/************************************************************************
* FILE:         ahl_tclBaseThread.h
* PROJECT:      VWLLNF
* SW-COMPONENT: (framework)
*----------------------------------------------------------------------
*
* DESCRIPTION:  base workthread's with event communication
*              
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2009 Robert Bosch GmbH, Hildesheim
* HISTORY:      
* Date      | Author   | Modification
* 30.07.08  | Hessling | added Thread with event comunication to AHL                     
* 16.02.09  | Hessling | suspend and resume added, more documentation done 
* 01.03.09  | Hessling | seperation in MessageThread and EventThread
*************************************************************************/
#ifndef AHL_BASETHREAD_H
#define AHL_BASETHREAD_H

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_THREAD_NAME "AHL________"
#define AHL_THREAD_START_WAIT_INFINITE ((tU32)(-1))
// use this switch to avoid interruptions caused by TRACE_ERROR etc.
//#define DISABLE_ASSERT

#ifdef DISABLE_ASSERT
#define FATAL_M_ASSERT(_INT_)             
#define NORMAL_M_ASSERT(_INT_)

#define FATAL_M_ASSERT_ALWAYS()
#define NORMAL_M_ASSERT_ALWAYS()
#endif

/**
* class ahl_tclThread is an minimal task having an event inbox to communicate
* with the thread function.
* By inheriting the ahl_tclThread the implementer defines:
* - events on which the thread shall react
* - parameters to be passed into the thread as membervariables of the thread
*   (don't forget semaphores if the varables are also used in other instances)
* - the bThreadFunction() which is called when an event occurs and just reacts 
*   on events coded in the event field. The function will be called
*   whenever an event has been posted by vPost().
* As typical for AHL the implementation is good practice and most commonly 
* used functionality. (about 64 times in di_middleware_server).
* It is the more simple and ressource saving as the ahl_BaseWorkThread becasue
* it does not use CCA message box.
* Design:
* tclThread care about thread handling functions and makes sure that the developper 
* has as little as possible to do with thread functions of OSAL. The thread is by
* default reached with event handling, wherein the developper can self define the 
* complete range of events. In parallel tclThread cares about controlling the thread 
* so that starting and stopping does avoid typical implementation errors like shutting
* down during still open handles, or freeing memory which is still in use by the thread.
* 
* @usage: "How to use this framework"
* 1. inherit the class and write an enum list of envents you want to handle in
*    bThreadFunction(). Put all data the thread will need as protected (or private)
*    Membervariables in your inheriting class. (e.g. a queue of values received from 
*    an ioPort being observed etc.)
* 2. According to typical class implementation put in AIL::bOnInit() the bStart() 
*    function and in AIL::vOnApplicationClose() bStop(). If you additionally want to 
*    save resources during OFF the entry may vSuspend() the thread and leaving may
*    vResume().
* 3. If you have to handle more than one thread you may want to shutdown them in 
*    parallel. For this case shutdown can be nonblocking initiated with vPostStopRequest()
*    you should afterwards wait for finla closure with blocking bStop() to make sure all
*    threads have really stopped.
*/
class ahl_tclBaseThread
{
public:

   /** CTOR creates the thread attributes. normally all threads should be defined via
   * the internal registry. For this case AppId and threadName should identify the 
   * thread. In cases where this data is not available you can also control the stack
   * size and priority with this constuctor. In case that no thread name is given, the 
   * this pointer address will be used to generate a unique string.
   * @param u16AppId your application id helping to find the entries in registry
   * @param tCString a string of max 31 chars defining the name also in registry
   *        if name is not given (0 or "") the constructor will create an individual   
   *        from the this pointer using usign the default parameters only.
   * @param u32DefaultStackSize if value cannot be retrieved from reg
   * @param u32DefaultPrio if value cannot be retrieved from reg
   */
   ahl_tclBaseThread(tU16 u16AppId=((tU16)-1), 
                     tCString szThreadName=0,
                     tS32 s32DefaultStackSize=1024, 
                     tU32 u32DefaultPrio=OSAL_C_U32_THREAD_PRIORITY_LOWEST);

   /** DTOR destroys the thread attributes. Before that bStop() is called to make sure 
   * that thread is shut down and will not further access any member variables.
   */
   virtual ~ahl_tclBaseThread();


   /** bStart() is called by the application to create and activate the thread 
   * @return true in case of succsefull activation. Additionally a wait timeout
   * can be given to get feedback if thread is finally in runnning state. 
   * The function will check each 10 msec if thread spawning was successfull.
   * The value 50msec is based on previous implementation and kept to behave 
   * unchanged if no parameter is given.
   * Please use 0 if you don't want to wait until thread runs and
   * use AHL_THREAD_START_WAIT_INFINITE to wait infinite for reaching the running state.
   * thread is already running. 
   */
   tBool bStart(tU32 u32WaitforStartTimeoutMsec = 50);

   /** bStop() is called by the application to stop the thread 
   * the thread internal function will not be reached again, but if one 
   * event processing is in progress suspension will be done afterwards.
   * The bStop() returns true when thread is stopped. bStop() will wait 
   * infinite until user function is done.
   * After bStop(); the thread can be restarted with bStart()
   */
   tBool bStop();

   /** vPostStopRequest() is called by the application to tell the thread 
   * to stop next time thread function is done. This function allows to 
   * put many threads to shut down. Use at later point with bStop(). until
   * all threads are really stopped before freeing resources used by the 
   * thread:
   * myThread1.vPostStopRequest();
   * myThread2.vPostStopRequest();
   * if(myThread1.bStop() && myThread2.bStop()) 
   *    TRACE("Threads are down.");
   * OSAL_DELETE myResource;
   */
   tVoid vPostStopRequest();

   /** vSuspend() puts the thread in inactive mode. I.e. no reaction on events.
   * you can use this function to avoid further execion e.g. in system state OFF.
   * The function waits until thread's user function is done and suspends the 
   * thread then.
   */
   tVoid vSuspend();

   /** vResume() reactivates the thread from suspend state.
   * the function does not clear the events already posted to the thread. If you want 
   * a clean start up call vClearEvents() before resuming the thread.
   */
   tVoid vResume();

   /** @return true in case that thread is spawned and in the active loop */
   tBool bGetIsThreadRunning() const {
      return _bThreadIsRunning;
   }
   /** @return trun in case that the thread is not suspended */
   tBool bGetIsThreadResumed() const {
      return _bThreadIsResumed;
   }

   /** vSetPriority() allows setting of the priority to a given value.
   * the function without parameter resets the priority to the default priority used 
   * during creation of the thread.
   */
   tVoid vSetPriority(tU32 u32Priority=0);

protected:
   /**
    * Clears the triggers (events / message queues) 
    */
   virtual tVoid vClearTriggers() =0;

   /**
    * Adds a trigger (event / message) which shall just brings the thread out of wait position. This function is called when base thread function wants to control thread condition. The trigger information will be removed and not reaching the bThreadFunction() 
    */
   virtual tVoid vTrigger() =0;

   /**
    * This function shall infinitively block at the trigger inbox
    * (eventWait / messageWait etc.)
    * @return tBool false in case that the wait function cannot be processed.
    */
   virtual tBool bWaitForTrigger() =0;

   /**
    * This function implements the thread work function the parameter is the
    * pointer to this of the given instance. The function enables inheriting
    * classes to define a thread function with the implementation specific
    * parameters.
    * @return tBool false if thread shall leave the loop and end.
    */
   virtual tBool bBaseThreadFunction(tPVoid pvInst) =0;

   /**
    * name of the thread but also used for creation of semaphores and events 
    */
   tChar _as8ThreadName[OSAL_C_U32_MAX_NAMELENGTH];

private:

   /** vThreadProc() is the thread loop function. It will run in endless loop 
   * waiting for events from the user and in case of events will call the 
   * bThreadFunction(event) to give inherited class the action to be executed 
   * in case of events.
   */
   static tVoid vThreadProc(tPVoid pvInst);

   /** this function provides the wait issuing asserts if we wait too long 
   * for more easy debugging im multi threaded environment.
   */
   tBool bWaitForThreadFunc(tBool &rbTest) const;

   /**
    * priority of the thread wherein 0-90 is reserved for Platform
    * 91 -150 is for applications
    * and 151 til 255 is used for low prio atcivities. 
    */
   tU32 _u32Prio;

   /**
    * stack for this thread 
    */
   tS32 _s32StackSize;

   /**
    * id given by creation for that thread 
    */
   OSAL_tThreadID _threadID;

   /**
    * pulled as long as loop is in work status. 
    * Not pulled if loog is suspended or powered down to OFF as internal state. 
    */
   OSAL_tSemHandle _hThreadFuncActive;

   /**
    * true as long as thread does not intend to leave the while loop 
    */
   tBool _bThreadIsRunning;

   /**
    * true if thread is in pause state, waiting for resumed. 
    */
   tBool _bThreadIsResumed;

   /**
    * user function requests to thread to stop as soon as the user function is ready by next . 
    */
   tBool _bThreadStopRequest;

   /**
    * Lets put the thread in Pause Mebr 
    */
   tBool _bThreadSuspendedRequest;
};

#endif // AHL_BASETHREAD_H
//EOF

