/************************************************************************
* FILE:         ahl_tclBaseThread.cpp
* PROJECT:      VWLLNF
* SW-COMPONENT: (framework)
*----------------------------------------------------------------------
*
* DESCRIPTION:  base workthread's with event communication
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2000 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author   | Modification
* 30.07.08  | Hessling | added Thread with event comunication to AHL
* 13.02.09  | Hessling | start and stop handling changed for improving against dead locks
* 01.03.09  | Hessling | seperation in MessageThread and EventThread
*************************************************************************/

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define SCD_S_IMPORT_INTERFACE_GENERIC
#include "scd_if.h" // for reading thread configuration from registry

#include "ahl_tclBaseThread.h"

#include "ahl_trace.h"
// -- trace identifier ------- define this value for __every__ file new
// regardless if used or not ;-)
#ifdef FILE_NUMBER
#undef FILE_NUMBER
#endif
#define FILE_NUMBER TRC::ahl_Thread_cpp // TODO
#ifdef FILE_NUMBER
#endif
// -- trace identifier ------- end


// use this define if you want to debug the stopping of your thread
#define WAIT_DIRECTLY_FOREVER

ahl_tclBaseThread::ahl_tclBaseThread(tU16 u16AppId,
                     tCString szThreadName,
                     tS32 s32DefaultStackSize,
                     tU32 u32DefaultPrio)
     :_u32Prio(u32DefaultPrio),_s32StackSize(s32DefaultStackSize),
      _threadID((OSAL_tThreadID)OSAL_ERROR),
      _hThreadFuncActive(OSAL_C_INVALID_HANDLE),
      _bThreadIsRunning(FALSE), _bThreadIsResumed(FALSE),
      _bThreadStopRequest(FALSE), _bThreadSuspendedRequest(FALSE)
   {
      TRACE_FLOW_DEF(TRC::FnConstructor);
      if(szThreadName && *szThreadName) { // at least one char in thread name
         tU32 len = OSAL_u32StringLength(szThreadName);
         if(OSAL_C_U32_MAX_NAMELENGTH>=len){
            len = (OSAL_C_U32_MAX_NAMELENGTH-1);
         }
         (tVoid)OSALUTIL_szSaveStringNCopy(_as8ThreadName, szThreadName, len);
         // make sure name is zero terminated
         _as8ThreadName[OSAL_C_U32_MAX_NAMELENGTH-1]=0;
         // know check for configuration
         (tVoid)scd_s32GetThreadConfiguration(u16AppId,_as8ThreadName,
            &_u32Prio, &_s32StackSize);
      } else {
         tU32 len = OSAL_u32StringLength(AHL_THREAD_NAME);
         (tVoid)OSALUTIL_szSaveStringNCopy(_as8ThreadName, AHL_THREAD_NAME, len);
         tU32 i = 2;
#ifdef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT
         for(uintptr_t u = (uintptr_t)this; u > 0; i++, u >>= 4) { // skips remaindering zeros
#else
         for(tU32 u = (tU32)this; u > 0; i++, u >>= 4) { // skips remaindering zeros
#endif	
            // put each nibble as char in the string      add 7 for hex A..F
            _as8ThreadName[len-i] = (tChar)(0x30 + (0x0f & u) + (((0x0f & u)<10)?0:7));
         }
      }
      // no priority inheritance for the worker thread function
      // put a '1' at last char to get different names
      tChar as8Name[OSAL_C_U32_MAX_NAMELENGTH];
      (tVoid)OSAL_szStringNCopy(as8Name,_as8ThreadName, OSAL_C_U32_MAX_NAMELENGTH);
      tU32 pos = OSAL_u32StringLength(as8Name)-1;
      as8Name[pos] ='1';
      TRACE_ON_ERROR_OSAL(TRC::FnConstructor,
         OSAL_s32SemaphoreCreate(as8Name, &_hThreadFuncActive, 0));
      // semaphore is posted when thread function does not run
      TRACE_ON_ERROR_OSAL(TRC::FnConstructor,
         OSAL_s32SemaphorePost(_hThreadFuncActive));
   }
ahl_tclBaseThread::~ahl_tclBaseThread(){
      TRACE_FLOW_DEF(TRC::FnDestructor);
      (tVoid)bStop(); //lint !e1551 Function may throw exception
      tChar as8Name[OSAL_C_U32_MAX_NAMELENGTH];
      (tVoid)OSAL_szStringNCopy(as8Name,_as8ThreadName, OSAL_C_U32_MAX_NAMELENGTH);
      tU32 pos = OSAL_u32StringLength(as8Name)-1;
      as8Name[pos] ='1';
      (tVoid)OSAL_s32SemaphoreClose(_hThreadFuncActive);
      (tVoid)OSAL_s32SemaphoreDelete(as8Name);
   }
tBool ahl_tclBaseThread::bStart(tU32 u32WaitforStartTimeoutMsec){
   TRACE_FLOW_DEF(TRC::FnStart);
   if(_bThreadIsRunning) {
      vResume();
      TRACE_INFO_RES(TRC::FnStart, bGetIsThreadRunning());
      TRACE_INFO_RES(TRC::FnStart, bGetIsThreadResumed());
      return TRUE;
   }
   OSAL_trThreadAttribute rThAttr;
   rThAttr.szName       = _as8ThreadName;
   rThAttr.u32Priority  = _u32Prio;
   rThAttr.s32StackSize = _s32StackSize;
   rThAttr.pfEntry      = (OSAL_tpfThreadEntry)vThreadProc;
   rThAttr.pvArg        = (tPVoid)this;

   // start the thread
   _bThreadIsRunning = FALSE;
   _bThreadIsResumed = FALSE;
   if((_threadID = OSAL_ThreadSpawn(&rThAttr))==OSAL_ERROR){
      TRACE_ERROR_OSAL(TRC::FnStart);
      _bThreadStopRequest = FALSE;
      _bThreadSuspendedRequest = FALSE;
      TRACE_INFO_RES(TRC::FnStart, bGetIsThreadRunning());
      TRACE_INFO_RES(TRC::FnStart, bGetIsThreadResumed());
      return FALSE;
   }
   _bThreadStopRequest = FALSE;
   _bThreadSuspendedRequest = FALSE;
   TRACE_DBG_LSN(TRC::FnStart, TR_LEVEL_USER_2, "u32WaitforStartTimeoutMsec", (tS32)u32WaitforStartTimeoutMsec);
   if (u32WaitforStartTimeoutMsec == 0) {
      return TRUE;
   }
   for (tU32 cnt=0; 
         (cnt < u32WaitforStartTimeoutMsec || 
          (u32WaitforStartTimeoutMsec == AHL_THREAD_START_WAIT_INFINITE)); 
         cnt += 10
       ) 
   {
      OSAL_s32ThreadWait(10); // leave some process time to the worker thread
      if (_bThreadIsRunning==TRUE) {
         break;
      }
   }
   TRACE_DBG_LSS(TRC::FnStart, TR_LEVEL_USER_2, "Thread spawned. ", _as8ThreadName);
   TRACE_INFO_RES(TRC::FnStart, bGetIsThreadRunning());
   TRACE_INFO_RES(TRC::FnStart, bGetIsThreadResumed());
   return (_bThreadIsRunning==TRUE);
}
tBool ahl_tclBaseThread::bStop(){
   TRACE_FLOW_DEF(TRC::FnStop);
   vPostStopRequest();
   tBool bRes = bWaitForThreadFunc(_bThreadIsRunning);
   _threadID = OSAL_ERROR;
   _bThreadIsRunning = FALSE;
   _bThreadIsResumed = FALSE;
   TRACE_INFO_RES(TRC::FnStop, bGetIsThreadRunning());
   TRACE_INFO_RES(TRC::FnStop, bGetIsThreadResumed());
   return bRes; //thread stopped
}
tVoid ahl_tclBaseThread::vPostStopRequest(){
   TRACE_FLOW_DEF(TRC::FnPostStopRequest);
   if(_bThreadIsRunning == FALSE) {
      return; //thread already stopped
   }
   //request stop
   _bThreadStopRequest = TRUE;
   // send the trigger to thread
   vTrigger();
   vResume();
}
tVoid ahl_tclBaseThread::vSuspend(){
   TRACE_FLOW_DEF(TRC::FnSuspend);
   if((_bThreadIsRunning)&&(_threadID!=OSAL_ERROR)&&
      (_bThreadIsResumed==TRUE)) {
      _bThreadSuspendedRequest = TRUE;
      // make sure at least one event activates the thread
      vTrigger();
      (tVoid)bWaitForThreadFunc(_bThreadIsResumed);
      TRACE_ON_ERROR_OSAL(TRC::FnSuspend,
         OSAL_s32ThreadSuspend(_threadID));
   }
   TRACE_INFO_RES(TRC::FnSuspend, bGetIsThreadRunning());
   TRACE_INFO_RES(TRC::FnSuspend, bGetIsThreadResumed());
}
tVoid ahl_tclBaseThread::vResume(){
   TRACE_FLOW_DEF(TRC::FnResume);
   if((_bThreadIsRunning)&&(_threadID!=OSAL_ERROR)&&
      (_bThreadIsResumed == FALSE)) {
      _bThreadSuspendedRequest = FALSE;
      TRACE_ON_ERROR_OSAL(TRC::FnSuspend,
         OSAL_s32ThreadResume(_threadID));
   }
   TRACE_INFO_RES(TRC::FnResume, bGetIsThreadRunning());
   TRACE_INFO_RES(TRC::FnResume, bGetIsThreadResumed());
}

tVoid ahl_tclBaseThread::vSetPriority(tU32 u32Priority){
   TRACE_FLOW_DEF(TRC::FnSetPriority);
   tU32 prio = _u32Prio; // default from ctor
   if(u32Priority != 0) {
      // use given priority
      prio = u32Priority;
   }
   TRACE_ON_ERROR_OSAL(TRC::FnSetPriority,
      OSAL_s32ThreadPriority(_threadID , prio ));
   tChar szText[100] = "Setting prio of thread ";
   TRACE_DBG_LSN(TRC::FnSetPriority, TR_LEVEL_USER_3,
      OSALUTIL_szSaveStringNConcat(szText, _as8ThreadName, OSAL_u32StringLength(_as8ThreadName))
      , prio);
}
tVoid ahl_tclBaseThread::vThreadProc(tPVoid pvInst){
   TRACE_FLOW_DEF(TRC::FnThreadProc);
   ahl_tclBaseThread* poThis = (ahl_tclBaseThread*)pvInst;
   if(poThis == 0) return;
   // signal that we work with the semaphore
   TRACE_ON_ERROR_OSAL(TRC::FnThreadProc,
      OSAL_s32SemaphoreWait(poThis->_hThreadFuncActive, OSAL_C_TIMEOUT_FOREVER));
   // go into endless loop and do the work
   poThis->_bThreadIsRunning = TRUE;
   poThis->_bThreadIsResumed = TRUE;
   TRACE_DBG_LSS(TRC::FnThreadProc, TR_LEVEL_USER_2, "Thread enters while loop.", poThis->_as8ThreadName);
   while(poThis->_bThreadIsRunning)
   {
      // check if event handle hasn't been closed
     if(!poThis->bWaitForTrigger())
      {
         // we can't do anything else than stopp
         TRACE_ERROR(TRC::FnThreadProc, TRC::ErrNullPointerInstanceMissing);
         poThis->_bThreadIsRunning = FALSE;
         break;
      }
      // wait for next action to do

      // in case of bStop we have sent any event to stop and
      // did put a STOP_REQUEST in the control flag
      if(poThis->_bThreadStopRequest)//thread stop requested
      {
         TRACE_DBG_LSS(TRC::FnThreadProc, TR_LEVEL_USER_2,
            "Thread stop requested, thread stops.", poThis->_as8ThreadName);
         poThis->_bThreadIsRunning = FALSE;
         break;
      }
      // in case of suspend we have sent any event to put thread in a safe status
      if(poThis->_bThreadSuspendedRequest)//thread suspend requested
      {
         TRACE_DBG_LSS(TRC::FnThreadProc, TR_LEVEL_USER_2,
            "Thread suspend requested, thread is idle.", poThis->_as8ThreadName);
         poThis->_bThreadIsResumed = FALSE;
         TRACE_ON_ERROR_OSAL(TRC::FnThreadProc,
            OSAL_s32SemaphorePost(poThis->_hThreadFuncActive));
         while(poThis->_bThreadSuspendedRequest) {
            // we idle as long as user thread does not reset the request
            (tVoid)OSAL_s32ThreadWait(100);
         }
         TRACE_ON_ERROR_OSAL(TRC::FnThreadProc,
            OSAL_s32SemaphoreWait(poThis->_hThreadFuncActive, OSAL_C_TIMEOUT_FOREVER));
         TRACE_DBG_LSS(TRC::FnThreadProc, TR_LEVEL_USER_2,
            "Thread resume requested, thread is running.", poThis->_as8ThreadName);
         poThis->_bThreadIsResumed = TRUE;
         continue; // put thread back in event wait status
      }

      // --------- call to the working function ---------
      if (poThis->bBaseThreadFunction(pvInst) == FALSE) // user requested stop;
      {
         TRACE_DBG_LSS(TRC::FnThreadProc, TR_LEVEL_USER_2,
            "Thread func returns false, thread stops.", poThis->_as8ThreadName);
         poThis->_bThreadIsRunning = FALSE;
         break;
      }
   } // while
   TRACE_DBG_LSS(TRC::FnThreadProc, TR_LEVEL_USER_2,
      "Thread left while loop.", poThis->_as8ThreadName);
   // signal read to waiting user
   TRACE_ON_ERROR_OSAL(TRC::FnThreadProc,
      OSAL_s32SemaphorePost(poThis->_hThreadFuncActive));
   // explicit exit here removes savely the stack
   OSAL_vThreadExit();
}
tBool ahl_tclBaseThread::bWaitForThreadFunc(tBool &rbTest) const{
   TRACE_FLOW_DEF(TRC::FnWaitForThreadFunc);
   // in the following sequence result of OSAL_s32SemaphoreWait is tested
   (tVoid)OSAL_s32SemaphoreWait(_hThreadFuncActive, 4000);
   if(rbTest != FALSE) {
#ifndef WAIT_DIRECTLY_FOREVER
      NORMAL_M_ASSERT((tU32)"Thread did not stop in 4 secs." == (tU32)_as8ThreadName); // address check just to transport the information
      (tVoid)OSAL_s32SemaphoreWait(_hThreadFuncActive, 6000);
   }
   if(rbTest != FALSE) {
      NORMAL_M_ASSERT((tU32)"Thread did not stop in 10 secs." == (tU32)_as8ThreadName); // address check just to transport the information
      (tVoid)OSAL_s32SemaphoreWait(_hThreadFuncActive, 10000);
   }
   if(rbTest != FALSE) {
      NORMAL_M_ASSERT((tU32)"Thread did not stop in 20 secs wait for ever." == (tU32)_as8ThreadName); // address check just to transport the information
#endif // WAIT_DIRECTLY_FOREVER
      (tVoid)OSAL_s32SemaphoreWait(_hThreadFuncActive, OSAL_C_TIMEOUT_FOREVER);
   }
   if(rbTest != FALSE) {
      // this code is never reached
      TRACE_WARN(TRC::FnWaitForThreadFunc, TRC::WarnThreadStopDidNotWorkOut);
      return FALSE;
   }
   TRACE_ON_ERROR_OSAL(TRC::FnWaitForThreadFunc,
      OSAL_s32SemaphorePost(_hThreadFuncActive));
   return TRUE;
}
