///////////////////////////////////////////////////////////
//  ahl_tclBaseConfigStore.h
//  Implementation of the Class ahl_tclBaseConfigStore
//  Created on:      20-Sep-2007
//  Original author: Matthias Heling
///////////////////////////////////////////////////////////

#if !defined(AHL_TCLBASECONFIGSTORE_H)
#define AHL_TCLBASECONFIGSTORE_H

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

  /**
   ---General---
   The class supports with all functionality needed to store configuration data into 
   the FFS in a standardized way. The developer inherits from the framework class and
   implements the set and get functions for his data.
   Additionally he can decide if data shall be stored in machine specific or user  
   specific file. He need not to care about which user is set; the framework checks 
   this already with the value set in the registry. The user id in registry is 
   expected to be only valid for the time when loadData is called. Therefore the 
   value is only read in loadData function and storeData uses this remembered value.
   This requires at least one successfull loadData before storeData is working.
   In exceptional case developer can adjust or retrieve the users id with given 
   functions. To set the user id from outside.
   ---Versioning---
   The framework also supports versioning with major and minor numbers. In case 
   developer wants to report backward compatibility (see. herunder) he increases
   the minor number only. In case of breaking backward compatibility increasing the
   major version number is required, which ensures old configuration data to be 
   deleted.
   ---Backward Compatibility---
   In case a system has been updated the old configuration data can be reused if 
   developer signals backward compatibility. In this case before reading the 
   configuration from the file, the new version of the default data is used to preset
   the memory as binary buffer. If now the neu version has one configuration byte more 
   then the old one, the new byte will stay with default value and the old version 
   data is read from the file. The storeData function in any case stores the actual
   version into the file.
   ---Defaultvalues---
   The framework requests default values for the configuration data which are used
   to preset the data buffer. In case of not implementing the setDefaultData function
   the framework will not create any file and also not load any file. (used to switch
   the function off.)
   ---Getter and Setter---
   The function vSetMachineData() and vSetUserData() provide for the time of call
   access to the configuration data read out of the config file. The developer 
   shall memcpy() this data as binary byte array in back into the start address of
   his configuration data. This function is triggered with the call of 
   vLoadMachineData() and vLoadUserData() at state transition into NORMAL state.
   The function vGetMachineData() and vGetUserData() same as the both xxxDefaultData()
   request access to a byte array containing the configuration data for the live 
   time of the class instance. This functions are triggered by the call to 
   StorexxxData() at state transition into OFF and DIAGNOSIS.
   */
class ahl_tclBaseConfigStore
{

public:
	/**
	 * the constructor defines the full qualyfied name where the data is sored.
	 * "/dev/ffs/MyDir/MySubDir/MyFileNAME" the constructor will concatinate
    *  "m-cfg.dat" etc. to the string to get ability to differentiate between 
    * the different profiles.
    * @param   szConfigFileName is expected to be a valid file name 
    * @param   u8Major the number controlling backward incompatible changes 
    * @param   u8Minor the number controlling backward compatible changes
	 */
	ahl_tclBaseConfigStore(tCString szConfigFileDir, tU16 u16Major, tU8 u8Minor);

   /**
    * the destructor frees the config file names 
    */
	virtual ~ahl_tclBaseConfigStore();

   /** -------------- machine depended configuration data ------------------- 
   * this section declares function for data being independent from the user
   * profile. 
   */
public:                        //lint !e1736 prio3 reviewed: added for explantory reasons
   /**
   * vLoadMachineData activates reading of the configuration data from FFS and 
   * should be called in the application state transition actions (e.g. NORMAL)
   */
	tVoid vLoadMachineData();

   /**
   * vStoreMachineData activates writing of the configuration data from FFS and 
   * should be called in the application state transition actions (e.g. OFF)
   */
	tVoid vStoreMachineData();

   /**
    * This function removes the machine specific data.
    * Also the file is deleted.
    */
   tVoid vRemoveMachineData(tVoid);

   /**
   * in case of loading configuration data application may want to know the version 
   * of the file read for configuration data. This function provides the version of
   * this file.
   * @param u16Major major version of file
   * @param u8Minor minor version of file
   * @return bool true if file version has been set by this function, false otherwise
   */
   tBool bGetConfigFileVersion(tU16 &ru16Major, tU8 &ru8Minor) const;

	/**
    * vSetMachineData() is the basic function providing the data of the machine 
    * configuration file. The framework reads the File and provides the data as 
    * binary array. It is required, that the application memcopies the data into 
    * its own memory the data behind the pointer becomes invalid immediately after 
    * leaving this function. In case that developer won't use machine dependent
    * config data he simply implements this function as empty function. "{}"
    * @param tPCU8 pcu8Buf reference to the const buffer to be copied by the App
    * @param tU32 u32Length length in bytes of the buffer
    */
	virtual tVoid vSetMachineData(tPCU8 pcu8Buf, tU32 u32Length)=0;

   /**
	 * This function requests a pointer to an array, containing the values to be
    * stored in this configuration file. The buffer shall be valid for the life time 
    * of this object. In case that developer won't use machine dependent config 
    * data he simply implements this function as empty function. "{}"
    * @param tPCU8 &rpcu8Buf reference to the buffer to be copied
    * @param tU32 &ru32Length length in bytes of the buffer
	 */
	virtual tVoid vGetMachineData(tPCU8 &rpcu8Buf, tU32 &ru32Length) =0;

   /**
	 * This function requests a pointer to an array, containing the default values
    * for this configuration files. The buffer shall be valid for the life time of
    * this object. In case that developer won't use machine dependent config 
    * data he simply implements this function as empty function. "{}"
    * @param tPCU8 &rpcu8Buf reference to the buffer to be copied
    * @param tU32 &ru32Length length in bytes of the buffer
	 */
	virtual tVoid vGetMachineDefaultData(tPCU8 &rpcu8Buf, tU32 &ru32Length) =0;

   /* -------------- user profiled configuration data functions ------------------- 
   * this section declares function for data being dependent from the user's
   * profile id. 
   */
public:                        //lint !e1736 prio3 reviewed: added for explantory reasons
   /**
	 * vSetUserId function allows setting explicitly of user profile actually used.
    * The function will deactivate the reading of the user id from the registry.
    * The userid actually supports up to 0x07 in the file name. Therefor only 
    * userids 0-7 are valid. 
    * @param u8UserId [0-7]
	 */
	tVoid vSetUserId(tU8 u8UserId);

	/**
    * This function returns true if profile id is valid and sets then ru8UserId 
    * to the given num.
	 * @param (out) number of actually activeated user profile
	 * @return ist TRUE if user id is currently valid.
	 */
	tBool bGetUserId(tU8& ru8UserId) const;

	/**
    * This function resets the flag to constructor default using Registry for reading
    * the user id from the registry.
	 * @param (bool) TRUE: use registry; FALSE: don't use registry 
	 */
	tVoid vSetUserIdFromReg(tBool bOn);

   /**
   * vLoadUserData activates reading of the user dependent configuration data from 
   * FFS and should be called in the application state transition actions (
   * e.g. NORMAL)
   */
	tVoid vLoadUserData();

   /**
   * vStoreUserData activates writing of the user dependent configuration data from 
   * FFS and should be called in the application state transition actions (e.g. OFF)
   */
	tVoid vStoreUserData();

   /**
    * This function removes the user specific data for the useer acually set.
    * Also the file is deleted.
    */
   tVoid vRemoveUserData(tVoid);

   /**
    * This function removes all user specific data.
    * Also the files are deleted.
    */
   tVoid vRemoveAllUserData(tVoid);

	/**
    * This is the basic function providing the data of the user configuration file.
    * The framework reads the File and provides the data as binary array. 
    * it is required, that the application memcopies the data into its own memory
    * the data behind the pointer becomes invalid immediately after leaving this 
    * function. In case that developer won't use machine dependent config 
    * data he simply implements this function as empty function. "{}"
    * @param tPCU8 pcu8Buf reference to the const buffer to be copied by the App
    * @param tU32 u32Length length in bytes of the buffer
    */
	virtual tVoid vSetUserData(tPCU8 pcu8Buf, tU32 u32Length) =0;

   /**
	 * This function requests a pointer to an array, containing the values to be
    * stored in this configuration file. The buffer shall be valid for the life time 
    * of this class. In case that developer won't use machine dependent config 
    * data he simply implements this function as empty function. "{}"
    * @param tPCU8 &rpcu8Buf reference to the buffer to be copied
    * @param tU32 &ru32Length length in bytes of the buffer
	 */
   virtual tVoid vGetUserData(tPCU8 &rpcu8Buf, tU32 &ru32Length) =0;

   /**
	 * This function requests a pointer to an array, containing the default values
    * for this configuration files. The buffer shall be valid for the life time of
    * this class. In case that developer won't use machine dependent config 
    * data he simply implements this function as empty function. "{}"
    * @param tPCU8 &rpcu8Buf reference to the buffer to be copied
    * @param tU32 &ru32Length length in bytes of the buffer
	 */
   virtual tVoid vGetUserDefaultData(tPCU8 &rpcu8Buf, tU32 &ru32Length) =0;

private:
   /**
   * structure used as header In any configfile application version set by write. 
      Main version is 16 bit to enable easy use is sizeof() as version which easily
      can be bigger as 255.
      Minor version have been made u8 to have at least one byte available
      to signal changes in the header structure of config files. This value is 
      zero in initial version.
      Together with the length coding 8 byte alignment is accomplished.
   */
   struct stCfgHeader {
      tU16 u16VersionMaj;   
      tU8 u8VersionMin;
      tU8 u8Reserved;
      tU32 u32BufLen;     // length of bytes following
   } mCfgHeader;
 
   /**
   * default constructor is forbidden, to make sure in any case 
   * filename to configuration file will be provided by the developer.
   */
   ahl_tclBaseConfigStore();     //lint !e1704 prio3 reviewed: default constructor forbidden for this class
   ahl_tclBaseConfigStore(const ahl_tclBaseConfigStore&); //lint !e1704 prio3 reviewed: copying also forbidden
   ahl_tclBaseConfigStore& operator=(const ahl_tclBaseConfigStore&);
   /**
    * create the directories requested by the user
    */
   tVoid vCreateDir(tCString szDir) const;
   
   /**
    * create a file with the given content
    */
   tVoid vCreateConfigFile(tCString szFileName, tPCU8 pcDefBuf, tU32 u32DefLen);

   /**
   * encapsulates the method how the user id is inserted into the user config
   * file name.
   * the function expects a valid mszUserConfigFile, mu8UserIdPos and mu8UserId
   * available and sets the number into the string.
   */
   tVoid vSetUserIdInFileName();
   
   /**
   * reads the user id value out of the registry 
   * /BLAUPUNKT/PROCESS/SYSTEM/USER_PROFILE_NUM = u32
   * valid values in VW project are 1-4 in case of no reg 1 is default in case of 
   * invalid higher numbers 4 is returned
   * @return   the user id after problem solution.
   */
   tU8 u8GetUserIdFromRegistry() const ;
   
   /**
   * general load function 
   * @param szConfigFileDir file to load from
   * @param rpcu8Buf buffer pointer to load into 
   * @param rpcu8DefBuf buffer containing default values
   * @return true if loading was successful, caller must delete[] rpcu8Buf
   *         false if no loading -> no buffer
   */
   tBool bLoadData(tCString szConfigFile, 
                   tPU8 &rpu8Buf, 
                   tU32 &ru32Length, 
                   tPCU8 pcu8DefBuf, 
                   tU32 u32DefLen);
   /**
   * general store function; checks avaiablility of th efile but does not create a 
   * new one. Creation of file was already tried in load function. The function
   * expects version in mCfgHeader correctly set and adjusts length in the header.
   * @param szConfigFile file to store to  
   * @param pcu8Buf data to store into
   */ 
   tVoid vStoreData(tCString szConfigFile,  
                    tPCU8 pcu8Buf, 
                    tU32 u32BufLen);

private:                      //lint !e1736 prio3 reviewed: added to seperate functions and variables 
   /**
    * number of the user used for profile data
    */
   tU8 mu8UserId;   

   /** 
   * set true if actual userid is valid
   * invalid then no store! and load looks in regitry
   */
   tBool mbUserIdOk;

   /**
   * set true if actual userid shall be read from registry calling vSetUserId()
   * switches registry off, which is not used anymore for the lifetime of this object
   */
   tBool mbUserIdFromReg;
                     
   /**
   * pos where num is in the file
   */
   tU8 mu8UserIdPos; 

   /**
    * name for machine configuration file  
    */
   tString mszMachineConfigFile;

   /**
    * name for user configuration file  
    */
   tString mszUserConfigFile;

   /**
   * remember version of configfiles read as last 
   */
   tU16 mu16FileVerMaj;
   tU8 mu8FileVerMin ;
   tBool mbFileVerOk ;

};
#endif // !defined(AHL_TCLBASECONFIGSTORE_H)
