#ifndef AHL_PORTS_HEADER
#define AHL_PORTS_HEADER

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#ifndef AMT_S_IMPORT_INTERFACE_GENERIC
#define AMT_S_IMPORT_INTERFACE_GENERIC
#endif
#include "amt_if.h"

#ifndef AIL_S_IMPORT_INTERFACE_GENERIC
#define AIL_S_IMPORT_INTERFACE_GENERIC
#endif
#include "ail_if.h"

#ifndef AHL_S_IMPORT_INTERFACE_GENERIC
#define AHL_S_IMPORT_INTERFACE_GENERIC
#endif
#include "ahl_if.h"         // use Application Help Library


// Communication Port library returns following return codes when
enum ahl_tenPortResult 
   {
// success codes 
      AHL_S_POST_OK,    // message successfully posted 
      AHL_S_MSG_BLOCKED,  // message blocked from sending
      AHL_S_UPDATE_OK,    // update executed successfully
      AHL_S_NO_UPREG_CLIENT,  //  server posts an UpReg message with no registered client. 
      AHL_S_UPREG_MSG_DELETED, // client code tries to send more than one UpReg message to server. 
      AHL_S_CAN_DELETE_ME,      // method stream port informs port manager with this return 
                              // code when it has sent all the messages to the client.
// error codes
      AHL_E_UNKNOWN_MSG,   // Port library recieves an unknown message *
      AHL_E_POST_FAILED,       // ail communication error
      AHL_E_LOW_MEMORY      // memory allocation failed.
   };

//* An unknown message is any message with opcodes which port library does not understand. 


/*
All UpReg status messages must have cmd counter as 
AHL_UPREG_CMD_COUNTER. This helps ServerPortMngr in 
differentiating a set/get status message from Property 
status message using the cmdcounter. 
*/
const tU16 AHL_UPREG_CMD_COUNTER  = 0xFFFF;


/*
This class provides a kind of message smart pointer.
It takes over the shared memory part of an AMT message, and
automatically deletes it if it is assigned another message.
*/
class ahl_MsgPtr : public amt_tclBaseMessage
{
public:
   ahl_MsgPtr();
   ahl_MsgPtr(const amt_tclBaseMessage* poMsg);
   ahl_MsgPtr(const ahl_MsgPtr& oRHS);
   virtual ~ahl_MsgPtr();  //ABR compiler warning

   ahl_MsgPtr& operator=(ahl_MsgPtr& oRHS);
   amt_tclBaseMessage* release() const;
   tVoid vDelete();

private:
   tBool m_bUsed;
};


/*
This class implements BitMap datasturcture
with basic functionality.

 Type can be any of the built in types.
 char,short,int,long
*/
template<class Type>
class ahl_Bitmap
{
public:
   ahl_Bitmap(Type state = 0) : m_state(state){}
   tVoid vAssign(Type flag)
   {
      m_state = flag;
   }
   
   tVoid vAppend(Type flag)
   {
      m_state |= flag;
   }

   tVoid vRemove(Type flag)
   {
      m_state &= ~flag;
   }

   tBool bIsSet(Type flag) const
   {
      // check whether all the bits in flag are set
      // in m_state.
      Type temp = m_state & flag;
      return temp == flag;
   }

private:
   Type m_state;
};


/*
- this is a client side port.
  ahl_tclMethodPort objects know how to store the messages.
  object's identity is with its FunctionID.
--  classes derived from ahl_tclMethodPort can customise the way messages are stored
    in the method port.
*/
class ahl_tclMethodPort 
{
public:
   ahl_tclMethodPort(amt_tclServiceData* poMsg = 0);
   virtual ~ahl_tclMethodPort(){}

   tBool bIsYourMsg(const amt_tclServiceData* poMsg) const;

   tVoid vGetMsg( ahl_MsgPtr& oMsg);

   virtual tVoid vStore(amt_tclServiceData* poMsg);

private:
   ahl_MsgPtr m_oMsg;
   tU16      m_u16FunctionID;
};



// defines conceptual port interface implemented by Every Port.
/*
class ahl_tclPort 
{
public:
   
   // when ail application posts a message to PortMngr, PortMngr identifies 
   //   appropriate port and calls enPost to process the message.   
   ahl_tenPortResult enPost(ail_tclAppInterfaceRestricted* pAilObj,amt_tclServiceData* poMsg );

   // Port Mngr informs port by calling enUpdate, 
   //   when an ail application recieves a new message.
   ahl_tenPortResult enUpdate(ail_tclAppInterfaceRestricted* pAilObj,const amt_tclServiceData* poMsg );

   // Port Mngr asks port to clear any pending message(s),target info by calling this function.
   tVoid vClear();

   // returns true if the message belongs to this port.
   tBool bIsYourMsg(const amt_tclServiceData* poMsg) const; 
};
*/

/*
- this is a client side port. 
Responsibility : handles all method and method stream requests under a given service ID.
Each service port object is identified with it's unique service ID.
implementation details : This class maintains one method ports list and one pending requests queue.
method ports list maintains the method port objects list.
pending requests queue maintains message sequence.Since messages are stored in method ports.
pending requests queue maintains pointers to method/method stream ports in FIFO.
*/
class ahl_tclServicePort 
{
public:
   ahl_tenPortResult enPost(ail_tclAppInterfaceRestricted* pAilObj,amt_tclServiceData* poMsg );
   ahl_tenPortResult enUpdate(ail_tclAppInterfaceRestricted* pAilObj,const amt_tclServiceData* poMsg );
   tVoid vClear();
   tBool bIsYourMsg(const amt_tclServiceData* poMsg) const; 

   ahl_tclServicePort(tU16 u16ServiceId):m_bPortClosed(FALSE),m_u16ServiceId(u16ServiceId){} //ABR compiler warning
   virtual ~ahl_tclServicePort();

private:
   ahl_tenPortResult enSendNextPendingMsg( ail_tclAppInterfaceRestricted* pAilObj);
   ahl_tenPortResult enSendAck(ail_tclAppInterfaceRestricted* pAilObj,const amt_tclServiceData* poMsg );
   ahl_tclMethodPort*  getPort(amt_tclServiceData* poMsg);
   ahl_tclMethodPort*  findPort(const amt_tclServiceData* poMsg,fwl_List<ahl_tclMethodPort*>& list) const;

   /* list of method port objects created by ServicePort or
      supplied by user. these pointers must be deleted explicitly.
   */
   fwl_List<ahl_tclMethodPort*> m_methodPortsList; // store method port objects. we have to delete these objects 
   fwl_List<ahl_tclMethodPort*> m_pendingMethodsQ; // pending methods queue.only pointers are stored
   tBool m_bPortClosed;
   tU16 m_u16ServiceId;
};


/* 
ahl_tclClientPropertyPort behaviour :
1) at a single instance, a property port can be open for
both  set (or get) and UpReg request. this means client can
send both set (or get) and UpReg with out blocking each other.
But in set/get, only one state is active at a time.
2) set request have higher priority than get request.
3) when a set request is to be posted and get request is in 
pending state, then get request is deleted and set request will be
in pending.
4) when a get request is to be posted with set request in pending state,
then new get request will be ignored(i.e deleted ).
*/

class ahl_tclClientPropertyPort 
{
public:
   ahl_tenPortResult enPost(ail_tclAppInterfaceRestricted* pAilObj,amt_tclServiceData* poMsg );
   ahl_tenPortResult enUpdate(ail_tclAppInterfaceRestricted* pAilObj,const amt_tclServiceData* poMsg );
   tVoid vClear();
   tBool bIsYourMsg(const amt_tclServiceData* poMsg) const ;


   ahl_tclClientPropertyPort(tU16 u16ServiceId,tU16 u16FunctionId);
   virtual ~ahl_tclClientPropertyPort(){};


private:
   typedef tU8 BitArray;

   enum 
   {
      INITIALISED = 0,
      WAITING_FOR_SET_GET_STATUS = 0x01,
      WAITING_FOR_UPREG_STATUS = 0x02,
      SET_MSG_PENDING = 0x04,
      GET_MSG_PENDING = 0x08
   };

   ahl_tenPortResult enSendAck( ail_tclAppInterfaceRestricted* pAilObj, const amt_tclServiceData* poMsg );

   ahl_Bitmap<BitArray> m_oState;
   tU16 m_u16ServiceId;
   tU16 m_u16FunctionId;
protected :
   virtual tVoid vStoreSetGetMsg( amt_tclServiceData* poMsg );

   ahl_MsgPtr m_oSetGetMsg; // to store set value;
};



/*
Behaviour : blocks more than one method request at a time, and stores the 
responses in m_pendingAnswersQ until acknowledgement is recieved from the client.
*/
class ahl_tclServerMethodStreamPort 
{
public:
   ahl_tenPortResult enPost(ail_tclAppInterfaceRestricted* pAilObj,amt_tclServiceData* poMsg );
   ahl_tenPortResult enUpdate(ail_tclAppInterfaceRestricted* pAilObj,const amt_tclServiceData* poMsg );
   tVoid vClear();
   tBool bIsYourMsg(const amt_tclServiceData* poMsg) const;


   ahl_tclServerMethodStreamPort(tU16 u16RegisterId,tU16 u16FunctionId)
      :m_bPortClosed(FALSE),m_u16RegisterId(u16RegisterId),m_u16FunctionId(u16FunctionId){}  //ABR Compiler warning
   virtual ~ahl_tclServerMethodStreamPort(){}


private:
   fwl_List<ahl_MsgPtr> m_pendingAnswersQ; // pending messages to be sent to client.
   tBool m_bPortClosed;
/*
Assumption : server's method stream response for a specific client request
can be uniquely identified with RegisterId and FunctionId
*/
   tU16 m_u16RegisterId;
   tU16 m_u16FunctionId;
};


/*
   helper class who maintains client identity ( with register id) and
   client specific locks ( to send or not to send a message )
   and knows how to store a message. 
*/
class ahl_tclClientInfo
{
public:
   // by default clientInfo is unlocked and can send message to client.
   ahl_tclClientInfo(tU16 u16RegisterID,tBool bLocked = FALSE);

   virtual ~ahl_tclClientInfo();

   tBool operator ==(const ahl_tclClientInfo& client) const
   {
      return m_u16RegisterID == client.m_u16RegisterID; 
   }

   tBool operator !=(const ahl_tclClientInfo& client) const
   {
      return ! operator == (client);
   }

   tBool bIsYourMsg(const amt_tclServiceData* poMsg ) const ;

   ahl_tenPortResult enPost
   (
      ail_tclAppInterfaceRestricted* pAilObj,
      amt_tclServiceData* poMsg,
      tBool bDoUnlock = FALSE
   );

   ahl_tenPortResult enUpdatePendingMsg
   (
      ail_tclAppInterfaceRestricted* pAilObj
   );

   tVoid vClear();

private :

typedef tU8 BitArray;

   enum { INITIALISED = 0, PENDING_MSG_AVAILABLE = 0x1, LOCKED = 0x2 };


   tVoid vStore(const amt_tclServiceData* poMsg )
   {
      ahl_MsgPtr oMsg(poMsg);
      m_oMsg = oMsg;
      m_oState.vAppend(PENDING_MSG_AVAILABLE);
   }


   ahl_Bitmap<BitArray> m_oState;
   ahl_MsgPtr m_oMsg;

   tU16  m_u16RegisterID; // indicates client identity.
};


/*
behaviour : handles property notification reqeusts on server side.
this class notifies the client and locks it until 
acknowledgement is recieved from client. after recieving acknowledgement from
client, if there is any pending msg, client is informed immediately otherwise
client will be informed when the next notification is posted by the server.
*/
class ahl_tclServerPropertyPort 
{
public:
   ahl_tenPortResult enPost(ail_tclAppInterfaceRestricted* pAilObj,amt_tclServiceData* poMsg );

   /* enUpdate will receive both new UpReg registration
     requests and Port Acknowledge messages from clients. */
   ahl_tenPortResult enUpdate(ail_tclAppInterfaceRestricted* pAilObj,const amt_tclServiceData* poMsg );

   tVoid vClear();

   tBool bIsYourMsg(const amt_tclServiceData* poMsg) const;

   ahl_tclServerPropertyPort(tU16 u16ServiceId,tU16 u16FunctionId)
      :m_u16ServiceId(u16ServiceId),m_u16FunctionId(u16FunctionId){}
   virtual ~ahl_tclServerPropertyPort(){}

private:
   fwl_List<ahl_tclClientInfo> m_RegisteredClients; // pending messages to be sent to client.
   tU16 m_u16ServiceId;
   tU16 m_u16FunctionId;
};



/*
 interface for ail client application. 
 Responsibility : delegates method and method stream reqeusts to appropriate service port
 and property reqeusts to appropriate property port.
 implementation detail : maintains service port list and property port list. 
*/
class ahl_tclClientPortMngr   
{
public:
  /* caller must allocate  amt_tclServiceData  object on heap.
   as only port knows when it will be sending the message to the target.*/
   ahl_tenPortResult enPost(amt_tclServiceData* poMsg );

   //  Inform PortManager about arrival of any new ail messages.
   ahl_tenPortResult enUpdate(const amt_tclServiceData* msg);
   
   // During state change or at any other time to clear all the pending messages call. 
   tVoid vClearAll();



   ahl_tclClientPortMngr(ail_tclAppInterfaceRestricted* pAilObj ):m_pAilObj(pAilObj){}
   
   virtual ~ahl_tclClientPortMngr();

private:

   ahl_tclClientPropertyPort* findPropertyPort( const amt_tclServiceData* poMsg );
   ahl_tclClientPropertyPort* getPropertyPort( const amt_tclServiceData* poMsg );

   ahl_tclServicePort* findServicePort(const amt_tclServiceData* poMsg );
   ahl_tclServicePort* getServicePort( const amt_tclServiceData* poMsg );

   ail_tclAppInterfaceRestricted* m_pAilObj;
   fwl_List<ahl_tclServicePort*> m_ServicePortList;
   fwl_List<ahl_tclClientPropertyPort* >m_PropertyPortList;
};


/*
 interface for ail Server application. 
 Responsibility : delegates method stream reqeusts to appropriate ahl_tclServerMethodStreamPorts
 and property reqeusts to appropriate ahl_tclServerPropertyPorts.
 implementation detail : maintains MethodStreamPort list and Property port list. 
*/

class ahl_tclServerPortMngr
{
public:
/* caller must allocate  amt_tclServiceData object on heap.
   as only port knows when it will be sending the message to the target.*/
   ahl_tenPortResult enPost(amt_tclServiceData* poMsg );

   //  Inform PortManager about arrival of any new ail messages.
   ahl_tenPortResult enUpdate(const amt_tclServiceData* poMsg);

   // During state change or at any other time to clear all the pending messages call 
   tVoid vClearAll( tBool bClearRegistration = TRUE );

   ahl_tclServerPortMngr(ail_tclAppInterfaceRestricted* pAilObj );

   virtual ~ahl_tclServerPortMngr();

private:

   ahl_tclServerPropertyPort* findPropertyPort( const amt_tclServiceData* poMsg );
   ahl_tclServerPropertyPort* getPropertyPort( const amt_tclServiceData* poMsg );

   ahl_tclServerMethodStreamPort* findMethodStreamPort(const amt_tclServiceData* poMsg );
   ahl_tclServerMethodStreamPort* getMethodStreamPort( const amt_tclServiceData* poMsg );

   ail_tclAppInterfaceRestricted* m_pAilObj;
   fwl_List<ahl_tclServerMethodStreamPort*> m_MethodStreamPortList;
   fwl_List<ahl_tclServerPropertyPort*> m_PropertyPortList;

   ahl_tclLock m_oPostLock;
};

#endif // AHL_PORTS_HEADER

