/******************************************************************************
* FILE:  ahl_ports.cpp
*
* SW-COMPONENT: CP
*
*------------------------------------------------------------------------------
* DESCRIPTION: This file provides implementation for classes used in
   Communication Port Library.
*------------------------------------------------------------------------------
*
* Author:      CM-DI/EHS4/RBIN Sreedhar
*
* COPYRIGHT:    (c) 1996 - 2002 Blaupunkt Werke GmbH
*
* HISTORY:
*      
* Date        | Author |Modification
*
* 10.Oct.2002  | Sreedhar |Initial revision
******************************************************************************/

#include "ahl_ports.h"



//#include "exh.h"

// Helper function to send ail messages.
static tBool ahl_enSend
   (
      ail_tclAppInterfaceRestricted* pAilObj,
      amt_tclServiceData* poMsg 
   )
{
   if( (pAilObj->enPostMessage(poMsg) == AIL_EN_N_NO_ERROR) )
   {
      return TRUE;
   }
   else
   {
      ahl_MsgPtr sink(poMsg);
      return  FALSE;
   }
}


template<class T>
tBool ahl_bPopFront(fwl_List<T>& rfList, T& rfElem)
{
   fwl_ListIterator<T> oIter(&rfList);
   if (oIter.bIsDone() == FALSE)
   {
      rfElem = oIter.oItem();
   }
   return rfList.bRemoveFirst();
}

// ahl_MsgPtr
ahl_MsgPtr::ahl_MsgPtr()
: amt_tclBaseMessage(), m_bUsed(FALSE)
{}

ahl_MsgPtr::ahl_MsgPtr(const amt_tclBaseMessage* poMsg)
: amt_tclBaseMessage(*poMsg), m_bUsed(TRUE)
{}

ahl_MsgPtr::ahl_MsgPtr(const ahl_MsgPtr& oRHS)
: amt_tclBaseMessage(oRHS), m_bUsed(oRHS.m_bUsed)
{
   const_cast<ahl_MsgPtr&>(oRHS).m_bUsed = FALSE;
}

ahl_MsgPtr::~ahl_MsgPtr()
{
   if (m_bUsed == TRUE)
   {
     try  /*ABR lint warning*/
     {
        vDelete();                   
     }
     catch (...)    /*ABR lint warning*/
     {
        OSAL_vAssert(FALSE);  
     }
     
   }/*end if*/
}

ahl_MsgPtr&
ahl_MsgPtr::operator=(ahl_MsgPtr& oRHS)
{
   if (&oRHS != this)
   {
      vDelete();
      if (oRHS.m_bUsed == TRUE)
      {
         //lint -e(530)    this is the compiler-generated assignment operator
         amt_tclMappableMessage::operator=(oRHS);
         m_bUsed = TRUE;
      }
      oRHS.m_bUsed = FALSE;
   }
   return *this;
}

amt_tclBaseMessage*
ahl_MsgPtr::release() const
{
   ahl_MsgPtr* me = (ahl_MsgPtr*)this;
   me->m_bUsed = FALSE;
   return me;
}

tVoid
ahl_MsgPtr::vDelete()
{
   if ((m_bUsed == TRUE) && (bIsValid() == TRUE))
   {
      bDelete();
   }
   m_bUsed = FALSE;
}


// ahl_tclMethodPort
ahl_tclMethodPort::ahl_tclMethodPort(amt_tclServiceData* poMsg)
 : m_oMsg(poMsg),m_u16FunctionID(0)
{
   if( poMsg)
   {
      m_u16FunctionID = poMsg->u16GetFunctionID();
   }
}



tVoid ahl_tclMethodPort::vStore(amt_tclServiceData* poMsg)
{
   ahl_MsgPtr omsg(poMsg);
   m_oMsg = omsg;
}


tBool ahl_tclMethodPort::bIsYourMsg(const amt_tclServiceData* poMsg) const
{
   return m_u16FunctionID == poMsg->u16GetFunctionID();
}



tVoid ahl_tclMethodPort::vGetMsg( ahl_MsgPtr& oMsg)
{
   oMsg = m_oMsg;
}



ahl_tclServicePort::~ahl_tclServicePort()
{
   ahl_tclMethodPort* pPort;   
    try  /*ABR lint warning*/
    {
      while( ahl_bPopFront(m_methodPortsList, pPort) )      
       delete pPort;                                      
    }
    catch (...)    /*ABR lint warning*/
    {
      OSAL_vAssert(FALSE);   
    }
    
}



tBool ahl_tclServicePort::bIsYourMsg(const amt_tclServiceData* poMsg) const
{
   return m_u16ServiceId == poMsg->u16GetServiceID();
}


ahl_tclMethodPort* ahl_tclServicePort::findPort
   (
      const amt_tclServiceData* poMsg,
      fwl_List<ahl_tclMethodPort*>& list
   ) const
{
   if( list.nSize() == 0)
      return 0;

   fwl_ListIterator<ahl_tclMethodPort* > Iterator( &list);
   for( Iterator.vFirst();! Iterator.bIsDone();Iterator.vNext() )
   {
      ahl_tclMethodPort* poPort = Iterator.oItem();
      if( poPort->bIsYourMsg(poMsg) )
         return poPort;
   }
   return 0;
}



/*
Algorithm :
 when enPost is called on a service port, enPost asks this function 
 to get a port associated with the service id and function id in the
 message.
 this function first searches in the pending methods queue to check
 whether a similar message request exists earlier in the queue, if so
 the associated method port is returned.
 if the port is not found, then it is searched in m_methodPortsList to
 check whether there exists a method port object to store this message.
 if the port does not exist in the method port list then a new method port
 is created and added to both pending methods q and method ports list 
 and the port is returned.
*/
ahl_tclMethodPort* ahl_tclServicePort::getPort(amt_tclServiceData* poMsg)
{
   ahl_tclMethodPort* poPort = 0;
   // does port exist in pending methodsQ
   poPort = findPort(poMsg,m_pendingMethodsQ);
   if( ! poPort) 
   {  // does port exist in methodports list.
      poPort = findPort(poMsg,m_methodPortsList);
      if( ! poPort)
      {  // create new default port.
         poPort = new ahl_tclMethodPort(poMsg);
         if( !poPort )
         {
            OSAL_vAssert( FALSE ); // low memory 
            return poPort;
         }
         m_methodPortsList.vAdd(poPort);
      }
      m_pendingMethodsQ.vAdd(poPort);
   }
   return poPort;
}



ahl_tenPortResult ahl_tclServicePort::enPost
   (
      ail_tclAppInterfaceRestricted* pAilObj,
      amt_tclServiceData* poMsg 
   )
{
   if( m_bPortClosed )
   {
      ahl_tclMethodPort* poPort = getPort(poMsg);
      if( ! poPort)
         return AHL_E_LOW_MEMORY;
      poPort->vStore(poMsg); 
      return AHL_S_MSG_BLOCKED;
   }
   else 
   {
      // handle the flag before the message is sent!!
      m_bPortClosed = TRUE;
      if( ahl_enSend(pAilObj,poMsg) )
      {
         return AHL_S_POST_OK ;
      }
      else
      {
        // restore the flag, if the message could not be sent
        m_bPortClosed = FALSE;
        return  AHL_E_POST_FAILED;
      }
   }
}


ahl_tenPortResult ahl_tclServicePort::enSendAck
   (
      ail_tclAppInterfaceRestricted* pAilObj,
      const amt_tclServiceData* poMsg 
   )
{
   amt_tclServiceDataMsg0 oServiceData;
   oServiceData.vInitServiceData
                  (     poMsg->u16GetTargetAppID(),  // Source,
                        poMsg->u16GetSourceAppID(),  // Target,
                        AMT_C_U8_CCAMSG_STREAMTYPE_NODATA, // StreamType
                        0,                           // StreamCounter
                        poMsg->u16GetRegisterID(),   // RegisterID
                        0,                           // nCmdCounter, 
                        poMsg->u16GetServiceID(),    // nServiceID, 
                        poMsg->u16GetFunctionID(),   // nFunctionID, 
                        AMT_C_U8_CCAMSG_OPCODE_CLIENT_PORT_ACK,// opcode 
                        0,                           // ACT
                        poMsg->u16GetTargetSubID(),  // swap SubIDs
                        poMsg->u16GetSourceSubID()
                  );
   return enPost(pAilObj, &oServiceData);
}



ahl_tenPortResult ahl_tclServicePort::enSendNextPendingMsg
   (
      ail_tclAppInterfaceRestricted* pAilObj
   )
{
   ahl_tclMethodPort* poPort;
   if( ahl_bPopFront(m_pendingMethodsQ, poPort) )
   {
      ahl_MsgPtr oMsg;
      poPort->vGetMsg(oMsg); 
      amt_tclServiceData oSData(oMsg.release());
      return enPost(pAilObj, &oSData);
   }
   else // m_pendingMethodsQ is empty. fine
      return AHL_S_UPDATE_OK; // no messages left to send.
}



ahl_tenPortResult ahl_tclServicePort::enUpdate
   (
      ail_tclAppInterfaceRestricted* pAilObj,
      const amt_tclServiceData* poMsg 
   )
{
   tU8 u8OpCode = poMsg->u8GetOpCode();
   tU8 u8StreamType = poMsg->u8GetStreamType();

   m_bPortClosed = FALSE;// open the port

   if( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_METHODRESULT )
   {
      if(  u8StreamType  == AMT_C_U8_CCAMSG_STREAMTYPE_NODATA   
           || u8StreamType  == AMT_C_U8_CCAMSG_STREAMTYPE_LASTDATA
           || u8StreamType  == AMT_C_U8_CCAMSG_STREAMTYPE_ABORTED
        )
      {  
         return enSendNextPendingMsg(pAilObj);
      }
      else
      if( u8StreamType  == AMT_C_U8_CCAMSG_STREAMTYPE_FIRSTDATA
          || u8StreamType  == AMT_C_U8_CCAMSG_STREAMTYPE_MIDDLEDATA )
      {  // method stream port 
         return enSendAck(pAilObj,poMsg);
      }
      else // unknown streamtypes 
      {
          // what to do ??
          return AHL_S_UPDATE_OK; 
      }
   }
   else 
   if( 
        u8OpCode == AMT_C_U8_CCAMSG_OPCODE_ABORTRESULT 
        ||
        u8OpCode == AMT_C_U8_CCAMSG_OPCODE_ERROR 
     )
   {
       return enSendNextPendingMsg(pAilObj);
   }
   else // unknown and unexpected optype 
   {
      return AHL_E_UNKNOWN_MSG;
   }
}



tVoid ahl_tclServicePort::vClear()
{
   m_pendingMethodsQ.vClear();
}


tBool ahl_tclServerMethodStreamPort::bIsYourMsg(const amt_tclServiceData* poMsg) const
{
   return m_u16RegisterId == poMsg->u16GetRegisterID()
          && m_u16FunctionId == poMsg->u16GetFunctionID();
}


ahl_tenPortResult ahl_tclServerMethodStreamPort::enPost
   (
      ail_tclAppInterfaceRestricted* pAilObj,
      amt_tclServiceData* poMsg 
   )
{
   
   if( m_bPortClosed )
   {
      ahl_MsgPtr oMsg(poMsg);
      m_pendingAnswersQ.vAdd(oMsg);
      return AHL_S_MSG_BLOCKED;
   }
   else
   {
      tU8 u8StreamType = poMsg->u8GetStreamType();

      // handle the flag before the message is sent!!
      if( u8StreamType != AMT_C_U8_CCAMSG_STREAMTYPE_LASTDATA )
      {
         m_bPortClosed = TRUE;
      }

      if( ahl_enSend(pAilObj,poMsg) )
      {
         return AHL_S_POST_OK;
      }
      else
      {
         // restore the flag, if the message could not be sent
         m_bPortClosed = FALSE;
         return  AHL_E_POST_FAILED;
      }
   }
}


ahl_tenPortResult ahl_tclServerMethodStreamPort::enUpdate
   (
      ail_tclAppInterfaceRestricted* pAilObj,
      const amt_tclServiceData* poMsg 
   )
{
   tU8 u8OpCode = poMsg->u8GetOpCode();
   //tU8 u8StreamType = poMsg->u8GetStreamType();   ABR Lint Warning
   ahl_tenPortResult result = AHL_E_UNKNOWN_MSG;

   if( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_CLIENT_PORT_ACK )
   {
      m_bPortClosed = FALSE;
      ahl_MsgPtr oMsg;
      if( ahl_bPopFront(m_pendingAnswersQ, oMsg) ) // if there is a pending msg post it.
      {
         amt_tclServiceData oSData(oMsg.release());
         result = enPost(pAilObj, &oSData);
      }
      
      if( m_pendingAnswersQ.nSize() == 0 ) 
      {
         // m_pendingAnswersQ is empty. fine, now time to delete this object.
         result = AHL_S_CAN_DELETE_ME;
      }
   }
   return result;
}


// can be called when client requests method stream abort.
tVoid ahl_tclServerMethodStreamPort::vClear()
{
   m_pendingAnswersQ.vClear();
   m_bPortClosed = FALSE; 
}


// ahl_tclClientPropertyPort 
ahl_tclClientPropertyPort::ahl_tclClientPropertyPort(tU16 u16ServiceId,tU16 u16FunctionId)
    :m_oState(INITIALISED),
    m_u16ServiceId(u16ServiceId),
    m_u16FunctionId(u16FunctionId){}  //ABR Compiler warning




tBool ahl_tclClientPropertyPort::bIsYourMsg(const amt_tclServiceData* poMsg) const
{
   return m_u16ServiceId == poMsg->u16GetServiceID()
          && m_u16FunctionId == poMsg->u16GetFunctionID();
}


tVoid ahl_tclClientPropertyPort::vStoreSetGetMsg( amt_tclServiceData* poMsg )
{
   ahl_MsgPtr msg(poMsg);
   m_oSetGetMsg = msg; 
}


ahl_tenPortResult ahl_tclClientPropertyPort::enSendAck
   (
      ail_tclAppInterfaceRestricted* pAilObj,
      const amt_tclServiceData* poMsg 
   )
{
   amt_tclServiceDataMsg0 oServiceData;
   oServiceData.vInitServiceData
                  (     poMsg->u16GetTargetAppID(),    // Source,
                        poMsg->u16GetSourceAppID(),    // Target,
                        AMT_C_U8_CCAMSG_STREAMTYPE_NODATA, // StreamType
                        0,                             // StreamCounter
                        poMsg->u16GetRegisterID(),     // RegisterID
                        AHL_UPREG_CMD_COUNTER,         // nCmdCounter, 
                        poMsg->u16GetServiceID(),      // nServiceID, 
                        poMsg->u16GetFunctionID(),     // nFunctionID, 
                        AMT_C_U8_CCAMSG_OPCODE_CLIENT_PORT_ACK,  // opcode 
                        0,                             // ACT
                        poMsg->u16GetTargetSubID(),    // swap SubIDs
                        poMsg->u16GetSourceSubID()
                  );
   return enPost(pAilObj, &oServiceData);
}




ahl_tenPortResult ahl_tclClientPropertyPort::enPost
   (
      ail_tclAppInterfaceRestricted* pAilObj,
      amt_tclServiceData* poMsg 
   )
{
   tU8 u8OpCode = poMsg->u8GetOpCode();   
   if( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_SET )
   { // set request
      if( m_oState.bIsSet(WAITING_FOR_SET_GET_STATUS ) )
      {
         vStoreSetGetMsg( poMsg);
         m_oState.vAppend( SET_MSG_PENDING ); 
         return AHL_S_MSG_BLOCKED;
      }
      else // no waiting post it.
      {
         // handle the flag before the message is sent!!
         m_oState.vAppend( WAITING_FOR_SET_GET_STATUS );

         if( ahl_enSend(pAilObj,poMsg) )
         {
            return AHL_S_POST_OK ;
         }
         else
         {
            // restore the flag, if the message could not be sent
            m_oState.vRemove( WAITING_FOR_SET_GET_STATUS );
            return  AHL_E_POST_FAILED;
         }
      }
   }
   else
   if( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_GET )
   {  // get reqeust
      if( m_oState.bIsSet(WAITING_FOR_SET_GET_STATUS ) )
      {  // is there a pending set message then ignore get message.
         if( m_oState.bIsSet( SET_MSG_PENDING ) )
         {
            ahl_MsgPtr sink(poMsg); // delete the message
            return  AHL_E_POST_FAILED;
         }
         else
         { // no pending get message so store get message.
            vStoreSetGetMsg( poMsg);
            m_oState.vAppend( GET_MSG_PENDING);
            return AHL_S_MSG_BLOCKED;
         }
      }
      else // no waiting post it.
      {
         // handle the flag before the message is sent!!
         m_oState.vAppend( WAITING_FOR_SET_GET_STATUS );

         if( ahl_enSend(pAilObj,poMsg) )
         {
            return AHL_S_POST_OK;
         }
         else
         {
            // restore the flag, if the message could not be sent
            m_oState.vRemove( WAITING_FOR_SET_GET_STATUS );
            return  AHL_E_POST_FAILED;
         }
      }
   }
   else
   if( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_UPREG )
   {
      if( m_oState.bIsSet( WAITING_FOR_UPREG_STATUS ) )
      { // port already sent one UpReg msg to server
        // so deleted the message.
         ahl_MsgPtr sink(poMsg);
         return  AHL_S_UPREG_MSG_DELETED;
      }
      
      // handle the flag before the message is sent!!
      m_oState.vAppend( WAITING_FOR_UPREG_STATUS );

      if( ahl_enSend(pAilObj,poMsg) )
      {
         return AHL_S_POST_OK;
      }
      else
      {
         // restore the flag, if the message could not be sent
         m_oState.vRemove( WAITING_FOR_UPREG_STATUS );
         return  AHL_E_POST_FAILED;
      }
   }
   else
   if( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_CLIENT_PORT_ACK  )
   {
      return ahl_enSend(pAilObj,poMsg) ? AHL_S_POST_OK : AHL_E_POST_FAILED;
   }
   else
   if( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_RELUPREG )
   {
      if( ahl_enSend(pAilObj,poMsg) )
      {
         m_oState.vRemove( WAITING_FOR_UPREG_STATUS );
         return AHL_S_POST_OK;
      }
      else
      {
         return  AHL_E_POST_FAILED;
      }
   }

   return AHL_E_UNKNOWN_MSG;
}


ahl_tenPortResult ahl_tclClientPropertyPort::enUpdate
   (
      ail_tclAppInterfaceRestricted* pAilObj,
      const amt_tclServiceData* poMsg 
   )
{
   tU8 u8OpCode         = poMsg->u8GetOpCode(); 
   tU16 u16CmdCounter   = poMsg->u16GetCmdCounter();
   if(  
        u8OpCode == AMT_C_U8_CCAMSG_OPCODE_STATUS
        || 
        u8OpCode == AMT_C_U8_CCAMSG_OPCODE_ERROR
     )
   {
      if( u16CmdCounter == AHL_UPREG_CMD_COUNTER )
      { // this is an UpReg status message 
         if( m_oState.bIsSet( WAITING_FOR_UPREG_STATUS ) )
         {
         	  if( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_ERROR )
         	  {
         	  	 m_oState.vRemove( WAITING_FOR_UPREG_STATUS );
         	  }
         	  
            return enSendAck(pAilObj,poMsg);
         }
         else if( m_oState.bIsSet( INITIALISED ) )
         {
            m_oState.vAppend(WAITING_FOR_UPREG_STATUS);
            return enSendAck(pAilObj,poMsg);
         }
         else
         {  // we might have recieved this UpReg status 
            // after sending UpRegRelease message to server.
            // no need to acknowledge as we are no more interested 
            // in UpReg messages.
            return AHL_S_UPDATE_OK;
         }
      }
      else
      {  // this is a set/get status message.
         m_oState.vRemove( WAITING_FOR_SET_GET_STATUS );
         if( m_oState.bIsSet( SET_MSG_PENDING )
            || m_oState.bIsSet(GET_MSG_PENDING )
           )
         { 
            m_oState.vRemove( SET_MSG_PENDING | GET_MSG_PENDING );
            amt_tclServiceData oSData(m_oSetGetMsg.release());

            // handle the flag before the message is sent!!
            m_oState.vAppend( WAITING_FOR_SET_GET_STATUS );

            if( ahl_enSend(pAilObj, &oSData )  )
            {
               return AHL_S_POST_OK;
            }
            else
            {
               m_oState.vRemove( WAITING_FOR_SET_GET_STATUS );
               return  AHL_E_POST_FAILED;
            }
         }
         else // no pending messages, fine
            return AHL_S_UPDATE_OK;
      }
   }
   else
   {
      return AHL_E_UNKNOWN_MSG;
   }
}



tVoid ahl_tclClientPropertyPort::vClear()
{
   m_oSetGetMsg.vDelete();
   m_oState.vAssign(INITIALISED);// clear all flags.
}


// ahl_tclClientInfo

ahl_tclClientInfo::ahl_tclClientInfo(tU16 u16RegisterID,tBool bLocked)
   :m_u16RegisterID(u16RegisterID)
{
   m_oState.vAppend(INITIALISED);
   if( bLocked ) m_oState.vAppend(LOCKED);
}

ahl_tclClientInfo::~ahl_tclClientInfo()
{
}


tBool ahl_tclClientInfo::bIsYourMsg(const amt_tclServiceData* poMsg ) const 
{
   return m_u16RegisterID == poMsg->u16GetRegisterID();
}



ahl_tenPortResult ahl_tclClientInfo::enPost
(
   ail_tclAppInterfaceRestricted* pAilObj,
   amt_tclServiceData* poMsg,
   tBool bDoUnlock
)
{
   ahl_tenPortResult  enRetVal = AHL_S_POST_OK;
   tBool bPendingFlagRemoved = FALSE;

   if ( bDoUnlock )
   {
      m_oState.vRemove(LOCKED);    // remove the lock

      if (m_oState.bIsSet(PENDING_MSG_AVAILABLE))
      {
         // pending (=stored) message must be sent
         m_oState.vRemove(PENDING_MSG_AVAILABLE);
      }
      else
      {
         // nothing to send
         enRetVal = AHL_S_UPDATE_OK;
      }
   }

   if ( AHL_S_POST_OK == enRetVal )
   {
      if( m_oState.bIsSet(LOCKED) )
      {
         vStore( poMsg);
         m_oState.vAppend(PENDING_MSG_AVAILABLE);
         enRetVal = AHL_S_MSG_BLOCKED;
      }
      else
      {
         amt_tclServiceData oStoredServiceData(m_oMsg.release());

         if ( NULL == poMsg )
         {
            // use the stored message for sending
            poMsg = &oStoredServiceData;
         }

         // handle the flags before the message is sent!!
         m_oState.vAppend(LOCKED);

         if( m_oState.bIsSet(PENDING_MSG_AVAILABLE) )
         {
            m_oState.vRemove(PENDING_MSG_AVAILABLE);
            bPendingFlagRemoved = TRUE;
         }

         if( ahl_enSend(pAilObj,poMsg) )
         {
            enRetVal = AHL_S_POST_OK;
         } 
         else
         {
            // restore the flags, if the message could not be sent
            m_oState.vRemove(LOCKED);
            if ( bPendingFlagRemoved )
            {
               m_oState.vAppend(PENDING_MSG_AVAILABLE);
            }
            enRetVal = AHL_E_POST_FAILED;
         }
      }
   }

   return enRetVal;
}


ahl_tenPortResult ahl_tclClientInfo::enUpdatePendingMsg
(
   ail_tclAppInterfaceRestricted* pAilObj
)
{
   // do an unlock and send the stored message
   return enPost( pAilObj, NULL, TRUE );
}

tVoid ahl_tclClientInfo::vClear()
{
   m_oState.vRemove(LOCKED | PENDING_MSG_AVAILABLE);
}



tBool ahl_tclServerPropertyPort::bIsYourMsg(const amt_tclServiceData* poMsg) const
{
   return m_u16ServiceId == poMsg->u16GetServiceID()
          && m_u16FunctionId == poMsg->u16GetFunctionID();
}



/* server code calls enPost as many times as the number of clients
  this function iterates through the client list and
  updates only if the specific client is unlocked.
*/
ahl_tenPortResult ahl_tclServerPropertyPort::enPost
   (
      ail_tclAppInterfaceRestricted* pAilObj,
      amt_tclServiceData* poMsg 
   )
{
   fwl_ListIterator<ahl_tclClientInfo> Iterator( &m_RegisteredClients);
   for( Iterator.vFirst();! Iterator.bIsDone();Iterator.vNext() )
   {
      ahl_tclClientInfo* poClientInfo = &Iterator.oItem();
      if( poClientInfo->bIsYourMsg(poMsg) )
      {
         return poClientInfo->enPost(pAilObj,poMsg);
      }
   }

   // property port without a client.
   return AHL_S_NO_UPREG_CLIENT;
}


ahl_tenPortResult ahl_tclServerPropertyPort::enUpdate
   (
      ail_tclAppInterfaceRestricted* pAilObj,
      const amt_tclServiceData* poMsg 
   )
{
   tU8 u8OpCode = poMsg->u8GetOpCode();
  //tU8 u8StreamType = poMsg->u8GetStreamType();  ABR Lint Warning

   if( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_CLIENT_PORT_ACK )
   {
      fwl_ListIterator<ahl_tclClientInfo> Iterator( &m_RegisteredClients);
      for( Iterator.vFirst();! Iterator.bIsDone();Iterator.vNext() )
      {
         ahl_tclClientInfo* poClientInfo = &Iterator.oItem();
         if( poClientInfo->bIsYourMsg(poMsg) )
         {
            return poClientInfo->enUpdatePendingMsg(pAilObj);
         }
      }

      return AHL_E_UNKNOWN_MSG;
   }
   else
   if( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_UPREG )
   {
      ahl_tclClientInfo clientInfo(poMsg->u16GetRegisterID());
      m_RegisteredClients.vAdd(clientInfo);
      return AHL_S_UPDATE_OK;
   }
   else
   if( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_RELUPREG )
   {
      ahl_tclClientInfo oClientInfo(poMsg->u16GetRegisterID());
      m_RegisteredClients.nRemove(oClientInfo);
      return AHL_S_UPDATE_OK;  // client unlocked
   }

      return AHL_E_UNKNOWN_MSG;
}



tVoid ahl_tclServerPropertyPort::vClear()
{
   m_RegisteredClients.vClear();
}




ahl_tclClientPortMngr::~ahl_tclClientPortMngr()
{
   ahl_tclServicePort* poPort;  

   try  /*ABR lint warning*/
   {
     while( ahl_bPopFront(m_ServicePortList, poPort) )                    
        delete poPort;                                                    

     ahl_tclClientPropertyPort* poPropertyPort;
     while( ahl_bPopFront(m_PropertyPortList, poPropertyPort ) )          
        delete poPropertyPort;                                            
   }
   catch (...)    /*ABR lint warning*/
   {
     OSAL_vAssert(FALSE);   
   }
   
   m_pAilObj=NULL;                /*ABR LINT Warning*/
}


/*
  clear all pending method or method stream requests and 
  pending set/get requests.
*/
tVoid ahl_tclClientPortMngr::vClearAll()
{
   fwl_ListIterator<ahl_tclServicePort* > Iterator( &m_ServicePortList);
   for( Iterator.vFirst();! Iterator.bIsDone();Iterator.vNext() )
   {
      ahl_tclServicePort* poPort = Iterator.oItem();
      poPort->vClear();
   }

   fwl_ListIterator<ahl_tclClientPropertyPort* > It( &m_PropertyPortList);
   for( It.vFirst();! It.bIsDone();It.vNext() )
   {
      ahl_tclClientPropertyPort* poPort = It.oItem();
      poPort->vClear();
   }
}



// ahl_tclClientPortMngr
ahl_tclServicePort* ahl_tclClientPortMngr::findServicePort
   (
      const amt_tclServiceData* poMsg
   )
{
   if( m_ServicePortList.nSize() == 0)
      return 0;

   fwl_ListIterator<ahl_tclServicePort* > Iterator( &m_ServicePortList);
   for( Iterator.vFirst();! Iterator.bIsDone();Iterator.vNext() )
   {
      ahl_tclServicePort* poPort = Iterator.oItem();
      if( poPort->bIsYourMsg(poMsg) )
         return poPort;
   }

   return 0;
}


ahl_tclServicePort* ahl_tclClientPortMngr::getServicePort
   (
      const amt_tclServiceData* poMsg
   )
{
   ahl_tclServicePort* poPort = findServicePort(poMsg);
   if( ! poPort)
   {
      poPort = new ahl_tclServicePort(poMsg->u16GetServiceID() );
      if( poPort)
         m_ServicePortList.vAdd(poPort);
      else
         OSAL_vAssert( FALSE);   // low memory
   }

   return poPort;
}



ahl_tclClientPropertyPort* ahl_tclClientPortMngr::findPropertyPort
   (
      const amt_tclServiceData* poMsg
   )
{
   if( m_PropertyPortList.nSize() == 0)
      return 0;

   fwl_ListIterator<ahl_tclClientPropertyPort* > Iterator( &m_PropertyPortList);
   for( Iterator.vFirst();! Iterator.bIsDone();Iterator.vNext() )
   {
      ahl_tclClientPropertyPort* poPort = Iterator.oItem();
      if( poPort->bIsYourMsg(poMsg) )
         return poPort;
   }

   return 0;
}


ahl_tclClientPropertyPort* ahl_tclClientPortMngr::getPropertyPort
   (
      const amt_tclServiceData* poMsg
   )
{
   ahl_tclClientPropertyPort* poPort = findPropertyPort(poMsg);
   if( ! poPort)
   {
      poPort = new ahl_tclClientPropertyPort(poMsg->u16GetServiceID(),poMsg->u16GetFunctionID() );
      if( poPort)
         m_PropertyPortList.vAdd(poPort);
      else
         OSAL_vAssert( FALSE);   // low memory
   }
   return poPort;
}



ahl_tenPortResult ahl_tclClientPortMngr::enPost(amt_tclServiceData* poMsg )
{
   tU8 u8OpCode = poMsg->u8GetOpCode();
   if( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_METHODSTART 
       || u8OpCode == AMT_C_U8_CCAMSG_OPCODE_METHODABORT
     )
   {  //  service port message 
      ahl_tclServicePort* poPort = getServicePort(poMsg);
      if( ! poPort)
         return AHL_E_LOW_MEMORY;
      return poPort->enPost(m_pAilObj,poMsg);
   }
   else
   if( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_SET
       || u8OpCode == AMT_C_U8_CCAMSG_OPCODE_GET
       || u8OpCode == AMT_C_U8_CCAMSG_OPCODE_UPREG
       || u8OpCode == AMT_C_U8_CCAMSG_OPCODE_RELUPREG
     )
   {   //  property port message 
      ahl_tclClientPropertyPort* poPort = getPropertyPort(poMsg);
      if( ! poPort)
         return AHL_E_LOW_MEMORY;
      return poPort->enPost(m_pAilObj,poMsg);
   }
   else  // unknown opCode ???
   {
      return AHL_E_UNKNOWN_MSG;
   }
}


ahl_tenPortResult ahl_tclClientPortMngr::enUpdate(const amt_tclServiceData* poMsg)
{
   tU8 u8OpCode = poMsg->u8GetOpCode();

   if(  u8OpCode  == AMT_C_U8_CCAMSG_OPCODE_METHODRESULT 
       || u8OpCode  == AMT_C_U8_CCAMSG_OPCODE_ABORTRESULT )
   {
      ahl_tclServicePort* poPort = findServicePort(poMsg);
      if( poPort)
      {
         return poPort->enUpdate(m_pAilObj,poMsg);
      }
      else // message without port on client side ? how come ??
      {
         return AHL_E_UNKNOWN_MSG;
      }
   }
   else 
   if(  u8OpCode  == AMT_C_U8_CCAMSG_OPCODE_STATUS )
   {  // property port 
      ahl_tclClientPropertyPort* poPort = findPropertyPort(poMsg);
      if( poPort)
      {
         return poPort->enUpdate(m_pAilObj,poMsg);
      }
      else // message without port on client side ? how come ??
      {
         return AHL_E_UNKNOWN_MSG;
      }
   }
   else 
   if(  u8OpCode == AMT_C_U8_CCAMSG_OPCODE_ERROR )
   {  // test, if property port is waiting for this function ID 
      ahl_tclClientPropertyPort* poPropertyPort = findPropertyPort(poMsg);
      if( poPropertyPort)
      {
         return poPropertyPort->enUpdate(m_pAilObj,poMsg);
      }
      else // test, if service port is waiting for this function ID
      {
         ahl_tclServicePort* poServicePort = findServicePort(poMsg);
         if( poServicePort)
         {
            return poServicePort->enUpdate(m_pAilObj,poMsg);
         }
         else // error message without port on client side ? how come ??
         {
            return AHL_E_UNKNOWN_MSG;
         }
      }
   }
   else  // unknown opCode ???
   {
         return AHL_E_UNKNOWN_MSG;
   }
}


ahl_tclServerPortMngr::ahl_tclServerPortMngr( ail_tclAppInterfaceRestricted* pAilObj )
 : m_pAilObj(pAilObj)
{
   (tVoid)m_oPostLock.bInit();
}


ahl_tclServerPortMngr::~ahl_tclServerPortMngr()
{
   ahl_tclServerMethodStreamPort* pPort;
   try  /*ABR lint warning*/
   {
     while( ahl_bPopFront(m_MethodStreamPortList, pPort) )     
        delete pPort;                                          

     ahl_tclServerPropertyPort* pPropertyPort;   
     while( ahl_bPopFront(m_PropertyPortList, pPropertyPort) )   
        delete pPropertyPort;                                     
   }
   catch (...)    /*ABR lint warning*/
   {
      OSAL_vAssert(FALSE);   
   }
   

   (tVoid)m_oPostLock.bDestroy();

   m_pAilObj = NULL;
}


ahl_tclServerPropertyPort* ahl_tclServerPortMngr::findPropertyPort(const amt_tclServiceData* poMsg )
{
   if( m_PropertyPortList.nSize() == 0)
      return 0;

   fwl_ListIterator<ahl_tclServerPropertyPort* > Iterator( &m_PropertyPortList);
   for( Iterator.vFirst();! Iterator.bIsDone();Iterator.vNext() )
   {
      ahl_tclServerPropertyPort* poPort = Iterator.oItem();
      if( poPort->bIsYourMsg(poMsg) )
         return poPort;
   }
   return 0;
}


ahl_tclServerPropertyPort* ahl_tclServerPortMngr::getPropertyPort( const amt_tclServiceData* poMsg )
{
   ahl_tclServerPropertyPort* poPort = findPropertyPort(poMsg);
   if( ! poPort)
   {
      poPort = new ahl_tclServerPropertyPort(poMsg->u16GetServiceID(),poMsg->u16GetFunctionID() );
      if( poPort)
         m_PropertyPortList.vAdd(poPort);
      else
         OSAL_vAssert( FALSE);  // low memory
   }
   return poPort;
}


ahl_tclServerMethodStreamPort* ahl_tclServerPortMngr::findMethodStreamPort(const amt_tclServiceData* poMsg )
{
   if( m_MethodStreamPortList.nSize() == 0)
      return 0;

   fwl_ListIterator<ahl_tclServerMethodStreamPort* > Iterator( &m_MethodStreamPortList);
   for( Iterator.vFirst();! Iterator.bIsDone();Iterator.vNext() )
   {
      ahl_tclServerMethodStreamPort* poPort = Iterator.oItem();
      if( poPort->bIsYourMsg(poMsg) )
         return poPort;
   }

   return 0;
}



ahl_tclServerMethodStreamPort* ahl_tclServerPortMngr::getMethodStreamPort( const amt_tclServiceData* poMsg )
{
   ahl_tclServerMethodStreamPort* poPort = findMethodStreamPort(poMsg);
   if( ! poPort)
   {
      poPort = new ahl_tclServerMethodStreamPort(poMsg->u16GetRegisterID(),poMsg->u16GetFunctionID() );
      if( poPort)
         m_MethodStreamPortList.vAdd(poPort);
      else
         OSAL_vAssert( FALSE);  // low memory
   }

   return poPort;
}


ahl_tenPortResult ahl_tclServerPortMngr::enPost(amt_tclServiceData* poMsg )
{
   tU8 u8OpCode = poMsg->u8GetOpCode();
   tU8 u8StreamType = poMsg->u8GetStreamType();

   if( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_METHODRESULT )
   { 
      if( u8StreamType  == AMT_C_U8_CCAMSG_STREAMTYPE_NODATA )
      {   // for method result no need to create a port
          // send it.
         return ahl_enSend(m_pAilObj,poMsg) ? AHL_S_POST_OK : AHL_E_POST_FAILED;
      }
      else
      if( ( u8StreamType  == AMT_C_U8_CCAMSG_STREAMTYPE_FIRSTDATA )
          ||
          ( u8StreamType  == AMT_C_U8_CCAMSG_STREAMTYPE_MIDDLEDATA )
          ||
          ( u8StreamType  == AMT_C_U8_CCAMSG_STREAMTYPE_LASTDATA )
        )
      { // method stream object
         ahl_tclServerMethodStreamPort* poPort = getMethodStreamPort(poMsg);
         if( ! poPort)
            return AHL_E_LOW_MEMORY;
         return poPort->enPost(m_pAilObj,poMsg); 
      }
   }
   else
   if( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_STATUS )
   { // this is a property message.
      tU16 u16CmdCounter   = poMsg->u16GetCmdCounter(); 
      if( u16CmdCounter == AHL_UPREG_CMD_COUNTER )
      {  // this is an UpReg message.
         ahl_tclServerPropertyPort* poPort = getPropertyPort(poMsg);
         if( ! poPort)
            return AHL_E_LOW_MEMORY;

         ahl_tenPortResult enPortResult;
         tBool bLockResult;
         tBool bUnlockResult;

         bLockResult = m_oPostLock.bLock();       // enter critical section

         enPortResult = poPort->enPost(m_pAilObj,poMsg);
         
         bUnlockResult = m_oPostLock.bUnlock();   // release critical section

         if ( bLockResult != bUnlockResult )
         {
            NORMAL_M_ASSERT_ALWAYS();
         }

         return enPortResult;
      }
      else
      {  /* this is a set/get status message.
            no need to create a port send it. */
         return ahl_enSend(m_pAilObj,poMsg) ? AHL_S_POST_OK : AHL_E_POST_FAILED;
      }
   }

   return AHL_E_UNKNOWN_MSG;
}


ahl_tenPortResult ahl_tclServerPortMngr::enUpdate(const amt_tclServiceData* poMsg)
{
   tU8 u8OpCode = poMsg->u8GetOpCode();
   //tU8 u8StreamType = poMsg->u8GetStreamType(); ABR LINT- Warning

   if( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_CLIENT_PORT_ACK )
   {
      ahl_tclServerMethodStreamPort* poPort = findMethodStreamPort(poMsg);
      if( poPort)
      {  // may be msg belongs to a method stream port.
         // delete concerned method stream port if it completes its job.
         ahl_tenPortResult result =  poPort->enUpdate(m_pAilObj,poMsg);
         if( result == AHL_S_CAN_DELETE_ME )
            m_MethodStreamPortList.nRemove( poPort);
         return result;
      }
      else
      {  // may be, msg belongs to a Property port.
         ahl_tclServerPropertyPort* poPortS = findPropertyPort(poMsg); /*ABR LINT Warning*/
         if( poPortS)
         {
            ahl_tenPortResult enPortResult;
            tBool bLockResult;
            tBool bUnlockResult;

            bLockResult = m_oPostLock.bLock();       // enter critical section

            enPortResult = poPortS->enUpdate(m_pAilObj,poMsg);

            bUnlockResult = m_oPostLock.bUnlock();   // release critical section

            if ( bLockResult != bUnlockResult )
            {
               NORMAL_M_ASSERT_ALWAYS();
            }
            
            return enPortResult;
         }
      }
   }
   else
   if( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_UPREG 
       || u8OpCode == AMT_C_U8_CCAMSG_OPCODE_RELUPREG
     )
   {
      ahl_tclServerPropertyPort* poPort = getPropertyPort(poMsg);
      if( ! poPort)
         return AHL_E_LOW_MEMORY;
      return poPort->enUpdate(m_pAilObj,poMsg); 
   }
   else
   if( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_SET
       || u8OpCode == AMT_C_U8_CCAMSG_OPCODE_GET
       || u8OpCode == AMT_C_U8_CCAMSG_OPCODE_METHODSTART
     )
   {
      // do nothing
      return AHL_S_UPDATE_OK;
   }
   else
   if ( u8OpCode == AMT_C_U8_CCAMSG_OPCODE_METHODABORT )
   {
      // delete the concerned method stream port object. 
      ahl_tclServerMethodStreamPort* poPort = findMethodStreamPort(poMsg);
      if( poPort)
      {
         m_MethodStreamPortList.nRemove( poPort);
         return AHL_S_UPDATE_OK;
      }
   }

   return AHL_E_UNKNOWN_MSG;
}


/*
  clear any pending methodstream pending answers and remove all registered clients
  for each UpReg server.
*/
tVoid ahl_tclServerPortMngr::vClearAll( tBool bClearRegistration )
{
   fwl_ListIterator<ahl_tclServerMethodStreamPort* > Iterator( &m_MethodStreamPortList);
   for( Iterator.vFirst();! Iterator.bIsDone();Iterator.vNext() )
   {
      ahl_tclServerMethodStreamPort* poPort = Iterator.oItem();
      poPort->vClear();
   }

   if( bClearRegistration )
   {
      fwl_ListIterator<ahl_tclServerPropertyPort* > It( &m_PropertyPortList);
      for( It.vFirst();! It.bIsDone();It.vNext() )
      {
         ahl_tclServerPropertyPort* poPort = It.oItem();
         poPort->vClear();
      }
   }
}

