/******************************************************************************
 * $Revision:   1.3  $
 * $Date:   01 Mar 2004 15:32:56  $
 * A changelog can be found at the end of this file.
 ******************************************************************************
 * FILE:         ahl_CCANotificationTable.cpp
 * SW-COMPONENT: AHL
 * DESCRIPTION:  A specialised notification table
 * AUTHOR:       CM-CR/EES4-Grosse Pawig
 * COPYRIGHT:    (c) 2003 Blaupunkt Werke GmbH
 * HISTORY:
 * 22.07.03      Rev 1.0
 *               Initial revision
 *****************************************************************************/

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AMT_S_IMPORT_INTERFACE_GENERIC
#include "amt_if.h"

#include "ahl_ccaNotificationTable.h"

ahl_tclCCANotificationTable::ahl_tclCCANotificationTable()
 : m_oNotTable()
 , m_hLock(OSAL_C_INVALID_HANDLE)
{
  m_szLockName[0] = '\0';
}

tBool ahl_tclCCANotificationTable::bInit(tU16 u16ApplID, tU16 u16ServiceID, tU16 u16Major, tU16 u16Minor, tU16 u16Patch)
{
  tBool bRetVal = FALSE;

  if (OSAL_C_INVALID_HANDLE == m_hLock)
  {
    (tVoid)OSALUTIL_s32NPrintFormat((tS8*)m_szLockName, sizeof(m_szLockName), "ahl_CC%04lx%04lx%04lx%04lx%04lx", 
                                    u16ApplID, u16ServiceID, u16Major, u16Minor, u16Patch);

    if (OSAL_s32SemaphoreCreate(m_szLockName, &m_hLock, 1) == OSAL_OK)
    {
      bRetVal = TRUE;
    }
  }

  return (bRetVal);
}

tBool ahl_tclCCANotificationTable::bDestroy()
{
  tBool bRetVal = FALSE;

  if (m_hLock != OSAL_C_INVALID_HANDLE)
  {
    if (OSAL_s32SemaphoreClose(m_hLock) == OSAL_OK)
    {
      if (OSAL_s32SemaphoreDelete(m_szLockName) == OSAL_OK)
      {
        bRetVal = TRUE;
      }
    }
  }

  return (bRetVal);
}

tBool ahl_tclCCANotificationTable::bLock()
{
   if ((m_hLock != OSAL_C_INVALID_HANDLE) &&
       (OSAL_s32SemaphoreWait(m_hLock, OSAL_C_TIMEOUT_FOREVER) == OSAL_OK))
   {
      return TRUE;
   }
   return FALSE;
}

tVoid ahl_tclCCANotificationTable::vUnlock()
{
   (tVoid)OSAL_s32SemaphorePost(m_hLock);
}

tBool ahl_tclCCANotificationTable::bAddNotification(const amt_tclServiceData& corfoMsg)
{
   tBool bSuccess = FALSE;

   if (bLock() == TRUE)
   {
      bSuccess =
         m_oNotTable.bAddNotification(corfoMsg.u16GetFunctionID(),
                                      corfoMsg.u16GetSourceAppID(),
                                      corfoMsg.u16GetRegisterID(),
                                      TRUE,
                                      corfoMsg.u16GetCmdCounter(),
                                      corfoMsg.u16GetSourceSubID(),
                                      corfoMsg.u8GetACT());
      
      m_FktIdMap[corfoMsg.u16GetFunctionID()]++;

      vUnlock();
   }
   return bSuccess;
}

ahl_tNotification* ahl_tclCCANotificationTable::poGetNotification(tU16 u16FktID, tU16 u16RegID)
{
   ahl_tNotification* poResult = NULL;
   ahl_tNotification* poCurrent = m_oNotTable.poGetNotificationList(u16FktID);

   while ((poCurrent != NULL) && (poResult == NULL))
   {
      if (poCurrent->u16RegisterID == u16RegID)
      {
         poResult = poCurrent;
      }
      poCurrent = poCurrent->pNext;
   }
   return poResult;
}

ahl_tNotification* ahl_tclCCANotificationTable::poGetNextNotification(tU16 u16FktID, const ahl_tNotification* poPrev)
{
   ahl_tNotification* poNext = NULL;
   
   if (poPrev != NULL)
   {
      poNext = poPrev->pNext;
   }
   else
   {
      poNext = m_oNotTable.poGetNotificationList(u16FktID);
   }
   return poNext;
}

tBool ahl_tclCCANotificationTable::bRemoveNotification(tU16 u16FktID, tU16 u16RegID)
{
   tBool bSuccess = FALSE;

   if (bLock() == TRUE)
   {
      ahl_tNotification* poNotify = poGetNotification(u16FktID, u16RegID);

      if (poNotify != NULL)
      {
         m_FktIdMap[u16FktID]--;
         bSuccess = bRemoveNotification(u16FktID, poNotify);
      }
      vUnlock();
   }
   return bSuccess;
}

tBool ahl_tclCCANotificationTable::bEraseNotifications(tU16 u16FktID, tU16 u16RegID)
{
   tBool bSuccess = FALSE;

   if (bLock() == TRUE)
   {
      ahl_tNotification* poNotify = m_oNotTable.poGetNotificationList(u16FktID);

      while (poNotify != NULL)
      {
         ahl_tNotification* poCur = poNotify;
         poNotify = poNotify->pNext;

         if (poCur->u16RegisterID ==  u16RegID)
         {
            bSuccess |= bRemoveNotification(u16FktID, poCur);
         }
      }
      vUnlock();
   }
   return bSuccess;
}

tBool ahl_tclCCANotificationTable::bRemoveNotification(tU16 u16FktID, const ahl_tNotification* poNotify)
{
   return m_oNotTable.bRemoveNotification(u16FktID,
                                          poNotify->u16AppID,
                                          poNotify->u16RegisterID,
                                          poNotify->bNotification,
                                          poNotify->u16CmdCounter,
                                          poNotify->u16SubID);
}

tBool ahl_tclCCANotificationTable::bRemoveAllEntriesWithRegID(tU16 u16RegID)
{
   return m_oNotTable.bRemoveAllEntriesWithRegID(u16RegID);
}

tVoid ahl_tclCCANotificationTable::vTraceTable(tU16 u16ClassID) const
{
   m_oNotTable.vTraceTable(u16ClassID);
}


