#ifndef AHL_TIMER_HEADER
#define AHL_TIMER_HEADER 1

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

// ahl_Callback
#include "ahl_Callback.h"

//*******************************************************************
//* CLASS:       ahl_tclTimer
//*
//* DESCRIPTION: A simple timer class.
//*
//*              This class provides a wrapper around the OSAL timer
//*              functions. It uses ahl_tclCallbackBase as its
//*              callback mechanism.
//*
//* AUTHOR:      CM-CR/EES4-Grosse Pawig
//*
//* COPYRIGHT:   (c) 2002 Blaupunkt GmbH
//*
//* HISTORY:
//* 12.07.2002 CM-CR/EES4-Grosse Pawig
//*            Initial Revision
//*******************************************************************
//##ModelId=3D2E8F49007D

class ahl_tclTimer 
{
  public:
      ahl_tclTimer();
      virtual ~ahl_tclTimer();


      //*****************************************************************
      //* FUNCTION:    bInit
      //*
      //* DESCRIPTION: Initialise the timer.
      //*
      //*              The necessary resources are allocated. On
      //*              successful return, the timer is ready to use.
      //*
      //* PARAMETER:   poCallback:  (->I)
      //*                 Pointer to a callback object that gets
      //*                 triggered on timer events.
      //*
      //* RETURNVALUE: TRUE on success, FALSE on failure.
      //*
      //* AUTHOR:      CM-CR/EES4-Grosse Pawig
      //*
      //* HISTORY:
      //* 12.07.2002 CM-CR/EES4-Grosse Pawig
      //*            Initial Revision
      //*****************************************************************
      //##ModelId=3D2E958D00AC
      tBool bInit (ahl_tclCallbackBase* poCallback);

      //*****************************************************************
      //* FUNCTION:    bDestroy
      //*
      //* DESCRIPTION: Free the resources of the timer.
      //*
      //*              On application shutdown, the resoures that have
      //*              been aqcuired by bInit() have to be freed by a
      //*              call to bDestroy().
      //*
      //* PARAMETER:   -
      //*
      //* RETURNVALUE: TRUE if the resources have successfully been
      //*              freed, FALSE otherwise (e.g. if the timer had not
      //*              been initialised at all).
      //*
      //* AUTHOR:      CM-CR/EES4-Grosse Pawig
      //*
      //* HISTORY:
      //* 12.07.2002 CM-CR/EES4-Grosse Pawig
      //*            Initial Revision
      //*****************************************************************
      //##ModelId=3D2E95930141
      tBool bDestroy ();

      //*****************************************************************
      //* FUNCTION:    bStart
      //*
      //* DESCRIPTION: Start the timer.
      //*
      //*              If the timer is currently active, the previous
      //*              settings will be overridden.
      //*
      //* PARAMETER:   msec:     (I)
      //*                 Number of milli seconds after which the timer
      //*                 should expire.
      //*
      //*              interval: (I)   (optional)
      //*                 If > 0, the timer will expire periodically
      //*                 every 'interval' milli seconds. The first tick
      //*                 happens after 'msec' milli seconds.
      //*                 Default: 0
      //*
      //* RETURNVALUE: TRUE on success, FALSE on failure.
      //*
      //* AUTHOR:      CM-CR/EES4-Grosse Pawig
      //*
      //* HISTORY:
      //* 12.07.2002 CM-CR/EES4-Grosse Pawig
      //*            Initial Revision
      //*****************************************************************
      //##ModelId=3D2E959803B5
      tBool bStart (OSAL_tMSecond msec, OSAL_tMSecond interval = 0);

      //*****************************************************************
      //* FUNCTION:    bStart
      //*
      //* DESCRIPTION: Start a periodic timer
      //*
      //*              If the timer is currently active, the previous
      //*              settings will be overridden.
      //*
      //* PARAMETER:   interval: (I)
      //*                 The timer will expire periodically every
      //*                 'interval' milli seconds.
      //*
      //* RETURNVALUE: TRUE on success, FALSE on failure.
      //*
      //* AUTHOR:      CM-CR/EES4-Grosse Pawig
      //*
      //* HISTORY:
      //* 12.07.2002 CM-CR/EES4-Grosse Pawig
      //*            Initial Revision
      //*****************************************************************
      //##ModelId=3D2EC4F103A0
      tBool bStartPeriodic (OSAL_tMSecond interval);

      //*****************************************************************
      //* FUNCTION:    bStop
      //*
      //* DESCRIPTION: Stop the timer.
      //*
      //* PARAMETER:   -
      //*
      //* RETURNVALUE: TRUE on success, FALSE on failure.
      //*
      //* AUTHOR:      CM-CR/EES4-Grosse Pawig
      //*
      //* HISTORY:
      //* 12.07.2002 CM-CR/EES4-Grosse Pawig
      //*            Initial Revision
      //*****************************************************************
      //##ModelId=3D2E95A00398
      tBool bStop ();

      //*****************************************************************
      //* FUNCTION:    bGetTime
      //*
      //* DESCRIPTION: Read the remaining time of the timer.
      //*
      //* PARAMETER:   pMSec:      (->O)
      //*                 On successful return, the variable pointed to
      //*                 by this pointer contains the remaining time of
      //*                 the timer.
      //*                 A NULL pointer may be passed if one is not
      //*                 interested in the remaining time.
      //*
      //*              pInterval:  (->O)
      //*                 On successful return, the variable pointed to
      //*                 by this pointer contains the interval length
      //*                 set for the timer.
      //*                 A NULL pointer may be passed if one is not
      //*                 interested in the interval length.
      //*
      //* RETURNVALUE: TRUE on success, FALSE on failure.
      //*
      //* AUTHOR:      CM-CR/EES4-Grosse Pawig
      //*
      //* HISTORY:
      //* 12.07.2002 CM-CR/EES4-Grosse Pawig
      //*            Initial Revision
      //*****************************************************************
      //##ModelId=3D2E95B8002C
      tBool bGetTime (OSAL_tMSecond* pMSec, OSAL_tMSecond* pInterval);

  private:
    // Data Members for Class Attributes

      OSAL_tTimerHandle m_hTimer;
};

//*******************************************************************
//* CLASS:       ahl_tclCallbackTimer
//*
//* DESCRIPTION: A forwarding timer.
//*
//*              This class provides a timer that on each timer tick
//*              calls a given method on a given object of a client
//*              class.
//*
//* PARAMETER:   T_Client
//*                 The class which is to be called back.
//*
//* AUTHOR:      CM-CR/EES4-Grosse Pawig
//*
//* COPYRIGHT:   (c) 2002 Blaupunkt GmbH
//*
//* HISTORY:
//* 12.07.2002 CM-CR/EES4-Grosse Pawig
//*            Initial Revision
//*******************************************************************
//##ModelId=3D2E90FB0013
template <class T_Client>
class ahl_tclCallbackTimer : public ahl_tclTimer
{
   // tCallback is a pointer to a member function of class T_Client
   //    that takes no parameters and returns tVoid.
   typedef tVoid (T_Client::*tCallback)();

  public:
      inline ahl_tclCallbackTimer();


      //*****************************************************************
      //* FUNCTION:    bInit
      //*
      //* DESCRIPTION: Initialise the timer.
      //*
      //*              This method initialises the timer and stores the
      //*              object and member function to call whenever the
      //*              timer expires.
      //*
      //* PARAMETER:   poClient:  (->I)
      //*                 Pointer to an object of the client class.
      //*                 Whenever the callback is triggered, the
      //*                 method indicated in 'pFun' is called on this
      //*                 object.
      //*
      //*              pFun:      (->I)
      //*                 Pointer to a member function of the client
      //*                 class that should be called whenever the
      //*                 callback is triggered.
      //*
      //*                 Only methods with the signature
      //*                    tVoid vFun();
      //*                 may be used, i.e. only methods with no
      //*                 parameters and no return value.
      //*
      //* RETURNVALUE: TRUE on success, FALSE on failure.
      //*
      //* AUTHOR:      CM-CR/EES4-Grosse Pawig
      //*
      //* HISTORY:
      //* 12.07.2002 CM-CR/EES4-Grosse Pawig
      //*            Initial Revision
      //*****************************************************************
      //##ModelId=3D2EA1BB029A
      tBool bInit (T_Client* poClient, tCallback pFun);             /*lint !e1511*/ //ABR=> Review Pawig

  private:
      ahl_tclCallbackTimer(const ahl_tclCallbackTimer< T_Client > &right);
    // Data Members for Associations


      ahl_tclObjectCallback<T_Client> m_oCallback;

};

// Class ahl_tclTimer 

// Parameterized Class ahl_tclCallbackTimer 


template <class T_Client>
inline ahl_tclCallbackTimer<T_Client>::ahl_tclCallbackTimer()
{
}


// Parameterized Class ahl_tclCallbackTimer 


template <class T_Client>
tBool ahl_tclCallbackTimer<T_Client>::bInit (T_Client* poClient, tCallback pFun)
{
   tBool bSuccess = FALSE;

   if ((poClient != NULL) && (pFun != NULL))
   {
      m_oCallback.vSetCallback(poClient, pFun);
      bSuccess = ahl_tclTimer::bInit(&m_oCallback);
   }
   return bSuccess;
}


//*******************************************************************
//* CLASS:       ahl_tclExCallbackTimer
//*
//* DESCRIPTION: A forwarding timer.
//*
//*              This class provides a timer that on each timer tick
//*              calls a given method on a given object of a client
//*              class.
//*
//* PARAMETER:   T_Client
//*                 The class which is to be called back.
//*
//* AUTHOR:      CM-CR/EES4-Grosse Pawig
//*
//* COPYRIGHT:   (c) 2002 Blaupunkt GmbH
//*
//* HISTORY:
//* 12.07.2002 CM-CR/EES4-Grosse Pawig
//*            Initial Revision
//*******************************************************************
//##ModelId=3D2E90FB0013
template <class T_Client>
class ahl_tclExCallbackTimer : public ahl_tclTimer
{
   // tCallback is a pointer to a member function of class T_Client
   //    that takes no parameters and returns tVoid.
   typedef tVoid (T_Client::*tExCallback)(tVoid* pParam);

  public:
      inline ahl_tclExCallbackTimer();


      //*****************************************************************
      //* FUNCTION:    bInit
      //*
      //* DESCRIPTION: Initialise the timer.
      //*
      //*              This method initialises the timer and stores the
      //*              object and member function to call whenever the
      //*              timer expires.
      //*
      //* PARAMETER:   poClient:  (->I)
      //*                 Pointer to an object of the client class.
      //*                 Whenever the callback is triggered, the
      //*                 method indicated in 'pFun' is called on this
      //*                 object.
      //*
      //*              pFun:      (->I)
      //*                 Pointer to a member function of the client
      //*                 class that should be called whenever the
      //*                 callback is triggered.
      //*
      //*                 Only methods with the signature
      //*                    tVoid vFun();
      //*                 may be used, i.e. only methods with no
      //*                 parameters and no return value.
      //*
      //* RETURNVALUE: TRUE on success, FALSE on failure.
      //*
      //* AUTHOR:      CM-CR/EES4-Grosse Pawig
      //*
      //* HISTORY:
      //* 12.07.2002 CM-CR/EES4-Grosse Pawig
      //*            Initial Revision
      //*****************************************************************
      //##ModelId=3D2EA1BB029A
      tBool bInit (T_Client* poClient, tExCallback pFun, tVoid* pParam);             /*lint !e1511*/ //ABR=> Review Pawig

  private:
      ahl_tclExCallbackTimer(const ahl_tclExCallbackTimer< T_Client > &right);
    // Data Members for Associations


      ahl_tclExObjectCallback<T_Client> m_oCallback;

};

// Class ahl_tclTimer 

// Parameterized Class ahl_tclExCallbackTimer 


template <class T_Client>
inline ahl_tclExCallbackTimer<T_Client>::ahl_tclExCallbackTimer()
{
}


// Parameterized Class ahl_tclExCallbackTimer 


template <class T_Client>
tBool ahl_tclExCallbackTimer<T_Client>::bInit(T_Client* poClient, tExCallback pFun, tVoid* pParam)
{
   tBool bSuccess = FALSE;

   if ((poClient != NULL) && (pFun != NULL))
   {
      m_oCallback.vSetCallback(poClient, pFun, pParam);
      bSuccess = ahl_tclTimer::bInit(&m_oCallback);
   }
   return bSuccess;
}


#endif
