/********************************************************************//**
 * @file         ahl_Streamer.h
 * @ingroup      Streamer
 *
 * @component    Application Help Library
 *
 * @description  This file defines a basic data stream class.
 *
 * @author       CM-AM/ENG-DI-Streitfeld
 * @copyright    2002,2006 Blaupunkt GmbH
 ************************************************************************/

#ifndef ahl_Streamer_h
#define ahl_Streamer_h

#define OSAL_S_IMPORT_INTERFACE_GENERIC   /*!< Import all OSAL features */
#include "osal_if.h"

/********************************************************************//**
 * @class       ahl_tclStreamer
 * @ingroup     Streamer
 *
 * @description A basic stream class.
 *
 *              This class provides the capabilities needed to stream
 *              basic data types into and out of a memory buffer.
 *
 *              It supports independent read and write operations.
 *              While writing is possible between the beginning and the
 *              end of the stream buffer, reading is only possible in
 *              the part that has already been written.
 *
 *              The stream supports working on application supplied
 *              memory buffers as well as maintaining its own buffer.
 *              In any case, the size of the buffer is fixed;
 *              dynamically growing buffers are not supported.
 *
 *              In case of an application supplied buffers, the
 *              application remains responsible for the memory
 *              management of the buffer.  Especially, the application
 *              must make sure that the buffer remains accessible while
 *              it is in use by a streamer instance.
 *
 * @author      CM-AM/ENG-DI-Streitfeld
 ************************************************************************/
//##ModelId=3BF91947005E
class ahl_tclStreamer
{
public:
   /********************************************************************//**
    * @method      ahl_tclStreamer
    *
    * @description Default constructor.
    *
    *              The default constructor constructs an empty stream.
    *              Before the instance can be used for data exchange,
    *              it has to be assigned an external stream buffer
    *              with vSetStreamBuffer().
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   ahl_tclStreamer();

   /********************************************************************//**
    * @method      ahl_tclStreamer
    *
    * @description Constructor.
    *
    *              This constructor constructs a stream with its own
    *              memory buffer.  The stream is initially empty.
    *
    * @param       u32StreamSize (I) \n
    *                 The maximum size of the stream buffer.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3BF919A40170
   ahl_tclStreamer (tU32 u32StreamSize);

   /********************************************************************//**
    * @method      ~ahl_tclStreamer
    *
    * @description Destructor.
    *
    *              If the streamer instance maintains its own buffer,
    *              the destructor releases the buffers memory.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   virtual ~ahl_tclStreamer();

   /********************************************************************//**
    * @method      pGetStream
    *
    * @description Retrieve pointer to the stream buffer.
    *
    *              This method returns a pointer to the whole stream
    *              buffer.  It is intended to be used after an output
    *              stream has been filled, to pass the collected data
    *              to some other function.
    *
    * @question    Why is this method not declared @c const?
    *
    * @return      A pointer to the stream buffer.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3BF937CB01A2
   tChar* pGetStream ()
      {
         return _pStreamBuffer;
      }

   /********************************************************************//**
    * @method      vSetStreamBuffer
    *
    * @description Set a new stream buffer.
    *
    *              This method provides the stream object with a new,
    *              application-supplied buffer.  It also supports
    *              "prefilled" buffers, which already contain some
    *              valid data.
    *
    *              This method sets the read position to the beginning
    *              of the buffer, while the write position is set to
    *              @a u16WrittenBytes, i.e. just beyond the data that
    *              is already present in the buffer.
    *
    *              If the stream object previously maintained its own
    *              buffer, this method will release that buffer's memory.
    *
    * @param       pBuffer       (->I) \n
    *                 A pointer to the stream buffer.
    *
    * @param       u32BufferSize   (I) \n
    *                 The size of @a pBuffer, in bytes.
    *
    * @param       u32WrittenBytes (I) \n
    *                 The amount of data already present in @a pBuffer.
    *
    * @return      -
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3C0E006503CE
   tVoid vSetStreamBuffer (tChar* pBuffer, tU32 u32BufferSize, tU32 u32WrittenBytes);

   /********************************************************************//**
    * @method      vResetReadWriteIndex
    *
    * @description Clear the stream.
    *
    *              This method discards all data that is currently
    *              present in the stream object.
    *
    * @return      -
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3D04849F0153
   tVoid vResetReadWriteIndex ()
      {
         _u32Size = 0;
         _u32ReadIndex = 0;
      }

   /********************************************************************//**
    * @method      vResetReadIndex
    *
    * @description Reset read position.
    *
    *              This method resets the read position of the stream,
    *              so that all data will be read again.
    *
    * @return      -
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DB3F1F8007E
   tVoid vResetReadIndex ()
   {
      _u32ReadIndex = 0;
   }


   tU32 u32IncrementReadIndex(tU32 u32Increment)
   {
      // there must be enough byte left to read
      _u32ReadIndex += u32Increment;
      OSAL_vAssert( _u32Size >= _u32ReadIndex);
      return _u32ReadIndex;
   }


   tU32 u32SetReadIndex(tU32 u32Pos)
   {
      // there must be enough byte left to read
      OSAL_vAssert( _u32Size >= u32Pos);
      _u32ReadIndex = u32Pos;
      return _u32ReadIndex;
   }

   /********************************************************************//**
    * @method      u16GetDataSize
    *
    * @description Retrieve amount of written data.
    *
    * @return      The number of bytes that have already been written
    *              to the stream.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DB0132B015C
   tU16 u16GetDataSize () const
      {
         OSAL_vAssert(_u32Size <= 0xFFFF);
         return (tU16)_u32Size;
      }

   tU32 u32GetDataSize () const
   {
      return _u32Size;
   }

   /********************************************************************//**
    * @method      u16GetReadIndex
    *
    * @description Retrieve amount of read data.
    *
    * @question    Why is this method not declared @c const?
    *
    * @return      The number of bytes that have already been read
    *              from the stream.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   tU16 u16GetReadIndex () const
   {
      OSAL_vAssert(_u32ReadIndex <= 0xFFFF);
      return (tU16)_u32ReadIndex;
   }

   tU32 u32GetReadIndex () const
   {
      return _u32ReadIndex;
   }

   /********************************************************************//**
    * @method      u16GetActualStringSize
    *
    * @description Determine length of a string at the current read
    *              position.
    *
    *              This methods interprets the stream data starting
    *              at the current read position as a @c 0 terminated
    *              string and returns the length of that string.
    *
    * @pre         The stream has at least one byte left to be read.
    *
    * @post        The current read position of the stream remains
    *              unchanged.
    *
    * @return      The length of the string, including the terminating
    *              @c 0 byte, or @c 0 if there is no @c 0 terminated
    *              string located at the current read position.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DB3FD7B02F2
   tU16 u16GetActualStringSize () const
   {
      tU32 u32StringSize = u32GetActualStringSize();
      OSAL_vAssert(u32StringSize <= 0xFFFF);
      return (tU16)u32StringSize;
   }

   tU32 u32GetActualStringSize () const;

   /** @name Writing data into a stream. */
   /*@{*/
   /********************************************************************//**
    * @method      operator <<
    *
    * @description Get data from another stream.
    *
    *              This method appends the unread data from the source
    *              stream @a prStrm to this stream object.
    *
    *              This operation leaves the source stream unchanged.
    *
    * @question    Is it intentional that @a prStrm remains unchanged?
    *              If it is, @a prStrm should be declared @c const.
    *
    * @pre         There has to be enough space left in the stream buffer
    *              to receive the data.
    *
    * @post        The data is stored in the stream, and the write
    *              position is adjusted accordingly.
    *
    * @param       prStrm (I) \n
    *                 The stream object whose data is to be copied.
    *
    * @return      This stream object; this allows chaining of stream
    *              operations.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3BF919F7027E
   ahl_tclStreamer& rfWriteStream (const ahl_tclStreamer& prStrm);

   ahl_tclStreamer& operator << (const bool bValue);

   /********************************************************************//**
    * @method      operator <<
    *
    * @description Append a tU8 value.
    *
    *              This method appends a single tU8 value to the stream.
    *
    * @pre         There has to be enough space left in the stream buffer
    *              to receive the data.
    *
    * @post        The data is stored in the stream, and the write
    *              position is adjusted accordingly.
    *
    * @param       u8Value (I) \n
    *                 The value to insert.
    *
    * @return      This stream object; this allows chaining of stream
    *              operations.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3BF91D6C00DC
   ahl_tclStreamer& operator << (const tU8 u8Value);

   /********************************************************************//**
    * @method      operator <<
    *
    * @description Append a tS8 value.
    *
    *              This method appends a single tS8 value to the stream.
    *
    * @pre         There has to be enough space left in the stream buffer
    *              to receive the data.
    *
    * @post        The data is stored in the stream, and the write
    *              position is adjusted accordingly.
    *
    * @param       s8Value (I) \n
    *                 The value to insert.
    *
    * @return      This stream object; this allows chaining of stream
    *              operations.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DB9198F035C
   ahl_tclStreamer& operator << (const tS8 s8Value)
      {
         //tU8 u8Value = (tU8)s8Value;
         return operator << ((const tU8)s8Value);
      }

   /********************************************************************//**
    * @method      operator <<
    *
    * @description Append a tU16 value.
    *
    *              This method appends a single tU16 value to the stream.
    *
    * @pre         There has to be enough space left in the stream buffer
    *              to receive the data.
    *
    * @post        The data is stored in the stream, and the write
    *              position is adjusted accordingly.
    *
    * @param       u16Value (I) \n
    *                 The value to insert.
    *
    * @return      This stream object; this allows chaining of stream
    *              operations.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3BF91DAD011C
   ahl_tclStreamer& operator << (const tU16 u16Value);

   /********************************************************************//**
    * @method      operator <<
    *
    * @description Append a tS16 value.
    *
    *              This method appends a single tS16 value to the stream.
    *
    * @pre         There has to be enough space left in the stream buffer
    *              to receive the data.
    *
    * @post        The data is stored in the stream, and the write
    *              position is adjusted accordingly.
    *
    * @param       s16Value (I) \n
    *                 The value to insert.
    *
    * @return      This stream object; this allows chaining of stream
    *              operations.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DB91870005C
   ahl_tclStreamer& operator << (const tS16 s16Value)
      {
         return operator << ((const tU16)s16Value);
      }

   /********************************************************************//**
    * @method      operator <<
    *
    * @description Append a tU32 value.
    *
    *              This method appends a single tU32 value to the stream.
    *
    * @pre         There has to be enough space left in the stream buffer
    *              to receive the data.
    *
    * @post        The data is stored in the stream, and the write
    *              position is adjusted accordingly.
    *
    * @param       u32Value (I) \n
    *                 The value to insert.
    *
    * @return      This stream object; this allows chaining of stream
    *              operations.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3BF934540251
   ahl_tclStreamer& operator << (const tU32 u32Value);

   ahl_tclStreamer& operator << (const tU64 u64Value);


   /********************************************************************//**
    * @method      operator <<
    *
    * @description Append a tS32 value.
    *
    *              This method appends a single tS32 value to the stream.
    *
    * @pre         There has to be enough space left in the stream buffer
    *              to receive the data.
    *
    * @post        The data is stored in the stream, and the write
    *              position is adjusted accordingly.
    *
    * @param       s32Value (I) \n
    *                 The value to insert.
    *
    * @return      This stream object; this allows chaining of stream
    *              operations.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DB918B7015C
   ahl_tclStreamer& operator << (const tS32 s32Value)
      {
         return operator << ((const tU32)s32Value);
      }
   ahl_tclStreamer& operator << (const tS64 s64Value)
      {
         return operator << ((const tU64)s64Value);
      }

#if defined(__APPLE__)
   ahl_tclStreamer& operator << (const uintptr_t value);

   ahl_tclStreamer& operator << (const intptr_t value)
   {
      return operator << ((const uintptr_t)value);
   }
#endif

   /********************************************************************//**
    * @method      vWriteString
    *
    * @description Append a bunch of bytes.
    *
    *              This method appends a bunch of @a u16StringSize bytes
    *              to the stream, taking the data from @a rfcoszString.
    *
    * @pre         There has to be enough space left in the stream buffer
    *              to receive the data.
    *
    * @post        The data is stored in the stream, and the write
    *              position is adjusted accordingly.
    *
    * @param       coszString (->I) \n
    *                 Specifies the data to insert.
    *
    * @param       u16StringSize  (I) \n
    *                 The number of bytes to insert.
    *
    * @return      -
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3CD687030370
   tVoid vWriteString (tCString coszString, tU16 u16StringSize);

   /********************************************************************//**
    * @method      vWriteStringToStream
    *
    * @description Append a @c 0 terminated string to the stream.
    *
    *              This method appends @a u16StringSize bytes to the
    *              stream, taking the data from @a rfszString.  The last
    *              character will be a @c 0 byte.
    *
    *              Except for the @c 0 termination, this method is
    *              identical to vWriteString().
    *
    * @pre         There has to be enough space left in the stream buffer
    *              to receive the data.
    *
    * @post        The data is stored in the stream, and the write
    *              position is adjusted accordingly.
    *
    * @param       rfszString  (->I) \n
    *                 Specifies the data to insert.
    *
    * @param       u16StringSize (I) \n
    *                 The number of bytes to insert.
    *
    * @return      -
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3E1D7CDF02C6
   tVoid vWriteStringToStream (tCString szString, tU32 u32StringSize);
   /*@}*/

   /** @name Reading data from a stream. */
   /*@{*/
   /********************************************************************//**
    * @method      operator >>
    *
    * @description Read data into another stream.
    *
    *              This method fills the target stream @a prStrm with
    *              unread data from this stream object.  It overwrites
    *              all data that @a prStrm may already contain.
    *
    * @pre         This stream contains enough unread bytes to fill
    *              the whole target stream.
    *
    * @post        The read position of this stream is adjusted
    *              accordingly.
    *
    * @post        The read position of @a prStrm is reset to the
    *              beginning of that stream.
    *
    * @param       prStrm (I) \n
    *                 The place to store the read value.
    *
    * @return      This stream object; this allows chaining of stream
    *              operations.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DB918370290
   ahl_tclStreamer& rfReadStream (ahl_tclStreamer& prStrm);

   /********************************************************************//**
    * @method      operator >>
    *
    * @description Read a tU8 value.
    *
    *              This method reads a single tU8 value from the stream.
    *
    * @pre         The stream contains enough unread bytes to fulfil
    *              the read request.
    *
    * @post        The read position is adjusted accordingly.
    *
    * @param       u8Value (I) \n
    *                 The place to store the read value.
    *
    * @return      This stream object; this allows chaining of stream
    *              operations.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3C0DFB1300CD
   ahl_tclStreamer& operator >> (tU8& u8Value);

   /********************************************************************//**
    * @method      operator >>
    *
    * @description Read a tS8 value.
    *
    *              This method reads a single tS8 value from the stream.
    *
    * @pre         The stream contains enough unread bytes to fulfil
    *              the read request.
    *
    * @post        The read position is adjusted accordingly.
    *
    * @param       s8Value (I) \n
    *                 The place to store the read value.
    *
    * @return      This stream object; this allows chaining of stream
    *              operations.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DB9190000C4
   ahl_tclStreamer& operator >> (tS8& s8Value)
      {
         return operator >> (*(tU8*)&s8Value);
      }

   /********************************************************************//**
    * @method      operator >>
    *
    * @description Read a tU16 value.
    *
    *              This method reads a single tU16 value from the stream.
    *
    * @pre         The stream contains enough unread bytes to fulfil
    *              the read request.
    *
    * @post        The read position is adjusted accordingly.
    *
    * @param       u16Value (I) \n
    *                 The place to store the read value.
    *
    * @return      This stream object; this allows chaining of stream
    *              operations.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3C0DFB1300D7
   ahl_tclStreamer& operator >> (tU16& u16Value);

   /********************************************************************//**
    * @method      operator >>
    *
    * @description Read a tS16 value.
    *
    *              This method reads a single tS16 value from the stream.
    *
    * @pre         The stream contains enough unread bytes to fulfil
    *              the read request.
    *
    * @post        The read position is adjusted accordingly.
    *
    * @param       s16Value (I) \n
    *                 The place to store the read value.
    *
    * @return      This stream object; this allows chaining of stream
    *              operations.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DB9190000F6
   ahl_tclStreamer& operator >> (tS16& s16Value)
      {
         return operator >> (*(tU16*)&s16Value);
      }

   /********************************************************************//**
    * @method      operator >>
    *
    * @description Read a tU32 value.
    *
    *              This method reads a single tU32 value from the stream.
    *
    * @pre         The stream contains enough unread bytes to fulfil
    *              the read request.
    *
    * @post        The read position is adjusted accordingly.
    *
    * @param       u32Value (I) \n
    *                 The place to store the read value.
    *
    * @return      This stream object; this allows chaining of stream
    *              operations.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3C0DFB1300EB
   ahl_tclStreamer& operator >> (tU32& u32Value);

   ahl_tclStreamer& operator >> (tU64& u64Value);

   /********************************************************************//**
    * @method      operator >>
    *
    * @description Read a tS32 value.
    *
    *              This method reads a single tS32 value from the stream.
    *
    * @pre         The stream contains enough unread bytes to fulfil
    *              the read request.
    *
    * @post        The read position is adjusted accordingly.
    *
    * @param       s32Value (I) \n
    *                 The place to store the read value.
    *
    * @return      This stream object; this allows chaining of stream
    *              operations.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DB919000132
   ahl_tclStreamer& operator >> (tS32& s32Value)
      {
         return operator >> (*(tU32*)&s32Value);
      }

   ahl_tclStreamer& operator >> (tS64& s64Value)
      {
         return operator >> (*(tU64*)&s64Value);
      }

#if defined(__APPLE__)
   ahl_tclStreamer& operator >> (uintptr_t& value);

   ahl_tclStreamer& operator >> (intptr_t& value)
   {
      return operator >> (*(uintptr_t*)&value);
   }
#endif

   /********************************************************************//**
    * @method      vReadString
    *
    * @description Read a @c 0 terminated string.
    *
    *              This method reads a @c 0 terminated string from the
    *              stream.
    *
    * @question    Why is the data sink supplied as a @e reference to
    *              a string?
    * @question    This method is identical to vReadStringFromStream().
    *              Is this intentional?
    *
    * @pre         The stream contains a @c 0 terminated string at the
    *              current read position.
    *
    * @pre         The string fits into the supplied buffer.
    *
    * @post        The read position is adjusted accordingly.
    *
    * @param       rfpcBuffer  (->O) \n
    *                 The place to store the read value.
    *
    * @param       u16BufferSize (I) \n
    *                 The size of @a rfpcBuffer.
    *
    * @return      -
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3CD687CF00E8
   tVoid vReadString (tChar*& rfpcBuffer, tU32 u32BufferSize);

   /********************************************************************//**
    * @method      vReadStringFromStream
    *
    * @description Read a @c 0 terminated string.
    *
    *              This method reads a @c 0 terminated string from the
    *              stream.
    *
    * @question    This method is identical to vReadString().
    *              Is this intentional?
    *
    * @pre         The stream contains a @c 0 terminated string at the
    *              current read position.
    *
    * @pre         The string fits into the supplied buffer.
    *
    * @post        The read position is adjusted accordingly.
    *
    * @param       rfszBuffer  (->O) \n
    *                 The place to store the read value.
    *
    * @param       u16BufferSize (I) \n
    *                 The size of @a rfpcBuffer.
    *
    * @return      -
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3E1D7CE60366
   tVoid vReadStringFromStream (tString szBuffer, tU32 u32BufferSize, tU32 u32KnownStringSize = 0);
   /*@}*/

protected:
   // Data Members for Class Attributes

   /** Pointer to the stream buffer. */
   tChar* _pStreamBuffer;

   /** The maximum size of the stream buffer. */
   tU32 _u32AllocatedMemory;

   /** Current write position. */
   tU32 _u32Size;

   /** Current read position. */
   tU32 _u32ReadIndex;

   /** If @c TRUE, #_pStreamBuffer has been supplied by the application
    *  and must not be released by the stream object.  Otherwise, the
    *  stream object owns the stream buffer. */
   tBool _bExternData;
private:
   // forbidden 
   ahl_tclStreamer& operator=(const ahl_tclStreamer& right){
      if(&right==this)
         return *this;
      else {
         _pStreamBuffer = 0;
         _u32AllocatedMemory = 0;
         _u32Size = 0;
         _u32ReadIndex = 0;
         _bExternData =0;
         return *this;
      }
   }

};


#endif
