/********************************************************************//**
 * @file         ahl_StreamableTypes.h
 * @ingroup      Streamer
 *
 * @component    Application Help Library
 *
 * @description  This file defines classes that allow easy insertion and
 *               extraction of complex data types into streams.
 *
 * @author       CM-AM/ENG-DI-Streitfeld
 * @copyright    2002,2006 Blaupunkt GmbH
 ************************************************************************/

#ifndef ahl_StreamableTypes_h
#define ahl_StreamableTypes_h

#define AHL_S_IMPORT_INTERFACE_LISTSTREAMER
#include "ahl_if.h"
#define AHL_S_IMPORT_INTERFACE_LIST
#include "ahl_if.h"
#define SYSTEM_S_IMPORT_INTERFACE_COMPLETE
#include "system_pif.h"
#include "ahl_List.h"



/********************************************************************//**
 * @interface   ahl_tclStreamableType
 * @ingroup     Streamer
 *
 * @description Interface for streamable types.
 *
 *              This class defines the interface that is implemented
 *              by all streamable types.
 *
 *              It provides a default implementation for all methods,
 *              which actually does nothing.
 *
 * @author      CM-AM/ENG-DI-Streitfeld
 ************************************************************************/
//##ModelId=3D92E50B00F6
class ahl_tclStreamableType
{
public:
   ahl_tclStreamableType();

   virtual ~ahl_tclStreamableType();

   /********************************************************************//**
    * @method      u32GetStreamSize
    *
    * @description How many bytes does this type occupy in a stream?
    *
    *              This method is used to determine how many bytes an
    *              instance of this type occupies when it is stored in
    *              a stream.
    *
    * @return      The size of the streamed representation, in bytes.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3D92EA08032D
   virtual tU32 u32GetStreamSize () const = 0;
   
   // for compatibility reasons
   virtual tU16 u16GetStreamSize () const 
   {
      tU32 u32ReturnValue = u32GetStreamSize();
      if (u32ReturnValue <= 0xFFFF)
      {
         return (tU16)u32ReturnValue;
      }
      else
      {
         OSAL_vAssert(u32ReturnValue <= 0xFFFF);
         return 0;
      }
   }


   /********************************************************************//**
    * @method      vSerializeType
    *
    * @description Write the object to a stream.
    *
    *              This method is used to write this object to the
    *              supplied stream.
    *
    * @param       rfoStream (O)
    *                 The stream this object is to be written to.
    *
    * @return      -
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3D92E98A028B
   virtual tVoid vSerializeType (ahl_tclListStreamer& rfoStream);

   /********************************************************************//**
    * @method      vFillType
    *
    * @description Read object data from a stream.
    *
    *              This method is used to read data from the supplied
    *              stream and assign it to this object.
    *
    * @param       rfoStream (O)
    *                 The stream this object is to be read from.
    *
    * @return      -
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3D92EB100138
   virtual tVoid vFillType (ahl_tclListStreamer& rfoStream);
protected:
   ahl_tclStreamableType& operator=(const ahl_tclStreamableType& right){ return (&right ==this)?*this:*this;}
};

/********************************************************************//**
 * @interface   ahl_tclStreamableList
 * @ingroup     Streamer
 *
 * @description A streamable list interface.
 *
 *              This class provides a wrapper around an ahl_tU16List
 *              of objects that provides it with the functionality of
 *              a streamable type.
 *
 *              This wrapper is unaware of the concrete kind of element
 *              stored in the list.  Since it needs to create instances
 *              of elements when the list is filled from a stream, a
 *              concrete list type must provide the necessary instance
 *              creation.
 *
 *              The objects in the list must themselves derive from
 *              ahl_tclStreamableType.
 *
 * @author      CM-AM/ENG-DI-Streitfeld
 ************************************************************************/
//##ModelId=3D92E425029A
class ahl_tclStreamableList : public ahl_tclStreamableType
{
public:
   //##ModelId=4141C21C0138
   ahl_tclStreamableList();

   //##ModelId=4141C21C0196
   virtual ~ahl_tclStreamableList();


   /********************************************************************//**
    * @method      u16GetElementNr
    *
    * @description Determine the number of elements in this list.
    *
    * @return      The number of elements currently stored in the list.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DC7C65100DA
   tU16 u16GetElementNr () const
      {
         return _oList.u16GetSize();
      }

   /** @name Sequential access */
   /*@{*/
   /********************************************************************//**
    * @method      vRewind
    *
    * @description Reset the sequential iteration sequence.
    *
    * @return      -
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DC7BE290040
   tVoid vRewind () const
      {
         _oList.vRewind();
      }

   /********************************************************************//**
    * @method      pvGetSequential
    *
    * @description Perform an iteration step.
    *
    *              This method returns the current element of the
    *              iteration sequence and advances the sequence by
    *              one step.
    *
    * @return      The object that has been current before the step
    *              has been performed.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DC7C03E028B
   tVoid* pvGetSequential () const
      {
         return _oList.pvGetSequential();
      }
   /*@}*/

   /** @name Insertion and removal
    *  @note These methods reset the sequential iteration sequence. */
   /*@{*/
   /********************************************************************//**
    * @method      bAppend
    *
    * @description Add an element to the tail of the list.
    *
    *              The object is only added if it does not already exist
    *              in the list; in the latter case, or if @a pvElement is
    *              @c NULL, this method returns with an error.
    *
    * @note        This method resets the sequential iteration sequence.
    *
    * @param       pvElement (->I) \n
    *                 The object that is to be added.
    *
    * @return      @c TRUE if the object has been added to the list,
    *              @c FALSE in case of an error.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DC7C0600173
   tBool bAppend (tVoid* pvElement)
      {
         return _oList.bAppend(pvElement);
      }

   /********************************************************************//**
    * @method      bAddAsFirst
    *
    * @description Add an element to the head of the list.
    *
    *              The object is only added if it does not already exist
    *              in the list; in the latter case, or if @a pvElement is
    *              @c NULL, this method returns with an error.
    *
    * @note        This method resets the sequential iteration sequence.
    *
    * @param       pvElement (->I) \n
    *                 The object that is to be added.
    *
    * @return      @c TRUE if the object has been added to the list,
    *              @c FALSE in case of an error.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3F4B4F810030
   tBool bAddAsFirst (tVoid* pvElement)
      {
         return _oList.bAddAsFirst(pvElement);
      }

   /********************************************************************//**
    * @method      u16InsertEntryBefore
    *
    * @description Insert an element into the list.
    *
    *              The object is only inserted if it does not already
    *              exist in the list; in the latter case, or if
    *              @a pvNewEntry is @c NULL, this method returns with an
    *              error.
    *
    *              If the insertion position @a u16Index is larger than
    *              the size of the list, the element is added to the
    *              end of the list.
    *
    * @note        This method resets the sequential iteration sequence.
    *
    * @param       u16Index   (I) \n
    *                 The index before which the new element is to be
    *                 inserted,
    *
    * @param       pvNewEntry (->I) \n
    *                 The object that is to be added.
    *
    * @return      The index at which the element has been inserted,
    *              or @c 0xFFFF in case of an error.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=415AB235031C
   tU16 u16InsertEntryBefore(tU16 u16Index, tVoid* pvNewEntry)
      {
         return _oList.u16InsertEntryBefore(u16Index,pvNewEntry);
      }

   /********************************************************************//**
    * @method      bRemove
    *
    * @description Remove an object from the list.
    *
    * @param       pvElement (->I) \n
    *                 The object that is to removed.
    *
    * @note        This method resets the sequential iteration sequence.
    *
    * @return      @c TRUE if the object has been removed from the list,
    *              @c FALSE if the object has not been in the list at all.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3F4B89400206
   tBool bRemove (const tVoid* pvElement)
      {
         return _oList.bRemove(pvElement);
      }

   /********************************************************************//**
    * @method      vRemoveFirstEntry
    *
    * @description Remove the first entry of the list.
    *
    * @note        This method resets the sequential iteration sequence.
    *
    * @return      -
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DC7C08503B1
   tVoid vRemoveFirstEntry ()
      {
         _oList.vRemoveFirstEntry();
      }

   /** @name Sorting */
   /*@{*/
   /********************************************************************//**
   * @method      vInsertSortList
   *
   * @description Sort list by insertion in ascending order 
   *              (tfunc_s32Compare < 0)
   *
   * @param       funcComp (->I) \n
   *                 Compare function to use for sorting.
   *
   * @return      -
   *
   * @author      CM-AM/ENG-DI-Streitfeld
   ************************************************************************/
   tVoid vInsertSortList(ahl_tclComparator &rfoComparator)
   {
      _oList.vInsertSortList(rfoComparator);
   }

   /********************************************************************//**
   * @method      vInsertSortParallelList
   *
   * @description Sort list by insertion in ascending order 
   *              (tfunc_s32Compare < 0) and apply the same sorting to external list
   *
   * @param       funcComp (->I) \n
   *                 Compare function to use for sorting.
   *
   * @param       funcComp (I) \n
   *                 External list which should be sorted in same way.
   *                 If nr of elements not the same, no sorting at all is done. 
   *
   * @return      -
   *
   * @author      CM-AM/ENG-DI-Streitfeld
   ************************************************************************/
   tVoid vInsertSortParallelList( ahl_tclComparator &rfoComparator, 
      ahl_tclStreamableList &rfoExtList)
   {
      _oList.vInsertSortParallelList(rfoComparator, rfoExtList._oList);
   }

   /********************************************************************//**
    * @method      vRemoveLastEntry
    *
    * @description Remove the last entry of the list.
    *
    * @note        This method resets the sequential iteration sequence.
    *
    * @return      -
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DCA64DB0066
   tVoid vRemoveLastEntry ()
      {
         _oList.bRemove(_oList.pvGetLast());
      }
   /*@}*/

   /** @name Member access */
   /*@{*/
   /********************************************************************//**
    * @method      pvGetFirstEntry
    *
    * @description Determine the first entry of the list.
    *
    * @return      The first entry of the list, or @c NULL if the list
    *              is empty.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DCA68C001E6
   tVoid* pvGetFirstEntry () const
      {
         return _oList.pvGetFirst();
      }

   /********************************************************************//**
    * @method      pvGetLastEntry
    *
    * @description Determine the last entry of the list.
    *
    * @return      The last entry of the list, or @c NULL if the list
    *              is empty.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DCA68EC01C1
   tVoid* pvGetLastEntry () const
      {
         return _oList.pvGetLast();
      }

   /********************************************************************//**
    * @method      pvGetByIndex
    *
    * @description Determine the list entry at a given index.
    *
    * @note        This method resets the sequential iteration sequence
    *              to continue at @a u16Index.
    *
    * @param       u16Index   (I) \n
    *                 Index of the element that is to be queried.
    *
    * @return      The last entry of the list, or @c NULL if the list
    *              is empty.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DC7D0A702CC
   tVoid* pvGetByIndex (tU16 u16Index) const
      {
         return _oList.pvGetByIndex(u16Index);
      }

   /********************************************************************//**
    * @method      pvReplaceEntry
    *
    * @description Replace a list element with another object.
    *
    * @note        This method resets the sequential iteration sequence
    *              to continue at @a u16Index.
    *
    * @param       u16Index   (I) \n
    *                 Index of the element that is to be replaced.
    *
    * @param       pvNewEntry (->I) \n
    *                 The new object that is to be stored in the list.
    *
    * @return      The previous object at @a u16Index, or @c NULL if
    *              @a u16Index is out of bounds or @a pvNewEntry is
    *              already stored in the list.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3E1AD1FD03B3
   tVoid* pvReplaceEntry (tU16 u16Index, tVoid* pvNewEntry)
      {
         return _oList.pvReplaceEntry(u16Index, pvNewEntry);
      }
   /*@}*/

   /** @name Streamable type interface */
   /*@{*/
   //##ModelId=3D9317FA0312
   virtual tU16 u16GetStreamSize() const;
   virtual tU32 u32GetStreamSize () const;

   //##ModelId=3DD274E3036A
   virtual tVoid vSerializeType (ahl_tclListStreamer& rfoStream);

   //##ModelId=3DD274E303CE
   virtual tVoid vFillType (ahl_tclListStreamer& rfoStream);
   /*@}*/


   /** @name Element maintenance */
   /*@{*/
   /********************************************************************//**
    * @method      poNewStreamableElement
    *
    * @description Create a new list element.
    *
    *              This method is used to create a new list element and
    *              insert it to the list.
    *
    *              Afterwards, it will be filled with data from a stream.
    *
    * @return      A pointer to the element, or @c NULL if the creation
    *              failed.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DC8EDDC0180
   virtual ahl_tclStreamableType* poNewStreamableElement () = 0;

   /********************************************************************//**
    * @method      poNewStreamableElement
    *
    * @description Create a new list element, copied from another element.
    *
    *              This method is used to create a new list element and
    *              insert it to the list.
    *
    *              The contents of the new element are to be taken from
    *              the supplied argument.
    *
    * @param       rfcoElement (I) \n
    *                 The source that the new element is to be initialised
    *                 from.
    *
    * @return      A pointer to the element, or @c NULL if the creation
    *              failed.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DC8EDFD0250
   virtual ahl_tclStreamableType* poNewStreamableElement (const ahl_tclStreamableType &rfcoElement) = 0;

   /********************************************************************//**
    * @method      vDeleteEntries
    *
    * @description Delete the contents of the list.
    *
    *              This method is used to remove all elements from the
    *              list and delete them.
    *
    * @return      -
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3F6033DA00C1
   virtual tVoid vDeleteEntries () = 0;
   /*@}*/

private:
   /** This list instance stores the actual elements. */
   //##ModelId=415AB3B400DA
   ahl_tclU16List _oList;
};

/********************************************************************//**
 * @class       ahl_tclFixedSizeElementList
 * @ingroup     Streamer
 *
 * @description A streamable, partitioned bunch of bytes.
 *
 * @question    What features does this class provide that
 *              ahl_tclStreamableArray does not?
 *
 * @author      CM-AM/ENG-DI-Streitfeld
 ************************************************************************/
//##ModelId=3DB9047C038C
class ahl_tclFixedSizeElementList : public ahl_tclStreamableType
{
public:
   /********************************************************************//**
    * @method      ahl_tclFixedSizeElementList
    *
    * @description Constructor
    *
    *              This constructor creates a list with a given number
    *              of elements.  The elements themselves remain
    *              uninitialised.
    *
    * @param       u16ElementSize  (I) \n
    *                 Size of a single element, in bytes.
    *
    * @param       u16NrOfElements (I) \n
    *                 Number of elements in the list.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DB906AC029F
   ahl_tclFixedSizeElementList (tU16 u16ElementSize, tU16 u16NrOfElements);

   /********************************************************************//**
    * @method      ahl_tclFixedSizeElementList
    *
    * @description Constructor
    *
    *              This constructor creates an empty list.
    *
    * @param       u16ElementSize (I) \n
    *                 Size of a single element, in bytes.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DB910D10214
   ahl_tclFixedSizeElementList (tU16 u16ElementSize);

   /********************************************************************//**
    * @method      ahl_tclFixedSizeElementList
    *
    * @description Copy constructor.
    *
    *              The copy constructor creates a list that contains the
    *              same entries as the source list @a right.
    *
    * @param       right  (I) \n
    *                 The list that is to be copied.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   ahl_tclFixedSizeElementList (const ahl_tclFixedSizeElementList &right);

   virtual ~ahl_tclFixedSizeElementList();


   //##ModelId=3DB905CE0294
   virtual tU32 u32GetStreamSize () const
      {
         return _u16NrOfElements*_u16ElementSize + 4;
      }

   //##ModelId=3DB905CE02B2
   virtual tVoid vSerializeType (ahl_tclListStreamer& rfoStream);

   //##ModelId=3DB905CE02D1
   virtual tVoid vFillType (ahl_tclListStreamer& rfoStream);

   // Data Members for Class Attributes

   /** Size of a single list element, in bytes. */
   tU16 _u16ElementSize;
   /** Number of elements in this list. */
   tU16 _u16NrOfElements;
   /** The list contents, addressed as a bunch of bytes. */
   tU8* _pu8Array;
};


/********************************************************************//**
 * @class       ahl_tclStreamableArray
 * @ingroup     Streamer
 *
 * @description A streamable array.
 *
 *              This class provides a wrapper around a plain array of
 *              objects that provides it with the functionality of a
 *              streamable type.
 *
 *              The array contents are transferred as raw byte data
 *              between the array and a stream.
 *
 * @param       ahl_tElementType
 *                 Specifies the kind of elements that are stored in
 *                 the array.
 *
 * @author      CM-AM/ENG-DI-Streitfeld
 ************************************************************************/
template <class ahl_tElementType>
class ahl_tclStreamableArray : public ahl_tclStreamableType
{
public:

   //Detecting Convertibility at Compile Time, Alexandrescu Modern C++
   template <class T, class U> 
   class Conversion 
   { 
      typedef char Small; 
      class Big { char dummy[2]; }; 
      static Small Test(U); 
      static Big Test(...); 
      static T MakeT(); 
   public: 
      enum { exists = sizeof(Test(MakeT())) == sizeof(Small) }; 
   };

   template <class tElementType1, bool Streamable>
   struct trSizeHelper
   {
      static tU32 u32Size(const tElementType1& oElem) { return sizeof(tElementType1); }
   };
   template <class tElementType2>
   struct trSizeHelper<tElementType2, true>
   {
      static tU32 u32Size(const tElementType2& oElem) { return oElem.u32GetStreamSize(); }
   };

   template <class tElementType3> 
   inline tU32 u32ElemSize(const tElementType3& oElem) const
   {
      return trSizeHelper<tElementType3, Conversion<tElementType3&, ahl_tclStreamableType&>::exists>::u32Size(oElem);
   }

   /********************************************************************//**
    * @method      ahl_tclStreamableArray
    *
    * @description Constructor
    *
    *              This constructor creates an array with a given number
    *              of elements.
    *
    * @param       u16NrOfElements (I) \n
    *                 Number of elements in the list.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   ahl_tclStreamableArray (tU16 u16NrOfElements);

   /********************************************************************//**
    * @method      ahl_tclStreamableArray
    *
    * @description Default constructor
    *
    *              This constructor creates an empty array.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   ahl_tclStreamableArray ();

   /********************************************************************//**
    * @method      ahl_tclStreamableArray
    *
    * @description Copy constructor.
    *
    *              The copy constructor creates an array that contains the
    *              same entries as the source array @a right.
    *
    * @param       right  (I) \n
    *                 The array that is to be copied.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   ahl_tclStreamableArray (const ahl_tclStreamableArray<ahl_tElementType> &right);

   virtual ~ahl_tclStreamableArray();


   virtual tU32 u32GetStreamSize () const
   {
      ahl_tElementType oDummy;
      return _u16NrOfElements*u32ElemSize(oDummy) + 4;
      // the ' + 4' takes the space for list size in the stream into account
   }

   virtual tVoid vSerializeType (ahl_tclListStreamer& rfoStream);

   virtual tVoid vFillType (ahl_tclListStreamer& rfoStream);

   /********************************************************************//**
    * @method      u16GetNrOfElements
    *
    * @description Query number of array elements.
    *
    * @return      The number of elements in this array.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   tU16 u16GetElementNr() const
   {
      return _u16NrOfElements;
   }

   // Data Members for Class Attributes

   /********************************************************************//**
    * @method      operator[]
    *
    * @description Element access.
    *
    *              The subscript operator provides access to individual
    *              array elements.
    *
    * @pre         @a u32Index does not exceed the number of elements
    *              in the array.
    *
    * @param       u32Index  (I) \n
    *                 Index of the array element of interest.
    *
    * @return      The array element at the indicated index.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   ahl_tElementType& operator[] (const tU32 u32Index)
   {
      OSAL_vAssert(_prArray != 0);
      OSAL_vAssert(u32Index < (tU32)_u16NrOfElements);

      return *_prArray[(tU16)u32Index];
   }

   const ahl_tElementType& operator[] (const tU32 u32Index) const
   {
      OSAL_vAssert(_prArray != 0);
      OSAL_vAssert(u32Index < (tU32)_u16NrOfElements);

      return *_prArray[(tU16)u32Index];
   }

   const ahl_tElementType * poGetByIndex (tU16 u16Index) const
   {
      return &(operator[](u16Index));
   }


private:
   /** Number of elements in this array. */
   tU16 _u16NrOfElements;
   /** The array elements. */
   ahl_tElementType** _prArray;
};


template <class ahl_tElementType>
ahl_tclStreamableArray<ahl_tElementType>::ahl_tclStreamableArray (tU16 u16NrOfElements)
:_u16NrOfElements(u16NrOfElements),_prArray(0)
{
   if (_u16NrOfElements != 0)
   {
      _prArray = OSAL_NEW ahl_tElementType*[_u16NrOfElements];
      for (tU32 i=0; i<_u16NrOfElements;++i)
      {
         _prArray[i] = OSAL_NEW ahl_tElementType();
      }
   }
}

template <class ahl_tElementType>
ahl_tclStreamableArray<ahl_tElementType>::ahl_tclStreamableArray ()
:_u16NrOfElements(0),_prArray(0)
{
}

template <class ahl_tElementType>
ahl_tclStreamableArray<ahl_tElementType>::ahl_tclStreamableArray (const ahl_tclStreamableArray<ahl_tElementType> &right)
:_u16NrOfElements(right._u16NrOfElements),_prArray(0)
{
   if (right._prArray != 0 && right._u16NrOfElements != 0)
   {
      _prArray = OSAL_NEW ahl_tElementType*[_u16NrOfElements];
      for (tU32 i=0; i<_u16NrOfElements;++i)
      {
         _prArray[i] = OSAL_NEW ahl_tElementType(*right._prArray[i]);
      }
   }
}


template <class ahl_tElementType>
ahl_tclStreamableArray<ahl_tElementType>::~ahl_tclStreamableArray()
{
   if (_prArray != 0)
   {
      for (tU32 i=0; i<_u16NrOfElements;++i)
      {
         OSAL_DELETE _prArray[i];
         _prArray[i]=0;
      }
   }

   OSAL_DELETE[] _prArray;
   _prArray=0;
}


template <class ahl_tElementType>
tVoid ahl_tclStreamableArray<ahl_tElementType>::vSerializeType (ahl_tclListStreamer& rfoStream)
{
   ahl_tclListStreamer oListStream;
   // tU32 u32ArraySize = u32GetStreamSize() - 4;
   tU32 u32ListSize = (tU32)_u16NrOfElements;
   rfoStream << u32ListSize;
   if (u32ListSize != 0 && _prArray != 0)
   {
      for (tU32 i=0; i<_u16NrOfElements;++i)
      {
         rfoStream << *(_prArray[i]);
      }
   }
}

template <class ahl_tElementType>
tVoid ahl_tclStreamableArray<ahl_tElementType>::vFillType (ahl_tclListStreamer& rfoStream)
{
   tU32 u32ListSize = 0;
   rfoStream >> u32ListSize;
   OSAL_vAssert(u32ListSize < (tU32)0xFFFF);

   if (_prArray != 0)
   {
      for (tU32 i=0; i<_u16NrOfElements;++i)
      {
         OSAL_DELETE _prArray[i];
         _prArray[i]=0;
      }

      OSAL_DELETE[] _prArray;
      _prArray = 0;
   }

   _u16NrOfElements = (tU16)u32ListSize;
   if (_u16NrOfElements != 0)
   {
      _prArray = OSAL_NEW ahl_tElementType*[_u16NrOfElements];

      for (tU32 i=0; i<_u16NrOfElements;++i)
      {
         _prArray[i] = OSAL_NEW ahl_tElementType;
         rfoStream >> *(_prArray[i]);
      }
   }
}




/********************************************************************//**
 * @class       ahl_tclTemplateList
 * @ingroup     Streamer
 *
 * @description A typed list.
 *
 *              While ahl_tclStreamableList only provides a generic
 *              list of streamable types (represented as @c void
 *              pointers), this template class augments its
 *              functionality with the correct element type.  This
 *              saves a lot of casting in the application.
 *
 * @param       ahl_tElementType
 *                 Specifies the kind of elements that are stored in
 *                 the list.  Must be derived from ahl_tclStreamableType.
 *
 * @author      CM-AM/ENG-DI-Streitfeld
 ************************************************************************/
//##ModelId=3DC7BA940332
template <class ahl_tElementType>
class ahl_tclTemplateList : public ahl_tclStreamableList
{
public:
   ahl_tclTemplateList();
   ahl_tclTemplateList(tBool bExternalData)
      : _bExternalData(bExternalData)
   {
      ;
   }

   //##ModelId=3DC7D3E3003E
   ahl_tclTemplateList (const ahl_tclTemplateList<ahl_tElementType> &rhs);

   virtual ~ahl_tclTemplateList();


   //##ModelId=3DC7D4140325
   ahl_tclTemplateList<ahl_tElementType>& operator = (const ahl_tclTemplateList<ahl_tElementType> &rhs);

   //##ModelId=3DC7D4C500A2
   ahl_tElementType * poNewElement ();

   //##ModelId=3DC7D46D028E
   ahl_tElementType * poNewElement (const ahl_tElementType &rfcoElement);

   //##ModelId=3DC7D5080027
   ahl_tElementType * poGetSequential () const
      {
        return (ahl_tElementType *)pvGetSequential();
      }

   //##ModelId=3DC7D519022B
   ahl_tElementType * poGetByIndex (tU16 u16Index) const
      {
        return (ahl_tElementType *)pvGetByIndex(u16Index);
      }

   //##ModelId=3DCA699B0267
   ahl_tElementType * poGetFirstElement () const
      {
        return (ahl_tElementType *)pvGetFirstEntry();
      }

   //##ModelId=3DCA69AD0247
   ahl_tElementType * poGetLastElement () const
      {
        return (ahl_tElementType *)pvGetLastEntry();
      }

   //##ModelId=3DC8EE48003D
   virtual ahl_tclStreamableType* poNewStreamableElement ();

   //##ModelId=3DC8EE48005B
   virtual ahl_tclStreamableType* poNewStreamableElement (const ahl_tclStreamableType &rfcoElement);

   //##ModelId=3DC7D1B80111
   virtual tVoid vDeleteEntries ();

   tU32 u32GetStreamSizeSpecial () const;

   tVoid vSerializeTypeSpecial (ahl_tclListStreamer& rfoStream);

   tVoid vFillTypeSpecial (ahl_tclListStreamer& rfoStream);

private:
   tBool _bExternalData;

};

/********************************************************************//**
 * @class       ahl_te8
 * @ingroup     Streamer
 *
 * @description Placeholder for an enumeration.
 *
 *              This class acts as a placeholder for enumerations.
 *              Using a dedicated class guarantees that the correct
 *              amount of bytes is read from or written to the data
 *              stream.
 *
 * @author      CM-AM/ENG-DI-Streitfeld
 ************************************************************************/
//##ModelId=3DE4F7DA00B4
class ahl_te8
{
public:
   ahl_te8()
      :_u8Enum(0)
      {
      }

   virtual ~ahl_te8()
      {
      }


   //##ModelId=3DECA75400F9
   operator tU8 () const
      {
         return _u8Enum;
      }


   tBool operator == (const ahl_te8 &roRight) const
      {
         return (_u8Enum == roRight._u8Enum);
      }

   tBool operator != (const ahl_te8 &roRight) const
      {
         return (_u8Enum != roRight._u8Enum);
      }

   //##ModelId=3E19998A0326
   tVoid vSetEnum (tU8 u8Enum)
      {
         _u8Enum = u8Enum;
      }

protected:
    // Data Members for Class Attributes

   /** The enumeration value is stored in this member. */
   tU8 _u8Enum;
};

/********************************************************************//**
 * @class       ahl_tb8
 * @ingroup     Streamer
 *
 * @description Placeholder for a bit field.
 *
 *              This class acts as a placeholder for bit field data.
 *              Using a dedicated class instead of the underlying
 *              data type (tU8) allows for explicit handling in
 *              a stream.
 *
 * @question    Could/should this be refactored with the ahl_Bitmap
 *              template from ahl_ports.h?
 *
 * @author      CM-AM/ENG-DI-Streitfeld
 ************************************************************************/
//##ModelId=3DEDEFF402DF
class ahl_tb8
{
public:
   enum tenum
   {
      EMPTY = 0x00
   };

   ahl_tb8()
      : _u8Bitfield(0)
      {
      }

   virtual ~ahl_tb8()
      {
      }


   //##ModelId=3DEDF08203DF
   operator tU8 () const
      {
         return _u8Bitfield;
      }

   //##ModelId=3E19996C0020
   tVoid vSetBitfield (tU8 u8Bitfield)
      {
         _u8Bitfield = u8Bitfield;
      }

  protected:
    // Data Members for Class Attributes

   /** The bitfield is stored in this member. */
   tU8 _u8Bitfield;
};

/********************************************************************//**
 * @class       ahl_tclStreamableString
 * @ingroup     Streamer
 *
 * @description A streamable string.
 *
 *              This class provides a wrapper around a plain C string
 *              that provides it with the functionality of a
 *              streamable type.
 *
 * @author      CM-AM/ENG-DI-Streitfeld
 ************************************************************************/
//##ModelId=3E2C1BE202EC
class ahl_tclStreamableString : public ahl_tclStreamableType
{
public:
   ahl_tclStreamableString();

   //##ModelId=3E2C22B101B2
   ahl_tclStreamableString (tCString rfszString);

   //##ModelId=3E2C24B900A2
   ahl_tclStreamableString (const ahl_tclStreamableString &right);

   virtual ~ahl_tclStreamableString();


   //##ModelId=3E2C229601DB
   virtual tU32 u32GetStreamSize () const;

   //##ModelId=3E2C22960218
   virtual tVoid vSerializeType (ahl_tclListStreamer& rfoStream);

   //##ModelId=3E2C2296024A
   virtual tVoid vFillType (ahl_tclListStreamer& rfoStream);
   virtual tVoid vSetString(tCString pString);

    // Data Members for Class Attributes

   /** The concrete string that is wrapped by this class. */
   tString _szString;
};

/********************************************************************//**
 * @class       ahl_tclLessAllocStreamableString
 * @ingroup     Streamer
 *
 * @description A streamable string.
 *
 *              This class provides a wrapper around a plain C string
 *              that provides it with the functionality of a
 *              streamable type. It encapsulates the plain C string
 *              and uses a fixed buffer for small strings
 *
 * @author      CM-AM/ENG-DI-Streitfeld
 ************************************************************************/
class ahl_tclLessAllocStreamableString : public ahl_tclStreamableType
{
public:
   ahl_tclLessAllocStreamableString();

   ahl_tclLessAllocStreamableString (tCString szString);

   ahl_tclLessAllocStreamableString (const ahl_tclStreamableString &right);

   ahl_tclLessAllocStreamableString (const ahl_tclLessAllocStreamableString &right);

   virtual ~ahl_tclLessAllocStreamableString();

   ahl_tclLessAllocStreamableString & operator = (const ahl_tclLessAllocStreamableString &rfoRight);

   virtual tU32 u32GetStreamSize () const;

   virtual tVoid vSerializeType (ahl_tclListStreamer& rfoStream);

   virtual tVoid vFillType (ahl_tclListStreamer& rfoStream);

   tCString szGetString() const
   {
      return _szString;
   }

    // Data Members for Class Attributes

private:

   /** The concrete string that is wrapped by this class. */
   tString _szString;

   enum
   {
      SHORTSTRINGLENGTH = 4
   };
   tChar _szShortString[SHORTSTRINGLENGTH];
};

/********************************************************************//**
 * @typedef     tdefStringList
 * @ingroup     Streamer
 *
 * @description A short-hand name for an instantiation of
 *              ahl_tclTemplateList with ahl_tclLessAllocStreamableString.
 ************************************************************************/
typedef ahl_tclTemplateList< ahl_tclLessAllocStreamableString  > tdefStringList;

//##ModelId=3E2C1BC20124

/********************************************************************//**
 * @class       ahl_tclStringList
 * @ingroup     Streamer
 *
 * @description A list of strings.
 *
 *              This class defines a streamable type that manages
 *              a list of strings.
 *
 * @author      CM-AM/ENG-DI-Streitfeld
 ************************************************************************/
class ahl_tclStringList : public tdefStringList
{
public:
   ahl_tclStringList();

   virtual ~ahl_tclStringList();
};

// Parameterized Class ahl_tclTemplateList


template <class ahl_tElementType>
ahl_tclTemplateList<ahl_tElementType>::ahl_tclTemplateList()
: _bExternalData(FALSE)
{
   ;
}

template <class ahl_tElementType>
ahl_tclTemplateList<ahl_tElementType>::ahl_tclTemplateList (const ahl_tclTemplateList<ahl_tElementType> &rhs)
:_bExternalData(FALSE)   // data is really created and must be destroyed
{
   ahl_tElementType *poActElement;
   rhs.vRewind();
   while (0 != (poActElement = rhs.poGetSequential()))
   {
      poNewElement(*poActElement);
   }
}


template <class ahl_tElementType>
ahl_tclTemplateList<ahl_tElementType>::~ahl_tclTemplateList()
{
   vDeleteEntries();
}



template <class ahl_tElementType>
ahl_tclTemplateList<ahl_tElementType>& ahl_tclTemplateList<ahl_tElementType>::operator = (const ahl_tclTemplateList<ahl_tElementType> &rhs)
{
   if (this == &rhs)
   {
      return *this;
   }

   vDeleteEntries();

   vRewind();
   rhs.vRewind();

   ahl_tElementType *poActElement;
   while (0 != (poActElement = rhs.poGetSequential()))
   {
      poNewElement(*poActElement);
   }

   // data is really created and must be destroyed
   _bExternalData = FALSE;

   return *this;
}

template <class ahl_tElementType>
ahl_tElementType * ahl_tclTemplateList<ahl_tElementType>::poNewElement ()
{
   ahl_tElementType *poElement = OSAL_NEW ahl_tElementType();

   if (FALSE == bAppend(poElement))
   {
      // avoid memleaks if element could not be added to list
      OSAL_DELETE poElement;
      poElement = 0;
   }

   return poElement;
}

template <class ahl_tElementType>
ahl_tElementType * ahl_tclTemplateList<ahl_tElementType>::poNewElement (const ahl_tElementType &rfcoElement)
{
   ahl_tElementType *poElement = OSAL_NEW ahl_tElementType(rfcoElement);

   if (FALSE == bAppend(poElement))
   {
      // avoid memleaks if element could not be added to list
      OSAL_DELETE poElement;
      poElement = 0;
   }

   return poElement;
}

template <class ahl_tElementType>
ahl_tclStreamableType* ahl_tclTemplateList<ahl_tElementType>::poNewStreamableElement ()
{
   return (ahl_tclStreamableType *)poNewElement();
}

template <class ahl_tElementType>
ahl_tclStreamableType* ahl_tclTemplateList<ahl_tElementType>::poNewStreamableElement (
   const ahl_tclStreamableType &rfcoElement)
{
   return (ahl_tclStreamableType *)poNewElement((const ahl_tElementType&)rfcoElement);
}

template <class ahl_tElementType>
tVoid ahl_tclTemplateList<ahl_tElementType>::vDeleteEntries ()
{
   ahl_tElementType *poFirstEntry = 0;
   while (0 != (poFirstEntry = poGetFirstElement()))
   {
      vRemoveFirstEntry();
      if(_bExternalData == FALSE)
      {
         OSAL_DELETE poFirstEntry;
         poFirstEntry=0;
      }
   }
}

template <class ahl_tElementType>
tU32 ahl_tclTemplateList<ahl_tElementType>::u32GetStreamSizeSpecial () const
{
   tU32 u32ReturnValue = 4; // list of
   vRewind();
   ahl_tElementType *poEntry;
   while (0 != 
      (poEntry = poGetSequential()))
   {
      u32ReturnValue += poEntry->u32GetStreamSizeSpecial();
   }

   return u32ReturnValue;
}

template <class ahl_tElementType>
tVoid ahl_tclTemplateList<ahl_tElementType>::vSerializeTypeSpecial (ahl_tclListStreamer& rfoStream)
{
   tU32 u32NumberOfEntries = (tU32)u16GetElementNr();
   rfoStream << u32NumberOfEntries;

   ahl_tElementType *poEntry;
   vRewind();

   while (0 != (poEntry = poGetSequential()))
   {
      poEntry->vSerializeTypeSpecial(rfoStream);
   }
}

template <class ahl_tElementType>
tVoid ahl_tclTemplateList<ahl_tElementType>::vFillTypeSpecial (ahl_tclListStreamer& rfoStream)
{
   vDeleteEntries();

   tU32 u32NumberOfEntries;
   ahl_tElementType *poEntry;

   rfoStream >> u32NumberOfEntries;

   OSAL_vAssert(u32NumberOfEntries <= 0xFFFF);

   while (u32NumberOfEntries-- != 0)
   {
      poEntry = poNewElement();
      if (0 != poEntry)
      {
         poEntry->vFillTypeSpecial(rfoStream);
      }
   }
}

#endif
