/********************************************************************//**
 * @file         ahl_Sequencer.h
 * @ingroup      Sequencer
 *
 * @component    Application Help Library
 *
 * @description  This file defines classes the make up the sequencer
 *               framework.
 *
 * @author       CM-AM/ENG-DI-Streitfeld
 * @copyright    2002,2006 Blaupunkt GmbH
 ************************************************************************/
#ifndef ahl_Sequencer_h
#define ahl_Sequencer_h

#define AHL_S_IMPORT_INTERFACE_LIST     /*!< Import AHL list interface. */
#include "ahl_if.h"

class ahl_tclSequenceAction;
class ahl_tclListStreamer;

/********************************************************************//**
 * @typedef     tvFctPointer
 * @ingroup     Sequencer
 *
 * @description Perform user-defined sequential action.
 *
 *              This type defines the interface for functions that
 *              perform the user-defined parts of sequential actions,
 *              like interpreting message contents and setting the
 *              stage for further steps.
 *
 * @param       pvArg  (->I) \n
 *                 Some arbitrary data.  This has been passed by the
 *                 application to the sequencer during registration of
 *                 the function, and is transparently passed on to the
 *                 invocation of this function.
 *
 * @param       poData (->I) \n
 *                 A data stream that the sequence works on.
 *
 * @param       u8ID     (I) \n
 *                 Identifier for the sequence this call belongs to.
 *
 * @return      -
 ************************************************************************/
typedef tVoid (* tvFctPointer)(tVoid* pvArg,
                               ahl_tclListStreamer * poData, tU8 u8ID);

/********************************************************************//**
 * @typedef     tbFctPointer
 * @ingroup     Sequencer
 *
 * @description Type of predicate functions.
 *
 *              This type defines the interface of predicate functions.
 *              They are used, for example, to check pre- and
 *              postconditions of a sequential step.
 *
 * @param       pvArg (->I) \n
 *                 Some arbitrary data the predicate works on.  This has
 *                 been passed by the application to the sequencer
 *                 during registration of the function, and is
 *                 transparently passed on to the invocation of this
 *                 function.
 *
 * @return      @c TRUE if the predicate is fulfilled,
 *              @c FALSE otherwise.
 ************************************************************************/
typedef tBool (* tbFctPointer)(tVoid* pvArg);

/** @name    Sequence status
 *  @ingroup Sequencer
 *
 * Different states of a sequencer.
 */
/*@{*/
/** The status of the sequence is unknown. */
#define AHL_SEQUENCE_STATUS_NOTAVAIL   0
/** The sequence has not been started yet. */
#define AHL_SEQUENCE_STATUS_NOTSTARTED 1
/** The sequence is currently being executed. */
#define AHL_SEQUENCE_STATUS_RUNNING    2
/** The sequence has been stopped during execution. */
#define AHL_SEQUENCE_STATUS_STOPPED    3
/** The sequence has reached its end. */
#define AHL_SEQUENCE_STATUS_FINISHED   4
/*@}*/


//##ModelId=3CC5720A036A
class ahl_tclSequenceResult 
{
  public:
      //##ModelId=3CEA29DD024A
      ahl_tclSequenceResult (tU16 u16FID);

      //##ModelId=3D09F68100C8
      ahl_tclSequenceResult (tU16 u16FID, tVoid* pvPtr2Object, tbFctPointer pbFctPointer);

      virtual ~ahl_tclSequenceResult();


      //##ModelId=3D09F825032E
      tBool bCheckResult ();

    // Data Members for Class Attributes


      tU16 _u16FunctionID;

      tBool _bFulfilled;

  private:
    // Data Members for Class Attributes


      tbFctPointer _pbFctPointer;

      tVoid* _pvPtr2Object;

};

//##ModelId=3CC56F7E039F
class ahl_tclSequenceAction 
{
  public:
      //##ModelId=3CC7D0E3035D
      ahl_tclSequenceAction (ahl_tclListStreamer *poParameter, tVoid* pvPtr2Object, tvFctPointer pvFctPointer, tBool bLocalAllocatedParameter);

      //##ModelId=3CEA346C039B
      ahl_tclSequenceAction (tVoid* pvPtr2Object, tvFctPointer pvFctPointer);

      virtual ~ahl_tclSequenceAction();


      //##ModelId=3CC57162016A
      tBool bExecute ();

      //##ModelId=3CFF76B80189
      tVoid vResetAction ();

      //##ModelId=3CF1F4650007
      tVoid vAddPrecondition (tU16 u16FID);

      //##ModelId=3D09F784002A
      tVoid vAddPrecondition (tVoid* pvPtr2Object, tbFctPointer pbFctPointer);

      //##ModelId=3CF1F57E0015
      tBool bFulfillPrecondition (tU16 u16FID = 0);

      //##ModelId=3D0460C502E6
      tVoid vSetResultParameter (ahl_tclListStreamer *poParameter);

      //##ModelId=3F3A3FC30339
      tVoid vMarkActionAsDone ()
      {
         _bDone = TRUE;
      }

      //##ModelId=3F7AB64E02FF
      tVoid vSetPrecondsAsFulfilled ()
      {
         _bPreconditions = FALSE;
      }

      //##ModelId=3F3A434D01FB
      tvFctPointer pvGetFctPointer () const
      {
         return _pvFctPointer;
      }

    // Data Members for Class Attributes


      tU8 _u8SeqID;

    // Data Members for Associations


      ahl_tclListStreamer *_poParameter;

  private:
    // Data Members for Class Attributes


      tvFctPointer _pvFctPointer;

      tVoid* _pvPtr2Object;

      tBool _bDone;

      tBool _bPreconditions;

      tBool _bLocalAllocatedParameter;

    // Data Members for Associations


      ahl_tclList _oPrecondList;

};

//##ModelId=3CC56E7700F7
class ahl_tclSequence 
{
  public:
      //##ModelId=3CC7CB9A0285
      ahl_tclSequence (tU8 u8SeqID);

      virtual ~ahl_tclSequence();


      //##ModelId=3CC56EEF029F
      ahl_tclSequenceAction * poAddAction (ahl_tclListStreamer *poParameter, tVoid* pvPtr2Object, tvFctPointer pvFctPointer, tBool bLocalAllocatedParameter);

      //##ModelId=3CEA33DA037D
      ahl_tclSequenceAction * poAddAction (tVoid* pvPtr2Object, tvFctPointer pvFctPointer);

      //##ModelId=3CFF70490112
      tVoid vStartSequence ();

      //##ModelId=3CFF724400C1
      tVoid vResetSequence ();

      //##ModelId=3CFF7A9D01A0
      tVoid vAbortSequence ();

      //##ModelId=3CC56EF400ED
      tVoid vExecute ();

      //##ModelId=3F3A44FD02F7
      static tVoid vExecuteIfWaiting (tVoid *vPtr2Object, ahl_tclListStreamer *poParameter, tU8 u8SeqID);

      //##ModelId=3CC572F4010E
      tBool bSequenceResult (tU16 u16FID);

      //##ModelId=3D00AF7B025F
      tBool bSequenceResult (tU16 u16FID, ahl_tclListStreamer *poParameter);

      //##ModelId=3F292F120212
      tBool bIsRunning () const
      {
         return _bIsRunning;
      }

      //##ModelId=3F3A34B40119
      tVoid vSetLastExecutedAction (tvFctPointer pvFctPointer);

      //##ModelId=3F3A36990329
      static tBool bCheckSequenceNotRunning (tVoid *vPtr2Object);

    // Data Members for Class Attributes


      tU8 _u8Status;

      tU8 _u8SeqID;

  private:
    // Data Members for Class Attributes


      tU8 _u8SeqCtr;

      tBool _bIsRunning;

    // Data Members for Associations


      ahl_tclList _oActionList;

      ahl_tclSequenceAction *_poCurrentAction;

};

//##ModelId=3CC69D7C022C

class ahl_tclSequencer 
{
  public:
      ahl_tclSequencer();

      virtual ~ahl_tclSequencer();


      //##ModelId=3CC69F7E03BB
      ahl_tclSequence* poAddSequence (tU8 u8SeqID);

      //##ModelId=3CC7CC59004F
      ahl_tclSequence* poFindSequence (tU8 u8SeqID) const;

  private:
    // Data Members for Associations


      ahl_tclList _oSequenceList;

};


#endif
