/*****************************************************************************
 * FILE:         ahl_password.cpp
 * PROJECT:      ELeNa
 * SW-COMPONENT: Help Library
 *----------------------------------------------------------------------------
 *
 * DESCRIPTION:  function for password handling 
 *              
 *----------------------------------------------------------------------------
 * COPYRIGHT:    (c) 2003 Robert Bosch GmbH, Hildesheim
 * HISTORY:      
 * Date        | Author        | Modification
 * 27-Aug-2008 | Jeryn Mathew  | Added the support for Nissan device PIN
 * 26-Sep-2008 | Jeryn Mathew  | Added new interfaces for Nissan device PIN
 *             |               | for diagnosis purposes.
 *****************************************************************************/

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define SYSTEM_S_IMPORT_INTERFACE_KDS_DEF
#include "system_pif.h"

#include "ahl_Password.h"

#ifdef VARIANT_S_FTR_ENABLE_DECRYPTION
#include "bpcl.h"
#include "bpcl_int.h"
#endif

/******************************************************************************
* Defines and macros
******************************************************************************/

#define AHL_PASSWORD_LENGTH         8
#define AHL_PASSWORD_LENGTH_NISSAN  4
#define AHL_MASTERW_BASE            (tU32)3628801
#define AHL_PASSW_BASE              (tU32)39916801
#define KDS_EDC_LENGTH              32  //256 bits
#define KDS_AES_KEY_LENGTH          16  //128 bits 
#define OEDT_TESTING                0   // Set this to 1 to run test Nissan Device PIN

#ifndef KDS_TARGET_SECURITY
#define KDS_TARGET_SECURITY         (0x12)
#endif
#define KDS_TYPE_EDC                (0x01)
#define KDS_TYPE_AES_DEV_KEY        (0x02)
//#define KDS_TYPE_CODE   0x0104

/*******************************************************************************
* Function Prototypes
*******************************************************************************/
/****************************************************************************
Function Name : ahl_bReadKDSEntry 

Description   : This function Reads the KDS entry corresponding 
                to the KDS type code, and returns the corresponding 
                values stored in it.
                          
Arguments     : Input  -   tEntry eTypeCode    -  KDS Entry Type Code 
                                                  identifying the Index
                Output -   tPU8 pu8Entry       -  Buffer to hold the 
                                                  KDS Entry read.
                Input  -   tU8  u8EntryLength  -  Length of the KDS Entry

Return        : TRUE   -  Success
                FALSE  -  Failure/Error
****************************************************************************/
static tBool ahl_bReadKDSEntry (tEntry eTypeCode, tPU8 pu8Entry, 
                                tU16   u16EntryLength);
/*****************************************************************************
Function Name : ahl_vDecryptAES

Description   : This function will decrypt the ciphertext encrypted 
                using AES Algorithm

Parameter     : *pau8Key        - Buffer containing the AESKey         - Input
                *pau8CipherText - Buffer containing the Encrypted Data - Input
                *pau8PlainText  - Buffer to store the Decrypted result - Output

Return Value  : None
*****************************************************************************/
static tVoid ahl_vDecryptAES   (tPU8   pau8Key, tPU8 pau8CipherText, 
                                tPU8   pau8PlainText );
/******************************************************************************
Function Name : ahl_s16ComputePIN

Description   : This function will compute the PIN from the Certificate
                passed as argument

Parameter     : *pu8Certificate - Input -  Buffer to store the Unencrypted 
                                           Certificate
Return Value  : s16PIN - The 4 digit PIN number
******************************************************************************/
static tU16 ahl_u16ComputePIN  (tPCU8  pu8Certificate );
/*****************************************************************************
Function Name : ahl_s16Compute_Device_PIN

Description   : This function will Read the Encrypted Device Certificate, 
                decrypt it, and then Compute PIN information out of this.

Parameter     : ps16PIN  - The 4 digit PIN number

Return Value  : s16ErrCode  - The 4 digit PIN number
*****************************************************************************/
static tS16 ahl_s16Compute_Device_PIN (tPU16 pu16PIN);
/******************************************************************************
*Function Name : ahl_bSetCopyOfPIN

Description      :  Read the original PIN from KDS and write a copy to FFS

Parameter        : None

Return Value    : TRUE   - PIN deactivated.
                            FALSE - PIN deactivation failed. 
*******************************************************************************/
static tBool ahl_bSetCopyOfPIN (tVoid); /* if TRUE, I can set a password in coded form */
                                       /* Set a password to eeprom */
/******************************************************************************
*FUNCTION       : ahl_bReadPasswordFromKdsNissan
*
*DESCRIPTION    : This function reads the password from KDS
*
*PARAMETER      : pau8Password - Pointer to buffer containing the PIN
*                 u8Length     - Length of Device PIN.
*
*RETURN VALUE   : TRUE  - PIN deactivated.
*                 FALSE - PIN deactivation failed. 
*******************************************************************************/
static tBool ahl_bReadPasswordFromKdsNissan (tPU8 pau8Password,
                                             tU8  u8Length);
/*****************************************************************************/
/******************************************************************************
*FUNCTION:          ahl_vPassAlg
*
*DESCRIPTION:       This function converts a given password into a fixed size 
*                   coded password                    
*******************************************************************************/

tVoid
ahl_vPassAlg(                       /* Compute password */
    tU32  u32MagicBase,             /* Base for the password compution */
    tU8   u8Length,                 /* Length of password */
    tPCU8 pau8Passw,                /* Pointer to password */
    tPU8  pau8Set)                  /* Pointer to coded password */
/*-----------------------------------------------------------------------------------------
 * @returnvalue
 *-----------------------------------------------------------------------------------------*/
{
    tU32 u32Base = 0;
    tU8  u8Index;
    tU8  u8Check = 0;

    for (u8Index=0 ; u8Index<u8Length ; ++u8Index)
    {
        u8Check+=pau8Passw[u8Index];
    }
    for (u8Index=0 ; (u8Index < u8Length) && (u8Index < AHL_PASSWORD_LENGTH) ; ++u8Index)
    {
      u32Base+= pau8Passw[u8Index] * pau8Passw[(u8Check+pau8Passw[u8Index])%u8Length];
      pau8Set[u8Index] = (tU8) ((((tU8 *)&u32MagicBase)[u8Index % 4]-u32Base) % 256);
    }

    for (; u8Index<AHL_PASSWORD_LENGTH ; ++u8Index)
    {
      u32Base+= pau8Set[u8Index-u8Length]*pau8Set[(u8Check+pau8Set[u8Index-u8Length])%u8Length];
      pau8Set[u8Index]=(tU8)((((tU8 *)&u32MagicBase)[u8Index % 4]-u32Base) % 256);
    }
    return;
}

/******************************************************************************
*FUNCTION:          ahl_bWritePasswordToKds
*
*DESCRIPTION:       This function converts a tU32 big endian value into a
*                                       little endian value
*******************************************************************************/
tBool
ahl_bWritePasswordToKds(
                        tPCU8  pau8Password,
                        tU8   u8Length)
{
   OSAL_tIODescriptor hKDS = 0;
   hKDS = OSAL_IOOpen(	OSAL_C_STRING_DEVICE_KDS, OSAL_EN_READWRITE);

   if ((hKDS == 0) || (hKDS == OSAL_ERROR))
   {
      return (FALSE);
   }
   
   (tVoid) OSAL_s32IOControl(hKDS, OSAL_C_S32_IOCTRL_KDS_WRITE_ENABLE, TRUE);
   
   tsKDSEntry tKDSEntry;
   tS32       s32OsalError;
   tKDSEntry.u16Entry = M_KDS_ENTRY(KDS_TARGET_DEVICE, KDS_TYPE_CODE);
   tKDSEntry.u16EntryLength = u8Length;
   tKDSEntry.u16EntryFlags = M_KDS_ENTRY_FLAG_NONE;

   tU8 u8Counter;
   for (u8Counter=0; u8Counter < u8Length; u8Counter++)
   {
      tKDSEntry.au8EntryData[u8Counter] = pau8Password[u8Counter];
   }

   s32OsalError = OSAL_s32IOWrite( hKDS,
      (tPCS8)&tKDSEntry,
      sizeof(tKDSEntry));

   if( s32OsalError == OSAL_ERROR )
   {
   /* ERROR: Error returned, check error value to check what
      happened */;
      (tVoid) OSAL_s32IOClose(hKDS); 
      return (FALSE);
   }
   
   (tVoid) OSAL_s32IOClose(hKDS); 
   
   return TRUE;
}

/******************************************************************************
*FUNCTION:          ahl_bIsPasswordInKds
*
*DESCRIPTION:       This function tries to read the password from KDS,
*                   If there is one, TRUE will be returned
*******************************************************************************/
tBool
ahl_bIsPasswordInKds(tVoid)
{
   OSAL_tIODescriptor hKDS = 0;
   hKDS = OSAL_IOOpen(	OSAL_C_STRING_DEVICE_KDS, OSAL_EN_READONLY);

   if ((hKDS == 0) || (hKDS == OSAL_ERROR))
   {
      return (FALSE);
   }

   tsKDSEntry tKDSEntry;
   tS32       s32OsalError;
   tKDSEntry.u16Entry = M_KDS_ENTRY(KDS_TARGET_DEVICE, KDS_TYPE_CODE);
   tKDSEntry.u16EntryLength = sizeof(tKDSEntry.au8EntryData);
   tKDSEntry.au8EntryData[0] = 0;
   
   s32OsalError = OSAL_s32IORead(hKDS,
                                 (tPS8)&tKDSEntry,
                                 sizeof(tKDSEntry));
  
   if( s32OsalError != OSAL_ERROR )
   {
      (tVoid) OSAL_s32IOClose(hKDS); 
      return TRUE;
   }
   else
   {
     /* ERROR: Error returned, check error value to check what
               happened */
//      tU32 u32OSALError = OSAL_u32ErrorCode ();
//
//      if (u32OSALError == KDS_Result_Not_In_List)
      {
         (tVoid) OSAL_s32IOClose(hKDS); 
         return FALSE;
      }
   }
   
  //return TRUE;  ABR Compiler warning
}

/******************************************************************************
*FUNCTION:          ahl_bReadPasswordFromKds
*
*DESCRIPTION:       This function reads the password from KDS
*******************************************************************************/
tBool
ahl_bReadPasswordFromKds(
                        tPU8  pau8Password,
                        tU8   u8Length)
{
   OSAL_tIODescriptor hKDS = 0;
   hKDS = OSAL_IOOpen(	OSAL_C_STRING_DEVICE_KDS, OSAL_EN_READONLY);

   if ((hKDS == 0) || (hKDS == OSAL_ERROR))
   {
      return (FALSE);
   }

   tsKDSEntry tKDSEntry;
   tS32       s32OsalError;
   tKDSEntry.u16Entry = M_KDS_ENTRY(KDS_TARGET_DEVICE, KDS_TYPE_CODE);
   tKDSEntry.u16EntryLength = sizeof(tKDSEntry.au8EntryData);
   tKDSEntry.au8EntryData[0] = 0;
   
   s32OsalError = OSAL_s32IORead(hKDS,
                                 (tPS8)&tKDSEntry,
                                 sizeof(tKDSEntry));
  
   if( s32OsalError != OSAL_ERROR )
   {
      if( (tKDSEntry.u16Entry != M_KDS_ENTRY(KDS_TARGET_DEVICE, KDS_TYPE_CODE)) &&
          (tKDSEntry.u16EntryLength != u8Length) )
     {
        /* ERROR: NOT the expected data received */;
         (tVoid) OSAL_s32IOClose(hKDS); 
         return FALSE;
     }
   }
   else
   {
     /* ERROR: Error returned, check error value to check what
               happened */;
      (tVoid) OSAL_s32IOClose(hKDS); 
      return FALSE;
   }
   
   (tVoid) OSAL_pvMemoryCopy(pau8Password,tKDSEntry.au8EntryData, u8Length);

   (tVoid) OSAL_s32IOClose(hKDS);   
   return TRUE;
}

/******************************************************************************
*FUNCTION:          ahl_bWritePasswordToFfs
*
*DESCRIPTION:       This function writes the password to FFS
*******************************************************************************/
tBool
ahl_bWritePasswordToFfs(
                        tPCU8  pau8Password,
                        tU8   u8Length)
{
   tBool bReturnValue = TRUE;

   OSAL_tIODescriptor hFFS = 0;

   // First try to open an existing file
   hFFS = OSAL_IOOpen(OSAL_C_STRING_DEVICE_FFS2"/Password.dat", OSAL_EN_READWRITE);

   //tU32 u32ErrorCode = OSAL_u32ErrorCode(); //ABR lint warning u32Errorcode

   if (hFFS == OSAL_ERROR)
   {
//      if (u32ErrorCode == OSAL_E_DOESNOTEXIST)
      {
         // File does not exist
         hFFS = OSAL_IOCreate(OSAL_C_STRING_DEVICE_FFS2"/Password.dat", OSAL_EN_READWRITE);

         if (hFFS == OSAL_ERROR)
         {
            return (FALSE);
         }
      }
//      else  // Unknown error
//      {
//         return FALSE;
//      }
   }
   
   tS32 s32OsalError = OSAL_s32IOWrite(hFFS, (tPS8)pau8Password, u8Length);
  
   if(s32OsalError == OSAL_ERROR)
   {
      bReturnValue = FALSE;
   }
   
   (tVoid) OSAL_s32IOClose(hFFS);

#ifdef AHL_PASSWORD_USE_SAVENOW
   /* sequence is not necessary anymore, because of flush in case of close of file 
   doing this can leave invalid data in other (!!!) files, because of flush
   in the middle of a transaction */

   if (bReturnValue)
   {
      hFFS = OSAL_IOOpen( OSAL_C_STRING_DEVICE_FFS2, OSAL_EN_READWRITE);
      
      if( hFFS != OSAL_ERROR)
      {
		 s32OsalError = OSAL_s32IOControl(hFFS, OSAL_C_S32_IOCTRL_FFS_SAVENOW, 0);

         if (s32OsalError == OSAL_ERROR)
         {
            bReturnValue = FALSE;
         }
         
         s32OsalError = OSAL_s32IOClose(hFFS);
      }
   }
#endif
   return (bReturnValue);
}

/******************************************************************************
*FUNCTION:          ahl_bReadPasswordFromFfs
*
*DESCRIPTION:       This function tries to read the password from FFS
*******************************************************************************/
tBool
ahl_bReadPasswordFromFfs(
                        tPU8  pau8Password,
                        tU8   u8Length)
{
   tBool bReturnValue = TRUE;

   OSAL_tIODescriptor hFFS = 0;

   // First try to open an existing file
   hFFS = OSAL_IOOpen(OSAL_C_STRING_DEVICE_FFS2"/Password.dat", OSAL_EN_READWRITE);

   if (hFFS == OSAL_ERROR)
   {
      return FALSE;
   }
   
   tS32 s32OsalError = OSAL_s32IORead(hFFS, (tPS8)pau8Password, u8Length);
  
   if(s32OsalError != u8Length)
   {
      bReturnValue = FALSE;
   }
   
   (tVoid) OSAL_s32IOClose(hFFS);
   
   return (bReturnValue);
}

/******************************************************************************
*FUNCTION:          ahl_bSetCopyOfPassword
*
*DESCRIPTION:       Read the original from KDS and write a copy to FFS
*******************************************************************************/
tBool                                   /* if TRUE, I can set a password in coded form */
ahl_bSetCopyOfPassword(tVoid)           /* Set a password to eeprom */
/*-----------------------------------------------------------------------------------------
 * @returnvalue If TRUE, I can set a password in coded form
 *-----------------------------------------------------------------------------------------*/
{
   tU8 au8Password[AHL_PASSWORD_LENGTH];

   if (ahl_bReadPasswordFromKds(au8Password, AHL_PASSWORD_LENGTH) == FALSE)
   /* if the master data cannot be read */
   {
      return(FALSE);
   }

   tU8 au8PasswordCopy[AHL_PASSWORD_LENGTH];
   ahl_vPassAlg(AHL_PASSW_BASE,AHL_PASSWORD_LENGTH,au8Password,au8PasswordCopy);

   if (ahl_bWritePasswordToFfs(au8PasswordCopy,AHL_PASSWORD_LENGTH)) /* if I can set the password */
   {
      return(TRUE);                   /* setting has success */
   }
   else
   {
      return(FALSE);                  /* setting is not correct */
   }
}

/******************************************************************************
*FUNCTION:          ahl_bActivatePassword
*
*DESCRIPTION:       This functions activates the password
*******************************************************************************/

tBool ahl_bActivatePassword(tVoid)
{
   tBool bReturnValue = TRUE;

   // Delete file in FFS
   tS32 s32OsalError;

   s32OsalError = OSAL_s32IORemove(OSAL_C_STRING_DEVICE_FFS2"/Password.dat");

   if (s32OsalError != OSAL_ERROR)
   {
#ifdef AHL_PASSWORD_USE_SAVENOW
/*
   sequence is not necessary anymore, because of flush in case of close of file 
   doing this can leave invalid data in other (!!!) files, because of flush
   in the middle of a transaction */

      OSAL_tIODescriptor hFFS = OSAL_IOOpen( OSAL_C_STRING_DEVICE_FFS2, OSAL_EN_READWRITE);
      
      if( hFFS != OSAL_ERROR)
      {
         s32OsalError = OSAL_s32IOControl(hFFS, OSAL_C_S32_IOCTRL_FFS_SAVENOW, 0);

         if (s32OsalError == OSAL_ERROR)
         {
            bReturnValue = FALSE;
         }
         
         s32OsalError = OSAL_s32IOClose(hFFS);
      }
#endif
   }
   else
   {
      bReturnValue = FALSE;
   }

   return(bReturnValue);
}


/******************************************************************************
*FUNCTION:          ahl_bDeactivatePassword
*
*DESCRIPTION:       This function deactivates the password
*******************************************************************************/

tBool ahl_bDeactivatePassword(tVoid)
{
    if (!ahl_bSetCopyOfPassword())
    {
        return(FALSE);
    }
    else
    {
        return(TRUE);
    }
}

/******************************************************************************
*FUNCTION:          ahl_bSetPassword
*
*DESCRIPTION:       This function stores the password in the KDS 
*                   and deactivates it
*******************************************************************************/

tBool                            /* if TRUE, I can set a password in coded form */
ahl_bSetPassword(                /* Set password and copy of password */
    tPCU8  pau8Passw,             /* Pointer to the password buffer */
    tU8   u8Length)              /* Length to be interpreted */
/*-----------------------------------------------------------------------------------------
 * @returnvalue If TRUE, I can set a mastercode
 *-----------------------------------------------------------------------------------------*/

{
    tBool bRetval;

    if (pau8Passw == NULL)                /* if clear match with mcode */
    {
        return(FALSE);            /* indicate no success */
    }

    tU8 au8CodedPassword[AHL_PASSWORD_LENGTH];

    ahl_vPassAlg(AHL_MASTERW_BASE,u8Length,pau8Passw,au8CodedPassword);

    bRetval = ahl_bWritePasswordToKds(au8CodedPassword,AHL_PASSWORD_LENGTH);

//    if (bRetval == TRUE)
//    {
//       bRetval = ahl_bSetCopyOfPassword();
//    }

    return (bRetval);
}

/******************************************************************************
*FUNCTION:          ahl_bIsPasswordActive
*
*DESCRIPTION:       This function checks if the password is activated
*******************************************************************************/

tBool ahl_bIsPasswordActive(tVoid)
{
   //tBool bReturnValue = FALSE; ABR compiler warning

   tU8 au8Password[AHL_PASSWORD_LENGTH];

   // Only check if file exists in FFS
   if (ahl_bReadPasswordFromFfs(au8Password, AHL_PASSWORD_LENGTH))
   {
      // if so, compare the original one with the copy in the FFS
      tU8 au8OriginalPassword[AHL_PASSWORD_LENGTH];

      if (ahl_bReadPasswordFromKds(au8OriginalPassword,AHL_PASSWORD_LENGTH))
      {
         tU8 au8CodedOriginal[AHL_PASSWORD_LENGTH];
         ahl_vPassAlg(AHL_PASSW_BASE,AHL_PASSWORD_LENGTH,au8OriginalPassword,au8CodedOriginal);

         tU8 u8Counter;
         for (u8Counter=0; u8Counter < AHL_PASSWORD_LENGTH; u8Counter++)
         {
            if (au8CodedOriginal[u8Counter] != au8Password[u8Counter])
            {
               return TRUE;
               //break;  ABR compiler warning
            }
         }

         return FALSE;
      }
      else
      {
         return TRUE;
      }
   }
   else  // file does not exist
   {
      return TRUE;
   }
}

/******************************************************************************
*FUNCTION:          ahl_bComparePassword
*
*DESCRIPTION:       This function compares the given password  
*                   with the one in the KDS
*******************************************************************************/
/* if TRUE, the password is ok *//* compare password with KDS */
tBool ahl_bComparePassword (tPCU8  pau8Passw, /* Pointer to the password buffer */            
							tU8   u8Length)  /* Length to be interpreted */                 
							                                  

/*-----------------------------------------------------------------------------------------
 * @returnvalue If TRUE, I can set a mastercode
 *-----------------------------------------------------------------------------------------*/

{
    if (pau8Passw == NULL)       /* if clear match with mcode */
	{
		return(FALSE);            /* indicate no success */
	}

    tU8 au8OriginalPassword[AHL_PASSWORD_LENGTH];

    if (ahl_bReadPasswordFromKds(au8OriginalPassword,AHL_PASSWORD_LENGTH))
	{
		tU8 au8CodedPassword[AHL_PASSWORD_LENGTH];

		ahl_vPassAlg(AHL_MASTERW_BASE,u8Length,pau8Passw,au8CodedPassword);

		if (OSAL_s32MemoryCompare (au8CodedPassword,
                                   au8OriginalPassword,
                                   AHL_PASSWORD_LENGTH) != 0)
		{
			return FALSE;
		}
		else
		{
			return TRUE;
		}
	}
	else
	{
		// No password in KDS?
		return FALSE;
	}
}

/*****************************************************************************
Function Name : ahl_vDecryptAES

Description   : This function will decrypt the ciphertext encrypted 
                using AES Algorithm

Parameter     : *pau8Key        - Buffer containing the AESKey         - Input
                *pau8CipherText - Buffer containing the Encrypted Data - Input
                *pau8PlainText  - Buffer to store the Decrypted result - Output

Return Value  : None
*****************************************************************************/
#ifdef VARIANT_S_FTR_ENABLE_DECRYPTION	
static tVoid ahl_vDecryptAES ( tPU8 pau8Key, 
                               tPU8 pau8CipherText, 
                               tPU8 pau8PlainText )
{
   BPCL_AES_Decrypt  ( BPCL_AES_MODE_CBC,    // BPCL_AES_MODE_*  
                        BPCL_AES_OP_ARGKEY,   // BPCL_AES_OP_*
                        BPCL_AES_KEYSIZE_128, // BPCL_AES_KEYSIZE_*
                        pau8Key,              // Buffer holding key
                        0,                    // Index of internal key to use
                        (tU8*)NULL,           // Buffer holding 128-bit IV
                        pau8CipherText,       // Buffer holding encrypted data
                        pau8PlainText,        // Buffer to carry plain data 
                        KDS_EDC_LENGTH );     // size of p_plain in bytes
#else
static tVoid ahl_vDecryptAES ( tPU8, tPU8, tPU8 )
{
#endif
   return;
}

/******************************************************************************
Function Name : ahl_s16ComputePIN

Description   : This function will compute the PIN from the Certificate
                passed as argument

Parameter     : *pu8Certificate - Input -  Buffer to store the Unencrypted 
                                           Certificate
Return Value  : s16PIN - The 4 digit PIN number
******************************************************************************/
static tU16 ahl_u16ComputePIN ( tPCU8 pu8Certificate )
{
    tU16 u16Res[4];
    tU32 u32Result;
    tU16 u16PIN;
    
    u16Res[0] = (pu8Certificate[0]  + pu8Certificate[3]) % 10;
    u16Res[1] = (pu8Certificate[6]  + pu8Certificate[9]) % 10;
    u16Res[2] = (pu8Certificate[10] + pu8Certificate[13]) % 10;
    u16Res[3] = (pu8Certificate[16] + pu8Certificate[19]) % 10;
    
    u32Result = (((u16Res[0] * 10) + u16Res[1]) * 10 + u16Res[2]) * 10 + u16Res[3];
    
    u16PIN    = (tU16)( u32Result % 10000 );
    
    return u16PIN;
}

/*****************************************************************************
Function Name : ahl_s16Compute_Device_PIN

Description   : This function will Read the Encrypted Device Certificate, 
                decrypt it, and then Compute PIN information out of this.

Parameter     : ps16PIN  - The 4 digit PIN number

Return Value  : s16ErrCode  - The 4 digit PIN number

History:
26-Sept-2008  | jev1kor |  Removed M_MASK_KEY() macro. Can avoid 
              |         | Linker error in Simulation build.
*****************************************************************************/
static tS16 ahl_s16Compute_Device_PIN (tPU16 pu16PIN)
{
    tU8   au8DevCert[KDS_EDC_LENGTH] = {0};
    tS16  s16ErrCode = 0;
        
    #if (OEDT_TESTING == 0)
    // In case of normal operation.
    /**********************************************************
    1. Read Encrypted Device Certificate, AES Key from the KDS.
    ***********************************************************/
    tU8 au8EDC[KDS_EDC_LENGTH] = {0};
    tU8 au8RandAESKey[KDS_AES_KEY_LENGTH]= {0};
    
    if (ahl_bReadKDSEntry ( KDS_TYPE_EDC, //E.D.C.
                            au8EDC, 
                            KDS_EDC_LENGTH ) == FALSE)
    {
        // Error in reading Data. Data may not exist in KDS
        s16ErrCode = -3;
        return s16ErrCode;
    }
    
    if (ahl_bReadKDSEntry ( KDS_TYPE_AES_DEV_KEY, //Random AES Key
                            au8RandAESKey, 
                            KDS_AES_KEY_LENGTH ) == FALSE)
    {
        // Error in reading AES Key> Data may not exist in KDS
        s16ErrCode = -4;
        return s16ErrCode;
    }
        
   
    #else
    //The following code is used to test the Nissan Device PIN functionality
    //The Device PIN value is 3693
    tU8 au8EDC[KDS_EDC_LENGTH]            = 
                           {0x40, 0x2D, 0xC5, 0xD2, 0x93, 0x60, 0x4A, 0x17,
                            0x00, 0x1F, 0x56, 0xC5, 0x3F, 0x18, 0x42, 0xF0,
                            0xE6, 0xE9, 0xB5, 0xED, 0x49, 0x7E, 0x8D, 0x18,
                            0x76, 0x5A, 0x03, 0x73, 0x49, 0xE8, 0xC8, 0x07};
    
    tU8 au8RandAESKey[KDS_AES_KEY_LENGTH] = 
                           {0x1B, 0x7D, 0x65, 0xD0, 0x17, 0x89, 0xD5, 0xD8, 
                            0x3B, 0x5F, 0x63, 0x98, 0xF5, 0x93, 0x0C, 0xEF};
                         
    tU8 au8FinalDevCert[KDS_EDC_LENGTH]   = 
                           {0xFD, 0x81, 0x6C, 0x14, 0x48, 0x49, 0x37, 0x64,
                            0x6C, 0x8D, 0xAE, 0x3A, 0xBD, 0xD7, 0xF7, 0x3A,
                            0x25, 0x23, 0xCD, 0xA6, 0x59, 0xB9, 0x53, 0xAB,
                            0xD0, 0xF9, 0x66, 0x60, 0xB9, 0xE8, 0xA1, 0xC0};
    #endif
    
    /*******************************************************
     2. Decrypt the Encrypted Device Certificate (EDC).
    ********************************************************/
    ahl_vDecryptAES ( au8RandAESKey, au8EDC, au8DevCert );
    
    /******  A little Housekeeping - Clearing content in buffer *****/
    (tVoid) OSAL_pvMemorySet ( au8RandAESKey, 0, KDS_AES_KEY_LENGTH );
    (tVoid) OSAL_pvMemorySet ( au8EDC, 0, KDS_EDC_LENGTH );
    /******  A little Housekeeping - Clearing content in buffer *****/
    
    
    #if (OEDT_TESTING)
    /*
     Compare if the Device certificate is same as expected.
    */
    
    if (OSAL_s32MemoryCompare ( au8DevCert, 
                                au8FinalDevCert, 
                                KDS_EDC_LENGTH ) == 0)
    {
        s16ErrCode = 0;
    }
    else
    {
        s16ErrCode = -5;
    }
    
    (tVoid) OSAL_pvMemorySet ( au8FinalDevCert, 0, KDS_EDC_LENGTH );
    /*
     This Device Certificate will return a PIN = 3693
    */
    #endif
  
    /********************************************************
            3. Compute PIN
    *********************************************************/
    *pu16PIN = ahl_u16ComputePIN (au8DevCert);
    
    /*****  A little Housekeeping - Clearing content in buffer *****/
    (tVoid) OSAL_pvMemorySet ( au8DevCert, 0, KDS_EDC_LENGTH );
    /*****  A little Housekeeping - Clearing content in buffer *****/
    
    return s16ErrCode;
}

/***************************************************************************
Function Name : ahl_bVerify_Device_PIN

Description   : This function will compare the User input PIN 
                (from HMI) and the PIN Computed from the device 
                certificate, and return a boolean TRUE or FALSE

Parameter     : tPU8  pau8Passw - buffer containing the User Input PIN
                tU8   u8Length  - Length of the PIN

Return Value  : bResult  - TRUE   - PIN is Correct
                         - FALSE  - PIN is Incorrect
                         
 Note         : This API is meant to be used for Nissan LCN Device 
 				Security feature
*****************************************************************************/
tBool ahl_bVerify_Device_PIN (tPCU8 pau8Passw, /* Pointer to password buffer*/
                              tU8   u8Length)
{
    tU16    u16DevicePIN = 0, u16UserPIN = 0;
    tS16    s16ErrCode = 0;    //This variable is used for debugging purposes.
    tBool   bResult = FALSE;
    tU8     u8Loop = 0, u8BaseVal = '0';   
    
    /***************************************************************     
    * Parameter Check                                              *
    * Case 1 : Check if password characters lie between '0' and '9'*
    * Case 2 : Check if the actual length of password is equal to  *
    *          (or greater than) length provided as argument       *
    ****************************************************************/

    while (u8Loop < u8Length)
    {
        //Case 1
        if (pau8Passw[u8Loop] >= '0' && pau8Passw[u8Loop] <='9') 
        {
            // In this case, the character is acceptable.
        }
        else
        {
            s16ErrCode = -1;
            return FALSE;
        }
        //Case 2
        if (pau8Passw[u8Loop] == '\0')
        {
            //Error. The actual length of password is less 
            //than that provided by the argument.
            s16ErrCode = -2;
            return FALSE;
        }
        u8Loop++;
    }
    /****************************************************************/    
    /* Convert the Character String to u16 number */
    for (u8Loop = 0; u8Loop < u8Length; u8Loop++)
    {
        u16UserPIN *= 10;
        // This is done because the PIN provided is a 
        //Character array of Numbers
        u16UserPIN += (pau8Passw[u8Loop] - u8BaseVal);
    }
    /****************************************************************/ 
    /* Obtain the Actual Device PIN*/
    s16ErrCode = ahl_s16Compute_Device_PIN ( &u16DevicePIN );
    /****************************************************************/ 
    /* Conduct Verify operation */
    
    if ( u16DevicePIN == u16UserPIN )
    {
        bResult = TRUE;
    }
    else 
    { 
        bResult = FALSE;
    }
    
    //To remove Compiler/Lint warnings
    s16ErrCode++; 
    s16ErrCode--;
    (tVoid)s16ErrCode;
    
    return  bResult;
}

/****************************************************************************
*FUNCTION      : ahl_bReadKDSEntry 
*
*DESCRIPTION   : This function Reads the KDS entry corresponding 
*                to the KDS type code, and returns the corresponding 
*                values stored in it.
*                          
*ARGUMENTS     : Input  -   tEntry eTypeCode    -  KDS Entry Type Code 
*                                                  identifying the Index
*                Output -   tPU8 pu8Entry       -  Buffer to hold the 
*                                                  KDS Entry read.
*                Input  -   tU8  u8EntryLength  -  Length of the KDS Entry
*
*RETURN VALUE  : TRUE   -  Success
*                FALSE  -  Failure/Error
****************************************************************************/
static tBool ahl_bReadKDSEntry (tEntry eTypeCode, //KDS Entry Type Code
                                tPU8   pu8Entry,  //Buffer to store the result
                                tU16   u16EntryLength) //Length of the Entry
{
    OSAL_tIODescriptor hKDS = 0;
    hKDS = OSAL_IOOpen ( OSAL_C_STRING_DEVICE_KDS, OSAL_EN_READONLY );

    if ((hKDS == 0) || (hKDS == OSAL_ERROR))
    {
        return (FALSE);
    }
    
    tsKDSEntry tKDSEntry;
    tS32       s32OsalError;
    tKDSEntry.u16Entry        = M_KDS_ENTRY ( KDS_TARGET_SECURITY, 
                                              eTypeCode );
    tKDSEntry.u16EntryLength  = u16EntryLength;
    tKDSEntry.au8EntryData[0] = 0;
    
    s32OsalError = OSAL_s32IORead ( hKDS,
                                    (tPS8)&tKDSEntry,
                                    sizeof (tKDSEntry) );
    
    if( s32OsalError != OSAL_ERROR )
    {
        if (( tKDSEntry.u16Entry != M_KDS_ENTRY (KDS_TARGET_DEVICE, eTypeCode)) &&
            ( tKDSEntry.u16EntryLength != u16EntryLength ))
        {
            /* ERROR: NOT the expected data received */
            (tVoid) OSAL_s32IOClose (hKDS); 
            return FALSE;
        }
    }
    else
    {
        /* ERROR: Error returned, check error value to check what
           happened 
        */
        (tVoid) OSAL_s32IOClose (hKDS); 
        return FALSE;
    }
    
    (tVoid) OSAL_pvMemoryCopy ( pu8Entry, 
                                tKDSEntry.au8EntryData, 
                                u16EntryLength );
    
    (tVoid) OSAL_s32IOClose (hKDS);
    return TRUE;
}

/******************************************************************************
*FUNCTION       : ahl_bDeactivatePasswordNissan
*
*DESCRIPTION    : This function will deactivate the PIN feature for Nissan
*
*PARAMETER      : None
*
*RETURN VALUE   : TRUE  - PIN deactivated.
*                 FALSE - PIN deactivation failed. 
*******************************************************************************/
tBool ahl_bDeactivatePasswordNissan (tVoid)
{
    if ( !ahl_bSetCopyOfPIN () )
    {
        /*Means  that either password cannot be read from KDS 
        or Cannot create/Write a Password.dat file.*/
        return(FALSE);
    }
    else
    {
        // Password.dat file created.
        return(TRUE);
    }
}

/******************************************************************************
*FUNCTION       : ahl_bSetCopyOfPIN
*
*DESCRIPTION    :  Read the original PIN from KDS and write a copy to FFS
*
*PARAMETER      : None
*
*RETURN VALUE   : TRUE  - PIN deactivated.
*                 FALSE - PIN deactivation failed. 
*******************************************************************************/
static tBool ahl_bSetCopyOfPIN (tVoid) /* if TRUE, I can set a password in coded form */
                                       /* Set a password to eeprom */
{
    tU8     au8PasswordCopy [AHL_PASSWORD_LENGTH] = {0};    
    tU8     au8Password [AHL_PASSWORD_LENGTH]     = {0};

    //Read Password from KDS
    if ( !ahl_bReadPasswordFromKdsNissan ( au8Password, 
                                           AHL_PASSWORD_LENGTH_NISSAN) )
    {
        //Cannot read from KDS
        return FALSE;
    }
    
    //Encode the PIN    
    ahl_vPassAlg ( (tU32)AHL_PASSW_BASE, AHL_PASSWORD_LENGTH,
                   au8Password, au8PasswordCopy );

    //Write the encoded password into FFS
    if ( ahl_bWritePasswordToFfs ( au8PasswordCopy, 
                                   AHL_PASSWORD_LENGTH) ) /* if I can set the password */
    {
        return TRUE;                   /* setting has success */
    }
    else
    {
        return FALSE;                  /* setting is not correct */
    }
}

/******************************************************************************
*FUNCTION       : ahl_bReadPasswordFromKdsNissan
*
*DESCRIPTION    : This function reads the password from KDS
*
*PARAMETER      : pau8Password - Pointer to buffer containing the PIN
*                 u8Length     - Length of Device PIN.
*
*RETURN VALUE   : TRUE  - PIN deactivated.
*                 FALSE - PIN deactivation failed. 
*******************************************************************************/
static tBool ahl_bReadPasswordFromKdsNissan ( tPU8  pau8Password, 
                                              tU8   u8Length)
{
    tU8     u8Loop     = 0;
    tU16    u16TempPIN = 0, u16DevPIN = 0;
    
    if ( ahl_s16Compute_Device_PIN (&u16DevPIN) < 0 )
    /* if the master data cannot be read */
    {
        return FALSE;
    }
    
    //Convert 4 digit U16 number to an array of characters
    //First get the reverse form of the PIN
    for (u8Loop = 0; u8Loop < u8Length; u8Loop++)
    {
        u16TempPIN  = u16TempPIN*10;
        u16TempPIN  = u16TempPIN + (u16DevPIN % 10);
        u16DevPIN   = u16DevPIN/10;
    }
    //Then extract each digit, convert to ASCII, and then store again in reverse in ARRAY
    for (u8Loop = 0; u8Loop < u8Length; u8Loop++)
    {
        pau8Password[u8Loop] = (tU8)(u16TempPIN % 10) + (tU8)'0';
        u16TempPIN           = u16TempPIN/10;       
    }
    //Finally Null terminate the string
    pau8Password[u8Length] = '\0';
    
    return TRUE;   
}

/******************************************************************************
*FUNCTION       : ahl_bIsPasswordActiveNissan
*
*DESCRIPTION    : This function checks if the password is activated
*
*PARAMETER      : None
*
*RETURN VALUE   : TRUE  - Password is Active.
*                 FALSE - Password is NOT active. 
*******************************************************************************/

tBool ahl_bIsPasswordActiveNissan (tVoid)
{
    //tBool bReturnValue = FALSE; ABR compiler warning
    tU8 au8Password [AHL_PASSWORD_LENGTH] = {0};
    
    // Only check if file exists in FFS
    if (ahl_bReadPasswordFromFfs (au8Password, AHL_PASSWORD_LENGTH))
    {
        // if so, compare the original one with the copy in the FFS
        tU8 au8OriginalPassword [AHL_PASSWORD_LENGTH] = {0};
        
        if (ahl_bReadPasswordFromKdsNissan (au8OriginalPassword,
                                            AHL_PASSWORD_LENGTH_NISSAN))
        {
            tU8 au8CodedOriginal [AHL_PASSWORD_LENGTH] = {0};
            
            ahl_vPassAlg ((tU32)AHL_PASSW_BASE, 
                          AHL_PASSWORD_LENGTH,
                          au8OriginalPassword,
                          au8CodedOriginal);
            
            tU8 u8Counter = 0;
            
            for (u8Counter = 0; u8Counter < AHL_PASSWORD_LENGTH; u8Counter++)
            {
                if (au8CodedOriginal[u8Counter] != au8Password[u8Counter])
                {
                    // password in FFS not same as that in KDS
                    return TRUE;
                    //break;  ABR compiler warning
                }
            }
            // Password in FFS and KDS are same
            return FALSE;
        }
        else
        {
            //Cannot read from KDS.
            return TRUE;
        }
    }
    else  
    {
        // file does not exist
        return TRUE;
    }
}

/******************************************************************************
*FUNCTION       : ahl_bIsPasswordInKdsNissan
*
*DESCRIPTION    : This function tries to read the password from KDS,
*                 If there is one, TRUE will be returned
*
*PARAMETER      : None
*
*RETURN VALUE   : TRUE  - KDS entries for EDC and Device key are present.
*                 FALSE - KDS Entries are not present. 
*******************************************************************************/
tBool ahl_bIsPasswordInKdsNissan (tVoid)
{
    tBool   bEDCCheck = FALSE, bDevKeyCheck = FALSE;
    tU8     u8Buffer [KDS_EDC_LENGTH] = {0};
    tBool   bRetVal = FALSE;
    
    //Read first KDS Entry
    bEDCCheck    = ahl_bReadKDSEntry ( KDS_TYPE_EDC, 
                                       u8Buffer, 
                                       KDS_EDC_LENGTH );
    //Read second KDS Entry                                       
    bDevKeyCheck = ahl_bReadKDSEntry ( KDS_TYPE_AES_DEV_KEY, 
                                       u8Buffer, 
                                       KDS_AES_KEY_LENGTH );
                                       
	//A Little housekeeping - Remove contents of Buffer
    (tVoid) OSAL_pvMemorySet ( u8Buffer, 0, KDS_EDC_LENGTH );
                                       
    if ((bEDCCheck == TRUE) && (bDevKeyCheck == TRUE))
    {
        //Both Entries exist
        bRetVal = TRUE;
    }
    else
    {
        // Atleast 1 entry is missing
        bRetVal = FALSE;
    }
    
    return bRetVal;
}

/* -- end of file ---------------------------------------------------------- */
