/*****************************************************************************
 * FILE:         ahl_tclNotificationTable.cpp
 * PROJECT:      ELeNa
 * SW-COMPONENT: 
 *----------------------------------------------------------------------------
 *
 * DESCRIPTION:  This class handles a notification table structure. 
 *              
 *----------------------------------------------------------------------------
 * COPYRIGHT:    (c) 2001 Robert Bosch GmbH, Hildesheim
 * HISTORY:      
 * Date      | Author            | Modification
 * 17.07.01  | CM/EPE Perick     | initial version
 * 23.11.01  | CM/EPE Perick     | removed dynamic list and changed the 
 *           |                   | NotifactionTable to a static mechanism
 *           |                   | (avoid memory fragmentation)
 * 18.07.02  | CM-CR/EES4 Perick | prevent UpReg from one client, two times
 * 07.04.08  | CM-DI/PJ-VW36 Hessling | dynamic enlarge of internal notify tab
 *****************************************************************************/
#include "ahl_NotificationTable.h"
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#include "ahl_trace.h"

// -- trace identifier ------- define this value for every class new 
#ifdef FILE_NUMBER
#undef FILE_NUMBER
#endif
#define FILE_NUMBER TRC::ahl_NotificationTable_cpp
// -- trace identifier ------- end

//-----------------------------------------------------------------------------
// default values for enlarging the internally controlled arrays                                      
//-----------------------------------------------------------------------------
#define AHL_FUNCARRAY_ENLARGE 20
#define AHL_NOTARRAY_ENLARGE 100
#define AHL_MAX_FUNCARRAY_ASSERT 500
#define AHL_MAX_NOTARRAY_ASSERT 2000

//-----------------------------------------------------------------------------
// constructor with external controlled array                                 
//-----------------------------------------------------------------------------
ahl_tclNotificationTable::ahl_tclNotificationTable (ahl_tFunction     *pFuncArray, 
                                                    tU16              u16FuncArraySize, 
                                                    ahl_tNotification *pNotArray, 
                                                    tU16              u16NotArraySize )
   : u16EnlrgFunc(0), u16EnlrgNot(0), u16FuncMax(0), u16NotMax(0)
{  
   _bIsArrayInternal = FALSE;

   _pFuncArray       = pFuncArray;
   _pNotArray        = pNotArray;
   _u16FuncArraySize = u16FuncArraySize;
   _u16NotArraySize  = u16NotArraySize;

   vInit();
}

//-----------------------------------------------------------------------------
// constructor with dynamic internal array                                         
//-----------------------------------------------------------------------------
ahl_tclNotificationTable::ahl_tclNotificationTable(tU16 u16FuncArraySize)
   : u16EnlrgFunc(AHL_FUNCARRAY_ENLARGE),
     u16EnlrgNot(AHL_NOTARRAY_ENLARGE),
     u16FuncMax(AHL_MAX_FUNCARRAY_ASSERT),
     u16NotMax(AHL_MAX_NOTARRAY_ASSERT)
{  
   _bIsArrayInternal = TRUE;
   if(u16FuncArraySize > 0) {
      _u16FuncArraySize = u16FuncArraySize;
   } else {
      // to make it running we need at least AHL_FUNCARRAY_ENLARGE.
      _u16FuncArraySize = AHL_FUNCARRAY_ENLARGE;
   }
   _pFuncArray       = OSAL_NEW ahl_tFunction[_u16FuncArraySize];
   TRACE_ON_ERROR_NULL(TRC::FnConstructor, _pFuncArray);
   _u16NotArraySize  = 0;
   _pNotArray        = 0;

   vInit();
   // prepaire some place for notifications
   vEnlargeNotificationArray();   
}

//-----------------------------------------------------------------------------
// initialisation is called from both constructors
//-----------------------------------------------------------------------------
tVoid ahl_tclNotificationTable::vInit(tVoid) 
{
   TRACE_FLOW_DEF(TRC::FnInit);
   // init all entries
   if(_pFuncArray) {
      tU32 uSize = (_u16FuncArraySize)*(sizeof(ahl_tFunction));
      OSAL_pvMemorySet(_pFuncArray, 0u, uSize);
   }
   if(_pNotArray) {
      tU32 uSize = (_u16NotArraySize)*sizeof(ahl_tNotification);
      OSAL_pvMemorySet(_pNotArray, 0u, uSize);
   }

}

//-----------------------------------------------------------------------------
// destructor                                         
//-----------------------------------------------------------------------------
ahl_tclNotificationTable::~ahl_tclNotificationTable()
{
   if(_bIsArrayInternal) {
      if(_pNotArray) {
         OSAL_DELETE[] _pNotArray;
         _pNotArray=NULL;            
      }
      if(_pFuncArray) {
         OSAL_DELETE[] _pFuncArray;
         _pFuncArray=NULL;           
      }
   }
} //lint !e1740 prio3 reviewed: The pointer is untouched if not in use only.

/*************************************************************************
*
* FUNCTION: vEnlargeFunctionArray()
* 
* DESCRIPTION: make sure we have some more place the function ids
*
* PARAMETER: void
*
* RETURNVALUE: void
*
*************************************************************************/
tVoid ahl_tclNotificationTable::vEnlargeFunctionArray(tVoid)
{
   TRACE_FLOW_DEF(TRC::FnEnlargeFunctionArray);
   if(_pFuncArray) {
      for(tInt i = 0; i < _u16FuncArraySize; i++ ) 
      {
         if(_pFuncArray[i].u16FuncID == 0) {
            /* switched of because no action required and it come very often
            TRACE_DBG_LS(TRC::FnEnlargeFunctionArray, TR_LEVEL_USER_3, 
               "No enlarge of FID array required."); 
            */
            return;
         }
      }
   }
   // enlarge the function array
   if((_u16FuncArraySize+AHL_FUNCARRAY_ENLARGE)>u16FuncMax) {
	   TRACE_DBG_LS(TRC::FnEnlargeFunctionArray,TR_LEVEL_WARNING, 
         "Maximum of FIDs in notification table reached. Use ahl_tclNotificationTable::u16FuncMax to enlarge if intended.");
   }
   // size = newCount * struct size
   tU32 uSize = (_u16FuncArraySize+AHL_FUNCARRAY_ENLARGE)*(sizeof(ahl_tFunction));
   ahl_tFunction* paNewFuncArray = OSAL_NEW ahl_tFunction[(_u16FuncArraySize+AHL_FUNCARRAY_ENLARGE)];
   TRACE_ON_ERROR_NULL(TRC::FnEnlargeFunctionArray, paNewFuncArray);
   if (paNewFuncArray==0) {
      // we could not allocate memory
      return;
   }
   OSAL_pvMemorySet(paNewFuncArray, 0u, uSize);
   if(_pFuncArray!=0) 
   {
      // copy old values to the new place
      TRACE_FLOW_INFO(TRC::FnEnlargeFunctionArray, TRC::FuncStep1);
      OSAL_pvMemoryCopy(paNewFuncArray, _pFuncArray, (_u16FuncArraySize*(sizeof(ahl_tFunction))));
      OSAL_DELETE[] _pFuncArray;
      _pFuncArray=NULL;
   }
   // in any case correct to the newly created pointer
   _pFuncArray = paNewFuncArray;
   _u16FuncArraySize += AHL_FUNCARRAY_ENLARGE;
   TRACE_DBG_LSN(TRC::FnEnlargeFunctionArray, TR_LEVEL_USER_2, 
      "FID array enlarged to:", _u16FuncArraySize); 
}

/*************************************************************************
*
* FUNCTION: vEnlargeNotificationTable()
* 
* DESCRIPTION: make sure we have some more place for the client who to be upReged
*
* PARAMETER: void
*
* RETURNVALUE: void
*
*************************************************************************/
tVoid ahl_tclNotificationTable::vEnlargeNotificationArray(tVoid)
{
   TRACE_FLOW_DEF(TRC::FnEnlargeNotificationArray);
   if(_pNotArray) {
      // check for free space
      for(tInt i = 0; i < _u16NotArraySize; i++) 
      {
         if(_pNotArray[i].bOccupied == false) 
         {
            // nothing to do we have a place left
            /* switched of because no action required and it come very often
            TRACE_DBG_LS(TRC::FnEnlargeNotificationArray, TR_LEVEL_USER_3, 
               "No enlarge of notification table required."); 
               */
            return;
         }
      }
   } 
   // Create new instance being bigger than before
   if((_u16NotArraySize+AHL_NOTARRAY_ENLARGE)>u16NotMax) {
      TRACE_ERROR_FATAL_STRING(TRC::FnEnlargeNotificationArray, TRC::ErrMaximumArrayReached,
         "Maximum of notifiers in notification table. Use ahl_tclNotificationTable::u16NotMax to enlarge if intended.");
   }
   // size = newCount * struct size
   tU32 uSize = (_u16NotArraySize+AHL_NOTARRAY_ENLARGE)*sizeof(ahl_tNotification);
   ahl_tNotification* paNewNotArray = OSAL_NEW ahl_tNotification[_u16NotArraySize+AHL_NOTARRAY_ENLARGE];
   TRACE_ON_ERROR_NULL(TRC::FnEnlargeNotificationArray, paNewNotArray);
   if(paNewNotArray){
      OSAL_pvMemorySet(paNewNotArray, 0u, uSize);
   }
   // if old values exist copy them
   if(_pNotArray && paNewNotArray) 
   {
      TRACE_FLOW_INFO(TRC::FnEnlargeNotificationArray, TRC::FuncStep1);
      // now copy value and correct the pointer
      OSAL_pvMemoryCopy(paNewNotArray, _pNotArray, _u16NotArraySize*sizeof(ahl_tNotification));
      intptr_t pointerOffset = (intptr_t)paNewNotArray - (intptr_t)_pNotArray;
      TRACE_FLOW_INFO(TRC::FnEnlargeNotificationArray, TRC::FuncStep2);
      for (tInt i1 = 0; i1 < _u16NotArraySize; i1++) 
      {
         if(paNewNotArray[i1].pNext != NULL) 
         {
            paNewNotArray[i1].pNext = (ahl_tNotification *)
               ((intptr_t)paNewNotArray[i1].pNext + pointerOffset);
         }
      }
      TRACE_FLOW_INFO(TRC::FnEnlargeNotificationArray, TRC::FuncStep3);
      if(_pFuncArray) {
         for (tInt i2 = 0; i2 < _u16FuncArraySize; i2++) 
         {
            if(_pFuncArray[i2].pFirst != NULL) 
            {
               _pFuncArray[i2].pFirst = (ahl_tNotification *)
                  ((intptr_t)_pFuncArray[i2].pFirst + pointerOffset);
            }
         }
      }
      OSAL_DELETE[] _pNotArray;
      _pNotArray=NULL;
   }
   // in any case correct to the newly created pointer
   _pNotArray = paNewNotArray;
   _u16NotArraySize += AHL_NOTARRAY_ENLARGE;
   TRACE_DBG_LSN(TRC::FnEnlargeNotificationArray, TR_LEVEL_USER_2, 
      "Notification table enlarged to:", _u16NotArraySize); 
}

//-----------------------------------------------------------------------------
// poGetNotificationList returns a pointer list to all notifier entries
//-----------------------------------------------------------------------------
ahl_tNotification* ahl_tclNotificationTable::poGetNotificationList(tU16 u16FuncID)
{
   tInt x;
   if(_pFuncArray) {
      for(x=0; x < _u16FuncArraySize; x++)
      {
          if(_pFuncArray[x].u16FuncID == u16FuncID)
              return _pFuncArray[x].pFirst;
      }
   }
   return NULL;
}//lint !e1762 prio3 reviewed: is left non-const because returned pointers are non-const 

//-----------------------------------------------------------------------------
// bAddNotification inserts notifier entry if different from existing
//-----------------------------------------------------------------------------
tBool ahl_tclNotificationTable::bAddNotification( tU16  u16FuncID, 
                                                  tU16  u16AppID,
                                                  tU16  u16RegisterID,
                                                  tBool bNotification, 
                                                  tU16  u16CmdCounter,
                                                  tU16  u16SubID,
                                                  tU8   u8Act)
{ 

    tInt fi, ni;
    tBool   bSuccess = false;
    ahl_tFunction     *pCurFunction     = NULL;
    ahl_tNotification *pNewNotification = NULL;
    ahl_tNotification *pCurNotification = NULL;
    
    if((_pNotArray==0) || (_pFuncArray==0)) {
       return false;
    }

    // ensure that arrays are big enough
    if(_bIsArrayInternal) {
       vEnlargeFunctionArray();
       vEnlargeNotificationArray();
    }
    
    // find NotificationListEntry for this function
    for(fi = 0; fi < _u16FuncArraySize; fi++)
    {
        if(_pFuncArray[fi].u16FuncID == u16FuncID)
        {
           pCurFunction = &_pFuncArray[fi];
           break;
        }
    }

    // if no list exist create new entry in FunctionArray
    if(pCurFunction == NULL)
    {
        for(fi = 0; fi < _u16FuncArraySize; fi++)
        {
            if(_pFuncArray[fi].u16FuncID == 0)
            {
                // if no list exist create new entry in FunctionArray
                if( fi < _u16FuncArraySize )
                {
                    _pFuncArray[fi].u16FuncID = u16FuncID;
                    _pFuncArray[fi].pFirst = NULL;
                    pCurFunction = &_pFuncArray[fi];
                    break;
                }
            }
        }
    }

    if(pCurFunction != NULL)
    {
        // check, if client call UpReg twice 
        if(bNotification)
        {
            // search equal notification entry
            pCurNotification = pCurFunction->pFirst;
            if(pCurNotification != NULL)
            {
                while(pCurNotification)
                {
                    if( pCurNotification->u16AppID      == u16AppID &&
                        pCurNotification->bNotification == 1        &&
                        pCurNotification->u16RegisterID == u16RegisterID &&
                        pCurNotification->u16SubID      == u16SubID)
                    {
                        // TODO: tbd. return TRUE or FALSE
                        // return FALSE;
                        return TRUE;
                    }
                    pCurNotification = pCurNotification->pNext;
                }
            }            
        }

        // find free space in Notification Array and insert values
        for( ni = 0; ni < _u16NotArraySize; ni++)
        {
            if(_pNotArray[ni].bOccupied == false)
            {
                pNewNotification = &_pNotArray[ni];
                pNewNotification->bOccupied     = true;
                pNewNotification->u16AppID      = u16AppID;
                pNewNotification->bNotification = bNotification;
                pNewNotification->u16CmdCounter = u16CmdCounter;
                pNewNotification->u16RegisterID = u16RegisterID;
                pNewNotification->u16SubID      = u16SubID;
                pNewNotification->u8Act         = u8Act;
                break;
            }
        }

        // connect new notification in list
        if(pNewNotification != NULL)
        {
            if(pCurFunction->pFirst)
            {
                // find end of notification list
                pCurNotification = pCurFunction->pFirst;
                while(pCurNotification->pNext)
                {
                    pCurNotification = pCurNotification->pNext;
                }
                pCurNotification->pNext = pNewNotification;
                bSuccess = true;
                
            }
            else
            {
                // if first entry in notification list
                pCurFunction->pFirst = pNewNotification;
                bSuccess = true;
            }
        }
        else
        {
            // could create entry in function array, but it is not enough space
            // in the notification array
            pCurFunction->u16FuncID = 0;
            pCurFunction->pFirst = NULL;
            bSuccess = false;
        }
    }

    return bSuccess;
}

//-----------------------------------------------------------------------------
// bRemoveNotification removes the entry being the same as specified
//-----------------------------------------------------------------------------
tBool ahl_tclNotificationTable::bRemoveNotification( tU16 u16FuncID, 
                                                     tU16 u16AppID,
                                                     tU16 /*u16RegisterID*/,
                                                     tBool bNotification, 
                                                     tU16 /*u16CmdCounter*/,
                                                     tU16 u16SubID)
{
    tInt fi;
    ahl_tNotification *pCurNotification  = NULL;
    ahl_tNotification *pPrevNotification = NULL;
    ahl_tFunction     *pCurFunction      = NULL;

    if(_pFuncArray==0) {
      return false;
    }
    // search function in function array
    for(fi=0; fi < _u16FuncArraySize; fi++)
    {
        if(_pFuncArray[fi].u16FuncID == u16FuncID)
        {
            pCurNotification = _pFuncArray[fi].pFirst;
            pCurFunction = &_pFuncArray[fi];
        }
    }

    if(!pCurFunction)
        return false;

    pCurNotification = pCurFunction->pFirst;
   
    while(pCurNotification)
    {
        if( pCurNotification->u16AppID      == u16AppID      &&
            pCurNotification->bNotification == bNotification &&
            pCurNotification->u16SubID      == u16SubID   /* &&
            pCurNotification->u16CmdCounter == u16CmdCounter */ )
        {
            // removes search hit
            pCurNotification->bOccupied     = false;
            pCurNotification->u16AppID      = 0;
            pCurNotification->bNotification = 0;
            pCurNotification->u16CmdCounter = 0;
            pCurNotification->u16RegisterID = 0;
            pCurNotification->u16SubID      = AMT_C_U16_DEFAULT_NULL;
            pCurNotification->u8Act         = 0;


            if(pPrevNotification)
            {
                pPrevNotification->pNext = pCurNotification->pNext;
                pCurNotification->pNext  = NULL;
            }
            else
            {
                pCurFunction->pFirst = pCurNotification->pNext;
                pCurNotification->pNext  = NULL;
                if(pCurFunction->pFirst == NULL)
                    pCurFunction->u16FuncID = 0;

            }
            return true;

        }

        pPrevNotification = pCurNotification;
        pCurNotification = pCurNotification->pNext;
    }

    return false;
}

//-----------------------------------------------------------------------------
// vTraceTable traces out the whole notification table
//-----------------------------------------------------------------------------
tVoid ahl_tclNotificationTable::vTraceTable(tU16 u16ClassID) const
{
   ahl_tNotification *pCurNotification  = NULL;
   if(_pFuncArray) {
      for(tU16 fi=0; fi < _u16FuncArraySize; fi++)
      {
         pCurNotification = _pFuncArray[fi].pFirst;
         
         while(pCurNotification)
         {
            ET_TRACE_ERROR(u16ClassID,
               "%i: FID: 0x%04X, AppID: 0x%04X, SubID: 0x%04X, RegId: %d, CmdCnt:%d, Not: %d (%p)" 
			   _ fi
               _ _pFuncArray[fi].u16FuncID
               _ pCurNotification->u16AppID
               _ pCurNotification->u16SubID
               _ pCurNotification->u16RegisterID
               _ pCurNotification->u16CmdCounter
               _ pCurNotification->bNotification
			   _ pCurNotification); //lint !e1776 prio3 reviewed: param3 is the trick to put the variable parameter list into a macro

            pCurNotification = pCurNotification->pNext;
         }
      }
   }
}

//-----------------------------------------------------------------------------
// bRemoveAllEntriesWithRegID removes all entries from one registration id
// use this function if the recipient did not respond anymore
//-----------------------------------------------------------------------------
tBool ahl_tclNotificationTable::bRemoveAllEntriesWithRegID(tU16 u16RegID)
{
   tInt fi;
   ahl_tNotification *pCurNotification  = NULL;
   ahl_tNotification *pPrevNotification = NULL;

   if(_pFuncArray) {
      for(fi=0; fi < _u16FuncArraySize; fi++)
      {
         pPrevNotification = NULL;
         pCurNotification = _pFuncArray[fi].pFirst;
         
         while(pCurNotification)
         {
            if(pCurNotification->u16RegisterID == u16RegID)
            {
               // removes search hit
               pCurNotification->bOccupied     = FALSE;
               pCurNotification->u16AppID      = 0;
               pCurNotification->bNotification = 0;
               pCurNotification->u16CmdCounter = 0;
               pCurNotification->u16RegisterID = 0;
               pCurNotification->u16SubID      = AMT_C_U16_DEFAULT_NULL;
               pCurNotification->u8Act         = 0;
               
               
               if(pPrevNotification != NULL)
               {
                  pPrevNotification->pNext = pCurNotification->pNext;
                  pCurNotification->pNext  = NULL;

                  pCurNotification = pPrevNotification->pNext;
               }
               else
               {
                  _pFuncArray[fi].pFirst = pCurNotification->pNext;
                  pCurNotification->pNext  = NULL;
                  if(_pFuncArray[fi].pFirst == NULL)
                  {
                     _pFuncArray[fi].u16FuncID = 0;
                  }
                  pCurNotification = _pFuncArray[fi].pFirst;
               }
            }
            else
            {
               pPrevNotification = pCurNotification;
               pCurNotification = pCurNotification->pNext;
            }
         }
      }
   }

   return TRUE;
}
