/********************************************************************//**
 * @file         ahl_List.h
 * @ingroup      Lists
 *
 * @component    Application Help Library
 *
 * @description  This file defines several kinds of singly linked lists.
 *
 * @author       CM-AM/ENG-DI-Gieseke
 * @author       CM-AM/ENG-DI-Streitfeld
 * @copyright    2002,2006 Blaupunkt GmbH
 ************************************************************************/

#ifndef AHL_LIST_H
#define AHL_LIST_H

#define OSAL_S_IMPORT_INTERFACE_TYPES
#define OSAL_S_IMPORT_INTERFACE_MEMORY
#include "osal_if.h"

// typedef tS32 (* tfunc_s32Compare)(tVoid* pvLeft,tVoid* pvRight);


class ahl_tclComparator
{
public:
   ahl_tclComparator()
      :_poNextComparator(0)
   {}

   virtual ~ahl_tclComparator()
   {
      _poNextComparator=0;
   }

   tVoid vSetNextComparison(ahl_tclComparator& rfoNextCmp)
   {
      _poNextComparator = &rfoNextCmp;
   }

   tS32 s32Compare(tVoid* pv1, tVoid* pv2);

protected:
   virtual tS32 s32ConcreteCompareVoid(tVoid* pv1, tVoid* pv2) = 0;

private:
   ahl_tclComparator* _poNextComparator;
};


template <class tElementType>
class ahl_tclComparatorTempl: public ahl_tclComparator
{
protected:
// JH - ARM complains of the virtual function override with the corresponding templated function.
// The void* version should not really be needed if we are using templates anyway.
//   virtual tS32 s32ConcreteCompare(tVoid* pv1, tVoid* pv2)
   virtual tS32 s32ConcreteCompareVoid(tVoid* pv1, tVoid* pv2)
   {
      return s32ConcreteCompare((tElementType*)pv1, (tElementType*)pv2);
   }

   virtual tS32 s32ConcreteCompare(tElementType* po1, tElementType* po2) = 0;
};



/********************************************************************//**
 * @class       ahl_tclU16List
 * @ingroup     Lists
 *
 * @description A singly linked list.
 *
 *              This list has the following features:
 *              - it knows how many entries it contains
 *              - element access by index
 *              - only unique elements
 *              - provision for sequential iteration over list elements
 *              - does not manage the memory of the contained elements;
 *                the application is responsible to free them
 *
 * @author      CM-AM/ENG-DI-Gieseke
 ************************************************************************/
//##ModelId=415AA32600BB
class ahl_tclU16List
{
   //   friend class	ahl_tclListIterator ;
public:
   /********************************************************************//**
    * @method      ahl_tclU16List
    *
    * @description Default constructor.
    *
    *              The default constructor constructs an empty list.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
    //##ModelId=415AA5350196
   ahl_tclU16List();

   /********************************************************************//**
    * @method      ~ahl_tclU16List
    *
    * @description Destructor.
    *
    *              The destructor frees only the memory of the list nodes,
    *              but it leaves the contained objects untouched.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
    //##ModelId=415AA53501B5
   virtual ~ahl_tclU16List();

   /********************************************************************//**
    * @method      ahl_tclU16List
    *
    * @description Copy constructor.
    *
    *              The copy constructor creates a list that contains the
    *              same nodes as the source list @a roRight.
    *
    * @note        The objects that the nodes of @a roRight refer to are
    *              @em not copied, so after the construction both lists
    *              refer to the same objects.
    *
    * @deprecated  Except for the initialisation of uc_tclDspBase::MAPLISTxxx
    *              in el_uc_ContextBase.cpp copy construction or assignment
    *              of lists is not used.  Thus, this method is not needed
    *              and should be removed (by making the declaration
    *              @c private and removing the implementation).
    *
    * @param       roRight (I) \n
    *                 The list that is to be copied.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
    //##ModelId=415AA53501C5
   ahl_tclU16List(const ahl_tclU16List &roRight);

   /********************************************************************//**
    * @method      operator=
    *
    * @description Assignment operator.
    *
    *              The assignment operator destroys the current contents
    *              of the list and recreates a list that contains the
    *              same nodes as the source list @a roRight.
    *
    *              Also, the current position of sequential iteration
    *              is copied from @a roRight.
    *
    * @note        The objects that the nodes of @a roRight refer to are
    *              @em not copied, so after the construction both lists
    *              refer to the same objects.
    *
    * @bug         There is a bug in setting up the sequential iteration
    *              position.  ahl_tclU16List#prActEntry is set to
    *              @a roRight's value, and @em not the corresponding
    *              element of the copied list.  I.e. if one calls
    *              pvGetSequential() on this object after the assignment,
    *              actually the elements of @a roRight are traversed.
    *
    * @deprecated  Except for the initialisation of uc_tclDspBase::MAPLISTxxx
    *              in el_uc_ContextBase.cpp copy construction or assignment
    *              of lists is not used.  Thus, this method is not needed
    *              and should be removed (by making the declaration
    *              @c private and removing the implementation).
    *
    * @param       roRight (I) \n
    *                 The list that is to be copied.
    *
    * @return      A reference to this list.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
    //##ModelId=415AA5350222
   const ahl_tclU16List & operator=(const ahl_tclU16List &roRight);

   /********************************************************************//**
    * @method      operator+=
    *
    * @description Append elements from another list.
    *
    *              The elements of the source list @a roRight are
    *              appended to this list.
    *
    * @note        The objects that the nodes of @a roRight refer to are
    *              @em not copied, so after the construction both lists
    *              refer to the same objects.
    *
    * @param       roRight (I) \n
    *                 The list that is to be appended.
    *
    * @return      A reference to this list.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
    //##ModelId=415AA53502EE
   const ahl_tclU16List& operator += (const ahl_tclU16List &roRight);

   /********************************************************************//**
    * @method      operator==
    *
    * @description Determine whether lists are equal.
    *
    *              This operator determines whether this list contains
    *              the same objects as the list referenced by @a roRight.
    *
    * @param       roRight (I) \n
    *                 The list that is to be compared.
    *
    * @return      @c TRUE if this list and @a roRight compare equal,
    *              @c FALSE otherwise.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
    //##ModelId=415AA5350280
   tBool operator==(const ahl_tclU16List &roRight);

   /********************************************************************//**
    * @method      u16GetSize
    *
    * @description Determine the number of elements in this list.
    *
    * @return      The number of elements currently stored in the list.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
    //##ModelId=415AA5360280
   inline tU16 u16GetSize() const
   {
      return u16Size;
   }

   /** @name Insertion and removal
    *  @note These methods reset the sequential iteration sequence. */
   /*@{*/
   /********************************************************************//**
    * @method      bAppend
    *
    * @description Add an element to the tail of the list.
    *
    *              The object is only added if it does not already exist
    *              in the list; in the latter case, or if @a pvObj is
    *              @c NULL, this method returns with an error.
    *
    * @note        This method resets the sequential iteration sequence.
    *
    * @param       pvObj (->I) \n
    *                 The object that is to be added.
    *
    * @return      @c TRUE if the object has been added to the list,
    *              @c FALSE in case of an error.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
    //##ModelId=415AA535035B
   tBool bAppend( tVoid *pvObj );

   /********************************************************************//**
    * @method      bAddAsFirst
    *
    * @description Add an element to the head of the list.
    *
    *              The object is only added if it does not already exist
    *              in the list; in the latter case, or if @a pvObj is
    *              @c NULL, this method returns with an error.
    *
    * @note        This method resets the sequential iteration sequence.
    *
    * @param       pvObj (->I) \n
    *                 The object that is to be added.
    *
    * @return      @c TRUE if the object has been added to the list,
    *              @c FALSE in case of an error.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
   //##ModelId=415AA53503C8
   tBool bAddAsFirst( tVoid *pvObj );

   /********************************************************************//**
    * @method      u16InsertEntryBefore
    *
    * @description Insert an element into the list.
    *
    *              The object is only inserted if it does not already
    *              exist in the list; in the latter case, or if
    *              @a pvNewObj is @c NULL, this method returns with an
    *              error.
    *
    *              If the insertion position @a u16Index is larger than
    *              the size of the list, the element is added to the
    *              end of the list.
    *
    * @note        This method resets the sequential iteration sequence.
    *
    * @param       u16Index   (I) \n
    *                 The index before which the new element is to be
    *                 inserted,
    *
    * @param       pvNewObj (->I) \n
    *                 The object that is to be added.
    *
    * @return      The index at which the element has been inserted,
    *              or @c 0xFFFF in case of an error.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
    //##ModelId=415AA76301B5
   tU16 u16InsertEntryBefore(tU16 u16Index, tVoid *pvNewObj);

   /********************************************************************//**
    * @method      bRemove
    *
    * @description Remove an object from the list.
    *
    * @param       pvObj (->I) \n
    *                 The object that is to removed.
    *
    * @note        This method resets the sequential iteration sequence.
    *
    * @return      @c TRUE if the object has been removed from the list,
    *              @c FALSE if the object has not been in the list at all.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
   //##ModelId=415AA536004E
   tBool bRemove(const tVoid *pvObj );

   /********************************************************************//**
    * @method      vRemoveFirstEntry
    *
    * @description Remove the first entry of the list.
    *
    * @note        This method resets the sequential iteration sequence.
    *
    * @return      -
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
    //##ModelId=415AA53600BB
   tVoid vRemoveFirstEntry();
   /*@}*/

   /** @name Sorting */
   /*@{*/
   /********************************************************************//**
    * @method      vInsertSortList
    *
    * @description Sort list by insertion in ascending order 
    *              (tfunc_s32Compare < 0)
    *
    * @param       funcComp (->I) \n
    *                 Compare function to use for sorting.
    *
    * @return      -
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   tVoid vInsertSortList(ahl_tclComparator &rfoComparator);

    /********************************************************************//**
    * @method      vInsertSortParallelList
    *
    * @description Sort list by insertion in ascending order 
    *              (tfunc_s32Compare < 0) and apply the same sorting to external list
    *
    * @param       funcComp (->I) \n
    *                 Compare function to use for sorting.
    *
    * @param       funcComp (I) \n
    *                 External list which should be sorted in same way.
    *                 If nr of elements not the same, no sorting at all is done. 
    *
    * @return      -
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   tVoid vInsertSortParallelList(ahl_tclComparator &rfoComparator, ahl_tclU16List &rfoExtList);
   /*@}*/

   /** @name Member access */
   /*@{*/
   /********************************************************************//**
    * @method      pvGetFirst
    *
    * @description Determine the first entry of the list.
    *
    * @return      The first entry of the list, or @c NULL if the list
    *              is empty.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
   //##ModelId=415AA53600EA
   tVoid *pvGetFirst() const;

   /********************************************************************//**
    * @method      pvGetLast
    *
    * @description Determine the last entry of the list.
    *
    * @return      The last entry of the list, or @c NULL if the list
    *              is empty.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
   //##ModelId=415AA5360119
   tVoid *pvGetLast() const;

   /********************************************************************//**
    * @method      pvGetByIndex
    *
    * @description Determine the list entry at a given index.
    *
    * @note        This method resets the sequential iteration sequence
    *              to continue at @a u16Index.
    *
    * @param       u16Index   (I) \n
    *                 Index of the element that is to be queried.
    *
    * @return      The last entry of the list, or @c NULL if the list
    *              is empty.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
    //##ModelId=415AA53601D4
   tVoid *pvGetByIndex(tU16 u16Index) const;

   /********************************************************************//**
    * @method      pvReplaceEntry
    *
    * @description Replace a list element with another object.
    *
    * @note        This method resets the sequential iteration sequence
    *              to continue at @a u16Index.
    *
    * @param       u16Index   (I) \n
    *                 Index of the element that is to be replaced.
    *
    * @param       pvNewObj (->I) \n
    *                 The new object that is to be stored in the list.
    *
    * @return      The previous object at @a u16Index, or @c NULL if
    *              @a u16Index is out of bounds or @a pvNewObj is
    *              already stored in the list.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
    //##ModelId=415AA53603B9
   tVoid *pvReplaceEntry(tU16 u16Index, tVoid *pvNewObj);
   /*@}*/

   /** @name Sequential access */
   /*@{*/
   /********************************************************************//**
    * @method      vRewind
    *
    * @description Reset the sequential iteration sequence.
    *
    * @return      -
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
   //##ModelId=415AA5360157
   inline tVoid vRewind() const
   {
      prActEntry = prBottomElement;
   }

   /********************************************************************//**
    * @method      pvGetSequential
    *
    * @description Perform an iteration step.
    *
    *              This method returns the current element of the
    *              iteration sequence and advances the sequence by
    *              one step.
    *
    * @return      The object that has been current before the step
    *              has been performed.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
    //##ModelId=415AA5360196
   tVoid *pvGetSequential() const;

   /********************************************************************//**
    * @method      pvGetActEntry
    *
    * @description Determine the current object.
    *
    * @return      The object at the current position of the iteration
    *              sequence, or @c NULL if the list is empty.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
    //##ModelId=415AA53602BF
   tVoid *pvGetActEntry() const;

   /********************************************************************//**
    * @method      bSetActEntry
    *
    * @description Replace the current object.
    *
    *              The object at the current position of the iteration
    *              sequence is replaced with @a pvObj.
    *
    * @param       pvObj (->I) \n
    *                 The new object.
    *
    * @return      @c TRUE if the current object has been replaced with,
    *              @a pvObj, @c FALSE if @a pvObj has already been
    *              contained in the list.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
    //##ModelId=415AA53602FD
   tBool bSetActEntry(const tVoid *pvObj ) const;
   /*@}*/


private:
   /*****************************************************************//**
    * @struct      rListEntry
    *
    * @description A list node.
    *
    *              Instances of this structure form the nodes of the
    *              linked list.  They contain the referenced data as
    *              well as a pointer to the next node.
    *********************************************************************/
   //##ModelId=415AA53500EA
   struct rListEntry {
        //##ModelId=415AA535010A
      tVoid *pvObject;             /*!< Pointer to the contained object. */
        //##ModelId=415AA535015A
      rListEntry* prNextListEntry; /*!< Pointer to the next list entry.  */
   };

   /********************************************************************//**
    * @method      prCreateNewEntry
    *
    * @description Create a new list node.
    *
    * @param       poObj (->I) \n
    *                 The object that is to be contained by the node.
    *
    * @return      A pointer to the new list node.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
    //##ModelId=415AA53700FA
   ahl_tclU16List::rListEntry* prCreateNewEntry(tVoid *poObj );

   /********************************************************************//**
    * @method      prFindListEntry
    *
    * @description Find the list node of a given object.
    *
    * @param       poObj (->I) \n
    *                 The object that is to be searched for.
    *
    * @return      A pointer to the first list node that contains @a poObj,
    *              or @c NULL if the object is not found.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
    //##ModelId=415AA53701A5
   ahl_tclU16List::rListEntry* prFindListEntry(const tVoid *poObj ) const;

   /********************************************************************//**
    * @method      vSwitchElements
    *
    * @description Switches the object pointer of 2 elements
    *
    * @param       prFirst any element of list != 0
    *
    * @param       prSecond any element of list != 0
    *
    * @return      -
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   tVoid vSwitchElements(rListEntry *prFirst, rListEntry *prSecond);

   /********************************************************************//**
    * @method      vMoveNextElementAfter
    *
    * @description Moves the next element of prSource after prDest
    *
    * @param       prDest any element of list != 0
    *
    * @param       prSource any element of list with next pointer != 0
    *
    * @return      -
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   tVoid vMoveNextElementAfter(rListEntry *prDest, rListEntry *prSource);

   /** Pointer to the first element of the list. */
    //##ModelId=415AA535015D
   rListEntry *prBottomElement;
   /** Pointer to the last element of the list. */
    //##ModelId=415AA5350168
   rListEntry *prHeadElement;
   /** Pointer to the current element for list traversal. */
    //##ModelId=415AA535016D
   mutable rListEntry *prActEntry;
   /** Current number of list elements. */
    //##ModelId=415AA5350177
   tU16 u16Size;
};

/********************************************************************//**
 * @class       ahl_tclList
 * @ingroup     Lists
 *
 * @description A variation of ahl_tclU16List.
 *
 *              This list supports all the features of ahl_tclU16List,
 *              but is restricted to at most 255 entries.
 *
 * @author      CM-AM/ENG-DI-Gieseke
 ************************************************************************/
//##ModelId=415AC15300FA
class ahl_tclList
{
   //   friend class	ahl_tclListIterator ;
public:
   /** @copydoc ahl_tclU16List::ahl_tclU16List() */
    //##ModelId=415AC1870000
   ahl_tclList();

   /** @copydoc ahl_tclU16List::~ahl_tclU16List() */
    //##ModelId=415AC187000F
   virtual ~ahl_tclList();

   /** @copydoc ahl_tclU16List::ahl_tclU16List(const ahl_tclU16List&) */
    //##ModelId=415AC187002E
   inline ahl_tclList(const ahl_tclList &roRight)
      : _oList(roRight._oList)
   {
   }

   /********************************************************************//**
    * @method      operator ahl_tclU16List
    *
    * @description Conversion to ahl_tclU16List.
    *
    *              This conversion operator lets ahl_tclList objects
    *              be passed to ahl_tclU16List methods that expect
    *              other ahl_tclU16List instances as their parameters.
    *
    * @bug         This should really be an
    *              <tt>operator const ahl_tclU16List&</tt>.  Currently,
    *              this operator creates a temporary copy of the list
    *              object.
    *
    * @return      The underlying ahl_tclU16List object.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
    //##ModelId=415AC187007D
   inline operator ahl_tclU16List() const
   {
      return _oList;
   }

   /** @copydoc ahl_tclU16List::operator=(const ahl_tclU16List&) */
    //##ModelId=415AC187009C
   inline const ahl_tclList & operator=(const ahl_tclList &roRight)
   {
      if (&roRight != this)
      {
         _oList.operator = ((ahl_tclU16List&)roRight);
      }
      return *this;
   }

   /** @copydoc ahl_tclU16List::operator==(const ahl_tclU16List&) */
    //##ModelId=415AC18700FA
   inline tBool operator==(const ahl_tclList &roRight)
   {
      return _oList.operator == ((ahl_tclU16List&)roRight);
   }

   /** @copydoc ahl_tclU16List::operator+=(const ahl_tclU16List&)
    *  @deprecated  This method has never been implemented. */
    //##ModelId=415AC1870157
   const ahl_tclList& operator += (const ahl_tclList &roRight);

   /** @copydoc ahl_tclU16List::u16GetSize() */
    //##ModelId=415AC18800BB
   inline tU8 u8GetSize() const
   {
      return (tU8)_oList.u16GetSize();
   }

   /** @name Insertion and removal
    *  @note These methods reset the sequential iteration sequence. */
   /*@{*/

   /********************************************************************//**
    * @method      bAppend
    *
    * @description Add an element to the tail of the list.
    *
    *              This method makes shure that the list contains no
    *              more than 255 elements.
    *
    *              The object is only added if it does not already exist
    *              in the list; in the latter case, or if @a pvObj is
    *              @c NULL, this method returns with an error.
    *
    * @note        This method resetd the sequential iteration sequence.
    *
    * @param       pvObj (->I) \n
    *                 The object that is to be added.
    *
    * @return      @c TRUE if the object has been added to the list,
    *              @c FALSE in case of an error.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
    //##ModelId=415AC18701C5
   tBool bAppend( tVoid *pvObj );

   /********************************************************************//**
    * @method      bAddAsFirst
    *
    * @description Add an element to the head of the list.
    *
    *              This method makes shure that the list contains no
    *              more than 255 elements.
    *
    *              The object is only added if it does not already exist
    *              in the list; in the latter case, or if @a pvObj is
    *              @c NULL, this method returns with an error.
    *
    * @note        This method resets the sequential iteration sequence.
    *
    * @param       pvObj (->I) \n
    *                 The object that is to be added.
    *
    * @return      @c TRUE if the object has been added to the list,
    *              @c FALSE in case of an error.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
    //##ModelId=415AC1870222
   tBool bAddAsFirst( tVoid *pvObj );

   /** @copydoc ahl_tclU16List::bRemove() */
    //##ModelId=415AC1870290
   inline tBool bRemove(const tVoid *pvObj )
   {
      return _oList.bRemove(pvObj);
   }

   /** @copydoc ahl_tclU16List::vRemoveFirstEntry() */
    //##ModelId=415AC18702FD
   inline tVoid vRemoveFirstEntry()
   {
      _oList.vRemoveFirstEntry();
   }
   /*@}*/

   /** @name Member access */
   /*@{*/
   /** @copydoc ahl_tclU16List::pvGetFirst() */
    //##ModelId=415AC187031C
   inline tVoid *pvGetFirst() const
   {
      return _oList.pvGetFirst();
   }

   /** @copydoc ahl_tclU16List::pvGetLast() */
    //##ModelId=415AC187035B
   inline tVoid *pvGetLast() const
   {
      return _oList.pvGetLast();
   }

   /********************************************************************//**
    * @method      pvGetByIndex
    *
    * @description Determine the list entry at a given index.
    *
    * @note        This method resets the sequential iteration sequence
    *              to continue at @a u16Index.
    *
    * @param       u8Index   (I) \n
    *                 Index of the element that is to be queried.
    *
    * @return      The last entry of the list, or @c NULL if the list
    *              is empty.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
    //##ModelId=415AC188001F
   inline tVoid *pvGetByIndex(tU8 u8Index) const
   {
      return _oList.pvGetByIndex(u8Index);
   }

   /********************************************************************//**
    * @method      pvReplaceEntry
    *
    * @description Replace a list element with another object.
    *
    * @note        This method resets the sequential iteration sequence
    *              to continue at @a u8Index.
    *
    * @param       u8Index   (I) \n
    *                 Index of the element that is to be replaced.
    *
    * @param       pvNewObj (->I) \n
    *                 The new object that is to be stored in the list.
    *
    * @return      The previous object at @a u8Index, or @c NULL if
    *              @a u8Index is out of bounds or @a pvNewObj is
    *              already stored in the list.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
    //##ModelId=415AC1880203
   inline tVoid *pvReplaceEntry(tU8 u8Index, tVoid *pvNewObj)
   {
      return _oList.pvReplaceEntry(u8Index, pvNewObj);
   }
   /*@}*/

   /** @name Sequential access */
   /*@{*/
   /** @copydoc ahl_tclU16List::vRewind() */
    //##ModelId=415AC187038A
   inline tVoid vRewind() const
   {
      _oList.vRewind();
   }

   /** @copydoc ahl_tclU16List::pvGetSequential() */
    //##ModelId=415AC18703C8
   inline tVoid *pvGetSequential() const
   {
      return _oList.pvGetSequential();
   }

   /** @copydoc ahl_tclU16List::pvGetActEntry() */
    //##ModelId=415AC18800FA
   inline tVoid *pvGetActEntry() const
   {
      return _oList.pvGetActEntry();
   }

   /** @copydoc ahl_tclU16List::bSetActEntry() */
    //##ModelId=415AC1880148
   inline tBool bSetActEntry(const tVoid *pvObj ) const
   {
      return _oList.bSetActEntry(pvObj);
   }
   /*@}*/

private:
   /** The underlying list implementation. */
    //##ModelId=415AC18603D9
   ahl_tclU16List _oList;
};

#endif   //AHL_LIST_H
