#ifndef AHL_CALLBACK_HEADER
#define AHL_CALLBACK_HEADER 1

#define SYSTEM_S_IMPORT_INTERFACE_COMPLETE
#include "system_pif.h"


//*******************************************************************
//* CLASS:       ahl_tclCallbackBase
//*
//* DESCRIPTION: Interface for callbacks.
//*
//*              This class provides a minimal interface for
//*              callbacks as e.g. used by the OSAL.
//*
//*              Quick guide:
//*              - for callback handling, derive a class from
//*                ahl_tclCallbackBase and implement your callback
//*                actions in the vExecute() method.
//*              - instantiate an object of your class.
//*              - use &ahl_tclCallbackBase::vCallback as address
//*                of the callback function and the address of the
//*                object as callback data for the any OSAL function
//*                that uses callbacks.
//*
//*              Example:
//*                class tclMyCallback : public ahl_tclCallbackBase
//*                {
//*                   public:
//*                      tVoid vExecute()
//*                      {
//*                         OSALUTIL_s32FPrintf(fdTrace,
//*                                             "Called back!");
//*                      }
//*                };
//*
//*                tclMyCallback oCallMe;
//*                OSAL_tTimerHandle hTimer;
//*
//*                OSAL_s32TimerCreate(&ahl_tclCallbackBase::vCallback,
//*                                    &oCallMe,
//*                                    &hTimer);
//*
//*                OSAL_s32TimerSetTime(hTimer, 50, 0);
//*
//* AUTHOR:      CM-CR/EES4-Grosse Pawig
//*
//* COPYRIGHT:   (c) 2002 Blaupunkt GmbH
//*
//* HISTORY:
//* 12.07.2002 CM-CR/EES4-Grosse Pawig
//*            Initial Revision
//*******************************************************************
//##ModelId=3D2E8F6A0201
class ahl_tclCallbackBase 
{
  public:

      //*****************************************************************
      //* FUNCTION:    vExecute
      //*
      //* DESCRIPTION: The per-object callback method.
      //*
      //*              Classes derived from ahl_tclCallbackBase should
      //*              re-implement this method for their specific
      //*              callback handling.
      //*
      //* PARAMETER:   -
      //*
      //* RETURNVALUE: -
      //*
      //* AUTHOR:      CM-CR/EES4-Grosse Pawig
      //*
      //* HISTORY:
      //* 12.07.2002 CM-CR/EES4-Grosse Pawig
      //*            Initial Revision
      //*****************************************************************
      //##ModelId=3D2E91F300D8
      virtual tVoid vExecute ();

      //*****************************************************************
      //* FUNCTION:    vCallback
      //*
      //* DESCRIPTION: Callback stub.
      //*
      //*              This method gets called by the OSAL and forward
      //*              to the supplied object.
      //*
      //* PARAMETER:   pvCallee:  (->I)
      //*                 Pointer to an object derived from
      //*                 ahl_tclCallbackBase whose vExecute() method is
      //*                 to be called.
      //*
      //* RETURNVALUE: -
      //*
      //* AUTHOR:      CM-CR/EES4-Grosse Pawig
      //*
      //* HISTORY:
      //* 12.07.2002 CM-CR/EES4-Grosse Pawig
      //*            Initial Revision
      //*****************************************************************
      //##ModelId=3D2ED48001B3
      static tVoid vCallback (tVoid* pvCallee);
      virtual ~ahl_tclCallbackBase() {};
  protected:
  private:
};

//*******************************************************************
//* CLASS:       ahl_tclObjectCallback
//*
//* DESCRIPTION: Callback into an object.
//*
//*              This class can be used to specify a callback that
//*              calls a specific method in a specific class.
//*
//* PARAMETER:   T_Client
//*                 The class which is to be called back.
//*
//* AUTHOR:      CM-CR/EES4-Grosse Pawig
//*
//* COPYRIGHT:   (c) 2002 Blaupunkt GmbH
//*
//* HISTORY:
//* 16.07.2002 CM-CR/EES4-Grosse Pawig
//*            Initial Revision
//*******************************************************************
//##ModelId=3D33DC2800BD
template <class T_Client>
class ahl_tclObjectCallback : public ahl_tclCallbackBase
{
   // tCallback is a pointer to a member function of class T_Client
   //    that takes no parameters and returns tVoid.
   typedef tVoid (T_Client::*tCallback)();

  public:
      inline ahl_tclObjectCallback();

      ahl_tclObjectCallback(const ahl_tclObjectCallback< T_Client > &right);


      //*****************************************************************
      //* FUNCTION:    vSetCallback
      //*
      //* DESCRIPTION: Set the destination of the callback.
      //*
      //*              This method stores the object and member function
      //*              to call whenever the callback is triggered.
      //*
      //* PARAMETER:   poClient:  (->I)
      //*                 Pointer to an object of the client class.
      //*                 Whenever the callback is triggered, the
      //*                 method indicated in 'pFun' is called on this
      //*                 object.
      //*
      //*              pFun:      (->I)
      //*                 Pointer to a member function of the client
      //*                 class that should be called whenever the
      //*                 callback is triggered.
      //*
      //*                 Only methods with the signature
      //*                    tVoid vFun();
      //*                 may be used, i.e. only methods with no
      //*                 parameters and no return value.
      //*
      //* RETURNVALUE: -
      //*
      //* AUTHOR:      CM-CR/EES4-Grosse Pawig
      //*
      //* HISTORY:
      //* 16.07.2002 CM-CR/EES4-Grosse Pawig
      //*            Initial Revision
      //*****************************************************************
      //##ModelId=3D33DDD603D3
      inline tVoid vSetCallback (T_Client* poClient, tCallback pFun);

      //*****************************************************************
      //* FUNCTION:    vExecute
      //*
      //* DESCRIPTION: The per-object callback method.
      //*
      //*              This methods calls the method on the object that
      //*              have been configured with bInit().
      //*
      //* PARAMETER:   -
      //*
      //* RETURNVALUE: -
      //*
      //* AUTHOR:      CM-CR/EES4-Grosse Pawig
      //*
      //* HISTORY:
      //* 12.07.2002 CM-CR/EES4-Grosse Pawig
      //*            Initial Revision
      //*****************************************************************
      //##ModelId=3D33DE7D0075
      virtual tVoid vExecute ();

  protected:
  private:
    // Data Members for Class Attributes


      T_Client* m_poClient;

      tCallback m_pFun;
};

// Class ahl_tclCallbackBase 

// Parameterized Class ahl_tclObjectCallback 


template <class T_Client>
inline ahl_tclObjectCallback<T_Client>::ahl_tclObjectCallback()
  : ahl_tclCallbackBase(), m_poClient(NULL), m_pFun(NULL)
{
}

template <class T_Client>
inline ahl_tclObjectCallback<T_Client>::ahl_tclObjectCallback(const ahl_tclObjectCallback<T_Client> &right)
  : ahl_tclCallbackBase(right),
    m_poClient(right.m_poClient), m_pFun(right.m_pFun)
{
}



template <class T_Client>
inline tVoid ahl_tclObjectCallback<T_Client>::vSetCallback (T_Client* poClient, tCallback pFun)
{
   if ((poClient != NULL) && (pFun != NULL))
   {
      m_poClient = poClient;
      m_pFun = pFun;
   }
}

template <class T_Client>
inline tVoid ahl_tclObjectCallback<T_Client>::vExecute ()
{
   if ((m_poClient != NULL) && (m_pFun != NULL))
   {
      // Note on ->*
      // This is the C++ syntax to call a method through a member
      // function pointer.
      (m_poClient->*m_pFun)();
   }
}

// Parameterized Class ahl_tclObjectCallback 

//*******************************************************************
//* CLASS:       ahl_tclExObjectCallback
//*
//* DESCRIPTION: Callback into an object.
//*
//*              This class can be used to specify a callback that
//*              calls a specific method in a specific class.
//*
//* PARAMETER:   T_Client
//*                 The class which is to be called back.
//*
//* AUTHOR:      CM-CR/EES4-Grosse Pawig
//*
//* COPYRIGHT:   (c) 2002 Blaupunkt GmbH
//*
//* HISTORY:
//* 16.07.2002 CM-CR/EES4-Grosse Pawig
//*            Initial Revision
//*******************************************************************
//##ModelId=3D33DC2800BD
template <class T_Client>
class ahl_tclExObjectCallback : public ahl_tclCallbackBase
{
   // tCallback is a pointer to a member function of class T_Client
   //    that takes no parameters and returns tVoid.
   typedef tVoid (T_Client::*tExCallback)(tVoid* pParam);

  public:
      inline ahl_tclExObjectCallback();

      ahl_tclExObjectCallback(const ahl_tclExObjectCallback< T_Client > &right);


      //*****************************************************************
      //* FUNCTION:    vSetCallback
      //*
      //* DESCRIPTION: Set the destination of the callback.
      //*
      //*              This method stores the object and member function
      //*              to call whenever the callback is triggered.
      //*
      //* PARAMETER:   poClient:  (->I)
      //*                 Pointer to an object of the client class.
      //*                 Whenever the callback is triggered, the
      //*                 method indicated in 'pFun' is called on this
      //*                 object.
      //*
      //*              pFun:      (->I)
      //*                 Pointer to a member function of the client
      //*                 class that should be called whenever the
      //*                 callback is triggered.
      //*
      //*                 Only methods with the signature
      //*                    tVoid vFun();
      //*                 may be used, i.e. only methods with no
      //*                 parameters and no return value.
      //*
      //* RETURNVALUE: -
      //*
      //* AUTHOR:      CM-CR/EES4-Grosse Pawig
      //*
      //* HISTORY:
      //* 16.07.2002 CM-CR/EES4-Grosse Pawig
      //*            Initial Revision
      //*****************************************************************
      //##ModelId=3D33DDD603D3
      inline tVoid vSetCallback (T_Client* poClient, tExCallback pFun, tVoid* pParam);

      //*****************************************************************
      //* FUNCTION:    vExecute
      //*
      //* DESCRIPTION: The per-object callback method.
      //*
      //*              This methods calls the method on the object that
      //*              have been configured with bInit().
      //*
      //* PARAMETER:   -
      //*
      //* RETURNVALUE: -
      //*
      //* AUTHOR:      CM-CR/EES4-Grosse Pawig
      //*
      //* HISTORY:
      //* 12.07.2002 CM-CR/EES4-Grosse Pawig
      //*            Initial Revision
      //*****************************************************************
      //##ModelId=3D33DE7D0075
      virtual tVoid vExecute ();

  protected:
  private:
    // Data Members for Class Attributes

      T_Client* m_poClient;

      tExCallback m_pFun;

      tVoid* m_pParam;
};

// Class ahl_tclCallbackBase 

// Parameterized Class ahl_tclExObjectCallback 


template <class T_Client>
inline ahl_tclExObjectCallback<T_Client>::ahl_tclExObjectCallback()
  : ahl_tclCallbackBase(), m_poClient(NULL), m_pFun(NULL), m_pParam(NULL)
{
}

template <class T_Client>
inline ahl_tclExObjectCallback<T_Client>::ahl_tclExObjectCallback(const ahl_tclExObjectCallback<T_Client> &right)
  : ahl_tclCallbackBase(right),
    m_poClient(right.m_poClient), m_pFun(right.m_pFun), m_pParam(right.m_pParam)
{
}



template <class T_Client>
inline tVoid ahl_tclExObjectCallback<T_Client>::vSetCallback (T_Client* poClient, tExCallback pFun, tVoid* pParam)
{
   if ((poClient != NULL) && (pFun != NULL))
   {
      m_poClient = poClient;
      m_pFun = pFun;
      m_pParam = pParam;
   }
}

template <class T_Client>
inline tVoid ahl_tclExObjectCallback<T_Client>::vExecute ()
{
   if ((m_poClient != NULL) && (m_pFun != NULL))
   {
      // Note on ->*
      // This is the C++ syntax to call a method through a member
      // function pointer.
      (m_poClient->*m_pFun)(m_pParam);
   }
}

#endif
