/************************************************************************
* FILE:          ahl_BaseClient.cpp
* PROJECT:       All
* SW-COMPONENT:  AHL
*----------------------------------------------------------------------
*
* DESCRIPTION: 
*              
*----------------------------------------------------------------------
*
* AUTHOR:  ESE-Neumann  (jn82hi)
*          TMS-Kempen   (kmr2hi)
*          TMS-Jentsch  (jhr2hi)           
*
* COPYRIGHT:    (c) 2008 Robert Bosch GmbH, Hildesheim
*
* HISTORY:      
*************************************************************************/

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"

/* Inclusion of OSAL Interface */
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AMT_S_IMPORT_INTERFACE_GENERIC
#include "amt_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ETRACE_S_IMPORT_INTERFACE_ERRMEM
#include "etrace_if.h"

#define SCD_S_IMPORT_INTERFACE_GENERIC
#include "scd_if.h"

#include "ahl_BaseOneThreadObject.h"
#include "ahl_BaseOneThreadClienthandler.h"
#include "ahl_BaseOneThreadApp.h"
    
#include "ahl_ccaExtensionTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_AHL_CCA_EXTENSION
#include "trcGenProj/Header/ahl_BaseOneThreadClienthandler.cpp.trc.h"
#endif

/************************************************
* Helper class for administration of a property 
*************************************************/

/************************************************
* Base class of a client handler
*************************************************/

/* constructor */
ahl_tclBaseOneThreadClientHandler::ahl_tclBaseOneThreadClientHandler( 
      ahl_tclBaseOneThreadApp*   poMainAppl,
      tU16            u16ServiceID,
      tU16            u16ServiceMajorVersion,
      tU16            u16ServiceMinorVersion,
      tU16            u16ServerAppID,
      tU16            u16OwnSubID
   )
   : ahl_tclBaseOneThreadObject(poMainAppl, u16ServiceID)
   , _tclRegisterPair(AMT_C_U32_STATE_INVALID, AMT_C_U32_STATE_OFF)
   , _tclUnregisterPair(AMT_C_U32_STATE_INVALID, AMT_C_U32_STATE_OFF)
   , _u16ServiceMajorVersion(u16ServiceMajorVersion)
   , _u16ServiceMinorVersion(u16ServiceMinorVersion)
   , _u16ServerAppID(u16ServerAppID)
   , _bDedicatedServer(FALSE)
   , _u16OwnSubID(u16OwnSubID)
   , _u16RegID(AMT_C_U16_REGID_INVALID)
{
   /* register this client to the Application */
   if (_poMainAppl != NULL)
   {
       _poMainAppl->vAddObject(this);
       _u16OwnAppID = _poMainAppl->u16GetAppId();
   }
   else
   {
       _u16OwnAppID = AMT_C_U16_APPID_INVALID;
       /* main application should never be NULL, please check constructor of derived class */
       ETG_TRACE_ERRMEM(("%x constructor: _poMainAppl should not be NULL", ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID) ));
       FATAL_M_ASSERT_ALWAYS();
   }

   if (_u16ServerAppID != AMT_C_U16_APPID_INVALID)
   {
     _bDedicatedServer = TRUE;
   }

   ahl_sAdminHelper emptyHelper;
   emptyHelper.u16FID = 0;
   emptyHelper.tclAdmin.vReset();

   _asAdminHelpers.resize(AHL_MAX_FUIDS);
   _asAdminHelpers.assign(AHL_MAX_FUIDS, emptyHelper);
   _nNumberOfFUIDs = 0;
   _bServiceAvailable = FALSE;

   _tclRegisterPair.bSetDefault(AHL_PAIRLIST_DEFAULT_REGISTER_FOR_SERVICE);
   _tclUnregisterPair.bSetDefault(AHL_PAIRLIST_DEFAULT_UNREGISTER_FOR_SERVICE);
}


/* destructor */
ahl_tclBaseOneThreadClientHandler::~ahl_tclBaseOneThreadClientHandler(tVoid)
{
    /* register this client to the Application */
    if (_poMainAppl != NULL)
    {
        _poMainAppl->vRemoveObject(this);
    }
    else
    {
       /* main application should never be NULL, please check constructor of derived class */
       ETG_TRACE_ERRMEM(("%x destructor: _poMainAppl should not be NULL", ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID) ));
    }
    _poMainAppl = NULL; 
}

tVoid ahl_tclBaseOneThreadClientHandler::vAddAutoRegisterForProperty( tU16 u16FID)
{
#ifdef AHL_CHECK_FUID_STILL_ADDED
    tU32 i;
    for (i=0; i<_nNumberOfFUIDs; ++i)
    {
        if (_asAdminHelpers[i].u16FID == u16FID)
        {
            // service id still added
            ETG_TRACE_FATAL(("%x vAddAutoRegisterForProperty: FID %d in ClientHandler for service %x App %x still added !", 
                             ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID) , u16FID, 
                             ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), ETG_ENUM(ail_u16AppId, (tU16)_u16ServerAppID) ));
            return;
        }
    }
#endif
    if (_nNumberOfFUIDs >= _asAdminHelpers.capacity())
    {
       ETG_TRACE_SYS_MIN(("%x vAddAutoRegisterForProperty: maximum number of FUID (%d) reached in Clienthandler for service %x App %x ! Double capacity now !", 
                        ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), _asAdminHelpers.capacity(), ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), ETG_ENUM(ail_u16AppId, (tU16)_u16ServerAppID) ));
       _asAdminHelpers.resize(2 * _asAdminHelpers.capacity()); // double the current size of the array
    }

    if (_nNumberOfFUIDs < _asAdminHelpers.capacity())
    {
       _asAdminHelpers[_nNumberOfFUIDs].u16FID = u16FID;
       _asAdminHelpers[_nNumberOfFUIDs].tclAdmin.vReset();
   
       // try to register for FIDs 
       if ( bIfServiceAvailable() )
       {
          if ( bRegisterForFID( u16FID) )  
          {
              _asAdminHelpers[_nNumberOfFUIDs].tclAdmin.vRegStarted();
          }
          else
          {
             ETG_TRACE_FATAL(("%x vAddAutoRegisterForProperty: could not register for service %x app %x !", 
                              ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID) , ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), ETG_ENUM(ail_u16AppId, (tU16)_u16ServerAppID) ));
          }
       }
   
       ++_nNumberOfFUIDs;
    }
    else
    {
       /* 
         maximum number of properties reached, please inform developer or increase AHL_MAX_FUIDS
       */
        ETG_TRACE_ERRMEM(("%x vAddAutoRegisterForProperty: maximum number of FUID reached in Clienthandler for service %x app %x !", 
                         ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID) , ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), ETG_ENUM(ail_u16AppId, (tU16)_u16ServerAppID) ));
    }
}

tVoid ahl_tclBaseOneThreadClientHandler::vRemoveAutoRegisterForProperty(tU16 u16FID)
{
   tU32 i;
   for (i=0; i<_nNumberOfFUIDs; ++i)
   {
       if (_asAdminHelpers[i].u16FID == u16FID)
       {

          // try to unregister 
          if ( _asAdminHelpers[i].tclAdmin.bIsReadyForWork() )
          {
              if ( bUnregisterForFID( _asAdminHelpers[i].u16FID ) )
              {
                 _asAdminHelpers[i].tclAdmin.vUnRegStarted();
              }
          }

          --_nNumberOfFUIDs;
          _asAdminHelpers[i] = _asAdminHelpers[_nNumberOfFUIDs];
       }
   }
}

/*************************************************************************
*
* FUNCTION: tVoid vOnNewAppState( tU32 u32OldAppState, tU32 u32AppState )
* 
* DESCRIPTION: handles state change messages form the spm
*
* PARAMETER: old state, new state
*
* RETURNVALUE: tVoid
*
*************************************************************************/
tVoid ahl_tclBaseOneThreadClientHandler::vOnNewAppState(tU32 u32OldAppState, tU32 u32AppState)
{
  if (u32OldAppState == u32AppState) 
  { 
     return; //switching to the same state
  } 

  if (_tclRegisterPair.bIfPairIncluded(u32OldAppState, u32AppState) == TRUE)
  {
    ETG_TRACE_USR2(("Application %x SubID %d Clienthandler vOnNewAppState: try to register for service in Clienthandler app %X !", 
                    ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID) , 
                    _u16OwnSubID, 
                    ETG_ENUM(ail_u16AppId, (tU16)_u16ServerAppID) ));

    if (bRegisterForService() == FALSE)
    {
      ETG_TRACE_ERRMEM(("Application %x SubID %d vOnNewAppState: bRegisterForService returns false for client with SrvId %X !", 
                         ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID) , 
                         _u16OwnSubID, 
                         ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID) ));
    }
  }

  if (_tclUnregisterPair.bIfPairIncluded(u32OldAppState, u32AppState) == TRUE)
  {
    ETG_TRACE_USR2(("Application %x SubID %d vOnNewAppState: call bUnregisterForService with SrvId %X !", 
                    ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID) , 
                    _u16OwnSubID, 
                    ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID) ));

    /* --- unregister for service --- */
    NORMAL_M_ASSERT(bUnregisterForService() == TRUE);
    vAutoResetConnection();
  }

  vOnNewAppStateNotify(u32OldAppState, u32AppState);
}


/*************************************************************************
*
* FUNCTION: tVoid vOnServiceState
* 
* DESCRIPTION: handle a state change of the service we use
*
* PARAMETER: 
*
* RETURNVALUE: tVoid
*
*************************************************************************/
tVoid ahl_tclBaseOneThreadClientHandler::vOnServiceState
( 
   tU16 u16ServiceId,
   tU16 /*u16ServerId*/,
   tU16 /*u16RegisterId*/,
   tU8  u8ServiceState,
   tU16 u16SubId
)
{

   // check the correct service ID
   if ( u16ServiceId == _u16ServiceID )
   {
      ETG_TRACE_USR2(("%x SubId %d vOnServiceState: SrvId %x ServiceState %d", 
                      ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), 
                      u16SubId,
                      ETG_ENUM(ail_u16ServiceId, (tU16) _u16ServiceID), 
                      ETG_ENUM(ail_u8ServiceState, (tU8) u8ServiceState) ));

      /* --- analyze the state of the server --- */
      switch ( u8ServiceState )
      {
         case AMT_C_U8_SVCSTATE_AVAILABLE:
         {
            _bServiceAvailable = TRUE;

            /* --- the service is now available. --- */
            vOnServiceAvailable();

            // Automatic Handling of FIDs
            if ( bCheckIds() )
            {
                   if ( bRegisterForAllFID() )
                   {
                      ETG_TRACE_USR2(("%x vOnServiceState: bRegisterForAllFID returns true", ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID)  ));
                   }
                   else
                   {
                      ETG_TRACE_FATAL(("%x vOnServiceState: bRegisterForAllFID returns false", ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID)  ));
                   }
            }
            else
            {
                ETG_TRACE_ERRMEM(("%x vOnServiceState: bCheckIds returns false", ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID)  ));
            }
            break;
         }
         case AMT_C_U8_SVCSTATE_NOT_AVAILABLE:
         {
            _bServiceAvailable = FALSE;

            /* --- the service is not available anymore.  implement your reaction if neccessary here. --- */
            vOnServiceUnavailable();
            break;
         }
         case AMT_C_U8_SVCSTATE_REG_DENIED:
         {
            _bServiceAvailable = FALSE;

            /* --- the server has lost our registration. --- */
            vAutoResetConnection();

            ETG_TRACE_SYS_MIN(("%x vOnServiceState(App 0x%X): AMT_C_U8_SVCSTATE_REG_DENIED", 
                               ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID) , ETG_ENUM(ail_u16AppId, (tU16)_u16ServerAppID) ));
            break;
         }
         default:
         {
            _bServiceAvailable = FALSE;

            vAutoResetConnection();
            ETG_TRACE_ERRMEM(("%x vOnServiceState(Srv %x App %x): unknown u8ServiceState (%d)", 
                             ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID) , 
                             ETG_ENUM(ail_u16AppId, (tU16)_u16ServerAppID), 
                             ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID),
                             ETG_ENUM(ail_u8ServiceState, (tU8)u8ServiceState) ));
         }
      }//switch
   }
   else
   {
       ETG_TRACE_ERRMEM(("%x vOnServiceState(App %x): Invalid service ID (%x != %x)", 
                        ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), 
                        ETG_ENUM(ail_u16AppId, (tU16)_u16ServerAppID), 
                        ETG_ENUM(ail_u16ServiceId, (tU16)u16ServiceId), 
                        ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID) ));
   }
}


/*************************************************************************
*  FUNCTION:    bRegisterForService();
*
*   DESCRIPTION: register for a service
*
*   PARAMETER:   tBool
*
*   RETURNVALUE: TRUE == NO_ERROR, FALSE == ERROR
*
*   History:
*   InitialVersion
* 
*************************************************************************/
tBool ahl_tclBaseOneThreadClientHandler::bRegisterForService()
{
  tBool bReturn = TRUE;

  ETG_TRACE_USR1(("Application %x SubID %d Clienthandler->bRegisterForService of service %x to App %x ", 
                  ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), 
                  _u16OwnSubID, 
                  ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), 
                  ETG_ENUM(ail_u16AppId, (tU16)_u16ServerAppID)  )); 

   /* ---  registration invalid ? --- */
  if (_u16RegID == AMT_C_U16_REGID_INVALID)
  {
       /* ---  try to register --- */
    if (_poMainAppl->bRegisterAsync(_u16ServiceID, 
                                    _u16ServiceMajorVersion, 
                                    _u16ServiceMinorVersion, 
                                    _u16OwnSubID, 
                                    _u16ServerAppID) == TRUE)
    {
      // all ok, registering in process, now wait for answer
      ETG_TRACE_USR1(("Application %x SubID %d Clienthandler->bRegisterAsync of service %x to App %x returns true ", 
                      ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), 
                      _u16OwnSubID, 
                      ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), 
                      ETG_ENUM(ail_u16AppId, (tU16)_u16ServerAppID)  )); 
    }
    else
    {
      bReturn = FALSE;
    }
  }
  else
  {
    /* +++
    it could be an error to call this function, if we are already registered
    decide if you need error handling here
    +++ */
    ETG_TRACE_COMP(("Application %x SubID %d Clienthandler->bRegisterAsync of service 0x%x to App %x while system is not ready ", 
                    ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), 
                    _u16OwnSubID, 
                    ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), 
                    ETG_ENUM(ail_u16AppId, (tU16)_u16ServerAppID) )); 
  }

  return (bReturn);
}

/*************************************************************************
*  FUNCTION:    bUnregisterForService();
*
*   DESCRIPTION: unregister for a service
*
*   PARAMETER:   tBool
*
*   RETURNVALUE: TRUE == NO_ERROR, FALSE == ERROR
*
*   History:
*   InitialVersion
* 
*************************************************************************/
tBool ahl_tclBaseOneThreadClientHandler::bUnregisterForService()
{
   /* --- unregister for Service --- */
   if ( _poMainAppl != NULL )
   {
       ETG_TRACE_USR1(("Application %x SubID %d Clienthandler->bUnregisterForService of service %x to App %x", 
                       ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), 
                       _u16OwnSubID,
                       ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), 
                       ETG_ENUM(ail_u16AppId, (tU16)_u16ServerAppID)  )); 
      _poMainAppl->vUnregisterService( _u16ServiceID,
                                       _u16ServerAppID, 
                                       AMT_C_U16_REGID_INVALID, 
                                       _u16OwnSubID);
   }
   else
   {
      ETG_TRACE_ERRMEM(("Application %x SubID %d Clienthandler->bUnregisterForService of service %x to App %x while _poMainAppl is NULL ", 
                       ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), 
                       _u16OwnSubID,
                       ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), 
                       ETG_ENUM(ail_u16AppId, (tU16)_u16ServerAppID)  )); 
      return FALSE;
   }
   
   if (_bDedicatedServer == FALSE)
   {
     _u16ServerAppID = AMT_C_U16_APPID_INVALID;
   }

   // Make register-ID invalid:
   _u16RegID = AMT_C_U16_REGID_INVALID;

   return TRUE;
}

/*************************************************************************
*  FUNCTION:    bCheckIds();
*
*   DESCRIPTION: check function for bRegisterForAllFID, bUnregisterForAllFID
*
*   PARAMETER:   tBool
*
*   RETURNVALUE: TRUE == NO_ERROR, FALSE == ERROR
*
*   History:
*   InitialVersion
* 
*************************************************************************/
tBool ahl_tclBaseOneThreadClientHandler::bCheckIds() const
{
   if (
         ( u16GetServerAppID() == AMT_C_U16_APPID_INVALID )
         ||
         ( u16GetRegID() == AMT_C_U16_REGID_INVALID )
      )
   {
       return FALSE;
   }
   else
   {    
      return TRUE;
   }
}

/*************************************************************************
* FUNCTION:    bRegisterForAllFID
*
* DESCRIPTION: register for all function with upreg.
*
* PARAMETER:   None
*
* RETURNVALUE: TRUE == NO_ERROR, FALSE == ERROR
*
*************************************************************************/
tBool ahl_tclBaseOneThreadClientHandler::bRegisterForAllFID()
{
    tBool bRetVal = TRUE;
    tU32 i;
    
    ETG_TRACE_USR1(("%x Clienthandler->bRegisterForAllFID of service %x to App %x", 
                    ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), 
                    ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), 
                    ETG_ENUM(ail_u16AppId, (tU16)_u16ServerAppID)  )); 
    
    for (i=0; i<_nNumberOfFUIDs; ++i)
    {
        if ( !_asAdminHelpers[i].tclAdmin.bIsRegDone())
        {
            if ( bRegisterForFID( _asAdminHelpers[i].u16FID) )  
            {
                _asAdminHelpers[i].tclAdmin.vRegStarted();
            }
            else
            {
                bRetVal = FALSE;
            }
        }
    }

    // return the Flag value after the individual FI Registration
   return bRetVal;
}

/*************************************************************************
* FUNCTION:    bUnRegisterForAllFID
*
* DESCRIPTION: Un register for all function with relupreg.
*
* PARAMETER:   None
*
* RETURNVALUE: TRUE == NO_ERROR, FALSE == ERROR
*
* History: InitialVersion
*
*************************************************************************/
tBool ahl_tclBaseOneThreadClientHandler::bUnRegisterForAllFID()
{
    tBool bRetVal = TRUE;
    tU32 i;
    
    ETG_TRACE_USR1(("%x Clienthandler->bUnregisterForAllFID of service %x to App %x", 
                    ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), 
                    ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), 
                    ETG_ENUM(ail_u16AppId, (tU16)_u16ServerAppID)  )); 
    
    for (i=0; i<_nNumberOfFUIDs; ++i)
    {
        if ( _asAdminHelpers[i].tclAdmin.bIsReadyForWork() )
        {
            if ( bUnregisterForFID( _asAdminHelpers[i].u16FID ) ) 
            {
               _asAdminHelpers[i].tclAdmin.vUnRegStarted();
            }
            else
            {
               bRetVal = FALSE;
            }
        }
    }
    
   // return the Flag value after the individual FI UNRegistration
   return bRetVal;
}

/*************************************************************************
* FUNCTION:    vAutoResetConnection
*
* DESCRIPTION: Reset all the flags of the current connections (which are registrated automatically)
*
* PARAMETER:   NONE
*
* RETURNVALUE: tVoid
*
* History: InitialVersion
*
*************************************************************************/
tVoid ahl_tclBaseOneThreadClientHandler::vAutoResetConnection()
{
   tU32 i;
   for (i=0; i<_nNumberOfFUIDs; ++i)
   {
      _asAdminHelpers[i].tclAdmin.vReset();
   }
}


/*************************************************************************
* FUNCTION:    bUpRegMessageFactory(tU16 u16FunctionId, amt_tclServiceData& roOutMsg)
*
* DESCRIPTION: factory function for property unregister messages
*
* PARAMETER:   u16FunctionId: Function ID of the function
*
* RETURNVALUE: TRUE == NO_ERROR, FALSE == ERROR
*
* History: InitialVersion
*
*************************************************************************/
tBool ahl_tclBaseOneThreadClientHandler::bUpRegMessageFactory(tU16 u16FunctionId, amt_tclServiceData& roOutMsg)
{
    /* form an empty CCA msg for FID registration */
    gm_tclEmptyMessage oUpRegMessage(_u16OwnAppID, 0, 0, 0, _u16ServiceID, u16FunctionId, AMT_C_U8_CCAMSG_OPCODE_UPREG);
    return oUpRegMessage.bHandOver(&roOutMsg);
}

/*************************************************************************
* FUNCTION:    bRelUpRegMessageFactory(tU16 u16FunctionId, amt_tclServiceData& roOutMsg)
*
* DESCRIPTION: factory function for property unregister messages
*
* PARAMETER:   u16FunctionId: Function ID of the function
*
* RETURNVALUE: TRUE == NO_ERROR, FALSE == ERROR
*
* History: InitialVersion
*
*************************************************************************/
tBool ahl_tclBaseOneThreadClientHandler::bRelUpRegMessageFactory(tU16 u16FunctionId, amt_tclServiceData& roOutMsg)
{
    /* form an empty CCA msg for FID unregistration */
    gm_tclEmptyMessage oRelUpRegMessage(_u16OwnAppID, 0, 0, 0, _u16ServiceID, u16FunctionId, AMT_C_U8_CCAMSG_OPCODE_RELUPREG);
    return oRelUpRegMessage.bHandOver(&roOutMsg);
}

/*************************************************************************
* FUNCTION:    bRegisterForFID( tU16 u16AppID, tU16 u16RegID, tU16 u16FID )
*
* DESCRIPTION: register for a function of a service
*               !!! before this, you must register for the service !!!
*
* PARAMETER:   u16FID:    FID of the function
*              u16RegID:  RegisterID
*              u16AppID:  ApplicationID
*
* RETURNVALUE: TRUE == NO_ERROR, FALSE == ERROR
*
* History: InitialVersion
*
*************************************************************************/
tBool ahl_tclBaseOneThreadClientHandler::bRegisterForFID(tU16 u16FID)
{
   tBool   bRetVal = TRUE;

   if (_poMainAppl != NULL)
   {
       amt_tclServiceData oUpRegFactoryMessage;

       if (bUpRegMessageFactory(u16FID, oUpRegFactoryMessage) == TRUE)
       {
           vInitServiceData(oUpRegFactoryMessage, AHL_C_U16_CMDCOUNT_UPREGMSG, u16FID, AMT_C_U8_CCAMSG_OPCODE_UPREG);

           /* error handling of the posted message is done in enPostMyMessage */
           ail_tenCommunicationError enError = _poMainAppl->enPostMessage( &oUpRegFactoryMessage, TRUE );
           if (enError  == AIL_EN_N_NO_ERROR )
           {
              /* +++ message send success, make a trace if you want +++ */
              ETG_TRACE_USR1(("%x Clienthandler->bRegisterForFID of FID %d service %x to App %x RegID %d", 
                              ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), 
                              u16FID, 
                              ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), 
                              ETG_ENUM(ail_u16AppId, (tU16)_u16ServerAppID) , _u16RegID )); 
           }
           else
           {
              bRetVal = FALSE;
              ETG_TRACE_ERRMEM(("%x Clienthandler->bRegisterForFID of FID %d service %x to App %x RegID %d", 
                                ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), 
                                u16FID, 
                                ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), 
                                ETG_ENUM(ail_u16AppId, (tU16)_u16ServerAppID) , _u16RegID )); 
              ETG_TRACE_ERRMEM(("%x Clienthandler->bRegisterForFID enPostMessage returns error %x", 
                                ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), enError )); 
           }
       }
       else
       {
           bRetVal = FALSE;
           ETG_TRACE_ERRMEM(("%x bRegisterForFID: the UpRegMessageFactory for service %x does not return a valid message for FID %d", 
                             ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), u16FID  ));
       }
   }
   else
   {
       ETG_TRACE_ERRMEM(("%x Clienthandler->bRegisterForFID() of FID %d service %x => _poMainAppl == NULL", 
                         ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), u16FID, ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID)   ));

       bRetVal = FALSE;
   }

   /* return the status of registration as TRUE or FALSE */
   return bRetVal;
}

/*************************************************************************
* FUNCTION:   bUnregisterForFID( tU16 u16AppID, tU16 u16RegID, tU16 u16FID )
*
* DESCRIPTION: unregister for a function of a service
*
* PARAMETER:   u16FID:    FID of the function
*              u16RegID:  RegisterID
*
* RETURNVALUE: TRUE == NO_ERROR, FALSE == ERROR
*
* History: InitialVersion
*
*************************************************************************/
tBool ahl_tclBaseOneThreadClientHandler::bUnregisterForFID(tU16 u16FID )
{
   tBool   bRetVal = TRUE;

   if (_poMainAppl != NULL)
   {
       amt_tclServiceData oRelUpRegFactoryMessage;

       if (bRelUpRegMessageFactory(u16FID, oRelUpRegFactoryMessage) == TRUE)
       {
           vInitServiceData(oRelUpRegFactoryMessage, AHL_C_U16_CMDCOUNT_UPREGMSG, u16FID, AMT_C_U8_CCAMSG_OPCODE_RELUPREG);
    
           /* error handling of the posted message is done in enPostMyMessage */
           ail_tenCommunicationError enError = _poMainAppl->enPostMessage( &oRelUpRegFactoryMessage, TRUE );
           if (enError  == AIL_EN_N_NO_ERROR )
           {
              ETG_TRACE_USR1(("%x Clienthandler->bUnregisterForFID of FID %d service %x to App %x RegID %d", 
                              ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), 
                              u16FID, 
                              ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), 
                              ETG_ENUM(ail_u16AppId, (tU16)_u16ServerAppID) , _u16RegID )); 
              /* +++ message send success, make a trace if you want +++ */
           }
           else
           {
              ETG_TRACE_ERRMEM(("%x Clienthandler->bUnregisterForFID of FID %d service %x to App %x RegID %d", 
                                ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), u16FID, 
                                ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), 
                                ETG_ENUM(ail_u16AppId, (tU16)_u16ServerAppID) , _u16RegID )); 
              ETG_TRACE_ERRMEM(("%x Clienthandler->bUnregisterForFID enPostMessage returns error %x", 
                                ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), enError )); 
              bRetVal= FALSE;
              /* +++ message send error, enter error handling ( trace or assert ) +++ */
           }
       }
       else
       {
           ETG_TRACE_ERRMEM(("%x bUnregisterForFID: the RelUpRegMessageFactory for service %x does not return a valid message for FID %d", 
               ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), u16FID  ));
       }
   }
   else
   {
       ETG_TRACE_ERRMEM(("%x Clienthandler->bUnregisterForFID() of FID %d service %x => _poMainAppl == NULL", 
                         ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), u16FID, ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID)   ));

       bRetVal= FALSE;
   }

   /* return the status of registration as TRUE or FALSE */
   return bRetVal;
}

tVoid ahl_tclBaseOneThreadClientHandler::vOnError(tU16 u16FID, amt_tclServiceData* pServiceDataMessage)
{
   ETG_TRACE_ERR(("%x vOnError message for service %x FID %d", 
                  ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID) , ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), u16FID )); 

   if(!(ahl_tclBaseOneThreadClientHandler::bDefaultSvcDataHandler(this, pServiceDataMessage)))
   {
       vOnUnknownMessage(pServiceDataMessage);
   }
}

tVoid ahl_tclBaseOneThreadClientHandler::vOnStatus(tU16 u16FID, amt_tclServiceData* pServiceDataMessage)
{
   ETG_TRACE_USR2(("%x vOnStatus message for service %x FID %d", 
                   ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID) , ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), u16FID )); 

   if(!(ahl_tclBaseOneThreadClientHandler::bDefaultSvcDataHandler(this, pServiceDataMessage)))
   {
       vOnUnknownMessage(pServiceDataMessage);
   }
}

tVoid ahl_tclBaseOneThreadClientHandler::vOnMethodResult(tU16 u16FID, amt_tclServiceData* pServiceDataMessage)
{
   ETG_TRACE_USR1(("%x vOnMethodResult message for service %x FID %d", 
                   ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID) , ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), u16FID )); 

   if(!(ahl_tclBaseOneThreadClientHandler::bDefaultSvcDataHandler(this, pServiceDataMessage)))
   {
       vOnUnknownMessage(pServiceDataMessage);
   }
}

tVoid ahl_tclBaseOneThreadClientHandler::vOnAbortResult(tU16 u16FID, amt_tclServiceData* pServiceDataMessage)
{
   ETG_TRACE_USR1(("%x vOnAbortResult message for service %x FID %d", 
                   ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID) , ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), u16FID )); 

   if(!(ahl_tclBaseOneThreadClientHandler::bDefaultSvcDataHandler(this, pServiceDataMessage)))
   {
       vOnUnknownMessage(pServiceDataMessage);
   }
}

tVoid ahl_tclBaseOneThreadClientHandler::vOnMethodResultFirst(tU16 u16FID, amt_tclServiceData* pServiceDataMessage)
{
   ETG_TRACE_USR1(("%x vOnMethodResultFirst message for service %x FID %d", 
                   ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID) , ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), u16FID )); 

   if(!(ahl_tclBaseOneThreadClientHandler::bDefaultSvcDataHandler(this, pServiceDataMessage)))
   {
       vOnUnknownMessage(pServiceDataMessage);
   }
}

tVoid ahl_tclBaseOneThreadClientHandler::vOnMethodResultMiddle(tU16 u16FID, amt_tclServiceData* pServiceDataMessage)
{
   ETG_TRACE_USR1(("%x vOnMethodResultMiddle message for service %x FID %d", 
                   ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID) , ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), u16FID )); 

   if(!(ahl_tclBaseOneThreadClientHandler::bDefaultSvcDataHandler(this, pServiceDataMessage)))
   {
       vOnUnknownMessage(pServiceDataMessage);
   }
}

tVoid ahl_tclBaseOneThreadClientHandler::vOnMethodResultLast(tU16 u16FID, amt_tclServiceData* pServiceDataMessage)
{
   ETG_TRACE_USR1(("%x vOnMethodResultLast message for service %x FID %d", 
                   ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID) , ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), u16FID )); 

   if(!(ahl_tclBaseOneThreadClientHandler::bDefaultSvcDataHandler(this, pServiceDataMessage)))
   {
       vOnUnknownMessage(pServiceDataMessage);
   }
}

tVoid ahl_tclBaseOneThreadClientHandler::vMyDispatchMessage(amt_tclServiceData *pServiceDataMessage)
{
    /* only for debugging

    const mm_tMsgMap* pMessageMap;
    const mm_tMsgMapEntry* pEntry;

    tU16 u16FunctionID = pServiceDataMessage->u16GetFunctionID();
    const mm_tMsgMapEntry* lpEntry;

    ET_TRACE_ERROR1(TR_COMP_SYSTEM, "vDispatchMessage %d", u16FunctionID); 

    // look through message map to see if it applies to us
    // printf("searching message map...\n");
    for (pMessageMap = GetMsgMap(); pMessageMap != NULL; pMessageMap = pMessageMap->pBaseMap)
    {
        ET_TRACE_ERROR2(TR_COMP_SYSTEM, "################### %04x %04x", pMessageMap, pMessageMap->pEntries); 

        // printf("searching entry ...\n");
        lpEntry = pMessageMap->pEntries;
        while ( lpEntry && ((lpEntry->u16FunctionID) != 0))
        {
             ET_TRACE_ERROR3(TR_COMP_SYSTEM, "################### %04x %04x %d", pMessageMap, lpEntry, lpEntry->u16FunctionID); 
             lpEntry++;
        }

       pEntry = pFindMsgEntry(pMessageMap->pEntries, pServiceDataMessage);
       if(pEntry != 0)
       {
           ET_TRACE_ERROR(TR_COMP_SYSTEM, "function found"); 
          // printf("function found: %d -> %x\n", pEntry->u16FunctionID, pEntry->pHandlerFunction);
       }
    }

    */
   ETG_TRACE_USR4(("%x DispatchMessage for App %x", ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID) , ETG_ENUM(ail_u16AppId, (tU16)_u16ServerAppID)  )); 

   tU16  u16CmdCounter = pServiceDataMessage->u16GetCmdCounter();

   switch ( pServiceDataMessage->u8GetOpCode() )
   {
      case AMT_C_U8_CCAMSG_OPCODE_STATUS: 
      {
         ahl_sAdminHelper* poStatusAdmin = sFindAdminHelper(pServiceDataMessage->u16GetFunctionID());

         /* +++ this can be an answer to GET / SET / UPREG or RELUPREG
         (up to now only UPREG and RELUPREG are used) +++ */

         if (poStatusAdmin != NULL)
         {
            // could happen on GET method, without register for property

            /* --- check the CmdCounter of the answer and call the administration function --- */
            if (  poStatusAdmin->tclAdmin.bIsRegInProgress()
                  &&
                  AHL_C_U16_CMDCOUNT_UPREGMSG == u16CmdCounter )
            {
               poStatusAdmin->tclAdmin.vRegFinished();
            }
            else if ( poStatusAdmin->tclAdmin.bIsUnRegInProgress()
                      &&
                      AHL_C_U16_CMDCOUNT_RELUPREGMSG == u16CmdCounter )
            {
               poStatusAdmin->tclAdmin.vUnRegFinished();
            }
         }

         vOnStatus(pServiceDataMessage->u16GetFunctionID(), pServiceDataMessage);
      }
      break;
      case AMT_C_U8_CCAMSG_OPCODE_ERROR:
      {
         ahl_sAdminHelper* poStatusAdmin = sFindAdminHelper(pServiceDataMessage->u16GetFunctionID());
         if (poStatusAdmin != NULL)
         {

            /* +++ this can be an answer to GET / SET / UPREG / RELUPREG and METHODSTART +++ */
            amt_tclServiceDataError oErrorMsg( pServiceDataMessage );
            tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();
   
            switch ( u16ErrorCode )
            {
               case AMT_C_U16_ERROR_UPREG_FAILURE:
               {
                  /* +++
                  To much clients which are already registered
                  or you tried to register twice.
                  +++ */
                  ETG_TRACE_SYS_MIN(("%x ahl_bStatusMessagePreProcessing: AMT_C_U16_ERROR_UPREG_FAILURE", ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID)  ));
                  poStatusAdmin->tclAdmin.vRegFailed();
               }
               break;
               case AMT_C_U16_ERROR_RELUPREG_FAILURE:
               {
                  /* +++
                  The de-registration from a property failed.
                  Are you really registered?
                  +++ */
                  ETG_TRACE_SYS_MIN(("%x ahl_bStatusMessagePreProcessing: AMT_C_U16_ERROR_RELUPREG_FAILURE", ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID)  ));
                  poStatusAdmin->tclAdmin.vUnRegFailed();
               }
               break;
               case AMT_C_U16_ERROR_UNKNOWN_REG_ID:
               {
                  /* --- This code is used by AIL, if connection is no longer accessible  --- */
                  ETG_TRACE_SYS_MIN(("%x ahl_bStatusMessagePreProcessing: AMT_C_U16_ERROR_UNKNOWN_REG_ID", ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID)  ));
                  poStatusAdmin->tclAdmin.vReset();
               }
               break;
               default:
               {
                  if ( poStatusAdmin->tclAdmin.bIsRegInProgress()
                       &&
                       AHL_C_U16_CMDCOUNT_UPREGMSG == u16CmdCounter )
                  {
                     poStatusAdmin->tclAdmin.vRegFailed();
                  }
                  else if ( poStatusAdmin->tclAdmin.bIsUnRegInProgress()
                            &&
                            AHL_C_U16_CMDCOUNT_RELUPREGMSG == u16CmdCounter )
                  {
                     poStatusAdmin->tclAdmin.vUnRegFailed();
                  }
                  ETG_TRACE_SYS_MIN(("%x ahl_bStatusMessagePreProcessing: unknown errorcode %x", ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID) , u16ErrorCode ));
               }
            }
         }

         vOnError(pServiceDataMessage->u16GetFunctionID(), pServiceDataMessage);
      }
      break;
      case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:
         vOnMethodResult(pServiceDataMessage->u16GetFunctionID(), pServiceDataMessage);
      break;
      case AMT_C_U8_CCAMSG_OPCODE_ABORTRESULT:
         vOnAbortResult(pServiceDataMessage->u16GetFunctionID(), pServiceDataMessage);
      break;
      case AMT_C_U8_CCAMSG_OPCODE_METHODRESULTFIRST:
         vOnMethodResultFirst(pServiceDataMessage->u16GetFunctionID(), pServiceDataMessage);
      break;
      case AMT_C_U8_CCAMSG_OPCODE_METHODRESULTMIDDLE:
         vOnMethodResultMiddle(pServiceDataMessage->u16GetFunctionID(), pServiceDataMessage);
      break;
      case AMT_C_U8_CCAMSG_OPCODE_METHODRESULTLAST:
         vOnMethodResultLast(pServiceDataMessage->u16GetFunctionID(), pServiceDataMessage);
      break;
      default:
      {
         vOnUnknownMessage(pServiceDataMessage);
         /* +++ unkown opcode +++ */
         ETG_TRACE_ERRMEM(("%x ahl_bStatusMessagePreProcessing: unknown message opcode %d", 
                          ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID) , 
                          ETG_ENUM(ail_u8OperationCode, (tU8)pServiceDataMessage->u8GetOpCode()) ));
      }
      break;
   }
}

ahl_sAdminHelper* ahl_tclBaseOneThreadClientHandler::sFindAdminHelper(tU32 nFunctionId)
{
   tU32 i;
   for (i=0; i<_nNumberOfFUIDs; ++i)
   {
       if (_asAdminHelpers[i].u16FID == nFunctionId)
       {
          return &_asAdminHelpers[i];
       }
   }
   return NULL;
}

/* this function is a helper to filter trace from the specific app, clienthandler, service */
tBool ahl_tclBaseOneThreadClientHandler::bTraceClassFilter(tU16 u16Level) const
{
   return _poMainAppl->bTraceClassFilter(u16Level);
}

/*************************************************************************
*
* FUNCTION:tVoid vOnUnknownMessage(amt_tclBaseMessage* poMessage)
* 
* DESCRIPTION: handle unkown message
*
* PARAMETER:  unkown message
*
* RETURNVALUE: tVoid
*
*************************************************************************/
tVoid ahl_tclBaseOneThreadClientHandler::vOnUnknownMessage( amt_tclBaseMessage* poMessage )
{
  /* +++
   the frame work calls this function when it receives a message with a unknown 
   FID or a message which is not handled in a client. You can use it for error handling.
  +++ */

   if(poMessage->u8GetType() == AMT_C_U8_CCAMSGTYPE_SVCDATA)
   {
      amt_tclServiceData oServiceData(poMessage);
      ETG_TRACE_ERR(("%x vOnUnknownMessage: SVDData from App=%x OpCode=%d SRV=%x FID=%d", 
                 ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), 
                 ETG_ENUM(ail_u16AppId, (tU16)u16GetServerAppID()), 
                 ETG_ENUM(ail_u8OperationCode, (tU8)oServiceData.u8GetOpCode()), 
                 ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), 
                 oServiceData.u16GetFunctionID() ));
   }
   else
   {
      ETG_TRACE_ERR(("%x vOnUnknownMessage: SVDData from App=%x Type=%d", 
                 ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), 
                 ETG_ENUM(ail_u16AppId, (tU16)u16GetServerAppID()), 
                 poMessage->u8GetType() ));
   }

  if (!poMessage->bDelete())
  {
     /* +++ delete failed enter error handling here +++ */  
     ETG_TRACE_ERRMEM(("%x vOnUnknownMessage: deleted of Message failed", ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID)  ));
  }
}

tVoid ahl_tclBaseOneThreadClientHandler::vOnServiceAvailable()
{
   ETG_TRACE_USR1(("%x vOnServiceAvailable: from App %x for SRV %x", 
                   ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), 
                   ETG_ENUM(ail_u16AppId, (tU16)_u16ServerAppID),
                   ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID) ));
}

tVoid ahl_tclBaseOneThreadClientHandler::vOnServiceUnavailable()
{
   ETG_TRACE_USR1(("%x vOnServiceUnavailable: from App %x for SRV %x", 
                   ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), 
                   ETG_ENUM(ail_u16AppId, (tU16)_u16ServerAppID),
                   ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID) ));
}

tBool ahl_tclBaseOneThreadClientHandler::bIfServiceAvailable() const
{
   if ((_bServiceAvailable != FALSE) && (bCheckIds() != FALSE))
   {
      return TRUE;
   }
   else
   {
      return FALSE;
   }
}

tVoid ahl_tclBaseOneThreadClientHandler::vRemoveRegistration()
{
  ETG_TRACE_USR1(("%x with SubID %x vRemoveRegistration: from App %x for service %x with register-ID %x", 
                  ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), 
                  _u16OwnSubID,
                  ETG_ENUM(ail_u16AppId, (tU16)_u16ServerAppID),
                  ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID),
                  _u16RegID ));

  vAutoResetConnection();

  if (TRUE == _bServiceAvailable)
  {
    _bServiceAvailable = FALSE;

    vOnServiceUnavailable();
  }

  if (_bDedicatedServer == FALSE)
  {
    _u16ServerAppID = AMT_C_U16_APPID_INVALID;
  }

  _u16RegID = AMT_C_U16_REGID_INVALID;
}

tVoid ahl_tclBaseOneThreadClientHandler::vTraceInternalState(TR_tenTraceLevel enLevel) const
{
   // tracing use ail trace class because of using callback from ail framework
   ail_vTraceMsg(enLevel, "     Clienthandler for Service 0x%04x ", u16GetServiceID());
   ail_vTraceMsg(enLevel, "       ask for Version %d.%d.%d ", _u16ServiceMajorVersion, _u16ServiceMinorVersion, 0);
   ail_vTraceMsg(enLevel, "       ServerApp 0x%04x (dedicated = %d) , RegId %d, SubId %d", _u16ServerAppID, _bDedicatedServer, _u16RegID, _u16OwnSubID);
   ail_vTraceMsg(enLevel, "       Service is available %d ", _bServiceAvailable);
   ail_vTraceMsg(enLevel, "       Number of FIDs %d ", _nNumberOfFUIDs);

   tU32 i;
   for (i=0; i<_nNumberOfFUIDs; ++i)
   {
      ail_vTraceMsg(enLevel, "         FID %d is ready %d", _asAdminHelpers[i].u16FID, _asAdminHelpers[i].tclAdmin.bIsReadyForWork());
   }
}

tVoid ahl_tclBaseOneThreadClientHandler::vInitServiceData(amt_tclServiceData& rServiceDataMessage, tU16 u16CmdCounter, tU16 u16FunctionId, tU8 u8Opcode) const
{
   rServiceDataMessage.vInitServiceData(
                 _u16OwnAppID,                         // Source AppID
                 _u16ServerAppID,                      // Target AppID
                 AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,    // StreamType
                 0,                                    // StreamCounter
                 _u16RegID,                            // RegisterID
                 u16CmdCounter,                        // CmdCounter,
                 _u16ServiceID,                        // ServiceID,
                 u16FunctionId,                        // Function ID
                 u8Opcode                              // Opcode STATUS 
                   );

   rServiceDataMessage.vSetSourceSubID(_u16OwnSubID);
}

tBool ahl_tclBaseOneThreadClientHandler::bIsTargetFor(tU16 u16ServiceId, tU16 u16AppId, tU16 /* u16RegisterId */, tU16 u16AppSubId ) const
{
   if (((_u16ServiceID   == u16ServiceId)  || (u16ServiceId    == AMT_C_U16_SERVICE_UNDEF)) &&
       ((_u16ServerAppID == u16AppId)      || (u16AppId        == AMT_C_U16_APPID_INVALID) || (_u16ServerAppID == AMT_C_U16_APPID_INVALID)) &&
       ((_u16OwnSubID    == u16AppSubId)   || (u16AppSubId     == AMT_C_U16_SUBID_INVALID))   )
   {
      return TRUE;
   }

   return FALSE;
}

tVoid ahl_tclBaseOneThreadClientHandler::vOnAsyncRegisterConf(tU16 u16RegisterId, tU16 u16ServerAppId, tU16 u16ServiceId, tU16 u16TargetSubId)
{
  ETG_TRACE_USR1(("Application %x SubID 0x%04x vOnAsyncRegisterConf(): Register-ID = 0x%04x, u16ServerAppId = %x, u16ServiceId = %x", 
                  ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID),
                  u16TargetSubId, 
                  u16RegisterId,
                  ETG_ENUM(ail_u16AppId, (tU16)u16ServerAppId),
                  ETG_ENUM(ail_u16ServiceId, (tU16)u16ServiceId)));

  // Intentionally do nothing. 
  
  // This method should stay until the introduction phase of the new interface method 
  // vOnAsyncRegisterConfExt() has ended and all of the CCX user applications have  
  // changed from vOnAsyncRegisterConf() as failure indication method to the new and 
  // prefered method vOnAsyncRegisterConfFailure().
}

/*******************************************************************************
*
* FUNCTION: ahl_tclBaseOneThreadClientHandler::vOnAsyncRegisterConfExt()
*
* DESCRIPTION: This method is called by the CCA-Framework to confirm a service 
*              registration request which was initiated by a call of the 
*              CCA-Framework method bRegisterAsync().
*
*              If the service registration has failed the method 
*              vOnAsyncRegisterConfFailure() is called to indicate the kind of 
*              failure to the user application.
*
* PARAMETER: [IN] u16RegisterId = Assigned register ID.
*            [IN] u16ServerAppId = Server application ID.
*            [IN] u16ServiceId = Service ID.
*            [IN] u16TargetSubId = Target Sub ID (Sub ID of this client handler).
*            [IN] u8AsyncRegisterConfStatus = Status of the registration 
*                                             confirmation which can be one of :
*
*                 - AMT_C_U8_REGCONF_SUCCESS
*                   = Registration of the service was successfull.
*                 - AMT_C_U8_REGCONF_SERVER_DOES_NOT_RESPOND_TIMEOUT
*                   = Registration failed because the server doesn't respond 
*                     at all (timeout).
*                 - AMT_C_U8_REGCONF_SERVICE_VERSION_NOT_SUPPORTED
*                   = Registration failed because the server doesn't support the 
*                     requested service version.
*                 - AMT_C_U8_REGCONF_SERVICE_ALREADY_REGISTERED
*                   = Registration failed because service is already registered.
*                 - AMT_C_U8_REGCONF_SERVICE_DOES_NOT_EXIST
*                   = Registration failed because service doesn't exist.
*                 - AMT_C_U8_REGCONF_SERVICE_REGISTRATION_PENDING
*                   = Registration failed because service was already requested 
*                     and the request is still pending.
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid ahl_tclBaseOneThreadClientHandler::vOnAsyncRegisterConfExt(tU16 u16RegisterId, tU16 u16ServerAppId, tU16 u16ServiceId, tU16 u16TargetSubId, tU8 u8AsyncRegisterConfStatus)
{
  tBool bRegistrationSuccessful = TRUE;

  switch (u8AsyncRegisterConfStatus)
  {
    /* --------------------------------------------------------------------- */

    case AMT_C_U8_REGCONF_SUCCESS :
    {
      if (_u16ServerAppID == AMT_C_U16_APPID_INVALID)
      {
        // None dedicated response => Update internal server application ID with the passed value.
        _u16ServerAppID = u16ServerAppId;
      }

      if (_u16RegID != AMT_C_U16_REGID_INVALID)
      {
         // Warning => Already assigned register ID is overwritten
         ETG_TRACE_SYS_MIN(("Application %x SubID 0x%04x  vOnAsyncRegisterConfExt(): RegisterID 0x%04x overwrites old RegisterID 0x%04x of ServiceID %x", 
                            ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID),
                            u16TargetSubId ,
                            u16RegisterId,
                            _u16RegID, 
                            ETG_ENUM(ail_u16ServiceId, (tU16)u16ServiceId) ));
      }

     break;
    }

    /* --------------------------------------------------------------------- */

    case AMT_C_U8_REGCONF_SERVICE_ALREADY_REGISTERED :
    {
      ETG_TRACE_ERRMEM(("Application %x SubID 0x%04x vOnAsyncRegisterConfExt(): Registration for service %x at application %x failed because this service is already registered", 
                       ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), 
                       u16TargetSubId,
                       ETG_ENUM(ail_u16ServiceId, (tU16)u16ServiceId), 
                       ETG_ENUM(ail_u16AppId, (tU16)u16ServerAppId)));

      bRegistrationSuccessful = FALSE;

      break;
    }

    /* --------------------------------------------------------------------- */

    case AMT_C_U8_REGCONF_SERVICE_DOES_NOT_EXIST :
    {
      ETG_TRACE_SYS_MIN(("Application %x SubID 0x%04x vOnAsyncRegisterConfExt(): Registration for service %x failed because there is no application which offers this service", 
                       ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), 
                       u16TargetSubId,
                       ETG_ENUM(ail_u16ServiceId, (tU16)u16ServiceId)));

      bRegistrationSuccessful = FALSE;

      break;
    }

    /* --------------------------------------------------------------------- */

    case AMT_C_U8_REGCONF_SERVICE_VERSION_NOT_SUPPORTED :
    {
      ETG_TRACE_SYS_MIN(("Application %x SubID 0x%04x vOnAsyncRegisterConfExt(): Registration for service %x failed because application %x doesn't offer the requested service version", 
                       ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), 
                       u16TargetSubId,
                       ETG_ENUM(ail_u16ServiceId, (tU16)u16ServiceId), 
                       ETG_ENUM(ail_u16AppId, (tU16)u16ServerAppId)));

      bRegistrationSuccessful = FALSE;

      break;
    }

    /* --------------------------------------------------------------------- */

    case AMT_C_U8_REGCONF_SERVER_DOES_NOT_RESPOND_TIMEOUT :
    {
      ETG_TRACE_FATAL(("Application %x SubID 0x%04x vOnAsyncRegisterConfExt(): Registration for service %x failed because application %x doesn't respond at all (timeout)", 
                       ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), 
                       u16TargetSubId,
                       ETG_ENUM(ail_u16ServiceId, (tU16)u16ServiceId), 
                       ETG_ENUM(ail_u16AppId, (tU16)u16ServerAppId)));

      bRegistrationSuccessful = FALSE;

      break;
    }

    /* --------------------------------------------------------------------- */

    case AMT_C_U8_REGCONF_SERVICE_REGISTRATION_PENDING :
    {
      ETG_TRACE_FATAL(("Application %x SubID 0x%04x vOnAsyncRegisterConfExt(): Registration for service %x failed because a registration for this service is already pending",
                       ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), 
                       u16TargetSubId,
                       ETG_ENUM(ail_u16ServiceId, (tU16)u16ServiceId)));

      bRegistrationSuccessful = FALSE;

      break;
    }

    /* --------------------------------------------------------------------- */

    default :
    {
      ETG_TRACE_ERRMEM(("Application %x SubID 0x%04x vOnAsyncRegisterConfExt(): Parameter u8AsyncRegisterConfStatus = %u is invalid", 
                        ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID) , 
                        u16TargetSubId,
                        u8AsyncRegisterConfStatus ));

      bRegistrationSuccessful = FALSE;
    }

    /* --------------------------------------------------------------------- */
  }

  // Use assigned register ID if registration was successful.
  if (TRUE == bRegistrationSuccessful)
  {
    ETG_TRACE_USR1(("Application %x SubID 0x%04x vOnAsyncRegisterConfExt(): Successfully registered for service %x at Application %x. Applied register-ID = 0x%04x", 
                    ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID),
                    u16TargetSubId, 
                    ETG_ENUM(ail_u16ServiceId, (tU16)u16ServiceId),
                    ETG_ENUM(ail_u16AppId, (tU16)u16ServerAppId),
                    u16RegisterId));

    _u16RegID = u16RegisterId;
  }

  vOnAsyncRegisterConfStatus(u16RegisterId ,u16ServerAppId, u16ServiceId, u16TargetSubId, u8AsyncRegisterConfStatus);
}


/*******************************************************************************
*
* FUNCTION: ahl_tclBaseOneThreadClientHandler::vOnAsyncRegisterConfStatus()
*
* DESCRIPTION: This method is called by the CCX client-handler when the 
*              registration request for a service has failed.
*
*              The method should be overwritten by the user application to be 
*              able to react to a failed service registration as per the 
*              application individual needs.
*
* PARAMETER: [IN] u16ServerAppId = Server application ID.
*            [IN] u16ServiceId = Service ID.
*            [IN] u16TargetSubId = Target Sub ID (Sub ID of this client handler).
*            [IN] u8AsyncRegisterConfStatus = Status of the registration 
*                                             confirmation which can be one of :
*
*                 - AIL_C_U8_REGCONF_SERVER_DOES_NOT_RESPOND_TIMEOUT
*                   = Registration failed because the server doesn't respond 
*                     at all (timeout).
*                 - AIL_C_U8_REGCONF_SERVICE_VERSION_NOT_SUPPORTED
*                   = Registration failed because the server doesn't support the 
*                     requested service version.
*                 - AIL_C_U8_REGCONF_SERVICE_ALREADY_REGISTERED
*                   = Registration failed because service is already registered.
*                 - AIL_C_U8_REGCONF_SERVICE_DOES_NOT_EXIST
*                   = Registration failed because service doesn't exist.
*                 - AIL_C_U8_REGCONF_SERVICE_ID_DOES_NOT_MATCH
*                   = Requested and confirmed service ID does not match.
*                 - AIL_C_U8_REGCONF_APPLICATION_ID_DOES_NOT_MATCH
*                    Requested and confirmed application ID does not match.
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid ahl_tclBaseOneThreadClientHandler::vOnAsyncRegisterConfStatus(tU16 u16RegisterId ,tU16 u16ServerAppId, tU16 u16ServiceId, tU16 u16TargetSubId, tU8 u8AsyncRegisterConfStatus)
{
  ETG_TRACE_USR1(("Application 0x%04x SubID 0x%04x : vOnAsyncRegisterConfStatus(): u8AsyncRegisterConfStatus = %u with register-ID = 0x%04x for service = %u of application = %u",
                  ETG_ENUM(ail_u16AppId, (tU16)_u16OwnAppID), 
                  u16TargetSubId,
                  ETG_ENUM(ail_u8AsyncRegisterConfStatus, (tU8)u8AsyncRegisterConfStatus),
                  u16RegisterId,
                  ETG_ENUM(ail_u16ServiceId, (tU16)u16ServiceId), 
                  ETG_ENUM(ail_u16AppId, (tU16)u16ServerAppId)));
}

