#define _AHL_BASEONETHREADAPP_CPP
/************************************************************************
* FILE:          ahl_BaseOneThreadApp.cpp (base_main_cpp)
* PROJECT:       all projects 
* SW-COMPONENT:  ahl
*----------------------------------------------------------------------
*
* DESCRIPTION: 
*              
*----------------------------------------------------------------------
*
* AUTHOR:  ESE-Neumann  (jn82hi)
*          TMS-Kempen   (kmr2hi)
*          TMS-Jentsch  (kmr2hi)
*
* COPYRIGHT:    (c) 2007 Robert Bosch GmbH, Hildesheim
*
* HISTORY:      
*************************************************************************/

/* --- 
read "---" comments to know what the code does
--- */

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ETRACE_S_IMPORT_INTERFACE_ERRMEM
#include "etrace_if.h"

//-----------------------------------------------------------------------------
// includes
//-----------------------------------------------------------------------------
#include "ahl_BaseOneThreadApp.h"
#include "ahl_ccaExtensionTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_AHL_CCA_EXTENSION
#include "trcGenProj/Header/ahl_BaseOneThreadApp.cpp.trc.h"
#endif

/*************************************************************************
*
* FUNCTION: ahl_tclBaseOneThreadApp::ahl_tclBaseOneThreadApp()
*
* DESCRIPTION:
*
*  Constructor: - Initialize AMT, create Sem, create notification table
*
*
* PARAMETER: void
*
* RETURNVALUE: none
*
*************************************************************************/
ahl_tclBaseOneThreadApp::ahl_tclBaseOneThreadApp(tU16 nAppID):
    _bAppClosed(FALSE),
	_bBreakLoop(FALSE),
    _bQueueCallbackInstalled(FALSE),
    _hEventHandle(OSAL_C_INVALID_HANDLE),
    _nNumberOfObjects(0),
    _pfTraceInput(NULL),
    _tclLoadPair(AMT_C_U32_STATE_INVALID, AMT_C_U32_STATE_OFF),
    _tclSavePair(AMT_C_U32_STATE_INVALID, AMT_C_U32_STATE_OFF)
{
   u16AppId = nAppID;

   /* --- initialize the AMT/CCA framework for this component --- */
   NORMAL_M_ASSERT(amt_bInit() == TRUE);

   // set all to zero
   _asObjects.reserve(AHL_STARTUP_NUMBER_OF_OBJECTS_LIST_CAPACITY);
   OSAL_pvMemorySet(_anTimerId, 0, sizeof(_anTimerId)); //lint !e534
   OSAL_pvMemorySet(_abTimerCreated, FALSE, sizeof(_abTimerCreated));//lint !e534
   OSAL_pvMemorySet(_abTimerStarted, FALSE, sizeof(_abTimerStarted));//lint !e534

   NORMAL_M_ASSERT(_tclLoadPair.bSetDefault(AHL_PAIRLIST_DEFAULT_LOAD_SETTINGS_FILE) == TRUE);
   NORMAL_M_ASSERT(_tclSavePair.bSetDefault(AHL_PAIRLIST_DEFAULT_SAVE_SETTINGS_FILE) == TRUE);
}

tVoid ahl_tclBaseOneThreadApp::vAddObject(ahl_tclBaseOneThreadObject* ptclBaseObject)
{
#ifdef DEBUG
    tU32 i;
    for (i=0; i<_nNumberOfObjects; ++i)
    {
        if (_asObjects[i] == ptclBaseObject)
        {
            // client still added
            ETG_TRACE_ERRMEM(("%x vAddObject: Object still added", ETG_ENUM(ail_u16AppId, (tU16)u16AppId) ));
        }
    }
#endif
    if (ptclBaseObject != NULL)
    {
        _asObjects.push_back(ptclBaseObject);
        ++_nNumberOfObjects;
    }
    else
    {
       ETG_TRACE_ERRMEM(("%x vAddObject: class try to add NULL object pointer", ETG_ENUM(ail_u16AppId, (tU16)u16AppId) ));
    }
}

tVoid ahl_tclBaseOneThreadApp::vRemoveObject(const ahl_tclBaseOneThreadObject* ptclBaseObject)
{
    tBool fFound = FALSE;
    tU32 i;
    for (i=0; i<_nNumberOfObjects; ++i)
    {
        if (_asObjects[i] == ptclBaseObject)
        {
            fFound = TRUE;
            --_nNumberOfObjects;
            _asObjects[i] = _asObjects[_nNumberOfObjects];
            _asObjects[_nNumberOfObjects] = NULL;
            _asObjects.pop_back();
        }
    }

    if (fFound == FALSE)
    {
        ETG_TRACE_ERRMEM(("%x vRemoveObject: Object which should be removed isn't found", ETG_ENUM(ail_u16AppId, (tU16)u16AppId) ));
    }
}


/*************************************************************************
*
* FUNCTION: ahl_tclBaseOneThreadApp::~ahl_tclBaseOneThreadApp()
*
* DESCRIPTION: destructor, frees all ressources
*             ( if not done by vOnApplicationClose before )
*
* PARAMETER: void
*
* RETURNVALUE: none
*
*************************************************************************/
ahl_tclBaseOneThreadApp::~ahl_tclBaseOneThreadApp()
{
   tU32 i;
   for (i=0;i<AHL_MAX_TIMERS;++i)
   {
      if (_abTimerCreated[i] == TRUE)
      {
      // Stop any running timers automatically
         (tVoid)bStopTimer(_anTimerId[i]);
      }
   }
  
   /* ---
   Normally the SPM calls vOnApplicationClose before this application - object is
   destroyed. Here we check if the ressources are already freed by vOnApplicationClose else
   we free the ressources
   --- */
}

/*************************************************************************
*
* FUNCTION: tBool ahl_tclBaseOneThreadApp::sFindObject()
*
* DESCRIPTION: looks for a special object
*
* PARAMETER: service ID
*
* RETURNVALUE: TRUE or FALSE
*
*************************************************************************/
ahl_tclBaseOneThreadObject* ahl_tclBaseOneThreadApp::sFindObject(tU16 u16SearchServiceId, tU16 u16SearchAppId, tU16 u16SearchRegisterId, tU16 u16SearchAppSubId) const
{
    tU32 i;
    for (i=0; i<_nNumberOfObjects; ++i)
    {
        if ( _asObjects[i]->bIsTargetFor(u16SearchServiceId, u16SearchAppId, u16SearchRegisterId, u16SearchAppSubId) == TRUE)
        {
            return _asObjects[i];
        }
    }

    // No object for this service found 
    ETG_TRACE_FATAL(("%x sFindObject: Handler for incoming message for service %04x could not be found (%d Objects)", ETG_ENUM(ail_u16AppId, (tU16)u16AppId), ETG_ENUM(ail_u16ServiceId, (tU16) u16SearchServiceId), _nNumberOfObjects ));
    ETG_TRACE_FATAL((" Check your Application: vAddObject and vRemoveObject should only be called in constructor/destructor or bOnit/vOnApplicationClose" ));
    
    return NULL;
}

/*************************************************************************
*
* FUNCTION: tVoid ahl_tclBaseOneThreadApp::vOnServiceState
*
* DESCRIPTION: I think the service of one of the servers to which I registered as client has changed
*
* PARAMETER:
*
* RETURNVALUE:
*
*************************************************************************/
tVoid ahl_tclBaseOneThreadApp::vOnServiceState( tU16 u16ServiceId, tU16 u16ServerId, tU16 u16RegisterId, tU8 u8ServiceState, tU16 u16SubId)
{
   // called for single or multiple clienthandlers

    /* ~~~ --- check which service has changed --- ~~~ */
    ahl_tclBaseOneThreadObject* psObject = sFindObject(u16ServiceId, u16ServerId, u16RegisterId, u16SubId);
    if (psObject != NULL)
    {
       /* ~~~ --- analyze new service state with the corresponding Handler --- ~~~ */
       psObject->vOnServiceState(u16ServiceId, u16ServerId, u16RegisterId, u8ServiceState, u16SubId);
    }
    else
    {
        /* --- My Client is not supposed to be NULL --- */
        ETG_TRACE_ERRMEM(("%x vOnServiceState: Handler for service %x could not be found", ETG_ENUM(ail_u16AppId, (tU16)u16AppId), ETG_ENUM(ail_u16ServiceId, (tU16) u16ServiceId) ));
    }

     /* --- call the corresponding method of the super class to give AIL the chance to do the  rest --- */
    ail_tclAppInterfaceRestricted::vOnServiceState(u16ServiceId, u16ServerId, u16RegisterId, u8ServiceState, u16SubId);
}

/*************************************************************************
*
* FUNCTION: tVoid ahl_tclBaseOneThreadApp::vOnNewAppState (tU32 u32OldAppState, tU32 u32AppState)
*
* DESCRIPTION: * is call from SPM to inform this application about new power states
*              * changes the state of the application
*              * sends ServiceAvailability state to all notified Objects
*              * changes state of downloadservice(-object)
*
* PARAMETER: tU32 u32OldAppState (actual state),
*            tU32 u32AppState (state to change to)
*
* RETURNVALUE: void
*
**************************************************************************/
tVoid ahl_tclBaseOneThreadApp::vOnNewAppState (tU32 u32OldAppState, tU32 u32AppState)
{
   tU32 i;

   ETG_TRACE_USR1(("%x vOnNewAppState: %d -> %d ", 
                   ETG_ENUM(ail_u16AppId, (tU16)u16AppId), 
                   ETG_ENUM(ail_u32CCAState, (tU32)u32OldAppState), 
                   ETG_ENUM(ail_u32CCAState, (tU32)u32AppState) ));

   // check if application should load here settings
   if ( _tclLoadPair.bIfPairIncluded(u32OldAppState, u32AppState) == TRUE)
   {
       vOnLoadSettings();
   }
   /* --- inform the all object handler about the new state --- */
   
    for (i=0; i<_nNumberOfObjects; ++i)
    {
        if (_asObjects[i] != NULL)
        {
            _asObjects[i]->vOnNewAppState(u32OldAppState, u32AppState);
        }
        else
        {
          //My object is not supposed to be NULL
          ETG_TRACE_ERRMEM(("%x vOnNewAppState: Handler pointer is zero ... !? ", 
                           ETG_ENUM(ail_u16AppId, (tU16)u16AppId) ));
        }
    }

    // check if application should save here settings
    if ( _tclSavePair.bIfPairIncluded(u32OldAppState, u32AppState) == TRUE)
    {
        vOnSaveSettings();
    }
   
   vOnNewAppStateNotify(u32OldAppState, u32AppState);

   /* --- create and send an acknowledge message so the SPM knows you have reached the new state --- */
   vAppStateChanged (u32AppState, 0);
}


/*************************************************************************
*
* FUNCTION: tVoid ahl_tclBaseOneThreadApp::vOnNewSWBlockState(tU32 u32newSWBlockID)
*
* DESCRIPTION: * is call from SPM client to inform this application about new SW Blocks
*              * sends new Blocks state to all notified clients
*
* PARAMETER: tU32 u32newSWBlockID (actual state),
*
* RETURNVALUE: void
*
**************************************************************************/

tVoid ahl_tclBaseOneThreadApp::vOnNewSWBlockState(tU32 /* u32newSWBlockID */)
{
}


/*************************************************************************
*
* FUNCTION: tVoid ahl_tclBaseOneThreadApp::vOnNewMessage (amt_tclBaseMessage* poMessage)
*
* DESCRIPTION: * casts Message to amt_tclServiceData (if legal)
*              * gives amt_tclServiceData - Message to service for analyzing
*              * ignores all unexpected messsages
*
* PARAMETER: amt_tclBaseMessage* poMessage ( Message to analyse)
*
* RETURNVALUE: void
*
*************************************************************************/
tVoid ahl_tclBaseOneThreadApp::vOnNewMessage (amt_tclBaseMessage* poMessage)
{
    /* ---  check if the received message is a AMT Service data message  --- */
    if(poMessage->u8GetType() == AMT_C_U8_CCAMSGTYPE_SVCDATA)
    {
        /* ---
        cast the received message to a AMT Service data message so we
        can access the standard members
        --- */

        amt_tclServiceData oServiceData(poMessage);
        tU16 u16ServiceId = oServiceData.u16GetServiceID();
        tU16 u16TargetSubId = oServiceData.u16GetTargetSubID();

        ETG_TRACE_USR3(("%x vOnNewMessage for Service %x and SubId %d", 
                        ETG_ENUM(ail_u16AppId, (tU16)u16AppId), 
                        ETG_ENUM(ail_u16ServiceId, (tU16) u16ServiceId),
                        u16TargetSubId));

        /* ---
        one server can offer more than one service. So we analyse the ServiceID
        to know which service handler we have to call.
        99% of all Servers only offer one service ( with a different function ID for each feature ).
        --- */

        if ((oServiceData.u16GetSourceAppID() == u16AppId) && (oServiceData.u16GetTargetAppID() == u16AppId))
        {
            // it is a private application callback
            vOnLoopback(u16ServiceId, &oServiceData);
            // only to hold backward compability
            vOnLoopback(u16ServiceId, poMessage);
        }
        else
        {
           ahl_tclBaseOneThreadObject* psObject = sFindObject(u16ServiceId, oServiceData.u16GetSourceAppID(), oServiceData.u16GetRegisterID(), u16TargetSubId);
           if (psObject != NULL)
           {
               /* ~~~ --- analyze new service state with the corresponding Object handler --- ~~~ */
               psObject->vMyDispatchMessage(&oServiceData);
           }
           else
           {
               // My handler is not supposed to be NULL
               ETG_TRACE_ERRMEM(("%x vOnNewMessage: Pointer to Object Handler is zero ... !? ", ETG_ENUM(ail_u16AppId, (tU16)u16AppId) ));
           }
        }

        /* --- alsways delete incoming message and free the ressources --- */
        if ((oServiceData.bIsValid() == TRUE) && (oServiceData.bDelete() != TRUE))
        {
            ETG_TRACE_SYS_MIN(("%x vOnNewMessage: message could not be deleted ... !? ", ETG_ENUM(ail_u16AppId, (tU16)u16AppId) ));
        }
    }
    else
    {
        /* ---
        we have received a message with an unknown format.
        this is an error so make a trace or assert here
        --- */
        ETG_TRACE_ERRMEM(("%x vOnNewMessage: received a message with an unknown format ... !? ", ETG_ENUM(ail_u16AppId, (tU16)u16AppId) ));

        /* --- alsways delete incoming message and free the ressources --- */
        if ((poMessage->bIsValid() == TRUE) && (poMessage->bDelete() != TRUE))
        {
            ETG_TRACE_SYS_MIN(("%x vOnNewMessage: message could not be deleted ... !? ", ETG_ENUM(ail_u16AppId, (tU16)u16AppId) ));
        }
    }
}

/*************************************************************************
*
* FUNCTION: tVoid ahl_tclBaseOneThreadApp::vOnApplicationClose()
*
* DESCRIPTION: releases all resources and sends close - message
*
* PARAMETER: void
*
* RETURNVALUE: void
*
*************************************************************************/
tVoid ahl_tclBaseOneThreadApp::vOnApplicationClose()
{
   tU32 i;
   for (i=0;i<AHL_MAX_TIMERS;++i)
   {
      if (_abTimerCreated[i] == TRUE)
      {
         // Stop any running timers automatically
         (tVoid)bStopTimer(_anTimerId[i]);
         ETG_TRACE_SYS_MIN(("%x vOnApplicationClose: your app haven't stop your timer %d ... !? ", 
                            ETG_ENUM(ail_u16AppId, (tU16)u16AppId), _anTimerId[i] ));
      }
   }

   _bAppClosed = TRUE;

   /* --- inform SPM about successful shut down --- */
   vApplicationCloseAcknowledge(AIL_C_U8_APP_END_SUCCESSFUL);
}

/******************************************************FunctionHeaderBegin******
 * CLASS       : ahl_tclBaseOneThreadApp
 * FUNCTION    : vPlugTraceIn
 * CREATED     : 2005-07-26
 * AUTHOR      :
 * DESCRIPTION :   -
 * SYNTAX      : vPlugTraceIn()
 * ARGUMENTS   :
 * RETURN VALUE:
 *               none
 * NOTES       :  What would you associate with this name?
 * Of course it does nothing, but plugging in the trace
 *******************************************************FunctionHeaderEnd******/
tVoid ahl_tclBaseOneThreadApp::vPlugTraceIn(TR_tenTraceChan enTraceChannel, OSAL_tpfCallback vTraceInput)
{
    OSAL_trIOCtrlLaunchChannel  MyTraceChannel;
    OSAL_tIODescriptor MyTraceHandle;
    tS32 s32Err;

    MyTraceHandle = OSAL_IOOpen(OSAL_C_STRING_DEVICE_TRACE, OSAL_EN_READWRITE);

    if (MyTraceHandle != OSAL_ERROR)
    {
       MyTraceChannel.enTraceChannel = enTraceChannel;
       MyTraceChannel.pCallback = vTraceInput;

       _pfTraceInput = MyTraceChannel.pCallback; // remember callback for vUnplugTrace()
   
       s32Err = OSAL_s32IOControl (MyTraceHandle, OSAL_C_S32_IOCTRL_CALLBACK_REG, (intptr_t) &MyTraceChannel);
       if (s32Err != OSAL_OK)
       {
           ETG_TRACE_ERRMEM(("%x vPlugTraceIn: could not plugin trace ... (%s) !? ", 
                            ETG_ENUM(ail_u16AppId, (tU16)u16AppId), OSAL_coszErrorText(OSAL_u32ErrorCode()) ));
       }
   
       s32Err = OSAL_s32IOClose(MyTraceHandle);
   
       if (s32Err != OSAL_OK)
       {
           ETG_TRACE_ERRMEM(("%x vPlugTraceIn: could not close trace ... (%s) !? ", 
                            ETG_ENUM(ail_u16AppId, (tU16)u16AppId), OSAL_coszErrorText(OSAL_u32ErrorCode()) ));
       }
    }
    else
    {
       ETG_TRACE_ERRMEM(("%x vPlugTraceIn: could not open trace ... (%s) !? ", 
                        ETG_ENUM(ail_u16AppId, (tU16)u16AppId), OSAL_coszErrorText(OSAL_u32ErrorCode()) ));
    }
}

/******************************************************FunctionHeaderBegin******
 * CLASS       : ahl_tclBaseOneThreadApp
 * FUNCTION    : vUnplugTrace
 * CREATED     : 2005-07-26
 * AUTHOR      :
 * DESCRIPTION :   -
 * SYNTAX      : vUnplugTrace()
 * ARGUMENTS   :
 * RETURN VALUE:
 *               none
 * NOTES       :  What would you associate with this name?
 * Of course it does nothing, but plugging out the trace
 *******************************************************FunctionHeaderEnd******/
tVoid ahl_tclBaseOneThreadApp::vUnplugTrace(TR_tenTraceChan enTraceChannel)
{
    OSAL_trIOCtrlLaunchChannel  MyTraceChannel;
    OSAL_tIODescriptor MyTraceHandle;
    tS32 s32Err;

    MyTraceHandle = OSAL_IOOpen(OSAL_C_STRING_DEVICE_TRACE, OSAL_EN_READWRITE);

    if (MyTraceHandle != OSAL_ERROR)
    {
       MyTraceChannel.enTraceChannel = enTraceChannel;
       MyTraceChannel.pCallback = _pfTraceInput;
    
       _pfTraceInput = NULL;

       s32Err = OSAL_s32IOControl (MyTraceHandle, OSAL_C_S32_IOCTRL_CALLBACK_UNREG, (intptr_t) &MyTraceChannel);
       if (s32Err != OSAL_OK)
       {
           ETG_TRACE_ERRMEM(("%x vUnplugTrace: could not Unplug trace ... (%s) !? ", 
                            ETG_ENUM(ail_u16AppId, (tU16)u16AppId), OSAL_coszErrorText(OSAL_u32ErrorCode()) ));
       }
    
       s32Err = OSAL_s32IOClose(MyTraceHandle);
    
       if (s32Err != OSAL_OK)
       {
           ETG_TRACE_ERRMEM(("%x vUnplugTrace: could not close trace ... (%s) !? ", 
                            ETG_ENUM(ail_u16AppId, (tU16)u16AppId), OSAL_coszErrorText(OSAL_u32ErrorCode()) ));
       }
    }
    else
    {
       ETG_TRACE_ERRMEM(("%x vUnplugTrace: could not close trace ... (%s) !? ", 
                        ETG_ENUM(ail_u16AppId, (tU16)u16AppId), OSAL_coszErrorText(OSAL_u32ErrorCode()) ));
    }
}

tBool ahl_tclBaseOneThreadApp::bGetServiceVersion(tU16 u16ServiceID, tU16 &rfu16MajorVersion, tU16 &rfu16MinorVersion, tU16 &rfu16PatchVersion)
{
   // only called for services

    /* ~~~ --- check which service version is necessary --- ~~~ */
    ahl_tclBaseOneThreadObject* psObject = sFindObject(u16ServiceID, AMT_C_U16_APPID_INVALID, AMT_C_U16_REGID_INVALID, AMT_C_U16_SUBID_DEFAULT);
    if (psObject != NULL)
    {
       /* ~~~ --- analyze new service state with the corresponding Clienthandler --- ~~~ */
       return psObject->bGetServiceVersion(u16ServiceID, rfu16MajorVersion, rfu16MinorVersion, rfu16PatchVersion);
    }
    else
    {
        /* --- My Client is not supposed to be NULL --- */
        ETG_TRACE_FATAL(("%x bGetServiceVersion: could not find client for service %x ... !? ", 
                         ETG_ENUM(ail_u16AppId, (tU16)u16AppId), ETG_ENUM(ail_u16ServiceId, (tU16) u16ServiceID) ));
        return FALSE;
    }
}

tBool ahl_tclBaseOneThreadApp::bStartTimer(tU16 nTimerID,  OSAL_tMSecond nTimeUntilFirstTimerTick, OSAL_tMSecond nTimerTickInterval)
{
   intptr_t i;

   if (_hEventHandle == OSAL_C_INVALID_HANDLE)
   {
      ETG_TRACE_SYS_MIN(("%x bStartTimer: Event could potentially not be send because application isn't started yet !", 
                         ETG_ENUM(ail_u16AppId, (tU16)u16AppId) ));
   }

   // first step, try to find timer for reuse. If not found try to create one
   for (i=0;i<AHL_MAX_TIMERS;++i)
   {
      if (_abTimerCreated[i] == TRUE)
      {
         // created timers stands at start of table
         if (_anTimerId[i] == nTimerID)
         {
            // stop searching
            break;
         }
      }
      else
      {
         // first empty place is found (only empty places follow !)
         if (_atclTimer[i].bInit(this, &ahl_tclBaseOneThreadApp::vOnTimerCallback, (tVoid*)i) == TRUE)
         {
            _abTimerCreated[i] = TRUE;
            _anTimerId[i] = nTimerID;
         }
         else
         {
            ETG_TRACE_ERRMEM(("%x bStartTimer: Timer of ID %d could not be created ! ", 
                             ETG_ENUM(ail_u16AppId, (tU16)u16AppId), nTimerID));
            return FALSE;
         }
         break;
      }
   }

   // check if there is a timer left
   if (i < AHL_MAX_TIMERS)
   {
      if (_abTimerStarted[i] == TRUE)
      {
         ETG_TRACE_SYS_MIN(("%x bStartTimer: Timer of ID %d always started ! ", 
                          ETG_ENUM(ail_u16AppId, (tU16)u16AppId), nTimerID));
      }

      if (_atclTimer[i].bStart(nTimeUntilFirstTimerTick, nTimerTickInterval) == TRUE)
      {
         return TRUE;
      }
      else
      {
         ETG_TRACE_ERRMEM(("%x bStartTimer: Timer of ID %d could not be started ! ", 
                          ETG_ENUM(ail_u16AppId, (tU16)u16AppId), nTimerID));
      }
   }
   else
   {
      ETG_TRACE_ERRMEM(("%x bStartTimer: max number of Timers reached, Timer with Id %d could not be started ! ", ETG_ENUM(ail_u16AppId, (tU16)u16AppId), nTimerID));
   }

   return FALSE;
}

tBool ahl_tclBaseOneThreadApp::bStopTimer(tU16 nTimerID)
{
   tU32 i;

   for (i=0;i<AHL_MAX_TIMERS;++i)
   {
      if (_abTimerCreated[i] == TRUE)
      {
         if (_anTimerId[i] == nTimerID)
         {
            if (_abTimerStarted[i] == FALSE)
            {
               ETG_TRACE_SYS_MIN(("%x bStopTimer: Timer of ID %d always stopped ! ", 
                                ETG_ENUM(ail_u16AppId, (tU16)u16AppId), nTimerID));
            }
   
            _abTimerStarted[i] = FALSE;
   
            if (_atclTimer[i].bStop() == TRUE)
            {
               return TRUE;
            }
            else
            {
               ETG_TRACE_ERRMEM(("%x bStopTimer: Timer of ID %d could not be stopped !", 
                                 ETG_ENUM(ail_u16AppId, (tU16)u16AppId), nTimerID));
            }
         }
      }
   }

   ETG_TRACE_SYS_MIN(("%x bStopTimer: Timer of ID %d is not created but should be stopped !", 
                      ETG_ENUM(ail_u16AppId, (tU16)u16AppId), nTimerID));

   return FALSE;
}

tVoid ahl_tclBaseOneThreadApp::vAppEntry()
{
  //## begin ail_tclAppInterfaceRestricted::vAppEntry%3A8D1E71014A.body preserve=yes
   tBool                   bContinueDispatching = TRUE;
   amt_tclBaseMessage      oMsgObject;
   tU32                    u32Prio;
   tChar                   szName[OSAL_C_U32_MAX_NAMELENGTH]="\0";

// registration at LPM-Proxy -> CCA-Message: POWER, Typ: Register
//   
// initialization -> initalize all your local threads
//...
// confirm initialize ready -> CCA-Message: POWER, Typ: State, Data: Initialized
//...
// while ...
// -> only OSAL-messages can be received, 
// -> first activity starts after receiving CCA-Message: POWER, Typ: SetState
      
      OSAL_tEventMask resultMask;

      if (!_bQueueCallbackInstalled)
      {
         // create eventhandle for main thread loop
         OSAL_s32PrintFormat((tString)szName,"ev_%i",u16AppId);//lint !e534
         if (OSAL_s32EventCreate(szName, &_hEventHandle) == OSAL_OK)
         {
            ETG_TRACE_COMP(("%x create Event handle %04x name %s ", 
                            ETG_ENUM(ail_u16AppId, (tU16)u16AppId), _hEventHandle, szName ));
         }
         else
         {
            ETG_TRACE_ERRMEM(("%x could not create event %32s (error: %s)", 
                             ETG_ENUM(ail_u16AppId, (tU16)u16AppId), szName, OSAL_coszErrorText(OSAL_u32ErrorCode()) ));
         }

         #ifdef VARIANT_S_FTR_ENABLE_MESSAGEQUEUE_EVENT_NOTIFICATION
            // use new functionality of posix in osal
            trMqEventInf rEvent;
            rEvent.coszName = szName;
            rEvent.mask     = CCA_EVENT;
            rEvent.enFlags  = OSAL_EN_EVENTMASK_OR;
   
            if (OSAL_s32MessageQueueNotify(hMyInQueue,(OSAL_tpfCallback)0xffffffff,(tPVoid)&rEvent) == OSAL_OK)
            {
               ETG_TRACE_COMP(("%x Install of message queue event successfully", ETG_ENUM(ail_u16AppId, (tU16)u16AppId) ));
            }
            else
            {
               ETG_TRACE_ERRMEM(("%x could not install message queue event (error: %s)", 
                                ETG_ENUM(ail_u16AppId, (tU16)u16AppId), OSAL_coszErrorText(OSAL_u32ErrorCode()) ));
            }
         #else
            // install callback for incoming messages
            if (OSAL_s32MessageQueueNotify(hMyInQueue, (OSAL_tpfCallback)vOnQueueCallback, this) == OSAL_OK)
            {
               ETG_TRACE_COMP(("%x Install of message queue callback successfully", ETG_ENUM(ail_u16AppId, (tU16)u16AppId) ));
            }
            else
            {
               ETG_TRACE_ERRMEM(("%x could not install message queue callback (error: %s)", 
                                ETG_ENUM(ail_u16AppId, (tU16)u16AppId), OSAL_coszErrorText(OSAL_u32ErrorCode()) ));
            }
         #endif

         _bQueueCallbackInstalled = TRUE;
      }

      ETG_TRACE_SYS_MIN(("Application %x: Hello Extension-world!", ETG_ENUM(ail_u16AppId, (tU16)u16AppId) ));

    //  try
      {
         // The CCA startup message AMT_C_U16_PWR_APP_START_REQ is only send from the ENTRY-thread for 
         // one-thread applications. For two-thread applications the CCA startup message is send from the
         // BODY-thread which is implemented by the superior AIL object where the ahl_BaseOneThreadApp 
         // object is derived from.
         #ifdef _AHL_BASEONETHREADAPP_CPP
           if ( u16LpmId != AMT_C_U16_APPID_INVALID )
           {
              if (bSendCCAPowerMsg(u16LpmId, hGetLpmInQueue(), OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST, AMT_C_U16_PWR_APP_START_REQ, (tU32) u16AppId, (tU32) 0x00) == FALSE)
              {
                 ETG_TRACE_ERRMEM(("%x send of start request message to SPM failed", ETG_ENUM(ail_u16AppId, (tU16)u16AppId) ));
              }
           }
           else
           {
              ETG_TRACE_ERRMEM(("%x could not send start request message to SPM because invalid SPM AppId", ETG_ENUM(ail_u16AppId, (tU16)u16AppId) ));
           }
         #endif

         /* application is ready to receive a command from the supervisor */
         enInterfaceState = AIL_EN_N_APPLICATION_REGISTERED;

         if ( poInternalDispatch != NULL )
         {
            tS32 s32WaitResult;
            // mainloop of thread
            while ( bContinueDispatching )
            {
               // prevent to get the same events after timeout
               resultMask = 0;
               // wait for message, timer or event
               s32WaitResult = OSAL_s32EventWait(_hEventHandle, CCA_EVENT | TIMER_EVENTS | OTHER_EVENTS, OSAL_EN_EVENTMASK_OR, AHL_WAITTIME_CCA_EVENTS, &resultMask);

               if (s32WaitResult != OSAL_OK)
               {
                  if (OSAL_u32ErrorCode() != OSAL_E_TIMEOUT)
                  {
                     ETG_TRACE_ERRMEM(("%x vAppEntry OSAL_s32EventWait returns error for handle %08x (%s)", ETG_ENUM(ail_u16AppId, (tU16)u16AppId), (unsigned int)_hEventHandle, OSAL_coszErrorText(OSAL_u32ErrorCode()) ));
                  }
               }

               if (OSAL_s32EventPost(_hEventHandle, ~resultMask, OSAL_EN_EVENTMASK_AND) != OSAL_OK )
               {
                  ETG_TRACE_ERRMEM(("%x vAppEntry could not clear event for handle %08x (error: %s)", 
                                   ETG_ENUM(ail_u16AppId, (tU16)u16AppId), (unsigned int)_hEventHandle, OSAL_coszErrorText(OSAL_u32ErrorCode()) ));
               }

               // process timer
               if  (resultMask & TIMER_EVENTS)
               {
                  tU32 i;
                  for (i=0;i<AHL_MAX_TIMERS;++i)
                  {
                     if (resultMask & (TIMER_EVENT_0 >> i))
                     {
                        vOnTimer(_anTimerId[i]);
                     }
                  }
               }

               // process events
               if (resultMask & OTHER_EVENTS)
               {
                  vOnEvent(resultMask & OTHER_EVENTS);
               }

               #ifndef AHL_CHECK_FOR_CCA_MESSAGES_ON_EACH_EVENT
               if (resultMask & CCA_EVENT)
               #endif
               {
                  tU32                    bMessageAvailable = TRUE;
                  // process CCA messages

                  // process all CCA messages
                  while (bContinueDispatching && bMessageAvailable)
                  {
                     // get messages without wait
                     bMessageAvailable = ail_bIpcMessageWait(u16AppId, hMyInQueue, &oMsgObject, &u32Prio, OSAL_C_TIMEOUT_NOBLOCKING);
                     if (bMessageAvailable)
                     {
                        #ifdef _AHL_BASEONETHREADAPP_CPP
                          bContinueDispatching = poInternalDispatch->bDispatchCCAMessages(&oMsgObject, OSAL_C_INVALID_HANDLE, FALSE, u32Prio);
                        #elif defined _AHL_BASETWOTHREADAPP_CPP
                          bContinueDispatching = poInternalDispatch->bDispatchCCAMessages(&oMsgObject, hIntermediateInQueue, FALSE, u32Prio);
                        #else
                          #error "Undefined identifier _AHL_BASEONETHREADAPP_CPP or _AHL_BASETWOTHREADAPP_CPP"
                        #endif
                     }
                     else
                     {
                        if (OSAL_u32ErrorCode() != OSAL_E_TIMEOUT)
                        {
                           ETG_TRACE_ERRMEM(("%x ail_bIpcMessageWait returns error for handle %08x an message (%s)", 
                                            ETG_ENUM(ail_u16AppId, (tU16)u16AppId), (unsigned int)hMyInQueue, OSAL_coszErrorText(OSAL_u32ErrorCode()) ));
                        }
                     }
					 
					 if (_bBreakLoop == TRUE && bMessageAvailable == TRUE)
					 {
						// give an event a chance, but set event again to process other mesasages
						vSendEvent(CCA_EVENT);
						_bBreakLoop = FALSE;
						break;
					 }
                  }
               }
            }
         }
         else
         {
            ETG_TRACE_ERR(("Application %x: poInternalDispatch==NULL", ETG_ENUM(ail_u16AppId, (tU16)u16AppId) ));
         }
      }
   /*   catch (...)
      {
         ail_vTraceMsg( TR_LEVEL_ERROR, "Application %x: unhandled exception occured in Entry-Thread", u16AppId );
         vOnUnhandledAilException();
      }
      
   */

      if (_bQueueCallbackInstalled == TRUE)
      {
        #ifdef VARIANT_S_FTR_ENABLE_MESSAGEQUEUE_EVENT_NOTIFICATION
           if (OSAL_s32MessageQueueNotify(hMyInQueue,(OSAL_tpfCallback)0xffffffff,NULL) == OSAL_OK)
           {
              ETG_TRACE_COMP(("%x Remove of  message queue event successfully", ETG_ENUM(ail_u16AppId, (tU16)u16AppId) ));
           }
           else
           {
              ETG_TRACE_ERRMEM(("%x could not remove message queue event (error: %s)", 
                               ETG_ENUM(ail_u16AppId, (tU16)u16AppId), OSAL_coszErrorText(OSAL_u32ErrorCode()) ));
           }
        #else
           // remove callback for incoming messages if installed
           if (OSAL_s32MessageQueueNotify(hMyInQueue, NULL, NULL) == OSAL_OK)
           {
              ETG_TRACE_COMP(("%x Remove of message queue callback successfully", ETG_ENUM(ail_u16AppId, (tU16)u16AppId) ));
           }
           else
           {
              ETG_TRACE_ERRMEM(("%x could not remove message queue callback (error: %s)", 
                               ETG_ENUM(ail_u16AppId, (tU16)u16AppId), OSAL_coszErrorText(OSAL_u32ErrorCode()) ));
           }
        #endif
      }
   
   if (_hEventHandle != OSAL_C_INVALID_HANDLE)
   {
     if (OSAL_s32EventClose(_hEventHandle) == OSAL_ERROR)
     {
       ETG_TRACE_ERRMEM(("ahl_tclBaseOneThreadApp::vAppEntry() failed: Call of OSAL_s32EventClose(%s) returned with OSAL_ERROR.", szName));
     }
     else
     {
       _hEventHandle = OSAL_C_INVALID_HANDLE;

       if (OSAL_s32EventDelete(szName) == OSAL_ERROR)
       {
         ETG_TRACE_ERRMEM(("ahl_tclBaseOneThreadApp::vAppEntry() failed: Call of OSAL_s32EventDelete(%s) returned with OSAL_ERROR.", szName));
       }
     }
   }

   ETG_TRACE_SYS_MIN(("Application %x: is gone", ETG_ENUM(ail_u16AppId, (tU16)u16AppId) ));

#ifdef _AHL_BASEONETHREADAPP_CPP
   /* release all basic class objects && inform LPX of correct shutdown */
   vTerminateMySelf ();
#elif defined _AHL_BASETWOTHREADAPP_CPP
   // give a trigger to body-thread that your job is done
   vEntryEndNotification();
#else
  #error "Undefined identifier _AHL_BASEONETHREADAPP_CPP or _AHL_BASETWOTHREADAPP_CPP"
#endif

  //## end ail_tclAppInterfaceRestricted::vAppEntry%3A8D1E71014A.body
}

tVoid ahl_tclBaseOneThreadApp::vOnTimerCallback(tVoid* pParam)
{
   uintptr_t nTimerId = (uintptr_t) pParam;
   if (nTimerId < AHL_MAX_TIMERS)
   {
      vSendEvent(TIMER_EVENT_0 >> nTimerId);
	  _bBreakLoop = TRUE;
   }
   else
   {
      ETG_TRACE_ERRMEM(("%x ahl_tclBaseOneThreadApp::vOnTimerCallback Unknown Timer Id %d (max %d)", ETG_ENUM(ail_u16AppId, (tU16)u16AppId), (int)nTimerId, AHL_MAX_TIMERS ));
   }
}

tVoid ahl_tclBaseOneThreadApp::vOnQueueCallback(tVoid* pArg)
{
   ahl_tclBaseOneThreadApp* p = (ahl_tclBaseOneThreadApp*)pArg;
   p->vSendEvent(CCA_EVENT);
}

/* This function is a historical one and should not used anymore */
tVoid ahl_tclBaseOneThreadApp::vSendEvent(tU32 nEvent)
{
   if(OSAL_OK != OSAL_s32EventPost(_hEventHandle, nEvent, OSAL_EN_EVENTMASK_OR))
   {
      /* occurs mostly if the timer is started in the constructor, but the application isnt started */
      ETG_TRACE_ERRMEM(("%x vSendEvent could not post event 0x%08x for handle 0x%08x (error: %s)", 
                       ETG_ENUM(ail_u16AppId, (tU16)u16AppId), nEvent, (unsigned int)_hEventHandle, OSAL_coszErrorText(OSAL_u32ErrorCode()) ));
   }
}

tBool ahl_tclBaseOneThreadApp::bPostEvent(tU32 nEvent)
{
   if ((nEvent & PROHIBITED_EVENTS) == 0)
   {
      if(OSAL_OK == OSAL_s32EventPost(_hEventHandle, nEvent, OSAL_EN_EVENTMASK_OR))
      {
		 _bBreakLoop = TRUE;
         return TRUE;
      }
      else
      {
         /* occurs mostly if the timer is started in the constructor, but the application isnt started */
         ETG_TRACE_ERRMEM(("%x bPostEvent could not post event 0x%08x for handle 0x%08x (error: %s)", 
                          ETG_ENUM(ail_u16AppId, (tU16)u16AppId), nEvent, (unsigned int)_hEventHandle, OSAL_coszErrorText(OSAL_u32ErrorCode()) ));
         return FALSE;
      }
   }
   else
   {
      /* some of the events are used internal for timers */
      ETG_TRACE_ERRMEM(("%x bPostEvent you try to use projibited (0x%08x are internal used) events 0x%08x", 
                       ETG_ENUM(ail_u16AppId, (tU16)u16AppId), PROHIBITED_EVENTS, nEvent));
      return FALSE;
   }
}


tVoid ahl_tclBaseOneThreadApp::vOnUnregister(tU16 u16ServiceId, tU16 u16RegisterId)
{
   // only called for services

   ahl_tclBaseOneThreadObject* psObject = sFindObject(u16ServiceId, AMT_C_U16_APPID_INVALID, u16RegisterId, AMT_C_U16_SUBID_DEFAULT);
   if (psObject != NULL)
   {
      /* ~~~ --- analyze new service state with the corresponding Clienthandler --- ~~~ */
      psObject->vOnUnregister(u16ServiceId, u16RegisterId);
   }
   else
   {
       /* --- My Client is not supposed to be NULL --- */
       ETG_TRACE_ERRMEM(("%x ahl_tclBaseOneThreadApp::vOnUnregister no service handler for this service (srvId = %04x, RegId = %04x)", 
                        ETG_ENUM(ail_u16AppId, (tU16)u16AppId), ETG_ENUM(ail_u16ServiceId, (tU16) u16ServiceId), u16RegisterId));
   }
}

tBool ahl_tclBaseOneThreadApp::bOnAcceptNewRegister (tU16 u16ServiceId, tU16 u16ClientAppId, tU16 u16ClientSubId, tU16* pu16RegisterId)
{
   // only called for services

   ahl_tclBaseOneThreadObject* psObject = sFindObject(u16ServiceId, u16ClientAppId, AMT_C_U16_REGID_INVALID, u16ClientSubId);
   if (psObject != NULL)
   {
      /* ~~~ --- analyze new service state with the corresponding handler --- ~~~ */
      if (psObject->bOnAcceptNewRegister (u16ServiceId, u16ClientAppId, u16ClientSubId, pu16RegisterId) != FALSE)
      {
         return ail_tclOneThreadAppInterface::bOnAcceptNewRegister(u16ServiceId, u16ClientAppId, u16ClientSubId, pu16RegisterId);
      }
      else
      {
         return FALSE;
      }
   }
   else
   {
       /* --- My object is not supposed to be NULL --- */
       ETG_TRACE_FATAL(("%x ahl_tclBaseOneThreadApp::bOnAcceptNewRegister no service handler for this service (srvId = %04x, ClientAppId = %04x)", 
                        ETG_ENUM(ail_u16AppId, (tU16)u16AppId), ETG_ENUM(ail_u16ServiceId, (tU16) u16ServiceId), ETG_ENUM(ail_u16AppId, (tU16)u16ClientAppId) ));
       return FALSE;
   }
}

tBool ahl_tclBaseOneThreadApp::bTraceClassFilter(tU16 u16Level) const
{
   return et_bIsTraceActive(TR_CLASS_AHL_CCA_EXTENSION, (tU16)u16Level);
}

tVoid ahl_tclBaseOneThreadApp::vOnLoadSettings()
{
   ETG_TRACE_USR1(("%x ahl_tclBaseOneThreadApp::vOnLoadSettings called ", ETG_ENUM(ail_u16AppId, (tU16)u16AppId) ));
}

tVoid ahl_tclBaseOneThreadApp::vOnSaveSettings()
{
   ETG_TRACE_USR1(("%x ahl_tclBaseOneThreadApp::vOnSaveSettings called ", ETG_ENUM(ail_u16AppId, (tU16)u16AppId) ));
}

tVoid ahl_tclBaseOneThreadApp::vOnLoopback(tU16 u16ServiceID, amt_tclBaseMessage* poMessage)
{
   // called for services and single or multiple clienthandlers

   amt_tclServiceData oServiceData(poMessage);
   tU16 u16TargetSubId = oServiceData.u16GetTargetSubID();

   ahl_tclBaseOneThreadObject* psObject = sFindObject(u16ServiceID, AMT_C_U16_APPID_INVALID, oServiceData.u16GetRegisterID(), u16TargetSubId);
   if (psObject != NULL)
   {
      ETG_TRACE_USR1(("%x vOnLoopback forward messages to object for service %x and SubId %d", 
                         ETG_ENUM(ail_u16AppId, (tU16)u16AppId), ETG_ENUM(ail_u16ServiceId, (tU16) u16ServiceID), u16TargetSubId));

      /* ~~~ --- analyze loopback message with the corresponding object handler --- ~~~ */
      psObject->vOnLoopback(u16ServiceID, &oServiceData);
   }
   else
   {
      ETG_TRACE_SYS_MIN(("%x ahl_tclBaseOneThreadApp::vOnLoopback unhandled messages for service %x and SubId = %d", 
                      ETG_ENUM(ail_u16AppId, (tU16)u16AppId), ETG_ENUM(ail_u16ServiceId, (tU16) u16ServiceID), u16TargetSubId));
   }
}

tVoid ahl_tclBaseOneThreadApp::vOnLoopback(tU16 u16ServiceID, amt_tclServiceData* poMessage)
{
   // called for services and single or multiple clienthandlers

   tU16 u16TargetSubId   = poMessage->u16GetTargetSubID();
   tU16 u16GetRegisterID = poMessage->u16GetRegisterID();

   ahl_tclBaseOneThreadObject* psObject = sFindObject(u16ServiceID, AMT_C_U16_APPID_INVALID, u16GetRegisterID, u16TargetSubId);
   if (psObject != NULL)
   {
      ETG_TRACE_USR1(("%x vOnLoopback forward messages to object for service %x and SubId %d", 
                         ETG_ENUM(ail_u16AppId, (tU16)u16AppId), ETG_ENUM(ail_u16ServiceId, (tU16) u16ServiceID), u16TargetSubId ));

      /* ~~~ --- analyze loopback message with the corresponding object handler --- ~~~ */
      psObject->vOnLoopback(u16ServiceID, poMessage);
   }
   else
   {
      ETG_TRACE_SYS_MIN(("%x vOnLoopback unhandled messages for service %x and SubId %d", 
                         ETG_ENUM(ail_u16AppId, (tU16)u16AppId), ETG_ENUM(ail_u16ServiceId, (tU16) u16ServiceID), u16TargetSubId ));
   }
}

tVoid ahl_tclBaseOneThreadApp::vOnTimer(tU16 nTimerId)
{
   ETG_TRACE_SYS_MIN(("%x ahl_tclBaseOneThreadApp::vOnTimer unhandled timer message for Timer ID %d", 
                      ETG_ENUM(ail_u16AppId, (tU16)u16AppId), nTimerId ));
}

tVoid ahl_tclBaseOneThreadApp::vOnEvent(tU32 nEvent)
{
   ETG_TRACE_SYS_MIN(("%x ahl_tclBaseOneThreadApp::vOnEvent unhandled event message %08x", 
                      ETG_ENUM(ail_u16AppId, (tU16)u16AppId), nEvent ));
}

tVoid ahl_tclBaseOneThreadApp::vOnRegistrationRemoved(tU16 u16ServiceId, tU16 u16ServerAppId, tU16 u16SourceSubId)
{
  ETG_TRACE_COMP(("%x ahl_tclBaseOneThreadApp::vOnRegistrationRemoved() => ServiceId = %x, ServerAppId = %x, SourceSubId = 0x%04X", 
                  ETG_ENUM(ail_u16AppId, (tU16)u16AppId), 
                  ETG_ENUM(ail_u16ServiceId, (tU16)u16ServiceId), 
                  ETG_ENUM(ail_u16AppId, (tU16)u16ServerAppId), 
                  u16SourceSubId));

  ahl_tclBaseOneThreadObject* psObject = sFindObject(u16ServiceId, u16ServerAppId, AMT_C_U16_REGID_INVALID, u16SourceSubId);

  if (psObject != NULL)
  {
    psObject->vRemoveRegistration();
  }
  else
  {
    ETG_TRACE_ERRMEM(("%x vOnRegistrationRemoved: Handler for Service %x, ServerApp %x and SourceSubId = 0x%04X could not be found", 
                      ETG_ENUM(ail_u16AppId, (tU16)u16AppId), 
                      ETG_ENUM(ail_u16ServiceId, (tU16) u16ServiceId),
                      ETG_ENUM(ail_u16AppId, (tU16)u16ServerAppId), 
                      u16SourceSubId));
  }
}

tVoid ahl_tclBaseOneThreadApp::vTraceInternalState(TR_tenTraceLevel enLevel)
{
   #ifdef AIL_DEBUG_APPLICATION
   // tracing use ail trace class because of using callback (virtual function) from ail framework
   ail_tclOneThreadAppInterface::vTraceInternalState(enLevel);
   #endif

   ail_vTraceMsg(enLevel, " #######  Additional Info of CCX Application ####### ");
   ail_vTraceMsg(enLevel, "   %d added Objects (Clienthandler and Services) max possible %d", _nNumberOfObjects, _asObjects.capacity());

   tU32 i;
   for (i=0; i<_nNumberOfObjects; ++i)
   {
      if (_asObjects[i] != NULL)
      {
         ail_vTraceMsg(enLevel, "   Object %d is", i);
         _asObjects[i]->vTraceInternalState(enLevel);
      }
      else
      {
         ail_vTraceMsg(enLevel, "   Object %d is NULL", i);
      }
   }

   for (i=0; i<AHL_MAX_TIMERS; ++i)
   {
      if (_abTimerCreated[i] == TRUE)
      {
         ail_vTraceMsg(enLevel, "   Timer %d with Id %d is running ", i, _anTimerId[i]);
      }
   }
}

tVoid ahl_tclBaseOneThreadApp::vOnAsyncRegisterConf(tU16 u16RegisterId, tU16 u16ServerAppId, tU16 u16ServiceId, tU16 u16TargetSubId)
{
   // called for single or multiple clienthandlers

   // Don't lookup the register-ID with the below call of sFindObject() as it will be set with the 
   // following call of vOnAsyncRegisterConf() and should not be used for a lookup before being set.
   ahl_tclBaseOneThreadObject* psObject = sFindObject(u16ServiceId, u16ServerAppId, AMT_C_U16_REGID_INVALID, u16TargetSubId);
   if (psObject != NULL)
   {
      /* ~~~ --- analyze new service state with the corresponding Clienthandler --- ~~~ */
      psObject->vOnAsyncRegisterConf(u16RegisterId, u16ServerAppId, u16ServiceId, u16TargetSubId);
   }
   else
   {
      /* --- My Client is not supposed to be NULL --- */
      ETG_TRACE_ERRMEM(("%x vOnAsyncRegisterConf no client for this service %x", ETG_ENUM(ail_u16AppId, (tU16)u16AppId), ETG_ENUM(ail_u16ServiceId, (tU16) u16ServiceId) ));
   }
}

tVoid ahl_tclBaseOneThreadApp::vOnAsyncRegisterConfExt(tU16 u16RegisterId, tU16 u16ServerAppId, tU16 u16ServiceId, tU16 u16TargetSubId, tU8 u8AsyncRegisterConfStatus)
{
   // called for single or multiple clienthandlers

   // Don't lookup the register-ID with the below call of sFindObject() as it will be set with the 
   // following call of vOnAsyncRegisterConf() and should not be used for a lookup before being set.
   ahl_tclBaseOneThreadObject* psObject = sFindObject(u16ServiceId, u16ServerAppId, AMT_C_U16_REGID_INVALID, u16TargetSubId);
   if (psObject != NULL)
   {
      /* ~~~ --- analyze new service state with the corresponding Clienthandler --- ~~~ */
      psObject->vOnAsyncRegisterConfExt(u16RegisterId, u16ServerAppId, u16ServiceId, u16TargetSubId, u8AsyncRegisterConfStatus);
   }
   else
   {
      /* --- My Client is not supposed to be NULL --- */
      ETG_TRACE_ERRMEM(("%x vOnAsyncRegisterConfExt no client for this service %x", ETG_ENUM(ail_u16AppId, (tU16)u16AppId), ETG_ENUM(ail_u16ServiceId, (tU16) u16ServiceId) ));
   }
}

#ifndef _AHL_BASEONETHREADAPP_CPP
  #error "Undefined identifier _AHL_BASEONETHREADAPP_CPP"
#endif

#undef _AHL_BASEONETHREADAPP_CPP
