/************************************************************************

  FILE:           utf_mcnet.cpp

  SW-COMPONENT:   UTFUtil

  DESCRIPTION:    UTF utility functions for mapping
                  MCNetM(odified) <--> UTF-8 byte sequences

  AUTHOR:         CM-DI/ESN-Saake

  COPYRIGHT:      (c) 2003 Blaupunkt GmbH

  HISTORY:
  $Log:   //hi230124/projects/Vasco/swnavi/archives/components/utf/utf_mcnet.cpp-arc  $
 *
 *    Rev 1.0   Sep 22 2003 09:15:26   OSR2HI
 * Initial revision.

************************************************************************/


// System
#define SYSTEM_S_IMPORT_INTERFACE_COMPLETE
#include "system_pif.h"

// OSAL

/*
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
*/

// Self include
#include "UTFUtil.h"


// Specifies start of code pair region. Smaller codes are mapped to 1 byte code.
#define UTFUTIL_U8_MCNET2UTFCODEPAIR_START        (tU8)0x80

// Symbol in MCNetM to map undefined UTF-8 symbols to
#define UTFUTIL_U8_MCNET2UTFCODEPAIR_UNDEFINED    (tU8)0x80

#define UTFUTIL_U8_START_LINEAR_CONVERSION        (tU8)0x20
#define UTFUTIL_U8_END_LINEAR_CONVERSION          (tU8)0x7F

#define UTFUTIL_U8_MCNET_EURO_SYMBOL              (tU8)0xA9

#define UTFUTIL_U8_MCNET_DOLLAR_SYMBOL            (tU8)0xAB
#define UTFUTIL_U8_UTF8_DOLLAR_SYMBOL             (tU8)0x24

#define UTFUTIL_U8_MCNET_CURRENCY_SYMBOL          (tU8)0x24

#define UTFUTIL_U8_MCNET_DOUBLE_PIPE_SYMBOL       (tU8)0x60


/*****************************************************************************
 * utfutil_tMCNet2UTFCodePairFirst32Symbols
 *
 * Table for mapping (modified) MCNet to UTF-8. Contains symbols in pre-
 * calculated UTF-8 sequence for the mapping of MCNet region [0x00..0x1F].
 * First symbol represents code value for 0x00... and so on. Last symbol
 * represents code value for 0x1F.
 *****************************************************************************/
struct utfutil_tMCNet2UTFCodePairFirst32Symbols
{
//  tU8  u8MCNETSymbol;
  tU8  u8UTFSymbol1; // first UTF-symbol
  tU8  u8UTFSymbol2; // second UTF-symbol, if defined in u8UTFLength
  tU8  u8UTFLength;  // number of symbols
} rMCNetSymTabFirst32[] =
{
   { 0x00, 0x00, 1 },   { 0xc4, 0x80, 2 },   { 0xc4, 0x84, 2 },   { 0xc4, 0x8e, 2 }, // 0x00 - 0x03
   { 0xc4, 0x92, 2 },   { 0xc4, 0x96, 2 },   { 0xc4, 0x98, 2 },   { 0x07, 0x00, 1 },
   { 0x08, 0x00, 1 },   { 0x09, 0x00, 1 },   { 0x0a, 0x00, 1 },   { 0x0b, 0x00, 1 }, // 0x08 - 0x0B
   { 0x0c, 0x00, 1 },   { 0x0d, 0x00, 1 },   { 0xc4, 0x9a, 2 },   { 0xc4, 0xa2, 2 },
   { 0xc4, 0xaa, 2 },   { 0xc4, 0xae, 2 },   { 0xc4, 0xb6, 2 },   { 0xc4, 0xb9, 2 }, // 0x10 - 0x13
   { 0xc4, 0xbb, 2 },   { 0xc4, 0xbd, 2 },   { 0xc5, 0x81, 2 },   { 0xc5, 0x83, 2 },
   { 0xc5, 0x85, 2 },   { 0xc5, 0x87, 2 },   { 0x1a, 0x00, 1 },   { 0xc5, 0x90, 2 }, // 0x18 - 0x1B
   { 0xc5, 0xa4, 2 },   { 0xc5, 0xaa, 2 },   { 0xc5, 0xae, 2 },   { 0xc5, 0xb0, 2 }
};

/*****************************************************************************
 * utfutil_tMCNet2UTFCodePair
 *
 * Table for mapping (modified) MCNet to UTF-8. Contains symbols in pre-calculated
 * UTF-8 sequence for the mapping of MCNet region [0x80..0xFF].
 * First symbol represents code value for 0x80... and so on. Last symbol
 * represents code value for 0xFF.
 *****************************************************************************/
struct utfutil_tMCNet2UTFCodePair
{
//  tU8  u8MCNETSymbol;
  tU8  u8UTFSymbol1; // first UTF-symbol
  tU8  u8UTFSymbol2; // second UTF-symbol
} rMCNetSymTab[] =
{
   { 0xc3, 0xa1 },   { 0xc3, 0xa0 },   { 0xc3, 0xa9 },   { 0xc3, 0xa8 }, // 0x80 - 0x83
   { 0xc3, 0xad },   { 0xc3, 0xac },   { 0xc3, 0xb3 },   { 0xc3, 0xb2 },
   { 0xc3, 0xba },   { 0xc3, 0xb9 },   { 0xc3, 0x91 },   { 0xc3, 0x87 }, // 0x88 - 0x8B
   { 0xc5, 0x9e },   { 0xc3, 0x9f },   { 0xc2, 0xa1 },   { 0xc4, 0xb2 },
   { 0xc3, 0xa2 },   { 0xc3, 0xa4 },   { 0xc3, 0xaa },   { 0xc3, 0xab }, // 0x90 - 0x93
   { 0xc3, 0xae },   { 0xc3, 0xaf },   { 0xc3, 0xb4 },   { 0xc3, 0xb6 },
   { 0xc3, 0xbb },   { 0xc3, 0xbc },   { 0xc3, 0xb1 },   { 0xc3, 0xa7 }, // 0x98 - 0x9B
   { 0xc5, 0x9f },   { 0xc4, 0x9f },   { 0xc4, 0x8f },   { 0xc4, 0xb3 },
   { 0xc2, 0xaa },   { 0xc4, 0x9b },   { 0xc2, 0xa9 },   { 0xc5, 0xa5 }, // 0xA0 - 0xA3
   { 0xc4, 0x9e },   { 0xc4, 0x95 },   { 0xc5, 0x88 },   { 0xc5, 0x91 },
   { 0xc5, 0xb2 },   { 0xFF, 0xFF },   { 0xc2, 0xa3 },   { 0xFF, 0xFF }, // 0xA8 - 0xAB // 0xA9: EURO-Zeichen (3 Bytes) [0xE2 82 AC] // 0xAB: Dollar-Zeichen (1 Byte) [0x24]
   { 0xc2, 0xa0 },   { 0xc2, 0xa0 },   { 0xc2, 0xa0 },   { 0xc2, 0xa0 }, // Pfeil-Symbole!?
   { 0xc2, 0xba },   { 0xc2, 0xb9 },   { 0xc2, 0xb2 },   { 0xc2, 0xb3 }, // 0xB0 - 0xB3
   { 0xc2, 0xb1 },   { 0xc4, 0xb0 },   { 0xc5, 0x84 },   { 0xc5, 0xb1 },
   { 0xc5, 0xbb },   { 0xc2, 0xbf },   { 0xc3, 0xb7 },   { 0xc2, 0xb0 }, // 0xB8 - 0xBB
   { 0xc2, 0xbc },   { 0xc2, 0xbd },   { 0xc2, 0xbe },   { 0xc2, 0xa7 },

   { 0xc3, 0x81 },   { 0xc3, 0x80 },   { 0xc3, 0x89 },   { 0xc3, 0x88 }, // 0xC0 - 0xC3
   { 0xc3, 0x8d },   { 0xc3, 0x8c },   { 0xc3, 0x93 },   { 0xc3, 0x92 },
   { 0xc3, 0x9a },   { 0xc3, 0x99 },   { 0xc5, 0x98 },   { 0xc4, 0x8c }, // 0xC8 - 0xCB
   { 0xc5, 0xa0 },   { 0xc5, 0xbd },   { 0xc4, 0x90 },   { 0xc4, 0xbf },
   { 0xc3, 0x82 },   { 0xc3, 0x84 },   { 0xc3, 0x8a },   { 0xc3, 0x8b }, // 0xD0 - 0xD3
   { 0xc3, 0x8e },   { 0xc3, 0x8f },   { 0xc3, 0x94 },   { 0xc3, 0x96 },
   { 0xc3, 0x9b },   { 0xc3, 0x9c },   { 0xc5, 0x99 },   { 0xc4, 0x8d }, // 0xD8 - 0xDB
   { 0xc5, 0xa1 },   { 0xc5, 0xbe },   { 0xc4, 0x91 },   { 0xc5, 0x80 },
   { 0xc3, 0x83 },   { 0xc3, 0x85 },   { 0xc3, 0x86 },   { 0xc5, 0x92 }, // 0xE0 - 0xE3
   { 0xc5, 0xb7 },   { 0xc3, 0x9d },   { 0xc3, 0x95 },   { 0xc3, 0x98 },
   { 0xc3, 0x9e },   { 0xc5, 0xb8 },   { 0xc5, 0x94 },   { 0xc4, 0x86 }, // 0xE8 - 0xEB
   { 0xc5, 0x9a },   { 0xc5, 0xb9 },   { 0xc5, 0xa6 },   { 0xc5, 0xaf },
   { 0xc3, 0xa3 },   { 0xc3, 0xa5 },   { 0xc3, 0xa6 },   { 0xc5, 0x93 }, // 0xF0 - 0xF3
   { 0xc5, 0xb5 },   { 0xc3, 0xbd },   { 0xc3, 0xb5 },   { 0xc3, 0xb8 },
   { 0xc3, 0xbe },   { 0xc3, 0xbf },   { 0xc5, 0x95 },   { 0xc4, 0x87 }, // 0xF8 - 0xFB
   { 0xc5, 0x9b },   { 0xc5, 0xba },   { 0xc5, 0xa7 },   { 0xc2, 0xa0 }  // 0xFF: reserved ???
};

/*****************************************************************************
 * utfutil_tUTF2MCNETValue
 *
 * Table for mapping UTF-8 to (modified) MCNet. Table must be sorted in
 * ascending order by the value "u32UTFSymbol" to be used in BINARY SEARCH in
 * "utfutil_u32ConvertUTF82MCNetMod".
 *****************************************************************************/
struct utfutil_tUTF2MCNetValue
{
  tU8  u8MCNetSymbol;
  tU32 u32UTFSymbol;
} rUtf8ToMCNetSymTab[] =
{                      // Position im Array (nur zur eigenen Orientierung)
  {   0x8E,0x00a1   }, // 0x00
  {   0xAA,0x00a3   },
  {   0x24,0x00a4   },
  {   0xBF,0x00a7   },
  {   0xA2,0x00a9   },
  {   0xA0,0x00aa   },
  {   0xBB,0x00b0   },
  {   0xB4,0x00b1   },
  {   0xB2,0x00b2   }, // 0x08
  {   0xB3,0x00b3   },
  {   0xB1,0x00b9   },
  {   0xB0,0x00ba   },
  {   0xBC,0x00bc   },
  {   0xBD,0x00bd   },
  {   0xBE,0x00be   },
  {   0xB9,0x00bf   },
  {   0xC1,0x00c0   }, // 0x10
  {   0xC0,0x00c1   },
  {   0xD0,0x00c2   },
  {   0xE0,0x00c3   },
  {   0xD1,0x00c4   },
  {   0xE1,0x00c5   },
  {   0xE2,0x00c6   },
  {   0x8B,0x00c7   },
  {   0xC3,0x00c8   }, // 0x18
  {   0xC2,0x00c9   },
  {   0xD2,0x00ca   },
  {   0xD3,0x00cb   },
  {   0xC5,0x00cc   },
  {   0xC4,0x00cd   },
  {   0xD4,0x00ce   },
  {   0xD5,0x00cf   },
  {   0x8A,0x00d1   }, // 0x20
  {   0xC7,0x00d2   },
  {   0xC6,0x00d3   },
  {   0xD6,0x00d4   },
  {   0xE6,0x00d5   },
  {   0xD7,0x00d6   },
  {   0xE7,0x00d8   },
  {   0xC9,0x00d9   },
  {   0xC8,0x00da   }, // 0x28
  {   0xD8,0x00db   },
  {   0xD9,0x00dc   },
  {   0xE5,0x00dd   },
  {   0xE8,0x00de   },
  {   0x8D,0x00df   },
  {   0x81,0x00e0   },
  {   0x80,0x00e1   },
  {   0x90,0x00e2   }, // 0x30
  {   0xF0,0x00e3   },
  {   0x91,0x00e4   },
  {   0xF1,0x00e5   },
  {   0xF2,0x00e6   },
  {   0x9B,0x00e7   },
  {   0x83,0x00e8   },
  {   0x82,0x00e9   },
  {   0x92,0x00ea   }, // 0x38
  {   0x93,0x00eb   },
  {   0x85,0x00ec   },
  {   0x84,0x00ed   },
  {   0x94,0x00ee   },
  {   0x95,0x00ef   },
  {   0x9A,0x00f1   },
  {   0x87,0x00f2   },
  {   0x86,0x00f3   }, // 0x40
  {   0x96,0x00f4   },
  {   0xF6,0x00f5   },
  {   0x97,0x00f6   },
  {   0xBA,0x00f7   },
  {   0xF7,0x00f8   },
  {   0x89,0x00f9   },
  {   0x88,0x00fa   },
  {   0x98,0x00fb   }, // 0x48
  {   0x99,0x00fc   },
  {   0xF5,0x00fd   },
  {   0xF8,0x00fe   },
  {   0xF9,0x00ff   },
  {   0x01,0x0100   },
  {   0x02,0x0104   },
  {   0xEB,0x0106   },
  {   0xFB,0x0107   }, // 0x50
  {   0xCB,0x010c   },
  {   0xDB,0x010d   },
  {   0x03,0x010e   },
  {   0x9E,0x010F   },
  {   0xCE,0x0110   },
  {   0xDE,0x0111   },
  {   0x04,0x0112   },
  {   0xA5,0x0115   }, // 0x58
  {   0x05,0x0116   },
  {   0x06,0x0118   },
  {   0x0E,0x011a   },
  {   0xA1,0x011b   },
  {   0xA4,0x011e   },
  {   0x9D,0x011f   },
  {   0x0F,0x0122   },
  {   0x10,0x012a   }, // 0x60
  {   0x11,0x012e   },
  {   0xB5,0x0130   },
  {   0x8F,0x0132   },
  {   0x9F,0x0133   },
  {   0x12,0x0136   },
  {   0x13,0x0139   },
  {   0x14,0x013b   },
  {   0x15,0x013d   }, // 0x68
  {   0xCF,0x013f   },
  {   0xDF,0x0140   },
  {   0x16,0x0141   },
  {   0x17,0x0143   },
  {   0xB6,0x0144   },
  {   0x18,0x0145   },
  {   0x19,0x0147   },
  {   0xA6,0x0148   }, // 0x70
  {   0x1B,0x0150   },
  {   0xA7,0x0151   },
  {   0xE3,0x0152   },
  {   0xF3,0x0153   },
  {   0xEA,0x0154   },
  {   0xFA,0x0155   },
  {   0xCA,0x0158   },
  {   0xDA,0x0159   }, // 0x78
  {   0xEC,0x015a   },
  {   0xFC,0x015b   },
  {   0x8C,0x015e   },
  {   0x9C,0x015f   },
  {   0xCC,0x0160   },
  {   0xDC,0x0161   },
  {   0x1C,0x0164   },
  {   0xA3,0x0165   }, // 0x80
  {   0xEE,0x0166   },
  {   0xFE,0x0167   },
  {   0x1D,0x016a   },
  {   0x1E,0x016e   },
  {   0xEF,0x016f   },
  {   0x1F,0x0170   },
  {   0xB7,0x0171   },
  {   0xA8,0x0172   }, // 0x88
  {   0xF4,0x0175   },
  {   0xE4,0x0177   },
  {   0xE9,0x0178   },
  {   0xED,0x0179   },
  {   0xFD,0x017a   },
  {   0xB8,0x017b   },
  {   0xCD,0x017d   }, // 0x90
  {   0xDD,0x017e   },
  {   0x60,0x01c1   },
  {   0xA9,0x20ac   }
   // mag zwar in dieser Anordnung etwas unbersichtlich sein, aber so lsst sich die Tabelle am einfachsten nachtrglich korrigieren, da die Sortierung immer beibehalten werden muss
};


/************************************************************************
  FUNCTION:    utfutil_u32ConvertMCNetMod2UTF8
  DESCRIPTION: Converts a sequence of (modified) MCNet characters to a
               corresponding sequence of UTF8-coded characters.
  PARAMETER:
     tChar* pcUTF8Buffer
                 Desti. buffer, allocated by client. If 0 this routine only
                 counts the maximum length which would be needed for
                 UTF-8 conversion (-> I)
     tU32   u32UTF8BufferLen
                 Length of destination buffer (-> I)
     const  tChar *MCNetCodedString
                 Source string in (modified) MCNet (-> I)
     tU32   u32Count (-> I)
                 number of characters to convert
  RETURNVALUE: Number of bytes in UTF-8 sequence
  HISTORY:
  12.09.2003 CM-DI/ESN-Saake
             modified copy 'utfutil_u32ConvertMCNETMod2UTF8' for MCNet conversion
************************************************************************/
tU32 utfutil_u32ConvertMCNetMod2UTF8 (tChar* pcUTF8Buffer,             // Desti. buffer, allocated by client
                                      tU32   u32UTF8BufferLen,         // Length of destination buffer
                                      const  tChar *MCNetCodedString,  // Source string in (modified) MCNet
                                      tU32   u32MCNetMax)              // number of characters to convert
{
   tU32 u32UTFCount=0;
   tU8  caConverted[3] = {0,0,0};         // Array for converted code
   tU32 u32ConversionLength=0;            // Stores lenght of used array
   tU32 u32MCNetCount = 0;
   tBool bContinue = (u32MCNetCount <= u32MCNetMax) ? TRUE : FALSE;

   while (bContinue == TRUE)
      //for(tU32 i=0; i<u32MCNetMax; i++)
   {
      tU8 u8MCNetCode = MCNetCodedString[u32MCNetCount];

      if( (u8MCNetCode >= UTFUTIL_U8_START_LINEAR_CONVERSION) &&
          (u8MCNetCode <= UTFUTIL_U8_END_LINEAR_CONVERSION)    )
      {
        if (u8MCNetCode == UTFUTIL_U8_MCNET_CURRENCY_SYMBOL)
        {
          // [0x24]  --> currency symbol = 2 byte code C2A4
          u32ConversionLength = 2;
          caConverted[0] = (tU8)0xC2;
          caConverted[1] = (tU8)0xA4;
        }
        else
        {
          // Check Regions: {0x20,..,0x7F}\{0x24} --> Standard-ASCII, 1 byte code
          u32ConversionLength = 1;
          caConverted[0] = u8MCNetCode;
        }
      }
      else
      {
         if (u8MCNetCode == UTFUTIL_U8_MCNET_EURO_SYMBOL)
         {
            // [0xA9]  --> Euro symbol = 3 byte code E282AC
            u32ConversionLength = 3;
            caConverted[0] = (tU8)0xE2;
            caConverted[1] = (tU8)0x82;
            caConverted[2] = (tU8)0xAC;
         }
         else if (u8MCNetCode == UTFUTIL_U8_MCNET_DOUBLE_PIPE_SYMBOL)
         {
            // [0x60]  --> double pipe symbol = 3 byte code E280B0
            u32ConversionLength = 3;
            caConverted[0] = (tU8)0xE2;
            caConverted[1] = (tU8)0x80;
            caConverted[2] = (tU8)0xB0;
         }
         else if (u8MCNetCode == UTFUTIL_U8_MCNET_DOLLAR_SYMBOL)
         {
            // [0xAB] --> Dollar symbol = 1 byte code 0x24
            u32ConversionLength = 1;
            caConverted[0] = (tU8)UTFUTIL_U8_UTF8_DOLLAR_SYMBOL;
         }
         else if (u8MCNetCode < UTFUTIL_U8_START_LINEAR_CONVERSION)
         {
            // [0x00..0x1F], 1 OR 2 byte code
           u32ConversionLength = rMCNetSymTabFirst32[u8MCNetCode].u8UTFLength;
           caConverted[0] = (tU8)(rMCNetSymTabFirst32[u8MCNetCode].u8UTFSymbol1);
           if (u32ConversionLength == 2)
           {
             caConverted[1] = (tU8)(rMCNetSymTabFirst32[u8MCNetCode].u8UTFSymbol2);
           }
         }
         else // u8MCNetCode > UTFUTIL_U8_END_LINEAR_CONVERSION
         {
            // 0x80-0xFF \ {0xA4} = 2 Byte Code
            tU8 u8CharIndex = u8MCNetCode - UTFUTIL_U8_MCNET2UTFCODEPAIR_START;
            // if((0 <= u32CharIndex) &&  (u32CharIndex <= (UTFUTIL_U8_MCNET2UTFCODEPAIR_MAX-UTFUTIL_U8_MCNET2UTFCODEPAIR_START+1)))
            if(u8CharIndex < (sizeof(rMCNetSymTab)/sizeof(struct utfutil_tMCNet2UTFCodePair)) )
            {
               u32ConversionLength = 2;
               caConverted[0] = (tU8)(rMCNetSymTab[u8CharIndex].u8UTFSymbol1);
               caConverted[1] = (tU8)(rMCNetSymTab[u8CharIndex].u8UTFSymbol2);
            }
         }
      }

      // Result: Array "caConverted" contains UTF8 conversion;
      //         u32ConversionLength contains length in bytes;

      // Only write if destination buffer present
      if(pcUTF8Buffer != NULL)
      {
         // Take care of destination buffer length
         if(u32UTFCount + u32ConversionLength <= u32UTF8BufferLen)
         {
            /* quick fix: replaced "OSAL_pvMemoryCopy" q&d by following "for"-loop
               OSAL_pvMemoryCopy(&pcUTF8Buffer[u32UTFCount], caConverted, u32ConversionLength);
            */
            for(tU32 i=0; i < u32ConversionLength; i++)
            {
               pcUTF8Buffer[u32UTFCount+i] = caConverted[i];
            }
         }
      }
      u32UTFCount += u32ConversionLength;

      u32MCNetCount++;

      // Check for end condition:
      // MCNetCount < MCNetMax or in case of MCNetMax=0 until 0-byte found
      // in MCNet-string
      if(u32MCNetMax == 0)
         bContinue = (u8MCNetCode != 0) ? TRUE : FALSE;
      else
         bContinue = (u32MCNetCount < u32MCNetMax) ? TRUE : FALSE;
   }
   return (u32UTFCount);
}  // end utfutil_u32ConvertMCNetMod2UTF8


#if 0 // unnoetig, kann durch u8GetStoredContainerLength ersetzt werden!
/************************************************************************
  FUNCTION:    u8GetStoredContainerLengthForMCNetConversion
  DESCRIPTION:
  PARAMETER:
  RETURNVALUE:
  HISTORY:     12.09.2003 CM-DI/ESN-Saake
               Initial Revision.
************************************************************************/
tU8 u8GetStoredContainerLengthForMCNetConversion(tU8 u8UTFByte1)
{
   tU8 u8UTFContainerLength=1;

   // Check for 1-byte character code
   if(u8UTFByte1 >= (tU8)0xC0)
   {
      // Multi-byte code -> Get container length
      u8UTFContainerLength=0;
      while(((tU8)((u8UTFByte1 << (++u8UTFContainerLength)) & (tU8)0x80)))
         ;
   }
   return u8UTFContainerLength;
}
#endif


/************************************************************************
  FUNCTION:    u8DecodeUTFContainer2MCNet
  DESCRIPTION:
  PARAMETER:
  RETURNVALUE:
  HISTORY:     12.09.2003 CM-DI/ESN-Saake
               Initial Revision.
************************************************************************/
tU8 u8DecodeUTFContainer2MCNet(tU8 *u8MCNetChar, const tU8 *pUTFBuffer)
{
   *u8MCNetChar = UTFUTIL_U8_MCNET2UTFCODEPAIR_UNDEFINED;
   tU8 u8UTFContainerLength = 1;

   // Get 1. byte
   tU8 u8UTFByte1 = *pUTFBuffer;

   // verify
//Wn   if((((tU8)0x00 <= u8UTFByte1) && (u8UTFByte1 <= (tU8)0x7F)) ||
//        (((tU8)0xC0 <= u8UTFByte1) && (u8UTFByte1 <= (tU8)0xFD)))
   if( (u8UTFByte1 <= (tU8)0x7F) ||
       ( ((tU8)0xC0 <= u8UTFByte1) && (u8UTFByte1 <= (tU8)0xFD) ) )
   {
      u8UTFContainerLength = u8GetStoredContainerLength(u8UTFByte1);
      if(u8UTFContainerLength == 1)
      {  // 1 byte code
        if (UTFUTIL_U8_UTF8_DOLLAR_SYMBOL == u8UTFByte1) // $ - Symbol [0x24]
        {
          *u8MCNetChar = UTFUTIL_U8_MCNET_DOLLAR_SYMBOL;
        }
        else
        {
          *u8MCNetChar = u8UTFByte1;
        }
      }
      else
      {  // > 1 byte code
         // get data bytes
         tU32 u32ContainerData = 0;

         // First data byte (data part smaller than 6)
         u8UTFByte1 <<= (u8UTFContainerLength+1);
         u8UTFByte1 >>= (u8UTFContainerLength+1);
         u32ContainerData |= u8UTFByte1;

         tU8 u8Index = 0;
         for(u8Index = 1; u8Index <= (u8UTFContainerLength-1); u8Index++)
         {
            tU8 u8DataItem = pUTFBuffer[u8Index];
            // valid data?
            if(((tU8)0x80 <= u8DataItem) && (u8DataItem <= (tU8)0xBF))
            {
               u32ContainerData <<= 6;
               u32ContainerData |= (tU32)((u8DataItem & (tU8)0x3F));
            }
            else
            {
               *u8MCNetChar = UTFUTIL_U8_MCNET2UTFCODEPAIR_UNDEFINED;
               break;
            }
         }
         u8UTFContainerLength = u8Index;

         // Decode
         switch(u8UTFContainerLength)
         {
            case 2:
              {
                /*
                // wird ber rUtf8ToMCNetSymTab[] abgefangen
                if (((tU32)0x00A4 == u32ContainerData))
                {
                  *u8MCNetChar = (tU8)UTFUTIL_U8_MCNET_CURRENCY_SYMBOL; // currency symbol
                }
                else
                */
                {
                  // region from table
                  tU32 u32Max = (sizeof(rUtf8ToMCNetSymTab) /
                    sizeof(struct utfutil_tUTF2MCNetValue));
                  for(tU32 u32TableIndex=0; u32TableIndex < u32Max; u32TableIndex++)
                  {
                    if(u32ContainerData == rUtf8ToMCNetSymTab[u32TableIndex].u32UTFSymbol)
                    {
                      *u8MCNetChar = rUtf8ToMCNetSymTab[u32TableIndex].u8MCNetSymbol;
                      break;
                    }
                  }
                }
                break;
              }
            case 3:
              {
                if(((tU32)0x20AC == u32ContainerData))
                {
                  *u8MCNetChar = (tU8)UTFUTIL_U8_MCNET_EURO_SYMBOL; // Euro symbol
                }
                else if(((tU32)0x2030 == u32ContainerData))
                {
                  *u8MCNetChar = (tU8)UTFUTIL_U8_MCNET_DOUBLE_PIPE_SYMBOL; // double pipe symbol
                }
                else {}
                break;
              }
            default:
                     *u8MCNetChar = UTFUTIL_U8_MCNET2UTFCODEPAIR_UNDEFINED;
         }
      }
   }
   return (u8UTFContainerLength);
} // end u8DecodeUTFContainer2MCNet


/************************************************************************
  FUNCTION:    utfutil_u32ConvertUTF82MCNetMod
  DESCRIPTION: Converts a sequence of UTF8-coded characters to a
               corresponding sequence of (modified) MCNet characters.
  PARAMETER:
     tChar* pcMCNetBuffer
               Desti. buffer, allocated by client. If 0 this routine only
               counts the maximum length which would be needed for
               UTF-8 conversion (->I).
     tU32   u32MCNetBufferLen
               Length of destination buffer. If a buffer length of "0"
               is given the conversion did not write to memory and just
               counts the resulting bytes in MCNet character (-> I).
     const  tChar *UTF8CodedString
               Source string in UTF-8 coding (-> I).
     tU32   u32UTFMax (->I).
               Number of characters to convert. A number of "0" converts
               until a 0-Byte termination is found (in first UTF-byte) in
               source string.

     Remark: Invalid UTF-Streams are transformed according to UTF-guidelines

  RETURNVALUE:   Number of bytes in MCNet sequence
  HISTORY:
  15.09.2003 CM-DI/ESN-Saake
             modified copy 'utfutil_u32ConvertUTF82MCNETMod' for MCNet conversion
************************************************************************/

tU32 utfutil_u32ConvertUTF82MCNetMod (tChar* pcMCNetBuffer,          // Desti. buffer, allocated by client
                                      tU32   u32MCNetBufferLen,      // Length of destination buffer
                                      const  tChar *UTF8CodedString, // Source string in UTF-8
                                      tU32   u32UTFMax)              // number of UTF-8 characters to convert
{
   tU32 u32UTFCount = 0;
   tU32 u32MCNetCount = 0;

   tBool bContinue = (u32UTFCount <= u32UTFMax) ? TRUE : FALSE;

   while(bContinue == TRUE) //  && (u32MCNetCount < u32MCNetBufferLen))
   {
      tU8 u8UTFByte1           = UTF8CodedString[u32UTFCount];
      tU8 u8UTFContainerLength = 1;
      tU8 u8MCNetCode          = UTFUTIL_U8_MCNET2UTFCODEPAIR_UNDEFINED;

      // Find first valid char (skip invalid ones)
//Wn      if((((tU8)0x00 <= u8UTFByte1) && (u8UTFByte1 <= (tU8)0x7F)) ||
//          (((tU8)0xC0 <= u8UTFByte1) && (u8UTFByte1 <= (tU8)0xFD)))
      if( (u8UTFByte1 <= (tU8)0x7F) ||
          ( ((tU8)0xC0 <= u8UTFByte1) && (u8UTFByte1 <= (tU8)0xFD) ) )
      {
         u8UTFContainerLength = u8DecodeUTFContainer2MCNet(&u8MCNetCode, (const tU8*)&UTF8CodedString[u32UTFCount]);

         // Write to target buffer only if "bufferlen > 0".
         // (condition always false if "u32MCNetBufferLen==0")

         if(u32MCNetCount < u32MCNetBufferLen)
         {
            pcMCNetBuffer[u32MCNetCount] = u8MCNetCode;
         }
         u32MCNetCount++;
      }
      u32UTFCount += (tU32)u8UTFContainerLength;

      // Check for end condition:
      // u32UTFCount < u32UTFMax or in case of UTFMax=0 until first
      // UTF-Byte == 0 found in UTF-string
      if(0 == u32UTFMax)
      {
         bContinue = (0 != u8UTFByte1) ? TRUE : FALSE;
      }
      else
      {
         bContinue = (u32UTFCount < u32UTFMax) ? TRUE : FALSE;
      }
   }

   return (u32MCNetCount);
}
