/*****************************************************************************
 * FILE:          UTFUtil.h
 * SW-COMPONENT:  UTFUtil
 *----------------------------------------------------------------------------
 * DESCRIPTION:   UTDF utility functions for mapping 
 *                ISO8859-15M(odified) <--> UTF-8 byte sequences
 *----------------------------------------------------------------------------
 * AUTHOR:        CM-DI/ESA2-Bruns
 * COPYRIGHT:     (c) 2003 Blaupunkt Werke GmbH
 * REMARK:
 *
 * Actually 4 functions are provided by this module.
 * + "utfutil_u32ConvertISOMod2UTF8": for mapping ISO --> UTF-8
 * + "utfutil_u32ConvertUTF82ISOMod": for mapping UTF-8 --> ISO
 * + "utfutil_u32ConvertMCNetMod2UTF8: for mapping MCNet --> UTF-8
 * + "utfutil_u32ConvertUTF82MCNetMod: for mapping UTF-8 --> MCNet
 *
 * HISTORY:
 * $Log:   //hi230124/projects/Vasco/swnavi/archives/components/utf/UTFUtil.h-arc  $
 * 
 *    Rev 1.1   Sep 22 2003 09:09:44   OSR2HI
 * added 2 interface functions for conversion MCNet <--> UTF8:
 * utfutil_u32ConvertMCNetMod2UTF8, utfutil_u32ConvertUTF82MCNetMod
 *                13.05.03 Rev. 1.0 CM-DI/ESA2-Bruns
 *                         Initial Revision;
 *    Rev 1.2    03/13/2008 2:13 PM   RBEI/EDI3 Sajith
 *    Added interface to convert ISO8849_1M to UTF8
 *****************************************************************************/



/*****************************************************************************
 * METHOD:        u8GetStoredContainerLength
 * ---------------------------------------------------------------------------
 * DESCRIPTION:   Calculates the utf-container-length of given utf8-container,
 *                suggested by the first byte of the contailer.
 * ---------------------------------------------------------------------------
 * PARAMETER:     
 *    tU8    u8UTFByte1 ( I )
 *                first byte of the utf8-container for length calculation
 * RETURNVALUE:   Number of bytes of the utf8-contailer (0...6)
 * ---------------------------------------------------------------------------
 * ---------------------------------------------------------------------------
 * HISTORY:
 * 13.05.2003 CM-DI/ESA2-Bruns
 *            Initial Revision.
 **************************************************************************/
tU8 u8GetStoredContainerLength(tU8 u8UTFByte1);

/*****************************************************************************
 * METHOD:        utfutil_u32ConvertMCNetMod2UTF8
 * ---------------------------------------------------------------------------
 * DESCRIPTION:   Converts a sequence of (modified) MCNet characters to a 
 *                corresponding sequence of UTF8-coded characters.
 * ---------------------------------------------------------------------------
 * PARAMETER:     
 *    tChar* pcUTF8Buffer
 *                Desti. buffer, allocated by client. If 0 this routine only 
 *                counts the maximum length which would be needed for
 *                UTF-8 conversion (-> I)
 *    tU32   u32UTF8BufferLen
 *                Length of destination buffer (-> I)
 *    const  tChar *MCNetCodedString
 *                Source string in (modified) MCNet (-> I)
 *    tU32   u32Count (-> I)
 *                number of characters to convert
 * RETURNVALUE:   Number of bytes in UTF-8 sequence
 * ---------------------------------------------------------------------------
 * ---------------------------------------------------------------------------
 * HISTORY:
 * 12.09.2003 CM-DI/ESN-Saake
 *            modified copy 'utfutil_u32ConvertISOMod2UTF8' for MCNet conversion
 **************************************************************************/
tU32  utfutil_u32ConvertMCNetMod2UTF8(tChar* pcUTF8Buffer, tU32 u32UTF8BufferLen, const tChar *MCNetCodedString, tU32 u32Count);


/*****************************************************************************
 * METHOD:        utfutil_u32ConvertUTF82MCNetMod
 * ---------------------------------------------------------------------------
 * DESCRIPTION:   Converts a sequence of UTF8-coded characters to a 
 *                corresponding sequence of (modified) MCNet characters.
 * ---------------------------------------------------------------------------
 * PARAMETER:     
 *    tChar* pcMCNetBuffer
 *                Desti. buffer, allocated by client. If 0 this routine only 
 *                counts the maximum length which would be needed for
 *                UTF-8 conversion (-> I)
 *    tU32   u32MCNetBufferLen
 *                Length of destination buffer (-> I)
 *    const  tChar *UTF8CodedString
 *                Source string in UTF-8 coding (-> I)
 *    tU32   u32UTFMax (-> I)
 *                number of characters to convert
 * RETURNVALUE:   Number of bytes in UTF-8 sequence
 * ---------------------------------------------------------------------------
 * HISTORY:
 * 12.09.2003 CM-DI/ESN-Saake
 *            modified copy 'utfutil_u32ConvertUTF82ISOMod' for MCNet conversion
 **************************************************************************/
tU32  utfutil_u32ConvertUTF82MCNetMod(tChar* pcUTF8Buffer, tU32 u32UTF8BufferLen, const tChar *MCNetCodedString, tU32 u32Count);


/*****************************************************************************
 * METHOD:        utfutil_u32ISOMod2UTF8
 * ---------------------------------------------------------------------------
 * DESCRIPTION:   Converts a sequence of ISO8859-M(odified) characters to a 
 *                corresponding sequence of UTF8-coded characters.
 * ---------------------------------------------------------------------------
 * PARAMETER:     
 *    tChar* pcUTF8Buffer
 *                Desti. buffer, allocated by client. If 0 this routine only 
 *                counts the maximum length which would be needed for
 *                UTF-8 conversion (->I)
 *    tU32   u32UTF8BufferLen
 *                Length of destination buffer (-> I)
 *    const  tChar *IsoCodedString
 *                Source string in ISO8859-15M(odified) (-> I)
 *    tU32   u32Count (->I)
 *                number of characters to convert
 * RETURNVALUE:   Number of bytes in UTF-8 sequence
 * ---------------------------------------------------------------------------
 * REMARK: This is a ISO8859-15M version. Thus the max. conversion length 
 *         for the array below is "3" is _hard_coded_!!
 * ---------------------------------------------------------------------------
 * HISTORY:
 * 13.05.2003 CM-DI/ESA2-Bruns
 *            Initial Revision.
 **************************************************************************/
tU32  utfutil_u32ConvertISOMod2UTF8(tChar* pcUTF8Buffer, tU32 u32UTF8BufferLen, const tChar *IsoCodedString, tU32 u32Count);




/*****************************************************************************
 * METHOD:        util_u32ConvertISO8859_1Mod2UTF8
 * ---------------------------------------------------------------------------
 * DESCRIPTION:   Converts a sequence of ISO8859_1M-coded characters to a 
 *                corresponding sequence of UTF8 characters.
 * ---------------------------------------------------------------------------
 * PARAMETER:     
 *    tChar* pcUTF8Buffer
 *                Desti. buffer, allocated by client. If 0 this routine only 
 *                counts the maximum length which would be needed for
 *                UTF-8 conversion (->I/O)
 *    tU32   u32UTF8BufferLen
 *                Length of destination buffer (-> I)
 *    const  tChar *UTF8CodedString
 *                Source string in ISO8859_1M coding (-> I)
 *    tU32   u32Count (->I)
 *                number of characters to convert
 * RETURNVALUE:   Number of bytes in UTF-8 sequence
 * ---------------------------------------------------------------------------
 * HISTORY:
 * 13.05.2003 RBEI/EDI3-Sajith
 *            Initial Revision.
 **************************************************************************/

tU32  util_u32ConvertISO8859_1Mod2UTF8(tChar* pcUTF8Buffer, tU32 u32UTF8BufferLen,
                                   const tChar *IsoCodedString, tU32 u32Count);


/*****************************************************************************
 * METHOD:        utfutil_u32ConvertUTF82ISOMod
 * ---------------------------------------------------------------------------
 * DESCRIPTION:   Converts a sequence of UTF8-coded characters to a 
 *                corresponding sequence of ISO8859-M(odified) characters.
 * ---------------------------------------------------------------------------
 * PARAMETER:     
 *    tChar* pcISOBuffer
 *                Desti. buffer, allocated by client. If 0 this routine only 
 *                counts the maximum length which would be needed for
 *                UTF-8 conversion (->I)
 *    tU32   u32ISOBufferLen
 *                Length of destination buffer (-> I)
 *    const  tChar *UTF8CodedString
 *                Source string in UTF-8 coding (-> I)
 *    tU32   u32UTFMax (->I)
 *                number of characters to convert
 * RETURNVALUE:   Number of bytes in UTF-8 sequence
 * ---------------------------------------------------------------------------
 * HISTORY:
 * 13.05.2003 CM-DI/ESA2-Bruns
 *            Initial Revision.
 **************************************************************************/
tU32  utfutil_u32ConvertUTF82ISOMod(tChar* pcUTF8Buffer, tU32 u32UTF8BufferLen, const tChar *IsoCodedString, tU32 u32Count);


#define utfutil_StrlenUTF8(pcUTFString) (utfutil_u32ConvertUTF82ISOMod(0,0,pcUTFString, 0)-1)

#define utfutil_SizeOfUTF8(pcIsoString) (utfutil_u32ConvertISOMod2UTF8(0,0,pcIsoString, 0)-1)

