/*****************************************************************************
 * FILE:          UTFUtil.cpp
 * SW-COMPONENT:  UTFUtil
 *----------------------------------------------------------------------------
 * DESCRIPTION:   UTDF utility functions for mapping
 *                ISO8859-15M(odified) <--> UTF-8 byte sequences
 *----------------------------------------------------------------------------
 * AUTHOR:        CM-DI/ESA2-Bruns
 * COPYRIGHT:     (c) 2003 Blaupunkt Werke GmbH
 * REMARK:
 *
 * Actually 2 functions are provided by this module.
 * + "utfutil_u32ConvertISOMod2UTF8": for mapping ISO --> UTF-8
 * + "utfutil_u32ConvertUTF82ISOMod": for mapping UTF-8 --> ISO
 *
 * - no endianess consideration, yet
 * -
 *
 * HISTORY:
 *                13.05.03 Rev. 1.0 CM-DI/ESA2-Bruns
 *                         Initial Revision;
 *    Rev 1.1    03/13/2008 2:13 PM   RBEI/EDI3 Sajith
 *    Added interface "util_u32ConvertISO8859_1Mod2UTF8"to convert 
 *    ISO8849_1M to UTF8 
 *****************************************************************************/


// System
#define SYSTEM_S_IMPORT_INTERFACE_COMPLETE
#include "system_pif.h"

// OSAL

/*
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
*/

// Self include
#include "UTFUtil.h"


// Specifies start of code pair region. Smaller codes are mapped to 1 byte code.
#define UTFUTIL_U8_ISO2UTFCODEPAIR_START        (tU8)0x80

// Symbol in ISO8859-15M to map undefined UTF-8 symbols to
#define UTFUTIL_U8_ISO2UTFCODEPAIR_UNDEFINED    (tU8)0x80



/*****************************************************************************
 * utfutil_tISO2UTFCodePair
 *
 * Table for mapping ISO8859-15M to UTF-8. Contains symbols in pre-calculated
 * UTF-8 sequence for the mapping of ISO region [0x80..0xBF].
 * First symbol represents code value for 0x80... and so on. Last symbol
 * represents code value for 0xBF.
 *****************************************************************************/
struct utfutil_tISO2UTFCodePair {
//  tU8  u8ISOSymbol;
  tU8  u8UTFSymbol1;
  tU8  u8UTFSymbol2;
} rISOSymTab[] = {
   { 0xc2, 0x80 },   { 0xc4, 0x84 },   { 0xc4, 0x8c },   { 0xc4, 0x86 },
   { 0xc4, 0x8e },   { 0xc4, 0x9a },   { 0xc4, 0x98 },   { 0xc5, 0x81 },
   { 0xc4, 0xbd },   { 0xc4, 0xb9 },   { 0xc5, 0x87 },   { 0xc5, 0x83 },
   { 0xc5, 0x90 },   { 0xc5, 0x98 },   { 0xc5, 0x94 },   { 0xc5, 0x9a },
   { 0xc5, 0xa4 },   { 0xc5, 0xae },   { 0xc5, 0xb0 },   { 0xc5, 0xb9 },
   { 0xc5, 0xbb },   { 0xc4, 0x96 },   { 0xc4, 0xae },   { 0xc5, 0xb2 },
   { 0xc5, 0xaa },   { 0xc4, 0x80 },   { 0xc4, 0x92 },   { 0xc4, 0xaa },
   { 0xc4, 0xa2 },   { 0xc4, 0xb6 },   { 0xc4, 0xbb },   { 0xc5, 0x85 },
   { 0xc2, 0xa0 },   { 0xc2, 0xa1 },   { 0xc2, 0xa2 },   { 0xc2, 0xa3 },
   { 0xc2, 0xac },   { 0xc2, 0xa5 },   { 0xc5, 0xa0 },   { 0xc2, 0xa7 },
   { 0xc5, 0xa1 },   { 0xc2, 0xa9 },   { 0xc2, 0xaa },   { 0xc2, 0xab },
   { 0xc2, 0xac },   { 0xc2, 0xad },   { 0xc2, 0xae },   { 0xc2, 0xaf },
   { 0xc2, 0xb0 },   { 0xc2, 0xb1 },   { 0xc2, 0xb2 },   { 0xc2, 0xb3 },
   { 0xc5, 0xbd },   { 0xc2, 0xb5 },   { 0xc2, 0xb6 },   { 0xc2, 0xb7 },
   { 0xc5, 0xbe },   { 0xc2, 0xb9 },   { 0xc2, 0xba },   { 0xc2, 0xbb },
   { 0xc5, 0x92 },   { 0xc5, 0x93 },   { 0xc5, 0xb8 },   { 0xc2, 0xbf }
};


/*****************************************************************************
 * utfutil_tISO1toUTFCodePair
 *
 * Table for mapping ISO8859-1M to UTF-8. Contains symbols in pre-calculated
 * UTF-8 sequence for the mapping of ISO8859-1M region [0x80..0xBF].
 * First symbol represents code value for 0x80... and so on. Last symbol
 * represents code value for 0xBF.
 *****************************************************************************/
struct utfutil_tISO1toUTFCodePair {
    //  tU8  u8ISOSymbol;
    tU8  u8UTFSymbol1;
    tU8  u8UTFSymbol2;
} rIso8859_1MToUtf8SymTab[] = {
    { 0xc2, 0x80 },   { 0xc4, 0x84 },   { 0xc4, 0x8c },   { 0xc4, 0x86 },
    { 0xc4, 0x8e },   { 0xc4, 0x9a },   { 0xc4, 0x98 },   { 0xc5, 0x81 },
    { 0xc4, 0xbd },   { 0xc4, 0xb9 },   { 0xc5, 0x87 },   { 0xc5, 0x83 },
    { 0xc5, 0x90 },   { 0xc5, 0x98 },   { 0xc5, 0x94 },   { 0xc5, 0x9a },
    { 0xc5, 0xa4 },   { 0xc5, 0xae },   { 0xc5, 0xb0 },   { 0xc5, 0xb9 },
    { 0xc5, 0xbb },   { 0xc4, 0x96 },   { 0xc4, 0xae },   { 0xc5, 0xb2 },
    { 0xc5, 0xaa },   { 0xc4, 0x80 },   { 0xc4, 0x92 },   { 0xc4, 0xaa },
    { 0xc4, 0xa2 },   { 0xc4, 0xb6 },   { 0xc4, 0xbb },   { 0xc5, 0x85 },
    { 0xc2, 0xa0 },   { 0xc2, 0xa1 },   { 0xc2, 0xa2 },   { 0xc2, 0xa3 },
    { 0xc2, 0xa4 },   { 0xc2, 0xa5 },   { 0xc2, 0xa6 },   { 0xc2, 0xa7 },
    { 0xc2, 0xa8 },   { 0xc2, 0xa9 },   { 0xc2, 0xaa },   { 0xc2, 0xab },
    { 0xc2, 0xac },   { 0xc2, 0xad },   { 0xc2, 0xae },   { 0xc2, 0xaf },
    { 0xc2, 0xb0 },   { 0xc2, 0xb1 },   { 0xc2, 0xb2 },   { 0xc2, 0xb3 },
    { 0xc2, 0xb4 },   { 0xc2, 0xb5 },   { 0xc2, 0xb6 },   { 0xc2, 0xb7 },
    { 0xc2, 0xb8 },   { 0xc2, 0xb9 },   { 0xc2, 0xba },   { 0xc2, 0xbb },
    { 0xc2, 0xbc },   { 0xc2, 0xbd },   { 0xc2, 0xbe },   { 0xc2, 0xbf }
};



/*****************************************************************************
 * utfutil_tUTF2ISOValue
 *
 * Table for mapping UTF-8 to ISO8859-15M. Table must be sorted in ascending
 * order by the value "u32UTFSymbol" to be used in BINARY SEARCH in
 * "utfutil_u32ConvertUTF82ISOMod".
 *****************************************************************************/
struct utfutil_tUTF2ISOValue {
  tU8  u8ISOSymbol;
  tU32 u32UTFSymbol;
} rUtf8ToIso8859MSymTab[] = {
   {   0x80,0x0080   },{   0xA0,0x00A0   },{   0xA1,0x00A1   },{   0xA2,0x00A2   },
   {   0xA3,0x00A3   },{   0xA5,0x00A5   },{   0xA7,0x00A7   },{   0xA9,0x00A9   },
   {   0xAA,0x00AA   },{   0xAB,0x00AB   },{   0xAC,0x00AC   },{   0xAD,0x00AD   },
   {   0xAE,0x00AE   },{   0xAF,0x00AF   },{   0xB0,0x00B0   },{   0xB1,0x00B1   },
   {   0xB2,0x00B2   },{   0xB3,0x00B3   },{   0xB5,0x00B5   },{   0xB6,0x00B6   },
   {   0xB7,0x00B7   },{   0xB9,0x00B9   },{   0xBA,0x00BA   },{   0xBB,0x00BB   },
   {   0xBF,0x00BF   },{   0x99,0x0100   },{   0x81,0x0104   },{   0x83,0x0106   },
   {   0x82,0x010C   },{   0x84,0x010E   },{   0x9A,0x0112   },{   0x95,0x0116   },
   {   0x86,0x0118   },{   0x85,0x011A   },{   0x9C,0x0122   },{   0x9B,0x012A   },
   {   0x96,0x012E   },{   0x9D,0x0136   },{   0x89,0x0139   },{   0x9E,0x013B   },
   {   0x88,0x013D   },{   0x87,0x0141   },{   0x8B,0x0143   },{   0x9F,0x0145   },
   {   0x8A,0x0147   },{   0x8C,0x0150   },{   0xBC,0x0152   },{   0xBD,0x0153   },
   {   0x8E,0x0154   },{   0x8D,0x0158   },{   0x8F,0x015A   },{   0xA6,0x0160   },
   {   0xA8,0x0161   },{   0x90,0x0164   },{   0x98,0x016A   },{   0x91,0x016E   },
   {   0x92,0x0170   },{   0x97,0x0172   },{   0xBE,0x0178   },{   0x93,0x0179   },
   {   0x94,0x017B   },{   0xB4,0x017D   },{   0xB8,0x017E   },{   0xA4,0x20AC   }
};




/*****************************************************************************
 * METHOD:        utfutil_u32ISOMod2UTF8
 * ---------------------------------------------------------------------------
 * DESCRIPTION:   Converts a sequence of ISO8859-M(odified) characters to a
 *                corresponding sequence of UTF8-coded characters.
 * ---------------------------------------------------------------------------
 * PARAMETER:
 *    tChar* pcUTF8Buffer
 *                Desti. buffer, allocated by client. If 0 this routine only
 *                counts the maximum length which would be needed for
 *                UTF-8 conversion (->I)
 *    tU32   u32UTF8BufferLen
 *                Length of destination buffer (-> I)
 *    const  tChar *IsoCodedString
 *                Source string in ISO8859-15M(odified) (-> I)
 *    tU32   u32Count (->I)
 *                number of characters to convert
 * RETURNVALUE:   Number of bytes in UTF-8 sequence
 * ---------------------------------------------------------------------------
 * REMARK: This is a ISO8859-15M version. Thus the max. conversion length
 *         for the array below is "3" is _hard_coded_!!
 * ---------------------------------------------------------------------------
 * HISTORY:
 * 13.05.2003 CM-DI/ESA2-Bruns
 *            Initial Revision.
 **************************************************************************/
tU32
utfutil_u32ConvertISOMod2UTF8(
         tChar* pcUTF8Buffer,          // Desti. buffer, allocated by client
         tU32   u32UTF8BufferLen,      // Length of destination buffer
         const  tChar *IsoCodedString, // Source string in ISO8859-15M(odified)
         tU32   u32ISOMax)              // number of characters to convert
{
   tU32 u32UTFCount=0;
   tU8  caConverted[3] = {0,0,0};                     // Array for converted code
   tU32 u32ConversionLength=0;              // Stores lenght of used array
   tU32 u32ISOCount = 0;
   tBool bContinue = (u32ISOCount <= u32ISOMax) ? TRUE : FALSE;

   while(TRUE == bContinue)
      //for(tU32 i=0; i<u32ISOMax; i++)
   {
      tU8 u8ISOCode = IsoCodedString[u32ISOCount];

      // Check Regions: [0x00-0x7F] --> Standard-ASCII, 1 byte code
//Wn     if(((tU8)0x00 <= u8ISOCode) && (u8ISOCode <=(tU8)0x7F))
      if(u8ISOCode <=(tU8)0x7F)
      {
         u32ConversionLength=1;
         caConverted[0] = u8ISOCode;
      }
      else
      {
         // [0xA4]  --> Euro symbol = 3 byte code E282AC
         if (((tU8)0xA4) == u8ISOCode)
         {
            u32ConversionLength=3;
            caConverted[0] = (tU8)0xE2;
            caConverted[1] = (tU8)0x82;
            caConverted[2] = (tU8)0xAC;
         }
         else if (((tU8)0xC0 <= u8ISOCode) /*&& (u8ISOCode <=(tU8)0xFF)*/)
         {
            // [0xC0..0xFF]
            // linear region 0xC0..0xFF --> U+00C0..U+00FF
            // Use formula for conversion
            u32ConversionLength = 2;
            caConverted[0] = ((tU8)u8ISOCode >> 6) | 0xC0;
            caConverted[1] = (tU8)(((tU8)u8ISOCode & 0x3F) + 0x80);
         }
         else
         {
            // 0x80-0xBF \ {0xA4} = 2 Byte Code
            tU32 u32CharIndex = u8ISOCode - UTFUTIL_U8_ISO2UTFCODEPAIR_START;
            if(u32CharIndex < (sizeof(rISOSymTab)/sizeof(struct utfutil_tISO2UTFCodePair)) )
            {
               u32ConversionLength=2;
               caConverted[0] = (tU8)(rISOSymTab[u32CharIndex].u8UTFSymbol1);
               caConverted[1] = (tU8)(rISOSymTab[u32CharIndex].u8UTFSymbol2);
            }
         }
      }

      // Result: Array "caConverted" contains conversion;
      //         u32ConversionLength contains length in bytes;

      // Only write if destination buffer present
      if(0 != pcUTF8Buffer)
      {
         // Take care of destination buffer length
         if(u32UTFCount + u32ConversionLength <= u32UTF8BufferLen)
         {
            /* quick fix: replaced "OSAL_pvMemoryCopy" q&d by following "for"-loop
               OSAL_pvMemoryCopy(&pcUTF8Buffer[u32UTFCount], caConverted, u32ConversionLength);
            */
            for(tU32 i=0; i < u32ConversionLength; i++)
            {
               pcUTF8Buffer[u32UTFCount+i] = caConverted[i];
            }
         }
      }
      u32UTFCount+=u32ConversionLength;

      u32ISOCount++;

      // Check for end condition:
      // IsoCount < IsoMax or in case of IsoMax=0 until 0-byte found
      // in ISO-string
      if(0 == u32ISOMax)
         bContinue = (0 != u8ISOCode) ? TRUE : FALSE;
      else
         bContinue =  (u32ISOCount < u32ISOMax) ? TRUE : FALSE;
   }
   return u32UTFCount;
}



tU8 u8GetStoredContainerLength(tU8 u8UTFByte1)
{
   tU8 u8UTFContainerLength=1;

   // Check for 1-byte character code
   if(u8UTFByte1 >= (tU8)0xC0)
   {
      // Multi-byte code -> Get container length
      u8UTFContainerLength=0;
      while(((tU8)((u8UTFByte1 << (++u8UTFContainerLength)) & (tU8)0x80)))
         ;
   }
   return u8UTFContainerLength;
}




tU8 u8DecodeUTFContainer(tU8 *u8IsoChar, const tU8 *pUTFBuffer)
{
   *u8IsoChar = UTFUTIL_U8_ISO2UTFCODEPAIR_UNDEFINED;
   tU8 u8UTFContainerLength = 1;

   // Get 1. byte
   tU8 u8UTFByte1 = *pUTFBuffer;

   // verify
//Wn   if((((tU8)0x00 <= u8UTFByte1) && (u8UTFByte1 <= (tU8)0x7F)) ||
//        (((tU8)0xC0 <= u8UTFByte1) && (u8UTFByte1 <= (tU8)0xFD)))
   if(    (u8UTFByte1 <= (tU8)0x7F)
       || (    ((tU8)0xC0 <= u8UTFByte1)
            && (u8UTFByte1 <= (tU8)0xFD)
          )
     )
   {
      u8UTFContainerLength = u8GetStoredContainerLength(u8UTFByte1);
      if(1 == u8UTFContainerLength)
      {  // 1 byte code
         *u8IsoChar = u8UTFByte1;
      }  // > 1 byte code
      else
      {
         // get data bytes
         tU32 u32ContainerData = 0;

         // First data byte (data part smaller than 6)
         u8UTFByte1 <<= (u8UTFContainerLength+1);
         u8UTFByte1 >>= (u8UTFContainerLength+1);
         u32ContainerData |= u8UTFByte1;

         tU8 u8Index = 0;
         for(u8Index = 1; u8Index <= (u8UTFContainerLength-1); u8Index++)
         {
            tU8 u8DataItem = pUTFBuffer[u8Index];
            // valid data?
            if(((tU8)0x80 <= u8DataItem) && (u8DataItem <= (tU8)0xBF))
            {
               u32ContainerData <<= 6;
               u32ContainerData |= (tU32)((u8DataItem & (tU8)0x3F));
            }
            else
            {
               *u8IsoChar = UTFUTIL_U8_ISO2UTFCODEPAIR_UNDEFINED;
               break;
            }
         }
         u8UTFContainerLength = u8Index;

         // Decode
         switch(u8UTFContainerLength)
         {
            case 2:  if(((tU32)0x00C0 <= u32ContainerData) && (u32ContainerData <=(tU32)0x00FF))
                     {  // [0xC380..0xC3BF] --> second linear region (-> formula)
                        *u8IsoChar = (tU8)(u32ContainerData & (tU32)0x00FF);
                     }
                     else
                     {  // region from table
                        tU32 u32Max = (sizeof(rUtf8ToIso8859MSymTab) /
                                       sizeof(struct utfutil_tUTF2ISOValue));
                        for(tU32 u32TableIndex=0; u32TableIndex < u32Max; u32TableIndex++)
                        {
                           if(u32ContainerData == rUtf8ToIso8859MSymTab[u32TableIndex].u32UTFSymbol)
                           {
                              *u8IsoChar = rUtf8ToIso8859MSymTab[u32TableIndex].u8ISOSymbol;
                              break;
                           }
                        }
                     }
                     break;
            case 3:  if(((tU32)0x20AC == u32ContainerData))
                        *u8IsoChar = (tU8)(0xA4); // Euro symbol
               break;
            default:
                     *u8IsoChar = UTFUTIL_U8_ISO2UTFCODEPAIR_UNDEFINED;
         }
      }
   }
   return u8UTFContainerLength;
}





/*****************************************************************************
 * METHOD:        util_u32ConvertISO8859_1Mod2UTF8
 * ---------------------------------------------------------------------------
 * DESCRIPTION:   Converts a sequence of ISO8859_1M-coded characters to a 
 *                corresponding sequence of UTF8 characters.
 * ---------------------------------------------------------------------------
 * PARAMETER:     
 *    tChar* pcUTF8Buffer
 *                Desti. buffer, allocated by client. If 0 this routine only 
 *                counts the maximum length which would be needed for
 *                UTF-8 conversion (->I/O)
 *    tU32   u32UTF8BufferLen
 *                Length of destination buffer (-> I)
 *    const  tChar *UTF8CodedString
 *                Source string in ISO8859_1M coding (-> I)
 *    tU32   u32Count (->I)
 *                number of characters to convert
 * RETURNVALUE:   Number of bytes in UTF-8 sequence
 * ---------------------------------------------------------------------------
 * HISTORY:
 * 13.05.2003 RBEI/EDI3-Sajith
 *            Initial Revision.
 **************************************************************************/

tU32  util_u32ConvertISO8859_1Mod2UTF8(
                   tChar* pcUTF8Buffer,          // Desti. buffer, allocated by client
                   tU32   u32UTF8BufferLen,      // Length of destination buffer
                   const  tChar *IsoCodedString, // Source string in ISO8859-1M
                   tU32   u32ISOMax)              // number of characters to convert
{
    tU32 u32UTFCount = 0;
    tU8  caConverted[2] = {0,0};            // Array for converted code
    tU32 u32ConversionLength=0;             // Stores lenght of used array
    tU32 u32ISOCount = 0;
    tBool bContinue = (u32ISOCount <= u32ISOMax) ? TRUE : FALSE;
    
    while(TRUE == bContinue)
        //for(tU32 i=0; i<u32ISOMax; i++)
    {
        tU8 u8ISOCode = IsoCodedString[u32ISOCount];
        
        // Check Regions: [0x00-0x7F] --> Standard-ASCII, 1 byte code
        //Wn     if(((tU8)0x00 <= u8ISOCode) && (u8ISOCode <=(tU8)0x7F))
        if(u8ISOCode <=(tU8)0x7F)
        {
            u32ConversionLength=1;
            caConverted[0] = u8ISOCode;
        }
        else
        {
            if (((tU8)0xC0 <= u8ISOCode) /*&& (u8ISOCode <=(tU8)0xFF)*/)
            {
                // [0xC0..0xFF]
                // linear region 0xC0..0xFF --> U+00C0..U+00FF
                // Use formula for conversion
                u32ConversionLength = 2;
                caConverted[0] = ((tU8)u8ISOCode >> 6) | 0xC0;
                caConverted[1] = (tU8)(((tU8)u8ISOCode & 0x3F) + 0x80);
            }
            else
            {
                // 0x80-0xBF \ {0xA4} = 2 Byte Code
                tU32 u32CharIndex = u8ISOCode - UTFUTIL_U8_ISO2UTFCODEPAIR_START;
                if( u32CharIndex < 
                    ( sizeof(rIso8859_1MToUtf8SymTab)/sizeof(struct utfutil_tISO1toUTFCodePair) ) 
                    )
                {
                    u32ConversionLength=2;
                    caConverted[0] = (tU8)(rIso8859_1MToUtf8SymTab[u32CharIndex].u8UTFSymbol1);
                    caConverted[1] = (tU8)(rIso8859_1MToUtf8SymTab[u32CharIndex].u8UTFSymbol2);
                }
            }
        }//else of if(u8ISOCode <=(tU8)0x7F)
        
        // Result: Array "caConverted" contains conversion;
        // u32ConversionLength contains length in bytes;
        
        // Only write if destination buffer present
        if(0 != pcUTF8Buffer)
        {
            // Take care of destination buffer length
            if(u32UTFCount + u32ConversionLength <= u32UTF8BufferLen)
            {
            /* quick fix: replaced "OSAL_pvMemoryCopy" q&d by following "for"-loop
            OSAL_pvMemoryCopy(&pcUTF8Buffer[u32UTFCount], caConverted, u32ConversionLength);
                */
                for(tU32 i=0; i < u32ConversionLength; i++)
                {
                    pcUTF8Buffer[u32UTFCount+i] = caConverted[i];
                }
            }
        }
        u32UTFCount+= u32ConversionLength;
        
        u32ISOCount++;
        
        // Check for end condition:
        // IsoCount < IsoMax or in case of IsoMax=0 until 0-byte found
        // in ISO-string
        if(0 == u32ISOMax)
            bContinue = (0 != u8ISOCode) ? TRUE : FALSE;
        else
            bContinue =  (u32ISOCount < u32ISOMax) ? TRUE : FALSE;
    }
    return u32UTFCount;
}






/*****************************************************************************
 * METHOD:        utfutil_u32ConvertUTF82ISOMod
 * ---------------------------------------------------------------------------
 * DESCRIPTION:   Converts a sequence of UTF8-coded characters to a
 *                corresponding sequence of ISO8859-M(odified) characters.
 * ---------------------------------------------------------------------------
 * PARAMETER:
 *    tChar* pcISOBuffer
 *                Desti. buffer, allocated by client. If 0 this routine only
 *                counts the maximum length which would be needed for
 *                UTF-8 conversion (->I).
 *    tU32   u32ISOBufferLen
 *                Length of destination buffer. If a buffer length of "0"
 *                is given the conversion did not write to memory and just
 *                counts the resulting bytes in ISO character (-> I).
 *    const  tChar *UTF8CodedString
 *                Source string in UTF-8 coding (-> I).
 *    tU32   u32UTFMax (->I).
 *                Number of characters to convert. A number of "0" converts
 *                until a 0-Byte termination is found (in first UTF-byte) in
 *                source string.
 *
 * Remark: Invalid UTF-Streams are transformed according to UTF-guidelines
 *
 * RETURNVALUE:   Number of bytes in ISO sequence
 * ---------------------------------------------------------------------------
 * HISTORY:
 * 13.05.2003 CM-DI/ESA2-Bruns
 *            Initial Revision.
 **************************************************************************/
tU32
utfutil_u32ConvertUTF82ISOMod(
         tChar* pcISOBuffer,           // Desti. buffer, allocated by client
         tU32   u32ISOBufferLen,       // Length of destination buffer
         const  tChar *UTF8CodedString,// Source string in UTF-8
         tU32   u32UTFMax)             // number of UTF-8 characters to convert
{
   tU32 u32UTFCount = 0;
   tU32 u32ISOCount = 0;

   tBool bContinue = (u32UTFCount <= u32UTFMax) ? TRUE : FALSE;
   while(TRUE == bContinue) //  && (u32ISOCount < u32ISOBufferLen))
   {
      tU8 u8UTFByte1          = UTF8CodedString[u32UTFCount];
      tU8 u8UTFContainerLength= 1;
      tU8 u8IsoCode           = UTFUTIL_U8_ISO2UTFCODEPAIR_UNDEFINED;

      // Find first valid char (skip invalid ones)
//Wn      if((((tU8)0x00 <= u8UTFByte1) && (u8UTFByte1 <= (tU8)0x7F)) ||
//          (((tU8)0xC0 <= u8UTFByte1) && (u8UTFByte1 <= (tU8)0xFD)))
      if(    (u8UTFByte1 <= (tU8)0x7F)
          || (    ((tU8)0xC0 <= u8UTFByte1)
               && (u8UTFByte1 <= (tU8)0xFD)
             )
        )
      {
         u8UTFContainerLength = u8DecodeUTFContainer(&u8IsoCode, (const tU8*)&UTF8CodedString[u32UTFCount]);

         // Write to target buffer only if "bufferlen > 0".
         // (condition always false if "u32ISOBufferLen==0")

         if(u32ISOCount < u32ISOBufferLen)
            pcISOBuffer[u32ISOCount] = u8IsoCode;
         u32ISOCount++;
      }
      u32UTFCount+=(tU32)u8UTFContainerLength;

      // Check for end condition:
      // u32UTFCount < u32UTFMax or in case of UTFMax=0 until first
      // UTF-Byte == 0 found in UTF-string
      if(0 == u32UTFMax)
         bContinue = (0 != u8UTFByte1) ? TRUE : FALSE;
      else
         bContinue =  (u32UTFCount < u32UTFMax) ? TRUE : FALSE;
   }

   return u32ISOCount;
}
