/*******************************************************************************
*
* FILE:          sortlib_table.c
*
* SW-COMPONENT:  Sorting Lib
*
* PROJECT:       Bosch-CMD
*
* DESCRIPTION:   Sorting Lib for Sqlite
*
* AUTHOR:
*
* COPYRIGHT:     (c) 2011 Robert Bosch GmbH, Hildesheim
*
*******************************************************************************/

/*******************************************************************************
*             INCLUDES
*******************************************************************************/

/*******************************************************************************
*             DEFINES
*******************************************************************************/
#define SORTLIB_DEFAULT_TABLE_VALUE 0xFFFF
//#define SORTLIB_DEFAULT_TABLE_VALUE 0x0000
#include "sortlib_helper.h"

static void sortlib_initialize_sort_table( unsigned sortTable[] );//Prototype
unsigned* sortlib_get_rule_table()
{
   static unsigned sortlib_ruleTable[SORTLIB_ARRAY_SIZE];
   static int done = 0;//Not using FALSE/TRUE here as it will cause inclusion of unwanted files/linkage

   if ( 0 == done )
   {
      done = 1;
      sortlib_initialize_sort_table( sortlib_ruleTable );
   }

   return sortlib_ruleTable;
}

#pragma GCC push_options
#pragma GCC optimize ("O0")

//This function would be updated as new data is made available
static void sortlib_initialize_sort_table( unsigned sortTable[] )
{
   unsigned i;

   for ( i = 0; i < SORTLIB_ARRAY_SIZE; ++i )
   {
      sortTable[i] = SORTLIB_DEFAULT_TABLE_VALUE;
   }

   // the following sorting table was calculated and prepared in an Excel file.
   // This is available under: https://hi-dms.de.bosch.com/docushare/dsweb/Get/Document-1094483/GlobalSortingTable.xlsm

   //Bosch Golden Sorting Table
   sortTable[0x0020]=0;   // SPACE
   sortTable[0x00A0]=0;   // NO-BREAK SPACE
   sortTable[0x0021]=2;   // EXCLAMATION MARK
   sortTable[0x0022]=3;   // QUOTATION MARK
   sortTable[0x0023]=4;   // NUMBER SIGN
   sortTable[0x0025]=5;   // PERCENT SIGN
   sortTable[0x0026]=6;   // AMPERSAND
   sortTable[0x0027]=7;   // APOSTROPHE
   sortTable[0x0028]=8;   // LEFT PARENTHESIS
   sortTable[0x0029]=9;   // RIGHT PARENTHESIS
   sortTable[0x002A]=10;   // ASTERISK
   sortTable[0x002B]=11;   // PLUS SIGN
   sortTable[0x002C]=12;   // COMMA
   sortTable[0x002D]=13;   // HYPHEN-MINUS
   sortTable[0x002E]=14;   // FULL STOP
   sortTable[0x002F]=15;   // SOLIDUS
   sortTable[0x003A]=16;   // COLON
   sortTable[0x003B]=17;   // SEMICOLON
   sortTable[0x003C]=18;   // LESS-THAN SIGN
   sortTable[0x003D]=19;   // EQUALS SIGN
   sortTable[0x003E]=20;   // GREATER-THAN SIGN
   sortTable[0x003F]=21;   // QUESTION MARK
   sortTable[0x0040]=22;   // COMMERCIAL AT
   sortTable[0x005B]=23;   // LEFT SQUARE BRACKET
   sortTable[0x005C]=24;   // REVERSE SOLIDUS
   sortTable[0x005D]=25;   // RIGHT SQUARE BRACKET
   sortTable[0x005E]=26;   // CIRCUMFLEX ACCENT
   sortTable[0x005F]=27;   // LOW LINE
   sortTable[0x007B]=28;   // LEFT CURLY BRACKET
   sortTable[0x007C]=29;   // VERTICAL LINE
   sortTable[0x007D]=30;   // RIGHT CURLY BRACKET
   sortTable[0x007E]=31;   // TILDE
   sortTable[0x00A1]=32;   // INVERTED EXCLAMATION MARK
   sortTable[0x00A7]=33;   // SECTION SIGN
   sortTable[0x00BF]=34;   // INVERTED QUESTION MARK
   sortTable[0x0024]=35;   // DOLLAR SIGN
   sortTable[0x00A2]=36;   // CENT SIGN
   sortTable[0x00A3]=37;   // POUND SIGN
   sortTable[0x00A4]=38;   // CURRENCY SIGN
   sortTable[0x00A5]=39;   // YEN SIGN
   sortTable[0x20AC]=40;   // EURO SIGN
   sortTable[0x00A6]=41;   // BROKEN BAR
   sortTable[0x00A8]=42;   // DIAERESIS
   sortTable[0x00A9]=43;   // COPYRIGHT SIGN
   sortTable[0x00AA]=44;   // FEMININE ORDINAL INDICATOR
   sortTable[0x00AB]=45;   // LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
   sortTable[0x00AC]=46;   // NOT SIGN
   sortTable[0x00AD]=47;   // SOFT HYPHEN
   sortTable[0x00AE]=48;   // REGISTERED SIGN
   sortTable[0x00AF]=49;   // MACRON
   sortTable[0x00B0]=50;   // DEGREE SIGN
   sortTable[0x00B1]=51;   // PLUS-MINUS SIGN
   sortTable[0x00B2]=52;   // SUPERSCRIPT TWO
   sortTable[0x00B3]=53;   // SUPERSCRIPT THREE
   sortTable[0x00B4]=54;   // ACUTE ACCENT
   sortTable[0x00B5]=55;   // MICRO SIGN
   sortTable[0x00B6]=56;   // PILCROW SIGN
   sortTable[0x00B7]=57;   // MIDDLE DOT
   sortTable[0x00B8]=58;   // CEDILLA
   sortTable[0x00B9]=59;   // SUPERSCRIPT ONE
   sortTable[0x00BA]=60;   // MASCULINE ORDINAL INDICATOR
   sortTable[0x00BB]=61;   // RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
   sortTable[0x00BC]=62;   // VULGAR FRACTION ONE QUARTER
   sortTable[0x00BD]=63;   // VULGAR FRACTION ONE HALF
   sortTable[0x00BE]=64;   // VULGAR FRACTION THREE QUARTERS
   sortTable[0x00D7]=65;   // MULTIPLICATION SIGN
   sortTable[0x00F7]=66;   // DIVISION SIGN
   sortTable[0x0030]=67;   // DIGIT ZERO
   sortTable[0x0031]=68;   // DIGIT ONE
   sortTable[0x0032]=69;   // DIGIT TWO
   sortTable[0x0033]=70;   // DIGIT THREE
   sortTable[0x0034]=71;   // DIGIT FOUR
   sortTable[0x0035]=72;   // DIGIT FIVE
   sortTable[0x0036]=73;   // DIGIT SIX
   sortTable[0x0037]=74;   // DIGIT SEVEN
   sortTable[0x0038]=75;   // DIGIT EIGHT
   sortTable[0x0039]=76;   // DIGIT NINE
   sortTable[0x0061]=512;   // LATIN SMALL LETTER A
   sortTable[0x0041]=513;   // LATIN CAPITAL LETTER A
   sortTable[0x00E1]=514;   // LATIN SMALL LETTER A WITH ACUTE
   sortTable[0x00C1]=515;   // LATIN CAPITAL LETTER A WITH ACUTE
   sortTable[0x00E0]=516;   // LATIN SMALL LETTER A WITH GRAVE
   sortTable[0x00C0]=517;   // LATIN CAPITAL LETTER A WITH GRAVE
   sortTable[0x0103]=518;   // LATIN SMALL LETTER A WITH BREVE
   sortTable[0x0102]=519;   // LATIN CAPITAL LETTER A WITH BREVE
   sortTable[0x00E2]=520;   // LATIN SMALL LETTER A WITH CIRCUMFLEX
   sortTable[0x00C2]=521;   // LATIN CAPITAL LETTER A WITH CIRCUMFLEX
   sortTable[0x01CE]=522;   // LATIN SMALL LETTER A WITH CARON
   sortTable[0x01CD]=523;   // LATIN CAPITAL LETTER A WITH CARON
   sortTable[0x00E5]=524;   // LATIN SMALL LETTER A WITH RING ABOVE
   sortTable[0x00C5]=525;   // LATIN CAPITAL LETTER A WITH RING ABOVE
   sortTable[0x00E4]=526;   // LATIN SMALL LETTER A WITH DIAERESIS
   sortTable[0x00C4]=527;   // LATIN CAPITAL LETTER A WITH DIAERESIS
   sortTable[0x00E3]=528;   // LATIN SMALL LETTER A WITH TILDE
   sortTable[0x00C3]=529;   // LATIN CAPITAL LETTER A WITH TILDE
   sortTable[0x0105]=530;   // LATIN SMALL LETTER A WITH OGONEK
   sortTable[0x0104]=531;   // LATIN CAPITAL LETTER A WITH OGONEK
   sortTable[0x0101]=532;   // LATIN SMALL LETTER A WITH MACRON
   sortTable[0x0100]=533;   // LATIN CAPITAL LETTER A WITH MACRON
   sortTable[0x00E6]=534;   // LATIN SMALL LETTER AE
   sortTable[0x00C6]=535;   // LATIN CAPITAL LETTER AE
   sortTable[0x0062]=1024;   // LATIN SMALL LETTER B
   sortTable[0x0042]=1025;   // LATIN CAPITAL LETTER B
   sortTable[0x0063]=2304;   // LATIN SMALL LETTER C
   sortTable[0x0043]=2305;   // LATIN CAPITAL LETTER C
   sortTable[0x0107]=2306;   // LATIN SMALL LETTER C WITH ACUTE
   sortTable[0x0106]=2307;   // LATIN CAPITAL LETTER C WITH ACUTE
   sortTable[0x0109]=2308;   // LATIN SMALL LETTER C WITH CIRCUMFLEX
   sortTable[0x0108]=2309;   // LATIN CAPITAL LETTER C WITH CIRCUMFLEX
   sortTable[0x010D]=2310;   // LATIN SMALL LETTER C WITH CARON
   sortTable[0x010C]=2311;   // LATIN CAPITAL LETTER C WITH CARON
   sortTable[0x010B]=2312;   // LATIN SMALL LETTER C WITH DOT ABOVE
   sortTable[0x010A]=2313;   // LATIN CAPITAL LETTER C WITH DOT ABOVE
   sortTable[0x00E7]=2314;   // LATIN SMALL LETTER C WITH CEDILLA
   sortTable[0x00C7]=2315;   // LATIN CAPITAL LETTER C WITH CEDILLA
   sortTable[0x0064]=4096;   // LATIN SMALL LETTER D
   sortTable[0x0044]=4097;   // LATIN CAPITAL LETTER D
   sortTable[0x010F]=4098;   // LATIN SMALL LETTER D WITH CARON
   sortTable[0x010E]=4099;   // LATIN CAPITAL LETTER D WITH CARON
   sortTable[0x0111]=4100;   // LATIN SMALL LETTER D WITH STROKE
   sortTable[0x0110]=4101;   // LATIN CAPITAL LETTER D WITH STROKE
   sortTable[0x00F0]=4102;   // LATIN SMALL LETTER ETH
   sortTable[0x00D0]=4103;   // LATIN CAPITAL LETTER ETH
   sortTable[0x0065]=5376;   // LATIN SMALL LETTER E
   sortTable[0x0045]=5377;   // LATIN CAPITAL LETTER E
   sortTable[0x00E9]=5378;   // LATIN SMALL LETTER E WITH ACUTE
   sortTable[0x00C9]=5379;   // LATIN CAPITAL LETTER E WITH ACUTE
   sortTable[0x00E8]=5380;   // LATIN SMALL LETTER E WITH GRAVE
   sortTable[0x00C8]=5381;   // LATIN CAPITAL LETTER E WITH GRAVE
   sortTable[0x00EA]=5382;   // LATIN SMALL LETTER E WITH CIRCUMFLEX
   sortTable[0x00CA]=5383;   // LATIN CAPITAL LETTER E WITH CIRCUMFLEX
   sortTable[0x011B]=5384;   // LATIN SMALL LETTER E WITH CARON
   sortTable[0x011A]=5385;   // LATIN CAPITAL LETTER E WITH CARON
   sortTable[0x00EB]=5386;   // LATIN SMALL LETTER E WITH DIAERESIS
   sortTable[0x00CB]=5387;   // LATIN CAPITAL LETTER E WITH DIAERESIS
   sortTable[0x0117]=5388;   // LATIN SMALL LETTER E WITH DOT ABOVE
   sortTable[0x0116]=5389;   // LATIN CAPITAL LETTER E WITH DOT ABOVE
   sortTable[0x0119]=5390;   // LATIN SMALL LETTER E WITH OGONEK
   sortTable[0x0118]=5391;   // LATIN CAPITAL LETTER E WITH OGONEK
   sortTable[0x0113]=5392;   // LATIN SMALL LETTER E WITH MACRON
   sortTable[0x0112]=5393;   // LATIN CAPITAL LETTER E WITH MACRON
   sortTable[0x0259]=5394;   // LATIN SMALL LETTER SCHWA
   sortTable[0x018F]=5395;   // LATIN CAPITAL LETTER SCHWA
   sortTable[0x0066]=5888;   // LATIN SMALL LETTER F
   sortTable[0x0046]=5889;   // LATIN CAPITAL LETTER F
   sortTable[0x0192]=5890;   // LATIN SMALL LETTER F WITH HOOK
   sortTable[0x0191]=5891;   // LATIN CAPITAL LETTER F WITH HOOK
   sortTable[0x0067]=6912;   // LATIN SMALL LETTER G
   sortTable[0x0047]=6913;   // LATIN CAPITAL LETTER G
   sortTable[0x011F]=6914;   // LATIN SMALL LETTER G WITH BREVE
   sortTable[0x011E]=6915;   // LATIN CAPITAL LETTER G WITH BREVE
   sortTable[0x011D]=6916;   // LATIN SMALL LETTER G WITH CIRCUMFLEX
   sortTable[0x011C]=6917;   // LATIN CAPITAL LETTER G WITH CIRCUMFLEX
   sortTable[0x01E7]=6918;   // LATIN SMALL LETTER G WITH CARON
   sortTable[0x01E6]=6919;   // LATIN CAPITAL LETTER G WITH CARON
   sortTable[0x0121]=6920;   // LATIN SMALL LETTER G WITH DOT ABOVE
   sortTable[0x0120]=6921;   // LATIN CAPITAL LETTER G WITH DOT ABOVE
   sortTable[0x0123]=6922;   // LATIN SMALL LETTER G WITH CEDILLA
   sortTable[0x0122]=6923;   // LATIN CAPITAL LETTER G WITH CEDILLA
   sortTable[0x01E5]=6924;   // LATIN SMALL LETTER G WITH STROKE
   sortTable[0x01E4]=6925;   // LATIN CAPITAL LETTER G WITH STROKE
   sortTable[0x0068]=8192;   // LATIN SMALL LETTER H
   sortTable[0x0048]=8193;   // LATIN CAPITAL LETTER H
   sortTable[0x0125]=8194;   // LATIN SMALL LETTER H WITH CIRCUMFLEX
   sortTable[0x0124]=8195;   // LATIN CAPITAL LETTER H WITH CIRCUMFLEX
   sortTable[0x0127]=8196;   // LATIN SMALL LETTER H WITH STROKE
   sortTable[0x0126]=8197;   // LATIN CAPITAL LETTER H WITH STROKE
   sortTable[0x0069]=9728;   // LATIN SMALL LETTER I
   sortTable[0x0049]=9729;   // LATIN CAPITAL LETTER I
   sortTable[0x00ED]=9730;   // LATIN SMALL LETTER I WITH ACUTE
   sortTable[0x00CD]=9731;   // LATIN CAPITAL LETTER I WITH ACUTE
   sortTable[0x00EC]=9732;   // LATIN SMALL LETTER I WITH GRAVE
   sortTable[0x00CC]=9733;   // LATIN CAPITAL LETTER I WITH GRAVE
   sortTable[0x00EE]=9734;   // LATIN SMALL LETTER I WITH CIRCUMFLEX
   sortTable[0x00CE]=9735;   // LATIN CAPITAL LETTER I WITH CIRCUMFLEX
   sortTable[0x01D0]=9736;   // LATIN SMALL LETTER I WITH CARON
   sortTable[0x01CF]=9737;   // LATIN CAPITAL LETTER I WITH CARON
   sortTable[0x00EF]=9738;   // LATIN SMALL LETTER I WITH DIAERESIS
   sortTable[0x00CF]=9739;   // LATIN CAPITAL LETTER I WITH DIAERESIS
   sortTable[0x0130]=9740;   // LATIN CAPITAL LETTER I WITH DOT ABOVE
   sortTable[0x012F]=9741;   // LATIN SMALL LETTER I WITH OGONEK
   sortTable[0x012E]=9742;   // LATIN CAPITAL LETTER I WITH OGONEK
   sortTable[0x012B]=9743;   // LATIN SMALL LETTER I WITH MACRON
   sortTable[0x012A]=9744;   // LATIN CAPITAL LETTER I WITH MACRON
   sortTable[0x0131]=9745;   // LATIN SMALL LETTER DOTLESS I
   sortTable[0x0133]=9746;   // LATIN SMALL LIGATURE IJ
   sortTable[0x0132]=9747;   // LATIN CAPITAL LIGATURE IJ
   sortTable[0x006A]=9984;   // LATIN SMALL LETTER J
   sortTable[0x004A]=9985;   // LATIN CAPITAL LETTER J
   sortTable[0x0135]=9986;   // LATIN SMALL LETTER J WITH CIRCUMFLEX
   sortTable[0x0134]=9987;   // LATIN CAPITAL LETTER J WITH CIRCUMFLEX
   sortTable[0x006B]=12032;   // LATIN SMALL LETTER K
   sortTable[0x004B]=12033;   // LATIN CAPITAL LETTER K
   sortTable[0x01E9]=12034;   // LATIN SMALL LETTER K WITH CARON
   sortTable[0x01E8]=12035;   // LATIN CAPITAL LETTER K WITH CARON
   sortTable[0x0137]=12036;   // LATIN SMALL LETTER K WITH CEDILLA
   sortTable[0x0136]=12037;   // LATIN CAPITAL LETTER K WITH CEDILLA
   sortTable[0x0138]=12038;   // LATIN SMALL LETTER KRA
   sortTable[0x006C]=13056;   // LATIN SMALL LETTER L
   sortTable[0x004C]=13057;   // LATIN CAPITAL LETTER L
   sortTable[0x013A]=13058;   // LATIN SMALL LETTER L WITH ACUTE
   sortTable[0x0139]=13059;   // LATIN CAPITAL LETTER L WITH ACUTE
   sortTable[0x013E]=13060;   // LATIN SMALL LETTER L WITH CARON
   sortTable[0x013D]=13061;   // LATIN CAPITAL LETTER L WITH CARON
   sortTable[0x013C]=13062;   // LATIN SMALL LETTER L WITH CEDILLA
   sortTable[0x013B]=13063;   // LATIN CAPITAL LETTER L WITH CEDILLA
   sortTable[0x0140]=13064;   // LATIN SMALL LETTER L WITH MIDDLE DOT
   sortTable[0x013F]=13065;   // LATIN CAPITAL LETTER L WITH MIDDLE DOT
   sortTable[0x0142]=13066;   // LATIN SMALL LETTER L WITH STROKE
   sortTable[0x0141]=13067;   // LATIN CAPITAL LETTER L WITH STROKE
   sortTable[0x006D]=15104;   // LATIN SMALL LETTER M
   sortTable[0x004D]=15105;   // LATIN CAPITAL LETTER M
   sortTable[0x006E]=16384;   // LATIN SMALL LETTER N
   sortTable[0x004E]=16385;   // LATIN CAPITAL LETTER N
   sortTable[0x0144]=16386;   // LATIN SMALL LETTER N WITH ACUTE
   sortTable[0x0143]=16387;   // LATIN CAPITAL LETTER N WITH ACUTE
   sortTable[0x0148]=16388;   // LATIN SMALL LETTER N WITH CARON
   sortTable[0x0147]=16389;   // LATIN CAPITAL LETTER N WITH CARON
   sortTable[0x00F1]=16390;   // LATIN SMALL LETTER N WITH TILDE
   sortTable[0x00D1]=16391;   // LATIN CAPITAL LETTER N WITH TILDE
   sortTable[0x0146]=16392;   // LATIN SMALL LETTER N WITH CEDILLA
   sortTable[0x0145]=16393;   // LATIN CAPITAL LETTER N WITH CEDILLA
   sortTable[0x0149]=16394;   // LATIN SMALL LETTER N PRECEDED BY APOSTROPHE
   sortTable[0x014B]=16395;   // LATIN SMALL LETTER ENG
   sortTable[0x014A]=16396;   // LATIN CAPITAL LETTER ENG
   sortTable[0x006F]=17152;   // LATIN SMALL LETTER O
   sortTable[0x004F]=17153;   // LATIN CAPITAL LETTER O
   sortTable[0x00F3]=17154;   // LATIN SMALL LETTER O WITH ACUTE
   sortTable[0x00D3]=17155;   // LATIN CAPITAL LETTER O WITH ACUTE
   sortTable[0x00F2]=17156;   // LATIN SMALL LETTER O WITH GRAVE
   sortTable[0x00D2]=17157;   // LATIN CAPITAL LETTER O WITH GRAVE
   sortTable[0x00F4]=17158;   // LATIN SMALL LETTER O WITH CIRCUMFLEX
   sortTable[0x00D4]=17159;   // LATIN CAPITAL LETTER O WITH CIRCUMFLEX
   sortTable[0x01D2]=17160;   // LATIN SMALL LETTER O WITH CARON
   sortTable[0x01D1]=17161;   // LATIN CAPITAL LETTER O WITH CARON
   sortTable[0x00F6]=17162;   // LATIN SMALL LETTER O WITH DIAERESIS
   sortTable[0x00D6]=17163;   // LATIN CAPITAL LETTER O WITH DIAERESIS
   sortTable[0x0151]=17164;   // LATIN SMALL LETTER O WITH DOUBLE ACUTE
   sortTable[0x0150]=17165;   // LATIN CAPITAL LETTER O WITH DOUBLE ACUTE
   sortTable[0x00F5]=17166;   // LATIN SMALL LETTER O WITH TILDE
   sortTable[0x00D5]=17167;   // LATIN CAPITAL LETTER O WITH TILDE
   sortTable[0x014D]=17168;   // LATIN SMALL LETTER O WITH MACRON
   sortTable[0x014C]=17169;   // LATIN CAPITAL LETTER O WITH MACRON
   sortTable[0x01A1]=17170;   // LATIN SMALL LETTER O WITH HORN
   sortTable[0x01A0]=17171;   // LATIN CAPITAL LETTER O WITH HORN
   sortTable[0x00F8]=17172;   // LATIN SMALL LETTER O WITH STROKE
   sortTable[0x00D8]=17173;   // LATIN CAPITAL LETTER O WITH STROKE
   sortTable[0x0153]=17174;   // LATIN SMALL LIGATURE OE
   sortTable[0x0152]=17175;   // LATIN CAPITAL LIGATURE OE
   sortTable[0x0070]=17664;   // LATIN SMALL LETTER P
   sortTable[0x0050]=17665;   // LATIN CAPITAL LETTER P
   sortTable[0x0071]=18688;   // LATIN SMALL LETTER Q
   sortTable[0x0051]=18689;   // LATIN CAPITAL LETTER Q
   sortTable[0x0072]=19968;   // LATIN SMALL LETTER R
   sortTable[0x0052]=19969;   // LATIN CAPITAL LETTER R
   sortTable[0x0155]=19970;   // LATIN SMALL LETTER R WITH ACUTE
   sortTable[0x0154]=19971;   // LATIN CAPITAL LETTER R WITH ACUTE
   sortTable[0x0159]=19972;   // LATIN SMALL LETTER R WITH CARON
   sortTable[0x0158]=19973;   // LATIN CAPITAL LETTER R WITH CARON
   sortTable[0x0157]=19974;   // LATIN SMALL LETTER R WITH CEDILLA
   sortTable[0x0156]=19975;   // LATIN CAPITAL LETTER R WITH CEDILLA
   sortTable[0x0073]=20736;   // LATIN SMALL LETTER S
   sortTable[0x0053]=20737;   // LATIN CAPITAL LETTER S
   sortTable[0x015B]=20738;   // LATIN SMALL LETTER S WITH ACUTE
   sortTable[0x015A]=20739;   // LATIN CAPITAL LETTER S WITH ACUTE
   sortTable[0x015D]=20740;   // LATIN SMALL LETTER S WITH CIRCUMFLEX
   sortTable[0x015C]=20741;   // LATIN CAPITAL LETTER S WITH CIRCUMFLEX
   sortTable[0x0161]=20742;   // LATIN SMALL LETTER S WITH CARON
   sortTable[0x0160]=20743;   // LATIN CAPITAL LETTER S WITH CARON
   sortTable[0x015F]=20744;   // LATIN SMALL LETTER S WITH CEDILLA
   sortTable[0x015E]=20745;   // LATIN CAPITAL LETTER S WITH CEDILLA
   sortTable[0x0219]=20746;   // LATIN SMALL LETTER S WITH COMMA BELOW
   sortTable[0x0218]=20747;   // LATIN CAPITAL LETTER S WITH COMMA BELOW
   sortTable[0x00DF]=20748;   // LATIN SMALL LETTER SHARP S
   sortTable[0x0074]=22528;   // LATIN SMALL LETTER T
   sortTable[0x0054]=22529;   // LATIN CAPITAL LETTER T
   sortTable[0x0165]=22530;   // LATIN SMALL LETTER T WITH CARON
   sortTable[0x0164]=22531;   // LATIN CAPITAL LETTER T WITH CARON
   sortTable[0x0162]=22532;   // LATIN CAPITAL LETTER T WITH CEDILLA
   sortTable[0x0163]=22533;   // LATIN SMALL LETTER T WITH CEDILLA
   sortTable[0x021B]=22534;   // LATIN SMALL LETTER T WITH COMMA BELOW
   sortTable[0x021A]=22535;   // LATIN CAPITAL LETTER T WITH COMMA BELOW
   sortTable[0x0167]=22536;   // LATIN SMALL LETTER T WITH STROKE
   sortTable[0x0166]=22537;   // LATIN CAPITAL LETTER T WITH STROKE
   sortTable[0x0075]=23808;   // LATIN SMALL LETTER U
   sortTable[0x0055]=23809;   // LATIN CAPITAL LETTER U
   sortTable[0x00FA]=23810;   // LATIN SMALL LETTER U WITH ACUTE
   sortTable[0x00DA]=23811;   // LATIN CAPITAL LETTER U WITH ACUTE
   sortTable[0x00F9]=23812;   // LATIN SMALL LETTER U WITH GRAVE
   sortTable[0x00D9]=23813;   // LATIN CAPITAL LETTER U WITH GRAVE
   sortTable[0x016D]=23814;   // LATIN SMALL LETTER U WITH BREVE
   sortTable[0x016C]=23815;   // LATIN CAPITAL LETTER U WITH BREVE
   sortTable[0x00FB]=23816;   // LATIN SMALL LETTER U WITH CIRCUMFLEX
   sortTable[0x00DB]=23817;   // LATIN CAPITAL LETTER U WITH CIRCUMFLEX
   sortTable[0x01D4]=23818;   // LATIN SMALL LETTER U WITH CARON
   sortTable[0x01D3]=23819;   // LATIN CAPITAL LETTER U WITH CARON
   sortTable[0x016F]=23820;   // LATIN SMALL LETTER U WITH RING ABOVE
   sortTable[0x016E]=23821;   // LATIN CAPITAL LETTER U WITH RING ABOVE
   sortTable[0x00FC]=23822;   // LATIN SMALL LETTER U WITH DIAERESIS
   sortTable[0x00DC]=23823;   // LATIN CAPITAL LETTER U WITH DIAERESIS
   sortTable[0x0171]=23824;   // LATIN SMALL LETTER U WITH DOUBLE ACUTE
   sortTable[0x0170]=23825;   // LATIN CAPITAL LETTER U WITH DOUBLE ACUTE
   sortTable[0x0173]=23826;   // LATIN SMALL LETTER U WITH OGONEK
   sortTable[0x0172]=23827;   // LATIN CAPITAL LETTER U WITH OGONEK
   sortTable[0x016B]=23828;   // LATIN SMALL LETTER U WITH MACRON
   sortTable[0x016A]=23829;   // LATIN CAPITAL LETTER U WITH MACRON
   sortTable[0x01B0]=23830;   // LATIN SMALL LETTER U WITH HORN
   sortTable[0x01AF]=23831;   // LATIN CAPITAL LETTER U WITH HORN
   sortTable[0x01D8]=23832;   // LATIN SMALL LETTER U WITH DIAERESIS AND ACUTE
   sortTable[0x01D7]=23833;   // LATIN CAPITAL LETTER U WITH DIAERESIS AND ACUTE
   sortTable[0x01DC]=23834;   // LATIN SMALL LETTER U WITH DIAERESIS AND GRAVE
   sortTable[0x01DB]=23835;   // LATIN CAPITAL LETTER U WITH DIAERESIS AND GRAVE
   sortTable[0x01DA]=23836;   // LATIN SMALL LETTER U WITH DIAERESIS AND CARON
   sortTable[0x01D9]=23837;   // LATIN CAPITAL LETTER U WITH DIAERESIS AND CARON
   sortTable[0x01D6]=23838;   // LATIN SMALL LETTER U WITH DIAERESIS AND MACRON
   sortTable[0x01D5]=23839;   // LATIN CAPITAL LETTER U WITH DIAERESIS AND MACRON
   sortTable[0x0076]=24064;   // LATIN SMALL LETTER V
   sortTable[0x0056]=24065;   // LATIN CAPITAL LETTER V
   sortTable[0x0077]=24320;   // LATIN SMALL LETTER W
   sortTable[0x0057]=24321;   // LATIN CAPITAL LETTER W
   sortTable[0x1E83]=24322;   // LATIN SMALL LETTER W WITH ACUTE
   sortTable[0x1E82]=24323;   // LATIN CAPITAL LETTER W WITH ACUTE
   sortTable[0x1E81]=24324;   // LATIN SMALL LETTER W WITH GRAVE
   sortTable[0x1E80]=24325;   // LATIN CAPITAL LETTER W WITH GRAVE
   sortTable[0x0175]=24326;   // LATIN SMALL LETTER W WITH CIRCUMFLEX
   sortTable[0x0174]=24327;   // LATIN CAPITAL LETTER W WITH CIRCUMFLEX
   sortTable[0x1E85]=24328;   // LATIN SMALL LETTER W WITH DIAERESIS
   sortTable[0x1E84]=24329;   // LATIN CAPITAL LETTER W WITH DIAERESIS
   sortTable[0x0078]=25344;   // LATIN SMALL LETTER X
   sortTable[0x0058]=25345;   // LATIN CAPITAL LETTER X
   sortTable[0x0079]=27136;   // LATIN SMALL LETTER Y
   sortTable[0x0059]=27137;   // LATIN CAPITAL LETTER Y
   sortTable[0x00FD]=27138;   // LATIN SMALL LETTER Y WITH ACUTE
   sortTable[0x00DD]=27139;   // LATIN CAPITAL LETTER Y WITH ACUTE
   sortTable[0x1EF3]=27140;   // LATIN SMALL LETTER Y WITH GRAVE
   sortTable[0x1EF2]=27141;   // LATIN CAPITAL LETTER Y WITH GRAVE
   sortTable[0x0177]=27142;   // LATIN SMALL LETTER Y WITH CIRCUMFLEX
   sortTable[0x0176]=27143;   // LATIN CAPITAL LETTER Y WITH CIRCUMFLEX
   sortTable[0x00FF]=27144;   // LATIN SMALL LETTER Y WITH DIAERESIS
   sortTable[0x0178]=27145;   // LATIN CAPITAL LETTER Y WITH DIAERESIS
   sortTable[0x007A]=29440;   // LATIN SMALL LETTER Z
   sortTable[0x005A]=29441;   // LATIN CAPITAL LETTER Z
   sortTable[0x017A]=29442;   // LATIN SMALL LETTER Z WITH ACUTE
   sortTable[0x0179]=29443;   // LATIN CAPITAL LETTER Z WITH ACUTE
   sortTable[0x017E]=29444;   // LATIN SMALL LETTER Z WITH CARON
   sortTable[0x017D]=29445;   // LATIN CAPITAL LETTER Z WITH CARON
   sortTable[0x017C]=29446;   // LATIN SMALL LETTER Z WITH DOT ABOVE
   sortTable[0x017B]=29447;   // LATIN CAPITAL LETTER Z WITH DOT ABOVE
   sortTable[0x0292]=29448;   // LATIN SMALL LETTER EZH
   sortTable[0x01B7]=29449;   // LATIN CAPITAL LETTER EZH
   sortTable[0x01EF]=29450;   // LATIN SMALL LETTER EZH WITH CARON
   sortTable[0x01EE]=29451;   // LATIN CAPITAL LETTER EZH WITH CARON
   sortTable[0x00FE]=29452;   // LATIN SMALL LETTER THORN
   sortTable[0x00DE]=29453;   // LATIN CAPITAL LETTER THORN
   sortTable[0x03B1]=31744;   // GREEK SMALL LETTER ALPHA
   sortTable[0x0391]=31745;   // GREEK CAPITAL LETTER ALPHA
   sortTable[0x03AC]=31746;   // GREEK SMALL LETTER ALPHA WITH TONOS
   sortTable[0x0386]=31747;   // GREEK CAPITAL LETTER ALPHA WITH TONOS
   sortTable[0x03B2]=31748;   // GREEK SMALL LETTER BETA
   sortTable[0x0392]=31749;   // GREEK CAPITAL LETTER BETA
   sortTable[0x03B3]=31750;   // GREEK SMALL LETTER GAMMA
   sortTable[0x0393]=31751;   // GREEK CAPITAL LETTER GAMMA
   sortTable[0x03B4]=31752;   // GREEK SMALL LETTER DELTA
   sortTable[0x0394]=31753;   // GREEK CAPITAL LETTER DELTA
   sortTable[0x03B5]=31754;   // GREEK SMALL LETTER EPSILON
   sortTable[0x0395]=31755;   // GREEK CAPITAL LETTER EPSILON
   sortTable[0x03AD]=31756;   // GREEK SMALL LETTER EPSILON WITH TONOS
   sortTable[0x0388]=31757;   // GREEK CAPITAL LETTER EPSILON WITH TONOS
   sortTable[0x03B6]=31758;   // GREEK SMALL LETTER ZETA
   sortTable[0x0396]=31759;   // GREEK CAPITAL LETTER ZETA
   sortTable[0x03B7]=31760;   // GREEK SMALL LETTER ETA
   sortTable[0x0397]=31761;   // GREEK CAPITAL LETTER ETA
   sortTable[0x03AE]=31762;   // GREEK SMALL LETTER ETA WITH TONOS
   sortTable[0x0389]=31763;   // GREEK CAPITAL LETTER ETA WITH TONOS
   sortTable[0x03B8]=31764;   // GREEK SMALL LETTER THETA
   sortTable[0x0398]=31765;   // GREEK CAPITAL LETTER THETA
   sortTable[0x03B9]=31766;   // GREEK SMALL LETTER IOTA
   sortTable[0x0399]=31767;   // GREEK CAPITAL LETTER IOTA
   sortTable[0x03AF]=31768;   // GREEK SMALL LETTER IOTA WITH TONOS
   sortTable[0x038A]=31769;   // GREEK CAPITAL LETTER IOTA WITH TONOS
   sortTable[0x03CA]=31770;   // GREEK SMALL LETTER IOTA WITH DIALYTICA
   sortTable[0x03AA]=31771;   // GREEK CAPITAL LETTER IOTA WITH DIALYTICA
   sortTable[0x0390]=31772;   // GREEK SMALL LETTER IOTA WITH DIALYTICA AND TONOS
   sortTable[0x03BA]=31773;   // GREEK SMALL LETTER KAPPA
   sortTable[0x039A]=31774;   // GREEK CAPITAL LETTER KAPPA
   sortTable[0x03BB]=31775;   // GREEK SMALL LETTER LAMDA
   sortTable[0x039B]=31776;   // GREEK CAPITAL LETTER LAMDA
   sortTable[0x03BC]=31777;   // GREEK SMALL LETTER MU
   sortTable[0x039C]=31778;   // GREEK CAPITAL LETTER MU
   sortTable[0x03BD]=31779;   // GREEK SMALL LETTER NU
   sortTable[0x039D]=31780;   // GREEK CAPITAL LETTER NU
   sortTable[0x03BE]=31781;   // GREEK SMALL LETTER XI
   sortTable[0x039E]=31782;   // GREEK CAPITAL LETTER XI
   sortTable[0x03BF]=31783;   // GREEK SMALL LETTER OMICRON
   sortTable[0x039F]=31784;   // GREEK CAPITAL LETTER OMICRON
   sortTable[0x03CC]=31785;   // GREEK SMALL LETTER OMICRON WITH TONOS
   sortTable[0x038C]=31786;   // GREEK CAPITAL LETTER OMICRON WITH TONOS
   sortTable[0x03C0]=31787;   // GREEK SMALL LETTER PI
   sortTable[0x03A0]=31788;   // GREEK CAPITAL LETTER PI
   sortTable[0x03C1]=31789;   // GREEK SMALL LETTER RHO
   sortTable[0x03A1]=31790;   // GREEK CAPITAL LETTER RHO
   sortTable[0x03C3]=31791;   // GREEK SMALL LETTER SIGMA
   sortTable[0x03C2]=31792;   // GREEK SMALL LETTER FINAL SIGMA
   sortTable[0x03A3]=31793;   // GREEK CAPITAL LETTER SIGMA
   sortTable[0x03C4]=31794;   // GREEK SMALL LETTER TAU
   sortTable[0x03A4]=31795;   // GREEK CAPITAL LETTER TAU
   sortTable[0x03C5]=31796;   // GREEK SMALL LETTER UPSILON
   sortTable[0x03A5]=31797;   // GREEK CAPITAL LETTER UPSILON
   sortTable[0x03CD]=31798;   // GREEK SMALL LETTER UPSILON WITH TONOS
   sortTable[0x038E]=31799;   // GREEK CAPITAL LETTER UPSILON WITH TONOS
   sortTable[0x03CB]=31800;   // GREEK SMALL LETTER UPSILON WITH DIALYTICA
   sortTable[0x03AB]=31801;   // GREEK CAPITAL LETTER UPSILON WITH DIALYTICA
   sortTable[0x03B0]=31802;   // GREEK SMALL LETTER UPSILON WITH DIALYTICA AND TONOS
   sortTable[0x03C6]=31803;   // GREEK SMALL LETTER PHI
   sortTable[0x03A6]=31804;   // GREEK CAPITAL LETTER PHI
   sortTable[0x03C7]=31805;   // GREEK SMALL LETTER CHI
   sortTable[0x03A7]=31806;   // GREEK CAPITAL LETTER CHI
   sortTable[0x03C8]=31807;   // GREEK SMALL LETTER PSI
   sortTable[0x03A8]=31808;   // GREEK CAPITAL LETTER PSI
   sortTable[0x03C9]=31809;   // GREEK SMALL LETTER OMEGA
   sortTable[0x03A9]=31810;   // GREEK CAPITAL LETTER OMEGA
   sortTable[0x03CE]=31811;   // GREEK SMALL LETTER OMEGA WITH TONOS
   sortTable[0x038F]=31812;   // GREEK CAPITAL LETTER OMEGA WITH TONOS
   sortTable[0x0430]=32000;   // CYRILLIC SMALL LETTER A
   sortTable[0x0410]=32001;   // CYRILLIC CAPITAL LETTER A
   sortTable[0x0431]=32002;   // CYRILLIC SMALL LETTER BE
   sortTable[0x0411]=32003;   // CYRILLIC CAPITAL LETTER BE
   sortTable[0x0432]=32004;   // CYRILLIC SMALL LETTER VE
   sortTable[0x0412]=32005;   // CYRILLIC CAPITAL LETTER VE
   sortTable[0x0433]=32006;   // CYRILLIC SMALL LETTER GHE
   sortTable[0x0413]=32007;   // CYRILLIC CAPITAL LETTER GHE
   sortTable[0x0491]=32008;   // CYRILLIC SMALL LETTER GHE WITH UPTURN
   sortTable[0x0490]=32009;   // CYRILLIC CAPITAL LETTER GHE WITH UPTURN
   sortTable[0x0434]=32010;   // CYRILLIC SMALL LETTER DE
   sortTable[0x0414]=32011;   // CYRILLIC CAPITAL LETTER DE
   sortTable[0x0452]=32012;   // CYRILLIC SMALL LETTER DJE
   sortTable[0x0402]=32013;   // CYRILLIC CAPITAL LETTER DJE
   sortTable[0x0453]=32014;   // CYRILLIC SMALL LETTER GJE
   sortTable[0x0403]=32015;   // CYRILLIC CAPITAL LETTER GJE
   sortTable[0x0435]=32016;   // CYRILLIC SMALL LETTER IE
   sortTable[0x0415]=32017;   // CYRILLIC CAPITAL LETTER IE
   sortTable[0x0451]=32018;   // CYRILLIC SMALL LETTER IO
   sortTable[0x0401]=32019;   // CYRILLIC CAPITAL LETTER IO
   sortTable[0x0454]=32020;   // CYRILLIC SMALL LETTER UKRAINIAN IE
   sortTable[0x0404]=32021;   // CYRILLIC CAPITAL LETTER UKRAINIAN IE
   sortTable[0x0436]=32022;   // CYRILLIC SMALL LETTER ZHE
   sortTable[0x0416]=32023;   // CYRILLIC CAPITAL LETTER ZHE
   sortTable[0x0437]=32024;   // CYRILLIC SMALL LETTER ZE
   sortTable[0x0417]=32025;   // CYRILLIC CAPITAL LETTER ZE
   sortTable[0x0455]=32026;   // CYRILLIC SMALL LETTER DZE
   sortTable[0x0405]=32027;   // CYRILLIC CAPITAL LETTER DZE
   sortTable[0x0438]=32028;   // CYRILLIC SMALL LETTER I
   sortTable[0x0418]=32029;   // CYRILLIC CAPITAL LETTER I
   sortTable[0x0456]=32030;   // CYRILLIC SMALL LETTER BELORUSIAN-UKRAINIAN I
   sortTable[0x0406]=32031;   // CYRILLIC CAPITAL LETTER BELORUSIAN-UKRAINIAN I
   sortTable[0x0457]=32032;   // CYRILLIC SMALL LETTER YI
   sortTable[0x0407]=32033;   // CYRILLIC CAPITAL LETTER YI
   sortTable[0x0439]=32034;   // CYRILLIC SMALL LETTER SHORT I
   sortTable[0x0419]=32035;   // CYRILLIC CAPITAL LETTER SHORT I
   sortTable[0x0458]=32036;   // CYRILLIC SMALL LETTER JE
   sortTable[0x0408]=32037;   // CYRILLIC CAPITAL LETTER JE
   sortTable[0x043A]=32038;   // CYRILLIC SMALL LETTER KA
   sortTable[0x041A]=32039;   // CYRILLIC CAPITAL LETTER KA
   sortTable[0x043B]=32040;   // CYRILLIC SMALL LETTER EL
   sortTable[0x041B]=32041;   // CYRILLIC CAPITAL LETTER EL
   sortTable[0x0459]=32042;   // CYRILLIC SMALL LETTER LJE
   sortTable[0x0409]=32043;   // CYRILLIC CAPITAL LETTER LJE
   sortTable[0x043C]=32044;   // CYRILLIC SMALL LETTER EM
   sortTable[0x041C]=32045;   // CYRILLIC CAPITAL LETTER EM
   sortTable[0x043D]=32046;   // CYRILLIC SMALL LETTER EN
   sortTable[0x041D]=32047;   // CYRILLIC CAPITAL LETTER EN
   sortTable[0x045A]=32048;   // CYRILLIC SMALL LETTER NJE
   sortTable[0x040A]=32049;   // CYRILLIC CAPITAL LETTER NJE
   sortTable[0x043E]=32050;   // CYRILLIC SMALL LETTER O
   sortTable[0x041E]=32051;   // CYRILLIC CAPITAL LETTER O
   sortTable[0x043F]=32052;   // CYRILLIC SMALL LETTER PE
   sortTable[0x041F]=32053;   // CYRILLIC CAPITAL LETTER PE
   sortTable[0x0440]=32054;   // CYRILLIC SMALL LETTER ER
   sortTable[0x0420]=32055;   // CYRILLIC CAPITAL LETTER ER
   sortTable[0x0441]=32056;   // CYRILLIC SMALL LETTER ES
   sortTable[0x0421]=32057;   // CYRILLIC CAPITAL LETTER ES
   sortTable[0x0442]=32058;   // CYRILLIC SMALL LETTER TE
   sortTable[0x0422]=32059;   // CYRILLIC CAPITAL LETTER TE
   sortTable[0x045B]=32060;   // CYRILLIC SMALL LETTER TSHE
   sortTable[0x040B]=32061;   // CYRILLIC CAPITAL LETTER TSHE
   sortTable[0x045C]=32062;   // CYRILLIC SMALL LETTER KJE
   sortTable[0x040C]=32063;   // CYRILLIC CAPITAL LETTER KJE
   sortTable[0x0443]=32064;   // CYRILLIC SMALL LETTER U
   sortTable[0x0423]=32065;   // CYRILLIC CAPITAL LETTER U
   sortTable[0x045E]=32066;   // CYRILLIC SMALL LETTER SHORT U
   sortTable[0x040E]=32067;   // CYRILLIC CAPITAL LETTER SHORT U
   sortTable[0x0444]=32068;   // CYRILLIC SMALL LETTER EF
   sortTable[0x0424]=32069;   // CYRILLIC CAPITAL LETTER EF
   sortTable[0x0445]=32070;   // CYRILLIC SMALL LETTER HA
   sortTable[0x0425]=32071;   // CYRILLIC CAPITAL LETTER HA
   sortTable[0x0446]=32072;   // CYRILLIC SMALL LETTER TSE
   sortTable[0x0426]=32073;   // CYRILLIC CAPITAL LETTER TSE
   sortTable[0x0447]=32074;   // CYRILLIC SMALL LETTER CHE
   sortTable[0x0427]=32075;   // CYRILLIC CAPITAL LETTER CHE
   sortTable[0x045F]=32076;   // CYRILLIC SMALL LETTER DZHE
   sortTable[0x040F]=32077;   // CYRILLIC CAPITAL LETTER DZHE
   sortTable[0x0448]=32078;   // CYRILLIC SMALL LETTER SHA
   sortTable[0x0428]=32079;   // CYRILLIC CAPITAL LETTER SHA
   sortTable[0x0449]=32080;   // CYRILLIC SMALL LETTER SHCHA
   sortTable[0x0429]=32081;   // CYRILLIC CAPITAL LETTER SHCHA
   sortTable[0x044A]=32082;   // CYRILLIC SMALL LETTER HARD SIGN
   sortTable[0x042A]=32083;   // CYRILLIC CAPITAL LETTER HARD SIGN
   sortTable[0x044B]=32084;   // CYRILLIC SMALL LETTER YERU
   sortTable[0x042B]=32085;   // CYRILLIC CAPITAL LETTER YERU
   sortTable[0x044C]=32086;   // CYRILLIC SMALL LETTER SOFT SIGN
   sortTable[0x042C]=32087;   // CYRILLIC CAPITAL LETTER SOFT SIGN
   sortTable[0x044D]=32088;   // CYRILLIC SMALL LETTER E
   sortTable[0x042D]=32089;   // CYRILLIC CAPITAL LETTER E
   sortTable[0x044E]=32090;   // CYRILLIC SMALL LETTER YU
   sortTable[0x042E]=32091;   // CYRILLIC CAPITAL LETTER YU
   sortTable[0x044F]=32092;   // CYRILLIC SMALL LETTER YA
   sortTable[0x042F]=32093;   // CYRILLIC CAPITAL LETTER YA
   sortTable[0x05D0]=32256;   // HEBREW LETTER ALEF
   sortTable[0x05D1]=32257;   // HEBREW LETTER BET
   sortTable[0x05D2]=32258;   // HEBREW LETTER GIMEL
   sortTable[0x05D3]=32259;   // HEBREW LETTER DALET
   sortTable[0x05D4]=32260;   // HEBREW LETTER HE
   sortTable[0x05D5]=32261;   // HEBREW LETTER VAV
   sortTable[0x05D6]=32262;   // HEBREW LETTER ZAYIN
   sortTable[0x05D7]=32263;   // HEBREW LETTER HET
   sortTable[0x05D8]=32264;   // HEBREW LETTER TET
   sortTable[0x05D9]=32265;   // HEBREW LETTER YOD
   sortTable[0x05DA]=32266;   // HEBREW LETTER FINAL KAF
   sortTable[0x05DB]=32267;   // HEBREW LETTER KAF
   sortTable[0x05DC]=32268;   // HEBREW LETTER LAMED
   sortTable[0x05DD]=32269;   // HEBREW LETTER FINAL MEM
   sortTable[0x05DE]=32270;   // HEBREW LETTER MEM
   sortTable[0x05DF]=32271;   // HEBREW LETTER FINAL NUN
   sortTable[0x05E0]=32272;   // HEBREW LETTER NUN
   sortTable[0x05E1]=32273;   // HEBREW LETTER SAMEKH
   sortTable[0x05E2]=32274;   // HEBREW LETTER AYIN
   sortTable[0x05E3]=32275;   // HEBREW LETTER FINAL PE
   sortTable[0x05E4]=32276;   // HEBREW LETTER PE
   sortTable[0x05E5]=32277;   // HEBREW LETTER FINAL TSADI
   sortTable[0x05E6]=32278;   // HEBREW LETTER TSADI
   sortTable[0x05E7]=32279;   // HEBREW LETTER QOF
   sortTable[0x05E8]=32280;   // HEBREW LETTER RESH
   sortTable[0x05E9]=32281;   // HEBREW LETTER SHIN
   sortTable[0x05EA]=32282;   // HEBREW LETTER TAV
   sortTable[0x0660]=32512;   // ARABIC-INDIC DIGIT ZERO
   sortTable[0x0661]=32513;   // ARABIC-INDIC DIGIT ONE
   sortTable[0x0662]=32514;   // ARABIC-INDIC DIGIT TWO
   sortTable[0x0663]=32515;   // ARABIC-INDIC DIGIT THREE
   sortTable[0x0664]=32516;   // ARABIC-INDIC DIGIT FOUR
   sortTable[0x0665]=32517;   // ARABIC-INDIC DIGIT FIVE
   sortTable[0x0666]=32518;   // ARABIC-INDIC DIGIT SIX
   sortTable[0x0667]=32519;   // ARABIC-INDIC DIGIT SEVEN
   sortTable[0x0668]=32520;   // ARABIC-INDIC DIGIT EIGHT
   sortTable[0x0669]=32521;   // ARABIC-INDIC DIGIT NINE
   sortTable[0x0621]=32522;   // ARABIC LETTER HAMZA
   sortTable[0x0623]=32523;   // ARABIC LETTER ALEF WITH HAMZA ABOVE
   sortTable[0x0624]=32524;   // ARABIC LETTER WAW WITH HAMZA ABOVE
   sortTable[0x0625]=32525;   // ARABIC LETTER ALEF WITH HAMZA BELOW
   sortTable[0x0626]=32526;   // ARABIC LETTER YEH WITH HAMZA ABOVE
   sortTable[0x0622]=32527;   // ARABIC LETTER ALEF WITH MADDA ABOVE
   sortTable[0x0627]=32528;   // ARABIC LETTER ALEF
   sortTable[0x0628]=32529;   // ARABIC LETTER BEH
   sortTable[0x062A]=32530;   // ARABIC LETTER TEH
   sortTable[0x0629]=32531;   // ARABIC LETTER TEH MARBUTA
   sortTable[0x062B]=32532;   // ARABIC LETTER THEH
   sortTable[0x062C]=32533;   // ARABIC LETTER JEEM
   sortTable[0x062D]=32534;   // ARABIC LETTER HAH
   sortTable[0x062E]=32535;   // ARABIC LETTER KHAH
   sortTable[0x062F]=32536;   // ARABIC LETTER DAL
   sortTable[0x0630]=32537;   // ARABIC LETTER THAL
   sortTable[0x0631]=32538;   // ARABIC LETTER REH
   sortTable[0x0632]=32539;   // ARABIC LETTER ZAIN
   sortTable[0x0633]=32540;   // ARABIC LETTER SEEN
   sortTable[0x0634]=32541;   // ARABIC LETTER SHEEN
   sortTable[0x0635]=32542;   // ARABIC LETTER SAD
   sortTable[0x0636]=32543;   // ARABIC LETTER DAD
   sortTable[0x0637]=32544;   // ARABIC LETTER TAH
   sortTable[0x0638]=32545;   // ARABIC LETTER ZAH
   sortTable[0x0639]=32546;   // ARABIC LETTER AIN
   sortTable[0x063A]=32547;   // ARABIC LETTER GHAIN
   sortTable[0x0641]=32548;   // ARABIC LETTER FEH
   sortTable[0x0642]=32549;   // ARABIC LETTER QAF
   sortTable[0x0643]=32550;   // ARABIC LETTER KAF
   sortTable[0x0644]=32551;   // ARABIC LETTER LAM
   sortTable[0x0645]=32552;   // ARABIC LETTER MEEM
   sortTable[0x0646]=32553;   // ARABIC LETTER NOON
   sortTable[0x0647]=32554;   // ARABIC LETTER HEH
   sortTable[0x0648]=32555;   // ARABIC LETTER WAW
   sortTable[0x0649]=32556;   // ARABIC LETTER ALEF MAKSURA
   sortTable[0x064A]=32557;   // ARABIC LETTER YEH
   sortTable[0x30A0]=256;   // KATAKANA-HIRAGANA DOUBLE HYPHEN
   sortTable[0x3041]=257;   // HIRAGANA LETTER SMALL A
   sortTable[0x30A1]=257;   // KATAKANA LETTER SMALL A
   sortTable[0xFF67]=257;   // HALFWIDTH KATAKANA LETTER SMALL A
   sortTable[0x3042]=257;   // HIRAGANA LETTER A
   sortTable[0x30A2]=257;   // KATAKANA LETTER A
   sortTable[0xFF71]=257;   // HALFWIDTH KATAKANA LETTER A
   sortTable[0x3043]=258;   // HIRAGANA LETTER SMALL I
   sortTable[0x30A3]=258;   // KATAKANA LETTER SMALL I
   sortTable[0xFF68]=258;   // HALFWIDTH KATAKANA LETTER SMALL I
   sortTable[0x3044]=258;   // HIRAGANA LETTER I
   sortTable[0x30A4]=258;   // KATAKANA LETTER I
   sortTable[0xFF72]=258;   // HALFWIDTH KATAKANA LETTER I
   sortTable[0x3045]=259;   // HIRAGANA LETTER SMALL U
   sortTable[0x30A5]=259;   // KATAKANA LETTER SMALL U
   sortTable[0xFF69]=259;   // HALFWIDTH KATAKANA LETTER SMALL U
   sortTable[0x3046]=259;   // HIRAGANA LETTER U
   sortTable[0x30A6]=259;   // KATAKANA LETTER U
   sortTable[0xFF73]=259;   // HALFWIDTH KATAKANA LETTER U
   sortTable[0x3094]=259;   // HIRAGANA LETTER VU
   sortTable[0x30F4]=259;   // KATAKANA LETTER VU
   sortTable[0x3047]=260;   // HIRAGANA LETTER SMALL E
   sortTable[0x30A7]=260;   // KATAKANA LETTER SMALL E
   sortTable[0xFF6A]=260;   // HALFWIDTH KATAKANA LETTER SMALL E
   sortTable[0x3048]=260;   // HIRAGANA LETTER E
   sortTable[0x30A8]=260;   // KATAKANA LETTER E
   sortTable[0xFF74]=260;   // HALFWIDTH KATAKANA LETTER E
   sortTable[0x3049]=261;   // HIRAGANA LETTER SMALL O
   sortTable[0x30A9]=261;   // KATAKANA LETTER SMALL O
   sortTable[0xFF6B]=261;   // HALFWIDTH KATAKANA LETTER SMALL O
   sortTable[0x304A]=261;   // HIRAGANA LETTER O
   sortTable[0x30AA]=261;   // KATAKANA LETTER O
   sortTable[0xFF75]=261;   // HALFWIDTH KATAKANA LETTER O
   sortTable[0x3095]=262;   // HIRAGANA LETTER SMALL KA
   sortTable[0x30F5]=262;   // KATAKANA LETTER SMALL KA
   sortTable[0x3096]=262;   // HIRAGANA LETTER SMALL KE
   sortTable[0x30F6]=262;   // KATAKANA LETTER SMALL KE
   sortTable[0x308E]=262;   // HIRAGANA LETTER SMALL WA
   sortTable[0x30EE]=262;   // KATAKANA LETTER SMALL WA
   sortTable[0x304B]=262;   // HIRAGANA LETTER KA
   sortTable[0x30AB]=262;   // KATAKANA LETTER KA
   sortTable[0xFF76]=262;   // HALFWIDTH KATAKANA LETTER KA
   sortTable[0x304C]=262;   // HIRAGANA LETTER GA
   sortTable[0x30AC]=262;   // KATAKANA LETTER GA
   sortTable[0x304D]=263;   // HIRAGANA LETTER KI
   sortTable[0x30AD]=263;   // KATAKANA LETTER KI
   sortTable[0xFF77]=263;   // HALFWIDTH KATAKANA LETTER KI
   sortTable[0x304E]=263;   // HIRAGANA LETTER GI
   sortTable[0x30AE]=263;   // KATAKANA LETTER GI
   sortTable[0x304F]=264;   // HIRAGANA LETTER KU
   sortTable[0x30AF]=264;   // KATAKANA LETTER KU
   sortTable[0xFF78]=264;   // HALFWIDTH KATAKANA LETTER KU
   sortTable[0x3050]=264;   // HIRAGANA LETTER GU
   sortTable[0x30B0]=264;   // KATAKANA LETTER GU
   sortTable[0x3051]=265;   // HIRAGANA LETTER KE
   sortTable[0x30B1]=265;   // KATAKANA LETTER KE
   sortTable[0xFF79]=265;   // HALFWIDTH KATAKANA LETTER KE
   sortTable[0x3052]=265;   // HIRAGANA LETTER GE
   sortTable[0x30B2]=265;   // KATAKANA LETTER GE
   sortTable[0x3053]=266;   // HIRAGANA LETTER KO
   sortTable[0x30B3]=266;   // KATAKANA LETTER KO
   sortTable[0xFF7A]=266;   // HALFWIDTH KATAKANA LETTER KO
   sortTable[0x3054]=266;   // HIRAGANA LETTER GO
   sortTable[0x30B4]=266;   // KATAKANA LETTER GO
   sortTable[0x3055]=267;   // HIRAGANA LETTER SA
   sortTable[0x30B5]=267;   // KATAKANA LETTER SA
   sortTable[0xFF7B]=267;   // HALFWIDTH KATAKANA LETTER SA
   sortTable[0x3056]=267;   // HIRAGANA LETTER ZA
   sortTable[0x30B6]=267;   // KATAKANA LETTER ZA
   sortTable[0x3057]=268;   // HIRAGANA LETTER SI
   sortTable[0x30B7]=268;   // KATAKANA LETTER SI
   sortTable[0xFF7C]=268;   // HALFWIDTH KATAKANA LETTER SI
   sortTable[0x3058]=268;   // HIRAGANA LETTER ZI
   sortTable[0x30B8]=268;   // KATAKANA LETTER ZI
   sortTable[0x3059]=269;   // HIRAGANA LETTER SU
   sortTable[0x30B9]=269;   // KATAKANA LETTER SU
   sortTable[0xFF7D]=269;   // HALFWIDTH KATAKANA LETTER SU
   sortTable[0x305A]=269;   // HIRAGANA LETTER ZU
   sortTable[0x30BA]=269;   // KATAKANA LETTER ZU
   sortTable[0x305B]=270;   // HIRAGANA LETTER SE
   sortTable[0x30BB]=270;   // KATAKANA LETTER SE
   sortTable[0xFF7E]=270;   // HALFWIDTH KATAKANA LETTER SE
   sortTable[0x305C]=270;   // HIRAGANA LETTER ZE
   sortTable[0x30BC]=270;   // KATAKANA LETTER ZE
   sortTable[0x305D]=271;   // HIRAGANA LETTER SO
   sortTable[0x30BD]=271;   // KATAKANA LETTER SO
   sortTable[0xFF7F]=271;   // HALFWIDTH KATAKANA LETTER SO
   sortTable[0x305E]=271;   // HIRAGANA LETTER ZO
   sortTable[0x30BE]=271;   // KATAKANA LETTER ZO
   sortTable[0x305F]=272;   // HIRAGANA LETTER TA
   sortTable[0x30BF]=272;   // KATAKANA LETTER TA
   sortTable[0xFF80]=272;   // HALFWIDTH KATAKANA LETTER TA
   sortTable[0x3060]=272;   // HIRAGANA LETTER DA
   sortTable[0x30C0]=272;   // KATAKANA LETTER DA
   sortTable[0x3061]=273;   // HIRAGANA LETTER TI
   sortTable[0x30C1]=273;   // KATAKANA LETTER TI
   sortTable[0xFF81]=273;   // HALFWIDTH KATAKANA LETTER TI
   sortTable[0x3062]=273;   // HIRAGANA LETTER DI
   sortTable[0x30C2]=273;   // KATAKANA LETTER DI
   sortTable[0x3063]=274;   // HIRAGANA LETTER SMALL TU
   sortTable[0x30C3]=274;   // KATAKANA LETTER SMALL TU
   sortTable[0xFF6F]=274;   // HALFWIDTH KATAKANA LETTER SMALL TU
   sortTable[0x3064]=274;   // HIRAGANA LETTER TU
   sortTable[0x30C4]=274;   // KATAKANA LETTER TU
   sortTable[0xFF82]=274;   // HALFWIDTH KATAKANA LETTER TU
   sortTable[0x3065]=274;   // HIRAGANA LETTER DU
   sortTable[0x30C5]=274;   // KATAKANA LETTER DU
   sortTable[0x3066]=275;   // HIRAGANA LETTER TE
   sortTable[0x30C6]=275;   // KATAKANA LETTER TE
   sortTable[0xFF83]=275;   // HALFWIDTH KATAKANA LETTER TE
   sortTable[0x3067]=275;   // HIRAGANA LETTER DE
   sortTable[0x30C7]=275;   // KATAKANA LETTER DE
   sortTable[0x3068]=276;   // HIRAGANA LETTER TO
   sortTable[0x30C8]=276;   // KATAKANA LETTER TO
   sortTable[0xFF84]=276;   // HALFWIDTH KATAKANA LETTER TO
   sortTable[0x3069]=276;   // HIRAGANA LETTER DO
   sortTable[0x30C9]=276;   // KATAKANA LETTER DO
   sortTable[0x306A]=277;   // HIRAGANA LETTER NA
   sortTable[0x30CA]=277;   // KATAKANA LETTER NA
   sortTable[0xFF85]=277;   // HALFWIDTH KATAKANA LETTER NA
   sortTable[0x306B]=278;   // HIRAGANA LETTER NI
   sortTable[0x30CB]=278;   // KATAKANA LETTER NI
   sortTable[0xFF86]=278;   // HALFWIDTH KATAKANA LETTER NI
   sortTable[0x306C]=279;   // HIRAGANA LETTER NU
   sortTable[0x30CC]=279;   // KATAKANA LETTER NU
   sortTable[0xFF87]=279;   // HALFWIDTH KATAKANA LETTER NU
   sortTable[0x306D]=280;   // HIRAGANA LETTER NE
   sortTable[0x30CD]=280;   // KATAKANA LETTER NE
   sortTable[0xFF88]=280;   // HALFWIDTH KATAKANA LETTER NE
   sortTable[0x306E]=281;   // HIRAGANA LETTER NO
   sortTable[0x30CE]=281;   // KATAKANA LETTER NO
   sortTable[0xFF89]=281;   // HALFWIDTH KATAKANA LETTER NO
   sortTable[0x306F]=282;   // HIRAGANA LETTER HA
   sortTable[0x30CF]=282;   // KATAKANA LETTER HA
   sortTable[0xFF8A]=282;   // HALFWIDTH KATAKANA LETTER HA
   sortTable[0x3070]=282;   // HIRAGANA LETTER BA
   sortTable[0x30D0]=282;   // KATAKANA LETTER BA
   sortTable[0x3071]=282;   // HIRAGANA LETTER PA
   sortTable[0x30D1]=282;   // KATAKANA LETTER PA
   sortTable[0x3072]=283;   // HIRAGANA LETTER HI
   sortTable[0x30D2]=283;   // KATAKANA LETTER HI
   sortTable[0xFF8B]=283;   // HALFWIDTH KATAKANA LETTER HI
   sortTable[0x3073]=283;   // HIRAGANA LETTER BI
   sortTable[0x30D3]=283;   // KATAKANA LETTER BI
   sortTable[0x3074]=283;   // HIRAGANA LETTER PI
   sortTable[0x30D4]=283;   // KATAKANA LETTER PI
   sortTable[0x3075]=284;   // HIRAGANA LETTER HU
   sortTable[0x30D5]=284;   // KATAKANA LETTER HU
   sortTable[0xFF8C]=284;   // HALFWIDTH KATAKANA LETTER HU
   sortTable[0x3076]=284;   // HIRAGANA LETTER BU
   sortTable[0x30D6]=284;   // KATAKANA LETTER BU
   sortTable[0x3077]=284;   // HIRAGANA LETTER PU
   sortTable[0x30D7]=284;   // KATAKANA LETTER PU
   sortTable[0x3078]=285;   // HIRAGANA LETTER HE
   sortTable[0x30D8]=285;   // KATAKANA LETTER HE
   sortTable[0xFF8D]=285;   // HALFWIDTH KATAKANA LETTER HE
   sortTable[0x3079]=285;   // HIRAGANA LETTER BE
   sortTable[0x30D9]=285;   // KATAKANA LETTER BE
   sortTable[0x307A]=285;   // HIRAGANA LETTER PE
   sortTable[0x30DA]=285;   // KATAKANA LETTER PE
   sortTable[0x307B]=286;   // HIRAGANA LETTER HO
   sortTable[0x30DB]=286;   // KATAKANA LETTER HO
   sortTable[0xFF8E]=286;   // HALFWIDTH KATAKANA LETTER HO
   sortTable[0x307C]=286;   // HIRAGANA LETTER BO
   sortTable[0x30DC]=286;   // KATAKANA LETTER BO
   sortTable[0x307D]=286;   // HIRAGANA LETTER PO
   sortTable[0x30DD]=286;   // KATAKANA LETTER PO
   sortTable[0x307E]=287;   // HIRAGANA LETTER MA
   sortTable[0x30DE]=287;   // KATAKANA LETTER MA
   sortTable[0xFF8F]=287;   // HALFWIDTH KATAKANA LETTER MA
   sortTable[0x307F]=288;   // HIRAGANA LETTER MI
   sortTable[0x30DF]=288;   // KATAKANA LETTER MI
   sortTable[0xFF90]=288;   // HALFWIDTH KATAKANA LETTER MI
   sortTable[0x3080]=289;   // HIRAGANA LETTER MU
   sortTable[0x30E0]=289;   // KATAKANA LETTER MU
   sortTable[0xFF91]=289;   // HALFWIDTH KATAKANA LETTER MU
   sortTable[0x3081]=290;   // HIRAGANA LETTER ME
   sortTable[0x30E1]=290;   // KATAKANA LETTER ME
   sortTable[0xFF92]=290;   // HALFWIDTH KATAKANA LETTER ME
   sortTable[0x3082]=291;   // HIRAGANA LETTER MO
   sortTable[0x30E2]=291;   // KATAKANA LETTER MO
   sortTable[0xFF93]=291;   // HALFWIDTH KATAKANA LETTER MO
   sortTable[0x3083]=292;   // HIRAGANA LETTER SMALL YA
   sortTable[0x30E3]=292;   // KATAKANA LETTER SMALL YA
   sortTable[0xFF6C]=292;   // HALFWIDTH KATAKANA LETTER SMALL YA
   sortTable[0x3084]=292;   // HIRAGANA LETTER YA
   sortTable[0x30E4]=292;   // KATAKANA LETTER YA
   sortTable[0xFF94]=292;   // HALFWIDTH KATAKANA LETTER YA
   sortTable[0x3085]=293;   // HIRAGANA LETTER SMALL YU
   sortTable[0x30E5]=293;   // KATAKANA LETTER SMALL YU
   sortTable[0xFF6D]=293;   // HALFWIDTH KATAKANA LETTER SMALL YU
   sortTable[0x3086]=293;   // HIRAGANA LETTER YU
   sortTable[0x30E6]=293;   // KATAKANA LETTER YU
   sortTable[0xFF95]=293;   // HALFWIDTH KATAKANA LETTER YU
   sortTable[0x3087]=294;   // HIRAGANA LETTER SMALL YO
   sortTable[0x30E7]=294;   // KATAKANA LETTER SMALL YO
   sortTable[0xFF6E]=294;   // HALFWIDTH KATAKANA LETTER SMALL YO
   sortTable[0x3088]=294;   // HIRAGANA LETTER YO
   sortTable[0x30E8]=294;   // KATAKANA LETTER YO
   sortTable[0xFF96]=294;   // HALFWIDTH KATAKANA LETTER YO
   sortTable[0x3089]=295;   // HIRAGANA LETTER RA
   sortTable[0x30E9]=295;   // KATAKANA LETTER RA
   sortTable[0xFF97]=295;   // HALFWIDTH KATAKANA LETTER RA
   sortTable[0x308A]=296;   // HIRAGANA LETTER RI
   sortTable[0x30EA]=296;   // KATAKANA LETTER RI
   sortTable[0xFF98]=296;   // HALFWIDTH KATAKANA LETTER RI
   sortTable[0x308B]=297;   // HIRAGANA LETTER RU
   sortTable[0x30EB]=297;   // KATAKANA LETTER RU
   sortTable[0xFF99]=297;   // HALFWIDTH KATAKANA LETTER RU
   sortTable[0x308C]=298;   // HIRAGANA LETTER RE
   sortTable[0x30EC]=298;   // KATAKANA LETTER RE
   sortTable[0xFF9A]=298;   // HALFWIDTH KATAKANA LETTER RE
   sortTable[0x308D]=299;   // HIRAGANA LETTER RO
   sortTable[0x30ED]=299;   // KATAKANA LETTER RO
   sortTable[0xFF9B]=299;   // HALFWIDTH KATAKANA LETTER RO
   sortTable[0x308F]=300;   // HIRAGANA LETTER WA
   sortTable[0x30EF]=300;   // KATAKANA LETTER WA
   sortTable[0xFF9C]=300;   // HALFWIDTH KATAKANA LETTER WA
   sortTable[0x30F7]=300;   // KATAKANA LETTER VA
   sortTable[0x3090]=301;   // HIRAGANA LETTER WI
   sortTable[0x30F0]=301;   // KATAKANA LETTER WI
   sortTable[0x30F8]=301;   // KATAKANA LETTER VI
   sortTable[0x3091]=302;   // HIRAGANA LETTER WE
   sortTable[0x30F1]=302;   // KATAKANA LETTER WE
   sortTable[0x30F9]=302;   // KATAKANA LETTER VE
   sortTable[0x3092]=303;   // HIRAGANA LETTER WO
   sortTable[0x30F2]=303;   // KATAKANA LETTER WO
   sortTable[0xFF66]=303;   // HALFWIDTH KATAKANA LETTER WO
   sortTable[0x30FA]=303;   // KATAKANA LETTER VO
   sortTable[0x3093]=304;   // HIRAGANA LETTER N
   sortTable[0x30F3]=304;   // KATAKANA LETTER N
   sortTable[0xFF9D]=304;   // HALFWIDTH KATAKANA LETTER N
   sortTable[0x0E00]=31488;
   sortTable[0x0E01]=31489;   // THAI CHARACTER KO KAI
   sortTable[0x0E02]=31490;   // THAI CHARACTER KHO KHAI
   sortTable[0x0E03]=31491;   // THAI CHARACTER KHO KHUAT
   sortTable[0x0E04]=31492;   // THAI CHARACTER KHO KHWAI
   sortTable[0x0E05]=31493;   // THAI CHARACTER KHO KHON
   sortTable[0x0E06]=31494;   // THAI CHARACTER KHO RAKHANG
   sortTable[0x0E07]=31495;   // THAI CHARACTER NGO NGU
   sortTable[0x0E08]=31496;   // THAI CHARACTER CHO CHAN
   sortTable[0x0E09]=31497;   // THAI CHARACTER CHO CHING
   sortTable[0x0E0A]=31498;   // THAI CHARACTER CHO CHANG
   sortTable[0x0E0B]=31499;   // THAI CHARACTER SO SO
   sortTable[0x0E0C]=31500;   // THAI CHARACTER CHO CHOE
   sortTable[0x0E0D]=31501;   // THAI CHARACTER YO YING
   sortTable[0x0E0E]=31502;   // THAI CHARACTER DO CHADA
   sortTable[0x0E0F]=31503;   // THAI CHARACTER TO PATAK
   sortTable[0x0E10]=31504;   // THAI CHARACTER THO THAN
   sortTable[0x0E11]=31505;   // THAI CHARACTER THO NANGMONTHO
   sortTable[0x0E12]=31506;   // THAI CHARACTER THO PHUTHAO
   sortTable[0x0E13]=31507;   // THAI CHARACTER NO NEN
   sortTable[0x0E14]=31508;   // THAI CHARACTER DO DEK
   sortTable[0x0E15]=31509;   // THAI CHARACTER TO TAO
   sortTable[0x0E16]=31510;   // THAI CHARACTER THO THUNG
   sortTable[0x0E17]=31511;   // THAI CHARACTER THO THAHAN
   sortTable[0x0E18]=31512;   // THAI CHARACTER THO THONG
   sortTable[0x0E19]=31513;   // THAI CHARACTER NO NU
   sortTable[0x0E1A]=31514;   // THAI CHARACTER BO BAIMAI
   sortTable[0x0E1B]=31515;   // THAI CHARACTER PO PLA
   sortTable[0x0E1C]=31516;   // THAI CHARACTER PHO PHUNG
   sortTable[0x0E1D]=31517;   // THAI CHARACTER FO FA
   sortTable[0x0E1E]=31518;   // THAI CHARACTER PHO PHAN
   sortTable[0x0E1F]=31519;   // THAI CHARACTER FO FAN
   sortTable[0x0E20]=31520;   // THAI CHARACTER PHO SAMPHAO
   sortTable[0x0E21]=31521;   // THAI CHARACTER MO MA
   sortTable[0x0E22]=31522;   // THAI CHARACTER YO YAK
   sortTable[0x0E23]=31523;   // THAI CHARACTER RO RUA
   sortTable[0x0E24]=31524;   // THAI CHARACTER RU
   sortTable[0x0E25]=31525;   // THAI CHARACTER LO LING
   sortTable[0x0E26]=31526;   // THAI CHARACTER LU
   sortTable[0x0E27]=31527;   // THAI CHARACTER WO WAEN
   sortTable[0x0E28]=31528;   // THAI CHARACTER SO SALA
   sortTable[0x0E29]=31529;   // THAI CHARACTER SO RUSI
   sortTable[0x0E2A]=31530;   // THAI CHARACTER SO SUA
   sortTable[0x0E2B]=31531;   // THAI CHARACTER HO HIP
   sortTable[0x0E2C]=31532;   // THAI CHARACTER LO CHULA
   sortTable[0x0E2D]=31533;   // THAI CHARACTER O ANG
   sortTable[0x0E2E]=31534;   // THAI CHARACTER HO NOKHUK
   sortTable[0x0E2F]=31535;   // THAI CHARACTER PAIYANNOI
   sortTable[0x0E30]=31536;   // THAI CHARACTER SARA A
   sortTable[0x0E31]=31537;   // THAI CHARACTER MAI HAN-AKAT
   sortTable[0x0E32]=31538;   // THAI CHARACTER SARA AA
   sortTable[0x0E33]=31539;   // THAI CHARACTER SARA AM
   sortTable[0x0E34]=31540;   // THAI CHARACTER SARA I
   sortTable[0x0E35]=31541;   // THAI CHARACTER SARA II
   sortTable[0x0E36]=31542;   // THAI CHARACTER SARA UE
   sortTable[0x0E37]=31543;   // THAI CHARACTER SARA UEE
   sortTable[0x0E38]=31544;   // THAI CHARACTER SARA U
   sortTable[0x0E39]=31545;   // THAI CHARACTER SARA UU
   sortTable[0x0E3A]=31546;   // THAI CHARACTER PHINTHU
   sortTable[0x0E3F]=31547;   // THAI CURRENCY SYMBOL BAHT
   sortTable[0x0E40]=31548;   // THAI CHARACTER SARA E
   sortTable[0x0E41]=31549;   // THAI CHARACTER SARA AE
   sortTable[0x0E42]=31550;   // THAI CHARACTER SARA O
   sortTable[0x0E43]=31551;   // THAI CHARACTER SARA AI MAIMUAN
   sortTable[0x0E44]=31552;   // THAI CHARACTER SARA AI MAIMALAI
   sortTable[0x0E45]=31553;   // THAI CHARACTER LAKKHANGYAO
   sortTable[0x0E46]=31554;   // THAI CHARACTER MAIYAMOK
   sortTable[0x0E47]=31555;   // THAI CHARACTER MAITAIKHU
   sortTable[0x0E48]=31556;   // THAI CHARACTER MAI EK
   sortTable[0x0E49]=31557;   // THAI CHARACTER MAI THO
   sortTable[0x0E4A]=31558;   // THAI CHARACTER MAI TRI
   sortTable[0x0E4B]=31559;   // THAI CHARACTER MAI CHATTAWA
   sortTable[0x0E4C]=31560;   // THAI CHARACTER THANTHAKHAT
   sortTable[0x0E4D]=31561;   // THAI CHARACTER NIKHAHIT
   sortTable[0x0E4E]=31562;   // THAI CHARACTER YAMAKKAN
   sortTable[0x0E4F]=31563;   // THAI CHARACTER FONGMAN
   sortTable[0x0E50]=31564;   // THAI DIGIT ZERO
   sortTable[0x0E51]=31565;   // THAI DIGIT ONE
   sortTable[0x0E52]=31566;   // THAI DIGIT TWO
   sortTable[0x0E53]=31567;   // THAI DIGIT THREE
   sortTable[0x0E54]=31568;   // THAI DIGIT FOUR
   sortTable[0x0E55]=31569;   // THAI DIGIT FIVE
   sortTable[0x0E56]=31570;   // THAI DIGIT SIX
   sortTable[0x0E57]=31571;   // THAI DIGIT SEVEN
   sortTable[0x0E58]=31572;   // THAI DIGIT EIGHT
   sortTable[0x0E59]=31573;   // THAI DIGIT NINE
   sortTable[0x0E5A]=31574;   // THAI CHARACTER ANGKHANKHU
   sortTable[0x0E5B]=31575;   // THAI CHARACTER KHOMUT
   sortTable[0x554A]=768;   // a
   sortTable[0x5475]=769;   // a
   sortTable[0x963F]=770;   // a
   sortTable[0x55C4]=771;   // a
   sortTable[0x9515]=772;   // a
   sortTable[0x7231]=773;   // ai
   sortTable[0x611B]=774;   // ai
   sortTable[0x54CE]=775;   // ai
   sortTable[0x6328]=776;   // ai
   sortTable[0x788D]=777;   // ai
   sortTable[0x7919]=778;   // ai
   sortTable[0x54C0]=779;   // ai
   sortTable[0x77EE]=780;   // ai
   sortTable[0x764C]=781;   // ai
   sortTable[0x57C3]=782;   // ai
   sortTable[0x853C]=783;   // ai
   sortTable[0x85F9]=784;   // ai
   sortTable[0x9698]=785;   // ai
   sortTable[0x827E]=786;   // ai
   sortTable[0x5509]=787;   // ai
   sortTable[0x55F3]=788;   // ai
   sortTable[0x66A7]=789;   // ai
   sortTable[0x6371]=790;   // ai
   sortTable[0x972D]=791;   // ai
   sortTable[0x55CC]=792;   // ai
   sortTable[0x7691]=793;   // ai
   sortTable[0x953F]=794;   // ai
   sortTable[0x5AD2]=795;   // ai
   sortTable[0x7839]=796;   // ai
   sortTable[0x7477]=797;   // ai
   sortTable[0x9A03]=798;   // ai
   sortTable[0x8586]=799;   // ai
   sortTable[0x9D31]=800;   // ai
   sortTable[0x8B7A]=801;   // ai
   sortTable[0x9749]=802;   // ai
   sortTable[0x784B]=803;   // ai
   sortTable[0x9932]=804;   // ai
   sortTable[0x9440]=805;   // ai
   sortTable[0x9744]=806;   // ai
   sortTable[0x4F0C]=807;   // ai
   sortTable[0x50FE]=808;   // ai
   sortTable[0x53C6]=809;   // ai
   sortTable[0x5540]=810;   // ai
   sortTable[0x560A]=811;   // ai
   sortTable[0x566F]=812;   // ai
   sortTable[0x5867]=813;   // ai
   sortTable[0x58D2]=814;   // ai
   sortTable[0x5A2D]=815;   // ai
   sortTable[0x5A3E]=816;   // ai
   sortTable[0x5B21]=817;   // ai
   sortTable[0x61D3]=818;   // ai
   sortTable[0x61DD]=819;   // ai
   sortTable[0x6571]=820;   // ai
   sortTable[0x6573]=821;   // ai
   sortTable[0x6639]=822;   // ai
   sortTable[0x66D6]=823;   // ai
   sortTable[0x6B38]=824;   // ai
   sortTable[0x6BD0]=825;   // ai
   sortTable[0x6EB0]=826;   // ai
   sortTable[0x6EBE]=827;   // ai
   sortTable[0x6FED]=828;   // ai
   sortTable[0x74A6]=829;   // ai
   sortTable[0x769A]=830;   // ai
   sortTable[0x76A7]=831;   // ai
   sortTable[0x77B9]=832;   // ai
   sortTable[0x8B6A]=833;   // ai
   sortTable[0x8EB7]=834;   // ai
   sortTable[0x92B0]=835;   // ai
   sortTable[0x9384]=836;   // ai
   sortTable[0x95A1]=837;   // ai
   sortTable[0x99A4]=838;   // ai
   sortTable[0x9C6B]=839;   // ai
   sortTable[0x5B89]=840;   // an
   sortTable[0x6309]=841;   // an
   sortTable[0x5CB8]=842;   // an
   sortTable[0x6697]=843;   // an
   sortTable[0x6848]=844;   // an
   sortTable[0x4FFA]=845;   // an
   sortTable[0x6C28]=846;   // an
   sortTable[0x9E4C]=847;   // an
   sortTable[0x9D6A]=848;   // an
   sortTable[0x9EEF]=849;   // an
   sortTable[0x5EB5]=850;   // an
   sortTable[0x978D]=851;   // an
   sortTable[0x8C19]=852;   // an
   sortTable[0x80FA]=853;   // an
   sortTable[0x95C7]=854;   // an
   sortTable[0x72B4]=855;   // an
   sortTable[0x6849]=856;   // an
   sortTable[0x83F4]=857;   // an
   sortTable[0x94F5]=858;   // an
   sortTable[0x57EF]=859;   // an
   sortTable[0x9B9F]=860;   // an
   sortTable[0x76E6]=861;   // an
   sortTable[0x844A]=862;   // an
   sortTable[0x97FD]=863;   // an
   sortTable[0x63DE]=864;   // an
   sortTable[0x8C7B]=865;   // an
   sortTable[0x9D95]=866;   // an
   sortTable[0x76EB]=867;   // an
   sortTable[0x978C]=868;   // an
   sortTable[0x7F6F]=869;   // an
   sortTable[0x84ED]=870;   // an
   sortTable[0x96F8]=871;   // an
   sortTable[0x4F92]=872;   // an
   sortTable[0x5111]=873;   // an
   sortTable[0x5535]=874;   // an
   sortTable[0x557D]=875;   // an
   sortTable[0x57B5]=876;   // an
   sortTable[0x5813]=877;   // an
   sortTable[0x5A69]=878;   // an
   sortTable[0x5A95]=879;   // an
   sortTable[0x5CD6]=880;   // an
   sortTable[0x667B]=881;   // an
   sortTable[0x6D1D]=882;   // an
   sortTable[0x73B5]=883;   // an
   sortTable[0x75F7]=884;   // an
   sortTable[0x8164]=885;   // an
   sortTable[0x834C]=886;   // an
   sortTable[0x843B]=887;   // an
   sortTable[0x8A9D]=888;   // an
   sortTable[0x8AF3]=889;   // an
   sortTable[0x92A8]=890;   // an
   sortTable[0x930C]=891;   // an
   sortTable[0x968C]=892;   // an
   sortTable[0x99A3]=893;   // an
   sortTable[0x6602]=894;   // ang
   sortTable[0x80AE]=895;   // ang
   sortTable[0x76CE]=896;   // ang
   sortTable[0x9AAF]=897;   // ang
   sortTable[0x536C]=898;   // ang
   sortTable[0x5C87]=899;   // ang
   sortTable[0x663B]=900;   // ang
   sortTable[0x678A]=901;   // ang
   sortTable[0x91A0]=902;   // ang
   sortTable[0x50B2]=903;   // ao
   sortTable[0x8884]=904;   // ao
   sortTable[0x8956]=905;   // ao
   sortTable[0x51F9]=906;   // ao
   sortTable[0x71AC]=907;   // ao
   sortTable[0x5965]=908;   // ao
   sortTable[0x6FB3]=909;   // ao
   sortTable[0x6556]=910;   // ao
   sortTable[0x9CCC]=911;   // ao
   sortTable[0x61CA]=912;   // ao
   sortTable[0x62D7]=913;   // ao
   sortTable[0x5AAA]=914;   // ao
   sortTable[0x7FF1]=915;   // ao
   sortTable[0x55F7]=916;   // ao
   sortTable[0x93CA]=917;   // ao
   sortTable[0x93D6]=918;   // ao
   sortTable[0x9068]=919;   // ao
   sortTable[0x5773]=920;   // ao
   sortTable[0x9A9C]=921;   // ao
   sortTable[0x9C32]=922;   // ao
   sortTable[0x87AF]=923;   // ao
   sortTable[0x5ED2]=924;   // ao
   sortTable[0x7352]=925;   // ao
   sortTable[0x78DD]=926;   // ao
   sortTable[0x8071]=927;   // ao
   sortTable[0x5C99]=928;   // ao
   sortTable[0x96A9]=929;   // ao
   sortTable[0x9F07]=930;   // ao
   sortTable[0x8B37]=931;   // ao
   sortTable[0x8B38]=932;   // ao
   sortTable[0x969E]=933;   // ao
   sortTable[0x9A41]=934;   // ao
   sortTable[0x82BA]=935;   // ao
   sortTable[0x851C]=936;   // ao
   sortTable[0x53AB]=937;   // ao
   sortTable[0x55F8]=938;   // ao
   sortTable[0x5787]=939;   // ao
   sortTable[0x58BA]=940;   // ao
   sortTable[0x5961]=941;   // ao
   sortTable[0x5967]=942;   // ao
   sortTable[0x5ABC]=943;   // ao
   sortTable[0x5AEF]=944;   // ao
   sortTable[0x5CB0]=945;   // ao
   sortTable[0x5D85]=946;   // ao
   sortTable[0x5DB4]=947;   // ao
   sortTable[0x6160]=948;   // ao
   sortTable[0x6277]=949;   // ao
   sortTable[0x629D]=950;   // ao
   sortTable[0x646E]=951;   // ao
   sortTable[0x64D9]=952;   // ao
   sortTable[0x67EA]=953;   // ao
   sortTable[0x688E]=954;   // ao
   sortTable[0x6EF6]=955;   // ao
   sortTable[0x720A]=956;   // ao
   sortTable[0x7353]=957;   // ao
   sortTable[0x7488]=958;   // ao
   sortTable[0x7FF6]=959;   // ao
   sortTable[0x7FFA]=960;   // ao
   sortTable[0x8EEA]=961;   // ao
   sortTable[0x957A]=962;   // ao
   sortTable[0x9DD4]=963;   // ao
   sortTable[0x628A]=1280;   // ba
   sortTable[0x5427]=1281;   // ba
   sortTable[0x7F62]=1282;   // ba
   sortTable[0x7F77]=1283;   // ba
   sortTable[0x516B]=1284;   // ba
   sortTable[0x7238]=1285;   // ba
   sortTable[0x5DF4]=1286;   // ba
   sortTable[0x62D4]=1287;   // ba
   sortTable[0x6252]=1288;   // ba
   sortTable[0x9738]=1289;   // ba
   sortTable[0x53ED]=1290;   // ba
   sortTable[0x575D]=1291;   // ba
   sortTable[0x58E9]=1292;   // ba
   sortTable[0x7B06]=1293;   // ba
   sortTable[0x8DCB]=1294;   // ba
   sortTable[0x75A4]=1295;   // ba
   sortTable[0x9776]=1296;   // ba
   sortTable[0x82AD]=1297;   // ba
   sortTable[0x8019]=1298;   // ba
   sortTable[0x7C91]=1299;   // ba
   sortTable[0x9B43]=1300;   // ba
   sortTable[0x634C]=1301;   // ba
   sortTable[0x705E]=1302;   // ba
   sortTable[0x94AF]=1303;   // ba
   sortTable[0x5C9C]=1304;   // ba
   sortTable[0x9C85]=1305;   // ba
   sortTable[0x83DD]=1306;   // ba
   sortTable[0x8307]=1307;   // ba
   sortTable[0x9C83]=1308;   // ba
   sortTable[0x9C8C]=1309;   // ba
   sortTable[0x9B8A]=1310;   // ba
   sortTable[0x8DC1]=1311;   // ba
   sortTable[0x9B5E]=1312;   // ba
   sortTable[0x8EF7]=1313;   // ba
   sortTable[0x8C5D]=1314;   // ba
   sortTable[0x7F93]=1315;   // ba
   sortTable[0x80C8]=1316;   // ba
   sortTable[0x91DB]=1317;   // ba
   sortTable[0x8686]=1318;   // ba
   sortTable[0x91DF]=1319;   // ba
   sortTable[0x9F25]=1320;   // ba
   sortTable[0x7D26]=1321;   // ba
   sortTable[0x8987]=1322;   // ba
   sortTable[0x4EC8]=1323;   // ba
   sortTable[0x53D0]=1324;   // ba
   sortTable[0x54F5]=1325;   // ba
   sortTable[0x577A]=1326;   // ba
   sortTable[0x57BB]=1327;   // ba
   sortTable[0x58A2]=1328;   // ba
   sortTable[0x593F]=1329;   // ba
   sortTable[0x59AD]=1330;   // ba
   sortTable[0x5CC7]=1331;   // ba
   sortTable[0x5DFC]=1332;   // ba
   sortTable[0x5F1D]=1333;   // ba
   sortTable[0x629C]=1334;   // ba
   sortTable[0x6733]=1335;   // ba
   sortTable[0x67ED]=1336;   // ba
   sortTable[0x6B1B]=1337;   // ba
   sortTable[0x70A6]=1338;   // ba
   sortTable[0x72AE]=1339;   // ba
   sortTable[0x7390]=1340;   // ba
   sortTable[0x7679]=1341;   // ba
   sortTable[0x77F2]=1342;   // ba
   sortTable[0x8A59]=1343;   // ba
   sortTable[0x9200]=1344;   // ba
   sortTable[0x98B0]=1345;   // ba
   sortTable[0x767E]=1346;   // bai
   sortTable[0x767D]=1347;   // bai
   sortTable[0x6446]=1348;   // bai
   sortTable[0x64FA]=1349;   // bai
   sortTable[0x8D25]=1350;   // bai
   sortTable[0x6557]=1351;   // bai
   sortTable[0x62DC]=1352;   // bai
   sortTable[0x67CF]=1353;   // bai
   sortTable[0x63B0]=1354;   // bai
   sortTable[0x7A17]=1355;   // bai
   sortTable[0x64D8]=1356;   // bai
   sortTable[0x4F70]=1357;   // bai
   sortTable[0x636D]=1358;   // bai
   sortTable[0x7CBA]=1359;   // bai
   sortTable[0x8D01]=1360;   // bai
   sortTable[0x7CA8]=1361;   // bai
   sortTable[0x7D54]=1362;   // bai
   sortTable[0x896C]=1363;   // bai
   sortTable[0x85AD]=1364;   // bai
   sortTable[0x86FD]=1365;   // bai
   sortTable[0x5161]=1366;   // bai
   sortTable[0x5E8D]=1367;   // bai
   sortTable[0x62DD]=1368;   // bai
   sortTable[0x6300]=1369;   // bai
   sortTable[0x6822]=1370;   // bai
   sortTable[0x7308]=1371;   // bai
   sortTable[0x74F8]=1372;   // bai
   sortTable[0x7AE1]=1373;   // bai
   sortTable[0x97DB]=1374;   // bai
   sortTable[0x529E]=1375;   // ban
   sortTable[0x8FA6]=1376;   // ban
   sortTable[0x534A]=1377;   // ban
   sortTable[0x822C]=1378;   // ban
   sortTable[0x677F]=1379;   // ban
   sortTable[0x73ED]=1380;   // ban
   sortTable[0x642C]=1381;   // ban
   sortTable[0x4F34]=1382;   // ban
   sortTable[0x7248]=1383;   // ban
   sortTable[0x74E3]=1384;   // ban
   sortTable[0x62CC]=1385;   // ban
   sortTable[0x626E]=1386;   // ban
   sortTable[0x9881]=1387;   // ban
   sortTable[0x6273]=1388;   // ban
   sortTable[0x9812]=1389;   // ban
   sortTable[0x6591]=1390;   // ban
   sortTable[0x7622]=1391;   // ban
   sortTable[0x7ECA]=1392;   // ban
   sortTable[0x962A]=1393;   // ban
   sortTable[0x5742]=1394;   // ban
   sortTable[0x8228]=1395;   // ban
   sortTable[0x94A3]=1396;   // ban
   sortTable[0x764D]=1397;   // ban
   sortTable[0x977D]=1398;   // ban
   sortTable[0x95C6]=1399;   // ban
   sortTable[0x8781]=1400;   // ban
   sortTable[0x9261]=1401;   // ban
   sortTable[0x9B6C]=1402;   // ban
   sortTable[0x7C84]=1403;   // ban
   sortTable[0x7D46]=1404;   // ban
   sortTable[0x9CFB]=1405;   // ban
   sortTable[0x5762]=1406;   // ban
   sortTable[0x59C5]=1407;   // ban
   sortTable[0x5C85]=1408;   // ban
   sortTable[0x6011]=1409;   // ban
   sortTable[0x653D]=1410;   // ban
   sortTable[0x6592]=1411;   // ban
   sortTable[0x6604]=1412;   // ban
   sortTable[0x67C8]=1413;   // ban
   sortTable[0x6E74]=1414;   // ban
   sortTable[0x74EA]=1415;   // ban
   sortTable[0x79DA]=1416;   // ban
   sortTable[0x8742]=1417;   // ban
   sortTable[0x878C]=1418;   // ban
   sortTable[0x8929]=1419;   // ban
   sortTable[0x8FAC]=1420;   // ban
   sortTable[0x9211]=1421;   // ban
   sortTable[0x5E2E]=1422;   // bang
   sortTable[0x5E6B]=1423;   // bang
   sortTable[0x8180]=1424;   // bang
   sortTable[0x68D2]=1425;   // bang
   sortTable[0x7ED1]=1426;   // bang
   sortTable[0x7D81]=1427;   // bang
   sortTable[0x508D]=1428;   // bang
   sortTable[0x699C]=1429;   // bang
   sortTable[0x78C5]=1430;   // bang
   sortTable[0x90A6]=1431;   // bang
   sortTable[0x6886]=1432;   // bang
   sortTable[0x9551]=1433;   // bang
   sortTable[0x8C24]=1434;   // bang
   sortTable[0x6D5C]=1435;   // bang
   sortTable[0x868C]=1436;   // bang
   sortTable[0x84A1]=1437;   // bang
   sortTable[0x8255]=1438;   // bang
   sortTable[0x872F]=1439;   // bang
   sortTable[0x7A16]=1440;   // bang
   sortTable[0x90AB]=1441;   // bang
   sortTable[0x57B9]=1442;   // bang
   sortTable[0x585D]=1443;   // bang
   sortTable[0x5E47]=1444;   // bang
   sortTable[0x5E5A]=1445;   // bang
   sortTable[0x6360]=1446;   // bang
   sortTable[0x6412]=1447;   // bang
   sortTable[0x68D3]=1448;   // bang
   sortTable[0x7253]=1449;   // bang
   sortTable[0x73A4]=1450;   // bang
   sortTable[0x7E0D]=1451;   // bang
   sortTable[0x8B17]=1452;   // bang
   sortTable[0x938A]=1453;   // bang
   sortTable[0x97A4]=1454;   // bang
   sortTable[0x9AC8]=1455;   // bang
   sortTable[0x62A5]=1456;   // bao
   sortTable[0x5831]=1457;   // bao
   sortTable[0x4FDD]=1458;   // bao
   sortTable[0x5305]=1459;   // bao
   sortTable[0x62B1]=1460;   // bao
   sortTable[0x5B9D]=1461;   // bao
   sortTable[0x5BF6]=1462;   // bao
   sortTable[0x7206]=1463;   // bao
   sortTable[0x66B4]=1464;   // bao
   sortTable[0x80DE]=1465;   // bao
   sortTable[0x9971]=1466;   // bao
   sortTable[0x98FD]=1467;   // bao
   sortTable[0x8584]=1468;   // bao
   sortTable[0x5821]=1469;   // bao
   sortTable[0x8C79]=1470;   // bao
   sortTable[0x96F9]=1471;   // bao
   sortTable[0x9C8D]=1472;   // bao
   sortTable[0x8912]=1473;   // bao
   sortTable[0x82DE]=1474;   // bao
   sortTable[0x9E28]=1475;   // bao
   sortTable[0x8446]=1476;   // bao
   sortTable[0x7172]=1477;   // bao
   sortTable[0x9F85]=1478;   // bao
   sortTable[0x5B62]=1479;   // bao
   sortTable[0x8913]=1480;   // bao
   sortTable[0x8DB5]=1481;   // bao
   sortTable[0x8663]=1482;   // bao
   sortTable[0x94C7]=1483;   // bao
   sortTable[0x888C]=1484;   // bao
   sortTable[0x83E2]=1485;   // bao
   sortTable[0x8CF2]=1486;   // bao
   sortTable[0x7A87]=1487;   // bao
   sortTable[0x7B23]=1488;   // bao
   sortTable[0x9AB2]=1489;   // bao
   sortTable[0x9AF1]=1490;   // bao
   sortTable[0x7DE5]=1491;   // bao
   sortTable[0x85F5]=1492;   // bao
   sortTable[0x9464]=1493;   // bao
   sortTable[0x9764]=1494;   // bao
   sortTable[0x9B91]=1495;   // bao
   sortTable[0x9CF5]=1496;   // bao
   sortTable[0x4F68]=1497;   // bao
   sortTable[0x5124]=1498;   // bao
   sortTable[0x52F9]=1499;   // bao
   sortTable[0x52FD]=1500;   // bao
   sortTable[0x5822]=1501;   // bao
   sortTable[0x5AAC]=1502;   // bao
   sortTable[0x5AD1]=1503;   // bao
   sortTable[0x5BB2]=1504;   // bao
   sortTable[0x5BDA]=1505;   // bao
   sortTable[0x5BF3]=1506;   // bao
   sortTable[0x5FC1]=1507;   // bao
   sortTable[0x6009]=1508;   // bao
   sortTable[0x66D3]=1509;   // bao
   sortTable[0x67B9]=1510;   // bao
   sortTable[0x73E4]=1511;   // bao
   sortTable[0x8554]=1512;   // bao
   sortTable[0x86AB]=1513;   // bao
   sortTable[0x8943]=1514;   // bao
   sortTable[0x924B]=1515;   // bao
   sortTable[0x95C1]=1516;   // bao
   sortTable[0x974C]=1517;   // bao
   sortTable[0x99C2]=1518;   // bao
   sortTable[0x9D07]=1519;   // bao
   sortTable[0x9E14]=1520;   // bao
   sortTable[0x9F59]=1521;   // bao
   sortTable[0x88AB]=1522;   // bei
   sortTable[0x5907]=1523;   // bei
   sortTable[0x5099]=1524;   // bei
   sortTable[0x5317]=1525;   // bei
   sortTable[0x80CC]=1526;   // bei
   sortTable[0x500D]=1527;   // bei
   sortTable[0x60B2]=1528;   // bei
   sortTable[0x676F]=1529;   // bei
   sortTable[0x8F88]=1530;   // bei
   sortTable[0x8F29]=1531;   // bei
   sortTable[0x7891]=1532;   // bei
   sortTable[0x8D1D]=1533;   // bei
   sortTable[0x8C9D]=1534;   // bei
   sortTable[0x5457]=1535;   // bei
   sortTable[0x5504]=1536;   // bei
   sortTable[0x5351]=1537;   // bei
   sortTable[0x72C8]=1538;   // bei
   sortTable[0x72FD]=1539;   // bei
   sortTable[0x6096]=1540;   // bei
   sortTable[0x60EB]=1541;   // bei
   sortTable[0x94A1]=1542;   // bei
   sortTable[0x5B5B]=1543;   // bei
   sortTable[0x7119]=1544;   // bei
   sortTable[0x9642]=1545;   // bei
   sortTable[0x97B4]=1546;   // bei
   sortTable[0x84D3]=1547;   // bei
   sortTable[0x943E]=1548;   // bei
   sortTable[0x789A]=1549;   // bei
   sortTable[0x9E4E]=1550;   // bei
   sortTable[0x8919]=1551;   // bei
   sortTable[0x90B6]=1552;   // bei
   sortTable[0x8A96]=1553;   // bei
   sortTable[0x7986]=1554;   // bei
   sortTable[0x7CD2]=1555;   // bei
   sortTable[0x90E5]=1556;   // bei
   sortTable[0x9AB3]=1557;   // bei
   sortTable[0x9101]=1558;   // bei
   sortTable[0x9303]=1559;   // bei
   sortTable[0x9781]=1560;   // bei
   sortTable[0x85E3]=1561;   // bei
   sortTable[0x92C7]=1562;   // bei
   sortTable[0x4FFB]=1563;   // bei
   sortTable[0x505D]=1564;   // bei
   sortTable[0x5079]=1565;   // bei
   sortTable[0x50C3]=1566;   // bei
   sortTable[0x6102]=1567;   // bei
   sortTable[0x618A]=1568;   // bei
   sortTable[0x63F9]=1569;   // bei
   sortTable[0x6601]=1570;   // bei
   sortTable[0x686E]=1571;   // bei
   sortTable[0x6896]=1572;   // bei
   sortTable[0x6911]=1573;   // bei
   sortTable[0x726C]=1574;   // bei
   sortTable[0x7295]=1575;   // bei
   sortTable[0x73FC]=1576;   // bei
   sortTable[0x7432]=1577;   // bei
   sortTable[0x76C3]=1578;   // bei
   sortTable[0x7999]=1579;   // bei
   sortTable[0x82DD]=1580;   // bei
   sortTable[0x8EF0]=1581;   // bei
   sortTable[0x9273]=1582;   // bei
   sortTable[0x9D6F]=1583;   // bei
   sortTable[0x672C]=1584;   // ben
   sortTable[0x5954]=1585;   // ben
   sortTable[0x7B28]=1586;   // ben
   sortTable[0x8D32]=1587;   // ben
   sortTable[0x82EF]=1588;   // ben
   sortTable[0x755A]=1589;   // ben
   sortTable[0x951B]=1590;   // ben
   sortTable[0x574C]=1591;   // ben
   sortTable[0x7FC9]=1592;   // ben
   sortTable[0x8F3D]=1593;   // ben
   sortTable[0x5034]=1594;   // ben
   sortTable[0x574B]=1595;   // ben
   sortTable[0x5959]=1596;   // ben
   sortTable[0x6379]=1597;   // ben
   sortTable[0x64AA]=1598;   // ben
   sortTable[0x681F]=1599;   // ben
   sortTable[0x6873]=1600;   // ben
   sortTable[0x694D]=1601;   // ben
   sortTable[0x6CCD]=1602;   // ben
   sortTable[0x6E00]=1603;   // ben
   sortTable[0x7287]=1604;   // ben
   sortTable[0x7356]=1605;   // ben
   sortTable[0x9029]=1606;   // ben
   sortTable[0x931B]=1607;   // ben
   sortTable[0x752D]=1608;   // beng
   sortTable[0x8E66]=1609;   // beng
   sortTable[0x5D29]=1610;   // beng
   sortTable[0x7EF7]=1611;   // beng
   sortTable[0x7E43]=1612;   // beng
   sortTable[0x6CF5]=1613;   // beng
   sortTable[0x8FF8]=1614;   // beng
   sortTable[0x5623]=1615;   // beng
   sortTable[0x750F]=1616;   // beng
   sortTable[0x83F6]=1617;   // beng
   sortTable[0x794A]=1618;   // beng
   sortTable[0x7D63]=1619;   // beng
   sortTable[0x93F0]=1620;   // beng
   sortTable[0x979B]=1621;   // beng
   sortTable[0x955A]=1622;   // beng
   sortTable[0x882F]=1623;   // beng
   sortTable[0x958D]=1624;   // beng
   sortTable[0x4F3B]=1625;   // beng
   sortTable[0x50B0]=1626;   // beng
   sortTable[0x57C4]=1627;   // beng
   sortTable[0x57F2]=1628;   // beng
   sortTable[0x5874]=1629;   // beng
   sortTable[0x595F]=1630;   // beng
   sortTable[0x5D6D]=1631;   // beng
   sortTable[0x63FC]=1632;   // beng
   sortTable[0x7423]=1633;   // beng
   sortTable[0x742B]=1634;   // beng
   sortTable[0x75ED]=1635;   // beng
   sortTable[0x7DB3]=1636;   // beng
   sortTable[0x902C]=1637;   // beng
   sortTable[0x6BD4]=1638;   // bi
   sortTable[0x5FC5]=1639;   // bi
   sortTable[0x7B14]=1640;   // bi
   sortTable[0x7B46]=1641;   // bi
   sortTable[0x9F3B]=1642;   // bi
   sortTable[0x907F]=1643;   // bi
   sortTable[0x6BD5]=1644;   // bi
   sortTable[0x58C1]=1645;   // bi
   sortTable[0x7562]=1646;   // bi
   sortTable[0x95ED]=1647;   // bi
   sortTable[0x9589]=1648;   // bi
   sortTable[0x903C]=1649;   // bi
   sortTable[0x81C2]=1650;   // bi
   sortTable[0x5F7C]=1651;   // bi
   sortTable[0x853D]=1652;   // bi
   sortTable[0x5E01]=1653;   // bi
   sortTable[0x5E63]=1654;   // bi
   sortTable[0x78A7]=1655;   // bi
   sortTable[0x6BD9]=1656;   // bi
   sortTable[0x6583]=1657;   // bi
   sortTable[0x9119]=1658;   // bi
   sortTable[0x75F9]=1659;   // bi
   sortTable[0x75FA]=1660;   // bi
   sortTable[0x965B]=1661;   // bi
   sortTable[0x5F0A]=1662;   // bi
   sortTable[0x5A62]=1663;   // bi
   sortTable[0x5F3C]=1664;   // bi
   sortTable[0x655D]=1665;   // bi
   sortTable[0x5E87]=1666;   // bi
   sortTable[0x74A7]=1667;   // bi
   sortTable[0x4FFE]=1668;   // bi
   sortTable[0x88E8]=1669;   // bi
   sortTable[0x5315]=1670;   // bi
   sortTable[0x8DF8]=1671;   // bi
   sortTable[0x5B16]=1672;   // bi
   sortTable[0x84D6]=1673;   // bi
   sortTable[0x6BD6]=1674;   // bi
   sortTable[0x9AC0]=1675;   // bi
   sortTable[0x859C]=1676;   // bi
   sortTable[0x610E]=1677;   // bi
   sortTable[0x5421]=1678;   // bi
   sortTable[0x7540]=1679;   // bi
   sortTable[0x59A3]=1680;   // bi
   sortTable[0x8406]=1681;   // bi
   sortTable[0x82FE]=1682;   // bi
   sortTable[0x7BE6]=1683;   // bi
   sortTable[0x822D]=1684;   // bi
   sortTable[0x835C]=1685;   // bi
   sortTable[0x895E]=1686;   // bi
   sortTable[0x6FDE]=1687;   // bi
   sortTable[0x54D4]=1688;   // bi
   sortTable[0x959F]=1689;   // bi
   sortTable[0x8BD0]=1690;   // bi
   sortTable[0x5EB3]=1691;   // bi
   sortTable[0x7B5A]=1692;   // bi
   sortTable[0x79D5]=1693;   // bi
   sortTable[0x89F1]=1694;   // bi
   sortTable[0x90B2]=1695;   // bi
   sortTable[0x72F4]=1696;   // bi
   sortTable[0x94CB]=1697;   // bi
   sortTable[0x8378]=1698;   // bi
   sortTable[0x8E84]=1699;   // bi
   sortTable[0x99DC]=1700;   // bi
   sortTable[0x7C83]=1701;   // bi
   sortTable[0x98F6]=1702;   // bi
   sortTable[0x7C8A]=1703;   // bi
   sortTable[0x805B]=1704;   // bi
   sortTable[0x7F7C]=1705;   // bi
   sortTable[0x97E0]=1706;   // bi
   sortTable[0x7B84]=1707;   // bi
   sortTable[0x7B85]=1708;   // bi
   sortTable[0x8795]=1709;   // bi
   sortTable[0x6ED7]=1710;   // bi
   sortTable[0x7DBC]=1711;   // bi
   sortTable[0x8C8F]=1712;   // bi
   sortTable[0x912A]=1713;   // bi
   sortTable[0x7E2A]=1714;   // bi
   sortTable[0x9DE9]=1715;   // bi
   sortTable[0x8D51]=1716;   // bi
   sortTable[0x97B8]=1717;   // bi
   sortTable[0x9AF2]=1718;   // bi
   sortTable[0x84FD]=1719;   // bi
   sortTable[0x9A46]=1720;   // bi
   sortTable[0x9F0A]=1721;   // bi
   sortTable[0x870C]=1722;   // bi
   sortTable[0x8890]=1723;   // bi
   sortTable[0x8A56]=1724;   // bi
   sortTable[0x9128]=1725;   // bi
   sortTable[0x7B13]=1726;   // bi
   sortTable[0x8177]=1727;   // bi
   sortTable[0x7E74]=1728;   // bi
   sortTable[0x8E83]=1729;   // bi
   sortTable[0x9946]=1730;   // bi
   sortTable[0x9CBE]=1731;   // bi
   sortTable[0x939E]=1732;   // bi
   sortTable[0x9B85]=1733;   // bi
   sortTable[0x7BF3]=1734;   // bi
   sortTable[0x8CC1]=1735;   // bi
   sortTable[0x8D14]=1736;   // bi
   sortTable[0x8E55]=1737;   // bi
   sortTable[0x924D]=1738;   // bi
   sortTable[0x9434]=1739;   // bi
   sortTable[0x999D]=1740;   // bi
   sortTable[0x4F4A]=1741;   // bi
   sortTable[0x4F56]=1742;   // bi
   sortTable[0x506A]=1743;   // bi
   sortTable[0x5302]=1744;   // bi
   sortTable[0x555A]=1745;   // bi
   sortTable[0x55F6]=1746;   // bi
   sortTable[0x5752]=1747;   // bi
   sortTable[0x581B]=1748;   // bi
   sortTable[0x5936]=1749;   // bi
   sortTable[0x5970]=1750;   // bi
   sortTable[0x59BC]=1751;   // bi
   sortTable[0x5B36]=1752;   // bi
   sortTable[0x5C44]=1753;   // bi
   sortTable[0x5E64]=1754;   // bi
   sortTable[0x5EE6]=1755;   // bi
   sortTable[0x5F3B]=1756;   // bi
   sortTable[0x5F43]=1757;   // bi
   sortTable[0x602D]=1758;   // bi
   sortTable[0x6036]=1759;   // bi
   sortTable[0x610A]=1760;   // bi
   sortTable[0x673C]=1761;   // bi
   sortTable[0x6788]=1762;   // bi
   sortTable[0x67C0]=1763;   // bi
   sortTable[0x67F2]=1764;   // bi
   sortTable[0x6890]=1765;   // bi
   sortTable[0x6945]=1766;   // bi
   sortTable[0x6BF4]=1767;   // bi
   sortTable[0x6C98]=1768;   // bi
   sortTable[0x6E62]=1769;   // bi
   sortTable[0x6EED]=1770;   // bi
   sortTable[0x6F77]=1771;   // bi
   sortTable[0x714F]=1772;   // bi
   sortTable[0x719A]=1773;   // bi
   sortTable[0x7358]=1774;   // bi
   sortTable[0x7359]=1775;   // bi
   sortTable[0x73CC]=1776;   // bi
   sortTable[0x7595]=1777;   // bi
   sortTable[0x75AA]=1778;   // bi
   sortTable[0x7695]=1779;   // bi
   sortTable[0x7764]=1780;   // bi
   sortTable[0x7B86]=1781;   // bi
   sortTable[0x841E]=1782;   // bi
   sortTable[0x8945]=1783;   // bi
   sortTable[0x8963]=1784;   // bi
   sortTable[0x8C4D]=1785;   // bi
   sortTable[0x8CB1]=1786;   // bi
   sortTable[0x93CE]=1787;   // bi
   sortTable[0x9587]=1788;   // bi
   sortTable[0x9B53]=1789;   // bi
   sortTable[0x9C0F]=1790;   // bi
   sortTable[0x9D56]=1791;   // bi
   sortTable[0x9DDD]=1792;   // bi
   sortTable[0x8FB9]=1793;   // bian
   sortTable[0x908A]=1794;   // bian
   sortTable[0x53D8]=1795;   // bian
   sortTable[0x8B8A]=1796;   // bian
   sortTable[0x4FBF]=1797;   // bian
   sortTable[0x904D]=1798;   // bian
   sortTable[0x7F16]=1799;   // bian
   sortTable[0x7DE8]=1800;   // bian
   sortTable[0x8FA9]=1801;   // bian
   sortTable[0x8FAF]=1802;   // bian
   sortTable[0x6241]=1803;   // bian
   sortTable[0x8FAB]=1804;   // bian
   sortTable[0x8FAE]=1805;   // bian
   sortTable[0x8FA8]=1806;   // bian
   sortTable[0x97AD]=1807;   // bian
   sortTable[0x533E]=1808;   // bian
   sortTable[0x8759]=1809;   // bian
   sortTable[0x8D2C]=1810;   // bian
   sortTable[0x6C74]=1811;   // bian
   sortTable[0x5F01]=1812;   // bian
   sortTable[0x7B3E]=1813;   // bian
   sortTable[0x535E]=1814;   // bian
   sortTable[0x82C4]=1815;   // bian
   sortTable[0x782D]=1816;   // bian
   sortTable[0x7178]=1817;   // bian
   sortTable[0x7A86]=1818;   // bian
   sortTable[0x890A]=1819;   // bian
   sortTable[0x5FED]=1820;   // bian
   sortTable[0x7F0F]=1821;   // bian
   sortTable[0x9CCA]=1822;   // bian
   sortTable[0x78A5]=1823;   // bian
   sortTable[0x91C6]=1824;   // bian
   sortTable[0x85CA]=1825;   // bian
   sortTable[0x8251]=1826;   // bian
   sortTable[0x7BAF]=1827;   // bian
   sortTable[0x9089]=1828;   // bian
   sortTable[0x898D]=1829;   // bian
   sortTable[0x9BFE]=1830;   // bian
   sortTable[0x9D18]=1831;   // bian
   sortTable[0x7DF6]=1832;   // bian
   sortTable[0x959E]=1833;   // bian
   sortTable[0x8439]=1834;   // bian
   sortTable[0x5325]=1835;   // bian
   sortTable[0x5909]=1836;   // bian
   sortTable[0x5CC5]=1837;   // bian
   sortTable[0x5FA7]=1838;   // bian
   sortTable[0x60FC]=1839;   // bian
   sortTable[0x6283]=1840;   // bian
   sortTable[0x63D9]=1841;   // bian
   sortTable[0x662A]=1842;   // bian
   sortTable[0x6C73]=1843;   // bian
   sortTable[0x709E]=1844;   // bian
   sortTable[0x7251]=1845;   // bian
   sortTable[0x7335]=1846;   // bian
   sortTable[0x73A3]=1847;   // bian
   sortTable[0x7502]=1848;   // bian
   sortTable[0x7A28]=1849;   // bian
   sortTable[0x7C69]=1850;   // bian
   sortTable[0x7CC4]=1851;   // bian
   sortTable[0x8CB6]=1852;   // bian
   sortTable[0x8FA1]=1853;   // bian
   sortTable[0x8FA7]=1854;   // bian
   sortTable[0x8FBA]=1855;   // bian
   sortTable[0x937D]=1856;   // bian
   sortTable[0x9BFF]=1857;   // bian
   sortTable[0x8868]=1858;   // biao
   sortTable[0x6807]=1859;   // biao
   sortTable[0x6A19]=1860;   // biao
   sortTable[0x5F6A]=1861;   // biao
   sortTable[0x9556]=1862;   // biao
   sortTable[0x9AA0]=1863;   // biao
   sortTable[0x9ADF]=1864;   // biao
   sortTable[0x5A4A]=1865;   // biao
   sortTable[0x98D9]=1866;   // biao
   sortTable[0x88F1]=1867;   // biao
   sortTable[0x6753]=1868;   // biao
   sortTable[0x9573]=1869;   // biao
   sortTable[0x98DA]=1870;   // biao
   sortTable[0x8198]=1871;   // biao
   sortTable[0x893E]=1872;   // biao
   sortTable[0x9CD4]=1873;   // biao
   sortTable[0x8D06]=1874;   // biao
   sortTable[0x98D1]=1875;   // biao
   sortTable[0x85E8]=1876;   // biao
   sortTable[0x813F]=1877;   // biao
   sortTable[0x81D5]=1878;   // biao
   sortTable[0x78E6]=1879;   // biao
   sortTable[0x762D]=1880;   // biao
   sortTable[0x8508]=1881;   // biao
   sortTable[0x8AD8]=1882;   // biao
   sortTable[0x98C6]=1883;   // biao
   sortTable[0x93E2]=1884;   // biao
   sortTable[0x9463]=1885;   // biao
   sortTable[0x98A9]=1886;   // biao
   sortTable[0x9A43]=1887;   // biao
   sortTable[0x9C3E]=1888;   // biao
   sortTable[0x4FF5]=1889;   // biao
   sortTable[0x5126]=1890;   // biao
   sortTable[0x5882]=1891;   // biao
   sortTable[0x5E56]=1892;   // biao
   sortTable[0x647D]=1893;   // biao
   sortTable[0x6AA6]=1894;   // biao
   sortTable[0x6DF2]=1895;   // biao
   sortTable[0x6EEE]=1896;   // biao
   sortTable[0x700C]=1897;   // biao
   sortTable[0x706C]=1898;   // biao
   sortTable[0x719B]=1899;   // biao
   sortTable[0x7202]=1900;   // biao
   sortTable[0x730B]=1901;   // biao
   sortTable[0x7A6E]=1902;   // biao
   sortTable[0x8B24]=1903;   // biao
   sortTable[0x9336]=1904;   // biao
   sortTable[0x98AE]=1905;   // biao
   sortTable[0x98B7]=1906;   // biao
   sortTable[0x98C7]=1907;   // biao
   sortTable[0x98C8]=1908;   // biao
   sortTable[0x98CA]=1909;   // biao
   sortTable[0x9A6B]=1910;   // biao
   sortTable[0x9A89]=1911;   // biao
   sortTable[0x522B]=1912;   // bie
   sortTable[0x5225]=1913;   // bie
   sortTable[0x618B]=1914;   // bie
   sortTable[0x762A]=1915;   // bie
   sortTable[0x765F]=1916;   // bie
   sortTable[0x9CD6]=1917;   // bie
   sortTable[0x8E69]=1918;   // bie
   sortTable[0x9F08]=1919;   // bie
   sortTable[0x8952]=1920;   // bie
   sortTable[0x9C49]=1921;   // bie
   sortTable[0x87DE]=1922;   // bie
   sortTable[0x8382]=1923;   // bie
   sortTable[0x86C2]=1924;   // bie
   sortTable[0x5487]=1925;   // bie
   sortTable[0x5F46]=1926;   // bie
   sortTable[0x5FB6]=1927;   // bie
   sortTable[0x864C]=1928;   // bie
   sortTable[0x9F9E]=1929;   // bie
   sortTable[0x6EE8]=1930;   // bin
   sortTable[0x6FF1]=1931;   // bin
   sortTable[0x5BBE]=1932;   // bin
   sortTable[0x8CD3]=1933;   // bin
   sortTable[0x658C]=1934;   // bin
   sortTable[0x5F6C]=1935;   // bin
   sortTable[0x9B13]=1936;   // bin
   sortTable[0x6BA1]=1937;   // bin
   sortTable[0x6FD2]=1938;   // bin
   sortTable[0x90A0]=1939;   // bin
   sortTable[0x7F24]=1940;   // bin
   sortTable[0x6448]=1941;   // bin
   sortTable[0x69DF]=1942;   // bin
   sortTable[0x50A7]=1943;   // bin
   sortTable[0x8191]=1944;   // bin
   sortTable[0x9ACC]=1945;   // bin
   sortTable[0x8C73]=1946;   // bin
   sortTable[0x73A2]=1947;   // bin
   sortTable[0x9554]=1948;   // bin
   sortTable[0x9AE9]=1949;   // bin
   sortTable[0x9B02]=1950;   // bin
   sortTable[0x9726]=1951;   // bin
   sortTable[0x81CF]=1952;   // bin
   sortTable[0x780F]=1953;   // bin
   sortTable[0x9AD5]=1954;   // bin
   sortTable[0x8CD4]=1955;   // bin
   sortTable[0x986E]=1956;   // bin
   sortTable[0x9B22]=1957;   // bin
   sortTable[0x5110]=1958;   // bin
   sortTable[0x64EF]=1959;   // bin
   sortTable[0x68B9]=1960;   // bin
   sortTable[0x6915]=1961;   // bin
   sortTable[0x6AB3]=1962;   // bin
   sortTable[0x6BAF]=1963;   // bin
   sortTable[0x6C1E]=1964;   // bin
   sortTable[0x6C43]=1965;   // bin
   sortTable[0x6FF5]=1966;   // bin
   sortTable[0x7015]=1967;   // bin
   sortTable[0x7478]=1968;   // bin
   sortTable[0x74B8]=1969;   // bin
   sortTable[0x7E7D]=1970;   // bin
   sortTable[0x8668]=1971;   // bin
   sortTable[0x8C69]=1972;   // bin
   sortTable[0x944C]=1973;   // bin
   sortTable[0x5E76]=1974;   // bing
   sortTable[0x75C5]=1975;   // bing
   sortTable[0x5175]=1976;   // bing
   sortTable[0x51B0]=1977;   // bing
   sortTable[0x997C]=1978;   // bing
   sortTable[0x9905]=1979;   // bing
   sortTable[0x4E19]=1980;   // bing
   sortTable[0x67C4]=1981;   // bing
   sortTable[0x7980]=1982;   // bing
   sortTable[0x79C9]=1983;   // bing
   sortTable[0x70B3]=1984;   // bing
   sortTable[0x6452]=1985;   // bing
   sortTable[0x90B4]=1986;   // bing
   sortTable[0x979E]=1987;   // bing
   sortTable[0x9235]=1988;   // bing
   sortTable[0x7A1F]=1989;   // bing
   sortTable[0x7ADD]=1990;   // bing
   sortTable[0x927C]=1991;   // bing
   sortTable[0x9786]=1992;   // bing
   sortTable[0x92F2]=1993;   // bing
   sortTable[0x7A89]=1994;   // bing
   sortTable[0x82EA]=1995;   // bing
   sortTable[0x772A]=1996;   // bing
   sortTable[0x4E26]=1997;   // bing
   sortTable[0x4ECC]=1998;   // bing
   sortTable[0x4ED2]=1999;   // bing
   sortTable[0x4F75]=2000;   // bing
   sortTable[0x5002]=2001;   // bing
   sortTable[0x504B]=2002;   // bing
   sortTable[0x50A1]=2003;   // bing
   sortTable[0x51AB]=2004;   // bing
   sortTable[0x57AA]=2005;   // bing
   sortTable[0x5BCE]=2006;   // bing
   sortTable[0x5E77]=2007;   // bing
   sortTable[0x5EB0]=2008;   // bing
   sortTable[0x6032]=2009;   // bing
   sortTable[0x62A6]=2010;   // bing
   sortTable[0x63A4]=2011;   // bing
   sortTable[0x661E]=2012;   // bing
   sortTable[0x663A]=2013;   // bing
   sortTable[0x6824]=2014;   // bing
   sortTable[0x68C5]=2015;   // bing
   sortTable[0x6C37]=2016;   // bing
   sortTable[0x86C3]=2017;   // bing
   sortTable[0x8A81]=2018;   // bing
   sortTable[0x9643]=2019;   // bing
   sortTable[0x9750]=2020;   // bing
   sortTable[0x9920]=2021;   // bing
   sortTable[0x9BA9]=2022;   // bing
   sortTable[0x73BB]=2023;   // bo
   sortTable[0x5265]=2024;   // bo
   sortTable[0x525D]=2025;   // bo
   sortTable[0x64AD]=2026;   // bo
   sortTable[0x4F2F]=2027;   // bo
   sortTable[0x535A]=2028;   // bo
   sortTable[0x6CE2]=2029;   // bo
   sortTable[0x8116]=2030;   // bo
   sortTable[0x62E8]=2031;   // bo
   sortTable[0x64A5]=2032;   // bo
   sortTable[0x52C3]=2033;   // bo
   sortTable[0x818A]=2034;   // bo
   sortTable[0x9A73]=2035;   // bo
   sortTable[0x640F]=2036;   // bo
   sortTable[0x99C1]=2037;   // bo
   sortTable[0x535C]=2038;   // bo
   sortTable[0x7C38]=2039;   // bo
   sortTable[0x8236]=2040;   // bo
   sortTable[0x8DDB]=2041;   // bo
   sortTable[0x5E1B]=2042;   // bo
   sortTable[0x5575]=2043;   // bo
   sortTable[0x6E24]=2044;   // bo
   sortTable[0x94B5]=2045;   // bo
   sortTable[0x4EB3]=2046;   // bo
   sortTable[0x997D]=2047;   // bo
   sortTable[0x7B94]=2048;   // bo
   sortTable[0x9E41]=2049;   // bo
   sortTable[0x83E0]=2050;   // bo
   sortTable[0x9548]=2051;   // bo
   sortTable[0x94B9]=2052;   // bo
   sortTable[0x8E23]=2053;   // bo
   sortTable[0x7934]=2054;   // bo
   sortTable[0x94C2]=2055;   // bo
   sortTable[0x6A97]=2056;   // bo
   sortTable[0x99EE]=2057;   // bo
   sortTable[0x999B]=2058;   // bo
   sortTable[0x9AC6]=2059;   // bo
   sortTable[0x896E]=2060;   // bo
   sortTable[0x939B]=2061;   // bo
   sortTable[0x7F3D]=2062;   // bo
   sortTable[0x889A]=2063;   // bo
   sortTable[0x8B52]=2064;   // bo
   sortTable[0x8467]=2065;   // bo
   sortTable[0x7921]=2066;   // bo
   sortTable[0x998E]=2067;   // bo
   sortTable[0x7886]=2068;   // bo
   sortTable[0x7C19]=2069;   // bo
   sortTable[0x7D34]=2070;   // bo
   sortTable[0x80C9]=2071;   // bo
   sortTable[0x946E]=2072;   // bo
   sortTable[0x8514]=2073;   // bo
   sortTable[0x88AF]=2074;   // bo
   sortTable[0x7BA5]=2075;   // bo
   sortTable[0x88B9]=2076;   // bo
   sortTable[0x9A4B]=2077;   // bo
   sortTable[0x9B81]=2078;   // bo
   sortTable[0x8E73]=2079;   // bo
   sortTable[0x9238]=2080;   // bo
   sortTable[0x92CD]=2081;   // bo
   sortTable[0x7CEA]=2082;   // bo
   sortTable[0x824A]=2083;   // bo
   sortTable[0x8421]=2084;   // bo
   sortTable[0x8617]=2085;   // bo
   sortTable[0x8C70]=2086;   // bo
   sortTable[0x9AC9]=2087;   // bo
   sortTable[0x4EE2]=2088;   // bo
   sortTable[0x4FBC]=2089;   // bo
   sortTable[0x50E0]=2090;   // bo
   sortTable[0x50F0]=2091;   // bo
   sortTable[0x54F1]=2092;   // bo
   sortTable[0x5697]=2093;   // bo
   sortTable[0x5B79]=2094;   // bo
   sortTable[0x5D93]=2095;   // bo
   sortTable[0x5E17]=2096;   // bo
   sortTable[0x613D]=2097;   // bo
   sortTable[0x61EA]=2098;   // bo
   sortTable[0x632C]=2099;   // bo
   sortTable[0x6B02]=2100;   // bo
   sortTable[0x6D61]=2101;   // bo
   sortTable[0x717F]=2102;   // bo
   sortTable[0x7254]=2103;   // bo
   sortTable[0x72A6]=2104;   // bo
   sortTable[0x72BB]=2105;   // bo
   sortTable[0x72DB]=2106;   // bo
   sortTable[0x733C]=2107;   // bo
   sortTable[0x74DD]=2108;   // bo
   sortTable[0x74DF]=2109;   // bo
   sortTable[0x7676]=2110;   // bo
   sortTable[0x7677]=2111;   // bo
   sortTable[0x76CB]=2112;   // bo
   sortTable[0x7835]=2113;   // bo
   sortTable[0x79E1]=2114;   // bo
   sortTable[0x8091]=2115;   // bo
   sortTable[0x82E9]=2116;   // bo
   sortTable[0x88B0]=2117;   // bo
   sortTable[0x894F]=2118;   // bo
   sortTable[0x90E3]=2119;   // bo
   sortTable[0x9251]=2120;   // bo
   sortTable[0x9262]=2121;   // bo
   sortTable[0x9911]=2122;   // bo
   sortTable[0x993A]=2123;   // bo
   sortTable[0x999E]=2124;   // bo
   sortTable[0x9C4D]=2125;   // bo
   sortTable[0x9D53]=2126;   // bo
   sortTable[0x4E0D]=2127;   // bu
   sortTable[0x90E8]=2128;   // bu
   sortTable[0x6B65]=2129;   // bu
   sortTable[0x5E03]=2130;   // bu
   sortTable[0x8865]=2131;   // bu
   sortTable[0x88DC]=2132;   // bu
   sortTable[0x6355]=2133;   // bu
   sortTable[0x6016]=2134;   // bu
   sortTable[0x54FA]=2135;   // bu
   sortTable[0x57E0]=2136;   // bu
   sortTable[0x7C3F]=2137;   // bu
   sortTable[0x57D4]=2138;   // bu
   sortTable[0x900B]=2139;   // bu
   sortTable[0x6661]=2140;   // bu
   sortTable[0x949A]=2141;   // bu
   sortTable[0x535F]=2142;   // bu
   sortTable[0x91AD]=2143;   // bu
   sortTable[0x74FF]=2144;   // bu
   sortTable[0x8500]=2145;   // bu
   sortTable[0x9914]=2146;   // bu
   sortTable[0x94B8]=2147;   // bu
   sortTable[0x7BF0]=2148;   // bu
   sortTable[0x9CEA]=2149;   // bu
   sortTable[0x8AA7]=2150;   // bu
   sortTable[0x8E04]=2151;   // bu
   sortTable[0x923D]=2152;   // bu
   sortTable[0x8379]=2153;   // bu
   sortTable[0x8F50]=2154;   // bu
   sortTable[0x9D4F]=2155;   // bu
   sortTable[0x4F48]=2156;   // bu
   sortTable[0x52CF]=2157;   // bu
   sortTable[0x5425]=2158;   // bu
   sortTable[0x5498]=2159;   // bu
   sortTable[0x55B8]=2160;   // bu
   sortTable[0x57D7]=2161;   // bu
   sortTable[0x5CEC]=2162;   // bu
   sortTable[0x5EAF]=2163;   // bu
   sortTable[0x6091]=2164;   // bu
   sortTable[0x62AA]=2165;   // bu
   sortTable[0x6357]=2166;   // bu
   sortTable[0x67E8]=2167;   // bu
   sortTable[0x6B68]=2168;   // bu
   sortTable[0x6B69]=2169;   // bu
   sortTable[0x90F6]=2170;   // bu
   sortTable[0x9922]=2171;   // bu
   sortTable[0x64E6]=2560;   // ca
   sortTable[0x5693]=2561;   // ca
   sortTable[0x7924]=2562;   // ca
   sortTable[0x56C3]=2563;   // ca
   sortTable[0x6503]=2564;   // ca
   sortTable[0x906A]=2565;   // ca
   sortTable[0x624D]=2566;   // cai
   sortTable[0x6750]=2567;   // cai
   sortTable[0x91C7]=2568;   // cai
   sortTable[0x83DC]=2569;   // cai
   sortTable[0x8D22]=2570;   // cai
   sortTable[0x8CA1]=2571;   // cai
   sortTable[0x5F69]=2572;   // cai
   sortTable[0x731C]=2573;   // cai
   sortTable[0x8E29]=2574;   // cai
   sortTable[0x88C1]=2575;   // cai
   sortTable[0x8521]=2576;   // cai
   sortTable[0x776C]=2577;   // cai
   sortTable[0x7E94]=2578;   // cai
   sortTable[0x7DB5]=2579;   // cai
   sortTable[0x8DF4]=2580;   // cai
   sortTable[0x7E29]=2581;   // cai
   sortTable[0x5038]=2582;   // cai
   sortTable[0x5072]=2583;   // cai
   sortTable[0x554B]=2584;   // cai
   sortTable[0x57F0]=2585;   // cai
   sortTable[0x5A47]=2586;   // cai
   sortTable[0x5BC0]=2587;   // cai
   sortTable[0x63A1]=2588;   // cai
   sortTable[0x68CC]=2589;   // cai
   sortTable[0x6BDD]=2590;   // cai
   sortTable[0x6EA8]=2591;   // cai
   sortTable[0x72B2]=2592;   // cai
   sortTable[0x53C2]=2593;   // can
   sortTable[0x53C3]=2594;   // can
   sortTable[0x6B8B]=2595;   // can
   sortTable[0x6B98]=2596;   // can
   sortTable[0x8695]=2597;   // can
   sortTable[0x8836]=2598;   // can
   sortTable[0x60E8]=2599;   // can
   sortTable[0x6158]=2600;   // can
   sortTable[0x9910]=2601;   // can
   sortTable[0x707F]=2602;   // can
   sortTable[0x71E6]=2603;   // can
   sortTable[0x60ED]=2604;   // can
   sortTable[0x615A]=2605;   // can
   sortTable[0x7CB2]=2606;   // can
   sortTable[0x63BA]=2607;   // can
   sortTable[0x74A8]=2608;   // can
   sortTable[0x9A96]=2609;   // can
   sortTable[0x5B71]=2610;   // can
   sortTable[0x9EEA]=2611;   // can
   sortTable[0x7BF8]=2612;   // can
   sortTable[0x98E1]=2613;   // can
   sortTable[0x7A47]=2614;   // can
   sortTable[0x9A42]=2615;   // can
   sortTable[0x50AA]=2616;   // can
   sortTable[0x510F]=2617;   // can
   sortTable[0x53C4]=2618;   // can
   sortTable[0x53C5]=2619;   // can
   sortTable[0x55B0]=2620;   // can
   sortTable[0x5B20]=2621;   // can
   sortTable[0x5B31]=2622;   // can
   sortTable[0x6159]=2623;   // can
   sortTable[0x61AF]=2624;   // can
   sortTable[0x647B]=2625;   // can
   sortTable[0x6701]=2626;   // can
   sortTable[0x6E4C]=2627;   // can
   sortTable[0x6FAF]=2628;   // can
   sortTable[0x7218]=2629;   // can
   sortTable[0x8592]=2630;   // can
   sortTable[0x8745]=2631;   // can
   sortTable[0x883A]=2632;   // can
   sortTable[0x8B32]=2633;   // can
   sortTable[0x9EF2]=2634;   // can
   sortTable[0x85CF]=2635;   // cang
   sortTable[0x8231]=2636;   // cang
   sortTable[0x8259]=2637;   // cang
   sortTable[0x82CD]=2638;   // cang
   sortTable[0x84BC]=2639;   // cang
   sortTable[0x4ED3]=2640;   // cang
   sortTable[0x5009]=2641;   // cang
   sortTable[0x6CA7]=2642;   // cang
   sortTable[0x4F27]=2643;   // cang
   sortTable[0x9E27]=2644;   // cang
   sortTable[0x9DAC]=2645;   // cang
   sortTable[0x7F49]=2646;   // cang
   sortTable[0x8CF6]=2647;   // cang
   sortTable[0x87A5]=2648;   // cang
   sortTable[0x4EFA]=2649;   // cang
   sortTable[0x5096]=2650;   // cang
   sortTable[0x5D62]=2651;   // cang
   sortTable[0x6B0C]=2652;   // cang
   sortTable[0x6EC4]=2653;   // cang
   sortTable[0x6FF8]=2654;   // cang
   sortTable[0x734A]=2655;   // cang
   sortTable[0x9476]=2656;   // cang
   sortTable[0x8349]=2657;   // cao
   sortTable[0x64CD]=2658;   // cao
   sortTable[0x69FD]=2659;   // cao
   sortTable[0x7CD9]=2660;   // cao
   sortTable[0x5608]=2661;   // cao
   sortTable[0x66F9]=2662;   // cao
   sortTable[0x6F15]=2663;   // cao
   sortTable[0x825A]=2664;   // cao
   sortTable[0x87AC]=2665;   // cao
   sortTable[0x8278]=2666;   // cao
   sortTable[0x808F]=2667;   // cao
   sortTable[0x84F8]=2668;   // cao
   sortTable[0x9135]=2669;   // cao
   sortTable[0x9A32]=2670;   // cao
   sortTable[0x5D86]=2671;   // cao
   sortTable[0x613A]=2672;   // cao
   sortTable[0x61C6]=2673;   // cao
   sortTable[0x64A1]=2674;   // cao
   sortTable[0x66FA]=2675;   // cao
   sortTable[0x8279]=2676;   // cao
   sortTable[0x893F]=2677;   // cao
   sortTable[0x8959]=2678;   // cao
   sortTable[0x93EA]=2679;   // cao
   sortTable[0x7B56]=2680;   // ce
   sortTable[0x6D4B]=2681;   // ce
   sortTable[0x6E2C]=2682;   // ce
   sortTable[0x4FA7]=2683;   // ce
   sortTable[0x5074]=2684;   // ce
   sortTable[0x518C]=2685;   // ce
   sortTable[0x518A]=2686;   // ce
   sortTable[0x5395]=2687;   // ce
   sortTable[0x5EC1]=2688;   // ce
   sortTable[0x607B]=2689;   // ce
   sortTable[0x7B74]=2690;   // ce
   sortTable[0x7CA3]=2691;   // ce
   sortTable[0x84DB]=2692;   // ce
   sortTable[0x7BA3]=2693;   // ce
   sortTable[0x7C0E]=2694;   // ce
   sortTable[0x8417]=2695;   // ce
   sortTable[0x7B27]=2696;   // ce
   sortTable[0x8434]=2697;   // ce
   sortTable[0x53A0]=2698;   // ce
   sortTable[0x5884]=2699;   // ce
   sortTable[0x60FB]=2700;   // ce
   sortTable[0x61A1]=2701;   // ce
   sortTable[0x62FA]=2702;   // ce
   sortTable[0x6547]=2703;   // ce
   sortTable[0x755F]=2704;   // ce
   sortTable[0x7B5E]=2705;   // ce
   sortTable[0x5C91]=2706;   // cen
   sortTable[0x6D94]=2707;   // cen
   sortTable[0x7B12]=2708;   // cen
   sortTable[0x5D7E]=2709;   // cen
   sortTable[0x68A3]=2710;   // cen
   sortTable[0x5C42]=2711;   // ceng
   sortTable[0x5C64]=2712;   // ceng
   sortTable[0x66FE]=2713;   // ceng
   sortTable[0x8E6D]=2714;   // ceng
   sortTable[0x564C]=2715;   // ceng
   sortTable[0x9A53]=2716;   // ceng
   sortTable[0x5D92]=2717;   // ceng
   sortTable[0x66FD]=2718;   // ceng
   sortTable[0x7AF2]=2719;   // ceng
   sortTable[0x67E5]=2720;   // cha
   sortTable[0x5BDF]=2721;   // cha
   sortTable[0x5DEE]=2722;   // cha
   sortTable[0x8336]=2723;   // cha
   sortTable[0x63D2]=2724;   // cha
   sortTable[0x8BE7]=2725;   // cha
   sortTable[0x8A6B]=2726;   // cha
   sortTable[0x53C9]=2727;   // cha
   sortTable[0x5C94]=2728;   // cha
   sortTable[0x9538]=2729;   // cha
   sortTable[0x69CE]=2730;   // cha
   sortTable[0x832C]=2731;   // cha
   sortTable[0x8869]=2732;   // cha
   sortTable[0x59F9]=2733;   // cha
   sortTable[0x643D]=2734;   // cha
   sortTable[0x6748]=2735;   // cha
   sortTable[0x78B4]=2736;   // cha
   sortTable[0x9987]=2737;   // cha
   sortTable[0x6C4A]=2738;   // cha
   sortTable[0x8A67]=2739;   // cha
   sortTable[0x6AAB]=2740;   // cha
   sortTable[0x929F]=2741;   // cha
   sortTable[0x8256]=2742;   // cha
   sortTable[0x79C5]=2743;   // cha
   sortTable[0x81FF]=2744;   // cha
   sortTable[0x8E45]=2745;   // cha
   sortTable[0x7339]=2746;   // cha
   sortTable[0x9572]=2747;   // cha
   sortTable[0x976B]=2748;   // cha
   sortTable[0x809E]=2749;   // cha
   sortTable[0x9364]=2750;   // cha
   sortTable[0x4F98]=2751;   // cha
   sortTable[0x505B]=2752;   // cha
   sortTable[0x55CF]=2753;   // cha
   sortTable[0x579E]=2754;   // cha
   sortTable[0x597C]=2755;   // cha
   sortTable[0x5D56]=2756;   // cha
   sortTable[0x6260]=2757;   // cha
   sortTable[0x633F]=2758;   // cha
   sortTable[0x63F7]=2759;   // cha
   sortTable[0x7580]=2760;   // cha
   sortTable[0x7D01]=2761;   // cha
   sortTable[0x9454]=2762;   // cha
   sortTable[0x9937]=2763;   // cha
   sortTable[0x67F4]=2764;   // chai
   sortTable[0x62C6]=2765;   // chai
   sortTable[0x8C7A]=2766;   // chai
   sortTable[0x9497]=2767;   // chai
   sortTable[0x867F]=2768;   // chai
   sortTable[0x4FAA]=2769;   // chai
   sortTable[0x7625]=2770;   // chai
   sortTable[0x831D]=2771;   // chai
   sortTable[0x8883]=2772;   // chai
   sortTable[0x8A0D]=2773;   // chai
   sortTable[0x7961]=2774;   // chai
   sortTable[0x8806]=2775;   // chai
   sortTable[0x9F5C]=2776;   // chai
   sortTable[0x5115]=2777;   // chai
   sortTable[0x558D]=2778;   // chai
   sortTable[0x56C6]=2779;   // chai
   sortTable[0x8286]=2780;   // chai
   sortTable[0x91F5]=2781;   // chai
   sortTable[0x4EA7]=2782;   // chan
   sortTable[0x7522]=2783;   // chan
   sortTable[0x98A4]=2784;   // chan
   sortTable[0x986B]=2785;   // chan
   sortTable[0x8749]=2786;   // chan
   sortTable[0x87EC]=2787;   // chan
   sortTable[0x7F20]=2788;   // chan
   sortTable[0x7E8F]=2789;   // chan
   sortTable[0x94F2]=2790;   // chan
   sortTable[0x93DF]=2791;   // chan
   sortTable[0x6400]=2792;   // chan
   sortTable[0x6519]=2793;   // chan
   sortTable[0x9610]=2794;   // chan
   sortTable[0x95E1]=2795;   // chan
   sortTable[0x998B]=2796;   // chan
   sortTable[0x995E]=2797;   // chan
   sortTable[0x7985]=2798;   // chan
   sortTable[0x8C17]=2799;   // chan
   sortTable[0x8C04]=2800;   // chan
   sortTable[0x87FE]=2801;   // chan
   sortTable[0x5FCF]=2802;   // chan
   sortTable[0x97C2]=2803;   // chan
   sortTable[0x5A75]=2804;   // chan
   sortTable[0x8E94]=2805;   // chan
   sortTable[0x6FB6]=2806;   // chan
   sortTable[0x6F7A]=2807;   // chan
   sortTable[0x89C7]=2808;   // chan
   sortTable[0x5EDB]=2809;   // chan
   sortTable[0x92CB]=2810;   // chan
   sortTable[0x9561]=2811;   // chan
   sortTable[0x895C]=2812;   // chan
   sortTable[0x7FBC]=2813;   // chan
   sortTable[0x8487]=2814;   // chan
   sortTable[0x5181]=2815;   // chan
   sortTable[0x9575]=2816;   // chan
   sortTable[0x913D]=2817;   // chan
   sortTable[0x8546]=2818;   // chan
   sortTable[0x8FBF]=2819;   // chan
   sortTable[0x8A97]=2820;   // chan
   sortTable[0x7E5F]=2821;   // chan
   sortTable[0x785F]=2822;   // chan
   sortTable[0x9471]=2823;   // chan
   sortTable[0x9141]=2824;   // chan
   sortTable[0x8B42]=2825;   // chan
   sortTable[0x9AA3]=2826;   // chan
   sortTable[0x7DFE]=2827;   // chan
   sortTable[0x88E7]=2828;   // chan
   sortTable[0x8B87]=2829;   // chan
   sortTable[0x95B3]=2830;   // chan
   sortTable[0x79AA]=2831;   // chan
   sortTable[0x7C05]=2832;   // chan
   sortTable[0x7E92]=2833;   // chan
   sortTable[0x8998]=2834;   // chan
   sortTable[0x8AC2]=2835;   // chan
   sortTable[0x4E33]=2836;   // chan
   sortTable[0x50DD]=2837;   // chan
   sortTable[0x5103]=2838;   // chan
   sortTable[0x5133]=2839;   // chan
   sortTable[0x522C]=2840;   // chan
   sortTable[0x5257]=2841;   // chan
   sortTable[0x5277]=2842;   // chan
   sortTable[0x5296]=2843;   // chan
   sortTable[0x5574]=2844;   // chan
   sortTable[0x563D]=2845;   // chan
   sortTable[0x56B5]=2846;   // chan
   sortTable[0x56C5]=2847;   // chan
   sortTable[0x58E5]=2848;   // chan
   sortTable[0x5B0B]=2849;   // chan
   sortTable[0x5D7C]=2850;   // chan
   sortTable[0x5DC9]=2851;   // chan
   sortTable[0x5E5D]=2852;   // chan
   sortTable[0x5E68]=2853;   // chan
   sortTable[0x61F4]=2854;   // chan
   sortTable[0x61FA]=2855;   // chan
   sortTable[0x644C]=2856;   // chan
   sortTable[0x6472]=2857;   // chan
   sortTable[0x65BA]=2858;   // chan
   sortTable[0x65F5]=2859;   // chan
   sortTable[0x68B4]=2860;   // chan
   sortTable[0x68CE]=2861;   // chan
   sortTable[0x6B03]=2862;   // chan
   sortTable[0x6BDA]=2863;   // chan
   sortTable[0x6D50]=2864;   // chan
   sortTable[0x6E79]=2865;   // chan
   sortTable[0x6EFB]=2866;   // chan
   sortTable[0x6F79]=2867;   // chan
   sortTable[0x700D]=2868;   // chan
   sortTable[0x703A]=2869;   // chan
   sortTable[0x705B]=2870;   // chan
   sortTable[0x7158]=2871;   // chan
   sortTable[0x71C0]=2872;   // chan
   sortTable[0x7351]=2873;   // chan
   sortTable[0x7523]=2874;   // chan
   sortTable[0x78DB]=2875;   // chan
   sortTable[0x826C]=2876;   // chan
   sortTable[0x8B92]=2877;   // chan
   sortTable[0x8FB4]=2878;   // chan
   sortTable[0x9246]=2879;   // chan
   sortTable[0x92D3]=2880;   // chan
   sortTable[0x5E38]=2881;   // chang
   sortTable[0x573A]=2882;   // chang
   sortTable[0x5834]=2883;   // chang
   sortTable[0x5382]=2884;   // chang
   sortTable[0x5EE0]=2885;   // chang
   sortTable[0x5531]=2886;   // chang
   sortTable[0x80A0]=2887;   // chang
   sortTable[0x8178]=2888;   // chang
   sortTable[0x5021]=2889;   // chang
   sortTable[0x5C1D]=2890;   // chang
   sortTable[0x5617]=2891;   // chang
   sortTable[0x655E]=2892;   // chang
   sortTable[0x7545]=2893;   // chang
   sortTable[0x66A2]=2894;   // chang
   sortTable[0x507F]=2895;   // chang
   sortTable[0x511F]=2896;   // chang
   sortTable[0x660C]=2897;   // chang
   sortTable[0x6005]=2898;   // chang
   sortTable[0x5AE6]=2899;   // chang
   sortTable[0x5A3C]=2900;   // chang
   sortTable[0x6636]=2901;   // chang
   sortTable[0x7316]=2902;   // chang
   sortTable[0x82CC]=2903;   // chang
   sortTable[0x9B2F]=2904;   // chang
   sortTable[0x960A]=2905;   // chang
   sortTable[0x6C05]=2906;   // chang
   sortTable[0x83D6]=2907;   // chang
   sortTable[0x4F25]=2908;   // chang
   sortTable[0x5F9C]=2909;   // chang
   sortTable[0x92F9]=2910;   // chang
   sortTable[0x60DD]=2911;   // chang
   sortTable[0x9CB3]=2912;   // chang
   sortTable[0x9CBF]=2913;   // chang
   sortTable[0x97D4]=2914;   // chang
   sortTable[0x95B6]=2915;   // chang
   sortTable[0x9520]=2916;   // chang
   sortTable[0x92FF]=2917;   // chang
   sortTable[0x9C68]=2918;   // chang
   sortTable[0x9F1A]=2919;   // chang
   sortTable[0x4EE7]=2920;   // chang
   sortTable[0x5000]=2921;   // chang
   sortTable[0x50D8]=2922;   // chang
   sortTable[0x514F]=2923;   // chang
   sortTable[0x53B0]=2924;   // chang
   sortTable[0x5690]=2925;   // chang
   sortTable[0x5872]=2926;   // chang
   sortTable[0x60B5]=2927;   // chang
   sortTable[0x667F]=2928;   // chang
   sortTable[0x6919]=2929;   // chang
   sortTable[0x6DD0]=2930;   // chang
   sortTable[0x713B]=2931;   // chang
   sortTable[0x739A]=2932;   // chang
   sortTable[0x7429]=2933;   // chang
   sortTable[0x7452]=2934;   // chang
   sortTable[0x747A]=2935;   // chang
   sortTable[0x74FA]=2936;   // chang
   sortTable[0x751E]=2937;   // chang
   sortTable[0x757C]=2938;   // chang
   sortTable[0x8193]=2939;   // chang
   sortTable[0x8407]=2940;   // chang
   sortTable[0x87D0]=2941;   // chang
   sortTable[0x88EE]=2942;   // chang
   sortTable[0x8AAF]=2943;   // chang
   sortTable[0x9329]=2944;   // chang
   sortTable[0x93DB]=2945;   // chang
   sortTable[0x9578]=2946;   // chang
   sortTable[0x9BE7]=2947;   // chang
   sortTable[0x671D]=2948;   // chao
   sortTable[0x8D85]=2949;   // chao
   sortTable[0x6F6E]=2950;   // chao
   sortTable[0x5435]=2951;   // chao
   sortTable[0x6284]=2952;   // chao
   sortTable[0x5DE2]=2953;   // chao
   sortTable[0x5632]=2954;   // chao
   sortTable[0x949E]=2955;   // chao
   sortTable[0x9214]=2956;   // chao
   sortTable[0x7092]=2957;   // chao
   sortTable[0x6641]=2958;   // chao
   sortTable[0x712F]=2959;   // chao
   sortTable[0x8016]=2960;   // chao
   sortTable[0x9F02]=2961;   // chao
   sortTable[0x600A]=2962;   // chao
   sortTable[0x8B3F]=2963;   // chao
   sortTable[0x911B]=2964;   // chao
   sortTable[0x9EA8]=2965;   // chao
   sortTable[0x8A2C]=2966;   // chao
   sortTable[0x7727]=2967;   // chao
   sortTable[0x7F7A]=2968;   // chao
   sortTable[0x4EE6]=2969;   // chao
   sortTable[0x4EEF]=2970;   // chao
   sortTable[0x52E6]=2971;   // chao
   sortTable[0x5DD0]=2972;   // chao
   sortTable[0x5DE3]=2973;   // chao
   sortTable[0x5F28]=2974;   // chao
   sortTable[0x6A14]=2975;   // chao
   sortTable[0x6B29]=2976;   // chao
   sortTable[0x6F05]=2977;   // chao
   sortTable[0x7123]=2978;   // chao
   sortTable[0x717C]=2979;   // chao
   sortTable[0x724A]=2980;   // chao
   sortTable[0x7AB2]=2981;   // chao
   sortTable[0x89D8]=2982;   // chao
   sortTable[0x8F48]=2983;   // chao
   sortTable[0x9F0C]=2984;   // chao
   sortTable[0x8F66]=2985;   // che
   sortTable[0x8ECA]=2986;   // che
   sortTable[0x5F7B]=2987;   // che
   sortTable[0x5FB9]=2988;   // che
   sortTable[0x64A4]=2989;   // che
   sortTable[0x626F]=2990;   // che
   sortTable[0x63A3]=2991;   // che
   sortTable[0x6F88]=2992;   // che
   sortTable[0x7817]=2993;   // che
   sortTable[0x577C]=2994;   // che
   sortTable[0x77AE]=2995;   // che
   sortTable[0x7869]=2996;   // che
   sortTable[0x86FC]=2997;   // che
   sortTable[0x8045]=2998;   // che
   sortTable[0x4F21]=2999;   // che
   sortTable[0x4FE5]=3000;   // che
   sortTable[0x5056]=3001;   // che
   sortTable[0x52F6]=3002;   // che
   sortTable[0x5513]=3003;   // che
   sortTable[0x5C6E]=3004;   // che
   sortTable[0x64A6]=3005;   // che
   sortTable[0x70E2]=3006;   // che
   sortTable[0x7221]=3007;   // che
   sortTable[0x7868]=3008;   // che
   sortTable[0x8397]=3009;   // che
   sortTable[0x8FE0]=3010;   // che
   sortTable[0x9819]=3011;   // che
   sortTable[0x6C89]=3012;   // chen
   sortTable[0x6668]=3013;   // chen
   sortTable[0x8D81]=3014;   // chen
   sortTable[0x5C18]=3015;   // chen
   sortTable[0x5875]=3016;   // chen
   sortTable[0x886C]=3017;   // chen
   sortTable[0x896F]=3018;   // chen
   sortTable[0x9648]=3019;   // chen
   sortTable[0x9673]=3020;   // chen
   sortTable[0x81E3]=3021;   // chen
   sortTable[0x5FF1]=3022;   // chen
   sortTable[0x8FB0]=3023;   // chen
   sortTable[0x741B]=3024;   // chen
   sortTable[0x5BB8]=3025;   // chen
   sortTable[0x8C0C]=3026;   // chen
   sortTable[0x55D4]=3027;   // chen
   sortTable[0x8C36]=3028;   // chen
   sortTable[0x90F4]=3029;   // chen
   sortTable[0x62BB]=3030;   // chen
   sortTable[0x6987]=3031;   // chen
   sortTable[0x789C]=3032;   // chen
   sortTable[0x9F80]=3033;   // chen
   sortTable[0x7D9D]=3034;   // chen
   sortTable[0x778B]=3035;   // chen
   sortTable[0x91A6]=3036;   // chen
   sortTable[0x9703]=3037;   // chen
   sortTable[0x831E]=3038;   // chen
   sortTable[0x852F]=3039;   // chen
   sortTable[0x8CDD]=3040;   // chen
   sortTable[0x9E8E]=3041;   // chen
   sortTable[0x8D02]=3042;   // chen
   sortTable[0x8D82]=3043;   // chen
   sortTable[0x8ED9]=3044;   // chen
   sortTable[0x8DBB]=3045;   // chen
   sortTable[0x8FE7]=3046;   // chen
   sortTable[0x9356]=3047;   // chen
   sortTable[0x8B96]=3048;   // chen
   sortTable[0x7876]=3049;   // chen
   sortTable[0x78E3]=3050;   // chen
   sortTable[0x8390]=3051;   // chen
   sortTable[0x8E38]=3052;   // chen
   sortTable[0x9202]=3053;   // chen
   sortTable[0x9F53]=3054;   // chen
   sortTable[0x512C]=3055;   // chen
   sortTable[0x512D]=3056;   // chen
   sortTable[0x56AB]=3057;   // chen
   sortTable[0x588B]=3058;   // chen
   sortTable[0x5926]=3059;   // chen
   sortTable[0x6116]=3060;   // chen
   sortTable[0x6375]=3061;   // chen
   sortTable[0x63E8]=3062;   // chen
   sortTable[0x6550]=3063;   // chen
   sortTable[0x66DF]=3064;   // chen
   sortTable[0x6A04]=3065;   // chen
   sortTable[0x6AEC]=3066;   // chen
   sortTable[0x70E5]=3067;   // chen
   sortTable[0x7141]=3068;   // chen
   sortTable[0x75A2]=3069;   // chen
   sortTable[0x760E]=3070;   // chen
   sortTable[0x7E1D]=3071;   // chen
   sortTable[0x8380]=3072;   // chen
   sortTable[0x85BC]=3073;   // chen
   sortTable[0x87B4]=3074;   // chen
   sortTable[0x8A26]=3075;   // chen
   sortTable[0x8AC3]=3076;   // chen
   sortTable[0x8AF6]=3077;   // chen
   sortTable[0x8B13]=3078;   // chen
   sortTable[0x9DD0]=3079;   // chen
   sortTable[0x9F54]=3080;   // chen
   sortTable[0x6210]=3081;   // cheng
   sortTable[0x7A0B]=3082;   // cheng
   sortTable[0x57CE]=3083;   // cheng
   sortTable[0x79F0]=3084;   // cheng
   sortTable[0x7A31]=3085;   // cheng
   sortTable[0x627F]=3086;   // cheng
   sortTable[0x4E58]=3087;   // cheng
   sortTable[0x8BDA]=3088;   // cheng
   sortTable[0x8AA0]=3089;   // cheng
   sortTable[0x5448]=3090;   // cheng
   sortTable[0x6491]=3091;   // cheng
   sortTable[0x6490]=3092;   // cheng
   sortTable[0x60E9]=3093;   // cheng
   sortTable[0x79E4]=3094;   // cheng
   sortTable[0x61F2]=3095;   // cheng
   sortTable[0x4E1E]=3096;   // cheng
   sortTable[0x6F84]=3097;   // cheng
   sortTable[0x901E]=3098;   // cheng
   sortTable[0x665F]=3099;   // cheng
   sortTable[0x9A8B]=3100;   // cheng
   sortTable[0x6A59]=3101;   // cheng
   sortTable[0x88CE]=3102;   // cheng
   sortTable[0x77A0]=3103;   // cheng
   sortTable[0x86CF]=3104;   // cheng
   sortTable[0x94D6]=3105;   // cheng
   sortTable[0x67A8]=3106;   // cheng
   sortTable[0x90D5]=3107;   // cheng
   sortTable[0x9172]=3108;   // cheng
   sortTable[0x584D]=3109;   // cheng
   sortTable[0x67FD]=3110;   // cheng
   sortTable[0x7748]=3111;   // cheng
   sortTable[0x8D6A]=3112;   // cheng
   sortTable[0x57D5]=3113;   // cheng
   sortTable[0x8100]=3114;   // cheng
   sortTable[0x7A6A]=3115;   // cheng
   sortTable[0x812D]=3116;   // cheng
   sortTable[0x9833]=3117;   // cheng
   sortTable[0x9A2C]=3118;   // cheng
   sortTable[0x8D6C]=3119;   // cheng
   sortTable[0x7AC0]=3120;   // cheng
   sortTable[0x7D7E]=3121;   // cheng
   sortTable[0x7880]=3122;   // cheng
   sortTable[0x7DFD]=3123;   // cheng
   sortTable[0x93F3]=3124;   // cheng
   sortTable[0x9953]=3125;   // cheng
   sortTable[0x9A01]=3126;   // cheng
   sortTable[0x7A9A]=3127;   // cheng
   sortTable[0x93FF]=3128;   // cheng
   sortTable[0x4E57]=3129;   // cheng
   sortTable[0x4FB1]=3130;   // cheng
   sortTable[0x5041]=3131;   // cheng
   sortTable[0x50DC]=3132;   // cheng
   sortTable[0x5818]=3133;   // cheng
   sortTable[0x5856]=3134;   // cheng
   sortTable[0x5A0D]=3135;   // cheng
   sortTable[0x5BAC]=3136;   // cheng
   sortTable[0x5CF8]=3137;   // cheng
   sortTable[0x5EB1]=3138;   // cheng
   sortTable[0x5F8E]=3139;   // cheng
   sortTable[0x609C]=3140;   // cheng
   sortTable[0x6186]=3141;   // cheng
   sortTable[0x6195]=3142;   // cheng
   sortTable[0x6330]=3143;   // cheng
   sortTable[0x6381]=3144;   // cheng
   sortTable[0x645A]=3145;   // cheng
   sortTable[0x673E]=3146;   // cheng
   sortTable[0x68D6]=3147;   // cheng
   sortTable[0x68E6]=3148;   // cheng
   sortTable[0x6909]=3149;   // cheng
   sortTable[0x6A55]=3150;   // cheng
   sortTable[0x6A89]=3151;   // cheng
   sortTable[0x6A99]=3152;   // cheng
   sortTable[0x6CDF]=3153;   // cheng
   sortTable[0x6D06]=3154;   // cheng
   sortTable[0x6D7E]=3155;   // cheng
   sortTable[0x6E5E]=3156;   // cheng
   sortTable[0x6E97]=3157;   // cheng
   sortTable[0x6F82]=3158;   // cheng
   sortTable[0x7013]=3159;   // cheng
   sortTable[0x722F]=3160;   // cheng
   sortTable[0x725A]=3161;   // cheng
   sortTable[0x73F5]=3162;   // cheng
   sortTable[0x73F9]=3163;   // cheng
   sortTable[0x7424]=3164;   // cheng
   sortTable[0x757B]=3165;   // cheng
   sortTable[0x7B6C]=3166;   // cheng
   sortTable[0x837F]=3167;   // cheng
   sortTable[0x87F6]=3168;   // cheng
   sortTable[0x92EE]=3169;   // cheng
   sortTable[0x9637]=3170;   // cheng
   sortTable[0x9757]=3171;   // cheng
   sortTable[0x9BCE]=3172;   // cheng
   sortTable[0x5403]=3173;   // chi
   sortTable[0x6301]=3174;   // chi
   sortTable[0x5C3A]=3175;   // chi
   sortTable[0x8FDF]=3176;   // chi
   sortTable[0x9072]=3177;   // chi
   sortTable[0x7FC5]=3178;   // chi
   sortTable[0x9F7F]=3179;   // chi
   sortTable[0x9F52]=3180;   // chi
   sortTable[0x6C60]=3181;   // chi
   sortTable[0x803B]=3182;   // chi
   sortTable[0x6065]=3183;   // chi
   sortTable[0x9A70]=3184;   // chi
   sortTable[0x65A5]=3185;   // chi
   sortTable[0x54E7]=3186;   // chi
   sortTable[0x99B3]=3187;   // chi
   sortTable[0x8D64]=3188;   // chi
   sortTable[0x86A9]=3189;   // chi
   sortTable[0x6555]=3190;   // chi
   sortTable[0x75F4]=3191;   // chi
   sortTable[0x53F1]=3192;   // chi
   sortTable[0x4F88]=3193;   // chi
   sortTable[0x996C]=3194;   // chi
   sortTable[0x70BD]=3195;   // chi
   sortTable[0x55E4]=3196;   // chi
   sortTable[0x5F1B]=3197;   // chi
   sortTable[0x7B1E]=3198;   // chi
   sortTable[0x761B]=3199;   // chi
   sortTable[0x557B]=3200;   // chi
   sortTable[0x9B51]=3201;   // chi
   sortTable[0x5880]=3202;   // chi
   sortTable[0x9E31]=3203;   // chi
   sortTable[0x5AB8]=3204;   // chi
   sortTable[0x892B]=3205;   // chi
   sortTable[0x87AD]=3206;   // chi
   sortTable[0x50BA]=3207;   // chi
   sortTable[0x577B]=3208;   // chi
   sortTable[0x8E1F]=3209;   // chi
   sortTable[0x7BEA]=3210;   // chi
   sortTable[0x7D7A]=3211;   // chi
   sortTable[0x830C]=3212;   // chi
   sortTable[0x9D92]=3213;   // chi
   sortTable[0x994E]=3214;   // chi
   sortTable[0x7735]=3215;   // chi
   sortTable[0x7AFE]=3216;   // chi
   sortTable[0x7C8E]=3217;   // chi
   sortTable[0x86B3]=3218;   // chi
   sortTable[0x906B]=3219;   // chi
   sortTable[0x7B8E]=3220;   // chi
   sortTable[0x8D8D]=3221;   // chi
   sortTable[0x8DA9]=3222;   // chi
   sortTable[0x98ED]=3223;   // chi
   sortTable[0x80F5]=3224;   // chi
   sortTable[0x9D1F]=3225;   // chi
   sortTable[0x7FE8]=3226;   // chi
   sortTable[0x834E]=3227;   // chi
   sortTable[0x96F4]=3228;   // chi
   sortTable[0x779D]=3229;   // chi
   sortTable[0x80E3]=3230;   // chi
   sortTable[0x88B2]=3231;   // chi
   sortTable[0x8B18]=3232;   // chi
   sortTable[0x7B42]=3233;   // chi
   sortTable[0x8687]=3234;   // chi
   sortTable[0x88ED]=3235;   // chi
   sortTable[0x8CBE]=3236;   // chi
   sortTable[0x8DEE]=3237;   // chi
   sortTable[0x9279]=3238;   // chi
   sortTable[0x9DD8]=3239;   // chi
   sortTable[0x9ED0]=3240;   // chi
   sortTable[0x9F5D]=3241;   // chi
   sortTable[0x4F99]=3242;   // chi
   sortTable[0x52C5]=3243;   // chi
   sortTable[0x52D1]=3244;   // chi
   sortTable[0x5376]=3245;   // chi
   sortTable[0x53FA]=3246;   // chi
   sortTable[0x544E]=3247;   // chi
   sortTable[0x55AB]=3248;   // chi
   sortTable[0x5644]=3249;   // chi
   sortTable[0x5791]=3250;   // chi
   sortTable[0x599B]=3251;   // chi
   sortTable[0x5CBB]=3252;   // chi
   sortTable[0x5F68]=3253;   // chi
   sortTable[0x5F72]=3254;   // chi
   sortTable[0x5F73]=3255;   // chi
   sortTable[0x605C]=3256;   // chi
   sortTable[0x6157]=3257;   // chi
   sortTable[0x618F]=3258;   // chi
   sortTable[0x61D8]=3259;   // chi
   sortTable[0x62B6]=3260;   // chi
   sortTable[0x645B]=3261;   // chi
   sortTable[0x6758]=3262;   // chi
   sortTable[0x6B3C]=3263;   // chi
   sortTable[0x6B6D]=3264;   // chi
   sortTable[0x6B6F]=3265;   // chi
   sortTable[0x6E41]=3266;   // chi
   sortTable[0x6F26]=3267;   // chi
   sortTable[0x707B]=3268;   // chi
   sortTable[0x70FE]=3269;   // chi
   sortTable[0x71BE]=3270;   // chi
   sortTable[0x74FB]=3271;   // chi
   sortTable[0x75D3]=3272;   // chi
   sortTable[0x75F8]=3273;   // chi
   sortTable[0x7608]=3274;   // chi
   sortTable[0x7661]=3275;   // chi
   sortTable[0x7873]=3276;   // chi
   sortTable[0x7FC4]=3277;   // chi
   sortTable[0x7FE4]=3278;   // chi
   sortTable[0x815F]=3279;   // chi
   sortTable[0x88B3]=3280;   // chi
   sortTable[0x8A35]=3281;   // chi
   sortTable[0x8ABA]=3282;   // chi
   sortTable[0x8D7F]=3283;   // chi
   sortTable[0x9045]=3284;   // chi
   sortTable[0x905F]=3285;   // chi
   sortTable[0x9253]=3286;   // chi
   sortTable[0x9290]=3287;   // chi
   sortTable[0x9EB6]=3288;   // chi
   sortTable[0x866B]=3289;   // chong
   sortTable[0x51B2]=3290;   // chong
   sortTable[0x6C96]=3291;   // chong
   sortTable[0x5145]=3292;   // chong
   sortTable[0x5D07]=3293;   // chong
   sortTable[0x61A7]=3294;   // chong
   sortTable[0x5BA0]=3295;   // chong
   sortTable[0x5FE1]=3296;   // chong
   sortTable[0x8202]=3297;   // chong
   sortTable[0x825F]=3298;   // chong
   sortTable[0x94F3]=3299;   // chong
   sortTable[0x7FC0]=3300;   // chong
   sortTable[0x87F2]=3301;   // chong
   sortTable[0x833A]=3302;   // chong
   sortTable[0x7F7F]=3303;   // chong
   sortTable[0x885D]=3304;   // chong
   sortTable[0x7DDF]=3305;   // chong
   sortTable[0x8E56]=3306;   // chong
   sortTable[0x9680]=3307;   // chong
   sortTable[0x5603]=3308;   // chong
   sortTable[0x57EB]=3309;   // chong
   sortTable[0x5BF5]=3310;   // chong
   sortTable[0x5D08]=3311;   // chong
   sortTable[0x5FB8]=3312;   // chong
   sortTable[0x6183]=3313;   // chong
   sortTable[0x63F0]=3314;   // chong
   sortTable[0x644F]=3315;   // chong
   sortTable[0x6D7A]=3316;   // chong
   sortTable[0x721E]=3317;   // chong
   sortTable[0x73EB]=3318;   // chong
   sortTable[0x8769]=3319;   // chong
   sortTable[0x8908]=3320;   // chong
   sortTable[0x9283]=3321;   // chong
   sortTable[0x62BD]=3322;   // chou
   sortTable[0x81ED]=3323;   // chou
   sortTable[0x4EC7]=3324;   // chou
   sortTable[0x6101]=3325;   // chou
   sortTable[0x7785]=3326;   // chou
   sortTable[0x4E11]=3327;   // chou
   sortTable[0x7EF8]=3328;   // chou
   sortTable[0x8E0C]=3329;   // chou
   sortTable[0x8E8A]=3330;   // chou
   sortTable[0x7DA2]=3331;   // chou
   sortTable[0x916C]=3332;   // chou
   sortTable[0x7A20]=3333;   // chou
   sortTable[0x7B79]=3334;   // chou
   sortTable[0x7574]=3335;   // chou
   sortTable[0x4FE6]=3336;   // chou
   sortTable[0x96E0]=3337;   // chou
   sortTable[0x60C6]=3338;   // chou
   sortTable[0x7633]=3339;   // chou
   sortTable[0x5E31]=3340;   // chou
   sortTable[0x77C1]=3341;   // chou
   sortTable[0x88EF]=3342;   // chou
   sortTable[0x7D2C]=3343;   // chou
   sortTable[0x7BD8]=3344;   // chou
   sortTable[0x8B8E]=3345;   // chou
   sortTable[0x9167]=3346;   // chou
   sortTable[0x91BB]=3347;   // chou
   sortTable[0x96D4]=3348;   // chou
   sortTable[0x919C]=3349;   // chou
   sortTable[0x7C4C]=3350;   // chou
   sortTable[0x83D7]=3351;   // chou
   sortTable[0x7D52]=3352;   // chou
   sortTable[0x8B90]=3353;   // chou
   sortTable[0x4E12]=3354;   // chou
   sortTable[0x4FB4]=3355;   // chou
   sortTable[0x5062]=3356;   // chou
   sortTable[0x5114]=3357;   // chou
   sortTable[0x541C]=3358;   // chou
   sortTable[0x568B]=3359;   // chou
   sortTable[0x5A64]=3360;   // chou
   sortTable[0x5B26]=3361;   // chou
   sortTable[0x5E6C]=3362;   // chou
   sortTable[0x601E]=3363;   // chou
   sortTable[0x61E4]=3364;   // chou
   sortTable[0x640A]=3365;   // chou
   sortTable[0x677B]=3366;   // chou
   sortTable[0x677D]=3367;   // chou
   sortTable[0x6826]=3368;   // chou
   sortTable[0x6906]=3369;   // chou
   sortTable[0x6BA0]=3370;   // chou
   sortTable[0x71FD]=3371;   // chou
   sortTable[0x72A8]=3372;   // chou
   sortTable[0x72AB]=3373;   // chou
   sortTable[0x7587]=3374;   // chou
   sortTable[0x7697]=3375;   // chou
   sortTable[0x81F0]=3376;   // chou
   sortTable[0x85B5]=3377;   // chou
   sortTable[0x905A]=3378;   // chou
   sortTable[0x9B57]=3379;   // chou
   sortTable[0x51FA]=3380;   // chu
   sortTable[0x5904]=3381;   // chu
   sortTable[0x8655]=3382;   // chu
   sortTable[0x9664]=3383;   // chu
   sortTable[0x521D]=3384;   // chu
   sortTable[0x7840]=3385;   // chu
   sortTable[0x790E]=3386;   // chu
   sortTable[0x695A]=3387;   // chu
   sortTable[0x89E6]=3388;   // chu
   sortTable[0x89F8]=3389;   // chu
   sortTable[0x53A8]=3390;   // chu
   sortTable[0x5EDA]=3391;   // chu
   sortTable[0x50A8]=3392;   // chu
   sortTable[0x5132]=3393;   // chu
   sortTable[0x8E87]=3394;   // chu
   sortTable[0x9504]=3395;   // chu
   sortTable[0x92E4]=3396;   // chu
   sortTable[0x755C]=3397;   // chu
   sortTable[0x77D7]=3398;   // chu
   sortTable[0x891A]=3399;   // chu
   sortTable[0x9EDC]=3400;   // chu
   sortTable[0x6A71]=3401;   // chu
   sortTable[0x520D]=3402;   // chu
   sortTable[0x96CF]=3403;   // chu
   sortTable[0x6775]=3404;   // chu
   sortTable[0x6410]=3405;   // chu
   sortTable[0x6EC1]=3406;   // chu
   sortTable[0x6A17]=3407;   // chu
   sortTable[0x7ECC]=3408;   // chu
   sortTable[0x696E]=3409;   // chu
   sortTable[0x6035]=3410;   // chu
   sortTable[0x8E70]=3411;   // chu
   sortTable[0x61B7]=3412;   // chu
   sortTable[0x870D]=3413;   // chu
   sortTable[0x924F]=3414;   // chu
   sortTable[0x8C99]=3415;   // chu
   sortTable[0x84A2]=3416;   // chu
   sortTable[0x81C5]=3417;   // chu
   sortTable[0x7BE8]=3418;   // chu
   sortTable[0x9F7C]=3419;   // chu
   sortTable[0x84AD]=3420;   // chu
   sortTable[0x9E00]=3421;   // chu
   sortTable[0x95A6]=3422;   // chu
   sortTable[0x9DB5]=3423;   // chu
   sortTable[0x84EB]=3424;   // chu
   sortTable[0x85F8]=3425;   // chu
   sortTable[0x9F63]=3426;   // chu
   sortTable[0x7ACC]=3427;   // chu
   sortTable[0x8021]=3428;   // chu
   sortTable[0x854F]=3429;   // chu
   sortTable[0x8C56]=3430;   // chu
   sortTable[0x82BB]=3431;   // chu
   sortTable[0x87F5]=3432;   // chu
   sortTable[0x8AD4]=3433;   // chu
   sortTable[0x8C60]=3434;   // chu
   sortTable[0x8E00]=3435;   // chu
   sortTable[0x9110]=3436;   // chu
   sortTable[0x4E8D]=3437;   // chu
   sortTable[0x4FF6]=3438;   // chu
   sortTable[0x5097]=3439;   // chu
   sortTable[0x510A]=3440;   // chu
   sortTable[0x51E6]=3441;   // chu
   sortTable[0x563C]=3442;   // chu
   sortTable[0x57F1]=3443;   // chu
   sortTable[0x5AB0]=3444;   // chu
   sortTable[0x5C80]=3445;   // chu
   sortTable[0x5E6E]=3446;   // chu
   sortTable[0x62C0]=3447;   // chu
   sortTable[0x6474]=3448;   // chu
   sortTable[0x654A]=3449;   // chu
   sortTable[0x65B6]=3450;   // chu
   sortTable[0x67F7]=3451;   // chu
   sortTable[0x6918]=3452;   // chu
   sortTable[0x698B]=3453;   // chu
   sortTable[0x6A7B]=3454;   // chu
   sortTable[0x6A9A]=3455;   // chu
   sortTable[0x6AC9]=3456;   // chu
   sortTable[0x6AE5]=3457;   // chu
   sortTable[0x6B2A]=3458;   // chu
   sortTable[0x6B5C]=3459;   // chu
   sortTable[0x6EC0]=3460;   // chu
   sortTable[0x6FCB]=3461;   // chu
   sortTable[0x7293]=3462;   // chu
   sortTable[0x73FF]=3463;   // chu
   sortTable[0x7421]=3464;   // chu
   sortTable[0x74B4]=3465;   // chu
   sortTable[0x7AD0]=3466;   // chu
   sortTable[0x7D40]=3467;   // chu
   sortTable[0x8D8E]=3468;   // chu
   sortTable[0x8E95]=3469;   // chu
   sortTable[0x96DB]=3470;   // chu
   sortTable[0x9F6D]=3471;   // chu
   sortTable[0x6B3B]=3472;   // chua
   sortTable[0x6B58]=3473;   // chua
   sortTable[0x63E3]=3474;   // chuai
   sortTable[0x555C]=3475;   // chuai
   sortTable[0x8E39]=3476;   // chuai
   sortTable[0x562C]=3477;   // chuai
   sortTable[0x640B]=3478;   // chuai
   sortTable[0x8197]=3479;   // chuai
   sortTable[0x81AA]=3480;   // chuai
   sortTable[0x8239]=3481;   // chuan
   sortTable[0x4F20]=3482;   // chuan
   sortTable[0x50B3]=3483;   // chuan
   sortTable[0x7A7F]=3484;   // chuan
   sortTable[0x4E32]=3485;   // chuan
   sortTable[0x5598]=3486;   // chuan
   sortTable[0x5DDD]=3487;   // chuan
   sortTable[0x948F]=3488;   // chuan
   sortTable[0x821B]=3489;   // chuan
   sortTable[0x693D]=3490;   // chuan
   sortTable[0x9044]=3491;   // chuan
   sortTable[0x6C1A]=3492;   // chuan
   sortTable[0x8221]=3493;   // chuan
   sortTable[0x8348]=3494;   // chuan
   sortTable[0x8E33]=3495;   // chuan
   sortTable[0x5DDB]=3496;   // chuan
   sortTable[0x8F32]=3497;   // chuan
   sortTable[0x7BC5]=3498;   // chuan
   sortTable[0x9DA8]=3499;   // chuan
   sortTable[0x4F1D]=3500;   // chuan
   sortTable[0x50E2]=3501;   // chuan
   sortTable[0x5276]=3502;   // chuan
   sortTable[0x570C]=3503;   // chuan
   sortTable[0x66B7]=3504;   // chuan
   sortTable[0x6B42]=3505;   // chuan
   sortTable[0x6C4C]=3506;   // chuan
   sortTable[0x732D]=3507;   // chuan
   sortTable[0x7394]=3508;   // chuan
   sortTable[0x744F]=3509;   // chuan
   sortTable[0x8229]=3510;   // chuan
   sortTable[0x8CD7]=3511;   // chuan
   sortTable[0x91E7]=3512;   // chuan
   sortTable[0x521B]=3513;   // chuang
   sortTable[0x5275]=3514;   // chuang
   sortTable[0x5E8A]=3515;   // chuang
   sortTable[0x7A97]=3516;   // chuang
   sortTable[0x95EF]=3517;   // chuang
   sortTable[0x95D6]=3518;   // chuang
   sortTable[0x5E62]=3519;   // chuang
   sortTable[0x75AE]=3520;   // chuang
   sortTable[0x6006]=3521;   // chuang
   sortTable[0x78E2]=3522;   // chuang
   sortTable[0x50B8]=3523;   // chuang
   sortTable[0x51D4]=3524;   // chuang
   sortTable[0x5205]=3525;   // chuang
   sortTable[0x5231]=3526;   // chuang
   sortTable[0x524F]=3527;   // chuang
   sortTable[0x5259]=3528;   // chuang
   sortTable[0x5647]=3529;   // chuang
   sortTable[0x6134]=3530;   // chuang
   sortTable[0x6450]=3531;   // chuang
   sortTable[0x6464]=3532;   // chuang
   sortTable[0x7240]=3533;   // chuang
   sortTable[0x724E]=3534;   // chuang
   sortTable[0x7255]=3535;   // chuang
   sortTable[0x7621]=3536;   // chuang
   sortTable[0x7A93]=3537;   // chuang
   sortTable[0x7ABB]=3538;   // chuang
   sortTable[0x5439]=3539;   // chui
   sortTable[0x5782]=3540;   // chui
   sortTable[0x9524]=3541;   // chui
   sortTable[0x6376]=3542;   // chui
   sortTable[0x9318]=3543;   // chui
   sortTable[0x708A]=3544;   // chui
   sortTable[0x690E]=3545;   // chui
   sortTable[0x69CC]=3546;   // chui
   sortTable[0x9672]=3547;   // chui
   sortTable[0x68F0]=3548;   // chui
   sortTable[0x7BA0]=3549;   // chui
   sortTable[0x939A]=3550;   // chui
   sortTable[0x83D9]=3551;   // chui
   sortTable[0x8144]=3552;   // chui
   sortTable[0x9840]=3553;   // chui
   sortTable[0x5015]=3554;   // chui
   sortTable[0x57C0]=3555;   // chui
   sortTable[0x6425]=3556;   // chui
   sortTable[0x9FA1]=3557;   // chui
   sortTable[0x6625]=3558;   // chun
   sortTable[0x7EAF]=3559;   // chun
   sortTable[0x7D14]=3560;   // chun
   sortTable[0x5507]=3561;   // chun
   sortTable[0x9E51]=3562;   // chun
   sortTable[0x9D89]=3563;   // chun
   sortTable[0x6DF3]=3564;   // chun
   sortTable[0x8822]=3565;   // chun
   sortTable[0x9187]=3566;   // chun
   sortTable[0x693F]=3567;   // chun
   sortTable[0x83BC]=3568;   // chun
   sortTable[0x8123]=3569;   // chun
   sortTable[0x877D]=3570;   // chun
   sortTable[0x931E]=3571;   // chun
   sortTable[0x8F34]=3572;   // chun
   sortTable[0x84F4]=3573;   // chun
   sortTable[0x7776]=3574;   // chun
   sortTable[0x8CF0]=3575;   // chun
   sortTable[0x9BD9]=3576;   // chun
   sortTable[0x9659]=3577;   // chun
   sortTable[0x7BBA]=3578;   // chun
   sortTable[0x81A5]=3579;   // chun
   sortTable[0x8436]=3580;   // chun
   sortTable[0x9D9E]=3581;   // chun
   sortTable[0x9195]=3582;   // chun
   sortTable[0x5046]=3583;   // chun
   sortTable[0x583E]=3584;   // chun
   sortTable[0x5A8B]=3585;   // chun
   sortTable[0x60F7]=3586;   // chun
   sortTable[0x65FE]=3587;   // chun
   sortTable[0x6699]=3588;   // chun
   sortTable[0x6776]=3589;   // chun
   sortTable[0x6A41]=3590;   // chun
   sortTable[0x6AC4]=3591;   // chun
   sortTable[0x6D71]=3592;   // chun
   sortTable[0x6E7B]=3593;   // chun
   sortTable[0x6EE3]=3594;   // chun
   sortTable[0x6F18]=3595;   // chun
   sortTable[0x7289]=3596;   // chun
   sortTable[0x7443]=3597;   // chun
   sortTable[0x8405]=3598;   // chun
   sortTable[0x8493]=3599;   // chun
   sortTable[0x9C06]=3600;   // chun
   sortTable[0x7EF0]=3601;   // chuo
   sortTable[0x8F8D]=3602;   // chuo
   sortTable[0x6233]=3603;   // chuo
   sortTable[0x8E14]=3604;   // chuo
   sortTable[0x9F8A]=3605;   // chuo
   sortTable[0x9034]=3606;   // chuo
   sortTable[0x8DA0]=3607;   // chuo
   sortTable[0x8FB5]=3608;   // chuo
   sortTable[0x9461]=3609;   // chuo
   sortTable[0x78ED]=3610;   // chuo
   sortTable[0x7E5B]=3611;   // chuo
   sortTable[0x814F]=3612;   // chuo
   sortTable[0x56BD]=3613;   // chuo
   sortTable[0x5A15]=3614;   // chuo
   sortTable[0x5A16]=3615;   // chuo
   sortTable[0x5A7C]=3616;   // chuo
   sortTable[0x60D9]=3617;   // chuo
   sortTable[0x64C9]=3618;   // chuo
   sortTable[0x6B60]=3619;   // chuo
   sortTable[0x6DB0]=3620;   // chuo
   sortTable[0x7DBD]=3621;   // chuo
   sortTable[0x8F1F]=3622;   // chuo
   sortTable[0x8FB6]=3623;   // chuo
   sortTable[0x916B]=3624;   // chuo
   sortTable[0x9F6A]=3625;   // chuo
   sortTable[0x6B21]=3626;   // ci
   sortTable[0x6B64]=3627;   // ci
   sortTable[0x523A]=3628;   // ci
   sortTable[0x78C1]=3629;   // ci
   sortTable[0x8BCD]=3630;   // ci
   sortTable[0x8A5E]=3631;   // ci
   sortTable[0x74F7]=3632;   // ci
   sortTable[0x4F3A]=3633;   // ci
   sortTable[0x8F9E]=3634;   // ci
   sortTable[0x8FAD]=3635;   // ci
   sortTable[0x96CC]=3636;   // ci
   sortTable[0x6148]=3637;   // ci
   sortTable[0x8D50]=3638;   // ci
   sortTable[0x7960]=3639;   // ci
   sortTable[0x8328]=3640;   // ci
   sortTable[0x75B5]=3641;   // ci
   sortTable[0x9E5A]=3642;   // ci
   sortTable[0x5472]=3643;   // ci
   sortTable[0x7CCD]=3644;   // ci
   sortTable[0x8308]=3645;   // ci
   sortTable[0x8DD0]=3646;   // ci
   sortTable[0x9908]=3647;   // ci
   sortTable[0x7920]=3648;   // ci
   sortTable[0x9B86]=3649;   // ci
   sortTable[0x9AB4]=3650;   // ci
   sortTable[0x98FA]=3651;   // ci
   sortTable[0x83BF]=3652;   // ci
   sortTable[0x8786]=3653;   // ci
   sortTable[0x7E12]=3654;   // ci
   sortTable[0x858B]=3655;   // ci
   sortTable[0x8CDC]=3656;   // ci
   sortTable[0x7D2A]=3657;   // ci
   sortTable[0x8326]=3658;   // ci
   sortTable[0x8800]=3659;   // ci
   sortTable[0x9D1C]=3660;   // ci
   sortTable[0x7D58]=3661;   // ci
   sortTable[0x4F4C]=3662;   // ci
   sortTable[0x4F7D]=3663;   // ci
   sortTable[0x5068]=3664;   // ci
   sortTable[0x523E]=3665;   // ci
   sortTable[0x5790]=3666;   // ci
   sortTable[0x5832]=3667;   // ci
   sortTable[0x5B28]=3668;   // ci
   sortTable[0x5E9B]=3669;   // ci
   sortTable[0x673F]=3670;   // ci
   sortTable[0x67CC]=3671;   // ci
   sortTable[0x6828]=3672;   // ci
   sortTable[0x6CDA]=3673;   // ci
   sortTable[0x6FE8]=3674;   // ci
   sortTable[0x73BC]=3675;   // ci
   sortTable[0x73C1]=3676;   // ci
   sortTable[0x7506]=3677;   // ci
   sortTable[0x7689]=3678;   // ci
   sortTable[0x86D3]=3679;   // ci
   sortTable[0x8D7C]=3680;   // ci
   sortTable[0x8D80]=3681;   // ci
   sortTable[0x8F9D]=3682;   // ci
   sortTable[0x8FA4]=3683;   // ci
   sortTable[0x9ACA]=3684;   // ci
   sortTable[0x9DBF]=3685;   // ci
   sortTable[0x9DC0]=3686;   // ci
   sortTable[0x9F79]=3687;   // ci
   sortTable[0x4ECE]=3688;   // cong
   sortTable[0x5F9E]=3689;   // cong
   sortTable[0x5306]=3690;   // cong
   sortTable[0x806A]=3691;   // cong
   sortTable[0x8070]=3692;   // cong
   sortTable[0x4E1B]=3693;   // cong
   sortTable[0x53E2]=3694;   // cong
   sortTable[0x56F1]=3695;   // cong
   sortTable[0x56EA]=3696;   // cong
   sortTable[0x8471]=3697;   // cong
   sortTable[0x8525]=3698;   // cong
   sortTable[0x6DD9]=3699;   // cong
   sortTable[0x742E]=3700;   // cong
   sortTable[0x82C1]=3701;   // cong
   sortTable[0x679E]=3702;   // cong
   sortTable[0x7481]=3703;   // cong
   sortTable[0x9AA2]=3704;   // cong
   sortTable[0x8CE8]=3705;   // cong
   sortTable[0x93E6]=3706;   // cong
   sortTable[0x85C2]=3707;   // cong
   sortTable[0x84EF]=3708;   // cong
   sortTable[0x8310]=3709;   // cong
   sortTable[0x8B25]=3710;   // cong
   sortTable[0x8066]=3711;   // cong
   sortTable[0x9A44]=3712;   // cong
   sortTable[0x8AB4]=3713;   // cong
   sortTable[0x779B]=3714;   // cong
   sortTable[0x8061]=3715;   // cong
   sortTable[0x9A18]=3716;   // cong
   sortTable[0x5A43]=3717;   // cong
   sortTable[0x5B6E]=3718;   // cong
   sortTable[0x5F93]=3719;   // cong
   sortTable[0x5F96]=3720;   // cong
   sortTable[0x5FE9]=3721;   // cong
   sortTable[0x6031]=3722;   // cong
   sortTable[0x60A4]=3723;   // cong
   sortTable[0x60B0]=3724;   // cong
   sortTable[0x6152]=3725;   // cong
   sortTable[0x6181]=3726;   // cong
   sortTable[0x66B0]=3727;   // cong
   sortTable[0x68C7]=3728;   // cong
   sortTable[0x6A05]=3729;   // cong
   sortTable[0x6A2C]=3730;   // cong
   sortTable[0x6A37]=3731;   // cong
   sortTable[0x6B09]=3732;   // cong
   sortTable[0x6F0E]=3733;   // cong
   sortTable[0x6F17]=3734;   // cong
   sortTable[0x6F40]=3735;   // cong
   sortTable[0x6F68]=3736;   // cong
   sortTable[0x7047]=3737;   // cong
   sortTable[0x7127]=3738;   // cong
   sortTable[0x719C]=3739;   // cong
   sortTable[0x71EA]=3740;   // cong
   sortTable[0x721C]=3741;   // cong
   sortTable[0x747D]=3742;   // cong
   sortTable[0x7BF5]=3743;   // cong
   sortTable[0x7DEB]=3744;   // cong
   sortTable[0x7E71]=3745;   // cong
   sortTable[0x87CC]=3746;   // cong
   sortTable[0x8CE9]=3747;   // cong
   sortTable[0x936F]=3748;   // cong
   sortTable[0x51D1]=3749;   // cou
   sortTable[0x6E4A]=3750;   // cou
   sortTable[0x8F8F]=3751;   // cou
   sortTable[0x8160]=3752;   // cou
   sortTable[0x8F33]=3753;   // cou
   sortTable[0x7C97]=3754;   // cu
   sortTable[0x4FC3]=3755;   // cu
   sortTable[0x918B]=3756;   // cu
   sortTable[0x7C07]=3757;   // cu
   sortTable[0x9162]=3758;   // cu
   sortTable[0x8E59]=3759;   // cu
   sortTable[0x731D]=3760;   // cu
   sortTable[0x6B82]=3761;   // cu
   sortTable[0x8E74]=3762;   // cu
   sortTable[0x5F82]=3763;   // cu
   sortTable[0x851F]=3764;   // cu
   sortTable[0x8E27]=3765;   // cu
   sortTable[0x9EA4]=3766;   // cu
   sortTable[0x9863]=3767;   // cu
   sortTable[0x9E84]=3768;   // cu
   sortTable[0x89D5]=3769;   // cu
   sortTable[0x8128]=3770;   // cu
   sortTable[0x8A8E]=3771;   // cu
   sortTable[0x8E75]=3772;   // cu
   sortTable[0x5648]=3773;   // cu
   sortTable[0x61B1]=3774;   // cu
   sortTable[0x7604]=3775;   // cu
   sortTable[0x762F]=3776;   // cu
   sortTable[0x7E2C]=3777;   // cu
   sortTable[0x8D97]=3778;   // cu
   sortTable[0x9E81]=3779;   // cu
   sortTable[0x9F00]=3780;   // cu
   sortTable[0x7A9C]=3781;   // cuan
   sortTable[0x7AC4]=3782;   // cuan
   sortTable[0x7BE1]=3783;   // cuan
   sortTable[0x8E7F]=3784;   // cuan
   sortTable[0x64BA]=3785;   // cuan
   sortTable[0x7228]=3786;   // cuan
   sortTable[0x6C46]=3787;   // cuan
   sortTable[0x9569]=3788;   // cuan
   sortTable[0x7A73]=3789;   // cuan
   sortTable[0x5DD1]=3790;   // cuan
   sortTable[0x651B]=3791;   // cuan
   sortTable[0x6AD5]=3792;   // cuan
   sortTable[0x6B11]=3793;   // cuan
   sortTable[0x6BA9]=3794;   // cuan
   sortTable[0x71B6]=3795;   // cuan
   sortTable[0x7C12]=3796;   // cuan
   sortTable[0x8EA5]=3797;   // cuan
   sortTable[0x92D1]=3798;   // cuan
   sortTable[0x9479]=3799;   // cuan
   sortTable[0x8106]=3800;   // cui
   sortTable[0x6467]=3801;   // cui
   sortTable[0x50AC]=3802;   // cui
   sortTable[0x7CB9]=3803;   // cui
   sortTable[0x7FE0]=3804;   // cui
   sortTable[0x5D14]=3805;   // cui
   sortTable[0x60B4]=3806;   // cui
   sortTable[0x8403]=3807;   // cui
   sortTable[0x7601]=3808;   // cui
   sortTable[0x5550]=3809;   // cui
   sortTable[0x69B1]=3810;   // cui
   sortTable[0x7480]=3811;   // cui
   sortTable[0x6DEC]=3812;   // cui
   sortTable[0x6BF3]=3813;   // cui
   sortTable[0x7F1E]=3814;   // cui
   sortTable[0x7C8B]=3815;   // cui
   sortTable[0x7E17]=3816;   // cui
   sortTable[0x81AC]=3817;   // cui
   sortTable[0x81CE]=3818;   // cui
   sortTable[0x813A]=3819;   // cui
   sortTable[0x7DB7]=3820;   // cui
   sortTable[0x7AC1]=3821;   // cui
   sortTable[0x81B5]=3822;   // cui
   sortTable[0x78EA]=3823;   // cui
   sortTable[0x8DA1]=3824;   // cui
   sortTable[0x9847]=3825;   // cui
   sortTable[0x7D23]=3826;   // cui
   sortTable[0x7FC6]=3827;   // cui
   sortTable[0x8103]=3828;   // cui
   sortTable[0x4E7C]=3829;   // cui
   sortTable[0x4F1C]=3830;   // cui
   sortTable[0x5005]=3831;   // cui
   sortTable[0x51D7]=3832;   // cui
   sortTable[0x555B]=3833;   // cui
   sortTable[0x5894]=3834;   // cui
   sortTable[0x5D89]=3835;   // cui
   sortTable[0x5FF0]=3836;   // cui
   sortTable[0x615B]=3837;   // cui
   sortTable[0x69EF]=3838;   // cui
   sortTable[0x6F3C]=3839;   // cui
   sortTable[0x6FE2]=3840;   // cui
   sortTable[0x7120]=3841;   // cui
   sortTable[0x7355]=3842;   // cui
   sortTable[0x75A9]=3843;   // cui
   sortTable[0x76A0]=3844;   // cui
   sortTable[0x894A]=3845;   // cui
   sortTable[0x93D9]=3846;   // cui
   sortTable[0x6751]=3847;   // cun
   sortTable[0x5B58]=3848;   // cun
   sortTable[0x5BF8]=3849;   // cun
   sortTable[0x5FD6]=3850;   // cun
   sortTable[0x90A8]=3851;   // cun
   sortTable[0x76B4]=3852;   // cun
   sortTable[0x8E06]=3853;   // cun
   sortTable[0x7C7F]=3854;   // cun
   sortTable[0x4F9F]=3855;   // cun
   sortTable[0x520C]=3856;   // cun
   sortTable[0x540B]=3857;   // cun
   sortTable[0x62F5]=3858;   // cun
   sortTable[0x6F8A]=3859;   // cun
   sortTable[0x7AF4]=3860;   // cun
   sortTable[0x9519]=3861;   // cuo
   sortTable[0x932F]=3862;   // cuo
   sortTable[0x63AA]=3863;   // cuo
   sortTable[0x632B]=3864;   // cuo
   sortTable[0x6413]=3865;   // cuo
   sortTable[0x64AE]=3866;   // cuo
   sortTable[0x78CB]=3867;   // cuo
   sortTable[0x539D]=3868;   // cuo
   sortTable[0x9E7E]=3869;   // cuo
   sortTable[0x9509]=3870;   // cuo
   sortTable[0x5D6F]=3871;   // cuo
   sortTable[0x8E49]=3872;   // cuo
   sortTable[0x75E4]=3873;   // cuo
   sortTable[0x77EC]=3874;   // cuo
   sortTable[0x9142]=3875;   // cuo
   sortTable[0x811E]=3876;   // cuo
   sortTable[0x839D]=3877;   // cuo
   sortTable[0x9E7A]=3878;   // cuo
   sortTable[0x7749]=3879;   // cuo
   sortTable[0x8658]=3880;   // cuo
   sortTable[0x919D]=3881;   // cuo
   sortTable[0x84AB]=3882;   // cuo
   sortTable[0x84CC]=3883;   // cuo
   sortTable[0x8516]=3884;   // cuo
   sortTable[0x8EA6]=3885;   // cuo
   sortTable[0x5249]=3886;   // cuo
   sortTable[0x5252]=3887;   // cuo
   sortTable[0x590E]=3888;   // cuo
   sortTable[0x5D73]=3889;   // cuo
   sortTable[0x65AE]=3890;   // cuo
   sortTable[0x68E4]=3891;   // cuo
   sortTable[0x6B75]=3892;   // cuo
   sortTable[0x7473]=3893;   // cuo
   sortTable[0x83A1]=3894;   // cuo
   sortTable[0x902A]=3895;   // cuo
   sortTable[0x9073]=3896;   // cuo
   sortTable[0x92BC]=3897;   // cuo
   sortTable[0x5927]=4352;   // da
   sortTable[0x6253]=4353;   // da
   sortTable[0x8FBE]=4354;   // da
   sortTable[0x9054]=4355;   // da
   sortTable[0x7B54]=4356;   // da
   sortTable[0x642D]=4357;   // da
   sortTable[0x7629]=4358;   // da
   sortTable[0x8037]=4359;   // da
   sortTable[0x6C93]=4360;   // da
   sortTable[0x9791]=4361;   // da
   sortTable[0x54D2]=4362;   // da
   sortTable[0x977C]=4363;   // da
   sortTable[0x7B2A]=4364;   // da
   sortTable[0x55D2]=4365;   // da
   sortTable[0x59B2]=4366;   // da
   sortTable[0x8921]=4367;   // da
   sortTable[0x8345]=4368;   // da
   sortTable[0x601B]=4369;   // da
   sortTable[0x87FD]=4370;   // da
   sortTable[0x9618]=4371;   // da
   sortTable[0x7E68]=4372;   // da
   sortTable[0x7F8D]=4373;   // da
   sortTable[0x939D]=4374;   // da
   sortTable[0x8359]=4375;   // da
   sortTable[0x7B1A]=4376;   // da
   sortTable[0x5273]=4377;   // da
   sortTable[0x5312]=4378;   // da
   sortTable[0x547E]=4379;   // da
   sortTable[0x5491]=4380;   // da
   sortTable[0x5660]=4381;   // da
   sortTable[0x57AF]=4382;   // da
   sortTable[0x58B6]=4383;   // da
   sortTable[0x6498]=4384;   // da
   sortTable[0x6C4F]=4385;   // da
   sortTable[0x709F]=4386;   // da
   sortTable[0x71F5]=4387;   // da
   sortTable[0x7557]=4388;   // da
   sortTable[0x7563]=4389;   // da
   sortTable[0x7714]=4390;   // da
   sortTable[0x8598]=4391;   // da
   sortTable[0x8A5A]=4392;   // da
   sortTable[0x8E82]=4393;   // da
   sortTable[0x8FD6]=4394;   // da
   sortTable[0x9039]=4395;   // da
   sortTable[0x9389]=4396;   // da
   sortTable[0x943D]=4397;   // da
   sortTable[0x97C3]=4398;   // da
   sortTable[0x9F96]=4399;   // da
   sortTable[0x9F98]=4400;   // da
   sortTable[0x4EE3]=4401;   // dai
   sortTable[0x5E26]=4402;   // dai
   sortTable[0x5E36]=4403;   // dai
   sortTable[0x5F85]=4404;   // dai
   sortTable[0x888B]=4405;   // dai
   sortTable[0x6234]=4406;   // dai
   sortTable[0x5446]=4407;   // dai
   sortTable[0x902E]=4408;   // dai
   sortTable[0x8D37]=4409;   // dai
   sortTable[0x9EDB]=4410;   // dai
   sortTable[0x6020]=4411;   // dai
   sortTable[0x6B86]=4412;   // dai
   sortTable[0x6B79]=4413;   // dai
   sortTable[0x5CB1]=4414;   // dai
   sortTable[0x8FE8]=4415;   // dai
   sortTable[0x73B3]=4416;   // dai
   sortTable[0x50A3]=4417;   // dai
   sortTable[0x7ED0]=4418;   // dai
   sortTable[0x5454]=4419;   // dai
   sortTable[0x7519]=4420;   // dai
   sortTable[0x57ED]=4421;   // dai
   sortTable[0x9A80]=4422;   // dai
   sortTable[0x8F6A]=4423;   // dai
   sortTable[0x8E5B]=4424;   // dai
   sortTable[0x7D3F]=4425;   // dai
   sortTable[0x825C]=4426;   // dai
   sortTable[0x7C24]=4427;   // dai
   sortTable[0x8976]=4428;   // dai
   sortTable[0x9D0F]=4429;   // dai
   sortTable[0x8EDA]=4430;   // dai
   sortTable[0x9746]=4431;   // dai
   sortTable[0x8CB8]=4432;   // dai
   sortTable[0x9B98]=4433;   // dai
   sortTable[0x53C7]=4434;   // dai
   sortTable[0x5788]=4435;   // dai
   sortTable[0x5E12]=4436;   // dai
   sortTable[0x5E2F]=4437;   // dai
   sortTable[0x5ED7]=4438;   // dai
   sortTable[0x61DB]=4439;   // dai
   sortTable[0x66C3]=4440;   // dai
   sortTable[0x67CB]=4441;   // dai
   sortTable[0x703B]=4442;   // dai
   sortTable[0x7343]=4443;   // dai
   sortTable[0x7447]=4444;   // dai
   sortTable[0x7DFF]=4445;   // dai
   sortTable[0x8ED1]=4446;   // dai
   sortTable[0x8EE9]=4447;   // dai
   sortTable[0x9734]=4448;   // dai
   sortTable[0x9EF1]=4449;   // dai
   sortTable[0x4F46]=4450;   // dan
   sortTable[0x5355]=4451;   // dan
   sortTable[0x55AE]=4452;   // dan
   sortTable[0x5F39]=4453;   // dan
   sortTable[0x5F48]=4454;   // dan
   sortTable[0x62C5]=4455;   // dan
   sortTable[0x64D4]=4456;   // dan
   sortTable[0x86CB]=4457;   // dan
   sortTable[0x6DE1]=4458;   // dan
   sortTable[0x80C6]=4459;   // dan
   sortTable[0x81BD]=4460;   // dan
   sortTable[0x8BDE]=4461;   // dan
   sortTable[0x8A95]=4462;   // dan
   sortTable[0x65E6]=4463;   // dan
   sortTable[0x803D]=4464;   // dan
   sortTable[0x4E39]=4465;   // dan
   sortTable[0x6C2E]=4466;   // dan
   sortTable[0x6FB9]=4467;   // dan
   sortTable[0x60EE]=4468;   // dan
   sortTable[0x90F8]=4469;   // dan
   sortTable[0x5556]=4470;   // dan
   sortTable[0x7708]=4471;   // dan
   sortTable[0x6B9A]=4472;   // dan
   sortTable[0x510B]=4473;   // dan
   sortTable[0x63B8]=4474;   // dan
   sortTable[0x840F]=4475;   // dan
   sortTable[0x8043]=4476;   // dan
   sortTable[0x75B8]=4477;   // dan
   sortTable[0x7BAA]=4478;   // dan
   sortTable[0x79AB]=4479;   // dan
   sortTable[0x7D1E]=4480;   // dan
   sortTable[0x7605]=4481;   // dan
   sortTable[0x8711]=4482;   // dan
   sortTable[0x8D55]=4483;   // dan
   sortTable[0x8EAD]=4484;   // dan
   sortTable[0x894C]=4485;   // dan
   sortTable[0x9EEE]=4486;   // dan
   sortTable[0x8078]=4487;   // dan
   sortTable[0x972E]=4488;   // dan
   sortTable[0x7A9E]=4489;   // dan
   sortTable[0x9AE7]=4490;   // dan
   sortTable[0x994F]=4491;   // dan
   sortTable[0x7C1E]=4492;   // dan
   sortTable[0x84DE]=4493;   // dan
   sortTable[0x9D20]=4494;   // dan
   sortTable[0x8D09]=4495;   // dan
   sortTable[0x9132]=4496;   // dan
   sortTable[0x99F3]=4497;   // dan
   sortTable[0x9ED5]=4498;   // dan
   sortTable[0x7E75]=4499;   // dan
   sortTable[0x891D]=4500;   // dan
   sortTable[0x89DB]=4501;   // dan
   sortTable[0x4EB6]=4502;   // dan
   sortTable[0x4F14]=4503;   // dan
   sortTable[0x50E4]=4504;   // dan
   sortTable[0x5210]=4505;   // dan
   sortTable[0x52EF]=4506;   // dan
   sortTable[0x5330]=4507;   // dan
   sortTable[0x5358]=4508;   // dan
   sortTable[0x5557]=4509;   // dan
   sortTable[0x557F]=4510;   // dan
   sortTable[0x563E]=4511;   // dan
   sortTable[0x5649]=4512;   // dan
   sortTable[0x56AA]=4513;   // dan
   sortTable[0x5989]=4514;   // dan
   sortTable[0x5A85]=4515;   // dan
   sortTable[0x5E0E]=4516;   // dan
   sortTable[0x5F3E]=4517;   // dan
   sortTable[0x619A]=4518;   // dan
   sortTable[0x61BA]=4519;   // dan
   sortTable[0x628C]=4520;   // dan
   sortTable[0x64A2]=4521;   // dan
   sortTable[0x64A3]=4522;   // dan
   sortTable[0x66BA]=4523;   // dan
   sortTable[0x67E6]=4524;   // dan
   sortTable[0x6BAB]=4525;   // dan
   sortTable[0x6C8A]=4526;   // dan
   sortTable[0x6CF9]=4527;   // dan
   sortTable[0x6FB8]=4528;   // dan
   sortTable[0x72DA]=4529;   // dan
   sortTable[0x73AC]=4530;   // dan
   sortTable[0x74ED]=4531;   // dan
   sortTable[0x7514]=4532;   // dan
   sortTable[0x758D]=4533;   // dan
   sortTable[0x7649]=4534;   // dan
   sortTable[0x765A]=4535;   // dan
   sortTable[0x7803]=4536;   // dan
   sortTable[0x803C]=4537;   // dan
   sortTable[0x8145]=4538;   // dan
   sortTable[0x8874]=4539;   // dan
   sortTable[0x9815]=4540;   // dan
   sortTable[0x99BE]=4541;   // dan
   sortTable[0x515A]=4542;   // dang
   sortTable[0x9EE8]=4543;   // dang
   sortTable[0x5F53]=4544;   // dang
   sortTable[0x7576]=4545;   // dang
   sortTable[0x6321]=4546;   // dang
   sortTable[0x64CB]=4547;   // dang
   sortTable[0x8361]=4548;   // dang
   sortTable[0x8569]=4549;   // dang
   sortTable[0x6863]=4550;   // dang
   sortTable[0x6A94]=4551;   // dang
   sortTable[0x94DB]=4552;   // dang
   sortTable[0x943A]=4553;   // dang
   sortTable[0x8C20]=4554;   // dang
   sortTable[0x5B95]=4555;   // dang
   sortTable[0x88C6]=4556;   // dang
   sortTable[0x51FC]=4557;   // dang
   sortTable[0x7800]=4558;   // dang
   sortTable[0x83EA]=4559;   // dang
   sortTable[0x7B5C]=4560;   // dang
   sortTable[0x7C1C]=4561;   // dang
   sortTable[0x903F]=4562;   // dang
   sortTable[0x8DA4]=4563;   // dang
   sortTable[0x8B61]=4564;   // dang
   sortTable[0x7C39]=4565;   // dang
   sortTable[0x76EA]=4566;   // dang
   sortTable[0x78AD]=4567;   // dang
   sortTable[0x7911]=4568;   // dang
   sortTable[0x8261]=4569;   // dang
   sortTable[0x87F7]=4570;   // dang
   sortTable[0x8B9C]=4571;   // dang
   sortTable[0x5105]=4572;   // dang
   sortTable[0x5679]=4573;   // dang
   sortTable[0x5735]=4574;   // dang
   sortTable[0x57B1]=4575;   // dang
   sortTable[0x58CB]=4576;   // dang
   sortTable[0x5A78]=4577;   // dang
   sortTable[0x5D63]=4578;   // dang
   sortTable[0x6113]=4579;   // dang
   sortTable[0x6529]=4580;   // dang
   sortTable[0x6B13]=4581;   // dang
   sortTable[0x6C39]=4582;   // dang
   sortTable[0x6F52]=4583;   // dang
   sortTable[0x6FA2]=4584;   // dang
   sortTable[0x7059]=4585;   // dang
   sortTable[0x73F0]=4586;   // dang
   sortTable[0x7497]=4587;   // dang
   sortTable[0x74AB]=4588;   // dang
   sortTable[0x74FD]=4589;   // dang
   sortTable[0x778A]=4590;   // dang
   sortTable[0x862F]=4591;   // dang
   sortTable[0x8960]=4592;   // dang
   sortTable[0x95E3]=4593;   // dang
   sortTable[0x96FC]=4594;   // dang
   sortTable[0x5230]=4595;   // dao
   sortTable[0x9053]=4596;   // dao
   sortTable[0x5BFC]=4597;   // dao
   sortTable[0x5C0E]=4598;   // dao
   sortTable[0x5012]=4599;   // dao
   sortTable[0x5200]=4600;   // dao
   sortTable[0x5C9B]=4601;   // dao
   sortTable[0x5CF6]=4602;   // dao
   sortTable[0x7A3B]=4603;   // dao
   sortTable[0x76D7]=4604;   // dao
   sortTable[0x76DC]=4605;   // dao
   sortTable[0x6363]=4606;   // dao
   sortTable[0x6417]=4607;   // dao
   sortTable[0x60BC]=4608;   // dao
   sortTable[0x8E48]=4609;   // dao
   sortTable[0x7977]=4610;   // dao
   sortTable[0x53E8]=4611;   // dao
   sortTable[0x7118]=4612;   // dao
   sortTable[0x7E9B]=4613;   // dao
   sortTable[0x6C18]=4614;   // dao
   sortTable[0x5FC9]=4615;   // dao
   sortTable[0x7FFF]=4616;   // dao
   sortTable[0x8220]=4617;   // dao
   sortTable[0x7982]=4618;   // dao
   sortTable[0x9C7D]=4619;   // dao
   sortTable[0x79B1]=4620;   // dao
   sortTable[0x969D]=4621;   // dao
   sortTable[0x885C]=4622;   // dao
   sortTable[0x7FE2]=4623;   // dao
   sortTable[0x8EC7]=4624;   // dao
   sortTable[0x7A32]=4625;   // dao
   sortTable[0x885F]=4626;   // dao
   sortTable[0x9B5B]=4627;   // dao
   sortTable[0x5202]=4628;   // dao
   sortTable[0x5675]=4629;   // dao
   sortTable[0x58D4]=4630;   // dao
   sortTable[0x5C76]=4631;   // dao
   sortTable[0x5D8B]=4632;   // dao
   sortTable[0x5D8C]=4633;   // dao
   sortTable[0x5DB9]=4634;   // dao
   sortTable[0x636F]=4635;   // dao
   sortTable[0x64E3]=4636;   // dao
   sortTable[0x6737]=4637;   // dao
   sortTable[0x6921]=4638;   // dao
   sortTable[0x69DD]=4639;   // dao
   sortTable[0x6AA4]=4640;   // dao
   sortTable[0x71FE]=4641;   // dao
   sortTable[0x74D9]=4642;   // dao
   sortTable[0x7B8C]=4643;   // dao
   sortTable[0x83FF]=4644;   // dao
   sortTable[0x91D6]=4645;   // dao
   sortTable[0x9666]=4646;   // dao
   sortTable[0x96AF]=4647;   // dao
   sortTable[0x7684]=4648;   // de
   sortTable[0x5730]=4649;   // de
   sortTable[0x5F97]=4650;   // de
   sortTable[0x5FB7]=4651;   // de
   sortTable[0x951D]=4652;   // de
   sortTable[0x8126]=4653;   // de
   sortTable[0x561A]=4654;   // de
   sortTable[0x5FB3]=4655;   // de
   sortTable[0x6074]=4656;   // de
   sortTable[0x60EA]=4657;   // de
   sortTable[0x68CF]=4658;   // de
   sortTable[0x6DC2]=4659;   // de
   sortTable[0x9340]=4660;   // de
   sortTable[0x6265]=4661;   // den
   sortTable[0x627D]=4662;   // den
   sortTable[0x7B49]=4663;   // deng
   sortTable[0x706F]=4664;   // deng
   sortTable[0x71C8]=4665;   // deng
   sortTable[0x767B]=4666;   // deng
   sortTable[0x51F3]=4667;   // deng
   sortTable[0x77AA]=4668;   // deng
   sortTable[0x8E6C]=4669;   // deng
   sortTable[0x9093]=4670;   // deng
   sortTable[0x7C26]=4671;   // deng
   sortTable[0x956B]=4672;   // deng
   sortTable[0x5654]=4673;   // deng
   sortTable[0x78F4]=4674;   // deng
   sortTable[0x5D9D]=4675;   // deng
   sortTable[0x6225]=4676;   // deng
   sortTable[0x96A5]=4677;   // deng
   sortTable[0x9127]=4678;   // deng
   sortTable[0x7AF3]=4679;   // deng
   sortTable[0x89B4]=4680;   // deng
   sortTable[0x8C4B]=4681;   // deng
   sortTable[0x58B1]=4682;   // deng
   sortTable[0x5B01]=4683;   // deng
   sortTable[0x6729]=4684;   // deng
   sortTable[0x6AC8]=4685;   // deng
   sortTable[0x7492]=4686;   // deng
   sortTable[0x8260]=4687;   // deng
   sortTable[0x9419]=4688;   // deng
   sortTable[0x7B2C]=4689;   // di
   sortTable[0x654C]=4690;   // di
   sortTable[0x6575]=4691;   // di
   sortTable[0x5E95]=4692;   // di
   sortTable[0x4F4E]=4693;   // di
   sortTable[0x5E1D]=4694;   // di
   sortTable[0x5F1F]=4695;   // di
   sortTable[0x62B5]=4696;   // di
   sortTable[0x6EF4]=4697;   // di
   sortTable[0x9012]=4698;   // di
   sortTable[0x905E]=4699;   // di
   sortTable[0x5824]=4700;   // di
   sortTable[0x6DA4]=4701;   // di
   sortTable[0x6ECC]=4702;   // di
   sortTable[0x5600]=4703;   // di
   sortTable[0x7F14]=4704;   // di
   sortTable[0x7DE0]=4705;   // di
   sortTable[0x7B1B]=4706;   // di
   sortTable[0x8FEA]=4707;   // di
   sortTable[0x72C4]=4708;   // di
   sortTable[0x8482]=4709;   // di
   sortTable[0x90B8]=4710;   // di
   sortTable[0x6C10]=4711;   // di
   sortTable[0x5AE1]=4712;   // di
   sortTable[0x8C1B]=4713;   // di
   sortTable[0x7C74]=4714;   // di
   sortTable[0x68E3]=4715;   // di
   sortTable[0x8BCB]=4716;   // di
   sortTable[0x837B]=4717;   // di
   sortTable[0x7825]=4718;   // di
   sortTable[0x7998]=4719;   // di
   sortTable[0x9AB6]=4720;   // di
   sortTable[0x5A23]=4721;   // di
   sortTable[0x7F9D]=4722;   // di
   sortTable[0x89CC]=4723;   // di
   sortTable[0x78FE]=4724;   // di
   sortTable[0x97AE]=4725;   // di
   sortTable[0x955D]=4726;   // di
   sortTable[0x7747]=4727;   // di
   sortTable[0x67E2]=4728;   // di
   sortTable[0x9814]=4729;   // di
   sortTable[0x78B2]=4730;   // di
   sortTable[0x7731]=4731;   // di
   sortTable[0x9684]=4732;   // di
   sortTable[0x91F1]=4733;   // di
   sortTable[0x889B]=4734;   // di
   sortTable[0x83E7]=4735;   // di
   sortTable[0x8E36]=4736;   // di
   sortTable[0x7BF4]=4737;   // di
   sortTable[0x9AE2]=4738;   // di
   sortTable[0x976E]=4739;   // di
   sortTable[0x9070]=4740;   // di
   sortTable[0x9349]=4741;   // di
   sortTable[0x89DD]=4742;   // di
   sortTable[0x83C2]=4743;   // di
   sortTable[0x8515]=4744;   // di
   sortTable[0x8743]=4745;   // di
   sortTable[0x8E62]=4746;   // di
   sortTable[0x9E10]=4747;   // di
   sortTable[0x9B04]=4748;   // di
   sortTable[0x87AE]=4749;   // di
   sortTable[0x8EE7]=4750;   // di
   sortTable[0x963A]=4751;   // di
   sortTable[0x9013]=4752;   // di
   sortTable[0x7976]=4753;   // di
   sortTable[0x7CF4]=4754;   // di
   sortTable[0x805C]=4755;   // di
   sortTable[0x85E1]=4756;   // di
   sortTable[0x8163]=4757;   // di
   sortTable[0x82D6]=4758;   // di
   sortTable[0x8AE6]=4759;   // di
   sortTable[0x99B0]=4760;   // di
   sortTable[0x9BF3]=4761;   // di
   sortTable[0x4EFE]=4762;   // di
   sortTable[0x4FE4]=4763;   // di
   sortTable[0x5059]=4764;   // di
   sortTable[0x50C0]=4765;   // di
   sortTable[0x538E]=4766;   // di
   sortTable[0x5467]=4767;   // di
   sortTable[0x5519]=4768;   // di
   sortTable[0x5547]=4769;   // di
   sortTable[0x5572]=4770;   // di
   sortTable[0x5681]=4771;   // di
   sortTable[0x5754]=4772;   // di
   sortTable[0x5758]=4773;   // di
   sortTable[0x57CA]=4774;   // di
   sortTable[0x57DE]=4775;   // di
   sortTable[0x5891]=4776;   // di
   sortTable[0x58AC]=4777;   // di
   sortTable[0x5943]=4778;   // di
   sortTable[0x5A82]=4779;   // di
   sortTable[0x5DB3]=4780;   // di
   sortTable[0x5EF8]=4781;   // di
   sortTable[0x5F24]=4782;   // di
   sortTable[0x5F7D]=4783;   // di
   sortTable[0x601F]=4784;   // di
   sortTable[0x6178]=4785;   // di
   sortTable[0x62DE]=4786;   // di
   sortTable[0x638B]=4787;   // di
   sortTable[0x6455]=4788;   // di
   sortTable[0x65F3]=4789;   // di
   sortTable[0x6755]=4790;   // di
   sortTable[0x688A]=4791;   // di
   sortTable[0x6891]=4792;   // di
   sortTable[0x6A00]=4793;   // di
   sortTable[0x6E27]=4794;   // di
   sortTable[0x710D]=4795;   // di
   sortTable[0x7274]=4796;   // di
   sortTable[0x7393]=4797;   // di
   sortTable[0x73F6]=4798;   // di
   sortTable[0x750B]=4799;   // di
   sortTable[0x830B]=4800;   // di
   sortTable[0x850B]=4801;   // di
   sortTable[0x8510]=4802;   // di
   sortTable[0x89BF]=4803;   // di
   sortTable[0x8A46]=4804;   // di
   sortTable[0x8C74]=4805;   // di
   sortTable[0x8D86]=4806;   // di
   sortTable[0x926A]=4807;   // di
   sortTable[0x93D1]=4808;   // di
   sortTable[0x55F2]=4809;   // dia
   sortTable[0x70B9]=4810;   // dian
   sortTable[0x9EDE]=4811;   // dian
   sortTable[0x7535]=4812;   // dian
   sortTable[0x96FB]=4813;   // dian
   sortTable[0x5E97]=4814;   // dian
   sortTable[0x5178]=4815;   // dian
   sortTable[0x98A0]=4816;   // dian
   sortTable[0x985B]=4817;   // dian
   sortTable[0x6BBF]=4818;   // dian
   sortTable[0x6DC0]=4819;   // dian
   sortTable[0x6FB1]=4820;   // dian
   sortTable[0x7538]=4821;   // dian
   sortTable[0x5960]=4822;   // dian
   sortTable[0x57AB]=4823;   // dian
   sortTable[0x588A]=4824;   // dian
   sortTable[0x4F43]=4825;   // dian
   sortTable[0x60E6]=4826;   // dian
   sortTable[0x8E2E]=4827;   // dian
   sortTable[0x766B]=4828;   // dian
   sortTable[0x7C1F]=4829;   // dian
   sortTable[0x6EC7]=4830;   // dian
   sortTable[0x73B7]=4831;   // dian
   sortTable[0x94BF]=4832;   // dian
   sortTable[0x5DC5]=4833;   // dian
   sortTable[0x7898]=4834;   // dian
   sortTable[0x6382]=4835;   // dian
   sortTable[0x765C]=4836;   // dian
   sortTable[0x975B]=4837;   // dian
   sortTable[0x576B]=4838;   // dian
   sortTable[0x963D]=4839;   // dian
   sortTable[0x8714]=4840;   // dian
   sortTable[0x84A7]=4841;   // dian
   sortTable[0x9A54]=4842;   // dian
   sortTable[0x8DD5]=4843;   // dian
   sortTable[0x9F7B]=4844;   // dian
   sortTable[0x8E4E]=4845;   // dian
   sortTable[0x8547]=4846;   // dian
   sortTable[0x985A]=4847;   // dian
   sortTable[0x508E]=4848;   // dian
   sortTable[0x53A7]=4849;   // dian
   sortTable[0x56B8]=4850;   // dian
   sortTable[0x58C2]=4851;   // dian
   sortTable[0x594C]=4852;   // dian
   sortTable[0x5A5D]=4853;   // dian
   sortTable[0x5A70]=4854;   // dian
   sortTable[0x5D6E]=4855;   // dian
   sortTable[0x5DD3]=4856;   // dian
   sortTable[0x5DD4]=4857;   // dian
   sortTable[0x6242]=4858;   // dian
   sortTable[0x6527]=4859;   // dian
   sortTable[0x6541]=4860;   // dian
   sortTable[0x655F]=4861;   // dian
   sortTable[0x6923]=4862;   // dian
   sortTable[0x69C7]=4863;   // dian
   sortTable[0x69D9]=4864;   // dian
   sortTable[0x6A42]=4865;   // dian
   sortTable[0x6A5D]=4866;   // dian
   sortTable[0x7320]=4867;   // dian
   sortTable[0x7414]=4868;   // dian
   sortTable[0x7628]=4869;   // dian
   sortTable[0x7672]=4870;   // dian
   sortTable[0x6389]=4871;   // diao
   sortTable[0x8C03]=4872;   // diao
   sortTable[0x8ABF]=4873;   // diao
   sortTable[0x9493]=4874;   // diao
   sortTable[0x91E3]=4875;   // diao
   sortTable[0x540A]=4876;   // diao
   sortTable[0x96D5]=4877;   // diao
   sortTable[0x53FC]=4878;   // diao
   sortTable[0x5201]=4879;   // diao
   sortTable[0x51CB]=4880;   // diao
   sortTable[0x8C82]=4881;   // diao
   sortTable[0x7889]=4882;   // diao
   sortTable[0x94EB]=4883;   // diao
   sortTable[0x9CB7]=4884;   // diao
   sortTable[0x9D70]=4885;   // diao
   sortTable[0x7A8E]=4886;   // diao
   sortTable[0x94DE]=4887;   // diao
   sortTable[0x85CB]=4888;   // diao
   sortTable[0x8A0B]=4889;   // diao
   sortTable[0x9BDB]=4890;   // diao
   sortTable[0x92FD]=4891;   // diao
   sortTable[0x84E7]=4892;   // diao
   sortTable[0x86C1]=4893;   // diao
   sortTable[0x7AB5]=4894;   // diao
   sortTable[0x9B61]=4895;   // diao
   sortTable[0x866D]=4896;   // diao
   sortTable[0x96FF]=4897;   // diao
   sortTable[0x9CED]=4898;   // diao
   sortTable[0x4F04]=4899;   // diao
   sortTable[0x595D]=4900;   // diao
   sortTable[0x5C4C]=4901;   // diao
   sortTable[0x5F14]=4902;   // diao
   sortTable[0x5F34]=4903;   // diao
   sortTable[0x5F6B]=4904;   // diao
   sortTable[0x625A]=4905;   // diao
   sortTable[0x6BA6]=4906;   // diao
   sortTable[0x6C48]=4907;   // diao
   sortTable[0x7431]=4908;   // diao
   sortTable[0x7639]=4909;   // diao
   sortTable[0x7797]=4910;   // diao
   sortTable[0x7AE8]=4911;   // diao
   sortTable[0x7C13]=4912;   // diao
   sortTable[0x92B1]=4913;   // diao
   sortTable[0x9443]=4914;   // diao
   sortTable[0x9B89]=4915;   // diao
   sortTable[0x9F26]=4916;   // diao
   sortTable[0x7239]=4917;   // die
   sortTable[0x8DCC]=4918;   // die
   sortTable[0x53E0]=4919;   // die
   sortTable[0x758A]=4920;   // die
   sortTable[0x8776]=4921;   // die
   sortTable[0x789F]=4922;   // die
   sortTable[0x8C0D]=4923;   // die
   sortTable[0x8FED]=4924;   // die
   sortTable[0x7252]=4925;   // die
   sortTable[0x558B]=4926;   // die
   sortTable[0x7ED6]=4927;   // die
   sortTable[0x581E]=4928;   // die
   sortTable[0x8E40]=4929;   // die
   sortTable[0x800B]=4930;   // die
   sortTable[0x63F2]=4931;   // die
   sortTable[0x7730]=4932;   // die
   sortTable[0x57A4]=4933;   // die
   sortTable[0x74DE]=4934;   // die
   sortTable[0x9CBD]=4935;   // die
   sortTable[0x7D70]=4936;   // die
   sortTable[0x890B]=4937;   // die
   sortTable[0x8253]=4938;   // die
   sortTable[0x8A44]=4939;   // die
   sortTable[0x8728]=4940;   // die
   sortTable[0x7723]=4941;   // die
   sortTable[0x800A]=4942;   // die
   sortTable[0x957B]=4943;   // die
   sortTable[0x9C08]=4944;   // die
   sortTable[0x82F5]=4945;   // die
   sortTable[0x54CB]=4946;   // die
   sortTable[0x5CCC]=4947;   // die
   sortTable[0x5D7D]=4948;   // die
   sortTable[0x5E49]=4949;   // die
   sortTable[0x604E]=4950;   // die
   sortTable[0x60F5]=4951;   // die
   sortTable[0x621C]=4952;   // die
   sortTable[0x6315]=4953;   // die
   sortTable[0x6633]=4954;   // die
   sortTable[0x66E1]=4955;   // die
   sortTable[0x6B9C]=4956;   // die
   sortTable[0x6C0E]=4957;   // die
   sortTable[0x7243]=4958;   // die
   sortTable[0x7573]=4959;   // die
   sortTable[0x7582]=4960;   // die
   sortTable[0x7589]=4961;   // die
   sortTable[0x80C5]=4962;   // die
   sortTable[0x81F7]=4963;   // die
   sortTable[0x893A]=4964;   // die
   sortTable[0x8ADC]=4965;   // die
   sortTable[0x8D83]=4966;   // die
   sortTable[0x5B9A]=4967;   // ding
   sortTable[0x9876]=4968;   // ding
   sortTable[0x9802]=4969;   // ding
   sortTable[0x8BA2]=4970;   // ding
   sortTable[0x8A02]=4971;   // ding
   sortTable[0x76EF]=4972;   // ding
   sortTable[0x9489]=4973;   // ding
   sortTable[0x91D8]=4974;   // ding
   sortTable[0x53EE]=4975;   // ding
   sortTable[0x4E01]=4976;   // ding
   sortTable[0x952D]=4977;   // ding
   sortTable[0x9320]=4978;   // ding
   sortTable[0x9F0E]=4979;   // ding
   sortTable[0x7887]=4980;   // ding
   sortTable[0x815A]=4981;   // ding
   sortTable[0x914A]=4982;   // ding
   sortTable[0x94E4]=4983;   // ding
   sortTable[0x4EC3]=4984;   // ding
   sortTable[0x5576]=4985;   // ding
   sortTable[0x738E]=4986;   // ding
   sortTable[0x7594]=4987;   // ding
   sortTable[0x8035]=4988;   // ding
   sortTable[0x9964]=4989;   // ding
   sortTable[0x77F4]=4990;   // ding
   sortTable[0x78F8]=4991;   // ding
   sortTable[0x8670]=4992;   // ding
   sortTable[0x9F11]=4993;   // ding
   sortTable[0x9424]=4994;   // ding
   sortTable[0x98E3]=4995;   // ding
   sortTable[0x78A0]=4996;   // ding
   sortTable[0x85A1]=4997;   // ding
   sortTable[0x9841]=4998;   // ding
   sortTable[0x5975]=4999;   // ding
   sortTable[0x5D7F]=5000;   // ding
   sortTable[0x5E04]=5001;   // ding
   sortTable[0x5FCA]=5002;   // ding
   sortTable[0x6917]=5003;   // ding
   sortTable[0x6FCE]=5004;   // ding
   sortTable[0x8062]=5005;   // ding
   sortTable[0x8423]=5006;   // ding
   sortTable[0x874A]=5007;   // ding
   sortTable[0x92CC]=5008;   // ding
   sortTable[0x976A]=5009;   // ding
   sortTable[0x4E22]=5010;   // diu
   sortTable[0x4E1F]=5011;   // diu
   sortTable[0x94E5]=5012;   // diu
   sortTable[0x92A9]=5013;   // diu
   sortTable[0x52A8]=5014;   // dong
   sortTable[0x52D5]=5015;   // dong
   sortTable[0x4E1C]=5016;   // dong
   sortTable[0x6771]=5017;   // dong
   sortTable[0x61C2]=5018;   // dong
   sortTable[0x51AC]=5019;   // dong
   sortTable[0x6D1E]=5020;   // dong
   sortTable[0x51BB]=5021;   // dong
   sortTable[0x51CD]=5022;   // dong
   sortTable[0x8463]=5023;   // dong
   sortTable[0x680B]=5024;   // dong
   sortTable[0x549A]=5025;   // dong
   sortTable[0x5CD2]=5026;   // dong
   sortTable[0x4F97]=5027;   // dong
   sortTable[0x606B]=5028;   // dong
   sortTable[0x80E8]=5029;   // dong
   sortTable[0x6C21]=5030;   // dong
   sortTable[0x9E2B]=5031;   // dong
   sortTable[0x7850]=5032;   // dong
   sortTable[0x5CBD]=5033;   // dong
   sortTable[0x80F4]=5034;   // dong
   sortTable[0x9D87]=5035;   // dong
   sortTable[0x8740]=5036;   // dong
   sortTable[0x578C]=5037;   // dong
   sortTable[0x9F15]=5038;   // dong
   sortTable[0x7BBD]=5039;   // dong
   sortTable[0x9718]=5040;   // dong
   sortTable[0x856B]=5041;   // dong
   sortTable[0x7B17]=5042;   // dong
   sortTable[0x99E7]=5043;   // dong
   sortTable[0x83C4]=5044;   // dong
   sortTable[0x9DAB]=5045;   // dong
   sortTable[0x8FF5]=5046;   // dong
   sortTable[0x9D24]=5047;   // dong
   sortTable[0x5032]=5048;   // dong
   sortTable[0x50CD]=5049;   // dong
   sortTable[0x57EC]=5050;   // dong
   sortTable[0x58A5]=5051;   // dong
   sortTable[0x59DB]=5052;   // dong
   sortTable[0x5A3B]=5053;   // dong
   sortTable[0x5B1E]=5054;   // dong
   sortTable[0x5D20]=5055;   // dong
   sortTable[0x5D2C]=5056;   // dong
   sortTable[0x5F9A]=5057;   // dong
   sortTable[0x6219]=5058;   // dong
   sortTable[0x630F]=5059;   // dong
   sortTable[0x6638]=5060;   // dong
   sortTable[0x68DF]=5061;   // dong
   sortTable[0x6C2D]=5062;   // dong
   sortTable[0x6DB7]=5063;   // dong
   sortTable[0x6E69]=5064;   // dong
   sortTable[0x7D67]=5065;   // dong
   sortTable[0x8156]=5066;   // dong
   sortTable[0x82F3]=5067;   // dong
   sortTable[0x8ACC]=5068;   // dong
   sortTable[0x9B97]=5069;   // dong
   sortTable[0x9BDF]=5070;   // dong
   sortTable[0x90FD]=5071;   // dou
   sortTable[0x6597]=5072;   // dou
   sortTable[0x8C46]=5073;   // dou
   sortTable[0x6296]=5074;   // dou
   sortTable[0x9017]=5075;   // dou
   sortTable[0x515C]=5076;   // dou
   sortTable[0x9661]=5077;   // dou
   sortTable[0x86AA]=5078;   // dou
   sortTable[0x7AA6]=5079;   // dou
   sortTable[0x75D8]=5080;   // dou
   sortTable[0x8130]=5081;   // dou
   sortTable[0x997E]=5082;   // dou
   sortTable[0x8254]=5083;   // dou
   sortTable[0x8538]=5084;   // dou
   sortTable[0x8373]=5085;   // dou
   sortTable[0x9916]=5086;   // dou
   sortTable[0x7AC7]=5087;   // dou
   sortTable[0x7BFC]=5088;   // dou
   sortTable[0x9597]=5089;   // dou
   sortTable[0x9158]=5090;   // dou
   sortTable[0x90D6]=5091;   // dou
   sortTable[0x95D8]=5092;   // dou
   sortTable[0x4E67]=5093;   // dou
   sortTable[0x5160]=5094;   // dou
   sortTable[0x543A]=5095;   // dou
   sortTable[0x5517]=5096;   // dou
   sortTable[0x551E]=5097;   // dou
   sortTable[0x65A3]=5098;   // dou
   sortTable[0x6793]=5099;   // dou
   sortTable[0x67A1]=5100;   // dou
   sortTable[0x68AA]=5101;   // dou
   sortTable[0x6A77]=5102;   // dou
   sortTable[0x6BED]=5103;   // dou
   sortTable[0x6D62]=5104;   // dou
   sortTable[0x9204]=5105;   // dou
   sortTable[0x9627]=5106;   // dou
   sortTable[0x9B25]=5107;   // dou
   sortTable[0x9B26]=5108;   // dou
   sortTable[0x9B2A]=5109;   // dou
   sortTable[0x9B2C]=5110;   // dou
   sortTable[0x9B2D]=5111;   // dou
   sortTable[0x5EA6]=5112;   // du
   sortTable[0x8BFB]=5113;   // du
   sortTable[0x8B80]=5114;   // du
   sortTable[0x72EC]=5115;   // du
   sortTable[0x7368]=5116;   // du
   sortTable[0x6BD2]=5117;   // du
   sortTable[0x6E21]=5118;   // du
   sortTable[0x809A]=5119;   // du
   sortTable[0x5835]=5120;   // du
   sortTable[0x7763]=5121;   // du
   sortTable[0x9540]=5122;   // du
   sortTable[0x934D]=5123;   // du
   sortTable[0x675C]=5124;   // du
   sortTable[0x8D4C]=5125;   // du
   sortTable[0x8CED]=5126;   // du
   sortTable[0x7B03]=5127;   // du
   sortTable[0x7779]=5128;   // du
   sortTable[0x5992]=5129;   // du
   sortTable[0x6E0E]=5130;   // du
   sortTable[0x561F]=5131;   // du
   sortTable[0x724D]=5132;   // du
   sortTable[0x8839]=5133;   // du
   sortTable[0x728A]=5134;   // du
   sortTable[0x828F]=5135;   // du
   sortTable[0x9EE9]=5136;   // du
   sortTable[0x9AD1]=5137;   // du
   sortTable[0x9607]=5138;   // du
   sortTable[0x691F]=5139;   // du
   sortTable[0x78A1]=5140;   // du
   sortTable[0x8B9F]=5141;   // du
   sortTable[0x89A9]=5142;   // du
   sortTable[0x97E3]=5143;   // du
   sortTable[0x95CD]=5144;   // du
   sortTable[0x79FA]=5145;   // du
   sortTable[0x97E5]=5146;   // du
   sortTable[0x97C7]=5147;   // du
   sortTable[0x9A33]=5148;   // du
   sortTable[0x88FB]=5149;   // du
   sortTable[0x945F]=5150;   // du
   sortTable[0x7BE4]=5151;   // du
   sortTable[0x8827]=5152;   // du
   sortTable[0x8773]=5153;   // du
   sortTable[0x8799]=5154;   // du
   sortTable[0x7C35]=5155;   // du
   sortTable[0x9316]=5156;   // du
   sortTable[0x51DF]=5157;   // du
   sortTable[0x5262]=5158;   // du
   sortTable[0x5335]=5159;   // du
   sortTable[0x53BE]=5160;   // du
   sortTable[0x59AC]=5161;   // du
   sortTable[0x5B3B]=5162;   // du
   sortTable[0x5E3E]=5163;   // du
   sortTable[0x6ADD]=5164;   // du
   sortTable[0x6BAC]=5165;   // du
   sortTable[0x6BB0]=5166;   // du
   sortTable[0x6D9C]=5167;   // du
   sortTable[0x7006]=5168;   // du
   sortTable[0x7258]=5169;   // du
   sortTable[0x72A2]=5170;   // du
   sortTable[0x743D]=5171;   // du
   sortTable[0x74C4]=5172;   // du
   sortTable[0x76BE]=5173;   // du
   sortTable[0x8370]=5174;   // du
   sortTable[0x8AAD]=5175;   // du
   sortTable[0x8C44]=5176;   // du
   sortTable[0x8D15]=5177;   // du
   sortTable[0x918F]=5178;   // du
   sortTable[0x976F]=5179;   // du
   sortTable[0x9EF7]=5180;   // du
   sortTable[0x65AD]=5181;   // duan
   sortTable[0x65B7]=5182;   // duan
   sortTable[0x6BB5]=5183;   // duan
   sortTable[0x77ED]=5184;   // duan
   sortTable[0x7AEF]=5185;   // duan
   sortTable[0x953B]=5186;   // duan
   sortTable[0x935B]=5187;   // duan
   sortTable[0x7F0E]=5188;   // duan
   sortTable[0x7DDE]=5189;   // duan
   sortTable[0x6934]=5190;   // duan
   sortTable[0x7145]=5191;   // duan
   sortTable[0x8011]=5192;   // duan
   sortTable[0x8176]=5193;   // duan
   sortTable[0x890D]=5194;   // duan
   sortTable[0x7C16]=5195;   // duan
   sortTable[0x846E]=5196;   // duan
   sortTable[0x5073]=5197;   // duan
   sortTable[0x526C]=5198;   // duan
   sortTable[0x5845]=5199;   // duan
   sortTable[0x5A8F]=5200;   // duan
   sortTable[0x6BC8]=5201;   // duan
   sortTable[0x7456]=5202;   // duan
   sortTable[0x78AB]=5203;   // duan
   sortTable[0x7C6A]=5204;   // duan
   sortTable[0x8E96]=5205;   // duan
   sortTable[0x9374]=5206;   // duan
   sortTable[0x5BF9]=5207;   // dui
   sortTable[0x5C0D]=5208;   // dui
   sortTable[0x961F]=5209;   // dui
   sortTable[0x968A]=5210;   // dui
   sortTable[0x5806]=5211;   // dui
   sortTable[0x5151]=5212;   // dui
   sortTable[0x603C]=5213;   // dui
   sortTable[0x7893]=5214;   // dui
   sortTable[0x619D]=5215;   // dui
   sortTable[0x9566]=5216;   // dui
   sortTable[0x794B]=5217;   // dui
   sortTable[0x9827]=5218;   // dui
   sortTable[0x9413]=5219;   // dui
   sortTable[0x8B48]=5220;   // dui
   sortTable[0x9D2D]=5221;   // dui
   sortTable[0x78D3]=5222;   // dui
   sortTable[0x941C]=5223;   // dui
   sortTable[0x966E]=5224;   // dui
   sortTable[0x514A]=5225;   // dui
   sortTable[0x514C]=5226;   // dui
   sortTable[0x5796]=5227;   // dui
   sortTable[0x5860]=5228;   // dui
   sortTable[0x5BFE]=5229;   // dui
   sortTable[0x5D5F]=5230;   // dui
   sortTable[0x619E]=5231;   // dui
   sortTable[0x61DF]=5232;   // dui
   sortTable[0x6FE7]=5233;   // dui
   sortTable[0x7029]=5234;   // dui
   sortTable[0x75FD]=5235;   // dui
   sortTable[0x7D90]=5236;   // dui
   sortTable[0x85B1]=5237;   // dui
   sortTable[0x8968]=5238;   // dui
   sortTable[0x76FE]=5239;   // dun
   sortTable[0x987F]=5240;   // dun
   sortTable[0x9813]=5241;   // dun
   sortTable[0x5428]=5242;   // dun
   sortTable[0x5678]=5243;   // dun
   sortTable[0x8E72]=5244;   // dun
   sortTable[0x58A9]=5245;   // dun
   sortTable[0x6566]=5246;   // dun
   sortTable[0x9041]=5247;   // dun
   sortTable[0x949D]=5248;   // dun
   sortTable[0x6C8C]=5249;   // dun
   sortTable[0x7096]=5250;   // dun
   sortTable[0x76F9]=5251;   // dun
   sortTable[0x8DB8]=5252;   // dun
   sortTable[0x56E4]=5253;   // dun
   sortTable[0x7818]=5254;   // dun
   sortTable[0x906F]=5255;   // dun
   sortTable[0x7905]=5256;   // dun
   sortTable[0x9A50]=5257;   // dun
   sortTable[0x8E7E]=5258;   // dun
   sortTable[0x920D]=5259;   // dun
   sortTable[0x8733]=5260;   // dun
   sortTable[0x9007]=5261;   // dun
   sortTable[0x8E89]=5262;   // dun
   sortTable[0x4F05]=5263;   // dun
   sortTable[0x58AA]=5264;   // dun
   sortTable[0x5E89]=5265;   // dun
   sortTable[0x60C7]=5266;   // dun
   sortTable[0x6489]=5267;   // dun
   sortTable[0x64B4]=5268;   // dun
   sortTable[0x696F]=5269;   // dun
   sortTable[0x6A54]=5270;   // dun
   sortTable[0x6F61]=5271;   // dun
   sortTable[0x71C9]=5272;   // dun
   sortTable[0x729C]=5273;   // dun
   sortTable[0x7364]=5274;   // dun
   sortTable[0x78B7]=5275;   // dun
   sortTable[0x8E32]=5276;   // dun
   sortTable[0x591A]=5277;   // duo
   sortTable[0x593A]=5278;   // duo
   sortTable[0x596A]=5279;   // duo
   sortTable[0x8EB2]=5280;   // duo
   sortTable[0x6735]=5281;   // duo
   sortTable[0x8E31]=5282;   // duo
   sortTable[0x54C6]=5283;   // duo
   sortTable[0x8DFA]=5284;   // duo
   sortTable[0x5815]=5285;   // duo
   sortTable[0x94CE]=5286;   // duo
   sortTable[0x5484]=5287;   // duo
   sortTable[0x60F0]=5288;   // duo
   sortTable[0x8235]=5289;   // duo
   sortTable[0x88F0]=5290;   // duo
   sortTable[0x6387]=5291;   // duo
   sortTable[0x579B]=5292;   // duo
   sortTable[0x54DA]=5293;   // duo
   sortTable[0x9438]=5294;   // duo
   sortTable[0x5241]=5295;   // duo
   sortTable[0x7F0D]=5296;   // duo
   sortTable[0x67C1]=5297;   // duo
   sortTable[0x9973]=5298;   // duo
   sortTable[0x964A]=5299;   // duo
   sortTable[0x8EC3]=5300;   // duo
   sortTable[0x9D7D]=5301;   // duo
   sortTable[0x8D93]=5302;   // duo
   sortTable[0x8EB1]=5303;   // duo
   sortTable[0x964F]=5304;   // duo
   sortTable[0x98FF]=5305;   // duo
   sortTable[0x9BB5]=5306;   // duo
   sortTable[0x4EB8]=5307;   // duo
   sortTable[0x51D9]=5308;   // duo
   sortTable[0x5234]=5309;   // duo
   sortTable[0x525F]=5310;   // duo
   sortTable[0x526B]=5311;   // duo
   sortTable[0x55A5]=5312;   // duo
   sortTable[0x5689]=5313;   // duo
   sortTable[0x56B2]=5314;   // duo
   sortTable[0x579C]=5315;   // duo
   sortTable[0x57F5]=5316;   // duo
   sortTable[0x58AE]=5317;   // duo
   sortTable[0x58AF]=5318;   // duo
   sortTable[0x591B]=5319;   // duo
   sortTable[0x5972]=5320;   // duo
   sortTable[0x5C2E]=5321;   // duo
   sortTable[0x5D1C]=5322;   // duo
   sortTable[0x5D9E]=5323;   // duo
   sortTable[0x60B3]=5324;   // duo
   sortTable[0x619C]=5325;   // duo
   sortTable[0x6305]=5326;   // duo
   sortTable[0x6306]=5327;   // duo
   sortTable[0x6553]=5328;   // duo
   sortTable[0x655A]=5329;   // duo
   sortTable[0x6560]=5330;   // duo
   sortTable[0x656A]=5331;   // duo
   sortTable[0x6736]=5332;   // duo
   sortTable[0x67A4]=5333;   // duo
   sortTable[0x67EE]=5334;   // duo
   sortTable[0x6857]=5335;   // duo
   sortTable[0x692F]=5336;   // duo
   sortTable[0x6BF2]=5337;   // duo
   sortTable[0x70A8]=5338;   // duo
   sortTable[0x7553]=5339;   // duo
   sortTable[0x75E5]=5340;   // duo
   sortTable[0x7D9E]=5341;   // duo
   sortTable[0x8DE2]=5342;   // duo
   sortTable[0x8DE5]=5343;   // duo
   sortTable[0x922C]=5344;   // duo
   sortTable[0x937A]=5345;   // duo
   sortTable[0x7565]=5632;   // e
   sortTable[0x997F]=5633;   // e
   sortTable[0x9913]=5634;   // e
   sortTable[0x6076]=5635;   // e
   sortTable[0x60E1]=5636;   // e
   sortTable[0x989D]=5637;   // e
   sortTable[0x984D]=5638;   // e
   sortTable[0x63A0]=5639;   // e
   sortTable[0x9E45]=5640;   // e
   sortTable[0x9D5D]=5641;   // e
   sortTable[0x6115]=5642;   // e
   sortTable[0x5443]=5643;   // e
   sortTable[0x9CC4]=5644;   // e
   sortTable[0x9C77]=5645;   // e
   sortTable[0x4FC4]=5646;   // e
   sortTable[0x5CE8]=5647;   // e
   sortTable[0x86FE]=5648;   // e
   sortTable[0x5A25]=5649;   // e
   sortTable[0x5384]=5650;   // e
   sortTable[0x9102]=5651;   // e
   sortTable[0x904F]=5652;   // e
   sortTable[0x627C]=5653;   // e
   sortTable[0x8BB9]=5654;   // e
   sortTable[0x8C14]=5655;   // e
   sortTable[0x843C]=5656;   // e
   sortTable[0x5669]=5657;   // e
   sortTable[0x9E57]=5658;   // e
   sortTable[0x960F]=5659;   // e
   sortTable[0x9537]=5660;   // e
   sortTable[0x5A40]=5661;   // e
   sortTable[0x989A]=5662;   // e
   sortTable[0x57A9]=5663;   // e
   sortTable[0x82CA]=5664;   // e
   sortTable[0x9507]=5665;   // e
   sortTable[0x8F6D]=5666;   // e
   sortTable[0x816D]=5667;   // e
   sortTable[0x5C59]=5668;   // e
   sortTable[0x83AA]=5669;   // e
   sortTable[0x9628]=5670;   // e
   sortTable[0x981E]=5671;   // e
   sortTable[0x94B6]=5672;   // e
   sortTable[0x8A7B]=5673;   // e
   sortTable[0x8B4C]=5674;   // e
   sortTable[0x8A1B]=5675;   // e
   sortTable[0x774B]=5676;   // e
   sortTable[0x9C2A]=5677;   // e
   sortTable[0x855A]=5678;   // e
   sortTable[0x8EF6]=5679;   // e
   sortTable[0x9D9A]=5680;   // e
   sortTable[0x922A]=5681;   // e
   sortTable[0x981F]=5682;   // e
   sortTable[0x8FD7]=5683;   // e
   sortTable[0x96B2]=5684;   // e
   sortTable[0x8EDB]=5685;   // e
   sortTable[0x9638]=5686;   // e
   sortTable[0x980B]=5687;   // e
   sortTable[0x984E]=5688;   // e
   sortTable[0x9F43]=5689;   // e
   sortTable[0x8685]=5690;   // e
   sortTable[0x8A90]=5691;   // e
   sortTable[0x9354]=5692;   // e
   sortTable[0x7828]=5693;   // e
   sortTable[0x7918]=5694;   // e
   sortTable[0x8741]=5695;   // e
   sortTable[0x8B8D]=5696;   // e
   sortTable[0x920B]=5697;   // e
   sortTable[0x9A00]=5698;   // e
   sortTable[0x9B65]=5699;   // e
   sortTable[0x9F76]=5700;   // e
   sortTable[0x5054]=5701;   // e
   sortTable[0x50EB]=5702;   // e
   sortTable[0x530E]=5703;   // e
   sortTable[0x537E]=5704;   // e
   sortTable[0x542A]=5705;   // e
   sortTable[0x545D]=5706;   // e
   sortTable[0x54A2]=5707;   // e
   sortTable[0x54B9]=5708;   // e
   sortTable[0x5641]=5709;   // e
   sortTable[0x56EE]=5710;   // e
   sortTable[0x580A]=5711;   // e
   sortTable[0x582E]=5712;   // e
   sortTable[0x59B8]=5713;   // e
   sortTable[0x59BF]=5714;   // e
   sortTable[0x59F6]=5715;   // e
   sortTable[0x5A3F]=5716;   // e
   sortTable[0x5C75]=5717;   // e
   sortTable[0x5C8B]=5718;   // e
   sortTable[0x5CC9]=5719;   // e
   sortTable[0x5CE9]=5720;   // e
   sortTable[0x5D3F]=5721;   // e
   sortTable[0x5EC5]=5722;   // e
   sortTable[0x60AA]=5723;   // e
   sortTable[0x6239]=5724;   // e
   sortTable[0x6424]=5725;   // e
   sortTable[0x6439]=5726;   // e
   sortTable[0x64DC]=5727;   // e
   sortTable[0x6799]=5728;   // e
   sortTable[0x6AEE]=5729;   // e
   sortTable[0x6B5E]=5730;   // e
   sortTable[0x6B7A]=5731;   // e
   sortTable[0x6D90]=5732;   // e
   sortTable[0x6E42]=5733;   // e
   sortTable[0x73F4]=5734;   // e
   sortTable[0x7427]=5735;   // e
   sortTable[0x75FE]=5736;   // e
   sortTable[0x7692]=5737;   // e
   sortTable[0x7808]=5738;   // e
   sortTable[0x7810]=5739;   // e
   sortTable[0x7846]=5740;   // e
   sortTable[0x78C0]=5741;   // e
   sortTable[0x89A8]=5742;   // e
   sortTable[0x8AE4]=5743;   // e
   sortTable[0x8C5F]=5744;   // e
   sortTable[0x8C96]=5745;   // e
   sortTable[0x904C]=5746;   // e
   sortTable[0x9469]=5747;   // e
   sortTable[0x95BC]=5748;   // e
   sortTable[0x9929]=5749;   // e
   sortTable[0x9B64]=5750;   // e
   sortTable[0x9C10]=5751;   // e
   sortTable[0x9D48]=5752;   // e
   sortTable[0x9D5E]=5753;   // e
   sortTable[0x8BF6]=5754;   // ei
   sortTable[0x8A92]=5755;   // ei
   sortTable[0x55EF]=5756;   // en
   sortTable[0x6069]=5757;   // en
   sortTable[0x6441]=5758;   // en
   sortTable[0x84BD]=5759;   // en
   sortTable[0x5940]=5760;   // en
   sortTable[0x5CCE]=5761;   // en
   sortTable[0x717E]=5762;   // en
   sortTable[0x97A5]=5763;   // eng
   sortTable[0x800C]=5764;   // er
   sortTable[0x513F]=5765;   // er
   sortTable[0x5152]=5766;   // er
   sortTable[0x4E8C]=5767;   // er
   sortTable[0x8033]=5768;   // er
   sortTable[0x5C14]=5769;   // er
   sortTable[0x723E]=5770;   // er
   sortTable[0x8D30]=5771;   // er
   sortTable[0x8FE9]=5772;   // er
   sortTable[0x9975]=5773;   // er
   sortTable[0x73E5]=5774;   // er
   sortTable[0x9E38]=5775;   // er
   sortTable[0x6D31]=5776;   // er
   sortTable[0x94D2]=5777;   // er
   sortTable[0x4F74]=5778;   // er
   sortTable[0x7CAB]=5779;   // er
   sortTable[0x8F5C]=5780;   // er
   sortTable[0x9C95]=5781;   // er
   sortTable[0x8F00]=5782;   // er
   sortTable[0x8A80]=5783;   // er
   sortTable[0x9087]=5784;   // er
   sortTable[0x9D2F]=5785;   // er
   sortTable[0x8DB0]=5786;   // er
   sortTable[0x99EC]=5787;   // er
   sortTable[0x80F9]=5788;   // er
   sortTable[0x85BE]=5789;   // er
   sortTable[0x9651]=5790;   // er
   sortTable[0x834B]=5791;   // er
   sortTable[0x8848]=5792;   // er
   sortTable[0x8CAE]=5793;   // er
   sortTable[0x990C]=5794;   // er
   sortTable[0x9AF5]=5795;   // er
   sortTable[0x4F95]=5796;   // er
   sortTable[0x5150]=5797;   // er
   sortTable[0x5235]=5798;   // er
   sortTable[0x53BC]=5799;   // er
   sortTable[0x54A1]=5800;   // er
   sortTable[0x5532]=5801;   // er
   sortTable[0x5C12]=5802;   // er
   sortTable[0x5C13]=5803;   // er
   sortTable[0x5CCF]=5804;   // er
   sortTable[0x5F0D]=5805;   // er
   sortTable[0x5F10]=5806;   // er
   sortTable[0x682D]=5807;   // er
   sortTable[0x682E]=5808;   // er
   sortTable[0x6A32]=5809;   // er
   sortTable[0x6BE6]=5810;   // er
   sortTable[0x6D0F]=5811;   // er
   sortTable[0x804F]=5812;   // er
   sortTable[0x88BB]=5813;   // er
   sortTable[0x8CB3]=5814;   // er
   sortTable[0x927A]=5815;   // er
   sortTable[0x96AD]=5816;   // er
   sortTable[0x9B9E]=5817;   // er
   sortTable[0x53D1]=6144;   // fa
   sortTable[0x767C]=6145;   // fa
   sortTable[0x6CD5]=6146;   // fa
   sortTable[0x4E4F]=6147;   // fa
   sortTable[0x4F10]=6148;   // fa
   sortTable[0x9600]=6149;   // fa
   sortTable[0x95A5]=6150;   // fa
   sortTable[0x7F5A]=6151;   // fa
   sortTable[0x7F70]=6152;   // fa
   sortTable[0x7B4F]=6153;   // fa
   sortTable[0x57A1]=6154;   // fa
   sortTable[0x781D]=6155;   // fa
   sortTable[0x73D0]=6156;   // fa
   sortTable[0x9AEE]=6157;   // fa
   sortTable[0x8337]=6158;   // fa
   sortTable[0x9AEA]=6159;   // fa
   sortTable[0x7782]=6160;   // fa
   sortTable[0x9166]=6161;   // fa
   sortTable[0x855F]=6162;   // fa
   sortTable[0x85C5]=6163;   // fa
   sortTable[0x91B1]=6164;   // fa
   sortTable[0x4F71]=6165;   // fa
   sortTable[0x50A0]=6166;   // fa
   sortTable[0x59C2]=6167;   // fa
   sortTable[0x5F42]=6168;   // fa
   sortTable[0x6830]=6169;   // fa
   sortTable[0x6A43]=6170;   // fa
   sortTable[0x6CB7]=6171;   // fa
   sortTable[0x6D4C]=6172;   // fa
   sortTable[0x704B]=6173;   // fa
   sortTable[0x743A]=6174;   // fa
   sortTable[0x75BA]=6175;   // fa
   sortTable[0x767A]=6176;   // fa
   sortTable[0x7F78]=6177;   // fa
   sortTable[0x9345]=6178;   // fa
   sortTable[0x53CD]=6179;   // fan
   sortTable[0x996D]=6180;   // fan
   sortTable[0x98EF]=6181;   // fan
   sortTable[0x7FFB]=6182;   // fan
   sortTable[0x72AF]=6183;   // fan
   sortTable[0x8303]=6184;   // fan
   sortTable[0x6CDB]=6185;   // fan
   sortTable[0x51E1]=6186;   // fan
   sortTable[0x7E41]=6187;   // fan
   sortTable[0x756A]=6188;   // fan
   sortTable[0x70E6]=6189;   // fan
   sortTable[0x7169]=6190;   // fan
   sortTable[0x8FD4]=6191;   // fan
   sortTable[0x5E06]=6192;   // fan
   sortTable[0x8D29]=6193;   // fan
   sortTable[0x8CA9]=6194;   // fan
   sortTable[0x8543]=6195;   // fan
   sortTable[0x85E9]=6196;   // fan
   sortTable[0x6A0A]=6197;   // fan
   sortTable[0x7C53]=6198;   // fan
   sortTable[0x5E61]=6199;   // fan
   sortTable[0x68B5]=6200;   // fan
   sortTable[0x71D4]=6201;   // fan
   sortTable[0x7548]=6202;   // fan
   sortTable[0x77FE]=6203;   // fan
   sortTable[0x8629]=6204;   // fan
   sortTable[0x9492]=6205;   // fan
   sortTable[0x9407]=6206;   // fan
   sortTable[0x8E6F]=6207;   // fan
   sortTable[0x7E59]=6208;   // fan
   sortTable[0x8F53]=6209;   // fan
   sortTable[0x7BC4]=6210;   // fan
   sortTable[0x81B0]=6211;   // fan
   sortTable[0x8A09]=6212;   // fan
   sortTable[0x881C]=6213;   // fan
   sortTable[0x8ED3]=6214;   // fan
   sortTable[0x7B32]=6215;   // fan
   sortTable[0x7B35]=6216;   // fan
   sortTable[0x9422]=6217;   // fan
   sortTable[0x98BF]=6218;   // fan
   sortTable[0x98F0]=6219;   // fan
   sortTable[0x85A0]=6220;   // fan
   sortTable[0x98DC]=6221;   // fan
   sortTable[0x76D5]=6222;   // fan
   sortTable[0x7DD0]=6223;   // fan
   sortTable[0x9C55]=6224;   // fan
   sortTable[0x9DED]=6225;   // fan
   sortTable[0x7C75]=6226;   // fan
   sortTable[0x7FB3]=6227;   // fan
   sortTable[0x8EEC]=6228;   // fan
   sortTable[0x91E9]=6229;   // fan
   sortTable[0x51E2]=6230;   // fan
   sortTable[0x51E3]=6231;   // fan
   sortTable[0x52EB]=6232;   // fan
   sortTable[0x5643]=6233;   // fan
   sortTable[0x58A6]=6234;   // fan
   sortTable[0x597F]=6235;   // fan
   sortTable[0x5A4F]=6236;   // fan
   sortTable[0x5B0E]=6237;   // fan
   sortTable[0x5B0F]=6238;   // fan
   sortTable[0x5FDB]=6239;   // fan
   sortTable[0x61A3]=6240;   // fan
   sortTable[0x6255]=6241;   // fan
   sortTable[0x65D9]=6242;   // fan
   sortTable[0x65DB]=6243;   // fan
   sortTable[0x674B]=6244;   // fan
   sortTable[0x67C9]=6245;   // fan
   sortTable[0x68E5]=6246;   // fan
   sortTable[0x6A4E]=6247;   // fan
   sortTable[0x6C3E]=6248;   // fan
   sortTable[0x6C4E]=6249;   // fan
   sortTable[0x6E22]=6250;   // fan
   sortTable[0x6EFC]=6251;   // fan
   sortTable[0x702A]=6252;   // fan
   sortTable[0x703F]=6253;   // fan
   sortTable[0x74A0]=6254;   // fan
   sortTable[0x792C]=6255;   // fan
   sortTable[0x8224]=6256;   // fan
   sortTable[0x8227]=6257;   // fan
   sortTable[0x894E]=6258;   // fan
   sortTable[0x65B9]=6259;   // fang
   sortTable[0x653E]=6260;   // fang
   sortTable[0x623F]=6261;   // fang
   sortTable[0x9632]=6262;   // fang
   sortTable[0x4EFF]=6263;   // fang
   sortTable[0x7EBA]=6264;   // fang
   sortTable[0x7D21]=6265;   // fang
   sortTable[0x8BBF]=6266;   // fang
   sortTable[0x8A2A]=6267;   // fang
   sortTable[0x59A8]=6268;   // fang
   sortTable[0x574A]=6269;   // fang
   sortTable[0x80AA]=6270;   // fang
   sortTable[0x82B3]=6271;   // fang
   sortTable[0x822B]=6272;   // fang
   sortTable[0x5F77]=6273;   // fang
   sortTable[0x531A]=6274;   // fang
   sortTable[0x9C82]=6275;   // fang
   sortTable[0x678B]=6276;   // fang
   sortTable[0x94AB]=6277;   // fang
   sortTable[0x90A1]=6278;   // fang
   sortTable[0x8684]=6279;   // fang
   sortTable[0x9AE3]=6280;   // fang
   sortTable[0x9D0B]=6281;   // fang
   sortTable[0x8DBD]=6282;   // fang
   sortTable[0x9C1F]=6283;   // fang
   sortTable[0x7706]=6284;   // fang
   sortTable[0x9B74]=6285;   // fang
   sortTable[0x9DAD]=6286;   // fang
   sortTable[0x5023]=6287;   // fang
   sortTable[0x57C5]=6288;   // fang
   sortTable[0x580F]=6289;   // fang
   sortTable[0x65CA]=6290;   // fang
   sortTable[0x6609]=6291;   // fang
   sortTable[0x6618]=6292;   // fang
   sortTable[0x6C78]=6293;   // fang
   sortTable[0x6DD3]=6294;   // fang
   sortTable[0x7265]=6295;   // fang
   sortTable[0x74EC]=6296;   // fang
   sortTable[0x9201]=6297;   // fang
   sortTable[0x933A]=6298;   // fang
   sortTable[0x98DE]=6299;   // fei
   sortTable[0x98DB]=6300;   // fei
   sortTable[0x975E]=6301;   // fei
   sortTable[0x8D39]=6302;   // fei
   sortTable[0x8CBB]=6303;   // fei
   sortTable[0x80A5]=6304;   // fei
   sortTable[0x532A]=6305;   // fei
   sortTable[0x5E9F]=6306;   // fei
   sortTable[0x5EE2]=6307;   // fei
   sortTable[0x80BA]=6308;   // fei
   sortTable[0x6CB8]=6309;   // fei
   sortTable[0x5561]=6310;   // fei
   sortTable[0x83F2]=6311;   // fei
   sortTable[0x5983]=6312;   // fei
   sortTable[0x6590]=6313;   // fei
   sortTable[0x7EEF]=6314;   // fei
   sortTable[0x5420]=6315;   // fei
   sortTable[0x8BFD]=6316;   // fei
   sortTable[0x72D2]=6317;   // fei
   sortTable[0x6249]=6318;   // fei
   sortTable[0x8153]=6319;   // fei
   sortTable[0x7FE1]=6320;   // fei
   sortTable[0x6DDD]=6321;   // fei
   sortTable[0x82BE]=6322;   // fei
   sortTable[0x871A]=6323;   // fei
   sortTable[0x970F]=6324;   // fei
   sortTable[0x69A7]=6325;   // fei
   sortTable[0x7BDA]=6326;   // fei
   sortTable[0x9544]=6327;   // fei
   sortTable[0x60B1]=6328;   // fei
   sortTable[0x9CB1]=6329;   // fei
   sortTable[0x75F1]=6330;   // fei
   sortTable[0x9A11]=6331;   // fei
   sortTable[0x99A1]=6332;   // fei
   sortTable[0x9BE1]=6333;   // fei
   sortTable[0x855C]=6334;   // fei
   sortTable[0x9428]=6335;   // fei
   sortTable[0x9F23]=6336;   // fei
   sortTable[0x80C7]=6337;   // fei
   sortTable[0x8730]=6338;   // fei
   sortTable[0x88F6]=6339;   // fei
   sortTable[0x9A1B]=6340;   // fei
   sortTable[0x7DCB]=6341;   // fei
   sortTable[0x8409]=6342;   // fei
   sortTable[0x87E6]=6343;   // fei
   sortTable[0x9925]=6344;   // fei
   sortTable[0x4FF7]=6345;   // fei
   sortTable[0x5255]=6346;   // fei
   sortTable[0x539E]=6347;   // fei
   sortTable[0x595C]=6348;   // fei
   sortTable[0x5A53]=6349;   // fei
   sortTable[0x5A54]=6350;   // fei
   sortTable[0x5C5D]=6351;   // fei
   sortTable[0x5EC3]=6352;   // fei
   sortTable[0x6632]=6353;   // fei
   sortTable[0x6683]=6354;   // fei
   sortTable[0x66CA]=6355;   // fei
   sortTable[0x670F]=6356;   // fei
   sortTable[0x676E]=6357;   // fei
   sortTable[0x68D0]=6358;   // fei
   sortTable[0x6AE0]=6359;   // fei
   sortTable[0x6E04]=6360;   // fei
   sortTable[0x6FF7]=6361;   // fei
   sortTable[0x7306]=6362;   // fei
   sortTable[0x75BF]=6363;   // fei
   sortTable[0x7648]=6364;   // fei
   sortTable[0x8AB9]=6365;   // fei
   sortTable[0x966B]=6366;   // fei
   sortTable[0x9745]=6367;   // fei
   sortTable[0x975F]=6368;   // fei
   sortTable[0x98DD]=6369;   // fei
   sortTable[0x5206]=6370;   // fen
   sortTable[0x594B]=6371;   // fen
   sortTable[0x596E]=6372;   // fen
   sortTable[0x7C89]=6373;   // fen
   sortTable[0x6124]=6374;   // fen
   sortTable[0x61A4]=6375;   // fen
   sortTable[0x7EB7]=6376;   // fen
   sortTable[0x7D1B]=6377;   // fen
   sortTable[0x4EFD]=6378;   // fen
   sortTable[0x5FFF]=6379;   // fen
   sortTable[0x5429]=6380;   // fen
   sortTable[0x575F]=6381;   // fen
   sortTable[0x58B3]=6382;   // fen
   sortTable[0x7CAA]=6383;   // fen
   sortTable[0x7CDE]=6384;   // fen
   sortTable[0x6C1B]=6385;   // fen
   sortTable[0x82AC]=6386;   // fen
   sortTable[0x711A]=6387;   // fen
   sortTable[0x6C7E]=6388;   // fen
   sortTable[0x507E]=6389;   // fen
   sortTable[0x915A]=6390;   // fen
   sortTable[0x68FC]=6391;   // fen
   sortTable[0x8561]=6392;   // fen
   sortTable[0x86A1]=6393;   // fen
   sortTable[0x96F0]=6394;   // fen
   sortTable[0x7FC2]=6395;   // fen
   sortTable[0x7035]=6396;   // fen
   sortTable[0x9F22]=6397;   // fen
   sortTable[0x81B9]=6398;   // fen
   sortTable[0x9CBC]=6399;   // fen
   sortTable[0x9959]=6400;   // fen
   sortTable[0x7F92]=6401;   // fen
   sortTable[0x7FB5]=6402;   // fen
   sortTable[0x9F16]=6403;   // fen
   sortTable[0x8A1C]=6404;   // fen
   sortTable[0x9216]=6405;   // fen
   sortTable[0x9B75]=6406;   // fen
   sortTable[0x8C6E]=6407;   // fen
   sortTable[0x8F52]=6408;   // fen
   sortTable[0x9934]=6409;   // fen
   sortTable[0x9EFA]=6410;   // fen
   sortTable[0x50E8]=6411;   // fen
   sortTable[0x515D]=6412;   // fen
   sortTable[0x517A]=6413;   // fen
   sortTable[0x54DB]=6414;   // fen
   sortTable[0x59A2]=6415;   // fen
   sortTable[0x5C8E]=6416;   // fen
   sortTable[0x5E09]=6417;   // fen
   sortTable[0x5E69]=6418;   // fen
   sortTable[0x5F05]=6419;   // fen
   sortTable[0x6610]=6420;   // fen
   sortTable[0x6706]=6421;   // fen
   sortTable[0x670C]=6422;   // fen
   sortTable[0x678C]=6423;   // fen
   sortTable[0x68A4]=6424;   // fen
   sortTable[0x68FB]=6425;   // fen
   sortTable[0x6A68]=6426;   // fen
   sortTable[0x6FC6]=6427;   // fen
   sortTable[0x7083]=6428;   // fen
   sortTable[0x71CC]=6429;   // fen
   sortTable[0x71D3]=6430;   // fen
   sortTable[0x79CE]=6431;   // fen
   sortTable[0x7AD5]=6432;   // fen
   sortTable[0x80A6]=6433;   // fen
   sortTable[0x84B6]=6434;   // fen
   sortTable[0x86A0]=6435;   // fen
   sortTable[0x886F]=6436;   // fen
   sortTable[0x8C76]=6437;   // fen
   sortTable[0x8EAE]=6438;   // fen
   sortTable[0x943C]=6439;   // fen
   sortTable[0x96AB]=6440;   // fen
   sortTable[0x999A]=6441;   // fen
   sortTable[0x99A9]=6442;   // fen
   sortTable[0x9C5D]=6443;   // fen
   sortTable[0x9EC2]=6444;   // fen
   sortTable[0x98CE]=6445;   // feng
   sortTable[0x98A8]=6446;   // feng
   sortTable[0x5C01]=6447;   // feng
   sortTable[0x4E30]=6448;   // feng
   sortTable[0x75AF]=6449;   // feng
   sortTable[0x760B]=6450;   // feng
   sortTable[0x8702]=6451;   // feng
   sortTable[0x5CF0]=6452;   // feng
   sortTable[0x7F1D]=6453;   // feng
   sortTable[0x7E2B]=6454;   // feng
   sortTable[0x950B]=6455;   // feng
   sortTable[0x92D2]=6456;   // feng
   sortTable[0x9022]=6457;   // feng
   sortTable[0x5949]=6458;   // feng
   sortTable[0x51E4]=6459;   // feng
   sortTable[0x9CF3]=6460;   // feng
   sortTable[0x8BBD]=6461;   // feng
   sortTable[0x8AF7]=6462;   // feng
   sortTable[0x51AF]=6463;   // feng
   sortTable[0x4FF8]=6464;   // feng
   sortTable[0x67AB]=6465;   // feng
   sortTable[0x70FD]=6466;   // feng
   sortTable[0x8451]=6467;   // feng
   sortTable[0x9146]=6468;   // feng
   sortTable[0x6CA3]=6469;   // feng
   sortTable[0x781C]=6470;   // feng
   sortTable[0x8D57]=6471;   // feng
   sortTable[0x552A]=6472;   // feng
   sortTable[0x8982]=6473;   // feng
   sortTable[0x8CF5]=6474;   // feng
   sortTable[0x9137]=6475;   // feng
   sortTable[0x9EB7]=6476;   // feng
   sortTable[0x8C50]=6477;   // feng
   sortTable[0x8634]=6478;   // feng
   sortTable[0x7D98]=6479;   // feng
   sortTable[0x8242]=6480;   // feng
   sortTable[0x99AE]=6481;   // feng
   sortTable[0x882D]=6482;   // feng
   sortTable[0x93E0]=6483;   // feng
   sortTable[0x76FD]=6484;   // feng
   sortTable[0x78B8]=6485;   // feng
   sortTable[0x7BC8]=6486;   // feng
   sortTable[0x974A]=6487;   // feng
   sortTable[0x98CC]=6488;   // feng
   sortTable[0x4EF9]=6489;   // feng
   sortTable[0x5051]=6490;   // feng
   sortTable[0x50FC]=6491;   // feng
   sortTable[0x51E8]=6492;   // feng
   sortTable[0x51EC]=6493;   // feng
   sortTable[0x51EE]=6494;   // feng
   sortTable[0x5838]=6495;   // feng
   sortTable[0x5906]=6496;   // feng
   sortTable[0x59A6]=6497;   // feng
   sortTable[0x5BF7]=6498;   // feng
   sortTable[0x5CEF]=6499;   // feng
   sortTable[0x5D36]=6500;   // feng
   sortTable[0x6340]=6501;   // feng
   sortTable[0x6453]=6502;   // feng
   sortTable[0x687B]=6503;   // feng
   sortTable[0x6953]=6504;   // feng
   sortTable[0x6A92]=6505;   // feng
   sortTable[0x6CA8]=6506;   // feng
   sortTable[0x6D72]=6507;   // feng
   sortTable[0x6E57]=6508;   // feng
   sortTable[0x6E84]=6509;   // feng
   sortTable[0x6F28]=6510;   // feng
   sortTable[0x7043]=6511;   // feng
   sortTable[0x7128]=6512;   // feng
   sortTable[0x7148]=6513;   // feng
   sortTable[0x728E]=6514;   // feng
   sortTable[0x7326]=6515;   // feng
   sortTable[0x7412]=6516;   // feng
   sortTable[0x752E]=6517;   // feng
   sortTable[0x8615]=6518;   // feng
   sortTable[0x93BD]=6519;   // feng
   sortTable[0x95CF]=6520;   // feng
   sortTable[0x973B]=6521;   // feng
   sortTable[0x9CEF]=6522;   // feng
   sortTable[0x9D0C]=6523;   // feng
   sortTable[0x8985]=6524;   // fiao
   sortTable[0x4ECF]=6525;   // fo
   sortTable[0x5772]=6526;   // fo
   sortTable[0x68BB]=6527;   // fo
   sortTable[0x5426]=6528;   // fou
   sortTable[0x7F36]=6529;   // fou
   sortTable[0x7D11]=6530;   // fou
   sortTable[0x7F3B]=6531;   // fou
   sortTable[0x7F39]=6532;   // fou
   sortTable[0x88E6]=6533;   // fou
   sortTable[0x9D00]=6534;   // fou
   sortTable[0x599A]=6535;   // fou
   sortTable[0x6B95]=6536;   // fou
   sortTable[0x96EC]=6537;   // fou
   sortTable[0x590D]=6538;   // fu
   sortTable[0x8907]=6539;   // fu
   sortTable[0x670D]=6540;   // fu
   sortTable[0x7236]=6541;   // fu
   sortTable[0x5E9C]=6542;   // fu
   sortTable[0x592B]=6543;   // fu
   sortTable[0x8D1F]=6544;   // fu
   sortTable[0x8CA0]=6545;   // fu
   sortTable[0x5987]=6546;   // fu
   sortTable[0x5A66]=6547;   // fu
   sortTable[0x5BCC]=6548;   // fu
   sortTable[0x526F]=6549;   // fu
   sortTable[0x4F5B]=6550;   // fu
   sortTable[0x5085]=6551;   // fu
   sortTable[0x798F]=6552;   // fu
   sortTable[0x9644]=6553;   // fu
   sortTable[0x6D6E]=6554;   // fu
   sortTable[0x5E45]=6555;   // fu
   sortTable[0x4F0F]=6556;   // fu
   sortTable[0x8150]=6557;   // fu
   sortTable[0x80A4]=6558;   // fu
   sortTable[0x819A]=6559;   // fu
   sortTable[0x7B26]=6560;   // fu
   sortTable[0x4ED8]=6561;   // fu
   sortTable[0x5490]=6562;   // fu
   sortTable[0x6276]=6563;   // fu
   sortTable[0x65A7]=6564;   // fu
   sortTable[0x88B1]=6565;   // fu
   sortTable[0x7F1A]=6566;   // fu
   sortTable[0x7E1B]=6567;   // fu
   sortTable[0x8F85]=6568;   // fu
   sortTable[0x4FD8]=6569;   // fu
   sortTable[0x8F90]=6570;   // fu
   sortTable[0x8F14]=6571;   // fu
   sortTable[0x8F3B]=6572;   // fu
   sortTable[0x8986]=6573;   // fu
   sortTable[0x8179]=6574;   // fu
   sortTable[0x4FEF]=6575;   // fu
   sortTable[0x6577]=6576;   // fu
   sortTable[0x5B75]=6577;   // fu
   sortTable[0x8760]=6578;   // fu
   sortTable[0x629A]=6579;   // fu
   sortTable[0x64AB]=6580;   // fu
   sortTable[0x8D74]=6581;   // fu
   sortTable[0x752B]=6582;   // fu
   sortTable[0x51EB]=6583;   // fu
   sortTable[0x9CE7]=6584;   // fu
   sortTable[0x5F17]=6585;   // fu
   sortTable[0x8D4B]=6586;   // fu
   sortTable[0x62C2]=6587;   // fu
   sortTable[0x8299]=6588;   // fu
   sortTable[0x961C]=6589;   // fu
   sortTable[0x5B5A]=6590;   // fu
   sortTable[0x9A78]=6591;   // fu
   sortTable[0x7954]=6592;   // fu
   sortTable[0x8151]=6593;   // fu
   sortTable[0x82FB]=6594;   // fu
   sortTable[0x91DC]=6595;   // fu
   sortTable[0x911C]=6596;   // fu
   sortTable[0x99A5]=6597;   // fu
   sortTable[0x62CA]=6598;   // fu
   sortTable[0x9C8B]=6599;   // fu
   sortTable[0x86A8]=6600;   // fu
   sortTable[0x8DBA]=6601;   // fu
   sortTable[0x83D4]=6602;   // fu
   sortTable[0x6DAA]=6603;   // fu
   sortTable[0x9EFC]=6604;   // fu
   sortTable[0x6C1F]=6605;   // fu
   sortTable[0x8BA3]=6606;   // fu
   sortTable[0x5310]=6607;   // fu
   sortTable[0x83A9]=6608;   // fu
   sortTable[0x8D59]=6609;   // fu
   sortTable[0x7829]=6610;   // fu
   sortTable[0x5452]=6611;   // fu
   sortTable[0x7F58]=6612;   // fu
   sortTable[0x8DD7]=6613;   // fu
   sortTable[0x876E]=6614;   // fu
   sortTable[0x9EFB]=6615;   // fu
   sortTable[0x544B]=6616;   // fu
   sortTable[0x7EC2]=6617;   // fu
   sortTable[0x90DB]=6618;   // fu
   sortTable[0x7C20]=6619;   // fu
   sortTable[0x832F]=6620;   // fu
   sortTable[0x5E5E]=6621;   // fu
   sortTable[0x6ECF]=6622;   // fu
   sortTable[0x6874]=6623;   // fu
   sortTable[0x602B]=6624;   // fu
   sortTable[0x8709]=6625;   // fu
   sortTable[0x7953]=6626;   // fu
   sortTable[0x9207]=6627;   // fu
   sortTable[0x9EB8]=6628;   // fu
   sortTable[0x7ECB]=6629;   // fu
   sortTable[0x982B]=6630;   // fu
   sortTable[0x8300]=6631;   // fu
   sortTable[0x97E8]=6632;   // fu
   sortTable[0x7D8D]=6633;   // fu
   sortTable[0x8274]=6634;   // fu
   sortTable[0x8659]=6635;   // fu
   sortTable[0x8946]=6636;   // fu
   sortTable[0x889D]=6637;   // fu
   sortTable[0x842F]=6638;   // fu
   sortTable[0x9B34]=6639;   // fu
   sortTable[0x8914]=6640;   // fu
   sortTable[0x7B99]=6641;   // fu
   sortTable[0x8342]=6642;   // fu
   sortTable[0x7A03]=6643;   // fu
   sortTable[0x82A3]=6644;   // fu
   sortTable[0x9CC6]=6645;   // fu
   sortTable[0x90D9]=6646;   // fu
   sortTable[0x8F39]=6647;   // fu
   sortTable[0x9D69]=6648;   // fu
   sortTable[0x7B30]=6649;   // fu
   sortTable[0x844D]=6650;   // fu
   sortTable[0x9D9D]=6651;   // fu
   sortTable[0x7D92]=6652;   // fu
   sortTable[0x8CE6]=6653;   // fu
   sortTable[0x91E1]=6654;   // fu
   sortTable[0x7F66]=6655;   // fu
   sortTable[0x9B84]=6656;   // fu
   sortTable[0x9351]=6657;   // fu
   sortTable[0x9CFA]=6658;   // fu
   sortTable[0x909E]=6659;   // fu
   sortTable[0x7806]=6660;   // fu
   sortTable[0x79A3]=6661;   // fu
   sortTable[0x7CB0]=6662;   // fu
   sortTable[0x7D28]=6663;   // fu
   sortTable[0x7B5F]=6664;   // fu
   sortTable[0x8A03]=6665;   // fu
   sortTable[0x97CD]=6666;   // fu
   sortTable[0x7D3C]=6667;   // fu
   sortTable[0x8567]=6668;   // fu
   sortTable[0x8705]=6669;   // fu
   sortTable[0x8984]=6670;   // fu
   sortTable[0x9362]=6671;   // fu
   sortTable[0x9AF4]=6672;   // fu
   sortTable[0x9EA9]=6673;   // fu
   sortTable[0x7D31]=6674;   // fu
   sortTable[0x7DEE]=6675;   // fu
   sortTable[0x8374]=6676;   // fu
   sortTable[0x86A5]=6677;   // fu
   sortTable[0x86B9]=6678;   // fu
   sortTable[0x886D]=6679;   // fu
   sortTable[0x8C67]=6680;   // fu
   sortTable[0x8CFB]=6681;   // fu
   sortTable[0x9D14]=6682;   // fu
   sortTable[0x4E40]=6683;   // fu
   sortTable[0x4E76]=6684;   // fu
   sortTable[0x4F15]=6685;   // fu
   sortTable[0x4FCC]=6686;   // fu
   sortTable[0x4FDB]=6687;   // fu
   sortTable[0x5069]=6688;   // fu
   sortTable[0x51A8]=6689;   // fu
   sortTable[0x51B9]=6690;   // fu
   sortTable[0x521C]=6691;   // fu
   sortTable[0x5488]=6692;   // fu
   sortTable[0x54F9]=6693;   // fu
   sortTable[0x5638]=6694;   // fu
   sortTable[0x5711]=6695;   // fu
   sortTable[0x577F]=6696;   // fu
   sortTable[0x5798]=6697;   // fu
   sortTable[0x57BA]=6698;   // fu
   sortTable[0x598B]=6699;   // fu
   sortTable[0x59C7]=6700;   // fu
   sortTable[0x5A10]=6701;   // fu
   sortTable[0x5A8D]=6702;   // fu
   sortTable[0x5B14]=6703;   // fu
   sortTable[0x5C03]=6704;   // fu
   sortTable[0x5CAA]=6705;   // fu
   sortTable[0x5CCA]=6706;   // fu
   sortTable[0x5DFF]=6707;   // fu
   sortTable[0x5F23]=6708;   // fu
   sortTable[0x5F7F]=6709;   // fu
   sortTable[0x5FA9]=6710;   // fu
   sortTable[0x6024]=6711;   // fu
   sortTable[0x61EF]=6712;   // fu
   sortTable[0x636C]=6713;   // fu
   sortTable[0x64A8]=6714;   // fu
   sortTable[0x65C9]=6715;   // fu
   sortTable[0x678E]=6716;   // fu
   sortTable[0x67CE]=6717;   // fu
   sortTable[0x67EB]=6718;   // fu
   sortTable[0x683F]=6719;   // fu
   sortTable[0x68F4]=6720;   // fu
   sortTable[0x6928]=6721;   // fu
   sortTable[0x6931]=6722;   // fu
   sortTable[0x6991]=6723;   // fu
   sortTable[0x6CED]=6724;   // fu
   sortTable[0x6D11]=6725;   // fu
   sortTable[0x6F93]=6726;   // fu
   sortTable[0x70A5]=6727;   // fu
   sortTable[0x70F0]=6728;   // fu
   sortTable[0x7124]=6729;   // fu
   sortTable[0x739E]=6730;   // fu
   sortTable[0x73B8]=6731;   // fu
   sortTable[0x7408]=6732;   // fu
   sortTable[0x7536]=6733;   // fu
   sortTable[0x7549]=6734;   // fu
   sortTable[0x7550]=6735;   // fu
   sortTable[0x75E1]=6736;   // fu
   sortTable[0x7641]=6737;   // fu
   sortTable[0x76D9]=6738;   // fu
   sortTable[0x79FF]=6739;   // fu
   sortTable[0x7A2A]=6740;   // fu
   sortTable[0x7ACE]=6741;   // fu
   sortTable[0x7CD0]=6742;   // fu
   sortTable[0x7D65]=6743;   // fu
   sortTable[0x7FC7]=6744;   // fu
   sortTable[0x80D5]=6745;   // fu
   sortTable[0x8240]=6746;   // fu
   sortTable[0x86D7]=6747;   // fu
   sortTable[0x875C]=6748;   // fu
   sortTable[0x8A42]=6749;   // fu
   sortTable[0x8AE8]=6750;   // fu
   sortTable[0x8E3E]=6751;   // fu
   sortTable[0x915C]=6752;   // fu
   sortTable[0x9258]=6753;   // fu
   sortTable[0x925C]=6754;   // fu
   sortTable[0x961D]=6755;   // fu
   sortTable[0x965A]=6756;   // fu
   sortTable[0x98AB]=6757;   // fu
   sortTable[0x99D9]=6758;   // fu
   sortTable[0x9B92]=6759;   // fu
   sortTable[0x9BB2]=6760;   // fu
   sortTable[0x9C12]=6761;   // fu
   sortTable[0x9CEC]=6762;   // fu
   sortTable[0x9EAC]=6763;   // fu
   sortTable[0x9EB1]=6764;   // fu
   sortTable[0x5C2C]=7168;   // ga
   sortTable[0x560E]=7169;   // ga
   sortTable[0x5676]=7170;   // ga
   sortTable[0x5477]=7171;   // ga
   sortTable[0x65EE]=7172;   // ga
   sortTable[0x9486]=7173;   // ga
   sortTable[0x5C15]=7174;   // ga
   sortTable[0x5C1C]=7175;   // ga
   sortTable[0x9B40]=7176;   // ga
   sortTable[0x5620]=7177;   // ga
   sortTable[0x738D]=7178;   // ga
   sortTable[0x9337]=7179;   // ga
   sortTable[0x6539]=7180;   // gai
   sortTable[0x8BE5]=7181;   // gai
   sortTable[0x8A72]=7182;   // gai
   sortTable[0x6982]=7183;   // gai
   sortTable[0x76D6]=7184;   // gai
   sortTable[0x84CB]=7185;   // gai
   sortTable[0x6E89]=7186;   // gai
   sortTable[0x9499]=7187;   // gai
   sortTable[0x9223]=7188;   // gai
   sortTable[0x4E10]=7189;   // gai
   sortTable[0x5793]=7190;   // gai
   sortTable[0x9654]=7191;   // gai
   sortTable[0x8D45]=7192;   // gai
   sortTable[0x8344]=7193;   // gai
   sortTable[0x8462]=7194;   // gai
   sortTable[0x7974]=7195;   // gai
   sortTable[0x6224]=7196;   // gai
   sortTable[0x9691]=7197;   // gai
   sortTable[0x7D6F]=7198;   // gai
   sortTable[0x8C65]=7199;   // gai
   sortTable[0x8CC5]=7200;   // gai
   sortTable[0x4E62]=7201;   // gai
   sortTable[0x4F85]=7202;   // gai
   sortTable[0x5303]=7203;   // gai
   sortTable[0x5304]=7204;   // gai
   sortTable[0x59DF]=7205;   // gai
   sortTable[0x5CD0]=7206;   // gai
   sortTable[0x5FCB]=7207;   // gai
   sortTable[0x6461]=7208;   // gai
   sortTable[0x6650]=7209;   // gai
   sortTable[0x675A]=7210;   // gai
   sortTable[0x69E9]=7211;   // gai
   sortTable[0x69EA]=7212;   // gai
   sortTable[0x6F11]=7213;   // gai
   sortTable[0x74C2]=7214;   // gai
   sortTable[0x7561]=7215;   // gai
   sortTable[0x7D60]=7216;   // gai
   sortTable[0x8CCC]=7217;   // gai
   sortTable[0x90C2]=7218;   // gai
   sortTable[0x9623]=7219;   // gai
   sortTable[0x5E72]=7220;   // gan
   sortTable[0x611F]=7221;   // gan
   sortTable[0x8D76]=7222;   // gan
   sortTable[0x8D95]=7223;   // gan
   sortTable[0x6562]=7224;   // gan
   sortTable[0x6746]=7225;   // gan
   sortTable[0x809D]=7226;   // gan
   sortTable[0x7518]=7227;   // gan
   sortTable[0x7AFF]=7228;   // gan
   sortTable[0x79C6]=7229;   // gan
   sortTable[0x7A08]=7230;   // gan
   sortTable[0x5C34]=7231;   // gan
   sortTable[0x8D63]=7232;   // gan
   sortTable[0x6A44]=7233;   // gan
   sortTable[0x6DE6]=7234;   // gan
   sortTable[0x67D1]=7235;   // gan
   sortTable[0x82F7]=7236;   // gan
   sortTable[0x65F0]=7237;   // gan
   sortTable[0x7EC0]=7238;   // gan
   sortTable[0x77F8]=7239;   // gan
   sortTable[0x64C0]=7240;   // gan
   sortTable[0x6CD4]=7241;   // gan
   sortTable[0x75B3]=7242;   // gan
   sortTable[0x6F89]=7243;   // gan
   sortTable[0x9150]=7244;   // gan
   sortTable[0x7B78]=7245;   // gan
   sortTable[0x5769]=7246;   // gan
   sortTable[0x9CE1]=7247;   // gan
   sortTable[0x7B34]=7248;   // gan
   sortTable[0x7C33]=7249;   // gan
   sortTable[0x9AAD]=7250;   // gan
   sortTable[0x9C64]=7251;   // gan
   sortTable[0x7C93]=7252;   // gan
   sortTable[0x8A4C]=7253;   // gan
   sortTable[0x7D3A]=7254;   // gan
   sortTable[0x9CF1]=7255;   // gan
   sortTable[0x4E79]=7256;   // gan
   sortTable[0x4E81]=7257;   // gan
   sortTable[0x4EE0]=7258;   // gan
   sortTable[0x501D]=7259;   // gan
   sortTable[0x51CE]=7260;   // gan
   sortTable[0x51F2]=7261;   // gan
   sortTable[0x5C32]=7262;   // gan
   sortTable[0x5C36]=7263;   // gan
   sortTable[0x5C37]=7264;   // gan
   sortTable[0x5E79]=7265;   // gan
   sortTable[0x5FD3]=7266;   // gan
   sortTable[0x625E]=7267;   // gan
   sortTable[0x653C]=7268;   // gan
   sortTable[0x687F]=7269;   // gan
   sortTable[0x69A6]=7270;   // gan
   sortTable[0x6A8A]=7271;   // gan
   sortTable[0x6C75]=7272;   // gan
   sortTable[0x6F27]=7273;   // gan
   sortTable[0x7068]=7274;   // gan
   sortTable[0x7395]=7275;   // gan
   sortTable[0x76AF]=7276;   // gan
   sortTable[0x76F0]=7277;   // gan
   sortTable[0x8289]=7278;   // gan
   sortTable[0x8866]=7279;   // gan
   sortTable[0x8D11]=7280;   // gan
   sortTable[0x8D1B]=7281;   // gan
   sortTable[0x8FC0]=7282;   // gan
   sortTable[0x9B50]=7283;   // gan
   sortTable[0x9C14]=7284;   // gan
   sortTable[0x521A]=7285;   // gang
   sortTable[0x525B]=7286;   // gang
   sortTable[0x94A2]=7287;   // gang
   sortTable[0x92FC]=7288;   // gang
   sortTable[0x7EB2]=7289;   // gang
   sortTable[0x7DB1]=7290;   // gang
   sortTable[0x5C97]=7291;   // gang
   sortTable[0x5D17]=7292;   // gang
   sortTable[0x6E2F]=7293;   // gang
   sortTable[0x7F38]=7294;   // gang
   sortTable[0x5188]=7295;   // gang
   sortTable[0x5CA1]=7296;   // gang
   sortTable[0x6760]=7297;   // gang
   sortTable[0x7F61]=7298;   // gang
   sortTable[0x809B]=7299;   // gang
   sortTable[0x6206]=7300;   // gang
   sortTable[0x91ED]=7301;   // gang
   sortTable[0x77FC]=7302;   // gang
   sortTable[0x7B7B]=7303;   // gang
   sortTable[0x7F41]=7304;   // gang
   sortTable[0x51AE]=7305;   // gang
   sortTable[0x5808]=7306;   // gang
   sortTable[0x583D]=7307;   // gang
   sortTable[0x6205]=7308;   // gang
   sortTable[0x6386]=7309;   // gang
   sortTable[0x68E1]=7310;   // gang
   sortTable[0x69D3]=7311;   // gang
   sortTable[0x7135]=7312;   // gang
   sortTable[0x7268]=7313;   // gang
   sortTable[0x7285]=7314;   // gang
   sortTable[0x7598]=7315;   // gang
   sortTable[0x7F53]=7316;   // gang
   sortTable[0x93A0]=7317;   // gang
   sortTable[0x9AD8]=7318;   // gao
   sortTable[0x544A]=7319;   // gao
   sortTable[0x641E]=7320;   // gao
   sortTable[0x7A3F]=7321;   // gao
   sortTable[0x818F]=7322;   // gao
   sortTable[0x7CD5]=7323;   // gao
   sortTable[0x7F94]=7324;   // gao
   sortTable[0x768B]=7325;   // gao
   sortTable[0x8BF0]=7326;   // gao
   sortTable[0x6772]=7327;   // gao
   sortTable[0x777E]=7328;   // gao
   sortTable[0x9550]=7329;   // gao
   sortTable[0x69C1]=7330;   // gao
   sortTable[0x7F1F]=7331;   // gao
   sortTable[0x7BD9]=7332;   // gao
   sortTable[0x85C1]=7333;   // gao
   sortTable[0x69D4]=7334;   // gao
   sortTable[0x90DC]=7335;   // gao
   sortTable[0x9506]=7336;   // gao
   sortTable[0x9F1B]=7337;   // gao
   sortTable[0x7A3E]=7338;   // gao
   sortTable[0x85F3]=7339;   // gao
   sortTable[0x9DF1]=7340;   // gao
   sortTable[0x83D2]=7341;   // gao
   sortTable[0x796E]=7342;   // gao
   sortTable[0x7B76]=7343;   // gao
   sortTable[0x8AA5]=7344;   // gao
   sortTable[0x7970]=7345;   // gao
   sortTable[0x81EF]=7346;   // gao
   sortTable[0x92EF]=7347;   // gao
   sortTable[0x993B]=7348;   // gao
   sortTable[0x799E]=7349;   // gao
   sortTable[0x7E1E]=7350;   // gao
   sortTable[0x7F99]=7351;   // gao
   sortTable[0x97DF]=7352;   // gao
   sortTable[0x52C2]=7353;   // gao
   sortTable[0x53DD]=7354;   // gao
   sortTable[0x543F]=7355;   // gao
   sortTable[0x5930]=7356;   // gao
   sortTable[0x66A0]=7357;   // gao
   sortTable[0x69C0]=7358;   // gao
   sortTable[0x69F9]=7359;   // gao
   sortTable[0x6A70]=7360;   // gao
   sortTable[0x6ABA]=7361;   // gao
   sortTable[0x6ADC]=7362;   // gao
   sortTable[0x6EDC]=7363;   // gao
   sortTable[0x7170]=7364;   // gao
   sortTable[0x7690]=7365;   // gao
   sortTable[0x9AD9]=7366;   // gao
   sortTable[0x9DCE]=7367;   // gao
   sortTable[0x4E2A]=7368;   // ge
   sortTable[0x500B]=7369;   // ge
   sortTable[0x9769]=7370;   // ge
   sortTable[0x5404]=7371;   // ge
   sortTable[0x54E5]=7372;   // ge
   sortTable[0x6B4C]=7373;   // ge
   sortTable[0x683C]=7374;   // ge
   sortTable[0x9694]=7375;   // ge
   sortTable[0x5272]=7376;   // ge
   sortTable[0x80F3]=7377;   // ge
   sortTable[0x6401]=7378;   // ge
   sortTable[0x64F1]=7379;   // ge
   sortTable[0x9ABC]=7380;   // ge
   sortTable[0x7599]=7381;   // ge
   sortTable[0x9E3D]=7382;   // ge
   sortTable[0x9D3F]=7383;   // ge
   sortTable[0x9601]=7384;   // ge
   sortTable[0x845B]=7385;   // ge
   sortTable[0x6208]=7386;   // ge
   sortTable[0x54AF]=7387;   // ge
   sortTable[0x7EA5]=7388;   // ge
   sortTable[0x643F]=7389;   // ge
   sortTable[0x55DD]=7390;   // ge
   sortTable[0x8238]=7391;   // ge
   sortTable[0x9549]=7392;   // ge
   sortTable[0x95A4]=7393;   // ge
   sortTable[0x867C]=7394;   // ge
   sortTable[0x88BC]=7395;   // ge
   sortTable[0x572A]=7396;   // ge
   sortTable[0x94EC]=7397;   // ge
   sortTable[0x54FF]=7398;   // ge
   sortTable[0x784C]=7399;   // ge
   sortTable[0x9B32]=7400;   // ge
   sortTable[0x8188]=7401;   // ge
   sortTable[0x9A14]=7402;   // ge
   sortTable[0x4EE1]=7403;   // ge
   sortTable[0x97D0]=7404;   // ge
   sortTable[0x8316]=7405;   // ge
   sortTable[0x8090]=7406;   // ge
   sortTable[0x8F35]=7407;   // ge
   sortTable[0x8F55]=7408;   // ge
   sortTable[0x89E1]=7409;   // ge
   sortTable[0x86D2]=7410;   // ge
   sortTable[0x95A3]=7411;   // ge
   sortTable[0x9D10]=7412;   // ge
   sortTable[0x8B0C]=7413;   // ge
   sortTable[0x5865]=7414;   // ge
   sortTable[0x93B6]=7415;   // ge
   sortTable[0x7B87]=7416;   // ge
   sortTable[0x88D3]=7417;   // ge
   sortTable[0x9788]=7418;   // ge
   sortTable[0x4F6E]=7419;   // ge
   sortTable[0x530C]=7420;   // ge
   sortTable[0x5444]=7421;   // ge
   sortTable[0x55F0]=7422;   // ge
   sortTable[0x5F41]=7423;   // ge
   sortTable[0x6105]=7424;   // ge
   sortTable[0x6213]=7425;   // ge
   sortTable[0x6228]=7426;   // ge
   sortTable[0x630C]=7427;   // ge
   sortTable[0x654B]=7428;   // ge
   sortTable[0x69C5]=7429;   // ge
   sortTable[0x6ACA]=7430;   // ge
   sortTable[0x6EC6]=7431;   // ge
   sortTable[0x6ED2]=7432;   // ge
   sortTable[0x726B]=7433;   // ge
   sortTable[0x7271]=7434;   // ge
   sortTable[0x72B5]=7435;   // ge
   sortTable[0x7366]=7436;   // ge
   sortTable[0x81F5]=7437;   // ge
   sortTable[0x8AFD]=7438;   // ge
   sortTable[0x97B7]=7439;   // ge
   sortTable[0x97DA]=7440;   // ge
   sortTable[0x9BAF]=7441;   // ge
   sortTable[0x9D1A]=7442;   // ge
   sortTable[0x7ED9]=7443;   // gei
   sortTable[0x7D66]=7444;   // gei
   sortTable[0x8DDF]=7445;   // gen
   sortTable[0x6839]=7446;   // gen
   sortTable[0x54CF]=7447;   // gen
   sortTable[0x4E98]=7448;   // gen
   sortTable[0x826E]=7449;   // gen
   sortTable[0x831B]=7450;   // gen
   sortTable[0x4E99]=7451;   // gen
   sortTable[0x63EF]=7452;   // gen
   sortTable[0x66F4]=7453;   // geng
   sortTable[0x8015]=7454;   // geng
   sortTable[0x6897]=7455;   // geng
   sortTable[0x54FD]=7456;   // geng
   sortTable[0x5E9A]=7457;   // geng
   sortTable[0x803F]=7458;   // geng
   sortTable[0x57C2]=7459;   // geng
   sortTable[0x7FB9]=7460;   // geng
   sortTable[0x8D53]=7461;   // geng
   sortTable[0x9CA0]=7462;   // geng
   sortTable[0x7EE0]=7463;   // geng
   sortTable[0x9BC1]=7464;   // geng
   sortTable[0x7D86]=7465;   // geng
   sortTable[0x7DEA]=7466;   // geng
   sortTable[0x7FAE]=7467;   // geng
   sortTable[0x7E06]=7468;   // geng
   sortTable[0x9D8A]=7469;   // geng
   sortTable[0x90E0]=7470;   // geng
   sortTable[0x9E52]=7471;   // geng
   sortTable[0x9ABE]=7472;   // geng
   sortTable[0x7D5A]=7473;   // geng
   sortTable[0x8CE1]=7474;   // geng
   sortTable[0x522F]=7475;   // geng
   sortTable[0x5829]=7476;   // geng
   sortTable[0x5CFA]=7477;   // geng
   sortTable[0x632D]=7478;   // geng
   sortTable[0x63B6]=7479;   // geng
   sortTable[0x6404]=7480;   // geng
   sortTable[0x6685]=7481;   // geng
   sortTable[0x6929]=7482;   // geng
   sortTable[0x6D6D]=7483;   // geng
   sortTable[0x713F]=7484;   // geng
   sortTable[0x754A]=7485;   // geng
   sortTable[0x8384]=7486;   // geng
   sortTable[0x83EE]=7487;   // geng
   sortTable[0x5DE5]=7488;   // gong
   sortTable[0x5171]=7489;   // gong
   sortTable[0x516C]=7490;   // gong
   sortTable[0x529F]=7491;   // gong
   sortTable[0x653B]=7492;   // gong
   sortTable[0x4F9B]=7493;   // gong
   sortTable[0x5DE9]=7494;   // gong
   sortTable[0x978F]=7495;   // gong
   sortTable[0x8D21]=7496;   // gong
   sortTable[0x8CA2]=7497;   // gong
   sortTable[0x62F1]=7498;   // gong
   sortTable[0x5BAB]=7499;   // gong
   sortTable[0x5BAE]=7500;   // gong
   sortTable[0x5F13]=7501;   // gong
   sortTable[0x8EAC]=7502;   // gong
   sortTable[0x6C5E]=7503;   // gong
   sortTable[0x86A3]=7504;   // gong
   sortTable[0x606D]=7505;   // gong
   sortTable[0x9F9A]=7506;   // gong
   sortTable[0x89E5]=7507;   // gong
   sortTable[0x80B1]=7508;   // gong
   sortTable[0x73D9]=7509;   // gong
   sortTable[0x9F8F]=7510;   // gong
   sortTable[0x7FBE]=7511;   // gong
   sortTable[0x83BB]=7512;   // gong
   sortTable[0x9AF8]=7513;   // gong
   sortTable[0x9F94]=7514;   // gong
   sortTable[0x8F01]=7515;   // gong
   sortTable[0x78BD]=7516;   // gong
   sortTable[0x5311]=7517;   // gong
   sortTable[0x53B7]=7518;   // gong
   sortTable[0x551D]=7519;   // gong
   sortTable[0x5868]=7520;   // gong
   sortTable[0x5E4A]=7521;   // gong
   sortTable[0x5EFE]=7522;   // gong
   sortTable[0x6129]=7523;   // gong
   sortTable[0x6150]=7524;   // gong
   sortTable[0x62F2]=7525;   // gong
   sortTable[0x675B]=7526;   // gong
   sortTable[0x6831]=7527;   // gong
   sortTable[0x7195]=7528;   // gong
   sortTable[0x739C]=7529;   // gong
   sortTable[0x7CFC]=7530;   // gong
   sortTable[0x89F5]=7531;   // gong
   sortTable[0x8EB3]=7532;   // gong
   sortTable[0x92DB]=7533;   // gong
   sortTable[0x591F]=7534;   // gou
   sortTable[0x5920]=7535;   // gou
   sortTable[0x6784]=7536;   // gou
   sortTable[0x69CB]=7537;   // gou
   sortTable[0x72D7]=7538;   // gou
   sortTable[0x6C9F]=7539;   // gou
   sortTable[0x6E9D]=7540;   // gou
   sortTable[0x52FE]=7541;   // gou
   sortTable[0x94A9]=7542;   // gou
   sortTable[0x9264]=7543;   // gou
   sortTable[0x8D2D]=7544;   // gou
   sortTable[0x8CFC]=7545;   // gou
   sortTable[0x82DF]=7546;   // gou
   sortTable[0x57A2]=7547;   // gou
   sortTable[0x8BDF]=7548;   // gou
   sortTable[0x5F40]=7549;   // gou
   sortTable[0x7B31]=7550;   // gou
   sortTable[0x5ABE]=7551;   // gou
   sortTable[0x9058]=7552;   // gou
   sortTable[0x97B2]=7553;   // gou
   sortTable[0x7F11]=7554;   // gou
   sortTable[0x7BDD]=7555;   // gou
   sortTable[0x67B8]=7556;   // gou
   sortTable[0x5CA3]=7557;   // gou
   sortTable[0x4F5D]=7558;   // gou
   sortTable[0x89CF]=7559;   // gou
   sortTable[0x96CA]=7560;   // gou
   sortTable[0x8007]=7561;   // gou
   sortTable[0x8920]=7562;   // gou
   sortTable[0x97DD]=7563;   // gou
   sortTable[0x88A7]=7564;   // gou
   sortTable[0x86BC]=7565;   // gou
   sortTable[0x8A3D]=7566;   // gou
   sortTable[0x8A6C]=7567;   // gou
   sortTable[0x8008]=7568;   // gou
   sortTable[0x7DF1]=7569;   // gou
   sortTable[0x8329]=7570;   // gou
   sortTable[0x89AF]=7571;   // gou
   sortTable[0x920E]=7572;   // gou
   sortTable[0x5193]=7573;   // gou
   sortTable[0x5778]=7574;   // gou
   sortTable[0x59E4]=7575;   // gou
   sortTable[0x6406]=7576;   // gou
   sortTable[0x6480]=7577;   // gou
   sortTable[0x7179]=7578;   // gou
   sortTable[0x73BD]=7579;   // gou
   sortTable[0x8009]=7580;   // gou
   sortTable[0x82B6]=7581;   // gou
   sortTable[0x8C7F]=7582;   // gou
   sortTable[0x53E4]=7583;   // gu
   sortTable[0x59D1]=7584;   // gu
   sortTable[0x6545]=7585;   // gu
   sortTable[0x56FA]=7586;   // gu
   sortTable[0x987E]=7587;   // gu
   sortTable[0x9867]=7588;   // gu
   sortTable[0x9F13]=7589;   // gu
   sortTable[0x80A1]=7590;   // gu
   sortTable[0x9AA8]=7591;   // gu
   sortTable[0x8C37]=7592;   // gu
   sortTable[0x4F30]=7593;   // gu
   sortTable[0x5B64]=7594;   // gu
   sortTable[0x96C7]=7595;   // gu
   sortTable[0x50F1]=7596;   // gu
   sortTable[0x5495]=7597;   // gu
   sortTable[0x94B4]=7598;   // gu
   sortTable[0x9237]=7599;   // gu
   sortTable[0x83C7]=7600;   // gu
   sortTable[0x8F9C]=7601;   // gu
   sortTable[0x6CBD]=7602;   // gu
   sortTable[0x9E58]=7603;   // gu
   sortTable[0x9E44]=7604;   // gu
   sortTable[0x86CA]=7605;   // gu
   sortTable[0x5471]=7606;   // gu
   sortTable[0x9522]=7607;   // gu
   sortTable[0x7B8D]=7608;   // gu
   sortTable[0x6BC2]=7609;   // gu
   sortTable[0x9E2A]=7610;   // gu
   sortTable[0x7F5F]=7611;   // gu
   sortTable[0x86C4]=7612;   // gu
   sortTable[0x7A40]=7613;   // gu
   sortTable[0x77BD]=7614;   // gu
   sortTable[0x8BC2]=7615;   // gu
   sortTable[0x560F]=7616;   // gu
   sortTable[0x6C69]=7617;   // gu
   sortTable[0x83F0]=7618;   // gu
   sortTable[0x688F]=7619;   // gu
   sortTable[0x5D2E]=7620;   // gu
   sortTable[0x9164]=7621;   // gu
   sortTable[0x89DA]=7622;   // gu
   sortTable[0x75FC]=7623;   // gu
   sortTable[0x8F71]=7624;   // gu
   sortTable[0x726F]=7625;   // gu
   sortTable[0x76EC]=7626;   // gu
   sortTable[0x7F96]=7627;   // gu
   sortTable[0x9989]=7628;   // gu
   sortTable[0x9CB4]=7629;   // gu
   sortTable[0x81CC]=7630;   // gu
   sortTable[0x7F5B]=7631;   // gu
   sortTable[0x9D23]=7632;   // gu
   sortTable[0x727F]=7633;   // gu
   sortTable[0x7A12]=7634;   // gu
   sortTable[0x82FD]=7635;   // gu
   sortTable[0x8F42]=7636;   // gu
   sortTable[0x9BDD]=7637;   // gu
   sortTable[0x7B1F]=7638;   // gu
   sortTable[0x797B]=7639;   // gu
   sortTable[0x7B9B]=7640;   // gu
   sortTable[0x84C7]=7641;   // gu
   sortTable[0x8135]=7642;   // gu
   sortTable[0x9232]=7643;   // gu
   sortTable[0x7CD3]=7644;   // gu
   sortTable[0x50A6]=7645;   // gu
   sortTable[0x51C5]=7646;   // gu
   sortTable[0x5502]=7647;   // gu
   sortTable[0x5503]=7648;   // gu
   sortTable[0x5552]=7649;   // gu
   sortTable[0x580C]=7650;   // gu
   sortTable[0x5903]=7651;   // gu
   sortTable[0x5AF4]=7652;   // gu
   sortTable[0x5C33]=7653;   // gu
   sortTable[0x5CE0]=7654;   // gu
   sortTable[0x5D13]=7655;   // gu
   sortTable[0x6132]=7656;   // gu
   sortTable[0x6262]=7657;   // gu
   sortTable[0x67E7]=7658;   // gu
   sortTable[0x68DD]=7659;   // gu
   sortTable[0x6996]=7660;   // gu
   sortTable[0x69BE]=7661;   // gu
   sortTable[0x6A6D]=7662;   // gu
   sortTable[0x6CD2]=7663;   // gu
   sortTable[0x6DC8]=7664;   // gu
   sortTable[0x6FF2]=7665;   // gu
   sortTable[0x7014]=7666;   // gu
   sortTable[0x76B7]=7667;   // gu
   sortTable[0x76BC]=7668;   // gu
   sortTable[0x7BD0]=7669;   // gu
   sortTable[0x7E0E]=7670;   // gu
   sortTable[0x85A3]=7671;   // gu
   sortTable[0x86CC]=7672;   // gu
   sortTable[0x8831]=7673;   // gu
   sortTable[0x8A41]=7674;   // gu
   sortTable[0x8EF1]=7675;   // gu
   sortTable[0x8EF2]=7676;   // gu
   sortTable[0x9027]=7677;   // gu
   sortTable[0x932E]=7678;   // gu
   sortTable[0x9936]=7679;   // gu
   sortTable[0x9B95]=7680;   // gu
   sortTable[0x9DBB]=7681;   // gu
   sortTable[0x9F14]=7682;   // gu
   sortTable[0x6302]=7683;   // gua
   sortTable[0x639B]=7684;   // gua
   sortTable[0x74DC]=7685;   // gua
   sortTable[0x522E]=7686;   // gua
   sortTable[0x5BE1]=7687;   // gua
   sortTable[0x8902]=7688;   // gua
   sortTable[0x5366]=7689;   // gua
   sortTable[0x8BD6]=7690;   // gua
   sortTable[0x80CD]=7691;   // gua
   sortTable[0x5250]=7692;   // gua
   sortTable[0x8052]=7693;   // gua
   sortTable[0x9E39]=7694;   // gua
   sortTable[0x681D]=7695;   // gua
   sortTable[0x7D53]=7696;   // gua
   sortTable[0x9A27]=7697;   // gua
   sortTable[0x7F63]=7698;   // gua
   sortTable[0x7DFA]=7699;   // gua
   sortTable[0x8A7F]=7700;   // gua
   sortTable[0x7F6B]=7701;   // gua
   sortTable[0x9D30]=7702;   // gua
   sortTable[0x92BD]=7703;   // gua
   sortTable[0x518E]=7704;   // gua
   sortTable[0x526E]=7705;   // gua
   sortTable[0x5280]=7706;   // gua
   sortTable[0x53E7]=7707;   // gua
   sortTable[0x5569]=7708;   // gua
   sortTable[0x576C]=7709;   // gua
   sortTable[0x6B44]=7710;   // gua
   sortTable[0x7171]=7711;   // gua
   sortTable[0x8D8F]=7712;   // gua
   sortTable[0x8E3B]=7713;   // gua
   sortTable[0x98AA]=7714;   // gua
   sortTable[0x98B3]=7715;   // gua
   sortTable[0x602A]=7716;   // guai
   sortTable[0x62D0]=7717;   // guai
   sortTable[0x4E56]=7718;   // guai
   sortTable[0x63B4]=7719;   // guai
   sortTable[0x7B89]=7720;   // guai
   sortTable[0x53CF]=7721;   // guai
   sortTable[0x592C]=7722;   // guai
   sortTable[0x6060]=7723;   // guai
   sortTable[0x6451]=7724;   // guai
   sortTable[0x67B4]=7725;   // guai
   sortTable[0x67FA]=7726;   // guai
   sortTable[0x5173]=7727;   // guan
   sortTable[0x95DC]=7728;   // guan
   sortTable[0x89C2]=7729;   // guan
   sortTable[0x89C0]=7730;   // guan
   sortTable[0x7BA1]=7731;   // guan
   sortTable[0x5B98]=7732;   // guan
   sortTable[0x9986]=7733;   // guan
   sortTable[0x9928]=7734;   // guan
   sortTable[0x60EF]=7735;   // guan
   sortTable[0x6163]=7736;   // guan
   sortTable[0x704C]=7737;   // guan
   sortTable[0x8D2F]=7738;   // guan
   sortTable[0x8CAB]=7739;   // guan
   sortTable[0x7F50]=7740;   // guan
   sortTable[0x68FA]=7741;   // guan
   sortTable[0x51A0]=7742;   // guan
   sortTable[0x6DAB]=7743;   // guan
   sortTable[0x500C]=7744;   // guan
   sortTable[0x76E5]=7745;   // guan
   sortTable[0x839E]=7746;   // guan
   sortTable[0x9E73]=7747;   // guan
   sortTable[0x9CCF]=7748;   // guan
   sortTable[0x63BC]=7749;   // guan
   sortTable[0x7B66]=7750;   // guan
   sortTable[0x797C]=7751;   // guan
   sortTable[0x96DA]=7752;   // guan
   sortTable[0x7936]=7753;   // guan
   sortTable[0x9327]=7754;   // guan
   sortTable[0x77D4]=7755;   // guan
   sortTable[0x9CE4]=7756;   // guan
   sortTable[0x93C6]=7757;   // guan
   sortTable[0x7F46]=7758;   // guan
   sortTable[0x8F28]=7759;   // guan
   sortTable[0x9475]=7760;   // guan
   sortTable[0x7AA4]=7761;   // guan
   sortTable[0x8484]=7762;   // guan
   sortTable[0x95A2]=7763;   // guan
   sortTable[0x9C25]=7764;   // guan
   sortTable[0x4E31]=7765;   // guan
   sortTable[0x60B9]=7766;   // guan
   sortTable[0x60BA]=7767;   // guan
   sortTable[0x645C]=7768;   // guan
   sortTable[0x6A0C]=7769;   // guan
   sortTable[0x6BCC]=7770;   // guan
   sortTable[0x6CF4]=7771;   // guan
   sortTable[0x6F45]=7772;   // guan
   sortTable[0x721F]=7773;   // guan
   sortTable[0x742F]=7774;   // guan
   sortTable[0x74D8]=7775;   // guan
   sortTable[0x75EF]=7776;   // guan
   sortTable[0x761D]=7777;   // guan
   sortTable[0x764F]=7778;   // guan
   sortTable[0x8218]=7779;   // guan
   sortTable[0x898C]=7780;   // guan
   sortTable[0x89B3]=7781;   // guan
   sortTable[0x9066]=7782;   // guan
   sortTable[0x95D7]=7783;   // guan
   sortTable[0x9C5E]=7784;   // guan
   sortTable[0x9C79]=7785;   // guan
   sortTable[0x9E1B]=7786;   // guan
   sortTable[0x5149]=7787;   // guang
   sortTable[0x5E7F]=7788;   // guang
   sortTable[0x5EE3]=7789;   // guang
   sortTable[0x901B]=7790;   // guang
   sortTable[0x80F1]=7791;   // guang
   sortTable[0x72B7]=7792;   // guang
   sortTable[0x54A3]=7793;   // guang
   sortTable[0x6844]=7794;   // guang
   sortTable[0x832A]=7795;   // guang
   sortTable[0x92A7]=7796;   // guang
   sortTable[0x8F04]=7797;   // guang
   sortTable[0x81E9]=7798;   // guang
   sortTable[0x4F8A]=7799;   // guang
   sortTable[0x4FC7]=7800;   // guang
   sortTable[0x50D9]=7801;   // guang
   sortTable[0x5799]=7802;   // guang
   sortTable[0x59EF]=7803;   // guang
   sortTable[0x5E83]=7804;   // guang
   sortTable[0x6497]=7805;   // guang
   sortTable[0x6B1F]=7806;   // guang
   sortTable[0x6D38]=7807;   // guang
   sortTable[0x706E]=7808;   // guang
   sortTable[0x7097]=7809;   // guang
   sortTable[0x709A]=7810;   // guang
   sortTable[0x709B]=7811;   // guang
   sortTable[0x70E1]=7812;   // guang
   sortTable[0x7377]=7813;   // guang
   sortTable[0x73D6]=7814;   // guang
   sortTable[0x81E6]=7815;   // guang
   sortTable[0x9EC6]=7816;   // guang
   sortTable[0x89C4]=7817;   // gui
   sortTable[0x898F]=7818;   // gui
   sortTable[0x9B3C]=7819;   // gui
   sortTable[0x8D35]=7820;   // gui
   sortTable[0x8CB4]=7821;   // gui
   sortTable[0x67DC]=7822;   // gui
   sortTable[0x5F52]=7823;   // gui
   sortTable[0x6B78]=7824;   // gui
   sortTable[0x8F68]=7825;   // gui
   sortTable[0x8ECC]=7826;   // gui
   sortTable[0x8DEA]=7827;   // gui
   sortTable[0x95FA]=7828;   // gui
   sortTable[0x95A8]=7829;   // gui
   sortTable[0x8BE1]=7830;   // gui
   sortTable[0x8A6D]=7831;   // gui
   sortTable[0x7845]=7832;   // gui
   sortTable[0x9F9F]=7833;   // gui
   sortTable[0x9F9C]=7834;   // gui
   sortTable[0x7678]=7835;   // gui
   sortTable[0x6842]=7836;   // gui
   sortTable[0x7470]=7837;   // gui
   sortTable[0x572D]=7838;   // gui
   sortTable[0x6867]=7839;   // gui
   sortTable[0x59AB]=7840;   // gui
   sortTable[0x6677]=7841;   // gui
   sortTable[0x7688]=7842;   // gui
   sortTable[0x5080]=7843;   // gui
   sortTable[0x5B84]=7844;   // gui
   sortTable[0x523D]=7845;   // gui
   sortTable[0x5326]=7846;   // gui
   sortTable[0x5E8B]=7847;   // gui
   sortTable[0x7094]=7848;   // gui
   sortTable[0x7C0B]=7849;   // gui
   sortTable[0x9C91]=7850;   // gui
   sortTable[0x90BD]=7851;   // gui
   sortTable[0x523F]=7852;   // gui
   sortTable[0x9CDC]=7853;   // gui
   sortTable[0x9C56]=7854;   // gui
   sortTable[0x79AC]=7855;   // gui
   sortTable[0x88BF]=7856;   // gui
   sortTable[0x9A29]=7857;   // gui
   sortTable[0x879D]=7858;   // gui
   sortTable[0x77B6]=7859;   // gui
   sortTable[0x796A]=7860;   // gui
   sortTable[0x81AD]=7861;   // gui
   sortTable[0x8958]=7862;   // gui
   sortTable[0x77A1]=7863;   // gui
   sortTable[0x7A90]=7864;   // gui
   sortTable[0x9C65]=7865;   // gui
   sortTable[0x7C02]=7866;   // gui
   sortTable[0x7B40]=7867;   // gui
   sortTable[0x86EB]=7868;   // gui
   sortTable[0x9652]=7869;   // gui
   sortTable[0x8325]=7870;   // gui
   sortTable[0x9B36]=7871;   // gui
   sortTable[0x9BAD]=7872;   // gui
   sortTable[0x84D5]=7873;   // gui
   sortTable[0x89E4]=7874;   // gui
   sortTable[0x4E80]=7875;   // gui
   sortTable[0x4F79]=7876;   // gui
   sortTable[0x528A]=7877;   // gui
   sortTable[0x528C]=7878;   // gui
   sortTable[0x532D]=7879;   // gui
   sortTable[0x5331]=7880;   // gui
   sortTable[0x53AC]=7881;   // gui
   sortTable[0x579D]=7882;   // gui
   sortTable[0x59FD]=7883;   // gui
   sortTable[0x5AAF]=7884;   // gui
   sortTable[0x5AE2]=7885;   // gui
   sortTable[0x5B00]=7886;   // gui
   sortTable[0x5DA1]=7887;   // gui
   sortTable[0x5DC2]=7888;   // gui
   sortTable[0x5E30]=7889;   // gui
   sortTable[0x5EAA]=7890;   // gui
   sortTable[0x5EC6]=7891;   // gui
   sortTable[0x6051]=7892;   // gui
   sortTable[0x646B]=7893;   // gui
   sortTable[0x648C]=7894;   // gui
   sortTable[0x6530]=7895;   // gui
   sortTable[0x6531]=7896;   // gui
   sortTable[0x660B]=7897;   // gui
   sortTable[0x6739]=7898;   // gui
   sortTable[0x691D]=7899;   // gui
   sortTable[0x6922]=7900;   // gui
   sortTable[0x69F6]=7901;   // gui
   sortTable[0x69FB]=7902;   // gui
   sortTable[0x69FC]=7903;   // gui
   sortTable[0x6A9C]=7904;   // gui
   sortTable[0x6AC3]=7905;   // gui
   sortTable[0x6AF7]=7906;   // gui
   sortTable[0x6C3F]=7907;   // gui
   sortTable[0x6E40]=7908;   // gui
   sortTable[0x7324]=7909;   // gui
   sortTable[0x73EA]=7910;   // gui
   sortTable[0x749D]=7911;   // gui
   sortTable[0x74CC]=7912;   // gui
   sortTable[0x7650]=7913;   // gui
   sortTable[0x80FF]=7914;   // gui
   sortTable[0x87E1]=7915;   // gui
   sortTable[0x90CC]=7916;   // gui
   sortTable[0x97BC]=7917;   // gui
   sortTable[0x9B39]=7918;   // gui
   sortTable[0x6EDA]=7919;   // gun
   sortTable[0x6EFE]=7920;   // gun
   sortTable[0x68CD]=7921;   // gun
   sortTable[0x886E]=7922;   // gun
   sortTable[0x7EF2]=7923;   // gun
   sortTable[0x9CA7]=7924;   // gun
   sortTable[0x8F8A]=7925;   // gun
   sortTable[0x78D9]=7926;   // gun
   sortTable[0x889E]=7927;   // gun
   sortTable[0x7754]=7928;   // gun
   sortTable[0x8B34]=7929;   // gun
   sortTable[0x9B8C]=7930;   // gun
   sortTable[0x84D8]=7931;   // gun
   sortTable[0x9BC0]=7932;   // gun
   sortTable[0x8F25]=7933;   // gun
   sortTable[0x7DC4]=7934;   // gun
   sortTable[0x7774]=7935;   // gun
   sortTable[0x88AC]=7936;   // gun
   sortTable[0x4E28]=7937;   // gun
   sortTable[0x60C3]=7938;   // gun
   sortTable[0x74AD]=7939;   // gun
   sortTable[0x8509]=7940;   // gun
   sortTable[0x56FD]=7941;   // guo
   sortTable[0x570B]=7942;   // guo
   sortTable[0x8FC7]=7943;   // guo
   sortTable[0x904E]=7944;   // guo
   sortTable[0x679C]=7945;   // guo
   sortTable[0x9505]=7946;   // guo
   sortTable[0x934B]=7947;   // guo
   sortTable[0x88F9]=7948;   // guo
   sortTable[0x90ED]=7949;   // guo
   sortTable[0x7313]=7950;   // guo
   sortTable[0x8662]=7951;   // guo
   sortTable[0x5D1E]=7952;   // guo
   sortTable[0x6901]=7953;   // guo
   sortTable[0x9998]=7954;   // guo
   sortTable[0x5E3C]=7955;   // guo
   sortTable[0x5459]=7956;   // guo
   sortTable[0x8748]=7957;   // guo
   sortTable[0x57DA]=7958;   // guo
   sortTable[0x9983]=7959;   // guo
   sortTable[0x873E]=7960;   // guo
   sortTable[0x805D]=7961;   // guo
   sortTable[0x8158]=7962;   // guo
   sortTable[0x8F20]=7963;   // guo
   sortTable[0x7DB6]=7964;   // guo
   sortTable[0x8195]=7965;   // guo
   sortTable[0x852E]=7966;   // guo
   sortTable[0x7CBF]=7967;   // guo
   sortTable[0x83D3]=7968;   // guo
   sortTable[0x54BC]=7969;   // guo
   sortTable[0x556F]=7970;   // guo
   sortTable[0x5613]=7971;   // guo
   sortTable[0x56EF]=7972;   // guo
   sortTable[0x56F6]=7973;   // guo
   sortTable[0x56FB]=7974;   // guo
   sortTable[0x5700]=7975;   // guo
   sortTable[0x581D]=7976;   // guo
   sortTable[0x588E]=7977;   // guo
   sortTable[0x5E57]=7978;   // guo
   sortTable[0x5F49]=7979;   // guo
   sortTable[0x5F4D]=7980;   // guo
   sortTable[0x60C8]=7981;   // guo
   sortTable[0x6156]=7982;   // guo
   sortTable[0x69E8]=7983;   // guo
   sortTable[0x6DC9]=7984;   // guo
   sortTable[0x6F0D]=7985;   // guo
   sortTable[0x6FC4]=7986;   // guo
   sortTable[0x7611]=7987;   // guo
   sortTable[0x87C8]=7988;   // guo
   sortTable[0x921B]=7989;   // guo
   sortTable[0x9301]=7990;   // guo
   sortTable[0x9439]=7991;   // guo
   sortTable[0x991C]=7992;   // guo
   sortTable[0x54C8]=8448;   // ha
   sortTable[0x86E4]=8449;   // ha
   sortTable[0x94EA]=8450;   // ha
   sortTable[0x4E37]=8451;   // ha
   sortTable[0x5964]=8452;   // ha
   sortTable[0x8FD8]=8453;   // hai
   sortTable[0x9084]=8454;   // hai
   sortTable[0x5B69]=8455;   // hai
   sortTable[0x6D77]=8456;   // hai
   sortTable[0x5BB3]=8457;   // hai
   sortTable[0x54B3]=8458;   // hai
   sortTable[0x55E8]=8459;   // hai
   sortTable[0x55D0]=8460;   // hai
   sortTable[0x9AB8]=8461;   // hai
   sortTable[0x4EA5]=8462;   // hai
   sortTable[0x9A87]=8463;   // hai
   sortTable[0x91A2]=8464;   // hai
   sortTable[0x6C26]=8465;   // hai
   sortTable[0x80F2]=8466;   // hai
   sortTable[0x99ED]=8467;   // hai
   sortTable[0x9900]=8468;   // hai
   sortTable[0x917C]=8469;   // hai
   sortTable[0x9826]=8470;   // hai
   sortTable[0x548D]=8471;   // hai
   sortTable[0x56A1]=8472;   // hai
   sortTable[0x5870]=8473;   // hai
   sortTable[0x598E]=8474;   // hai
   sortTable[0x70F8]=8475;   // hai
   sortTable[0x995A]=8476;   // hai
   sortTable[0x558A]=8477;   // han
   sortTable[0x542B]=8478;   // han
   sortTable[0x6C49]=8479;   // han
   sortTable[0x6F22]=8480;   // han
   sortTable[0x5BD2]=8481;   // han
   sortTable[0x710A]=8482;   // han
   sortTable[0x65F1]=8483;   // han
   sortTable[0x6C57]=8484;   // han
   sortTable[0x61BE]=8485;   // han
   sortTable[0x64BC]=8486;   // han
   sortTable[0x634D]=8487;   // han
   sortTable[0x97E9]=8488;   // han
   sortTable[0x7FF0]=8489;   // han
   sortTable[0x7F55]=8490;   // han
   sortTable[0x51FD]=8491;   // han
   sortTable[0x6DB5]=8492;   // han
   sortTable[0x608D]=8493;   // han
   sortTable[0x9163]=8494;   // han
   sortTable[0x90AF]=8495;   // han
   sortTable[0x701A]=8496;   // han
   sortTable[0x9894]=8497;   // han
   sortTable[0x61A8]=8498;   // han
   sortTable[0x7113]=8499;   // han
   sortTable[0x9F3E]=8500;   // han
   sortTable[0x83E1]=8501;   // han
   sortTable[0x6657]=8502;   // han
   sortTable[0x86B6]=8503;   // han
   sortTable[0x961A]=8504;   // han
   sortTable[0x9097]=8505;   // han
   sortTable[0x9878]=8506;   // han
   sortTable[0x95EC]=8507;   // han
   sortTable[0x8C8B]=8508;   // han
   sortTable[0x92B2]=8509;   // han
   sortTable[0x7745]=8510;   // han
   sortTable[0x8B40]=8511;   // han
   sortTable[0x8C3D]=8512;   // han
   sortTable[0x99AF]=8513;   // han
   sortTable[0x9588]=8514;   // han
   sortTable[0x97D3]=8515;   // han
   sortTable[0x99FB]=8516;   // han
   sortTable[0x6496]=8517;   // han
   sortTable[0x80A3]=8518;   // han
   sortTable[0x850A]=8519;   // han
   sortTable[0x8677]=8520;   // han
   sortTable[0x872C]=8521;   // han
   sortTable[0x9837]=8522;   // han
   sortTable[0x9B7D]=8523;   // han
   sortTable[0x8C43]=8524;   // han
   sortTable[0x91EC]=8525;   // han
   sortTable[0x92CE]=8526;   // han
   sortTable[0x92E1]=8527;   // han
   sortTable[0x9844]=8528;   // han
   sortTable[0x4E06]=8529;   // han
   sortTable[0x4F44]=8530;   // han
   sortTable[0x50BC]=8531;   // han
   sortTable[0x516F]=8532;   // han
   sortTable[0x51FE]=8533;   // han
   sortTable[0x5388]=8534;   // han
   sortTable[0x5481]=8535;   // han
   sortTable[0x54FB]=8536;   // han
   sortTable[0x5505]=8537;   // han
   sortTable[0x5705]=8538;   // han
   sortTable[0x57BE]=8539;   // han
   sortTable[0x5A22]=8540;   // han
   sortTable[0x5AE8]=8541;   // han
   sortTable[0x5C7D]=8542;   // han
   sortTable[0x5CBE]=8543;   // han
   sortTable[0x5D21]=8544;   // han
   sortTable[0x5D45]=8545;   // han
   sortTable[0x6658]=8546;   // han
   sortTable[0x6665]=8547;   // han
   sortTable[0x66B5]=8548;   // han
   sortTable[0x6892]=8549;   // han
   sortTable[0x6B5B]=8550;   // han
   sortTable[0x6D5B]=8551;   // han
   sortTable[0x6D6B]=8552;   // han
   sortTable[0x6D86]=8553;   // han
   sortTable[0x6F8F]=8554;   // han
   sortTable[0x70B6]=8555;   // han
   sortTable[0x71AF]=8556;   // han
   sortTable[0x7233]=8557;   // han
   sortTable[0x7302]=8558;   // han
   sortTable[0x7400]=8559;   // han
   sortTable[0x751D]=8560;   // han
   sortTable[0x7694]=8561;   // han
   sortTable[0x7B68]=8562;   // han
   sortTable[0x839F]=8563;   // han
   sortTable[0x862B]=8564;   // han
   sortTable[0x86FF]=8565;   // han
   sortTable[0x872D]=8566;   // han
   sortTable[0x8792]=8567;   // han
   sortTable[0x96D7]=8568;   // han
   sortTable[0x9807]=8569;   // han
   sortTable[0x99A0]=8570;   // han
   sortTable[0x9B2B]=8571;   // han
   sortTable[0x9DBE]=8572;   // han
   sortTable[0x822A]=8573;   // hang
   sortTable[0x676D]=8574;   // hang
   sortTable[0x592F]=8575;   // hang
   sortTable[0x6C86]=8576;   // hang
   sortTable[0x73E9]=8577;   // hang
   sortTable[0x9883]=8578;   // hang
   sortTable[0x8FD2]=8579;   // hang
   sortTable[0x86A2]=8580;   // hang
   sortTable[0x7B10]=8581;   // hang
   sortTable[0x7ED7]=8582;   // hang
   sortTable[0x9B67]=8583;   // hang
   sortTable[0x8CA5]=8584;   // hang
   sortTable[0x82C0]=8585;   // hang
   sortTable[0x57B3]=8586;   // hang
   sortTable[0x65BB]=8587;   // hang
   sortTable[0x7B55]=8588;   // hang
   sortTable[0x7D4E]=8589;   // hang
   sortTable[0x980F]=8590;   // hang
   sortTable[0x597D]=8591;   // hao
   sortTable[0x53F7]=8592;   // hao
   sortTable[0x865F]=8593;   // hao
   sortTable[0x6BEB]=8594;   // hao
   sortTable[0x8017]=8595;   // hao
   sortTable[0x8C6A]=8596;   // hao
   sortTable[0x58D5]=8597;   // hao
   sortTable[0x6D69]=8598;   // hao
   sortTable[0x7693]=8599;   // hao
   sortTable[0x90DD]=8600;   // hao
   sortTable[0x660A]=8601;   // hao
   sortTable[0x6FE0]=8602;   // hao
   sortTable[0x568E]=8603;   // hao
   sortTable[0x98A2]=8604;   // hao
   sortTable[0x704F]=8605;   // hao
   sortTable[0x55E5]=8606;   // hao
   sortTable[0x84BF]=8607;   // hao
   sortTable[0x869D]=8608;   // hao
   sortTable[0x8C89]=8609;   // hao
   sortTable[0x8585]=8610;   // hao
   sortTable[0x9117]=8611;   // hao
   sortTable[0x5686]=8612;   // hao
   sortTable[0x79CF]=8613;   // hao
   sortTable[0x9865]=8614;   // hao
   sortTable[0x8814]=8615;   // hao
   sortTable[0x8AD5]=8616;   // hao
   sortTable[0x8320]=8617;   // hao
   sortTable[0x85A7]=8618;   // hao
   sortTable[0x7C47]=8619;   // hao
   sortTable[0x9C1D]=8620;   // hao
   sortTable[0x8583]=8621;   // hao
   sortTable[0x5090]=8622;   // hao
   sortTable[0x512B]=8623;   // hao
   sortTable[0x54E0]=8624;   // hao
   sortTable[0x5637]=8625;   // hao
   sortTable[0x5651]=8626;   // hao
   sortTable[0x5CFC]=8627;   // hao
   sortTable[0x604F]=8628;   // hao
   sortTable[0x608E]=8629;   // hao
   sortTable[0x6626]=8630;   // hao
   sortTable[0x6667]=8631;   // hao
   sortTable[0x66A4]=8632;   // hao
   sortTable[0x66AD]=8633;   // hao
   sortTable[0x66CD]=8634;   // hao
   sortTable[0x6903]=8635;   // hao
   sortTable[0x6BDC]=8636;   // hao
   sortTable[0x6DCF]=8637;   // hao
   sortTable[0x6EC8]=8638;   // hao
   sortTable[0x6F94]=8639;   // hao
   sortTable[0x705D]=8640;   // hao
   sortTable[0x7346]=8641;   // hao
   sortTable[0x734B]=8642;   // hao
   sortTable[0x7354]=8643;   // hao
   sortTable[0x769C]=8644;   // hao
   sortTable[0x769E]=8645;   // hao
   sortTable[0x76A1]=8646;   // hao
   sortTable[0x76A5]=8647;   // hao
   sortTable[0x7AD3]=8648;   // hao
   sortTable[0x8055]=8649;   // hao
   sortTable[0x8B79]=8650;   // hao
   sortTable[0x93AC]=8651;   // hao
   sortTable[0x548C]=8652;   // he
   sortTable[0x5408]=8653;   // he
   sortTable[0x6CB3]=8654;   // he
   sortTable[0x4F55]=8655;   // he
   sortTable[0x559D]=8656;   // he
   sortTable[0x76D2]=8657;   // he
   sortTable[0x8377]=8658;   // he
   sortTable[0x6838]=8659;   // he
   sortTable[0x8D3A]=8660;   // he
   sortTable[0x8CC0]=8661;   // he
   sortTable[0x55EC]=8662;   // he
   sortTable[0x8910]=8663;   // he
   sortTable[0x8D6B]=8664;   // he
   sortTable[0x9E64]=8665;   // he
   sortTable[0x52BE]=8666;   // he
   sortTable[0x79BE]=8667;   // he
   sortTable[0x58D1]=8668;   // he
   sortTable[0x9616]=8669;   // he
   sortTable[0x66F7]=8670;   // he
   sortTable[0x6DB8]=8671;   // he
   sortTable[0x8BC3]=8672;   // he
   sortTable[0x83CF]=8673;   // he
   sortTable[0x9602]=8674;   // he
   sortTable[0x988C]=8675;   // he
   sortTable[0x76CD]=8676;   // he
   sortTable[0x90C3]=8677;   // he
   sortTable[0x9FA2]=8678;   // he
   sortTable[0x86B5]=8679;   // he
   sortTable[0x7FEE]=8680;   // he
   sortTable[0x97A8]=8681;   // he
   sortTable[0x8988]=8682;   // he
   sortTable[0x9F81]=8683;   // he
   sortTable[0x9E56]=8684;   // he
   sortTable[0x9978]=8685;   // he
   sortTable[0x7909]=8686;   // he
   sortTable[0x9EA7]=8687;   // he
   sortTable[0x7C7A]=8688;   // he
   sortTable[0x9C84]=8689;   // he
   sortTable[0x8A65]=8690;   // he
   sortTable[0x881A]=8691;   // he
   sortTable[0x76C7]=8692;   // he
   sortTable[0x7FEF]=8693;   // he
   sortTable[0x8B1E]=8694;   // he
   sortTable[0x7CAD]=8695;   // he
   sortTable[0x8894]=8696;   // he
   sortTable[0x924C]=8697;   // he
   sortTable[0x9DA1]=8698;   // he
   sortTable[0x76C9]=8699;   // he
   sortTable[0x79F4]=8700;   // he
   sortTable[0x7D07]=8701;   // he
   sortTable[0x879B]=8702;   // he
   sortTable[0x9449]=8703;   // he
   sortTable[0x974E]=8704;   // he
   sortTable[0x981C]=8705;   // he
   sortTable[0x9B7A]=8706;   // he
   sortTable[0x9DB4]=8707;   // he
   sortTable[0x9E16]=8708;   // he
   sortTable[0x9F55]=8709;   // he
   sortTable[0x4F6B]=8710;   // he
   sortTable[0x5392]=8711;   // he
   sortTable[0x548A]=8712;   // he
   sortTable[0x54EC]=8713;   // he
   sortTable[0x555D]=8714;   // he
   sortTable[0x55C3]=8715;   // he
   sortTable[0x578E]=8716;   // he
   sortTable[0x59C0]=8717;   // he
   sortTable[0x5CC6]=8718;   // he
   sortTable[0x60D2]=8719;   // he
   sortTable[0x62B2]=8720;   // he
   sortTable[0x6546]=8721;   // he
   sortTable[0x67C7]=8722;   // he
   sortTable[0x6941]=8723;   // he
   sortTable[0x6B31]=8724;   // he
   sortTable[0x6BFC]=8725;   // he
   sortTable[0x6E2E]=8726;   // he
   sortTable[0x6F95]=8727;   // he
   sortTable[0x7103]=8728;   // he
   sortTable[0x7142]=8729;   // he
   sortTable[0x7186]=8730;   // he
   sortTable[0x7187]=8731;   // he
   sortTable[0x7200]=8732;   // he
   sortTable[0x72E2]=8733;   // he
   sortTable[0x764B]=8734;   // he
   sortTable[0x76AC]=8735;   // he
   sortTable[0x788B]=8736;   // he
   sortTable[0x7A52]=8737;   // he
   sortTable[0x7BD5]=8738;   // he
   sortTable[0x8402]=8739;   // he
   sortTable[0x8A36]=8740;   // he
   sortTable[0x8A38]=8741;   // he
   sortTable[0x8C88]=8742;   // he
   sortTable[0x8F05]=8743;   // he
   sortTable[0x95D4]=8744;   // he
   sortTable[0x974D]=8745;   // he
   sortTable[0x974F]=8746;   // he
   sortTable[0x9DAE]=8747;   // he
   sortTable[0x9ED1]=8748;   // hei
   sortTable[0x563F]=8749;   // hei
   sortTable[0x9ED2]=8750;   // hei
   sortTable[0x6F76]=8751;   // hei
   sortTable[0x5F88]=8752;   // hen
   sortTable[0x6068]=8753;   // hen
   sortTable[0x72E0]=8754;   // hen
   sortTable[0x75D5]=8755;   // hen
   sortTable[0x8A6A]=8756;   // hen
   sortTable[0x978E]=8757;   // hen
   sortTable[0x4F77]=8758;   // hen
   sortTable[0x62EB]=8759;   // hen
   sortTable[0x54FC]=8760;   // heng
   sortTable[0x6A2A]=8761;   // heng
   sortTable[0x6A6B]=8762;   // heng
   sortTable[0x8861]=8763;   // heng
   sortTable[0x6052]=8764;   // heng
   sortTable[0x6046]=8765;   // heng
   sortTable[0x4EA8]=8766;   // heng
   sortTable[0x8605]=8767;   // heng
   sortTable[0x6841]=8768;   // heng
   sortTable[0x9E3B]=8769;   // heng
   sortTable[0x811D]=8770;   // heng
   sortTable[0x9445]=8771;   // heng
   sortTable[0x9D34]=8772;   // heng
   sortTable[0x80FB]=8773;   // heng
   sortTable[0x5548]=8774;   // heng
   sortTable[0x583C]=8775;   // heng
   sortTable[0x59EE]=8776;   // heng
   sortTable[0x6099]=8777;   // heng
   sortTable[0x6DA5]=8778;   // heng
   sortTable[0x70C6]=8779;   // heng
   sortTable[0x9D46]=8780;   // heng
   sortTable[0x5677]=8781;   // hm
   sortTable[0x7EA2]=8782;   // hong
   sortTable[0x7D05]=8783;   // hong
   sortTable[0x8F70]=8784;   // hong
   sortTable[0x8F5F]=8785;   // hong
   sortTable[0x6D2A]=8786;   // hong
   sortTable[0x5B8F]=8787;   // hong
   sortTable[0x54C4]=8788;   // hong
   sortTable[0x70D8]=8789;   // hong
   sortTable[0x9E3F]=8790;   // hong
   sortTable[0x5F18]=8791;   // hong
   sortTable[0x85A8]=8792;   // hong
   sortTable[0x8679]=8793;   // hong
   sortTable[0x95F3]=8794;   // hong
   sortTable[0x6CD3]=8795;   // hong
   sortTable[0x7EAE]=8796;   // hong
   sortTable[0x7AD1]=8797;   // hong
   sortTable[0x836D]=8798;   // hong
   sortTable[0x9277]=8799;   // hong
   sortTable[0x8A07]=8800;   // hong
   sortTable[0x8BA7]=8801;   // hong
   sortTable[0x9EC9]=8802;   // hong
   sortTable[0x7FC3]=8803;   // hong
   sortTable[0x92D0]=8804;   // hong
   sortTable[0x857B]=8805;   // hong
   sortTable[0x9B5F]=8806;   // hong
   sortTable[0x8C3C]=8807;   // hong
   sortTable[0x9367]=8808;   // hong
   sortTable[0x8A0C]=8809;   // hong
   sortTable[0x95A7]=8810;   // hong
   sortTable[0x7D18]=8811;   // hong
   sortTable[0x803E]=8812;   // hong
   sortTable[0x8F37]=8813;   // hong
   sortTable[0x8C39]=8814;   // hong
   sortTable[0x7CA0]=8815;   // hong
   sortTable[0x7D8B]=8816;   // hong
   sortTable[0x8453]=8817;   // hong
   sortTable[0x9D3B]=8818;   // hong
   sortTable[0x7AE4]=8819;   // hong
   sortTable[0x82F0]=8820;   // hong
   sortTable[0x9710]=8821;   // hong
   sortTable[0x7FDD]=8822;   // hong
   sortTable[0x7D2D]=8823;   // hong
   sortTable[0x8452]=8824;   // hong
   sortTable[0x8EE3]=8825;   // hong
   sortTable[0x9ECC]=8826;   // hong
   sortTable[0x7861]=8827;   // hong
   sortTable[0x8C3E]=8828;   // hong
   sortTable[0x95C2]=8829;   // hong
   sortTable[0x9783]=8830;   // hong
   sortTable[0x4EDC]=8831;   // hong
   sortTable[0x53FF]=8832;   // hong
   sortTable[0x5430]=8833;   // hong
   sortTable[0x543D]=8834;   // hong
   sortTable[0x544D]=8835;   // hong
   sortTable[0x55CA]=8836;   // hong
   sortTable[0x569D]=8837;   // hong
   sortTable[0x57AC]=8838;   // hong
   sortTable[0x5985]=8839;   // hong
   sortTable[0x5A02]=8840;   // hong
   sortTable[0x5B96]=8841;   // hong
   sortTable[0x5CF5]=8842;   // hong
   sortTable[0x5F4B]=8843;   // hong
   sortTable[0x63C8]=8844;   // hong
   sortTable[0x6494]=8845;   // hong
   sortTable[0x664E]=8846;   // hong
   sortTable[0x6C6F]=8847;   // hong
   sortTable[0x6D64]=8848;   // hong
   sortTable[0x6E31]=8849;   // hong
   sortTable[0x6E39]=8850;   // hong
   sortTable[0x6F42]=8851;   // hong
   sortTable[0x6F8B]=8852;   // hong
   sortTable[0x6F92]=8853;   // hong
   sortTable[0x7074]=8854;   // hong
   sortTable[0x7122]=8855;   // hong
   sortTable[0x7392]=8856;   // hong
   sortTable[0x7854]=8857;   // hong
   sortTable[0x921C]=8858;   // hong
   sortTable[0x92BE]=8859;   // hong
   sortTable[0x958E]=8860;   // hong
   sortTable[0x971F]=8861;   // hong
   sortTable[0x9B28]=8862;   // hong
   sortTable[0x540E]=8863;   // hou
   sortTable[0x5019]=8864;   // hou
   sortTable[0x539A]=8865;   // hou
   sortTable[0x7334]=8866;   // hou
   sortTable[0x543C]=8867;   // hou
   sortTable[0x5589]=8868;   // hou
   sortTable[0x4FAF]=8869;   // hou
   sortTable[0x5F8C]=8870;   // hou
   sortTable[0x9005]=8871;   // hou
   sortTable[0x9C8E]=8872;   // hou
   sortTable[0x7CC7]=8873;   // hou
   sortTable[0x7BCC]=8874;   // hou
   sortTable[0x9ABA]=8875;   // hou
   sortTable[0x5820]=8876;   // hou
   sortTable[0x90C8]=8877;   // hou
   sortTable[0x9F41]=8878;   // hou
   sortTable[0x777A]=8879;   // hou
   sortTable[0x9931]=8880;   // hou
   sortTable[0x936D]=8881;   // hou
   sortTable[0x760A]=8882;   // hou
   sortTable[0x7FED]=8883;   // hou
   sortTable[0x9107]=8884;   // hou
   sortTable[0x8454]=8885;   // hou
   sortTable[0x5795]=8886;   // hou
   sortTable[0x5E3F]=8887;   // hou
   sortTable[0x6D09]=8888;   // hou
   sortTable[0x72BC]=8889;   // hou
   sortTable[0x77E6]=8890;   // hou
   sortTable[0x7FF5]=8891;   // hou
   sortTable[0x8C5E]=8892;   // hou
   sortTable[0x9B9C]=8893;   // hou
   sortTable[0x9BF8]=8894;   // hou
   sortTable[0x9C5F]=8895;   // hou
   sortTable[0x9C98]=8896;   // hou
   sortTable[0x5FFD]=8897;   // hu
   sortTable[0x62A4]=8898;   // hu
   sortTable[0x8B77]=8899;   // hu
   sortTable[0x547C]=8900;   // hu
   sortTable[0x4E4E]=8901;   // hu
   sortTable[0x4E92]=8902;   // hu
   sortTable[0x6E56]=8903;   // hu
   sortTable[0x80E1]=8904;   // hu
   sortTable[0x864E]=8905;   // hu
   sortTable[0x6237]=8906;   // hu
   sortTable[0x6236]=8907;   // hu
   sortTable[0x7CCA]=8908;   // hu
   sortTable[0x58F6]=8909;   // hu
   sortTable[0x58FA]=8910;   // hu
   sortTable[0x8774]=8911;   // hu
   sortTable[0x846B]=8912;   // hu
   sortTable[0x72D0]=8913;   // hu
   sortTable[0x745A]=8914;   // hu
   sortTable[0x552C]=8915;   // hu
   sortTable[0x60DA]=8916;   // hu
   sortTable[0x5F27]=8917;   // hu
   sortTable[0x659B]=8918;   // hu
   sortTable[0x6CAA]=8919;   // hu
   sortTable[0x6248]=8920;   // hu
   sortTable[0x7B0F]=8921;   // hu
   sortTable[0x6D52]=8922;   // hu
   sortTable[0x795C]=8923;   // hu
   sortTable[0x7173]=8924;   // hu
   sortTable[0x7425]=8925;   // hu
   sortTable[0x74E0]=8926;   // hu
   sortTable[0x9E55]=8927;   // hu
   sortTable[0x69F2]=8928;   // hu
   sortTable[0x6019]=8929;   // hu
   sortTable[0x89F3]=8930;   // hu
   sortTable[0x70C0]=8931;   // hu
   sortTable[0x5CB5]=8932;   // hu
   sortTable[0x9E71]=8933;   // hu
   sortTable[0x6EF9]=8934;   // hu
   sortTable[0x7E20]=8935;   // hu
   sortTable[0x7322]=8936;   // hu
   sortTable[0x9120]=8937;   // hu
   sortTable[0x56EB]=8938;   // hu
   sortTable[0x553F]=8939;   // hu
   sortTable[0x9190]=8940;   // hu
   sortTable[0x623D]=8941;   // hu
   sortTable[0x8656]=8942;   // hu
   sortTable[0x7C04]=8943;   // hu
   sortTable[0x992C]=8944;   // hu
   sortTable[0x51B1]=8945;   // hu
   sortTable[0x81B4]=8946;   // hu
   sortTable[0x96BA]=8947;   // hu
   sortTable[0x7C90]=8948;   // hu
   sortTable[0x9C6F]=8949;   // hu
   sortTable[0x7BB6]=8950;   // hu
   sortTable[0x9800]=8951;   // hu
   sortTable[0x865D]=8952;   // hu
   sortTable[0x7D94]=8953;   // hu
   sortTable[0x8290]=8954;   // hu
   sortTable[0x9359]=8955;   // hu
   sortTable[0x9B71]=8956;   // hu
   sortTable[0x8F77]=8957;   // hu
   sortTable[0x9CF8]=8958;   // hu
   sortTable[0x9BF1]=8959;   // hu
   sortTable[0x9CE0]=8960;   // hu
   sortTable[0x8B3C]=8961;   // hu
   sortTable[0x9378]=8962;   // hu
   sortTable[0x96FD]=8963;   // hu
   sortTable[0x879C]=8964;   // hu
   sortTable[0x8EE4]=8965;   // hu
   sortTable[0x4E55]=8966;   // hu
   sortTable[0x4E65]=8967;   // hu
   sortTable[0x4E6F]=8968;   // hu
   sortTable[0x4FFF]=8969;   // hu
   sortTable[0x51B4]=8970;   // hu
   sortTable[0x5322]=8971;   // hu
   sortTable[0x532B]=8972;   // hu
   sortTable[0x5596]=8973;   // hu
   sortTable[0x55C0]=8974;   // hu
   sortTable[0x5611]=8975;   // hu
   sortTable[0x561D]=8976;   // hu
   sortTable[0x569B]=8977;   // hu
   sortTable[0x5780]=8978;   // hu
   sortTable[0x58F7]=8979;   // hu
   sortTable[0x5A5F]=8980;   // hu
   sortTable[0x5AA9]=8981;   // hu
   sortTable[0x5AED]=8982;   // hu
   sortTable[0x5AEE]=8983;   // hu
   sortTable[0x5BE3]=8984;   // hu
   sortTable[0x5E0D]=8985;   // hu
   sortTable[0x5E60]=8986;   // hu
   sortTable[0x5F16]=8987;   // hu
   sortTable[0x6018]=8988;   // hu
   sortTable[0x6057]=8989;   // hu
   sortTable[0x622F]=8990;   // hu
   sortTable[0x6238]=8991;   // hu
   sortTable[0x6287]=8992;   // hu
   sortTable[0x6430]=8993;   // hu
   sortTable[0x6462]=8994;   // hu
   sortTable[0x6608]=8995;   // hu
   sortTable[0x6612]=8996;   // hu
   sortTable[0x66F6]=8997;   // hu
   sortTable[0x6791]=8998;   // hu
   sortTable[0x695B]=8999;   // hu
   sortTable[0x695C]=9000;   // hu
   sortTable[0x69F4]=9001;   // hu
   sortTable[0x6B51]=9002;   // hu
   sortTable[0x6C7B]=9003;   // hu
   sortTable[0x6C8D]=9004;   // hu
   sortTable[0x6CD8]=9005;   // hu
   sortTable[0x6DF4]=9006;   // hu
   sortTable[0x6EEC]=9007;   // hu
   sortTable[0x6EF8]=9008;   // hu
   sortTable[0x702B]=9009;   // hu
   sortTable[0x7100]=9010;   // hu
   sortTable[0x71A9]=9011;   // hu
   sortTable[0x74F3]=9012;   // hu
   sortTable[0x7D57]=9013;   // hu
   sortTable[0x82F8]=9014;   // hu
   sortTable[0x8400]=9015;   // hu
   sortTable[0x851B]=9016;   // hu
   sortTable[0x8530]=9017;   // hu
   sortTable[0x864D]=9018;   // hu
   sortTable[0x885A]=9019;   // hu
   sortTable[0x933F]=9020;   // hu
   sortTable[0x96D0]=9021;   // hu
   sortTable[0x97C4]=9022;   // hu
   sortTable[0x9836]=9023;   // hu
   sortTable[0x9B0D]=9024;   // hu
   sortTable[0x9C17]=9025;   // hu
   sortTable[0x9D60]=9026;   // hu
   sortTable[0x9D98]=9027;   // hu
   sortTable[0x9DA6]=9028;   // hu
   sortTable[0x9E0C]=9029;   // hu
   sortTable[0x8BDD]=9030;   // hua
   sortTable[0x8A71]=9031;   // hua
   sortTable[0x5316]=9032;   // hua
   sortTable[0x82B1]=9033;   // hua
   sortTable[0x753B]=9034;   // hua
   sortTable[0x756B]=9035;   // hua
   sortTable[0x5212]=9036;   // hua
   sortTable[0x6ED1]=9037;   // hua
   sortTable[0x54D7]=9038;   // hua
   sortTable[0x5629]=9039;   // hua
   sortTable[0x534E]=9040;   // hua
   sortTable[0x83EF]=9041;   // hua
   sortTable[0x733E]=9042;   // hua
   sortTable[0x6866]=9043;   // hua
   sortTable[0x6A3A]=9044;   // hua
   sortTable[0x9A85]=9045;   // hua
   sortTable[0x94E7]=9046;   // hua
   sortTable[0x8B41]=9047;   // hua
   sortTable[0x92D8]=9048;   // hua
   sortTable[0x7CC0]=9049;   // hua
   sortTable[0x7E63]=9050;   // hua
   sortTable[0x89DF]=9051;   // hua
   sortTable[0x91EA]=9052;   // hua
   sortTable[0x8B6E]=9053;   // hua
   sortTable[0x9A4A]=9054;   // hua
   sortTable[0x8796]=9055;   // hua
   sortTable[0x9DE8]=9056;   // hua
   sortTable[0x82B2]=9057;   // hua
   sortTable[0x8AE3]=9058;   // hua
   sortTable[0x5283]=9059;   // hua
   sortTable[0x57D6]=9060;   // hua
   sortTable[0x593B]=9061;   // hua
   sortTable[0x59E1]=9062;   // hua
   sortTable[0x5A72]=9063;   // hua
   sortTable[0x5A73]=9064;   // hua
   sortTable[0x5AFF]=9065;   // hua
   sortTable[0x5B05]=9066;   // hua
   sortTable[0x5D0B]=9067;   // hua
   sortTable[0x6433]=9068;   // hua
   sortTable[0x6466]=9069;   // hua
   sortTable[0x64B6]=9070;   // hua
   sortTable[0x6779]=9071;   // hua
   sortTable[0x691B]=9072;   // hua
   sortTable[0x69EC]=9073;   // hua
   sortTable[0x6F85]=9074;   // hua
   sortTable[0x748D]=9075;   // hua
   sortTable[0x7575]=9076;   // hua
   sortTable[0x7874]=9077;   // hua
   sortTable[0x78C6]=9078;   // hua
   sortTable[0x8219]=9079;   // hua
   sortTable[0x848A]=9080;   // hua
   sortTable[0x8550]=9081;   // hua
   sortTable[0x8AAE]=9082;   // hua
   sortTable[0x91EB]=9083;   // hua
   sortTable[0x9335]=9084;   // hua
   sortTable[0x93F5]=9085;   // hua
   sortTable[0x9ECA]=9086;   // hua
   sortTable[0x574F]=9087;   // huai
   sortTable[0x58DE]=9088;   // huai
   sortTable[0x6000]=9089;   // huai
   sortTable[0x61F7]=9090;   // huai
   sortTable[0x5F8A]=9091;   // huai
   sortTable[0x69D0]=9092;   // huai
   sortTable[0x6DEE]=9093;   // huai
   sortTable[0x8E1D]=9094;   // huai
   sortTable[0x8931]=9095;   // huai
   sortTable[0x8922]=9096;   // huai
   sortTable[0x8AD9]=9097;   // huai
   sortTable[0x863E]=9098;   // huai
   sortTable[0x54B6]=9099;   // huai
   sortTable[0x58CA]=9100;   // huai
   sortTable[0x61D0]=9101;   // huai
   sortTable[0x6AF0]=9102;   // huai
   sortTable[0x7024]=9103;   // huai
   sortTable[0x8032]=9104;   // huai
   sortTable[0x8639]=9105;   // huai
   sortTable[0x6362]=9106;   // huan
   sortTable[0x63DB]=9107;   // huan
   sortTable[0x73AF]=9108;   // huan
   sortTable[0x74B0]=9109;   // huan
   sortTable[0x6B22]=9110;   // huan
   sortTable[0x6B61]=9111;   // huan
   sortTable[0x7F13]=9112;   // huan
   sortTable[0x7DE9]=9113;   // huan
   sortTable[0x5E7B]=9114;   // huan
   sortTable[0x5524]=9115;   // huan
   sortTable[0x559A]=9116;   // huan
   sortTable[0x60A3]=9117;   // huan
   sortTable[0x6853]=9118;   // huan
   sortTable[0x5BA6]=9119;   // huan
   sortTable[0x7115]=9120;   // huan
   sortTable[0x8C62]=9121;   // huan
   sortTable[0x75EA]=9122;   // huan
   sortTable[0x6D63]=9123;   // huan
   sortTable[0x6D39]=9124;   // huan
   sortTable[0x9B1F]=9125;   // huan
   sortTable[0x6DA3]=9126;   // huan
   sortTable[0x5942]=9127;   // huan
   sortTable[0x5BF0]=9128;   // huan
   sortTable[0x571C]=9129;   // huan
   sortTable[0x8411]=9130;   // huan
   sortTable[0x737E]=9131;   // huan
   sortTable[0x902D]=9132;   // huan
   sortTable[0x953E]=9133;   // huan
   sortTable[0x64D0]=9134;   // huan
   sortTable[0x9A69]=9135;   // huan
   sortTable[0x90C7]=9136;   // huan
   sortTable[0x7F33]=9137;   // huan
   sortTable[0x8F58]=9138;   // huan
   sortTable[0x8B99]=9139;   // huan
   sortTable[0x956E]=9140;   // huan
   sortTable[0x7746]=9141;   // huan
   sortTable[0x9E6E]=9142;   // huan
   sortTable[0x6F36]=9143;   // huan
   sortTable[0x7D59]=9144;   // huan
   sortTable[0x961B]=9145;   // huan
   sortTable[0x9CA9]=9146;   // huan
   sortTable[0x9370]=9147;   // huan
   sortTable[0x85E7]=9148;   // huan
   sortTable[0x8C86]=9149;   // huan
   sortTable[0x8C9B]=9150;   // huan
   sortTable[0x8341]=9151;   // huan
   sortTable[0x95E4]=9152;   // huan
   sortTable[0x9BC7]=9153;   // huan
   sortTable[0x7CEB]=9154;   // huan
   sortTable[0x7D84]=9155;   // huan
   sortTable[0x8C72]=9156;   // huan
   sortTable[0x9436]=9157;   // huan
   sortTable[0x96C8]=9158;   // huan
   sortTable[0x77A3]=9159;   // huan
   sortTable[0x7E6F]=9160;   // huan
   sortTable[0x7FA6]=9161;   // huan
   sortTable[0x8408]=9162;   // huan
   sortTable[0x9144]=9163;   // huan
   sortTable[0x559B]=9164;   // huan
   sortTable[0x56BE]=9165;   // huan
   sortTable[0x5950]=9166;   // huan
   sortTable[0x5B1B]=9167;   // huan
   sortTable[0x5BCF]=9168;   // huan
   sortTable[0x5CD8]=9169;   // huan
   sortTable[0x5D48]=9170;   // huan
   sortTable[0x610C]=9171;   // huan
   sortTable[0x61FD]=9172;   // huan
   sortTable[0x650C]=9173;   // huan
   sortTable[0x6899]=9174;   // huan
   sortTable[0x69F5]=9175;   // huan
   sortTable[0x6B53]=9176;   // huan
   sortTable[0x6E19]=9177;   // huan
   sortTable[0x6FA3]=9178;   // huan
   sortTable[0x6FB4]=9179;   // huan
   sortTable[0x70C9]=9180;   // huan
   sortTable[0x7165]=9181;   // huan
   sortTable[0x72BF]=9182;   // huan
   sortTable[0x72DF]=9183;   // huan
   sortTable[0x744D]=9184;   // huan
   sortTable[0x74DB]=9185;   // huan
   sortTable[0x7613]=9186;   // huan
   sortTable[0x8092]=9187;   // huan
   sortTable[0x926E]=9188;   // huan
   sortTable[0x9C00]=9189;   // huan
   sortTable[0x9D05]=9190;   // huan
   sortTable[0x9D4D]=9191;   // huan
   sortTable[0x9EC4]=9192;   // huang
   sortTable[0x9EC3]=9193;   // huang
   sortTable[0x614C]=9194;   // huang
   sortTable[0x7687]=9195;   // huang
   sortTable[0x8757]=9196;   // huang
   sortTable[0x8352]=9197;   // huang
   sortTable[0x6643]=9198;   // huang
   sortTable[0x714C]=9199;   // huang
   sortTable[0x78FA]=9200;   // huang
   sortTable[0x604D]=9201;   // huang
   sortTable[0x8C0E]=9202;   // huang
   sortTable[0x8B0A]=9203;   // huang
   sortTable[0x51F0]=9204;   // huang
   sortTable[0x60F6]=9205;   // huang
   sortTable[0x5E4C]=9206;   // huang
   sortTable[0x7C27]=9207;   // huang
   sortTable[0x6E5F]=9208;   // huang
   sortTable[0x968D]=9209;   // huang
   sortTable[0x749C]=9210;   // huang
   sortTable[0x6F62]=9211;   // huang
   sortTable[0x9051]=9212;   // huang
   sortTable[0x8093]=9213;   // huang
   sortTable[0x5FA8]=9214;   // huang
   sortTable[0x7BC1]=9215;   // huang
   sortTable[0x953D]=9216;   // huang
   sortTable[0x9CC7]=9217;   // huang
   sortTable[0x87E5]=9218;   // huang
   sortTable[0x7640]=9219;   // huang
   sortTable[0x824E]=9220;   // huang
   sortTable[0x9404]=9221;   // huang
   sortTable[0x9C09]=9222;   // huang
   sortTable[0x8A64]=9223;   // huang
   sortTable[0x9C51]=9224;   // huang
   sortTable[0x992D]=9225;   // huang
   sortTable[0x845F]=9226;   // huang
   sortTable[0x8DAA]=9227;   // huang
   sortTable[0x9360]=9228;   // huang
   sortTable[0x9DEC]=9229;   // huang
   sortTable[0x97F9]=9230;   // huang
   sortTable[0x9A1C]=9231;   // huang
   sortTable[0x505F]=9232;   // huang
   sortTable[0x5164]=9233;   // huang
   sortTable[0x55A4]=9234;   // huang
   sortTable[0x582D]=9235;   // huang
   sortTable[0x5843]=9236;   // huang
   sortTable[0x58B4]=9237;   // huang
   sortTable[0x595B]=9238;   // huang
   sortTable[0x5A93]=9239;   // huang
   sortTable[0x5BBA]=9240;   // huang
   sortTable[0x5D32]=9241;   // huang
   sortTable[0x5DDF]=9242;   // huang
   sortTable[0x6033]=9243;   // huang
   sortTable[0x6130]=9244;   // huang
   sortTable[0x6644]=9245;   // huang
   sortTable[0x66C2]=9246;   // huang
   sortTable[0x671A]=9247;   // huang
   sortTable[0x697B]=9248;   // huang
   sortTable[0x69A5]=9249;   // huang
   sortTable[0x6ACE]=9250;   // huang
   sortTable[0x6EC9]=9251;   // huang
   sortTable[0x70BE]=9252;   // huang
   sortTable[0x7180]=9253;   // huang
   sortTable[0x71BF]=9254;   // huang
   sortTable[0x735A]=9255;   // huang
   sortTable[0x745D]=9256;   // huang
   sortTable[0x769D]=9257;   // huang
   sortTable[0x76A9]=9258;   // huang
   sortTable[0x7A54]=9259;   // huang
   sortTable[0x7BCA]=9260;   // huang
   sortTable[0x7E28]=9261;   // huang
   sortTable[0x8841]=9262;   // huang
   sortTable[0x8AFB]=9263;   // huang
   sortTable[0x93A4]=9264;   // huang
   sortTable[0x4F1A]=9265;   // hui
   sortTable[0x6703]=9266;   // hui
   sortTable[0x56DE]=9267;   // hui
   sortTable[0x6325]=9268;   // hui
   sortTable[0x63EE]=9269;   // hui
   sortTable[0x7070]=9270;   // hui
   sortTable[0x6062]=9271;   // hui
   sortTable[0x8F89]=9272;   // hui
   sortTable[0x8F1D]=9273;   // hui
   sortTable[0x6BC1]=9274;   // hui
   sortTable[0x6BC0]=9275;   // hui
   sortTable[0x6C47]=9276;   // hui
   sortTable[0x532F]=9277;   // hui
   sortTable[0x6167]=9278;   // hui
   sortTable[0x6094]=9279;   // hui
   sortTable[0x7ED8]=9280;   // hui
   sortTable[0x7E6A]=9281;   // hui
   sortTable[0x60E0]=9282;   // hui
   sortTable[0x5FBD]=9283;   // hui
   sortTable[0x8BB3]=9284;   // hui
   sortTable[0x8D3F]=9285;   // hui
   sortTable[0x6666]=9286;   // hui
   sortTable[0x605A]=9287;   // hui
   sortTable[0x6656]=9288;   // hui
   sortTable[0x79FD]=9289;   // hui
   sortTable[0x9EBE]=9290;   // hui
   sortTable[0x8BF2]=9291;   // hui
   sortTable[0x8BD9]=9292;   // hui
   sortTable[0x8559]=9293;   // hui
   sortTable[0x5F57]=9294;   // hui
   sortTable[0x86D4]=9295;   // hui
   sortTable[0x5349]=9296;   // hui
   sortTable[0x5599]=9297;   // hui
   sortTable[0x87EA]=9298;   // hui
   sortTable[0x54B4]=9299;   // hui
   sortTable[0x867A]=9300;   // hui
   sortTable[0x835F]=9301;   // hui
   sortTable[0x96B3]=9302;   // hui
   sortTable[0x6D04]=9303;   // hui
   sortTable[0x7F0B]=9304;   // hui
   sortTable[0x70E9]=9305;   // hui
   sortTable[0x73F2]=9306;   // hui
   sortTable[0x7FDA]=9307;   // hui
   sortTable[0x7FD9]=9308;   // hui
   sortTable[0x6D4D]=9309;   // hui
   sortTable[0x8334]=9310;   // hui
   sortTable[0x54D5]=9311;   // hui
   sortTable[0x8886]=9312;   // hui
   sortTable[0x8B53]=9313;   // hui
   sortTable[0x9613]=9314;   // hui
   sortTable[0x8FF4]=9315;   // hui
   sortTable[0x8A6F]=9316;   // hui
   sortTable[0x7FEC]=9317;   // hui
   sortTable[0x7BF2]=9318;   // hui
   sortTable[0x8918]=9319;   // hui
   sortTable[0x8A7C]=9320;   // hui
   sortTable[0x7FFD]=9321;   // hui
   sortTable[0x8C57]=9322;   // hui
   sortTable[0x8698]=9323;   // hui
   sortTable[0x8589]=9324;   // hui
   sortTable[0x9892]=9325;   // hui
   sortTable[0x8633]=9326;   // hui
   sortTable[0x93F8]=9327;   // hui
   sortTable[0x9767]=9328;   // hui
   sortTable[0x986A]=9329;   // hui
   sortTable[0x942C]=9330;   // hui
   sortTable[0x95E0]=9331;   // hui
   sortTable[0x7773]=9332;   // hui
   sortTable[0x8588]=9333;   // hui
   sortTable[0x9BB0]=9334;   // hui
   sortTable[0x8AF1]=9335;   // hui
   sortTable[0x8CC4]=9336;   // hui
   sortTable[0x7988]=9337;   // hui
   sortTable[0x7A62]=9338;   // hui
   sortTable[0x8294]=9339;   // hui
   sortTable[0x8527]=9340;   // hui
   sortTable[0x8B7F]=9341;   // hui
   sortTable[0x4F6A]=9342;   // hui
   sortTable[0x50E1]=9343;   // hui
   sortTable[0x5136]=9344;   // hui
   sortTable[0x5612]=9345;   // hui
   sortTable[0x5645]=9346;   // hui
   sortTable[0x5655]=9347;   // hui
   sortTable[0x5666]=9348;   // hui
   sortTable[0x5696]=9349;   // hui
   sortTable[0x56D8]=9350;   // hui
   sortTable[0x56EC]=9351;   // hui
   sortTable[0x571A]=9352;   // hui
   sortTable[0x5A4E]=9353;   // hui
   sortTable[0x5A88]=9354;   // hui
   sortTable[0x5B12]=9355;   // hui
   sortTable[0x5B48]=9356;   // hui
   sortTable[0x5BED]=9357;   // hui
   sortTable[0x5C77]=9358;   // hui
   sortTable[0x5E51]=9359;   // hui
   sortTable[0x5EFB]=9360;   // hui
   sortTable[0x5EFD]=9361;   // hui
   sortTable[0x5F59]=9362;   // hui
   sortTable[0x5F5A]=9363;   // hui
   sortTable[0x5FBB]=9364;   // hui
   sortTable[0x605B]=9365;   // hui
   sortTable[0x6075]=9366;   // hui
   sortTable[0x6193]=9367;   // hui
   sortTable[0x61F3]=9368;   // hui
   sortTable[0x62FB]=9369;   // hui
   sortTable[0x649D]=9370;   // hui
   sortTable[0x6689]=9371;   // hui
   sortTable[0x66B3]=9372;   // hui
   sortTable[0x694E]=9373;   // hui
   sortTable[0x69E5]=9374;   // hui
   sortTable[0x6A5E]=9375;   // hui
   sortTable[0x6A93]=9376;   // hui
   sortTable[0x6AD8]=9377;   // hui
   sortTable[0x6BA8]=9378;   // hui
   sortTable[0x6BC7]=9379;   // hui
   sortTable[0x6CCB]=9380;   // hui
   sortTable[0x6D03]=9381;   // hui
   sortTable[0x6E4F]=9382;   // hui
   sortTable[0x6ED9]=9383;   // hui
   sortTable[0x6F53]=9384;   // hui
   sortTable[0x6FAE]=9385;   // hui
   sortTable[0x6FCA]=9386;   // hui
   sortTable[0x7008]=9387;   // hui
   sortTable[0x7073]=9388;   // hui
   sortTable[0x70E0]=9389;   // hui
   sortTable[0x70E3]=9390;   // hui
   sortTable[0x7147]=9391;   // hui
   sortTable[0x71EC]=9392;   // hui
   sortTable[0x71F4]=9393;   // hui
   sortTable[0x7369]=9394;   // hui
   sortTable[0x74A4]=9395;   // hui
   sortTable[0x74AF]=9396;   // hui
   sortTable[0x75D0]=9397;   // hui
   sortTable[0x7623]=9398;   // hui
   sortTable[0x77BA]=9399;   // hui
   sortTable[0x7D75]=9400;   // hui
   sortTable[0x7E62]=9401;   // hui
   sortTable[0x85F1]=9402;   // hui
   sortTable[0x86D5]=9403;   // hui
   sortTable[0x8716]=9404;   // hui
   sortTable[0x8AA8]=9405;   // hui
   sortTable[0x8B6D]=9406;   // hui
   sortTable[0x9025]=9407;   // hui
   sortTable[0x9693]=9408;   // hui
   sortTable[0x982E]=9409;   // hui
   sortTable[0x992F]=9410;   // hui
   sortTable[0x9C34]=9411;   // hui
   sortTable[0x6DF7]=9412;   // hun
   sortTable[0x660F]=9413;   // hun
   sortTable[0x6D51]=9414;   // hun
   sortTable[0x6E3E]=9415;   // hun
   sortTable[0x5A5A]=9416;   // hun
   sortTable[0x9B42]=9417;   // hun
   sortTable[0x8BE8]=9418;   // hun
   sortTable[0x8364]=9419;   // hun
   sortTable[0x960D]=9420;   // hun
   sortTable[0x9984]=9421;   // hun
   sortTable[0x6EB7]=9422;   // hun
   sortTable[0x8AE2]=9423;   // hun
   sortTable[0x8477]=9424;   // hun
   sortTable[0x9F32]=9425;   // hun
   sortTable[0x9BF6]=9426;   // hun
   sortTable[0x7E49]=9427;   // hun
   sortTable[0x8F4B]=9428;   // hun
   sortTable[0x7767]=9429;   // hun
   sortTable[0x95BD]=9430;   // hun
   sortTable[0x4FD2]=9431;   // hun
   sortTable[0x5031]=9432;   // hun
   sortTable[0x5702]=9433;   // hun
   sortTable[0x581A]=9434;   // hun
   sortTable[0x5FF6]=9435;   // hun
   sortTable[0x60DB]=9436;   // hun
   sortTable[0x6141]=9437;   // hun
   sortTable[0x638D]=9438;   // hun
   sortTable[0x662C]=9439;   // hun
   sortTable[0x68A1]=9440;   // hun
   sortTable[0x68D4]=9441;   // hun
   sortTable[0x6B99]=9442;   // hun
   sortTable[0x6DBD]=9443;   // hun
   sortTable[0x711D]=9444;   // hun
   sortTable[0x743F]=9445;   // hun
   sortTable[0x776F]=9446;   // hun
   sortTable[0x89E8]=9447;   // hun
   sortTable[0x991B]=9448;   // hun
   sortTable[0x6D3B]=9449;   // huo
   sortTable[0x6216]=9450;   // huo
   sortTable[0x706B]=9451;   // huo
   sortTable[0x4F19]=9452;   // huo
   sortTable[0x8D27]=9453;   // huo
   sortTable[0x8CA8]=9454;   // huo
   sortTable[0x83B7]=9455;   // huo
   sortTable[0x7372]=9456;   // huo
   sortTable[0x8C41]=9457;   // huo
   sortTable[0x7978]=9458;   // huo
   sortTable[0x798D]=9459;   // huo
   sortTable[0x60D1]=9460;   // huo
   sortTable[0x970D]=9461;   // huo
   sortTable[0x8816]=9462;   // huo
   sortTable[0x5925]=9463;   // huo
   sortTable[0x85FF]=9464;   // huo
   sortTable[0x7809]=9465;   // huo
   sortTable[0x956C]=9466;   // huo
   sortTable[0x952A]=9467;   // huo
   sortTable[0x56AF]=9468;   // huo
   sortTable[0x94AC]=9469;   // huo
   sortTable[0x5290]=9470;   // huo
   sortTable[0x8020]=9471;   // huo
   sortTable[0x6509]=9472;   // huo
   sortTable[0x81DB]=9473;   // huo
   sortTable[0x7A6B]=9474;   // huo
   sortTable[0x84A6]=9475;   // huo
   sortTable[0x9743]=9476;   // huo
   sortTable[0x9A1E]=9477;   // huo
   sortTable[0x77D0]=9478;   // huo
   sortTable[0x79F3]=9479;   // huo
   sortTable[0x77C6]=9480;   // huo
   sortTable[0x8267]=9481;   // huo
   sortTable[0x79EE]=9482;   // huo
   sortTable[0x944A]=9483;   // huo
   sortTable[0x4F78]=9484;   // huo
   sortTable[0x4FF0]=9485;   // huo
   sortTable[0x5268]=9486;   // huo
   sortTable[0x5419]=9487;   // huo
   sortTable[0x549F]=9488;   // huo
   sortTable[0x5684]=9489;   // huo
   sortTable[0x56BF]=9490;   // huo
   sortTable[0x596F]=9491;   // huo
   sortTable[0x6347]=9492;   // huo
   sortTable[0x639D]=9493;   // huo
   sortTable[0x65E4]=9494;   // huo
   sortTable[0x66E4]=9495;   // huo
   sortTable[0x6947]=9496;   // huo
   sortTable[0x6AB4]=9497;   // huo
   sortTable[0x6C8E]=9498;   // huo
   sortTable[0x6E71]=9499;   // huo
   sortTable[0x6F37]=9500;   // huo
   sortTable[0x6FE9]=9501;   // huo
   sortTable[0x7016]=9502;   // huo
   sortTable[0x7668]=9503;   // huo
   sortTable[0x7713]=9504;   // huo
   sortTable[0x802F]=9505;   // huo
   sortTable[0x8B0B]=9506;   // huo
   sortTable[0x90A9]=9507;   // huo
   sortTable[0x9225]=9508;   // huo
   sortTable[0x9343]=9509;   // huo
   sortTable[0x9584]=9510;   // huo
   sortTable[0x5DF1]=10240;   // ji
   sortTable[0x673A]=10241;   // ji
   sortTable[0x6A5F]=10242;   // ji
   sortTable[0x51E0]=10243;   // ji
   sortTable[0x5E7E]=10244;   // ji
   sortTable[0x7EA7]=10245;   // ji
   sortTable[0x7D1A]=10246;   // ji
   sortTable[0x8BA1]=10247;   // ji
   sortTable[0x8A08]=10248;   // ji
   sortTable[0x6D4E]=10249;   // ji
   sortTable[0x6FDF]=10250;   // ji
   sortTable[0x6781]=10251;   // ji
   sortTable[0x6975]=10252;   // ji
   sortTable[0x57FA]=10253;   // ji
   sortTable[0x8BB0]=10254;   // ji
   sortTable[0x8A18]=10255;   // ji
   sortTable[0x96C6]=10256;   // ji
   sortTable[0x53CA]=10257;   // ji
   sortTable[0x6280]=10258;   // ji
   sortTable[0x9645]=10259;   // ji
   sortTable[0x969B]=10260;   // ji
   sortTable[0x51FB]=10261;   // ji
   sortTable[0x64CA]=10262;   // ji
   sortTable[0x6025]=10263;   // ji
   sortTable[0x79EF]=10264;   // ji
   sortTable[0x7A4D]=10265;   // ji
   sortTable[0x5373]=10266;   // ji
   sortTable[0x7EAA]=10267;   // ji
   sortTable[0x7D00]=10268;   // ji
   sortTable[0x6FC0]=10269;   // ji
   sortTable[0x7EE7]=10270;   // ji
   sortTable[0x7E7C]=10271;   // ji
   sortTable[0x65E2]=10272;   // ji
   sortTable[0x9E21]=10273;   // ji
   sortTable[0x96DE]=10274;   // ji
   sortTable[0x5B63]=10275;   // ji
   sortTable[0x6324]=10276;   // ji
   sortTable[0x64E0]=10277;   // ji
   sortTable[0x7EE9]=10278;   // ji
   sortTable[0x7E3E]=10279;   // ji
   sortTable[0x5BC4]=10280;   // ji
   sortTable[0x8FF9]=10281;   // ji
   sortTable[0x8DE1]=10282;   // ji
   sortTable[0x5242]=10283;   // ji
   sortTable[0x5291]=10284;   // ji
   sortTable[0x5BC2]=10285;   // ji
   sortTable[0x75BE]=10286;   // ji
   sortTable[0x808C]=10287;   // ji
   sortTable[0x8F91]=10288;   // ji
   sortTable[0x8F2F]=10289;   // ji
   sortTable[0x810A]=10290;   // ji
   sortTable[0x573E]=10291;   // ji
   sortTable[0x9965]=10292;   // ji
   sortTable[0x98E2]=10293;   // ji
   sortTable[0x7C4D]=10294;   // ji
   sortTable[0x8BA5]=10295;   // ji
   sortTable[0x8B4F]=10296;   // ji
   sortTable[0x5409]=10297;   // ji
   sortTable[0x796D]=10298;   // ji
   sortTable[0x5FCC]=10299;   // ji
   sortTable[0x7A3D]=10300;   // ji
   sortTable[0x5180]=10301;   // ji
   sortTable[0x59EC]=10302;   // ji
   sortTable[0x7391]=10303;   // ji
   sortTable[0x7A37]=10304;   // ji
   sortTable[0x85C9]=10305;   // ji
   sortTable[0x5993]=10306;   // ji
   sortTable[0x8BD8]=10307;   // ji
   sortTable[0x6C72]=10308;   // ji
   sortTable[0x5AC9]=10309;   // ji
   sortTable[0x4E9F]=10310;   // ji
   sortTable[0x68D8]=10311;   // ji
   sortTable[0x757F]=10312;   // ji
   sortTable[0x7F81]=10313;   // ji
   sortTable[0x7F09]=10314;   // ji
   sortTable[0x621F]=10315;   // ji
   sortTable[0x8D4D]=10316;   // ji
   sortTable[0x4F0E]=10317;   // ji
   sortTable[0x84DF]=10318;   // ji
   sortTable[0x77F6]=10319;   // ji
   sortTable[0x5048]=10320;   // ji
   sortTable[0x66A8]=10321;   // ji
   sortTable[0x7B95]=10322;   // ji
   sortTable[0x8DFB]=10323;   // ji
   sortTable[0x53FD]=10324;   // ji
   sortTable[0x9AFB]=10325;   // ji
   sortTable[0x9AA5]=10326;   // ji
   sortTable[0x60B8]=10327;   // ji
   sortTable[0x7578]=10328;   // ji
   sortTable[0x54AD]=10329;   // ji
   sortTable[0x5527]=10330;   // ji
   sortTable[0x7B08]=10331;   // ji
   sortTable[0x7DDD]=10332;   // ji
   sortTable[0x7620]=10333;   // ji
   sortTable[0x9701]=10334;   // ji
   sortTable[0x696B]=10335;   // ji
   sortTable[0x4E69]=10336;   // ji
   sortTable[0x7B04]=10337;   // ji
   sortTable[0x89CA]=10338;   // ji
   sortTable[0x5D47]=10339;   // ji
   sortTable[0x5C8C]=10340;   // ji
   sortTable[0x6D0E]=10341;   // ji
   sortTable[0x58BC]=10342;   // ji
   sortTable[0x82A8]=10343;   // ji
   sortTable[0x82B0]=10344;   // ji
   sortTable[0x6222]=10345;   // ji
   sortTable[0x9F51]=10346;   // ji
   sortTable[0x866E]=10347;   // ji
   sortTable[0x8DFD]=10348;   // ji
   sortTable[0x9CAB]=10349;   // ji
   sortTable[0x5C50]=10350;   // ji
   sortTable[0x4F76]=10351;   // ji
   sortTable[0x638E]=10352;   // ji
   sortTable[0x7284]=10353;   // ji
   sortTable[0x6B9B]=10354;   // ji
   sortTable[0x9E82]=10355;   // ji
   sortTable[0x5D74]=10356;   // ji
   sortTable[0x84BA]=10357;   // ji
   sortTable[0x7F7D]=10358;   // ji
   sortTable[0x8360]=10359;   // ji
   sortTable[0x8024]=10360;   // ji
   sortTable[0x54DC]=10361;   // ji
   sortTable[0x8E50]=10362;   // ji
   sortTable[0x9C9A]=10363;   // ji
   sortTable[0x79A8]=10364;   // ji
   sortTable[0x525E]=10365;   // ji
   sortTable[0x9F4E]=10366;   // ji
   sortTable[0x8E16]=10367;   // ji
   sortTable[0x7D12]=10368;   // ji
   sortTable[0x7A44]=10369;   // ji
   sortTable[0x96AE]=10370;   // ji
   sortTable[0x857A]=10371;   // ji
   sortTable[0x9E61]=10372;   // ji
   sortTable[0x7F87]=10373;   // ji
   sortTable[0x9491]=10374;   // ji
   sortTable[0x7A4A]=10375;   // ji
   sortTable[0x830D]=10376;   // ji
   sortTable[0x8E5F]=10377;   // ji
   sortTable[0x8265]=10378;   // ji
   sortTable[0x8324]=10379;   // ji
   sortTable[0x9C7E]=10380;   // ji
   sortTable[0x7A18]=10381;   // ji
   sortTable[0x9951]=10382;   // ji
   sortTable[0x81EE]=10383;   // ji
   sortTable[0x8507]=10384;   // ji
   sortTable[0x8640]=10385;   // ji
   sortTable[0x89ED]=10386;   // ji
   sortTable[0x913F]=10387;   // ji
   sortTable[0x97BF]=10388;   // ji
   sortTable[0x97F2]=10389;   // ji
   sortTable[0x802D]=10390;   // ji
   sortTable[0x9BDA]=10391;   // ji
   sortTable[0x9DBA]=10392;   // ji
   sortTable[0x9353]=10393;   // ji
   sortTable[0x9C36]=10394;   // ji
   sortTable[0x9C40]=10395;   // ji
   sortTable[0x799D]=10396;   // ji
   sortTable[0x7A67]=10397;   // ji
   sortTable[0x7E4B]=10398;   // ji
   sortTable[0x860E]=10399;   // ji
   sortTable[0x8AC5]=10400;   // ji
   sortTable[0x8D8C]=10401;   // ji
   sortTable[0x9F4F]=10402;   // ji
   sortTable[0x7B53]=10403;   // ji
   sortTable[0x78EF]=10404;   // ji
   sortTable[0x7F88]=10405;   // ji
   sortTable[0x846A]=10406;   // ji
   sortTable[0x8540]=10407;   // ji
   sortTable[0x894B]=10408;   // ji
   sortTable[0x9288]=10409;   // ji
   sortTable[0x92A1]=10410;   // ji
   sortTable[0x9416]=10411;   // ji
   sortTable[0x9735]=10412;   // ji
   sortTable[0x9B62]=10413;   // ji
   sortTable[0x9C3F]=10414;   // ji
   sortTable[0x9D4B]=10415;   // ji
   sortTable[0x7A56]=10416;   // ji
   sortTable[0x7BBF]=10417;   // ji
   sortTable[0x862E]=10418;   // ji
   sortTable[0x878F]=10419;   // ji
   sortTable[0x8989]=10420;   // ji
   sortTable[0x898A]=10421;   // ji
   sortTable[0x89D9]=10422;   // ji
   sortTable[0x8C3B]=10423;   // ji
   sortTable[0x8CF7]=10424;   // ji
   sortTable[0x8F5A]=10425;   // ji
   sortTable[0x9218]=10426;   // ji
   sortTable[0x96E6]=10427;   // ji
   sortTable[0x9A65]=10428;   // ji
   sortTable[0x9D8F]=10429;   // ji
   sortTable[0x9DC4]=10430;   // ji
   sortTable[0x9DD1]=10431;   // ji
   sortTable[0x4E0C]=10432;   // ji
   sortTable[0x4E2E]=10433;   // ji
   sortTable[0x4EBC]=10434;   // ji
   sortTable[0x4EBD]=10435;   // ji
   sortTable[0x4F0B]=10436;   // ji
   sortTable[0x506E]=10437;   // ji
   sortTable[0x50DF]=10438;   // ji
   sortTable[0x517E]=10439;   // ji
   sortTable[0x5209]=10440;   // ji
   sortTable[0x520F]=10441;   // ji
   sortTable[0x5264]=10442;   // ji
   sortTable[0x52E3]=10443;   // ji
   sortTable[0x5359]=10444;   // ji
   sortTable[0x537D]=10445;   // ji
   sortTable[0x559E]=10446;   // ji
   sortTable[0x55D8]=10447;   // ji
   sortTable[0x5630]=10448;   // ji
   sortTable[0x568C]=10449;   // ji
   sortTable[0x5756]=10450;   // ji
   sortTable[0x578D]=10451;   // ji
   sortTable[0x5848]=10452;   // ji
   sortTable[0x5849]=10453;   // ji
   sortTable[0x5980]=10454;   // ji
   sortTable[0x59DE]=10455;   // ji
   sortTable[0x59EB]=10456;   // ji
   sortTable[0x5CDC]=10457;   // ji
   sortTable[0x5D46]=10458;   // ji
   sortTable[0x5DAF]=10459;   // ji
   sortTable[0x5EB4]=10460;   // ji
   sortTable[0x5EED]=10461;   // ji
   sortTable[0x5F50]=10462;   // ji
   sortTable[0x5F51]=10463;   // ji
   sortTable[0x5F76]=10464;   // ji
   sortTable[0x5F9B]=10465;   // ji
   sortTable[0x5FE3]=10466;   // ji
   sortTable[0x60CE]=10467;   // ji
   sortTable[0x6131]=10468;   // ji
   sortTable[0x61FB]=10469;   // ji
   sortTable[0x63E4]=10470;   // ji
   sortTable[0x6483]=10471;   // ji
   sortTable[0x64A0]=10472;   // ji
   sortTable[0x64EE]=10473;   // ji
   sortTable[0x6567]=10474;   // ji
   sortTable[0x65E1]=10475;   // ji
   sortTable[0x65E3]=10476;   // ji
   sortTable[0x66A9]=10477;   // ji
   sortTable[0x66C1]=10478;   // ji
   sortTable[0x671E]=10479;   // ji
   sortTable[0x6785]=10480;   // ji
   sortTable[0x689E]=10481;   // ji
   sortTable[0x69C9]=10482;   // ji
   sortTable[0x69E3]=10483;   // ji
   sortTable[0x6A2D]=10484;   // ji
   sortTable[0x6A76]=10485;   // ji
   sortTable[0x6A95]=10486;   // ji
   sortTable[0x6A9D]=10487;   // ji
   sortTable[0x6AB5]=10488;   // ji
   sortTable[0x6AC5]=10489;   // ji
   sortTable[0x6BC4]=10490;   // ji
   sortTable[0x6CF2]=10491;   // ji
   sortTable[0x6E08]=10492;   // ji
   sortTable[0x6E52]=10493;   // ji
   sortTable[0x6F03]=10494;   // ji
   sortTable[0x6F08]=10495;   // ji
   sortTable[0x6F57]=10496;   // ji
   sortTable[0x6FC8]=10497;   // ji
   sortTable[0x7031]=10498;   // ji
   sortTable[0x710F]=10499;   // ji
   sortTable[0x72B1]=10500;   // ji
   sortTable[0x72E4]=10501;   // ji
   sortTable[0x74A3]=10502;   // ji
   sortTable[0x75F5]=10503;   // ji
   sortTable[0x7660]=10504;   // ji
   sortTable[0x766A]=10505;   // ji
   sortTable[0x7680]=10506;   // ji
   sortTable[0x768D]=10507;   // ji
   sortTable[0x7A29]=10508;   // ji
   sortTable[0x7C0A]=10509;   // ji
   sortTable[0x7D99]=10510;   // ji
   sortTable[0x818C]=10511;   // ji
   sortTable[0x858A]=10512;   // ji
   sortTable[0x85BA]=10513;   // ji
   sortTable[0x863B]=10514;   // ji
   sortTable[0x87E3]=10515;   // ji
   sortTable[0x88DA]=10516;   // ji
   sortTable[0x8900]=10517;   // ji
   sortTable[0x8940]=10518;   // ji
   sortTable[0x89AC]=10519;   // ji
   sortTable[0x8A8B]=10520;   // ji
   sortTable[0x8B64]=10521;   // ji
   sortTable[0x8CEB]=10522;   // ji
   sortTable[0x8E8B]=10523;   // ji
   sortTable[0x8EB8]=10524;   // ji
   sortTable[0x90C6]=10525;   // ji
   sortTable[0x9324]=10526;   // ji
   sortTable[0x93F6]=10527;   // ji
   sortTable[0x9447]=10528;   // ji
   sortTable[0x9459]=10529;   // ji
   sortTable[0x96E7]=10530;   // ji
   sortTable[0x973D]=10531;   // ji
   sortTable[0x9B3E]=10532;   // ji
   sortTable[0x9B55]=10533;   // ji
   sortTable[0x9C6D]=10534;   // ji
   sortTable[0x9CEE]=10535;   // ji
   sortTable[0x9E04]=10536;   // ji
   sortTable[0x9F4C]=10537;   // ji
   sortTable[0x5BB6]=10538;   // jia
   sortTable[0x52A0]=10539;   // jia
   sortTable[0x4EF7]=10540;   // jia
   sortTable[0x50F9]=10541;   // jia
   sortTable[0x67B6]=10542;   // jia
   sortTable[0x5047]=10543;   // jia
   sortTable[0x7A3C]=10544;   // jia
   sortTable[0x5939]=10545;   // jia
   sortTable[0x593E]=10546;   // jia
   sortTable[0x7532]=10547;   // jia
   sortTable[0x9A7E]=10548;   // jia
   sortTable[0x99D5]=10549;   // jia
   sortTable[0x5AC1]=10550;   // jia
   sortTable[0x4F73]=10551;   // jia
   sortTable[0x988A]=10552;   // jia
   sortTable[0x8304]=10553;   // jia
   sortTable[0x9830]=10554;   // jia
   sortTable[0x8D3E]=10555;   // jia
   sortTable[0x5609]=10556;   // jia
   sortTable[0x8FE6]=10557;   // jia
   sortTable[0x4F3D]=10558;   // jia
   sortTable[0x67B7]=10559;   // jia
   sortTable[0x90CF]=10560;   // jia
   sortTable[0x835A]=10561;   // jia
   sortTable[0x8DCF]=10562;   // jia
   sortTable[0x80DB]=10563;   // jia
   sortTable[0x7B33]=10564;   // jia
   sortTable[0x5CAC]=10565;   // jia
   sortTable[0x88B7]=10566;   // jia
   sortTable[0x8888]=10567;   // jia
   sortTable[0x94BE]=10568;   // jia
   sortTable[0x846D]=10569;   // jia
   sortTable[0x6D43]=10570;   // jia
   sortTable[0x86F1]=10571;   // jia
   sortTable[0x621B]=10572;   // jia
   sortTable[0x75C2]=10573;   // jia
   sortTable[0x73C8]=10574;   // jia
   sortTable[0x605D]=10575;   // jia
   sortTable[0x9553]=10576;   // jia
   sortTable[0x94D7]=10577;   // jia
   sortTable[0x8C6D]=10578;   // jia
   sortTable[0x7615]=10579;   // jia
   sortTable[0x9240]=10580;   // jia
   sortTable[0x88CC]=10581;   // jia
   sortTable[0x9782]=10582;   // jia
   sortTable[0x9E9A]=10583;   // jia
   sortTable[0x8CC8]=10584;   // jia
   sortTable[0x86FA]=10585;   // jia
   sortTable[0x8DF2]=10586;   // jia
   sortTable[0x9D36]=10587;   // jia
   sortTable[0x9D4A]=10588;   // jia
   sortTable[0x8C91]=10589;   // jia
   sortTable[0x83A2]=10590;   // jia
   sortTable[0x90DF]=10591;   // jia
   sortTable[0x801E]=10592;   // jia
   sortTable[0x8175]=10593;   // jia
   sortTable[0x9904]=10594;   // jia
   sortTable[0x4E6B]=10595;   // jia
   sortTable[0x4EEE]=10596;   // jia
   sortTable[0x50A2]=10597;   // jia
   sortTable[0x53DA]=10598;   // jia
   sortTable[0x550A]=10599;   // jia
   sortTable[0x55E7]=10600;   // jia
   sortTable[0x573F]=10601;   // jia
   sortTable[0x57C9]=10602;   // jia
   sortTable[0x5A7D]=10603;   // jia
   sortTable[0x5E4F]=10604;   // jia
   sortTable[0x5FA6]=10605;   // jia
   sortTable[0x5FE6]=10606;   // jia
   sortTable[0x621E]=10607;   // jia
   sortTable[0x6274]=10608;   // jia
   sortTable[0x62B8]=10609;   // jia
   sortTable[0x62C1]=10610;   // jia
   sortTable[0x659A]=10611;   // jia
   sortTable[0x659D]=10612;   // jia
   sortTable[0x689C]=10613;   // jia
   sortTable[0x6935]=10614;   // jia
   sortTable[0x698E]=10615;   // jia
   sortTable[0x69A2]=10616;   // jia
   sortTable[0x69DA]=10617;   // jia
   sortTable[0x6A9F]=10618;   // jia
   sortTable[0x6BE0]=10619;   // jia
   sortTable[0x6CC7]=10620;   // jia
   sortTable[0x6D79]=10621;   // jia
   sortTable[0x728C]=10622;   // jia
   sortTable[0x7333]=10623;   // jia
   sortTable[0x73BE]=10624;   // jia
   sortTable[0x7CD8]=10625;   // jia
   sortTable[0x926B]=10626;   // jia
   sortTable[0x927F]=10627;   // jia
   sortTable[0x92CF]=10628;   // jia
   sortTable[0x93B5]=10629;   // jia
   sortTable[0x982C]=10630;   // jia
   sortTable[0x89C1]=10631;   // jian
   sortTable[0x898B]=10632;   // jian
   sortTable[0x95F4]=10633;   // jian
   sortTable[0x9593]=10634;   // jian
   sortTable[0x5EFA]=10635;   // jian
   sortTable[0x4EF6]=10636;   // jian
   sortTable[0x575A]=10637;   // jian
   sortTable[0x5805]=10638;   // jian
   sortTable[0x6E10]=10639;   // jian
   sortTable[0x6F38]=10640;   // jian
   sortTable[0x7B80]=10641;   // jian
   sortTable[0x7C21]=10642;   // jian
   sortTable[0x68C0]=10643;   // jian
   sortTable[0x6AA2]=10644;   // jian
   sortTable[0x8DF5]=10645;   // jian
   sortTable[0x8E10]=10646;   // jian
   sortTable[0x51CF]=10647;   // jian
   sortTable[0x6E1B]=10648;   // jian
   sortTable[0x8270]=10649;   // jian
   sortTable[0x8271]=10650;   // jian
   sortTable[0x80A9]=10651;   // jian
   sortTable[0x5C16]=10652;   // jian
   sortTable[0x5065]=10653;   // jian
   sortTable[0x76D1]=10654;   // jian
   sortTable[0x76E3]=10655;   // jian
   sortTable[0x6B7C]=10656;   // jian
   sortTable[0x6BB2]=10657;   // jian
   sortTable[0x7BAD]=10658;   // jian
   sortTable[0x62E3]=10659;   // jian
   sortTable[0x63C0]=10660;   // jian
   sortTable[0x5251]=10661;   // jian
   sortTable[0x528D]=10662;   // jian
   sortTable[0x952E]=10663;   // jian
   sortTable[0x9375]=10664;   // jian
   sortTable[0x517C]=10665;   // jian
   sortTable[0x526A]=10666;   // jian
   sortTable[0x9274]=10667;   // jian
   sortTable[0x9452]=10668;   // jian
   sortTable[0x78B1]=10669;   // jian
   sortTable[0x9E7C]=10670;   // jian
   sortTable[0x8D31]=10671;   // jian
   sortTable[0x8CE4]=10672;   // jian
   sortTable[0x6361]=10673;   // jian
   sortTable[0x64BF]=10674;   // jian
   sortTable[0x6E85]=10675;   // jian
   sortTable[0x6FFA]=10676;   // jian
   sortTable[0x714E]=10677;   // jian
   sortTable[0x8230]=10678;   // jian
   sortTable[0x8266]=10679;   // jian
   sortTable[0x5978]=10680;   // jian
   sortTable[0x8350]=10681;   // jian
   sortTable[0x8327]=10682;   // jian
   sortTable[0x7E6D]=10683;   // jian
   sortTable[0x4FED]=10684;   // jian
   sortTable[0x5109]=10685;   // jian
   sortTable[0x8C0F]=10686;   // jian
   sortTable[0x7F04]=10687;   // jian
   sortTable[0x50ED]=10688;   // jian
   sortTable[0x7B3A]=10689;   // jian
   sortTable[0x67EC]=10690;   // jian
   sortTable[0x6DA7]=10691;   // jian
   sortTable[0x996F]=10692;   // jian
   sortTable[0x8E47]=10693;   // jian
   sortTable[0x7FE6]=10694;   // jian
   sortTable[0x622C]=10695;   // jian
   sortTable[0x950F]=10696;   // jian
   sortTable[0x97AF]=10697;   // jian
   sortTable[0x7B15]=10698;   // jian
   sortTable[0x7F23]=10699;   // jian
   sortTable[0x8B07]=10700;   // jian
   sortTable[0x7751]=10701;   // jian
   sortTable[0x728D]=10702;   // jian
   sortTable[0x6E54]=10703;   // jian
   sortTable[0x83C5]=10704;   // jian
   sortTable[0x88E5]=10705;   // jian
   sortTable[0x9E63]=10706;   // jian
   sortTable[0x8DBC]=10707;   // jian
   sortTable[0x97AC]=10708;   // jian
   sortTable[0x8171]=10709;   // jian
   sortTable[0x7877]=10710;   // jian
   sortTable[0x84B9]=10711;   // jian
   sortTable[0x7BEF]=10712;   // jian
   sortTable[0x8C2B]=10713;   // jian
   sortTable[0x620B]=10714;   // jian
   sortTable[0x6957]=10715;   // jian
   sortTable[0x67A7]=10716;   // jian
   sortTable[0x56DD]=10717;   // jian
   sortTable[0x6BFD]=10718;   // jian
   sortTable[0x8AD3]=10719;   // jian
   sortTable[0x9451]=10720;   // jian
   sortTable[0x77B7]=10721;   // jian
   sortTable[0x9CA3]=10722;   // jian
   sortTable[0x641B]=10723;   // jian
   sortTable[0x81F6]=10724;   // jian
   sortTable[0x89B8]=10725;   // jian
   sortTable[0x78F5]=10726;   // jian
   sortTable[0x7C5B]=10727;   // jian
   sortTable[0x844C]=10728;   // jian
   sortTable[0x9D73]=10729;   // jian
   sortTable[0x8573]=10730;   // jian
   sortTable[0x8F5E]=10731;   // jian
   sortTable[0x8E3A]=10732;   // jian
   sortTable[0x77BC]=10733;   // jian
   sortTable[0x7E5D]=10734;   // jian
   sortTable[0x9473]=10735;   // jian
   sortTable[0x9B0B]=10736;   // jian
   sortTable[0x9CD2]=10737;   // jian
   sortTable[0x791B]=10738;   // jian
   sortTable[0x852A]=10739;   // jian
   sortTable[0x8A43]=10740;   // jian
   sortTable[0x9C39]=10741;   // jian
   sortTable[0x9E89]=10742;   // jian
   sortTable[0x7906]=10743;   // jian
   sortTable[0x92FB]=10744;   // jian
   sortTable[0x9417]=10745;   // jian
   sortTable[0x9E7B]=10746;   // jian
   sortTable[0x726E]=10747;   // jian
   sortTable[0x7900]=10748;   // jian
   sortTable[0x7DD8]=10749;   // jian
   sortTable[0x85A6]=10750;   // jian
   sortTable[0x8643]=10751;   // jian
   sortTable[0x8B2D]=10752;   // jian
   sortTable[0x9427]=10753;   // jian
   sortTable[0x946C]=10754;   // jian
   sortTable[0x7B8B]=10755;   // jian
   sortTable[0x83FA]=10756;   // jian
   sortTable[0x8465]=10757;   // jian
   sortTable[0x85C6]=10758;   // jian
   sortTable[0x8812]=10759;   // jian
   sortTable[0x88B8]=10760;   // jian
   sortTable[0x8C5C]=10761;   // jian
   sortTable[0x9930]=10762;   // jian
   sortTable[0x7B67]=10763;   // jian
   sortTable[0x8947]=10764;   // jian
   sortTable[0x897A]=10765;   // jian
   sortTable[0x89B5]=10766;   // jian
   sortTable[0x8D9D]=10767;   // jian
   sortTable[0x9373]=10768;   // jian
   sortTable[0x9CFD]=10769;   // jian
   sortTable[0x4FF4]=10770;   // jian
   sortTable[0x5039]=10771;   // jian
   sortTable[0x5042]=10772;   // jian
   sortTable[0x51BF]=10773;   // jian
   sortTable[0x5263]=10774;   // jian
   sortTable[0x5271]=10775;   // jian
   sortTable[0x528E]=10776;   // jian
   sortTable[0x5292]=10777;   // jian
   sortTable[0x5294]=10778;   // jian
   sortTable[0x5297]=10779;   // jian
   sortTable[0x56CF]=10780;   // jian
   sortTable[0x583F]=10781;   // jian
   sortTable[0x58B9]=10782;   // jian
   sortTable[0x59E6]=10783;   // jian
   sortTable[0x59E7]=10784;   // jian
   sortTable[0x5BCB]=10785;   // jian
   sortTable[0x5E75]=10786;   // jian
   sortTable[0x5F3F]=10787;   // jian
   sortTable[0x5F45]=10788;   // jian
   sortTable[0x5FA4]=10789;   // jian
   sortTable[0x60E4]=10790;   // jian
   sortTable[0x6214]=10791;   // jian
   sortTable[0x6229]=10792;   // jian
   sortTable[0x6338]=10793;   // jian
   sortTable[0x63C3]=10794;   // jian
   sortTable[0x64F6]=10795;   // jian
   sortTable[0x65D4]=10796;   // jian
   sortTable[0x6695]=10797;   // jian
   sortTable[0x682B]=10798;   // jian
   sortTable[0x6898]=10799;   // jian
   sortTable[0x691C]=10800;   // jian
   sortTable[0x6937]=10801;   // jian
   sortTable[0x693E]=10802;   // jian
   sortTable[0x6997]=10803;   // jian
   sortTable[0x6A2B]=10804;   // jian
   sortTable[0x6A7A]=10805;   // jian
   sortTable[0x6AFC]=10806;   // jian
   sortTable[0x6BB1]=10807;   // jian
   sortTable[0x6D0A]=10808;   // jian
   sortTable[0x6E55]=10809;   // jian
   sortTable[0x6F97]=10810;   // jian
   sortTable[0x7010]=10811;   // jian
   sortTable[0x7033]=10812;   // jian
   sortTable[0x7038]=10813;   // jian
   sortTable[0x703D]=10814;   // jian
   sortTable[0x719E]=10815;   // jian
   sortTable[0x71B8]=10816;   // jian
   sortTable[0x724B]=10817;   // jian
   sortTable[0x730F]=10818;   // jian
   sortTable[0x73AA]=10819;   // jian
   sortTable[0x73D4]=10820;   // jian
   sortTable[0x744A]=10821;   // jian
   sortTable[0x7450]=10822;   // jian
   sortTable[0x7777]=10823;   // jian
   sortTable[0x788A]=10824;   // jian
   sortTable[0x7CCB]=10825;   // jian
   sortTable[0x7D78]=10826;   // jian
   sortTable[0x7E11]=10827;   // jian
   sortTable[0x8551]=10828;   // jian
   sortTable[0x87B9]=10829;   // jian
   sortTable[0x8949]=10830;   // jian
   sortTable[0x8AEB]=10831;   // jian
   sortTable[0x8B7C]=10832;   // jian
   sortTable[0x8B7E]=10833;   // jian
   sortTable[0x8C63]=10834;   // jian
   sortTable[0x8CCE]=10835;   // jian
   sortTable[0x91FC]=10836;   // jian
   sortTable[0x93E9]=10837;   // jian
   sortTable[0x9431]=10838;   // jian
   sortTable[0x946F]=10839;   // jian
   sortTable[0x97C0]=10840;   // jian
   sortTable[0x97C9]=10841;   // jian
   sortTable[0x991E]=10842;   // jian
   sortTable[0x99A2]=10843;   // jian
   sortTable[0x9C0E]=10844;   // jian
   sortTable[0x9DBC]=10845;   // jian
   sortTable[0x9E78]=10846;   // jian
   sortTable[0x5C06]=10847;   // jiang
   sortTable[0x5C07]=10848;   // jiang
   sortTable[0x8BB2]=10849;   // jiang
   sortTable[0x8B1B]=10850;   // jiang
   sortTable[0x6C5F]=10851;   // jiang
   sortTable[0x964D]=10852;   // jiang
   sortTable[0x5320]=10853;   // jiang
   sortTable[0x5956]=10854;   // jiang
   sortTable[0x734E]=10855;   // jiang
   sortTable[0x9171]=10856;   // jiang
   sortTable[0x91AC]=10857;   // jiang
   sortTable[0x50F5]=10858;   // jiang
   sortTable[0x6D46]=10859;   // jiang
   sortTable[0x7F30]=10860;   // jiang
   sortTable[0x97C1]=10861;   // jiang
   sortTable[0x6F3F]=10862;   // jiang
   sortTable[0x6868]=10863;   // jiang
   sortTable[0x69F3]=10864;   // jiang
   sortTable[0x7586]=10865;   // jiang
   sortTable[0x59DC]=10866;   // jiang
   sortTable[0x848B]=10867;   // jiang
   sortTable[0x7EDB]=10868;   // jiang
   sortTable[0x729F]=10869;   // jiang
   sortTable[0x8C47]=10870;   // jiang
   sortTable[0x7CE8]=10871;   // jiang
   sortTable[0x6D1A]=10872;   // jiang
   sortTable[0x7913]=10873;   // jiang
   sortTable[0x8591]=10874;   // jiang
   sortTable[0x8029]=10875;   // jiang
   sortTable[0x9CC9]=10876;   // jiang
   sortTable[0x8780]=10877;   // jiang
   sortTable[0x8523]=10878;   // jiang
   sortTable[0x8333]=10879;   // jiang
   sortTable[0x8B3D]=10880;   // jiang
   sortTable[0x985C]=10881;   // jiang
   sortTable[0x9C42]=10882;   // jiang
   sortTable[0x7D73]=10883;   // jiang
   sortTable[0x7FDE]=10884;   // jiang
   sortTable[0x508B]=10885;   // jiang
   sortTable[0x52E5]=10886;   // jiang
   sortTable[0x531E]=10887;   // jiang
   sortTable[0x58C3]=10888;   // jiang
   sortTable[0x5905]=10889;   // jiang
   sortTable[0x5968]=10890;   // jiang
   sortTable[0x596C]=10891;   // jiang
   sortTable[0x5D79]=10892;   // jiang
   sortTable[0x5F1C]=10893;   // jiang
   sortTable[0x5F36]=10894;   // jiang
   sortTable[0x5F4A]=10895;   // jiang
   sortTable[0x646A]=10896;   // jiang
   sortTable[0x647E]=10897;   // jiang
   sortTable[0x6762]=10898;   // jiang
   sortTable[0x6A7F]=10899;   // jiang
   sortTable[0x6AE4]=10900;   // jiang
   sortTable[0x6BAD]=10901;   // jiang
   sortTable[0x6EF0]=10902;   // jiang
   sortTable[0x7555]=10903;   // jiang
   sortTable[0x757A]=10904;   // jiang
   sortTable[0x7585]=10905;   // jiang
   sortTable[0x7CE1]=10906;   // jiang
   sortTable[0x7E6E]=10907;   // jiang
   sortTable[0x8199]=10908;   // jiang
   sortTable[0x8441]=10909;   // jiang
   sortTable[0x87BF]=10910;   // jiang
   sortTable[0x88B6]=10911;   // jiang
   sortTable[0x91A4]=10912;   // jiang
   sortTable[0x53EB]=10913;   // jiao
   sortTable[0x6559]=10914;   // jiao
   sortTable[0x8F83]=10915;   // jiao
   sortTable[0x8F03]=10916;   // jiao
   sortTable[0x4EA4]=10917;   // jiao
   sortTable[0x811A]=10918;   // jiao
   sortTable[0x8173]=10919;   // jiao
   sortTable[0x89D2]=10920;   // jiao
   sortTable[0x7126]=10921;   // jiao
   sortTable[0x9A84]=10922;   // jiao
   sortTable[0x9A55]=10923;   // jiao
   sortTable[0x6D47]=10924;   // jiao
   sortTable[0x6F86]=10925;   // jiao
   sortTable[0x6405]=10926;   // jiao
   sortTable[0x652A]=10927;   // jiao
   sortTable[0x80F6]=10928;   // jiao
   sortTable[0x81A0]=10929;   // jiao
   sortTable[0x8F7F]=10930;   // jiao
   sortTable[0x8F4E]=10931;   // jiao
   sortTable[0x90CA]=10932;   // jiao
   sortTable[0x527F]=10933;   // jiao
   sortTable[0x72E1]=10934;   // jiao
   sortTable[0x7F34]=10935;   // jiao
   sortTable[0x7E73]=10936;   // jiao
   sortTable[0x8549]=10937;   // jiao
   sortTable[0x5A07]=10938;   // jiao
   sortTable[0x6912]=10939;   // jiao
   sortTable[0x7A96]=10940;   // jiao
   sortTable[0x5B0C]=10941;   // jiao
   sortTable[0x997A]=10942;   // jiao
   sortTable[0x9903]=10943;   // jiao
   sortTable[0x9175]=10944;   // jiao
   sortTable[0x77EB]=10945;   // jiao
   sortTable[0x7EDE]=10946;   // jiao
   sortTable[0x86DF]=10947;   // jiao
   sortTable[0x7901]=10948;   // jiao
   sortTable[0x4FA5]=10949;   // jiao
   sortTable[0x91AE]=10950;   // jiao
   sortTable[0x768E]=10951;   // jiao
   sortTable[0x5CE4]=10952;   // jiao
   sortTable[0x8DE4]=10953;   // jiao
   sortTable[0x6E6B]=10954;   // jiao
   sortTable[0x5FBC]=10955;   // jiao
   sortTable[0x94F0]=10956;   // jiao
   sortTable[0x4F7C]=10957;   // jiao
   sortTable[0x9E6A]=10958;   // jiao
   sortTable[0x50EC]=10959;   // jiao
   sortTable[0x656B]=10960;   // jiao
   sortTable[0x59E3]=10961;   // jiao
   sortTable[0x827D]=10962;   // jiao
   sortTable[0x9C9B]=10963;   // jiao
   sortTable[0x832D]=10964;   // jiao
   sortTable[0x564D]=10965;   // jiao
   sortTable[0x6322]=10966;   // jiao
   sortTable[0x87DC]=10967;   // jiao
   sortTable[0x91C2]=10968;   // jiao
   sortTable[0x77EF]=10969;   // jiao
   sortTable[0x87ED]=10970;   // jiao
   sortTable[0x7A8C]=10971;   // jiao
   sortTable[0x9DEE]=10972;   // jiao
   sortTable[0x96A6]=10973;   // jiao
   sortTable[0x9D41]=10974;   // jiao
   sortTable[0x8F47]=10975;   // jiao
   sortTable[0x81B2]=10976;   // jiao
   sortTable[0x7C25]=10977;   // jiao
   sortTable[0x85E0]=10978;   // jiao
   sortTable[0x8DAD]=10979;   // jiao
   sortTable[0x8A06]=10980;   // jiao
   sortTable[0x940E]=10981;   // jiao
   sortTable[0x9DE6]=10982;   // jiao
   sortTable[0x7A5A]=10983;   // jiao
   sortTable[0x7D5E]=10984;   // jiao
   sortTable[0x8B65]=10985;   // jiao
   sortTable[0x9D64]=10986;   // jiao
   sortTable[0x8B51]=10987;   // jiao
   sortTable[0x50E5]=10988;   // jiao
   sortTable[0x510C]=10989;   // jiao
   sortTable[0x528B]=10990;   // jiao
   sortTable[0x544C]=10991;   // jiao
   sortTable[0x5602]=10992;   // jiao
   sortTable[0x5604]=10993;   // jiao
   sortTable[0x5626]=10994;   // jiao
   sortTable[0x566D]=10995;   // jiao
   sortTable[0x5B13]=10996;   // jiao
   sortTable[0x5B42]=10997;   // jiao
   sortTable[0x5CE7]=10998;   // jiao
   sortTable[0x5D95]=10999;   // jiao
   sortTable[0x5DA0]=11000;   // jiao
   sortTable[0x5DA3]=11001;   // jiao
   sortTable[0x5FBA]=11002;   // jiao
   sortTable[0x6054]=11003;   // jiao
   sortTable[0x618D]=11004;   // jiao
   sortTable[0x61BF]=11005;   // jiao
   sortTable[0x630D]=11006;   // jiao
   sortTable[0x6341]=11007;   // jiao
   sortTable[0x6477]=11008;   // jiao
   sortTable[0x649F]=11009;   // jiao
   sortTable[0x64B9]=11010;   // jiao
   sortTable[0x654E]=11011;   // jiao
   sortTable[0x657D]=11012;   // jiao
   sortTable[0x657F]=11013;   // jiao
   sortTable[0x65A0]=11014;   // jiao
   sortTable[0x6648]=11015;   // jiao
   sortTable[0x669E]=11016;   // jiao
   sortTable[0x66D2]=11017;   // jiao
   sortTable[0x6AF5]=11018;   // jiao
   sortTable[0x6E6C]=11019;   // jiao
   sortTable[0x6ED8]=11020;   // jiao
   sortTable[0x6F16]=11021;   // jiao
   sortTable[0x6F50]=11022;   // jiao
   sortTable[0x705A]=11023;   // jiao
   sortTable[0x70C4]=11024;   // jiao
   sortTable[0x714D]=11025;   // jiao
   sortTable[0x71CB]=11026;   // jiao
   sortTable[0x71DE]=11027;   // jiao
   sortTable[0x7365]=11028;   // jiao
   sortTable[0x73D3]=11029;   // jiao
   sortTable[0x74AC]=11030;   // jiao
   sortTable[0x76A6]=11031;   // jiao
   sortTable[0x76AD]=11032;   // jiao
   sortTable[0x7E90]=11033;   // jiao
   sortTable[0x81EB]=11034;   // jiao
   sortTable[0x8281]=11035;   // jiao
   sortTable[0x832E]=11036;   // jiao
   sortTable[0x8660]=11037;   // jiao
   sortTable[0x8CCB]=11038;   // jiao
   sortTable[0x8E0B]=11039;   // jiao
   sortTable[0x9278]=11040;   // jiao
   sortTable[0x9BAB]=11041;   // jiao
   sortTable[0x9C4E]=11042;   // jiao
   sortTable[0x9DCD]=11043;   // jiao
   sortTable[0x9636]=11044;   // jie
   sortTable[0x968E]=11045;   // jie
   sortTable[0x89E3]=11046;   // jie
   sortTable[0x7ED3]=11047;   // jie
   sortTable[0x7D50]=11048;   // jie
   sortTable[0x754C]=11049;   // jie
   sortTable[0x63A5]=11050;   // jie
   sortTable[0x8282]=11051;   // jie
   sortTable[0x7BC0]=11052;   // jie
   sortTable[0x8857]=11053;   // jie
   sortTable[0x59D0]=11054;   // jie
   sortTable[0x501F]=11055;   // jie
   sortTable[0x4ECB]=11056;   // jie
   sortTable[0x63ED]=11057;   // jie
   sortTable[0x6D01]=11058;   // jie
   sortTable[0x6F54]=11059;   // jie
   sortTable[0x5C4A]=11060;   // jie
   sortTable[0x5C46]=11061;   // jie
   sortTable[0x622A]=11062;   // jie
   sortTable[0x6212]=11063;   // jie
   sortTable[0x7AED]=11064;   // jie
   sortTable[0x6770]=11065;   // jie
   sortTable[0x6377]=11066;   // jie
   sortTable[0x776B]=11067;   // jie
   sortTable[0x7686]=11068;   // jie
   sortTable[0x8BEB]=11069;   // jie
   sortTable[0x8AA1]=11070;   // jie
   sortTable[0x52AB]=11071;   // jie
   sortTable[0x55DF]=11072;   // jie
   sortTable[0x9889]=11073;   // jie
   sortTable[0x79F8]=11074;   // jie
   sortTable[0x6840]=11075;   // jie
   sortTable[0x62EE]=11076;   // jie
   sortTable[0x5A55]=11077;   // jie
   sortTable[0x7FAF]=11078;   // jie
   sortTable[0x82A5]=11079;   // jie
   sortTable[0x86A7]=11080;   // jie
   sortTable[0x8BA6]=11081;   // jie
   sortTable[0x78A3]=11082;   // jie
   sortTable[0x5B51]=11083;   // jie
   sortTable[0x9AB1]=11084;   // jie
   sortTable[0x7D5C]=11085;   // jie
   sortTable[0x75A5]=11086;   // jie
   sortTable[0x5588]=11087;   // jie
   sortTable[0x7596]=11088;   // jie
   sortTable[0x9C92]=11089;   // jie
   sortTable[0x813B]=11090;   // jie
   sortTable[0x7A2D]=11091;   // jie
   sortTable[0x8A70]=11092;   // jie
   sortTable[0x84F5]=11093;   // jie
   sortTable[0x892F]=11094;   // jie
   sortTable[0x874D]=11095;   // jie
   sortTable[0x98F7]=11096;   // jie
   sortTable[0x86F6]=11097;   // jie
   sortTable[0x8710]=11098;   // jie
   sortTable[0x881E]=11099;   // jie
   sortTable[0x8871]=11100;   // jie
   sortTable[0x978A]=11101;   // jie
   sortTable[0x9B6A]=11102;   // jie
   sortTable[0x780E]=11103;   // jie
   sortTable[0x83E8]=11104;   // jie
   sortTable[0x9B9A]=11105;   // jie
   sortTable[0x9D9B]=11106;   // jie
   sortTable[0x88BA]=11107;   // jie
   sortTable[0x8A10]=11108;   // jie
   sortTable[0x8B2F]=11109;   // jie
   sortTable[0x8E15]=11110;   // jie
   sortTable[0x8FFC]=11111;   // jie
   sortTable[0x9B5D]=11112;   // jie
   sortTable[0x8878]=11113;   // jie
   sortTable[0x4E2F]=11114;   // jie
   sortTable[0x5022]=11115;   // jie
   sortTable[0x507C]=11116;   // jie
   sortTable[0x5091]=11117;   // jie
   sortTable[0x5226]=11118;   // jie
   sortTable[0x5227]=11119;   // jie
   sortTable[0x523C]=11120;   // jie
   sortTable[0x52BC]=11121;   // jie
   sortTable[0x5369]=11122;   // jie
   sortTable[0x536A]=11123;   // jie
   sortTable[0x5424]=11124;   // jie
   sortTable[0x55BC]=11125;   // jie
   sortTable[0x5826]=11126;   // jie
   sortTable[0x583A]=11127;   // jie
   sortTable[0x5A8E]=11128;   // jie
   sortTable[0x5A98]=11129;   // jie
   sortTable[0x5AC5]=11130;   // jie
   sortTable[0x5C10]=11131;   // jie
   sortTable[0x5C8A]=11132;   // jie
   sortTable[0x5C95]=11133;   // jie
   sortTable[0x5D28]=11134;   // jie
   sortTable[0x5D65]=11135;   // jie
   sortTable[0x5DC0]=11136;   // jie
   sortTable[0x5E6F]=11137;   // jie
   sortTable[0x5E8E]=11138;   // jie
   sortTable[0x5FA3]=11139;   // jie
   sortTable[0x6088]=11140;   // jie
   sortTable[0x63B2]=11141;   // jie
   sortTable[0x64D1]=11142;   // jie
   sortTable[0x6605]=11143;   // jie
   sortTable[0x685D]=11144;   // jie
   sortTable[0x6904]=11145;   // jie
   sortTable[0x6950]=11146;   // jie
   sortTable[0x696C]=11147;   // jie
   sortTable[0x6976]=11148;   // jie
   sortTable[0x69A4]=11149;   // jie
   sortTable[0x6A9E]=11150;   // jie
   sortTable[0x6AED]=11151;   // jie
   sortTable[0x6BD1]=11152;   // jie
   sortTable[0x6E5D]=11153;   // jie
   sortTable[0x6ED0]=11154;   // jie
   sortTable[0x716F]=11155;   // jie
   sortTable[0x7297]=11156;   // jie
   sortTable[0x73A0]=11157;   // jie
   sortTable[0x743E]=11158;   // jie
   sortTable[0x754D]=11159;   // jie
   sortTable[0x758C]=11160;   // jie
   sortTable[0x75CE]=11161;   // jie
   sortTable[0x7664]=11162;   // jie
   sortTable[0x83AD]=11163;   // jie
   sortTable[0x8754]=11164;   // jie
   sortTable[0x8818]=11165;   // jie
   sortTable[0x883D]=11166;   // jie
   sortTable[0x89E7]=11167;   // jie
   sortTable[0x8AB1]=11168;   // jie
   sortTable[0x8EA4]=11169;   // jie
   sortTable[0x9263]=11170;   // jie
   sortTable[0x937B]=11171;   // jie
   sortTable[0x9385]=11172;   // jie
   sortTable[0x8FDB]=11173;   // jin
   sortTable[0x9032]=11174;   // jin
   sortTable[0x4ECA]=11175;   // jin
   sortTable[0x7D27]=11176;   // jin
   sortTable[0x7DCA]=11177;   // jin
   sortTable[0x8FD1]=11178;   // jin
   sortTable[0x91D1]=11179;   // jin
   sortTable[0x4EC5]=11180;   // jin
   sortTable[0x50C5]=11181;   // jin
   sortTable[0x65A4]=11182;   // jin
   sortTable[0x52B2]=11183;   // jin
   sortTable[0x52C1]=11184;   // jin
   sortTable[0x5C3D]=11185;   // jin
   sortTable[0x76E1]=11186;   // jin
   sortTable[0x5DFE]=11187;   // jin
   sortTable[0x7981]=11188;   // jin
   sortTable[0x7B4B]=11189;   // jin
   sortTable[0x6D78]=11190;   // jin
   sortTable[0x9526]=11191;   // jin
   sortTable[0x9326]=11192;   // jin
   sortTable[0x6D25]=11193;   // jin
   sortTable[0x8C28]=11194;   // jin
   sortTable[0x8B39]=11195;   // jin
   sortTable[0x895F]=11196;   // jin
   sortTable[0x664B]=11197;   // jin
   sortTable[0x8369]=11198;   // jin
   sortTable[0x77DC]=11199;   // jin
   sortTable[0x747E]=11200;   // jin
   sortTable[0x89D0]=11201;   // jin
   sortTable[0x7F19]=11202;   // jin
   sortTable[0x5997]=11203;   // jin
   sortTable[0x70EC]=11204;   // jin
   sortTable[0x9991]=11205;   // jin
   sortTable[0x9773]=11206;   // jin
   sortTable[0x887F]=11207;   // jin
   sortTable[0x9485]=11208;   // jin
   sortTable[0x5664]=11209;   // jin
   sortTable[0x5807]=11210;   // jin
   sortTable[0x69FF]=11211;   // jin
   sortTable[0x537A]=11212;   // jin
   sortTable[0x8D46]=11213;   // jin
   sortTable[0x7972]=11214;   // jin
   sortTable[0x5ED1]=11215;   // jin
   sortTable[0x89D4]=11216;   // jin
   sortTable[0x7D1F]=11217;   // jin
   sortTable[0x83EB]=11218;   // jin
   sortTable[0x9EC5]=11219;   // jin
   sortTable[0x9F7D]=11220;   // jin
   sortTable[0x8CEE]=11221;   // jin
   sortTable[0x84F3]=11222;   // jin
   sortTable[0x91FF]=11223;   // jin
   sortTable[0x781B]=11224;   // jin
   sortTable[0x4F12]=11225;   // jin
   sortTable[0x4FAD]=11226;   // jin
   sortTable[0x50F8]=11227;   // jin
   sortTable[0x5118]=11228;   // jin
   sortTable[0x5153]=11229;   // jin
   sortTable[0x51DA]=11230;   // jin
   sortTable[0x52A4]=11231;   // jin
   sortTable[0x53AA]=11232;   // jin
   sortTable[0x552B]=11233;   // jin
   sortTable[0x568D]=11234;   // jin
   sortTable[0x57D0]=11235;   // jin
   sortTable[0x583B]=11236;   // jin
   sortTable[0x5890]=11237;   // jin
   sortTable[0x58D7]=11238;   // jin
   sortTable[0x5AE4]=11239;   // jin
   sortTable[0x5B27]=11240;   // jin
   sortTable[0x5BD6]=11241;   // jin
   sortTable[0x5D9C]=11242;   // jin
   sortTable[0x5DF9]=11243;   // jin
   sortTable[0x60CD]=11244;   // jin
   sortTable[0x6422]=11245;   // jin
   sortTable[0x6649]=11246;   // jin
   sortTable[0x669C]=11247;   // jin
   sortTable[0x6783]=11248;   // jin
   sortTable[0x6B4F]=11249;   // jin
   sortTable[0x6BA3]=11250;   // jin
   sortTable[0x6D55]=11251;   // jin
   sortTable[0x6E8D]=11252;   // jin
   sortTable[0x6F0C]=11253;   // jin
   sortTable[0x6FC5]=11254;   // jin
   sortTable[0x6FDC]=11255;   // jin
   sortTable[0x71FC]=11256;   // jin
   sortTable[0x73D2]=11257;   // jin
   sortTable[0x740E]=11258;   // jin
   sortTable[0x743B]=11259;   // jin
   sortTable[0x7468]=11260;   // jin
   sortTable[0x74A1]=11261;   // jin
   sortTable[0x74B6]=11262;   // jin
   sortTable[0x7E09]=11263;   // jin
   sortTable[0x8355]=11264;   // jin
   sortTable[0x85CE]=11265;   // jin
   sortTable[0x89B2]=11266;   // jin
   sortTable[0x8D10]=11267;   // jin
   sortTable[0x91D2]=11268;   // jin
   sortTable[0x9949]=11269;   // jin
   sortTable[0x9E76]=11270;   // jin
   sortTable[0x7ECF]=11271;   // jing
   sortTable[0x7D93]=11272;   // jing
   sortTable[0x775B]=11273;   // jing
   sortTable[0x7CBE]=11274;   // jing
   sortTable[0x9759]=11275;   // jing
   sortTable[0x975C]=11276;   // jing
   sortTable[0x60CA]=11277;   // jing
   sortTable[0x9A5A]=11278;   // jing
   sortTable[0x7ADF]=11279;   // jing
   sortTable[0x666F]=11280;   // jing
   sortTable[0x955C]=11281;   // jing
   sortTable[0x93E1]=11282;   // jing
   sortTable[0x5883]=11283;   // jing
   sortTable[0x8B66]=11284;   // jing
   sortTable[0x51C0]=11285;   // jing
   sortTable[0x6DE8]=11286;   // jing
   sortTable[0x4E95]=11287;   // jing
   sortTable[0x656C]=11288;   // jing
   sortTable[0x7ADE]=11289;   // jing
   sortTable[0x7AF6]=11290;   // jing
   sortTable[0x5F84]=11291;   // jing
   sortTable[0x5F91]=11292;   // jing
   sortTable[0x9CB8]=11293;   // jing
   sortTable[0x9BE8]=11294;   // jing
   sortTable[0x6676]=11295;   // jing
   sortTable[0x8148]=11296;   // jing
   sortTable[0x830E]=11297;   // jing
   sortTable[0x8396]=11298;   // jing
   sortTable[0x9888]=11299;   // jing
   sortTable[0x9838]=11300;   // jing
   sortTable[0x61AC]=11301;   // jing
   sortTable[0x4EAC]=11302;   // jing
   sortTable[0x8346]=11303;   // jing
   sortTable[0x9756]=11304;   // jing
   sortTable[0x65CC]=11305;   // jing
   sortTable[0x6CFE]=11306;   // jing
   sortTable[0x5162]=11307;   // jing
   sortTable[0x8FF3]=11308;   // jing
   sortTable[0x80EB]=11309;   // jing
   sortTable[0x83C1]=11310;   // jing
   sortTable[0x9631]=11311;   // jing
   sortTable[0x5106]=11312;   // jing
   sortTable[0x75C9]=11313;   // jing
   sortTable[0x9753]=11314;   // jing
   sortTable[0x80BC]=11315;   // jing
   sortTable[0x734D]=11316;   // jing
   sortTable[0x5A67]=11317;   // jing
   sortTable[0x7CB3]=11318;   // jing
   sortTable[0x522D]=11319;   // jing
   sortTable[0x9E96]=11320;   // jing
   sortTable[0x79D4]=11321;   // jing
   sortTable[0x9D84]=11322;   // jing
   sortTable[0x9F31]=11323;   // jing
   sortTable[0x7A7D]=11324;   // jing
   sortTable[0x7AEB]=11325;   // jing
   sortTable[0x811B]=11326;   // jing
   sortTable[0x5F2A]=11327;   // jing
   sortTable[0x7A09]=11328;   // jing
   sortTable[0x9015]=11329;   // jing
   sortTable[0x7AF8]=11330;   // jing
   sortTable[0x834A]=11331;   // jing
   sortTable[0x87FC]=11332;   // jing
   sortTable[0x844F]=11333;   // jing
   sortTable[0x981A]=11334;   // jing
   sortTable[0x9D5B]=11335;   // jing
   sortTable[0x7AE7]=11336;   // jing
   sortTable[0x4E3C]=11337;   // jing
   sortTable[0x4EB0]=11338;   // jing
   sortTable[0x4FD3]=11339;   // jing
   sortTable[0x501E]=11340;   // jing
   sortTable[0x50B9]=11341;   // jing
   sortTable[0x51C8]=11342;   // jing
   sortTable[0x5244]=11343;   // jing
   sortTable[0x5753]=11344;   // jing
   sortTable[0x5755]=11345;   // jing
   sortTable[0x5759]=11346;   // jing
   sortTable[0x598C]=11347;   // jing
   sortTable[0x5A59]=11348;   // jing
   sortTable[0x5A5B]=11349;   // jing
   sortTable[0x5B91]=11350;   // jing
   sortTable[0x5DE0]=11351;   // jing
   sortTable[0x5E5C]=11352;   // jing
   sortTable[0x5F33]=11353;   // jing
   sortTable[0x61BC]=11354;   // jing
   sortTable[0x65CD]=11355;   // jing
   sortTable[0x66BB]=11356;   // jing
   sortTable[0x66D4]=11357;   // jing
   sortTable[0x6871]=11358;   // jing
   sortTable[0x68B7]=11359;   // jing
   sortTable[0x6A78]=11360;   // jing
   sortTable[0x6C6B]=11361;   // jing
   sortTable[0x6C6C]=11362;   // jing
   sortTable[0x6D44]=11363;   // jing
   sortTable[0x6D87]=11364;   // jing
   sortTable[0x6FEA]=11365;   // jing
   sortTable[0x701E]=11366;   // jing
   sortTable[0x71DB]=11367;   // jing
   sortTable[0x71DD]=11368;   // jing
   sortTable[0x7304]=11369;   // jing
   sortTable[0x749F]=11370;   // jing
   sortTable[0x74A5]=11371;   // jing
   sortTable[0x75D9]=11372;   // jing
   sortTable[0x7D4C]=11373;   // jing
   sortTable[0x8059]=11374;   // jing
   sortTable[0x8AA9]=11375;   // jing
   sortTable[0x8E01]=11376;   // jing
   sortTable[0x975A]=11377;   // jing
   sortTable[0x9D81]=11378;   // jing
   sortTable[0x9EA0]=11379;   // jing
   sortTable[0x70AF]=11380;   // jiong
   sortTable[0x8FE5]=11381;   // jiong
   sortTable[0x7A98]=11382;   // jiong
   sortTable[0x7085]=11383;   // jiong
   sortTable[0x988E]=11384;   // jiong
   sortTable[0x6243]=11385;   // jiong
   sortTable[0x8927]=11386;   // jiong
   sortTable[0x99C9]=11387;   // jiong
   sortTable[0x7D45]=11388;   // jiong
   sortTable[0x9008]=11389;   // jiong
   sortTable[0x99EB]=11390;   // jiong
   sortTable[0x7D97]=11391;   // jiong
   sortTable[0x8614]=11392;   // jiong
   sortTable[0x4FB0]=11393;   // jiong
   sortTable[0x50D2]=11394;   // jiong
   sortTable[0x5182]=11395;   // jiong
   sortTable[0x518B]=11396;   // jiong
   sortTable[0x518F]=11397;   // jiong
   sortTable[0x56E7]=11398;   // jiong
   sortTable[0x5770]=11399;   // jiong
   sortTable[0x57DB]=11400;   // jiong
   sortTable[0x6CC2]=11401;   // jiong
   sortTable[0x6D7B]=11402;   // jiong
   sortTable[0x6F83]=11403;   // jiong
   sortTable[0x70F1]=11404;   // jiong
   sortTable[0x715A]=11405;   // jiong
   sortTable[0x715B]=11406;   // jiong
   sortTable[0x71B2]=11407;   // jiong
   sortTable[0x860F]=11408;   // jiong
   sortTable[0x5C31]=11409;   // jiu
   sortTable[0x4E5D]=11410;   // jiu
   sortTable[0x7A76]=11411;   // jiu
   sortTable[0x65E7]=11412;   // jiu
   sortTable[0x820A]=11413;   // jiu
   sortTable[0x4E45]=11414;   // jiu
   sortTable[0x9152]=11415;   // jiu
   sortTable[0x6551]=11416;   // jiu
   sortTable[0x7EA0]=11417;   // jiu
   sortTable[0x7CFE]=11418;   // jiu
   sortTable[0x8205]=11419;   // jiu
   sortTable[0x63EA]=11420;   // jiu
   sortTable[0x557E]=11421;   // jiu
   sortTable[0x548E]=11422;   // jiu
   sortTable[0x759A]=11423;   // jiu
   sortTable[0x9E20]=11424;   // jiu
   sortTable[0x67E9]=11425;   // jiu
   sortTable[0x97ED]=11426;   // jiu
   sortTable[0x81FC]=11427;   // jiu
   sortTable[0x53A9]=11428;   // jiu
   sortTable[0x50E6]=11429;   // jiu
   sortTable[0x9604]=11430;   // jiu
   sortTable[0x7078]=11431;   // jiu
   sortTable[0x9E6B]=11432;   // jiu
   sortTable[0x7396]=11433;   // jiu
   sortTable[0x8D73]=11434;   // jiu
   sortTable[0x6855]=11435;   // jiu
   sortTable[0x7CFA]=11436;   // jiu
   sortTable[0x97EE]=11437;   // jiu
   sortTable[0x9B0F]=11438;   // jiu
   sortTable[0x7D24]=11439;   // jiu
   sortTable[0x9DF2]=11440;   // jiu
   sortTable[0x9BE6]=11441;   // jiu
   sortTable[0x9F68]=11442;   // jiu
   sortTable[0x9CE9]=11443;   // jiu
   sortTable[0x9E94]=11444;   // jiu
   sortTable[0x4E29]=11445;   // jiu
   sortTable[0x4E46]=11446;   // jiu
   sortTable[0x4E63]=11447;   // jiu
   sortTable[0x5003]=11448;   // jiu
   sortTable[0x52FC]=11449;   // jiu
   sortTable[0x5313]=11450;   // jiu
   sortTable[0x531B]=11451;   // jiu
   sortTable[0x5336]=11452;   // jiu
   sortTable[0x597A]=11453;   // jiu
   sortTable[0x5AA8]=11454;   // jiu
   sortTable[0x5EC4]=11455;   // jiu
   sortTable[0x5ECF]=11456;   // jiu
   sortTable[0x5ED0]=11457;   // jiu
   sortTable[0x6166]=11458;   // jiu
   sortTable[0x6344]=11459;   // jiu
   sortTable[0x63C2]=11460;   // jiu
   sortTable[0x63EB]=11461;   // jiu
   sortTable[0x644E]=11462;   // jiu
   sortTable[0x673B]=11463;   // jiu
   sortTable[0x6766]=11464;   // jiu
   sortTable[0x67FE]=11465;   // jiu
   sortTable[0x6A1B]=11466;   // jiu
   sortTable[0x6B0D]=11467;   // jiu
   sortTable[0x6BA7]=11468;   // jiu
   sortTable[0x6C63]=11469;   // jiu
   sortTable[0x725E]=11470;   // jiu
   sortTable[0x820F]=11471;   // jiu
   sortTable[0x841B]=11472;   // jiu
   sortTable[0x9579]=11473;   // jiu
   sortTable[0x9B2E]=11474;   // jiu
   sortTable[0x636E]=11475;   // ju
   sortTable[0x64DA]=11476;   // ju
   sortTable[0x5177]=11477;   // ju
   sortTable[0x53E5]=11478;   // ju
   sortTable[0x4E3E]=11479;   // ju
   sortTable[0x8209]=11480;   // ju
   sortTable[0x5C40]=11481;   // ju
   sortTable[0x5DE8]=11482;   // ju
   sortTable[0x5C45]=11483;   // ju
   sortTable[0x8DDD]=11484;   // ju
   sortTable[0x5267]=11485;   // ju
   sortTable[0x5287]=11486;   // ju
   sortTable[0x805A]=11487;   // ju
   sortTable[0x62D2]=11488;   // ju
   sortTable[0x6A58]=11489;   // ju
   sortTable[0x77E9]=11490;   // ju
   sortTable[0x97A0]=11491;   // ju
   sortTable[0x4FF1]=11492;   // ju
   sortTable[0x62D8]=11493;   // ju
   sortTable[0x60E7]=11494;   // ju
   sortTable[0x61FC]=11495;   // ju
   sortTable[0x5480]=11496;   // ju
   sortTable[0x952F]=11497;   // ju
   sortTable[0x92F8]=11498;   // ju
   sortTable[0x83CA]=11499;   // ju
   sortTable[0x6854]=11500;   // ju
   sortTable[0x9A79]=11501;   // ju
   sortTable[0x99D2]=11502;   // ju
   sortTable[0x6CAE]=11503;   // ju
   sortTable[0x907D]=11504;   // ju
   sortTable[0x97AB]=11505;   // ju
   sortTable[0x949C]=11506;   // ju
   sortTable[0x83F9]=11507;   // ju
   sortTable[0x70AC]=11508;   // ju
   sortTable[0x8E1E]=11509;   // ju
   sortTable[0x75BD]=11510;   // ju
   sortTable[0x8392]=11511;   // ju
   sortTable[0x72D9]=11512;   // ju
   sortTable[0x5028]=11513;   // ju
   sortTable[0x88FE]=11514;   // ju
   sortTable[0x63AC]=11515;   // ju
   sortTable[0x98D3]=11516;   // ju
   sortTable[0x82F4]=11517;   // ju
   sortTable[0x8BB5]=11518;   // ju
   sortTable[0x91B5]=11519;   // ju
   sortTable[0x9F83]=11520;   // ju
   sortTable[0x96CE]=11521;   // ju
   sortTable[0x7AAD]=11522;   // ju
   sortTable[0x82E3]=11523;   // ju
   sortTable[0x6998]=11524;   // ju
   sortTable[0x741A]=11525;   // ju
   sortTable[0x6989]=11526;   // ju
   sortTable[0x5C66]=11527;   // ju
   sortTable[0x9514]=11528;   // ju
   sortTable[0x8D84]=11529;   // ju
   sortTable[0x8661]=11530;   // ju
   sortTable[0x8E3D]=11531;   // ju
   sortTable[0x79EC]=11532;   // ju
   sortTable[0x9B88]=11533;   // ju
   sortTable[0x7F5D]=11534;   // ju
   sortTable[0x8DFC]=11535;   // ju
   sortTable[0x7B65]=11536;   // ju
   sortTable[0x824D]=11537;   // ju
   sortTable[0x8E18]=11538;   // ju
   sortTable[0x9B94]=11539;   // ju
   sortTable[0x943B]=11540;   // ju
   sortTable[0x8893]=11541;   // ju
   sortTable[0x6910]=11542;   // ju
   sortTable[0x8152]=11543;   // ju
   sortTable[0x849F]=11544;   // ju
   sortTable[0x8C97]=11545;   // ju
   sortTable[0x7C34]=11546;   // ju
   sortTable[0x9245]=11547;   // ju
   sortTable[0x7CB7]=11548;   // ju
   sortTable[0x7C94]=11549;   // ju
   sortTable[0x95B0]=11550;   // ju
   sortTable[0x9D59]=11551;   // ju
   sortTable[0x728B]=11552;   // ju
   sortTable[0x8F02]=11553;   // ju
   sortTable[0x9D8B]=11554;   // ju
   sortTable[0x8DD4]=11555;   // ju
   sortTable[0x8C66]=11556;   // ju
   sortTable[0x99F6]=11557;   // ju
   sortTable[0x7820]=11558;   // ju
   sortTable[0x9D74]=11559;   // ju
   sortTable[0x8065]=11560;   // ju
   sortTable[0x861C]=11561;   // ju
   sortTable[0x8E86]=11562;   // ju
   sortTable[0x99CF]=11563;   // ju
   sortTable[0x9D21]=11564;   // ju
   sortTable[0x801F]=11565;   // ju
   sortTable[0x8ACA]=11566;   // ju
   sortTable[0x8DD9]=11567;   // ju
   sortTable[0x8E6B]=11568;   // ju
   sortTable[0x9671]=11569;   // ju
   sortTable[0x9F33]=11570;   // ju
   sortTable[0x4E6C]=11571;   // ju
   sortTable[0x4FB7]=11572;   // ju
   sortTable[0x5036]=11573;   // ju
   sortTable[0x50EA]=11574;   // ju
   sortTable[0x51A3]=11575;   // ju
   sortTable[0x51E5]=11576;   // ju
   sortTable[0x521F]=11577;   // ju
   sortTable[0x52EE]=11578;   // ju
   sortTable[0x530A]=11579;   // ju
   sortTable[0x5579]=11580;   // ju
   sortTable[0x57E7]=11581;   // ju
   sortTable[0x57FE]=11582;   // ju
   sortTable[0x58C9]=11583;   // ju
   sortTable[0x59D6]=11584;   // ju
   sortTable[0x5A35]=11585;   // ju
   sortTable[0x5A45]=11586;   // ju
   sortTable[0x5A6E]=11587;   // ju
   sortTable[0x5BE0]=11588;   // ju
   sortTable[0x5C68]=11589;   // ju
   sortTable[0x5CA0]=11590;   // ju
   sortTable[0x5D0C]=11591;   // ju
   sortTable[0x5DC8]=11592;   // ju
   sortTable[0x5DEA]=11593;   // ju
   sortTable[0x5F06]=11594;   // ju
   sortTable[0x6007]=11595;   // ju
   sortTable[0x6010]=11596;   // ju
   sortTable[0x601A]=11597;   // ju
   sortTable[0x6133]=11598;   // ju
   sortTable[0x61C5]=11599;   // ju
   sortTable[0x6285]=11600;   // ju
   sortTable[0x62E0]=11601;   // ju
   sortTable[0x6319]=11602;   // ju
   sortTable[0x6336]=11603;   // ju
   sortTable[0x64E7]=11604;   // ju
   sortTable[0x661B]=11605;   // ju
   sortTable[0x68AE]=11606;   // ju
   sortTable[0x6907]=11607;   // ju
   sortTable[0x6908]=11608;   // ju
   sortTable[0x6A8B]=11609;   // ju
   sortTable[0x6AF8]=11610;   // ju
   sortTable[0x6B05]=11611;   // ju
   sortTable[0x6B6B]=11612;   // ju
   sortTable[0x6BE9]=11613;   // ju
   sortTable[0x6BF1]=11614;   // ju
   sortTable[0x6CC3]=11615;   // ju
   sortTable[0x6CE6]=11616;   // ju
   sortTable[0x6D30]=11617;   // ju
   sortTable[0x6DBA]=11618;   // ju
   sortTable[0x6DD7]=11619;   // ju
   sortTable[0x6E68]=11620;   // ju
   sortTable[0x6FBD]=11621;   // ju
   sortTable[0x7117]=11622;   // ju
   sortTable[0x7220]=11623;   // ju
   sortTable[0x7291]=11624;   // ju
   sortTable[0x72CA]=11625;   // ju
   sortTable[0x75C0]=11626;   // ju
   sortTable[0x7717]=11627;   // ju
   sortTable[0x7AB6]=11628;   // ju
   sortTable[0x86B7]=11629;   // ju
   sortTable[0x871B]=11630;   // ju
   sortTable[0x8977]=11631;   // ju
   sortTable[0x8A4E]=11632;   // ju
   sortTable[0x8D9C]=11633;   // ju
   sortTable[0x8E19]=11634;   // ju
   sortTable[0x8EB9]=11635;   // ju
   sortTable[0x90AD]=11636;   // ju
   sortTable[0x90F9]=11637;   // ju
   sortTable[0x92E6]=11638;   // ju
   sortTable[0x98B6]=11639;   // ju
   sortTable[0x9A67]=11640;   // ju
   sortTable[0x9DAA]=11641;   // ju
   sortTable[0x9F5F]=11642;   // ju
   sortTable[0x5377]=11643;   // juan
   sortTable[0x5026]=11644;   // juan
   sortTable[0x7EE2]=11645;   // juan
   sortTable[0x7D79]=11646;   // juan
   sortTable[0x9E43]=11647;   // juan
   sortTable[0x9D51]=11648;   // juan
   sortTable[0x6350]=11649;   // juan
   sortTable[0x7737]=11650;   // juan
   sortTable[0x5A1F]=11651;   // juan
   sortTable[0x8832]=11652;   // juan
   sortTable[0x6D93]=11653;   // juan
   sortTable[0x954C]=11654;   // juan
   sortTable[0x96BD]=11655;   // juan
   sortTable[0x9104]=11656;   // juan
   sortTable[0x9529]=11657;   // juan
   sortTable[0x72F7]=11658;   // juan
   sortTable[0x7760]=11659;   // juan
   sortTable[0x7F65]=11660;   // juan
   sortTable[0x774A]=11661;   // juan
   sortTable[0x684A]=11662;   // juan
   sortTable[0x81C7]=11663;   // juan
   sortTable[0x8127]=11664;   // juan
   sortTable[0x7E33]=11665;   // juan
   sortTable[0x83E4]=11666;   // juan
   sortTable[0x9308]=11667;   // juan
   sortTable[0x7F82]=11668;   // juan
   sortTable[0x942B]=11669;   // juan
   sortTable[0x96CB]=11670;   // juan
   sortTable[0x7D6D]=11671;   // juan
   sortTable[0x8528]=11672;   // juan
   sortTable[0x93B8]=11673;   // juan
   sortTable[0x52B5]=11674;   // juan
   sortTable[0x52CC]=11675;   // juan
   sortTable[0x52EC]=11676;   // juan
   sortTable[0x545F]=11677;   // juan
   sortTable[0x57CD]=11678;   // juan
   sortTable[0x5946]=11679;   // juan
   sortTable[0x59E2]=11680;   // juan
   sortTable[0x5DFB]=11681;   // juan
   sortTable[0x5E23]=11682;   // juan
   sortTable[0x617B]=11683;   // juan
   sortTable[0x6372]=11684;   // juan
   sortTable[0x6DC3]=11685;   // juan
   sortTable[0x7106]=11686;   // juan
   sortTable[0x7367]=11687;   // juan
   sortTable[0x74F9]=11688;   // juan
   sortTable[0x88D0]=11689;   // juan
   sortTable[0x98EC]=11690;   // juan
   sortTable[0x990B]=11691;   // juan
   sortTable[0x51B3]=11692;   // jue
   sortTable[0x6C7A]=11693;   // jue
   sortTable[0x89C9]=11694;   // jue
   sortTable[0x89BA]=11695;   // jue
   sortTable[0x7EDD]=11696;   // jue
   sortTable[0x7D55]=11697;   // jue
   sortTable[0x6485]=11698;   // jue
   sortTable[0x56BC]=11699;   // jue
   sortTable[0x6398]=11700;   // jue
   sortTable[0x7235]=11701;   // jue
   sortTable[0x53A5]=11702;   // jue
   sortTable[0x8BC0]=11703;   // jue
   sortTable[0x5D1B]=11704;   // jue
   sortTable[0x7357]=11705;   // jue
   sortTable[0x5014]=11706;   // jue
   sortTable[0x6A5B]=11707;   // jue
   sortTable[0x6289]=11708;   // jue
   sortTable[0x652B]=11709;   // jue
   sortTable[0x8E76]=11710;   // jue
   sortTable[0x73CF]=11711;   // jue
   sortTable[0x77CD]=11712;   // jue
   sortTable[0x8C32]=11713;   // jue
   sortTable[0x8568]=11714;   // jue
   sortTable[0x5658]=11715;   // jue
   sortTable[0x5671]=11716;   // jue
   sortTable[0x9562]=11717;   // jue
   sortTable[0x6877]=11718;   // jue
   sortTable[0x89D6]=11719;   // jue
   sortTable[0x8E7B]=11720;   // jue
   sortTable[0x5282]=11721;   // jue
   sortTable[0x721D]=11722;   // jue
   sortTable[0x940D]=11723;   // jue
   sortTable[0x5B53]=11724;   // jue
   sortTable[0x99C3]=11725;   // jue
   sortTable[0x855D]=11726;   // jue
   sortTable[0x9D02]=11727;   // jue
   sortTable[0x9481]=11728;   // jue
   sortTable[0x89FC]=11729;   // jue
   sortTable[0x8DB9]=11730;   // jue
   sortTable[0x81C4]=11731;   // jue
   sortTable[0x9D03]=11732;   // jue
   sortTable[0x8C9C]=11733;   // jue
   sortTable[0x8E77]=11734;   // jue
   sortTable[0x8B4E]=11735;   // jue
   sortTable[0x8EA9]=11736;   // jue
   sortTable[0x87E8]=11737;   // jue
   sortTable[0x77E1]=11738;   // jue
   sortTable[0x82B5]=11739;   // jue
   sortTable[0x941D]=11740;   // jue
   sortTable[0x9D8C]=11741;   // jue
   sortTable[0x9FA3]=11742;   // jue
   sortTable[0x87E9]=11743;   // jue
   sortTable[0x8990]=11744;   // jue
   sortTable[0x899A]=11745;   // jue
   sortTable[0x8A23]=11746;   // jue
   sortTable[0x8D89]=11747;   // jue
   sortTable[0x920C]=11748;   // jue
   sortTable[0x7D76]=11749;   // jue
   sortTable[0x8D7D]=11750;   // jue
   sortTable[0x9DE2]=11751;   // jue
   sortTable[0x4E85]=11752;   // jue
   sortTable[0x5095]=11753;   // jue
   sortTable[0x5214]=11754;   // jue
   sortTable[0x52EA]=11755;   // jue
   sortTable[0x5337]=11756;   // jue
   sortTable[0x5B52]=11757;   // jue
   sortTable[0x5C69]=11758;   // jue
   sortTable[0x5C6B]=11759;   // jue
   sortTable[0x5DA5]=11760;   // jue
   sortTable[0x5F21]=11761;   // jue
   sortTable[0x5F4F]=11762;   // jue
   sortTable[0x61A0]=11763;   // jue
   sortTable[0x61B0]=11764;   // jue
   sortTable[0x6204]=11765;   // jue
   sortTable[0x6317]=11766;   // jue
   sortTable[0x6354]=11767;   // jue
   sortTable[0x64A7]=11768;   // jue
   sortTable[0x658D]=11769;   // jue
   sortTable[0x6A5C]=11770;   // jue
   sortTable[0x6B14]=11771;   // jue
   sortTable[0x6B2E]=11772;   // jue
   sortTable[0x6B8C]=11773;   // jue
   sortTable[0x6C12]=11774;   // jue
   sortTable[0x6CEC]=11775;   // jue
   sortTable[0x7133]=11776;   // jue
   sortTable[0x71A6]=11777;   // jue
   sortTable[0x7211]=11778;   // jue
   sortTable[0x7234]=11779;   // jue
   sortTable[0x7383]=11780;   // jue
   sortTable[0x73A6]=11781;   // jue
   sortTable[0x73A8]=11782;   // jue
   sortTable[0x7474]=11783;   // jue
   sortTable[0x75A6]=11784;   // jue
   sortTable[0x761A]=11785;   // jue
   sortTable[0x7804]=11786;   // jue
   sortTable[0x8673]=11787;   // jue
   sortTable[0x8697]=11788;   // jue
   sortTable[0x902B]=11789;   // jue
   sortTable[0x519B]=11790;   // jun
   sortTable[0x8ECD]=11791;   // jun
   sortTable[0x5747]=11792;   // jun
   sortTable[0x83CC]=11793;   // jun
   sortTable[0x541B]=11794;   // jun
   sortTable[0x5CFB]=11795;   // jun
   sortTable[0x90E1]=11796;   // jun
   sortTable[0x4FCA]=11797;   // jun
   sortTable[0x6D5A]=11798;   // jun
   sortTable[0x94A7]=11799;   // jun
   sortTable[0x9A8F]=11800;   // jun
   sortTable[0x7AE3]=11801;   // jun
   sortTable[0x9E87]=11802;   // jun
   sortTable[0x6343]=11803;   // jun
   sortTable[0x76B2]=11804;   // jun
   sortTable[0x9982]=11805;   // jun
   sortTable[0x9E95]=11806;   // jun
   sortTable[0x8880]=11807;   // jun
   sortTable[0x7B98]=11808;   // jun
   sortTable[0x9CAA]=11809;   // jun
   sortTable[0x89A0]=11810;   // jun
   sortTable[0x9E8F]=11811;   // jun
   sortTable[0x8399]=11812;   // jun
   sortTable[0x929E]=11813;   // jun
   sortTable[0x9D54]=11814;   // jun
   sortTable[0x7885]=11815;   // jun
   sortTable[0x9915]=11816;   // jun
   sortTable[0x8720]=11817;   // jun
   sortTable[0x9656]=11818;   // jun
   sortTable[0x5101]=11819;   // jun
   sortTable[0x5441]=11820;   // jun
   sortTable[0x57C8]=11821;   // jun
   sortTable[0x59F0]=11822;   // jun
   sortTable[0x5BEF]=11823;   // jun
   sortTable[0x61CF]=11824;   // jun
   sortTable[0x6508]=11825;   // jun
   sortTable[0x651F]=11826;   // jun
   sortTable[0x6659]=11827;   // jun
   sortTable[0x687E]=11828;   // jun
   sortTable[0x68DE]=11829;   // jun
   sortTable[0x6C6E]=11830;   // jun
   sortTable[0x6FEC]=11831;   // jun
   sortTable[0x710C]=11832;   // jun
   sortTable[0x71C7]=11833;   // jun
   sortTable[0x73FA]=11834;   // jun
   sortTable[0x756F]=11835;   // jun
   sortTable[0x76B8]=11836;   // jun
   sortTable[0x76B9]=11837;   // jun
   sortTable[0x7B9F]=11838;   // jun
   sortTable[0x8690]=11839;   // jun
   sortTable[0x921E]=11840;   // jun
   sortTable[0x9281]=11841;   // jun
   sortTable[0x9355]=11842;   // jun
   sortTable[0x99FF]=11843;   // jun
   sortTable[0x9BB6]=11844;   // jun
   sortTable[0x9D58]=11845;   // jun
   sortTable[0x5361]=12288;   // ka
   sortTable[0x5496]=12289;   // ka
   sortTable[0x5580]=12290;   // ka
   sortTable[0x5494]=12291;   // ka
   sortTable[0x80E9]=12292;   // ka
   sortTable[0x4F67]=12293;   // ka
   sortTable[0x8849]=12294;   // ka
   sortTable[0x57B0]=12295;   // ka
   sortTable[0x64D6]=12296;   // ka
   sortTable[0x88C3]=12297;   // ka
   sortTable[0x9272]=12298;   // ka
   sortTable[0x5F00]=12299;   // kai
   sortTable[0x958B]=12300;   // kai
   sortTable[0x6168]=12301;   // kai
   sortTable[0x63E9]=12302;   // kai
   sortTable[0x51EF]=12303;   // kai
   sortTable[0x6977]=12304;   // kai
   sortTable[0x607A]=12305;   // kai
   sortTable[0x94E0]=12306;   // kai
   sortTable[0x9534]=12307;   // kai
   sortTable[0x950E]=12308;   // kai
   sortTable[0x95FF]=12309;   // kai
   sortTable[0x5FFE]=12310;   // kai
   sortTable[0x5240]=12311;   // kai
   sortTable[0x57B2]=12312;   // kai
   sortTable[0x8488]=12313;   // kai
   sortTable[0x8F06]=12314;   // kai
   sortTable[0x98BD]=12315;   // kai
   sortTable[0x95D3]=12316;   // kai
   sortTable[0x9426]=12317;   // kai
   sortTable[0x51F1]=12318;   // kai
   sortTable[0x5274]=12319;   // kai
   sortTable[0x52D3]=12320;   // kai
   sortTable[0x5605]=12321;   // kai
   sortTable[0x584F]=12322;   // kai
   sortTable[0x5952]=12323;   // kai
   sortTable[0x5D66]=12324;   // kai
   sortTable[0x6112]=12325;   // kai
   sortTable[0x6137]=12326;   // kai
   sortTable[0x613E]=12327;   // kai
   sortTable[0x669F]=12328;   // kai
   sortTable[0x6B2C]=12329;   // kai
   sortTable[0x708C]=12330;   // kai
   sortTable[0x708F]=12331;   // kai
   sortTable[0x70D7]=12332;   // kai
   sortTable[0x9347]=12333;   // kai
   sortTable[0x938E]=12334;   // kai
   sortTable[0x93A7]=12335;   // kai
   sortTable[0x770B]=12336;   // kan
   sortTable[0x780D]=12337;   // kan
   sortTable[0x520A]=12338;   // kan
   sortTable[0x582A]=12339;   // kan
   sortTable[0x69DB]=12340;   // kan
   sortTable[0x6ABB]=12341;   // kan
   sortTable[0x52D8]=12342;   // kan
   sortTable[0x574E]=12343;   // kan
   sortTable[0x4F83]=12344;   // kan
   sortTable[0x9F9B]=12345;   // kan
   sortTable[0x77B0]=12346;   // kan
   sortTable[0x6221]=12347;   // kan
   sortTable[0x884E]=12348;   // kan
   sortTable[0x9851]=12349;   // kan
   sortTable[0x7AF7]=12350;   // kan
   sortTable[0x8F57]=12351;   // kan
   sortTable[0x78E1]=12352;   // kan
   sortTable[0x77D9]=12353;   // kan
   sortTable[0x9F95]=12354;   // kan
   sortTable[0x8F21]=12355;   // kan
   sortTable[0x5058]=12356;   // kan
   sortTable[0x519A]=12357;   // kan
   sortTable[0x57F3]=12358;   // kan
   sortTable[0x586A]=12359;   // kan
   sortTable[0x5888]=12360;   // kan
   sortTable[0x5D01]=12361;   // kan
   sortTable[0x5D41]=12362;   // kan
   sortTable[0x60C2]=12363;   // kan
   sortTable[0x681E]=12364;   // kan
   sortTable[0x6B3F]=12365;   // kan
   sortTable[0x6B41]=12366;   // kan
   sortTable[0x83B0]=12367;   // kan
   sortTable[0x95DE]=12368;   // kan
   sortTable[0x6297]=12369;   // kang
   sortTable[0x7095]=12370;   // kang
   sortTable[0x5EB7]=12371;   // kang
   sortTable[0x625B]=12372;   // kang
   sortTable[0x6177]=12373;   // kang
   sortTable[0x4EA2]=12374;   // kang
   sortTable[0x94AA]=12375;   // kang
   sortTable[0x7CE0]=12376;   // kang
   sortTable[0x95F6]=12377;   // kang
   sortTable[0x4F09]=12378;   // kang
   sortTable[0x9C47]=12379;   // kang
   sortTable[0x9227]=12380;   // kang
   sortTable[0x909F]=12381;   // kang
   sortTable[0x7A45]=12382;   // kang
   sortTable[0x7C87]=12383;   // kang
   sortTable[0x8EBF]=12384;   // kang
   sortTable[0x958C]=12385;   // kang
   sortTable[0x531F]=12386;   // kang
   sortTable[0x56E5]=12387;   // kang
   sortTable[0x5ADD]=12388;   // kang
   sortTable[0x5D7B]=12389;   // kang
   sortTable[0x5FFC]=12390;   // kang
   sortTable[0x6443]=12391;   // kang
   sortTable[0x69FA]=12392;   // kang
   sortTable[0x6F2E]=12393;   // kang
   sortTable[0x72BA]=12394;   // kang
   sortTable[0x780A]=12395;   // kang
   sortTable[0x93EE]=12396;   // kang
   sortTable[0x9760]=12397;   // kao
   sortTable[0x8003]=12398;   // kao
   sortTable[0x70E4]=12399;   // kao
   sortTable[0x62F7]=12400;   // kao
   sortTable[0x7292]=12401;   // kao
   sortTable[0x94D0]=12402;   // kao
   sortTable[0x5C3B]=12403;   // kao
   sortTable[0x6832]=12404;   // kao
   sortTable[0x9BCC]=12405;   // kao
   sortTable[0x9ADB]=12406;   // kao
   sortTable[0x7A01]=12407;   // kao
   sortTable[0x9C93]=12408;   // kao
   sortTable[0x4E02]=12409;   // kao
   sortTable[0x6537]=12410;   // kao
   sortTable[0x6D18]=12411;   // kao
   sortTable[0x71FA]=12412;   // kao
   sortTable[0x92AC]=12413;   // kao
   sortTable[0x9BB3]=12414;   // kao
   sortTable[0x53EF]=12415;   // ke
   sortTable[0x79D1]=12416;   // ke
   sortTable[0x523B]=12417;   // ke
   sortTable[0x5BA2]=12418;   // ke
   sortTable[0x8BFE]=12419;   // ke
   sortTable[0x8AB2]=12420;   // ke
   sortTable[0x9897]=12421;   // ke
   sortTable[0x9846]=12422;   // ke
   sortTable[0x514B]=12423;   // ke
   sortTable[0x68F5]=12424;   // ke
   sortTable[0x58F3]=12425;   // ke
   sortTable[0x6BBC]=12426;   // ke
   sortTable[0x6E34]=12427;   // ke
   sortTable[0x78D5]=12428;   // ke
   sortTable[0x778C]=12429;   // ke
   sortTable[0x874C]=12430;   // ke
   sortTable[0x67EF]=12431;   // ke
   sortTable[0x606A]=12432;   // ke
   sortTable[0x82DB]=12433;   // ke
   sortTable[0x73C2]=12434;   // ke
   sortTable[0x8F72]=12435;   // ke
   sortTable[0x7AA0]=12436;   // ke
   sortTable[0x5777]=12437;   // ke
   sortTable[0x55D1]=12438;   // ke
   sortTable[0x9AC1]=12439;   // ke
   sortTable[0x6C2A]=12440;   // ke
   sortTable[0x5CA2]=12441;   // ke
   sortTable[0x7F02]=12442;   // ke
   sortTable[0x951E]=12443;   // ke
   sortTable[0x988F]=12444;   // ke
   sortTable[0x75B4]=12445;   // ke
   sortTable[0x7A1E]=12446;   // ke
   sortTable[0x6E98]=12447;   // ke
   sortTable[0x9A92]=12448;   // ke
   sortTable[0x8DB7]=12449;   // ke
   sortTable[0x7822]=12450;   // ke
   sortTable[0x8250]=12451;   // ke
   sortTable[0x8596]=12452;   // ke
   sortTable[0x9A0D]=12453;   // ke
   sortTable[0x790A]=12454;   // ke
   sortTable[0x9233]=12455;   // ke
   sortTable[0x78A6]=12456;   // ke
   sortTable[0x7FD7]=12457;   // ke
   sortTable[0x8EFB]=12458;   // ke
   sortTable[0x524B]=12459;   // ke
   sortTable[0x52C0]=12460;   // ke
   sortTable[0x52CA]=12461;   // ke
   sortTable[0x533C]=12462;   // ke
   sortTable[0x5801]=12463;   // ke
   sortTable[0x5A14]=12464;   // ke
   sortTable[0x5C05]=12465;   // ke
   sortTable[0x5D51]=12466;   // ke
   sortTable[0x5D59]=12467;   // ke
   sortTable[0x5DB1]=12468;   // ke
   sortTable[0x6119]=12469;   // ke
   sortTable[0x63E2]=12470;   // ke
   sortTable[0x6415]=12471;   // ke
   sortTable[0x6564]=12472;   // ke
   sortTable[0x69BC]=12473;   // ke
   sortTable[0x6A16]=12474;   // ke
   sortTable[0x6E07]=12475;   // ke
   sortTable[0x70A3]=12476;   // ke
   sortTable[0x7241]=12477;   // ke
   sortTable[0x7290]=12478;   // ke
   sortTable[0x790D]=12479;   // ke
   sortTable[0x791A]=12480;   // ke
   sortTable[0x7DD9]=12481;   // ke
   sortTable[0x80E2]=12482;   // ke
   sortTable[0x842A]=12483;   // ke
   sortTable[0x9198]=12484;   // ke
   sortTable[0x9312]=12485;   // ke
   sortTable[0x80AF]=12486;   // ken
   sortTable[0x6073]=12487;   // ken
   sortTable[0x61C7]=12488;   // ken
   sortTable[0x5543]=12489;   // ken
   sortTable[0x57A6]=12490;   // ken
   sortTable[0x58BE]=12491;   // ken
   sortTable[0x9F88]=12492;   // ken
   sortTable[0x88C9]=12493;   // ken
   sortTable[0x80BB]=12494;   // ken
   sortTable[0x8C64]=12495;   // ken
   sortTable[0x9F66]=12496;   // ken
   sortTable[0x63AF]=12497;   // ken
   sortTable[0x808E]=12498;   // ken
   sortTable[0x8903]=12499;   // ken
   sortTable[0x9339]=12500;   // ken
   sortTable[0x5751]=12501;   // keng
   sortTable[0x542D]=12502;   // keng
   sortTable[0x94FF]=12503;   // keng
   sortTable[0x7841]=12504;   // keng
   sortTable[0x962C]=12505;   // keng
   sortTable[0x785C]=12506;   // keng
   sortTable[0x935E]=12507;   // keng
   sortTable[0x787B]=12508;   // keng
   sortTable[0x52A5]=12509;   // keng
   sortTable[0x5994]=12510;   // keng
   sortTable[0x6333]=12511;   // keng
   sortTable[0x647C]=12512;   // keng
   sortTable[0x727C]=12513;   // keng
   sortTable[0x8A99]=12514;   // keng
   sortTable[0x92B5]=12515;   // keng
   sortTable[0x93D7]=12516;   // keng
   sortTable[0x7A7A]=12517;   // kong
   sortTable[0x63A7]=12518;   // kong
   sortTable[0x5B54]=12519;   // kong
   sortTable[0x6050]=12520;   // kong
   sortTable[0x5D06]=12521;   // kong
   sortTable[0x5025]=12522;   // kong
   sortTable[0x7B9C]=12523;   // kong
   sortTable[0x979A]=12524;   // kong
   sortTable[0x787F]=12525;   // kong
   sortTable[0x8EBB]=12526;   // kong
   sortTable[0x9D7C]=12527;   // kong
   sortTable[0x57EA]=12528;   // kong
   sortTable[0x60BE]=12529;   // kong
   sortTable[0x6DB3]=12530;   // kong
   sortTable[0x9313]=12531;   // kong
   sortTable[0x53E3]=12532;   // kou
   sortTable[0x6263]=12533;   // kou
   sortTable[0x5BC7]=12534;   // kou
   sortTable[0x53E9]=12535;   // kou
   sortTable[0x82A4]=12536;   // kou
   sortTable[0x62A0]=12537;   // kou
   sortTable[0x7B58]=12538;   // kou
   sortTable[0x853B]=12539;   // kou
   sortTable[0x9DC7]=12540;   // kou
   sortTable[0x770D]=12541;   // kou
   sortTable[0x91E6]=12542;   // kou
   sortTable[0x7789]=12543;   // kou
   sortTable[0x7A9B]=12544;   // kou
   sortTable[0x8532]=12545;   // kou
   sortTable[0x7C06]=12546;   // kou
   sortTable[0x51A6]=12547;   // kou
   sortTable[0x527E]=12548;   // kou
   sortTable[0x52B6]=12549;   // kou
   sortTable[0x5BBC]=12550;   // kou
   sortTable[0x5F44]=12551;   // kou
   sortTable[0x6473]=12552;   // kou
   sortTable[0x6542]=12553;   // kou
   sortTable[0x6EF1]=12554;   // kou
   sortTable[0x7798]=12555;   // kou
   sortTable[0x82E6]=12556;   // ku
   sortTable[0x54ED]=12557;   // ku
   sortTable[0x5E93]=12558;   // ku
   sortTable[0x5EAB]=12559;   // ku
   sortTable[0x88E4]=12560;   // ku
   sortTable[0x8932]=12561;   // ku
   sortTable[0x9177]=12562;   // ku
   sortTable[0x67AF]=12563;   // ku
   sortTable[0x7A9F]=12564;   // ku
   sortTable[0x9AB7]=12565;   // ku
   sortTable[0x55BE]=12566;   // ku
   sortTable[0x88B4]=12567;   // ku
   sortTable[0x5233]=12568;   // ku
   sortTable[0x7ED4]=12569;   // ku
   sortTable[0x5800]=12570;   // ku
   sortTable[0x77FB]=12571;   // ku
   sortTable[0x8DCD]=12572;   // ku
   sortTable[0x8DB6]=12573;   // ku
   sortTable[0x9BAC]=12574;   // ku
   sortTable[0x90C0]=12575;   // ku
   sortTable[0x79D9]=12576;   // ku
   sortTable[0x4FC8]=12577;   // ku
   sortTable[0x56B3]=12578;   // ku
   sortTable[0x5710]=12579;   // ku
   sortTable[0x5D2B]=12580;   // ku
   sortTable[0x5EE4]=12581;   // ku
   sortTable[0x625D]=12582;   // ku
   sortTable[0x684D]=12583;   // ku
   sortTable[0x7105]=12584;   // ku
   sortTable[0x72DC]=12585;   // ku
   sortTable[0x7614]=12586;   // ku
   sortTable[0x7D5D]=12587;   // ku
   sortTable[0x80D0]=12588;   // ku
   sortTable[0x8DE8]=12589;   // kua
   sortTable[0x5938]=12590;   // kua
   sortTable[0x57AE]=12591;   // kua
   sortTable[0x630E]=12592;   // kua
   sortTable[0x80EF]=12593;   // kua
   sortTable[0x4F89]=12594;   // kua
   sortTable[0x9299]=12595;   // kua
   sortTable[0x8A87]=12596;   // kua
   sortTable[0x9ABB]=12597;   // kua
   sortTable[0x54B5]=12598;   // kua
   sortTable[0x59F1]=12599;   // kua
   sortTable[0x823F]=12600;   // kua
   sortTable[0x5FEB]=12601;   // kuai
   sortTable[0x5757]=12602;   // kuai
   sortTable[0x584A]=12603;   // kuai
   sortTable[0x7B77]=12604;   // kuai
   sortTable[0x84AF]=12605;   // kuai
   sortTable[0x54D9]=12606;   // kuai
   sortTable[0x810D]=12607;   // kuai
   sortTable[0x4FA9]=12608;   // kuai
   sortTable[0x72EF]=12609;   // kuai
   sortTable[0x9C99]=12610;   // kuai
   sortTable[0x90D0]=12611;   // kuai
   sortTable[0x9136]=12612;   // kuai
   sortTable[0x9C60]=12613;   // kuai
   sortTable[0x81BE]=12614;   // kuai
   sortTable[0x5108]=12615;   // kuai
   sortTable[0x51F7]=12616;   // kuai
   sortTable[0x5672]=12617;   // kuai
   sortTable[0x5726]=12618;   // kuai
   sortTable[0x58A4]=12619;   // kuai
   sortTable[0x5DDC]=12620;   // kuai
   sortTable[0x5EE5]=12621;   // kuai
   sortTable[0x64D3]=12622;   // kuai
   sortTable[0x65DD]=12623;   // kuai
   sortTable[0x736A]=12624;   // kuai
   sortTable[0x7CE9]=12625;   // kuai
   sortTable[0x5BBD]=12626;   // kuan
   sortTable[0x5BEC]=12627;   // kuan
   sortTable[0x6B3E]=12628;   // kuan
   sortTable[0x9ACB]=12629;   // kuan
   sortTable[0x7ABE]=12630;   // kuan
   sortTable[0x81D7]=12631;   // kuan
   sortTable[0x5BDB]=12632;   // kuan
   sortTable[0x6B35]=12633;   // kuan
   sortTable[0x6B40]=12634;   // kuan
   sortTable[0x7ABD]=12635;   // kuan
   sortTable[0x9467]=12636;   // kuan
   sortTable[0x9AD6]=12637;   // kuan
   sortTable[0x51B5]=12638;   // kuang
   sortTable[0x6CC1]=12639;   // kuang
   sortTable[0x77FF]=12640;   // kuang
   sortTable[0x7926]=12641;   // kuang
   sortTable[0x72C2]=12642;   // kuang
   sortTable[0x7B50]=12643;   // kuang
   sortTable[0x6846]=12644;   // kuang
   sortTable[0x7736]=12645;   // kuang
   sortTable[0x65F7]=12646;   // kuang
   sortTable[0x66E0]=12647;   // kuang
   sortTable[0x5321]=12648;   // kuang
   sortTable[0x8BF3]=12649;   // kuang
   sortTable[0x8BD3]=12650;   // kuang
   sortTable[0x54D0]=12651;   // kuang
   sortTable[0x8D36]=12652;   // kuang
   sortTable[0x909D]=12653;   // kuang
   sortTable[0x5739]=12654;   // kuang
   sortTable[0x7EA9]=12655;   // kuang
   sortTable[0x593C]=12656;   // kuang
   sortTable[0x7716]=12657;   // kuang
   sortTable[0x7A6C]=12658;   // kuang
   sortTable[0x945B]=12659;   // kuang
   sortTable[0x7E8A]=12660;   // kuang
   sortTable[0x8EED]=12661;   // kuang
   sortTable[0x8E80]=12662;   // kuang
   sortTable[0x8ED6]=12663;   // kuang
   sortTable[0x90BC]=12664;   // kuang
   sortTable[0x913A]=12665;   // kuang
   sortTable[0x9ECB]=12666;   // kuang
   sortTable[0x7844]=12667;   // kuang
   sortTable[0x7D56]=12668;   // kuang
   sortTable[0x9D5F]=12669;   // kuang
   sortTable[0x77CC]=12670;   // kuang
   sortTable[0x8A91]=12671;   // kuang
   sortTable[0x5123]=12672;   // kuang
   sortTable[0x52BB]=12673;   // kuang
   sortTable[0x5329]=12674;   // kuang
   sortTable[0x535D]=12675;   // kuang
   sortTable[0x58D9]=12676;   // kuang
   sortTable[0x5CB2]=12677;   // kuang
   sortTable[0x5FF9]=12678;   // kuang
   sortTable[0x6047]=12679;   // kuang
   sortTable[0x61EC]=12680;   // kuang
   sortTable[0x61ED]=12681;   // kuang
   sortTable[0x6282]=12682;   // kuang
   sortTable[0x663F]=12683;   // kuang
   sortTable[0x6D2D]=12684;   // kuang
   sortTable[0x720C]=12685;   // kuang
   sortTable[0x783F]=12686;   // kuang
   sortTable[0x7B7A]=12687;   // kuang
   sortTable[0x7D4B]=12688;   // kuang
   sortTable[0x8A86]=12689;   // kuang
   sortTable[0x8CBA]=12690;   // kuang
   sortTable[0x8EE6]=12691;   // kuang
   sortTable[0x9271]=12692;   // kuang
   sortTable[0x4E8F]=12693;   // kui
   sortTable[0x8667]=12694;   // kui
   sortTable[0x6127]=12695;   // kui
   sortTable[0x76D4]=12696;   // kui
   sortTable[0x8475]=12697;   // kui
   sortTable[0x6E83]=12698;   // kui
   sortTable[0x6F70]=12699;   // kui
   sortTable[0x9B41]=12700;   // kui
   sortTable[0x9988]=12701;   // kui
   sortTable[0x7AA5]=12702;   // kui
   sortTable[0x594E]=12703;   // kui
   sortTable[0x9035]=12704;   // kui
   sortTable[0x5914]=12705;   // kui
   sortTable[0x532E]=12706;   // kui
   sortTable[0x63C6]=12707;   // kui
   sortTable[0x9697]=12708;   // kui
   sortTable[0x559F]=12709;   // kui
   sortTable[0x9997]=12710;   // kui
   sortTable[0x668C]=12711;   // kui
   sortTable[0x777D]=12712;   // kui
   sortTable[0x6126]=12713;   // kui
   sortTable[0x8DEC]=12714;   // kui
   sortTable[0x8069]=12715;   // kui
   sortTable[0x5CBF]=12716;   // kui
   sortTable[0x609D]=12717;   // kui
   sortTable[0x7BD1]=12718;   // kui
   sortTable[0x8489]=12719;   // kui
   sortTable[0x9A99]=12720;   // kui
   sortTable[0x55B9]=12721;   // kui
   sortTable[0x980D]=12722;   // kui
   sortTable[0x8770]=12723;   // kui
   sortTable[0x993D]=12724;   // kui
   sortTable[0x985D]=12725;   // kui
   sortTable[0x806D]=12726;   // kui
   sortTable[0x7786]=12727;   // kui
   sortTable[0x95DA]=12728;   // kui
   sortTable[0x994B]=12729;   // kui
   sortTable[0x8E5E]=12730;   // kui
   sortTable[0x7C44]=12731;   // kui
   sortTable[0x8067]=12732;   // kui
   sortTable[0x9400]=12733;   // kui
   sortTable[0x9804]=12734;   // kui
   sortTable[0x9A24]=12735;   // kui
   sortTable[0x8143]=12736;   // kui
   sortTable[0x8075]=12737;   // kui
   sortTable[0x8562]=12738;   // kui
   sortTable[0x9108]=12739;   // kui
   sortTable[0x85C8]=12740;   // kui
   sortTable[0x862C]=12741;   // kui
   sortTable[0x8641]=12742;   // kui
   sortTable[0x8EA8]=12743;   // kui
   sortTable[0x9368]=12744;   // kui
   sortTable[0x5232]=12745;   // kui
   sortTable[0x5633]=12746;   // kui
   sortTable[0x5ABF]=12747;   // kui
   sortTable[0x5B07]=12748;   // kui
   sortTable[0x5C2F]=12749;   // kui
   sortTable[0x5DCB]=12750;   // kui
   sortTable[0x5DD9]=12751;   // kui
   sortTable[0x6192]=12752;   // kui
   sortTable[0x6223]=12753;   // kui
   sortTable[0x6646]=12754;   // kui
   sortTable[0x694F]=12755;   // kui
   sortTable[0x6951]=12756;   // kui
   sortTable[0x6A3B]=12757;   // kui
   sortTable[0x6AC6]=12758;   // kui
   sortTable[0x6B33]=12759;   // kui
   sortTable[0x7143]=12760;   // kui
   sortTable[0x72AA]=12761;   // kui
   sortTable[0x7ABA]=12762;   // kui
   sortTable[0x7C23]=12763;   // kui
   sortTable[0x8637]=12764;   // kui
   sortTable[0x8B09]=12765;   // kui
   sortTable[0x9377]=12766;   // kui
   sortTable[0x944E]=12767;   // kui
   sortTable[0x982F]=12768;   // kui
   sortTable[0x56F0]=12769;   // kun
   sortTable[0x6606]=12770;   // kun
   sortTable[0x6346]=12771;   // kun
   sortTable[0x5764]=12772;   // kun
   sortTable[0x7428]=12773;   // kun
   sortTable[0x9AE1]=12774;   // kun
   sortTable[0x9603]=12775;   // kun
   sortTable[0x951F]=12776;   // kun
   sortTable[0x9CB2]=12777;   // kun
   sortTable[0x6083]=12778;   // kun
   sortTable[0x9E4D]=12779;   // kun
   sortTable[0x918C]=12780;   // kun
   sortTable[0x7871]=12781;   // kun
   sortTable[0x88C8]=12782;   // kun
   sortTable[0x890C]=12783;   // kun
   sortTable[0x774F]=12784;   // kun
   sortTable[0x88E9]=12785;   // kun
   sortTable[0x9AE0]=12786;   // kun
   sortTable[0x7A1B]=12787;   // kun
   sortTable[0x7D91]=12788;   // kun
   sortTable[0x9A09]=12789;   // kun
   sortTable[0x9AE8]=12790;   // kun
   sortTable[0x7A07]=12791;   // kun
   sortTable[0x83CE]=12792;   // kun
   sortTable[0x872B]=12793;   // kun
   sortTable[0x9BE4]=12794;   // kun
   sortTable[0x9DA4]=12795;   // kun
   sortTable[0x95B8]=12796;   // kun
   sortTable[0x5803]=12797;   // kun
   sortTable[0x5812]=12798;   // kun
   sortTable[0x58F8]=12799;   // kun
   sortTable[0x58FC]=12800;   // kun
   sortTable[0x5A6B]=12801;   // kun
   sortTable[0x5C21]=12802;   // kun
   sortTable[0x5D10]=12803;   // kun
   sortTable[0x5D11]=12804;   // kun
   sortTable[0x665C]=12805;   // kun
   sortTable[0x68B1]=12806;   // kun
   sortTable[0x6D83]=12807;   // kun
   sortTable[0x6F49]=12808;   // kun
   sortTable[0x711C]=12809;   // kun
   sortTable[0x71B4]=12810;   // kun
   sortTable[0x7311]=12811;   // kun
   sortTable[0x747B]=12812;   // kun
   sortTable[0x7975]=12813;   // kun
   sortTable[0x88CD]=12814;   // kun
   sortTable[0x8C87]=12815;   // kun
   sortTable[0x9315]=12816;   // kun
   sortTable[0x95AB]=12817;   // kun
   sortTable[0x9D7E]=12818;   // kun
   sortTable[0x9F6B]=12819;   // kun
   sortTable[0x62EC]=12820;   // kuo
   sortTable[0x6269]=12821;   // kuo
   sortTable[0x64F4]=12822;   // kuo
   sortTable[0x9614]=12823;   // kuo
   sortTable[0x95CA]=12824;   // kuo
   sortTable[0x5ED3]=12825;   // kuo
   sortTable[0x86DE]=12826;   // kuo
   sortTable[0x7B48]=12827;   // kuo
   sortTable[0x97B9]=12828;   // kuo
   sortTable[0x979F]=12829;   // kuo
   sortTable[0x9729]=12830;   // kuo
   sortTable[0x9B20]=12831;   // kuo
   sortTable[0x843F]=12832;   // kuo
   sortTable[0x9AFA]=12833;   // kuo
   sortTable[0x61D6]=12834;   // kuo
   sortTable[0x62E1]=12835;   // kuo
   sortTable[0x6304]=12836;   // kuo
   sortTable[0x6870]=12837;   // kuo
   sortTable[0x6FF6]=12838;   // kuo
   sortTable[0x8440]=12839;   // kuo
   sortTable[0x97D5]=12840;   // kuo
   sortTable[0x9822]=12841;   // kuo
   sortTable[0x5566]=13312;   // la
   sortTable[0x62C9]=13313;   // la
   sortTable[0x8721]=13314;   // la
   sortTable[0x881F]=13315;   // la
   sortTable[0x5783]=13316;   // la
   sortTable[0x5587]=13317;   // la
   sortTable[0x8FA3]=13318;   // la
   sortTable[0x814A]=13319;   // la
   sortTable[0x524C]=13320;   // la
   sortTable[0x65EF]=13321;   // la
   sortTable[0x782C]=13322;   // la
   sortTable[0x908B]=13323;   // la
   sortTable[0x760C]=13324;   // la
   sortTable[0x9574]=13325;   // la
   sortTable[0x97A1]=13326;   // la
   sortTable[0x81C8]=13327;   // la
   sortTable[0x85DE]=13328;   // la
   sortTable[0x78D6]=13329;   // la
   sortTable[0x7FCB]=13330;   // la
   sortTable[0x81D8]=13331;   // la
   sortTable[0x83C8]=13332;   // la
   sortTable[0x8772]=13333;   // la
   sortTable[0x8FA2]=13334;   // la
   sortTable[0x9B0E]=13335;   // la
   sortTable[0x56B9]=13336;   // la
   sortTable[0x63E6]=13337;   // la
   sortTable[0x63E7]=13338;   // la
   sortTable[0x641A]=13339;   // la
   sortTable[0x650B]=13340;   // la
   sortTable[0x67C6]=13341;   // la
   sortTable[0x694B]=13342;   // la
   sortTable[0x6E82]=13343;   // la
   sortTable[0x7209]=13344;   // la
   sortTable[0x74CE]=13345;   // la
   sortTable[0x874B]=13346;   // la
   sortTable[0x945E]=13347;   // la
   sortTable[0x9BFB]=13348;   // la
   sortTable[0x6765]=13349;   // lai
   sortTable[0x4F86]=13350;   // lai
   sortTable[0x8D56]=13351;   // lai
   sortTable[0x8CF4]=13352;   // lai
   sortTable[0x83B1]=13353;   // lai
   sortTable[0x6D9E]=13354;   // lai
   sortTable[0x8D49]=13355;   // lai
   sortTable[0x765E]=13356;   // lai
   sortTable[0x7C41]=13357;   // lai
   sortTable[0x7750]=13358;   // lai
   sortTable[0x6FD1]=13359;   // lai
   sortTable[0x5D03]=13360;   // lai
   sortTable[0x5F95]=13361;   // lai
   sortTable[0x7C5F]=13362;   // lai
   sortTable[0x94FC]=13363;   // lai
   sortTable[0x90F2]=13364;   // lai
   sortTable[0x9A0B]=13365;   // lai
   sortTable[0x85FE]=13366;   // lai
   sortTable[0x775E]=13367;   // lai
   sortTable[0x9BE0]=13368;   // lai
   sortTable[0x9D63]=13369;   // lai
   sortTable[0x9D86]=13370;   // lai
   sortTable[0x840A]=13371;   // lai
   sortTable[0x9028]=13372;   // lai
   sortTable[0x983C]=13373;   // lai
   sortTable[0x8CDA]=13374;   // lai
   sortTable[0x4FEB]=13375;   // lai
   sortTable[0x5008]=13376;   // lai
   sortTable[0x553B]=13377;   // lai
   sortTable[0x5A61]=13378;   // lai
   sortTable[0x5D0D]=13379;   // lai
   sortTable[0x5EB2]=13380;   // lai
   sortTable[0x5FA0]=13381;   // lai
   sortTable[0x68BE]=13382;   // lai
   sortTable[0x68F6]=13383;   // lai
   sortTable[0x6AF4]=13384;   // lai
   sortTable[0x6DF6]=13385;   // lai
   sortTable[0x7028]=13386;   // lai
   sortTable[0x702C]=13387;   // lai
   sortTable[0x730D]=13388;   // lai
   sortTable[0x741C]=13389;   // lai
   sortTable[0x7669]=13390;   // lai
   sortTable[0x7B59]=13391;   // lai
   sortTable[0x7B82]=13392;   // lai
   sortTable[0x8970]=13393;   // lai
   sortTable[0x9338]=13394;   // lai
   sortTable[0x9842]=13395;   // lai
   sortTable[0x9EB3]=13396;   // lai
   sortTable[0x84DD]=13397;   // lan
   sortTable[0x70C2]=13398;   // lan
   sortTable[0x85CD]=13399;   // lan
   sortTable[0x721B]=13400;   // lan
   sortTable[0x7BEE]=13401;   // lan
   sortTable[0x7C43]=13402;   // lan
   sortTable[0x62E6]=13403;   // lan
   sortTable[0x6514]=13404;   // lan
   sortTable[0x89C8]=13405;   // lan
   sortTable[0x89BD]=13406;   // lan
   sortTable[0x680F]=13407;   // lan
   sortTable[0x6B04]=13408;   // lan
   sortTable[0x61D2]=13409;   // lan
   sortTable[0x61F6]=13410;   // lan
   sortTable[0x6EE5]=13411;   // lan
   sortTable[0x6FEB]=13412;   // lan
   sortTable[0x5170]=13413;   // lan
   sortTable[0x5C9A]=13414;   // lan
   sortTable[0x6F9C]=13415;   // lan
   sortTable[0x63FD]=13416;   // lan
   sortTable[0x7F06]=13417;   // lan
   sortTable[0x9611]=13418;   // lan
   sortTable[0x5A6A]=13419;   // lan
   sortTable[0x6593]=13420;   // lan
   sortTable[0x6984]=13421;   // lan
   sortTable[0x7F71]=13422;   // lan
   sortTable[0x8C30]=13423;   // lan
   sortTable[0x8934]=13424;   // lan
   sortTable[0x8955]=13425;   // lan
   sortTable[0x9567]=13426;   // lan
   sortTable[0x6F24]=13427;   // lan
   sortTable[0x862D]=13428;   // lan
   sortTable[0x8B95]=13429;   // lan
   sortTable[0x847B]=13430;   // lan
   sortTable[0x8974]=13431;   // lan
   sortTable[0x7CF7]=13432;   // lan
   sortTable[0x7E7F]=13433;   // lan
   sortTable[0x7C63]=13434;   // lan
   sortTable[0x897D]=13435;   // lan
   sortTable[0x8B4B]=13436;   // lan
   sortTable[0x8E9D]=13437;   // lan
   sortTable[0x9484]=13438;   // lan
   sortTable[0x95CC]=13439;   // lan
   sortTable[0x9872]=13440;   // lan
   sortTable[0x5116]=13441;   // lan
   sortTable[0x53B1]=13442;   // lan
   sortTable[0x5682]=13443;   // lan
   sortTable[0x56D2]=13444;   // lan
   sortTable[0x56D5]=13445;   // lan
   sortTable[0x58C8]=13446;   // lan
   sortTable[0x5B3E]=13447;   // lan
   sortTable[0x5B44]=13448;   // lan
   sortTable[0x5B4F]=13449;   // lan
   sortTable[0x5D50]=13450;   // lan
   sortTable[0x5E71]=13451;   // lan
   sortTable[0x60CF]=13452;   // lan
   sortTable[0x61E2]=13453;   // lan
   sortTable[0x64E5]=13454;   // lan
   sortTable[0x652C]=13455;   // lan
   sortTable[0x6595]=13456;   // lan
   sortTable[0x6B16]=13457;   // lan
   sortTable[0x6B17]=13458;   // lan
   sortTable[0x6D68]=13459;   // lan
   sortTable[0x703E]=13460;   // lan
   sortTable[0x7046]=13461;   // lan
   sortTable[0x7060]=13462;   // lan
   sortTable[0x7061]=13463;   // lan
   sortTable[0x71D7]=13464;   // lan
   sortTable[0x71E3]=13465;   // lan
   sortTable[0x71F7]=13466;   // lan
   sortTable[0x7201]=13467;   // lan
   sortTable[0x7224]=13468;   // lan
   sortTable[0x7226]=13469;   // lan
   sortTable[0x74BC]=13470;   // lan
   sortTable[0x74D3]=13471;   // lan
   sortTable[0x7937]=13472;   // lan
   sortTable[0x7E9C]=13473;   // lan
   sortTable[0x8964]=13474;   // lan
   sortTable[0x89A7]=13475;   // lan
   sortTable[0x9182]=13476;   // lan
   sortTable[0x946D]=13477;   // lan
   sortTable[0x97CA]=13478;   // lan
   sortTable[0x6D6A]=13479;   // lang
   sortTable[0x72FC]=13480;   // lang
   sortTable[0x6717]=13481;   // lang
   sortTable[0x90CE]=13482;   // lang
   sortTable[0x5ECA]=13483;   // lang
   sortTable[0x7405]=13484;   // lang
   sortTable[0x9606]=13485;   // lang
   sortTable[0x8782]=13486;   // lang
   sortTable[0x6994]=13487;   // lang
   sortTable[0x9512]=13488;   // lang
   sortTable[0x83A8]=13489;   // lang
   sortTable[0x5577]=13490;   // lang
   sortTable[0x93AF]=13491;   // lang
   sortTable[0x8497]=13492;   // lang
   sortTable[0x7A02]=13493;   // lang
   sortTable[0x7B64]=13494;   // lang
   sortTable[0x870B]=13495;   // lang
   sortTable[0x7860]=13496;   // lang
   sortTable[0x8246]=13497;   // lang
   sortTable[0x90DE]=13498;   // lang
   sortTable[0x95AC]=13499;   // lang
   sortTable[0x84E2]=13500;   // lang
   sortTable[0x99FA]=13501;   // lang
   sortTable[0x52C6]=13502;   // lang
   sortTable[0x5525]=13503;   // lang
   sortTable[0x57CC]=13504;   // lang
   sortTable[0x5871]=13505;   // lang
   sortTable[0x5ACF]=13506;   // lang
   sortTable[0x5D00]=13507;   // lang
   sortTable[0x658F]=13508;   // lang
   sortTable[0x6716]=13509;   // lang
   sortTable[0x6724]=13510;   // lang
   sortTable[0x6879]=13511;   // lang
   sortTable[0x6A03]=13512;   // lang
   sortTable[0x6B34]=13513;   // lang
   sortTable[0x70FA]=13514;   // lang
   sortTable[0x746F]=13515;   // lang
   sortTable[0x84C8]=13516;   // lang
   sortTable[0x8A8F]=13517;   // lang
   sortTable[0x8EB4]=13518;   // lang
   sortTable[0x90D2]=13519;   // lang
   sortTable[0x92C3]=13520;   // lang
   sortTable[0x8001]=13521;   // lao
   sortTable[0x52B3]=13522;   // lao
   sortTable[0x52DE]=13523;   // lao
   sortTable[0x635E]=13524;   // lao
   sortTable[0x6488]=13525;   // lao
   sortTable[0x7262]=13526;   // lao
   sortTable[0x59E5]=13527;   // lao
   sortTable[0x6F66]=13528;   // lao
   sortTable[0x70D9]=13529;   // lao
   sortTable[0x916A]=13530;   // lao
   sortTable[0x4F6C]=13531;   // lao
   sortTable[0x5520]=13532;   // lao
   sortTable[0x6D9D]=13533;   // lao
   sortTable[0x91AA]=13534;   // lao
   sortTable[0x5D02]=13535;   // lao
   sortTable[0x75E8]=13536;   // lao
   sortTable[0x6833]=13537;   // lao
   sortTable[0x94D1]=13538;   // lao
   sortTable[0x8022]=13539;   // lao
   sortTable[0x985F]=13540;   // lao
   sortTable[0x7CA9]=13541;   // lao
   sortTable[0x8EBC]=13542;   // lao
   sortTable[0x86EF]=13543;   // lao
   sortTable[0x8F51]=13544;   // lao
   sortTable[0x94F9]=13545;   // lao
   sortTable[0x8356]=13546;   // lao
   sortTable[0x87E7]=13547;   // lao
   sortTable[0x78F1]=13548;   // lao
   sortTable[0x7C29]=13549;   // lao
   sortTable[0x802E]=13550;   // lao
   sortTable[0x9BB1]=13551;   // lao
   sortTable[0x50D7]=13552;   // lao
   sortTable[0x52B4]=13553;   // lao
   sortTable[0x54BE]=13554;   // lao
   sortTable[0x54F0]=13555;   // lao
   sortTable[0x562E]=13556;   // lao
   sortTable[0x5AEA]=13557;   // lao
   sortTable[0x5D97]=13558;   // lao
   sortTable[0x6045]=13559;   // lao
   sortTable[0x61A5]=13560;   // lao
   sortTable[0x61A6]=13561;   // lao
   sortTable[0x6725]=13562;   // lao
   sortTable[0x6A51]=13563;   // lao
   sortTable[0x6A6F]=13564;   // lao
   sortTable[0x6D76]=13565;   // lao
   sortTable[0x6F87]=13566;   // lao
   sortTable[0x72EB]=13567;   // lao
   sortTable[0x73EF]=13568;   // lao
   sortTable[0x7646]=13569;   // lao
   sortTable[0x7853]=13570;   // lao
   sortTable[0x7A82]=13571;   // lao
   sortTable[0x8002]=13572;   // lao
   sortTable[0x8EC2]=13573;   // lao
   sortTable[0x92A0]=13574;   // lao
   sortTable[0x9412]=13575;   // lao
   sortTable[0x9ADD]=13576;   // lao
   sortTable[0x4E86]=13577;   // le
   sortTable[0x4E50]=13578;   // le
   sortTable[0x6A02]=13579;   // le
   sortTable[0x808B]=13580;   // le
   sortTable[0x53FB]=13581;   // le
   sortTable[0x6CD0]=13582;   // le
   sortTable[0x4EC2]=13583;   // le
   sortTable[0x9979]=13584;   // le
   sortTable[0x9CD3]=13585;   // le
   sortTable[0x7C15]=13586;   // le
   sortTable[0x97F7]=13587;   // le
   sortTable[0x961E]=13588;   // le
   sortTable[0x7AFB]=13589;   // le
   sortTable[0x9C33]=13590;   // le
   sortTable[0x5FC7]=13591;   // le
   sortTable[0x6250]=13592;   // le
   sortTable[0x697D]=13593;   // le
   sortTable[0x6C3B]=13594;   // le
   sortTable[0x738F]=13595;   // le
   sortTable[0x7833]=13596;   // le
   sortTable[0x827B]=13597;   // le
   sortTable[0x990E]=13598;   // le
   sortTable[0x7C7B]=13599;   // lei
   sortTable[0x985E]=13600;   // lei
   sortTable[0x6CEA]=13601;   // lei
   sortTable[0x6DDA]=13602;   // lei
   sortTable[0x7D2F]=13603;   // lei
   sortTable[0x96F7]=13604;   // lei
   sortTable[0x5792]=13605;   // lei
   sortTable[0x58D8]=13606;   // lei
   sortTable[0x52D2]=13607;   // lei
   sortTable[0x78CA]=13608;   // lei
   sortTable[0x64C2]=13609;   // lei
   sortTable[0x8BD4]=13610;   // lei
   sortTable[0x857E]=13611;   // lei
   sortTable[0x5121]=13612;   // lei
   sortTable[0x7FB8]=13613;   // lei
   sortTable[0x8012]=13614;   // lei
   sortTable[0x956D]=13615;   // lei
   sortTable[0x5AD8]=13616;   // lei
   sortTable[0x561E]=13617;   // lei
   sortTable[0x7F4D]=13618;   // lei
   sortTable[0x7F27]=13619;   // lei
   sortTable[0x9179]=13620;   // lei
   sortTable[0x790C]=13621;   // lei
   sortTable[0x7D6B]=13622;   // lei
   sortTable[0x85DF]=13623;   // lei
   sortTable[0x98A3]=13624;   // lei
   sortTable[0x7927]=13625;   // lei
   sortTable[0x9287]=13626;   // lei
   sortTable[0x6A91]=13627;   // lei
   sortTable[0x78E5]=13628;   // lei
   sortTable[0x8502]=13629;   // lei
   sortTable[0x8646]=13630;   // lei
   sortTable[0x881D]=13631;   // lei
   sortTable[0x7E8D]=13632;   // lei
   sortTable[0x7928]=13633;   // lei
   sortTable[0x8142]=13634;   // lei
   sortTable[0x7E32]=13635;   // lei
   sortTable[0x854C]=13636;   // lei
   sortTable[0x8631]=13637;   // lei
   sortTable[0x9433]=13638;   // lei
   sortTable[0x9458]=13639;   // lei
   sortTable[0x9478]=13640;   // lei
   sortTable[0x9E13]=13641;   // lei
   sortTable[0x50AB]=13642;   // lei
   sortTable[0x513D]=13643;   // lei
   sortTable[0x53BD]=13644;   // lei
   sortTable[0x5841]=13645;   // lei
   sortTable[0x58E8]=13646;   // lei
   sortTable[0x6502]=13647;   // lei
   sortTable[0x6A0F]=13648;   // lei
   sortTable[0x6AD0]=13649;   // lei
   sortTable[0x6AD1]=13650;   // lei
   sortTable[0x6B19]=13651;   // lei
   sortTable[0x6D21]=13652;   // lei
   sortTable[0x6D99]=13653;   // lei
   sortTable[0x7045]=13654;   // lei
   sortTable[0x74C3]=13655;   // lei
   sortTable[0x757E]=13656;   // lei
   sortTable[0x7657]=13657;   // lei
   sortTable[0x79B7]=13658;   // lei
   sortTable[0x7E87]=13659;   // lei
   sortTable[0x7E9D]=13660;   // lei
   sortTable[0x8632]=13661;   // lei
   sortTable[0x863D]=13662;   // lei
   sortTable[0x8A84]=13663;   // lei
   sortTable[0x8B84]=13664;   // lei
   sortTable[0x8F60]=13665;   // lei
   sortTable[0x9311]=13666;   // lei
   sortTable[0x9741]=13667;   // lei
   sortTable[0x981B]=13668;   // lei
   sortTable[0x982A]=13669;   // lei
   sortTable[0x9C69]=13670;   // lei
   sortTable[0x9F3A]=13671;   // lei
   sortTable[0x51B7]=13672;   // leng
   sortTable[0x6123]=13673;   // leng
   sortTable[0x68F1]=13674;   // leng
   sortTable[0x695E]=13675;   // leng
   sortTable[0x7A1C]=13676;   // leng
   sortTable[0x5844]=13677;   // leng
   sortTable[0x7756]=13678;   // leng
   sortTable[0x8590]=13679;   // leng
   sortTable[0x8F18]=13680;   // leng
   sortTable[0x8E1C]=13681;   // leng
   sortTable[0x5030]=13682;   // leng
   sortTable[0x580E]=13683;   // leng
   sortTable[0x5D1A]=13684;   // leng
   sortTable[0x7890]=13685;   // leng
   sortTable[0x91CC]=13686;   // li
   sortTable[0x529B]=13687;   // li
   sortTable[0x7406]=13688;   // li
   sortTable[0x5229]=13689;   // li
   sortTable[0x7ACB]=13690;   // li
   sortTable[0x5386]=13691;   // li
   sortTable[0x6B77]=13692;   // li
   sortTable[0x79BB]=13693;   // li
   sortTable[0x96E2]=13694;   // li
   sortTable[0x88E1]=13695;   // li
   sortTable[0x4F8B]=13696;   // li
   sortTable[0x54E9]=13697;   // li
   sortTable[0x4E3D]=13698;   // li
   sortTable[0x9E97]=13699;   // li
   sortTable[0x7483]=13700;   // li
   sortTable[0x5389]=13701;   // li
   sortTable[0x53B2]=13702;   // li
   sortTable[0x793C]=13703;   // li
   sortTable[0x79AE]=13704;   // li
   sortTable[0x7C92]=13705;   // li
   sortTable[0x52B1]=13706;   // li
   sortTable[0x52F5]=13707;   // li
   sortTable[0x96B6]=13708;   // li
   sortTable[0x96B8]=13709;   // li
   sortTable[0x5398]=13710;   // li
   sortTable[0x674E]=13711;   // li
   sortTable[0x9ECE]=13712;   // li
   sortTable[0x8354]=13713;   // li
   sortTable[0x6CA5]=13714;   // li
   sortTable[0x701D]=13715;   // li
   sortTable[0x72F8]=13716;   // li
   sortTable[0x7BF1]=13717;   // li
   sortTable[0x7C6C]=13718;   // li
   sortTable[0x68A8]=13719;   // li
   sortTable[0x9E42]=13720;   // li
   sortTable[0x9E1D]=13721;   // li
   sortTable[0x9CA4]=13722;   // li
   sortTable[0x9BC9]=13723;   // li
   sortTable[0x7281]=13724;   // li
   sortTable[0x783E]=13725;   // li
   sortTable[0x792B]=13726;   // li
   sortTable[0x540F]=13727;   // li
   sortTable[0x8389]=13728;   // li
   sortTable[0x6817]=13729;   // li
   sortTable[0x4FDA]=13730;   // li
   sortTable[0x7B20]=13731;   // li
   sortTable[0x7F79]=13732;   // li
   sortTable[0x91B4]=13733;   // li
   sortTable[0x623E]=13734;   // li
   sortTable[0x8821]=13735;   // li
   sortTable[0x6F13]=13736;   // li
   sortTable[0x4FD0]=13737;   // li
   sortTable[0x8A48]=13738;   // li
   sortTable[0x96F3]=13739;   // li
   sortTable[0x75E2]=13740;   // li
   sortTable[0x8385]=13741;   // li
   sortTable[0x85DC]=13742;   // li
   sortTable[0x4FEA]=13743;   // li
   sortTable[0x9A8A]=13744;   // li
   sortTable[0x783A]=13745;   // li
   sortTable[0x9026]=13746;   // li
   sortTable[0x6FA7]=13747;   // li
   sortTable[0x91D0]=13748;   // li
   sortTable[0x90E6]=13749;   // li
   sortTable[0x7C9D]=13750;   // li
   sortTable[0x86CE]=13751;   // li
   sortTable[0x5AE0]=13752;   // li
   sortTable[0x680E]=13753;   // li
   sortTable[0x8DDE]=13754;   // li
   sortTable[0x7F21]=13755;   // li
   sortTable[0x5A0C]=13756;   // li
   sortTable[0x75A0]=13757;   // li
   sortTable[0x6EA7]=13758;   // li
   sortTable[0x67A5]=13759;   // li
   sortTable[0x5456]=13760;   // li
   sortTable[0x5533]=13761;   // li
   sortTable[0x870A]=13762;   // li
   sortTable[0x82C8]=13763;   // li
   sortTable[0x9502]=13764;   // li
   sortTable[0x8F79]=13765;   // li
   sortTable[0x7BE5]=13766;   // li
   sortTable[0x5088]=13767;   // li
   sortTable[0x9CA1]=13768;   // li
   sortTable[0x9EE7]=13769;   // li
   sortTable[0x9CE2]=13770;   // li
   sortTable[0x8C4A]=13771;   // li
   sortTable[0x55B1]=13772;   // li
   sortTable[0x7E9A]=13773;   // li
   sortTable[0x7301]=13774;   // li
   sortTable[0x7D9F]=13775;   // li
   sortTable[0x84E0]=13776;   // li
   sortTable[0x8935]=13777;   // li
   sortTable[0x75AC]=13778;   // li
   sortTable[0x76ED]=13779;   // li
   sortTable[0x9C67]=13780;   // li
   sortTable[0x8C8D]=13781;   // li
   sortTable[0x8243]=13782;   // li
   sortTable[0x91A8]=13783;   // li
   sortTable[0x78FF]=13784;   // li
   sortTable[0x7CB4]=13785;   // li
   sortTable[0x9E9C]=13786;   // li
   sortTable[0x575C]=13787;   // li
   sortTable[0x82D9]=13788;   // li
   sortTable[0x7C9A]=13789;   // li
   sortTable[0x9BCF]=13790;   // li
   sortTable[0x7805]=13791;   // li
   sortTable[0x8B88]=13792;   // li
   sortTable[0x88CF]=13793;   // li
   sortTable[0x86E0]=13794;   // li
   sortTable[0x908C]=13795;   // li
   sortTable[0x8137]=13796;   // li
   sortTable[0x83DE]=13797;   // li
   sortTable[0x8B27]=13798;   // li
   sortTable[0x92EB]=13799;   // li
   sortTable[0x849E]=13800;   // li
   sortTable[0x85F6]=13801;   // li
   sortTable[0x9B01]=13802;   // li
   sortTable[0x8823]=13803;   // li
   sortTable[0x7CF2]=13804;   // li
   sortTable[0x8318]=13805;   // li
   sortTable[0x849A]=13806;   // li
   sortTable[0x8727]=13807;   // li
   sortTable[0x89FB]=13808;   // li
   sortTable[0x8F63]=13809;   // li
   sortTable[0x9305]=13810;   // li
   sortTable[0x76E0]=13811;   // li
   sortTable[0x7A72]=13812;   // li
   sortTable[0x7B63]=13813;   // li
   sortTable[0x86B8]=13814;   // li
   sortTable[0x8D72]=13815;   // li
   sortTable[0x8E92]=13816;   // li
   sortTable[0x925D]=13817;   // li
   sortTable[0x92F0]=13818;   // li
   sortTable[0x9398]=13819;   // li
   sortTable[0x9BEC]=13820;   // li
   sortTable[0x9C73]=13821;   // li
   sortTable[0x9C7A]=13822;   // li
   sortTable[0x9D17]=13823;   // li
   sortTable[0x512E]=13824;   // li
   sortTable[0x5137]=13825;   // li
   sortTable[0x5163]=13826;   // li
   sortTable[0x51D3]=13827;   // li
   sortTable[0x5215]=13828;   // li
   sortTable[0x5253]=13829;   // li
   sortTable[0x527A]=13830;   // li
   sortTable[0x5299]=13831;   // li
   sortTable[0x53A4]=13832;   // li
   sortTable[0x53AF]=13833;   // li
   sortTable[0x550E]=13834;   // li
   sortTable[0x569F]=13835;   // li
   sortTable[0x56A6]=13836;   // li
   sortTable[0x56C4]=13837;   // li
   sortTable[0x56C7]=13838;   // li
   sortTable[0x585B]=13839;   // li
   sortTable[0x58E2]=13840;   // li
   sortTable[0x5A33]=13841;   // li
   sortTable[0x5A6F]=13842;   // li
   sortTable[0x5B4B]=13843;   // li
   sortTable[0x5B77]=13844;   // li
   sortTable[0x5C74]=13845;   // li
   sortTable[0x5CA6]=13846;   // li
   sortTable[0x5CDB]=13847;   // li
   sortTable[0x5CE2]=13848;   // li
   sortTable[0x5CF2]=13849;   // li
   sortTable[0x5DC1]=13850;   // li
   sortTable[0x5EF2]=13851;   // li
   sortTable[0x60A1]=13852;   // li
   sortTable[0x60A7]=13853;   // li
   sortTable[0x6144]=13854;   // li
   sortTable[0x642E]=13855;   // li
   sortTable[0x650A]=13856;   // li
   sortTable[0x6521]=13857;   // li
   sortTable[0x6526]=13858;   // li
   sortTable[0x652D]=13859;   // li
   sortTable[0x6584]=13860;   // li
   sortTable[0x66A6]=13861;   // li
   sortTable[0x66C6]=13862;   // li
   sortTable[0x66DE]=13863;   // li
   sortTable[0x6738]=13864;   // li
   sortTable[0x675D]=13865;   // li
   sortTable[0x6803]=13866;   // li
   sortTable[0x681B]=13867;   // li
   sortTable[0x68A9]=13868;   // li
   sortTable[0x68B8]=13869;   // li
   sortTable[0x68C3]=13870;   // li
   sortTable[0x68D9]=13871;   // li
   sortTable[0x6A06]=13872;   // li
   sortTable[0x6AAA]=13873;   // li
   sortTable[0x6AD4]=13874;   // li
   sortTable[0x6ADF]=13875;   // li
   sortTable[0x6AEA]=13876;   // li
   sortTable[0x6B10]=13877;   // li
   sortTable[0x6B1A]=13878;   // li
   sortTable[0x6B74]=13879;   // li
   sortTable[0x6CB4]=13880;   // li
   sortTable[0x6D6C]=13881;   // li
   sortTable[0x6D96]=13882;   // li
   sortTable[0x6FFF]=13883;   // li
   sortTable[0x7055]=13884;   // li
   sortTable[0x7204]=13885;   // li
   sortTable[0x720F]=13886;   // li
   sortTable[0x7282]=13887;   // li
   sortTable[0x72A1]=13888;   // li
   sortTable[0x73D5]=13889;   // li
   sortTable[0x740D]=13890;   // li
   sortTable[0x746E]=13891;   // li
   sortTable[0x74C5]=13892;   // li
   sortTable[0x74C8]=13893;   // li
   sortTable[0x74D1]=13894;   // li
   sortTable[0x74E5]=13895;   // li
   sortTable[0x7658]=13896;   // li
   sortTable[0x7667]=13897;   // li
   sortTable[0x76AA]=13898;   // li
   sortTable[0x775D]=13899;   // li
   sortTable[0x77CB]=13900;   // li
   sortTable[0x792A]=13901;   // li
   sortTable[0x7930]=13902;   // li
   sortTable[0x79B2]=13903;   // li
   sortTable[0x79DD]=13904;   // li
   sortTable[0x7CCE]=13905;   // li
   sortTable[0x7E2D]=13906;   // li
   sortTable[0x8372]=13907;   // li
   sortTable[0x853E]=13908;   // li
   sortTable[0x863A]=13909;   // li
   sortTable[0x8777]=13910;   // li
   sortTable[0x87CD]=13911;   // li
   sortTable[0x87F8]=13912;   // li
   sortTable[0x8807]=13913;   // li
   sortTable[0x882B]=13914;   // li
   sortTable[0x8F62]=13915;   // li
   sortTable[0x9090]=13916;   // li
   sortTable[0x9148]=13917;   // li
   sortTable[0x93EB]=13918;   // li
   sortTable[0x9457]=13919;   // li
   sortTable[0x96B7]=13920;   // li
   sortTable[0x9742]=13921;   // li
   sortTable[0x974B]=13922;   // li
   sortTable[0x9A39]=13923;   // li
   sortTable[0x9A6A]=13924;   // li
   sortTable[0x9C71]=13925;   // li
   sortTable[0x9CE8]=13926;   // li
   sortTable[0x9D79]=13927;   // li
   sortTable[0x9DC5]=13928;   // li
   sortTable[0x4FE9]=13929;   // lia
   sortTable[0x5006]=13930;   // lia
   sortTable[0x8FDE]=13931;   // lian
   sortTable[0x9023]=13932;   // lian
   sortTable[0x8138]=13933;   // lian
   sortTable[0x81C9]=13934;   // lian
   sortTable[0x8054]=13935;   // lian
   sortTable[0x806F]=13936;   // lian
   sortTable[0x70BC]=13937;   // lian
   sortTable[0x7149]=13938;   // lian
   sortTable[0x7EC3]=13939;   // lian
   sortTable[0x7DF4]=13940;   // lian
   sortTable[0x601C]=13941;   // lian
   sortTable[0x6190]=13942;   // lian
   sortTable[0x5E18]=13943;   // lian
   sortTable[0x9570]=13944;   // lian
   sortTable[0x942E]=13945;   // lian
   sortTable[0x5EC9]=13946;   // lian
   sortTable[0x604B]=13947;   // lian
   sortTable[0x6200]=13948;   // lian
   sortTable[0x94FE]=13949;   // lian
   sortTable[0x93C8]=13950;   // lian
   sortTable[0x83B2]=13951;   // lian
   sortTable[0x655B]=13952;   // lian
   sortTable[0x740F]=13953;   // lian
   sortTable[0x6D9F]=13954;   // lian
   sortTable[0x6B93]=13955;   // lian
   sortTable[0x81C1]=13956;   // lian
   sortTable[0x5941]=13957;   // lian
   sortTable[0x6FC2]=13958;   // lian
   sortTable[0x880A]=13959;   // lian
   sortTable[0x88E2]=13960;   // lian
   sortTable[0x88E3]=13961;   // lian
   sortTable[0x9CA2]=13962;   // lian
   sortTable[0x6F4B]=13963;   // lian
   sortTable[0x695D]=13964;   // lian
   sortTable[0x934A]=13965;   // lian
   sortTable[0x84EE]=13966;   // lian
   sortTable[0x938C]=13967;   // lian
   sortTable[0x8539]=13968;   // lian
   sortTable[0x7C3E]=13969;   // lian
   sortTable[0x9B11]=13970;   // lian
   sortTable[0x78CF]=13971;   // lian
   sortTable[0x8B30]=13972;   // lian
   sortTable[0x9C31]=13973;   // lian
   sortTable[0x8068]=13974;   // lian
   sortTable[0x7FB7]=13975;   // lian
   sortTable[0x806B]=13976;   // lian
   sortTable[0x7E3A]=13977;   // lian
   sortTable[0x8595]=13978;   // lian
   sortTable[0x878A]=13979;   // lian
   sortTable[0x913B]=13980;   // lian
   sortTable[0x932C]=13981;   // lian
   sortTable[0x806E]=13982;   // lian
   sortTable[0x8430]=13983;   // lian
   sortTable[0x895D]=13984;   // lian
   sortTable[0x4EB7]=13985;   // lian
   sortTable[0x50C6]=13986;   // lian
   sortTable[0x5286]=13987;   // lian
   sortTable[0x5332]=13988;   // lian
   sortTable[0x5333]=13989;   // lian
   sortTable[0x55F9]=13990;   // lian
   sortTable[0x5652]=13991;   // lian
   sortTable[0x581C]=13992;   // lian
   sortTable[0x5969]=13993;   // lian
   sortTable[0x5AA1]=13994;   // lian
   sortTable[0x5AFE]=13995;   // lian
   sortTable[0x5B1A]=13996;   // lian
   sortTable[0x6169]=13997;   // lian
   sortTable[0x6459]=13998;   // lian
   sortTable[0x6582]=13999;   // lian
   sortTable[0x68BF]=14000;   // lian
   sortTable[0x69E4]=14001;   // lian
   sortTable[0x6AE3]=14002;   // lian
   sortTable[0x6BAE]=14003;   // lian
   sortTable[0x6D70]=14004;   // lian
   sortTable[0x6E45]=14005;   // lian
   sortTable[0x6E93]=14006;   // lian
   sortTable[0x6F23]=14007;   // lian
   sortTable[0x6FB0]=14008;   // lian
   sortTable[0x6FD3]=14009;   // lian
   sortTable[0x7032]=14010;   // lian
   sortTable[0x7191]=14011;   // lian
   sortTable[0x71EB]=14012;   // lian
   sortTable[0x7453]=14013;   // lian
   sortTable[0x7489]=14014;   // lian
   sortTable[0x7C62]=14015;   // lian
   sortTable[0x7C68]=14016;   // lian
   sortTable[0x7E9E]=14017;   // lian
   sortTable[0x7FF4]=14018;   // lian
   sortTable[0x861D]=14019;   // lian
   sortTable[0x861E]=14020;   // lian
   sortTable[0x8933]=14021;   // lian
   sortTable[0x899D]=14022;   // lian
   sortTable[0x8E65]=14023;   // lian
   sortTable[0x9C0A]=14024;   // lian
   sortTable[0x4E24]=14025;   // liang
   sortTable[0x5169]=14026;   // liang
   sortTable[0x91CF]=14027;   // liang
   sortTable[0x4EAE]=14028;   // liang
   sortTable[0x7CAE]=14029;   // liang
   sortTable[0x7CE7]=14030;   // liang
   sortTable[0x826F]=14031;   // liang
   sortTable[0x8F86]=14032;   // liang
   sortTable[0x8F1B]=14033;   // liang
   sortTable[0x51C9]=14034;   // liang
   sortTable[0x6DBC]=14035;   // liang
   sortTable[0x6881]=14036;   // liang
   sortTable[0x8C05]=14037;   // liang
   sortTable[0x8AD2]=14038;   // liang
   sortTable[0x667E]=14039;   // liang
   sortTable[0x690B]=14040;   // liang
   sortTable[0x8E09]=14041;   // liang
   sortTable[0x7CB1]=14042;   // liang
   sortTable[0x9B49]=14043;   // liang
   sortTable[0x8F8C]=14044;   // liang
   sortTable[0x88F2]=14045;   // liang
   sortTable[0x8F2C]=14046;   // liang
   sortTable[0x7DC9]=14047;   // liang
   sortTable[0x589A]=14048;   // liang
   sortTable[0x813C]=14049;   // liang
   sortTable[0x873D]=14050;   // liang
   sortTable[0x7DA1]=14051;   // liang
   sortTable[0x4E21]=14052;   // liang
   sortTable[0x4FCD]=14053;   // liang
   sortTable[0x54F4]=14054;   // liang
   sortTable[0x5521]=14055;   // liang
   sortTable[0x5562]=14056;   // liang
   sortTable[0x55A8]=14057;   // liang
   sortTable[0x60A2]=14058;   // liang
   sortTable[0x639A]=14059;   // liang
   sortTable[0x6A11]=14060;   // liang
   sortTable[0x6E78]=14061;   // liang
   sortTable[0x7177]=14062;   // liang
   sortTable[0x7C17]=14063;   // liang
   sortTable[0x8F0C]=14064;   // liang
   sortTable[0x9344]=14065;   // liang
   sortTable[0x9B4E]=14066;   // liang
   sortTable[0x6599]=14067;   // liao
   sortTable[0x77AD]=14068;   // liao
   sortTable[0x7597]=14069;   // liao
   sortTable[0x7642]=14070;   // liao
   sortTable[0x50DA]=14071;   // liao
   sortTable[0x8FBD]=14072;   // liao
   sortTable[0x907C]=14073;   // liao
   sortTable[0x804A]=14074;   // liao
   sortTable[0x64A9]=14075;   // liao
   sortTable[0x6482]=14076;   // liao
   sortTable[0x5ED6]=14077;   // liao
   sortTable[0x5BE5]=14078;   // liao
   sortTable[0x7F2D]=14079;   // liao
   sortTable[0x71CE]=14080;   // liao
   sortTable[0x7360]=14081;   // liao
   sortTable[0x5BEE]=14082;   // liao
   sortTable[0x948C]=14083;   // liao
   sortTable[0x9563]=14084;   // liao
   sortTable[0x9E69]=14085;   // liao
   sortTable[0x84FC]=14086;   // liao
   sortTable[0x5639]=14087;   // liao
   sortTable[0x5C25]=14088;   // liao
   sortTable[0x7C1D]=14089;   // liao
   sortTable[0x818B]=14090;   // liao
   sortTable[0x8C42]=14091;   // liao
   sortTable[0x87DF]=14092;   // liao
   sortTable[0x81AB]=14093;   // liao
   sortTable[0x9410]=14094;   // liao
   sortTable[0x8CFF]=14095;   // liao
   sortTable[0x7E5A]=14096;   // liao
   sortTable[0x85D4]=14097;   // liao
   sortTable[0x9DEF]=14098;   // liao
   sortTable[0x8E7D]=14099;   // liao
   sortTable[0x8E58]=14100;   // liao
   sortTable[0x911D]=14101;   // liao
   sortTable[0x91D5]=14102;   // liao
   sortTable[0x53FE]=14103;   // liao
   sortTable[0x5AFD]=14104;   // liao
   sortTable[0x5C1E]=14105;   // liao
   sortTable[0x5C26]=14106;   // liao
   sortTable[0x5C6A]=14107;   // liao
   sortTable[0x5D7A]=14108;   // liao
   sortTable[0x5D9A]=14109;   // liao
   sortTable[0x5D9B]=14110;   // liao
   sortTable[0x5EEB]=14111;   // liao
   sortTable[0x6180]=14112;   // liao
   sortTable[0x61AD]=14113;   // liao
   sortTable[0x6579]=14114;   // liao
   sortTable[0x66B8]=14115;   // liao
   sortTable[0x66E2]=14116;   // liao
   sortTable[0x6F3B]=14117;   // liao
   sortTable[0x7093]=14118;   // liao
   sortTable[0x720E]=14119;   // liao
   sortTable[0x7212]=14120;   // liao
   sortTable[0x7499]=14121;   // liao
   sortTable[0x7AB7]=14122;   // liao
   sortTable[0x957D]=14123;   // liao
   sortTable[0x98C9]=14124;   // liao
   sortTable[0x9ACE]=14125;   // liao
   sortTable[0x70C8]=14126;   // lie
   sortTable[0x5217]=14127;   // lie
   sortTable[0x730E]=14128;   // lie
   sortTable[0x7375]=14129;   // lie
   sortTable[0x88C2]=14130;   // lie
   sortTable[0x54A7]=14131;   // lie
   sortTable[0x52A3]=14132;   // lie
   sortTable[0x6D0C]=14133;   // lie
   sortTable[0x51BD]=14134;   // lie
   sortTable[0x9B23]=14135;   // lie
   sortTable[0x57D2]=14136;   // lie
   sortTable[0x8D94]=14137;   // lie
   sortTable[0x6369]=14138;   // lie
   sortTable[0x8E90]=14139;   // lie
   sortTable[0x9C72]=14140;   // lie
   sortTable[0x8322]=14141;   // lie
   sortTable[0x9D37]=14142;   // lie
   sortTable[0x86DA]=14143;   // lie
   sortTable[0x9B1B]=14144;   // lie
   sortTable[0x8FFE]=14145;   // lie
   sortTable[0x7759]=14146;   // lie
   sortTable[0x811F]=14147;   // lie
   sortTable[0x5120]=14148;   // lie
   sortTable[0x52BD]=14149;   // lie
   sortTable[0x54F7]=14150;   // lie
   sortTable[0x57D3]=14151;   // lie
   sortTable[0x59F4]=14152;   // lie
   sortTable[0x5DE4]=14153;   // lie
   sortTable[0x6312]=14154;   // lie
   sortTable[0x6318]=14155;   // lie
   sortTable[0x64F8]=14156;   // lie
   sortTable[0x6835]=14157;   // lie
   sortTable[0x6BDF]=14158;   // lie
   sortTable[0x6D56]=14159;   // lie
   sortTable[0x70EE]=14160;   // lie
   sortTable[0x716D]=14161;   // lie
   sortTable[0x72A3]=14162;   // lie
   sortTable[0x731F]=14163;   // lie
   sortTable[0x8057]=14164;   // lie
   sortTable[0x98B2]=14165;   // lie
   sortTable[0x9BA4]=14166;   // lie
   sortTable[0x6797]=14167;   // lin
   sortTable[0x4E34]=14168;   // lin
   sortTable[0x81E8]=14169;   // lin
   sortTable[0x6DCB]=14170;   // lin
   sortTable[0x90BB]=14171;   // lin
   sortTable[0x9130]=14172;   // lin
   sortTable[0x78F7]=14173;   // lin
   sortTable[0x9E9F]=14174;   // lin
   sortTable[0x7433]=14175;   // lin
   sortTable[0x9CDE]=14176;   // lin
   sortTable[0x9716]=14177;   // lin
   sortTable[0x51DB]=14178;   // lin
   sortTable[0x8D41]=14179;   // lin
   sortTable[0x5EEA]=14180;   // lin
   sortTable[0x541D]=14181;   // lin
   sortTable[0x62CE]=14182;   // lin
   sortTable[0x7CBC]=14183;   // lin
   sortTable[0x8E8F]=14184;   // lin
   sortTable[0x61D4]=14185;   // lin
   sortTable[0x5D99]=14186;   // lin
   sortTable[0x9074]=14187;   // lin
   sortTable[0x853A]=14188;   // lin
   sortTable[0x9A4E]=14189;   // lin
   sortTable[0x8F9A]=14190;   // lin
   sortTable[0x6AA9]=14191;   // lin
   sortTable[0x5549]=14192;   // lin
   sortTable[0x93FB]=14193;   // lin
   sortTable[0x83FB]=14194;   // lin
   sortTable[0x7B96]=14195;   // lin
   sortTable[0x77DD]=14196;   // lin
   sortTable[0x77B5]=14197;   // lin
   sortTable[0x7CA6]=14198;   // lin
   sortTable[0x8F65]=14199;   // lin
   sortTable[0x81A6]=14200;   // lin
   sortTable[0x9C57]=14201;   // lin
   sortTable[0x9E90]=14202;   // lin
   sortTable[0x95B5]=14203;   // lin
   sortTable[0x7884]=14204;   // lin
   sortTable[0x7E57]=14205;   // lin
   sortTable[0x96A3]=14206;   // lin
   sortTable[0x7FF7]=14207;   // lin
   sortTable[0x85FA]=14208;   // lin
   sortTable[0x8E78]=14209;   // lin
   sortTable[0x4E83]=14210;   // lin
   sortTable[0x50EF]=14211;   // lin
   sortTable[0x51A7]=14212;   // lin
   sortTable[0x51DC]=14213;   // lin
   sortTable[0x53B8]=14214;   // lin
   sortTable[0x58E3]=14215;   // lin
   sortTable[0x5D0A]=14216;   // lin
   sortTable[0x5EE9]=14217;   // lin
   sortTable[0x6061]=14218;   // lin
   sortTable[0x608B]=14219;   // lin
   sortTable[0x61CD]=14220;   // lin
   sortTable[0x649B]=14221;   // lin
   sortTable[0x65B4]=14222;   // lin
   sortTable[0x667D]=14223;   // lin
   sortTable[0x66BD]=14224;   // lin
   sortTable[0x6A49]=14225;   // lin
   sortTable[0x6A81]=14226;   // lin
   sortTable[0x6F7E]=14227;   // lin
   sortTable[0x6F9F]=14228;   // lin
   sortTable[0x7036]=14229;   // lin
   sortTable[0x711B]=14230;   // lin
   sortTable[0x71D0]=14231;   // lin
   sortTable[0x735C]=14232;   // lin
   sortTable[0x7498]=14233;   // lin
   sortTable[0x7510]=14234;   // lin
   sortTable[0x7584]=14235;   // lin
   sortTable[0x75F3]=14236;   // lin
   sortTable[0x765B]=14237;   // lin
   sortTable[0x765D]=14238;   // lin
   sortTable[0x8CC3]=14239;   // lin
   sortTable[0x8E99]=14240;   // lin
   sortTable[0x8EAA]=14241;   // lin
   sortTable[0x8F54]=14242;   // lin
   sortTable[0x9886]=14243;   // ling
   sortTable[0x9818]=14244;   // ling
   sortTable[0x53E6]=14245;   // ling
   sortTable[0x4EE4]=14246;   // ling
   sortTable[0x96F6]=14247;   // ling
   sortTable[0x7075]=14248;   // ling
   sortTable[0x9748]=14249;   // ling
   sortTable[0x94C3]=14250;   // ling
   sortTable[0x9234]=14251;   // ling
   sortTable[0x5CAD]=14252;   // ling
   sortTable[0x5DBA]=14253;   // ling
   sortTable[0x9F84]=14254;   // ling
   sortTable[0x9F61]=14255;   // ling
   sortTable[0x9675]=14256;   // ling
   sortTable[0x4F36]=14257;   // ling
   sortTable[0x73B2]=14258;   // ling
   sortTable[0x51CC]=14259;   // ling
   sortTable[0x83F1]=14260;   // ling
   sortTable[0x7EEB]=14261;   // ling
   sortTable[0x86C9]=14262;   // ling
   sortTable[0x7FCE]=14263;   // ling
   sortTable[0x5464]=14264;   // ling
   sortTable[0x8046]=14265;   // ling
   sortTable[0x82D3]=14266;   // ling
   sortTable[0x7F9A]=14267;   // ling
   sortTable[0x6CE0]=14268;   // ling
   sortTable[0x74F4]=14269;   // ling
   sortTable[0x68C2]=14270;   // ling
   sortTable[0x9143]=14271;   // ling
   sortTable[0x56F9]=14272;   // ling
   sortTable[0x9CAE]=14273;   // ling
   sortTable[0x8EE8]=14274;   // ling
   sortTable[0x67C3]=14275;   // ling
   sortTable[0x9E30]=14276;   // ling
   sortTable[0x7B2D]=14277;   // ling
   sortTable[0x91BD]=14278;   // ling
   sortTable[0x8232]=14279;   // ling
   sortTable[0x971D]=14280;   // ling
   sortTable[0x888A]=14281;   // ling
   sortTable[0x971B]=14282;   // ling
   sortTable[0x8DC9]=14283;   // ling
   sortTable[0x797E]=14284;   // ling
   sortTable[0x84E4]=14285;   // ling
   sortTable[0x970A]=14286;   // ling
   sortTable[0x9EA2]=14287;   // ling
   sortTable[0x9F97]=14288;   // ling
   sortTable[0x8626]=14289;   // ling
   sortTable[0x99D6]=14290;   // ling
   sortTable[0x9D12]=14291;   // ling
   sortTable[0x8506]=14292;   // ling
   sortTable[0x8A45]=14293;   // ling
   sortTable[0x9F62]=14294;   // ling
   sortTable[0x79E2]=14295;   // ling
   sortTable[0x7ADB]=14296;   // ling
   sortTable[0x9302]=14297;   // ling
   sortTable[0x963E]=14298;   // ling
   sortTable[0x9B7F]=14299;   // ling
   sortTable[0x5222]=14300;   // ling
   sortTable[0x577D]=14301;   // ling
   sortTable[0x590C]=14302;   // ling
   sortTable[0x59C8]=14303;   // ling
   sortTable[0x5A48]=14304;   // ling
   sortTable[0x5B41]=14305;   // ling
   sortTable[0x5CBA]=14306;   // ling
   sortTable[0x5F7E]=14307;   // ling
   sortTable[0x6395]=14308;   // ling
   sortTable[0x6624]=14309;   // ling
   sortTable[0x670E]=14310;   // ling
   sortTable[0x6AFA]=14311;   // ling
   sortTable[0x6B1E]=14312;   // ling
   sortTable[0x6DE9]=14313;   // ling
   sortTable[0x6FAA]=14314;   // ling
   sortTable[0x702E]=14315;   // ling
   sortTable[0x70A9]=14316;   // ling
   sortTable[0x71EF]=14317;   // ling
   sortTable[0x7227]=14318;   // ling
   sortTable[0x72D1]=14319;   // ling
   sortTable[0x740C]=14320;   // ling
   sortTable[0x768A]=14321;   // ling
   sortTable[0x7831]=14322;   // ling
   sortTable[0x7D37]=14323;   // ling
   sortTable[0x7DBE]=14324;   // ling
   sortTable[0x8576]=14325;   // ling
   sortTable[0x8851]=14326;   // ling
   sortTable[0x88EC]=14327;   // ling
   sortTable[0x959D]=14328;   // ling
   sortTable[0x9717]=14329;   // ling
   sortTable[0x9BEA]=14330;   // ling
   sortTable[0x9E77]=14331;   // ling
   sortTable[0x6D41]=14332;   // liu
   sortTable[0x516D]=14333;   // liu
   sortTable[0x7559]=14334;   // liu
   sortTable[0x69B4]=14335;   // liu
   sortTable[0x67F3]=14336;   // liu
   sortTable[0x6E9C]=14337;   // liu
   sortTable[0x786B]=14338;   // liu
   sortTable[0x5218]=14339;   // liu
   sortTable[0x7624]=14340;   // liu
   sortTable[0x7409]=14341;   // liu
   sortTable[0x6D4F]=14342;   // liu
   sortTable[0x7198]=14343;   // liu
   sortTable[0x954F]=14344;   // liu
   sortTable[0x65D2]=14345;   // liu
   sortTable[0x950D]=14346;   // liu
   sortTable[0x9E68]=14347;   // liu
   sortTable[0x7EFA]=14348;   // liu
   sortTable[0x9560]=14349;   // liu
   sortTable[0x905B]=14350;   // liu
   sortTable[0x938F]=14351;   // liu
   sortTable[0x9A9D]=14352;   // liu
   sortTable[0x998F]=14353;   // liu
   sortTable[0x8E53]=14354;   // liu
   sortTable[0x9724]=14355;   // liu
   sortTable[0x98D7]=14356;   // liu
   sortTable[0x9E60]=14357;   // liu
   sortTable[0x7F76]=14358;   // liu
   sortTable[0x7F80]=14359;   // liu
   sortTable[0x98C2]=14360;   // liu
   sortTable[0x9C21]=14361;   // liu
   sortTable[0x7FCF]=14362;   // liu
   sortTable[0x93D0]=14363;   // liu
   sortTable[0x99F5]=14364;   // liu
   sortTable[0x9DB9]=14365;   // liu
   sortTable[0x87C9]=14366;   // liu
   sortTable[0x99E0]=14367;   // liu
   sortTable[0x98F9]=14368;   // liu
   sortTable[0x85F0]=14369;   // liu
   sortTable[0x9A51]=14370;   // liu
   sortTable[0x9E8D]=14371;   // liu
   sortTable[0x7DB9]=14372;   // liu
   sortTable[0x98C0]=14373;   // liu
   sortTable[0x9B38]=14374;   // liu
   sortTable[0x5289]=14375;   // liu
   sortTable[0x56A0]=14376;   // liu
   sortTable[0x586F]=14377;   // liu
   sortTable[0x5AB9]=14378;   // liu
   sortTable[0x5B3C]=14379;   // liu
   sortTable[0x5D67]=14380;   // liu
   sortTable[0x5EC7]=14381;   // liu
   sortTable[0x61F0]=14382;   // liu
   sortTable[0x65C8]=14383;   // liu
   sortTable[0x6801]=14384;   // liu
   sortTable[0x685E]=14385;   // liu
   sortTable[0x687A]=14386;   // liu
   sortTable[0x6A4A]=14387;   // liu
   sortTable[0x6A6E]=14388;   // liu
   sortTable[0x6CA0]=14389;   // liu
   sortTable[0x6F91]=14390;   // liu
   sortTable[0x700F]=14391;   // liu
   sortTable[0x71AE]=14392;   // liu
   sortTable[0x73CB]=14393;   // liu
   sortTable[0x7460]=14394;   // liu
   sortTable[0x746C]=14395;   // liu
   sortTable[0x74A2]=14396;   // liu
   sortTable[0x7542]=14397;   // liu
   sortTable[0x7544]=14398;   // liu
   sortTable[0x7571]=14399;   // liu
   sortTable[0x7581]=14400;   // liu
   sortTable[0x7645]=14401;   // liu
   sortTable[0x78C2]=14402;   // liu
   sortTable[0x78DF]=14403;   // liu
   sortTable[0x84A5]=14404;   // liu
   sortTable[0x84C5]=14405;   // liu
   sortTable[0x88D7]=14406;   // liu
   sortTable[0x925A]=14407;   // liu
   sortTable[0x92F6]=14408;   // liu
   sortTable[0x93A6]=14409;   // liu
   sortTable[0x9402]=14410;   // liu
   sortTable[0x96E1]=14411;   // liu
   sortTable[0x98C5]=14412;   // liu
   sortTable[0x993E]=14413;   // liu
   sortTable[0x9A2E]=14414;   // liu
   sortTable[0x9DDA]=14415;   // liu
   sortTable[0x56D6]=14416;   // lo
   sortTable[0x9F99]=14417;   // long
   sortTable[0x9F8D]=14418;   // long
   sortTable[0x62E2]=14419;   // long
   sortTable[0x650F]=14420;   // long
   sortTable[0x9686]=14421;   // long
   sortTable[0x7B3C]=14422;   // long
   sortTable[0x7C60]=14423;   // long
   sortTable[0x80E7]=14424;   // long
   sortTable[0x6727]=14425;   // long
   sortTable[0x5784]=14426;   // long
   sortTable[0x58DF]=14427;   // long
   sortTable[0x5499]=14428;   // long
   sortTable[0x7ABF]=14429;   // long
   sortTable[0x56A8]=14430;   // long
   sortTable[0x73D1]=14431;   // long
   sortTable[0x74CF]=14432;   // long
   sortTable[0x9647]=14433;   // long
   sortTable[0x804B]=14434;   // long
   sortTable[0x830F]=14435;   // long
   sortTable[0x680A]=14436;   // long
   sortTable[0x783B]=14437;   // long
   sortTable[0x6CF7]=14438;   // long
   sortTable[0x5785]=14439;   // long
   sortTable[0x772C]=14440;   // long
   sortTable[0x7643]=14441;   // long
   sortTable[0x9468]=14442;   // long
   sortTable[0x9F92]=14443;   // long
   sortTable[0x8E98]=14444;   // long
   sortTable[0x9E17]=14445;   // long
   sortTable[0x7932]=14446;   // long
   sortTable[0x9F93]=14447;   // long
   sortTable[0x9733]=14448;   // long
   sortTable[0x7C3C]=14449;   // long
   sortTable[0x856F]=14450;   // long
   sortTable[0x8971]=14451;   // long
   sortTable[0x77D3]=14452;   // long
   sortTable[0x9A61]=14453;   // long
   sortTable[0x7931]=14454;   // long
   sortTable[0x882A]=14455;   // long
   sortTable[0x7BE2]=14456;   // long
   sortTable[0x7BED]=14457;   // long
   sortTable[0x807E]=14458;   // long
   sortTable[0x882C]=14459;   // long
   sortTable[0x5131]=14460;   // long
   sortTable[0x54E2]=14461;   // long
   sortTable[0x58E0]=14462;   // long
   sortTable[0x5C78]=14463;   // long
   sortTable[0x5D90]=14464;   // long
   sortTable[0x5DC3]=14465;   // long
   sortTable[0x5DC4]=14466;   // long
   sortTable[0x5FBF]=14467;   // long
   sortTable[0x6335]=14468;   // long
   sortTable[0x663D]=14469;   // long
   sortTable[0x66E8]=14470;   // long
   sortTable[0x6887]=14471;   // long
   sortTable[0x69DE]=14472;   // long
   sortTable[0x6AF3]=14473;   // long
   sortTable[0x6E70]=14474;   // long
   sortTable[0x6EDD]=14475;   // long
   sortTable[0x6F0B]=14476;   // long
   sortTable[0x7027]=14477;   // long
   sortTable[0x7216]=14478;   // long
   sortTable[0x7AC9]=14479;   // long
   sortTable[0x7ADC]=14480;   // long
   sortTable[0x8622]=14481;   // long
   sortTable[0x8C45]=14482;   // long
   sortTable[0x8D1A]=14483;   // long
   sortTable[0x93E7]=14484;   // long
   sortTable[0x96B4]=14485;   // long
   sortTable[0x9747]=14486;   // long
   sortTable[0x697C]=14487;   // lou
   sortTable[0x6A13]=14488;   // lou
   sortTable[0x6F0F]=14489;   // lou
   sortTable[0x55BD]=14490;   // lou
   sortTable[0x560D]=14491;   // lou
   sortTable[0x6402]=14492;   // lou
   sortTable[0x645F]=14493;   // lou
   sortTable[0x964B]=14494;   // lou
   sortTable[0x5A04]=14495;   // lou
   sortTable[0x9542]=14496;   // lou
   sortTable[0x9AC5]=14497;   // lou
   sortTable[0x507B]=14498;   // lou
   sortTable[0x7BD3]=14499;   // lou
   sortTable[0x5D5D]=14500;   // lou
   sortTable[0x877C]=14501;   // lou
   sortTable[0x7618]=14502;   // lou
   sortTable[0x848C]=14503;   // lou
   sortTable[0x8027]=14504;   // lou
   sortTable[0x825B]=14505;   // lou
   sortTable[0x97BB]=14506;   // lou
   sortTable[0x8B31]=14507;   // lou
   sortTable[0x7C0D]=14508;   // lou
   sortTable[0x851E]=14509;   // lou
   sortTable[0x9071]=14510;   // lou
   sortTable[0x93E4]=14511;   // lou
   sortTable[0x779C]=14512;   // lou
   sortTable[0x9ACF]=14513;   // lou
   sortTable[0x50C2]=14514;   // lou
   sortTable[0x5245]=14515;   // lou
   sortTable[0x587F]=14516;   // lou
   sortTable[0x5A41]=14517;   // lou
   sortTable[0x5C5A]=14518;   // lou
   sortTable[0x5D81]=14519;   // lou
   sortTable[0x5ED4]=14520;   // lou
   sortTable[0x617A]=14521;   // lou
   sortTable[0x6E87]=14522;   // lou
   sortTable[0x6F0A]=14523;   // lou
   sortTable[0x71A1]=14524;   // lou
   sortTable[0x750A]=14525;   // lou
   sortTable[0x763A]=14526;   // lou
   sortTable[0x763B]=14527;   // lou
   sortTable[0x802C]=14528;   // lou
   sortTable[0x87BB]=14529;   // lou
   sortTable[0x8EC1]=14530;   // lou
   sortTable[0x8DEF]=14531;   // lu
   sortTable[0x9732]=14532;   // lu
   sortTable[0x9646]=14533;   // lu
   sortTable[0x9678]=14534;   // lu
   sortTable[0x7089]=14535;   // lu
   sortTable[0x7210]=14536;   // lu
   sortTable[0x5F55]=14537;   // lu
   sortTable[0x9304]=14538;   // lu
   sortTable[0x82A6]=14539;   // lu
   sortTable[0x8606]=14540;   // lu
   sortTable[0x864F]=14541;   // lu
   sortTable[0x865C]=14542;   // lu
   sortTable[0x788C]=14543;   // lu
   sortTable[0x9E7F]=14544;   // lu
   sortTable[0x9C81]=14545;   // lu
   sortTable[0x5362]=14546;   // lu
   sortTable[0x7984]=14547;   // lu
   sortTable[0x5E90]=14548;   // lu
   sortTable[0x64B8]=14549;   // lu
   sortTable[0x622E]=14550;   // lu
   sortTable[0x8D42]=14551;   // lu
   sortTable[0x6F5E]=14552;   // lu
   sortTable[0x5364]=14553;   // lu
   sortTable[0x80EA]=14554;   // lu
   sortTable[0x565C]=14555;   // lu
   sortTable[0x9885]=14556;   // lu
   sortTable[0x9E93]=14557;   // lu
   sortTable[0x63B3]=14558;   // lu
   sortTable[0x8F82]=14559;   // lu
   sortTable[0x6A79]=14560;   // lu
   sortTable[0x6E0C]=14561;   // lu
   sortTable[0x9E6D]=14562;   // lu
   sortTable[0x7B93]=14563;   // lu
   sortTable[0x902F]=14564;   // lu
   sortTable[0x7490]=14565;   // lu
   sortTable[0x6F09]=14566;   // lu
   sortTable[0x8F98]=14567;   // lu
   sortTable[0x9C88]=14568;   // lu
   sortTable[0x6CF8]=14569;   // lu
   sortTable[0x823B]=14570;   // lu
   sortTable[0x9E2C]=14571;   // lu
   sortTable[0x680C]=14572;   // lu
   sortTable[0x8F73]=14573;   // lu
   sortTable[0x5786]=14574;   // lu
   sortTable[0x7C0F]=14575;   // lu
   sortTable[0x76DD]=14576;   // lu
   sortTable[0x6C07]=14577;   // lu
   sortTable[0x9A04]=14578;   // lu
   sortTable[0x9181]=14579;   // lu
   sortTable[0x9A3C]=14580;   // lu
   sortTable[0x9871]=14581;   // lu
   sortTable[0x83C9]=14582;   // lu
   sortTable[0x7A11]=14583;   // lu
   sortTable[0x9565]=14584;   // lu
   sortTable[0x7C2C]=14585;   // lu
   sortTable[0x7E91]=14586;   // lu
   sortTable[0x9229]=14587;   // lu
   sortTable[0x7CB6]=14588;   // lu
   sortTable[0x8263]=14589;   // lu
   sortTable[0x84FE]=14590;   // lu
   sortTable[0x8557]=14591;   // lu
   sortTable[0x7849]=14592;   // lu
   sortTable[0x7A4B]=14593;   // lu
   sortTable[0x7C59]=14594;   // lu
   sortTable[0x9332]=14595;   // lu
   sortTable[0x9DFA]=14596;   // lu
   sortTable[0x9B6F]=14597;   // lu
   sortTable[0x9E15]=14598;   // lu
   sortTable[0x9EF8]=14599;   // lu
   sortTable[0x7769]=14600;   // lu
   sortTable[0x93D5]=14601;   // lu
   sortTable[0x946A]=14602;   // lu
   sortTable[0x797F]=14603;   // lu
   sortTable[0x7F4F]=14604;   // lu
   sortTable[0x8826]=14605;   // lu
   sortTable[0x9E75]=14606;   // lu
   sortTable[0x76E7]=14607;   // lu
   sortTable[0x77D1]=14608;   // lu
   sortTable[0x7875]=14609;   // lu
   sortTable[0x78E0]=14610;   // lu
   sortTable[0x81DA]=14611;   // lu
   sortTable[0x826B]=14612;   // lu
   sortTable[0x8642]=14613;   // lu
   sortTable[0x87B0]=14614;   // lu
   sortTable[0x8CC2]=14615;   // lu
   sortTable[0x8E57]=14616;   // lu
   sortTable[0x93F4]=14617;   // lu
   sortTable[0x4F93]=14618;   // lu
   sortTable[0x50C7]=14619;   // lu
   sortTable[0x5279]=14620;   // lu
   sortTable[0x52CE]=14621;   // lu
   sortTable[0x52E0]=14622;   // lu
   sortTable[0x5695]=14623;   // lu
   sortTable[0x56A7]=14624;   // lu
   sortTable[0x5725]=14625;   // lu
   sortTable[0x5774]=14626;   // lu
   sortTable[0x5876]=14627;   // lu
   sortTable[0x5877]=14628;   // lu
   sortTable[0x58DA]=14629;   // lu
   sortTable[0x5A3D]=14630;   // lu
   sortTable[0x5CCD]=14631;   // lu
   sortTable[0x5ED8]=14632;   // lu
   sortTable[0x5EEC]=14633;   // lu
   sortTable[0x5F54]=14634;   // lu
   sortTable[0x645D]=14635;   // lu
   sortTable[0x64C4]=14636;   // lu
   sortTable[0x64FC]=14637;   // lu
   sortTable[0x650E]=14638;   // lu
   sortTable[0x66E5]=14639;   // lu
   sortTable[0x67A6]=14640;   // lu
   sortTable[0x6902]=14641;   // lu
   sortTable[0x6A10]=14642;   // lu
   sortTable[0x6A1A]=14643;   // lu
   sortTable[0x6AD3]=14644;   // lu
   sortTable[0x6ADA]=14645;   // lu
   sortTable[0x6AE8]=14646;   // lu
   sortTable[0x6C0C]=14647;   // lu
   sortTable[0x6DD5]=14648;   // lu
   sortTable[0x6DE5]=14649;   // lu
   sortTable[0x6EF7]=14650;   // lu
   sortTable[0x6F9B]=14651;   // lu
   sortTable[0x7002]=14652;   // lu
   sortTable[0x7018]=14653;   // lu
   sortTable[0x719D]=14654;   // lu
   sortTable[0x7379]=14655;   // lu
   sortTable[0x7388]=14656;   // lu
   sortTable[0x742D]=14657;   // lu
   sortTable[0x74B7]=14658;   // lu
   sortTable[0x74D0]=14659;   // lu
   sortTable[0x752A]=14660;   // lu
   sortTable[0x7C36]=14661;   // lu
   sortTable[0x7C5A]=14662;   // lu
   sortTable[0x8194]=14663;   // lu
   sortTable[0x822E]=14664;   // lu
   sortTable[0x826A]=14665;   // lu
   sortTable[0x850D]=14666;   // lu
   sortTable[0x89EE]=14667;   // lu
   sortTable[0x8DA2]=14668;   // lu
   sortTable[0x8E1B]=14669;   // lu
   sortTable[0x8F46]=14670;   // lu
   sortTable[0x8F64]=14671;   // lu
   sortTable[0x9334]=14672;   // lu
   sortTable[0x93C0]=14673;   // lu
   sortTable[0x942A]=14674;   // lu
   sortTable[0x9465]=14675;   // lu
   sortTable[0x9AD7]=14676;   // lu
   sortTable[0x9B72]=14677;   // lu
   sortTable[0x9BE5]=14678;   // lu
   sortTable[0x9C78]=14679;   // lu
   sortTable[0x9D66]=14680;   // lu
   sortTable[0x9D71]=14681;   // lu
   sortTable[0x4E71]=14682;   // luan
   sortTable[0x4E82]=14683;   // luan
   sortTable[0x5375]=14684;   // luan
   sortTable[0x5CE6]=14685;   // luan
   sortTable[0x5DD2]=14686;   // luan
   sortTable[0x9E3E]=14687;   // luan
   sortTable[0x631B]=14688;   // luan
   sortTable[0x92AE]=14689;   // luan
   sortTable[0x683E]=14690;   // luan
   sortTable[0x6EE6]=14691;   // luan
   sortTable[0x5B6A]=14692;   // luan
   sortTable[0x8114]=14693;   // luan
   sortTable[0x5A08]=14694;   // luan
   sortTable[0x91E0]=14695;   // luan
   sortTable[0x947E]=14696;   // luan
   sortTable[0x7F89]=14697;   // luan
   sortTable[0x81E0]=14698;   // luan
   sortTable[0x864A]=14699;   // luan
   sortTable[0x571D]=14700;   // luan
   sortTable[0x571E]=14701;   // luan
   sortTable[0x5971]=14702;   // luan
   sortTable[0x5B4C]=14703;   // luan
   sortTable[0x5B7F]=14704;   // luan
   sortTable[0x6523]=14705;   // luan
   sortTable[0x66EB]=14706;   // luan
   sortTable[0x6B12]=14707;   // luan
   sortTable[0x7053]=14708;   // luan
   sortTable[0x7064]=14709;   // luan
   sortTable[0x7674]=14710;   // luan
   sortTable[0x7675]=14711;   // luan
   sortTable[0x9D49]=14712;   // luan
   sortTable[0x9E1E]=14713;   // luan
   sortTable[0x950A]=14714;   // lüe
   sortTable[0x5719]=14715;   // lüe
   sortTable[0x64FD]=14716;   // lüe
   sortTable[0x7567]=14717;   // lüe
   sortTable[0x7A24]=14718;   // lüe
   sortTable[0x92DD]=14719;   // lüe
   sortTable[0x92E2]=14720;   // lüe
   sortTable[0x8BBA]=14721;   // lun
   sortTable[0x8AD6]=14722;   // lun
   sortTable[0x8F6E]=14723;   // lun
   sortTable[0x8F2A]=14724;   // lun
   sortTable[0x7EB6]=14725;   // lun
   sortTable[0x7DB8]=14726;   // lun
   sortTable[0x62A1]=14727;   // lun
   sortTable[0x6384]=14728;   // lun
   sortTable[0x4F26]=14729;   // lun
   sortTable[0x4ED1]=14730;   // lun
   sortTable[0x6CA6]=14731;   // lun
   sortTable[0x56F5]=14732;   // lun
   sortTable[0x7A10]=14733;   // lun
   sortTable[0x83D5]=14734;   // lun
   sortTable[0x9300]=14735;   // lun
   sortTable[0x78EE]=14736;   // lun
   sortTable[0x8140]=14737;   // lun
   sortTable[0x9BE9]=14738;   // lun
   sortTable[0x4F96]=14739;   // lun
   sortTable[0x502B]=14740;   // lun
   sortTable[0x5707]=14741;   // lun
   sortTable[0x57E8]=14742;   // lun
   sortTable[0x5A68]=14743;   // lun
   sortTable[0x5D18]=14744;   // lun
   sortTable[0x5D19]=14745;   // lun
   sortTable[0x60C0]=14746;   // lun
   sortTable[0x68C6]=14747;   // lun
   sortTable[0x6DEA]=14748;   // lun
   sortTable[0x6EA3]=14749;   // lun
   sortTable[0x7896]=14750;   // lun
   sortTable[0x8023]=14751;   // lun
   sortTable[0x8726]=14752;   // lun
   sortTable[0x8E1A]=14753;   // lun
   sortTable[0x966F]=14754;   // lun
   sortTable[0x843D]=14755;   // luo
   sortTable[0x9A86]=14756;   // luo
   sortTable[0x99F1]=14757;   // luo
   sortTable[0x9523]=14758;   // luo
   sortTable[0x947C]=14759;   // luo
   sortTable[0x7EDC]=14760;   // luo
   sortTable[0x7D61]=14761;   // luo
   sortTable[0x841D]=14762;   // luo
   sortTable[0x863F]=14763;   // luo
   sortTable[0x903B]=14764;   // luo
   sortTable[0x908F]=14765;   // luo
   sortTable[0x9AA1]=14766;   // luo
   sortTable[0x9A3E]=14767;   // luo
   sortTable[0x7BA9]=14768;   // luo
   sortTable[0x7C6E]=14769;   // luo
   sortTable[0x7F57]=14770;   // luo
   sortTable[0x6D1B]=14771;   // luo
   sortTable[0x88F8]=14772;   // luo
   sortTable[0x87BA]=14773;   // luo
   sortTable[0x6924]=14774;   // luo
   sortTable[0x96D2]=14775;   // luo
   sortTable[0x502E]=14776;   // luo
   sortTable[0x6CFA]=14777;   // luo
   sortTable[0x645E]=14778;   // luo
   sortTable[0x73DE]=14779;   // luo
   sortTable[0x8366]=14780;   // luo
   sortTable[0x6F2F]=14781;   // luo
   sortTable[0x8803]=14782;   // luo
   sortTable[0x8136]=14783;   // luo
   sortTable[0x7321]=14784;   // luo
   sortTable[0x7630]=14785;   // luo
   sortTable[0x81DD]=14786;   // luo
   sortTable[0x84CF]=14787;   // luo
   sortTable[0x8999]=14788;   // luo
   sortTable[0x9559]=14789;   // luo
   sortTable[0x9960]=14790;   // luo
   sortTable[0x9D3C]=14791;   // luo
   sortTable[0x9A58]=14792;   // luo
   sortTable[0x7F85]=14793;   // luo
   sortTable[0x93CD]=14794;   // luo
   sortTable[0x9E01]=14795;   // luo
   sortTable[0x8EB6]=14796;   // luo
   sortTable[0x9D45]=14797;   // luo
   sortTable[0x7B3F]=14798;   // luo
   sortTable[0x89BC]=14799;   // luo
   sortTable[0x9BA5]=14800;   // luo
   sortTable[0x5138]=14801;   // luo
   sortTable[0x5246]=14802;   // luo
   sortTable[0x5570]=14803;   // luo
   sortTable[0x55E0]=14804;   // luo
   sortTable[0x56C9]=14805;   // luo
   sortTable[0x5CC8]=14806;   // luo
   sortTable[0x651E]=14807;   // luo
   sortTable[0x66EA]=14808;   // luo
   sortTable[0x6B0F]=14809;   // luo
   sortTable[0x6D1C]=14810;   // luo
   sortTable[0x6FFC]=14811;   // luo
   sortTable[0x7296]=14812;   // luo
   sortTable[0x7380]=14813;   // luo
   sortTable[0x7673]=14814;   // luo
   sortTable[0x7866]=14815;   // luo
   sortTable[0x7E99]=14816;   // luo
   sortTable[0x7F56]=14817;   // luo
   sortTable[0x8161]=14818;   // luo
   sortTable[0x89B6]=14819;   // luo
   sortTable[0x927B]=14820;   // luo
   sortTable[0x9831]=14821;   // luo
   sortTable[0x5F8B]=14822;   // lv
   sortTable[0x7EFF]=14823;   // lv
   sortTable[0x7DA0]=14824;   // lv
   sortTable[0x7387]=14825;   // lv
   sortTable[0x65C5]=14826;   // lv
   sortTable[0x8651]=14827;   // lv
   sortTable[0x616E]=14828;   // lv
   sortTable[0x9A74]=14829;   // lv
   sortTable[0x9A62]=14830;   // lv
   sortTable[0x94DD]=14831;   // lv
   sortTable[0x92C1]=14832;   // lv
   sortTable[0x7F15]=14833;   // lv
   sortTable[0x7E37]=14834;   // lv
   sortTable[0x5415]=14835;   // lv
   sortTable[0x5C65]=14836;   // lv
   sortTable[0x5C61]=14837;   // lv
   sortTable[0x4FA3]=14838;   // lv
   sortTable[0x95FE]=14839;   // lv
   sortTable[0x6C2F]=14840;   // lv
   sortTable[0x6EE4]=14841;   // lv
   sortTable[0x6988]=14842;   // lv
   sortTable[0x634B]=14843;   // lv
   sortTable[0x8182]=14844;   // lv
   sortTable[0x891B]=14845;   // lv
   sortTable[0x844E]=14846;   // lv
   sortTable[0x85D8]=14847;   // lv
   sortTable[0x81A2]=14848;   // lv
   sortTable[0x7A06]=14849;   // lv
   sortTable[0x7BBB]=14850;   // lv
   sortTable[0x9462]=14851;   // lv
   sortTable[0x7D7D]=14852;   // lv
   sortTable[0x95AD]=14853;   // lv
   sortTable[0x7E42]=14854;   // lv
   sortTable[0x819F]=14855;   // lv
   sortTable[0x7A6D]=14856;   // lv
   sortTable[0x7A5E]=14857;   // lv
   sortTable[0x90D8]=14858;   // lv
   sortTable[0x8190]=14859;   // lv
   sortTable[0x8938]=14860;   // lv
   sortTable[0x99BF]=14861;   // lv
   sortTable[0x9DDC]=14862;   // lv
   sortTable[0x4FB6]=14863;   // lv
   sortTable[0x5122]=14864;   // lv
   sortTable[0x52F4]=14865;   // lv
   sortTable[0x5442]=14866;   // lv
   sortTable[0x578F]=14867;   // lv
   sortTable[0x5BFD]=14868;   // lv
   sortTable[0x5C62]=14869;   // lv
   sortTable[0x5D42]=14870;   // lv
   sortTable[0x6314]=14871;   // lv
   sortTable[0x635B]=14872;   // lv
   sortTable[0x68A0]=14873;   // lv
   sortTable[0x6AD6]=14874;   // lv
   sortTable[0x6C00]=14875;   // lv
   sortTable[0x6FFE]=14876;   // lv
   sortTable[0x7112]=14877;   // lv
   sortTable[0x7208]=14878;   // lv
   sortTable[0x7963]=14879;   // lv
   sortTable[0x7DD1]=14880;   // lv
   sortTable[0x5463]=15360;   // ḿ
   sortTable[0x5417]=15361;   // ma
   sortTable[0x55CE]=15362;   // ma
   sortTable[0x5988]=15363;   // ma
   sortTable[0x5ABD]=15364;   // ma
   sortTable[0x9A6C]=15365;   // ma
   sortTable[0x99AC]=15366;   // ma
   sortTable[0x561B]=15367;   // ma
   sortTable[0x9A82]=15368;   // ma
   sortTable[0x7F75]=15369;   // ma
   sortTable[0x9EBB]=15370;   // ma
   sortTable[0x7801]=15371;   // ma
   sortTable[0x78BC]=15372;   // ma
   sortTable[0x8682]=15373;   // ma
   sortTable[0x879E]=15374;   // ma
   sortTable[0x87C6]=15375;   // ma
   sortTable[0x739B]=15376;   // ma
   sortTable[0x5B37]=15377;   // ma
   sortTable[0x6769]=15378;   // ma
   sortTable[0x72B8]=15379;   // ma
   sortTable[0x551B]=15380;   // ma
   sortTable[0x7943]=15381;   // ma
   sortTable[0x7770]=15382;   // ma
   sortTable[0x79A1]=15383;   // ma
   sortTable[0x8534]=15384;   // ma
   sortTable[0x87C7]=15385;   // ma
   sortTable[0x9C22]=15386;   // ma
   sortTable[0x93B7]=15387;   // ma
   sortTable[0x99E1]=15388;   // ma
   sortTable[0x4E87]=15389;   // ma
   sortTable[0x508C]=15390;   // ma
   sortTable[0x561C]=15391;   // ma
   sortTable[0x5AF2]=15392;   // ma
   sortTable[0x5B24]=15393;   // ma
   sortTable[0x5B56]=15394;   // ma
   sortTable[0x69AA]=15395;   // ma
   sortTable[0x6EA4]=15396;   // ma
   sortTable[0x7298]=15397;   // ma
   sortTable[0x7341]=15398;   // ma
   sortTable[0x746A]=15399;   // ma
   sortTable[0x75F2]=15400;   // ma
   sortTable[0x7923]=15401;   // ma
   sortTable[0x9064]=15402;   // ma
   sortTable[0x9581]=15403;   // ma
   sortTable[0x9B15]=15404;   // ma
   sortTable[0x9DCC]=15405;   // ma
   sortTable[0x4E70]=15406;   // mai
   sortTable[0x8CB7]=15407;   // mai
   sortTable[0x5356]=15408;   // mai
   sortTable[0x8CE3]=15409;   // mai
   sortTable[0x9EA6]=15410;   // mai
   sortTable[0x9EA5]=15411;   // mai
   sortTable[0x8109]=15412;   // mai
   sortTable[0x8108]=15413;   // mai
   sortTable[0x57CB]=15414;   // mai
   sortTable[0x8FC8]=15415;   // mai
   sortTable[0x9081]=15416;   // mai
   sortTable[0x973E]=15417;   // mai
   sortTable[0x52A2]=15418;   // mai
   sortTable[0x9721]=15419;   // mai
   sortTable[0x836C]=15420;   // mai
   sortTable[0x85B6]=15421;   // mai
   sortTable[0x9DF6]=15422;   // mai
   sortTable[0x8552]=15423;   // mai
   sortTable[0x4F45]=15424;   // mai
   sortTable[0x52F1]=15425;   // mai
   sortTable[0x562A]=15426;   // mai
   sortTable[0x58F2]=15427;   // mai
   sortTable[0x8847]=15428;   // mai
   sortTable[0x9722]=15429;   // mai
   sortTable[0x6EE1]=15430;   // man
   sortTable[0x6EFF]=15431;   // man
   sortTable[0x6162]=15432;   // man
   sortTable[0x6F2B]=15433;   // man
   sortTable[0x9992]=15434;   // man
   sortTable[0x9945]=15435;   // man
   sortTable[0x7792]=15436;   // man
   sortTable[0x779E]=15437;   // man
   sortTable[0x66FC]=15438;   // man
   sortTable[0x86EE]=15439;   // man
   sortTable[0x8513]=15440;   // man
   sortTable[0x8C29]=15441;   // man
   sortTable[0x5E54]=15442;   // man
   sortTable[0x71B3]=15443;   // man
   sortTable[0x9794]=15444;   // man
   sortTable[0x7F26]=15445;   // man
   sortTable[0x9CD7]=15446;   // man
   sortTable[0x87A8]=15447;   // man
   sortTable[0x989F]=15448;   // man
   sortTable[0x5881]=15449;   // man
   sortTable[0x9B18]=15450;   // man
   sortTable[0x9124]=15451;   // man
   sortTable[0x9558]=15452;   // man
   sortTable[0x774C]=15453;   // man
   sortTable[0x883B]=15454;   // man
   sortTable[0x93CB]=15455;   // man
   sortTable[0x77D5]=15456;   // man
   sortTable[0x8954]=15457;   // man
   sortTable[0x8504]=15458;   // man
   sortTable[0x8B3E]=15459;   // man
   sortTable[0x9B17]=15460;   // man
   sortTable[0x9C3B]=15461;   // man
   sortTable[0x50C8]=15462;   // man
   sortTable[0x59CF]=15463;   // man
   sortTable[0x5ADA]=15464;   // man
   sortTable[0x5C58]=15465;   // man
   sortTable[0x6097]=15466;   // man
   sortTable[0x6172]=15467;   // man
   sortTable[0x6471]=15468;   // man
   sortTable[0x69FE]=15469;   // man
   sortTable[0x6A20]=15470;   // man
   sortTable[0x6E80]=15471;   // man
   sortTable[0x6FB7]=15472;   // man
   sortTable[0x734C]=15473;   // man
   sortTable[0x7E35]=15474;   // man
   sortTable[0x8630]=15475;   // man
   sortTable[0x87CE]=15476;   // man
   sortTable[0x93DD]=15477;   // man
   sortTable[0x9862]=15478;   // man
   sortTable[0x5FD9]=15479;   // mang
   sortTable[0x832B]=15480;   // mang
   sortTable[0x76F2]=15481;   // mang
   sortTable[0x8292]=15482;   // mang
   sortTable[0x6C13]=15483;   // mang
   sortTable[0x83BD]=15484;   // mang
   sortTable[0x87D2]=15485;   // mang
   sortTable[0x9099]=15486;   // mang
   sortTable[0x786D]=15487;   // mang
   sortTable[0x99F9]=15488;   // mang
   sortTable[0x94D3]=15489;   // mang
   sortTable[0x6F2D]=15490;   // mang
   sortTable[0x83BE]=15491;   // mang
   sortTable[0x7865]=15492;   // mang
   sortTable[0x86D6]=15493;   // mang
   sortTable[0x833B]=15494;   // mang
   sortTable[0x7B00]=15495;   // mang
   sortTable[0x880E]=15496;   // mang
   sortTable[0x91EF]=15497;   // mang
   sortTable[0x5402]=15498;   // mang
   sortTable[0x54E4]=15499;   // mang
   sortTable[0x58FE]=15500;   // mang
   sortTable[0x5A0F]=15501;   // mang
   sortTable[0x5C28]=15502;   // mang
   sortTable[0x5EAC]=15503;   // mang
   sortTable[0x607E]=15504;   // mang
   sortTable[0x6757]=15505;   // mang
   sortTable[0x6767]=15506;   // mang
   sortTable[0x6C52]=15507;   // mang
   sortTable[0x6D5D]=15508;   // mang
   sortTable[0x7264]=15509;   // mang
   sortTable[0x727B]=15510;   // mang
   sortTable[0x72F5]=15511;   // mang
   sortTable[0x75DD]=15512;   // mang
   sortTable[0x92E9]=15513;   // mang
   sortTable[0x6BDB]=15514;   // mao
   sortTable[0x77DB]=15515;   // mao
   sortTable[0x5E3D]=15516;   // mao
   sortTable[0x5192]=15517;   // mao
   sortTable[0x8C8C]=15518;   // mao
   sortTable[0x732B]=15519;   // mao
   sortTable[0x8C93]=15520;   // mao
   sortTable[0x8D38]=15521;   // mao
   sortTable[0x8CBF]=15522;   // mao
   sortTable[0x8302]=15523;   // mao
   sortTable[0x8305]=15524;   // mao
   sortTable[0x9AE6]=15525;   // mao
   sortTable[0x7266]=15526;   // mao
   sortTable[0x729B]=15527;   // mao
   sortTable[0x536F]=15528;   // mao
   sortTable[0x8004]=15529;   // mao
   sortTable[0x61CB]=15530;   // mao
   sortTable[0x7441]=15531;   // mao
   sortTable[0x6634]=15532;   // mao
   sortTable[0x88A4]=15533;   // mao
   sortTable[0x65C4]=15534;   // mao
   sortTable[0x951A]=15535;   // mao
   sortTable[0x87CA]=15536;   // mao
   sortTable[0x5CC1]=15537;   // mao
   sortTable[0x7780]=15538;   // mao
   sortTable[0x8306]=15539;   // mao
   sortTable[0x94C6]=15540;   // mao
   sortTable[0x770A]=15541;   // mao
   sortTable[0x911A]=15542;   // mao
   sortTable[0x6CD6]=15543;   // mao
   sortTable[0x8765]=15544;   // mao
   sortTable[0x912E]=15545;   // mao
   sortTable[0x9AF3]=15546;   // mao
   sortTable[0x82BC]=15547;   // mao
   sortTable[0x84E9]=15548;   // mao
   sortTable[0x9D9C]=15549;   // mao
   sortTable[0x9155]=15550;   // mao
   sortTable[0x8750]=15551;   // mao
   sortTable[0x8EDE]=15552;   // mao
   sortTable[0x9328]=15553;   // mao
   sortTable[0x7F5E]=15554;   // mao
   sortTable[0x4E6E]=15555;   // mao
   sortTable[0x515E]=15556;   // mao
   sortTable[0x5183]=15557;   // mao
   sortTable[0x5187]=15558;   // mao
   sortTable[0x5190]=15559;   // mao
   sortTable[0x5825]=15560;   // mao
   sortTable[0x5918]=15561;   // mao
   sortTable[0x5AA2]=15562;   // mao
   sortTable[0x5AF9]=15563;   // mao
   sortTable[0x6117]=15564;   // mao
   sortTable[0x623C]=15565;   // mao
   sortTable[0x6693]=15566;   // mao
   sortTable[0x6786]=15567;   // mao
   sortTable[0x67D5]=15568;   // mao
   sortTable[0x6959]=15569;   // mao
   sortTable[0x6BF7]=15570;   // mao
   sortTable[0x6C02]=15571;   // mao
   sortTable[0x6E35]=15572;   // mao
   sortTable[0x7683]=15573;   // mao
   sortTable[0x7B37]=15574;   // mao
   sortTable[0x843A]=15575;   // mao
   sortTable[0x8992]=15576;   // mao
   sortTable[0x4E48]=15577;   // me
   sortTable[0x9EBC]=15578;   // me
   sortTable[0x5692]=15579;   // me
   sortTable[0x569C]=15580;   // me
   sortTable[0x6FF9]=15581;   // me
   sortTable[0x7666]=15582;   // me
   sortTable[0x6CA1]=15583;   // mei
   sortTable[0x6C92]=15584;   // mei
   sortTable[0x6BCF]=15585;   // mei
   sortTable[0x7F8E]=15586;   // mei
   sortTable[0x7164]=15587;   // mei
   sortTable[0x59B9]=15588;   // mei
   sortTable[0x7709]=15589;   // mei
   sortTable[0x6885]=15590;   // mei
   sortTable[0x9709]=15591;   // mei
   sortTable[0x6627]=15592;   // mei
   sortTable[0x679A]=15593;   // mei
   sortTable[0x9541]=15594;   // mei
   sortTable[0x9382]=15595;   // mei
   sortTable[0x9176]=15596;   // mei
   sortTable[0x9EF4]=15597;   // mei
   sortTable[0x5A92]=15598;   // mei
   sortTable[0x5A9A]=15599;   // mei
   sortTable[0x73AB]=15600;   // mei
   sortTable[0x9B45]=15601;   // mei
   sortTable[0x5BD0]=15602;   // mei
   sortTable[0x8882]=15603;   // mei
   sortTable[0x5D4B]=15604;   // mei
   sortTable[0x6D7C]=15605;   // mei
   sortTable[0x6963]=15606;   // mei
   sortTable[0x6E44]=15607;   // mei
   sortTable[0x8393]=15608;   // mei
   sortTable[0x9545]=15609;   // mei
   sortTable[0x9E5B]=15610;   // mei
   sortTable[0x90FF]=15611;   // mei
   sortTable[0x7996]=15612;   // mei
   sortTable[0x7BC3]=15613;   // mei
   sortTable[0x771B]=15614;   // mei
   sortTable[0x97CE]=15615;   // mei
   sortTable[0x7A48]=15616;   // mei
   sortTable[0x8DCA]=15617;   // mei
   sortTable[0x7338]=15618;   // mei
   sortTable[0x862A]=15619;   // mei
   sortTable[0x92C2]=15620;   // mei
   sortTable[0x77C0]=15621;   // mei
   sortTable[0x8122]=15622;   // mei
   sortTable[0x8104]=15623;   // mei
   sortTable[0x9B3D]=15624;   // mei
   sortTable[0x7959]=15625;   // mei
   sortTable[0x9387]=15626;   // mei
   sortTable[0x51C2]=15627;   // mei
   sortTable[0x5445]=15628;   // mei
   sortTable[0x5746]=15629;   // mei
   sortTable[0x5833]=15630;   // mei
   sortTable[0x587A]=15631;   // mei
   sortTable[0x5A12]=15632;   // mei
   sortTable[0x5A84]=15633;   // mei
   sortTable[0x5ABA]=15634;   // mei
   sortTable[0x5B0D]=15635;   // mei
   sortTable[0x5D44]=15636;   // mei
   sortTable[0x5FBE]=15637;   // mei
   sortTable[0x62BA]=15638;   // mei
   sortTable[0x6334]=15639;   // mei
   sortTable[0x6517]=15640;   // mei
   sortTable[0x65C0]=15641;   // mei
   sortTable[0x6802]=15642;   // mei
   sortTable[0x6973]=15643;   // mei
   sortTable[0x69D1]=15644;   // mei
   sortTable[0x6BCE]=15645;   // mei
   sortTable[0x6CAC]=15646;   // mei
   sortTable[0x6E3C]=15647;   // mei
   sortTable[0x6E48]=15648;   // mei
   sortTable[0x715D]=15649;   // mei
   sortTable[0x71D8]=15650;   // mei
   sortTable[0x73FB]=15651;   // mei
   sortTable[0x7442]=15652;   // mei
   sortTable[0x75D7]=15653;   // mei
   sortTable[0x7742]=15654;   // mei
   sortTable[0x7778]=15655;   // mei
   sortTable[0x815C]=15656;   // mei
   sortTable[0x82FA]=15657;   // mei
   sortTable[0x847F]=15658;   // mei
   sortTable[0x875E]=15659;   // mei
   sortTable[0x8EBE]=15660;   // mei
   sortTable[0x9DA5]=15661;   // mei
   sortTable[0x9EE3]=15662;   // mei
   sortTable[0x4EEC]=15663;   // men
   sortTable[0x5011]=15664;   // men
   sortTable[0x95E8]=15665;   // men
   sortTable[0x9580]=15666;   // men
   sortTable[0x95F7]=15667;   // men
   sortTable[0x60B6]=15668;   // men
   sortTable[0x9494]=15669;   // men
   sortTable[0x626A]=15670;   // men
   sortTable[0x7116]=15671;   // men
   sortTable[0x61D1]=15672;   // men
   sortTable[0x83DB]=15673;   // men
   sortTable[0x864B]=15674;   // men
   sortTable[0x4EB9]=15675;   // men
   sortTable[0x61E3]=15676;   // men
   sortTable[0x636B]=15677;   // men
   sortTable[0x66AA]=15678;   // men
   sortTable[0x691A]=15679;   // men
   sortTable[0x71DC]=15680;   // men
   sortTable[0x73A7]=15681;   // men
   sortTable[0x748A]=15682;   // men
   sortTable[0x9346]=15683;   // men
   sortTable[0x9585]=15684;   // men
   sortTable[0x731B]=15685;   // meng
   sortTable[0x68A6]=15686;   // meng
   sortTable[0x5922]=15687;   // meng
   sortTable[0x76DF]=15688;   // meng
   sortTable[0x8499]=15689;   // meng
   sortTable[0x6726]=15690;   // meng
   sortTable[0x5B5F]=15691;   // meng
   sortTable[0x840C]=15692;   // meng
   sortTable[0x8813]=15693;   // meng
   sortTable[0x6AAC]=15694;   // meng
   sortTable[0x61F5]=15695;   // meng
   sortTable[0x867B]=15696;   // meng
   sortTable[0x9530]=15697;   // meng
   sortTable[0x8268]=15698;   // meng
   sortTable[0x824B]=15699;   // meng
   sortTable[0x52D0]=15700;   // meng
   sortTable[0x8722]=15701;   // meng
   sortTable[0x77C7]=15702;   // meng
   sortTable[0x750D]=15703;   // meng
   sortTable[0x77A2]=15704;   // meng
   sortTable[0x9133]=15705;   // meng
   sortTable[0x791E]=15706;   // meng
   sortTable[0x973F]=15707;   // meng
   sortTable[0x77D2]=15708;   // meng
   sortTable[0x9138]=15709;   // meng
   sortTable[0x995B]=15710;   // meng
   sortTable[0x986D]=15711;   // meng
   sortTable[0x8420]=15712;   // meng
   sortTable[0x9BCD]=15713;   // meng
   sortTable[0x9F06]=15714;   // meng
   sortTable[0x511A]=15715;   // meng
   sortTable[0x51A1]=15716;   // meng
   sortTable[0x5923]=15717;   // meng
   sortTable[0x5E6A]=15718;   // meng
   sortTable[0x61DC]=15719;   // meng
   sortTable[0x61DE]=15720;   // meng
   sortTable[0x63B9]=15721;   // meng
   sortTable[0x64DD]=15722;   // meng
   sortTable[0x66DA]=15723;   // meng
   sortTable[0x6A57]=15724;   // meng
   sortTable[0x6C0B]=15725;   // meng
   sortTable[0x6E95]=15726;   // meng
   sortTable[0x6FDB]=15727;   // meng
   sortTable[0x7374]=15728;   // meng
   sortTable[0x74FE]=15729;   // meng
   sortTable[0x753F]=15730;   // meng
   sortTable[0x8394]=15731;   // meng
   sortTable[0x8544]=15732;   // meng
   sortTable[0x8609]=15733;   // meng
   sortTable[0x8771]=15734;   // meng
   sortTable[0x9333]=15735;   // meng
   sortTable[0x9725]=15736;   // meng
   sortTable[0x9740]=15737;   // meng
   sortTable[0x9BED]=15738;   // meng
   sortTable[0x9E0F]=15739;   // meng
   sortTable[0x9E72]=15740;   // meng
   sortTable[0x7C73]=15741;   // mi
   sortTable[0x5BC6]=15742;   // mi
   sortTable[0x79D8]=15743;   // mi
   sortTable[0x8FF7]=15744;   // mi
   sortTable[0x871C]=15745;   // mi
   sortTable[0x772F]=15746;   // mi
   sortTable[0x7787]=15747;   // mi
   sortTable[0x5F25]=15748;   // mi
   sortTable[0x5F4C]=15749;   // mi
   sortTable[0x6CCC]=15750;   // mi
   sortTable[0x9761]=15751;   // mi
   sortTable[0x89C5]=15752;   // mi
   sortTable[0x8C1C]=15753;   // mi
   sortTable[0x7CDC]=15754;   // mi
   sortTable[0x8C27]=15755;   // mi
   sortTable[0x54AA]=15756;   // mi
   sortTable[0x5B93]=15757;   // mi
   sortTable[0x5F2D]=15758;   // mi
   sortTable[0x6549]=15759;   // mi
   sortTable[0x7E3B]=15760;   // mi
   sortTable[0x9E8B]=15761;   // mi
   sortTable[0x5627]=15762;   // mi
   sortTable[0x919A]=15763;   // mi
   sortTable[0x7962]=15764;   // mi
   sortTable[0x863C]=15765;   // mi
   sortTable[0x6C68]=15766;   // mi
   sortTable[0x8288]=15767;   // mi
   sortTable[0x7315]=15768;   // mi
   sortTable[0x5E42]=15769;   // mi
   sortTable[0x7955]=15770;   // mi
   sortTable[0x91BF]=15771;   // mi
   sortTable[0x899B]=15772;   // mi
   sortTable[0x9E9B]=15773;   // mi
   sortTable[0x8112]=15774;   // mi
   sortTable[0x9F0F]=15775;   // mi
   sortTable[0x9E8A]=15776;   // mi
   sortTable[0x91BE]=15777;   // mi
   sortTable[0x7F8B]=15778;   // mi
   sortTable[0x845E]=15779;   // mi
   sortTable[0x772B]=15780;   // mi
   sortTable[0x79B0]=15781;   // mi
   sortTable[0x7F83]=15782;   // mi
   sortTable[0x84BE]=15783;   // mi
   sortTable[0x8993]=15784;   // mi
   sortTable[0x8994]=15785;   // mi
   sortTable[0x8A78]=15786;   // mi
   sortTable[0x91C4]=15787;   // mi
   sortTable[0x9E0D]=15788;   // mi
   sortTable[0x7F59]=15789;   // mi
   sortTable[0x8524]=15790;   // mi
   sortTable[0x92A4]=15791;   // mi
   sortTable[0x4F8E]=15792;   // mi
   sortTable[0x5196]=15793;   // mi
   sortTable[0x519E]=15794;   // mi
   sortTable[0x51AA]=15795;   // mi
   sortTable[0x5853]=15796;   // mi
   sortTable[0x5B4A]=15797;   // mi
   sortTable[0x5BBB]=15798;   // mi
   sortTable[0x5CDA]=15799;   // mi
   sortTable[0x5E4E]=15800;   // mi
   sortTable[0x5E66]=15801;   // mi
   sortTable[0x6202]=15802;   // mi
   sortTable[0x64DF]=15803;   // mi
   sortTable[0x6520]=15804;   // mi
   sortTable[0x6993]=15805;   // mi
   sortTable[0x6A12]=15806;   // mi
   sortTable[0x6AC1]=15807;   // mi
   sortTable[0x6C95]=15808;   // mi
   sortTable[0x6CB5]=15809;   // mi
   sortTable[0x6D23]=15810;   // mi
   sortTable[0x6DE7]=15811;   // mi
   sortTable[0x6DFF]=15812;   // mi
   sortTable[0x6E33]=15813;   // mi
   sortTable[0x6EF5]=15814;   // mi
   sortTable[0x6F1E]=15815;   // mi
   sortTable[0x6FD4]=15816;   // mi
   sortTable[0x6FD7]=15817;   // mi
   sortTable[0x7030]=15818;   // mi
   sortTable[0x7056]=15819;   // mi
   sortTable[0x7190]=15820;   // mi
   sortTable[0x7222]=15821;   // mi
   sortTable[0x737C]=15822;   // mi
   sortTable[0x74D5]=15823;   // mi
   sortTable[0x7C1A]=15824;   // mi
   sortTable[0x7CF8]=15825;   // mi
   sortTable[0x851D]=15826;   // mi
   sortTable[0x85CC]=15827;   // mi
   sortTable[0x8B0E]=15828;   // mi
   sortTable[0x8B10]=15829;   // mi
   sortTable[0x957E]=15830;   // mi
   sortTable[0x9762]=15831;   // mian
   sortTable[0x68C9]=15832;   // mian
   sortTable[0x514D]=15833;   // mian
   sortTable[0x7720]=15834;   // mian
   sortTable[0x52C9]=15835;   // mian
   sortTable[0x7EF5]=15836;   // mian
   sortTable[0x7DBF]=15837;   // mian
   sortTable[0x7F05]=15838;   // mian
   sortTable[0x5195]=15839;   // mian
   sortTable[0x6C94]=15840;   // mian
   sortTable[0x6E4E]=15841;   // mian
   sortTable[0x5A29]=15842;   // mian
   sortTable[0x9EFE]=15843;   // mian
   sortTable[0x6E11]=15844;   // mian
   sortTable[0x817C]=15845;   // mian
   sortTable[0x7704]=15846;   // mian
   sortTable[0x81F1]=15847;   // mian
   sortTable[0x7D7B]=15848;   // mian
   sortTable[0x9BB8]=15849;   // mian
   sortTable[0x77CF]=15850;   // mian
   sortTable[0x9763]=15851;   // mian
   sortTable[0x77CA]=15852;   // mian
   sortTable[0x7DDC]=15853;   // mian
   sortTable[0x8752]=15854;   // mian
   sortTable[0x9EAA]=15855;   // mian
   sortTable[0x9EB5]=15856;   // mian
   sortTable[0x4E0F]=15857;   // mian
   sortTable[0x506D]=15858;   // mian
   sortTable[0x52D4]=15859;   // mian
   sortTable[0x5595]=15860;   // mian
   sortTable[0x5A42]=15861;   // mian
   sortTable[0x5A94]=15862;   // mian
   sortTable[0x5B35]=15863;   // mian
   sortTable[0x5B80]=15864;   // mian
   sortTable[0x6110]=15865;   // mian
   sortTable[0x6AB0]=15866;   // mian
   sortTable[0x6ACB]=15867;   // mian
   sortTable[0x6C45]=15868;   // mian
   sortTable[0x6FA0]=15869;   // mian
   sortTable[0x77C8]=15870;   // mian
   sortTable[0x7CC6]=15871;   // mian
   sortTable[0x7DEC]=15872;   // mian
   sortTable[0x8287]=15873;   // mian
   sortTable[0x8442]=15874;   // mian
   sortTable[0x9EAB]=15875;   // mian
   sortTable[0x9EBA]=15876;   // mian
   sortTable[0x9EFD]=15877;   // mian
   sortTable[0x79D2]=15878;   // miao
   sortTable[0x5999]=15879;   // miao
   sortTable[0x82D7]=15880;   // miao
   sortTable[0x63CF]=15881;   // miao
   sortTable[0x5E99]=15882;   // miao
   sortTable[0x5EDF]=15883;   // miao
   sortTable[0x6E3A]=15884;   // miao
   sortTable[0x7784]=15885;   // miao
   sortTable[0x7F2A]=15886;   // miao
   sortTable[0x9088]=15887;   // miao
   sortTable[0x85D0]=15888;   // miao
   sortTable[0x7F08]=15889;   // miao
   sortTable[0x7707]=15890;   // miao
   sortTable[0x6DFC]=15891;   // miao
   sortTable[0x676A]=15892;   // miao
   sortTable[0x55B5]=15893;   // miao
   sortTable[0x9E4B]=15894;   // miao
   sortTable[0x7DF2]=15895;   // miao
   sortTable[0x7BCE]=15896;   // miao
   sortTable[0x7DE2]=15897;   // miao
   sortTable[0x9D93]=15898;   // miao
   sortTable[0x5A8C]=15899;   // miao
   sortTable[0x5EBF]=15900;   // miao
   sortTable[0x7385]=15901;   // miao
   sortTable[0x7AD7]=15902;   // miao
   sortTable[0x9C59]=15903;   // miao
   sortTable[0x706D]=15904;   // mie
   sortTable[0x6EC5]=15905;   // mie
   sortTable[0x8511]=15906;   // mie
   sortTable[0x881B]=15907;   // mie
   sortTable[0x4E5C]=15908;   // mie
   sortTable[0x7BFE]=15909;   // mie
   sortTable[0x54A9]=15910;   // mie
   sortTable[0x884A]=15911;   // mie
   sortTable[0x8995]=15912;   // mie
   sortTable[0x9456]=15913;   // mie
   sortTable[0x9D13]=15914;   // mie
   sortTable[0x9C74]=15915;   // mie
   sortTable[0x5400]=15916;   // mie
   sortTable[0x54F6]=15917;   // mie
   sortTable[0x5B6D]=15918;   // mie
   sortTable[0x5E6D]=15919;   // mie
   sortTable[0x61F1]=15920;   // mie
   sortTable[0x6423]=15921;   // mie
   sortTable[0x6AD7]=15922;   // mie
   sortTable[0x70D5]=15923;   // mie
   sortTable[0x858E]=15924;   // mie
   sortTable[0x6C11]=15925;   // min
   sortTable[0x654F]=15926;   // min
   sortTable[0x62BF]=15927;   // min
   sortTable[0x95F5]=15928;   // min
   sortTable[0x95FD]=15929;   // min
   sortTable[0x60AF]=15930;   // min
   sortTable[0x7F17]=15931;   // min
   sortTable[0x610D]=15932;   // min
   sortTable[0x76BF]=15933;   // min
   sortTable[0x5CB7]=15934;   // min
   sortTable[0x6CEF]=15935;   // min
   sortTable[0x73C9]=15936;   // min
   sortTable[0x82E0]=15937;   // min
   sortTable[0x7DE1]=15938;   // min
   sortTable[0x7DCD]=15939;   // min
   sortTable[0x9CD8]=15940;   // min
   sortTable[0x7C22]=15941;   // min
   sortTable[0x9309]=15942;   // min
   sortTable[0x9231]=15943;   // min
   sortTable[0x76FF]=15944;   // min
   sortTable[0x7888]=15945;   // min
   sortTable[0x9C35]=15946;   // min
   sortTable[0x7B22]=15947;   // min
   sortTable[0x7F60]=15948;   // min
   sortTable[0x9594]=15949;   // min
   sortTable[0x50F6]=15950;   // min
   sortTable[0x51BA]=15951;   // min
   sortTable[0x5221]=15952;   // min
   sortTable[0x52C4]=15953;   // min
   sortTable[0x578A]=15954;   // min
   sortTable[0x59C4]=15955;   // min
   sortTable[0x5D0F]=15956;   // min
   sortTable[0x5FDE]=15957;   // min
   sortTable[0x600B]=15958;   // min
   sortTable[0x60FD]=15959;   // min
   sortTable[0x615C]=15960;   // min
   sortTable[0x61AB]=15961;   // min
   sortTable[0x636A]=15962;   // min
   sortTable[0x6543]=15963;   // min
   sortTable[0x656F]=15964;   // min
   sortTable[0x65FB]=15965;   // min
   sortTable[0x65FC]=15966;   // min
   sortTable[0x668B]=15967;   // min
   sortTable[0x6E63]=15968;   // min
   sortTable[0x6F63]=15969;   // min
   sortTable[0x7418]=15970;   // min
   sortTable[0x7449]=15971;   // min
   sortTable[0x75FB]=15972;   // min
   sortTable[0x7807]=15973;   // min
   sortTable[0x7B3D]=15974;   // min
   sortTable[0x8820]=15975;   // min
   sortTable[0x9372]=15976;   // min
   sortTable[0x95A9]=15977;   // min
   sortTable[0x9D16]=15978;   // min
   sortTable[0x547D]=15979;   // ming
   sortTable[0x660E]=15980;   // ming
   sortTable[0x540D]=15981;   // ming
   sortTable[0x9E23]=15982;   // ming
   sortTable[0x9CF4]=15983;   // ming
   sortTable[0x94ED]=15984;   // ming
   sortTable[0x51A5]=15985;   // ming
   sortTable[0x8317]=15986;   // ming
   sortTable[0x7791]=15987;   // ming
   sortTable[0x669D]=15988;   // ming
   sortTable[0x6E9F]=15989;   // ming
   sortTable[0x879F]=15990;   // ming
   sortTable[0x9169]=15991;   // ming
   sortTable[0x84C2]=15992;   // ming
   sortTable[0x7733]=15993;   // ming
   sortTable[0x8A7A]=15994;   // ming
   sortTable[0x910D]=15995;   // ming
   sortTable[0x9298]=15996;   // ming
   sortTable[0x7700]=15997;   // ming
   sortTable[0x89AD]=15998;   // ming
   sortTable[0x4F72]=15999;   // ming
   sortTable[0x51D5]=16000;   // ming
   sortTable[0x59F3]=16001;   // ming
   sortTable[0x5AC7]=16002;   // ming
   sortTable[0x614F]=16003;   // ming
   sortTable[0x63B5]=16004;   // ming
   sortTable[0x6719]=16005;   // ming
   sortTable[0x6927]=16006;   // ming
   sortTable[0x69A0]=16007;   // ming
   sortTable[0x6D3A]=16008;   // ming
   sortTable[0x733D]=16009;   // ming
   sortTable[0x8C2C]=16010;   // miu
   sortTable[0x8B2C]=16011;   // miu
   sortTable[0x6A21]=16012;   // mo
   sortTable[0x6478]=16013;   // mo
   sortTable[0x9ED8]=16014;   // mo
   sortTable[0x6F20]=16015;   // mo
   sortTable[0x78E8]=16016;   // mo
   sortTable[0x819C]=16017;   // mo
   sortTable[0x9B54]=16018;   // mo
   sortTable[0x672B]=16019;   // mo
   sortTable[0x58A8]=16020;   // mo
   sortTable[0x62B9]=16021;   // mo
   sortTable[0x83AB]=16022;   // mo
   sortTable[0x5BDE]=16023;   // mo
   sortTable[0x964C]=16024;   // mo
   sortTable[0x6CAB]=16025;   // mo
   sortTable[0x6469]=16026;   // mo
   sortTable[0x8611]=16027;   // mo
   sortTable[0x8C1F]=16028;   // mo
   sortTable[0x6479]=16029;   // mo
   sortTable[0x9EBD]=16030;   // mo
   sortTable[0x84E6]=16031;   // mo
   sortTable[0x6B81]=16032;   // mo
   sortTable[0x8309]=16033;   // mo
   sortTable[0x998D]=16034;   // mo
   sortTable[0x9546]=16035;   // mo
   sortTable[0x79E3]=16036;   // mo
   sortTable[0x5AEB]=16037;   // mo
   sortTable[0x977A]=16038;   // mo
   sortTable[0x8C8A]=16039;   // mo
   sortTable[0x763C]=16040;   // mo
   sortTable[0x9ED9]=16041;   // mo
   sortTable[0x8B29]=16042;   // mo
   sortTable[0x8C98]=16043;   // mo
   sortTable[0x7E86]=16044;   // mo
   sortTable[0x9EBF]=16045;   // mo
   sortTable[0x8031]=16046;   // mo
   sortTable[0x773D]=16047;   // mo
   sortTable[0x85E6]=16048;   // mo
   sortTable[0x8B28]=16049;   // mo
   sortTable[0x9B79]=16050;   // mo
   sortTable[0x9A40]=16051;   // mo
   sortTable[0x7790]=16052;   // mo
   sortTable[0x7CE2]=16053;   // mo
   sortTable[0x995D]=16054;   // mo
   sortTable[0x8C83]=16055;   // mo
   sortTable[0x771C]=16056;   // mo
   sortTable[0x781E]=16057;   // mo
   sortTable[0x9B69]=16058;   // mo
   sortTable[0x773F]=16059;   // mo
   sortTable[0x86E8]=16060;   // mo
   sortTable[0x9286]=16061;   // mo
   sortTable[0x93CC]=16062;   // mo
   sortTable[0x7C96]=16063;   // mo
   sortTable[0x87D4]=16064;   // mo
   sortTable[0x5298]=16065;   // mo
   sortTable[0x52B0]=16066;   // mo
   sortTable[0x551C]=16067;   // mo
   sortTable[0x55FC]=16068;   // mo
   sortTable[0x56A4]=16069;   // mo
   sortTable[0x56A9]=16070;   // mo
   sortTable[0x56B0]=16071;   // mo
   sortTable[0x573D]=16072;   // mo
   sortTable[0x587B]=16073;   // mo
   sortTable[0x59BA]=16074;   // mo
   sortTable[0x5AFC]=16075;   // mo
   sortTable[0x5C1B]=16076;   // mo
   sortTable[0x5E13]=16077;   // mo
   sortTable[0x5E1E]=16078;   // mo
   sortTable[0x5E85]=16079;   // mo
   sortTable[0x603D]=16080;   // mo
   sortTable[0x61E1]=16081;   // mo
   sortTable[0x64F5]=16082;   // mo
   sortTable[0x6629]=16083;   // mo
   sortTable[0x66AF]=16084;   // mo
   sortTable[0x67BA]=16085;   // mo
   sortTable[0x6A45]=16086;   // mo
   sortTable[0x6B7E]=16087;   // mo
   sortTable[0x6B7F]=16088;   // mo
   sortTable[0x6E50]=16089;   // mo
   sortTable[0x700E]=16090;   // mo
   sortTable[0x7205]=16091;   // mo
   sortTable[0x734F]=16092;   // mo
   sortTable[0x768C]=16093;   // mo
   sortTable[0x7799]=16094;   // mo
   sortTable[0x7933]=16095;   // mo
   sortTable[0x7D48]=16096;   // mo
   sortTable[0x8388]=16097;   // mo
   sortTable[0x9943]=16098;   // mo
   sortTable[0x9ACD]=16099;   // mo
   sortTable[0x67D0]=16100;   // mou
   sortTable[0x8C0B]=16101;   // mou
   sortTable[0x8B00]=16102;   // mou
   sortTable[0x725F]=16103;   // mou
   sortTable[0x7738]=16104;   // mou
   sortTable[0x4F94]=16105;   // mou
   sortTable[0x86D1]=16106;   // mou
   sortTable[0x936A]=16107;   // mou
   sortTable[0x54DE]=16108;   // mou
   sortTable[0x9EB0]=16109;   // mou
   sortTable[0x927E]=16110;   // mou
   sortTable[0x9D3E]=16111;   // mou
   sortTable[0x7E46]=16112;   // mou
   sortTable[0x52BA]=16113;   // mou
   sortTable[0x6048]=16114;   // mou
   sortTable[0x6D20]=16115;   // mou
   sortTable[0x77B4]=16116;   // mou
   sortTable[0x8E0E]=16117;   // mou
   sortTable[0x76EE]=16118;   // mu
   sortTable[0x6BCD]=16119;   // mu
   sortTable[0x6728]=16120;   // mu
   sortTable[0x5E55]=16121;   // mu
   sortTable[0x4EA9]=16122;   // mu
   sortTable[0x755D]=16123;   // mu
   sortTable[0x7267]=16124;   // mu
   sortTable[0x6155]=16125;   // mu
   sortTable[0x5893]=16126;   // mu
   sortTable[0x59C6]=16127;   // mu
   sortTable[0x7261]=16128;   // mu
   sortTable[0x62C7]=16129;   // mu
   sortTable[0x7A46]=16130;   // mu
   sortTable[0x66AE]=16131;   // mu
   sortTable[0x52DF]=16132;   // mu
   sortTable[0x7766]=16133;   // mu
   sortTable[0x6C90]=16134;   // mu
   sortTable[0x82DC]=16135;   // mu
   sortTable[0x94BC]=16136;   // mu
   sortTable[0x4EEB]=16137;   // mu
   sortTable[0x5776]=16138;   // mu
   sortTable[0x96EE]=16139;   // mu
   sortTable[0x9702]=16140;   // mu
   sortTable[0x8252]=16141;   // mu
   sortTable[0x7E38]=16142;   // mu
   sortTable[0x9267]=16143;   // mu
   sortTable[0x782A]=16144;   // mu
   sortTable[0x869E]=16145;   // mu
   sortTable[0x97AA]=16146;   // mu
   sortTable[0x51E9]=16147;   // mu
   sortTable[0x58B2]=16148;   // mu
   sortTable[0x5CD4]=16149;   // mu
   sortTable[0x5E59]=16150;   // mu
   sortTable[0x6154]=16151;   // mu
   sortTable[0x6730]=16152;   // mu
   sortTable[0x6958]=16153;   // mu
   sortTable[0x6BE3]=16154;   // mu
   sortTable[0x6BEA]=16155;   // mu
   sortTable[0x6C01]=16156;   // mu
   sortTable[0x7091]=16157;   // mu
   sortTable[0x7273]=16158;   // mu
   sortTable[0x72C7]=16159;   // mu
   sortTable[0x7546]=16160;   // mu
   sortTable[0x7552]=16161;   // mu
   sortTable[0x755E]=16162;   // mu
   sortTable[0x756E]=16163;   // mu
   sortTable[0x80DF]=16164;   // mu
   sortTable[0x83AF]=16165;   // mu
   sortTable[0x8E07]=16166;   // mu
   sortTable[0x926C]=16167;   // mu
   sortTable[0x90A3]=16640;   // na
   sortTable[0x54EA]=16641;   // na
   sortTable[0x62FF]=16642;   // na
   sortTable[0x7EB3]=16643;   // na
   sortTable[0x7D0D]=16644;   // na
   sortTable[0x5A1C]=16645;   // na
   sortTable[0x94A0]=16646;   // na
   sortTable[0x637A]=16647;   // na
   sortTable[0x8872]=16648;   // na
   sortTable[0x80AD]=16649;   // na
   sortTable[0x954E]=16650;   // na
   sortTable[0x8C7D]=16651;   // na
   sortTable[0x9779]=16652;   // na
   sortTable[0x9B76]=16653;   // na
   sortTable[0x88A6]=16654;   // na
   sortTable[0x93BF]=16655;   // na
   sortTable[0x8C80]=16656;   // na
   sortTable[0x8EDC]=16657;   // na
   sortTable[0x9209]=16658;   // na
   sortTable[0x84B3]=16659;   // na
   sortTable[0x4E78]=16660;   // na
   sortTable[0x55F1]=16661;   // na
   sortTable[0x59A0]=16662;   // na
   sortTable[0x62CF]=16663;   // na
   sortTable[0x6310]=16664;   // na
   sortTable[0x7B1D]=16665;   // na
   sortTable[0x96EB]=16666;   // na
   sortTable[0x5976]=16667;   // nai
   sortTable[0x8010]=16668;   // nai
   sortTable[0x4E43]=16669;   // nai
   sortTable[0x5948]=16670;   // nai
   sortTable[0x6C16]=16671;   // nai
   sortTable[0x9F10]=16672;   // nai
   sortTable[0x827F]=16673;   // nai
   sortTable[0x8418]=16674;   // nai
   sortTable[0x67F0]=16675;   // nai
   sortTable[0x800F]=16676;   // nai
   sortTable[0x91E2]=16677;   // nai
   sortTable[0x8FFA]=16678;   // nai
   sortTable[0x879A]=16679;   // nai
   sortTable[0x8926]=16680;   // nai
   sortTable[0x5037]=16681;   // nai
   sortTable[0x59B3]=16682;   // nai
   sortTable[0x5B2D]=16683;   // nai
   sortTable[0x5B7B]=16684;   // nai
   sortTable[0x5EFC]=16685;   // nai
   sortTable[0x6468]=16686;   // nai
   sortTable[0x6E3F]=16687;   // nai
   sortTable[0x718B]=16688;   // nai
   sortTable[0x7593]=16689;   // nai
   sortTable[0x8149]=16690;   // nai
   sortTable[0x933C]=16691;   // nai
   sortTable[0x96BE]=16692;   // nan
   sortTable[0x96E3]=16693;   // nan
   sortTable[0x5357]=16694;   // nan
   sortTable[0x7537]=16695;   // nan
   sortTable[0x5583]=16696;   // nan
   sortTable[0x6960]=16697;   // nan
   sortTable[0x8D67]=16698;   // nan
   sortTable[0x56E1]=16699;   // nan
   sortTable[0x877B]=16700;   // nan
   sortTable[0x8169]=16701;   // nan
   sortTable[0x8AF5]=16702;   // nan
   sortTable[0x4FBD]=16703;   // nan
   sortTable[0x5A1A]=16704;   // nan
   sortTable[0x5A7B]=16705;   // nan
   sortTable[0x6201]=16706;   // nan
   sortTable[0x63C7]=16707;   // nan
   sortTable[0x6694]=16708;   // nan
   sortTable[0x678F]=16709;   // nan
   sortTable[0x67AC]=16710;   // nan
   sortTable[0x67DF]=16711;   // nan
   sortTable[0x6E73]=16712;   // nan
   sortTable[0x7558]=16713;   // nan
   sortTable[0x83AE]=16714;   // nan
   sortTable[0x8433]=16715;   // nan
   sortTable[0x9056]=16716;   // nan
   sortTable[0x56CA]=16717;   // nang
   sortTable[0x9995]=16718;   // nang
   sortTable[0x66E9]=16719;   // nang
   sortTable[0x56D4]=16720;   // nang
   sortTable[0x652E]=16721;   // nang
   sortTable[0x9962]=16722;   // nang
   sortTable[0x8830]=16723;   // nang
   sortTable[0x4E6A]=16724;   // nang
   sortTable[0x513E]=16725;   // nang
   sortTable[0x56A2]=16726;   // nang
   sortTable[0x64C3]=16727;   // nang
   sortTable[0x6B1C]=16728;   // nang
   sortTable[0x7062]=16729;   // nang
   sortTable[0x8B68]=16730;   // nang
   sortTable[0x9B1E]=16731;   // nang
   sortTable[0x9F49]=16732;   // nang
   sortTable[0x8111]=16733;   // nao
   sortTable[0x8166]=16734;   // nao
   sortTable[0x95F9]=16735;   // nao
   sortTable[0x9B27]=16736;   // nao
   sortTable[0x607C]=16737;   // nao
   sortTable[0x60F1]=16738;   // nao
   sortTable[0x6320]=16739;   // nao
   sortTable[0x6DD6]=16740;   // nao
   sortTable[0x7459]=16741;   // nao
   sortTable[0x5476]=16742;   // nao
   sortTable[0x86F2]=16743;   // nao
   sortTable[0x94D9]=16744;   // nao
   sortTable[0x7847]=16745;   // nao
   sortTable[0x7331]=16746;   // nao
   sortTable[0x5B6C]=16747;   // nao
   sortTable[0x57B4]=16748;   // nao
   sortTable[0x81D1]=16749;   // nao
   sortTable[0x8B4A]=16750;   // nao
   sortTable[0x8133]=16751;   // nao
   sortTable[0x87EF]=16752;   // nao
   sortTable[0x8A49]=16753;   // nao
   sortTable[0x78AF]=16754;   // nao
   sortTable[0x5318]=16755;   // nao
   sortTable[0x5816]=16756;   // nao
   sortTable[0x5912]=16757;   // nao
   sortTable[0x5A65]=16758;   // nao
   sortTable[0x5AD0]=16759;   // nao
   sortTable[0x5CF1]=16760;   // nao
   sortTable[0x5DA9]=16761;   // nao
   sortTable[0x5DCE]=16762;   // nao
   sortTable[0x6013]=16763;   // nao
   sortTable[0x60A9]=16764;   // nao
   sortTable[0x61B9]=16765;   // nao
   sortTable[0x6493]=16766;   // nao
   sortTable[0x7376]=16767;   // nao
   sortTable[0x737F]=16768;   // nao
   sortTable[0x7899]=16769;   // nao
   sortTable[0x9403]=16770;   // nao
   sortTable[0x9599]=16771;   // nao
   sortTable[0x5462]=16772;   // ne
   sortTable[0x5450]=16773;   // ne
   sortTable[0x5436]=16774;   // ne
   sortTable[0x8BB7]=16775;   // ne
   sortTable[0x7732]=16776;   // ne
   sortTable[0x6290]=16777;   // ne
   sortTable[0x7592]=16778;   // ne
   sortTable[0x8A25]=16779;   // ne
   sortTable[0x5185]=16780;   // nei
   sortTable[0x5167]=16781;   // nei
   sortTable[0x9981]=16782;   // nei
   sortTable[0x9BBE]=16783;   // nei
   sortTable[0x9BD8]=16784;   // nei
   sortTable[0x812E]=16785;   // nei
   sortTable[0x8147]=16786;   // nei
   sortTable[0x9317]=16787;   // nei
   sortTable[0x9912]=16788;   // nei
   sortTable[0x5A1E]=16789;   // nei
   sortTable[0x6C1D]=16790;   // nei
   sortTable[0x5AE9]=16791;   // nen
   sortTable[0x6041]=16792;   // nen
   sortTable[0x5AF0]=16793;   // nen
   sortTable[0x80FD]=16794;   // neng
   sortTable[0x4F60]=16795;   // ni
   sortTable[0x6CE5]=16796;   // ni
   sortTable[0x62DF]=16797;   // ni
   sortTable[0x64EC]=16798;   // ni
   sortTable[0x5C3C]=16799;   // ni
   sortTable[0x9006]=16800;   // ni
   sortTable[0x59AE]=16801;   // ni
   sortTable[0x502A]=16802;   // ni
   sortTable[0x533F]=16803;   // ni
   sortTable[0x6EBA]=16804;   // ni
   sortTable[0x817B]=16805;   // ni
   sortTable[0x4F32]=16806;   // ni
   sortTable[0x9713]=16807;   // ni
   sortTable[0x6635]=16808;   // ni
   sortTable[0x7768]=16809;   // ni
   sortTable[0x65CE]=16810;   // ni
   sortTable[0x730A]=16811;   // ni
   sortTable[0x6029]=16812;   // ni
   sortTable[0x9CB5]=16813;   // ni
   sortTable[0x873A]=16814;   // ni
   sortTable[0x94CC]=16815;   // ni
   sortTable[0x9E91]=16816;   // ni
   sortTable[0x85BF]=16817;   // ni
   sortTable[0x8F17]=16818;   // ni
   sortTable[0x922E]=16819;   // ni
   sortTable[0x576D]=16820;   // ni
   sortTable[0x81A9]=16821;   // ni
   sortTable[0x81E1]=16822;   // ni
   sortTable[0x90F3]=16823;   // ni
   sortTable[0x8C8E]=16824;   // ni
   sortTable[0x82E8]=16825;   // ni
   sortTable[0x7E0C]=16826;   // ni
   sortTable[0x9BD3]=16827;   // ni
   sortTable[0x807B]=16828;   // ni
   sortTable[0x7724]=16829;   // ni
   sortTable[0x8063]=16830;   // ni
   sortTable[0x8FE1]=16831;   // ni
   sortTable[0x9F6F]=16832;   // ni
   sortTable[0x80D2]=16833;   // ni
   sortTable[0x815D]=16834;   // ni
   sortTable[0x8ABD]=16835;   // ni
   sortTable[0x8DDC]=16836;   // ni
   sortTable[0x9BE2]=16837;   // ni
   sortTable[0x4F31]=16838;   // ni
   sortTable[0x5117]=16839;   // ni
   sortTable[0x511E]=16840;   // ni
   sortTable[0x57FF]=16841;   // ni
   sortTable[0x5804]=16842;   // ni
   sortTable[0x5A57]=16843;   // ni
   sortTable[0x5ADF]=16844;   // ni
   sortTable[0x5B3A]=16845;   // ni
   sortTable[0x5B74]=16846;   // ni
   sortTable[0x5C54]=16847;   // ni
   sortTable[0x5C70]=16848;   // ni
   sortTable[0x60C4]=16849;   // ni
   sortTable[0x6135]=16850;   // ni
   sortTable[0x62B3]=16851;   // ni
   sortTable[0x6672]=16852;   // ni
   sortTable[0x66B1]=16853;   // ni
   sortTable[0x67C5]=16854;   // ni
   sortTable[0x68FF]=16855;   // ni
   sortTable[0x6AB7]=16856;   // ni
   sortTable[0x6C3C]=16857;   // ni
   sortTable[0x6DE3]=16858;   // ni
   sortTable[0x72D4]=16859;   // ni
   sortTable[0x79DC]=16860;   // ni
   sortTable[0x7C7E]=16861;   // ni
   sortTable[0x86AD]=16862;   // ni
   sortTable[0x88AE]=16863;   // ni
   sortTable[0x89EC]=16864;   // ni
   sortTable[0x96AC]=16865;   // ni
   sortTable[0x999C]=16866;   // ni
   sortTable[0x5E74]=16867;   // nian
   sortTable[0x5FF5]=16868;   // nian
   sortTable[0x9ECF]=16869;   // nian
   sortTable[0x637B]=16870;   // nian
   sortTable[0x78BE]=16871;   // nian
   sortTable[0x5EFF]=16872;   // nian
   sortTable[0x64B5]=16873;   // nian
   sortTable[0x6506]=16874;   // nian
   sortTable[0x8F87]=16875;   // nian
   sortTable[0x62C8]=16876;   // nian
   sortTable[0x8F97]=16877;   // nian
   sortTable[0x9C87]=16878;   // nian
   sortTable[0x852B]=16879;   // nian
   sortTable[0x9CB6]=16880;   // nian
   sortTable[0x57DD]=16881;   // nian
   sortTable[0x8E68]=16882;   // nian
   sortTable[0x9F30]=16883;   // nian
   sortTable[0x824C]=16884;   // nian
   sortTable[0x8E4D]=16885;   // nian
   sortTable[0x8DC8]=16886;   // nian
   sortTable[0x8F26]=16887;   // nian
   sortTable[0x9B8E]=16888;   // nian
   sortTable[0x5344]=16889;   // nian
   sortTable[0x54D6]=16890;   // nian
   sortTable[0x5538]=16891;   // nian
   sortTable[0x59E9]=16892;   // nian
   sortTable[0x649A]=16893;   // nian
   sortTable[0x6D8A]=16894;   // nian
   sortTable[0x6DF0]=16895;   // nian
   sortTable[0x713E]=16896;   // nian
   sortTable[0x79CA]=16897;   // nian
   sortTable[0x79E5]=16898;   // nian
   sortTable[0x7C10]=16899;   // nian
   sortTable[0x8E8E]=16900;   // nian
   sortTable[0x9BF0]=16901;   // nian
   sortTable[0x9D47]=16902;   // nian
   sortTable[0x5A18]=16903;   // niang
   sortTable[0x917F]=16904;   // niang
   sortTable[0x91C0]=16905;   // niang
   sortTable[0x5B22]=16906;   // niang
   sortTable[0x5B43]=16907;   // niang
   sortTable[0x91B8]=16908;   // niang
   sortTable[0x9E1F]=16909;   // niao
   sortTable[0x9CE5]=16910;   // niao
   sortTable[0x5C3F]=16911;   // niao
   sortTable[0x8885]=16912;   // niao
   sortTable[0x5B32]=16913;   // niao
   sortTable[0x8311]=16914;   // niao
   sortTable[0x8132]=16915;   // niao
   sortTable[0x892D]=16916;   // niao
   sortTable[0x8526]=16917;   // niao
   sortTable[0x5ACB]=16918;   // niao
   sortTable[0x5B1D]=16919;   // niao
   sortTable[0x6A22]=16920;   // niao
   sortTable[0x88CA]=16921;   // niao
   sortTable[0x634F]=16922;   // nie
   sortTable[0x6D85]=16923;   // nie
   sortTable[0x5B7D]=16924;   // nie
   sortTable[0x8042]=16925;   // nie
   sortTable[0x8E51]=16926;   // nie
   sortTable[0x55EB]=16927;   // nie
   sortTable[0x556E]=16928;   // nie
   sortTable[0x81EC]=16929;   // nie
   sortTable[0x8616]=16930;   // nie
   sortTable[0x954D]=16931;   // nie
   sortTable[0x954A]=16932;   // nie
   sortTable[0x989E]=16933;   // nie
   sortTable[0x9667]=16934;   // nie
   sortTable[0x9F67]=16935;   // nie
   sortTable[0x7CF5]=16936;   // nie
   sortTable[0x95D1]=16937;   // nie
   sortTable[0x7CF1]=16938;   // nie
   sortTable[0x81F2]=16939;   // nie
   sortTable[0x82F6]=16940;   // nie
   sortTable[0x83CD]=16941;   // nie
   sortTable[0x8B98]=16942;   // nie
   sortTable[0x8825]=16943;   // nie
   sortTable[0x7C4B]=16944;   // nie
   sortTable[0x9480]=16945;   // nie
   sortTable[0x9873]=16946;   // nie
   sortTable[0x7BDE]=16947;   // nie
   sortTable[0x9477]=16948;   // nie
   sortTable[0x8076]=16949;   // nie
   sortTable[0x93B3]=16950;   // nie
   sortTable[0x9689]=16951;   // nie
   sortTable[0x55A6]=16952;   // nie
   sortTable[0x565B]=16953;   // nie
   sortTable[0x5699]=16954;   // nie
   sortTable[0x56C1]=16955;   // nie
   sortTable[0x56D3]=16956;   // nie
   sortTable[0x573C]=16957;   // nie
   sortTable[0x5B7C]=16958;   // nie
   sortTable[0x5D72]=16959;   // nie
   sortTable[0x5DAD]=16960;   // nie
   sortTable[0x5DD5]=16961;   // nie
   sortTable[0x5E07]=16962;   // nie
   sortTable[0x60D7]=16963;   // nie
   sortTable[0x63D1]=16964;   // nie
   sortTable[0x6470]=16965;   // nie
   sortTable[0x655C]=16966;   // nie
   sortTable[0x67BF]=16967;   // nie
   sortTable[0x69F7]=16968;   // nie
   sortTable[0x6AF1]=16969;   // nie
   sortTable[0x6E7C]=16970;   // nie
   sortTable[0x75C6]=16971;   // nie
   sortTable[0x8E02]=16972;   // nie
   sortTable[0x8E17]=16973;   // nie
   sortTable[0x8EA1]=16974;   // nie
   sortTable[0x931C]=16975;   // nie
   sortTable[0x9448]=16976;   // nie
   sortTable[0x60A8]=16977;   // nin
   sortTable[0x56DC]=16978;   // nin
   sortTable[0x62F0]=16979;   // nin
   sortTable[0x810C]=16980;   // nin
   sortTable[0x51DD]=16981;   // ning
   sortTable[0x62E7]=16982;   // ning
   sortTable[0x64F0]=16983;   // ning
   sortTable[0x5B81]=16984;   // ning
   sortTable[0x5BE7]=16985;   // ning
   sortTable[0x549B]=16986;   // ning
   sortTable[0x5680]=16987;   // ning
   sortTable[0x4F5E]=16988;   // ning
   sortTable[0x72DE]=16989;   // ning
   sortTable[0x752F]=16990;   // ning
   sortTable[0x6CDE]=16991;   // ning
   sortTable[0x67E0]=16992;   // ning
   sortTable[0x804D]=16993;   // ning
   sortTable[0x82E7]=16994;   // ning
   sortTable[0x9E0B]=16995;   // ning
   sortTable[0x8079]=16996;   // ning
   sortTable[0x77C3]=16997;   // ning
   sortTable[0x9B21]=16998;   // ning
   sortTable[0x4FAB]=16999;   // ning
   sortTable[0x511C]=17000;   // ning
   sortTable[0x5B23]=17001;   // ning
   sortTable[0x5BCD]=17002;   // ning
   sortTable[0x5BD5]=17003;   // ning
   sortTable[0x5BD7]=17004;   // ning
   sortTable[0x5BDC]=17005;   // ning
   sortTable[0x6A63]=17006;   // ning
   sortTable[0x6AB8]=17007;   // ning
   sortTable[0x6F9D]=17008;   // ning
   sortTable[0x6FD8]=17009;   // ning
   sortTable[0x7370]=17010;   // ning
   sortTable[0x85B4]=17011;   // ning
   sortTable[0x944F]=17012;   // ning
   sortTable[0x725B]=17013;   // niu
   sortTable[0x626D]=17014;   // niu
   sortTable[0x7EBD]=17015;   // niu
   sortTable[0x7D10]=17016;   // niu
   sortTable[0x94AE]=17017;   // niu
   sortTable[0x599E]=17018;   // niu
   sortTable[0x5FF8]=17019;   // niu
   sortTable[0x72C3]=17020;   // niu
   sortTable[0x8842]=17021;   // niu
   sortTable[0x9775]=17022;   // niu
   sortTable[0x83A5]=17023;   // niu
   sortTable[0x9215]=17024;   // niu
   sortTable[0x6C7C]=17025;   // niu
   sortTable[0x7084]=17026;   // niu
   sortTable[0x725C]=17027;   // niu
   sortTable[0x519C]=17028;   // nong
   sortTable[0x8FB2]=17029;   // nong
   sortTable[0x5F04]=17030;   // nong
   sortTable[0x6D53]=17031;   // nong
   sortTable[0x6FC3]=17032;   // nong
   sortTable[0x4FAC]=17033;   // nong
   sortTable[0x54DD]=17034;   // nong
   sortTable[0x8113]=17035;   // nong
   sortTable[0x79FE]=17036;   // nong
   sortTable[0x91B2]=17037;   // nong
   sortTable[0x81BF]=17038;   // nong
   sortTable[0x7A60]=17039;   // nong
   sortTable[0x857D]=17040;   // nong
   sortTable[0x895B]=17041;   // nong
   sortTable[0x7E77]=17042;   // nong
   sortTable[0x5102]=17043;   // nong
   sortTable[0x5665]=17044;   // nong
   sortTable[0x630A]=17045;   // nong
   sortTable[0x6A82]=17046;   // nong
   sortTable[0x6B01]=17047;   // nong
   sortTable[0x71F6]=17048;   // nong
   sortTable[0x7651]=17049;   // nong
   sortTable[0x79AF]=17050;   // nong
   sortTable[0x8FB3]=17051;   // nong
   sortTable[0x9F48]=17052;   // nong
   sortTable[0x8028]=17053;   // nou
   sortTable[0x7FBA]=17054;   // nou
   sortTable[0x941E]=17055;   // nou
   sortTable[0x5542]=17056;   // nou
   sortTable[0x69C8]=17057;   // nou
   sortTable[0x6ABD]=17058;   // nou
   sortTable[0x7373]=17059;   // nou
   sortTable[0x8B73]=17060;   // nou
   sortTable[0x9392]=17061;   // nou
   sortTable[0x52AA]=17062;   // nu
   sortTable[0x6012]=17063;   // nu
   sortTable[0x5974]=17064;   // nu
   sortTable[0x5F29]=17065;   // nu
   sortTable[0x5B65]=17066;   // nu
   sortTable[0x9A7D]=17067;   // nu
   sortTable[0x80EC]=17068;   // nu
   sortTable[0x782E]=17069;   // nu
   sortTable[0x7B2F]=17070;   // nu
   sortTable[0x4F2E]=17071;   // nu
   sortTable[0x5089]=17072;   // nu
   sortTable[0x6419]=17073;   // nu
   sortTable[0x99D1]=17074;   // nu
   sortTable[0x6696]=17075;   // nuan
   sortTable[0x992A]=17076;   // nuan
   sortTable[0x597B]=17077;   // nuan
   sortTable[0x6E1C]=17078;   // nuan
   sortTable[0x7156]=17079;   // nuan
   sortTable[0x7157]=17080;   // nuan
   sortTable[0x8650]=17081;   // nüe
   sortTable[0x759F]=17082;   // nüe
   sortTable[0x7878]=17083;   // nüe
   sortTable[0x7627]=17084;   // nüe
   sortTable[0x9EC1]=17085;   // nun
   sortTable[0x8BFA]=17086;   // nuo
   sortTable[0x632A]=17087;   // nuo
   sortTable[0x61E6]=17088;   // nuo
   sortTable[0x50A9]=17089;   // nuo
   sortTable[0x558F]=17090;   // nuo
   sortTable[0x7CEF]=17091;   // nuo
   sortTable[0x6426]=17092;   // nuo
   sortTable[0x9518]=17093;   // nuo
   sortTable[0x8E43]=17094;   // nuo
   sortTable[0x90CD]=17095;   // nuo
   sortTable[0x7A2C]=17096;   // nuo
   sortTable[0x513A]=17097;   // nuo
   sortTable[0x611E]=17098;   // nuo
   sortTable[0x61E7]=17099;   // nuo
   sortTable[0x63BF]=17100;   // nuo
   sortTable[0x643B]=17101;   // nuo
   sortTable[0x689B]=17102;   // nuo
   sortTable[0x6992]=17103;   // nuo
   sortTable[0x6A60]=17104;   // nuo
   sortTable[0x7A64]=17105;   // nuo
   sortTable[0x7CD1]=17106;   // nuo
   sortTable[0x7CE5]=17107;   // nuo
   sortTable[0x8AFE]=17108;   // nuo
   sortTable[0x903D]=17109;   // nuo
   sortTable[0x5973]=17110;   // nv
   sortTable[0x9495]=17111;   // nv
   sortTable[0x8844]=17112;   // nv
   sortTable[0x6067]=17113;   // nv
   sortTable[0x7C79]=17114;   // nv
   sortTable[0x6712]=17115;   // nv
   sortTable[0x6C91]=17116;   // nv
   sortTable[0x91F9]=17117;   // nv
   sortTable[0x5662]=17408;   // o
   sortTable[0x5594]=17409;   // o
   sortTable[0x54E6]=17410;   // o
   sortTable[0x7B7D]=17411;   // o
   sortTable[0x5076]=17412;   // ou
   sortTable[0x6B27]=17413;   // ou
   sortTable[0x6B50]=17414;   // ou
   sortTable[0x9E25]=17415;   // ou
   sortTable[0x9DD7]=17416;   // ou
   sortTable[0x5455]=17417;   // ou
   sortTable[0x6BB4]=17418;   // ou
   sortTable[0x85D5]=17419;   // ou
   sortTable[0x74EF]=17420;   // ou
   sortTable[0x6CA4]=17421;   // ou
   sortTable[0x8BB4]=17422;   // ou
   sortTable[0x8026]=17423;   // ou
   sortTable[0x6004]=17424;   // ou
   sortTable[0x8B33]=17425;   // ou
   sortTable[0x8545]=17426;   // ou
   sortTable[0x8162]=17427;   // ou
   sortTable[0x93C2]=17428;   // ou
   sortTable[0x9F75]=17429;   // ou
   sortTable[0x85F2]=17430;   // ou
   sortTable[0x5418]=17431;   // ou
   sortTable[0x5614]=17432;   // ou
   sortTable[0x5878]=17433;   // ou
   sortTable[0x616A]=17434;   // ou
   sortTable[0x6AD9]=17435;   // ou
   sortTable[0x6BC6]=17436;   // ou
   sortTable[0x6F1A]=17437;   // ou
   sortTable[0x71B0]=17438;   // ou
   sortTable[0x750C]=17439;   // ou
   sortTable[0x8192]=17440;   // ou
   sortTable[0x9D0E]=17441;   // ou
   sortTable[0x6015]=17920;   // pa
   sortTable[0x722C]=17921;   // pa
   sortTable[0x556A]=17922;   // pa
   sortTable[0x8DB4]=17923;   // pa
   sortTable[0x5E15]=17924;   // pa
   sortTable[0x7436]=17925;   // pa
   sortTable[0x8469]=17926;   // pa
   sortTable[0x6777]=17927;   // pa
   sortTable[0x7B62]=17928;   // pa
   sortTable[0x8899]=17929;   // pa
   sortTable[0x8225]=17930;   // pa
   sortTable[0x5991]=17931;   // pa
   sortTable[0x5E0A]=17932;   // pa
   sortTable[0x63B1]=17933;   // pa
   sortTable[0x6F56]=17934;   // pa
   sortTable[0x7685]=17935;   // pa
   sortTable[0x6D3E]=17936;   // pai
   sortTable[0x6392]=17937;   // pai
   sortTable[0x62CD]=17938;   // pai
   sortTable[0x724C]=17939;   // pai
   sortTable[0x5F98]=17940;   // pai
   sortTable[0x6E43]=17941;   // pai
   sortTable[0x4FF3]=17942;   // pai
   sortTable[0x54CC]=17943;   // pai
   sortTable[0x848E]=17944;   // pai
   sortTable[0x7C30]=17945;   // pai
   sortTable[0x7C32]=17946;   // pai
   sortTable[0x8F2B]=17947;   // pai
   sortTable[0x5EF9]=17948;   // pai
   sortTable[0x68D1]=17949;   // pai
   sortTable[0x72A4]=17950;   // pai
   sortTable[0x7305]=17951;   // pai
   sortTable[0x9383]=17952;   // pai
   sortTable[0x5224]=17953;   // pan
   sortTable[0x76D8]=17954;   // pan
   sortTable[0x76E4]=17955;   // pan
   sortTable[0x76FC]=17956;   // pan
   sortTable[0x53DB]=17957;   // pan
   sortTable[0x6500]=17958;   // pan
   sortTable[0x7554]=17959;   // pan
   sortTable[0x6F58]=17960;   // pan
   sortTable[0x62DA]=17961;   // pan
   sortTable[0x78D0]=17962;   // pan
   sortTable[0x87E0]=17963;   // pan
   sortTable[0x88A2]=17964;   // pan
   sortTable[0x8E52]=17965;   // pan
   sortTable[0x6CEE]=17966;   // pan
   sortTable[0x97B6]=17967;   // pan
   sortTable[0x78FB]=17968;   // pan
   sortTable[0x92EC]=17969;   // pan
   sortTable[0x897B]=17970;   // pan
   sortTable[0x7705]=17971;   // pan
   sortTable[0x9816]=17972;   // pan
   sortTable[0x7E0F]=17973;   // pan
   sortTable[0x8DD8]=17974;   // pan
   sortTable[0x8E63]=17975;   // pan
   sortTable[0x7819]=17976;   // pan
   sortTable[0x947B]=17977;   // pan
   sortTable[0x84B0]=17978;   // pan
   sortTable[0x51B8]=17979;   // pan
   sortTable[0x5ABB]=17980;   // pan
   sortTable[0x5E4B]=17981;   // pan
   sortTable[0x642B]=17982;   // pan
   sortTable[0x69C3]=17983;   // pan
   sortTable[0x6C9C]=17984;   // pan
   sortTable[0x6D00]=17985;   // pan
   sortTable[0x6EBF]=17986;   // pan
   sortTable[0x700A]=17987;   // pan
   sortTable[0x708D]=17988;   // pan
   sortTable[0x723F]=17989;   // pan
   sortTable[0x7249]=17990;   // pan
   sortTable[0x7568]=17991;   // pan
   sortTable[0x8041]=17992;   // pan
   sortTable[0x8A4A]=17993;   // pan
   sortTable[0x939C]=17994;   // pan
   sortTable[0x9D65]=17995;   // pan
   sortTable[0x65C1]=17996;   // pang
   sortTable[0x80D6]=17997;   // pang
   sortTable[0x4E53]=17998;   // pang
   sortTable[0x5E9E]=17999;   // pang
   sortTable[0x9F90]=18000;   // pang
   sortTable[0x6EC2]=18001;   // pang
   sortTable[0x8783]=18002;   // pang
   sortTable[0x802A]=18003;   // pang
   sortTable[0x9004]=18004;   // pang
   sortTable[0x96F1]=18005;   // pang
   sortTable[0x9CD1]=18006;   // pang
   sortTable[0x9F8E]=18007;   // pang
   sortTable[0x5390]=18008;   // pang
   sortTable[0x5396]=18009;   // pang
   sortTable[0x55D9]=18010;   // pang
   sortTable[0x5ACE]=18011;   // pang
   sortTable[0x5FAC]=18012;   // pang
   sortTable[0x6C97]=18013;   // pang
   sortTable[0x7090]=18014;   // pang
   sortTable[0x80A8]=18015;   // pang
   sortTable[0x80EE]=18016;   // pang
   sortTable[0x8196]=18017;   // pang
   sortTable[0x823D]=18018;   // pang
   sortTable[0x89AB]=18019;   // pang
   sortTable[0x9736]=18020;   // pang
   sortTable[0x8DD1]=18021;   // pao
   sortTable[0x70AE]=18022;   // pao
   sortTable[0x6CE1]=18023;   // pao
   sortTable[0x629B]=18024;   // pao
   sortTable[0x62CB]=18025;   // pao
   sortTable[0x888D]=18026;   // pao
   sortTable[0x5486]=18027;   // pao
   sortTable[0x5228]=18028;   // pao
   sortTable[0x5E96]=18029;   // pao
   sortTable[0x7832]=18030;   // pao
   sortTable[0x530F]=18031;   // pao
   sortTable[0x75B1]=18032;   // pao
   sortTable[0x812C]=18033;   // pao
   sortTable[0x72CD]=18034;   // pao
   sortTable[0x9E83]=18035;   // pao
   sortTable[0x792E]=18036;   // pao
   sortTable[0x9784]=18037;   // pao
   sortTable[0x9E85]=18038;   // pao
   sortTable[0x791F]=18039;   // pao
   sortTable[0x8422]=18040;   // pao
   sortTable[0x891C]=18041;   // pao
   sortTable[0x5789]=18042;   // pao
   sortTable[0x5945]=18043;   // pao
   sortTable[0x70B0]=18044;   // pao
   sortTable[0x722E]=18045;   // pao
   sortTable[0x76B0]=18046;   // pao
   sortTable[0x8EF3]=18047;   // pao
   sortTable[0x9EAD]=18048;   // pao
   sortTable[0x914D]=18049;   // pei
   sortTable[0x57F9]=18050;   // pei
   sortTable[0x966A]=18051;   // pei
   sortTable[0x5478]=18052;   // pei
   sortTable[0x4F69]=18053;   // pei
   sortTable[0x8D54]=18054;   // pei
   sortTable[0x8CE0]=18055;   // pei
   sortTable[0x6C9B]=18056;   // pei
   sortTable[0x88F4]=18057;   // pei
   sortTable[0x8F94]=18058;   // pei
   sortTable[0x80DA]=18059;   // pei
   sortTable[0x65C6]=18060;   // pei
   sortTable[0x952B]=18061;   // pei
   sortTable[0x5E14]=18062;   // pei
   sortTable[0x9708]=18063;   // pei
   sortTable[0x9185]=18064;   // pei
   sortTable[0x8843]=18065;   // pei
   sortTable[0x84DC]=18066;   // pei
   sortTable[0x80A7]=18067;   // pei
   sortTable[0x99CD]=18068;   // pei
   sortTable[0x4F02]=18069;   // pei
   sortTable[0x4FD6]=18070;   // pei
   sortTable[0x59F5]=18071;   // pei
   sortTable[0x5D8F]=18072;   // pei
   sortTable[0x600C]=18073;   // pei
   sortTable[0x65BE]=18074;   // pei
   sortTable[0x67F8]=18075;   // pei
   sortTable[0x6BF0]=18076;   // pei
   sortTable[0x6D7F]=18077;   // pei
   sortTable[0x73EE]=18078;   // pei
   sortTable[0x7B29]=18079;   // pei
   sortTable[0x88F5]=18080;   // pei
   sortTable[0x8F61]=18081;   // pei
   sortTable[0x962B]=18082;   // pei
   sortTable[0x99B7]=18083;   // pei
   sortTable[0x76C6]=18084;   // pen
   sortTable[0x55B7]=18085;   // pen
   sortTable[0x5674]=18086;   // pen
   sortTable[0x6E53]=18087;   // pen
   sortTable[0x8450]=18088;   // pen
   sortTable[0x7FF8]=18089;   // pen
   sortTable[0x5460]=18090;   // pen
   sortTable[0x55AF]=18091;   // pen
   sortTable[0x6B55]=18092;   // pen
   sortTable[0x74EB]=18093;   // pen
   sortTable[0x670B]=18094;   // peng
   sortTable[0x78B0]=18095;   // peng
   sortTable[0x6367]=18096;   // peng
   sortTable[0x68DA]=18097;   // peng
   sortTable[0x81A8]=18098;   // peng
   sortTable[0x84EC]=18099;   // peng
   sortTable[0x6026]=18100;   // peng
   sortTable[0x7BF7]=18101;   // peng
   sortTable[0x7830]=18102;   // peng
   sortTable[0x6F8E]=18103;   // peng
   sortTable[0x5F6D]=18104;   // peng
   sortTable[0x9E4F]=18105;   // peng
   sortTable[0x70F9]=18106;   // peng
   sortTable[0x580B]=18107;   // peng
   sortTable[0x62A8]=18108;   // peng
   sortTable[0x87DB]=18109;   // peng
   sortTable[0x562D]=18110;   // peng
   sortTable[0x787C]=18111;   // peng
   sortTable[0x8283]=18112;   // peng
   sortTable[0x959B]=18113;   // peng
   sortTable[0x8F23]=18114;   // peng
   sortTable[0x7BE3]=18115;   // peng
   sortTable[0x9B05]=18116;   // peng
   sortTable[0x97FC]=18117;   // peng
   sortTable[0x78DE]=18118;   // peng
   sortTable[0x7E84]=18119;   // peng
   sortTable[0x8EEF]=18120;   // peng
   sortTable[0x9AFC]=18121;   // peng
   sortTable[0x7AFC]=18122;   // peng
   sortTable[0x8E2B]=18123;   // peng
   sortTable[0x930B]=18124;   // peng
   sortTable[0x97F8]=18125;   // peng
   sortTable[0x9D6C]=18126;   // peng
   sortTable[0x5017]=18127;   // peng
   sortTable[0x527B]=18128;   // peng
   sortTable[0x5309]=18129;   // peng
   sortTable[0x5873]=18130;   // peng
   sortTable[0x5F38]=18131;   // peng
   sortTable[0x6072]=18132;   // peng
   sortTable[0x6189]=18133;   // peng
   sortTable[0x6337]=18134;   // peng
   sortTable[0x63BD]=18135;   // peng
   sortTable[0x6888]=18136;   // peng
   sortTable[0x6916]=18137;   // peng
   sortTable[0x692A]=18138;   // peng
   sortTable[0x69F0]=18139;   // peng
   sortTable[0x6A25]=18140;   // peng
   sortTable[0x6DCE]=18141;   // peng
   sortTable[0x6F30]=18142;   // peng
   sortTable[0x71A2]=18143;   // peng
   sortTable[0x768F]=18144;   // peng
   sortTable[0x7851]=18145;   // peng
   sortTable[0x7A1D]=18146;   // peng
   sortTable[0x8391]=18147;   // peng
   sortTable[0x87DA]=18148;   // peng
   sortTable[0x945D]=18149;   // peng
   sortTable[0x9A2F]=18150;   // peng
   sortTable[0x9B14]=18151;   // peng
   sortTable[0x6279]=18152;   // pi
   sortTable[0x76AE]=18153;   // pi
   sortTable[0x62AB]=18154;   // pi
   sortTable[0x813E]=18155;   // pi
   sortTable[0x5339]=18156;   // pi
   sortTable[0x75B2]=18157;   // pi
   sortTable[0x8F9F]=18158;   // pi
   sortTable[0x5C41]=18159;   // pi
   sortTable[0x8B6C]=18160;   // pi
   sortTable[0x576F]=18161;   // pi
   sortTable[0x5288]=18162;   // pi
   sortTable[0x5564]=18163;   // pi
   sortTable[0x50FB]=18164;   // pi
   sortTable[0x4E15]=18165;   // pi
   sortTable[0x6BD7]=18166;   // pi
   sortTable[0x7435]=18167;   // pi
   sortTable[0x9739]=18168;   // pi
   sortTable[0x7656]=18169;   // pi
   sortTable[0x90B3]=18170;   // pi
   sortTable[0x7F74]=18171;   // pi
   sortTable[0x5AB2]=18172;   // pi
   sortTable[0x9F19]=18173;   // pi
   sortTable[0x6DE0]=18174;   // pi
   sortTable[0x7513]=18175;   // pi
   sortTable[0x75DE]=18176;   // pi
   sortTable[0x868D]=18177;   // pi
   sortTable[0x572E]=18178;   // pi
   sortTable[0x5E80]=18179;   // pi
   sortTable[0x567C]=18180;   // pi
   sortTable[0x6787]=18181;   // pi
   sortTable[0x7812]=18182;   // pi
   sortTable[0x9674]=18183;   // pi
   sortTable[0x8C94]=18184;   // pi
   sortTable[0x7EB0]=18185;   // pi
   sortTable[0x7765]=18186;   // pi
   sortTable[0x90EB]=18187;   // pi
   sortTable[0x94CD]=18188;   // pi
   sortTable[0x8298]=18189;   // pi
   sortTable[0x57E4]=18190;   // pi
   sortTable[0x8731]=18191;   // pi
   sortTable[0x64D7]=18192;   // pi
   sortTable[0x4EF3]=18193;   // pi
   sortTable[0x9C8F]=18194;   // pi
   sortTable[0x9E0A]=18195;   // pi
   sortTable[0x79E0]=18196;   // pi
   sortTable[0x925F]=18197;   // pi
   sortTable[0x9DFF]=18198;   // pi
   sortTable[0x818D]=18199;   // pi
   sortTable[0x99D3]=18200;   // pi
   sortTable[0x9B7E]=18201;   // pi
   sortTable[0x921A]=18202;   // pi
   sortTable[0x9B8D]=18203;   // pi
   sortTable[0x7914]=18204;   // pi
   sortTable[0x7BFA]=18205;   // pi
   sortTable[0x9294]=18206;   // pi
   sortTable[0x92A2]=18207;   // pi
   sortTable[0x95E2]=18208;   // pi
   sortTable[0x8157]=18209;   // pi
   sortTable[0x7FCD]=18210;   // pi
   sortTable[0x91FD]=18211;   // pi
   sortTable[0x930D]=18212;   // pi
   sortTable[0x9630]=18213;   // pi
   sortTable[0x9B6E]=18214;   // pi
   sortTable[0x7D15]=18215;   // pi
   sortTable[0x7F86]=18216;   // pi
   sortTable[0x8134]=18217;   // pi
   sortTable[0x86BE]=18218;   // pi
   sortTable[0x87B7]=18219;   // pi
   sortTable[0x9AEC]=18220;   // pi
   sortTable[0x9D04]=18221;   // pi
   sortTable[0x9D67]=18222;   // pi
   sortTable[0x4F13]=18223;   // pi
   sortTable[0x4F3E]=18224;   // pi
   sortTable[0x567D]=18225;   // pi
   sortTable[0x568A]=18226;   // pi
   sortTable[0x56AD]=18227;   // pi
   sortTable[0x58C0]=18228;   // pi
   sortTable[0x5AD3]=18229;   // pi
   sortTable[0x5CAF]=18230;   // pi
   sortTable[0x5D25]=18231;   // pi
   sortTable[0x6082]=18232;   // pi
   sortTable[0x61B5]=18233;   // pi
   sortTable[0x62B7]=18234;   // pi
   sortTable[0x63CA]=18235;   // pi
   sortTable[0x65C7]=18236;   // pi
   sortTable[0x6707]=18237;   // pi
   sortTable[0x698C]=18238;   // pi
   sortTable[0x6BD8]=18239;   // pi
   sortTable[0x6BDE]=18240;   // pi
   sortTable[0x6E12]=18241;   // pi
   sortTable[0x6F4E]=18242;   // pi
   sortTable[0x6FBC]=18243;   // pi
   sortTable[0x708B]=18244;   // pi
   sortTable[0x7137]=18245;   // pi
   sortTable[0x72C9]=18246;   // pi
   sortTable[0x72D3]=18247;   // pi
   sortTable[0x7588]=18248;   // pi
   sortTable[0x758B]=18249;   // pi
   sortTable[0x78C7]=18250;   // pi
   sortTable[0x7915]=18251;   // pi
   sortTable[0x79DB]=18252;   // pi
   sortTable[0x7A2B]=18253;   // pi
   sortTable[0x801A]=18254;   // pi
   sortTable[0x80B6]=18255;   // pi
   sortTable[0x82C9]=18256;   // pi
   sortTable[0x86BD]=18257;   // pi
   sortTable[0x8AC0]=18258;   // pi
   sortTable[0x8C7C]=18259;   // pi
   sortTable[0x8C7E]=18260;   // pi
   sortTable[0x9208]=18261;   // pi
   sortTable[0x9239]=18262;   // pi
   sortTable[0x7247]=18263;   // pian
   sortTable[0x7BC7]=18264;   // pian
   sortTable[0x504F]=18265;   // pian
   sortTable[0x9A97]=18266;   // pian
   sortTable[0x9A19]=18267;   // pian
   sortTable[0x7FE9]=18268;   // pian
   sortTable[0x8C1D]=18269;   // pian
   sortTable[0x9A88]=18270;   // pian
   sortTable[0x80FC]=18271;   // pian
   sortTable[0x8E41]=18272;   // pian
   sortTable[0x728F]=18273;   // pian
   sortTable[0x8DF0]=18274;   // pian
   sortTable[0x9B78]=18275;   // pian
   sortTable[0x8991]=18276;   // pian
   sortTable[0x8ADE]=18277;   // pian
   sortTable[0x8ADA]=18278;   // pian
   sortTable[0x9342]=18279;   // pian
   sortTable[0x99E2]=18280;   // pian
   sortTable[0x9ABF]=18281;   // pian
   sortTable[0x56E8]=18282;   // pian
   sortTable[0x5AA5]=18283;   // pian
   sortTable[0x6944]=18284;   // pian
   sortTable[0x6969]=18285;   // pian
   sortTable[0x8141]=18286;   // pian
   sortTable[0x8CB5]=18287;   // pian
   sortTable[0x8CC6]=18288;   // pian
   sortTable[0x9A08]=18289;   // pian
   sortTable[0x9A17]=18290;   // pian
   sortTable[0x9DA3]=18291;   // pian
   sortTable[0x7968]=18292;   // piao
   sortTable[0x98D8]=18293;   // piao
   sortTable[0x98C4]=18294;   // piao
   sortTable[0x6F02]=18295;   // piao
   sortTable[0x74E2]=18296;   // piao
   sortTable[0x5AD6]=18297;   // piao
   sortTable[0x527D]=18298;   // piao
   sortTable[0x7F25]=18299;   // piao
   sortTable[0x779F]=18300;   // piao
   sortTable[0x560C]=18301;   // piao
   sortTable[0x6B8D]=18302;   // piao
   sortTable[0x87B5]=18303;   // piao
   sortTable[0x9860]=18304;   // piao
   sortTable[0x7FF2]=18305;   // piao
   sortTable[0x7AC2]=18306;   // piao
   sortTable[0x7E39]=18307;   // piao
   sortTable[0x91A5]=18308;   // piao
   sortTable[0x9B52]=18309;   // piao
   sortTable[0x50C4]=18310;   // piao
   sortTable[0x52E1]=18311;   // piao
   sortTable[0x5F6F]=18312;   // piao
   sortTable[0x5FB1]=18313;   // piao
   sortTable[0x6153]=18314;   // piao
   sortTable[0x65DA]=18315;   // piao
   sortTable[0x72A5]=18316;   // piao
   sortTable[0x76AB]=18317;   // piao
   sortTable[0x7BFB]=18318;   // piao
   sortTable[0x85B8]=18319;   // piao
   sortTable[0x95DD]=18320;   // piao
   sortTable[0x98C3]=18321;   // piao
   sortTable[0x77A5]=18322;   // pie
   sortTable[0x6487]=18323;   // pie
   sortTable[0x82E4]=18324;   // pie
   sortTable[0x6C15]=18325;   // pie
   sortTable[0x4E3F]=18326;   // pie
   sortTable[0x5AF3]=18327;   // pie
   sortTable[0x6486]=18328;   // pie
   sortTable[0x66BC]=18329;   // pie
   sortTable[0x9405]=18330;   // pie
   sortTable[0x54C1]=18331;   // pin
   sortTable[0x62FC]=18332;   // pin
   sortTable[0x8D2B]=18333;   // pin
   sortTable[0x8CA7]=18334;   // pin
   sortTable[0x9891]=18335;   // pin
   sortTable[0x983B]=18336;   // pin
   sortTable[0x8058]=18337;   // pin
   sortTable[0x5AD4]=18338;   // pin
   sortTable[0x725D]=18339;   // pin
   sortTable[0x98A6]=18340;   // pin
   sortTable[0x59D8]=18341;   // pin
   sortTable[0x8819]=18342;   // pin
   sortTable[0x6980]=18343;   // pin
   sortTable[0x99AA]=18344;   // pin
   sortTable[0x77C9]=18345;   // pin
   sortTable[0x9870]=18346;   // pin
   sortTable[0x9A5E]=18347;   // pin
   sortTable[0x56AC]=18348;   // pin
   sortTable[0x5A26]=18349;   // pin
   sortTable[0x5B2A]=18350;   // pin
   sortTable[0x6C56]=18351;   // pin
   sortTable[0x7371]=18352;   // pin
   sortTable[0x73AD]=18353;   // pin
   sortTable[0x7415]=18354;   // pin
   sortTable[0x7917]=18355;   // pin
   sortTable[0x7A66]=18356;   // pin
   sortTable[0x85B2]=18357;   // pin
   sortTable[0x5E73]=18358;   // ping
   sortTable[0x8BC4]=18359;   // ping
   sortTable[0x8A55]=18360;   // ping
   sortTable[0x51ED]=18361;   // ping
   sortTable[0x6191]=18362;   // ping
   sortTable[0x74F6]=18363;   // ping
   sortTable[0x82F9]=18364;   // ping
   sortTable[0x860B]=18365;   // ping
   sortTable[0x4E52]=18366;   // ping
   sortTable[0x5C4F]=18367;   // ping
   sortTable[0x840D]=18368;   // ping
   sortTable[0x576A]=18369;   // ping
   sortTable[0x5A09]=18370;   // ping
   sortTable[0x9C86]=18371;   // ping
   sortTable[0x67B0]=18372;   // ping
   sortTable[0x4FDC]=18373;   // ping
   sortTable[0x8EFF]=18374;   // ping
   sortTable[0x8F27]=18375;   // ping
   sortTable[0x7F3E]=18376;   // ping
   sortTable[0x8353]=18377;   // ping
   sortTable[0x86E2]=18378;   // ping
   sortTable[0x90F1]=18379;   // ping
   sortTable[0x8060]=18380;   // ping
   sortTable[0x84F1]=18381;   // ping
   sortTable[0x7BB3]=18382;   // ping
   sortTable[0x782F]=18383;   // ping
   sortTable[0x80D3]=18384;   // ping
   sortTable[0x51F4]=18385;   // ping
   sortTable[0x546F]=18386;   // ping
   sortTable[0x5840]=18387;   // ping
   sortTable[0x5C5B]=18388;   // ping
   sortTable[0x5CBC]=18389;   // ping
   sortTable[0x5E21]=18390;   // ping
   sortTable[0x5E32]=18391;   // ping
   sortTable[0x5E48]=18392;   // ping
   sortTable[0x617F]=18393;   // ping
   sortTable[0x6A98]=18394;   // ping
   sortTable[0x6CD9]=18395;   // ping
   sortTable[0x6D34]=18396;   // ping
   sortTable[0x6D84]=18397;   // ping
   sortTable[0x6DDC]=18398;   // ping
   sortTable[0x7129]=18399;   // ping
   sortTable[0x73B6]=18400;   // ping
   sortTable[0x7501]=18401;   // ping
   sortTable[0x7539]=18402;   // ping
   sortTable[0x7AEE]=18403;   // ping
   sortTable[0x7C08]=18404;   // ping
   sortTable[0x8275]=18405;   // ping
   sortTable[0x86B2]=18406;   // ping
   sortTable[0x9829]=18407;   // ping
   sortTable[0x9B83]=18408;   // ping
   sortTable[0x7834]=18409;   // po
   sortTable[0x8FEB]=18410;   // po
   sortTable[0x5761]=18411;   // po
   sortTable[0x5A46]=18412;   // po
   sortTable[0x9887]=18413;   // po
   sortTable[0x9817]=18414;   // po
   sortTable[0x6CFC]=18415;   // po
   sortTable[0x6F51]=18416;   // po
   sortTable[0x6CCA]=18417;   // po
   sortTable[0x9B44]=18418;   // po
   sortTable[0x73C0]=18419;   // po
   sortTable[0x9131]=18420;   // po
   sortTable[0x53F5]=18421;   // po
   sortTable[0x7B38]=18422;   // po
   sortTable[0x7C95]=18423;   // po
   sortTable[0x948B]=18424;   // po
   sortTable[0x76A4]=18425;   // po
   sortTable[0x94B7]=18426;   // po
   sortTable[0x91D9]=18427;   // po
   sortTable[0x8B08]=18428;   // po
   sortTable[0x84AA]=18429;   // po
   sortTable[0x99CA]=18430;   // po
   sortTable[0x5619]=18431;   // po
   sortTable[0x5C00]=18432;   // po
   sortTable[0x5CA5]=18433;   // po
   sortTable[0x5CB6]=18434;   // po
   sortTable[0x6540]=18435;   // po
   sortTable[0x6622]=18436;   // po
   sortTable[0x6872]=18437;   // po
   sortTable[0x6AC7]=18438;   // po
   sortTable[0x6D26]=18439;   // po
   sortTable[0x6E8C]=18440;   // po
   sortTable[0x7087]=18441;   // po
   sortTable[0x70DE]=18442;   // po
   sortTable[0x7836]=18443;   // po
   sortTable[0x8522]=18444;   // po
   sortTable[0x9197]=18445;   // po
   sortTable[0x9255]=18446;   // po
   sortTable[0x93FA]=18447;   // po
   sortTable[0x5256]=18448;   // pou
   sortTable[0x88D2]=18449;   // pou
   sortTable[0x638A]=18450;   // pou
   sortTable[0x9307]=18451;   // pou
   sortTable[0x5485]=18452;   // pou
   sortTable[0x54E3]=18453;   // pou
   sortTable[0x5A1D]=18454;   // pou
   sortTable[0x5A44]=18455;   // pou
   sortTable[0x5ECD]=18456;   // pou
   sortTable[0x6294]=18457;   // pou
   sortTable[0x6299]=18458;   // pou
   sortTable[0x634A]=18459;   // pou
   sortTable[0x7283]=18460;   // pou
   sortTable[0x7B81]=18461;   // pou
   sortTable[0x666E]=18462;   // pu
   sortTable[0x6251]=18463;   // pu
   sortTable[0x64B2]=18464;   // pu
   sortTable[0x94FA]=18465;   // pu
   sortTable[0x92EA]=18466;   // pu
   sortTable[0x6734]=18467;   // pu
   sortTable[0x8461]=18468;   // pu
   sortTable[0x812F]=18469;   // pu
   sortTable[0x7011]=18470;   // pu
   sortTable[0x83E9]=18471;   // pu
   sortTable[0x4EC6]=18472;   // pu
   sortTable[0x6D66]=18473;   // pu
   sortTable[0x84B2]=18474;   // pu
   sortTable[0x8C31]=18475;   // pu
   sortTable[0x6FEE]=18476;   // pu
   sortTable[0x5703]=18477;   // pu
   sortTable[0x6EA5]=18478;   // pu
   sortTable[0x749E]=18479;   // pu
   sortTable[0x66DD]=18480;   // pu
   sortTable[0x5657]=18481;   // pu
   sortTable[0x8386]=18482;   // pu
   sortTable[0x530D]=18483;   // pu
   sortTable[0x917A]=18484;   // pu
   sortTable[0x6534]=18485;   // pu
   sortTable[0x8E7C]=18486;   // pu
   sortTable[0x84B1]=18487;   // pu
   sortTable[0x9568]=18488;   // pu
   sortTable[0x6C06]=18489;   // pu
   sortTable[0x9660]=18490;   // pu
   sortTable[0x9564]=18491;   // pu
   sortTable[0x9BC6]=18492;   // pu
   sortTable[0x8216]=18493;   // pu
   sortTable[0x7A59]=18494;   // pu
   sortTable[0x8D0C]=18495;   // pu
   sortTable[0x8B5C]=18496;   // pu
   sortTable[0x83D0]=18497;   // pu
   sortTable[0x7E80]=18498;   // pu
   sortTable[0x9420]=18499;   // pu
   sortTable[0x50D5]=18500;   // pu
   sortTable[0x5724]=18501;   // pu
   sortTable[0x58A3]=18502;   // pu
   sortTable[0x5DEC]=18503;   // pu
   sortTable[0x5DED]=18504;   // pu
   sortTable[0x64C8]=18505;   // pu
   sortTable[0x6A38]=18506;   // pu
   sortTable[0x6A8F]=18507;   // pu
   sortTable[0x6F7D]=18508;   // pu
   sortTable[0x70F3]=18509;   // pu
   sortTable[0x735B]=18510;   // pu
   sortTable[0x77A8]=18511;   // pu
   sortTable[0x8217]=18512;   // pu
   sortTable[0x8965]=18513;   // pu
   sortTable[0x8AE9]=18514;   // pu
   sortTable[0x93F7]=18515;   // pu
   sortTable[0x99C7]=18516;   // pu
   sortTable[0x8D77]=18944;   // qi
   sortTable[0x6C23]=18945;   // qi
   sortTable[0x6C14]=18946;   // qi
   sortTable[0x5176]=18947;   // qi
   sortTable[0x671F]=18948;   // qi
   sortTable[0x4E03]=18949;   // qi
   sortTable[0x5668]=18950;   // qi
   sortTable[0x6C7D]=18951;   // qi
   sortTable[0x5947]=18952;   // qi
   sortTable[0x4F01]=18953;   // qi
   sortTable[0x65D7]=18954;   // qi
   sortTable[0x9F50]=18955;   // qi
   sortTable[0x9F4A]=18956;   // qi
   sortTable[0x9A91]=18957;   // qi
   sortTable[0x9A0E]=18958;   // qi
   sortTable[0x6F06]=18959;   // qi
   sortTable[0x59BB]=18960;   // qi
   sortTable[0x542F]=18961;   // qi
   sortTable[0x555F]=18962;   // qi
   sortTable[0x6B3A]=18963;   // qi
   sortTable[0x621A]=18964;   // qi
   sortTable[0x5C82]=18965;   // qi
   sortTable[0x5F03]=18966;   // qi
   sortTable[0x8C48]=18967;   // qi
   sortTable[0x68C4]=18968;   // qi
   sortTable[0x780C]=18969;   // qi
   sortTable[0x68CB]=18970;   // qi
   sortTable[0x51C4]=18971;   // qi
   sortTable[0x6DD2]=18972;   // qi
   sortTable[0x6CE3]=18973;   // qi
   sortTable[0x5D0E]=18974;   // qi
   sortTable[0x6C8F]=18975;   // qi
   sortTable[0x4E5E]=18976;   // qi
   sortTable[0x5951]=18977;   // qi
   sortTable[0x7948]=18978;   // qi
   sortTable[0x4FDF]=18979;   // qi
   sortTable[0x6B67]=18980;   // qi
   sortTable[0x6816]=18981;   // qi
   sortTable[0x8BAB]=18982;   // qi
   sortTable[0x8FC4]=18983;   // qi
   sortTable[0x5C90]=18984;   // qi
   sortTable[0x7EEE]=18985;   // qi
   sortTable[0x7941]=18986;   // qi
   sortTable[0x7426]=18987;   // qi
   sortTable[0x8006]=18988;   // qi
   sortTable[0x742A]=18989;   // qi
   sortTable[0x9CCD]=18990;   // qi
   sortTable[0x675E]=18991;   // qi
   sortTable[0x69ED]=18992;   // qi
   sortTable[0x797A]=18993;   // qi
   sortTable[0x5601]=18994;   // qi
   sortTable[0x8E4A]=18995;   // qi
   sortTable[0x9E92]=18996;   // qi
   sortTable[0x6DC7]=18997;   // qi
   sortTable[0x8572]=18998;   // qi
   sortTable[0x789B]=18999;   // qi
   sortTable[0x86F4]=19000;   // qi
   sortTable[0x82AA]=19001;   // qi
   sortTable[0x7947]=19002;   // qi
   sortTable[0x8110]=19003;   // qi
   sortTable[0x951C]=19004;   // qi
   sortTable[0x847A]=19005;   // qi
   sortTable[0x61A9]=19006;   // qi
   sortTable[0x8401]=19007;   // qi
   sortTable[0x840B]=19008;   // qi
   sortTable[0x4E93]=19009;   // qi
   sortTable[0x7DA6]=19010;   // qi
   sortTable[0x8291]=19011;   // qi
   sortTable[0x573B]=19012;   // qi
   sortTable[0x7566]=19013;   // qi
   sortTable[0x9A90]=19014;   // qi
   sortTable[0x9880]=19015;   // qi
   sortTable[0x8DC2]=19016;   // qi
   sortTable[0x7DAE]=19017;   // qi
   sortTable[0x90EA]=19018;   // qi
   sortTable[0x6864]=19019;   // qi
   sortTable[0x6C54]=19020;   // qi
   sortTable[0x67D2]=19021;   // qi
   sortTable[0x78B6]=19022;   // qi
   sortTable[0x871E]=19023;   // qi
   sortTable[0x7881]=19024;   // qi
   sortTable[0x5C7A]=19025;   // qi
   sortTable[0x9B10]=19026;   // qi
   sortTable[0x8AC6]=19027;   // qi
   sortTable[0x80B5]=19028;   // qi
   sortTable[0x76F5]=19029;   // qi
   sortTable[0x81CD]=19030;   // qi
   sortTable[0x9321]=19031;   // qi
   sortTable[0x7895]=19032;   // qi
   sortTable[0x78E7]=19033;   // qi
   sortTable[0x79A5]=19034;   // qi
   sortTable[0x7F3C]=19035;   // qi
   sortTable[0x93DA]=19036;   // qi
   sortTable[0x9094]=19037;   // qi
   sortTable[0x86E3]=19038;   // qi
   sortTable[0x9CAF]=19039;   // qi
   sortTable[0x9F1C]=19040;   // qi
   sortTable[0x78DC]=19041;   // qi
   sortTable[0x8691]=19042;   // qi
   sortTable[0x8694]=19043;   // qi
   sortTable[0x78E9]=19044;   // qi
   sortTable[0x9BD5]=19045;   // qi
   sortTable[0x9EA1]=19046;   // qi
   sortTable[0x7C2F]=19047;   // qi
   sortTable[0x980E]=19048;   // qi
   sortTable[0x7CB8]=19049;   // qi
   sortTable[0x9B4C]=19050;   // qi
   sortTable[0x85C4]=19051;   // qi
   sortTable[0x8E11]=19052;   // qi
   sortTable[0x9D80]=19053;   // qi
   sortTable[0x9D88]=19054;   // qi
   sortTable[0x7C31]=19055;   // qi
   sortTable[0x7DA5]=19056;   // qi
   sortTable[0x7F4A]=19057;   // qi
   sortTable[0x8269]=19058;   // qi
   sortTable[0x87A7]=19059;   // qi
   sortTable[0x87FF]=19060;   // qi
   sortTable[0x970B]=19061;   // qi
   sortTable[0x9B3F]=19062;   // qi
   sortTable[0x9C2D]=19063;   // qi
   sortTable[0x77F5]=19064;   // qi
   sortTable[0x7AD2]=19065;   // qi
   sortTable[0x7C4F]=19066;   // qi
   sortTable[0x7DC0]=19067;   // qi
   sortTable[0x7E83]=19068;   // qi
   sortTable[0x8415]=19069;   // qi
   sortTable[0x869A]=19070;   // qi
   sortTable[0x8A16]=19071;   // qi
   sortTable[0x8EDD]=19072;   // qi
   sortTable[0x95D9]=19073;   // qi
   sortTable[0x4E9D]=19074;   // qi
   sortTable[0x501B]=19075;   // qi
   sortTable[0x50DB]=19076;   // qi
   sortTable[0x5258]=19077;   // qi
   sortTable[0x5447]=19078;   // qi
   sortTable[0x546E]=19079;   // qi
   sortTable[0x54A0]=19080;   // qi
   sortTable[0x5518]=19081;   // qi
   sortTable[0x552D]=19082;   // qi
   sortTable[0x5553]=19083;   // qi
   sortTable[0x5554]=19084;   // qi
   sortTable[0x5650]=19085;   // qi
   sortTable[0x57FC]=19086;   // qi
   sortTable[0x5921]=19087;   // qi
   sortTable[0x5A38]=19088;   // qi
   sortTable[0x5A4D]=19089;   // qi
   sortTable[0x5C93]=19090;   // qi
   sortTable[0x5E3A]=19091;   // qi
   sortTable[0x5FD4]=19092;   // qi
   sortTable[0x5FEF]=19093;   // qi
   sortTable[0x60BD]=19094;   // qi
   sortTable[0x612D]=19095;   // qi
   sortTable[0x617C]=19096;   // qi
   sortTable[0x617D]=19097;   // qi
   sortTable[0x6187]=19098;   // qi
   sortTable[0x61E0]=19099;   // qi
   sortTable[0x6391]=19100;   // qi
   sortTable[0x6456]=19101;   // qi
   sortTable[0x6532]=19102;   // qi
   sortTable[0x6589]=19103;   // qi
   sortTable[0x658A]=19104;   // qi
   sortTable[0x65C2]=19105;   // qi
   sortTable[0x6675]=19106;   // qi
   sortTable[0x66A3]=19107;   // qi
   sortTable[0x6814]=19108;   // qi
   sortTable[0x687C]=19109;   // qi
   sortTable[0x68CA]=19110;   // qi
   sortTable[0x68E8]=19111;   // qi
   sortTable[0x68F2]=19112;   // qi
   sortTable[0x69BF]=19113;   // qi
   sortTable[0x6AB1]=19114;   // qi
   sortTable[0x6AC0]=19115;   // qi
   sortTable[0x6B2B]=19116;   // qi
   sortTable[0x6C17]=19117;   // qi
   sortTable[0x6E0F]=19118;   // qi
   sortTable[0x6E46]=19119;   // qi
   sortTable[0x6E47]=19120;   // qi
   sortTable[0x6FDD]=19121;   // qi
   sortTable[0x7081]=19122;   // qi
   sortTable[0x7309]=19123;   // qi
   sortTable[0x7382]=19124;   // qi
   sortTable[0x7398]=19125;   // qi
   sortTable[0x7482]=19126;   // qi
   sortTable[0x7508]=19127;   // qi
   sortTable[0x7541]=19128;   // qi
   sortTable[0x75A7]=19129;   // qi
   sortTable[0x76C0]=19130;   // qi
   sortTable[0x7DA8]=19131;   // qi
   sortTable[0x7DBA]=19132;   // qi
   sortTable[0x7DD5]=19133;   // qi
   sortTable[0x829E]=19134;   // qi
   sortTable[0x8604]=19135;   // qi
   sortTable[0x871D]=19136;   // qi
   sortTable[0x8810]=19137;   // qi
   sortTable[0x8904]=19138;   // qi
   sortTable[0x8AEC]=19139;   // qi
   sortTable[0x8AFF]=19140;   // qi
   sortTable[0x8FC9]=19141;   // qi
   sortTable[0x91EE]=19142;   // qi
   sortTable[0x9A0F]=19143;   // qi
   sortTable[0x9D78]=19144;   // qi
   sortTable[0x6070]=19145;   // qia
   sortTable[0x6390]=19146;   // qia
   sortTable[0x6D3D]=19147;   // qia
   sortTable[0x845C]=19148;   // qia
   sortTable[0x9AC2]=19149;   // qia
   sortTable[0x8DD2]=19150;   // qia
   sortTable[0x9790]=19151;   // qia
   sortTable[0x7848]=19152;   // qia
   sortTable[0x51BE]=19153;   // qia
   sortTable[0x5736]=19154;   // qia
   sortTable[0x5E22]=19155;   // qia
   sortTable[0x6118]=19156;   // qia
   sortTable[0x62E4]=19157;   // qia
   sortTable[0x6B8E]=19158;   // qia
   sortTable[0x9160]=19159;   // qia
   sortTable[0x524D]=19160;   // qian
   sortTable[0x5343]=19161;   // qian
   sortTable[0x94B1]=19162;   // qian
   sortTable[0x9322]=19163;   // qian
   sortTable[0x7275]=19164;   // qian
   sortTable[0x727D]=19165;   // qian
   sortTable[0x6D45]=19166;   // qian
   sortTable[0x6DFA]=19167;   // qian
   sortTable[0x7B7E]=19168;   // qian
   sortTable[0x7C3D]=19169;   // qian
   sortTable[0x94C5]=19170;   // qian
   sortTable[0x925B]=19171;   // qian
   sortTable[0x6B20]=19172;   // qian
   sortTable[0x6F5C]=19173;   // qian
   sortTable[0x6F5B]=19174;   // qian
   sortTable[0x8FC1]=19175;   // qian
   sortTable[0x9077]=19176;   // qian
   sortTable[0x9063]=19177;   // qian
   sortTable[0x8C26]=19178;   // qian
   sortTable[0x8B19]=19179;   // qian
   sortTable[0x6B49]=19180;   // qian
   sortTable[0x5D4C]=19181;   // qian
   sortTable[0x94B3]=19182;   // qian
   sortTable[0x9257]=19183;   // qian
   sortTable[0x8C34]=19184;   // qian
   sortTable[0x8B74]=19185;   // qian
   sortTable[0x4E7E]=19186;   // qian
   sortTable[0x8654]=19187;   // qian
   sortTable[0x5029]=19188;   // qian
   sortTable[0x4F65]=19189;   // qian
   sortTable[0x5811]=19190;   // qian
   sortTable[0x94A4]=19191;   // qian
   sortTable[0x831C]=19192;   // qian
   sortTable[0x9ED4]=19193;   // qian
   sortTable[0x6106]=19194;   // qian
   sortTable[0x9A9E]=19195;   // qian
   sortTable[0x80B7]=19196;   // qian
   sortTable[0x6434]=19197;   // qian
   sortTable[0x82A1]=19198;   // qian
   sortTable[0x8930]=19199;   // qian
   sortTable[0x9621]=19200;   // qian
   sortTable[0x7F31]=19201;   // qian
   sortTable[0x828A]=19202;   // qian
   sortTable[0x614A]=19203;   // qian
   sortTable[0x60AD]=19204;   // qian
   sortTable[0x7B9D]=19205;   // qian
   sortTable[0x63AE]=19206;   // qian
   sortTable[0x84A8]=19207;   // qian
   sortTable[0x4EDF]=19208;   // qian
   sortTable[0x6266]=19209;   // qian
   sortTable[0x948E]=19210;   // qian
   sortTable[0x6920]=19211;   // qian
   sortTable[0x7C64]=19212;   // qian
   sortTable[0x9206]=19213;   // qian
   sortTable[0x8C38]=19214;   // qian
   sortTable[0x5C8D]=19215;   // qian
   sortTable[0x7DAA]=19216;   // qian
   sortTable[0x9C1C]=19217;   // qian
   sortTable[0x976C]=19218;   // qian
   sortTable[0x8AD0]=19219;   // qian
   sortTable[0x8F24]=19220;   // qian
   sortTable[0x8EE1]=19221;   // qian
   sortTable[0x8541]=19222;   // qian
   sortTable[0x9A1D]=19223;   // qian
   sortTable[0x9C2C]=19224;   // qian
   sortTable[0x7C56]=19225;   // qian
   sortTable[0x7E34]=19226;   // qian
   sortTable[0x8181]=19227;   // qian
   sortTable[0x9B1C]=19228;   // qian
   sortTable[0x9E50]=19229;   // qian
   sortTable[0x9EDA]=19230;   // qian
   sortTable[0x7ACF]=19231;   // qian
   sortTable[0x7BDF]=19232;   // qian
   sortTable[0x8688]=19233;   // qian
   sortTable[0x9A2B]=19234;   // qian
   sortTable[0x9B1D]=19235;   // qian
   sortTable[0x7B9E]=19236;   // qian
   sortTable[0x7BCF]=19237;   // qian
   sortTable[0x7FAC]=19238;   // qian
   sortTable[0x8125]=19239;   // qian
   sortTable[0x8738]=19240;   // qian
   sortTable[0x97C6]=19241;   // qian
   sortTable[0x9845]=19242;   // qian
   sortTable[0x4EF1]=19243;   // qian
   sortTable[0x4FD4]=19244;   // qian
   sortTable[0x5094]=19245;   // qian
   sortTable[0x50C9]=19246;   // qian
   sortTable[0x5119]=19247;   // qian
   sortTable[0x515B]=19248;   // qian
   sortTable[0x51F5]=19249;   // qian
   sortTable[0x520B]=19250;   // qian
   sortTable[0x55DB]=19251;   // qian
   sortTable[0x5731]=19252;   // qian
   sortTable[0x5732]=19253;   // qian
   sortTable[0x5879]=19254;   // qian
   sortTable[0x5898]=19255;   // qian
   sortTable[0x58CD]=19256;   // qian
   sortTable[0x5977]=19257;   // qian
   sortTable[0x5A5C]=19258;   // qian
   sortTable[0x5A8A]=19259;   // qian
   sortTable[0x5B45]=19260;   // qian
   sortTable[0x5B6F]=19261;   // qian
   sortTable[0x5C92]=19262;   // qian
   sortTable[0x5D70]=19263;   // qian
   sortTable[0x5FF4]=19264;   // qian
   sortTable[0x6093]=19265;   // qian
   sortTable[0x6173]=19266;   // qian
   sortTable[0x6272]=19267;   // qian
   sortTable[0x62D1]=19268;   // qian
   sortTable[0x62EA]=19269;   // qian
   sortTable[0x6394]=19270;   // qian
   sortTable[0x63F5]=19271;   // qian
   sortTable[0x6481]=19272;   // qian
   sortTable[0x6510]=19273;   // qian
   sortTable[0x6511]=19274;   // qian
   sortTable[0x6513]=19275;   // qian
   sortTable[0x6744]=19276;   // qian
   sortTable[0x68C8]=19277;   // qian
   sortTable[0x69A9]=19278;   // qian
   sortTable[0x69CF]=19279;   // qian
   sortTable[0x69E7]=19280;   // qian
   sortTable[0x6A6C]=19281;   // qian
   sortTable[0x6AB6]=19282;   // qian
   sortTable[0x6ACF]=19283;   // qian
   sortTable[0x6B26]=19284;   // qian
   sortTable[0x6B6C]=19285;   // qian
   sortTable[0x6C58]=19286;   // qian
   sortTable[0x6C67]=19287;   // qian
   sortTable[0x6FF3]=19288;   // qian
   sortTable[0x704A]=19289;   // qian
   sortTable[0x74E9]=19290;   // qian
   sortTable[0x7698]=19291;   // qian
   sortTable[0x7C81]=19292;   // qian
   sortTable[0x7E7E]=19293;   // qian
   sortTable[0x81E4]=19294;   // qian
   sortTable[0x833E]=19295;   // qian
   sortTable[0x8533]=19296;   // qian
   sortTable[0x91FA]=19297;   // qian
   sortTable[0x9210]=19298;   // qian
   sortTable[0x92AD]=19299;   // qian
   sortTable[0x9386]=19300;   // qian
   sortTable[0x93F2]=19301;   // qian
   sortTable[0x9453]=19302;   // qian
   sortTable[0x96C3]=19303;   // qian
   sortTable[0x9A1A]=19304;   // qian
   sortTable[0x9D6E]=19305;   // qian
   sortTable[0x5F3A]=19306;   // qiang
   sortTable[0x5F37]=19307;   // qiang
   sortTable[0x67AA]=19308;   // qiang
   sortTable[0x69CD]=19309;   // qiang
   sortTable[0x5899]=19310;   // qiang
   sortTable[0x7246]=19311;   // qiang
   sortTable[0x62A2]=19312;   // qiang
   sortTable[0x6436]=19313;   // qiang
   sortTable[0x8154]=19314;   // qiang
   sortTable[0x7F8C]=19315;   // qiang
   sortTable[0x8DC4]=19316;   // qiang
   sortTable[0x545B]=19317;   // qiang
   sortTable[0x9535]=19318;   // qiang
   sortTable[0x8537]=19319;   // qiang
   sortTable[0x6A2F]=19320;   // qiang
   sortTable[0x6215]=19321;   // qiang
   sortTable[0x5AF1]=19322;   // qiang
   sortTable[0x9516]=19323;   // qiang
   sortTable[0x8941]=19324;   // qiang
   sortTable[0x709D]=19325;   // qiang
   sortTable[0x7F9F]=19326;   // qiang
   sortTable[0x9397]=19327;   // qiang
   sortTable[0x956A]=19328;   // qiang
   sortTable[0x8723]=19329;   // qiang
   sortTable[0x6217]=19330;   // qiang
   sortTable[0x7E66]=19331;   // qiang
   sortTable[0x7F97]=19332;   // qiang
   sortTable[0x8E61]=19333;   // qiang
   sortTable[0x8262]=19334;   // qiang
   sortTable[0x9306]=19335;   // qiang
   sortTable[0x7BEC]=19336;   // qiang
   sortTable[0x7E48]=19337;   // qiang
   sortTable[0x7FAB]=19338;   // qiang
   sortTable[0x7FBB]=19339;   // qiang
   sortTable[0x93D8]=19340;   // qiang
   sortTable[0x8594]=19341;   // qiang
   sortTable[0x8E4C]=19342;   // qiang
   sortTable[0x93F9]=19343;   // qiang
   sortTable[0x4E2C]=19344;   // qiang
   sortTable[0x5534]=19345;   // qiang
   sortTable[0x55C6]=19346;   // qiang
   sortTable[0x55F4]=19347;   // qiang
   sortTable[0x588F]=19348;   // qiang
   sortTable[0x58BB]=19349;   // qiang
   sortTable[0x5B19]=19350;   // qiang
   sortTable[0x5D88]=19351;   // qiang
   sortTable[0x5EE7]=19352;   // qiang
   sortTable[0x6227]=19353;   // qiang
   sortTable[0x65A8]=19354;   // qiang
   sortTable[0x690C]=19355;   // qiang
   sortTable[0x6AA3]=19356;   // qiang
   sortTable[0x6EAC]=19357;   // qiang
   sortTable[0x6F12]=19358;   // qiang
   sortTable[0x7197]=19359;   // qiang
   sortTable[0x7244]=19360;   // qiang
   sortTable[0x7310]=19361;   // qiang
   sortTable[0x7347]=19362;   // qiang
   sortTable[0x73B1]=19363;   // qiang
   sortTable[0x7472]=19364;   // qiang
   sortTable[0x7FA5]=19365;   // qiang
   sortTable[0x8503]=19366;   // qiang
   sortTable[0x8620]=19367;   // qiang
   sortTable[0x8B12]=19368;   // qiang
   sortTable[0x6865]=19369;   // qiao
   sortTable[0x6A4B]=19370;   // qiao
   sortTable[0x77A7]=19371;   // qiao
   sortTable[0x6084]=19372;   // qiao
   sortTable[0x5DE7]=19373;   // qiao
   sortTable[0x6572]=19374;   // qiao
   sortTable[0x7FD8]=19375;   // qiao
   sortTable[0x7FF9]=19376;   // qiao
   sortTable[0x9539]=19377;   // qiao
   sortTable[0x936C]=19378;   // qiao
   sortTable[0x5CED]=19379;   // qiao
   sortTable[0x4E54]=19380;   // qiao
   sortTable[0x4FA8]=19381;   // qiao
   sortTable[0x7A8D]=19382;   // qiao
   sortTable[0x4FCF]=19383;   // qiao
   sortTable[0x6A35]=19384;   // qiao
   sortTable[0x9798]=19385;   // qiao
   sortTable[0x8DF7]=19386;   // qiao
   sortTable[0x8BEE]=19387;   // qiao
   sortTable[0x64AC]=19388;   // qiao
   sortTable[0x6194]=19389;   // qiao
   sortTable[0x8C2F]=19390;   // qiao
   sortTable[0x6A47]=19391;   // qiao
   sortTable[0x9792]=19392;   // qiao
   sortTable[0x6100]=19393;   // qiao
   sortTable[0x7857]=19394;   // qiao
   sortTable[0x835E]=19395;   // qiao
   sortTable[0x83EC]=19396;   // qiao
   sortTable[0x8DAB]=19397;   // qiao
   sortTable[0x5281]=19398;   // qiao
   sortTable[0x8DAC]=19399;   // qiao
   sortTable[0x7F32]=19400;   // qiao
   sortTable[0x9121]=19401;   // qiao
   sortTable[0x7AC5]=19402;   // qiao
   sortTable[0x9AB9]=19403;   // qiao
   sortTable[0x785A]=19404;   // qiao
   sortTable[0x7E51]=19405;   // qiao
   sortTable[0x78FD]=19406;   // qiao
   sortTable[0x85EE]=19407;   // qiao
   sortTable[0x834D]=19408;   // qiao
   sortTable[0x91E5]=19409;   // qiao
   sortTable[0x7904]=19410;   // qiao
   sortTable[0x9408]=19411;   // qiao
   sortTable[0x9430]=19412;   // qiao
   sortTable[0x9657]=19413;   // qiao
   sortTable[0x8E88]=19414;   // qiao
   sortTable[0x9125]=19415;   // qiao
   sortTable[0x936B]=19416;   // qiao
   sortTable[0x97BD]=19417;   // qiao
   sortTable[0x97D2]=19418;   // qiao
   sortTable[0x981D]=19419;   // qiao
   sortTable[0x50D1]=19420;   // qiao
   sortTable[0x50FA]=19421;   // qiao
   sortTable[0x55AC]=19422;   // qiao
   sortTable[0x563A]=19423;   // qiao
   sortTable[0x589D]=19424;   // qiao
   sortTable[0x58BD]=19425;   // qiao
   sortTable[0x5AF6]=19426;   // qiao
   sortTable[0x5D6A]=19427;   // qiao
   sortTable[0x5E29]=19428;   // qiao
   sortTable[0x5E67]=19429;   // qiao
   sortTable[0x64BD]=19430;   // qiao
   sortTable[0x69D7]=19431;   // qiao
   sortTable[0x6BBB]=19432;   // qiao
   sortTable[0x6BC3]=19433;   // qiao
   sortTable[0x71C6]=19434;   // qiao
   sortTable[0x729E]=19435;   // qiao
   sortTable[0x7644]=19436;   // qiao
   sortTable[0x854E]=19437;   // qiao
   sortTable[0x8A9A]=19438;   // qiao
   sortTable[0x8B59]=19439;   // qiao
   sortTable[0x8E0D]=19440;   // qiao
   sortTable[0x8E7A]=19441;   // qiao
   sortTable[0x90FB]=19442;   // qiao
   sortTable[0x9866]=19443;   // qiao
   sortTable[0x9ADA]=19444;   // qiao
   sortTable[0x9ADC]=19445;   // qiao
   sortTable[0x4E14]=19446;   // qie
   sortTable[0x5207]=19447;   // qie
   sortTable[0x7A83]=19448;   // qie
   sortTable[0x7ACA]=19449;   // qie
   sortTable[0x602F]=19450;   // qie
   sortTable[0x59BE]=19451;   // qie
   sortTable[0x6308]=19452;   // qie
   sortTable[0x60EC]=19453;   // qie
   sortTable[0x7BA7]=19454;   // qie
   sortTable[0x9532]=19455;   // qie
   sortTable[0x90C4]=19456;   // qie
   sortTable[0x9411]=19457;   // qie
   sortTable[0x807A]=19458;   // qie
   sortTable[0x8E25]=19459;   // qie
   sortTable[0x9365]=19460;   // qie
   sortTable[0x7B21]=19461;   // qie
   sortTable[0x7C61]=19462;   // qie
   sortTable[0x7BCB]=19463;   // qie
   sortTable[0x7DC1]=19464;   // qie
   sortTable[0x86EA]=19465;   // qie
   sortTable[0x503F]=19466;   // qie
   sortTable[0x5327]=19467;   // qie
   sortTable[0x5AAB]=19468;   // qie
   sortTable[0x608F]=19469;   // qie
   sortTable[0x611C]=19470;   // qie
   sortTable[0x6705]=19471;   // qie
   sortTable[0x6D2F]=19472;   // qie
   sortTable[0x6DC1]=19473;   // qie
   sortTable[0x767F]=19474;   // qie
   sortTable[0x7A55]=19475;   // qie
   sortTable[0x82C6]=19476;   // qie
   sortTable[0x85D2]=19477;   // qie
   sortTable[0x9BDC]=19478;   // qie
   sortTable[0x4EB2]=19479;   // qin
   sortTable[0x89AA]=19480;   // qin
   sortTable[0x4FB5]=19481;   // qin
   sortTable[0x52E4]=19482;   // qin
   sortTable[0x7434]=19483;   // qin
   sortTable[0x94A6]=19484;   // qin
   sortTable[0x6B3D]=19485;   // qin
   sortTable[0x79E6]=19486;   // qin
   sortTable[0x64D2]=19487;   // qin
   sortTable[0x5BDD]=19488;   // qin
   sortTable[0x79BD]=19489;   // qin
   sortTable[0x6C81]=19490;   // qin
   sortTable[0x887E]=19491;   // qin
   sortTable[0x82B9]=19492;   // qin
   sortTable[0x82A9]=19493;   // qin
   sortTable[0x5423]=19494;   // qin
   sortTable[0x55EA]=19495;   // qin
   sortTable[0x6A8E]=19496;   // qin
   sortTable[0x5659]=19497;   // qin
   sortTable[0x9513]=19498;   // qin
   sortTable[0x63FF]=19499;   // qin
   sortTable[0x6EB1]=19500;   // qin
   sortTable[0x9A8E]=19501;   // qin
   sortTable[0x8793]=19502;   // qin
   sortTable[0x9BBC]=19503;   // qin
   sortTable[0x99F8]=19504;   // qin
   sortTable[0x8804]=19505;   // qin
   sortTable[0x83E6]=19506;   // qin
   sortTable[0x9219]=19507;   // qin
   sortTable[0x9B35]=19508;   // qin
   sortTable[0x7D85]=19509;   // qin
   sortTable[0x9772]=19510;   // qin
   sortTable[0x7B09]=19511;   // qin
   sortTable[0x83E3]=19512;   // qin
   sortTable[0x83F3]=19513;   // qin
   sortTable[0x85FD]=19514;   // qin
   sortTable[0x92DF]=19515;   // qin
   sortTable[0x96C2]=19516;   // qin
   sortTable[0x9849]=19517;   // qin
   sortTable[0x5422]=19518;   // qin
   sortTable[0x551A]=19519;   // qin
   sortTable[0x5745]=19520;   // qin
   sortTable[0x57C1]=19521;   // qin
   sortTable[0x5A87]=19522;   // qin
   sortTable[0x5AC0]=19523;   // qin
   sortTable[0x5BD1]=19524;   // qin
   sortTable[0x5BE2]=19525;   // qin
   sortTable[0x5BF4]=19526;   // qin
   sortTable[0x5D5A]=19527;   // qin
   sortTable[0x5D94]=19528;   // qin
   sortTable[0x5E88]=19529;   // qin
   sortTable[0x616C]=19530;   // qin
   sortTable[0x61C3]=19531;   // qin
   sortTable[0x61C4]=19532;   // qin
   sortTable[0x628B]=19533;   // qin
   sortTable[0x6366]=19534;   // qin
   sortTable[0x6407]=19535;   // qin
   sortTable[0x64B3]=19536;   // qin
   sortTable[0x65B3]=19537;   // qin
   sortTable[0x6611]=19538;   // qin
   sortTable[0x68AB]=19539;   // qin
   sortTable[0x6FBF]=19540;   // qin
   sortTable[0x7019]=19541;   // qin
   sortTable[0x73E1]=19542;   // qin
   sortTable[0x7439]=19543;   // qin
   sortTable[0x763D]=19544;   // qin
   sortTable[0x8039]=19545;   // qin
   sortTable[0x8699]=19546;   // qin
   sortTable[0x87BC]=19547;   // qin
   sortTable[0x8A9B]=19548;   // qin
   sortTable[0x8D7E]=19549;   // qin
   sortTable[0x9CF9]=19550;   // qin
   sortTable[0x9D6D]=19551;   // qin
   sortTable[0x60C5]=19552;   // qing
   sortTable[0x8F7B]=19553;   // qing
   sortTable[0x8F15]=19554;   // qing
   sortTable[0x9752]=19555;   // qing
   sortTable[0x6E05]=19556;   // qing
   sortTable[0x8BF7]=19557;   // qing
   sortTable[0x8ACB]=19558;   // qing
   sortTable[0x503E]=19559;   // qing
   sortTable[0x50BE]=19560;   // qing
   sortTable[0x6674]=19561;   // qing
   sortTable[0x5E86]=19562;   // qing
   sortTable[0x6176]=19563;   // qing
   sortTable[0x9877]=19564;   // qing
   sortTable[0x9803]=19565;   // qing
   sortTable[0x873B]=19566;   // qing
   sortTable[0x6C22]=19567;   // qing
   sortTable[0x6C2B]=19568;   // qing
   sortTable[0x537F]=19569;   // qing
   sortTable[0x64CE]=19570;   // qing
   sortTable[0x7F44]=19571;   // qing
   sortTable[0x78EC]=19572;   // qing
   sortTable[0x9EE5]=19573;   // qing
   sortTable[0x6C30]=19574;   // qing
   sortTable[0x6AA0]=19575;   // qing
   sortTable[0x7B90]=19576;   // qing
   sortTable[0x9CAD]=19577;   // qing
   sortTable[0x570A]=19578;   // qing
   sortTable[0x82D8]=19579;   // qing
   sortTable[0x8B26]=19580;   // qing
   sortTable[0x7883]=19581;   // qing
   sortTable[0x845D]=19582;   // qing
   sortTable[0x78D8]=19583;   // qing
   sortTable[0x9758]=19584;   // qing
   sortTable[0x51CA]=19585;   // qing
   sortTable[0x5260]=19586;   // qing
   sortTable[0x52CD]=19587;   // qing
   sortTable[0x57E5]=19588;   // qing
   sortTable[0x591D]=19589;   // qing
   sortTable[0x5BC8]=19590;   // qing
   sortTable[0x5EBC]=19591;   // qing
   sortTable[0x5ECE]=19592;   // qing
   sortTable[0x6385]=19593;   // qing
   sortTable[0x64CF]=19594;   // qing
   sortTable[0x6692]=19595;   // qing
   sortTable[0x68FE]=19596;   // qing
   sortTable[0x6A08]=19597;   // qing
   sortTable[0x6ABE]=19598;   // qing
   sortTable[0x6AE6]=19599;   // qing
   sortTable[0x6B91]=19600;   // qing
   sortTable[0x6BB8]=19601;   // qing
   sortTable[0x6DF8]=19602;   // qing
   sortTable[0x6F00]=19603;   // qing
   sortTable[0x72C5]=19604;   // qing
   sortTable[0x7520]=19605;   // qing
   sortTable[0x7858]=19606;   // qing
   sortTable[0x90EC]=19607;   // qing
   sortTable[0x944B]=19608;   // qing
   sortTable[0x9751]=19609;   // qing
   sortTable[0x7A77]=19610;   // qiong
   sortTable[0x7AAE]=19611;   // qiong
   sortTable[0x743C]=19612;   // qiong
   sortTable[0x7A79]=19613;   // qiong
   sortTable[0x86E9]=19614;   // qiong
   sortTable[0x909B]=19615;   // qiong
   sortTable[0x8DEB]=19616;   // qiong
   sortTable[0x828E]=19617;   // qiong
   sortTable[0x928E]=19618;   // qiong
   sortTable[0x7B47]=19619;   // qiong
   sortTable[0x8315]=19620;   // qiong
   sortTable[0x85ED]=19621;   // qiong
   sortTable[0x85D1]=19622;   // qiong
   sortTable[0x7758]=19623;   // qiong
   sortTable[0x86EC]=19624;   // qiong
   sortTable[0x823C]=19625;   // qiong
   sortTable[0x778F]=19626;   // qiong
   sortTable[0x8D79]=19627;   // qiong
   sortTable[0x511D]=19628;   // qiong
   sortTable[0x5314]=19629;   // qiong
   sortTable[0x536D]=19630;   // qiong
   sortTable[0x5B86]=19631;   // qiong
   sortTable[0x60F8]=19632;   // qiong
   sortTable[0x618C]=19633;   // qiong
   sortTable[0x684F]=19634;   // qiong
   sortTable[0x6A69]=19635;   // qiong
   sortTable[0x712A]=19636;   // qiong
   sortTable[0x712D]=19637;   // qiong
   sortTable[0x7162]=19638;   // qiong
   sortTable[0x718D]=19639;   // qiong
   sortTable[0x749A]=19640;   // qiong
   sortTable[0x74CA]=19641;   // qiong
   sortTable[0x74D7]=19642;   // qiong
   sortTable[0x7AC6]=19643;   // qiong
   sortTable[0x7B3B]=19644;   // qiong
   sortTable[0x6C42]=19645;   // qiu
   sortTable[0x7403]=19646;   // qiu
   sortTable[0x79CB]=19647;   // qiu
   sortTable[0x4E18]=19648;   // qiu
   sortTable[0x86AF]=19649;   // qiu
   sortTable[0x56DA]=19650;   // qiu
   sortTable[0x90B1]=19651;   // qiu
   sortTable[0x88D8]=19652;   // qiu
   sortTable[0x914B]=19653;   // qiu
   sortTable[0x866C]=19654;   // qiu
   sortTable[0x4FC5]=19655;   // qiu
   sortTable[0x9CC5]=19656;   // qiu
   sortTable[0x9011]=19657;   // qiu
   sortTable[0x6978]=19658;   // qiu
   sortTable[0x6CC5]=19659;   // qiu
   sortTable[0x9052]=19660;   // qiu
   sortTable[0x8D47]=19661;   // qiu
   sortTable[0x7CD7]=19662;   // qiu
   sortTable[0x5DEF]=19663;   // qiu
   sortTable[0x9194]=19664;   // qiu
   sortTable[0x92B6]=19665;   // qiu
   sortTable[0x9C3D]=19666;   // qiu
   sortTable[0x9E59]=19667;   // qiu
   sortTable[0x72B0]=19668;   // qiu
   sortTable[0x8764]=19669;   // qiu
   sortTable[0x97A6]=19670;   // qiu
   sortTable[0x8A04]=19671;   // qiu
   sortTable[0x7D7F]=19672;   // qiu
   sortTable[0x8DA5]=19673;   // qiu
   sortTable[0x8429]=19674;   // qiu
   sortTable[0x97A7]=19675;   // qiu
   sortTable[0x866F]=19676;   // qiu
   sortTable[0x91DA]=19677;   // qiu
   sortTable[0x79CC]=19678;   // qiu
   sortTable[0x89E9]=19679;   // qiu
   sortTable[0x86F7]=19680;   // qiu
   sortTable[0x9D96]=19681;   // qiu
   sortTable[0x7DE7]=19682;   // qiu
   sortTable[0x8A05]=19683;   // qiu
   sortTable[0x9BC4]=19684;   // qiu
   sortTable[0x9F3D]=19685;   // qiu
   sortTable[0x84F2]=19686;   // qiu
   sortTable[0x9F9D]=19687;   // qiu
   sortTable[0x7A50]=19688;   // qiu
   sortTable[0x808D]=19689;   // qiu
   sortTable[0x89D3]=19690;   // qiu
   sortTable[0x900E]=19691;   // qiu
   sortTable[0x91D3]=19692;   // qiu
   sortTable[0x9B82]=19693;   // qiu
   sortTable[0x9C0C]=19694;   // qiu
   sortTable[0x4E20]=19695;   // qiu
   sortTable[0x53F4]=19696;   // qiu
   sortTable[0x5512]=19697;   // qiu
   sortTable[0x5775]=19698;   // qiu
   sortTable[0x5A9D]=19699;   // qiu
   sortTable[0x5D37]=19700;   // qiu
   sortTable[0x5DF0]=19701;   // qiu
   sortTable[0x6058]=19702;   // qiu
   sortTable[0x624F]=19703;   // qiu
   sortTable[0x641D]=19704;   // qiu
   sortTable[0x6882]=19705;   // qiu
   sortTable[0x6B8F]=19706;   // qiu
   sortTable[0x6BEC]=19707;   // qiu
   sortTable[0x6C53]=19708;   // qiu
   sortTable[0x6D57]=19709;   // qiu
   sortTable[0x6E1E]=19710;   // qiu
   sortTable[0x6E6D]=19711;   // qiu
   sortTable[0x716A]=19712;   // qiu
   sortTable[0x738C]=19713;   // qiu
   sortTable[0x7486]=19714;   // qiu
   sortTable[0x76B3]=19715;   // qiu
   sortTable[0x76DA]=19716;   // qiu
   sortTable[0x7BCD]=19717;   // qiu
   sortTable[0x7D0C]=19718;   // qiu
   sortTable[0x838D]=19719;   // qiu
   sortTable[0x8612]=19720;   // qiu
   sortTable[0x8775]=19721;   // qiu
   sortTable[0x87D7]=19722;   // qiu
   sortTable[0x8824]=19723;   // qiu
   sortTable[0x8CD5]=19724;   // qiu
   sortTable[0x91FB]=19725;   // qiu
   sortTable[0x9C0D]=19726;   // qiu
   sortTable[0x53BB]=19727;   // qu
   sortTable[0x533A]=19728;   // qu
   sortTable[0x5340]=19729;   // qu
   sortTable[0x53D6]=19730;   // qu
   sortTable[0x8DA3]=19731;   // qu
   sortTable[0x66F2]=19732;   // qu
   sortTable[0x6E20]=19733;   // qu
   sortTable[0x5A36]=19734;   // qu
   sortTable[0x5C48]=19735;   // qu
   sortTable[0x8D8B]=19736;   // qu
   sortTable[0x8DA8]=19737;   // qu
   sortTable[0x5C96]=19738;   // qu
   sortTable[0x5D87]=19739;   // qu
   sortTable[0x9A71]=19740;   // qu
   sortTable[0x9A45]=19741;   // qu
   sortTable[0x8EAF]=19742;   // qu
   sortTable[0x86C6]=19743;   // qu
   sortTable[0x8EC0]=19744;   // qu
   sortTable[0x9F8B]=19745;   // qu
   sortTable[0x77BF]=19746;   // qu
   sortTable[0x89D1]=19747;   // qu
   sortTable[0x8862]=19748;   // qu
   sortTable[0x883C]=19749;   // qu
   sortTable[0x8556]=19750;   // qu
   sortTable[0x6710]=19751;   // qu
   sortTable[0x9612]=19752;   // qu
   sortTable[0x8BCE]=19753;   // qu
   sortTable[0x86D0]=19754;   // qu
   sortTable[0x8627]=19755;   // qu
   sortTable[0x52AC]=19756;   // qu
   sortTable[0x795B]=19757;   // qu
   sortTable[0x766F]=19758;   // qu
   sortTable[0x74A9]=19759;   // qu
   sortTable[0x9E32]=19760;   // qu
   sortTable[0x9EB9]=19761;   // qu
   sortTable[0x9EE2]=19762;   // qu
   sortTable[0x9EB4]=19763;   // qu
   sortTable[0x88AA]=19764;   // qu
   sortTable[0x6C0D]=19765;   // qu
   sortTable[0x78F2]=19766;   // qu
   sortTable[0x80E0]=19767;   // qu
   sortTable[0x7D47]=19768;   // qu
   sortTable[0x99C6]=19769;   // qu
   sortTable[0x9F29]=19770;   // qu
   sortTable[0x81DE]=19771;   // qu
   sortTable[0x83C3]=19772;   // qu
   sortTable[0x7C67]=19773;   // qu
   sortTable[0x95B4]=19774;   // qu
   sortTable[0x9E1C]=19775;   // qu
   sortTable[0x877A]=19776;   // qu
   sortTable[0x7CAC]=19777;   // qu
   sortTable[0x8FF2]=19778;   // qu
   sortTable[0x80CA]=19779;   // qu
   sortTable[0x9639]=19780;   // qu
   sortTable[0x9EAF]=19781;   // qu
   sortTable[0x7D36]=19782;   // qu
   sortTable[0x8EE5]=19783;   // qu
   sortTable[0x7FD1]=19784;   // qu
   sortTable[0x844B]=19785;   // qu
   sortTable[0x8AB3]=19786;   // qu
   sortTable[0x9C38]=19787;   // qu
   sortTable[0x9F01]=19788;   // qu
   sortTable[0x957C]=19789;   // qu
   sortTable[0x95C3]=19790;   // qu
   sortTable[0x8837]=19791;   // qu
   sortTable[0x99C8]=19792;   // qu
   sortTable[0x9AF7]=19793;   // qu
   sortTable[0x9F72]=19794;   // qu
   sortTable[0x89BB]=19795;   // qu
   sortTable[0x947A]=19796;   // qu
   sortTable[0x9EAE]=19797;   // qu
   sortTable[0x4F39]=19798;   // qu
   sortTable[0x4F49]=19799;   // qu
   sortTable[0x4F62]=19800;   // qu
   sortTable[0x521E]=19801;   // qu
   sortTable[0x5324]=19802;   // qu
   sortTable[0x53BA]=19803;   // qu
   sortTable[0x547F]=19804;   // qu
   sortTable[0x551F]=19805;   // qu
   sortTable[0x5765]=19806;   // qu
   sortTable[0x5CA8]=19807;   // qu
   sortTable[0x5CB4]=19808;   // qu
   sortTable[0x5FC2]=19809;   // qu
   sortTable[0x6188]=19810;   // qu
   sortTable[0x6235]=19811;   // qu
   sortTable[0x62BE]=19812;   // qu
   sortTable[0x657A]=19813;   // qu
   sortTable[0x65AA]=19814;   // qu
   sortTable[0x6B0B]=19815;   // qu
   sortTable[0x6D40]=19816;   // qu
   sortTable[0x6DED]=19817;   // qu
   sortTable[0x7048]=19818;   // qu
   sortTable[0x7496]=19819;   // qu
   sortTable[0x7AD8]=19820;   // qu
   sortTable[0x7AEC]=19821;   // qu
   sortTable[0x7B41]=19822;   // qu
   sortTable[0x801D]=19823;   // qu
   sortTable[0x87B6]=19824;   // qu
   sortTable[0x87DD]=19825;   // qu
   sortTable[0x8850]=19826;   // qu
   sortTable[0x89B0]=19827;   // qu
   sortTable[0x89B7]=19828;   // qu
   sortTable[0x8A53]=19829;   // qu
   sortTable[0x8A58]=19830;   // qu
   sortTable[0x8EA3]=19831;   // qu
   sortTable[0x9B7C]=19832;   // qu
   sortTable[0x9C4B]=19833;   // qu
   sortTable[0x9D1D]=19834;   // qu
   sortTable[0x5168]=19835;   // quan
   sortTable[0x6743]=19836;   // quan
   sortTable[0x6B0A]=19837;   // quan
   sortTable[0x5708]=19838;   // quan
   sortTable[0x529D]=19839;   // quan
   sortTable[0x52F8]=19840;   // quan
   sortTable[0x62F3]=19841;   // quan
   sortTable[0x6CC9]=19842;   // quan
   sortTable[0x72AC]=19843;   // quan
   sortTable[0x5238]=19844;   // quan
   sortTable[0x8BE0]=19845;   // quan
   sortTable[0x94E8]=19846;   // quan
   sortTable[0x75CA]=19847;   // quan
   sortTable[0x7EFB]=19848;   // quan
   sortTable[0x8343]=19849;   // quan
   sortTable[0x8737]=19850;   // quan
   sortTable[0x609B]=19851;   // quan
   sortTable[0x754E]=19852;   // quan
   sortTable[0x98A7]=19853;   // quan
   sortTable[0x919B]=19854;   // quan
   sortTable[0x7B4C]=19855;   // quan
   sortTable[0x9B08]=19856;   // quan
   sortTable[0x9409]=19857;   // quan
   sortTable[0x8A6E]=19858;   // quan
   sortTable[0x9CC8]=19859;   // quan
   sortTable[0x7E13]=19860;   // quan
   sortTable[0x7D5F]=19861;   // quan
   sortTable[0x9C01]=19862;   // quan
   sortTable[0x8E21]=19863;   // quan
   sortTable[0x8DE7]=19864;   // quan
   sortTable[0x8472]=19865;   // quan
   sortTable[0x8F81]=19866;   // quan
   sortTable[0x8838]=19867;   // quan
   sortTable[0x9293]=19868;   // quan
   sortTable[0x99E9]=19869;   // quan
   sortTable[0x7842]=19870;   // quan
   sortTable[0x8647]=19871;   // quan
   sortTable[0x9874]=19872;   // quan
   sortTable[0x4F7A]=19873;   // quan
   sortTable[0x52E7]=19874;   // quan
   sortTable[0x5573]=19875;   // quan
   sortTable[0x570F]=19876;   // quan
   sortTable[0x57E2]=19877;   // quan
   sortTable[0x59FE]=19878;   // quan
   sortTable[0x5A58]=19879;   // quan
   sortTable[0x5B49]=19880;   // quan
   sortTable[0x5CD1]=19881;   // quan
   sortTable[0x5DCF]=19882;   // quan
   sortTable[0x5F2E]=19883;   // quan
   sortTable[0x606E]=19884;   // quan
   sortTable[0x60D3]=19885;   // quan
   sortTable[0x643C]=19886;   // quan
   sortTable[0x68EC]=19887;   // quan
   sortTable[0x6926]=19888;   // quan
   sortTable[0x697E]=19889;   // quan
   sortTable[0x6A29]=19890;   // quan
   sortTable[0x6C71]=19891;   // quan
   sortTable[0x6D24]=19892;   // quan
   sortTable[0x6E76]=19893;   // quan
   sortTable[0x70C7]=19894;   // quan
   sortTable[0x7276]=19895;   // quan
   sortTable[0x7277]=19896;   // quan
   sortTable[0x7288]=19897;   // quan
   sortTable[0x72AD]=19898;   // quan
   sortTable[0x7454]=19899;   // quan
   sortTable[0x7DA3]=19900;   // quan
   sortTable[0x89E0]=19901;   // quan
   sortTable[0x8F07]=19902;   // quan
   sortTable[0x95CE]=19903;   // quan
   sortTable[0x97CF]=19904;   // quan
   sortTable[0x9A21]=19905;   // quan
   sortTable[0x9F64]=19906;   // quan
   sortTable[0x5374]=19907;   // que
   sortTable[0x537B]=19908;   // que
   sortTable[0x786E]=19909;   // que
   sortTable[0x78BA]=19910;   // que
   sortTable[0x7F3A]=19911;   // que
   sortTable[0x96C0]=19912;   // que
   sortTable[0x9E4A]=19913;   // que
   sortTable[0x9D72]=19914;   // que
   sortTable[0x9619]=19915;   // que
   sortTable[0x69B7]=19916;   // que
   sortTable[0x9615]=19917;   // que
   sortTable[0x7638]=19918;   // que
   sortTable[0x60AB]=19919;   // que
   sortTable[0x78BB]=19920;   // que
   sortTable[0x788F]=19921;   // que
   sortTable[0x7910]=19922;   // que
   sortTable[0x95D5]=19923;   // que
   sortTable[0x785E]=19924;   // que
   sortTable[0x849B]=19925;   // que
   sortTable[0x57C6]=19926;   // que
   sortTable[0x5859]=19927;   // que
   sortTable[0x58A7]=19928;   // que
   sortTable[0x5BC9]=19929;   // que
   sortTable[0x5D05]=19930;   // que
   sortTable[0x6128]=19931;   // que
   sortTable[0x6164]=19932;   // que
   sortTable[0x6409]=19933;   // que
   sortTable[0x704D]=19934;   // que
   sortTable[0x71E9]=19935;   // que
   sortTable[0x7437]=19936;   // que
   sortTable[0x76B5]=19937;   // que
   sortTable[0x792D]=19938;   // que
   sortTable[0x8D9E]=19939;   // que
   sortTable[0x95CB]=19940;   // que
   sortTable[0x7FA4]=19941;   // qun
   sortTable[0x88D9]=19942;   // qun
   sortTable[0x9021]=19943;   // qun
   sortTable[0x56F7]=19944;   // qun
   sortTable[0x590B]=19945;   // qun
   sortTable[0x5BAD]=19946;   // qun
   sortTable[0x5CEE]=19947;   // qun
   sortTable[0x5E2C]=19948;   // qun
   sortTable[0x7FA3]=19949;   // qun
   sortTable[0x88E0]=19950;   // qun
   sortTable[0x7136]=20224;   // ran
   sortTable[0x71C3]=20225;   // ran
   sortTable[0x67D3]=20226;   // ran
   sortTable[0x5189]=20227;   // ran
   sortTable[0x9AEF]=20228;   // ran
   sortTable[0x82D2]=20229;   // ran
   sortTable[0x86BA]=20230;   // ran
   sortTable[0x8887]=20231;   // ran
   sortTable[0x88A1]=20232;   // ran
   sortTable[0x9AE5]=20233;   // ran
   sortTable[0x80B0]=20234;   // ran
   sortTable[0x5184]=20235;   // ran
   sortTable[0x5465]=20236;   // ran
   sortTable[0x562B]=20237;   // ran
   sortTable[0x59CC]=20238;   // ran
   sortTable[0x5AA3]=20239;   // ran
   sortTable[0x6A6A]=20240;   // ran
   sortTable[0x73C3]=20241;   // ran
   sortTable[0x7E4E]=20242;   // ran
   sortTable[0x8485]=20243;   // ran
   sortTable[0x86A6]=20244;   // ran
   sortTable[0x887B]=20245;   // ran
   sortTable[0x8BA9]=20246;   // rang
   sortTable[0x8B93]=20247;   // rang
   sortTable[0x56B7]=20248;   // rang
   sortTable[0x58E4]=20249;   // rang
   sortTable[0x6518]=20250;   // rang
   sortTable[0x7A70]=20251;   // rang
   sortTable[0x79B3]=20252;   // rang
   sortTable[0x74E4]=20253;   // rang
   sortTable[0x8618]=20254;   // rang
   sortTable[0x7E95]=20255;   // rang
   sortTable[0x5134]=20256;   // rang
   sortTable[0x52F7]=20257;   // rang
   sortTable[0x58CC]=20258;   // rang
   sortTable[0x61F9]=20259;   // rang
   sortTable[0x703C]=20260;   // rang
   sortTable[0x7219]=20261;   // rang
   sortTable[0x737D]=20262;   // rang
   sortTable[0x7A63]=20263;   // rang
   sortTable[0x8B72]=20264;   // rang
   sortTable[0x8E9F]=20265;   // rang
   sortTable[0x9B24]=20266;   // rang
   sortTable[0x7ED5]=20267;   // rao
   sortTable[0x7E5E]=20268;   // rao
   sortTable[0x6270]=20269;   // rao
   sortTable[0x64FE]=20270;   // rao
   sortTable[0x9976]=20271;   // rao
   sortTable[0x9952]=20272;   // rao
   sortTable[0x6861]=20273;   // rao
   sortTable[0x835B]=20274;   // rao
   sortTable[0x5A06]=20275;   // rao
   sortTable[0x9076]=20276;   // rao
   sortTable[0x8558]=20277;   // rao
   sortTable[0x8953]=20278;   // rao
   sortTable[0x5B08]=20279;   // rao
   sortTable[0x6A48]=20280;   // rao
   sortTable[0x96A2]=20281;   // rao
   sortTable[0x70ED]=20282;   // re
   sortTable[0x71B1]=20283;   // re
   sortTable[0x60F9]=20284;   // re
   sortTable[0x4EBA]=20285;   // ren
   sortTable[0x8BA4]=20286;   // ren
   sortTable[0x8A8D]=20287;   // ren
   sortTable[0x4EFB]=20288;   // ren
   sortTable[0x5FCD]=20289;   // ren
   sortTable[0x97E7]=20290;   // ren
   sortTable[0x97CC]=20291;   // ren
   sortTable[0x4EC1]=20292;   // ren
   sortTable[0x58EC]=20293;   // ren
   sortTable[0x5203]=20294;   // ren
   sortTable[0x7EAB]=20295;   // ren
   sortTable[0x598A]=20296;   // ren
   sortTable[0x7A14]=20297;   // ren
   sortTable[0x4EDE]=20298;   // ren
   sortTable[0x996A]=20299;   // ren
   sortTable[0x887D]=20300;   // ren
   sortTable[0x834F]=20301;   // ren
   sortTable[0x845A]=20302;   // ren
   sortTable[0x8F6B]=20303;   // ren
   sortTable[0x7EB4]=20304;   // ren
   sortTable[0x8A12]=20305;   // ren
   sortTable[0x8BB1]=20306;   // ren
   sortTable[0x98EA]=20307;   // ren
   sortTable[0x8ED4]=20308;   // ren
   sortTable[0x88B5]=20309;   // ren
   sortTable[0x9B5C]=20310;   // ren
   sortTable[0x928B]=20311;   // ren
   sortTable[0x79F9]=20312;   // ren
   sortTable[0x8EE0]=20313;   // ren
   sortTable[0x9901]=20314;   // ren
   sortTable[0x9D40]=20315;   // ren
   sortTable[0x7D1D]=20316;   // ren
   sortTable[0x9213]=20317;   // ren
   sortTable[0x976D]=20318;   // ren
   sortTable[0x8095]=20319;   // ren
   sortTable[0x82A2]=20320;   // ren
   sortTable[0x7D4D]=20321;   // ren
   sortTable[0x814D]=20322;   // ren
   sortTable[0x8375]=20323;   // ren
   sortTable[0x9771]=20324;   // ren
   sortTable[0x4EBB]=20325;   // ren
   sortTable[0x4EED]=20326;   // ren
   sortTable[0x5204]=20327;   // ren
   sortTable[0x59D9]=20328;   // ren
   sortTable[0x5C7B]=20329;   // ren
   sortTable[0x5C83]=20330;   // ren
   sortTable[0x5FC8]=20331;   // ren
   sortTable[0x5FCE]=20332;   // ren
   sortTable[0x6268]=20333;   // ren
   sortTable[0x6732]=20334;   // ren
   sortTable[0x6752]=20335;   // ren
   sortTable[0x6820]=20336;   // ren
   sortTable[0x6823]=20337;   // ren
   sortTable[0x6895]=20338;   // ren
   sortTable[0x68EF]=20339;   // ren
   sortTable[0x7263]=20340;   // ren
   sortTable[0x794D]=20341;   // ren
   sortTable[0x79C2]=20342;   // ren
   sortTable[0x7D09]=20343;   // ren
   sortTable[0x7D9B]=20344;   // ren
   sortTable[0x8EB5]=20345;   // ren
   sortTable[0x4ECD]=20346;   // reng
   sortTable[0x6254]=20347;   // reng
   sortTable[0x793D]=20348;   // reng
   sortTable[0x82BF]=20349;   // reng
   sortTable[0x8FB8]=20350;   // reng
   sortTable[0x967E]=20351;   // reng
   sortTable[0x65E5]=20352;   // ri
   sortTable[0x9224]=20353;   // ri
   sortTable[0x9A72]=20354;   // ri
   sortTable[0x99B9]=20355;   // ri
   sortTable[0x56F8]=20356;   // ri
   sortTable[0x91F0]=20357;   // ri
   sortTable[0x5BB9]=20358;   // rong
   sortTable[0x8363]=20359;   // rong
   sortTable[0x69AE]=20360;   // rong
   sortTable[0x6EB6]=20361;   // rong
   sortTable[0x7ED2]=20362;   // rong
   sortTable[0x7D68]=20363;   // rong
   sortTable[0x878D]=20364;   // rong
   sortTable[0x7194]=20365;   // rong
   sortTable[0x8338]=20366;   // rong
   sortTable[0x6995]=20367;   // rong
   sortTable[0x620E]=20368;   // rong
   sortTable[0x84C9]=20369;   // rong
   sortTable[0x5197]=20370;   // rong
   sortTable[0x9555]=20371;   // rong
   sortTable[0x5D58]=20372;   // rong
   sortTable[0x809C]=20373;   // rong
   sortTable[0x877E]=20374;   // rong
   sortTable[0x72E8]=20375;   // rong
   sortTable[0x9394]=20376;   // rong
   sortTable[0x8319]=20377;   // rong
   sortTable[0x8EF5]=20378;   // rong
   sortTable[0x9D27]=20379;   // rong
   sortTable[0x7A43]=20380;   // rong
   sortTable[0x7FA2]=20381;   // rong
   sortTable[0x8811]=20382;   // rong
   sortTable[0x7E19]=20383;   // rong
   sortTable[0x8923]=20384;   // rong
   sortTable[0x99E5]=20385;   // rong
   sortTable[0x5087]=20386;   // rong
   sortTable[0x5748]=20387;   // rong
   sortTable[0x5AB6]=20388;   // rong
   sortTable[0x5AC6]=20389;   // rong
   sortTable[0x5B2B]=20390;   // rong
   sortTable[0x5B82]=20391;   // rong
   sortTable[0x5D64]=20392;   // rong
   sortTable[0x5DB8]=20393;   // rong
   sortTable[0x5DC6]=20394;   // rong
   sortTable[0x6408]=20395;   // rong
   sortTable[0x6411]=20396;   // rong
   sortTable[0x66E7]=20397;   // rong
   sortTable[0x6804]=20398;   // rong
   sortTable[0x69B5]=20399;   // rong
   sortTable[0x6BE7]=20400;   // rong
   sortTable[0x6C04]=20401;   // rong
   sortTable[0x701C]=20402;   // rong
   sortTable[0x70FF]=20403;   // rong
   sortTable[0x7203]=20404;   // rong
   sortTable[0x7462]=20405;   // rong
   sortTable[0x7A41]=20406;   // rong
   sortTable[0x878E]=20407;   // rong
   sortTable[0x9AF6]=20408;   // rong
   sortTable[0x8089]=20409;   // rou
   sortTable[0x67D4]=20410;   // rou
   sortTable[0x63C9]=20411;   // rou
   sortTable[0x8E42]=20412;   // rou
   sortTable[0x97A3]=20413;   // rou
   sortTable[0x7CC5]=20414;   // rou
   sortTable[0x8F2E]=20415;   // rou
   sortTable[0x875A]=20416;   // rou
   sortTable[0x8447]=20417;   // rou
   sortTable[0x816C]=20418;   // rou
   sortTable[0x9D94]=20419;   // rou
   sortTable[0x7C88]=20420;   // rou
   sortTable[0x9A25]=20421;   // rou
   sortTable[0x53B9]=20422;   // rou
   sortTable[0x5A83]=20423;   // rou
   sortTable[0x5B8D]=20424;   // rou
   sortTable[0x697A]=20425;   // rou
   sortTable[0x6E18]=20426;   // rou
   sortTable[0x7163]=20427;   // rou
   sortTable[0x7448]=20428;   // rou
   sortTable[0x74C7]=20429;   // rou
   sortTable[0x79B8]=20430;   // rou
   sortTable[0x9352]=20431;   // rou
   sortTable[0x97D6]=20432;   // rou
   sortTable[0x9C07]=20433;   // rou
   sortTable[0x5982]=20434;   // ru
   sortTable[0x5165]=20435;   // ru
   sortTable[0x4E73]=20436;   // ru
   sortTable[0x8FB1]=20437;   // ru
   sortTable[0x8815]=20438;   // ru
   sortTable[0x6C5D]=20439;   // ru
   sortTable[0x5112]=20440;   // ru
   sortTable[0x5B7A]=20441;   // ru
   sortTable[0x8925]=20442;   // ru
   sortTable[0x8339]=20443;   // ru
   sortTable[0x6D33]=20444;   // ru
   sortTable[0x6FE1]=20445;   // ru
   sortTable[0x7F1B]=20446;   // ru
   sortTable[0x5685]=20447;   // ru
   sortTable[0x85B7]=20448;   // ru
   sortTable[0x8966]=20449;   // ru
   sortTable[0x84D0]=20450;   // ru
   sortTable[0x6EBD]=20451;   // ru
   sortTable[0x910F]=20452;   // ru
   sortTable[0x94F7]=20453;   // ru
   sortTable[0x9D3D]=20454;   // ru
   sortTable[0x88BD]=20455;   // ru
   sortTable[0x98A5]=20456;   // ru
   sortTable[0x9CF0]=20457;   // ru
   sortTable[0x986C]=20458;   // ru
   sortTable[0x8FBC]=20459;   // ru
   sortTable[0x8761]=20460;   // ru
   sortTable[0x909A]=20461;   // ru
   sortTable[0x8498]=20462;   // ru
   sortTable[0x91B9]=20463;   // ru
   sortTable[0x8097]=20464;   // ru
   sortTable[0x8560]=20465;   // ru
   sortTable[0x4F9E]=20466;   // ru
   sortTable[0x55D5]=20467;   // ru
   sortTable[0x5AB7]=20468;   // ru
   sortTable[0x5B2C]=20469;   // ru
   sortTable[0x5DBF]=20470;   // ru
   sortTable[0x5E24]=20471;   // ru
   sortTable[0x6256]=20472;   // ru
   sortTable[0x64E9]=20473;   // ru
   sortTable[0x66D8]=20474;   // ru
   sortTable[0x6741]=20475;   // ru
   sortTable[0x6847]=20476;   // ru
   sortTable[0x6E2A]=20477;   // ru
   sortTable[0x71F8]=20478;   // ru
   sortTable[0x7B4E]=20479;   // ru
   sortTable[0x7E1F]=20480;   // ru
   sortTable[0x92A3]=20481;   // ru
   sortTable[0x9C6C]=20482;   // ru
   sortTable[0x9D11]=20483;   // ru
   sortTable[0x633C]=20484;   // rua
   sortTable[0x8F6F]=20485;   // ruan
   sortTable[0x8EDF]=20486;   // ruan
   sortTable[0x962E]=20487;   // ruan
   sortTable[0x670A]=20488;   // ruan
   sortTable[0x800E]=20489;   // ruan
   sortTable[0x7DDB]=20490;   // ruan
   sortTable[0x8F2D]=20491;   // ruan
   sortTable[0x789D]=20492;   // ruan
   sortTable[0x5044]=20493;   // ruan
   sortTable[0x5827]=20494;   // ruan
   sortTable[0x58D6]=20495;   // ruan
   sortTable[0x5A86]=20496;   // ruan
   sortTable[0x648B]=20497;   // ruan
   sortTable[0x744C]=20498;   // ruan
   sortTable[0x74C0]=20499;   // ruan
   sortTable[0x791D]=20500;   // ruan
   sortTable[0x9510]=20501;   // rui
   sortTable[0x92B3]=20502;   // rui
   sortTable[0x745E]=20503;   // rui
   sortTable[0x777F]=20504;   // rui
   sortTable[0x854A]=20505;   // rui
   sortTable[0x82AE]=20506;   // rui
   sortTable[0x8564]=20507;   // rui
   sortTable[0x868B]=20508;   // rui
   sortTable[0x6798]=20509;   // rui
   sortTable[0x7DCC]=20510;   // rui
   sortTable[0x7E60]=20511;   // rui
   sortTable[0x92ED]=20512;   // rui
   sortTable[0x8602]=20513;   // rui
   sortTable[0x854B]=20514;   // rui
   sortTable[0x8739]=20515;   // rui
   sortTable[0x53E1]=20516;   // rui
   sortTable[0x58E1]=20517;   // rui
   sortTable[0x5A51]=20518;   // rui
   sortTable[0x6875]=20519;   // rui
   sortTable[0x6A64]=20520;   // rui
   sortTable[0x6C6D]=20521;   // rui
   sortTable[0x7524]=20522;   // rui
   sortTable[0x8603]=20523;   // rui
   sortTable[0x6DA6]=20524;   // run
   sortTable[0x6F64]=20525;   // run
   sortTable[0x95F0]=20526;   // run
   sortTable[0x81B6]=20527;   // run
   sortTable[0x77A4]=20528;   // run
   sortTable[0x958F]=20529;   // run
   sortTable[0x6A4D]=20530;   // run
   sortTable[0x95A0]=20531;   // run
   sortTable[0x5F31]=20532;   // ruo
   sortTable[0x82E5]=20533;   // ruo
   sortTable[0x504C]=20534;   // ruo
   sortTable[0x7BAC]=20535;   // ruo
   sortTable[0x9100]=20536;   // ruo
   sortTable[0x84BB]=20537;   // ruo
   sortTable[0x9DB8]=20538;   // ruo
   sortTable[0x7BDB]=20539;   // ruo
   sortTable[0x53D2]=20540;   // ruo
   sortTable[0x5D76]=20541;   // ruo
   sortTable[0x637C]=20542;   // ruo
   sortTable[0x6949]=20543;   // ruo
   sortTable[0x6E03]=20544;   // ruo
   sortTable[0x712B]=20545;   // ruo
   sortTable[0x7207]=20546;   // ruo
   sortTable[0x9C19]=20547;   // ruo
   sortTable[0x9C2F]=20548;   // ruo
   sortTable[0x6492]=20992;   // sa
   sortTable[0x6D12]=20993;   // sa
   sortTable[0x8428]=20994;   // sa
   sortTable[0x85A9]=20995;   // sa
   sortTable[0x98D2]=20996;   // sa
   sortTable[0x5345]=20997;   // sa
   sortTable[0x4EE8]=20998;   // sa
   sortTable[0x6332]=20999;   // sa
   sortTable[0x9778]=21000;   // sa
   sortTable[0x9212]=21001;   // sa
   sortTable[0x810E]=21002;   // sa
   sortTable[0x99BA]=21003;   // sa
   sortTable[0x8644]=21004;   // sa
   sortTable[0x8EA0]=21005;   // sa
   sortTable[0x6331]=21006;   // sa
   sortTable[0x644B]=21007;   // sa
   sortTable[0x6AD2]=21008;   // sa
   sortTable[0x6CE7]=21009;   // sa
   sortTable[0x6F75]=21010;   // sa
   sortTable[0x7051]=21011;   // sa
   sortTable[0x8A2F]=21012;   // sa
   sortTable[0x96A1]=21013;   // sa
   sortTable[0x98AF]=21014;   // sa
   sortTable[0x8D5B]=21015;   // sai
   sortTable[0x8CFD]=21016;   // sai
   sortTable[0x585E]=21017;   // sai
   sortTable[0x816E]=21018;   // sai
   sortTable[0x9CC3]=21019;   // sai
   sortTable[0x567B]=21020;   // sai
   sortTable[0x7C3A]=21021;   // sai
   sortTable[0x50FF]=21022;   // sai
   sortTable[0x55EE]=21023;   // sai
   sortTable[0x5625]=21024;   // sai
   sortTable[0x6122]=21025;   // sai
   sortTable[0x63CC]=21026;   // sai
   sortTable[0x6BE2]=21027;   // sai
   sortTable[0x6BF8]=21028;   // sai
   sortTable[0x984B]=21029;   // sai
   sortTable[0x9C13]=21030;   // sai
   sortTable[0x4E09]=21031;   // san
   sortTable[0x6563]=21032;   // san
   sortTable[0x4F1E]=21033;   // san
   sortTable[0x5098]=21034;   // san
   sortTable[0x53C1]=21035;   // san
   sortTable[0x7CC1]=21036;   // san
   sortTable[0x9730]=21037;   // san
   sortTable[0x9993]=21038;   // san
   sortTable[0x7E56]=21039;   // san
   sortTable[0x6BF5]=21040;   // san
   sortTable[0x93FE]=21041;   // san
   sortTable[0x7CE4]=21042;   // san
   sortTable[0x9B16]=21043;   // san
   sortTable[0x7CC2]=21044;   // san
   sortTable[0x4ED0]=21045;   // san
   sortTable[0x4FD5]=21046;   // san
   sortTable[0x5381]=21047;   // san
   sortTable[0x58ED]=21048;   // san
   sortTable[0x5E34]=21049;   // san
   sortTable[0x5F0E]=21050;   // san
   sortTable[0x60B7]=21051;   // san
   sortTable[0x6A75]=21052;   // san
   sortTable[0x6BF6]=21053;   // san
   sortTable[0x6BFF]=21054;   // san
   sortTable[0x7299]=21055;   // san
   sortTable[0x7CDD]=21056;   // san
   sortTable[0x7CE3]=21057;   // san
   sortTable[0x93D2]=21058;   // san
   sortTable[0x9590]=21059;   // san
   sortTable[0x994A]=21060;   // san
   sortTable[0x55D3]=21061;   // sang
   sortTable[0x4E27]=21062;   // sang
   sortTable[0x55AA]=21063;   // sang
   sortTable[0x6851]=21064;   // sang
   sortTable[0x98A1]=21065;   // sang
   sortTable[0x6421]=21066;   // sang
   sortTable[0x78C9]=21067;   // sang
   sortTable[0x9859]=21068;   // sang
   sortTable[0x6852]=21069;   // sang
   sortTable[0x69E1]=21070;   // sang
   sortTable[0x892C]=21071;   // sang
   sortTable[0x939F]=21072;   // sang
   sortTable[0x5AC2]=21073;   // sao
   sortTable[0x626B]=21074;   // sao
   sortTable[0x6383]=21075;   // sao
   sortTable[0x6414]=21076;   // sao
   sortTable[0x9A9A]=21077;   // sao
   sortTable[0x57FD]=21078;   // sao
   sortTable[0x7F2B]=21079;   // sao
   sortTable[0x81CA]=21080;   // sao
   sortTable[0x7619]=21081;   // sao
   sortTable[0x9ADE]=21082;   // sao
   sortTable[0x9CCB]=21083;   // sao
   sortTable[0x77C2]=21084;   // sao
   sortTable[0x9C20]=21085;   // sao
   sortTable[0x7E45]=21086;   // sao
   sortTable[0x87A6]=21087;   // sao
   sortTable[0x9A37]=21088;   // sao
   sortTable[0x6145]=21089;   // sao
   sortTable[0x63BB]=21090;   // sao
   sortTable[0x6C09]=21091;   // sao
   sortTable[0x6E9E]=21092;   // sao
   sortTable[0x9A12]=21093;   // sao
   sortTable[0x9C62]=21094;   // sao
   sortTable[0x8272]=21095;   // se
   sortTable[0x745F]=21096;   // se
   sortTable[0x7A51]=21097;   // se
   sortTable[0x6DA9]=21098;   // se
   sortTable[0x556C]=21099;   // se
   sortTable[0x94EF]=21100;   // se
   sortTable[0x96ED]=21101;   // se
   sortTable[0x8B45]=21102;   // se
   sortTable[0x7A6F]=21103;   // se
   sortTable[0x8F56]=21104;   // se
   sortTable[0x7E6C]=21105;   // se
   sortTable[0x92AB]=21106;   // se
   sortTable[0x98CB]=21107;   // se
   sortTable[0x55C7]=21108;   // se
   sortTable[0x61CE]=21109;   // se
   sortTable[0x64CC]=21110;   // se
   sortTable[0x681C]=21111;   // se
   sortTable[0x6B6E]=21112;   // se
   sortTable[0x6B70]=21113;   // se
   sortTable[0x6D13]=21114;   // se
   sortTable[0x6E0B]=21115;   // se
   sortTable[0x6F80]=21116;   // se
   sortTable[0x6F81]=21117;   // se
   sortTable[0x6FC7]=21118;   // se
   sortTable[0x6FCF]=21119;   // se
   sortTable[0x7012]=21120;   // se
   sortTable[0x7417]=21121;   // se
   sortTable[0x74B1]=21122;   // se
   sortTable[0x7637]=21123;   // se
   sortTable[0x7A61]=21124;   // se
   sortTable[0x93FC]=21125;   // se
   sortTable[0x95AA]=21126;   // se
   sortTable[0x68EE]=21127;   // sen
   sortTable[0x8942]=21128;   // sen
   sortTable[0x692E]=21129;   // sen
   sortTable[0x69EE]=21130;   // sen
   sortTable[0x50E7]=21131;   // seng
   sortTable[0x9B19]=21132;   // seng
   sortTable[0x6C99]=21133;   // sha
   sortTable[0x6740]=21134;   // sha
   sortTable[0x6BBA]=21135;   // sha
   sortTable[0x5565]=21136;   // sha
   sortTable[0x7EB1]=21137;   // sha
   sortTable[0x7D17]=21138;   // sha
   sortTable[0x50BB]=21139;   // sha
   sortTable[0x7802]=21140;   // sha
   sortTable[0x715E]=21141;   // sha
   sortTable[0x5239]=21142;   // sha
   sortTable[0x53A6]=21143;   // sha
   sortTable[0x524E]=21144;   // sha
   sortTable[0x5EC8]=21145;   // sha
   sortTable[0x970E]=21146;   // sha
   sortTable[0x838E]=21147;   // sha
   sortTable[0x94E9]=21148;   // sha
   sortTable[0x88DF]=21149;   // sha
   sortTable[0x9CA8]=21150;   // sha
   sortTable[0x6B43]=21151;   // sha
   sortTable[0x75E7]=21152;   // sha
   sortTable[0x7FE3]=21153;   // sha
   sortTable[0x553C]=21154;   // sha
   sortTable[0x9BCA]=21155;   // sha
   sortTable[0x7FDC]=21156;   // sha
   sortTable[0x9BCB]=21157;   // sha
   sortTable[0x7B91]=21158;   // sha
   sortTable[0x8410]=21159;   // sha
   sortTable[0x7E4C]=21160;   // sha
   sortTable[0x7870]=21161;   // sha
   sortTable[0x9B66]=21162;   // sha
   sortTable[0x7C86]=21163;   // sha
   sortTable[0x95AF]=21164;   // sha
   sortTable[0x8531]=21165;   // sha
   sortTable[0x4E77]=21166;   // sha
   sortTable[0x503D]=21167;   // sha
   sortTable[0x510D]=21168;   // sha
   sortTable[0x5526]=21169;   // sha
   sortTable[0x5551]=21170;   // sha
   sortTable[0x55A2]=21171;   // sha
   sortTable[0x5E39]=21172;   // sha
   sortTable[0x686C]=21173;   // sha
   sortTable[0x699D]=21174;   // sha
   sortTable[0x6A27]=21175;   // sha
   sortTable[0x6BEE]=21176;   // sha
   sortTable[0x7300]=21177;   // sha
   sortTable[0x93A9]=21178;   // sha
   sortTable[0x6652]=21179;   // shai
   sortTable[0x7B5B]=21180;   // shai
   sortTable[0x7BE9]=21181;   // shai
   sortTable[0x917E]=21182;   // shai
   sortTable[0x95B7]=21183;   // shai
   sortTable[0x7C01]=21184;   // shai
   sortTable[0x7C1B]=21185;   // shai
   sortTable[0x7E7A]=21186;   // shai
   sortTable[0x91C3]=21187;   // shai
   sortTable[0x66EC]=21188;   // shai
   sortTable[0x5C71]=21189;   // shan
   sortTable[0x95EA]=21190;   // shan
   sortTable[0x9583]=21191;   // shan
   sortTable[0x5584]=21192;   // shan
   sortTable[0x886B]=21193;   // shan
   sortTable[0x6247]=21194;   // shan
   sortTable[0x6749]=21195;   // shan
   sortTable[0x73CA]=21196;   // shan
   sortTable[0x9655]=21197;   // shan
   sortTable[0x64C5]=21198;   // shan
   sortTable[0x5220]=21199;   // shan
   sortTable[0x81B3]=21200;   // shan
   sortTable[0x8D61]=21201;   // shan
   sortTable[0x7F2E]=21202;   // shan
   sortTable[0x717D]=21203;   // shan
   sortTable[0x8BAA]=21204;   // shan
   sortTable[0x87EE]=21205;   // shan
   sortTable[0x6C55]=21206;   // shan
   sortTable[0x8DDA]=21207;   // shan
   sortTable[0x82EB]=21208;   // shan
   sortTable[0x912F]=21209;   // shan
   sortTable[0x5B17]=21210;   // shan
   sortTable[0x829F]=21211;   // shan
   sortTable[0x59D7]=21212;   // shan
   sortTable[0x5261]=21213;   // shan
   sortTable[0x9490]=21214;   // shan
   sortTable[0x6F78]=21215;   // shan
   sortTable[0x9CDD]=21216;   // shan
   sortTable[0x81BB]=21217;   // shan
   sortTable[0x8222]=21218;   // shan
   sortTable[0x759D]=21219;   // shan
   sortTable[0x994D]=21220;   // shan
   sortTable[0x57CF]=21221;   // shan
   sortTable[0x5F61]=21222;   // shan
   sortTable[0x7752]=21223;   // shan
   sortTable[0x9A9F]=21224;   // shan
   sortTable[0x7E3F]=21225;   // shan
   sortTable[0x7FB6]=21226;   // shan
   sortTable[0x91E4]=21227;   // shan
   sortTable[0x9C53]=21228;   // shan
   sortTable[0x87FA]=21229;   // shan
   sortTable[0x8120]=21230;   // shan
   sortTable[0x8A15]=21231;   // shan
   sortTable[0x7FB4]=21232;   // shan
   sortTable[0x8D78]=21233;   // shan
   sortTable[0x965D]=21234;   // shan
   sortTable[0x9BC5]=21235;   // shan
   sortTable[0x7E55]=21236;   // shan
   sortTable[0x8ED5]=21237;   // shan
   sortTable[0x9425]=21238;   // shan
   sortTable[0x78F0]=21239;   // shan
   sortTable[0x8D0D]=21240;   // shan
   sortTable[0x928F]=21241;   // shan
   sortTable[0x5093]=21242;   // shan
   sortTable[0x50D0]=21243;   // shan
   sortTable[0x522A]=21244;   // shan
   sortTable[0x527C]=21245;   // shan
   sortTable[0x5607]=21246;   // shan
   sortTable[0x5738]=21247;   // shan
   sortTable[0x58A0]=21248;   // shan
   sortTable[0x58A1]=21249;   // shan
   sortTable[0x59CD]=21250;   // shan
   sortTable[0x5E53]=21251;   // shan
   sortTable[0x633B]=21252;   // shan
   sortTable[0x639E]=21253;   // shan
   sortTable[0x6427]=21254;   // shan
   sortTable[0x657E]=21255;   // shan
   sortTable[0x6671]=21256;   // shan
   sortTable[0x6763]=21257;   // shan
   sortTable[0x67F5]=21258;   // shan
   sortTable[0x6A3F]=21259;   // shan
   sortTable[0x6A86]=21260;   // shan
   sortTable[0x6B5A]=21261;   // shan
   sortTable[0x6F6C]=21262;   // shan
   sortTable[0x6F98]=21263;   // shan
   sortTable[0x7057]=21264;   // shan
   sortTable[0x7154]=21265;   // shan
   sortTable[0x718C]=21266;   // shan
   sortTable[0x72E6]=21267;   // shan
   sortTable[0x75C1]=21268;   // shan
   sortTable[0x7B18]=21269;   // shan
   sortTable[0x89A2]=21270;   // shan
   sortTable[0x8B06]=21271;   // shan
   sortTable[0x8B71]=21272;   // shan
   sortTable[0x9096]=21273;   // shan
   sortTable[0x958A]=21274;   // shan
   sortTable[0x9A38]=21275;   // shan
   sortTable[0x9C54]=21276;   // shan
   sortTable[0x9FC3]=21277;   // shan
   sortTable[0x4E0A]=21278;   // shang
   sortTable[0x5546]=21279;   // shang
   sortTable[0x4F24]=21280;   // shang
   sortTable[0x50B7]=21281;   // shang
   sortTable[0x5C1A]=21282;   // shang
   sortTable[0x8D4F]=21283;   // shang
   sortTable[0x8CDE]=21284;   // shang
   sortTable[0x88F3]=21285;   // shang
   sortTable[0x664C]=21286;   // shang
   sortTable[0x89DE]=21287;   // shang
   sortTable[0x5892]=21288;   // shang
   sortTable[0x7EF1]=21289;   // shang
   sortTable[0x6B87]=21290;   // shang
   sortTable[0x71B5]=21291;   // shang
   sortTable[0x57A7]=21292;   // shang
   sortTable[0x9B3A]=21293;   // shang
   sortTable[0x8B2A]=21294;   // shang
   sortTable[0x850F]=21295;   // shang
   sortTable[0x979D]=21296;   // shang
   sortTable[0x87AA]=21297;   // shang
   sortTable[0x4E04]=21298;   // shang
   sortTable[0x4EE9]=21299;   // shang
   sortTable[0x5C19]=21300;   // shang
   sortTable[0x6066]=21301;   // shang
   sortTable[0x616F]=21302;   // shang
   sortTable[0x6244]=21303;   // shang
   sortTable[0x6BA4]=21304;   // shang
   sortTable[0x6EF3]=21305;   // shang
   sortTable[0x6F21]=21306;   // shang
   sortTable[0x7DD4]=21307;   // shang
   sortTable[0x89F4]=21308;   // shang
   sortTable[0x8D18]=21309;   // shang
   sortTable[0x945C]=21310;   // shang
   sortTable[0x5C11]=21311;   // shao
   sortTable[0x70E7]=21312;   // shao
   sortTable[0x71D2]=21313;   // shao
   sortTable[0x7ECD]=21314;   // shao
   sortTable[0x7D39]=21315;   // shao
   sortTable[0x7A0D]=21316;   // shao
   sortTable[0x54E8]=21317;   // shao
   sortTable[0x68A2]=21318;   // shao
   sortTable[0x634E]=21319;   // shao
   sortTable[0x90B5]=21320;   // shao
   sortTable[0x97F6]=21321;   // shao
   sortTable[0x828D]=21322;   // shao
   sortTable[0x52FA]=21323;   // shao
   sortTable[0x52AD]=21324;   // shao
   sortTable[0x8244]=21325;   // shao
   sortTable[0x82D5]=21326;   // shao
   sortTable[0x7B72]=21327;   // shao
   sortTable[0x86F8]=21328;   // shao
   sortTable[0x8891]=21329;   // shao
   sortTable[0x6F72]=21330;   // shao
   sortTable[0x7744]=21331;   // shao
   sortTable[0x83A6]=21332;   // shao
   sortTable[0x9AFE]=21333;   // shao
   sortTable[0x9BB9]=21334;   // shao
   sortTable[0x98B5]=21335;   // shao
   sortTable[0x5372]=21336;   // shao
   sortTable[0x5A0B]=21337;   // shao
   sortTable[0x5F30]=21338;   // shao
   sortTable[0x65D3]=21339;   // shao
   sortTable[0x67D6]=21340;   // shao
   sortTable[0x713C]=21341;   // shao
   sortTable[0x73BF]=21342;   // shao
   sortTable[0x7AF0]=21343;   // shao
   sortTable[0x7DA4]=21344;   // shao
   sortTable[0x8571]=21345;   // shao
   sortTable[0x8F0E]=21346;   // shao
   sortTable[0x793E]=21347;   // she
   sortTable[0x8BBE]=21348;   // she
   sortTable[0x8A2D]=21349;   // she
   sortTable[0x5C04]=21350;   // she
   sortTable[0x6444]=21351;   // she
   sortTable[0x651D]=21352;   // she
   sortTable[0x86C7]=21353;   // she
   sortTable[0x820D]=21354;   // she
   sortTable[0x820C]=21355;   // she
   sortTable[0x6D89]=21356;   // she
   sortTable[0x8D66]=21357;   // she
   sortTable[0x5962]=21358;   // she
   sortTable[0x6151]=21359;   // she
   sortTable[0x9E9D]=21360;   // she
   sortTable[0x6B59]=21361;   // she
   sortTable[0x8D4A]=21362;   // she
   sortTable[0x7572]=21363;   // she
   sortTable[0x538D]=21364;   // she
   sortTable[0x4F58]=21365;   // she
   sortTable[0x6EE0]=21366;   // she
   sortTable[0x731E]=21367;   // she
   sortTable[0x8675]=21368;   // she
   sortTable[0x850E]=21369;   // she
   sortTable[0x97D8]=21370;   // she
   sortTable[0x8F0B]=21371;   // she
   sortTable[0x86E5]=21372;   // she
   sortTable[0x8802]=21373;   // she
   sortTable[0x9A07]=21374;   // she
   sortTable[0x820E]=21375;   // she
   sortTable[0x5399]=21376;   // she
   sortTable[0x5F3D]=21377;   // she
   sortTable[0x6174]=21378;   // she
   sortTable[0x61FE]=21379;   // she
   sortTable[0x6368]=21380;   // she
   sortTable[0x6442]=21381;   // she
   sortTable[0x6475]=21382;   // she
   sortTable[0x6AA8]=21383;   // she
   sortTable[0x6B07]=21384;   // she
   sortTable[0x6DBB]=21385;   // she
   sortTable[0x6E09]=21386;   // she
   sortTable[0x7044]=21387;   // she
   sortTable[0x756C]=21388;   // she
   sortTable[0x8CD2]=21389;   // she
   sortTable[0x8CD6]=21390;   // she
   sortTable[0x4EC0]=21391;   // shen
   sortTable[0x751A]=21392;   // shen
   sortTable[0x8EAB]=21393;   // shen
   sortTable[0x795E]=21394;   // shen
   sortTable[0x6DF1]=21395;   // shen
   sortTable[0x4F38]=21396;   // shen
   sortTable[0x5A76]=21397;   // shen
   sortTable[0x5B38]=21398;   // shen
   sortTable[0x5BA1]=21399;   // shen
   sortTable[0x5BE9]=21400;   // shen
   sortTable[0x614E]=21401;   // shen
   sortTable[0x6E17]=21402;   // shen
   sortTable[0x6EF2]=21403;   // shen
   sortTable[0x7533]=21404;   // shen
   sortTable[0x7EC5]=21405;   // shen
   sortTable[0x7D33]=21406;   // shen
   sortTable[0x547B]=21407;   // shen
   sortTable[0x6C88]=21408;   // shen
   sortTable[0x80BE]=21409;   // shen
   sortTable[0x8BDC]=21410;   // shen
   sortTable[0x8398]=21411;   // shen
   sortTable[0x8703]=21412;   // shen
   sortTable[0x5A20]=21413;   // shen
   sortTable[0x54C2]=21414;   // shen
   sortTable[0x6E16]=21415;   // shen
   sortTable[0x80C2]=21416;   // shen
   sortTable[0x7837]=21417;   // shen
   sortTable[0x8C02]=21418;   // shen
   sortTable[0x77E7]=21419;   // shen
   sortTable[0x814E]=21420;   // shen
   sortTable[0x6939]=21421;   // shen
   sortTable[0x8124]=21422;   // shen
   sortTable[0x8460]=21423;   // shen
   sortTable[0x77AB]=21424;   // shen
   sortTable[0x8A75]=21425;   // shen
   sortTable[0x9B6B]=21426;   // shen
   sortTable[0x8C09]=21427;   // shen
   sortTable[0x90A5]=21428;   // shen
   sortTable[0x99EA]=21429;   // shen
   sortTable[0x8704]=21430;   // shen
   sortTable[0x9CB9]=21431;   // shen
   sortTable[0x9C30]=21432;   // shen
   sortTable[0x84E1]=21433;   // shen
   sortTable[0x8518]=21434;   // shen
   sortTable[0x8AD7]=21435;   // shen
   sortTable[0x92E0]=21436;   // shen
   sortTable[0x77E4]=21437;   // shen
   sortTable[0x7973]=21438;   // shen
   sortTable[0x7A7C]=21439;   // shen
   sortTable[0x9BF5]=21440;   // shen
   sortTable[0x7712]=21441;   // shen
   sortTable[0x7718]=21442;   // shen
   sortTable[0x7C76]=21443;   // shen
   sortTable[0x88D1]=21444;   // shen
   sortTable[0x9823]=21445;   // shen
   sortTable[0x9C3A]=21446;   // shen
   sortTable[0x4F81]=21447;   // shen
   sortTable[0x4FBA]=21448;   // shen
   sortTable[0x515F]=21449;   // shen
   sortTable[0x5814]=21450;   // shen
   sortTable[0x59BD]=21451;   // shen
   sortTable[0x59FA]=21452;   // shen
   sortTable[0x5BB7]=21453;   // shen
   sortTable[0x5C7E]=21454;   // shen
   sortTable[0x5CF7]=21455;   // shen
   sortTable[0x5F1E]=21456;   // shen
   sortTable[0x613C]=21457;   // shen
   sortTable[0x625F]=21458;   // shen
   sortTable[0x6552]=21459;   // shen
   sortTable[0x661A]=21460;   // shen
   sortTable[0x66CB]=21461;   // shen
   sortTable[0x66D1]=21462;   // shen
   sortTable[0x67DB]=21463;   // shen
   sortTable[0x68FD]=21464;   // shen
   sortTable[0x698A]=21465;   // shen
   sortTable[0x6C20]=21466;   // shen
   sortTable[0x6D81]=21467;   // shen
   sortTable[0x700B]=21468;   // shen
   sortTable[0x71CA]=21469;   // shen
   sortTable[0x73C5]=21470;   // shen
   sortTable[0x7521]=21471;   // shen
   sortTable[0x7527]=21472;   // shen
   sortTable[0x7606]=21473;   // shen
   sortTable[0x762E]=21474;   // shen
   sortTable[0x7C78]=21475;   // shen
   sortTable[0x7F67]=21476;   // shen
   sortTable[0x8593]=21477;   // shen
   sortTable[0x89BE]=21478;   // shen
   sortTable[0x8A20]=21479;   // shen
   sortTable[0x8A37]=21480;   // shen
   sortTable[0x8B85]=21481;   // shen
   sortTable[0x9825]=21482;   // shen
   sortTable[0x9D62]=21483;   // shen
   sortTable[0x751F]=21484;   // sheng
   sortTable[0x58F0]=21485;   // sheng
   sortTable[0x8072]=21486;   // sheng
   sortTable[0x80DC]=21487;   // sheng
   sortTable[0x52DD]=21488;   // sheng
   sortTable[0x7701]=21489;   // sheng
   sortTable[0x5269]=21490;   // sheng
   sortTable[0x5347]=21491;   // sheng
   sortTable[0x7272]=21492;   // sheng
   sortTable[0x7EF3]=21493;   // sheng
   sortTable[0x7E69]=21494;   // sheng
   sortTable[0x76DB]=21495;   // sheng
   sortTable[0x5723]=21496;   // sheng
   sortTable[0x8056]=21497;   // sheng
   sortTable[0x7525]=21498;   // sheng
   sortTable[0x7B19]=21499;   // sheng
   sortTable[0x5D4A]=21500;   // sheng
   sortTable[0x771A]=21501;   // sheng
   sortTable[0x8CF8]=21502;   // sheng
   sortTable[0x82FC]=21503;   // sheng
   sortTable[0x965E]=21504;   // sheng
   sortTable[0x9F2A]=21505;   // sheng
   sortTable[0x8542]=21506;   // sheng
   sortTable[0x8B5D]=21507;   // sheng
   sortTable[0x8CB9]=21508;   // sheng
   sortTable[0x5057]=21509;   // sheng
   sortTable[0x5270]=21510;   // sheng
   sortTable[0x544F]=21511;   // sheng
   sortTable[0x58AD]=21512;   // sheng
   sortTable[0x61B4]=21513;   // sheng
   sortTable[0x6598]=21514;   // sheng
   sortTable[0x6607]=21515;   // sheng
   sortTable[0x6660]=21516;   // sheng
   sortTable[0x66FB]=21517;   // sheng
   sortTable[0x680D]=21518;   // sheng
   sortTable[0x69BA]=21519;   // sheng
   sortTable[0x6A73]=21520;   // sheng
   sortTable[0x6B85]=21521;   // sheng
   sortTable[0x6CE9]=21522;   // sheng
   sortTable[0x6E3B]=21523;   // sheng
   sortTable[0x6E66]=21524;   // sheng
   sortTable[0x713A]=21525;   // sheng
   sortTable[0x72CC]=21526;   // sheng
   sortTable[0x73C4]=21527;   // sheng
   sortTable[0x741E]=21528;   // sheng
   sortTable[0x7AD4]=21529;   // sheng
   sortTable[0x924E]=21530;   // sheng
   sortTable[0x9629]=21531;   // sheng
   sortTable[0x9679]=21532;   // sheng
   sortTable[0x9D7F]=21533;   // sheng
   sortTable[0x662F]=21534;   // shi
   sortTable[0x65F6]=21535;   // shi
   sortTable[0x6642]=21536;   // shi
   sortTable[0x5341]=21537;   // shi
   sortTable[0x4E8B]=21538;   // shi
   sortTable[0x5B9E]=21539;   // shi
   sortTable[0x5BE6]=21540;   // shi
   sortTable[0x4F7F]=21541;   // shi
   sortTable[0x4E16]=21542;   // shi
   sortTable[0x5E08]=21543;   // shi
   sortTable[0x5E2B]=21544;   // shi
   sortTable[0x77F3]=21545;   // shi
   sortTable[0x8BC6]=21546;   // shi
   sortTable[0x8B58]=21547;   // shi
   sortTable[0x53F2]=21548;   // shi
   sortTable[0x5E02]=21549;   // shi
   sortTable[0x59CB]=21550;   // shi
   sortTable[0x4F3C]=21551;   // shi
   sortTable[0x5F0F]=21552;   // shi
   sortTable[0x5931]=21553;   // shi
   sortTable[0x58EB]=21554;   // shi
   sortTable[0x793A]=21555;   // shi
   sortTable[0x52BF]=21556;   // shi
   sortTable[0x52E2]=21557;   // shi
   sortTable[0x5BA4]=21558;   // shi
   sortTable[0x89C6]=21559;   // shi
   sortTable[0x8996]=21560;   // shi
   sortTable[0x8BD5]=21561;   // shi
   sortTable[0x8A66]=21562;   // shi
   sortTable[0x9002]=21563;   // shi
   sortTable[0x9069]=21564;   // shi
   sortTable[0x98DF]=21565;   // shi
   sortTable[0x65BD]=21566;   // shi
   sortTable[0x6E7F]=21567;   // shi
   sortTable[0x6FD5]=21568;   // shi
   sortTable[0x8BD7]=21569;   // shi
   sortTable[0x8A69]=21570;   // shi
   sortTable[0x91CA]=21571;   // shi
   sortTable[0x9A76]=21572;   // shi
   sortTable[0x91CB]=21573;   // shi
   sortTable[0x99DB]=21574;   // shi
   sortTable[0x8680]=21575;   // shi
   sortTable[0x8755]=21576;   // shi
   sortTable[0x62FE]=21577;   // shi
   sortTable[0x6C0F]=21578;   // shi
   sortTable[0x9970]=21579;   // shi
   sortTable[0x98FE]=21580;   // shi
   sortTable[0x901D]=21581;   // shi
   sortTable[0x8A93]=21582;   // shi
   sortTable[0x72EE]=21583;   // shi
   sortTable[0x7345]=21584;   // shi
   sortTable[0x4F8D]=21585;   // shi
   sortTable[0x5C38]=21586;   // shi
   sortTable[0x5C4E]=21587;   // shi
   sortTable[0x5319]=21588;   // shi
   sortTable[0x62ED]=21589;   // shi
   sortTable[0x8C25]=21590;   // shi
   sortTable[0x4ED5]=21591;   // shi
   sortTable[0x77E2]=21592;   // shi
   sortTable[0x6043]=21593;   // shi
   sortTable[0x55DC]=21594;   // shi
   sortTable[0x8F7C]=21595;   // shi
   sortTable[0x566C]=21596;   // shi
   sortTable[0x5F11]=21597;   // shi
   sortTable[0x8C55]=21598;   // shi
   sortTable[0x67FF]=21599;   // shi
   sortTable[0x7B6E]=21600;   // shi
   sortTable[0x8210]=21601;   // shi
   sortTable[0x8671]=21602;   // shi
   sortTable[0x794F]=21603;   // shi
   sortTable[0x8AE1]=21604;   // shi
   sortTable[0x87AB]=21605;   // shi
   sortTable[0x84CD]=21606;   // shi
   sortTable[0x70BB]=21607;   // shi
   sortTable[0x8D33]=21608;   // shi
   sortTable[0x7D41]=21609;   // shi
   sortTable[0x83B3]=21610;   // shi
   sortTable[0x8C49]=21611;   // shi
   sortTable[0x9CA5]=21612;   // shi
   sortTable[0x94C8]=21613;   // shi
   sortTable[0x8937]=21614;   // shi
   sortTable[0x57D8]=21615;   // shi
   sortTable[0x7757]=21616;   // shi
   sortTable[0x8492]=21617;   // shi
   sortTable[0x8ADF]=21618;   // shi
   sortTable[0x9E24]=21619;   // shi
   sortTable[0x90BF]=21620;   // shi
   sortTable[0x9F2B]=21621;   // shi
   sortTable[0x770E]=21622;   // shi
   sortTable[0x7721]=21623;   // shi
   sortTable[0x7C2D]=21624;   // shi
   sortTable[0x8FBB]=21625;   // shi
   sortTable[0x8479]=21626;   // shi
   sortTable[0x9048]=21627;   // shi
   sortTable[0x9CF2]=21628;   // shi
   sortTable[0x7702]=21629;   // shi
   sortTable[0x7C42]=21630;   // shi
   sortTable[0x9CBA]=21631;   // shi
   sortTable[0x9243]=21632;   // shi
   sortTable[0x9C24]=21633;   // shi
   sortTable[0x9366]=21634;   // shi
   sortTable[0x896B]=21635;   // shi
   sortTable[0x9242]=21636;   // shi
   sortTable[0x9250]=21637;   // shi
   sortTable[0x927D]=21638;   // shi
   sortTable[0x9C23]=21639;   // shi
   sortTable[0x9CFE]=21640;   // shi
   sortTable[0x9230]=21641;   // shi
   sortTable[0x9DB3]=21642;   // shi
   sortTable[0x8213]=21643;   // shi
   sortTable[0x8768]=21644;   // shi
   sortTable[0x8B1A]=21645;   // shi
   sortTable[0x8CB0]=21646;   // shi
   sortTable[0x907E]=21647;   // shi
   sortTable[0x921F]=21648;   // shi
   sortTable[0x991D]=21649;   // shi
   sortTable[0x9F2D]=21650;   // shi
   sortTable[0x4E17]=21651;   // shi
   sortTable[0x4E68]=21652;   // shi
   sortTable[0x4E6D]=21653;   // shi
   sortTable[0x4E8A]=21654;   // shi
   sortTable[0x4F66]=21655;   // shi
   sortTable[0x5158]=21656;   // shi
   sortTable[0x519F]=21657;   // shi
   sortTable[0x534B]=21658;   // shi
   sortTable[0x53D3]=21659;   // shi
   sortTable[0x545E]=21660;   // shi
   sortTable[0x5469]=21661;   // shi
   sortTable[0x5852]=21662;   // shi
   sortTable[0x596D]=21663;   // shi
   sortTable[0x59FC]=21664;   // shi
   sortTable[0x5A9E]=21665;   // shi
   sortTable[0x5B15]=21666;   // shi
   sortTable[0x5B9F]=21667;   // shi
   sortTable[0x5BA9]=21668;   // shi
   sortTable[0x5BD4]=21669;   // shi
   sortTable[0x5C4D]=21670;   // shi
   sortTable[0x5CD5]=21671;   // shi
   sortTable[0x5D3C]=21672;   // shi
   sortTable[0x5D75]=21673;   // shi
   sortTable[0x5F12]=21674;   // shi
   sortTable[0x5FA5]=21675;   // shi
   sortTable[0x5FD5]=21676;   // shi
   sortTable[0x6040]=21677;   // shi
   sortTable[0x623A]=21678;   // shi
   sortTable[0x63D3]=21679;   // shi
   sortTable[0x65F9]=21680;   // shi
   sortTable[0x6630]=21681;   // shi
   sortTable[0x67BE]=21682;   // shi
   sortTable[0x67F9]=21683;   // shi
   sortTable[0x683B]=21684;   // shi
   sortTable[0x6981]=21685;   // shi
   sortTable[0x69AF]=21686;   // shi
   sortTable[0x6D49]=21687;   // shi
   sortTable[0x6E5C]=21688;   // shi
   sortTable[0x6E64]=21689;   // shi
   sortTable[0x6EA1]=21690;   // shi
   sortTable[0x6EAE]=21691;   // shi
   sortTable[0x6EBC]=21692;   // shi
   sortTable[0x6FA8]=21693;   // shi
   sortTable[0x70D2]=21694;   // shi
   sortTable[0x7176]=21695;   // shi
   sortTable[0x7461]=21696;   // shi
   sortTable[0x793B]=21697;   // shi
   sortTable[0x7ACD]=21698;   // shi
   sortTable[0x7B36]=21699;   // shi
   sortTable[0x7BD2]=21700;   // shi
   sortTable[0x8494]=21701;   // shi
   sortTable[0x8979]=21702;   // shi
   sortTable[0x89E2]=21703;   // shi
   sortTable[0x8EFE]=21704;   // shi
   sortTable[0x91C8]=21705;   // shi
   sortTable[0x91F6]=21706;   // shi
   sortTable[0x9247]=21707;   // shi
   sortTable[0x9248]=21708;   // shi
   sortTable[0x92B4]=21709;   // shi
   sortTable[0x98E0]=21710;   // shi
   sortTable[0x9919]=21711;   // shi
   sortTable[0x9963]=21712;   // shi
   sortTable[0x9B96]=21713;   // shi
   sortTable[0x9BF4]=21714;   // shi
   sortTable[0x9C18]=21715;   // shi
   sortTable[0x5159]=21716;   // shig
   sortTable[0x624B]=21717;   // shou
   sortTable[0x53D7]=21718;   // shou
   sortTable[0x6536]=21719;   // shou
   sortTable[0x9996]=21720;   // shou
   sortTable[0x5B88]=21721;   // shou
   sortTable[0x6388]=21722;   // shou
   sortTable[0x7626]=21723;   // shou
   sortTable[0x552E]=21724;   // shou
   sortTable[0x517D]=21725;   // shou
   sortTable[0x7378]=21726;   // shou
   sortTable[0x5BFF]=21727;   // shou
   sortTable[0x58FD]=21728;   // shou
   sortTable[0x7EF6]=21729;   // shou
   sortTable[0x72E9]=21730;   // shou
   sortTable[0x824F]=21731;   // shou
   sortTable[0x93C9]=21732;   // shou
   sortTable[0x53CE]=21733;   // shou
   sortTable[0x57A8]=21734;   // shou
   sortTable[0x5900]=21735;   // shou
   sortTable[0x624C]=21736;   // shou
   sortTable[0x6DAD]=21737;   // shou
   sortTable[0x7363]=21738;   // shou
   sortTable[0x75E9]=21739;   // shou
   sortTable[0x7DAC]=21740;   // shou
   sortTable[0x4E66]=21741;   // shu
   sortTable[0x66F8]=21742;   // shu
   sortTable[0x6570]=21743;   // shu
   sortTable[0x6578]=21744;   // shu
   sortTable[0x672F]=21745;   // shu
   sortTable[0x672E]=21746;   // shu
   sortTable[0x6811]=21747;   // shu
   sortTable[0x6A39]=21748;   // shu
   sortTable[0x719F]=21749;   // shu
   sortTable[0x5C5E]=21750;   // shu
   sortTable[0x5C6C]=21751;   // shu
   sortTable[0x8F93]=21752;   // shu
   sortTable[0x675F]=21753;   // shu
   sortTable[0x8F38]=21754;   // shu
   sortTable[0x53D4]=21755;   // shu
   sortTable[0x8212]=21756;   // shu
   sortTable[0x8FF0]=21757;   // shu
   sortTable[0x6B8A]=21758;   // shu
   sortTable[0x9F20]=21759;   // shu
   sortTable[0x852C]=21760;   // shu
   sortTable[0x7AD6]=21761;   // shu
   sortTable[0x68B3]=21762;   // shu
   sortTable[0x8C4E]=21763;   // shu
   sortTable[0x7F72]=21764;   // shu
   sortTable[0x758F]=21765;   // shu
   sortTable[0x67A2]=21766;   // shu
   sortTable[0x6A1E]=21767;   // shu
   sortTable[0x6691]=21768;   // shu
   sortTable[0x85AF]=21769;   // shu
   sortTable[0x5EB6]=21770;   // shu
   sortTable[0x8700]=21771;   // shu
   sortTable[0x6055]=21772;   // shu
   sortTable[0x620D]=21773;   // shu
   sortTable[0x6DD1]=21774;   // shu
   sortTable[0x8D4E]=21775;   // shu
   sortTable[0x5B70]=21776;   // shu
   sortTable[0x7EBE]=21777;   // shu
   sortTable[0x500F]=21778;   // shu
   sortTable[0x5885]=21779;   // shu
   sortTable[0x6292]=21780;   // shu
   sortTable[0x6F31]=21781;   // shu
   sortTable[0x66D9]=21782;   // shu
   sortTable[0x587E]=21783;   // shu
   sortTable[0x9ECD]=21784;   // shu
   sortTable[0x83FD]=21785;   // shu
   sortTable[0x6F8D]=21786;   // shu
   sortTable[0x59DD]=21787;   // shu
   sortTable[0x6BB3]=21788;   // shu
   sortTable[0x6445]=21789;   // shu
   sortTable[0x79EB]=21790;   // shu
   sortTable[0x6CAD]=21791;   // shu
   sortTable[0x8853]=21792;   // shu
   sortTable[0x8167]=21793;   // shu
   sortTable[0x9103]=21794;   // shu
   sortTable[0x9683]=21795;   // shu
   sortTable[0x6BF9]=21796;   // shu
   sortTable[0x7D80]=21797;   // shu
   sortTable[0x88CB]=21798;   // shu
   sortTable[0x8481]=21799;   // shu
   sortTable[0x9265]=21800;   // shu
   sortTable[0x85F7]=21801;   // shu
   sortTable[0x9D90]=21802;   // shu
   sortTable[0x9483]=21803;   // shu
   sortTable[0x8961]=21804;   // shu
   sortTable[0x7D13]=21805;   // shu
   sortTable[0x93E3]=21806;   // shu
   sortTable[0x85A5]=21807;   // shu
   sortTable[0x8DFE]=21808;   // shu
   sortTable[0x9D68]=21809;   // shu
   sortTable[0x9F21]=21810;   // shu
   sortTable[0x4FB8]=21811;   // shu
   sortTable[0x5010]=21812;   // shu
   sortTable[0x5135]=21813;   // shu
   sortTable[0x51C1]=21814;   // shu
   sortTable[0x54B0]=21815;   // shu
   sortTable[0x5A4C]=21816;   // shu
   sortTable[0x5C0C]=21817;   // shu
   sortTable[0x5C17]=21818;   // shu
   sortTable[0x5EBB]=21819;   // shu
   sortTable[0x6037]=21820;   // shu
   sortTable[0x6352]=21821;   // shu
   sortTable[0x6393]=21822;   // shu
   sortTable[0x6504]=21823;   // shu
   sortTable[0x668F]=21824;   // shu
   sortTable[0x6778]=21825;   // shu
   sortTable[0x6A7E]=21826;   // shu
   sortTable[0x6F44]=21827;   // shu
   sortTable[0x6F7B]=21828;   // shu
   sortTable[0x6FD6]=21829;   // shu
   sortTable[0x702D]=21830;   // shu
   sortTable[0x7102]=21831;   // shu
   sortTable[0x7479]=21832;   // shu
   sortTable[0x74B9]=21833;   // shu
   sortTable[0x758E]=21834;   // shu
   sortTable[0x7659]=21835;   // shu
   sortTable[0x7AEA]=21836;   // shu
   sortTable[0x7CEC]=21837;   // shu
   sortTable[0x7D49]=21838;   // shu
   sortTable[0x8357]=21839;   // shu
   sortTable[0x866A]=21840;   // shu
   sortTable[0x8834]=21841;   // shu
   sortTable[0x8969]=21842;   // shu
   sortTable[0x8D16]=21843;   // shu
   sortTable[0x8E08]=21844;   // shu
   sortTable[0x8ED7]=21845;   // shu
   sortTable[0x9330]=21846;   // shu
   sortTable[0x964E]=21847;   // shu
   sortTable[0x9B9B]=21848;   // shu
   sortTable[0x9C6A]=21849;   // shu
   sortTable[0x9C70]=21850;   // shu
   sortTable[0x5237]=21851;   // shua
   sortTable[0x5530]=21852;   // shua
   sortTable[0x800D]=21853;   // shua
   sortTable[0x8A9C]=21854;   // shua
   sortTable[0x6454]=21855;   // shuai
   sortTable[0x7529]=21856;   // shuai
   sortTable[0x87C0]=21857;   // shuai
   sortTable[0x8870]=21858;   // shuai
   sortTable[0x5E05]=21859;   // shuai
   sortTable[0x5E25]=21860;   // shuai
   sortTable[0x535B]=21861;   // shuai
   sortTable[0x62F4]=21862;   // shuan
   sortTable[0x6813]=21863;   // shuan
   sortTable[0x95E9]=21864;   // shuan
   sortTable[0x6DAE]=21865;   // shuan
   sortTable[0x8168]=21866;   // shuan
   sortTable[0x9582]=21867;   // shuan
   sortTable[0x53CC]=21868;   // shuang
   sortTable[0x96D9]=21869;   // shuang
   sortTable[0x971C]=21870;   // shuang
   sortTable[0x723D]=21871;   // shuang
   sortTable[0x5B40]=21872;   // shuang
   sortTable[0x9E74]=21873;   // shuang
   sortTable[0x9A3B]=21874;   // shuang
   sortTable[0x9AA6]=21875;   // shuang
   sortTable[0x9DDE]=21876;   // shuang
   sortTable[0x7935]=21877;   // shuang
   sortTable[0x826D]=21878;   // shuang
   sortTable[0x9A66]=21879;   // shuang
   sortTable[0x9E18]=21880;   // shuang
   sortTable[0x587D]=21881;   // shuang
   sortTable[0x5B47]=21882;   // shuang
   sortTable[0x6161]=21883;   // shuang
   sortTable[0x6A09]=21884;   // shuang
   sortTable[0x6B06]=21885;   // shuang
   sortTable[0x6F3A]=21886;   // shuang
   sortTable[0x7040]=21887;   // shuang
   sortTable[0x7E14]=21888;   // shuang
   sortTable[0x93EF]=21889;   // shuang
   sortTable[0x6C34]=21890;   // shui
   sortTable[0x8C01]=21891;   // shui
   sortTable[0x8AB0]=21892;   // shui
   sortTable[0x7761]=21893;   // shui
   sortTable[0x7A0E]=21894;   // shui
   sortTable[0x7A05]=21895;   // shui
   sortTable[0x813D]=21896;   // shui
   sortTable[0x7793]=21897;   // shui
   sortTable[0x88DE]=21898;   // shui
   sortTable[0x7971]=21899;   // shui
   sortTable[0x9596]=21900;   // shui
   sortTable[0x5E28]=21901;   // shui
   sortTable[0x6C35]=21902;   // shui
   sortTable[0x6C3A]=21903;   // shui
   sortTable[0x6D97]=21904;   // shui
   sortTable[0x6D9A]=21905;   // shui
   sortTable[0x987A]=21906;   // shun
   sortTable[0x9806]=21907;   // shun
   sortTable[0x821C]=21908;   // shun
   sortTable[0x77AC]=21909;   // shun
   sortTable[0x542E]=21910;   // shun
   sortTable[0x8563]=21911;   // shun
   sortTable[0x779A]=21912;   // shun
   sortTable[0x9B0A]=21913;   // shun
   sortTable[0x6A53]=21914;   // shun
   sortTable[0x8BF4]=21915;   // shuo
   sortTable[0x8AAA]=21916;   // shuo
   sortTable[0x70C1]=21917;   // shuo
   sortTable[0x720D]=21918;   // shuo
   sortTable[0x6714]=21919;   // shuo
   sortTable[0x7855]=21920;   // shuo
   sortTable[0x69CA]=21921;   // shuo
   sortTable[0x94C4]=21922;   // shuo
   sortTable[0x6420]=21923;   // shuo
   sortTable[0x5981]=21924;   // shuo
   sortTable[0x84B4]=21925;   // shuo
   sortTable[0x7BBE]=21926;   // shuo
   sortTable[0x77DF]=21927;   // shuo
   sortTable[0x78A9]=21928;   // shuo
   sortTable[0x8AAC]=21929;   // shuo
   sortTable[0x54FE]=21930;   // shuo
   sortTable[0x6B36]=21931;   // shuo
   sortTable[0x7361]=21932;   // shuo
   sortTable[0x9399]=21933;   // shuo
   sortTable[0x9460]=21934;   // shuo
   sortTable[0x56DB]=21935;   // si
   sortTable[0x601D]=21936;   // si
   sortTable[0x6B7B]=21937;   // si
   sortTable[0x53F8]=21938;   // si
   sortTable[0x4E1D]=21939;   // si
   sortTable[0x7D72]=21940;   // si
   sortTable[0x79C1]=21941;   // si
   sortTable[0x65AF]=21942;   // si
   sortTable[0x6495]=21943;   // si
   sortTable[0x9972]=21944;   // si
   sortTable[0x98FC]=21945;   // si
   sortTable[0x5636]=21946;   // si
   sortTable[0x5BFA]=21947;   // si
   sortTable[0x55E3]=21948;   // si
   sortTable[0x5DF3]=21949;   // si
   sortTable[0x7940]=21950;   // si
   sortTable[0x8086]=21951;   // si
   sortTable[0x53AE]=21952;   // si
   sortTable[0x6CD7]=21953;   // si
   sortTable[0x7B25]=21954;   // si
   sortTable[0x9A77]=21955;   // si
   sortTable[0x86F3]=21956;   // si
   sortTable[0x7F0C]=21957;   // si
   sortTable[0x9536]=21958;   // si
   sortTable[0x6C5C]=21959;   // si
   sortTable[0x549D]=21960;   // si
   sortTable[0x59D2]=21961;   // si
   sortTable[0x9E36]=21962;   // si
   sortTable[0x801C]=21963;   // si
   sortTable[0x5155]=21964;   // si
   sortTable[0x79A9]=21965;   // si
   sortTable[0x6F8C]=21966;   // si
   sortTable[0x9DE5]=21967;   // si
   sortTable[0x8652]=21968;   // si
   sortTable[0x98D4]=21969;   // si
   sortTable[0x7F73]=21970;   // si
   sortTable[0x7997]=21971;   // si
   sortTable[0x79A0]=21972;   // si
   sortTable[0x9236]=21973;   // si
   sortTable[0x9F36]=21974;   // si
   sortTable[0x8997]=21975;   // si
   sortTable[0x7AE2]=21976;   // si
   sortTable[0x857C]=21977;   // si
   sortTable[0x8082]=21978;   // si
   sortTable[0x8784]=21979;   // si
   sortTable[0x78C3]=21980;   // si
   sortTable[0x7DE6]=21981;   // si
   sortTable[0x8724]=21982;   // si
   sortTable[0x92AF]=21983;   // si
   sortTable[0x856C]=21984;   // si
   sortTable[0x87F4]=21985;   // si
   sortTable[0x8C84]=21986;   // si
   sortTable[0x923B]=21987;   // si
   sortTable[0x9401]=21988;   // si
   sortTable[0x98E4]=21989;   // si
   sortTable[0x99DF]=21990;   // si
   sortTable[0x4E96]=21991;   // si
   sortTable[0x4F40]=21992;   // si
   sortTable[0x4FA1]=21993;   // si
   sortTable[0x4FEC]=21994;   // si
   sortTable[0x5129]=21995;   // si
   sortTable[0x51D8]=21996;   // si
   sortTable[0x53B6]=21997;   // si
   sortTable[0x565D]=21998;   // si
   sortTable[0x5A30]=21999;   // si
   sortTable[0x5AA4]=22000;   // si
   sortTable[0x5B60]=22001;   // si
   sortTable[0x5EDD]=22002;   // si
   sortTable[0x6056]=22003;   // si
   sortTable[0x676B]=22004;   // si
   sortTable[0x67F6]=22005;   // si
   sortTable[0x6952]=22006;   // si
   sortTable[0x69B9]=22007;   // si
   sortTable[0x6CC0]=22008;   // si
   sortTable[0x6CE4]=22009;   // si
   sortTable[0x6D0D]=22010;   // si
   sortTable[0x6D98]=22011;   // si
   sortTable[0x7003]=22012;   // si
   sortTable[0x71CD]=22013;   // si
   sortTable[0x726D]=22014;   // si
   sortTable[0x7C6D]=22015;   // si
   sortTable[0x7CF9]=22016;   // si
   sortTable[0x7E9F]=22017;   // si
   sortTable[0x87D6]=22018;   // si
   sortTable[0x91F2]=22019;   // si
   sortTable[0x9270]=22020;   // si
   sortTable[0x92D6]=22021;   // si
   sortTable[0x98B8]=22022;   // si
   sortTable[0x9A26]=22023;   // si
   sortTable[0x9001]=22024;   // song
   sortTable[0x677E]=22025;   // song
   sortTable[0x8038]=22026;   // song
   sortTable[0x8073]=22027;   // song
   sortTable[0x9882]=22028;   // song
   sortTable[0x980C]=22029;   // song
   sortTable[0x8BBC]=22030;   // song
   sortTable[0x8A1F]=22031;   // song
   sortTable[0x8BF5]=22032;   // song
   sortTable[0x8AA6]=22033;   // song
   sortTable[0x5B8B]=22034;   // song
   sortTable[0x5D69]=22035;   // song
   sortTable[0x51C7]=22036;   // song
   sortTable[0x6002]=22037;   // song
   sortTable[0x609A]=22038;   // song
   sortTable[0x7AE6]=22039;   // song
   sortTable[0x6DDE]=22040;   // song
   sortTable[0x5D27]=22041;   // song
   sortTable[0x83D8]=22042;   // song
   sortTable[0x5FEA]=22043;   // song
   sortTable[0x9B06]=22044;   // song
   sortTable[0x99F7]=22045;   // song
   sortTable[0x8719]=22046;   // song
   sortTable[0x502F]=22047;   // song
   sortTable[0x50B1]=22048;   // song
   sortTable[0x5A00]=22049;   // song
   sortTable[0x5D77]=22050;   // song
   sortTable[0x5EBA]=22051;   // song
   sortTable[0x612F]=22052;   // song
   sortTable[0x616B]=22053;   // song
   sortTable[0x61BD]=22054;   // song
   sortTable[0x6780]=22055;   // song
   sortTable[0x67A9]=22056;   // song
   sortTable[0x67D7]=22057;   // song
   sortTable[0x68A5]=22058;   // song
   sortTable[0x6964]=22059;   // song
   sortTable[0x6AA7]=22060;   // song
   sortTable[0x6FCD]=22061;   // song
   sortTable[0x7879]=22062;   // song
   sortTable[0x9376]=22063;   // song
   sortTable[0x93B9]=22064;   // song
   sortTable[0x9938]=22065;   // song
   sortTable[0x641C]=22066;   // sou
   sortTable[0x55FD]=22067;   // sou
   sortTable[0x8258]=22068;   // sou
   sortTable[0x53DF]=22069;   // sou
   sortTable[0x64DE]=22070;   // sou
   sortTable[0x55D6]=22071;   // sou
   sortTable[0x85AE]=22072;   // sou
   sortTable[0x98D5]=22073;   // sou
   sortTable[0x998A]=22074;   // sou
   sortTable[0x6EB2]=22075;   // sou
   sortTable[0x55FE]=22076;   // sou
   sortTable[0x953C]=22077;   // sou
   sortTable[0x8490]=22078;   // sou
   sortTable[0x778D]=22079;   // sou
   sortTable[0x878B]=22080;   // sou
   sortTable[0x910B]=22081;   // sou
   sortTable[0x7C54]=22082;   // sou
   sortTable[0x93AA]=22083;   // sou
   sortTable[0x98BC]=22084;   // sou
   sortTable[0x8184]=22085;   // sou
   sortTable[0x84C3]=22086;   // sou
   sortTable[0x85EA]=22087;   // sou
   sortTable[0x9199]=22088;   // sou
   sortTable[0x9A2A]=22089;   // sou
   sortTable[0x5081]=22090;   // sou
   sortTable[0x53DC]=22091;   // sou
   sortTable[0x5EC0]=22092;   // sou
   sortTable[0x5ECB]=22093;   // sou
   sortTable[0x635C]=22094;   // sou
   sortTable[0x6449]=22095;   // sou
   sortTable[0x6457]=22096;   // sou
   sortTable[0x64FB]=22097;   // sou
   sortTable[0x6AE2]=22098;   // sou
   sortTable[0x7340]=22099;   // sou
   sortTable[0x7636]=22100;   // sou
   sortTable[0x98BE]=22101;   // sou
   sortTable[0x993F]=22102;   // sou
   sortTable[0x901F]=22103;   // su
   sortTable[0x8BC9]=22104;   // su
   sortTable[0x8A34]=22105;   // su
   sortTable[0x7D20]=22106;   // su
   sortTable[0x8083]=22107;   // su
   sortTable[0x8085]=22108;   // su
   sortTable[0x5851]=22109;   // su
   sortTable[0x5BBF]=22110;   // su
   sortTable[0x4FD7]=22111;   // su
   sortTable[0x82CF]=22112;   // su
   sortTable[0x8607]=22113;   // su
   sortTable[0x7C9F]=22114;   // su
   sortTable[0x7A23]=22115;   // su
   sortTable[0x9165]=22116;   // su
   sortTable[0x6EAF]=22117;   // su
   sortTable[0x5919]=22118;   // su
   sortTable[0x8C21]=22119;   // su
   sortTable[0x612B]=22120;   // su
   sortTable[0x7C0C]=22121;   // su
   sortTable[0x55C9]=22122;   // su
   sortTable[0x6D91]=22123;   // su
   sortTable[0x7AA3]=22124;   // su
   sortTable[0x9917]=22125;   // su
   sortTable[0x50F3]=22126;   // su
   sortTable[0x850C]=22127;   // su
   sortTable[0x89EB]=22128;   // su
   sortTable[0x9A95]=22129;   // su
   sortTable[0x9DEB]=22130;   // su
   sortTable[0x9E54]=22131;   // su
   sortTable[0x9C50]=22132;   // su
   sortTable[0x83A4]=22133;   // su
   sortTable[0x8E5C]=22134;   // su
   sortTable[0x9061]=22135;   // su
   sortTable[0x9A4C]=22136;   // su
   sortTable[0x7C9B]=22137;   // su
   sortTable[0x8186]=22138;   // su
   sortTable[0x85D7]=22139;   // su
   sortTable[0x8D9A]=22140;   // su
   sortTable[0x9BC2]=22141;   // su
   sortTable[0x7E24]=22142;   // su
   sortTable[0x8613]=22143;   // su
   sortTable[0x906C]=22144;   // su
   sortTable[0x5083]=22145;   // su
   sortTable[0x56CC]=22146;   // su
   sortTable[0x5850]=22147;   // su
   sortTable[0x5ACA]=22148;   // su
   sortTable[0x612C]=22149;   // su
   sortTable[0x619F]=22150;   // su
   sortTable[0x6880]=22151;   // su
   sortTable[0x69A1]=22152;   // su
   sortTable[0x6A0E]=22153;   // su
   sortTable[0x6A15]=22154;   // su
   sortTable[0x6A5A]=22155;   // su
   sortTable[0x6AEF]=22156;   // su
   sortTable[0x6B90]=22157;   // su
   sortTable[0x6CDD]=22158;   // su
   sortTable[0x6D2C]=22159;   // su
   sortTable[0x6EB8]=22160;   // su
   sortTable[0x6F5A]=22161;   // su
   sortTable[0x6F65]=22162;   // su
   sortTable[0x738A]=22163;   // su
   sortTable[0x73DF]=22164;   // su
   sortTable[0x749B]=22165;   // su
   sortTable[0x7526]=22166;   // su
   sortTable[0x78BF]=22167;   // su
   sortTable[0x7A4C]=22168;   // su
   sortTable[0x8B16]=22169;   // su
   sortTable[0x92C9]=22170;   // su
   sortTable[0x7B97]=22171;   // suan
   sortTable[0x9178]=22172;   // suan
   sortTable[0x849C]=22173;   // suan
   sortTable[0x72FB]=22174;   // suan
   sortTable[0x7B6D]=22175;   // suan
   sortTable[0x7B07]=22176;   // suan
   sortTable[0x7958]=22177;   // suan
   sortTable[0x5334]=22178;   // suan
   sortTable[0x75E0]=22179;   // suan
   sortTable[0x968F]=22180;   // sui
   sortTable[0x96A8]=22181;   // sui
   sortTable[0x867D]=22182;   // sui
   sortTable[0x96D6]=22183;   // sui
   sortTable[0x5C81]=22184;   // sui
   sortTable[0x6B72]=22185;   // sui
   sortTable[0x788E]=22186;   // sui
   sortTable[0x96A7]=22187;   // sui
   sortTable[0x7A57]=22188;   // sui
   sortTable[0x9042]=22189;   // sui
   sortTable[0x968B]=22190;   // sui
   sortTable[0x7EE5]=22191;   // sui
   sortTable[0x837D]=22192;   // sui
   sortTable[0x7762]=22193;   // sui
   sortTable[0x9AD3]=22194;   // sui
   sortTable[0x795F]=22195;   // sui
   sortTable[0x6FC9]=22196;   // sui
   sortTable[0x9083]=22197;   // sui
   sortTable[0x71E7]=22198;   // sui
   sortTable[0x8C07]=22199;   // sui
   sortTable[0x6535]=22200;   // sui
   sortTable[0x772D]=22201;   // sui
   sortTable[0x895A]=22202;   // sui
   sortTable[0x775F]=22203;   // sui
   sortTable[0x9429]=22204;   // sui
   sortTable[0x7E50]=22205;   // sui
   sortTable[0x8470]=22206;   // sui
   sortTable[0x7A5F]=22207;   // sui
   sortTable[0x79AD]=22208;   // sui
   sortTable[0x7E78]=22209;   // sui
   sortTable[0x8B62]=22210;   // sui
   sortTable[0x81B8]=22211;   // sui
   sortTable[0x9796]=22212;   // sui
   sortTable[0x7D8F]=22213;   // sui
   sortTable[0x7E40]=22214;   // sui
   sortTable[0x837E]=22215;   // sui
   sortTable[0x4E97]=22216;   // sui
   sortTable[0x5020]=22217;   // sui
   sortTable[0x54F8]=22218;   // sui
   sortTable[0x57E3]=22219;   // sui
   sortTable[0x590A]=22220;   // sui
   sortTable[0x5B18]=22221;   // sui
   sortTable[0x5D57]=22222;   // sui
   sortTable[0x65DE]=22223;   // sui
   sortTable[0x6A85]=22224;   // sui
   sortTable[0x6A96]=22225;   // sui
   sortTable[0x6B73]=22226;   // sui
   sortTable[0x6D7D]=22227;   // sui
   sortTable[0x6ED6]=22228;   // sui
   sortTable[0x6FBB]=22229;   // sui
   sortTable[0x7021]=22230;   // sui
   sortTable[0x716B]=22231;   // sui
   sortTable[0x71A3]=22232;   // sui
   sortTable[0x74B2]=22233;   // sui
   sortTable[0x74CD]=22234;   // sui
   sortTable[0x7815]=22235;   // sui
   sortTable[0x7A42]=22236;   // sui
   sortTable[0x8295]=22237;   // sui
   sortTable[0x8AB6]=22238;   // sui
   sortTable[0x8CE5]=22239;   // sui
   sortTable[0x9040]=22240;   // sui
   sortTable[0x9406]=22241;   // sui
   sortTable[0x97E2]=22242;   // sui
   sortTable[0x9AC4]=22243;   // sui
   sortTable[0x635F]=22244;   // sun
   sortTable[0x640D]=22245;   // sun
   sortTable[0x5B59]=22246;   // sun
   sortTable[0x5B6B]=22247;   // sun
   sortTable[0x7B0B]=22248;   // sun
   sortTable[0x96BC]=22249;   // sun
   sortTable[0x98E7]=22250;   // sun
   sortTable[0x836A]=22251;   // sun
   sortTable[0x69AB]=22252;   // sun
   sortTable[0x72F2]=22253;   // sun
   sortTable[0x7C28]=22254;   // sun
   sortTable[0x7BB0]=22255;   // sun
   sortTable[0x7B4D]=22256;   // sun
   sortTable[0x98F1]=22257;   // sun
   sortTable[0x8575]=22258;   // sun
   sortTable[0x9DBD]=22259;   // sun
   sortTable[0x640E]=22260;   // sun
   sortTable[0x69C2]=22261;   // sun
   sortTable[0x733B]=22262;   // sun
   sortTable[0x84C0]=22263;   // sun
   sortTable[0x859E]=22264;   // sun
   sortTable[0x93A8]=22265;   // sun
   sortTable[0x6240]=22266;   // suo
   sortTable[0x7F29]=22267;   // suo
   sortTable[0x7E2E]=22268;   // suo
   sortTable[0x7D22]=22269;   // suo
   sortTable[0x9501]=22270;   // suo
   sortTable[0x9396]=22271;   // suo
   sortTable[0x55E6]=22272;   // suo
   sortTable[0x7410]=22273;   // suo
   sortTable[0x7463]=22274;   // suo
   sortTable[0x68AD]=22275;   // suo
   sortTable[0x5A11]=22276;   // suo
   sortTable[0x5506]=22277;   // suo
   sortTable[0x55CD]=22278;   // suo
   sortTable[0x84D1]=22279;   // suo
   sortTable[0x7FA7]=22280;   // suo
   sortTable[0x7743]=22281;   // suo
   sortTable[0x5522]=22282;   // suo
   sortTable[0x686B]=22283;   // suo
   sortTable[0x93C1]=22284;   // suo
   sortTable[0x9024]=22285;   // suo
   sortTable[0x9AFF]=22286;   // suo
   sortTable[0x9BBB]=22287;   // suo
   sortTable[0x8928]=22288;   // suo
   sortTable[0x93BB]=22289;   // suo
   sortTable[0x7C11]=22290;   // suo
   sortTable[0x8736]=22291;   // suo
   sortTable[0x938D]=22292;   // suo
   sortTable[0x4E7A]=22293;   // suo
   sortTable[0x509E]=22294;   // suo
   sortTable[0x55E9]=22295;   // suo
   sortTable[0x60E2]=22296;   // suo
   sortTable[0x644D]=22297;   // suo
   sortTable[0x669B]=22298;   // suo
   sortTable[0x6E91]=22299;   // suo
   sortTable[0x6EB9]=22300;   // suo
   sortTable[0x7411]=22301;   // suo
   sortTable[0x7485]=22302;   // suo
   sortTable[0x7C14]=22303;   // suo
   sortTable[0x838F]=22304;   // suo
   sortTable[0x8D96]=22305;   // suo
   sortTable[0x9388]=22306;   // suo
   sortTable[0x4ED6]=22784;   // ta
   sortTable[0x5979]=22785;   // ta
   sortTable[0x5B83]=22786;   // ta
   sortTable[0x8E0F]=22787;   // ta
   sortTable[0x5854]=22788;   // ta
   sortTable[0x584C]=22789;   // ta
   sortTable[0x8E4B]=22790;   // ta
   sortTable[0x62D3]=22791;   // ta
   sortTable[0x69BB]=22792;   // ta
   sortTable[0x9CCE]=22793;   // ta
   sortTable[0x736D]=22794;   // ta
   sortTable[0x631E]=22795;   // ta
   sortTable[0x95FC]=22796;   // ta
   sortTable[0x8DBF]=22797;   // ta
   sortTable[0x9062]=22798;   // ta
   sortTable[0x6EBB]=22799;   // ta
   sortTable[0x94CA]=22800;   // ta
   sortTable[0x8DF6]=22801;   // ta
   sortTable[0x905D]=22802;   // ta
   sortTable[0x9314]=22803;   // ta
   sortTable[0x7942]=22804;   // ta
   sortTable[0x9C28]=22805;   // ta
   sortTable[0x8ABB]=22806;   // ta
   sortTable[0x8E79]=22807;   // ta
   sortTable[0x95D2]=22808;   // ta
   sortTable[0x891F]=22809;   // ta
   sortTable[0x979C]=22810;   // ta
   sortTable[0x79A2]=22811;   // ta
   sortTable[0x97B3]=22812;   // ta
   sortTable[0x8EA2]=22813;   // ta
   sortTable[0x4EA3]=22814;   // ta
   sortTable[0x4FA4]=22815;   // ta
   sortTable[0x549C]=22816;   // ta
   sortTable[0x5683]=22817;   // ta
   sortTable[0x56BA]=22818;   // ta
   sortTable[0x5896]=22819;   // ta
   sortTable[0x5D09]=22820;   // ta
   sortTable[0x6428]=22821;   // ta
   sortTable[0x64BB]=22822;   // ta
   sortTable[0x6999]=22823;   // ta
   sortTable[0x6A7D]=22824;   // ta
   sortTable[0x6BFE]=22825;   // ta
   sortTable[0x6DBE]=22826;   // ta
   sortTable[0x6E9A]=22827;   // ta
   sortTable[0x6FBE]=22828;   // ta
   sortTable[0x6FCC]=22829;   // ta
   sortTable[0x7260]=22830;   // ta
   sortTable[0x72E7]=22831;   // ta
   sortTable[0x737A]=22832;   // ta
   sortTable[0x8B76]=22833;   // ta
   sortTable[0x95E5]=22834;   // ta
   sortTable[0x95E7]=22835;   // ta
   sortTable[0x9B99]=22836;   // ta
   sortTable[0x592A]=22837;   // tai
   sortTable[0x53F0]=22838;   // tai
   sortTable[0x6001]=22839;   // tai
   sortTable[0x614B]=22840;   // tai
   sortTable[0x62AC]=22841;   // tai
   sortTable[0x80CE]=22842;   // tai
   sortTable[0x6C70]=22843;   // tai
   sortTable[0x6CF0]=22844;   // tai
   sortTable[0x82D4]=22845;   // tai
   sortTable[0x80BD]=22846;   // tai
   sortTable[0x949B]=22847;   // tai
   sortTable[0x70B1]=22848;   // tai
   sortTable[0x85B9]=22849;   // tai
   sortTable[0x8DC6]=22850;   // tai
   sortTable[0x90B0]=22851;   // tai
   sortTable[0x9C90]=22852;   // tai
   sortTable[0x915E]=22853;   // tai
   sortTable[0x9226]=22854;   // tai
   sortTable[0x7C8F]=22855;   // tai
   sortTable[0x81FA]=22856;   // tai
   sortTable[0x83ED]=22857;   // tai
   sortTable[0x9B90]=22858;   // tai
   sortTable[0x7B88]=22859;   // tai
   sortTable[0x8226]=22860;   // tai
   sortTable[0x98B1]=22861;   // tai
   sortTable[0x5113]=22862;   // tai
   sortTable[0x51AD]=22863;   // tai
   sortTable[0x56FC]=22864;   // tai
   sortTable[0x576E]=22865;   // tai
   sortTable[0x5933]=22866;   // tai
   sortTable[0x5B2F]=22867;   // tai
   sortTable[0x5B61]=22868;   // tai
   sortTable[0x5FF2]=22869;   // tai
   sortTable[0x64E1]=22870;   // tai
   sortTable[0x65F2]=22871;   // tai
   sortTable[0x67B1]=22872;   // tai
   sortTable[0x6AAF]=22873;   // tai
   sortTable[0x6E99]=22874;   // tai
   sortTable[0x70B2]=22875;   // tai
   sortTable[0x71E4]=22876;   // tai
   sortTable[0x7C49]=22877;   // tai
   sortTable[0x99D8]=22878;   // tai
   sortTable[0x8C08]=22879;   // tan
   sortTable[0x8AC7]=22880;   // tan
   sortTable[0x53F9]=22881;   // tan
   sortTable[0x5606]=22882;   // tan
   sortTable[0x63A2]=22883;   // tan
   sortTable[0x5766]=22884;   // tan
   sortTable[0x70AD]=22885;   // tan
   sortTable[0x644A]=22886;   // tan
   sortTable[0x6524]=22887;   // tan
   sortTable[0x6EE9]=22888;   // tan
   sortTable[0x7058]=22889;   // tan
   sortTable[0x78B3]=22890;   // tan
   sortTable[0x8D2A]=22891;   // tan
   sortTable[0x8CAA]=22892;   // tan
   sortTable[0x6F6D]=22893;   // tan
   sortTable[0x6BEF]=22894;   // tan
   sortTable[0x75F0]=22895;   // tan
   sortTable[0x575B]=22896;   // tan
   sortTable[0x8C2D]=22897;   // tan
   sortTable[0x6A80]=22898;   // tan
   sortTable[0x762B]=22899;   // tan
   sortTable[0x6619]=22900;   // tan
   sortTable[0x8892]=22901;   // tan
   sortTable[0x574D]=22902;   // tan
   sortTable[0x8983]=22903;   // tan
   sortTable[0x90EF]=22904;   // tan
   sortTable[0x5FD0]=22905;   // tan
   sortTable[0x952C]=22906;   // tan
   sortTable[0x94BD]=22907;   // tan
   sortTable[0x83FC]=22908;   // tan
   sortTable[0x8962]=22909;   // tan
   sortTable[0x9193]=22910;   // tan
   sortTable[0x9924]=22911;   // tan
   sortTable[0x78F9]=22912;   // tan
   sortTable[0x7F48]=22913;   // tan
   sortTable[0x91B0]=22914;   // tan
   sortTable[0x7F4E]=22915;   // tan
   sortTable[0x85EB]=22916;   // tan
   sortTable[0x9188]=22917;   // tan
   sortTable[0x8C9A]=22918;   // tan
   sortTable[0x931F]=22919;   // tan
   sortTable[0x5013]=22920;   // tan
   sortTable[0x509D]=22921;   // tan
   sortTable[0x50CB]=22922;   // tan
   sortTable[0x55FF]=22923;   // tan
   sortTable[0x57EE]=22924;   // tan
   sortTable[0x58B0]=22925;   // tan
   sortTable[0x58B5]=22926;   // tan
   sortTable[0x58C7]=22927;   // tan
   sortTable[0x58DC]=22928;   // tan
   sortTable[0x5A52]=22929;   // tan
   sortTable[0x6039]=22930;   // tan
   sortTable[0x60D4]=22931;   // tan
   sortTable[0x619B]=22932;   // tan
   sortTable[0x61B3]=22933;   // tan
   sortTable[0x61BB]=22934;   // tan
   sortTable[0x62A9]=22935;   // tan
   sortTable[0x64F9]=22936;   // tan
   sortTable[0x66C7]=22937;   // tan
   sortTable[0x6983]=22938;   // tan
   sortTable[0x6B4E]=22939;   // tan
   sortTable[0x6E60]=22940;   // tan
   sortTable[0x71C2]=22941;   // tan
   sortTable[0x74AE]=22942;   // tan
   sortTable[0x75D1]=22943;   // tan
   sortTable[0x7671]=22944;   // tan
   sortTable[0x8211]=22945;   // tan
   sortTable[0x8215]=22946;   // tan
   sortTable[0x8B5A]=22947;   // tan
   sortTable[0x8B60]=22948;   // tan
   sortTable[0x8CE7]=22949;   // tan
   sortTable[0x926D]=22950;   // tan
   sortTable[0x9843]=22951;   // tan
   sortTable[0x8EBA]=22952;   // tang
   sortTable[0x5802]=22953;   // tang
   sortTable[0x7CD6]=22954;   // tang
   sortTable[0x8D9F]=22955;   // tang
   sortTable[0x5858]=22956;   // tang
   sortTable[0x5018]=22957;   // tang
   sortTable[0x6C64]=22958;   // tang
   sortTable[0x6E6F]=22959;   // tang
   sortTable[0x70EB]=22960;   // tang
   sortTable[0x71D9]=22961;   // tang
   sortTable[0x819B]=22962;   // tang
   sortTable[0x6DCC]=22963;   // tang
   sortTable[0x642A]=22964;   // tang
   sortTable[0x5510]=22965;   // tang
   sortTable[0x68E0]=22966;   // tang
   sortTable[0x50A5]=22967;   // tang
   sortTable[0x5E11]=22968;   // tang
   sortTable[0x7FB0]=22969;   // tang
   sortTable[0x9967]=22970;   // tang
   sortTable[0x746D]=22971;   // tang
   sortTable[0x87B3]=22972;   // tang
   sortTable[0x91A3]=22973;   // tang
   sortTable[0x9557]=22974;   // tang
   sortTable[0x6A18]=22975;   // tang
   sortTable[0x8E5A]=22976;   // tang
   sortTable[0x6E8F]=22977;   // tang
   sortTable[0x799F]=22978;   // tang
   sortTable[0x8797]=22979;   // tang
   sortTable[0x954B]=22980;   // tang
   sortTable[0x8025]=22981;   // tang
   sortTable[0x77D8]=22982;   // tang
   sortTable[0x910C]=22983;   // tang
   sortTable[0x93DC]=22984;   // tang
   sortTable[0x9F1E]=22985;   // tang
   sortTable[0x93B2]=22986;   // tang
   sortTable[0x97BA]=22987;   // tang
   sortTable[0x8E3C]=22988;   // tang
   sortTable[0x940B]=22989;   // tang
   sortTable[0x9482]=22990;   // tang
   sortTable[0x95DB]=22991;   // tang
   sortTable[0x969A]=22992;   // tang
   sortTable[0x78C4]=22993;   // tang
   sortTable[0x7BD6]=22994;   // tang
   sortTable[0x876A]=22995;   // tang
   sortTable[0x94F4]=22996;   // tang
   sortTable[0x9944]=22997;   // tang
   sortTable[0x9DB6]=22998;   // tang
   sortTable[0x4F16]=22999;   // tang
   sortTable[0x5052]=23000;   // tang
   sortTable[0x508F]=23001;   // tang
   sortTable[0x513B]=23002;   // tang
   sortTable[0x528F]=23003;   // tang
   sortTable[0x557A]=23004;   // tang
   sortTable[0x5621]=23005;   // tang
   sortTable[0x5763]=23006;   // tang
   sortTable[0x6203]=23007;   // tang
   sortTable[0x6465]=23008;   // tang
   sortTable[0x66ED]=23009;   // tang
   sortTable[0x69B6]=23010;   // tang
   sortTable[0x6A56]=23011;   // tang
   sortTable[0x6F1F]=23012;   // tang
   sortTable[0x717B]=23013;   // tang
   sortTable[0x7223]=23014;   // tang
   sortTable[0x7CC3]=23015;   // tang
   sortTable[0x7CDB]=23016;   // tang
   sortTable[0x8185]=23017;   // tang
   sortTable[0x84CE]=23018;   // tang
   sortTable[0x859A]=23019;   // tang
   sortTable[0x8D6F]=23020;   // tang
   sortTable[0x9395]=23021;   // tang
   sortTable[0x9933]=23022;   // tang
   sortTable[0x9939]=23023;   // tang
   sortTable[0x8BA8]=23024;   // tao
   sortTable[0x8A0E]=23025;   // tao
   sortTable[0x5957]=23026;   // tao
   sortTable[0x9003]=23027;   // tao
   sortTable[0x638F]=23028;   // tao
   sortTable[0x6843]=23029;   // tao
   sortTable[0x8404]=23030;   // tao
   sortTable[0x6DD8]=23031;   // tao
   sortTable[0x9676]=23032;   // tao
   sortTable[0x6D9B]=23033;   // tao
   sortTable[0x6FE4]=23034;   // tao
   sortTable[0x6ED4]=23035;   // tao
   sortTable[0x97EC]=23036;   // tao
   sortTable[0x9955]=23037;   // tao
   sortTable[0x6D2E]=23038;   // tao
   sortTable[0x7EE6]=23039;   // tao
   sortTable[0x9F17]=23040;   // tao
   sortTable[0x5555]=23041;   // tao
   sortTable[0x7DAF]=23042;   // tao
   sortTable[0x7EF9]=23043;   // tao
   sortTable[0x7E1A]=23044;   // tao
   sortTable[0x7979]=23045;   // tao
   sortTable[0x9A0A]=23046;   // tao
   sortTable[0x8FEF]=23047;   // tao
   sortTable[0x9789]=23048;   // tao
   sortTable[0x9780]=23049;   // tao
   sortTable[0x92FE]=23050;   // tao
   sortTable[0x7D5B]=23051;   // tao
   sortTable[0x8B1F]=23052;   // tao
   sortTable[0x9184]=23053;   // tao
   sortTable[0x98F8]=23054;   // tao
   sortTable[0x99E3]=23055;   // tao
   sortTable[0x872A]=23056;   // tao
   sortTable[0x7E27]=23057;   // tao
   sortTable[0x8F41]=23058;   // tao
   sortTable[0x932D]=23059;   // tao
   sortTable[0x530B]=23060;   // tao
   sortTable[0x54B7]=23061;   // tao
   sortTable[0x5932]=23062;   // tao
   sortTable[0x5ACD]=23063;   // tao
   sortTable[0x5E4D]=23064;   // tao
   sortTable[0x5F22]=23065;   // tao
   sortTable[0x6146]=23066;   // tao
   sortTable[0x642F]=23067;   // tao
   sortTable[0x68BC]=23068;   // tao
   sortTable[0x69C4]=23069;   // tao
   sortTable[0x6AAE]=23070;   // tao
   sortTable[0x746B]=23071;   // tao
   sortTable[0x88EA]=23072;   // tao
   sortTable[0x8A5C]=23073;   // tao
   sortTable[0x97B1]=23074;   // tao
   sortTable[0x97DC]=23075;   // tao
   sortTable[0x9940]=23076;   // tao
   sortTable[0x7279]=23077;   // te
   sortTable[0x5FD2]=23078;   // te
   sortTable[0x615D]=23079;   // te
   sortTable[0x5FD1]=23080;   // te
   sortTable[0x87A3]=23081;   // te
   sortTable[0x8CA3]=23082;   // te
   sortTable[0x94FD]=23083;   // te
   sortTable[0x87D8]=23084;   // te
   sortTable[0x92F1]=23085;   // te
   sortTable[0x86AE]=23086;   // te
   sortTable[0x75BC]=23087;   // teng
   sortTable[0x817E]=23088;   // teng
   sortTable[0x9A30]=23089;   // teng
   sortTable[0x85E4]=23090;   // teng
   sortTable[0x6ED5]=23091;   // teng
   sortTable[0x8A8A]=23092;   // teng
   sortTable[0x7E22]=23093;   // teng
   sortTable[0x9F1F]=23094;   // teng
   sortTable[0x8B04]=23095;   // teng
   sortTable[0x81AF]=23096;   // teng
   sortTable[0x9086]=23097;   // teng
   sortTable[0x7C50]=23098;   // teng
   sortTable[0x99E6]=23099;   // teng
   sortTable[0x9C27]=23100;   // teng
   sortTable[0x512F]=23101;   // teng
   sortTable[0x5E50]=23102;   // teng
   sortTable[0x6F1B]=23103;   // teng
   sortTable[0x71A5]=23104;   // teng
   sortTable[0x75CB]=23105;   // teng
   sortTable[0x7C58]=23106;   // teng
   sortTable[0x8645]=23107;   // teng
   sortTable[0x972F]=23108;   // teng
   sortTable[0x9A63]=23109;   // teng
   sortTable[0x4F53]=23110;   // ti
   sortTable[0x9AD4]=23111;   // ti
   sortTable[0x9898]=23112;   // ti
   sortTable[0x984C]=23113;   // ti
   sortTable[0x63D0]=23114;   // ti
   sortTable[0x66FF]=23115;   // ti
   sortTable[0x8E22]=23116;   // ti
   sortTable[0x68AF]=23117;   // ti
   sortTable[0x5C49]=23118;   // ti
   sortTable[0x5C5C]=23119;   // ti
   sortTable[0x60D5]=23120;   // ti
   sortTable[0x8E44]=23121;   // ti
   sortTable[0x6D95]=23122;   // ti
   sortTable[0x557C]=23123;   // ti
   sortTable[0x5254]=23124;   // ti
   sortTable[0x5243]=23125;   // ti
   sortTable[0x9016]=23126;   // ti
   sortTable[0x608C]=23127;   // ti
   sortTable[0x88FC]=23128;   // ti
   sortTable[0x501C]=23129;   // ti
   sortTable[0x568F]=23130;   // ti
   sortTable[0x7F07]=23131;   // ti
   sortTable[0x7EE8]=23132;   // ti
   sortTable[0x9E48]=23133;   // ti
   sortTable[0x918D]=23134;   // ti
   sortTable[0x8351]=23135;   // ti
   sortTable[0x9511]=23136;   // ti
   sortTable[0x8599]=23137;   // ti
   sortTable[0x8DAF]=23138;   // ti
   sortTable[0x9BA7]=23139;   // ti
   sortTable[0x9A20]=23140;   // ti
   sortTable[0x9D97]=23141;   // ti
   sortTable[0x9CC0]=23142;   // ti
   sortTable[0x9D3A]=23143;   // ti
   sortTable[0x7D88]=23144;   // ti
   sortTable[0x8EC6]=23145;   // ti
   sortTable[0x7A0A]=23146;   // ti
   sortTable[0x9037]=23147;   // ti
   sortTable[0x9DC9]=23148;   // ti
   sortTable[0x79B5]=23149;   // ti
   sortTable[0x8DA7]=23150;   // ti
   sortTable[0x7B39]=23151;   // ti
   sortTable[0x9046]=23152;   // ti
   sortTable[0x9DE4]=23153;   // ti
   sortTable[0x855B]=23154;   // ti
   sortTable[0x7DF9]=23155;   // ti
   sortTable[0x8E4F]=23156;   // ti
   sortTable[0x8FCF]=23157;   // ti
   sortTable[0x9357]=23158;   // ti
   sortTable[0x7C4A]=23159;   // ti
   sortTable[0x9B00]=23160;   // ti
   sortTable[0x9BF7]=23161;   // ti
   sortTable[0x9D99]=23162;   // ti
   sortTable[0x78AE]=23163;   // ti
   sortTable[0x876D]=23164;   // ti
   sortTable[0x8905]=23165;   // ti
   sortTable[0x8906]=23166;   // ti
   sortTable[0x9BB7]=23167;   // ti
   sortTable[0x504D]=23168;   // ti
   sortTable[0x5397]=23169;   // ti
   sortTable[0x55C1]=23170;   // ti
   sortTable[0x5694]=23171;   // ti
   sortTable[0x5D39]=23172;   // ti
   sortTable[0x5D5C]=23173;   // ti
   sortTable[0x5FB2]=23174;   // ti
   sortTable[0x6090]=23175;   // ti
   sortTable[0x60D6]=23176;   // ti
   sortTable[0x60FF]=23177;   // ti
   sortTable[0x623B]=23178;   // ti
   sortTable[0x632E]=23179;   // ti
   sortTable[0x63A6]=23180;   // ti
   sortTable[0x63E5]=23181;   // ti
   sortTable[0x64FF]=23182;   // ti
   sortTable[0x6711]=23183;   // ti
   sortTable[0x6974]=23184;   // ti
   sortTable[0x6B52]=23185;   // ti
   sortTable[0x6BA2]=23186;   // ti
   sortTable[0x6D1F]=23187;   // ti
   sortTable[0x6F3D]=23188;   // ti
   sortTable[0x7445]=23189;   // ti
   sortTable[0x74CB]=23190;   // ti
   sortTable[0x7F64]=23191;   // ti
   sortTable[0x82D0]=23192;   // ti
   sortTable[0x8B15]=23193;   // ti
   sortTable[0x8EB0]=23194;   // ti
   sortTable[0x92BB]=23195;   // ti
   sortTable[0x9AB5]=23196;   // ti
   sortTable[0x9AF0]=23197;   // ti
   sortTable[0x9D5C]=23198;   // ti
   sortTable[0x9DC8]=23199;   // ti
   sortTable[0x5929]=23200;   // tian
   sortTable[0x7530]=23201;   // tian
   sortTable[0x6DFB]=23202;   // tian
   sortTable[0x751C]=23203;   // tian
   sortTable[0x586B]=23204;   // tian
   sortTable[0x8214]=23205;   // tian
   sortTable[0x606C]=23206;   // tian
   sortTable[0x5FDD]=23207;   // tian
   sortTable[0x754B]=23208;   // tian
   sortTable[0x6B84]=23209;   // tian
   sortTable[0x9617]=23210;   // tian
   sortTable[0x8146]=23211;   // tian
   sortTable[0x7AB4]=23212;   // tian
   sortTable[0x9766]=23213;   // tian
   sortTable[0x9902]=23214;   // tian
   sortTable[0x63AD]=23215;   // tian
   sortTable[0x89CD]=23216;   // tian
   sortTable[0x9D2B]=23217;   // tian
   sortTable[0x9369]=23218;   // tian
   sortTable[0x923F]=23219;   // tian
   sortTable[0x78CC]=23220;   // tian
   sortTable[0x83FE]=23221;   // tian
   sortTable[0x89A5]=23222;   // tian
   sortTable[0x76F7]=23223;   // tian
   sortTable[0x9EC7]=23224;   // tian
   sortTable[0x7753]=23225;   // tian
   sortTable[0x777C]=23226;   // tian
   sortTable[0x78B5]=23227;   // tian
   sortTable[0x8CDF]=23228;   // tian
   sortTable[0x9754]=23229;   // tian
   sortTable[0x9DCF]=23230;   // tian
   sortTable[0x500E]=23231;   // tian
   sortTable[0x5172]=23232;   // tian
   sortTable[0x553A]=23233;   // tian
   sortTable[0x5861]=23234;   // tian
   sortTable[0x5A56]=23235;   // tian
   sortTable[0x5C47]=23236;   // tian
   sortTable[0x60BF]=23237;   // tian
   sortTable[0x6437]=23238;   // tian
   sortTable[0x666A]=23239;   // tian
   sortTable[0x6CBA]=23240;   // tian
   sortTable[0x6DDF]=23241;   // tian
   sortTable[0x6E49]=23242;   // tian
   sortTable[0x7420]=23243;   // tian
   sortTable[0x74B3]=23244;   // tian
   sortTable[0x751B]=23245;   // tian
   sortTable[0x7551]=23246;   // tian
   sortTable[0x7560]=23247;   // tian
   sortTable[0x75F6]=23248;   // tian
   sortTable[0x7DC2]=23249;   // tian
   sortTable[0x80CB]=23250;   // tian
   sortTable[0x821A]=23251;   // tian
   sortTable[0x915F]=23252;   // tian
   sortTable[0x932A]=23253;   // tian
   sortTable[0x95D0]=23254;   // tian
   sortTable[0x975D]=23255;   // tian
   sortTable[0x9DC6]=23256;   // tian
   sortTable[0x6761]=23257;   // tiao
   sortTable[0x689D]=23258;   // tiao
   sortTable[0x8DF3]=23259;   // tiao
   sortTable[0x6311]=23260;   // tiao
   sortTable[0x7C9C]=23261;   // tiao
   sortTable[0x7CF6]=23262;   // tiao
   sortTable[0x773A]=23263;   // tiao
   sortTable[0x8FE2]=23264;   // tiao
   sortTable[0x7A95]=23265;   // tiao
   sortTable[0x7B24]=23266;   // tiao
   sortTable[0x7967]=23267;   // tiao
   sortTable[0x9F86]=23268;   // tiao
   sortTable[0x4F7B]=23269;   // tiao
   sortTable[0x9AEB]=23270;   // tiao
   sortTable[0x84DA]=23271;   // tiao
   sortTable[0x8729]=23272;   // tiao
   sortTable[0x9BC8]=23273;   // tiao
   sortTable[0x9CA6]=23274;   // tiao
   sortTable[0x9797]=23275;   // tiao
   sortTable[0x84E8]=23276;   // tiao
   sortTable[0x8280]=23277;   // tiao
   sortTable[0x899C]=23278;   // tiao
   sortTable[0x7AB1]=23279;   // tiao
   sortTable[0x9C37]=23280;   // tiao
   sortTable[0x7D69]=23281;   // tiao
   sortTable[0x8A82]=23282;   // tiao
   sortTable[0x8101]=23283;   // tiao
   sortTable[0x5B25]=23284;   // tiao
   sortTable[0x5BA8]=23285;   // tiao
   sortTable[0x5CA7]=23286;   // tiao
   sortTable[0x5CB9]=23287;   // tiao
   sortTable[0x5EA3]=23288;   // tiao
   sortTable[0x604C]=23289;   // tiao
   sortTable[0x65A2]=23290;   // tiao
   sortTable[0x65EB]=23291;   // tiao
   sortTable[0x6640]=23292;   // tiao
   sortTable[0x6713]=23293;   // tiao
   sortTable[0x6A24]=23294;   // tiao
   sortTable[0x7952]=23295;   // tiao
   sortTable[0x804E]=23296;   // tiao
   sortTable[0x8414]=23297;   // tiao
   sortTable[0x87A9]=23298;   // tiao
   sortTable[0x8D92]=23299;   // tiao
   sortTable[0x92DA]=23300;   // tiao
   sortTable[0x93A5]=23301;   // tiao
   sortTable[0x9F60]=23302;   // tiao
   sortTable[0x94C1]=23303;   // tie
   sortTable[0x9435]=23304;   // tie
   sortTable[0x8D34]=23305;   // tie
   sortTable[0x8CBC]=23306;   // tie
   sortTable[0x5E16]=23307;   // tie
   sortTable[0x992E]=23308;   // tie
   sortTable[0x841C]=23309;   // tie
   sortTable[0x98FB]=23310;   // tie
   sortTable[0x9A56]=23311;   // tie
   sortTable[0x9295]=23312;   // tie
   sortTable[0x9D29]=23313;   // tie
   sortTable[0x86C8]=23314;   // tie
   sortTable[0x50E3]=23315;   // tie
   sortTable[0x546B]=23316;   // tie
   sortTable[0x6017]=23317;   // tie
   sortTable[0x8051]=23318;   // tie
   sortTable[0x92E8]=23319;   // tie
   sortTable[0x9421]=23320;   // tie
   sortTable[0x542C]=23321;   // ting
   sortTable[0x807D]=23322;   // ting
   sortTable[0x505C]=23323;   // ting
   sortTable[0x633A]=23324;   // ting
   sortTable[0x5EAD]=23325;   // ting
   sortTable[0x5385]=23326;   // ting
   sortTable[0x5EF3]=23327;   // ting
   sortTable[0x4EAD]=23328;   // ting
   sortTable[0x8247]=23329;   // ting
   sortTable[0x8713]=23330;   // ting
   sortTable[0x5EF7]=23331;   // ting
   sortTable[0x9706]=23332;   // ting
   sortTable[0x6C40]=23333;   // ting
   sortTable[0x5A77]=23334;   // ting
   sortTable[0x753A]=23335;   // ting
   sortTable[0x988B]=23336;   // ting
   sortTable[0x6883]=23337;   // ting
   sortTable[0x7D8E]=23338;   // ting
   sortTable[0x70C3]=23339;   // ting
   sortTable[0x8476]=23340;   // ting
   sortTable[0x9793]=23341;   // ting
   sortTable[0x8074]=23342;   // ting
   sortTable[0x839B]=23343;   // ting
   sortTable[0x8121]=23344;   // ting
   sortTable[0x95AE]=23345;   // ting
   sortTable[0x9F2E]=23346;   // ting
   sortTable[0x7B73]=23347;   // ting
   sortTable[0x827C]=23348;   // ting
   sortTable[0x9832]=23349;   // ting
   sortTable[0x4FB9]=23350;   // ting
   sortTable[0x539B]=23351;   // ting
   sortTable[0x5722]=23352;   // ting
   sortTable[0x5A17]=23353;   // ting
   sortTable[0x5D49]=23354;   // ting
   sortTable[0x5E81]=23355;   // ting
   sortTable[0x5EF0]=23356;   // ting
   sortTable[0x686F]=23357;   // ting
   sortTable[0x695F]=23358;   // ting
   sortTable[0x69B3]=23359;   // ting
   sortTable[0x6D8F]=23360;   // ting
   sortTable[0x6E1F]=23361;   // ting
   sortTable[0x70F4]=23362;   // ting
   sortTable[0x70F6]=23363;   // ting
   sortTable[0x73FD]=23364;   // ting
   sortTable[0x753C]=23365;   // ting
   sortTable[0x8013]=23366;   // ting
   sortTable[0x8064]=23367;   // ting
   sortTable[0x807C]=23368;   // ting
   sortTable[0x874F]=23369;   // ting
   sortTable[0x8A94]=23370;   // ting
   sortTable[0x8AEA]=23371;   // ting
   sortTable[0x9092]=23372;   // ting
   sortTable[0x540C]=23373;   // tong
   sortTable[0x901A]=23374;   // tong
   sortTable[0x7EDF]=23375;   // tong
   sortTable[0x7D71]=23376;   // tong
   sortTable[0x75DB]=23377;   // tong
   sortTable[0x94DC]=23378;   // tong
   sortTable[0x9285]=23379;   // tong
   sortTable[0x6876]=23380;   // tong
   sortTable[0x7AE5]=23381;   // tong
   sortTable[0x7B52]=23382;   // tong
   sortTable[0x6850]=23383;   // tong
   sortTable[0x4F5F]=23384;   // tong
   sortTable[0x6F7C]=23385;   // tong
   sortTable[0x50EE]=23386;   // tong
   sortTable[0x6078]=23387;   // tong
   sortTable[0x5F64]=23388;   // tong
   sortTable[0x6345]=23389;   // tong
   sortTable[0x77B3]=23390;   // tong
   sortTable[0x55F5]=23391;   // tong
   sortTable[0x916E]=23392;   // tong
   sortTable[0x4EDD]=23393;   // tong
   sortTable[0x783C]=23394;   // tong
   sortTable[0x7B69]=23395;   // tong
   sortTable[0x772E]=23396;   // tong
   sortTable[0x9C96]=23397;   // tong
   sortTable[0x7CA1]=23398;   // tong
   sortTable[0x833C]=23399;   // tong
   sortTable[0x8A77]=23400;   // tong
   sortTable[0x8855]=23401;   // tong
   sortTable[0x84EA]=23402;   // tong
   sortTable[0x79F1]=23403;   // tong
   sortTable[0x9BA6]=23404;   // tong
   sortTable[0x7D82]=23405;   // tong
   sortTable[0x8692]=23406;   // tong
   sortTable[0x9907]=23407;   // tong
   sortTable[0x52ED]=23408;   // tong
   sortTable[0x54C3]=23409;   // tong
   sortTable[0x56F2]=23410;   // tong
   sortTable[0x5CC2]=23411;   // tong
   sortTable[0x5CDD]=23412;   // tong
   sortTable[0x5E9D]=23413;   // tong
   sortTable[0x615F]=23414;   // tong
   sortTable[0x6185]=23415;   // tong
   sortTable[0x664D]=23416;   // tong
   sortTable[0x66C8]=23417;   // tong
   sortTable[0x6723]=23418;   // tong
   sortTable[0x6A0B]=23419;   // tong
   sortTable[0x6A66]=23420;   // tong
   sortTable[0x6C03]=23421;   // tong
   sortTable[0x6D75]=23422;   // tong
   sortTable[0x70B5]=23423;   // tong
   sortTable[0x70D4]=23424;   // tong
   sortTable[0x71D1]=23425;   // tong
   sortTable[0x729D]=23426;   // tong
   sortTable[0x72EA]=23427;   // tong
   sortTable[0x735E]=23428;   // tong
   sortTable[0x75CC]=23429;   // tong
   sortTable[0x81A7]=23430;   // tong
   sortTable[0x8D68]=23431;   // tong
   sortTable[0x9256]=23432;   // tong
   sortTable[0x9275]=23433;   // tong
   sortTable[0x5934]=23434;   // tou
   sortTable[0x982D]=23435;   // tou
   sortTable[0x900F]=23436;   // tou
   sortTable[0x6295]=23437;   // tou
   sortTable[0x5077]=23438;   // tou
   sortTable[0x9AB0]=23439;   // tou
   sortTable[0x94AD]=23440;   // tou
   sortTable[0x936E]=23441;   // tou
   sortTable[0x9EC8]=23442;   // tou
   sortTable[0x7DF0]=23443;   // tou
   sortTable[0x7D0F]=23444;   // tou
   sortTable[0x7D89]=23445;   // tou
   sortTable[0x98F3]=23446;   // tou
   sortTable[0x92C0]=23447;   // tou
   sortTable[0x4EA0]=23448;   // tou
   sortTable[0x5078]=23449;   // tou
   sortTable[0x59B5]=23450;   // tou
   sortTable[0x5A7E]=23451;   // tou
   sortTable[0x5AAE]=23452;   // tou
   sortTable[0x6568]=23453;   // tou
   sortTable[0x8623]=23454;   // tou
   sortTable[0x571F]=23455;   // tu
   sortTable[0x56FE]=23456;   // tu
   sortTable[0x5716]=23457;   // tu
   sortTable[0x7A81]=23458;   // tu
   sortTable[0x9014]=23459;   // tu
   sortTable[0x5F92]=23460;   // tu
   sortTable[0x6D82]=23461;   // tu
   sortTable[0x5154]=23462;   // tu
   sortTable[0x5410]=23463;   // tu
   sortTable[0x51F8]=23464;   // tu
   sortTable[0x5C60]=23465;   // tu
   sortTable[0x79C3]=23466;   // tu
   sortTable[0x79BF]=23467;   // tu
   sortTable[0x837C]=23468;   // tu
   sortTable[0x948D]=23469;   // tu
   sortTable[0x83DF]=23470;   // tu
   sortTable[0x9174]=23471;   // tu
   sortTable[0x99FC]=23472;   // tu
   sortTable[0x7A0C]=23473;   // tu
   sortTable[0x816F]=23474;   // tu
   sortTable[0x580D]=23475;   // tu
   sortTable[0x92F5]=23476;   // tu
   sortTable[0x9D9F]=23477;   // tu
   sortTable[0x83B5]=23478;   // tu
   sortTable[0x8FCC]=23479;   // tu
   sortTable[0x934E]=23480;   // tu
   sortTable[0x8456]=23481;   // tu
   sortTable[0x8DFF]=23482;   // tu
   sortTable[0x84A4]=23483;   // tu
   sortTable[0x9D4C]=23484;   // tu
   sortTable[0x9D75]=23485;   // tu
   sortTable[0x9F35]=23486;   // tu
   sortTable[0x7B61]=23487;   // tu
   sortTable[0x922F]=23488;   // tu
   sortTable[0x9DCB]=23489;   // tu
   sortTable[0x9DF5]=23490;   // tu
   sortTable[0x514E]=23491;   // tu
   sortTable[0x51C3]=23492;   // tu
   sortTable[0x550B]=23493;   // tu
   sortTable[0x56F3]=23494;   // tu
   sortTable[0x5715]=23495;   // tu
   sortTable[0x5717]=23496;   // tu
   sortTable[0x5721]=23497;   // tu
   sortTable[0x5817]=23498;   // tu
   sortTable[0x5857]=23499;   // tu
   sortTable[0x5B8A]=23500;   // tu
   sortTable[0x5CF9]=23501;   // tu
   sortTable[0x5D5E]=23502;   // tu
   sortTable[0x5D80]=23503;   // tu
   sortTable[0x5EA9]=23504;   // tu
   sortTable[0x5EDC]=23505;   // tu
   sortTable[0x6022]=23506;   // tu
   sortTable[0x6087]=23507;   // tu
   sortTable[0x6348]=23508;   // tu
   sortTable[0x6378]=23509;   // tu
   sortTable[0x63EC]=23510;   // tu
   sortTable[0x688C]=23511;   // tu
   sortTable[0x6C62]=23512;   // tu
   sortTable[0x6D8B]=23513;   // tu
   sortTable[0x6E65]=23514;   // tu
   sortTable[0x6F73]=23515;   // tu
   sortTable[0x75DC]=23516;   // tu
   sortTable[0x760F]=23517;   // tu
   sortTable[0x91F7]=23518;   // tu
   sortTable[0x999F]=23519;   // tu
   sortTable[0x9D5A]=23520;   // tu
   sortTable[0x56E2]=23521;   // tuan
   sortTable[0x5718]=23522;   // tuan
   sortTable[0x629F]=23523;   // tuan
   sortTable[0x6E4D]=23524;   // tuan
   sortTable[0x7583]=23525;   // tuan
   sortTable[0x5F56]=23526;   // tuan
   sortTable[0x8916]=23527;   // tuan
   sortTable[0x8C92]=23528;   // tuan
   sortTable[0x7CF0]=23529;   // tuan
   sortTable[0x93C4]=23530;   // tuan
   sortTable[0x9DFB]=23531;   // tuan
   sortTable[0x9DD2]=23532;   // tuan
   sortTable[0x5278]=23533;   // tuan
   sortTable[0x56E3]=23534;   // tuan
   sortTable[0x6171]=23535;   // tuan
   sortTable[0x6476]=23536;   // tuan
   sortTable[0x69EB]=23537;   // tuan
   sortTable[0x6AB2]=23538;   // tuan
   sortTable[0x6E6A]=23539;   // tuan
   sortTable[0x6F19]=23540;   // tuan
   sortTable[0x7153]=23541;   // tuan
   sortTable[0x732F]=23542;   // tuan
   sortTable[0x7BFF]=23543;   // tuan
   sortTable[0x63A8]=23544;   // tui
   sortTable[0x817F]=23545;   // tui
   sortTable[0x9000]=23546;   // tui
   sortTable[0x9893]=23547;   // tui
   sortTable[0x892A]=23548;   // tui
   sortTable[0x8715]=23549;   // tui
   sortTable[0x717A]=23550;   // tui
   sortTable[0x9B4B]=23551;   // tui
   sortTable[0x96A4]=23552;   // tui
   sortTable[0x7A68]=23553;   // tui
   sortTable[0x8E46]=23554;   // tui
   sortTable[0x84F7]=23555;   // tui
   sortTable[0x85EC]=23556;   // tui
   sortTable[0x8E6A]=23557;   // tui
   sortTable[0x983A]=23558;   // tui
   sortTable[0x86FB]=23559;   // tui
   sortTable[0x9839]=23560;   // tui
   sortTable[0x99FE]=23561;   // tui
   sortTable[0x4FBB]=23562;   // tui
   sortTable[0x4FC0]=23563;   // tui
   sortTable[0x50D3]=23564;   // tui
   sortTable[0x5A27]=23565;   // tui
   sortTable[0x5C35]=23566;   // tui
   sortTable[0x5F1A]=23567;   // tui
   sortTable[0x8608]=23568;   // tui
   sortTable[0x983D]=23569;   // tui
   sortTable[0x9ABD]=23570;   // tui
   sortTable[0x541E]=23571;   // tun
   sortTable[0x5C6F]=23572;   // tun
   sortTable[0x81C0]=23573;   // tun
   sortTable[0x8C5A]=23574;   // tun
   sortTable[0x9968]=23575;   // tun
   sortTable[0x66BE]=23576;   // tun
   sortTable[0x9C80]=23577;   // tun
   sortTable[0x9B68]=23578;   // tun
   sortTable[0x6C3D]=23579;   // tun
   sortTable[0x829A]=23580;   // tun
   sortTable[0x98E9]=23581;   // tun
   sortTable[0x81CB]=23582;   // tun
   sortTable[0x8ED8]=23583;   // tun
   sortTable[0x9715]=23584;   // tun
   sortTable[0x5451]=23585;   // tun
   sortTable[0x554D]=23586;   // tun
   sortTable[0x564B]=23587;   // tun
   sortTable[0x5749]=23588;   // tun
   sortTable[0x5FF3]=23589;   // tun
   sortTable[0x65FD]=23590;   // tun
   sortTable[0x671C]=23591;   // tun
   sortTable[0x6D92]=23592;   // tun
   sortTable[0x711E]=23593;   // tun
   sortTable[0x757D]=23594;   // tun
   sortTable[0x8C58]=23595;   // tun
   sortTable[0x9ED7]=23596;   // tun
   sortTable[0x8131]=23597;   // tuo
   sortTable[0x812B]=23598;   // tuo
   sortTable[0x62D6]=23599;   // tuo
   sortTable[0x9A7C]=23600;   // tuo
   sortTable[0x99DD]=23601;   // tuo
   sortTable[0x6258]=23602;   // tuo
   sortTable[0x59A5]=23603;   // tuo
   sortTable[0x9A6E]=23604;   // tuo
   sortTable[0x99B1]=23605;   // tuo
   sortTable[0x553E]=23606;   // tuo
   sortTable[0x692D]=23607;   // tuo
   sortTable[0x6A62]=23608;   // tuo
   sortTable[0x9640]=23609;   // tuo
   sortTable[0x6A50]=23610;   // tuo
   sortTable[0x9E35]=23611;   // tuo
   sortTable[0x4F57]=23612;   // tuo
   sortTable[0x6CB1]=23613;   // tuo
   sortTable[0x8DCE]=23614;   // tuo
   sortTable[0x4E47]=23615;   // tuo
   sortTable[0x9F0D]=23616;   // tuo
   sortTable[0x7823]=23617;   // tuo
   sortTable[0x8BAC]=23618;   // tuo
   sortTable[0x5EB9]=23619;   // tuo
   sortTable[0x67DD]=23620;   // tuo
   sortTable[0x7BA8]=23621;   // tuo
   sortTable[0x5768]=23622;   // tuo
   sortTable[0x9F09]=23623;   // tuo
   sortTable[0x9641]=23624;   // tuo
   sortTable[0x9161]=23625;   // tuo
   sortTable[0x6CB2]=23626;   // tuo
   sortTable[0x8DC5]=23627;   // tuo
   sortTable[0x88A5]=23628;   // tuo
   sortTable[0x9B80]=23629;   // tuo
   sortTable[0x8A17]=23630;   // tuo
   sortTable[0x9966]=23631;   // tuo
   sortTable[0x8FF1]=23632;   // tuo
   sortTable[0x841A]=23633;   // tuo
   sortTable[0x9A52]=23634;   // tuo
   sortTable[0x9B60]=23635;   // tuo
   sortTable[0x8600]=23636;   // tuo
   sortTable[0x8889]=23637;   // tuo
   sortTable[0x9B0C]=23638;   // tuo
   sortTable[0x7D3D]=23639;   // tuo
   sortTable[0x99DE]=23640;   // tuo
   sortTable[0x7824]=23641;   // tuo
   sortTable[0x7C5C]=23642;   // tuo
   sortTable[0x98E5]=23643;   // tuo
   sortTable[0x9A5D]=23644;   // tuo
   sortTable[0x9C16]=23645;   // tuo
   sortTable[0x9F27]=23646;   // tuo
   sortTable[0x4EDB]=23647;   // tuo
   sortTable[0x4F82]=23648;   // tuo
   sortTable[0x5483]=23649;   // tuo
   sortTable[0x5836]=23650;   // tuo
   sortTable[0x5AA0]=23651;   // tuo
   sortTable[0x5AF7]=23652;   // tuo
   sortTable[0x5CAE]=23653;   // tuo
   sortTable[0x5F75]=23654;   // tuo
   sortTable[0x6261]=23655;   // tuo
   sortTable[0x62D5]=23656;   // tuo
   sortTable[0x6329]=23657;   // tuo
   sortTable[0x635D]=23658;   // tuo
   sortTable[0x6754]=23659;   // tuo
   sortTable[0x6955]=23660;   // tuo
   sortTable[0x69D6]=23661;   // tuo
   sortTable[0x6BE4]=23662;   // tuo
   sortTable[0x6BFB]=23663;   // tuo
   sortTable[0x6C51]=23664;   // tuo
   sortTable[0x6CB0]=23665;   // tuo
   sortTable[0x6DB6]=23666;   // tuo
   sortTable[0x72CF]=23667;   // tuo
   sortTable[0x78A2]=23668;   // tuo
   sortTable[0x838C]=23669;   // tuo
   sortTable[0x99C4]=23670;   // tuo
   sortTable[0x9A28]=23671;   // tuo
   sortTable[0x9D15]=23672;   // tuo
   sortTable[0x9D4E]=23673;   // tuo
   sortTable[0x6316]=24576;   // wa
   sortTable[0x54C7]=24577;   // wa
   sortTable[0x86D9]=24578;   // wa
   sortTable[0x74E6]=24579;   // wa
   sortTable[0x5A03]=24580;   // wa
   sortTable[0x889C]=24581;   // wa
   sortTable[0x896A]=24582;   // wa
   sortTable[0x6D3C]=24583;   // wa
   sortTable[0x5A32]=24584;   // wa
   sortTable[0x4F64]=24585;   // wa
   sortTable[0x97C8]=24586;   // wa
   sortTable[0x7A8A]=24587;   // wa
   sortTable[0x817D]=24588;   // wa
   sortTable[0x7A75]=24589;   // wa
   sortTable[0x7AAA]=24590;   // wa
   sortTable[0x90B7]=24591;   // wa
   sortTable[0x52B8]=24592;   // wa
   sortTable[0x5493]=24593;   // wa
   sortTable[0x55D7]=24594;   // wa
   sortTable[0x55E2]=24595;   // wa
   sortTable[0x5AA7]=24596;   // wa
   sortTable[0x5C72]=24597;   // wa
   sortTable[0x6432]=24598;   // wa
   sortTable[0x6528]=24599;   // wa
   sortTable[0x6E9B]=24600;   // wa
   sortTable[0x6F25]=24601;   // wa
   sortTable[0x74F2]=24602;   // wa
   sortTable[0x7556]=24603;   // wa
   sortTable[0x8049]=24604;   // wa
   sortTable[0x8183]=24605;   // wa
   sortTable[0x97E4]=24606;   // wa
   sortTable[0x9F03]=24607;   // wa
   sortTable[0x5916]=24608;   // wai
   sortTable[0x6B6A]=24609;   // wai
   sortTable[0x5D34]=24610;   // wai
   sortTable[0x9861]=24611;   // wai
   sortTable[0x7AF5]=24612;   // wai
   sortTable[0x558E]=24613;   // wai
   sortTable[0x591E]=24614;   // wai
   sortTable[0x5B8C]=24615;   // wan
   sortTable[0x4E07]=24616;   // wan
   sortTable[0x842C]=24617;   // wan
   sortTable[0x665A]=24618;   // wan
   sortTable[0x7897]=24619;   // wan
   sortTable[0x73A9]=24620;   // wan
   sortTable[0x5F2F]=24621;   // wan
   sortTable[0x5F4E]=24622;   // wan
   sortTable[0x633D]=24623;   // wan
   sortTable[0x987D]=24624;   // wan
   sortTable[0x9811]=24625;   // wan
   sortTable[0x6E7E]=24626;   // wan
   sortTable[0x7063]=24627;   // wan
   sortTable[0x60CB]=24628;   // wan
   sortTable[0x4E38]=24629;   // wan
   sortTable[0x8C4C]=24630;   // wan
   sortTable[0x5B9B]=24631;   // wan
   sortTable[0x5A49]=24632;   // wan
   sortTable[0x8155]=24633;   // wan
   sortTable[0x7EA8]=24634;   // wan
   sortTable[0x7EFE]=24635;   // wan
   sortTable[0x7696]=24636;   // wan
   sortTable[0x7579]=24637;   // wan
   sortTable[0x8284]=24638;   // wan
   sortTable[0x742C]=24639;   // wan
   sortTable[0x83C0]=24640;   // wan
   sortTable[0x873F]=24641;   // wan
   sortTable[0x525C]=24642;   // wan
   sortTable[0x70F7]=24643;   // wan
   sortTable[0x8F13]=24644;   // wan
   sortTable[0x8118]=24645;   // wan
   sortTable[0x7FEB]=24646;   // wan
   sortTable[0x7755]=24647;   // wan
   sortTable[0x8E20]=24648;   // wan
   sortTable[0x92C4]=24649;   // wan
   sortTable[0x8115]=24650;   // wan
   sortTable[0x8CA6]=24651;   // wan
   sortTable[0x8F10]=24652;   // wan
   sortTable[0x909C]=24653;   // wan
   sortTable[0x93AB]=24654;   // wan
   sortTable[0x76CC]=24655;   // wan
   sortTable[0x5007]=24656;   // wan
   sortTable[0x5213]=24657;   // wan
   sortTable[0x534D]=24658;   // wan
   sortTable[0x5350]=24659;   // wan
   sortTable[0x550D]=24660;   // wan
   sortTable[0x57E6]=24661;   // wan
   sortTable[0x5846]=24662;   // wan
   sortTable[0x58EA]=24663;   // wan
   sortTable[0x59A7]=24664;   // wan
   sortTable[0x5A60]=24665;   // wan
   sortTable[0x5C8F]=24666;   // wan
   sortTable[0x5E35]=24667;   // wan
   sortTable[0x5FE8]=24668;   // wan
   sortTable[0x628F]=24669;   // wan
   sortTable[0x6356]=24670;   // wan
   sortTable[0x6365]=24671;   // wan
   sortTable[0x6669]=24672;   // wan
   sortTable[0x667C]=24673;   // wan
   sortTable[0x6764]=24674;   // wan
   sortTable[0x689A]=24675;   // wan
   sortTable[0x6900]=24676;   // wan
   sortTable[0x6C4D]=24677;   // wan
   sortTable[0x6F6B]=24678;   // wan
   sortTable[0x6FAB]=24679;   // wan
   sortTable[0x7413]=24680;   // wan
   sortTable[0x7B02]=24681;   // wan
   sortTable[0x7D08]=24682;   // wan
   sortTable[0x7DA9]=24683;   // wan
   sortTable[0x7DB0]=24684;   // wan
   sortTable[0x8416]=24685;   // wan
   sortTable[0x858D]=24686;   // wan
   sortTable[0x87C3]=24687;   // wan
   sortTable[0x8D03]=24688;   // wan
   sortTable[0x8D0E]=24689;   // wan
   sortTable[0x92D4]=24690;   // wan
   sortTable[0x933D]=24691;   // wan
   sortTable[0x671B]=24692;   // wang
   sortTable[0x5F80]=24693;   // wang
   sortTable[0x5FD8]=24694;   // wang
   sortTable[0x738B]=24695;   // wang
   sortTable[0x7F51]=24696;   // wang
   sortTable[0x7DB2]=24697;   // wang
   sortTable[0x4EA1]=24698;   // wang
   sortTable[0x5984]=24699;   // wang
   sortTable[0x65FA]=24700;   // wang
   sortTable[0x6789]=24701;   // wang
   sortTable[0x6C6A]=24702;   // wang
   sortTable[0x7F54]=24703;   // wang
   sortTable[0x60D8]=24704;   // wang
   sortTable[0x9B4D]=24705;   // wang
   sortTable[0x8F8B]=24706;   // wang
   sortTable[0x8744]=24707;   // wang
   sortTable[0x8FCB]=24708;   // wang
   sortTable[0x83F5]=24709;   // wang
   sortTable[0x8AB7]=24710;   // wang
   sortTable[0x8F1E]=24711;   // wang
   sortTable[0x76F3]=24712;   // wang
   sortTable[0x83A3]=24713;   // wang
   sortTable[0x86E7]=24714;   // wang
   sortTable[0x4EBE]=24715;   // wang
   sortTable[0x4EFC]=24716;   // wang
   sortTable[0x5166]=24717;   // wang
   sortTable[0x5C23]=24718;   // wang
   sortTable[0x5C29]=24719;   // wang
   sortTable[0x5C2A]=24720;   // wang
   sortTable[0x5C2B]=24721;   // wang
   sortTable[0x5F7A]=24722;   // wang
   sortTable[0x5F83]=24723;   // wang
   sortTable[0x5F8D]=24724;   // wang
   sortTable[0x6680]=24725;   // wang
   sortTable[0x6722]=24726;   // wang
   sortTable[0x68E2]=24727;   // wang
   sortTable[0x7007]=24728;   // wang
   sortTable[0x7139]=24729;   // wang
   sortTable[0x7F52]=24730;   // wang
   sortTable[0x869F]=24731;   // wang
   sortTable[0x4E3A]=24732;   // wei
   sortTable[0x70BA]=24733;   // wei
   sortTable[0x4F4D]=24734;   // wei
   sortTable[0x59D4]=24735;   // wei
   sortTable[0x56F4]=24736;   // wei
   sortTable[0x570D]=24737;   // wei
   sortTable[0x4F1F]=24738;   // wei
   sortTable[0x5049]=24739;   // wei
   sortTable[0x5FAE]=24740;   // wei
   sortTable[0x7EF4]=24741;   // wei
   sortTable[0x7DAD]=24742;   // wei
   sortTable[0x536B]=24743;   // wei
   sortTable[0x885B]=24744;   // wei
   sortTable[0x672A]=24745;   // wei
   sortTable[0x5371]=24746;   // wei
   sortTable[0x5473]=24747;   // wei
   sortTable[0x552F]=24748;   // wei
   sortTable[0x5A01]=24749;   // wei
   sortTable[0x8C13]=24750;   // wei
   sortTable[0x8B02]=24751;   // wei
   sortTable[0x5C3E]=24752;   // wei
   sortTable[0x5582]=24753;   // wei
   sortTable[0x60DF]=24754;   // wei
   sortTable[0x8FDD]=24755;   // wei
   sortTable[0x9055]=24756;   // wei
   sortTable[0x80C3]=24757;   // wei
   sortTable[0x6170]=24758;   // wei
   sortTable[0x4F2A]=24759;   // wei
   sortTable[0x507D]=24760;   // wei
   sortTable[0x7EAC]=24761;   // wei
   sortTable[0x7DEF]=24762;   // wei
   sortTable[0x82C7]=24763;   // wei
   sortTable[0x8466]=24764;   // wei
   sortTable[0x754F]=24765;   // wei
   sortTable[0x504E]=24766;   // wei
   sortTable[0x6845]=24767;   // wei
   sortTable[0x732C]=24768;   // wei
   sortTable[0x875F]=24769;   // wei
   sortTable[0x5DCD]=24770;   // wei
   sortTable[0x9B4F]=24771;   // wei
   sortTable[0x5C09]=24772;   // wei
   sortTable[0x97E6]=24773;   // wei
   sortTable[0x6E2D]=24774;   // wei
   sortTable[0x5E37]=24775;   // wei
   sortTable[0x851A]=24776;   // wei
   sortTable[0x840E]=24777;   // wei
   sortTable[0x8587]=24778;   // wei
   sortTable[0x95F1]=24779;   // wei
   sortTable[0x7168]=24780;   // wei
   sortTable[0x8BFF]=24781;   // wei
   sortTable[0x9036]=24782;   // wei
   sortTable[0x7325]=24783;   // wei
   sortTable[0x709C]=24784;   // wei
   sortTable[0x5A13]=24785;   // wei
   sortTable[0x5729]=24786;   // wei
   sortTable[0x73AE]=24787;   // wei
   sortTable[0x5E0F]=24788;   // wei
   sortTable[0x6DA0]=24789;   // wei
   sortTable[0x5D6C]=24790;   // wei
   sortTable[0x8249]=24791;   // wei
   sortTable[0x97EA]=24792;   // wei
   sortTable[0x6F4D]=24793;   // wei
   sortTable[0x75FF]=24794;   // wei
   sortTable[0x9820]=24795;   // wei
   sortTable[0x6D27]=24796;   // wei
   sortTable[0x9C94]=24797;   // wei
   sortTable[0x9688]=24798;   // wei
   sortTable[0x8473]=24799;   // wei
   sortTable[0x9CDA]=24800;   // wei
   sortTable[0x7859]=24801;   // wei
   sortTable[0x97E1]=24802;   // wei
   sortTable[0x848D]=24803;   // wei
   sortTable[0x85B3]=24804;   // wei
   sortTable[0x9927]=24805;   // wei
   sortTable[0x9728]=24806;   // wei
   sortTable[0x9CC2]=24807;   // wei
   sortTable[0x9AAB]=24808;   // wei
   sortTable[0x9BA0]=24809;   // wei
   sortTable[0x873C]=24810;   // wei
   sortTable[0x8A74]=24811;   // wei
   sortTable[0x6CA9]=24812;   // wei
   sortTable[0x7DED]=24813;   // wei
   sortTable[0x9687]=24814;   // wei
   sortTable[0x934F]=24815;   // wei
   sortTable[0x912C]=24816;   // wei
   sortTable[0x78D1]=24817;   // wei
   sortTable[0x853F]=24818;   // wei
   sortTable[0x7F7B]=24819;   // wei
   sortTable[0x82FF]=24820;   // wei
   sortTable[0x87B1]=24821;   // wei
   sortTable[0x8F4A]=24822;   // wei
   sortTable[0x78A8]=24823;   // wei
   sortTable[0x8468]=24824;   // wei
   sortTable[0x78C8]=24825;   // wei
   sortTable[0x84F6]=24826;   // wei
   sortTable[0x85EF]=24827;   // wei
   sortTable[0x8624]=24828;   // wei
   sortTable[0x8ECE]=24829;   // wei
   sortTable[0x8B8F]=24830;   // wei
   sortTable[0x784A]=24831;   // wei
   sortTable[0x83CB]=24832;   // wei
   sortTable[0x8AC9]=24833;   // wei
   sortTable[0x93CF]=24834;   // wei
   sortTable[0x9935]=24835;   // wei
   sortTable[0x829B]=24836;   // wei
   sortTable[0x8732]=24837;   // wei
   sortTable[0x95C8]=24838;   // wei
   sortTable[0x97D9]=24839;   // wei
   sortTable[0x9956]=24840;   // wei
   sortTable[0x9AA9]=24841;   // wei
   sortTable[0x9BAA]=24842;   // wei
   sortTable[0x893D]=24843;   // wei
   sortTable[0x8E97]=24844;   // wei
   sortTable[0x8E9B]=24845;   // wei
   sortTable[0x98B9]=24846;   // wei
   sortTable[0x9AAA]=24847;   // wei
   sortTable[0x9B87]=24848;   // wei
   sortTable[0x50DE]=24849;   // wei
   sortTable[0x5130]=24850;   // wei
   sortTable[0x5383]=24851;   // wei
   sortTable[0x53DE]=24852;   // wei
   sortTable[0x55A1]=24853;   // wei
   sortTable[0x55B4]=24854;   // wei
   sortTable[0x56D7]=24855;   // wei
   sortTable[0x589B]=24856;   // wei
   sortTable[0x58DD]=24857;   // wei
   sortTable[0x5A81]=24858;   // wei
   sortTable[0x5A99]=24859;   // wei
   sortTable[0x5AA6]=24860;   // wei
   sortTable[0x5BEA]=24861;   // wei
   sortTable[0x5C57]=24862;   // wei
   sortTable[0x5CD7]=24863;   // wei
   sortTable[0x5CDE]=24864;   // wei
   sortTable[0x5D23]=24865;   // wei
   sortTable[0x5D54]=24866;   // wei
   sortTable[0x5DB6]=24867;   // wei
   sortTable[0x5E43]=24868;   // wei
   sortTable[0x5FAB]=24869;   // wei
   sortTable[0x6104]=24870;   // wei
   sortTable[0x6107]=24871;   // wei
   sortTable[0x61C0]=24872;   // wei
   sortTable[0x6364]=24873;   // wei
   sortTable[0x63CB]=24874;   // wei
   sortTable[0x63FB]=24875;   // wei
   sortTable[0x64B1]=24876;   // wei
   sortTable[0x6596]=24877;   // wei
   sortTable[0x6690]=24878;   // wei
   sortTable[0x68B6]=24879;   // wei
   sortTable[0x6932]=24880;   // wei
   sortTable[0x6933]=24881;   // wei
   sortTable[0x6972]=24882;   // wei
   sortTable[0x6B08]=24883;   // wei
   sortTable[0x6D08]=24884;   // wei
   sortTable[0x6D58]=24885;   // wei
   sortTable[0x6E28]=24886;   // wei
   sortTable[0x6E4B]=24887;   // wei
   sortTable[0x6E88]=24888;   // wei
   sortTable[0x6EA6]=24889;   // wei
   sortTable[0x6F59]=24890;   // wei
   sortTable[0x6F7F]=24891;   // wei
   sortTable[0x6FF0]=24892;   // wei
   sortTable[0x6FFB]=24893;   // wei
   sortTable[0x7022]=24894;   // wei
   sortTable[0x70D3]=24895;   // wei
   sortTable[0x7140]=24896;   // wei
   sortTable[0x7152]=24897;   // wei
   sortTable[0x715F]=24898;   // wei
   sortTable[0x71AD]=24899;   // wei
   sortTable[0x71F0]=24900;   // wei
   sortTable[0x7232]=24901;   // wei
   sortTable[0x729A]=24902;   // wei
   sortTable[0x72A9]=24903;   // wei
   sortTable[0x741F]=24904;   // wei
   sortTable[0x744B]=24905;   // wei
   sortTable[0x748F]=24906;   // wei
   sortTable[0x75CF]=24907;   // wei
   sortTable[0x7653]=24908;   // wei
   sortTable[0x7E05]=24909;   // wei
   sortTable[0x8172]=24910;   // wei
   sortTable[0x8371]=24911;   // wei
   sortTable[0x8636]=24912;   // wei
   sortTable[0x875B]=24913;   // wei
   sortTable[0x885E]=24914;   // wei
   sortTable[0x89A3]=24915;   // wei
   sortTable[0x89B9]=24916;   // wei
   sortTable[0x8B86]=24917;   // wei
   sortTable[0x8E13]=24918;   // wei
   sortTable[0x9180]=24919;   // wei
   sortTable[0x9361]=24920;   // wei
   sortTable[0x973A]=24921;   // wei
   sortTable[0x97CB]=24922;   // wei
   sortTable[0x97D1]=24923;   // wei
   sortTable[0x9C03]=24924;   // wei
   sortTable[0x9C04]=24925;   // wei
   sortTable[0x95EE]=24926;   // wen
   sortTable[0x554F]=24927;   // wen
   sortTable[0x6587]=24928;   // wen
   sortTable[0x6E29]=24929;   // wen
   sortTable[0x6EAB]=24930;   // wen
   sortTable[0x7A33]=24931;   // wen
   sortTable[0x7A69]=24932;   // wen
   sortTable[0x95FB]=24933;   // wen
   sortTable[0x805E]=24934;   // wen
   sortTable[0x7EB9]=24935;   // wen
   sortTable[0x7D0B]=24936;   // wen
   sortTable[0x868A]=24937;   // wen
   sortTable[0x543B]=24938;   // wen
   sortTable[0x96EF]=24939;   // wen
   sortTable[0x7D0A]=24940;   // wen
   sortTable[0x761F]=24941;   // wen
   sortTable[0x6C76]=24942;   // wen
   sortTable[0x520E]=24943;   // wen
   sortTable[0x960C]=24944;   // wen
   sortTable[0x74BA]=24945;   // wen
   sortTable[0x9850]=24946;   // wen
   sortTable[0x8F92]=24947;   // wen
   sortTable[0x8570]=24948;   // wen
   sortTable[0x739F]=24949;   // wen
   sortTable[0x95BA]=24950;   // wen
   sortTable[0x95C5]=24951;   // wen
   sortTable[0x8F40]=24952;   // wen
   sortTable[0x9CC1]=24953;   // wen
   sortTable[0x8117]=24954;   // wen
   sortTable[0x9B70]=24955;   // wen
   sortTable[0x9F24]=24956;   // wen
   sortTable[0x97B0]=24957;   // wen
   sortTable[0x95BF]=24958;   // wen
   sortTable[0x82A0]=24959;   // wen
   sortTable[0x8689]=24960;   // wen
   sortTable[0x87A1]=24961;   // wen
   sortTable[0x87C1]=24962;   // wen
   sortTable[0x922B]=24963;   // wen
   sortTable[0x95E6]=24964;   // wen
   sortTable[0x9CFC]=24965;   // wen
   sortTable[0x5301]=24966;   // wen
   sortTable[0x545A]=24967;   // wen
   sortTable[0x5461]=24968;   // wen
   sortTable[0x586D]=24969;   // wen
   sortTable[0x598F]=24970;   // wen
   sortTable[0x5F63]=24971;   // wen
   sortTable[0x5FDF]=24972;   // wen
   sortTable[0x6286]=24973;   // wen
   sortTable[0x63FE]=24974;   // wen
   sortTable[0x6435]=24975;   // wen
   sortTable[0x6637]=24976;   // wen
   sortTable[0x687D]=24977;   // wen
   sortTable[0x6985]=24978;   // wen
   sortTable[0x6B9F]=24979;   // wen
   sortTable[0x6E02]=24980;   // wen
   sortTable[0x7086]=24981;   // wen
   sortTable[0x73F3]=24982;   // wen
   sortTable[0x741D]=24983;   // wen
   sortTable[0x7465]=24984;   // wen
   sortTable[0x7612]=24985;   // wen
   sortTable[0x7A4F]=24986;   // wen
   sortTable[0x80B3]=24987;   // wen
   sortTable[0x83AC]=24988;   // wen
   sortTable[0x8C71]=24989;   // wen
   sortTable[0x8F3C]=24990;   // wen
   sortTable[0x93BE]=24991;   // wen
   sortTable[0x99BC]=24992;   // wen
   sortTable[0x9C1B]=24993;   // wen
   sortTable[0x9C2E]=24994;   // wen
   sortTable[0x9D0D]=24995;   // wen
   sortTable[0x55E1]=24996;   // weng
   sortTable[0x7FC1]=24997;   // weng
   sortTable[0x74EE]=24998;   // weng
   sortTable[0x8579]=24999;   // weng
   sortTable[0x84CA]=25000;   // weng
   sortTable[0x806C]=25001;   // weng
   sortTable[0x8789]=25002;   // weng
   sortTable[0x9E5F]=25003;   // weng
   sortTable[0x9F46]=25004;   // weng
   sortTable[0x7F4B]=25005;   // weng
   sortTable[0x7788]=25006;   // weng
   sortTable[0x9DB2]=25007;   // weng
   sortTable[0x52DC]=25008;   // weng
   sortTable[0x5855]=25009;   // weng
   sortTable[0x5963]=25010;   // weng
   sortTable[0x5D61]=25011;   // weng
   sortTable[0x66A1]=25012;   // weng
   sortTable[0x6EC3]=25013;   // weng
   sortTable[0x7515]=25014;   // weng
   sortTable[0x9393]=25015;   // weng
   sortTable[0x6211]=25016;   // wo
   sortTable[0x63E1]=25017;   // wo
   sortTable[0x7A9D]=25018;   // wo
   sortTable[0x7AA9]=25019;   // wo
   sortTable[0x5367]=25020;   // wo
   sortTable[0x81E5]=25021;   // wo
   sortTable[0x6DA1]=25022;   // wo
   sortTable[0x6E26]=25023;   // wo
   sortTable[0x6C83]=25024;   // wo
   sortTable[0x8717]=25025;   // wo
   sortTable[0x8778]=25026;   // wo
   sortTable[0x502D]=25027;   // wo
   sortTable[0x6E25]=25028;   // wo
   sortTable[0x65A1]=25029;   // wo
   sortTable[0x631D]=25030;   // wo
   sortTable[0x5E44]=25031;   // wo
   sortTable[0x786A]=25032;   // wo
   sortTable[0x9F8C]=25033;   // wo
   sortTable[0x809F]=25034;   // wo
   sortTable[0x83B4]=25035;   // wo
   sortTable[0x96D8]=25036;   // wo
   sortTable[0x81D2]=25037;   // wo
   sortTable[0x8435]=25038;   // wo
   sortTable[0x8E12]=25039;   // wo
   sortTable[0x4EF4]=25040;   // wo
   sortTable[0x5053]=25041;   // wo
   sortTable[0x5529]=25042;   // wo
   sortTable[0x5A50]=25043;   // wo
   sortTable[0x5A89]=25044;   // wo
   sortTable[0x6370]=25045;   // wo
   sortTable[0x637E]=25046;   // wo
   sortTable[0x64BE]=25047;   // wo
   sortTable[0x64ED]=25048;   // wo
   sortTable[0x6782]=25049;   // wo
   sortTable[0x6943]=25050;   // wo
   sortTable[0x6DB4]=25051;   // wo
   sortTable[0x6DB9]=25052;   // wo
   sortTable[0x6FE3]=25053;   // wo
   sortTable[0x7125]=25054;   // wo
   sortTable[0x7327]=25055;   // wo
   sortTable[0x74C1]=25056;   // wo
   sortTable[0x7783]=25057;   // wo
   sortTable[0x815B]=25058;   // wo
   sortTable[0x9F77]=25059;   // wo
   sortTable[0x7269]=25060;   // wu
   sortTable[0x65E0]=25061;   // wu
   sortTable[0x7121]=25062;   // wu
   sortTable[0x4E94]=25063;   // wu
   sortTable[0x5C4B]=25064;   // wu
   sortTable[0x8BEF]=25065;   // wu
   sortTable[0x8AA4]=25066;   // wu
   sortTable[0x52A1]=25067;   // wu
   sortTable[0x52D9]=25068;   // wu
   sortTable[0x6B66]=25069;   // wu
   sortTable[0x5348]=25070;   // wu
   sortTable[0x4F0D]=25071;   // wu
   sortTable[0x821E]=25072;   // wu
   sortTable[0x6C61]=25073;   // wu
   sortTable[0x96FE]=25074;   // wu
   sortTable[0x9727]=25075;   // wu
   sortTable[0x4E4C]=25076;   // wu
   sortTable[0x609F]=25077;   // wu
   sortTable[0x70CF]=25078;   // wu
   sortTable[0x4FAE]=25079;   // wu
   sortTable[0x6342]=25080;   // wu
   sortTable[0x8BEC]=25081;   // wu
   sortTable[0x8AA3]=25082;   // wu
   sortTable[0x545C]=25083;   // wu
   sortTable[0x8708]=25084;   // wu
   sortTable[0x55DA]=25085;   // wu
   sortTable[0x543E]=25086;   // wu
   sortTable[0x5434]=25087;   // wu
   sortTable[0x620A]=25088;   // wu
   sortTable[0x52FF]=25089;   // wu
   sortTable[0x5DEB]=25090;   // wu
   sortTable[0x5140]=25091;   // wu
   sortTable[0x6BCB]=25092;   // wu
   sortTable[0x6664]=25093;   // wu
   sortTable[0x68A7]=25094;   // wu
   sortTable[0x829C]=25095;   // wu
   sortTable[0x575E]=25096;   // wu
   sortTable[0x5FE4]=25097;   // wu
   sortTable[0x9E49]=25098;   // wu
   sortTable[0x5514]=25099;   // wu
   sortTable[0x5E91]=25100;   // wu
   sortTable[0x5BE4]=25101;   // wu
   sortTable[0x5A7A]=25102;   // wu
   sortTable[0x92C8]=25103;   // wu
   sortTable[0x6D6F]=25104;   // wu
   sortTable[0x59A9]=25105;   // wu
   sortTable[0x7110]=25106;   // wu
   sortTable[0x9E5C]=25107;   // wu
   sortTable[0x90AC]=25108;   // wu
   sortTable[0x9A9B]=25109;   // wu
   sortTable[0x8FD5]=25110;   // wu
   sortTable[0x674C]=25111;   // wu
   sortTable[0x4EF5]=25112;   // wu
   sortTable[0x9622]=25113;   // wu
   sortTable[0x82B4]=25114;   // wu
   sortTable[0x94A8]=25115;   // wu
   sortTable[0x6003]=25116;   // wu
   sortTable[0x572C]=25117;   // wu
   sortTable[0x727E]=25118;   // wu
   sortTable[0x9C1E]=25119;   // wu
   sortTable[0x9F2F]=25120;   // wu
   sortTable[0x7991]=25121;   // wu
   sortTable[0x94FB]=25122;   // wu
   sortTable[0x90DA]=25123;   // wu
   sortTable[0x96FA]=25124;   // wu
   sortTable[0x75E6]=25125;   // wu
   sortTable[0x971A]=25126;   // wu
   sortTable[0x8323]=25127;   // wu
   sortTable[0x9770]=25128;   // wu
   sortTable[0x7894]=25129;   // wu
   sortTable[0x9E40]=25130;   // wu
   sortTable[0x7966]=25131;   // wu
   sortTable[0x907B]=25132;   // wu
   sortTable[0x856A]=25133;   // wu
   sortTable[0x9D2E]=25134;   // wu
   sortTable[0x8790]=25135;   // wu
   sortTable[0x7AB9]=25136;   // wu
   sortTable[0x9696]=25137;   // wu
   sortTable[0x8381]=25138;   // wu
   sortTable[0x8601]=25139;   // wu
   sortTable[0x87F1]=25140;   // wu
   sortTable[0x933B]=25141;   // wu
   sortTable[0x9F3F]=25142;   // wu
   sortTable[0x77F9]=25143;   // wu
   sortTable[0x7A8F]=25144;   // wu
   sortTable[0x901C]=25145;   // wu
   sortTable[0x9BC3]=25146;   // wu
   sortTable[0x9DA9]=25147;   // wu
   sortTable[0x9F40]=25148;   // wu
   sortTable[0x4E44]=25149;   // wu
   sortTable[0x4F06]=25150;   // wu
   sortTable[0x4FC9]=25151;   // wu
   sortTable[0x5035]=25152;   // wu
   sortTable[0x511B]=25153;   // wu
   sortTable[0x526D]=25154;   // wu
   sortTable[0x537C]=25155;   // wu
   sortTable[0x5433]=25156;   // wu
   sortTable[0x5449]=25157;   // wu
   sortTable[0x554E]=25158;   // wu
   sortTable[0x5862]=25159;   // wu
   sortTable[0x5966]=25160;   // wu
   sortTable[0x5A2A]=25161;   // wu
   sortTable[0x5A2C]=25162;   // wu
   sortTable[0x5AF5]=25163;   // wu
   sortTable[0x5C7C]=25164;   // wu
   sortTable[0x5C89]=25165;   // wu
   sortTable[0x5D4D]=25166;   // wu
   sortTable[0x5D68]=25167;   // wu
   sortTable[0x5EE1]=25168;   // wu
   sortTable[0x5F19]=25169;   // wu
   sortTable[0x5FE2]=25170;   // wu
   sortTable[0x609E]=25171;   // wu
   sortTable[0x60AE]=25172;   // wu
   sortTable[0x61AE]=25173;   // wu
   sortTable[0x6264]=25174;   // wu
   sortTable[0x6440]=25175;   // wu
   sortTable[0x6544]=25176;   // wu
   sortTable[0x65FF]=25177;   // wu
   sortTable[0x6747]=25178;   // wu
   sortTable[0x6A46]=25179;   // wu
   sortTable[0x6B4D]=25180;   // wu
   sortTable[0x6C59]=25181;   // wu
   sortTable[0x6C5A]=25182;   // wu
   sortTable[0x6D16]=25183;   // wu
   sortTable[0x6D3F]=25184;   // wu
   sortTable[0x6EA9]=25185;   // wu
   sortTable[0x6F55]=25186;   // wu
   sortTable[0x7183]=25187;   // wu
   sortTable[0x7193]=25188;   // wu
   sortTable[0x739D]=25189;   // wu
   sortTable[0x73F7]=25190;   // wu
   sortTable[0x73F8]=25191;   // wu
   sortTable[0x7466]=25192;   // wu
   sortTable[0x7491]=25193;   // wu
   sortTable[0x7512]=25194;   // wu
   sortTable[0x7BBC]=25195;   // wu
   sortTable[0x7C85]=25196;   // wu
   sortTable[0x8A88]=25197;   // wu
   sortTable[0x8B55]=25198;   // wu
   sortTable[0x8E8C]=25199;   // wu
   sortTable[0x9114]=25200;   // wu
   sortTable[0x93A2]=25201;   // wu
   sortTable[0x9A16]=25202;   // wu
   sortTable[0x9D50]=25203;   // wu
   sortTable[0x9D61]=25204;   // wu
   sortTable[0x9DE1]=25205;   // wu
   sortTable[0x897F]=25600;   // xi
   sortTable[0x7EC6]=25601;   // xi
   sortTable[0x7D30]=25602;   // xi
   sortTable[0x4E60]=25603;   // xi
   sortTable[0x7FD2]=25604;   // xi
   sortTable[0x5E2D]=25605;   // xi
   sortTable[0x7CFB]=25606;   // xi
   sortTable[0x559C]=25607;   // xi
   sortTable[0x606F]=25608;   // xi
   sortTable[0x5438]=25609;   // xi
   sortTable[0x5E0C]=25610;   // xi
   sortTable[0x6D17]=25611;   // xi
   sortTable[0x620F]=25612;   // xi
   sortTable[0x6232]=25613;   // xi
   sortTable[0x6790]=25614;   // xi
   sortTable[0x60DC]=25615;   // xi
   sortTable[0x6089]=25616;   // xi
   sortTable[0x727A]=25617;   // xi
   sortTable[0x72A7]=25618;   // xi
   sortTable[0x7A00]=25619;   // xi
   sortTable[0x5AB3]=25620;   // xi
   sortTable[0x88AD]=25621;   // xi
   sortTable[0x8972]=25622;   // xi
   sortTable[0x563B]=25623;   // xi
   sortTable[0x6EAA]=25624;   // xi
   sortTable[0x819D]=25625;   // xi
   sortTable[0x9699]=25626;   // xi
   sortTable[0x6670]=25627;   // xi
   sortTable[0x7184]=25628;   // xi
   sortTable[0x5915]=25629;   // xi
   sortTable[0x9521]=25630;   // xi
   sortTable[0x932B]=25631;   // xi
   sortTable[0x87CB]=25632;   // xi
   sortTable[0x7280]=25633;   // xi
   sortTable[0x6614]=25634;   // xi
   sortTable[0x7199]=25635;   // xi
   sortTable[0x5F99]=25636;   // xi
   sortTable[0x516E]=25637;   // xi
   sortTable[0x79A7]=25638;   // xi
   sortTable[0x595A]=25639;   // xi
   sortTable[0x8785]=25640;   // xi
   sortTable[0x73BA]=25641;   // xi
   sortTable[0x6A84]=25642;   // xi
   sortTable[0x50D6]=25643;   // xi
   sortTable[0x7FB2]=25644;   // xi
   sortTable[0x5B09]=25645;   // xi
   sortTable[0x71B9]=25646;   // xi
   sortTable[0x89FD]=25647;   // xi
   sortTable[0x66E6]=25648;   // xi
   sortTable[0x91AF]=25649;   // xi
   sortTable[0x94E3]=25650;   // xi
   sortTable[0x960B]=25651;   // xi
   sortTable[0x6DC5]=25652;   // xi
   sortTable[0x7FD5]=25653;   // xi
   sortTable[0x96B0]=25654;   // xi
   sortTable[0x9969]=25655;   // xi
   sortTable[0x6D60]=25656;   // xi
   sortTable[0x7699]=25657;   // xi
   sortTable[0x90D7]=25658;   // xi
   sortTable[0x7A78]=25659;   // xi
   sortTable[0x6C50]=25660;   // xi
   sortTable[0x8725]=25661;   // xi
   sortTable[0x550F]=25662;   // xi
   sortTable[0x8204]=25663;   // xi
   sortTable[0x6B37]=25664;   // xi
   sortTable[0x823E]=25665;   // xi
   sortTable[0x70EF]=25666;   // xi
   sortTable[0x83E5]=25667;   // xi
   sortTable[0x8C3F]=25668;   // xi
   sortTable[0x90E4]=25669;   // xi
   sortTable[0x7C9E]=25670;   // xi
   sortTable[0x8C68]=25671;   // xi
   sortTable[0x7AB8]=25672;   // xi
   sortTable[0x7852]=25673;   // xi
   sortTable[0x5C63]=25674;   // xi
   sortTable[0x89CB]=25675;   // xi
   sortTable[0x77FD]=25676;   // xi
   sortTable[0x6A28]=25677;   // xi
   sortTable[0x80B8]=25678;   // xi
   sortTable[0x798A]=25679;   // xi
   sortTable[0x9268]=25680;   // xi
   sortTable[0x89FF]=25681;   // xi
   sortTable[0x8478]=25682;   // xi
   sortTable[0x78CE]=25683;   // xi
   sortTable[0x9F37]=25684;   // xi
   sortTable[0x9145]=25685;   // xi
   sortTable[0x774E]=25686;   // xi
   sortTable[0x8B35]=25687;   // xi
   sortTable[0x84F0]=25688;   // xi
   sortTable[0x95DF]=25689;   // xi
   sortTable[0x76FB]=25690;   // xi
   sortTable[0x9E02]=25691;   // xi
   sortTable[0x8EA7]=25692;   // xi
   sortTable[0x7EE4]=25693;   // xi
   sortTable[0x8D69]=25694;   // xi
   sortTable[0x91F3]=25695;   // xi
   sortTable[0x8669]=25696;   // xi
   sortTable[0x96DF]=25697;   // xi
   sortTable[0x7E30]=25698;   // xi
   sortTable[0x8AF0]=25699;   // xi
   sortTable[0x8E5D]=25700;   // xi
   sortTable[0x7E6B]=25701;   // xi
   sortTable[0x910E]=25702;   // xi
   sortTable[0x9BD1]=25703;   // xi
   sortTable[0x78F6]=25704;   // xi
   sortTable[0x7CE6]=25705;   // xi
   sortTable[0x7D8C]=25706;   // xi
   sortTable[0x7DC6]=25707;   // xi
   sortTable[0x80B9]=25708;   // xi
   sortTable[0x9474]=25709;   // xi
   sortTable[0x972B]=25710;   // xi
   sortTable[0x87E2]=25711;   // xi
   sortTable[0x90CB]=25712;   // xi
   sortTable[0x8C95]=25713;   // xi
   sortTable[0x91F8]=25714;   // xi
   sortTable[0x9CDB]=25715;   // xi
   sortTable[0x84C6]=25716;   // xi
   sortTable[0x89F9]=25717;   // xi
   sortTable[0x8B46]=25718;   // xi
   sortTable[0x8C40]=25719;   // xi
   sortTable[0x77D6]=25720;   // xi
   sortTable[0x7E65]=25721;   // xi
   sortTable[0x8787]=25722;   // xi
   sortTable[0x884B]=25723;   // xi
   sortTable[0x8D65]=25724;   // xi
   sortTable[0x93B4]=25725;   // xi
   sortTable[0x9A31]=25726;   // xi
   sortTable[0x9A3D]=25727;   // xi
   sortTable[0x7902]=25728;   // xi
   sortTable[0x7FD6]=25729;   // xi
   sortTable[0x8582]=25730;   // xi
   sortTable[0x8835]=25731;   // xi
   sortTable[0x93ED]=25732;   // xi
   sortTable[0x7A27]=25733;   // xi
   sortTable[0x8203]=25734;   // xi
   sortTable[0x8383]=25735;   // xi
   sortTable[0x8448]=25736;   // xi
   sortTable[0x84A0]=25737;   // xi
   sortTable[0x856E]=25738;   // xi
   sortTable[0x89A1]=25739;   // xi
   sortTable[0x8B11]=25740;   // xi
   sortTable[0x8C6F]=25741;   // xi
   sortTable[0x969F]=25742;   // xi
   sortTable[0x993C]=25743;   // xi
   sortTable[0x997B]=25744;   // xi
   sortTable[0x9A68]=25745;   // xi
   sortTable[0x9B29]=25746;   // xi
   sortTable[0x9C3C]=25747;   // xi
   sortTable[0x9D57]=25748;   // xi
   sortTable[0x4FC2]=25749;   // xi
   sortTable[0x4FD9]=25750;   // xi
   sortTable[0x5092]=25751;   // xi
   sortTable[0x51DE]=25752;   // xi
   sortTable[0x5338]=25753;   // xi
   sortTable[0x534C]=25754;   // xi
   sortTable[0x5365]=25755;   // xi
   sortTable[0x5380]=25756;   // xi
   sortTable[0x546C]=25757;   // xi
   sortTable[0x54A5]=25758;   // xi
   sortTable[0x553D]=25759;   // xi
   sortTable[0x55BA]=25760;   // xi
   sortTable[0x564F]=25761;   // xi
   sortTable[0x56B1]=25762;   // xi
   sortTable[0x56CD]=25763;   // xi
   sortTable[0x588D]=25764;   // xi
   sortTable[0x58D0]=25765;   // xi
   sortTable[0x5B06]=25766;   // xi
   sortTable[0x5C43]=25767;   // xi
   sortTable[0x5C56]=25768;   // xi
   sortTable[0x5C6D]=25769;   // xi
   sortTable[0x5D60]=25770;   // xi
   sortTable[0x5D8D]=25771;   // xi
   sortTable[0x5DB2]=25772;   // xi
   sortTable[0x5DC7]=25773;   // xi
   sortTable[0x5F86]=25774;   // xi
   sortTable[0x5FAF]=25775;   // xi
   sortTable[0x5FDA]=25776;   // xi
   sortTable[0x5FE5]=25777;   // xi
   sortTable[0x602C]=25778;   // xi
   sortTable[0x6038]=25779;   // xi
   sortTable[0x6044]=25780;   // xi
   sortTable[0x6053]=25781;   // xi
   sortTable[0x6095]=25782;   // xi
   sortTable[0x60C1]=25783;   // xi
   sortTable[0x6140]=25784;   // xi
   sortTable[0x6198]=25785;   // xi
   sortTable[0x6199]=25786;   // xi
   sortTable[0x6231]=25787;   // xi
   sortTable[0x6271]=25788;   // xi
   sortTable[0x6278]=25789;   // xi
   sortTable[0x637F]=25790;   // xi
   sortTable[0x665E]=25791;   // xi
   sortTable[0x6673]=25792;   // xi
   sortTable[0x66BF]=25793;   // xi
   sortTable[0x67B2]=25794;   // xi
   sortTable[0x6878]=25795;   // xi
   sortTable[0x691E]=25796;   // xi
   sortTable[0x693A]=25797;   // xi
   sortTable[0x69BD]=25798;   // xi
   sortTable[0x69E2]=25799;   // xi
   sortTable[0x6A40]=25800;   // xi
   sortTable[0x6A72]=25801;   // xi
   sortTable[0x6B2F]=25802;   // xi
   sortTable[0x6B56]=25803;   // xi
   sortTable[0x6C25]=25804;   // xi
   sortTable[0x6E13]=25805;   // xi
   sortTable[0x6ECA]=25806;   // xi
   sortTable[0x6F07]=25807;   // xi
   sortTable[0x6F1D]=25808;   // xi
   sortTable[0x6F5D]=25809;   // xi
   sortTable[0x6F5F]=25810;   // xi
   sortTable[0x6F99]=25811;   // xi
   sortTable[0x7101]=25812;   // xi
   sortTable[0x7108]=25813;   // xi
   sortTable[0x711F]=25814;   // xi
   sortTable[0x712C]=25815;   // xi
   sortTable[0x7155]=25816;   // xi
   sortTable[0x7182]=25817;   // xi
   sortTable[0x7188]=25818;   // xi
   sortTable[0x71BA]=25819;   // xi
   sortTable[0x71BB]=25820;   // xi
   sortTable[0x71E8]=25821;   // xi
   sortTable[0x7214]=25822;   // xi
   sortTable[0x7294]=25823;   // xi
   sortTable[0x72A0]=25824;   // xi
   sortTable[0x72F6]=25825;   // xi
   sortTable[0x740B]=25826;   // xi
   sortTable[0x74BD]=25827;   // xi
   sortTable[0x761C]=25828;   // xi
   sortTable[0x77A6]=25829;   // xi
   sortTable[0x7E18]=25830;   // xi
   sortTable[0x84B5]=25831;   // xi
   sortTable[0x8980]=25832;   // xi
   sortTable[0x89A4]=25833;   // xi
   sortTable[0x8D87]=25834;   // xi
   sortTable[0x8D98]=25835;   // xi
   sortTable[0x9222]=25836;   // xi
   sortTable[0x9269]=25837;   // xi
   sortTable[0x96B5]=25838;   // xi
   sortTable[0x973C]=25839;   // xi
   sortTable[0x98C1]=25840;   // xi
   sortTable[0x990F]=25841;   // xi
   sortTable[0x9C5A]=25842;   // xi
   sortTable[0x9ED6]=25843;   // xi
   sortTable[0x4E0B]=25844;   // xia
   sortTable[0x590F]=25845;   // xia
   sortTable[0x5413]=25846;   // xia
   sortTable[0x5687]=25847;   // xia
   sortTable[0x5CE1]=25848;   // xia
   sortTable[0x5CFD]=25849;   // xia
   sortTable[0x778E]=25850;   // xia
   sortTable[0x867E]=25851;   // xia
   sortTable[0x8766]=25852;   // xia
   sortTable[0x72ED]=25853;   // xia
   sortTable[0x72F9]=25854;   // xia
   sortTable[0x971E]=25855;   // xia
   sortTable[0x4FA0]=25856;   // xia
   sortTable[0x8F96]=25857;   // xia
   sortTable[0x6687]=25858;   // xia
   sortTable[0x5323]=25859;   // xia
   sortTable[0x9050]=25860;   // xia
   sortTable[0x7455]=25861;   // xia
   sortTable[0x72CE]=25862;   // xia
   sortTable[0x9EE0]=25863;   // xia
   sortTable[0x796B]=25864;   // xia
   sortTable[0x7F45]=25865;   // xia
   sortTable[0x7856]=25866;   // xia
   sortTable[0x67D9]=25867;   // xia
   sortTable[0x9C15]=25868;   // xia
   sortTable[0x965C]=25869;   // xia
   sortTable[0x7E16]=25870;   // xia
   sortTable[0x821D]=25871;   // xia
   sortTable[0x93EC]=25872;   // xia
   sortTable[0x823A]=25873;   // xia
   sortTable[0x93BC]=25874;   // xia
   sortTable[0x938B]=25875;   // xia
   sortTable[0x9A22]=25876;   // xia
   sortTable[0x7FC8]=25877;   // xia
   sortTable[0x8578]=25878;   // xia
   sortTable[0x8C3A]=25879;   // xia
   sortTable[0x8D6E]=25880;   // xia
   sortTable[0x8F44]=25881;   // xia
   sortTable[0x935C]=25882;   // xia
   sortTable[0x967F]=25883;   // xia
   sortTable[0x9DB7]=25884;   // xia
   sortTable[0x7771]=25885;   // xia
   sortTable[0x7E00]=25886;   // xia
   sortTable[0x8672]=25887;   // xia
   sortTable[0x9B7B]=25888;   // xia
   sortTable[0x4E05]=25889;   // xia
   sortTable[0x4E64]=25890;   // xia
   sortTable[0x4FE0]=25891;   // xia
   sortTable[0x5084]=25892;   // xia
   sortTable[0x5737]=25893;   // xia
   sortTable[0x5913]=25894;   // xia
   sortTable[0x61D7]=25895;   // xia
   sortTable[0x656E]=25896;   // xia
   sortTable[0x68BA]=25897;   // xia
   sortTable[0x6E8A]=25898;   // xia
   sortTable[0x70A0]=25899;   // xia
   sortTable[0x70DA]=25900;   // xia
   sortTable[0x7146]=25901;   // xia
   sortTable[0x7175]=25902;   // xia
   sortTable[0x73E8]=25903;   // xia
   sortTable[0x759C]=25904;   // xia
   sortTable[0x75A8]=25905;   // xia
   sortTable[0x7864]=25906;   // xia
   sortTable[0x78AC]=25907;   // xia
   sortTable[0x78CD]=25908;   // xia
   sortTable[0x7B6A]=25909;   // xia
   sortTable[0x9595]=25910;   // xia
   sortTable[0x959C]=25911;   // xia
   sortTable[0x98AC]=25912;   // xia
   sortTable[0x73B0]=25913;   // xian
   sortTable[0x73FE]=25914;   // xian
   sortTable[0x5148]=25915;   // xian
   sortTable[0x7EBF]=25916;   // xian
   sortTable[0x7DDA]=25917;   // xian
   sortTable[0x663E]=25918;   // xian
   sortTable[0x986F]=25919;   // xian
   sortTable[0x53BF]=25920;   // xian
   sortTable[0x7E23]=25921;   // xian
   sortTable[0x9650]=25922;   // xian
   sortTable[0x9669]=25923;   // xian
   sortTable[0x96AA]=25924;   // xian
   sortTable[0x7EA4]=25925;   // xian
   sortTable[0x7E96]=25926;   // xian
   sortTable[0x9C9C]=25927;   // xian
   sortTable[0x9BAE]=25928;   // xian
   sortTable[0x732E]=25929;   // xian
   sortTable[0x737B]=25930;   // xian
   sortTable[0x95F2]=25931;   // xian
   sortTable[0x9591]=25932;   // xian
   sortTable[0x9677]=25933;   // xian
   sortTable[0x6380]=25934;   // xian
   sortTable[0x5BAA]=25935;   // xian
   sortTable[0x61B2]=25936;   // xian
   sortTable[0x5ACC]=25937;   // xian
   sortTable[0x7FA1]=25938;   // xian
   sortTable[0x7FA8]=25939;   // xian
   sortTable[0x4ED9]=25940;   // xian
   sortTable[0x5F26]=25941;   // xian
   sortTable[0x8854]=25942;   // xian
   sortTable[0x929C]=25943;   // xian
   sortTable[0x54B8]=25944;   // xian
   sortTable[0x9985]=25945;   // xian
   sortTable[0x9921]=25946;   // xian
   sortTable[0x8D24]=25947;   // xian
   sortTable[0x817A]=25948;   // xian
   sortTable[0x9170]=25949;   // xian
   sortTable[0x8237]=25950;   // xian
   sortTable[0x5A34]=25951;   // xian
   sortTable[0x6D8E]=25952;   // xian
   sortTable[0x66B9]=25953;   // xian
   sortTable[0x8DE3]=25954;   // xian
   sortTable[0x85D3]=25955;   // xian
   sortTable[0x82CB]=25956;   // xian
   sortTable[0x9528]=25957;   // xian
   sortTable[0x86AC]=25958;   // xian
   sortTable[0x5C98]=25959;   // xian
   sortTable[0x8DF9]=25960;   // xian
   sortTable[0x75EB]=25961;   // xian
   sortTable[0x7C7C]=25962;   // xian
   sortTable[0x83B6]=25963;   // xian
   sortTable[0x9E47]=25964;   // xian
   sortTable[0x7946]=25965;   // xian
   sortTable[0x8AF4]=25966;   // xian
   sortTable[0x71F9]=25967;   // xian
   sortTable[0x7B45]=25968;   // xian
   sortTable[0x774D]=25969;   // xian
   sortTable[0x9592]=25970;   // xian
   sortTable[0x7303]=25971;   // xian
   sortTable[0x51BC]=25972;   // xian
   sortTable[0x94E6]=25973;   // xian
   sortTable[0x6C19]=25974;   // xian
   sortTable[0x929B]=25975;   // xian
   sortTable[0x7CAF]=25976;   // xian
   sortTable[0x7D43]=25977;   // xian
   sortTable[0x9E79]=25978;   // xian
   sortTable[0x8B63]=25979;   // xian
   sortTable[0x81FD]=25980;   // xian
   sortTable[0x86BF]=25981;   // xian
   sortTable[0x934C]=25982;   // xian
   sortTable[0x8E6E]=25983;   // xian
   sortTable[0x8CE2]=25984;   // xian
   sortTable[0x97C5]=25985;   // xian
   sortTable[0x85D6]=25986;   // xian
   sortTable[0x9DF4]=25987;   // xian
   sortTable[0x784D]=25988;   // xian
   sortTable[0x918E]=25989;   // xian
   sortTable[0x92E7]=25990;   // xian
   sortTable[0x97F1]=25991;   // xian
   sortTable[0x79C8]=25992;   // xian
   sortTable[0x8706]=25993;   // xian
   sortTable[0x97EF]=25994;   // xian
   sortTable[0x9DF3]=25995;   // xian
   sortTable[0x770C]=25996;   // xian
   sortTable[0x7992]=25997;   // xian
   sortTable[0x7A34]=25998;   // xian
   sortTable[0x7E4A]=25999;   // xian
   sortTable[0x8C4F]=26000;   // xian
   sortTable[0x8F31]=26001;   // xian
   sortTable[0x9291]=26002;   // xian
   sortTable[0x9855]=26003;   // xian
   sortTable[0x9C7B]=26004;   // xian
   sortTable[0x9DFC]=26005;   // xian
   sortTable[0x9E99]=26006;   // xian
   sortTable[0x7925]=26007;   // xian
   sortTable[0x7CEE]=26008;   // xian
   sortTable[0x7D64]=26009;   // xian
   sortTable[0x7F10]=26010;   // xian
   sortTable[0x80D8]=26011;   // xian
   sortTable[0x8858]=26012;   // xian
   sortTable[0x8A2E]=26013;   // xian
   sortTable[0x8D12]=26014;   // xian
   sortTable[0x967A]=26015;   // xian
   sortTable[0x9F38]=26016;   // xian
   sortTable[0x4EDA]=26017;   // xian
   sortTable[0x4F23]=26018;   // xian
   sortTable[0x4F2D]=26019;   // xian
   sortTable[0x4F61]=26020;   // xian
   sortTable[0x50CA]=26021;   // xian
   sortTable[0x50E9]=26022;   // xian
   sortTable[0x50F2]=26023;   // xian
   sortTable[0x50F4]=26024;   // xian
   sortTable[0x549E]=26025;   // xian
   sortTable[0x54EF]=26026;   // xian
   sortTable[0x550C]=26027;   // xian
   sortTable[0x5563]=26028;   // xian
   sortTable[0x5615]=26029;   // xian
   sortTable[0x57B7]=26030;   // xian
   sortTable[0x58CF]=26031;   // xian
   sortTable[0x597E]=26032;   // xian
   sortTable[0x59B6]=26033;   // xian
   sortTable[0x59ED]=26034;   // xian
   sortTable[0x5A0A]=26035;   // xian
   sortTable[0x5A28]=26036;   // xian
   sortTable[0x5A39]=26037;   // xian
   sortTable[0x5A71]=26038;   // xian
   sortTable[0x5AFA]=26039;   // xian
   sortTable[0x5AFB]=26040;   // xian
   sortTable[0x5B10]=26041;   // xian
   sortTable[0x5C1F]=26042;   // xian
   sortTable[0x5C20]=26043;   // xian
   sortTable[0x5C73]=26044;   // xian
   sortTable[0x5CF4]=26045;   // xian
   sortTable[0x5D04]=26046;   // xian
   sortTable[0x5DAE]=26047;   // xian
   sortTable[0x5E70]=26048;   // xian
   sortTable[0x5EEF]=26049;   // xian
   sortTable[0x5FFA]=26050;   // xian
   sortTable[0x61AA]=26051;   // xian
   sortTable[0x61B8]=26052;   // xian
   sortTable[0x6326]=26053;   // xian
   sortTable[0x641F]=26054;   // xian
   sortTable[0x648A]=26055;   // xian
   sortTable[0x648F]=26056;   // xian
   sortTable[0x6507]=26057;   // xian
   sortTable[0x6515]=26058;   // xian
   sortTable[0x665B]=26059;   // xian
   sortTable[0x6774]=26060;   // xian
   sortTable[0x67AE]=26061;   // xian
   sortTable[0x6A4C]=26062;   // xian
   sortTable[0x6AF6]=26063;   // xian
   sortTable[0x6BE8]=26064;   // xian
   sortTable[0x6D80]=26065;   // xian
   sortTable[0x6F96]=26066;   // xian
   sortTable[0x7017]=26067;   // xian
   sortTable[0x7066]=26068;   // xian
   sortTable[0x70CD]=26069;   // xian
   sortTable[0x71C5]=26070;   // xian
   sortTable[0x72DD]=26071;   // xian
   sortTable[0x736B]=26072;   // xian
   sortTable[0x736E]=26073;   // xian
   sortTable[0x7381]=26074;   // xian
   sortTable[0x73D7]=26075;   // xian
   sortTable[0x7509]=26076;   // xian
   sortTable[0x7647]=26077;   // xian
   sortTable[0x764E]=26078;   // xian
   sortTable[0x77AF]=26079;   // xian
   sortTable[0x7BB2]=26080;   // xian
   sortTable[0x7DAB]=26081;   // xian
   sortTable[0x7E8E]=26082;   // xian
   sortTable[0x81D4]=26083;   // xian
   sortTable[0x82EE]=26084;   // xian
   sortTable[0x83A7]=26085;   // xian
   sortTable[0x859F]=26086;   // xian
   sortTable[0x861A]=26087;   // xian
   sortTable[0x86DD]=26088;   // xian
   sortTable[0x893C]=26089;   // xian
   sortTable[0x8973]=26090;   // xian
   sortTable[0x8AA2]=26091;   // xian
   sortTable[0x8AB8]=26092;   // xian
   sortTable[0x8D7B]=26093;   // xian
   sortTable[0x8E9A]=26094;   // xian
   sortTable[0x930E]=26095;   // xian
   sortTable[0x9341]=26096;   // xian
   sortTable[0x9466]=26097;   // xian
   sortTable[0x9665]=26098;   // xian
   sortTable[0x99A6]=26099;   // xian
   sortTable[0x9DB1]=26100;   // xian
   sortTable[0x9EB2]=26101;   // xian
   sortTable[0x60F3]=26102;   // xiang
   sortTable[0x8C61]=26103;   // xiang
   sortTable[0x5411]=26104;   // xiang
   sortTable[0x76F8]=26105;   // xiang
   sortTable[0x54CD]=26106;   // xiang
   sortTable[0x97FF]=26107;   // xiang
   sortTable[0x4E61]=26108;   // xiang
   sortTable[0x9109]=26109;   // xiang
   sortTable[0x9879]=26110;   // xiang
   sortTable[0x9805]=26111;   // xiang
   sortTable[0x50CF]=26112;   // xiang
   sortTable[0x9999]=26113;   // xiang
   sortTable[0x7BB1]=26114;   // xiang
   sortTable[0x8BE6]=26115;   // xiang
   sortTable[0x8A73]=26116;   // xiang
   sortTable[0x4EAB]=26117;   // xiang
   sortTable[0x5DF7]=26118;   // xiang
   sortTable[0x6A61]=26119;   // xiang
   sortTable[0x7965]=26120;   // xiang
   sortTable[0x53A2]=26121;   // xiang
   sortTable[0x5EC2]=26122;   // xiang
   sortTable[0x9576]=26123;   // xiang
   sortTable[0x9472]=26124;   // xiang
   sortTable[0x7FD4]=26125;   // xiang
   sortTable[0x8944]=26126;   // xiang
   sortTable[0x6E58]=26127;   // xiang
   sortTable[0x9977]=26128;   // xiang
   sortTable[0x98E8]=26129;   // xiang
   sortTable[0x9AA7]=26130;   // xiang
   sortTable[0x5EA0]=26131;   // xiang
   sortTable[0x8297]=26132;   // xiang
   sortTable[0x7F03]=26133;   // xiang
   sortTable[0x995F]=26134;   // xiang
   sortTable[0x8683]=26135;   // xiang
   sortTable[0x9C9E]=26136;   // xiang
   sortTable[0x8856]=26137;   // xiang
   sortTable[0x8459]=26138;   // xiang
   sortTable[0x9BD7]=26139;   // xiang
   sortTable[0x7A25]=26140;   // xiang
   sortTable[0x87D3]=26141;   // xiang
   sortTable[0x95C0]=26142;   // xiang
   sortTable[0x81B7]=26143;   // xiang
   sortTable[0x8801]=26144;   // xiang
   sortTable[0x7F3F]=26145;   // xiang
   sortTable[0x9297]=26146;   // xiang
   sortTable[0x9909]=26147;   // xiang
   sortTable[0x7DD7]=26148;   // xiang
   sortTable[0x8950]=26149;   // xiang
   sortTable[0x9284]=26150;   // xiang
   sortTable[0x9C4C]=26151;   // xiang
   sortTable[0x9E98]=26152;   // xiang
   sortTable[0x4EAF]=26153;   // xiang
   sortTable[0x4F6D]=26154;   // xiang
   sortTable[0x52E8]=26155;   // xiang
   sortTable[0x554C]=26156;   // xiang
   sortTable[0x56AE]=26157;   // xiang
   sortTable[0x5842]=26158;   // xiang
   sortTable[0x59E0]=26159;   // xiang
   sortTable[0x5D91]=26160;   // xiang
   sortTable[0x5FC0]=26161;   // xiang
   sortTable[0x6651]=26162;   // xiang
   sortTable[0x66CF]=26163;   // xiang
   sortTable[0x6819]=26164;   // xiang
   sortTable[0x697F]=26165;   // xiang
   sortTable[0x6B00]=26166;   // xiang
   sortTable[0x73E6]=26167;   // xiang
   sortTable[0x74D6]=26168;   // xiang
   sortTable[0x74E8]=26169;   // xiang
   sortTable[0x7D74]=26170;   // xiang
   sortTable[0x842B]=26171;   // xiang
   sortTable[0x858C]=26172;   // xiang
   sortTable[0x8DED]=26173;   // xiang
   sortTable[0x90F7]=26174;   // xiang
   sortTable[0x910A]=26175;   // xiang
   sortTable[0x9115]=26176;   // xiang
   sortTable[0x940C]=26177;   // xiang
   sortTable[0x9957]=26178;   // xiang
   sortTable[0x9A64]=26179;   // xiang
   sortTable[0x9B9D]=26180;   // xiang
   sortTable[0x9C5C]=26181;   // xiang
   sortTable[0x9C76]=26182;   // xiang
   sortTable[0x5C0F]=26183;   // xiao
   sortTable[0x7B11]=26184;   // xiao
   sortTable[0x6D88]=26185;   // xiao
   sortTable[0x6821]=26186;   // xiao
   sortTable[0x6548]=26187;   // xiao
   sortTable[0x9500]=26188;   // xiao
   sortTable[0x92B7]=26189;   // xiao
   sortTable[0x6653]=26190;   // xiao
   sortTable[0x66C9]=26191;   // xiao
   sortTable[0x8096]=26192;   // xiao
   sortTable[0x5BB5]=26193;   // xiao
   sortTable[0x5578]=26194;   // xiao
   sortTable[0x562F]=26195;   // xiao
   sortTable[0x54EE]=26196;   // xiao
   sortTable[0x785D]=26197;   // xiao
   sortTable[0x7BAB]=26198;   // xiao
   sortTable[0x7C2B]=26199;   // xiao
   sortTable[0x5B5D]=26200;   // xiao
   sortTable[0x8427]=26201;   // xiao
   sortTable[0x9704]=26202;   // xiao
   sortTable[0x9A81]=26203;   // xiao
   sortTable[0x56A3]=26204;   // xiao
   sortTable[0x6F47]=26205;   // xiao
   sortTable[0x67AD]=26206;   // xiao
   sortTable[0x900D]=26207;   // xiao
   sortTable[0x6DC6]=26208;   // xiao
   sortTable[0x7B71]=26209;   // xiao
   sortTable[0x9B48]=26210;   // xiao
   sortTable[0x7EE1]=26211;   // xiao
   sortTable[0x67B5]=26212;   // xiao
   sortTable[0x5D24]=26213;   // xiao
   sortTable[0x8653]=26214;   // xiao
   sortTable[0x54D3]=26215;   // xiao
   sortTable[0x7FDB]=26216;   // xiao
   sortTable[0x9E2E]=26217;   // xiao
   sortTable[0x8B0F]=26218;   // xiao
   sortTable[0x7BE0]=26219;   // xiao
   sortTable[0x81AE]=26220;   // xiao
   sortTable[0x9D1E]=26221;   // xiao
   sortTable[0x7B4A]=26222;   // xiao
   sortTable[0x87CF]=26223;   // xiao
   sortTable[0x9D35]=26224;   // xiao
   sortTable[0x7A99]=26225;   // xiao
   sortTable[0x8A68]=26226;   // xiao
   sortTable[0x7863]=26227;   // xiao
   sortTable[0x87C2]=26228;   // xiao
   sortTable[0x9AD0]=26229;   // xiao
   sortTable[0x7B7F]=26230;   // xiao
   sortTable[0x856D]=26231;   // xiao
   sortTable[0x8648]=26232;   // xiao
   sortTable[0x90E9]=26233;   // xiao
   sortTable[0x7C18]=26234;   // xiao
   sortTable[0x85C3]=26235;   // xiao
   sortTable[0x8828]=26236;   // xiao
   sortTable[0x8E03]=26237;   // xiao
   sortTable[0x9A4D]=26238;   // xiao
   sortTable[0x9AC7]=26239;   // xiao
   sortTable[0x4FBE]=26240;   // xiao
   sortTable[0x4FF2]=26241;   // xiao
   sortTable[0x509A]=26242;   // xiao
   sortTable[0x52B9]=26243;   // xiao
   sortTable[0x547A]=26244;   // xiao
   sortTable[0x54B2]=26245;   // xiao
   sortTable[0x560B]=26246;   // xiao
   sortTable[0x5610]=26247;   // xiao
   sortTable[0x5628]=26248;   // xiao
   sortTable[0x5635]=26249;   // xiao
   sortTable[0x56BB]=26250;   // xiao
   sortTable[0x56C2]=26251;   // xiao
   sortTable[0x5A4B]=26252;   // xiao
   sortTable[0x5BAF]=26253;   // xiao
   sortTable[0x5EA8]=26254;   // xiao
   sortTable[0x5F47]=26255;   // xiao
   sortTable[0x6077]=26256;   // xiao
   sortTable[0x61A2]=26257;   // xiao
   sortTable[0x63F1]=26258;   // xiao
   sortTable[0x6569]=26259;   // xiao
   sortTable[0x6585]=26260;   // xiao
   sortTable[0x6586]=26261;   // xiao
   sortTable[0x6681]=26262;   // xiao
   sortTable[0x689F]=26263;   // xiao
   sortTable[0x6AF9]=26264;   // xiao
   sortTable[0x6B4A]=26265;   // xiao
   sortTable[0x6B57]=26266;   // xiao
   sortTable[0x6BBD]=26267;   // xiao
   sortTable[0x6BCA]=26268;   // xiao
   sortTable[0x6D28]=26269;   // xiao
   sortTable[0x6D8D]=26270;   // xiao
   sortTable[0x6EE7]=26271;   // xiao
   sortTable[0x701F]=26272;   // xiao
   sortTable[0x7071]=26273;   // xiao
   sortTable[0x7072]=26274;   // xiao
   sortTable[0x7107]=26275;   // xiao
   sortTable[0x71BD]=26276;   // xiao
   sortTable[0x7307]=26277;   // xiao
   sortTable[0x7362]=26278;   // xiao
   sortTable[0x75DA]=26279;   // xiao
   sortTable[0x75DF]=26280;   // xiao
   sortTable[0x769B]=26281;   // xiao
   sortTable[0x76A2]=26282;   // xiao
   sortTable[0x7A58]=26283;   // xiao
   sortTable[0x7B05]=26284;   // xiao
   sortTable[0x7D83]=26285;   // xiao
   sortTable[0x8437]=26286;   // xiao
   sortTable[0x87F0]=26287;   // xiao
   sortTable[0x8A24]=26288;   // xiao
   sortTable[0x8A9F]=26289;   // xiao
   sortTable[0x8AB5]=26290;   // xiao
   sortTable[0x97A9]=26291;   // xiao
   sortTable[0x4E9B]=26292;   // xie
   sortTable[0x5199]=26293;   // xie
   sortTable[0x5BEB]=26294;   // xie
   sortTable[0x978B]=26295;   // xie
   sortTable[0x8C22]=26296;   // xie
   sortTable[0x8B1D]=26297;   // xie
   sortTable[0x68B0]=26298;   // xie
   sortTable[0x534F]=26299;   // xie
   sortTable[0x5354]=26300;   // xie
   sortTable[0x659C]=26301;   // xie
   sortTable[0x6B47]=26302;   // xie
   sortTable[0x80C1]=26303;   // xie
   sortTable[0x8105]=26304;   // xie
   sortTable[0x5378]=26305;   // xie
   sortTable[0x643A]=26306;   // xie
   sortTable[0x651C]=26307;   // xie
   sortTable[0x631F]=26308;   // xie
   sortTable[0x633E]=26309;   // xie
   sortTable[0x6CC4]=26310;   // xie
   sortTable[0x5C51]=26311;   // xie
   sortTable[0x90AA]=26312;   // xie
   sortTable[0x8C10]=26313;   // xie
   sortTable[0x61C8]=26314;   // xie
   sortTable[0x6CFB]=26315;   // xie
   sortTable[0x5055]=26316;   // xie
   sortTable[0x87F9]=26317;   // xie
   sortTable[0x4EB5]=26318;   // xie
   sortTable[0x71EE]=26319;   // xie
   sortTable[0x69AD]=26320;   // xie
   sortTable[0x52F0]=26321;   // xie
   sortTable[0x874E]=26322;   // xie
   sortTable[0x6E2B]=26323;   // xie
   sortTable[0x7F2C]=26324;   // xie
   sortTable[0x6954]=26325;   // xie
   sortTable[0x736C]=26326;   // xie
   sortTable[0x9082]=26327;   // xie
   sortTable[0x5EE8]=26328;   // xie
   sortTable[0x64B7]=26329;   // xie
   sortTable[0x698D]=26330;   // xie
   sortTable[0x7EC1]=26331;   // xie
   sortTable[0x79BC]=26332;   // xie
   sortTable[0x8E9E]=26333;   // xie
   sortTable[0x85A4]=26334;   // xie
   sortTable[0x7023]=26335;   // xie
   sortTable[0x818E]=26336;   // xie
   sortTable[0x8B97]=26337;   // xie
   sortTable[0x7DE4]=26338;   // xie
   sortTable[0x9FA4]=26339;   // xie
   sortTable[0x7D4F]=26340;   // xie
   sortTable[0x99F4]=26341;   // xie
   sortTable[0x887A]=26342;   // xie
   sortTable[0x97A2]=26343;   // xie
   sortTable[0x97F0]=26344;   // xie
   sortTable[0x7944]=26345;   // xie
   sortTable[0x85DB]=26346;   // xie
   sortTable[0x8909]=26347;   // xie
   sortTable[0x880F]=26348;   // xie
   sortTable[0x85A2]=26349;   // xie
   sortTable[0x880D]=26350;   // xie
   sortTable[0x7D8A]=26351;   // xie
   sortTable[0x7F37]=26352;   // xie
   sortTable[0x7FD3]=26353;   // xie
   sortTable[0x9F58]=26354;   // xie
   sortTable[0x7D32]=26355;   // xie
   sortTable[0x7E72]=26356;   // xie
   sortTable[0x9F5B]=26357;   // xie
   sortTable[0x8762]=26358;   // xie
   sortTable[0x893B]=26359;   // xie
   sortTable[0x896D]=26360;   // xie
   sortTable[0x8AE7]=26361;   // xie
   sortTable[0x97B5]=26362;   // xie
   sortTable[0x9F42]=26363;   // xie
   sortTable[0x4F33]=26364;   // xie
   sortTable[0x505E]=26365;   // xie
   sortTable[0x5070]=26366;   // xie
   sortTable[0x50C1]=26367;   // xie
   sortTable[0x51A9]=26368;   // xie
   sortTable[0x52A6]=26369;   // xie
   sortTable[0x5368]=26370;   // xie
   sortTable[0x55CB]=26371;   // xie
   sortTable[0x5667]=26372;   // xie
   sortTable[0x57A5]=26373;   // xie
   sortTable[0x586E]=26374;   // xie
   sortTable[0x5911]=26375;   // xie
   sortTable[0x594A]=26376;   // xie
   sortTable[0x5A0E]=26377;   // xie
   sortTable[0x5A9F]=26378;   // xie
   sortTable[0x5C53]=26379;   // xie
   sortTable[0x5C5F]=26380;   // xie
   sortTable[0x5C67]=26381;   // xie
   sortTable[0x5CEB]=26382;   // xie
   sortTable[0x5DB0]=26383;   // xie
   sortTable[0x5FA2]=26384;   // xie
   sortTable[0x604A]=26385;   // xie
   sortTable[0x6136]=26386;   // xie
   sortTable[0x62F9]=26387;   // xie
   sortTable[0x63F3]=26388;   // xie
   sortTable[0x64D5]=26389;   // xie
   sortTable[0x64F7]=26390;   // xie
   sortTable[0x65EA]=26391;   // xie
   sortTable[0x66AC]=26392;   // xie
   sortTable[0x6D29]=26393;   // xie
   sortTable[0x6FA5]=26394;   // xie
   sortTable[0x7009]=26395;   // xie
   sortTable[0x707A]=26396;   // xie
   sortTable[0x70A7]=26397;   // xie
   sortTable[0x70F2]=26398;   // xie
   sortTable[0x710E]=26399;   // xie
   sortTable[0x7181]=26400;   // xie
   sortTable[0x71F2]=26401;   // xie
   sortTable[0x7215]=26402;   // xie
   sortTable[0x7332]=26403;   // xie
   sortTable[0x744E]=26404;   // xie
   sortTable[0x7CCF]=26405;   // xie
   sortTable[0x7D6C]=26406;   // xie
   sortTable[0x7DF3]=26407;   // xie
   sortTable[0x7E88]=26408;   // xie
   sortTable[0x8107]=26409;   // xie
   sortTable[0x810B]=26410;   // xie
   sortTable[0x9821]=26411;   // xie
   sortTable[0x9F65]=26412;   // xie
   sortTable[0x5FC3]=26413;   // xin
   sortTable[0x65B0]=26414;   // xin
   sortTable[0x4FE1]=26415;   // xin
   sortTable[0x8F9B]=26416;   // xin
   sortTable[0x6B23]=26417;   // xin
   sortTable[0x85AA]=26418;   // xin
   sortTable[0x82AF]=26419;   // xin
   sortTable[0x950C]=26420;   // xin
   sortTable[0x92C5]=26421;   // xin
   sortTable[0x8845]=26422;   // xin
   sortTable[0x820B]=26423;   // xin
   sortTable[0x99A8]=26424;   // xin
   sortTable[0x5FFB]=26425;   // xin
   sortTable[0x6B46]=26426;   // xin
   sortTable[0x946B]=26427;   // xin
   sortTable[0x6615]=26428;   // xin
   sortTable[0x56DF]=26429;   // xin
   sortTable[0x8A22]=26430;   // xin
   sortTable[0x8ED0]=26431;   // xin
   sortTable[0x812A]=26432;   // xin
   sortTable[0x90A4]=26433;   // xin
   sortTable[0x91C1]=26434;   // xin
   sortTable[0x9856]=26435;   // xin
   sortTable[0x4F08]=26436;   // xin
   sortTable[0x4F29]=26437;   // xin
   sortTable[0x4FFD]=26438;   // xin
   sortTable[0x567A]=26439;   // xin
   sortTable[0x59A1]=26440;   // xin
   sortTable[0x5B1C]=26441;   // xin
   sortTable[0x5B5E]=26442;   // xin
   sortTable[0x5EDE]=26443;   // xin
   sortTable[0x5FC4]=26444;   // xin
   sortTable[0x60DE]=26445;   // xin
   sortTable[0x677A]=26446;   // xin
   sortTable[0x6794]=26447;   // xin
   sortTable[0x7098]=26448;   // xin
   sortTable[0x712E]=26449;   // xin
   sortTable[0x7161]=26450;   // xin
   sortTable[0x76FA]=26451;   // xin
   sortTable[0x8951]=26452;   // xin
   sortTable[0x8A2B]=26453;   // xin
   sortTable[0x920A]=26454;   // xin
   sortTable[0x9414]=26455;   // xin
   sortTable[0x9620]=26456;   // xin
   sortTable[0x99AB]=26457;   // xin
   sortTable[0x99B8]=26458;   // xin
   sortTable[0x884C]=26459;   // xing
   sortTable[0x6027]=26460;   // xing
   sortTable[0x5F62]=26461;   // xing
   sortTable[0x661F]=26462;   // xing
   sortTable[0x5174]=26463;   // xing
   sortTable[0x8208]=26464;   // xing
   sortTable[0x578B]=26465;   // xing
   sortTable[0x5E78]=26466;   // xing
   sortTable[0x59D3]=26467;   // xing
   sortTable[0x9192]=26468;   // xing
   sortTable[0x5211]=26469;   // xing
   sortTable[0x7329]=26470;   // xing
   sortTable[0x8165]=26471;   // xing
   sortTable[0x60FA]=26472;   // xing
   sortTable[0x674F]=26473;   // xing
   sortTable[0x90A2]=26474;   // xing
   sortTable[0x8365]=26475;   // xing
   sortTable[0x60BB]=26476;   // xing
   sortTable[0x784E]=26477;   // xing
   sortTable[0x9649]=26478;   // xing
   sortTable[0x8347]=26479;   // xing
   sortTable[0x94CF]=26480;   // xing
   sortTable[0x64E4]=26481;   // xing
   sortTable[0x9A8D]=26482;   // xing
   sortTable[0x9498]=26483;   // xing
   sortTable[0x9276]=26484;   // xing
   sortTable[0x9A02]=26485;   // xing
   sortTable[0x7BB5]=26486;   // xing
   sortTable[0x935F]=26487;   // xing
   sortTable[0x7BC2]=26488;   // xing
   sortTable[0x8395]=26489;   // xing
   sortTable[0x86F5]=26490;   // xing
   sortTable[0x81D6]=26491;   // xing
   sortTable[0x89F2]=26492;   // xing
   sortTable[0x9203]=26493;   // xing
   sortTable[0x9292]=26494;   // xing
   sortTable[0x4F80]=26495;   // xing
   sortTable[0x5016]=26496;   // xing
   sortTable[0x54D8]=26497;   // xing
   sortTable[0x57B6]=26498;   // xing
   sortTable[0x5A19]=26499;   // xing
   sortTable[0x5A5E]=26500;   // xing
   sortTable[0x5B39]=26501;   // xing
   sortTable[0x66D0]=26502;   // xing
   sortTable[0x6D10]=26503;   // xing
   sortTable[0x6DAC]=26504;   // xing
   sortTable[0x6ECE]=26505;   // xing
   sortTable[0x714B]=26506;   // xing
   sortTable[0x7446]=26507;   // xing
   sortTable[0x76A8]=26508;   // xing
   sortTable[0x7772]=26509;   // xing
   sortTable[0x7DC8]=26510;   // xing
   sortTable[0x88C4]=26511;   // xing
   sortTable[0x89EA]=26512;   // xing
   sortTable[0x8B03]=26513;   // xing
   sortTable[0x90C9]=26514;   // xing
   sortTable[0x92DE]=26515;   // xing
   sortTable[0x9658]=26516;   // xing
   sortTable[0x9B8F]=26517;   // xing
   sortTable[0x9BF9]=26518;   // xing
   sortTable[0x96C4]=26519;   // xiong
   sortTable[0x80F8]=26520;   // xiong
   sortTable[0x5144]=26521;   // xiong
   sortTable[0x51F6]=26522;   // xiong
   sortTable[0x718A]=26523;   // xiong
   sortTable[0x6C79]=26524;   // xiong
   sortTable[0x6D36]=26525;   // xiong
   sortTable[0x5308]=26526;   // xiong
   sortTable[0x8BC7]=26527;   // xiong
   sortTable[0x8A7E]=26528;   // xiong
   sortTable[0x8BBB]=26529;   // xiong
   sortTable[0x8CEF]=26530;   // xiong
   sortTable[0x8A57]=26531;   // xiong
   sortTable[0x5147]=26532;   // xiong
   sortTable[0x54C5]=26533;   // xiong
   sortTable[0x5910]=26534;   // xiong
   sortTable[0x5FF7]=26535;   // xiong
   sortTable[0x605F]=26536;   // xiong
   sortTable[0x657B]=26537;   // xiong
   sortTable[0x7138]=26538;   // xiong
   sortTable[0x713D]=26539;   // xiong
   sortTable[0x80F7]=26540;   // xiong
   sortTable[0x8A29]=26541;   // xiong
   sortTable[0x4FEE]=26542;   // xiu
   sortTable[0x4F11]=26543;   // xiu
   sortTable[0x8896]=26544;   // xiu
   sortTable[0x79C0]=26545;   // xiu
   sortTable[0x673D]=26546;   // xiu
   sortTable[0x7EE3]=26547;   // xiu
   sortTable[0x9508]=26548;   // xiu
   sortTable[0x93FD]=26549;   // xiu
   sortTable[0x7E61]=26550;   // xiu
   sortTable[0x55C5]=26551;   // xiu
   sortTable[0x7F9E]=26552;   // xiu
   sortTable[0x5CAB]=26553;   // xiu
   sortTable[0x9990]=26554;   // xiu
   sortTable[0x5EA5]=26555;   // xiu
   sortTable[0x8129]=26556;   // xiu
   sortTable[0x9E3A]=26557;   // xiu
   sortTable[0x54BB]=26558;   // xiu
   sortTable[0x6EB4]=26559;   // xiu
   sortTable[0x9AF9]=26560;   // xiu
   sortTable[0x8C85]=26561;   // xiu
   sortTable[0x890E]=26562;   // xiu
   sortTable[0x8119]=26563;   // xiu
   sortTable[0x9AE4]=26564;   // xiu
   sortTable[0x7E4D]=26565;   // xiu
   sortTable[0x8791]=26566;   // xiu
   sortTable[0x7D87]=26567;   // xiu
   sortTable[0x9948]=26568;   // xiu
   sortTable[0x9D42]=26569;   // xiu
   sortTable[0x7CD4]=26570;   // xiu
   sortTable[0x890F]=26571;   // xiu
   sortTable[0x9F45]=26572;   // xiu
   sortTable[0x82EC]=26573;   // xiu
   sortTable[0x929D]=26574;   // xiu
   sortTable[0x4FE2]=26575;   // xiu
   sortTable[0x5CC0]=26576;   // xiu
   sortTable[0x6A07]=26577;   // xiu
   sortTable[0x6EEB]=26578;   // xiu
   sortTable[0x70CB]=26579;   // xiu
   sortTable[0x70CC]=26580;   // xiu
   sortTable[0x73DB]=26581;   // xiu
   sortTable[0x7407]=26582;   // xiu
   sortTable[0x7493]=26583;   // xiu
   sortTable[0x81F9]=26584;   // xiu
   sortTable[0x92B9]=26585;   // xiu
   sortTable[0x9380]=26586;   // xiu
   sortTable[0x93C5]=26587;   // xiu
   sortTable[0x93E5]=26588;   // xiu
   sortTable[0x98CD]=26589;   // xiu
   sortTable[0x9BB4]=26590;   // xiu
   sortTable[0x9C43]=26591;   // xiu
   sortTable[0x8BB8]=26592;   // xu
   sortTable[0x8A31]=26593;   // xu
   sortTable[0x9700]=26594;   // xu
   sortTable[0x987B]=26595;   // xu
   sortTable[0x9808]=26596;   // xu
   sortTable[0x7EED]=26597;   // xu
   sortTable[0x7E8C]=26598;   // xu
   sortTable[0x5E8F]=26599;   // xu
   sortTable[0x7EEA]=26600;   // xu
   sortTable[0x7DD2]=26601;   // xu
   sortTable[0x865A]=26602;   // xu
   sortTable[0x865B]=26603;   // xu
   sortTable[0x5F90]=26604;   // xu
   sortTable[0x5401]=26605;   // xu
   sortTable[0x53D9]=26606;   // xu
   sortTable[0x6558]=26607;   // xu
   sortTable[0x84C4]=26608;   // xu
   sortTable[0x5618]=26609;   // xu
   sortTable[0x5653]=26610;   // xu
   sortTable[0x620C]=26611;   // xu
   sortTable[0x5A7F]=26612;   // xu
   sortTable[0x6064]=26613;   // xu
   sortTable[0x65ED]=26614;   // xu
   sortTable[0x7D6E]=26615;   // xu
   sortTable[0x80E5]=26616;   // xu
   sortTable[0x589F]=26617;   // xu
   sortTable[0x8BE9]=26618;   // xu
   sortTable[0x987C]=26619;   // xu
   sortTable[0x6E86]=26620;   // xu
   sortTable[0x84FF]=26621;   // xu
   sortTable[0x52D6]=26622;   // xu
   sortTable[0x7166]=26623;   // xu
   sortTable[0x6829]=26624;   // xu
   sortTable[0x76F1]=26625;   // xu
   sortTable[0x9157]=26626;   // xu
   sortTable[0x8C1E]=26627;   // xu
   sortTable[0x7E7B]=26628;   // xu
   sortTable[0x6D2B]=26629;   // xu
   sortTable[0x7CC8]=26630;   // xu
   sortTable[0x9B46]=26631;   // xu
   sortTable[0x9191]=26632;   // xu
   sortTable[0x8A0F]=26633;   // xu
   sortTable[0x8A39]=26634;   // xu
   sortTable[0x85DA]=26635;   // xu
   sortTable[0x8053]=26636;   // xu
   sortTable[0x980A]=26637;   // xu
   sortTable[0x8ADD]=26638;   // xu
   sortTable[0x9C6E]=26639;   // xu
   sortTable[0x9126]=26640;   // xu
   sortTable[0x85C7]=26641;   // xu
   sortTable[0x9B56]=26642;   // xu
   sortTable[0x8751]=26643;   // xu
   sortTable[0x8A61]=26644;   // xu
   sortTable[0x8657]=26645;   // xu
   sortTable[0x76E8]=26646;   // xu
   sortTable[0x82A7]=26647;   // xu
   sortTable[0x8B43]=26648;   // xu
   sortTable[0x928A]=26649;   // xu
   sortTable[0x9A49]=26650;   // xu
   sortTable[0x76E2]=26651;   // xu
   sortTable[0x7A30]=26652;   // xu
   sortTable[0x7A38]=26653;   // xu
   sortTable[0x7E03]=26654;   // xu
   sortTable[0x805F]=26655;   // xu
   sortTable[0x9450]=26656;   // xu
   sortTable[0x9B1A]=26657;   // xu
   sortTable[0x9B63]=26658;   // xu
   sortTable[0x7D9A]=26659;   // xu
   sortTable[0x84A3]=26660;   // xu
   sortTable[0x8566]=26661;   // xu
   sortTable[0x4F35]=26662;   // xu
   sortTable[0x4F90]=26663;   // xu
   sortTable[0x4FC6]=26664;   // xu
   sortTable[0x5066]=26665;   // xu
   sortTable[0x5194]=26666;   // xu
   sortTable[0x52D7]=26667;   // xu
   sortTable[0x5379]=26668;   // xu
   sortTable[0x5474]=26669;   // xu
   sortTable[0x55A3]=26670;   // xu
   sortTable[0x57BF]=26671;   // xu
   sortTable[0x58FB]=26672;   // xu
   sortTable[0x59C1]=26673;   // xu
   sortTable[0x5AAD]=26674;   // xu
   sortTable[0x5B03]=26675;   // xu
   sortTable[0x5E41]=26676;   // xu
   sortTable[0x6034]=26677;   // xu
   sortTable[0x6149]=26678;   // xu
   sortTable[0x63DF]=26679;   // xu
   sortTable[0x654D]=26680;   // xu
   sortTable[0x65F4]=26681;   // xu
   sortTable[0x662B]=26682;   // xu
   sortTable[0x6647]=26683;   // xu
   sortTable[0x668A]=26684;   // xu
   sortTable[0x6702]=26685;   // xu
   sortTable[0x6948]=26686;   // xu
   sortTable[0x69D2]=26687;   // xu
   sortTable[0x6B28]=26688;   // xu
   sortTable[0x6B30]=26689;   // xu
   sortTable[0x6B54]=26690;   // xu
   sortTable[0x6B88]=26691;   // xu
   sortTable[0x6C7F]=26692;   // xu
   sortTable[0x6C80]=26693;   // xu
   sortTable[0x6E51]=26694;   // xu
   sortTable[0x6F35]=26695;   // xu
   sortTable[0x6F4A]=26696;   // xu
   sortTable[0x70C5]=26697;   // xu
   sortTable[0x70FC]=26698;   // xu
   sortTable[0x735D]=26699;   // xu
   sortTable[0x73DD]=26700;   // xu
   sortTable[0x73EC]=26701;   // xu
   sortTable[0x759E]=26702;   // xu
   sortTable[0x7781]=26703;   // xu
   sortTable[0x77B2]=26704;   // xu
   sortTable[0x7AA2]=26705;   // xu
   sortTable[0x7DD6]=26706;   // xu
   sortTable[0x88C7]=26707;   // xu
   sortTable[0x8CC9]=26708;   // xu
   sortTable[0x9009]=26709;   // xuan
   sortTable[0x9078]=26710;   // xuan
   sortTable[0x5BA3]=26711;   // xuan
   sortTable[0x65CB]=26712;   // xuan
   sortTable[0x60AC]=26713;   // xuan
   sortTable[0x61F8]=26714;   // xuan
   sortTable[0x6F29]=26715;   // xuan
   sortTable[0x7EDA]=26716;   // xuan
   sortTable[0x7D62]=26717;   // xuan
   sortTable[0x6684]=26718;   // xuan
   sortTable[0x7384]=26719;   // xuan
   sortTable[0x8F69]=26720;   // xuan
   sortTable[0x7487]=26721;   // xuan
   sortTable[0x55A7]=26722;   // xuan
   sortTable[0x7729]=26723;   // xuan
   sortTable[0x6E32]=26724;   // xuan
   sortTable[0x8C16]=26725;   // xuan
   sortTable[0x70AB]=26726;   // xuan
   sortTable[0x714A]=26727;   // xuan
   sortTable[0x94C9]=26728;   // xuan
   sortTable[0x8431]=26729;   // xuan
   sortTable[0x7663]=26730;   // xuan
   sortTable[0x6966]=26731;   // xuan
   sortTable[0x6CEB]=26732;   // xuan
   sortTable[0x5107]=26733;   // xuan
   sortTable[0x63CE]=26734;   // xuan
   sortTable[0x7FFE]=26735;   // xuan
   sortTable[0x8852]=26736;   // xuan
   sortTable[0x955F]=26737;   // xuan
   sortTable[0x8AE0]=26738;   // xuan
   sortTable[0x88A8]=26739;   // xuan
   sortTable[0x92D7]=26740;   // xuan
   sortTable[0x7734]=26741;   // xuan
   sortTable[0x8701]=26742;   // xuan
   sortTable[0x857F]=26743;   // xuan
   sortTable[0x8B5E]=26744;   // xuan
   sortTable[0x78B9]=26745;   // xuan
   sortTable[0x75C3]=26746;   // xuan
   sortTable[0x8ED2]=26747;   // xuan
   sortTable[0x7FE7]=26748;   // xuan
   sortTable[0x9799]=26749;   // xuan
   sortTable[0x7BAE]=26750;   // xuan
   sortTable[0x77CE]=26751;   // xuan
   sortTable[0x7E3C]=26752;   // xuan
   sortTable[0x8610]=26753;   // xuan
   sortTable[0x8809]=26754;   // xuan
   sortTable[0x9848]=26755;   // xuan
   sortTable[0x99FD]=26756;   // xuan
   sortTable[0x84D2]=26757;   // xuan
   sortTable[0x8D19]=26758;   // xuan
   sortTable[0x93C7]=26759;   // xuan
   sortTable[0x777B]=26760;   // xuan
   sortTable[0x79A4]=26761;   // xuan
   sortTable[0x8519]=26762;   // xuan
   sortTable[0x8756]=26763;   // xuan
   sortTable[0x5405]=26764;   // xuan
   sortTable[0x54BA]=26765;   // xuan
   sortTable[0x5847]=26766;   // xuan
   sortTable[0x5A97]=26767;   // xuan
   sortTable[0x5AD9]=26768;   // xuan
   sortTable[0x5F32]=26769;   // xuan
   sortTable[0x6030]=26770;   // xuan
   sortTable[0x6103]=26771;   // xuan
   sortTable[0x610B]=26772;   // xuan
   sortTable[0x61C1]=26773;   // xuan
   sortTable[0x660D]=26774;   // xuan
   sortTable[0x6621]=26775;   // xuan
   sortTable[0x6645]=26776;   // xuan
   sortTable[0x66B6]=26777;   // xuan
   sortTable[0x688B]=26778;   // xuan
   sortTable[0x6965]=26779;   // xuan
   sortTable[0x6A88]=26780;   // xuan
   sortTable[0x70DC]=26781;   // xuan
   sortTable[0x73B9]=26782;   // xuan
   sortTable[0x7401]=26783;   // xuan
   sortTable[0x7404]=26784;   // xuan
   sortTable[0x7444]=26785;   // xuan
   sortTable[0x74BF]=26786;   // xuan
   sortTable[0x766C]=26787;   // xuan
   sortTable[0x7E07]=26788;   // xuan
   sortTable[0x7E4F]=26789;   // xuan
   sortTable[0x8432]=26790;   // xuan
   sortTable[0x85FC]=26791;   // xuan
   sortTable[0x8AFC]=26792;   // xuan
   sortTable[0x8B82]=26793;   // xuan
   sortTable[0x9249]=26794;   // xuan
   sortTable[0x9379]=26795;   // xuan
   sortTable[0x98B4]=26796;   // xuan
   sortTable[0x9C1A]=26797;   // xuan
   sortTable[0x5B66]=26798;   // xue
   sortTable[0x5B78]=26799;   // xue
   sortTable[0x96EA]=26800;   // xue
   sortTable[0x8840]=26801;   // xue
   sortTable[0x524A]=26802;   // xue
   sortTable[0x9774]=26803;   // xue
   sortTable[0x859B]=26804;   // xue
   sortTable[0x7A74]=26805;   // xue
   sortTable[0x8C11]=26806;   // xue
   sortTable[0x8E05]=26807;   // xue
   sortTable[0x6CF6]=26808;   // xue
   sortTable[0x9CD5]=26809;   // xue
   sortTable[0x89F7]=26810;   // xue
   sortTable[0x97BE]=26811;   // xue
   sortTable[0x9E34]=26812;   // xue
   sortTable[0x8895]=26813;   // xue
   sortTable[0x9DFD]=26814;   // xue
   sortTable[0x825D]=26815;   // xue
   sortTable[0x8313]=26816;   // xue
   sortTable[0x81A4]=26817;   // xue
   sortTable[0x8D90]=26818;   // xue
   sortTable[0x4E74]=26819;   // xue
   sortTable[0x5437]=26820;   // xue
   sortTable[0x5779]=26821;   // xue
   sortTable[0x58C6]=26822;   // xue
   sortTable[0x5CA4]=26823;   // xue
   sortTable[0x5CC3]=26824;   // xue
   sortTable[0x5DA8]=26825;   // xue
   sortTable[0x6588]=26826;   // xue
   sortTable[0x6856]=26827;   // xue
   sortTable[0x6A30]=26828;   // xue
   sortTable[0x6FA9]=26829;   // xue
   sortTable[0x7025]=26830;   // xue
   sortTable[0x71E2]=26831;   // xue
   sortTable[0x72D8]=26832;   // xue
   sortTable[0x75B6]=26833;   // xue
   sortTable[0x8486]=26834;   // xue
   sortTable[0x8B14]=26835;   // xue
   sortTable[0x8F4C]=26836;   // xue
   sortTable[0x8FA5]=26837;   // xue
   sortTable[0x8FAA]=26838;   // xue
   sortTable[0x96E4]=26839;   // xue
   sortTable[0x9C48]=26840;   // xue
   sortTable[0x8FC5]=26841;   // xun
   sortTable[0x8BAF]=26842;   // xun
   sortTable[0x8A0A]=26843;   // xun
   sortTable[0x5BFB]=26844;   // xun
   sortTable[0x5C0B]=26845;   // xun
   sortTable[0x8BAD]=26846;   // xun
   sortTable[0x8A13]=26847;   // xun
   sortTable[0x5FAA]=26848;   // xun
   sortTable[0x65EC]=26849;   // xun
   sortTable[0x8BE2]=26850;   // xun
   sortTable[0x8A62]=26851;   // xun
   sortTable[0x718F]=26852;   // xun
   sortTable[0x5DE1]=26853;   // xun
   sortTable[0x6C5B]=26854;   // xun
   sortTable[0x900A]=26855;   // xun
   sortTable[0x905C]=26856;   // xun
   sortTable[0x52CB]=26857;   // xun
   sortTable[0x8368]=26858;   // xun
   sortTable[0x9A6F]=26859;   // xun
   sortTable[0x8340]=26860;   // xun
   sortTable[0x5F87]=26861;   // xun
   sortTable[0x6B89]=26862;   // xun
   sortTable[0x85B0]=26863;   // xun
   sortTable[0x6D54]=26864;   // xun
   sortTable[0x6042]=26865;   // xun
   sortTable[0x91BA]=26866;   // xun
   sortTable[0x66DB]=26867;   // xun
   sortTable[0x6D35]=26868;   // xun
   sortTable[0x5DFD]=26869;   // xun
   sortTable[0x5CCB]=26870;   // xun
   sortTable[0x7AA8]=26871;   // xun
   sortTable[0x736F]=26872;   // xun
   sortTable[0x9129]=26873;   // xun
   sortTable[0x57D9]=26874;   // xun
   sortTable[0x7E81]=26875;   // xun
   sortTable[0x9C9F]=26876;   // xun
   sortTable[0x8548]=26877;   // xun
   sortTable[0x7D03]=26878;   // xun
   sortTable[0x8A19]=26879;   // xun
   sortTable[0x81D0]=26880;   // xun
   sortTable[0x9D55]=26881;   // xun
   sortTable[0x77C4]=26882;   // xun
   sortTable[0x9868]=26883;   // xun
   sortTable[0x99E8]=26884;   // xun
   sortTable[0x7A04]=26885;   // xun
   sortTable[0x87F3]=26886;   // xun
   sortTable[0x99B4]=26887;   // xun
   sortTable[0x9C4F]=26888;   // xun
   sortTable[0x4F28]=26889;   // xun
   sortTable[0x4F9A]=26890;   // xun
   sortTable[0x5071]=26891;   // xun
   sortTable[0x52DB]=26892;   // xun
   sortTable[0x52F2]=26893;   // xun
   sortTable[0x52F3]=26894;   // xun
   sortTable[0x5342]=26895;   // xun
   sortTable[0x5640]=26896;   // xun
   sortTable[0x565A]=26897;   // xun
   sortTable[0x5691]=26898;   // xun
   sortTable[0x5743]=26899;   // xun
   sortTable[0x5864]=26900;   // xun
   sortTable[0x58CE]=26901;   // xun
   sortTable[0x58E6]=26902;   // xun
   sortTable[0x595E]=26903;   // xun
   sortTable[0x5DFA]=26904;   // xun
   sortTable[0x5EF5]=26905;   // xun
   sortTable[0x613B]=26906;   // xun
   sortTable[0x63D7]=26907;   // xun
   sortTable[0x6533]=26908;   // xun
   sortTable[0x674A]=26909;   // xun
   sortTable[0x6812]=26910;   // xun
   sortTable[0x686A]=26911;   // xun
   sortTable[0x69C6]=26912;   // xun
   sortTable[0x6A33]=26913;   // xun
   sortTable[0x6BBE]=26914;   // xun
   sortTable[0x6BE5]=26915;   // xun
   sortTable[0x6F43]=26916;   // xun
   sortTable[0x6F60]=26917;   // xun
   sortTable[0x6F6F]=26918;   // xun
   sortTable[0x7065]=26919;   // xun
   sortTable[0x7104]=26920;   // xun
   sortTable[0x71D6]=26921;   // xun
   sortTable[0x71FB]=26922;   // xun
   sortTable[0x720B]=26923;   // xun
   sortTable[0x72E5]=26924;   // xun
   sortTable[0x73E3]=26925;   // xun
   sortTable[0x7495]=26926;   // xun
   sortTable[0x7543]=26927;   // xun
   sortTable[0x8512]=26928;   // xun
   sortTable[0x85AB]=26929;   // xun
   sortTable[0x860D]=26930;   // xun
   sortTable[0x8CD0]=26931;   // xun
   sortTable[0x8FFF]=26932;   // xun
   sortTable[0x9442]=26933;   // xun
   sortTable[0x9C58]=26934;   // xun
   sortTable[0x5440]=27392;   // ya
   sortTable[0x538B]=27393;   // ya
   sortTable[0x58D3]=27394;   // ya
   sortTable[0x7259]=27395;   // ya
   sortTable[0x9E26]=27396;   // ya
   sortTable[0x9D09]=27397;   // ya
   sortTable[0x4E2B]=27398;   // ya
   sortTable[0x5D16]=27399;   // ya
   sortTable[0x9E2D]=27400;   // ya
   sortTable[0x9D28]=27401;   // ya
   sortTable[0x62BC]=27402;   // ya
   sortTable[0x8BB6]=27403;   // ya
   sortTable[0x8A1D]=27404;   // ya
   sortTable[0x82BD]=27405;   // ya
   sortTable[0x4E9A]=27406;   // ya
   sortTable[0x4E9E]=27407;   // ya
   sortTable[0x8F67]=27408;   // ya
   sortTable[0x8ECB]=27409;   // ya
   sortTable[0x8859]=27410;   // ya
   sortTable[0x869C]=27411;   // ya
   sortTable[0x54D1]=27412;   // ya
   sortTable[0x555E]=27413;   // ya
   sortTable[0x96C5]=27414;   // ya
   sortTable[0x6DAF]=27415;   // ya
   sortTable[0x5A05]=27416;   // ya
   sortTable[0x4F22]=27417;   // ya
   sortTable[0x8FD3]=27418;   // ya
   sortTable[0x6C29]=27419;   // ya
   sortTable[0x740A]=27420;   // ya
   sortTable[0x7811]=27421;   // ya
   sortTable[0x5416]=27422;   // ya
   sortTable[0x6860]=27423;   // ya
   sortTable[0x775A]=27424;   // ya
   sortTable[0x57AD]=27425;   // ya
   sortTable[0x75D6]=27426;   // ya
   sortTable[0x63E0]=27427;   // ya
   sortTable[0x9D76]=27428;   // ya
   sortTable[0x5C88]=27429;   // ya
   sortTable[0x9F7E]=27430;   // ya
   sortTable[0x94D4]=27431;   // ya
   sortTable[0x7A0F]=27432;   // ya
   sortTable[0x930F]=27433;   // ya
   sortTable[0x7AAB]=27434;   // ya
   sortTable[0x9F56]=27435;   // ya
   sortTable[0x897E]=27436;   // ya
   sortTable[0x8050]=27437;   // ya
   sortTable[0x8565]=27438;   // ya
   sortTable[0x4E5B]=27439;   // ya
   sortTable[0x4E9C]=27440;   // ya
   sortTable[0x4FF9]=27441;   // ya
   sortTable[0x529C]=27442;   // ya
   sortTable[0x538A]=27443;   // ya
   sortTable[0x5391]=27444;   // ya
   sortTable[0x5393]=27445;   // ya
   sortTable[0x5516]=27446;   // ya
   sortTable[0x5714]=27447;   // ya
   sortTable[0x5720]=27448;   // ya
   sortTable[0x5727]=27449;   // ya
   sortTable[0x57E1]=27450;   // ya
   sortTable[0x5810]=27451;   // ya
   sortTable[0x5A6D]=27452;   // ya
   sortTable[0x5B72]=27453;   // ya
   sortTable[0x5D15]=27454;   // ya
   sortTable[0x5E8C]=27455;   // ya
   sortTable[0x5E98]=27456;   // ya
   sortTable[0x631C]=27457;   // ya
   sortTable[0x6397]=27458;   // ya
   sortTable[0x6792]=27459;   // ya
   sortTable[0x690F]=27460;   // ya
   sortTable[0x6C2C]=27461;   // ya
   sortTable[0x6F04]=27462;   // ya
   sortTable[0x72BD]=27463;   // ya
   sortTable[0x731A]=27464;   // ya
   sortTable[0x7330]=27465;   // ya
   sortTable[0x73A1]=27466;   // ya
   sortTable[0x7458]=27467;   // ya
   sortTable[0x7602]=27468;   // ya
   sortTable[0x7B0C]=27469;   // ya
   sortTable[0x941A]=27470;   // ya
   sortTable[0x773C]=27471;   // yan
   sortTable[0x7814]=27472;   // yan
   sortTable[0x9A8C]=27473;   // yan
   sortTable[0x9A57]=27474;   // yan
   sortTable[0x4E25]=27475;   // yan
   sortTable[0x56B4]=27476;   // yan
   sortTable[0x70DF]=27477;   // yan
   sortTable[0x7159]=27478;   // yan
   sortTable[0x6F14]=27479;   // yan
   sortTable[0x8A00]=27480;   // yan
   sortTable[0x6CBF]=27481;   // yan
   sortTable[0x989C]=27482;   // yan
   sortTable[0x984F]=27483;   // yan
   sortTable[0x63A9]=27484;   // yan
   sortTable[0x5CA9]=27485;   // yan
   sortTable[0x76D0]=27486;   // yan
   sortTable[0x9E7D]=27487;   // yan
   sortTable[0x538C]=27488;   // yan
   sortTable[0x53AD]=27489;   // yan
   sortTable[0x71D5]=27490;   // yan
   sortTable[0x5EF6]=27491;   // yan
   sortTable[0x708E]=27492;   // yan
   sortTable[0x6DF9]=27493;   // yan
   sortTable[0x8273]=27494;   // yan
   sortTable[0x8277]=27495;   // yan
   sortTable[0x5BB4]=27496;   // yan
   sortTable[0x7130]=27497;   // yan
   sortTable[0x54BD]=27498;   // yan
   sortTable[0x6A90]=27499;   // yan
   sortTable[0x5830]=27500;   // yan
   sortTable[0x884D]=27501;   // yan
   sortTable[0x7109]=27502;   // yan
   sortTable[0x5F66]=27503;   // yan
   sortTable[0x960E]=27504;   // yan
   sortTable[0x96C1]=27505;   // yan
   sortTable[0x664F]=27506;   // yan
   sortTable[0x7B75]=27507;   // yan
   sortTable[0x5043]=27508;   // yan
   sortTable[0x4FE8]=27509;   // yan
   sortTable[0x5944]=27510;   // yan
   sortTable[0x5AE3]=27511;   // yan
   sortTable[0x6E6E]=27512;   // yan
   sortTable[0x9609]=27513;   // yan
   sortTable[0x781A]=27514;   // yan
   sortTable[0x5156]=27515;   // yan
   sortTable[0x9B47]=27516;   // yan
   sortTable[0x8C1A]=27517;   // yan
   sortTable[0x7430]=27518;   // yan
   sortTable[0x8712]=27519;   // yan
   sortTable[0x598D]=27520;   // yan
   sortTable[0x990D]=27521;   // yan
   sortTable[0x9122]=27522;   // yan
   sortTable[0x80ED]=27523;   // yan
   sortTable[0x8D5D]=27524;   // yan
   sortTable[0x83F8]=27525;   // yan
   sortTable[0x814C]=27526;   // yan
   sortTable[0x53A3]=27527;   // yan
   sortTable[0x7131]=27528;   // yan
   sortTable[0x82AB]=27529;   // yan
   sortTable[0x90FE]=27530;   // yan
   sortTable[0x6079]=27531;   // yan
   sortTable[0x8C33]=27532;   // yan
   sortTable[0x95EB]=27533;   // yan
   sortTable[0x7F68]=27534;   // yan
   sortTable[0x5501]=27535;   // yan
   sortTable[0x6EDF]=27536;   // yan
   sortTable[0x5D26]=27537;   // yan
   sortTable[0x9D08]=27538;   // yan
   sortTable[0x7E2F]=27539;   // yan
   sortTable[0x9F39]=27540;   // yan
   sortTable[0x917D]=27541;   // yan
   sortTable[0x7D96]=27542;   // yan
   sortTable[0x9EE1]=27543;   // yan
   sortTable[0x9DC3]=27544;   // yan
   sortTable[0x89FE]=27545;   // yan
   sortTable[0x9854]=27546;   // yan
   sortTable[0x784F]=27547;   // yan
   sortTable[0x8A7D]=27548;   // yan
   sortTable[0x839A]=27549;   // yan
   sortTable[0x9D33]=27550;   // yan
   sortTable[0x9869]=27551;   // yan
   sortTable[0x8758]=27552;   // yan
   sortTable[0x8B9E]=27553;   // yan
   sortTable[0x95BB]=27554;   // yan
   sortTable[0x9DA0]=27555;   // yan
   sortTable[0x8EC5]=27556;   // yan
   sortTable[0x9C0B]=27557;   // yan
   sortTable[0x7C37]=27558;   // yan
   sortTable[0x787D]=27559;   // yan
   sortTable[0x91BC]=27560;   // yan
   sortTable[0x9681]=27561;   // yan
   sortTable[0x9EA3]=27562;   // yan
   sortTable[0x9F91]=27563;   // yan
   sortTable[0x90D4]=27564;   // yan
   sortTable[0x8664]=27565;   // yan
   sortTable[0x8D0B]=27566;   // yan
   sortTable[0x9140]=27567;   // yan
   sortTable[0x9183]=27568;   // yan
   sortTable[0x995C]=27569;   // yan
   sortTable[0x9F74]=27570;   // yan
   sortTable[0x786F]=27571;   // yan
   sortTable[0x7939]=27572;   // yan
   sortTable[0x9153]=27573;   // yan
   sortTable[0x9692]=27574;   // yan
   sortTable[0x9F34]=27575;   // yan
   sortTable[0x789E]=27576;   // yan
   sortTable[0x8276]=27577;   // yan
   sortTable[0x8412]=27578;   // yan
   sortTable[0x8505]=27579;   // yan
   sortTable[0x88FA]=27580;   // yan
   sortTable[0x8AFA]=27581;   // yan
   sortTable[0x8B8C]=27582;   // yan
   sortTable[0x91B6]=27583;   // yan
   sortTable[0x95B9]=27584;   // yan
   sortTable[0x9A34]=27585;   // yan
   sortTable[0x9A60]=27586;   // yan
   sortTable[0x9B33]=27587;   // yan
   sortTable[0x9EE4]=27588;   // yan
   sortTable[0x9EEB]=27589;   // yan
   sortTable[0x9EED]=27590;   // yan
   sortTable[0x9EF6]=27591;   // yan
   sortTable[0x4E75]=27592;   // yan
   sortTable[0x5050]=27593;   // yan
   sortTable[0x5063]=27594;   // yan
   sortTable[0x50BF]=27595;   // yan
   sortTable[0x513C]=27596;   // yan
   sortTable[0x5157]=27597;   // yan
   sortTable[0x5266]=27598;   // yan
   sortTable[0x533D]=27599;   // yan
   sortTable[0x53B3]=27600;   // yan
   sortTable[0x53B4]=27601;   // yan
   sortTable[0x5571]=27602;   // yan
   sortTable[0x55AD]=27603;   // yan
   sortTable[0x565E]=27604;   // yan
   sortTable[0x56A5]=27605;   // yan
   sortTable[0x5869]=27606;   // yan
   sortTable[0x5895]=27607;   // yan
   sortTable[0x58DB]=27608;   // yan
   sortTable[0x58E7]=27609;   // yan
   sortTable[0x5935]=27610;   // yan
   sortTable[0x599F]=27611;   // yan
   sortTable[0x59F2]=27612;   // yan
   sortTable[0x59F8]=27613;   // yan
   sortTable[0x5A2B]=27614;   // yan
   sortTable[0x5A2E]=27615;   // yan
   sortTable[0x5B0A]=27616;   // yan
   sortTable[0x5B2E]=27617;   // yan
   sortTable[0x5B3F]=27618;   // yan
   sortTable[0x5B4D]=27619;   // yan
   sortTable[0x5D43]=27620;   // yan
   sortTable[0x5D52]=27621;   // yan
   sortTable[0x5D53]=27622;   // yan
   sortTable[0x5D96]=27623;   // yan
   sortTable[0x5DCC]=27624;   // yan
   sortTable[0x5DD6]=27625;   // yan
   sortTable[0x5DD7]=27626;   // yan
   sortTable[0x5DD8]=27627;   // yan
   sortTable[0x5DDA]=27628;   // yan
   sortTable[0x5F07]=27629;   // yan
   sortTable[0x5F65]=27630;   // yan
   sortTable[0x611D]=27631;   // yan
   sortTable[0x61D5]=27632;   // yan
   sortTable[0x61E8]=27633;   // yan
   sortTable[0x622D]=27634;   // yan
   sortTable[0x624A]=27635;   // yan
   sortTable[0x6281]=27636;   // yan
   sortTable[0x63C5]=27637;   // yan
   sortTable[0x63DC]=27638;   // yan
   sortTable[0x6565]=27639;   // yan
   sortTable[0x6616]=27640;   // yan
   sortTable[0x66A5]=27641;   // yan
   sortTable[0x66D5]=27642;   // yan
   sortTable[0x66E3]=27643;   // yan
   sortTable[0x66EE]=27644;   // yan
   sortTable[0x68EA]=27645;   // yan
   sortTable[0x693B]=27646;   // yan
   sortTable[0x693C]=27647;   // yan
   sortTable[0x694C]=27648;   // yan
   sortTable[0x6A2E]=27649;   // yan
   sortTable[0x6ABF]=27650;   // yan
   sortTable[0x6AE9]=27651;   // yan
   sortTable[0x6B15]=27652;   // yan
   sortTable[0x6C87]=27653;   // yan
   sortTable[0x6DCA]=27654;   // yan
   sortTable[0x6E30]=27655;   // yan
   sortTable[0x6E37]=27656;   // yan
   sortTable[0x6E7A]=27657;   // yan
   sortTable[0x6E8E]=27658;   // yan
   sortTable[0x6F39]=27659;   // yan
   sortTable[0x704E]=27660;   // yan
   sortTable[0x7054]=27661;   // yan
   sortTable[0x7067]=27662;   // yan
   sortTable[0x7069]=27663;   // yan
   sortTable[0x70FB]=27664;   // yan
   sortTable[0x7111]=27665;   // yan
   sortTable[0x7114]=27666;   // yan
   sortTable[0x7196]=27667;   // yan
   sortTable[0x71C4]=27668;   // yan
   sortTable[0x7213]=27669;   // yan
   sortTable[0x726A]=27670;   // yan
   sortTable[0x72FF]=27671;   // yan
   sortTable[0x7312]=27672;   // yan
   sortTable[0x73DA]=27673;   // yan
   sortTable[0x7402]=27674;   // yan
   sortTable[0x7517]=27675;   // yan
   sortTable[0x7BF6]=27676;   // yan
   sortTable[0x81D9]=27677;   // yan
   sortTable[0x8455]=27678;   // yan
   sortTable[0x8917]=27679;   // yan
   sortTable[0x898E]=27680;   // yan
   sortTable[0x89C3]=27681;   // yan
   sortTable[0x8A01]=27682;   // yan
   sortTable[0x8BA0]=27683;   // yan
   sortTable[0x8C53]=27684;   // yan
   sortTable[0x8C54]=27685;   // yan
   sortTable[0x8D17]=27686;   // yan
   sortTable[0x8EBD]=27687;   // yan
   sortTable[0x9043]=27688;   // yan
   sortTable[0x91C5]=27689;   // yan
   sortTable[0x9586]=27690;   // yan
   sortTable[0x9A10]=27691;   // yan
   sortTable[0x9A13]=27692;   // yan
   sortTable[0x9B58]=27693;   // yan
   sortTable[0x9CEB]=27694;   // yan
   sortTable[0x9DF0]=27695;   // yan
   sortTable[0x9EEC]=27696;   // yan
   sortTable[0x9F5E]=27697;   // yan
   sortTable[0x6837]=27698;   // yang
   sortTable[0x6A23]=27699;   // yang
   sortTable[0x9633]=27700;   // yang
   sortTable[0x967D]=27701;   // yang
   sortTable[0x517B]=27702;   // yang
   sortTable[0x990A]=27703;   // yang
   sortTable[0x592E]=27704;   // yang
   sortTable[0x6D0B]=27705;   // yang
   sortTable[0x7F8A]=27706;   // yang
   sortTable[0x626C]=27707;   // yang
   sortTable[0x63DA]=27708;   // yang
   sortTable[0x6C27]=27709;   // yang
   sortTable[0x6768]=27710;   // yang
   sortTable[0x694A]=27711;   // yang
   sortTable[0x4EF0]=27712;   // yang
   sortTable[0x79E7]=27713;   // yang
   sortTable[0x6F3E]=27714;   // yang
   sortTable[0x75D2]=27715;   // yang
   sortTable[0x9E2F]=27716;   // yang
   sortTable[0x7080]=27717;   // yang
   sortTable[0x9785]=27718;   // yang
   sortTable[0x4F6F]=27719;   // yang
   sortTable[0x6B83]=27720;   // yang
   sortTable[0x6059]=27721;   // yang
   sortTable[0x600F]=27722;   // yang
   sortTable[0x75A1]=27723;   // yang
   sortTable[0x70CA]=27724;   // yang
   sortTable[0x86D8]=27725;   // yang
   sortTable[0x98CF]=27726;   // yang
   sortTable[0x6CF1]=27727;   // yang
   sortTable[0x5F89]=27728;   // yang
   sortTable[0x7F95]=27729;   // yang
   sortTable[0x9496]=27730;   // yang
   sortTable[0x99DA]=27731;   // yang
   sortTable[0x8AF9]=27732;   // yang
   sortTable[0x9D39]=27733;   // yang
   sortTable[0x773B]=27734;   // yang
   sortTable[0x98BA]=27735;   // yang
   sortTable[0x9D26]=27736;   // yang
   sortTable[0x935A]=27737;   // yang
   sortTable[0x8EEE]=27738;   // yang
   sortTable[0x8F30]=27739;   // yang
   sortTable[0x9C11]=27740;   // yang
   sortTable[0x770F]=27741;   // yang
   sortTable[0x7993]=27742;   // yang
   sortTable[0x7D3B]=27743;   // yang
   sortTable[0x7F8F]=27744;   // yang
   sortTable[0x8A47]=27745;   // yang
   sortTable[0x4F52]=27746;   // yang
   sortTable[0x509F]=27747;   // yang
   sortTable[0x52B7]=27748;   // yang
   sortTable[0x5489]=27749;   // yang
   sortTable[0x5771]=27750;   // yang
   sortTable[0x579F]=27751;   // yang
   sortTable[0x594D]=27752;   // yang
   sortTable[0x59CE]=27753;   // yang
   sortTable[0x5C9F]=27754;   // yang
   sortTable[0x5D35]=27755;   // yang
   sortTable[0x5D38]=27756;   // yang
   sortTable[0x6143]=27757;   // yang
   sortTable[0x61E9]=27758;   // yang
   sortTable[0x62B0]=27759;   // yang
   sortTable[0x6501]=27760;   // yang
   sortTable[0x656D]=27761;   // yang
   sortTable[0x65F8]=27762;   // yang
   sortTable[0x661C]=27763;   // yang
   sortTable[0x6698]=27764;   // yang
   sortTable[0x67CD]=27765;   // yang
   sortTable[0x6967]=27766;   // yang
   sortTable[0x69D8]=27767;   // yang
   sortTable[0x6C1C]=27768;   // yang
   sortTable[0x6C31]=27769;   // yang
   sortTable[0x7001]=27770;   // yang
   sortTable[0x70B4]=27771;   // yang
   sortTable[0x716C]=27772;   // yang
   sortTable[0x73DC]=27773;   // yang
   sortTable[0x760D]=27774;   // yang
   sortTable[0x7662]=27775;   // yang
   sortTable[0x7922]=27776;   // yang
   sortTable[0x7FAA]=27777;   // yang
   sortTable[0x80E6]=27778;   // yang
   sortTable[0x8746]=27779;   // yang
   sortTable[0x9260]=27780;   // yang
   sortTable[0x940A]=27781;   // yang
   sortTable[0x9626]=27782;   // yang
   sortTable[0x96F5]=27783;   // yang
   sortTable[0x9737]=27784;   // yang
   sortTable[0x9E09]=27785;   // yang
   sortTable[0x8981]=27786;   // yao
   sortTable[0x836F]=27787;   // yao
   sortTable[0x85E5]=27788;   // yao
   sortTable[0x6447]=27789;   // yao
   sortTable[0x6416]=27790;   // yao
   sortTable[0x54AC]=27791;   // yao
   sortTable[0x8170]=27792;   // yao
   sortTable[0x7A91]=27793;   // yao
   sortTable[0x7AAF]=27794;   // yao
   sortTable[0x8000]=27795;   // yao
   sortTable[0x5406]=27796;   // yao
   sortTable[0x9080]=27797;   // yao
   sortTable[0x9065]=27798;   // yao
   sortTable[0x9059]=27799;   // yao
   sortTable[0x8C23]=27800;   // yao
   sortTable[0x8200]=27801;   // yao
   sortTable[0x8B20]=27802;   // yao
   sortTable[0x94A5]=27803;   // yao
   sortTable[0x9470]=27804;   // yao
   sortTable[0x5996]=27805;   // yao
   sortTable[0x59DA]=27806;   // yao
   sortTable[0x5C27]=27807;   // yao
   sortTable[0x7476]=27808;   // yao
   sortTable[0x592D]=27809;   // yao
   sortTable[0x66DC]=27810;   // yao
   sortTable[0x80B4]=27811;   // yao
   sortTable[0x7E47]=27812;   // yao
   sortTable[0x6773]=27813;   // yao
   sortTable[0x9E5E]=27814;   // yao
   sortTable[0x723B]=27815;   // yao
   sortTable[0x7A88]=27816;   // yao
   sortTable[0x5FAD]=27817;   // yao
   sortTable[0x5E7A]=27818;   // yao
   sortTable[0x5D3E]=27819;   // yao
   sortTable[0x8F7A]=27820;   // yao
   sortTable[0x7945]=27821;   // yao
   sortTable[0x98D6]=27822;   // yao
   sortTable[0x7A85]=27823;   // yao
   sortTable[0x73E7]=27824;   // yao
   sortTable[0x9CD0]=27825;   // yao
   sortTable[0x9A15]=27826;   // yao
   sortTable[0x977F]=27827;   // yao
   sortTable[0x8A1E]=27828;   // yao
   sortTable[0x7A94]=27829;   // yao
   sortTable[0x888E]=27830;   // yao
   sortTable[0x9864]=27831;   // yao
   sortTable[0x7711]=27832;   // yao
   sortTable[0x825E]=27833;   // yao
   sortTable[0x991A]=27834;   // yao
   sortTable[0x77C5]=27835;   // yao
   sortTable[0x846F]=27836;   // yao
   sortTable[0x847D]=27837;   // yao
   sortTable[0x95C4]=27838;   // yao
   sortTable[0x9DD5]=27839;   // yao
   sortTable[0x7A7E]=27840;   // yao
   sortTable[0x929A]=27841;   // yao
   sortTable[0x9906]=27842;   // yao
   sortTable[0x9D22]=27843;   // yao
   sortTable[0x7B44]=27844;   // yao
   sortTable[0x84D4]=27845;   // yao
   sortTable[0x899E]=27846;   // yao
   sortTable[0x8B21]=27847;   // yao
   sortTable[0x8B91]=27848;   // yao
   sortTable[0x98BB]=27849;   // yao
   sortTable[0x9C29]=27850;   // yao
   sortTable[0x9F3C]=27851;   // yao
   sortTable[0x4EF8]=27852;   // yao
   sortTable[0x5004]=27853;   // yao
   sortTable[0x5060]=27854;   // yao
   sortTable[0x509C]=27855;   // yao
   sortTable[0x5593]=27856;   // yao
   sortTable[0x55C2]=27857;   // yao
   sortTable[0x579A]=27858;   // yao
   sortTable[0x582F]=27859;   // yao
   sortTable[0x5A79]=27860;   // yao
   sortTable[0x5AB1]=27861;   // yao
   sortTable[0x5B8E]=27862;   // yao
   sortTable[0x5C2D]=27863;   // yao
   sortTable[0x5C86]=27864;   // yao
   sortTable[0x5CE3]=27865;   // yao
   sortTable[0x5DA2]=27866;   // yao
   sortTable[0x5DA4]=27867;   // yao
   sortTable[0x612E]=27868;   // yao
   sortTable[0x62AD]=27869;   // yao
   sortTable[0x63FA]=27870;   // yao
   sortTable[0x669A]=27871;   // yao
   sortTable[0x6796]=27872;   // yao
   sortTable[0x67FC]=27873;   // yao
   sortTable[0x6946]=27874;   // yao
   sortTable[0x699A]=27875;   // yao
   sortTable[0x69A3]=27876;   // yao
   sortTable[0x6B80]=27877;   // yao
   sortTable[0x6E94]=27878;   // yao
   sortTable[0x70D1]=27879;   // yao
   sortTable[0x718E]=27880;   // yao
   sortTable[0x71FF]=27881;   // yao
   sortTable[0x72D5]=27882;   // yao
   sortTable[0x733A]=27883;   // yao
   sortTable[0x735F]=27884;   // yao
   sortTable[0x7464]=27885;   // yao
   sortTable[0x7AB0]=27886;   // yao
   sortTable[0x7E85]=27887;   // yao
   sortTable[0x82ED]=27888;   // yao
   sortTable[0x85AC]=27889;   // yao
   sortTable[0x8628]=27890;   // yao
   sortTable[0x8A4F]=27891;   // yao
   sortTable[0x8EFA]=27892;   // yao
   sortTable[0x908E]=27893;   // yao
   sortTable[0x9390]=27894;   // yao
   sortTable[0x9D01]=27895;   // yao
   sortTable[0x9DC2]=27896;   // yao
   sortTable[0x9F69]=27897;   // yao
   sortTable[0x4E5F]=27898;   // ye
   sortTable[0x4E1A]=27899;   // ye
   sortTable[0x696D]=27900;   // ye
   sortTable[0x591C]=27901;   // ye
   sortTable[0x7237]=27902;   // ye
   sortTable[0x723A]=27903;   // ye
   sortTable[0x91CE]=27904;   // ye
   sortTable[0x53F6]=27905;   // ye
   sortTable[0x8449]=27906;   // ye
   sortTable[0x6DB2]=27907;   // ye
   sortTable[0x51B6]=27908;   // ye
   sortTable[0x9875]=27909;   // ye
   sortTable[0x9801]=27910;   // ye
   sortTable[0x8036]=27911;   // ye
   sortTable[0x8C12]=27912;   // ye
   sortTable[0x6396]=27913;   // ye
   sortTable[0x66F3]=27914;   // ye
   sortTable[0x9134]=27915;   // ye
   sortTable[0x6654]=27916;   // ye
   sortTable[0x90BA]=27917;   // ye
   sortTable[0x814B]=27918;   // ye
   sortTable[0x564E]=27919;   // ye
   sortTable[0x70E8]=27920;   // ye
   sortTable[0x6930]=27921;   // ye
   sortTable[0x9765]=27922;   // ye
   sortTable[0x63F6]=27923;   // ye
   sortTable[0x94D8]=27924;   // ye
   sortTable[0x998C]=27925;   // ye
   sortTable[0x882E]=27926;   // ye
   sortTable[0x9371]=27927;   // ye
   sortTable[0x9941]=27928;   // ye
   sortTable[0x9381]=27929;   // ye
   sortTable[0x9437]=27930;   // ye
   sortTable[0x790F]=27931;   // ye
   sortTable[0x4EAA]=27932;   // ye
   sortTable[0x4EB1]=27933;   // ye
   sortTable[0x503B]=27934;   // ye
   sortTable[0x50F7]=27935;   // ye
   sortTable[0x5414]=27936;   // ye
   sortTable[0x5558]=27937;   // ye
   sortTable[0x5622]=27938;   // ye
   sortTable[0x5688]=27939;   // ye
   sortTable[0x57DC]=27940;   // ye
   sortTable[0x5828]=27941;   // ye
   sortTable[0x58B7]=27942;   // ye
   sortTable[0x58C4]=27943;   // ye
   sortTable[0x5DAA]=27944;   // ye
   sortTable[0x5DAB]=27945;   // ye
   sortTable[0x62B4]=27946;   // ye
   sortTable[0x6353]=27947;   // ye
   sortTable[0x64DB]=27948;   // ye
   sortTable[0x64E8]=27949;   // ye
   sortTable[0x64EA]=27950;   // ye
   sortTable[0x64EB]=27951;   // ye
   sortTable[0x668D]=27952;   // ye
   sortTable[0x66C4]=27953;   // ye
   sortTable[0x66C5]=27954;   // ye
   sortTable[0x66D7]=27955;   // ye
   sortTable[0x66F5]=27956;   // ye
   sortTable[0x67BC]=27957;   // ye
   sortTable[0x67BD]=27958;   // ye
   sortTable[0x696A]=27959;   // ye
   sortTable[0x6B4B]=27960;   // ye
   sortTable[0x6B97]=27961;   // ye
   sortTable[0x6F1C]=27962;   // ye
   sortTable[0x6F71]=27963;   // ye
   sortTable[0x6FB2]=27964;   // ye
   sortTable[0x71C1]=27965;   // ye
   sortTable[0x7217]=27966;   // ye
   sortTable[0x76A3]=27967;   // ye
   sortTable[0x77B1]=27968;   // ye
   sortTable[0x77B8]=27969;   // ye
   sortTable[0x8B01]=27970;   // ye
   sortTable[0x9113]=27971;   // ye
   sortTable[0x91FE]=27972;   // ye
   sortTable[0x92E3]=27973;   // ye
   sortTable[0x9391]=27974;   // ye
   sortTable[0x9768]=27975;   // ye
   sortTable[0x9923]=27976;   // ye
   sortTable[0x9A5C]=27977;   // ye
   sortTable[0x9D7A]=27978;   // ye
   sortTable[0x9E08]=27979;   // ye
   sortTable[0x4E00]=27980;   // yi
   sortTable[0x4EE5]=27981;   // yi
   sortTable[0x4E49]=27982;   // yi
   sortTable[0x7FA9]=27983;   // yi
   sortTable[0x5DF2]=27984;   // yi
   sortTable[0x610F]=27985;   // yi
   sortTable[0x8863]=27986;   // yi
   sortTable[0x827A]=27987;   // yi
   sortTable[0x85DD]=27988;   // yi
   sortTable[0x8BAE]=27989;   // yi
   sortTable[0x8B70]=27990;   // yi
   sortTable[0x6613]=27991;   // yi
   sortTable[0x533B]=27992;   // yi
   sortTable[0x91AB]=27993;   // yi
   sortTable[0x4F9D]=27994;   // yi
   sortTable[0x76CA]=27995;   // yi
   sortTable[0x79FB]=27996;   // yi
   sortTable[0x4EBF]=27997;   // yi
   sortTable[0x5104]=27998;   // yi
   sortTable[0x7591]=27999;   // yi
   sortTable[0x5F02]=28000;   // yi
   sortTable[0x7570]=28001;   // yi
   sortTable[0x5F79]=28002;   // yi
   sortTable[0x4EEA]=28003;   // yi
   sortTable[0x5100]=28004;   // yi
   sortTable[0x5FC6]=28005;   // yi
   sortTable[0x61B6]=28006;   // yi
   sortTable[0x6905]=28007;   // yi
   sortTable[0x9057]=28008;   // yi
   sortTable[0x907A]=28009;   // yi
   sortTable[0x8BD1]=28010;   // yi
   sortTable[0x8B6F]=28011;   // yi
   sortTable[0x5B9C]=28012;   // yi
   sortTable[0x59E8]=28013;   // yi
   sortTable[0x6291]=28014;   // yi
   sortTable[0x4E59]=28015;   // yi
   sortTable[0x54A6]=28016;   // yi
   sortTable[0x8681]=28017;   // yi
   sortTable[0x87FB]=28018;   // yi
   sortTable[0x4EA6]=28019;   // yi
   sortTable[0x501A]=28020;   // yi
   sortTable[0x8C0A]=28021;   // yi
   sortTable[0x8ABC]=28022;   // yi
   sortTable[0x6BC5]=28023;   // yi
   sortTable[0x7FFC]=28024;   // yi
   sortTable[0x5C79]=28025;   // yi
   sortTable[0x4F0A]=28026;   // yi
   sortTable[0x75AB]=28027;   // yi
   sortTable[0x77E3]=28028;   // yi
   sortTable[0x5937]=28029;   // yi
   sortTable[0x9091]=28030;   // yi
   sortTable[0x8BE3]=28031;   // yi
   sortTable[0x9038]=28032;   // yi
   sortTable[0x6EA2]=28033;   // yi
   sortTable[0x61FF]=28034;   // yi
   sortTable[0x9A7F]=28035;   // yi
   sortTable[0x63D6]=28036;   // yi
   sortTable[0x6021]=28037;   // yi
   sortTable[0x9890]=28038;   // yi
   sortTable[0x88D4]=28039;   // yi
   sortTable[0x5955]=28040;   // yi
   sortTable[0x6339]=28041;   // yi
   sortTable[0x6C82]=28042;   // yi
   sortTable[0x5F5D]=28043;   // yi
   sortTable[0x8D3B]=28044;   // yi
   sortTable[0x7ECE]=28045;   // yi
   sortTable[0x7FCA]=28046;   // yi
   sortTable[0x80F0]=28047;   // yi
   sortTable[0x7FCC]=28048;   // yi
   sortTable[0x7F22]=28049;   // yi
   sortTable[0x8FE4]=28050;   // yi
   sortTable[0x4F5A]=28051;   // yi
   sortTable[0x58F9]=28052;   // yi
   sortTable[0x7317]=28053;   // yi
   sortTable[0x8BD2]=28054;   // yi
   sortTable[0x5F08]=28055;   // yi
   sortTable[0x9552]=28056;   // yi
   sortTable[0x5F0B]=28057;   // yi
   sortTable[0x81C6]=28058;   // yi
   sortTable[0x9974]=28059;   // yi
   sortTable[0x6F2A]=28060;   // yi
   sortTable[0x8F76]=28061;   // yi
   sortTable[0x7617]=28062;   // yi
   sortTable[0x5208]=28063;   // yi
   sortTable[0x5DB7]=28064;   // yi
   sortTable[0x7FF3]=28065;   // yi
   sortTable[0x9857]=28066;   // yi
   sortTable[0x5453]=28067;   // yi
   sortTable[0x82E1]=28068;   // yi
   sortTable[0x858F]=28069;   // yi
   sortTable[0x603F]=28070;   // yi
   sortTable[0x794E]=28071;   // yi
   sortTable[0x65D6]=28072;   // yi
   sortTable[0x566B]=28073;   // yi
   sortTable[0x5CC4]=28074;   // yi
   sortTable[0x7719]=28075;   // yi
   sortTable[0x8084]=28076;   // yi
   sortTable[0x71A0]=28077;   // yi
   sortTable[0x94F1]=28078;   // yi
   sortTable[0x7FBF]=28079;   // yi
   sortTable[0x54BF]=28080;   // yi
   sortTable[0x6B39]=28081;   // yi
   sortTable[0x9809]=28082;   // yi
   sortTable[0x8734]=28083;   // yi
   sortTable[0x4F7E]=28084;   // yi
   sortTable[0x75CD]=28085;   // yi
   sortTable[0x9487]=28086;   // yi
   sortTable[0x6092]=28087;   // yi
   sortTable[0x6BAA]=28088;   // yi
   sortTable[0x5293]=28089;   // yi
   sortTable[0x57F8]=28090;   // yi
   sortTable[0x572F]=28091;   // yi
   sortTable[0x84FA]=28092;   // yi
   sortTable[0x8223]=28093;   // yi
   sortTable[0x9EDF]=28094;   // yi
   sortTable[0x9ED3]=28095;   // yi
   sortTable[0x9F6E]=28096;   // yi
   sortTable[0x88DB]=28097;   // yi
   sortTable[0x7654]=28098;   // yi
   sortTable[0x9E62]=28099;   // yi
   sortTable[0x7E44]=28100;   // yi
   sortTable[0x8E26]=28101;   // yi
   sortTable[0x91F4]=28102;   // yi
   sortTable[0x9E65]=28103;   // yi
   sortTable[0x8798]=28104;   // yi
   sortTable[0x8FFB]=28105;   // yi
   sortTable[0x91B3]=28106;   // yi
   sortTable[0x8A63]=28107;   // yi
   sortTable[0x8CA4]=28108;   // yi
   sortTable[0x9D82]=28109;   // yi
   sortTable[0x914F]=28110;   // yi
   sortTable[0x7E76]=28111;   // yi
   sortTable[0x776A]=28112;   // yi
   sortTable[0x8B89]=28113;   // yi
   sortTable[0x9571]=28114;   // yi
   sortTable[0x9DC1]=28115;   // yi
   sortTable[0x8A11]=28116;   // yi
   sortTable[0x8C77]=28117;   // yi
   sortTable[0x9E03]=28118;   // yi
   sortTable[0x93B0]=28119;   // yi
   sortTable[0x88A3]=28120;   // yi
   sortTable[0x8B3B]=28121;   // yi
   sortTable[0x8F59]=28122;   // yi
   sortTable[0x85D9]=28123;   // yi
   sortTable[0x8D00]=28124;   // yi
   sortTable[0x86E1]=28125;   // yi
   sortTable[0x91D4]=28126;   // yi
   sortTable[0x8875]=28127;   // yi
   sortTable[0x8DC7]=28128;   // yi
   sortTable[0x8FB7]=28129;   // yi
   sortTable[0x9950]=28130;   // yi
   sortTable[0x91B7]=28131;   // yi
   sortTable[0x7995]=28132;   // yi
   sortTable[0x808A]=28133;   // yi
   sortTable[0x8264]=28134;   // yi
   sortTable[0x9BA8]=28135;   // yi
   sortTable[0x9D83]=28136;   // yi
   sortTable[0x9DD6]=28137;   // yi
   sortTable[0x7C03]=28138;   // yi
   sortTable[0x9CE6]=28139;   // yi
   sortTable[0x9DE7]=28140;   // yi
   sortTable[0x9F78]=28141;   // yi
   sortTable[0x7BB7]=28142;   // yi
   sortTable[0x7F9B]=28143;   // yi
   sortTable[0x86E6]=28144;   // yi
   sortTable[0x8794]=28145;   // yi
   sortTable[0x8CF9]=28146;   // yi
   sortTable[0x9220]=28147;   // yi
   sortTable[0x7B16]=28148;   // yi
   sortTable[0x8B9B]=28149;   // yi
   sortTable[0x972C]=28150;   // yi
   sortTable[0x9EF3]=28151;   // yi
   sortTable[0x8257]=28152;   // yi
   sortTable[0x8649]=28153;   // yi
   sortTable[0x8898]=28154;   // yi
   sortTable[0x89FA]=28155;   // yi
   sortTable[0x8A52]=28156;   // yi
   sortTable[0x8DE0]=28157;   // yi
   sortTable[0x8F22]=28158;   // yi
   sortTable[0x8FC6]=28159;   // yi
   sortTable[0x9824]=28160;   // yi
   sortTable[0x9DFE]=28161;   // yi
   sortTable[0x7912]=28162;   // yi
   sortTable[0x8189]=28163;   // yi
   sortTable[0x86DC]=28164;   // yi
   sortTable[0x87A0]=28165;   // yi
   sortTable[0x8939]=28166;   // yi
   sortTable[0x9018]=28167;   // yi
   sortTable[0x92A5]=28168;   // yi
   sortTable[0x966D]=28169;   // yi
   sortTable[0x96BF]=28170;   // yi
   sortTable[0x977E]=28171;   // yi
   sortTable[0x9D8D]=28172;   // yi
   sortTable[0x9E5D]=28173;   // yi
   sortTable[0x79C7]=28174;   // yi
   sortTable[0x7FA0]=28175;   // yi
   sortTable[0x8034]=28176;   // yi
   sortTable[0x8094]=28177;   // yi
   sortTable[0x8285]=28178;   // yi
   sortTable[0x82C5]=28179;   // yi
   sortTable[0x8619]=28180;   // yi
   sortTable[0x88FF]=28181;   // yi
   sortTable[0x897C]=28182;   // yi
   sortTable[0x8A4D]=28183;   // yi
   sortTable[0x8A83]=28184;   // yi
   sortTable[0x8C59]=28185;   // yi
   sortTable[0x8CBD]=28186;   // yi
   sortTable[0x943F]=28187;   // yi
   sortTable[0x9A5B]=28188;   // yi
   sortTable[0x9AAE]=28189;   // yi
   sortTable[0x9DCA]=28190;   // yi
   sortTable[0x4E41]=28191;   // yi
   sortTable[0x4E42]=28192;   // yi
   sortTable[0x4E4A]=28193;   // yi
   sortTable[0x4E84]=28194;   // yi
   sortTable[0x4F07]=28195;   // yi
   sortTable[0x4F3F]=28196;   // yi
   sortTable[0x4F41]=28197;   // yi
   sortTable[0x4F87]=28198;   // yi
   sortTable[0x4FCB]=28199;   // yi
   sortTable[0x506F]=28200;   // yi
   sortTable[0x517F]=28201;   // yi
   sortTable[0x519D]=28202;   // yi
   sortTable[0x51D2]=28203;   // yi
   sortTable[0x52AE]=28204;   // yi
   sortTable[0x52DA]=28205;   // yi
   sortTable[0x52E9]=28206;   // yi
   sortTable[0x5307]=28207;   // yi
   sortTable[0x531C]=28208;   // yi
   sortTable[0x541A]=28209;   // yi
   sortTable[0x546D]=28210;   // yi
   sortTable[0x5479]=28211;   // yi
   sortTable[0x5508]=28212;   // yi
   sortTable[0x56C8]=28213;   // yi
   sortTable[0x571B]=28214;   // yi
   sortTable[0x5744]=28215;   // yi
   sortTable[0x57BC]=28216;   // yi
   sortTable[0x57F6]=28217;   // yi
   sortTable[0x58BF]=28218;   // yi
   sortTable[0x58F1]=28219;   // yi
   sortTable[0x5901]=28220;   // yi
   sortTable[0x5A90]=28221;   // yi
   sortTable[0x5AD5]=28222;   // yi
   sortTable[0x5ADB]=28223;   // yi
   sortTable[0x5B04]=28224;   // yi
   sortTable[0x5B11]=28225;   // yi
   sortTable[0x5B1F]=28226;   // yi
   sortTable[0x5B90]=28227;   // yi
   sortTable[0x5BA7]=28228;   // yi
   sortTable[0x5BF1]=28229;   // yi
   sortTable[0x5BF2]=28230;   // yi
   sortTable[0x5CD3]=28231;   // yi
   sortTable[0x5D3A]=28232;   // yi
   sortTable[0x5DA7]=28233;   // yi
   sortTable[0x5DAC]=28234;   // yi
   sortTable[0x5DF8]=28235;   // yi
   sortTable[0x5E1F]=28236;   // yi
   sortTable[0x5E20]=28237;   // yi
   sortTable[0x5E46]=28238;   // yi
   sortTable[0x5EA1]=28239;   // yi
   sortTable[0x5ED9]=28240;   // yi
   sortTable[0x5F0C]=28241;   // yi
   sortTable[0x5F2C]=28242;   // yi
   sortTable[0x5F5B]=28243;   // yi
   sortTable[0x5F5C]=28244;   // yi
   sortTable[0x5F5E]=28245;   // yi
   sortTable[0x6008]=28246;   // yi
   sortTable[0x605E]=28247;   // yi
   sortTable[0x6098]=28248;   // yi
   sortTable[0x60A5]=28249;   // yi
   sortTable[0x61CC]=28250;   // yi
   sortTable[0x6245]=28251;   // yi
   sortTable[0x6246]=28252;   // yi
   sortTable[0x62F8]=28253;   // yi
   sortTable[0x6359]=28254;   // yi
   sortTable[0x639C]=28255;   // yi
   sortTable[0x648E]=28256;   // yi
   sortTable[0x653A]=28257;   // yi
   sortTable[0x6561]=28258;   // yi
   sortTable[0x657C]=28259;   // yi
   sortTable[0x6581]=28260;   // yi
   sortTable[0x65D1]=28261;   // yi
   sortTable[0x6679]=28262;   // yi
   sortTable[0x6686]=28263;   // yi
   sortTable[0x66C0]=28264;   // yi
   sortTable[0x66CE]=28265;   // yi
   sortTable[0x6759]=28266;   // yi
   sortTable[0x678D]=28267;   // yi
   sortTable[0x67BB]=28268;   // yi
   sortTable[0x67C2]=28269;   // yi
   sortTable[0x6818]=28270;   // yi
   sortTable[0x6827]=28271;   // yi
   sortTable[0x683A]=28272;   // yi
   sortTable[0x684B]=28273;   // yi
   sortTable[0x68ED]=28274;   // yi
   sortTable[0x692C]=28275;   // yi
   sortTable[0x6938]=28276;   // yi
   sortTable[0x698F]=28277;   // yi
   sortTable[0x69F8]=28278;   // yi
   sortTable[0x6A8D]=28279;   // yi
   sortTable[0x6AA5]=28280;   // yi
   sortTable[0x6AB9]=28281;   // yi
   sortTable[0x6B25]=28282;   // yi
   sortTable[0x6B2D]=28283;   // yi
   sortTable[0x6B5D]=28284;   // yi
   sortTable[0x6B94]=28285;   // yi
   sortTable[0x6BB9]=28286;   // yi
   sortTable[0x6BC9]=28287;   // yi
   sortTable[0x6CB6]=28288;   // yi
   sortTable[0x6CC6]=28289;   // yi
   sortTable[0x6D02]=28290;   // yi
   sortTable[0x6D22]=28291;   // yi
   sortTable[0x6D42]=28292;   // yi
   sortTable[0x6D65]=28293;   // yi
   sortTable[0x6D73]=28294;   // yi
   sortTable[0x6E59]=28295;   // yi
   sortTable[0x6F69]=28296;   // yi
   sortTable[0x6FBA]=28297;   // yi
   sortTable[0x7037]=28298;   // yi
   sortTable[0x7088]=28299;   // yi
   sortTable[0x7132]=28300;   // yi
   sortTable[0x71A4]=28301;   // yi
   sortTable[0x71AA]=28302;   // yi
   sortTable[0x71BC]=28303;   // yi
   sortTable[0x71DA]=28304;   // yi
   sortTable[0x71E1]=28305;   // yi
   sortTable[0x71F1]=28306;   // yi
   sortTable[0x72CB]=28307;   // yi
   sortTable[0x7348]=28308;   // yi
   sortTable[0x73B4]=28309;   // yi
   sortTable[0x73C6]=28310;   // yi
   sortTable[0x747F]=28311;   // yi
   sortTable[0x74F5]=28312;   // yi
   sortTable[0x7569]=28313;   // yi
   sortTable[0x75EC]=28314;   // yi
   sortTable[0x761E]=28315;   // yi
   sortTable[0x7631]=28316;   // yi
   sortTable[0x7796]=28317;   // yi
   sortTable[0x785B]=28318;   // yi
   sortTable[0x7A26]=28319;   // yi
   sortTable[0x7A53]=28320;   // yi
   sortTable[0x7AE9]=28321;   // yi
   sortTable[0x7C4E]=28322;   // yi
   sortTable[0x7E0A]=28323;   // yi
   sortTable[0x7E79]=28324;   // yi
   sortTable[0x801B]=28325;   // yi
   sortTable[0x82E2]=28326;   // yi
   sortTable[0x8413]=28327;   // yi
   sortTable[0x841F]=28328;   // yi
   sortTable[0x8864]=28329;   // yi
   sortTable[0x886A]=28330;   // yi
   sortTable[0x8A32]=28331;   // yi
   sortTable[0x8A33]=28332;   // yi
   sortTable[0x8A51]=28333;   // yi
   sortTable[0x8B69]=28334;   // yi
   sortTable[0x8C5B]=28335;   // yi
   sortTable[0x8EFC]=28336;   // yi
   sortTable[0x90FC]=28337;   // yi
   sortTable[0x926F]=28338;   // yi
   sortTable[0x93D4]=28339;   // yi
   sortTable[0x984A]=28340;   // yi
   sortTable[0x98F4]=28341;   // yi
   sortTable[0x99C5]=28342;   // yi
   sortTable[0x9BE3]=28343;   // yi
   sortTable[0x56E0]=28344;   // yin
   sortTable[0x97F3]=28345;   // yin
   sortTable[0x5F15]=28346;   // yin
   sortTable[0x5370]=28347;   // yin
   sortTable[0x94F6]=28348;   // yin
   sortTable[0x9280]=28349;   // yin
   sortTable[0x9634]=28350;   // yin
   sortTable[0x9670]=28351;   // yin
   sortTable[0x9690]=28352;   // yin
   sortTable[0x96B1]=28353;   // yin
   sortTable[0x996E]=28354;   // yin
   sortTable[0x98F2]=28355;   // yin
   sortTable[0x541F]=28356;   // yin
   sortTable[0x59FB]=28357;   // yin
   sortTable[0x8693]=28358;   // yin
   sortTable[0x5BC5]=28359;   // yin
   sortTable[0x6BB7]=28360;   // yin
   sortTable[0x5C39]=28361;   // yin
   sortTable[0x6DEB]=28362;   // yin
   sortTable[0x80E4]=28363;   // yin
   sortTable[0x836B]=28364;   // yin
   sortTable[0x8335]=28365;   // yin
   sortTable[0x763E]=28366;   // yin
   sortTable[0x5924]=28367;   // yin
   sortTable[0x6C24]=28368;   // yin
   sortTable[0x5591]=28369;   // yin
   sortTable[0x798B]=28370;   // yin
   sortTable[0x911E]=28371;   // yin
   sortTable[0x57A0]=28372;   // yin
   sortTable[0x8AF2]=28373;   // yin
   sortTable[0x972A]=28374;   // yin
   sortTable[0x5819]=28375;   // yin
   sortTable[0x9A83]=28376;   // yin
   sortTable[0x79F5]=28377;   // yin
   sortTable[0x7D6A]=28378;   // yin
   sortTable[0x95C9]=28379;   // yin
   sortTable[0x5432]=28380;   // yin
   sortTable[0x88C0]=28381;   // yin
   sortTable[0x72FA]=28382;   // yin
   sortTable[0x8A1A]=28383;   // yin
   sortTable[0x8ABE]=28384;   // yin
   sortTable[0x7C8C]=28385;   // yin
   sortTable[0x9173]=28386;   // yin
   sortTable[0x9F82]=28387;   // yin
   sortTable[0x6D07]=28388;   // yin
   sortTable[0x87BE]=28389;   // yin
   sortTable[0x94DF]=28390;   // yin
   sortTable[0x8A14]=28391;   // yin
   sortTable[0x831A]=28392;   // yin
   sortTable[0x87EB]=28393;   // yin
   sortTable[0x9777]=28394;   // yin
   sortTable[0x99F0]=28395;   // yin
   sortTable[0x9BA3]=28396;   // yin
   sortTable[0x9DE3]=28397;   // yin
   sortTable[0x9712]=28398;   // yin
   sortTable[0x8B94]=28399;   // yin
   sortTable[0x967B]=28400;   // yin
   sortTable[0x9682]=28401;   // yin
   sortTable[0x9720]=28402;   // yin
   sortTable[0x98EE]=28403;   // yin
   sortTable[0x78E4]=28404;   // yin
   sortTable[0x8529]=28405;   // yin
   sortTable[0x920F]=28406;   // yin
   sortTable[0x92A6]=28407;   // yin
   sortTable[0x9625]=28408;   // yin
   sortTable[0x9787]=28409;   // yin
   sortTable[0x97FE]=28410;   // yin
   sortTable[0x9F57]=28411;   // yin
   sortTable[0x7892]=28412;   // yin
   sortTable[0x7DF8]=28413;   // yin
   sortTable[0x8491]=28414;   // yin
   sortTable[0x8A21]=28415;   // yin
   sortTable[0x96A0]=28416;   // yin
   sortTable[0x4E51]=28417;   // yin
   sortTable[0x4E5A]=28418;   // yin
   sortTable[0x4F8C]=28419;   // yin
   sortTable[0x5198]=28420;   // yin
   sortTable[0x51D0]=28421;   // yin
   sortTable[0x5656]=28422;   // yin
   sortTable[0x567E]=28423;   // yin
   sortTable[0x569A]=28424;   // yin
   sortTable[0x56D9]=28425;   // yin
   sortTable[0x5701]=28426;   // yin
   sortTable[0x5794]=28427;   // yin
   sortTable[0x57BD]=28428;   // yin
   sortTable[0x5837]=28429;   // yin
   sortTable[0x5A63]=28430;   // yin
   sortTable[0x5A6C]=28431;   // yin
   sortTable[0x5CFE]=28432;   // yin
   sortTable[0x5D1F]=28433;   // yin
   sortTable[0x5D2F]=28434;   // yin
   sortTable[0x5DBE]=28435;   // yin
   sortTable[0x5ED5]=28436;   // yin
   sortTable[0x5EF4]=28437;   // yin
   sortTable[0x6114]=28438;   // yin
   sortTable[0x6147]=28439;   // yin
   sortTable[0x616D]=28440;   // yin
   sortTable[0x6196]=28441;   // yin
   sortTable[0x6197]=28442;   // yin
   sortTable[0x61DA]=28443;   // yin
   sortTable[0x647F]=28444;   // yin
   sortTable[0x65A6]=28445;   // yin
   sortTable[0x6704]=28446;   // yin
   sortTable[0x6836]=28447;   // yin
   sortTable[0x6A83]=28448;   // yin
   sortTable[0x6AAD]=28449;   // yin
   sortTable[0x6ABC]=28450;   // yin
   sortTable[0x6AFD]=28451;   // yin
   sortTable[0x6B45]=28452;   // yin
   sortTable[0x6BA5]=28453;   // yin
   sortTable[0x6CFF]=28454;   // yin
   sortTable[0x6D15]=28455;   // yin
   sortTable[0x6DFE]=28456;   // yin
   sortTable[0x6E5A]=28457;   // yin
   sortTable[0x6EB5]=28458;   // yin
   sortTable[0x6EDB]=28459;   // yin
   sortTable[0x6FE5]=28460;   // yin
   sortTable[0x6FE6]=28461;   // yin
   sortTable[0x70CE]=28462;   // yin
   sortTable[0x72BE]=28463;   // yin
   sortTable[0x730C]=28464;   // yin
   sortTable[0x73E2]=28465;   // yin
   sortTable[0x748C]=28466;   // yin
   sortTable[0x7616]=28467;   // yin
   sortTable[0x764A]=28468;   // yin
   sortTable[0x766E]=28469;   // yin
   sortTable[0x7B43]=28470;   // yin
   sortTable[0x82C2]=28471;   // yin
   sortTable[0x8376]=28472;   // yin
   sortTable[0x852D]=28473;   // yin
   sortTable[0x861F]=28474;   // yin
   sortTable[0x8D7A]=28475;   // yin
   sortTable[0x8D9B]=28476;   // yin
   sortTable[0x8F11]=28477;   // yin
   sortTable[0x921D]=28478;   // yin
   sortTable[0x5E94]=28479;   // ying
   sortTable[0x61C9]=28480;   // ying
   sortTable[0x5F71]=28481;   // ying
   sortTable[0x8425]=28482;   // ying
   sortTable[0x71DF]=28483;   // ying
   sortTable[0x6620]=28484;   // ying
   sortTable[0x82F1]=28485;   // ying
   sortTable[0x786C]=28486;   // ying
   sortTable[0x8FCE]=28487;   // ying
   sortTable[0x8747]=28488;   // ying
   sortTable[0x8805]=28489;   // ying
   sortTable[0x5A74]=28490;   // ying
   sortTable[0x5B30]=28491;   // ying
   sortTable[0x9E70]=28492;   // ying
   sortTable[0x9DF9]=28493;   // ying
   sortTable[0x8424]=28494;   // ying
   sortTable[0x87A2]=28495;   // ying
   sortTable[0x83B9]=28496;   // ying
   sortTable[0x7469]=28497;   // ying
   sortTable[0x9896]=28498;   // ying
   sortTable[0x7A4E]=28499;   // ying
   sortTable[0x76C8]=28500;   // ying
   sortTable[0x8D62]=28501;   // ying
   sortTable[0x6A31]=28502;   // ying
   sortTable[0x988D]=28503;   // ying
   sortTable[0x83BA]=28504;   // ying
   sortTable[0x8367]=28505;   // ying
   sortTable[0x81BA]=28506;   // ying
   sortTable[0x90E2]=28507;   // ying
   sortTable[0x745B]=28508;   // ying
   sortTable[0x701B]=28509;   // ying
   sortTable[0x9E66]=28510;   // ying
   sortTable[0x8426]=28511;   // ying
   sortTable[0x7F28]=28512;   // ying
   sortTable[0x5624]=28513;   // ying
   sortTable[0x6979]=28514;   // ying
   sortTable[0x7F42]=28515;   // ying
   sortTable[0x5B34]=28516;   // ying
   sortTable[0x8314]=28517;   // ying
   sortTable[0x5AB5]=28518;   // ying
   sortTable[0x6EE2]=28519;   // ying
   sortTable[0x763F]=28520;   // ying
   sortTable[0x6F46]=28521;   // ying
   sortTable[0x748E]=28522;   // ying
   sortTable[0x6484]=28523;   // ying
   sortTable[0x9DEA]=28524;   // ying
   sortTable[0x7F43]=28525;   // ying
   sortTable[0x84E5]=28526;   // ying
   sortTable[0x7EEC]=28527;   // ying
   sortTable[0x97FA]=28528;   // ying
   sortTable[0x9719]=28529;   // ying
   sortTable[0x7C5D]=28530;   // ying
   sortTable[0x9795]=28531;   // ying
   sortTable[0x9533]=28532;   // ying
   sortTable[0x8B7B]=28533;   // ying
   sortTable[0x9D2C]=28534;   // ying
   sortTable[0x877F]=28535;   // ying
   sortTable[0x8D0F]=28536;   // ying
   sortTable[0x77E8]=28537;   // ying
   sortTable[0x892E]=28538;   // ying
   sortTable[0x8EC8]=28539;   // ying
   sortTable[0x7C6F]=28540;   // ying
   sortTable[0x8B0D]=28541;   // ying
   sortTable[0x944D]=28542;   // ying
   sortTable[0x9895]=28543;   // ying
   sortTable[0x9C66]=28544;   // ying
   sortTable[0x7E08]=28545;   // ying
   sortTable[0x85C0]=28546;   // ying
   sortTable[0x8621]=28547;   // ying
   sortTable[0x9D91]=28548;   // ying
   sortTable[0x9DAF]=28549;   // ying
   sortTable[0x5040]=28550;   // ying
   sortTable[0x50CC]=28551;   // ying
   sortTable[0x5568]=28552;   // ying
   sortTable[0x55B6]=28553;   // ying
   sortTable[0x565F]=28554;   // ying
   sortTable[0x56B6]=28555;   // ying
   sortTable[0x584B]=28556;   // ying
   sortTable[0x5A96]=28557;   // ying
   sortTable[0x5AC8]=28558;   // ying
   sortTable[0x5B46]=28559;   // ying
   sortTable[0x5B7E]=28560;   // ying
   sortTable[0x5DCA]=28561;   // ying
   sortTable[0x5EEE]=28562;   // ying
   sortTable[0x5FDC]=28563;   // ying
   sortTable[0x6125]=28564;   // ying
   sortTable[0x646C]=28565;   // ying
   sortTable[0x650D]=28566;   // ying
   sortTable[0x6516]=28567;   // ying
   sortTable[0x651A]=28568;   // ying
   sortTable[0x668E]=28569;   // ying
   sortTable[0x6720]=28570;   // ying
   sortTable[0x685C]=28571;   // ying
   sortTable[0x68AC]=28572;   // ying
   sortTable[0x6AFB]=28573;   // ying
   sortTable[0x6AFF]=28574;   // ying
   sortTable[0x6D67]=28575;   // ying
   sortTable[0x6E36]=28576;   // ying
   sortTable[0x6E81]=28577;   // ying
   sortTable[0x6E8B]=28578;   // ying
   sortTable[0x6F41]=28579;   // ying
   sortTable[0x6FD9]=28580;   // ying
   sortTable[0x6FDA]=28581;   // ying
   sortTable[0x6FF4]=28582;   // ying
   sortTable[0x7005]=28583;   // ying
   sortTable[0x7020]=28584;   // ying
   sortTable[0x702F]=28585;   // ying
   sortTable[0x7034]=28586;   // ying
   sortTable[0x7050]=28587;   // ying
   sortTable[0x705C]=28588;   // ying
   sortTable[0x7150]=28589;   // ying
   sortTable[0x7192]=28590;   // ying
   sortTable[0x73F1]=28591;   // ying
   sortTable[0x7484]=28592;   // ying
   sortTable[0x74D4]=28593;   // ying
   sortTable[0x7507]=28594;   // ying
   sortTable[0x7516]=28595;   // ying
   sortTable[0x766D]=28596;   // ying
   sortTable[0x76C1]=28597;   // ying
   sortTable[0x78A4]=28598;   // ying
   sortTable[0x792F]=28599;   // ying
   sortTable[0x7DD3]=28600;   // ying
   sortTable[0x7E04]=28601;   // ying
   sortTable[0x7E93]=28602;   // ying
   sortTable[0x7F4C]=28603;   // ying
   sortTable[0x81A1]=28604;   // ying
   sortTable[0x843E]=28605;   // ying
   sortTable[0x86CD]=28606;   // ying
   sortTable[0x8767]=28607;   // ying
   sortTable[0x8833]=28608;   // ying
   sortTable[0x89AE]=28609;   // ying
   sortTable[0x8B4D]=28610;   // ying
   sortTable[0x8CCF]=28611;   // ying
   sortTable[0x9348]=28612;   // ying
   sortTable[0x93A3]=28613;   // ying
   sortTable[0x941B]=28614;   // ying
   sortTable[0x9834]=28615;   // ying
   sortTable[0x9DA7]=28616;   // ying
   sortTable[0x9E0E]=28617;   // ying
   sortTable[0x9E1A]=28618;   // ying
   sortTable[0x54DF]=28619;   // yo
   sortTable[0x55B2]=28620;   // yo
   sortTable[0x5537]=28621;   // yo
   sortTable[0x7528]=28622;   // yong
   sortTable[0x6C38]=28623;   // yong
   sortTable[0x52C7]=28624;   // yong
   sortTable[0x6D8C]=28625;   // yong
   sortTable[0x62E5]=28626;   // yong
   sortTable[0x64C1]=28627;   // yong
   sortTable[0x6CF3]=28628;   // yong
   sortTable[0x5EB8]=28629;   // yong
   sortTable[0x96CD]=28630;   // yong
   sortTable[0x4F63]=28631;   // yong
   sortTable[0x548F]=28632;   // yong
   sortTable[0x9095]=28633;   // yong
   sortTable[0x8E0A]=28634;   // yong
   sortTable[0x86F9]=28635;   // yong
   sortTable[0x58C5]=28636;   // yong
   sortTable[0x9954]=28637;   // yong
   sortTable[0x607F]=28638;   // yong
   sortTable[0x9899]=28639;   // yong
   sortTable[0x6175]=28640;   // yong
   sortTable[0x4FD1]=28641;   // yong
   sortTable[0x752C]=28642;   // yong
   sortTable[0x75C8]=28643;   // yong
   sortTable[0x5889]=28644;   // yong
   sortTable[0x81C3]=28645;   // yong
   sortTable[0x955B]=28646;   // yong
   sortTable[0x5581]=28647;   // yong
   sortTable[0x9852]=28648;   // yong
   sortTable[0x9118]=28649;   // yong
   sortTable[0x799C]=28650;   // yong
   sortTable[0x9CD9]=28651;   // yong
   sortTable[0x96DD]=28652;   // yong
   sortTable[0x8A60]=28653;   // yong
   sortTable[0x9BD2]=28654;   // yong
   sortTable[0x9C2B]=28655;   // yong
   sortTable[0x9CAC]=28656;   // yong
   sortTable[0x8E34]=28657;   // yong
   sortTable[0x90FA]=28658;   // yong
   sortTable[0x509B]=28659;   // yong
   sortTable[0x50AD]=28660;   // yong
   sortTable[0x52C8]=28661;   // yong
   sortTable[0x55C8]=28662;   // yong
   sortTable[0x5670]=28663;   // yong
   sortTable[0x57C7]=28664;   // yong
   sortTable[0x584E]=28665;   // yong
   sortTable[0x5ADE]=28666;   // yong
   sortTable[0x5D71]=28667;   // yong
   sortTable[0x5EF1]=28668;   // yong
   sortTable[0x5F6E]=28669;   // yong
   sortTable[0x603A]=28670;   // yong
   sortTable[0x6080]=28671;   // yong
   sortTable[0x60E5]=28672;   // yong
   sortTable[0x6111]=28673;   // yong
   sortTable[0x6139]=28674;   // yong
   sortTable[0x6142]=28675;   // yong
   sortTable[0x63D8]=28676;   // yong
   sortTable[0x67E1]=28677;   // yong
   sortTable[0x6810]=28678;   // yong
   sortTable[0x69E6]=28679;   // yong
   sortTable[0x6E67]=28680;   // yong
   sortTable[0x6EFD]=28681;   // yong
   sortTable[0x6FAD]=28682;   // yong
   sortTable[0x7049]=28683;   // yong
   sortTable[0x7245]=28684;   // yong
   sortTable[0x7655]=28685;   // yong
   sortTable[0x7670]=28686;   // yong
   sortTable[0x783D]=28687;   // yong
   sortTable[0x7867]=28688;   // yong
   sortTable[0x82DA]=28689;   // yong
   sortTable[0x919F]=28690;   // yong
   sortTable[0x93DE]=28691;   // yong
   sortTable[0x9C45]=28692;   // yong
   sortTable[0x9DDB]=28693;   // yong
   sortTable[0x6709]=28694;   // you
   sortTable[0x53C8]=28695;   // you
   sortTable[0x7531]=28696;   // you
   sortTable[0x6CB9]=28697;   // you
   sortTable[0x53F3]=28698;   // you
   sortTable[0x53CB]=28699;   // you
   sortTable[0x6E38]=28700;   // you
   sortTable[0x4F18]=28701;   // you
   sortTable[0x512A]=28702;   // you
   sortTable[0x5C24]=28703;   // you
   sortTable[0x5E7C]=28704;   // you
   sortTable[0x72B9]=28705;   // you
   sortTable[0x7336]=28706;   // you
   sortTable[0x90AE]=28707;   // you
   sortTable[0x90F5]=28708;   // you
   sortTable[0x5FE7]=28709;   // you
   sortTable[0x6182]=28710;   // you
   sortTable[0x94C0]=28711;   // you
   sortTable[0x923E]=28712;   // you
   sortTable[0x60A0]=28713;   // you
   sortTable[0x8BF1]=28714;   // you
   sortTable[0x8A98]=28715;   // you
   sortTable[0x4F51]=28716;   // you
   sortTable[0x5E7D]=28717;   // you
   sortTable[0x9149]=28718;   // you
   sortTable[0x7950]=28719;   // you
   sortTable[0x6538]=28720;   // you
   sortTable[0x91C9]=28721;   // you
   sortTable[0x5BA5]=28722;   // you
   sortTable[0x7337]=28723;   // you
   sortTable[0x86B4]=28724;   // you
   sortTable[0x7256]=28725;   // you
   sortTable[0x9EDD]=28726;   // you
   sortTable[0x5363]=28727;   // you
   sortTable[0x75A3]=28728;   // you
   sortTable[0x67DA]=28729;   // you
   sortTable[0x56FF]=28730;   // you
   sortTable[0x83B8]=28731;   // you
   sortTable[0x86B0]=28732;   // you
   sortTable[0x4F91]=28733;   // you
   sortTable[0x5466]=28734;   // you
   sortTable[0x9C7F]=28735;   // you
   sortTable[0x5C22]=28736;   // you
   sortTable[0x83A0]=28737;   // you
   sortTable[0x839C]=28738;   // you
   sortTable[0x9F2C]=28739;   // you
   sortTable[0x8763]=28740;   // you
   sortTable[0x900C]=28741;   // you
   sortTable[0x8F36]=28742;   // you
   sortTable[0x7F91]=28743;   // you
   sortTable[0x9E80]=28744;   // you
   sortTable[0x94D5]=28745;   // you
   sortTable[0x8030]=28746;   // you
   sortTable[0x9C89]=28747;   // you
   sortTable[0x904A]=28748;   // you
   sortTable[0x870F]=28749;   // you
   sortTable[0x913E]=28750;   // you
   sortTable[0x8FF6]=28751;   // you
   sortTable[0x8A27]=28752;   // you
   sortTable[0x9B77]=28753;   // you
   sortTable[0x9B8B]=28754;   // you
   sortTable[0x80AC]=28755;   // you
   sortTable[0x811C]=28756;   // you
   sortTable[0x82C3]=28757;   // you
   sortTable[0x7989]=28758;   // you
   sortTable[0x79DE]=28759;   // you
   sortTable[0x7E8B]=28760;   // you
   sortTable[0x4E23]=28761;   // you
   sortTable[0x4EB4]=28762;   // you
   sortTable[0x5064]=28763;   // you
   sortTable[0x54CA]=28764;   // you
   sortTable[0x5500]=28765;   // you
   sortTable[0x5698]=28766;   // you
   sortTable[0x59F7]=28767;   // you
   sortTable[0x5B67]=28768;   // you
   sortTable[0x5CDF]=28769;   // you
   sortTable[0x5CF3]=28770;   // you
   sortTable[0x5EAE]=28771;   // you
   sortTable[0x6023]=28772;   // you
   sortTable[0x602E]=28773;   // you
   sortTable[0x61EE]=28774;   // you
   sortTable[0x65BF]=28775;   // you
   sortTable[0x682F]=28776;   // you
   sortTable[0x6884]=28777;   // you
   sortTable[0x6962]=28778;   // you
   sortTable[0x69F1]=28779;   // you
   sortTable[0x6ACC]=28780;   // you
   sortTable[0x6AFE]=28781;   // you
   sortTable[0x6C8B]=28782;   // you
   sortTable[0x6CD1]=28783;   // you
   sortTable[0x6D5F]=28784;   // you
   sortTable[0x6E75]=28785;   // you
   sortTable[0x6EFA]=28786;   // you
   sortTable[0x7000]=28787;   // you
   sortTable[0x7257]=28788;   // you
   sortTable[0x7270]=28789;   // you
   sortTable[0x72D6]=28790;   // you
   sortTable[0x7CFF]=28791;   // you
   sortTable[0x7F90]=28792;   // you
   sortTable[0x8048]=28793;   // you
   sortTable[0x848F]=28794;   // you
   sortTable[0x8555]=28795;   // you
   sortTable[0x8C81]=28796;   // you
   sortTable[0x8F0F]=28797;   // you
   sortTable[0x9030]=28798;   // you
   sortTable[0x916D]=28799;   // you
   sortTable[0x92AA]=28800;   // you
   sortTable[0x99C0]=28801;   // you
   sortTable[0x4E8E]=28802;   // yu
   sortTable[0x4E0E]=28803;   // yu
   sortTable[0x8207]=28804;   // yu
   sortTable[0x80B2]=28805;   // yu
   sortTable[0x96E8]=28806;   // yu
   sortTable[0x9C7C]=28807;   // yu
   sortTable[0x9B5A]=28808;   // yu
   sortTable[0x4F59]=28809;   // yu
   sortTable[0x8BED]=28810;   // yu
   sortTable[0x9047]=28811;   // yu
   sortTable[0x8A9E]=28812;   // yu
   sortTable[0x9884]=28813;   // yu
   sortTable[0x9810]=28814;   // yu
   sortTable[0x5B87]=28815;   // yu
   sortTable[0x6108]=28816;   // yu
   sortTable[0x57DF]=28817;   // yu
   sortTable[0x7389]=28818;   // yu
   sortTable[0x72F1]=28819;   // yu
   sortTable[0x7344]=28820;   // yu
   sortTable[0x6109]=28821;   // yu
   sortTable[0x6B32]=28822;   // yu
   sortTable[0x5FA1]=28823;   // yu
   sortTable[0x6E14]=28824;   // yu
   sortTable[0x6F01]=28825;   // yu
   sortTable[0x4E88]=28826;   // yu
   sortTable[0x8C6B]=28827;   // yu
   sortTable[0x8A89]=28828;   // yu
   sortTable[0x8B7D]=28829;   // yu
   sortTable[0x5C7F]=28830;   // yu
   sortTable[0x5DBC]=28831;   // yu
   sortTable[0x88D5]=28832;   // yu
   sortTable[0x7FBD]=28833;   // yu
   sortTable[0x5A31]=28834;   // yu
   sortTable[0x7AFD]=28835;   // yu
   sortTable[0x5A1B]=28836;   // yu
   sortTable[0x90C1]=28837;   // yu
   sortTable[0x5BD3]=28838;   // yu
   sortTable[0x8206]=28839;   // yu
   sortTable[0x8F3F]=28840;   // yu
   sortTable[0x611A]=28841;   // yu
   sortTable[0x8C15]=28842;   // yu
   sortTable[0x9980]=28843;   // yu
   sortTable[0x65BC]=28844;   // yu
   sortTable[0x865E]=28845;   // yu
   sortTable[0x903E]=28846;   // yu
   sortTable[0x6D74]=28847;   // yu
   sortTable[0x55BB]=28848;   // yu
   sortTable[0x79B9]=28849;   // yu
   sortTable[0x745C]=28850;   // yu
   sortTable[0x6BD3]=28851;   // yu
   sortTable[0x4FDE]=28852;   // yu
   sortTable[0x6B24]=28853;   // yu
   sortTable[0x715C]=28854;   // yu
   sortTable[0x9685]=28855;   // yu
   sortTable[0x6986]=28856;   // yu
   sortTable[0x8FC2]=28857;   // yu
   sortTable[0x81FE]=28858;   // yu
   sortTable[0x5EBE]=28859;   // yu
   sortTable[0x9A6D]=28860;   // yu
   sortTable[0x79BA]=28861;   // yu
   sortTable[0x8C00]=28862;   // yu
   sortTable[0x94B0]=28863;   // yu
   sortTable[0x807F]=28864;   // yu
   sortTable[0x6DE4]=28865;   // yu
   sortTable[0x59AA]=28866;   // yu
   sortTable[0x9B3B]=28867;   // yu
   sortTable[0x6E1D]=28868;   // yu
   sortTable[0x6631]=28869;   // yu
   sortTable[0x76C2]=28870;   // yu
   sortTable[0x4FE3]=28871;   // yu
   sortTable[0x9E6C]=28872;   // yu
   sortTable[0x9AC3]=28873;   // yu
   sortTable[0x5CEA]=28874;   // yu
   sortTable[0x9E46]=28875;   // yu
   sortTable[0x996B]=28876;   // yu
   sortTable[0x8174]=28877;   // yu
   sortTable[0x9608]=28878;   // yu
   sortTable[0x9F89]=28879;   // yu
   sortTable[0x96E9]=28880;   // yu
   sortTable[0x5709]=28881;   // yu
   sortTable[0x8438]=28882;   // yu
   sortTable[0x828B]=28883;   // yu
   sortTable[0x5D5B]=28884;   // yu
   sortTable[0x89CE]=28885;   // yu
   sortTable[0x7AB3]=28886;   // yu
   sortTable[0x8E30]=28887;   // yu
   sortTable[0x7AAC]=28888;   // yu
   sortTable[0x7600]=28889;   // yu
   sortTable[0x7EA1]=28890;   // yu
   sortTable[0x8201]=28891;   // yu
   sortTable[0x59A4]=28892;   // yu
   sortTable[0x5704]=28893;   // yu
   sortTable[0x63C4]=28894;   // yu
   sortTable[0x872E]=28895;   // yu
   sortTable[0x8080]=28896;   // yu
   sortTable[0x71E0]=28897;   // yu
   sortTable[0x9079]=28898;   // yu
   sortTable[0x4F1B]=28899;   // yu
   sortTable[0x7610]=28900;   // yu
   sortTable[0x8753]=28901;   // yu
   sortTable[0x8915]=28902;   // yu
   sortTable[0x8581]=28903;   // yu
   sortTable[0x77DE]=28904;   // yu
   sortTable[0x8F5D]=28905;   // yu
   sortTable[0x9D25]=28906;   // yu
   sortTable[0x8248]=28907;   // yu
   sortTable[0x84E3]=28908;   // yu
   sortTable[0x9098]=28909;   // yu
   sortTable[0x9918]=28910;   // yu
   sortTable[0x72F3]=28911;   // yu
   sortTable[0x923A]=28912;   // yu
   sortTable[0x7C5E]=28913;   // yu
   sortTable[0x9105]=28914;   // yu
   sortTable[0x776E]=28915;   // yu
   sortTable[0x9B31]=28916;   // yu
   sortTable[0x7964]=28917;   // yu
   sortTable[0x8C90]=28918;   // yu
   sortTable[0x7907]=28919;   // yu
   sortTable[0x7E58]=28920;   // yu
   sortTable[0x8B23]=28921;   // yu
   sortTable[0x92CA]=28922;   // yu
   sortTable[0x791C]=28923;   // yu
   sortTable[0x8245]=28924;   // yu
   sortTable[0x842D]=28925;   // yu
   sortTable[0x8577]=28926;   // yu
   sortTable[0x9828]=28927;   // yu
   sortTable[0x9BBD]=28928;   // yu
   sortTable[0x9E8C]=28929;   // yu
   sortTable[0x9FA5]=28930;   // yu
   sortTable[0x79A6]=28931;   // yu
   sortTable[0x9A48]=28932;   // yu
   sortTable[0x9E12]=28933;   // yu
   sortTable[0x7F6D]=28934;   // yu
   sortTable[0x7FAD]=28935;   // yu
   sortTable[0x9325]=28936;   // yu
   sortTable[0x9D2A]=28937;   // yu
   sortTable[0x7872]=28938;   // yu
   sortTable[0x7A7B]=28939;   // yu
   sortTable[0x8330]=28940;   // yu
   sortTable[0x9289]=28941;   // yu
   sortTable[0x7DCE]=28942;   // yu
   sortTable[0x8ADB]=28943;   // yu
   sortTable[0x7C72]=28944;   // yu
   sortTable[0x8EC9]=28945;   // yu
   sortTable[0x942D]=28946;   // yu
   sortTable[0x95BE]=28947;   // yu
   sortTable[0x98EB]=28948;   // yu
   sortTable[0x9A1F]=28949;   // yu
   sortTable[0x9C05]=28950;   // yu
   sortTable[0x9CFF]=28951;   // yu
   sortTable[0x76D3]=28952;   // yu
   sortTable[0x7A22]=28953;   // yu
   sortTable[0x7BFD]=28954;   // yu
   sortTable[0x7C45]=28955;   // yu
   sortTable[0x831F]=28956;   // yu
   sortTable[0x8362]=28957;   // yu
   sortTable[0x84AE]=28958;   // yu
   sortTable[0x854D]=28959;   // yu
   sortTable[0x860C]=28960;   // yu
   sortTable[0x861B]=28961;   // yu
   sortTable[0x87B8]=28962;   // yu
   sortTable[0x935D]=28963;   // yu
   sortTable[0x9653]=28964;   // yu
   sortTable[0x96D3]=28965;   // yu
   sortTable[0x9947]=28966;   // yu
   sortTable[0x9C4A]=28967;   // yu
   sortTable[0x9D52]=28968;   // yu
   sortTable[0x9DF8]=28969;   // yu
   sortTable[0x9E06]=28970;   // yu
   sortTable[0x4E7B]=28971;   // yu
   sortTable[0x4E90]=28972;   // yu
   sortTable[0x4F03]=28973;   // yu
   sortTable[0x4FC1]=28974;   // yu
   sortTable[0x4FFC]=28975;   // yu
   sortTable[0x504A]=28976;   // yu
   sortTable[0x50B4]=28977;   // yu
   sortTable[0x5125]=28978;   // yu
   sortTable[0x516A]=28979;   // yu
   sortTable[0x532C]=28980;   // yu
   sortTable[0x5539]=28981;   // yu
   sortTable[0x5585]=28982;   // yu
   sortTable[0x5590]=28983;   // yu
   sortTable[0x55A9]=28984;   // yu
   sortTable[0x564A]=28985;   // yu
   sortTable[0x5673]=28986;   // yu
   sortTable[0x572B]=28987;   // yu
   sortTable[0x5809]=28988;   // yu
   sortTable[0x5823]=28989;   // yu
   sortTable[0x582C]=28990;   // yu
   sortTable[0x5A2F]=28991;   // yu
   sortTable[0x5A80]=28992;   // yu
   sortTable[0x5AD7]=28993;   // yu
   sortTable[0x5B29]=28994;   // yu
   sortTable[0x5BD9]=28995;   // yu
   sortTable[0x5CFF]=28996;   // yu
   sortTable[0x5D33]=28997;   // yu
   sortTable[0x5D4E]=28998;   // yu
   sortTable[0x5D8E]=28999;   // yu
   sortTable[0x5EBD]=29000;   // yu
   sortTable[0x5F67]=29001;   // yu
   sortTable[0x5FEC]=29002;   // yu
   sortTable[0x6086]=29003;   // yu
   sortTable[0x60D0]=29004;   // yu
   sortTable[0x617E]=29005;   // yu
   sortTable[0x61D9]=29006;   // yu
   sortTable[0x622B]=29007;   // yu
   sortTable[0x625C]=29008;   // yu
   sortTable[0x6275]=29009;   // yu
   sortTable[0x6327]=29010;   // yu
   sortTable[0x6554]=29011;   // yu
   sortTable[0x6594]=29012;   // yu
   sortTable[0x659E]=29013;   // yu
   sortTable[0x65D5]=29014;   // yu
   sortTable[0x65DF]=29015;   // yu
   sortTable[0x6745]=29016;   // yu
   sortTable[0x6859]=29017;   // yu
   sortTable[0x68DB]=29018;   // yu
   sortTable[0x68DC]=29019;   // yu
   sortTable[0x68EB]=29020;   // yu
   sortTable[0x6940]=29021;   // yu
   sortTable[0x6961]=29022;   // yu
   sortTable[0x6970]=29023;   // yu
   sortTable[0x6AF2]=29024;   // yu
   sortTable[0x6B0E]=29025;   // yu
   sortTable[0x6B1D]=29026;   // yu
   sortTable[0x6B48]=29027;   // yu
   sortTable[0x6B5F]=29028;   // yu
   sortTable[0x6B76]=29029;   // yu
   sortTable[0x6BFA]=29030;   // yu
   sortTable[0x6DE2]=29031;   // yu
   sortTable[0x6DEF]=29032;   // yu
   sortTable[0x6E61]=29033;   // yu
   sortTable[0x6EEA]=29034;   // yu
   sortTable[0x6F4F]=29035;   // yu
   sortTable[0x6F9A]=29036;   // yu
   sortTable[0x6F9E]=29037;   // yu
   sortTable[0x6FA6]=29038;   // yu
   sortTable[0x706A]=29039;   // yu
   sortTable[0x7134]=29040;   // yu
   sortTable[0x71CF]=29041;   // yu
   sortTable[0x7229]=29042;   // yu
   sortTable[0x724F]=29043;   // yu
   sortTable[0x7397]=29044;   // yu
   sortTable[0x7399]=29045;   // yu
   sortTable[0x7419]=29046;   // yu
   sortTable[0x7440]=29047;   // yu
   sortTable[0x74B5]=29048;   // yu
   sortTable[0x756D]=29049;   // yu
   sortTable[0x7609]=29050;   // yu
   sortTable[0x7652]=29051;   // yu
   sortTable[0x7821]=29052;   // yu
   sortTable[0x7862]=29053;   // yu
   sortTable[0x7916]=29054;   // yu
   sortTable[0x79D7]=29055;   // yu
   sortTable[0x7A36]=29056;   // yu
   sortTable[0x7A65]=29057;   // yu
   sortTable[0x7B8A]=29058;   // yu
   sortTable[0x7D06]=29059;   // yu
   sortTable[0x828C]=29060;   // yu
   sortTable[0x842E]=29061;   // yu
   sortTable[0x84F9]=29062;   // yu
   sortTable[0x8676]=29063;   // yu
   sortTable[0x871F]=29064;   // yu
   sortTable[0x8867]=29065;   // yu
   sortTable[0x89A6]=29066;   // yu
   sortTable[0x8AED]=29067;   // yu
   sortTable[0x8F0D]=29068;   // yu
   sortTable[0x8FC3]=29069;   // yu
   sortTable[0x9033]=29070;   // yu
   sortTable[0x9151]=29071;   // yu
   sortTable[0x91A7]=29072;   // yu
   sortTable[0x92D9]=29073;   // yu
   sortTable[0x9731]=29074;   // yu
   sortTable[0x99AD]=29075;   // yu
   sortTable[0x9AAC]=29076;   // yu
   sortTable[0x9B30]=29077;   // yu
   sortTable[0x9B4A]=29078;   // yu
   sortTable[0x9BF2]=29079;   // yu
   sortTable[0x9DE0]=29080;   // yu
   sortTable[0x9F6C]=29081;   // yu
   sortTable[0x5458]=29082;   // yuan
   sortTable[0x54E1]=29083;   // yuan
   sortTable[0x539F]=29084;   // yuan
   sortTable[0x8FDC]=29085;   // yuan
   sortTable[0x9060]=29086;   // yuan
   sortTable[0x9662]=29087;   // yuan
   sortTable[0x613F]=29088;   // yuan
   sortTable[0x6E90]=29089;   // yuan
   sortTable[0x5143]=29090;   // yuan
   sortTable[0x56ED]=29091;   // yuan
   sortTable[0x5712]=29092;   // yuan
   sortTable[0x5706]=29093;   // yuan
   sortTable[0x5713]=29094;   // yuan
   sortTable[0x63F4]=29095;   // yuan
   sortTable[0x733F]=29096;   // yuan
   sortTable[0x7F18]=29097;   // yuan
   sortTable[0x7DE3]=29098;   // yuan
   sortTable[0x51A4]=29099;   // yuan
   sortTable[0x6028]=29100;   // yuan
   sortTable[0x8881]=29101;   // yuan
   sortTable[0x6E0A]=29102;   // yuan
   sortTable[0x82D1]=29103;   // yuan
   sortTable[0x57A3]=29104;   // yuan
   sortTable[0x9E33]=29105;   // yuan
   sortTable[0x8F95]=29106;   // yuan
   sortTable[0x7230]=29107;   // yuan
   sortTable[0x63BE]=29108;   // yuan
   sortTable[0x5A9B]=29109;   // yuan
   sortTable[0x6C85]=29110;   // yuan
   sortTable[0x9F0B]=29111;   // yuan
   sortTable[0x7457]=29112;   // yuan
   sortTable[0x9E22]=29113;   // yuan
   sortTable[0x8788]=29114;   // yuan
   sortTable[0x6A7C]=29115;   // yuan
   sortTable[0x7722]=29116;   // yuan
   sortTable[0x586C]=29117;   // yuan
   sortTable[0x9E53]=29118;   // yuan
   sortTable[0x875D]=29119;   // yuan
   sortTable[0x8B1C]=29120;   // yuan
   sortTable[0x57B8]=29121;   // yuan
   sortTable[0x8696]=29122;   // yuan
   sortTable[0x9B6D]=29123;   // yuan
   sortTable[0x7BA2]=29124;   // yuan
   sortTable[0x9228]=29125;   // yuan
   sortTable[0x8312]=29126;   // yuan
   sortTable[0x7B0E]=29127;   // yuan
   sortTable[0x8099]=29128;   // yuan
   sortTable[0x870E]=29129;   // yuan
   sortTable[0x9A35]=29130;   // yuan
   sortTable[0x92FA]=29131;   // yuan
   sortTable[0x90A7]=29132;   // yuan
   sortTable[0x7FB1]=29133;   // yuan
   sortTable[0x8D20]=29134;   // yuan
   sortTable[0x9858]=29135;   // yuan
   sortTable[0x876F]=29136;   // yuan
   sortTable[0x884F]=29137;   // yuan
   sortTable[0x9D77]=29138;   // yuan
   sortTable[0x847E]=29139;   // yuan
   sortTable[0x8911]=29140;   // yuan
   sortTable[0x908D]=29141;   // yuan
   sortTable[0x9EFF]=29142;   // yuan
   sortTable[0x9F18]=29143;   // yuan
   sortTable[0x9F1D]=29144;   // yuan
   sortTable[0x88F7]=29145;   // yuan
   sortTable[0x9DA2]=29146;   // yuan
   sortTable[0x7990]=29147;   // yuan
   sortTable[0x7E01]=29148;   // yuan
   sortTable[0x8597]=29149;   // yuan
   sortTable[0x8C9F]=29150;   // yuan
   sortTable[0x99CC]=29151;   // yuan
   sortTable[0x9CF6]=29152;   // yuan
   sortTable[0x5086]=29153;   // yuan
   sortTable[0x5186]=29154;   // yuan
   sortTable[0x5248]=29155;   // yuan
   sortTable[0x53A1]=29156;   // yuan
   sortTable[0x53B5]=29157;   // yuan
   sortTable[0x566E]=29158;   // yuan
   sortTable[0x56E6]=29159;   // yuan
   sortTable[0x570E]=29160;   // yuan
   sortTable[0x5917]=29161;   // yuan
   sortTable[0x59B4]=29162;   // yuan
   sortTable[0x5AB4]=29163;   // yuan
   sortTable[0x5AC4]=29164;   // yuan
   sortTable[0x5B3D]=29165;   // yuan
   sortTable[0x5BC3]=29166;   // yuan
   sortTable[0x6081]=29167;   // yuan
   sortTable[0x60CC]=29168;   // yuan
   sortTable[0x676C]=29169;   // yuan
   sortTable[0x68E9]=29170;   // yuan
   sortTable[0x699E]=29171;   // yuan
   sortTable[0x69AC]=29172;   // yuan
   sortTable[0x6ADE]=29173;   // yuan
   sortTable[0x6DF5]=29174;   // yuan
   sortTable[0x6E01]=29175;   // yuan
   sortTable[0x6E06]=29176;   // yuan
   sortTable[0x6E15]=29177;   // yuan
   sortTable[0x6E72]=29178;   // yuan
   sortTable[0x6E92]=29179;   // yuan
   sortTable[0x7041]=29180;   // yuan
   sortTable[0x7328]=29181;   // yuan
   sortTable[0x7342]=29182;   // yuan
   sortTable[0x76F6]=29183;   // yuan
   sortTable[0x849D]=29184;   // yuan
   sortTable[0x84AC]=29185;   // yuan
   sortTable[0x8735]=29186;   // yuan
   sortTable[0x88EB]=29187;   // yuan
   sortTable[0x8924]=29188;   // yuan
   sortTable[0x8F45]=29189;   // yuan
   sortTable[0x903A]=29190;   // yuan
   sortTable[0x915B]=29191;   // yuan
   sortTable[0x93B1]=29192;   // yuan
   sortTable[0x9D1B]=29193;   // yuan
   sortTable[0x9DB0]=29194;   // yuan
   sortTable[0x6708]=29195;   // yue
   sortTable[0x8D8A]=29196;   // yue
   sortTable[0x7EA6]=29197;   // yue
   sortTable[0x7D04]=29198;   // yue
   sortTable[0x8DC3]=29199;   // yue
   sortTable[0x8E8D]=29200;   // yue
   sortTable[0x60A6]=29201;   // yue
   sortTable[0x6085]=29202;   // yue
   sortTable[0x9605]=29203;   // yue
   sortTable[0x95B1]=29204;   // yue
   sortTable[0x66F0]=29205;   // yue
   sortTable[0x5CB3]=29206;   // yue
   sortTable[0x7CA4]=29207;   // yue
   sortTable[0x94BA]=29208;   // yue
   sortTable[0x5216]=29209;   // yue
   sortTable[0x6A3E]=29210;   // yue
   sortTable[0x9FA0]=29211;   // yue
   sortTable[0x7C65]=29212;   // yue
   sortTable[0x7039]=29213;   // yue
   sortTable[0x8ECF]=29214;   // yue
   sortTable[0x9E11]=29215;   // yue
   sortTable[0x793F]=29216;   // yue
   sortTable[0x79B4]=29217;   // yue
   sortTable[0x77F1]=29218;   // yue
   sortTable[0x7CB5]=29219;   // yue
   sortTable[0x8DC0]=29220;   // yue
   sortTable[0x7BB9]=29221;   // yue
   sortTable[0x868F]=29222;   // yue
   sortTable[0x7C46]=29223;   // yue
   sortTable[0x9EE6]=29224;   // yue
   sortTable[0x7C70]=29225;   // yue
   sortTable[0x8625]=29226;   // yue
   sortTable[0x599C]=29227;   // yue
   sortTable[0x5B33]=29228;   // yue
   sortTable[0x5C84]=29229;   // yue
   sortTable[0x5DBD]=29230;   // yue
   sortTable[0x5F5F]=29231;   // yue
   sortTable[0x5F60]=29232;   // yue
   sortTable[0x6071]=29233;   // yue
   sortTable[0x6209]=29234;   // yue
   sortTable[0x6288]=29235;   // yue
   sortTable[0x6373]=29236;   // yue
   sortTable[0x66F1]=29237;   // yue
   sortTable[0x721A]=29238;   // yue
   sortTable[0x73A5]=29239;   // yue
   sortTable[0x7BD7]=29240;   // yue
   sortTable[0x868E]=29241;   // yue
   sortTable[0x9205]=29242;   // yue
   sortTable[0x925E]=29243;   // yue
   sortTable[0x95B2]=29244;   // yue
   sortTable[0x9E19]=29245;   // yue
   sortTable[0x8FD0]=29246;   // yun
   sortTable[0x904B]=29247;   // yun
   sortTable[0x4E91]=29248;   // yun
   sortTable[0x5141]=29249;   // yun
   sortTable[0x5300]=29250;   // yun
   sortTable[0x52FB]=29251;   // yun
   sortTable[0x6655]=29252;   // yun
   sortTable[0x6688]=29253;   // yun
   sortTable[0x8574]=29254;   // yun
   sortTable[0x860A]=29255;   // yun
   sortTable[0x97F5]=29256;   // yun
   sortTable[0x5B55]=29257;   // yun
   sortTable[0x9668]=29258;   // yun
   sortTable[0x82B8]=29259;   // yun
   sortTable[0x915D]=29260;   // yun
   sortTable[0x7B60]=29261;   // yun
   sortTable[0x90E7]=29262;   // yun
   sortTable[0x90D3]=29263;   // yun
   sortTable[0x7EAD]=29264;   // yun
   sortTable[0x6C32]=29265;   // yun
   sortTable[0x8018]=29266;   // yun
   sortTable[0x6120]=29267;   // yun
   sortTable[0x607D]=29268;   // yun
   sortTable[0x6B92]=29269;   // yun
   sortTable[0x92C6]=29270;   // yun
   sortTable[0x97EB]=29271;   // yun
   sortTable[0x71A8]=29272;   // yun
   sortTable[0x8D5F]=29273;   // yun
   sortTable[0x6600]=29274;   // yun
   sortTable[0x9835]=29275;   // yun
   sortTable[0x9196]=29276;   // yun
   sortTable[0x72C1]=29277;   // yun
   sortTable[0x7F0A]=29278;   // yun
   sortTable[0x97FB]=29279;   // yun
   sortTable[0x9723]=29280;   // yun
   sortTable[0x992B]=29281;   // yun
   sortTable[0x85F4]=29282;   // yun
   sortTable[0x7B7C]=29283;   // yun
   sortTable[0x8D07]=29284;   // yun
   sortTable[0x96F2]=29285;   // yun
   sortTable[0x8580]=29286;   // yun
   sortTable[0x7E15]=29287;   // yun
   sortTable[0x84B7]=29288;   // yun
   sortTable[0x837A]=29289;   // yun
   sortTable[0x7DF7]=29290;   // yun
   sortTable[0x816A]=29291;   // yun
   sortTable[0x8779]=29292;   // yun
   sortTable[0x9217]=29293;   // yun
   sortTable[0x99A7]=29294;   // yun
   sortTable[0x7DFC]=29295;   // yun
   sortTable[0x97D7]=29296;   // yun
   sortTable[0x7703]=29297;   // yun
   sortTable[0x891E]=29298;   // yun
   sortTable[0x99BB]=29299;   // yun
   sortTable[0x7E1C]=29300;   // yun
   sortTable[0x8CF1]=29301;   // yun
   sortTable[0x9F73]=29302;   // yun
   sortTable[0x7BD4]=29303;   // yun
   sortTable[0x7E67]=29304;   // yun
   sortTable[0x8480]=29305;   // yun
   sortTable[0x8495]=29306;   // yun
   sortTable[0x919E]=29307;   // yun
   sortTable[0x962D]=29308;   // yun
   sortTable[0x508A]=29309;   // yun
   sortTable[0x5597]=29310;   // yun
   sortTable[0x56E9]=29311;   // yun
   sortTable[0x593D]=29312;   // yun
   sortTable[0x596B]=29313;   // yun
   sortTable[0x5998]=29314;   // yun
   sortTable[0x60F2]=29315;   // yun
   sortTable[0x612A]=29316;   // yun
   sortTable[0x614D]=29317;   // yun
   sortTable[0x628E]=29318;   // yun
   sortTable[0x62A3]=29319;   // yun
   sortTable[0x679F]=29320;   // yun
   sortTable[0x69B2]=29321;   // yun
   sortTable[0x6A52]=29322;   // yun
   sortTable[0x6B9E]=29323;   // yun
   sortTable[0x6C33]=29324;   // yun
   sortTable[0x6C84]=29325;   // yun
   sortTable[0x6DA2]=29326;   // yun
   sortTable[0x6EB3]=29327;   // yun
   sortTable[0x6F90]=29328;   // yun
   sortTable[0x7174]=29329;   // yun
   sortTable[0x7185]=29330;   // yun
   sortTable[0x7189]=29331;   // yun
   sortTable[0x7547]=29332;   // yun
   sortTable[0x78D2]=29333;   // yun
   sortTable[0x79D0]=29334;   // yun
   sortTable[0x7D1C]=29335;   // yun
   sortTable[0x803A]=29336;   // yun
   sortTable[0x8553]=29337;   // yun
   sortTable[0x9106]=29338;   // yun
   sortTable[0x9116]=29339;   // yun
   sortTable[0x9695]=29340;   // yun
   sortTable[0x97DE]=29341;   // yun
   sortTable[0x9942]=29342;   // yun
   sortTable[0x6742]=29696;   // za
   sortTable[0x96DC]=29697;   // za
   sortTable[0x7838]=29698;   // za
   sortTable[0x5482]=29699;   // za
   sortTable[0x548B]=29700;   // za
   sortTable[0x531D]=29701;   // za
   sortTable[0x62F6]=29702;   // za
   sortTable[0x81DC]=29703;   // za
   sortTable[0x7D25]=29704;   // za
   sortTable[0x81E2]=29705;   // za
   sortTable[0x7D2E]=29706;   // za
   sortTable[0x894D]=29707;   // za
   sortTable[0x9B73]=29708;   // za
   sortTable[0x97F4]=29709;   // za
   sortTable[0x78FC]=29710;   // za
   sortTable[0x8FCA]=29711;   // za
   sortTable[0x9254]=29712;   // za
   sortTable[0x96D1]=29713;   // za
   sortTable[0x96E5]=29714;   // za
   sortTable[0x507A]=29715;   // za
   sortTable[0x5592]=29716;   // za
   sortTable[0x56CB]=29717;   // za
   sortTable[0x56D0]=29718;   // za
   sortTable[0x5DBB]=29719;   // za
   sortTable[0x5E00]=29720;   // za
   sortTable[0x6C9E]=29721;   // za
   sortTable[0x5728]=29722;   // zai
   sortTable[0x518D]=29723;   // zai
   sortTable[0x707E]=29724;   // zai
   sortTable[0x707D]=29725;   // zai
   sortTable[0x8F7D]=29726;   // zai
   sortTable[0x8F09]=29727;   // zai
   sortTable[0x683D]=29728;   // zai
   sortTable[0x5BB0]=29729;   // zai
   sortTable[0x5D3D]=29730;   // zai
   sortTable[0x54C9]=29731;   // zai
   sortTable[0x83D1]=29732;   // zai
   sortTable[0x753E]=29733;   // zai
   sortTable[0x7E21]=29734;   // zai
   sortTable[0x8CF3]=29735;   // zai
   sortTable[0x9168]=29736;   // zai
   sortTable[0x4FA2]=29737;   // zai
   sortTable[0x50A4]=29738;   // zai
   sortTable[0x510E]=29739;   // zai
   sortTable[0x6257]=29740;   // zai
   sortTable[0x6D05]=29741;   // zai
   sortTable[0x6E3D]=29742;   // zai
   sortTable[0x70D6]=29743;   // zai
   sortTable[0x7775]=29744;   // zai
   sortTable[0x54B1]=29745;   // zan
   sortTable[0x8D5E]=29746;   // zan
   sortTable[0x8D0A]=29747;   // zan
   sortTable[0x6682]=29748;   // zan
   sortTable[0x66AB]=29749;   // zan
   sortTable[0x6512]=29750;   // zan
   sortTable[0x6522]=29751;   // zan
   sortTable[0x7C2A]=29752;   // zan
   sortTable[0x74D2]=29753;   // zan
   sortTable[0x933E]=29754;   // zan
   sortTable[0x661D]=29755;   // zan
   sortTable[0x8DB1]=29756;   // zan
   sortTable[0x7CCC]=29757;   // zan
   sortTable[0x9147]=29758;   // zan
   sortTable[0x8E54]=29759;   // zan
   sortTable[0x9415]=29760;   // zan
   sortTable[0x9961]=29761;   // zan
   sortTable[0x8B9A]=29762;   // zan
   sortTable[0x7C2E]=29763;   // zan
   sortTable[0x8DB2]=29764;   // zan
   sortTable[0x79B6]=29765;   // zan
   sortTable[0x8978]=29766;   // zan
   sortTable[0x941F]=29767;   // zan
   sortTable[0x5127]=29768;   // zan
   sortTable[0x5139]=29769;   // zan
   sortTable[0x5142]=29770;   // zan
   sortTable[0x5646]=29771;   // zan
   sortTable[0x5BC1]=29772;   // zan
   sortTable[0x63DD]=29773;   // zan
   sortTable[0x648D]=29774;   // zan
   sortTable[0x6505]=29775;   // zan
   sortTable[0x685A]=29776;   // zan
   sortTable[0x6CAF]=29777;   // zan
   sortTable[0x6FFD]=29778;   // zan
   sortTable[0x7052]=29779;   // zan
   sortTable[0x74C9]=29780;   // zan
   sortTable[0x74DA]=29781;   // zan
   sortTable[0x7938]=29782;   // zan
   sortTable[0x8B83]=29783;   // zan
   sortTable[0x8CDB]=29784;   // zan
   sortTable[0x913C]=29785;   // zan
   sortTable[0x93E8]=29786;   // zan
   sortTable[0x810F]=29787;   // zang
   sortTable[0x81DF]=29788;   // zang
   sortTable[0x846C]=29789;   // zang
   sortTable[0x8D43]=29790;   // zang
   sortTable[0x81E7]=29791;   // zang
   sortTable[0x5958]=29792;   // zang
   sortTable[0x9A75]=29793;   // zang
   sortTable[0x9AD2]=29794;   // zang
   sortTable[0x81D3]=29795;   // zang
   sortTable[0x8D13]=29796;   // zang
   sortTable[0x99D4]=29797;   // zang
   sortTable[0x8535]=29798;   // zang
   sortTable[0x8D1C]=29799;   // zang
   sortTable[0x5328]=29800;   // zang
   sortTable[0x585F]=29801;   // zang
   sortTable[0x5F09]=29802;   // zang
   sortTable[0x7242]=29803;   // zang
   sortTable[0x7F98]=29804;   // zang
   sortTable[0x8CCD]=29805;   // zang
   sortTable[0x8CD8]=29806;   // zang
   sortTable[0x92BA]=29807;   // zang
   sortTable[0x9020]=29808;   // zao
   sortTable[0x65E9]=29809;   // zao
   sortTable[0x906D]=29810;   // zao
   sortTable[0x7CDF]=29811;   // zao
   sortTable[0x7682]=29812;   // zao
   sortTable[0x51FF]=29813;   // zao
   sortTable[0x947F]=29814;   // zao
   sortTable[0x8E81]=29815;   // zao
   sortTable[0x71E5]=29816;   // zao
   sortTable[0x7076]=29817;   // zao
   sortTable[0x6FA1]=29818;   // zao
   sortTable[0x67A3]=29819;   // zao
   sortTable[0x68D7]=29820;   // zao
   sortTable[0x566A]=29821;   // zao
   sortTable[0x85FB]=29822;   // zao
   sortTable[0x86A4]=29823;   // zao
   sortTable[0x5523]=29824;   // zao
   sortTable[0x7C09]=29825;   // zao
   sortTable[0x8241]=29826;   // zao
   sortTable[0x8B5F]=29827;   // zao
   sortTable[0x8DAE]=29828;   // zao
   sortTable[0x8E67]=29829;   // zao
   sortTable[0x7E70]=29830;   // zao
   sortTable[0x7AC3]=29831;   // zao
   sortTable[0x7AC8]=29832;   // zao
   sortTable[0x50AE]=29833;   // zao
   sortTable[0x5515]=29834;   // zao
   sortTable[0x55BF]=29835;   // zao
   sortTable[0x6165]=29836;   // zao
   sortTable[0x6806]=29837;   // zao
   sortTable[0x688D]=29838;   // zao
   sortTable[0x74AA]=29839;   // zao
   sortTable[0x7681]=29840;   // zao
   sortTable[0x85BB]=29841;   // zao
   sortTable[0x91A9]=29842;   // zao
   sortTable[0x5219]=29843;   // ze
   sortTable[0x5247]=29844;   // ze
   sortTable[0x8D23]=29845;   // ze
   sortTable[0x8CAC]=29846;   // ze
   sortTable[0x62E9]=29847;   // ze
   sortTable[0x64C7]=29848;   // ze
   sortTable[0x6CFD]=29849;   // ze
   sortTable[0x8D5C]=29850;   // ze
   sortTable[0x4EC4]=29851;   // ze
   sortTable[0x5567]=29852;   // ze
   sortTable[0x7B2E]=29853;   // ze
   sortTable[0x5E3B]=29854;   // ze
   sortTable[0x6603]=29855;   // ze
   sortTable[0x8234]=29856;   // ze
   sortTable[0x7BA6]=29857;   // ze
   sortTable[0x8FEE]=29858;   // ze
   sortTable[0x835D]=29859;   // ze
   sortTable[0x9F70]=29860;   // ze
   sortTable[0x9F5A]=29861;   // ze
   sortTable[0x790B]=29862;   // ze
   sortTable[0x880C]=29863;   // ze
   sortTable[0x8957]=29864;   // ze
   sortTable[0x9E05]=29865;   // ze
   sortTable[0x77E0]=29866;   // ze
   sortTable[0x8B2E]=29867;   // ze
   sortTable[0x7794]=29868;   // ze
   sortTable[0x4F2C]=29869;   // ze
   sortTable[0x5536]=29870;   // ze
   sortTable[0x5616]=29871;   // ze
   sortTable[0x5928]=29872;   // ze
   sortTable[0x5AE7]=29873;   // ze
   sortTable[0x5D31]=29874;   // ze
   sortTable[0x5E58]=29875;   // ze
   sortTable[0x5E82]=29876;   // ze
   sortTable[0x629E]=29877;   // ze
   sortTable[0x6351]=29878;   // ze
   sortTable[0x6617]=29879;   // ze
   sortTable[0x6A0D]=29880;   // ze
   sortTable[0x6C44]=29881;   // ze
   sortTable[0x6CA2]=29882;   // ze
   sortTable[0x6CCE]=29883;   // ze
   sortTable[0x6EAD]=29884;   // ze
   sortTable[0x6FA4]=29885;   // ze
   sortTable[0x769F]=29886;   // ze
   sortTable[0x7C00]=29887;   // ze
   sortTable[0x8536]=29888;   // ze
   sortTable[0x8ACE]=29889;   // ze
   sortTable[0x8CFE]=29890;   // ze
   sortTable[0x8D3C]=29891;   // zei
   sortTable[0x8CCA]=29892;   // zei
   sortTable[0x8808]=29893;   // zei
   sortTable[0x9C97]=29894;   // zei
   sortTable[0x9BFD]=29895;   // zei
   sortTable[0x621D]=29896;   // zei
   sortTable[0x9C02]=29897;   // zei
   sortTable[0x9C61]=29898;   // zei
   sortTable[0x600E]=29899;   // zen
   sortTable[0x8C2E]=29900;   // zen
   sortTable[0x8B5B]=29901;   // zen
   sortTable[0x56CE]=29902;   // zen
   sortTable[0x8B56]=29903;   // zen
   sortTable[0x589E]=29904;   // zeng
   sortTable[0x8D60]=29905;   // zeng
   sortTable[0x618E]=29906;   // zeng
   sortTable[0x7F2F]=29907;   // zeng
   sortTable[0x7511]=29908;   // zeng
   sortTable[0x7F7E]=29909;   // zeng
   sortTable[0x9503]=29910;   // zeng
   sortTable[0x912B]=29911;   // zeng
   sortTable[0x77F0]=29912;   // zeng
   sortTable[0x8D08]=29913;   // zeng
   sortTable[0x8B44]=29914;   // zeng
   sortTable[0x7E52]=29915;   // zeng
   sortTable[0x78F3]=29916;   // zeng
   sortTable[0x5897]=29917;   // zeng
   sortTable[0x6A67]=29918;   // zeng
   sortTable[0x71B7]=29919;   // zeng
   sortTable[0x7494]=29920;   // zeng
   sortTable[0x92E5]=29921;   // zeng
   sortTable[0x9C5B]=29922;   // zeng
   sortTable[0x70B8]=29923;   // zha
   sortTable[0x624E]=29924;   // zha
   sortTable[0x7728]=29925;   // zha
   sortTable[0x6E23]=29926;   // zha
   sortTable[0x95F8]=29927;   // zha
   sortTable[0x9598]=29928;   // zha
   sortTable[0x8BC8]=29929;   // zha
   sortTable[0x672D]=29930;   // zha
   sortTable[0x6805]=29931;   // zha
   sortTable[0x4E4D]=29932;   // zha
   sortTable[0x5412]=29933;   // zha
   sortTable[0x94E1]=29934;   // zha
   sortTable[0x55B3]=29935;   // zha
   sortTable[0x86B1]=29936;   // zha
   sortTable[0x69A8]=29937;   // zha
   sortTable[0x54A4]=29938;   // zha
   sortTable[0x67DE]=29939;   // zha
   sortTable[0x6942]=29940;   // zha
   sortTable[0x781F]=29941;   // zha
   sortTable[0x9705]=29942;   // zha
   sortTable[0x9C8A]=29943;   // zha
   sortTable[0x63F8]=29944;   // zha
   sortTable[0x9C9D]=29945;   // zha
   sortTable[0x91A1]=29946;   // zha
   sortTable[0x54F3]=29947;   // zha
   sortTable[0x75C4]=29948;   // zha
   sortTable[0x9F44]=29949;   // zha
   sortTable[0x82F2]=29950;   // zha
   sortTable[0x9358]=29951;   // zha
   sortTable[0x7B9A]=29952;   // zha
   sortTable[0x8B57]=29953;   // zha
   sortTable[0x8E37]=29954;   // zha
   sortTable[0x9B93]=29955;   // zha
   sortTable[0x9F47]=29956;   // zha
   sortTable[0x86BB]=29957;   // zha
   sortTable[0x89F0]=29958;   // zha
   sortTable[0x8A50]=29959;   // zha
   sortTable[0x5067]=29960;   // zha
   sortTable[0x5284]=29961;   // zha
   sortTable[0x538F]=29962;   // zha
   sortTable[0x5953]=29963;   // zha
   sortTable[0x5BB1]=29964;   // zha
   sortTable[0x62AF]=29965;   // zha
   sortTable[0x62C3]=29966;   // zha
   sortTable[0x6313]=29967;   // zha
   sortTable[0x6429]=29968;   // zha
   sortTable[0x643E]=29969;   // zha
   sortTable[0x6463]=29970;   // zha
   sortTable[0x67E4]=29971;   // zha
   sortTable[0x67FB]=29972;   // zha
   sortTable[0x6A1D]=29973;   // zha
   sortTable[0x6EA0]=29974;   // zha
   sortTable[0x7079]=29975;   // zha
   sortTable[0x7160]=29976;   // zha
   sortTable[0x7250]=29977;   // zha
   sortTable[0x7534]=29978;   // zha
   sortTable[0x76B6]=29979;   // zha
   sortTable[0x76BB]=29980;   // zha
   sortTable[0x802B]=29981;   // zha
   sortTable[0x8B47]=29982;   // zha
   sortTable[0x9BBA]=29983;   // zha
   sortTable[0x6458]=29984;   // zhai
   sortTable[0x5B85]=29985;   // zhai
   sortTable[0x7A84]=29986;   // zhai
   sortTable[0x503A]=29987;   // zhai
   sortTable[0x50B5]=29988;   // zhai
   sortTable[0x5BE8]=29989;   // zhai
   sortTable[0x658B]=29990;   // zhai
   sortTable[0x7FDF]=29991;   // zhai
   sortTable[0x7826]=29992;   // zhai
   sortTable[0x7635]=29993;   // zhai
   sortTable[0x9F4B]=29994;   // zhai
   sortTable[0x5908]=29995;   // zhai
   sortTable[0x635A]=29996;   // zhai
   sortTable[0x658E]=29997;   // zhai
   sortTable[0x69B8]=29998;   // zhai
   sortTable[0x6AA1]=29999;   // zhai
   sortTable[0x7C82]=30000;   // zhai
   sortTable[0x9259]=30001;   // zhai
   sortTable[0x6218]=30002;   // zhan
   sortTable[0x6230]=30003;   // zhan
   sortTable[0x5C55]=30004;   // zhan
   sortTable[0x7AD9]=30005;   // zhan
   sortTable[0x5360]=30006;   // zhan
   sortTable[0x76CF]=30007;   // zhan
   sortTable[0x76DE]=30008;   // zhan
   sortTable[0x6CBE]=30009;   // zhan
   sortTable[0x5D2D]=30010;   // zhan
   sortTable[0x5D84]=30011;   // zhan
   sortTable[0x7C98]=30012;   // zhan
   sortTable[0x6BE1]=30013;   // zhan
   sortTable[0x6C08]=30014;   // zhan
   sortTable[0x77BB]=30015;   // zhan
   sortTable[0x8638]=30016;   // zhan
   sortTable[0x65A9]=30017;   // zhan
   sortTable[0x8A79]=30018;   // zhan
   sortTable[0x6E5B]=30019;   // zhan
   sortTable[0x6808]=30020;   // zhan
   sortTable[0x7EFD]=30021;   // zhan
   sortTable[0x65C3]=30022;   // zhan
   sortTable[0x98D0]=30023;   // zhan
   sortTable[0x9711]=30024;   // zhan
   sortTable[0x8C35]=30025;   // zhan
   sortTable[0x9E6F]=30026;   // zhan
   sortTable[0x9958]=30027;   // zhan
   sortTable[0x9085]=30028;   // zhan
   sortTable[0x9CE3]=30029;   // zhan
   sortTable[0x9C63]=30030;   // zhan
   sortTable[0x9186]=30031;   // zhan
   sortTable[0x640C]=30032;   // zhan
   sortTable[0x859D]=30033;   // zhan
   sortTable[0x8B9D]=30034;   // zhan
   sortTable[0x98E6]=30035;   // zhan
   sortTable[0x8F3E]=30036;   // zhan
   sortTable[0x9A4F]=30037;   // zhan
   sortTable[0x9E07]=30038;   // zhan
   sortTable[0x9EF5]=30039;   // zhan
   sortTable[0x83DA]=30040;   // zhan
   sortTable[0x8A40]=30041;   // zhan
   sortTable[0x98AD]=30042;   // zhan
   sortTable[0x8666]=30043;   // zhan
   sortTable[0x9A59]=30044;   // zhan
   sortTable[0x8665]=30045;   // zhan
   sortTable[0x8B6B]=30046;   // zhan
   sortTable[0x8F1A]=30047;   // zhan
   sortTable[0x86C5]=30048;   // zhan
   sortTable[0x8F4F]=30049;   // zhan
   sortTable[0x4F54]=30050;   // zhan
   sortTable[0x5061]=30051;   // zhan
   sortTable[0x5661]=30052;   // zhan
   sortTable[0x5AF8]=30053;   // zhan
   sortTable[0x5D83]=30054;   // zhan
   sortTable[0x5D98]=30055;   // zhan
   sortTable[0x5DA6]=30056;   // zhan
   sortTable[0x60C9]=30057;   // zhan
   sortTable[0x6226]=30058;   // zhan
   sortTable[0x65AC]=30059;   // zhan
   sortTable[0x65DC]=30060;   // zhan
   sortTable[0x6834]=30061;   // zhan
   sortTable[0x685F]=30062;   // zhan
   sortTable[0x68E7]=30063;   // zhan
   sortTable[0x692B]=30064;   // zhan
   sortTable[0x6990]=30065;   // zhan
   sortTable[0x6A4F]=30066;   // zhan
   sortTable[0x6C0A]=30067;   // zhan
   sortTable[0x7416]=30068;   // zhan
   sortTable[0x7DBB]=30069;   // zhan
   sortTable[0x89B1]=30070;   // zhan
   sortTable[0x8B67]=30071;   // zhan
   sortTable[0x8D88]=30072;   // zhan
   sortTable[0x959A]=30073;   // zhan
   sortTable[0x9B59]=30074;   // zhan
   sortTable[0x957F]=30075;   // zhang
   sortTable[0x9577]=30076;   // zhang
   sortTable[0x5F20]=30077;   // zhang
   sortTable[0x5F35]=30078;   // zhang
   sortTable[0x7AE0]=30079;   // zhang
   sortTable[0x638C]=30080;   // zhang
   sortTable[0x5E10]=30081;   // zhang
   sortTable[0x5E33]=30082;   // zhang
   sortTable[0x4ED7]=30083;   // zhang
   sortTable[0x4E08]=30084;   // zhang
   sortTable[0x969C]=30085;   // zhang
   sortTable[0x6DA8]=30086;   // zhang
   sortTable[0x6F32]=30087;   // zhang
   sortTable[0x80C0]=30088;   // zhang
   sortTable[0x8139]=30089;   // zhang
   sortTable[0x6A1F]=30090;   // zhang
   sortTable[0x6756]=30091;   // zhang
   sortTable[0x8D26]=30092;   // zhang
   sortTable[0x5F70]=30093;   // zhang
   sortTable[0x748B]=30094;   // zhang
   sortTable[0x6F33]=30095;   // zhang
   sortTable[0x7634]=30096;   // zhang
   sortTable[0x5D82]=30097;   // zhang
   sortTable[0x9123]=30098;   // zhang
   sortTable[0x7350]=30099;   // zhang
   sortTable[0x87D1]=30100;   // zhang
   sortTable[0x5ADC]=30101;   // zhang
   sortTable[0x9E9E]=30102;   // zhang
   sortTable[0x5E5B]=30103;   // zhang
   sortTable[0x4EC9]=30104;   // zhang
   sortTable[0x9926]=30105;   // zhang
   sortTable[0x7CBB]=30106;   // zhang
   sortTable[0x9423]=30107;   // zhang
   sortTable[0x7795]=30108;   // zhang
   sortTable[0x7903]=30109;   // zhang
   sortTable[0x9067]=30110;   // zhang
   sortTable[0x93F1]=30111;   // zhang
   sortTable[0x8501]=30112;   // zhang
   sortTable[0x9A3F]=30113;   // zhang
   sortTable[0x50BD]=30114;   // zhang
   sortTable[0x5887]=30115;   // zhang
   sortTable[0x5E65]=30116;   // zhang
   sortTable[0x615E]=30117;   // zhang
   sortTable[0x6259]=30118;   // zhang
   sortTable[0x66B2]=30119;   // zhang
   sortTable[0x6DB1]=30120;   // zhang
   sortTable[0x75EE]=30121;   // zhang
   sortTable[0x762C]=30122;   // zhang
   sortTable[0x7C80]=30123;   // zhang
   sortTable[0x8CEC]=30124;   // zhang
   sortTable[0x9C46]=30125;   // zhang
   sortTable[0x627E]=30126;   // zhao
   sortTable[0x7167]=30127;   // zhao
   sortTable[0x53EC]=30128;   // zhao
   sortTable[0x62DB]=30129;   // zhao
   sortTable[0x7F69]=30130;   // zhao
   sortTable[0x722A]=30131;   // zhao
   sortTable[0x8BCF]=30132;   // zhao
   sortTable[0x8D75]=30133;   // zhao
   sortTable[0x662D]=30134;   // zhao
   sortTable[0x5146]=30135;   // zhao
   sortTable[0x6CBC]=30136;   // zhao
   sortTable[0x8087]=30137;   // zhao
   sortTable[0x948A]=30138;   // zhao
   sortTable[0x68F9]=30139;   // zhao
   sortTable[0x7B0A]=30140;   // zhao
   sortTable[0x5541]=30141;   // zhao
   sortTable[0x9BA1]=30142;   // zhao
   sortTable[0x77BE]=30143;   // zhao
   sortTable[0x8D99]=30144;   // zhao
   sortTable[0x91D7]=30145;   // zhao
   sortTable[0x7ABC]=30146;   // zhao
   sortTable[0x9363]=30147;   // zhao
   sortTable[0x7F40]=30148;   // zhao
   sortTable[0x8081]=30149;   // zhao
   sortTable[0x8088]=30150;   // zhao
   sortTable[0x99CB]=30151;   // zhao
   sortTable[0x4F4B]=30152;   // zhao
   sortTable[0x5797]=30153;   // zhao
   sortTable[0x59B1]=30154;   // zhao
   sortTable[0x5DF6]=30155;   // zhao
   sortTable[0x65D0]=30156;   // zhao
   sortTable[0x66CC]=30157;   // zhao
   sortTable[0x679B]=30158;   // zhao
   sortTable[0x6AC2]=30159;   // zhao
   sortTable[0x70A4]=30160;   // zhao
   sortTable[0x71F3]=30161;   // zhao
   sortTable[0x722B]=30162;   // zhao
   sortTable[0x72E3]=30163;   // zhao
   sortTable[0x7475]=30164;   // zhao
   sortTable[0x76BD]=30165;   // zhao
   sortTable[0x76C4]=30166;   // zhao
   sortTable[0x7F84]=30167;   // zhao
   sortTable[0x8A54]=30168;   // zhao
   sortTable[0x924A]=30169;   // zhao
   sortTable[0x8FD9]=30170;   // zhe
   sortTable[0x9019]=30171;   // zhe
   sortTable[0x7740]=30172;   // zhe
   sortTable[0x8457]=30173;   // zhe
   sortTable[0x8005]=30174;   // zhe
   sortTable[0x54F2]=30175;   // zhe
   sortTable[0x6298]=30176;   // zhe
   sortTable[0x906E]=30177;   // zhe
   sortTable[0x8517]=30178;   // zhe
   sortTable[0x6D59]=30179;   // zhe
   sortTable[0x8F84]=30180;   // zhe
   sortTable[0x647A]=30181;   // zhe
   sortTable[0x8C2A]=30182;   // zhe
   sortTable[0x8F99]=30183;   // zhe
   sortTable[0x8936]=30184;   // zhe
   sortTable[0x9E67]=30185;   // zhe
   sortTable[0x86F0]=30186;   // zhe
   sortTable[0x8D6D]=30187;   // zhe
   sortTable[0x67D8]=30188;   // zhe
   sortTable[0x78D4]=30189;   // zhe
   sortTable[0x8707]=30190;   // zhe
   sortTable[0x9517]=30191;   // zhe
   sortTable[0x8A5F]=30192;   // zhe
   sortTable[0x7987]=30193;   // zhe
   sortTable[0x9BBF]=30194;   // zhe
   sortTable[0x8B3A]=30195;   // zhe
   sortTable[0x88A9]=30196;   // zhe
   sortTable[0x8F19]=30197;   // zhe
   sortTable[0x7C8D]=30198;   // zhe
   sortTable[0x77FA]=30199;   // zhe
   sortTable[0x99B2]=30200;   // zhe
   sortTable[0x8975]=30201;   // zhe
   sortTable[0x8B2B]=30202;   // zhe
   sortTable[0x87C5]=30203;   // zhe
   sortTable[0x8B81]=30204;   // zhe
   sortTable[0x8B8B]=30205;   // zhe
   sortTable[0x8F12]=30206;   // zhe
   sortTable[0x4E7D]=30207;   // zhe
   sortTable[0x5387]=30208;   // zhe
   sortTable[0x5560]=30209;   // zhe
   sortTable[0x556B]=30210;   // zhe
   sortTable[0x5586]=30211;   // zhe
   sortTable[0x55FB]=30212;   // zhe
   sortTable[0x569E]=30213;   // zhe
   sortTable[0x57D1]=30214;   // zhe
   sortTable[0x5AEC]=30215;   // zhe
   sortTable[0x608A]=30216;   // zhe
   sortTable[0x6662]=30217;   // zhe
   sortTable[0x6663]=30218;   // zhe
   sortTable[0x6A1C]=30219;   // zhe
   sortTable[0x6B7D]=30220;   // zhe
   sortTable[0x6DDB]=30221;   // zhe
   sortTable[0x6F6A]=30222;   // zhe
   sortTable[0x7813]=30223;   // zhe
   sortTable[0x7C77]=30224;   // zhe
   sortTable[0x8674]=30225;   // zhe
   sortTable[0x87C4]=30226;   // zhe
   sortTable[0x8F4D]=30227;   // zhe
   sortTable[0x92B8]=30228;   // zhe
   sortTable[0x9DD3]=30229;   // zhe
   sortTable[0x771F]=30230;   // zhen
   sortTable[0x9635]=30231;   // zhen
   sortTable[0x9663]=30232;   // zhen
   sortTable[0x9488]=30233;   // zhen
   sortTable[0x91DD]=30234;   // zhen
   sortTable[0x9707]=30235;   // zhen
   sortTable[0x9547]=30236;   // zhen
   sortTable[0x93AE]=30237;   // zhen
   sortTable[0x73CD]=30238;   // zhen
   sortTable[0x632F]=30239;   // zhen
   sortTable[0x4FA6]=30240;   // zhen
   sortTable[0x5075]=30241;   // zhen
   sortTable[0x6795]=30242;   // zhen
   sortTable[0x8BCA]=30243;   // zhen
   sortTable[0x8A3A]=30244;   // zhen
   sortTable[0x8D1E]=30245;   // zhen
   sortTable[0x6715]=30246;   // zhen
   sortTable[0x5733]=30247;   // zhen
   sortTable[0x8D48]=30248;   // zhen
   sortTable[0x659F]=30249;   // zhen
   sortTable[0x796F]=30250;   // zhen
   sortTable[0x7504]=30251;   // zhen
   sortTable[0x7A39]=30252;   // zhen
   sortTable[0x81FB]=30253;   // zhen
   sortTable[0x6862]=30254;   // zhen
   sortTable[0x75B9]=30255;   // zhen
   sortTable[0x8F78]=30256;   // zhen
   sortTable[0x7BB4]=30257;   // zhen
   sortTable[0x9E29]=30258;   // zhen
   sortTable[0x699B]=30259;   // zhen
   sortTable[0x7F1C]=30260;   // zhen
   sortTable[0x755B]=30261;   // zhen
   sortTable[0x80D7]=30262;   // zhen
   sortTable[0x7827]=30263;   // zhen
   sortTable[0x84C1]=30264;   // zhen
   sortTable[0x799B]=30265;   // zhen
   sortTable[0x9156]=30266;   // zhen
   sortTable[0x6D48]=30267;   // zhen
   sortTable[0x8897]=30268;   // zhen
   sortTable[0x7EBC]=30269;   // zhen
   sortTable[0x8474]=30270;   // zhen
   sortTable[0x9241]=30271;   // zhen
   sortTable[0x7D7C]=30272;   // zhen
   sortTable[0x7D3E]=30273;   // zhen
   sortTable[0x9B12]=30274;   // zhen
   sortTable[0x937C]=30275;   // zhen
   sortTable[0x93AD]=30276;   // zhen
   sortTable[0x798E]=30277;   // zhen
   sortTable[0x8C9E]=30278;   // zhen
   sortTable[0x9C75]=30279;   // zhen
   sortTable[0x7739]=30280;   // zhen
   sortTable[0x7715]=30281;   // zhen
   sortTable[0x78AA]=30282;   // zhen
   sortTable[0x7C48]=30283;   // zhen
   sortTable[0x9EF0]=30284;   // zhen
   sortTable[0x7D16]=30285;   // zhen
   sortTable[0x8F43]=30286;   // zhen
   sortTable[0x771E]=30287;   // zhen
   sortTable[0x85BD]=30288;   // zhen
   sortTable[0x92F4]=30289;   // zhen
   sortTable[0x4FB2]=30290;   // zhen
   sortTable[0x5866]=30291;   // zhen
   sortTable[0x5AC3]=30292;   // zhen
   sortTable[0x5BCA]=30293;   // zhen
   sortTable[0x5C52]=30294;   // zhen
   sortTable[0x5E2A]=30295;   // zhen
   sortTable[0x5F2B]=30296;   // zhen
   sortTable[0x62AE]=30297;   // zhen
   sortTable[0x630B]=30298;   // zhen
   sortTable[0x63D5]=30299;   // zhen
   sortTable[0x6438]=30300;   // zhen
   sortTable[0x6576]=30301;   // zhen
   sortTable[0x6623]=30302;   // zhen
   sortTable[0x6815]=30303;   // zhen
   sortTable[0x681A]=30304;   // zhen
   sortTable[0x686D]=30305;   // zhen
   sortTable[0x6968]=30306;   // zhen
   sortTable[0x6A3C]=30307;   // zhen
   sortTable[0x6B9D]=30308;   // zhen
   sortTable[0x6F67]=30309;   // zhen
   sortTable[0x6FB5]=30310;   // zhen
   sortTable[0x7349]=30311;   // zhen
   sortTable[0x73CE]=30312;   // zhen
   sortTable[0x7467]=30313;   // zhen
   sortTable[0x7471]=30314;   // zhen
   sortTable[0x753D]=30315;   // zhen
   sortTable[0x7E25]=30316;   // zhen
   sortTable[0x8044]=30317;   // zhen
   sortTable[0x8419]=30318;   // zhen
   sortTable[0x8496]=30319;   // zhen
   sortTable[0x88D6]=30320;   // zhen
   sortTable[0x8AAB]=30321;   // zhen
   sortTable[0x8CD1]=30322;   // zhen
   sortTable[0x8EEB]=30323;   // zhen
   sortTable[0x9049]=30324;   // zhen
   sortTable[0x9159]=30325;   // zhen
   sortTable[0x9331]=30326;   // zhen
   sortTable[0x9755]=30327;   // zhen
   sortTable[0x99D7]=30328;   // zhen
   sortTable[0x9D06]=30329;   // zhen
   sortTable[0x6B63]=30330;   // zheng
   sortTable[0x653F]=30331;   // zheng
   sortTable[0x4E89]=30332;   // zheng
   sortTable[0x722D]=30333;   // zheng
   sortTable[0x6574]=30334;   // zheng
   sortTable[0x8BC1]=30335;   // zheng
   sortTable[0x8A3C]=30336;   // zheng
   sortTable[0x5F81]=30337;   // zheng
   sortTable[0x7741]=30338;   // zheng
   sortTable[0x775C]=30339;   // zheng
   sortTable[0x84B8]=30340;   // zheng
   sortTable[0x6323]=30341;   // zheng
   sortTable[0x6399]=30342;   // zheng
   sortTable[0x75C7]=30343;   // zheng
   sortTable[0x90D1]=30344;   // zheng
   sortTable[0x912D]=30345;   // zheng
   sortTable[0x6014]=30346;   // zheng
   sortTable[0x94EE]=30347;   // zheng
   sortTable[0x5FB5]=30348;   // zheng
   sortTable[0x62EF]=30349;   // zheng
   sortTable[0x5E27]=30350;   // zheng
   sortTable[0x8BE4]=30351;   // zheng
   sortTable[0x7B5D]=30352;   // zheng
   sortTable[0x72F0]=30353;   // zheng
   sortTable[0x5CE5]=30354;   // zheng
   sortTable[0x94B2]=30355;   // zheng
   sortTable[0x8B49]=30356;   // zheng
   sortTable[0x9BD6]=30357;   // zheng
   sortTable[0x7B8F]=30358;   // zheng
   sortTable[0x7BDC]=30359;   // zheng
   sortTable[0x9266]=30360;   // zheng
   sortTable[0x931A]=30361;   // zheng
   sortTable[0x9D0A]=30362;   // zheng
   sortTable[0x4F42]=30363;   // zheng
   sortTable[0x51E7]=30364;   // zheng
   sortTable[0x57E9]=30365;   // zheng
   sortTable[0x5863]=30366;   // zheng
   sortTable[0x59C3]=30367;   // zheng
   sortTable[0x5A9C]=30368;   // zheng
   sortTable[0x5D1D]=30369;   // zheng
   sortTable[0x5D22]=30370;   // zheng
   sortTable[0x5E40]=30371;   // zheng
   sortTable[0x5FB0]=30372;   // zheng
   sortTable[0x5FB4]=30373;   // zheng
   sortTable[0x6138]=30374;   // zheng
   sortTable[0x628D]=30375;   // zheng
   sortTable[0x639F]=30376;   // zheng
   sortTable[0x63C1]=30377;   // zheng
   sortTable[0x649C]=30378;   // zheng
   sortTable[0x6678]=30379;   // zheng
   sortTable[0x6C36]=30380;   // zheng
   sortTable[0x70A1]=30381;   // zheng
   sortTable[0x70DD]=30382;   // zheng
   sortTable[0x7319]=30383;   // zheng
   sortTable[0x7665]=30384;   // zheng
   sortTable[0x7710]=30385;   // zheng
   sortTable[0x7CFD]=30386;   // zheng
   sortTable[0x8047]=30387;   // zheng
   sortTable[0x8ACD]=30388;   // zheng
   sortTable[0x8E2D]=30389;   // zheng
   sortTable[0x9B07]=30390;   // zheng
   sortTable[0x53EA]=30391;   // zhi
   sortTable[0x77E5]=30392;   // zhi
   sortTable[0x4E4B]=30393;   // zhi
   sortTable[0x5FD7]=30394;   // zhi
   sortTable[0x5236]=30395;   // zhi
   sortTable[0x76F4]=30396;   // zhi
   sortTable[0x6307]=30397;   // zhi
   sortTable[0x6CBB]=30398;   // zhi
   sortTable[0x8D28]=30399;   // zhi
   sortTable[0x8CEA]=30400;   // zhi
   sortTable[0x81F3]=30401;   // zhi
   sortTable[0x7EC7]=30402;   // zhi
   sortTable[0x7E54]=30403;   // zhi
   sortTable[0x652F]=30404;   // zhi
   sortTable[0x7EB8]=30405;   // zhi
   sortTable[0x7D19]=30406;   // zhi
   sortTable[0x6B62]=30407;   // zhi
   sortTable[0x503C]=30408;   // zhi
   sortTable[0x81F4]=30409;   // zhi
   sortTable[0x679D]=30410;   // zhi
   sortTable[0x6267]=30411;   // zhi
   sortTable[0x690D]=30412;   // zhi
   sortTable[0x57F7]=30413;   // zhi
   sortTable[0x804C]=30414;   // zhi
   sortTable[0x8077]=30415;   // zhi
   sortTable[0x7F6E]=30416;   // zhi
   sortTable[0x6B96]=30417;   // zhi
   sortTable[0x667A]=30418;   // zhi
   sortTable[0x5E1C]=30419;   // zhi
   sortTable[0x5E5F]=30420;   // zhi
   sortTable[0x79E9]=30421;   // zhi
   sortTable[0x8102]=30422;   // zhi
   sortTable[0x80A2]=30423;   // zhi
   sortTable[0x8718]=30424;   // zhi
   sortTable[0x5740]=30425;   // zhi
   sortTable[0x7A1A]=30426;   // zhi
   sortTable[0x63B7]=30427;   // zhi
   sortTable[0x64F2]=30428;   // zhi
   sortTable[0x829D]=30429;   // zhi
   sortTable[0x6C41]=30430;   // zhi
   sortTable[0x65E8]=30431;   // zhi
   sortTable[0x4F84]=30432;   // zhi
   sortTable[0x6EDE]=30433;   // zhi
   sortTable[0x7957]=30434;   // zhi
   sortTable[0x631A]=30435;   // zhi
   sortTable[0x89EF]=30436;   // zhi
   sortTable[0x8DBE]=30437;   // zhi
   sortTable[0x5431]=30438;   // zhi
   sortTable[0x5CD9]=30439;   // zhi
   sortTable[0x7A92]=30440;   // zhi
   sortTable[0x7099]=30441;   // zhi
   sortTable[0x965F]=30442;   // zhi
   sortTable[0x96C9]=30443;   // zhi
   sortTable[0x86ED]=30444;   // zhi
   sortTable[0x82B7]=30445;   // zhi
   sortTable[0x8D3D]=30446;   // zhi
   sortTable[0x7949]=30447;   // zhi
   sortTable[0x54AB]=30448;   // zhi
   sortTable[0x536E]=30449;   // zhi
   sortTable[0x75D4]=30450;   // zhi
   sortTable[0x6809]=30451;   // zhi
   sortTable[0x80DD]=30452;   // zhi
   sortTable[0x8879]=30453;   // zhi
   sortTable[0x90C5]=30454;   // zhi
   sortTable[0x9E37]=30455;   // zhi
   sortTable[0x646D]=30456;   // zhi
   sortTable[0x9A98]=30457;   // zhi
   sortTable[0x75E3]=30458;   // zhi
   sortTable[0x5F58]=30459;   // zhi
   sortTable[0x684E]=30460;   // zhi
   sortTable[0x67B3]=30461;   // zhi
   sortTable[0x7D77]=30462;   // zhi
   sortTable[0x8DD6]=30463;   // zhi
   sortTable[0x962F]=30464;   // zhi
   sortTable[0x916F]=30465;   // zhi
   sortTable[0x5E19]=30466;   // zhi
   sortTable[0x8E2F]=30467;   // zhi
   sortTable[0x8E2C]=30468;   // zhi
   sortTable[0x8F75]=30469;   // zhi
   sortTable[0x5FEE]=30470;   // zhi
   sortTable[0x6800]=30471;   // zhi
   sortTable[0x57F4]=30472;   // zhi
   sortTable[0x9EF9]=30473;   // zhi
   sortTable[0x7983]=30474;   // zhi
   sortTable[0x7994]=30475;   // zhi
   sortTable[0x9CF7]=30476;   // zhi
   sortTable[0x8F7E]=30477;   // zhi
   sortTable[0x7929]=30478;   // zhi
   sortTable[0x9527]=30479;   // zhi
   sortTable[0x94DA]=30480;   // zhi
   sortTable[0x8E60]=30481;   // zhi
   sortTable[0x8DF1]=30482;   // zhi
   sortTable[0x88FD]=30483;   // zhi
   sortTable[0x8A8C]=30484;   // zhi
   sortTable[0x79F2]=30485;   // zhi
   sortTable[0x88A0]=30486;   // zhi
   sortTable[0x9624]=30487;   // zhi
   sortTable[0x7A49]=30488;   // zhi
   sortTable[0x7A19]=30489;   // zhi
   sortTable[0x7D29]=30490;   // zhi
   sortTable[0x9D32]=30491;   // zhi
   sortTable[0x9A3A]=30492;   // zhi
   sortTable[0x9455]=30493;   // zhi
   sortTable[0x96BB]=30494;   // zhi
   sortTable[0x79EA]=30495;   // zhi
   sortTable[0x7DFB]=30496;   // zhi
   sortTable[0x9D44]=30497;   // zhi
   sortTable[0x889F]=30498;   // zhi
   sortTable[0x79D6]=30499;   // zhi
   sortTable[0x7FD0]=30500;   // zhi
   sortTable[0x80D1]=30501;   // zhi
   sortTable[0x81B1]=30502;   // zhi
   sortTable[0x9244]=30503;   // zhi
   sortTable[0x928D]=30504;   // zhi
   sortTable[0x7D95]=30505;   // zhi
   sortTable[0x8296]=30506;   // zhi
   sortTable[0x92D5]=30507;   // zhi
   sortTable[0x887C]=30508;   // zhi
   sortTable[0x85E2]=30509;   // zhi
   sortTable[0x9A47]=30510;   // zhi
   sortTable[0x7951]=30511;   // zhi
   sortTable[0x8635]=30512;   // zhi
   sortTable[0x899F]=30513;   // zhi
   sortTable[0x8D04]=30514;   // zhi
   sortTable[0x8FE3]=30515;   // zhi
   sortTable[0x99BD]=30516;   // zhi
   sortTable[0x9A2D]=30517;   // zhi
   sortTable[0x79F7]=30518;   // zhi
   sortTable[0x8B22]=30519;   // zhi
   sortTable[0x8CAD]=30520;   // zhi
   sortTable[0x8EC4]=30521;   // zhi
   sortTable[0x9D19]=30522;   // zhi
   sortTable[0x9F05]=30523;   // zhi
   sortTable[0x4E7F]=30524;   // zhi
   sortTable[0x4FE7]=30525;   // zhi
   sortTable[0x5001]=30526;   // zhi
   sortTable[0x5024]=30527;   // zhi
   sortTable[0x506B]=30528;   // zhi
   sortTable[0x5082]=30529;   // zhi
   sortTable[0x5128]=30530;   // zhi
   sortTable[0x51EA]=30531;   // zhi
   sortTable[0x5295]=30532;   // zhi
   sortTable[0x52A7]=30533;   // zhi
   sortTable[0x5394]=30534;   // zhi
   sortTable[0x55ED]=30535;   // zhi
   sortTable[0x5741]=30536;   // zhi
   sortTable[0x5767]=30537;   // zhi
   sortTable[0x5781]=30538;   // zhi
   sortTable[0x5886]=30539;   // zhi
   sortTable[0x588C]=30540;   // zhi
   sortTable[0x5902]=30541;   // zhi
   sortTable[0x59B7]=30542;   // zhi
   sortTable[0x59EA]=30543;   // zhi
   sortTable[0x5A21]=30544;   // zhi
   sortTable[0x5B02]=30545;   // zhi
   sortTable[0x5BD8]=30546;   // zhi
   sortTable[0x5D3B]=30547;   // zhi
   sortTable[0x5DF5]=30548;   // zhi
   sortTable[0x5E0B]=30549;   // zhi
   sortTable[0x5EA2]=30550;   // zhi
   sortTable[0x5EA4]=30551;   // zhi
   sortTable[0x5ECC]=30552;   // zhi
   sortTable[0x5F8F]=30553;   // zhi
   sortTable[0x5F94]=30554;   // zhi
   sortTable[0x5F9D]=30555;   // zhi
   sortTable[0x603E]=30556;   // zhi
   sortTable[0x6049]=30557;   // zhi
   sortTable[0x6179]=30558;   // zhi
   sortTable[0x6184]=30559;   // zhi
   sortTable[0x61E5]=30560;   // zhi
   sortTable[0x61EB]=30561;   // zhi
   sortTable[0x6220]=30562;   // zhi
   sortTable[0x627A]=30563;   // zhi
   sortTable[0x627B]=30564;   // zhi
   sortTable[0x62A7]=30565;   // zhi
   sortTable[0x6303]=30566;   // zhi
   sortTable[0x6418]=30567;   // zhi
   sortTable[0x6431]=30568;   // zhi
   sortTable[0x646F]=30569;   // zhi
   sortTable[0x64F3]=30570;   // zhi
   sortTable[0x65D8]=30571;   // zhi
   sortTable[0x664A]=30572;   // zhi
   sortTable[0x67E3]=30573;   // zhi
   sortTable[0x6894]=30574;   // zhi
   sortTable[0x68BD]=30575;   // zhi
   sortTable[0x6925]=30576;   // zhi
   sortTable[0x6956]=30577;   // zhi
   sortTable[0x69B0]=30578;   // zhi
   sortTable[0x6A34]=30579;   // zhi
   sortTable[0x6ACD]=30580;   // zhi
   sortTable[0x6ADB]=30581;   // zhi
   sortTable[0x6C65]=30582;   // zhi
   sortTable[0x6C66]=30583;   // zhi
   sortTable[0x6C9A]=30584;   // zhi
   sortTable[0x6CDC]=30585;   // zhi
   sortTable[0x6D14]=30586;   // zhi
   sortTable[0x6D37]=30587;   // zhi
   sortTable[0x6DD4]=30588;   // zhi
   sortTable[0x6DFD]=30589;   // zhi
   sortTable[0x6ECD]=30590;   // zhi
   sortTable[0x6EEF]=30591;   // zhi
   sortTable[0x6F10]=30592;   // zhi
   sortTable[0x6F4C]=30593;   // zhi
   sortTable[0x7004]=30594;   // zhi
   sortTable[0x71AB]=30595;   // zhi
   sortTable[0x7286]=30596;   // zhi
   sortTable[0x72FE]=30597;   // zhi
   sortTable[0x7318]=30598;   // zhi
   sortTable[0x74C6]=30599;   // zhi
   sortTable[0x74E1]=30600;   // zhi
   sortTable[0x7564]=30601;   // zhi
   sortTable[0x7590]=30602;   // zhi
   sortTable[0x75B7]=30603;   // zhi
   sortTable[0x75BB]=30604;   // zhi
   sortTable[0x780B]=30605;   // zhi
   sortTable[0x796C]=30606;   // zhi
   sortTable[0x79D3]=30607;   // zhi
   sortTable[0x7A3A]=30608;   // zhi
   sortTable[0x7B6B]=30609;   // zhi
   sortTable[0x7E36]=30610;   // zhi
   sortTable[0x8040]=30611;   // zhi
   sortTable[0x81A3]=30612;   // zhi
   sortTable[0x81F8]=30613;   // zhi
   sortTable[0x87B2]=30614;   // zhi
   sortTable[0x87D9]=30615;   // zhi
   sortTable[0x8901]=30616;   // zhi
   sortTable[0x8967]=30617;   // zhi
   sortTable[0x89D7]=30618;   // zhi
   sortTable[0x89F6]=30619;   // zhi
   sortTable[0x8A28]=30620;   // zhi
   sortTable[0x8C51]=30621;   // zhi
   sortTable[0x8C52]=30622;   // zhi
   sortTable[0x8C78]=30623;   // zhi
   sortTable[0x8E91]=30624;   // zhi
   sortTable[0x8E93]=30625;   // zhi
   sortTable[0x8EF9]=30626;   // zhi
   sortTable[0x8EFD]=30627;   // zhi
   sortTable[0x8F0A]=30628;   // zhi
   sortTable[0x91DE]=30629;   // zhi
   sortTable[0x99B6]=30630;   // zhi
   sortTable[0x99E4]=30631;   // zhi
   sortTable[0x9BEF]=30632;   // zhi
   sortTable[0x9DD9]=30633;   // zhi
   sortTable[0x4E2D]=30634;   // zhong
   sortTable[0x79CD]=30635;   // zhong
   sortTable[0x7A2E]=30636;   // zhong
   sortTable[0x91CD]=30637;   // zhong
   sortTable[0x4F17]=30638;   // zhong
   sortTable[0x773E]=30639;   // zhong
   sortTable[0x949F]=30640;   // zhong
   sortTable[0x9418]=30641;   // zhong
   sortTable[0x7EC8]=30642;   // zhong
   sortTable[0x7D42]=30643;   // zhong
   sortTable[0x5FE0]=30644;   // zhong
   sortTable[0x80BF]=30645;   // zhong
   sortTable[0x816B]=30646;   // zhong
   sortTable[0x4EF2]=30647;   // zhong
   sortTable[0x8877]=30648;   // zhong
   sortTable[0x51A2]=30649;   // zhong
   sortTable[0x8E35]=30650;   // zhong
   sortTable[0x953A]=30651;   // zhong
   sortTable[0x76C5]=30652;   // zhong
   sortTable[0x822F]=30653;   // zhong
   sortTable[0x87BD]=30654;   // zhong
   sortTable[0x8846]=30655;   // zhong
   sortTable[0x7A5C]=30656;   // zhong
   sortTable[0x8E71]=30657;   // zhong
   sortTable[0x8AE5]=30658;   // zhong
   sortTable[0x8876]=30659;   // zhong
   sortTable[0x92BF]=30660;   // zhong
   sortTable[0x9F28]=30661;   // zhong
   sortTable[0x833D]=30662;   // zhong
   sortTable[0x87A4]=30663;   // zhong
   sortTable[0x937E]=30664;   // zhong
   sortTable[0x8520]=30665;   // zhong
   sortTable[0x869B]=30666;   // zhong
   sortTable[0x8FDA]=30667;   // zhong
   sortTable[0x9221]=30668;   // zhong
   sortTable[0x4F00]=30669;   // zhong
   sortTable[0x5045]=30670;   // zhong
   sortTable[0x5223]=30671;   // zhong
   sortTable[0x55A0]=30672;   // zhong
   sortTable[0x5839]=30673;   // zhong
   sortTable[0x585A]=30674;   // zhong
   sortTable[0x585C]=30675;   // zhong
   sortTable[0x5990]=30676;   // zhong
   sortTable[0x5995]=30677;   // zhong
   sortTable[0x5A91]=30678;   // zhong
   sortTable[0x5C30]=30679;   // zhong
   sortTable[0x5E52]=30680;   // zhong
   sortTable[0x5F78]=30681;   // zhong
   sortTable[0x67CA]=30682;   // zhong
   sortTable[0x6B71]=30683;   // zhong
   sortTable[0x6C77]=30684;   // zhong
   sortTable[0x6CC8]=30685;   // zhong
   sortTable[0x7082]=30686;   // zhong
   sortTable[0x7144]=30687;   // zhong
   sortTable[0x72C6]=30688;   // zhong
   sortTable[0x7607]=30689;   // zhong
   sortTable[0x794C]=30690;   // zhong
   sortTable[0x7B57]=30691;   // zhong
   sortTable[0x7C66]=30692;   // zhong
   sortTable[0x8873]=30693;   // zhong
   sortTable[0x5468]=30694;   // zhou
   sortTable[0x6D32]=30695;   // zhou
   sortTable[0x5B99]=30696;   // zhou
   sortTable[0x76B1]=30697;   // zhou
   sortTable[0x76BA]=30698;   // zhou
   sortTable[0x7CA5]=30699;   // zhou
   sortTable[0x8F74]=30700;   // zhou
   sortTable[0x8EF8]=30701;   // zhou
   sortTable[0x5DDE]=30702;   // zhou
   sortTable[0x663C]=30703;   // zhou
   sortTable[0x665D]=30704;   // zhou
   sortTable[0x9AA4]=30705;   // zhou
   sortTable[0x9A5F]=30706;   // zhou
   sortTable[0x821F]=30707;   // zhou
   sortTable[0x5E1A]=30708;   // zhou
   sortTable[0x5492]=30709;   // zhou
   sortTable[0x80C4]=30710;   // zhou
   sortTable[0x7EA3]=30711;   // zhou
   sortTable[0x8098]=30712;   // zhou
   sortTable[0x7EC9]=30713;   // zhou
   sortTable[0x8BCC]=30714;   // zhou
   sortTable[0x59AF]=30715;   // zhou
   sortTable[0x76E9]=30716;   // zhou
   sortTable[0x7C40]=30717;   // zhou
   sortTable[0x914E]=30718;   // zhou
   sortTable[0x8F80]=30719;   // zhou
   sortTable[0x8D52]=30720;   // zhou
   sortTable[0x8BEA]=30721;   // zhou
   sortTable[0x9031]=30722;   // zhou
   sortTable[0x8A76]=30723;   // zhou
   sortTable[0x8F08]=30724;   // zhou
   sortTable[0x970C]=30725;   // zhou
   sortTable[0x9282]=30726;   // zhou
   sortTable[0x7B92]=30727;   // zhou
   sortTable[0x8F16]=30728;   // zhou
   sortTable[0x83F7]=30729;   // zhou
   sortTable[0x8B78]=30730;   // zhou
   sortTable[0x7C52]=30731;   // zhou
   sortTable[0x9D43]=30732;   // zhou
   sortTable[0x9E3C]=30733;   // zhou
   sortTable[0x7C99]=30734;   // zhou
   sortTable[0x7D02]=30735;   // zhou
   sortTable[0x836E]=30736;   // zhou
   sortTable[0x8464]=30737;   // zhou
   sortTable[0x8A4B]=30738;   // zhou
   sortTable[0x99CE]=30739;   // zhou
   sortTable[0x9A06]=30740;   // zhou
   sortTable[0x9BDE]=30741;   // zhou
   sortTable[0x4F37]=30742;   // zhou
   sortTable[0x4F9C]=30743;   // zhou
   sortTable[0x50FD]=30744;   // zhou
   sortTable[0x5191]=30745;   // zhou
   sortTable[0x546A]=30746;   // zhou
   sortTable[0x54AE]=30747;   // zhou
   sortTable[0x558C]=30748;   // zhou
   sortTable[0x5663]=30749;   // zhou
   sortTable[0x5F9F]=30750;   // zhou
   sortTable[0x63AB]=30751;   // zhou
   sortTable[0x666D]=30752;   // zhou
   sortTable[0x6DCD]=30753;   // zhou
   sortTable[0x70BF]=30754;   // zhou
   sortTable[0x70D0]=30755;   // zhou
   sortTable[0x73D8]=30756;   // zhou
   sortTable[0x7503]=30757;   // zhou
   sortTable[0x759B]=30758;   // zhou
   sortTable[0x776D]=30759;   // zhou
   sortTable[0x77EA]=30760;   // zhou
   sortTable[0x7C55]=30761;   // zhou
   sortTable[0x7E10]=30762;   // zhou
   sortTable[0x8B05]=30763;   // zhou
   sortTable[0x8CD9]=30764;   // zhou
   sortTable[0x90EE]=30765;   // zhou
   sortTable[0x99F2]=30766;   // zhou
   sortTable[0x4E3B]=30767;   // zhu
   sortTable[0x4F4F]=30768;   // zhu
   sortTable[0x52A9]=30769;   // zhu
   sortTable[0x6CE8]=30770;   // zhu
   sortTable[0x9010]=30771;   // zhu
   sortTable[0x7B51]=30772;   // zhu
   sortTable[0x73E0]=30773;   // zhu
   sortTable[0x7AF9]=30774;   // zhu
   sortTable[0x795D]=30775;   // zhu
   sortTable[0x732A]=30776;   // zhu
   sortTable[0x8C6C]=30777;   // zhu
   sortTable[0x67F1]=30778;   // zhu
   sortTable[0x682A]=30779;   // zhu
   sortTable[0x70DB]=30780;   // zhu
   sortTable[0x71ED]=30781;   // zhu
   sortTable[0x716E]=30782;   // zhu
   sortTable[0x94F8]=30783;   // zhu
   sortTable[0x9444]=30784;   // zhu
   sortTable[0x5631]=30785;   // zhu
   sortTable[0x56D1]=30786;   // zhu
   sortTable[0x86DB]=30787;   // zhu
   sortTable[0x9A7B]=30788;   // zhu
   sortTable[0x99D0]=30789;   // zhu
   sortTable[0x8BF8]=30790;   // zhu
   sortTable[0x8AF8]=30791;   // zhu
   sortTable[0x62C4]=30792;   // zhu
   sortTable[0x86C0]=30793;   // zhu
   sortTable[0x6731]=30794;   // zhu
   sortTable[0x8BDB]=30795;   // zhu
   sortTable[0x7843]=30796;   // zhu
   sortTable[0x8D2E]=30797;   // zhu
   sortTable[0x7AFA]=30798;   // zhu
   sortTable[0x6E1A]=30799;   // zhu
   sortTable[0x94E2]=30800;   // zhu
   sortTable[0x77A9]=30801;   // zhu
   sortTable[0x4F2B]=30802;   // zhu
   sortTable[0x7BB8]=30803;   // zhu
   sortTable[0x4F8F]=30804;   // zhu
   sortTable[0x90BE]=30805;   // zhu
   sortTable[0x677C]=30806;   // zhu
   sortTable[0x8233]=30807;   // zhu
   sortTable[0x70B7]=30808;   // zhu
   sortTable[0x6D19]=30809;   // zhu
   sortTable[0x9E88]=30810;   // zhu
   sortTable[0x69E0]=30811;   // zhu
   sortTable[0x82CE]=30812;   // zhu
   sortTable[0x7FE5]=30813;   // zhu
   sortTable[0x6F74]=30814;   // zhu
   sortTable[0x7EBB]=30815;   // zhu
   sortTable[0x6A65]=30816;   // zhu
   sortTable[0x8E85]=30817;   // zhu
   sortTable[0x8331]=30818;   // zhu
   sortTable[0x8A3B]=30819;   // zhu
   sortTable[0x7A8B]=30820;   // zhu
   sortTable[0x782B]=30821;   // zhu
   sortTable[0x880B]=30822;   // zhu
   sortTable[0x7603]=30823;   // zhu
   sortTable[0x7D35]=30824;   // zhu
   sortTable[0x75B0]=30825;   // zhu
   sortTable[0x967C]=30826;   // zhu
   sortTable[0x8A5D]=30827;   // zhu
   sortTable[0x9E86]=30828;   // zhu
   sortTable[0x7F9C]=30829;   // zhu
   sortTable[0x8DE6]=30830;   // zhu
   sortTable[0x8CAF]=30831;   // zhu
   sortTable[0x8DD3]=30832;   // zhu
   sortTable[0x9714]=30833;   // zhu
   sortTable[0x9D38]=30834;   // zhu
   sortTable[0x7969]=30835;   // zhu
   sortTable[0x88BE]=30836;   // zhu
   sortTable[0x7ADA]=30837;   // zhu
   sortTable[0x7BC9]=30838;   // zhu
   sortTable[0x7B6F]=30839;   // zhu
   sortTable[0x8829]=30840;   // zhu
   sortTable[0x9C41]=30841;   // zhu
   sortTable[0x77DA]=30842;   // zhu
   sortTable[0x883E]=30843;   // zhu
   sortTable[0x7B01]=30844;   // zhu
   sortTable[0x8FEC]=30845;   // zhu
   sortTable[0x9296]=30846;   // zhu
   sortTable[0x99EF]=30847;   // zhu
   sortTable[0x79FC]=30848;   // zhu
   sortTable[0x7B1C]=30849;   // zhu
   sortTable[0x7D38]=30850;   // zhu
   sortTable[0x833F]=30851;   // zhu
   sortTable[0x9F04]=30852;   // zhu
   sortTable[0x4E36]=30853;   // zhu
   sortTable[0x4F47]=30854;   // zhu
   sortTable[0x529A]=30855;   // zhu
   sortTable[0x52AF]=30856;   // zhu
   sortTable[0x577E]=30857;   // zhu
   sortTable[0x58B8]=30858;   // zhu
   sortTable[0x58F4]=30859;   // zhu
   sortTable[0x5B4E]=30860;   // zhu
   sortTable[0x5B94]=30861;   // zhu
   sortTable[0x5D40]=30862;   // zhu
   sortTable[0x65B8]=30863;   // zhu
   sortTable[0x66EF]=30864;   // zhu
   sortTable[0x6A26]=30865;   // zhu
   sortTable[0x6AE7]=30866;   // zhu
   sortTable[0x6AEB]=30867;   // zhu
   sortTable[0x6B18]=30868;   // zhu
   sortTable[0x6BB6]=30869;   // zhu
   sortTable[0x6CCF]=30870;   // zhu
   sortTable[0x6FD0]=30871;   // zhu
   sortTable[0x7026]=30872;   // zhu
   sortTable[0x705F]=30873;   // zhu
   sortTable[0x70A2]=30874;   // zhu
   sortTable[0x7151]=30875;   // zhu
   sortTable[0x7225]=30876;   // zhu
   sortTable[0x771D]=30877;   // zhu
   sortTable[0x7BEB]=30878;   // zhu
   sortTable[0x7D51]=30879;   // zhu
   sortTable[0x7F5C]=30880;   // zhu
   sortTable[0x8387]=30881;   // zhu
   sortTable[0x876B]=30882;   // zhu
   sortTable[0x8A85]=30883;   // zhu
   sortTable[0x8EF4]=30884;   // zhu
   sortTable[0x9252]=30885;   // zhu
   sortTable[0x92F3]=30886;   // zhu
   sortTable[0x99B5]=30887;   // zhu
   sortTable[0x9BA2]=30888;   // zhu
   sortTable[0x9BFA]=30889;   // zhu
   sortTable[0x6293]=30890;   // zhua
   sortTable[0x9AFD]=30891;   // zhua
   sortTable[0x7C3B]=30892;   // zhua
   sortTable[0x81BC]=30893;   // zhua
   sortTable[0x6A9B]=30894;   // zhua
   sortTable[0x62FD]=30895;   // zhuai
   sortTable[0x8DE9]=30896;   // zhuai
   sortTable[0x8F6C]=30897;   // zhuan
   sortTable[0x8F49]=30898;   // zhuan
   sortTable[0x4E13]=30899;   // zhuan
   sortTable[0x5C08]=30900;   // zhuan
   sortTable[0x7816]=30901;   // zhuan
   sortTable[0x78DA]=30902;   // zhuan
   sortTable[0x8D5A]=30903;   // zhuan
   sortTable[0x8CFA]=30904;   // zhuan
   sortTable[0x64B0]=30905;   // zhuan
   sortTable[0x9994]=30906;   // zhuan
   sortTable[0x7BC6]=30907;   // zhuan
   sortTable[0x989B]=30908;   // zhuan
   sortTable[0x556D]=30909;   // zhuan
   sortTable[0x8948]=30910;   // zhuan
   sortTable[0x8B54]=30911;   // zhuan
   sortTable[0x9C44]=30912;   // zhuan
   sortTable[0x911F]=30913;   // zhuan
   sortTable[0x7BF9]=30914;   // zhuan
   sortTable[0x994C]=30915;   // zhuan
   sortTable[0x819E]=30916;   // zhuan
   sortTable[0x9853]=30917;   // zhuan
   sortTable[0x8EE2]=30918;   // zhuan
   sortTable[0x8483]=30919;   // zhuan
   sortTable[0x815E]=30920;   // zhuan
   sortTable[0x50CE]=30921;   // zhuan
   sortTable[0x53C0]=30922;   // zhuan
   sortTable[0x56C0]=30923;   // zhuan
   sortTable[0x581F]=30924;   // zhuan
   sortTable[0x587C]=30925;   // zhuan
   sortTable[0x5AE5]=30926;   // zhuan
   sortTable[0x5B68]=30927;   // zhuan
   sortTable[0x5C02]=30928;   // zhuan
   sortTable[0x7077]=30929;   // zhuan
   sortTable[0x7451]=30930;   // zhuan
   sortTable[0x747C]=30931;   // zhuan
   sortTable[0x750E]=30932;   // zhuan
   sortTable[0x78D7]=30933;   // zhuan
   sortTable[0x7AF1]=30934;   // zhuan
   sortTable[0x7C51]=30935;   // zhuan
   sortTable[0x87E4]=30936;   // zhuan
   sortTable[0x8AEF]=30937;   // zhuan
   sortTable[0x88C5]=30938;   // zhuang
   sortTable[0x88DD]=30939;   // zhuang
   sortTable[0x5E84]=30940;   // zhuang
   sortTable[0x838A]=30941;   // zhuang
   sortTable[0x72B6]=30942;   // zhuang
   sortTable[0x72C0]=30943;   // zhuang
   sortTable[0x58EE]=30944;   // zhuang
   sortTable[0x58EF]=30945;   // zhuang
   sortTable[0x649E]=30946;   // zhuang
   sortTable[0x6869]=30947;   // zhuang
   sortTable[0x6A01]=30948;   // zhuang
   sortTable[0x5986]=30949;   // zhuang
   sortTable[0x7CA7]=30950;   // zhuang
   sortTable[0x8358]=30951;   // zhuang
   sortTable[0x58F5]=30952;   // zhuang
   sortTable[0x599D]=30953;   // zhuang
   sortTable[0x5A24]=30954;   // zhuang
   sortTable[0x5E92]=30955;   // zhuang
   sortTable[0x6207]=30956;   // zhuang
   sortTable[0x6889]=30957;   // zhuang
   sortTable[0x6E77]=30958;   // zhuang
   sortTable[0x6F34]=30959;   // zhuang
   sortTable[0x710B]=30960;   // zhuang
   sortTable[0x7CDA]=30961;   // zhuang
   sortTable[0x8FFD]=30962;   // zhui
   sortTable[0x7F00]=30963;   // zhui
   sortTable[0x7DB4]=30964;   // zhui
   sortTable[0x5760]=30965;   // zhui
   sortTable[0x589C]=30966;   // zhui
   sortTable[0x9525]=30967;   // zhui
   sortTable[0x8D58]=30968;   // zhui
   sortTable[0x60F4]=30969;   // zhui
   sortTable[0x7F12]=30970;   // zhui
   sortTable[0x96B9]=30971;   // zhui
   sortTable[0x9A93]=30972;   // zhui
   sortTable[0x918A]=30973;   // zhui
   sortTable[0x8D05]=30974;   // zhui
   sortTable[0x7B0D]=30975;   // zhui
   sortTable[0x8187]=30976;   // zhui
   sortTable[0x8F5B]=30977;   // zhui
   sortTable[0x7E0B]=30978;   // zhui
   sortTable[0x991F]=30979;   // zhui
   sortTable[0x787E]=30980;   // zhui
   sortTable[0x7908]=30981;   // zhui
   sortTable[0x8AC8]=30982;   // zhui
   sortTable[0x9310]=30983;   // zhui
   sortTable[0x9D7B]=30984;   // zhui
   sortTable[0x9323]=30985;   // zhui
   sortTable[0x9A05]=30986;   // zhui
   sortTable[0x5A37]=30987;   // zhui
   sortTable[0x6858]=30988;   // zhui
   sortTable[0x6C9D]=30989;   // zhui
   sortTable[0x7500]=30990;   // zhui
   sortTable[0x7577]=30991;   // zhui
   sortTable[0x8B75]=30992;   // zhui
   sortTable[0x9446]=30993;   // zhui
   sortTable[0x51C6]=30994;   // zhun
   sortTable[0x8C06]=30995;   // zhun
   sortTable[0x80AB]=30996;   // zhun
   sortTable[0x7A80]=30997;   // zhun
   sortTable[0x8FCD]=30998;   // zhun
   sortTable[0x8A30]=30999;   // zhun
   sortTable[0x7DA7]=31000;   // zhun
   sortTable[0x8AC4]=31001;   // zhun
   sortTable[0x7A15]=31002;   // zhun
   sortTable[0x8860]=31003;   // zhun
   sortTable[0x51D6]=31004;   // zhun
   sortTable[0x57FB]=31005;   // zhun
   sortTable[0x5B92]=31006;   // zhun
   sortTable[0x6E96]=31007;   // zhun
   sortTable[0x684C]=31008;   // zhuo
   sortTable[0x6349]=31009;   // zhuo
   sortTable[0x6D4A]=31010;   // zhuo
   sortTable[0x6FC1]=31011;   // zhuo
   sortTable[0x5544]=31012;   // zhuo
   sortTable[0x8301]=31013;   // zhuo
   sortTable[0x5353]=31014;   // zhuo
   sortTable[0x64E2]=31015;   // zhuo
   sortTable[0x914C]=31016;   // zhuo
   sortTable[0x62D9]=31017;   // zhuo
   sortTable[0x707C]=31018;   // zhuo
   sortTable[0x6DBF]=31019;   // zhuo
   sortTable[0x65AB]=31020;   // zhuo
   sortTable[0x956F]=31021;   // zhuo
   sortTable[0x502C]=31022;   // zhuo
   sortTable[0x6FEF]=31023;   // zhuo
   sortTable[0x6D5E]=31024;   // zhuo
   sortTable[0x8BFC]=31025;   // zhuo
   sortTable[0x9DDF]=31026;   // zhuo
   sortTable[0x9D6B]=31027;   // zhuo
   sortTable[0x799A]=31028;   // zhuo
   sortTable[0x7F6C]=31029;   // zhuo
   sortTable[0x8AD1]=31030;   // zhuo
   sortTable[0x7AA1]=31031;   // zhuo
   sortTable[0x8817]=31032;   // zhuo
   sortTable[0x7A5B]=31033;   // zhuo
   sortTable[0x7A71]=31034;   // zhuo
   sortTable[0x92DC]=31035;   // zhuo
   sortTable[0x7BE7]=31036;   // zhuo
   sortTable[0x8B36]=31037;   // zhuo
   sortTable[0x9432]=31038;   // zhuo
   sortTable[0x4E35]=31039;   // zhuo
   sortTable[0x5285]=31040;   // zhuo
   sortTable[0x53D5]=31041;   // zhuo
   sortTable[0x5545]=31042;   // zhuo
   sortTable[0x5734]=31043;   // zhuo
   sortTable[0x59B0]=31044;   // zhuo
   sortTable[0x5A3A]=31045;   // zhuo
   sortTable[0x5F74]=31046;   // zhuo
   sortTable[0x64AF]=31047;   // zhuo
   sortTable[0x64C6]=31048;   // zhuo
   sortTable[0x6580]=31049;   // zhuo
   sortTable[0x65B1]=31050;   // zhuo
   sortTable[0x65B2]=31051;   // zhuo
   sortTable[0x65B5]=31052;   // zhuo
   sortTable[0x666B]=31053;   // zhuo
   sortTable[0x68B2]=31054;   // zhuo
   sortTable[0x68C1]=31055;   // zhuo
   sortTable[0x68F3]=31056;   // zhuo
   sortTable[0x6913]=31057;   // zhuo
   sortTable[0x69D5]=31058;   // zhuo
   sortTable[0x6AE1]=31059;   // zhuo
   sortTable[0x6C4B]=31060;   // zhuo
   sortTable[0x7042]=31061;   // zhuo
   sortTable[0x70AA]=31062;   // zhuo
   sortTable[0x70F5]=31063;   // zhuo
   sortTable[0x72B3]=31064;   // zhuo
   sortTable[0x7438]=31065;   // zhuo
   sortTable[0x787A]=31066;   // zhuo
   sortTable[0x7AA7]=31067;   // zhuo
   sortTable[0x7C57]=31068;   // zhuo
   sortTable[0x7C71]=31069;   // zhuo
   sortTable[0x883F]=31070;   // zhuo
   sortTable[0x8AC1]=31071;   // zhuo
   sortTable[0x942F]=31072;   // zhuo
   sortTable[0x5B50]=31073;   // zi
   sortTable[0x81EA]=31074;   // zi
   sortTable[0x8D44]=31075;   // zi
   sortTable[0x8CC7]=31076;   // zi
   sortTable[0x5B57]=31077;   // zi
   sortTable[0x4ED4]=31078;   // zi
   sortTable[0x7D2B]=31079;   // zi
   sortTable[0x59FF]=31080;   // zi
   sortTable[0x6ECB]=31081;   // zi
   sortTable[0x59CA]=31082;   // zi
   sortTable[0x7C7D]=31083;   // zi
   sortTable[0x5179]=31084;   // zi
   sortTable[0x54A8]=31085;   // zi
   sortTable[0x5B5C]=31086;   // zi
   sortTable[0x6063]=31087;   // zi
   sortTable[0x6893]=31088;   // zi
   sortTable[0x6DC4]=31089;   // zi
   sortTable[0x8C18]=31090;   // zi
   sortTable[0x7B2B]=31091;   // zi
   sortTable[0x8F8E]=31092;   // zi
   sortTable[0x8D40]=31093;   // zi
   sortTable[0x6E0D]=31094;   // zi
   sortTable[0x6ED3]=31095;   // zi
   sortTable[0x7CA2]=31096;   // zi
   sortTable[0x7F01]=31097;   // zi
   sortTable[0x9F87]=31098;   // zi
   sortTable[0x9AED]=31099;   // zi
   sortTable[0x5B73]=31100;   // zi
   sortTable[0x8A3E]=31101;   // zi
   sortTable[0x89DC]=31102;   // zi
   sortTable[0x7726]=31103;   // zi
   sortTable[0x9531]=31104;   // zi
   sortTable[0x5D6B]=31105;   // zi
   sortTable[0x79ED]=31106;   // zi
   sortTable[0x8D91]=31107;   // zi
   sortTable[0x80D4]=31108;   // zi
   sortTable[0x9CBB]=31109;   // zi
   sortTable[0x8678]=31110;   // zi
   sortTable[0x80FE]=31111;   // zi
   sortTable[0x8DA6]=31112;   // zi
   sortTable[0x9111]=31113;   // zi
   sortTable[0x93A1]=31114;   // zi
   sortTable[0x8014]=31115;   // zi
   sortTable[0x9F12]=31116;   // zi
   sortTable[0x8332]=31117;   // zi
   sortTable[0x8A3F]=31118;   // zi
   sortTable[0x9BD4]=31119;   // zi
   sortTable[0x9543]=31120;   // zi
   sortTable[0x9D85]=31121;   // zi
   sortTable[0x9F4D]=31122;   // zi
   sortTable[0x80CF]=31123;   // zi
   sortTable[0x84FB]=31124;   // zi
   sortTable[0x7725]=31125;   // zi
   sortTable[0x79C4]=31126;   // zi
   sortTable[0x798C]=31127;   // zi
   sortTable[0x830A]=31128;   // zi
   sortTable[0x8F3A]=31129;   // zi
   sortTable[0x9C26]=31130;   // zi
   sortTable[0x79F6]=31131;   // zi
   sortTable[0x8458]=31132;   // zi
   sortTable[0x922D]=31133;   // zi
   sortTable[0x8AEE]=31134;   // zi
   sortTable[0x8321]=31135;   // zi
   sortTable[0x8CB2]=31136;   // zi
   sortTable[0x983E]=31137;   // zi
   sortTable[0x8293]=31138;   // zi
   sortTable[0x4E72]=31139;   // zi
   sortTable[0x5033]=31140;   // zi
   sortTable[0x525A]=31141;   // zi
   sortTable[0x5407]=31142;   // zi
   sortTable[0x5470]=31143;   // zi
   sortTable[0x5559]=31144;   // zi
   sortTable[0x55DE]=31145;   // zi
   sortTable[0x59C9]=31146;   // zi
   sortTable[0x59D5]=31147;   // zi
   sortTable[0x5B76]=31148;   // zi
   sortTable[0x5D30]=31149;   // zi
   sortTable[0x674D]=31150;   // zi
   sortTable[0x6825]=31151;   // zi
   sortTable[0x6914]=31152;   // zi
   sortTable[0x699F]=31153;   // zi
   sortTable[0x6A74]=31154;   // zi
   sortTable[0x6E7D]=31155;   // zi
   sortTable[0x6F2C]=31156;   // zi
   sortTable[0x6FAC]=31157;   // zi
   sortTable[0x7278]=31158;   // zi
   sortTable[0x7386]=31159;   // zi
   sortTable[0x74BE]=31160;   // zi
   sortTable[0x77F7]=31161;   // zi
   sortTable[0x7A35]=31162;   // zi
   sortTable[0x7D0E]=31163;   // zi
   sortTable[0x7DC7]=31164;   // zi
   sortTable[0x8F1C]=31165;   // zi
   sortTable[0x91E8]=31166;   // zi
   sortTable[0x9319]=31167;   // zi
   sortTable[0x937F]=31168;   // zi
   sortTable[0x983F]=31169;   // zi
   sortTable[0x603B]=31170;   // zong
   sortTable[0x7E3D]=31171;   // zong
   sortTable[0x7EFC]=31172;   // zong
   sortTable[0x7D9C]=31173;   // zong
   sortTable[0x5B97]=31174;   // zong
   sortTable[0x7EB5]=31175;   // zong
   sortTable[0x7E31]=31176;   // zong
   sortTable[0x8E2A]=31177;   // zong
   sortTable[0x8E64]=31178;   // zong
   sortTable[0x68D5]=31179;   // zong
   sortTable[0x9B03]=31180;   // zong
   sortTable[0x7CBD]=31181;   // zong
   sortTable[0x506C]=31182;   // zong
   sortTable[0x8159]=31183;   // zong
   sortTable[0x9B37]=31184;   // zong
   sortTable[0x9350]=31185;   // zong
   sortTable[0x9BEE]=31186;   // zong
   sortTable[0x7DC3]=31187;   // zong
   sortTable[0x78EB]=31188;   // zong
   sortTable[0x931D]=31189;   // zong
   sortTable[0x7DF5]=31190;   // zong
   sortTable[0x8C75]=31191;   // zong
   sortTable[0x9A23]=31192;   // zong
   sortTable[0x9A94]=31193;   // zong
   sortTable[0x7A2F]=31194;   // zong
   sortTable[0x7E02]=31195;   // zong
   sortTable[0x7FEA]=31196;   // zong
   sortTable[0x847C]=31197;   // zong
   sortTable[0x7CC9]=31198;   // zong
   sortTable[0x84D7]=31199;   // zong
   sortTable[0x8E28]=31200;   // zong
   sortTable[0x9B09]=31201;   // zong
   sortTable[0x500A]=31202;   // zong
   sortTable[0x5027]=31203;   // zong
   sortTable[0x50AF]=31204;   // zong
   sortTable[0x582B]=31205;   // zong
   sortTable[0x5D4F]=31206;   // zong
   sortTable[0x5D55]=31207;   // zong
   sortTable[0x5D78]=31208;   // zong
   sortTable[0x60E3]=31209;   // zong
   sortTable[0x60FE]=31210;   // zong
   sortTable[0x6121]=31211;   // zong
   sortTable[0x6374]=31212;   // zong
   sortTable[0x63D4]=31213;   // zong
   sortTable[0x6403]=31214;   // zong
   sortTable[0x6460]=31215;   // zong
   sortTable[0x662E]=31216;   // zong
   sortTable[0x6721]=31217;   // zong
   sortTable[0x6936]=31218;   // zong
   sortTable[0x6F48]=31219;   // zong
   sortTable[0x71A7]=31220;   // zong
   sortTable[0x7314]=31221;   // zong
   sortTable[0x7323]=31222;   // zong
   sortTable[0x75AD]=31223;   // zong
   sortTable[0x7632]=31224;   // zong
   sortTable[0x7882]=31225;   // zong
   sortTable[0x7CED]=31226;   // zong
   sortTable[0x7DCF]=31227;   // zong
   sortTable[0x7E26]=31228;   // zong
   sortTable[0x876C]=31229;   // zong
   sortTable[0x93D3]=31230;   // zong
   sortTable[0x9441]=31231;   // zong
   sortTable[0x9A0C]=31232;   // zong
   sortTable[0x9BFC]=31233;   // zong
   sortTable[0x8D70]=31234;   // zou
   sortTable[0x594F]=31235;   // zou
   sortTable[0x63CD]=31236;   // zou
   sortTable[0x90B9]=31237;   // zou
   sortTable[0x8BF9]=31238;   // zou
   sortTable[0x9A7A]=31239;   // zou
   sortTable[0x966C]=31240;   // zou
   sortTable[0x9139]=31241;   // zou
   sortTable[0x9CB0]=31242;   // zou
   sortTable[0x9BD0]=31243;   // zou
   sortTable[0x83C6]=31244;   // zou
   sortTable[0x7DC5]=31245;   // zou
   sortTable[0x9F7A]=31246;   // zou
   sortTable[0x6971]=31247;   // zou
   sortTable[0x90F0]=31248;   // zou
   sortTable[0x9F71]=31249;   // zou
   sortTable[0x8ACF]=31250;   // zou
   sortTable[0x68F7]=31251;   // zou
   sortTable[0x68F8]=31252;   // zou
   sortTable[0x7B83]=31253;   // zou
   sortTable[0x8D71]=31254;   // zou
   sortTable[0x9112]=31255;   // zou
   sortTable[0x9A36]=31256;   // zou
   sortTable[0x9BEB]=31257;   // zou
   sortTable[0x9EC0]=31258;   // zou
   sortTable[0x7EC4]=31259;   // zu
   sortTable[0x7D44]=31260;   // zu
   sortTable[0x65CF]=31261;   // zu
   sortTable[0x7956]=31262;   // zu
   sortTable[0x8DB3]=31263;   // zu
   sortTable[0x963B]=31264;   // zu
   sortTable[0x79DF]=31265;   // zu
   sortTable[0x5352]=31266;   // zu
   sortTable[0x4FCE]=31267;   // zu
   sortTable[0x8BC5]=31268;   // zu
   sortTable[0x955E]=31269;   // zu
   sortTable[0x8445]=31270;   // zu
   sortTable[0x84A9]=31271;   // zu
   sortTable[0x7BA4]=31272;   // zu
   sortTable[0x8A5B]=31273;   // zu
   sortTable[0x93BA]=31274;   // zu
   sortTable[0x93C3]=31275;   // zu
   sortTable[0x977B]=31276;   // zu
   sortTable[0x50B6]=31277;   // zu
   sortTable[0x5346]=31278;   // zu
   sortTable[0x54EB]=31279;   // zu
   sortTable[0x5D12]=31280;   // zu
   sortTable[0x5D2A]=31281;   // zu
   sortTable[0x723C]=31282;   // zu
   sortTable[0x73C7]=31283;   // zu
   sortTable[0x8E24]=31284;   // zu
   sortTable[0x8E3F]=31285;   // zu
   sortTable[0x94BB]=31286;   // zuan
   sortTable[0x947D]=31287;   // zuan
   sortTable[0x6525]=31288;   // zuan
   sortTable[0x7E82]=31289;   // zuan
   sortTable[0x7F35]=31290;   // zuan
   sortTable[0x8E9C]=31291;   // zuan
   sortTable[0x7E98]=31292;   // zuan
   sortTable[0x7C6B]=31293;   // zuan
   sortTable[0x7E64]=31294;   // zuan
   sortTable[0x7E89]=31295;   // zuan
   sortTable[0x945A]=31296;   // zuan
   sortTable[0x6700]=31297;   // zui
   sortTable[0x5634]=31298;   // zui
   sortTable[0x7F6A]=31299;   // zui
   sortTable[0x9189]=31300;   // zui
   sortTable[0x855E]=31301;   // zui
   sortTable[0x7A21]=31302;   // zui
   sortTable[0x92F7]=31303;   // zui
   sortTable[0x87D5]=31304;   // zui
   sortTable[0x797D]=31305;   // zui
   sortTable[0x7E97]=31306;   // zui
   sortTable[0x539C]=31307;   // zui
   sortTable[0x55FA]=31308;   // zui
   sortTable[0x567F]=31309;   // zui
   sortTable[0x5D8A]=31310;   // zui
   sortTable[0x5DB5]=31311;   // zui
   sortTable[0x666C]=31312;   // zui
   sortTable[0x6718]=31313;   // zui
   sortTable[0x67A0]=31314;   // zui
   sortTable[0x682C]=31315;   // zui
   sortTable[0x69DC]=31316;   // zui
   sortTable[0x6A36]=31317;   // zui
   sortTable[0x6A87]=31318;   // zui
   sortTable[0x6A8C]=31319;   // zui
   sortTable[0x74BB]=31320;   // zui
   sortTable[0x7A5D]=31321;   // zui
   sortTable[0x7D4A]=31322;   // zui
   sortTable[0x8FA0]=31323;   // zui
   sortTable[0x9154]=31324;   // zui
   sortTable[0x917B]=31325;   // zui
   sortTable[0x930A]=31326;   // zui
   sortTable[0x5C0A]=31327;   // zun
   sortTable[0x9075]=31328;   // zun
   sortTable[0x6A3D]=31329;   // zun
   sortTable[0x6499]=31330;   // zun
   sortTable[0x9CDF]=31331;   // zun
   sortTable[0x940F]=31332;   // zun
   sortTable[0x7F47]=31333;   // zun
   sortTable[0x9D8E]=31334;   // zun
   sortTable[0x9DF7]=31335;   // zun
   sortTable[0x7E5C]=31336;   // zun
   sortTable[0x8B50]=31337;   // zun
   sortTable[0x928C]=31338;   // zun
   sortTable[0x50D4]=31339;   // zun
   sortTable[0x5642]=31340;   // zun
   sortTable[0x58AB]=31341;   // zun
   sortTable[0x58FF]=31342;   // zun
   sortTable[0x5D9F]=31343;   // zun
   sortTable[0x6358]=31344;   // zun
   sortTable[0x9C52]=31345;   // zun
   sortTable[0x4F5C]=31346;   // zuo
   sortTable[0x505A]=31347;   // zuo
   sortTable[0x5750]=31348;   // zuo
   sortTable[0x5DE6]=31349;   // zuo
   sortTable[0x5EA7]=31350;   // zuo
   sortTable[0x6628]=31351;   // zuo
   sortTable[0x7422]=31352;   // zuo
   sortTable[0x4F50]=31353;   // zuo
   sortTable[0x795A]=31354;   // zuo
   sortTable[0x963C]=31355;   // zuo
   sortTable[0x80D9]=31356;   // zuo
   sortTable[0x5511]=31357;   // zuo
   sortTable[0x600D]=31358;   // zuo
   sortTable[0x838B]=31359;   // zuo
   sortTable[0x7B70]=31360;   // zuo
   sortTable[0x7A13]=31361;   // zuo
   sortTable[0x7E53]=31362;   // zuo
   sortTable[0x84D9]=31363;   // zuo
   sortTable[0x888F]=31364;   // zuo
   sortTable[0x8444]=31365;   // zuo
   sortTable[0x923C]=31366;   // zuo
   sortTable[0x98F5]=31367;   // zuo
   sortTable[0x4FB3]=31368;   // zuo
   sortTable[0x5497]=31369;   // zuo
   sortTable[0x5528]=31370;   // zuo
   sortTable[0x5C9D]=31371;   // zuo
   sortTable[0x5C9E]=31372;   // zuo
   sortTable[0x637D]=31373;   // zuo
   sortTable[0x690A]=31374;   // zuo
   sortTable[0x79E8]=31375;   // zuo
   sortTable[0x7CF3]=31376;   // zuo
   sortTable[0x8443]=31377;   // zuo
   sortTable[0x685B]=31378;   // zzz
   sortTable[0x70EA]=31379;   // zzz
   sortTable[0x74E7]=31380;   // zzz
   sortTable[0x74F0]=31381;   // zzz
   sortTable[0x74F1]=31382;   // zzz
   sortTable[0x74FC]=31383;   // zzz
   sortTable[0x7505]=31384;   // zzz
   sortTable[0x3131]=33024;   // HANGUL SYLLABLE
   sortTable[0x1100]=33025;   // HANGUL CHOSEONG KIYEOK
   sortTable[0x11A8]=33026;   // HANGUL JONGSEONG KIYEOK
   sortTable[0xAC00]=33027;   // HANGUL SYLLABLE
   sortTable[0xAC01]=33028;   // HANGUL SYLLABLE
   sortTable[0xAC02]=33029;   // HANGUL SYLLABLE
   sortTable[0xAC03]=33030;   // HANGUL SYLLABLE
   sortTable[0xAC04]=33031;   // HANGUL SYLLABLE
   sortTable[0xAC05]=33032;   // HANGUL SYLLABLE
   sortTable[0xAC06]=33033;   // HANGUL SYLLABLE
   sortTable[0xAC07]=33034;   // HANGUL SYLLABLE
   sortTable[0xAC08]=33035;   // HANGUL SYLLABLE
   sortTable[0xAC09]=33036;   // HANGUL SYLLABLE
   sortTable[0xAC0A]=33037;   // HANGUL SYLLABLE
   sortTable[0xAC0B]=33038;   // HANGUL SYLLABLE
   sortTable[0xAC0C]=33039;   // HANGUL SYLLABLE
   sortTable[0xAC0D]=33040;   // HANGUL SYLLABLE
   sortTable[0xAC0E]=33041;   // HANGUL SYLLABLE
   sortTable[0xAC0F]=33042;   // HANGUL SYLLABLE
   sortTable[0xAC10]=33043;   // HANGUL SYLLABLE
   sortTable[0xAC11]=33044;   // HANGUL SYLLABLE
   sortTable[0xAC12]=33045;   // HANGUL SYLLABLE
   sortTable[0xAC13]=33046;   // HANGUL SYLLABLE
   sortTable[0xAC14]=33047;   // HANGUL SYLLABLE
   sortTable[0xAC15]=33048;   // HANGUL SYLLABLE
   sortTable[0xAC16]=33049;   // HANGUL SYLLABLE
   sortTable[0xAC17]=33050;   // HANGUL SYLLABLE
   sortTable[0xAC18]=33051;   // HANGUL SYLLABLE
   sortTable[0xAC19]=33052;   // HANGUL SYLLABLE
   sortTable[0xAC1A]=33053;   // HANGUL SYLLABLE
   sortTable[0xAC1B]=33054;   // HANGUL SYLLABLE
   sortTable[0xAC1C]=33055;   // HANGUL SYLLABLE
   sortTable[0xAC1D]=33056;   // HANGUL SYLLABLE
   sortTable[0xAC1E]=33057;   // HANGUL SYLLABLE
   sortTable[0xAC1F]=33058;   // HANGUL SYLLABLE
   sortTable[0xAC20]=33059;   // HANGUL SYLLABLE
   sortTable[0xAC21]=33060;   // HANGUL SYLLABLE
   sortTable[0xAC22]=33061;   // HANGUL SYLLABLE
   sortTable[0xAC23]=33062;   // HANGUL SYLLABLE
   sortTable[0xAC24]=33063;   // HANGUL SYLLABLE
   sortTable[0xAC25]=33064;   // HANGUL SYLLABLE
   sortTable[0xAC26]=33065;   // HANGUL SYLLABLE
   sortTable[0xAC27]=33066;   // HANGUL SYLLABLE
   sortTable[0xAC28]=33067;   // HANGUL SYLLABLE
   sortTable[0xAC29]=33068;   // HANGUL SYLLABLE
   sortTable[0xAC2A]=33069;   // HANGUL SYLLABLE
   sortTable[0xAC2B]=33070;   // HANGUL SYLLABLE
   sortTable[0xAC2C]=33071;   // HANGUL SYLLABLE
   sortTable[0xAC2D]=33072;   // HANGUL SYLLABLE
   sortTable[0xAC2E]=33073;   // HANGUL SYLLABLE
   sortTable[0xAC2F]=33074;   // HANGUL SYLLABLE
   sortTable[0xAC30]=33075;   // HANGUL SYLLABLE
   sortTable[0xAC31]=33076;   // HANGUL SYLLABLE
   sortTable[0xAC32]=33077;   // HANGUL SYLLABLE
   sortTable[0xAC33]=33078;   // HANGUL SYLLABLE
   sortTable[0xAC34]=33079;   // HANGUL SYLLABLE
   sortTable[0xAC35]=33080;   // HANGUL SYLLABLE
   sortTable[0xAC36]=33081;   // HANGUL SYLLABLE
   sortTable[0xAC37]=33082;   // HANGUL SYLLABLE
   sortTable[0xAC38]=33083;   // HANGUL SYLLABLE
   sortTable[0xAC39]=33084;   // HANGUL SYLLABLE
   sortTable[0xAC3A]=33085;   // HANGUL SYLLABLE
   sortTable[0xAC3B]=33086;   // HANGUL SYLLABLE
   sortTable[0xAC3C]=33087;   // HANGUL SYLLABLE
   sortTable[0xAC3D]=33088;   // HANGUL SYLLABLE
   sortTable[0xAC3E]=33089;   // HANGUL SYLLABLE
   sortTable[0xAC3F]=33090;   // HANGUL SYLLABLE
   sortTable[0xAC40]=33091;   // HANGUL SYLLABLE
   sortTable[0xAC41]=33092;   // HANGUL SYLLABLE
   sortTable[0xAC42]=33093;   // HANGUL SYLLABLE
   sortTable[0xAC43]=33094;   // HANGUL SYLLABLE
   sortTable[0xAC44]=33095;   // HANGUL SYLLABLE
   sortTable[0xAC45]=33096;   // HANGUL SYLLABLE
   sortTable[0xAC46]=33097;   // HANGUL SYLLABLE
   sortTable[0xAC47]=33098;   // HANGUL SYLLABLE
   sortTable[0xAC48]=33099;   // HANGUL SYLLABLE
   sortTable[0xAC49]=33100;   // HANGUL SYLLABLE
   sortTable[0xAC4A]=33101;   // HANGUL SYLLABLE
   sortTable[0xAC4B]=33102;   // HANGUL SYLLABLE
   sortTable[0xAC4C]=33103;   // HANGUL SYLLABLE
   sortTable[0xAC4D]=33104;   // HANGUL SYLLABLE
   sortTable[0xAC4E]=33105;   // HANGUL SYLLABLE
   sortTable[0xAC4F]=33106;   // HANGUL SYLLABLE
   sortTable[0xAC50]=33107;   // HANGUL SYLLABLE
   sortTable[0xAC51]=33108;   // HANGUL SYLLABLE
   sortTable[0xAC52]=33109;   // HANGUL SYLLABLE
   sortTable[0xAC53]=33110;   // HANGUL SYLLABLE
   sortTable[0xAC54]=33111;   // HANGUL SYLLABLE
   sortTable[0xAC55]=33112;   // HANGUL SYLLABLE
   sortTable[0xAC56]=33113;   // HANGUL SYLLABLE
   sortTable[0xAC57]=33114;   // HANGUL SYLLABLE
   sortTable[0xAC58]=33115;   // HANGUL SYLLABLE
   sortTable[0xAC59]=33116;   // HANGUL SYLLABLE
   sortTable[0xAC5A]=33117;   // HANGUL SYLLABLE
   sortTable[0xAC5B]=33118;   // HANGUL SYLLABLE
   sortTable[0xAC5C]=33119;   // HANGUL SYLLABLE
   sortTable[0xAC5D]=33120;   // HANGUL SYLLABLE
   sortTable[0xAC5E]=33121;   // HANGUL SYLLABLE
   sortTable[0xAC5F]=33122;   // HANGUL SYLLABLE
   sortTable[0xAC60]=33123;   // HANGUL SYLLABLE
   sortTable[0xAC61]=33124;   // HANGUL SYLLABLE
   sortTable[0xAC62]=33125;   // HANGUL SYLLABLE
   sortTable[0xAC63]=33126;   // HANGUL SYLLABLE
   sortTable[0xAC64]=33127;   // HANGUL SYLLABLE
   sortTable[0xAC65]=33128;   // HANGUL SYLLABLE
   sortTable[0xAC66]=33129;   // HANGUL SYLLABLE
   sortTable[0xAC67]=33130;   // HANGUL SYLLABLE
   sortTable[0xAC68]=33131;   // HANGUL SYLLABLE
   sortTable[0xAC69]=33132;   // HANGUL SYLLABLE
   sortTable[0xAC6A]=33133;   // HANGUL SYLLABLE
   sortTable[0xAC6B]=33134;   // HANGUL SYLLABLE
   sortTable[0xAC6C]=33135;   // HANGUL SYLLABLE
   sortTable[0xAC6D]=33136;   // HANGUL SYLLABLE
   sortTable[0xAC6E]=33137;   // HANGUL SYLLABLE
   sortTable[0xAC6F]=33138;   // HANGUL SYLLABLE
   sortTable[0xAC70]=33139;   // HANGUL SYLLABLE
   sortTable[0xAC71]=33140;   // HANGUL SYLLABLE
   sortTable[0xAC72]=33141;   // HANGUL SYLLABLE
   sortTable[0xAC73]=33142;   // HANGUL SYLLABLE
   sortTable[0xAC74]=33143;   // HANGUL SYLLABLE
   sortTable[0xAC75]=33144;   // HANGUL SYLLABLE
   sortTable[0xAC76]=33145;   // HANGUL SYLLABLE
   sortTable[0xAC77]=33146;   // HANGUL SYLLABLE
   sortTable[0xAC78]=33147;   // HANGUL SYLLABLE
   sortTable[0xAC79]=33148;   // HANGUL SYLLABLE
   sortTable[0xAC7A]=33149;   // HANGUL SYLLABLE
   sortTable[0xAC7B]=33150;   // HANGUL SYLLABLE
   sortTable[0xAC7C]=33151;   // HANGUL SYLLABLE
   sortTable[0xAC7D]=33152;   // HANGUL SYLLABLE
   sortTable[0xAC7E]=33153;   // HANGUL SYLLABLE
   sortTable[0xAC7F]=33154;   // HANGUL SYLLABLE
   sortTable[0xAC80]=33155;   // HANGUL SYLLABLE
   sortTable[0xAC81]=33156;   // HANGUL SYLLABLE
   sortTable[0xAC82]=33157;   // HANGUL SYLLABLE
   sortTable[0xAC83]=33158;   // HANGUL SYLLABLE
   sortTable[0xAC84]=33159;   // HANGUL SYLLABLE
   sortTable[0xAC85]=33160;   // HANGUL SYLLABLE
   sortTable[0xAC86]=33161;   // HANGUL SYLLABLE
   sortTable[0xAC87]=33162;   // HANGUL SYLLABLE
   sortTable[0xAC88]=33163;   // HANGUL SYLLABLE
   sortTable[0xAC89]=33164;   // HANGUL SYLLABLE
   sortTable[0xAC8A]=33165;   // HANGUL SYLLABLE
   sortTable[0xAC8B]=33166;   // HANGUL SYLLABLE
   sortTable[0xAC8C]=33167;   // HANGUL SYLLABLE
   sortTable[0xAC8D]=33168;   // HANGUL SYLLABLE
   sortTable[0xAC8E]=33169;   // HANGUL SYLLABLE
   sortTable[0xAC8F]=33170;   // HANGUL SYLLABLE
   sortTable[0xAC90]=33171;   // HANGUL SYLLABLE
   sortTable[0xAC91]=33172;   // HANGUL SYLLABLE
   sortTable[0xAC92]=33173;   // HANGUL SYLLABLE
   sortTable[0xAC93]=33174;   // HANGUL SYLLABLE
   sortTable[0xAC94]=33175;   // HANGUL SYLLABLE
   sortTable[0xAC95]=33176;   // HANGUL SYLLABLE
   sortTable[0xAC96]=33177;   // HANGUL SYLLABLE
   sortTable[0xAC97]=33178;   // HANGUL SYLLABLE
   sortTable[0xAC98]=33179;   // HANGUL SYLLABLE
   sortTable[0xAC99]=33180;   // HANGUL SYLLABLE
   sortTable[0xAC9A]=33181;   // HANGUL SYLLABLE
   sortTable[0xAC9B]=33182;   // HANGUL SYLLABLE
   sortTable[0xAC9C]=33183;   // HANGUL SYLLABLE
   sortTable[0xAC9D]=33184;   // HANGUL SYLLABLE
   sortTable[0xAC9E]=33185;   // HANGUL SYLLABLE
   sortTable[0xAC9F]=33186;   // HANGUL SYLLABLE
   sortTable[0xACA0]=33187;   // HANGUL SYLLABLE
   sortTable[0xACA1]=33188;   // HANGUL SYLLABLE
   sortTable[0xACA2]=33189;   // HANGUL SYLLABLE
   sortTable[0xACA3]=33190;   // HANGUL SYLLABLE
   sortTable[0xACA4]=33191;   // HANGUL SYLLABLE
   sortTable[0xACA5]=33192;   // HANGUL SYLLABLE
   sortTable[0xACA6]=33193;   // HANGUL SYLLABLE
   sortTable[0xACA7]=33194;   // HANGUL SYLLABLE
   sortTable[0xACA8]=33195;   // HANGUL SYLLABLE
   sortTable[0xACA9]=33196;   // HANGUL SYLLABLE
   sortTable[0xACAA]=33197;   // HANGUL SYLLABLE
   sortTable[0xACAB]=33198;   // HANGUL SYLLABLE
   sortTable[0xACAC]=33199;   // HANGUL SYLLABLE
   sortTable[0xACAD]=33200;   // HANGUL SYLLABLE
   sortTable[0xACAE]=33201;   // HANGUL SYLLABLE
   sortTable[0xACAF]=33202;   // HANGUL SYLLABLE
   sortTable[0xACB0]=33203;   // HANGUL SYLLABLE
   sortTable[0xACB1]=33204;   // HANGUL SYLLABLE
   sortTable[0xACB2]=33205;   // HANGUL SYLLABLE
   sortTable[0xACB3]=33206;   // HANGUL SYLLABLE
   sortTable[0xACB4]=33207;   // HANGUL SYLLABLE
   sortTable[0xACB5]=33208;   // HANGUL SYLLABLE
   sortTable[0xACB6]=33209;   // HANGUL SYLLABLE
   sortTable[0xACB7]=33210;   // HANGUL SYLLABLE
   sortTable[0xACB8]=33211;   // HANGUL SYLLABLE
   sortTable[0xACB9]=33212;   // HANGUL SYLLABLE
   sortTable[0xACBA]=33213;   // HANGUL SYLLABLE
   sortTable[0xACBB]=33214;   // HANGUL SYLLABLE
   sortTable[0xACBC]=33215;   // HANGUL SYLLABLE
   sortTable[0xACBD]=33216;   // HANGUL SYLLABLE
   sortTable[0xACBE]=33217;   // HANGUL SYLLABLE
   sortTable[0xACBF]=33218;   // HANGUL SYLLABLE
   sortTable[0xACC0]=33219;   // HANGUL SYLLABLE
   sortTable[0xACC1]=33220;   // HANGUL SYLLABLE
   sortTable[0xACC2]=33221;   // HANGUL SYLLABLE
   sortTable[0xACC3]=33222;   // HANGUL SYLLABLE
   sortTable[0xACC4]=33223;   // HANGUL SYLLABLE
   sortTable[0xACC5]=33224;   // HANGUL SYLLABLE
   sortTable[0xACC6]=33225;   // HANGUL SYLLABLE
   sortTable[0xACC7]=33226;   // HANGUL SYLLABLE
   sortTable[0xACC8]=33227;   // HANGUL SYLLABLE
   sortTable[0xACC9]=33228;   // HANGUL SYLLABLE
   sortTable[0xACCA]=33229;   // HANGUL SYLLABLE
   sortTable[0xACCB]=33230;   // HANGUL SYLLABLE
   sortTable[0xACCC]=33231;   // HANGUL SYLLABLE
   sortTable[0xACCD]=33232;   // HANGUL SYLLABLE
   sortTable[0xACCE]=33233;   // HANGUL SYLLABLE
   sortTable[0xACCF]=33234;   // HANGUL SYLLABLE
   sortTable[0xACD0]=33235;   // HANGUL SYLLABLE
   sortTable[0xACD1]=33236;   // HANGUL SYLLABLE
   sortTable[0xACD2]=33237;   // HANGUL SYLLABLE
   sortTable[0xACD3]=33238;   // HANGUL SYLLABLE
   sortTable[0xACD4]=33239;   // HANGUL SYLLABLE
   sortTable[0xACD5]=33240;   // HANGUL SYLLABLE
   sortTable[0xACD6]=33241;   // HANGUL SYLLABLE
   sortTable[0xACD7]=33242;   // HANGUL SYLLABLE
   sortTable[0xACD8]=33243;   // HANGUL SYLLABLE
   sortTable[0xACD9]=33244;   // HANGUL SYLLABLE
   sortTable[0xACDA]=33245;   // HANGUL SYLLABLE
   sortTable[0xACDB]=33246;   // HANGUL SYLLABLE
   sortTable[0xACDC]=33247;   // HANGUL SYLLABLE
   sortTable[0xACDD]=33248;   // HANGUL SYLLABLE
   sortTable[0xACDE]=33249;   // HANGUL SYLLABLE
   sortTable[0xACDF]=33250;   // HANGUL SYLLABLE
   sortTable[0xACE0]=33251;   // HANGUL SYLLABLE
   sortTable[0xACE1]=33252;   // HANGUL SYLLABLE
   sortTable[0xACE2]=33253;   // HANGUL SYLLABLE
   sortTable[0xACE3]=33254;   // HANGUL SYLLABLE
   sortTable[0xACE4]=33255;   // HANGUL SYLLABLE
   sortTable[0xACE5]=33256;   // HANGUL SYLLABLE
   sortTable[0xACE6]=33257;   // HANGUL SYLLABLE
   sortTable[0xACE7]=33258;   // HANGUL SYLLABLE
   sortTable[0xACE8]=33259;   // HANGUL SYLLABLE
   sortTable[0xACE9]=33260;   // HANGUL SYLLABLE
   sortTable[0xACEA]=33261;   // HANGUL SYLLABLE
   sortTable[0xACEB]=33262;   // HANGUL SYLLABLE
   sortTable[0xACEC]=33263;   // HANGUL SYLLABLE
   sortTable[0xACED]=33264;   // HANGUL SYLLABLE
   sortTable[0xACEE]=33265;   // HANGUL SYLLABLE
   sortTable[0xACEF]=33266;   // HANGUL SYLLABLE
   sortTable[0xACF0]=33267;   // HANGUL SYLLABLE
   sortTable[0xACF1]=33268;   // HANGUL SYLLABLE
   sortTable[0xACF2]=33269;   // HANGUL SYLLABLE
   sortTable[0xACF3]=33270;   // HANGUL SYLLABLE
   sortTable[0xACF4]=33271;   // HANGUL SYLLABLE
   sortTable[0xACF5]=33272;   // HANGUL SYLLABLE
   sortTable[0xACF6]=33273;   // HANGUL SYLLABLE
   sortTable[0xACF7]=33274;   // HANGUL SYLLABLE
   sortTable[0xACF8]=33275;   // HANGUL SYLLABLE
   sortTable[0xACF9]=33276;   // HANGUL SYLLABLE
   sortTable[0xACFA]=33277;   // HANGUL SYLLABLE
   sortTable[0xACFB]=33278;   // HANGUL SYLLABLE
   sortTable[0xACFC]=33279;   // HANGUL SYLLABLE
   sortTable[0xACFD]=33280;   // HANGUL SYLLABLE
   sortTable[0xACFE]=33281;   // HANGUL SYLLABLE
   sortTable[0xACFF]=33282;   // HANGUL SYLLABLE
   sortTable[0xAD00]=33283;   // HANGUL SYLLABLE
   sortTable[0xAD01]=33284;   // HANGUL SYLLABLE
   sortTable[0xAD02]=33285;   // HANGUL SYLLABLE
   sortTable[0xAD03]=33286;   // HANGUL SYLLABLE
   sortTable[0xAD04]=33287;   // HANGUL SYLLABLE
   sortTable[0xAD05]=33288;   // HANGUL SYLLABLE
   sortTable[0xAD06]=33289;   // HANGUL SYLLABLE
   sortTable[0xAD07]=33290;   // HANGUL SYLLABLE
   sortTable[0xAD08]=33291;   // HANGUL SYLLABLE
   sortTable[0xAD09]=33292;   // HANGUL SYLLABLE
   sortTable[0xAD0A]=33293;   // HANGUL SYLLABLE
   sortTable[0xAD0B]=33294;   // HANGUL SYLLABLE
   sortTable[0xAD0C]=33295;   // HANGUL SYLLABLE
   sortTable[0xAD0D]=33296;   // HANGUL SYLLABLE
   sortTable[0xAD0E]=33297;   // HANGUL SYLLABLE
   sortTable[0xAD0F]=33298;   // HANGUL SYLLABLE
   sortTable[0xAD10]=33299;   // HANGUL SYLLABLE
   sortTable[0xAD11]=33300;   // HANGUL SYLLABLE
   sortTable[0xAD12]=33301;   // HANGUL SYLLABLE
   sortTable[0xAD13]=33302;   // HANGUL SYLLABLE
   sortTable[0xAD14]=33303;   // HANGUL SYLLABLE
   sortTable[0xAD15]=33304;   // HANGUL SYLLABLE
   sortTable[0xAD16]=33305;   // HANGUL SYLLABLE
   sortTable[0xAD17]=33306;   // HANGUL SYLLABLE
   sortTable[0xAD18]=33307;   // HANGUL SYLLABLE
   sortTable[0xAD19]=33308;   // HANGUL SYLLABLE
   sortTable[0xAD1A]=33309;   // HANGUL SYLLABLE
   sortTable[0xAD1B]=33310;   // HANGUL SYLLABLE
   sortTable[0xAD1C]=33311;   // HANGUL SYLLABLE
   sortTable[0xAD1D]=33312;   // HANGUL SYLLABLE
   sortTable[0xAD1E]=33313;   // HANGUL SYLLABLE
   sortTable[0xAD1F]=33314;   // HANGUL SYLLABLE
   sortTable[0xAD20]=33315;   // HANGUL SYLLABLE
   sortTable[0xAD21]=33316;   // HANGUL SYLLABLE
   sortTable[0xAD22]=33317;   // HANGUL SYLLABLE
   sortTable[0xAD23]=33318;   // HANGUL SYLLABLE
   sortTable[0xAD24]=33319;   // HANGUL SYLLABLE
   sortTable[0xAD25]=33320;   // HANGUL SYLLABLE
   sortTable[0xAD26]=33321;   // HANGUL SYLLABLE
   sortTable[0xAD27]=33322;   // HANGUL SYLLABLE
   sortTable[0xAD28]=33323;   // HANGUL SYLLABLE
   sortTable[0xAD29]=33324;   // HANGUL SYLLABLE
   sortTable[0xAD2A]=33325;   // HANGUL SYLLABLE
   sortTable[0xAD2B]=33326;   // HANGUL SYLLABLE
   sortTable[0xAD2C]=33327;   // HANGUL SYLLABLE
   sortTable[0xAD2D]=33328;   // HANGUL SYLLABLE
   sortTable[0xAD2E]=33329;   // HANGUL SYLLABLE
   sortTable[0xAD2F]=33330;   // HANGUL SYLLABLE
   sortTable[0xAD30]=33331;   // HANGUL SYLLABLE
   sortTable[0xAD31]=33332;   // HANGUL SYLLABLE
   sortTable[0xAD32]=33333;   // HANGUL SYLLABLE
   sortTable[0xAD33]=33334;   // HANGUL SYLLABLE
   sortTable[0xAD34]=33335;   // HANGUL SYLLABLE
   sortTable[0xAD35]=33336;   // HANGUL SYLLABLE
   sortTable[0xAD36]=33337;   // HANGUL SYLLABLE
   sortTable[0xAD37]=33338;   // HANGUL SYLLABLE
   sortTable[0xAD38]=33339;   // HANGUL SYLLABLE
   sortTable[0xAD39]=33340;   // HANGUL SYLLABLE
   sortTable[0xAD3A]=33341;   // HANGUL SYLLABLE
   sortTable[0xAD3B]=33342;   // HANGUL SYLLABLE
   sortTable[0xAD3C]=33343;   // HANGUL SYLLABLE
   sortTable[0xAD3D]=33344;   // HANGUL SYLLABLE
   sortTable[0xAD3E]=33345;   // HANGUL SYLLABLE
   sortTable[0xAD3F]=33346;   // HANGUL SYLLABLE
   sortTable[0xAD40]=33347;   // HANGUL SYLLABLE
   sortTable[0xAD41]=33348;   // HANGUL SYLLABLE
   sortTable[0xAD42]=33349;   // HANGUL SYLLABLE
   sortTable[0xAD43]=33350;   // HANGUL SYLLABLE
   sortTable[0xAD44]=33351;   // HANGUL SYLLABLE
   sortTable[0xAD45]=33352;   // HANGUL SYLLABLE
   sortTable[0xAD46]=33353;   // HANGUL SYLLABLE
   sortTable[0xAD47]=33354;   // HANGUL SYLLABLE
   sortTable[0xAD48]=33355;   // HANGUL SYLLABLE
   sortTable[0xAD49]=33356;   // HANGUL SYLLABLE
   sortTable[0xAD4A]=33357;   // HANGUL SYLLABLE
   sortTable[0xAD4B]=33358;   // HANGUL SYLLABLE
   sortTable[0xAD4C]=33359;   // HANGUL SYLLABLE
   sortTable[0xAD4D]=33360;   // HANGUL SYLLABLE
   sortTable[0xAD4E]=33361;   // HANGUL SYLLABLE
   sortTable[0xAD4F]=33362;   // HANGUL SYLLABLE
   sortTable[0xAD50]=33363;   // HANGUL SYLLABLE
   sortTable[0xAD51]=33364;   // HANGUL SYLLABLE
   sortTable[0xAD52]=33365;   // HANGUL SYLLABLE
   sortTable[0xAD53]=33366;   // HANGUL SYLLABLE
   sortTable[0xAD54]=33367;   // HANGUL SYLLABLE
   sortTable[0xAD55]=33368;   // HANGUL SYLLABLE
   sortTable[0xAD56]=33369;   // HANGUL SYLLABLE
   sortTable[0xAD57]=33370;   // HANGUL SYLLABLE
   sortTable[0xAD58]=33371;   // HANGUL SYLLABLE
   sortTable[0xAD59]=33372;   // HANGUL SYLLABLE
   sortTable[0xAD5A]=33373;   // HANGUL SYLLABLE
   sortTable[0xAD5B]=33374;   // HANGUL SYLLABLE
   sortTable[0xAD5C]=33375;   // HANGUL SYLLABLE
   sortTable[0xAD5D]=33376;   // HANGUL SYLLABLE
   sortTable[0xAD5E]=33377;   // HANGUL SYLLABLE
   sortTable[0xAD5F]=33378;   // HANGUL SYLLABLE
   sortTable[0xAD60]=33379;   // HANGUL SYLLABLE
   sortTable[0xAD61]=33380;   // HANGUL SYLLABLE
   sortTable[0xAD62]=33381;   // HANGUL SYLLABLE
   sortTable[0xAD63]=33382;   // HANGUL SYLLABLE
   sortTable[0xAD64]=33383;   // HANGUL SYLLABLE
   sortTable[0xAD65]=33384;   // HANGUL SYLLABLE
   sortTable[0xAD66]=33385;   // HANGUL SYLLABLE
   sortTable[0xAD67]=33386;   // HANGUL SYLLABLE
   sortTable[0xAD68]=33387;   // HANGUL SYLLABLE
   sortTable[0xAD69]=33388;   // HANGUL SYLLABLE
   sortTable[0xAD6A]=33389;   // HANGUL SYLLABLE
   sortTable[0xAD6B]=33390;   // HANGUL SYLLABLE
   sortTable[0xAD6C]=33391;   // HANGUL SYLLABLE
   sortTable[0xAD6D]=33392;   // HANGUL SYLLABLE
   sortTable[0xAD6E]=33393;   // HANGUL SYLLABLE
   sortTable[0xAD6F]=33394;   // HANGUL SYLLABLE
   sortTable[0xAD70]=33395;   // HANGUL SYLLABLE
   sortTable[0xAD71]=33396;   // HANGUL SYLLABLE
   sortTable[0xAD72]=33397;   // HANGUL SYLLABLE
   sortTable[0xAD73]=33398;   // HANGUL SYLLABLE
   sortTable[0xAD74]=33399;   // HANGUL SYLLABLE
   sortTable[0xAD75]=33400;   // HANGUL SYLLABLE
   sortTable[0xAD76]=33401;   // HANGUL SYLLABLE
   sortTable[0xAD77]=33402;   // HANGUL SYLLABLE
   sortTable[0xAD78]=33403;   // HANGUL SYLLABLE
   sortTable[0xAD79]=33404;   // HANGUL SYLLABLE
   sortTable[0xAD7A]=33405;   // HANGUL SYLLABLE
   sortTable[0xAD7B]=33406;   // HANGUL SYLLABLE
   sortTable[0xAD7C]=33407;   // HANGUL SYLLABLE
   sortTable[0xAD7D]=33408;   // HANGUL SYLLABLE
   sortTable[0xAD7E]=33409;   // HANGUL SYLLABLE
   sortTable[0xAD7F]=33410;   // HANGUL SYLLABLE
   sortTable[0xAD80]=33411;   // HANGUL SYLLABLE
   sortTable[0xAD81]=33412;   // HANGUL SYLLABLE
   sortTable[0xAD82]=33413;   // HANGUL SYLLABLE
   sortTable[0xAD83]=33414;   // HANGUL SYLLABLE
   sortTable[0xAD84]=33415;   // HANGUL SYLLABLE
   sortTable[0xAD85]=33416;   // HANGUL SYLLABLE
   sortTable[0xAD86]=33417;   // HANGUL SYLLABLE
   sortTable[0xAD87]=33418;   // HANGUL SYLLABLE
   sortTable[0xAD88]=33419;   // HANGUL SYLLABLE
   sortTable[0xAD89]=33420;   // HANGUL SYLLABLE
   sortTable[0xAD8A]=33421;   // HANGUL SYLLABLE
   sortTable[0xAD8B]=33422;   // HANGUL SYLLABLE
   sortTable[0xAD8C]=33423;   // HANGUL SYLLABLE
   sortTable[0xAD8D]=33424;   // HANGUL SYLLABLE
   sortTable[0xAD8E]=33425;   // HANGUL SYLLABLE
   sortTable[0xAD8F]=33426;   // HANGUL SYLLABLE
   sortTable[0xAD90]=33427;   // HANGUL SYLLABLE
   sortTable[0xAD91]=33428;   // HANGUL SYLLABLE
   sortTable[0xAD92]=33429;   // HANGUL SYLLABLE
   sortTable[0xAD93]=33430;   // HANGUL SYLLABLE
   sortTable[0xAD94]=33431;   // HANGUL SYLLABLE
   sortTable[0xAD95]=33432;   // HANGUL SYLLABLE
   sortTable[0xAD96]=33433;   // HANGUL SYLLABLE
   sortTable[0xAD97]=33434;   // HANGUL SYLLABLE
   sortTable[0xAD98]=33435;   // HANGUL SYLLABLE
   sortTable[0xAD99]=33436;   // HANGUL SYLLABLE
   sortTable[0xAD9A]=33437;   // HANGUL SYLLABLE
   sortTable[0xAD9B]=33438;   // HANGUL SYLLABLE
   sortTable[0xAD9C]=33439;   // HANGUL SYLLABLE
   sortTable[0xAD9D]=33440;   // HANGUL SYLLABLE
   sortTable[0xAD9E]=33441;   // HANGUL SYLLABLE
   sortTable[0xAD9F]=33442;   // HANGUL SYLLABLE
   sortTable[0xADA0]=33443;   // HANGUL SYLLABLE
   sortTable[0xADA1]=33444;   // HANGUL SYLLABLE
   sortTable[0xADA2]=33445;   // HANGUL SYLLABLE
   sortTable[0xADA3]=33446;   // HANGUL SYLLABLE
   sortTable[0xADA4]=33447;   // HANGUL SYLLABLE
   sortTable[0xADA5]=33448;   // HANGUL SYLLABLE
   sortTable[0xADA6]=33449;   // HANGUL SYLLABLE
   sortTable[0xADA7]=33450;   // HANGUL SYLLABLE
   sortTable[0xADA8]=33451;   // HANGUL SYLLABLE
   sortTable[0xADA9]=33452;   // HANGUL SYLLABLE
   sortTable[0xADAA]=33453;   // HANGUL SYLLABLE
   sortTable[0xADAB]=33454;   // HANGUL SYLLABLE
   sortTable[0xADAC]=33455;   // HANGUL SYLLABLE
   sortTable[0xADAD]=33456;   // HANGUL SYLLABLE
   sortTable[0xADAE]=33457;   // HANGUL SYLLABLE
   sortTable[0xADAF]=33458;   // HANGUL SYLLABLE
   sortTable[0xADB0]=33459;   // HANGUL SYLLABLE
   sortTable[0xADB1]=33460;   // HANGUL SYLLABLE
   sortTable[0xADB2]=33461;   // HANGUL SYLLABLE
   sortTable[0xADB3]=33462;   // HANGUL SYLLABLE
   sortTable[0xADB4]=33463;   // HANGUL SYLLABLE
   sortTable[0xADB5]=33464;   // HANGUL SYLLABLE
   sortTable[0xADB6]=33465;   // HANGUL SYLLABLE
   sortTable[0xADB7]=33466;   // HANGUL SYLLABLE
   sortTable[0xADB8]=33467;   // HANGUL SYLLABLE
   sortTable[0xADB9]=33468;   // HANGUL SYLLABLE
   sortTable[0xADBA]=33469;   // HANGUL SYLLABLE
   sortTable[0xADBB]=33470;   // HANGUL SYLLABLE
   sortTable[0xADBC]=33471;   // HANGUL SYLLABLE
   sortTable[0xADBD]=33472;   // HANGUL SYLLABLE
   sortTable[0xADBE]=33473;   // HANGUL SYLLABLE
   sortTable[0xADBF]=33474;   // HANGUL SYLLABLE
   sortTable[0xADC0]=33475;   // HANGUL SYLLABLE
   sortTable[0xADC1]=33476;   // HANGUL SYLLABLE
   sortTable[0xADC2]=33477;   // HANGUL SYLLABLE
   sortTable[0xADC3]=33478;   // HANGUL SYLLABLE
   sortTable[0xADC4]=33479;   // HANGUL SYLLABLE
   sortTable[0xADC5]=33480;   // HANGUL SYLLABLE
   sortTable[0xADC6]=33481;   // HANGUL SYLLABLE
   sortTable[0xADC7]=33482;   // HANGUL SYLLABLE
   sortTable[0xADC8]=33483;   // HANGUL SYLLABLE
   sortTable[0xADC9]=33484;   // HANGUL SYLLABLE
   sortTable[0xADCA]=33485;   // HANGUL SYLLABLE
   sortTable[0xADCB]=33486;   // HANGUL SYLLABLE
   sortTable[0xADCC]=33487;   // HANGUL SYLLABLE
   sortTable[0xADCD]=33488;   // HANGUL SYLLABLE
   sortTable[0xADCE]=33489;   // HANGUL SYLLABLE
   sortTable[0xADCF]=33490;   // HANGUL SYLLABLE
   sortTable[0xADD0]=33491;   // HANGUL SYLLABLE
   sortTable[0xADD1]=33492;   // HANGUL SYLLABLE
   sortTable[0xADD2]=33493;   // HANGUL SYLLABLE
   sortTable[0xADD3]=33494;   // HANGUL SYLLABLE
   sortTable[0xADD4]=33495;   // HANGUL SYLLABLE
   sortTable[0xADD5]=33496;   // HANGUL SYLLABLE
   sortTable[0xADD6]=33497;   // HANGUL SYLLABLE
   sortTable[0xADD7]=33498;   // HANGUL SYLLABLE
   sortTable[0xADD8]=33499;   // HANGUL SYLLABLE
   sortTable[0xADD9]=33500;   // HANGUL SYLLABLE
   sortTable[0xADDA]=33501;   // HANGUL SYLLABLE
   sortTable[0xADDB]=33502;   // HANGUL SYLLABLE
   sortTable[0xADDC]=33503;   // HANGUL SYLLABLE
   sortTable[0xADDD]=33504;   // HANGUL SYLLABLE
   sortTable[0xADDE]=33505;   // HANGUL SYLLABLE
   sortTable[0xADDF]=33506;   // HANGUL SYLLABLE
   sortTable[0xADE0]=33507;   // HANGUL SYLLABLE
   sortTable[0xADE1]=33508;   // HANGUL SYLLABLE
   sortTable[0xADE2]=33509;   // HANGUL SYLLABLE
   sortTable[0xADE3]=33510;   // HANGUL SYLLABLE
   sortTable[0xADE4]=33511;   // HANGUL SYLLABLE
   sortTable[0xADE5]=33512;   // HANGUL SYLLABLE
   sortTable[0xADE6]=33513;   // HANGUL SYLLABLE
   sortTable[0xADE7]=33514;   // HANGUL SYLLABLE
   sortTable[0xADE8]=33515;   // HANGUL SYLLABLE
   sortTable[0xADE9]=33516;   // HANGUL SYLLABLE
   sortTable[0xADEA]=33517;   // HANGUL SYLLABLE
   sortTable[0xADEB]=33518;   // HANGUL SYLLABLE
   sortTable[0xADEC]=33519;   // HANGUL SYLLABLE
   sortTable[0xADED]=33520;   // HANGUL SYLLABLE
   sortTable[0xADEE]=33521;   // HANGUL SYLLABLE
   sortTable[0xADEF]=33522;   // HANGUL SYLLABLE
   sortTable[0xADF0]=33523;   // HANGUL SYLLABLE
   sortTable[0xADF1]=33524;   // HANGUL SYLLABLE
   sortTable[0xADF2]=33525;   // HANGUL SYLLABLE
   sortTable[0xADF3]=33526;   // HANGUL SYLLABLE
   sortTable[0xADF4]=33527;   // HANGUL SYLLABLE
   sortTable[0xADF5]=33528;   // HANGUL SYLLABLE
   sortTable[0xADF6]=33529;   // HANGUL SYLLABLE
   sortTable[0xADF7]=33530;   // HANGUL SYLLABLE
   sortTable[0xADF8]=33531;   // HANGUL SYLLABLE
   sortTable[0xADF9]=33532;   // HANGUL SYLLABLE
   sortTable[0xADFA]=33533;   // HANGUL SYLLABLE
   sortTable[0xADFB]=33534;   // HANGUL SYLLABLE
   sortTable[0xADFC]=33535;   // HANGUL SYLLABLE
   sortTable[0xADFD]=33536;   // HANGUL SYLLABLE
   sortTable[0xADFE]=33537;   // HANGUL SYLLABLE
   sortTable[0xADFF]=33538;   // HANGUL SYLLABLE
   sortTable[0xAE00]=33539;   // HANGUL SYLLABLE
   sortTable[0xAE01]=33540;   // HANGUL SYLLABLE
   sortTable[0xAE02]=33541;   // HANGUL SYLLABLE
   sortTable[0xAE03]=33542;   // HANGUL SYLLABLE
   sortTable[0xAE04]=33543;   // HANGUL SYLLABLE
   sortTable[0xAE05]=33544;   // HANGUL SYLLABLE
   sortTable[0xAE06]=33545;   // HANGUL SYLLABLE
   sortTable[0xAE07]=33546;   // HANGUL SYLLABLE
   sortTable[0xAE08]=33547;   // HANGUL SYLLABLE
   sortTable[0xAE09]=33548;   // HANGUL SYLLABLE
   sortTable[0xAE0A]=33549;   // HANGUL SYLLABLE
   sortTable[0xAE0B]=33550;   // HANGUL SYLLABLE
   sortTable[0xAE0C]=33551;   // HANGUL SYLLABLE
   sortTable[0xAE0D]=33552;   // HANGUL SYLLABLE
   sortTable[0xAE0E]=33553;   // HANGUL SYLLABLE
   sortTable[0xAE0F]=33554;   // HANGUL SYLLABLE
   sortTable[0xAE10]=33555;   // HANGUL SYLLABLE
   sortTable[0xAE11]=33556;   // HANGUL SYLLABLE
   sortTable[0xAE12]=33557;   // HANGUL SYLLABLE
   sortTable[0xAE13]=33558;   // HANGUL SYLLABLE
   sortTable[0xAE14]=33559;   // HANGUL SYLLABLE
   sortTable[0xAE15]=33560;   // HANGUL SYLLABLE
   sortTable[0xAE16]=33561;   // HANGUL SYLLABLE
   sortTable[0xAE17]=33562;   // HANGUL SYLLABLE
   sortTable[0xAE18]=33563;   // HANGUL SYLLABLE
   sortTable[0xAE19]=33564;   // HANGUL SYLLABLE
   sortTable[0xAE1A]=33565;   // HANGUL SYLLABLE
   sortTable[0xAE1B]=33566;   // HANGUL SYLLABLE
   sortTable[0xAE1C]=33567;   // HANGUL SYLLABLE
   sortTable[0xAE1D]=33568;   // HANGUL SYLLABLE
   sortTable[0xAE1E]=33569;   // HANGUL SYLLABLE
   sortTable[0xAE1F]=33570;   // HANGUL SYLLABLE
   sortTable[0xAE20]=33571;   // HANGUL SYLLABLE
   sortTable[0xAE21]=33572;   // HANGUL SYLLABLE
   sortTable[0xAE22]=33573;   // HANGUL SYLLABLE
   sortTable[0xAE23]=33574;   // HANGUL SYLLABLE
   sortTable[0xAE24]=33575;   // HANGUL SYLLABLE
   sortTable[0xAE25]=33576;   // HANGUL SYLLABLE
   sortTable[0xAE26]=33577;   // HANGUL SYLLABLE
   sortTable[0xAE27]=33578;   // HANGUL SYLLABLE
   sortTable[0xAE28]=33579;   // HANGUL SYLLABLE
   sortTable[0xAE29]=33580;   // HANGUL SYLLABLE
   sortTable[0xAE2A]=33581;   // HANGUL SYLLABLE
   sortTable[0xAE2B]=33582;   // HANGUL SYLLABLE
   sortTable[0xAE2C]=33583;   // HANGUL SYLLABLE
   sortTable[0xAE2D]=33584;   // HANGUL SYLLABLE
   sortTable[0xAE2E]=33585;   // HANGUL SYLLABLE
   sortTable[0xAE2F]=33586;   // HANGUL SYLLABLE
   sortTable[0xAE30]=33587;   // HANGUL SYLLABLE
   sortTable[0xAE31]=33588;   // HANGUL SYLLABLE
   sortTable[0xAE32]=33589;   // HANGUL SYLLABLE
   sortTable[0xAE33]=33590;   // HANGUL SYLLABLE
   sortTable[0xAE34]=33591;   // HANGUL SYLLABLE
   sortTable[0xAE35]=33592;   // HANGUL SYLLABLE
   sortTable[0xAE36]=33593;   // HANGUL SYLLABLE
   sortTable[0xAE37]=33594;   // HANGUL SYLLABLE
   sortTable[0xAE38]=33595;   // HANGUL SYLLABLE
   sortTable[0xAE39]=33596;   // HANGUL SYLLABLE
   sortTable[0xAE3A]=33597;   // HANGUL SYLLABLE
   sortTable[0xAE3B]=33598;   // HANGUL SYLLABLE
   sortTable[0xAE3C]=33599;   // HANGUL SYLLABLE
   sortTable[0xAE3D]=33600;   // HANGUL SYLLABLE
   sortTable[0xAE3E]=33601;   // HANGUL SYLLABLE
   sortTable[0xAE3F]=33602;   // HANGUL SYLLABLE
   sortTable[0xAE40]=33603;   // HANGUL SYLLABLE
   sortTable[0xAE41]=33604;   // HANGUL SYLLABLE
   sortTable[0xAE42]=33605;   // HANGUL SYLLABLE
   sortTable[0xAE43]=33606;   // HANGUL SYLLABLE
   sortTable[0xAE44]=33607;   // HANGUL SYLLABLE
   sortTable[0xAE45]=33608;   // HANGUL SYLLABLE
   sortTable[0xAE46]=33609;   // HANGUL SYLLABLE
   sortTable[0xAE47]=33610;   // HANGUL SYLLABLE
   sortTable[0xAE48]=33611;   // HANGUL SYLLABLE
   sortTable[0xAE49]=33612;   // HANGUL SYLLABLE
   sortTable[0xAE4A]=33613;   // HANGUL SYLLABLE
   sortTable[0xAE4B]=33614;   // HANGUL SYLLABLE
   sortTable[0x3132]=33615;   // HANGUL SYLLABLE
   sortTable[0x1101]=33616;   // HANGUL CHOSEONG SSANGKIYEOK
   sortTable[0x11A9]=33617;   // HANGUL JONGSEONG SSANGKIYEOK
   sortTable[0xAE4C]=33618;   // HANGUL SYLLABLE
   sortTable[0xAE4D]=33619;   // HANGUL SYLLABLE
   sortTable[0xAE4E]=33620;   // HANGUL SYLLABLE
   sortTable[0xAE4F]=33621;   // HANGUL SYLLABLE
   sortTable[0xAE50]=33622;   // HANGUL SYLLABLE
   sortTable[0xAE51]=33623;   // HANGUL SYLLABLE
   sortTable[0xAE52]=33624;   // HANGUL SYLLABLE
   sortTable[0xAE53]=33625;   // HANGUL SYLLABLE
   sortTable[0xAE54]=33626;   // HANGUL SYLLABLE
   sortTable[0xAE55]=33627;   // HANGUL SYLLABLE
   sortTable[0xAE56]=33628;   // HANGUL SYLLABLE
   sortTable[0xAE57]=33629;   // HANGUL SYLLABLE
   sortTable[0xAE58]=33630;   // HANGUL SYLLABLE
   sortTable[0xAE59]=33631;   // HANGUL SYLLABLE
   sortTable[0xAE5A]=33632;   // HANGUL SYLLABLE
   sortTable[0xAE5B]=33633;   // HANGUL SYLLABLE
   sortTable[0xAE5C]=33634;   // HANGUL SYLLABLE
   sortTable[0xAE5D]=33635;   // HANGUL SYLLABLE
   sortTable[0xAE5E]=33636;   // HANGUL SYLLABLE
   sortTable[0xAE5F]=33637;   // HANGUL SYLLABLE
   sortTable[0xAE60]=33638;   // HANGUL SYLLABLE
   sortTable[0xAE61]=33639;   // HANGUL SYLLABLE
   sortTable[0xAE62]=33640;   // HANGUL SYLLABLE
   sortTable[0xAE63]=33641;   // HANGUL SYLLABLE
   sortTable[0xAE64]=33642;   // HANGUL SYLLABLE
   sortTable[0xAE65]=33643;   // HANGUL SYLLABLE
   sortTable[0xAE66]=33644;   // HANGUL SYLLABLE
   sortTable[0xAE67]=33645;   // HANGUL SYLLABLE
   sortTable[0xAE68]=33646;   // HANGUL SYLLABLE
   sortTable[0xAE69]=33647;   // HANGUL SYLLABLE
   sortTable[0xAE6A]=33648;   // HANGUL SYLLABLE
   sortTable[0xAE6B]=33649;   // HANGUL SYLLABLE
   sortTable[0xAE6C]=33650;   // HANGUL SYLLABLE
   sortTable[0xAE6D]=33651;   // HANGUL SYLLABLE
   sortTable[0xAE6E]=33652;   // HANGUL SYLLABLE
   sortTable[0xAE6F]=33653;   // HANGUL SYLLABLE
   sortTable[0xAE70]=33654;   // HANGUL SYLLABLE
   sortTable[0xAE71]=33655;   // HANGUL SYLLABLE
   sortTable[0xAE72]=33656;   // HANGUL SYLLABLE
   sortTable[0xAE73]=33657;   // HANGUL SYLLABLE
   sortTable[0xAE74]=33658;   // HANGUL SYLLABLE
   sortTable[0xAE75]=33659;   // HANGUL SYLLABLE
   sortTable[0xAE76]=33660;   // HANGUL SYLLABLE
   sortTable[0xAE77]=33661;   // HANGUL SYLLABLE
   sortTable[0xAE78]=33662;   // HANGUL SYLLABLE
   sortTable[0xAE79]=33663;   // HANGUL SYLLABLE
   sortTable[0xAE7A]=33664;   // HANGUL SYLLABLE
   sortTable[0xAE7B]=33665;   // HANGUL SYLLABLE
   sortTable[0xAE7C]=33666;   // HANGUL SYLLABLE
   sortTable[0xAE7D]=33667;   // HANGUL SYLLABLE
   sortTable[0xAE7E]=33668;   // HANGUL SYLLABLE
   sortTable[0xAE7F]=33669;   // HANGUL SYLLABLE
   sortTable[0xAE80]=33670;   // HANGUL SYLLABLE
   sortTable[0xAE81]=33671;   // HANGUL SYLLABLE
   sortTable[0xAE82]=33672;   // HANGUL SYLLABLE
   sortTable[0xAE83]=33673;   // HANGUL SYLLABLE
   sortTable[0xAE84]=33674;   // HANGUL SYLLABLE
   sortTable[0xAE85]=33675;   // HANGUL SYLLABLE
   sortTable[0xAE86]=33676;   // HANGUL SYLLABLE
   sortTable[0xAE87]=33677;   // HANGUL SYLLABLE
   sortTable[0xAE88]=33678;   // HANGUL SYLLABLE
   sortTable[0xAE89]=33679;   // HANGUL SYLLABLE
   sortTable[0xAE8A]=33680;   // HANGUL SYLLABLE
   sortTable[0xAE8B]=33681;   // HANGUL SYLLABLE
   sortTable[0xAE8C]=33682;   // HANGUL SYLLABLE
   sortTable[0xAE8D]=33683;   // HANGUL SYLLABLE
   sortTable[0xAE8E]=33684;   // HANGUL SYLLABLE
   sortTable[0xAE8F]=33685;   // HANGUL SYLLABLE
   sortTable[0xAE90]=33686;   // HANGUL SYLLABLE
   sortTable[0xAE91]=33687;   // HANGUL SYLLABLE
   sortTable[0xAE92]=33688;   // HANGUL SYLLABLE
   sortTable[0xAE93]=33689;   // HANGUL SYLLABLE
   sortTable[0xAE94]=33690;   // HANGUL SYLLABLE
   sortTable[0xAE95]=33691;   // HANGUL SYLLABLE
   sortTable[0xAE96]=33692;   // HANGUL SYLLABLE
   sortTable[0xAE97]=33693;   // HANGUL SYLLABLE
   sortTable[0xAE98]=33694;   // HANGUL SYLLABLE
   sortTable[0xAE99]=33695;   // HANGUL SYLLABLE
   sortTable[0xAE9A]=33696;   // HANGUL SYLLABLE
   sortTable[0xAE9B]=33697;   // HANGUL SYLLABLE
   sortTable[0xAE9C]=33698;   // HANGUL SYLLABLE
   sortTable[0xAE9D]=33699;   // HANGUL SYLLABLE
   sortTable[0xAE9E]=33700;   // HANGUL SYLLABLE
   sortTable[0xAE9F]=33701;   // HANGUL SYLLABLE
   sortTable[0xAEA0]=33702;   // HANGUL SYLLABLE
   sortTable[0xAEA1]=33703;   // HANGUL SYLLABLE
   sortTable[0xAEA2]=33704;   // HANGUL SYLLABLE
   sortTable[0xAEA3]=33705;   // HANGUL SYLLABLE
   sortTable[0xAEA4]=33706;   // HANGUL SYLLABLE
   sortTable[0xAEA5]=33707;   // HANGUL SYLLABLE
   sortTable[0xAEA6]=33708;   // HANGUL SYLLABLE
   sortTable[0xAEA7]=33709;   // HANGUL SYLLABLE
   sortTable[0xAEA8]=33710;   // HANGUL SYLLABLE
   sortTable[0xAEA9]=33711;   // HANGUL SYLLABLE
   sortTable[0xAEAA]=33712;   // HANGUL SYLLABLE
   sortTable[0xAEAB]=33713;   // HANGUL SYLLABLE
   sortTable[0xAEAC]=33714;   // HANGUL SYLLABLE
   sortTable[0xAEAD]=33715;   // HANGUL SYLLABLE
   sortTable[0xAEAE]=33716;   // HANGUL SYLLABLE
   sortTable[0xAEAF]=33717;   // HANGUL SYLLABLE
   sortTable[0xAEB0]=33718;   // HANGUL SYLLABLE
   sortTable[0xAEB1]=33719;   // HANGUL SYLLABLE
   sortTable[0xAEB2]=33720;   // HANGUL SYLLABLE
   sortTable[0xAEB3]=33721;   // HANGUL SYLLABLE
   sortTable[0xAEB4]=33722;   // HANGUL SYLLABLE
   sortTable[0xAEB5]=33723;   // HANGUL SYLLABLE
   sortTable[0xAEB6]=33724;   // HANGUL SYLLABLE
   sortTable[0xAEB7]=33725;   // HANGUL SYLLABLE
   sortTable[0xAEB8]=33726;   // HANGUL SYLLABLE
   sortTable[0xAEB9]=33727;   // HANGUL SYLLABLE
   sortTable[0xAEBA]=33728;   // HANGUL SYLLABLE
   sortTable[0xAEBB]=33729;   // HANGUL SYLLABLE
   sortTable[0xAEBC]=33730;   // HANGUL SYLLABLE
   sortTable[0xAEBD]=33731;   // HANGUL SYLLABLE
   sortTable[0xAEBE]=33732;   // HANGUL SYLLABLE
   sortTable[0xAEBF]=33733;   // HANGUL SYLLABLE
   sortTable[0xAEC0]=33734;   // HANGUL SYLLABLE
   sortTable[0xAEC1]=33735;   // HANGUL SYLLABLE
   sortTable[0xAEC2]=33736;   // HANGUL SYLLABLE
   sortTable[0xAEC3]=33737;   // HANGUL SYLLABLE
   sortTable[0xAEC4]=33738;   // HANGUL SYLLABLE
   sortTable[0xAEC5]=33739;   // HANGUL SYLLABLE
   sortTable[0xAEC6]=33740;   // HANGUL SYLLABLE
   sortTable[0xAEC7]=33741;   // HANGUL SYLLABLE
   sortTable[0xAEC8]=33742;   // HANGUL SYLLABLE
   sortTable[0xAEC9]=33743;   // HANGUL SYLLABLE
   sortTable[0xAECA]=33744;   // HANGUL SYLLABLE
   sortTable[0xAECB]=33745;   // HANGUL SYLLABLE
   sortTable[0xAECC]=33746;   // HANGUL SYLLABLE
   sortTable[0xAECD]=33747;   // HANGUL SYLLABLE
   sortTable[0xAECE]=33748;   // HANGUL SYLLABLE
   sortTable[0xAECF]=33749;   // HANGUL SYLLABLE
   sortTable[0xAED0]=33750;   // HANGUL SYLLABLE
   sortTable[0xAED1]=33751;   // HANGUL SYLLABLE
   sortTable[0xAED2]=33752;   // HANGUL SYLLABLE
   sortTable[0xAED3]=33753;   // HANGUL SYLLABLE
   sortTable[0xAED4]=33754;   // HANGUL SYLLABLE
   sortTable[0xAED5]=33755;   // HANGUL SYLLABLE
   sortTable[0xAED6]=33756;   // HANGUL SYLLABLE
   sortTable[0xAED7]=33757;   // HANGUL SYLLABLE
   sortTable[0xAED8]=33758;   // HANGUL SYLLABLE
   sortTable[0xAED9]=33759;   // HANGUL SYLLABLE
   sortTable[0xAEDA]=33760;   // HANGUL SYLLABLE
   sortTable[0xAEDB]=33761;   // HANGUL SYLLABLE
   sortTable[0xAEDC]=33762;   // HANGUL SYLLABLE
   sortTable[0xAEDD]=33763;   // HANGUL SYLLABLE
   sortTable[0xAEDE]=33764;   // HANGUL SYLLABLE
   sortTable[0xAEDF]=33765;   // HANGUL SYLLABLE
   sortTable[0xAEE0]=33766;   // HANGUL SYLLABLE
   sortTable[0xAEE1]=33767;   // HANGUL SYLLABLE
   sortTable[0xAEE2]=33768;   // HANGUL SYLLABLE
   sortTable[0xAEE3]=33769;   // HANGUL SYLLABLE
   sortTable[0xAEE4]=33770;   // HANGUL SYLLABLE
   sortTable[0xAEE5]=33771;   // HANGUL SYLLABLE
   sortTable[0xAEE6]=33772;   // HANGUL SYLLABLE
   sortTable[0xAEE7]=33773;   // HANGUL SYLLABLE
   sortTable[0xAEE8]=33774;   // HANGUL SYLLABLE
   sortTable[0xAEE9]=33775;   // HANGUL SYLLABLE
   sortTable[0xAEEA]=33776;   // HANGUL SYLLABLE
   sortTable[0xAEEB]=33777;   // HANGUL SYLLABLE
   sortTable[0xAEEC]=33778;   // HANGUL SYLLABLE
   sortTable[0xAEED]=33779;   // HANGUL SYLLABLE
   sortTable[0xAEEE]=33780;   // HANGUL SYLLABLE
   sortTable[0xAEEF]=33781;   // HANGUL SYLLABLE
   sortTable[0xAEF0]=33782;   // HANGUL SYLLABLE
   sortTable[0xAEF1]=33783;   // HANGUL SYLLABLE
   sortTable[0xAEF2]=33784;   // HANGUL SYLLABLE
   sortTable[0xAEF3]=33785;   // HANGUL SYLLABLE
   sortTable[0xAEF4]=33786;   // HANGUL SYLLABLE
   sortTable[0xAEF5]=33787;   // HANGUL SYLLABLE
   sortTable[0xAEF6]=33788;   // HANGUL SYLLABLE
   sortTable[0xAEF7]=33789;   // HANGUL SYLLABLE
   sortTable[0xAEF8]=33790;   // HANGUL SYLLABLE
   sortTable[0xAEF9]=33791;   // HANGUL SYLLABLE
   sortTable[0xAEFA]=33792;   // HANGUL SYLLABLE
   sortTable[0xAEFB]=33793;   // HANGUL SYLLABLE
   sortTable[0xAEFC]=33794;   // HANGUL SYLLABLE
   sortTable[0xAEFD]=33795;   // HANGUL SYLLABLE
   sortTable[0xAEFE]=33796;   // HANGUL SYLLABLE
   sortTable[0xAEFF]=33797;   // HANGUL SYLLABLE
   sortTable[0xAF00]=33798;   // HANGUL SYLLABLE
   sortTable[0xAF01]=33799;   // HANGUL SYLLABLE
   sortTable[0xAF02]=33800;   // HANGUL SYLLABLE
   sortTable[0xAF03]=33801;   // HANGUL SYLLABLE
   sortTable[0xAF04]=33802;   // HANGUL SYLLABLE
   sortTable[0xAF05]=33803;   // HANGUL SYLLABLE
   sortTable[0xAF06]=33804;   // HANGUL SYLLABLE
   sortTable[0xAF07]=33805;   // HANGUL SYLLABLE
   sortTable[0xAF08]=33806;   // HANGUL SYLLABLE
   sortTable[0xAF09]=33807;   // HANGUL SYLLABLE
   sortTable[0xAF0A]=33808;   // HANGUL SYLLABLE
   sortTable[0xAF0B]=33809;   // HANGUL SYLLABLE
   sortTable[0xAF0C]=33810;   // HANGUL SYLLABLE
   sortTable[0xAF0D]=33811;   // HANGUL SYLLABLE
   sortTable[0xAF0E]=33812;   // HANGUL SYLLABLE
   sortTable[0xAF0F]=33813;   // HANGUL SYLLABLE
   sortTable[0xAF10]=33814;   // HANGUL SYLLABLE
   sortTable[0xAF11]=33815;   // HANGUL SYLLABLE
   sortTable[0xAF12]=33816;   // HANGUL SYLLABLE
   sortTable[0xAF13]=33817;   // HANGUL SYLLABLE
   sortTable[0xAF14]=33818;   // HANGUL SYLLABLE
   sortTable[0xAF15]=33819;   // HANGUL SYLLABLE
   sortTable[0xAF16]=33820;   // HANGUL SYLLABLE
   sortTable[0xAF17]=33821;   // HANGUL SYLLABLE
   sortTable[0xAF18]=33822;   // HANGUL SYLLABLE
   sortTable[0xAF19]=33823;   // HANGUL SYLLABLE
   sortTable[0xAF1A]=33824;   // HANGUL SYLLABLE
   sortTable[0xAF1B]=33825;   // HANGUL SYLLABLE
   sortTable[0xAF1C]=33826;   // HANGUL SYLLABLE
   sortTable[0xAF1D]=33827;   // HANGUL SYLLABLE
   sortTable[0xAF1E]=33828;   // HANGUL SYLLABLE
   sortTable[0xAF1F]=33829;   // HANGUL SYLLABLE
   sortTable[0xAF20]=33830;   // HANGUL SYLLABLE
   sortTable[0xAF21]=33831;   // HANGUL SYLLABLE
   sortTable[0xAF22]=33832;   // HANGUL SYLLABLE
   sortTable[0xAF23]=33833;   // HANGUL SYLLABLE
   sortTable[0xAF24]=33834;   // HANGUL SYLLABLE
   sortTable[0xAF25]=33835;   // HANGUL SYLLABLE
   sortTable[0xAF26]=33836;   // HANGUL SYLLABLE
   sortTable[0xAF27]=33837;   // HANGUL SYLLABLE
   sortTable[0xAF28]=33838;   // HANGUL SYLLABLE
   sortTable[0xAF29]=33839;   // HANGUL SYLLABLE
   sortTable[0xAF2A]=33840;   // HANGUL SYLLABLE
   sortTable[0xAF2B]=33841;   // HANGUL SYLLABLE
   sortTable[0xAF2C]=33842;   // HANGUL SYLLABLE
   sortTable[0xAF2D]=33843;   // HANGUL SYLLABLE
   sortTable[0xAF2E]=33844;   // HANGUL SYLLABLE
   sortTable[0xAF2F]=33845;   // HANGUL SYLLABLE
   sortTable[0xAF30]=33846;   // HANGUL SYLLABLE
   sortTable[0xAF31]=33847;   // HANGUL SYLLABLE
   sortTable[0xAF32]=33848;   // HANGUL SYLLABLE
   sortTable[0xAF33]=33849;   // HANGUL SYLLABLE
   sortTable[0xAF34]=33850;   // HANGUL SYLLABLE
   sortTable[0xAF35]=33851;   // HANGUL SYLLABLE
   sortTable[0xAF36]=33852;   // HANGUL SYLLABLE
   sortTable[0xAF37]=33853;   // HANGUL SYLLABLE
   sortTable[0xAF38]=33854;   // HANGUL SYLLABLE
   sortTable[0xAF39]=33855;   // HANGUL SYLLABLE
   sortTable[0xAF3A]=33856;   // HANGUL SYLLABLE
   sortTable[0xAF3B]=33857;   // HANGUL SYLLABLE
   sortTable[0xAF3C]=33858;   // HANGUL SYLLABLE
   sortTable[0xAF3D]=33859;   // HANGUL SYLLABLE
   sortTable[0xAF3E]=33860;   // HANGUL SYLLABLE
   sortTable[0xAF3F]=33861;   // HANGUL SYLLABLE
   sortTable[0xAF40]=33862;   // HANGUL SYLLABLE
   sortTable[0xAF41]=33863;   // HANGUL SYLLABLE
   sortTable[0xAF42]=33864;   // HANGUL SYLLABLE
   sortTable[0xAF43]=33865;   // HANGUL SYLLABLE
   sortTable[0xAF44]=33866;   // HANGUL SYLLABLE
   sortTable[0xAF45]=33867;   // HANGUL SYLLABLE
   sortTable[0xAF46]=33868;   // HANGUL SYLLABLE
   sortTable[0xAF47]=33869;   // HANGUL SYLLABLE
   sortTable[0xAF48]=33870;   // HANGUL SYLLABLE
   sortTable[0xAF49]=33871;   // HANGUL SYLLABLE
   sortTable[0xAF4A]=33872;   // HANGUL SYLLABLE
   sortTable[0xAF4B]=33873;   // HANGUL SYLLABLE
   sortTable[0xAF4C]=33874;   // HANGUL SYLLABLE
   sortTable[0xAF4D]=33875;   // HANGUL SYLLABLE
   sortTable[0xAF4E]=33876;   // HANGUL SYLLABLE
   sortTable[0xAF4F]=33877;   // HANGUL SYLLABLE
   sortTable[0xAF50]=33878;   // HANGUL SYLLABLE
   sortTable[0xAF51]=33879;   // HANGUL SYLLABLE
   sortTable[0xAF52]=33880;   // HANGUL SYLLABLE
   sortTable[0xAF53]=33881;   // HANGUL SYLLABLE
   sortTable[0xAF54]=33882;   // HANGUL SYLLABLE
   sortTable[0xAF55]=33883;   // HANGUL SYLLABLE
   sortTable[0xAF56]=33884;   // HANGUL SYLLABLE
   sortTable[0xAF57]=33885;   // HANGUL SYLLABLE
   sortTable[0xAF58]=33886;   // HANGUL SYLLABLE
   sortTable[0xAF59]=33887;   // HANGUL SYLLABLE
   sortTable[0xAF5A]=33888;   // HANGUL SYLLABLE
   sortTable[0xAF5B]=33889;   // HANGUL SYLLABLE
   sortTable[0xAF5C]=33890;   // HANGUL SYLLABLE
   sortTable[0xAF5D]=33891;   // HANGUL SYLLABLE
   sortTable[0xAF5E]=33892;   // HANGUL SYLLABLE
   sortTable[0xAF5F]=33893;   // HANGUL SYLLABLE
   sortTable[0xAF60]=33894;   // HANGUL SYLLABLE
   sortTable[0xAF61]=33895;   // HANGUL SYLLABLE
   sortTable[0xAF62]=33896;   // HANGUL SYLLABLE
   sortTable[0xAF63]=33897;   // HANGUL SYLLABLE
   sortTable[0xAF64]=33898;   // HANGUL SYLLABLE
   sortTable[0xAF65]=33899;   // HANGUL SYLLABLE
   sortTable[0xAF66]=33900;   // HANGUL SYLLABLE
   sortTable[0xAF67]=33901;   // HANGUL SYLLABLE
   sortTable[0xAF68]=33902;   // HANGUL SYLLABLE
   sortTable[0xAF69]=33903;   // HANGUL SYLLABLE
   sortTable[0xAF6A]=33904;   // HANGUL SYLLABLE
   sortTable[0xAF6B]=33905;   // HANGUL SYLLABLE
   sortTable[0xAF6C]=33906;   // HANGUL SYLLABLE
   sortTable[0xAF6D]=33907;   // HANGUL SYLLABLE
   sortTable[0xAF6E]=33908;   // HANGUL SYLLABLE
   sortTable[0xAF6F]=33909;   // HANGUL SYLLABLE
   sortTable[0xAF70]=33910;   // HANGUL SYLLABLE
   sortTable[0xAF71]=33911;   // HANGUL SYLLABLE
   sortTable[0xAF72]=33912;   // HANGUL SYLLABLE
   sortTable[0xAF73]=33913;   // HANGUL SYLLABLE
   sortTable[0xAF74]=33914;   // HANGUL SYLLABLE
   sortTable[0xAF75]=33915;   // HANGUL SYLLABLE
   sortTable[0xAF76]=33916;   // HANGUL SYLLABLE
   sortTable[0xAF77]=33917;   // HANGUL SYLLABLE
   sortTable[0xAF78]=33918;   // HANGUL SYLLABLE
   sortTable[0xAF79]=33919;   // HANGUL SYLLABLE
   sortTable[0xAF7A]=33920;   // HANGUL SYLLABLE
   sortTable[0xAF7B]=33921;   // HANGUL SYLLABLE
   sortTable[0xAF7C]=33922;   // HANGUL SYLLABLE
   sortTable[0xAF7D]=33923;   // HANGUL SYLLABLE
   sortTable[0xAF7E]=33924;   // HANGUL SYLLABLE
   sortTable[0xAF7F]=33925;   // HANGUL SYLLABLE
   sortTable[0xAF80]=33926;   // HANGUL SYLLABLE
   sortTable[0xAF81]=33927;   // HANGUL SYLLABLE
   sortTable[0xAF82]=33928;   // HANGUL SYLLABLE
   sortTable[0xAF83]=33929;   // HANGUL SYLLABLE
   sortTable[0xAF84]=33930;   // HANGUL SYLLABLE
   sortTable[0xAF85]=33931;   // HANGUL SYLLABLE
   sortTable[0xAF86]=33932;   // HANGUL SYLLABLE
   sortTable[0xAF87]=33933;   // HANGUL SYLLABLE
   sortTable[0xAF88]=33934;   // HANGUL SYLLABLE
   sortTable[0xAF89]=33935;   // HANGUL SYLLABLE
   sortTable[0xAF8A]=33936;   // HANGUL SYLLABLE
   sortTable[0xAF8B]=33937;   // HANGUL SYLLABLE
   sortTable[0xAF8C]=33938;   // HANGUL SYLLABLE
   sortTable[0xAF8D]=33939;   // HANGUL SYLLABLE
   sortTable[0xAF8E]=33940;   // HANGUL SYLLABLE
   sortTable[0xAF8F]=33941;   // HANGUL SYLLABLE
   sortTable[0xAF90]=33942;   // HANGUL SYLLABLE
   sortTable[0xAF91]=33943;   // HANGUL SYLLABLE
   sortTable[0xAF92]=33944;   // HANGUL SYLLABLE
   sortTable[0xAF93]=33945;   // HANGUL SYLLABLE
   sortTable[0xAF94]=33946;   // HANGUL SYLLABLE
   sortTable[0xAF95]=33947;   // HANGUL SYLLABLE
   sortTable[0xAF96]=33948;   // HANGUL SYLLABLE
   sortTable[0xAF97]=33949;   // HANGUL SYLLABLE
   sortTable[0xAF98]=33950;   // HANGUL SYLLABLE
   sortTable[0xAF99]=33951;   // HANGUL SYLLABLE
   sortTable[0xAF9A]=33952;   // HANGUL SYLLABLE
   sortTable[0xAF9B]=33953;   // HANGUL SYLLABLE
   sortTable[0xAF9C]=33954;   // HANGUL SYLLABLE
   sortTable[0xAF9D]=33955;   // HANGUL SYLLABLE
   sortTable[0xAF9E]=33956;   // HANGUL SYLLABLE
   sortTable[0xAF9F]=33957;   // HANGUL SYLLABLE
   sortTable[0xAFA0]=33958;   // HANGUL SYLLABLE
   sortTable[0xAFA1]=33959;   // HANGUL SYLLABLE
   sortTable[0xAFA2]=33960;   // HANGUL SYLLABLE
   sortTable[0xAFA3]=33961;   // HANGUL SYLLABLE
   sortTable[0xAFA4]=33962;   // HANGUL SYLLABLE
   sortTable[0xAFA5]=33963;   // HANGUL SYLLABLE
   sortTable[0xAFA6]=33964;   // HANGUL SYLLABLE
   sortTable[0xAFA7]=33965;   // HANGUL SYLLABLE
   sortTable[0xAFA8]=33966;   // HANGUL SYLLABLE
   sortTable[0xAFA9]=33967;   // HANGUL SYLLABLE
   sortTable[0xAFAA]=33968;   // HANGUL SYLLABLE
   sortTable[0xAFAB]=33969;   // HANGUL SYLLABLE
   sortTable[0xAFAC]=33970;   // HANGUL SYLLABLE
   sortTable[0xAFAD]=33971;   // HANGUL SYLLABLE
   sortTable[0xAFAE]=33972;   // HANGUL SYLLABLE
   sortTable[0xAFAF]=33973;   // HANGUL SYLLABLE
   sortTable[0xAFB0]=33974;   // HANGUL SYLLABLE
   sortTable[0xAFB1]=33975;   // HANGUL SYLLABLE
   sortTable[0xAFB2]=33976;   // HANGUL SYLLABLE
   sortTable[0xAFB3]=33977;   // HANGUL SYLLABLE
   sortTable[0xAFB4]=33978;   // HANGUL SYLLABLE
   sortTable[0xAFB5]=33979;   // HANGUL SYLLABLE
   sortTable[0xAFB6]=33980;   // HANGUL SYLLABLE
   sortTable[0xAFB7]=33981;   // HANGUL SYLLABLE
   sortTable[0xAFB8]=33982;   // HANGUL SYLLABLE
   sortTable[0xAFB9]=33983;   // HANGUL SYLLABLE
   sortTable[0xAFBA]=33984;   // HANGUL SYLLABLE
   sortTable[0xAFBB]=33985;   // HANGUL SYLLABLE
   sortTable[0xAFBC]=33986;   // HANGUL SYLLABLE
   sortTable[0xAFBD]=33987;   // HANGUL SYLLABLE
   sortTable[0xAFBE]=33988;   // HANGUL SYLLABLE
   sortTable[0xAFBF]=33989;   // HANGUL SYLLABLE
   sortTable[0xAFC0]=33990;   // HANGUL SYLLABLE
   sortTable[0xAFC1]=33991;   // HANGUL SYLLABLE
   sortTable[0xAFC2]=33992;   // HANGUL SYLLABLE
   sortTable[0xAFC3]=33993;   // HANGUL SYLLABLE
   sortTable[0xAFC4]=33994;   // HANGUL SYLLABLE
   sortTable[0xAFC5]=33995;   // HANGUL SYLLABLE
   sortTable[0xAFC6]=33996;   // HANGUL SYLLABLE
   sortTable[0xAFC7]=33997;   // HANGUL SYLLABLE
   sortTable[0xAFC8]=33998;   // HANGUL SYLLABLE
   sortTable[0xAFC9]=33999;   // HANGUL SYLLABLE
   sortTable[0xAFCA]=34000;   // HANGUL SYLLABLE
   sortTable[0xAFCB]=34001;   // HANGUL SYLLABLE
   sortTable[0xAFCC]=34002;   // HANGUL SYLLABLE
   sortTable[0xAFCD]=34003;   // HANGUL SYLLABLE
   sortTable[0xAFCE]=34004;   // HANGUL SYLLABLE
   sortTable[0xAFCF]=34005;   // HANGUL SYLLABLE
   sortTable[0xAFD0]=34006;   // HANGUL SYLLABLE
   sortTable[0xAFD1]=34007;   // HANGUL SYLLABLE
   sortTable[0xAFD2]=34008;   // HANGUL SYLLABLE
   sortTable[0xAFD3]=34009;   // HANGUL SYLLABLE
   sortTable[0xAFD4]=34010;   // HANGUL SYLLABLE
   sortTable[0xAFD5]=34011;   // HANGUL SYLLABLE
   sortTable[0xAFD6]=34012;   // HANGUL SYLLABLE
   sortTable[0xAFD7]=34013;   // HANGUL SYLLABLE
   sortTable[0xAFD8]=34014;   // HANGUL SYLLABLE
   sortTable[0xAFD9]=34015;   // HANGUL SYLLABLE
   sortTable[0xAFDA]=34016;   // HANGUL SYLLABLE
   sortTable[0xAFDB]=34017;   // HANGUL SYLLABLE
   sortTable[0xAFDC]=34018;   // HANGUL SYLLABLE
   sortTable[0xAFDD]=34019;   // HANGUL SYLLABLE
   sortTable[0xAFDE]=34020;   // HANGUL SYLLABLE
   sortTable[0xAFDF]=34021;   // HANGUL SYLLABLE
   sortTable[0xAFE0]=34022;   // HANGUL SYLLABLE
   sortTable[0xAFE1]=34023;   // HANGUL SYLLABLE
   sortTable[0xAFE2]=34024;   // HANGUL SYLLABLE
   sortTable[0xAFE3]=34025;   // HANGUL SYLLABLE
   sortTable[0xAFE4]=34026;   // HANGUL SYLLABLE
   sortTable[0xAFE5]=34027;   // HANGUL SYLLABLE
   sortTable[0xAFE6]=34028;   // HANGUL SYLLABLE
   sortTable[0xAFE7]=34029;   // HANGUL SYLLABLE
   sortTable[0xAFE8]=34030;   // HANGUL SYLLABLE
   sortTable[0xAFE9]=34031;   // HANGUL SYLLABLE
   sortTable[0xAFEA]=34032;   // HANGUL SYLLABLE
   sortTable[0xAFEB]=34033;   // HANGUL SYLLABLE
   sortTable[0xAFEC]=34034;   // HANGUL SYLLABLE
   sortTable[0xAFED]=34035;   // HANGUL SYLLABLE
   sortTable[0xAFEE]=34036;   // HANGUL SYLLABLE
   sortTable[0xAFEF]=34037;   // HANGUL SYLLABLE
   sortTable[0xAFF0]=34038;   // HANGUL SYLLABLE
   sortTable[0xAFF1]=34039;   // HANGUL SYLLABLE
   sortTable[0xAFF2]=34040;   // HANGUL SYLLABLE
   sortTable[0xAFF3]=34041;   // HANGUL SYLLABLE
   sortTable[0xAFF4]=34042;   // HANGUL SYLLABLE
   sortTable[0xAFF5]=34043;   // HANGUL SYLLABLE
   sortTable[0xAFF6]=34044;   // HANGUL SYLLABLE
   sortTable[0xAFF7]=34045;   // HANGUL SYLLABLE
   sortTable[0xAFF8]=34046;   // HANGUL SYLLABLE
   sortTable[0xAFF9]=34047;   // HANGUL SYLLABLE
   sortTable[0xAFFA]=34048;   // HANGUL SYLLABLE
   sortTable[0xAFFB]=34049;   // HANGUL SYLLABLE
   sortTable[0xAFFC]=34050;   // HANGUL SYLLABLE
   sortTable[0xAFFD]=34051;   // HANGUL SYLLABLE
   sortTable[0xAFFE]=34052;   // HANGUL SYLLABLE
   sortTable[0xAFFF]=34053;   // HANGUL SYLLABLE
   sortTable[0xB000]=34054;   // HANGUL SYLLABLE
   sortTable[0xB001]=34055;   // HANGUL SYLLABLE
   sortTable[0xB002]=34056;   // HANGUL SYLLABLE
   sortTable[0xB003]=34057;   // HANGUL SYLLABLE
   sortTable[0xB004]=34058;   // HANGUL SYLLABLE
   sortTable[0xB005]=34059;   // HANGUL SYLLABLE
   sortTable[0xB006]=34060;   // HANGUL SYLLABLE
   sortTable[0xB007]=34061;   // HANGUL SYLLABLE
   sortTable[0xB008]=34062;   // HANGUL SYLLABLE
   sortTable[0xB009]=34063;   // HANGUL SYLLABLE
   sortTable[0xB00A]=34064;   // HANGUL SYLLABLE
   sortTable[0xB00B]=34065;   // HANGUL SYLLABLE
   sortTable[0xB00C]=34066;   // HANGUL SYLLABLE
   sortTable[0xB00D]=34067;   // HANGUL SYLLABLE
   sortTable[0xB00E]=34068;   // HANGUL SYLLABLE
   sortTable[0xB00F]=34069;   // HANGUL SYLLABLE
   sortTable[0xB010]=34070;   // HANGUL SYLLABLE
   sortTable[0xB011]=34071;   // HANGUL SYLLABLE
   sortTable[0xB012]=34072;   // HANGUL SYLLABLE
   sortTable[0xB013]=34073;   // HANGUL SYLLABLE
   sortTable[0xB014]=34074;   // HANGUL SYLLABLE
   sortTable[0xB015]=34075;   // HANGUL SYLLABLE
   sortTable[0xB016]=34076;   // HANGUL SYLLABLE
   sortTable[0xB017]=34077;   // HANGUL SYLLABLE
   sortTable[0xB018]=34078;   // HANGUL SYLLABLE
   sortTable[0xB019]=34079;   // HANGUL SYLLABLE
   sortTable[0xB01A]=34080;   // HANGUL SYLLABLE
   sortTable[0xB01B]=34081;   // HANGUL SYLLABLE
   sortTable[0xB01C]=34082;   // HANGUL SYLLABLE
   sortTable[0xB01D]=34083;   // HANGUL SYLLABLE
   sortTable[0xB01E]=34084;   // HANGUL SYLLABLE
   sortTable[0xB01F]=34085;   // HANGUL SYLLABLE
   sortTable[0xB020]=34086;   // HANGUL SYLLABLE
   sortTable[0xB021]=34087;   // HANGUL SYLLABLE
   sortTable[0xB022]=34088;   // HANGUL SYLLABLE
   sortTable[0xB023]=34089;   // HANGUL SYLLABLE
   sortTable[0xB024]=34090;   // HANGUL SYLLABLE
   sortTable[0xB025]=34091;   // HANGUL SYLLABLE
   sortTable[0xB026]=34092;   // HANGUL SYLLABLE
   sortTable[0xB027]=34093;   // HANGUL SYLLABLE
   sortTable[0xB028]=34094;   // HANGUL SYLLABLE
   sortTable[0xB029]=34095;   // HANGUL SYLLABLE
   sortTable[0xB02A]=34096;   // HANGUL SYLLABLE
   sortTable[0xB02B]=34097;   // HANGUL SYLLABLE
   sortTable[0xB02C]=34098;   // HANGUL SYLLABLE
   sortTable[0xB02D]=34099;   // HANGUL SYLLABLE
   sortTable[0xB02E]=34100;   // HANGUL SYLLABLE
   sortTable[0xB02F]=34101;   // HANGUL SYLLABLE
   sortTable[0xB030]=34102;   // HANGUL SYLLABLE
   sortTable[0xB031]=34103;   // HANGUL SYLLABLE
   sortTable[0xB032]=34104;   // HANGUL SYLLABLE
   sortTable[0xB033]=34105;   // HANGUL SYLLABLE
   sortTable[0xB034]=34106;   // HANGUL SYLLABLE
   sortTable[0xB035]=34107;   // HANGUL SYLLABLE
   sortTable[0xB036]=34108;   // HANGUL SYLLABLE
   sortTable[0xB037]=34109;   // HANGUL SYLLABLE
   sortTable[0xB038]=34110;   // HANGUL SYLLABLE
   sortTable[0xB039]=34111;   // HANGUL SYLLABLE
   sortTable[0xB03A]=34112;   // HANGUL SYLLABLE
   sortTable[0xB03B]=34113;   // HANGUL SYLLABLE
   sortTable[0xB03C]=34114;   // HANGUL SYLLABLE
   sortTable[0xB03D]=34115;   // HANGUL SYLLABLE
   sortTable[0xB03E]=34116;   // HANGUL SYLLABLE
   sortTable[0xB03F]=34117;   // HANGUL SYLLABLE
   sortTable[0xB040]=34118;   // HANGUL SYLLABLE
   sortTable[0xB041]=34119;   // HANGUL SYLLABLE
   sortTable[0xB042]=34120;   // HANGUL SYLLABLE
   sortTable[0xB043]=34121;   // HANGUL SYLLABLE
   sortTable[0xB044]=34122;   // HANGUL SYLLABLE
   sortTable[0xB045]=34123;   // HANGUL SYLLABLE
   sortTable[0xB046]=34124;   // HANGUL SYLLABLE
   sortTable[0xB047]=34125;   // HANGUL SYLLABLE
   sortTable[0xB048]=34126;   // HANGUL SYLLABLE
   sortTable[0xB049]=34127;   // HANGUL SYLLABLE
   sortTable[0xB04A]=34128;   // HANGUL SYLLABLE
   sortTable[0xB04B]=34129;   // HANGUL SYLLABLE
   sortTable[0xB04C]=34130;   // HANGUL SYLLABLE
   sortTable[0xB04D]=34131;   // HANGUL SYLLABLE
   sortTable[0xB04E]=34132;   // HANGUL SYLLABLE
   sortTable[0xB04F]=34133;   // HANGUL SYLLABLE
   sortTable[0xB050]=34134;   // HANGUL SYLLABLE
   sortTable[0xB051]=34135;   // HANGUL SYLLABLE
   sortTable[0xB052]=34136;   // HANGUL SYLLABLE
   sortTable[0xB053]=34137;   // HANGUL SYLLABLE
   sortTable[0xB054]=34138;   // HANGUL SYLLABLE
   sortTable[0xB055]=34139;   // HANGUL SYLLABLE
   sortTable[0xB056]=34140;   // HANGUL SYLLABLE
   sortTable[0xB057]=34141;   // HANGUL SYLLABLE
   sortTable[0xB058]=34142;   // HANGUL SYLLABLE
   sortTable[0xB059]=34143;   // HANGUL SYLLABLE
   sortTable[0xB05A]=34144;   // HANGUL SYLLABLE
   sortTable[0xB05B]=34145;   // HANGUL SYLLABLE
   sortTable[0xB05C]=34146;   // HANGUL SYLLABLE
   sortTable[0xB05D]=34147;   // HANGUL SYLLABLE
   sortTable[0xB05E]=34148;   // HANGUL SYLLABLE
   sortTable[0xB05F]=34149;   // HANGUL SYLLABLE
   sortTable[0xB060]=34150;   // HANGUL SYLLABLE
   sortTable[0xB061]=34151;   // HANGUL SYLLABLE
   sortTable[0xB062]=34152;   // HANGUL SYLLABLE
   sortTable[0xB063]=34153;   // HANGUL SYLLABLE
   sortTable[0xB064]=34154;   // HANGUL SYLLABLE
   sortTable[0xB065]=34155;   // HANGUL SYLLABLE
   sortTable[0xB066]=34156;   // HANGUL SYLLABLE
   sortTable[0xB067]=34157;   // HANGUL SYLLABLE
   sortTable[0xB068]=34158;   // HANGUL SYLLABLE
   sortTable[0xB069]=34159;   // HANGUL SYLLABLE
   sortTable[0xB06A]=34160;   // HANGUL SYLLABLE
   sortTable[0xB06B]=34161;   // HANGUL SYLLABLE
   sortTable[0xB06C]=34162;   // HANGUL SYLLABLE
   sortTable[0xB06D]=34163;   // HANGUL SYLLABLE
   sortTable[0xB06E]=34164;   // HANGUL SYLLABLE
   sortTable[0xB06F]=34165;   // HANGUL SYLLABLE
   sortTable[0xB070]=34166;   // HANGUL SYLLABLE
   sortTable[0xB071]=34167;   // HANGUL SYLLABLE
   sortTable[0xB072]=34168;   // HANGUL SYLLABLE
   sortTable[0xB073]=34169;   // HANGUL SYLLABLE
   sortTable[0xB074]=34170;   // HANGUL SYLLABLE
   sortTable[0xB075]=34171;   // HANGUL SYLLABLE
   sortTable[0xB076]=34172;   // HANGUL SYLLABLE
   sortTable[0xB077]=34173;   // HANGUL SYLLABLE
   sortTable[0xB078]=34174;   // HANGUL SYLLABLE
   sortTable[0xB079]=34175;   // HANGUL SYLLABLE
   sortTable[0xB07A]=34176;   // HANGUL SYLLABLE
   sortTable[0xB07B]=34177;   // HANGUL SYLLABLE
   sortTable[0xB07C]=34178;   // HANGUL SYLLABLE
   sortTable[0xB07D]=34179;   // HANGUL SYLLABLE
   sortTable[0xB07E]=34180;   // HANGUL SYLLABLE
   sortTable[0xB07F]=34181;   // HANGUL SYLLABLE
   sortTable[0xB080]=34182;   // HANGUL SYLLABLE
   sortTable[0xB081]=34183;   // HANGUL SYLLABLE
   sortTable[0xB082]=34184;   // HANGUL SYLLABLE
   sortTable[0xB083]=34185;   // HANGUL SYLLABLE
   sortTable[0xB084]=34186;   // HANGUL SYLLABLE
   sortTable[0xB085]=34187;   // HANGUL SYLLABLE
   sortTable[0xB086]=34188;   // HANGUL SYLLABLE
   sortTable[0xB087]=34189;   // HANGUL SYLLABLE
   sortTable[0xB088]=34190;   // HANGUL SYLLABLE
   sortTable[0xB089]=34191;   // HANGUL SYLLABLE
   sortTable[0xB08A]=34192;   // HANGUL SYLLABLE
   sortTable[0xB08B]=34193;   // HANGUL SYLLABLE
   sortTable[0xB08C]=34194;   // HANGUL SYLLABLE
   sortTable[0xB08D]=34195;   // HANGUL SYLLABLE
   sortTable[0xB08E]=34196;   // HANGUL SYLLABLE
   sortTable[0xB08F]=34197;   // HANGUL SYLLABLE
   sortTable[0xB090]=34198;   // HANGUL SYLLABLE
   sortTable[0xB091]=34199;   // HANGUL SYLLABLE
   sortTable[0xB092]=34200;   // HANGUL SYLLABLE
   sortTable[0xB093]=34201;   // HANGUL SYLLABLE
   sortTable[0xB094]=34202;   // HANGUL SYLLABLE
   sortTable[0xB095]=34203;   // HANGUL SYLLABLE
   sortTable[0xB096]=34204;   // HANGUL SYLLABLE
   sortTable[0xB097]=34205;   // HANGUL SYLLABLE
   sortTable[0x3134]=34206;   // HANGUL SYLLABLE
   sortTable[0x1102]=34207;   // HANGUL CHOSEONG NIEUN
   sortTable[0x11AB]=34208;   // HANGUL JONGSEONG NIEUN
   sortTable[0xB098]=34209;   // HANGUL SYLLABLE
   sortTable[0xB099]=34210;   // HANGUL SYLLABLE
   sortTable[0xB09A]=34211;   // HANGUL SYLLABLE
   sortTable[0xB09B]=34212;   // HANGUL SYLLABLE
   sortTable[0xB09C]=34213;   // HANGUL SYLLABLE
   sortTable[0xB09D]=34214;   // HANGUL SYLLABLE
   sortTable[0xB09E]=34215;   // HANGUL SYLLABLE
   sortTable[0xB09F]=34216;   // HANGUL SYLLABLE
   sortTable[0xB0A0]=34217;   // HANGUL SYLLABLE
   sortTable[0xB0A1]=34218;   // HANGUL SYLLABLE
   sortTable[0xB0A2]=34219;   // HANGUL SYLLABLE
   sortTable[0xB0A3]=34220;   // HANGUL SYLLABLE
   sortTable[0xB0A4]=34221;   // HANGUL SYLLABLE
   sortTable[0xB0A5]=34222;   // HANGUL SYLLABLE
   sortTable[0xB0A6]=34223;   // HANGUL SYLLABLE
   sortTable[0xB0A7]=34224;   // HANGUL SYLLABLE
   sortTable[0xB0A8]=34225;   // HANGUL SYLLABLE
   sortTable[0xB0A9]=34226;   // HANGUL SYLLABLE
   sortTable[0xB0AA]=34227;   // HANGUL SYLLABLE
   sortTable[0xB0AB]=34228;   // HANGUL SYLLABLE
   sortTable[0xB0AC]=34229;   // HANGUL SYLLABLE
   sortTable[0xB0AD]=34230;   // HANGUL SYLLABLE
   sortTable[0xB0AE]=34231;   // HANGUL SYLLABLE
   sortTable[0xB0AF]=34232;   // HANGUL SYLLABLE
   sortTable[0xB0B0]=34233;   // HANGUL SYLLABLE
   sortTable[0xB0B1]=34234;   // HANGUL SYLLABLE
   sortTable[0xB0B2]=34235;   // HANGUL SYLLABLE
   sortTable[0xB0B3]=34236;   // HANGUL SYLLABLE
   sortTable[0xB0B4]=34237;   // HANGUL SYLLABLE
   sortTable[0xB0B5]=34238;   // HANGUL SYLLABLE
   sortTable[0xB0B6]=34239;   // HANGUL SYLLABLE
   sortTable[0xB0B7]=34240;   // HANGUL SYLLABLE
   sortTable[0xB0B8]=34241;   // HANGUL SYLLABLE
   sortTable[0xB0B9]=34242;   // HANGUL SYLLABLE
   sortTable[0xB0BA]=34243;   // HANGUL SYLLABLE
   sortTable[0xB0BB]=34244;   // HANGUL SYLLABLE
   sortTable[0xB0BC]=34245;   // HANGUL SYLLABLE
   sortTable[0xB0BD]=34246;   // HANGUL SYLLABLE
   sortTable[0xB0BE]=34247;   // HANGUL SYLLABLE
   sortTable[0xB0BF]=34248;   // HANGUL SYLLABLE
   sortTable[0xB0C0]=34249;   // HANGUL SYLLABLE
   sortTable[0xB0C1]=34250;   // HANGUL SYLLABLE
   sortTable[0xB0C2]=34251;   // HANGUL SYLLABLE
   sortTable[0xB0C3]=34252;   // HANGUL SYLLABLE
   sortTable[0xB0C4]=34253;   // HANGUL SYLLABLE
   sortTable[0xB0C5]=34254;   // HANGUL SYLLABLE
   sortTable[0xB0C6]=34255;   // HANGUL SYLLABLE
   sortTable[0xB0C7]=34256;   // HANGUL SYLLABLE
   sortTable[0xB0C8]=34257;   // HANGUL SYLLABLE
   sortTable[0xB0C9]=34258;   // HANGUL SYLLABLE
   sortTable[0xB0CA]=34259;   // HANGUL SYLLABLE
   sortTable[0xB0CB]=34260;   // HANGUL SYLLABLE
   sortTable[0xB0CC]=34261;   // HANGUL SYLLABLE
   sortTable[0xB0CD]=34262;   // HANGUL SYLLABLE
   sortTable[0xB0CE]=34263;   // HANGUL SYLLABLE
   sortTable[0xB0CF]=34264;   // HANGUL SYLLABLE
   sortTable[0xB0D0]=34265;   // HANGUL SYLLABLE
   sortTable[0xB0D1]=34266;   // HANGUL SYLLABLE
   sortTable[0xB0D2]=34267;   // HANGUL SYLLABLE
   sortTable[0xB0D3]=34268;   // HANGUL SYLLABLE
   sortTable[0xB0D4]=34269;   // HANGUL SYLLABLE
   sortTable[0xB0D5]=34270;   // HANGUL SYLLABLE
   sortTable[0xB0D6]=34271;   // HANGUL SYLLABLE
   sortTable[0xB0D7]=34272;   // HANGUL SYLLABLE
   sortTable[0xB0D8]=34273;   // HANGUL SYLLABLE
   sortTable[0xB0D9]=34274;   // HANGUL SYLLABLE
   sortTable[0xB0DA]=34275;   // HANGUL SYLLABLE
   sortTable[0xB0DB]=34276;   // HANGUL SYLLABLE
   sortTable[0xB0DC]=34277;   // HANGUL SYLLABLE
   sortTable[0xB0DD]=34278;   // HANGUL SYLLABLE
   sortTable[0xB0DE]=34279;   // HANGUL SYLLABLE
   sortTable[0xB0DF]=34280;   // HANGUL SYLLABLE
   sortTable[0xB0E0]=34281;   // HANGUL SYLLABLE
   sortTable[0xB0E1]=34282;   // HANGUL SYLLABLE
   sortTable[0xB0E2]=34283;   // HANGUL SYLLABLE
   sortTable[0xB0E3]=34284;   // HANGUL SYLLABLE
   sortTable[0xB0E4]=34285;   // HANGUL SYLLABLE
   sortTable[0xB0E5]=34286;   // HANGUL SYLLABLE
   sortTable[0xB0E6]=34287;   // HANGUL SYLLABLE
   sortTable[0xB0E7]=34288;   // HANGUL SYLLABLE
   sortTable[0xB0E8]=34289;   // HANGUL SYLLABLE
   sortTable[0xB0E9]=34290;   // HANGUL SYLLABLE
   sortTable[0xB0EA]=34291;   // HANGUL SYLLABLE
   sortTable[0xB0EB]=34292;   // HANGUL SYLLABLE
   sortTable[0xB0EC]=34293;   // HANGUL SYLLABLE
   sortTable[0xB0ED]=34294;   // HANGUL SYLLABLE
   sortTable[0xB0EE]=34295;   // HANGUL SYLLABLE
   sortTable[0xB0EF]=34296;   // HANGUL SYLLABLE
   sortTable[0xB0F0]=34297;   // HANGUL SYLLABLE
   sortTable[0xB0F1]=34298;   // HANGUL SYLLABLE
   sortTable[0xB0F2]=34299;   // HANGUL SYLLABLE
   sortTable[0xB0F3]=34300;   // HANGUL SYLLABLE
   sortTable[0xB0F4]=34301;   // HANGUL SYLLABLE
   sortTable[0xB0F5]=34302;   // HANGUL SYLLABLE
   sortTable[0xB0F6]=34303;   // HANGUL SYLLABLE
   sortTable[0xB0F7]=34304;   // HANGUL SYLLABLE
   sortTable[0xB0F8]=34305;   // HANGUL SYLLABLE
   sortTable[0xB0F9]=34306;   // HANGUL SYLLABLE
   sortTable[0xB0FA]=34307;   // HANGUL SYLLABLE
   sortTable[0xB0FB]=34308;   // HANGUL SYLLABLE
   sortTable[0xB0FC]=34309;   // HANGUL SYLLABLE
   sortTable[0xB0FD]=34310;   // HANGUL SYLLABLE
   sortTable[0xB0FE]=34311;   // HANGUL SYLLABLE
   sortTable[0xB0FF]=34312;   // HANGUL SYLLABLE
   sortTable[0xB100]=34313;   // HANGUL SYLLABLE
   sortTable[0xB101]=34314;   // HANGUL SYLLABLE
   sortTable[0xB102]=34315;   // HANGUL SYLLABLE
   sortTable[0xB103]=34316;   // HANGUL SYLLABLE
   sortTable[0xB104]=34317;   // HANGUL SYLLABLE
   sortTable[0xB105]=34318;   // HANGUL SYLLABLE
   sortTable[0xB106]=34319;   // HANGUL SYLLABLE
   sortTable[0xB107]=34320;   // HANGUL SYLLABLE
   sortTable[0xB108]=34321;   // HANGUL SYLLABLE
   sortTable[0xB109]=34322;   // HANGUL SYLLABLE
   sortTable[0xB10A]=34323;   // HANGUL SYLLABLE
   sortTable[0xB10B]=34324;   // HANGUL SYLLABLE
   sortTable[0xB10C]=34325;   // HANGUL SYLLABLE
   sortTable[0xB10D]=34326;   // HANGUL SYLLABLE
   sortTable[0xB10E]=34327;   // HANGUL SYLLABLE
   sortTable[0xB10F]=34328;   // HANGUL SYLLABLE
   sortTable[0xB110]=34329;   // HANGUL SYLLABLE
   sortTable[0xB111]=34330;   // HANGUL SYLLABLE
   sortTable[0xB112]=34331;   // HANGUL SYLLABLE
   sortTable[0xB113]=34332;   // HANGUL SYLLABLE
   sortTable[0xB114]=34333;   // HANGUL SYLLABLE
   sortTable[0xB115]=34334;   // HANGUL SYLLABLE
   sortTable[0xB116]=34335;   // HANGUL SYLLABLE
   sortTable[0xB117]=34336;   // HANGUL SYLLABLE
   sortTable[0xB118]=34337;   // HANGUL SYLLABLE
   sortTable[0xB119]=34338;   // HANGUL SYLLABLE
   sortTable[0xB11A]=34339;   // HANGUL SYLLABLE
   sortTable[0xB11B]=34340;   // HANGUL SYLLABLE
   sortTable[0xB11C]=34341;   // HANGUL SYLLABLE
   sortTable[0xB11D]=34342;   // HANGUL SYLLABLE
   sortTable[0xB11E]=34343;   // HANGUL SYLLABLE
   sortTable[0xB11F]=34344;   // HANGUL SYLLABLE
   sortTable[0xB120]=34345;   // HANGUL SYLLABLE
   sortTable[0xB121]=34346;   // HANGUL SYLLABLE
   sortTable[0xB122]=34347;   // HANGUL SYLLABLE
   sortTable[0xB123]=34348;   // HANGUL SYLLABLE
   sortTable[0xB124]=34349;   // HANGUL SYLLABLE
   sortTable[0xB125]=34350;   // HANGUL SYLLABLE
   sortTable[0xB126]=34351;   // HANGUL SYLLABLE
   sortTable[0xB127]=34352;   // HANGUL SYLLABLE
   sortTable[0xB128]=34353;   // HANGUL SYLLABLE
   sortTable[0xB129]=34354;   // HANGUL SYLLABLE
   sortTable[0xB12A]=34355;   // HANGUL SYLLABLE
   sortTable[0xB12B]=34356;   // HANGUL SYLLABLE
   sortTable[0xB12C]=34357;   // HANGUL SYLLABLE
   sortTable[0xB12D]=34358;   // HANGUL SYLLABLE
   sortTable[0xB12E]=34359;   // HANGUL SYLLABLE
   sortTable[0xB12F]=34360;   // HANGUL SYLLABLE
   sortTable[0xB130]=34361;   // HANGUL SYLLABLE
   sortTable[0xB131]=34362;   // HANGUL SYLLABLE
   sortTable[0xB132]=34363;   // HANGUL SYLLABLE
   sortTable[0xB133]=34364;   // HANGUL SYLLABLE
   sortTable[0xB134]=34365;   // HANGUL SYLLABLE
   sortTable[0xB135]=34366;   // HANGUL SYLLABLE
   sortTable[0xB136]=34367;   // HANGUL SYLLABLE
   sortTable[0xB137]=34368;   // HANGUL SYLLABLE
   sortTable[0xB138]=34369;   // HANGUL SYLLABLE
   sortTable[0xB139]=34370;   // HANGUL SYLLABLE
   sortTable[0xB13A]=34371;   // HANGUL SYLLABLE
   sortTable[0xB13B]=34372;   // HANGUL SYLLABLE
   sortTable[0xB13C]=34373;   // HANGUL SYLLABLE
   sortTable[0xB13D]=34374;   // HANGUL SYLLABLE
   sortTable[0xB13E]=34375;   // HANGUL SYLLABLE
   sortTable[0xB13F]=34376;   // HANGUL SYLLABLE
   sortTable[0xB140]=34377;   // HANGUL SYLLABLE
   sortTable[0xB141]=34378;   // HANGUL SYLLABLE
   sortTable[0xB142]=34379;   // HANGUL SYLLABLE
   sortTable[0xB143]=34380;   // HANGUL SYLLABLE
   sortTable[0xB144]=34381;   // HANGUL SYLLABLE
   sortTable[0xB145]=34382;   // HANGUL SYLLABLE
   sortTable[0xB146]=34383;   // HANGUL SYLLABLE
   sortTable[0xB147]=34384;   // HANGUL SYLLABLE
   sortTable[0xB148]=34385;   // HANGUL SYLLABLE
   sortTable[0xB149]=34386;   // HANGUL SYLLABLE
   sortTable[0xB14A]=34387;   // HANGUL SYLLABLE
   sortTable[0xB14B]=34388;   // HANGUL SYLLABLE
   sortTable[0xB14C]=34389;   // HANGUL SYLLABLE
   sortTable[0xB14D]=34390;   // HANGUL SYLLABLE
   sortTable[0xB14E]=34391;   // HANGUL SYLLABLE
   sortTable[0xB14F]=34392;   // HANGUL SYLLABLE
   sortTable[0xB150]=34393;   // HANGUL SYLLABLE
   sortTable[0xB151]=34394;   // HANGUL SYLLABLE
   sortTable[0xB152]=34395;   // HANGUL SYLLABLE
   sortTable[0xB153]=34396;   // HANGUL SYLLABLE
   sortTable[0xB154]=34397;   // HANGUL SYLLABLE
   sortTable[0xB155]=34398;   // HANGUL SYLLABLE
   sortTable[0xB156]=34399;   // HANGUL SYLLABLE
   sortTable[0xB157]=34400;   // HANGUL SYLLABLE
   sortTable[0xB158]=34401;   // HANGUL SYLLABLE
   sortTable[0xB159]=34402;   // HANGUL SYLLABLE
   sortTable[0xB15A]=34403;   // HANGUL SYLLABLE
   sortTable[0xB15B]=34404;   // HANGUL SYLLABLE
   sortTable[0xB15C]=34405;   // HANGUL SYLLABLE
   sortTable[0xB15D]=34406;   // HANGUL SYLLABLE
   sortTable[0xB15E]=34407;   // HANGUL SYLLABLE
   sortTable[0xB15F]=34408;   // HANGUL SYLLABLE
   sortTable[0xB160]=34409;   // HANGUL SYLLABLE
   sortTable[0xB161]=34410;   // HANGUL SYLLABLE
   sortTable[0xB162]=34411;   // HANGUL SYLLABLE
   sortTable[0xB163]=34412;   // HANGUL SYLLABLE
   sortTable[0xB164]=34413;   // HANGUL SYLLABLE
   sortTable[0xB165]=34414;   // HANGUL SYLLABLE
   sortTable[0xB166]=34415;   // HANGUL SYLLABLE
   sortTable[0xB167]=34416;   // HANGUL SYLLABLE
   sortTable[0xB168]=34417;   // HANGUL SYLLABLE
   sortTable[0xB169]=34418;   // HANGUL SYLLABLE
   sortTable[0xB16A]=34419;   // HANGUL SYLLABLE
   sortTable[0xB16B]=34420;   // HANGUL SYLLABLE
   sortTable[0xB16C]=34421;   // HANGUL SYLLABLE
   sortTable[0xB16D]=34422;   // HANGUL SYLLABLE
   sortTable[0xB16E]=34423;   // HANGUL SYLLABLE
   sortTable[0xB16F]=34424;   // HANGUL SYLLABLE
   sortTable[0xB170]=34425;   // HANGUL SYLLABLE
   sortTable[0xB171]=34426;   // HANGUL SYLLABLE
   sortTable[0xB172]=34427;   // HANGUL SYLLABLE
   sortTable[0xB173]=34428;   // HANGUL SYLLABLE
   sortTable[0xB174]=34429;   // HANGUL SYLLABLE
   sortTable[0xB175]=34430;   // HANGUL SYLLABLE
   sortTable[0xB176]=34431;   // HANGUL SYLLABLE
   sortTable[0xB177]=34432;   // HANGUL SYLLABLE
   sortTable[0xB178]=34433;   // HANGUL SYLLABLE
   sortTable[0xB179]=34434;   // HANGUL SYLLABLE
   sortTable[0xB17A]=34435;   // HANGUL SYLLABLE
   sortTable[0xB17B]=34436;   // HANGUL SYLLABLE
   sortTable[0xB17C]=34437;   // HANGUL SYLLABLE
   sortTable[0xB17D]=34438;   // HANGUL SYLLABLE
   sortTable[0xB17E]=34439;   // HANGUL SYLLABLE
   sortTable[0xB17F]=34440;   // HANGUL SYLLABLE
   sortTable[0xB180]=34441;   // HANGUL SYLLABLE
   sortTable[0xB181]=34442;   // HANGUL SYLLABLE
   sortTable[0xB182]=34443;   // HANGUL SYLLABLE
   sortTable[0xB183]=34444;   // HANGUL SYLLABLE
   sortTable[0xB184]=34445;   // HANGUL SYLLABLE
   sortTable[0xB185]=34446;   // HANGUL SYLLABLE
   sortTable[0xB186]=34447;   // HANGUL SYLLABLE
   sortTable[0xB187]=34448;   // HANGUL SYLLABLE
   sortTable[0xB188]=34449;   // HANGUL SYLLABLE
   sortTable[0xB189]=34450;   // HANGUL SYLLABLE
   sortTable[0xB18A]=34451;   // HANGUL SYLLABLE
   sortTable[0xB18B]=34452;   // HANGUL SYLLABLE
   sortTable[0xB18C]=34453;   // HANGUL SYLLABLE
   sortTable[0xB18D]=34454;   // HANGUL SYLLABLE
   sortTable[0xB18E]=34455;   // HANGUL SYLLABLE
   sortTable[0xB18F]=34456;   // HANGUL SYLLABLE
   sortTable[0xB190]=34457;   // HANGUL SYLLABLE
   sortTable[0xB191]=34458;   // HANGUL SYLLABLE
   sortTable[0xB192]=34459;   // HANGUL SYLLABLE
   sortTable[0xB193]=34460;   // HANGUL SYLLABLE
   sortTable[0xB194]=34461;   // HANGUL SYLLABLE
   sortTable[0xB195]=34462;   // HANGUL SYLLABLE
   sortTable[0xB196]=34463;   // HANGUL SYLLABLE
   sortTable[0xB197]=34464;   // HANGUL SYLLABLE
   sortTable[0xB198]=34465;   // HANGUL SYLLABLE
   sortTable[0xB199]=34466;   // HANGUL SYLLABLE
   sortTable[0xB19A]=34467;   // HANGUL SYLLABLE
   sortTable[0xB19B]=34468;   // HANGUL SYLLABLE
   sortTable[0xB19C]=34469;   // HANGUL SYLLABLE
   sortTable[0xB19D]=34470;   // HANGUL SYLLABLE
   sortTable[0xB19E]=34471;   // HANGUL SYLLABLE
   sortTable[0xB19F]=34472;   // HANGUL SYLLABLE
   sortTable[0xB1A0]=34473;   // HANGUL SYLLABLE
   sortTable[0xB1A1]=34474;   // HANGUL SYLLABLE
   sortTable[0xB1A2]=34475;   // HANGUL SYLLABLE
   sortTable[0xB1A3]=34476;   // HANGUL SYLLABLE
   sortTable[0xB1A4]=34477;   // HANGUL SYLLABLE
   sortTable[0xB1A5]=34478;   // HANGUL SYLLABLE
   sortTable[0xB1A6]=34479;   // HANGUL SYLLABLE
   sortTable[0xB1A7]=34480;   // HANGUL SYLLABLE
   sortTable[0xB1A8]=34481;   // HANGUL SYLLABLE
   sortTable[0xB1A9]=34482;   // HANGUL SYLLABLE
   sortTable[0xB1AA]=34483;   // HANGUL SYLLABLE
   sortTable[0xB1AB]=34484;   // HANGUL SYLLABLE
   sortTable[0xB1AC]=34485;   // HANGUL SYLLABLE
   sortTable[0xB1AD]=34486;   // HANGUL SYLLABLE
   sortTable[0xB1AE]=34487;   // HANGUL SYLLABLE
   sortTable[0xB1AF]=34488;   // HANGUL SYLLABLE
   sortTable[0xB1B0]=34489;   // HANGUL SYLLABLE
   sortTable[0xB1B1]=34490;   // HANGUL SYLLABLE
   sortTable[0xB1B2]=34491;   // HANGUL SYLLABLE
   sortTable[0xB1B3]=34492;   // HANGUL SYLLABLE
   sortTable[0xB1B4]=34493;   // HANGUL SYLLABLE
   sortTable[0xB1B5]=34494;   // HANGUL SYLLABLE
   sortTable[0xB1B6]=34495;   // HANGUL SYLLABLE
   sortTable[0xB1B7]=34496;   // HANGUL SYLLABLE
   sortTable[0xB1B8]=34497;   // HANGUL SYLLABLE
   sortTable[0xB1B9]=34498;   // HANGUL SYLLABLE
   sortTable[0xB1BA]=34499;   // HANGUL SYLLABLE
   sortTable[0xB1BB]=34500;   // HANGUL SYLLABLE
   sortTable[0xB1BC]=34501;   // HANGUL SYLLABLE
   sortTable[0xB1BD]=34502;   // HANGUL SYLLABLE
   sortTable[0xB1BE]=34503;   // HANGUL SYLLABLE
   sortTable[0xB1BF]=34504;   // HANGUL SYLLABLE
   sortTable[0xB1C0]=34505;   // HANGUL SYLLABLE
   sortTable[0xB1C1]=34506;   // HANGUL SYLLABLE
   sortTable[0xB1C2]=34507;   // HANGUL SYLLABLE
   sortTable[0xB1C3]=34508;   // HANGUL SYLLABLE
   sortTable[0xB1C4]=34509;   // HANGUL SYLLABLE
   sortTable[0xB1C5]=34510;   // HANGUL SYLLABLE
   sortTable[0xB1C6]=34511;   // HANGUL SYLLABLE
   sortTable[0xB1C7]=34512;   // HANGUL SYLLABLE
   sortTable[0xB1C8]=34513;   // HANGUL SYLLABLE
   sortTable[0xB1C9]=34514;   // HANGUL SYLLABLE
   sortTable[0xB1CA]=34515;   // HANGUL SYLLABLE
   sortTable[0xB1CB]=34516;   // HANGUL SYLLABLE
   sortTable[0xB1CC]=34517;   // HANGUL SYLLABLE
   sortTable[0xB1CD]=34518;   // HANGUL SYLLABLE
   sortTable[0xB1CE]=34519;   // HANGUL SYLLABLE
   sortTable[0xB1CF]=34520;   // HANGUL SYLLABLE
   sortTable[0xB1D0]=34521;   // HANGUL SYLLABLE
   sortTable[0xB1D1]=34522;   // HANGUL SYLLABLE
   sortTable[0xB1D2]=34523;   // HANGUL SYLLABLE
   sortTable[0xB1D3]=34524;   // HANGUL SYLLABLE
   sortTable[0xB1D4]=34525;   // HANGUL SYLLABLE
   sortTable[0xB1D5]=34526;   // HANGUL SYLLABLE
   sortTable[0xB1D6]=34527;   // HANGUL SYLLABLE
   sortTable[0xB1D7]=34528;   // HANGUL SYLLABLE
   sortTable[0xB1D8]=34529;   // HANGUL SYLLABLE
   sortTable[0xB1D9]=34530;   // HANGUL SYLLABLE
   sortTable[0xB1DA]=34531;   // HANGUL SYLLABLE
   sortTable[0xB1DB]=34532;   // HANGUL SYLLABLE
   sortTable[0xB1DC]=34533;   // HANGUL SYLLABLE
   sortTable[0xB1DD]=34534;   // HANGUL SYLLABLE
   sortTable[0xB1DE]=34535;   // HANGUL SYLLABLE
   sortTable[0xB1DF]=34536;   // HANGUL SYLLABLE
   sortTable[0xB1E0]=34537;   // HANGUL SYLLABLE
   sortTable[0xB1E1]=34538;   // HANGUL SYLLABLE
   sortTable[0xB1E2]=34539;   // HANGUL SYLLABLE
   sortTable[0xB1E3]=34540;   // HANGUL SYLLABLE
   sortTable[0xB1E4]=34541;   // HANGUL SYLLABLE
   sortTable[0xB1E5]=34542;   // HANGUL SYLLABLE
   sortTable[0xB1E6]=34543;   // HANGUL SYLLABLE
   sortTable[0xB1E7]=34544;   // HANGUL SYLLABLE
   sortTable[0xB1E8]=34545;   // HANGUL SYLLABLE
   sortTable[0xB1E9]=34546;   // HANGUL SYLLABLE
   sortTable[0xB1EA]=34547;   // HANGUL SYLLABLE
   sortTable[0xB1EB]=34548;   // HANGUL SYLLABLE
   sortTable[0xB1EC]=34549;   // HANGUL SYLLABLE
   sortTable[0xB1ED]=34550;   // HANGUL SYLLABLE
   sortTable[0xB1EE]=34551;   // HANGUL SYLLABLE
   sortTable[0xB1EF]=34552;   // HANGUL SYLLABLE
   sortTable[0xB1F0]=34553;   // HANGUL SYLLABLE
   sortTable[0xB1F1]=34554;   // HANGUL SYLLABLE
   sortTable[0xB1F2]=34555;   // HANGUL SYLLABLE
   sortTable[0xB1F3]=34556;   // HANGUL SYLLABLE
   sortTable[0xB1F4]=34557;   // HANGUL SYLLABLE
   sortTable[0xB1F5]=34558;   // HANGUL SYLLABLE
   sortTable[0xB1F6]=34559;   // HANGUL SYLLABLE
   sortTable[0xB1F7]=34560;   // HANGUL SYLLABLE
   sortTable[0xB1F8]=34561;   // HANGUL SYLLABLE
   sortTable[0xB1F9]=34562;   // HANGUL SYLLABLE
   sortTable[0xB1FA]=34563;   // HANGUL SYLLABLE
   sortTable[0xB1FB]=34564;   // HANGUL SYLLABLE
   sortTable[0xB1FC]=34565;   // HANGUL SYLLABLE
   sortTable[0xB1FD]=34566;   // HANGUL SYLLABLE
   sortTable[0xB1FE]=34567;   // HANGUL SYLLABLE
   sortTable[0xB1FF]=34568;   // HANGUL SYLLABLE
   sortTable[0xB200]=34569;   // HANGUL SYLLABLE
   sortTable[0xB201]=34570;   // HANGUL SYLLABLE
   sortTable[0xB202]=34571;   // HANGUL SYLLABLE
   sortTable[0xB203]=34572;   // HANGUL SYLLABLE
   sortTable[0xB204]=34573;   // HANGUL SYLLABLE
   sortTable[0xB205]=34574;   // HANGUL SYLLABLE
   sortTable[0xB206]=34575;   // HANGUL SYLLABLE
   sortTable[0xB207]=34576;   // HANGUL SYLLABLE
   sortTable[0xB208]=34577;   // HANGUL SYLLABLE
   sortTable[0xB209]=34578;   // HANGUL SYLLABLE
   sortTable[0xB20A]=34579;   // HANGUL SYLLABLE
   sortTable[0xB20B]=34580;   // HANGUL SYLLABLE
   sortTable[0xB20C]=34581;   // HANGUL SYLLABLE
   sortTable[0xB20D]=34582;   // HANGUL SYLLABLE
   sortTable[0xB20E]=34583;   // HANGUL SYLLABLE
   sortTable[0xB20F]=34584;   // HANGUL SYLLABLE
   sortTable[0xB210]=34585;   // HANGUL SYLLABLE
   sortTable[0xB211]=34586;   // HANGUL SYLLABLE
   sortTable[0xB212]=34587;   // HANGUL SYLLABLE
   sortTable[0xB213]=34588;   // HANGUL SYLLABLE
   sortTable[0xB214]=34589;   // HANGUL SYLLABLE
   sortTable[0xB215]=34590;   // HANGUL SYLLABLE
   sortTable[0xB216]=34591;   // HANGUL SYLLABLE
   sortTable[0xB217]=34592;   // HANGUL SYLLABLE
   sortTable[0xB218]=34593;   // HANGUL SYLLABLE
   sortTable[0xB219]=34594;   // HANGUL SYLLABLE
   sortTable[0xB21A]=34595;   // HANGUL SYLLABLE
   sortTable[0xB21B]=34596;   // HANGUL SYLLABLE
   sortTable[0xB21C]=34597;   // HANGUL SYLLABLE
   sortTable[0xB21D]=34598;   // HANGUL SYLLABLE
   sortTable[0xB21E]=34599;   // HANGUL SYLLABLE
   sortTable[0xB21F]=34600;   // HANGUL SYLLABLE
   sortTable[0xB220]=34601;   // HANGUL SYLLABLE
   sortTable[0xB221]=34602;   // HANGUL SYLLABLE
   sortTable[0xB222]=34603;   // HANGUL SYLLABLE
   sortTable[0xB223]=34604;   // HANGUL SYLLABLE
   sortTable[0xB224]=34605;   // HANGUL SYLLABLE
   sortTable[0xB225]=34606;   // HANGUL SYLLABLE
   sortTable[0xB226]=34607;   // HANGUL SYLLABLE
   sortTable[0xB227]=34608;   // HANGUL SYLLABLE
   sortTable[0xB228]=34609;   // HANGUL SYLLABLE
   sortTable[0xB229]=34610;   // HANGUL SYLLABLE
   sortTable[0xB22A]=34611;   // HANGUL SYLLABLE
   sortTable[0xB22B]=34612;   // HANGUL SYLLABLE
   sortTable[0xB22C]=34613;   // HANGUL SYLLABLE
   sortTable[0xB22D]=34614;   // HANGUL SYLLABLE
   sortTable[0xB22E]=34615;   // HANGUL SYLLABLE
   sortTable[0xB22F]=34616;   // HANGUL SYLLABLE
   sortTable[0xB230]=34617;   // HANGUL SYLLABLE
   sortTable[0xB231]=34618;   // HANGUL SYLLABLE
   sortTable[0xB232]=34619;   // HANGUL SYLLABLE
   sortTable[0xB233]=34620;   // HANGUL SYLLABLE
   sortTable[0xB234]=34621;   // HANGUL SYLLABLE
   sortTable[0xB235]=34622;   // HANGUL SYLLABLE
   sortTable[0xB236]=34623;   // HANGUL SYLLABLE
   sortTable[0xB237]=34624;   // HANGUL SYLLABLE
   sortTable[0xB238]=34625;   // HANGUL SYLLABLE
   sortTable[0xB239]=34626;   // HANGUL SYLLABLE
   sortTable[0xB23A]=34627;   // HANGUL SYLLABLE
   sortTable[0xB23B]=34628;   // HANGUL SYLLABLE
   sortTable[0xB23C]=34629;   // HANGUL SYLLABLE
   sortTable[0xB23D]=34630;   // HANGUL SYLLABLE
   sortTable[0xB23E]=34631;   // HANGUL SYLLABLE
   sortTable[0xB23F]=34632;   // HANGUL SYLLABLE
   sortTable[0xB240]=34633;   // HANGUL SYLLABLE
   sortTable[0xB241]=34634;   // HANGUL SYLLABLE
   sortTable[0xB242]=34635;   // HANGUL SYLLABLE
   sortTable[0xB243]=34636;   // HANGUL SYLLABLE
   sortTable[0xB244]=34637;   // HANGUL SYLLABLE
   sortTable[0xB245]=34638;   // HANGUL SYLLABLE
   sortTable[0xB246]=34639;   // HANGUL SYLLABLE
   sortTable[0xB247]=34640;   // HANGUL SYLLABLE
   sortTable[0xB248]=34641;   // HANGUL SYLLABLE
   sortTable[0xB249]=34642;   // HANGUL SYLLABLE
   sortTable[0xB24A]=34643;   // HANGUL SYLLABLE
   sortTable[0xB24B]=34644;   // HANGUL SYLLABLE
   sortTable[0xB24C]=34645;   // HANGUL SYLLABLE
   sortTable[0xB24D]=34646;   // HANGUL SYLLABLE
   sortTable[0xB24E]=34647;   // HANGUL SYLLABLE
   sortTable[0xB24F]=34648;   // HANGUL SYLLABLE
   sortTable[0xB250]=34649;   // HANGUL SYLLABLE
   sortTable[0xB251]=34650;   // HANGUL SYLLABLE
   sortTable[0xB252]=34651;   // HANGUL SYLLABLE
   sortTable[0xB253]=34652;   // HANGUL SYLLABLE
   sortTable[0xB254]=34653;   // HANGUL SYLLABLE
   sortTable[0xB255]=34654;   // HANGUL SYLLABLE
   sortTable[0xB256]=34655;   // HANGUL SYLLABLE
   sortTable[0xB257]=34656;   // HANGUL SYLLABLE
   sortTable[0xB258]=34657;   // HANGUL SYLLABLE
   sortTable[0xB259]=34658;   // HANGUL SYLLABLE
   sortTable[0xB25A]=34659;   // HANGUL SYLLABLE
   sortTable[0xB25B]=34660;   // HANGUL SYLLABLE
   sortTable[0xB25C]=34661;   // HANGUL SYLLABLE
   sortTable[0xB25D]=34662;   // HANGUL SYLLABLE
   sortTable[0xB25E]=34663;   // HANGUL SYLLABLE
   sortTable[0xB25F]=34664;   // HANGUL SYLLABLE
   sortTable[0xB260]=34665;   // HANGUL SYLLABLE
   sortTable[0xB261]=34666;   // HANGUL SYLLABLE
   sortTable[0xB262]=34667;   // HANGUL SYLLABLE
   sortTable[0xB263]=34668;   // HANGUL SYLLABLE
   sortTable[0xB264]=34669;   // HANGUL SYLLABLE
   sortTable[0xB265]=34670;   // HANGUL SYLLABLE
   sortTable[0xB266]=34671;   // HANGUL SYLLABLE
   sortTable[0xB267]=34672;   // HANGUL SYLLABLE
   sortTable[0xB268]=34673;   // HANGUL SYLLABLE
   sortTable[0xB269]=34674;   // HANGUL SYLLABLE
   sortTable[0xB26A]=34675;   // HANGUL SYLLABLE
   sortTable[0xB26B]=34676;   // HANGUL SYLLABLE
   sortTable[0xB26C]=34677;   // HANGUL SYLLABLE
   sortTable[0xB26D]=34678;   // HANGUL SYLLABLE
   sortTable[0xB26E]=34679;   // HANGUL SYLLABLE
   sortTable[0xB26F]=34680;   // HANGUL SYLLABLE
   sortTable[0xB270]=34681;   // HANGUL SYLLABLE
   sortTable[0xB271]=34682;   // HANGUL SYLLABLE
   sortTable[0xB272]=34683;   // HANGUL SYLLABLE
   sortTable[0xB273]=34684;   // HANGUL SYLLABLE
   sortTable[0xB274]=34685;   // HANGUL SYLLABLE
   sortTable[0xB275]=34686;   // HANGUL SYLLABLE
   sortTable[0xB276]=34687;   // HANGUL SYLLABLE
   sortTable[0xB277]=34688;   // HANGUL SYLLABLE
   sortTable[0xB278]=34689;   // HANGUL SYLLABLE
   sortTable[0xB279]=34690;   // HANGUL SYLLABLE
   sortTable[0xB27A]=34691;   // HANGUL SYLLABLE
   sortTable[0xB27B]=34692;   // HANGUL SYLLABLE
   sortTable[0xB27C]=34693;   // HANGUL SYLLABLE
   sortTable[0xB27D]=34694;   // HANGUL SYLLABLE
   sortTable[0xB27E]=34695;   // HANGUL SYLLABLE
   sortTable[0xB27F]=34696;   // HANGUL SYLLABLE
   sortTable[0xB280]=34697;   // HANGUL SYLLABLE
   sortTable[0xB281]=34698;   // HANGUL SYLLABLE
   sortTable[0xB282]=34699;   // HANGUL SYLLABLE
   sortTable[0xB283]=34700;   // HANGUL SYLLABLE
   sortTable[0xB284]=34701;   // HANGUL SYLLABLE
   sortTable[0xB285]=34702;   // HANGUL SYLLABLE
   sortTable[0xB286]=34703;   // HANGUL SYLLABLE
   sortTable[0xB287]=34704;   // HANGUL SYLLABLE
   sortTable[0xB288]=34705;   // HANGUL SYLLABLE
   sortTable[0xB289]=34706;   // HANGUL SYLLABLE
   sortTable[0xB28A]=34707;   // HANGUL SYLLABLE
   sortTable[0xB28B]=34708;   // HANGUL SYLLABLE
   sortTable[0xB28C]=34709;   // HANGUL SYLLABLE
   sortTable[0xB28D]=34710;   // HANGUL SYLLABLE
   sortTable[0xB28E]=34711;   // HANGUL SYLLABLE
   sortTable[0xB28F]=34712;   // HANGUL SYLLABLE
   sortTable[0xB290]=34713;   // HANGUL SYLLABLE
   sortTable[0xB291]=34714;   // HANGUL SYLLABLE
   sortTable[0xB292]=34715;   // HANGUL SYLLABLE
   sortTable[0xB293]=34716;   // HANGUL SYLLABLE
   sortTable[0xB294]=34717;   // HANGUL SYLLABLE
   sortTable[0xB295]=34718;   // HANGUL SYLLABLE
   sortTable[0xB296]=34719;   // HANGUL SYLLABLE
   sortTable[0xB297]=34720;   // HANGUL SYLLABLE
   sortTable[0xB298]=34721;   // HANGUL SYLLABLE
   sortTable[0xB299]=34722;   // HANGUL SYLLABLE
   sortTable[0xB29A]=34723;   // HANGUL SYLLABLE
   sortTable[0xB29B]=34724;   // HANGUL SYLLABLE
   sortTable[0xB29C]=34725;   // HANGUL SYLLABLE
   sortTable[0xB29D]=34726;   // HANGUL SYLLABLE
   sortTable[0xB29E]=34727;   // HANGUL SYLLABLE
   sortTable[0xB29F]=34728;   // HANGUL SYLLABLE
   sortTable[0xB2A0]=34729;   // HANGUL SYLLABLE
   sortTable[0xB2A1]=34730;   // HANGUL SYLLABLE
   sortTable[0xB2A2]=34731;   // HANGUL SYLLABLE
   sortTable[0xB2A3]=34732;   // HANGUL SYLLABLE
   sortTable[0xB2A4]=34733;   // HANGUL SYLLABLE
   sortTable[0xB2A5]=34734;   // HANGUL SYLLABLE
   sortTable[0xB2A6]=34735;   // HANGUL SYLLABLE
   sortTable[0xB2A7]=34736;   // HANGUL SYLLABLE
   sortTable[0xB2A8]=34737;   // HANGUL SYLLABLE
   sortTable[0xB2A9]=34738;   // HANGUL SYLLABLE
   sortTable[0xB2AA]=34739;   // HANGUL SYLLABLE
   sortTable[0xB2AB]=34740;   // HANGUL SYLLABLE
   sortTable[0xB2AC]=34741;   // HANGUL SYLLABLE
   sortTable[0xB2AD]=34742;   // HANGUL SYLLABLE
   sortTable[0xB2AE]=34743;   // HANGUL SYLLABLE
   sortTable[0xB2AF]=34744;   // HANGUL SYLLABLE
   sortTable[0xB2B0]=34745;   // HANGUL SYLLABLE
   sortTable[0xB2B1]=34746;   // HANGUL SYLLABLE
   sortTable[0xB2B2]=34747;   // HANGUL SYLLABLE
   sortTable[0xB2B3]=34748;   // HANGUL SYLLABLE
   sortTable[0xB2B4]=34749;   // HANGUL SYLLABLE
   sortTable[0xB2B5]=34750;   // HANGUL SYLLABLE
   sortTable[0xB2B6]=34751;   // HANGUL SYLLABLE
   sortTable[0xB2B7]=34752;   // HANGUL SYLLABLE
   sortTable[0xB2B8]=34753;   // HANGUL SYLLABLE
   sortTable[0xB2B9]=34754;   // HANGUL SYLLABLE
   sortTable[0xB2BA]=34755;   // HANGUL SYLLABLE
   sortTable[0xB2BB]=34756;   // HANGUL SYLLABLE
   sortTable[0xB2BC]=34757;   // HANGUL SYLLABLE
   sortTable[0xB2BD]=34758;   // HANGUL SYLLABLE
   sortTable[0xB2BE]=34759;   // HANGUL SYLLABLE
   sortTable[0xB2BF]=34760;   // HANGUL SYLLABLE
   sortTable[0xB2C0]=34761;   // HANGUL SYLLABLE
   sortTable[0xB2C1]=34762;   // HANGUL SYLLABLE
   sortTable[0xB2C2]=34763;   // HANGUL SYLLABLE
   sortTable[0xB2C3]=34764;   // HANGUL SYLLABLE
   sortTable[0xB2C4]=34765;   // HANGUL SYLLABLE
   sortTable[0xB2C5]=34766;   // HANGUL SYLLABLE
   sortTable[0xB2C6]=34767;   // HANGUL SYLLABLE
   sortTable[0xB2C7]=34768;   // HANGUL SYLLABLE
   sortTable[0xB2C8]=34769;   // HANGUL SYLLABLE
   sortTable[0xB2C9]=34770;   // HANGUL SYLLABLE
   sortTable[0xB2CA]=34771;   // HANGUL SYLLABLE
   sortTable[0xB2CB]=34772;   // HANGUL SYLLABLE
   sortTable[0xB2CC]=34773;   // HANGUL SYLLABLE
   sortTable[0xB2CD]=34774;   // HANGUL SYLLABLE
   sortTable[0xB2CE]=34775;   // HANGUL SYLLABLE
   sortTable[0xB2CF]=34776;   // HANGUL SYLLABLE
   sortTable[0xB2D0]=34777;   // HANGUL SYLLABLE
   sortTable[0xB2D1]=34778;   // HANGUL SYLLABLE
   sortTable[0xB2D2]=34779;   // HANGUL SYLLABLE
   sortTable[0xB2D3]=34780;   // HANGUL SYLLABLE
   sortTable[0xB2D4]=34781;   // HANGUL SYLLABLE
   sortTable[0xB2D5]=34782;   // HANGUL SYLLABLE
   sortTable[0xB2D6]=34783;   // HANGUL SYLLABLE
   sortTable[0xB2D7]=34784;   // HANGUL SYLLABLE
   sortTable[0xB2D8]=34785;   // HANGUL SYLLABLE
   sortTable[0xB2D9]=34786;   // HANGUL SYLLABLE
   sortTable[0xB2DA]=34787;   // HANGUL SYLLABLE
   sortTable[0xB2DB]=34788;   // HANGUL SYLLABLE
   sortTable[0xB2DC]=34789;   // HANGUL SYLLABLE
   sortTable[0xB2DD]=34790;   // HANGUL SYLLABLE
   sortTable[0xB2DE]=34791;   // HANGUL SYLLABLE
   sortTable[0xB2DF]=34792;   // HANGUL SYLLABLE
   sortTable[0xB2E0]=34793;   // HANGUL SYLLABLE
   sortTable[0xB2E1]=34794;   // HANGUL SYLLABLE
   sortTable[0xB2E2]=34795;   // HANGUL SYLLABLE
   sortTable[0xB2E3]=34796;   // HANGUL SYLLABLE
   sortTable[0x3137]=34797;   // HANGUL LETTER TIKEUT
   sortTable[0x1103]=34798;   // HANGUL CHOSEONG TIKEUT
   sortTable[0x11AE]=34799;   // HANGUL JONGSEONG TIKEUT
   sortTable[0xB2E4]=34800;   // HANGUL SYLLABLE
   sortTable[0xB2E5]=34801;   // HANGUL SYLLABLE
   sortTable[0xB2E6]=34802;   // HANGUL SYLLABLE
   sortTable[0xB2E7]=34803;   // HANGUL SYLLABLE
   sortTable[0xB2E8]=34804;   // HANGUL SYLLABLE
   sortTable[0xB2E9]=34805;   // HANGUL SYLLABLE
   sortTable[0xB2EA]=34806;   // HANGUL SYLLABLE
   sortTable[0xB2EB]=34807;   // HANGUL SYLLABLE
   sortTable[0xB2EC]=34808;   // HANGUL SYLLABLE
   sortTable[0xB2ED]=34809;   // HANGUL SYLLABLE
   sortTable[0xB2EE]=34810;   // HANGUL SYLLABLE
   sortTable[0xB2EF]=34811;   // HANGUL SYLLABLE
   sortTable[0xB2F0]=34812;   // HANGUL SYLLABLE
   sortTable[0xB2F1]=34813;   // HANGUL SYLLABLE
   sortTable[0xB2F2]=34814;   // HANGUL SYLLABLE
   sortTable[0xB2F3]=34815;   // HANGUL SYLLABLE
   sortTable[0xB2F4]=34816;   // HANGUL SYLLABLE
   sortTable[0xB2F5]=34817;   // HANGUL SYLLABLE
   sortTable[0xB2F6]=34818;   // HANGUL SYLLABLE
   sortTable[0xB2F7]=34819;   // HANGUL SYLLABLE
   sortTable[0xB2F8]=34820;   // HANGUL SYLLABLE
   sortTable[0xB2F9]=34821;   // HANGUL SYLLABLE
   sortTable[0xB2FA]=34822;   // HANGUL SYLLABLE
   sortTable[0xB2FB]=34823;   // HANGUL SYLLABLE
   sortTable[0xB2FC]=34824;   // HANGUL SYLLABLE
   sortTable[0xB2FD]=34825;   // HANGUL SYLLABLE
   sortTable[0xB2FE]=34826;   // HANGUL SYLLABLE
   sortTable[0xB2FF]=34827;   // HANGUL SYLLABLE
   sortTable[0xB300]=34828;   // HANGUL SYLLABLE
   sortTable[0xB301]=34829;   // HANGUL SYLLABLE
   sortTable[0xB302]=34830;   // HANGUL SYLLABLE
   sortTable[0xB303]=34831;   // HANGUL SYLLABLE
   sortTable[0xB304]=34832;   // HANGUL SYLLABLE
   sortTable[0xB305]=34833;   // HANGUL SYLLABLE
   sortTable[0xB306]=34834;   // HANGUL SYLLABLE
   sortTable[0xB307]=34835;   // HANGUL SYLLABLE
   sortTable[0xB308]=34836;   // HANGUL SYLLABLE
   sortTable[0xB309]=34837;   // HANGUL SYLLABLE
   sortTable[0xB30A]=34838;   // HANGUL SYLLABLE
   sortTable[0xB30B]=34839;   // HANGUL SYLLABLE
   sortTable[0xB30C]=34840;   // HANGUL SYLLABLE
   sortTable[0xB30D]=34841;   // HANGUL SYLLABLE
   sortTable[0xB30E]=34842;   // HANGUL SYLLABLE
   sortTable[0xB30F]=34843;   // HANGUL SYLLABLE
   sortTable[0xB310]=34844;   // HANGUL SYLLABLE
   sortTable[0xB311]=34845;   // HANGUL SYLLABLE
   sortTable[0xB312]=34846;   // HANGUL SYLLABLE
   sortTable[0xB313]=34847;   // HANGUL SYLLABLE
   sortTable[0xB314]=34848;   // HANGUL SYLLABLE
   sortTable[0xB315]=34849;   // HANGUL SYLLABLE
   sortTable[0xB316]=34850;   // HANGUL SYLLABLE
   sortTable[0xB317]=34851;   // HANGUL SYLLABLE
   sortTable[0xB318]=34852;   // HANGUL SYLLABLE
   sortTable[0xB319]=34853;   // HANGUL SYLLABLE
   sortTable[0xB31A]=34854;   // HANGUL SYLLABLE
   sortTable[0xB31B]=34855;   // HANGUL SYLLABLE
   sortTable[0xB31C]=34856;   // HANGUL SYLLABLE
   sortTable[0xB31D]=34857;   // HANGUL SYLLABLE
   sortTable[0xB31E]=34858;   // HANGUL SYLLABLE
   sortTable[0xB31F]=34859;   // HANGUL SYLLABLE
   sortTable[0xB320]=34860;   // HANGUL SYLLABLE
   sortTable[0xB321]=34861;   // HANGUL SYLLABLE
   sortTable[0xB322]=34862;   // HANGUL SYLLABLE
   sortTable[0xB323]=34863;   // HANGUL SYLLABLE
   sortTable[0xB324]=34864;   // HANGUL SYLLABLE
   sortTable[0xB325]=34865;   // HANGUL SYLLABLE
   sortTable[0xB326]=34866;   // HANGUL SYLLABLE
   sortTable[0xB327]=34867;   // HANGUL SYLLABLE
   sortTable[0xB328]=34868;   // HANGUL SYLLABLE
   sortTable[0xB329]=34869;   // HANGUL SYLLABLE
   sortTable[0xB32A]=34870;   // HANGUL SYLLABLE
   sortTable[0xB32B]=34871;   // HANGUL SYLLABLE
   sortTable[0xB32C]=34872;   // HANGUL SYLLABLE
   sortTable[0xB32D]=34873;   // HANGUL SYLLABLE
   sortTable[0xB32E]=34874;   // HANGUL SYLLABLE
   sortTable[0xB32F]=34875;   // HANGUL SYLLABLE
   sortTable[0xB330]=34876;   // HANGUL SYLLABLE
   sortTable[0xB331]=34877;   // HANGUL SYLLABLE
   sortTable[0xB332]=34878;   // HANGUL SYLLABLE
   sortTable[0xB333]=34879;   // HANGUL SYLLABLE
   sortTable[0xB334]=34880;   // HANGUL SYLLABLE
   sortTable[0xB335]=34881;   // HANGUL SYLLABLE
   sortTable[0xB336]=34882;   // HANGUL SYLLABLE
   sortTable[0xB337]=34883;   // HANGUL SYLLABLE
   sortTable[0xB338]=34884;   // HANGUL SYLLABLE
   sortTable[0xB339]=34885;   // HANGUL SYLLABLE
   sortTable[0xB33A]=34886;   // HANGUL SYLLABLE
   sortTable[0xB33B]=34887;   // HANGUL SYLLABLE
   sortTable[0xB33C]=34888;   // HANGUL SYLLABLE
   sortTable[0xB33D]=34889;   // HANGUL SYLLABLE
   sortTable[0xB33E]=34890;   // HANGUL SYLLABLE
   sortTable[0xB33F]=34891;   // HANGUL SYLLABLE
   sortTable[0xB340]=34892;   // HANGUL SYLLABLE
   sortTable[0xB341]=34893;   // HANGUL SYLLABLE
   sortTable[0xB342]=34894;   // HANGUL SYLLABLE
   sortTable[0xB343]=34895;   // HANGUL SYLLABLE
   sortTable[0xB344]=34896;   // HANGUL SYLLABLE
   sortTable[0xB345]=34897;   // HANGUL SYLLABLE
   sortTable[0xB346]=34898;   // HANGUL SYLLABLE
   sortTable[0xB347]=34899;   // HANGUL SYLLABLE
   sortTable[0xB348]=34900;   // HANGUL SYLLABLE
   sortTable[0xB349]=34901;   // HANGUL SYLLABLE
   sortTable[0xB34A]=34902;   // HANGUL SYLLABLE
   sortTable[0xB34B]=34903;   // HANGUL SYLLABLE
   sortTable[0xB34C]=34904;   // HANGUL SYLLABLE
   sortTable[0xB34D]=34905;   // HANGUL SYLLABLE
   sortTable[0xB34E]=34906;   // HANGUL SYLLABLE
   sortTable[0xB34F]=34907;   // HANGUL SYLLABLE
   sortTable[0xB350]=34908;   // HANGUL SYLLABLE
   sortTable[0xB351]=34909;   // HANGUL SYLLABLE
   sortTable[0xB352]=34910;   // HANGUL SYLLABLE
   sortTable[0xB353]=34911;   // HANGUL SYLLABLE
   sortTable[0xB354]=34912;   // HANGUL SYLLABLE
   sortTable[0xB355]=34913;   // HANGUL SYLLABLE
   sortTable[0xB356]=34914;   // HANGUL SYLLABLE
   sortTable[0xB357]=34915;   // HANGUL SYLLABLE
   sortTable[0xB358]=34916;   // HANGUL SYLLABLE
   sortTable[0xB359]=34917;   // HANGUL SYLLABLE
   sortTable[0xB35A]=34918;   // HANGUL SYLLABLE
   sortTable[0xB35B]=34919;   // HANGUL SYLLABLE
   sortTable[0xB35C]=34920;   // HANGUL SYLLABLE
   sortTable[0xB35D]=34921;   // HANGUL SYLLABLE
   sortTable[0xB35E]=34922;   // HANGUL SYLLABLE
   sortTable[0xB35F]=34923;   // HANGUL SYLLABLE
   sortTable[0xB360]=34924;   // HANGUL SYLLABLE
   sortTable[0xB361]=34925;   // HANGUL SYLLABLE
   sortTable[0xB362]=34926;   // HANGUL SYLLABLE
   sortTable[0xB363]=34927;   // HANGUL SYLLABLE
   sortTable[0xB364]=34928;   // HANGUL SYLLABLE
   sortTable[0xB365]=34929;   // HANGUL SYLLABLE
   sortTable[0xB366]=34930;   // HANGUL SYLLABLE
   sortTable[0xB367]=34931;   // HANGUL SYLLABLE
   sortTable[0xB368]=34932;   // HANGUL SYLLABLE
   sortTable[0xB369]=34933;   // HANGUL SYLLABLE
   sortTable[0xB36A]=34934;   // HANGUL SYLLABLE
   sortTable[0xB36B]=34935;   // HANGUL SYLLABLE
   sortTable[0xB36C]=34936;   // HANGUL SYLLABLE
   sortTable[0xB36D]=34937;   // HANGUL SYLLABLE
   sortTable[0xB36E]=34938;   // HANGUL SYLLABLE
   sortTable[0xB36F]=34939;   // HANGUL SYLLABLE
   sortTable[0xB370]=34940;   // HANGUL SYLLABLE
   sortTable[0xB371]=34941;   // HANGUL SYLLABLE
   sortTable[0xB372]=34942;   // HANGUL SYLLABLE
   sortTable[0xB373]=34943;   // HANGUL SYLLABLE
   sortTable[0xB374]=34944;   // HANGUL SYLLABLE
   sortTable[0xB375]=34945;   // HANGUL SYLLABLE
   sortTable[0xB376]=34946;   // HANGUL SYLLABLE
   sortTable[0xB377]=34947;   // HANGUL SYLLABLE
   sortTable[0xB378]=34948;   // HANGUL SYLLABLE
   sortTable[0xB379]=34949;   // HANGUL SYLLABLE
   sortTable[0xB37A]=34950;   // HANGUL SYLLABLE
   sortTable[0xB37B]=34951;   // HANGUL SYLLABLE
   sortTable[0xB37C]=34952;   // HANGUL SYLLABLE
   sortTable[0xB37D]=34953;   // HANGUL SYLLABLE
   sortTable[0xB37E]=34954;   // HANGUL SYLLABLE
   sortTable[0xB37F]=34955;   // HANGUL SYLLABLE
   sortTable[0xB380]=34956;   // HANGUL SYLLABLE
   sortTable[0xB381]=34957;   // HANGUL SYLLABLE
   sortTable[0xB382]=34958;   // HANGUL SYLLABLE
   sortTable[0xB383]=34959;   // HANGUL SYLLABLE
   sortTable[0xB384]=34960;   // HANGUL SYLLABLE
   sortTable[0xB385]=34961;   // HANGUL SYLLABLE
   sortTable[0xB386]=34962;   // HANGUL SYLLABLE
   sortTable[0xB387]=34963;   // HANGUL SYLLABLE
   sortTable[0xB388]=34964;   // HANGUL SYLLABLE
   sortTable[0xB389]=34965;   // HANGUL SYLLABLE
   sortTable[0xB38A]=34966;   // HANGUL SYLLABLE
   sortTable[0xB38B]=34967;   // HANGUL SYLLABLE
   sortTable[0xB38C]=34968;   // HANGUL SYLLABLE
   sortTable[0xB38D]=34969;   // HANGUL SYLLABLE
   sortTable[0xB38E]=34970;   // HANGUL SYLLABLE
   sortTable[0xB38F]=34971;   // HANGUL SYLLABLE
   sortTable[0xB390]=34972;   // HANGUL SYLLABLE
   sortTable[0xB391]=34973;   // HANGUL SYLLABLE
   sortTable[0xB392]=34974;   // HANGUL SYLLABLE
   sortTable[0xB393]=34975;   // HANGUL SYLLABLE
   sortTable[0xB394]=34976;   // HANGUL SYLLABLE
   sortTable[0xB395]=34977;   // HANGUL SYLLABLE
   sortTable[0xB396]=34978;   // HANGUL SYLLABLE
   sortTable[0xB397]=34979;   // HANGUL SYLLABLE
   sortTable[0xB398]=34980;   // HANGUL SYLLABLE
   sortTable[0xB399]=34981;   // HANGUL SYLLABLE
   sortTable[0xB39A]=34982;   // HANGUL SYLLABLE
   sortTable[0xB39B]=34983;   // HANGUL SYLLABLE
   sortTable[0xB39C]=34984;   // HANGUL SYLLABLE
   sortTable[0xB39D]=34985;   // HANGUL SYLLABLE
   sortTable[0xB39E]=34986;   // HANGUL SYLLABLE
   sortTable[0xB39F]=34987;   // HANGUL SYLLABLE
   sortTable[0xB3A0]=34988;   // HANGUL SYLLABLE
   sortTable[0xB3A1]=34989;   // HANGUL SYLLABLE
   sortTable[0xB3A2]=34990;   // HANGUL SYLLABLE
   sortTable[0xB3A3]=34991;   // HANGUL SYLLABLE
   sortTable[0xB3A4]=34992;   // HANGUL SYLLABLE
   sortTable[0xB3A5]=34993;   // HANGUL SYLLABLE
   sortTable[0xB3A6]=34994;   // HANGUL SYLLABLE
   sortTable[0xB3A7]=34995;   // HANGUL SYLLABLE
   sortTable[0xB3A8]=34996;   // HANGUL SYLLABLE
   sortTable[0xB3A9]=34997;   // HANGUL SYLLABLE
   sortTable[0xB3AA]=34998;   // HANGUL SYLLABLE
   sortTable[0xB3AB]=34999;   // HANGUL SYLLABLE
   sortTable[0xB3AC]=35000;   // HANGUL SYLLABLE
   sortTable[0xB3AD]=35001;   // HANGUL SYLLABLE
   sortTable[0xB3AE]=35002;   // HANGUL SYLLABLE
   sortTable[0xB3AF]=35003;   // HANGUL SYLLABLE
   sortTable[0xB3B0]=35004;   // HANGUL SYLLABLE
   sortTable[0xB3B1]=35005;   // HANGUL SYLLABLE
   sortTable[0xB3B2]=35006;   // HANGUL SYLLABLE
   sortTable[0xB3B3]=35007;   // HANGUL SYLLABLE
   sortTable[0xB3B4]=35008;   // HANGUL SYLLABLE
   sortTable[0xB3B5]=35009;   // HANGUL SYLLABLE
   sortTable[0xB3B6]=35010;   // HANGUL SYLLABLE
   sortTable[0xB3B7]=35011;   // HANGUL SYLLABLE
   sortTable[0xB3B8]=35012;   // HANGUL SYLLABLE
   sortTable[0xB3B9]=35013;   // HANGUL SYLLABLE
   sortTable[0xB3BA]=35014;   // HANGUL SYLLABLE
   sortTable[0xB3BB]=35015;   // HANGUL SYLLABLE
   sortTable[0xB3BC]=35016;   // HANGUL SYLLABLE
   sortTable[0xB3BD]=35017;   // HANGUL SYLLABLE
   sortTable[0xB3BE]=35018;   // HANGUL SYLLABLE
   sortTable[0xB3BF]=35019;   // HANGUL SYLLABLE
   sortTable[0xB3C0]=35020;   // HANGUL SYLLABLE
   sortTable[0xB3C1]=35021;   // HANGUL SYLLABLE
   sortTable[0xB3C2]=35022;   // HANGUL SYLLABLE
   sortTable[0xB3C3]=35023;   // HANGUL SYLLABLE
   sortTable[0xB3C4]=35024;   // HANGUL SYLLABLE
   sortTable[0xB3C5]=35025;   // HANGUL SYLLABLE
   sortTable[0xB3C6]=35026;   // HANGUL SYLLABLE
   sortTable[0xB3C7]=35027;   // HANGUL SYLLABLE
   sortTable[0xB3C8]=35028;   // HANGUL SYLLABLE
   sortTable[0xB3C9]=35029;   // HANGUL SYLLABLE
   sortTable[0xB3CA]=35030;   // HANGUL SYLLABLE
   sortTable[0xB3CB]=35031;   // HANGUL SYLLABLE
   sortTable[0xB3CC]=35032;   // HANGUL SYLLABLE
   sortTable[0xB3CD]=35033;   // HANGUL SYLLABLE
   sortTable[0xB3CE]=35034;   // HANGUL SYLLABLE
   sortTable[0xB3CF]=35035;   // HANGUL SYLLABLE
   sortTable[0xB3D0]=35036;   // HANGUL SYLLABLE
   sortTable[0xB3D1]=35037;   // HANGUL SYLLABLE
   sortTable[0xB3D2]=35038;   // HANGUL SYLLABLE
   sortTable[0xB3D3]=35039;   // HANGUL SYLLABLE
   sortTable[0xB3D4]=35040;   // HANGUL SYLLABLE
   sortTable[0xB3D5]=35041;   // HANGUL SYLLABLE
   sortTable[0xB3D6]=35042;   // HANGUL SYLLABLE
   sortTable[0xB3D7]=35043;   // HANGUL SYLLABLE
   sortTable[0xB3D8]=35044;   // HANGUL SYLLABLE
   sortTable[0xB3D9]=35045;   // HANGUL SYLLABLE
   sortTable[0xB3DA]=35046;   // HANGUL SYLLABLE
   sortTable[0xB3DB]=35047;   // HANGUL SYLLABLE
   sortTable[0xB3DC]=35048;   // HANGUL SYLLABLE
   sortTable[0xB3DD]=35049;   // HANGUL SYLLABLE
   sortTable[0xB3DE]=35050;   // HANGUL SYLLABLE
   sortTable[0xB3DF]=35051;   // HANGUL SYLLABLE
   sortTable[0xB3E0]=35052;   // HANGUL SYLLABLE
   sortTable[0xB3E1]=35053;   // HANGUL SYLLABLE
   sortTable[0xB3E2]=35054;   // HANGUL SYLLABLE
   sortTable[0xB3E3]=35055;   // HANGUL SYLLABLE
   sortTable[0xB3E4]=35056;   // HANGUL SYLLABLE
   sortTable[0xB3E5]=35057;   // HANGUL SYLLABLE
   sortTable[0xB3E6]=35058;   // HANGUL SYLLABLE
   sortTable[0xB3E7]=35059;   // HANGUL SYLLABLE
   sortTable[0xB3E8]=35060;   // HANGUL SYLLABLE
   sortTable[0xB3E9]=35061;   // HANGUL SYLLABLE
   sortTable[0xB3EA]=35062;   // HANGUL SYLLABLE
   sortTable[0xB3EB]=35063;   // HANGUL SYLLABLE
   sortTable[0xB3EC]=35064;   // HANGUL SYLLABLE
   sortTable[0xB3ED]=35065;   // HANGUL SYLLABLE
   sortTable[0xB3EE]=35066;   // HANGUL SYLLABLE
   sortTable[0xB3EF]=35067;   // HANGUL SYLLABLE
   sortTable[0xB3F0]=35068;   // HANGUL SYLLABLE
   sortTable[0xB3F1]=35069;   // HANGUL SYLLABLE
   sortTable[0xB3F2]=35070;   // HANGUL SYLLABLE
   sortTable[0xB3F3]=35071;   // HANGUL SYLLABLE
   sortTable[0xB3F4]=35072;   // HANGUL SYLLABLE
   sortTable[0xB3F5]=35073;   // HANGUL SYLLABLE
   sortTable[0xB3F6]=35074;   // HANGUL SYLLABLE
   sortTable[0xB3F7]=35075;   // HANGUL SYLLABLE
   sortTable[0xB3F8]=35076;   // HANGUL SYLLABLE
   sortTable[0xB3F9]=35077;   // HANGUL SYLLABLE
   sortTable[0xB3FA]=35078;   // HANGUL SYLLABLE
   sortTable[0xB3FB]=35079;   // HANGUL SYLLABLE
   sortTable[0xB3FC]=35080;   // HANGUL SYLLABLE
   sortTable[0xB3FD]=35081;   // HANGUL SYLLABLE
   sortTable[0xB3FE]=35082;   // HANGUL SYLLABLE
   sortTable[0xB3FF]=35083;   // HANGUL SYLLABLE
   sortTable[0xB400]=35084;   // HANGUL SYLLABLE
   sortTable[0xB401]=35085;   // HANGUL SYLLABLE
   sortTable[0xB402]=35086;   // HANGUL SYLLABLE
   sortTable[0xB403]=35087;   // HANGUL SYLLABLE
   sortTable[0xB404]=35088;   // HANGUL SYLLABLE
   sortTable[0xB405]=35089;   // HANGUL SYLLABLE
   sortTable[0xB406]=35090;   // HANGUL SYLLABLE
   sortTable[0xB407]=35091;   // HANGUL SYLLABLE
   sortTable[0xB408]=35092;   // HANGUL SYLLABLE
   sortTable[0xB409]=35093;   // HANGUL SYLLABLE
   sortTable[0xB40A]=35094;   // HANGUL SYLLABLE
   sortTable[0xB40B]=35095;   // HANGUL SYLLABLE
   sortTable[0xB40C]=35096;   // HANGUL SYLLABLE
   sortTable[0xB40D]=35097;   // HANGUL SYLLABLE
   sortTable[0xB40E]=35098;   // HANGUL SYLLABLE
   sortTable[0xB40F]=35099;   // HANGUL SYLLABLE
   sortTable[0xB410]=35100;   // HANGUL SYLLABLE
   sortTable[0xB411]=35101;   // HANGUL SYLLABLE
   sortTable[0xB412]=35102;   // HANGUL SYLLABLE
   sortTable[0xB413]=35103;   // HANGUL SYLLABLE
   sortTable[0xB414]=35104;   // HANGUL SYLLABLE
   sortTable[0xB415]=35105;   // HANGUL SYLLABLE
   sortTable[0xB416]=35106;   // HANGUL SYLLABLE
   sortTable[0xB417]=35107;   // HANGUL SYLLABLE
   sortTable[0xB418]=35108;   // HANGUL SYLLABLE
   sortTable[0xB419]=35109;   // HANGUL SYLLABLE
   sortTable[0xB41A]=35110;   // HANGUL SYLLABLE
   sortTable[0xB41B]=35111;   // HANGUL SYLLABLE
   sortTable[0xB41C]=35112;   // HANGUL SYLLABLE
   sortTable[0xB41D]=35113;   // HANGUL SYLLABLE
   sortTable[0xB41E]=35114;   // HANGUL SYLLABLE
   sortTable[0xB41F]=35115;   // HANGUL SYLLABLE
   sortTable[0xB420]=35116;   // HANGUL SYLLABLE
   sortTable[0xB421]=35117;   // HANGUL SYLLABLE
   sortTable[0xB422]=35118;   // HANGUL SYLLABLE
   sortTable[0xB423]=35119;   // HANGUL SYLLABLE
   sortTable[0xB424]=35120;   // HANGUL SYLLABLE
   sortTable[0xB425]=35121;   // HANGUL SYLLABLE
   sortTable[0xB426]=35122;   // HANGUL SYLLABLE
   sortTable[0xB427]=35123;   // HANGUL SYLLABLE
   sortTable[0xB428]=35124;   // HANGUL SYLLABLE
   sortTable[0xB429]=35125;   // HANGUL SYLLABLE
   sortTable[0xB42A]=35126;   // HANGUL SYLLABLE
   sortTable[0xB42B]=35127;   // HANGUL SYLLABLE
   sortTable[0xB42C]=35128;   // HANGUL SYLLABLE
   sortTable[0xB42D]=35129;   // HANGUL SYLLABLE
   sortTable[0xB42E]=35130;   // HANGUL SYLLABLE
   sortTable[0xB42F]=35131;   // HANGUL SYLLABLE
   sortTable[0xB430]=35132;   // HANGUL SYLLABLE
   sortTable[0xB431]=35133;   // HANGUL SYLLABLE
   sortTable[0xB432]=35134;   // HANGUL SYLLABLE
   sortTable[0xB433]=35135;   // HANGUL SYLLABLE
   sortTable[0xB434]=35136;   // HANGUL SYLLABLE
   sortTable[0xB435]=35137;   // HANGUL SYLLABLE
   sortTable[0xB436]=35138;   // HANGUL SYLLABLE
   sortTable[0xB437]=35139;   // HANGUL SYLLABLE
   sortTable[0xB438]=35140;   // HANGUL SYLLABLE
   sortTable[0xB439]=35141;   // HANGUL SYLLABLE
   sortTable[0xB43A]=35142;   // HANGUL SYLLABLE
   sortTable[0xB43B]=35143;   // HANGUL SYLLABLE
   sortTable[0xB43C]=35144;   // HANGUL SYLLABLE
   sortTable[0xB43D]=35145;   // HANGUL SYLLABLE
   sortTable[0xB43E]=35146;   // HANGUL SYLLABLE
   sortTable[0xB43F]=35147;   // HANGUL SYLLABLE
   sortTable[0xB440]=35148;   // HANGUL SYLLABLE
   sortTable[0xB441]=35149;   // HANGUL SYLLABLE
   sortTable[0xB442]=35150;   // HANGUL SYLLABLE
   sortTable[0xB443]=35151;   // HANGUL SYLLABLE
   sortTable[0xB444]=35152;   // HANGUL SYLLABLE
   sortTable[0xB445]=35153;   // HANGUL SYLLABLE
   sortTable[0xB446]=35154;   // HANGUL SYLLABLE
   sortTable[0xB447]=35155;   // HANGUL SYLLABLE
   sortTable[0xB448]=35156;   // HANGUL SYLLABLE
   sortTable[0xB449]=35157;   // HANGUL SYLLABLE
   sortTable[0xB44A]=35158;   // HANGUL SYLLABLE
   sortTable[0xB44B]=35159;   // HANGUL SYLLABLE
   sortTable[0xB44C]=35160;   // HANGUL SYLLABLE
   sortTable[0xB44D]=35161;   // HANGUL SYLLABLE
   sortTable[0xB44E]=35162;   // HANGUL SYLLABLE
   sortTable[0xB44F]=35163;   // HANGUL SYLLABLE
   sortTable[0xB450]=35164;   // HANGUL SYLLABLE
   sortTable[0xB451]=35165;   // HANGUL SYLLABLE
   sortTable[0xB452]=35166;   // HANGUL SYLLABLE
   sortTable[0xB453]=35167;   // HANGUL SYLLABLE
   sortTable[0xB454]=35168;   // HANGUL SYLLABLE
   sortTable[0xB455]=35169;   // HANGUL SYLLABLE
   sortTable[0xB456]=35170;   // HANGUL SYLLABLE
   sortTable[0xB457]=35171;   // HANGUL SYLLABLE
   sortTable[0xB458]=35172;   // HANGUL SYLLABLE
   sortTable[0xB459]=35173;   // HANGUL SYLLABLE
   sortTable[0xB45A]=35174;   // HANGUL SYLLABLE
   sortTable[0xB45B]=35175;   // HANGUL SYLLABLE
   sortTable[0xB45C]=35176;   // HANGUL SYLLABLE
   sortTable[0xB45D]=35177;   // HANGUL SYLLABLE
   sortTable[0xB45E]=35178;   // HANGUL SYLLABLE
   sortTable[0xB45F]=35179;   // HANGUL SYLLABLE
   sortTable[0xB460]=35180;   // HANGUL SYLLABLE
   sortTable[0xB461]=35181;   // HANGUL SYLLABLE
   sortTable[0xB462]=35182;   // HANGUL SYLLABLE
   sortTable[0xB463]=35183;   // HANGUL SYLLABLE
   sortTable[0xB464]=35184;   // HANGUL SYLLABLE
   sortTable[0xB465]=35185;   // HANGUL SYLLABLE
   sortTable[0xB466]=35186;   // HANGUL SYLLABLE
   sortTable[0xB467]=35187;   // HANGUL SYLLABLE
   sortTable[0xB468]=35188;   // HANGUL SYLLABLE
   sortTable[0xB469]=35189;   // HANGUL SYLLABLE
   sortTable[0xB46A]=35190;   // HANGUL SYLLABLE
   sortTable[0xB46B]=35191;   // HANGUL SYLLABLE
   sortTable[0xB46C]=35192;   // HANGUL SYLLABLE
   sortTable[0xB46D]=35193;   // HANGUL SYLLABLE
   sortTable[0xB46E]=35194;   // HANGUL SYLLABLE
   sortTable[0xB46F]=35195;   // HANGUL SYLLABLE
   sortTable[0xB470]=35196;   // HANGUL SYLLABLE
   sortTable[0xB471]=35197;   // HANGUL SYLLABLE
   sortTable[0xB472]=35198;   // HANGUL SYLLABLE
   sortTable[0xB473]=35199;   // HANGUL SYLLABLE
   sortTable[0xB474]=35200;   // HANGUL SYLLABLE
   sortTable[0xB475]=35201;   // HANGUL SYLLABLE
   sortTable[0xB476]=35202;   // HANGUL SYLLABLE
   sortTable[0xB477]=35203;   // HANGUL SYLLABLE
   sortTable[0xB478]=35204;   // HANGUL SYLLABLE
   sortTable[0xB479]=35205;   // HANGUL SYLLABLE
   sortTable[0xB47A]=35206;   // HANGUL SYLLABLE
   sortTable[0xB47B]=35207;   // HANGUL SYLLABLE
   sortTable[0xB47C]=35208;   // HANGUL SYLLABLE
   sortTable[0xB47D]=35209;   // HANGUL SYLLABLE
   sortTable[0xB47E]=35210;   // HANGUL SYLLABLE
   sortTable[0xB47F]=35211;   // HANGUL SYLLABLE
   sortTable[0xB480]=35212;   // HANGUL SYLLABLE
   sortTable[0xB481]=35213;   // HANGUL SYLLABLE
   sortTable[0xB482]=35214;   // HANGUL SYLLABLE
   sortTable[0xB483]=35215;   // HANGUL SYLLABLE
   sortTable[0xB484]=35216;   // HANGUL SYLLABLE
   sortTable[0xB485]=35217;   // HANGUL SYLLABLE
   sortTable[0xB486]=35218;   // HANGUL SYLLABLE
   sortTable[0xB487]=35219;   // HANGUL SYLLABLE
   sortTable[0xB488]=35220;   // HANGUL SYLLABLE
   sortTable[0xB489]=35221;   // HANGUL SYLLABLE
   sortTable[0xB48A]=35222;   // HANGUL SYLLABLE
   sortTable[0xB48B]=35223;   // HANGUL SYLLABLE
   sortTable[0xB48C]=35224;   // HANGUL SYLLABLE
   sortTable[0xB48D]=35225;   // HANGUL SYLLABLE
   sortTable[0xB48E]=35226;   // HANGUL SYLLABLE
   sortTable[0xB48F]=35227;   // HANGUL SYLLABLE
   sortTable[0xB490]=35228;   // HANGUL SYLLABLE
   sortTable[0xB491]=35229;   // HANGUL SYLLABLE
   sortTable[0xB492]=35230;   // HANGUL SYLLABLE
   sortTable[0xB493]=35231;   // HANGUL SYLLABLE
   sortTable[0xB494]=35232;   // HANGUL SYLLABLE
   sortTable[0xB495]=35233;   // HANGUL SYLLABLE
   sortTable[0xB496]=35234;   // HANGUL SYLLABLE
   sortTable[0xB497]=35235;   // HANGUL SYLLABLE
   sortTable[0xB498]=35236;   // HANGUL SYLLABLE
   sortTable[0xB499]=35237;   // HANGUL SYLLABLE
   sortTable[0xB49A]=35238;   // HANGUL SYLLABLE
   sortTable[0xB49B]=35239;   // HANGUL SYLLABLE
   sortTable[0xB49C]=35240;   // HANGUL SYLLABLE
   sortTable[0xB49D]=35241;   // HANGUL SYLLABLE
   sortTable[0xB49E]=35242;   // HANGUL SYLLABLE
   sortTable[0xB49F]=35243;   // HANGUL SYLLABLE
   sortTable[0xB4A0]=35244;   // HANGUL SYLLABLE
   sortTable[0xB4A1]=35245;   // HANGUL SYLLABLE
   sortTable[0xB4A2]=35246;   // HANGUL SYLLABLE
   sortTable[0xB4A3]=35247;   // HANGUL SYLLABLE
   sortTable[0xB4A4]=35248;   // HANGUL SYLLABLE
   sortTable[0xB4A5]=35249;   // HANGUL SYLLABLE
   sortTable[0xB4A6]=35250;   // HANGUL SYLLABLE
   sortTable[0xB4A7]=35251;   // HANGUL SYLLABLE
   sortTable[0xB4A8]=35252;   // HANGUL SYLLABLE
   sortTable[0xB4A9]=35253;   // HANGUL SYLLABLE
   sortTable[0xB4AA]=35254;   // HANGUL SYLLABLE
   sortTable[0xB4AB]=35255;   // HANGUL SYLLABLE
   sortTable[0xB4AC]=35256;   // HANGUL SYLLABLE
   sortTable[0xB4AD]=35257;   // HANGUL SYLLABLE
   sortTable[0xB4AE]=35258;   // HANGUL SYLLABLE
   sortTable[0xB4AF]=35259;   // HANGUL SYLLABLE
   sortTable[0xB4B0]=35260;   // HANGUL SYLLABLE
   sortTable[0xB4B1]=35261;   // HANGUL SYLLABLE
   sortTable[0xB4B2]=35262;   // HANGUL SYLLABLE
   sortTable[0xB4B3]=35263;   // HANGUL SYLLABLE
   sortTable[0xB4B4]=35264;   // HANGUL SYLLABLE
   sortTable[0xB4B5]=35265;   // HANGUL SYLLABLE
   sortTable[0xB4B6]=35266;   // HANGUL SYLLABLE
   sortTable[0xB4B7]=35267;   // HANGUL SYLLABLE
   sortTable[0xB4B8]=35268;   // HANGUL SYLLABLE
   sortTable[0xB4B9]=35269;   // HANGUL SYLLABLE
   sortTable[0xB4BA]=35270;   // HANGUL SYLLABLE
   sortTable[0xB4BB]=35271;   // HANGUL SYLLABLE
   sortTable[0xB4BC]=35272;   // HANGUL SYLLABLE
   sortTable[0xB4BD]=35273;   // HANGUL SYLLABLE
   sortTable[0xB4BE]=35274;   // HANGUL SYLLABLE
   sortTable[0xB4BF]=35275;   // HANGUL SYLLABLE
   sortTable[0xB4C0]=35276;   // HANGUL SYLLABLE
   sortTable[0xB4C1]=35277;   // HANGUL SYLLABLE
   sortTable[0xB4C2]=35278;   // HANGUL SYLLABLE
   sortTable[0xB4C3]=35279;   // HANGUL SYLLABLE
   sortTable[0xB4C4]=35280;   // HANGUL SYLLABLE
   sortTable[0xB4C5]=35281;   // HANGUL SYLLABLE
   sortTable[0xB4C6]=35282;   // HANGUL SYLLABLE
   sortTable[0xB4C7]=35283;   // HANGUL SYLLABLE
   sortTable[0xB4C8]=35284;   // HANGUL SYLLABLE
   sortTable[0xB4C9]=35285;   // HANGUL SYLLABLE
   sortTable[0xB4CA]=35286;   // HANGUL SYLLABLE
   sortTable[0xB4CB]=35287;   // HANGUL SYLLABLE
   sortTable[0xB4CC]=35288;   // HANGUL SYLLABLE
   sortTable[0xB4CD]=35289;   // HANGUL SYLLABLE
   sortTable[0xB4CE]=35290;   // HANGUL SYLLABLE
   sortTable[0xB4CF]=35291;   // HANGUL SYLLABLE
   sortTable[0xB4D0]=35292;   // HANGUL SYLLABLE
   sortTable[0xB4D1]=35293;   // HANGUL SYLLABLE
   sortTable[0xB4D2]=35294;   // HANGUL SYLLABLE
   sortTable[0xB4D3]=35295;   // HANGUL SYLLABLE
   sortTable[0xB4D4]=35296;   // HANGUL SYLLABLE
   sortTable[0xB4D5]=35297;   // HANGUL SYLLABLE
   sortTable[0xB4D6]=35298;   // HANGUL SYLLABLE
   sortTable[0xB4D7]=35299;   // HANGUL SYLLABLE
   sortTable[0xB4D8]=35300;   // HANGUL SYLLABLE
   sortTable[0xB4D9]=35301;   // HANGUL SYLLABLE
   sortTable[0xB4DA]=35302;   // HANGUL SYLLABLE
   sortTable[0xB4DB]=35303;   // HANGUL SYLLABLE
   sortTable[0xB4DC]=35304;   // HANGUL SYLLABLE
   sortTable[0xB4DD]=35305;   // HANGUL SYLLABLE
   sortTable[0xB4DE]=35306;   // HANGUL SYLLABLE
   sortTable[0xB4DF]=35307;   // HANGUL SYLLABLE
   sortTable[0xB4E0]=35308;   // HANGUL SYLLABLE
   sortTable[0xB4E1]=35309;   // HANGUL SYLLABLE
   sortTable[0xB4E2]=35310;   // HANGUL SYLLABLE
   sortTable[0xB4E3]=35311;   // HANGUL SYLLABLE
   sortTable[0xB4E4]=35312;   // HANGUL SYLLABLE
   sortTable[0xB4E5]=35313;   // HANGUL SYLLABLE
   sortTable[0xB4E6]=35314;   // HANGUL SYLLABLE
   sortTable[0xB4E7]=35315;   // HANGUL SYLLABLE
   sortTable[0xB4E8]=35316;   // HANGUL SYLLABLE
   sortTable[0xB4E9]=35317;   // HANGUL SYLLABLE
   sortTable[0xB4EA]=35318;   // HANGUL SYLLABLE
   sortTable[0xB4EB]=35319;   // HANGUL SYLLABLE
   sortTable[0xB4EC]=35320;   // HANGUL SYLLABLE
   sortTable[0xB4ED]=35321;   // HANGUL SYLLABLE
   sortTable[0xB4EE]=35322;   // HANGUL SYLLABLE
   sortTable[0xB4EF]=35323;   // HANGUL SYLLABLE
   sortTable[0xB4F0]=35324;   // HANGUL SYLLABLE
   sortTable[0xB4F1]=35325;   // HANGUL SYLLABLE
   sortTable[0xB4F2]=35326;   // HANGUL SYLLABLE
   sortTable[0xB4F3]=35327;   // HANGUL SYLLABLE
   sortTable[0xB4F4]=35328;   // HANGUL SYLLABLE
   sortTable[0xB4F5]=35329;   // HANGUL SYLLABLE
   sortTable[0xB4F6]=35330;   // HANGUL SYLLABLE
   sortTable[0xB4F7]=35331;   // HANGUL SYLLABLE
   sortTable[0xB4F8]=35332;   // HANGUL SYLLABLE
   sortTable[0xB4F9]=35333;   // HANGUL SYLLABLE
   sortTable[0xB4FA]=35334;   // HANGUL SYLLABLE
   sortTable[0xB4FB]=35335;   // HANGUL SYLLABLE
   sortTable[0xB4FC]=35336;   // HANGUL SYLLABLE
   sortTable[0xB4FD]=35337;   // HANGUL SYLLABLE
   sortTable[0xB4FE]=35338;   // HANGUL SYLLABLE
   sortTable[0xB4FF]=35339;   // HANGUL SYLLABLE
   sortTable[0xB500]=35340;   // HANGUL SYLLABLE
   sortTable[0xB501]=35341;   // HANGUL SYLLABLE
   sortTable[0xB502]=35342;   // HANGUL SYLLABLE
   sortTable[0xB503]=35343;   // HANGUL SYLLABLE
   sortTable[0xB504]=35344;   // HANGUL SYLLABLE
   sortTable[0xB505]=35345;   // HANGUL SYLLABLE
   sortTable[0xB506]=35346;   // HANGUL SYLLABLE
   sortTable[0xB507]=35347;   // HANGUL SYLLABLE
   sortTable[0xB508]=35348;   // HANGUL SYLLABLE
   sortTable[0xB509]=35349;   // HANGUL SYLLABLE
   sortTable[0xB50A]=35350;   // HANGUL SYLLABLE
   sortTable[0xB50B]=35351;   // HANGUL SYLLABLE
   sortTable[0xB50C]=35352;   // HANGUL SYLLABLE
   sortTable[0xB50D]=35353;   // HANGUL SYLLABLE
   sortTable[0xB50E]=35354;   // HANGUL SYLLABLE
   sortTable[0xB50F]=35355;   // HANGUL SYLLABLE
   sortTable[0xB510]=35356;   // HANGUL SYLLABLE
   sortTable[0xB511]=35357;   // HANGUL SYLLABLE
   sortTable[0xB512]=35358;   // HANGUL SYLLABLE
   sortTable[0xB513]=35359;   // HANGUL SYLLABLE
   sortTable[0xB514]=35360;   // HANGUL SYLLABLE
   sortTable[0xB515]=35361;   // HANGUL SYLLABLE
   sortTable[0xB516]=35362;   // HANGUL SYLLABLE
   sortTable[0xB517]=35363;   // HANGUL SYLLABLE
   sortTable[0xB518]=35364;   // HANGUL SYLLABLE
   sortTable[0xB519]=35365;   // HANGUL SYLLABLE
   sortTable[0xB51A]=35366;   // HANGUL SYLLABLE
   sortTable[0xB51B]=35367;   // HANGUL SYLLABLE
   sortTable[0xB51C]=35368;   // HANGUL SYLLABLE
   sortTable[0xB51D]=35369;   // HANGUL SYLLABLE
   sortTable[0xB51E]=35370;   // HANGUL SYLLABLE
   sortTable[0xB51F]=35371;   // HANGUL SYLLABLE
   sortTable[0xB520]=35372;   // HANGUL SYLLABLE
   sortTable[0xB521]=35373;   // HANGUL SYLLABLE
   sortTable[0xB522]=35374;   // HANGUL SYLLABLE
   sortTable[0xB523]=35375;   // HANGUL SYLLABLE
   sortTable[0xB524]=35376;   // HANGUL SYLLABLE
   sortTable[0xB525]=35377;   // HANGUL SYLLABLE
   sortTable[0xB526]=35378;   // HANGUL SYLLABLE
   sortTable[0xB527]=35379;   // HANGUL SYLLABLE
   sortTable[0xB528]=35380;   // HANGUL SYLLABLE
   sortTable[0xB529]=35381;   // HANGUL SYLLABLE
   sortTable[0xB52A]=35382;   // HANGUL SYLLABLE
   sortTable[0xB52B]=35383;   // HANGUL SYLLABLE
   sortTable[0xB52C]=35384;   // HANGUL SYLLABLE
   sortTable[0xB52D]=35385;   // HANGUL SYLLABLE
   sortTable[0xB52E]=35386;   // HANGUL SYLLABLE
   sortTable[0xB52F]=35387;   // HANGUL SYLLABLE
   sortTable[0x3138]=35388;   // HANGUL LETTER SSANGTIKEUT
   sortTable[0x1104]=35389;   // HANGUL CHOSEONG SSANGTIKEUT
   sortTable[0xB530]=35390;   // HANGUL SYLLABLE
   sortTable[0xB531]=35391;   // HANGUL SYLLABLE
   sortTable[0xB532]=35392;   // HANGUL SYLLABLE
   sortTable[0xB533]=35393;   // HANGUL SYLLABLE
   sortTable[0xB534]=35394;   // HANGUL SYLLABLE
   sortTable[0xB535]=35395;   // HANGUL SYLLABLE
   sortTable[0xB536]=35396;   // HANGUL SYLLABLE
   sortTable[0xB537]=35397;   // HANGUL SYLLABLE
   sortTable[0xB538]=35398;   // HANGUL SYLLABLE
   sortTable[0xB539]=35399;   // HANGUL SYLLABLE
   sortTable[0xB53A]=35400;   // HANGUL SYLLABLE
   sortTable[0xB53B]=35401;   // HANGUL SYLLABLE
   sortTable[0xB53C]=35402;   // HANGUL SYLLABLE
   sortTable[0xB53D]=35403;   // HANGUL SYLLABLE
   sortTable[0xB53E]=35404;   // HANGUL SYLLABLE
   sortTable[0xB53F]=35405;   // HANGUL SYLLABLE
   sortTable[0xB540]=35406;   // HANGUL SYLLABLE
   sortTable[0xB541]=35407;   // HANGUL SYLLABLE
   sortTable[0xB542]=35408;   // HANGUL SYLLABLE
   sortTable[0xB543]=35409;   // HANGUL SYLLABLE
   sortTable[0xB544]=35410;   // HANGUL SYLLABLE
   sortTable[0xB545]=35411;   // HANGUL SYLLABLE
   sortTable[0xB546]=35412;   // HANGUL SYLLABLE
   sortTable[0xB547]=35413;   // HANGUL SYLLABLE
   sortTable[0xB548]=35414;   // HANGUL SYLLABLE
   sortTable[0xB549]=35415;   // HANGUL SYLLABLE
   sortTable[0xB54A]=35416;   // HANGUL SYLLABLE
   sortTable[0xB54B]=35417;   // HANGUL SYLLABLE
   sortTable[0xB54C]=35418;   // HANGUL SYLLABLE
   sortTable[0xB54D]=35419;   // HANGUL SYLLABLE
   sortTable[0xB54E]=35420;   // HANGUL SYLLABLE
   sortTable[0xB54F]=35421;   // HANGUL SYLLABLE
   sortTable[0xB550]=35422;   // HANGUL SYLLABLE
   sortTable[0xB551]=35423;   // HANGUL SYLLABLE
   sortTable[0xB552]=35424;   // HANGUL SYLLABLE
   sortTable[0xB553]=35425;   // HANGUL SYLLABLE
   sortTable[0xB554]=35426;   // HANGUL SYLLABLE
   sortTable[0xB555]=35427;   // HANGUL SYLLABLE
   sortTable[0xB556]=35428;   // HANGUL SYLLABLE
   sortTable[0xB557]=35429;   // HANGUL SYLLABLE
   sortTable[0xB558]=35430;   // HANGUL SYLLABLE
   sortTable[0xB559]=35431;   // HANGUL SYLLABLE
   sortTable[0xB55A]=35432;   // HANGUL SYLLABLE
   sortTable[0xB55B]=35433;   // HANGUL SYLLABLE
   sortTable[0xB55C]=35434;   // HANGUL SYLLABLE
   sortTable[0xB55D]=35435;   // HANGUL SYLLABLE
   sortTable[0xB55E]=35436;   // HANGUL SYLLABLE
   sortTable[0xB55F]=35437;   // HANGUL SYLLABLE
   sortTable[0xB560]=35438;   // HANGUL SYLLABLE
   sortTable[0xB561]=35439;   // HANGUL SYLLABLE
   sortTable[0xB562]=35440;   // HANGUL SYLLABLE
   sortTable[0xB563]=35441;   // HANGUL SYLLABLE
   sortTable[0xB564]=35442;   // HANGUL SYLLABLE
   sortTable[0xB565]=35443;   // HANGUL SYLLABLE
   sortTable[0xB566]=35444;   // HANGUL SYLLABLE
   sortTable[0xB567]=35445;   // HANGUL SYLLABLE
   sortTable[0xB568]=35446;   // HANGUL SYLLABLE
   sortTable[0xB569]=35447;   // HANGUL SYLLABLE
   sortTable[0xB56A]=35448;   // HANGUL SYLLABLE
   sortTable[0xB56B]=35449;   // HANGUL SYLLABLE
   sortTable[0xB56C]=35450;   // HANGUL SYLLABLE
   sortTable[0xB56D]=35451;   // HANGUL SYLLABLE
   sortTable[0xB56E]=35452;   // HANGUL SYLLABLE
   sortTable[0xB56F]=35453;   // HANGUL SYLLABLE
   sortTable[0xB570]=35454;   // HANGUL SYLLABLE
   sortTable[0xB571]=35455;   // HANGUL SYLLABLE
   sortTable[0xB572]=35456;   // HANGUL SYLLABLE
   sortTable[0xB573]=35457;   // HANGUL SYLLABLE
   sortTable[0xB574]=35458;   // HANGUL SYLLABLE
   sortTable[0xB575]=35459;   // HANGUL SYLLABLE
   sortTable[0xB576]=35460;   // HANGUL SYLLABLE
   sortTable[0xB577]=35461;   // HANGUL SYLLABLE
   sortTable[0xB578]=35462;   // HANGUL SYLLABLE
   sortTable[0xB579]=35463;   // HANGUL SYLLABLE
   sortTable[0xB57A]=35464;   // HANGUL SYLLABLE
   sortTable[0xB57B]=35465;   // HANGUL SYLLABLE
   sortTable[0xB57C]=35466;   // HANGUL SYLLABLE
   sortTable[0xB57D]=35467;   // HANGUL SYLLABLE
   sortTable[0xB57E]=35468;   // HANGUL SYLLABLE
   sortTable[0xB57F]=35469;   // HANGUL SYLLABLE
   sortTable[0xB580]=35470;   // HANGUL SYLLABLE
   sortTable[0xB581]=35471;   // HANGUL SYLLABLE
   sortTable[0xB582]=35472;   // HANGUL SYLLABLE
   sortTable[0xB583]=35473;   // HANGUL SYLLABLE
   sortTable[0xB584]=35474;   // HANGUL SYLLABLE
   sortTable[0xB585]=35475;   // HANGUL SYLLABLE
   sortTable[0xB586]=35476;   // HANGUL SYLLABLE
   sortTable[0xB587]=35477;   // HANGUL SYLLABLE
   sortTable[0xB588]=35478;   // HANGUL SYLLABLE
   sortTable[0xB589]=35479;   // HANGUL SYLLABLE
   sortTable[0xB58A]=35480;   // HANGUL SYLLABLE
   sortTable[0xB58B]=35481;   // HANGUL SYLLABLE
   sortTable[0xB58C]=35482;   // HANGUL SYLLABLE
   sortTable[0xB58D]=35483;   // HANGUL SYLLABLE
   sortTable[0xB58E]=35484;   // HANGUL SYLLABLE
   sortTable[0xB58F]=35485;   // HANGUL SYLLABLE
   sortTable[0xB590]=35486;   // HANGUL SYLLABLE
   sortTable[0xB591]=35487;   // HANGUL SYLLABLE
   sortTable[0xB592]=35488;   // HANGUL SYLLABLE
   sortTable[0xB593]=35489;   // HANGUL SYLLABLE
   sortTable[0xB594]=35490;   // HANGUL SYLLABLE
   sortTable[0xB595]=35491;   // HANGUL SYLLABLE
   sortTable[0xB596]=35492;   // HANGUL SYLLABLE
   sortTable[0xB597]=35493;   // HANGUL SYLLABLE
   sortTable[0xB598]=35494;   // HANGUL SYLLABLE
   sortTable[0xB599]=35495;   // HANGUL SYLLABLE
   sortTable[0xB59A]=35496;   // HANGUL SYLLABLE
   sortTable[0xB59B]=35497;   // HANGUL SYLLABLE
   sortTable[0xB59C]=35498;   // HANGUL SYLLABLE
   sortTable[0xB59D]=35499;   // HANGUL SYLLABLE
   sortTable[0xB59E]=35500;   // HANGUL SYLLABLE
   sortTable[0xB59F]=35501;   // HANGUL SYLLABLE
   sortTable[0xB5A0]=35502;   // HANGUL SYLLABLE
   sortTable[0xB5A1]=35503;   // HANGUL SYLLABLE
   sortTable[0xB5A2]=35504;   // HANGUL SYLLABLE
   sortTable[0xB5A3]=35505;   // HANGUL SYLLABLE
   sortTable[0xB5A4]=35506;   // HANGUL SYLLABLE
   sortTable[0xB5A5]=35507;   // HANGUL SYLLABLE
   sortTable[0xB5A6]=35508;   // HANGUL SYLLABLE
   sortTable[0xB5A7]=35509;   // HANGUL SYLLABLE
   sortTable[0xB5A8]=35510;   // HANGUL SYLLABLE
   sortTable[0xB5A9]=35511;   // HANGUL SYLLABLE
   sortTable[0xB5AA]=35512;   // HANGUL SYLLABLE
   sortTable[0xB5AB]=35513;   // HANGUL SYLLABLE
   sortTable[0xB5AC]=35514;   // HANGUL SYLLABLE
   sortTable[0xB5AD]=35515;   // HANGUL SYLLABLE
   sortTable[0xB5AE]=35516;   // HANGUL SYLLABLE
   sortTable[0xB5AF]=35517;   // HANGUL SYLLABLE
   sortTable[0xB5B0]=35518;   // HANGUL SYLLABLE
   sortTable[0xB5B1]=35519;   // HANGUL SYLLABLE
   sortTable[0xB5B2]=35520;   // HANGUL SYLLABLE
   sortTable[0xB5B3]=35521;   // HANGUL SYLLABLE
   sortTable[0xB5B4]=35522;   // HANGUL SYLLABLE
   sortTable[0xB5B5]=35523;   // HANGUL SYLLABLE
   sortTable[0xB5B6]=35524;   // HANGUL SYLLABLE
   sortTable[0xB5B7]=35525;   // HANGUL SYLLABLE
   sortTable[0xB5B8]=35526;   // HANGUL SYLLABLE
   sortTable[0xB5B9]=35527;   // HANGUL SYLLABLE
   sortTable[0xB5BA]=35528;   // HANGUL SYLLABLE
   sortTable[0xB5BB]=35529;   // HANGUL SYLLABLE
   sortTable[0xB5BC]=35530;   // HANGUL SYLLABLE
   sortTable[0xB5BD]=35531;   // HANGUL SYLLABLE
   sortTable[0xB5BE]=35532;   // HANGUL SYLLABLE
   sortTable[0xB5BF]=35533;   // HANGUL SYLLABLE
   sortTable[0xB5C0]=35534;   // HANGUL SYLLABLE
   sortTable[0xB5C1]=35535;   // HANGUL SYLLABLE
   sortTable[0xB5C2]=35536;   // HANGUL SYLLABLE
   sortTable[0xB5C3]=35537;   // HANGUL SYLLABLE
   sortTable[0xB5C4]=35538;   // HANGUL SYLLABLE
   sortTable[0xB5C5]=35539;   // HANGUL SYLLABLE
   sortTable[0xB5C6]=35540;   // HANGUL SYLLABLE
   sortTable[0xB5C7]=35541;   // HANGUL SYLLABLE
   sortTable[0xB5C8]=35542;   // HANGUL SYLLABLE
   sortTable[0xB5C9]=35543;   // HANGUL SYLLABLE
   sortTable[0xB5CA]=35544;   // HANGUL SYLLABLE
   sortTable[0xB5CB]=35545;   // HANGUL SYLLABLE
   sortTable[0xB5CC]=35546;   // HANGUL SYLLABLE
   sortTable[0xB5CD]=35547;   // HANGUL SYLLABLE
   sortTable[0xB5CE]=35548;   // HANGUL SYLLABLE
   sortTable[0xB5CF]=35549;   // HANGUL SYLLABLE
   sortTable[0xB5D0]=35550;   // HANGUL SYLLABLE
   sortTable[0xB5D1]=35551;   // HANGUL SYLLABLE
   sortTable[0xB5D2]=35552;   // HANGUL SYLLABLE
   sortTable[0xB5D3]=35553;   // HANGUL SYLLABLE
   sortTable[0xB5D4]=35554;   // HANGUL SYLLABLE
   sortTable[0xB5D5]=35555;   // HANGUL SYLLABLE
   sortTable[0xB5D6]=35556;   // HANGUL SYLLABLE
   sortTable[0xB5D7]=35557;   // HANGUL SYLLABLE
   sortTable[0xB5D8]=35558;   // HANGUL SYLLABLE
   sortTable[0xB5D9]=35559;   // HANGUL SYLLABLE
   sortTable[0xB5DA]=35560;   // HANGUL SYLLABLE
   sortTable[0xB5DB]=35561;   // HANGUL SYLLABLE
   sortTable[0xB5DC]=35562;   // HANGUL SYLLABLE
   sortTable[0xB5DD]=35563;   // HANGUL SYLLABLE
   sortTable[0xB5DE]=35564;   // HANGUL SYLLABLE
   sortTable[0xB5DF]=35565;   // HANGUL SYLLABLE
   sortTable[0xB5E0]=35566;   // HANGUL SYLLABLE
   sortTable[0xB5E1]=35567;   // HANGUL SYLLABLE
   sortTable[0xB5E2]=35568;   // HANGUL SYLLABLE
   sortTable[0xB5E3]=35569;   // HANGUL SYLLABLE
   sortTable[0xB5E4]=35570;   // HANGUL SYLLABLE
   sortTable[0xB5E5]=35571;   // HANGUL SYLLABLE
   sortTable[0xB5E6]=35572;   // HANGUL SYLLABLE
   sortTable[0xB5E7]=35573;   // HANGUL SYLLABLE
   sortTable[0xB5E8]=35574;   // HANGUL SYLLABLE
   sortTable[0xB5E9]=35575;   // HANGUL SYLLABLE
   sortTable[0xB5EA]=35576;   // HANGUL SYLLABLE
   sortTable[0xB5EB]=35577;   // HANGUL SYLLABLE
   sortTable[0xB5EC]=35578;   // HANGUL SYLLABLE
   sortTable[0xB5ED]=35579;   // HANGUL SYLLABLE
   sortTable[0xB5EE]=35580;   // HANGUL SYLLABLE
   sortTable[0xB5EF]=35581;   // HANGUL SYLLABLE
   sortTable[0xB5F0]=35582;   // HANGUL SYLLABLE
   sortTable[0xB5F1]=35583;   // HANGUL SYLLABLE
   sortTable[0xB5F2]=35584;   // HANGUL SYLLABLE
   sortTable[0xB5F3]=35585;   // HANGUL SYLLABLE
   sortTable[0xB5F4]=35586;   // HANGUL SYLLABLE
   sortTable[0xB5F5]=35587;   // HANGUL SYLLABLE
   sortTable[0xB5F6]=35588;   // HANGUL SYLLABLE
   sortTable[0xB5F7]=35589;   // HANGUL SYLLABLE
   sortTable[0xB5F8]=35590;   // HANGUL SYLLABLE
   sortTable[0xB5F9]=35591;   // HANGUL SYLLABLE
   sortTable[0xB5FA]=35592;   // HANGUL SYLLABLE
   sortTable[0xB5FB]=35593;   // HANGUL SYLLABLE
   sortTable[0xB5FC]=35594;   // HANGUL SYLLABLE
   sortTable[0xB5FD]=35595;   // HANGUL SYLLABLE
   sortTable[0xB5FE]=35596;   // HANGUL SYLLABLE
   sortTable[0xB5FF]=35597;   // HANGUL SYLLABLE
   sortTable[0xB600]=35598;   // HANGUL SYLLABLE
   sortTable[0xB601]=35599;   // HANGUL SYLLABLE
   sortTable[0xB602]=35600;   // HANGUL SYLLABLE
   sortTable[0xB603]=35601;   // HANGUL SYLLABLE
   sortTable[0xB604]=35602;   // HANGUL SYLLABLE
   sortTable[0xB605]=35603;   // HANGUL SYLLABLE
   sortTable[0xB606]=35604;   // HANGUL SYLLABLE
   sortTable[0xB607]=35605;   // HANGUL SYLLABLE
   sortTable[0xB608]=35606;   // HANGUL SYLLABLE
   sortTable[0xB609]=35607;   // HANGUL SYLLABLE
   sortTable[0xB60A]=35608;   // HANGUL SYLLABLE
   sortTable[0xB60B]=35609;   // HANGUL SYLLABLE
   sortTable[0xB60C]=35610;   // HANGUL SYLLABLE
   sortTable[0xB60D]=35611;   // HANGUL SYLLABLE
   sortTable[0xB60E]=35612;   // HANGUL SYLLABLE
   sortTable[0xB60F]=35613;   // HANGUL SYLLABLE
   sortTable[0xB610]=35614;   // HANGUL SYLLABLE
   sortTable[0xB611]=35615;   // HANGUL SYLLABLE
   sortTable[0xB612]=35616;   // HANGUL SYLLABLE
   sortTable[0xB613]=35617;   // HANGUL SYLLABLE
   sortTable[0xB614]=35618;   // HANGUL SYLLABLE
   sortTable[0xB615]=35619;   // HANGUL SYLLABLE
   sortTable[0xB616]=35620;   // HANGUL SYLLABLE
   sortTable[0xB617]=35621;   // HANGUL SYLLABLE
   sortTable[0xB618]=35622;   // HANGUL SYLLABLE
   sortTable[0xB619]=35623;   // HANGUL SYLLABLE
   sortTable[0xB61A]=35624;   // HANGUL SYLLABLE
   sortTable[0xB61B]=35625;   // HANGUL SYLLABLE
   sortTable[0xB61C]=35626;   // HANGUL SYLLABLE
   sortTable[0xB61D]=35627;   // HANGUL SYLLABLE
   sortTable[0xB61E]=35628;   // HANGUL SYLLABLE
   sortTable[0xB61F]=35629;   // HANGUL SYLLABLE
   sortTable[0xB620]=35630;   // HANGUL SYLLABLE
   sortTable[0xB621]=35631;   // HANGUL SYLLABLE
   sortTable[0xB622]=35632;   // HANGUL SYLLABLE
   sortTable[0xB623]=35633;   // HANGUL SYLLABLE
   sortTable[0xB624]=35634;   // HANGUL SYLLABLE
   sortTable[0xB625]=35635;   // HANGUL SYLLABLE
   sortTable[0xB626]=35636;   // HANGUL SYLLABLE
   sortTable[0xB627]=35637;   // HANGUL SYLLABLE
   sortTable[0xB628]=35638;   // HANGUL SYLLABLE
   sortTable[0xB629]=35639;   // HANGUL SYLLABLE
   sortTable[0xB62A]=35640;   // HANGUL SYLLABLE
   sortTable[0xB62B]=35641;   // HANGUL SYLLABLE
   sortTable[0xB62C]=35642;   // HANGUL SYLLABLE
   sortTable[0xB62D]=35643;   // HANGUL SYLLABLE
   sortTable[0xB62E]=35644;   // HANGUL SYLLABLE
   sortTable[0xB62F]=35645;   // HANGUL SYLLABLE
   sortTable[0xB630]=35646;   // HANGUL SYLLABLE
   sortTable[0xB631]=35647;   // HANGUL SYLLABLE
   sortTable[0xB632]=35648;   // HANGUL SYLLABLE
   sortTable[0xB633]=35649;   // HANGUL SYLLABLE
   sortTable[0xB634]=35650;   // HANGUL SYLLABLE
   sortTable[0xB635]=35651;   // HANGUL SYLLABLE
   sortTable[0xB636]=35652;   // HANGUL SYLLABLE
   sortTable[0xB637]=35653;   // HANGUL SYLLABLE
   sortTable[0xB638]=35654;   // HANGUL SYLLABLE
   sortTable[0xB639]=35655;   // HANGUL SYLLABLE
   sortTable[0xB63A]=35656;   // HANGUL SYLLABLE
   sortTable[0xB63B]=35657;   // HANGUL SYLLABLE
   sortTable[0xB63C]=35658;   // HANGUL SYLLABLE
   sortTable[0xB63D]=35659;   // HANGUL SYLLABLE
   sortTable[0xB63E]=35660;   // HANGUL SYLLABLE
   sortTable[0xB63F]=35661;   // HANGUL SYLLABLE
   sortTable[0xB640]=35662;   // HANGUL SYLLABLE
   sortTable[0xB641]=35663;   // HANGUL SYLLABLE
   sortTable[0xB642]=35664;   // HANGUL SYLLABLE
   sortTable[0xB643]=35665;   // HANGUL SYLLABLE
   sortTable[0xB644]=35666;   // HANGUL SYLLABLE
   sortTable[0xB645]=35667;   // HANGUL SYLLABLE
   sortTable[0xB646]=35668;   // HANGUL SYLLABLE
   sortTable[0xB647]=35669;   // HANGUL SYLLABLE
   sortTable[0xB648]=35670;   // HANGUL SYLLABLE
   sortTable[0xB649]=35671;   // HANGUL SYLLABLE
   sortTable[0xB64A]=35672;   // HANGUL SYLLABLE
   sortTable[0xB64B]=35673;   // HANGUL SYLLABLE
   sortTable[0xB64C]=35674;   // HANGUL SYLLABLE
   sortTable[0xB64D]=35675;   // HANGUL SYLLABLE
   sortTable[0xB64E]=35676;   // HANGUL SYLLABLE
   sortTable[0xB64F]=35677;   // HANGUL SYLLABLE
   sortTable[0xB650]=35678;   // HANGUL SYLLABLE
   sortTable[0xB651]=35679;   // HANGUL SYLLABLE
   sortTable[0xB652]=35680;   // HANGUL SYLLABLE
   sortTable[0xB653]=35681;   // HANGUL SYLLABLE
   sortTable[0xB654]=35682;   // HANGUL SYLLABLE
   sortTable[0xB655]=35683;   // HANGUL SYLLABLE
   sortTable[0xB656]=35684;   // HANGUL SYLLABLE
   sortTable[0xB657]=35685;   // HANGUL SYLLABLE
   sortTable[0xB658]=35686;   // HANGUL SYLLABLE
   sortTable[0xB659]=35687;   // HANGUL SYLLABLE
   sortTable[0xB65A]=35688;   // HANGUL SYLLABLE
   sortTable[0xB65B]=35689;   // HANGUL SYLLABLE
   sortTable[0xB65C]=35690;   // HANGUL SYLLABLE
   sortTable[0xB65D]=35691;   // HANGUL SYLLABLE
   sortTable[0xB65E]=35692;   // HANGUL SYLLABLE
   sortTable[0xB65F]=35693;   // HANGUL SYLLABLE
   sortTable[0xB660]=35694;   // HANGUL SYLLABLE
   sortTable[0xB661]=35695;   // HANGUL SYLLABLE
   sortTable[0xB662]=35696;   // HANGUL SYLLABLE
   sortTable[0xB663]=35697;   // HANGUL SYLLABLE
   sortTable[0xB664]=35698;   // HANGUL SYLLABLE
   sortTable[0xB665]=35699;   // HANGUL SYLLABLE
   sortTable[0xB666]=35700;   // HANGUL SYLLABLE
   sortTable[0xB667]=35701;   // HANGUL SYLLABLE
   sortTable[0xB668]=35702;   // HANGUL SYLLABLE
   sortTable[0xB669]=35703;   // HANGUL SYLLABLE
   sortTable[0xB66A]=35704;   // HANGUL SYLLABLE
   sortTable[0xB66B]=35705;   // HANGUL SYLLABLE
   sortTable[0xB66C]=35706;   // HANGUL SYLLABLE
   sortTable[0xB66D]=35707;   // HANGUL SYLLABLE
   sortTable[0xB66E]=35708;   // HANGUL SYLLABLE
   sortTable[0xB66F]=35709;   // HANGUL SYLLABLE
   sortTable[0xB670]=35710;   // HANGUL SYLLABLE
   sortTable[0xB671]=35711;   // HANGUL SYLLABLE
   sortTable[0xB672]=35712;   // HANGUL SYLLABLE
   sortTable[0xB673]=35713;   // HANGUL SYLLABLE
   sortTable[0xB674]=35714;   // HANGUL SYLLABLE
   sortTable[0xB675]=35715;   // HANGUL SYLLABLE
   sortTable[0xB676]=35716;   // HANGUL SYLLABLE
   sortTable[0xB677]=35717;   // HANGUL SYLLABLE
   sortTable[0xB678]=35718;   // HANGUL SYLLABLE
   sortTable[0xB679]=35719;   // HANGUL SYLLABLE
   sortTable[0xB67A]=35720;   // HANGUL SYLLABLE
   sortTable[0xB67B]=35721;   // HANGUL SYLLABLE
   sortTable[0xB67C]=35722;   // HANGUL SYLLABLE
   sortTable[0xB67D]=35723;   // HANGUL SYLLABLE
   sortTable[0xB67E]=35724;   // HANGUL SYLLABLE
   sortTable[0xB67F]=35725;   // HANGUL SYLLABLE
   sortTable[0xB680]=35726;   // HANGUL SYLLABLE
   sortTable[0xB681]=35727;   // HANGUL SYLLABLE
   sortTable[0xB682]=35728;   // HANGUL SYLLABLE
   sortTable[0xB683]=35729;   // HANGUL SYLLABLE
   sortTable[0xB684]=35730;   // HANGUL SYLLABLE
   sortTable[0xB685]=35731;   // HANGUL SYLLABLE
   sortTable[0xB686]=35732;   // HANGUL SYLLABLE
   sortTable[0xB687]=35733;   // HANGUL SYLLABLE
   sortTable[0xB688]=35734;   // HANGUL SYLLABLE
   sortTable[0xB689]=35735;   // HANGUL SYLLABLE
   sortTable[0xB68A]=35736;   // HANGUL SYLLABLE
   sortTable[0xB68B]=35737;   // HANGUL SYLLABLE
   sortTable[0xB68C]=35738;   // HANGUL SYLLABLE
   sortTable[0xB68D]=35739;   // HANGUL SYLLABLE
   sortTable[0xB68E]=35740;   // HANGUL SYLLABLE
   sortTable[0xB68F]=35741;   // HANGUL SYLLABLE
   sortTable[0xB690]=35742;   // HANGUL SYLLABLE
   sortTable[0xB691]=35743;   // HANGUL SYLLABLE
   sortTable[0xB692]=35744;   // HANGUL SYLLABLE
   sortTable[0xB693]=35745;   // HANGUL SYLLABLE
   sortTable[0xB694]=35746;   // HANGUL SYLLABLE
   sortTable[0xB695]=35747;   // HANGUL SYLLABLE
   sortTable[0xB696]=35748;   // HANGUL SYLLABLE
   sortTable[0xB697]=35749;   // HANGUL SYLLABLE
   sortTable[0xB698]=35750;   // HANGUL SYLLABLE
   sortTable[0xB699]=35751;   // HANGUL SYLLABLE
   sortTable[0xB69A]=35752;   // HANGUL SYLLABLE
   sortTable[0xB69B]=35753;   // HANGUL SYLLABLE
   sortTable[0xB69C]=35754;   // HANGUL SYLLABLE
   sortTable[0xB69D]=35755;   // HANGUL SYLLABLE
   sortTable[0xB69E]=35756;   // HANGUL SYLLABLE
   sortTable[0xB69F]=35757;   // HANGUL SYLLABLE
   sortTable[0xB6A0]=35758;   // HANGUL SYLLABLE
   sortTable[0xB6A1]=35759;   // HANGUL SYLLABLE
   sortTable[0xB6A2]=35760;   // HANGUL SYLLABLE
   sortTable[0xB6A3]=35761;   // HANGUL SYLLABLE
   sortTable[0xB6A4]=35762;   // HANGUL SYLLABLE
   sortTable[0xB6A5]=35763;   // HANGUL SYLLABLE
   sortTable[0xB6A6]=35764;   // HANGUL SYLLABLE
   sortTable[0xB6A7]=35765;   // HANGUL SYLLABLE
   sortTable[0xB6A8]=35766;   // HANGUL SYLLABLE
   sortTable[0xB6A9]=35767;   // HANGUL SYLLABLE
   sortTable[0xB6AA]=35768;   // HANGUL SYLLABLE
   sortTable[0xB6AB]=35769;   // HANGUL SYLLABLE
   sortTable[0xB6AC]=35770;   // HANGUL SYLLABLE
   sortTable[0xB6AD]=35771;   // HANGUL SYLLABLE
   sortTable[0xB6AE]=35772;   // HANGUL SYLLABLE
   sortTable[0xB6AF]=35773;   // HANGUL SYLLABLE
   sortTable[0xB6B0]=35774;   // HANGUL SYLLABLE
   sortTable[0xB6B1]=35775;   // HANGUL SYLLABLE
   sortTable[0xB6B2]=35776;   // HANGUL SYLLABLE
   sortTable[0xB6B3]=35777;   // HANGUL SYLLABLE
   sortTable[0xB6B4]=35778;   // HANGUL SYLLABLE
   sortTable[0xB6B5]=35779;   // HANGUL SYLLABLE
   sortTable[0xB6B6]=35780;   // HANGUL SYLLABLE
   sortTable[0xB6B7]=35781;   // HANGUL SYLLABLE
   sortTable[0xB6B8]=35782;   // HANGUL SYLLABLE
   sortTable[0xB6B9]=35783;   // HANGUL SYLLABLE
   sortTable[0xB6BA]=35784;   // HANGUL SYLLABLE
   sortTable[0xB6BB]=35785;   // HANGUL SYLLABLE
   sortTable[0xB6BC]=35786;   // HANGUL SYLLABLE
   sortTable[0xB6BD]=35787;   // HANGUL SYLLABLE
   sortTable[0xB6BE]=35788;   // HANGUL SYLLABLE
   sortTable[0xB6BF]=35789;   // HANGUL SYLLABLE
   sortTable[0xB6C0]=35790;   // HANGUL SYLLABLE
   sortTable[0xB6C1]=35791;   // HANGUL SYLLABLE
   sortTable[0xB6C2]=35792;   // HANGUL SYLLABLE
   sortTable[0xB6C3]=35793;   // HANGUL SYLLABLE
   sortTable[0xB6C4]=35794;   // HANGUL SYLLABLE
   sortTable[0xB6C5]=35795;   // HANGUL SYLLABLE
   sortTable[0xB6C6]=35796;   // HANGUL SYLLABLE
   sortTable[0xB6C7]=35797;   // HANGUL SYLLABLE
   sortTable[0xB6C8]=35798;   // HANGUL SYLLABLE
   sortTable[0xB6C9]=35799;   // HANGUL SYLLABLE
   sortTable[0xB6CA]=35800;   // HANGUL SYLLABLE
   sortTable[0xB6CB]=35801;   // HANGUL SYLLABLE
   sortTable[0xB6CC]=35802;   // HANGUL SYLLABLE
   sortTable[0xB6CD]=35803;   // HANGUL SYLLABLE
   sortTable[0xB6CE]=35804;   // HANGUL SYLLABLE
   sortTable[0xB6CF]=35805;   // HANGUL SYLLABLE
   sortTable[0xB6D0]=35806;   // HANGUL SYLLABLE
   sortTable[0xB6D1]=35807;   // HANGUL SYLLABLE
   sortTable[0xB6D2]=35808;   // HANGUL SYLLABLE
   sortTable[0xB6D3]=35809;   // HANGUL SYLLABLE
   sortTable[0xB6D4]=35810;   // HANGUL SYLLABLE
   sortTable[0xB6D5]=35811;   // HANGUL SYLLABLE
   sortTable[0xB6D6]=35812;   // HANGUL SYLLABLE
   sortTable[0xB6D7]=35813;   // HANGUL SYLLABLE
   sortTable[0xB6D8]=35814;   // HANGUL SYLLABLE
   sortTable[0xB6D9]=35815;   // HANGUL SYLLABLE
   sortTable[0xB6DA]=35816;   // HANGUL SYLLABLE
   sortTable[0xB6DB]=35817;   // HANGUL SYLLABLE
   sortTable[0xB6DC]=35818;   // HANGUL SYLLABLE
   sortTable[0xB6DD]=35819;   // HANGUL SYLLABLE
   sortTable[0xB6DE]=35820;   // HANGUL SYLLABLE
   sortTable[0xB6DF]=35821;   // HANGUL SYLLABLE
   sortTable[0xB6E0]=35822;   // HANGUL SYLLABLE
   sortTable[0xB6E1]=35823;   // HANGUL SYLLABLE
   sortTable[0xB6E2]=35824;   // HANGUL SYLLABLE
   sortTable[0xB6E3]=35825;   // HANGUL SYLLABLE
   sortTable[0xB6E4]=35826;   // HANGUL SYLLABLE
   sortTable[0xB6E5]=35827;   // HANGUL SYLLABLE
   sortTable[0xB6E6]=35828;   // HANGUL SYLLABLE
   sortTable[0xB6E7]=35829;   // HANGUL SYLLABLE
   sortTable[0xB6E8]=35830;   // HANGUL SYLLABLE
   sortTable[0xB6E9]=35831;   // HANGUL SYLLABLE
   sortTable[0xB6EA]=35832;   // HANGUL SYLLABLE
   sortTable[0xB6EB]=35833;   // HANGUL SYLLABLE
   sortTable[0xB6EC]=35834;   // HANGUL SYLLABLE
   sortTable[0xB6ED]=35835;   // HANGUL SYLLABLE
   sortTable[0xB6EE]=35836;   // HANGUL SYLLABLE
   sortTable[0xB6EF]=35837;   // HANGUL SYLLABLE
   sortTable[0xB6F0]=35838;   // HANGUL SYLLABLE
   sortTable[0xB6F1]=35839;   // HANGUL SYLLABLE
   sortTable[0xB6F2]=35840;   // HANGUL SYLLABLE
   sortTable[0xB6F3]=35841;   // HANGUL SYLLABLE
   sortTable[0xB6F4]=35842;   // HANGUL SYLLABLE
   sortTable[0xB6F5]=35843;   // HANGUL SYLLABLE
   sortTable[0xB6F6]=35844;   // HANGUL SYLLABLE
   sortTable[0xB6F7]=35845;   // HANGUL SYLLABLE
   sortTable[0xB6F8]=35846;   // HANGUL SYLLABLE
   sortTable[0xB6F9]=35847;   // HANGUL SYLLABLE
   sortTable[0xB6FA]=35848;   // HANGUL SYLLABLE
   sortTable[0xB6FB]=35849;   // HANGUL SYLLABLE
   sortTable[0xB6FC]=35850;   // HANGUL SYLLABLE
   sortTable[0xB6FD]=35851;   // HANGUL SYLLABLE
   sortTable[0xB6FE]=35852;   // HANGUL SYLLABLE
   sortTable[0xB6FF]=35853;   // HANGUL SYLLABLE
   sortTable[0xB700]=35854;   // HANGUL SYLLABLE
   sortTable[0xB701]=35855;   // HANGUL SYLLABLE
   sortTable[0xB702]=35856;   // HANGUL SYLLABLE
   sortTable[0xB703]=35857;   // HANGUL SYLLABLE
   sortTable[0xB704]=35858;   // HANGUL SYLLABLE
   sortTable[0xB705]=35859;   // HANGUL SYLLABLE
   sortTable[0xB706]=35860;   // HANGUL SYLLABLE
   sortTable[0xB707]=35861;   // HANGUL SYLLABLE
   sortTable[0xB708]=35862;   // HANGUL SYLLABLE
   sortTable[0xB709]=35863;   // HANGUL SYLLABLE
   sortTable[0xB70A]=35864;   // HANGUL SYLLABLE
   sortTable[0xB70B]=35865;   // HANGUL SYLLABLE
   sortTable[0xB70C]=35866;   // HANGUL SYLLABLE
   sortTable[0xB70D]=35867;   // HANGUL SYLLABLE
   sortTable[0xB70E]=35868;   // HANGUL SYLLABLE
   sortTable[0xB70F]=35869;   // HANGUL SYLLABLE
   sortTable[0xB710]=35870;   // HANGUL SYLLABLE
   sortTable[0xB711]=35871;   // HANGUL SYLLABLE
   sortTable[0xB712]=35872;   // HANGUL SYLLABLE
   sortTable[0xB713]=35873;   // HANGUL SYLLABLE
   sortTable[0xB714]=35874;   // HANGUL SYLLABLE
   sortTable[0xB715]=35875;   // HANGUL SYLLABLE
   sortTable[0xB716]=35876;   // HANGUL SYLLABLE
   sortTable[0xB717]=35877;   // HANGUL SYLLABLE
   sortTable[0xB718]=35878;   // HANGUL SYLLABLE
   sortTable[0xB719]=35879;   // HANGUL SYLLABLE
   sortTable[0xB71A]=35880;   // HANGUL SYLLABLE
   sortTable[0xB71B]=35881;   // HANGUL SYLLABLE
   sortTable[0xB71C]=35882;   // HANGUL SYLLABLE
   sortTable[0xB71D]=35883;   // HANGUL SYLLABLE
   sortTable[0xB71E]=35884;   // HANGUL SYLLABLE
   sortTable[0xB71F]=35885;   // HANGUL SYLLABLE
   sortTable[0xB720]=35886;   // HANGUL SYLLABLE
   sortTable[0xB721]=35887;   // HANGUL SYLLABLE
   sortTable[0xB722]=35888;   // HANGUL SYLLABLE
   sortTable[0xB723]=35889;   // HANGUL SYLLABLE
   sortTable[0xB724]=35890;   // HANGUL SYLLABLE
   sortTable[0xB725]=35891;   // HANGUL SYLLABLE
   sortTable[0xB726]=35892;   // HANGUL SYLLABLE
   sortTable[0xB727]=35893;   // HANGUL SYLLABLE
   sortTable[0xB728]=35894;   // HANGUL SYLLABLE
   sortTable[0xB729]=35895;   // HANGUL SYLLABLE
   sortTable[0xB72A]=35896;   // HANGUL SYLLABLE
   sortTable[0xB72B]=35897;   // HANGUL SYLLABLE
   sortTable[0xB72C]=35898;   // HANGUL SYLLABLE
   sortTable[0xB72D]=35899;   // HANGUL SYLLABLE
   sortTable[0xB72E]=35900;   // HANGUL SYLLABLE
   sortTable[0xB72F]=35901;   // HANGUL SYLLABLE
   sortTable[0xB730]=35902;   // HANGUL SYLLABLE
   sortTable[0xB731]=35903;   // HANGUL SYLLABLE
   sortTable[0xB732]=35904;   // HANGUL SYLLABLE
   sortTable[0xB733]=35905;   // HANGUL SYLLABLE
   sortTable[0xB734]=35906;   // HANGUL SYLLABLE
   sortTable[0xB735]=35907;   // HANGUL SYLLABLE
   sortTable[0xB736]=35908;   // HANGUL SYLLABLE
   sortTable[0xB737]=35909;   // HANGUL SYLLABLE
   sortTable[0xB738]=35910;   // HANGUL SYLLABLE
   sortTable[0xB739]=35911;   // HANGUL SYLLABLE
   sortTable[0xB73A]=35912;   // HANGUL SYLLABLE
   sortTable[0xB73B]=35913;   // HANGUL SYLLABLE
   sortTable[0xB73C]=35914;   // HANGUL SYLLABLE
   sortTable[0xB73D]=35915;   // HANGUL SYLLABLE
   sortTable[0xB73E]=35916;   // HANGUL SYLLABLE
   sortTable[0xB73F]=35917;   // HANGUL SYLLABLE
   sortTable[0xB740]=35918;   // HANGUL SYLLABLE
   sortTable[0xB741]=35919;   // HANGUL SYLLABLE
   sortTable[0xB742]=35920;   // HANGUL SYLLABLE
   sortTable[0xB743]=35921;   // HANGUL SYLLABLE
   sortTable[0xB744]=35922;   // HANGUL SYLLABLE
   sortTable[0xB745]=35923;   // HANGUL SYLLABLE
   sortTable[0xB746]=35924;   // HANGUL SYLLABLE
   sortTable[0xB747]=35925;   // HANGUL SYLLABLE
   sortTable[0xB748]=35926;   // HANGUL SYLLABLE
   sortTable[0xB749]=35927;   // HANGUL SYLLABLE
   sortTable[0xB74A]=35928;   // HANGUL SYLLABLE
   sortTable[0xB74B]=35929;   // HANGUL SYLLABLE
   sortTable[0xB74C]=35930;   // HANGUL SYLLABLE
   sortTable[0xB74D]=35931;   // HANGUL SYLLABLE
   sortTable[0xB74E]=35932;   // HANGUL SYLLABLE
   sortTable[0xB74F]=35933;   // HANGUL SYLLABLE
   sortTable[0xB750]=35934;   // HANGUL SYLLABLE
   sortTable[0xB751]=35935;   // HANGUL SYLLABLE
   sortTable[0xB752]=35936;   // HANGUL SYLLABLE
   sortTable[0xB753]=35937;   // HANGUL SYLLABLE
   sortTable[0xB754]=35938;   // HANGUL SYLLABLE
   sortTable[0xB755]=35939;   // HANGUL SYLLABLE
   sortTable[0xB756]=35940;   // HANGUL SYLLABLE
   sortTable[0xB757]=35941;   // HANGUL SYLLABLE
   sortTable[0xB758]=35942;   // HANGUL SYLLABLE
   sortTable[0xB759]=35943;   // HANGUL SYLLABLE
   sortTable[0xB75A]=35944;   // HANGUL SYLLABLE
   sortTable[0xB75B]=35945;   // HANGUL SYLLABLE
   sortTable[0xB75C]=35946;   // HANGUL SYLLABLE
   sortTable[0xB75D]=35947;   // HANGUL SYLLABLE
   sortTable[0xB75E]=35948;   // HANGUL SYLLABLE
   sortTable[0xB75F]=35949;   // HANGUL SYLLABLE
   sortTable[0xB760]=35950;   // HANGUL SYLLABLE
   sortTable[0xB761]=35951;   // HANGUL SYLLABLE
   sortTable[0xB762]=35952;   // HANGUL SYLLABLE
   sortTable[0xB763]=35953;   // HANGUL SYLLABLE
   sortTable[0xB764]=35954;   // HANGUL SYLLABLE
   sortTable[0xB765]=35955;   // HANGUL SYLLABLE
   sortTable[0xB766]=35956;   // HANGUL SYLLABLE
   sortTable[0xB767]=35957;   // HANGUL SYLLABLE
   sortTable[0xB768]=35958;   // HANGUL SYLLABLE
   sortTable[0xB769]=35959;   // HANGUL SYLLABLE
   sortTable[0xB76A]=35960;   // HANGUL SYLLABLE
   sortTable[0xB76B]=35961;   // HANGUL SYLLABLE
   sortTable[0xB76C]=35962;   // HANGUL SYLLABLE
   sortTable[0xB76D]=35963;   // HANGUL SYLLABLE
   sortTable[0xB76E]=35964;   // HANGUL SYLLABLE
   sortTable[0xB76F]=35965;   // HANGUL SYLLABLE
   sortTable[0xB770]=35966;   // HANGUL SYLLABLE
   sortTable[0xB771]=35967;   // HANGUL SYLLABLE
   sortTable[0xB772]=35968;   // HANGUL SYLLABLE
   sortTable[0xB773]=35969;   // HANGUL SYLLABLE
   sortTable[0xB774]=35970;   // HANGUL SYLLABLE
   sortTable[0xB775]=35971;   // HANGUL SYLLABLE
   sortTable[0xB776]=35972;   // HANGUL SYLLABLE
   sortTable[0xB777]=35973;   // HANGUL SYLLABLE
   sortTable[0xB778]=35974;   // HANGUL SYLLABLE
   sortTable[0xB779]=35975;   // HANGUL SYLLABLE
   sortTable[0xB77A]=35976;   // HANGUL SYLLABLE
   sortTable[0xB77B]=35977;   // HANGUL SYLLABLE
   sortTable[0x3139]=35978;   // HANGUL LETTER RIEUL
   sortTable[0x1105]=35979;   // HANGUL CHOSEONG RIEUL
   sortTable[0x11AF]=35980;   // HANGUL JONGSEONG RIEUL
   sortTable[0xB77C]=35981;   // HANGUL SYLLABLE
   sortTable[0xB77D]=35982;   // HANGUL SYLLABLE
   sortTable[0xB77E]=35983;   // HANGUL SYLLABLE
   sortTable[0xB77F]=35984;   // HANGUL SYLLABLE
   sortTable[0xB780]=35985;   // HANGUL SYLLABLE
   sortTable[0xB781]=35986;   // HANGUL SYLLABLE
   sortTable[0xB782]=35987;   // HANGUL SYLLABLE
   sortTable[0xB783]=35988;   // HANGUL SYLLABLE
   sortTable[0xB784]=35989;   // HANGUL SYLLABLE
   sortTable[0xB785]=35990;   // HANGUL SYLLABLE
   sortTable[0xB786]=35991;   // HANGUL SYLLABLE
   sortTable[0xB787]=35992;   // HANGUL SYLLABLE
   sortTable[0xB788]=35993;   // HANGUL SYLLABLE
   sortTable[0xB789]=35994;   // HANGUL SYLLABLE
   sortTable[0xB78A]=35995;   // HANGUL SYLLABLE
   sortTable[0xB78B]=35996;   // HANGUL SYLLABLE
   sortTable[0xB78C]=35997;   // HANGUL SYLLABLE
   sortTable[0xB78D]=35998;   // HANGUL SYLLABLE
   sortTable[0xB78E]=35999;   // HANGUL SYLLABLE
   sortTable[0xB78F]=36000;   // HANGUL SYLLABLE
   sortTable[0xB790]=36001;   // HANGUL SYLLABLE
   sortTable[0xB791]=36002;   // HANGUL SYLLABLE
   sortTable[0xB792]=36003;   // HANGUL SYLLABLE
   sortTable[0xB793]=36004;   // HANGUL SYLLABLE
   sortTable[0xB794]=36005;   // HANGUL SYLLABLE
   sortTable[0xB795]=36006;   // HANGUL SYLLABLE
   sortTable[0xB796]=36007;   // HANGUL SYLLABLE
   sortTable[0xB797]=36008;   // HANGUL SYLLABLE
   sortTable[0xB798]=36009;   // HANGUL SYLLABLE
   sortTable[0xB799]=36010;   // HANGUL SYLLABLE
   sortTable[0xB79A]=36011;   // HANGUL SYLLABLE
   sortTable[0xB79B]=36012;   // HANGUL SYLLABLE
   sortTable[0xB79C]=36013;   // HANGUL SYLLABLE
   sortTable[0xB79D]=36014;   // HANGUL SYLLABLE
   sortTable[0xB79E]=36015;   // HANGUL SYLLABLE
   sortTable[0xB79F]=36016;   // HANGUL SYLLABLE
   sortTable[0xB7A0]=36017;   // HANGUL SYLLABLE
   sortTable[0xB7A1]=36018;   // HANGUL SYLLABLE
   sortTable[0xB7A2]=36019;   // HANGUL SYLLABLE
   sortTable[0xB7A3]=36020;   // HANGUL SYLLABLE
   sortTable[0xB7A4]=36021;   // HANGUL SYLLABLE
   sortTable[0xB7A5]=36022;   // HANGUL SYLLABLE
   sortTable[0xB7A6]=36023;   // HANGUL SYLLABLE
   sortTable[0xB7A7]=36024;   // HANGUL SYLLABLE
   sortTable[0xB7A8]=36025;   // HANGUL SYLLABLE
   sortTable[0xB7A9]=36026;   // HANGUL SYLLABLE
   sortTable[0xB7AA]=36027;   // HANGUL SYLLABLE
   sortTable[0xB7AB]=36028;   // HANGUL SYLLABLE
   sortTable[0xB7AC]=36029;   // HANGUL SYLLABLE
   sortTable[0xB7AD]=36030;   // HANGUL SYLLABLE
   sortTable[0xB7AE]=36031;   // HANGUL SYLLABLE
   sortTable[0xB7AF]=36032;   // HANGUL SYLLABLE
   sortTable[0xB7B0]=36033;   // HANGUL SYLLABLE
   sortTable[0xB7B1]=36034;   // HANGUL SYLLABLE
   sortTable[0xB7B2]=36035;   // HANGUL SYLLABLE
   sortTable[0xB7B3]=36036;   // HANGUL SYLLABLE
   sortTable[0xB7B4]=36037;   // HANGUL SYLLABLE
   sortTable[0xB7B5]=36038;   // HANGUL SYLLABLE
   sortTable[0xB7B6]=36039;   // HANGUL SYLLABLE
   sortTable[0xB7B7]=36040;   // HANGUL SYLLABLE
   sortTable[0xB7B8]=36041;   // HANGUL SYLLABLE
   sortTable[0xB7B9]=36042;   // HANGUL SYLLABLE
   sortTable[0xB7BA]=36043;   // HANGUL SYLLABLE
   sortTable[0xB7BB]=36044;   // HANGUL SYLLABLE
   sortTable[0xB7BC]=36045;   // HANGUL SYLLABLE
   sortTable[0xB7BD]=36046;   // HANGUL SYLLABLE
   sortTable[0xB7BE]=36047;   // HANGUL SYLLABLE
   sortTable[0xB7BF]=36048;   // HANGUL SYLLABLE
   sortTable[0xB7C0]=36049;   // HANGUL SYLLABLE
   sortTable[0xB7C1]=36050;   // HANGUL SYLLABLE
   sortTable[0xB7C2]=36051;   // HANGUL SYLLABLE
   sortTable[0xB7C3]=36052;   // HANGUL SYLLABLE
   sortTable[0xB7C4]=36053;   // HANGUL SYLLABLE
   sortTable[0xB7C5]=36054;   // HANGUL SYLLABLE
   sortTable[0xB7C6]=36055;   // HANGUL SYLLABLE
   sortTable[0xB7C7]=36056;   // HANGUL SYLLABLE
   sortTable[0xB7C8]=36057;   // HANGUL SYLLABLE
   sortTable[0xB7C9]=36058;   // HANGUL SYLLABLE
   sortTable[0xB7CA]=36059;   // HANGUL SYLLABLE
   sortTable[0xB7CB]=36060;   // HANGUL SYLLABLE
   sortTable[0xB7CC]=36061;   // HANGUL SYLLABLE
   sortTable[0xB7CD]=36062;   // HANGUL SYLLABLE
   sortTable[0xB7CE]=36063;   // HANGUL SYLLABLE
   sortTable[0xB7CF]=36064;   // HANGUL SYLLABLE
   sortTable[0xB7D0]=36065;   // HANGUL SYLLABLE
   sortTable[0xB7D1]=36066;   // HANGUL SYLLABLE
   sortTable[0xB7D2]=36067;   // HANGUL SYLLABLE
   sortTable[0xB7D3]=36068;   // HANGUL SYLLABLE
   sortTable[0xB7D4]=36069;   // HANGUL SYLLABLE
   sortTable[0xB7D5]=36070;   // HANGUL SYLLABLE
   sortTable[0xB7D6]=36071;   // HANGUL SYLLABLE
   sortTable[0xB7D7]=36072;   // HANGUL SYLLABLE
   sortTable[0xB7D8]=36073;   // HANGUL SYLLABLE
   sortTable[0xB7D9]=36074;   // HANGUL SYLLABLE
   sortTable[0xB7DA]=36075;   // HANGUL SYLLABLE
   sortTable[0xB7DB]=36076;   // HANGUL SYLLABLE
   sortTable[0xB7DC]=36077;   // HANGUL SYLLABLE
   sortTable[0xB7DD]=36078;   // HANGUL SYLLABLE
   sortTable[0xB7DE]=36079;   // HANGUL SYLLABLE
   sortTable[0xB7DF]=36080;   // HANGUL SYLLABLE
   sortTable[0xB7E0]=36081;   // HANGUL SYLLABLE
   sortTable[0xB7E1]=36082;   // HANGUL SYLLABLE
   sortTable[0xB7E2]=36083;   // HANGUL SYLLABLE
   sortTable[0xB7E3]=36084;   // HANGUL SYLLABLE
   sortTable[0xB7E4]=36085;   // HANGUL SYLLABLE
   sortTable[0xB7E5]=36086;   // HANGUL SYLLABLE
   sortTable[0xB7E6]=36087;   // HANGUL SYLLABLE
   sortTable[0xB7E7]=36088;   // HANGUL SYLLABLE
   sortTable[0xB7E8]=36089;   // HANGUL SYLLABLE
   sortTable[0xB7E9]=36090;   // HANGUL SYLLABLE
   sortTable[0xB7EA]=36091;   // HANGUL SYLLABLE
   sortTable[0xB7EB]=36092;   // HANGUL SYLLABLE
   sortTable[0xB7EC]=36093;   // HANGUL SYLLABLE
   sortTable[0xB7ED]=36094;   // HANGUL SYLLABLE
   sortTable[0xB7EE]=36095;   // HANGUL SYLLABLE
   sortTable[0xB7EF]=36096;   // HANGUL SYLLABLE
   sortTable[0xB7F0]=36097;   // HANGUL SYLLABLE
   sortTable[0xB7F1]=36098;   // HANGUL SYLLABLE
   sortTable[0xB7F2]=36099;   // HANGUL SYLLABLE
   sortTable[0xB7F3]=36100;   // HANGUL SYLLABLE
   sortTable[0xB7F4]=36101;   // HANGUL SYLLABLE
   sortTable[0xB7F5]=36102;   // HANGUL SYLLABLE
   sortTable[0xB7F6]=36103;   // HANGUL SYLLABLE
   sortTable[0xB7F7]=36104;   // HANGUL SYLLABLE
   sortTable[0xB7F8]=36105;   // HANGUL SYLLABLE
   sortTable[0xB7F9]=36106;   // HANGUL SYLLABLE
   sortTable[0xB7FA]=36107;   // HANGUL SYLLABLE
   sortTable[0xB7FB]=36108;   // HANGUL SYLLABLE
   sortTable[0xB7FC]=36109;   // HANGUL SYLLABLE
   sortTable[0xB7FD]=36110;   // HANGUL SYLLABLE
   sortTable[0xB7FE]=36111;   // HANGUL SYLLABLE
   sortTable[0xB7FF]=36112;   // HANGUL SYLLABLE
   sortTable[0xB800]=36113;   // HANGUL SYLLABLE
   sortTable[0xB801]=36114;   // HANGUL SYLLABLE
   sortTable[0xB802]=36115;   // HANGUL SYLLABLE
   sortTable[0xB803]=36116;   // HANGUL SYLLABLE
   sortTable[0xB804]=36117;   // HANGUL SYLLABLE
   sortTable[0xB805]=36118;   // HANGUL SYLLABLE
   sortTable[0xB806]=36119;   // HANGUL SYLLABLE
   sortTable[0xB807]=36120;   // HANGUL SYLLABLE
   sortTable[0xB808]=36121;   // HANGUL SYLLABLE
   sortTable[0xB809]=36122;   // HANGUL SYLLABLE
   sortTable[0xB80A]=36123;   // HANGUL SYLLABLE
   sortTable[0xB80B]=36124;   // HANGUL SYLLABLE
   sortTable[0xB80C]=36125;   // HANGUL SYLLABLE
   sortTable[0xB80D]=36126;   // HANGUL SYLLABLE
   sortTable[0xB80E]=36127;   // HANGUL SYLLABLE
   sortTable[0xB80F]=36128;   // HANGUL SYLLABLE
   sortTable[0xB810]=36129;   // HANGUL SYLLABLE
   sortTable[0xB811]=36130;   // HANGUL SYLLABLE
   sortTable[0xB812]=36131;   // HANGUL SYLLABLE
   sortTable[0xB813]=36132;   // HANGUL SYLLABLE
   sortTable[0xB814]=36133;   // HANGUL SYLLABLE
   sortTable[0xB815]=36134;   // HANGUL SYLLABLE
   sortTable[0xB816]=36135;   // HANGUL SYLLABLE
   sortTable[0xB817]=36136;   // HANGUL SYLLABLE
   sortTable[0xB818]=36137;   // HANGUL SYLLABLE
   sortTable[0xB819]=36138;   // HANGUL SYLLABLE
   sortTable[0xB81A]=36139;   // HANGUL SYLLABLE
   sortTable[0xB81B]=36140;   // HANGUL SYLLABLE
   sortTable[0xB81C]=36141;   // HANGUL SYLLABLE
   sortTable[0xB81D]=36142;   // HANGUL SYLLABLE
   sortTable[0xB81E]=36143;   // HANGUL SYLLABLE
   sortTable[0xB81F]=36144;   // HANGUL SYLLABLE
   sortTable[0xB820]=36145;   // HANGUL SYLLABLE
   sortTable[0xB821]=36146;   // HANGUL SYLLABLE
   sortTable[0xB822]=36147;   // HANGUL SYLLABLE
   sortTable[0xB823]=36148;   // HANGUL SYLLABLE
   sortTable[0xB824]=36149;   // HANGUL SYLLABLE
   sortTable[0xB825]=36150;   // HANGUL SYLLABLE
   sortTable[0xB826]=36151;   // HANGUL SYLLABLE
   sortTable[0xB827]=36152;   // HANGUL SYLLABLE
   sortTable[0xB828]=36153;   // HANGUL SYLLABLE
   sortTable[0xB829]=36154;   // HANGUL SYLLABLE
   sortTable[0xB82A]=36155;   // HANGUL SYLLABLE
   sortTable[0xB82B]=36156;   // HANGUL SYLLABLE
   sortTable[0xB82C]=36157;   // HANGUL SYLLABLE
   sortTable[0xB82D]=36158;   // HANGUL SYLLABLE
   sortTable[0xB82E]=36159;   // HANGUL SYLLABLE
   sortTable[0xB82F]=36160;   // HANGUL SYLLABLE
   sortTable[0xB830]=36161;   // HANGUL SYLLABLE
   sortTable[0xB831]=36162;   // HANGUL SYLLABLE
   sortTable[0xB832]=36163;   // HANGUL SYLLABLE
   sortTable[0xB833]=36164;   // HANGUL SYLLABLE
   sortTable[0xB834]=36165;   // HANGUL SYLLABLE
   sortTable[0xB835]=36166;   // HANGUL SYLLABLE
   sortTable[0xB836]=36167;   // HANGUL SYLLABLE
   sortTable[0xB837]=36168;   // HANGUL SYLLABLE
   sortTable[0xB838]=36169;   // HANGUL SYLLABLE
   sortTable[0xB839]=36170;   // HANGUL SYLLABLE
   sortTable[0xB83A]=36171;   // HANGUL SYLLABLE
   sortTable[0xB83B]=36172;   // HANGUL SYLLABLE
   sortTable[0xB83C]=36173;   // HANGUL SYLLABLE
   sortTable[0xB83D]=36174;   // HANGUL SYLLABLE
   sortTable[0xB83E]=36175;   // HANGUL SYLLABLE
   sortTable[0xB83F]=36176;   // HANGUL SYLLABLE
   sortTable[0xB840]=36177;   // HANGUL SYLLABLE
   sortTable[0xB841]=36178;   // HANGUL SYLLABLE
   sortTable[0xB842]=36179;   // HANGUL SYLLABLE
   sortTable[0xB843]=36180;   // HANGUL SYLLABLE
   sortTable[0xB844]=36181;   // HANGUL SYLLABLE
   sortTable[0xB845]=36182;   // HANGUL SYLLABLE
   sortTable[0xB846]=36183;   // HANGUL SYLLABLE
   sortTable[0xB847]=36184;   // HANGUL SYLLABLE
   sortTable[0xB848]=36185;   // HANGUL SYLLABLE
   sortTable[0xB849]=36186;   // HANGUL SYLLABLE
   sortTable[0xB84A]=36187;   // HANGUL SYLLABLE
   sortTable[0xB84B]=36188;   // HANGUL SYLLABLE
   sortTable[0xB84C]=36189;   // HANGUL SYLLABLE
   sortTable[0xB84D]=36190;   // HANGUL SYLLABLE
   sortTable[0xB84E]=36191;   // HANGUL SYLLABLE
   sortTable[0xB84F]=36192;   // HANGUL SYLLABLE
   sortTable[0xB850]=36193;   // HANGUL SYLLABLE
   sortTable[0xB851]=36194;   // HANGUL SYLLABLE
   sortTable[0xB852]=36195;   // HANGUL SYLLABLE
   sortTable[0xB853]=36196;   // HANGUL SYLLABLE
   sortTable[0xB854]=36197;   // HANGUL SYLLABLE
   sortTable[0xB855]=36198;   // HANGUL SYLLABLE
   sortTable[0xB856]=36199;   // HANGUL SYLLABLE
   sortTable[0xB857]=36200;   // HANGUL SYLLABLE
   sortTable[0xB858]=36201;   // HANGUL SYLLABLE
   sortTable[0xB859]=36202;   // HANGUL SYLLABLE
   sortTable[0xB85A]=36203;   // HANGUL SYLLABLE
   sortTable[0xB85B]=36204;   // HANGUL SYLLABLE
   sortTable[0xB85C]=36205;   // HANGUL SYLLABLE
   sortTable[0xB85D]=36206;   // HANGUL SYLLABLE
   sortTable[0xB85E]=36207;   // HANGUL SYLLABLE
   sortTable[0xB85F]=36208;   // HANGUL SYLLABLE
   sortTable[0xB860]=36209;   // HANGUL SYLLABLE
   sortTable[0xB861]=36210;   // HANGUL SYLLABLE
   sortTable[0xB862]=36211;   // HANGUL SYLLABLE
   sortTable[0xB863]=36212;   // HANGUL SYLLABLE
   sortTable[0xB864]=36213;   // HANGUL SYLLABLE
   sortTable[0xB865]=36214;   // HANGUL SYLLABLE
   sortTable[0xB866]=36215;   // HANGUL SYLLABLE
   sortTable[0xB867]=36216;   // HANGUL SYLLABLE
   sortTable[0xB868]=36217;   // HANGUL SYLLABLE
   sortTable[0xB869]=36218;   // HANGUL SYLLABLE
   sortTable[0xB86A]=36219;   // HANGUL SYLLABLE
   sortTable[0xB86B]=36220;   // HANGUL SYLLABLE
   sortTable[0xB86C]=36221;   // HANGUL SYLLABLE
   sortTable[0xB86D]=36222;   // HANGUL SYLLABLE
   sortTable[0xB86E]=36223;   // HANGUL SYLLABLE
   sortTable[0xB86F]=36224;   // HANGUL SYLLABLE
   sortTable[0xB870]=36225;   // HANGUL SYLLABLE
   sortTable[0xB871]=36226;   // HANGUL SYLLABLE
   sortTable[0xB872]=36227;   // HANGUL SYLLABLE
   sortTable[0xB873]=36228;   // HANGUL SYLLABLE
   sortTable[0xB874]=36229;   // HANGUL SYLLABLE
   sortTable[0xB875]=36230;   // HANGUL SYLLABLE
   sortTable[0xB876]=36231;   // HANGUL SYLLABLE
   sortTable[0xB877]=36232;   // HANGUL SYLLABLE
   sortTable[0xB878]=36233;   // HANGUL SYLLABLE
   sortTable[0xB879]=36234;   // HANGUL SYLLABLE
   sortTable[0xB87A]=36235;   // HANGUL SYLLABLE
   sortTable[0xB87B]=36236;   // HANGUL SYLLABLE
   sortTable[0xB87C]=36237;   // HANGUL SYLLABLE
   sortTable[0xB87D]=36238;   // HANGUL SYLLABLE
   sortTable[0xB87E]=36239;   // HANGUL SYLLABLE
   sortTable[0xB87F]=36240;   // HANGUL SYLLABLE
   sortTable[0xB880]=36241;   // HANGUL SYLLABLE
   sortTable[0xB881]=36242;   // HANGUL SYLLABLE
   sortTable[0xB882]=36243;   // HANGUL SYLLABLE
   sortTable[0xB883]=36244;   // HANGUL SYLLABLE
   sortTable[0xB884]=36245;   // HANGUL SYLLABLE
   sortTable[0xB885]=36246;   // HANGUL SYLLABLE
   sortTable[0xB886]=36247;   // HANGUL SYLLABLE
   sortTable[0xB887]=36248;   // HANGUL SYLLABLE
   sortTable[0xB888]=36249;   // HANGUL SYLLABLE
   sortTable[0xB889]=36250;   // HANGUL SYLLABLE
   sortTable[0xB88A]=36251;   // HANGUL SYLLABLE
   sortTable[0xB88B]=36252;   // HANGUL SYLLABLE
   sortTable[0xB88C]=36253;   // HANGUL SYLLABLE
   sortTable[0xB88D]=36254;   // HANGUL SYLLABLE
   sortTable[0xB88E]=36255;   // HANGUL SYLLABLE
   sortTable[0xB88F]=36256;   // HANGUL SYLLABLE
   sortTable[0xB890]=36257;   // HANGUL SYLLABLE
   sortTable[0xB891]=36258;   // HANGUL SYLLABLE
   sortTable[0xB892]=36259;   // HANGUL SYLLABLE
   sortTable[0xB893]=36260;   // HANGUL SYLLABLE
   sortTable[0xB894]=36261;   // HANGUL SYLLABLE
   sortTable[0xB895]=36262;   // HANGUL SYLLABLE
   sortTable[0xB896]=36263;   // HANGUL SYLLABLE
   sortTable[0xB897]=36264;   // HANGUL SYLLABLE
   sortTable[0xB898]=36265;   // HANGUL SYLLABLE
   sortTable[0xB899]=36266;   // HANGUL SYLLABLE
   sortTable[0xB89A]=36267;   // HANGUL SYLLABLE
   sortTable[0xB89B]=36268;   // HANGUL SYLLABLE
   sortTable[0xB89C]=36269;   // HANGUL SYLLABLE
   sortTable[0xB89D]=36270;   // HANGUL SYLLABLE
   sortTable[0xB89E]=36271;   // HANGUL SYLLABLE
   sortTable[0xB89F]=36272;   // HANGUL SYLLABLE
   sortTable[0xB8A0]=36273;   // HANGUL SYLLABLE
   sortTable[0xB8A1]=36274;   // HANGUL SYLLABLE
   sortTable[0xB8A2]=36275;   // HANGUL SYLLABLE
   sortTable[0xB8A3]=36276;   // HANGUL SYLLABLE
   sortTable[0xB8A4]=36277;   // HANGUL SYLLABLE
   sortTable[0xB8A5]=36278;   // HANGUL SYLLABLE
   sortTable[0xB8A6]=36279;   // HANGUL SYLLABLE
   sortTable[0xB8A7]=36280;   // HANGUL SYLLABLE
   sortTable[0xB8A8]=36281;   // HANGUL SYLLABLE
   sortTable[0xB8A9]=36282;   // HANGUL SYLLABLE
   sortTable[0xB8AA]=36283;   // HANGUL SYLLABLE
   sortTable[0xB8AB]=36284;   // HANGUL SYLLABLE
   sortTable[0xB8AC]=36285;   // HANGUL SYLLABLE
   sortTable[0xB8AD]=36286;   // HANGUL SYLLABLE
   sortTable[0xB8AE]=36287;   // HANGUL SYLLABLE
   sortTable[0xB8AF]=36288;   // HANGUL SYLLABLE
   sortTable[0xB8B0]=36289;   // HANGUL SYLLABLE
   sortTable[0xB8B1]=36290;   // HANGUL SYLLABLE
   sortTable[0xB8B2]=36291;   // HANGUL SYLLABLE
   sortTable[0xB8B3]=36292;   // HANGUL SYLLABLE
   sortTable[0xB8B4]=36293;   // HANGUL SYLLABLE
   sortTable[0xB8B5]=36294;   // HANGUL SYLLABLE
   sortTable[0xB8B6]=36295;   // HANGUL SYLLABLE
   sortTable[0xB8B7]=36296;   // HANGUL SYLLABLE
   sortTable[0xB8B8]=36297;   // HANGUL SYLLABLE
   sortTable[0xB8B9]=36298;   // HANGUL SYLLABLE
   sortTable[0xB8BA]=36299;   // HANGUL SYLLABLE
   sortTable[0xB8BB]=36300;   // HANGUL SYLLABLE
   sortTable[0xB8BC]=36301;   // HANGUL SYLLABLE
   sortTable[0xB8BD]=36302;   // HANGUL SYLLABLE
   sortTable[0xB8BE]=36303;   // HANGUL SYLLABLE
   sortTable[0xB8BF]=36304;   // HANGUL SYLLABLE
   sortTable[0xB8C0]=36305;   // HANGUL SYLLABLE
   sortTable[0xB8C1]=36306;   // HANGUL SYLLABLE
   sortTable[0xB8C2]=36307;   // HANGUL SYLLABLE
   sortTable[0xB8C3]=36308;   // HANGUL SYLLABLE
   sortTable[0xB8C4]=36309;   // HANGUL SYLLABLE
   sortTable[0xB8C5]=36310;   // HANGUL SYLLABLE
   sortTable[0xB8C6]=36311;   // HANGUL SYLLABLE
   sortTable[0xB8C7]=36312;   // HANGUL SYLLABLE
   sortTable[0xB8C8]=36313;   // HANGUL SYLLABLE
   sortTable[0xB8C9]=36314;   // HANGUL SYLLABLE
   sortTable[0xB8CA]=36315;   // HANGUL SYLLABLE
   sortTable[0xB8CB]=36316;   // HANGUL SYLLABLE
   sortTable[0xB8CC]=36317;   // HANGUL SYLLABLE
   sortTable[0xB8CD]=36318;   // HANGUL SYLLABLE
   sortTable[0xB8CE]=36319;   // HANGUL SYLLABLE
   sortTable[0xB8CF]=36320;   // HANGUL SYLLABLE
   sortTable[0xB8D0]=36321;   // HANGUL SYLLABLE
   sortTable[0xB8D1]=36322;   // HANGUL SYLLABLE
   sortTable[0xB8D2]=36323;   // HANGUL SYLLABLE
   sortTable[0xB8D3]=36324;   // HANGUL SYLLABLE
   sortTable[0xB8D4]=36325;   // HANGUL SYLLABLE
   sortTable[0xB8D5]=36326;   // HANGUL SYLLABLE
   sortTable[0xB8D6]=36327;   // HANGUL SYLLABLE
   sortTable[0xB8D7]=36328;   // HANGUL SYLLABLE
   sortTable[0xB8D8]=36329;   // HANGUL SYLLABLE
   sortTable[0xB8D9]=36330;   // HANGUL SYLLABLE
   sortTable[0xB8DA]=36331;   // HANGUL SYLLABLE
   sortTable[0xB8DB]=36332;   // HANGUL SYLLABLE
   sortTable[0xB8DC]=36333;   // HANGUL SYLLABLE
   sortTable[0xB8DD]=36334;   // HANGUL SYLLABLE
   sortTable[0xB8DE]=36335;   // HANGUL SYLLABLE
   sortTable[0xB8DF]=36336;   // HANGUL SYLLABLE
   sortTable[0xB8E0]=36337;   // HANGUL SYLLABLE
   sortTable[0xB8E1]=36338;   // HANGUL SYLLABLE
   sortTable[0xB8E2]=36339;   // HANGUL SYLLABLE
   sortTable[0xB8E3]=36340;   // HANGUL SYLLABLE
   sortTable[0xB8E4]=36341;   // HANGUL SYLLABLE
   sortTable[0xB8E5]=36342;   // HANGUL SYLLABLE
   sortTable[0xB8E6]=36343;   // HANGUL SYLLABLE
   sortTable[0xB8E7]=36344;   // HANGUL SYLLABLE
   sortTable[0xB8E8]=36345;   // HANGUL SYLLABLE
   sortTable[0xB8E9]=36346;   // HANGUL SYLLABLE
   sortTable[0xB8EA]=36347;   // HANGUL SYLLABLE
   sortTable[0xB8EB]=36348;   // HANGUL SYLLABLE
   sortTable[0xB8EC]=36349;   // HANGUL SYLLABLE
   sortTable[0xB8ED]=36350;   // HANGUL SYLLABLE
   sortTable[0xB8EE]=36351;   // HANGUL SYLLABLE
   sortTable[0xB8EF]=36352;   // HANGUL SYLLABLE
   sortTable[0xB8F0]=36353;   // HANGUL SYLLABLE
   sortTable[0xB8F1]=36354;   // HANGUL SYLLABLE
   sortTable[0xB8F2]=36355;   // HANGUL SYLLABLE
   sortTable[0xB8F3]=36356;   // HANGUL SYLLABLE
   sortTable[0xB8F4]=36357;   // HANGUL SYLLABLE
   sortTable[0xB8F5]=36358;   // HANGUL SYLLABLE
   sortTable[0xB8F6]=36359;   // HANGUL SYLLABLE
   sortTable[0xB8F7]=36360;   // HANGUL SYLLABLE
   sortTable[0xB8F8]=36361;   // HANGUL SYLLABLE
   sortTable[0xB8F9]=36362;   // HANGUL SYLLABLE
   sortTable[0xB8FA]=36363;   // HANGUL SYLLABLE
   sortTable[0xB8FB]=36364;   // HANGUL SYLLABLE
   sortTable[0xB8FC]=36365;   // HANGUL SYLLABLE
   sortTable[0xB8FD]=36366;   // HANGUL SYLLABLE
   sortTable[0xB8FE]=36367;   // HANGUL SYLLABLE
   sortTable[0xB8FF]=36368;   // HANGUL SYLLABLE
   sortTable[0xB900]=36369;   // HANGUL SYLLABLE
   sortTable[0xB901]=36370;   // HANGUL SYLLABLE
   sortTable[0xB902]=36371;   // HANGUL SYLLABLE
   sortTable[0xB903]=36372;   // HANGUL SYLLABLE
   sortTable[0xB904]=36373;   // HANGUL SYLLABLE
   sortTable[0xB905]=36374;   // HANGUL SYLLABLE
   sortTable[0xB906]=36375;   // HANGUL SYLLABLE
   sortTable[0xB907]=36376;   // HANGUL SYLLABLE
   sortTable[0xB908]=36377;   // HANGUL SYLLABLE
   sortTable[0xB909]=36378;   // HANGUL SYLLABLE
   sortTable[0xB90A]=36379;   // HANGUL SYLLABLE
   sortTable[0xB90B]=36380;   // HANGUL SYLLABLE
   sortTable[0xB90C]=36381;   // HANGUL SYLLABLE
   sortTable[0xB90D]=36382;   // HANGUL SYLLABLE
   sortTable[0xB90E]=36383;   // HANGUL SYLLABLE
   sortTable[0xB90F]=36384;   // HANGUL SYLLABLE
   sortTable[0xB910]=36385;   // HANGUL SYLLABLE
   sortTable[0xB911]=36386;   // HANGUL SYLLABLE
   sortTable[0xB912]=36387;   // HANGUL SYLLABLE
   sortTable[0xB913]=36388;   // HANGUL SYLLABLE
   sortTable[0xB914]=36389;   // HANGUL SYLLABLE
   sortTable[0xB915]=36390;   // HANGUL SYLLABLE
   sortTable[0xB916]=36391;   // HANGUL SYLLABLE
   sortTable[0xB917]=36392;   // HANGUL SYLLABLE
   sortTable[0xB918]=36393;   // HANGUL SYLLABLE
   sortTable[0xB919]=36394;   // HANGUL SYLLABLE
   sortTable[0xB91A]=36395;   // HANGUL SYLLABLE
   sortTable[0xB91B]=36396;   // HANGUL SYLLABLE
   sortTable[0xB91C]=36397;   // HANGUL SYLLABLE
   sortTable[0xB91D]=36398;   // HANGUL SYLLABLE
   sortTable[0xB91E]=36399;   // HANGUL SYLLABLE
   sortTable[0xB91F]=36400;   // HANGUL SYLLABLE
   sortTable[0xB920]=36401;   // HANGUL SYLLABLE
   sortTable[0xB921]=36402;   // HANGUL SYLLABLE
   sortTable[0xB922]=36403;   // HANGUL SYLLABLE
   sortTable[0xB923]=36404;   // HANGUL SYLLABLE
   sortTable[0xB924]=36405;   // HANGUL SYLLABLE
   sortTable[0xB925]=36406;   // HANGUL SYLLABLE
   sortTable[0xB926]=36407;   // HANGUL SYLLABLE
   sortTable[0xB927]=36408;   // HANGUL SYLLABLE
   sortTable[0xB928]=36409;   // HANGUL SYLLABLE
   sortTable[0xB929]=36410;   // HANGUL SYLLABLE
   sortTable[0xB92A]=36411;   // HANGUL SYLLABLE
   sortTable[0xB92B]=36412;   // HANGUL SYLLABLE
   sortTable[0xB92C]=36413;   // HANGUL SYLLABLE
   sortTable[0xB92D]=36414;   // HANGUL SYLLABLE
   sortTable[0xB92E]=36415;   // HANGUL SYLLABLE
   sortTable[0xB92F]=36416;   // HANGUL SYLLABLE
   sortTable[0xB930]=36417;   // HANGUL SYLLABLE
   sortTable[0xB931]=36418;   // HANGUL SYLLABLE
   sortTable[0xB932]=36419;   // HANGUL SYLLABLE
   sortTable[0xB933]=36420;   // HANGUL SYLLABLE
   sortTable[0xB934]=36421;   // HANGUL SYLLABLE
   sortTable[0xB935]=36422;   // HANGUL SYLLABLE
   sortTable[0xB936]=36423;   // HANGUL SYLLABLE
   sortTable[0xB937]=36424;   // HANGUL SYLLABLE
   sortTable[0xB938]=36425;   // HANGUL SYLLABLE
   sortTable[0xB939]=36426;   // HANGUL SYLLABLE
   sortTable[0xB93A]=36427;   // HANGUL SYLLABLE
   sortTable[0xB93B]=36428;   // HANGUL SYLLABLE
   sortTable[0xB93C]=36429;   // HANGUL SYLLABLE
   sortTable[0xB93D]=36430;   // HANGUL SYLLABLE
   sortTable[0xB93E]=36431;   // HANGUL SYLLABLE
   sortTable[0xB93F]=36432;   // HANGUL SYLLABLE
   sortTable[0xB940]=36433;   // HANGUL SYLLABLE
   sortTable[0xB941]=36434;   // HANGUL SYLLABLE
   sortTable[0xB942]=36435;   // HANGUL SYLLABLE
   sortTable[0xB943]=36436;   // HANGUL SYLLABLE
   sortTable[0xB944]=36437;   // HANGUL SYLLABLE
   sortTable[0xB945]=36438;   // HANGUL SYLLABLE
   sortTable[0xB946]=36439;   // HANGUL SYLLABLE
   sortTable[0xB947]=36440;   // HANGUL SYLLABLE
   sortTable[0xB948]=36441;   // HANGUL SYLLABLE
   sortTable[0xB949]=36442;   // HANGUL SYLLABLE
   sortTable[0xB94A]=36443;   // HANGUL SYLLABLE
   sortTable[0xB94B]=36444;   // HANGUL SYLLABLE
   sortTable[0xB94C]=36445;   // HANGUL SYLLABLE
   sortTable[0xB94D]=36446;   // HANGUL SYLLABLE
   sortTable[0xB94E]=36447;   // HANGUL SYLLABLE
   sortTable[0xB94F]=36448;   // HANGUL SYLLABLE
   sortTable[0xB950]=36449;   // HANGUL SYLLABLE
   sortTable[0xB951]=36450;   // HANGUL SYLLABLE
   sortTable[0xB952]=36451;   // HANGUL SYLLABLE
   sortTable[0xB953]=36452;   // HANGUL SYLLABLE
   sortTable[0xB954]=36453;   // HANGUL SYLLABLE
   sortTable[0xB955]=36454;   // HANGUL SYLLABLE
   sortTable[0xB956]=36455;   // HANGUL SYLLABLE
   sortTable[0xB957]=36456;   // HANGUL SYLLABLE
   sortTable[0xB958]=36457;   // HANGUL SYLLABLE
   sortTable[0xB959]=36458;   // HANGUL SYLLABLE
   sortTable[0xB95A]=36459;   // HANGUL SYLLABLE
   sortTable[0xB95B]=36460;   // HANGUL SYLLABLE
   sortTable[0xB95C]=36461;   // HANGUL SYLLABLE
   sortTable[0xB95D]=36462;   // HANGUL SYLLABLE
   sortTable[0xB95E]=36463;   // HANGUL SYLLABLE
   sortTable[0xB95F]=36464;   // HANGUL SYLLABLE
   sortTable[0xB960]=36465;   // HANGUL SYLLABLE
   sortTable[0xB961]=36466;   // HANGUL SYLLABLE
   sortTable[0xB962]=36467;   // HANGUL SYLLABLE
   sortTable[0xB963]=36468;   // HANGUL SYLLABLE
   sortTable[0xB964]=36469;   // HANGUL SYLLABLE
   sortTable[0xB965]=36470;   // HANGUL SYLLABLE
   sortTable[0xB966]=36471;   // HANGUL SYLLABLE
   sortTable[0xB967]=36472;   // HANGUL SYLLABLE
   sortTable[0xB968]=36473;   // HANGUL SYLLABLE
   sortTable[0xB969]=36474;   // HANGUL SYLLABLE
   sortTable[0xB96A]=36475;   // HANGUL SYLLABLE
   sortTable[0xB96B]=36476;   // HANGUL SYLLABLE
   sortTable[0xB96C]=36477;   // HANGUL SYLLABLE
   sortTable[0xB96D]=36478;   // HANGUL SYLLABLE
   sortTable[0xB96E]=36479;   // HANGUL SYLLABLE
   sortTable[0xB96F]=36480;   // HANGUL SYLLABLE
   sortTable[0xB970]=36481;   // HANGUL SYLLABLE
   sortTable[0xB971]=36482;   // HANGUL SYLLABLE
   sortTable[0xB972]=36483;   // HANGUL SYLLABLE
   sortTable[0xB973]=36484;   // HANGUL SYLLABLE
   sortTable[0xB974]=36485;   // HANGUL SYLLABLE
   sortTable[0xB975]=36486;   // HANGUL SYLLABLE
   sortTable[0xB976]=36487;   // HANGUL SYLLABLE
   sortTable[0xB977]=36488;   // HANGUL SYLLABLE
   sortTable[0xB978]=36489;   // HANGUL SYLLABLE
   sortTable[0xB979]=36490;   // HANGUL SYLLABLE
   sortTable[0xB97A]=36491;   // HANGUL SYLLABLE
   sortTable[0xB97B]=36492;   // HANGUL SYLLABLE
   sortTable[0xB97C]=36493;   // HANGUL SYLLABLE
   sortTable[0xB97D]=36494;   // HANGUL SYLLABLE
   sortTable[0xB97E]=36495;   // HANGUL SYLLABLE
   sortTable[0xB97F]=36496;   // HANGUL SYLLABLE
   sortTable[0xB980]=36497;   // HANGUL SYLLABLE
   sortTable[0xB981]=36498;   // HANGUL SYLLABLE
   sortTable[0xB982]=36499;   // HANGUL SYLLABLE
   sortTable[0xB983]=36500;   // HANGUL SYLLABLE
   sortTable[0xB984]=36501;   // HANGUL SYLLABLE
   sortTable[0xB985]=36502;   // HANGUL SYLLABLE
   sortTable[0xB986]=36503;   // HANGUL SYLLABLE
   sortTable[0xB987]=36504;   // HANGUL SYLLABLE
   sortTable[0xB988]=36505;   // HANGUL SYLLABLE
   sortTable[0xB989]=36506;   // HANGUL SYLLABLE
   sortTable[0xB98A]=36507;   // HANGUL SYLLABLE
   sortTable[0xB98B]=36508;   // HANGUL SYLLABLE
   sortTable[0xB98C]=36509;   // HANGUL SYLLABLE
   sortTable[0xB98D]=36510;   // HANGUL SYLLABLE
   sortTable[0xB98E]=36511;   // HANGUL SYLLABLE
   sortTable[0xB98F]=36512;   // HANGUL SYLLABLE
   sortTable[0xB990]=36513;   // HANGUL SYLLABLE
   sortTable[0xB991]=36514;   // HANGUL SYLLABLE
   sortTable[0xB992]=36515;   // HANGUL SYLLABLE
   sortTable[0xB993]=36516;   // HANGUL SYLLABLE
   sortTable[0xB994]=36517;   // HANGUL SYLLABLE
   sortTable[0xB995]=36518;   // HANGUL SYLLABLE
   sortTable[0xB996]=36519;   // HANGUL SYLLABLE
   sortTable[0xB997]=36520;   // HANGUL SYLLABLE
   sortTable[0xB998]=36521;   // HANGUL SYLLABLE
   sortTable[0xB999]=36522;   // HANGUL SYLLABLE
   sortTable[0xB99A]=36523;   // HANGUL SYLLABLE
   sortTable[0xB99B]=36524;   // HANGUL SYLLABLE
   sortTable[0xB99C]=36525;   // HANGUL SYLLABLE
   sortTable[0xB99D]=36526;   // HANGUL SYLLABLE
   sortTable[0xB99E]=36527;   // HANGUL SYLLABLE
   sortTable[0xB99F]=36528;   // HANGUL SYLLABLE
   sortTable[0xB9A0]=36529;   // HANGUL SYLLABLE
   sortTable[0xB9A1]=36530;   // HANGUL SYLLABLE
   sortTable[0xB9A2]=36531;   // HANGUL SYLLABLE
   sortTable[0xB9A3]=36532;   // HANGUL SYLLABLE
   sortTable[0xB9A4]=36533;   // HANGUL SYLLABLE
   sortTable[0xB9A5]=36534;   // HANGUL SYLLABLE
   sortTable[0xB9A6]=36535;   // HANGUL SYLLABLE
   sortTable[0xB9A7]=36536;   // HANGUL SYLLABLE
   sortTable[0xB9A8]=36537;   // HANGUL SYLLABLE
   sortTable[0xB9A9]=36538;   // HANGUL SYLLABLE
   sortTable[0xB9AA]=36539;   // HANGUL SYLLABLE
   sortTable[0xB9AB]=36540;   // HANGUL SYLLABLE
   sortTable[0xB9AC]=36541;   // HANGUL SYLLABLE
   sortTable[0xB9AD]=36542;   // HANGUL SYLLABLE
   sortTable[0xB9AE]=36543;   // HANGUL SYLLABLE
   sortTable[0xB9AF]=36544;   // HANGUL SYLLABLE
   sortTable[0xB9B0]=36545;   // HANGUL SYLLABLE
   sortTable[0xB9B1]=36546;   // HANGUL SYLLABLE
   sortTable[0xB9B2]=36547;   // HANGUL SYLLABLE
   sortTable[0xB9B3]=36548;   // HANGUL SYLLABLE
   sortTable[0xB9B4]=36549;   // HANGUL SYLLABLE
   sortTable[0xB9B5]=36550;   // HANGUL SYLLABLE
   sortTable[0xB9B6]=36551;   // HANGUL SYLLABLE
   sortTable[0xB9B7]=36552;   // HANGUL SYLLABLE
   sortTable[0xB9B8]=36553;   // HANGUL SYLLABLE
   sortTable[0xB9B9]=36554;   // HANGUL SYLLABLE
   sortTable[0xB9BA]=36555;   // HANGUL SYLLABLE
   sortTable[0xB9BB]=36556;   // HANGUL SYLLABLE
   sortTable[0xB9BC]=36557;   // HANGUL SYLLABLE
   sortTable[0xB9BD]=36558;   // HANGUL SYLLABLE
   sortTable[0xB9BE]=36559;   // HANGUL SYLLABLE
   sortTable[0xB9BF]=36560;   // HANGUL SYLLABLE
   sortTable[0xB9C0]=36561;   // HANGUL SYLLABLE
   sortTable[0xB9C1]=36562;   // HANGUL SYLLABLE
   sortTable[0xB9C2]=36563;   // HANGUL SYLLABLE
   sortTable[0xB9C3]=36564;   // HANGUL SYLLABLE
   sortTable[0xB9C4]=36565;   // HANGUL SYLLABLE
   sortTable[0xB9C5]=36566;   // HANGUL SYLLABLE
   sortTable[0xB9C6]=36567;   // HANGUL SYLLABLE
   sortTable[0xB9C7]=36568;   // HANGUL SYLLABLE
   sortTable[0x3141]=36569;   // HANGUL LETTER MIEUM
   sortTable[0x1106]=36570;   // HANGUL CHOSEONG MIEUM
   sortTable[0x11B7]=36571;   // HANGUL JONGSEONG MIEUM
   sortTable[0xB9C8]=36572;   // HANGUL SYLLABLE
   sortTable[0xB9C9]=36573;   // HANGUL SYLLABLE
   sortTable[0xB9CA]=36574;   // HANGUL SYLLABLE
   sortTable[0xB9CB]=36575;   // HANGUL SYLLABLE
   sortTable[0xB9CC]=36576;   // HANGUL SYLLABLE
   sortTable[0xB9CD]=36577;   // HANGUL SYLLABLE
   sortTable[0xB9CE]=36578;   // HANGUL SYLLABLE
   sortTable[0xB9CF]=36579;   // HANGUL SYLLABLE
   sortTable[0xB9D0]=36580;   // HANGUL SYLLABLE
   sortTable[0xB9D1]=36581;   // HANGUL SYLLABLE
   sortTable[0xB9D2]=36582;   // HANGUL SYLLABLE
   sortTable[0xB9D3]=36583;   // HANGUL SYLLABLE
   sortTable[0xB9D4]=36584;   // HANGUL SYLLABLE
   sortTable[0xB9D5]=36585;   // HANGUL SYLLABLE
   sortTable[0xB9D6]=36586;   // HANGUL SYLLABLE
   sortTable[0xB9D7]=36587;   // HANGUL SYLLABLE
   sortTable[0xB9D8]=36588;   // HANGUL SYLLABLE
   sortTable[0xB9D9]=36589;   // HANGUL SYLLABLE
   sortTable[0xB9DA]=36590;   // HANGUL SYLLABLE
   sortTable[0xB9DB]=36591;   // HANGUL SYLLABLE
   sortTable[0xB9DC]=36592;   // HANGUL SYLLABLE
   sortTable[0xB9DD]=36593;   // HANGUL SYLLABLE
   sortTable[0xB9DE]=36594;   // HANGUL SYLLABLE
   sortTable[0xB9DF]=36595;   // HANGUL SYLLABLE
   sortTable[0xB9E0]=36596;   // HANGUL SYLLABLE
   sortTable[0xB9E1]=36597;   // HANGUL SYLLABLE
   sortTable[0xB9E2]=36598;   // HANGUL SYLLABLE
   sortTable[0xB9E3]=36599;   // HANGUL SYLLABLE
   sortTable[0xB9E4]=36600;   // HANGUL SYLLABLE
   sortTable[0xB9E5]=36601;   // HANGUL SYLLABLE
   sortTable[0xB9E6]=36602;   // HANGUL SYLLABLE
   sortTable[0xB9E7]=36603;   // HANGUL SYLLABLE
   sortTable[0xB9E8]=36604;   // HANGUL SYLLABLE
   sortTable[0xB9E9]=36605;   // HANGUL SYLLABLE
   sortTable[0xB9EA]=36606;   // HANGUL SYLLABLE
   sortTable[0xB9EB]=36607;   // HANGUL SYLLABLE
   sortTable[0xB9EC]=36608;   // HANGUL SYLLABLE
   sortTable[0xB9ED]=36609;   // HANGUL SYLLABLE
   sortTable[0xB9EE]=36610;   // HANGUL SYLLABLE
   sortTable[0xB9EF]=36611;   // HANGUL SYLLABLE
   sortTable[0xB9F0]=36612;   // HANGUL SYLLABLE
   sortTable[0xB9F1]=36613;   // HANGUL SYLLABLE
   sortTable[0xB9F2]=36614;   // HANGUL SYLLABLE
   sortTable[0xB9F3]=36615;   // HANGUL SYLLABLE
   sortTable[0xB9F4]=36616;   // HANGUL SYLLABLE
   sortTable[0xB9F5]=36617;   // HANGUL SYLLABLE
   sortTable[0xB9F6]=36618;   // HANGUL SYLLABLE
   sortTable[0xB9F7]=36619;   // HANGUL SYLLABLE
   sortTable[0xB9F8]=36620;   // HANGUL SYLLABLE
   sortTable[0xB9F9]=36621;   // HANGUL SYLLABLE
   sortTable[0xB9FA]=36622;   // HANGUL SYLLABLE
   sortTable[0xB9FB]=36623;   // HANGUL SYLLABLE
   sortTable[0xB9FC]=36624;   // HANGUL SYLLABLE
   sortTable[0xB9FD]=36625;   // HANGUL SYLLABLE
   sortTable[0xB9FE]=36626;   // HANG