/*****************************************************************************
* FILE:         veda_appl_main.h
* PROJECT:      G3G project
* SW-COMPONENT: VehicleData
*----------------------------------------------------------------------------
*
* DESCRIPTION:  Language handler for VehicleData
*              
*----------------------------------------------------------------------------
* COPYRIGHT:    (c) 2015 Robert Bosch GmbH, Hildesheim
*****************************************************************************/

#ifndef __VEDA_LANGUAGE_HANLDER__H_
#define __VEDA_LANGUAGE_HANLDER__H_

/******************************************************************************/
/* INCLUDES                                                                   */
/******************************************************************************/
#include "I_veda_language_handler.h"

//! Class Definition
class vehicle_fi_tcl_Language_SourceTable;
class Iveda_tclServiceVehicleIf;
class I_veda_tclClientVcan;

class veda_tclLanguageHandler : public Iveda_tclLanguageHandler
{
public:
   //! Default Constructor
   veda_tclLanguageHandler();

   //! User Defined Constructor
   veda_tclLanguageHandler(veda_tclAppMain* poMainAppl);

   //! Destructor
   virtual ~veda_tclLanguageHandler();

   virtual const tChar* getName() const { return "veda_tclLanguageHandler"; }

   virtual tVoid vHandleMessage(TMsg* pMsg);
   virtual tVoid vHandleTraceMessage(const tUChar* puchData);

   //! Get any pointers which this class may need
   tVoid vGetReferences();

   //! Most of the applications are initialized now
   tVoid vStartCommunication();
   tVoid vTraceInfo();

private:

   /**
   * @brief Handle the Set Language changed for Renault
   * @param u8SourceId
   * @param u8Language
   * @return
   */
   tVoid vHandleSetLanguage(const tU8 u8SourceId, tU8 u8Language);

   /**
   * @brief Handle the Set Language Method
   * @param u8SourceId
   * @param u8Language
   * @return
   */
   tVoid vSetLanguageTriggered(const tU8 u8SourceId, const tU8 u8Language);

   /**
   * @brief Handle the Set LanguageSyncSource Property
   * @param u8SourceId
   * @return
   */
   tVoid vLanguageSyncTriggered(const tU8 u8SourceId);

   /**
   * Read the Configured region from KDS
   * @param
   * @return Read Region
   */
   tU8 _u8ReadConfiguredRegion(tVoid) const;

   /**
   * Read the "Default Language" KDS configuration
   * @param u8Language
   * @return
   */
   tVoid vGetKDSDefaultLanguage(tU8& u8Language) const;

   /**
    * @brief Sets the System Language and HMI Language to default
    *        for eg. when factory-reset triggered or device re-configured
    * @param None
    * @return None
    */
   tVoid _vSetDefaultLanguage(tVoid);

   /**
   * @brief Fetch all the supported languages according to the KDS region
   *        Source of the supported languages is the static table
   * @param Region
   * @return None
   */
   tVoid _vFetchRegionBasedSupportedLanguages(tU8 u8Region);

   /**
   * @brief Read the last mode values persistent Memory data
   * @param None
   * @return None
   */
   tVoid _vReadPersistentData(tVoid);

   /**
   * @brief Handle the New Language request for a particular source
   * @param u8SourceId - Language Source Id
   * @param u8Language - New Language to be set for Source Id
   * @return TRUE If source language is changed
   */
   tBool _bHandleSourceLanguageUpdate(const tU8 u8SourceId, const tU8 u8Language);

   /**
   * @brief Utility Functions to get the current language selected by a particular Source
   * @param u8SourceId
   * @return Returns the Language for a particular source
   */
   tU8 _u8GetSourceLanguage(const tU8 u8SourceId) const;

   /**
   * @brief Utility Functions to Check if a particular Language is supported currently
   * @param u8Language
   * @return True (Supported) /False (Not-Supported)
   */
   tBool _bIsRequestedLanguageSupported(const tU8 u8Language) const;

   /**
   * @brief Trigger a Service/Property Update for SystemLanguage Property
   * @param None
   * @return None
   */
   tVoid _vTriggerSystemLanguageServiceUpdate(tVoid) const;

   /**
   * @brief Trigger a Service/Property Update for LanguageSyncSource Property
   * @param None
   * @return None
   */
   tVoid _vTriggerLanguageSyncSourceServiceUpdate(tVoid) const;

   /**
   * @brief Save the system language and trigger DP update
   * @param u8Language
   * @return TRUE if system language is changed
   */
   tBool _bSaveSystemLanguage(const tU8 u8Language);

   /**
   * @brief Save the Language Sync Source and Update the DP
   * @param u8SourceId
   * @return
   */
   tVoid _vSaveLanguageSyncSource(const tU8 u8SourceId);

   /**
   * @brief Update the Language Sync Source DP
   * @param None
   * @return None
   */
   tVoid _vHandleLanguageSyncSourceDPUpdate(tVoid) const;

   /**
   * @brief Initialize the Source Table List with the Unused Default Languages
   * @param None
   * @return None
   */
   tVoid _vInitializeSourceTableList(tVoid);

   /**
   * @brief Saves the Language data for a particular source to persistent DP
   * @param u8Source
   * @return None
   */
   tVoid _vHandleDPSaveLangForSource(const tU8 u8Source) const;

   /**
    * Fetch the Language data for a particular source from persistent DP
    * @param u8Source
    * @return
    */
   tU8 _u8GetDPLangForSource(const tU8 u8Source) const;
   
   /**
    * Handle the Language Sync request from HMI
    * @param u8Request
    * @return
    */
   tVoid _vHandleMeterLanguageSyncRequest(const tU8 u8Request) const;
   /**
    * Handle the Language Update from Profile Manager
    * @param u8Request
    * @returne
    */
   tVoid _vUpdateLanguageProfile();

   //! List of supported languages
   bpstl::vector<tU8> vSupportedLanguages;

   //! List to be updated to the service handlers
   bpstl::vector<vehicle_fi_tcl_Language_SourceTable, bpstl::allocator<vehicle_fi_tcl_Language_SourceTable> > vLangSourceTable;

   //! Default Auto Mode according to the static data
   tBool _bDefaultAutoMode;
   //! Default KDS language
   tU8 _u8DefaultLanguage;   
   //! Final Decided System Language 
   tU8 _u8Language;
   //! The Source Component with the which the current system language is synced
   tU8 _u8LangSyncSource;

   Iveda_tclServiceVehicleIf* _poCCAServiceHdl;
   I_veda_tclClientVcan*      _poClientVcan;
};

#endif // __VEDA_LANGUAGE_HANLDER__H_
