/*****************************************************************************
* FILE:         veda_language_handler.cpp
* PROJECT:      G3G project
* SW-COMPONENT: VehicleData
*----------------------------------------------------------------------------
*
* DESCRIPTION:  Language handler for VehicleData
*
*----------------------------------------------------------------------------
* COPYRIGHT:    (c) 2015 Robert Bosch GmbH, Hildesheim
*****************************************************************************/


/******************************************************************************/
/* INCLUDES                                                                   */
/******************************************************************************/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#include "veda_appl_handler.h"
#include "veda_language_handler.h"
#include "I_veda_ServiceVehicleIf.h"
#include "Iveda_ClientVcan.h"
#include "LanguageList.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#include "dp_vd_vehicle_data_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VEHICLEDATA_LANGUAGE_HANDLER
#include "trcGenProj/Header/veda_language_handler.cpp.trc.h"
#endif


veda_tclLanguageHandler::veda_tclLanguageHandler()
: Iveda_tclLanguageHandler(NULL)
, _poCCAServiceHdl(NULL)
, _poClientVcan(NULL)
{
   ETG_TRACE_ERR(("veda_tclLanguageHandler() entered."));

   _bDefaultAutoMode = FALSE;
   _u8DefaultLanguage = static_cast<tU8>(vehicle_fi_tcl_e8_Language_Code::FI_EN_UNKNOWN);
   _u8Language = static_cast<tU8>(vehicle_fi_tcl_e8_Language_Code::FI_EN_UNKNOWN);

   //! Explicitly set the Language Sync Source as HMI in case Persistent data read fails
   _u8LangSyncSource = static_cast<tU8>(vehicle_fi_tcl_e8_Language_SourceId::FI_EN_HMI);

   vSupportedLanguages.clear();
   vLangSourceTable.clear();

   vLangSourceTable.reserve(10);

   //! default constructor should not be used
   NORMAL_M_ASSERT_ALWAYS();
}

veda_tclLanguageHandler::veda_tclLanguageHandler(veda_tclAppMain* poMainAppl)
: Iveda_tclLanguageHandler(poMainAppl)
, _poCCAServiceHdl(NULL)
, _poClientVcan(NULL)
{
   ETG_TRACE_USR1(("veda_tclLanguageHandler() entered."));

   _bDefaultAutoMode = FALSE;
   _u8DefaultLanguage = static_cast<tU8>(vehicle_fi_tcl_e8_Language_Code::FI_EN_UNKNOWN);
   _u8Language = static_cast<tU8>(vehicle_fi_tcl_e8_Language_Code::FI_EN_UNKNOWN);
   // by default sync source is kept as HMI, on KDS def lang basis it is either changed or kept the same
   _u8LangSyncSource = static_cast<tU8>(vehicle_fi_tcl_e8_Language_SourceId::FI_EN_HMI);

   vSupportedLanguages.clear();
   vLangSourceTable.clear();

   vLangSourceTable.reserve(10);
}

veda_tclLanguageHandler::~veda_tclLanguageHandler()
{
   ETG_TRACE_USR1(("~veda_tclLanguageHandler() Destructor entered."));

   _poCCAServiceHdl = NULL;
   _poClientVcan = NULL;
}

tVoid veda_tclLanguageHandler::vHandleMessage(TMsg* pMsg)
{
   ETG_TRACE_USR4(("veda_tclApplHandler::vHandleMessage() entered %u -> data: %d.", ETG_CENUM(vehfunc_tclBaseIf::ECmdTypes, pMsg->eCmd), pMsg->u.u32Data));
   switch(pMsg->eCmd)
   {
   case eSetLanguage:
      {
         vHandleSetLanguage(pMsg->u.au8Data[0], pMsg->u.au8Data[1]);
      }
      break;
   case eSetLanguageSyncSource:
      {
         vLanguageSyncTriggered(pMsg->u.u8Data);
      }
      break;
   case eLanguageSyncMenuRequested:
      {
         _vHandleMeterLanguageSyncRequest(pMsg->u.u8Data);
      }
      break;
   case eDefaultSetting:
      {
         _vSetDefaultLanguage();
      }
      break;
   default:
      break;
   }
}

tVoid veda_tclLanguageHandler::vHandleTraceMessage(const tUChar* puchData)
{
   VEDA_NULL_POINTER_CHECK(puchData);

   tU32 u32MsgCode = ((puchData[1]<<8) | puchData[2]);
   ETG_TRACE_USR4(("veda_tclApplHandler::vHandleTraceMessage(): trace command %d", u32MsgCode ));

   switch(u32MsgCode)
   {
   case VEHICLEDATA_VEDA_LANGUAGE_INPUT:
      {
         // Input the Language Set input from HMI System
         tU8 param1 = puchData[3];
         tU8 param2 = puchData[4];

         veda_tclBaseIf::TMsg pMsg;
         (tVoid)memset((tVoid*)&pMsg, OSAL_NULL, sizeof(pMsg));
         pMsg.eCmd = veda_tclBaseIf::eSetLanguage;
         strncpy(pMsg.strClassName, "Iveda_tclLanguageHandler", sizeof("Iveda_tclLanguageHandler"));
         pMsg.u.au8Data[0] = (tU8) param1;
         pMsg.u.au8Data[1] = (tU8) param2;

         veda_tclAppMain::theServer()->vPostInternalMessage(&pMsg);
      }
      break;

   default:
      break;
   }
}

tVoid veda_tclLanguageHandler::vGetReferences()
{
   ETG_TRACE_USR4(("veda_tclLanguageHandler::vGetReferences() entered."));

   _poCCAServiceHdl = dynamic_cast<Iveda_tclServiceVehicleIf*>(_cpoMain->getHandler("Iveda_tclServiceVehicleIf"));
   VEDA_NULL_POINTER_CHECK(_poCCAServiceHdl);
   
   _poClientVcan = dynamic_cast<I_veda_tclClientVcan*>(_cpoMain->getHandler("I_veda_tclClientVcan"));
   VEDA_NULL_POINTER_CHECK(_poClientVcan);
}

tVoid veda_tclLanguageHandler::vStartCommunication()
{
   ETG_TRACE_USR4(("veda_tclLanguageHandler::vStartCommunication() entered."));

   // Initialize the Member Variable Language Source Table list
   _vInitializeSourceTableList();

   // Read the KDS configured region
   tU8 u8Region = _u8ReadConfiguredRegion();

   // Populate the supported languages table according to the configured region (KDS)
   _vFetchRegionBasedSupportedLanguages(u8Region);

   // Load all the last mode values from persistent data pool
   _vReadPersistentData();

   _vTriggerSystemLanguageServiceUpdate();

   _vTriggerLanguageSyncSourceServiceUpdate();
}

tVoid veda_tclLanguageHandler::vTraceInfo()
{
   ETG_TRACE_USR4(("veda_tclLanguageHandler::vTraceInfo() entered."));

   // Trace the Supported Languages
   bpstl::vector<tU8>::const_iterator it = vSupportedLanguages.begin();
   for (; it != vSupportedLanguages.end(); ++it)
   {
      ETG_TRACE_USR1(("veda_tclLanguageHandler::vTraceInfo() Supported Language %u.", ETG_ENUM(VEDA_E8_LANGUAGE_CODE, *it)));
   }

   // Current state of the Language Source Table
   bpstl::vector<vehicle_fi_tcl_Language_SourceTable, bpstl::allocator<vehicle_fi_tcl_Language_SourceTable> >::const_iterator it2 = vLangSourceTable.begin();
   for (; it2 != vLangSourceTable.end(); ++it2)
   {
      ETG_TRACE_USR1(("veda_tclLanguageHandler::vTraceInfo() SrcId %u Language %u", \
         ETG_ENUM(VEDA_E8_LANGUAGE_SOURCEID, it2->enLangSrcId.enType), ETG_ENUM(VEDA_E8_LANGUAGE_CODE, it2->enLanguage.enType)));
   }

   // Default Auto Mode according to the static data
   ETG_TRACE_USR1(("veda_tclLanguageHandler::vTraceInfo() _bDefaultAutoMode %u", _bDefaultAutoMode));
   // Default KDS language
   ETG_TRACE_USR1(("veda_tclLanguageHandler::vTraceInfo() _u8DefaultLanguage %u", ETG_ENUM(VEDA_E8_LANGUAGE_CODE, _u8DefaultLanguage)));
   // The Source Component with the which the current system language is synced
   ETG_TRACE_USR1(("veda_tclLanguageHandler::vTraceInfo() _u8LangSyncSource %u", ETG_ENUM(VEDA_E8_LANGUAGE_SOURCEID, _u8LangSyncSource)));
}

tVoid veda_tclLanguageHandler::_vInitializeSourceTableList(tVoid)
{
   tU8 u8LangCode = static_cast<tU8>(vehicle_fi_tcl_e8_Language_Code::FI_EN_UNKNOWN);

   // Push default values for all the sources
   for (tU8 u8SourceId = static_cast<tU8>(HMI); u8SourceId < static_cast<tU8>(SOURCE_LIST_END); u8SourceId++)
   {
      (tVoid) _bHandleSourceLanguageUpdate(u8SourceId, u8LangCode);
   }
}

tU8 veda_tclLanguageHandler::_u8ReadConfiguredRegion(tVoid) const
{
   // Read the KDS Region
   tU8 u8Region = 0xFF;

   if ( DP_S32_NO_ERR == DP_s32GetConfigItem("VehicleInformation", "DestinationRegion1", &u8Region, 1) )
   {
      ETG_TRACE_USR1(("_u8ReadConfiguredRegion(): KDS Region is 0x%02x.", u8Region));
   }
   else
   {
      // Set the Default region as USA
      u8Region = USA;
      ETG_TRACE_ERR(("_u8ReadConfiguredRegion(): KDS Region Read Failed. Setting region as USA"));
   }

   return u8Region;
}

tVoid veda_tclLanguageHandler::_vFetchRegionBasedSupportedLanguages(tU8 u8Region)
{
#if (LANGUAGELIST_H_VERSION == 1)
   if ( u8Region <= KDS_REGION_TYPE_MAX )
   {
      // Read the default settings
      _u8DefaultLanguage = mapRegionAndLanguage[u8Region].u8DefaultLanguage;
      _bDefaultAutoMode = mapRegionAndLanguage[u8Region].bDefaultMode;

      // Read all the supported languages to the Member Variable List (Vector)
      for (tU8 i = 0; i < mapRegionAndLanguage[u8Region].u8SizeOfLangArray; i++)
      {
         tU8 u8TempLanguage = mapRegionAndLanguage[u8Region].pu8LangList[i];
         ETG_TRACE_USR1(("_vFetchRegionBasedSupportedLanguages(): Supported Language %u.", ETG_ENUM(VEDA_E8_LANGUAGE_CODE, u8TempLanguage)));
         vSupportedLanguages.push_back(u8TempLanguage);
      }
   }
#else
#pragma message ( "WARNING!! LANGUAGELIST_H_VERSION does not match. Language List will not be loaded" )
#endif
}

tVoid veda_tclLanguageHandler::_vReadPersistentData(tVoid)
{
   tU8 u8KDSLanguage = static_cast<tU8>(vehicle_fi_tcl_e8_Language_Code::FI_EN_UNSUPPORTED);
   vGetKDSDefaultLanguage(u8KDSLanguage);

   if ( static_cast<tU8>(vehicle_fi_tcl_e8_Language_Code::FI_EN_UNKNOWN) == u8KDSLanguage )
   {
      u8KDSLanguage = _u8DefaultLanguage;
   }

   // Read Language Sync Source data from Vehicle data DP
   dp_tclVehDataProfileLanguageSyncSource oLanguageSyncSource;
   if ( DP_U8_ELEM_STATUS_VALID == oLanguageSyncSource.u8GetElementStatus() )
   {
      // DP data Valid load into member variable
      _u8LangSyncSource = oLanguageSyncSource.tGetData();
      ETG_TRACE_USR1(("_vReadPersistentData(): LanguageSyncSource DP Read Success, SyncSource = %d", ETG_ENUM(VEDA_E8_LANGUAGE_SOURCEID, _u8LangSyncSource)));
   }
   else
   {
      _vHandleLanguageSyncSourceDPUpdate();
   }

   // Read the last mode language for Meter
   tU8 u8MeterLang = static_cast<tU8>(_u8GetDPLangForSource(static_cast<tU8>(METER)));

   if ( u8MeterLang == static_cast<tU8>(vehicle_fi_tcl_e8_Language_Code::FI_EN_UNKNOWN) )
   {
      u8MeterLang = static_cast<tU8>(vehicle_fi_tcl_e8_Language_Code::FI_EN_UNUSED);
   }
   (tVoid) _bHandleSourceLanguageUpdate(static_cast<tU8>(vehicle_fi_tcl_e8_Language_SourceId::FI_EN_METER), u8MeterLang);

   // Read the last mode language for HMI
   (tVoid) _bHandleSourceLanguageUpdate(static_cast<tU8>(vehicle_fi_tcl_e8_Language_SourceId::FI_EN_HMI), _u8GetDPLangForSource(static_cast<tU8>(HMI)));

   // Read the last mode language for SDS
   (tVoid) _bHandleSourceLanguageUpdate(vehicle_fi_tcl_e8_Language_SourceId::FI_EN_SDS, _u8GetDPLangForSource(SDS));

   // Read the last mode language for DIAG
   (tVoid) _bHandleSourceLanguageUpdate(vehicle_fi_tcl_e8_Language_SourceId::FI_EN_DIAG, _u8GetDPLangForSource(DIAG));

   // Read the last mode language for System
   tU8 u8Language = static_cast<tU8>( _u8GetDPLangForSource( static_cast<tU8>(SYSTEM) ) );

   // System language cannot not be any of the below
   // Also check if the read language is still supported, in case if the KDS region is changed meanwhile
   if ( ( u8Language == static_cast<tU8>(vehicle_fi_tcl_e8_Language_Code::FI_EN_UNKNOWN) ) ||
      ( u8Language == static_cast<tU8>(vehicle_fi_tcl_e8_Language_Code::FI_EN_UNUSED) ) ||
      ( u8Language == static_cast<tU8>(vehicle_fi_tcl_e8_Language_Code::FI_EN_UNSUPPORTED) ) ||
      ( FALSE == _bIsRequestedLanguageSupported(u8Language) ) )
   {
      // if the language is not supported than fall back to the default KDS Language
      _u8DefaultLanguage = u8KDSLanguage;
      u8Language = u8KDSLanguage;
   }

   (tVoid) _bHandleSourceLanguageUpdate(_u8LangSyncSource, u8Language);

   // Save the last mode language into the member variable
   (tVoid) _bSaveSystemLanguage(u8Language);
}

tVoid veda_tclLanguageHandler::vGetKDSDefaultLanguage(tU8& u8Language) const
{
   // Read KDS default Language data if cold start
   if ( DP_S32_NO_ERR != DP_s32GetConfigItem("SystemConfiguration1", "DefaultLanguage", &u8Language, 1) )
   {
      // If KDS read for Default language fails,
      // Set the language as the Default language known from the static data
      u8Language = _u8DefaultLanguage;
      ETG_TRACE_USR1(("vGetKDSDefaultLanguage(): KDS Default Language Read Error using static data %u", ETG_ENUM(VEDA_E8_LANGUAGE_CODE, u8Language)));
   }
   else
   {
      // NCG3D-31449: Fallback solution so that the HMI will not be blank even if the configured default language is not really supported
      if ( FALSE == _bIsRequestedLanguageSupported(u8Language) )
      {
         ETG_TRACE_FATAL(("veda_tclLanguageHandler: W A R N I N G KDS Default Language %u is not supported ignored. Instead using %u", \
                           ETG_ENUM(VEDA_E8_LANGUAGE_CODE, u8Language), ETG_ENUM(VEDA_E8_LANGUAGE_CODE, _u8DefaultLanguage)));
         u8Language = _u8DefaultLanguage;
      }
      ETG_TRACE_USR1(("vGetKDSDefaultLanguage(): Using KDS Default Language %u.", ETG_ENUM(VEDA_E8_LANGUAGE_CODE, u8Language)));
   }
}

tVoid veda_tclLanguageHandler::_vSetDefaultLanguage(tVoid)
{
   ETG_TRACE_USR1(("_vSetDefaultLanguage() entered."));

   // Read the KDS configured region
   tU8 u8Region = _u8ReadConfiguredRegion();

   // Populate the supported languages table according to the configured region (KDS)
   _vFetchRegionBasedSupportedLanguages(u8Region);

   tU8 u8Language;
   vGetKDSDefaultLanguage(u8Language);

   if ( static_cast<tU8>(vehicle_fi_tcl_e8_Language_Code::FI_EN_UNKNOWN) == u8Language )
   {
      u8Language = _u8DefaultLanguage;
   }
   
   vSetLanguageTriggered(static_cast<tU8>(vehicle_fi_tcl_e8_Language_SourceId::FI_EN_HMI), u8Language);
   vLanguageSyncTriggered(static_cast<tU8>(vehicle_fi_tcl_e8_Language_SourceId::FI_EN_HMI));   

   _vTriggerSystemLanguageServiceUpdate();

   _vTriggerLanguageSyncSourceServiceUpdate();
}

tVoid veda_tclLanguageHandler::vHandleSetLanguage(const tU8 u8SourceId, tU8 u8Language)
{
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
    if (METER == u8SourceId)
    {
       tBool bFound = FALSE;
       for (tU8 i = 0; i < sizeof(mapMeterLanguage)/sizeof(sMapMeterLanguage); i++)
       {
          if (mapMeterLanguage[i].u8MeterLanguage == u8Language)
          {
             ETG_TRACE_USR1(("veda_tclLanguageHandler()::vHandleMessage MeterLang -> SystemLang %d -> %d .", \
                                                       u8Language, mapMeterLanguage[i].u8SystemLanguage));
             u8Language = mapMeterLanguage[i].u8SystemLanguage;
             bFound = TRUE;
             break;
          }
       }
       
       if (bFound == FALSE)
       {
          u8Language = _u8DefaultLanguage;
          ETG_TRACE_FATAL(("veda_tclLanguageHandler()::vHandleMessage Unmapped value from Source %u", ETG_CENUM(enLangSources, u8SourceId)));
       }
    }
    else if (HMI == u8SourceId)
    {
       tBool bFound = FALSE;
       for (tU8 i = 0; i < sizeof(mapMeterLanguage)/sizeof(sMapMeterLanguage); i++)
       {
          if (mapMeterLanguage[i].u8SystemLanguage == u8Language)
          {
             ETG_TRACE_USR1(("veda_tclLanguageHandler()::vHandleMessage SystemLang -> MeterLang %d -> %d .", \
                                                       u8Language, mapMeterLanguage[i].u8MeterLanguage));
             VEDA_NULL_POINTER_CHECK(_poClientVcan);
             _poClientVcan->vSendCockpitLanguageUpdate(mapMeterLanguage[i].u8MeterLanguage);
             bFound = TRUE;
             break;
          }
       }
       
       if (bFound == FALSE)
       {
          u8Language = _u8DefaultLanguage;
          ETG_TRACE_FATAL(("veda_tclLanguageHandler()::vHandleMessage Unmapped value from Source %u", ETG_CENUM(enLangSources, u8SourceId)));
       }
    }
    else
    {
    }
#endif
    vSetLanguageTriggered(u8SourceId, u8Language);
}

tVoid veda_tclLanguageHandler::vSetLanguageTriggered(const tU8 u8SourceId, const tU8 u8Language)
{
   ETG_TRACE_USR1(("vSetLanguageTriggered() with SourceId %u and Language %u.", \
      ETG_ENUM(VEDA_E8_LANGUAGE_SOURCEID, u8SourceId), ETG_ENUM(VEDA_E8_LANGUAGE_CODE, u8Language)));

   // Store the Method Result as UnSuccessful by default
   tBool bSetLangSuccess = FALSE;
   tBool bSrcLangChanged = FALSE;
   tBool bSystemLangChanged = FALSE;

   // Update the (member variable list) Source Table
   bSrcLangChanged = _bHandleSourceLanguageUpdate(u8SourceId, u8Language);

   if (u8SourceId != vehicle_fi_tcl_e8_Language_SourceId::FI_EN_DIAG)
   {
		ETG_TRACE_USR1(("vSetLanguageTriggered(): Not diag 'if' entered"));
		// Check if the requested language is supported in the currently configured Region (Country)
		if ( TRUE == _bIsRequestedLanguageSupported(u8Language) )
		{
			// Trigger Method Result as Success/OK
			bSetLangSuccess = TRUE;

			// Update the System Language as well if the update requested from the Language Sync Source
			if ( u8SourceId == _u8LangSyncSource )
			{
				bSystemLangChanged = _bSaveSystemLanguage(u8Language);
			}
		}
		// System Language should not be updated but Method Result shall still be TRUE
		else if ( ( u8Language == vehicle_fi_tcl_e8_Language_Code::FI_EN_UNKNOWN ) ||
			( u8Language == vehicle_fi_tcl_e8_Language_Code::FI_EN_UNSUPPORTED ) ||
			( u8Language == vehicle_fi_tcl_e8_Language_Code::FI_EN_UNUSED ) )
		{
			// Trigger Method Result as Success/OK
			bSetLangSuccess = TRUE;
		}
		else
		{
			ETG_TRACE_USR1(("vSetLanguageTriggered() Language %u is not supported", ETG_ENUM(VEDA_E8_LANGUAGE_CODE, u8Language)));
		}
   }
   else if (u8SourceId == vehicle_fi_tcl_e8_Language_SourceId::FI_EN_DIAG)
   {
	   ETG_TRACE_USR1(("vSetLanguageTriggered(): diag 'else if' entered"));
	   bSetLangSuccess = TRUE;
   }

   // Trigger the Method Result
   VEDA_NULL_POINTER_CHECK(_poCCAServiceHdl);
   _poCCAServiceHdl->sendSetLanguageMResult(bSetLangSuccess);

   // update only if something has changed
   if ( bSrcLangChanged || bSystemLangChanged )
   {
	   _u8Language = u8Language;
       // Trigger a SystemLanguage property update
       _vTriggerSystemLanguageServiceUpdate();
   }
}

tVoid veda_tclLanguageHandler::vLanguageSyncTriggered(const tU8 u8SourceId)
{
   ETG_TRACE_USR1(("vLanguageSyncTriggered() with SourceId %u ", ETG_ENUM(VEDA_E8_LANGUAGE_SOURCEID, u8SourceId)));

   // Save the requested Language Sync Source into the member variable and DP
   _vSaveLanguageSyncSource(u8SourceId);

   // Get the current Language of the new Language Sync Source
   // update the system Language if the currently Language Sync
   tU8 u8Lang = _u8GetSourceLanguage(u8SourceId);

   // Check if the current Language of the new Language Sync Source is supported in the currently configured Region (Country)
   if ( _bIsRequestedLanguageSupported(u8Lang) )
   {
      // Update the System Language as well if the update requested from the Language Sync Source
      if ( _bSaveSystemLanguage(u8Lang) )
      {
          // Trigger a SystemLanguage property update
          _vTriggerSystemLanguageServiceUpdate();
      }
   }
}

tBool veda_tclLanguageHandler::_bHandleSourceLanguageUpdate(const tU8 u8SourceId, const tU8 u8Language)
{
    ETG_TRACE_USR4(("veda_tclLanguageHandler::_bHandleSourceLanguageUpdate update Source %u with Lang %u", \
            ETG_ENUM(VEDA_E8_LANGUAGE_SOURCEID, u8SourceId), ETG_ENUM(VEDA_E8_LANGUAGE_CODE, u8Language)));

    tBool bSourceFound = FALSE;
    tBool bChanged = FALSE;
    bpstl::vector<vehicle_fi_tcl_Language_SourceTable, bpstl::allocator<vehicle_fi_tcl_Language_SourceTable> >::iterator it = vLangSourceTable.begin();

    // Iterate through the Language Source table
    for ( ; it != vLangSourceTable.end(); it++ )
    {
        // Change the language for the particular source
        if ( it->enLangSrcId.enType == (vehicle_fi_tcl_e8_Language_SourceId::tenType) u8SourceId )
        {
            bSourceFound = TRUE;

            // Update the language only if it is changed
            if ( it->enLanguage.enType != static_cast<vehicle_fi_tcl_e8_Language_Code::tenType>(u8Language) )
            {
                it->enLanguage.enType = static_cast<vehicle_fi_tcl_e8_Language_Code::tenType>(u8Language);
                it->bHasChanged = TRUE;

                // Language is changed for a particular source, trigger persistent data update
                bChanged = TRUE;
            }
        }
        else
        {
            // All the elements not updated recently should be updates as FALSE
            it->bHasChanged = FALSE;
        }
    }

    // If a new source is updated, append to the existing table
    // No need to handle new sources To be checked - Reset Issue NCG3D-6549
    if ( !bSourceFound )
    {
        ETG_TRACE_USR4(("veda_tclLanguageHandler::_bHandleSourceLanguageUpdate pushing a new Source %u Lang %u", \
                ETG_ENUM(VEDA_E8_LANGUAGE_SOURCEID, u8SourceId), ETG_ENUM(VEDA_E8_LANGUAGE_CODE, u8Language)));

        // Push the new Source into the list
        vehicle_fi_tcl_Language_SourceTable oSourceMap;
        vLangSourceTable.push_back(oSourceMap);

        if ( it != vLangSourceTable.end() )
        {
            vLangSourceTable.back().enLangSrcId.enType = static_cast<vehicle_fi_tcl_e8_Language_SourceId::tenType>(u8SourceId);
            vLangSourceTable.back().enLanguage.enType = static_cast<vehicle_fi_tcl_e8_Language_Code::tenType>(u8Language);
            vLangSourceTable.back().bHasChanged = TRUE;
        }
    }

    // Update the DP if the language of the u8SourceId has changed
    if (bChanged)
    {
        _vHandleDPSaveLangForSource(u8SourceId);
    }

    return bChanged;
}

tVoid veda_tclLanguageHandler::_vTriggerSystemLanguageServiceUpdate(tVoid) const
{
   vehicle_fi_tcl_e8_Language_Code oLangCode;
   oLangCode.enType = (vehicle_fi_tcl_e8_Language_Code::tenType) _u8Language;
   ETG_TRACE_USR1(("_vTriggerSystemLanguageServiceUpdate() Language %u", ETG_ENUM(VEDA_E8_LANGUAGE_CODE, _u8Language)));

   // Trigger the Property Update
   VEDA_NULL_POINTER_CHECK(_poCCAServiceHdl);
   _poCCAServiceHdl->sendLanguageStatus(oLangCode, vLangSourceTable);
}

tVoid veda_tclLanguageHandler::_vTriggerLanguageSyncSourceServiceUpdate(tVoid) const
{
   vehicle_fi_tcl_e8_Language_SourceId oSyncSource;
   oSyncSource.enType = (vehicle_fi_tcl_e8_Language_SourceId::tenType) _u8LangSyncSource;

   //! Trigger the Property Update
   VEDA_NULL_POINTER_CHECK(_poCCAServiceHdl);
   _poCCAServiceHdl->sendLanguageSyncSourceStatus(oSyncSource);
}

tU8 veda_tclLanguageHandler::_u8GetSourceLanguage(const tU8 u8SourceId) const
{
   tU8 u8Lang = static_cast<tU8>(vehicle_fi_tcl_e8_Language_Code::FI_EN_UNKNOWN);

   bpstl::vector<vehicle_fi_tcl_Language_SourceTable, bpstl::allocator<vehicle_fi_tcl_Language_SourceTable> >::const_iterator it = vLangSourceTable.begin();

   for ( ; it != vLangSourceTable.end(); it++ )
   {
      //! Get The Language for the particular Source
      if ( it->enLangSrcId.enType == (vehicle_fi_tcl_e8_Language_SourceId::tenType) u8SourceId )
      {
         u8Lang = (tU8) it->enLanguage.enType;
         break;
      }
   }

   return u8Lang;
}

tBool veda_tclLanguageHandler::_bIsRequestedLanguageSupported(const tU8 u8Language) const
{
   // Not supported by default
   tBool bAvailable = FALSE;

   // Only if the requested language is found in the Supported Language table
   std::vector<tU8>::const_iterator it = ::std::find(vSupportedLanguages.begin(), vSupportedLanguages.end(), u8Language);
   if ( it != vSupportedLanguages.end() ) // Present in the list ??
   {
      bAvailable = TRUE;
   }

   return bAvailable;
}

tBool veda_tclLanguageHandler::_bSaveSystemLanguage(const tU8 u8Language)
{
   tBool bLangChanged = FALSE;

   ETG_TRACE_USR1(("_bSaveSystemLanguage() with Language as %u ", ETG_ENUM(VEDA_E8_LANGUAGE_CODE, u8Language)));

   // Set the Language only if changed
   if ( _u8Language != u8Language )
   {
       bLangChanged = TRUE;
      _u8Language = u8Language;
      _bHandleSourceLanguageUpdate(SYSTEM, u8Language);
   }

   return bLangChanged;
}

tVoid veda_tclLanguageHandler::_vSaveLanguageSyncSource(const tU8 u8SourceId)
{
   ETG_TRACE_USR1(("_vSaveLanguageSyncSource() with Language Sync Source as %u ", ETG_ENUM(VEDA_E8_LANGUAGE_SOURCEID, u8SourceId)));

   //! Set the Language only if changed
   if ( _u8LangSyncSource != u8SourceId )
   {
      _u8LangSyncSource = u8SourceId;
      _vHandleLanguageSyncSourceDPUpdate();
   }
}

tVoid veda_tclLanguageHandler::_vHandleLanguageSyncSourceDPUpdate(tVoid) const
{
   ETG_TRACE_USR1(("_vHandleLanguageSyncSourceDPUpdate() entered"));

   dp_tclVehDataProfileLanguageSyncSource oSyncSource;
   oSyncSource << _u8LangSyncSource;
}

tVoid veda_tclLanguageHandler::_vHandleDPSaveLangForSource(const tU8 u8Source) const
{
   tU8 u8Lang = _u8GetSourceLanguage(u8Source);

   ETG_TRACE_USR1(("_vHandleDPSaveLangForSource() entered for Source %u Lang %u", \
      ETG_ENUM(VEDA_E8_LANGUAGE_SOURCEID, u8Source), ETG_ENUM(VEDA_E8_LANGUAGE_CODE, u8Lang)));

   switch(u8Source)
   {
   case HMI:
      {
    	 dp_tclVehDataHMILanguage oHMILang;
         oHMILang << u8Lang;
      }
      break;
   case METER:
      {
         dp_tclVehDataMeterLanguage oMeterLang;
         oMeterLang << u8Lang;
      }
      break;
   case SDS:
      {
         dp_tclVehDataSDSLanguage oSDSLang;
         oSDSLang << u8Lang;
      }
      break;
   case SYSTEM:
      {
         dp_tclVehDataProfileLanguage oVehLang;
         oVehLang << u8Lang;
      }
      break;
   case DIAG:
     {
    	 dp_tclVehDataDiagLanguage oDiagLang;
    	 oDiagLang << u8Lang;
     }
     break;
   default:
      break;
   }
}

tU8 veda_tclLanguageHandler::_u8GetDPLangForSource(const tU8 u8Source) const
{
   tU8 u8Lang = static_cast<tU8>(vehicle_fi_tcl_e8_Language_Code::FI_EN_UNKNOWN);

   switch(u8Source)
   {
   case HMI:
      {
    	 dp_tclVehDataHMILanguage oHMILang;
         if (DP_U8_ELEM_STATUS_VALID == oHMILang.u8GetElementStatus())
         {
            oHMILang >> u8Lang;
         }
      }
      break;
   case METER:
      {
         dp_tclVehDataMeterLanguage oMeterLang;
         if (DP_U8_ELEM_STATUS_VALID == oMeterLang.u8GetElementStatus())
         {
            oMeterLang >> u8Lang;
         }
      }
      break;
   case SDS:
      {
         dp_tclVehDataSDSLanguage oSDSLang;
         if (DP_U8_ELEM_STATUS_VALID == oSDSLang.u8GetElementStatus())
         {
            oSDSLang >> u8Lang;
         }
      }
      break;
   case SYSTEM:
      {
         dp_tclVehDataProfileLanguage oVehLang;
         if (DP_U8_ELEM_STATUS_VALID == oVehLang.u8GetElementStatus())
         {
            oVehLang >> u8Lang;
         }
      }
      break;
   case DIAG:
      {
    	 dp_tclVehDataDiagLanguage oDiagLang;
         if (DP_U8_ELEM_STATUS_VALID == oDiagLang.u8GetElementStatus())
         {
        	 oDiagLang >> u8Lang;
         }
      }
      break;
   default:
      break;
   }

   ETG_TRACE_USR1(("_u8GetDPLangForSource() entered for Source %u Lang %u", \
      ETG_ENUM(VEDA_E8_LANGUAGE_SOURCEID, u8Source), ETG_ENUM(VEDA_E8_LANGUAGE_CODE, u8Lang)));

   return u8Lang;
}

tVoid veda_tclLanguageHandler::_vHandleMeterLanguageSyncRequest(const tU8 u8Request) const
{
   VEDA_NULL_POINTER_CHECK(_poClientVcan);
   _poClientVcan->vSendMeterLanguageSyncMenuUpdate(u8Request);
   
   VEDA_NULL_POINTER_CHECK(_poCCAServiceHdl);
   _poCCAServiceHdl->sendRequestMeterLanguageSyncMResult(TRUE);
}

tVoid veda_tclLanguageHandler::_vUpdateLanguageProfile()
{
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
   // Read Language Sync Source data from Vehicle data DP
   tU8 _u8Language = static_cast<tU8>(_u8GetDPLangForSource(static_cast<tU8>(SYSTEM)));

   if ( ( _u8Language == static_cast<tU8>(vehicle_fi_tcl_e8_Language_Code::FI_EN_UNKNOWN) ) ||
      ( _u8Language == static_cast<tU8>(vehicle_fi_tcl_e8_Language_Code::FI_EN_UNUSED) ) ||
      ( _u8Language == static_cast<tU8>(vehicle_fi_tcl_e8_Language_Code::FI_EN_UNSUPPORTED) ) ||
      ( FALSE == _bIsRequestedLanguageSupported(_u8Language) ) )
   {
	   vGetKDSDefaultLanguage(_u8Language);
   }

   dp_tclVehDataProfileLanguageSyncSource oLanguageSyncSource;
   if (DP_U8_ELEM_STATUS_VALID == oLanguageSyncSource.u8GetElementStatus())
   {
      // DP data Valid load into member variable
      _u8LangSyncSource = oLanguageSyncSource.tGetData();
      ETG_TRACE_USR1(("_vReadPersistentData(): LanguageSyncSource DP Read Success, SyncSource = %d", ETG_ENUM(VEDA_E8_LANGUAGE_SOURCEID, _u8LangSyncSource)));
   }
   else
   {
	  _u8LangSyncSource = static_cast<tU8>(vehicle_fi_tcl_e8_Language_SourceId::FI_EN_HMI);
   }

   vHandleSetLanguage(_u8LangSyncSource, _u8Language);

#endif
}
// EOF

