/**
 * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
 *               The reproduction, distribution and utilization of this file as well as the
 *               communication of its contents to others without express authorization is prohibited.
 *               Offenders will be held liable for the payment of damages.
 *               All rights reserved in the event of the grant of a patent, utility model or design.
 * @brief        class dp_tclDpPersPddAccess, which access to the PDD dynamical
 * @addtogroup   Datapool persistent
 * @{
 */

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include <dlfcn.h>
#include "dpIfGlob.h"
#include "dpPersPDDAcc.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS DP_TRACE_CLASS
#include "trcGenProj/Header/dpPersPDDAcc.cpp.trc.h"
#endif


 /************************************************************************
 |defines and macros (scope: global)
 |-----------------------------------------------------------------------*/
#define DP_TRACE_CLASS                              (tU16)(TR_COMP_DATAPOOL + 0x04)

#define DP_PDD_SHARED_FUNC_GET_SIZE                 "pdd_get_data_stream_size"
#define DP_PDD_SHARED_FUNC_READ_STREAM              "pdd_read_datastream"
#define DP_PDD_SHARED_FUNC_WRITE_STREAM             "pdd_write_datastream"
#define DP_PDD_SHARED_FUNC_DELETE_STREAM            "pdd_delete_data_stream"
#define DP_PDD_SHARED_FUNC_DELETE_OLDPOOL           "pdd_delete_old_DataPool"

#define DP_PDD_SHARED_FUNC_SYNC_SCC_STREAMS         "pdd_sync_scc_streams"
#define DP_PDD_SHARED_FUNC_SYNC_NOR_STREAMS         "pdd_sync_nor_user_streams"
#define DP_PDD_SHARED_FUNC_SYNC_RAW_EMMC_STREAMS    "pdd_sync_raw_emmc_user_streams"
#define DP_PDD_SHARED_FUNC_TRACE_COMMAND            "pdd_vTraceCommand"

#define DP_PDD_SHARED_LIB                           "/opt/bosch/base/lib/libpdd_so.so"
 /******************************************************************************/
 /* public function                                                            */
 /******************************************************************************/

 /******************************************************************************
 * dp_tclDpPersPddAccess::dp_tclDpPersPddAccess()
 *
 * DESCRIPTION: constructor
 *
 *****************************************************************************/
dp_tclDpPersPddAccess::dp_tclDpPersPddAccess()
{
    ETG_TRACE_USR1(("dp_tclDpPersPddAccess(): constructor process pid: '%d'", OSAL_ProcessWhoAmI()));
    vpHandlePdd = NULL;
    fpPddGetDataStreamSize = NULL;
    fpPddReadDataStream = NULL;
    fpPddWriteDataStream = NULL;
    fpPddDeleteDataStream = NULL;
    fpPddDeleteOldDataPool = NULL;
    fpPddSyncSccStreams = NULL;
    fpPddSyncNorStreams = NULL;
    fpPddTraceCmd = NULL;
    fpPddSyncRaweMMCStreams = NULL;
}
/******************************************************************************
* dp_tclDpPersPddAccess::~dp_tclDpPersPddAccess()
*
* DESCRIPTION: destructor
*
*****************************************************************************/
dp_tclDpPersPddAccess::~dp_tclDpPersPddAccess()
{
    ETG_TRACE_USR1(("dp_tclDpPersPddAccess(): destructor process pid: '%d'", OSAL_ProcessWhoAmI()));
    vpHandlePdd = NULL;
    fpPddGetDataStreamSize = NULL;
    fpPddReadDataStream = NULL;
    fpPddWriteDataStream = NULL;
    fpPddDeleteDataStream = NULL;
    fpPddDeleteOldDataPool = NULL;
    fpPddSyncSccStreams = NULL;
    fpPddSyncNorStreams = NULL;
    fpPddTraceCmd = NULL;
    fpPddSyncRaweMMCStreams = NULL;
}
/******************************************************************************
* FUNCTION: s32PddGetDataStreamSize()
*
* DESCRIPTION: get the dynamic address pointer for the function of the pdd
*              and call the function to get the size of the datastream
*
* PARAMETERS:
*      PtsDataStreamName:  the name of the datastream
*      PeLoctaion:         location from where the datastream should be read
*
* RETURNS:
*      positive value: no error
*      negative value: error code
*****************************************************************************/
tS32 dp_tclDpPersPddAccess::s32PddGetDataStreamSize(const char* PtsDataStreamName, eDpLocation PeLocation)
{
#ifndef DP_DATAPOOL_UNIT_TEST
    tS32 Vs32Return = s32LoadLibPdd();
    if (Vs32Return == DP_S32_NO_ERR)
    {
        if (fpPddGetDataStreamSize != NULL)
        {
            Vs32Return = fpPddGetDataStreamSize(PtsDataStreamName, eGetPddLocation(PeLocation));
        }
        else
        {
            Vs32Return = DP_S32_ERR_IO;
        }
    }
#else
    tS32 Vs32Return = pdd_get_data_stream_size(PtsDataStreamName, eGetPddLocation(PeLocation));
#endif
    return(Vs32Return);
}
/******************************************************************************
* FUNCTION: s32PddReadDataStream()
*
* DESCRIPTION: get the dynamic address pointer for the function of the pdd
*              and call the function to read the data of the datastream
*
* PARAMETERS:
*      PtsDataStreamName:  the name of the datastream
*      PeLocation:         location from where the datastream should be read
*      Ppu8ReadBuffer:     pointer for the read data
*      Ps32SizeReadBuffer: size of the read data
*      Pu32Version:        version of the datastream
*      Pu8Info:            pointer to get the information, if the data read from backup or not
*
* RETURNS:
*      positive value: read size
*      negative value: error code
*****************************************************************************/
tS32 dp_tclDpPersPddAccess::s32PddReadDataStream(const char* PtsDataStreamName, eDpLocation PeLocation, tU8 *Ppu8ReadBuffer, tS32 Ps32SizeReadBuffer, tU32 Pu32Version, tU8* Pu8Info)
{
#ifndef DP_DATAPOOL_UNIT_TEST
    tS32 Vs32Return = s32LoadLibPdd();
    if (Vs32Return == DP_S32_NO_ERR)
    {
        if (fpPddReadDataStream != NULL)
        {
            Vs32Return = fpPddReadDataStream(PtsDataStreamName, eGetPddLocation(PeLocation), Ppu8ReadBuffer, Ps32SizeReadBuffer, Pu32Version, Pu8Info);
        }
        else
        {
            Vs32Return = DP_S32_ERR_IO;
        }
    }
#else
    tS32 Vs32Return = pdd_read_datastream(PtsDataStreamName, eGetPddLocation(PeLocation), Ppu8ReadBuffer, Ps32SizeReadBuffer, Pu32Version, Pu8Info);
#endif
    return(Vs32Return);
}
/******************************************************************************
* FUNCTION: s32PddWriteDataStream()
*
* DESCRIPTION: get the dynamic address pointer for the function of the pdd
*              and call the function to write the data
*
* PARAMETERS:
*      PtsDataStreamName:    the name of the datastream
*      PeLocation:           location from where the datastream should be read
*      Ppu8WriteBuffer:      pointer for the write data
*      Ps32SizeBytestoWrite: size of the write data
*      Pu32Version:          version of the datastream
*      PbfSync:              information to the PDD, if the file should be sync or not
*
* RETURNS:
*      positive value: write size
*      negative value: error code
*****************************************************************************/
tS32 dp_tclDpPersPddAccess::s32PddWriteDataStream(const char* PtsDataStreamName, eDpLocation PeLocation, tU8 *Ppu8WriteBuffer, tS32 Ps32SizeBytestoWrite, tU32 Pu32Version, tBool PbfSync)
{
#ifndef DP_DATAPOOL_UNIT_TEST
    tS32 Vs32Return = s32LoadLibPdd();
    if (Vs32Return == DP_S32_NO_ERR)
    {
        if (fpPddWriteDataStream != NULL)
        {
            Vs32Return = fpPddWriteDataStream(PtsDataStreamName, eGetPddLocation(PeLocation), Ppu8WriteBuffer, Ps32SizeBytestoWrite, Pu32Version, PbfSync);
        }
        else
        {
            Vs32Return = DP_S32_ERR_IO;
        }
    }
#else
    tS32 Vs32Return = pdd_write_datastream(PtsDataStreamName, eGetPddLocation(PeLocation), Ppu8WriteBuffer, Ps32SizeBytestoWrite, Pu32Version, PbfSync);
#endif
    return(Vs32Return);
}
/******************************************************************************
* FUNCTION: s32PddDeleteDataStream()
*
* DESCRIPTION: get the dynamic address pointer for the function of the pdd
*              and call the function to delete a data stream
*
* PARAMETERS:
*      PtsDataStreamName:    the name of the datastream
*      PeLocation:           location from where the datastream should be read
*
* RETURNS:
*      positive value: success
*      negative value: error code
*****************************************************************************/
tS32 dp_tclDpPersPddAccess::s32PddDeleteDataStream(const char* PtsDataStreamName, eDpLocation PeLocation)
{
#ifndef DP_DATAPOOL_UNIT_TEST
    tS32 Vs32Return = s32LoadLibPdd();
    if (Vs32Return == DP_S32_NO_ERR)
    {
        if (fpPddDeleteDataStream != NULL)
        {
            Vs32Return = fpPddDeleteDataStream(PtsDataStreamName, eGetPddLocation(PeLocation));
        }
        else
        {
            Vs32Return = DP_S32_ERR_IO;
        }
    }
#else
    tS32  Vs32Return = pdd_delete_data_stream(PtsDataStreamName, eGetPddLocation(PeLocation));
#endif
    return(Vs32Return);
}
tS32 dp_tclDpPersPddAccess::s32PddDeleteOldDataPool(const char* PtsDataPool, eDpLocation PeLocation)
{
#ifndef DP_DATAPOOL_UNIT_TEST
    tS32 Vs32Return = s32LoadLibPdd();
    if (Vs32Return == DP_S32_NO_ERR)
    {
        if (fpPddDeleteOldDataPool != NULL)
        {
            Vs32Return = fpPddDeleteOldDataPool(PtsDataPool, eGetPddLocation(PeLocation));
        }
        else
        {
            Vs32Return = DP_S32_ERR_IO;
        }
    }
#else
    tS32  Vs32Return = pdd_delete_old_DataPool(PtsDataPool, eGetPddLocation(PeLocation));
#endif
    return(Vs32Return);
}
/******************************************************************************
* FUNCTION: s32PddSyncSccStreams()
*
* DESCRIPTION: get the dynamic address pointer for the function of the pdd
*              and call the function to sync all SCC streams
*
* RETURNS:
*      positive value: success
*      negative value: error code
*****************************************************************************/
tS32 dp_tclDpPersPddAccess::s32PddSyncSccStreams(void)
{
#ifndef DP_DATAPOOL_UNIT_TEST
    tS32 Vs32Return = s32LoadLibPdd();
    if (Vs32Return == DP_S32_NO_ERR)
    {
        if (fpPddSyncSccStreams != NULL)
        {
            fpPddSyncSccStreams();
        }
        else
        {
            Vs32Return = DP_S32_ERR_IO;
        }
    }
#else
    tS32  Vs32Return = DP_S32_NO_ERR;
    pdd_sync_scc_streams();
#endif
    return(Vs32Return);
}
/******************************************************************************
* FUNCTION: s32PddSyncNorStreams()
*
* DESCRIPTION: get the dynamic address pointer for the function of the pdd
*              and call the function to sync all NOR streams
*
* RETURNS:
*      positive value: success
*      negative value: error code
*****************************************************************************/
tS32 dp_tclDpPersPddAccess::s32PddSyncNorStreams(void)
{
#ifndef DP_DATAPOOL_UNIT_TEST
    tS32 Vs32Return = s32LoadLibPdd();
    if (Vs32Return == DP_S32_NO_ERR)
    {
        if (fpPddSyncNorStreams != NULL)
        {
            fpPddSyncNorStreams();
        }
        else
        {
            Vs32Return = DP_S32_ERR_IO;
        }
    }
#else
    tS32  Vs32Return = DP_S32_NO_ERR;
    pdd_sync_nor_user_streams();
#endif
    return(Vs32Return);
}
/******************************************************************************
* FUNCTION: s32PddSyncRaweMMCStreams()
*
* DESCRIPTION: get the dynamic address pointer for the function of the pdd
*              and call the function to sync all Raw eMMC streams
*
* RETURNS:
*      positive value: success
*      negative value: error code
*****************************************************************************/
tS32 dp_tclDpPersPddAccess::s32PddSyncRaweMMCStreams(void)
{
    tS32  Vs32Return = DP_S32_NO_ERR;
#ifdef PDD_RAWEMMC_EXIST
#ifndef DP_DATAPOOL_UNIT_TEST
    Vs32Return = s32LoadLibPdd();
    if (Vs32Return == DP_S32_NO_ERR)
    {
        if (fpPddSyncRaweMMCStreams != NULL)
        {
            fpPddSyncRaweMMCStreams();
        }
        else
        {
            Vs32Return = DP_S32_ERR_IO;
        }
    }
#else
    pdd_sync_raw_emmc_user_streams();
#endif
#endif
    return(Vs32Return);
}
/******************************************************************************
* FUNCTION: vPddTraceCommand()
*
* DESCRIPTION: get the dynamic pointer and call the PDD for trace commands to Putty
*
*****************************************************************************/
void dp_tclDpPersPddAccess::vPddTraceCommand(char* Pu8pData)
{
#ifndef DP_DATAPOOL_UNIT_TEST
    if (s32LoadLibPdd() == DP_S32_NO_ERR)
    {
        if (fpPddTraceCmd != NULL)
        {
            fpPddTraceCmd(Pu8pData);
        }
    }
#else
    pdd_vTraceCommand(Pu8pData);
#endif
}
/******************************************************************************
* FUNCTION: vDlClose()
*
* DESCRIPTION: call function dlclose to release resource
*
*****************************************************************************/
void dp_tclDpPersPddAccess::vDlClose(void)
{
    if (vpHandlePdd != NULL)
    {
        if (dlclose(vpHandlePdd) == 0)
        {
            ETG_TRACE_USR1(("dp_tclDpPersPddAccess::vDlClose(): success for process pid: '%d'", OSAL_ProcessWhoAmI()));
        }
        else
        {
            ETG_TRACE_FATAL(("dp_tclDpPersPddAccess::vDlClose(): error for process pid: '%d'", OSAL_ProcessWhoAmI()));
        }
        vpHandlePdd = NULL;
        fpPddGetDataStreamSize = NULL;
        fpPddReadDataStream = NULL;
        fpPddWriteDataStream = NULL;
        fpPddDeleteDataStream = NULL;
        fpPddDeleteOldDataPool = NULL;
        fpPddSyncSccStreams = NULL;
        fpPddSyncNorStreams = NULL;
        fpPddTraceCmd = NULL;
    }
}
/******************************************************************************/
/* private function                                                            */
/******************************************************************************/

/******************************************************************************
* FUNCTION: s32LoadLibPdd()
*
* DESCRIPTION: load library PDD and determine the function address pointer fro the PDD
*
* RETURNS: success or error code
*
*****************************************************************************/
tS32 dp_tclDpPersPddAccess::s32LoadLibPdd()
{
    tS32   Vs32ReturnCode = DP_S32_NO_ERR;

    if (vpHandlePdd == NULL)
    {
        dlerror();    /* Clear any existing error */
        vpHandlePdd = dlopen(DP_PDD_SHARED_LIB, RTLD_NOW);
        if (vpHandlePdd == NULL)
        {
            tPChar VpcErrorStr = dlerror();
            if (VpcErrorStr != NULL)
            {
                ETG_TRACE_ERRMEM(("dp_tclDpPersPddAccess::s32LoadLibPdd(): dlopen() fails & dlerror:'%s' ", VpcErrorStr));
            }
            else
            {
                ETG_TRACE_ERRMEM(("dp_tclDpPersPddAccess::s32LoadLibPdd(): dlopen() fails & dlerror returned NULL"));
            }
            Vs32ReturnCode = DP_S32_ERR_IO;
        }
        else
        {
            fpPddGetDataStreamSize = (tfpPddGetDataStreamSize)dlsym(vpHandlePdd, DP_PDD_SHARED_FUNC_GET_SIZE);                      /*lint !e611 */
            if (fpPddGetDataStreamSize == NULL)
            {
                vTraceErrorForDLsym(DP_PDD_SHARED_FUNC_GET_SIZE);
                Vs32ReturnCode = DP_S32_ERR_IO;
            }
            if (Vs32ReturnCode == DP_S32_NO_ERR)
            {
                fpPddReadDataStream = (tfpPddReadDataStream)dlsym(vpHandlePdd, DP_PDD_SHARED_FUNC_READ_STREAM);                     /*lint !e611 */
                if (fpPddReadDataStream == NULL)
                {
                    vTraceErrorForDLsym(DP_PDD_SHARED_FUNC_READ_STREAM);
                    Vs32ReturnCode = DP_S32_ERR_IO;
                }
            }
            if (Vs32ReturnCode == DP_S32_NO_ERR)
            {
                fpPddWriteDataStream = (tfpPddWriteDataStream)dlsym(vpHandlePdd, DP_PDD_SHARED_FUNC_WRITE_STREAM);                  /*lint !e611 */
                if (fpPddWriteDataStream == NULL)
                {
                    vTraceErrorForDLsym(DP_PDD_SHARED_FUNC_WRITE_STREAM);
                    Vs32ReturnCode = DP_S32_ERR_IO;
                }
            }
            if (Vs32ReturnCode == DP_S32_NO_ERR)
            {
                fpPddDeleteDataStream = (tfpPddDeleteDataStream)dlsym(vpHandlePdd, DP_PDD_SHARED_FUNC_DELETE_STREAM);               /*lint !e611 */
                if (fpPddDeleteDataStream == NULL)
                {
                    vTraceErrorForDLsym(DP_PDD_SHARED_FUNC_DELETE_STREAM);
                    Vs32ReturnCode = DP_S32_ERR_IO;
                }
            }
            if (Vs32ReturnCode == DP_S32_NO_ERR)
            {
                fpPddDeleteOldDataPool = (tfpPddDeleteOldDataPool)dlsym(vpHandlePdd, DP_PDD_SHARED_FUNC_DELETE_OLDPOOL);            /*lint !e611 */
                if (fpPddDeleteOldDataPool == NULL)
                {
                    vTraceErrorForDLsym(DP_PDD_SHARED_FUNC_DELETE_OLDPOOL);
                    Vs32ReturnCode = DP_S32_ERR_IO;
                }
            }
            if (Vs32ReturnCode == DP_S32_NO_ERR)
            {
                fpPddSyncSccStreams = (tfpPddSyncSccStreams)dlsym(vpHandlePdd, DP_PDD_SHARED_FUNC_SYNC_SCC_STREAMS);                /*lint !e611 */
                if (fpPddSyncSccStreams == NULL)
                {
                    vTraceErrorForDLsym(DP_PDD_SHARED_FUNC_SYNC_SCC_STREAMS);
                    Vs32ReturnCode = DP_S32_ERR_IO;
                }
            }
            if (Vs32ReturnCode == DP_S32_NO_ERR)
            {
                fpPddSyncNorStreams = (tfpPddSyncNorStreams)dlsym(vpHandlePdd, DP_PDD_SHARED_FUNC_SYNC_NOR_STREAMS);                /*lint !e611 */
                if (fpPddSyncNorStreams == NULL)
                {
                    vTraceErrorForDLsym(DP_PDD_SHARED_FUNC_SYNC_NOR_STREAMS);
                    Vs32ReturnCode = DP_S32_ERR_IO;
                }
            }
            if (Vs32ReturnCode == DP_S32_NO_ERR)
            {
                fpPddTraceCmd = (tfpPddTraceCmd)dlsym(vpHandlePdd, DP_PDD_SHARED_FUNC_TRACE_COMMAND);                               /*lint !e611 */
                if (fpPddTraceCmd == NULL)
                {
                    vTraceErrorForDLsym(DP_PDD_SHARED_FUNC_TRACE_COMMAND);
                    Vs32ReturnCode = DP_S32_ERR_IO;
                }
            }
            #ifdef PDD_RAWEMMC_EXIST
            if (Vs32ReturnCode == DP_S32_NO_ERR)
            {
                fpPddSyncRaweMMCStreams = (tfpPddSyncRaweMMCStreams)dlsym(vpHandlePdd, DP_PDD_SHARED_FUNC_SYNC_RAW_EMMC_STREAMS);   /*lint !e611 */
                if (fpPddSyncNorStreams == NULL)
                {
                    vTraceErrorForDLsym(DP_PDD_SHARED_FUNC_SYNC_RAW_EMMC_STREAMS);
                    Vs32ReturnCode = DP_S32_ERR_IO;
                }
            }
            #endif
        }
        if ((Vs32ReturnCode == DP_S32_ERR_IO) && (vpHandlePdd != NULL))
        {/*close handle*/
            dlclose(vpHandlePdd);
            vpHandlePdd = NULL;
        }
    }
    return(Vs32ReturnCode);
}
/******************************************************************************
* FUNCTION: vTraceErrorForDLsym()
*
* DESCRIPTION: trace error into error memory, if adress function could not be determine
*
* PARAMETERS:  PcpFunction: name of the function
*
*****************************************************************************/
void dp_tclDpPersPddAccess::vTraceErrorForDLsym(const tChar* PcpFunction)
{
    tPChar VpcErrorStr = dlerror();
    if (VpcErrorStr != NULL)
    {
        tChar   Vstr[255] = { 0 };
        snprintf(Vstr, sizeof(Vstr), "'%s' for function '%s'", VpcErrorStr, PcpFunction);
        ETG_TRACE_ERRMEM(("dp_tclDpPersPddAccess::s32LoadLibPdd(): dlsym() fails & dlerror:%s", Vstr));
    }
    else
    {
        ETG_TRACE_ERRMEM(("dp_tclDpPersPddAccess::s32LoadLibPdd(): dlsym() fails & dlerror returned NULL for function '%s' ", PcpFunction));
    }
}
/******************************************************************************
* FUNCTION: eGetPddLocation()
*
* DESCRIPTION: get the location value for the PDD
*
* PARAMETERS:
*      eLocation:        location enum for datapool
*
* RETURNS:
*      location value for PDD
*****************************************************************************/
tePddLocation dp_tclDpPersPddAccess::eGetPddLocation(eDpLocation eLocation)
{
    tePddLocation ePddLocation;
#ifdef OSAL_GEN3_SIM
    ePddLocation = PDD_LOCATION_FS;
#else
    switch (eLocation)
    {
    case eDpLocation_FILE_SYSTEM:
    {
        ePddLocation = PDD_LOCATION_FS;
    }break;
    case eDpLocation_FILE_SYSTEM_SECURE:
    {
        ePddLocation = PDD_LOCATION_FS_SECURE;
    }break;
    case eDpLocation_RAW_NOR:
    {
        ePddLocation = PDD_LOCATION_NOR_USER;
    }break;
    case eDpLocation_PDD_KERNEL:
    {
        ePddLocation = PDD_LOCATION_NOR_KERNEL;
    }break;
    case eDpLocation_V850:
    {
        ePddLocation = PDD_LOCATION_SCC;
    }break;
    case eDpLocation_RAW_EMMC:
#ifdef PDD_RAWEMMC_EXIST
    {
        ePddLocation = PDD_LOCATION_RAW_EMMC_USER;
    }break;
#endif
    default:
    {
        ePddLocation = PDD_LOCATION_FS;
    }break;
    }
#endif
    return(ePddLocation);
}
//EOF
