/**
 * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
 *               The reproduction, distribution and utilization of this file as well as the
 *               communication of its contents to others without express authorization is prohibited.
 *               Offenders will be held liable for the payment of damages.
 *               All rights reserved in the event of the grant of a patent, utility model or design.
 * @brief        Function for the access to the different flash location, of the persistent data.
 *               With the define OSAL_GEN3_SIM the persistent data saved all to the file system.
 * @addtogroup   Datapool persistent
 * @{
 */
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define DP_S_IMPORT_INTERFACE_DPPERSACCESS
#include "dp_if.h"
#include "dpPersPDDAcc.h"

#include "bpcl.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS DP_TRACE_CLASS
#include "trcGenProj/Header/dpPersMemAcc.cpp.trc.h"
#endif

/************************************************************************
|defines and macros (scope: global)
|-----------------------------------------------------------------------*/
#define DP_TRACE_CLASS                   (tU16)(TR_COMP_DATAPOOL + 0x04)

 /************************************************************************
 |static variable process global
 |-----------------------------------------------------------------------*/
static dp_tclDpPersPddAccess vObjDpPdd;

/******************************************************************************/
/* public function                                                            */
/******************************************************************************/
extern tVoid vFillStreamFromRawData(tCString pSourceFilename, void* vData, tU8** pOutData, tU32& ru32OutLength);
extern tVoid vFillRawDataFromStream(tCString pSourceFilename, tU8* pData, tU32 u32Length, tU8** pOutData, tU32& ru32OutLength);

/******************************************************************************
* dp_tclDpPersMemAccess::dp_tclDpPersMemAccess()
*
* DESCRIPTION: constructor
*
*****************************************************************************/
dp_tclDpPersMemAccess::dp_tclDpPersMemAccess()
{
    ETG_TRACE_USR1(("dp_tclDpPersMemAccess(): constructor process pid: '%d'", OSAL_ProcessWhoAmI()));
}
/******************************************************************************
* dp_tclDpPersMemAccess::~dp_tclDpPersMemAccess()
*
* DESCRIPTION: destructor
*
*****************************************************************************/
dp_tclDpPersMemAccess::~dp_tclDpPersMemAccess()
{
    ETG_TRACE_USR1(("dp_tclDpPersMemAccess(): destructor process pid: '%d'", OSAL_ProcessWhoAmI()));
    vObjDpPdd.vDlClose();
}
/******************************************************************************
* FUNCTION: s32GetDataStreamName()
*
* DESCRIPTION: get the name of the file. For the simulation the whole path
* returns. For the PDD access strName=strReferenceName
*
* PARAMETERS:
*      u16Pool:          pool identification
*      strName:          name of the pool
*      strReferenceName: name(and path)of the file
*
* RETURNS:
*      positive value: no error
*      negative value: error code
*****************************************************************************/
tS32  dp_tclDpPersMemAccess::s32GetDataStreamName(tU16 u16Pool, const tChar* strName, tChar* strReferenceName)
{
    tS32 Vs32ReturnCode = 0;
    /*check parameter*/
    if ((strName != NULL) && (strReferenceName != NULL))
    { /*set reference buffer to zero*/
        tChar strAppId[8] = { 0 };
        tS32   ViLen = 0;
        memset(strReferenceName, 0, DP_MAX_LEN_FILE);
        OSAL_s32PrintFormat(strAppId, "%04x", u16Pool);
        /*add id as folder*/
        ViLen = ((tS32)strlen(strReferenceName));
        OSAL_szStringNConcat((tString)strReferenceName, (tString)strAppId, DP_MAX_LEN_FILE - ViLen);
        ViLen = ((tS32)strlen(strReferenceName));
        OSAL_szStringNConcat((tString)strReferenceName, "/", DP_MAX_LEN_FILE - ViLen);
        /*add name*/
        ViLen = ((tS32)strlen(strReferenceName));
        OSAL_szStringNConcat((tString)strReferenceName, strName, DP_MAX_LEN_FILE - ViLen);
    }
    else
    { /*error */
        Vs32ReturnCode = -1;
    }
    return(Vs32ReturnCode);
}
/******************************************************************************
* FUNCTION: s32GetPersDataSize()
*
* DESCRIPTION: get the size of one datastream.
* This isn't necessary for the simulation
*
* PARAMETERS:
*      pSourceFilename:  the name of the datastream
*      eLocation:        location from where the datastream should be read
*
* RETURNS:
*      positive value: no error
*      negative value: error code
*****************************************************************************/
tS32  dp_tclDpPersMemAccess::s32GetPersDataSize(tCString pSourceFilename, eDpLocation eLocation)
{ /* check parmeter */
    tS32 Vs32Size = -1;
    if (pSourceFilename != NULL)
    { /*get location*/
        Vs32Size = vObjDpPdd.s32PddGetDataStreamSize(pSourceFilename, eLocation);
        if (Vs32Size < 0)
            ETG_TRACE_USR1(("s32GetPersDataSize(): Error s32PddGetDataStreamSize returns  %d", Vs32Size));
    }
    return(Vs32Size);
}
/******************************************************************************
* FUNCTION: s32ImportPersData()
*
* DESCRIPTION: import peristent data stream from flash.
*
* PARAMETERS:
*      pSourceFilename:  the name of the datastream
*      pu8Data:          pointer for the read data
*      eLocation:        location from where the datastream should be read
*      u32Version:       version of the datastream
*
* RETURNS:
*      positive value: no error
*      negative value: error code
*****************************************************************************/
tS32  dp_tclDpPersMemAccess::s32ImportPersData(tCString pSourceFilename, tU8 **pu8Data, eDpLocation eLocation, tU32 u32Version)
{
    tS32 s32Result = -1;
    /*check parameter*/
    if ((pSourceFilename != NULL) && (pu8Data != NULL))
    {
        /*get size */
        tS32 Vs32Size = vObjDpPdd.s32PddGetDataStreamSize(pSourceFilename, eLocation);
        if (Vs32Size > 0)
        {/*allocate buffer => delete buffer in upper function*/
            tU8* pu8Buffer;
            try
            {
                pu8Buffer = new tU8[(tU32)Vs32Size];
            }
            catch (const std::bad_alloc&)
            {
                pu8Buffer = NULL;
                ETG_TRACE_ERR(("s32ImportPersData():Unable to allocate %s ", Vs32Size));
            }

            if (pu8Buffer != NULL)
            {
                tU8 Vu8Info = PDD_READ_INFO_NORMAL_FILE;
                s32Result = vObjDpPdd.s32PddReadDataStream(pSourceFilename, eLocation, pu8Buffer, Vs32Size, u32Version, &Vu8Info);
                if (s32Result > PDD_OK)
                {
                    ETG_TRACE_USR1(("s32ImportPersData(): File found: %d bytes read -> '%s' loaded.", s32Result, pSourceFilename));
                    if ((eLocation == eDpLocation_V850) || (eLocation == eDpLocation_PDD_KERNEL))
                    {
                        tU8* pData = NULL;
                        tU32 u32Len = 0;
                        ETG_TRACE_USR1(("s32ImportPersData(): for fill stream: %d bytes read -> '%s' loaded.", s32Result, pSourceFilename));
                        vFillStreamFromRawData(pSourceFilename, pu8Buffer, &pData, u32Len);
                        s32Result = (tS32)u32Len;
                        ETG_TRACE_USR1(("s32ImportPersData(): after fill stream: %d bytes read -> '%s' loaded.", s32Result, pSourceFilename));
                        if (pData != NULL)
                        {
                            delete[] pu8Buffer;
                            pu8Buffer = NULL;  /*Bug 123144: Safe handling*/
                        }
                        pu8Buffer = pData;
                    }
                    ETG_TRACE_USR1(("s32ImportPersData():%02x", ETG_LIST_LEN((s32Result)), ETG_LIST_PTR_T8(pu8Buffer)));
                }
                else
                {
                    ETG_TRACE_USR1(("s32ImportPersData(): Error s32PddReadDataStream returns  %d", s32Result));
                }
                *pu8Data = pu8Buffer;
            }/*if buffer != NULL*/
        }
        else
            ETG_TRACE_USR1(("s32ImportPersData(): Error s32PddGetDataStreamSize returns  %d", Vs32Size));
    }
    return s32Result;
}
/******************************************************************************
* FUNCTION: bWriteFfsData()
*
* DESCRIPTION: write peristent data stream to flash.
*
* PARAMETERS:
*      pSourceFilename:  the name of the datastream
*      pu8MyData:        pointer for the write data
*      s32Size:          size of the write data
*      eLocation:        location where the datastream should be write
*      u32Version:       version of the datastream
*      bfSync:           information to the PDD, if the file should be sync or not
*
* RETURNS:
*      TRUE: no error
*      FALSE: error
*****************************************************************************/
tBool dp_tclDpPersMemAccess::bWriteFfsData(tCString pSourceFilename, tU8 *pu8MyData, tU32 u32Size, eDpLocation eLocation, tU32 u32Version, tBool bfSync)
{
    tBool bRet = FALSE;
    /*check parameter*/
    if ((pSourceFilename != NULL) && (pu8MyData != NULL))
    {
        /*get location*/
        tU8*          pData = NULL;//pu8MyData;
        tU32          u32FileLen = u32Size;
        tS32          Vs32Length = -1;

        ETG_TRACE_USR1(("bWriteFfsData(): %02x", ETG_LIST_LEN((u32Size)), ETG_LIST_PTR_T8(pu8MyData)));
        if ((eLocation == eDpLocation_V850) || (eLocation == eDpLocation_PDD_KERNEL))
        {
            vFillRawDataFromStream(pSourceFilename, pu8MyData, u32Size, &pData, u32FileLen);
            ETG_TRACE_USR1(("bWriteFfsData(): after vFillRawDataFromStream %02x", ETG_LIST_LEN((u32FileLen)), ETG_LIST_PTR_T8(pData)));
        }
        else
        {
            pData = pu8MyData;
        }
         
        if (pData != NULL)
        { /*write to pdd*/
            Vs32Length = vObjDpPdd.s32PddWriteDataStream(pSourceFilename, eLocation, pData, (tS32)u32FileLen, u32Version, bfSync);
        }
        /*delete buffer*/
        if (((eLocation == eDpLocation_V850) || (eLocation == eDpLocation_PDD_KERNEL)) && (pData != NULL))
        {
            delete[] pData;
            pData = NULL; /*Bug 123144: Safe handling*/
        }
        /*error*/
        if (Vs32Length >= 0)
        {
            bRet = TRUE;
        }
        else
        {
            ETG_TRACE_FATAL(("bWriteFfsData(): s32PddWriteDataStream returns  %d", Vs32Length));
        }
    }
    return bRet;
}
/******************************************************************************
* FUNCTION: bCalcHashForFile()
*
* DESCRIPTION: read size of a datastream, the datastrean and calc the hash value
*
* PARAMETERS:
*      pSourceFilename:  the name of the datastream
*      pu32MyData:       pointer for get the hash value
*      eLocation:        location from where the datastream should be read
*      u32Version:       version of the datastream
*
* RETURNS:
*      TRUE: no error
*      FALSE: error
*****************************************************************************/
tBool dp_tclDpPersMemAccess::bCalcHashForFile(tCString pSourceFilename, tU32 *pu32MyData, eDpLocation eLocation, tU32 u32Version)
{
    tBool bRet = FALSE;
    tS32 s32Result;
    /* check parameter*/
    if ((pSourceFilename != NULL) && (pu32MyData != NULL))
    {/* get size*/
        s32Result = vObjDpPdd.s32PddGetDataStreamSize(pSourceFilename, eLocation);
        if (s32Result < 0)
        {
            ETG_TRACE_ERR(("dp_tclDatapool::bCalcHashForFile(): data dosen't exist '%s' -> must be created.", pSourceFilename));
        }
        if (s32Result > 0)
        {
            /*read datastream*/
            tU8* pu8FileBuffer = NULL;
            try
            {
                pu8FileBuffer = new tU8[(tU32)s32Result];
            }
            catch (const std::bad_alloc&)
            {
                pu8FileBuffer = NULL;
                ETG_TRACE_ERR(("dp_tclDatapool::bCalcHashForFile(): Unable to allocate %d.", s32Result));
            }
            if (pu8FileBuffer)
            {/* read data*/
                tU8 Vu8Info = PDD_READ_INFO_NORMAL_FILE;
                tS32 s32ReadLen = vObjDpPdd.s32PddReadDataStream(pSourceFilename, eLocation, pu8FileBuffer, s32Result, u32Version, &Vu8Info);
                if (s32ReadLen > 0)
                {//calc hash
                    bRet = bGenHash(pu8FileBuffer, s32ReadLen, pu32MyData);
                }
                else
                {
                    ETG_TRACE_USR1(("bCalcHashForFile(): Error s32PddReadDataStream returns  %d", s32ReadLen));
                }
                delete[] pu8FileBuffer;
                pu8FileBuffer = NULL; /*Bug 123144: Safe handling*/
            }
        }
    }
    return(bRet);
}
/******************************************************************************
* FUNCTION: bGenHash()
*
* DESCRIPTION: get the hash of a buffer
*
* PARAMETERS:
*      pData:            data buffer
*      u32DataLen:       length of the data
*      pu32MyData:       pointer for get the hash value
*
* RETURNS:
*      TRUE: no error
*      FALSE: error
*****************************************************************************/
tBool dp_tclDpPersMemAccess::bGenHash(tU8* pData, tU32 u32DataLen, tU32 *pu32MyData)
{
    tBool bRet = FALSE;
    /*check parameter*/
    if ((pData != NULL) && (u32DataLen > 0) && (pu32MyData != NULL))
    {// create hashing context  SHA: Secure Hash Algorithm
        tU8* p = pData;
        tsSHA1Context shaCtx;
        tErrCode err = BPCL_SHA1_Init(&shaCtx);
        if (BPCL_OK == err)
        { // Load content for hashing
            err = BPCL_SHA1_Update(&shaCtx, p, u32DataLen);
            if (BPCL_OK == err)
            {// get hash
                err = BPCL_SHA1_Finish(&shaCtx, pu32MyData);
                if (BPCL_OK == err)
                {
                    bRet = TRUE;
                }
            }
        }
    }
    return(bRet);
}
/******************************************************************************
* FUNCTION: vSynchronizePersDataWithBackup()
*
* DESCRIPTION: Synchronize persistent data with backup
*             (for pools with the location SCC(V850) and NOR_USER.
*****************************************************************************/
void dp_tclDpPersMemAccess::vSynchronizePersDataWithBackup(void)
{
#ifndef OSAL_GEN3_SIM
    /*sync for V850/SCC datastreams*/
    vObjDpPdd.s32PddSyncSccStreams();

#ifdef OSAL_GEN4

    /*sync for raw emmc user streams*/
    vObjDpPdd.s32PddSyncRaweMMCStreams();
#else
    /*sync for raw nor user streams*/
    vObjDpPdd.s32PddSyncNorStreams();
#endif /*OSAL_GEN4 */

#endif /* OSAL_GEN3_SIM */
}
/******************************************************************************
* FUNCTION: s32DeletePersData()
*
* DESCRIPTION: delete persistent data
*
* PARAMETERS:
*      pSourceFilename:  the name of the datastream
*      eLocation:        location from where the datastream should be delete
*
* RETURNS:
*      TRUE: no error
*      FALSE: error
*
* HISTORY: Created 14.01.15
*****************************************************************************/
tS32  dp_tclDpPersMemAccess::s32DeletePersData(tCString pSourceFilename, eDpLocation eLocation)
{
    tS32 Vs32Result = DP_S32_ERR_UNKNOWN;
    /*check parameter*/
    if (pSourceFilename != NULL)
    {
        ETG_TRACE_USR1(("s32DeletePersData(): delete file: '%s' ", pSourceFilename));
        /*get location*/
        Vs32Result = vObjDpPdd.s32PddDeleteDataStream(pSourceFilename, eLocation);
    }
    return(Vs32Result);
}

tS32  dp_tclDpPersMemAccess::s32DeleteOldPool(tCString pSourceFilename, eDpLocation eLocation)
{
    tS32 Vs32Result = DP_S32_ERR_UNKNOWN;
    /*check parameter*/
    if (pSourceFilename != NULL)
    {
        ETG_TRACE_USR1(("s32DeletePersData(): delete Folder: '%s' ", pSourceFilename));
        /*get location*/
        Vs32Result = vObjDpPdd.s32PddDeleteOldDataPool(pSourceFilename, eLocation);
    }
    return(Vs32Result);
}
//EOF
