/**
 * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
 *               The reproduction, distribution and utilization of this file as well as the
 *               communication of its contents to others without express authorization is prohibited.
 *               Offenders will be held liable for the payment of damages.
 *               All rights reserved in the event of the grant of a patent, utility model or design.
 * @brief        This class managed the information about the end user
 *               If the pool "DpEndUserMode" is defined in the global project configuration
 *               the feature is enabled (define DP_U32_POOL_ID_DPENDUSERMODE)
 *               If the first process access to an end user pool, this class read the 
 *               pool infomation from file to shared memory. Then all processes have access 
 *               to this information. The pool "DpEndUserMode" isn't available in any process.           
 *               If set the user information, this modul save the pool information to the file. 
 * @addtogroup   Datapool end user
 * @{
 */
#include <fcntl.h>	
#include <unistd.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <errno.h>
#include <grp.h>
#include <errno.h>
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC		   
#include "osal_if.h"

#define DP_S_IMPORT_INTERFACE_BASE
#define DP_S_IMPORT_INTERFACE_DPSEM
#define DP_S_IMPORT_INTERFACE_DPENDUSER
#include "dp_if.h"

#include "ccaservice.h"

#ifdef DP_U32_POOL_ID_DPENDUSERMODE  //feature is only possible if pool for user mode exist

//for trace
#define DP_END_USER_TRACE_CLASS              (tU16)(TR_COMP_DATAPOOL + 0x06) 
//pool information 
#define DP_END_USER_STR_POOL_FILE_NAME    "effe/DpEndUserMode"
#define DP_END_USER_POOL_LOCATION         eDpLocation_FILE_SYSTEM_SECURE
#define DP_END_USER_POOL_VERSION          0x0001
/*configuration files*/
#define DP_CONFIG_FILE_DEFAULT            "/etc/datapool/datapool-default.conf"
#define DP_CONFIG_FILE                    "/etc/datapool/datapool.conf"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS DP_END_USER_TRACE_CLASS
#include "trcGenProj/Header/dpEndUser.cpp.trc.h"
#endif

/******************************************************************************* 
|defines and macros 
|------------------------------------------------------------------------------*/
/* name for semaphore*/
#define DP_SEM_NAME_END_USER_INIT         "DP-SemEUInit"
#define DP_SEM_NAME_END_USER_INIT_IN_SHM  "/dev/shm/sem.DP-SemEUInit"
/* name for shared memory*/
#define DP_SHM_NAME_END_USER              "DP-ShmEU"

/******************************************************************************/
/* static  variable                                                           */
/******************************************************************************/
dp_tclDpEndUser* dp_tclDpEndUser::_poInstance=NULL;

// *****************************************************************************
// ********   initialization part 
// *****************************************************************************

/******************************************************************************
* dp_tclDpEndUser::dp_tclDpEndUser() 
*
* DESCRIPTION: constructor  
*
* HISTORY: Created 2014 12 16
*****************************************************************************/
dp_tclDpEndUser::dp_tclDpEndUser()
{
   _ptDPShmEndUser=NULL;
   _poPersMemAccessHdl=NULL;
}
/******************************************************************************
* dp_tclDpEndUser::~dp_tclDpEndUser() 
*
* DESCRIPTION: destructor  
*
* HISTORY: Created 2014 12 16
*****************************************************************************/
dp_tclDpEndUser::~dp_tclDpEndUser() 
{
  ETG_TRACE_USR4(("dp_tclDpEndUser(): destructor"));
  if(_poPersMemAccessHdl!=NULL)
    _poPersMemAccessHdl=NULL;
}
/******************************************************************************
* dp_tclDpEndUser::dp_tclDpEndUser() 
*
* DESCRIPTION: get the pointer of the instance
*     
* RETURNS: pointer of the instance
*
* HISTORY: Created 2014 12 16
*****************************************************************************/
dp_tclDpEndUser* dp_tclDpEndUser::pGetInstance()
{
   if (_poInstance == NULL)
   {
      _poInstance = new dp_tclDpEndUser();
   }
   return _poInstance;
}
/******************************************************************************
* void dp_tclDpEndUser::vInitInstance(dp_tclDpPersMemAccess* poPersMem)
*
* DESCRIPTION: the first process creates the shared memory and the semaphore 
*              to protect the shared memory
*
* PARAMETERS:  poPersMem access handle for persistent access
*
* RETURNS: void  
*
* HISTORY: Created 2014 12 16
*****************************************************************************/
void dp_tclDpEndUser::vInitInstance(dp_tclDpPersMemAccess* poPersMem)
{
    _poPersMemAccessHdl = poPersMem; 
    /*if shared memory pointer not valid*/
    if (_ptDPShmEndUser==NULL)
    { 
        tBool       VbFirstAttach = TRUE;
        sem_t*      VSemLockInit;      
        tS32           VShmId;  
        dp_tclDpSem VmyDpSem;

        /* protect against initializing race, the second process has to wait until this is finished*/
        /* Linux Programmer's Manual: 
        If O_CREAT is specified in oflag, then the semaphore is created if it does not
        already exist. If both O_CREAT and O_EXCL are
        specified in oflag, then an error is returned if a semaphore with the given
        name already exists.*/
        /* try to create semaphore with name;(if success semaphore's value is set to zeror)*/
        VSemLockInit = sem_open(DP_SEM_NAME_END_USER_INIT, O_EXCL | O_CREAT, VmyDpSem.s32GetAccessRight(), 0);
        if(VSemLockInit != SEM_FAILED)
        { /* no error; semaphore create => first attach */
            /* first attach */
            VbFirstAttach = TRUE;
            /* create shared memory at first process attach */
            VShmId= shm_open(DP_SHM_NAME_END_USER, O_EXCL|O_RDWR|O_CREAT|O_TRUNC, VmyDpSem.s32GetAccessRight());
            if(VShmId != -1)
            {
                if(ftruncate(VShmId, sizeof(tDpShmEndUser)) == -1)
                {
                    VShmId=-1;
                }
            }
            else
               ETG_TRACE_ERR(("dp_tclDpEndUser(): vInitInstance(): shm_open failed %d",errno));
            
        }
        else
        { /* open semaphore*/
            VSemLockInit= sem_open(DP_SEM_NAME_END_USER_INIT, 0);
            if(VSemLockInit != SEM_FAILED)
            {
                /* lock semaphore*/
                if(sem_wait(VSemLockInit) != 0)
                    ETG_TRACE_USR4(("dp_tclDpEndUser(): vInitInstance(): sem_wait %d",errno));//coverity error    
                /* not first attach*/
                VbFirstAttach = FALSE;    
                /* open shared memory*/
                VShmId = shm_open(DP_SHM_NAME_END_USER, O_RDWR ,0); 
                if(VShmId == -1)
                    ETG_TRACE_ERR(("dp_tclDpEndUser(): vInitInstance(): shm_open without create failed %d",errno));                    
            }
            else
            {
                VShmId=-1;
                ETG_TRACE_ERR(("dp_tclDpEndUser(): vInitInstance(): sem_open without create failed %d",errno));
            }
        }
        if(VShmId!=-1)
        { /* map global shared memory into address space */
            _ptDPShmEndUser = (tDpShmEndUser*) mmap( NULL,sizeof(tDpShmEndUser),PROT_READ | PROT_WRITE,MAP_SHARED,VShmId,0);       
            /* if no error */
            if(_ptDPShmEndUser != MAP_FAILED)
            { /* if first proccess access */
                if(VbFirstAttach)
                { /*set all to zero*/                      
                    memset(_ptDPShmEndUser,0,sizeof(tDpShmEndUser));    
                    /* create semaphore for access shm property*/          
                    VmyDpSem.vCreateSemShm(&_ptDPShmEndUser->semShmEndUser);            
                    /*set access rights to Group "eco_pdd*/
                    VmyDpSem.s32ChangeGroupAccess(DP_KIND_RES_SHM,VShmId,"VShmId");       
                    VmyDpSem.s32ChangeGroupAccess(DP_KIND_RES_SEM,-1,DP_SEM_NAME_END_USER_INIT_IN_SHM);              
                    /*set values in shared memeory to default*/
                    _ptDPShmEndUser->bLoad=FALSE;
                    _ptDPShmEndUser->u16EndUserConfigNotStore=DP_END_USER_NOT_STORED_FEATURE_DISABLED;
                    vSetEndUserInfoInSharedMemToDefault();                
                }        
            }
            /* init finished, let's go! */
            sem_post(VSemLockInit);
        }
    }
}

/******************************************************************************
* tS32 dp_tclDpEndUser::s32GetEndUser()
*
* DESCRIPTION: get current end user
*
* PARAMETERS:  Pu8CurrentEndUser: current end user information
*
* RETURNS: tS32 error code
*
* HISTORY: Created 2015 01 21
*****************************************************************************/
tS32 dp_tclDpEndUser::s32GetEndUser(tU8& Pu8CurrentEndUser)
{
  tS32              Vs32Return=DP_S32_NO_ERR;
  dp_tclDpSem       VmyDpSem;
  
  if(_ptDPShmEndUser==NULL)
  {
    Vs32Return=DP_S32_ERR_UNKNOWN;
    NORMAL_M_ASSERT_ALWAYS();
  }
  else
  { /*lock semaphore*/
    VmyDpSem.s32LockSemShm(&_ptDPShmEndUser->semShmEndUser);
    if(_ptDPShmEndUser->bLoad==FALSE)
    { //load end user info to shared memory 
	    vLoadEndUserInfoToShm();
    }
    //get end user
    Pu8CurrentEndUser=_ptDPShmEndUser->u8EndUser;
    ETG_TRACE_USR4(("s32GetEndUser(): end user %d",Pu8CurrentEndUser));
    /*unlock semaphore*/
    VmyDpSem.s32UnLockSemShm(&_ptDPShmEndUser->semShmEndUser);
  }
  return(Vs32Return);
}

/******************************************************************************
* void dp_tclDpEndUser::s32SetEndUser()
*
* DESCRIPTION: set end user and check if end user in user list
*
* PARAMETERS:  u8EndUser:   end user to set
*              u16AccessId: access id
*
* RETURNS: error code or DP_S32_NO_ERR
*
* HISTORY: Created 2014 12 16
*****************************************************************************/
tS32     dp_tclDpEndUser::s32SetEndUser(tU8 u8EndUser,tU16 u16AccessId)
{
  tS32                                Vs32ReturnCode=DP_S32_NO_ERR;
  dp_tclDpSem                         VmyDpSem;
  tBool                               VbFoundEndUserInList=FALSE;
  tU8                                 Vu8Inc;
 
  if(_ptDPShmEndUser==NULL)
  {
    Vs32ReturnCode=DP_S32_ERR_UNKNOWN;
    NORMAL_M_ASSERT_ALWAYS();
  }
  else
  {
    if(u16AccessId==(tU16)(DP_ACCESS_ID_END_USER_APP))
    { /*lock semaphore*/
      VmyDpSem.s32LockSemShm(&_ptDPShmEndUser->semShmEndUser);
	    /*check if user information load from file*/
	    if(_ptDPShmEndUser->bLoad==FALSE)
      { //load end user info to shared memory 
	      vLoadEndUserInfoToShm();
      }
      /*-----------------  set end user to shared memory----------------------------*/
      _ptDPShmEndUser->u8EndUser=u8EndUser;
	    ETG_TRACE_USR4(("dp_tclDpEndUser(): s32SetEndUser(): end user %d",_ptDPShmEndUser->u8EndUser));	
      /*-----------------  check if new end user exist in list ----------------------------*/	
	    for (Vu8Inc=0; Vu8Inc<_ptDPShmEndUser->u8EndUserUsedMax && !VbFoundEndUserInList; Vu8Inc++)
      {
	      if(u8EndUser==_ptDPShmEndUser->u8EndUserUsed[Vu8Inc])
	      {
	        VbFoundEndUserInList=TRUE;
	      }
	    }
	    //if not found
	    if(VbFoundEndUserInList==FALSE)
	    { /*save to list*/
	      _ptDPShmEndUser->u8EndUserUsed[_ptDPShmEndUser->u8EndUserUsedMax]=u8EndUser;
	      _ptDPShmEndUser->u8EndUserUsedMax++;	
	    }
	    /*store pool information to file */
	    vStoreEndUserInfoFromShm();
      /*unlock semaphore*/
      VmyDpSem.s32UnLockSemShm(&_ptDPShmEndUser->semShmEndUser);
    }
    else
    {
      Vs32ReturnCode=DP_S32_ERR_ACCESS_RIGHT;
    }
  }
  return(Vs32ReturnCode);
}
/******************************************************************************
* void dp_tclDpEndUser::s32GetElementFromFile()
*
* DESCRIPTION: get element from file
*
* PARAMETERS: Pu8EndUser: end user
*             Pu32PoolId: pool od
*             PcPoolName: pool name
*             PeLocation: pool location
*             Pu32Version: pool version
*             PpDpElement: pointer for the element
*             Pu16AccessId: access id
*
* RETURNS: success or error code
*
* HISTORY: Created 2015 01 08
*****************************************************************************/
tS32     dp_tclDpEndUser::s32GetElementFromFile(tU8 Pu8EndUser,tU32 Pu32PoolId, const tChar* PcPoolName, eDpLocation PeLocation,tU32 Pu32Version,dp_tclBaseElement* PpDpElement, tU16 Pu16AccessId)
{
  tS32          Vs32ReturnCode=DP_S32_ERR_NO_ELEMENT;
  tBool         VbCodingElem = FALSE;
  tBool         VbOwner = ((tU32)(Pu16AccessId << (tU16)16) == (Pu32PoolId & (tU32)0xffff0000));
  tU16          Vu16SubpoolId;
  tChar         VstrFileName[DP_MAX_LEN_FILE];
  dp_tclDpSem   VmyDpSem;

  if((_ptDPShmEndUser==NULL)||(_poPersMemAccessHdl==NULL))
  {
    Vs32ReturnCode=DP_S32_ERR_UNKNOWN;
    NORMAL_M_ASSERT_ALWAYS();
  }
  else
  { /*lock semaphore*/
    VmyDpSem.s32LockSemShm(&_ptDPShmEndUser->semShmEndUser);
    /*check if user information load from file*/
    if(_ptDPShmEndUser->bLoad==FALSE)
    { //load end user info to shared memory 
      vLoadEndUserInfoToShm();
    }
    //check for undef element
    if((Pu16AccessId >= CCA_C_U16_APP_UNDEFINED_BEGIN)&&(Pu16AccessId<=CCA_C_U16_APP_UNDEFINED_END))
    { /*not allowed for undef element*/
      Vs32ReturnCode=DP_S32_ERR_NOT_SUPPORTED;
	    ETG_TRACE_ERR(("dp_tclDpEndUser(): s32GetElementFromFile(): undef element are not supported for multi end user (%s)",PpDpElement->pcGetElementName()));
    }
    else
    { //check for global/coding pool
      Vu16SubpoolId = (Pu32PoolId & 0x0000ffff);
      if (Vu16SubpoolId >= DP_U16_SUBPOOL_CODING_RANGE) 
      {
        VbCodingElem = TRUE;
      }  
      //check access
	    if (VbOwner || VbCodingElem) 
	    {/* read access available */
	      if (  ( VbOwner && (PpDpElement->u16GetAccessType() & DP_U16_ACCESS_RD_OWNER_0004)) 
	          ||(!VbOwner && (PpDpElement->u16GetAccessType() & DP_U16_ACCESS_RD_OTHER_0440)))
        {
	        tU8*    Vpu8Buffer = NULL;
          tS32    Vs32ReadBytes=0;			     
		      //get file name  		
		      _poPersMemAccessHdl->s32GetDataStreamName((tU16) (Pu32PoolId >> 16), PcPoolName,VstrFileName);
		      vAddExt(Pu8EndUser,VstrFileName);
          //import data
		      Vs32ReadBytes = _poPersMemAccessHdl->s32ImportPersData(VstrFileName,&Vpu8Buffer,PeLocation,Pu32Version);
          if (0 < Vs32ReadBytes) 
		      { //extract data 
            tU8* Vpu8CurPos = Vpu8Buffer;
            while ((Vpu8CurPos + sizeof(tU32)) < (Vpu8Buffer + Vs32ReadBytes)) //lint !e662 :Possible creation of out-of-bounds pointer (4 beyond end of data) by operator 'ptr+int' -> buffer ranges checked
		        { //create empty element
              dp_tclBaseElement VmyDataElem;		
              //stream data to context
              dp_tclInStreamCtxt VoInContext(Vpu8CurPos);
              VoInContext >> VmyDataElem;
			        //check if hash equal
			        if(VmyDataElem.s32GetHash()==PpDpElement->s32GetHash())
			        {//copy data
                PpDpElement->vCopyElement(VmyDataElem);
			          ETG_TRACE_USR4(("dp_tclDpEndUser(): s32GetElementFromFile(): element '%s' found",PpDpElement->pcGetElementName()));
			          Vs32ReturnCode=DP_S32_NO_ERR;
                break;
			        }
              Vpu8CurPos = VoInContext.pu8GetEndPosition();
            }/*end while*/
          }/*end if 0 < Vs32ReadBytes*/
	        delete[] Vpu8Buffer;
            Vpu8Buffer = NULL;/*Bug 123144: Safe handling*/
		      if(Vs32ReturnCode!=DP_S32_NO_ERR)
          {
		        ETG_TRACE_USR4(("dp_tclDpEndUser(): s32GetElementFromFile(): no element '%s' found",PpDpElement->pcGetElementName()));
		        //return default value; already saved in PpDpElement
            Vs32ReturnCode=DP_S32_NO_ERR;
          }		
        }/*access*/
	      else
	      {
		      ETG_TRACE_ERR(("dp_tclDpEndUser(): s32GetElementFromFile(): error -> no access right to read element '%s'.", PpDpElement->pcGetElementName()));
          Vs32ReturnCode = DP_S32_ERR_ACCESS_RIGHT;
        }
      }/*access*/
	    else
      {
	      ETG_TRACE_ERR(("dp_tclDpEndUser(): s32GetElementFromFile(): error -> no access right to read element '%s'.", PpDpElement->pcGetElementName()));
        Vs32ReturnCode = DP_S32_ERR_ACCESS_RIGHT;
      }
    }/*undef*/
    /*unlock semaphore*/
    VmyDpSem.s32UnLockSemShm(&_ptDPShmEndUser->semShmEndUser);
  }
  return(Vs32ReturnCode);
}
/******************************************************************************
* tS32     dp_tclDpEndUser::s32CopyEndUserDataStream(
*
* DESCRIPTION: copy datastream from Pu8EndUserFrom to Pu8EndUserTo
*
* PARAMETERS: Pu8EndUserFrom: from end user
*             PcPoolName: to end user
*             PeLocation: pool location
*             Pu32Version: pool version
*
* RETURNS: success or error code
*
* HISTORY: Created  2016 07 26
*****************************************************************************/
tS32     dp_tclDpEndUser::s32CopyEndUserDataStream(tU8 Pu8EndUserFrom,tU8 Pu8EndUserTo,const tChar* PcPoolName,eDpLocation PeLocation,tU32 Pu32Version)
{
  tS32          Vs32ReturnCode=DP_S32_NO_ERR;

  /*check parameter*/
  if (Pu8EndUserFrom==Pu8EndUserTo)
  {
    ETG_TRACE_USR4(("dp_tclDpEndUser(): s32CopyEndUserFile(): equal user '%d':'%d' ", Pu8EndUserFrom,Pu8EndUserTo));
  } 
  else if (PcPoolName==NULL)
  {
    ETG_TRACE_ERR(("dp_tclDpEndUser(): s32CopyEndUserFile(): error parameter PcPoolName "));
    Vs32ReturnCode=DP_S32_ERR_TYPE;
  }
  else if(_poPersMemAccessHdl==NULL)
  {
    ETG_TRACE_ERR(("dp_tclDpEndUser(): s32CopyEndUserFile(): error _poPersMemAccessHdl not exist "));
	  Vs32ReturnCode=DP_S32_ERR_UNKNOWN;
    NORMAL_M_ASSERT_ALWAYS();
  }
  else if(_ptDPShmEndUser==NULL)
  {
    ETG_TRACE_ERR(("dp_tclDpEndUser(): s32CopyEndUserFile(): error _ptDPShmEndUser not exist "));
	  Vs32ReturnCode=DP_S32_ERR_UNKNOWN;
    NORMAL_M_ASSERT_ALWAYS();
  }
  else if(_ptDPShmEndUser->u16EndUserConfigNotStore==Pu8EndUserTo)
  {
    ETG_TRACE_USR4(("dp_tclDpEndUser(): s32CopyEndUserFile(): 'Pu8EndUserTo:%d' should not be stored 'u16EndUserConfigNotStore:%d' ", Pu8EndUserFrom,_ptDPShmEndUser->u16EndUserConfigNotStore));
  }
  else
  {
    tS32          ViLen=((tS32)strlen(PcPoolName));
    tChar         VstrFileNameFrom[DP_MAX_LEN_FILE]={0};
    tChar         VstrFileNameTo[DP_MAX_LEN_FILE]={0};
    tU8*          Vpu8Buffer = NULL;

    //get both datastream name
    strncpy(VstrFileNameFrom,PcPoolName,ViLen);     
    strncpy(VstrFileNameTo,PcPoolName,ViLen); 
    vAddExt(Pu8EndUserFrom,VstrFileNameFrom);   
    vAddExt(Pu8EndUserTo,VstrFileNameTo);   
    tS32 Vs32ReadBytes = _poPersMemAccessHdl->s32ImportPersData(VstrFileNameFrom,&Vpu8Buffer,PeLocation,Pu32Version);
    if (0 < Vs32ReadBytes) 
    { //write data
      if(_poPersMemAccessHdl->bWriteFfsData(VstrFileNameTo,Vpu8Buffer,Vs32ReadBytes,PeLocation,Pu32Version, TRUE)==FALSE)
      {
         ETG_TRACE_ERR(("dp_tclDpEndUser(): s32CopyEndUserFile(): error -> not write datastream '%s' ", VstrFileNameTo));
         Vs32ReturnCode=DP_S32_ERR_UNKNOWN;
      }
    }
    else
    {
      ETG_TRACE_USR4(("dp_tclDpEndUser(): s32CopyEndUserFile(): not import datastream '%s' ",VstrFileNameFrom));
    }
    delete[] Vpu8Buffer;
    Vpu8Buffer=NULL;/*Bug 123144: Safe handling*/
  }
  return(Vs32ReturnCode);
}

/******************************************************************************
* tBool     dp_tclDpEndUser::bCheckEndUserShouldStore(tChar* PcPoolName,tU8 Pu8Bank)
*
* DESCRIPTION: check if the pool should be store
*
* PARAMETERS: PcPoolName: name of the pool 
*             Pu8Bank: if Pu8Bank == DP_U8_NO_BANK_ACTION (poolname with bank)
*               
*
* RETURNS: return TRUE if the pool should be stored
*          return FALSE if the pool should not be stored
*
* HISTORY: Created 2016 07 26
*****************************************************************************/
tBool     dp_tclDpEndUser::bCheckEndUserShouldStore(tChar* PcPoolName,tU8 Pu8Bank)
{
  tBool             VbReturnStore=TRUE;
  dp_tclDpSem       VmyDpSem;

  if(_ptDPShmEndUser!=NULL)  
  { /*lock semaphore*/
    VmyDpSem.s32LockSemShm(&_ptDPShmEndUser->semShmEndUser);
    if(_ptDPShmEndUser->bLoad==FALSE)
    { //load end user info to shared memory 
	    vLoadEndUserInfoToShm();
    }
    /* valid range for end user 0..0xff*/
    if(_ptDPShmEndUser->u16EndUserConfigNotStore<=0xff)
    { //get end user from pool name
      tS32    ViLenExt;
      tS32    ViLen=((tS32)strlen(PcPoolName));
      char*   VstrPos = NULL;
      if(Pu8Bank!=DP_U8_NO_BANK_ACTION)   
        ViLenExt=4;
      else
        ViLenExt=2;
      if(ViLen>=ViLenExt)
      {
        tChar   VstrEndUserPoolName[ViLenExt+1];
        tChar   VstrEndUserNotStored[ViLenExt+1];

        memset(VstrEndUserPoolName,0,ViLenExt+1);
        memset(VstrEndUserNotStored,0,ViLenExt+1);
        VstrPos=PcPoolName+ViLen-ViLenExt;       
        strncpy(VstrEndUserPoolName,VstrPos,ViLenExt);     
        //get end user string not stored
        vAddExt((tU8)_ptDPShmEndUser->u16EndUserConfigNotStore,VstrEndUserNotStored);
        /*compare both string*/
        if(strncmp(VstrEndUserNotStored,VstrEndUserPoolName,2)==0)
        {
          VbReturnStore=FALSE;
        }
      }
    }
    /*unlock semaphore*/
    VmyDpSem.s32UnLockSemShm(&_ptDPShmEndUser->semShmEndUser);
  }
  return(VbReturnStore);
}

/******************************************************************************
* void dp_tclDpEndUser::s32GetBank()
*
* DESCRIPTION: get bank for the current user
*
* PARAMETERS: Pu8Bank: buffer for the bank of the current user
*
* RETURNS: success or error code
*
* HISTORY: Created 2015 05 06
*****************************************************************************/
tS32     dp_tclDpEndUser::s32GetBank(tU8& Pu8Bank)
{
  tS32              Vs32Return=DP_S32_NO_ERR;
  dp_tclDpSem       VmyDpSem;
  
  if(_ptDPShmEndUser==NULL)
  {
    Vs32Return=DP_S32_ERR_UNKNOWN;
    NORMAL_M_ASSERT_ALWAYS();
  }
  else
  {  /*lock semaphore*/
    VmyDpSem.s32LockSemShm(&_ptDPShmEndUser->semShmEndUser);
    if(_ptDPShmEndUser->bLoad==FALSE)
    { //load end user info to shared memory 
	    vLoadEndUserInfoToShm();
    }
    //get bank for the current user
    Pu8Bank=_ptDPShmEndUser->u8EndUserBank[_ptDPShmEndUser->u8EndUser];
    ETG_TRACE_USR4(("s32GetBank(): current bank %d for end user %d",Pu8Bank,_ptDPShmEndUser->u8EndUser));    
    /*unlock semaphore*/
    VmyDpSem.s32UnLockSemShm(&_ptDPShmEndUser->semShmEndUser);
  }
  return(Vs32Return);
}

/******************************************************************************
* void dp_tclDpEndUser::s32SetBank()
*
* DESCRIPTION: set bank for the current user
*
* PARAMETERS: Pu8Bank: bank of the current user
*
* RETURNS: success or error code
*
* HISTORY: Created 2015 05 06
*****************************************************************************/
tS32     dp_tclDpEndUser::s32SetBank(tU8 Pu8Bank)
{
  tS32              Vs32Return=DP_S32_NO_ERR;
  dp_tclDpSem       VmyDpSem;
  
  if(_ptDPShmEndUser==NULL)
  {
	Vs32Return=DP_S32_ERR_UNKNOWN;
    NORMAL_M_ASSERT_ALWAYS();
  }
  else
  {  /*lock semaphore*/
    VmyDpSem.s32LockSemShm(&_ptDPShmEndUser->semShmEndUser);
    if(_ptDPShmEndUser->bLoad==FALSE)
    { //load end user info to shared memory 
	    vLoadEndUserInfoToShm();
    }
    //set bank for the current user
    _ptDPShmEndUser->u8EndUserBank[_ptDPShmEndUser->u8EndUser]=Pu8Bank;
    ETG_TRACE_USR4(("s32SetBank(): current bank %d for end user %d",Pu8Bank,_ptDPShmEndUser->u8EndUser));   
    /*store pool information to file */
    vStoreEndUserInfoFromShm();
    /*unlock semaphore*/
    VmyDpSem.s32UnLockSemShm(&_ptDPShmEndUser->semShmEndUser);
  }
  return(Vs32Return);
}
/******************************************************************************
* void dp_tclDpEndUser::s32SetBankUsed()
*
* DESCRIPTION: check if bank in used list; if not saved new bank to used list
*
* PARAMETERS: Pu8Bank: bank of the current user
*
* RETURNS: success or error code
*
* HISTORY: Created 2015 05 06
*****************************************************************************/
tS32     dp_tclDpEndUser::s32SetBankUsed(tU8 Pu8Bank)
{
  tS32              Vs32Return=DP_S32_NO_ERR;
  dp_tclDpSem       VmyDpSem;
  tBool             VbFoundEndUserBankInList=FALSE;
  tU8               Vu8Inc;
  
  if(_ptDPShmEndUser==NULL)
  {
    Vs32Return=DP_S32_ERR_UNKNOWN;
    NORMAL_M_ASSERT_ALWAYS();
  }
  else
  { /*lock semaphore*/
    VmyDpSem.s32LockSemShm(&_ptDPShmEndUser->semShmEndUser);
    if(_ptDPShmEndUser->bLoad==FALSE)
    { //load end user info to shared memory 
	    vLoadEndUserInfoToShm();
    }
    /*-----------------  check if new end user bank exist in list ----------------------------*/	
    for (Vu8Inc=0; Vu8Inc < _ptDPShmEndUser->u8EndUserBankUsedMax && !VbFoundEndUserBankInList; Vu8Inc++)
    {
      if(Pu8Bank==_ptDPShmEndUser->u8EndUserBankUsed[Vu8Inc])
	    {
	      VbFoundEndUserBankInList=TRUE;
	    }
    }
    //if not found
    if(VbFoundEndUserBankInList==FALSE)
    { /*save to list*/
      _ptDPShmEndUser->u8EndUserBankUsed[_ptDPShmEndUser->u8EndUserBankUsedMax]=Pu8Bank;
	    _ptDPShmEndUser->u8EndUserBankUsedMax++;	
	    /*store pool information to file */
      vStoreEndUserInfoFromShm();
    }  
    /*unlock semaphore*/
    VmyDpSem.s32UnLockSemShm(&_ptDPShmEndUser->semShmEndUser);
  }
  return(Vs32Return);
}
/******************************************************************************
* void dp_tclDpEndUser::vSetEndUserModeToDefault()
*
* DESCRIPTION: set end user mode to default (in shared memory and in file)
*
* PARAMETERS:  void
*
* RETURNS: void
*
* HISTORY: Created 2014 12 16
*****************************************************************************/
void     dp_tclDpEndUser::vSetEndUserModeToDefault(void)
{
  dp_tclDpSem                         VmyDpSem;
 
  if(_ptDPShmEndUser==NULL)
  {
    NORMAL_M_ASSERT_ALWAYS();
  }
  else
  { /*lock semaphore*/
    VmyDpSem.s32LockSemShm(&_ptDPShmEndUser->semShmEndUser);
    /*check if user information load from file*/
    if(_ptDPShmEndUser->bLoad==FALSE)
    { //load end user info to shared memory 
      vLoadEndUserInfoToShm();
    }
    /* set to default*/
    vSetEndUserInfoInSharedMemToDefault(); 
    /*store pool information to file */
    vStoreEndUserInfoFromShm();
    /*unlock semaphore*/
    VmyDpSem.s32UnLockSemShm(&_ptDPShmEndUser->semShmEndUser);
  }
}
/******************************************************************************
* void dp_tclDpEndUser::vSetBankForEndUserToDefault(tU8 u8EndUser)
*
* DESCRIPTION: set end user mode to default (in shared memory and in file)
*
* PARAMETERS:  u8EndUser end user
*
* RETURNS: void
*
* HISTORY: Created  2014 12 16
*****************************************************************************/
void     dp_tclDpEndUser::vSetBankForEndUserToDefault(tU8 u8EndUser)
{
  dp_tclDpSem                         VmyDpSem;
 
  if(_ptDPShmEndUser==NULL)
  {
    NORMAL_M_ASSERT_ALWAYS();
  }
  else
  { /*lock semaphore*/
    VmyDpSem.s32LockSemShm(&_ptDPShmEndUser->semShmEndUser);
    /*check if user information load from file*/
    if(_ptDPShmEndUser->bLoad==FALSE)
    { //load end user info to shared memory 
      vLoadEndUserInfoToShm();
    }
    /* set bank for user to default*/
    _ptDPShmEndUser->u8EndUserBank[u8EndUser]=0;
    /*store pool information to file */
    vStoreEndUserInfoFromShm();
    /*unlock semaphore*/
    VmyDpSem.s32UnLockSemShm(&_ptDPShmEndUser->semShmEndUser);
  }
}
/******************************************************************************
* void dp_tclDpEndUser::bPerformDefSetEndUserPool()
*
* DESCRIPTION: set end user pool to default => delete the pool
*
* PARAMETERS:  PptDpElementList: list of elements in pool
*              u8EndUser:        end user
*              u32DefSetType:    type of defset
*              cPoolName:        pool name              
*              PeLocation:       location
*              Pu32Version:      version of the pool
*
* RETURNS: TRUE: ==DP_DEFSET_END_USER=> only current pool set to default/delete; 
*                !=DP_DEFSET_END_USER: all pools set to default/delete
*          FALSE:  DP_DEFSET_END_USER=> not the current pool  
*
* HISTORY: Created  2015 01 26
*****************************************************************************/
tBool     dp_tclDpEndUser::bPerformDefSetEndUserPool(TListDatapoolElements* PptDpElementList,tU8 u8EndUser,tU32 u32DefSetType, tChar* cPoolName, eDpLocation PeLocation,tU32 Pu32Version)
{
  tBool         VbCurrentPool=TRUE;
  dp_tclDpSem   VmyDpSem; 
  if(_ptDPShmEndUser==NULL)
  {
    FATAL_M_ASSERT_ALWAYS();
  }
  else
  { /*lock semaphore*/
    VmyDpSem.s32LockSemShm(&_ptDPShmEndUser->semShmEndUser);
    /*check if user information load from file*/
    if(_ptDPShmEndUser->bLoad==FALSE)
    { //load end user info to shared memory 
      vLoadEndUserInfoToShm();
    }
    if(u32DefSetType==DP_DEFSET_END_USER)
    {/* set one user to default*/
      if(u8EndUser!=_ptDPShmEndUser->u8EndUser)
	    {/*delete not the current pool*/
	      VbCurrentPool=FALSE;	   
        s32SetEndUserPoolToDefault(PptDpElementList,u8EndUser,cPoolName,PeLocation,Pu32Version);
	    }
    }
    else if (u32DefSetType!=DP_DEFSET_END_USER_BANK)
    {/*delete all pools*/
      tU8  Vu8Inc;
	    for (Vu8Inc=0; Vu8Inc < _ptDPShmEndUser->u8EndUserUsedMax; Vu8Inc++)
      {
	      s32SetEndUserPoolToDefault(PptDpElementList,_ptDPShmEndUser->u8EndUserUsed[Vu8Inc],cPoolName,PeLocation,Pu32Version);
	    }
    }
    /*unlock semaphore*/
    VmyDpSem.s32UnLockSemShm(&_ptDPShmEndUser->semShmEndUser);
  }
  return(VbCurrentPool);
}

/******************************************************************************
* void dp_tclDpEndUser::bPerformDefSetEndUserBank()
*
* DESCRIPTION: set bank to default
*
* PARAMETERS:  PptDpElementList: list of elements in pool
*              u8EndUserBank:    end user bank
*              u32DefSetType:    type of defset
*              cPoolName:        pool name              
*              PeLocation:       location
*              Pu32Version:      version of the pool
*
* RETURNS: return TRUE
*
* HISTORY: Created  2015 01 26
*****************************************************************************/
tBool     dp_tclDpEndUser::bPerformDefSetEndUserBank(TListDatapoolElements* PptDpElementList,tU8 u8EndUserBank, tU32 u32DefSetType,tChar* cPoolName, eDpLocation PeLocation,tU32 Pu32Version)
{
  dp_tclDpSem   VmyDpSem;
  if(_ptDPShmEndUser==NULL)
  {
    NORMAL_M_ASSERT_ALWAYS();
	return(FALSE);
  }
  else
  {  /*lock semaphore*/
    VmyDpSem.s32LockSemShm(&_ptDPShmEndUser->semShmEndUser);
    /*check if user information load from file*/
    if(_ptDPShmEndUser->bLoad==FALSE)
    { //load end user info to shared memory 
      vLoadEndUserInfoToShm();
    }   
    if(u32DefSetType==DP_DEFSET_END_USER_BANK)
    {
      s32SetEndUserBankToDefault(PptDpElementList,_ptDPShmEndUser->u8EndUser,u8EndUserBank,cPoolName,PeLocation,Pu32Version);
    }
    else if(u32DefSetType==DP_DEFSET_END_USER)
    {/*set all banks for the current end user to default*/
      tU8  Vu8IncBank;
	    //set all banks to default
	    for (Vu8IncBank=0; Vu8IncBank < _ptDPShmEndUser->u8EndUserBankUsedMax; Vu8IncBank++)
      {
	      s32SetEndUserBankToDefault(PptDpElementList,u8EndUserBank,_ptDPShmEndUser->u8EndUserBankUsed[Vu8IncBank],cPoolName,PeLocation,Pu32Version);
	    }
	    //set end user pool to default
	    s32SetEndUserPoolToDefault(PptDpElementList,u8EndUserBank,cPoolName,PeLocation,Pu32Version);
    }
    else
    {/*set all banks for all end user to default*/
      tU8  Vu8IncUser;
	    tU8  Vu8IncBank;
	    for (Vu8IncUser=0; Vu8IncUser < _ptDPShmEndUser->u8EndUserUsedMax; Vu8IncUser++)
      {
	      for (Vu8IncBank=0; Vu8IncBank < _ptDPShmEndUser->u8EndUserBankUsedMax; Vu8IncBank++)
        {
	        s32SetEndUserBankToDefault(PptDpElementList,_ptDPShmEndUser->u8EndUserUsed[Vu8IncUser],_ptDPShmEndUser->u8EndUserBankUsed[Vu8IncBank],cPoolName,PeLocation,Pu32Version);
	      }
	      s32SetEndUserPoolToDefault(PptDpElementList,_ptDPShmEndUser->u8EndUserUsed[Vu8IncUser],cPoolName,PeLocation,Pu32Version);
	    }
    }
    /*unlock semaphore*/
    VmyDpSem.s32UnLockSemShm(&_ptDPShmEndUser->semShmEndUser);
  }
  return(TRUE);
}
/******************************************************************************
* void dp_tclDpEndUser::vAddCurrentEndUserExt()
*
* DESCRIPTION: add current end user extension 
*
* PARAMETERS:  VstrWithUserExt: String to add extansion
*
* RETURNS: void  
*
* HISTORY: Created  2014 12 16
*****************************************************************************/
void     dp_tclDpEndUser::vAddCurrentEndUserExt(tChar* VstrWithUserExt)
{
  dp_tclDpSem   VmyDpSem;
  if(_ptDPShmEndUser==NULL)
  {
    NORMAL_M_ASSERT_ALWAYS();
  }
  else
  {  /*lock semaphore*/
    VmyDpSem.s32LockSemShm(&_ptDPShmEndUser->semShmEndUser);
    /*check if user information load from file*/
    if(_ptDPShmEndUser->bLoad==FALSE)
    { //load end user info to shared memory 
      vLoadEndUserInfoToShm();
    }
    // get extansion for the current end user
    vAddExt(_ptDPShmEndUser->u8EndUser,VstrWithUserExt);   
    /*unlock semaphore*/
    VmyDpSem.s32UnLockSemShm(&_ptDPShmEndUser->semShmEndUser);
  }
}

/******************************************************************************
* void dp_tclDpEndUser::vAddEndUserBankExt()
*
* DESCRIPTION: add bank extansion
*
* PARAMETERS:  VstrWithUserExt: String to add extansion
*
* RETURNS: void  
*
* HISTORY: Created  2014 12 16
*****************************************************************************/
void     dp_tclDpEndUser::vAddEndUserBankExt(tU8 Pu8Bank,tChar* VstrFileWithBankExt)
{
  dp_tclDpSem   VmyDpSem; 
  if(_ptDPShmEndUser==NULL)
  {
    NORMAL_M_ASSERT_ALWAYS();
  }
  else
  {  /*lock semaphore*/
    VmyDpSem.s32LockSemShm(&_ptDPShmEndUser->semShmEndUser);
    /*check if user information load from file*/
    if(_ptDPShmEndUser->bLoad==FALSE)
    { //load end user info to shared memory 
      vLoadEndUserInfoToShm();
    }
    // get extansion 
    vAddExt(Pu8Bank,VstrFileWithBankExt);   
    /*unlock semaphore*/
    VmyDpSem.s32UnLockSemShm(&_ptDPShmEndUser->semShmEndUser);
  }
}
/******************************************************************************
* void dp_tclDpEndUser::vAddExt()
*
* DESCRIPTION: get end user extensiion for the current end user
*
* PARAMETERS:  PeModeEndUser: end user mode
*              Pu8EndUser:    end user 
*              strEndUserExtension:   string for the extension
*
* RETURNS: void
*
* HISTORY: Created  2015 01 26
*****************************************************************************/
void dp_tclDpEndUser::vAddExt(tU8 Pu8Value,tChar* VstrStringWithExt)
{
  tChar  VstrEndUserExt[DP_END_USER_FILE_LEN_EXT]={0};
  int    ViLen;
  OSAL_s32PrintFormat(VstrEndUserExt, "%02x", Pu8Value);
  ViLen=((tS32)strlen(VstrStringWithExt));
  OSAL_szStringNConcat((tString)VstrStringWithExt,VstrEndUserExt, DP_MAX_LEN_FILE-ViLen); 
}

/******************************************************************************
* void dp_tclDpEndUser::vGetFileNameWithoutExt()
*
* DESCRIPTION: get the file name without end user extension
*
* PARAMETERS:  VstrStringWithExt: string with extension
*              VstrStringWithoutExt: string without extension
*
* RETURNS: void
*
* HISTORY: Created  2015 01 26
*****************************************************************************/
void dp_tclDpEndUser::vGetFileNameWithoutExt(tChar* VstrStringWithExt,tChar* VstrStringWithoutExt)
{
    if((VstrStringWithExt!=NULL)&&(VstrStringWithoutExt!=NULL))
    {
      tS32  ViLen=((tS32)strlen(VstrStringWithExt));
      strncpy(VstrStringWithoutExt,VstrStringWithExt,ViLen);
      if(ViLen>2)
        VstrStringWithoutExt[ViLen-2]=0x00;
    }
}

/******************************************************************************
* void dp_tclDpEndUser::s32SetEndUserPoolToDefault()
*
* DESCRIPTION: set end user pool to default. 
*              First check if pool has secure elements. 
*              If no secure elements => delete file
*
* PARAMETERS:  PptDpElementList: list of elements in pool
*              Pu8EndUser: end user
*              Pu32PoolId: pool id
*              PcPoolName: pool name
*              PeLocation: pool location
*              Pu32Version: version of the pool
*
* RETURNS: success or error code  
*
* HISTORY: Created  2015 01 26
*****************************************************************************/
tS32 dp_tclDpEndUser::s32SetEndUserPoolToDefault(TListDatapoolElements* PptDpElementList,tU8 Pu8EndUser,tChar* PcPoolName, eDpLocation PeLocation,tU32 Pu32Version)
{
  tS32   Vs32ReturnCode=DP_S32_NO_ERR;
  tChar  VstrFileName[DP_MAX_LEN_FILE]={0};

  /*get the file name without extension */            
  vGetFileNameWithoutExt(PcPoolName,VstrFileName);
  /*get new name of the datastream*/
  vAddExt(Pu8EndUser,VstrFileName); 
  //check for elements with secure
  if(bCheckElementsInFileForDefSetModeSecure(PptDpElementList,VstrFileName,PeLocation,Pu32Version)==FALSE)
  {/*delete persistent data*/
    if(_poPersMemAccessHdl==NULL)
    {
	    Vs32ReturnCode=DP_S32_ERR_UNKNOWN;
      NORMAL_M_ASSERT_ALWAYS();
    }
	  else
	  {
      Vs32ReturnCode=_poPersMemAccessHdl->s32DeletePersData(VstrFileName,PeLocation);   
	  }
  }
  return(Vs32ReturnCode);
}
/******************************************************************************
* void dp_tclDpEndUser::s32SetEndUserBankToDefault()
*
* DESCRIPTION: set bank pool to default. 
*              First check if bank pool has secure elements. 
*              If no secure elements => delete file
*
* PARAMETERS:  PptDpElementList: list of elements in pool
*              Pu8EndUser: end user
*              Pu8Bank:    bank
*              Pu32PoolId: pool id
*              PcPoolName: pool name
*              PeLocation: pool location
*              Pu32Version: version of the pool
*
* RETURNS: success or error code  
*
* HISTORY: Created  2015 01 26
*****************************************************************************/
tS32 dp_tclDpEndUser::s32SetEndUserBankToDefault(TListDatapoolElements* PptDpElementList,tU8 Pu8EndUser,tU8 Pu8Bank,tChar* PcPoolName, eDpLocation PeLocation,tU32 Pu32Version)
{
  tS32   Vs32ReturnCode=DP_S32_NO_ERR;
  tChar  VstrFileName[DP_MAX_LEN_FILE]={0};  

  /*get the file name without extension */            
  vGetFileNameWithoutExt(PcPoolName,VstrFileName);
  /*add ext user and bank ext*/
  vAddExt(Pu8EndUser,VstrFileName);  
  vAddExt(Pu8Bank,VstrFileName); 
  //check for elements with secure
  if(bCheckElementsInFileForDefSetModeSecure(PptDpElementList,VstrFileName,PeLocation,Pu32Version)==FALSE)
  { /*delete persistent data*/
    if(_poPersMemAccessHdl==NULL)
    {
	    Vs32ReturnCode=DP_S32_ERR_UNKNOWN;
      NORMAL_M_ASSERT_ALWAYS();
    }
	  else	
	  {
      Vs32ReturnCode=_poPersMemAccessHdl->s32DeletePersData(VstrFileName,PeLocation);  
	  }
  }
  return(Vs32ReturnCode);
}
/******************************************************************************
* void dp_tclDpEndUser::bCheckElementsInFileForDefSetModeSecure()
*
* DESCRIPTION: check if in file elements exists, which are secure and should not set to default.
*              This elements are copied to a filebuffer and save to file.
*
* PARAMETERS:  PptDpElementList: list of elements in pool
*              VpSourceFilename: filename
*              PeLocation: pool location
*              Pu32Version: version of the pool
*
* RETURNS: TRUE:   set file to default
*          FALSE:  no secure elements exist; set file not to default
*
* HISTORY: Created  2015 01 26
*****************************************************************************/
tBool dp_tclDpEndUser::bCheckElementsInFileForDefSetModeSecure(TListDatapoolElements* PptDpElementList,tCString VpSourceFilename, eDpLocation PeLocation,tU32 Pu32Version)
{
  tBool   VbReturnDone=FALSE;
  tU8*    Vpu8Buffer = NULL;
  tS32    Vs32ReadBytes=0;
  
  if(_poPersMemAccessHdl==NULL)
  {
    NORMAL_M_ASSERT_ALWAYS();
  }
  else
  { //import file
    Vs32ReadBytes = _poPersMemAccessHdl->s32ImportPersData(VpSourceFilename,&Vpu8Buffer,PeLocation,Pu32Version);
    if (0 < Vs32ReadBytes) 
    { //extract data 
        tU8*                            VpBufferFileSecure = NULL;     
        tU32                            Vu32FileLenSecure=0;
        tU8*                            Vpu8CurPosFileSecure;
        tU8*                            Vpu8CurPos = Vpu8Buffer;
        TListDatapoolElements::iterator VposElement;
        try
        {
            VpBufferFileSecure = new tU8[Vs32ReadBytes];
        }
         catch(const std::bad_alloc&)
        {
            VpBufferFileSecure = NULL;
            ETG_TRACE_ERR(("bCheckElementsInFileForDefSetModeSecure(): unable to allocate %d Bytes",Vs32ReadBytes));
        }
        Vpu8CurPosFileSecure = VpBufferFileSecure;

        if ((VpBufferFileSecure)&&(Vpu8Buffer!= NULL))
        { //for each element in file
            while ((Vpu8CurPos + sizeof(tU32)) < (Vpu8Buffer + Vs32ReadBytes)) //lint !e662 :Possible creation of out-of-bounds pointer (4 beyond end of data) by operator 'ptr+int' -> buffer ranges checked
            { //create empty element
                dp_tclBaseElement VmyDataElem;        
                //stream data to context
                dp_tclInStreamCtxt VoInContext(Vpu8CurPos);
                try
                {
                    VoInContext >> VmyDataElem;
                }
                catch(const std::bad_alloc&)
                {
                    ETG_TRACE_ERR(("bCheckElementsInFileForDefSetModeSecure(): unable to read the inStream to VmyDataElem"));
                    break;
                }
          
                //check element in list
                VposElement = PptDpElementList->find(VmyDataElem.s32GetHash());
                if (VposElement != PptDpElementList->end()) 
                { //check element defset mode secure
                    if(VposElement->second.u32GetDefSetMode()==DP_DEFSET_NONE)
                    { //secure element don't delete
                        //copy element data to VpBufferFileSecure
                        tU32 Vu32Len = VposElement->second.u32GetStreamLength();
                        if((Vpu8CurPosFileSecure)&&((Vpu8CurPosFileSecure + Vu32Len) <= (VpBufferFileSecure + Vs32ReadBytes))) //lint !e662 :Possible creation of out-of-bounds pointer (4 beyond end of data) by operator 'ptr+int' -> buffer ranges checked
                        {
                            dp_tclOutStreamCtxt VoContextFile(Vpu8CurPosFileSecure, Vu32Len);
                            VoContextFile << VposElement->second;
                            Vpu8CurPosFileSecure = VoContextFile.pu8GetEndPosition();
                            Vu32FileLenSecure+=Vu32Len;
                        }
                        else
                        {
                            ETG_TRACE_ERR(("bCheckElementsInFileForDefSetModeSecure(): Overflow Detected allocated %d needed %d",Vs32ReadBytes,Vu32FileLenSecure+Vu32Len));
                        }
                    }
                }
                Vpu8CurPos = VoInContext.pu8GetEndPosition();
            }/*end while*/
            if (Vu32FileLenSecure!= 0)
            { //save file
                if(_poPersMemAccessHdl->bWriteFfsData(VpSourceFilename,VpBufferFileSecure,Vu32FileLenSecure,PeLocation,Pu32Version, TRUE)==TRUE)
                {
                    VbReturnDone=TRUE;
                }
            }
            delete[] VpBufferFileSecure;
            VpBufferFileSecure = NULL;/*Bug 123144: Safe handling*/
        }/*end if buffer*/
    }/*end if 0 < Vs32ReadBytes*/
    delete[] Vpu8Buffer;
    Vpu8Buffer=NULL;/*Bug 123144: Safe handling*/
  }
  return(VbReturnDone);
}
/******************************************************************************
* void dp_tclDpEndUser::vSetEndUserInfoInSharedMemToDefault()
*
* DESCRIPTION: set end user information in shared memory to default
*
* PARAMETERS:  void
*
* RETURNS: void
*
* HISTORY: Created  2015 01 26
*****************************************************************************/
void dp_tclDpEndUser::vSetEndUserInfoInSharedMemToDefault(void)
{
  if(_ptDPShmEndUser==NULL)
  {
    NORMAL_M_ASSERT_ALWAYS();
  }
  else
  {
    /*As per the Non-store requirement. If the Valet Mode is set to 0
    then on Default Settings the Enduser Used should be 1*/
    if(_ptDPShmEndUser->u16EndUserConfigNotStore != 0)
        _ptDPShmEndUser->u8EndUser=0; 
    else
        _ptDPShmEndUser->u8EndUser=1;
    
    _ptDPShmEndUser->u8EndUserUsedMax=1;                                      //the default u8EndUserUsedMax = '1' (current = end user 0)
    _ptDPShmEndUser->u8EndUserBankUsedMax=0;
    memset(_ptDPShmEndUser->u8EndUserUsed,0xff,DP_END_USER_USED_MAX);
    _ptDPShmEndUser->u8EndUserUsed[0]= _ptDPShmEndUser->u8EndUser;            //the default u8EndUserUsedMax = '1' (current = end user 0)
    memset(_ptDPShmEndUser->u8EndUserBank,0x00,DP_END_USER_USED_MAX);         //default 0
    memset(_ptDPShmEndUser->u8EndUserBankUsed,0xff,DP_END_USER_USED_MAX); 
  }
}
/******************************************************************************
* void dp_tclDpEndUser::s32LoadEndUserInformationToSharedMemory()
*
* DESCRIPTION: load end user information to shared memory
*
* PARAMETERS: void
*
* RETURNS: void
*
* HISTORY: Created  2015 01 26
*****************************************************************************/
void dp_tclDpEndUser::vLoadEndUserInfoToShm(void)
{
  tU8*                    Vpu8Buffer = NULL;
  tS32                    Vs32ReadBytes=0;
  char*                   VstrLine=NULL;

  if((_ptDPShmEndUser==NULL)||(_poPersMemAccessHdl==NULL))
  {
    NORMAL_M_ASSERT_ALWAYS();
  }
  else
  {            
    Vs32ReadBytes = _poPersMemAccessHdl->s32ImportPersData(DP_END_USER_STR_POOL_FILE_NAME,&Vpu8Buffer,DP_END_USER_POOL_LOCATION,DP_END_USER_POOL_VERSION); 
    if (0 < Vs32ReadBytes)
    {     		
	    tU8* Vpu8CurPos = Vpu8Buffer;
	    // end user current
	    dp_tclBaseElement VmyDpElemEndUserCurrent;
      dp_tclInStreamCtxt oInContext1(Vpu8CurPos);
      oInContext1 >> VmyDpElemEndUserCurrent;
      Vpu8CurPos = oInContext1.pu8GetEndPosition();
	    memcpy(&_ptDPShmEndUser->u8EndUser,VmyDpElemEndUserCurrent.pvGetData(),sizeof(_ptDPShmEndUser->u8EndUser));
      // end user used
	    dp_tclBaseElement VmyDpElemEndUserUsed;
      dp_tclInStreamCtxt oInContext2(Vpu8CurPos);
      oInContext2 >> VmyDpElemEndUserUsed;
      Vpu8CurPos = oInContext2.pu8GetEndPosition();
	    _ptDPShmEndUser->u8EndUserUsedMax=(tU8) VmyDpElemEndUserUsed.u32GetDataLength();
	    if(_ptDPShmEndUser->u8EndUserUsedMax!=0)
	    {
        memcpy(&_ptDPShmEndUser->u8EndUserUsed[0],VmyDpElemEndUserUsed.pvGetData(),_ptDPShmEndUser->u8EndUserUsedMax);
	    }	  
	    // end user bank
	    dp_tclBaseElement VmyDpElemEndUserBank;
      dp_tclInStreamCtxt oInContext3(Vpu8CurPos);
      oInContext3 >> VmyDpElemEndUserBank;
      Vpu8CurPos = oInContext3.pu8GetEndPosition();
      if (VmyDpElemEndUserBank.u32GetDataLength()!=0)
      {
	      memcpy(&_ptDPShmEndUser->u8EndUserBank[0],VmyDpElemEndUserBank.pvGetData(),VmyDpElemEndUserBank.u32GetDataLength());
      }
	    // end user bank used
	    dp_tclBaseElement VmyDpElemEndUserBankUsed;
      dp_tclInStreamCtxt oInContext4(Vpu8CurPos);
      oInContext4 >> VmyDpElemEndUserBankUsed;
      Vpu8CurPos = oInContext4.pu8GetEndPosition();
	    _ptDPShmEndUser->u8EndUserBankUsedMax=(tU8) VmyDpElemEndUserBankUsed.u32GetDataLength();
	    if(_ptDPShmEndUser->u8EndUserBankUsedMax!=0)
	    {
        memcpy(&_ptDPShmEndUser->u8EndUserBankUsed[0],VmyDpElemEndUserBankUsed.pvGetData(),_ptDPShmEndUser->u8EndUserBankUsedMax);
	    }	  
    }
    else
    { 
      ETG_TRACE_USR4(("vLoadEndUserInfoToShm(): no file %s read => set default",DP_END_USER_STR_POOL_FILE_NAME));
	    vSetEndUserInfoInSharedMemToDefault();
    }
    delete[] Vpu8Buffer;
    Vpu8Buffer=NULL;/*Bug 123144: Safe handling*/
    /*read information from configuration file*/  
    VstrLine=strFileReadConfigLine("DP_END_USER_NOT_STORED");
    if(VstrLine!=NULL)
    {
      ETG_TRACE_USR4(("vLoadEndUserInfoToShm():strFileReadConfigLine '%s'",VstrLine));
      char*  Vstr=strrchr(VstrLine,'='); //search last "="
      if(Vstr!=NULL)
      {
         tS32 Vs32Read=-1;
         if(strlen(Vstr)>2)
         {
           Vs32Read=atoi(Vstr+1);
           /*check value Vs32Read = 0; this is only valid, if string '0' found */
           Vstr=strrchr(VstrLine,'0');
           if((Vs32Read==0)&&(Vstr==NULL))
           { //set to unvalid
             Vs32Read=-1;   
           }
         }
         if(Vs32Read > DP_END_USER_USED_MAX)
         {
           _ptDPShmEndUser->u16EndUserConfigNotStore=DP_END_USER_NOT_STORED_FEATURE_DISABLED;
         }
         else
         {
           _ptDPShmEndUser->u16EndUserConfigNotStore=(tU16) Vs32Read;
           ETG_TRACE_USR4(("vLoadEndUserInfoToShm(): _ptDPShmEndUser->u16EndUserConfigNotStore:%d",_ptDPShmEndUser->u16EndUserConfigNotStore));
            if (0 >= Vs32ReadBytes)
            {
                vSetEndUserInfoInSharedMemToDefault();
                ETG_TRACE_USR4(("vLoadEndUserInfoToShm(): not store = %d => %s set default",Vs32Read, DP_END_USER_STR_POOL_FILE_NAME));
            } 
         }
      }
      free(VstrLine);
    }
    _ptDPShmEndUser->bLoad=TRUE;    
    /*trace out end user info */
    vTraceOutEndUserInfo();
  }
}

/******************************************************************************
* void dp_tclDpEndUser::vStoreEndUserInfoFromShm()
*
* DESCRIPTION: store  end user information from shared memory 
*
* PARAMETERS: void
*
* RETURNS: void
*
* HISTORY: Created  2015 02 18
*****************************************************************************/
void     dp_tclDpEndUser::vStoreEndUserInfoFromShm(void)
{
  dp_tclBaseElement       VmyDpElemEndUserCurrent("DpEndUserCurrent",(tS32)0x96a35491L); 
  dp_tclBaseElement       VmyDpElemEndUserUsed("DpEndUserUsed", (tS32)0x5b5135ebL);
  dp_tclBaseElement       VmyDpElemEndUserBank("DpEndUserBank", (tS32)0x5b512b16L);
  dp_tclBaseElement       VmyDpElemEndUserBankUsed("DpEndUserBankUsed", (tS32)0xf12a67cfL);
  tU32                    Vu32FileLen=0;
  tU8*                    Vpu8Buffer = NULL;

  if((_ptDPShmEndUser==NULL)||(_poPersMemAccessHdl==NULL))
  {
    NORMAL_M_ASSERT_ALWAYS();
  }
  else
  { /*trace out end user info */   
    vTraceOutEndUserInfo();
    /* fill and get length end user VmyDpElemEndUserCurrent*/
    (tVoid)VmyDpElemEndUserCurrent.bFillData((tVoid*)&_ptDPShmEndUser->u8EndUser, sizeof(tU8));
    Vu32FileLen+=VmyDpElemEndUserCurrent.u32GetStreamLength();
    /* fill and get lenght end user list  VmyDpElemEndUserUsed*/ 
    (tVoid)VmyDpElemEndUserUsed.bFillData((tVoid*)&_ptDPShmEndUser->u8EndUserUsed[0], _ptDPShmEndUser->u8EndUserUsedMax);
    Vu32FileLen+=VmyDpElemEndUserUsed.u32GetStreamLength();
    /* fill and get lenght end user list  VmyDpElemEndUserBank*/ 
    (tVoid)VmyDpElemEndUserBank.bFillData((tVoid*)&_ptDPShmEndUser->u8EndUserBank[0],u8GetGreatestEndUser()+1);
    Vu32FileLen+=VmyDpElemEndUserBank.u32GetStreamLength();
    /* fill and get lenght end user list  VmyDpElemEndUserUsed*/ 
    (tVoid)VmyDpElemEndUserBankUsed.bFillData((tVoid*)&_ptDPShmEndUser->u8EndUserBankUsed[0], _ptDPShmEndUser->u8EndUserBankUsedMax);
    Vu32FileLen+=VmyDpElemEndUserBankUsed.u32GetStreamLength();
    /*create stream */
    ETG_TRACE_USR4(("vStoreEndUserInfoFromShm() Vu32FileLen: %d\n",Vu32FileLen));
    Vpu8Buffer = new tU8[Vu32FileLen];
    if (Vpu8Buffer) 
    {
      tU32 Vu32Len;
      tU8* Vpu8CurPos = Vpu8Buffer;
	    // end user current
	    Vu32Len = VmyDpElemEndUserCurrent.u32GetStreamLength();
	    dp_tclOutStreamCtxt oContext1(Vpu8CurPos, Vu32Len);
      oContext1 << VmyDpElemEndUserCurrent;
      Vpu8CurPos = oContext1.pu8GetEndPosition();
	    // end user used
	    Vu32Len = VmyDpElemEndUserUsed.u32GetStreamLength();
	    dp_tclOutStreamCtxt oContext2(Vpu8CurPos, Vu32Len);
      oContext2 << VmyDpElemEndUserUsed;
      Vpu8CurPos = oContext2.pu8GetEndPosition();
	    // end user bank
	    Vu32Len = VmyDpElemEndUserBank.u32GetStreamLength();
	    dp_tclOutStreamCtxt oContext3(Vpu8CurPos, Vu32Len);
      oContext3 << VmyDpElemEndUserBank;
      Vpu8CurPos = oContext3.pu8GetEndPosition();
	    // end user bank used
	    Vu32Len = VmyDpElemEndUserBankUsed.u32GetStreamLength();
	    dp_tclOutStreamCtxt oContext4(Vpu8CurPos, Vu32Len);
      oContext4 << VmyDpElemEndUserBankUsed;
      Vpu8CurPos = oContext3.pu8GetEndPosition();
    }
	  if(_poPersMemAccessHdl->bWriteFfsData(DP_END_USER_STR_POOL_FILE_NAME,Vpu8Buffer,Vu32FileLen,DP_END_USER_POOL_LOCATION,DP_END_USER_POOL_VERSION, TRUE)==FALSE)
    {/*fatal error */
      ETG_TRACE_FATAL(("vStoreEndUserInfoFromShm(): pool '%s' not saved", DP_END_USER_STR_POOL_FILE_NAME));
    }
    delete[] Vpu8Buffer;
    Vpu8Buffer=NULL;/*Bug 123144: Safe handling*/
  }
}
/******************************************************************************
* void dp_tclDpEndUser::vTraceOutEndUserInfo()
*
* DESCRIPTION: trace out end user info
*
* PARAMETERS:  
*
* RETURNS: 
*
* HISTORY: Created  2015 01 26
*****************************************************************************/
void dp_tclDpEndUser::vTraceOutEndUserInfo(void)
{
  if(_ptDPShmEndUser==NULL)
  {
    NORMAL_M_ASSERT_ALWAYS();
  }
  else 
  {         
     ETG_TRACE_USR4(("_ptDPShmEndUser->bLoad '%d' ",_ptDPShmEndUser->bLoad));
     ETG_TRACE_USR4(("_ptDPShmEndUser->u8EndUser '%d' ",_ptDPShmEndUser->u8EndUser));
     ETG_TRACE_USR4(("_ptDPShmEndUser->u8EndUserUsedMax '%d' ",_ptDPShmEndUser->u8EndUserUsedMax));
     ETG_TRACE_USR4(("_ptDPShmEndUser->u8EndUserUsed: %d",ETG_LIST_LEN((_ptDPShmEndUser->u8EndUserUsedMax)), ETG_LIST_PTR_T8(_ptDPShmEndUser->u8EndUserUsed)));	
     ETG_TRACE_USR4(("_ptDPShmEndUser->u8EndUserBank: %d",ETG_LIST_LEN((u8GetGreatestEndUser()+1)), ETG_LIST_PTR_T8(_ptDPShmEndUser->u8EndUserBank)));	
     ETG_TRACE_USR4(("_ptDPShmEndUser->u8EndUserBankUsedMax '%d' ",_ptDPShmEndUser->u8EndUserBankUsedMax));
     ETG_TRACE_USR4(("_ptDPShmEndUser->u8EndUserBankUsed: %d",ETG_LIST_LEN((_ptDPShmEndUser->u8EndUserBankUsedMax)), ETG_LIST_PTR_T8(_ptDPShmEndUser->u8EndUserBankUsed)));	
     ETG_TRACE_USR4(("_ptDPShmEndUser->u16EndUserConfigNotStore: '%d'",_ptDPShmEndUser->u16EndUserConfigNotStore));	
  }
}

/******************************************************************************
* void dp_tclDpEndUser::u8GetGreatestEndUser()
*
* DESCRIPTION: get greatest end user. For each end suer a bank exist
*
*
* RETURNS: greatest end user
*
* HISTORY: Created  2016 02 05
*****************************************************************************/
tU8 dp_tclDpEndUser::u8GetGreatestEndUser()
{
  tU8 Vu8GreatestEndUser=0;
  if(_ptDPShmEndUser==NULL)
  {
    NORMAL_M_ASSERT_ALWAYS();
  }
  else 
  {    
    tU8 Vu8Inc;
    for (Vu8Inc=0;Vu8Inc < _ptDPShmEndUser->u8EndUserUsedMax; Vu8Inc++)
    {
      if(_ptDPShmEndUser->u8EndUserUsed[Vu8Inc]!=0xff) //0xff is unused
      {
        if(_ptDPShmEndUser->u8EndUserUsed[Vu8Inc]>Vu8GreatestEndUser)
          Vu8GreatestEndUser=_ptDPShmEndUser->u8EndUserUsed[Vu8Inc];
      }
    }
  }
  return(Vu8GreatestEndUser);
}

/******************************************************************************
* char* dp_tclDpEndUser::strFileReadConfigLine(const char* PstrConfigStr)
*
* DESCRIPTION: read the config line from file datapool configuration file
*
* PARAMETERS:
*      PstrConfigStr: config string to read
*
* RETURNS: 
*      NULL config string not found 
*      !=NULL config string
*
* HISTORY:Created  2016 03 10
*****************************************************************************/
char* dp_tclDpEndUser::strFileReadConfigLine(const char* PstrConfigStr)
{
  FILE*   VpFile;
  size_t  VtsLen = 0;
  char*   VstrLine = NULL;
  ssize_t VtsRead;
  /*first try to open the config file 'DP_CONFIG_FILE'*/
  VpFile=fopen(DP_CONFIG_FILE,"r");
  if(VpFile==NULL)
  {/*if config file not exist open the 'DP_CONFIG_FILE_DEFAULT'*/
    VpFile=fopen(DP_CONFIG_FILE_DEFAULT,"r");     
  }
  if(VpFile!=NULL)
  { /*read line until found or end*/ 
    do
    { 
      VtsRead=getline(&VstrLine, &VtsLen, VpFile);
      if(VtsRead!=-1)
      { /*check for comment as first char*/
        if (VstrLine[0] != '#')
        {        
          if(strstr(VstrLine,PstrConfigStr)!=0)
            break;
        }
      }
    }while(VtsRead!=-1);      
    fclose(VpFile);
    if(VtsRead==-1)
    {
      free(VstrLine);
      VstrLine=NULL;
    }   
    else
    { /*On success, getline() return the number of characters read, including the delimiter character, but not including the terminating null byte ('\0').  
        This value can be used to handle embedded null bytes in the line read.*/
      VstrLine[VtsRead]=0;    
    }
  }
  else
  {   
    ETG_TRACE_ERR(("dp_tclDpEndUser(): strFileReadConfigLine(): could  not open config file(%s)",DP_CONFIG_FILE_DEFAULT));
  }
  return(VstrLine);
}
/******************************************************************************
* tVoid dp_tclDpEndUser::vEnableReloadEndUserSharedMemory(tVoid)
*
* DESCRIPTION: only for oedt test => set bLoad to FALSE; 
*              the next access reads the configuration again
*
* PARAMETERS:
*
* RETURNS: 
*
* HISTORY:Created  2016 08 3
*****************************************************************************/
#ifdef DP_U32_POOL_ID_PDDDPTESTENDUSER
tVoid dp_tclDpEndUser::vEnableReloadEndUserSharedMemory(tVoid)
{
  if(_ptDPShmEndUser!=NULL)
  {
    if(_ptDPShmEndUser->bLoad==TRUE)
    {
      _ptDPShmEndUser->bLoad=FALSE;
    }
  }
}
#endif

#endif       
		
//EOF