/******************************************************************************/
/**
* \file    dispvidctrl_tclGstRvc.cpp
* \ingroup
*
* \brief
*
* \remark  Copyright : (c) 2015 Robert Bosch GmbH, Hildesheim
* \remark  Author    : 
* \remark  Scope     : AIVI
*
* \todo
*/
/******************************************************************************/

/*******************************************************************************
Includes
*******************************************************************************/

#include "dispvidctrl_tclVideoPlayer.h"

#include "dispvidctrl_tclGstRvc.h"

#include "dispvidctrl_tclControlGraphic.h"
#include "dispvidctrl_datapool.h"

#include <linux/videodev2.h>
#include <linux/v4l2-controls.h>
#include <sys/ioctl.h>
#ifndef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
#include <media/adv718x.h>
#endif
#include <ilm/ilm_client.h>
#include <ilm/ilm_control.h>

#include "VideoPlayer_shm.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_CLIENT_VIDEOPLAYER
#include "trcGenProj/Header/dispvidctrl_tclGstRvc.cpp.trc.h"
#endif

/*******************************************************************************
Defines
*******************************************************************************/

dispvidctrl_tclGstRvc* dispvidctrl_tclGstRvc::_pMyStaticRef = NULL;

const tChar* const aStrVideoNames[] = {
  "/dev/video0",
  "/dev/video1",
  "/dev/video2",
  "/dev/video3",
  "/dev/video4"
};
const tChar* const aStrVideoSources[] = {
  "/sys/class/video4linux/video0/name",
  "/sys/class/video4linux/video1/name",
  "/sys/class/video4linux/video2/name",
  "/sys/class/video4linux/video3/name",
  "/sys/class/video4linux/video4/name"
};
#define GSTRVC_NUM_POSSIBLE_VIDEO_SOURCES  ( sizeof(aStrVideoSources) / sizeof(aStrVideoSources[0]) )


#ifndef VARIANT_S_FTR_ENABLE_AIVI_NPIVI
tChar  strAnalogInputName[] = "ADV7182 Composite on Ain1";
#else
tChar  strAnalogInputName[] = "ADV7182 Composite on Ain4";
#endif //VARIANT_S_FTR_ENABLE_AIVI_NPIVI

tChar  strDigitalInputName[] = "DS90UH940 RVC";
tChar  strDigitalMediaInInputName[] = "DS90UH940 Media In";

/******************************************************************************/
/* FUNCTION     dispvidctrl_tclGstRvc()                                     */
/******************************************************************************/
/**
*  \brief       constructor
*
*  \param       pointer to main application
*  \return      none
*/
/******************************************************************************/

dispvidctrl_tclGstRvc::dispvidctrl_tclGstRvc() : dispvidctrl_tclGstPipeline() {
    ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::dispvidctrl_tclGstRvc() entered.", OSAL_ClockGetElapsedTime()));
    _pMyStaticRef                   = this;

    _u16LayerNumber                 = 0;
    _u16GraphicsLayerNumber         = 0;
    _u16SurfaceNumber               = 0;
    _u16GuidelineSurfaceNumber      = 0;
    _u16ScreenWidth                 = 0;
    _u16ScreenHeight                = 0;
    _u16PixFmt                      = 0;

    _u16CropUp                      = 0;
    _u16CropDown                    = 0;
    _u16CropLeft                    = 0;
    _u16CropRight                   = 0;
    _u16LayerSourceRectangle_X      = 0;
    _u16LayerSourceRectangle_Y      = 0;
    _u16LayerSourceRectangle_Width  = 0;
    _u16LayerSourceRectangle_Height = 0;
    _u16LayerDestinationRectangle_X = 0;
    _u16LayerDestinationRectangle_Y = 0;
    _u16LayerDestinationRectangle_Width = 0;
    _u16LayerDestinationRectangle_Height = 0;
    _u8CameraConnectionType         = 0;
    _u8CameraConnection             = 0;
    _u16CameraWidth                 = 0;
    _u16CameraHeight                = 0;
    _u8CameraImageMirror            = 0;
	
    _u32StartPlayTime               = 0;

     _strVideoSourceName            = 0;
     _strVideoOutName               = 0;
     _strMediaInSourceName          = 0;
     
    _u8AnalogInputChannel           = 0xFF;
    _u8DigitalInputChannel          = 0xFF;
    _u8DigitalMediaInInputChannel   = 0xFF;

    _bRvcActive                     = FALSE;
    _bRvcVisible                    = FALSE;
    _bRvcBlackInComplete            = FALSE;
    _bRvcRendered                   = FALSE;
    _bVideoSignalError              = FALSE;
    _bPipelineActive                = FALSE;
    _bHandoverReady                 = FALSE;
    _bCroppingSet                   = FALSE;
    _bLayerSourceRectangleSet       = FALSE;
    _bLeftHandDrive                 = FALSE;
    
    _bVideoSourceFound              = FALSE;
    _bVideoOutFound                 = FALSE;
    _bMediaInSourceFound            = FALSE;

    _u32StartPlayTime               = 0;

    _poControlGraphic               = 0;
    _povideomanager                 = 0;
}

dispvidctrl_tclGstRvc::dispvidctrl_tclGstRvc(const TEarlyConfiguration& tEarlyConfig, Idispvidctrl_tclClientVideomanager* poClientVideoManager): dispvidctrl_tclGstPipeline() {
    ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::dispvidctrl_tclGstRvc() entered.", OSAL_ClockGetElapsedTime()));
    _pMyStaticRef                 = this;

    _povideomanager               = poClientVideoManager;
    if(_povideomanager == 0) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::dispvidctrl_tclGstRvc() - ERROR - poClientVideoManager is NULL", OSAL_ClockGetElapsedTime()) );
    }

    _u16LayerNumber                       = tEarlyConfig.u32RvcLayer;
    _u16GraphicsLayerNumber               = tEarlyConfig.u32RvcGraphicsLayer;
    _u16SurfaceNumber                     = tEarlyConfig.u32RvcSurface;
    _u16GuidelineSurfaceNumber            = tEarlyConfig.u32GuidelineSurface;
    _u16PixFmt                            = tEarlyConfig.u32RvcPixFmt;

    _u16CropUp                            = tEarlyConfig.tEarlyNorForce.u16VideoCropping_Up;
    _u16CropDown                          = tEarlyConfig.tEarlyNorForce.u16VideoCropping_Down;
    _u16CropLeft                          = tEarlyConfig.tEarlyNorForce.u16VideoCropping_Left;
    _u16CropRight                         = tEarlyConfig.tEarlyNorForce.u16VideoCropping_Right;
    _u16LayerSourceRectangle_X            = tEarlyConfig.tEarlyNorForce.u16VideoSourceRectangle_X;
    _u16LayerSourceRectangle_Y            = tEarlyConfig.tEarlyNorForce.u16VideoSourceRectangle_Y;
    _u16LayerSourceRectangle_Width        = tEarlyConfig.tEarlyNorForce.u16VideoSourceRectangle_Width;
    _u16LayerSourceRectangle_Height       = tEarlyConfig.tEarlyNorForce.u16VideoSourceRectangle_Height;
    _u16LayerDestinationRectangle_X       = tEarlyConfig.tEarlyNorForce.u16VideoDestinationRectangle_X;
    _u16LayerDestinationRectangle_Y       = tEarlyConfig.tEarlyNorForce.u16VideoDestinationRectangle_Y;
    _u16LayerDestinationRectangle_Width   = tEarlyConfig.tEarlyNorForce.u16VideoDestinationRectangle_Width;
    _u16LayerDestinationRectangle_Height  = tEarlyConfig.tEarlyNorForce.u16VideoDestinationRectangle_Height;
    
    _u8CameraConnectionType               = tEarlyConfig.tEarlyNorForce.u8CameraConnectionType;
    _u8CameraConnection                   = _u8CameraConnectionType;

    _u16CameraWidth                    = tEarlyConfig.tEarlyNorForce.u16CameraWidth;
    _u16CameraHeight                   = tEarlyConfig.tEarlyNorForce.u16CameraHeight;
    _u16ScreenWidth                    = tEarlyConfig.tEarlyNorForce.u16Display1Width;
    _u16ScreenHeight                   = tEarlyConfig.tEarlyNorForce.u16Display1Height;
    _u8CameraImageMirror               = tEarlyConfig.tEarlyNorForce.u8CameraImageMirror;
    
    _u8AnalogInputChannel                 = 0xFF;
    _u8DigitalInputChannel                = 0xFF;
    _u8DigitalMediaInInputChannel         = 0xFF;

    _bRvcActive                           = FALSE;
    _bRvcVisible                          = FALSE;
    _bRvcBlackInComplete                  = FALSE;
    _bRvcRendered                         = FALSE;
    _bVideoSignalError                    = FALSE;
    _bPipelineActive                      = FALSE;
    _bHandoverReady                       = FALSE;
    _bCroppingSet                         = tEarlyConfig.tEarlyNorForce.bVideoCroppingSet;
    _bLayerSourceRectangleSet             = tEarlyConfig.tEarlyNorForce.bVideoScalingSet;
    _bLeftHandDrive                       = tEarlyConfig.tEarlyNorForce.bLeftHandSteeringPosition;
    
    _bVideoSourceFound                    = FALSE;
    _bVideoOutFound                       = FALSE;
    _bMediaInSourceFound                  = FALSE;

    _u32StartPlayTime                      = 0;
    _strVideoSourceName                    = 0;
    _strVideoOutName                       = 0;
    _strMediaInSourceName                  = 0;

#ifndef GEN3X86
    // To find the video source device
    if (FALSE == bFindCameraVideoSource())
    {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::dispvidctrl_tclGstRvc() - ERROR - FAILED to find video Source device", OSAL_ClockGetElapsedTime()) );
    }
#endif //GEN3X86

    // Graphic Control Interface
    _poControlGraphic            = new dispvidctrl_tclControlGraphic(tEarlyConfig);


#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE1
    // Requirement for Scope 2 is Blue Screen
    vSetBlackScreenOnCameraDisconnection();
#endif //VARIANT_S_FTR_ENABLE_AIVI_SCOPE1

#ifndef GEN3X86
    vSetAce(FALSE);
#endif //GEN3X86
}


/******************************************************************************/
/* FUNCTION     ~dispvidctrl_tclGstRvc                                      */
/******************************************************************************/
/**
*  \brief       destructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclGstRvc::~dispvidctrl_tclGstRvc(tVoid) {
    ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::~dispvidctrl_tclGstRvc() entered.", OSAL_ClockGetElapsedTime()));

    delete _poControlGraphic;
    _strVideoSourceName = 0;
    _strVideoOutName    = 0;
}


tVoid dispvidctrl_tclGstRvc::vStart(tU8 /*u8Type*/) {
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif

   // vStart performs Layer Manager preparation and starts capturing for the early case
   //     For the late case, capturing is started when videomanager says it is ok by responding with a change of status
   
   // Destination Rectangle parameters for Layer Manager Layer (for the video layer)
   // Note - at this time, we can only make changes to the RVC Layer. We cannot change the RVC surface
   //        because it does not exist until the pipeline is running. This is done when the first frame
   //        is rendered.
   if(_bLayerSourceRectangleSet == TRUE)
   {
      ETG_TRACE_FATAL( ( "dispvidctrl_tclGstRvc::vStart() SourceRectangle Params - X [%d], Y [%d], Width [%d], Height [%d]", 
                        _u16LayerSourceRectangle_X, _u16LayerSourceRectangle_Y, _u16LayerSourceRectangle_Width, _u16LayerSourceRectangle_Height));
      ETG_TRACE_FATAL( ( "dispvidctrl_tclGstRvc::vStart() DestinationRectangle Params - X [%d], Y [%d], Width [%d], Height [%d]", 
                        _u16LayerDestinationRectangle_X, _u16LayerDestinationRectangle_Y, _u16LayerDestinationRectangle_Width, _u16LayerDestinationRectangle_Height));

      //Cropping parameters - only for Renault
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
      if(_bCroppingSet == TRUE)
      {
         ETG_TRACE_FATAL( ( "dispvidctrl_tclGstRvc::vStart() Raw Cropping Parameters - Up [%d], Down [%d], Left [%d], Right [%d]", _u16CropUp,_u16CropDown, _u16CropLeft, _u16CropRight));
         tU16 _u16LayerSourceRectangle_X_cropped = _u16LayerSourceRectangle_X + (_u16CropLeft * _u16LayerSourceRectangle_Width) / 100;
         tU16 _u16LayerSourceRectangle_Y_cropped = _u16LayerSourceRectangle_Y + (_u16CropUp * _u16LayerSourceRectangle_Height) / 100;
         tU16 _u16LayerSourceRectangle_Width_cropped = ( (100 - _u16CropLeft - _u16CropRight) * _u16LayerSourceRectangle_Width) / 100;
         tU16 _u16LayerSourceRectangle_Height_cropped = ( (100 - _u16CropUp - _u16CropDown) * _u16LayerSourceRectangle_Height) / 100;

         ETG_TRACE_FATAL( ( "dispvidctrl_tclGstRvc::vStart() Cropped SourceRectangle Params - X [%d], Y [%d], Width [%d], Height [%d]", 
                        _u16LayerSourceRectangle_X_cropped, _u16LayerSourceRectangle_Y_cropped, _u16LayerSourceRectangle_Width_cropped, _u16LayerSourceRectangle_Height_cropped));
                        
         vSetLayerDestinationRectangle(_u16LayerNumber, _u16LayerDestinationRectangle_X, _u16LayerDestinationRectangle_Y, _u16LayerDestinationRectangle_Width, _u16LayerDestinationRectangle_Height, FALSE);
      }
      else
      {
         vSetLayerDestinationRectangle(_u16LayerNumber, _u16LayerDestinationRectangle_X, _u16LayerDestinationRectangle_Y, _u16LayerDestinationRectangle_Width, _u16LayerDestinationRectangle_Height, FALSE);
      }
#else
      vSetLayerDestinationRectangle(_u16LayerNumber, _u16LayerDestinationRectangle_X, _u16LayerDestinationRectangle_Y, _u16LayerDestinationRectangle_Width, _u16LayerDestinationRectangle_Height, FALSE);

#endif //VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
   }
   
   bool lRc = (ILM_SUCCESS == ilm_commitChanges());
   if(lRc != TRUE) {
       ETG_TRACE_FATAL (("[%d ms] dispvidctrl_tclGstRvc::vStart() - LayerManagerControl Commit Failed [%d]", OSAL_ClockGetElapsedTime(), lRc));
   }
   
   // Start capturing for the early case
   if (_pMyStaticRef->_povideomanager != NULL) {
       if ( (_bHandoverReady == FALSE) || (_pMyStaticRef->_povideomanager->bVideoManagerIsAvailable() == FALSE) ) {
           ETG_TRACE_FATAL (("[%d ms] dispvidctrl_tclGstRvc::vStart() - Start RVC Playing - _bHandoverReady[%d], bVideoManagerIsAvailable[%d]", OSAL_ClockGetElapsedTime(), _bHandoverReady, _pMyStaticRef->_povideomanager->bVideoManagerIsAvailable()));
           vStartRVCPlaying();
       }
       else {
           ETG_TRACE_FATAL (("[%d ms] dispvidctrl_tclGstRvc::vStart() - Delaying RVC Start Playing until videomanager says start", OSAL_ClockGetElapsedTime()));
       }
   }
}

tVoid dispvidctrl_tclGstRvc::vStartRVCPlaying() {
   ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vStartRVCPlaying() entered.", OSAL_ClockGetElapsedTime()));
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif

   DISPVIDCTRL_NULL_POINTER_CHECK(_pGstPipeline);
   if (!_bRvcActive)
   {
     gst_element_set_state(GST_ELEMENT(_pGstPipeline), GST_STATE_PLAYING); //lint !e826 PQM_authorized_multi_415

     _bRvcActive = TRUE;
     _bRvcRendered = FALSE; //Set when first frame has been rendered
  }
   _u32StartPlayTime = OSAL_ClockGetElapsedTime();
}

tVoid dispvidctrl_tclGstRvc::vPause() {
    ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vPause() entered.", OSAL_ClockGetElapsedTime()));
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif

    DISPVIDCTRL_NULL_POINTER_CHECK(_pGstPipeline);
    gst_element_set_state(GST_ELEMENT(_pGstPipeline), GST_STATE_PAUSED); //lint !e826 PQM_authorized_multi_415

    _bRvcActive = FALSE;
}

tVoid dispvidctrl_tclGstRvc::vStop() {
    ETG_TRACE_FATAL (("[%d ms] dispvidctrl_tclGstRvc::vStop() - LayerManagerControl Surface %d and Layers %d, %d (Visibility -> 0), Remove Surface %d", OSAL_ClockGetElapsedTime(), _pMyStaticRef->_u16GuidelineSurfaceNumber, _pMyStaticRef->_u16GraphicsLayerNumber, _pMyStaticRef->_u16LayerNumber, _pMyStaticRef->_u16GuidelineSurfaceNumber));
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif

    bool lRc = (ILM_SUCCESS == ilm_surfaceSetVisibility((unsigned int)_pMyStaticRef->_u16GuidelineSurfaceNumber,0));
    lRc = lRc && (ILM_SUCCESS == ilm_layerRemoveSurface((unsigned int)_pMyStaticRef->_u16GraphicsLayerNumber,(unsigned int)_pMyStaticRef->_u16GuidelineSurfaceNumber));
    lRc = lRc && (ILM_SUCCESS == ilm_layerSetVisibility((unsigned int)_pMyStaticRef->_u16LayerNumber,0));
    lRc = lRc && (ILM_SUCCESS == ilm_layerSetVisibility((unsigned int)_pMyStaticRef->_u16GraphicsLayerNumber,0));

    if(_poControlGraphic != 0) {
        _poControlGraphic->vStopGuidelines();
    }
    else {
        ETG_TRACE_FATAL (("[%d ms] dispvidctrl_tclGstRvc::vStop() - _poControlGraphic is zero", OSAL_ClockGetElapsedTime()));
    }

    // Commit changes to ILM
    lRc = lRc && (ILM_SUCCESS == ilm_commitChanges());
    if(lRc != TRUE) {
        ETG_TRACE_FATAL (("[%d ms] dispvidctrl_tclGstRvc::vStop() - LayerManagerControl Commit Failed [%d]", OSAL_ClockGetElapsedTime(), lRc));
    }

    // If LVDS configured, switch input back to media input, i.e. set input to 5
    // This will help diagnostic components(Tuner and Heatctrl) which needs to access the external TV box with I2C via LVDS.
    // Otherwise DTV application has to send this command each time RVC operation finished even when DTV operation is not needed.

    if(_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_LVDS) {
       ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vStop() LVDS - switch input to Media In- input channel %d", OSAL_ClockGetElapsedTime(), _u8DigitalMediaInInputChannel));
       if(_bMediaInSourceFound==TRUE) {
          if(_pGstSource != 0) {
             g_object_set (G_OBJECT (_pGstSource), "input", _u8DigitalMediaInInputChannel, NULL); //lint !e826 PQM_authorized_multi_415
          }
       }
    }

    DISPVIDCTRL_NULL_POINTER_CHECK(_pGstPipeline);
    gst_element_set_state(GST_ELEMENT(_pGstPipeline), GST_STATE_NULL); //lint !e826 PQM_authorized_multi_415

    _bRvcActive = FALSE;
    _bRvcVisible = FALSE;
}

//******************************************************************************

//FUNCTION:    tVoid dispvidctrl_tclGstRvc::vCompleteCVBSPipelineCsc(tVoid)

//Links v4l_csc into the GStreamer pipeline
// It is needed to implement Mirroring which is a Renault feature

//!

//PARAMETER:

//RETURNVALUE: None

//******************************************************************************
tVoid dispvidctrl_tclGstRvc::vCompleteCVBSPipelineCsc(tVoid) {
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif

    _pGstQueue_Csc = gst_element_factory_make ("queue", "_pGstQueue_Csc");
    if (_pGstQueue_Csc == 0) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vCompleteCVBSPipelineCsc() late Could not create queue_Csc", OSAL_ClockGetElapsedTime()));
    }

    _pGstCsc = gst_element_factory_make ("v4l_csc", "rvc_v4l_csc");
    if(_pGstCsc == 0) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vCompleteCVBSPipelineCsc() late Could not create v4l_csc", OSAL_ClockGetElapsedTime()));
    }

    gst_bin_add_many (GST_BIN (_pGstPipeline), _pGstSource, _pGstQueue, _pGstCsc, _pGstQueue_Csc, _pGstSink, NULL); //lint !e826 PQM_authorized_multi_415

    if( !gst_element_link_many (_pGstSource, _pGstQueue, NULL)) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vCompleteCVBSPipelineCsc() late Could not link _pGstSource to _pGstQueue", OSAL_ClockGetElapsedTime()));
    }

    if( !gst_element_link_many (_pGstQueue, _pGstCsc, NULL)) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vCompleteCVBSPipelineCsc() late Could not link _pGstQueue to _pGstCsc", OSAL_ClockGetElapsedTime()));
    }

    if( !gst_element_link_many (_pGstCsc, _pGstQueue_Csc, NULL)) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vCompleteCVBSPipelineCsc() late Could not link _pGstCsc to _pGstQueue_Csc", OSAL_ClockGetElapsedTime()));
    }

    if( !gst_element_link_many (_pGstQueue_Csc, _pGstSink, NULL)) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vCompleteCVBSPipelineCsc() late Could not link _pGstQueue_Csc to _pGstSink", OSAL_ClockGetElapsedTime()));
    }
}

//******************************************************************************

//FUNCTION:    tVoid dispvidctrl_tclGstRvc::vCompleteCVBSPipelineNoCsc(tVoid)

//Complete the GStreamer pipeline for the simple case - no Csc

//!

//PARAMETER:

//RETURNVALUE: None

//******************************************************************************
tVoid dispvidctrl_tclGstRvc::vCompleteCVBSPipelineNoCsc(tVoid) {
        gst_bin_add_many (GST_BIN (_pGstPipeline), _pGstSource, _pGstQueue, _pGstSink, NULL); //lint !e826 PQM_authorized_multi_415
        
        if( !gst_element_link_many (_pGstSource, _pGstQueue, NULL)) {
            ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vCompleteCVBSPipelineNoCsc() late Could not link _pGstSource to _pGstQueue", OSAL_ClockGetElapsedTime()));
        }
        
        if( !gst_element_link_many (_pGstQueue, _pGstSink, NULL)) {
            ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vCompleteCVBSPipelineNoCsc() late Could not link _pGstQueue to _pGstSink", OSAL_ClockGetElapsedTime()));
        }
}

void dispvidctrl_tclGstRvc::setGraphicslayerVisibility(unsigned int visibility)
{
	graphiclayervisibility(visibility);
}
//******************************************************************************

//FUNCTION:    tVoid dispvidctrl_tclGstRvc::vCompleteLVDSPipeline(tVoid)

//Links videorate and Csc into the GStreamer pipeline
// videorate is needed for LVDS to reduce the external frame rate (60fps) to 30fps to reduce risk of flickering due to bandwidth performance problems
// Csc is needed for colour, brightness, etc settings

//!

//PARAMETER:

//RETURNVALUE: None

//******************************************************************************
tVoid dispvidctrl_tclGstRvc::vCompleteLVDSPipeline(tVoid) {
    
#ifdef ENABLE_CSC_FOR_LVDS
    _pGstQueue_Csc = gst_element_factory_make ("queue", "_pGstQueue_Csc");
    if (_pGstQueue_Csc == 0) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vCompleteLVDSPipeline() late Could not create queue_Csc", OSAL_ClockGetElapsedTime()));
    }

    _pGstCsc = gst_element_factory_make ("v4l_csc", "rvc_v4l_csc");
    if(_pGstCsc == 0) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vCompleteLVDSPipeline() late Could not create v4l_csc", OSAL_ClockGetElapsedTime()));
    }
#endif //ENABLE_CSC_FOR_LVDS
  
#ifdef ENABLE_60_TO_30_REDUCTION
    _pGstVideoRate = gst_element_factory_make ("videorate", "rvc_videorate");
    if(_pGstVideoRate == 0) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vCompleteLVDSPipeline() Could not create videorate", OSAL_ClockGetElapsedTime()));
    }

    _pGstCapsFilter = gst_element_factory_make("capsfilter","cfilter");
    if(_pGstCapsFilter == 0) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vCompleteLVDSPipeline() Could not create videorate caps filter", OSAL_ClockGetElapsedTime()));
    }

    gst_bin_add_many (GST_BIN (_pGstPipeline), _pGstSource, _pGstVideoRate, _pGstCapsFilter, _pGstQueue, _pGstCsc, _pGstQueue_Csc, _pGstSink, NULL); //lint !e826 PQM_authorized_multi_415

    if( !gst_element_link_many (_pGstSource, _pGstVideoRate, NULL)) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vCompleteLVDSPipeline() Could not link _pGstSource to _pGstVideoRate", OSAL_ClockGetElapsedTime()));
    }

    if( !gst_element_link_many (_pGstVideoRate, _pGstCapsFilter, NULL)) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vCompleteLVDSPipeline() Could not link _pGstVideoRate to _pGstCapsFilter", OSAL_ClockGetElapsedTime()));
    }

    if( !gst_element_link_many (_pGstCapsFilter, _pGstQueue, NULL)) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vCompleteLVDSPipeline() Could not link _pGstCapsFilter to _pGstQueue", OSAL_ClockGetElapsedTime()));
    }

    if( !gst_element_link_many (_pGstQueue, _pGstCsc, NULL)) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vCompleteLVDSPipeline() Could not link _pGstCapsFilter to _pGstQueue", OSAL_ClockGetElapsedTime()));
    }

    if( !gst_element_link_many (_pGstCsc, _pGstQueue_Csc, NULL)) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vCompleteLVDSPipeline() Could not link _pGstCapsFilter to _pGstQueue", OSAL_ClockGetElapsedTime()));
    }
#else
 
#ifdef ENABLE_CSC_FOR_LVDS
    gst_bin_add_many (GST_BIN (_pGstPipeline), _pGstSource, _pGstQueue, _pGstCsc, _pGstQueue_Csc, _pGstSink, NULL); //lint !e826 PQM_authorized_multi_415

    if( !gst_element_link_many (_pGstSource, _pGstQueue, NULL)) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vCompleteLVDSPipeline() Could not link _pGstCapsFilter to _pGstQueue", OSAL_ClockGetElapsedTime()));
    }

    if( !gst_element_link_many (_pGstQueue, _pGstCsc, NULL)) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vCompleteLVDSPipeline() Could not link _pGstCapsFilter to _pGstQueue", OSAL_ClockGetElapsedTime()));
    }

    if( !gst_element_link_many (_pGstCsc, _pGstQueue_Csc, NULL)) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vCompleteLVDSPipeline() Could not link _pGstCapsFilter to _pGstQueue", OSAL_ClockGetElapsedTime()));
    } 
#else
    gst_bin_add_many (GST_BIN (_pGstPipeline), _pGstSource, _pGstQueue, _pGstSink, NULL); //lint !e826 PQM_authorized_multi_415

    if( !gst_element_link_many (_pGstSource, _pGstQueue, NULL)) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vCompleteLVDSPipeline() Could not link _pGstSource to _pGstQueue", OSAL_ClockGetElapsedTime()));
    }
#endif //ENABLE_CSC_FOR_LVDS

#endif //ENABLE_60_TO_30_REDUCTION

    if( !gst_element_link_many (_pGstQueue, _pGstSink, NULL)) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vCompleteLVDSPipeline() Could not link _pGstQueue to _pGstSink", OSAL_ClockGetElapsedTime()));
    }
}

tBool dispvidctrl_tclGstRvc::bInit(GMainLoop *pGstMainLoop) {
    ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::bInit() entered.", OSAL_ClockGetElapsedTime()));
    if (_pGstSource != NULL && _pGstSink != NULL) {
        // we are already initialized
        return TRUE;
    }

    tU32 u32StartTime = OSAL_ClockGetElapsedTime();

#ifndef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    _pGstMainLoop = pGstMainLoop;
    if (_pGstMainLoop == 0) {
        ETG_TRACE_FATAL(( "dispvidctrl_tclGstRvc::bInit() pGstMainLoop is NULL"));
    }

    _pGstPipeline = gst_pipeline_new ("RVC");
    if (_pGstPipeline == 0) {
        ETG_TRACE_FATAL(( "dispvidctrl_tclGstRvc::bInit() Could not create RVC pipeline"));
    }

    /* we add a message handler */
    _pGstBus = gst_pipeline_get_bus (GST_PIPELINE (_pGstPipeline)); //lint !e826 PQM_authorized_multi_415
    if (_pGstBus == 0) {
        ETG_TRACE_FATAL(( "dispvidctrl_tclGstRvc::bInit() Could not create RVC pipeline message handler"));
    }

    gst_bus_add_watch (_pGstBus, dispvidctrl_tclGstRvc::bBusCall, _pGstMainLoop);

    _pGstSource = gst_element_factory_make ("mfw_v4lsrc", "v4lsource");
    if (_pGstSource == 0) {
        ETG_TRACE_FATAL(( "dispvidctrl_tclGstRvc::bInit() RVC pipeline Could not create mfw_v4lsrc"));
    } else {
        dispvidctrl_tclAppMain::vWriteStartupLog("[RVC-EARLY]: RVC pipeline create mfw_v4lsrc done", OSAL_ClockGetElapsedTime() - u32StartTime);
    }

    g_signal_connect(G_OBJECT(_pGstSource), "v4l-buf-flag-error", G_CALLBACK(dispvidctrl_tclGstRvc::vBufferErrorCB), NULL); //lint !e826 PQM_authorized_multi_415
    g_signal_connect(G_OBJECT(_pGstSource), "v4l-buf-flag-ok", G_CALLBACK(dispvidctrl_tclGstRvc::vBufferErrorDoneCB), NULL); //lint !e826 PQM_authorized_multi_415

    _pGstQueue = gst_element_factory_make ("queue", "_pGstQueue");
    if (_pGstQueue == 0) {
        ETG_TRACE_FATAL(( "dispvidctrl_tclGstRvc::bInit() Could not create RVC Queue"));
    }

    _pGstSink = gst_element_factory_make("gst_apx_sink", "apxsink");
    if (_pGstSink == 0) {
        ETG_TRACE_FATAL( ( "dispvidctrl_tclGstRvc::bInit() RVC pipeline Could not create gst_apx_sink"));
    } else {
        dispvidctrl_tclAppMain::vWriteStartupLog("[RVC-EARLY]: RVC pipeline create gst_apx_sink done", OSAL_ClockGetElapsedTime() - u32StartTime);
    }

    g_signal_connect(_pGstSink, "first-videoframe-rendered", G_CALLBACK (dispvidctrl_tclGstRvc::vFirstFrameRendered), NULL);

    // Complete the pipeline for the following cases:-
    //   LVDS
    //   CVBS for Renault - includes CSC
    //   CVBS for Nissan
    if (_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_LVDS) {
        vCompleteLVDSPipeline();
    }
    else if (_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_CVBS) {

        //CSC is needed for Renault
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
        vCompleteCVBSPipelineCsc();
#else
        vCompleteCVBSPipelineNoCsc();
#endif //VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
    } 
    else {
       // do nothing
    }

    //Set all GStreamer Pipeline Objects
    vSetAllRvcObjects();
#endif
    dispvidctrl_tclAppMain::vWriteStartupLog("[RVC-EARLY]: RVC pipeline creation complete", OSAL_ClockGetElapsedTime() - u32StartTime);
    return TRUE;
}

//******************************************************************************

//FUNCTION:    tVoid vSetMirroring(tVoid)

//Sets v4l_csc mirroring parameters (hflip and vflip)
// This is a Renault only feature

//!

//PARAMETER:

//RETURNVALUE: None

//******************************************************************************
tVoid dispvidctrl_tclGstRvc::vSetMirroring()
{
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif

   if (_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_CVBS) {

       // Set horizontal/vertical flipping
       ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vSetMirroring() CameraImageMirror %d", OSAL_ClockGetElapsedTime(), _u8CameraImageMirror));
       
       switch (_u8CameraImageMirror)
       {
       case DISPVIDCTRL_AllianceCameraImageMirror_None:
           {
               ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vSetMirroring() Setting vflip=false hflip=false", OSAL_ClockGetElapsedTime()));
               g_object_set (G_OBJECT (_pGstCsc), "vflip", 0, NULL); //lint !e826 PQM_authorized_multi_415
               g_object_set (G_OBJECT (_pGstCsc), "hflip", 0, NULL); //lint !e826 PQM_authorized_multi_415
               break;
           }
       case DISPVIDCTRL_AllianceCameraImageMirror_Vertical:
           {
               ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vSetMirroring() Setting vflip=true hflip=false", OSAL_ClockGetElapsedTime()));
               g_object_set (G_OBJECT (_pGstCsc), "vflip", 1, NULL); //lint !e826 PQM_authorized_multi_415
               g_object_set (G_OBJECT (_pGstCsc), "hflip", 0, NULL); //lint !e826 PQM_authorized_multi_415
               break;
           }
       case DISPVIDCTRL_AllianceCameraImageMirror_Horizontal:
           {
               ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vSetMirroring() Setting vflip=false hflip=true", OSAL_ClockGetElapsedTime()));
               g_object_set (G_OBJECT (_pGstCsc), "vflip", 0, NULL); //lint !e826 PQM_authorized_multi_415
               g_object_set (G_OBJECT (_pGstCsc), "hflip", 1, NULL); //lint !e826 PQM_authorized_multi_415
               break;
           }
       
       case DISPVIDCTRL_AllianceCameraImageMirror_Vertical_plus_Horizontal:
           {
               ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vSetMirroring() Setting vflip=true and hflip=true", OSAL_ClockGetElapsedTime()));
               g_object_set (G_OBJECT (_pGstCsc), "hflip", 1, NULL); //lint !e826 PQM_authorized_multi_415
               g_object_set (G_OBJECT (_pGstCsc), "vflip", 1, NULL); //lint !e826 PQM_authorized_multi_415
               break;
       }
       default:
           break;
       }
   }
}

//******************************************************************************

//FUNCTION:    tVoid vSetCscObjects(tVoid)

//Sets v4l_csc specific parameters
// This is a Renault only feature

//!

//PARAMETER:

//RETURNVALUE: None

//******************************************************************************
tVoid dispvidctrl_tclGstRvc::vSetCscObjects()
{
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif

   if (_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_CVBS) {

       DISPVIDCTRL_NULL_POINTER_CHECK(_pGstQueue_Csc);
       g_object_set (G_OBJECT(_pGstQueue_Csc), "leaky", 2, NULL); //lint !e826 PQM_authorized_multi_415
       g_object_set (G_OBJECT (_pGstQueue_Csc), "max-size-buffers", 2, NULL); //lint !e826 PQM_authorized_multi_415
       
       DISPVIDCTRL_NULL_POINTER_CHECK(_pGstCsc)
       if (_strVideoOutName != NULL)
       {
           g_object_set (G_OBJECT (_pGstCsc), "devicename", _strVideoOutName, NULL); //lint !e826 PQM_authorized_multi_415
       }
   }
}

#ifdef ENABLE_60_TO_30_REDUCTION
//******************************************************************************

//FUNCTION:    tVoid vSetVideoRateObjects(tVoid)

//Sets videorate specific parameters
// This is an LVDS only feature

//!

//PARAMETER:

//RETURNVALUE: None

//******************************************************************************
tVoid dispvidctrl_tclGstRvc::vSetVideoRateObjects()
{
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif
   ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vSetVideoRateObjects() entered", OSAL_ClockGetElapsedTime()));
   if (_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_LVDS) {
       DISPVIDCTRL_NULL_POINTER_CHECK(_pGstVideoRate);
       g_object_set (G_OBJECT (_pGstVideoRate), "drop-only", true, NULL); //lint !e826 PQM_authorized_multi_415

       DISPVIDCTRL_NULL_POINTER_CHECK(_pGstCapsFilter);
        g_object_set(G_OBJECT(_pGstCapsFilter),"caps",gst_caps_new_simple("video/x-raw,framerate=30/1",NULL),NULL);
   }
}
#endif //ENABLE_60_TO_30_REDUCTION

tVoid dispvidctrl_tclGstRvc::vSetAllRvcObjects()
{
    ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vSetAllRvcObjects() entered", OSAL_ClockGetElapsedTime()));
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif

    DISPVIDCTRL_NULL_POINTER_CHECK(_pGstSource);
    
    if (_strVideoSourceName != NULL) {
        g_object_set (G_OBJECT (_pGstSource), "device", _strVideoSourceName, NULL); //lint !e826 PQM_authorized_multi_415
    } else {
        ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vSetAllRvcObjects() Video device Not Found!!! Setting NULL", OSAL_ClockGetElapsedTime()));
        g_object_set (G_OBJECT (_pGstSource), "device", "", NULL); //lint !e826 PQM_authorized_multi_415
    }

    g_object_set (G_OBJECT (_pGstSource), "pix-fmt", _u16PixFmt, NULL); //lint !e826 PQM_authorized_multi_415

    if(_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_CVBS) {
	   // motion parameter is not supported for LVDS
       // Deinterlace motion parameter setting - motion=3: high motion
       // because of reduced vertical resolution for motion 3 we have changed to motion 1
       // (NCG3D-20522, NCG3D-11143)
       g_object_set (G_OBJECT (_pGstSource), "motion", 1, NULL); //lint !e826 PQM_authorized_multi_415
	}

    if(_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_LVDS) {
       ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vSetAllRvcObjects() LVDS - setting input=%d", OSAL_ClockGetElapsedTime(), _u8DigitalInputChannel));
       g_object_set (G_OBJECT (_pGstSource), "input", _u8DigitalInputChannel, NULL); //lint !e826 PQM_authorized_multi_415

       ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vSetAllRvcObjects() LVDS - setting capture area 1280 x 720", OSAL_ClockGetElapsedTime()));
//TODO - Keep hard coded values as we don't have capture_width and capture_height defined currently
       g_object_set (G_OBJECT (_pGstSource), "capture-width", 1280, NULL); //lint !e826 PQM_authorized_multi_415
       g_object_set (G_OBJECT (_pGstSource), "capture-height", 720, NULL); //lint !e826 PQM_authorized_multi_415
       g_object_set (G_OBJECT (_pGstSource), "fps-n", 60, NULL); //lint !e826 PQM_authorized_multi_415
       g_object_set (G_OBJECT (_pGstSource), "fps-d", 1, NULL); //lint !e826 PQM_authorized_multi_415
       
#ifdef ENABLE_60_TO_30_REDUCTION
       g_object_set (G_OBJECT (_pGstSource), "fps-out-n", 30, NULL); //lint !e826 PQM_authorized_multi_415
       g_object_set (G_OBJECT (_pGstSource), "fps-out-d", 1, NULL); //lint !e826 PQM_authorized_multi_415
#endif //ENABLE_60_TO_30_REDUCTION

       // Setting stream-good-buf-only avoids pink frames. Option description "Only good buffers are pushed to next element"
	   // Note - this is only set for LVDS currently as for CVBS the Blue screen no longer works
	   g_object_set (G_OBJECT (_pGstSource), "stream-good-buf-only", 1, NULL); //lint !e826 PQM_authorized_multi_415
    }
    else if(_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_CVBS) {
       ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vSetAllRvcObjects() CVBS - setting input channel as %d", OSAL_ClockGetElapsedTime(), _u8AnalogInputChannel));
       g_object_set (G_OBJECT (_pGstSource), "input", _u8AnalogInputChannel, NULL); //lint !e826 PQM_authorized_multi_415
    }
	
    DISPVIDCTRL_NULL_POINTER_CHECK(_pGstSink);
    g_object_set (G_OBJECT (_pGstSink), "layer-id", _u16LayerNumber, NULL); //lint !e826 PQM_authorized_multi_415
    g_object_set (G_OBJECT (_pGstSink), "surface-id", _u16SurfaceNumber, NULL); //lint !e826 PQM_authorized_multi_415
    g_object_set (G_OBJECT (_pGstSink), "display-width", _u16ScreenWidth, NULL); //lint !e826 PQM_authorized_multi_415
    g_object_set (G_OBJECT (_pGstSink), "display-height", _u16ScreenHeight, NULL); //lint !e826 PQM_authorized_multi_415

    g_object_set (G_OBJECT(_pGstQueue), "leaky", 2, NULL); //lint !e826 PQM_authorized_multi_415
    g_object_set (G_OBJECT (_pGstQueue), "max-size-buffers", 2, NULL); //lint !e826 PQM_authorized_multi_415

    // For Renault Analogue and Digital, we need v4l_csc for flipping
    // For all Digital, we need v4l_csc for colour, brightness, etc settings
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
    if (_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_CVBS) {
       vSetMirroring();
       vSetCscObjects();
    }
#endif //VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1

    if (_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_LVDS) {
#ifdef ENABLE_CSC_FOR_LVDS
       vSetCscObjects();
#endif //ENABLE_CSC_FOR_LVDS

#ifdef ENABLE_60_TO_30_REDUCTION
       vSetVideoRateObjects();
#endif //ENABLE_60_TO_30_REDUCTION
    }
}

tVoid dispvidctrl_tclGstRvc::vShowRvc()
{
    ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vShowRvc() entered", OSAL_ClockGetElapsedTime()));
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif

    _bRvcBlackInComplete = TRUE; //Set when the the Black In period is complete
	
    //Only start drawing guidelines if first frame rendered - it may have happened during black in
    if(_bRvcRendered == TRUE) {
        if(_bRvcVisible == FALSE) {
            DISPVIDCTRL_NULL_POINTER_CHECK(_pMyStaticRef);
            DISPVIDCTRL_NULL_POINTER_CHECK(_pMyStaticRef->_poControlGraphic);
            _pMyStaticRef->_poControlGraphic->vSwitchToGuidelinesDisplay();

            _bRvcVisible = TRUE;
        }
        else {
            ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vShowRvc() ignoring as RVC already visible", OSAL_ClockGetElapsedTime()));
        }
    }
}

tVoid dispvidctrl_tclGstRvc::vShowBlackScreen(tU8 u8BlackScreenType)
{
    ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vShowBlackScreen() entered", OSAL_ClockGetElapsedTime()));
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif

    if (u8BlackScreenType != 0) {
        _bRvcBlackInComplete = FALSE; // Set to show that Black In period is not yet complete
    }
	
    DISPVIDCTRL_NULL_POINTER_CHECK(_pMyStaticRef);
    DISPVIDCTRL_NULL_POINTER_CHECK(_pMyStaticRef->_poControlGraphic);
    if (!_pMyStaticRef->_bPipelineActive) {
        ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vShowBlackScreen() starting guidelines", OSAL_ClockGetElapsedTime()));
        _pMyStaticRef->_poControlGraphic->vStartGuidelines();
        _pMyStaticRef->_bPipelineActive = TRUE;
    }

    _pMyStaticRef->_poControlGraphic->vSwitchToBlackScreen(u8BlackScreenType);
    _bRvcVisible = FALSE;

    //If this is the lead in black screen then set Video Manager RVC state to active
    if(_bRvcVisible == FALSE) {
        ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vShowBlackScreen() Set videomanager RVC state to active", OSAL_ClockGetElapsedTime()));
        //DISPVIDCTRL_NULL_POINTER_CHECK(_pMyStaticRef->_povideomanager);
        if(_pMyStaticRef->_povideomanager != NULL) {
           _pMyStaticRef->_povideomanager->vSetRVCStateActive();
        }
        else {
            ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vShowBlackScreen() _pMyStaticRef->_povideomanager is NULL", OSAL_ClockGetElapsedTime()));
        }
    }

    ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vShowBlackScreen() exited", OSAL_ClockGetElapsedTime()));
}

// vShowGreenScreen is used when there is no LVDS signal
tVoid dispvidctrl_tclGstRvc::vShowGreenScreen(tU8 u8BlackScreenType)
{
    ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vShowGreenScreen() entered", OSAL_ClockGetElapsedTime()));

    set_green_screen();
    DISPVIDCTRL_NULL_POINTER_CHECK(_pMyStaticRef);
    DISPVIDCTRL_NULL_POINTER_CHECK(_pMyStaticRef->_poControlGraphic);
    if (!_pMyStaticRef->_bPipelineActive) {
        ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vShowGreenScreen() starting guidelines", OSAL_ClockGetElapsedTime()));
        _pMyStaticRef->_poControlGraphic->vStartGuidelines();
        _pMyStaticRef->_bPipelineActive = TRUE;
    }

    _pMyStaticRef->_poControlGraphic->vSwitchToBlackScreen(u8BlackScreenType);
}

tVoid dispvidctrl_tclGstRvc::vPreInitGraphics()
{
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif

    DISPVIDCTRL_NULL_POINTER_CHECK(_pMyStaticRef);
    DISPVIDCTRL_NULL_POINTER_CHECK(_pMyStaticRef->_poControlGraphic);
    _pMyStaticRef->_poControlGraphic->vOnPreInit();
}

tVoid dispvidctrl_tclGstRvc::vFirstFrameRendered(GstElement * element, GstPad *pad, gpointer user_data)
{
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vFirstFrameRendered GST Callback()", OSAL_ClockGetElapsedTime()));
    element = element;                       // Unused
    pad = pad;                               // Unused
    user_data = user_data;                   // Unused

    dispvidctrl_tclAppMain::vWriteStartupLog("[RVC-EARLY]: RVC FirstFrameRendered GST Callback", OSAL_ClockGetElapsedTime() - _pMyStaticRef->_u32StartPlayTime);
    dispvidctrl_tclVideoPlayer::vFirstFrame();
}

// Handling the first frame in a synchronization with our Main thread
tVoid dispvidctrl_tclGstRvc::vHandleFirstFrameRendered(tVoid)
{
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vHandleFirstFrameRendered()", OSAL_ClockGetElapsedTime()));
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif
    _bRvcRendered = TRUE; //Indicates first frame rendered - used by vShowRvc to check whether to start drawing guidelines
	
    if (_bRvcActive)
    {
       if(_pMyStaticRef->_bHandoverReady == FALSE)
       {
           //For the early case the surface visibility is set now
           //For the late case, it is delayed until black screen lead in is complete
           ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vHandleFirstFrameRendered() calling vShowRvc - _bHandoverReady[%d]", OSAL_ClockGetElapsedTime(), _pMyStaticRef->_bHandoverReady));
           _pMyStaticRef->vShowRvc();

           // Source Rectangle parameters for Layer Manager Layer (for the video layer)
           // Destination Rectangle parameters for Layer Manager Layer (for the video layer)
		   // Note - this needs to be done here because Layer Manager does not allow the Source Rectangle to be set until the surface has been created
           if(_bLayerSourceRectangleSet == TRUE)
           {
               ETG_TRACE_FATAL( ( "dispvidctrl_tclGstRvc::vHandleFirstFrameRendered() SourceRectangle Params - X [%d], Y [%d], Width [%d], Height [%d]", 
                        _u16LayerSourceRectangle_X, _u16LayerSourceRectangle_Y, _u16LayerSourceRectangle_Width, _u16LayerSourceRectangle_Height));
               ETG_TRACE_FATAL( ( "dispvidctrl_tclGstRvc::vHandleFirstFrameRendered() DestinationRectangle Params - X [%d], Y [%d], Width [%d], Height [%d]", 
                        _u16LayerDestinationRectangle_X, _u16LayerDestinationRectangle_Y, _u16LayerDestinationRectangle_Width, _u16LayerDestinationRectangle_Height));

               //Cropping parameters - only for Renault
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
               if(_bCroppingSet == TRUE)
               {
                   ETG_TRACE_FATAL( ( "dispvidctrl_tclGstRvc::vHandleFirstFrameRendered() Raw Cropping Parameters - Up [%d], Down [%d], Left [%d], Right [%d]", _u16CropUp,_u16CropDown, _u16CropLeft, _u16CropRight));
                   tU16 _u16LayerSourceRectangle_X_cropped = _u16LayerSourceRectangle_X + (_u16CropLeft * _u16LayerSourceRectangle_Width) / 100;
                   tU16 _u16LayerSourceRectangle_Y_cropped = _u16LayerSourceRectangle_Y + (_u16CropUp * _u16LayerSourceRectangle_Height) / 100;
                   tU16 _u16LayerSourceRectangle_Width_cropped = ( (100 - _u16CropLeft - _u16CropRight) * _u16LayerSourceRectangle_Width) / 100;
                   tU16 _u16LayerSourceRectangle_Height_cropped = ( (100 - _u16CropUp - _u16CropDown) * _u16LayerSourceRectangle_Height) / 100;

                   ETG_TRACE_FATAL( ( "dispvidctrl_tclGstRvc::vHandleFirstFrameRendered() Cropped SourceRectangle Params - X [%d], Y [%d], Width [%d], Height [%d]", 
                        _u16LayerSourceRectangle_X_cropped, _u16LayerSourceRectangle_Y_cropped, _u16LayerSourceRectangle_Width_cropped, _u16LayerSourceRectangle_Height_cropped));
         
                   vSetSurfaceSourceRectangle(_u16SurfaceNumber, _u16LayerSourceRectangle_X_cropped, _u16LayerSourceRectangle_Y_cropped, _u16LayerSourceRectangle_Width_cropped, _u16LayerSourceRectangle_Height_cropped, FALSE);
               }
               else
               {
                   vSetSurfaceSourceRectangle(_u16SurfaceNumber, _u16LayerSourceRectangle_X, _u16LayerSourceRectangle_Y, _u16LayerSourceRectangle_Width, _u16LayerSourceRectangle_Height, FALSE);
               }
#else
               vSetSurfaceSourceRectangle(_u16SurfaceNumber, _u16LayerSourceRectangle_X, _u16LayerSourceRectangle_Y, _u16LayerSourceRectangle_Width, _u16LayerSourceRectangle_Height, FALSE);
#endif //VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
		   }

           ETG_TRACE_FATAL (("[%d ms] dispvidctrl_tclGstRvc::vHandleFirstFrameRendered() - LayerManagerControl Set Surface %d Visibility -> 1 and Layer [%d] Visibility -> 1", OSAL_ClockGetElapsedTime(), _pMyStaticRef->_u16SurfaceNumber, _pMyStaticRef->_u16LayerNumber));
           bool lRc = (ILM_SUCCESS == ilm_surfaceSetVisibility((unsigned int)_pMyStaticRef->_u16SurfaceNumber,1));
           lRc = lRc && (ILM_SUCCESS == ilm_layerSetVisibility((unsigned int)_pMyStaticRef->_u16LayerNumber,1));

           // Commit changes to ILM
           lRc = lRc && (ILM_SUCCESS == ilm_commitChanges());
           if(lRc != TRUE) {
               ETG_TRACE_FATAL (("[%d ms] dispvidctrl_tclGstRvc::vHandleFirstFrameRendered() - LayerManagerControl Commit Failed [%d]", OSAL_ClockGetElapsedTime(), lRc));
           }
		   
           if (_pMyStaticRef->_poControlGraphic && !_pMyStaticRef->_bPipelineActive)
           {
              _pMyStaticRef->_poControlGraphic->vStartGuidelines();
           }

           _pMyStaticRef->_bPipelineActive = TRUE;
       }
	   else {
           //late case - only start guideline drawing if black in period is complete
           if(_bRvcBlackInComplete == TRUE) {
               if (!_pMyStaticRef->_bPipelineActive)
               {
				// CID: 174908
				
                  if(_pMyStaticRef->_poControlGraphic)
				  {
					  _pMyStaticRef->_poControlGraphic->vStartGuidelines();
					  _pMyStaticRef->_bPipelineActive = TRUE;
				  }
               }
               else
               {
                  _pMyStaticRef->_poControlGraphic->vSwitchToGuidelinesDisplay();
               }
           }
       }
    }
    else 
    {
       ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vHandleFirstFrameRendered() RVC was stopped", OSAL_ClockGetElapsedTime()));
    }
}

gboolean dispvidctrl_tclGstRvc::bBusCall(GstBus *bus, GstMessage *msg,gpointer data)
{
    bus = bus; // Unused
    data = data; // Unused
    GstState oldstate;
    GstState newstate;
    GstState pending;

    switch (GST_MESSAGE_TYPE(msg))
    {
    case GST_MESSAGE_STREAM_STATUS:
        {
            GstStreamStatusType type;
            GstElement* owner;
            gst_message_parse_stream_status(msg, &type, &owner);

            ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::bBusCall() streamstatus type: %d owner: %s", OSAL_ClockGetElapsedTime(),type,GST_OBJECT_NAME(owner)));
            break;
        }
    case GST_MESSAGE_STATE_CHANGED:
        {
            gst_message_parse_state_changed (msg, &oldstate, &newstate, &pending);

            ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::bBusCall() old: %d new: %d pending: %d state-changed: %s", OSAL_ClockGetElapsedTime(),oldstate,newstate,pending, GST_MESSAGE_SRC_NAME(msg)));

            break;
        }
    case GST_MESSAGE_EOS:
        {
            ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::bBusCall() End of stream", OSAL_ClockGetElapsedTime()));
            break;
        }

    case GST_MESSAGE_ERROR:
        {
            gchar* debug;
            GError* error;

            gst_message_parse_error(msg, &error, &debug);
            g_free(debug);

            ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::bBusCall() Error: %s", OSAL_ClockGetElapsedTime(), error->message));

            g_error_free(error);
            if (_pMyStaticRef->_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_LVDS)
            {
                ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::bBusCall() LVDS signal loss", OSAL_ClockGetElapsedTime()));

                _pMyStaticRef->_bVideoSignalError = TRUE;
                dispvidctrl_tclVideoPlayer::vNewVideoStatus(); // to post a message to parkassist video control part
            }
            break;
        }
    default:
        break;
    }

    return TRUE;
}


//******************************************************************************

//FUNCTION:    void v4lBufferErrorCB(GstElement * element, GstPad *pad, gpointer user_data)

//!

//!

//PARAMETER:

//RETURNVALUE: None

//******************************************************************************
tVoid dispvidctrl_tclGstRvc::vBufferErrorCB(GstElement*, GstPad*, gpointer) {
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::v4lBufferErrorCB() Rear View Camera Video Signal Failure", OSAL_ClockGetElapsedTime()));

    _pMyStaticRef->_bVideoSignalError = TRUE;
    dispvidctrl_tclVideoPlayer::vNewVideoStatus(); // to post a message to parkassist video control part
}


//******************************************************************************

//FUNCTION:    void v4lBufferErrorDoneCB(GstElement * element, GstPad *pad, gpointer user_data)

//!

//!

//PARAMETER:

//RETURNVALUE: None

//******************************************************************************
tVoid dispvidctrl_tclGstRvc::vBufferErrorDoneCB(GstElement*, GstPad*, gpointer)
{
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::v4lBufferErrorDoneCB() Rear View Camera Video Signal Restored", OSAL_ClockGetElapsedTime()));

    _pMyStaticRef->_bVideoSignalError = FALSE;
    dispvidctrl_tclVideoPlayer::vNewVideoStatus(); // to post a message to parkassist video control part
}

tVoid dispvidctrl_tclGstRvc::vSetBlackScreenOnCameraDisconnection() {
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vSetBlackScreenOnCameraDisconnection()", OSAL_ClockGetElapsedTime()));

    if (_strVideoSourceName != NULL)
    {
#ifndef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
        int fd = open(_strVideoSourceName, O_RDWR);
        if(fd != -1)
        {
            struct v4l2_control ctrl;
       
            // Set black screen for camera disconnection
       
            // Y
            ctrl.id = V4L2_CID_ADV718x_FREERUN_Y;
            ctrl.value = 4;
            if (ioctl(fd, VIDIOC_S_CTRL, &ctrl) < 0) {
                ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vSet_Black_Screen_On_Camera_Disconnection() Failed to set YCbCr (Y)", OSAL_ClockGetElapsedTime()));
            }
       
            // CB
            ctrl.id = V4L2_CID_ADV718x_FREERUN_CB;
            ctrl.value = 8;
            if (ioctl(fd, VIDIOC_S_CTRL, &ctrl) < 0) {
                ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vSet_Black_Screen_On_Camera_Disconnection() Failed to set YCbCr (Cb)", OSAL_ClockGetElapsedTime()));
            }
       
            // CR
            ctrl.id = V4L2_CID_ADV718x_FREERUN_CR;
            ctrl.value = 8;
            if (ioctl(fd, VIDIOC_S_CTRL, &ctrl) < 0) {
                ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vSet_Black_Screen_On_Camera_Disconnection() Failed to set YCbCr (Cr)", OSAL_ClockGetElapsedTime()));
            }
       
            close(fd);
        }
#endif
    }
    else
    {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vSet_Black_Screen_On_Camera_Disconnection() Video device name is NULL", OSAL_ClockGetElapsedTime()));
    }
}

tVoid dispvidctrl_tclGstRvc::vHandleLVDSSignalLoss(tVoid)
{
    // Check that this is LVDS only
    if (_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_LVDS)
    {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vHandleLVDSSignalLoss() entered", OSAL_ClockGetElapsedTime()));
        // Check that we have an error
        if (_bVideoSignalError == true)
        {
           // Change visibility of camera
           vSetLayerVisibility(_u16LayerNumber, false);

           // Show a green screen
           vShowGreenScreen(EN_VIDEOPLAYER__BLACKSCREENTYPE_FULL);
        }
    }
}

#ifndef GEN3X86
tVoid dispvidctrl_tclGstRvc::vSetAce(tBool bEnable)
{
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vSetAce()", OSAL_ClockGetElapsedTime()));
   if (_strVideoSourceName != NULL)
   {
#ifndef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
      int fd = open(_strVideoSourceName, O_RDWR);
      if(fd != -1)
      {
         struct v4l2_control ctrl;
      
         // Set AceEnable value
         ctrl.id = V4L2_CID_ADV7182_ACE_ENABLE;
         ctrl.value = (tS32) bEnable;
      
         if (ioctl(fd, VIDIOC_S_CTRL, &ctrl) < 0) {
            ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vSetAce() Failed to set", OSAL_ClockGetElapsedTime()));
         }
      
         close(fd);
      }
#endif
   }
   else
   {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vSetAce() Video device name is NULL", OSAL_ClockGetElapsedTime()));
   }
}
#endif //GEN3X86

tVoid dispvidctrl_tclGstRvc::vSetCropping(tU32 u32CropUp, tU32 u32CropDown, tU32 u32CropLeft, tU32 u32CropRight) {
    if(bCroppingParametersInRange(u32CropUp, u32CropDown, u32CropLeft, u32CropRight) == true) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vSetCropping() Input Cropping Paras - Crop Up [%d], Crop Down [%d], Crop Left [%d], Crop Right [%d]", 
                       OSAL_ClockGetElapsedTime(), u32CropUp, u32CropDown, u32CropLeft, u32CropRight));
        if((u32CropUp == 0) && (u32CropDown == 0) && (u32CropLeft == 0) && (u32CropRight == 0))
        {
            _bCroppingSet = FALSE;
            _u16CropUp = 0;
            _u16CropDown = 0;
            _u16CropLeft = 0;
            _u16CropRight = 0;
        }
        else
        {
            _bCroppingSet = TRUE;
            _u16CropUp = (tU16)u32CropUp;
            _u16CropDown = (tU16)u32CropDown;
            _u16CropLeft = (tU16)u32CropLeft;
            _u16CropRight = (tU16)u32CropRight;
        }
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vSetCropping() Cropping Paras Set to Cropping Set [%d], Crop Up [%d], Crop Down [%d], Crop Left [%d], Crop Right [%d]", 
                       OSAL_ClockGetElapsedTime(), _bCroppingSet, _u16CropUp, _u16CropDown, _u16CropLeft, _u16CropRight));
    }
    else
    {
        _bCroppingSet = FALSE;
        _u16CropUp = 0;
        _u16CropDown = 0;
        _u16CropLeft = 0;
        _u16CropRight = 0;
    }
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vSetCropping() Cropping Paras Set to Cropping Set [%d], Crop Up [%d], Crop Down [%d], Crop Left [%d], Crop Right [%d]", 
        OSAL_ClockGetElapsedTime(), _bCroppingSet, _u16CropUp, _u16CropDown, _u16CropLeft, _u16CropRight));
}

tBool dispvidctrl_tclGstRvc::bCroppingParametersInRange(tU32 u32CropUp, tU32 u32CropDown, tU32 u32CropLeft, tU32 u32CropRight) {
    tBool res = TRUE;
    if(u32CropUp >= 100) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::bCroppingParametersInRange() Crop Up [%d] - ERROR - OUT OF RANGE", OSAL_ClockGetElapsedTime(), u32CropUp));
        res = FALSE;
    }
    if(u32CropDown >= 100) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::bCroppingParametersInRange() Crop Down [%d] - ERROR - OUT OF RANGE", OSAL_ClockGetElapsedTime(), u32CropDown));
        res = FALSE;
    }
    if(u32CropLeft >= 100) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::bCroppingParametersInRange() Crop Left [%d] - ERROR - OUT OF RANGE", OSAL_ClockGetElapsedTime(), u32CropLeft));
        res = FALSE;
    }
    if(u32CropRight >= 100) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::bCroppingParametersInRange() Crop Right [%d] - ERROR - OUT OF RANGE", OSAL_ClockGetElapsedTime(), u32CropRight));
        res = FALSE;
    }
    if(u32CropUp + u32CropDown >= 100) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::bCroppingParametersInRange() Crop Up plus Crop Down [%d] - ERROR - OUT OF RANGE", OSAL_ClockGetElapsedTime(), u32CropUp + u32CropDown));
        res = FALSE;
    }
    if(u32CropLeft + u32CropRight >= 100) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::bCroppingParametersInRange() Crop Left plus Crop Right [%d] - ERROR - OUT OF RANGE", OSAL_ClockGetElapsedTime(), u32CropLeft + u32CropRight));
        res = FALSE;
    }
    return res;
}

tVoid dispvidctrl_tclGstRvc::vSetCameraImageMirror(tU32 u32CameraImageMirror)
{
    _u8CameraImageMirror = (tU8)u32CameraImageMirror;
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vSetCameraImageMirror() CameraImageMirror set to [%d]", OSAL_ClockGetElapsedTime(), _u8CameraImageMirror));
    vSetMirroring();
}

tVoid dispvidctrl_tclGstRvc::vSetLayerVisibility(tU16 u16LayerId, tBool bVisible)
{
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vSetLayerVisibility() LayerManagerControl Set Layer [%d] Visible [%d]", OSAL_ClockGetElapsedTime(), u16LayerId, bVisible));
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif

    //Set the surface visibility
    bool lRc = (ILM_SUCCESS == ilm_layerSetVisibility((unsigned int)u16LayerId, bVisible));

    // Commit changes to ILM
    lRc = lRc && (ILM_SUCCESS == ilm_commitChanges());
    if(lRc != TRUE) {
        ETG_TRACE_FATAL (("[%d ms] dispvidctrl_tclGstRvc::vSetLayerVisibility() - LayerManagerControl Commit Failed [%d]", OSAL_ClockGetElapsedTime(), lRc));
    }
}

tVoid dispvidctrl_tclGstRvc::vSetLayerOpacity(tU16 u16LayerId, tBool bOpacity)
{
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vSetLayerOpacity() LayerId [%d] Opacity [%d]", OSAL_ClockGetElapsedTime(), u16LayerId, bOpacity));
}

tVoid dispvidctrl_tclGstRvc::vSetLayerSourceRectangle(tU16 u16LayerId, tU32 u32X, tU32 u32Y, tU32 u32Width, tU32 u32Height, tBool bCommit)
{
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vSetLayerSourceRectangle() LayerId [%d], X [%d], Y [%d], Width [%d], Height [%d]", 
                       OSAL_ClockGetElapsedTime(), u16LayerId, u32X, u32Y, u32Width, u32Height));
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif

    _bLayerSourceRectangleSet = TRUE;
    _u16LayerSourceRectangle_X        = (tU16) u32X;
    _u16LayerSourceRectangle_Y        = (tU16) u32Y;
    _u16LayerSourceRectangle_Width    = (tU16) u32Width;
    _u16LayerSourceRectangle_Height   = (tU16) u32Height;

    bool lRc = (ILM_SUCCESS == ilm_layerSetSourceRectangle((int) u16LayerId, (int) _u16LayerSourceRectangle_X, (int) _u16LayerSourceRectangle_Y, (int) _u16LayerSourceRectangle_Width, (int) _u16LayerSourceRectangle_Height));

    if ( bCommit ) {
       // Commit changes to ILM
       lRc = lRc && (ILM_SUCCESS == ilm_commitChanges());
       if(lRc != TRUE) {
           ETG_TRACE_FATAL (("[%d ms] dispvidctrl_tclGstRvc::vSetLayerSourceRectangle() - LayerManagerControl Commit Failed [%d]", OSAL_ClockGetElapsedTime(), lRc));
       }
    }
}

tVoid dispvidctrl_tclGstRvc::vSetLayerDestinationRectangle(tU16 u16LayerId, tU32 u32X, tU32 u32Y, tU32 u32Width, tU32 u32Height, tBool bCommit) {
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vSetLayerDestinationRectangle() SurfaceId [%d], X [%d], Y [%d], Width [%d], Height [%d]", 
                       OSAL_ClockGetElapsedTime(), u16LayerId, u32X, u32Y, u32Width, u32Height));
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif

    _u16LayerDestinationRectangle_X = (tU16) u32X;
    _u16LayerDestinationRectangle_Y = (tU16) u32Y;
    _u16LayerDestinationRectangle_Width = (tU16) u32Width;
    _u16LayerDestinationRectangle_Height = (tU16) u32Height;

    bool lRc = (ILM_SUCCESS == ilm_layerSetDestinationRectangle((int) u16LayerId, (int) _u16LayerDestinationRectangle_X, (int) _u16LayerDestinationRectangle_Y, (int) _u16LayerDestinationRectangle_Width, (int) _u16LayerDestinationRectangle_Height));

    if ( bCommit ) {
       // Commit changes to ILM
       lRc = lRc && (ILM_SUCCESS == ilm_commitChanges());
       if(lRc != TRUE) {
           ETG_TRACE_FATAL (("[%d ms] dispvidctrl_tclGstRvc::vSetLayerDestinationRectangle() - LayerManagerControl Commit Failed [%d]", OSAL_ClockGetElapsedTime(), lRc));
       }
    }
}

tVoid dispvidctrl_tclGstRvc::vSetSurfaceVisibility(tU16 u16SurfaceId, tBool bVisible)
{
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vSetSurfaceVisibility() LayerManagerControl Set Surface [%d] Visible [%d]", OSAL_ClockGetElapsedTime(), u16SurfaceId, bVisible));
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif

    if ( _bRvcActive == TRUE ) {
        //Set the surface visibility
        bool lRc = (ILM_SUCCESS == ilm_surfaceSetVisibility((unsigned int)u16SurfaceId, bVisible));

        // Commit changes to ILM
        lRc = lRc && (ILM_SUCCESS == ilm_commitChanges());
        if(lRc != TRUE) {
            ETG_TRACE_FATAL (("[%d ms] dispvidctrl_tclGstRvc::vSetSurfaceVisibility() - LayerManagerControl Commit Failed [%d]", OSAL_ClockGetElapsedTime(), lRc));
        }
    }
}

tVoid dispvidctrl_tclGstRvc::vSetSurfaceOpacity(tU16 u16SurfaceId, tBool bOpacity)
{
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vSetSurfaceOpacity() SurfaceId [%d] Opacity [%d]", OSAL_ClockGetElapsedTime(), u16SurfaceId, bOpacity));
}

tVoid dispvidctrl_tclGstRvc::vSetSurfaceSourceRectangle(tU16 u16SurfaceId, tU32 u32X, tU32 u32Y, tU32 u32Width, tU32 u32Height, tBool bCommit)
{
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vSetSurfaceSourceRectangle() SurfaceId [%d], X [%d], Y [%d], Width [%d], Height [%d]", 
                       OSAL_ClockGetElapsedTime(), u16SurfaceId, u32X, u32Y, u32Width, u32Height));
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif
    if ( _bRvcActive == TRUE ) {
        bool lRc = (ILM_SUCCESS == ilm_surfaceSetSourceRectangle((int) u16SurfaceId, (int) u32X, (int) u32Y, (int) u32Width, (int) u32Height));

        if ( bCommit ) {
           // Commit changes to ILM
           lRc = lRc && (ILM_SUCCESS == ilm_commitChanges());
           if(lRc != TRUE) {
               ETG_TRACE_FATAL (("[%d ms] dispvidctrl_tclGstRvc::vSetSurfaceSourceRectangle() - LayerManagerControl Commit Failed [%d]", OSAL_ClockGetElapsedTime(), lRc));
           }
       }
    }
}

tVoid dispvidctrl_tclGstRvc::vSetSurfaceDestinationRectangle(tU16 u16SurfaceId, tU32 u32X, tU32 u32Y, tU32 u32Width, tU32 u32Height, tBool bCommit) {
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vSetSurfaceDestinationRectangle() SurfaceId [%d], X [%d], Y [%d], Width [%d], Height [%d]", 
                       OSAL_ClockGetElapsedTime(), u16SurfaceId, u32X, u32Y, u32Width, u32Height));
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif

    if ( _bRvcActive == TRUE ) {
        bool lRc = (ILM_SUCCESS == ilm_surfaceSetDestinationRectangle((int) u16SurfaceId, (int) u32X, (int) u32Y, (int) u32Width, (int) u32Height));

        if ( bCommit ) {
           // Commit changes to ILM
           lRc = lRc && (ILM_SUCCESS == ilm_commitChanges());
           if(lRc != TRUE) {
               ETG_TRACE_FATAL (("[%d ms] dispvidctrl_tclGstRvc::vSetSurfaceDestinationRectangle() - LayerManagerControl Commit Failed [%d]", OSAL_ClockGetElapsedTime(), lRc));
           }
       }
    }
}

tVoid dispvidctrl_tclGstRvc::vCommitChanges(tVoid) {
   // Commit changes to ILM
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif
   bool lRc = (ILM_SUCCESS == ilm_commitChanges());
   if(lRc != TRUE) {
       ETG_TRACE_FATAL (("[%d ms] dispvidctrl_tclGstRvc::vCommitChanges() - LayerManagerControl Commit Failed [%d]", OSAL_ClockGetElapsedTime(), lRc));
   }
}

tVoid dispvidctrl_tclGstRvc::vReConfigGuidelineParams(tU8 u8VideoType, tU8 u8VideoParamIndex)
{
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vReConfigGuidelineParams() u8VideoType [%d]", OSAL_ClockGetElapsedTime(), ETG_CENUM(tenControl_VideoResize, u8VideoType)));
    if (_poControlGraphic) _poControlGraphic->vOnSetupGuidelineParams(u8VideoType, u8VideoParamIndex);
}

tVoid dispvidctrl_tclGstRvc::vSetWarningTextLanguage(tU8 u8Language) {
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vSetWarningTextLanguage() Language [%d]", OSAL_ClockGetElapsedTime(), u8Language));
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif

    if (_poControlGraphic) _poControlGraphic->vSetWarningTextLanguage(u8Language);
}

tVoid dispvidctrl_tclGstRvc::vSetGuideLineVisibility(tU32 u32Visibility) {

    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vSetGuideLineVisibility() Guideline Visibility [%d]", OSAL_ClockGetElapsedTime(), u32Visibility));
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif
    if (_poControlGraphic) _poControlGraphic->vSetGuideLineVisibility(u32Visibility);
}

tVoid dispvidctrl_tclGstRvc::vSetWarningTextVisibility(tU32 u32Visibility) {
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::vSetWarningTextVisibility() Warning Text Visibility [%d]", OSAL_ClockGetElapsedTime(), u32Visibility));
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif
    if (_poControlGraphic) _poControlGraphic->vSetWarningTextVisibility(u32Visibility);
}

tVoid dispvidctrl_tclGstRvc::vSetSteeringWheelAngle(tS32 s32SteeringWheelAngle) {
    ETG_TRACE_USR4( ( "[%d ms] dispvidctrl_tclGstRvc::vSetSteeringWheelAngle() Steering Wheel Angle [%d]", OSAL_ClockGetElapsedTime(), s32SteeringWheelAngle));
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif
    if (_poControlGraphic) _poControlGraphic->vSetSteeringWheelAngle(s32SteeringWheelAngle);
}

tVoid dispvidctrl_tclGstRvc::vSetRearWheelAngle(tS32 s32RearWheelAngle) {
    ETG_TRACE_USR4( ( "[%d ms] dispvidctrl_tclGstRvc::vSetRearWheelAngle() Rear Wheel Angle [%d]", OSAL_ClockGetElapsedTime(), s32RearWheelAngle));
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif
    if (_poControlGraphic) _poControlGraphic->vSetRearWheelAngle(s32RearWheelAngle);
}

tVoid dispvidctrl_tclGstRvc::vSetHandoverReady() {
    ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vSetHandoverReady() setting _bHandoverReady = TRUE", OSAL_ClockGetElapsedTime()));
    _bHandoverReady = TRUE;
}

tVoid dispvidctrl_tclGstRvc::vSetHmiAvailable()
{
    ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstRvc::vSetHmiAvailable() so Remove Early Graphics", OSAL_ClockGetElapsedTime()));
    // should be called when HMI is available and the Early ONly Graphics are not needed anymore
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif
    if (_poControlGraphic) _poControlGraphic->vRemoveEarlyOnlyGraphics();    
}

tVoid dispvidctrl_tclGstRvc::vTraceDebugInformation(tVoid) {

    ETG_TRACE_FATAL((" CA-RVC   :: RVC Configuration"));
    ETG_TRACE_FATAL((" CA-RVC   :: _u16LayerNumber = ................ %d", _u16LayerNumber));
    ETG_TRACE_FATAL((" CA-RVC   :: _u16SurfaceNumber = .............. %d", _u16SurfaceNumber));
    ETG_TRACE_FATAL((" CA-RVC   :: _u16GraphicsLayerNumber = ........ %d", _u16GraphicsLayerNumber));
    ETG_TRACE_FATAL((" CA-RVC   :: _u16GuidelineSurfaceNumber = ..... %d", _u16GuidelineSurfaceNumber));
    ETG_TRACE_FATAL((" CA-RVC   :: _u16ScreenWidth = ................ %d", _u16ScreenWidth));
    ETG_TRACE_FATAL((" CA-RVC   :: _u16ScreenHeight = ............... %d", _u16ScreenHeight));
    ETG_TRACE_FATAL((" CA-RVC   :: _u16PixFmt = ..................... %d", _u16PixFmt));
    ETG_TRACE_FATAL((" CA-RVC   :: _u16CropUp = ..................... %d", _u16CropUp));
    ETG_TRACE_FATAL((" CA-RVC   :: _u16CropDown = ................... %d", _u16CropDown));
    ETG_TRACE_FATAL((" CA-RVC   :: _u16CropLeft = ................... %d", _u16CropLeft));
    ETG_TRACE_FATAL((" CA-RVC   :: _u16CropRight = .................. %d", _u16CropRight));
    ETG_TRACE_FATAL((" CA-RVC   :: _u16LayerSourceRectangle_X = ..... %d", _u16LayerSourceRectangle_X));
    ETG_TRACE_FATAL((" CA-RVC   :: _u16LayerSourceRectangle_Y = ..... %d", _u16LayerSourceRectangle_Y));
    ETG_TRACE_FATAL((" CA-RVC   :: _u16LayerSourceRectangle_Width = . %d", _u16LayerSourceRectangle_Width));
    ETG_TRACE_FATAL((" CA-RVC   :: _u16LayerSourceRectangle_Height =  %d", _u16LayerSourceRectangle_Height));
    ETG_TRACE_FATAL((" CA-RVC   :: _u16LayerDestinationRectangle_X =  %d", _u16LayerDestinationRectangle_X));
    ETG_TRACE_FATAL((" CA-RVC   :: _u16LayerDestinationRectangle_Y =  %d", _u16LayerDestinationRectangle_Y));
    ETG_TRACE_FATAL((" CA-RVC   :: _u16LayerDestRectangle_Width    =  %d", _u16LayerDestinationRectangle_Width));
    ETG_TRACE_FATAL((" CA-RVC   :: _u16LayerDestRectangle_Height   =  %d", _u16LayerDestinationRectangle_Height));
    ETG_TRACE_FATAL((" CA-RVC   :: _strVideoSourceName = ............ %s", _strVideoSourceName));
    ETG_TRACE_FATAL((" CA-RVC   :: _strVideoOutName = ............... %s", _strVideoOutName));
    ETG_TRACE_FATAL((" CA-RVC   :: _bRvcActive = .................... %d", _bRvcActive));
    ETG_TRACE_FATAL((" CA-RVC   :: _bRvcVisible = ................... %d", _bRvcVisible));
    ETG_TRACE_FATAL((" CA-RVC   :: _bVideoSignalError = ............. %d", _bVideoSignalError));
    ETG_TRACE_FATAL((" CA-RVC   :: _bPipelineActive = ............... %d", _bPipelineActive));
    ETG_TRACE_FATAL((" CA-RVC   :: _bHandoverReady = ................ %d", _bHandoverReady));
    ETG_TRACE_FATAL((" CA-RVC   :: _bCroppingSet = .................. %d", _bCroppingSet));
    ETG_TRACE_FATAL((" CA-RVC   :: _bLayerSourceRectangleSet = ...... %d", _bLayerSourceRectangleSet));
    ETG_TRACE_FATAL((" CA-RVC   :: _bVideoSourceFound = ............. %d", _bVideoSourceFound));
    ETG_TRACE_FATAL((" CA-RVC   :: _bVideoOutFound = ................ %d", _bVideoOutFound));
    ETG_TRACE_FATAL((" CA-RVC   :: _bMediaInSourceFound = ........... %d", _bMediaInSourceFound));
    ETG_TRACE_FATAL((" CA-RVC   :: _u8CameraConnectionType = ........ %d", _u8CameraConnectionType));
    ETG_TRACE_FATAL((" CA-RVC   :: _u8AnalogInputChannel = .......... %d", _u8AnalogInputChannel));
    ETG_TRACE_FATAL((" CA-RVC   :: _u8DigitalInputChannel = ......... %d", _u8DigitalInputChannel));
    ETG_TRACE_FATAL((" CA-RVC   :: _u8DigitalMediaInInputChannel = .. %d", _u8DigitalMediaInInputChannel));


    t_ilm_bool rvc_layer_visible = FALSE;
    t_ilm_bool graphics_layer_visible = FALSE;
    t_ilm_bool rvc_surface_visible = FALSE;
    t_ilm_bool graphics_surface_visible = FALSE;
    bool lRc = (ILM_SUCCESS == ilm_layerGetVisibility((unsigned int)_u16LayerNumber,&rvc_layer_visible));
    lRc = lRc && (ILM_SUCCESS == ilm_layerGetVisibility((unsigned int)_u16GraphicsLayerNumber,&graphics_layer_visible));
    lRc = lRc && (ILM_SUCCESS == ilm_surfaceGetVisibility((unsigned int)_u16GuidelineSurfaceNumber,&graphics_surface_visible));

    ETG_TRACE_FATAL((" CA-RVC   :: RVC Layer [%d] visibility = ....... %d", _u16LayerNumber, rvc_layer_visible));
    ETG_TRACE_FATAL((" CA-RVC   :: Graphics Layer [%d] visibility = .. %d", _u16GraphicsLayerNumber, graphics_layer_visible));

    if(lRc == TRUE) {
       if (ILM_SUCCESS == ilm_surfaceGetVisibility((unsigned int)_u16SurfaceNumber,&rvc_surface_visible)) {
          ETG_TRACE_FATAL((" CA-RVC   :: RVC Surface [%d] visibility = ..... %d", _u16SurfaceNumber, rvc_surface_visible));
       } else {
          ETG_TRACE_FATAL((" CA-RVC   :: RVC Surface [%d] does not exist at this time", _u16SurfaceNumber));
       }
       ETG_TRACE_FATAL((" CA-RVC   :: Graphics Surface [%d] visibility =  %d", _u16GuidelineSurfaceNumber, graphics_surface_visible));
    }

    DISPVIDCTRL_NULL_POINTER_CHECK(_pMyStaticRef);
    DISPVIDCTRL_NULL_POINTER_CHECK(_pMyStaticRef->_poControlGraphic);
    _pMyStaticRef->_poControlGraphic->vTraceDebugInformation();
}

tVoid dispvidctrl_tclGstRvc::vIndVehicleConfigChanged_DynGuideline(tVoid)
{
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif
   if (_poControlGraphic)
   {
      _poControlGraphic->vIndVehicleConfigChanged_DynGuideline();
   }
}

vd_rvc_tstVehicleParams* dispvidctrl_tclGstRvc::vGetVehicleConfigPtr_DynGuideline(tVoid)
{
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return NULL;
#endif
   if (_poControlGraphic)
   {
      return _poControlGraphic->vGetVehicleConfigPtr_DynGuideline();
   }
   else
   {
      return NULL;
   }
}

tVoid dispvidctrl_tclGstRvc::vSetVehicleConfig_DynGuideline(tVoid)
{
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif
   if (_poControlGraphic)
   {
      _poControlGraphic->vSetVehicleConfig_DynGuideline();
   }
}

tVoid dispvidctrl_tclGstRvc::vSetTemporaryCalibGuidelnStyle(tBool bActive)
{
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return;
#endif
   if (_poControlGraphic)
   {
      _poControlGraphic->vSetTemporaryCalibGuidelnStyle(bActive);
   }
}

tVoid dispvidctrl_tclGstRvc::vSetInitialblackscreen(tU8  u8camConfigType)
{
    ETG_TRACE_FATAL((" CA-RVC   :: vSetInitialblackscreen = ................ %d", u8camConfigType));
   if (_poControlGraphic)
   {
       _poControlGraphic->vSetInitialblackscreenMask(u8camConfigType);
   }
}

#ifndef GEN3X86
tBool dispvidctrl_tclGstRvc::bFindCameraVideoSource(tVoid)
{
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    return FALSE;
#endif
   // It is necessary to check the device naming as it cannot be guaranteed by platform
   FILE* fp;
   
   _bVideoSourceFound = _bMediaInSourceFound = _bVideoOutFound = false;
   for (tU32 u32Idx=0; u32Idx<GSTRVC_NUM_POSSIBLE_VIDEO_SOURCES; ++u32Idx)
   {
      tU32 u32TimeStamp = OSAL_ClockGetElapsedTime();
      
      if ( _bVideoSourceFound && _bMediaInSourceFound && _bVideoOutFound )
      {
         break;
      }

      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::bFindCameraVideoSource() - checking video source %s ...", u32TimeStamp, aStrVideoSources[u32Idx]));
      
      fp = fopen(aStrVideoSources[u32Idx], "r");
      if(fp != 0)
      {
         char device [30];
         fgets(device, 30, fp);
         fclose(fp);
   
         ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::bFindCameraVideoSource() device = %s", u32TimeStamp, device));
   
         if (_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_CVBS)
         {
            if ( !_bVideoSourceFound )
            {
               if (0 == strcmp("ipu1_csi0 capture\n", device))
               {
                  ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::bFindCameraVideoSource() Analogue Camera device is %s", u32TimeStamp, aStrVideoNames[u32Idx]));
                  _strVideoSourceName = aStrVideoNames[u32Idx];
                  _bVideoSourceFound = TRUE;
               }
            }
         }
         else if (_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_LVDS)
         {
            if ( !_bVideoSourceFound )
            {
               if (0 == strcmp("ipu1_csi1 capture\n", device))
               {
                  ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::bFindCameraVideoSource() Digital Camera device is %s", u32TimeStamp, aStrVideoNames[u32Idx]));
                  _strVideoSourceName = aStrVideoNames[u32Idx];
                  _bVideoSourceFound = TRUE;
               }
            }
            
            if ( !_bMediaInSourceFound )
            {
               if (0 == strncmp("ds90ux940", device, strlen("ds90ux940")))
               {
                  ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::bFindCameraVideoSource() Digital Media In device is %s", u32TimeStamp, aStrVideoNames[u32Idx]));
                  _strMediaInSourceName = aStrVideoNames[u32Idx];
                  _bMediaInSourceFound = TRUE;
               }
            }
         }
         else { /* do nothing */ }
         
         // v4l_csc device name check
         if ( !_bVideoOutFound )
         {
            if (0 == strcmp("mx6-m2m\n", device))
            {
               ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::bFindCameraVideoSource() Digital Output device is %s", u32TimeStamp, aStrVideoNames[u32Idx]));
               _strVideoOutName = aStrVideoNames[u32Idx];
               _bVideoOutFound = TRUE;
            }
         }
      }
      else
      {
         ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstRvc::bFindCameraVideoSource() - ERROR - FAILED to open %s", u32TimeStamp, aStrVideoSources[u32Idx]));
      }
   }

   // Find the right input channel for Analog and Digital Video Inputs
   if ( _bVideoSourceFound == TRUE )
   {
      int fd = open(_strVideoSourceName, O_RDONLY);
      if(fd != -1)
      {
         struct v4l2_input vin;
         memset(&vin, 0, sizeof(v4l2_input));
         
         for (tU8 u8Index = 0; u8Index <= 5; u8Index++ )
         {
            vin.index = u8Index;

            if (ioctl(fd, VIDIOC_ENUMINPUT, &vin) < 0) {
               ETG_TRACE_USR2( ( "[%d ms] dispvidctrl_tclGstRvc::bFindCameraVideoSource() Failed to read VIDIOC_ENUMINPUT Index %u", OSAL_ClockGetElapsedTime(), vin.index));
            }
            else {
               int resAnalog  = strcmp(strAnalogInputName, reinterpret_cast<char*>(vin.name));
               int resDigital = strcmp(strDigitalInputName, reinterpret_cast<char*>(vin.name));
               int resMediaIn = strcmp(strDigitalMediaInInputName, reinterpret_cast<char*>(vin.name));

               if (resAnalog == 0) {
                  ETG_TRACE_FATAL(( "[%d ms] dispvidctrl_tclGstRvc::bFindCameraVideoSource() RVC Analog Input Channel Found \"%u\" Name \"%s\" ", OSAL_ClockGetElapsedTime(), vin.index, vin.name));
                  _u8AnalogInputChannel = vin.index;
               } 
               else if (resDigital == 0) {
                  ETG_TRACE_FATAL(( "[%d ms] dispvidctrl_tclGstRvc::bFindCameraVideoSource() RVC Digital Input Channel Found \"%u\" Name \"%s\" ", OSAL_ClockGetElapsedTime(), vin.index, vin.name));
                  _u8DigitalInputChannel = vin.index;
               }
               else if (resMediaIn == 0) {
                  ETG_TRACE_FATAL(( "[%d ms] dispvidctrl_tclGstRvc::bFindCameraVideoSource() Media In Digital Input Channel Found \"%u\" Name \"%s\" ", OSAL_ClockGetElapsedTime(), vin.index, vin.name));
                  _u8DigitalMediaInInputChannel = vin.index;
               }
               else {
                  // do somehting ?
               }
            }
         }
         close(fd);
      }
   }

   return _bVideoSourceFound;
}
#endif //GEN3X86

