/******************************************************************************/
/**
* \file    vd_rvc_tclGraphics_DynGuideline_Plotter.cpp
* \ingroup
*
* \brief
*
* \remark  Copyright : (c) 2012 Robert Bosch GmbH, Hildesheim
* \remark  Author    :
* \remark  Scope     :
*
* \todo
*/
/******************************************************************************/


/*******************************************************************************
                        Includes
*******************************************************************************/

#include "vd_rvc_tclGraphics_DynGuideline_Types.h"
#include "vd_rvc_tclGraphics_DynGuideline_Plotter.h"
#include "vd_rvc_tclGraphics_DynGuideline_Util.h"
#include <cmath>


/*******************************************************************************
                  Specific defines for this component
*******************************************************************************/


#define INSIDE_BB 0x00
#define X_SMALL   0x01
#define X_BIG     0x02
#define Y_SMALL   0x04
#define Y_BIG     0x08


/*******************************************************************************
    Class implementation
*******************************************************************************/

vd_rvc_tclGraphics_DynGuideline_Plotter::vd_rvc_tclGraphics_DynGuideline_Plotter(tVoid)
  : m_pu32OutBuf(NULL), m_s32NumBufXpix(0), m_s32NumBufYpix(0),
    m_u32Color(LINEPLOTTER_COLOR_RGB_AS_U32(0xFF,0x80,0xFF)),  // piggy-pink
    m_bAntiAliasOn(false),
    m_bEraseModeOn(false),
    m_s32BbXmin(0), m_s32BbXmax(0), m_s32BbYmin(0), m_s32BbYmax(0),
    m_bIsBoundingBoxSet(false)
{
}


vd_rvc_tclGraphics_DynGuideline_Plotter::~vd_rvc_tclGraphics_DynGuideline_Plotter()
{
   m_pu32OutBuf = NULL;
}


tVoid vd_rvc_tclGraphics_DynGuideline_Plotter::vSetOutBuffer(const vd_rvc_tstOutputBufferParams *pstBufParams)
{
   m_pu32OutBuf    = (tU32*)pstBufParams->pvBuf;
   m_s32NumBufXpix = (tS32)pstBufParams->u32NumBufXpix;
   m_s32NumBufYpix = (tS32)pstBufParams->u32NumBufYpix;
}


tVoid vd_rvc_tclGraphics_DynGuideline_Plotter::vClearOutBufferContent(const vd_rvc_tstImageAreaParams *pstImgParams)
{
   if (NULL == m_pu32OutBuf)
   {
      return;
   }

   tU32* pBuf         = m_pu32OutBuf + pstImgParams->u32ImgYoffs*(tU32)m_s32NumBufXpix + pstImgParams->u32ImgXoffs;
   tU32 u32ImgRowSize = LINEPLOTTER_NUM_BYTES_PER_PIX * pstImgParams->u32NumImgXpix;
   for (tU32 u32YIdx=0; u32YIdx<pstImgParams->u32NumImgYpix; u32YIdx++, pBuf+=m_s32NumBufXpix)
   {
      (tVoid)OSAL_pvMemorySet(pBuf, LINEPLOTTER_FULLY_TRANSPARENT, u32ImgRowSize);
   }
}


tVoid vd_rvc_tclGraphics_DynGuideline_Plotter::vSetBoundingBox(tS32 s32BbXmin, tS32 s32BbXmax, tS32 s32BbYmin, tS32 s32BbYmax)
{
   m_s32BbXmin = s32BbXmin;
   m_s32BbXmax = s32BbXmax;
   m_s32BbYmin = s32BbYmin;
   m_s32BbYmax = s32BbYmax;
   m_bIsBoundingBoxSet = true;
}


tBool vd_rvc_tclGraphics_DynGuideline_Plotter::bClipToBoundingBox(vd_rvc_t2IntVect * poPsta, vd_rvc_t2IntVect * poPend) const
{
   // implements the Cohen-Sutherland-algorithm to perform line clipping to the viewport area, if required
   tFloat fDx = poPsta->x() - poPend->x();
   tFloat fDy = poPsta->y() - poPend->y();
   
   tU32 u32PosFlags1 = u32GetPosFlags(poPsta);	// position code of starting point
   tU32 u32PosFlags2 = u32GetPosFlags(poPend);	// position code of end point

   tU32 u32Cnt = 0;
   while(u32PosFlags1 | u32PosFlags2)  // loop until both ends of the line have been clipped correctly
   {
      if (++u32Cnt > 2)  // the algorithm should need at most two loops; however, there *are* "degenerate" cases where one of the points continuously
      {                  // "jumps" diagonally one pixel hence and forth across one of the corners of the bounding box (=>endless loop). This 'if' catches these cases.
         return false;
      }
      
      if (u32PosFlags1 & u32PosFlags2)  // both starting and end point have some coordinate(s) which fall(s) out of the bounding box on the same side => so does the whole line
      {
        return false;
      }
      if (u32PosFlags1)
      {
         u32PosFlags1 = u32ShiftToBoundingBoxBorders(poPsta, u32PosFlags1, fDx,fDy);
      }
        
      if (u32PosFlags1 & u32PosFlags2)  // both starting and end point have some coordinate(s) which fall(s) out of the bounding box on the same side => so does the whole line
      {
         return false;
      }
      if (u32PosFlags2)
      {
         u32PosFlags2 = u32ShiftToBoundingBoxBorders(poPend, u32PosFlags2, fDx,fDy);
      }
   }
   return true;
}


tU32 vd_rvc_tclGraphics_DynGuideline_Plotter::u32GetPosFlags(const vd_rvc_t2IntVect * poP) const
{
   tU32 u32PosFlags = INSIDE_BB;
   
   if      (poP->x() < m_s32BbXmin)
   {
      u32PosFlags |= X_SMALL;
   }
   else if (poP->x() > m_s32BbXmax)
   {
      u32PosFlags |= X_BIG;
   }
      
   if      (poP->y() < m_s32BbYmin)
   {
      u32PosFlags |= Y_SMALL;
   }
   else if (poP->y() > m_s32BbYmax)
   {
      u32PosFlags |= Y_BIG;
   }

   return u32PosFlags;
}


tU32 vd_rvc_tclGraphics_DynGuideline_Plotter::u32ShiftToBoundingBoxBorders(vd_rvc_t2IntVect * poP, tU32 u32PosFlags,  tFloat fDx, tFloat fDy) const
{
   if      (u32PosFlags & (tU32)Y_SMALL)
   {
      poP->vSet( poP->x() + (m_s32BbYmin-poP->y()) * fDx / fDy,  m_s32BbYmin );
   }
   else if (u32PosFlags & (tU32)Y_BIG  )
   {
      poP->vSet( poP->x() + (m_s32BbYmax-poP->y()) * fDx / fDy,  m_s32BbYmax );
   }
   
   if (u32PosFlags & (tU32)X_SMALL)
   {
      poP->vSet( m_s32BbXmin,  poP->y() + (m_s32BbXmin-poP->x()) * fDy / fDx );
   }
   else if (u32PosFlags & (tU32)X_BIG)
   {
      poP->vSet( m_s32BbXmax,  poP->y() + (m_s32BbXmax-poP->x()) * fDy / fDx );
   }
   
   return( u32GetPosFlags(poP) );
}


tVoid vd_rvc_tclGraphics_DynGuideline_Plotter::vPlotLine(const vd_rvc_t2IntVect * poPsta, const vd_rvc_t2IntVect * poPend,  tFloat fLineWidth)
{
   if ((NULL == m_pu32OutBuf) || (0.0f >= fLineWidth))
   {
      return;
   }
   
   vd_rvc_t2IntVect oPsta = *poPsta, oPend = *poPend;
   if ( (false==m_bIsBoundingBoxSet)                  ||  // if no bounding box was set before, better don't plot anything at all
        (false==bClipToBoundingBox( & oPsta, & oPend)) )  // and if no part of the line falls inside the bounding box, there's nothing to do
   {
      return;
   }
   
   // select the applicable specialized Bresenham implementation
   if (1.0f == fLineWidth)
   {
      if (m_bAntiAliasOn)
      {
         vPlotLine_1_AA(oPsta.x(),oPsta.y(),  oPend.x(),oPend.y());
      }
      else
      {
         vPlotLine_1   (oPsta.x(),oPsta.y(),  oPend.x(),oPend.y());
      }
   }
   else
   {
      if (m_bAntiAliasOn)
      {
         vPlotLine_N_AA(oPsta.x(),oPsta.y(),  oPend.x(),oPend.y(),  fLineWidth);
      }
      else
      {
         vPlotLine_N   (oPsta.x(),oPsta.y(),  oPend.x(),oPend.y(),  fLineWidth);
      }
   }
}


tVoid vd_rvc_tclGraphics_DynGuideline_Plotter::vPlotLine_1(tS32 s32Xsta, tS32 s32Ysta,  tS32 s32Xend, tS32 s32Yend)
{
   tS32 s32AbsDx =  abs(s32Xend-s32Xsta),  s32SgnDx = (s32Xsta<=s32Xend) ? 1 : -1;
   tS32 s32AbsDy = -abs(s32Yend-s32Ysta),  s32SgnDy = (s32Ysta<=s32Yend) ? 1 : -1;
   tS32 s32ErrXY = s32AbsDx + s32AbsDy;

   tS32 s32Xcur = s32Xsta,  s32Ycur = s32Ysta;
   for (;;)  // pixel loop
   {
      // set line pixel at current position
      if (! m_bEraseModeOn) {
         vSetPixel(s32Xcur,s32Ycur);
      } else {
         vClrPixel(s32Xcur,s32Ycur);
      }
      if ( (s32Xcur == s32Xend) && (s32Ycur == s32Yend) )
      {
         break;
      }
      
      // determine position of next line pixel
      tS32 s32Err2 = 2*s32ErrXY;
      if (s32Err2 >= s32AbsDy)  // x step needed?
      {
         s32ErrXY += s32AbsDy;
         s32Xcur  += s32SgnDx;
      }      
      if (s32Err2 <= s32AbsDx)  // y step needed?
      {
         s32ErrXY += s32AbsDx;
         s32Ycur  += s32SgnDy;
      }
   }
}


tVoid vd_rvc_tclGraphics_DynGuideline_Plotter::vPlotLine_1_AA(tS32 s32Xsta, tS32 s32Ysta,  tS32 s32Xend, tS32 s32Yend)
{
   tS32 s32AbsDx = abs(s32Xend-s32Xsta),  s32SgnDx = (s32Xsta<=s32Xend) ? 1 : -1;
   tS32 s32AbsDy = abs(s32Yend-s32Ysta),  s32SgnDy = (s32Ysta<=s32Yend) ? 1 : -1;
   
   tS32   s32ErrXY = s32AbsDx - s32AbsDy;    // error value e_xy
   tFloat fErrAA   = (s32AbsDx+s32AbsDy == 0) ? 1.0 : (tFloat)sqrt( (tDouble)s32AbsDx*s32AbsDx + (tDouble)s32AbsDy*s32AbsDy );  // antialiasing error value
   tFloat fD2TFact = LINEPLOTTER_ALPHA_MAXVAL / fErrAA;  // distance-from-line-to-transparency factor; pre-computed here to save divisions inside the loop
   
   tS32   s32Xcur = s32Xsta,  s32Ycur = s32Ysta;
   tFloat fTransp;
   for (;;)  // pixel loop
   {
      // set 1st partly transparent line pixel at current position
      if (! m_bEraseModeOn) {
         fTransp = abs(s32ErrXY+s32AbsDy-s32AbsDx) * fD2TFact;
         vSetPixel(s32Xcur, s32Ycur, fTransp);
      } else {
         vClrPixel(s32Xcur, s32Ycur);
      }

      // set 2nd partly transparent line pixel next to current position (if needed) and update current position
      tS32 s32Err2 = s32ErrXY;      
      tS32 s32Xtmp = s32Xcur;
      if (2*s32Err2 >= -s32AbsDx)  // x step needed?
      {
         if (s32Xcur == s32Xend)
         {
            break;
         }
         if (s32AbsDy+s32Err2 < fErrAA)  // set an additional vertically offset AA pixel?
         {
            if (! m_bEraseModeOn) {
               fTransp = (s32AbsDy+s32Err2) * fD2TFact;
               vSetPixel(s32Xcur, s32Ycur+s32SgnDy, fTransp);
            } else {
               vClrPixel(s32Xcur, s32Ycur+s32SgnDy);
            }
         }
         s32ErrXY -= s32AbsDy;
         s32Xcur  += s32SgnDx;
      }
      if (2*s32Err2 <=  s32AbsDy)  // y step needed?
      {
         if (s32Ycur == s32Yend)
         {
            break;
         }
         if (s32AbsDx-s32Err2 < fErrAA)  // set an additional horizontally offset AA pixel?
         {
            if (! m_bEraseModeOn) {
               fTransp = (s32AbsDx-s32Err2) * fD2TFact;
               vSetPixel(s32Xtmp+s32SgnDx, s32Ycur, fTransp);
            } else {
               vClrPixel(s32Xtmp+s32SgnDx, s32Ycur);
            }
         }
         s32ErrXY += s32AbsDx;
         s32Ycur  += s32SgnDy;
      }
   }
}


tVoid vd_rvc_tclGraphics_DynGuideline_Plotter::vPlotLine_N(tS32 s32Xsta, tS32 s32Ysta,  tS32 s32Xend, tS32 s32Yend,  tFloat fWidth)
{
   if (0.5F > fWidth)
      return;  // don't plot line if it is too thin
   tS32 s32Width = (tS32)(fWidth+0.5F);  // so that the rest of the code can run on integers only
      
   tS32 s32AbsDx = abs(s32Xend-s32Xsta),  s32SgnDx = (s32Xsta<=s32Xend) ? 1 : -1;
   tS32 s32AbsDy = abs(s32Yend-s32Ysta),  s32SgnDy = (s32Ysta<=s32Yend) ? 1 : -1;
   tS32 s32ErrXY = s32AbsDx - s32AbsDy;    // error value e_xy   
   tS32 s32Wsta  = -s32Width/2,   s32Wend = s32Wsta + s32Width;  // line width control variables
   
   tS32 s32Xcur = s32Xsta,  s32Ycur = s32Ysta;
   for (;;)  // pixel loop
   {
      // plot line "slice" at current position
      if (s32AbsDx > s32AbsDy)  // "slice" runs vertically (<=> "shallow" line) ?
      {
          if (! m_bEraseModeOn) {
             for(tS32 s32Wy=s32Wsta; s32Wy<s32Wend; ++s32Wy) {
                vSetPixel(s32Xcur, s32Ycur+s32Wy);
             }
          } else {
             for(tS32 s32Wy=s32Wsta; s32Wy<s32Wend; ++s32Wy) {
                vClrPixel(s32Xcur, s32Ycur+s32Wy);
             }
          }
      }
      else  // "slice" runs horizontally (<=> "steep" line) ?
      {
          if (! m_bEraseModeOn) {
             for(tS32 s32Wx=s32Wsta; s32Wx<s32Wend; ++s32Wx) {
                vSetPixel(s32Xcur+s32Wx, s32Ycur);
             }
          } else {
             for(tS32 s32Wx=s32Wsta; s32Wx<s32Wend; ++s32Wx) {
                vClrPixel(s32Xcur+s32Wx, s32Ycur);
             }
          }
      }
      if ( (s32Xcur == s32Xend) && (s32Ycur == s32Yend) )
      {
         break;
      }
         
      // determine position of next line "slice"
      tS32 s32Err2 = 2*s32ErrXY;
      if (s32Err2 >= -s32AbsDy)  // x step needed?
      {
        s32ErrXY -= s32AbsDy;
        s32Xcur  += s32SgnDx;
      }
      if (s32Err2 <=  s32AbsDx)  // y step needed?
      {
        s32ErrXY += s32AbsDx;
        s32Ycur  += s32SgnDy;
      }
   }
}


tVoid vd_rvc_tclGraphics_DynGuideline_Plotter::vPlotLine_N_AA(tS32 s32Xsta, tS32 s32Ysta,  tS32 s32Xend, tS32 s32Yend,  tFloat fWidth)
{
   if (0.0F >= fWidth)
      return;  // don't plot line in this case
      
   tS32 s32AbsDx = abs(s32Xend-s32Xsta),  s32SgnDx = (s32Xsta<=s32Xend) ? 1 : -1;
   tS32 s32AbsDy = abs(s32Yend-s32Ysta),  s32SgnDy = (s32Ysta<=s32Yend) ? 1 : -1;
   
   tS32   s32ErrXY = s32AbsDx - s32AbsDy;    // error value e_xy
   tFloat fErrAA = (s32AbsDx+s32AbsDy == 0) ? 1.0 : (tFloat)sqrt( (tDouble)s32AbsDx*s32AbsDx + (tDouble)s32AbsDy*s32AbsDy );  // antialiasing error value

   tS32   s32Xcen = s32Xsta + s32Xend,  s32Ycen = s32Ysta + s32Yend;  // doubled line center coords
   tFloat fWidth2 = (fWidth+1) / 2.0F;
   
   tFloat fErrAA_x_fWidth2 = fErrAA*fWidth2;
   
   tS32   s32Xcur = s32Xsta,  s32Ycur = s32Ysta;
   tFloat fTransp;
   for (;;)  // pixel loop
   {
      if (! m_bEraseModeOn) {       
         fTransp = (tFloat)floor( (tDouble)(LINEPLOTTER_ALPHA_MAXVAL * (abs(s32ErrXY+s32AbsDy-s32AbsDx)/fErrAA + 1 - fWidth2) ) );
         vSetPixel(s32Xcur, s32Ycur, fTransp);
      } else {
         vClrPixel(s32Xcur, s32Ycur);
      }
      
      tS32 s32Err2 = s32ErrXY;
      tS32 s32Xtmp = s32Xcur;
      if (2*s32Err2 >= -s32AbsDx)  // x step needed?
      {
         tS32 s32Ytmp = s32Ycur;
         s32Err2 = s32AbsDy + s32Err2;
         while ( s32Err2 < fErrAA_x_fWidth2 && (s32Ytmp != s32Yend || s32AbsDx > s32AbsDy) )
         {
            s32Ytmp += s32SgnDy;
            if (! m_bEraseModeOn) {
               fTransp = (tFloat)floor( (tDouble)(LINEPLOTTER_ALPHA_MAXVAL * (abs(s32Err2)/(tDouble)fErrAA + 1 - fWidth2) ) );
               vSetPixel(        s32Xcur,         s32Ytmp, fTransp);  // for far better optical appearance,
               vSetPixel(s32Xcen-s32Xcur, s32Ycen-s32Ytmp, fTransp);  //    set the pixel "mirrored" at the line center to the same color/alpha
            } else {
               vClrPixel(        s32Xcur,         s32Ytmp);
               vClrPixel(s32Xcen-s32Xcur, s32Ycen-s32Ytmp);
            }
            s32Err2 += s32AbsDx;
         }
         if (s32Xcur == s32Xend)
         {
            break;
         }
         s32Err2  =  s32ErrXY;
         s32ErrXY -= s32AbsDy;
         s32Xcur  += s32SgnDx; 
      }
      if (2*s32Err2 <= s32AbsDy)  // y step needed?
      {
         s32Err2 = s32AbsDx - s32Err2;
         while ( s32Err2 < fErrAA_x_fWidth2 && (s32Xtmp != s32Xend || s32AbsDx < s32AbsDy) )
         {
            s32Xtmp += s32SgnDx;
            if (! m_bEraseModeOn) {
               fTransp = (tFloat)floor( (tDouble)(LINEPLOTTER_ALPHA_MAXVAL * (abs(s32Err2)/fErrAA + 1 - fWidth2) ) );
               vSetPixel(        s32Xtmp,         s32Ycur, fTransp);  // for far better optical appearance,
               vSetPixel(s32Xcen-s32Xtmp, s32Ycen-s32Ycur, fTransp);  // set the pixel "mirrored" at the line center to the same color/alpha
            } else {
               vClrPixel(        s32Xtmp,         s32Ycur);
               vClrPixel(s32Xcen-s32Xtmp, s32Ycen-s32Ycur);
            }
            s32Err2 += s32AbsDy;
         }
         if (s32Ycur == s32Yend)
         {
            break;
         }
         s32ErrXY += s32AbsDx;
         s32Ycur  += s32SgnDy; 
      }
   }
}

