/******************************************************************************/
/**
* \file    vd_rvc_tclGraphics_DynGuideline_CamParams.cpp
* \ingroup
*
* \brief
*
* \remark  Copyright : (c) 2012 Robert Bosch GmbH, Hildesheim
* \remark  Author    :
* \remark  Scope     :
*
* \todo
*/
/******************************************************************************/


/*******************************************************************************
                        Includes
*******************************************************************************/

#include "vd_rvc_tclGraphics_DynGuideline_Cameras.h"
#include "vd_rvc_tclGraphics_DynGuideline_Util.h"     // to get the GET_U16_FROM_U8_ARRAY and GET_U16_FROM_U8_ARRAY macros and the classes vd_rvc_tclFlexFloat and vd_rvc_tclCrc32

#include <cmath>
#include <cstring>

#include "dispvidctrl_AppMain_Trace.h"
// ETG defines
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

// defines and include of generated code
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_CLIENT_VIDEOPLAYER
#include "trcGenProj/Header/vd_rvc_tclGraphics_DynGuideline_Cameras.cpp.trc.h"
#endif




/*******************************************************************************
                  Specific defines for this component
*******************************************************************************/

#define CAMERA_RN2MAX_UMLIMITED   1.0e+30

#ifdef EIS2HI_DEB_TEST_ENV  // compiling for my "playground" PC environment?
   #define CAM_COLLECTION_PROVIDE_DUMMY_CAMERA  1    // set to 1 at any rate
#else  // compiling for the target
   #define CAM_COLLECTION_PROVIDE_DUMMY_CAMERA  1    // set to 1 or 0 as needed
#endif

/*******************************************************************************
                  Internal helper types and declaration of locally used derived classes
*******************************************************************************/

struct vd_rvc_tstSplineParam
{
   tDouble dXmax;
   tDouble dC3;
   tDouble dC2;
   tDouble dC1;
   tDouble dC0;
};


/*******************************************************************************
                  methods of the interface base camera class
*******************************************************************************/

tBool vd_rvc_tclCamera::bCamParamsValid(tVoid) const
{
   return ((NULL != m_pstCamParams) && (0 < m_pstCamParams->dFocLen) && (0 < m_pstCamParams->dSensorXsize) && (0 < m_pstCamParams->dSensorYsize)) ? true : false;
}



/*******************************************************************************
                  locally used derived camera classes
*******************************************************************************/

/*** camera without any optical distortion ***/

class vd_rvc_tclCameraWithoutDistortion : public vd_rvc_tclCamera
{
   public:
      vd_rvc_tclCameraWithoutDistortion(tVoid)
         : vd_rvc_tclCamera()                      {}
      vd_rvc_tclCameraWithoutDistortion(vd_rvc_tenCameraKdsId enKdsId, const vd_rvc_tstCameraParams * pstCamParams)
         : vd_rvc_tclCamera(enKdsId,pstCamParams)  {}
      virtual ~vd_rvc_tclCameraWithoutDistortion() {}
      
      virtual tDouble dGetMaxRNormSquare(tVoid) const                          { return CAMERA_RN2MAX_UMLIMITED;           }
      virtual tDouble dCalcRadialDistortionFactor(tDouble dRNormSquare) const  { return (0.0 >= dRNormSquare) ? 0.0 : 1.0; }
};


/*** camera which computes the radial distortion correction factor by means of interpolating a cubic spline approximation ***/

class vd_rvc_tclCameraWithSplineInterp : public vd_rvc_tclCamera
{
   private:
      const vd_rvc_tstSplineParam * m_pstSplineParams;
      tU32  m_u32NumSplineParams;
      tBool m_bUseLinExtrapol;
      
   public:
      vd_rvc_tclCameraWithSplineInterp(tVoid)
         : vd_rvc_tclCamera(), m_pstSplineParams(NULL), m_u32NumSplineParams(0), m_bUseLinExtrapol(false)  {}
      vd_rvc_tclCameraWithSplineInterp(vd_rvc_tenCameraKdsId enKdsId, const vd_rvc_tstCameraParams * pstCamParams,
                                       const vd_rvc_tstSplineParam * pstSplineParams, tU32 u32NumSplineParams, tBool bUseLinExtrapol)
         : vd_rvc_tclCamera(enKdsId, pstCamParams), m_pstSplineParams(pstSplineParams), m_u32NumSplineParams(u32NumSplineParams), m_bUseLinExtrapol(bUseLinExtrapol)  {}
      virtual ~vd_rvc_tclCameraWithSplineInterp() { m_pstSplineParams = NULL; }

      virtual tDouble dGetMaxRNormSquare(tVoid) const;
      virtual tDouble dCalcRadialDistortionFactor(tDouble dRNormSquare) const;

   protected:
      tVoid           vSetSplineParams(const vd_rvc_tstSplineParam * pstSplineParams, tU32 u32NumSplineParams, tBool bUseLinExtrapol);      
};

tVoid vd_rvc_tclCameraWithSplineInterp::vSetSplineParams(const vd_rvc_tstSplineParam * pstSplineParams, tU32 u32NumSplineParams, tBool bUseLinExtrapol)
{
   m_u32NumSplineParams = u32NumSplineParams;
   m_pstSplineParams    = (m_u32NumSplineParams) ? pstSplineParams : NULL;
   m_bUseLinExtrapol    = bUseLinExtrapol;
}

tDouble vd_rvc_tclCameraWithSplineInterp::dGetMaxRNormSquare(tVoid) const
{
   if ((NULL == m_pstSplineParams) || (0 == m_u32NumSplineParams) || (true == m_bUseLinExtrapol))
   {
      return CAMERA_RN2MAX_UMLIMITED;
   }
   tDouble dR = m_pstSplineParams[m_u32NumSplineParams-1].dXmax;
   return dR*dR;
}

tDouble vd_rvc_tclCameraWithSplineInterp::dCalcRadialDistortionFactor(tDouble dRNormSquare) const
{
   if (0.0 >= dRNormSquare)
   {
      return 0.0;
   }
   if ((NULL == m_pstSplineParams) || (0 == m_u32NumSplineParams))
   {
      return 1.0;
   }
   
   tDouble dRNorm = sqrt(dRNormSquare);
   // bisection search for the proper interpolation segment
   tU32 u32Lo = 0;
   tU32 u32Hi = m_u32NumSplineParams - 1;
   tU32 u32Mid;
   while (u32Lo < u32Hi)
   {
      u32Mid = (u32Lo+u32Hi) >> 1;
      if (dRNorm <= m_pstSplineParams[u32Mid].dXmax)
      {
         u32Hi = u32Mid;
      }
      else
      {
         u32Lo = u32Mid+1;
      }
   }
   
   const vd_rvc_tstSplineParam* pPar = & m_pstSplineParams[u32Lo];           // pointer to the spline segment to use for inter- or extrapolation ...
   tDouble dR0 = (0 == u32Lo)  ?  0.0  : m_pstSplineParams[u32Lo-1].dXmax;   // ... and its starting point
   tDouble dRCorr = 0.0;
   if (pPar->dXmax < dRNorm)  // point lies outside the definition range of the spline (i.e. beyond the last defined spline segment) ?
   {
      if (m_bUseLinExtrapol)  // we may still be allowed to use linear extrapolation
      {
         tDouble dR        = pPar->dXmax - dR0;                                              // full length of the last defined spline segment
         tDouble dLinExtC0 = (pPar->dC0 + dR*(pPar->dC1 + dR*(pPar->dC2 + dR*pPar->dC3)));   // C0 = y value of the end point of the last defined spline segment
         tDouble dLinExtC1 = (pPar->dC1 + dR*(2*pPar->dC2 + dR*3*pPar->dC3));                // C1 = slope of the last defined spline segment at its end point
         dRCorr            = (dLinExtC0 + dLinExtC1*(dRNorm-pPar->dXmax));                   // y_extrapol = C0_end + C1_end * (x - x_end)
      }  // else stay with dRCorr = 0.0
   }
   else
   {
      tDouble dR = dRNorm - dR0;    // spline coefficients for each interval [x_i,x_j] have been computed so that for given x, we need to compute the polynomial as a function of (x-x_i)
      dRCorr = (pPar->dC0 + dR*(pPar->dC1 + dR*(pPar->dC2 + dR*pPar->dC3)));   // use Horner scheme to evaluate polynomials efficiently
   }
   
   return  dRCorr / dRNorm;
}


/*** EOL configurable camera which computes the radial distortion correction factor by means of interpolating a cubic spline approximation ***/

class vd_rvc_tclEolConfigurableCamera : public vd_rvc_tclCameraWithSplineInterp
{
   private:
      vd_rvc_tstCameraParams m_stCameraParams;
      vd_rvc_tstSplineParam  m_astSplineParams[16];

   public:
      vd_rvc_tclEolConfigurableCamera(tVoid)
         : m_stCameraParams(), m_astSplineParams(), vd_rvc_tclCameraWithSplineInterp()  {}
      vd_rvc_tclEolConfigurableCamera(vd_rvc_tenCameraKdsId enKdsId)
         : m_stCameraParams(), m_astSplineParams(),vd_rvc_tclCameraWithSplineInterp(enKdsId, NULL, NULL,0, false)  {}
      virtual ~vd_rvc_tclEolConfigurableCamera()  {}
     
      tBool bSetEolConfig(const tU8 * pu8EolCfgString);
};

tBool vd_rvc_tclEolConfigurableCamera::bSetEolConfig(const tU8 * pu8EolCfgString)
{
   vSetCameraParams(NULL);
   vSetSplineParams(NULL,0,false);

   if (NULL == pu8EolCfgString)
      return false;
  
   tU8   u8CfgStrVersNo  =  pu8EolCfgString[0] & 0x0F;
   tBool bUseLinExtrapol = (pu8EolCfgString[0] & 0x10) ? true : false;
   ETG_TRACE_USR1(("DYN-GL: Camera parameter string version = %d", u8CfgStrVersNo));
   
   // determine what we have to evaluate, depending on the string version
   tU32  u32NumSplineSegs = 16;  // number of spline segments coded in the string; currently the same for all supported formats
   tU32  u32BytesPerFloat = 0;
   tU32  u32NumFmtBytes   = 0;
   tBool bFixedXmaxFmt    = false;
   switch(u8CfgStrVersNo)
   {
      case 0x00:   // version 0: using 24bit float coding; one common format byte for all spline model parameters xmax and c3...c0
         u32BytesPerFloat = 3;
         u32NumFmtBytes   = 1;
         bFixedXmaxFmt    = false;
         break;
      case 0x01:   // version 1: using 24bit float coding; one common format byte for for spline coefficients c3...c0, plus one separate, dedicated (fixed) format for xmax
         u32BytesPerFloat = 3;
         u32NumFmtBytes   = 1;
         bFixedXmaxFmt    = true;
         break;
      case 0x02:   // version 2: using 16bit float coding; one separate format byte for each of the spline model parameters xmax and c3...c0
         u32BytesPerFloat = 2;
         u32NumFmtBytes   = 5;
         bFixedXmaxFmt    = false;
         break;
      default:
         ETG_TRACE_USR1(("DYN-GL: ERROR: The above camera parameter string version is not supported by this SW"));
         return false;
   }
   ETG_TRACE_USR1(("DYN-GL: => NumSplineSegments=%u BytesPerFloat=%u NumFormatBytes=%u FixedXmaxFormat=%u UseLinExtrapol=%d",
                               u32NumSplineSegs,  u32BytesPerFloat,  u32NumFmtBytes,   bFixedXmaxFmt,     bUseLinExtrapol));

   tU32 u32FmtHdrSize = 1 + u32NumFmtBytes;
   tU32 u32FullDataLen = u32FmtHdrSize + (5*2) + (u32NumSplineSegs*(5*u32BytesPerFloat)) + 4;  // "(5*2)": global camera parameters (see further down); "4": CRC32 at the end
   vd_rvc_tclCrc32 oCrc(vd_rvc_tclCrc32::CRC32_KOOPMAN);
   if (false == oCrc.bCheck(pu8EolCfgString, u32FullDataLen))
   {
      ETG_TRACE_USR1(("DYN-GL: ERROR: CRC32 check of camera parameter string failed"));
      return false;
   }
   ++pu8EolCfgString;  // only after checking the CRC, we may safely advance pu8EolCfgString to the (first) format byte
   
   // decode the format byte(s) and set up the FlexFloat's accordingly
   vd_rvc_tclFlexFloat oFlexFloat[5];  // need 5 FlexFloat's for the spline model parameters xmax, c3, c2, c1 and c0 (in that order in this array)
   tU32 u32FmtByteIdx = 0;
   for (tU32 u32Idx=0; u32Idx<5; ++u32Idx)  // loop over the FlexFloat objects to set up
   {
      tS8   s8FloatFmt   = (tS8)(pu8EolCfgString[u32FmtByteIdx]);
      tU8   u8NumExpBits = (tU8)(s8FloatFmt & 0x07);
      tS16  s16ExpBias   = (s8FloatFmt & ~0x07) / 8;                  // this is equivalent to "s8FloatFmt >> 3", but lint doesn't like ">>" on a signed quantity
      u8NumExpBits += 2;                                              // FlexFloats always have at least 2 exponent bits, so only the number of bits in excess of this is coded in the format specifier
      s16ExpBias   -= 5;                                              // all biases known so far are in the range -11...+1, so they are coded relative to the "central" value -5 in the format specifier
      tBool bSigned      = (1==u32Idx || 2==u32Idx) ? true : false;   // only spline coefficients c3 and c2 can be <0; xmax, c1 and c0 are always >=0 per definition/construction of the spline => don't have sign bit
      if (0==u32Idx && bFixedXmaxFmt)   // "0==u32Idx" means "FlexFloat for xmax"
      {
         oFlexFloat[0].vSetup((tU8)(8*u32BytesPerFloat), 4, 1, false);       // xmax was coded with an own, dedicated (fixed) format: 4 bits exponent, exponent bias of 1, unsigned => set up FlexFloat accordingly
         ETG_TRACE_USR1(("DYN-GL:    FlexFloat format for dXmax = fix  <=> (%u,%u,%d,%u)", 8*u32BytesPerFloat, 4u, 1, false));
      }
      else   // for the spline coefficients c0...c3, but also for xmax if it was *not* coded with an own, dedicated (fixed) format, set up the FlexFloat using the data from the format specifier
      {
         oFlexFloat[u32Idx].vSetup((tU8)(8*u32BytesPerFloat), u8NumExpBits, s16ExpBias, bSigned);   // xmax is also coded using a format specifier byte (just as the spline coefficients)
         if (0==u32Idx) {   // means "xmax?"
            ETG_TRACE_USR1(("DYN-GL:    FlexFloat format for dXmax = 0x%02X <=> (%u,%u,%d,%u)",            s8FloatFmt & 0xFF, 8*u32BytesPerFloat, u8NumExpBits, s16ExpBias, bSigned));
         } else {            // c3...c0
            ETG_TRACE_USR1(("DYN-GL:    FlexFloat format for dC%u   = 0x%02X <=> (%u,%u,%d,%u)", 4-u32Idx, s8FloatFmt & 0xFF, 8*u32BytesPerFloat, u8NumExpBits, s16ExpBias, bSigned));
         }
      }
      if (! oFlexFloat[u32Idx].bFormatValid() )
      {
         ETG_TRACE_USR1(("DYN-GL: ERROR: invalid FlexFloat format specified in camera parameter string"));
         return false;
      }
      if (1 != u32NumFmtBytes)
         u32FmtByteIdx++;
      // else re-read and evaluate the one and only float format byte over and over again <=> same number of exponent bits and same exponent bias for all parameters (unless xmax has its own format, see above)
   }
   pu8EolCfgString += u32NumFmtBytes;

   // evaluate the global camera parameters
   m_stCameraParams.dFocLen         = GET_U16_FROM_U8_ARRAY(pu8EolCfgString,0) * 0.0001;
   m_stCameraParams.dSensorXsize    = GET_U16_FROM_U8_ARRAY(pu8EolCfgString,2) * 0.0001;
   m_stCameraParams.dSensorYsize    = GET_U16_FROM_U8_ARRAY(pu8EolCfgString,4) * 0.0001;
   m_stCameraParams.dOptAxisXoffs   = GET_S16_FROM_U8_ARRAY(pu8EolCfgString,6) * 0.1;
   m_stCameraParams.dOptAxisYoffs   = GET_S16_FROM_U8_ARRAY(pu8EolCfgString,8) * 0.1;
   m_stCameraParams.bIsADummyCamera = false;
   ETG_TRACE_USR1(("DYN-GL: m_stCameraParams:"));
   ETG_TRACE_USR1(("DYN-GL: .dFocLen       = %f mm", (float)(m_stCameraParams.dFocLen      )));  /*lint !e740*/
   ETG_TRACE_USR1(("DYN-GL: .dSensorXsize  = %f mm", (float)(m_stCameraParams.dSensorXsize )));  /*lint !e740*/
   ETG_TRACE_USR1(("DYN-GL: .dSensorYsize  = %f mm", (float)(m_stCameraParams.dSensorYsize )));  /*lint !e740*/
   ETG_TRACE_USR1(("DYN-GL: .dOptAxisXoffs = %f um", (float)(m_stCameraParams.dOptAxisXoffs)));  /*lint !e740*/
   ETG_TRACE_USR1(("DYN-GL: .dOptAxisYoffs = %f um", (float)(m_stCameraParams.dOptAxisYoffs)));  /*lint !e740*/
   pu8EolCfgString += (5*2);
   
   // finally decode the spline model, using the FlexFloat's we set up further up
   tU32 u32NumSegments;
   ETG_TRACE_USR1(("DYN-GL: m_astSplineParams[]:"));
   for (u32NumSegments=0; u32NumSegments<NUM_ARRAY_ELEMENTS(m_astSplineParams); u32NumSegments++)
   {
      vd_rvc_tstSplineParam* pParams = & m_astSplineParams[u32NumSegments];
      pParams->dXmax = oFlexFloat[0].dDecode(pu8EolCfgString                    , false);  // "false": in the EOL config string, all values are stored in BigEndian byte order
      if (0.0 == m_astSplineParams[u32NumSegments].dXmax)                                  // xmax==0 marks the end of the list of used spline segments
         break;
      pParams->dC3   = oFlexFloat[1].dDecode(pu8EolCfgString += u32BytesPerFloat, false);
      pParams->dC2   = oFlexFloat[2].dDecode(pu8EolCfgString += u32BytesPerFloat, false);
      pParams->dC1   = oFlexFloat[3].dDecode(pu8EolCfgString += u32BytesPerFloat, false);
      pParams->dC0   = oFlexFloat[4].dDecode(pu8EolCfgString += u32BytesPerFloat, false);
      pu8EolCfgString += u32BytesPerFloat;
      ETG_TRACE_USR1(("DYN-GL:  [%2u]          = { dXmax=%f, dC3=%f, dC2=%f, dC1=%f, dC0=%f }",
                           u32NumSegments, (float)(pParams->dXmax), (float)(pParams->dC3), (float)(pParams->dC2), (float)(pParams->dC1), (float)(pParams->dC0) ));  /*lint !e740*/
   }

   vSetCameraParams( & m_stCameraParams);
   vSetSplineParams( & m_astSplineParams[0], u32NumSegments, bUseLinExtrapol);
   
   return bCamParamsValid();
}


/*******************************************************************************
       Methods and static data of the collection of supported cameras
*******************************************************************************/


/*************** static data ***************/

#include "vd_rvc_tclGraphics_DynGuideline_CameraConfigs.cpp"


/*** the supported camera objects ***/

/** "real world" cameras **/

// helper macros to create a static const struct named   oCamera_<CameraName>  holding the configuration parameters of a camera
#define LENS_MODEL_PARAMS(LensModelName)   & astSplineParams_##LensModelName[0], NUM_ARRAY_ELEMENTS(astSplineParams_##LensModelName)
#define DEFINE_CAMERA_PARAMS(CameraName,KdsIdName,LensModelName,UseLinExtrapol) \
   static const vd_rvc_tclCameraWithSplineInterp   oCamera_##CameraName(VD_RVC_CAMERA_KDS_ID_##KdsIdName, & stCameraParams_##CameraName, LENS_MODEL_PARAMS(LensModelName), UseLinExtrapol)   
// define one object oCamera_XYZ for each known camera
DEFINE_CAMERA_PARAMS( Sony2400s_Nissan     , SONY_2400S_NISSAN         , Sony2400s_W130_6 , false);  // vFOV = 92°
#ifdef DYNGL_SUPPORT_CAM_SONY2400_NISSAN_V109
DEFINE_CAMERA_PARAMS( Sony2400s_Nissan_V109, SONY_2400S_NISSAN_VFOV109 , Sony2400s_W130_6 , false);  // vFOV = 109° , used (so far) only in H60A
#endif
DEFINE_CAMERA_PARAMS( Sony2200s            , SONY_2200S                , Sony2200s        , false);
// DEFINE_CAMERA_PARAMS( Panasonic_old        , PANASONIC                 , Panasonic_old    , false);  // not used anymore, replaced by Panasonic 7.5G with VFOV=102°
// DEFINE_CAMERA_PARAMS( Panasonic_75G_V113   , PANASONIC                 , Panasonic_75G    , false);  // Panasonic 7.5G will only be used with VFOV=102°
DEFINE_CAMERA_PARAMS( Panasonic_75G_V102   , PANASONIC                 , Panasonic_75G    , false);  // note that the Panasonic 7.5G "recycles" the KDS ID of the "old" Panasonic camera
DEFINE_CAMERA_PARAMS( Sony2400s_Renault    , SONY_2400S_RENAULT        , Sony2400s_W130_6 , false);
DEFINE_CAMERA_PARAMS( SemcoLogo            , SEMCO_LOGO                , SemcoLogo        , false);
DEFINE_CAMERA_PARAMS( SemcoGeneric_V109    , SEMCO_GENERIC_VFOV109     , SemcoLogo        , false);
DEFINE_CAMERA_PARAMS( SemcoL47             , SEMCO_L47                 , SemcoLogo        , false);
DEFINE_CAMERA_PARAMS( Sony2500s            , SONY_2500S                , Sony2400s_W130_6 , false);
DEFINE_CAMERA_PARAMS( Whetron              , WHETRON                   , Whetron          , false);
#ifdef DYNGL_SUPPORT_CAM_MVC_SEMCO
DEFINE_CAMERA_PARAMS( MVC_Semco            , MVC_SEMCO                 , MVC_Semco        , true );  // uses linear extrapolation because the RDC model just about covers the FOV
#endif
#undef LENS_MODEL_PARAMS     // not needed anymore
#undef DEFINE_CAMERA_PARAMS  // ditto

/** "special" cameras **/

static       vd_rvc_tclEolConfigurableCamera    oCamera_EolConfigurable  ( VD_RVC_CAMERA_KDS_ID_EOL_CONFIGURABLE );

#if (CAM_COLLECTION_PROVIDE_DUMMY_CAMERA == 1)
static const vd_rvc_tclCameraWithoutDistortion  oCamera_Dummy            ( VD_RVC_CAMERA_KDS_ID_DUMMY_CAMERA,          & stCameraParams_Dummy );
#endif


/*** the set of actually supported (i.e. user-selectable) cameras ***/

const vd_rvc_tclCamera * const vd_rvc_tclCameraCollection::m_apoSelectableCameras[] =
{
   & oCamera_Sony2400s_Nissan,       // vFOV = 92°
   #ifdef DYNGL_SUPPORT_CAM_SONY2400_NISSAN_V109
   & oCamera_Sony2400s_Nissan_V109,  // vFOV = 109° , used (so far) only in H60A
   #endif
   & oCamera_Sony2200s,
   & oCamera_Panasonic_75G_V102,
   & oCamera_Sony2400s_Renault,
   & oCamera_SemcoLogo,
   & oCamera_SemcoGeneric_V109,
   & oCamera_SemcoL47,
   & oCamera_Sony2500s,
   & oCamera_Whetron,
   #ifdef DYNGL_SUPPORT_CAM_MVC_SEMCO
   & oCamera_MVC_Semco,
   #endif
   #if (CAM_COLLECTION_PROVIDE_DUMMY_CAMERA == 1) && (defined EIS2HI_DEB_TEST_ENV)  // in my "playground" PC environment (and only there), I want to be able to explicitly select the dummy camera, too, via a call to the poSelectCamera() API 
   & oCamera_Dummy,
   #endif
   & oCamera_EolConfigurable
};
const tU32 vd_rvc_tclCameraCollection::m_u32NumSelectableCameras = NUM_ARRAY_ELEMENTS(vd_rvc_tclCameraCollection::m_apoSelectableCameras);


/*** methods ***/

const vd_rvc_tclCamera * vd_rvc_tclCameraCollection::poSelectCamera(tU8 u8CameraType)
{
   tU32 u32Idx;
   for(u32Idx=0; u32Idx<m_u32NumSelectableCameras; u32Idx++)
   {
      if ( m_apoSelectableCameras[u32Idx]->enGetKdsId() == (vd_rvc_tenCameraKdsId)u8CameraType )
      {
         return m_apoSelectableCameras[u32Idx];
      }
   }
   return NULL;
}

const vd_rvc_tclCamera * vd_rvc_tclCameraCollection::poGetDummyCamera(tVoid)
{
   #if (CAM_COLLECTION_PROVIDE_DUMMY_CAMERA == 1)
   return & oCamera_Dummy;
   #else
   return NULL;
   #endif
}

tBool vd_rvc_tclCameraCollection::bSetParamsOfConfigurableCamera(const tU8 * pu8EolCfgString)
{
   return oCamera_EolConfigurable.bSetEolConfig(pu8EolCfgString);
}
