/******************************************************************************/
/**
* \file    vd_rvc_tclGraphics_DynGuideline_Calib.cpp
* \ingroup
*
* \brief
*
* \remark  Copyright : (c) 2015 Robert Bosch GmbH, Hildesheim
* \remark  Author    :
* \remark  Scope     :
*
* \todo
*/
/******************************************************************************/


/*******************************************************************************
                        Includes
*******************************************************************************/

#include "vd_rvc_tclGraphics_DynGuideline_Calib.h"
#include "vd_rvc_tclGraphics_DynGuideline_CamPoint.h"

#include <cmath>

#ifndef TRUE
   #define TRUE ((tBool)true)
#endif
#ifndef FALSE
   #define FALSE ((tBool)false)
#endif

/*******************************************************************************
                  Specific defines for this component
*******************************************************************************/

#define DYN_GUIDELINE_TUNEANGLE_NONE   0x00
#define DYN_GUIDELINE_TUNEANGLE_YAW    0x01
#define DYN_GUIDELINE_TUNEANGLE_PITCH  0x02
#define DYN_GUIDELINE_TUNEANGLE_ROLL   0x04


#ifndef M_PI
  #define M_PI       3.141592653589793238462643383279
#endif
#define DEG2RAD(x)   ((x)*M_PI/180.0)


/*******************************************************************************
                        Static member variables
*******************************************************************************/

tBool                           vd_rvc_tclGraphics_GuidelineCalib::m_bTuningEnabled   = false;
tBool                           vd_rvc_tclGraphics_GuidelineCalib::m_bShiftAndRot     = false;
tDouble                         vd_rvc_tclGraphics_GuidelineCalib::m_dDefaultCropDist = 0.0;

const vd_rvc_tstVehicleParams * vd_rvc_tclGraphics_GuidelineCalib::m_ptVehParams;
vd_rvc_tstTuneParams            vd_rvc_tclGraphics_GuidelineCalib::m_stTuneParams;
tDouble                         vd_rvc_tclGraphics_GuidelineCalib::m_dNearRefYpos;
tDouble                         vd_rvc_tclGraphics_GuidelineCalib::m_dFarRefYpos;
tDouble                         vd_rvc_tclGraphics_GuidelineCalib::m_dGuidelnOffs;
tDouble                         vd_rvc_tclGraphics_GuidelineCalib::m_dInputTunePitch;

vd_rvc_tstSingleSideCalib       vd_rvc_tclGraphics_GuidelineCalib::m_stLftCalib       = { vd_rvc_t2Vector(), vd_rvc_t2Vector(), vd_rvc_tcl2x2Matrix(), 0.0 };
vd_rvc_tstSingleSideCalib       vd_rvc_tclGraphics_GuidelineCalib::m_stRgtCalib       = { vd_rvc_t2Vector(), vd_rvc_t2Vector(), vd_rvc_tcl2x2Matrix(), 0.0 };


/*******************************************************************************
                        Methods
*******************************************************************************/


vd_rvc_tclGraphics_GuidelineCalib::vd_rvc_tclGraphics_GuidelineCalib(tVoid)
{
}


vd_rvc_tclGraphics_GuidelineCalib::~vd_rvc_tclGraphics_GuidelineCalib(tVoid)
{
}


tVoid vd_rvc_tclGraphics_GuidelineCalib::vSetConfig(const vd_rvc_tstVehicleParams * ptVehParams)
{
   m_bTuningEnabled   = false;  // prevent ourselves from performing any tuning while we are still busy setting up the calibration

   m_ptVehParams      = ptVehParams;                  // remark: only needed until this method exits, but more convenient to have it as member instead of passing it along through the call tree
   m_stTuneParams     = m_ptVehParams->stTuneParams;  // make a writable local copy of the input tuning values (also only needed until this method exits)
   m_dNearRefYpos     = m_ptVehParams->adGuidelnStatDist[0];
   m_dFarRefYpos      = m_ptVehParams->adGuidelnStatDist[2];
   m_dGuidelnOffs     = m_ptVehParams->dGuidelnOffs;
   m_dDefaultCropDist = m_ptVehParams->dGuidelnStartDist;
   m_dInputTunePitch  = 0.0;    // relevant/used only if the selected method uses tuning angle estimation (both with and without residual offset correction)

   m_bShiftAndRot     = false;  // default; will be overwritten later acsording to selected tuning method
   tBool   bCorrResid = false;  // if set to true later, this implies m_bShiftAndRot==true as well
   tBool   bSuccess   = true;   // starting assumption
   switch (m_stTuneParams.u8Method)
   {
      default:
      case DYN_GUIDELINE_TUNEMETH_NONE:
         bSuccess = false;  // to reset all tuning later on
         break;
         
      case DYN_GUIDELINE_TUNEMETH_USE_YAW_PITCH_ROLL:
         // nothing special to do, keep all input tuning angles as-is
         break;
         
      case DYN_GUIDELINE_TUNEMETH_USE_PITCH_TUNE_DXY:
         m_bShiftAndRot         = true;
         m_stTuneParams.dCamYaw = m_stTuneParams.dCamRoll = 0.0;  // use only input pitch tuning angle, reset others to 0
         break;
         
      case DYN_GUIDELINE_TUNEMETH_USE_PITCH_CALC_YAW_ROLL:
      case DYN_GUIDELINE_TUNEMETH_USE_PITCH_CALC_YAW_ROLL_CORR_RESID:
         bCorrResid        = (DYN_GUIDELINE_TUNEMETH_USE_PITCH_CALC_YAW_ROLL_CORR_RESID == m_stTuneParams.u8Method);
         m_dInputTunePitch = m_stTuneParams.dCamPitch;  // we will need this later during yaw/roll estimation, and optionally also during residual offset correction
         bSuccess          = bEstimateYawAndRoll();
         break;
         
      case DYN_GUIDELINE_TUNEMETH_CALC_YAW_PITCH_ROLL:
      case DYN_GUIDELINE_TUNEMETH_CALC_YAW_PITCH_ROLL_CORR_RESID:
         bCorrResid        = (DYN_GUIDELINE_TUNEMETH_CALC_YAW_PITCH_ROLL_CORR_RESID == m_stTuneParams.u8Method);
         m_dInputTunePitch = 0.0;  // method has no initial pitch tuning angle available as input, hence starting with a "clean slate" from zero pitch tuning angle
         bSuccess          = bEstimateYawPitchAndRoll();
         break;
         
      case DYN_GUIDELINE_TUNEMETH_IMPROVE_PITCH_CALC_YAW_ROLL:
      case DYN_GUIDELINE_TUNEMETH_IMPROVE_PITCH_CALC_YAW_ROLL_CORR_RESID:
         bCorrResid        = (DYN_GUIDELINE_TUNEMETH_IMPROVE_PITCH_CALC_YAW_ROLL_CORR_RESID == m_stTuneParams.u8Method);
         m_dInputTunePitch = m_stTuneParams.dCamPitch;  // we will need this later during yaw/pitch/roll estimation, and optionally also during residual offset correction
         bSuccess          = bEstimateYawPitchAndRoll();
         break;
   }
   
   if (bSuccess)
   {
      // apply all configured and/or estimated tuning angles
      vd_rvc_tclCamPoint::vSetCamTuningAngles(m_stTuneParams.dCamYaw, m_stTuneParams.dCamPitch, m_stTuneParams.dCamRoll);
      if (bCorrResid)  // this implies shift-and-rotate; but we won't use the full input offsets for the shift-and-rotate, but instead only
      {                // the residuals which remain after all (input or estimated) tuning angles have been accounted for
         bSuccess       = bCalcResidualXyOffsets();  
         m_bShiftAndRot = bSuccess;
      }
   }
   if (!bSuccess)
      vd_rvc_tclCamPoint::vSetCamTuningAngles(0.0, 0.0, 0.0);
   
   m_stLftCalib.dNearLen = (bSuccess) ? m_stTuneParams.stOffsLft.dNearLineLen : 0.0;
   m_stRgtCalib.dNearLen = (bSuccess) ? m_stTuneParams.stOffsRgt.dNearLineLen : 0.0;
   
   if (bSuccess && m_bShiftAndRot)
   {
      vSetupLineShiftAndRot((tBool)true );  // 'true'  <=> left curve
      vSetupLineShiftAndRot((tBool)false);  // 'false' <=> right curve
   }
   else
   {
      vResetLineShiftAndRot((tBool)true );  // 'true'  <=> left curve
      vResetLineShiftAndRot((tBool)false);  // 'false' <=> right curve
   }
   
   m_bTuningEnabled = bSuccess;   
}


tBool vd_rvc_tclGraphics_GuidelineCalib::bEstimateYawAndRoll(tVoid)
{
   vd_rvc_t3Vector oLineVarTot, oLineVarYaw, oLineVarRoll;
   
   // determine the line movement descriptor values (see below) of the reference points in case of pure yaw/roll tuning
   vCalcGuidelnVarianceWithTuningAngle(DYN_GUIDELINE_TUNEANGLE_YAW,  & oLineVarYaw );  // movements per degree of yaw tuning
   vCalcGuidelnVarianceWithTuningAngle(DYN_GUIDELINE_TUNEANGLE_ROLL, & oLineVarRoll);  // movements per degree of roll tuning
   
   // use the descriptor values to build the matrix of the system of linear equations
   vd_rvc_tcl2x2Matrix oAngVar2LineVar = vd_rvc_tcl2x2Matrix( oLineVarYaw.x(), oLineVarRoll.x(),   // xshift: average shift of left and right reference point in x direction
                                                              oLineVarYaw.z(), oLineVarRoll.z() ); // dydist: change of distance between left and right point in y direction
   // invert the matrix
   vd_rvc_tcl2x2Matrix oLineVar2AngVar;
   if (! oAngVar2LineVar.bInvert( & oLineVar2AngVar ))   // should never happen, but then again: you also never know ...
      return false;
   // this inverse matrix can now be used to compute the required tune angles from the observed average shifts and distance in/decreases
      
   // convert input x/y offset values of the near reference points (measured in cm by means of the calibration carpet) to movement descriptor values
   vCalcGuidelnVarianceForTuningOffsets(& oLineVarTot);
   
   // calculate output delta yaw/pitch/roll value estimates from those quantities by applying the solution of the system of linear equations to them
   vd_rvc_t2Vector oAngVar = oLineVar2AngVar * vd_rvc_t2Vector(oLineVarTot.x(), oLineVarTot.z());
   m_stTuneParams.dCamYaw  = oAngVar.x();
   m_stTuneParams.dCamRoll = oAngVar.y();
   
   return true;
}


tBool vd_rvc_tclGraphics_GuidelineCalib::bEstimateYawPitchAndRoll(tVoid)
{
   vd_rvc_t3Vector oLineVarTot, oLineVarYaw, oLineVarPitch, oLineVarRoll;
      
   // determine the line movement descriptor values (see below) of the reference points in case of pure yaw/pitch/roll tuning
   vCalcGuidelnVarianceWithTuningAngle(DYN_GUIDELINE_TUNEANGLE_YAW,   & oLineVarYaw  );  // movements per degree of yaw tuning
   vCalcGuidelnVarianceWithTuningAngle(DYN_GUIDELINE_TUNEANGLE_PITCH, & oLineVarPitch);  // movements per degree of pitch tuning
   vCalcGuidelnVarianceWithTuningAngle(DYN_GUIDELINE_TUNEANGLE_ROLL,  & oLineVarRoll );  // movements per degree of roll tuning

   // use the descriptor values to build the matrix of the system of linear equations
   vd_rvc_tcl3x3Matrix oAngVar2LineVar = vd_rvc_tcl3x3Matrix( oLineVarYaw.x(), oLineVarPitch.x(), oLineVarRoll.x(),   // xshift: average shift of left and right reference point in x direction
                                                              oLineVarYaw.y(), oLineVarPitch.y(), oLineVarRoll.y(),   // yshift: average shift of left and right reference point in y direction
                                                              oLineVarYaw.z(), oLineVarPitch.z(), oLineVarRoll.z() ); // dydist: change of distance between left and right point in y direction
   // invert the matrix
   vd_rvc_tcl3x3Matrix oLineVar2AngVar;
   if (! oAngVar2LineVar.bInvert( & oLineVar2AngVar ))   // should never happen, but then again: you also never know ...
      return false;
   // this inverse matrix can now be used to compute the required tune angles from the observed average shifts and distance in/decreases
   
   // convert input x/y offset values of the near reference points (measured in cm by means of the calibration carpet) to movement descriptor values
   vCalcGuidelnVarianceForTuningOffsets(& oLineVarTot);
   
   // calculate output delta yaw/pitch/roll value estimates from those quantities by applying the solution of the system of linear equations to them
   vd_rvc_t3Vector oAngVar  = oLineVar2AngVar * oLineVarTot;
   m_stTuneParams.dCamYaw   = oAngVar.x();
   m_stTuneParams.dCamPitch = oAngVar.y() + m_dInputTunePitch;
   m_stTuneParams.dCamRoll  = oAngVar.z();
   
   return true;
}


tVoid vd_rvc_tclGraphics_GuidelineCalib::vCalcGuidelnVarianceForTuningOffsets(vd_rvc_t3Vector *poVariance)
{
   // setup the initial tuning values as supplied as input (yaw and roll are not part of the tuning string, hence always 0.0)
   vd_rvc_tclCamPoint::vSetCamTuningAngles(0.0, m_dInputTunePitch, 0.0);

   // compute x/y sensor coordinates of left and near right reference point where they *should* theoretically appear
   vd_rvc_tclCamPoint oPLftExpPos, oPRgtExpPos;
   oPLftExpPos.vSetGroundCoords(+m_dGuidelnOffs,                                  m_dNearRefYpos,                                  0.0, DYN_GUIDELINE_TUNE_SIDE_NONE);  // near left point
   oPRgtExpPos.vSetGroundCoords(-m_dGuidelnOffs,                                  m_dNearRefYpos,                                  0.0, DYN_GUIDELINE_TUNE_SIDE_NONE);  // near right point
   
   // compute x/y sensor coordinates of left and near right reference point where they actually are *observed* (i.e. including the measured ground offsets)
   vd_rvc_tclCamPoint oPLftObsPos, oPRgtObsPos;
   oPLftObsPos.vSetGroundCoords(+m_dGuidelnOffs+m_stTuneParams.stOffsLft.dNearDx, m_dNearRefYpos+m_stTuneParams.stOffsLft.dNearDy, 0.0, DYN_GUIDELINE_TUNE_SIDE_NONE);  // shifted near left point
   oPRgtObsPos.vSetGroundCoords(-m_dGuidelnOffs+m_stTuneParams.stOffsRgt.dNearDx, m_dNearRefYpos+m_stTuneParams.stOffsRgt.dNearDy, 0.0, DYN_GUIDELINE_TUNE_SIDE_NONE);  // shifted near right point
   
   // finally compute the three "characteristic quantities" describing how the sensor coordinates of the two reference points changed due to the ground offets
   vCalcGuidelnVarianceParams(oPLftObsPos, oPLftExpPos,  oPRgtObsPos, oPRgtExpPos,  poVariance);
}


tVoid vd_rvc_tclGraphics_GuidelineCalib::vCalcGuidelnVarianceWithTuningAngle(tU32 u32VarAngleId, vd_rvc_t3Vector *poVariance)
{
   #define DYN_GUIDELINE_TUNEANGLE_VAR_VAL  DEG2RAD(2.0)  // we will vary +-2°, which should not be too far off the range of real-world camera mount angle errors
   
   // compute x/y sensor coordinates of left and right reference point for positive tuning angle variance
   vSetupVariedTuningAngle(u32VarAngleId, +DYN_GUIDELINE_TUNEANGLE_VAR_VAL);
   vd_rvc_tclCamPoint oPLftPosVar, oPRgtPosVar;
   oPLftPosVar.vSetGroundCoords(+m_dGuidelnOffs, m_dNearRefYpos, 0.0, DYN_GUIDELINE_TUNE_SIDE_NONE);  // near left point
   oPRgtPosVar.vSetGroundCoords(-m_dGuidelnOffs, m_dNearRefYpos, 0.0, DYN_GUIDELINE_TUNE_SIDE_NONE);  // near right point
   
   // same for negative tuning angle variance
   vSetupVariedTuningAngle(u32VarAngleId, -DYN_GUIDELINE_TUNEANGLE_VAR_VAL);
   vd_rvc_tclCamPoint oPLftNegVar, oPRgtNegVar;
   oPLftNegVar.vSetGroundCoords(+m_dGuidelnOffs, m_dNearRefYpos, 0.0, DYN_GUIDELINE_TUNE_SIDE_NONE);  // near left point
   oPRgtNegVar.vSetGroundCoords(-m_dGuidelnOffs, m_dNearRefYpos, 0.0, DYN_GUIDELINE_TUNE_SIDE_NONE);  // near right point

   // finally compute the three "characteristic quantities" describing how the sensor coordinates of the two reference points change due to the tuning angle variation
   vCalcGuidelnVarianceParams(oPLftNegVar, oPLftPosVar,  oPRgtNegVar, oPRgtPosVar,  poVariance);
   *poVariance *= (1.0/(2*DYN_GUIDELINE_TUNEANGLE_VAR_VAL));  // variance per unit tuning angle change
}


tVoid vd_rvc_tclGraphics_GuidelineCalib::vSetupVariedTuningAngle(tU32 u32VarAngleId, tDouble dVarAngleVal)
{
   switch(u32VarAngleId)
   {
      default:
      case DYN_GUIDELINE_TUNEANGLE_NONE:  vd_rvc_tclCamPoint::vSetCamTuningAngles(0.0,          m_dInputTunePitch,              0.0         );  break;
      case DYN_GUIDELINE_TUNEANGLE_YAW:   vd_rvc_tclCamPoint::vSetCamTuningAngles(dVarAngleVal, m_dInputTunePitch,              0.0         );  break;
      case DYN_GUIDELINE_TUNEANGLE_PITCH: vd_rvc_tclCamPoint::vSetCamTuningAngles(0.0,          m_dInputTunePitch+dVarAngleVal, 0.0         );  break;
      case DYN_GUIDELINE_TUNEANGLE_ROLL:  vd_rvc_tclCamPoint::vSetCamTuningAngles(0.0,          m_dInputTunePitch,              dVarAngleVal);  break;
   }
}


tVoid vd_rvc_tclGraphics_GuidelineCalib::vCalcGuidelnVarianceParams(const vd_rvc_tclCamPoint &poLftFrom, const vd_rvc_tclCamPoint &poLftTo,
                                                                    const vd_rvc_tclCamPoint &poRgtFrom, const vd_rvc_tclCamPoint &poRgtTo, vd_rvc_t3Vector *poVariance)
{
   tDouble dDXlft = poLftTo.m_tSensCoo.x() - poLftFrom.m_tSensCoo.x();  // calculate movement of left point in x and y direction
   tDouble dDYlft = poLftTo.m_tSensCoo.y() - poLftFrom.m_tSensCoo.y();
   tDouble dDXrgt = poRgtTo.m_tSensCoo.x() - poRgtFrom.m_tSensCoo.x();  // same for right point
   tDouble dDYrgt = poRgtTo.m_tSensCoo.y() - poRgtFrom.m_tSensCoo.y();
   
   tDouble dXshift = ( (dDXlft + dDXrgt)/2 );   // average shift of left and right reference point in x direction
   tDouble dYshift = ( (dDYlft + dDYrgt)/2 );   // average shift of left and right reference point in y direction
   tDouble dDYdist = ( (dDYlft - dDYrgt)   );   // change of distance/separation of left and right point in y direction
   *poVariance = vd_rvc_t3Vector(dXshift,dYshift,dDYdist);
}


tBool vd_rvc_tclGraphics_GuidelineCalib::bCalcResidualXyOffsets(tVoid)
{
   // first setup the initial angle tuning (which may or may not contain a non-zero pitch tuning angle supplied as input)
   // (note: the input dx/y offsets were measured with this initial tuning already being in effect)
   vd_rvc_tclCamPoint::vSetCamTuningAngles(0.0, m_dInputTunePitch, 0.0);

   // calculate the sensor coordinates of the reference point for this initial tuning
   vd_rvc_tclCamPoint oPNearLft, oPNearRgt,  oPFarLft, oPFarRgt;
   oPNearLft.vSetGroundCoords(+m_dGuidelnOffs, m_dNearRefYpos, 0.0, DYN_GUIDELINE_TUNE_SIDE_NONE);  // near left point
   oPNearRgt.vSetGroundCoords(-m_dGuidelnOffs, m_dNearRefYpos, 0.0, DYN_GUIDELINE_TUNE_SIDE_NONE);  // near right point
   oPFarLft .vSetGroundCoords(+m_dGuidelnOffs, m_dFarRefYpos,  0.0, DYN_GUIDELINE_TUNE_SIDE_NONE);  // far left point
   oPFarRgt .vSetGroundCoords(-m_dGuidelnOffs, m_dFarRefYpos,  0.0, DYN_GUIDELINE_TUNE_SIDE_NONE);  // far right point
   
   // now setup the final tuning angles (including all estimated values as well)
   vd_rvc_tclCamPoint::vSetCamTuningAngles(m_stTuneParams.dCamYaw, m_stTuneParams.dCamPitch, m_stTuneParams.dCamRoll);
   
   // For this final angle tuning, calculate by how much we need to shift the ground coordinates of each reference point so that it falls onto
   // the same sensor coordinates as calculated above for the initial angle tuning. This is the portion of the total input dx/y offsets which has
   // already been accounted for by applying the estimated tuning angles => only the remainder has to be "tweaked away" by line shift-and-rotate
   vd_rvc_t2Vector oDxyGround;

   if (! bCalcRequiredGroundCoordinateOffset(oPNearLft, & oDxyGround))  // near left point
      return false;
   m_stTuneParams.stOffsLft.dNearDx -= oDxyGround.x();
   m_stTuneParams.stOffsLft.dNearDy -= oDxyGround.y();

   if (! bCalcRequiredGroundCoordinateOffset(oPNearRgt, & oDxyGround))  // near right point
      return false;
   m_stTuneParams.stOffsRgt.dNearDx -= oDxyGround.x();
   m_stTuneParams.stOffsRgt.dNearDy -= oDxyGround.y();
   
   if (! bCalcRequiredGroundCoordinateOffset(oPFarLft, & oDxyGround))  // far left point
      return false;
   m_stTuneParams.stOffsLft.dFarDx -= oDxyGround.x();
   
   if (! bCalcRequiredGroundCoordinateOffset(oPFarRgt, & oDxyGround))  // far right point
      return false;
   m_stTuneParams.stOffsRgt.dFarDx -= oDxyGround.x();
   
   return true;
}


tBool vd_rvc_tclGraphics_GuidelineCalib::bCalcRequiredGroundCoordinateOffset(const vd_rvc_tclCamPoint &oPGoal, vd_rvc_t2Vector *oDxyGround)
{
   // The point oPGoal has (a) ground coordinates, and (b) sensor coordinates which have been calculated from (a) using the tuning angle setup which was active at that time.
   // This method calculates by how much we need to move away from the ground coordinates (a) so that these shifted coordinates give the same sensor coordinates (b),
   // using the tuning angle setup which is active *now*.
   // Or expressed differently:
   // Changing the tuning angle setup causes the sensor coordinates resulting from certain ground coordinates to change as well. The same change of sensor coordinates could be achieved
   // by keeping the tuning angle setup *unchanged*, but instead shifting the *ground* coordinates by a certain amount. This method calculates this amount.
   
   vd_rvc_t2Vector     oXyGndGoal = vd_rvc_t2Vector(oPGoal.m_tGroundCoo);
   vd_rvc_tclCamPoint  oPCurr, oPXinc, oPYinc;
   vd_rvc_t2Vector     oDxySnsCurrXinc, oDxySnsCurrYinc, oDxySnsCurrGoal, oDxyGnd;
   vd_rvc_tcl2x2Matrix oDxyGnd2DxySns, oDxySns2DxyGnd;
   
   // we start with the ground coordinates of the point oPGoal, and then we will successively modify them:
   vd_rvc_t2Vector     oXyGndCurr = oXyGndGoal;
   // due to the wild non-linearity of the whole ground-coordinates-to-sensor-coordinates transformation, we need to do it iteratively, using a linear approximation in each step
   for (tU8 u8Idx=0; u8Idx<5; u8Idx++)
   {
      // compute by how much the sensor coordinates of the current position differ from those of the goal position:
      oPCurr.vSetGroundCoords(oXyGndCurr,                            DYN_GUIDELINE_TUNE_SIDE_NONE);
      oDxySnsCurrGoal = oPGoal.m_tSensCoo - oPCurr.m_tSensCoo;
      
      // then compute by how much the sensor coordinates of the current position change if we move away from it by one unit in either x or y direction:
      oPXinc.vSetGroundCoords(oXyGndCurr.x()+1,oXyGndCurr.y()  ,0.0, DYN_GUIDELINE_TUNE_SIDE_NONE);
      oPYinc.vSetGroundCoords(oXyGndCurr.x()  ,oXyGndCurr.y()+1,0.0, DYN_GUIDELINE_TUNE_SIDE_NONE);
      oDxySnsCurrXinc = oPXinc.m_tSensCoo - oPCurr.m_tSensCoo;
      oDxySnsCurrYinc = oPYinc.m_tSensCoo - oPCurr.m_tSensCoo;

      // Now cast these variances into a matrix and invert it
      oDxyGnd2DxySns = vd_rvc_tcl2x2Matrix(oDxySnsCurrXinc.x(), oDxySnsCurrYinc.x(),
                                           oDxySnsCurrXinc.y(), oDxySnsCurrYinc.y());
      if (! oDxyGnd2DxySns.bInvert(& oDxySns2DxyGnd))   // should never happen, but then again: you also never know ...
         return false;

      // Finally use the inverse matrix to convert the current (goal-current) sensor coordinates difference into an estimate of the according ground coordinate shift.
      // Note that this is the step where we exploit the "local almost-linearity" of the underlying ground-to-sensor coordinate transformation, and that the "almost"
      // is actually the reason why we need iterations (though typically only very few)
      oDxyGnd     = oDxySns2DxyGnd * oDxySnsCurrGoal;
      oXyGndCurr += oDxyGnd;      // update the current position with the estimated offset
      if ( oDxyGnd*oDxyGnd < 1.0 )  // if the last ground coordinate correction we applied was "small enough" (less than 1cm), bail out
         break;
   }
   
   *oDxyGround = oXyGndCurr - oXyGndGoal;  // that's it!
   return true;
}


tVoid vd_rvc_tclGraphics_GuidelineCalib::vResetLineShiftAndRot(tBool bLeftCurve)
{
   vd_rvc_tstSingleSideCalib * ptCalib = (bLeftCurve) ? &  m_stLftCalib : & m_stRgtCalib;
   ptCalib->oShiftDxy = vd_rvc_t2Vector();
   ptCalib->oRotCenXy = vd_rvc_t2Vector();
   ptCalib->oRotation = vd_rvc_tcl2x2Matrix();
}


tVoid vd_rvc_tclGraphics_GuidelineCalib::vSetupLineShiftAndRot(tBool bLeftCurve)
{
   const vd_rvc_tstTuneOffsParams & pstOffs = (bLeftCurve) ? m_stTuneParams.stOffsLft : m_stTuneParams.stOffsRgt;
   vd_rvc_tstSingleSideCalib      * ptCalib = (bLeftCurve) ? &  m_stLftCalib          : & m_stRgtCalib;
   
   // linear shift == x/y offset of the near guideline reference point:
   ptCalib->oShiftDxy = vd_rvc_t2Vector(pstOffs.dNearDx, pstOffs.dNearDy);
   // line rotation center point == near reference point on the current guideline (left or right):
   ptCalib->oRotCenXy = vd_rvc_t2Vector( (bLeftCurve) ? m_dGuidelnOffs : -m_dGuidelnOffs, m_dNearRefYpos);
   // line rotation angle is given by the difference of the x offsets of the near and far guideline reference point and their distance in y direction:
   tDouble dRotDx     = pstOffs.dFarDx - pstOffs.dNearDx;
   tDouble dRotDy     = m_dFarRefYpos  - m_dNearRefYpos;
   tDouble dRotHyp    = sqrt(dRotDx*dRotDx + dRotDy*dRotDy);
   tDouble dRotCos    =   dRotDy / dRotHyp;
   tDouble dRotSin    = - dRotDx / dRotHyp;
   ptCalib->oRotation = vd_rvc_tcl2x2Matrix(dRotCos,dRotSin, -dRotSin,dRotCos);
}


/////// output methods: public APIs for system requests to apply tuning 


tDouble vd_rvc_tclGraphics_GuidelineCalib::vGetLineCropDistance(tDouble dTuneSide)
{
   if (!m_bTuningEnabled)         // configured for tuning method "none", or tuning configuration failed (or is still in progress)?
      return m_dDefaultCropDist;  // => return the general untuned vehicle crop distance
      
   tDouble dDCropLft = ((0 < m_stLftCalib.dNearLen) ? (m_dNearRefYpos - m_stLftCalib.dNearLen) : m_dDefaultCropDist) - m_stLftCalib.oShiftDxy.y();
   tDouble dDCropRgt = ((0 < m_stRgtCalib.dNearLen) ? (m_dNearRefYpos - m_stRgtCalib.dNearLen) : m_dDefaultCropDist) - m_stRgtCalib.oShiftDxy.y();

   tDouble dDCrop;
   if      (+1 <= dTuneSide)   // requested position lies on (or even left of) left curve:
   {
      dDCrop = dDCropLft;      // saturate at left line crop distance 
   }
   else if (-1 >= dTuneSide)   // requested position lies on (or even right of) right curve:
   {
      dDCrop = dDCropRgt;      // saturate at right line crop distance 
   }
   else                        // requested position lies strictly between the two curves
   {
      tDouble dLeftWeight = (dTuneSide+1)/2;  // weight of left value varies from 1 (if dTuneSide==+1==left) to 0 (if dTuneSide==-1==right) 
      dDCrop = dDCropRgt + (dDCropLft - dDCropRgt) * dLeftWeight;  // interpolate between the crop distances of the left and right curve
   }
   
   return dDCrop;
}


tVoid vd_rvc_tclGraphics_GuidelineCalib::vTunePoint(vd_rvc_t3Vector * poPoint, tDouble dTuneSide)
{
   if (! (m_bTuningEnabled && m_bShiftAndRot))  // configured for a tuning method (including "none") which doesn't require shift-and-rotate, or tuning configuration failed (or is still in progress)?
      return;                                   // => don't touch the point
      
   vd_rvc_t2Vector oP1 = vd_rvc_t2Vector(*poPoint);
   
   if      (+1 <= dTuneSide)               // requested position lies on (or even left of) left curve:
   {
      vTuneXyCoords(& oP1, (tBool)true);   // apply pure left-curve tuning ('true' <=> left curve)
   }
   else if (-1 >= dTuneSide)               // requested position lies on (or even right of) right curve:
   {
      vTuneXyCoords(& oP1, (tBool)false);  // apply pure right-curve tuning ('false' <=> right curve)
   }
   else                                    // requested position lies strictly between the two curves
   {
      // apply position-weighted average of left- and right-curve tuning
      vd_rvc_t2Vector oP2 = oP1;    // clone
      vTuneXyCoords(& oP1, (tBool)false);     // 'false' <=> right curve
      vTuneXyCoords(& oP2, (tBool)true );     // 'true'  <=> left curve
      tDouble dLeftWeight = (dTuneSide+1)/2;  // weight of left vector varies from 1 (if dTuneSide==+1==left) to 0 (if dTuneSide==-1==right) 
      oP1 += (oP2 - oP1) * dLeftWeight;       // interpolate between the shifts and rotations of the left and right curve
   }
   
   *poPoint = vd_rvc_t3Vector(oP1.x(), oP1.y(), poPoint->z() );  // keep input z position untouched
}

// a private helper for the above method

tVoid vd_rvc_tclGraphics_GuidelineCalib::vTuneXyCoords(vd_rvc_t2Vector * poPoint, tBool bLeftCurve)
{
   const vd_rvc_tstSingleSideCalib & oCalib = (bLeftCurve) ? m_stLftCalib : m_stRgtCalib;
   *poPoint = ( oCalib.oRotation * ((*poPoint - oCalib.oShiftDxy) - oCalib.oRotCenXy) )  +  oCalib.oRotCenXy;
   // 1)              |                     first apply linear shift      |                       |
   // 2)              |                                 then translate origin to rot. center      |
   // 3) now rotate around (new) origin                                                           |
   // 4)                                                                     finally translate origin back to old position
}
