/******************************************************************************/
/**
* \file    vd_rvc_tclGraphics_DynGuideline.cpp
* \ingroup
*
* \brief
*
* \remark  Copyright : (c) 2012 Robert Bosch GmbH, Hildesheim
* \remark  Author    :
* \remark  Scope     :
*
* \todo
*/
/******************************************************************************/


/*******************************************************************************
                        Includes
*******************************************************************************/

#include "vd_rvc_tclGraphics_DynGuideline.h"
#include "vd_rvc_tclGraphics_DynGuideline_Config.h"
#include "vd_rvc_tclGraphics_DynGuideline_Cameras.h"
#include "vd_rvc_tclGraphics_DynGuideline_CamPoint.h"
#include "vd_rvc_tclGraphics_DynGuideline_Calib.h"
#include "vd_rvc_tclGraphics_DynGuideline_Plotter.h"

#include <cmath>

#ifndef TRUE
   #define TRUE ((tBool)true)
#endif
#ifndef FALSE
   #define FALSE ((tBool)false)
#endif

#include "dispvidctrl_AppMain_Trace.h"
// ETG defines
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

// defines and include of generated code
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_CLIENT_VIDEOPLAYER
#include "trcGenProj/Header/vd_rvc_tclGraphics_DynGuideline.cpp.trc.h"
#endif

/*******************************************************************************
                  Specific defines for this component
*******************************************************************************/


/*** defines to control the behaviour on errors while selecting or configuring the camera ***/

#ifdef EIS2HI_DEB_TEST_ENV  // compiling for my "playground" PC environment?
   #define DYN_GUIDELINE_USE_DUMMY_CAMERA_ON_INVALID_CAM_SELECTION  1  // set to 1 at any rate
   #define DYN_GUIDELINE_USE_DUMMY_CAMERA_ON_INVALID_CAM_PARAMS     1  // set to 1 at any rate
#else  // compiling for the target
   #define DYN_GUIDELINE_USE_DUMMY_CAMERA_ON_INVALID_CAM_SELECTION  0  // set to 1 or 0 as needed
   #define DYN_GUIDELINE_USE_DUMMY_CAMERA_ON_INVALID_CAM_PARAMS     0  // set to 1 or 0 as needed
#endif


/*** steering related configuration defines ***/

#define CONVERT_SWA_TO_FWA(swa)            ((swa)*(m_tVehParams.dSWAtoFWA_Coeff1 + (swa)*(swa)*m_tVehParams.dSWAtoFWA_Coeff3))


#define TIRE_ANGLE_SATURATION_MODE_HIDE_DYNLINES    0
#define TIRE_ANGLE_SATURATION_MODE_FREEZE_DYNLINES  1
#define TIRE_ANGLE_SATURATION_MODE                 (TIRE_ANGLE_SATURATION_MODE_FREEZE_DYNLINES)

#if (TIRE_ANGLE_SATURATION_MODE != TIRE_ANGLE_SATURATION_MODE_HIDE_DYNLINES)  && \
    (TIRE_ANGLE_SATURATION_MODE != TIRE_ANGLE_SATURATION_MODE_FREEZE_DYNLINES)
   #error "invalid TIRE_ANGLE_SATURATION_MODE configuration"
#endif



/*** defines to determine the max. arc length and resolution of the dynamic guidelines ***/

#define DYN_GUIDELINE_MAX_ARC_ANGLE_DEG    90.0  // degrees (output)
#define DYN_GUIDELINE_MAX_ARC_ANGLE_RAD    (DEG2RAD(DYN_GUIDELINE_MAX_ARC_ANGLE_DEG))

#define DYN_GUIDELINE_ARC_ANGLE_STEP_DEG   5.0    // degrees (output)
#define DYN_GUIDELINE_NUM_ANG_STEPS        ((DYN_GUIDELINE_MAX_ARC_ANGLE_DEG)/(DYN_GUIDELINE_ARC_ANGLE_STEP_DEG) + 1)

#define DYN_GUIDELINE_NUM_END_LINE_STEPS   5   // configure to 0 to disable the dynamic end bar completely
#define DYN_GUIDELINE_NUM_CENT_LINE_STEPS  8   // configure to 0 to disable the static center line completely


/*** graphics layout and content related defines ***/

// some macros to ease access to the RGB values for the different curve(part)s
#define DYN_GUIDELINE_CURVE_RGB(idx)       m_tVehParams.au8GuidelnRgb[idx][0], m_tVehParams.au8GuidelnRgb[idx][1], m_tVehParams.au8GuidelnRgb[idx][2]
// a special explicit RGB color
#define DYN_GUIDELINE_RGBCOLOR_WARN_IND    0xFF,0x00,0x00    // color of warning indicators

// defines for the type of test config warner to be used
#define WARNING_IND_TYPE_NONE              0
#define WARNING_IND_TYPE_EXCL_MARKS        1
#define WARNING_IND_TYPE_BIG_CROSS         2
#define WARNING_IND_TYPE                  (WARNING_IND_TYPE_BIG_CROSS)

#if (WARNING_IND_TYPE != WARNING_IND_TYPE_NONE)        && \
    (WARNING_IND_TYPE != WARNING_IND_TYPE_EXCL_MARKS)  && \
    (WARNING_IND_TYPE != WARNING_IND_TYPE_BIG_CROSS)
   #error "invalid WARNING_IND_TYPE configuration"
#endif

#define DYN_GUIDELINE_USE_ADAPTIVE_DYN_STEPSIZE
#ifdef  DYN_GUIDELINE_USE_ADAPTIVE_DYN_STEPSIZE
   #define DYN_GUIDELINE_MIN_DYNLINE_PLOTLEN   8   // see comment in vDrawDynamicLines() for what this is good for
#endif

#define DYN_GUIDELINE_NUM_STAT_HOR_BAR_STEPS   12  // number of line steps used to draw full-vehicle-width static horizontal guideline bar (Renault)

#define DYN_GUIDELINE_USE_IMGSIZE_DEPENDENT_LINEWIDTH  // comment out this #define if the plotted lines shall *not* get thinner for guidelne image sizes < screen size


/*** utility defines ***/

#ifndef M_PI
  #define M_PI       3.141592653589793238462643383279
#endif
#define DEG2RAD(x)   ((x)*M_PI/180.0)

#define SWAP(type, v1,v2)  {type __tmpval__ = v1; v1 = v2; v2 = __tmpval__;}


/*******************************************************************************
                  Local helper types
*******************************************************************************/

struct tstDynLineTrackGeometry
{
   vd_rvc_t2Vector     oPCen;
   vd_rvc_t2Vector     oVecDxy;
   tDouble             dFactLft2Rgt;
   vd_rvc_tcl2x2Matrix oMAngStep;
};



/*******************************************************************************
                  Local constants
*******************************************************************************/


/*******************************************************************************
    Class implementation
*******************************************************************************/

/*** static member variables ***/

vd_rvc_tclGraphics_DynGuideline_Plotter vd_rvc_tclGraphics_DynGuideline::m_oLinePlotter      = vd_rvc_tclGraphics_DynGuideline_Plotter();
vd_rvc_tstVehicleParams                 vd_rvc_tclGraphics_DynGuideline::m_tVehParams;
vd_rvc_tstCaptureAreaParams             vd_rvc_tclGraphics_DynGuideline::m_tCapParams;
vd_rvc_tstImageAreaParams               vd_rvc_tclGraphics_DynGuideline::m_tImgParams;
vd_rvc_tstOutputBufferParams            vd_rvc_tclGraphics_DynGuideline::m_tBufParams;
const vd_rvc_tclCamera *                vd_rvc_tclGraphics_DynGuideline::m_poCamera          = NULL;
tDouble                                 vd_rvc_tclGraphics_DynGuideline::m_dLastUsedFrontAng = 0.0;
tDouble                                 vd_rvc_tclGraphics_DynGuideline::m_dLastUsedRearAng  = 0.0;
tU32                                    vd_rvc_tclGraphics_DynGuideline::m_u32LastMode       = DYN_GUIDELINE_DRAW_NONE;

tBool                                   vd_rvc_tclGraphics_DynGuideline::m_bBufCfgValid     = false;
tBool                                   vd_rvc_tclGraphics_DynGuideline::m_bVehCfgValid     = false;
tBool                                   vd_rvc_tclGraphics_DynGuideline::m_bImgCfgValid     = false;
tBool                                   vd_rvc_tclGraphics_DynGuideline::m_bCamCfgValid     = false;
tBool                                   vd_rvc_tclGraphics_DynGuideline::m_bCapCfgValid     = false;
tBool                                   vd_rvc_tclGraphics_DynGuideline::m_bCanDraw         = false;
tBool                                   vd_rvc_tclGraphics_DynGuideline::m_bIsRealCfg       = false;
tBool                                   vd_rvc_tclGraphics_DynGuideline::m_bWarnIfTestCfg   = true;

tBool                                   vd_rvc_tclGraphics_DynGuideline::m_bTempCalibStyleActive      = false;
tBool                                   vd_rvc_tclGraphics_DynGuideline::m_bLastTempCalibStyleActive  = false;
tBool                                   vd_rvc_tclGraphics_DynGuideline::m_bDoFullClearBeforeNextPlot = true;

/*** methods ***/

vd_rvc_tclGraphics_DynGuideline::vd_rvc_tclGraphics_DynGuideline(tVoid)
{
}


vd_rvc_tclGraphics_DynGuideline::~vd_rvc_tclGraphics_DynGuideline()
{
}


tVoid vd_rvc_tclGraphics_DynGuideline::vSetOutBuffer(const vd_rvc_tstOutputBufferParams * pstBufParams) const
{
   m_bBufCfgValid = (NULL != pstBufParams) && (NULL != pstBufParams->pvBuf) && (0 < pstBufParams->u32NumBufXpix) && (0 < pstBufParams->u32NumBufYpix);
   if (m_bBufCfgValid && (NULL != pstBufParams))  // checking pstBufParams against NULL again because lint likes it better this way ...
      m_tBufParams = *pstBufParams;
   vInitConfig();
}


tVoid vd_rvc_tclGraphics_DynGuideline::vSetCaptureConfig(const vd_rvc_tstCaptureAreaParams * pstCaptureParams) const
{
   m_bCapCfgValid = (NULL != pstCaptureParams) && (0 < pstCaptureParams->u32NumVideoImgXpix)    && (0 < pstCaptureParams->u32NumVideoImgYpix)
                                               && (0 < pstCaptureParams->u32NumCaptureAreaXpix) && (0 < pstCaptureParams->u32NumCaptureAreaYpix);
   if (m_bCapCfgValid && (NULL != pstCaptureParams))  // checking pstCaptureParams against NULL again because lint likes it better this way ...
      m_tCapParams = *pstCaptureParams;
   vInitConfig();
}


tVoid vd_rvc_tclGraphics_DynGuideline::vSetImageConfig(const vd_rvc_tstImageAreaParams * pstImgParams) const
{
   m_bImgCfgValid = (NULL != pstImgParams) && (0 < pstImgParams->u32NumImgXpix) && (0 < pstImgParams->u32NumImgYpix);
   if (m_bImgCfgValid && (NULL != pstImgParams))  // checking pstImgParams against NULL again because lint likes it better this way ...
      m_tImgParams = *pstImgParams;
   vInitConfig();
}


tVoid vd_rvc_tclGraphics_DynGuideline::vSetVehicleConfig(const vd_rvc_tstKdsVehicleConfig * pstKdsVehConfig) const
{
   m_bVehCfgValid = vd_rvc_tclGraphics_DynGuideline_Config::bEvalConfigString(pstKdsVehConfig, & m_tVehParams);
   vInitConfig();
}


tVoid vd_rvc_tclGraphics_DynGuideline::vSetVehicleConfig(const char * sPredefCfgName) const
{
   m_bVehCfgValid = vd_rvc_tclGraphics_DynGuideline_Config::bSetPredefConfig(sPredefCfgName, & m_tVehParams);
   vInitConfig();
}

tVoid vd_rvc_tclGraphics_DynGuideline::vIndVehicleConfigChanged(tVoid) const
{
   m_bVehCfgValid = vd_rvc_tclGraphics_DynGuideline_Config::bFinalizeVehicleConfig(& m_tVehParams);
   vInitConfig();
}


tVoid vd_rvc_tclGraphics_DynGuideline::vSelectCamera(tU8 u8CameraType) const
{
   m_tVehParams.u8CameraType = u8CameraType;
   vInitConfig();
}

tBool vd_rvc_tclGraphics_DynGuideline::bSetParamsOfConfigurableCamera(const tU8 * pu8EolCfgString) const
{
   tBool bSuccess = vd_rvc_tclCameraCollection::bSetParamsOfConfigurableCamera(pu8EolCfgString);
   vInitConfig();
   return bSuccess;
}


tVoid vd_rvc_tclGraphics_DynGuideline::vSetCalibConfig(const tU8 * pu8CalibString) const
{
   (tVoid) vd_rvc_tclGraphics_DynGuideline_Config::bEvalCalibString(pu8CalibString, & m_tVehParams);
   vInitConfig();
}


tVoid vd_rvc_tclGraphics_DynGuideline::vSetTemporaryCalibGuidelnStyle(tBool bActive) const
{
   m_bTempCalibStyleActive = bActive;
   m_bDoFullClearBeforeNextPlot = true;
}

tVoid vd_rvc_tclGraphics_DynGuideline::vAllowWarningIndIfUsingDummyConfig(tBool bAllow) const
{
   m_bWarnIfTestCfg = bAllow;
   m_u32LastMode = DYN_GUIDELINE_DRAW_NONE;  // to force drawing on next bDraw() call which draws anything
   m_bDoFullClearBeforeNextPlot = true;
}

tVoid vd_rvc_tclGraphics_DynGuideline::vTraceDebugInformation(tVoid) const
{
   vd_rvc_tclGraphics_DynGuideline_Config::vDebugVehicleParams(& m_tVehParams, m_bVehCfgValid);
   vd_rvc_tclGraphics_DynGuideline_Config::vDebugTuningParams(& m_tVehParams, m_bVehCfgValid);
}


tVoid vd_rvc_tclGraphics_DynGuideline::vSetupCamera(tVoid) const
{
   m_poCamera = vd_rvc_tclCameraCollection::poSelectCamera(m_tVehParams.u8CameraType);
   #if (DYN_GUIDELINE_USE_DUMMY_CAMERA_ON_INVALID_CAM_SELECTION == 1)
   if ( NULL == m_poCamera )
   {
      m_poCamera = vd_rvc_tclCameraCollection::poGetDummyCamera();  // might return NULL (namely if vd_rvc_tclCameraCollection does not provide a dummy camera)
   }
   #endif
   #if (DYN_GUIDELINE_USE_DUMMY_CAMERA_ON_INVALID_CAM_PARAMS == 1)
   if ( (NULL != m_poCamera) && ! m_poCamera->bCamParamsValid() )
   {
      m_poCamera = vd_rvc_tclCameraCollection::poGetDummyCamera();  // might return NULL (namely if vd_rvc_tclCameraCollection does not provide a dummy camera)
   }
   #endif
   m_bCamCfgValid = (NULL != m_poCamera) && m_poCamera->bCamParamsValid();
}

tVoid vd_rvc_tclGraphics_DynGuideline::vInitConfig(tVoid) const
{
   vSetupCamera();

   m_bCanDraw   = m_bCamCfgValid && m_bBufCfgValid && m_bImgCfgValid && m_bCapCfgValid && m_bVehCfgValid;
   m_bIsRealCfg = m_bCanDraw && (! m_poCamera->bIsADummyCamera()) && (! m_tVehParams.bIsATestConfig);

   if (m_bBufCfgValid)
   {
      m_oLinePlotter.vSetOutBuffer(& m_tBufParams);  // do this already now, even if other config parts are still missing, so that we can at least *clear* the output buffer if required
   }
   
   if (m_bCanDraw)  // do the rest only if we finally have all config pieces together
   {
      // make sure that captured video image area isn't larger than the video camera image area, and reduce if required:
      m_tCapParams.u32NumCaptureAreaXpix = (m_tCapParams.u32NumCaptureAreaXpix <= m_tCapParams.u32NumVideoImgXpix) ? m_tCapParams.u32NumCaptureAreaXpix : m_tCapParams.u32NumVideoImgXpix;
      m_tCapParams.u32NumCaptureAreaYpix = (m_tCapParams.u32NumCaptureAreaYpix <= m_tCapParams.u32NumVideoImgYpix) ? m_tCapParams.u32NumCaptureAreaYpix : m_tCapParams.u32NumVideoImgYpix;
      
      // make sure that the "shifted" capture area does not extend beyond the video camera image, and reduce shift if required:
      if (m_tCapParams.u32CaptureAreaXoffs + m_tCapParams.u32NumCaptureAreaXpix > m_tCapParams.u32NumVideoImgXpix)
      {
         m_tCapParams.u32CaptureAreaXoffs = m_tCapParams.u32NumVideoImgXpix - m_tCapParams.u32NumCaptureAreaXpix;
      }
      if (m_tCapParams.u32CaptureAreaYoffs + m_tCapParams.u32NumCaptureAreaYpix > m_tCapParams.u32NumVideoImgYpix)
      {
         m_tCapParams.u32CaptureAreaYoffs = m_tCapParams.u32NumVideoImgYpix - m_tCapParams.u32NumCaptureAreaYpix;
      }      
      
      // make sure that output image area isn't larger than the logical surface buffer area, and reduce if required:
      m_tImgParams.u32NumImgXpix = (m_tImgParams.u32NumImgXpix <= m_tBufParams.u32NumBufXpix) ? m_tImgParams.u32NumImgXpix : m_tBufParams.u32NumBufXpix;
      m_tImgParams.u32NumImgYpix = (m_tImgParams.u32NumImgYpix <= m_tBufParams.u32NumBufYpix) ? m_tImgParams.u32NumImgYpix : m_tBufParams.u32NumBufYpix;
   
      // make sure that the "shifted" output image area does not extend beyond the logical surface buffer area, and reduce shift if required:
      if (m_tImgParams.u32ImgXoffs + m_tImgParams.u32NumImgXpix > m_tBufParams.u32NumBufXpix)
      {
         m_tImgParams.u32ImgXoffs = m_tBufParams.u32NumBufXpix - m_tImgParams.u32NumImgXpix;
      }
      if (m_tImgParams.u32ImgYoffs + m_tImgParams.u32NumImgYpix > m_tBufParams.u32NumBufYpix)
      {
         m_tImgParams.u32ImgYoffs = m_tBufParams.u32NumBufYpix - m_tImgParams.u32NumImgYpix;
      }
      
      vd_rvc_tclCamPoint::vSetConfig(& m_tVehParams, & m_tCapParams, & m_tImgParams, m_poCamera);
   
      tS32 s32Xmin = (tS32)m_tImgParams.u32ImgXoffs;
      tS32 s32Xmax = s32Xmin + (tS32)m_tImgParams.u32NumImgXpix - 1;
      tS32 s32Ymin = (tS32)m_tImgParams.u32ImgYoffs;
      tS32 s32Ymax = s32Ymin + (tS32)m_tImgParams.u32NumImgYpix - 1;
      m_oLinePlotter.vSetBoundingBox(s32Xmin,s32Xmax, s32Ymin,s32Ymax);
      
      // finally, set the effective rear wheel angle threshold values for guideline hiding and guideline updating hysteresis
      if (! m_tVehParams.bFourWheelSteering)  // 2WS case:
      {
         m_tVehParams.dEffRearTireThresh = m_tVehParams.dEffMinRearTireChg = 0.0;  // configured values for RWA are irrelevant
      }
      else                                    // 4WS case:
      {
         if (0 == m_tVehParams.dFrontTireThresh)    // if *no* hiding threshold is configured for FWA, this disables the hiding of dyn. guidelines in general
            m_tVehParams.dEffRearTireThresh = 0.0;  // => set the effective RWA threshold also to 0
         else if (0 < m_tVehParams.dRearTireThresh)                          // else, if an RWA hiding threshold is explicitly configured, ...
            m_tVehParams.dEffRearTireThresh = m_tVehParams.dRearTireThresh;  // ... use that configured value as-is;
         else                                                                     // but if *no* explicit RWA hiding threshold is configured, ...
            m_tVehParams.dEffRearTireThresh = m_tVehParams.dFrontTireThresh / 2;  // ... compute a suitable value (which must be strictly >0!) from the configured *FWA* threshold
           
         if (0 == m_tVehParams.dMinFrontTireChg)    // use the same logic for the RWA hysteresis value
            m_tVehParams.dEffMinRearTireChg = 0.0;
         else if (0 < m_tVehParams.dMinRearTireChg)
            m_tVehParams.dEffMinRearTireChg = m_tVehParams.dMinRearTireChg;
         else
            m_tVehParams.dEffMinRearTireChg = m_tVehParams.dMinFrontTireChg / 2;
      }
      
      // compute the "line width scale factor" (<=1.0) to be used for the current screen(=buffer) and output image area
      #ifdef DYN_GUIDELINE_USE_IMGSIZE_DEPENDENT_LINEWIDTH
      m_tVehParams.fLineWidthScalingFact = (tFloat)(1.25 * m_tImgParams.u32NumImgXpix / m_tBufParams.u32NumBufXpix);   // "1.25 * ..." to make it <1.0 (<=> thinner lines) only if the output image covers less than 80% of the whole screen width
      if (m_tVehParams.fLineWidthScalingFact > 1.0)                                                                    // (but make sure that "1.25 * " does not unintentionally make the lines *thicker* than initially configured)
         m_tVehParams.fLineWidthScalingFact = 1.0F;
      #else
      m_tVehParams.fLineWidthScalingFact = 1.0F;
      #endif      
   }
   
   m_u32LastMode = DYN_GUIDELINE_FORCE_UPDATE;  // to force at least a ClearBuffer on the next bDraw() call
   m_bDoFullClearBeforeNextPlot = true;
}


tBool vd_rvc_tclGraphics_DynGuideline::bDraw(tDouble dSteerWheelAngleDeg, tU32 u32Mode, tDouble dRearWheelAngleDeg) const
{
   if (!m_bCanDraw || DYN_GUIDELINE_STYLE_UNDEFINED==m_tVehParams.u8GuidelnStyle)
   {
      if (m_bBufCfgValid && m_bImgCfgValid)
      {
         m_oLinePlotter.vClearOutBufferContent(& m_tImgParams);  // at least this little courtesy we will do to our caller ...
         // Extra debug info added to understand why we occasionally get here
         ETG_TRACE_USR4(("DYN-GL: vd_rvc_tclGraphics_DynGuideline::bDraw(): Buffer cleared only - m_bBufCfgValid=%d m_bImgCfgValid=%d m_bCanDraw=%d m_tVehParams.u8GuidelnStyle=%d", m_bBufCfgValid,m_bImgCfgValid, m_bCanDraw, m_tVehParams.u8GuidelnStyle));
         return true;  // buffer changed
      }
      ETG_TRACE_USR4(("DYN-GL: vd_rvc_tclGraphics_DynGuideline::bDraw(): Buffer unchanged - m_bBufCfgValid=%d m_bImgCfgValid=%d", m_bBufCfgValid,m_bImgCfgValid));
      return false;  // buffer unchanged
   }
   
   if (m_bTempCalibStyleActive != m_bLastTempCalibStyleActive)
   {
      u32Mode |= DYN_GUIDELINE_FORCE_UPDATE;
      m_bLastTempCalibStyleActive = m_bTempCalibStyleActive;
   }
   
   // first save the relevant values which were used to produce the previous plot (we may need them further down, but the call to bEvalSteerAngleInput() below will update them)
   tDouble dFrontWheelAngInPrevPlot = m_dLastUsedFrontAng;
   tDouble dRearWheelAngInPrevPlot  = m_dLastUsedRearAng;
   tU32    u32ModeInPrevPlot        = m_u32LastMode;
   
   // now find out what needs to be done (if anything at all)
   tDouble dFrontWheelAngleDeg = 0.0;
   if (false == bEvalSteerAngleInput(dSteerWheelAngleDeg, & dFrontWheelAngleDeg, & dRearWheelAngleDeg, & u32Mode))
   {
      ETG_TRACE_USR4(("DYN-GL: vd_rvc_tclGraphics_DynGuideline::bDraw(): bEvalSteerAngleInput()==false -> nothing to do"));
      return false;  // nothing to do => keep previous output buffer content unchanged and report back
   }
   
   // remove the guidelines currently on the screen
   if (m_bDoFullClearBeforeNextPlot) {
      // do it the traditional, hard, costly, brute force way
      m_oLinePlotter.vClearOutBufferContent(& m_tImgParams);
      m_bDoFullClearBeforeNextPlot = false;  // but at the next plot, we will again try to do a smart-erase (unless someone changes e.g. some configuration item(s) in between)
   } else {
      // do it by exactly reproducing the previous plot, but this time using fully transparent pixels (actually MUCH faster than the above!)
      m_oLinePlotter.vSetEraseMode(TRUE);
      vDrawLines(dFrontWheelAngInPrevPlot,dRearWheelAngInPrevPlot,u32ModeInPrevPlot);
   }
      
   // do the actual plotting of the static and dynamic lines, the warning cross, ..., according to u32Mode
   m_oLinePlotter.vSetEraseMode(FALSE);
   vDrawLines(dFrontWheelAngleDeg,dRearWheelAngleDeg,u32Mode);

   return true;  // buffer changed (because it was least cleared above)
}

tVoid vd_rvc_tclGraphics_DynGuideline::vDrawLines(tDouble dFrontWheelAngleDeg, tDouble dRearWheelAngleDeg, tU32 u32Mode) const
{
   m_oLinePlotter.vSetAntiAliasing(TRUE);
   
   tBool bLinesDrawn = false;
   if (u32Mode & (DYN_GUIDELINE_DRAW_STATIC_SIDE|DYN_GUIDELINE_DRAW_STATIC_CENT))
   {
      bLinesDrawn = true;
      vDrawStaticLines(u32Mode);
   }
   if (u32Mode & DYN_GUIDELINE_DRAW_DYNAMIC)
   {
      bLinesDrawn = true;
      vDrawDynamicLines(dFrontWheelAngleDeg, dRearWheelAngleDeg);
   }

   if ((!m_bIsRealCfg) && m_bWarnIfTestCfg && bLinesDrawn)
   {
      vDrawWarningIndicator();
   }   
}

tBool vd_rvc_tclGraphics_DynGuideline::bEvalSteerAngleInput(tDouble dSteerWheelAngleDeg, tDouble *pdFrontWheelAngleDeg, tDouble *pdRearWheelAngleDeg, tU32 *pu32Mode) const
{
   tBool bShowDyn      = (*pu32Mode     & DYN_GUIDELINE_DRAW_DYNAMIC)     ? true : false;  // dyn. guideline request as sent by caller
   tBool bLastShowDyn  = (m_u32LastMode & DYN_GUIDELINE_DRAW_DYNAMIC)     ? true : false;  // did we draw the dyn. lines in the last call?
   tBool bShowStat     = (*pu32Mode     & DYN_GUIDELINE_DRAW_STATIC_SIDE) ? true : false;  // stat. guideline request as sent by caller
   tBool bForcedUpdate = (*pu32Mode     & DYN_GUIDELINE_FORCE_UPDATE)     ? true : false;

   // (A) input value preparation
   if (bShowDyn)
   {
      // step 0: calibrate away angle sensor zero point offsets
      dSteerWheelAngleDeg  -= m_tVehParams.stTuneParams.dSwaZeroOffset;
      *pdRearWheelAngleDeg -= m_tVehParams.stTuneParams.dRwaZeroOffset;
      
      // step 1: check (and, if necessary, limit) input angle value ranges
      if (fabs(dSteerWheelAngleDeg) > m_tVehParams.dMaxSteerWhlAngle)
      {
         #if   (TIRE_ANGLE_SATURATION_MODE == TIRE_ANGLE_SATURATION_MODE_HIDE_DYNLINES)
         bShowDyn = false;   // => don't draw dyn. guidelines
         #elif (TIRE_ANGLE_SATURATION_MODE == TIRE_ANGLE_SATURATION_MODE_FREEZE_DYNLINES)
         dSteerWheelAngleDeg = (dSteerWheelAngleDeg >= 0) ? +m_tVehParams.dMaxSteerWhlAngle : -m_tVehParams.dMaxSteerWhlAngle;
         #else
         #error "configured value of TIRE_ANGLE_SATURATION_MODE unsupported"
         #endif
      }
      
      if (fabs(*pdRearWheelAngleDeg) > m_tVehParams.dMaxRearTireAngle)
      {
         #if   (TIRE_ANGLE_SATURATION_MODE == TIRE_ANGLE_SATURATION_MODE_HIDE_DYNLINES)
         bShowDyn = false;   // => don't draw dyn. guidelines
         #elif (TIRE_ANGLE_SATURATION_MODE == TIRE_ANGLE_SATURATION_MODE_FREEZE_DYNLINES)
         *pdRearWheelAngleDeg = (*pdRearWheelAngleDeg >= 0) ? +m_tVehParams.dMaxRearTireAngle : -m_tVehParams.dMaxRearTireAngle;
         #else
         #error "configured value of TIRE_ANGLE_SATURATION_MODE unsupported"
         #endif
      }      
   }
   if (bShowDyn)   // request still holds?
   {
      // step 2: convert steering wheel angle to front tire angle and normalize turn direction sense
      *pdFrontWheelAngleDeg = CONVERT_SWA_TO_FWA(dSteerWheelAngleDeg);
      if (true == m_tVehParams.bPosTireAngleIsCcw)
      {
         *pdFrontWheelAngleDeg = - *pdFrontWheelAngleDeg;  // within this class, positive FWA/RWA always mean clockwise tire turns
         *pdRearWheelAngleDeg  = - *pdRearWheelAngleDeg;
      }
      if (false == m_tVehParams.bFourWheelSteering)
      {
         *pdRearWheelAngleDeg  = 0.0;
      }
   }
   
   // (B) check lower angle ("hide") thresholds and hysteresis
   tBool bVisibleDynUpd = false;  // starting assumption
   if (bShowDyn)   // request still holds?
   {
      tDouble dFwaThresh = m_tVehParams.dFrontTireThresh,    dFwaHyst = m_tVehParams.dMinFrontTireChg;
      tDouble dRwaThresh = m_tVehParams.dEffRearTireThresh,  dRwaHyst = m_tVehParams.dEffMinRearTireChg;
      
      // step 3: first check if we passed a hide/show threshold for the dyn. guidelines, and hide them if needed (but only if we are going to draw stat. side lines as well)
      if (bShowStat && 0<dFwaThresh /* && 0<dRwaThresh */)  // if dFwaThresh>0, also dRwaThresh>0 (ensured elsewhere), so checking dFwaThresh is sufficient
      {
         // an angle is considered too small if it falls below its hide threshold; but if it was too small already on the last call, it must rise above the
         // hide-threshold plus the hysteresis value to be considered large enough again (in order to avoid hide-show-hide flicker on small changes around the threshold)
         if (fabs(m_dLastUsedFrontAng) < dFwaThresh)
            dFwaThresh += dFwaHyst;
         if (fabs(m_dLastUsedRearAng ) < dRwaThresh)
            dRwaThresh += dRwaHyst;
         tBool bFwaTooSmall = (fabs(*pdFrontWheelAngleDeg) < dFwaThresh);
         tBool bRwaTooSmall = (fabs(*pdRearWheelAngleDeg ) < dRwaThresh);
         if (bFwaTooSmall && (!m_tVehParams.bFourWheelSteering || bRwaTooSmall))  // note: in the 2WS case, it's enough if FWA is too small
            bShowDyn = false;   // hide dyn. guidelines
      }
      // step 4: if dyn. guidelines were visible after the last call and will be visible now, too, check input angle values for changes above the hysteresis threshold (which may be 0, note!)
      if (bShowDyn && bLastShowDyn)
      {
         tDouble dAbsFwaChg = fabs(*pdFrontWheelAngleDeg - m_dLastUsedFrontAng);
         tDouble dAbsRwaChg = fabs(*pdRearWheelAngleDeg  - m_dLastUsedRearAng );
         tBool bSignifFwaChg = (0==dFwaHyst) ? (0<dAbsFwaChg) : ((dFwaHyst-1E-10)<dAbsFwaChg); // "-1E-10" to cope with roundoff errors in "currangle - lastangle" lines above
         tBool bSignifRwaChg = (0==dRwaHyst) ? (0<dAbsRwaChg) : ((dRwaHyst-1E-10)<dAbsRwaChg); // note: RWA may change independently of FWA (e.g. due to speed changes) => check separately
         bVisibleDynUpd = (bSignifFwaChg || bSignifRwaChg);  // if true, this means "dyn. guidelines were and still are visible, but angle(s) changed so much that we need to update them"
      }      
   }      

   // (C) manage current and last mode and last drawn angles
   if (!bShowDyn)  // was there any reason not to show the dyn. guidelines?
      *pu32Mode &= ~DYN_GUIDELINE_DRAW_DYNAMIC;          // clear the according flag in the mode byte
   *pu32Mode &= ~DYN_GUIDELINE_FORCE_UPDATE;             // the decision whether or not a screen update is needed will be delivered back via the return value
   tBool bAnyModeChanged = (*pu32Mode != m_u32LastMode); // (remember: the current request may also contain changed flags other than that for dyn. guidelines)
   m_u32LastMode         = *pu32Mode;                    // keep in mind what we did (or did not) in this call
   
   if ((bShowDyn != bLastShowDyn) || bVisibleDynUpd ||   // change of dyn. guideline visibility, or any relevant angle change on still visible guidelines?
       (bShowDyn && bForcedUpdate)                   )   // or forced redraw even if angles changed only within hysteresis?
   {
      m_dLastUsedFrontAng = *pdFrontWheelAngleDeg;       // keep in mind the last angle values which lead to showing, hiding, updating or redrawing the dyn. guidelines
      m_dLastUsedRearAng  = *pdRearWheelAngleDeg;
   }

   return (bAnyModeChanged || bVisibleDynUpd || bForcedUpdate);
}


tVoid vd_rvc_tclGraphics_DynGuideline::vCalcTrackGeometry(tstDynLineTrackGeometry *pstGeom, tDouble dFrontWheelAngle, tDouble dRearWheelAngle) const  // angle inputs are in degrees
{
   tDouble dHalfVehWidth = m_tVehParams.dVehWidth / 2.0;
   
   // convert input front and rear wheel angles to radians (note: positive value of input angles always mean a clockwise (i.e. right) turn of the wheel; ensured by caller)
   dFrontWheelAngle = DEG2RAD(dFrontWheelAngle);
   dRearWheelAngle  = DEG2RAD(dRearWheelAngle);

   // ensure that the input angle values don't lead to mathematical problems 
   tS8     s8TurnDir        = (dFrontWheelAngle<0) ? -1 : +1;
   tDouble dWheelAngleDiff  = dFrontWheelAngle - dRearWheelAngle;
   if (fabs(dWheelAngleDiff) < 1E-5)  // we will divide by sin(dWheelAngleDiff) below, so we need prevent dWheelAngleDiff from being ==0.
                                     // Note: dWheelAngleDiff==0 (i.e. FWA==RWA) corresponds to front and rear wheels travelling on parallel lines, so except the case that RWA==FWA==0,
                                     // this may be seen as a somewhat "pathological" case for today's steering mechanics, where RWA is a small (positive or negative) fraction of FWA.
                                     // But maybe tomorrow, RWA will be more or less independent of FWA (e.g. to allow "sideward" vehicle movement for parking in small gaps),
                                     // and then this check will be relevant. At any rate, even today it doesn't hurt, since we have to deal with the case RWA==FWA==0 anyway.
   {
      dWheelAngleDiff = (dWheelAngleDiff>0) ? +1E-5 : -1E-5;  // set the angle difference to a very small fixed value (which will lead to a LAAARGE turn circle radius, which in turn
      dFrontWheelAngle = dRearWheelAngle + dWheelAngleDiff;   // will lead to curves being undistinguishable from parallel lines), and adjust the front wheel angle accordingly
   }
   
   // compute center coordinates of turn circle (relative to the coordinates of the "angle reference side rear wheel")
   tDouble dRearWheelTurnRadius = m_tVehParams.dWheelBase * cos(dFrontWheelAngle) / sin(dWheelAngleDiff);  // special case RWA=0: => r = wheelbase*cos(FWA)/sin(FWA-0) = wheelbase/tan(FWA), ok
   tDouble dXcen = - dRearWheelTurnRadius * cos(dRearWheelAngle);
   tDouble dYcen = + dRearWheelTurnRadius * sin(dRearWheelAngle);
   // so far, dXcen is relative to the "angle reference wheel side" => now make it absolute (i.e. relative to rear axle center)
   if (TIRE_ANGLE_REFERS_TO_OUTER_TIRE == m_tVehParams.u8TireAngleRefPoint)
   {
      dXcen += s8TurnDir * dHalfVehWidth;
   }
   else if (TIRE_ANGLE_REFERS_TO_INNER_TIRE == m_tVehParams.u8TireAngleRefPoint)
   {
      dXcen -= s8TurnDir * dHalfVehWidth;
   }
   else
   {
      // else it already refers to the axle center => nothing more to do
   }
   // finally "stretch" dXcen by the radius tuning factor (i.e. shift turn circle center away from the vehicle)
   dXcen *= m_tVehParams.dRadiusTuningFact;

   // compute radius of the effective (i.e. "tune-factored") final turn circle, measured from the turn circle center to the center of the rear axle
   tDouble dRturn = sqrt(dXcen*dXcen + dYcen*dYcen);
   if (dXcen > 0)  // due to the sqrt(), we lost the sign information dRearWheelTurnRadius initially had, so we need to restore it (using dXcen):
   {
      dRturn = - dRturn;  // dRturn must be "directed" from (xcen,ycen) towards the vehicle, i.e. must be >0 if xcen<0 and vice versa
   }

   // now compute the arc angle the curve will cover (in radians):
   tDouble dArcAngle = (m_tVehParams.dGuidelnLenDyn + m_tVehParams.dRearOverhang) / dRturn;  // may be >0 or <0, depending on whether it's a left- or right-bent curve
     // note: dRearOverhang is a pretty good approximation of the true arc length between the rear axle center and the vehicle rear end;
     //       for "extreme" 4WS with rear wheel angle > approx. 20, dRearOverhang/cos(dRearWheelAngle) should be used
   if      (dArcAngle > +DYN_GUIDELINE_MAX_ARC_ANGLE_RAD) // limit it to the max. length we want to plot
   {
      dArcAngle =       +DYN_GUIDELINE_MAX_ARC_ANGLE_RAD;
   }
   else if (dArcAngle < -DYN_GUIDELINE_MAX_ARC_ANGLE_RAD)
   {
      dArcAngle =       -DYN_GUIDELINE_MAX_ARC_ANGLE_RAD;
   }
   
   // compute the start angle of the guideline to be drawn, as seen from the turn circle center:
   tDouble dPhiSta = atan(dYcen / dXcen);   
   // so far, dYcen was relative to the rear axle => now, after the above line, finally we can make it absolute (i.e. relative to the vehicle rear end):
   dYcen -= m_tVehParams.dRearOverhang;
   
   // now compute the difference of the radii of the circle on which the rear axle center travels and the left (or right) guideline
   tDouble dDeltaRturn  = m_tVehParams.dGuidelnOffs * cos(dRearWheelAngle);  // i.e. outer/inner curve radius = axle_center_radius +/- (vehicle_width/2 + safety_margin)*cos(RWA)
   // ... and the "delta vector" from the turn circle center to the start of the left guideline curve:
   tDouble dDxLeft      = (dRturn + dDeltaRturn) * cos(dPhiSta);
   tDouble dDyLeft      = (dRturn + dDeltaRturn) * sin(dPhiSta);
   // ... and the scaling factor between the "delta vectors" for the left and right guideline (i.e. the ratio of their radii):
   tDouble dFactLft2Rgt = (dRturn - dDeltaRturn) / (dRturn + dDeltaRturn);  // remember: dRturn may be >0 or <0 (depending on right or left turn), so this factor f may be <1 or >1,
                                                                            //           which carries the information whether the left curve is the outer (f<1) or the inner (f>1) one
   // compute the sine and cosine of the angle by which the "delta vector" gets turned around the turn circle center in each angle step:
   tDouble dDphi    = dArcAngle / DYN_GUIDELINE_NUM_ANG_STEPS;
   tDouble dSinDphi = sin(dDphi);
   tDouble dCosDphi = cos(dDphi);
   
   // finally package all relevant values into the output struct:
   pstGeom->oPCen        = vd_rvc_t2Vector(dXcen, dYcen);
   pstGeom->oVecDxy      = vd_rvc_t2Vector(dDxLeft, dDyLeft);
   pstGeom->dFactLft2Rgt = dFactLft2Rgt;
   pstGeom->oMAngStep    = vd_rvc_tcl2x2Matrix(dCosDphi,-dSinDphi,  dSinDphi,dCosDphi);
}


tVoid vd_rvc_tclGraphics_DynGuideline::vDrawDynamicLines(tDouble dFrontWheelAngleDeg, tDouble dRearWheelAngleDeg) const
{
   tstDynLineTrackGeometry stGeom;
   vCalcTrackGeometry( & stGeom, dFrontWheelAngleDeg, dRearWheelAngleDeg);

   // for each of the left and right guideline, create 2 point objects (for line segment start end end) plus pointers to them
   vd_rvc_tclCamPoint  oP1Lft, oP2Lft,   oP1Rgt, oP2Rgt;
   vd_rvc_tclCamPoint *poPStaLft = & oP1Lft,  *poPEndLft = & oP2Lft;
   vd_rvc_tclCamPoint *poPStaRgt = & oP1Rgt,  *poPEndRgt = & oP2Rgt;
   tFloat fLwidth;
   
   m_oLinePlotter.vSetColor(DYN_GUIDELINE_CURVE_RGB(DYN_GUIDELINE_COLORIDX_DYN));

   // first draw the two side lines

   #ifdef DYN_GUIDELINE_USE_ADAPTIVE_DYN_STEPSIZE
   // squared minimum line segment length to be plotted (in pixel), i.e. the minimum allowed (dy^2+dy^2) of theline start and end pixel coordinates
   // if a line segment is shorter than this, its start point will be kept and used together with the end point of the next computed line segment
   // in order to draw a longer line. This is done to "mask" frequent "wrinkles" in the lines due to pixel coordinates being int's
   tU32  u32MinLen2 = (DYN_GUIDELINE_MIN_DYNLINE_PLOTLEN)*(DYN_GUIDELINE_MIN_DYNLINE_PLOTLEN);
   #else
   tU32  u32MinLen2 = 0;
   #endif
   tDouble dYcropLft = vd_rvc_tclGraphics_GuidelineCalib::vGetLineCropDistance(+1.0);
   tDouble dYcropRgt = vd_rvc_tclGraphics_GuidelineCalib::vGetLineCropDistance(-1.0);
   
   // use the rotated "delta" vector from the steer circle's center to the starting point of the left line to setup the segment start point of the left track:
   poPStaLft->vSetGroundCoords(stGeom.oPCen + stGeom.oVecDxy                    , DYN_GUIDELINE_TUNE_SIDE_LEFT);
   // per construction, the "delta" vector for the right track is just a scaled version of the one for the left track:
   poPStaRgt->vSetGroundCoords(stGeom.oPCen + stGeom.oVecDxy*stGeom.dFactLft2Rgt, DYN_GUIDELINE_TUNE_SIDE_RIGHT);

   tBool bFirstVisLft=true, bFirstVisRgt=true, bCurrEndIsNextStart;
   for (tU32 u32Idx=1; u32Idx<=DYN_GUIDELINE_NUM_ANG_STEPS; u32Idx++)
   {
      // rotate the "delta" vector by one angular step:
      stGeom.oVecDxy = stGeom.oMAngStep * stGeom.oVecDxy;

      // setup the end points of the current segments:
      poPEndLft->vSetGroundCoords(stGeom.oPCen + stGeom.oVecDxy                    , DYN_GUIDELINE_TUNE_SIDE_LEFT);
      poPEndRgt->vSetGroundCoords(stGeom.oPCen + stGeom.oVecDxy*stGeom.dFactLft2Rgt, DYN_GUIDELINE_TUNE_SIDE_RIGHT);
      
      if (DYN_GUIDELINE_NUM_ANG_STEPS == u32Idx)  // last point of the curve?
         u32MinLen2 = 0;                          // ==> force it not to be skipped because of being too close to the previous line end point
      
      // start plotting only after the line has reached/exceeded the minimum distance from the vehicle's rear, and chop off if needed:
      bCurrEndIsNextStart = true;  // assumption to begin with
      if ( poPStaLft->bEnsureMinimumYgroundOnWholePathTo(poPEndLft, dYcropLft) )
      {
         fLwidth = fCalcLineWidth(DYN_GUIDELINE_LWIDTH_IDX_DYN, poPStaLft->m_tGroundCoo.y(), poPEndLft->m_tGroundCoo.y());
         if ( (bCurrEndIsNextStart = bDrawConnectingLine(poPStaLft, poPEndLft, fLwidth, u32MinLen2, (tBool)!bFirstVisLft, TRUE)) == TRUE )
              // "!bFirstVisLft": the first visible segment must never get a "head" (in order to keep the chopping at the near end exact, see above), all later ones may;
              // "true": all visible segments may get a "tail" (which is located at the segment's far end)
            bFirstVisLft = false;  // else line segment was not plotted because it was too short => keep current starting point
      }
      if (bCurrEndIsNextStart)
         SWAP(vd_rvc_tclCamPoint*, poPStaLft,poPEndLft) // make current segment's end point the starting point of the next segment
      
      bCurrEndIsNextStart = true;
      if ( poPStaRgt->bEnsureMinimumYgroundOnWholePathTo(poPEndRgt, dYcropRgt) )
      {
         fLwidth = fCalcLineWidth(DYN_GUIDELINE_LWIDTH_IDX_DYN, poPStaRgt->m_tGroundCoo.y(), poPEndRgt->m_tGroundCoo.y());
         if ((bCurrEndIsNextStart = bDrawConnectingLine(poPStaRgt, poPEndRgt, fLwidth, u32MinLen2, (tBool)!bFirstVisRgt, TRUE)) == TRUE )
            bFirstVisRgt = false;         
      }
      if (bCurrEndIsNextStart)
         SWAP(vd_rvc_tclCamPoint*, poPStaRgt,poPEndRgt)
   }

   #if (DYN_GUIDELINE_NUM_END_LINE_STEPS) > 0  // dynamic lines lateral end bar not disabled completely?
   if (m_tVehParams.bConnectDynGlEnds)
   {
     // after the last swap inside the for() loop, poPStaLft and poPStaRgt are the end points of the left and right track;
     // these two points we need to connect now:
     vd_rvc_t3Vector oStep = (poPStaRgt->m_tGroundCoo - poPStaLft->m_tGroundCoo) * (1.0/DYN_GUIDELINE_NUM_END_LINE_STEPS);
     for(tU32 u32Idx=1; u32Idx<=(DYN_GUIDELINE_NUM_END_LINE_STEPS); u32Idx++)
     {
        poPEndLft->vSetGroundCoords(poPStaLft->m_tGroundCoo + oStep, DYN_GUIDELINE_TUNE_SIDE_NONE);  // no more tuning here: start and end point (and with that also the oStep) are alreay tuned
        fLwidth = fCalcLineWidth(DYN_GUIDELINE_LWIDTH_IDX_DYN, poPStaLft->m_tGroundCoo.y(), poPEndLft->m_tGroundCoo.y());
        (tVoid)bDrawConnectingLine(poPStaLft, poPEndLft, fLwidth, 0, (tBool)(1<u32Idx), (tBool)(DYN_GUIDELINE_NUM_END_LINE_STEPS>u32Idx) );
           // "(1<u32Idx), (DYN_GUIDELINE_NUM_END_LINE_STEPS>u32Idx)": first segment must never get a "head", last segment must never get a "tail"; all "inner" segments may have both
        SWAP(vd_rvc_tclCamPoint*, poPStaLft,poPEndLft)
     }
   }
   #endif
}


tVoid vd_rvc_tclGraphics_DynGuideline::vDrawStaticLines(tU32 u32Mode) const
{
   switch (m_tVehParams.u8GuidelnStyle)
   {
      case DYN_GUIDELINE_STYLE_NISSAN:
      case DYN_GUIDELINE_STYLE_NISSAN_US:
      case DYN_GUIDELINE_STYLE_MITSUBISHI:
         if (m_bTempCalibStyleActive)
            vDrawStaticLines_NissanCalib(u32Mode);
         else
            vDrawStaticLines_Nissan(u32Mode);
         break;
         
      case DYN_GUIDELINE_STYLE_RENAULT:
         vDrawStaticLines_Renault(u32Mode);      
         break;
         
      default:
         break;
   }
}


tVoid vd_rvc_tclGraphics_DynGuideline::vDrawStaticLines_Nissan(tU32 u32Mode) const
{
   struct tstSideLineSeg {
      tU16 u16DistSta, u16DistEnd,   u16DistHor, u16LenHor;
      tU8  u8ColorIdx;
   };

   const tstSideLineSeg* pstLineSegs = NULL; // a pointer to the line segment specifiers to be used ...
   tU32                  u32NumLineSegs = 0; // ... and their number

   if (DYN_GUIDELINE_STYLE_NISSAN_US != m_tVehParams.u8GuidelnStyle)
   {
       static const tstSideLineSeg aLineSegs_gen[] =   // define the set of line segment specifiers to be used in general (i.e. for ROW)
       {
          { 40, 60,   50, 25,   DYN_GUIDELINE_COLORIDX_STAT_NEAR},
          { 70, 80,   0,  0,    DYN_GUIDELINE_COLORIDX_STAT_NEAR},
          { 90,120,   100,28,   DYN_GUIDELINE_COLORIDX_STAT_MID },
          {140,160,   0,  0,    DYN_GUIDELINE_COLORIDX_STAT_MID },
          {180,240,   200,32,   DYN_GUIDELINE_COLORIDX_STAT_FAR },
          {260,370,   300,34,   DYN_GUIDELINE_COLORIDX_STAT_FAR },
       };
       pstLineSegs    = & aLineSegs_gen[0];
       u32NumLineSegs = NUM_ARRAY_ELEMENTS(aLineSegs_gen);
   }
   else
   {
       static const tstSideLineSeg aLineSegs_US[] =    // define the alternate set of line segment specifiers to be used for US/CAN
       {
          { 40, 60,   50, 25,   DYN_GUIDELINE_COLORIDX_STAT_NEAR},
          { 70, 80,   0,  0,    DYN_GUIDELINE_COLORIDX_STAT_MID },
          { 90,120,   100,28,   DYN_GUIDELINE_COLORIDX_STAT_MID },
          {140,160,   0,  0,    DYN_GUIDELINE_COLORIDX_STAT_FAR },
          {180,200,   200,32,   DYN_GUIDELINE_COLORIDX_STAT_FAR },
       };
       pstLineSegs    = & aLineSegs_US[0];
       u32NumLineSegs = NUM_ARRAY_ELEMENTS(aLineSegs_US);
   }
   
   vd_rvc_tclCamPoint oP1, oP2;
   vd_rvc_tclCamPoint *poPSta = & oP1, *poPEnd = & oP2;
   tFloat fLwidth;
   
   if (u32Mode & DYN_GUIDELINE_DRAW_STATIC_SIDE)
   {
      const tstSideLineSeg *pstSeg = pstLineSegs;
      for (tU32 u32Idx=0; u32Idx<u32NumLineSegs; u32Idx++, pstSeg++)
      {
         m_oLinePlotter.vSetColor(DYN_GUIDELINE_CURVE_RGB(pstSeg->u8ColorIdx));
         
         tBool bDoTweak = (DYN_GUIDELINE_STYLE_NISSAN_US==m_tVehParams.u8GuidelnStyle && u32NumLineSegs-1==u32Idx && 0<pstSeg->u16LenHor) ? TRUE : FALSE;
         
         // draw horizontal dashes if required
         if (0 < pstSeg->u16LenHor)
         {
            tDouble dDistHLine = (tDouble)(pstSeg->u16DistHor);
            fLwidth = fCalcLineWidth(DYN_GUIDELINE_LWIDTH_IDX_STAT, dDistHLine,dDistHLine);
            
            // a "tweak" for better optical appearance of the intersection of the last horizontal dash and the side line in the US style:
            // shift the outer ends of the last horizontal dashes outwards by an "empirical" fraction of the line width of the last side line segment
            tFloat fDxTweak = bDoTweak ? (0.3F * fLwidth*m_tVehParams.fLineWdthWgtVertLines) : 0.0F;
         
            poPSta->vSetGroundCoords(+(m_tVehParams.dGuidelnOffs + fDxTweak         ), dDistHLine, 0.0, DYN_GUIDELINE_TUNE_SIDE_LEFT);   // outer dash end on left side line
            poPEnd->vSetGroundCoords(+(m_tVehParams.dGuidelnOffs - pstSeg->u16LenHor), dDistHLine, 0.0, DYN_GUIDELINE_TUNE_SIDE_LEFT);   // inner dash end
            (tVoid)bDrawConnectingLine(poPSta, poPEnd, fLwidth, 0, FALSE, FALSE);
            
            poPSta->vSetGroundCoords(-(m_tVehParams.dGuidelnOffs + fDxTweak         ), dDistHLine, 0.0, DYN_GUIDELINE_TUNE_SIDE_RIGHT);  // outer dash end on right side line
            poPEnd->vSetGroundCoords(-(m_tVehParams.dGuidelnOffs - pstSeg->u16LenHor), dDistHLine, 0.0, DYN_GUIDELINE_TUNE_SIDE_RIGHT);  // inner dash end
            (tVoid)bDrawConnectingLine(poPSta, poPEnd, fLwidth, 0, FALSE, FALSE);
         }
         
         // draw "vertical" side line segment
         tDouble dDistSta = (tDouble)(pstSeg->u16DistSta);
         tDouble dDistEnd = (tDouble)(pstSeg->u16DistEnd);
         fLwidth = fCalcLineWidth(DYN_GUIDELINE_LWIDTH_IDX_STAT, dDistSta, dDistEnd) * m_tVehParams.fLineWdthWgtVertLines;

         // a "tweak" for better optical appearance of the intersection of the last horizontal dash and the side line in the US style:
         // shift the farther end of the last side line segment farther away by an "empirical" fraction of the line width of the last horizontal dash
         tFloat fDyTweak = bDoTweak ? (0.6F * fLwidth) : 0.0F;
         fLwidth *= m_tVehParams.fLineWdthWgtVertLines;
         
         poPSta->vSetGroundCoords(+m_tVehParams.dGuidelnOffs, (dDistSta           ), 0.0, DYN_GUIDELINE_TUNE_SIDE_LEFT);   // near end of left segment
         poPEnd->vSetGroundCoords(+m_tVehParams.dGuidelnOffs, (dDistEnd + fDyTweak), 0.0, DYN_GUIDELINE_TUNE_SIDE_LEFT);   // far end
         (tVoid)bDrawConnectingLine(poPSta, poPEnd, fLwidth, 0, FALSE, FALSE);       
         
         poPSta->vSetGroundCoords(-m_tVehParams.dGuidelnOffs, (dDistSta           ), 0.0, DYN_GUIDELINE_TUNE_SIDE_RIGHT);  // near end of right segment
         poPEnd->vSetGroundCoords(-m_tVehParams.dGuidelnOffs, (dDistEnd + fDyTweak), 0.0, DYN_GUIDELINE_TUNE_SIDE_RIGHT);  // far end
         (tVoid)bDrawConnectingLine(poPSta, poPEnd, fLwidth, 0, FALSE, FALSE);            
      }
   }

   #if (DYN_GUIDELINE_NUM_CENT_LINE_STEPS) > 0   
   if (u32Mode & DYN_GUIDELINE_DRAW_STATIC_CENT)
   {
      tDouble dYSta  = (tDouble)(pstLineSegs[0].u16DistSta);  // start/end distance of center line == start/end distance of side lines
      tDouble dYStep = (pstLineSegs[u32NumLineSegs-1].u16DistEnd - dYSta) / (DYN_GUIDELINE_NUM_CENT_LINE_STEPS);
      
      m_oLinePlotter.vSetColor(DYN_GUIDELINE_CURVE_RGB(DYN_GUIDELINE_COLORIDX_STAT_CENT));
      
      poPSta->vSetGroundCoords(0.0, dYSta, 0.0, DYN_GUIDELINE_TUNE_SIDE_CENT);
      for (tU32 u32Idx=1; u32Idx<=(DYN_GUIDELINE_NUM_CENT_LINE_STEPS); u32Idx++)
      {
         poPEnd->vSetGroundCoords(0.0, dYSta+=dYStep, 0.0, DYN_GUIDELINE_TUNE_SIDE_CENT);
         fLwidth = fCalcLineWidth(DYN_GUIDELINE_LWIDTH_IDX_CENT, poPSta->m_tGroundCoo.y(), poPEnd->m_tGroundCoo.y());
         (tVoid)bDrawConnectingLine(poPSta, poPEnd, fLwidth, 0, FALSE, FALSE);
         SWAP(vd_rvc_tclCamPoint*, poPSta,poPEnd)
      }
   }
   #endif   
}


tVoid vd_rvc_tclGraphics_DynGuideline::vDrawStaticLines_NissanCalib(tU32 u32Mode) const
{
   struct tstCalibMark {
      tU16 u16LongPos, u8LatPos, u8LatLen, u8LongLen;
      tU8  u8ColorIdx;
   };
#if(0)  // initial guess
   static const tstCalibMark aCalibMarks[] = 
   {
      { 50, 100, 20,20,   DYN_GUIDELINE_COLORIDX_STAT_FAR },
      {150, 100, 20,20,   DYN_GUIDELINE_COLORIDX_STAT_FAR },
      {150,  60, 20, 0,   DYN_GUIDELINE_COLORIDX_STAT_NEAR},
      {170,   0, 20, 0,   DYN_GUIDELINE_COLORIDX_STAT_NEAR},
   };
   tDouble dGuidelnOffs = m_tVehParams.dGuidelnOffs;
#else  // resembles the sketches in the spec more than the above
   static const tstCalibMark aCalibMarks[] = 
   {
      { 50, 100, 15, 10,  DYN_GUIDELINE_COLORIDX_STAT_FAR },
      {100, 100, 15, 20,  DYN_GUIDELINE_COLORIDX_STAT_FAR },
      {100,  70, 10,  0,  DYN_GUIDELINE_COLORIDX_STAT_NEAR},
      {110,   0, 10,  0,  DYN_GUIDELINE_COLORIDX_STAT_NEAR},
   };
   tDouble dGuidelnOffs = m_tVehParams.dVehWidth/2.0 + 20.0;
#endif
   
   tU32 u32NumLineSegs = NUM_ARRAY_ELEMENTS(aCalibMarks);
   
   vd_rvc_tclCamPoint oPSta, oPEnd;
   tDouble dFracLatPos, dXcen, dYcen, dXsta, dXend, dYsta, dYend;
   tFloat  fLwidth;
   
   if (u32Mode & DYN_GUIDELINE_DRAW_STATIC_SIDE)
   {
      const tstCalibMark *pstMark = & aCalibMarks[0];
      for (tU32 u32SegIdx=0; u32SegIdx<u32NumLineSegs; u32SegIdx++, pstMark++)  // loop over segments
      {
         m_oLinePlotter.vSetColor(DYN_GUIDELINE_CURVE_RGB(pstMark->u8ColorIdx));

         dFracLatPos = (tDouble)(pstMark->u8LatPos) / 100;  // u8LatPos is the percentage of the full guideline separation
         dYcen       = pstMark->u16LongPos;
         dXcen       = dGuidelnOffs * dFracLatPos;
         dXsta       = dXcen - pstMark->u8LatLen;
         dXend       = dXcen + pstMark->u8LatLen;
         dYsta       = dYcen - pstMark->u8LongLen;
         dYend       = dYcen + pstMark->u8LongLen;
         fLwidth     = 1.5F * fCalcLineWidth(DYN_GUIDELINE_LWIDTH_IDX_STAT, dYcen,dYcen);           
         
         oPSta.vSetGroundCoords(+dXsta, dYcen, 0.0, +dFracLatPos);  // draw left lateral line segment
         oPEnd.vSetGroundCoords(+dXend, dYcen, 0.0, +dFracLatPos);
         (tVoid)bDrawConnectingLine(&oPSta, &oPEnd, fLwidth, 0, FALSE, FALSE);
         if (0 < pstMark->u8LongLen)  // draw left longitudinal line segment, if required
         {
            oPSta.vSetGroundCoords(+dXcen, dYsta, 0.0, +dFracLatPos);
            oPEnd.vSetGroundCoords(+dXcen, dYend, 0.0, +dFracLatPos);
            (tVoid)bDrawConnectingLine(&oPSta, &oPEnd, fLwidth, 0, FALSE, FALSE);
         }
         
         if (0 != pstMark->u8LatPos)  // not a single calibration mark on the center line?
         {
            oPSta.vSetGroundCoords(-dXsta, dYcen, 0.0, -dFracLatPos);  // draw right lateral line segment
            oPEnd.vSetGroundCoords(-dXend, dYcen, 0.0, -dFracLatPos);
            (tVoid)bDrawConnectingLine(&oPSta, &oPEnd, fLwidth, 0, FALSE, FALSE);
            if (0 < pstMark->u8LongLen)  // draw right longitudinal line segment, if required
            {
               oPSta.vSetGroundCoords(-dXcen, dYsta, 0.0, -dFracLatPos);
               oPEnd.vSetGroundCoords(-dXcen, dYend, 0.0, -dFracLatPos);
               (tVoid)bDrawConnectingLine(&oPSta, &oPEnd, fLwidth, 0, FALSE, FALSE);
            }
         }
      }
   }
}


tVoid vd_rvc_tclGraphics_DynGuideline::vDrawStaticLines_Renault(tU32 u32Mode) const
{
   if (! (u32Mode & DYN_GUIDELINE_DRAW_STATIC_SIDE) )
      return;

   // definition of the side line sections
   struct tstSideLineSeg {
      tU8 u8NumSteps;
      tU8 u8ColorIdx;
   };
   static const tstSideLineSeg aLineSegs[] = 
   {
      { 3, DYN_GUIDELINE_COLORIDX_STAT_NEAR },
      { 3, DYN_GUIDELINE_COLORIDX_STAT_MID  },
      { 3, DYN_GUIDELINE_COLORIDX_STAT_FAR  },
      { 3, DYN_GUIDELINE_COLORIDX_STAT_FAR  },
   };
   tU32 u32NumLineSegs = NUM_ARRAY_ELEMENTS(aLineSegs);
   
   // definition of the horizontal lines/bars
   struct tstHorLine {
      tBool  bFullWidth;
      tU8    u8ColorIdx;
   };
   static const tstHorLine aHorLines[] = 
   {
      { true,  DYN_GUIDELINE_COLORIDX_STAT_NEAR },
      { false, DYN_GUIDELINE_COLORIDX_STAT_MID  },
      { false, DYN_GUIDELINE_COLORIDX_STAT_FAR  },
   };
   tU32 u32NumHorLines = NUM_ARRAY_ELEMENTS(aHorLines);
   
   // draw longitudinal side line segments
   
   // for each of the left and right guideline, create 2 point objects (for line segment start end end) plus pointers to them
   vd_rvc_tclCamPoint  oP1Lft, oP2Lft,   oP1Rgt, oP2Rgt;
   vd_rvc_tclCamPoint *poPStaLft = & oP1Lft,  *poPEndLft = & oP2Lft;
   vd_rvc_tclCamPoint *poPStaRgt = & oP1Rgt,  *poPEndRgt = & oP2Rgt;
   tFloat fLwidth;

   tDouble dYsta  = 0.0, dYend = 0.0;
   tDouble dYcropLft = vd_rvc_tclGraphics_GuidelineCalib::vGetLineCropDistance(+1.0);
   tDouble dYcropRgt = vd_rvc_tclGraphics_GuidelineCalib::vGetLineCropDistance(-1.0);
   
   poPStaLft->vSetGroundCoords(+m_tVehParams.dGuidelnOffs, dYsta, 0.0, DYN_GUIDELINE_TUNE_SIDE_LEFT);   // near end of first segment of left line
   poPStaRgt->vSetGroundCoords(-m_tVehParams.dGuidelnOffs, dYsta, 0.0, DYN_GUIDELINE_TUNE_SIDE_RIGHT);  // near end of first segment of right line
   
   tBool bFirstVisLft=true, bFirstVisRgt=true;
   for (tU32 u32Seg=0; u32Seg<u32NumLineSegs; u32Seg++)
   {
      m_oLinePlotter.vSetColor(DYN_GUIDELINE_CURVE_RGB(aLineSegs[u32Seg].u8ColorIdx));
      
      tU32    u32NumSteps = aLineSegs[u32Seg].u8NumSteps;      // number of drawing segments in this section
      tDouble dYstep      = (m_tVehParams.adGuidelnStatDist[u32Seg] - dYend) / u32NumSteps;

      for (tU32 u32Stp=1; u32Stp<=u32NumSteps; u32Stp++)
      {
         dYsta  = dYend;
         dYend += dYstep;
         fLwidth = fCalcLineWidth(DYN_GUIDELINE_LWIDTH_IDX_STAT, dYsta, dYend);  // and also the line widths of the segments are the same
         
         // a "tweak" for better optical appearance of the places where lateral and horizontal lines meet
         if ( u32Stp == u32NumSteps )  // last line segment of one color section?
         {
            // shift the line segment end by half the line width of the horizontal dash/bar which ends here, multiplied my some "heuristical" tweak factor for the closet one
            tDouble dTweakFact = 1.0;  // for the color sections further away from the car, this is quite ok
            if ( 0 == u32Seg )         // special case: first color section - this one (1) is very close, and (2) may have a greater line width => both effects cause bigger "gaps"
            {
               tDouble dVehGeomFact = (m_tVehParams.dCamZOffs/120.0) * (45.0/m_tVehParams.dCamPitch);  // camera height and pitch angle affect the appearent size of the "gaps"
               if ( dVehGeomFact < 0.9 )   // note that the "scale factors" 120cm, 45 and 0.9 were determined "by eye" using the HTML guideline tool
                  dVehGeomFact = 0.9;
               dTweakFact = (m_tVehParams.fLineWdthWgtNearHbar/2.0) * dVehGeomFact;  // also, the line width weight of the first segment has some influence
            }
            dYend += 0.5 * fLwidth * dTweakFact;
         }
         
         poPEndLft->vSetGroundCoords(+m_tVehParams.dGuidelnOffs, dYend, 0.0, DYN_GUIDELINE_TUNE_SIDE_LEFT);  // far end of current segment         
         if ( poPStaLft->bEnsureMinimumYgroundOnWholePathTo(poPEndLft, dYcropLft) )
         {
            (tVoid)bDrawConnectingLine(poPStaLft, poPEndLft, fLwidth, 0, (tBool)!bFirstVisLft, TRUE);
                 // "!bFirstVisLft": the first visible segment must never get a "head" (in order to keep the chopping at the near end exact, see above), all later ones may;
                 // "true": all visible segments may get a "tail" (which is located at the segment's far end)
            bFirstVisLft = false;
         } 

         poPEndRgt->vSetGroundCoords(-m_tVehParams.dGuidelnOffs, dYend, 0.0, DYN_GUIDELINE_TUNE_SIDE_RIGHT);
         if ( poPStaRgt->bEnsureMinimumYgroundOnWholePathTo(poPEndRgt, dYcropRgt) )
         {
            (tVoid)bDrawConnectingLine(poPStaRgt, poPEndRgt, fLwidth, 0, (tBool)!bFirstVisRgt, TRUE);
            bFirstVisRgt = false;
         } 
         
         SWAP(vd_rvc_tclCamPoint*, poPStaLft,poPEndLft)  // end point of current segment = start point of next segment
         SWAP(vd_rvc_tclCamPoint*, poPStaRgt,poPEndRgt)  // end point of current segment = start point of next segment
      }
   }
   
   // draw the horizontal bars and dashes
   
   for (tU32 u32Idx=0; u32Idx<u32NumHorLines; u32Idx++)
   {
      tDouble dDist       = m_tVehParams.adGuidelnStatDist[u32Idx];
      tFloat fLineWdthWgt = (0==u32Idx) ? m_tVehParams.fLineWdthWgtNearHbar : 1.0;  // only first color segment may get a larger weight

      m_oLinePlotter.vSetColor(DYN_GUIDELINE_CURVE_RGB(aHorLines[u32Idx].u8ColorIdx));
      fLwidth = fCalcLineWidth(DYN_GUIDELINE_LWIDTH_IDX_STAT, dDist, dDist) * fLineWdthWgt;
      
      tDouble dDashLen;
      tU32    u32NumSteps;
      if (aHorLines[u32Idx].bFullWidth)  // draw a full-width bar by drawing two "half bars" of half the full width
      {
         dDashLen    = m_tVehParams.dGuidelnOffs;
         u32NumSteps = (0 == u32Idx) ? (tU32)ceil(DYN_GUIDELINE_NUM_STAT_HOR_BAR_STEPS/2.0) : (tU32)ceil(dDashLen/30);  // 2nd option intended for future bars further away
      }
      else  // draw two short dashes on left and right line
      {
         dDashLen    = m_tVehParams.dGuidelnLenHorBar;
         u32NumSteps = (tU32)ceil(dDashLen/30);  // use 30cm-max segments
      }
      tDouble dWidthFraction = dDashLen / (2*m_tVehParams.dGuidelnOffs); // fraction of full width between left to right line which is covered by the dash or "half bar"
      
      poPStaLft->vSetGroundCoords(+m_tVehParams.dGuidelnOffs, dDist, 0.0, DYN_GUIDELINE_TUNE_SIDE_LEFT);   // tuned position of outer end of left "dash"
      poPStaRgt->vSetGroundCoords(-m_tVehParams.dGuidelnOffs, dDist, 0.0, DYN_GUIDELINE_TUNE_SIDE_RIGHT);  // tuned position of outer end of right "dash"
      vd_rvc_t3Vector oTunedDelta = (poPStaLft->m_tGroundCoo - poPStaRgt->m_tGroundCoo) * (dWidthFraction / u32NumSteps);  // tuned step vector
      
      for (tU32 u32Step=0; u32Step<u32NumSteps; u32Step++)
      {
         poPEndLft->vSetGroundCoords(poPStaLft->m_tGroundCoo - oTunedDelta, DYN_GUIDELINE_TUNE_SIDE_NONE);  // moving inward from outer left side => - oTunedDelta
         dYcropLft = vd_rvc_tclGraphics_GuidelineCalib::vGetLineCropDistance(poPStaLft->m_tGroundCoo.x() / m_tVehParams.dGuidelnOffs);
         if ( poPStaLft->bEnsureMinimumYgroundOnWholePathTo(poPEndLft, dYcropLft) )
            (tVoid)bDrawConnectingLine(poPStaLft, poPEndLft, fLwidth, 0, FALSE, FALSE);
         SWAP(vd_rvc_tclCamPoint*, poPStaLft,poPEndLft)  // end point of current segment = start point of next segment         
         
         poPEndRgt->vSetGroundCoords(poPStaRgt->m_tGroundCoo + oTunedDelta, DYN_GUIDELINE_TUNE_SIDE_NONE);  // moving inward from outer left side => + oTunedDelta
         dYcropRgt = vd_rvc_tclGraphics_GuidelineCalib::vGetLineCropDistance(poPStaRgt->m_tGroundCoo.x() / m_tVehParams.dGuidelnOffs);
         if ( poPStaRgt->bEnsureMinimumYgroundOnWholePathTo(poPEndRgt, dYcropRgt) )
            (tVoid)bDrawConnectingLine(poPStaRgt, poPEndRgt, fLwidth, 0, FALSE, FALSE);               
         SWAP(vd_rvc_tclCamPoint*, poPStaRgt,poPEndRgt)
      }
   }      
}

tVoid vd_rvc_tclGraphics_DynGuideline::vDrawWarningIndicator(tVoid) const
{
#if (WARNING_IND_TYPE != WARNING_IND_TYPE_NONE)
   m_oLinePlotter.vSetColor(DYN_GUIDELINE_RGBCOLOR_WARN_IND);
   m_oLinePlotter.vSetAntiAliasing(FALSE);

   vd_rvc_t2IntVect pSta, pEnd;
   
   #if (WARNING_IND_TYPE == WARNING_IND_TYPE_EXCL_MARKS)
   
   // draw four "exclamation marks" in the corners of the active image area
   tS32   s32Length=64, s32Xoff=20, s32YoffTop=20, s32YoffBot=100; // all values in image pixels
   tFloat fLineWidth = 8.0f;
   
   for (tU32 u32Idx=0; u32Idx<2; u32Idx++)
   {
      tS32 s32Xcen = m_tImgParams.u32ImgXoffs + ((u32Idx==0) ? s32Xoff : (m_tImgParams.u32NumImgXpix-s32Xoff));
      for (tS32 u32Jdx=0; u32Jdx<2; u32Jdx++)
      {
         tS32 s32Ytop = m_tImgParams.u32ImgYoffs + ((u32Jdx==0) ? s32YoffTop : (m_tImgParams.u32NumImgYpix-s32Length-s32YoffBot));
         // stem of the exclamation mark (stem length = 11 sixteenth of the whole length)
         pSta.vSet(s32Xcen, s32Ytop                );
         pEnd.vSet(s32Xcen, s32Ytop+s32Length*11/16);
         m_oLinePlotter.vPlotLine(&pSta,&pEnd,fLineWidth);
         // dot of the exclamation mark (dot length = 2 sixteenth of the whole length, leaving a gap of 3 sixteenth between stem and dot) 
         pSta.vSet(s32Xcen, s32Ytop+s32Length*14/16);
         pEnd.vSet(s32Xcen, s32Ytop+s32Length      );
         m_oLinePlotter.vPlotLine(&pSta,&pEnd,fLineWidth);
      }
   }
   
   #elif (WARNING_IND_TYPE == WARNING_IND_TYPE_BIG_CROSS)
   
   // draw a big cross across the active image area
   tS32   s32Xoff=20, s32YoffTop=20, s32YoffBot=100; // all values in image pixels
   tFloat fLineWidth = 1.0f;
   
   // line left top to bottom right
   pSta.vSet((tS32)m_tImgParams.u32ImgXoffs+s32Xoff,                                  (tS32)m_tImgParams.u32ImgYoffs+s32YoffTop                                 );
   pEnd.vSet((tS32)m_tImgParams.u32ImgXoffs+(tS32)m_tImgParams.u32NumImgXpix-s32Xoff, (tS32)m_tImgParams.u32ImgYoffs+(tS32)m_tImgParams.u32NumImgYpix-s32YoffBot);
   m_oLinePlotter.vPlotLine(&pSta,&pEnd,fLineWidth);   
   // line from left bottom to right top
   pSta.vSet((tS32)m_tImgParams.u32ImgXoffs+s32Xoff,                                  (tS32)m_tImgParams.u32ImgYoffs+(tS32)m_tImgParams.u32NumImgYpix-s32YoffBot);
   pEnd.vSet((tS32)m_tImgParams.u32ImgXoffs+(tS32)m_tImgParams.u32NumImgXpix-s32Xoff, (tS32)m_tImgParams.u32ImgYoffs+s32YoffTop                                 );
   m_oLinePlotter.vPlotLine(&pSta,&pEnd,fLineWidth);
   
   #else
   #error "configured value of WARNING_IND_TYPE unsupported"
   #endif
#endif
}


tBool vd_rvc_tclGraphics_DynGuideline::bDrawConnectingLine(vd_rvc_tclCamPoint *poPSta, vd_rvc_tclCamPoint *poPEnd, tFloat fLineWidth,
                                                           tU32 u32MinLen2, tBool bAddHead, tBool bAddTail) const
{
   if (poPSta->bCalcPlotCooForPathTo(poPEnd))  // something remains to be plotted after all calculating and ground/camera clipping has been done?
   {
      
      tS32 s32Dx = poPEnd->m_tPlotCoo.x() - poPSta->m_tPlotCoo.x();
      tS32 s32Dy = poPEnd->m_tPlotCoo.y() - poPSta->m_tPlotCoo.y();
   
      if ( (tU32)(s32Dx*s32Dx + s32Dy*s32Dy) < u32MinLen2 )  // check the minimum-length constraint, if requested
         return false;

      if (bAddHead || bAddTail)   // check whether expanding the line at the ends is requested, if necessary
      {
         // this is an attempt to "fill" the ugly gaps which tend to appear in near-diagonal parts of the guidelines when using larger line widths
         
         // compute the head/tail length
         tS32 s32AddLen = (tS32) floor(fLineWidth/2.0);  // based on heuristics - good for line widths up to ~6
         if (s32AddLen > 2)
            s32AddLen = 2;
         
         // determine whether this line segment is near-diagonal
         tS32 s32AbsDx  = abs(s32Dx),  s32AbsDy = abs(s32Dy);
         if (5*s32AbsDx < 6*s32AbsDy && 6*s32AbsDx > 5*s32AbsDy )  // good integer-only approximation of "if (tan(alpha) == 40 ... 50)" 
//         if (5*s32AbsDx < 7*s32AbsDy && 7*s32AbsDx > 5*s32AbsDy )  // good approximation of "if (tan(alpha) == 35 ... 55)" , if this should be needed
         {
            tS32 s32DDx = (s32Dx<0) ? -s32AddLen : +s32AddLen;
            tS32 s32DDy = (s32Dy<0) ? -s32AddLen : +s32AddLen;
            if (bAddHead)
               poPSta->m_tPlotCoo.vSet(poPSta->m_tPlotCoo.x() - s32DDx, poPSta->m_tPlotCoo.y() - s32DDy);
            if (bAddTail)
               poPEnd->m_tPlotCoo.vSet(poPEnd->m_tPlotCoo.x() + s32DDx, poPEnd->m_tPlotCoo.y() + s32DDy);
         }
      }
         
      m_oLinePlotter.vPlotLine( & poPSta->m_tPlotCoo, & poPEnd->m_tPlotCoo, fLineWidth);
   }
   return true;
}

tFloat vd_rvc_tclGraphics_DynGuideline::fCalcLineWidth(tU32 u32LwidthIdx, tDouble dYsta, tDouble dYend) const
{
   vd_rvc_tstLineWidthDef *pLineWdthDef = & m_tVehParams.astLineWidth[u32LwidthIdx];
   
   tFloat fLwidth;
   if (0 == pLineWdthDef->s8Delta)  // in this case, the arithmetic below is superfluous - constant line width all the way
   {
      fLwidth = (tFloat) pLineWdthDef->u8Near;
   }
   else
   {
      fLwidth = (tFloat)(pLineWdthDef->u8Near + pLineWdthDef->s8Delta * ((dYsta - m_tVehParams.dGuidelnStartDist) / m_tVehParams.dDistScaleForLineWidthCalc));
      (void)dYend;    // currently, we use only dYsta for line width calculation; but other choices (such as computing the line width at the middle point (dYsta+dYend)/2  would be possible, too
      // (note: the "scale length" dDistScaleForLineWidthCalc gets computed only once in vConvertKdsToWorkVehicleParams())
   
      // now only make sure that we stay within the defined limits:
      if      (fLwidth < pLineWdthDef->u8LoLim)
         fLwidth = pLineWdthDef->u8LoLim;
      else if (fLwidth > pLineWdthDef->u8HiLim)
         fLwidth = pLineWdthDef->u8HiLim;
   }

   return fLwidth * m_tVehParams.fLineWidthScalingFact;
}
