/******************************************************************************/
/**
* \file    dispvidctrl_tclFsm_OnOff.cpp
* \ingroup
*
* \brief   state machine for OnOff handling
*
* \remark  Copyright : (c) 2015 Robert Bosch GmbH, Hildesheim
* \remark  Author    : Michael Niemann CM-AI/PJ-CB32
* \remark  Scope     : A-IVI
*
* \todo
*/
/******************************************************************************/
#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#include "dispvidctrl_AppMain.h"
#include "dispvidctrl_tclControl.h"
#include "dispvidctrl_tclFsm_OnOff.h"
#include "dispvidctrl_tclFsm_Hmi.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_APPLICATION
#include "trcGenProj/Header/dispvidctrl_tclFsm_OnOff.cpp.trc.h"
#endif

#define DISPVIDCTRL_TRACE_ID_FSM_ONOFF 0x01

#define ELEMENTE(array)       ((tU16)(sizeof(array)/sizeof(array[0])))

dispvidctrl_tclFsm_OnOff*   dispvidctrl_tclFsm_OnOff::m_poInstance  = OSAL_NULL;

/*******************************************************************************
*                                  STATE TABLE
*******************************************************************************/
dispvidctrl_tclFsm_OnOff::TFsmStateConf dispvidctrl_tclFsm_OnOff::aFsm_OnOff_StateTable[eState_Max] =
{
   {
      eState_OFF_EARLY,
      0,                                   //u32Timeout
      NULL,                                //tFsmStateEntryFunc
      vDo_OffEarly,                        //tFsmStateDoFunc
      NULL,                                //tFsmStateExitFunc
   },
   {
      eState_OFF,
      0,                                   //u32Timeout
      vEntry_Off,                          //tFsmStateEntryFunc
      vDo_Off,                             //tFsmStateDoFunc
      NULL,                                //tFsmStateExitFunc
   },
   {
      eState_ON,
      0,                                   //u32Timeout
      vEntry_On,                           //tFsmStateEntryFunc
      vDo_On,                              //tFsmStateDoFunc
      NULL,                                //tFsmStateExitFunc
   }
};

/*******************************************************************************
 *                             STATE TRANSITION TABLE
 *******************************************************************************/
dispvidctrl_tclFsm_OnOff::TStateTransitions dispvidctrl_tclFsm_OnOff::aFsm_OnOff_StateTransitionTable[] =
{
   // ===== eState_EARLY_OFF ============
   // ----- transition actions ----------
   {  100,                                // transitionId
      eState_OFF_EARLY,                   // current FSM State
      eTrigger_VIDEO_STATUS_RECEIVED,     // trigger update
      (eTrigger_START | eTrigger_APPLICATION_STATE_NORMAL | eTrigger_CRITICAL_VOLTAGE_END), // current active trigger(s)
      FSM_DONT_CARE,                      // current not active trigger(s)
      eState_ON                           // next FSM State
   },
   {  101,                                // transitionId
      eState_OFF_EARLY,                   // current FSM State
      eTrigger_VIDEO_STATUS_RECEIVED,     // trigger update
      eTrigger_START,                     // current active trigger(s)
      FSM_DONT_CARE,                      // current not active trigger(s)
      eState_OFF                          // next FSM State
   },
   // ---- do actions -------------------
   {  130,                                // transitionId
      eState_OFF_EARLY,                   // current FSM State
      eTrigger_START,                     // trigger update
      FSM_DONT_CARE,                      // current active trigger(s)
      FSM_DONT_CARE,                      // current not active trigger(s)
      eState_OFF_EARLY                    // next FSM State
   },
   {  131,                                // transitionId
      eState_OFF_EARLY,                   // current FSM State
      eTrigger_HMI_AVAILABLE_TRUE,        // trigger update
      FSM_DONT_CARE,                      // current active trigger(s)
      FSM_DONT_CARE,                      // current not active trigger(s)
      eState_OFF_EARLY                    // next FSM State
   },

   // ===== eState_OFF ==================
   // ----- transition actions ----------
   {  200,                                // transitionId
      eState_OFF,                         // current FSM State
      eTrigger_CRITICAL_VOLTAGE_END,      // trigger update
      eTrigger_APPLICATION_STATE_NORMAL,  // current active trigger(s)
      FSM_DONT_CARE,                      // current not active trigger(s)
      eState_ON                           // next FSM State
   },
   {  201,                                // transitionId
      eState_OFF,                         // current FSM State
      eTrigger_APPLICATION_STATE_NORMAL,  // trigger update
      eTrigger_CRITICAL_VOLTAGE_END,      // current active trigger(s)
      FSM_DONT_CARE,                      // current not active trigger(s)
      eState_ON                           // next FSM State
   },
   // ---- do actions -------------------
   {  230,                                // transitionId
      eState_OFF,                         // current FSM State
      eTrigger_HMI_AVAILABLE_TRUE,        // trigger update
      eTrigger_START,                     // current active trigger(s)
      FSM_DONT_CARE,                      // current not active trigger(s)
      eState_OFF                          // next FSM State
   },

   // ===== eState_ON ===================
   // ----- transition actions ----------
   {  300,                                // transitionId
      eState_ON,                          // current FSM State
      eTrigger_APPLICATION_STATE_NOT_NORMAL, // trigger update
      FSM_DONT_CARE,                      // current active trigger(s)
      FSM_DONT_CARE,                      // current not active trigger(s)
      eState_OFF                          // next FSM State
   },
   {  301,                                // transitionId
      eState_ON,                          // current FSM State
      eTrigger_CRITICAL_VOLTAGE_START,    // trigger update
      FSM_DONT_CARE,                      // current active trigger(s)
      FSM_DONT_CARE,                      // current not active trigger(s)
      eState_OFF                          // next FSM State
   },
   // ---- do actions -------------------
   {  330,                                // transitionId
      eState_ON,                          // current FSM State
      eTrigger_HMI_AVAILABLE_TRUE,        // trigger update
      eTrigger_START,                     // current active trigger(s)
      FSM_DONT_CARE,                      // current not active trigger(s)
      eState_ON                           // next FSM State
   }
};


/******************************************************************************/
/* FUNCTION     ~dispvidctrl_tclFsm_OnOff                                          */
/******************************************************************************/
/**
*  \brief       Constructor
*
*
*
*  \param       pointer to main application
*               ahl_tclFsmBase( tU32 u32InitFsmSTate,
*                               tU32 u32MaxTransitionEntry,
*                               TStateTransitions* paFsmTransitionTable,
*                               tU32 u32MaxStateEntry,
*                               FsmStateConf* paFsmStateTable,
*                               tU32 u32TraceClass
*                               tU8  u8FsmTraceId);
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclFsm_OnOff::dispvidctrl_tclFsm_OnOff(dispvidctrl_tclControl* poControl)
: dispvidctrl_tclFsm_Base(
     eState_OFF_EARLY,
     ELEMENTE(aFsm_OnOff_StateTransitionTable), aFsm_OnOff_StateTransitionTable,
     ELEMENTE(aFsm_OnOff_StateTable)          , aFsm_OnOff_StateTable,
     (tU32) TR_CLASS_DISPVIDCTRL_APPLICATION,
     DISPVIDCTRL_TRACE_ID_FSM_ONOFF)
, m_hTimerHandle_FsmStateTimeout(OSAL_C_INVALID_HANDLE)
, m_poControl(poControl)
, m_bStateTimeout(FALSE)
, m_bEarly(TRUE)
{
   // start timer for FSM state timeouts used for state verification
   if( OSAL_s32TimerCreate( (OSAL_tpfCallback)_pfCallbackTimer_FsmStateTimeout, (tPVoid)this, &m_hTimerHandle_FsmStateTimeout) != OSAL_OK)
   {
      ETG_TRACE_USR4(("ERROR - Could not create FSM timer!"));
      m_hTimerHandle_FsmStateTimeout = OSAL_C_INVALID_HANDLE;
   }
}


/******************************************************************************/
/* FUNCTION     ~dispvidctrl_tclFsm_OnOff                                          */
/******************************************************************************/
/**
*  \brief       destructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclFsm_OnOff::~dispvidctrl_tclFsm_OnOff(tVoid)
{
   // delete timers
   if (OSAL_C_INVALID_HANDLE != m_hTimerHandle_FsmStateTimeout)
   {
      OSAL_s32TimerSetTime(m_hTimerHandle_FsmStateTimeout, 0, 0);
      OSAL_s32TimerDelete(m_hTimerHandle_FsmStateTimeout);
   }

   m_poInstance = NULL;
   m_poControl = NULL;
}


/******************************************************************************/
/* FUNCTION     tclCreateInstance                                             */
/******************************************************************************/
/**
*  \brief       Create instance (singleton pattern)
*
*  \param       pointer to main application
*  \return      pointer to instance
*/
/******************************************************************************/
dispvidctrl_tclFsm_OnOff* dispvidctrl_tclFsm_OnOff::tclCreateInstance(dispvidctrl_tclControl* poControl)
{
   if (m_poInstance == NULL)
   {
      m_poInstance = OSAL_NEW dispvidctrl_tclFsm_OnOff(poControl);
   }
   return m_poInstance;
}


/******************************************************************************/
/* FUNCTION     vNewTrigger                                                   */
/******************************************************************************/
/**
*  \brief       To be called to enter a new Trigger
*
*  \param       Trigger and State
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_OnOff::vNewTrigger(tU32 u32Trigger, tBool bTriggerState) const
{
   if (NULL != m_poInstance)
   {
      m_poInstance->u32CalcNewFsmState(u32Trigger, bTriggerState);
   }
}


/******************************************************************************/
/* FUNCTION     vStateChangeDetected                                          */
/******************************************************************************/
/**
*  \brief       called when a state change happens
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_OnOff::vStateChangeDetected(tVoid)
{
   if (NULL != m_poInstance)
   {
      m_poInstance->vTraceStatusInfo(0);
   }

   return;
}

/******************************************************************************/
/* FUNCTION     vUpdateRelatedTrigger                                         */
/******************************************************************************/
/**
*  \brief       to correct "hold" trigger states which are related to actual
*               trigger,
*               overwritten base class function
*
*  \param       actual Trigger
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_OnOff::vUpdateRelatedTrigger(tU32 u32Trigger)
{
   switch (u32Trigger)
   {
      case dispvidctrl_tclFsm_OnOff::eTrigger_HMI_AVAILABLE_FALSE:
         vUpdateTrigger(eTrigger_HMI_AVAILABLE_TRUE, FALSE);
         break;
      case dispvidctrl_tclFsm_OnOff::eTrigger_HMI_AVAILABLE_TRUE:
         vUpdateTrigger(eTrigger_HMI_AVAILABLE_FALSE, FALSE);
         // delete old (early) trigger from Video Player
         vUpdateTrigger(eTrigger_VIDEO_STATUS_RECEIVED,FALSE);
         break;
      case dispvidctrl_tclFsm_OnOff::eTrigger_APPLICATION_STATE_NORMAL:
         vUpdateTrigger(eTrigger_APPLICATION_STATE_NOT_NORMAL, FALSE);
         break;
      case dispvidctrl_tclFsm_OnOff::eTrigger_APPLICATION_STATE_NOT_NORMAL:
         vUpdateTrigger(eTrigger_APPLICATION_STATE_NORMAL, FALSE);
         break;
      case dispvidctrl_tclFsm_OnOff::eTrigger_CRITICAL_VOLTAGE_START:
         vUpdateTrigger(eTrigger_CRITICAL_VOLTAGE_END, FALSE);
         break;
      case dispvidctrl_tclFsm_OnOff::eTrigger_CRITICAL_VOLTAGE_END:
         vUpdateTrigger(eTrigger_CRITICAL_VOLTAGE_START, FALSE);
         break;
      default:
         break;
   }
}

/******************************************************************************/
/* FUNCTION     _pfCallbackTimer_FsmStateTimeout                              */
/******************************************************************************/
/**
*  \brief
*
*  \param       pArg - pointer to this class
*  \return      none
*/
/******************************************************************************/
OSAL_tpfCallback dispvidctrl_tclFsm_OnOff::_pfCallbackTimer_FsmStateTimeout(tVoid* pArg)
{
   (tVoid) pArg;

   //ETG_TRACE_USR4(("  ====>>  : _pfCallbackTimer_FsmStateTimeout()"));

   //eTrigger_StateTimeout
   if ((m_poInstance) && (m_poInstance->m_poControl))
   {
      m_poInstance->m_poControl->vHandleFsmOutEvent_OnOff(FSM_ONOFF_OUT__FSM_ONOFF_INPUT_EVENT,
                                                                         (tU32) eTrigger_STATE_VERIFICATION_TIMEOUT);
   }

   return 0;
};


/******************************************************************************/
/* FUNCTION     vStartFsmStateTimer                                           */
/******************************************************************************/
/**
*  \brief       start or stop the timer
*
*  \param       timeout value
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_OnOff::vStartFsmStateTimer(tU32 u32Time)
{
   if(u32Time > 0)
   {
      OSAL_s32TimerSetTime(m_hTimerHandle_FsmStateTimeout, u32Time, 0);      // set Shutdown-Timer active
   }
   else
   {
      OSAL_s32TimerSetTime(m_hTimerHandle_FsmStateTimeout,       0, 0);     // reset Shutdown-Timer
   }
}


/******************************************************************************/
/******************************************************************************/
/******************************************************************************/
/* State Functions                                                            */
/******************************************************************************/

/******************************************************************************/
/* FUNCTION     vDo_OffEarly                                                  */
/******************************************************************************/
/**
*  \brief
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_OnOff::vDo_OffEarly(tVoid)
{
// trigger(s):
// eTrigger_START
// eTrigger_HMI_AVAILABLE_TRUE

   if ( (NULL != m_poInstance) && (NULL != m_poInstance->m_poControl) )
   {
      if (eTrigger_START == m_poInstance->u32GetCurrentTrigger() )
      {
         // with START trigger we will switch off early video control (vd_early , video player)
         // as response we should get current camera visibility and initialisation state
         // todo: time out implementation for the case that vd_early/video player will not response
         m_poInstance->m_poControl->vHandleFsmOutEvent_OnOff(FSM_ONOFF_OUT__EARLY_VIDEO_CONTROL_OFF);      
      }
      else if (eTrigger_HMI_AVAILABLE_TRUE == m_poInstance->u32GetCurrentTrigger() )
      {
         m_poInstance->m_poControl->vHandleFsmOutEvent_OnOff(FSM_ONOFF_OUT__HMI_AVAILABLE);      
      }
      else
      {
      }
   }
}


/******************************************************************************/
/* FUNCTION     vEntry_Off                                                    */
/******************************************************************************/
/**
*  \brief
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_OnOff::vEntry_Off(tVoid)
{
// trigger(s):
// eTrigger_APPLICATION_STATE_NOT_NORMAL
// eTrigger_CRITICAL_VOLTAGE_START
// eTrigger_VIDEO_STATUS_RECEIVED [Guard: ((eTrigger_CRITICAL_VOLTAGE_END && eTrigger_APPLICATION_STATE_NORMAL) == FALSE) && (eTrigger_START == TRUE)]

   if (NULL != m_poInstance)
   {
      if (m_poInstance->m_poControl)
      {
         if (m_poInstance->m_bEarly == FALSE)
         {
            if (TRUE == m_poInstance->bGetTriggerState(eTrigger_APPLICATION_STATE_NOT_NORMAL) )
            {
               m_poInstance->m_poControl->vHandleFsmOutEvent_OnOff(FSM_ONOFF_OUT__DISABLE_CONTROLS, (tU32) dispvidctrl_tclFsm_Hmi::EN_VD_RVC_HMI_VISUALISATION_OFF__APPLICATION_STATE_NOT_NORMAL);
            }
            else if (TRUE == m_poInstance->bGetTriggerState(eTrigger_CRITICAL_VOLTAGE_START) )
            {
               m_poInstance->m_poControl->vHandleFsmOutEvent_OnOff(FSM_ONOFF_OUT__DISABLE_CONTROLS, (tU32) dispvidctrl_tclFsm_Hmi::EN_VD_RVC_HMI_VISUALISATION_OFF__CRITICAL_UNDERVOLTAGE);
            }
            else 
            {}
         }
         else  // first transition into this state (previous state == OffEarly)
         {
            m_poInstance->m_bEarly = FALSE;

            m_poInstance->m_poControl->vHandleFsmOutEvent_OnOff(FSM_ONOFF_OUT__DISABLE_CONTROLS, (tU32) dispvidctrl_tclFsm_Hmi::EN_VD_RVC_HMI_VISUALISATION_OFF__INITIAL_OFF);
            // todo: Probably we have to send always the right disable reason, also if it was the early state transition.
            //       So we have to check the active trigger states normally.
            // We are using the Reason "Initial Off" to trigger the system status update to VISUALISATION_OFF state.
         }
      }
   }
}

/******************************************************************************/
/* FUNCTION     vDo_Off                                                       */
/******************************************************************************/
/**
*  \brief
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_OnOff::vDo_Off(tVoid)
{
// trigger(s):
// eTrigger_HMI_AVAILABLE_TRUE [Guard: eTrigger_START == TRUE]

   if (NULL != m_poInstance)
   {
      if (m_poInstance->m_poControl)
      {
         m_poInstance->m_poControl->vHandleFsmOutEvent_OnOff(FSM_ONOFF_OUT__HMI_AVAILABLE);
      }
   }
}

/******************************************************************************/
/* FUNCTION     vEntry_On                                                     */
/******************************************************************************/
/**
*  \brief
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_OnOff::vEntry_On(tVoid)
{
// trigger(s):
// eTrigger_APPLICATION_STATE_NORMAL
// eTrigger_CRITICAL_VOLTAGE_END
// eTrigger_VIDEO_STATUS_RECEIVED

   if (NULL != m_poInstance)
   {
      if (m_poInstance->m_poControl)
      {
         if (m_poInstance->m_bEarly == FALSE)
         {
            m_poInstance->m_poControl->vHandleFsmOutEvent_OnOff(FSM_ONOFF_OUT__ENABLE_CONTROLS);
         }
         else  // first transition into this state (previous state == OffEarly)
         {
            // -> enable own control functionality, takeover of current visualisation state
            m_poInstance->m_poControl->vHandleFsmOutEvent_OnOff(FSM_ONOFF_OUT__ENABLE_CONTROLS_INITIAL_WITH_EARLY_VIDEO_STATUS);

            m_poInstance->m_bEarly = FALSE;
         }
      }
   }
}

/******************************************************************************/
/* FUNCTION     vDo_On                                                        */
/******************************************************************************/
/**
*  \brief
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_OnOff::vDo_On(tVoid)
{
// trigger(s):
// eTrigger_HMI_AVAILABLE_TRUE [Guard: eTrigger_START == TRUE]

   if (NULL != m_poInstance)
   {
      if (m_poInstance->m_poControl)
      {
         m_poInstance->m_poControl->vHandleFsmOutEvent_OnOff(FSM_ONOFF_OUT__HMI_AVAILABLE);
      }
   }
}

/*******************************************************************************
*                                       E O F
*******************************************************************************/
