/* ******************************************************FileHeaderBegin** *//**
 *
 * @file        dispvidctrl_tclFsm_Base.cpp
 *  ShortInfo
 *
 *  LongInfo
 *
 * @date        2009-03-14
 *
 * @note
 *
 *  &copy; Copyright Robert Bosch Car Multimedia. All Rights reserved!
 *
 *//* ***************************************************FileHeaderEnd******* */

/* TENGINE Header */
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#include "dispvidctrl_tclFsm_Base.h"





/******************************************************************************/
/* FUNCTION     dispvidctrl_tclFsm_Base                                            */
/******************************************************************************/
/**
*  \brief       constructor
*
*  \param       
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclFsm_Base::dispvidctrl_tclFsm_Base(tU32 u32InitFsmState,
                                           tU32 u32MaxTransitionEntry,
                                           const TStateTransitions* paFsmTransitionTable,
                                           tU32 u32MaxStateEntry,
                                           const TFsmStateConf* paFsmStateTable,
                                           tU32 u32TraceClass,
                                           tU8 u8FsmTraceId,
                                           tBool bUseReenterTransitions)
{
   _u32Trigger = 0;
   _u32UpdateTrigger = 0;
   _u32LastTransitionId = FSM_TRANSITION_ID_NONE;
   _u32TimeOut = 0;
   _u32FsmState = u32InitFsmState;
   _u32LastFsmState = FSM_STATE_ID_NONE; //u32InitFsmState;

   _u32MaxStateEntry = u32MaxStateEntry;
   _paFsmStateTable = paFsmStateTable;

   _u32MaxTransitionEntry = u32MaxTransitionEntry;
   _paFsmTransitionTable = paFsmTransitionTable;

   _u32TraceClass = u32TraceClass;
   _u8FsmTraceId = u8FsmTraceId;
   _bUseReenterTransitions = bUseReenterTransitions;
   
   _stTransitionInfo.u32TransitionId     = FSM_TRANSITION_ID_NONE;
   _stTransitionInfo.u32CurrTrigger      = 0x00;
   _stTransitionInfo.bCurrTriggerState   = FALSE;
   _stTransitionInfo.u32OldTriggerStates = 0x00;
   _stTransitionInfo.u32UpdTriggerStates = 0x00;
   _stTransitionInfo.u32CfgOldState      = FSM_STATE_ID_NONE;
   _stTransitionInfo.u32CfgNewState      = FSM_STATE_ID_NONE;
   _stTransitionInfo.u32OldState         = FSM_STATE_ID_NONE;
   _stTransitionInfo.u32NewState         = FSM_STATE_ID_NONE;
}


/******************************************************************************/
/* FUNCTION     ~dispvidctrl_tclFsm_Base                                           */
/******************************************************************************/
/**
*  \brief       destructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclFsm_Base::~dispvidctrl_tclFsm_Base(){
   _paFsmStateTable = NULL;
   _paFsmTransitionTable = NULL;
}


/******************************************************************************/
/* FUNCTION     vStateChangeDetected                                          */
/******************************************************************************/
/**
*  \brief       
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_Base::vStateChangeDetected(){
   return;
}


/******************************************************************************/
/* FUNCTION     vStartFsmStateTimer                                           */
/******************************************************************************/
/**
*  \brief       
*
*  \param       time in ms
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_Base::vStartFsmStateTimer(tU32 u32Time)
{
   (tVoid) u32Time;
   return;
}


/******************************************************************************/
/* FUNCTION     vActStateExit                                                 */
/******************************************************************************/
/**
*  \brief       Called when new system state detected by state machine and
*               current state is left.
*
*  \param       State to exit
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_Base::vActStateExit(tU32 u32FsmState) const
{
   tBool bFound;
   const TFsmStateConf* prStateDescr;

   if (u32FsmState < _u32MaxStateEntry)
   {
      bFound = FALSE;
      prStateDescr = _paFsmStateTable;

      /* while state not found and not end of state table */
      while ( (bFound==FALSE) && (prStateDescr < (_paFsmStateTable + _u32MaxStateEntry) ) )
      {
         /* if state is active */
         if (u32FsmState == prStateDescr->u32FsmState)
         {
            /* call state exit function */
            bFound=TRUE;
            if (prStateDescr->pfnvStateExit) {
               (* prStateDescr->pfnvStateExit)();
            }
         }
         prStateDescr++;
      }
   }
}


/******************************************************************************/
/* FUNCTION     u32ActStateEntry                                              */
/******************************************************************************/
/**
*  \brief       Called when new system state detected by state machine and
*               current state has to be entered.
*
*  \param       u32NewSystemState -> system state detected by state machine
*  \return      u32Timeout -> time out value in ms for new system state
*/
/******************************************************************************/
tU32 dispvidctrl_tclFsm_Base::u32ActStateEntry(tU32 u32NewFsmState) const
{
   tBool bFound;
   const TFsmStateConf* prStateDescr;
   tU32 u32Timeout = 0;

   if (u32NewFsmState < _u32MaxStateEntry)
   {
      bFound = FALSE;
      prStateDescr = _paFsmStateTable;

      /* while state not found and not end of state table */
      while ( (bFound==FALSE) && (prStateDescr < (_paFsmStateTable + _u32MaxStateEntry) ) )
      {
         /* if state is active */
         if (u32NewFsmState == prStateDescr->u32FsmState)
         {
            /* call state entry function */
            bFound=TRUE;
            if (prStateDescr->pfnvStateEntry) {
               (* prStateDescr->pfnvStateEntry)();
            }
            u32Timeout = prStateDescr->u32Timeout;
         }
         prStateDescr++;
      }
   }
   return u32Timeout;
}


/******************************************************************************/
/* FUNCTION     vActStateDo                                                   */
/******************************************************************************/
/**
*  \brief       Called when state needs trigger action without state transition.
*
*  \param       system state
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_Base::vActStateDo(tU32 u32FsmState) const
{
   tBool bFound;
   const TFsmStateConf* prStateDescr;

   if (u32FsmState < _u32MaxStateEntry)
   {
      bFound = FALSE;
      prStateDescr = _paFsmStateTable;

      /* while state not found and not end of state table */
      while ( (bFound==FALSE) && (prStateDescr < (_paFsmStateTable + _u32MaxStateEntry) ) )
      {
         /* if state is active */
         if (u32FsmState == prStateDescr->u32FsmState)
         {
            /* call state do function */
            bFound=TRUE;
            if (prStateDescr->pfnvStateDo) {
               (* prStateDescr->pfnvStateDo)();
            }
         }
         prStateDescr++;
      }
   }
   return;
}


/******************************************************************************/
/* FUNCTION     vUpdateTrigger                                                */
/******************************************************************************/
/**
*  \brief       to modify "hold" trigger information without doing an state change
*
*  \param       Trigger and State
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_Base::vUpdateTrigger(tU32 u32Trigger, tBool bState)
{
   if (bState) {
      _u32Trigger |= u32Trigger;
   } else {
      _u32Trigger &= ~u32Trigger;
   }
}


/******************************************************************************/
/* FUNCTION     vUpdateRelatedTrigger                                         */
/******************************************************************************/
/**
*  \brief       to correct "hold" trigger states which are related to actual
*               trigger,
*               derived class has to overwrite this function
*
*  \param       actual Trigger
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_Base::vUpdateRelatedTrigger(tU32 u32Trigger)
{
   (tVoid) u32Trigger;
}


/******************************************************************************/
/* FUNCTION     u32CalcNewFsmState                                            */
/******************************************************************************/
/**
*  \brief       check for new system state in state transition table
*               _arStateTable[].
*
*  \param       actual Trigger
*               u32PreviousSystemState -> last valid system state
*               u32NewSystemState      -> new system state, if different to
*                                         u32PreviousSystemState, system state
*                                         transition is still active 
*  \return      _u32FsmState -> system state after the transition
*/
/******************************************************************************/
tU32 dispvidctrl_tclFsm_Base::u32CalcNewFsmState(tU32 u32Trigger, tBool bState)
{
   tU32 u32ConfigCurrSystemState  = FSM_DONT_CARE;      // the from-state which is configured in the transition table (may be a valid state ID or FSM_DONT_CARE)
   tU32 u32ConfigNextSystemState  = FSM_STATE_ID_NONE;  // the goto-state which is configured in the transition table (may be a valid state ID or FSM_STATE_ID_NONE)
   tU32 u32TransitionId           = FSM_TRANSITION_ID_NONE;
   tU32 u32TransitionActionId     = FSM_TRANSITION_ACTION_ID_NONE;
   tU32 u32TransitionActionParam1 = 0x00;
   tU32 u32TransitionActionParam2 = 0x00;

   _u32UpdateTrigger = u32Trigger; // hold the new received trigger
   
   tU32 u32TriggerStatesOnEntry = _u32Trigger;
   if (bState) {
      _u32Trigger |= u32Trigger;
   } else {
      _u32Trigger &= ~u32Trigger;
   }
   vUpdateRelatedTrigger(u32Trigger);

   ET_TRACE_INFO_BIN( _u32TraceClass,
           ET_EN_T8 _ _u8FsmTraceId _
          ET_EN_T16 _ FSM_TRACE_ID_TRIGGER _
          ET_EN_T32 _ _u32UpdateTrigger _
          ET_EN_T32 _ _u32Trigger _
         ET_EN_DONE);


   tU32 u32TabEntries = _u32MaxTransitionEntry;
   const TStateTransitions* prTransitionDescr     = _paFsmTransitionTable;
   const TStateTransitions* prLastTransitionEntry = _paFsmTransitionTable + u32TabEntries;

   if ((u32TabEntries > 0) && prTransitionDescr) {
      /* while end state not found and not end of state table */
      while ( (u32TransitionId == FSM_TRANSITION_ID_NONE) && (prTransitionDescr < prLastTransitionEntry) )
      {
         // check system state
         if ((prTransitionDescr->u32FsmState == _u32FsmState) || (prTransitionDescr->u32FsmState == FSM_DONT_CARE)) {
            // check update Trigger
            if (((prTransitionDescr->u32UpdateTrigger&_u32UpdateTrigger)==prTransitionDescr->u32UpdateTrigger) || (prTransitionDescr->u32UpdateTrigger == FSM_DONT_CARE)){
               // check if an active trigger as guard condition is required
               if (((prTransitionDescr->u32TriggerSet&_u32Trigger)==prTransitionDescr->u32TriggerSet)|| (prTransitionDescr->u32TriggerSet == FSM_DONT_CARE)) {
                  // check if an not active trigger as guard condition is required
                  if (((prTransitionDescr->u32TriggerNotSet & ~_u32Trigger) ==  prTransitionDescr->u32TriggerNotSet)|| (prTransitionDescr->u32TriggerNotSet == FSM_DONT_CARE)){
                     // found a table entry
                     u32ConfigCurrSystemState  = prTransitionDescr->u32FsmState;
                     u32ConfigNextSystemState  = prTransitionDescr->u32NextFsmState;
                     u32TransitionId           = prTransitionDescr->u32TransitionId;
                     u32TransitionActionId     = prTransitionDescr->u32TransitionActionId;
                     u32TransitionActionParam1 = prTransitionDescr->u32TransitionActionParam1;
                     u32TransitionActionParam2 = prTransitionDescr->u32TransitionActionParam2;
                  }
               }
            }
         }

         prTransitionDescr++;
      }
   }

   if (u32TransitionId != FSM_TRANSITION_ID_NONE)
   {      
      tU32  u32ResultingSystemState; // the state in which the FSM will be after the transition is done (may be the same as before)
      tBool bPureDoTransition;       // TRUE => pure do-transition inside current state; FALSE => exit current + enter result state (even if both are the same <-> reentry)
      
      // determine *where* we will actually go (next state may configured as DONT_CARE, i.e. next==curr), and *how* we will go there (e.g. via exit+reentry or via do)
      if (FSM_STATE_ID_NONE == u32ConfigNextSystemState) {    // next-state not configured in the transition table => stay where we are
         u32ResultingSystemState = _u32FsmState;
         bPureDoTransition       = TRUE;
      } else {
         u32ResultingSystemState = u32ConfigNextSystemState;  // this is where we are bound to go; now determine how to get there:
         if (FSM_DONT_CARE == u32ConfigCurrSystemState) {     // if we may currently be in *any* state, check if we *effectively* do or don't have a state change:
            bPureDoTransition = (u32ResultingSystemState == _u32FsmState);   // next!=current => (exit+enter) ; next==current => always (do) (for backward compatibility)
         } else {                        // if current state is *explicitly* specified => almost as above, but if next==current, it depends on the reentry-feature switch
            bPureDoTransition = (u32ResultingSystemState == _u32FsmState) && !_bUseReenterTransitions;  // false if curr!=next, and also if curr==next && useReenter=true
         }
      }

      // then setup the info with all the details of the transition we are about to carry out,
      // so that Entry/Do/Exit/TransitionAction functions may be able to use them, if required
      _stTransitionInfo.u32TransitionId     = u32TransitionId;
      _stTransitionInfo.u32CurrTrigger      = _u32UpdateTrigger;
      _stTransitionInfo.bCurrTriggerState   = bState;
      _stTransitionInfo.u32OldTriggerStates = u32TriggerStatesOnEntry;
      _stTransitionInfo.u32UpdTriggerStates = _u32Trigger;
      _stTransitionInfo.u32CfgOldState      = u32ConfigCurrSystemState; // may be DONT_CARE
      _stTransitionInfo.u32CfgNewState      = u32ConfigNextSystemState; // ditto
      _stTransitionInfo.u32OldState         = _u32FsmState;             // will always be a real state != DONT_CARE
      _stTransitionInfo.u32NewState         = u32ResultingSystemState;  // will always be a real state != DONT_CARE
      
      // finally carry out the required actions
      if (! bPureDoTransition)
      {
         // transition to a new state: call vExit...() and vEntry...()
         vActStateExit(_u32FsmState);
         if (FSM_TRANSITION_ACTION_ID_NONE != u32TransitionActionId) {
             vTransitionAction(u32TransitionActionId,u32TransitionActionParam1,u32TransitionActionParam2);
         }
         _u32TimeOut = u32ActStateEntry(u32ResultingSystemState);
         vStartFsmStateTimer(_u32TimeOut);
      }
      else
      {
         // no transition but trigger update: call vDo...() or do transition action
         if (FSM_TRANSITION_ACTION_ID_NONE != u32TransitionActionId) {
             vTransitionAction(u32TransitionActionId,u32TransitionActionParam1,u32TransitionActionParam2);
         } else {
             vActStateDo(_u32FsmState);
         }
      }
      
      // set new states
      _u32LastTransitionId = u32TransitionId;
      _u32LastFsmState     = _u32FsmState;
      _u32FsmState         = u32ResultingSystemState;
      vStateChangeDetected();

      ET_TRACE_INFO_BIN( _u32TraceClass,
      ET_EN_T8 _ _u8FsmTraceId _
      ET_EN_T16 _ FSM_TRACE_ID_NEW_STATE _
      ET_EN_T32 _ _u32FsmState _
      ET_EN_T32 _ _u32LastFsmState _
      ET_EN_T32 _ _u32LastTransitionId _
      ET_EN_DONE);

   } else {
      //no new state found
      vStateNoTransition();

      ET_TRACE_INFO_BIN( _u32TraceClass,
          ET_EN_T8 _ _u8FsmTraceId _
         ET_EN_T16 _ FSM_TRACE_ID_NO_NEW_STATE _
         ET_EN_T32 _ _u32FsmState _
         ET_EN_DONE);
   }

   return(_u32FsmState);
}


/******************************************************************************/
/* FUNCTION     vStateNoTransition                                            */
/******************************************************************************/
/**
*  \brief       
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_Base::vStateNoTransition(tVoid) const
{
   return;
}


/******************************************************************************/
/* FUNCTION     vTraceStatusInfo                                              */
/******************************************************************************/
/**
*  \brief       
*
*  \param       u32RemainingTime
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_Base::vTraceStatusInfo(tU32 u32RemainingTime)
{
      ET_TRACE_INFO_BIN( _u32TraceClass,
          ET_EN_T8 _ _u8FsmTraceId _
         ET_EN_T16 _  FSM_TRACE_ID_INFO _
         ET_EN_T32 _ _u32FsmState _
         ET_EN_T32 _ _u32LastFsmState _
         ET_EN_T32 _ _u32LastTransitionId _
         ET_EN_T32 _ _u32UpdateTrigger _
         ET_EN_T32 _ _u32Trigger _
         ET_EN_T32 _ _u32TimeOut _
         ET_EN_T32 _ u32RemainingTime _
         ET_EN_DONE);
}


