/******************************************************************************/
/**
* \file    dispvidctrl_tclFsm_Avm.cpp
* \ingroup
*
* \brief   state machine for Avm
*
* \remark  Copyright : (c) 2015 Robert Bosch GmbH, Hildesheim
* \remark  Author    : Michael Niemann CM-AI/PJ-CB32
* \remark  Scope     : A-IVI
*
* \todo
*/
/******************************************************************************/

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#include "dispvidctrl_AppMain.h"
#include "dispvidctrl_tclFsm_Hmi.h"
#include "dispvidctrl_tclControl_Avm.h"
#include "dispvidctrl_tclFsm_Avm.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_APPLICATION
#include "trcGenProj/Header/dispvidctrl_tclFsm_Avm.cpp.trc.h"
#endif

#define DISPVIDCTRL_RVC_TRACE_ID_FSM_AVM 0x05

#define ELEMENTE(array)       ((tU16)(sizeof(array)/sizeof(array[0])))

#define DISPVIDCTRL_AVM_REINIT_TIME_IN_MS                 2100
#define DISPVIDCTRL_AVM_DELAY_OF_CAN_MSG_TIMOUT_FALSE_TIME_IN_MS   600
#define DISPVIDCTRL_AVM_CAMERA_OFF_REQUEST_DELAY          500

#ifndef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
   #define DISPVIDCTRL_AVM_CAMERA_ON_REQUEST_TIMEOUT         600
#else
   #define DISPVIDCTRL_AVM_CAMERA_ON_REQUEST_TIMEOUT         3000
#endif

dispvidctrl_tclFsm_Avm*   dispvidctrl_tclFsm_Avm::m_poInstance  = OSAL_NULL;

/*******************************************************************************
*                                  STATE TABLE
*******************************************************************************/
dispvidctrl_tclFsm_Avm::TFsmStateConf dispvidctrl_tclFsm_Avm::aFsm_Avm_StateTable[eState_Max] =
{
   {
        eState_AVM_UNINITIALISED,
        0,                                              //u32Timeout
        vEntry_AvmUninitialised,                        //tFsmStateEntryFunc
        vDo_AvmUnitialized,                             //tFsmStateDoFunc
        vExit_AvmUninitialised,                         //tFsmStateExitFunc
    },
    {
        eState_AVM_OFF,
        0,                                              //u32Timeout
        vEntry_AvmOff,                                  //tFsmStateEntryFunc
        vDo_AvmOff,                                     //tFsmStateDoFunc
        vExit_AvmOff,                                   //tFsmStateExitFunc
    },
    {
        eState_AVM_ON,
        0,                                              //u32Timeout
        vEntry_AvmOn,                                   //tFsmStateEntryFunc
        vDo_AvmOn,                                      //tFsmStateDoFunc
        vExit_AvmOn,                                    //tFsmStateExitFunc
    },
    {
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF,
        0,                                              //u32Timeout
        vEntry_AvmOnWaitForReverseOff,                  //tFsmStateEntryFunc
        vDo_AvmOnWaitForReverseOff,                     //tFsmStateDoFunc
        vExit_AvmOnWaitForReverseOff,                   //tFsmStateExitFunc
    },
    {
        eState_AVM_OFF_CAMERA_SETTINGS_WAIT_FOR_ON,
        0,                                              //u32Timeout
        vEntry_AvmOffCameraSettingsWaitForOn,           //tFsmStateEntryFunc
        vDo_AvmOffCameraSettingsWaitForOn,              //tFsmStateDoFunc
        vExit_AvmOffCameraSettingsWaitForOn,            //tFsmStateExitFunc
    },
    {
        eState_AVM_ON_CAMERA_SETTINGS,
        0,                                              //u32Timeout
        vEntry_AvmOnCameraSettings,                     //tFsmStateEntryFunc
        vDo_AvmOnCameraSettings,                        //tFsmStateDoFunc
        vExit_AvmOnCameraSettings,                      //tFsmStateExitFunc
    },
    {
        eState_AVM_ON_CAMERA_SETTINGS_TEMPORAL,
        0,                                              //u32Timeout
        vEntry_AvmOnCameraSettingsTemporal,             //tFsmStateEntryFunc
        vDo_AvmOnCameraSettingsTemporal,                //tFsmStateDoFunc
        vExit_AvmOnCameraSettingsTemporal,              //tFsmStateExitFunc
    }
};

/*******************************************************************************
 *                             STATE TRANSITION TABLE
 *******************************************************************************/
dispvidctrl_tclFsm_Avm::TStateTransitions dispvidctrl_tclFsm_Avm::aFsm_Avm_StateTransitionTable[] =
{
   //  ***********************************************
    //  MAIN STATES
    //  ***********************************************
    //  ===== eState_AVM_UNINITIALISED ================
    //  ---- transition actions -----------------------
    {   100,                                            // transitionId
        eState_AVM_UNINITIALISED,                       // current FSM State
        eTrigger_AVM_EV_INIT_DONE,                      // trigger update
        eTrigger_AVM_EV_IGNITION_ON,                    // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_OFF                                  // next FSM State
    },
    //  ---- do actions--------------------------------
    {   130,                                            // transitionId
        eState_AVM_UNINITIALISED,                       // current FSM State
        eTrigger_AVM_EV_IGNITION_ON,                    // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_UNINITIALISED                        // next FSM State
    },
    {   131,                                            // transitionId
        eState_AVM_UNINITIALISED,                       // current FSM State
        eTrigger_AVM_EV_IGNITION_OFF,                   // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_UNINITIALISED                        // next FSM State
    },
    {   132,                                            // transitionId
        eState_AVM_UNINITIALISED,                       // current FSM State
        eTrigger_AVM_EV_REINIT_TIMEOUT,                 // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_UNINITIALISED                        // next FSM State
    },
    {   133,                                            // transitionId
        eState_AVM_UNINITIALISED,                       // current FSM State
        eTrigger_AVM_EV_CAN_MSG_TIMEOUT_TRUE,           // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_UNINITIALISED                        // next FSM State
    },
    {   134,                                            // transitionId
        eState_AVM_UNINITIALISED,                       // current FSM State
        eTrigger_AVM_EV_CAN_MSG_TIMEOUT_FALSE,          // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_UNINITIALISED                        // next FSM State
    },
    {   135,                                            // transitionId
        eState_AVM_UNINITIALISED,                       // current FSM State
        eTrigger_AVM_EV_INIT_ACTIVE,                    // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_UNINITIALISED                        // next FSM State
    },


    //  == In all states > eState_AVM_UNINITIALISED ===
    //  ---- transition actions -----------------------
    {   200,                                            // transitionId
        FSM_DONT_CARE,                                  // current FSM State
        eTrigger_AVM_EV_IGNITION_OFF,                   // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_UNINITIALISED                        // next FSM State
    },
    {   201,                                            // transitionId
        FSM_DONT_CARE,                                  // current FSM State
        eTrigger_AVM_EV_INIT_ACTIVE,                    // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_UNINITIALISED                        // next FSM State
    },
    {   202,                                            // transitionId
        FSM_DONT_CARE,                                  // current FSM State
        eTrigger_AVM_EV_CAN_MSG_TIMEOUT_TRUE,           // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_UNINITIALISED                        // next FSM State
    },

    //  ===== eState_AVM_OFF ==========================
    //  ---- transition actions -----------------------
    {   300,                                            // transitionId
        eState_AVM_OFF,                                 // current FSM State
        eTrigger_AVM_EV_CAMERA_SCREEN_ON,               // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON                                   // next FSM State
    },
    {   301,                                            // transitionId
        eState_AVM_OFF,                                 // current FSM State
        eTrigger_AVM_EV_CAMERA_SETTINGS_ON,             // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_OFF_CAMERA_SETTINGS_WAIT_FOR_ON      // next FSM State
    },
    //  ---- do actions--------------------------------
    {   332,                                            // transitionId
        eState_AVM_OFF,                                 // current FSM State
        eTrigger_AVM_EV_CAMERA_BUTTON_PRESS,            // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_OFF                                  // next FSM State
    },
    {   333,                                            // transitionId
        eState_AVM_OFF,                                 // current FSM State
        eTrigger_AVM_EV_CAMERA_BUTTON_LONG_PRESS,       // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_OFF                                  // next FSM State
    },
    {   334,                                            // transitionId
        eState_AVM_OFF,                                 // current FSM State
        eTrigger_AVM_EV_NORMAL_BEEP,                    // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_OFF                                  // next FSM State
    },
    {   335,                                            // transitionId
        eState_AVM_OFF,                                 // current FSM State
        eTrigger_AVM_EV_ERROR_BEEP,                     // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_OFF                                  // next FSM State
    },
    {   336,                                            // transitionId
        eState_AVM_OFF,                                 // current FSM State
        eTrigger_AVM_EV_AFFORDABLE_ITS,                 // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_OFF                                  // next FSM State
    },
    {   337,                                            // transitionId
        eState_AVM_OFF,                                 // current FSM State
        eTrigger_AVM_EV_TIMEOUT_VIEW_CHANGE_REQUEST_WAIT, // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_OFF                                  // next FSM State
    },

    //  ===== eState_AVM_ON ===========================
    //  ---- transition actions -----------------------
    {   501,                                            // transitionId
        eState_AVM_ON,                                  // current FSM State
        eTrigger_AVM_EV_CAMERA_SCREEN_OFF,              // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_OFF                                  // next FSM State
    },
    {   502,                                            // transitionId
        eState_AVM_ON,                                  // current FSM State
        eTrigger_AVM_EV_CAMERA_SETTINGS_ON,             // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_CAMERA_SETTINGS_TEMPORAL          // next FSM State
    },
    //  ---- do actions--------------------------------
    {   530,                                            // transitionId
        eState_AVM_ON,                                  // current FSM State
        eTrigger_AVM_EV_CAMERA_BUTTON_PRESS,            // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON                                   // next FSM State
    },
    {   531,                                            // transitionId
        eState_AVM_ON,                                  // current FSM State
        eTrigger_AVM_EV_CAMERA_BUTTON_LONG_PRESS,       // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON                                   // next FSM State
    },
    {   532,                                            // transitionId
        eState_AVM_ON,                                  // current FSM State
        eTrigger_AVM_EV_NORMAL_BEEP,                    // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON                                   // next FSM State
    },
    {   533,                                            // transitionId
        eState_AVM_ON,                                  // current FSM State
        eTrigger_AVM_EV_ERROR_BEEP,                     // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON                                   // next FSM State
    },
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE1
    {   534,                                            // transitionId
        eState_AVM_ON,                                  // current FSM State
        eTrigger_AVM_EV_PURPOSE_SWITCH,                 // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        eTrigger_AVM_EV_REVERSE_ON_DEBOUNCED,           // current not active trigger(s)
        eState_AVM_ON                                   // next FSM State
    },
#else
    {   534,                                            // transitionId
        eState_AVM_ON,                                  // current FSM State
        eTrigger_AVM_EV_PURPOSE_SWITCH,                 // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON                                   // next FSM State
    },
#endif
    {   535,                                            // transitionId
        eState_AVM_ON,                                  // current FSM State
        eTrigger_AVM_EV_AFFORDABLE_ITS,                 // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON                                   // next FSM State
    },
    {   536,                                            // transitionId
        eState_AVM_ON,                                  // current FSM State
        eTrigger_AVM_EV_VERIFY_AFTER_VIEW_STATUS_STOPPED, // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON                                   // next FSM State
    },
    {   537,                                            // transitionId
        eState_AVM_ON,                                  // current FSM State
        eTrigger_AVM_EV_VIDEO_SIGNAL_FAILURE,           // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON                                   // next FSM State
    },
    {   538,                                            // transitionId
        eState_AVM_ON,                                  // current FSM State
        eTrigger_AVM_EV_VIDEO_SIGNAL_RESTORED,          // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON                                   // next FSM State
    },
    {   539,                                            // transitionId
        eState_AVM_ON,                                  // current FSM State
        eTrigger_AVM_EV_CAMERA_SETTINGS_OFF,            // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON                                   // next FSM State
    },
    {   540,                                            // transitionId
        eState_AVM_ON,                                  // current FSM State
        eTrigger_AVM_EV_TIMEOUT_VIEW_CHANGE_REQUEST_WAIT, // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON                                   // next FSM State
    },
    {   541,                                            // transitionId
        eState_AVM_ON,                                  // current FSM State
        eTrigger_AVM_EV_VIEW_STATUS_UPDATE,             // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON                                   // next FSM State
    },
    {   542,                                            // transitionId
        eState_AVM_ON,                                  // current FSM State
        eTrigger_AVM_EV_ERROR_TRUE,                     // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON                                   // next FSM State
    },
    {   543,                                            // transitionId
        eState_AVM_ON,                                  // current FSM State
        eTrigger_AVM_EV_ERROR_FALSE,                    // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON                                   // next FSM State
    },

    //  ===== eState_AVM_ON_WAIT_FOR_REVERSE_OFF ======
    //  ---- transition actions -----------------------
    {   602,                                            // transitionId
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF,             // current FSM State
        eTrigger_AVM_EV_CAMERA_SCREEN_OFF,              // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_OFF                                  // next FSM State
    },
    {   604,                                            // transitionId
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF,             // current FSM State
        eTrigger_AVM_EV_CAMERA_BUTTON_PRESS,            // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON                                   // next FSM State
    },
    {   605,                                            // transitionId
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF,             // current FSM State
        eTrigger_AVM_EV_CAMERA_BUTTON_LONG_PRESS,       // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON                                   // next FSM State
    },
    {   606,                                            // transitionId
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF,             // current FSM State
        eTrigger_AVM_EV_TIMEOUT_CAMERA_OFF_REQUEST_DELAY, // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON                                   // next FSM State
    },
    //  ---- do actions--------------------------------
    {   630,                                            // transitionId
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF,             // current FSM State
        eTrigger_AVM_EV_NORMAL_BEEP,                    // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF              // next FSM State
    },
    {   631,                                            // transitionId
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF,             // current FSM State
        eTrigger_AVM_EV_ERROR_BEEP,                     // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF              // next FSM State
    },
    {   632,                                            // transitionId
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF,             // current FSM State
        eTrigger_AVM_EV_PURPOSE_SWITCH,                 // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        eTrigger_AVM_EV_REVERSE_ON_DEBOUNCED,           // current not active trigger(s)
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF              // next FSM State
    },
    {   633,                                            // transitionId
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF,             // current FSM State
        eTrigger_AVM_EV_AFFORDABLE_ITS,                 // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF              // next FSM State
    },
    {   634,                                            // transitionId
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF,             // current FSM State
        eTrigger_AVM_EV_REVERSE_OFF,                    // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF              // next FSM State
    },
    {   635,                                            // transitionId
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF,             // current FSM State
        eTrigger_AVM_EV_VERIFY_AFTER_VIEW_STATUS_STOPPED, // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF              // next FSM State
    },
    {   636,                                            // transitionId
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF,             // current FSM State
        eTrigger_AVM_EV_REVERSE_ON_DEBOUNCED,           // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF              // next FSM State
    },
    {   637,                                            // transitionId
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF,             // current FSM State
        eTrigger_AVM_EV_VIDEO_SIGNAL_FAILURE,           // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF              // next FSM State
    },
    {   638,                                            // transitionId
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF,             // current FSM State
        eTrigger_AVM_EV_VIDEO_SIGNAL_RESTORED,          // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF              // next FSM State
    },
    {   639,                                            // transitionId
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF,             // current FSM State
        eTrigger_AVM_EV_TIMEOUT_VIEW_CHANGE_REQUEST_WAIT, // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF              // next FSM State
    },
    {   640,                                            // transitionId
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF,             // current FSM State
        eTrigger_AVM_EV_CAMERA_SETTINGS_OFF,            // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF              // next FSM State
    },

    {   641,                                            // transitionId
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF,             // current FSM State
        eTrigger_AVM_EV_ERROR_TRUE,                     // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF              // next FSM State
    },
    {  642,                                             // transitionId
       eState_AVM_ON_WAIT_FOR_REVERSE_OFF,              // current FSM State
       eTrigger_AVM_EV_ERROR_FALSE,                     // trigger update
       FSM_DONT_CARE,                                   // current active trigger(s)
       FSM_DONT_CARE,                                   // current not active trigger(s)
       eState_AVM_ON_WAIT_FOR_REVERSE_OFF               // next FSM State
    },
    //  ===== eState_AVM_OFF_CAMERA_SETTINGS_WAIT_FOR_ON
    //  ---- transition actions -----------------------
    {   702,                                            // transitionId
        eState_AVM_OFF_CAMERA_SETTINGS_WAIT_FOR_ON,     // current FSM State
        eTrigger_AVM_EV_CAMERA_SCREEN_ON,               // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        eTrigger_AVM_EV_REVERSE_ON_DEBOUNCED,           // current not active trigger(s)
        eState_AVM_ON_CAMERA_SETTINGS                   // next FSM State
    },
    {   703,                                            // transitionId
        eState_AVM_OFF_CAMERA_SETTINGS_WAIT_FOR_ON,     // current FSM State
        eTrigger_AVM_EV_CAMERA_SCREEN_ON,               // trigger update
        eTrigger_AVM_EV_REVERSE_ON_DEBOUNCED,           // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON                                   // next FSM State
    },
    {   705,                                            // transitionId
        eState_AVM_OFF_CAMERA_SETTINGS_WAIT_FOR_ON,     // current FSM State
        eTrigger_AVM_EV_TIMEOUT_VIEW_CHANGE_REQUEST_WAIT, // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_OFF                                  // next FSM State
    },
    {   706,                                            // transitionId
        eState_AVM_OFF_CAMERA_SETTINGS_WAIT_FOR_ON,     // current FSM State
        eTrigger_AVM_EV_ERROR_BEEP,                     // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_OFF                                  // next FSM State
    },
    //  ---- do actions--------------------------------
    {   730,                                            // transitionId
        eState_AVM_OFF_CAMERA_SETTINGS_WAIT_FOR_ON,     // current FSM State
        eTrigger_AVM_EV_AFFORDABLE_ITS,                 // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_OFF_CAMERA_SETTINGS_WAIT_FOR_ON      // next FSM State
    },


    //  ===== eState_AVM_ON_CAMERA_SETTINGS ===========
    //  ---- transition actions -----------------------
    {   803,                                            // transitionId
        eState_AVM_ON_CAMERA_SETTINGS,                  // current FSM State
        eTrigger_AVM_EV_CAMERA_SCREEN_OFF,              // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_OFF                                  // next FSM State
    },
    {   805,                                            // transitionId
        eState_AVM_ON_CAMERA_SETTINGS,                  // current FSM State
        eTrigger_AVM_EV_REVERSE_ON_DEBOUNCED,           // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_WAIT_FOR_REVERSE_OFF              // next FSM State
    },

    //  ---- do actions--------------------------------
    {   830,                                            // transitionId
        eState_AVM_ON_CAMERA_SETTINGS,                  // current FSM State
        eTrigger_AVM_EV_ERROR_BEEP,                     // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_CAMERA_SETTINGS                   // next FSM State
    },
    {   831,                                            // transitionId
        eState_AVM_ON_CAMERA_SETTINGS,                  // current FSM State
        eTrigger_AVM_EV_AFFORDABLE_ITS,                 // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_CAMERA_SETTINGS                   // next FSM State
    },
    {   831,                                            // transitionId
        eState_AVM_ON_CAMERA_SETTINGS,                  // current FSM State
        eTrigger_AVM_EV_PURPOSE_SWITCH,                 // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_CAMERA_SETTINGS                   // next FSM State
    },
    {   832,                                            // transitionId
        eState_AVM_ON_CAMERA_SETTINGS,                  // current FSM State
        eTrigger_AVM_EV_VERIFY_AFTER_VIEW_STATUS_STOPPED, // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_CAMERA_SETTINGS                   // next FSM State
    },
    {   833,                                            // transitionId
        eState_AVM_ON_CAMERA_SETTINGS,                  // current FSM State
        eTrigger_AVM_EV_VIDEO_SIGNAL_FAILURE,           // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_CAMERA_SETTINGS                   // next FSM State
    },
    {   834,                                            // transitionId
        eState_AVM_ON_CAMERA_SETTINGS,                  // current FSM State
        eTrigger_AVM_EV_VIDEO_SIGNAL_RESTORED,          // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_CAMERA_SETTINGS                   // next FSM State
    },
    {   835,                                            // transitionId
        eState_AVM_ON_CAMERA_SETTINGS,                  // current FSM State
        eTrigger_AVM_EV_CAMERA_SETTINGS_OFF,            // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_CAMERA_SETTINGS                   // next FSM State
    },
    {   836,                                            // transitionId
        eState_AVM_ON_CAMERA_SETTINGS,                  // current FSM State
        eTrigger_AVM_EV_TIMEOUT_VIEW_CHANGE_REQUEST_WAIT,  // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_CAMERA_SETTINGS                   // next FSM State
    },
    {   837,                                            // transitionId
        eState_AVM_ON_CAMERA_SETTINGS,                  // current FSM State
        eTrigger_AVM_EV_ERROR_TRUE,                     // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_CAMERA_SETTINGS                   // next FSM State
    },
    {   838,                                            // transitionId
        eState_AVM_ON_CAMERA_SETTINGS,                  // current FSM State
        eTrigger_AVM_EV_ERROR_FALSE,                    // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_CAMERA_SETTINGS                   // next FSM State
    },    
    
    //  ===== eState_AVM_ON_CAMERA_SETTINGS_TEMPORAL ===========
    //  ---- transition actions -----------------------
    {   900,                                            // transitionId
        eState_AVM_ON_CAMERA_SETTINGS_TEMPORAL,         // current FSM State
        eTrigger_AVM_EV_CAMERA_SCREEN_OFF,              // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_OFF                                  // next FSM State
    },
    {   901,                                            // transitionId
        eState_AVM_ON_CAMERA_SETTINGS_TEMPORAL,         // current FSM State
        eTrigger_AVM_EV_CAMERA_SETTINGS_OFF,            // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON                                   // next FSM State
    },
    
    //  ---- do actions--------------------------------
    {   910,                                            // transitionId
        eState_AVM_ON_CAMERA_SETTINGS_TEMPORAL,         // current FSM State
        eTrigger_AVM_EV_VIDEO_SIGNAL_FAILURE,           // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_CAMERA_SETTINGS_TEMPORAL,         // current FSM State
    },
    {   911,                                            // transitionId
        eState_AVM_ON_CAMERA_SETTINGS_TEMPORAL,         // current FSM State
        eTrigger_AVM_EV_VIDEO_SIGNAL_RESTORED,          // trigger update
        FSM_DONT_CARE,                                  // current active trigger(s)
        FSM_DONT_CARE,                                  // current not active trigger(s)
        eState_AVM_ON_CAMERA_SETTINGS_TEMPORAL,         // current FSM State
    }
};


/******************************************************************************/
/* FUNCTION     ~dispvidctrl_tclFsm_Avm                                            */
/******************************************************************************/
/**
*  \brief       Constructor
*
*
*
*  \param       pointer to main application
*               ahl_tclFsmBase( tU32 u32InitFsmSTate,
*                               tU32 u32MaxTransitionEntry,
*                               TStateTransitions* paFsmTransitionTable,
*                               tU32 u32MaxStateEntry,
*                               FsmStateConf* paFsmStateTable,
*                               tU32 u32TraceClass
*                               tU8  u8FsmTraceId);
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclFsm_Avm::dispvidctrl_tclFsm_Avm(dispvidctrl_tclControl_Avm* poControl_Avm)
: dispvidctrl_tclFsm_Base(
     eState_AVM_UNINITIALISED,
     ELEMENTE(aFsm_Avm_StateTransitionTable), aFsm_Avm_StateTransitionTable,
     ELEMENTE(aFsm_Avm_StateTable)          , aFsm_Avm_StateTable,
     (tU32) TR_CLASS_DISPVIDCTRL_APPLICATION,
     DISPVIDCTRL_RVC_TRACE_ID_FSM_AVM)
, m_hTimerHandle_AvmReinit(OSAL_C_INVALID_HANDLE)
, m_hTimerHandle_ViewChangeRequestWait(OSAL_C_INVALID_HANDLE)
, m_hTimerHandle_CameraOffRequestDelay(OSAL_C_INVALID_HANDLE)
, m_hTimerHandle_FsmStateChange(OSAL_C_INVALID_HANDLE)
, m_poControl_Avm(poControl_Avm)
{
   if( OSAL_s32TimerCreate( (OSAL_tpfCallback)_pfCallbackTimer_AvmReinit, (tPVoid)this, &m_hTimerHandle_AvmReinit) != OSAL_OK)
   {
      ETG_TRACE_USR4(("ERROR - Could not create AvmReinit timer!"));
      m_hTimerHandle_AvmReinit = OSAL_C_INVALID_HANDLE;
   }
   if( OSAL_s32TimerCreate( (OSAL_tpfCallback)_pfCallbackTimer_ViewChangeRequestWait, (tPVoid)this, &m_hTimerHandle_ViewChangeRequestWait) != OSAL_OK)
   {
      ETG_TRACE_USR4(("ERROR - Could not create ViewChangeRequestWait timer!"));
      m_hTimerHandle_ViewChangeRequestWait = OSAL_C_INVALID_HANDLE;
   }
   if( OSAL_s32TimerCreate( (OSAL_tpfCallback)_pfCallbackTimer_CameraOffRequestDelay, (tPVoid)this, &m_hTimerHandle_CameraOffRequestDelay) != OSAL_OK)
   {
      ETG_TRACE_USR4(("ERROR - Could not create CameraOffRequestDelay timer!"));
      m_hTimerHandle_CameraOffRequestDelay = OSAL_C_INVALID_HANDLE;
   }
   // start timer for FSM state timeouts used for state verification
   if( OSAL_s32TimerCreate( (OSAL_tpfCallback)_pfCallbackTimer_FsmStateChange, (tPVoid)this, &m_hTimerHandle_FsmStateChange) != OSAL_OK)
   {
      ETG_TRACE_USR4(("ERROR - Could not create FSM timer!"));
      m_hTimerHandle_FsmStateChange = OSAL_C_INVALID_HANDLE;
   }
}


/******************************************************************************/
/* FUNCTION     ~dispvidctrl_tclFsm_Avm                                  */
/******************************************************************************/
/**
*  \brief       destructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclFsm_Avm::~dispvidctrl_tclFsm_Avm(tVoid)
{
   //stop and delete all timer
   if (OSAL_C_INVALID_HANDLE != m_hTimerHandle_AvmReinit)
   {
      (void) OSAL_s32TimerSetTime(m_hTimerHandle_AvmReinit, 0, 0);
      (void) OSAL_s32TimerDelete(m_hTimerHandle_AvmReinit);
   }

   if (OSAL_C_INVALID_HANDLE != m_hTimerHandle_ViewChangeRequestWait)
   {
      (void) OSAL_s32TimerSetTime(m_hTimerHandle_ViewChangeRequestWait, 0, 0);
      (void) OSAL_s32TimerDelete(m_hTimerHandle_ViewChangeRequestWait);
   }

   if (OSAL_C_INVALID_HANDLE != m_hTimerHandle_CameraOffRequestDelay)
   {
      (void) OSAL_s32TimerSetTime(m_hTimerHandle_CameraOffRequestDelay, 0, 0);
      (void) OSAL_s32TimerDelete(m_hTimerHandle_CameraOffRequestDelay);
   }

   if (OSAL_C_INVALID_HANDLE != m_hTimerHandle_FsmStateChange)
   {
      (void) OSAL_s32TimerSetTime(m_hTimerHandle_FsmStateChange, 0, 0);
      (void) OSAL_s32TimerDelete(m_hTimerHandle_FsmStateChange);
   }

   m_poInstance = NULL;
   m_poControl_Avm = NULL;
}


/******************************************************************************/
/* FUNCTION     tclCreateInstance                                             */
/******************************************************************************/
/**
*  \brief       Create instance (singleton pattern)
*
*  \param       pointer to main application
*  \return      pointer to instance
*/
/******************************************************************************/
dispvidctrl_tclFsm_Avm* dispvidctrl_tclFsm_Avm::tclCreateInstance(dispvidctrl_tclControl_Avm* poControl_Avm)
{
   if (m_poInstance == NULL) {
      m_poInstance = OSAL_NEW dispvidctrl_tclFsm_Avm(poControl_Avm);
   }
   return m_poInstance;
}


/******************************************************************************/
/* FUNCTION     vNewTrigger                                                   */
/******************************************************************************/
/**
*  \brief       To be called to enter a new Trigger
*
*  \param       Trigger and State
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vNewTrigger(tU32 u32Trigger, tBool bTriggerState) const
{
   if (NULL != m_poInstance) {
      m_poInstance->u32CalcNewFsmState(u32Trigger, bTriggerState);
   }
}


/******************************************************************************/
/* FUNCTION     vStateChangeDetected                                          */
/******************************************************************************/
/**
*  \brief       called when a state change happens
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vStateChangeDetected(tVoid)
{
   if (NULL != m_poInstance) {
      m_poInstance->vTraceStatusInfo(0);
   }

   return;
}


/*******************************************************************************
* FUNCTION   : dispvidctrl_tclFsm_Avm::vTransitionActionHandler
* DESCRIPTION: HandlerRoutine to execute transition action common for all stated
* PARAMETER  : timeout value in milliseconds
* RETURNVALUE: None
*******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vTransitionActionHandler(tU32 u32Trigger)
{
   if ( m_poInstance->m_poControl_Avm ) {
      switch (u32Trigger)
      {
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_AFFORDABLE_ITS:
            m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__AVM_UPDATE_AFFORDABLE_ITS, 0 );
            break;
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_BUTTON_PRESS:
            // Clear the Request if no repsonse received until Timeout (3s)
            m_poInstance->_bSetTimer_ViewChangeRequestWait(DISPVIDCTRL_AVM_CAMERA_ON_REQUEST_TIMEOUT);
            m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__CSM_WRITE_CAMERA_SWITCH_PRESS_DISPLAY_SWITCH_STATUS );
            break;
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_BUTTON_LONG_PRESS:
            m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__CSM_WRITE_CAMERA_SWITCH_KEEP_DISPLAY_SWITCH_STATUS );
            break;
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_NORMAL_BEEP:
            m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__AUDIO_SEND_OPERATION_BEEP );
            break;
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_ERROR_BEEP:
            m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__AUDIO_SEND_ERROR_BEEP );
            break;
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_SETTINGS_OFF:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_PURPOSE_SWITCH:
            m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__CSM_WRITE_CAMERA_OFF_REQUEST_DISPLAY_SWITCH_STATUS );
            break;
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_TIMEOUT_VIEW_CHANGE_REQUEST_WAIT:
            m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__CSM_WRITE_CLEAR_DISPLAY_REQUESTS );
            break;
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_VIEW_STATUS_UPDATE:
            // Repsonse received stop the running timer
            m_poInstance->_bSetTimer_ViewChangeRequestWait(0);
            m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__CSM_WRITE_CLEAR_DISPLAY_REQUESTS );
            break;
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_ERROR_TRUE:
           m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm(FSM_AVM_OUT__AVM_ERROR_ACTIVE);
           break;
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_ERROR_FALSE:
           m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm(FSM_AVM_OUT__AVM_ERROR_DEACTIVE);
           break;
         default:
            break;
      }
   }
}

/******************************************************************************/
/* FUNCTION     vUpdateRelatedTrigger                                         */
/******************************************************************************/
/**
*  \brief       to correct "hold" trigger states which are related to actual
*               trigger,
*               overwritten base class function
*
*  \param       actual Trigger
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vUpdateRelatedTrigger(tU32 u32Trigger)
{
   switch (u32Trigger)
   {
      case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_IGNITION_ON:
         vUpdateTrigger(eTrigger_AVM_EV_IGNITION_OFF, FALSE);
         break;
      case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_IGNITION_OFF:
         vUpdateTrigger(eTrigger_AVM_EV_IGNITION_ON, FALSE);
         break;

      case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_INIT_ACTIVE:
         vUpdateTrigger(eTrigger_AVM_EV_INIT_DONE, FALSE);
         break;
      case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_INIT_DONE:
         vUpdateTrigger(eTrigger_AVM_EV_INIT_ACTIVE, FALSE);
         break;

      case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_SCREEN_ON:
         vUpdateTrigger(eTrigger_AVM_EV_CAMERA_SCREEN_OFF, FALSE);
         break;
      case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_SCREEN_OFF:
         vUpdateTrigger(eTrigger_AVM_EV_CAMERA_SCREEN_ON, FALSE);
         break;

      case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAN_MSG_TIMEOUT_TRUE:
         vUpdateTrigger(eTrigger_AVM_EV_CAN_MSG_TIMEOUT_FALSE, FALSE);
         vUpdateTrigger(eTrigger_AVM_EV_INIT_DONE, FALSE);
         vUpdateTrigger(eTrigger_AVM_EV_INIT_ACTIVE, FALSE);
         break;
      case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAN_MSG_TIMEOUT_FALSE:
         vUpdateTrigger(eTrigger_AVM_EV_CAN_MSG_TIMEOUT_TRUE, FALSE);
         break;

      case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_REVERSE_ON_DEBOUNCED:
         vUpdateTrigger(eTrigger_AVM_EV_REVERSE_OFF, FALSE);
         break;
      case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_REVERSE_OFF:
         vUpdateTrigger(eTrigger_AVM_EV_REVERSE_ON_DEBOUNCED, FALSE);
         break;
      case eTrigger_AVM_EV_VIDEO_SIGNAL_FAILURE:
         vUpdateTrigger(eTrigger_AVM_EV_VIDEO_SIGNAL_RESTORED, FALSE);
         break;
      case eTrigger_AVM_EV_VIDEO_SIGNAL_RESTORED:
         vUpdateTrigger(eTrigger_AVM_EV_VIDEO_SIGNAL_FAILURE, FALSE);
         break;
         
      case eTrigger_AVM_EV_ERROR_TRUE:
        vUpdateTrigger(eTrigger_AVM_EV_ERROR_FALSE, FALSE);
        break;
      case eTrigger_AVM_EV_ERROR_FALSE:
        vUpdateTrigger(eTrigger_AVM_EV_ERROR_TRUE, FALSE);
        break;

      default:
         break;
   }
}

/*******************************************************************************
* FUNCTION   : dispvidctrl_tclFsm_Avm::vEntry_AvmUninitialised
* DESCRIPTION:
* PARAMETER  :
* RETURNVALUE: None
*******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vEntry_AvmUninitialised(tVoid)
{
// possible entry trigger are:
// eTrigger_AVM_EV_IGNITION_OFF
// eTrigger_AVM_EV_INIT_ACTIVE
// eTrigger_AVM_EV_CAN_MSG_TIMEOUT_TRUE

   if (NULL != m_poInstance) {
      if ( m_poInstance->m_poControl_Avm ) {
         
         // Visualisation could be on -> switch off
         tU32 u32OffReason = dispvidctrl_tclFsm_Hmi::EN_VD_RVC_HMI_VISUALISATION_OFF__SERVICE_REQUEST_OFF;
         
         if ( m_poInstance->u32GetCurrentTrigger() == eTrigger_AVM_EV_CAN_MSG_TIMEOUT_TRUE )
         {
            u32OffReason = dispvidctrl_tclFsm_Hmi::EN_VD_RVC_HMI_VISUALISATION_OFF__SYSTEM_MALFUNCTION;
         }
         
         m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__HMI_VISUALISATION_OFF, u32OffReason );
         m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__CSM_WRITE_IPA_ACTIVATION_FLAG_STATUS, FALSE );
         ETG_TRACE_USR1((" FSM-AVM  :: vEntry_AvmUninitialised() eTrigger_AVM_EV_IGNITION_OFF -> AVM_EV_IPA_REQST_NORMAL"));

         // set CAN message initializing bit
         m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__CSM_WRITE_INITIALISING_DISPLAY_SWITCH_STATUS, 1 );
         m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__CSM_REINIT_DISPLAY_SWITCH_STATUS );

         // disable HMI Menu entry
         // In case of no CAN communication no operation for the
         // Affordable ITS related feature (switching on/off MOD, BSW or LDW)
         // is possible. Therefore we want to switch of the HMI "Driving Aids" menu.
         m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__AVM_UPDATE_AFFORDABLE_ITS, 1 );
         // IPA state is initializing
         m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__AVM_UPDATE_IPA_STATUS, DISPVIDCTRL_CONTROL_IPA__INITIALIZING);

         if ( m_poInstance->u32GetCurrentTrigger() == eTrigger_AVM_EV_IGNITION_OFF )
         {
            // With Ignition == OFF AVM will stop CAN message transmission.
            // A active visualisation request will be not reset before.
            // At the earliest with CAN message timeout.
            // If Ignition is switched to ON before CAN message timeout we would continuoue with this
            // signal values (which could be wrong) and reactivate AVM screen.
            // We need to invalidate CAN signals values/states which can trigger a Visualisation (Initializing, ViewChangeRequest).
            // The CSM will indicate the "Initalizing" signal always with every message receiption.
            // So we will get a update after Ignition ON with first message from AVM.
            m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__INVALIDATE_CSM_RX);
         }
      }
   }
}

/*******************************************************************************
* FUNCTION   : dispvidctrl_tclFsm_Avm::vExit_AvmUninitialised
* DESCRIPTION:
* PARAMETER  :
* RETURNVALUE: None
*******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vExit_AvmUninitialised(tVoid)
{
// possible exit trigger are:
// eTrigger_AVM_EV_INIT_DONE [GUARD: IGNITION_ON]

   if (NULL != m_poInstance)
   {
      // stop reinit timer
      if (m_poInstance->_bSetTimer_AvmReinit(0) == FALSE)   {
         ETG_TRACE_ERR((" FSM-AVM  :: E R R O R in vExit_AvmUninitialised():   AVM Reinit Timer stop failed!           "));
      }

      if ( m_poInstance->m_poControl_Avm ) {
         // clear CAN message initializing bit
         m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__CSM_WRITE_INITIALISING_DISPLAY_SWITCH_STATUS, 0 );

         // check if ViewChangeRequest bit is set (could be already set from AVM during initialization)
         m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__AVM_VIEW_CHANGE_REQUEST_CHANGE );

         // enable/disable HMI Menu entry depending on CAN signal value
          m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__AVM_UPDATE_AFFORDABLE_ITS, 0 );

          // IPA state is Available if configured
          m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__AVM_UPDATE_IPA_STATUS, DISPVIDCTRL_CONTROL_IPA__AVAILABLE);
      }
   }
}

/*******************************************************************************
* FUNCTION   : dispvidctrl_tclFsm_Avm::vDo_AvmUninitialised
* DESCRIPTION:
* PARAMETER  :
* RETURNVALUE: None
*******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vDo_AvmUnitialized(tVoid)
{
// trigger we have to process while state is active without doing a state change
// eTrigger_AVM_EV_IGNITION_ON
// eTrigger_AVM_EV_CAN_MSG_TIMEOUT_FALSE
// eTrigger_AVM_EV_REINIT_TIMEOUT

// trigger we have to process to stay in this state
// (otherwise the statetransition table will find an "in all states" entry (200 - 202)
// eTrigger_AVM_EV_IGNITION_OFF
// eTrigger_AVM_EV_CAN_MSG_TIMEOUT_TRUE
// eTrigger_AVM_EV_INIT_ACTIVE

   if (NULL != m_poInstance)
   {

      tU32 u32CurrentTrigger;
      u32CurrentTrigger = m_poInstance->u32GetCurrentTrigger();

      switch (u32CurrentTrigger)
      {
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_IGNITION_ON:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_REINIT_TIMEOUT:
            {
               if ( m_poInstance->m_poControl_Avm ) {
                  // check AVM initialization status
                  m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__CSM_CHECK_INTIALISING_SIGNAL );

                  // start re-init timer for verification in case
                  // no init active/done event will come
                  if (m_poInstance->_bSetTimer_AvmReinit(DISPVIDCTRL_AVM_REINIT_TIME_IN_MS) == FALSE) {
                     ETG_TRACE_ERR((" FSM-AVM  :: E R R O R in vExit_AvmUninitialised():   AVM Reinit Timer start failed!          "));
                  }
               }
            }
            break;
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAN_MSG_TIMEOUT_FALSE:
            {
               // In case of CAN message timeout FALSE we have to make sure, that AVM unit can receive at least
               // one message where the initialize bit is set.
               // (Because we may have reset other signals of the message and AVM would handle this as a request
               // if initialize bit is not set (if AVM can not detect that we have done a reinit)
               // The time raster is 500 ms for 5FE message, so timer should be a little bit more.
               // restart re-init timer for verification with a timeout value > 500 ms
               if (m_poInstance->_bSetTimer_AvmReinit(DISPVIDCTRL_AVM_DELAY_OF_CAN_MSG_TIMOUT_FALSE_TIME_IN_MS) == FALSE) {
                  ETG_TRACE_ERR((" FSM-AVM  :: E R R O R in vExit_AvmUninitialised():   AVM Reinit Timer start failed!          "));
               }
            }
            break;
         case eTrigger_AVM_EV_CAN_MSG_TIMEOUT_TRUE:
         case eTrigger_AVM_EV_IGNITION_OFF:
         case eTrigger_AVM_EV_INIT_ACTIVE:
            {
               //To be in sync with Hmi FSM during enabling
               m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__HMI_VISUALISATION_OFF );
            }
            break;
         default:
            break;
      }

   }
}

/*******************************************************************************
* FUNCTION   : dispvidctrl_tclFsm_Avm::vEntry_AvmOff
* DESCRIPTION:
* PARAMETER  :
* RETURNVALUE: None
*******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vEntry_AvmOff(tVoid)
{
// possible entry trigger are:
// eTrigger_AVM_EV_INIT_DONE
// eTrigger_AVM_EV_CAMERA_SCREEN_OFF
// eTrigger_AVM_EV_ERROR_BEEP
// eTrigger_AVM_EV_TIMEOUT_VIEW_CHANGE_REQUEST_WAIT
// eTrigger_AVM_VIDEO_SIGNAL_FAILURE

   if (NULL != m_poInstance && NULL != m_poInstance->m_poControl_Avm)
   {
      tU32 u32CurrentTrigger;
      u32CurrentTrigger = m_poInstance->u32GetCurrentTrigger();

      // reset old trigger used for guard conditions of exit triggers
      // HW_START_OK and GRAPHICS_INIT_OK we don't reset because they will come only once during start up
      m_poInstance->vUpdateTrigger(eTrigger_AVM_EV_CAMERA_SCREEN_ON, FALSE);
      
      m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__CSM_WRITE_IPA_ACTIVATION_FLAG_STATUS, FALSE );
      
      // if View Scene exit because of the error condition, inform the Ipa Unit that we have switched Off
      if ( TRUE == m_poInstance->bGetTriggerState( eTrigger_AVM_EV_ERROR_TRUE ) )
      {
         m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__CSM_WRITE_CAMERA_OFF_REQUEST_DISPLAY_SWITCH_STATUS );
         m_poInstance->vUpdateTrigger(eTrigger_AVM_EV_ERROR_TRUE, FALSE);
      }

      // additional state specific entry actions
      switch (u32CurrentTrigger)
      {
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_SCREEN_OFF:
            {
                if ( m_poInstance->m_poControl_Avm ) {
                  // Visualisation is on -> switch off
                  m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__CSM_WRITE_CLEAR_DISPLAY_REQUESTS );
                  m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__HMI_VISUALISATION_OFF );
                }
            }
            break;

         default:
            break;
      }

   }
}

/*******************************************************************************
* FUNCTION   : dispvidctrl_tclFsm_Avm::vExit_AvmOff
* DESCRIPTION:
* PARAMETER  :
* RETURNVALUE: None
*******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vExit_AvmOff(tVoid)
{
// possible exit trigger are:
// eTrigger_AVM_EV_CAMERA_SCREEN_ON
// eTrigger_AVM_EV_CAMERA_SETTINGS_ON

// eTrigger_AVM_EV_IGNITION_OFF
// eTrigger_AVM_EV_CAN_MSG_TIMEOUT_TRUE
// eTrigger_AVM_EV_INIT_ACTIVE

   if (NULL != m_poInstance)
   {

      tU32 u32CurrentTrigger;
      u32CurrentTrigger = m_poInstance->u32GetCurrentTrigger();

      switch (u32CurrentTrigger)
      {
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_SETTINGS_ON:
            {
               if ( m_poInstance->m_poControl_Avm ) {
                   m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__CSM_WRITE_CAMERA_SWITCH_PRESS_DISPLAY_SWITCH_STATUS );
               }
            }
            break;

         default:
            break;
      }

   }
}

/*******************************************************************************
* FUNCTION   : dispvidctrl_tclFsm_Avm::vDo_AvmOff
* DESCRIPTION:
* PARAMETER  :
* RETURNVALUE: None
*******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vDo_AvmOff(tVoid)
{
// trigger we have to process while state is active without doing a state change
// eTrigger_AVM_EV_CAMERA_SCREEN_OFF
// eTrigger_AVM_EV_AFFORDABLE_ITS
// eTrigger_AVM_EV_CAMERA_BUTTON_PRESS
// eTrigger_AVM_EV_CAMERA_BUTTON_LONG_PRESS
// eTrigger_AVM_EV_NORMAL_BEEP
// eTrigger_AVM_EV_ERROR_BEEP

   if (NULL != m_poInstance)
   {

      tU32 u32CurrentTrigger;
      u32CurrentTrigger = m_poInstance->u32GetCurrentTrigger();

      switch (u32CurrentTrigger)
      {
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_SCREEN_OFF:

            {
               // Visualisation may be on -> switch off
               if ( m_poInstance->m_poControl_Avm ) {
                  m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__HMI_VISUALISATION_OFF );
               }
            }
            break;
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_AFFORDABLE_ITS:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_BUTTON_LONG_PRESS:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_TIMEOUT_VIEW_CHANGE_REQUEST_WAIT:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_BUTTON_PRESS:
             m_poInstance->vTransitionActionHandler(u32CurrentTrigger);
             break;

         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_NORMAL_BEEP:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_ERROR_BEEP:
            // NCG3D-26735: If switched Off due to Camera Settings do not request for Audio confirmation beep.
            if ((m_poInstance->_u32LastFsmState != dispvidctrl_tclFsm_Avm::eState_AVM_ON_CAMERA_SETTINGS) &&
                (m_poInstance->_u32LastFsmState != dispvidctrl_tclFsm_Avm::eState_AVM_OFF_CAMERA_SETTINGS_WAIT_FOR_ON))
               m_poInstance->vTransitionActionHandler(u32CurrentTrigger);
            else
                ETG_TRACE_USR1((" FSM-AVM  :: W A R N I N G vDo_AvmOff() Audio Beep Not triggered because coming back from Camera Settings!"));
            break;
            
         

         default:
            break;
      }
   }
}

/*******************************************************************************
* FUNCTION   : dispvidctrl_tclFsm_Avm::vEntry_AvmOn
* DESCRIPTION:
* PARAMETER  :
* RETURNVALUE: None
*******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vEntry_AvmOn(tVoid)
{
// possible entry trigger are:
// eTrigger_AVM_EV_CAMERA_SCREEN_ON

// eTrigger_AVM_EV_CAMERA_SCREEN_ON [Guard: eTrigger_AVM_EV_REVERSE_ON_DEBOUNCED == TRUE]
// eTrigger_AVM_EV_CAMERA_BUTTON_PRESS
// eTrigger_AVM_EV_CAMERA_BUTTON_LONG_PRESS
// eTrigger_AVM_EV_TIMEOUT_CAMERA_OFF_REQUEST_DELAY

   if (NULL != m_poInstance)
   {

      tU32 u32CurrentTrigger;
      u32CurrentTrigger = m_poInstance->u32GetCurrentTrigger();

      switch (u32CurrentTrigger)
      {
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_SCREEN_ON:
            {
                // (Needed for AVM C1A to clear the SwitchRequest_AVM)
                m_poInstance->_bSetTimer_ViewChangeRequestWait(0); // Stop already running timer 
                m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__CSM_WRITE_CLEAR_DISPLAY_REQUESTS );

                if ( m_poInstance->m_poControl_Avm ) {
                  // post event for switching on the visualisation
                  m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__CSM_WRITE_IPA_ACTIVATION_FLAG_STATUS, TRUE );
                  m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__HMI_VISUALISATION_ON );
                }
            }
            break;

         default:
            break;
      }
   }
}

/*******************************************************************************
* FUNCTION   : dispvidctrl_tclFsm_Avm::vExit_AvmOn
* DESCRIPTION:
* PARAMETER  :
* RETURNVALUE: None
*******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vExit_AvmOn(tVoid)
{
// possible exit trigger are:
// eTrigger_AVM_EV_CAMERA_SCREEN_OFF
// eTrigger_AVM_EV_CAMERA_SETTINGS_ON

// eTrigger_AVM_EV_IGNITION_OFF
// eTrigger_AVM_EV_CAN_MSG_TIMEOUT_TRUE
// eTrigger_AVM_EV_INIT_ACTIVE

   if (NULL != m_poInstance)
   {
      tU32 u32CurrentTrigger = m_poInstance->u32GetCurrentTrigger();

      switch (u32CurrentTrigger)
      {
         default:
            break;
      }
   }
}

/*******************************************************************************
* FUNCTION   : dispvidctrl_tclFsm_Avm::vDo_AvmOn
* DESCRIPTION:
* PARAMETER  :
* RETURNVALUE: None
*******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vDo_AvmOn(tVoid)
{
// trigger we have to process while state is active without doing a state change
// eTrigger_AVM_EV_VERIFY_AFTER_VIEW_STATUS_STOPPED
// eTrigger_AVM_EV_TIMEOUT_VIEW_CHANGE_REQUEST_WAIT
// eTrigger_AVM_EV_CAMERA_BUTTON_PRESS
// eTrigger_AVM_EV_CAMERA_BUTTON_LONG_PRESS
// eTrigger_AVM_EV_NORMAL_BEEP
// eTrigger_AVM_EV_ERROR_BEEP
// eTrigger_AVM_EV_PURPOSE_SWITCH
// eTrigger_AVM_EV_AFFORDABLE_ITS
// eTrigger_AVM_EV_CAMERA_SETTINGS_OFF
// eTrigger_AVM_EV_VIEW_STATUS_UPDATE

   if (NULL != m_poInstance)
   {

      tU32 u32CurrentTrigger;
      u32CurrentTrigger = m_poInstance->u32GetCurrentTrigger();

      switch (u32CurrentTrigger)
      {
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_VERIFY_AFTER_VIEW_STATUS_STOPPED:
            {
               if ( m_poInstance->m_poControl_Avm ) {
                  // post event for switching on the visualisation
                  m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__HMI_VISUALISATION_ON );
               }
            }
            break;
         
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_BUTTON_PRESS:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_BUTTON_LONG_PRESS:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_NORMAL_BEEP:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_ERROR_BEEP:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_PURPOSE_SWITCH:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_AFFORDABLE_ITS:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_TIMEOUT_VIEW_CHANGE_REQUEST_WAIT:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_SETTINGS_OFF:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_VIEW_STATUS_UPDATE:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_ERROR_TRUE:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_ERROR_FALSE:
            m_poInstance->vTransitionActionHandler(u32CurrentTrigger);
            break;

         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_VIDEO_SIGNAL_RESTORED:
            if ( m_poInstance->m_poControl_Avm ) {
              //m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__AVM_UPDATE_IPA_STATUS, DISPVIDCTRL_CONTROL_IPA__AVAILABLE);
            }
            break;
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_VIDEO_SIGNAL_FAILURE:
            if ( m_poInstance->m_poControl_Avm ) {
              //m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__AVM_UPDATE_IPA_STATUS, DISPVIDCTRL_CONTROL_IPA__MALFUNCTION);
            }
            break;

         default:
            break;
      }
   }
}

/*******************************************************************************
* FUNCTION   : dispvidctrl_tclFsm_Avm::vEntry_AvmOnWaitForReverseOff
* DESCRIPTION:
* PARAMETER  :
* RETURNVALUE: None
*******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vEntry_AvmOnWaitForReverseOff(tVoid)
{
// possible entry trigger are:
// eTrigger_AVM_EV_REVERSE_ON_DEBOUNCED

   // nothing to do currently
}

/*******************************************************************************
* FUNCTION   : dispvidctrl_tclFsm_Avm::vExit_AvmOnWaitForReverseOff
* DESCRIPTION:
* PARAMETER  :
* RETURNVALUE: None
*******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vExit_AvmOnWaitForReverseOff(tVoid)
{
// possible exit trigger are:
// eTrigger_AVM_EV_CAMERA_BUTTON_PRESS
// eTrigger_AVM_EV_CAMERA_BUTTON_LONG_PRESS
// eTrigger_AVM_EV_TIMEOUT_CAMERA_OFF_REQUEST_DELAY
// eTrigger_AVM_EV_CAMERA_SCREEN_OFF

// eTrigger_AVM_EV_IGNITION_OFF
// eTrigger_AVM_EV_CAN_MSG_TIMEOUT_TRUE
// eTrigger_AVM_EV_INIT_ACTIVE

   if (NULL != m_poInstance)
   {
      tU32 u32CurrentTrigger;
      u32CurrentTrigger = m_poInstance->u32GetCurrentTrigger();

      // stop timer
      (tVoid) m_poInstance->_bSetTimer_CameraOffRequestDelay(0);

      switch (u32CurrentTrigger)
      {
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_BUTTON_PRESS:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_BUTTON_LONG_PRESS:
            m_poInstance->vTransitionActionHandler(u32CurrentTrigger);
            break;
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_TIMEOUT_CAMERA_OFF_REQUEST_DELAY:
            {
               if ( m_poInstance->m_poControl_Avm ) {
                   m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__CSM_WRITE_CAMERA_OFF_REQUEST_DISPLAY_SWITCH_STATUS );
               }
            }
            break;
            
         case eTrigger_AVM_EV_CAN_MSG_TIMEOUT_TRUE:
         case eTrigger_AVM_EV_INIT_ACTIVE:
            break;

         default:
            break;
      }
   }
}

/*******************************************************************************
* FUNCTION   : dispvidctrl_tclFsm_Avm::vDo_AvmOnWaitForReverseOff
* DESCRIPTION:
* PARAMETER  :
* RETURNVALUE: None
*******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vDo_AvmOnWaitForReverseOff(tVoid)
{
// trigger we have to process while state is active without doing a state change
// eTrigger_AVM_EV_VERIFY_AFTER_VIEW_STATUS_STOPPED
// eTrigger_AVM_EV_NORMAL_BEEP
// eTrigger_AVM_EV_ERROR_BEEP
// eTrigger_AVM_EV_PURPOSE_SWITCH
// eTrigger_AVM_EV_AFFORDABLE_ITS
// eTrigger_AVM_EV_REVERSE_OFF
// eTrigger_AVM_EV_REVERSE_ON_DEBOUNCED
// eTrigger_AVM_EV_CAMERA_SETTINGS_OFF

   if (NULL != m_poInstance)
   {

      tU32 u32CurrentTrigger;
      u32CurrentTrigger = m_poInstance->u32GetCurrentTrigger();

      switch (u32CurrentTrigger)
      {
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_NORMAL_BEEP:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_ERROR_BEEP:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_PURPOSE_SWITCH:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_AFFORDABLE_ITS:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_TIMEOUT_VIEW_CHANGE_REQUEST_WAIT:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_SETTINGS_OFF:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_ERROR_TRUE:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_ERROR_FALSE:
            m_poInstance->vTransitionActionHandler(u32CurrentTrigger);
            break;
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_REVERSE_OFF:
            {
               //start CameraOffRequest delay timer
               if (m_poInstance->_bSetTimer_CameraOffRequestDelay(DISPVIDCTRL_AVM_CAMERA_OFF_REQUEST_DELAY) == FALSE)
               {
                  ETG_TRACE_ERR((" FSM-AVM  :: E R R O R vExit_AvmOnWaitForReverseOff() CameraOffRequestDelay TimerStart failed!"));
               }
            }
            break;
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_REVERSE_ON_DEBOUNCED:
            {
               //stop CameraOffRequest delay timer
               if (m_poInstance->_bSetTimer_CameraOffRequestDelay(0) == FALSE)
               {
                  ETG_TRACE_ERR((" FSM-AVM  :: E R R O R vExit_AvmOnWaitForReverseOff() CameraOffRequestDelay TimerStop failed!"));
               }
            }
            break;
         case eTrigger_AVM_EV_VERIFY_AFTER_VIEW_STATUS_STOPPED:
            {
               if ( m_poInstance->m_poControl_Avm ) {
                  // post event for switching on the visualisation
                  m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__HMI_VISUALISATION_ON );
               }
            }
            break;

         default:
            break;
      }
   }
}

/*******************************************************************************
* FUNCTION   : dispvidctrl_tclFsm_Avm::vEntry_AvmOffCameraSettingsWaitForOn
* DESCRIPTION:
* PARAMETER  :
* RETURNVALUE: None
*******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vEntry_AvmOffCameraSettingsWaitForOn(tVoid)
{
// possible entry trigger are:
// eTrigger_AVM_EV_CAMERA_SETTINGS_ON

   if (NULL != m_poInstance)
   {
      // start timer
      (tVoid) m_poInstance->_bSetTimer_ViewChangeRequestWait(DISPVIDCTRL_AVM_CAMERA_ON_REQUEST_TIMEOUT);
   }
}

/*******************************************************************************
* FUNCTION   : dispvidctrl_tclFsm_Avm::vExit_AvmOffCameraSettingsWaitForOn
* DESCRIPTION:
* PARAMETER  :
* RETURNVALUE: None
*******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vExit_AvmOffCameraSettingsWaitForOn(tVoid)
{
// possible exit trigger are:
// eTrigger_AVM_TIMEOUT
// eTrigger_AVM_EV_ERROR_BEEP
// eTrigger_AVM_EV_CAMERA_SCREEN_ON [GUARD: REVERSE_ON_DEBOUNCED == TRUE]
//                                  [GUARD: REVERSE_ON_DEBOUNCED == FALSE]

// eTrigger_AVM_EV_IGNITION_OFF
// eTrigger_AVM_EV_CAN_MSG_TIMEOUT_TRUE
// eTrigger_AVM_EV_INIT_ACTIVE

   if (NULL != m_poInstance)
   {

      tU32 u32CurrentTrigger;
      u32CurrentTrigger = m_poInstance->u32GetCurrentTrigger();

      // stop timer
      (tVoid) m_poInstance->_bSetTimer_ViewChangeRequestWait(0);

      // reset Trigger
      m_poInstance->vUpdateTrigger(eTrigger_AVM_EV_CAMERA_SETTINGS_ON, FALSE);

      switch (u32CurrentTrigger)
      {
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_ERROR_BEEP:
            m_poInstance->vTransitionActionHandler(u32CurrentTrigger);
            break;

#ifndef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
         // Switch Off the request for Renault Only
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_TIMEOUT_VIEW_CHANGE_REQUEST_WAIT:
            m_poInstance->vTransitionActionHandler(u32CurrentTrigger);
            break;
#endif

         default:
            break;
      }
   }
}

/*******************************************************************************
* FUNCTION   : dispvidctrl_tclFsm_Avm::vDo_AvmOffCameraSettingsWaitForOn
* DESCRIPTION:
* PARAMETER  :
* RETURNVALUE: None
*******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vDo_AvmOffCameraSettingsWaitForOn(tVoid)
{
// trigger we have to process while state is active without doing a state change
// eTrigger_AVM_EV_AFFORDABLE_ITS

   if (NULL != m_poInstance)
   {
      tU32 u32CurrentTrigger;
      u32CurrentTrigger = m_poInstance->u32GetCurrentTrigger();

      switch (u32CurrentTrigger)
      {
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_AFFORDABLE_ITS:
            m_poInstance->vTransitionActionHandler(u32CurrentTrigger);
            break;

         default:
            break;
      }
   }
}

/*******************************************************************************
* FUNCTION   : dispvidctrl_tclFsm_Avm::vEntry_AvmOnCameraSettings
* DESCRIPTION:
* PARAMETER  :
* RETURNVALUE: None
*******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vEntry_AvmOnCameraSettings(tVoid)
{
// possible entry trigger are:
// eTrigger_AVM_EV_CAMERA_SCREEN_ON [Guard: eTrigger_AVM_EV_REVERSE_ON_DEBOUNCED == FALSE]

   if (NULL != m_poInstance)
   {
      if ( m_poInstance->m_poControl_Avm ) {
         // post event for switching on the visualisation
         m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__CSM_WRITE_IPA_ACTIVATION_FLAG_STATUS, TRUE );
         m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__HMI_VISUALISATION_ON_CAMERA_SETTINGS );
      }
   }
}

/*******************************************************************************
* FUNCTION   : dispvidctrl_tclFsm_Avm::vExit_AvmOnCameraSettings
* DESCRIPTION:
* PARAMETER  :
* RETURNVALUE: None
*******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vExit_AvmOnCameraSettings(tVoid)
{
// possible exit trigger are:
// eTrigger_AVM_EV_REVERSE_ON_DEBOUNCED
// eTrigger_AVM_EV_VIDEO_SIGNAL_FAILURE
// eTrigger_AVM_EV_CAMERA_SCREEN_OFF

// eTrigger_AVM_EV_IGNITION_OFF
// eTrigger_AVM_EV_CAN_MSG_TIMEOUT_TRUE
// eTrigger_AVM_EV_INIT_ACTIVE

   if (NULL != m_poInstance)
   {
      tU32 u32CurrentTrigger;
      u32CurrentTrigger = m_poInstance->u32GetCurrentTrigger();

      switch (u32CurrentTrigger)
      {
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_VIDEO_SIGNAL_FAILURE:
            {
            }
            break;

         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_REVERSE_ON_DEBOUNCED:
            {
               if ( m_poInstance->m_poControl_Avm ) {
                  // set AVM View On
                  // post event for switching on the visualisation
                  m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__HMI_VISUALISATION_ON );
               }
            }
            break;
         default:
            break;
      }
   }
}

/*******************************************************************************
* FUNCTION   : dispvidctrl_tclFsm_Avm::vDo_AvmOnCameraSettings
* DESCRIPTION:
* PARAMETER  :
* RETURNVALUE: None
*******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vDo_AvmOnCameraSettings(tVoid)
{
// trigger we have to process while state is active without doing a state change
// eTrigger_AVM_EV_ERROR_BEEP
// eTrigger_AVM_EV_AFFORDABLE_ITS
// eTrigger_AVM_EV_PURPOSE_SWITCH
// eTrigger_AVM_EV_VERIFY_AFTER_VIEW_STATUS_STOPPED
// eTrigger_AVM_EV_TIMEOUT_VIEW_CHANGE_REQUEST_WAIT
// eTrigger_AVM_EV_CAMERA_SETTINGS_OFF

   if (NULL != m_poInstance)
   {

      tU32 u32CurrentTrigger;
      u32CurrentTrigger = m_poInstance->u32GetCurrentTrigger();

      switch (u32CurrentTrigger)
      {
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_ERROR_BEEP:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_AFFORDABLE_ITS:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_PURPOSE_SWITCH:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_TIMEOUT_VIEW_CHANGE_REQUEST_WAIT:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_SETTINGS_OFF:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_ERROR_TRUE:
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_ERROR_FALSE:
            m_poInstance->vTransitionActionHandler(u32CurrentTrigger);
            break;

         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_VERIFY_AFTER_VIEW_STATUS_STOPPED:
            {
               if ( m_poInstance->m_poControl_Avm ) {
                   m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__CSM_WRITE_CAMERA_OFF_REQUEST_DISPLAY_SWITCH_STATUS );
               }
            }
            break;

          default:
            break;
      }

   }
}

/*******************************************************************************
* FUNCTION   : vEntry_AvmOnCameraSettingsTemporal
* DESCRIPTION:
* PARAMETER  :
* RETURNVALUE: None
*******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vEntry_AvmOnCameraSettingsTemporal(tVoid)
{
   if (NULL != m_poInstance)
   {
      tU32 u32CurrentTrigger;
      u32CurrentTrigger = m_poInstance->u32GetCurrentTrigger();

      // additional state specific entry actions
      switch (u32CurrentTrigger)
      {
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_SETTINGS_ON:
            {
                if ( m_poInstance->m_poControl_Avm ) {
                  // Visualisation is on Settings -> switch on now
                  m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__HMI_VISUALISATION_ON_CAMERA_SETTINGS );
                }
            }
            break;

         default:
            break;
      }
   }
}

/*******************************************************************************
* FUNCTION   : vExit_AvmOnCameraSettingsTemporal
* DESCRIPTION:
* PARAMETER  :
* RETURNVALUE: None
*******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vExit_AvmOnCameraSettingsTemporal(tVoid)
{
   if (NULL != m_poInstance)
   {
      tU32 u32CurrentTrigger;
      u32CurrentTrigger = m_poInstance->u32GetCurrentTrigger();

      // additional state specific entry actions
      switch (u32CurrentTrigger)
      {
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_SCREEN_OFF:
            {
                if ( m_poInstance->m_poControl_Avm ) {
                  // Visualisation is on -> switch off
                  m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__HMI_VISUALISATION_OFF );
                }
            }
            break;
            
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_SETTINGS_OFF:
            {
                if ( m_poInstance->m_poControl_Avm ) {
                  // Visualisation is on Settings -> switch on now
                  m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm( FSM_AVM_OUT__HMI_VISUALISATION_ON );
                }
            }
            break;

         default:
            break;
      }
   }
}

/*******************************************************************************
* FUNCTION   : vDo_AvmOnCameraSettingsTemporal
* DESCRIPTION:
* PARAMETER  :
* RETURNVALUE: None
*******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vDo_AvmOnCameraSettingsTemporal(tVoid)
{
   if (NULL != m_poInstance)
   {
      tU32 u32CurrentTrigger;
      u32CurrentTrigger = m_poInstance->u32GetCurrentTrigger();

      // additional state specific entry actions
      switch (u32CurrentTrigger)
      {
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_VIDEO_SIGNAL_FAILURE:
            break;
            
         case dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_VIDEO_SIGNAL_RESTORED:
            break;

         default:
            break;
      }
   }
   
}


/******************************************************************************/
/* FUNCTION     _pfCallbackTimer_ViewChangeRequestWait                               */
/******************************************************************************/
/**
*  \brief
*
*  \param       pArg - pointer to this class
*  \return      none
*/
/******************************************************************************/
OSAL_tpfCallback dispvidctrl_tclFsm_Avm::_pfCallbackTimer_AvmReinit(tVoid* pArg)
{
   (tVoid) pArg;

   //ETG_TRACE_USR4(("  ====>>  : _pfCallbackTimer_AvmReinit()"));

   if ((m_poInstance) && (m_poInstance->m_poControl_Avm)) {
      m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm(FSM_AVM_OUT__FSM_AVM_INPUT_EVENT,
                                                             (tU32) eTrigger_AVM_EV_REINIT_TIMEOUT );
   }

   return 0;
};

/******************************************************************************/
/* FUNCTION     _bSetTimer_AvmReinit                                          */
/******************************************************************************/
/**
*  \brief       start or stop the timer
*
*  \param       timeout value
*  \return      tBool - timer start success state
*/
/******************************************************************************/
tBool  dispvidctrl_tclFsm_Avm::_bSetTimer_AvmReinit( OSAL_tMSecond msTimeout )
{
    tBool bSuccess = TRUE;
    if ( OSAL_C_INVALID_HANDLE != m_hTimerHandle_AvmReinit )
    {
        if ( OSAL_s32TimerSetTime( m_hTimerHandle_AvmReinit, msTimeout, 0 ) != OSAL_OK )
        {
            bSuccess = FALSE;
        }
    }
    return(bSuccess);
}

/******************************************************************************/
/* FUNCTION     _pfCallbackTimer_ViewChangeRequestWait                               */
/******************************************************************************/
/**
*  \brief
*
*  \param       pArg - pointer to this class
*  \return      none
*/
/******************************************************************************/
OSAL_tpfCallback dispvidctrl_tclFsm_Avm::_pfCallbackTimer_ViewChangeRequestWait(tVoid* pArg)
{
   (tVoid) pArg;

   //ETG_TRACE_USR4(("  ====>>  : _pfCallbackTimer_ViewChangeRequestWait()"));

   //eTrigger_AVM_EV_TIMEOUT_VIEW_CHANGE_REQUEST_WAIT
   if ((m_poInstance) && (m_poInstance->m_poControl_Avm))
   {
         m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm(FSM_AVM_OUT__FSM_AVM_INPUT_EVENT,
                                                                         (tU32) eTrigger_AVM_EV_TIMEOUT_VIEW_CHANGE_REQUEST_WAIT);
   }

   return 0;
};

/******************************************************************************/
/* FUNCTION     _bSetTimer_ViewChangeRequestWait                              */
/******************************************************************************/
/**
*  \brief       start or stop the timer
*
*  \param       timeout value
*  \return      tBool - timer start success state
*/
/******************************************************************************/
tBool  dispvidctrl_tclFsm_Avm::_bSetTimer_ViewChangeRequestWait( OSAL_tMSecond msTimeout )
{
    tBool bSuccess = TRUE;
    if ( OSAL_C_INVALID_HANDLE != m_hTimerHandle_ViewChangeRequestWait )
    {
        if ( OSAL_s32TimerSetTime( m_hTimerHandle_ViewChangeRequestWait, msTimeout, 0 ) != OSAL_OK )
        {
            bSuccess = FALSE;
        }
    }
    return(bSuccess);
}

/******************************************************************************/
/* FUNCTION     _pfCallbackTimer_CameraOffRequestDelay                               */
/******************************************************************************/
/**
*  \brief
*
*  \param       pArg - pointer to this class
*  \return      none
*/
/******************************************************************************/
OSAL_tpfCallback dispvidctrl_tclFsm_Avm::_pfCallbackTimer_CameraOffRequestDelay(tVoid* pArg)
{
   (tVoid) pArg;

   //ETG_TRACE_USR4(("  ====>>  : _pfCallbackTimer_CameraOffRequestDelay()"));

   //eTrigger_StateTimeout
   if ((m_poInstance) && (m_poInstance->m_poControl_Avm)) {
         m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm(FSM_AVM_OUT__FSM_AVM_INPUT_EVENT,
                                                                         (tU32) eTrigger_AVM_EV_TIMEOUT_CAMERA_OFF_REQUEST_DELAY);

   }

   return 0;
};

/******************************************************************************/
/* FUNCTION     _bSetTimer_CameraOffRequestDelay                                     */
/******************************************************************************/
/**
*  \brief       start or stop the timer
*
*  \param       timeout value
*  \return      tBool - timer start success state
*/
/******************************************************************************/
tBool  dispvidctrl_tclFsm_Avm::_bSetTimer_CameraOffRequestDelay( OSAL_tMSecond msTimeout )
{
    tBool bSuccess = TRUE;
    if ( OSAL_C_INVALID_HANDLE != m_hTimerHandle_CameraOffRequestDelay )
    {
        if ( OSAL_s32TimerSetTime( m_hTimerHandle_CameraOffRequestDelay, msTimeout, 0 ) != OSAL_OK )
        {
            bSuccess = FALSE;
        }
    }
    return(bSuccess);
}

/******************************************************************************/
/* FUNCTION     _pfCallbackTimer_FsmStateChange                               */
/******************************************************************************/
/**
*  \brief
*
*  \param       pArg - pointer to this class
*  \return      none
*/
/******************************************************************************/
OSAL_tpfCallback dispvidctrl_tclFsm_Avm::_pfCallbackTimer_FsmStateChange(tVoid* pArg)
{
   (tVoid) pArg;

   //ETG_TRACE_USR4(("  ====>>  : _pfCallbackTimer_FsmStateChange()"));

   //eTrigger_StateTimeout
   if ((m_poInstance) && (m_poInstance->m_poControl_Avm)) {
      m_poInstance->m_poControl_Avm->vHandleFsmOutEvent_Avm(FSM_AVM_OUT__FSM_AVM_INPUT_EVENT,
                                                                         (tU32) eTrigger_StateTimeout);
   }

   return 0;
};

/******************************************************************************/
/* FUNCTION     _bSetTimer_FsmStateChange                                     */
/******************************************************************************/
/**
*  \brief       start or stop the timer
*
*  \param       timeout value
*  \return      tBool - timer start success state
*/
/******************************************************************************/
tBool  dispvidctrl_tclFsm_Avm::_bSetTimer_FsmStateChange( OSAL_tMSecond msTimeout )
{
    tBool bSuccess = TRUE;
    if ( OSAL_C_INVALID_HANDLE != m_hTimerHandle_FsmStateChange )
    {
        if ( OSAL_s32TimerSetTime( m_hTimerHandle_FsmStateChange, msTimeout, 0 ) != OSAL_OK )
        {
            bSuccess = FALSE;
        }
    }
    return(bSuccess);
}

/******************************************************************************/
/* FUNCTION     vStartFsmStateTimer                                           */
/******************************************************************************/
/**
*  \brief       start or stop the timer
*
*  \param       timeout value
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vStartFsmStateTimer(tU32 u32Time)
{
   if(u32Time > 0) {
      OSAL_s32TimerSetTime(m_hTimerHandle_FsmStateChange, u32Time, 0);  // set
   }
   else {
      OSAL_s32TimerSetTime(m_hTimerHandle_FsmStateChange,       0, 0);  // reset
   }
}

/******************************************************************************/
/* FUNCTION     vUpdateTriggerState                                           */
/******************************************************************************/
/**
*  \brief       To be called to correct Trigger states
*
*  \param       Trigger and State
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_Avm::vUpdateTriggerState(tU32 u32Trigger, tBool bTriggerState)
{
    vUpdateTrigger(u32Trigger, bTriggerState);
    vUpdateRelatedTrigger(u32Trigger);
}



/*******************************************************************************
*                                       E O F
*******************************************************************************/
