/******************************************************************************/
/**
* \file    dispvidctrl_tclControl_Rvc.cpp
* \ingroup
*
* \brief
*
* \remark  Copyright : (c) 2015 Robert Bosch GmbH, Hildesheim
* \remark  Author    : Michael Niemann
* \remark  Scope     : AIVI
*
* \todo
*/
/******************************************************************************/

/*******************************************************************************
                        Includes
*******************************************************************************/
#include "dispvidctrl_tclControl.h"
#include "dispvidctrl_tclControl_Rvc.h"
#include "Idispvidctrl_ClientVcan.h"
#include "dispvidctrl_tclFsm_ReverseSignal.h"
#include "dispvidctrl_tclFsm_CameraSupply.h"
#include "dispvidctrl_tclFsm_CameraSupplySupervision.h"
#include "dispvidctrl_tclFsm_Hmi.h"
#include "I_dispvidctrl_ClientDiaglogIf.h"
#include "I_dispvidctrl_ServiceRvcIf.h"
#include "I_dispvidctrl_ClientSpmIf.h"
#include "dispvidctrl_tclControl_Video.h"
#include "dispvidctrl_datapool.h"

#define VD_DIAGLOG_S_IMPORT_INTERFACE_MSG
#include "vd_diaglog_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_APPLICATION
#include "trcGenProj/Header/dispvidctrl_tclControl_Rvc.cpp.trc.h"
#endif

/*******************************************************************************
                        Defines
*******************************************************************************/
#define DISPVIDCTRL_CAMERA_OVERCURRENT_LEVEL       195 // [mA]
#define DISPVIDCTRL_CAMERA_OPEN_LOAD_LEVEL          10 // [mA]
#define DISPVIDCTRL_CAMERA_SHORT_TO_BATTERY_LEVEL 8000 // [mV]
#define DISPVIDCTRL_CAMERA_SHORT_TO_GND_VOLTAGE_LEVEL 4000 //[mV]
#define DISPVIDCTRL_VEHICLESPEED_TRESHOLD_SLOW      10 // [km/h]
#define DISPVIDCTRL_VEHICLESPEED_TRESHOLD_HIGH      15 // [km/h]

// the below values kept the same as Camera, needs to be adapted for mic
#define DISPVIDCTRL_MICROPHONE_OVERCURRENT_LEVEL       195 // [mA]
#define DISPVIDCTRL_MICROPHONE_OPEN_LOAD_LEVEL           2 // [mA]
#define DISPVIDCTRL_MICROPHONE_SHORT_TO_BATTERY_LEVEL 8000 // [mV]
#define DISPVIDCTRL_MICROPHONE_SHORT_TO_GND_VOLTAGE_LEVEL 4000 //[mV]

// Camera
#define DISPVIDCTRL_ELMOS_A2_VOLTAGE_ERROR  (tU8) 0x10
#define DISPVIDCTRL_ELMOS_A2_OPEN_LOAD      (tU8) 0x20
#define DISPVIDCTRL_ELMOS_A2_OVER_CURRENT   (tU8) 0x30

// Microphone
#define DISPVIDCTRL_ELMOS_A1_VOLTAGE_ERROR  (tU8) 0x04
#define DISPVIDCTRL_ELMOS_A1_OPEN_LOAD      (tU8) 0x08
#define DISPVIDCTRL_ELMOS_A1_OVER_CURRENT   (tU8) 0x0C

// Elmos Over Temperature
#define DISPVIDCTRL_ELMOS_OVER_TEMPERATURE  (tU8) 0x02

// Elmos Error present
#define DISPVIDCTRL_ELMOS_ANY_ERROR_FLAG_MASK (tU8) 0x3F

// Reverse Gear Debouncing Time Values [ms]
#ifndef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
// NISSAN:
// Debouncing is needed
#define DISPVIDCTRL_RVC_FSM_REVERSESIGNAL_DEBOUNCE_LEVEL_1_TIME_IN_MS_NISSAN  50
#define DISPVIDCTRL_RVC_FSM_REVERSESIGNAL_DEBOUNCE_LEVEL_2_TIME_IN_MS_NISSAN  300
#else
// RENAULT:
// No Debouncing is needed for Manual Transmission (Manual Gear Box)
#define DISPVIDCTRL_RVC_FSM_REVERSESIGNAL_DEBOUNCE_LEVEL_1_TIME_IN_MS_RENAULT_MT  5
#define DISPVIDCTRL_RVC_FSM_REVERSESIGNAL_DEBOUNCE_LEVEL_2_TIME_IN_MS_RENAULT_MT  5
// Debouncing is needed for Automatic Transmission (Automatic Gear Box)
#define DISPVIDCTRL_RVC_FSM_REVERSESIGNAL_DEBOUNCE_LEVEL_1_TIME_IN_MS_RENAULT_AT  5
#define DISPVIDCTRL_RVC_FSM_REVERSESIGNAL_DEBOUNCE_LEVEL_2_TIME_IN_MS_RENAULT_AT  500
#endif

/******************************************************************************/
/* FUNCTION     dispvidctrl_tclControl_Rvc()                                  */
/******************************************************************************/
/**
*  \brief       default constructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclControl_Rvc::dispvidctrl_tclControl_Rvc()
: dispvidctrl_tclBaseIf(OSAL_NULL)
, m_hTimerHandle_ElmosInit(OSAL_C_INVALID_HANDLE)
, m_poFsm_ReverseSignal(OSAL_NULL)
, m_poFsm_CameraSupply(OSAL_NULL)
, m_poFsm_CameraSupplySupervision(OSAL_NULL)
, m_poClientVcan(OSAL_NULL)
, m_poClientDiaglogIf(OSAL_NULL)
, m_poServiceRvcIf(OSAL_NULL)
, m_enReverseSignal_State(DISPVIDCTRL_REVERSE_SIGNAL_STATE__OFF)
, m_enFsmControl_Rvc_SuperState(DISPVIDCTRL_CONTROL_RVC__DISABLED)
, m_GpioId_Reverse(0)
, m_hGpioHandle_Reverse(OSAL_ERROR)
, m_bReverseSignal_InputSignalState(FALSE)
, m_bReverseStateUpdatePending(TRUE)
, m_bReverseSimulationModeViaTTFis(FALSE)
, m_GpioId_CameraSupplyError(0)
, m_hGpioHandle_CameraSupplyError(OSAL_ERROR)
, m_bCameraSupplyError_InputSignalState(FALSE)
, m_enCameraVccCurrentState_ShortGnd(EN_CurrentStateUNKNOWN)
, m_enCameraVccCurrentState_OpenLoad(EN_CurrentStateUNKNOWN)
, m_enCameraVccVccState(EN_VccStateUNKNOWN)
, m_enMicVccCurrentState_ShortGnd(EN_CurrentStateUNKNOWN)
, m_enMicVccCurrentState_OpenLoad(EN_CurrentStateUNKNOWN)
, m_enMicVccVccState(EN_VccStateUNKNOWN)
, m_bCameraState(TRUE)
, m_bDiagCameraReq(FALSE)
, m_bDiagCameraFreeze(FALSE)
, m_bMicState(FALSE)
, m_bDiagMicReq(FALSE)
, m_bDiagMicFreeze(FALSE)
, m_bCameraSupplyFreeze(FALSE)
, m_bRvcMode(FALSE)
, m_bRvcType_Renault(FALSE)

#ifdef VARIANT_S_FTR_ENABLE_SMART
, m_Config_u8VCANDiversity(DISPVIDCTRL_AllianceVCANSlot_Type_C1A)
#else
, m_Config_u8VCANDiversity(DISPVIDCTRL_AllianceVCANSlot_Type_0)
#endif
, m_bVideoSignalState(FALSE)
, m_tU8CntVerificationShortToBattery(0)
, m_bAccState(FALSE)
, m_bIgnition(FALSE)
, m_bCameraSupplyHardwareInitDone(FALSE)
, m_u8CameraSupplyHardwareInitRetryCounter(0)
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
// RENAULT
, m_bOverSpeed(FALSE)
, m_bRecoverCameraAllow(FALSE)
#endif
{
}

/******************************************************************************/
/* FUNCTION     dispvidctrl_tclControl_Rvc()                                  */
/******************************************************************************/
/**
*  \brief       constructor
*
*  \param       pointer to main application
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclControl_Rvc::dispvidctrl_tclControl_Rvc(dispvidctrl_tclAppMain* poMainAppl)
: dispvidctrl_tclBaseIf(poMainAppl)
, m_hTimerHandle_ElmosInit(OSAL_C_INVALID_HANDLE)
, m_poFsm_ReverseSignal(OSAL_NULL)
, m_poFsm_CameraSupply(OSAL_NULL)
, m_poFsm_CameraSupplySupervision(OSAL_NULL)
, m_poClientVcan(OSAL_NULL)
, m_poClientDiaglogIf(OSAL_NULL)
, m_poServiceRvcIf(OSAL_NULL)
, m_enReverseSignal_State(DISPVIDCTRL_REVERSE_SIGNAL_STATE__OFF)
, m_enFsmControl_Rvc_SuperState(DISPVIDCTRL_CONTROL_RVC__DISABLED)
, m_GpioId_Reverse(0)
, m_hGpioHandle_Reverse(OSAL_ERROR)
, m_bReverseSignal_InputSignalState(FALSE)
, m_bReverseStateUpdatePending(TRUE)
, m_bReverseSimulationModeViaTTFis(FALSE)
, m_GpioId_CameraSupplyError(0)
, m_hGpioHandle_CameraSupplyError(OSAL_ERROR)
, m_bCameraSupplyError_InputSignalState(FALSE)
, m_enCameraVccCurrentState_ShortGnd(EN_CurrentStateUNKNOWN)
, m_enCameraVccCurrentState_OpenLoad(EN_CurrentStateUNKNOWN)
, m_enCameraVccVccState(EN_VccStateUNKNOWN)
, m_enMicVccCurrentState_ShortGnd(EN_CurrentStateUNKNOWN)
, m_enMicVccCurrentState_OpenLoad(EN_CurrentStateUNKNOWN)
, m_enMicVccVccState(EN_VccStateUNKNOWN)
, m_bCameraState(TRUE)
, m_bDiagCameraReq(FALSE)
, m_bDiagCameraFreeze(FALSE)
, m_bMicState(FALSE)
, m_bDiagMicReq(FALSE)
, m_bDiagMicFreeze(FALSE)
, m_bCameraSupplyFreeze(FALSE)
, m_bRvcMode(FALSE)
, m_bRvcType_Renault(FALSE)
#ifdef VARIANT_S_FTR_ENABLE_SMART
, m_Config_u8VCANDiversity(DISPVIDCTRL_AllianceVCANSlot_Type_C1A)
#else
, m_Config_u8VCANDiversity(DISPVIDCTRL_AllianceVCANSlot_Type_0)
#endif
, m_bVideoSignalState(FALSE)
, m_tU8CntVerificationShortToBattery(0)
, m_bAccState(FALSE)
, m_bIgnition(FALSE)
, m_bCameraSupplyHardwareInitDone(FALSE)
, m_u8CameraSupplyHardwareInitRetryCounter(0)
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
// RENAULT
, m_bOverSpeed(FALSE)
, m_bRecoverCameraAllow(FALSE)
#endif
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Rvc() entered."));

   // create new FSM ReverseSignal
   m_poFsm_ReverseSignal = dispvidctrl_tclFsm_ReverseSignal::tclCreateInstance(this);
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poFsm_ReverseSignal);

   // create new FSM CameraSupply
   m_poFsm_CameraSupply = dispvidctrl_tclFsm_CameraSupply::tclCreateInstance(this);
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poFsm_CameraSupply);

   // create new FSM CameraSupplySupervision
   m_poFsm_CameraSupplySupervision = dispvidctrl_tclFsm_CameraSupplySupervision::tclCreateInstance(this);
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poFsm_CameraSupplySupervision);

   // create timer for Elmos initalisation
   if( OSAL_s32TimerCreate( (OSAL_tpfCallback)_pfCallbackTimer_ElmosInit, (tPVoid)this, &m_hTimerHandle_ElmosInit) != OSAL_OK)
   {
      ETG_TRACE_FATAL(("ERROR - Could not create ElmosInit timer!"));
      m_hTimerHandle_ElmosInit = OSAL_C_INVALID_HANDLE;
   }
}


/******************************************************************************/
/* FUNCTION     ~dispvidctrl_tclControl_Rvc                                   */
/******************************************************************************/
/**
*  \brief       destructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclControl_Rvc::~dispvidctrl_tclControl_Rvc(tVoid)
{
   ETG_TRACE_USR4(("~dispvidctrl_tclControl_Rvc() entered."));

   // de-initialise/close initialised parts
   _vCloseGpio_Reverse();
   _vCloseGpio_CameraSupplyError();

   if ( m_poFsm_ReverseSignal != OSAL_NULL )
   {
       delete m_poFsm_ReverseSignal;
       m_poFsm_ReverseSignal = OSAL_NULL;
   }
   if ( m_poFsm_CameraSupply != OSAL_NULL )
   {
       delete m_poFsm_CameraSupply;
       m_poFsm_CameraSupply = OSAL_NULL;
   }
   if ( m_poFsm_CameraSupplySupervision != OSAL_NULL )
   {
       delete m_poFsm_CameraSupplySupervision;
       m_poFsm_CameraSupplySupervision = OSAL_NULL;
   }

   m_poClientDiaglogIf = OSAL_NULL;
   m_poClientVcan = OSAL_NULL;
   m_poServiceRvcIf = OSAL_NULL;
   DispvidctrlParser::vDeleteInstance();
}

/******************************************************************************/
/* FUNCTION     vGetConfiguration                                             */
/******************************************************************************/
/**
*  \brief       Function to get all configuration values needed by this class.
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Rvc::vGetConfiguration(const TConfiguration* pStConfigurationValues)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Rvc::vGetConfiguration() entered."));
   TConfiguration_Fsm_CameraSupply StConfiguration_Fsm_CameraSupply;

   m_bRvcMode = pStConfigurationValues->bRvcMode;
   m_bRvcType_Renault = pStConfigurationValues->bRvcType_Renault;

   if (m_bRvcMode)
   {
      StConfiguration_Fsm_CameraSupply.u8Brand = pStConfigurationValues->u8Brand;
   }
   else
   {
      StConfiguration_Fsm_CameraSupply.u8Brand = 0;
   }
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Rvc::vGetConfiguration() u8Brand = %u", StConfiguration_Fsm_CameraSupply.u8Brand));
   if (m_poFsm_CameraSupply)
   {
      m_poFsm_CameraSupply->vConfigure(&StConfiguration_Fsm_CameraSupply);
   }

   if (m_poFsm_ReverseSignal)
   {
#ifndef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
      // NISSAN:
      m_poFsm_ReverseSignal->vSetDebouncingLevel1TimeInMs(DISPVIDCTRL_RVC_FSM_REVERSESIGNAL_DEBOUNCE_LEVEL_1_TIME_IN_MS_NISSAN);
      m_poFsm_ReverseSignal->vSetDebouncingLevel2TimeInMs(DISPVIDCTRL_RVC_FSM_REVERSESIGNAL_DEBOUNCE_LEVEL_2_TIME_IN_MS_NISSAN);
#else
      // RENAULT:
      if (pStConfigurationValues->bManualTransmission == TRUE)
      {
         m_poFsm_ReverseSignal->vSetDebouncingLevel1TimeInMs(DISPVIDCTRL_RVC_FSM_REVERSESIGNAL_DEBOUNCE_LEVEL_1_TIME_IN_MS_RENAULT_MT);
         m_poFsm_ReverseSignal->vSetDebouncingLevel2TimeInMs(DISPVIDCTRL_RVC_FSM_REVERSESIGNAL_DEBOUNCE_LEVEL_2_TIME_IN_MS_RENAULT_MT);
      }
      else
      {
         // for all other gear box types (AT, CVT or unknown) we will set the AT debouncing values
         m_poFsm_ReverseSignal->vSetDebouncingLevel1TimeInMs(DISPVIDCTRL_RVC_FSM_REVERSESIGNAL_DEBOUNCE_LEVEL_1_TIME_IN_MS_RENAULT_AT);
         m_poFsm_ReverseSignal->vSetDebouncingLevel2TimeInMs(DISPVIDCTRL_RVC_FSM_REVERSESIGNAL_DEBOUNCE_LEVEL_2_TIME_IN_MS_RENAULT_AT);
      }
#endif
   }

   m_Config_u8VCANDiversity = pStConfigurationValues->u8VCANDiversity;
}

/******************************************************************************/
/* FUNCTION     vGetReferences                                                */
/******************************************************************************/
/**
*  \brief       Function to get all reference needed by this class.
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Rvc::vGetReferences(tVoid)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Rvc::vGetReferences() entered."));

   m_poClientDiaglogIf = dynamic_cast<Idispvidctrl_tclClientDiaglogIf*>(_cpoMain->getHandler("Idispvidctrl_tclClientDiaglogIf"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poClientDiaglogIf);

   m_poClientVcan = dynamic_cast<I_dispvidctrl_tclClientVcan*>(_cpoMain->getHandler("I_dispvidctrl_tclClientVcan"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poClientVcan);

   m_poServiceRvcIf = dynamic_cast<Idispvidctrl_tclServiceRvcIf*>(_cpoMain->getHandler("Idispvidctrl_tclServiceRvcIf"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poServiceRvcIf);

    if (NULL == DispvidctrlParser::pGetInstance())
    {
        ETG_TRACE_USR4(("Dispvidcrl Parser Object creation failed"));
    }
}

/******************************************************************************/
/* FUNCTION     vStartCommunication                                           */
/******************************************************************************/
/**
*  \brief       Function to start all dynamic objects e.g. threads, ...
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Rvc::vStartCommunication(tVoid)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Rvc::vStartCommunication() entered."));

   _vConfigureGpio_Reverse();
   _vConfigureGpio_CameraSupplyError();

   // IC2 communication is not working at this time (driver not loaded).
   // We can initialize the Elmos only with delay.
   // start timer for Elmos initialisation
   if ( OSAL_C_INVALID_HANDLE != m_hTimerHandle_ElmosInit )   {
      if ( OSAL_s32TimerSetTime( m_hTimerHandle_ElmosInit, 2000, 0 ) != OSAL_OK )
      {
         ETG_TRACE_FATAL(("ERROR - Could not start ElmosInit timer!"));
      }
   }
}


/******************************************************************************/
/* FUNCTION     _pfCallbackTimer_ElmosInit                                    */
/******************************************************************************/
/**
*  \brief
*
*  \param       pArg - pointer to this class
*  \return      none
*/
/******************************************************************************/
OSAL_tpfCallback dispvidctrl_tclControl_Rvc::_pfCallbackTimer_ElmosInit(tVoid* pArg)
{
   ETG_TRACE_USR4(("   CA-HMI :: _pfCallbackTimer_ElmosInit(): ... ====>>                                  "));

   dispvidctrl_tclControl_Rvc* po_tclControl_Rvc = (dispvidctrl_tclControl_Rvc*) pArg;
   if (po_tclControl_Rvc)
   {
      if ((po_tclControl_Rvc->_cpoMain) && (po_tclControl_Rvc->_cpoMain->theServer()))
      {
         // post loop-back message to Control_Rvc
         tChar strIfName[] = "dispvidctrl_tclControl_Rvc";
         po_tclControl_Rvc->_cpoMain->theServer()->vPostMsgControlRvc(strIfName, EN_CONTROL_RVC__ELMOS_INIT_TIMEOUT);
      }
   }
   return 0;
}


/******************************************************************************/
/* FUNCTION     vHandleMessage(TMsg* pMsg)                                    */
/******************************************************************************/
/**
*  \brief       Handle worker events.
*               Forward to vHandleControl to handle all input events in one method.
*
*  \param       message pointer
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Rvc::vHandleMessage(dispvidctrl_tclBaseIf::TMsg* pMsg)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Rvc::vHandleMessage() entered %u -> data: %d.", ETG_CENUM(dispvidctrl_tclBaseIf::ECmdTypes , (tU32)pMsg->eCmd), pMsg->u.u32Data));

   switch (pMsg->eCmd)
   {
      case dispvidctrl_tclBaseIf::eControlRvc:
         vHandleControl((tenControl_Rvc) pMsg->u.u32Data);
         break;
      case dispvidctrl_tclBaseIf::eControlModeEnabled:
         vHandleControl(EN_CONTROL_RVC__ENABLE, pMsg->u.u32Data);
         break;
      case dispvidctrl_tclBaseIf::eControlModeDisabled:
         vHandleControl(EN_CONTROL_RVC__DISABLE, pMsg->u.u32Data);
         break;
      case dispvidctrl_tclBaseIf::eFsmReverseSignal:
         vHandleControl(EN_CONTROL_RVC__FSM_REVERSESIGNAL_INPUT_EVENT, pMsg->u.u32Data);
         break;
      case dispvidctrl_tclBaseIf::eFsmCameraSupply:
         vHandleControl(EN_CONTROL_RVC__FSM_CAMERASUPPLY_INPUT_EVENT, pMsg->u.u32Data);
         break;
      case dispvidctrl_tclBaseIf::eFsmCameraSupplySupervision:
         vHandleControl(EN_CONTROL_RVC__FSM_CAMERASUPPLYSUPERVISION_INPUT_EVENT, pMsg->u.u32Data);
         break;
      case dispvidctrl_tclBaseIf::eReverseOff:
         vHandleControl(EN_CONTROL_RVC__REVERSE_OFF);
         break;
      case dispvidctrl_tclBaseIf::eReverseOn:
         vHandleControl(EN_CONTROL_RVC__REVERSE_ON);
         break;
      case dispvidctrl_tclBaseIf::eReverseOnDebounced:
         vHandleControl(EN_CONTROL_RVC__REVERSE_ON_DEBOUNCED);
         break;
      case dispvidctrl_tclBaseIf::eAccChanged:
         if ( TRUE == (tBool) pMsg->u.u32Data ) {
            m_bAccState = TRUE;
         } else {
            m_bAccState = FALSE;
         }
         break;
      case dispvidctrl_tclBaseIf::eVideoStatus:
         if (pMsg->u.u32Data & DISPVIDCTRL_VIDEO__MASK_VIDEO_SIGNAL_STATUS)
         {
            vHandleControl(EN_CONTROL_RVC__VIDEO_SIGNAL_TRUE, (tU32) TRUE);
         }
         else
         {
            vHandleControl(EN_CONTROL_RVC__VIDEO_SIGNAL_FALSE, (tU32) TRUE);
         }
         break;
      case dispvidctrl_tclBaseIf::eTriggerRecheckITC:
         vHandleControl(EN_CONTROL_RVC__SEND_NEXT_TESTRESULT);
         break;
      default:
         break;
   }
}


/******************************************************************************/
/* FUNCTION     vHandleControl()                                              */
/******************************************************************************/
/**
*  \brief       direct call interface
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Rvc::vHandleControl(tenControl_Rvc enControl, tU32 u32Value)
{
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32Value );

   ETG_TRACE_USR4(("   CA-RVC :: ............vHandleControl(): %d     ", ETG_CENUM(tenControl_Rvc, enControl)));
   ETG_TRACE_USR4(("   CA-RVC :: Value of m_bRvcMode is %d", m_bRvcMode));

   switch (enControl)
   {
      case EN_CONTROL_RVC__DISABLE:
         if (m_bRvcMode == TRUE) {
            _vOnEvent_FsmControl_Rvc(enControl, u32Value);
         }
         break;
      case EN_CONTROL_RVC__ENABLE:
         if (m_bRvcMode == TRUE) {
            _vOnEvent_FsmControl_Rvc(enControl, u32Value);
         }
         break;
      //-------------------------------------------------------------------------------------------
      case EN_CONTROL_RVC__FSM_REVERSESIGNAL_INPUT_EVENT:
         if (m_poFsm_ReverseSignal) {m_poFsm_ReverseSignal->vNewTrigger(u32Value, TRUE);}
         break;
      case EN_CONTROL_RVC__FSM_CAMERASUPPLY_INPUT_EVENT:
         if (m_poFsm_CameraSupply) {m_poFsm_CameraSupply->vNewTrigger(u32Value, TRUE);}
         break;
      case EN_CONTROL_RVC__FSM_CAMERASUPPLYSUPERVISION_INPUT_EVENT:
         if (m_poFsm_CameraSupplySupervision) {m_poFsm_CameraSupplySupervision->vNewTrigger(u32Value, TRUE);}
         break;
      //-------------------------------------------------------------------------------------------
      case EN_CONTROL_RVC__STATE_CHANGE_REVERSE: // input via message from GPIO callback
         if (m_poFsm_ReverseSignal) { m_poFsm_ReverseSignal->vNewTrigger((tU32) dispvidctrl_tclFsm_ReverseSignal::eTrigger_SignalChange, TRUE);}
         break;
      case EN_CONTROL_RVC__REVERSE_OFF:          // input via message from reverse signal FSM out
         if ( m_poFsm_CameraSupplySupervision ) {
            m_poFsm_CameraSupplySupervision->vNewTrigger(dispvidctrl_tclFsm_CameraSupplySupervision::eTrigger_SpeedSupervisionActive, TRUE);
         }
         _vOnEvent_FsmControl_Rvc(enControl, u32Value);
         break;
      case EN_CONTROL_RVC__REVERSE_ON:           // input via message from reverse signal FSM out
         // for Nissan, speed supervision not active while in Reverse
         // for Renault, speed supervision active while in Reverse
         if ( m_poFsm_CameraSupplySupervision ) {
             m_poFsm_CameraSupplySupervision->vNewTrigger(dispvidctrl_tclFsm_CameraSupplySupervision::eTrigger_SpeedSupervisionActive, m_bRvcType_Renault);
         }
         _vOnEvent_FsmControl_Rvc(enControl, u32Value);
         break;
      case EN_CONTROL_RVC__REVERSE_ON_DEBOUNCED: // input via message from reverse signal FSM out
         _vOnEvent_FsmControl_Rvc(enControl, u32Value);
         break;
      //--------------------------------------------------------------------------------------------
      case EN_CONTROL_RVC__SERVICE_REQUEST_VISUALISATION_OFF:  // from HMI via RVC service
      case EN_CONTROL_RVC__SERVICE_REQUEST_VISUALISATION_ON:   // from HMI via RVC service
         _vOnEvent_FsmControl_Rvc(enControl, u32Value);
         break;
      //--------------------------------------------------------------------------------------------
      case EN_CONTROL_RVC__CAMERA_OFF:                    // request from HMI state machine
         // switch also supervision to OFF
         if (m_poFsm_CameraSupplySupervision) {
            m_poFsm_CameraSupplySupervision->vNewTrigger((tU32) dispvidctrl_tclFsm_CameraSupplySupervision::eTrigger_SupervisionOff, TRUE);
         }
         _vOnEvent_FsmControl_Rvc(enControl, u32Value);
         break;
      case EN_CONTROL_RVC__VERIFY_AFTER_VIEW_STATUS_STOPPED:  // request from HMI state machine
         // trigger visualization ON to HMI state machine if still valid
         // therefore send reverse gear state trigger
         if (m_poFsm_ReverseSignal) {m_poFsm_ReverseSignal->vNewTrigger((tU32) dispvidctrl_tclFsm_ReverseSignal::eTrigger_AskForReverseDebouncedEvent, TRUE);}
         break;
      case EN_CONTROL_RVC__VIDEO_SIGNAL_FALSE:
         m_bVideoSignalState = FALSE;
         break;
      case EN_CONTROL_RVC__VIDEO_SIGNAL_TRUE:
         m_bVideoSignalState = TRUE;
         break;
      case EN_CONTROL_RVC__USER_SETTING_UPDATE__REAR_VIEW_CAMERA:
      case EN_CONTROL_RVC__USER_SETTING_VALUE__REAR_VIEW_CAMERA:
         // todo:  handle user setting for Rear View Camera deactivation/activation
         break;
      case EN_CONTROL_RVC__IGNITION:
         m_bIgnition = (tBool) u32Value;
         if (m_bIgnition) {
            _vOnEvent_FsmControl_Rvc(EN_CONTROL_RVC__IGNITION_ON, u32Value);
         }
         else {
            _vOnEvent_FsmControl_Rvc(EN_CONTROL_RVC__IGNITION_OFF, u32Value);
            if (!m_bAccState) {
               tChar strIfName[] = "dispvidctrl_tclControl_Rvc";
               // post loop-back message to Switch Off Settings, u32Value will be eTrigger... value
               dispvidctrl_tclAppMain::theServer()->vPostMsgControlRvc(strIfName, EN_CONTROL_RVC__SERVICE_REQUEST_VISUALISATION_OFF);
            }
         }
         break;
      case EN_CONTROL_RVC__SEND_NEXT_TESTRESULT:
         // ITC to report:
         // enITC_ID_RVC_SHORT_BAT,
         // enITC_ID_RVC_SHORT_GND
         // enITC_ID_RVC_OPEN,
         // "FI_EN_NORESULT" in case of RVC not configured (in case Control is disabled)
         // "FI_EN_NORESULT" when Supervision is not active (no active camera), else force to send test result


         // maybe the following "if/else" check is not required because Supervision should not become active when
         // RVC is not configured or Control is disabled for any other reason
         // -> then Supervision StateMachine should be in state OFF normally and test result should be sent as FI_EN_NORESULT
         if (m_enFsmControl_Rvc_SuperState == DISPVIDCTRL_CONTROL_RVC__ENABLED)
         {
            // "FI_EN_NORESULT" when Supervision is not active, else trigger measurement
            // -> trigger input to FSM Supervision, which is handled only in eState_On state -> do function trigger
            // --> when running, reset the current fault state and call measurement (force update of result)
            if (m_poFsm_CameraSupplySupervision)
            {
               m_poFsm_CameraSupplySupervision->vNewTrigger((tU32) dispvidctrl_tclFsm_CameraSupplySupervision::eTrigger_SendNextTestResult, TRUE);
            }
         }
         else
         {
            // "FI_EN_NORESULT" in case of RVC not configured (in case Control is disabled)
            if (m_poClientDiaglogIf)
            {
               m_poClientDiaglogIf->vSetITC(enITC_ID_RVC_SHORT_BAT, midw_fi_tcl_e8_TestResult::FI_EN_NORESULT);
               m_poClientDiaglogIf->vSetITC(enITC_ID_RVC_OPEN, midw_fi_tcl_e8_TestResult::FI_EN_NORESULT);
               m_poClientDiaglogIf->vSetITC(enITC_ID_RVC_SHORT_GND, midw_fi_tcl_e8_TestResult::FI_EN_NORESULT);
            }
         }
         break;
      case EN_CONTROL_RVC__CAMERA_MICROPHONE_SUPPLY_ERROR:
         // as long as we handle the error only for Camera (not for Microphone) we can send a trigger to the
         // Camera Supply Supervision State Machine
         if (m_poFsm_CameraSupplySupervision)
         {
            m_poFsm_CameraSupplySupervision->vNewTrigger((tU32) dispvidctrl_tclFsm_CameraSupplySupervision::eTrigger_CameraSupplyError, TRUE);
         }
         break;

      // Production Diag Session via IO Control Service requests
      case EN_CONTROL_RVC__DIAG_IOCONTROL_CAMERA_VCC_FREEZE_STATE_CHANGED:
      {
         // Freeze/Un-freeze are the only possible values
         m_bDiagCameraFreeze = (tBool) u32Value;

         // switch the camera state to OFF when exiting diagnosis freeze
         if ( FALSE == m_bDiagCameraFreeze ) {
            (tVoid) _bSetCameraVccOn( FALSE );
            // stop diag supervision
            if ( m_poFsm_CameraSupplySupervision ) {
               m_poFsm_CameraSupplySupervision->vNewTrigger((tU32) dispvidctrl_tclFsm_CameraSupplySupervision::eTrigger_DiagCameraSupervisionOff, TRUE);
            }
         }

         // no need to check the camera's current consumption if Camera is OFF
         if ( m_bCameraState ) {
            _vCheckCameraCurrentConsumption(FALSE);
         }
      }
      break;

      case EN_CONTROL_RVC__DIAG_IOCONTROL_CAMERA_VCC_STATE_CHANGED:
      {
          // On/off are the only possible values
          m_bDiagCameraReq = (tBool) u32Value;
          tU32 u32Trigger = 0;

          // if camera is requested as ON from diagnosis, switch ON the diag supervision
          if ( TRUE == m_bDiagCameraReq ) {
              u32Trigger = dispvidctrl_tclFsm_CameraSupplySupervision::eTrigger_DiagCameraSupervisionOn;
          } else {
              u32Trigger = dispvidctrl_tclFsm_CameraSupplySupervision::eTrigger_DiagCameraSupervisionOff;
          }

          if ( m_poFsm_CameraSupplySupervision ) {
              m_poFsm_CameraSupplySupervision->vNewTrigger(u32Trigger, TRUE);
          }

          (tVoid) _bSetCameraVccOn( m_bDiagCameraReq );
          if ( m_bCameraState ) {
              _vCheckCameraCurrentConsumption(FALSE);
          }
      }
      break;

      case EN_CONTROL_RVC__DIAG_IOCONTROL_MICROPHONE_VCC_FREEZE_STATE_CHANGED:
      {
          // Freeze/Un-freeze are the only possible values
          m_bDiagMicFreeze = (tBool) u32Value;

          // switch ON the Mic State when exiting the Diagnosis Freeze (Mic is always ON by default)
          if ( FALSE == m_bDiagMicFreeze ) {
              (tVoid) bSetMicrophoneVccOn( TRUE );
          }

          // no need to check the microphone's current consumption if Microphone is OFF
          if ( TRUE == m_bMicState ) {
              _vCheckMicCurrentConsumption();
          }
      }
      break;

      case EN_CONTROL_RVC__DIAG_IOCONTROL_MICROPHONE_VCC_STATE_CHANGED:
      {
          // On/off are the only possible values
          m_bDiagMicReq = (tBool) u32Value;
          tU32 u32Trigger = 0;

          // if microphone is requested as ON from diagnosis, switch ON the diag supervision
          if ( TRUE == m_bDiagMicReq ) {
              u32Trigger = dispvidctrl_tclFsm_CameraSupplySupervision::eTrigger_DiagMicSupervisionOn;
          } else {
              u32Trigger = dispvidctrl_tclFsm_CameraSupplySupervision::eTrigger_DiagMicSupervisionOff;
          }

          if ( OSAL_NULL != m_poFsm_CameraSupplySupervision ) {
              m_poFsm_CameraSupplySupervision->vNewTrigger(u32Trigger, TRUE);
          }

          (tVoid) bSetMicrophoneVccOn( m_bDiagMicReq );
          if ( TRUE == m_bMicState ) {
              _vCheckMicCurrentConsumption(FALSE);
          }
      }
      break;

      case EN_CONTROL_RVC__DIAG_VIDEO_SCREEN_CAMERA_SUPPLY:
      {
         // used for Diagnosis Video screen and for GuidelineAdjustement video screen
         // a camera supply deactivation shall not happen while these screens are active
         m_bCameraSupplyFreeze = (tBool) u32Value;
         //switch camera VCC on or off
         (tVoid) _bSetCameraVccOn((tBool) u32Value);
		 ETG_TRACE_USR4(("Value of u32value is %d",u32Value));
	  }
      break;

      case EN_CONTROL_RVC__ELMOS_INIT_TIMEOUT:
      {
         _vConfigureElmos_CameraVccOn();

         // On startup, the camera supply is active by default.
         // In non RvcMode switch camera supply to OFF.
         if (m_bRvcMode == FALSE) {
            (tVoid) _bSetCameraVccOn(FALSE);
         }
         else {
            // If RvcControl is enabled and if FSM_CameraSupply is in state OFF, we can deactivate the camera supply voltage.
            if (m_poFsm_CameraSupply) {
               if (m_enFsmControl_Rvc_SuperState == DISPVIDCTRL_CONTROL_RVC__ENABLED) {
                  if (m_poFsm_CameraSupply->u32GetFsmState() == dispvidctrl_tclFsm_CameraSupply::eState_CameraOff) {
                     (tVoid) _bSetCameraVccOn(FALSE);
                  }
               }
            }
         }

         // for production diagnosis we need to update the current
         // camera power supply state
         // microphone power supply state

         // in case when camera is never switched ON the camera power supply state needs to be updated
         // this will be read in production diagnosis
         // if the camera is already ON, the status update would have happened anyway
         if ( FALSE == m_bCameraState )
         {
            // Updated diagnosis with the De-activated State
            rvc_fi_tcl_e8_Camera_PowerSupplyStatus oCameraPwrStatus;
            oCameraPwrStatus.enType = rvc_fi_tcl_e8_Camera_PowerSupplyStatus::FI_EN_DEACTIVATED;

            ETG_TRACE_USR4(("   CA-RVC :: Updating Elmos Camera Power Supply Status as %u Current 0 mA", \
                  ETG_ENUM(DISPVIDCTRL_E8_CAMERA_POWERSUPPLYSTATUS, oCameraPwrStatus.enType)));

            if ( m_poServiceRvcIf )
               m_poServiceRvcIf->sendCamera_PwrSupplyStatusStatus(0, oCameraPwrStatus, 0);
         }
      }
      break;
      
      case EN_CONTROL_RVC__TAILGATE_OPEN:
      case EN_CONTROL_RVC__TAILGATE_CLOSE:
         _vOnEvent_FsmControl_Rvc(enControl, u32Value);
         break;

      default:
         break;
   }
}


/******************************************************************************/
/* FUNCTION     vHandleFsmOutEvent_ReverseSignal()                            */
/******************************************************************************/
/**
*  \brief       handle function to post output events from FSM ReverseSignal
*
*  \param       event
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Rvc::vHandleFsmOutEvent_ReverseSignal(tenFsmOutEvent_ReverseSignal enFsmOutEvent_ReverseSignal, tU32 u32Value)
{
   // ...FSM_OUT__AVM...     // input events for FSM AVM (Loopback to FSM)
   // ...FSM_OUT__HMI...     // input events for FSM HMI

   ETG_TRACE_USR4(("   CA-RVC :: vHandleFsmOutEvent_ReverseSignal(): %d     ", ETG_CENUM(tenFsmOutEvent_ReverseSignal, enFsmOutEvent_ReverseSignal)));

   switch (enFsmOutEvent_ReverseSignal)
   {
      case FSM_REVERSESIGNAL_OUT__FSM_REVERSESIGNAL_INPUT_EVENT:
         {
            tChar strIfName[] = "dispvidctrl_tclControl_Rvc";
            // post loop-back message to Reverse Signal FSM, u32Value will be eTrigger... value
            dispvidctrl_tclAppMain::theServer()->vPostMsgFsmReverseSignal(strIfName, u32Value);
         }
         break;
      case FSM_REVERSESIGNAL_OUT__REVERSE_OFF:
         {
			if(m_bRvcMode == TRUE)
			{
            tChar strIfName[] = "";
            m_enReverseSignal_State = DISPVIDCTRL_REVERSE_SIGNAL_STATE__OFF;
            dispvidctrl_tclAppMain::theServer()->vPostMsgReverseOff(strIfName);
			}
         }
         break;
      case FSM_REVERSESIGNAL_OUT__REVERSE_ON:
         {
			if(m_bRvcMode == TRUE)
			{
				tChar strIfName[] = "";
				m_enReverseSignal_State = DISPVIDCTRL_REVERSE_SIGNAL_STATE__ON;
				dispvidctrl_tclAppMain::theServer()->vPostMsgReverseOn(strIfName);
			}
         }
         break;
      case FSM_REVERSESIGNAL_OUT__REVERSE_ON_DEBOUNCED:
         {
			if(m_bRvcMode == TRUE)
			{
            tChar strIfName[] = "";
            m_enReverseSignal_State = DISPVIDCTRL_REVERSE_SIGNAL_STATE__ON_DEBOUNCED;
            dispvidctrl_tclAppMain::theServer()->vPostMsgReverseOnDebounced(strIfName);
			}
         }
         break;
      default:
         break;
   }
}


/******************************************************************************/
/* FUNCTION     vHandleFsmOutEvent_CameraSupply()                             */
/******************************************************************************/
/**
*  \brief       handle function to post output events from FSM CameraSupply
*
*  \param       event
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Rvc::vHandleFsmOutEvent_CameraSupply(tenFsmOutEvent_CameraSupply enFsmOutEvent_CameraSupply, tU32 u32Value)
{
   // ...FSM_OUT__AVM...     // input events for FSM AVM (Loopback to FSM)
   // ...FSM_OUT__HMI...     // input events for FSM HMI

   ETG_TRACE_USR4(("   CA-RVC :: vHandleFsmOutEvent_CameraSupply(): %d     ", ETG_CENUM(tenFsmOutEvent_CameraSupply, enFsmOutEvent_CameraSupply)));

   switch (enFsmOutEvent_CameraSupply)
   {
      case FSM_CAMERASUPPLY_OUT__FSM_CAMERASUPPLY_INPUT_EVENT:
         {
            tChar strIfName[] = "dispvidctrl_tclControl_Rvc";
            // post loop-back message to Camera Supply FSM, u32Value will be eTrigger... value
            dispvidctrl_tclAppMain::theServer()->vPostMsgFsmCameraSupply(strIfName, u32Value);
         }
         break;
      case FSM_CAMERASUPPLY_OUT__SWITCH_CAMERA_VCC_OFF:
	  #ifdef VARIANT_S_FTR_ENABLE_SMART
         (tVoid) _bSetCameraVccOn(FALSE);
		 #endif
         break;
      case FSM_CAMERASUPPLY_OUT__SWITCH_CAMERA_VCC_ON:
	  #ifdef VARIANT_S_FTR_ENABLE_SMART
         (tVoid) _bSetCameraVccOn(TRUE);
		 #endif
         break;
      case FSM_CAMERASUPPLY_OUT__HMI_EV_VISUALISATION_OFF:
         {
            tChar strIfName[] = "dispvidctrl_tclControl_Hmi";
            dispvidctrl_tclAppMain::theServer()->vPostMsgHmiVisualisationOff(strIfName, u32Value);
            if (m_bRvcMode)
            {
               // deactivate supervision for current consumption
			   #ifdef VARIANT_S_FTR_ENABLE_SMART
               if (m_poFsm_CameraSupplySupervision) {m_poFsm_CameraSupplySupervision->vNewTrigger((tU32) dispvidctrl_tclFsm_CameraSupplySupervision::eTrigger_SupervisionOff, TRUE);}
			   #endif
            }
         }
         break;
      case FSM_CAMERASUPPLY_OUT__HMI_EV_VISUALISATION_ON:
         {
            tChar strIfName[] = "dispvidctrl_tclControl_Hmi";
            dispvidctrl_tclAppMain::theServer()->vPostMsgHmiVisualisationOn(strIfName);
            if (m_bRvcMode)
            {
               // activate supervision for current consumption
			   #ifdef VARIANT_S_FTR_ENABLE_SMART
               if (m_poFsm_CameraSupplySupervision) {m_poFsm_CameraSupplySupervision->vNewTrigger((tU32) dispvidctrl_tclFsm_CameraSupplySupervision::eTrigger_SupervisionOn, TRUE);}
			   #endif
            }
         }
         break;
      case FSM_CAMERASUPPLY_OUT__HMI_EV_VISUALISATION_ON_CAMERA_SETTINGS:
         {
            tChar strIfName[] = "dispvidctrl_tclControl_Hmi";
            dispvidctrl_tclAppMain::theServer()->vPostMsgHmiVisualisationOnCameraSettings(strIfName);
            if (m_bRvcMode)
            {
               // activate supervision for current consumption
			   #ifdef VARIANT_S_FTR_ENABLE_SMART
               if (m_poFsm_CameraSupplySupervision) {m_poFsm_CameraSupplySupervision->vNewTrigger((tU32) dispvidctrl_tclFsm_CameraSupplySupervision::eTrigger_SupervisionOn, TRUE);}
			   #endif
            }
         }
         break;
      default:
         break;
   }
}

/******************************************************************************/
/* FUNCTION     vHandleFsmOutEvent_CameraSupplySupervision()                  */
/******************************************************************************/
/**
*  \brief       handle function to post output events from FSM CameraSupplySupervision
*
*  \param       event
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Rvc::vHandleFsmOutEvent_CameraSupplySupervision(tenFsmOutEvent_CameraSupplySupervision enFsmOutEvent_CameraSupplySupervision, tU32 u32Value)
{
   // ...FSM_OUT__AVM...     // input events for FSM AVM (Loopback to FSM)
   // ...FSM_OUT__HMI...     // input events for FSM HMI

   ETG_TRACE_USR4(("   CA-RVC :: vHandleFsmOutEvent_CameraSupplySupervision(): %d     ", ETG_CENUM(tenFsmOutEvent_CameraSupplySupervision, enFsmOutEvent_CameraSupplySupervision)));

   switch (enFsmOutEvent_CameraSupplySupervision)
   {
      case FSM_CAMERASUPPLYSUPERVISION_OUT__FSM_CAMERASUPPLYSUPERVISION_INPUT_EVENT:
         {
            tChar strIfName[] = "dispvidctrl_tclControl_Rvc";
            // post loop-back message to Camera Supply Supervision FSM, u32Value will be eTrigger... value
            dispvidctrl_tclAppMain::theServer()->vPostMsgFsmCameraSupplySupervision(strIfName, u32Value);
         }
         break;
      case FSM_CAMERASUPPLYSUPERVISION_OUT__CHECK_CURRENT_SPEED:
         {
            tenSpeedStatus enSpeedStatus = EN_SpeedUNKNOWN;
            if (bGetSpeedStatus(enSpeedStatus))
            {
               if (EN_SpeedHIGH == enSpeedStatus)
               {
                  ETG_TRACE_USR4(("   CA-RVC :: vHandleFsmOutEvent_CameraSupplySupervision(): high speed limit exceeded!   "));
                  // trigger service request off due to exceeded speed limit
                  // and stop supervision for this visualisation cycle
#ifndef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
      // NISSAN:
                  tChar strIfName[] = "dispvidctrl_tclControl_Rvc";
                  // post loop-back message to Camera Supply Supervision FSM, u32Value will be eTrigger... value
                  dispvidctrl_tclAppMain::theServer()->vPostMsgFsmCameraSupply(strIfName, (tU32) dispvidctrl_tclFsm_CameraSupply::eTrigger_ServiceRequestOff);
                  dispvidctrl_tclAppMain::theServer()->vPostMsgFsmCameraSupplySupervision(strIfName, (tU32) dispvidctrl_tclFsm_CameraSupplySupervision::eTrigger_SupervisionOff);
                }
                else
                {
                  ETG_TRACE_USR4(("   CA-RVC :: vHandleFsmOutEvent_CameraSupplySupervision(): enSpeedStatus = %d ", ETG_CENUM(tenSpeedStatus, enSpeedStatus)));
                }
#else
      // RENAULT: 
                 m_bOverSpeed = TRUE;
                 tChar strIfName[] = "dispvidctrl_tclControl_Hmi";
                 dispvidctrl_tclAppMain::theServer()->vPostMsgHmiVisualisationOff(strIfName,
                 (tU32) ((dispvidctrl_tclFsm_Hmi::tenHmiVisualisationOffReason) dispvidctrl_tclFsm_Hmi:: EN_VD_RVC_HMI_VISUALISATION_OFF__OVERSPEED));
                }
                else if ((EN_SpeedLOW == enSpeedStatus) && m_bOverSpeed && m_bRecoverCameraAllow)
                {
                   ETG_TRACE_USR4(("   CA-RVC :: vHandleFsmOutEvent_CameraSupplySupervision(): enSpeedStatus = %d ", ETG_CENUM(tenSpeedStatus, enSpeedStatus)));
                   m_bOverSpeed = FALSE;
                   tChar strIfName[] = "dispvidctrl_tclControl_Hmi";
                   dispvidctrl_tclAppMain::theServer()->vPostMsgHmiVisualisationOn(strIfName);
                }
                else
                {
                  ETG_TRACE_USR4(("   CA-RVC :: vHandleFsmOutEvent_CameraSupplySupervision(): enSpeedStatus = %d ", ETG_CENUM(tenSpeedStatus, enSpeedStatus)));
                }
#endif      
            }
         }
         break;
      case FSM_CAMERASUPPLYSUPERVISION_OUT__CHECK_CURRENT_CONSUMPTION:
         _vCheckCameraCurrentConsumption((tBool) u32Value);
         break;
      case FSM_CAMERASUPPLYSUPERVISION_OUT__SEND_NEXT_TEST_RESULT:
         // reset states to force a update to diaglog
         _vResetCameraCurrentState();
         _vResetMicrophoneCurrentState(); // Harmless, but to be checked: do we need this also ?
         _vCheckCameraCurrentConsumption(FALSE);
         break;
      case FSM_CAMERASUPPLYSUPERVISION_OUT__SEND_NO_RESULT:
         if (m_poClientDiaglogIf)
            {
               m_poClientDiaglogIf->vSetITC(enITC_ID_RVC_SHORT_BAT, midw_fi_tcl_e8_TestResult::FI_EN_NORESULT);
               m_poClientDiaglogIf->vSetITC(enITC_ID_RVC_OPEN, midw_fi_tcl_e8_TestResult::FI_EN_NORESULT);
               m_poClientDiaglogIf->vSetITC(enITC_ID_RVC_SHORT_GND, midw_fi_tcl_e8_TestResult::FI_EN_NORESULT);
               // reset states to force a update to diaglog on next activation within this power cycle
               _vResetCameraCurrentState();
               _vResetMicrophoneCurrentState(); // Harmless, but to be checked: do we need this also ?
            }
         break;

      case FSM_CAMERASUPPLYSUPERVISION_OUT__DIAG_CAMERA_VCC_MONITOR_TIMEOUT:
      {
          // new monitoring cycle, reset the saved states
          // set the power supply as per the requested Diag State again
          // check the current consumption again and update the power supply state

          Idispvidctrl_tclClientSpmIf* poClientSpmIf = dynamic_cast<Idispvidctrl_tclClientSpmIf*>(_cpoMain->getHandler("Idispvidctrl_tclClientSpmIf"));

          _vResetCameraCurrentState();
          if (poClientSpmIf) {
              if ( !(poClientSpmIf->bIsCriticalVoltageActive()) ) {
                  (tVoid) _bSetCameraVccOn( m_bDiagCameraReq );
                  _vCheckCameraCurrentConsumption(FALSE);
              } else {
                  ETG_TRACE_USR4(("   CA-RVC :: ............vHandleControl(): Critical Voltage, Monitoring cycle until recovery"));
                  (tVoid) _bSetCameraVccOn( FALSE );
              }
          }
      }
      break;

      case FSM_CAMERASUPPLYSUPERVISION_OUT__DIAG_MICROPHONE_VCC_MONITOR_TIMEOUT:
      {
          // new monitoring cycle, reset the saved states
          // set the power supply as per the requested Diag State again
          // check the current consumption again and update the power supply state

          Idispvidctrl_tclClientSpmIf* poClientSpmIf = dynamic_cast<Idispvidctrl_tclClientSpmIf*>(_cpoMain->getHandler("Idispvidctrl_tclClientSpmIf"));

          _vResetMicrophoneCurrentState();
          if (poClientSpmIf) {
              if ( !(poClientSpmIf->bIsCriticalVoltageActive()) ) {
                  (tVoid) bSetMicrophoneVccOn( m_bDiagMicReq );
                  _vCheckMicCurrentConsumption(FALSE);
              } else {
                  ETG_TRACE_USR4(("   CA-RVC :: ............vHandleControl(): Critical Voltage, Monitoring cycle until recovery"));
                  (tVoid) bSetMicrophoneVccOn( FALSE );
              }
          }
      }
      break;

      default:
         break;
   }
}

/******************************************************************************/
/* FUNCTION     _vCheckCameraCurrentConsumption                               */
/******************************************************************************/
/**
*  \brief       do the measurement for camera supply supervision and report ITC
*               to diaglog
*
*  \param       tBool Elmos GPIO callback indication
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Rvc::_vCheckCameraCurrentConsumption(tBool bElmosErrorIndication)
{
   rvc_fi_tcl_e8_Camera_PowerSupplyStatus oCameraPwrStatus;

   tBool bLevelOutOfRange = FALSE;
   tU8 u8Error = 0;
   tU8 u8SupVoltage = 0;
   tU8 u8VoltageA2 = 0;
   tU8 u8CurrentLowA2 = 0;
   tU8 u8CurrentHighA2 = 0;
   tU32 u32SupVoltageValue = 0;
   tU32 u32CurrentConsumptionValue = 0;
   tU32 u32VoltageOutValue = 0;
   tBool bRegisterReadSuccess = FALSE;

//   tBool bElmosErr_A2_VoltageError = FALSE;
//   tBool bElmosErr_A2_OpenLoad     = FALSE;
   tBool bElmosErr_A2_OverCurrent  = FALSE;
//   tBool bElmosErr_OverTemperature = FALSE;

   // read from Elmos IC the error state, voltage and current consumption value (ADC value)
   // Camera is output A2, microphone is A1

   if (TRUE == _bReadRegister((tU8) 0x00, u8Error))
   {
      ETG_TRACE_USR4(("   CA-RVC ::              error register: 0x%x     ", u8Error));

      if (TRUE == _bReadRegister((tU8) 0x02, u8SupVoltage))
      {
         ETG_TRACE_USR4(("   CA-RVC ::       u8SupVoltage register: 0x%x     ", u8SupVoltage));
         u32SupVoltageValue = (tU32) ((tU32) u8SupVoltage * 100); // [mV]
      }

      // camera A2 read
      if (TRUE == _bReadRegister((tU8) 0x09, u8VoltageA2))
      {
         u32VoltageOutValue = (tU32) (((tU32) u8VoltageA2 * 7059) / 100);
         ETG_TRACE_USR4(("   CA-RVC ::       u8VoltageA2 register: 0x%x     ", u8VoltageA2));

         if (TRUE == _bReadRegister((tU8) 0x0A, u8CurrentLowA2))
         {
            ETG_TRACE_USR4(("   CA-RVC ::    u8CurrentLowA2 register: 0x%x     ", u8CurrentLowA2));

            if (TRUE == _bReadRegister((tU8) 0x0B, u8CurrentHighA2))
            {
               ETG_TRACE_USR4(("   CA-RVC ::   u8CurrentHighA2 register: 0x%x     ", u8CurrentHighA2));
               bRegisterReadSuccess = TRUE;

               // current consumption
               if ((u8CurrentLowA2 == 0xFF) && (u8CurrentHighA2 > 0x3F))
               {
                  u32CurrentConsumptionValue = (tU32) (((tU32) u8CurrentHighA2 * 784) / 1000);
               }
               else
               {
                  u32CurrentConsumptionValue = (tU32) (((tU32) u8CurrentLowA2 * 196) / 1000);
               }
            }
         }
      }
   }
   ETG_TRACE_USR4(("   CA-RVC :: Elmos supply voltage:       %d mV   ", u32SupVoltageValue));
   ETG_TRACE_USR4(("   CA-RVC :: camera current consumption: %d mA   ", u32CurrentConsumptionValue));
   ETG_TRACE_USR4(("   CA-RVC :: camera supply voltage:      %d mV   ", u32VoltageOutValue));
   ETG_TRACE_USR4(("   CA-RVC :: m_tU8CntVerificationShortToBattery:      %d   ", m_tU8CntVerificationShortToBattery));

   // check error register value
   // Elmos IC will report over-current error

   // Error Register
   //  MSB --------------------------------------------------------- LSB
   // |      |      | A2-ER1 | A2-ER0 | A1-ER1 | A1-ER0 |    OT | SUP-OV|
   //  -----------------------------------------------------------------
   // ER1, ER0    01 voltage error
   //             10 open load
   //             11 over-current
   // OT  over-temperature

   // check if error is for A2 and which error
//   if ((u8Error & DISPVIDCTRL_ELMOS_A2_VOLTAGE_ERROR) == DISPVIDCTRL_ELMOS_A2_VOLTAGE_ERROR)
//   {
//      bElmosErr_A2_VoltageError = TRUE;
//      ETG_TRACE_USR4(("   CA-RVC :: Elmos CAMERA_SUPPLY_VOLTAGE_ERROR"));
//   }
//   if ((u8Error & DISPVIDCTRL_ELMOS_A2_OPEN_LOAD) == DISPVIDCTRL_ELMOS_A2_OPEN_LOAD)
//   {
//      bElmosErr_A2_OpenLoad = TRUE;
//      ETG_TRACE_USR4(("   CA-RVC :: Elmos DISPVIDCTRL_CAMERA_SUPPLY_OPEN_LOAD"));
//   }
   if ((u8Error & DISPVIDCTRL_ELMOS_A2_OVER_CURRENT) == DISPVIDCTRL_ELMOS_A2_OVER_CURRENT)
   {
      bElmosErr_A2_OverCurrent = TRUE;
      ETG_TRACE_USR4(("   CA-RVC :: Elmos DISPVIDCTRL_CAMERA_SUPPLY_OVER_CURRENT"));
   }
//   if ((u8Error & DISPVIDCTRL_ELMOS_OVER_TEMPERATURE) == DISPVIDCTRL_ELMOS_OVER_TEMPERATURE)
//   {
//      bElmosErr_OverTemperature = TRUE;
//      ETG_TRACE_USR4(("   CA-RVC :: Elmos OVER_TEMPERATURE"));
//   }

   if (bRegisterReadSuccess == TRUE)
   {
      if (  (bElmosErr_A2_OverCurrent)
          ||(u32VoltageOutValue < DISPVIDCTRL_CAMERA_SHORT_TO_GND_VOLTAGE_LEVEL))
      {
         m_enCameraVccCurrentState_ShortGnd = EN_CurrentStateOVER;
         bLevelOutOfRange = TRUE;

         oCameraPwrStatus.enType = rvc_fi_tcl_e8_Camera_PowerSupplyStatus::FI_EN_SHORTTOGROUND;
   
         // report ITC as Failed
         if (m_poClientDiaglogIf)
         {
            ETG_TRACE_USR4(("   CA-RVC :: Short to GND detected "));
            m_poClientDiaglogIf->vSetITC(enITC_ID_RVC_SHORT_GND, midw_fi_tcl_e8_TestResult::FI_EN_FAILED);
         }
      }
      else
      {
         // We don't have a over-current condition, so voltage register value should be valid.
         // We will check first for short to battery (verifying the output voltage)
         // because for an short to battery the current consumption will be low also.
         if (u32VoltageOutValue > (tU32) DISPVIDCTRL_CAMERA_SHORT_TO_BATTERY_LEVEL)
         {
            m_enCameraVccVccState = EN_VccStateOVER;
            bLevelOutOfRange = TRUE;

            oCameraPwrStatus.enType = rvc_fi_tcl_e8_Camera_PowerSupplyStatus::FI_EN_SHORTTOBATTERY;
   
            // report ITC as Failed
            if (m_poClientDiaglogIf)
            {
               ETG_TRACE_USR4(("   CA-RVC :: Short to Battery detected "));
               m_poClientDiaglogIf->vSetITC(enITC_ID_RVC_SHORT_BAT, midw_fi_tcl_e8_TestResult::FI_EN_FAILED);
            }
         }
         else if (u32CurrentConsumptionValue < DISPVIDCTRL_CAMERA_OPEN_LOAD_LEVEL)
         {
            m_enCameraVccCurrentState_OpenLoad = EN_CurrentStateUNDER;
            bLevelOutOfRange = TRUE;
   
            oCameraPwrStatus.enType = rvc_fi_tcl_e8_Camera_PowerSupplyStatus::FI_EN_OPENLOAD;
            // report ITC as Failed
            if (m_poClientDiaglogIf)
            {
               ETG_TRACE_USR4(("   CA-RVC :: Open Load detected "));
               m_poClientDiaglogIf->vSetITC(enITC_ID_RVC_OPEN, midw_fi_tcl_e8_TestResult::FI_EN_FAILED);
            }
         }
         else
         {
         }
      }
   
   
      if (bLevelOutOfRange == FALSE)
      {
         // no error detected
         // set ITC's to passed which where not reported or which were reported as FAILED within the operation cycle
   
         // in case the state is not OKAY change to OKAY and report ITC as PASSED
         if (m_enCameraVccCurrentState_ShortGnd != EN_CurrentStateOKAY)
         {
            m_enCameraVccCurrentState_ShortGnd = EN_CurrentStateOKAY;
            if (m_poClientDiaglogIf)
            {
               ETG_TRACE_USR4(("   CA-RVC :: update Short to GND to passed "));
               m_poClientDiaglogIf->vSetITC(enITC_ID_RVC_SHORT_GND, midw_fi_tcl_e8_TestResult::FI_EN_PASSED);
            }
         }
   
         if (m_enCameraVccCurrentState_OpenLoad != EN_CurrentStateOKAY)
         {
            m_enCameraVccCurrentState_OpenLoad = EN_CurrentStateOKAY;
            if (m_poClientDiaglogIf)
            {
               ETG_TRACE_USR4(("   CA-RVC :: update Short to Battery to passed "));
               m_poClientDiaglogIf->vSetITC(enITC_ID_RVC_OPEN, midw_fi_tcl_e8_TestResult::FI_EN_PASSED);
            }
         }

         if ( m_enCameraVccVccState != EN_VccStateOKAY )
         {
            m_enCameraVccVccState = EN_VccStateOKAY;
            if (m_poClientDiaglogIf)
            {
               ETG_TRACE_USR4(("   CA-RVC :: update Open Load to passed "));
               m_poClientDiaglogIf->vSetITC(enITC_ID_RVC_SHORT_BAT, midw_fi_tcl_e8_TestResult::FI_EN_PASSED);
            }
         }

         oCameraPwrStatus.enType = rvc_fi_tcl_e8_Camera_PowerSupplyStatus::FI_EN_OK;
      }
   
      if (bLevelOutOfRange == TRUE)
      {
         // switch supply voltage to OFF in case of: over current, short to battery, open load
         (tVoid) _bSetCameraVccOn(FALSE);
         // we have to reset the error register with supervision stop or with switching output voltage to off
         _bResetElmosErrorRegister();
   
         // stop supervision for this visualisation cycle
         {
            // post loop-back message to Camera Supply Supervision FSM, u32Value will be eTrigger... value
            tChar strIfName[] = "dispvidctrl_tclControl_Rvc";
            dispvidctrl_tclAppMain::theServer()->vPostMsgFsmCameraSupplySupervision(strIfName, (tU32) dispvidctrl_tclFsm_CameraSupplySupervision::eTrigger_SupervisionOff);
         }
   
         // Not in all cases when switching the supply voltage to off we will get the video signal failure indication to leave the camera screen.
         // In case of the Short to Battery the video signal could be still fine, so we trigger a visualisation OFF always.
         {
            tChar strIfName[] = "dispvidctrl_tclControl_Hmi";
            dispvidctrl_tclAppMain::theServer()->vPostMsgHmiVisualisationOff(strIfName,
                         (tU32) ((dispvidctrl_tclFsm_Hmi::tenHmiVisualisationOffReason) dispvidctrl_tclFsm_Hmi::EN_VD_RVC_HMI_VISUALISATION_OFF__RVC_ELECTRIC_FAILURE));
         }
      }
      else if (   (bElmosErrorIndication == TRUE)
               || (u8Error & DISPVIDCTRL_ELMOS_ANY_ERROR_FLAG_MASK))
      {
         // was error register set because of microphone supply error?
         // reset error register
         _bResetElmosErrorRegister();
      }
      else
      {
      }
   }
   else
   {
      oCameraPwrStatus.enType = rvc_fi_tcl_e8_Camera_PowerSupplyStatus::FI_EN_UNKNOWN;
      ETG_TRACE_FATAL(("   CA-RVC :: _vCheckCameraCurrentConsumption:: reading Elmos Registers failed "));
   }

   ETG_TRACE_USR4(("   CA-RVC :: Updating Elmos Camera Power Supply Status as %u Current %u", \
           ETG_ENUM(DISPVIDCTRL_E8_CAMERA_POWERSUPPLYSTATUS, oCameraPwrStatus.enType), u32CurrentConsumptionValue));

   DISPVIDCTRL_NULL_POINTER_CHECK(m_poServiceRvcIf);
   m_poServiceRvcIf->sendCamera_PwrSupplyStatusStatus(0, oCameraPwrStatus, u32CurrentConsumptionValue);
}

/******************************************************************************/
/* FUNCTION     _vCheckMicCurrentConsumption                                  */
/******************************************************************************/
/**
*  \brief       do the measurement for microphone supply supervision and report
*               supply state to diagnosis
*
*  \param       tBool Elmos GPIO callback indication
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Rvc::_vCheckMicCurrentConsumption(tBool /* bElmosErrorIndication */)
{
    rvc_fi_tcl_e8_Microphone_PowerSupplyStatus oMicPwrStatus;

    tBool bLevelOutOfRange = FALSE;
    tU8 u8Error = 0;
    tU8 u8SupVoltage = 0;
    tU32 u32SupVoltageValue = 0;
    tU8 u8VoltageA1 = 0;
    tU8 u8CurrentLowA1 = 0;
    tU8 u8CurrentHighA1 = 0;
    tU32 u32CurrentMicConsumptionValue = 0;
    tU32 u32VoltageOutValueA1 = 0;
    tBool bRegisterReadSuccess = FALSE;

//    tBool bElmosErr_A1_VoltageError = FALSE;
//    tBool bElmosErr_A1_OpenLoad     = FALSE;
    tBool bElmosErr_A1_OverCurrent  = FALSE;
//    tBool bElmosErr_OverTemperature = FALSE;

    // read from Elmos IC the error state, voltage and current consumption value (ADC value)
    // Camera is output A2, microphone is A1
    if (TRUE == _bReadRegister((tU8) 0x00, u8Error))
    {
       ETG_TRACE_USR4(("   CA-RVC ::              error register: 0x%x     ", u8Error));

       if (TRUE == _bReadRegister((tU8) 0x02, u8SupVoltage))
       {
          ETG_TRACE_USR4(("   CA-RVC ::       u8SupVoltage register: 0x%x     ", u8SupVoltage));
          u32SupVoltageValue = (tU32) ((tU32) u8SupVoltage * 100); // [mV]
       }

       // Microphone A1 read
       if (TRUE == _bReadRegister((tU8) 0x04, u8VoltageA1))
       {
          u32VoltageOutValueA1 = (tU32) (((tU32) u8VoltageA1 * 7059) / 100);
          ETG_TRACE_USR4(("   CA-RVC ::       u8VoltageA1 register: 0x%x     ", u8VoltageA1));

          if (TRUE == _bReadRegister((tU8) 0x05, u8CurrentLowA1))
          {
             ETG_TRACE_USR4(("   CA-RVC ::    u8CurrentLowA1 register: 0x%x     ", u8CurrentLowA1));

             if (TRUE == _bReadRegister((tU8) 0x06, u8CurrentHighA1))
             {
                ETG_TRACE_USR4(("   CA-RVC ::   u8CurrentHighA1 register: 0x%x     ", u8CurrentHighA1));
                bRegisterReadSuccess = TRUE;

                // current consumption
                if ((u8CurrentLowA1 == 0xFF) && (u8CurrentHighA1 > 0x3F))
                {
                    u32CurrentMicConsumptionValue = (tU32) (((tU32) u8CurrentHighA1 * 784) / 1000);
                }
                else
                {
                    u32CurrentMicConsumptionValue = (tU32) (((tU32) u8CurrentLowA1 * 196) / 1000);
                }
             }
          }
       }

       ETG_TRACE_USR4(("   CA-RVC :: Elmos supply voltage:       %d mV   ", u32SupVoltageValue));
       ETG_TRACE_USR4(("   CA-RVC :: mic current consumption:    %d mA   ", u32CurrentMicConsumptionValue));
       ETG_TRACE_USR4(("   CA-RVC :: mic supply voltage:         %d mV   ", u32VoltageOutValueA1));
       ETG_TRACE_USR4(("   CA-RVC :: m_tU8CntVerificationShortToBattery:      %d   ", m_tU8CntVerificationShortToBattery));
    }

    // check error register value
    // Elmos IC will report over-current error

    // Error Register
    //  MSB --------------------------------------------------------- LSB
    // |      |      | A2-ER1 | A2-ER0 | A1-ER1 | A1-ER0 |    OT | SUP-OV|
    //  -----------------------------------------------------------------
    // ER1, ER0    01 voltage error
    //             10 open load
    //             11 over-current
    // OT  over-temperature
    if ((u8Error & DISPVIDCTRL_ELMOS_A1_OVER_CURRENT) == DISPVIDCTRL_ELMOS_A1_OVER_CURRENT)
    {
       bElmosErr_A1_OverCurrent = TRUE;
       ETG_TRACE_USR4(("   CA-RVC :: Elmos DISPVIDCTRL_MICROPHONE_SUPPLY_OVER_CURRENT"));
    }

    if (bRegisterReadSuccess == TRUE)
    {
       if (  (bElmosErr_A1_OverCurrent)
           ||(u32VoltageOutValueA1 < DISPVIDCTRL_MICROPHONE_SHORT_TO_GND_VOLTAGE_LEVEL))
       {
          m_enMicVccCurrentState_ShortGnd = EN_CurrentStateOVER;
          bLevelOutOfRange = TRUE;

          oMicPwrStatus.enType = rvc_fi_tcl_e8_Microphone_PowerSupplyStatus::FI_EN_SHORTTOGROUND;
       }
       else
       {
          // We don't have a over-current condition, so voltage register value should be valid.
          // We will check first for short to battery (verifying the output voltage)
          // because for an short to battery the current consumption will be low also.
          if (u32VoltageOutValueA1 > (tU32) DISPVIDCTRL_MICROPHONE_SHORT_TO_BATTERY_LEVEL)
          {
              m_enMicVccVccState = EN_VccStateOVER;
             bLevelOutOfRange = TRUE;

             oMicPwrStatus.enType = rvc_fi_tcl_e8_Microphone_PowerSupplyStatus::FI_EN_SHORTTOBATTERY;
          }
          else if (u32CurrentMicConsumptionValue < DISPVIDCTRL_MICROPHONE_OPEN_LOAD_LEVEL)
          {
              m_enMicVccCurrentState_OpenLoad = EN_CurrentStateUNDER;
             bLevelOutOfRange = TRUE;

             oMicPwrStatus.enType = rvc_fi_tcl_e8_Microphone_PowerSupplyStatus::FI_EN_OPENLOAD;
          }
          else
          {
          }
       }

       if (bLevelOutOfRange == FALSE)
       {
          // no error detected
          // change all the Mic states to OKAY
          m_enMicVccCurrentState_ShortGnd = EN_CurrentStateOKAY;
          m_enMicVccCurrentState_OpenLoad = EN_CurrentStateOKAY;
          m_enCameraVccVccState = EN_VccStateOKAY;

          oMicPwrStatus.enType = rvc_fi_tcl_e8_Microphone_PowerSupplyStatus::FI_EN_OK;
       }
    }
    else
    {
       oMicPwrStatus.enType = rvc_fi_tcl_e8_Microphone_PowerSupplyStatus::FI_EN_UNKNOWN;
       ETG_TRACE_FATAL(("   CA-RVC :: _vCheckMicCurrentConsumption::reading Elmos Registers failed "));
    }

    ETG_TRACE_USR4(("   CA-RVC :: Updating Elmos Microphone Power Supply Status as %u and Current %u", \
            ETG_ENUM(DISPVIDCTRL_E8_MICROPHONE_POWERSUPPLYSTATUS, oMicPwrStatus.enType), u32CurrentMicConsumptionValue));

    DISPVIDCTRL_NULL_POINTER_CHECK(m_poServiceRvcIf);
    m_poServiceRvcIf->sendMicrophone_PwrSupplyStatusStatus(0, oMicPwrStatus, u32CurrentMicConsumptionValue);
}

/******************************************************************************/
/* FUNCTION     _vCheckMicCurrentConsumption                                  */
/******************************************************************************/
/**
*  \brief       do the measurement for microphone supply supervision and report
*               supply state to diagnosis
*
*  \param       CCA Message to be filled
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Rvc::_vCheckMicCurrentConsumption(rvc_ccafi_tclMsgMicrophone_PwrSupplyStatusStatus& oStatus)
{
    oStatus.current = 0;
    oStatus.pwrSupplyID = 0;

    tBool bLevelOutOfRange = FALSE;
    tU8 u8Error = 0;
    tU8 u8SupVoltage = 0;
    tU32 u32SupVoltageValue = 0;
    tU8 u8VoltageA1 = 0;
    tU8 u8CurrentLowA1 = 0;
    tU8 u8CurrentHighA1 = 0;
    tU32 u32CurrentMicConsumptionValue = 0;
    tU32 u32VoltageOutValueA1 = 0;
    tBool bRegisterReadSuccess = FALSE;

//    tBool bElmosErr_A1_VoltageError = FALSE;
//    tBool bElmosErr_A1_OpenLoad     = FALSE;
    tBool bElmosErr_A1_OverCurrent  = FALSE;
//    tBool bElmosErr_OverTemperature = FALSE;

    // read from Elmos IC the error state, voltage and current consumption value (ADC value)
    // Camera is output A2, microphone is A1
    if (TRUE == _bReadRegister((tU8) 0x00, u8Error))
    {
       ETG_TRACE_USR4(("   CA-RVC ::              error register: 0x%x     ", u8Error));

       if (TRUE == _bReadRegister((tU8) 0x02, u8SupVoltage))
       {
          ETG_TRACE_USR4(("   CA-RVC ::       u8SupVoltage register: 0x%x     ", u8SupVoltage));
          u32SupVoltageValue = (tU32) ((tU32) u8SupVoltage * 100); // [mV]
       }

       // Microphone A1 read
       if (TRUE == _bReadRegister((tU8) 0x04, u8VoltageA1))
       {
          u32VoltageOutValueA1 = (tU32) (((tU32) u8VoltageA1 * 7059) / 100);
          ETG_TRACE_USR4(("   CA-RVC ::       u8VoltageA1 register: 0x%x     ", u8VoltageA1));

          if (TRUE == _bReadRegister((tU8) 0x05, u8CurrentLowA1))
          {
             ETG_TRACE_USR4(("   CA-RVC ::    u8CurrentLowA1 register: 0x%x     ", u8CurrentLowA1));

             if (TRUE == _bReadRegister((tU8) 0x06, u8CurrentHighA1))
             {
                ETG_TRACE_USR4(("   CA-RVC ::   u8CurrentHighA1 register: 0x%x     ", u8CurrentHighA1));
                bRegisterReadSuccess = TRUE;

                // current consumption
                if ((u8CurrentLowA1 == 0xFF) && (u8CurrentHighA1 > 0x3F))
                {
                    u32CurrentMicConsumptionValue = (tU32) (((tU32) u8CurrentHighA1 * 784) / 1000);
                }
                else
                {
                    u32CurrentMicConsumptionValue = (tU32) (((tU32) u8CurrentLowA1 * 196) / 1000);
                }
             }
          }
       }

       ETG_TRACE_USR4(("   CA-RVC :: Elmos supply voltage:       %d mV   ", u32SupVoltageValue));
       ETG_TRACE_USR4(("   CA-RVC :: mic current consumption:    %d mA   ", u32CurrentMicConsumptionValue));
       ETG_TRACE_USR4(("   CA-RVC :: mic supply voltage:         %d mV   ", u32VoltageOutValueA1));
       ETG_TRACE_USR4(("   CA-RVC :: m_tU8CntVerificationShortToBattery:      %d   ", m_tU8CntVerificationShortToBattery));
    }

    // check error register value
    // Elmos IC will report over-current error

    // Error Register
    //  MSB --------------------------------------------------------- LSB
    // |      |      | A2-ER1 | A2-ER0 | A1-ER1 | A1-ER0 |    OT | SUP-OV|
    //  -----------------------------------------------------------------
    // ER1, ER0    01 voltage error
    //             10 open load
    //             11 over-current
    // OT  over-temperature
    if ((u8Error & DISPVIDCTRL_ELMOS_A1_OVER_CURRENT) == DISPVIDCTRL_ELMOS_A1_OVER_CURRENT)
    {
       bElmosErr_A1_OverCurrent = TRUE;
       ETG_TRACE_USR4(("   CA-RVC :: Elmos DISPVIDCTRL_MICROPHONE_SUPPLY_OVER_CURRENT"));
    }

    if (bRegisterReadSuccess == TRUE)
    {
       if (  (bElmosErr_A1_OverCurrent)
           ||(u32VoltageOutValueA1 < DISPVIDCTRL_MICROPHONE_SHORT_TO_GND_VOLTAGE_LEVEL))
       {
          m_enMicVccCurrentState_ShortGnd = EN_CurrentStateOVER;
          bLevelOutOfRange = TRUE;

          oStatus.status.enType = rvc_fi_tcl_e8_Microphone_PowerSupplyStatus::FI_EN_SHORTTOGROUND;
       }
       else
       {
          // We don't have a over-current condition, so voltage register value should be valid.
          // We will check first for short to battery (verifying the output voltage)
          // because for an short to battery the current consumption will be low also.
          if (u32VoltageOutValueA1 > (tU32) DISPVIDCTRL_MICROPHONE_SHORT_TO_BATTERY_LEVEL)
          {
              m_enMicVccVccState = EN_VccStateOVER;
             bLevelOutOfRange = TRUE;

             oStatus.status.enType = rvc_fi_tcl_e8_Microphone_PowerSupplyStatus::FI_EN_SHORTTOBATTERY;
          }
          else if (u32CurrentMicConsumptionValue < DISPVIDCTRL_MICROPHONE_OPEN_LOAD_LEVEL)
          {
              m_enMicVccCurrentState_OpenLoad = EN_CurrentStateUNDER;
             bLevelOutOfRange = TRUE;

             oStatus.status.enType = rvc_fi_tcl_e8_Microphone_PowerSupplyStatus::FI_EN_OPENLOAD;
          }
          else
          {
          }
       }

       if (bLevelOutOfRange == FALSE)
       {
          // no error detected
          // change all the Mic states to OKAY
          m_enMicVccCurrentState_ShortGnd = EN_CurrentStateOKAY;
          m_enMicVccCurrentState_OpenLoad = EN_CurrentStateOKAY;
          m_enCameraVccVccState = EN_VccStateOKAY;

          oStatus.status.enType = rvc_fi_tcl_e8_Microphone_PowerSupplyStatus::FI_EN_OK;
       }
    }
    else
    {
       oStatus.status.enType = rvc_fi_tcl_e8_Microphone_PowerSupplyStatus::FI_EN_UNKNOWN;
       ETG_TRACE_FATAL(("   CA-RVC :: _u8CheckMicCurrentConsumption::reading Elmos Registers failed "));
    }

    ETG_TRACE_USR4(("   CA-RVC :: Updating Elmos Microphone Power Supply Status as %u and Current %u", \
            ETG_ENUM(DISPVIDCTRL_E8_MICROPHONE_POWERSUPPLYSTATUS, oStatus.status.enType), u32CurrentMicConsumptionValue));

    oStatus.current = u32CurrentMicConsumptionValue;
}

/******************************************************************************/
/* FUNCTION     _vOnEvent_FsmControl_Rvc                                      */
/******************************************************************************/
/**
*  \brief
*
*  \param       event, data
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Rvc::_vOnEvent_FsmControl_Rvc(tenControl_Rvc enControl, tU32 u32Value)
{
   ETG_TRACE_USR4(("   CA-RVC :: ........._vOnEvent_FsmControl_Rvc(): %d     ", ETG_CENUM(tenControl_Rvc, enControl)));
   ETG_TRACE_USR4(("   CA-RVC :: . FSM Control_Rvc Super State Entry: %u     ", ETG_CENUM(tenFsmControl_Rvc_SuperState, m_enFsmControl_Rvc_SuperState)));

   tBool bEventProcessed = TRUE;

   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(u32Value);

   switch (m_enFsmControl_Rvc_SuperState)
   {
      /* ------------------------------------------------------------- */
      /*                                      D I S A B L E D          */
      /* ------------------------------------------------------------- */
      // activation via reverse gear on possible
      case DISPVIDCTRL_CONTROL_RVC__DISABLED:
         {
            switch (enControl)
            {
               case EN_CONTROL_RVC__ENABLE:
                  {
                     if (m_poFsm_CameraSupply)
                     {
                        if (m_bIgnition)
                        {
                           m_poFsm_CameraSupply->vUpdateTriggerState((tU32) dispvidctrl_tclFsm_CameraSupply::eTrigger_IgnitionOn, TRUE);
                           m_poFsm_CameraSupply->vUpdateTriggerState((tU32) dispvidctrl_tclFsm_CameraSupply::eTrigger_IgnitionOff, FALSE);
                        }
                        else
                        {
                           m_poFsm_CameraSupply->vUpdateTriggerState((tU32) dispvidctrl_tclFsm_CameraSupply::eTrigger_IgnitionOn, FALSE);
                           m_poFsm_CameraSupply->vUpdateTriggerState((tU32) dispvidctrl_tclFsm_CameraSupply::eTrigger_IgnitionOff, TRUE);
                        }

                        // first LATE transition after EARLY: use direct transition
                        if ((u32Value & DISPVIDCTRL_VIDEO__MASK_EARLY_ACTIV) > 0)
                        {
                           switch ((tenVideoStatus) (u32Value & DISPVIDCTRL_VIDEO__MASK_VIDEO_CAPTURING_STATUS))
                           {
                              case EN_VIDEO_STATUS__BLACK_OUT:
                              case EN_VIDEO_STATUS__OFF:
                                 m_poFsm_CameraSupply->vNewTrigger((tU32) dispvidctrl_tclFsm_CameraSupply::eTrigger_CameraSupplyOff, TRUE);
                                 break;
                              case EN_VIDEO_STATUS__BLACK_IN:
                              case EN_VIDEO_STATUS__ON:
                                 m_poFsm_CameraSupply->vNewTrigger((tU32) dispvidctrl_tclFsm_CameraSupply::eTrigger_DirectOnReverseReq, TRUE);
                                 // we have to activate supply supervision for camera current consumption also, access to Elmos (I2C driver) might be not yet possible at this time
                                 if (m_poFsm_CameraSupplySupervision)
                                 {
                                    m_poFsm_CameraSupplySupervision->vNewTrigger((tU32) dispvidctrl_tclFsm_CameraSupplySupervision::eTrigger_SupervisionOn, TRUE);
                                 }
                                 break;
                              default:
                                 bEventProcessed = FALSE;
                                 break;
                           }

                           // Prepare the graphics screen already.
                           // NCG3D-4737/NCG3d-4738: issue of the first time entry in RVC takes longer than the subsequent entries.
                           dispvidctrl_tclControl_Video* poControlVideo = dynamic_cast<dispvidctrl_tclControl_Video*>(_cpoMain->getHandler("dispvidctrl_tclControl_Video"));
                           DISPVIDCTRL_NULL_POINTER_CHECK(poControlVideo);

                           poControlVideo->vHandleControl(EN_CONTROL_VIDEO__PREPARE_GRAPHICS, (tU32) 0);

                        }
                     }
                     
                     // re-trigger activation request
                     tChar strIfName[] = "dispvidctrl_tclControl_Rvc";
                     dispvidctrl_tclAppMain::theServer()->vPostMsgControlRvc(strIfName, (tU32) EN_CONTROL_RVC__STATE_CHANGE_REVERSE);

                     m_enFsmControl_Rvc_SuperState = DISPVIDCTRL_CONTROL_RVC__ENABLED;
                  }
                  break;
            /*
             * following events are handled to clean up the system in case something was pending and could not be stopped before
             * (there could be some events queued before)
             */
            case EN_CONTROL_RVC__CAMERA_OFF:
               {
                  //switch camera VCC off
                  (tVoid) _bSetCameraVccOn(FALSE);
               }
               break;
            default:
               {
                  bEventProcessed = FALSE;
               }
               break;
            }
         }
         break;
      /* ------------------------------------------------------------- */
      /*                                        E N A B L E D          */
      /* ------------------------------------------------------------- */
      case DISPVIDCTRL_CONTROL_RVC__ENABLED:
      {
         if (EN_CONTROL_RVC__DISABLE == enControl)
         {
            if (m_poFsm_CameraSupply)
            {
               // to stop camera boot timer
               m_poFsm_CameraSupply->vNewTrigger((tU32) dispvidctrl_tclFsm_CameraSupply::eTrigger_Disable, TRUE);
            }

            m_enFsmControl_Rvc_SuperState = DISPVIDCTRL_CONTROL_RVC__DISABLED;
         }
         else
         {
            tU32 u32FSM_Event;

            bEventProcessed = TRUE;
            switch (enControl)
            {
               case EN_CONTROL_RVC__REVERSE_ON:
               {
                  u32FSM_Event = dispvidctrl_tclFsm_CameraSupply::eTrigger_ReverseOn;
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
// RENAULT
                  m_bRecoverCameraAllow =  TRUE;
#endif
               }
               break;
               case EN_CONTROL_RVC__REVERSE_OFF:
               {
                  u32FSM_Event = dispvidctrl_tclFsm_CameraSupply::eTrigger_ReverseOff;
               }
               break;
               case EN_CONTROL_RVC__REVERSE_ON_DEBOUNCED              : u32FSM_Event = dispvidctrl_tclFsm_CameraSupply::eTrigger_ReverseDebounced;  break;
               case EN_CONTROL_RVC__SERVICE_REQUEST_VISUALISATION_ON  : 
                  u32FSM_Event = dispvidctrl_tclFsm_CameraSupply::eTrigger_SettingsOn;
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
// RENAULT
                  m_bRecoverCameraAllow =  FALSE;
#endif
                  break;
               case EN_CONTROL_RVC__SERVICE_REQUEST_VISUALISATION_OFF : u32FSM_Event = dispvidctrl_tclFsm_CameraSupply::eTrigger_SettingsOff;       break;
               case EN_CONTROL_RVC__CAMERA_OFF                        : u32FSM_Event = dispvidctrl_tclFsm_CameraSupply::eTrigger_CameraOff;         break;

               case EN_CONTROL_RVC__IGNITION_ON                       : u32FSM_Event = dispvidctrl_tclFsm_CameraSupply::eTrigger_IgnitionOn;        break;
               case EN_CONTROL_RVC__IGNITION_OFF                      : u32FSM_Event = dispvidctrl_tclFsm_CameraSupply::eTrigger_IgnitionOff;       break;
               case EN_CONTROL_RVC__TAILGATE_OPEN                     : u32FSM_Event = dispvidctrl_tclFsm_CameraSupply::eTrigger_TailgateOpen;      break;
               case EN_CONTROL_RVC__TAILGATE_CLOSE                    : u32FSM_Event = dispvidctrl_tclFsm_CameraSupply::eTrigger_TailgateClose;     break;
               default:
               {
                    bEventProcessed = FALSE;
                    u32FSM_Event = dispvidctrl_tclFsm_CameraSupply::eTrigger_None;
               }
               break;
            }

            if (bEventProcessed == TRUE)
            {
               if (m_poFsm_CameraSupply)
               {
                  m_poFsm_CameraSupply->vNewTrigger(u32FSM_Event, TRUE);
               }
            }
         }
      }
      break;

      default:
         break;
   } // switch super state

   ETG_TRACE_USR4(("   CA-RVC :: ................... Event Processed: %u     ", ETG_ENUM(DISPVIDCTRL_NoYes, bEventProcessed)));
   ETG_TRACE_USR4(("   CA-RVC :: .. FSM Control_Rvc Super State Exit: %u     ", ETG_CENUM(tenFsmControl_Rvc_SuperState, m_enFsmControl_Rvc_SuperState)));
}


/******************************************************************************/
/* FUNCTION     bGetReverseState                                              */
/******************************************************************************/
/**
*  \brief       returns the current state of the reverse gear
*
*  \param       none
*  \return      tBool - TRUE  = reverse gear engaged
*                     - FALSE = reverse gear not engaged
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Rvc::bGetReverseState(tVoid)
{
   if (m_bReverseSimulationModeViaTTFis == FALSE)
   {
#ifndef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
      // Nissan Scope 2.0 and Nissan Scope 2.1 (C1A - Alliance platform) are build using the same customer project filter "RNAIVI"
      // but for Nissan C1A (Alliance) we have to use a CAN signal for reverse instead hardwired signal
      if (m_Config_u8VCANDiversity == DISPVIDCTRL_AllianceVCANSlot_Type_C1A) {
         if (m_poClientVcan)
         {
            m_bReverseSignal_InputSignalState = m_poClientVcan->bGet_ReverseGearEngagedStatus();
         }
         else
         {
            ETG_TRACE_ERR(("dispvidctrl_tclControl_Rvc::bGetReverseState m_poClientVcan is NULL!!"));
         }
      }
      else {
         m_bReverseSignal_InputSignalState = _bGetGPIOState_Reverse();   
      }
#else
      if (m_poClientVcan)
      {
         m_bReverseSignal_InputSignalState = m_poClientVcan->bGet_ReverseGearEngagedStatus();
      }
      else
      {
         ETG_TRACE_ERR(("dispvidctrl_tclControl_Rvc::bGetReverseState m_poClientVcan is NULL!!"));
      }
#endif
   }
   m_bReverseStateUpdatePending = FALSE;

   return (m_bReverseSignal_InputSignalState);
}


/******************************************************************************/
/* FUNCTION     bGetReverseDebouncedState                                     */
/******************************************************************************/
/**
*  \brief       returns true if the reverse gear signal is on and stable
*               (debouncing time).
*
*  \param       none
*  \return      tBool - TRUE  = reverse gear is on and stable (debounced)
*                     - FALSE = else
*/
/******************************************************************************/
// todo : maybe we can get this state from state machine directly
tBool dispvidctrl_tclControl_Rvc::bGetReverseDebouncedState(tVoid) const
{
   tBool bReturnValue = FALSE;

   if (m_enReverseSignal_State == DISPVIDCTRL_REVERSE_SIGNAL_STATE__ON_DEBOUNCED)
   {
      bReturnValue = TRUE;
   }

   return (bReturnValue);
}





//##############################################################################
// GPIO  Reverse Signal
//##############################################################################
/******************************************************************************/
/* FUNCTION     _vConfigureGpio_Reverse                                       */
/******************************************************************************/
/**
*  \brief       configures the GPIO used to detect the reverse gear / RFLS state
*
*               a (call-back)function shall be called on every level (low/high)
*               change
*               - open dev/gpio
*               - set GPIO as input
*               - register call-back function
*               - set trigger edges
*               - enable interrupts
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Rvc::_vConfigureGpio_Reverse(tVoid)
{
   ETG_TRACE_USR4(("_vConfigureGpio_Reverse() entered."));
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
   return;
#endif
//#if not defined (LSIM)
                      tU32  u32OSALError;
           OSAL_trGPIOData  pinData;
   OSAL_trGPIOCallbackData  gpioCallbackData;


      m_GpioId_Reverse = (OSAL_tGPIODevID) OSAL_EN_REVERSE_DETECT;
      m_hGpioHandle_Reverse = OSAL_IOOpen(OSAL_C_STRING_DEVICE_GPIO, OSAL_EN_READWRITE);
      if (m_hGpioHandle_Reverse == OSAL_ERROR)
      {
         u32OSALError = OSAL_u32ErrorCode();
         ETG_TRACE_ERR(("_vConfigureGpio_Reverse() OSAL_IOOpen for dev_GPIO failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
      }
      else
      {
         // Configure pin as input
         if (OSAL_ERROR == OSAL_s32IOControl(m_hGpioHandle_Reverse, OSAL_C_32_IOCTRL_GPIO_SET_INPUT, (tS32) m_GpioId_Reverse))
         {
            u32OSALError = OSAL_u32ErrorCode();
            ETG_TRACE_ERR(("_vConfigureGpio_Reverse() Set GPIO as input failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
         }
         else
         {
            // parameter to pass to the call-back function
            gpioCallbackData.rData.unData.pfCallback = (OSAL_tpfGPIOCallback) _pfCallbackGpioStateChange_Reverse; //my call-back function
            gpioCallbackData.pvArg = (tVoid*) this;
            gpioCallbackData.rData.tId = m_GpioId_Reverse;

            //parameter for setting the trigger edge
            pinData.unData.u16Edge = OSAL_GPIO_EDGE_BOTH;
            pinData.tId = m_GpioId_Reverse;

            // register call-back
            if (OSAL_ERROR == OSAL_s32IOControl(m_hGpioHandle_Reverse, OSAL_C_32_IOCTRL_GPIO_SET_CALLBACK, (intptr_t)(&gpioCallbackData)))
            {
               u32OSALError = OSAL_u32ErrorCode();
               ETG_TRACE_ERR(("_vConfigureGpio_Reverse() callback registration failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
            }
            // Set trigger edge BOTH
            else if (OSAL_ERROR == OSAL_s32IOControl(m_hGpioHandle_Reverse, OSAL_C_32_IOCTRL_GPIO_SET_TRIGGER, (intptr_t)(&pinData)))
            {
               u32OSALError = OSAL_u32ErrorCode();
               ETG_TRACE_ERR(("_vConfigureGpio_Reverse() Set Trigger Edge (Both Edges) failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
               // unregister call-back
               gpioCallbackData.rData.unData.pfCallback = NULL;
               OSAL_s32IOControl(m_hGpioHandle_Reverse, OSAL_C_32_IOCTRL_GPIO_SET_CALLBACK, (intptr_t)&gpioCallbackData);
               //NORMAL_M_ASSERT_ALWAYS();
            }
            else
            {
               //parameter for setting enabling interrupt
               pinData.unData.bState = TRUE;
               // enable interrupts
               if (OSAL_s32IOControl(m_hGpioHandle_Reverse, OSAL_C_32_IOCTRL_GPIO_ENABLE_INT, (intptr_t)(&pinData)) == OSAL_ERROR)
               {
                  u32OSALError = OSAL_u32ErrorCode();
                  ETG_TRACE_ERR(("_vConfigureGpio_Reverse() Enable Interrupt failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));

                  // unregister call-back
                  gpioCallbackData.rData.unData.pfCallback = NULL;
                  OSAL_s32IOControl(m_hGpioHandle_Reverse, OSAL_C_32_IOCTRL_GPIO_SET_CALLBACK, (intptr_t)&gpioCallbackData);
               }
            }
         }
      }


//#endif // #if not defined (LSIM)
}


/******************************************************************************/
/* FUNCTION     _vCloseGpio_Reverse                                           */
/******************************************************************************/
/**
*  \brief       disable interrupt
*               unregister callback
*               close
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Rvc::_vCloseGpio_Reverse(tVoid)
{
   ETG_TRACE_USR4(("_vCloseGpio_Reverse() entered."));
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
   return;
#endif

//#if not defined (LSIM)
   OSAL_trGPIOData  pinData;
   OSAL_trGPIOCallbackData  gpioCallbackData;

   if ((m_GpioId_Reverse != 0) && (m_hGpioHandle_Reverse != OSAL_ERROR))
   {
      // disable interrupt
      //parameter for setting disabling interrupt
      pinData.tId = m_GpioId_Reverse;
      pinData.unData.bState = FALSE;
      (tVoid) OSAL_s32IOControl(m_hGpioHandle_Reverse, OSAL_C_32_IOCTRL_GPIO_ENABLE_INT, (intptr_t)(&pinData));

      // unregister call-back
      // parameter to pass to the call-back function
      gpioCallbackData.rData.unData.pfCallback = NULL;
      gpioCallbackData.pvArg = (tVoid*) this;
      gpioCallbackData.rData.tId = m_GpioId_Reverse;
      (tVoid) OSAL_s32IOControl(m_hGpioHandle_Reverse, OSAL_C_32_IOCTRL_GPIO_SET_CALLBACK, (intptr_t)&gpioCallbackData);

      // close device
      (tVoid) OSAL_s32IOClose(m_hGpioHandle_Reverse);
      m_hGpioHandle_Reverse = OSAL_ERROR;
   }

//#endif // #if not defined (LSIM)
}


/******************************************************************************/
/* FUNCTION     _pfCallbackGpioStateChange_Reverse                            */
/******************************************************************************/
/**
*  \brief       call-back function to get informed about a reverse / RFLS
*               GPIO state change
*               function will set a event for the worker thread
*
*  \param       pointer to myself
*  \return
*/
/******************************************************************************/
OSAL_tpfGPIOCallback dispvidctrl_tclControl_Rvc::_pfCallbackGpioStateChange_Reverse(tVoid* pArg)
{
   // get pointer to myself from argument
   //dispvidctrl_tclControl_Rvc* poThis = (dispvidctrl_tclControl_Rvc*) pArg;
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);

   ETG_TRACE_USR3(("_vCallbackGpioStateChange_Reverse() entered"));

   //post message STATE_CHANGE_REVERSE
   {
      tChar strIfName[] = "dispvidctrl_tclControl_Rvc";
      //poThis->_cpoMain->theServer()->vPostMsgControlRvc(strIfName, (tU32) EN_CONTROL_RVC__STATE_CHANGE_REVERSE);
      dispvidctrl_tclAppMain::theServer()->vPostMsgControlRvc(strIfName, (tU32) EN_CONTROL_RVC__STATE_CHANGE_REVERSE);
   }

   return (0);
}


/******************************************************************************/
/* FUNCTION     _bGetGPIOState_Reverse                                        */
/******************************************************************************/
/**
*  \brief       requests the GPIO state used to detect the reverse / RFLS
*
*
*  \param       none
*  \return      GPIO state
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Rvc::_bGetGPIOState_Reverse(tVoid)
{
   ETG_TRACE_USR4(("_bGetGPIOState_Reverse() entered."));
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
   return FALSE;
#endif
             tBool  bRet = FALSE;
   OSAL_trGPIOData  gpioData;
              tU32  u32OSALError;

   if ((m_GpioId_Reverse != 0) && (m_hGpioHandle_Reverse != OSAL_ERROR))   {
       gpioData.tId = m_GpioId_Reverse;

       if (OSAL_ERROR == OSAL_s32IOControl(m_hGpioHandle_Reverse, OSAL_C_32_IOCTRL_GPIO_IS_STATE_ACTIVE, (intptr_t)(&gpioData)))   {
           u32OSALError = OSAL_u32ErrorCode();
           ETG_TRACE_ERR(("_bGetGPIOState_Reverse() Get GPIO State failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
       }
       else   {
           //state
           bRet = gpioData.unData.bState;
           ETG_TRACE_USR3(("Reverse Gear State = %u", bRet));
       }
   }

   return (bRet);
}


//##############################################################################
// GPIO  Camera Supply Error Signal
//##############################################################################
/******************************************************************************/
/* FUNCTION     _vConfigureGpio_CameraSupplyError                             */
/******************************************************************************/
/**
*  \brief       configures the GPIO used to detect the error state from
*               Micro/Camera supply (Elmos IC)
*               Note: The Elmos IC is used to provide the supply voltage for Microphone also!
*               (it's a common error state pin)
*
*               a (call-back)function shall be called on high to low level change
*               - open dev/gpio
*               - set GPIO as input
*               - register call-back function
*               - set trigger edges
*               - enable interrupts
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Rvc::_vConfigureGpio_CameraSupplyError(tVoid)
{
   //ETG_TRACE_USR4(("_vConfigureGpio_CameraSupplyError() entered."));
   ETG_TRACE_FATAL(("_vConfigureGpio_CameraSupplyError() entered."));
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
   return;
#endif

//#if not defined (LSIM)
                      tU32  u32OSALError;
           OSAL_trGPIOData  pinData;
   OSAL_trGPIOCallbackData  gpioCallbackData;


      m_GpioId_CameraSupplyError = (OSAL_tGPIODevID) OSAL_EN_ELMOS52240_NFLT_3;
      m_hGpioHandle_CameraSupplyError = OSAL_IOOpen(OSAL_C_STRING_DEVICE_GPIO, OSAL_EN_READWRITE);
      if (m_hGpioHandle_CameraSupplyError == OSAL_ERROR)
      {
         u32OSALError = OSAL_u32ErrorCode();
         //ETG_TRACE_ERR(("_vConfigureGpio_CameraSupplyError() OSAL_IOOpen for dev_GPIO failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
         ETG_TRACE_FATAL(("_vConfigureGpio_CameraSupplyError() OSAL_IOOpen for dev_GPIO failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
      }
      else
      {
         // Configure pin as input
         if (OSAL_ERROR == OSAL_s32IOControl(m_hGpioHandle_CameraSupplyError, OSAL_C_32_IOCTRL_GPIO_SET_INPUT, (tS32) m_GpioId_CameraSupplyError))
         {
            u32OSALError = OSAL_u32ErrorCode();
            //ETG_TRACE_ERR(("_vConfigureGpio_CameraSupplyError() Set GPIO as input failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
            ETG_TRACE_FATAL(("_vConfigureGpio_CameraSupplyError() Set GPIO as input failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
         }
         else
         {
            // parameter to pass to the call-back function
            gpioCallbackData.rData.unData.pfCallback = (OSAL_tpfGPIOCallback) _pfCallbackGpioStateChange_CameraSupplyError; //my call-back function
            gpioCallbackData.pvArg = (tVoid*) this;
            gpioCallbackData.rData.tId = m_GpioId_CameraSupplyError;

            //parameter for setting the trigger edge
            pinData.unData.u16Edge = OSAL_GPIO_EDGE_LOW; // low active signal
            pinData.tId = m_GpioId_CameraSupplyError;

            // register call-back
            if (OSAL_ERROR == OSAL_s32IOControl(m_hGpioHandle_CameraSupplyError, OSAL_C_32_IOCTRL_GPIO_SET_CALLBACK, (intptr_t)(&gpioCallbackData)))
            {
               u32OSALError = OSAL_u32ErrorCode();
               //ETG_TRACE_ERR(("_vConfigureGpio_CameraSupplyError() callback registration failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
               ETG_TRACE_FATAL(("_vConfigureGpio_CameraSupplyError() callback registration failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
            }
            // Set trigger edge
            else if (OSAL_ERROR == OSAL_s32IOControl(m_hGpioHandle_CameraSupplyError, OSAL_C_32_IOCTRL_GPIO_SET_TRIGGER, (intptr_t)(&pinData)))
            {
               u32OSALError = OSAL_u32ErrorCode();
               //ETG_TRACE_ERR(("_vConfigureGpio_CameraSupplyError() Set Trigger Edge failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
               ETG_TRACE_FATAL(("_vConfigureGpio_CameraSupplyError() Set Trigger Edge failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
               // unregister call-back
               gpioCallbackData.rData.unData.pfCallback = NULL;
               OSAL_s32IOControl(m_hGpioHandle_CameraSupplyError, OSAL_C_32_IOCTRL_GPIO_SET_CALLBACK, (intptr_t)&gpioCallbackData);
               //NORMAL_M_ASSERT_ALWAYS();
            }
            else
            {
               //parameter for setting enabling interrupt
               pinData.unData.bState = TRUE;
               // enable interrupts
               if (OSAL_s32IOControl(m_hGpioHandle_CameraSupplyError, OSAL_C_32_IOCTRL_GPIO_ENABLE_INT, (intptr_t)(&pinData)) == OSAL_ERROR)
               {
                  u32OSALError = OSAL_u32ErrorCode();
                  //ETG_TRACE_ERR(("_vConfigureGpio_CameraSupplyError() Enable Interrupt failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
                  ETG_TRACE_FATAL(("_vConfigureGpio_CameraSupplyError() Enable Interrupt failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));

                  // unregister call-back
                  gpioCallbackData.rData.unData.pfCallback = NULL;
                  OSAL_s32IOControl(m_hGpioHandle_CameraSupplyError, OSAL_C_32_IOCTRL_GPIO_SET_CALLBACK, (intptr_t)&gpioCallbackData);
               }
            }
         }
      }
//#endif // #if not defined (LSIM)
}

/******************************************************************************/
/* FUNCTION     _vCloseGpio_CameraSupplyError                                 */
/******************************************************************************/
/**
*  \brief       disable interrupt
*               unregister callback
*               close
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Rvc::_vCloseGpio_CameraSupplyError(tVoid)
{
   ETG_TRACE_USR4(("_vCloseGpio_CameraSupplyError() entered."));
#ifdef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
   return;
#endif

//#if not defined (LSIM)
   OSAL_trGPIOData  pinData;
   OSAL_trGPIOCallbackData  gpioCallbackData;

   if ((m_GpioId_CameraSupplyError != 0) && (m_hGpioHandle_CameraSupplyError != OSAL_ERROR))
   {
      // disable interrupt
      //parameter for setting disabling interrupt
      pinData.tId = m_GpioId_CameraSupplyError;
      pinData.unData.bState = FALSE;
      (tVoid) OSAL_s32IOControl(m_hGpioHandle_CameraSupplyError, OSAL_C_32_IOCTRL_GPIO_ENABLE_INT, (intptr_t)(&pinData));

      // unregister call-back
      // parameter to pass to the call-back function
      gpioCallbackData.rData.unData.pfCallback = NULL;
      gpioCallbackData.pvArg = (tVoid*) this;
      gpioCallbackData.rData.tId = m_GpioId_CameraSupplyError;
      (tVoid) OSAL_s32IOControl(m_hGpioHandle_CameraSupplyError, OSAL_C_32_IOCTRL_GPIO_SET_CALLBACK, (intptr_t)&gpioCallbackData);

      // close device
      (tVoid) OSAL_s32IOClose(m_hGpioHandle_CameraSupplyError);
      m_hGpioHandle_CameraSupplyError = OSAL_ERROR;
   }

//#endif // #if not defined (LSIM)
}

/******************************************************************************/
/* FUNCTION     _pfCallbackGpioStateChange_CameraSupplyError                  */
/******************************************************************************/
/**
*  \brief       call-back function to get informed about a GPIO state change
*               function will set a event for the worker thread
*
*  \param       pointer to myself
*  \return
*/
/******************************************************************************/
OSAL_tpfGPIOCallback dispvidctrl_tclControl_Rvc::_pfCallbackGpioStateChange_CameraSupplyError(tVoid* pArg)
{
   // get pointer to myself from argument
   //dispvidctrl_tclControl_Rvc* poThis = (dispvidctrl_tclControl_Rvc*) pArg;
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);

   ETG_TRACE_FATAL(("_pfCallbackGpioStateChange_CameraSupplyError() entered"));

   //post message CAMERA SUPPLY ERROR
   {
      tChar strIfName[] = "dispvidctrl_tclControl_Rvc";
      //poThis->_cpoMain->theServer()->vPostMsgControlRvc(strIfName, (tU32) EN_CONTROL_RVC__CAMERA_MICROPHONE_SUPPLY_ERROR);
      dispvidctrl_tclAppMain::theServer()->vPostMsgControlRvc(strIfName, (tU32) EN_CONTROL_RVC__CAMERA_MICROPHONE_SUPPLY_ERROR);
   }

   return (0);
}





/******************************************************************************/
/* FUNCTION     bGetSpeedStatus                                               */
/******************************************************************************/
/**
*  \brief       requests the vehicle speed from Vcan and evaluates it to limits
*
*
*  \param       none
*  \return      bSuccess
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Rvc::bGetSpeedStatus(tenSpeedStatus& enSpeedStatus)
{
   tBool bSuccess = TRUE;
   tU32 u32VehicleSpeedValue = 0;

   if (m_poClientVcan)
   {
      if (TRUE == m_poClientVcan->bGet_VehicleSpeedSignal(u32VehicleSpeedValue))
      {
         if (u32VehicleSpeedValue > DISPVIDCTRL_VEHICLESPEED_TRESHOLD_HIGH)
         {
            enSpeedStatus = EN_SpeedHIGH;
         }
         else if (u32VehicleSpeedValue < DISPVIDCTRL_VEHICLESPEED_TRESHOLD_SLOW)
         {
            enSpeedStatus = EN_SpeedLOW;
         }
         else
         {
            enSpeedStatus = EN_SpeedMID;
         }
      }
      else
      {
         //enSpeedStatus = EN_SpeedUNKNOWN;
         ETG_TRACE_USR4(("   CA-RVC :: bGetSpeedStatus() failed!                     "));
         bSuccess = FALSE;
      }
   }
   else
   {
      bSuccess = FALSE;
   }
   return (bSuccess);
}

//##############################################################################
// Elmos  CameraVcc
//##############################################################################

/******************************************************************************/
/* FUNCTION     _vConfigureElmos_CameraVccOn                                  */
/******************************************************************************/
/**
*  \brief       configures the Elmos used
*               to switch on/off the power supply for the camera via I2C
*               output A2 is used for Camera Supply
*               - first I2C access of status register
*               - default after Reset:
*                 - SUP overvoltage detection is disabled
*                 --> keep disabled
*               - overcurrent detection is enabled (200mA = maximum possible level)
*                 --> keep enabled at this level
*               - open load detection is disabled
*                 --> keep disabled, we will measure manually
*               - AIN level trigger is disabled
*                 --> keep disabled
*               - auto retry is enabled
*                 --> keep enabled -> error state is not latched in this case,
*                                     we have to react on NFLT Pin
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Rvc::_vConfigureElmos_CameraVccOn(tVoid)
{
   ETG_TRACE_USR4((" CA-RVC   :: _vConfigureElmos_CameraVccOn(): ......... entered!                                "));

   // keep all register content as is
   // - supply overvoltage detection disabled
   // - open load detection disabled
   // - overcurrent detection enabled - it is set to the maximum possible level (200mA)
   // - auto retry enabled
   //   -> we have to register for NFLT Pin to get informed about the error

   // NCG3D-22491: change Microphone overcurrent level to 80mA (output A1)

   if (FALSE == m_bCameraSupplyHardwareInitDone)
   {
      tU8 u8TempRead  = 0x00;
      // access of status register to check I2C access
      if (_bReadRegister(( tU8)0x01, u8TempRead))
      {

         ETG_TRACE_USR4((" _vConfigureElmos_CameraVccOn(): Status register ->  = %x ", u8TempRead));

         // set A1 over current limit to 80mA
         if (_bWriteRegister((tU8)0x07, (tU8) 0x66))
         {
            m_bCameraSupplyHardwareInitDone = TRUE;
         }
      }
      else
      {
         ETG_TRACE_USR4((" _vConfigureElmos_CameraVccOn(): Status register read failed"));
         if (m_u8CameraSupplyHardwareInitRetryCounter < 5)
         {
            m_u8CameraSupplyHardwareInitRetryCounter += 1;
            // restart Elomos Init timer for initalisation retry
            if ( OSAL_C_INVALID_HANDLE != m_hTimerHandle_ElmosInit )   {
               if ( OSAL_s32TimerSetTime( m_hTimerHandle_ElmosInit, 2000, 0 ) != OSAL_OK )
               {
                  ETG_TRACE_FATAL(("ERROR - Could not re-start ElmosInit timer!"));
               }
            }
         }
      }
   }
}


/******************************************************************************/
/* FUNCTION     _bSetCameraVccOn                                              */
/******************************************************************************/
/**
*  \brief       Set Elmos register for camera vcc on.
*
*  \param       tBool bOnOff - Request to switch On (TRUE) or Off (FALSE)
*  \return      TRUE => Elmos set successfully
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Rvc::_bSetCameraVccOn(tBool bOnOff)
{
   tBool bSuccess = FALSE;

    // check if we are really changing something ?
    if ( m_bCameraState != bOnOff )
    {
        // check for Diagnosis freeze and critical voltage conditions
        bOnOff = _bCalcGpio_CameraVccOn(bOnOff);

        tU8 u8TempRead  = 0x03; // A1 and A2 On by default

        // read current status register value
        (tVoid) _bReadRegister(( tU8)0x01, u8TempRead);

        // update status register value (modify only A2A bit)
        if( bOnOff == FALSE) {
            bSuccess = _bWriteRegister((tU8)0x01, (u8TempRead & 0xFD));  //clear A2A bit (Camera VCC On/Off)

            // Updated diagnosis with the De-activated State
            rvc_fi_tcl_e8_Camera_PowerSupplyStatus oCameraPwrStatus;
            oCameraPwrStatus.enType = rvc_fi_tcl_e8_Camera_PowerSupplyStatus::FI_EN_DEACTIVATED;

            ETG_TRACE_USR4(("   CA-RVC :: Updating Elmos Camera Power Supply Status as %u", \
                       ETG_ENUM(DISPVIDCTRL_E8_CAMERA_POWERSUPPLYSTATUS, oCameraPwrStatus.enType)));

            if ( m_poServiceRvcIf )
                m_poServiceRvcIf->sendCamera_PwrSupplyStatusStatus(0, oCameraPwrStatus, 0);
        }
        else {
            bSuccess = _bWriteRegister((tU8)0x01, (u8TempRead | 0x02));  //set A2A bit (Camera VCC On/Off)
        }

        if( bSuccess == TRUE) {
            m_bCameraState = bOnOff;
            ETG_TRACE_FATAL(( "set Camera Power supply to %d", bOnOff ));
        }
        else {
            ETG_TRACE_FATAL(( "set Camera Power supply to %d failed!", bOnOff ));
        }
    }
    else
    {
        ETG_TRACE_USR1(( "Camera Power supply already set to %d ", bOnOff ));
    }

    return( bSuccess);
}

/******************************************************************************/
/* FUNCTION     bSetMicrophoneVccOn                                              */
/******************************************************************************/
/**
*  \brief       Set Elmos register for microphone vcc on.
*
*  \param       tBool bOnOff - Request to switch On (TRUE) or Off (FALSE)
*  \return      TRUE => Elmos set successfully
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Rvc::bSetMicrophoneVccOn(tBool bOnOff)
{
    tBool bSuccess = FALSE;

    // check if we are really changing something ?
    if ( m_bMicState != bOnOff )
    {
        // check for Diagnosis freeze and critical voltage conditions
        bOnOff = _bCalcGpio_MicrophoneVccOn(bOnOff);

        tU8 u8TempRead  = 0x03; // A1 and A2 On by default

        // read current status register value
        (tVoid) _bReadRegister(( tU8)0x01, u8TempRead);

        // update status register value (modify only A1A bit)
        if( bOnOff == FALSE) {
            bSuccess = _bWriteRegister((tU8)0x01, (u8TempRead & 0xFE));  //clear A1A bit (Microphone VCC On/Off)

            // Updated diagnosis with the De-activated State
            rvc_fi_tcl_e8_Microphone_PowerSupplyStatus oMicPwrStatus;
            oMicPwrStatus.enType = rvc_fi_tcl_e8_Microphone_PowerSupplyStatus::FI_EN_DEACTIVATED;

            ETG_TRACE_USR4(("   CA-RVC :: Updating Elmos Microphone Power Supply Status as %u", \
                    ETG_ENUM(DISPVIDCTRL_E8_MICROPHONE_POWERSUPPLYSTATUS, oMicPwrStatus.enType)));

            if ( m_poServiceRvcIf )
                m_poServiceRvcIf->sendMicrophone_PwrSupplyStatusStatus(0, oMicPwrStatus, 0);
        }
        else {
            bSuccess = _bWriteRegister((tU8)0x01, (u8TempRead | 0x01));  //set A1A bit (Microphone VCC On/Off)
        }

        if( bSuccess == TRUE) {
            m_bMicState = bOnOff;
            ETG_TRACE_FATAL(( "set Microphone Power supply to %d", bOnOff ));
        }
        else {
            ETG_TRACE_FATAL(( "set Microphone Power supply to %d failed!", bOnOff ));
        }
    }
    else
    {
        ETG_TRACE_USR1(( "Microphone Power supply already set to %d ", bOnOff ));
    }

    return( bSuccess);
}


/******************************************************************************/
/* FUNCTION     _bResetElmosErrorRegister                                     */
/******************************************************************************/
/**
*  \brief       write into Elmos error register to reset the error states.
*
*  \param       none
*  \return      TRUE => Elmos set successfully
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Rvc::_bResetElmosErrorRegister(tVoid)
{
   tBool bSuccess = FALSE;

   bSuccess = _bWriteRegister((tU8)0x00, (tU8) 0x00);

   ETG_TRACE_USR4(("_bResetElmosErrorRegister %d ", bSuccess));
   return( bSuccess);
}

/******************************************************************************/
/* FUNCTION:     _bReadRegister                                               */
/******************************************************************************/
/**
*  \brief       Function to read Elmos register via I2C.
*
*  \param       u8RegisterAddress, tU8& u8RegisterData
*  \return      TRUE => Elmos read successfully
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Rvc::_bReadRegister(tU8 u8RegisterAddress, tU8& u8RegisterData)
{
   // To read an register value the ELMOS requires an 4 Byte sequence.
   // The first Byte contains the address of the device(7bit) and the R/W bit. For
   // transfer master -> slave ( write register) this bit has to be 0.
   // The next Byte contains the address of the register which shall
   // be accessed. After the second byte one more start condition (repeated start condition)
   // has to be sent followed by a byte again containing the device address and the R/W Bit. After
   // that the device will send the contents of the register and waits for an acknowledge from the master.

   // This sequence can't be realised with an standard "write" followed by an "read" access because
   // in between there would be an stop condition and not an repeated start condition.
   // We need to use the ioctl "I2C_RDWR" which is doing combined read/write transaction without stop in between.

   // https://www.kernel.org/doc/Documentation/i2c/dev-interface

   ETG_TRACE_USR4(("_bReadRegister(register address: 0x%02x) entered.", u8RegisterAddress ));

   tBool bResult = FALSE;
   const char *dev = "/dev/i2c-1";
   const tU8  u8SlaveAddress = 0x52;
   tS32 fd;

   // open I2C bus to read the register
   fd = open(dev, O_RDWR);
   if(fd == -1)
   {
      ETG_TRACE_ERR(("_bReadRegister: I2C bus open failed!"));
   }
   else
   {
      // set device address of the slave
      if( ioctl(fd, I2C_SLAVE, u8SlaveAddress) < 0)
      {
         ETG_TRACE_ERR(("_bReadRegister: setting slave address failed!"));
      }
      else
      {
         // prepare message sequence
         tU8 aReadBuffer[1] = {0};
         tU8 aWriteBuffer[1] = {0};
         struct i2c_msg messages[2];
         memset(messages, OSAL_NULL, sizeof(messages));
         struct i2c_rdwr_ioctl_data data;


         messages[0].addr = u8SlaveAddress;
         messages[0].flags = 0x0; //write mode
         messages[0].len = 1;
         messages[0].buf = aWriteBuffer;
         messages[0].buf[0] = u8RegisterAddress;

         messages[1].addr = u8SlaveAddress;
         messages[1].flags = I2C_M_RD; // read mode
         messages[1].len = 1;
         messages[1].buf = aReadBuffer;

         data.msgs = messages;
         data.nmsgs = 2;

         // read data / register
         if(ioctl(fd, I2C_RDWR, &data) < 0)
         {
           ETG_TRACE_ERR(("_bReadRegister: reading of data failed!"));
         }
         else
         {
            //ETG_TRACE_USR2(("_bReadRegister: aReadBuffer = 0x%02x", aReadBuffer[0]));
            // set return data
            u8RegisterData = aReadBuffer[0];
            bResult = TRUE;
         }
      }
      // close I2C bus
      close(fd);
   }
   ETG_TRACE_USR4(("_bReadRegister(%d, data: 0x%02x) exited.", bResult, u8RegisterData));
   return bResult;
}

/******************************************************************************/
/* FUNCTION:     _bReadRegisters                                              */
/******************************************************************************/
/**
*  \brief       Function to read consecutive Elmos register via I2C.
*
*  \param       u8RegisterAddress -> start address
*               u8NumberOfRegisters - register count to read
*               tU8* au8RegisterData -> pointer to data array
*  \return      TRUE => Elmos read successfully
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Rvc::_bReadRegisters(tU8 u8RegisterStartAddress, tU8 u8NumberOfRegisters, tU8* au8RegisterDataBuffer)
{
   // To read an register value the ELMOS requires an 4 Byte sequence.
   // The first Byte contains the address of the device(7bit) and the R/W bit. For
   // transfer master -> slave ( write register) this bit has to be 0.
   // The next Byte contains the address of the register which shall
   // be accessed. After the second byte one more start condition (repeated start condition)
   // has to be sent followed by a byte again containing the device address and the R/W Bit. After
   // that the device will send the contents of the register and waits for an acknowledge from the master.

   // This sequence can't be realised with an standard "write" followed by an "read" access because
   // in between there would be an stop condition and not an repeated start condition.
   // We need to use the ioctl "I2C_RDWR" which is doing combined read/write transaction without stop in between.

   // https://www.kernel.org/doc/Documentation/i2c/dev-interface

   ETG_TRACE_USR4(("_bReadRegisters(start register address: 0x%02x) entered.", u8RegisterStartAddress ));

   tBool bResult = FALSE;
   const char *dev = "/dev/i2c-1";
   const tU8  u8SlaveAddress = 0x52;
   tS32 fd;

   // open I2C bus to read the register
   fd = open(dev, O_RDWR);
   if(fd == -1)
   {
      ETG_TRACE_ERR(("_bReadRegisters: I2C bus open failed!"));
   }
   else
   {
      // set device address of the slave
      if( ioctl(fd, I2C_SLAVE, u8SlaveAddress) < 0)
      {
         ETG_TRACE_ERR(("_bReadRegisters: setting slave address failed!"));
      }
      else
      {
         // prepare message sequence
         tU8 aReadBuffer[u8NumberOfRegisters];
         tU8 aWriteBuffer[1] = {0};
         struct i2c_msg messages[2];
         memset(messages, OSAL_NULL, sizeof(messages));
         struct i2c_rdwr_ioctl_data data;


         messages[0].addr = u8SlaveAddress;
         messages[0].flags = 0x0; //write mode
         messages[0].len = 1;
         messages[0].buf = aWriteBuffer;
         messages[0].buf[0] = u8RegisterStartAddress;

         messages[1].addr = u8SlaveAddress;
         messages[1].flags = I2C_M_RD; // read mode
         messages[1].len = u8NumberOfRegisters;
         messages[1].buf = aReadBuffer;

         data.msgs = messages;
         data.nmsgs = 2;

         // read data / register
         if(ioctl(fd, I2C_RDWR, &data) < 0)
         {
           ETG_TRACE_ERR(("_bReadRegisters: reading of data failed!"));
         }
         else
         {
            //ETG_TRACE_USR2(("_bReadRegister: aReadBuffer = 0x%02x", aReadBuffer[0]));
            // set return data
            for (tU8 i=0; i<u8NumberOfRegisters; i++)
            {
               au8RegisterDataBuffer[i] = aReadBuffer[i];
            }
            bResult = TRUE;
         }
      }
      // close I2C bus
      close(fd);
   }
   ETG_TRACE_USR4(("_bReadRegisters(%d, data: 0x%02x) exited.", bResult, au8RegisterDataBuffer[0]));
   return bResult;
}

/******************************************************************************/
/* FUNCTION:     _bWriteRegister                                              */
/******************************************************************************/
/**
*  \brief       Function to write Elmos register via I2C.
*
*  \param       const tU8 u8RegisterAddress, tU8 u8RegisterData
*  \return      tBool - write success
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Rvc::_bWriteRegister(const tU8 u8RegisterAddress, tU8 u8RegisterData) const
{
   tBool bResult = FALSE;
   const char *dev = "/dev/i2c-1";
   const tU8  u8SlaveAddress = 0x52;
   tS32 fd;

   // open I2C bus to read the register
   fd = open(dev, O_RDWR);
   if(fd == -1)
   {
      ETG_TRACE_ERR(("_bWriteRegister: I2C bus open failed!"));
   }
   else
   {
      // set device address of the slave
      if( ioctl(fd, I2C_SLAVE, u8SlaveAddress) < 0)
      {
         ETG_TRACE_ERR(("_bWriteRegister: setting slave address failed!"));
      }
      else
      {
         tU8 aWriteBuffer[2] = {0};
         aWriteBuffer[0] = u8RegisterAddress; //Address of Register
         aWriteBuffer[1] = u8RegisterData; //Data
         if (write(fd, aWriteBuffer, 2) != 2)
         {
            ETG_TRACE_ERR(("_bWriteRegister: Error in writing Register"));
         }
         else
         {
            bResult = TRUE;
         }
      }
      close(fd);
   }
   return (bResult);
}

/******************************************************************************/
/* FUNCTION:     _bCalcGpio_CameraVccOn                                       */
/******************************************************************************/
/**
*  \brief       Function to calculate the camera power supply state
*
*  \param       bOnOff - Requested State
*  \return      tBool - Calculated State
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Rvc::_bCalcGpio_CameraVccOn(tBool bOnOff)
{
    tBool bRetVal = FALSE;

    Idispvidctrl_tclClientSpmIf* m_poClientSpmIf = dynamic_cast<Idispvidctrl_tclClientSpmIf*>(_cpoMain->getHandler("Idispvidctrl_tclClientSpmIf"));
    DISPVIDCTRL_NULL_POINTER_CHECK_VAL(m_poClientSpmIf);

    // if diag freeze is inactive
    if ( FALSE == m_bDiagCameraFreeze )
    {
        // while in critical voltage we should not switch the Camera ON
        if ( FALSE == m_poClientSpmIf->bIsCriticalVoltageActive() )
        {
            bRetVal = bOnOff;

            if (m_bCameraSupplyFreeze == TRUE)
            {
               // don't switch OFF when a Diagnosis video screen or GuidelienAdjustment screen is active
               //( for diagnosis video screen diag might not have triggered the camera supply activation via diag service)
               bRetVal = TRUE;
            }
        }
    }
    else
    {
        bRetVal = m_bDiagCameraReq;
    }

    ETG_TRACE_USR4(("_bCalcGpio_CameraVccOn: Calculated State is %d", bRetVal));

    return bRetVal;
}

/******************************************************************************/
/* FUNCTION:     _bCalcGpio_MicrophoneVccOn                                   */
/******************************************************************************/
/**
*  \brief       Function to calculate the camera power supply state
*
*  \param       bOnOff - Requested State
*  \return      tBool - Calculated State
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Rvc::_bCalcGpio_MicrophoneVccOn(tBool bOnOff)
{
    tBool bRetVal = FALSE;

    Idispvidctrl_tclClientSpmIf* m_poClientSpmIf = dynamic_cast<Idispvidctrl_tclClientSpmIf*>(_cpoMain->getHandler("Idispvidctrl_tclClientSpmIf"));
    DISPVIDCTRL_NULL_POINTER_CHECK_VAL(m_poClientSpmIf);

    // if diag freeze is inactive
    if ( FALSE == m_bDiagMicFreeze )
    {
        // while in critical voltage we should not switch the Microphone ON
        if ( FALSE == m_poClientSpmIf->bIsCriticalVoltageActive() )
        {
            bRetVal = bOnOff;
        }
    }
    else
    {
        bRetVal = m_bDiagMicReq;
    }

    ETG_TRACE_USR4(("_bCalcGpio_MicrophoneVccOn: Calculated State is %d", bRetVal));

    return bRetVal;
}

tVoid dispvidctrl_tclControl_Rvc::_vResetCameraCurrentState()
{
    m_enCameraVccCurrentState_ShortGnd = EN_CurrentStateUNKNOWN;
    m_enCameraVccCurrentState_OpenLoad = EN_CurrentStateUNKNOWN;
    m_enCameraVccVccState = EN_VccStateUNKNOWN;
}

tVoid dispvidctrl_tclControl_Rvc::_vResetMicrophoneCurrentState()
{
    m_enMicVccCurrentState_ShortGnd = EN_CurrentStateUNKNOWN;
    m_enMicVccCurrentState_OpenLoad = EN_CurrentStateUNKNOWN;
    m_enMicVccVccState = EN_VccStateUNKNOWN;
}

//##############################################################################
// TTFis
//##############################################################################
/******************************************************************************/
/* FUNCTION     vHandleTraceMessage                                           */
/******************************************************************************/
/**
*  \brief       handle TTFis commands
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Rvc::vHandleTraceMessage(const tUChar* puchData)
{
   DISPVIDCTRL_NULL_POINTER_CHECK(puchData);

   tU32 u32MsgCode = ((puchData[1]<<8) | puchData[2]);
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Rvc::vHandleTraceMessage(): trace command %d ", u32MsgCode ));

   switch (u32MsgCode)
   {
      case DISPVIDCTRL_DISPVIDCTRL_STIMULATE_CONTROL_RVC:
      {
         // Stimulate tclControl_Rvc, Parameter = control name and u32Data value)
         tU8 param1 = puchData[3];
         tU32 param2 = ((tU32)puchData[4] << 24) | ((tU32)puchData[5] << 16) | ((tU32)puchData[6] <<  8) | (tU32)puchData[7];

         vHandleControl((tenControl_Rvc) param1, param2);
      }
      break;
      case DISPVIDCTRL_DISPVIDCTRL_STIMULATE_FSM_REVERSESIGNAL:
      {
         // Stimulate FSM ReverseSignal, Parameter = trigger)
         tU32 param1 = ((tU32)puchData[3] << 24) | ((tU32)puchData[4] << 16) | ((tU32)puchData[5] <<  8) | (tU32)puchData[6];

         vHandleControl(EN_CONTROL_RVC__FSM_REVERSESIGNAL_INPUT_EVENT, param1);
      }
      break;
      case DISPVIDCTRL_DISPVIDCTRL_STIMULATE_FSM_CAMERASUPPLY:
      {
         // Stimulate FSM CameraSupply, Parameter = trigger)
         tU32 param1 = ((tU32)puchData[3] << 24) | ((tU32)puchData[4] << 16) | ((tU32)puchData[5] <<  8) | (tU32)puchData[6];

         vHandleControl(EN_CONTROL_RVC__FSM_CAMERASUPPLY_INPUT_EVENT, param1);
      }
      break;
      case DISPVIDCTRL_DISPVIDCTRL_SWITCH_REVERSEGEAR:
      {
         // simulates switching the reverse gear (GPIO Pin)
         tU8 param1 = puchData[3];

         m_bReverseSimulationModeViaTTFis = TRUE;
         m_bReverseSignal_InputSignalState = (tBool) param1;

         vHandleControl(EN_CONTROL_RVC__STATE_CHANGE_REVERSE);
      }
      break;
      case DISPVIDCTRL_DISPVIDCTRL_SWITCH_CAMERASUPPLYVOLTAGE:
      {
         // Switch camera supply voltage (Elmos register)
         tU8 param1 = puchData[3];
         (tVoid) _bSetCameraVccOn((tBool)param1);
      }
      break;

      default:
      break;
   }
}


/******************************************************************************/
/* FUNCTION     vTraceInfo()                                                  */
/******************************************************************************/
/**
*  \brief       trace information
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Rvc::vTraceInfo()
{
   ETG_TRACE_FATAL(("   CA-RVC :: vTraceInfo() *********** CA-RVC states ******************************************"));
   ETG_TRACE_FATAL(("   CA-RVC :: m_bRvcMode ................................ %-40u", m_bRvcMode));
   ETG_TRACE_FATAL(("   CA-RVC :: Reverse GPIO    : GPIO ID ................. %-40u", m_GpioId_Reverse));
   ETG_TRACE_FATAL(("   CA-RVC :: Reverse GPIO    : current GPIO value . .... %-40u", _bGetGPIOState_Reverse()));
   ETG_TRACE_FATAL(("   CA-RVC :: Reverse Signal  : Update Pending State .... %-40u", m_bReverseStateUpdatePending));
   ETG_TRACE_FATAL(("   CA-RVC :: Reverse Signal  : Input Signal State ...... %-40u", m_bReverseSignal_InputSignalState));
   ETG_TRACE_FATAL(("   CA-RVC :: Reverse Signal  : Signal State ............ %-40u", ETG_CENUM(tenReverseSignal_State, m_enReverseSignal_State)));
   ETG_TRACE_FATAL(("   CA-RVC :: Reverse Signal  : simulation mode via TTFis %-40u", m_bReverseSimulationModeViaTTFis));
   ETG_TRACE_FATAL(("   CA-RVC :: Camera Supply   : hardware State .......... %-40u", m_bCameraState));
   ETG_TRACE_FATAL(("   CA-RVC :: FSM Control_Rvc : Super State ............. %-40u", ETG_CENUM(tenFsmControl_Rvc_SuperState, m_enFsmControl_Rvc_SuperState)));
   ETG_TRACE_FATAL(("   CA-RVC :: m_enCameraVccCurrentState_OpenLoad ........ %-40u", ETG_CENUM(tenCurrentState, m_enCameraVccCurrentState_OpenLoad)));
   ETG_TRACE_FATAL(("   CA-RVC :: m_enCameraVccCurrentState_ShortGnd ........ %-40u", ETG_CENUM(tenCurrentState, m_enCameraVccCurrentState_ShortGnd)));
   ETG_TRACE_FATAL(("   CA-RVC :: m_enCameraVccVccState ..................... %-40u", ETG_CENUM(tenVccState, m_enCameraVccVccState)));
   ETG_TRACE_FATAL(("   CA-RVC :: m_enMicVccCurrentState_OpenLoad ........... %-40u", ETG_CENUM(tenCurrentState, m_enMicVccCurrentState_OpenLoad)));
   ETG_TRACE_FATAL(("   CA-RVC :: m_enMicVccCurrentState_ShortGnd ........... %-40u", ETG_CENUM(tenCurrentState, m_enMicVccCurrentState_ShortGnd)));
   ETG_TRACE_FATAL(("   CA-RVC :: m_enMicVccVccState ........................ %-40u", ETG_CENUM(tenVccState, m_enMicVccVccState)));
   ETG_TRACE_FATAL(("   CA-RVC :: m_enMicVccCurrentState_ShortGnd ........... %-40u", ETG_CENUM(tenCurrentState, m_enMicVccCurrentState_ShortGnd)));
   ETG_TRACE_FATAL(("   CA-RVC :: m_enMicVccVccState ........................ %-40u", ETG_CENUM(tenVccState, m_enMicVccVccState)));
   ETG_TRACE_FATAL(("   CA-RVC :: *********************************************************************************"));
   if (m_poFsm_ReverseSignal)
   {
      m_poFsm_ReverseSignal->vTraceDebugInfo();
   }
}

