/******************************************************************************/
/**
* \file    dispvidctrl_tclControl_Hmi.cpp
* \ingroup
*
* \brief
*
* \remark  Copyright : (c) 2015 Robert Bosch GmbH, Hildesheim
* \remark  Author    : Michael Niemann, Harry Wentland
* \remark  Scope     : AIVI
*
* \todo
*/
/******************************************************************************/

/*******************************************************************************
                        Includes
*******************************************************************************/
#include "dispvidctrl_tclControl_Mvc.h"
#include "dispvidctrl_tclFsm_Mvc.h"
#include "I_dispvidctrl_ServiceRvcIf.h"
#include "Idispvidctrl_ClientVcan.h"
#include "dispvidctrl_tclControl_Video.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_APPLICATION
#include "trcGenProj/Header/dispvidctrl_tclControl_Mvc.cpp.trc.h"
#endif

#define DISPVIDCTRL_VEHICLESPEED_TRESHOLD_SLOW      10 // [km/h]
#define DISPVIDCTRL_VEHICLESPEED_TRESHOLD_HIGH      15 // [km/h]

/******************************************************************************/
/* FUNCTION     dispvidctrl_tclControl_Mvc()                                  */
/******************************************************************************/
/**
*  \brief       default constructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclControl_Mvc::dispvidctrl_tclControl_Mvc()
: dispvidctrl_tclBaseIf(NULL)
, m_poFsm_Mvc(NULL)
, m_poControl_Video(NULL)
, m_poClientVcan(NULL)
, m_poServiceRvcIf(NULL)
, m_enFsmControl_Mvc_SuperState(DISPVIDCTRL_CONTROL_MVC__DISABLED)
, m_enRequestedView(EN_MVC_VIEW_MODE_NO_VIEW)
, m_enCurrentView(EN_MVC_VIEW_MODE_NO_VIEW)
, m_bAccState(false)
, m_bIgnition(false)
, m_bStaticGuidelineVisible(TRUE)
, m_bStaticGuidelineUserSetting(TRUE)
, m_bMvcMode(FALSE)
, m_bSpeedHigh(FALSE)
{
   NORMAL_M_ASSERT_ALWAYS();
}

/******************************************************************************/
/* FUNCTION     dispvidctrl_tclControl_Mvc()                                  */
/******************************************************************************/
/**
*  \brief       user defined constructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclControl_Mvc::dispvidctrl_tclControl_Mvc(dispvidctrl_tclAppMain* poMainAppl)
: dispvidctrl_tclBaseIf(poMainAppl)
, m_poControl_Video(NULL)
, m_poClientVcan(NULL)
, m_poServiceRvcIf(NULL)
, m_enFsmControl_Mvc_SuperState(DISPVIDCTRL_CONTROL_MVC__DISABLED)
, m_enRequestedView(EN_MVC_VIEW_MODE_NO_VIEW)
, m_enCurrentView(EN_MVC_VIEW_MODE_NO_VIEW)
, m_bAccState(false)
, m_bIgnition(false)
, m_bStaticGuidelineVisible(TRUE)
, m_bStaticGuidelineUserSetting(TRUE)
, m_bMvcMode(FALSE)
, m_bSpeedHigh(FALSE)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Mvc() entered."));

   // create new Hmi FSM
   m_poFsm_Mvc = dispvidctrl_tclFsm_Mvc::tclCreateInstance(this); // this: dispvidctrl_tclControl_Mvc*;
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poFsm_Mvc);
}

/******************************************************************************/
/* FUNCTION     dispvidctrl_tclControl_Mvc()                                  */
/******************************************************************************/
/**
*  \brief       default destructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclControl_Mvc::~dispvidctrl_tclControl_Mvc()
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Hmi() entered."));
   m_poFsm_Mvc = NULL;
   m_poControl_Video = NULL;
   m_poClientVcan = NULL;
   m_poServiceRvcIf = NULL;
}

/******************************************************************************/
/* FUNCTION     dispvidctrl_tclControl_Mvc()                                  */
/******************************************************************************/
/**
*  \brief       default destructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Mvc::vHandleMessage(dispvidctrl_tclBaseIf::TMsg* pMsg)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Mvc::vHandleMessage() entered %u -> data: %d.", ETG_CENUM(dispvidctrl_tclBaseIf::ECmdTypes , (tU32)pMsg->eCmd), pMsg->u.u32Data));

   switch(pMsg->eCmd)
   {
   case dispvidctrl_tclBaseIf::eControlModeEnabled:
      if (m_bMvcMode == TRUE) {
         vHandleControl(EN_CONTROL_MVC__ENABLE, pMsg->u.u32Data);
      }
      break;
   case dispvidctrl_tclBaseIf::eControlModeDisabled:
      if (m_bMvcMode == TRUE) {
         vHandleControl(EN_CONTROL_MVC__DISABLE, pMsg->u.u32Data);
      }
      break;

   case dispvidctrl_tclBaseIf::eControlMvc:
      vHandleControl((tenControl_Mvc) pMsg->u.u32Data);
      break;
      
   case dispvidctrl_tclBaseIf::eFsmMvc:
      vHandleControl(EN_CONTROL_MVC__FSM_MVC_INPUT_EVENT, pMsg->u.u32Data);
      break;
      
   case dispvidctrl_tclBaseIf::eAccChanged:
      if ( TRUE == (tBool) pMsg->u.u32Data ) {
         m_bAccState = TRUE;
      } else {
         m_bAccState = FALSE;
      }
      break;
      
   default:
      break;
   }
}

/******************************************************************************/
/* FUNCTION     dispvidctrl_tclControl_Mvc()                                  */
/******************************************************************************/
/**
*  \brief       default destructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Mvc::vHandleTraceMessage(const tUChar* puchData)
{
   DISPVIDCTRL_NULL_POINTER_CHECK(puchData);

   tU32 u32MsgCode = ((puchData[1]<<8) | puchData[2]);
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Hmi::vHandleTraceMessage(): trace command %d", u32MsgCode ));
}

/******************************************************************************/
/* FUNCTION     vGetReferences()                                              */
/******************************************************************************/
/**
*  \brief       default destructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Mvc::vGetReferences()
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Mvc::vGetReferences() entered."));

   m_poClientVcan = dynamic_cast<I_dispvidctrl_tclClientVcan*>(_cpoMain->getHandler("I_dispvidctrl_tclClientVcan"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poClientVcan);

   m_poControl_Video = dynamic_cast<dispvidctrl_tclControl_Video*>(_cpoMain->getHandler("dispvidctrl_tclControl_Video"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poControl_Video);

   m_poServiceRvcIf = dynamic_cast<Idispvidctrl_tclServiceRvcIf*>(_cpoMain->getHandler("Idispvidctrl_tclServiceRvcIf"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poServiceRvcIf);
}

/******************************************************************************/
/* FUNCTION     vStartCommunication()                                         */
/******************************************************************************/
/**
*  \brief       Start Communication
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Mvc::vStartCommunication()
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Mvc::vStartCommunication() entered."));
}

/******************************************************************************/
/* FUNCTION     vTraceInfo()                                                  */
/******************************************************************************/
/**
*  \brief       Trace all the member variables and other necessary info
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Mvc::vTraceInfo()
{
   ETG_TRACE_FATAL(("   CA-MVC :: vTraceInfo() *********** CA-MVC states ******************************************"));
   ETG_TRACE_FATAL(("   CA-MVC :: m_bStaticGuidelineVisible ................. %-40u", m_bStaticGuidelineVisible));
   ETG_TRACE_FATAL(("   CA-MVC :: m_bMvcMode ................................ %-40u", m_bMvcMode));
   ETG_TRACE_FATAL(("   CA-MVC :: m_enRequestedView ......................... %-40u", m_enRequestedView));
   ETG_TRACE_FATAL(("   CA-MVC :: m_enCurrentView  .......................... %-40u", m_enCurrentView));
   ETG_TRACE_FATAL(("   CA-MVC :: FSM Control_MVC : Super State ............. %-40u", ETG_CENUM(tenFsmControl_Mvc_SuperState, m_enFsmControl_Mvc_SuperState)));
   ETG_TRACE_FATAL(("   CA-MVC :: *********************************************************************************"));
}

/******************************************************************************/
/* FUNCTION     vGetConfiguration()                                           */
/******************************************************************************/
/**
*  \brief       Fetch all the KDS values
*
*  \param       pStConfigurationValues
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Mvc::vGetConfiguration(const TConfiguration* pStConfigurationValues)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Mvc::vGetConfiguration() entered."));
   
   m_bMvcMode = pStConfigurationValues->bMvcMode;
}

/******************************************************************************/
/* FUNCTION     vHandleControl()                                              */
/******************************************************************************/
/**
*  \brief       Fetch all the KDS values
*
*  \param       pStConfigurationValues
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Mvc::vHandleControl(tenControl_Mvc enControl, tU32 u32Value)
{
   ETG_TRACE_USR4(("   CA-MVC :: ............vHandleControl(): %d     ", ETG_CENUM(tenControl_Mvc, enControl)));

   switch(enControl)
   {
   case EN_CONTROL_MVC__FSM_MVC_INPUT_EVENT:
      if (m_poFsm_Mvc) {m_poFsm_Mvc->vNewTrigger(u32Value, TRUE);}
      break;

   case EN_CONTROL_MVC__ENABLE:
   case EN_CONTROL_MVC__DISABLE:
   case EN_CONTROL_MVC__HMI_VIEW_CHANGE_REQUEST:   
      _vOnEvent_FsmControl_Mvc(enControl, u32Value);
      break;
   
   // Mvc Ecu Requests a new View
   case EN_CONTROL_MVC__VIEW_MODE_CHANGED:
      {
         tU8 u8ViewStatus = EN_MVC_VIEW_MODE_NO_VIEW;
         if (m_poClientVcan)
         {
            m_poClientVcan->bGet_MVC_ViewModeStatus(u8ViewStatus);
         }
         
         _vOnEvent_FsmControl_Mvc(enControl, (tU32) u8ViewStatus);
      }
      break;
      
   case EN_CONTROL_MVC__WARNING_DISPLAY_STATUS_UPDATE:
      if (m_poServiceRvcIf) {
         m_poServiceRvcIf->sendMvcAlertRequestStatus((tBool) u32Value);
      }
      break;
      
   case EN_CONTROL_MVC__IGNITION:
      if ( (tBool) u32Value == TRUE ) { // Ignition Active
         m_bIgnition = TRUE;
         _vOnEvent_FsmControl_Mvc(EN_CONTROL_MVC__IGNITION_ON);
      }
      else { // Ignition Off
         m_bIgnition = FALSE;
         _vOnEvent_FsmControl_Mvc(EN_CONTROL_MVC__IGNITION_OFF);
      }
      break;

   case EN_CONTROL_MVC__CAN_MSG_TIMEOUT_TRUE:
   case EN_CONTROL_MVC__CAN_MSG_TIMEOUT_FALSE:
   case EN_CONTROL_MVC__CAN_MSG_TIMEOUT_CONFIRMED:
      _vOnEvent_FsmControl_Mvc(enControl, u32Value);
      break;
      
   case EN_CONTROL_MVC__STATIC_GUIDELINE_USER_SETTING_CHANGE:
      if (u32Value == 0x00) {
         m_bStaticGuidelineUserSetting = FALSE;
      } else {
         m_bStaticGuidelineUserSetting = TRUE;
      }
      //Removed as HMI user setting is handled in dispvidctrl_tclControl_Video.cpp NCG3D-77436
      //_vUpdateGuidelineState(m_enCurrentView == EN_MVC_VIEW_MODE_REAR);
      break;
      
   case EN_CONTROL_MVC__SETTINGS_OFF:
   case EN_CONTROL_MVC__SETTINGS_ON:
      _vOnEvent_FsmControl_Mvc(enControl, 0);
      break;
      
   case EN_CONTROL_MVC__SERVICE_REQUEST_VISUALISATION_OFF:
      // trigger to SM
      _vOnEvent_FsmControl_Mvc(EN_CONTROL_MVC__SETTINGS_OFF, 0);
      // Recheck the current view status and redraw the static guidelines if needed
      _bCheckAndUpdateMvcViewState();
      break;
      
   case EN_CONTROL_MVC__SERVICE_REQUEST_VISUALISATION_ON: // when switching between ( Mvc Settings <-> Mvc Maneuver Screen )      
      // Request to enter Mvc Maneuver and exit Mvc Settings
      if ( u32Value == rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_MVC_VISUALISATION_ON_CAMERA_SETTINGS )
      {
         // trigger to SM
         _vOnEvent_FsmControl_Mvc(EN_CONTROL_MVC__SETTINGS_ON, 0);

         // when entering the Settings view we must hide the static guidelines
         // _vUpdateGuidelineState(FALSE);
      }
      break;

   default:
      break;
   }
}

/******************************************************************************/
/* FUNCTION     vHandleFsmOutEvent_Mvc()                                      */
/******************************************************************************/
/**
*  \brief       Handler of the events from the FSM MVC
*
*  \param       Event and Param
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Mvc::vHandleFsmOutEvent_Mvc(tenFsmOutEvent_Mvc enFsmMvcOutEvent, tU32 u32Value)
{
   ETG_TRACE_USR4(("   CA-MVC :: ............vHandleFsmOutEvent_Mvc(): %d     ", ETG_CENUM(tenFsmOutEvent_Mvc, enFsmMvcOutEvent)));

   switch (enFsmMvcOutEvent)
   {
   case FSM_MVC_OUT__FSM_MVC_INPUT_EVENT:
      {
         tChar strIfName[] = "dispvidctrl_tclControl_Mvc";
         // post loop-back message to MVC FSM, u32Value will be eTrigger... value
         dispvidctrl_tclAppMain::theServer()->vPostMsgFsmMvc(strIfName, u32Value);
      }
      break;

   case FSM_MVC_OUT__HMI_VISUALISATION_ON:
      {
         tChar strIfName[] = "dispvidctrl_tclControl_Hmi";
         dispvidctrl_tclAppMain::theServer()->vPostMsgHmiVisualisationOn(strIfName);
      }
      break;

   case FSM_MVC_OUT__HMI_VISUALISATION_OFF:
      {
         tChar strIfName[] = "dispvidctrl_tclControl_Hmi";
         dispvidctrl_tclAppMain::theServer()->vPostMsgHmiVisualisationOff(strIfName, (tU32)0);
      }
      break;
      
   case FSM_MVC_OUT__HMI_VISUALISATION_ON_SETTINGS:
      {
         tChar strIfName[] = "dispvidctrl_tclControl_Hmi";
         dispvidctrl_tclAppMain::theServer()->vPostMsgHmiVisualisationOnCameraSettings(strIfName);
      }
      break;

   case FSM_MVC_OUT__MVC_CHECK_MVC_VIEW_REQUEST:
      {
         tChar strIfName[] = "dispvidctrl_tclControl_Mvc";
         // post loop-back message to Control MVC, u32Value will be Control event
         dispvidctrl_tclAppMain::theServer()->vPostMsgControlMvc(strIfName, EN_CONTROL_MVC__VIEW_MODE_CHANGED);
      }
      break;

   case FSM_MVC_OUT__MVC_CLEAR_VIEW_REQUEST:
      {
         // Clear the Request to MVC
         if (m_poClientVcan)
         {
            m_poClientVcan->vWriteMVC_ViewModeRequest(EN_MVC_VIEW_MODE_NO_VIEW);
         }
      }
      break;
      
   case FSM_MVC_OUT__MVC_SEND_REQUEST:
      {
         // Send Request to MVC
         if (m_poClientVcan)
         {
            m_poClientVcan->vWriteMVC_ViewModeRequest(m_enRequestedView);
         }
         // Request sent, clear it now
         m_enRequestedView = EN_MVC_VIEW_MODE_NO_VIEW;
      }
      break;
      
   case FSM_MVC_OUT__MVC_CAN_SIGNAL_TIMEOUT:
      _bCheckAndUpdateMvcViewState();
      break;
      
   case FSM_MVC_OUT__MVC_CHECK_CURRENT_SPEED:
      {
         tBool bSpeedHigh = _bIsVehicleSpeedHigh();
         
         if ( bSpeedHigh != m_bSpeedHigh )
         {
            m_bSpeedHigh = bSpeedHigh;
            
            tChar strIfName[] = "dispvidctrl_tclControl_Mvc";
            tU32 u32Trigger = dispvidctrl_tclFsm_Mvc::eTrigger_SpeedTooHighFalse;
            
            if ( m_bSpeedHigh )
            {
               u32Trigger = dispvidctrl_tclFsm_Mvc::eTrigger_SpeedTooHighTrue;
            }
            
            dispvidctrl_tclAppMain::theServer()->vPostMsgFsmMvc(strIfName, u32Trigger);
         }
      }
      break;

   default:
      break;
   }
}

/******************************************************************************/
/* FUNCTION     _vHandleFrameStatusUpdate()                                   */
/******************************************************************************/
/**
*  \brief       Fetch the current MVC View Mode Status
*
*  \param       u8CurrentStatus
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Mvc::_vHandleFrameStatusUpdate(tenControl_Mvc enControl, tU32 u32Value)
{
   ETG_TRACE_USR4(("   CA-MVC :: ............_vHandleFrameStatusUpdate(): %d     ", ETG_CENUM(tenControl_Mvc, enControl)));
   (tVoid) u32Value;

   // Current Timeout Status update to HMI
   rvc_fi_tcl_e8_RVC_SignalStatus oSignalStatus;

   switch(enControl)
   {
   case EN_CONTROL_MVC__CAN_MSG_TIMEOUT_TRUE:
      oSignalStatus.enType = rvc_fi_tcl_e8_RVC_SignalStatus::FI_EN_CURRENT_FAILURE;
      break;

   case EN_CONTROL_MVC__CAN_MSG_TIMEOUT_FALSE:
      oSignalStatus.enType = rvc_fi_tcl_e8_RVC_SignalStatus::FI_EN_SIGNAL_OK;
      break;

   case EN_CONTROL_MVC__CAN_MSG_TIMEOUT_CONFIRMED:
      oSignalStatus.enType = rvc_fi_tcl_e8_RVC_SignalStatus::FI_EN_CONFIRMED_FAILURE;
      break;

   default:
      break;
   }

   if (m_poServiceRvcIf) {
      m_poServiceRvcIf->sendCANFrameStatusStatus(oSignalStatus);
   }
}

/******************************************************************************/
/* FUNCTION     _vOnEvent_FsmControl_Mvc()                                    */
/******************************************************************************/
/**
*  \brief       Fetch the current MVC View Mode Status
*
*  \param       u8CurrentStatus
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Mvc::_vOnEvent_FsmControl_Mvc(tenControl_Mvc enControl, tU32 u32Value)
{
   ETG_TRACE_USR4(("   CA-MVC :: ........._vOnEvent_FsmControl_Mvc(): %d     ", ETG_CENUM(tenControl_Mvc, enControl)));
   ETG_TRACE_USR4(("   CA-MVC :: . FSM Control_Mvc Super State Entry: %u     ", ETG_CENUM(tenFsmControl_Mvc_SuperState, m_enFsmControl_Mvc_SuperState)));

   tBool bEventProcessed = TRUE;
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(u32Value);

   switch (m_enFsmControl_Mvc_SuperState)
   {
      /* ------------------------------------------------------------- */
      /*                                      D I S A B L E D          */
      /* ------------------------------------------------------------- */
      // activation via reverse gear on possible
   case DISPVIDCTRL_CONTROL_MVC__DISABLED:
      {
         if (EN_CONTROL_MVC__ENABLE == enControl)
         {
            // set AVM state machine into uninitialized state
            if (m_poFsm_Mvc)
            {
               m_poFsm_Mvc->vNewTrigger((tU32) dispvidctrl_tclFsm_Mvc::eTrigger_Init, TRUE);

               // new super state
               m_enFsmControl_Mvc_SuperState = DISPVIDCTRL_CONTROL_MVC__ENABLED;
            }
         }
         else 
         {
            bEventProcessed = FALSE;
         }
      }
      break;
      
      /* ------------------------------------------------------------- */
      /*                                      S T O P P E D            */
      /* todo
      * - Application state not Normal
      * - critical low voltage
      */
      /* --------------------------------------------------------- */

      /* ------------------------------------------------------------- */
      /*                                        E N A B L E D          */
      /* ------------------------------------------------------------- */
   case DISPVIDCTRL_CONTROL_MVC__ENABLED:
      {
         tU32 u32Trigger = dispvidctrl_tclFsm_Mvc::eTrigger_NONE;

         if (EN_CONTROL_MVC__DISABLE == enControl) 
         {
            // set AVM state machine into uninitialized state
            u32Trigger = dispvidctrl_tclFsm_Mvc::eTrigger_DeInit;
            
            // new super state
            m_enFsmControl_Mvc_SuperState = DISPVIDCTRL_CONTROL_MVC__DISABLED;
            
            // No Request should be sent to MVC ECU after this
            m_enRequestedView = EN_MVC_VIEW_MODE_NO_VIEW;
         }
         else 
         {
            switch(enControl)
            {
            case EN_CONTROL_MVC__VIEW_MODE_CHANGED:
               if (!_bCheckAndUpdateMvcViewState())
               {
                  u32Trigger = dispvidctrl_tclFsm_Mvc::eTrigger_VisualisationOff;
               }
               else
               {
                  u32Trigger = dispvidctrl_tclFsm_Mvc::eTrigger_VisualisationOn;
               }
               break;

            case EN_CONTROL_MVC__HMI_VIEW_CHANGE_REQUEST:
               u32Trigger = dispvidctrl_tclFsm_Mvc::eTrigger_View_Change_Request;
               m_enRequestedView = static_cast<ten_Mvc_ViewMode>(u32Value);
               break;
               
            case EN_CONTROL_MVC__CAN_MSG_TIMEOUT_TRUE:
               _vHandleFrameStatusUpdate(enControl, u32Value);
               break;

            case EN_CONTROL_MVC__CAN_MSG_TIMEOUT_FALSE:
               u32Trigger = dispvidctrl_tclFsm_Mvc::eTrigger_Frame_Timeout_False;
               _vHandleFrameStatusUpdate(enControl, u32Value);
               break;

            case EN_CONTROL_MVC__CAN_MSG_TIMEOUT_CONFIRMED:
               u32Trigger = dispvidctrl_tclFsm_Mvc::eTrigger_Frame_Timeout_True;
               _vHandleFrameStatusUpdate(enControl, u32Value);
               break;
               
            case EN_CONTROL_MVC__SETTINGS_OFF:
               u32Trigger = dispvidctrl_tclFsm_Mvc::eTrigger_VisualisationSettingsOff;
               break;
            case EN_CONTROL_MVC__SETTINGS_ON:
               u32Trigger = dispvidctrl_tclFsm_Mvc::eTrigger_VisualisationSettingsOn;
               break;
               
            case EN_CONTROL_MVC__IGNITION_ON:
               u32Trigger = dispvidctrl_tclFsm_Mvc::eTrigger_IgnitionOn;
               break;
            case EN_CONTROL_MVC__IGNITION_OFF:
               u32Trigger = dispvidctrl_tclFsm_Mvc::eTrigger_IgnitionOff;
               break;

            default:
               bEventProcessed = FALSE;
               u32Trigger = dispvidctrl_tclFsm_Mvc::eTrigger_NONE;
               break;
            }
         }
         
         if (bEventProcessed && m_poFsm_Mvc)
         {
            m_poFsm_Mvc->vNewTrigger(u32Trigger, TRUE);
         }
      }
      break;

   default:
      bEventProcessed = FALSE;
      break;

   } // switch super state

   ETG_TRACE_USR4(("   CA-MVC :: ................... Event Processed: %u     ", ETG_ENUM(DISPVIDCTRL_NoYes, bEventProcessed)));
   ETG_TRACE_USR4(("   CA-MVC :: .. FSM Control_Mvc Super State Exit: %u     ", ETG_CENUM(tenFsmControl_Mvc_SuperState, m_enFsmControl_Mvc_SuperState)));
}


/******************************************************************************/
/* FUNCTION     _vUpdateGuidelineState()                                      */
/******************************************************************************/
/**
*  \brief       Is it the rear view on which guidelines should be shown
*
*  \param       u8NewStatus
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Mvc::_vUpdateGuidelineState(const tU8 u8NewStatus)
{
   ETG_TRACE_FATAL(("   CA-MVC :: ........._vUpdateGuidelineState(): %u     ", ETG_CENUM(ten_Mvc_ViewMode, u8NewStatus)));

   DISPVIDCTRL_NULL_POINTER_CHECK(m_poControl_Video);
   
   //Display Guidelines only on the mvc rear view
   if(u8NewStatus == EN_MVC_VIEW_MODE_REAR)
   {
      ETG_TRACE_FATAL(("   CA-MVC :: ........._vUpdateGuidelineState(): Current MVC view is [%u], Update guidelines", ETG_CENUM(ten_Mvc_ViewMode, u8NewStatus)));
      m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__STATIC_GUIDELINE_UPDATE, 1);
   }
   else
   {
      ETG_TRACE_FATAL(("   CA-MVC :: ........._vUpdateGuidelineState(): Current MVC view is [%u], Guidelines is not displayed", ETG_CENUM(ten_Mvc_ViewMode, u8NewStatus)));
      m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__STATIC_GUIDELINE_UPDATE, 0);
   }
}

/******************************************************************************/
/* FUNCTION     _bCheckAndUpdateMvcViewState()                                */
/******************************************************************************/
/**
*  \brief       Is Mvc Visualisation On, also updates the ToggleGuideline Flag
*
*  \param       u8NewStatus
*  \return      none
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Mvc::_bCheckAndUpdateMvcViewState(tVoid)
{
   tBool bReturn = FALSE;
   
   tU8 u8NewStatus = EN_MVC_VIEW_MODE_NO_VIEW;
   if (m_poClientVcan)
   {
      m_poClientVcan->bGet_MVC_ViewModeStatus(u8NewStatus);
   }  
   
   ETG_TRACE_USR4(("   CA-MVC :: ........._bCheckAndUpdateMvcViewState(): %u     ", ETG_CENUM(ten_Mvc_ViewMode, u8NewStatus)));
   
   _vUpdateGuidelineState(u8NewStatus);
   
   if ( m_enCurrentView != static_cast<ten_Mvc_ViewMode>(u8NewStatus) )
   {
      if (m_poServiceRvcIf) {
         rvc_fi_tcl_e8_MVCViewingState oViewState;
         oViewState.enType = static_cast<rvc_fi_tcl_e8_MVCViewingState::tenType>(u8NewStatus);
         m_poServiceRvcIf->sendMvcViewStatusStatus(oViewState);
      }

      m_enCurrentView = static_cast<ten_Mvc_ViewMode>(u8NewStatus);
   }
   
   if ( ( m_enCurrentView >= EN_MVC_VIEW_MODE_FRONT ) && ( m_enCurrentView <= EN_MVC_VIEW_MODE_LEFT ) )
   {
      bReturn = TRUE;
   }
   else
   {
      bReturn = FALSE;
   }
   
   return bReturn;
}

/******************************************************************************/
/* FUNCTION     _bIsVehicleSpeedHigh                                          */
/******************************************************************************/
/**
*  \brief       requests the vehicle speed from Vcan and evaluates it to limits
*
*
*  \param       none
*  \return      bSuccess
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Mvc::_bIsVehicleSpeedHigh() const
{
   tU32 u32VehicleSpeedValue = 0;
   tBool bSpeedHigh = FALSE;

   if (m_poClientVcan)
   {
      if (TRUE == m_poClientVcan->bGet_VehicleSpeedSignal(u32VehicleSpeedValue))
      {
         if (u32VehicleSpeedValue > DISPVIDCTRL_VEHICLESPEED_TRESHOLD_HIGH)
         {
            bSpeedHigh = TRUE;
         }
         else if (u32VehicleSpeedValue < DISPVIDCTRL_VEHICLESPEED_TRESHOLD_SLOW)
         {
            bSpeedHigh = FALSE;
         }
         else
         {
            bSpeedHigh = FALSE;
         }
      }
      else
      {
         //enSpeedStatus = EN_SpeedUNKNOWN;
         ETG_TRACE_USR4(("   CA-RVC :: bGetSpeedStatus() failed!                     "));
         bSpeedHigh = FALSE;
      }
   }

   return (bSpeedHigh);
}

/******************************************************************************/
/* FUNCTION     IsMvcViewModeRear                                          */
/******************************************************************************/
/**
*  \brief       Returns true if the current MVC View Mode is Rear
*
*
*  \param       none
*  \return      bSuccess
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Mvc::IsMvcViewModeRear()
{
   ETG_TRACE_FATAL(("dispvidctrl_tclControl_Mvc::IsMvcViewModeRear Current MVC Mode = %u", ETG_CENUM(ten_Mvc_ViewMode, m_enCurrentView)));
   tBool bReturn = FALSE;

   if(m_enCurrentView == EN_MVC_VIEW_MODE_REAR)
   {
	   bReturn = TRUE;
   }

   return bReturn;
}
