/******************************************************************************/
/**
* \file    dispvidctrl_tclControl_Hmi.cpp
* \ingroup
*
* \brief
*
* \remark  Copyright : (c) 2015 Robert Bosch GmbH, Hildesheim
* \remark  Author    : Michael Niemann, Harry Wentland
* \remark  Scope     : AIVI
*
* \todo
*/
/******************************************************************************/

/*******************************************************************************
                        Includes
*******************************************************************************/
#include "dispvidctrl_tclFsm_Hmi.h"
#include "dispvidctrl_tclControl.h"
#include "dispvidctrl_tclControl_Avm.h"
#include "dispvidctrl_tclControl_Hmi.h"
#include "dispvidctrl_tclControl_Rvc.h"
#include "dispvidctrl_tclControl_Video.h"
#include "I_dispvidctrl_ClientDiaglogIf.h"
#include "Idispvidctrl_ClientVcan.h"
#include "dispvidctrl_tclControl_Mvc.h"
#include "dispvidctrl_datapool.h"
#include "I_dispvidctrl_ServiceRvcIf.h"

#define VD_DIAGLOG_S_IMPORT_INTERFACE_MSG
#include "vd_diaglog_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_APPLICATION
#include "trcGenProj/Header/dispvidctrl_tclControl_Hmi.cpp.trc.h"
#endif

/*******************************************************************************
                        Defines
*******************************************************************************/

//Timeout to wait for Video signal check
#ifndef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
  #define DISPVIDCTRL_VIDEO_SIGNAL_RECHECK_TIME_IN_MS     2000
#else
  #define DISPVIDCTRL_VIDEO_SIGNAL_RECHECK_TIME_IN_MS     3000
#endif

#ifndef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
  #define DISPVIDCTRL_BLACK_IN_TIME_IN_MS_AVM               250
  #define DISPVIDCTRL_BLACK_IN_TIME_IN_MS_RVC               300
  #define DISPVIDCTRL_BLACK_IN_TIME_IN_MS_MVC                10
  #define DISPVIDCTRL_BLACK_OUT_TIME_IN_MS_AVM              250
  #define DISPVIDCTRL_BLACK_OUT_TIME_IN_MS_RVC              300
  #define DISPVIDCTRL_BLACK_OUT_TIME_IN_MS_MVC               10
#else
  #define DISPVIDCTRL_BLACK_IN_TIME_IN_MS_AVM                10
  #define DISPVIDCTRL_BLACK_IN_TIME_IN_MS_RVC                10
  #define DISPVIDCTRL_BLACK_IN_TIME_IN_MS_MVC                10
  #define DISPVIDCTRL_BLACK_OUT_TIME_IN_MS_AVM               10
  #define DISPVIDCTRL_BLACK_OUT_TIME_IN_MS_RVC               10
  #define DISPVIDCTRL_BLACK_OUT_TIME_IN_MS_MVC               10
#endif

//------------------------------------------------------------------------------
//display back-light level for Video screen (Dimming)
#define DISPVIDCTRL_VIDEO_BACKLIGHT_LEVEL_DAY   (tU8) 18
#define DISPVIDCTRL_VIDEO_BACKLIGHT_LEVEL_NIGHT (tU8) 18
#define DISPVIDCTRL_VIDEO_BACKLIGHT_REQUEST_OFF (tU8) 255

/******************************************************************************/
/* FUNCTION     dispvidctrl_tclControl_Hmi()                                       */
/******************************************************************************/
/**
*  \brief       default constructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclControl_Hmi::dispvidctrl_tclControl_Hmi()
: dispvidctrl_tclBaseIf(OSAL_NULL)
//, m_poClientDimmingIf(OSAL_NULL)
, m_poControl(OSAL_NULL)
, m_poControl_Avm(OSAL_NULL)
, m_poControl_Rvc(OSAL_NULL)
, m_poControl_Video(OSAL_NULL)
, m_poClientDiaglogIf(OSAL_NULL)
, m_poControl_Mvc(OSAL_NULL)
, m_poFsm_Hmi(OSAL_NULL)
, m_poRvcServiceIf(OSAL_NULL)
, m_enFsmControl_Hmi_SuperState(DISPVIDCTRL_CONTROL_HMI__DISABLED)
, m_bVideoSignalCheck(FALSE)
, m_bVideoSignalState(FALSE)
, m_u8VideoSignalStateTestResult(midw_fi_tcl_e8_TestResult::FI_EN_NORESULT)
, _hTimerHandle_VideoSignalCheck(OSAL_C_INVALID_HANDLE)
, m_u8RearTrunkBehaviour(DISPVIDCTRL_AllianceRearTrunkBehave_NoAction)
, m_u8VideoResizedType(0xFF)
, m_bAvmMode(FALSE)
, m_bRvcMode(FALSE)
, m_bMvcMode(FALSE)
, m_bFapConfigured(FALSE)
, m_bFapModeActive(FALSE)
, m_bDiagVideoLayer(FALSE)
, m_bHmiAppVehicleReady(FALSE)
, m_bGuidelineAdjustmentScreen(FALSE)
{
}

/******************************************************************************/
/* FUNCTION     dispvidctrl_tclControl_Hmi()                                       */
/******************************************************************************/
/**
*  \brief       constructor
*
*  \param       pointer to control //main application
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclControl_Hmi::dispvidctrl_tclControl_Hmi(dispvidctrl_tclAppMain* poMainAppl)
: dispvidctrl_tclBaseIf(poMainAppl)
//, m_poClientDimmingIf(OSAL_NULL)
, m_poControl(OSAL_NULL)
, m_poControl_Avm(OSAL_NULL)
, m_poControl_Rvc(OSAL_NULL)
, m_poControl_Video(OSAL_NULL)
, m_poClientDiaglogIf(OSAL_NULL)
, m_poControl_Mvc(OSAL_NULL)
, m_poFsm_Hmi(OSAL_NULL)
, m_poRvcServiceIf(OSAL_NULL)
, m_enFsmControl_Hmi_SuperState(DISPVIDCTRL_CONTROL_HMI__DISABLED)
, m_bVideoSignalCheck(FALSE)
, m_bVideoSignalState(FALSE)
, m_u8VideoSignalStateTestResult(midw_fi_tcl_e8_TestResult::FI_EN_NORESULT)
, _hTimerHandle_VideoSignalCheck(OSAL_C_INVALID_HANDLE)
, m_u8RearTrunkBehaviour(DISPVIDCTRL_AllianceRearTrunkBehave_NoAction)
, m_u8VideoResizedType(0xFF)
, m_bAvmMode(FALSE)
, m_bRvcMode(FALSE)
, m_bMvcMode(FALSE)
, m_bFapConfigured(FALSE)
, m_bFapModeActive(FALSE)
, m_bDiagVideoLayer(FALSE)
, m_bHmiAppVehicleReady(FALSE)
, m_bGuidelineAdjustmentScreen(FALSE)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Hmi() entered."));

   // create new Hmi FSM
   m_poFsm_Hmi = dispvidctrl_tclFsm_Hmi::tclCreateInstance(this); // this: dispvidctrl_tclControl_Hmi*;
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poFsm_Hmi);

   // create Video signal check timer(for 2 seconds)
   // if the Video signal is not available even after 2 seconds of Black-in then an error-pop should be triggered
   if( OSAL_s32TimerCreate((OSAL_tpfCallback) _pfCallbackTimer_VideoSignalCheck, (tVoid*) this, &_hTimerHandle_VideoSignalCheck) != OSAL_OK)
   {
      //FATAL_M_ASSERT_ALWAYS();
      ETG_TRACE_USR4((" WORKER   :: E R R O R in Thread_Worker(): .......... creation Video Signal Check timer fails!"));
      _hTimerHandle_VideoSignalCheck = OSAL_C_INVALID_HANDLE;
   }
}


/******************************************************************************/
/* FUNCTION     ~dispvidctrl_tclControl_Hmi                                        */
/******************************************************************************/
/**
*  \brief       destructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclControl_Hmi::~dispvidctrl_tclControl_Hmi(tVoid)
{
   ETG_TRACE_USR4(("~dispvidctrl_tclControl_Hmi() entered."));

   //m_poClientDimmingIf = OSAL_NULL;
   m_poControl_Avm = OSAL_NULL;
   m_poControl_Rvc = OSAL_NULL;
   m_poControl_Video = OSAL_NULL;
   m_poClientDiaglogIf = OSAL_NULL;
   m_poFsm_Hmi  = OSAL_NULL;
   m_poControl_Mvc = OSAL_NULL;

   if (OSAL_C_INVALID_HANDLE != _hTimerHandle_VideoSignalCheck)
   {
      (void) OSAL_s32TimerSetTime(_hTimerHandle_VideoSignalCheck, 0, 0);
      (void) OSAL_s32TimerDelete(_hTimerHandle_VideoSignalCheck);
   }
}

/******************************************************************************/
/* FUNCTION     vGetConfiguration                                             */
/******************************************************************************/
/**
*  \brief       Function to get all configuration values needed by this class.
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Hmi::vGetConfiguration(const TConfiguration* pStConfigurationValues)
{
   TConfiguration_Fsm_Hmi StConfiguration_Fsm_Hmi;

   ETG_TRACE_USR4(("dispvidctrl_tclControl_Hmi::vGetConfiguration() entered."));

   m_bAvmMode = pStConfigurationValues->bAvmMode;
   m_bRvcMode = pStConfigurationValues->bRvcMode;
   m_bMvcMode = pStConfigurationValues->bMvcMode;
   m_bFapConfigured = pStConfigurationValues->bAvmType_Fap;  
   m_u8RearTrunkBehaviour = pStConfigurationValues->u8CameraRearTrunkBehaviour;
   
   StConfiguration_Fsm_Hmi.bRvcMode = m_bRvcMode;
   
   if (m_bAvmMode)
   {
      StConfiguration_Fsm_Hmi.u32BlackInTime = DISPVIDCTRL_BLACK_IN_TIME_IN_MS_AVM;
      StConfiguration_Fsm_Hmi.u32BlackOutTime = DISPVIDCTRL_BLACK_OUT_TIME_IN_MS_AVM;
   }
   else if (m_bMvcMode)
   {
      StConfiguration_Fsm_Hmi.u32BlackInTime = DISPVIDCTRL_BLACK_IN_TIME_IN_MS_MVC;
      StConfiguration_Fsm_Hmi.u32BlackOutTime = DISPVIDCTRL_BLACK_OUT_TIME_IN_MS_MVC;
   }
   else
   {
      StConfiguration_Fsm_Hmi.u32BlackInTime = DISPVIDCTRL_BLACK_IN_TIME_IN_MS_RVC;
      StConfiguration_Fsm_Hmi.u32BlackOutTime = DISPVIDCTRL_BLACK_OUT_TIME_IN_MS_RVC;
   }
   if (m_poFsm_Hmi)
   {
      m_poFsm_Hmi->vConfigure(&StConfiguration_Fsm_Hmi);
   }
}

/******************************************************************************/
/* FUNCTION     vGetReferences                                                */
/******************************************************************************/
/**
*  \brief       Function to get all reference needed by this class.
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Hmi::vGetReferences(tVoid)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Video::vGetReferences() entered."));

//   m_poClientDimmingIf = dynamic_cast<Idispvidctrl_tclClientDimmingIf*>(_cpoMain->getHandler("Idispvidctrl_tclClientDimmingIf"));
//   DISPVIDCTRL_NULL_POINTER_CHECK(m_poClientDimmingIf);

   m_poControl = dynamic_cast<dispvidctrl_tclControl*>(_cpoMain->getHandler("dispvidctrl_tclControl"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poControl);

   m_poControl_Avm = dynamic_cast<dispvidctrl_tclControl_Avm*>(_cpoMain->getHandler("dispvidctrl_tclControl_Avm"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poControl_Avm);

   m_poControl_Rvc = dynamic_cast<dispvidctrl_tclControl_Rvc*>(_cpoMain->getHandler("dispvidctrl_tclControl_Rvc"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poControl_Rvc);

   m_poControl_Video = dynamic_cast<dispvidctrl_tclControl_Video*>(_cpoMain->getHandler("dispvidctrl_tclControl_Video"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poControl_Video);

   m_poClientDiaglogIf = dynamic_cast<Idispvidctrl_tclClientDiaglogIf*>(_cpoMain->getHandler("Idispvidctrl_tclClientDiaglogIf"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poClientDiaglogIf);

   m_poControl_Mvc = dynamic_cast<dispvidctrl_tclControl_Mvc*>(_cpoMain->getHandler("dispvidctrl_tclControl_Mvc"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poControl_Mvc);
   
   m_poRvcServiceIf = dynamic_cast<Idispvidctrl_tclServiceRvcIf*>(_cpoMain->getHandler("Idispvidctrl_tclServiceRvcIf"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poRvcServiceIf);
}


/******************************************************************************/
/* FUNCTION     vStartCommunication                                           */
/******************************************************************************/
/**
*  \brief       Function to start all dynamic objects e.g. threads, ...
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Hmi::vStartCommunication(tVoid)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Hmi::vStartCommunication() entered."));
}


/******************************************************************************/
/* FUNCTION     _pfCallbackTimer_VideoSignalCheck                             */
/******************************************************************************/
/**
*  \brief
*
*  \param       pArg - pointer to this class
*  \return      none
*/
/******************************************************************************/
OSAL_tpfCallback dispvidctrl_tclControl_Hmi::_pfCallbackTimer_VideoSignalCheck(tVoid* pArg)
{
   //(tVoid) pArg;

   ETG_TRACE_USR4(("   CA-HMI :: _pfCallbackTimer_VideoSignalCheck(): ... ====>>                                  "));

   dispvidctrl_tclControl_Hmi* po_tclControl_Hmi = (dispvidctrl_tclControl_Hmi*)pArg;
   if (po_tclControl_Hmi)
   {
      if ((po_tclControl_Hmi->_cpoMain) && (po_tclControl_Hmi->_cpoMain->theServer()))
      {
         // post loop-back message to control _Hmi
         tChar strIfName[] = "dispvidctrl_tclControl_Hmi";
         po_tclControl_Hmi->_cpoMain->theServer()->vPostMsgControlHmi(strIfName, EN_CONTROL_HMI__VIDEO_SIGNAL_CHECK_TIMEOUT);
      }
   }
   return 0;
}

/******************************************************************************/
/* FUNCTION     _bSetTimer_VideoSignalCheck                                   */
/******************************************************************************/
/**
*  \brief       start or stop the timer
*
*  \param       timeout value
*  \return      tBool - timer start success state
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Hmi::_bSetTimer_VideoSignalCheck( OSAL_tMSecond msTimeout )
{
    tBool bSuccess = TRUE;
    if ( OSAL_C_INVALID_HANDLE != _hTimerHandle_VideoSignalCheck )   {
        if ( OSAL_s32TimerSetTime( _hTimerHandle_VideoSignalCheck, msTimeout, 0 ) != OSAL_OK )
        {
            bSuccess = FALSE;
        }
    }
    return(bSuccess);
}


/******************************************************************************/
/* FUNCTION     vHandleMessage(TMsg* pMsg)                                    */
/******************************************************************************/
/**
*  \brief       Handle worker events.
*
*  \param       message pointer
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Hmi::vHandleMessage(dispvidctrl_tclBaseIf::TMsg* pMsg)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Hmi::vHandleMessage() entered %u -> data: %d.", ETG_CENUM(dispvidctrl_tclBaseIf::ECmdTypes , (tU32)pMsg->eCmd), pMsg->u.u32Data));

   switch (pMsg->eCmd)
   {
      case dispvidctrl_tclBaseIf::eControlHmi:
         vHandleControl((tenControl_Hmi) pMsg->u.u32Data);
         break;
      case dispvidctrl_tclBaseIf::eFsmHmi:
         vHandleControl(EN_CONTROL_HMI__FSM_HMI_INPUT_EVENT, pMsg->u.u32Data);
         break;
      case dispvidctrl_tclBaseIf::eControlModeEnabled:
         vHandleControl(EN_CONTROL_HMI__ENABLE, pMsg->u.u32Data);
         break;
      case dispvidctrl_tclBaseIf::eControlModeDisabled:
         vHandleControl(EN_CONTROL_HMI__DISABLE, pMsg->u.u32Data);
         break;
      case dispvidctrl_tclBaseIf::eDiagVideoLayer:
         vHandleControl(EN_CONTROL_HMI__DIAG_VIDEO_LAYER, pMsg->u.u32Data);
         break;
      case dispvidctrl_tclBaseIf::eHmiVisualisationOff:
         vHandleControl( EN_CONTROL_HMI__VISUALISATION_OFF, pMsg->u.u32Data);
         break;
      case dispvidctrl_tclBaseIf::eHmiVisualisationOn:
         vHandleControl(EN_CONTROL_HMI__VISUALISATION_ON, pMsg->u.u32Data);
         break;
      case dispvidctrl_tclBaseIf::eHmiVisualisationOnCameraSettings:
         vHandleControl(EN_CONTROL_HMI__VISUALISATION_ON_CAMERA_SETTINGS, pMsg->u.u32Data);
         break;
      case dispvidctrl_tclBaseIf::eVideoStatus:
         if (pMsg->u.u32Data & DISPVIDCTRL_VIDEO__MASK_VIDEO_SIGNAL_STATUS)
         {
            ETG_TRACE_USR1(("CONTROL_HMI:  VIDEO_SIGNAL_TRUE"));
            vHandleControl(EN_CONTROL_HMI__VIDEO_SIGNAL_TRUE, (tU32) TRUE);
         }
         else
         {  
            ETG_TRACE_USR1(("CONTROL_HMI:  VIDEO_SIGNAL_FALSE"));
            vHandleControl(EN_CONTROL_HMI__VIDEO_SIGNAL_FALSE, (tU32) TRUE);
         }
         break;
      case dispvidctrl_tclBaseIf::eTriggerRecheckITC:
         vHandleControl(EN_CONTROL_HMI__SEND_NEXT_TESTRESULT);
         break;
      case dispvidctrl_tclBaseIf::eReverseOff:
         vHandleControl(EN_CONTROL_HMI__REVERSE_OFF);
         break;
      case dispvidctrl_tclBaseIf::eReverseOn:
         vHandleControl(EN_CONTROL_HMI__REVERSE_ON);
         break;         
      case dispvidctrl_tclBaseIf::eFapModeActive:
         if (m_bFapConfigured == TRUE) {
            if (1 == pMsg->u.u8Data || 2 == pMsg->u.u8Data) {
                m_bFapModeActive = TRUE;
            } else {
                m_bFapModeActive = FALSE;
            }
         }
         break; 
      default:
         break;
   }
}

/******************************************************************************/
/* FUNCTION     vHandleTraceMessage                                           */
/******************************************************************************/
/**
*  \brief       handle TTFis commands
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Hmi::vHandleTraceMessage(const tUChar* puchData)
{
   DISPVIDCTRL_NULL_POINTER_CHECK(puchData);

   tU32 u32MsgCode = ((puchData[1]<<8) | puchData[2]);
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Hmi::vHandleTraceMessage(): trace command %d", u32MsgCode ));

   switch (u32MsgCode)
   {
      case DISPVIDCTRL_DISPVIDCTRL_STIMULATE_FSM_HMI:
         {
            // Set once eTrigger for FSM_Hmi
            tU32 param1 = ((tU32)puchData[3] << 24) | ((tU32)puchData[4] << 16) | ((tU32)puchData[5] <<  8) | (tU32)puchData[6];

            vHandleControl(EN_CONTROL_HMI__FSM_HMI_INPUT_EVENT, param1);
         }
         break;
      case DISPVIDCTRL_DISPVIDCTRL_STIMULATE_CONTROL_HMI:
         {
            tU8 param1 = puchData[3];
            tU32 param2 = ((tU32)puchData[4] << 24) | ((tU32)puchData[5] << 16) | ((tU32)puchData[6] <<  8) | (tU32)puchData[7];

            vHandleControl((tenControl_Hmi) param1, param2);
         }
         break;
      default:
         break;
   }
}


/******************************************************************************/
/* FUNCTION     vHandleControl()                                              */
/******************************************************************************/
/**
*  \brief       trace information
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Hmi::vHandleControl(tenControl_Hmi enControl, tU32 u32Value)
{
   //OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32Value );

   ETG_TRACE_USR2(("   CA-HMI :: ............vHandleControl(): %d     ", ETG_CENUM(tenControl_Hmi, enControl)));

   switch (enControl)
   {
      case EN_CONTROL_HMI__DISABLE:
      case EN_CONTROL_HMI__ENABLE:
         _vOnEvent_FsmControl_Hmi(enControl, u32Value);
         break;
      //-------------------------------------------------------------------------------------------
      case EN_CONTROL_HMI__FSM_HMI_INPUT_EVENT:
         if (m_poFsm_Hmi) {m_poFsm_Hmi->vNewTrigger(u32Value, TRUE);}
         break;
      //-------------------------------------------------------------------------------------------
      case EN_CONTROL_HMI__VISUALISATION_OFF:
         _vOnEvent_FsmControl_Hmi(enControl, u32Value);
         break;
      case EN_CONTROL_HMI__VISUALISATION_ON:
      case EN_CONTROL_HMI__VISUALISATION_ON_CAMERA_SETTINGS:
         if (m_poControl_Video && m_poControl) {
            if ( FALSE == m_poControl_Video->bIsVideoSourceFound() ) {
               m_poControl->vHandleControl(EN_CONTROL__RETRY_MECHANISM_TIMEOUT, 0x01); // Force Retry
            }
         }
         _vOnEvent_FsmControl_Hmi(enControl, u32Value);
         break;
      case EN_CONTROL_HMI__OFF_REASON_CLEAN_UP:
      //-------------------------------------------------------------------------------------------
      case EN_CONTROL_HMI__HMI_AVAILABLE:
         if (m_poControl_Video) {
            m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__SET_B_HANDOVER_READY);
         }
         if (m_poFsm_Hmi)
         {
            _vSend_SetStatusHmiMResult(m_poFsm_Hmi->u32GetFsmState());
         }
         _vOnEvent_FsmControl_Hmi(enControl, u32Value);
         break;
      case EN_CONTROL_HMI__TOGGLE_GUIDELINES:
      case EN_CONTROL_HMI__DIAG_VIDEO_LAYER:
      case EN_CONTROL_HMI__DIMMING:
      case EN_CONTROL_HMI__GDL_ADJ_SCREEN:
         _vOnEvent_FsmControl_Hmi(enControl, u32Value);
         break;
      case EN_CONTROL_HMI__VIDEO_SIGNAL_FALSE:
         m_bVideoSignalState = FALSE;
         _vOnEvent_FsmVideoSignalCheck(EN_CONTROL_VID_TIMER__VIDEO_SIGNAL_FALSE);
         _vSetVideoSignalStatus(enControl, u32Value);
         break;
      case EN_CONTROL_HMI__VIDEO_SIGNAL_TRUE:
         m_bVideoSignalState = TRUE;
         _vOnEvent_FsmVideoSignalCheck(EN_CONTROL_VID_TIMER__VIDEO_SIGNAL_TRUE);
         _vSetVideoSignalStatus(enControl, u32Value);
         break;
      case EN_CONTROL_HMI__VIDEO_SIGNAL_CHECK_TIMEOUT:
         _vOnEvent_FsmVideoSignalCheck(EN_CONTROL_VID_TIMER__TIMEOUT);
         break;
      case EN_CONTROL_HMI__SEND_NEXT_TESTRESULT:
         _vUpdateITCStatus((tU8) midw_fi_tcl_e8_TestResult::FI_EN_NORESULT);
         break;
      case EN_CONTROL_HMI__SYSTEMSTATUS_CONFIRMATION:
         // for removing the early IPA mask
         if (m_bHmiAppVehicleReady == FALSE) {
            m_bHmiAppVehicleReady = TRUE;
            if (m_poControl_Video) {
               m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__HMI_AVAILABLE);
            }
         }
         _vHandleHMIConfirmationStatus(u32Value);
         break;
      case EN_CONTROL_HMI__VIDEO_SIGNAL_FALSE_CONFIRMED:
          _vSetVideoSignalStatus(enControl, u32Value);
          break;
          
      case EN_CONTROL_HMI__TAILGATE_STATUS:
          if ( m_u8RearTrunkBehaviour != DISPVIDCTRL_AllianceRearTrunkBehave_NoAction ) {
             _vHandleRearTrunkStateChange();
          }
          break;
      case EN_CONTROL_HMI__REVERSE_OFF:
          if (m_poFsm_Hmi)
          {
              m_poFsm_Hmi->vUpdateTriggerState(dispvidctrl_tclFsm_Hmi::eTrigger_REVERSE_OFF, TRUE);
              m_poFsm_Hmi->vUpdateTriggerState(dispvidctrl_tclFsm_Hmi::eTrigger_REVERSE_ON, FALSE);
          }
          _vOnEvent_FsmControl_Hmi(enControl, 0);
          break;
      case EN_CONTROL_HMI__REVERSE_ON:
          if (m_poFsm_Hmi)
          {
              m_poFsm_Hmi->vUpdateTriggerState(dispvidctrl_tclFsm_Hmi::eTrigger_REVERSE_ON, TRUE);
              m_poFsm_Hmi->vUpdateTriggerState(dispvidctrl_tclFsm_Hmi::eTrigger_REVERSE_OFF, FALSE);
          }
          break;
          
      case EN_CONTROL_HMI__VIDEO_FORCE_GRAPHICS_ON:
      case EN_CONTROL_HMI__VIDEO_FORCE_GRAPHICS_OFF:
          _vOnEvent_FsmControl_Hmi(enControl, u32Value);
          break;

      default:
         break;
   }
}

/******************************************************************************/
/* FUNCTION     vHandleFsmOutEvent_Hmi()                                 */
/******************************************************************************/
/**
*  \brief       handle function to post output events from FSM HMI
*
*  \param       event
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Hmi::vHandleFsmOutEvent_Hmi(tenFsmOutEvent_Hmi enFsmOutEvent_Hmi, tU32 u32Value)
{
   // FSM_HMI_OUT__FSM...       // input events for FSM HMI (Loopback to FSM HMI)
   // FSM_HMI_OUT__EVENTNAME... // input events for other FSM/Target

   ETG_TRACE_USR2(("   CA-HMI :: vHandleFsmOutEvent_Hmi(): %d     ", ETG_CENUM(tenFsmOutEvent_Hmi, enFsmOutEvent_Hmi)));

   switch (enFsmOutEvent_Hmi)
   {
      case FSM_HMI_OUT__FSM_HMI_INPUT_EVENT:
         {
            // post loop-back message to Hmi FSM, u32Value will be eTrigger... value
            tChar strIfName[] = "dispvidctrl_tclControl_Hmi";
            _cpoMain->theServer()->vPostMsgFsmHmi(strIfName, u32Value);
         }
         break;
      //-------------------------------------------------------------------------------------------
      case FSM_HMI_OUT__HMI_VISUALISATION_OFF:
      case FSM_HMI_OUT__HMI_VISUALISATION_OFF_VIDEO_SIGNAL_FAILURE:
      case FSM_HMI_OUT__HMI_VISUALISATION_OFF_SYSTEM_MALFUNCTION:
         _vSetSystemStatus(enFsmOutEvent_Hmi);
         break;
      case FSM_HMI_OUT__HMI_VISUALISATION_ON:
      case FSM_HMI_OUT__HMI_VISUALISATION_ON_CAMERA_SETTINGS:
      case FSM_HMI_OUT__HMI_VISUALISATION_ON_GUIDELINE_ADJUSTMENT:
         if (m_poControl_Video) {
            m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__SHOW_RVC);
         }
         _vSetSystemStatus(enFsmOutEvent_Hmi);
         break;
      case FSM_HMI_OUT__HMI_CONFIRM_VISUALISATION_BLACK_IN:
      case FSM_HMI_OUT__HMI_CONFIRM_VISUALISATION_BLACK_OUT:
         if (m_poControl_Video) {
            m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__SET_B_HANDOVER_READY);
            m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__BLACK_SCREEN_ON);
         }
         break;
      case FSM_HMI_OUT__HMI_VISUALISATION_BLACK_OUT:
      case FSM_HMI_OUT__HMI_VISUALISATION_BLACK_OUT_CAMERA_SETTINGS:
          if (m_poControl_Video) {
              m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__SET_B_HANDOVER_READY);
              if ( u32Value != 0xFFFFFFFF ) { // Should we draw Black Screen on Rvc Graphics Surface ?
                 m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__BLACK_SCREEN_ON);
              }
          }
          _vSetSystemStatus(enFsmOutEvent_Hmi);
          break;
      case FSM_HMI_OUT__HMI_VISUALISATION_BLACK_IN:
         _vSetSystemStatus(enFsmOutEvent_Hmi);
         //Resize video based on PSD
         if ( m_u8VideoResizedType != RVC_AVM_VIDEO_FULL_SCREEN )
         {
            m_u8VideoResizedType = RVC_AVM_VIDEO_FULL_SCREEN;
            if (m_poControl_Video) {
               m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__RESIZE_VIDEO, m_u8VideoResizedType);
            }
         }
         break;
      case FSM_HMI_OUT__HMI_VISUALISATION_BLACK_IN_CAMERA_SETTINGS:
          _vSetSystemStatus(enFsmOutEvent_Hmi);
          if ( m_u8VideoResizedType != RVC_AVM_VIDEO_SETTINGS_SCREEN )
          {
             m_u8VideoResizedType = RVC_AVM_VIDEO_SETTINGS_SCREEN;
             if (m_poControl_Video) {
                 m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__RESIZE_VIDEO, m_u8VideoResizedType);
             }
          }
          break;
      case FSM_HMI_OUT__HMI_VISUALISATION_DIRECT_ON:
         if (m_poControl_Video) {
            m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__DIRECT_ON);
         }
         _vSetSystemStatus(enFsmOutEvent_Hmi);
         break;
      case FSM_HMI_OUT__HMI_VISUALISATION_DIRECT_BLACK_IN:
      case FSM_HMI_OUT__HMI_VISUALISATION_DIRECT_BLACK_OUT:
         _vSetSystemStatus(enFsmOutEvent_Hmi);
         break;
      //-------------------------------------------------------------------------------------------
      case FSM_HMI_OUT__RVC_CAMERA_OFF:
         {
            if (m_poControl_Rvc) {
               m_poControl_Rvc->vHandleControl(EN_CONTROL_RVC__CAMERA_OFF);
            }
         }
         break;
      //-------------------------------------------------------------------------------------------
      case FSM_HMI_OUT__AVM_VIDEO_SIGNAL_FAILURE:
         {
            if (m_poControl_Avm)
            {
               m_poControl_Avm->vHandleControl(EN_CONTROL_AVM__VIDEO_SIGNAL_FAILURE);
            }
         }
         break;
      case FSM_HMI_OUT__AVM_VIDEO_SIGNAL_RESTORED:
         {
            if (m_poControl_Avm)
            {
               m_poControl_Avm->vHandleControl(EN_CONTROL_AVM__VIDEO_SIGNAL_RESTORED);
            }
         }
         break;
      //-------------------------------------------------------------------------------------------
      case FSM_HMI_OUT__VIDEO_SET_GAMMA_OFF:
            // via video interface
            if (m_poControl_Video) {
//todo: HMI FSM is not sending Gamma off currently
               m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__SET_VIDEO_GAMMA_OFF);
            }
         break;
      case FSM_HMI_OUT__VIDEO_SET_GAMMA_ON:
            // via video interface
            if (m_poControl_Video) {
               m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__SET_VIDEO_GAMMA_ON);
            }
         break;
      case FSM_HMI_OUT__VIDEO_CAPTURING_STOP:
            if (m_poControl_Video) {
               m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__CAPTURING_OFF);
            }
            // stop video signal detection check
            _vOnEvent_FsmVideoSignalCheck(EN_CONTROL_VID_TIMER__STOP);
         break;
      case FSM_HMI_OUT__VIDEO_CAPTURING_START:
            if (m_poControl_Video) {
               m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__CAPTURING_ON);
            }
         break;
      case FSM_HMI_OUT__VIDEO_CAPTURING_DIAG_START:
            // we have to activate the camera power supply normally only in case of RVC mode but we will do it general
            if (m_poControl_Rvc) {
               m_poControl_Rvc->vHandleControl(EN_CONTROL_RVC__DIAG_VIDEO_SCREEN_CAMERA_SUPPLY, 1);
            }
            // in case of AVM configuration we will send a "Camera Switch Press" CAN signal to AVM unit
            // for the case that a real AVM is connected and shall provide the video input
            if (m_bAvmMode) {
               if (m_poControl_Avm) {
                  m_poControl_Avm->vHandleControl(EN_CONTROL_AVM__CAMERA_BUTTON_PRESS);
               }
            }
            if (m_poControl_Video) {
               m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__CAPTURING_ON_DIAG);
            }
         break;
         
      case FSM_HMI_OUT__VIDEO_CAPTURING_GDL_ADJ_START:
         // we have to activate the camera power supply
         if (m_poControl_Rvc) {
            m_poControl_Rvc->vHandleControl(EN_CONTROL_RVC__DIAG_VIDEO_SCREEN_CAMERA_SUPPLY, 1);
         }
         if (m_poControl_Video) {
            m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__CAPTURING_ON_GDL_ADJ);
         }
         break;
      case FSM_HMI_OUT__VIDEO_SET_VIDEO_CONTROLS:
            if (m_poControl_Video) {
               m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__SET_VIDEO_CONTROLS);
            }
         break;
       case FSM_HMI_OUT__VIDEO_GUIDELINE_WARNINGTEXT_TOGGLE:
            if (m_poControl_Video) {
               m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__TOGGLE_GUIDELINE);
            }
         break;
      case FSM_HMI_OUT__VIDEO_GRAPHICS_OFF:
            if (m_poControl_Video) {
               m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__GRAPHICS_OFF);
            }
         break;
      case FSM_HMI_OUT__VIDEO_GRAPHICS_ON:
            if (m_poControl_Video) {
               m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__GRAPHICS_ON);
            }
         break;
      case FSM_HMI_OUT__VIDEO_SIGNAL_CHECK:
         {
            _vOnEvent_FsmVideoSignalCheck((tenControl_VideoSignalCheck) u32Value );
         }
         break;
      //-------------------------------------------------------------------------------------------
      case FSM_HMI_OUT__DIMMING_BACKLIGHT_CAMERA_MODE_OFF:
         //if (m_poClientDimmingIf) {
         //   (tVoid) m_poClientDimmingIf->sendDIM_DISP_SetVideoBrightnessLevelMStart((tU8) DISPVIDCTRL_VIDEO_BACKLIGHT_REQUEST_OFF, (tU8) DISPVIDCTRL_VIDEO_BACKLIGHT_REQUEST_OFF);
         //}
         break;
      case FSM_HMI_OUT__DIMMING_BACKLIGHT_CAMERA_MODE_ON:
         //if (m_poClientDimmingIf) {
         //   (tVoid) m_poClientDimmingIf->sendDIM_DISP_SetVideoBrightnessLevelMStart((tU8) DISPVIDCTRL_VIDEO_BACKLIGHT_LEVEL_DAY, (tU8) DISPVIDCTRL_VIDEO_BACKLIGHT_LEVEL_NIGHT);
         //}
         break;

      default:
      break;
   }
}

/******************************************************************************/
/* FUNCTION     _vOnEvent_FsmControl_Hmi                                      */
/******************************************************************************/
/**
*  \brief
*
*  \param       event, data
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Hmi::_vOnEvent_FsmControl_Hmi(tenControl_Hmi enControl, tU32 u32Value)
{
   ETG_TRACE_USR4(("   CA-HMI :: ........._vOnEvent_FsmControl_Hmi(): %d     ", ETG_CENUM(tenControl_Hmi, enControl)));
   ETG_TRACE_USR4(("   CA-HMI :: . FSM Control_Hmi Super State Entry: %u     ", ETG_CENUM(tenFsmControl_Hmi_SuperState, m_enFsmControl_Hmi_SuperState)));

   tBool bEventProcessed = TRUE;

   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(u32Value);

   switch (m_enFsmControl_Hmi_SuperState)
   {
      /* ------------------------------------------------------------- */
      /*                                      D I S A B L E D          */
      /* ------------------------------------------------------------- */
      case DISPVIDCTRL_CONTROL_HMI__DISABLED:
         {
            switch (enControl)
            {
            case EN_CONTROL_HMI__ENABLE:
               {
                  // first LATE transition after EARLY: use direct transition
                  if ((u32Value & DISPVIDCTRL_VIDEO__MASK_EARLY_ACTIV) > 0)
                  {
                     switch ((tenVideoStatus) (u32Value & DISPVIDCTRL_VIDEO__MASK_VIDEO_CAPTURING_STATUS))
                     {
                        case EN_VIDEO_STATUS__BLACK_OUT:
                           {
                              if (m_poFsm_Hmi)
                              {
                                 m_poFsm_Hmi->vNewTrigger(dispvidctrl_tclFsm_Hmi::eTrigger_DIRECT_BLACK_OUT, TRUE);
                              }
                           }
                           break;
                        case EN_VIDEO_STATUS__BLACK_IN:
                           {
                              if (m_poFsm_Hmi)
                              {
                                 m_poFsm_Hmi->vNewTrigger(dispvidctrl_tclFsm_Hmi::eTrigger_DIRECT_BLACK_IN, TRUE);
                              }
                           }
                           break;
                        case EN_VIDEO_STATUS__ON:
                           {
                              if (m_poFsm_Hmi)
                              {
                                 m_poFsm_Hmi->vNewTrigger(dispvidctrl_tclFsm_Hmi::eTrigger_DIRECT_ON, TRUE);
                              }
                           }
                           break;
                        case EN_VIDEO_STATUS__OFF:
                        default:
                           {
                              if (m_poFsm_Hmi)
                              {
                                 m_poFsm_Hmi->vNewTrigger(dispvidctrl_tclFsm_Hmi::eTrigger_DIRECT_OFF, TRUE);
                              }
                           }
                           break;
                     }// switch
                  }
                  else
                  {
                     // If we did stop because of critical under voltage or
                     // application state not normal, we have to clean up this
                     // remembered off trigger.
                     // Otherwise we will not change to black in if an new
                     // Visualization ON Request because of Screen reactivation will be received.
                     // Required only as long as we are in black out.
                     if (m_poFsm_Hmi)
                     {
                        m_poFsm_Hmi->vNewTrigger(dispvidctrl_tclFsm_Hmi::eTrigger_OFF_REASON_CLEAN_UP, TRUE);
                     }
                  }
                  // new super state
                  m_enFsmControl_Hmi_SuperState = DISPVIDCTRL_CONTROL_HMI__ENABLED;
               }
               break;
            case EN_CONTROL_HMI__DISABLE:
               {
                  if (((dispvidctrl_tclFsm_Hmi::tenHmiVisualisationOffReason) u32Value) == dispvidctrl_tclFsm_Hmi::EN_VD_RVC_HMI_VISUALISATION_OFF__INITIAL_OFF)
                  {
                     _vSetSystemStatus(FSM_HMI_OUT__HMI_VISUALISATION_OFF);
                  }
               }
               break;
            default:
               {
                  bEventProcessed = FALSE;
               }
               break;
            }
         }
         break;
      /* ------------------------------------------------------------- */
      /*                                        E N A B L E D          */
      /* ------------------------------------------------------------- */
      case DISPVIDCTRL_CONTROL_HMI__ENABLED:
      {
         if (EN_CONTROL_HMI__DISABLE == enControl)
         {
            // remember off reason with highest priority
            if (m_poFsm_Hmi) {
               if (((dispvidctrl_tclFsm_Hmi::tenHmiVisualisationOffReason) u32Value) > m_poFsm_Hmi->m_enHmiVisualisationOffReason)
               {
                  m_poFsm_Hmi->m_enHmiVisualisationOffReason = (dispvidctrl_tclFsm_Hmi::tenHmiVisualisationOffReason) u32Value;
               }
               m_poFsm_Hmi->vNewTrigger(dispvidctrl_tclFsm_Hmi::eTrigger_VISUALISATION_OFF, TRUE);
            }
            // new super state
            m_enFsmControl_Hmi_SuperState = DISPVIDCTRL_CONTROL_HMI__DISABLED;
         }
         else if (m_bDiagVideoLayer == TRUE)
         {
            bEventProcessed = TRUE;
            switch (enControl)
            {
               //Diagnose is not longer activ -> disable VideoLayer and DiagVideoLayer mode
               case EN_CONTROL_HMI__DIAG_VIDEO_LAYER:
                  {
                     if ((u32Value == 0) && (m_poFsm_Hmi != OSAL_NULL))
                     {
                        m_poFsm_Hmi->m_enHmiVisualisationOffReason = (dispvidctrl_tclFsm_Hmi::tenHmiVisualisationOffReason) dispvidctrl_tclFsm_Hmi::EN_VD_RVC_HMI_VISUALISATION_OFF__DIAG_VIDEO_LAYER_OFF;
                        m_poFsm_Hmi->vNewTrigger(dispvidctrl_tclFsm_Hmi::eTrigger_VISUALISATION_OFF, TRUE);
                        m_bDiagVideoLayer = FALSE;

                        // we can deactivate the camera power supply (was propably switched to ON in RVC mode only)
                        if (m_poControl_Rvc) {
                           m_poControl_Rvc->vHandleControl(EN_CONTROL_RVC__DIAG_VIDEO_SCREEN_CAMERA_SUPPLY, 0);
                        }
                        // we can send "Camera OFF Request" CAN signal to AVM unit
                        if (m_bAvmMode) {
                           if (m_poControl_Avm) {
                              m_poControl_Avm->vHandleControl(EN_CONTROL_AVM__PURPOSE_SWITCH);
                           }
                        }
                     }
                  }
                  break;
              default:
               {
                    ETG_TRACE_USR4(("   CA-HMI :: ... event blocked due to activated video layer from diagnose!"));
                    bEventProcessed = FALSE;
               }
               break;
            }
         }
         else if (m_bGuidelineAdjustmentScreen == TRUE)
         {
            bEventProcessed = TRUE;
            switch (enControl)
            {
               //Diagnose is not longer activ -> disable VideoLayer and GuidelineAdjustment mode
               case EN_CONTROL_HMI__GDL_ADJ_SCREEN:
                  {
                     if ((u32Value == 0) && (m_poFsm_Hmi != OSAL_NULL))
                     {
                        m_poFsm_Hmi->m_enHmiVisualisationOffReason = (dispvidctrl_tclFsm_Hmi::tenHmiVisualisationOffReason) dispvidctrl_tclFsm_Hmi::EN_VD_RVC_HMI_VISUALISATION_OFF__GUIDELINE_ADJUSTMENT_OFF;
                        m_poFsm_Hmi->vNewTrigger(dispvidctrl_tclFsm_Hmi::eTrigger_VISUALISATION_OFF, TRUE);
                        m_bGuidelineAdjustmentScreen = FALSE;

                        // we can deactivate the camera power supply
                        if (m_poControl_Rvc) {
                           m_poControl_Rvc->vHandleControl(EN_CONTROL_RVC__DIAG_VIDEO_SCREEN_CAMERA_SUPPLY, 0);
                        }
                     }
                  }
                  break;
              default:
               {
                    ETG_TRACE_USR4(("   CA-HMI :: ... event blocked due to activated video layer from guideline adjustment screen!"));
                    bEventProcessed = FALSE;
               }
               break;
            }
         }
         else
         {
            tU32 u32FSM_Event;

            bEventProcessed = TRUE;
            switch (enControl)
            {
               case EN_CONTROL_HMI__FSM_HMI_INPUT_EVENT                 : u32FSM_Event = u32Value; break;
               case EN_CONTROL_HMI__VISUALISATION_OFF                   : u32FSM_Event = dispvidctrl_tclFsm_Hmi::eTrigger_VISUALISATION_OFF;
                  {
                     // remember off reason with highest priority
                     if (m_poFsm_Hmi) {
                        if (((dispvidctrl_tclFsm_Hmi::tenHmiVisualisationOffReason) u32Value) > m_poFsm_Hmi->m_enHmiVisualisationOffReason) {
                           m_poFsm_Hmi->m_enHmiVisualisationOffReason = (dispvidctrl_tclFsm_Hmi::tenHmiVisualisationOffReason) u32Value;
                        }
                     }
                  }
                  break;
               case EN_CONTROL_HMI__DIAG_VIDEO_LAYER:
                  {
                     if (u32Value == 1)
                     {
                        u32FSM_Event = dispvidctrl_tclFsm_Hmi::eTrigger_VISUALISATION_ON_CAMERA_DIAG;
                        //Diagnose is activ -> show hard VideoLayer, disable all video signal and video ITC control mechanism
                        m_bDiagVideoLayer = TRUE;
                     }
                     else
                     {
                        u32FSM_Event = dispvidctrl_tclFsm_Hmi::eTrigger_NONE;
                     }
                  }
                  break;
               case EN_CONTROL_HMI__GDL_ADJ_SCREEN:
                  {
                     if (u32Value == 1)
                     {
                        u32FSM_Event = dispvidctrl_tclFsm_Hmi::eTrigger_VISUALISATION_ON_GUIDELINE_ADJUSTMENT;
                        m_bGuidelineAdjustmentScreen = TRUE;
                     }
                     else
                     {
                        u32FSM_Event = dispvidctrl_tclFsm_Hmi::eTrigger_NONE;
                     }
                  }
                  break;
               case EN_CONTROL_HMI__VISUALISATION_ON                    : u32FSM_Event = dispvidctrl_tclFsm_Hmi::eTrigger_VISUALISATION_ON; break;
               case EN_CONTROL_HMI__VISUALISATION_ON_CAMERA_SETTINGS    : u32FSM_Event = dispvidctrl_tclFsm_Hmi::eTrigger_VISUALISATION_ON_CAMERA_SETTINGS; break;
               case EN_CONTROL_HMI__VIDEO_SIGNAL_FALSE_CONFIRMED        : u32FSM_Event = dispvidctrl_tclFsm_Hmi::eTrigger_VIDEO_SIGNAL_FALSE; break;
               case EN_CONTROL_HMI__VIDEO_SIGNAL_TRUE                   : u32FSM_Event = dispvidctrl_tclFsm_Hmi::eTrigger_VIDEO_SIGNAL_TRUE; break;
               case EN_CONTROL_HMI__OFF_REASON_CLEAN_UP                 : u32FSM_Event = dispvidctrl_tclFsm_Hmi::eTrigger_OFF_REASON_CLEAN_UP; break;
               //-------------------------------------------------------------------------------------------
               case EN_CONTROL_HMI__TOGGLE_GUIDELINES                   : u32FSM_Event = dispvidctrl_tclFsm_Hmi::eTrigger_TOGGLE_GUIDELINES; break;
               case EN_CONTROL_HMI__REVERSE_OFF                         : u32FSM_Event = dispvidctrl_tclFsm_Hmi::eTrigger_REVERSE_OFF; break;
               case EN_CONTROL_HMI__HMI_AVAILABLE                       : u32FSM_Event = dispvidctrl_tclFsm_Hmi::eTrigger_HMI_AVAILABLE; break;
               case EN_CONTROL_HMI__VIDEO_FORCE_GRAPHICS_OFF            : u32FSM_Event = dispvidctrl_tclFsm_Hmi::eTrigger_GRAPHICS_OFF; break;
               case EN_CONTROL_HMI__VIDEO_FORCE_GRAPHICS_ON             : u32FSM_Event = dispvidctrl_tclFsm_Hmi::eTrigger_GRAPHICS_ON; break;
               default:
               {
                    bEventProcessed = FALSE;
                    u32FSM_Event = dispvidctrl_tclFsm_Hmi::eTrigger_NONE;
               }
               break;
            }

            if (bEventProcessed == TRUE) {
               if (m_poFsm_Hmi) {
                  m_poFsm_Hmi->vNewTrigger(u32FSM_Event, TRUE);
               }
            }
         }
      }
      break;

      default:
         break;
   } // switch super state

   ETG_TRACE_USR4(("   CA-HMI :: ................... Event Processed: %u     ", ETG_ENUM(DISPVIDCTRL_NoYes, bEventProcessed)));
   ETG_TRACE_USR4(("   CA-HMI :: .. FSM Control_Hmi Super State Exit: %u     ", ETG_CENUM(tenFsmControl_Hmi_SuperState, m_enFsmControl_Hmi_SuperState)));
}


/******************************************************************************/
/* FUNCTION     _vOnEvent_FsmVideoSignalCheck                                 */
/******************************************************************************/
/**
*  \brief
*
*  \param       event, data
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Hmi::_vOnEvent_FsmVideoSignalCheck(tenControl_VideoSignalCheck enControl, tU32 u32Value)
{
   ETG_TRACE_USR4(("   CA-HMI :: ...._vOnEvent_FsmVideoSignalCheck(): %d     ", ETG_CENUM(tenControl_VideoSignalCheck, enControl)));

   tBool bEventProcessed = TRUE;

   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(u32Value);

   if (m_bVideoSignalCheck == FALSE )  {
      /* ------------------------------------------------------------- */
      /*                                      D I S A B L E D          */
      /* ------------------------------------------------------------- */
      switch (enControl)
      {
         case EN_CONTROL_VID_TIMER__START:
            {
               if (m_bDiagVideoLayer == FALSE)
               {
                  m_bVideoSignalCheck = TRUE;
                  _bSetTimer_VideoSignalCheck( (OSAL_tMSecond) DISPVIDCTRL_VIDEO_SIGNAL_RECHECK_TIME_IN_MS );
               }
               else
               {
                  ETG_TRACE_USR4(("   CA-HMI :: ... video signal check not started: diagnose activated video layer!"));
                  bEventProcessed = FALSE;
               }
            }
            break;
         default:
            {
               bEventProcessed = FALSE;
            }
            break;
      }
   }
   else {
      /* ------------------------------------------------------------- */
      /*                                        E N A B L E D          */
      /* ------------------------------------------------------------- */
      switch (enControl)
      {
         case EN_CONTROL_VID_TIMER__STOP:
            {
               m_bVideoSignalCheck = FALSE;
               if (_bSetTimer_VideoSignalCheck((OSAL_tMSecond) 0) == FALSE) {
                  ETG_TRACE_ERR(("  CA-HMI :: E R R O R in _bSetTimer_VideoSignalCheck():  stop Timer VideoSignalCheck failed! "));
                  //Assert, update Error trace
               }
            }
            break;
         case EN_CONTROL_VID_TIMER__VIDEO_SIGNAL_FALSE:
            {
               // Start Timer, Video Signal State will be checked after 2000ms (Nissan) / 3000ms (Renault)
               _bSetTimer_VideoSignalCheck( (OSAL_tMSecond) DISPVIDCTRL_VIDEO_SIGNAL_RECHECK_TIME_IN_MS );
            }
            break;
         case EN_CONTROL_VID_TIMER__VIDEO_SIGNAL_TRUE:
            {
               //Stop Timer, Video signal check can be terminated immediately
               if (_bSetTimer_VideoSignalCheck((OSAL_tMSecond) 0) == FALSE) {
                  ETG_TRACE_ERR(("  CA-HMI :: E R R O R in _bSetTimer_VideoSignalCheck():  stop Timer VideoSignalCheck failed! "));
                  //Assert, update Error trace
               }

               _vOnEvent_FsmControl_Hmi(EN_CONTROL_HMI__VIDEO_SIGNAL_TRUE);

               _vUpdateITCStatus((tU8) midw_fi_tcl_e8_TestResult::FI_EN_PASSED);
            }
            break;
         case EN_CONTROL_VID_TIMER__TIMEOUT:
            {
               if(m_bVideoSignalState == TRUE ) {
                  _vOnEvent_FsmControl_Hmi(EN_CONTROL_HMI__VIDEO_SIGNAL_TRUE);

                  // report ITC as Passed
                  _vUpdateITCStatus((tU8) midw_fi_tcl_e8_TestResult::FI_EN_PASSED);
               }
               else  {
                  // VideoSignalState is still FALSE, Trigger is now evaluated
                  _vOnEvent_FsmControl_Hmi(EN_CONTROL_HMI__VIDEO_SIGNAL_FALSE_CONFIRMED);

                  // report ITC as Failed
                  _vUpdateITCStatus((tU8) midw_fi_tcl_e8_TestResult::FI_EN_FAILED);

                  // Set HMI Video Signal State
                  _vSetVideoSignalStatus(EN_CONTROL_HMI__VIDEO_SIGNAL_FALSE_CONFIRMED);
               }
            }
            break;
         default:
            {
               bEventProcessed = FALSE;
            }
            break;
      }
   }

   ETG_TRACE_USR4(("   CA-HMI :: ................... Event Processed: %u     ", ETG_ENUM(DISPVIDCTRL_NoYes, bEventProcessed)));
}

/******************************************************************************/
/* FUNCTION     _vSetVideoSignalStatus                                        */
/******************************************************************************/
/**
*  \brief       send system status corresponding to configuration (AVM or RVC)
*
*  \param       tU32 u32Fsm_HmiState
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Hmi::_vSetVideoSignalStatus(tenControl_Hmi enControl, tU32 /*u32Value*/)
{
   ETG_TRACE_USR4(("   CA-HMI :: ...._vSetVideoSignalStatus(): %d     ", ETG_CENUM(tenControl_Hmi, enControl)));

   rvc_fi_tcl_e8_RVC_SignalStatus e8VideoStatus;

   switch(enControl)
   {
   case EN_CONTROL_HMI__VIDEO_SIGNAL_TRUE:
       e8VideoStatus.enType = rvc_fi_tcl_e8_RVC_SignalStatus::FI_EN_SIGNAL_OK;
       break;
   case EN_CONTROL_HMI__VIDEO_SIGNAL_FALSE:
       e8VideoStatus.enType = rvc_fi_tcl_e8_RVC_SignalStatus::FI_EN_CURRENT_FAILURE;
       break;
   case EN_CONTROL_HMI__VIDEO_SIGNAL_FALSE_CONFIRMED:
       e8VideoStatus.enType = rvc_fi_tcl_e8_RVC_SignalStatus::FI_EN_CONFIRMED_FAILURE;
       break;
   default:
//todo: should be checked if default value should Signal OK?
       e8VideoStatus.enType = rvc_fi_tcl_e8_RVC_SignalStatus::FI_EN_SIGNAL_OK;
       break;
   }

   // NCG3D-33462: Do not use the NULL_CHECK macro here, because, during Early Video Service Handler might not be ready to handle this request
   if (m_poRvcServiceIf) {
      m_poRvcServiceIf->sendVideoSignalStatusStatus(e8VideoStatus);
   }
}

/******************************************************************************/
/* FUNCTION     _vSend_SetStatusHmiMResult                                    */
/******************************************************************************/
/**
*  \brief       send system status corresponding to configuration (AVM or RVC)
*
*  \param       tU32 u32Fsm_HmiState
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Hmi::_vSend_SetStatusHmiMResult(tU32 u32Fsm_HmiState)
{
   rvc_fi_tcl_e8_RVC_SystemStatus e8SystemStatus;

   switch(u32Fsm_HmiState)
   {
      case dispvidctrl_tclFsm_Hmi::eState_BLACK_IN:
         if (m_bAvmMode) {
            e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_AVM_VISUALISATION_BLACK_IN;
         } else if (m_bMvcMode) {
            e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_MVC_VISUALISATION_BLACK_IN;
         } else {
            e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_RVC_VISUALISATION_BLACK_IN;
         }
         break;
      case dispvidctrl_tclFsm_Hmi::eState_ON:
         if (m_bAvmMode) {
            e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_AVM_VISUALISATION_ON;
         } else if (m_bMvcMode) {
            e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_MVC_VISUALISATION_ON;
         }  else {
            e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_RVC_VISUALISATION_ON;
         }
         break;
      case dispvidctrl_tclFsm_Hmi::eState_BLACK_OUT:
         if (m_bAvmMode) {
            e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_AVM_VISUALISATION_BLACK_OUT;
         } else if (m_bMvcMode) {
            e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_MVC_VISUALISATION_BLACK_OUT;
         }  else {
            e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_RVC_VISUALISATION_BLACK_OUT;
         }
         break;
      case dispvidctrl_tclFsm_Hmi::eState_OFF:
      default:
         e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_VISUALISATION_OFF;
         break;
   }

   // send MethodResult for SetStatusHmi
   // NCG3D-33462: Do not use the NULL_CHECK macro here, because, during Early Video Service Handler might not be ready to handle this request
   if (m_poRvcServiceIf) {
      m_poRvcServiceIf->sendSetStatusHmiMResult(e8SystemStatus);
   }
}

/******************************************************************************/
/* FUNCTION     _vSetSystemStatus                                             */
/******************************************************************************/
/**
*  \brief       send system status corresponding to configuration (AVM or RVC)
*
*  \param       tenFsmOutEvent_Hmi enFsmHmiOutEvent
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Hmi::_vSetSystemStatus(tenFsmOutEvent_Hmi enFsmHmiOutEvent)
{
   rvc_fi_tcl_e8_RVC_SystemStatus e8SystemStatus;

   switch (enFsmHmiOutEvent)
   {
      case FSM_HMI_OUT__HMI_VISUALISATION_OFF:
         e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_VISUALISATION_OFF;
         break;
      case FSM_HMI_OUT__HMI_VISUALISATION_OFF_VIDEO_SIGNAL_FAILURE:
         e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_VISUALISATION_OFF_VIDEO_SIGNAL_FAILURE;
         break;
      case FSM_HMI_OUT__HMI_VISUALISATION_OFF_SYSTEM_MALFUNCTION:
         e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_VISUALISATION_OFF_SYSTEM_MALFUNCTION;
         break;
      case FSM_HMI_OUT__HMI_VISUALISATION_DIRECT_ON:
      case FSM_HMI_OUT__HMI_VISUALISATION_ON:
         if (m_bAvmMode) {
            e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_AVM_VISUALISATION_ON;
         } else if (m_bMvcMode) {
            e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_MVC_VISUALISATION_ON;
         } else {
            e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_RVC_VISUALISATION_ON;
         }
         break;
      case FSM_HMI_OUT__HMI_VISUALISATION_ON_CAMERA_SETTINGS:
         if (m_bAvmMode) {
            e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_AVM_VISUALISATION_ON_CAMERA_SETTINGS;
         } else if (m_bMvcMode) {
            e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_MVC_VISUALISATION_ON_CAMERA_SETTINGS;
         } else {
            e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_RVC_VISUALISATION_ON_CAMERA_SETTINGS;
         }
         break;
      case FSM_HMI_OUT__HMI_VISUALISATION_ON_GUIDELINE_ADJUSTMENT:
            e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_RVC_VISUALISATION_ON_GUIDELINE_ADJUSTMENT;
         break;
      case FSM_HMI_OUT__HMI_VISUALISATION_DIRECT_BLACK_IN:
      case FSM_HMI_OUT__HMI_VISUALISATION_BLACK_IN:
         if (m_bAvmMode) {
            e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_AVM_VISUALISATION_BLACK_IN;
         } else if (m_bMvcMode) {
            e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_MVC_VISUALISATION_BLACK_IN;
         } else {
            e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_RVC_VISUALISATION_BLACK_IN;
         }
         break;
      case FSM_HMI_OUT__HMI_VISUALISATION_DIRECT_BLACK_OUT:
      case FSM_HMI_OUT__HMI_VISUALISATION_BLACK_OUT:
         if (m_bAvmMode) {
            e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_AVM_VISUALISATION_BLACK_OUT;
         } else if (m_bMvcMode) {
            e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_MVC_VISUALISATION_BLACK_OUT;
         } else {
            e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_RVC_VISUALISATION_BLACK_OUT;
         }
         break;
      case FSM_HMI_OUT__HMI_VISUALISATION_BLACK_IN_CAMERA_SETTINGS:
          e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_VISUALISATION_BLACK_IN_CAMERA_SETTINGS;
          break;
      case FSM_HMI_OUT__HMI_VISUALISATION_BLACK_OUT_CAMERA_SETTINGS:
          e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_VISUALISATION_BLACK_OUT_CAMERA_SETTINGS;
          break;
      default:
         e8SystemStatus.enType = rvc_fi_tcl_e8_RVC_SystemStatus::FI_EN_VISUALISATION_OFF;
         break;
   }

   // NCG3D-33462: Do not use the NULL_CHECK macro here, because, during Early Video Service Handler might not be ready to handle this request
   if (m_poRvcServiceIf) {
      ETG_TRACE_USR4(("   CA-HMI :: _vSetSystemStatus(%u)", ETG_ENUM(DISPVIDCTRL_E8_RVC_SYSTEMSTATUS,e8SystemStatus.enType) ));
      m_poRvcServiceIf->sendSystemStatusStatus(e8SystemStatus);      
   }
}


/******************************************************************************/
/* FUNCTION     _vUpdateITCStatus()                                           */
/******************************************************************************/
/**
*  \brief       trace information
*
*  \param       new ITC Status for CVBS In
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Hmi::_vUpdateITCStatus(tU8 u8ITCStatus)
{
   tenITC_ID enITCCode = enITC_ID_VIDEO_NO_SIGNAL;

   // "FI_EN_NORESULT" in case of RVC not configured
   if (m_bRvcMode == TRUE) 
   {
      enITCCode = enITC_ID_VIDEO_NO_SIGNAL;
   }
   else if (m_bAvmMode == TRUE)
   {
      enITCCode = enITC_ID_VIDEO_AVM_NO_SIGNAL;
   }
   else 
   {
      u8ITCStatus = (tU8) midw_fi_tcl_e8_TestResult::FI_EN_NORESULT;
   }

   // report to Diaglog only if the ITC status has changed
   if (m_u8VideoSignalStateTestResult != u8ITCStatus)
   {
      // report ITC as Passed
      if (m_poClientDiaglogIf)
      {
         m_u8VideoSignalStateTestResult = u8ITCStatus;
         m_poClientDiaglogIf->vSetITC(enITCCode, static_cast<midw_fi_tcl_e8_TestResult::tenType>(u8ITCStatus));
      }
   }
}

/******************************************************************************/
/* FUNCTION     _vHandleHMIConfirmationStatus()                               */
/******************************************************************************/
/**
*  \brief       HMI has entered into the SystemStatus as decided by the FSM HMI 
*
*  \param       System Status in which HMI as entered
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Hmi::_vHandleHMIConfirmationStatus(const tU32 u32Value)
{
   tU32 u32Trigger = dispvidctrl_tclFsm_Hmi::eTrigger_NONE;

   switch (u32Value)
   {
   case EN_HMI_CONFIRM_SYSTEMSTATUS__OFF:
   case EN_HMI_CONFIRM_SYSTEMSTATUS__OFF_VIDEO_SIGNAL_FAILURE:
     u32Trigger = dispvidctrl_tclFsm_Hmi::eTrigger_HMI_OFF_CONFIRM;
     // always reset to full video size when going off.
     m_u8VideoResizedType = RVC_AVM_VIDEO_FULL_SCREEN;
     if (m_poControl_Video) {
        m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__RESIZE_VIDEO, m_u8VideoResizedType);
     }
     break;

   case EN_HMI_CONFIRM_SYSTEMSTATUS__RVC_ON_GUIDELINE_ADJUSTMENT:
     // do nothing
     break;

   case EN_HMI_CONFIRM_SYSTEMSTATUS__RVC_BLACK_IN:
   case EN_HMI_CONFIRM_SYSTEMSTATUS__AVM_BLACK_IN:
   case EN_HMI_CONFIRM_SYSTEMSTATUS__MVC_BLACK_IN:
     u32Trigger = dispvidctrl_tclFsm_Hmi::eTrigger_HMI_BLACKIN_CONFIRM;
     break;

   case EN_HMI_CONFIRM_SYSTEMSTATUS__RVC_ON:
   case EN_HMI_CONFIRM_SYSTEMSTATUS__AVM_ON:
   case EN_HMI_CONFIRM_SYSTEMSTATUS__MVC_ON:
     u32Trigger = dispvidctrl_tclFsm_Hmi::eTrigger_HMI_ON_CONFIRM;
     //Resize video based on PSD
     if ( m_u8VideoResizedType != RVC_AVM_VIDEO_FULL_SCREEN )
     {
        m_u8VideoResizedType = RVC_AVM_VIDEO_FULL_SCREEN;
        if (m_poControl_Video) {
           m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__RESIZE_VIDEO, m_u8VideoResizedType);
        }
     }
     break;

   case EN_HMI_CONFIRM_SYSTEMSTATUS__RVC_ON_CAMERA_SETTINGS:
   case EN_HMI_CONFIRM_SYSTEMSTATUS__AVM_ON_CAMERA_SETTINGS:
   case EN_HMI_CONFIRM_SYSTEMSTATUS__MVC_ON_CAMERA_SETTINGS:
     u32Trigger = dispvidctrl_tclFsm_Hmi::eTrigger_HMI_ON_CAMERASETTINGS;
     break;

   case EN_HMI_CONFIRM_SYSTEMSTATUS__RVC__BLACK_OUT:
   case EN_HMI_CONFIRM_SYSTEMSTATUS__AVM__BLACK_OUT:
   case EN_HMI_CONFIRM_SYSTEMSTATUS__MVC__BLACK_OUT:
     u32Trigger = dispvidctrl_tclFsm_Hmi::eTrigger_HMI_BLACKOUT_CONFIRM;
     break;

   case EN_HMI_CONFIRM_SYSTEMSTATUS__BLACK_IN_CAM_SETTINGS:
     u32Trigger = dispvidctrl_tclFsm_Hmi::eTrigger_HMI_BLACKIN_CONFIRM;
     break;

   case EN_HMI_CONFIRM_SYSTEMSTATUS__BLACK_OUT_CAM_SETTINGS:
     u32Trigger = dispvidctrl_tclFsm_Hmi::eTrigger_HMI_BLACKOUT_CONFIRM;
     break;

   case EN_HMI_CONFIRM_SYSTEMSTATUS__RVC_VISUALISATION_ON_WITH_UPA:
   case EN_HMI_CONFIRM_SYSTEMSTATUS__AVM_VISUALISATION_ON_WITH_UPA:
     //Resize video based on PSD
     if ( m_u8VideoResizedType != RVC_AVM_VIDEO_WITH_UPA_VIDEO_SCREEN )
     {
        m_u8VideoResizedType = RVC_AVM_VIDEO_WITH_UPA_VIDEO_SCREEN;
        if (m_poControl_Video) {
           m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__RESIZE_VIDEO, m_u8VideoResizedType);
        }
     }
     break;

   case EN_HMI_CONFIRM_SYSTEMSTATUS__RVC_VISUALISATION_ON_WITH_UPA_HFP:
   case EN_HMI_CONFIRM_SYSTEMSTATUS__AVM_VISUALISATION_ON_WITH_UPA_HFP:
     //Resize video based on PSD
     if ( m_u8VideoResizedType != RVC_AVM_VIDEO_WITH_UPA_HFP_VIDEO_SCREEN )
     {
        m_u8VideoResizedType = RVC_AVM_VIDEO_WITH_UPA_HFP_VIDEO_SCREEN;
        if (m_poControl_Video) {
           m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__RESIZE_VIDEO, m_u8VideoResizedType);
        }
     }
     break;

   default:
     break;
   }

   DISPVIDCTRL_NULL_POINTER_CHECK(m_poFsm_Hmi);
   m_poFsm_Hmi->vNewTrigger(u32Trigger, TRUE);
}

/******************************************************************************/
/* FUNCTION     _vHandleRearTrunkStateChange                                  */
/******************************************************************************/
/**
*  \brief       trace information
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Hmi::_vHandleRearTrunkStateChange(tVoid)
{
   tBool bTailgateOpen = FALSE;
   tBool bShowWarningText = TRUE;
   tBool bBlackScreen = FALSE;
   tBool bSwitchOff = FALSE;
   
   // Rear trunk behaviour is only applicable on Rvc Mode
   // in all the other Avm, Mvc modes, warning text should be displayed when Tailgate is open
   if ( m_bRvcMode )
   {
      switch ( m_u8RearTrunkBehaviour )
      {
         case DISPVIDCTRL_AllianceRearTrunkBehave_SwitchOff:
         case DISPVIDCTRL_AllianceRearTrunkBehave_SwitchOff_ShowWarning:
            bSwitchOff = TRUE;
            bBlackScreen = FALSE;
            break;
         
         case DISPVIDCTRL_AllianceRearTrunkBehave_BlackScreen:
         case DISPVIDCTRL_AllianceRearTrunkBehave_BlackScreen_ShowWarning:
            bBlackScreen = TRUE;
            bSwitchOff = FALSE;
            break;
      
         case DISPVIDCTRL_AllianceRearTrunkBehave_ShowWarningText:
            bShowWarningText = TRUE;
            break;

         case DISPVIDCTRL_AllianceRearTrunkBehave_NoAction:
         default:
            bShowWarningText = FALSE;
            bBlackScreen = FALSE;
            bSwitchOff = FALSE;
            break;
      }
   }

   I_dispvidctrl_tclClientVcan* poClientVcan = dynamic_cast<I_dispvidctrl_tclClientVcan*>(_cpoMain->getHandler("I_dispvidctrl_tclClientVcan"));
   DISPVIDCTRL_NULL_POINTER_CHECK(poClientVcan);
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poControl_Rvc);
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poControl_Video);
   
   if ( TRUE == poClientVcan->bGet_TailgateStatus(bTailgateOpen) )
   {
      rvc_fi_tcl_e8_Tailgate_State oState;

      if ( bTailgateOpen ) 
      {
         oState.enType = rvc_fi_tcl_e8_Tailgate_State::FI_EN_OPEN;
         
         if ( bBlackScreen ) {
            // show black
            m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__BLACK_SCREEN_ON, EN_CONTROL_VIDEO__BLACKSCREENTYPE_FULL);
         } else if ( bSwitchOff ) {
            m_poControl_Rvc->vHandleControl(EN_CONTROL_RVC__TAILGATE_OPEN);
         } else {
            // do nothing
         }
      } 
      else 
      {
         oState.enType = rvc_fi_tcl_e8_Tailgate_State::FI_EN_CLOSED;
         
         if ( bBlackScreen ) {
            // hide black
            m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__SHOW_RVC);
         } else if ( bSwitchOff ) {
            m_poControl_Rvc->vHandleControl(EN_CONTROL_RVC__TAILGATE_CLOSE);
         } else {
            // do nothing
         }
      }
      
      DISPVIDCTRL_NULL_POINTER_CHECK(m_poRvcServiceIf);
      if ( TRUE == bShowWarningText ) {
          m_poRvcServiceIf->sendTailgate_StatusStatus(oState);
      }
   }
}

/******************************************************************************/
/* FUNCTION     _bIsGraphicsDisplayOn                                         */
/******************************************************************************/
/**
*  \brief       trace information
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Hmi::_bIsGraphicsDisplayOn(tVoid) const
{
   DISPVIDCTRL_NULL_POINTER_CHECK_VAL(m_poControl_Avm);
   tBool bIsGraphicsOn = TRUE;
   
   if ( m_bAvmMode )
   {
      bIsGraphicsOn = m_poControl_Avm->bIsAvmGraphicsOn();
   }
   
   return bIsGraphicsOn;
}

/******************************************************************************/
/* FUNCTION     vTraceInfo()                                                  */
/******************************************************************************/
/**
*  \brief       trace information
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Hmi::vTraceInfo()
{
   ETG_TRACE_FATAL(("   CA-HMI :: vTraceInfo() *********** CA-HMI states ******************************************"));
   ETG_TRACE_FATAL(("   CA-HMI :: m_bAvmMode ................................ %-40u", m_bAvmMode));
   ETG_TRACE_FATAL(("   CA-HMI :: m_bRvcMode ................................ %-40u", m_bRvcMode));
   ETG_TRACE_FATAL(("   CA-HMI :: m_bMvcMode ................................ %-40u", m_bMvcMode));
   ETG_TRACE_FATAL(("   CA-HMI :: m_bFapConfigured .......................... %-40u", m_bFapConfigured));
   ETG_TRACE_FATAL(("   CA-HMI :: m_bFapModeActive .......................... %-40u", m_bFapModeActive));
   ETG_TRACE_FATAL(("   CA-HMI :: Video Signal    : Check State ............. %-40u", m_bVideoSignalCheck));
   ETG_TRACE_FATAL(("   CA-HMI :: Video Signal    : State ................... %-40u", m_bVideoSignalState));
   ETG_TRACE_FATAL(("   CA-HMI :: Video Signal Test Result : ................ %-40u", m_u8VideoSignalStateTestResult));
   ETG_TRACE_FATAL(("   CA-HMI :: m_bHmiAppVehicleReady : ................... %-40u", m_bHmiAppVehicleReady));
   ETG_TRACE_FATAL(("   CA-HMI :: FSM Control_Hmi : Super State ............. %-40u", ETG_CENUM(tenFsmControl_Hmi_SuperState, m_enFsmControl_Hmi_SuperState)));
   ETG_TRACE_FATAL(("   CA-HMI :: *********************************************************************************"));
}
