/******************************************************************************/
/**
* \file    dispvidctrl_tclControl_Avm.cpp
* \ingroup
*
* \brief
*
* \remark  Copyright : (c) 2015 Robert Bosch GmbH, Hildesheim
* \remark  Author    : Michael Niemann
* \remark  Scope     : AIVI
*
* \todo
*/
/******************************************************************************/

/*******************************************************************************
                        Includes
*******************************************************************************/

#include "dispvidctrl_tclFsm_Avm.h"
#include "dispvidctrl_tclControl.h"
#include "dispvidctrl_tclControl_Avm.h"
#include "dispvidctrl_tclControl_Rvc.h"
#include "I_dispvidctrl_ClientAudioIf.h"
#include "Idispvidctrl_ClientVcan.h"
#include "I_dispvidctrl_ServiceRvcIf.h"
#include "dispvidctrl_tclPersistentData.h"
#include "dispvidctrl_tclControl_Video.h"
#include "dispvidctrl_tclFsm_Hmi.h"
#include "dispvidctrl_tclControl_Hmi.h"
#include "dispvidctrl_tclFsm_IpaSwitchOperation.h"
#include "I_dispvidctrl_InputHandler.h"
#include "I_dispvidctrl_tclAudioPlayer.h"

#ifndef VARIANT_S_FTR_ENABLE_AIVI_SCOPE1
#include "I_dispvidctrl_ClientSds_ttsIf.h"
#endif
#include "dispvidctrl_tclAudioRouting.h"
#include "dispvidctrl_datapool.h"


#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_APPLICATION
#include "trcGenProj/Header/dispvidctrl_tclControl_Avm.cpp.trc.h"
#endif

/*******************************************************************************
                        Defines
*******************************************************************************/
#define DISPVIDCTRL_DATA__HMI_LANGUAGE_NOT_RECEIVED (tU8) 0xFF

#define DISPVIDCTRL_DATA__AVM_LANGUAGE_ENGLISH_US            0  //for U.S.
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_SPANISH_EUR           1  //Spanish for other than Mexico (e.g. Europe)
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_FRENCH_EUR            2  //French for other than Canada (e.g. Europe)
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_DUTCH                 3
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_GERMAN                4
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_ITALIAN               5
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_PORTUGUESE            6
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_RUSSIAN               7
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_JAPANESE              8
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_CHINESE_TRADITIONAL   9  //Honkong, Taiwan, Japan
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_CHINESE_SIMPLIFIED   10  //VR_China    (Mandarin Chinesisch?)
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_KOREAN               11
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_ARABIC               12
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_ENGLISH_UK           13  //for U.K.
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_SPANISH_MEX          14  //for Mexico
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_FRENCH_CAN           15  //for Canada
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_PORTUGUESE_BRAZIL    16  //for Brazil
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_TURKISH              17
//#define DISPVIDCTRL_DATA__AVM_LANGUAGE_GREEK_LATIN          18
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_GREEK_GREEK_ALPHABET 19
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_POLISH               20
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_CZECH                21
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_SLOVAK               22
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_HUNGARIAN            23
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_DANISH               24
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_SWEDISH              25
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_FINNISH              26
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_NORWEGIAN            27
#define DISPVIDCTRL_DATA__AVM_LANGUAGE_THAI                 28
//#define DISPVIDCTRL_DATA__AVM_LANGUAGE_BLANK                29  // blank means, that AVM shall not show a warning text

#define DISPVIDCTRL_DATA__AVM_LANGUAGE_DEFAULT              DISPVIDCTRL_DATA__AVM_LANGUAGE_ENGLISH_UK
#define DISPVIDCTRL_FAP_KEY_CODE_INPUT                      49 // Offset used to Fap key code (Ipbutton id - Fap settings key)

static StLanguageMappingEntryAvm stLanguageMappingAvm[] =
{
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_ENGLISH_US,                    DISPVIDCTRL_DATA__AVM_LANGUAGE_ENGLISH_US,            },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_FRENCH_CAN,                    DISPVIDCTRL_DATA__AVM_LANGUAGE_FRENCH_CAN,            },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_SPANISH_LATIN_AMERICAN,        DISPVIDCTRL_DATA__AVM_LANGUAGE_SPANISH_MEX,           },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_ENGLISH_UK,                    DISPVIDCTRL_DATA__AVM_LANGUAGE_ENGLISH_UK,            },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_GERMAN,                        DISPVIDCTRL_DATA__AVM_LANGUAGE_GERMAN,                },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_FRENCH_EUR,                    DISPVIDCTRL_DATA__AVM_LANGUAGE_FRENCH_EUR,            },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_ITALIAN,                       DISPVIDCTRL_DATA__AVM_LANGUAGE_ITALIAN,               },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_DUTCH,                         DISPVIDCTRL_DATA__AVM_LANGUAGE_DUTCH,                 },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_SPANISH_EUR,                   DISPVIDCTRL_DATA__AVM_LANGUAGE_SPANISH_EUR,           },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_PORTUGUESE,                    DISPVIDCTRL_DATA__AVM_LANGUAGE_PORTUGUESE,            },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_RUSSIAN,                       DISPVIDCTRL_DATA__AVM_LANGUAGE_RUSSIAN,               },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_TURKISH,                       DISPVIDCTRL_DATA__AVM_LANGUAGE_TURKISH,               },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_DANISH,                        DISPVIDCTRL_DATA__AVM_LANGUAGE_DANISH,                },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_SWEDISH,                       DISPVIDCTRL_DATA__AVM_LANGUAGE_SWEDISH,               },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_FINNISH,                       DISPVIDCTRL_DATA__AVM_LANGUAGE_FINNISH,               },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_NORWEGIAN,                     DISPVIDCTRL_DATA__AVM_LANGUAGE_NORWEGIAN,             },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_GREEK,                         DISPVIDCTRL_DATA__AVM_LANGUAGE_GREEK_GREEK_ALPHABET,  },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_POLISH,                        DISPVIDCTRL_DATA__AVM_LANGUAGE_POLISH,                },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_CZECH,                         DISPVIDCTRL_DATA__AVM_LANGUAGE_CZECH,                 },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_SLOVAK,                        DISPVIDCTRL_DATA__AVM_LANGUAGE_SLOVAK,                },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_HUNGARIAN,                     DISPVIDCTRL_DATA__AVM_LANGUAGE_HUNGARIAN,             },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_ARABIC,                        DISPVIDCTRL_DATA__AVM_LANGUAGE_ARABIC,                },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_THAI,                          DISPVIDCTRL_DATA__AVM_LANGUAGE_THAI,                  },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_AUSTRALIAN,                    DISPVIDCTRL_DATA__AVM_LANGUAGE_DEFAULT,               },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_BRAZILIAN,                     DISPVIDCTRL_DATA__AVM_LANGUAGE_PORTUGUESE_BRAZIL,     },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_UKRAINIAN,                     DISPVIDCTRL_DATA__AVM_LANGUAGE_DEFAULT,               },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_KOREAN,                        DISPVIDCTRL_DATA__AVM_LANGUAGE_KOREAN,                },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_CHINESE_SIMPLIFIED,            DISPVIDCTRL_DATA__AVM_LANGUAGE_CHINESE_SIMPLIFIED,    },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_CHINESE_TRADITIONAL_CANTONESE, DISPVIDCTRL_DATA__AVM_LANGUAGE_CHINESE_TRADITIONAL,   },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_CHINESE_TRADITIONAL_TAIWANESE, DISPVIDCTRL_DATA__AVM_LANGUAGE_CHINESE_SIMPLIFIED,    },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_INDONESIAN,                    DISPVIDCTRL_DATA__AVM_LANGUAGE_DEFAULT,               },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_ROMANIAN,                      DISPVIDCTRL_DATA__AVM_LANGUAGE_DEFAULT,               },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_ALBANESE,                      DISPVIDCTRL_DATA__AVM_LANGUAGE_DEFAULT,               },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_BULGARIAN,                     DISPVIDCTRL_DATA__AVM_LANGUAGE_DEFAULT,               },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_CROATIAN,                      DISPVIDCTRL_DATA__AVM_LANGUAGE_DEFAULT,               },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_ESTONIAN,                      DISPVIDCTRL_DATA__AVM_LANGUAGE_DEFAULT,               },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_FLEMISH,                       DISPVIDCTRL_DATA__AVM_LANGUAGE_DEFAULT,               },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_HEBREW,                        DISPVIDCTRL_DATA__AVM_LANGUAGE_DEFAULT,               },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_HINDI,                         DISPVIDCTRL_DATA__AVM_LANGUAGE_DEFAULT,               },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_JAPANESE,                      DISPVIDCTRL_DATA__AVM_LANGUAGE_JAPANESE,              },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_LATVIAN,                       DISPVIDCTRL_DATA__AVM_LANGUAGE_DEFAULT,               },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_LITHUANIAN,                    DISPVIDCTRL_DATA__AVM_LANGUAGE_DEFAULT,               },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_MALAY,                         DISPVIDCTRL_DATA__AVM_LANGUAGE_DEFAULT,               },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_PERSIAN_FARSI_IRANIAN,         DISPVIDCTRL_DATA__AVM_LANGUAGE_DEFAULT,               },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_PHILIPPINES,                   DISPVIDCTRL_DATA__AVM_LANGUAGE_DEFAULT,               },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_SERBIAN,                       DISPVIDCTRL_DATA__AVM_LANGUAGE_DEFAULT,               },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_SLOVENIAN,                     DISPVIDCTRL_DATA__AVM_LANGUAGE_DEFAULT,               },
   {DISPVIDCTRL_DATA__HMI_LANGUAGE_VIETNAMESE,                    DISPVIDCTRL_DATA__AVM_LANGUAGE_DEFAULT,               }
};

static StCSMFapInputKeyCode stCSMInputFAPKey[] =
{
        {FAP_CSMINPUT_SETTING_TYPE__VOICE_GUIDANCE,                (FAP_SETTING_TYPE__FAP_SETTING01 + DISPVIDCTRL_FAP_KEY_CODE_INPUT),            },
		{FAP_CSMINPUT_SETTING_TYPE__AUTO_SELECTION,                (FAP_SETTING_TYPE__FAP_SETTING03 + DISPVIDCTRL_FAP_KEY_CODE_INPUT),            },
		{FAP_CSMINPUT_SETTING_TYPE__BAYPARKING,                    (FAP_SETTING_TYPE__FAP_SETTING05 + DISPVIDCTRL_FAP_KEY_CODE_INPUT),            },
		{FAP_CSMINPUT_SETTING_TYPE__SLOTPARKING,                   (FAP_SETTING_TYPE__FAP_SETTING06 + DISPVIDCTRL_FAP_KEY_CODE_INPUT),            },
		{FAP_CSMINPUT_SETTING_TYPE__FORWARDPARKING,                (FAP_SETTING_TYPE__FAP_SETTING07 + DISPVIDCTRL_FAP_KEY_CODE_INPUT),            },
		{FAP_CSMINPUT_SETTING_TYPE__ANGLE_BACKWARD_PARKING,        (FAP_SETTING_TYPE__FAP_SETTING08 + DISPVIDCTRL_FAP_KEY_CODE_INPUT),            },
		{FAP_CSMINPUT_SETTING_TYPE__ANGLE_FORWARD_PARKING,         (FAP_SETTING_TYPE__FAP_SETTING09 + DISPVIDCTRL_FAP_KEY_CODE_INPUT),            },
		{FAP_CSMINPUT_SETTING_TYPE__PREVIOUS_PARKING_TYPE,         (FAP_SETTING_TYPE__FAP_SETTING04 + DISPVIDCTRL_FAP_KEY_CODE_INPUT),            },
};


#define DISPVIDCTRL_DATA_LANGUAGE_MAPPING_AVM_TABLE_SIZE  (sizeof(stLanguageMappingAvm)/sizeof(StLanguageMappingEntryAvm))
#define DISPVIDCTRL_FAP_KEY_CODE_TABLE_SIZE    (sizeof(stCSMInputFAPKey)/sizeof(StCSMFapInputKeyCode))

#define DISPVIDCTRL_AVM_REINIT_TIME_IN_MS                        2100
#define DISPVIDCTRL_AVM_RENAULT_ERROR_CONFIRMATION_TIME_IN_MS   15000
#define DISPVIDCTRL_AVM__NISSAN_ERROR_CONFIRMATION_TIME_IN_MS    5000

/******************************************************************************/
/* FUNCTION     dispvidctrl_tclControl_Avm()                                  */
/******************************************************************************/
/**
*  \brief       default constructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclControl_Avm::dispvidctrl_tclControl_Avm()
: dispvidctrl_tclBaseIf(OSAL_NULL)
, m_poClientAudioIf(OSAL_NULL)
, m_poClientVcan(OSAL_NULL)
, m_poControlRvc(OSAL_NULL)
, m_poServiceRvcIf(OSAL_NULL)
, m_poAudioRouting(OSAL_NULL)
, m_enFsmControl_Avm_SuperState(DISPVIDCTRL_CONTROL_AVM__DISABLED)
, m_u8IPAStatus(DISPVIDCTRL_CONTROL_IPA__NOTAVAILABLE)
, m_bAvmMode(FALSE)
, m_bAvmTypeIpa(FALSE)
, m_bAvmTypeRenault(FALSE)
, m_bAvmTypeFap(FALSE)
, m_bDaVariant(FALSE)
, m_bFapModeActive(FALSE)
, m_bFapPurposeSwitchAvailable(TRUE)
, m_bAvmRGraphicsOn(FALSE)
, m_bAccState(TRUE)
, m_bIgnition(FALSE)
, m_bIgnitionStateValid(FALSE)
, m_bCanMsgTimeOutState(FALSE)
, m_bReverseGearState(FALSE)
, m_hTimerHandle_AvmErrorConfirm(OSAL_C_INVALID_HANDLE)
, m_poFsm_IpaSwitchOperation(OSAL_NULL)
, m_u32HmiLanguage(DISPVIDCTRL_DATA__HMI_LANGUAGE_NOT_RECEIVED)
, m_u8AvmLanguage(DISPVIDCTRL_DATA__HMI_LANGUAGE_NOT_RECEIVED)
#ifdef VARIANT_S_FTR_ENABLE_SMART
, m_u8VCANDiversity(DISPVIDCTRL_AllianceVCANSlot_Type_C1A)
#else
, m_u8VCANDiversity(DISPVIDCTRL_AllianceVCANSlot_Type_0)
#endif
, m_bAudioBeepsMuted(FALSE)
, m_bAudioChannelAvailable(FALSE)
, m_u8FAPKeyCode(0)
, m_bAvmFapSourceAvailable(FALSE)
, m_u8IpaErrorCode(DISPVIDCTRL_CONTROL_IPA_NO_ERROR)
, m_bTimerErrorConfirmRunning(FALSE)
, m_bIpaErrorHandlerON(TRUE)
{
   m_poFsm_Avm = OSAL_NULL;
   tIPA_Voice_Texts.clear();
}

/******************************************************************************/
/* FUNCTION     dispvidctrl_tclControl_Avm()                                  */
/******************************************************************************/
/**
*  \brief       constructor
*
*  \param       pointer to main application
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclControl_Avm::dispvidctrl_tclControl_Avm(dispvidctrl_tclAppMain* poMainAppl)
: dispvidctrl_tclBaseIf(poMainAppl)
, m_poClientAudioIf(OSAL_NULL)
, m_poClientVcan(OSAL_NULL)
, m_poControlRvc(OSAL_NULL)
, m_poServiceRvcIf(OSAL_NULL)
, m_poAudioRouting(OSAL_NULL)
, m_enFsmControl_Avm_SuperState(DISPVIDCTRL_CONTROL_AVM__DISABLED)
, m_u8IPAStatus(DISPVIDCTRL_CONTROL_IPA__NOTAVAILABLE)
, m_bAvmMode(FALSE)
, m_bAvmTypeIpa(FALSE)
, m_bAvmTypeRenault(FALSE)
, m_bAvmTypeFap(FALSE)
, m_bDaVariant(FALSE)
, m_bFapModeActive(FALSE)
, m_bFapPurposeSwitchAvailable(TRUE)
, m_bAvmRGraphicsOn(FALSE)
, m_bAccState(TRUE)
, m_bIgnition(FALSE)
, m_bIgnitionStateValid(FALSE)
, m_bCanMsgTimeOutState(FALSE)
, m_bReverseGearState(FALSE)
, m_hTimerHandle_AvmErrorConfirm(OSAL_C_INVALID_HANDLE)
, m_u32HmiLanguage(DISPVIDCTRL_DATA__HMI_LANGUAGE_NOT_RECEIVED)
, m_u8AvmLanguage(DISPVIDCTRL_DATA__HMI_LANGUAGE_NOT_RECEIVED)
#ifdef VARIANT_S_FTR_ENABLE_SMART
, m_u8VCANDiversity(DISPVIDCTRL_AllianceVCANSlot_Type_C1A)
#else
, m_u8VCANDiversity(DISPVIDCTRL_AllianceVCANSlot_Type_0)
#endif
, m_bAudioBeepsMuted(FALSE)
, m_bAudioChannelAvailable(FALSE)
, m_u8FAPKeyCode(0)
, m_bAvmFapSourceAvailable(FALSE)
, m_u8IpaErrorCode(DISPVIDCTRL_CONTROL_IPA_NO_ERROR)
, m_bTimerErrorConfirmRunning(FALSE)
, m_bIpaErrorHandlerON(TRUE)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Avm() entered."));

   tIPA_Voice_Texts.clear();

   // create new FSM Avm
   m_poFsm_Avm = dispvidctrl_tclFsm_Avm::tclCreateInstance(this);
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poFsm_Avm);

   // create new FSM Ipa Switch Operation
   m_poFsm_IpaSwitchOperation = dispvidctrl_tclFsm_IpaSwitchOperation::tclCreateInstance(this);
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poFsm_IpaSwitchOperation);
   
   if ( OSAL_s32TimerCreate( (OSAL_tpfCallback)_pfCallbackTimer_AvmErrorConfirm, (tPVoid)this, &m_hTimerHandle_AvmErrorConfirm) != OSAL_OK )
   {
      ETG_TRACE_FATAL(("ERROR - Could not create AvmIpaMalfunction timer!"));
      m_hTimerHandle_AvmErrorConfirm = OSAL_C_INVALID_HANDLE;
   }
}

/******************************************************************************/
/* FUNCTION     ~dispvidctrl_tclControl_Avm                                   */
/******************************************************************************/
/**
*  \brief       destructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclControl_Avm::~dispvidctrl_tclControl_Avm(tVoid)
{
   ETG_TRACE_USR4(("~dispvidctrl_tclControl_Avm() entered."));

   m_poClientAudioIf = OSAL_NULL;
   m_poClientVcan = OSAL_NULL;
   m_poControlRvc = OSAL_NULL;
   m_poServiceRvcIf = (OSAL_NULL);

   if ( m_poFsm_Avm )
   {
       delete m_poFsm_Avm;
       m_poFsm_Avm = OSAL_NULL;
   }

   if ( m_poFsm_IpaSwitchOperation )
   {
       delete m_poFsm_IpaSwitchOperation;
       m_poFsm_IpaSwitchOperation = OSAL_NULL;
   }

   //stop and delete all timer
   if (OSAL_C_INVALID_HANDLE != m_hTimerHandle_AvmErrorConfirm)
   {
      (void) OSAL_s32TimerSetTime(m_hTimerHandle_AvmErrorConfirm, 0, 0);
      (void) OSAL_s32TimerDelete(m_hTimerHandle_AvmErrorConfirm);
   }

   m_poAudioRouting = OSAL_NULL;
}

/******************************************************************************/
/* FUNCTION     vGetConfiguration                                             */
/******************************************************************************/
/**
*  \brief       Function to get all configuration values needed by this class.
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::vGetConfiguration(const TConfiguration* pStConfigurationValues)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Avm::vGetConfiguration() entered."));
   m_bAvmMode = pStConfigurationValues->bAvmMode;
   m_bAvmTypeIpa = pStConfigurationValues->bAvmType_Ipa;
   m_u8VCANDiversity = pStConfigurationValues->u8VCANDiversity;
   m_bAvmTypeRenault = pStConfigurationValues->bAvmType_Renault;
   m_bAvmTypeFap = pStConfigurationValues->bAvmType_Fap;

   if ( 0x02 == pStConfigurationValues->u8DeviceVariantType ) {
      m_bDaVariant = TRUE;
   } else {
      m_bDaVariant = FALSE;
   }
}

/******************************************************************************/
/* FUNCTION     vGetReferences                                                */
/******************************************************************************/
/**
*  \brief       Function to get all reference needed by this class.
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::vGetReferences(tVoid)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Video::vGetReferences() entered."));

   m_poClientAudioIf = dynamic_cast<Idispvidctrl_tclClientAudioIf*>(_cpoMain->getHandler("Idispvidctrl_tclClientAudioIf"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poClientAudioIf);

   m_poClientVcan = dynamic_cast<I_dispvidctrl_tclClientVcan*>(_cpoMain->getHandler("I_dispvidctrl_tclClientVcan"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poClientVcan);

   m_poControlRvc = dynamic_cast<dispvidctrl_tclControl_Rvc*>(_cpoMain->getHandler("dispvidctrl_tclControl_Rvc"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poControlRvc);

   m_poServiceRvcIf = dynamic_cast<Idispvidctrl_tclServiceRvcIf*>(_cpoMain->getHandler("Idispvidctrl_tclServiceRvcIf"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poServiceRvcIf);

   m_poAudioRouting = dynamic_cast<dispvidctrl_tclAudio_Routing*>(_cpoMain->getHandler("dispvidctrl_tclAudio_Routing"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poAudioRouting);
}


/******************************************************************************/
/* FUNCTION     vStartCommunication                                           */
/******************************************************************************/
/**
*  \brief       Function to start all dynamic objects e.g. threads, ...
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::vStartCommunication(tVoid)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Avm::vStartCommunication() entered."));

   // Update all the Fap Settings current state
   if (m_bAvmTypeFap) {
      _vUpdateFapAvmSettings();
   }
}


/******************************************************************************/
/* FUNCTION     vHandleMessage(TMsg* pMsg)                                    */
/******************************************************************************/
/**
*  \brief       Handle worker events.
*
*  \param       message pointer
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::vHandleMessage(dispvidctrl_tclBaseIf::TMsg* pMsg)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Avm::vHandleMessage() entered %u -> data: %d.", ETG_CENUM(dispvidctrl_tclBaseIf::ECmdTypes , (tU32)pMsg->eCmd), pMsg->u.u32Data));
   //ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclControl_Avm::vHandleMessage() entered %u -> data: %d.", OSAL_ClockGetElapsedTime(), ETG_CENUM(dispvidctrl_tclBaseIf::ECmdTypes , (tU32)pMsg->eCmd), pMsg->u.u32Data));

   switch (pMsg->eCmd)
   {
      case dispvidctrl_tclBaseIf::eControlAvm:
         vHandleControl((tenControl_Avm) pMsg->u.u32Data);
         break;
      case dispvidctrl_tclBaseIf::eFsmAvm:
         vHandleControl(EN_CONTROL_AVM__FSM_AVM_INPUT_EVENT, pMsg->u.u32Data);
         break;
      case dispvidctrl_tclBaseIf::eReverseOff:
         m_bReverseGearState = FALSE;
         vHandleControl(EN_CONTROL_AVM__REVERSE_OFF);
         break;
      case dispvidctrl_tclBaseIf::eReverseOnDebounced:
         m_bReverseGearState = TRUE;
         vHandleControl(EN_CONTROL_AVM__REVERSE_ON_DEBOUNCED);
         break;
      case dispvidctrl_tclBaseIf::eControlModeEnabled:
         if (m_bAvmMode == TRUE) {
            vHandleControl(EN_CONTROL_AVM__ENABLE, pMsg->u.u32Data);
         }
         break;
      case dispvidctrl_tclBaseIf::eControlModeDisabled:
         if (m_bAvmMode == TRUE) {
            vHandleControl(EN_CONTROL_AVM__DISABLE, pMsg->u.u32Data);
         }
         break;
      case dispvidctrl_tclBaseIf::eAccChanged:
         if ( TRUE == (tBool) pMsg->u.u32Data ) {
            m_bAccState = TRUE;
         } else {
            m_bAccState = FALSE;
         }
         break;
      case dispvidctrl_tclBaseIf::eFsmIpaSwitchOperation:
         vHandleControl(EN_CONTROL_AVM__FSM_IPASWITCHOPERATION_INPUT_EVENT, pMsg->u.u32Data);
         break;
      case dispvidctrl_tclBaseIf::eFapModeActive:
         vHandleControl(EN_CONTROL_AVM__FAP_MODE_CHANGE, (tU32) pMsg->u.u8Data);
         break;
      default:
         break;
   }
}

/******************************************************************************/
/* FUNCTION     vHandleTraceMessage                                           */
/******************************************************************************/
/**
*  \brief       handle TTFis commands
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::vHandleTraceMessage(const tUChar* puchData)
{
   DISPVIDCTRL_NULL_POINTER_CHECK(puchData);

   tU32 u32MsgCode = ((puchData[1]<<8) | puchData[2]);
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Avm::vHandleTraceMessage(): trace command %d", u32MsgCode ));

   switch (u32MsgCode)
   {
      case DISPVIDCTRL_DISPVIDCTRL_STIMULATE_FSM_AVM:
         {
            // Set once eTrigger for FSM_Avm
            tU32 param1 = ((tU32)puchData[3] << 24) | ((tU32)puchData[4] << 16) | ((tU32)puchData[5] <<  8) | (tU32)puchData[6];

            vHandleControl(EN_CONTROL_AVM__FSM_AVM_INPUT_EVENT, param1);
         }
         break;
      case DISPVIDCTRL_DISPVIDCTRL_STIMULATE_CONTROL_AVM:
         {
            tU8 param1 = puchData[3];
            tU32 param2 = ((tU32)puchData[4] << 24) | ((tU32)puchData[5] << 16) | ((tU32)puchData[6] <<  8) | (tU32)puchData[7];

            vHandleControl((tenControl_Avm) param1, param2);
         }
         break;
      case DISPVIDCTRL_DISPVIDCTRL_STIMULATE_FSMOUT_AVM:
         {
            tU8 param1 = puchData[3];
            tU32 param2 = ((tU32)puchData[4] << 24) | ((tU32)puchData[5] << 16) | ((tU32)puchData[6] <<  8) | (tU32)puchData[7];

            vHandleFsmOutEvent_Avm((tenFsmOutEvent_Avm) param1, param2);
        }
        break;
      default:
         break;
   }
}


/******************************************************************************/
/* FUNCTION     vHandleControl()                                              */
/******************************************************************************/
/**
*  \brief       trace information
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::vHandleControl(tenControl_Avm enControl, tU32 u32Value)
{
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32Value );

   ETG_TRACE_USR4(("   CA-AVM :: ............vHandleControl(): %d     ", ETG_CENUM(tenControl_Avm, enControl)));
   //ETG_TRACE_FATAL(("[%d ms]    CA-AVM :: ............vHandleControl(): %d     ", OSAL_ClockGetElapsedTime(), ETG_CENUM(tenControl_Avm, enControl)));

   switch (enControl)
   {
      case EN_CONTROL_AVM__DISABLE:
      case EN_CONTROL_AVM__ENABLE:
         _vOnEvent_FsmControl_Avm(enControl, u32Value);
         break;
      //-------------------------------------------------------------------- direct event(s) to FSM, Timeouts
      case EN_CONTROL_AVM__FSM_AVM_INPUT_EVENT:
         if (m_poFsm_Avm) {m_poFsm_Avm->vNewTrigger(u32Value, TRUE);}
         break;

      case EN_CONTROL_AVM__TIMEOUT_VIEW_CHANGE_REQUEST_WAIT:
         _vOnEvent_FsmControl_Avm(enControl, u32Value);
         break;
      //-------------------------------------------------------------------------------------------
      case EN_CONTROL_AVM__FSM_IPASWITCHOPERATION_INPUT_EVENT:
         if (m_bAvmTypeIpa && m_poFsm_IpaSwitchOperation) {m_poFsm_IpaSwitchOperation->vNewTrigger(u32Value, TRUE);}
         break;

      //-------------------------------------------------------------------------------------------
      case EN_CONTROL_AVM__IGNITION:
         {
            if ( (tBool) u32Value == TRUE ) { // Ignition Active
               m_bIgnition = TRUE;
               _vOnEvent_FsmControl_Avm(EN_CONTROL_AVM__IGNITION_ON);
            }
            else { // Ignition Off
               m_bIgnition = FALSE;
               _vOnEvent_FsmControl_Avm(EN_CONTROL_AVM__IGNITION_OFF);

               if ( m_bAccState == FALSE ) { // If both Ignition and Acc are Off, Switch Off Settings
                  tChar strIfName[] = "dispvidctrl_tclControl_Avm";
                  // post loop-back message to AVM FSM, u32Value will be eTrigger... value
                  dispvidctrl_tclAppMain::theServer()->vPostMsgControlAvm(strIfName, EN_CONTROL_AVM__SERVICE_REQUEST_VISUALISATION_OFF);
               }
            }
            m_bIgnitionStateValid = TRUE;
         }
         break;
      case EN_CONTROL_AVM__UPDATE_LANGUAGE:
         _vOnEvent_FsmControl_Avm(enControl, u32Value);
         break;

      // --------------------------------------------------------------------
      case EN_CONTROL_AVM__SWITCH_PRESS_MOD:
         if (m_poClientVcan) {
            if ( DISPVIDCTRL_AllianceVCANSlot_Type_C1A != m_u8VCANDiversity ) {
               tBool bReinit = FALSE;
               m_poClientVcan->vWriteUPA_MOD_Switch_Press(bReinit);
            } else {
			   // todo
               //m_poClientVcan->vWriteAVM_MOD_Activation_Request(u8Request);
            }
         }
         break;
      case EN_CONTROL_AVM__SWITCH_PRESS_LDW:
         if (m_poClientVcan) {
            //tBool bReinit = FALSE;
            //m_poClientVcan->vWriteLDW_PRESS_5FE(bReinit);
         }
         break;
      case EN_CONTROL_AVM__SWITCH_PRESS_BSW:
         if (m_poClientVcan) {
            //tBool bReinit = FALSE;
            //m_poClientVcan->vWriteBSW_PRESS_5FE(bReinit);
         }
         break;

      case EN_CONTROL_AVM__PURPOSE_SWITCH:
         // Are we in AVM FAP Mode ?
         if (m_bAvmTypeFap == TRUE) {
            // if reverse is On dont care about the Purpose switch info from FAP
            if (FALSE == m_bReverseGearState) {
               if (TRUE == m_bFapPurposeSwitchAvailable) {
                  _vOnEvent_FsmControl_Avm(EN_CONTROL_AVM__PURPOSE_SWITCH);
               }
            }
         } else { // For AVM (without FAP) we should always forward it to AVM
            _vOnEvent_FsmControl_Avm(EN_CONTROL_AVM__PURPOSE_SWITCH);
         }
         break;

      case EN_CONTROL_AVM__SERVICE_REQUEST_VISUALISATION_OFF:
         _vOnEvent_FsmControl_Avm(EN_CONTROL_AVM__CAMERA_SETTINGS_OFF);
         break;

      case EN_CONTROL_AVM__SERVICE_REQUEST_VISUALISATION_ON:
         // do this when View CHange not ON already
         _vOnEvent_FsmControl_Avm(EN_CONTROL_AVM__CAMERA_SETTINGS_ON);
         // else just update Fsm HMI with Settings request
         break;

      case EN_CONTROL_AVM__REVERSE_OFF:
      case EN_CONTROL_AVM__REVERSE_ON_DEBOUNCED:
      case EN_CONTROL_AVM__AFFORDABLE_ITS:
      case EN_CONTROL_AVM__CAMERA_BUTTON_PRESS:
      case EN_CONTROL_AVM__CAMERA_BUTTON_RELEASE:
      case EN_CONTROL_AVM__CAMERA_BUTTON_LONG_PRESS:
      case EN_CONTROL_AVM__CAMERA_SCREEN_OFF:
      case EN_CONTROL_AVM__CAMERA_SCREEN_ON:
      case EN_CONTROL_AVM__CAN_MSG_TIMEOUT_CONFIRMED:
      case EN_CONTROL_AVM__INIT_CHANGE:
      case EN_CONTROL_AVM__INIT_DONE:
      case EN_CONTROL_AVM__INIT_ACTIVE:
      case EN_CONTROL_AVM__VERIFY_AFTER_VIEW_STATUS_STOPPED:
      case EN_CONTROL_AVM__VIEW_CHANGE_REQUEST_CHANGE:
      case EN_CONTROL_AVM__IPA_ACTIVATION_CHANGE:
      case EN_CONTROL_AVM__ERROR_CONFIRM_TIMEOUT:
         _vOnEvent_FsmControl_Avm(enControl, u32Value);
         break;

      case EN_CONTROL_AVM__CAN_MSG_TIMEOUT_TRUE:
         if ( m_bIpaErrorHandlerON && m_bAvmTypeIpa )
         {
            m_u8IpaErrorCode |= DISPVIDCTRL_CONTROL_IPA_CAN_TIMEOUT;
            _vOnEvent_FsmControl_Avm(EN_CONTROL_AVM__ERROR_ACTIVE);
         } 
         else
         {
            _vOnEvent_FsmControl_Avm(enControl, u32Value);
         }
         break;
     case EN_CONTROL_AVM__CAN_MSG_TIMEOUT_FALSE:
         if ( m_bIpaErrorHandlerON && m_bAvmTypeIpa )
         {
            m_u8IpaErrorCode &= ~DISPVIDCTRL_CONTROL_IPA_CAN_TIMEOUT;
            if ( DISPVIDCTRL_CONTROL_IPA_NO_ERROR == m_u8IpaErrorCode ) {
               _vOnEvent_FsmControl_Avm(EN_CONTROL_AVM__ERROR_DEACTIVE);
            }
         }
         else
         {
            _vOnEvent_FsmControl_Avm(enControl, u32Value);
         }
         break;

   case EN_CONTROL_AVM__VIDEO_SIGNAL_FAILURE:
        if ( m_bIpaErrorHandlerON && m_bAvmTypeIpa ) 
        {
           m_u8IpaErrorCode |= DISPVIDCTRL_CONTROL_IPA_VIDEO_SIGNAL;
           _vOnEvent_FsmControl_Avm(EN_CONTROL_AVM__ERROR_ACTIVE);
        }
        else
        {
           _vOnEvent_FsmControl_Avm(enControl, u32Value);
        }
        break;

   case EN_CONTROL_AVM__VIDEO_SIGNAL_RESTORED:
        if ( m_bIpaErrorHandlerON && m_bAvmTypeIpa ) 
        {
           m_u8IpaErrorCode &= ~DISPVIDCTRL_CONTROL_IPA_VIDEO_SIGNAL;
           if ( DISPVIDCTRL_CONTROL_IPA_NO_ERROR == m_u8IpaErrorCode ) {
              _vOnEvent_FsmControl_Avm(EN_CONTROL_AVM__ERROR_DEACTIVE);
           }
        } 
        else 
        {
           _vOnEvent_FsmControl_Avm(enControl, u32Value);
        }
        break;

   case EN_CONTROL_AVM__IPA_OPERATION_MALFUNCTION:
      if ( m_bIpaErrorHandlerON && m_bAvmTypeIpa ) 
      {
         m_u8IpaErrorCode |= DISPVIDCTRL_CONTROL_IPA_MALFUNCTION;
         _vOnEvent_FsmControl_Avm(EN_CONTROL_AVM__ERROR_ACTIVE);
      } 
      else 
      {
         _vOnEvent_FsmControl_Avm(enControl, u32Value);
      }
      break;
   case EN_CONTROL_AVM__IPA_OPERATION_NORMAL:
      if ( m_bIpaErrorHandlerON && m_bAvmTypeIpa ) 
      {
         m_u8IpaErrorCode &= ~DISPVIDCTRL_CONTROL_IPA_MALFUNCTION;
         if ( DISPVIDCTRL_CONTROL_IPA_NO_ERROR == m_u8IpaErrorCode ) {
            _vOnEvent_FsmControl_Avm(EN_CONTROL_AVM__ERROR_DEACTIVE);
         }
      } 
      else 
      {
         _vOnEvent_FsmControl_Avm(enControl, u32Value);
      }
      break;

      case EN_CONTROL_AVM__NORMAL_BEEP:
      case EN_CONTROL_AVM__ERROR_BEEP:
      case EN_CONTROL_AVM__AOD_BEEP_1:
         // No Audio Beeps should be heard for Entry Avm Renault
         if ( FALSE == m_bAvmTypeRenault ) {
            _vOnEvent_FsmControl_Avm(enControl, u32Value);
         }
         break;

    // IPA related input events
    case EN_CONTROL_AVM__IPA_CONFIGURATION_CHANGE:
        if ( m_bAvmTypeIpa == TRUE ) {
            _vUpdate_IPAConfiguration();
        }
        break;

    case EN_CONTROL_AVM__IPA_REQUEST_NORMAL:
    case EN_CONTROL_AVM__IPA_REQUEST_IPA:
        if ( m_bAvmTypeIpa == TRUE ) {
            _vUpdate_IPAStatus();
            _vUpdate_IPAConfiguration();
        }
        break;

    case EN_CONTROL_AVM__IPA_SWITCHES_CHANGE:
        if ( m_bAvmTypeIpa == TRUE ) {
            _vUpdate_IPASwitches();
        }
        break;

    case EN_CONTROL_AVM__IPA_REQUEST_CHANGE:
        if ( m_bAvmTypeIpa == TRUE ) {
            _vUpdate_IPARequests();
        }
        break;

    case EN_CONTROL_AVM__IPA_DIRECTION_CHANGE:
        if ( m_bAvmTypeIpa == TRUE ) {
            _vUpdate_IPADirectionIndicator();
        }
        break;

    case EN_CONTROL_AVM__IPA_BUTTON_CHANGE:
        if ( m_bAvmTypeIpa == TRUE ) {
            _vHandleIPAButtonChange(u32Value);
        }
        break;

    case EN_CONTROL_AVM__IPA_VOICE_OUTPUT_REQUEST:
        if ( m_bAvmTypeIpa == TRUE ) {
            // Voice request to TTS ??
            _vHandleIPA_VoiceOutputRequest(u32Value);
        }
        else if (m_bAvmTypeFap && m_poAudioRouting)
        {
           if (u32Value == 0 || u32Value == 15)
           {
              // request for audio channel
              m_poAudioRouting->bRequestAudioRoute(ARL_SRC_IPA_TTS, ARL_EN_ISRC_ACT_OFF);
           }
           else if (u32Value < 14)
           {
              // request for audio channel
              m_poAudioRouting->bRequestAudioRoute(ARL_SRC_IPA_TTS, ARL_EN_ISRC_ACT_ON);
           }
           else
           {}
        }
        else
        {}
        break;

    case EN_CONTROL_AVM__IPA_SOUND_OUTPUT_REQUEST:
        if ( m_bAvmTypeIpa == TRUE ) {
            _vHandleIPA_AudioBeepRequest(u32Value);
        }
        break;

    case EN_CONTROL_AVM__IPA_VOICE_TEXTS_UPDATE:
        if ( m_bAvmTypeIpa == TRUE ) {
           _vHandleIpa_Voice_Request_Texts();
        }
        break;
        
    case EN_CONTROL_AVM__ERROR_HANDLER_ON:
        m_bIpaErrorHandlerON = TRUE;
        break;

    case EN_CONTROL_AVM__ERROR_HANDLER_OFF:
        m_bIpaErrorHandlerON = FALSE;
        m_u8IpaErrorCode = DISPVIDCTRL_CONTROL_IPA_NO_ERROR;
        _vOnEvent_FsmControl_Avm(EN_CONTROL_AVM__ERROR_DEACTIVE);
        break;

    case EN_CONTROL_AVM__CURRENT_VIEW_STATUS:
        _vOnEvent_FsmControl_Avm(enControl, u32Value);
        _vUpdate_AVM_CurrentViewStatus(u32Value);
        break;

    case EN_CONTROL_AVM__VIEW_EXIT_ALLOWED_STATUS:
        _vUpdate_AVM_ExitViewStatus(u32Value);
        break;

    case EN_CONTROL_AVM__VIEW_CHANGE_REQUEST_FROM_HMI:
        if (m_poClientVcan) {
           if (DISPVIDCTRL_AllianceVCANSlot_Type_C1A == m_u8VCANDiversity && m_bAvmTypeRenault) {
              m_poClientVcan->vWriteAVM_View_Change_Request((tU8)u32Value);

              // todo: MPU Settings
           }
        }
        break;

    case EN_CONTROL_AVM__AVM_BEEPS_STATUS_CHANGE:
        m_bAudioBeepsMuted = (tBool) u32Value;
        break;

    case EN_CONTROL_AVM__FAP_MODE_CHANGE:
        // update registered clients
        if (m_bAvmTypeFap == TRUE)
        {
           if ( u32Value == 1 || u32Value == 2 ) {
              m_bFapModeActive = TRUE;
           } else {
              m_bFapModeActive = FALSE;
           }

           _vOnEvent_FsmControl_Avm(EN_CONTROL_AVM__IPA_ACTIVATION_CHANGE, m_bFapModeActive);
           //_vOnEvent_FsmControl_Avm(EN_CONTROL_AVM__VIEW_CHANGE_REQUEST_CHANGE, m_bFapModeActive);
            
           if ( m_poServiceRvcIf ) {
              m_poServiceRvcIf->sendFAPModeStatus(m_bFapModeActive);
           }
        }
        break;

     case EN_CONTROL_AVM__FAP_PURPOSE_SWITCH_AVAILABLE:
        if (m_bAvmTypeFap == TRUE) {
           if (u32Value == 0) {
              m_bFapPurposeSwitchAvailable = FALSE;
           } else {
              m_bFapPurposeSwitchAvailable = TRUE;
           }
        }
        break;

     case EN_CONTROL_AVM__FAP_VOICE_PATTERN_REQUEST:
        if (m_bAvmTypeFap && m_poAudioRouting)
        {
           if (u32Value == 0)
           {
              // request for audio channel
              m_poAudioRouting->bRequestAudioRoute(ARL_SRC_IPA_TTS, ARL_EN_ISRC_ACT_OFF);
           }
           else if (u32Value < 64)
           {
              // request for audio channel
              m_poAudioRouting->bRequestAudioRoute(ARL_SRC_IPA_TTS, ARL_EN_ISRC_ACT_ON);
           }
           else
           {}
        }
        break;

     case EN_CONTROL_AVM__FAP_SOUND_PATTERN_REQUEST:
        if (m_bAvmTypeFap && m_poAudioRouting)
        {
           if (u32Value == 0)
           {
              // request for audio channel OFF
              m_poAudioRouting->bRequestAudioRoute(ARL_SRC_IPA_TTS, ARL_EN_ISRC_ACT_OFF);
           }
           else if (u32Value < 16) // Valid Sound pattern request
           {
              // request for audio channel
              m_poAudioRouting->bRequestAudioRoute(ARL_SRC_IPA_TTS, ARL_EN_ISRC_ACT_ON);
           }
           else
           {}
        }
        break;


     case EN_CONTROL_AVM__FAP_CAMERA_SETTING_AVAILABLE:
        if (m_bAvmTypeFap == TRUE) {
           _vUpdateFapAvmSettings(FAP_SETTING_TYPE__FAP_SETTING_AVAILABLE, FAP_SETTING_TYPE__AVM_SETTING_AVAILABLE);
        }
        break;

     case EN_CONTROL_AVM__FAP_SETTING_STATUS:
        if (m_bAvmTypeFap == TRUE) {
           _vUpdateFapSetting(static_cast<tenCsmFapSettingType>(u32Value));
           _vUpdateFapSettingkey(u32Value);
        }
        break;

     case EN_CONTROL_AVM__FAP_AVM_SETTING_STATUS:
        if (m_bAvmTypeFap == TRUE) {
           _vUpdateAvmSetting(static_cast<tenCsmFapAvmSettingType>(u32Value));
        }
        break;

     case EN_CONTROL_AVM__FAP_SOUND_DIRECTION:
     case EN_CONTROL_AVM__FAP_SOUND_LEVEL_CONTROL:
        if (m_bAvmTypeFap == TRUE) {
           // todo:
        }
        break;

    default:
        break;
   }
}

/******************************************************************************/
/* FUNCTION     _vHandleIPAButtonChange()                                     */
/******************************************************************************/
/**
*  \brief       handle function for IPA button events
*
*  \param       event
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::_vHandleIPAButtonChange(tU32 u32Value)
{
    tU8 u8Key = ((u32Value & 0xFF00) >> 8 );
    tU8 u8Action = (u32Value & 0xFF);

    ETG_TRACE_USR4(("   CA-AVM :: ............_vHandleIPAButtonChange(): Button %u  Action %u    ", \
            ETG_ENUM(DISPVIDCTRL_E8_IPA_BUTTON_ID, u8Key), ETG_ENUM(DISPVIDCTRL_E8_RVC_BUTTONSTATE, u8Action)));

    tU32 u32Trigger = dispvidctrl_tclFsm_IpaSwitchOperation::eTrigger_None;

    switch (u8Action)
    {
    case rvc_fi_tcl_e8_RVC_ButtonState::FI_EN_PRESSED:
    case rvc_fi_tcl_e8_RVC_ButtonState::FI_EN_SHORTPRESSED:
        u32Trigger = dispvidctrl_tclFsm_IpaSwitchOperation::eTrigger_KeyPress;
        break;
    case rvc_fi_tcl_e8_RVC_ButtonState::FI_EN_RELEASED:
        u32Trigger = dispvidctrl_tclFsm_IpaSwitchOperation::eTrigger_KeyRelease;
        break;
    case rvc_fi_tcl_e8_RVC_ButtonState::FI_EN_LONGPRESSED:
        u32Trigger = dispvidctrl_tclFsm_IpaSwitchOperation::eTrigger_KeyLongPress;
        break;
    default:
        // nothing to do
        break;
    }

    DISPVIDCTRL_NULL_POINTER_CHECK(m_poFsm_IpaSwitchOperation);
    m_poFsm_IpaSwitchOperation->vNewTrigger(u32Trigger, TRUE, u8Key);
}

/******************************************************************************/
/* FUNCTION     _vHandleIPA_AudioBeepRequest()                                */
/******************************************************************************/
/**
*  \brief       handle function for IPA sound output requests
*               map the output sound to audio enum and send a request
*
*  \param       Sound output request code
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::_vHandleIPA_AudioBeepRequest(const tU32 u32Request)
{
    ETG_TRACE_USR4(("   CA-AVM :: ............_vHandleIPA_AudioBeepRequest(): %d     ", ETG_CENUM(tenCSMInput_IPASoundRequests, u32Request)));

    midw_fi_tcl_e8_AudioBeepType oBeepType;
    midw_fi_tcl_e8_ResourceNo oAudioResource;
    oBeepType.enType = midw_fi_tcl_e8_AudioBeepType::FI_EN_AUDIO_BEEP_IPA_NOSOUND;
    oAudioResource.enType = midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS;

    switch (u32Request)
    {
        case IPA_CSM_SOUND_REQUEST__IPA_NO_SOUND:
            oBeepType.enType = midw_fi_tcl_e8_AudioBeepType::FI_EN_AUDIO_BEEP_IPA_NOSOUND;
            break;
        case IPA_CSM_SOUND_REQUEST__IPA_DOUBLE_CAUTION:
            oBeepType.enType = midw_fi_tcl_e8_AudioBeepType::FI_EN_AUDIO_BEEP_IPA_3;
            break;
        case IPA_CSM_SOUND_REQUEST__IPA_LONG_SINGLE_CAUTION:
            oBeepType.enType = midw_fi_tcl_e8_AudioBeepType::FI_EN_AUDIO_BEEP_IPA_7;
            break;
        case IPA_CSM_SOUND_REQUEST__IPA_LONG_DOUBLE_CAUTION:
            oBeepType.enType = midw_fi_tcl_e8_AudioBeepType::FI_EN_AUDIO_BEEP_IPA_8;
            break;
        case IPA_CSM_SOUND_REQUEST__IPA_STOP_SOUND:
            oBeepType.enType = midw_fi_tcl_e8_AudioBeepType::FI_EN_AUDIO_BEEP_IPA_STOP_SOUND;
            break;

        default:
            break;
    }

    DISPVIDCTRL_NULL_POINTER_CHECK(m_poClientAudioIf);
    (tVoid) m_poClientAudioIf->sendBeepMStart(oBeepType, oAudioResource );
}

/******************************************************************************/
/* FUNCTION     _vHandleIpa_Voice_Request_Texts()                             */
/******************************************************************************/
/**
*  \brief       handle function for IPA voice output requests
*               map the output sound to sds and send a request
*
*  \param       Voice Request Code
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::_vHandleIpa_Voice_Request_Texts(tVoid)
{
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poServiceRvcIf);
   rvc_ccafi_tclMsgIPA_Voice_TextsMethodStart oIpaVoiceStringMStart;
   m_poServiceRvcIf->vGetIPA_Voice_TextsMsg(oIpaVoiceStringMStart);

   tIPA_Voice_Texts = oIpaVoiceStringMStart.arTexts;
}

tVoid dispvidctrl_tclControl_Avm::vRequestAudioRoute(tVoid)
{
	ETG_TRACE_USR4(("   CA-AVM :: ............vRequestAudioRoute(): %d  ",m_bAvmFapSourceAvailable ));
	tBool breturn = false;
	DISPVIDCTRL_NULL_POINTER_CHECK(m_poAudioRouting);
	if(m_bAvmFapSourceAvailable == false)
	{
		m_bAvmFapSourceAvailable = m_poAudioRouting->bRequestAudioSourceAvailability(ARL_SRC_DRVASS_VIDEO);
	}
	else
	{
		breturn = m_poAudioRouting->bRequestAudioRoute(ARL_SRC_DRVASS_VIDEO, ARL_EN_ISRC_ACT_ON);
	}
}

/******************************************************************************/
/* FUNCTION     _vHandleIPA_VoiceOutputRequest()                              */
/******************************************************************************/
/**
*  \brief       handle function for IPA voice output requests
*               map the output sound to sds and send a request
*
*  \param       Voice Request Code
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::_vHandleIPA_VoiceOutputRequest(const tU32 u32Request)
{
#ifndef VARIANT_S_FTR_ENABLE_AIVI_SCOPE1
    ETG_TRACE_USR4(("   CA-AVM :: ............_vHandleIPA_VoiceOutputRequest(): %d     ", ETG_CENUM(tenCSMInput_IPAVoiceRequests, u32Request)));

    Idispvidctrl_tclClientSds_ttsIf* poSdsTTS = dynamic_cast<Idispvidctrl_tclClientSds_ttsIf*>(_cpoMain->getHandler("Idispvidctrl_tclClientSds_ttsIf"));
    DISPVIDCTRL_NULL_POINTER_CHECK(poSdsTTS);

    I_dispvidctrl_tclAudioPlayer* poAudioPlayer = dynamic_cast<I_dispvidctrl_tclAudioPlayer*>(_cpoMain->getHandler("dispvidctrl_tclAudioPlayer"));
    DISPVIDCTRL_NULL_POINTER_CHECK(poAudioPlayer);

    DISPVIDCTRL_NULL_POINTER_CHECK(m_poAudioRouting);

    if ( m_bDaVariant == TRUE )
    {
       poAudioPlayer->vPrepareIpaRequest(static_cast<tU8>(u32Request));
    }

    switch ( u32Request )
    {
        case IPA_CSM_VOICE_REQUEST__IPA_START_PARK_ASSIST:
        case IPA_CSM_VOICE_REQUEST__IPA_PARK_ASSIST_FINISHED:
        case IPA_CSM_VOICE_REQUEST__IPA_PARK_ASSIST_CANCELED:
        case IPA_CSM_VOICE_REQUEST__IPA_PARK_ASSIST_COULD_NOT_START:
        case IPA_CSM_VOICE_REQUEST__IPA_SELECT_REVERSE_GEAR:
        case IPA_CSM_VOICE_REQUEST__IPA_SELECT_DRIVE_GEAR:
        case IPA_CSM_VOICE_REQUEST__IPA_DRIVE_FORWARD:
        case IPA_CSM_VOICE_REQUEST__IPA_DRIVE_REARWARD:
        {
           if ( m_bDaVariant == FALSE )
           {
              std::string szTextandTag = "";
              std::vector<rvc_fi_tcl_IPA_Voice_Text_Strings>::const_iterator cit = tIPA_Voice_Texts.begin();
              
              for (; cit != tIPA_Voice_Texts.end(); cit++)
              {
                 if ( u32Request == static_cast<tU32>(cit->e8RequestId.enType) )
                 {
                    szTextandTag = cit->szText;
                    break;
                 }
              }
              
              ETG_TRACE_USR4(("   CA-AVM :: ............Voice output TextAndTags String : %s", szTextandTag.c_str()));

              // set the input string
              sds_fi_tclString szText;
              szText.bSet(szTextandTag.c_str(), sds_fi_tclString::FI_EN_UTF8);
              poSdsTTS->setTTSString(szText);
           }
           
           if ( FALSE == m_bAudioChannelAvailable )
           {
              // request for audio channel if not active already
              m_poAudioRouting->bRequestAudioRoute(ARL_SRC_IPA_TTS, ARL_EN_ISRC_ACT_ON);
           }
           else
           {
              if ( m_bDaVariant == FALSE )
              {
                 poSdsTTS->RestartTTS();
              }
              else
              {
                 poAudioPlayer->vActivateIpaAudio(TRUE);
              }
           }
        }
        break;

        case IPA_CSM_VOICE_REQUEST__IPA_STOP_VOICE:
        case IPA_CSM_VOICE_REQUEST__IPA_NO_VOICE:
        default:
        {
            if ( m_bDaVariant == FALSE )
            {
               // clear the string in SDS client
               sds_fi_tclString szText;
               szText.bSet("\0", sds_fi_tclString::FI_EN_UTF8);
               poSdsTTS->setTTSString(szText);
            }

            // Abort if some voice prompt is played already, handled in Audio Routing
            m_poAudioRouting->bRequestAudioRoute(ARL_SRC_IPA_TTS, ARL_EN_ISRC_ACT_OFF);
        }
        break;
    }
#else
    ETG_TRACE_USR4(("dispvidctrl_tclControl_Avm::_vHandleIPA_VoiceOutputRequest SDS Not Mounted, use audio player maybe"));
#endif
}

tVoid dispvidctrl_tclControl_Avm::vHandleAudioChannelStateChange(tU8 u8NewState)
{
    ETG_TRACE_USR4(("   CA-AVM :: ............vHandleAudioChannelStateChange entered : %u", u8NewState));

    switch(u8NewState)
    {
    case AUDIO_EN_OFF:
    {
        m_bAudioChannelAvailable = FALSE;
    }
    break;

    case AUDIO_EN_ON:
    {
        m_bAudioChannelAvailable = TRUE;
    }
    break;

    case AUDIO_EN_PAUSE:
    {
        m_bAudioChannelAvailable = TRUE;
    }
    break;

    default:
        break;
    }
}

/******************************************************************************/
/* FUNCTION     vHandleFsmOutEvent_Avm()                                      */
/******************************************************************************/
/**
*  \brief       handle function to post output events from FSM Avm
*
*  \param       event
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::vHandleFsmOutEvent_Avm(tenFsmOutEvent_Avm enFsmOutEvent_Avm, tU32 u32Value)
{
   // ...FSM_OUT__FSM...     // input events for FSM AVM (Loopback to FSM)
   // ...FSM_OUT__HMI...     // input events for FSM HMI

   ETG_TRACE_USR4(("   CA-AVM :: vHandleFsmOutEvent_Avm(): %d  ,u32Value = %d   ", ETG_CENUM(tenFsmOutEvent_Avm, enFsmOutEvent_Avm), u32Value));
   //ETG_TRACE_FATAL(("[%d ms]   CA-AVM :: vHandleFsmOutEvent_Avm(): %d  ,u32Value = %d   ", OSAL_ClockGetElapsedTime(), ETG_CENUM(tenFsmOutEvent_Avm, enFsmOutEvent_Avm), u32Value));

   switch (enFsmOutEvent_Avm)
   {
      case FSM_AVM_OUT__FSM_AVM_INPUT_EVENT:
         {
            tChar strIfName[] = "dispvidctrl_tclControl_Avm";
            // post loop-back message to AVM FSM, u32Value will be eTrigger... value
            dispvidctrl_tclAppMain::theServer()->vPostMsgFsmAvm(strIfName, u32Value);
         }
         break;

      case FSM_AVM_OUT__HMI_VISUALISATION_OFF:
         {
            tChar strIfName[] = "dispvidctrl_tclControl_Hmi";
            dispvidctrl_tclAppMain::theServer()->vPostMsgHmiVisualisationOff(strIfName, u32Value);

            if (m_bAvmTypeFap)
            {
               // de-initialize touch interface
               // post event to InputHandler to stop touch opserver
               tChar strIfName[] = "dispvidctrl_tclInputHandler";
               dispvidctrl_tclAppMain::theServer()->vPostMsgControlFap(strIfName, (tU32) EN_INPUT_HANDLER__FAP_VISUALISATION_OFF);

               if ( u32Value == dispvidctrl_tclFsm_Hmi::EN_VD_RVC_HMI_VISUALISATION_OFF__SYSTEM_MALFUNCTION )
               {
                  if (m_poClientVcan) {
                     m_poClientVcan->vWriteIPA_Activation_Flag((tBool) 0);
                  }
               }
            }
         }
         break;
      case FSM_AVM_OUT__HMI_VISUALISATION_ON:
         {
            tChar strIfName[] = "dispvidctrl_tclControl_Hmi";
            dispvidctrl_tclAppMain::theServer()->vPostMsgHmiVisualisationOn(strIfName);

            if (m_bAvmTypeFap)
            {
               // initialize touch interface
               // post event to InputHandler to start touch opserver
               tChar strIfName[] = "dispvidctrl_tclInputHandler";
               dispvidctrl_tclAppMain::theServer()->vPostMsgControlFap(strIfName, (tU32) EN_INPUT_HANDLER__FAP_VISUALISATION_ON);
            }
         }
         break;
      case FSM_AVM_OUT__HMI_VISUALISATION_ON_CAMERA_SETTINGS:
         {
            tChar strIfName[] = "dispvidctrl_tclControl_Hmi";
            dispvidctrl_tclAppMain::theServer()->vPostMsgHmiVisualisationOnCameraSettings(strIfName);
         }
         break;
      // -------------------------------------------------
      case FSM_AVM_OUT__INVALIDATE_CSM_RX:
         if (m_poClientVcan) {
// [Michael 2018_Feb_06]
// todo: activate first when CSM "Indicate Always" implementation is available
//            m_poClientVcan->vInvalidateCsmRx();
         }
         break;
      // -------------------------------------------------
      case FSM_AVM_OUT__CSM_WRITE_INITIALISING_DISPLAY_SWITCH_STATUS:
         if (m_poClientVcan) {
            m_poClientVcan->vWriteUPA_TX_Initializing((tU8) u32Value);
         }
         break;
      case FSM_AVM_OUT__CSM_WRITE_CAMERA_SWITCH_KEEP_DISPLAY_SWITCH_STATUS:
         if (m_poClientVcan) {
            tBool bReinit = FALSE;
            m_poClientVcan->vWriteUPA_Camera_Switch_Keep(bReinit);
         }
         break;
      case FSM_AVM_OUT__CSM_WRITE_CAMERA_SWITCH_PRESS_DISPLAY_SWITCH_STATUS:
         if (m_poClientVcan) {
            if (DISPVIDCTRL_AllianceVCANSlot_Type_C1A != m_u8VCANDiversity) {
               tBool bReinit = FALSE;
               m_poClientVcan->vWriteUPA_Camera_Switch_Press(bReinit);
            } else {
               m_poClientVcan->vWriteAVM_Switch_Request(0x01); // Switch Press
            }
         }
         break;
      case FSM_AVM_OUT__CSM_WRITE_CAMERA_OFF_REQUEST_DISPLAY_SWITCH_STATUS:
         if (m_poClientVcan) {
             if (DISPVIDCTRL_AllianceVCANSlot_Type_C1A != m_u8VCANDiversity) {
                 tBool bReinit = FALSE;
                 m_poClientVcan->vWriteUPA_Camera_OFF_Request(bReinit);
             } else {
               m_poClientVcan->vWriteAVM_Switch_Request(0x00); // Switch Release
               m_poClientVcan->vWriteAVM_View_Change_Request(DISPVIDCTRL_DATA__AVM_R_REQUEST_OFF);
             }
          }
          break;

      case FSM_AVM_OUT__CSM_REINIT_DISPLAY_SWITCH_STATUS:
         _bReinitCsmMessage_DisplaySwitchStatus();
         break;

      // -------------------------------------------------

      case FSM_AVM_OUT__CSM_WRITE_CLEAR_DISPLAY_REQUESTS:
         if (DISPVIDCTRL_AllianceVCANSlot_Type_C1A == m_u8VCANDiversity)
         {
            m_poClientVcan->vWriteAVM_Switch_Request(0x00);
            m_poClientVcan->vWriteAVM_View_Change_Request(DISPVIDCTRL_DATA__AVM_R_REQUEST_NO_REQUEST);
         }
         break;
      case FSM_AVM_OUT__CSM_CHECK_INTIALISING_SIGNAL:
         if (m_poClientVcan) {
             m_poClientVcan->vCheckInitialisingSignal();
         }
         break;

      // -------------------------------------------------
      case FSM_AVM_OUT__AVM_UPDATE_AFFORDABLE_ITS:
         {
            vUpdateAffordableIts((tBool) u32Value);
         }
         break;
      case FSM_AVM_OUT__AVM_VIEW_CHANGE_REQUEST_CHANGE:
         {
            tChar strIfName[] = "dispvidctrl_tclControl_Avm";
            dispvidctrl_tclAppMain::theServer()->vPostMsgControlAvm(strIfName, (tU32) EN_CONTROL_AVM__VIEW_CHANGE_REQUEST_CHANGE);
         }
         break;
      case FSM_AVM_OUT__AUDIO_SEND_OPERATION_BEEP:
         if (m_poClientAudioIf && !m_bAudioBeepsMuted) {
            midw_fi_tcl_e8_AudioBeepType oBeepType;
            midw_fi_tcl_e8_ResourceNo oAudioResource;
            oBeepType.enType = midw_fi_tcl_e8_AudioBeepType::FI_EN_AUDIO_BEEP_AVM_OPERATION;
            oAudioResource.enType = midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS;

            (tVoid) m_poClientAudioIf->sendBeepMStart(oBeepType, oAudioResource );
         }
         break;
      case FSM_AVM_OUT__AUDIO_SEND_ERROR_BEEP:
         if (m_poClientAudioIf && !m_bAudioBeepsMuted) {
            midw_fi_tcl_e8_AudioBeepType oBeepType;
            midw_fi_tcl_e8_ResourceNo oAudioResource;
            oBeepType.enType = midw_fi_tcl_e8_AudioBeepType::FI_EN_AUDIO_BEEP_AVM_ERROR;
            oAudioResource.enType = midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS;

            (tVoid) m_poClientAudioIf->sendBeepMStart(oBeepType, oAudioResource );
         }
         break;

      case FSM_AVM_OUT__AVM_UPDATE_IPA_STATUS:
        if ( TRUE == m_bAvmTypeIpa )
        {
           if ( m_u8IPAStatus != static_cast<tenIPAStatus>(u32Value) )
           {
              m_u8IPAStatus = static_cast<tenIPAStatus>(u32Value);
              _vUpdate_IPAConfiguration();
           }
        }
        break;

      case FSM_AVM_OUT__CSM_WRITE_IPA_ACTIVATION_FLAG_STATUS:
        if ( TRUE == m_bAvmTypeIpa || TRUE == m_bAvmTypeFap )
        {
           if (m_poClientVcan) {
              m_poClientVcan->vWriteIPA_Activation_Flag((tBool) u32Value);
              m_poClientVcan->vWriteFAP_Screen_Activation_flag(0x03); //todo: hard coded for FAP to work
           }
        }
        break;
        
      case FSM_AVM_OUT__AVM_ERROR_ACTIVE:
         if ( FALSE == m_bTimerErrorConfirmRunning ) {
            if ( m_bAvmTypeRenault ){
               _bSetTimer_AvmErrorConfirm(DISPVIDCTRL_AVM_RENAULT_ERROR_CONFIRMATION_TIME_IN_MS);
            } else if ( m_bAvmTypeIpa ) {
               _bSetTimer_AvmErrorConfirm(DISPVIDCTRL_AVM__NISSAN_ERROR_CONFIRMATION_TIME_IN_MS);
            }
            m_bTimerErrorConfirmRunning = TRUE;
         }
         _vUpdate_IPAConfiguration();
         break;

      case FSM_AVM_OUT__AVM_ERROR_DEACTIVE:
         if ( TRUE == m_bTimerErrorConfirmRunning ) {
            _bSetTimer_AvmErrorConfirm(0);
            m_bTimerErrorConfirmRunning = FALSE;
         }
         _vUpdate_IPAConfiguration();
         break;

      case FSM_AVM_OUT__CSM_WRITE_IPA_SETTING_CHANGE_REQUEST_STATUS:
         // not required to handle
         break;

      case FSM_AVM_OUT__AVM_FAP_FAIL_SAFE_OPERATON:
         // FAP Fail Safe Handling
         if ( m_bAvmTypeFap )
         {

         }
         break;

      default:
         break;
   }
}

/******************************************************************************/
/* FUNCTION     vHandleFsmOutEvent_IpaSwitchOperation()                       */
/******************************************************************************/
/**
*  \brief       handle function to post output events from FSM IpaSwitchOperation
*
*  \param       event
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::vHandleFsmOutEvent_IpaSwitchOperation(tenFsmOutEvent_IpaSwitchOperation enFsmOutEvent_IpaSwitchOperation, tU32 u32Value)
{
   // ...FSM_OUT__AVM...     // input events for FSM AVM (Loopback to FSM)
   // ...FSM_OUT__HMI...     // input events for FSM HMI

   ETG_TRACE_USR4(("   CA-AVM :: vHandleFsmOutEvent_IpaSwitchOperation(): %d     ", ETG_CENUM(tenFsmOutEvent_IpaSwitchOperation, enFsmOutEvent_IpaSwitchOperation)));

   switch (enFsmOutEvent_IpaSwitchOperation)
   {
      case FSM_IPASWITCHOPERATION_OUT__FSM_IPASWITCHOPERATION_INPUT_EVENT:
         {
            tChar strIfName[] = "dispvidctrl_tclControl_Avm";
            // post loop-back message to IpaSwitchOperation FSM, u32Value will be eTrigger... value
            dispvidctrl_tclAppMain::theServer()->vPostMsgFsmIpaSwitchOperation(strIfName, u32Value);
         }
         break;
      case FSM_AVM_OUT__CSM_WRITE_IPA_SW_LONG_PRESS_STATUS:
          if (m_poClientVcan) {
             tBool bReinit = (tBool) u32Value;
             m_poClientVcan->vWriteIPA_SW_Long_Press(bReinit);
           }
           break;
      case FSM_AVM_OUT__CSM_WRITE_IPA_SW_OPERATION_PRESS_STATUS:
          if (m_poClientVcan) {
             tBool bReinit = (tBool) u32Value;
             m_poClientVcan->vWriteIPA_SW_Operation_Status(bReinit);
           }
           break;
      default:
         break;
   }
}

/******************************************************************************/
/* FUNCTION     _vOnEvent_FsmControl_Avm                                      */
/******************************************************************************/
/**
*  \brief
*
*  \param       event, data
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::_vOnEvent_FsmControl_Avm(tenControl_Avm enControl, tU32 u32Value)
{
   ETG_TRACE_USR4(("   CA-AVM :: ........._vOnEvent_FsmControl_Avm(): %d     ", ETG_CENUM(tenControl_Avm, enControl)));
   ETG_TRACE_USR4(("   CA-AVM :: . FSM Control_Avm Super State Entry: %u     ", ETG_CENUM(tenFsmControl_Avm_SuperState, m_enFsmControl_Avm_SuperState)));
   //ETG_TRACE_FATAL(("[%d ms]   CA-AVM :: ........._vOnEvent_FsmControl_Avm(): %d     ", OSAL_ClockGetElapsedTime(), ETG_CENUM(tenControl_Avm, enControl)));
   //ETG_TRACE_FATAL(("[%d ms]   CA-AVM :: . FSM Control_Avm Super State Entry: %u     ", OSAL_ClockGetElapsedTime(), ETG_CENUM(tenFsmControl_Avm_SuperState, m_enFsmControl_Avm_SuperState)));

   tBool bEventProcessed = TRUE;
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(u32Value);

   switch (m_enFsmControl_Avm_SuperState)
   {
      /* ------------------------------------------------------------- */
      /*                                      D I S A B L E D          */
      /* ------------------------------------------------------------- */
      // activation via reverse gear on possible
      case DISPVIDCTRL_CONTROL_AVM__DISABLED:
         {
            switch (enControl)
            {
            case EN_CONTROL_AVM__ENABLE:
            {
               tBool bAvmInitFlagTx = TRUE;
               // update CAN signals we can send always to AVM unit
               // in case of first time enable we have to correct the 5FE init and language signal of the CSM Proxy
               // according to the value we have send via INC to V850 earlyApp
               // this we can do only with a multiple signal write
// todo: use multiple signal write
               // the can signal values which we have modified in Sync-Handler for early AVM must be repeated here
               // (init signal and language)

               // is it the first LATE transition after EARLY?
               if ((u32Value & DISPVIDCTRL_VIDEO__MASK_EARLY_ACTIV) > 0)
               {
                  if ((u32Value & DISPVIDCTRL_VIDEO__MASK_AVM_INIT_FLAG_TX) == 0)
                  {
                     // it was cleared in Sync-Handler already, so send initializing signal as cleared
                     bAvmInitFlagTx = FALSE;
                  }
               }

               if (m_poClientVcan)
               {
                  m_poClientVcan->vWriteUPA_TX_Initializing((tU8) bAvmInitFlagTx);
               }

                if (m_u32HmiLanguage == DISPVIDCTRL_DATA__HMI_LANGUAGE_NOT_RECEIVED)
                {
                    dispvidctrl_tclPersistentData* poPersistentData =
                            dynamic_cast<dispvidctrl_tclPersistentData*>(_cpoMain->getHandler("dispvidctrl_tclPersistentData"));
                    if (poPersistentData)
                    {
                        (tVoid) poPersistentData->bGetPersistent_WarningTextLanguage(m_u32HmiLanguage);
                    }
                }

                if (m_u32HmiLanguage != DISPVIDCTRL_DATA__HMI_LANGUAGE_NOT_RECEIVED)
                {
                    (tVoid) bOnUpdate_Language(m_u32HmiLanguage);
                }

                // we are ready with initialization, check AVM initialization status
                if ( m_poClientVcan )
                {
                    m_poClientVcan->vCheckInitialisingSignal(); // this is posting a message InitActive or InitDone
                }
                // start re-init timer for verification in case
                // no init active/done event will come
                if(m_poFsm_Avm)
                {
                    // update Ignition status
                    if (m_bIgnition)
                    {
                        m_poFsm_Avm->vUpdateTriggerState((tU32) dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_IGNITION_ON, TRUE);
                    }
                    else
                    {
                        // In case it is the first activation with early video status and early AVM video is active:
                        //  When IGN was going OFF meanwhile we have to post a HMI Visualisation OFF
                        //  Therefore we must process the Ignition OFF event in Uninitialised State as "Do" action,
                        //  else we have to update the trigger state only.

                        // is it the first LATE transition after EARLY? and Ignition Info was received from SPM already
                        if (   ((u32Value & DISPVIDCTRL_VIDEO__MASK_EARLY_ACTIV) > 0)
                            && (m_bIgnitionStateValid == TRUE))
                        {
                           m_poFsm_Avm->vNewTrigger((tU32) dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_IGNITION_OFF, TRUE);
                        }
                        else
                        {
                           m_poFsm_Avm->vUpdateTriggerState((tU32) dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_IGNITION_OFF, TRUE);
                        }
                    }
                    // update reverse debounced status
                    tBool bReverseDebounced;
                    if(m_poControlRvc)
                    {
                        bReverseDebounced = m_poControlRvc->bGetReverseDebouncedState();
                        if (bReverseDebounced)
                        {
                            m_poFsm_Avm->vUpdateTriggerState((tU32) dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_REVERSE_ON_DEBOUNCED, TRUE);
                        }
                        else
                        {
                            m_poFsm_Avm->vUpdateTriggerState((tU32) dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_REVERSE_OFF, TRUE);
                        }
                    }
                    if (m_poFsm_Avm->bGetSetTimer_AvmReinit(DISPVIDCTRL_AVM_REINIT_TIME_IN_MS) == FALSE)
                    {
                        ETG_TRACE_ERR((" CA-AVM   :: E R R O R in _vHandleEvent(): .......... AVM Reinit Timer start failed!          "));
                    }
                }

                // If IPA Configuredu8SwitchesStatus
                if ( m_bAvmTypeIpa == TRUE )
                {
                    _vUpdate_IPAConfiguration();
                    _vUpdate_IPAStatus();
                    _vUpdate_IPASwitches();
                    _vUpdate_IPARequests();
                    _vUpdate_IPADirectionIndicator();
                }

                // is it the first LATE transition after EARLY?
                // Prepare the graphics screen already.
                // NCG3D-4737/NCG3d-4738: issue of the first time entry in AVM takes longer than the subsequent entries.
                if ((u32Value & DISPVIDCTRL_VIDEO__MASK_EARLY_ACTIV) > 0)
                {
                    if ((u32Value & DISPVIDCTRL_VIDEO__MASK_AVM_INIT_FLAG_TX) == 0)
                    {
                        dispvidctrl_tclControl_Video* poControlVideo = dynamic_cast<dispvidctrl_tclControl_Video*>(_cpoMain->getHandler("dispvidctrl_tclControl_Video"));
                        DISPVIDCTRL_NULL_POINTER_CHECK(poControlVideo);

                        poControlVideo->vHandleControl(EN_CONTROL_VIDEO__PREPARE_GRAPHICS, (tU32) 0);
                    }
                }

                m_enFsmControl_Avm_SuperState = DISPVIDCTRL_CONTROL_AVM__ENABLED;
            }
            break;
            /*
            * As vd_early could have written 0x5FE message signals on V850 already
            * we can not write any signal until we got the enable request with the
            * with the value of the "Initialising" signal value from vd_early.
            * Language Signal update can be done first when AVM control was enabled.
            */
            case EN_CONTROL_AVM__UPDATE_LANGUAGE:
               m_u32HmiLanguage = u32Value;
               break;
            /*
             * following events are handled to clean up the system in case something was pending and could not be stopped before
             * (there could be some events queued before)
             */
            // todo

            default:
               bEventProcessed = FALSE;
               break;
            }
         }
         break;
      /* ------------------------------------------------------------- */
      /*                                      S T O P P E D            */
      /* todo
      * - Application state not Normal
      * - critical low voltage
      */
      /* --------------------------------------------------------- */


      /* ------------------------------------------------------------- */
      /*                                        E N A B L E D          */
      /* ------------------------------------------------------------- */
      case DISPVIDCTRL_CONTROL_AVM__ENABLED:
         {
            if (EN_CONTROL_AVM__DISABLE == enControl) {
               // set AVM state machine into uninitialized state
               if (m_poFsm_Avm != OSAL_NULL)
               {
                  m_poFsm_Avm->vNewTrigger((tU32) dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_INIT_ACTIVE, TRUE);

                  // new super state
                  m_enFsmControl_Avm_SuperState = DISPVIDCTRL_CONTROL_AVM__DISABLED;
               }
            }
            else {
               tU32 u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_NONE;

               bEventProcessed = TRUE;
               switch (enControl)
               {
                  case EN_CONTROL_AVM__CAN_MSG_TIMEOUT_TRUE:
                  {
                     m_bCanMsgTimeOutState = TRUE;
                     if (!m_bAvmTypeRenault) u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAN_MSG_TIMEOUT_TRUE;
                     _vUpdateCANFrameStatus(enControl);
                  }
                  break;
                  case EN_CONTROL_AVM__CAN_MSG_TIMEOUT_CONFIRMED: // special for AVM Renault
                  {
                     m_bCanMsgTimeOutState = TRUE;
                     if (m_bAvmTypeRenault) u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAN_MSG_TIMEOUT_TRUE;
                     _vUpdateCANFrameStatus(enControl);
                  }
                  break;
                  case EN_CONTROL_AVM__CAN_MSG_TIMEOUT_FALSE:
                  {
                     m_bCanMsgTimeOutState = FALSE;
                     u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAN_MSG_TIMEOUT_FALSE;
                     _vUpdateCANFrameStatus(enControl);
                  }
                  break;
                  case EN_CONTROL_AVM__IGNITION_ON                      :  u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_IGNITION_ON;             break;
                  case EN_CONTROL_AVM__IGNITION_OFF                     :  u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_IGNITION_OFF;            break;
                  case EN_CONTROL_AVM__CAMERA_BUTTON_LONG_PRESS         :  u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_BUTTON_LONG_PRESS;break;
                  case EN_CONTROL_AVM__NORMAL_BEEP                      :  u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_NORMAL_BEEP;             break;
                  case EN_CONTROL_AVM__ERROR_BEEP                       :  u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_ERROR_BEEP;              break;
                  case EN_CONTROL_AVM__AOD_BEEP_1:
                     if (u32Value)
                     {
                        u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_NORMAL_BEEP;
                     }
                     else
                     {
                        bEventProcessed = FALSE;
                        u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_NONE;
                     }
                     break;
                  case EN_CONTROL_AVM__PURPOSE_SWITCH                   :  u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_PURPOSE_SWITCH;          break;
                  case EN_CONTROL_AVM__INIT_ACTIVE                      :  u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_INIT_ACTIVE;             break;
                  case EN_CONTROL_AVM__INIT_DONE                        :  u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_INIT_DONE;               break;
                  case EN_CONTROL_AVM__CAMERA_SCREEN_ON                 :  u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_SCREEN_ON;        break;
                  case EN_CONTROL_AVM__CAMERA_SCREEN_OFF                :  u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_SCREEN_OFF;       break;
                  case EN_CONTROL_AVM__REVERSE_ON_DEBOUNCED             :  u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_REVERSE_ON_DEBOUNCED;    break;
                  case EN_CONTROL_AVM__REVERSE_OFF                      :  u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_REVERSE_OFF;             break;
                  case EN_CONTROL_AVM__CAMERA_BUTTON_PRESS              :  u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_BUTTON_PRESS;     break;
                  case EN_CONTROL_AVM__AFFORDABLE_ITS                   :  u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_AFFORDABLE_ITS;          break;
                  case EN_CONTROL_AVM__VIDEO_SIGNAL_FAILURE             :  u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_VIDEO_SIGNAL_FAILURE;    break;
                  case EN_CONTROL_AVM__VIDEO_SIGNAL_RESTORED            :  u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_VIDEO_SIGNAL_RESTORED;   break;
                  case EN_CONTROL_AVM__VERIFY_AFTER_VIEW_STATUS_STOPPED :  u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_VERIFY_AFTER_VIEW_STATUS_STOPPED;    break;
                  case EN_CONTROL_AVM__CAMERA_SETTINGS_ON               :  u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_SETTINGS_ON;      break;
                  case EN_CONTROL_AVM__CAMERA_SETTINGS_OFF              :  u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_SETTINGS_OFF;     break;
                  case EN_CONTROL_AVM__TIMEOUT_VIEW_CHANGE_REQUEST_WAIT :  u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_TIMEOUT_VIEW_CHANGE_REQUEST_WAIT;    break;
                  case EN_CONTROL_AVM__CURRENT_VIEW_STATUS              :  u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_VIEW_STATUS_UPDATE;    break;
                  case EN_CONTROL_AVM__ERROR_ACTIVE                     :  u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_ERROR_TRUE;   break;
                  case EN_CONTROL_AVM__ERROR_DEACTIVE                   :  u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_ERROR_FALSE;  break;

                  case EN_CONTROL_AVM__CAMERA_BUTTON_RELEASE            :
                  case EN_CONTROL_AVM__TIMEOUT_CAMERA_BUTTON_LONG_PRESS :
                  case EN_CONTROL_AVM__NORMAL_TO_PAUSE                  :
                  case EN_CONTROL_AVM__PAUSE_TO_NORMAL                  :

                  default:
                     {
                        bEventProcessed = FALSE;
                        u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_NONE;
                     }
                     break;
               };

               if (bEventProcessed == TRUE) {
                  if (m_poFsm_Avm) {
                     m_poFsm_Avm->vNewTrigger(u32FSM_Event, TRUE);
                  }
               }

               if (EN_CONTROL_AVM__INIT_CHANGE == enControl ) {
                  ETG_TRACE_USR4((" CA-AVM   :: _vHandleEvent(): ....................... AVM_EV_INIT_CHANGE                      "));
                  // check AVM initialization signal
                  if (m_poClientVcan) {
                     m_poClientVcan->vCheckInitialisingSignal();
                  }
                  bEventProcessed = TRUE;
               }

               if (EN_CONTROL_AVM__ERROR_CONFIRM_TIMEOUT == enControl ) {
                   ETG_TRACE_USR4((" CA-AVM   :: _vHandleEvent(): ....................... EN_CONTROL_AVM__TRY_RECOVER_ERROR_TIMEOUT       "));
                   ETG_TRACE_USR4((" CA-AVM   :: _vHandleEvent(): ....................... m_u8IpaErrorCode: %d", m_u8IpaErrorCode ));
                   m_bTimerErrorConfirmRunning = FALSE;
                   if ( m_poFsm_Avm && ( m_u8IpaErrorCode != DISPVIDCTRL_CONTROL_IPA_NO_ERROR ) ) {
                       // If CAN Timout is true than inform this to Fsm
                       if ( m_u8IpaErrorCode & DISPVIDCTRL_CONTROL_IPA_CAN_TIMEOUT ) {
                          m_poFsm_Avm->vNewTrigger(dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAN_MSG_TIMEOUT_TRUE, TRUE);
                       } else {
                          m_poFsm_Avm->vNewTrigger(dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_SCREEN_OFF, TRUE);
                       }
                   }
               }

               if (EN_CONTROL_AVM__VIEW_CHANGE_REQUEST_CHANGE == enControl ) {
                  ETG_TRACE_USR4((" CA-AVM   :: _vHandleEvent(): ....................... AVM_EV_VIEW_CHANGE_REQUEST_CHANGE       "));

                  // check AVM view change request signal
                  if (m_poClientVcan) {
                     tBool bReturnValue; // = FALSE; -> LINT 838
                     tU8 u8ViewChangeSignalValue = 0;
                     u32FSM_Event = 0;

                     if (FALSE == m_bAvmTypeFap)
                     {
                        bReturnValue = m_poClientVcan->bGetViewChangeRequestSignal(u8ViewChangeSignalValue);
                     }
                     else // if (TRUE == m_bFapModeActive) // Fap is in Active Mode (Just an additional check for Simulation testing)
                     {
                        bReturnValue = m_poClientVcan->bGet_Fap_ViewChangeRequestSignal(u8ViewChangeSignalValue);
                        vRequestAudioRoute();
                     }

                     if ((bReturnValue == TRUE) && (m_poFsm_Avm)) {
                        if(u8ViewChangeSignalValue == 1) {
                           u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_SCREEN_ON;
                           // unlock updates for Display content
                           // m_DisplayUpdatesLocked = FALSE;
                           // trigger an update (post event)
                           //if ((m_poMainAppl) && (m_poMainAppl->poGetInstance_Thread_Worker()))
                           {
                              //m_poMainAppl->poGetInstance_Thread_Worker()->vPostMessage(rThreadMsg_Worker(EN_ACTION_WORKER__AVM_CONTROL_EVENT, AVM_EV_IPA_REQUEST_CHANGE));
                           }
                        }
                        else {
                           u32FSM_Event = dispvidctrl_tclFsm_Avm::eTrigger_AVM_EV_CAMERA_SCREEN_OFF;

                           // lock updates for Display content
                           //m_DisplayUpdatesLocked = TRUE;
                        }
                        m_poFsm_Avm->vNewTrigger(u32FSM_Event, TRUE);
                     }
                  }
                  bEventProcessed = TRUE;
               }

               if ( EN_CONTROL_AVM__IPA_CONFIGURATION_CHANGE == enControl )
               {
                   _vUpdate_IPAConfiguration();
                   bEventProcessed = TRUE;
               }

               if ( EN_CONTROL_AVM__IPA_ACTIVATION_CHANGE == enControl )
               {
                  if ( ( m_bAvmTypeIpa == TRUE ) || ( m_bAvmTypeFap == TRUE ) ) {
                      if (m_poClientVcan) {
                          m_poClientVcan->vWriteIPA_Activation_Flag((tBool) u32Value);
                      }
                  } else {}
               }

               if (EN_CONTROL_AVM__UPDATE_LANGUAGE == enControl )
               {
                  m_u32HmiLanguage = u32Value;
                  (tVoid) bOnUpdate_Language(u32Value);
               }
            }
         }
         break;
      default:
         bEventProcessed = FALSE;
         break;

   } // switch super state

   ETG_TRACE_USR4(("   CA-AVM :: ................... Event Processed: %u     ", ETG_ENUM(DISPVIDCTRL_NoYes, bEventProcessed)));
   ETG_TRACE_USR4(("   CA-AVM :: .. FSM Control_Avm Super State Exit: %u     ", ETG_CENUM(tenFsmControl_Avm_SuperState, m_enFsmControl_Avm_SuperState)));
   //ETG_TRACE_FATAL(("[%d ms]   CA-AVM :: ................... Event Processed: %u     ",OSAL_ClockGetElapsedTime(), ETG_ENUM(DISPVIDCTRL_NoYes, bEventProcessed)));
   //ETG_TRACE_FATAL(("[%d ms]   CA-AVM :: .. FSM Control_Avm Super State Exit: %u     ",OSAL_ClockGetElapsedTime(), ETG_CENUM(tenFsmControl_Avm_SuperState, m_enFsmControl_Avm_SuperState)));
}


/******************************************************************************/
/* FUNCTION     vUpdateAffordableIts                                          */
/******************************************************************************/
/**
*  \brief       reads the affordable ITS signals and update registered clients
*               for property "Affordable_ITS"
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::vUpdateAffordableIts(tBool bSetToUnknown)
{
   tBool bItsMod;
   //tBool bItsBsw;
   //tBool bItsLdw;
   tBool bItsAvailable;

   rvc_fi_tcl_e8_AffordableIts_Status oAffordableItsAvailable;
   rvc_fi_tcl_e8_AffordableIts_StatusFeature oBswStatus;
   rvc_fi_tcl_e8_AffordableIts_StatusFeature oModStatus;
   rvc_fi_tcl_e8_AffordableIts_StatusFeature oLdwStatus;

   oAffordableItsAvailable.enType = rvc_fi_tcl_e8_AffordableIts_Status::FI_EN_NOT_AVAILABLE;
   oBswStatus.enType = rvc_fi_tcl_e8_AffordableIts_StatusFeature::FI_EN_UNKNOWN;
   oModStatus.enType = rvc_fi_tcl_e8_AffordableIts_StatusFeature::FI_EN_UNKNOWN;
   oLdwStatus.enType = rvc_fi_tcl_e8_AffordableIts_StatusFeature::FI_EN_UNKNOWN;


   if (    (bSetToUnknown == FALSE)
         &&(m_poClientVcan)
         &&(m_poClientVcan->bGetAffordableItsSignals(bItsMod, bItsAvailable))  )
                 //bItsBsw,
                 //bItsLdw,
                 //bItsAvailable))  )
   {
/*
      if (bItsBsw == TRUE)
      {
         oBswStatus.enType = rvc_fi_tcl_e8_AffordableIts_StatusFeature::FI_EN_ON;
      }
      else
      {
         oBswStatus.enType = rvc_fi_tcl_e8_AffordableIts_StatusFeature::FI_EN_OFF;
      }
*/

      if (bItsMod == TRUE)
      {
         oModStatus.enType = rvc_fi_tcl_e8_AffordableIts_StatusFeature::FI_EN_ON;
      }
      else
      {
         oModStatus.enType = rvc_fi_tcl_e8_AffordableIts_StatusFeature::FI_EN_OFF;
      }

/*
       if (bItsLdw == TRUE)
      {
         oLdwStatus.enType = rvc_fi_tcl_e8_AffordableIts_StatusFeature::FI_EN_ON;
      }
      else
      {
         oLdwStatus.enType = rvc_fi_tcl_e8_AffordableIts_StatusFeature::FI_EN_OFF;
      }
*/

      if (bItsAvailable == TRUE)
      {
         oAffordableItsAvailable.enType = rvc_fi_tcl_e8_AffordableIts_Status::FI_EN_AVAILABLE;
      }
      else
      {
         oAffordableItsAvailable.enType = rvc_fi_tcl_e8_AffordableIts_Status::FI_EN_NOT_AVAILABLE;
      }
   }

   // update registered clients
   if ( m_poServiceRvcIf ) {
       m_poServiceRvcIf->sendAffordable_ITSStatus(oAffordableItsAvailable, oBswStatus, oModStatus, oLdwStatus);
   }
}


/******************************************************************************/
/* FUNCTION     _vOnUpdate_Language                                           */
/******************************************************************************/
/**
*  \brief       Update language signal for AVM.
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Avm::bOnUpdate_Language(tU32 u32NewLanguage)
{
   tBool bLanguageFound = FALSE;

   // mapping HMI - AVM Language
   for(tU8 u8Count = 0; u8Count < DISPVIDCTRL_DATA_LANGUAGE_MAPPING_AVM_TABLE_SIZE; u8Count++)
   {
      if(stLanguageMappingAvm[u8Count].u8HmiLanguageCode == (tU8) u32NewLanguage) {
         m_u8AvmLanguage = stLanguageMappingAvm[u8Count].u8AvmLanguageCode;
         bLanguageFound = TRUE;
      }
   }

   if (bLanguageFound == FALSE)
   {
      // Use English_US if not found in the table
      m_u8AvmLanguage = DISPVIDCTRL_DATA__AVM_LANGUAGE_ENGLISH_US;
      ETG_TRACE_FATAL((" CA-AVM   :: _vOnUpdate_Language(): ................. No Language found for AVM set! Using English_US"));
   }

   if (m_poClientVcan)
   {
      m_poClientVcan->vWriteUPA_Setting_Language_Request(m_u8AvmLanguage);
   }

   {
      dispvidctrl_tclPersistentData* poPersistentData = dynamic_cast<dispvidctrl_tclPersistentData*>(_cpoMain->getHandler("dispvidctrl_tclPersistentData"));
      if (poPersistentData)
      {
         poPersistentData->vSet_Language_Avm((tU32) m_u8AvmLanguage);
      }
   }

   return bLanguageFound;
}


/******************************************************************************/
/* FUNCTION     bIsAvmGraphicsOn                                              */
/******************************************************************************/
/**
*  \brief       resets the toggle signals
*
*  \param       none
*  \return      write success state
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Avm::bIsAvmGraphicsOn()
{
   return m_bAvmRGraphicsOn;
}


/******************************************************************************/
/* FUNCTION     _bReinitCsmMessage_DisplaySwitchStatus                        */
/******************************************************************************/
/**
*  \brief       resets the toggle signals
*
*  \param       none
*  \return      write success state
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Avm::_bReinitCsmMessage_DisplaySwitchStatus(tVoid)
{
   tBool bRetVal = FALSE;

   DISPVIDCTRL_NULL_POINTER_CHECK_VAL(m_poClientVcan);

   if (DISPVIDCTRL_AllianceVCANSlot_Type_C1A != m_u8VCANDiversity)
   {
      tBool bReinit = TRUE;
      // write reinit:
      //   CameraOffRequest
      //   CameraSwitchKeep
      //   CameraSwitchPress
      //   LdwSwitchPress
      //   BswSwitchPress
      //   ModSwitchPress
      m_poClientVcan->vWriteUPA_Camera_OFF_Request(bReinit);
      m_poClientVcan->vWriteUPA_Camera_Switch_Keep(bReinit);
      m_poClientVcan->vWriteUPA_Camera_Switch_Press(bReinit);
      //m_poClientVcan->vWriteLDW_PRESS_5FE(bReinit);
      //m_poClientVcan->vWriteBSW_PRESS_5FE(bReinit);
      m_poClientVcan->vWriteUPA_MOD_Switch_Press(bReinit);
   }
   else // CAN messages valid only for C1A
   {
      tU8 u8InitSetting = 0x00;
      m_poClientVcan->vWriteAVM_Switch_Request(u8InitSetting);
      m_poClientVcan->vWriteAVM_View_Change_Request(u8InitSetting);
   }

   bRetVal = TRUE;

   return (bRetVal);
}

/******************************************************************************/
/* FUNCTION     _vUpdate_AVM_CurrentViewStatus                                */
/******************************************************************************/
/**
*  \brief       resets the toggle signals
*
*  \param       u8CurrentView displayed on AVM
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::_vUpdate_AVM_CurrentViewStatus(const tU8 u8CurrentView)
{
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poServiceRvcIf);

   tBool bGraphicsOn = TRUE;
   rvc_fi_tcl_e8_AVM_View_Status eViewStatus;
   eViewStatus.enType = rvc_fi_tcl_e8_AVM_View_Status::FI_EN_NO_REQUEST;

   switch (u8CurrentView)
   {
   case DISPVIDCTRL_DATA__AVM_R_VIEW_FRONT_BIRD:
   case DISPVIDCTRL_DATA__AVM_R_VIEW_FRONT_SIDE:
   case DISPVIDCTRL_DATA__AVM_R_VIEW__REAR_BIRD:
   case DISPVIDCTRL_DATA__AVM_R_VIEW__REAR_SIDE:
   case DISPVIDCTRL_DATA__AVM_R_VIEW__ONLY_REAR:
      eViewStatus.enType = static_cast<rvc_fi_tcl_e8_AVM_View_Status::tenType>(u8CurrentView);
      break;

   case DISPVIDCTRL_DATA__AVM_R_VIEW_DEALERMODE:
      bGraphicsOn = FALSE;
      eViewStatus.enType = static_cast<rvc_fi_tcl_e8_AVM_View_Status::tenType>(u8CurrentView);
      break;
   default:
      bGraphicsOn = FALSE;
      eViewStatus.enType = rvc_fi_tcl_e8_AVM_View_Status::FI_EN_NO_REQUEST;
      break;
   }

   ETG_TRACE_USR1((" CA-AVM   :: _vUpdate_AVM_CurrentViewStatus(): Avm Renault View Status is %u", eViewStatus.enType));

   m_poServiceRvcIf->sendAvmR_Current_ViewStatus(eViewStatus);

   // Update Graphics if needed
   if ( bGraphicsOn != m_bAvmRGraphicsOn)
   {
      m_bAvmRGraphicsOn = bGraphicsOn;

      dispvidctrl_tclControl_Video* poControl_Video = dynamic_cast<dispvidctrl_tclControl_Video*>(_cpoMain->getHandler("dispvidctrl_tclControl_Video"));

      if (poControl_Video)
      {
         if ( m_bAvmRGraphicsOn )
         {
            poControl_Video->vHandleControl(EN_CONTROL_VIDEO__RESIZE_VIDEO, RVC_AVM_VIDEO_FULL_SCREEN);
         }
         else
         {
            poControl_Video->vHandleControl(EN_CONTROL_VIDEO__RESIZE_VIDEO, RVC_AVM_VIDEO_FORCE_FULL_SCREEN);
         }
      }
   }
}

/******************************************************************************/
/* FUNCTION     _vUpdate_AVM_ExitViewStatus                                   */
/******************************************************************************/
/**
*  \brief       resets the toggle signals
*
*  \param       u8ExitAllowed - Is Closing AVM allowed from the HMI
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::_vUpdate_AVM_ExitViewStatus(const tU8 u8ExitAllowed)
{
   ETG_TRACE_USR1((" CA-AVM   :: _vUpdate_AVM_ExitViewStatus(): u8ExitAllowed %d", u8ExitAllowed));

   DISPVIDCTRL_NULL_POINTER_CHECK(m_poServiceRvcIf);

   rvc_fi_tcl_e8_Exit_Authorized oExitAuthorized;
   oExitAuthorized.enType = static_cast<rvc_fi_tcl_e8_Exit_Authorized::tenType>(u8ExitAllowed);

   m_poServiceRvcIf->sendView_Close_AuthorizedStatus(oExitAuthorized);
}

/******************************************************************************/
/* FUNCTION     _vUpdateCANFrameStatus                                        */
/******************************************************************************/
/**
*  \brief       sends the current state of AVM CAN frame for C1A
*
*  \param       Frame Status
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::_vUpdateCANFrameStatus(const tU32 u32FrameState)
{
   ETG_TRACE_USR4(("   CA-AVM :: ........._vUpdateCANFrameStatus(): %d     ", ETG_CENUM(tenControl_Avm, u32FrameState)));

   rvc_fi_tcl_e8_RVC_SignalStatus oSignalState;

   if (EN_CONTROL_AVM__CAN_MSG_TIMEOUT_FALSE == u32FrameState)
   {
       oSignalState.enType = rvc_fi_tcl_e8_RVC_SignalStatus::FI_EN_SIGNAL_OK;
   }
   else if (EN_CONTROL_AVM__CAN_MSG_TIMEOUT_TRUE == u32FrameState)
   {
       oSignalState.enType = rvc_fi_tcl_e8_RVC_SignalStatus::FI_EN_CURRENT_FAILURE;
   }
   else if (EN_CONTROL_AVM__CAN_MSG_TIMEOUT_CONFIRMED == u32FrameState)
   {
       oSignalState.enType = rvc_fi_tcl_e8_RVC_SignalStatus::FI_EN_CONFIRMED_FAILURE;
   }
   else // is this ok?
   {
       oSignalState.enType = rvc_fi_tcl_e8_RVC_SignalStatus::FI_EN_SIGNAL_OK;
   }

   // if (m_bAvmTypeRenault)
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poServiceRvcIf);
   m_poServiceRvcIf->sendCANFrameStatusStatus(oSignalState);
}

/******************************************************************************/
/* FUNCTION     _vUpdateFapSetting                                            */
/******************************************************************************/
/**
*  \brief       Fetch only the updated FAP Setting Item and update to the clients
*
*  \param       Setting Type which is changed
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::_vUpdateFapSetting(const tenCsmFapSettingType FapSettingType) const
{
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poServiceRvcIf);
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poClientVcan);

   std::vector<rvc_fi_tcl_FAP_Settings_State, std::allocator<rvc_fi_tcl_FAP_Settings_State> > FapSettings;
   std::vector<rvc_fi_tcl_AVM_Settings_State, std::allocator<rvc_fi_tcl_AVM_Settings_State> > AvmSettings;

   rvc_fi_tcl_FAP_Settings_State oSetting;
   oSetting.e8Type.enType = static_cast<rvc_fi_tcl_e8_FAP_SettingItem::tenType>(FapSettingType);
   oSetting.e8State.enType = static_cast<rvc_fi_tcl_e8_AVM_FAP_SettingState::tenType>(m_poClientVcan->u8GetFapSettingState(FapSettingType));

   ETG_TRACE_USR4(("   CA-AVM :: ........._vUpdateFapSetting(): Fap Setting Type %u Current State %u     ", ETG_CENUM(tenCsmFapSettingType, FapSettingType), oSetting.e8State.enType));

   FapSettings.push_back(oSetting);

   m_poServiceRvcIf->sendFAPSettingsStatus(FapSettings, AvmSettings);
}


/******************************************************************************/
/* FUNCTION     _vUpdateFapSetting                                            */
/******************************************************************************/
/**
*  \brief       Fetch only the updated FAP Setting Item and update to the clients
*
*  \param       Setting Type which is changed
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::_vUpdateFapSettingkey(tU32 u32Value)
{
	tU8 u8Key = ((u32Value & 0xFF00) >> 8 );
	tU8 u8Action = (u32Value & 0xFF);
	bool bkeycodeFound = false;

	ETG_TRACE_USR4(("   CA-AVM :: ............_vUpdateFapSetting(): Button %u  Action %u    ", \
			ETG_ENUM(DISPVIDCTRL_E8_IPA_BUTTON_ID, u8Key), ETG_ENUM(DISPVIDCTRL_E8_RVC_BUTTONSTATE, u8Action)));


	// mapping FAP Settings -- FAP Key Code
	for(tU8 u8Count = 0; u8Count < DISPVIDCTRL_FAP_KEY_CODE_TABLE_SIZE; u8Count++)
	{
		if(stCSMInputFAPKey[u8Count].u8IPAKeyID == (tU8)u8Key) {
			m_u8FAPKeyCode = stCSMInputFAPKey[u8Count].u8FAPKeyCode;
			bkeycodeFound = TRUE;
		}
	}

	if (m_poClientVcan && bkeycodeFound)
	{
		m_poClientVcan->vWriteIPA_SW_Operation_Status(m_u8FAPKeyCode);
	}
}
/******************************************************************************/
/* FUNCTION     _vUpdateAvmSetting                                            */
/******************************************************************************/
/**
*  \brief       Fetch only the updated AVM Setting Item and update to the clients
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::_vUpdateAvmSetting(const tenCsmFapAvmSettingType AvmSettingType) const
{
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poServiceRvcIf);
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poClientVcan);

   std::vector<rvc_fi_tcl_FAP_Settings_State, std::allocator<rvc_fi_tcl_FAP_Settings_State> > FapSettings;
   std::vector<rvc_fi_tcl_AVM_Settings_State, std::allocator<rvc_fi_tcl_AVM_Settings_State> > AvmSettings;

   rvc_fi_tcl_AVM_Settings_State oSetting;
   oSetting.e8Type.enType = static_cast<rvc_fi_tcl_e8_AVM_SettingItem::tenType>(AvmSettingType);
   oSetting.e8State.enType = static_cast<rvc_fi_tcl_e8_AVM_FAP_SettingState::tenType>(m_poClientVcan->u8GetAvmSettingState(AvmSettingType));

   ETG_TRACE_USR4(("   CA-AVM :: ........._vUpdateAvmSetting(): Avm Setting Type %u Current State %u     ", ETG_CENUM(tenCsmFapAvmSettingType, AvmSettingType), oSetting.e8State.enType));

   AvmSettings.push_back(oSetting);

   m_poServiceRvcIf->sendFAPSettingsStatus(FapSettings, AvmSettings);
}

/******************************************************************************/
/* FUNCTION     _vUpdateFapAvmSettings                                        */
/******************************************************************************/
/**
*  \brief       Fetch the FAP Settings and update them to the clients
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::_vUpdateFapAvmSettings(const tenCsmFapSettingType FapSettingType, const tenCsmFapAvmSettingType AvmSettingType) const
{
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poServiceRvcIf);
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poClientVcan);

   std::vector<rvc_fi_tcl_FAP_Settings_State, std::allocator<rvc_fi_tcl_FAP_Settings_State> > FapSettings;
   std::vector<rvc_fi_tcl_AVM_Settings_State, std::allocator<rvc_fi_tcl_AVM_Settings_State> > AvmSettings;

   // fetch the Fap Setting value
   rvc_fi_tcl_FAP_Settings_State oFapSetting;
   oFapSetting.e8Type.enType = static_cast<rvc_fi_tcl_e8_FAP_SettingItem::tenType>(FapSettingType);
   oFapSetting.e8State.enType = static_cast<rvc_fi_tcl_e8_AVM_FAP_SettingState::tenType>(m_poClientVcan->u8GetFapSettingState(FapSettingType));

   ETG_TRACE_USR4(("   CA-AVM :: ........._vUpdateFapAvmSettings(): Fap Setting Type %u Current State %u     ", ETG_CENUM(tenCsmFapSettingType, FapSettingType), oFapSetting.e8State.enType));

   FapSettings.push_back(oFapSetting);

   // fetch the Fap Avm Setting value
   rvc_fi_tcl_AVM_Settings_State oAvmSetting;
   oAvmSetting.e8Type.enType = static_cast<rvc_fi_tcl_e8_AVM_SettingItem::tenType>(AvmSettingType);
   oAvmSetting.e8State.enType = static_cast<rvc_fi_tcl_e8_AVM_FAP_SettingState::tenType>(m_poClientVcan->u8GetAvmSettingState(AvmSettingType));

   ETG_TRACE_USR4(("   CA-AVM :: ........._vUpdateFapAvmSettings(): Avm Setting Type %u Current State %u     ", ETG_CENUM(tenCsmFapAvmSettingType, AvmSettingType), oAvmSetting.e8State.enType));

   AvmSettings.push_back(oAvmSetting);

   m_poServiceRvcIf->sendFAPSettingsStatus(FapSettings, AvmSettings);
}

/******************************************************************************/
/* FUNCTION     _vUpdateFapAvmSettings                                        */
/******************************************************************************/
/**
*  \brief       Fetch the FAP Settings and update them to the clients
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::_vUpdateFapAvmSettings(tVoid) const
{
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poServiceRvcIf);
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poClientVcan);

   std::vector<rvc_fi_tcl_FAP_Settings_State, std::allocator<rvc_fi_tcl_FAP_Settings_State> > FapSettings;
   std::vector<rvc_fi_tcl_AVM_Settings_State, std::allocator<rvc_fi_tcl_AVM_Settings_State> > AvmSettings;

   // reserve memory to avoid reallocation of vector
   FapSettings.reserve(FAP_SETTING_TYPE__FAP_SETTING10);
   AvmSettings.reserve(FAP_SETTING_TYPE__AVM_SETTING09);

   // fetch all the Fap Setting values
   for ( tU8 i = FAP_SETTING_TYPE__FAP_SETTING_AVAILABLE; i <= FAP_SETTING_TYPE__FAP_SETTING10; i++ )
   {
      rvc_fi_tcl_FAP_Settings_State oSetting;
      oSetting.e8Type.enType = static_cast<rvc_fi_tcl_e8_FAP_SettingItem::tenType>(i);
      oSetting.e8State.enType = static_cast<rvc_fi_tcl_e8_AVM_FAP_SettingState::tenType>(m_poClientVcan->u8GetFapSettingState(i));

      ETG_TRACE_USR4(("   CA-AVM :: ........._vUpdateFapAvmSettings(): Fap Setting Type %u Current State %u     ", ETG_CENUM(tenCsmFapSettingType, i), oSetting.e8State.enType));

      FapSettings.push_back(oSetting);
   }

   // fetch all the Fap Avm Setting values
   for ( tU8 j = FAP_SETTING_TYPE__AVM_SETTING_AVAILABLE; j <= FAP_SETTING_TYPE__AVM_SETTING09; j++ )
   {
      rvc_fi_tcl_AVM_Settings_State oSetting;
      oSetting.e8Type.enType = static_cast<rvc_fi_tcl_e8_AVM_SettingItem::tenType>(j);
      oSetting.e8State.enType = static_cast<rvc_fi_tcl_e8_AVM_FAP_SettingState::tenType>(m_poClientVcan->u8GetAvmSettingState(j));

      ETG_TRACE_USR4(("   CA-AVM :: ........._vUpdateFapAvmSettings(): Avm Setting Type %u Current State %u     ", ETG_CENUM(tenCsmFapAvmSettingType, j), oSetting.e8State.enType));

      AvmSettings.push_back(oSetting);
   }

   m_poServiceRvcIf->sendFAPSettingsStatus(FapSettings, AvmSettings);
}

/******************************************************************************/
/* FUNCTION     _vUpdate_IPAConfiguration                                     */
/******************************************************************************/
/**
*  \brief       resets the toggle signals
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::_vUpdate_IPAConfiguration(tVoid)
{
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poClientVcan);
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poServiceRvcIf);
   
   ETG_TRACE_USR4((" CA-AVM: _vUpdate_IPAConfiguration(): m_u8IpaErrorCode %d", m_u8IpaErrorCode));
   if ( ( m_u8IPAStatus != DISPVIDCTRL_CONTROL_IPA__INITIALIZING ) && ( m_u8IPAStatus != DISPVIDCTRL_CONTROL_IPA__NOTAVAILABLE ) )
   {
      if ( m_u8IpaErrorCode > DISPVIDCTRL_CONTROL_IPA_VIDEO_SIGNAL ) {
          m_u8IPAStatus = DISPVIDCTRL_CONTROL_IPA__ERROR;
      } else if ( DISPVIDCTRL_CONTROL_IPA_VIDEO_SIGNAL & m_u8IpaErrorCode ) {
          m_u8IPAStatus = DISPVIDCTRL_CONTROL_IPA_VIDEO_SIGNAL_FAIL;
      } else { // No Error
          m_u8IPAStatus = DISPVIDCTRL_CONTROL_IPA__AVAILABLE;
      }
   }

   tU8 u8Region = rvc_fi_tcl_e8_IPA_RegionalSelection::FI_EN_INVALID;
   rvc_fi_tcl_e8_IPA_AvailabilityStatus oAvailabilityStatus;
   oAvailabilityStatus.enType = static_cast<rvc_fi_tcl_e8_IPA_AvailabilityStatus::tenType>(m_u8IPAStatus);

   rvc_fi_tcl_e8_IPA_RegionalSelection e8RegionalSelection;
   e8RegionalSelection.enType = static_cast<rvc_fi_tcl_e8_IPA_RegionalSelection::tenType>(u8Region);

   // check Region Selection signal
   if ( TRUE == m_poClientVcan->bGet_IPA_ConfigurationStatus(u8Region) )
   {
      e8RegionalSelection.enType = static_cast<rvc_fi_tcl_e8_IPA_RegionalSelection::tenType>(u8Region);
   }

   ETG_TRACE_USR1((" CA-AVM   :: _vUpdate_IPAConfiguration(): u8IPAOpStatus %d, u8Region %d", \
           ETG_CENUM(tenIPAStatus, m_u8IPAStatus), ETG_ENUM(DISPVIDCTRL_E8_IPA_REGIONALSELECTION, u8Region)));

   m_poServiceRvcIf->sendIPAConfigurationStatus(oAvailabilityStatus, e8RegionalSelection);
}

/******************************************************************************/
/* FUNCTION     _vUpdate_IPAStatus                                            */
/******************************************************************************/
/**
*  \brief       fetch the IPA status and updates it to HMI
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::_vUpdate_IPAStatus()
{
    DISPVIDCTRL_NULL_POINTER_CHECK(m_poClientVcan);
    DISPVIDCTRL_NULL_POINTER_CHECK(m_poServiceRvcIf);

    // in case of CAN Signal Timeout these values shall be updated to HALL
    tU8 u8IPAOpStatus = rvc_fi_tcl_e8_IPA_OperationStatus::FI_EN_MALFUNCTION;
    tU8 u8IPAModeRequest = rvc_fi_tcl_e8_IPA_ModeChangeRequest::FI_EN_NORMAL;

    rvc_fi_tcl_e8_IPA_OperationStatus e8OperationStatus;
    e8OperationStatus.enType = static_cast<rvc_fi_tcl_e8_IPA_OperationStatus::tenType>(u8IPAOpStatus);
    rvc_fi_tcl_e8_IPA_ModeChangeRequest e8Request;
    e8Request.enType = (static_cast<rvc_fi_tcl_e8_IPA_ModeChangeRequest::tenType>(u8IPAModeRequest));

    if ( TRUE == m_poClientVcan->bGet_IPA_OperationStatus(u8IPAOpStatus, u8IPAModeRequest) )
    {
        e8OperationStatus.enType = static_cast<rvc_fi_tcl_e8_IPA_OperationStatus::tenType>(u8IPAOpStatus);
        e8Request.enType = static_cast<rvc_fi_tcl_e8_IPA_ModeChangeRequest::tenType>(u8IPAModeRequest);
    }

    ETG_TRACE_USR1((" CA-AVM   :: _vUpdate_IPAStatus(): u8IPAOpStatus %d, u8IPAModeRequest %d", \
            ETG_ENUM(DISPVIDCTRL_E8_IPA_OPERATIONSTATUS, u8IPAOpStatus), ETG_ENUM(DISPVIDCTRL_E8_IPA_MODECHANGEREQUEST, u8IPAModeRequest)));

    // update it to the HMI
    m_poServiceRvcIf->sendIPAStatusStatus(e8OperationStatus, e8Request);
}

/******************************************************************************/
/* FUNCTION     _vUpdate_IPASwitches                                          */
/******************************************************************************/
/**
*  \brief       fetch the IPA Switch status and update to HMI
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::_vUpdate_IPASwitches(tVoid)
{
    DISPVIDCTRL_NULL_POINTER_CHECK(m_poClientVcan);
    DISPVIDCTRL_NULL_POINTER_CHECK(m_poServiceRvcIf);

    tU8 u8SwitchesStatus = 0x00;
    tU8 u8SwitchMaskStatus = rvc_fi_tcl_e8_IPA_SwitchMask::FI_EN_NOMASK;
    rvc_fi_tcl_e8_IPA_SwitchMask e8SwitchMask;
    e8SwitchMask.enType = static_cast<rvc_fi_tcl_e8_IPA_SwitchMask::tenType>(u8SwitchMaskStatus);

    // fetch the data
    (tVoid) m_poClientVcan->bGet_IPA_SwitchesSignals(u8SwitchesStatus);
    if ( TRUE == m_poClientVcan->bGet_IPA_SwitchMaskStatus(u8SwitchMaskStatus) )
    {
        e8SwitchMask.enType = static_cast<rvc_fi_tcl_e8_IPA_SwitchMask::tenType>(u8SwitchMaskStatus);
    }

    ETG_TRACE_USR1((" CA-AVM   :: _vUpdate_IPASwitches(): Switch Mask Status is %d", ETG_ENUM(DISPVIDCTRL_E8_IPA_SWITCHMASK, u8SwitchMaskStatus)));

    // update to registered clients
    m_poServiceRvcIf->sendIPASwitchGreyOutRequestStatus(u8SwitchesStatus);
    m_poServiceRvcIf->sendIPASwitchMaskRequestStatus(e8SwitchMask);
}

/******************************************************************************/
/* FUNCTION     _vUpdate_IPARequests                                          */
/******************************************************************************/
/**
*  \brief       resets the toggle signals
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::_vUpdate_IPARequests(tVoid)
{
    DISPVIDCTRL_NULL_POINTER_CHECK(m_poClientVcan);
    DISPVIDCTRL_NULL_POINTER_CHECK(m_poServiceRvcIf);

    tU8 u8MessageDisplayValue = 0x00; // no message displayed
    tU8 u8PatternId = 0xFF; // no pattern displayed

    rvc_fi_tcl_e8_IPA_PatternID e8DisplayPattern;
    e8DisplayPattern.enType = static_cast<rvc_fi_tcl_e8_IPA_PatternID::tenType>(u8PatternId); // reserved value, no pattern
    rvc_fi_tcl_e8_IPA_MessageID e8MessageID;
    e8MessageID.enType = (rvc_fi_tcl_e8_IPA_MessageID::FI_EN_NO_MESSAGE_DISPLAYED);

    if ( TRUE == m_poClientVcan->bGet_IPA_PatternDisplayStatus(u8PatternId) )
        e8DisplayPattern.enType = static_cast<rvc_fi_tcl_e8_IPA_PatternID::tenType>(u8PatternId);

    if ( TRUE == m_poClientVcan->bGet_IPA_MessageDisplaySignals(u8MessageDisplayValue) )
    {
       switch (u8MessageDisplayValue)
       {
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_NO_MESSAGE_DISPLAYED                         :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_ENSURE_IT_IS_SAFE_BEFORE_MANEUVERING         :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_STOP_BESIDE_PARK_SLOT_TO_START_PARK_ASSIST   :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_DRIVE_FORWARD                                :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_REDUCE_SPEED                                 :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_STOP_AND_SELECT_REVERSE_GEAR                 :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_STOP_AND_SELECT_DRIVE_GEAR                   :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_DRIVE_REARWARD                               :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_STOP_NEXT_TO_EMPTY_PARKING_POSITION          :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_PRESS_ADJUST_TO_SET_PARKING_POSITION         :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_ADJUST_PARK_POSITION                         :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_PARK_ASSIST_FINISHED                         :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_SELECT_DRIVE_GEAR                            :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_CENTER_STEERING_WHEEL                        :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_PARK_ASSIST_FAULT                            :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_TURN_SIGNAL_TO_SWITCH_SIDE                   :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_PARK_ASSIST_UNAVAILABLE                      :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_SPACE_FOUND                                  :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_STOP                                         :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_SELECT_REVERSE_GEAR                          :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_DRIVE_REARWARD_1                             :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_REDUCE_SPEED_1                               :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_DRIVE_FORWARD_1                              :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_PARK_ASSIST_CANCELLED                        :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_PARK_ASSIST_FINISHED_1                       :
          case rvc_fi_tcl_e8_IPA_MessageID::FI_EN_PARK_ASSIST_CANCELLED_1                      :
             {
                e8MessageID.enType = static_cast<rvc_fi_tcl_e8_IPA_MessageID::tenType>(u8MessageDisplayValue);
             }
             break;

          default:
             break;
       }
    }

    ETG_TRACE_USR1((" CA-AVM   :: _vUpdate_IPARequests(): PatternID %u MessageDisplayValue %u", \
            ETG_ENUM(DISPVIDCTRL_E8_IPA_PATTERNID, u8PatternId), ETG_ENUM(DISPVIDCTRL_E8_IPA_MESSAGEID, u8MessageDisplayValue)));

    m_poServiceRvcIf->sendIPAPatternRequestStatus(e8DisplayPattern);
    m_poServiceRvcIf->sendIPAMessageRequestStatus(e8MessageID);
}

/******************************************************************************/
/* FUNCTION     _vUpdate_IPADirectionIndicator                                */
/******************************************************************************/
/**
*  \brief       resets the toggle signals
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::_vUpdate_IPADirectionIndicator(tVoid)
{
    DISPVIDCTRL_NULL_POINTER_CHECK(m_poClientVcan);
    DISPVIDCTRL_NULL_POINTER_CHECK(m_poServiceRvcIf);

    tU8 u8IPADirectionInd;
    rvc_fi_tcl_e8_IPA_DirectionIndicator eDirection;

    if ( m_poClientVcan->bGet_IPA_DirectionStatus(u8IPADirectionInd) )
    {
        eDirection.enType = static_cast<rvc_fi_tcl_e8_IPA_DirectionIndicator::tenType>(u8IPADirectionInd);
    }

    ETG_TRACE_USR1((" CA-AVM   :: _vUpdate_IPADirectionIndicator(): Direction is %d", ETG_ENUM(DISPVIDCTRL_E8_IPA_DIRECTIONINDICATOR, u8IPADirectionInd)));

    m_poServiceRvcIf->sendIPADirectionIndicatorStatus(eDirection);
}

/******************************************************************************/
/* FUNCTION     _pfCallbackTimer_AvmErrorConfirm                              */
/******************************************************************************/
/**
*  \brief
*
*  \param       pArg - pointer to this class
*  \return      none
*/
/******************************************************************************/
OSAL_tpfCallback dispvidctrl_tclControl_Avm::_pfCallbackTimer_AvmErrorConfirm(tVoid* pArg)
{
   ETG_TRACE_USR4((" CA-AVM   :: _pfCallbackTimer_AvmErrorConfirm(): ... ====>>                                  "));

   dispvidctrl_tclControl_Avm* po_tclControl_Avm = (dispvidctrl_tclControl_Avm*)pArg;
   if (po_tclControl_Avm)
   {
      if ((po_tclControl_Avm->_cpoMain) && (po_tclControl_Avm->_cpoMain->theServer()))
      {
         // post loop-back message to control _Avm
         const tChar strIfName[] = "dispvidctrl_tclControl_Avm";
         po_tclControl_Avm->_cpoMain->theServer()->vPostMsgControlAvm(strIfName, EN_CONTROL_AVM__ERROR_CONFIRM_TIMEOUT);
      }
   }
   return 0;
};

/******************************************************************************/
/* FUNCTION     _bSetTimer_AvmErrorConfirm()                                  */
/******************************************************************************/
/**
*  \brief       set/reset the timer for IPA malfunction
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Avm::_bSetTimer_AvmErrorConfirm(OSAL_tMSecond msTimeout)
{
    ETG_TRACE_USR4((" CA-AVM   :: _bSetTimer_AvmErrorConfirm(): msTimeout %u", msTimeout));

    tBool bSuccess = TRUE;
    if ( OSAL_C_INVALID_HANDLE != m_hTimerHandle_AvmErrorConfirm )
    {
        if ( OSAL_s32TimerSetTime( m_hTimerHandle_AvmErrorConfirm, msTimeout, 0 ) != OSAL_OK )
        {
            bSuccess = FALSE;
        }
    }
    return(bSuccess);
}

/******************************************************************************/
/* FUNCTION     vTraceInfo()                                                  */
/******************************************************************************/
/**
*  \brief       trace information
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Avm::vTraceInfo()
{
   ETG_TRACE_FATAL((" CA-AVM   :: vTraceInfo()    : ****** CA-AVM states ******************************************"));
   ETG_TRACE_FATAL((" CA-AVM   :: Avm Mode is ............................ %-40u", m_bAvmMode));
   ETG_TRACE_FATAL((" CA-AVM   :: Avm Mode with IPA is ................... %-40u", m_bAvmTypeIpa));
   ETG_TRACE_FATAL((" CA-AVM   :: Avm Mode with FAP is ................... %-40u", m_bAvmTypeFap));
   ETG_TRACE_FATAL((" CA-AVM   :: FAP Mode is Active ..................... %-40u", m_bFapModeActive));
   ETG_TRACE_FATAL((" CA-AVM   :: Ignition is ............................ %-40u", m_bIgnition));
   ETG_TRACE_FATAL((" CA-AVM   :: Ignition is Valid ...................... %-40u", m_bIgnitionStateValid));
   ETG_TRACE_FATAL((" CA-AVM   :: m_bTimerErrorConfirmRunning ............ %-40u", m_bTimerErrorConfirmRunning));
   ETG_TRACE_FATAL((" CA-AVM   :: m_bTimerErrorConfirmRunning ............ %-40u", m_bIpaErrorHandlerON));
   ETG_TRACE_FATAL((" CA-AVM   :: FSM Control_Avm : Super State .......... %-40u", ETG_CENUM(tenFsmControl_Avm_SuperState, m_enFsmControl_Avm_SuperState)));
   ETG_TRACE_FATAL((" CA-AVM   :: CanMsgTimeOutState is .................. 0x%-38x", m_bCanMsgTimeOutState));
   ETG_TRACE_FATAL((" CA-AVM   :: m_u32HmiLanguage ....................... %-40u", m_u32HmiLanguage));
   ETG_TRACE_FATAL((" CA-AVM   :: m_u8AvmLanguage ........................ %-40u", m_u8AvmLanguage));
   ETG_TRACE_FATAL((" CA-AVM   :: m_u8IPAStatus  ......................... %-40u", ETG_CENUM(tenIPAStatus, m_u8IPAStatus)));
   ETG_TRACE_FATAL((" CA-AVM   :: *********************************************************************************"));
}

