/******************************************************************************/
/**
* \file    dispvidctrl_tclControl.cpp
* \ingroup
*
* \brief
*
* \remark  Copyright : (c) 2015 Robert Bosch GmbH, Hildesheim
* \remark  Author    : Michael Niemann
* \remark  Scope     : AIVI
*
* \todo
*/
/******************************************************************************/

/*******************************************************************************
                        Includes
*******************************************************************************/
#include "dispvidctrl_tclControl.h"
#include "dispvidctrl_tclFsm_OnOff.h"
#include "dispvidctrl_tclControl_Avm.h"
#include "dispvidctrl_tclControl_Hmi.h"
#include "dispvidctrl_tclControl_HmiAvm3.h"
#include "dispvidctrl_tclControl_Rvc.h"
#include "dispvidctrl_tclControl_Video.h"
#include "I_dispvidctrl_ServiceRvcIf.h"
#include "I_dispvidctrl_ClientSpmIf.h"
#include "I_dispvidctrl_tclSyncHandler.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_APPLICATION
#include "trcGenProj/Header/dispvidctrl_tclControl.cpp.trc.h"
#endif

/*******************************************************************************
                        Defines
*******************************************************************************/

#define DISPVIDCTRL_CONTROL__MAX_NO_OF_RETRY                          5
#define DISPVIDCTRL_CONTROL__WAIT_BEFORE_RETRY_TIME_IN_MS          2000

/******************************************************************************/
/* FUNCTION     dispvidctrl_tclControl()                                      */
/******************************************************************************/
/**
*  \brief       default constructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclControl::dispvidctrl_tclControl()
: dispvidctrl_tclBaseIf(OSAL_NULL)
, m_hTimerHandle_RetryMechanism(OSAL_C_INVALID_HANDLE)
, m_poFsm_OnOff(OSAL_NULL)
, m_poControl_Avm(OSAL_NULL)
, m_poControl_Rvc(OSAL_NULL)
, m_poControl_Video(OSAL_NULL)
, m_poRvcServiceIf(OSAL_NULL)
, m_poSpmClientIf(OSAL_NULL)
//------------------------------ system (KDS) configuration info
, m_Config_bAvmMode(FALSE)
, m_Config_bRvcMode(FALSE)
, m_Config_bAvmType_Phase3(FALSE)
, m_u8NoOfRetriesExecuted(0)
, m_tU32VideoStatus(OSAL_NULL)
{
}


/******************************************************************************/
/* FUNCTION     dispvidctrl_tclControl()                                      */
/******************************************************************************/
/**
*  \brief       constructor
*
*  \param       pointer to main application
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclControl::dispvidctrl_tclControl(const dispvidctrl_tclAppMain* poMainAppl)
: dispvidctrl_tclBaseIf(poMainAppl)
, m_hTimerHandle_RetryMechanism(OSAL_C_INVALID_HANDLE)
, m_poFsm_OnOff(OSAL_NULL)
, m_poControl_Avm(OSAL_NULL)
, m_poControl_Rvc(OSAL_NULL)
, m_poControl_Video(OSAL_NULL)
, m_poRvcServiceIf(OSAL_NULL)
, m_poSpmClientIf(OSAL_NULL)
//------------------------------ system (KDS) configuration info
, m_Config_bAvmMode(FALSE)
, m_Config_bRvcMode(FALSE)
, m_Config_bAvmType_Phase3(FALSE)
, m_u8NoOfRetriesExecuted(0)
, m_tU32VideoStatus(OSAL_NULL)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl() entered."));

   // create new FSM OnOff
   m_poFsm_OnOff = dispvidctrl_tclFsm_OnOff::tclCreateInstance(this);
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poFsm_OnOff);
   
   // create Retry Mechanism timer(for 2 seconds)
   // this is a generic timer needed for any sort of retry mechanism in the parkassist area (eg. i2c comm fail, elmos init fail, video source file open error .. etc.. )
   if ( OSAL_s32TimerCreate((OSAL_tpfCallback) _pfCallbackTimer_RetryMechanism, (tVoid*) this, &m_hTimerHandle_RetryMechanism) != OSAL_OK)
   {
      //FATAL_M_ASSERT_ALWAYS();
      ETG_TRACE_USR4((" dispvidctrl_tclControl :: E R R O R in .......... Retry Mechanism timer creation fails!"));
      m_hTimerHandle_RetryMechanism = OSAL_C_INVALID_HANDLE;
   }
}


/******************************************************************************/
/* FUNCTION     ~dispvidctrl_tclControl                                       */
/******************************************************************************/
/**
*  \brief       destructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclControl::~dispvidctrl_tclControl(tVoid)
{
   ETG_TRACE_USR4(("~dispvidctrl_tclControl() entered."));

   // de-initialise/close initialised parts
   m_poFsm_OnOff     = OSAL_NULL;
   m_poControl_Avm   = OSAL_NULL;
   m_poControl_Rvc   = OSAL_NULL;
   m_poControl_Video = OSAL_NULL;
   m_poRvcServiceIf  = OSAL_NULL;
   m_poSpmClientIf   = OSAL_NULL;
   
   if (OSAL_C_INVALID_HANDLE != m_hTimerHandle_RetryMechanism)
   {
      (void) OSAL_s32TimerSetTime(m_hTimerHandle_RetryMechanism, 0, 0);
      (void) OSAL_s32TimerDelete(m_hTimerHandle_RetryMechanism);
   }
}

/******************************************************************************/
/* FUNCTION     vGetConfiguration                                             */
/******************************************************************************/
/**
*  \brief       Function to get all configuration values needed by this class.
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl::vGetConfiguration(const TConfiguration* pStConfigurationValues)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl::vGetConfiguration() entered."));

   m_Config_bAvmMode = pStConfigurationValues->bAvmMode;
   m_Config_bRvcMode = pStConfigurationValues->bRvcMode;
   m_Config_bAvmType_Phase3 = pStConfigurationValues->bAvmType_Phase3;
   
}

/******************************************************************************/
/* FUNCTION     vGetReferences                                                */
/******************************************************************************/
/**
*  \brief       Function to get all reference needed by this class.
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl::vGetReferences(tVoid)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl::vGetReferences() entered."));

   m_poControl_Avm = dynamic_cast<dispvidctrl_tclControl_Avm*>(_cpoMain->getHandler("dispvidctrl_tclControl_Avm"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poControl_Avm);

   m_poControl_Rvc = dynamic_cast<dispvidctrl_tclControl_Rvc*>(_cpoMain->getHandler("dispvidctrl_tclControl_Rvc"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poControl_Rvc);

   m_poControl_Video = dynamic_cast<dispvidctrl_tclControl_Video*>(_cpoMain->getHandler("dispvidctrl_tclControl_Video"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poControl_Video);

   m_poRvcServiceIf = dynamic_cast<Idispvidctrl_tclServiceRvcIf*>(_cpoMain->getHandler("Idispvidctrl_tclServiceRvcIf"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poRvcServiceIf);

   m_poSpmClientIf = dynamic_cast<Idispvidctrl_tclClientSpmIf*>(_cpoMain->getHandler("Idispvidctrl_tclClientSpmIf"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poSpmClientIf);
}

/******************************************************************************/
/* FUNCTION     vStartCommunication                                           */
/******************************************************************************/
/**
*  \brief       Function to start all dynamic objects e.g. threads, ...
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl::vStartCommunication(tVoid)
{
   spm_fi_tcl_SPM_e32_CVM_EVENT rCvmEvent;
   tU32 tU32ControlEvent = (tU32) EN_CONTROL__NONE;
   tU32 tU32VoltageLevel = 0;

   ETG_TRACE_USR4(("dispvidctrl_tclControl::vStartCommunication() entered."));

   // cca eTrigger_APPLICATION_STATE_NORMAL is set by default
   if (m_poFsm_OnOff)
   {
      m_poFsm_OnOff->vNewTrigger((tU32) dispvidctrl_tclFsm_OnOff::eTrigger_APPLICATION_STATE_NORMAL, TRUE);
   }

   if (m_poSpmClientIf)
   {
      m_poSpmClientIf->getCvmEventStatus(rCvmEvent);
      if(m_poSpmClientIf->bMapCvmEventStatus(rCvmEvent, tU32ControlEvent, tU32VoltageLevel))
      {
         if (m_poFsm_OnOff)
         {
            if ((tenControl) tU32ControlEvent == EN_CONTROL__CRITICAL_VOLTAGE_END)
            {
               m_poFsm_OnOff->vNewTrigger((tU32) dispvidctrl_tclFsm_OnOff::eTrigger_CRITICAL_VOLTAGE_END, TRUE);
            }
            if ((tenControl) tU32ControlEvent == EN_CONTROL__CRITICAL_VOLTAGE_START)
            {
               m_poFsm_OnOff->vNewTrigger((tU32) dispvidctrl_tclFsm_OnOff::eTrigger_CRITICAL_VOLTAGE_START, TRUE);
            }
         }
      }
   }

   // post start trigger
   tChar strIfName[] = "dispvidctrl_tclControl";
   dispvidctrl_tclAppMain::theServer()->vPostMsgControl(strIfName, EN_CONTROL__START);
   // For re-checking if the video source is found, elmos initialized etc..
   dispvidctrl_tclAppMain::theServer()->vPostMsgControl(strIfName, EN_CONTROL__RETRY_MECHANISM_TIMEOUT);
}

/******************************************************************************/
/* FUNCTION     vHandleMessage(TMsg* pMsg)                                    */
/******************************************************************************/
/**
*  \brief       Handle worker events.
*               Forward to vHandleControl to handle all input events in one method.
*
*  \param       message pointer
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl::vHandleMessage(dispvidctrl_tclBaseIf::TMsg* pMsg)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl::vHandleMessage() entered %u -> data: %d.", ETG_CENUM(dispvidctrl_tclBaseIf::ECmdTypes , (tU32)pMsg->eCmd), pMsg->u.u32Data));

   switch (pMsg->eCmd)
   {
      case dispvidctrl_tclBaseIf::eControl:
         vHandleControl((tenControl) pMsg->u.u32Data);
         break;
      case dispvidctrl_tclBaseIf::eFsmOnOff:
         vHandleControl(EN_CONTROL__FSM_ONOFF_INPUT_EVENT, pMsg->u.u32Data);
         break;

      default:
         break;
   }
}


/******************************************************************************/
/* FUNCTION     vHandleControl()                                              */
/******************************************************************************/
/**
*  \brief       direct call interface
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl::vHandleControl(tenControl enControl, tU32 u32Value)
{
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32Value );

   ETG_TRACE_USR4(("  CONTROL :: ............vHandleControl(): %d  u32Value = %u", ETG_CENUM(tenControl, enControl), u32Value));

   switch (enControl)
   {
      case EN_CONTROL__START:
         if (m_poFsm_OnOff) {m_poFsm_OnOff->vNewTrigger((tU32) dispvidctrl_tclFsm_OnOff::eTrigger_START, TRUE);}
         break;
      case EN_CONTROL__HMI_AVAILABLE:
         if ((tBool) u32Value == TRUE) {
            if (m_poFsm_OnOff) {m_poFsm_OnOff->vNewTrigger((tU32) dispvidctrl_tclFsm_OnOff::eTrigger_HMI_AVAILABLE_TRUE, TRUE);}
         } else {
            if (m_poFsm_OnOff) {m_poFsm_OnOff->vNewTrigger((tU32) dispvidctrl_tclFsm_OnOff::eTrigger_HMI_AVAILABLE_FALSE, TRUE);}
         }
         break;
      case EN_CONTROL__VIDEO_STATUS_RECEIVED:
         {
            m_tU32VideoStatus = u32Value & DISPVIDCTRL_VIDEO__MASK_VIDEO_CAPTURING_STATUS;
            if (m_poFsm_OnOff) {m_poFsm_OnOff->vNewTrigger((tU32) dispvidctrl_tclFsm_OnOff::eTrigger_VIDEO_STATUS_RECEIVED, TRUE);}
         }
         break; 
      case EN_CONTROL__CRITICAL_VOLTAGE_START:
         if (m_poFsm_OnOff) {m_poFsm_OnOff->vNewTrigger((tU32) dispvidctrl_tclFsm_OnOff::eTrigger_CRITICAL_VOLTAGE_START, TRUE);}
         break;
      case EN_CONTROL__CRITICAL_VOLTAGE_END:
         if (m_poFsm_OnOff) {m_poFsm_OnOff->vNewTrigger((tU32) dispvidctrl_tclFsm_OnOff::eTrigger_CRITICAL_VOLTAGE_END, TRUE);}
         break;
      case EN_CONTROL__APPLICATION_STATE_NORMAL:
         if (m_poFsm_OnOff) {m_poFsm_OnOff->vNewTrigger((tU32) dispvidctrl_tclFsm_OnOff::eTrigger_APPLICATION_STATE_NORMAL, TRUE);}
         break;
      case EN_CONTROL__APPLICATION_STATE_NOT_NORMAL:
         if (m_poFsm_OnOff) {m_poFsm_OnOff->vNewTrigger((tU32) dispvidctrl_tclFsm_OnOff::eTrigger_APPLICATION_STATE_NOT_NORMAL, TRUE);}
         break;
      case EN_CONTROL__RETRY_MECHANISM_TIMEOUT:
         {
            tBool bForce = FALSE;
            if ( u32Value == 0x01 )
            {
               bForce = TRUE;
            }
            _vHandleRetryMechanismTimeout(bForce);
         }
         break;
      //-------------------------------------------------------------------------------------------
      case EN_CONTROL__FSM_ONOFF_INPUT_EVENT:
         if (m_poFsm_OnOff) {m_poFsm_OnOff->vNewTrigger(u32Value, TRUE);}
         break;
      //-------------------------------------------------------------------------------------------
      default:
         break;
   }
}


/******************************************************************************/
/* FUNCTION     vHandleFsmOutEvent_OnOff()                                    */
/******************************************************************************/
/**
*  \brief       handle function to post output events from FSM OnOff
*
*  \param       event
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl::vHandleFsmOutEvent_OnOff(tenFsmOutEvent_OnOff enFsmOutEvent_OnOff, tU32 u32Value)
{
   // ...FSM_OUT__AVM...     // input events for FSM AVM (Loopback to FSM)
   // ...FSM_OUT__HMI...     // input events for FSM HMI

   ETG_TRACE_USR4(("  CONTROL :: vHandleFsmOutEvent_OnOff(): %d    u32Value = %u", ETG_CENUM(tenFsmOutEvent_OnOff, enFsmOutEvent_OnOff), u32Value));

   switch (enFsmOutEvent_OnOff)
   {
      case FSM_ONOFF_OUT__FSM_ONOFF_INPUT_EVENT:
         {
            tChar strIfName[] = "dispvidctrl_tclControl";
            // post loop-back message to FSM OnOff, u32Value will be eTrigger... value
            dispvidctrl_tclAppMain::theServer()->vPostMsgFsmOnOff(strIfName, u32Value);
         }
         break;
      case FSM_ONOFF_OUT__HMI_AVAILABLE:
         {
            if (m_Config_bAvmType_Phase3) {
                dispvidctrl_tclAppMain::theServer()->vPostMsgControlHmiAvm3("dispvidctrl_tclControl_HmiAvm3", (tU32) EN_CONTROL_HMIAVM3__HMI_AVAILABLE);
            } else {
                dispvidctrl_tclAppMain::theServer()->vPostMsgControlHmi    ("dispvidctrl_tclControl_Hmi",     (tU32) EN_CONTROL_HMI__HMI_AVAILABLE);
            }
         }
         break;
      case FSM_ONOFF_OUT__ENABLE_CONTROLS:
         {
            // post broadcast message to all dispvidctrl controls
            tChar strIfName[] = "";
            dispvidctrl_tclAppMain::theServer()->vPostMsgControlModeEnabled(strIfName, 0);
         }
         break;
      case FSM_ONOFF_OUT__ENABLE_CONTROLS_INITIAL_WITH_EARLY_VIDEO_STATUS:
         {
            // post broadcast message to all dispvidctrl controls
            tChar strIfName[] = "";
            tU32 tU32VideoStatusEarly = m_tU32VideoStatus;
            tBool bAvmInitFlagTx = TRUE;
            I_dispvidctrl_tclSyncHandler* _poSyncHandler = dynamic_cast<I_dispvidctrl_tclSyncHandler*>(_cpoMain->getHandler("I_dispvidctrl_tclSyncHandler"));
            if (_poSyncHandler)
            {
               _poSyncHandler->vGetAvmInitFlagTx(bAvmInitFlagTx);
            }

            tU32VideoStatusEarly |= DISPVIDCTRL_VIDEO__MASK_EARLY_ACTIV;
            if (bAvmInitFlagTx)
            {
               tU32VideoStatusEarly |= DISPVIDCTRL_VIDEO__MASK_AVM_INIT_FLAG_TX;
            }

            dispvidctrl_tclAppMain::theServer()->vPostMsgControlModeEnabled(strIfName, tU32VideoStatusEarly);
         }
         break;
      case FSM_ONOFF_OUT__DISABLE_CONTROLS:
         {
            // post broadcast message to all dispvidctrl controls
            tChar strIfName[] = "";
            dispvidctrl_tclAppMain::theServer()->vPostMsgControlModeDisabled(strIfName, u32Value);
         }
         break;
      case FSM_ONOFF_OUT__EARLY_VIDEO_CONTROL_OFF:
         {
            {
               // NCG3D-10438: Synchronous handling of VideoStatus event to FSM OnOff
               tChar strIfName[] = "dispvidctrl_tclControl_Video";
               dispvidctrl_tclAppMain::theServer()->vPostMsgControlVideo(strIfName, (tU32) EN_CONTROL_VIDEO__EARLY_VIDEO_CONTROL_OFF);
            
               //if (m_poControl_Video) {
               //   m_poControl_Video->vHandleControl(EN_CONTROL_VIDEO__EARLY_VIDEO_CONTROL_OFF);
               //}
            }
            {
               // takeover of Camera screen activation/deactivation from early control
               // we have to set eSyncTriggerDrvAssCtrl within SyncFsm (SyncHandler)
               tChar strIfName[] = "I_dispvidctrl_tclSyncHandler";
               dispvidctrl_tclAppMain::theServer()->vPostMsgControlSyncHandler(strIfName, EN_CONTROL_SH__EARLY_CAMERA_SCREEN_CONTROL_OFF);
            }
         }
         break;
      default:
         break;
   }
}

/******************************************************************************/
/* FUNCTION     _pfCallbackTimer_RetryMechanism                               */
/******************************************************************************/
/**
*  \brief       Timer expired now post a loopback message to recheck the 
*               required conditions again.
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
OSAL_tpfCallback dispvidctrl_tclControl::_pfCallbackTimer_RetryMechanism(tVoid* pArg)
{
   //(tVoid) pArg;

   ETG_TRACE_USR4(("  CONTROL :: _pfCallbackTimer_RetryMechanism(): ... ====>>                 "));

   dispvidctrl_tclControl* po_tclControl = (dispvidctrl_tclControl*) pArg;
   if (po_tclControl)
   {
      if ((po_tclControl->_cpoMain) && (po_tclControl->_cpoMain->theServer()))
      {
         // post loop-back message to control
         tChar strIfName[] = "dispvidctrl_tclControl";
         po_tclControl->_cpoMain->theServer()->vPostMsgControl(strIfName, EN_CONTROL__RETRY_MECHANISM_TIMEOUT);
      }
   }
   return 0;
}

/******************************************************************************/
/* FUNCTION     _bSetTimer_RetryMechanism                                     */
/******************************************************************************/
/**
*  \brief       Set a timer after which we can recheck the needed states
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tBool dispvidctrl_tclControl::_bSetTimer_RetryMechanism(OSAL_tMSecond msTimeout)
{
    tBool bSuccess = TRUE;
    if ( OSAL_C_INVALID_HANDLE != m_hTimerHandle_RetryMechanism )   {
        if ( OSAL_s32TimerSetTime( m_hTimerHandle_RetryMechanism, msTimeout, 0 ) != OSAL_OK )
        {
            bSuccess = FALSE;
        }
    }
    return(bSuccess);
}

/******************************************************************************/
/* FUNCTION     _vHandleRetryMechanismTimeout                                 */
/******************************************************************************/
/**
*  \brief       handle TTFis commands
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl::_vHandleRetryMechanismTimeout(const tBool bForceRetry)
{
   tBool bRetryNeeded = FALSE;   
   
   // Force the retry mechanism again ?
   if ( TRUE == bForceRetry )
   {
      m_u8NoOfRetriesExecuted = 0;
   }

   // re-check for the Video Device
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poControl_Video);
   if ( FALSE == m_poControl_Video->bIsVideoSourceFound() )
   {
      if ( FALSE == m_poControl_Video->bFindVideoSource() )
      {
         ETG_TRACE_FATAL(("DISPVIDCTRL :: Rear View Camera Video Device Not found!!!"));
         bRetryNeeded = TRUE;
      }
      else
      {
         ETG_TRACE_FATAL(("DISPVIDCTRL :: Rear View Camera Video Device found"));
      }
   }
   
   if ( bRetryNeeded )
   {
      if ( m_u8NoOfRetriesExecuted < DISPVIDCTRL_CONTROL__MAX_NO_OF_RETRY )
      {
         m_u8NoOfRetriesExecuted++;
         _bSetTimer_RetryMechanism(DISPVIDCTRL_CONTROL__WAIT_BEFORE_RETRY_TIME_IN_MS);
         ETG_TRACE_ERRMEM(("DISPVIDCTRL :: Rear View Camera Video Device Not Found!!! m_u8NoOfRetriesExecuted %d. Retrying again in 2s...", m_u8NoOfRetriesExecuted));
      }
      else
      {
         ETG_TRACE_ERRMEM(("DISPVIDCTRL :: Rear View Camera Video Device Not Found!!! m_u8NoOfRetriesExecuted %d. Max retries over", m_u8NoOfRetriesExecuted));
      }
   }
}

//##############################################################################
// TTFis
//##############################################################################
/******************************************************************************/
/* FUNCTION     vHandleTraceMessage                                           */
/******************************************************************************/
/**
*  \brief       handle TTFis commands
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl::vHandleTraceMessage(const tUChar* puchData)
{
   DISPVIDCTRL_NULL_POINTER_CHECK(puchData);

   tU32 u32MsgCode = ((puchData[1]<<8) | puchData[2]);
   ETG_TRACE_USR4(("dispvidctrl_tclControl::vHandleTraceMessage(): trace command %d", u32MsgCode ));

   switch (u32MsgCode)
   {
      case DISPVIDCTRL_DISPVIDCTRL_STIMULATE_CONTROL:
      {
         // Stimulate tclControl, Parameter = control name and u32Data value)
         tU8 param1 = puchData[3];
         tU32 param2 = ((tU32)puchData[4] << 24) | ((tU32)puchData[5] << 16) | ((tU32)puchData[6] <<  8) | (tU32)puchData[7];

         vHandleControl((tenControl) param1, param2);
      }
      break;
      case DISPVIDCTRL_DISPVIDCTRL_STIMULATE_FSM_ONOFF:
      {
         // Stimulate FSM OnOff, Parameter = trigger)
         tU32 param1 = ((tU32)puchData[3] << 24) | ((tU32)puchData[4] << 16) | ((tU32)puchData[5] <<  8) | (tU32)puchData[6];

         vHandleControl(EN_CONTROL__FSM_ONOFF_INPUT_EVENT, param1);
      }
      break;

      default:
      break;
   }
}


/******************************************************************************/
/* FUNCTION     vTraceInfo()                                                  */
/******************************************************************************/
/**
*  \brief       trace information
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl::vTraceInfo()
{
   ETG_TRACE_FATAL(("  CONTROL :: *********** Control states ******************************************************"));
   ETG_TRACE_FATAL(("  CONTROL :: m_Config_bAvmMode ... ..................... %-40u", m_Config_bAvmMode));
   ETG_TRACE_FATAL(("  CONTROL :: m_Config_bRvcMode ......................... %-40u", m_Config_bRvcMode));
   ETG_TRACE_FATAL(("  CONTROL :: Video Capturing State  .................... %-40u", ETG_CENUM(tenVideoStatus, (m_tU32VideoStatus & DISPVIDCTRL_VIDEO__MASK_VIDEO_CAPTURING_STATUS))));
   ETG_TRACE_FATAL(("  CONTROL :: m_u8NoOfRetriesExecuted ................... %-40u", m_u8NoOfRetriesExecuted));
   ETG_TRACE_FATAL(("  CONTROL :: *********************************************************************************"));
}

