/******************************************************************************/
/**
* \file    dispvidctrl_tclAudioRouting.cpp
* \ingroup
*
* \brief
*
* \remark  Copyright : (c) 2015 Robert Bosch GmbH, Hildesheim
* \remark  Author    : deo2kor
* \remark  Scope     : AIVI
*
* \todo
*/
/******************************************************************************/

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#include "dispvidctrl_tclAudioRouting.h"
#include "I_dispvidctrl_ClientSds_ttsIf.h"
#include "dispvidctrl_tclControl_Avm.h"
#include "I_dispvidctrl_tclAudioPlayer.h"
#include "dispvidctrl_AppMain.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_APPLICATION
#include "trcGenProj/Header/dispvidctrl_tclAudioRouting.cpp.trc.h"
#endif


// Private Constructor
dispvidctrl_tclAudio_Routing::dispvidctrl_tclAudio_Routing(dispvidctrl_tclAppMain* poMainAppl)
: arl_tclISource(poMainAppl)
, dispvidctrl_tclBaseIf(poMainAppl)
{
   ETG_TRACE_USR4(("dispvidctrl_tclAudio_Routing:: Constructor"));
   m_poControlAvm = OSAL_NULL;
   m_poSdsTTS = OSAL_NULL;
   m_poAudioPlayer = OSAL_NULL;

   m_bTtsAvailable = TRUE;
   m_AvmIpaConfig = FALSE;
   m_AvmTypeFap = FALSE;
   m_IpaSourceName = ARL_SRC_IPA_TTS; // using only one Audio channel for all kind of Parking Audio channel (currently IPA)
   m_AvmFapSourceName = ARL_SRC_DRVASS_VIDEO; // seperate audio channel for FAP/
   m_AudioChannelState = AUDIO_DEALLOCATED;
   m_SrcActivityState = AUDIO_EN_UNKNOWN;
}

// Destructor
dispvidctrl_tclAudio_Routing::~dispvidctrl_tclAudio_Routing()
{
   ETG_TRACE_USR4(("dispvidctrl_tclAudio_Routing:: Destructor"));
   m_poControlAvm = OSAL_NULL;
   m_poSdsTTS = OSAL_NULL;
   m_poAudioPlayer = OSAL_NULL;
}

tVoid dispvidctrl_tclAudio_Routing::vHandleMessage(dispvidctrl_tclBaseIf::TMsg* /*pMsg*/)
{
   ETG_TRACE_USR4(("dispvidctrl_tclAudio_Routing:: vHandleMessage entered"));
}

tVoid dispvidctrl_tclAudio_Routing::vHandleTraceMessage(const tUChar* /*puchData*/)
{
   ETG_TRACE_USR4(("dispvidctrl_tclAudio_Routing:: vHandleTraceMessage entered"));
}

tVoid dispvidctrl_tclAudio_Routing::vGetReferences()
{
   ETG_TRACE_USR4(("dispvidctrl_tclAudio_Routing:: vGetReferences entered"));

   m_poSdsTTS = dynamic_cast<Idispvidctrl_tclClientSds_ttsIf*>(_cpoMain->getHandler("Idispvidctrl_tclClientSds_ttsIf"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poSdsTTS);
   
   m_poControlAvm = dynamic_cast<dispvidctrl_tclControl_Avm*>(_cpoMain->getHandler("dispvidctrl_tclControl_Avm"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poControlAvm);
   
   m_poAudioPlayer = dynamic_cast<I_dispvidctrl_tclAudioPlayer*>(_cpoMain->getHandler("dispvidctrl_tclAudioPlayer"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poAudioPlayer);
}

tVoid dispvidctrl_tclAudio_Routing::vStartCommunication()
{
   ETG_TRACE_USR4(("dispvidctrl_tclAudio_Routing:: vStartCommunication entered"));
}

tVoid dispvidctrl_tclAudio_Routing::vTraceInfo()
{
   ETG_TRACE_FATAL(("dispvidctrl_tclAudio_Routing::vTraceInfo() entered"));
   ETG_TRACE_FATAL(("dispvidctrl_tclAudio_Routing::vTraceInfo() m_bTtsAvailable ............... = %u", m_bTtsAvailable));
   ETG_TRACE_FATAL(("dispvidctrl_tclAudio_Routing::vTraceInfo() m_IpaSourceName ................ = %u", m_IpaSourceName));
   ETG_TRACE_FATAL(("dispvidctrl_tclAudio_Routing::vTraceInfo() m_AudioChannelState ........ = %u", ETG_CENUM(dispvidctrl_tenAudioChannelState, m_AudioChannelState)));
   ETG_TRACE_FATAL(("dispvidctrl_tclAudio_Routing::vTraceInfo() m_SrcActivityState ......... = %u", ETG_CENUM(dispvidctrl_tenSrcActivityState, m_SrcActivityState)));
}

tVoid dispvidctrl_tclAudio_Routing::vGetConfiguration(const TConfiguration* pStConfigurationValues)
{
   ETG_TRACE_USR4(("dispvidctrl_tclAudio_Routing:: vGetConfiguration entered"));
   
   if (pStConfigurationValues)
   {
      m_bTtsAvailable = (0x02 == pStConfigurationValues->u8DeviceVariantType);
      m_AvmIpaConfig = pStConfigurationValues->bAvmType_Ipa;
      m_AvmTypeFap = pStConfigurationValues->bAvmType_Fap;
   }
}


/*******************************************************************************
 *
 * FUNCTION: tBool fc_sxm_arl_tclISource::bOnAllocate()
 *
 * DESCRIPTION: This function is called by the AUDIO-ROUTING-LIB after Allocate
 *              is processed.
 *
 * PARAMETER:[enSrcNum]:  (I) Source Number.
 *           [rfcoAllocRoute]: (I) Reference to Allocate route result
 *
 * RETURNVALUE: [tBool]: TRUE, if Application performed operations successfully,
 *              FALSE otherwise
 *
 ********************************************************************************
 * Overrides method arl_tclISource::bOnAllocate(..)
 *******************************************************************************/
tBool dispvidctrl_tclAudio_Routing::bOnAllocate(arl_tenSource enSrcNum, const arl_tAllocRouteResult& rfcoAllocRoute)
{
   ETG_TRACE_USR1(("dispvidctrl_tclAudio_Routing::bOnAllocate(), enSrcNum = %d", enSrcNum));
   
   if ( enSrcNum == m_IpaSourceName )
   {
      m_AudioChannelState = AUDIO_ALLOCATED;
   }
   
   std::vector<midw_fi_tclString, std::allocator<midw_fi_tclString> >::const_iterator it = rfcoAllocRoute.listOutputDev.strALSADev.begin();
   
   for ( ; it != rfcoAllocRoute.listOutputDev.strALSADev.end(); it++ )
   {
      ETG_TRACE_USR4(("dispvidctrl_tclAudio_Routing::bOnAllocate(), listOutputDev.strALSADev = %s", it->szValue));
      DISPVIDCTRL_NULL_POINTER_CHECK_VAL(m_poAudioPlayer);
      m_poAudioPlayer->vSetAlsaDeviceName(it->szValue);
   }
   
   return TRUE;
}

/*******************************************************************************
 *
 * FUNCTION: tBool bOnDeAllocate()
 *
 * DESCRIPTION: This function is called by the CALLED BY AUDIO-ROUTING-LIB
 *               after DeAllocate is processed.
 *              
 * PARAMETER: [enSrcNum]:  (I) Source Number..
 *
 * RETURNVALUE: [tBool]: TRUE, if Application performed operations successfully,
 *               FALSE otherwise.
 *
 ********************************************************************************
 * Overrides method arl_tclISource::bOnDeAllocate(..)
 *******************************************************************************/
tBool dispvidctrl_tclAudio_Routing::bOnDeAllocate(arl_tenSource enSrcNum)
{
   ETG_TRACE_USR4(("dispvidctrl_tclAudio_Routing::bOnDeAllocate(), enSrcNum = %d", enSrcNum));
   if ( enSrcNum == m_IpaSourceName )
   {
      m_AudioChannelState = AUDIO_DEALLOCATED;
   }
   
   return TRUE;
}

/*******************************************************************************
 *
 * FUNCTION: tBool bOnSrcActivity()
 *
 * DESCRIPTION: This function is called by the CALLED BY AUDIO-ROUTING-LIB
 *               on Source Activity start.
 *              
 * PARAMETER: [enSrcNum]:  (I) Source Number..
 *            [rfcoSrcActivity]: (I) Source Activity
 *
 * RETURNVALUE: [tBool]: TRUE, if source activity was successful,
 *               FALSE otherwise.
 *
 ********************************************************************************
 * Overrides method arl_tclISource::bOnSrcActivity(..)
 *******************************************************************************/
tBool dispvidctrl_tclAudio_Routing::bOnSrcActivity(arl_tenSource enSrcNum, const arl_tSrcActivity& rfcoSrcActivity)
{
   tBool bResult = TRUE;
     ETG_TRACE_USR4(("dispvidctrl_tclAudio_Routing:: IPA_TTS bOnSrcActivity() m_SrcActivityState %u", enSrcNum));

   if (enSrcNum == m_IpaSourceName)
   {
	   switch( rfcoSrcActivity.enType )
	   {
	   case midw_fi_tcl_e8_SrcActivity::FI_EN_ON:
		   m_SrcActivityState = AUDIO_EN_ON;
		   (tVoid) bPrepareTts();
		   break;
	   case midw_fi_tcl_e8_SrcActivity::FI_EN_OFF:
		   m_SrcActivityState = AUDIO_EN_OFF;
		   (tVoid) bAbortTts();
		   break;
	   case midw_fi_tcl_e8_SrcActivity::FI_EN_PAUSE:
		   ETG_TRACE_FATAL(("dispvidctrl_tclAudio_Routing::bOnSrcActivity() IPA_TTS PAUSED entered, not handled!"));
		   m_SrcActivityState = AUDIO_EN_PAUSE;
		   break;
	   default:
		   m_SrcActivityState = AUDIO_EN_UNKNOWN;
		   break;
	   }

	   ETG_TRACE_USR4(("dispvidctrl_tclAudio_Routing::bOnSrcActivity() source %u", ETG_CENUM(dispvidctrl_tenSrcActivityState, m_SrcActivityState)));

	   DISPVIDCTRL_NULL_POINTER_CHECK_VAL(m_poControlAvm);
	   m_poControlAvm->vHandleAudioChannelStateChange(m_SrcActivityState);

	   vSourceActivityResult(enSrcNum, rfcoSrcActivity);
   }
   else if (enSrcNum == m_AvmFapSourceName)
   {
	   switch( rfcoSrcActivity.enType)
	   {
	   case midw_fi_tcl_e8_SrcActivity::FI_EN_ON:
		   vSourceActivityResult(enSrcNum, ARL_EN_ISRC_ACT_ON);
		   break;
	   case midw_fi_tcl_e8_SrcActivity::FI_EN_OFF:
		   vSourceActivityResult(enSrcNum, ARL_EN_ISRC_ACT_OFF);
		   break;
	   case midw_fi_tcl_e8_SrcActivity::FI_EN_PAUSE:
		   vSourceActivityResult(enSrcNum, ARL_EN_ISRC_ACT_PAUSE);
		   break;
	   default:
		   //nothing to be done
		   break;
	   }
   }
   else
   {
      ETG_TRACE_USR4(("dispvidctrl_tclAudio_Routing::bOnSrcActivity() not for DisplayVideoController source!"));
   }
   return bResult;
}

/*******************************************************************************
 *
 * FUNCTION: tBool bOnMuteState()
 *
 * DESCRIPTION: This function is called by the CALLED BY AUDIO-ROUTING-LIB
 *              when AudioRouting changed Mute state for given Source.
 *              
 *
 * PARAMETER: [enSrcNum]:  (I) Source Number..
 *            [enMuteState]: systems actual mute state
 *
 * RETURNVALUE: [tBool]: TRUE
 *                       FALSE 
 *
 ********************************************************************************
 * Overrides method arl_tclISource::bOnMuteState(..)
 *******************************************************************************/
tBool dispvidctrl_tclAudio_Routing::bOnMuteState(arl_tenSource enSrcNum, arl_tenMuteState enMuteState)
{
   ETG_TRACE_USR4(("dispvidctrl_tclAudio_Routing::bOnMuteState() entered, SrcName %u MuteState %u", \
                  ETG_CENUM(arl_tenSource, enSrcNum), ETG_CENUM(arl_tenMuteState, enMuteState)));
   
   if ( enSrcNum == m_IpaSourceName )
   {
      // Our Audio Channel Muted, do we need to do something ?
   }

   return FALSE;
}

/*******************************************************************************
 *
 * FUNCTION: tBool vInformAudioPlaybackFinished()
 *
 * DESCRIPTION: This function is called by the Audio player (GST)/ SDS_TTS
 *              when Audio which had to be played is finished.
 *              
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *
 *
  *******************************************************************************/
tVoid dispvidctrl_tclAudio_Routing::vInformAudioPlaybackFinished(const arl_tenSource enAudioSource)
{
   ETG_TRACE_USR4(("dispvidctrl_tclAudio_Routing::vInformAudioPlaybackFinished() entered"));
   
   if ( m_SrcActivityState == AUDIO_EN_ON )
   {
      bRequestAudioRoute(enAudioSource, ARL_EN_ISRC_ACT_OFF);
   }
}

/*******************************************************************************
 *
 * FUNCTION: tBool bRequestAudioRoute()
 *
 * DESCRIPTION: This function is to be called when we need to allocate the 
 *              audio channel
 *
 * PARAMETER: [enAudioSource]:  (I) Source Number..
 *            [enActivity]:     Allocate/Deallocate etc..
 *            [bTtsAvailable]:  TTS available or not
 *
 * RETURNVALUE: [tBool]: TRUE
 *                       FALSE 
 *
 *******************************************************************************/
tBool dispvidctrl_tclAudio_Routing::bRequestAudioRoute(arl_tenSource enAudioSource, arl_tenActivity enActivity)
{
   ETG_TRACE_USR4(("dispvidctrl_tclAudio_Routing::bRequestAudioRoute() entered, enAudioSource %u enActivity %u", \
                      ETG_CENUM(arl_tenSource, enAudioSource), ETG_CENUM(arl_tenActivity, enActivity)));
                      
   tBool bReturn = FALSE;

   if (enAudioSource == m_AvmFapSourceName)
   {
	   bReturn = bSetAudioRouteRequest(enAudioSource, enActivity);
   }
   else if (enAudioSource == m_IpaSourceName)// handled only for ARL_SRC_IPA_TTS as of now
   {
      if (((enActivity == ARL_EN_ISRC_ACT_ON)  && (m_SrcActivityState == AUDIO_EN_OFF))
      ||  ((enActivity == ARL_EN_ISRC_ACT_OFF) && (m_SrcActivityState != AUDIO_EN_OFF)))
      {
    	  bReturn = bSetAudioRouteRequest(enAudioSource, enActivity);
      }
   }
   else
   {
	   ETG_TRACE_USR1(( "dispvidctrl_tclAudio_Routing::bRequestAudioRoute() Request not for Dispvidctrl Source"));
   }

   return bReturn;
}

/*******************************************************************************
 *
 * FUNCTION: tBool bRequestAudioSourceAvailability()
 *
 * DESCRIPTION: This function is to be called when we need to allocate the
 *               source for Audio.
 *               In this case the FAP Audio is a external audio from FAP unit
 *
 * PARAMETER: [enSrcAvailability] ARL_EN_SRC_PLAYABLE/ playable Content found
 *            [enAudioSource]:  (I) Source Number..
 *            [enAvailabilityReason]:     availabilityReason /same media/ Device was already there before
 *            [sub source ]: 0 by default
 *             *
 * RETURNVALUE: [tBool]: TRUE
 *                       FALSE
 *
 *******************************************************************************/
tBool dispvidctrl_tclAudio_Routing::bRequestAudioSourceAvailability(arl_tenSource enAudioSource)
{
   ETG_TRACE_USR4(("dispvidctrl_tclAudio_Routing::bRequestAudioRoute() entered, enAudioSource %u ", enAudioSource));

   tBool bReturn = FALSE;

   if (enAudioSource == m_AvmFapSourceName)// handled only for FAP Audio as of now
   {
	   bReturn = bSetSourceAvailability(ARL_EN_SRC_PLAYABLE,ARL_EN_REASON_SAMEMEDIA,enAudioSource,0);

   }
   else
   {
	   ETG_TRACE_USR1(( "dispvidctrl_tclAudio_Routing::bRequestAudioSourceAvailability() Request not matching to FAP Source"));
   }
   return bReturn;
}

tBool dispvidctrl_tclAudio_Routing::bPrepareTts() const
{
   if ( m_AvmIpaConfig )
   {
      if ( m_bTtsAvailable )
      {
         ETG_TRACE_USR1(( "dispvidctrl_tclAudio_Routing::bPrepareTts ............ sendPrepareVoiceOutputMStart() Posted MStart"));
         DISPVIDCTRL_NULL_POINTER_CHECK_VAL(m_poSdsTTS);
         m_poSdsTTS->requestPrepareVoiceOutput();
         return TRUE;
      }
      else // Da Variant (Non-navi)
      {
         if ( m_poAudioPlayer )
         {
            ETG_TRACE_USR1(( "dispvidctrl_tclAudio_Routing::bPrepareTts bActivate Ipa Audio through GST Audio Player"));
            m_poAudioPlayer->vActivateIpaAudio(TRUE);
            return TRUE;
         }
         else
         {
            ETG_TRACE_FATAL(( "dispvidctrl_tclAudio_Routing::bPrepareTts m_poAudioPlayer is NULL"));
         }
      }
   }
   else 
   {
      ETG_TRACE_USR1(( "dispvidctrl_tclAudio_Routing::bPrepareTts IPA Not configured, only allocated Audio Channel"));
   }

   return FALSE;
}

tBool dispvidctrl_tclAudio_Routing::bAbortTts() const
{
   tBool bSuccess = FALSE;
   
   if ( m_AvmIpaConfig )
   {
      if ( m_bTtsAvailable )
      {
         DISPVIDCTRL_NULL_POINTER_CHECK_VAL(m_poSdsTTS);
         
         sds_fi_tcl_e8_PromptAbortOption oAbortOption;
         oAbortOption.enType = sds_fi_tcl_e8_PromptAbortOption::FI_EN_IMMEDIATE;
         m_poSdsTTS->sendAbortPromptMStart(oAbortOption);
         ETG_TRACE_USR1(( "dispvidctrl_tclAudio_Routing::bAbortTts ............ sendAbortPromptMStart() Posted MStart %u", bSuccess));
      }
      else // Da Variant (Non-navi)
      {
         if ( m_poAudioPlayer )
         {
            m_poAudioPlayer->vActivateIpaAudio(FALSE);
         }
         else
         {
            ETG_TRACE_FATAL(( "dispvidctrl_tclAudio_Routing::bPrepareTts m_poAudioPlayer is NULL"));
         }
         
         bSuccess = TRUE;
      }
   }
   else 
   {
      ETG_TRACE_USR1(( "dispvidctrl_tclAudio_Routing::bPrepareTts IPA Not configured, only allocated Audio Channel"));
   }
   
   return bSuccess;
}

// EOF
