/************************************************************************
* FILE:         dispvidctrl_InputHandler.cpp
* PROJECT:
* SW-COMPONENT: DisplayVideoController
*----------------------------------------------------------------------
*
* DESCRIPTION: interface of Input event observer class
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 24.02.2017| RBEI/ECV Deora     | init
*
*************************************************************************/

// this handler shall provide touch/gesture information from event observer to FAP unit
// we need:
// for update to FAP via VCN message
// - state "press" (as long as screen is touched)
// - Coordinate x,y
// for update to FAP via I2C
// - state "not touch" (touch panel is not touched and not in flick status)
// - state "press"     (touch panel is pressed with one finger and touch position does not change)
// - state "drag"      (touch panel is pressed with one finger and touch position is changing)
// - state "pinch"     (touch panel is pressed with two fingers and touch position is changing)
// - state "flick"     (touch panel is not touched and touch position was changing fast during drag status)
// - coordinates x,y for 1st and 2nd touch position (2nd touch position only in state "pinch")



/*******************************************************************************
                        Includes
*******************************************************************************/
#include "dispvidctrl_InputHandler.h"
#include "dispvidctrl_EventObserver.h"
#include "dispvidctrl_TouchDataHandler.h"
#include "Idispvidctrl_ClientVcan.h"

//I2C interface
#include <linux/i2c-dev.h>
#include <linux/i2c.h>
#include <sys/ioctl.h>
#include <errno.h>

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_APPLICATION_INPUT
#include "trcGenProj/Header/dispvidctrl_InputHandler.cpp.trc.h"
#endif

/*******************************************************************************
                        Defines
*******************************************************************************/

#define GET_HI_BYTE(x) (tU8) (( x >> 8) & 0x00FF )
#define GET_LO_BYTE(x) (tU8) (  x       & 0x00FF )

#define DISPVIDCTRL_INPUT_INITALISATION_DELAY_TIME_IN_MS        1000
#define DISPVIDCTRL_INPUT_FAP_I2C_UPDATE_CYCLE_TIME_IN_MS        100

dispvidctrl_tclInputHandler* dispvidctrl_tclInputHandler::m_poMyStaticRef = NULL;

/******************************************************************************/
/* FUNCTION     dispvidctrl_tclInputHandler()                                 */
/******************************************************************************/
/**
*  \brief       constructor
*
*  \param       pointer to main application
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclInputHandler::dispvidctrl_tclInputHandler(dispvidctrl_tclAppMain* poMainAppl)
: I_dispvidctrl_tclInputHandler(poMainAppl)
, m_hTimerHandle_InputHandler(OSAL_C_INVALID_HANDLE)
, m_hTimerHandle_FapI2cMessageUpdate(OSAL_C_INVALID_HANDLE)
, m_bAvmTypeFap(FALSE)
, m_bFapModeActive(FALSE)
, m_bObserverRunning(FALSE)
, m_ObserverReady(FALSE)
, m_fHeightFactor(1.0)
, m_fWidthFactor(1.0)
, m_enTouchGestureState(EN_TOUCH_GESTURE_NOT_TOUCH)
{
   ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclInputHandler::dispvidctrl_tclInputHandler() entered.", OSAL_ClockGetElapsedTime()));

   (tVoid)memset((tVoid*) m_au8FapI2cTouchDataMsg, 0x00, sizeof(m_au8FapI2cTouchDataMsg));
   (tVoid)memset((tVoid*) &m_sTouchInfoForCan, 0x00, sizeof(m_sTouchInfoForCan));

   m_poMyStaticRef = this;
   m_poEventObserver = NULL;
   m_poClientVcan = NULL;

   // create general purpose timer e.g. used to delay initalisation of observer (not cyclic timer)
   if ( OSAL_s32TimerCreate( (OSAL_tpfCallback)_pfCallbackTimer_InputHandler, (tPVoid)this, &m_hTimerHandle_InputHandler) != OSAL_OK)
   {
      ETG_TRACE_FATAL(("ERROR - Could not create InputHandler timer!"));
      m_hTimerHandle_InputHandler = OSAL_C_INVALID_HANDLE;
   }
   // create FAP Message I2C Message update timer (cyclic timer)
   if ( OSAL_s32TimerCreate( (OSAL_tpfCallback)_pfCallbackTimer_FapI2cMessageUpdate, (tPVoid)this, &m_hTimerHandle_FapI2cMessageUpdate) != OSAL_OK)
   {
      ETG_TRACE_FATAL(("ERROR - Could not create FapI2cMessageUpdate timer!"));
      m_hTimerHandle_FapI2cMessageUpdate = OSAL_C_INVALID_HANDLE;
   }
}

/******************************************************************************/
/* FUNCTION     ~dispvidctrl_tclInputHandler                                  */
/******************************************************************************/
/**
*  \brief       destructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclInputHandler::~dispvidctrl_tclInputHandler(tVoid)
{
   ETG_TRACE_FATAL(("[%d ms] ~dispvidctrl_tclInputHandler() entered.", OSAL_ClockGetElapsedTime()));
   m_poMyStaticRef = NULL;

   if ( m_bObserverRunning )
   {
      m_poEventObserver->stopObserver();
   }

   m_poEventObserver = NULL;
   m_poClientVcan = NULL;

   // stop and delete timer(s)
   if ( m_hTimerHandle_InputHandler != OSAL_C_INVALID_HANDLE )
   {
      (void) OSAL_s32TimerSetTime(m_hTimerHandle_InputHandler, 0, 0);
      (void) OSAL_s32TimerDelete(m_hTimerHandle_InputHandler);
   }
   if ( m_hTimerHandle_FapI2cMessageUpdate != OSAL_C_INVALID_HANDLE )
   {
      (void) OSAL_s32TimerSetTime(m_hTimerHandle_FapI2cMessageUpdate, 0, 0);
      (void) OSAL_s32TimerDelete(m_hTimerHandle_FapI2cMessageUpdate);
   }
}

/******************************************************************************/
/* FUNCTION     vGetReferences                                                */
/******************************************************************************/
/**
*  \brief       Function to get all reference needed by this class.
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclInputHandler::vGetReferences(tVoid)
{
   //ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclInputHandler::vGetReferences() entered.", OSAL_ClockGetElapsedTime()));

   m_poEventObserver = dynamic_cast<dispvidctrl_tclEventObserver*>(_cpoMain->getHandler("dispvidctrl_tclEventObserver"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poEventObserver);

   m_poClientVcan = dynamic_cast<I_dispvidctrl_tclClientVcan*>(_cpoMain->getHandler("I_dispvidctrl_tclClientVcan"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poClientVcan);
}

/******************************************************************************/
/* FUNCTION     vStartCommunication                                           */
/******************************************************************************/
/**
*  \brief       Function to start all dynamic objects e.g. threads, ...
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclInputHandler::vStartCommunication(tVoid)
{
   ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclInputHandler::vStartCommunication() entered.", OSAL_ClockGetElapsedTime()));

   // delay initalisation of observer/observer ready status
   (tVoid) _bSetTimer_InputHandler (DISPVIDCTRL_INPUT_INITALISATION_DELAY_TIME_IN_MS);
}

/******************************************************************************/
/* FUNCTION     vHandleMessage(TMsg* pMsg)                                    */
/******************************************************************************/
/**
*  \brief       Handle worker events.
*
*  \param       message pointer
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclInputHandler::vHandleMessage(dispvidctrl_tclBaseIf::TMsg* pMsg)
{
   ETG_TRACE_USR4(("[%d ms] dispvidctrl_tclInputHandler::vHandleMessage() entered %u -> data: %d.", OSAL_ClockGetElapsedTime(), ETG_CENUM(dispvidctrl_tclBaseIf::ECmdTypes , (tU32)pMsg->eCmd), pMsg->u.u32Data));

   switch (pMsg->eCmd)
   {
      case dispvidctrl_tclBaseIf::eControlFap:
         vHandleControl((tenControl_InputHandler) pMsg->u.u32Data);
         break;

      default:
         break;
   }
}

/******************************************************************************/
/* FUNCTION     vHandleControl()                                              */
/******************************************************************************/
/**
*  \brief       processing of control event
*
*  \param       event, data
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclInputHandler::vHandleControl(tenControl_InputHandler enControl, tU32 u32Value)
{
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32Value );

   ETG_TRACE_USR4((" InpHndlr :: ............vHandleControl(): %d     ", ETG_CENUM(tenControl_InputHandler, enControl)));

   switch (enControl)
   {
      case EN_INPUT_HANDLER__TIMEOUT_INPUT_HANDLER_TIMER:
         // first occurence is used for delaying the initalisation
         if (m_ObserverReady == FALSE)
         {
            if (m_bAvmTypeFap)
            {
               m_ObserverReady = TRUE;
               // in case start is pending
               _vStartStopObserver();
            }
         }
         else
         {
         }
         break;

      case EN_INPUT_HANDLER__TIMEOUT_FAP_I2C_MESSAGE_UPDATE_TIMER:
         // cyclic time out -> send I2C Message to FAP
         if ( m_bAvmTypeFap && m_bFapModeActive )
         {
            _bSendI2CMessage();
         }
         break;

      case EN_INPUT_HANDLER__FAP_VISUALISATION_OFF: // trigger from Control AVM when Visualisation is switched Off
         m_bFapModeActive = FALSE;
         // stop cyclic I2C Message transmission
         _bSetTimer_FapI2cMessageUpdate(0);

         // with Visualisation stop we have to reset Touch Data which we send via CAN signals or I2C
         _vResetFapTouchInfo();
         // send I2C message as event message and CAN message with touch data reseted
         _bSendI2CMessage();
         _bSendCanMessage();
         // reset gesture state
         m_enTouchGestureState = EN_TOUCH_GESTURE_NOT_TOUCH;

         // stop observer
         _vStartStopObserver();
         break;

      case EN_INPUT_HANDLER__FAP_VISUALISATION_ON: // trigger from Control AVM when Visualisation is switched On

         // with Visualisation start we have to reset Touch Data which we send via CAN signals or I2C
         _vResetFapTouchInfo();
         // send I2C message as event message and CAN message with touch data reseted
         _bSendI2CMessage();
         _bSendCanMessage();
         // reset gesture state
         m_enTouchGestureState = EN_TOUCH_GESTURE_NOT_TOUCH;

         // start observer
         m_bFapModeActive = TRUE;
         _vStartStopObserver();
         // start cyclic I2C Message transmission
         _bSetTimer_FapI2cMessageUpdate(DISPVIDCTRL_INPUT_FAP_I2C_UPDATE_CYCLE_TIME_IN_MS);
         break;

      default:
         break;
   }
}


/******************************************************************************/
/* FUNCTION     vHandleTraceMessage                                           */
/******************************************************************************/
/**
*  \brief       handle TTFis commands
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclInputHandler::vHandleTraceMessage(const tUChar* puchData)
{
   DISPVIDCTRL_NULL_POINTER_CHECK(puchData);

   tU32 u32MsgCode = ((puchData[1]<<8) | puchData[2]);
}

/******************************************************************************/
/* FUNCTION     vTraceInfo()                                                  */
/******************************************************************************/
/**
*  \brief       trace information
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclInputHandler::vTraceInfo()
{
   ETG_TRACE_USR4(("dispvidctrl_tclInputHandler::vTraceInfo() entered."));
   ETG_TRACE_USR4(("dispvidctrl_tclInputHandler::m_bObserverRunning....... %d", m_bObserverRunning));
}

/******************************************************************************/
/* FUNCTION     vGetConfiguration()                                           */
/******************************************************************************/
/**
*  \brief       trace information
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclInputHandler::vGetConfiguration(const TConfiguration* pStConfigurationValues)
{
   ETG_TRACE_USR4(("dispvidctrl_tclInputHandler::vGetConfiguration() entered."));

   if (pStConfigurationValues)
   {
      m_bAvmTypeFap = pStConfigurationValues->bAvmType_Fap;
      m_fHeightFactor = 255.0 / pStConfigurationValues->u32DisplayResolutionHeight;
      m_fWidthFactor = 255.0 / pStConfigurationValues->u32DisplayResolutionWidth;
   }
}


/******************************************************************************/
/* FUNCTION     _bSetTimer_InputHandler()                                     */
/******************************************************************************/
/**
*  \brief       set/reset the Input Handler general purpose timer
*               timer is used as non cyclic timer
*
*  \param       time out value
*  \return      none
*/
/******************************************************************************/
tBool dispvidctrl_tclInputHandler::_bSetTimer_InputHandler(OSAL_tMSecond msTimeout)
{
    tBool bSuccess = TRUE;
    if ( OSAL_C_INVALID_HANDLE != m_hTimerHandle_InputHandler ) {
        if ( OSAL_s32TimerSetTime( m_hTimerHandle_InputHandler, msTimeout, 0 ) != OSAL_OK ) {
            bSuccess = FALSE;
        }
    }
    return(bSuccess);
}
/******************************************************************************/
/* FUNCTION     _pfCallbackTimer_InputHandler                                 */
/******************************************************************************/
/**
*  \brief       timer callback
*
*  \param       pointer to argument
*  \return
*/
/******************************************************************************/
OSAL_tpfCallback dispvidctrl_tclInputHandler::_pfCallbackTimer_InputHandler(tVoid* pArg)
{
   //ETG_TRACE_USR4(("dispvidctrl_tclInputHandler::_pfCallbackTimer_InputHandler(): called"));

   dispvidctrl_tclInputHandler* po_tclControl_InputHandler = (dispvidctrl_tclInputHandler*) pArg;
   if (po_tclControl_InputHandler) {
      if ((po_tclControl_InputHandler->_cpoMain) && (po_tclControl_InputHandler->_cpoMain->theServer())) {
         // post loop-back message to Input Handler itself
         tChar strIfName[] = "dispvidctrl_tclInputHandler";
         po_tclControl_InputHandler->_cpoMain->theServer()->vPostMsgControlFap(strIfName, (tU32) EN_INPUT_HANDLER__TIMEOUT_INPUT_HANDLER_TIMER);
      }
   }
   return 0;
}


/******************************************************************************/
/* FUNCTION     _bSetTimer_FapI2cMessageUpdate()                              */
/******************************************************************************/
/**
*  \brief       set/reset
*               timer is used as cyclic timer
*
*  \param       cyclic time out value
*  \return      none
*/
/******************************************************************************/
tBool dispvidctrl_tclInputHandler::_bSetTimer_FapI2cMessageUpdate(OSAL_tMSecond msTimeout)
{
    tBool bSuccess = TRUE;
    if ( OSAL_C_INVALID_HANDLE != m_hTimerHandle_FapI2cMessageUpdate ) {
        if ( OSAL_s32TimerSetTime( m_hTimerHandle_FapI2cMessageUpdate, msTimeout, msTimeout ) != OSAL_OK ) {
            bSuccess = FALSE;
        }
    }
    return(bSuccess);
}
/******************************************************************************/
/* FUNCTION     _pfCallbackTimer_FapI2cMessageUpdate                          */
/******************************************************************************/
/**
*  \brief       timer callback
*
*  \param       pointer to argument
*  \return
*/
/******************************************************************************/
OSAL_tpfCallback dispvidctrl_tclInputHandler::_pfCallbackTimer_FapI2cMessageUpdate(tVoid* pArg)
{
   ETG_TRACE_USR4(("dispvidctrl_tclInputHandler::_pfCallbackTimer_FapI2cMessageUpdate(): called"));

   dispvidctrl_tclInputHandler* po_tclControl_InputHandler = (dispvidctrl_tclInputHandler*) pArg;
   if (po_tclControl_InputHandler) {
      if ((po_tclControl_InputHandler->_cpoMain) && (po_tclControl_InputHandler->_cpoMain->theServer())) {
         // post loop-back message to Input Handler itself
         tChar strIfName[] = "dispvidctrl_tclInputHandler";
         po_tclControl_InputHandler->_cpoMain->theServer()->vPostMsgControlFap(strIfName, (tU32) EN_INPUT_HANDLER__TIMEOUT_FAP_I2C_MESSAGE_UPDATE_TIMER);
      }
   }
   return 0;
}


/******************************************************************************/
/* FUNCTION     _vStartStopObserver                                           */
/******************************************************************************/
/**
*  \brief       start/stop observer
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclInputHandler::_vStartStopObserver(tVoid)
{
   ETG_TRACE_USR4(("dispvidctrl_tclInputHandler::_vStartStopObserver() entered"));

   if ( m_poEventObserver != NULL)
   {
      if ( m_ObserverReady ) // init delay done
      {
         if ( m_bAvmTypeFap && m_bFapModeActive ) // Add the conditions which should activate the observer
         {
            if ( !m_bObserverRunning )
            {
               m_bObserverRunning = TRUE;
               tS32 s32StartObserver = m_poEventObserver->startObserver();
               ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclInputHandler::_vStartStopObserver() startObserver %d", OSAL_ClockGetElapsedTime(), s32StartObserver));
            }
         }
         else // if no condition to keep the observer active, then Stop it.
         {
            if ( m_bObserverRunning )
            {
               m_bObserverRunning = FALSE;
               m_poEventObserver->stopObserver();
               ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclInputHandler::_vStartStopObserver() stopObserver", OSAL_ClockGetElapsedTime()));
            }
         }
      }
   }
}

/******************************************************************************/
/* FUNCTION     _vResetTouchInfo                                              */
/******************************************************************************/
/**
*  \brief
*
*  \param
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclInputHandler::_vResetFapTouchInfo(tVoid)
{
      ETG_TRACE_USR1(("[%d ms] dispvidctrl_tclInputHandler::_vResetTouchInfo()", OSAL_ClockGetElapsedTime()));

      if ( m_bAvmTypeFap )
      {
         m_au8FapI2cTouchDataMsg[EN_FAP_I2C_TOUCH_MSG_BYTE_COMMAND]        = 0;
         m_au8FapI2cTouchDataMsg[EN_FAP_I2C_TOUCH_MSG_BYTE_TOUCHSTATE]     = EN_TOUCH_GESTURE_NOT_TOUCH; // = 0
         m_au8FapI2cTouchDataMsg[EN_FAP_I2C_TOUCH_MSG_BYTE_HIX1]           = 0;
         m_au8FapI2cTouchDataMsg[EN_FAP_I2C_TOUCH_MSG_BYTE_LOX1]           = 0;
         m_au8FapI2cTouchDataMsg[EN_FAP_I2C_TOUCH_MSG_BYTE_HIY1]           = 0;
         m_au8FapI2cTouchDataMsg[EN_FAP_I2C_TOUCH_MSG_BYTE_LOY1]           = 0;
         m_au8FapI2cTouchDataMsg[EN_FAP_I2C_TOUCH_MSG_BYTE_FLICKSPEED]     = 0;
         m_au8FapI2cTouchDataMsg[EN_FAP_I2C_TOUCH_MSG_BYTE_FLICKDIRECTION] = 0;
         m_au8FapI2cTouchDataMsg[EN_FAP_I2C_TOUCH_MSG_BYTE_CHECKSUM]       = 0;

         m_sTouchInfoForCan.x_ordinate = 0;
         m_sTouchInfoForCan.y_ordinate = 0;
         m_sTouchInfoForCan.status     = EN_RELEASE; // = 0
      }
}

/******************************************************************************/
/* FUNCTION     vUpdateTouchInfo                                              */
/******************************************************************************/
/**
*  \brief       called from observer
*
*  \param
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclInputHandler::vUpdateTouchInfo(const sTouchListElement* const poTouchData, const tU8 u8Slot) const
{
      ETG_TRACE_USR1(("[%d ms] dispvidctrl_tclInputHandler::vUpdateTouchInfo() RAW (x,y) = (%u,%u) Status = %u", \
                         OSAL_ClockGetElapsedTime(), poTouchData->u16XCoord, poTouchData->u16YCoord, ETG_CENUM(tenTouchStatus, poTouchData->u8TouchStatus)));

      tU32  u16Checksum = 0;
      tBool bI2cMessageContentUpdate = FALSE;
      tBool bI2cMessageSendEvent     = FALSE;
      tBool bCanMessageContentUpdateAndSendEvent = FALSE;

      // Depending on the current gesture state and the touch event (gesture change or coordinate change)
      // we have to send I2C and CAN message with new content,
      // or to ...,
      // or to update I2C message content only.

      // info for FAP unit
      if ( m_bAvmTypeFap && m_bFapModeActive )
      {

         // switch current gesture state
         switch (m_enTouchGestureState)
         {
            case EN_TOUCH_GESTURE_NOT_TOUCH:
               if (poTouchData->u8TouchStatus == EN_PRESS)
               {
                  bI2cMessageContentUpdate = TRUE;
                  bI2cMessageSendEvent     = TRUE;
                  bCanMessageContentUpdateAndSendEvent = TRUE;

                  // new state is "PRESS"
                  m_poMyStaticRef->m_enTouchGestureState = EN_TOUCH_GESTURE_PRESS;
               }

               break;

            case EN_TOUCH_GESTURE_PRESS:
               if (poTouchData->u8TouchStatus == EN_PRESS)
               {
                  // change of touch coordinates x,y ?
                  bI2cMessageContentUpdate = TRUE;
                  bCanMessageContentUpdateAndSendEvent = TRUE;
               }
               else if (poTouchData->u8TouchStatus == EN_RELEASE)
               {
                  bI2cMessageContentUpdate = TRUE;
                  bI2cMessageSendEvent     = TRUE;
                  bCanMessageContentUpdateAndSendEvent = TRUE;

                  // new state is "NOT_TOUCHED"
                  m_poMyStaticRef->m_enTouchGestureState = EN_TOUCH_GESTURE_NOT_TOUCH;
               }
               else if (poTouchData->u8TouchStatus == EN_INVALID)
               {
                  // reset

                  bI2cMessageContentUpdate = TRUE;
                  bI2cMessageSendEvent     = TRUE;
                  bCanMessageContentUpdateAndSendEvent = TRUE;

                  // new state is "NOT_TOUCHED"
                  m_poMyStaticRef->m_enTouchGestureState = EN_TOUCH_GESTURE_NOT_TOUCH;
               }
               else
               {
               }
               break;
            case EN_TOUCH_GESTURE_PINCH:
               break;
            case EN_TOUCH_GESTURE_DRAG:
               break;
            case EN_TOUCH_GESTURE_FLICK:
               break;
            default:
               break;
         }


         // for I2C message
         if (bI2cMessageContentUpdate == TRUE)
         {
            m_poMyStaticRef->m_au8FapI2cTouchDataMsg[EN_FAP_I2C_TOUCH_MSG_BYTE_COMMAND]        = 0;
            m_poMyStaticRef->m_au8FapI2cTouchDataMsg[EN_FAP_I2C_TOUCH_MSG_BYTE_TOUCHSTATE]     = m_enTouchGestureState;
            if ( u8Slot == 0 )
            {
               m_poMyStaticRef->m_au8FapI2cTouchDataMsg[EN_FAP_I2C_TOUCH_MSG_BYTE_HIX1]        = GET_HI_BYTE(poTouchData->u16XCoord);
               m_poMyStaticRef->m_au8FapI2cTouchDataMsg[EN_FAP_I2C_TOUCH_MSG_BYTE_LOX1]        = GET_LO_BYTE(poTouchData->u16XCoord);
               m_poMyStaticRef->m_au8FapI2cTouchDataMsg[EN_FAP_I2C_TOUCH_MSG_BYTE_HIY1]        = GET_HI_BYTE(poTouchData->u16YCoord);
               m_poMyStaticRef->m_au8FapI2cTouchDataMsg[EN_FAP_I2C_TOUCH_MSG_BYTE_LOY1]        = GET_LO_BYTE(poTouchData->u16YCoord);
            }
            else if ( u8Slot == 1 )
            {
               m_poMyStaticRef->m_au8FapI2cTouchDataMsg[EN_FAP_I2C_TOUCH_MSG_BYTE_HIX2]        = GET_HI_BYTE(poTouchData->u16XCoord);
               m_poMyStaticRef->m_au8FapI2cTouchDataMsg[EN_FAP_I2C_TOUCH_MSG_BYTE_LOX2]        = GET_LO_BYTE(poTouchData->u16XCoord);
               m_poMyStaticRef->m_au8FapI2cTouchDataMsg[EN_FAP_I2C_TOUCH_MSG_BYTE_HIY2]        = GET_HI_BYTE(poTouchData->u16YCoord);
               m_poMyStaticRef->m_au8FapI2cTouchDataMsg[EN_FAP_I2C_TOUCH_MSG_BYTE_LOY2]        = GET_LO_BYTE(poTouchData->u16YCoord);
            }
            m_poMyStaticRef->m_au8FapI2cTouchDataMsg[EN_FAP_I2C_TOUCH_MSG_BYTE_FLICKSPEED]     = 0;
            m_poMyStaticRef->m_au8FapI2cTouchDataMsg[EN_FAP_I2C_TOUCH_MSG_BYTE_FLICKDIRECTION] = 0;

            // checksum shall be a simple sum of all previous message bytes
            for (tU8 i = EN_FAP_I2C_TOUCH_MSG_BYTE_COMMAND; i < EN_FAP_I2C_TOUCH_MSG_BYTE_CHECKSUM; i++)
            {
                u16Checksum += m_au8FapI2cTouchDataMsg[i];
            }
            m_poMyStaticRef->m_au8FapI2cTouchDataMsg[EN_FAP_I2C_TOUCH_MSG_BYTE_CHECKSUM] = (tU8) (u16Checksum & 0x00FF);

            // trace I2C touch info
            for ( tU8 i = EN_FAP_I2C_TOUCH_MSG_BYTE_COMMAND; i < EN_FAP_I2C_TOUCH_MSG_NO_OF_BYTES; i++)
            {
               ETG_TRACE_USR4((" InpHndlr :: bOnFapTouchEvent(): Input m_au8FapI2cTouchDataMsg[0x%02x] = 0x%02x", i, m_au8FapI2cTouchDataMsg[i]));
            }
         }

         if (bI2cMessageSendEvent == TRUE)
         {
            m_poMyStaticRef->_bSendI2CMessage();
         }


         // for CAN signals
         if (bCanMessageContentUpdateAndSendEvent == TRUE)
         {
            if (poTouchData->u8TouchStatus == EN_PRESS)
            {
               m_poMyStaticRef->m_sTouchInfoForCan.x_ordinate = static_cast<tU8>(poTouchData->u16XCoord * m_fWidthFactor);
               m_poMyStaticRef->m_sTouchInfoForCan.y_ordinate = static_cast<tU8>(poTouchData->u16YCoord * m_fHeightFactor);
               m_poMyStaticRef->m_sTouchInfoForCan.status = static_cast<tenTouchStatus>(poTouchData->u8TouchStatus);
            }
            else
            {
               m_poMyStaticRef->m_sTouchInfoForCan.x_ordinate = 0;
               m_poMyStaticRef->m_sTouchInfoForCan.y_ordinate = 0;
               m_poMyStaticRef->m_sTouchInfoForCan.status = EN_RELEASE;  // 0
            }

            // trace CAN touch info
            ETG_TRACE_USR4(("[%d ms] dispvidctrl_tclInputHandler::vUpdateTouchInfo() RESOLVED (x,y) = (%u,%u) Status = %u", \
                         OSAL_ClockGetElapsedTime(), m_sTouchInfoForCan.x_ordinate, m_sTouchInfoForCan.y_ordinate, ETG_CENUM(enTouchStatus, m_sTouchInfoForCan.status)));

            // send event messages
            m_poMyStaticRef->_bSendCanMessage();
         }
      }
}


/******************************************************************************/
/* FUNCTION:     _bSendCanMessage                                             */
/******************************************************************************/
/**
*  \brief       Function to write Touch Data via I2C for FAP
*
*  \param       Touch Data Buffer
*  \return      tBool - write success
*/
/******************************************************************************/
tBool dispvidctrl_tclInputHandler::_bSendCanMessage(tVoid)
{
   tBool bResult = FALSE;

   ETG_TRACE_USR4(("dispvidctrl_tclInputHandler::_bSendCanMessage()"));

   // Broadcast over V-CAN
   if ( m_poClientVcan )
   {
       // trace CAN touch info
       ETG_TRACE_USR4(("dispvidctrl_tclInputHandler::_bSendCanMessage() (%u,%u) Status = %u", \
                         m_sTouchInfoForCan.x_ordinate, m_sTouchInfoForCan.y_ordinate, ETG_CENUM(enTouchStatus, m_sTouchInfoForCan.status)));

       m_poClientVcan->vWriteFAP_ScreenPressCoordinateX(m_sTouchInfoForCan.x_ordinate);
       m_poClientVcan->vWriteFAP_ScreenPressCoordinateY(m_sTouchInfoForCan.y_ordinate);
       m_poClientVcan->vWriteFAP_ScreenPressDetected(m_sTouchInfoForCan.status);

       bResult = TRUE;
   }
   return (bResult);
}

/******************************************************************************/
/* FUNCTION:     _bSendI2CMessage                                             */
/******************************************************************************/
/**
*  \brief       Function to write Touch Data via I2C for FAP
*
*  \param       Touch Data Buffer
*  \return      tBool - write success
*/
/******************************************************************************/
tBool dispvidctrl_tclInputHandler::_bSendI2CMessage(tVoid)
{
   tBool bResult = FALSE;
   const char *dev = "/dev/i2c-1";
   const tU8 u8SlaveAddress = 0x70; // 7 bit adress
   tS32 fd;

   ETG_TRACE_USR4(("dispvidctrl_tclInputHandler::_bSendI2CMessage()"));

   // open I2C bus to read/write the register
   fd = open(dev, O_RDWR);
   if(fd == -1)
   {
      ETG_TRACE_ERR(("dispvidctrl_tclInputHandler::_bSendI2CMessage: I2C bus open failed!"));
   }
   else
   {
      // set device address of the slave
      if( ioctl(fd, I2C_SLAVE, u8SlaveAddress) < 0)
      {
         ETG_TRACE_ERR(("dispvidctrl_tclInputHandler::_bSendI2CMessage: setting slave address failed!"));
      }
      else
      {
         struct i2c_msg messages[1];
         memset(messages, OSAL_NULL, sizeof(messages));
         struct i2c_rdwr_ioctl_data data;

         messages[0].addr = u8SlaveAddress;
         messages[0].flags = 0x0; // write mode
         messages[0].len = EN_FAP_I2C_TOUCH_MSG_NO_OF_BYTES;
         messages[0].buf = m_au8FapI2cTouchDataMsg;

         data.msgs = messages;
         data.nmsgs = 1;

         if (ioctl(fd, I2C_RDWR, &data) < 0)
         {
            ETG_TRACE_ERR(("dispvidctrl_tclInputHandler::_bSendI2CMessage: Error in writing Register"));
            ETG_TRACE_ERR(("dispvidctrl_tclInputHandler::_bSendI2CMessage: ioctl() FAILED: %s\n", strerror(errno)));
         }
         else
         {
            bResult = TRUE;
         }
      }
      close(fd);
   }
   return (bResult);
}
