/************************************************************************
* FILE:        dispvidctrl_config_early.cpp
* PROJECT:
* SW-COMPONENT:vd_dimming
*----------------------------------------------------------------------
*
* DESCRIPTION: implementation of EarlyConfig class
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 20.08.2015| CM-AI/EPB2 Bernard | init
*
*************************************************************************/

/*******************************************************************************
| includes: system- and project- includes
|-----------------------------------------------------------------------------*/
// Basic OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

/*******************************************************************************
| set own module id
|-----------------------------------------------------------------------------*/
#ifndef DISPVIDCTRL_USE_OSAL
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#endif //#ifndef DISPVIDCTRL_USE_OSAL

/*******************************************************************************
| includes: needed interfaces from external components
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| includes: internal and external interfaces from this component
|-----------------------------------------------------------------------------*/

#include "dispvidctrl_config_early.h"
#include "dispvidctrl_AppMain.h"
#include "dispvidctrl_tclTypes.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_APPLICATION
#include "trcGenProj/Header/dispvidctrl_config_early.cpp.trc.h"
#endif

/*******************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| typedefs (scope: module-local)
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function prototype (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementation (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
*
* FUNCTION:    dispvidctrl_tclEarlyConfig()
*
* DESCRIPTION: constructor
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
dispvidctrl_tclEarlyConfig::dispvidctrl_tclEarlyConfig()
{
   ETG_TRACE_USR4(("dispvidctrl_tclEarlyConfig() entered."));
   // EarlyConfig_Display
   (tVoid)OSAL_pvMemorySet((tVoid*)m_strDisplayResolution, OSAL_NULL, sizeof(m_strDisplayResolution));
   m_s32DisplayBackwardComp = -1;
   m_s32DisplayLowFrequency = -1;
   m_bKDSDisplayLoaded = FALSE;
   // EarlyConfig_RTCDriver
   (tVoid)OSAL_pvMemorySet((tVoid*)m_strRTCDriverName, OSAL_NULL, sizeof(m_strRTCDriverName));
   m_bKDSRTCDriverLoaded = FALSE;
   // EarlyConfig_Touch
   (tVoid)OSAL_pvMemorySet((tVoid*)m_strTouchDriverName, OSAL_NULL, sizeof(m_strTouchDriverName));
   (tVoid)OSAL_pvMemorySet((tVoid*)m_strTouchConfigFileName, OSAL_NULL, sizeof(m_strTouchConfigFileName));
   m_u8TouchDriverI2CAddress = 0;
   (tVoid)OSAL_pvMemorySet((tVoid*)m_strTouchDriverI2CAddress, OSAL_NULL, sizeof(m_strTouchDriverI2CAddress));
   m_bKDSTouchDriverLoaded = FALSE;
   // EarlyConfig_TwoDisplays
   (tVoid)OSAL_pvMemorySet((tVoid*)m_strTimingLvds1, OSAL_NULL, sizeof(m_strTimingLvds1));
   m_s8ClockEdgeSelectLVDS1 = -1;
   m_s32ClockGenerationLVDS1 = -1;
   (tVoid)OSAL_pvMemorySet((tVoid*)m_strTimingLvds2, OSAL_NULL, sizeof(m_strTimingLvds2));
   m_s8ClockEdgeSelectLVDS2 = -1;
   m_s32ClockGenerationLVDS2 = -1;
   m_bKDSTwoDisplaysLoaded = FALSE;
   // EarlyConfig ... Customer Touch Screen Configuration
   m_u8KDSCustomTouchScreen = 0;
   m_bKDSCustomTouchScreenLoaded = FALSE;
   // EarlyConfig_LayerManager
   m_u8FlushDesignConfigFile = OSAL_C_U8_MAX;
   (tVoid)OSAL_pvMemorySet((tVoid*)m_strFlushDesignConfigFile, OSAL_NULL, sizeof(m_strFlushDesignConfigFile));
   (tVoid)OSAL_pvMemorySet((tVoid*)m_au8DisplayOrder, -1, sizeof(m_au8DisplayOrder));
   m_bKDSLayerManagerLoaded = FALSE;
   // earlyConfig_TowTouchDriver
   (tVoid)OSAL_pvMemorySet((tVoid*)m_strTouchDriverName2, OSAL_NULL, sizeof(m_strTouchDriverName2));
   (tVoid)OSAL_pvMemorySet((tVoid*)m_strTouchConfigFileName2, OSAL_NULL, sizeof(m_strTouchConfigFileName2));
   m_u8TouchDriverI2CAddress2 = 0;
   (tVoid)OSAL_pvMemorySet((tVoid*)m_strTouchDriverI2CAddress2, OSAL_NULL, sizeof(m_strTouchDriverI2CAddress2));
   m_bKDSTouchDriverLoaded2 = FALSE;
   m_bTwoTouchChanged = FALSE;
}

/*******************************************************************************
*
* FUNCTION:    ~dispvidctrl_tclEarlyConfig()
*
* DESCRIPTION: destructor
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
dispvidctrl_tclEarlyConfig::~dispvidctrl_tclEarlyConfig()
{
   ETG_TRACE_USR4(("~dispvidctrl_tclEarlyConfig() entered."));
}

// structure of custom touch screen configurations
struct tCustomTouchConfig {
	const char* strTouchScreenConfigName;
	const char* strDisplayDriverName;
	const char* strTouchDriverName;
	const char* strTouchConfigFileName;
	tS32 s32BackwardComp;
	tS32 s32LowFrequency;
	tS32 s32ClockEdgeSelect;
	tS32 s32ClockGeneration;
};

// known custom touch screen configurations
const tCustomTouchConfig catCustomTouchScreenConfig[] =
{
	// touch screen configuration name
	//      display driver,      touch driver,      touch configuration file name
	//      backward comp,       low frequency		clock_edge_select,	ClockGeneration
	{ "Type 0 (0000)	Attached fascia (A-IVI) --> loading PD information",
			"", "", "",
			-1, -1, -1, -1
	},
	{ "Type 1 (0001)	CMF1 7 inches central panel carry back",
			"valeocp7g", "cyttsp4_i2c", "A-IVI_C7D_TMA884.raw",
			-1, -1, -1, -1
	},
	{ "Type 2 (0010)	C7D 7 inches display carry back",
			"valeoc7d", "cyttsp4_i2c", "A-IVI_C7D_TMA884.raw",
			-1, -1, 1, -1
	},
	{ "Type 3 (0011)	CMFCDph2 9.3 inches central panel",
			"lge_9_3inch", "atmel_mxt_ts", "",
			-1, -1, -1, -1
	},
	{ "Type 4 (0100)	CMFB 9.3 inches central panel",
			"lge_9_3inch", "atmel_mxt_ts", "",
			-1, -1, -1, -1
	},
	{ "Type 5 (0101)	CMFB 7 affordable display",
			"cmfb7affordable", "atmel_mxt_ts", "",
			-1, -1, -1, -1
	},
	{ "Type 6 (0110)	CMFB 7 best level display",
			"cmfb7bestlevel", "atmel_mxt_ts", "",
			-1, -1, -1, -1
	},
	{ "Type 7 (0111)	CMF1 8.7 central panel carry back --> obsolete",
			"valeocp8.7", "", "",
			-1, -1, -1, -1
	},
	{ "Type 8 (1000)	HFE ph2 specific central panel",
			"", "", "",
			-1, -1, -1, -1
	}
};
const tCustomTouchConfig ctCustomTouchScreenConfigUnkown = {
		"",
			"", "", "",
			-1, -1, -1, -1
};
/*******************************************************************************
*
* FUNCTION:    bLoadKDSCustom_TouchScreenConfig()
*
* DESCRIPTION: loading KDS EarlyConfig_TouchDriver data
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bLoadKDSCustom_TouchScreenConfig()
{
   ETG_TRACE_USR4(("bLoadKDSCustom_TouchScreenConfig() entered."));

   m_bKDSCustomTouchScreenLoaded = FALSE;
   m_u8KDSCustomTouchScreen = 0;

   tU8 _u8Value = OSAL_C_U8_MAX;
   tS32 _s32ReturnCode = DP_s32GetConfigItem( "SystemConfiguration1", "TouchScreenConfiguration",
		   &_u8Value, sizeof(_u8Value), FALSE );

   if (DP_S32_NO_ERR > _s32ReturnCode)
   {
	   ETG_TRACE_USR1(("bLoadKDSCustom_TouchScreenConfig: TouchScreenConfig could not loaded! (value=%u, returnval=%u) ",
			   _u8Value, _s32ReturnCode ));
   }
   else
   {
	   m_u8KDSCustomTouchScreen = _u8Value;
	   m_bKDSCustomTouchScreenLoaded = TRUE;

	  /** dp_tclDimmingDPEarlyConfig_Display_Type _oDisPlayType;

		   if (DP_S32_NO_ERR != _oDisPlayType.s32SetData(m_u8KDSCustomTouchScreen))
		   {
			   ETG_TRACE_FATAL(("bLoadKDSCustom_TouchScreenConfig: Failed to set display type %d to early config!",
					   m_u8KDSCustomTouchScreen ));
			   return FALSE;
		   }
		   else
		   {
			   ETG_TRACE_USR2(("bLoadKDSCustom_TouchScreenConfig: Successful set display type %d to early config!",
					   m_u8KDSCustomTouchScreen ));
		   } **/

	   
	   ETG_TRACE_USR1(("bLoadKDSCustom_TouchScreenConfig: TouchScreenConfig successful loaded with %u ",
			   _u8Value ));
   }

   return m_bKDSCustomTouchScreenLoaded;
}

/*******************************************************************************
*
* FUNCTION:    bIsValidKDSCustom_TouchScreenConfig()
*
* DESCRIPTION: loading KDS EarlyConfig_TouchDriver data
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsValidKDSCustom_TouchScreenConfig() const
{
   ETG_TRACE_USR4(("bIsValidKDSCustom_TouchScreenConfig() entered."));

   // check loaded
   if (TRUE != bIsLoadedKDSCustom_TouchScreenConfig())
	   return FALSE;
   // check value
   else if (u8GetTouchScreenConfig() >= EARLYCONFIG_ARRAY_ELEMENTS(catCustomTouchScreenConfig))
	   return FALSE;
   // check diplay driver
   else if (OSAL_NULL == cstrGetCustomDisplayDriver())
	   return FALSE;
   else if (0 == strlen(cstrGetCustomDisplayDriver()))
	   return FALSE;
   // check touch driver
   else if (OSAL_NULL == cstrGetCustomTouchDriver())
	   return FALSE;
   else if (0 == strlen(cstrGetCustomTouchDriver()))
	   return FALSE;
   // check touch config file

   return TRUE;
}

/*******************************************************************************
*
* FUNCTION:    cstrGetTouchScreenConfig()
*
* DESCRIPTION: returns the touch screen configuration name of loaded custom
*              touch screen configuration
*
* PARAMETER:   None
*
* RETURNVALUE: const char *                   {OSAL_NULL - failed | else string}
*
*******************************************************************************/
const char* dispvidctrl_tclEarlyConfig::cstrGetTouchScreenConfig(tVoid) const
{
   ETG_TRACE_USR4(("cstrGetTouchScreenConfig() entered."));

   if (TRUE == bIsLoadedKDSCustom_TouchScreenConfig()
	   && u8GetTouchScreenConfig() < EARLYCONFIG_ARRAY_ELEMENTS(catCustomTouchScreenConfig))
	   return catCustomTouchScreenConfig[u8GetTouchScreenConfig()].strTouchScreenConfigName;
   return ctCustomTouchScreenConfigUnkown.strTouchScreenConfigName;
}

/*******************************************************************************
*
* FUNCTION:    cstrGetCustomDisplayDriver()
*
* DESCRIPTION: returns the display driver name of loaded custom touch screen configuration
*
* PARAMETER:   None
*
* RETURNVALUE: const char *                   {OSAL_NULL - failed | else string}
*
*******************************************************************************/
const char* dispvidctrl_tclEarlyConfig::cstrGetCustomDisplayDriver(tVoid) const
{
   ETG_TRACE_USR4(("cstrGetCustomDisplayDriver() entered."));

   if (TRUE == bIsLoadedKDSCustom_TouchScreenConfig()
	   && u8GetTouchScreenConfig() < EARLYCONFIG_ARRAY_ELEMENTS(catCustomTouchScreenConfig))
	   return catCustomTouchScreenConfig[u8GetTouchScreenConfig()].strDisplayDriverName;
   return ctCustomTouchScreenConfigUnkown.strDisplayDriverName;
}

/*******************************************************************************
*
* FUNCTION:    s32GetCustomDisplayDriverBackwardComp()
*
* DESCRIPTION: returns the display backwardr comp. of loaded custom touch screen configuration
*
* PARAMETER:   None
*
* RETURNVALUE: tS32
*
*******************************************************************************/
tS32 dispvidctrl_tclEarlyConfig::s32GetCustomDisplayDriverBackwardComp(tVoid) const
{
   ETG_TRACE_USR4(("s32GetCustomDisplayDriverBackwardComp() entered."));

   if (TRUE == bIsLoadedKDSCustom_TouchScreenConfig()
	   && u8GetTouchScreenConfig() < EARLYCONFIG_ARRAY_ELEMENTS(catCustomTouchScreenConfig))
	   return catCustomTouchScreenConfig[u8GetTouchScreenConfig()].s32BackwardComp;
   return ctCustomTouchScreenConfigUnkown.s32BackwardComp;
}

/*******************************************************************************
*
* FUNCTION:    s32GetCustomDisplayDriverLowFrequency()
*
* DESCRIPTION: returns the display low frequency of loaded custom touch screen configuration
*
* PARAMETER:   None
*
* RETURNVALUE: tS32
*
*******************************************************************************/
tS32 dispvidctrl_tclEarlyConfig::s32GetCustomDisplayDriverLowFrequency(tVoid) const
{
   ETG_TRACE_USR4(("s32GetCustomDisplayDriverLowFrequency() entered."));

   if (TRUE == bIsLoadedKDSCustom_TouchScreenConfig()
	   && u8GetTouchScreenConfig() < EARLYCONFIG_ARRAY_ELEMENTS(catCustomTouchScreenConfig))
	   return catCustomTouchScreenConfig[u8GetTouchScreenConfig()].s32LowFrequency;
   return ctCustomTouchScreenConfigUnkown.s32LowFrequency;
}

/*******************************************************************************
*
* FUNCTION:    s32GetCustomDisplayDriverClockEdgeSelect()
*
* DESCRIPTION: returns the display clock edge select of loaded custom touch
*              screen configuration
*
* PARAMETER:   None
*
* RETURNVALUE: tS32
*
*******************************************************************************/
tS32 dispvidctrl_tclEarlyConfig::s32GetCustomDisplayDriverClockEdgeSelect(tVoid) const
{
   ETG_TRACE_USR4(("s32GetCustomDisplayDriverClockEdgeSelect() entered."));

   if (TRUE == bIsLoadedKDSCustom_TouchScreenConfig()
	   && u8GetTouchScreenConfig() < EARLYCONFIG_ARRAY_ELEMENTS(catCustomTouchScreenConfig))
	   return catCustomTouchScreenConfig[u8GetTouchScreenConfig()].s32ClockEdgeSelect;
   return ctCustomTouchScreenConfigUnkown.s32ClockEdgeSelect;
}

/*******************************************************************************
*
* FUNCTION:    s32GetCustomDisplayDriverClockGeneration()
*
* DESCRIPTION: returns the display spread spectrum clock generation of loaded
*              custom touch screen configuration
*
* PARAMETER:   None
*
* RETURNVALUE: tS32
*
*******************************************************************************/
tS32 dispvidctrl_tclEarlyConfig::s32GetCustomDisplayDriverClockGeneration(tVoid) const
{
   ETG_TRACE_USR4(("s32GetCustomDisplayDriverClockGeneration() entered."));

   if (TRUE == bIsLoadedKDSCustom_TouchScreenConfig()
	   && u8GetTouchScreenConfig() < EARLYCONFIG_ARRAY_ELEMENTS(catCustomTouchScreenConfig))
	   return catCustomTouchScreenConfig[u8GetTouchScreenConfig()].s32ClockGeneration;
   return ctCustomTouchScreenConfigUnkown.s32ClockGeneration;
}

/*******************************************************************************
*
* FUNCTION:    cstrGetCustomTouchDriver()
*
* DESCRIPTION: returns the touch driver name of loaded custom touch screen configuration
*
* PARAMETER:   None
*
* RETURNVALUE: const char *                   {OSAL_NULL - failed | else string}
*
*******************************************************************************/
const char* dispvidctrl_tclEarlyConfig::cstrGetCustomTouchDriver(tVoid) const
{
   ETG_TRACE_USR4(("cstrGetCustomTouchDriver() entered."));

   if (TRUE == bIsLoadedKDSCustom_TouchScreenConfig()
	   && u8GetTouchScreenConfig() < EARLYCONFIG_ARRAY_ELEMENTS(catCustomTouchScreenConfig))
	   return catCustomTouchScreenConfig[u8GetTouchScreenConfig()].strTouchDriverName;
   return ctCustomTouchScreenConfigUnkown.strTouchDriverName;
}

/*******************************************************************************
*
* FUNCTION:    cstrGetCustomTouchConfigFile()
*
* DESCRIPTION: returns the touch driver name of loaded custom touch screen configuration
*
* PARAMETER:   None
*
* RETURNVALUE: const char *                   {OSAL_NULL - failed | else string}
*
*******************************************************************************/
const char* dispvidctrl_tclEarlyConfig::cstrGetCustomTouchConfigFile(tVoid) const
{
   ETG_TRACE_USR4(("cstrGetCustomTouchConfigFile() entered."));

   if (TRUE == bIsLoadedKDSCustom_TouchScreenConfig()
	   && u8GetTouchScreenConfig() < EARLYCONFIG_ARRAY_ELEMENTS(catCustomTouchScreenConfig))
	   return catCustomTouchScreenConfig[u8GetTouchScreenConfig()].strTouchConfigFileName;
   return ctCustomTouchScreenConfigUnkown.strTouchConfigFileName;
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfoEarlyConfig_TouchScreenConfig() const
*
* DESCRIPTION: Trace information EarlyConfig Custom Touch Screen Configuration
*
* PARAMETER:   const char* strIdentifier
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid dispvidctrl_tclEarlyConfig::vTraceInfoEarlyConfig_TouchScreenConfig(const char* strIdentifier) const
{
	ETG_TRACE_USR4(("vTraceInfoEarlyConfig_Display() entered."));

	ETG_TRACE_FATAL(("%26s: <<< Custom Touch Screen Configuration >>> ", strIdentifier ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "         CTSC_LOADED",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsLoadedKDSCustom_TouchScreenConfig()) ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "          CTSC_VALID",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsLoadedKDSCustom_TouchScreenConfig()) ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "          CTSC_VALUE",
			u8GetTouchScreenConfig() ));
	if (OSAL_NULL != cstrGetTouchScreenConfig())
	{
		ETG_TRACE_FATAL(("%26s: %30s=%s ",   strIdentifier, "           CTSC_NAME",
				cstrGetTouchScreenConfig() ));
	}
	else
	{
		ETG_TRACE_FATAL(("%26s: %30s=%s ",   strIdentifier, "           CTSC_NAME",
				"-- undefined --" ));
	}
	if (OSAL_NULL != cstrGetCustomDisplayDriver())
	{
		ETG_TRACE_FATAL(("%26s: %30s=%s ",   strIdentifier, "        CTSC_DISPDRV",
				cstrGetCustomDisplayDriver() ));
	}
	else
	{
		ETG_TRACE_FATAL(("%26s: %30s=%s ",   strIdentifier, "        CTSC_DISPDRV",
				"-- undefined --" ));
	}
	if (OSAL_NULL != cstrGetCustomTouchDriver())
	{
		ETG_TRACE_FATAL(("%26s: %30s=%s ",   strIdentifier, "       CTSC_TOUCHDRV",
				cstrGetCustomTouchDriver() ));
	}
	else
	{
		ETG_TRACE_FATAL(("%26s: %30s=%s ",   strIdentifier, "       CTSC_TOUCHDRV",
				"-- undefined --" ));
	}
	if (OSAL_NULL != cstrGetCustomTouchConfigFile())
	{
		ETG_TRACE_FATAL(("%26s: %30s=%s ",   strIdentifier, "   CTSC_TOUCHCFGFILE",
				cstrGetCustomTouchConfigFile() ));
	}
	else
	{
		ETG_TRACE_FATAL(("%26s: %30s=%s ",   strIdentifier, "   CTSC_TOUCHCFGFILE",
				"-- undefined --" ));
	}
	ETG_TRACE_FATAL(("%26s: %30s=%d ",   strIdentifier, "  CTSC_DISP_BACKCOMP",
			s32GetCustomDisplayDriverBackwardComp() ));
	ETG_TRACE_FATAL(("%26s: %30s=%d ",   strIdentifier, "   CTSC_DISP_LOWFREQ",
			s32GetCustomDisplayDriverLowFrequency() ));
}

/*******************************************************************************
*
* FUNCTION:    bLoadKDSEarlyConfig()
*
* DESCRIPTION: loading of EarlyConfig data from KDS
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bLoadKDSEarlyConfig()
{
   ETG_TRACE_USR4(("bLoadKDSEarlyConfig() entered."));
   // load custom touch screen configuration
   if (TRUE == bLoadKDSCustom_TouchScreenConfig()
	   && TRUE == bIsValidKDSCustom_TouchScreenConfig()
	   && 0 != u8GetTouchScreenConfig())
   {
	   // custom touch screen configuration active

	   // EarlyConfig_RTCDriver
	   (tVoid)bLoadKDSEarlyConfig_RTCDriver();
	   // EarlyConfig_LayerManager
	   (tVoid)bLoadKDSearlyConfig_LayerManager();
	   // EarlyConfig_TwoDisplays
	   (tVoid)bLoadKDSEarlyConfig_TwoDisplays();
	   // EarlyConfig_TwoTouchDriver
	  // (tVoid)bLoadKDSearlyConfig_TwoTouchDriver();
	   ETG_TRACE_USR2(("bLoadKDSEarlyConfig(): returns with RTCDriver: %d, LayerManager: %d, TwoDisplays: %d",
			   bIsLoadedKDSEarlyConfig_RTCDriver(), bIsLoadedKDSEarlyConfig_LayerManager(),
			   bIsLoadedKDSEarlyConfig_TwoDisplays()));
	   return ((TRUE == bIsLoadedKDSEarlyConfig_RTCDriver())
			   && (TRUE == bIsLoadedKDSEarlyConfig_LayerManager())
			   && (TRUE == bIsLoadedKDSEarlyConfig_TwoDisplays())
			   );
   }
   else
   {
	   // EarlyConfig_Display
	   (tVoid)bLoadKDSEarlyConfig_Display();
	   // EarlyConfig_RTCDriver
	   (tVoid)bLoadKDSEarlyConfig_RTCDriver();
	   // EarlyConfig_Touch
	   (tVoid)bLoadKDSEarlyConfig_TouchDriver();
	   // EarlyConfig_LayerManager
	   (tVoid)bLoadKDSearlyConfig_LayerManager();
	   // EarlyConfig_TwoDisplays
	   (tVoid)bLoadKDSEarlyConfig_TwoDisplays();
	   // EarlyConfig_TwoTouchDriver
	//   (tVoid)bLoadKDSearlyConfig_TwoTouchDriver();
   }
   ETG_TRACE_USR2(("bLoadKDSEarlyConfig(): returns with Display: %d, RTCDriver: %d, TouchDriver: %d, LayerManager: %d, TwoDisplays: %d",
		   bIsLoadedKDSEarlyConfig_Display(), bIsLoadedKDSEarlyConfig_RTCDriver(),
		   bIsLoadedKDSEarlyConfig_TouchDriver(), bIsLoadedKDSEarlyConfig_LayerManager(),
		   bIsLoadedKDSEarlyConfig_TwoDisplays()));
   return ((TRUE == bIsLoadedKDSEarlyConfig_Display())
		   && (TRUE == bIsLoadedKDSEarlyConfig_RTCDriver())
		   && (TRUE == bIsLoadedKDSEarlyConfig_TouchDriver())
		   && (TRUE == bIsLoadedKDSEarlyConfig_TwoDisplays())
		   && (TRUE == bIsLoadedKDSEarlyConfig_LayerManager())
		   );
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig()
*
* DESCRIPTION: storing of EarlyConfig data loaded from KDS
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig()
{
   ETG_TRACE_USR4(("bWriteEarlyConfig() entered."));
   // EarlyConfig_Display
   if (FALSE == bWriteEarlyConfig_Display())
   {
	   ETG_TRACE_ERR(("bWriteEarlyConfig: writing of EarlyConfig_Display failed!"));
   }
   // EarlyConfig_RTCDriver
   if (TRUE == bWriteEarlyConfig_RTCDriver())
   {
	   ETG_TRACE_ERR(("bWriteEarlyConfig: writing of EarlyConfig_RTCDriver failed!"));
   }
   // EarlyConfig_Touch
   if (FALSE == bWriteEarlyConfig_TouchDriver())
   {
	   ETG_TRACE_ERR(("bWriteEarlyConfig: writing of EarlyConfig_Touch failed!"));
   }
   // EarlyConfig_LayerManager
   if (FALSE == bWriteEarlyConfig_LayerManager())
   {
	   ETG_TRACE_ERR(("bWriteEarlyConfig: writing of EarlyConfig_LayerManager failed!"));
   }
   // EarlyConfig_TwoDisplays
   if (FALSE == bWriteEarlyConfig_TwoDisplays())
   {
	   ETG_TRACE_ERR(("bWriteEarlyConfig: writing of EarlyConfig_TwoDisplays failed!"));
   }
   // EarlyConfig_TwoTouchDriver
   /*if (FALSE == bWriteEarlyConfig_TwoTouchDriver())
   {
	   ETG_TRACE_ERR(("bWriteEarlyConfig: writing of EarlyConfig_TwoTouch failed!"));
   }*/
   // all ealy config data should be saved and should not be different
   return ((FALSE == bIsChangedEarlyConfig_Display())
		   && (FALSE == bIsChangedEarlyConfig_RTCDriver())
		   && (FALSE == bIsChangedEarlyConfig_TouchDriver())
		   && (FALSE == bIsChangedEarlyConfig_TwoDisplays())
		   && (FALSE == bIsChangedEarlyConfig_LayerManager()));
		   //&& (FALSE == bIsChangedEarlyConfig_TwoTouchDriver()));
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfoEarlyConfig() const
*
* DESCRIPTION: Trace information EarlyConfig
*
* PARAMETER:   const char* strIdentifier
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid dispvidctrl_tclEarlyConfig::vTraceInfoEarlyConfig(const char* strIdentifier) const
{
   ETG_TRACE_USR4(("vTraceInfoEarlyConfig() entered."));
   vTraceInfoEarlyConfig_Display(strIdentifier);
   vTraceInfoEarlyConfig_RTCDriver(strIdentifier);
   vTraceInfoEarlyConfig_TouchDriver(strIdentifier);
   vTraceInfoEarlyConfig_TwoDisplays(strIdentifier);
   vTraceInfoEarlyConfig_TouchScreenConfig(strIdentifier);
   vTraceInfoEarlyConfig_LayerManager(strIdentifier);
   vTraceInfoEarlyConfig_TwoTouchDriver(strIdentifier);
}

/*******************************************************************************
*
* FUNCTION:    bIsEarlyConfig_DisplayResolutionManuallySet()
*
* DESCRIPTION: returns TRUE if EarlyConfig_Display resolution manually set
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsEarlyConfig_DisplayResolutionManuallySet(tVoid) const
{
   ETG_TRACE_USR4(("bIsEarlyConfig_DisplayResolutionManuallySet() entered."));

   tChar _strDisplayResolution_Manual[sizeof(m_strDisplayResolution)] = {0};
   dp_tclDimmingDPEarlyConfig_DisplayResulution_Manually _oDisplayResolutionManual;
   tS32 _s32ReadStatus = _oDisplayResolutionManual.s32GetData(_strDisplayResolution_Manual, sizeof(_strDisplayResolution_Manual)-1);

   // check for update of manual EarlyConfig display resolution
   if (DP_S32_NO_ERR > _oDisplayResolutionManual.s32GetStatus())
   {
	   // not initialized
	   ETG_TRACE_USR4(("bIsEarlyConfig_DisplayResolutionManuallySet: Status DisplayResolution manual: %d.", _oDisplayResolutionManual.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   if (0 < strlen(_strDisplayResolution_Manual))
	   {
		   ETG_TRACE_FATAL(("bIsEarlyConfig_DisplayResolutionManuallySet: Status DisplayResolution manual: (len=%d) %s.",
				   strlen(_strDisplayResolution_Manual), _strDisplayResolution_Manual ));
		   return TRUE;
	   }

   }
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bIsEarlyConfig_DisplayBackwardCompManuallySet()
*
* DESCRIPTION: returns TRUE if EarlyConfig_Display backward comp manually set
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsEarlyConfig_DisplayBackwardCompManuallySet(tVoid) const
{
   ETG_TRACE_USR4(("bIsEarlyConfig_DisplayBackwardCompManuallySet() entered."));

   tBool _bReturn = FALSE;
   dp_tclDimmingDPEarlyConfig_DisplayBackwardComp_Manually _oDisplayBackwardCompManual;
   tS32 _s32Value = _oDisplayBackwardCompManual.tGetData();

   if (DP_S32_NO_ERR != _oDisplayBackwardCompManual.s32GetStatus())
   {
	   ETG_TRACE_FATAL(("bIsEarlyConfig_DisplayBackwardCompManuallySet: Status DisplayBackwardComp manual is %d!",
			   _oDisplayBackwardCompManual.s32GetStatus() ));
   }
   else
   {
	   if ( -1 <= _s32Value && 1 >= _s32Value)
	   {
		   ETG_TRACE_USR2(("bIsEarlyConfig_DisplayBackwardCompManuallySet: Value of DisplayBackwardComp is set manually to %d.",
				   _s32Value ));
		   _bReturn = TRUE;
	   }
	   else
	   {
		   ETG_TRACE_USR2(("bIsEarlyConfig_DisplayBackwardCompManuallySet: Value of DisplayBackwardComp isn't set manually and is %d.",
				   _s32Value ));
	   }
   }

   return _bReturn;
}

/*******************************************************************************
*
* FUNCTION:    bIsEarlyConfig_DisplayLowFrequencyManuallySet()
*
* DESCRIPTION: returns TRUE if EarlyConfig_Display low frequency manually set
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsEarlyConfig_DisplayLowFrequencyManuallySet(tVoid) const
{
   ETG_TRACE_USR4(("bIsEarlyConfig_DisplayLowFrequencyManuallySet() entered."));

   tBool _bReturn = FALSE;
   dp_tclDimmingDPEarlyConfig_DisplayLowFrequency_Manually _oDisplayLowFrequencyManual;
   tS32 _s32Value = _oDisplayLowFrequencyManual.tGetData();

   if (DP_S32_NO_ERR != _oDisplayLowFrequencyManual.s32GetStatus())
   {
	   ETG_TRACE_FATAL(("bIsEarlyConfig_DisplayLowFrequencyManuallySet: Status DisplayLowFrequency manual is %d!",
			   _oDisplayLowFrequencyManual.s32GetStatus() ));
   }
   else
   {
	   if ( -1 <= _s32Value && 1 >= _s32Value)
	   {
		   ETG_TRACE_USR2(("bIsEarlyConfig_DisplayLowFrequencyManuallySet: Value of DisplayLowFrequency is set manually to %d.",
				   _s32Value ));
		   _bReturn = TRUE;
	   }
	   else
	   {
		   ETG_TRACE_USR2(("bIsEarlyConfig_DisplayLowFrequencyManuallySet: Value of DisplayLowFrequency isn't set manually and is %d.",
				   _s32Value ));
	   }
   }

   return _bReturn;
}

/*******************************************************************************
*
* FUNCTION:    bLoadKDSEarlyConfig_Display()
*
* DESCRIPTION: loading KDS EarlyConfig_Display data
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bLoadKDSEarlyConfig_Display()
{
   ETG_TRACE_USR4(("bLoadKDSEarlyConfig_Display() entered."));

   m_bKDSDisplayLoaded = FALSE;
   tU8 _u8LoadedItems = 0;
   tKDS_EarlyConfig_DisplayRaw _tKDSDisplayRaw;
   (tVoid)OSAL_pvMemorySet((tVoid*)&_tKDSDisplayRaw, OSAL_NULL, sizeof(_tKDSDisplayRaw));
   // resolution
   tS32 _s32ReturnCode = DP_s32GetConfigItem( "EarlyConfig_Display", "Resolution",
		   _tKDSDisplayRaw.tDisplayRaw.au8Resolution, sizeof(_tKDSDisplayRaw.tDisplayRaw.au8Resolution), FALSE);
   if  (DP_S32_NO_ERR > _s32ReturnCode)
   {
	   ETG_TRACE_ERR(("bLoadKDSEarlyConfig_Display: EarlyConfig_Display.Resolution could not read (error=0x%08x)", _s32ReturnCode));
   }
   else
   {
	   (tVoid)OSAL_pvMemorySet((tVoid*)m_strDisplayResolution, OSAL_NULL, sizeof(m_strDisplayResolution));
	   (tVoid)OSAL_szStringNCopy(m_strDisplayResolution, (const char*)_tKDSDisplayRaw.tDisplayRaw.au8Resolution, sizeof(m_strDisplayResolution)-1);
	   ++_u8LoadedItems;
   }
   // backward comp.
   _s32ReturnCode = DP_s32GetConfigItem( "EarlyConfig_Display", "BackwardCompatibility",
		   _tKDSDisplayRaw.tDisplayRaw.au8BackwardComp, sizeof(_tKDSDisplayRaw.tDisplayRaw.au8BackwardComp), FALSE);
   if  (DP_S32_NO_ERR > _s32ReturnCode)
   {
	   ETG_TRACE_ERR(("bLoadKDSEarlyConfig_Display: EarlyConfig_Display.BackwardComp could not read (error=0x%08x)", _s32ReturnCode));
   }
   else
   {
	   m_s32DisplayBackwardComp = (tS32)((_tKDSDisplayRaw.tDisplayRaw.au8BackwardComp[0] << 24)
			   | (_tKDSDisplayRaw.tDisplayRaw.au8BackwardComp[1] << 16)
			   | (_tKDSDisplayRaw.tDisplayRaw.au8BackwardComp[2] << 8)
			   | (_tKDSDisplayRaw.tDisplayRaw.au8BackwardComp[3]));
	   ++_u8LoadedItems;
   }
   // low frequency
   _s32ReturnCode = DP_s32GetConfigItem( "EarlyConfig_Display", "LowFrequency",
		   _tKDSDisplayRaw.tDisplayRaw.au8LowFrequency, sizeof(_tKDSDisplayRaw.tDisplayRaw.au8LowFrequency), FALSE);
   if  (DP_S32_NO_ERR > _s32ReturnCode)
   {
	   ETG_TRACE_ERR(("bLoadKDSEarlyConfig_Display: EarlyConfig_Display.LowFrequency could not read (error=0x%08x)", _s32ReturnCode));
   }
   else
   {
	   m_s32DisplayLowFrequency = (tS32)((_tKDSDisplayRaw.tDisplayRaw.au8LowFrequency[0] << 24)
			   | (_tKDSDisplayRaw.tDisplayRaw.au8LowFrequency[1] << 16)
			   | (_tKDSDisplayRaw.tDisplayRaw.au8LowFrequency[2] << 8)
			   | (_tKDSDisplayRaw.tDisplayRaw.au8LowFrequency[3]));
	   ++_u8LoadedItems;
   }
   if (3 == _u8LoadedItems)
   {
	   ETG_TRACE_USR2(("bLoadKDSEarlyConfig_Display: KDS early config display loaded with resolution=%40s, backward comp=%d and low forward=%d",
			   m_strDisplayResolution, m_s32DisplayBackwardComp, m_s32DisplayLowFrequency ));
	   m_bKDSDisplayLoaded = TRUE;
   }

   return bIsLoadedKDSEarlyConfig_Display();
}

/*******************************************************************************
*
* FUNCTION:    bIsValidKDSEarlyConfig_Display()
*
* DESCRIPTION: returns TRUE if loaded KDS data are valid
*              - configured resolution must be known
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - valid | else - not valid}
*
*******************************************************************************/
// possible display resolutions for A-IVI
const char* m_astrEarlyConfigDisplayResolutions[] =
{
		"lq070y5dg36", 			// Sharp-LQ070Y5DG36
		"lq080y5dw02", 			// ITG5 8"
		"samsung_ltr080sl01",	// samsung_ltr080sl01 - 8 inch display
		"samsung_ltr070sl02",	// samsung_ltr070sl02 - 7 inch display
		"samsung_ltr070sl03",	// samsung_ltr070sl03 - 7 inch display
		"auo_c070eat01"         // P-IVI A2 display resolution / timing
};
tBool dispvidctrl_tclEarlyConfig::bIsValidKDSEarlyConfig_Display() const
{
   ETG_TRACE_USR4(("bIsValidKDSEarlyConfig_Display() entered."));

   tBool _bValid = FALSE;
   if (TRUE == bIsLoadedKDSEarlyConfig_Display())
   {
	   for (tU32 _u32Idx = 0; EARLYCONFIG_ARRAY_ELEMENTS(m_astrEarlyConfigDisplayResolutions) > _u32Idx; ++_u32Idx)
	   {
		   if (OSAL_NULL == OSAL_s32StringCompare(m_astrEarlyConfigDisplayResolutions[_u32Idx], m_strDisplayResolution))
		   {
			   _bValid = TRUE;
		   }
	   }
   }
   return _bValid;
}

/*******************************************************************************
*
* FUNCTION:    bIsChangedEarlyConfig_Display()
*
* DESCRIPTION: returns TRUE if KDS data not the same like early config data
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - new | else - not new}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsChangedEarlyConfig_Display() const
{
   ETG_TRACE_USR4(("bIsChangedEarlyConfig_Display() entered."));
   tBool _bChanged = FALSE;
   // Resolution
   dp_tclEarlyConfigDisplayTrResolution _oDisplayResolution;
   tChar _strDisplayResolution[sizeof(m_strDisplayResolution)] = {0};
   tS32 _s32ReadStatus = _oDisplayResolution.s32GetData(_strDisplayResolution, sizeof(_strDisplayResolution)-1);
   if (DP_S32_NO_ERR > _oDisplayResolution.s32GetStatus())
   {
	   // not initialized
	   _bChanged = bIsLoadedKDSEarlyConfig_Display();
	   ETG_TRACE_USR2(("bIsChangedEarlyConfig_Display: Status DisplayResolution: %d.", _oDisplayResolution.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   // could be loaded
	   tS32 _s32Compare = OSAL_s32StringCompare(m_strDisplayResolution,_strDisplayResolution);
	   if (OSAL_NULL != _s32Compare)
	   {
		   _bChanged = bIsLoadedKDSEarlyConfig_Display();
		   ETG_TRACE_USR3(("bIsChangedEarlyConfig_Display: DisplayResolution Compare: %d.", _s32Compare ));
	   }
	   else
	   {
		   ETG_TRACE_USR3(("bIsChangedEarlyConfig_Display: DisplayResolution Compare: %d.", _s32Compare ));
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("bIsChangedEarlyConfig_Display: Read Error DisplayResolution detected %d!", _s32ReadStatus ));
   }
   // BackwardComp
   dp_tclEarlyConfigDisplayTrBackwardComp _oBackwardComp;
   if (DP_S32_NO_ERR != _oBackwardComp.s32GetStatus())
   {
	   // not initialized
	   _bChanged = bIsLoadedKDSEarlyConfig_Display();
	   ETG_TRACE_USR2(("bIsChangedEarlyConfig_Display: Status BackwardComp: %d.", _oBackwardComp.s32GetStatus() ));
   }
   else if (m_s32DisplayBackwardComp != _oBackwardComp.tGetData())
   {
	   _bChanged = bIsLoadedKDSEarlyConfig_Display();
	   ETG_TRACE_USR3(("bIsChangedEarlyConfig_Display: BackwardComp Compare: %d.", (m_s32DisplayBackwardComp != _oBackwardComp.tGetData()) ));
   }
   else
   {
	   ETG_TRACE_USR3(("bIsChangedEarlyConfig_Display: No BackwardComp changes detected!" ));
   }
   // LowFrequency
   dp_tclEarlyConfigDisplayTrLowFrequency _oLowFrequence;
   if (DP_S32_NO_ERR != _oLowFrequence.s32GetStatus())
   {
	   // not initialized
	   _bChanged = bIsLoadedKDSEarlyConfig_Display();
	   ETG_TRACE_USR2(("bIsChangedEarlyConfig_Display: Status LowFrequence: %d.", _oLowFrequence.s32GetStatus() ));
   }
   else if (m_s32DisplayLowFrequency != _oLowFrequence.tGetData())
   {
	   _bChanged = bIsLoadedKDSEarlyConfig_Display();
	   ETG_TRACE_USR3(("bIsChangedEarlyConfig_Display: LowFrequence Compare: %d.", (m_s32DisplayLowFrequency != _oLowFrequence.tGetData()) ));
   }
   else
   {
	   ETG_TRACE_USR3(("bIsChangedEarlyConfig_Display: No LowFrequence changes detected!" ));
   }
   return _bChanged;
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig_Display()
*
* DESCRIPTION: write the EarlyConfig_Display
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig_Display()
{
   ETG_TRACE_USR4(("bWriteEarlyConfig_Display() entered."));

   tBool _bReturn = TRUE;
   tBool _bDoSetResolution = FALSE;
   char _strDisplayResolution[sizeof(m_strDisplayResolution)];
   (tVoid)memset((tVoid*)_strDisplayResolution, OSAL_NULL, sizeof(_strDisplayResolution));
   tBool _bDoSetBackwardComp = FALSE;
   tS32 _s32BackwardComp = -1;
   tBool _bDoSetLowFrequency = FALSE;
   tS32 _s32LowFrequency = -1;

   if (0 != u8GetTouchScreenConfig())
   {
	   if (TRUE == bIsValidKDSCustom_TouchScreenConfig())
	   {
		   if (FALSE == bIsEarlyConfig_DisplayResolutionManuallySet())
		   {
			   (tVoid)OSAL_szStringNCopy(_strDisplayResolution, cstrGetCustomDisplayDriver(), sizeof(_strDisplayResolution)-1);
			   _bDoSetResolution = TRUE;
		   }
		   if (FALSE == bIsEarlyConfig_DisplayBackwardCompManuallySet())
		   {
			   _s32BackwardComp = s32GetCustomDisplayDriverBackwardComp();
			   _bDoSetBackwardComp = TRUE;
		   }
		   if (FALSE == bIsEarlyConfig_DisplayLowFrequencyManuallySet())
		   {
			   _s32LowFrequency = s32GetCustomDisplayDriverLowFrequency();
			   _bDoSetLowFrequency = TRUE;
		   }
	   }
	   else
	   {
		   _bReturn = FALSE;
	   }
   }
   else if (TRUE == bIsValidKDSEarlyConfig_Display())
   {
	   if (FALSE == bIsEarlyConfig_DisplayResolutionManuallySet())
	   {
		   (tVoid)OSAL_szStringNCopy(_strDisplayResolution, m_strDisplayResolution, sizeof(_strDisplayResolution)-1);
		   _bDoSetResolution = TRUE;
	   }
	   if (FALSE == bIsEarlyConfig_DisplayBackwardCompManuallySet())
	   {
		   _s32BackwardComp = m_s32DisplayBackwardComp;
		   _bDoSetBackwardComp = TRUE;
	   }
	   if (FALSE == bIsEarlyConfig_DisplayLowFrequencyManuallySet())
	   {
		   _s32LowFrequency = m_s32DisplayLowFrequency;
		   _bDoSetLowFrequency = TRUE;
	   }
   }
   else
   {
	   _bReturn = FALSE;
   }
   if (TRUE == _bReturn)
   {
	   if (TRUE == _bDoSetResolution)
	   {
		   dp_tclEarlyConfigDisplayTrResolution _oDisplayResolution;
		   if (DP_S32_NO_ERR != _oDisplayResolution.s32SetData(_strDisplayResolution))
		   {
			   ETG_TRACE_FATAL(("bWriteEarlyConfig_Display: Failed to set display resolution %s to early config!",
					   _strDisplayResolution ));
			   _bReturn = FALSE;
		   }
		   else
		   {
			   ETG_TRACE_USR2(("bWriteEarlyConfig_Display: Successful set display resolution %s to early config!",
					   _strDisplayResolution ));
		   }
	   }
	   if (TRUE == _bDoSetBackwardComp)
	   {
		   dp_tclEarlyConfigDisplayTrBackwardComp _oBackwardComp;
		   if (DP_S32_NO_ERR != _oBackwardComp.s32SetData(_s32BackwardComp))
		   {
			   ETG_TRACE_FATAL(("bWriteEarlyConfig_Display: Failed to set display backlight comp %d to early config!",
					   _s32BackwardComp ));
			   _bReturn = FALSE;
		   }
		   else
		   {
			   ETG_TRACE_USR2(("bWriteEarlyConfig_Display: Successful set display backlight comp %d to early config!",
					   _s32BackwardComp ));
		   }
	   }
	   if (TRUE == _bDoSetLowFrequency)
	   {
		   dp_tclEarlyConfigDisplayTrLowFrequency _oLowFrequence;
		   if (DP_S32_NO_ERR != _oLowFrequence.s32SetData(_s32LowFrequency))
		   {
			   ETG_TRACE_FATAL(("bWriteEarlyConfig_Display: Failed to set display low frequency %d to early config!",
					   _s32LowFrequency ));
			   _bReturn = FALSE;
		   }
		   else
		   {
			   ETG_TRACE_USR2(("bWriteEarlyConfig_Display: Successful set display low frequency %d to early config!",
					   _s32LowFrequency ));
		   }
	   }
	   dp_tclSrvIf _oDpSrvIf;
	   tU32 _u32DpID = DP_U32_POOL_ID_EARLYCONFIGDISPLAY;
	   tS32 _s32ErrorCode = _oDpSrvIf.s32StorePoolNow(_u32DpID);
	   if (DP_S32_NO_ERR != _s32ErrorCode)
	   {
	     ETG_TRACE_FATAL(("bWriteEarlyConfig_Display: storing of PDD 0x%08x failed with error code 0x%08x!", _u32DpID, _s32ErrorCode ));
	   }
   }
   return _bReturn;
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig_DisplayResolution()
*
* DESCRIPTION: write the EarlyConfig_Display resolution directly without validation
*
* PARAMETER:   const tChar* pStrResolution (max. 31 char)
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig_DisplayResolution(const tChar* pStrResolution) const
{
   ETG_TRACE_USR4(("bWriteEarlyConfig_DisplayResolution(%31s) entered.", pStrResolution));

   tBool _bChanged = FALSE;
   tChar _strDisplayResolution_Manual[sizeof(m_strDisplayResolution)] = {0};
   dp_tclDimmingDPEarlyConfig_DisplayResulution_Manually _oDisplayResolutionManual;
   tS32 _s32ReadStatus = _oDisplayResolutionManual.s32GetData(_strDisplayResolution_Manual, sizeof(_strDisplayResolution_Manual)-1);

   // check for update of manual EarlyConfig display resolution
   if (DP_S32_NO_ERR > _oDisplayResolutionManual.s32GetStatus())
   {
	   // not initialized
	   _bChanged = TRUE;
	   ETG_TRACE_USR1(("bWriteEarlyConfig_DisplayResolution: Status DisplayResolution manual: %d.", _oDisplayResolutionManual.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   // could be loaded
	   tS32 _s32Compare = OSAL_s32StringCompare(pStrResolution, _strDisplayResolution_Manual);
	   if (OSAL_NULL != _s32Compare)
	   {
		   _bChanged = TRUE;
		   ETG_TRACE_USR1(("bWriteEarlyConfig_DisplayResolution: DisplayResolution manual changed (Compare: %d).", _s32Compare ));
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_DisplayResolution: DisplayResolution manual not changed (Compare: %d).", _s32Compare ));
	   }
   }
   // set EarlyConfig display resolution manual
   if (TRUE == _bChanged)
   {
	   (tVoid)OSAL_pvMemorySet((tVoid*)_strDisplayResolution_Manual, OSAL_NULL, sizeof(_strDisplayResolution_Manual));
	   (tVoid)OSAL_szStringNCopy(_strDisplayResolution_Manual, pStrResolution, sizeof(_strDisplayResolution_Manual)-1);

	   // write EarlyConfig data pool
	   dp_tclEarlyConfigDisplayTrResolution _oDisplayResolution;
	   if (DP_S32_NO_ERR != _oDisplayResolution.s32SetData(_strDisplayResolution_Manual))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_DisplayResolution: Failed to manually set EarlyConfig display resolution %s!",
				   _strDisplayResolution_Manual ));
		   return FALSE;
	   }
	   else
	   {
		   ETG_TRACE_USR1(("bWriteEarlyConfig_DisplayResolution: Successful manually set EarlyConfig display resolution %s.",
				   _strDisplayResolution_Manual ));
	   }
	   // write own display resolution data pool
	   if (DP_S32_NO_ERR != _oDisplayResolutionManual.s32SetData(_strDisplayResolution_Manual))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_DisplayResolution: Failed to manually set own display resolution %s!",
				   _strDisplayResolution_Manual ));
	   }
	   else
	   {
		   ETG_TRACE_USR1(("bWriteEarlyConfig_DisplayResolution: Successful manually set own display resolution %s.",
				   _strDisplayResolution_Manual ));
		   return TRUE;
	   }
   }
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig_BackwardComp()
*
* DESCRIPTION: Writes the early config display backward comp configuration
*              (possible values: -1, 0, 1). Please use an undefined value to
*              reset manually settings.
*
* PARAMETER:   const tS32 s32BackwardComp
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig_BackwardComp(const tS32 s32BackwardComp) const
{
   ETG_TRACE_USR4(("bWriteEarlyConfig_BackwardComp(%d) entered.", s32BackwardComp));

   tBool _bReturn = TRUE;
   tS32 _s32Return = DP_S32_NO_ERR;

   dp_tclDimmingDPEarlyConfig_DisplayBackwardComp_Manually _oBackwardComp_Manually;
   _s32Return = _oBackwardComp_Manually.s32SetData(s32BackwardComp);
   if (DP_S32_NO_ERR != _s32Return)
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_BackwardComp: Failed to set manual display backward comp %d with return %d!",
			   s32BackwardComp, _s32Return ));
	   _bReturn = FALSE;
   }
   else
   {
	   ETG_TRACE_USR2(("bWriteEarlyConfig_BackwardComp: Successful set manual display backward comp %d with return %d.",
			   s32BackwardComp, _s32Return ));
   }

   dp_tclEarlyConfigDisplayTrBackwardComp _oBackwardComp;
   _s32Return = _oBackwardComp.s32SetData(s32BackwardComp);
   if (DP_S32_NO_ERR != _s32Return)
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_BackwardComp: Failed to set display backlight comp %d to early config with return %d!",
			   s32BackwardComp, _s32Return ));
	   _bReturn = FALSE;
   }
   else
   {
	   ETG_TRACE_USR2(("bWriteEarlyConfig_BackwardComp: Successful set display backlight comp %d to early config with return %d.",
			   s32BackwardComp, _s32Return ));
   }

   return _bReturn;
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig_LowFrequency()
*
* DESCRIPTION: Writes the early config display low frequency configuration
*              (possible values: -1, 0, 1). Please use an undefined value to
*              reset manually settings.
*
* PARAMETER:   const tS32 s32LowFrequency
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig_LowFrequency(const tS32 s32LowFrequency) const
{
   ETG_TRACE_USR4(("bWriteEarlyConfig_LowFrequency(%d) entered.", s32LowFrequency));

   tBool _bReturn = TRUE;
   tS32 _s32Return = DP_S32_NO_ERR;

   dp_tclDimmingDPEarlyConfig_DisplayLowFrequency_Manually _oLowFrequency_Manually;
   _s32Return = _oLowFrequency_Manually.s32SetData(s32LowFrequency);
   if (DP_S32_NO_ERR != _s32Return)
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_LowFrequency: Failed to set manual display low frequency %d with return %d!",
			   s32LowFrequency, _s32Return ));
	   _bReturn = FALSE;
   }
   else
   {
	   ETG_TRACE_USR2(("bWriteEarlyConfig_LowFrequency: Successful set manual display low frequency %d with return %d.",
			   s32LowFrequency, _s32Return ));
   }

   dp_tclEarlyConfigDisplayTrLowFrequency _oLowFrequency;
   _s32Return = _oLowFrequency.s32SetData(s32LowFrequency);
   if (DP_S32_NO_ERR != _s32Return)
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_LowFrequency: Failed to set display low frequency %d to early config with return %d!",
			   s32LowFrequency, _s32Return ));
	   _bReturn = FALSE;
   }
   else
   {
	   ETG_TRACE_USR2(("bWriteEarlyConfig_LowFrequency: Successful set display low frequency %d to early config with return %d.",
			   s32LowFrequency, _s32Return ));
   }

   return _bReturn;
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfoEarlyConfig_Display() const
*
* DESCRIPTION: Trace information EarlyConfig_Display
*
* PARAMETER:   const char* strIdentifier
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid dispvidctrl_tclEarlyConfig::vTraceInfoEarlyConfig_Display(const char* strIdentifier) const
{
	ETG_TRACE_USR4(("vTraceInfoEarlyConfig_Display() entered."));
	ETG_TRACE_FATAL(("%26s: <<< EarlyConfig_Display >>> ", strIdentifier ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "   DISP_RESOL_MANUAL",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsEarlyConfig_DisplayResolutionManuallySet()) ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "   DISP_BCOMP_MANUAL",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsEarlyConfig_DisplayBackwardCompManuallySet()) ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "   DISP_LFREQ_MANUAL",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsEarlyConfig_DisplayLowFrequencyManuallySet()) ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "     KDS_DISP_LOADED",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsLoadedKDSEarlyConfig_Display()) ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "      KDS_DISP_VALID",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsValidKDSEarlyConfig_Display()) ));
	ETG_TRACE_FATAL(("%26s: %30s=%50s ", strIdentifier, "      KDS_DISP_RESOL",
			m_strDisplayResolution ));
	ETG_TRACE_FATAL(("%26s: %30s=%d ",   strIdentifier, "   KDS_DISP_BACKCOMP",
			m_s32DisplayBackwardComp ));
	ETG_TRACE_FATAL(("%26s: %30s=%d ",   strIdentifier, "    KDS_DISP_LOWFREQ",
			m_s32DisplayLowFrequency ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "     EC_DISP_CHANGED",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsChangedEarlyConfig_Display()) ));
	dp_tclEarlyConfigDisplayTrResolution _oDisplayResolution;
	tChar _strDisplayResolution[sizeof(m_strDisplayResolution)] = {0};
	(tVoid)_oDisplayResolution.s32GetData(_strDisplayResolution, sizeof(_strDisplayResolution)-1);
	ETG_TRACE_FATAL(("%26s: %30s=%50s ", strIdentifier, "       EC_DISP_RESOL",
			_strDisplayResolution ));
	dp_tclEarlyConfigDisplayTrBackwardComp _oBackwardComp;
	ETG_TRACE_FATAL(("%26s: %30s=%d ",   strIdentifier, "    EC_DISP_BACKCOMP",
			_oBackwardComp.tGetData() ));
	dp_tclEarlyConfigDisplayTrLowFrequency _oLowFrequence;
	ETG_TRACE_FATAL(("%26s: %30s=%d ",   strIdentifier, "     EC_DISP_LOWFREQ",
			_oLowFrequence.tGetData() ));
}

/*******************************************************************************
*
* FUNCTION:    bIsEarlyConfig_RTCDriverManuallySet()
*
* DESCRIPTION: returns TRUE if EarlyConfig_RTC driver name manually set
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsEarlyConfig_RTCDriverManuallySet(tVoid) const
{
   ETG_TRACE_USR4(("bIsEarlyConfig_RTCDriverManuallySet() entered."));

   tChar _strRTCDriverName_Manual[sizeof(m_strRTCDriverName)] = {0};
   dp_tclDimmingDPEarlyConfig_RTCDriverName_Manually _oRTCDriverNameManual;
   tS32 _s32ReadStatus = _oRTCDriverNameManual.s32GetData(_strRTCDriverName_Manual, sizeof(_strRTCDriverName_Manual)-1);

   // check for update of manual EarlyConfig display resolution
   if (DP_S32_NO_ERR > _oRTCDriverNameManual.s32GetStatus())
   {
	   // not initialized
	   ETG_TRACE_FATAL(("bIsEarlyConfig_RTCDriverManuallySet: Status RTCDriverName manual: %d.", _oRTCDriverNameManual.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   if (0 < strlen(_strRTCDriverName_Manual))
	   {
		   return TRUE;
	   }

   }
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bLoadKDSEarlyConfig_RTCDriver()
*
* DESCRIPTION: loading KDS EarlyConfig_RTCDriver data
*              - only for project A-IVI depending mouting state of GNSS module
*                ... GNSS module mounted     - RTC driver name "rtc-inc"
*                ... GNSS module not mounted - RTC driver name "rtc-rx6110-i2c"
*
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
// possible RTC driver names for A-IVI
const char* m_astrEarlyConfigRTCDriverNames[] =
{
		// depending of CMVariantCoding.
		"rtc-rx6110-i2c", // RTC connected to iMx
		"rtc-inc"         // RTC on the GNSS(TESEO), which is connected to the V850
};
const char* m_astrEarlyConfigRTCDriverNamesByEnum[] =
{
		// depending of CMVariantCoding.RTC as CMRTCInfoType
		"",               // [0x00] not available (Information is not available.)
		"rtc-inc",        // [0x01] GNSS (GNSS provdes RTC information.)
		"rtc-rx6110-i2c", // [0x02] RX6110
		"rtc-rv8803"      // [0x03] RX8900
};
tBool dispvidctrl_tclEarlyConfig::bLoadKDSEarlyConfig_RTCDriver()
{
   ETG_TRACE_USR4(("bLoadKDSEarlyConfig_RTCDriver() entered."));

   m_bKDSRTCDriverLoaded = FALSE;
   dp_tclKdsCMVariantCoding _oVariantCoding;
   tU8 _u8Data = 0;
   // depending of RTC
   tU8 _u8Status = _oVariantCoding.u8GetRTC(_u8Data);
   if (DP_U8_ELEM_STATUS_VALID != _u8Status)
   {
	   ETG_TRACE_ERR(("bLoadKDSEarlyConfig_RTCDriver: VariantCoding.RTC for EarlyConfig_RTCDriver could not read (status=0x%08x)", _u8Status ));
   }
   else if ((DISPVIDCTRL_C_U16_ARRAYELEMENTS(m_astrEarlyConfigRTCDriverNamesByEnum) > (tU16)_u8Data) && (0 < _u8Data))
   {
	   (tVoid)OSAL_pvMemorySet((tVoid*)m_strRTCDriverName, OSAL_NULL, sizeof(m_strRTCDriverName));
	   (tVoid)OSAL_szStringNCopy(m_strRTCDriverName, m_astrEarlyConfigRTCDriverNamesByEnum[_u8Data], sizeof(m_strRTCDriverName)-1);
	   ETG_TRACE_USR2(("bLoadKDSEarlyConfig_RTCDriver: KDS early config rtc driver loaded with driver name=%64s (data_RTC: 0x%02x, status: %d)",
			   m_strRTCDriverName, _u8Data, _u8Status ));
	   m_bKDSRTCDriverLoaded = TRUE;
	   return bIsLoadedKDSEarlyConfig_RTCDriver();
   }
   // depending of GNSS mounted
   _u8Status = _oVariantCoding.u8GetGNSS(_u8Data);
   if (DP_U8_ELEM_STATUS_VALID != _u8Status)
   {
	   ETG_TRACE_ERR(("bLoadKDSEarlyConfig_RTCDriver: VariantCoding.GNSS for EarlyConfig_RTCDriver could not read (status=0x%08x)", _u8Status ));
   }
   else if ((0 < _u8Data) && (2 >= _u8Data))
   {
	   (tVoid)OSAL_pvMemorySet((tVoid*)m_strRTCDriverName, OSAL_NULL, sizeof(m_strRTCDriverName));
	   if (1 == _u8Data)
		   (tVoid)OSAL_szStringNCopy(m_strRTCDriverName, m_astrEarlyConfigRTCDriverNames[1], sizeof(m_strRTCDriverName)-1);
	   else
		   (tVoid)OSAL_szStringNCopy(m_strRTCDriverName, m_astrEarlyConfigRTCDriverNames[0], sizeof(m_strRTCDriverName)-1);
	   ETG_TRACE_USR2(("bLoadKDSEarlyConfig_RTCDriver: KDS early config rtc driver loaded with driver name=%64s (data_GNSS: 0x%02x, status: %d)",
			   m_strRTCDriverName, _u8Data, _oVariantCoding.u8GetGNSS(_u8Data) ));
	   m_bKDSRTCDriverLoaded = TRUE;
   }
   return bIsLoadedKDSEarlyConfig_RTCDriver();
}

/*******************************************************************************
*
* FUNCTION:    bIsValidKDSEarlyConfig_RTCDriver()
*
* DESCRIPTION: returns TRUE if loaded KDS data are valid
*              - configured RTC driver name must be known
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - valid | else - not valid}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsValidKDSEarlyConfig_RTCDriver() const
{
   ETG_TRACE_USR4(("bIsValidKDSEarlyConfig_RTCDriver() entered."));

   if (TRUE == bIsLoadedKDSEarlyConfig_RTCDriver())
   {
	   for (tU16 _u16Idx = 0; DISPVIDCTRL_C_U16_ARRAYELEMENTS(m_astrEarlyConfigRTCDriverNamesByEnum) > _u16Idx; ++_u16Idx)
	   {
		   if (OSAL_NULL == OSAL_s32StringCompare(m_astrEarlyConfigRTCDriverNamesByEnum[_u16Idx], m_strRTCDriverName)
			   && 0 < OSAL_u32StringLength(m_strRTCDriverName))
		   {
			   return TRUE;
		   }
	   }
	   for (tU16 _u16Idx = 0; DISPVIDCTRL_C_U16_ARRAYELEMENTS(m_astrEarlyConfigRTCDriverNames) > _u16Idx; ++_u16Idx)
	   {
		   if (OSAL_NULL == OSAL_s32StringCompare(m_astrEarlyConfigRTCDriverNames[_u16Idx], m_strRTCDriverName))
		   {
			   return TRUE;
		   }
	   }
   }
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bIsChangedEarlyConfig_RTCDriver()
*
* DESCRIPTION: returns TRUE if KDS data not the same like early config data
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - new | else - not new}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsChangedEarlyConfig_RTCDriver() const
{
   ETG_TRACE_USR4(("bIsChangedEarlyConfig_RTCDriver() entered."));
   tBool _bChanged = FALSE;
   // driver name
   dp_tclEarlyConfigRTCDriverTrRtcDriverName _oRTCDriverName;
   tChar _strRTCDriverName[sizeof(m_strRTCDriverName)];
   (tVoid)OSAL_pvMemorySet((tVoid*)_strRTCDriverName, OSAL_NULL, sizeof(_strRTCDriverName));
   tS32 _s32ReadStatus = _oRTCDriverName.s32GetData(_strRTCDriverName, sizeof(_strRTCDriverName)-1);
   if (DP_S32_NO_ERR > _oRTCDriverName.s32GetStatus())
   {
	   // not initialized
	   _bChanged = bIsLoadedKDSEarlyConfig_RTCDriver();
	   ETG_TRACE_USR2(("bIsChangedEarlyConfig_RTCDriver: Status RTCDriverName: %d.", _oRTCDriverName.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   // could be loaded
	   tS32 _s32Compare = OSAL_s32StringCompare(m_strRTCDriverName,_strRTCDriverName);
	   if (OSAL_NULL != _s32Compare)
	   {
		   _bChanged = bIsLoadedKDSEarlyConfig_RTCDriver();
		   ETG_TRACE_USR3(("bIsChangedEarlyConfig_RTCDriver: RTCDriverName Compare: %d.", _s32Compare ));
	   }
	   else
	   {
		   ETG_TRACE_USR3(("bIsChangedEarlyConfig_RTCDriver: RTCDriverName Compare: %d.", _s32Compare ));
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("bIsChangedEarlyConfig_RTCDriver: Read Error RTCDriverName detected %d!", _s32ReadStatus ));
   }
   return _bChanged;
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig_RTCDriver()
*
* DESCRIPTION: write the EarlyConfig_RTCDriver
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig_RTCDriver()
{
   ETG_TRACE_USR4(("bWriteEarlyConfig_RTCDriver() entered."));
   if (TRUE == bIsValidKDSEarlyConfig_RTCDriver() && TRUE == bIsChangedEarlyConfig_RTCDriver()
	   && FALSE == bIsEarlyConfig_RTCDriverManuallySet())
   {
	   dp_tclEarlyConfigRTCDriverTrRtcDriverName _oRTCDriverName;
	   if (DP_S32_NO_ERR != _oRTCDriverName.s32SetData(m_strRTCDriverName))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_RTCDriver: Failed to set rtc driver name %s to early config!",
				   m_strRTCDriverName ));
		   return FALSE;
	   }
	   else
	   {
		   ETG_TRACE_USR2(("bWriteEarlyConfig_RTCDriver: Successful set rtc driver name %s to early config!",
				   m_strRTCDriverName ));
	   }
	   ETG_TRACE_ERRMEM(( "VD_DIMMING: early config rtc driver set with driver name=%64s", m_strRTCDriverName ));
	   dp_tclSrvIf _oDpSrvIf;
	   tU32 _u32DpID = DP_U32_POOL_ID_EARLYCONFIGRTCDRIVER;
	   tS32 _s32ErrorCode = _oDpSrvIf.s32StorePoolNow(_u32DpID);
	   if (DP_S32_NO_ERR != _s32ErrorCode)
	   {
	     ETG_TRACE_FATAL(("bWriteEarlyConfig_RTCDriver: storing of PDD 0x%08x failed with error code 0x%08x!", _u32DpID, _s32ErrorCode ));
	   }
	   return TRUE;
   }
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig_RTCDriverName()
*
* DESCRIPTION: write the EarlyConfig_RTC driver name directly without validation
*
* PARAMETER:   const tChar* pStrDriverName (max. 63 char)
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig_RTCDriverName(const tChar* pStrDriverName) const
{
   ETG_TRACE_USR4(("bWriteEarlyConfig_RTCDriverName(%63s) entered.", pStrDriverName));

   tBool _bChanged = FALSE;
   tChar _strRTCDriverName_Manual[sizeof(m_strRTCDriverName)] = {0};
   dp_tclDimmingDPEarlyConfig_RTCDriverName_Manually _oRTCDriverNameManual;
   tS32 _s32ReadStatus = _oRTCDriverNameManual.s32GetData(_strRTCDriverName_Manual, sizeof(_strRTCDriverName_Manual)-1);

   // check for update of manual EarlyConfig RTC driver name
   if (DP_S32_NO_ERR > _oRTCDriverNameManual.s32GetStatus())
   {
	   // not initialized
	   _bChanged = TRUE;
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_RTCDriverName: Status RTCDriverName manual: %d.", _oRTCDriverNameManual.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   // could be loaded
	   tS32 _s32Compare = OSAL_s32StringCompare(pStrDriverName, _strRTCDriverName_Manual);
	   if (OSAL_NULL != _s32Compare)
	   {
		   _bChanged = TRUE;
		   ETG_TRACE_USR1(("bWriteEarlyConfig_RTCDriverName: RTCDriverName manual changed (Compare: %d).", _s32Compare ));
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_RTCDriverName: RTCDriverName manual not changed (Compare: %d).", _s32Compare ));
	   }
   }
   // set EarlyConfig RTC driver name manual
   if (TRUE == _bChanged)
   {
	   (tVoid)OSAL_pvMemorySet((tVoid*)_strRTCDriverName_Manual, OSAL_NULL, sizeof(_strRTCDriverName_Manual));
	   (tVoid)OSAL_szStringNCopy(_strRTCDriverName_Manual, pStrDriverName, sizeof(_strRTCDriverName_Manual)-1);

	   // write EarlyConfig data pool
	   dp_tclEarlyConfigRTCDriverTrRtcDriverName _oRTCDriverName;
	   if (DP_S32_NO_ERR != _oRTCDriverName.s32SetData(_strRTCDriverName_Manual))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_RTCDriverName: Failed to manually set EarlyConfig RTC driver name %s!",
				   _strRTCDriverName_Manual ));
		   return FALSE;
	   }
	   else
	   {
		   ETG_TRACE_USR1(("bWriteEarlyConfig_RTCDriverName: Successful manually set EarlyConfig RTC driver name %s.",
				   _strRTCDriverName_Manual ));
	   }
	   // write own RTC driver name data pool
	   if (DP_S32_NO_ERR != _oRTCDriverNameManual.s32SetData(_strRTCDriverName_Manual))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_RTCDriverName: Failed to manually set own RTC driver name %s!",
				   _strRTCDriverName_Manual ));
	   }
	   else
	   {
		   ETG_TRACE_USR1(("bWriteEarlyConfig_RTCDriverName: Successful manually set own RTC driver name %s.",
				   _strRTCDriverName_Manual ));
		   return TRUE;
	   }
   }

   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfoEarlyConfig_RTCDriver() const
*
* DESCRIPTION: Trace information EarlyConfig_Display
*
* PARAMETER:   const char* strIdentifier
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid dispvidctrl_tclEarlyConfig::vTraceInfoEarlyConfig_RTCDriver(const char* strIdentifier) const
{
	ETG_TRACE_USR4(("vTraceInfoEarlyConfig_RTCDriver() entered."));
	ETG_TRACE_FATAL(("%26s: <<< EarlyConfig_RTCDriver >>> ", strIdentifier ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "       RTCDRV_MANUAL",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsEarlyConfig_RTCDriverManuallySet()) ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "   KDS_RTCDRV_LOADED",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsLoadedKDSEarlyConfig_RTCDriver()) ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "    KDS_RTCDRV_VALID",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsValidKDSEarlyConfig_RTCDriver()) ));
	ETG_TRACE_FATAL(("%26s: %30s=%64s ", strIdentifier, "     KDS_RTCDRV_NAME",
			m_strRTCDriverName ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "   EC_RTCDRV_CHANGED",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsChangedEarlyConfig_RTCDriver()) ));
	dp_tclEarlyConfigRTCDriverTrRtcDriverName _oRTCDriverName;
	tChar _strRTCDriverName[sizeof(m_strRTCDriverName)];
	(tVoid)OSAL_pvMemorySet((tVoid*)_strRTCDriverName, OSAL_NULL, sizeof(_strRTCDriverName));
	(tVoid)_oRTCDriverName.s32GetData(_strRTCDriverName,sizeof(_strRTCDriverName)-1);
	ETG_TRACE_FATAL(("%26s: %30s=%64s ", strIdentifier, "      EC_RTCDRV_NAME",
			_strRTCDriverName ));
}

/*******************************************************************************
*
* FUNCTION:    bIsEarlyConfig_TouchDriverManuallySet()
*
* DESCRIPTION: returns TRUE if EarlyConfig_Touch driver name manually set
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsEarlyConfig_TouchDriverManuallySet(tVoid) const
{
   ETG_TRACE_USR4(("bIsEarlyConfig_TouchDriverManuallySet() entered."));

   tChar _strTouchDriverName_Manual[sizeof(m_strTouchDriverName)] = {0};
   dp_tclDimmingDPEarlyConfig_TouchDriverName_Manually _oTouchDriverNameManual;
   tS32 _s32ReadStatus = _oTouchDriverNameManual.s32GetData(_strTouchDriverName_Manual, sizeof(_strTouchDriverName_Manual)-1);

   // check for update of manual EarlyConfig display resolution
   if (DP_S32_NO_ERR > _oTouchDriverNameManual.s32GetStatus())
   {
	   // not initialized
	   ETG_TRACE_FATAL(("bIsEarlyConfig_TouchDriverManuallySet: Status Touch Driver Name manual: %d.", _oTouchDriverNameManual.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   if (0 < strlen(_strTouchDriverName_Manual))
	   {
		   return TRUE;
	   }

   }
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bIsEarlyConfig_TouchDriverConfigFileManuallySet()
*
* DESCRIPTION: returns TRUE if EarlyConfig_Touch driver name manually set
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsEarlyConfig_TouchDriverConfigFileManuallySet(tVoid) const
{
   ETG_TRACE_USR4(("bIsEarlyConfig_TouchDriverConfigFileManuallySet() entered."));

   tChar _strTouchConfigFileName_Manual[sizeof(m_strTouchConfigFileName)] = {0};
   dp_tclDimmingDPEarlyConfig_TouchConfigFile_Manually _oTouchConfigFileNameManual;
   tS32 _s32ReadStatus = _oTouchConfigFileNameManual.s32GetData(_strTouchConfigFileName_Manual, sizeof(_strTouchConfigFileName_Manual)-1);

   // check for update of manual EarlyConfig display resolution
   if (DP_S32_NO_ERR > _oTouchConfigFileNameManual.s32GetStatus())
   {
	   // not initialized
	   ETG_TRACE_FATAL(("bIsEarlyConfig_TouchDriverConfigFileManuallySet: Status Touch Driver Config File manual: %d.", _oTouchConfigFileNameManual.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   if (0 < strlen(_strTouchConfigFileName_Manual))
	   {
		   return TRUE;
	   }

   }
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bIsEarlyConfig_TouchI2CAddressManuallySet()
*
* DESCRIPTION: returns TRUE if EarlyConfig_Touch driver I2C address manually set
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsEarlyConfig_TouchI2CAddressManuallySet(tVoid) const
{
   ETG_TRACE_USR4(("bIsEarlyConfig_TouchI2CAddressManuallySet() entered."));

   tChar _strTouchDriverI2CAddress_Manual[DP_U8_NORLEN_EARLYCONFIG_TOUCHI2CADDRESS] = {0};
   dp_tclDimmingDPEarlyConfig_TouchI2CAddress_Manually _oTouchDriverI2CAddressManual;
   tS32 _s32ReadStatus = _oTouchDriverI2CAddressManual.s32GetData(_strTouchDriverI2CAddress_Manual, sizeof(_strTouchDriverI2CAddress_Manual)-1);

   // check for update of manual EarlyConfig display resolution
   if (DP_S32_NO_ERR > _oTouchDriverI2CAddressManual.s32GetStatus())
   {
	   // not initialized
	   ETG_TRACE_FATAL(("bIsEarlyConfig_TouchDriverConfigFileManuallySet: Status TouchDriverI2CAddress manual: %d.", _oTouchDriverI2CAddressManual.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   if (0 < strlen(_strTouchDriverI2CAddress_Manual))
	   {
		   return TRUE;
	   }

   }
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bLoadKDSEarlyConfig_TouchDriver()
*
* DESCRIPTION: loading KDS EarlyConfig_TouchDriver data
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bLoadKDSEarlyConfig_TouchDriver()
{
   ETG_TRACE_USR4(("bLoadKDSEarlyConfig_TouchDriver() entered."));

   m_bKDSTouchDriverLoaded = FALSE;
   tU8 _u8LoadedItems = 0;
   tKDS_EarlyConfig_TouchDriverRaw _tKDSTouchDriverRaw;
   (tVoid)OSAL_pvMemorySet((tVoid*)&_tKDSTouchDriverRaw, OSAL_NULL, sizeof(_tKDSTouchDriverRaw));
   // driver name
   tS32 _s32ReturnCode = DP_s32GetConfigItem( "EarlyConfig_Touch", "TouchDriverName",
		   _tKDSTouchDriverRaw.tTouchDriverRaw.au8Name, sizeof(_tKDSTouchDriverRaw.tTouchDriverRaw.au8Name), FALSE);
   if  (DP_S32_NO_ERR > _s32ReturnCode)
   {
	   ETG_TRACE_ERR(("bLoadKDSEarlyConfig_TouchDriver: EarlyConfig_TouchDriver.Name could not read (error=0x%08x)", _s32ReturnCode));
   }
   else
   {
	   (tVoid)OSAL_pvMemorySet((tVoid*)m_strTouchDriverName, OSAL_NULL, sizeof(m_strTouchDriverName));
	   (tVoid)OSAL_szStringNCopy(m_strTouchDriverName, (const char*)_tKDSTouchDriverRaw.tTouchDriverRaw.au8Name, sizeof(m_strTouchDriverName)-1);
	   ++_u8LoadedItems;
   }
   // driver config file
   _s32ReturnCode = DP_s32GetConfigItem( "EarlyConfig_Touch", "TouchConfigFileName",
		   _tKDSTouchDriverRaw.tTouchDriverRaw.au8File, sizeof(_tKDSTouchDriverRaw.tTouchDriverRaw.au8File), FALSE);
   if  (DP_S32_NO_ERR > _s32ReturnCode)
   {
	   ETG_TRACE_ERR(("bLoadKDSEarlyConfig_TouchDriver: EarlyConfig_TouchDriver.File could not read (error=0x%08x)", _s32ReturnCode));
   }
   else
   {
	   (tVoid)OSAL_pvMemorySet((tVoid*)m_strTouchConfigFileName, OSAL_NULL, sizeof(m_strTouchConfigFileName));
	   (tVoid)OSAL_szStringNCopy(m_strTouchConfigFileName, (const char*)_tKDSTouchDriverRaw.tTouchDriverRaw.au8File, sizeof(m_strTouchConfigFileName)-1);
	   ++_u8LoadedItems;
   }
   // i2c address
   (tVoid)OSAL_pvMemorySet((tVoid*)m_strTouchDriverI2CAddress, OSAL_NULL, sizeof(m_strTouchDriverI2CAddress));
   m_u8TouchDriverI2CAddress = 0;
#ifdef DP_U8_KDSLEN_AIVIVARIANTCODING_TOUCHI2CADDRESS
   tU8 _u8I2CAddress[DP_U8_KDSLEN_AIVIVARIANTCODING_TOUCHI2CADDRESS] = {0};
   _s32ReturnCode = DP_s32GetConfigItem( "AIVIVariantCoding", "TouchI2CAddress", _u8I2CAddress, sizeof(_u8I2CAddress), FALSE);
   if  (DP_S32_NO_ERR > _s32ReturnCode)
   {
	   ETG_TRACE_ERR(("bLoadKDSEarlyConfig_TouchDriver: EarlyConfig_TouchDriver.I2CAddress could not read (error=0x%08x)", _s32ReturnCode));
   }
   else
   {
	   m_u8TouchDriverI2CAddress = _u8I2CAddress[0];
	   ++_u8LoadedItems;
   }
   if (DP_U8_EARLYCONFIG_TOUCHI2CADDRESS_NOTSET != m_u8TouchDriverI2CAddress)
   {
	   snprintf(m_strTouchDriverI2CAddress, sizeof(m_strTouchDriverI2CAddress)-1, "i2c-%u/", m_u8TouchDriverI2CAddress);
   }
#else
   ++_u8LoadedItems;
#endif //#ifdef DP_U8_KDSLEN_AIVIVARIANTCODING_TOUCHI2CADDRESS
   // summaries result
   if(3 == _u8LoadedItems)
   {
	   ETG_TRACE_USR2(("bLoadKDSEarlyConfig_TouchDriver: KDS early config touch loaded with driver_name=%64s and file_name=%64s and I2CAddress=%u",
			   m_strTouchDriverName, m_strTouchConfigFileName, m_u8TouchDriverI2CAddress ));
	   m_bKDSTouchDriverLoaded = TRUE;
   }

   return bIsLoadedKDSEarlyConfig_TouchDriver();
}

/*******************************************************************************
*
* FUNCTION:    bIsValidKDSEarlyConfig_TouchDriver()
*
* DESCRIPTION: returns TRUE if loaded KDS data are valid
*              - configured driver name must be known
*              - configured config file name must be known
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - valid | else - not valid}
*
*******************************************************************************/
// possible touch driver names for A-IVI
const char* m_astrEarlyConfigTouchDriverNames[] =
{
		"atmel_mxt_ts",
		"cyttsp4_i2c"
};
// possible touch config file names for A-IVI
const char* m_astrEarlyConfigTouchConfigFiles[] =
{
		"Nissan_Scope1_JLR-TP-01.raw",
		"Nissan_Scope1_ITG5-TP-02.raw",
		"Nissan_Scope2-8inch_ITG5-TP.raw",
		"DTS2020-mxtConfig8inch-V03.raw"
};
tBool dispvidctrl_tclEarlyConfig::bIsValidKDSEarlyConfig_TouchDriver() const
{
   ETG_TRACE_USR4(("bIsValidKDSEarlyConfig_TouchDriver() entered."));

   tBool _bValid = FALSE;
   // valid as specified
   if (TRUE == bIsLoadedKDSEarlyConfig_TouchDriver())
   {
	   tBool _bValName = FALSE;
	   for (tU32 _u32Idx = 0; (sizeof(m_astrEarlyConfigTouchDriverNames)/sizeof(m_astrEarlyConfigTouchDriverNames[0])) > _u32Idx; ++_u32Idx)
	   {
		   if (OSAL_NULL == OSAL_s32StringCompare(m_astrEarlyConfigTouchDriverNames[_u32Idx], m_strTouchDriverName))
		   {
			   _bValName = TRUE;
		   }
	   }
	   if (FALSE == _bValName)
	   {
		   ETG_TRACE_USR3(("bIsValidKDSEarlyConfig_TouchDriver: invalid driver name %s ",
				   m_strTouchDriverName ));
	   }
	   tBool _bValFile = (OSAL_NULL == OSAL_u32StringLength(m_astrEarlyConfigTouchConfigFiles)) ? TRUE : FALSE;
	   for (tU32 _u32Idx = 0; (sizeof(m_astrEarlyConfigTouchConfigFiles)/sizeof(m_astrEarlyConfigTouchConfigFiles[0])) > _u32Idx; ++_u32Idx)
	   {
		   if (OSAL_NULL == OSAL_s32StringCompare(m_astrEarlyConfigTouchConfigFiles[_u32Idx], m_strTouchConfigFileName))
		   {
			   _bValFile = TRUE;
		   }
	   }
	   if (FALSE == _bValFile)
	   {
		   ETG_TRACE_USR3(("bIsValidKDSEarlyConfig_TouchDriver: invalid touch config file name %s ",
				   m_strTouchConfigFileName ));
	   }
	   _bValid = ((TRUE == _bValName) && (TRUE == _bValFile));
   }
   // valid as required ... if touch config file exist
   if (FALSE == _bValid && TRUE == bIsExistKDSEarlyConfig_TouchConfigFile(m_strTouchDriverName, m_strTouchConfigFileName))
   {
	   _bValid = TRUE;
   }

   return _bValid;
}

/*******************************************************************************
*
* FUNCTION:    bIsChangedEarlyConfig_TouchDriver()
*
* DESCRIPTION: returns TRUE if KDS data not the same like early config data
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - new | else - not new}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsChangedEarlyConfig_TouchDriver() const
{
   ETG_TRACE_USR4(("bIsChangedEarlyConfig_TouchDriver() entered."));
   tBool _bChanged = TRUE;
   // touch driver name
   dp_tclEarlyConfigTouchDriverTrTouchDriverName _oTouchDriverName;
   tChar _strTouchDriverName[sizeof(m_strTouchDriverName)];
   (tVoid)OSAL_pvMemorySet((tVoid*)_strTouchDriverName, OSAL_NULL, sizeof(_strTouchDriverName));
   tS32 _s32ReadStatus = _oTouchDriverName.s32GetData(_strTouchDriverName, sizeof(_strTouchDriverName)-1);
   if (DP_S32_NO_ERR > _oTouchDriverName.s32GetStatus())
   {
	   // not initialized
	   _bChanged = bIsLoadedKDSEarlyConfig_TouchDriver();
	   ETG_TRACE_USR2(("bIsChangedEarlyConfig_TouchDriver: Status TouchDriverName: %d.", _oTouchDriverName.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   // could be loaded
	   tS32 _s32Result = OSAL_s32StringCompare(m_strTouchDriverName,_strTouchDriverName);
	   if (OSAL_NULL != _s32Result)
	   {
		   _bChanged = bIsLoadedKDSEarlyConfig_TouchDriver();
	   }
	   ETG_TRACE_USR3(("bIsChangedEarlyConfig_TouchDriver: TouchDriverName Compare: %d.", _s32Result ));
   }
   else
   {
	   ETG_TRACE_FATAL(("bIsChangedEarlyConfig_TouchDriver: Read Error TouchDriverName detected %d!", _s32ReadStatus ));
   }
   // touch config file name
   dp_tclEarlyConfigTouchDriverTrTouchConfigFileName _oTouchConfigFileName;
   tChar _strTouchConfigFileName[sizeof(m_strTouchConfigFileName)];
   (tVoid)OSAL_pvMemorySet((tVoid*)_strTouchConfigFileName, OSAL_NULL, sizeof(_strTouchConfigFileName));
   _s32ReadStatus = _oTouchConfigFileName.s32GetData(_strTouchConfigFileName, sizeof(_strTouchConfigFileName)-1);
   if (DP_S32_NO_ERR > _oTouchConfigFileName.s32GetStatus())
   {
	   // not initialized
	   _bChanged = bIsLoadedKDSEarlyConfig_TouchDriver();
	   ETG_TRACE_USR2(("bIsChangedEarlyConfig_TouchDriver: Status TouchConfigFileName: %d.", _oTouchConfigFileName.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   // could be loaded
	   tS32 _s32Result = OSAL_s32StringCompare(m_strTouchConfigFileName,_strTouchConfigFileName);
	   if (OSAL_NULL != _s32Result)
	   {
		   _bChanged = bIsLoadedKDSEarlyConfig_TouchDriver();
	   }
	   ETG_TRACE_USR3(("bIsChangedEarlyConfig_TouchDriver: TouchConfigFileName Compare: %d.", _s32Result ));
   }
   else
   {
	   ETG_TRACE_FATAL(("bIsChangedEarlyConfig_TouchDriver: Read Error TouchConfigFileName detected %d!", _s32ReadStatus ));
   }
   // touch I2C address
#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
   dp_tclEarlyConfigTouchDriverTrTouchI2CAddr _oTouchICAddress;
   tChar _strTouchI2CAddress[DP_U8_NORLEN_EARLYCONFIG_TOUCHI2CADDRESS];
   (tVoid)OSAL_pvMemorySet((tVoid*)_strTouchI2CAddress, OSAL_NULL, sizeof(_strTouchI2CAddress));
   _s32ReadStatus = _oTouchICAddress.s32GetData(_strTouchI2CAddress, sizeof(_strTouchI2CAddress)-1);
   if (DP_S32_NO_ERR > _oTouchICAddress.s32GetStatus())
   {
	   // not initialized
	   _bChanged = bIsLoadedKDSEarlyConfig_TouchDriver();
	   ETG_TRACE_USR2(("bIsChangedEarlyConfig_TouchDriver: Status TouchI2CAddress: %d.", _oTouchICAddress.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   // could be loaded
	   tS32 _s32Result = OSAL_s32StringCompare(m_strTouchDriverI2CAddress,_strTouchI2CAddress);
	   if (OSAL_NULL != _s32Result)
	   {
		   _bChanged = bIsLoadedKDSEarlyConfig_TouchDriver();
	   }
	   ETG_TRACE_USR3(("bIsChangedEarlyConfig_TouchDriver: TouchI2CAddress Compare: %d.", _s32Result ));
   }
   else
   {
	   ETG_TRACE_FATAL(("bIsChangedEarlyConfig_TouchDriver: Read Error TouchI2CAddress detected %d!", _s32ReadStatus ));
   }
#else
   ETG_TRACE_FATAL(("bIsChangedEarlyConfig_TouchDriver: TouchI2CAddress unsupported!" ));
#endif //#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER

   return _bChanged;
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig_TouchDriver()
*
* DESCRIPTION: write the EarlyConfig_TouchDriver
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig_TouchDriver()
{
   ETG_TRACE_USR4(("bWriteEarlyConfig_TouchDriver() entered."));
   if (TRUE == bIsValidKDSCustom_TouchScreenConfig() && 0 != u8GetTouchScreenConfig())
   {
	   if (FALSE == bIsEarlyConfig_TouchDriverManuallySet())
	   {
		   dp_tclEarlyConfigTouchDriverTrTouchDriverName _oTouchDriverName;
		   char _strTouchDriverName[sizeof(m_strTouchDriverName)];
		   (tVoid)memset((tVoid*)_strTouchDriverName, OSAL_NULL, sizeof(_strTouchDriverName));
		   (tVoid)OSAL_szStringNCopy(_strTouchDriverName, cstrGetCustomTouchDriver(), sizeof(_strTouchDriverName)-1);
		   if (DP_S32_NO_ERR != _oTouchDriverName.s32SetData(_strTouchDriverName))
		   {
			   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriver: Failed to set driver name %s to early config!",
					   _strTouchDriverName ));
			   return FALSE;
		   }
		   else
		   {
			   ETG_TRACE_USR2(("bWriteEarlyConfig_TouchDriver: Successful set driver name %s to early config!",
					   _strTouchDriverName ));
		   }
	   }
	   if (FALSE == bIsEarlyConfig_TouchDriverConfigFileManuallySet())
	   {
		   dp_tclEarlyConfigTouchDriverTrTouchConfigFileName _oTouchConfigFileName;
		   char _strTouchConfigFileName[sizeof(m_strTouchConfigFileName)];
		   (tVoid)memset((tVoid*)_strTouchConfigFileName, OSAL_NULL, sizeof(_strTouchConfigFileName));
		   (tVoid)OSAL_szStringNCopy(_strTouchConfigFileName, cstrGetCustomTouchConfigFile(), sizeof(_strTouchConfigFileName)-1);
		   if (DP_S32_NO_ERR != _oTouchConfigFileName.s32SetData(_strTouchConfigFileName))
		   {
			   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriver: Failed to set config file name %s to early config!",
					   _strTouchConfigFileName ));
			   return FALSE;
		   }
		   else
		   {
			   ETG_TRACE_USR2(("bWriteEarlyConfig_TouchDriver: Successful set config file name %s to early config!",
					   _strTouchConfigFileName ));
		   }
	   }
	   return TRUE;
   }
   else
   {
	   if (TRUE == bIsValidKDSEarlyConfig_TouchDriver() && TRUE == bIsChangedEarlyConfig_TouchDriver())
	   {
		   tBool _bChanged = FALSE;
		   if (FALSE == bIsEarlyConfig_TouchDriverManuallySet())
		   {
			   dp_tclEarlyConfigTouchDriverTrTouchDriverName _oTouchDriverName;
			   if (DP_S32_NO_ERR != _oTouchDriverName.s32SetData(m_strTouchDriverName))
			   {
				   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriver: Failed to set driver name %s to early config!",
						   m_strTouchDriverName ));
				   return FALSE;
			   }
			   else
			   {
				   _bChanged = TRUE;
				   ETG_TRACE_USR2(("bWriteEarlyConfig_TouchDriver: Successful set driver name %s to early config!",
						   m_strTouchDriverName ));
			   }
		   }
		   if (FALSE == bIsEarlyConfig_TouchDriverConfigFileManuallySet())
		   {
			   dp_tclEarlyConfigTouchDriverTrTouchConfigFileName _oTouchConfigFileName;
			   if (DP_S32_NO_ERR != _oTouchConfigFileName.s32SetData(m_strTouchConfigFileName))
			   {
				   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriver: Failed to set config file name %s to early config!",
						   m_strTouchConfigFileName ));
				   return FALSE;
			   }
			   else
			   {
				   _bChanged = TRUE;
				   ETG_TRACE_USR2(("bWriteEarlyConfig_TouchDriver: Successful set config file name %s to early config!",
						   m_strTouchConfigFileName ));
			   }
		   }
		   if (FALSE == bIsEarlyConfig_TouchI2CAddressManuallySet())
		   {
#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
			   dp_tclEarlyConfigTouchDriverTrTouchI2CAddr _oTouchI2CAddress;
			   tChar _strTouchIsCAddress[DP_U8_NORLEN_EARLYCONFIG_TOUCHI2CADDRESS] = {0};
			   (tVoid)OSAL_szStringNCopy(_strTouchIsCAddress, m_strTouchDriverI2CAddress, sizeof(_strTouchIsCAddress)-1);
			   if (DP_S32_NO_ERR != _oTouchI2CAddress.s32SetData(_strTouchIsCAddress))
			   {
				   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriver: Failed to set I2C address %s to early config!",
						   m_strTouchDriverI2CAddress ));
				   return FALSE;
			   }
			   else
			   {
				   _bChanged = TRUE;
				   ETG_TRACE_USR2(("bWriteEarlyConfig_TouchDriver: Successful set I2C address %s to early config!",
						   m_strTouchDriverI2CAddress ));
			   }
#else
			   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriver: touch I2C address unsupported!"));
#endif //#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
		   }
		   dp_tclSrvIf _oDpSrvIf;
		   tU32 _u32DpID = DP_U32_POOL_ID_EARLYCONFIGTOUCHDRIVER;
		   tS32 _s32ErrorCode = _oDpSrvIf.s32StorePoolNow(_u32DpID);
		   if (DP_S32_NO_ERR != _s32ErrorCode)
		   {
		     ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriver: storing of PDD 0x%08x failed with error code 0x%08x!", _u32DpID, _s32ErrorCode ));
		   }
		   if (TRUE == _bChanged)
		   {
			   ETG_TRACE_ERRMEM(( "VD_DIMMING: early config touch set with driver_name=%64s, config_file=%64s, I2C_address=%12s",
					   m_strTouchDriverName, m_strTouchConfigFileName, m_strTouchDriverI2CAddress ));
			   return TRUE;
		   }
	   }
   }
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig_TouchDriverName()
*
* DESCRIPTION: write the EarlyConfig_Touch driver name directly without validation
*
* PARAMETER:   const tChar* pStrDriverName (max. 63 char)
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig_TouchDriverName(const tChar* pStrDriverName) const
{
   ETG_TRACE_USR4(("bWriteEarlyConfig_TouchDriverName(%63s) entered.", pStrDriverName));

   tBool _bChanged = FALSE;
   tChar _strTouchDriverName_Manual[sizeof(m_strTouchDriverName)] = {0};
   dp_tclDimmingDPEarlyConfig_TouchDriverName_Manually _oTouchDriverNameManual;
   tS32 _s32ReadStatus = _oTouchDriverNameManual.s32GetData(_strTouchDriverName_Manual, sizeof(_strTouchDriverName_Manual)-1);

   // check for update of manual EarlyConfig touch driver name
   if (DP_S32_NO_ERR > _oTouchDriverNameManual.s32GetStatus())
   {
	   // not initialized
	   _bChanged = TRUE;
	   ETG_TRACE_USR1(("bWriteEarlyConfig_TouchDriverName: Status TouchDriverName manual: %d.", _oTouchDriverNameManual.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   // could be loaded
	   tS32 _s32Compare = OSAL_s32StringCompare(pStrDriverName, _strTouchDriverName_Manual);
	   if (OSAL_NULL != _s32Compare)
	   {
		   _bChanged = TRUE;
		   ETG_TRACE_USR1(("bWriteEarlyConfig_TouchDriverName: TouchDriverName manual changed (Compare: %d).", _s32Compare ));
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverName: TouchDriverName manual not changed (Compare: %d).", _s32Compare ));
	   }
   }
   // set EarlyConfig touch driver name manual
   if (TRUE == _bChanged)
   {
	   (tVoid)OSAL_pvMemorySet((tVoid*)_strTouchDriverName_Manual, OSAL_NULL, sizeof(_strTouchDriverName_Manual));
	   (tVoid)strncpy(_strTouchDriverName_Manual, pStrDriverName, sizeof(_strTouchDriverName_Manual)-1);

	   // write EarlyConfig data pool
	   dp_tclEarlyConfigTouchDriverTrTouchDriverName _oTouchDriverName;
	   if (DP_S32_NO_ERR != _oTouchDriverName.s32SetData(_strTouchDriverName_Manual))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverName: Failed to manually set EarlyConfig touch driver name %s!",
				   _strTouchDriverName_Manual ));
		   return FALSE;
	   }
	   else
	   {
		   ETG_TRACE_USR1(("bWriteEarlyConfig_TouchDriverName: Successful manually set EarlyConfig touch driver name %s.",
				   _strTouchDriverName_Manual ));
	   }
	   // write own touch driver name data pool
	   if (DP_S32_NO_ERR != _oTouchDriverNameManual.s32SetData(_strTouchDriverName_Manual))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverName: Failed to manually set own touch driver name %s!",
				   _strTouchDriverName_Manual ));
	   }
	   else
	   {
		   ETG_TRACE_USR1(("bWriteEarlyConfig_TouchDriverName: Successful manually set own touch driver name %s.",
				   _strTouchDriverName_Manual ));
		   return TRUE;
	   }
   }

   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig_TouchDriverConfigFile()
*
* DESCRIPTION: write the EarlyConfig_Touch driver configuration file name
*              directly without validation
*
* PARAMETER:   const tChar* pStrFilename (max. 63 char)
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig_TouchDriverConfigFile(const tChar* pStrFilename) const
{
   ETG_TRACE_USR4(("bWriteEarlyConfig_TouchDriverConfigFile(%63s) entered.", pStrFilename));

   tBool _bChanged = FALSE;
   tChar _strTouchConfigFileName_Manual[sizeof(m_strTouchConfigFileName)] = {0};
   dp_tclDimmingDPEarlyConfig_TouchConfigFile_Manually _oTouchConfigFileNameManual;
   tS32 _s32ReadStatus = _oTouchConfigFileNameManual.s32GetData(_strTouchConfigFileName_Manual, sizeof(_strTouchConfigFileName_Manual)-1);


   // check for update of manual EarlyConfig touch driver name
   if (DP_S32_NO_ERR > _oTouchConfigFileNameManual.s32GetStatus())
   {
	   // not initialized
	   _bChanged = TRUE;
	   ETG_TRACE_USR1(("bWriteEarlyConfig_TouchDriverConfigFile: Status TouchConfigFileName manual: %d.", _oTouchConfigFileNameManual.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   // could be loaded
	   tS32 _s32Compare = OSAL_s32StringCompare(pStrFilename, _strTouchConfigFileName_Manual);
	   if (OSAL_NULL != _s32Compare)
	   {
		   _bChanged = TRUE;
		   ETG_TRACE_USR1(("bWriteEarlyConfig_TouchDriverConfigFile: TouchConfigFileName manual changed (Compare: %d).", _s32Compare ));
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverConfigFile: TouchConfigFileName manual not changed (Compare: %d).", _s32Compare ));
	   }
   }
   // set EarlyConfig touch config file name manual
   if (TRUE == _bChanged)
   {
	   (tVoid)OSAL_pvMemorySet((tVoid*)_strTouchConfigFileName_Manual, OSAL_NULL, sizeof(_strTouchConfigFileName_Manual));
	   (tVoid)strncpy(_strTouchConfigFileName_Manual, pStrFilename, sizeof(_strTouchConfigFileName_Manual)-1);

	   // write EarlyConfig data pool
	   dp_tclEarlyConfigTouchDriverTrTouchConfigFileName _oTouchConfigFileName;
	   if (DP_S32_NO_ERR != _oTouchConfigFileName.s32SetData(_strTouchConfigFileName_Manual))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverConfigFile: Failed to manually set EarlyConfig touch config file name %s!",
				   _strTouchConfigFileName_Manual ));
		   return FALSE;
	   }
	   else
	   {
		   ETG_TRACE_USR1(("bWriteEarlyConfig_TouchDriverConfigFile: Successful manually set EarlyConfig touch config file name %s.",
				   _strTouchConfigFileName_Manual ));
	   }
	   // write own touch config file name data pool
	   if (DP_S32_NO_ERR != _oTouchConfigFileNameManual.s32SetData(_strTouchConfigFileName_Manual))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverConfigFile: Failed to manually set own touch config file name %s!",
				   _strTouchConfigFileName_Manual ));
		   return FALSE;
	   }
	   else
	   {
		   ETG_TRACE_USR1(("bWriteEarlyConfig_TouchDriverConfigFile: Successful manually set own touch config file name %s.",
				   _strTouchConfigFileName_Manual ));
		   return TRUE;
	   }
   }

   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig_TouchDriverI2CAddress()
*
* DESCRIPTION: write the EarlyConfig_Touch driver I2C address
*              directly without validation
*
* PARAMETER:   const tChar* pStrI2CAddress (max. 11 char)
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig_TouchDriverI2CAddress(const tChar* pStrI2CAddress) const
{
   ETG_TRACE_USR4(("bWriteEarlyConfig_TouchDriverI2CAddress(%11s) entered.", pStrI2CAddress));

   tBool _bChanged = FALSE;
   tChar _strTouchI2CAddress_Manual[DP_U8_NORLEN_EARLYCONFIG_TOUCHI2CADDRESS] = {0};
   dp_tclDimmingDPEarlyConfig_TouchI2CAddress_Manually _oTouchI2CAddressManual;
   tS32 _s32ReadStatus = _oTouchI2CAddressManual.s32GetData(_strTouchI2CAddress_Manual, sizeof(_strTouchI2CAddress_Manual)-1);


   // check for update of manual EarlyConfig touch driver I2C address
   if (OSAL_NULL == pStrI2CAddress)
   {
	   // NULL point
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverI2CAddress: TouchI2CAddress is NULL!" ));
   }
   else if (DP_U8_NORLEN_EARLYCONFIG_TOUCHI2CADDRESS <= strlen(pStrI2CAddress))
   {
	   // string too long
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverI2CAddress: TouchI2CAddress %50s manual too long (length: %d bytes)!", pStrI2CAddress, strlen(pStrI2CAddress) ));
   }
   else if (DP_S32_NO_ERR > _oTouchI2CAddressManual.s32GetStatus())
   {
	   // not initialized
	   _bChanged = TRUE;
	   ETG_TRACE_USR1(("bWriteEarlyConfig_TouchDriverI2CAddress: Status TouchI2CAddress manual: %d.", _oTouchI2CAddressManual.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   // could be loaded
	   tS32 _s32Compare = OSAL_s32StringCompare(pStrI2CAddress, _strTouchI2CAddress_Manual);
	   if (OSAL_NULL != _s32Compare)
	   {
		   _bChanged = TRUE;
		   ETG_TRACE_USR1(("bWriteEarlyConfig_TouchDriverI2CAddress: TouchI2CAddress manual changed (Compare: %d).", _s32Compare ));
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverI2CAddress: TouchI2CAddress manual not changed (Compare: %d)!", _s32Compare ));
	   }
   }
   // set EarlyConfig touch config file name manual
   if (TRUE == _bChanged)
   {
	   (tVoid)OSAL_pvMemorySet((tVoid*)_strTouchI2CAddress_Manual, OSAL_NULL, sizeof(_strTouchI2CAddress_Manual));
	   (tVoid)strncpy(_strTouchI2CAddress_Manual, pStrI2CAddress, sizeof(_strTouchI2CAddress_Manual)-1);

	   // write EarlyConfig data pool
#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
	   dp_tclEarlyConfigTouchDriverTrTouchI2CAddr _oTouchI2CAddress;
	   if (DP_S32_NO_ERR != _oTouchI2CAddress.s32SetData(_strTouchI2CAddress_Manual))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverI2CAddress: Failed to manually set EarlyConfig touch I2C address %s!",
				   _strTouchI2CAddress_Manual ));
		   return FALSE;
	   }
	   else
	   {
		   ETG_TRACE_USR1(("bWriteEarlyConfig_TouchDriverI2CAddress: Successful manually set EarlyConfig touch I2C address %s.",
				   _strTouchI2CAddress_Manual ));
	   }
	   // write own touch config file name data pool
	   if (DP_S32_NO_ERR != _oTouchI2CAddressManual.s32SetData(_strTouchI2CAddress_Manual))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverI2CAddress: Failed to manually set own touch I2C address %s!",
				   _strTouchI2CAddress_Manual ));
	   }
	   else
	   {
		   ETG_TRACE_USR1(("bWriteEarlyConfig_TouchDriverI2CAddress: Successful manually set own touch I2C address %s.",
				   _strTouchI2CAddress_Manual ));
		   return TRUE;
	   }
#else
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverI2CAddress: Failed to manually set own EarlyConfig touch I2C address %s! UNSUPPORTED DATAPOOL!",
			   _strTouchI2CAddress_Manual ));
#endif //#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
   }

   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig_TouchDriverName2()
*
* DESCRIPTION: write the EarlyConfig_Touch driver name 2 directly without validation
*
* PARAMETER:   const tChar* pStrDriverName (max. 63 char)
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig_TouchDriverName2(const tChar* pStrDriverName) const
{
   ETG_TRACE_USR4(("bWriteEarlyConfig_TouchDriverName2(%63s) entered.", pStrDriverName));

   tBool _bChanged = FALSE;
   tChar _strTouchDriverName_Manual[sizeof(m_strTouchDriverName2)] = {0};
   dp_tclDimmingDPEarlyConfig_TouchDriverName2_Manually _oTouchDriverNameManual;
   tS32 _s32ReadStatus = _oTouchDriverNameManual.s32GetData(_strTouchDriverName_Manual, sizeof(_strTouchDriverName_Manual)-1);

   // check for update of manual EarlyConfig touch driver name
   if (OSAL_NULL == pStrDriverName)
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverName2: manual TouchDriverName2 is NULL!" ));
   }
   else if (sizeof(m_strTouchDriverName2) <= strlen(pStrDriverName))
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverName2: manual TouchDriverName2 %64s is too long (length: %d bytes)!", pStrDriverName, strlen(pStrDriverName) ));
   }
   else if (DP_S32_NO_ERR > _oTouchDriverNameManual.s32GetStatus())
   {
	   // not initialized
	   _bChanged = TRUE;
	   ETG_TRACE_USR1(("bWriteEarlyConfig_TouchDriverName2: Status TouchDriverName2 manual: %d.", _oTouchDriverNameManual.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   // could be loaded
	   tS32 _s32Compare = OSAL_s32StringCompare(pStrDriverName, _strTouchDriverName_Manual);
	   if (OSAL_NULL != _s32Compare)
	   {
		   _bChanged = TRUE;
		   ETG_TRACE_USR1(("bWriteEarlyConfig_TouchDriverName2: TouchDriverName2 manual changed (Compare: %d).", _s32Compare ));
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverName2: TouchDriverName2 manual not changed (Compare: %d).", _s32Compare ));
	   }
   }
   // set EarlyConfig touch driver name manual
#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
   if (TRUE == _bChanged)
   {
	   (tVoid)OSAL_pvMemorySet((tVoid*)_strTouchDriverName_Manual, OSAL_NULL, sizeof(_strTouchDriverName_Manual));
	   (tVoid)strncpy(_strTouchDriverName_Manual, pStrDriverName, sizeof(_strTouchDriverName_Manual)-1);

	   // write EarlyConfig data pool
	   dp_tclEarlyConfigTwoTouchDriverTrTouchDriverName2 _oTouchDriverName;
	   if (DP_S32_NO_ERR != _oTouchDriverName.s32SetData(_strTouchDriverName_Manual))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverName2: Failed to manually set EarlyConfig touch driver name 2 %s!",
				   _strTouchDriverName_Manual ));
		   return FALSE;
	   }
	   else
	   {
		   ETG_TRACE_USR1(("bWriteEarlyConfig_TouchDriverName2: Successful manually set EarlyConfig touch driver name 2 %s.",
				   _strTouchDriverName_Manual ));
	   }
	   // write own touch driver name data pool
	   if (DP_S32_NO_ERR != _oTouchDriverNameManual.s32SetData(_strTouchDriverName_Manual))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverName2: Failed to manually set own touch driver name 2 %s!",
				   _strTouchDriverName_Manual ));
	   }
	   else
	   {
		   ETG_TRACE_USR1(("bWriteEarlyConfig_TouchDriverName2: Successful manually set own touch driver name 2 %s.",
				   _strTouchDriverName_Manual ));
		   return TRUE;
	   }
   }
#else
   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverName2: Failed to manually set own touch driver name 2 %64s! UNSUPPORTED DATAPOOL!",
		   pStrDriverName ));
#endif // #ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER

   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig_TouchDriverConfigFile2()
*
* DESCRIPTION: write the EarlyConfig_Touch driver configuration file name 2
*              directly without validation
*
* PARAMETER:   const tChar* pStrFilename (max. 63 char)
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig_TouchDriverConfigFile2(const tChar* pStrFilename) const
{
   ETG_TRACE_USR4(("bWriteEarlyConfig_TouchDriverConfigFile2(%63s) entered.", pStrFilename));

   tBool _bChanged = FALSE;
   tChar _strTouchConfigFileName_Manual[sizeof(m_strTouchConfigFileName2)] = {0};
   dp_tclDimmingDPEarlyConfig_TouchConfigFile2_Manually _oTouchConfigFileNameManual;
   tS32 _s32ReadStatus = _oTouchConfigFileNameManual.s32GetData(_strTouchConfigFileName_Manual, sizeof(_strTouchConfigFileName_Manual)-1);


   // check for update of manual EarlyConfig touch driver name
   if (OSAL_NULL == pStrFilename)
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverConfigFile2: manual TouchConfigFileName2 is NULL!" ));
   }
   else if (sizeof(m_strTouchDriverName2) <= strlen(pStrFilename))
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverConfigFile2: manual TouchConfigFileName2 %64s is too long (length: %d bytes)!", pStrFilename, strlen(pStrFilename) ));
   }
   else if (DP_S32_NO_ERR > _oTouchConfigFileNameManual.s32GetStatus())
   {
	   // not initialized
	   _bChanged = TRUE;
	   ETG_TRACE_USR1(("bWriteEarlyConfig_TouchDriverConfigFile2: Status TouchConfigFileName2 manual: %d.", _oTouchConfigFileNameManual.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   // could be loaded
	   tS32 _s32Compare = OSAL_s32StringCompare(pStrFilename, _strTouchConfigFileName_Manual);
	   if (OSAL_NULL != _s32Compare)
	   {
		   _bChanged = TRUE;
		   ETG_TRACE_USR1(("bWriteEarlyConfig_TouchDriverConfigFile2: TouchConfigFileName2 manual changed (Compare: %d).", _s32Compare ));
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverConfigFile2: TouchConfigFileName2 manual not changed (Compare: %d).", _s32Compare ));
	   }
   }
   // set EarlyConfig touch config file name manual
#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
   if (TRUE == _bChanged)
   {
	   (tVoid)OSAL_pvMemorySet((tVoid*)_strTouchConfigFileName_Manual, OSAL_NULL, sizeof(_strTouchConfigFileName_Manual));
	   (tVoid)strncpy(_strTouchConfigFileName_Manual, pStrFilename, sizeof(_strTouchConfigFileName_Manual)-1);

	   // write EarlyConfig data pool
	   dp_tclEarlyConfigTwoTouchDriverTrTouchConfigFileName2 _oTouchConfigFileName;
	   if (DP_S32_NO_ERR != _oTouchConfigFileName.s32SetData(_strTouchConfigFileName_Manual))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverConfigFile2: Failed to manually set EarlyConfig touch config file name 2 %s!",
				   _strTouchConfigFileName_Manual ));
		   return FALSE;
	   }
	   else
	   {
		   ETG_TRACE_USR1(("bWriteEarlyConfig_TouchDriverConfigFile2: Successful manually set EarlyConfig touch config file name 2 %s.",
				   _strTouchConfigFileName_Manual ));
	   }
	   // write own touch config file name data pool
	   if (DP_S32_NO_ERR != _oTouchConfigFileNameManual.s32SetData(_strTouchConfigFileName_Manual))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverConfigFile2: Failed to manually set own touch config file name 2 %s!",
				   _strTouchConfigFileName_Manual ));
		   return FALSE;
	   }
	   else
	   {
		   ETG_TRACE_USR1(("bWriteEarlyConfig_TouchDriverConfigFile2: Successful manually set own touch config file name 2 %s.",
				   _strTouchConfigFileName_Manual ));
		   return TRUE;
	   }
   }
#else
   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverConfigFile2: Failed to manually set own touch config file name 2 %64s! UNSUPPORTED DATAPOOL!",
		   pStrFilename ));
#endif // #ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER

   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig_TouchDriverI2CAddress2()
*
* DESCRIPTION: write the EarlyConfig_Touch driver I2C address 2
*              directly without validation
*
* PARAMETER:   const tChar* pStrI2CAddress (max. 11 char)
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig_TouchDriverI2CAddress2(const tChar* pStrI2CAddress) const
{
   ETG_TRACE_USR4(("bWriteEarlyConfig_TouchDriverI2CAddress2(%11s) entered.", pStrI2CAddress));

   tBool _bChanged = FALSE;
   tChar _strTouchI2CAddress_Manual[DP_U8_NORLEN_EARLYCONFIG_TOUCHI2CADDRESS] = {0};
   dp_tclDimmingDPEarlyConfig_TouchI2CAddress2_Manually _oTouchI2CAddressManual;
   tS32 _s32ReadStatus = _oTouchI2CAddressManual.s32GetData(_strTouchI2CAddress_Manual, sizeof(_strTouchI2CAddress_Manual)-1);


   // check for update of manual EarlyConfig touch driver I2C address
   if (OSAL_NULL == pStrI2CAddress)
   {
	   // NULL point
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverI2CAddress2: TouchI2CAddress2 is NULL!" ));
   }
   else if (DP_U8_NORLEN_EARLYCONFIG_TOUCHI2CADDRESS <= strlen(pStrI2CAddress))
   {
	   // string too long
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverI2CAddress2: TouchI2CAddress2 %50s manual too long (length: %d bytes)!", pStrI2CAddress, strlen(pStrI2CAddress) ));
   }
   else if (DP_S32_NO_ERR > _oTouchI2CAddressManual.s32GetStatus())
   {
	   // not initialized
	   _bChanged = TRUE;
	   ETG_TRACE_USR1(("bWriteEarlyConfig_TouchDriverI2CAddress2: Status TouchI2CAddress2 manual: %d.", _oTouchI2CAddressManual.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   // could be loaded
	   tS32 _s32Compare = OSAL_s32StringCompare(pStrI2CAddress, _strTouchI2CAddress_Manual);
	   if (OSAL_NULL != _s32Compare)
	   {
		   _bChanged = TRUE;
		   ETG_TRACE_USR1(("bWriteEarlyConfig_TouchDriverI2CAddress2: TouchI2CAddress2 manual changed (Compare: %d).", _s32Compare ));
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverI2CAddress2: TouchI2CAddress2 manual not changed (Compare: %d)!", _s32Compare ));
	   }
   }
   // set EarlyConfig touch config file name manual
   if (TRUE == _bChanged)
   {
	   (tVoid)OSAL_pvMemorySet((tVoid*)_strTouchI2CAddress_Manual, OSAL_NULL, sizeof(_strTouchI2CAddress_Manual));
	   (tVoid)strncpy(_strTouchI2CAddress_Manual, pStrI2CAddress, sizeof(_strTouchI2CAddress_Manual)-1);

	   // write EarlyConfig data pool
#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
	   dp_tclEarlyConfigTwoTouchDriverTrTouchI2CAddr2 _oTouchI2CAddress;
	   if (DP_S32_NO_ERR != _oTouchI2CAddress.s32SetData(_strTouchI2CAddress_Manual))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverI2CAddress2: Failed to manually set EarlyConfig touch I2C address 2 %s!",
				   _strTouchI2CAddress_Manual ));
		   return FALSE;
	   }
	   else
	   {
		   ETG_TRACE_USR1(("bWriteEarlyConfig_TouchDriverI2CAddress2: Successful manually set EarlyConfig touch I2C address 2 %s.",
				   _strTouchI2CAddress_Manual ));
	   }
	   // write own touch config file name data pool
	   if (DP_S32_NO_ERR != _oTouchI2CAddressManual.s32SetData(_strTouchI2CAddress_Manual))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverI2CAddress2: Failed to manually set own touch I2C address 2 %s!",
				   _strTouchI2CAddress_Manual ));
	   }
	   else
	   {
		   ETG_TRACE_USR1(("bWriteEarlyConfig_TouchDriverI2CAddress2: Successful manually set own touch I2C address 2 %s.",
				   _strTouchI2CAddress_Manual ));
		   return TRUE;
	   }
#else
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TouchDriverI2CAddress2: Failed to manually set own EarlyConfig touch I2C address 2 %s! UNSUPPORTED DATAPOOL!",
			   _strTouchI2CAddress_Manual ));
#endif //#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
   }

   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfoEarlyConfig_TouchDriver() const
*
* DESCRIPTION: Trace information EarlyConfig_TouchDriver
*
* PARAMETER:   const char* strIdentifier
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid dispvidctrl_tclEarlyConfig::vTraceInfoEarlyConfig_TouchDriver(const char* strIdentifier) const
{
	ETG_TRACE_USR4(("vTraceInfoEarlyConfig_TouchDriver() entered."));
	ETG_TRACE_FATAL(("%26s: <<< EarlyConfig_TouchDriver >>> ", strIdentifier ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "    TOUCH_DRV_MANUAL",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsEarlyConfig_TouchDriverManuallySet()) ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, " TOUCH_DRVCFG_MANUAL",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsEarlyConfig_TouchDriverConfigFileManuallySet()) ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "    KDS_TOUCH_LOADED",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsLoadedKDSEarlyConfig_TouchDriver()) ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "     KDS_TOUCH_VALID",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsValidKDSEarlyConfig_TouchDriver()) ));
	ETG_TRACE_FATAL(("%26s: %30s='%64s' ", strIdentifier, "   KDS_TOUCH_DRVNAME",
			m_strTouchDriverName ));
	ETG_TRACE_FATAL(("%26s: %30s='%64s' ", strIdentifier, "  KDS_TOUCH_FILENAME",
			m_strTouchConfigFileName ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier,   "  KDS_TOUCH_I2CBUSNR",
			m_u8TouchDriverI2CAddress ));
	ETG_TRACE_FATAL(("%26s: %30s='%12s' ", strIdentifier, "    KDS_TOUCH_I2CADR",
			m_strTouchDriverI2CAddress ));

	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "KDS_TOUCH_FILE_EXIST",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsExistKDSEarlyConfig_TouchConfigFile(m_strTouchDriverName, m_strTouchConfigFileName)) ));

	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "    EC_TOUCH_CHANGED",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsChangedEarlyConfig_TouchDriver()) ));
	dp_tclEarlyConfigTouchDriverTrTouchDriverName _oTouchDriverName;
	tChar _strTouchDriverName[sizeof(m_strTouchDriverName)];
	(tVoid)OSAL_pvMemorySet((tVoid*)_strTouchDriverName, OSAL_NULL, sizeof(_strTouchDriverName));
	(tVoid)_oTouchDriverName.s32GetData(_strTouchDriverName, sizeof(_strTouchDriverName)-1);
	ETG_TRACE_FATAL(("%26s: %30s='%64s' ", strIdentifier, "    EC_TOUCH_DRVNAME",
			_strTouchDriverName ));
	dp_tclEarlyConfigTouchDriverTrTouchConfigFileName _oTouchConfigFileName;
	tChar _strTouchConfigFileName[sizeof(m_strTouchConfigFileName)];
	(tVoid)OSAL_pvMemorySet((tVoid*)_strTouchConfigFileName, OSAL_NULL, sizeof(_strTouchConfigFileName));
	(tVoid)_oTouchConfigFileName.s32GetData(_strTouchConfigFileName, sizeof(_strTouchConfigFileName)-1);
	ETG_TRACE_FATAL(("%26s: %30s='%64s' ", strIdentifier, "   EC_TOUCH_FILENAME",
			_strTouchConfigFileName ));
#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
	dp_tclEarlyConfigTouchDriverTrTouchI2CAddr _oTouchI2CAddress;
	tChar _strTouchI2CAddress[DP_U8_NORLEN_EARLYCONFIG_TOUCHI2CADDRESS];
	(tVoid)OSAL_pvMemorySet((tVoid*)_strTouchI2CAddress, OSAL_NULL, sizeof(_strTouchI2CAddress));
	(tVoid)_oTouchI2CAddress.s32GetData(_strTouchI2CAddress, sizeof(_strTouchI2CAddress)-1);
	ETG_TRACE_FATAL(("%26s: %30s='%64s' ", strIdentifier, "     EC_TOUCH_I2CADR",
			_strTouchI2CAddress ));
#else
	ETG_TRACE_FATAL(("%26s: %30s='%64s' ", strIdentifier, "     EC_TOUCH_I2CADR",
			"---" ));
#endif //#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
}

/*******************************************************************************
*
* FUNCTION:    bIsEarlyConfig_TimingLvds1ManuallySet()
*
* DESCRIPTION: returns TRUE if EarlyConfig_TwoDisplays timming LVDS 1 manually set
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - manually set | else - not manually set}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsEarlyConfig_TimingLvds1ManuallySet(tVoid) const
{
   ETG_TRACE_USR4(("bIsEarlyConfig_TimingLvds1ManuallySet() entered."));

   tChar _strTimingLvds1_Manual[sizeof(m_strTimingLvds1)] = {0};
   dp_tclDimmingDPEarlyConfig_TimingLvds1_Manually _oTimingLvds1;
   tS32 _s32ReadStatus = _oTimingLvds1.s32GetData(_strTimingLvds1_Manual, sizeof(_strTimingLvds1_Manual)-1);

   // check for update of manual EarlyConfig timing LVDS 1
   if (DP_S32_NO_ERR > _oTimingLvds1.s32GetStatus())
   {
	   // not initialized
	   ETG_TRACE_FATAL(("bIsEarlyConfig_TimingLvds1ManuallySet: Status TwoDisplays.TimingLVDS1 manual: %d.", _oTimingLvds1.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   if (0 < strlen(_strTimingLvds1_Manual))
	   {
		   return TRUE;
	   }

   }
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bIsEarlyConfig_TimingLvds2ManuallySet()
*
* DESCRIPTION: returns TRUE if EarlyConfig_TwoDisplays timming LVDS 2 manually set
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - manually set | else - not manually set}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsEarlyConfig_TimingLvds2ManuallySet(tVoid) const
{
   ETG_TRACE_USR4(("bIsEarlyConfig_TimingLvds2ManuallySet() entered."));

   tChar _strTimingLvds2_Manual[sizeof(m_strTimingLvds2)] = {0};
   dp_tclDimmingDPEarlyConfig_TimingLvds2_Manually _oTimingLvds2;
   tS32 _s32ReadStatus = _oTimingLvds2.s32GetData(_strTimingLvds2_Manual, sizeof(_strTimingLvds2_Manual)-1);

   // check for update of manual EarlyConfig timing LVDS 2
   if (DP_S32_NO_ERR > _oTimingLvds2.s32GetStatus())
   {
	   // not initialized
	   ETG_TRACE_FATAL(("bIsEarlyConfig_TimingLvds1ManuallySet: Status TwoDisplays.TimingLVDS2 manual: %d.", _oTimingLvds2.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   if (0 < strlen(_strTimingLvds2_Manual))
	   {
		   return TRUE;
	   }

   }
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bIsEarlyConfig_ClockEdgeLvds1ManuallySet()
*
* DESCRIPTION: returns TRUE if EarlyConfig_TwoDisplays clock edge LVDS 1 manually set
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - manually set | else - not manually set}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsEarlyConfig_ClockEdgeLvds1ManuallySet(tVoid) const
{
   ETG_TRACE_USR4(("bIsEarlyConfig_ClockEdgeLvds1ManuallySet() entered."));

   dp_tclDimmingDPEarlyConfig_ClockEdgeLvds1_Manually _oClockEdge;
   tS32 _s32ClockEdge = _oClockEdge.tGetData();
   tBool _bReturn = FALSE;

   if (-1 <= _s32ClockEdge && 1 >= _s32ClockEdge)
   {
	   _bReturn = TRUE;
   }

   return _bReturn;
}

/*******************************************************************************
*
* FUNCTION:    bIsEarlyConfig_ClockEdgeLvds2ManuallySet()
*
* DESCRIPTION: returns TRUE if EarlyConfig_TwoDisplays clock edge LVDS 2 manually set
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - manually set | else - not manually set}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsEarlyConfig_ClockEdgeLvds2ManuallySet(tVoid) const
{
   ETG_TRACE_USR4(("bIsEarlyConfig_ClockEdgeLvds1ManuallySet() entered."));

   dp_tclDimmingDPEarlyConfig_ClockEdgeLvds2_Manually _oClockEdge;
   tS32 _s32ClockEdge = _oClockEdge.tGetData();
   tBool _bReturn = FALSE;

   if (-1 <= _s32ClockEdge && 1 >= _s32ClockEdge)
   {
	   _bReturn = TRUE;
   }

   return _bReturn;
}

/*******************************************************************************
*
* FUNCTION:    bIsEarlyConfig_ClockGenerationLvds1ManuallySet()
*
* DESCRIPTION: returns TRUE if EarlyConfig_TwoDisplays clock generation LVDS 1 manually set
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - manually set | else - not manually set}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsEarlyConfig_ClockGenerationLvds1ManuallySet(tVoid) const
{
   ETG_TRACE_USR4(("bIsEarlyConfig_ClockGenerationLvds1ManuallySet() entered."));

   dp_tclDimmingDPEarlyConfig_ClockGenerationLvds1_Manually _oClockGeneration;
   tS32 _s32ClockGeneration = _oClockGeneration.tGetData();
   tBool _bReturn = FALSE;

   if ((-1 <= _s32ClockGeneration && 1 >= _s32ClockGeneration) || (8 <= _s32ClockGeneration && 15 >= _s32ClockGeneration))
   {
	   _bReturn = TRUE;
   }

   return _bReturn;
}

/*******************************************************************************
*
* FUNCTION:    bIsEarlyConfig_ClockGenerationLvds2ManuallySet()
*
* DESCRIPTION: returns TRUE if EarlyConfig_TwoDisplays clock generation LVDS 2 manually set
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - manually set | else - not manually set}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsEarlyConfig_ClockGenerationLvds2ManuallySet(tVoid) const
{
   ETG_TRACE_USR4(("bIsEarlyConfig_ClockGenerationLvds1ManuallySet() entered."));

   dp_tclDimmingDPEarlyConfig_ClockGenerationLvds2_Manually _oClockGeneration;
   tS32 _s32ClockGeneration = _oClockGeneration.tGetData();
   tBool _bReturn = FALSE;

   if ((-1 <= _s32ClockGeneration && 1 >= _s32ClockGeneration) || (8 <= _s32ClockGeneration && 15 >= _s32ClockGeneration))
   {
	   _bReturn = TRUE;
   }

   return _bReturn;
}

/*******************************************************************************
*
* FUNCTION:    bLoadKDSEarlyConfig_TwoDisplays()
*
* DESCRIPTION: loading KDS EarlyConfig_TwoDisplays data
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bLoadKDSEarlyConfig_TwoDisplays()
{
   ETG_TRACE_USR4(("bLoadKDSEarlyConfig_TwoDisplays() entered."));

   tBool _bReturn = TRUE;
   m_bKDSTwoDisplaysLoaded = FALSE;
   tKDS_EarlyConfig_TwoDisplaysRaw _tKDSTwoDisplaysRaw;
   (tVoid)OSAL_pvMemorySet((tVoid*)&_tKDSTwoDisplaysRaw, OSAL_NULL, sizeof(_tKDSTwoDisplaysRaw));
   // load timing LVDS1
   tS32 _s32ReturnCode = DP_s32GetConfigItem( "EarlyConfig2Displays", "TrTimingLVDS1",
		   _tKDSTwoDisplaysRaw.tTwoDisplaysRaw.au8TimingLvds1, sizeof(_tKDSTwoDisplaysRaw.tTwoDisplaysRaw.au8TimingLvds1), FALSE);
   if  (DP_S32_NO_ERR > _s32ReturnCode)
   {
	   ETG_TRACE_USR3(("bLoadKDSEarlyConfig_TwoDisplays: EarlyConfig2Displays.timingLVDS1 could not read (error=0x%08x)", _s32ReturnCode));
	   _bReturn = FALSE;
   }
   else
   {
	   ETG_TRACE_USR3(("bLoadKDSEarlyConfig_TwoDisplays: timingLVDS1=%32s", (tChar*)_tKDSTwoDisplaysRaw.tTwoDisplaysRaw.au8TimingLvds1 ));
	   (tVoid)OSAL_pvMemorySet((tVoid*)m_strTimingLvds1, OSAL_NULL, sizeof(m_strTimingLvds1));
	   (tVoid)OSAL_szStringNCopy(m_strTimingLvds1, (const char*)_tKDSTwoDisplaysRaw.tTwoDisplaysRaw.au8TimingLvds1, sizeof(m_strTimingLvds1)-1);
   }
   // load timing LVDS2
   _s32ReturnCode = DP_s32GetConfigItem( "EarlyConfig2Displays", "TrTimingLVDS2",
		   _tKDSTwoDisplaysRaw.tTwoDisplaysRaw.au8TimingLvds2, sizeof(_tKDSTwoDisplaysRaw.tTwoDisplaysRaw.au8TimingLvds2), FALSE);
   if  (DP_S32_NO_ERR > _s32ReturnCode)
   {
	   ETG_TRACE_USR3(("bLoadKDSEarlyConfig_TwoDisplays: EarlyConfig2Displays.timingLVDS2 could not read (error=0x%08x)", _s32ReturnCode));
	   _bReturn = FALSE;
   }
   else
   {
	   ETG_TRACE_USR3(("bLoadKDSEarlyConfig_TwoDisplays: timingLVDS2=%32s", (tChar*)_tKDSTwoDisplaysRaw.tTwoDisplaysRaw.au8TimingLvds2 ));
	   (tVoid)OSAL_pvMemorySet((tVoid*)m_strTimingLvds2, OSAL_NULL, sizeof(m_strTimingLvds2));
	   (tVoid)OSAL_szStringNCopy(m_strTimingLvds2, (const char*)_tKDSTwoDisplaysRaw.tTwoDisplaysRaw.au8TimingLvds2, sizeof(m_strTimingLvds2)-1);
   }
   // DispTrClockEdgeSelectLVDS1
   tU8 _u8TrClockEdgeSelectLVDS = OSAL_C_U8_MAX;
   _s32ReturnCode = DP_s32GetConfigItem( "AIVIVariantCoding", "DispTrClockEdgeSelectLVDS1", &_u8TrClockEdgeSelectLVDS, sizeof(_u8TrClockEdgeSelectLVDS), FALSE);
   if (DP_S32_NO_ERR > _s32ReturnCode)
   {
	   ETG_TRACE_USR3(("bLoadKDSEarlyConfig_TwoDisplays: EarlyConfig2Displays TRClockEdgeLVDS1 could not read (error=0x%08x)", _s32ReturnCode));
	   _bReturn = FALSE;
   }
   else if (1 >=  _u8TrClockEdgeSelectLVDS)
   {
	   m_s8ClockEdgeSelectLVDS1 = (tS8)_u8TrClockEdgeSelectLVDS;
   }
   else
   {
	   m_s8ClockEdgeSelectLVDS1 = -1;
   }
   // DispTrClockEdgeSelectLVDS2
   _u8TrClockEdgeSelectLVDS = OSAL_C_U8_MAX;
   _s32ReturnCode = DP_s32GetConfigItem( "AIVIVariantCoding", "DispTrClockEdgeSelectLVDS2", &_u8TrClockEdgeSelectLVDS, sizeof(_u8TrClockEdgeSelectLVDS), FALSE);
   if (DP_S32_NO_ERR > _s32ReturnCode)
   {
	   ETG_TRACE_USR3(("bLoadKDSEarlyConfig_TwoDisplays: EarlyConfig2Displays TRClockEdgeLVDS2 could not read (error=0x%08x)", _s32ReturnCode));
	   _bReturn = FALSE;
   }
   else if (1 >=  _u8TrClockEdgeSelectLVDS)
   {
	   m_s8ClockEdgeSelectLVDS2 = (tS8)_u8TrClockEdgeSelectLVDS;
   }
   else
   {
	   m_s8ClockEdgeSelectLVDS2 = -1;
   }
   // SpreadSpectrumClockGenerationLVDS
#ifdef DP_U8_KDSLEN_AIVIVARIANTCODING_SPREADSPECTRUMCLOCKGENERATIONLVDS
   tU8 _u8ClockGeneration[DP_U8_KDSLEN_AIVIVARIANTCODING_SPREADSPECTRUMCLOCKGENERATIONLVDS] = {0xFF};
   (tVoid)OSAL_pvMemorySet((tVoid*)_u8ClockGeneration, OSAL_C_U32_MAX, sizeof(_u8ClockGeneration));
   _s32ReturnCode = DP_s32GetConfigItem( "AIVIVariantCoding", "SpreadSpectrumClockGenerationLVDS", _u8ClockGeneration, sizeof(_u8ClockGeneration), FALSE);
   if (DP_S32_NO_ERR > _s32ReturnCode)
   {
	   ETG_TRACE_ERR(("bLoadKDSEarlyConfig_TwoDisplays: EarlyConfig2Displays SpreadSpectrumClockGenerationLVDS could not read (error=0x%08x)", _s32ReturnCode));
   }
   if (((8 <= _u8ClockGeneration[0]) && (15 >= _u8ClockGeneration[0])) || (0 == _u8ClockGeneration[0]))
   {
	   m_s32ClockGenerationLVDS1 = (tS32)_u8ClockGeneration[0];
   }
   else
   {
	   m_s32ClockGenerationLVDS1 = -1;
   }
   if (((8 <= _u8ClockGeneration[1]) && (15 >= _u8ClockGeneration[1])) || (0 == _u8ClockGeneration[1]))
   {
	   m_s32ClockGenerationLVDS2 = (tS32)_u8ClockGeneration[1];
   }
   else
   {
	   m_s32ClockGenerationLVDS2 = -1;
   }
#else
   ETG_TRACE_FATAL(("bLoadKDSEarlyConfig_TwoDisplays: EarlyConfig2Displays SpreadSpectrumClockGenerationLVDS unsupported!"));
   m_s32ClockGenerationLVDS1 = -1;
   m_s32ClockGenerationLVDS2 = -1;
#endif //#ifdef DP_U8_KDSLEN_AIVIVARIANTCODING_SPREADSPECTRUMCLOCKGENERATIONLVDS

   if (TRUE == _bReturn)
   {
       ETG_TRACE_USR2(("bLoadKDSEarlyConfig_TwoDisplays: KDS early config two displays loaded for LVDS1 with timing=%32s, clock_edge=%d, clock_generation=%d ",
    		   m_strTimingLvds1, m_s8ClockEdgeSelectLVDS1, m_s32ClockGenerationLVDS1));
       ETG_TRACE_USR2(("bLoadKDSEarlyConfig_TwoDisplays: KDS early config two displays loaded for LVDS2 with timing=%32s, clock_edge=%d, clock_generation=%d ",
    		   m_strTimingLvds2, m_s8ClockEdgeSelectLVDS2, m_s32ClockGenerationLVDS2));
	   m_bKDSTwoDisplaysLoaded = TRUE;
   }
   else
   {
       ETG_TRACE_ERR(("bLoadKDSEarlyConfig_TwoDisplays: KDS early config two displays could not loaded!"));
   }

   return bIsLoadedKDSEarlyConfig_TwoDisplays();
}

/*******************************************************************************
*
* FUNCTION:    bIsValidKDSEarlyConfig_TwoDisplays()
*
* DESCRIPTION: returns TRUE if loaded KDS data are valid
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - valid | else - not valid}
*
*******************************************************************************/
// possible timing LVDS 1 names for A-IVI
const char* m_astrEarlyConfigTimingLvds1Names[] =
{
		"displayGM_800x480_60Hz",
		"displayGM_1280x800_60Hz"
};
// possible timing LVDS 2 names for A-IVI
const char* m_astrEarlyConfigTimingLvds2Names[] =
{
		"lq070y5dg36",
		"lq080y5dw02"
};
tBool dispvidctrl_tclEarlyConfig::bIsValidKDSEarlyConfig_TwoDisplays() const
{
   ETG_TRACE_USR4(("bIsValidKDSEarlyConfig_TwoDisplays() entered."));

   tBool _bValid = bIsLoadedKDSEarlyConfig_TwoDisplays();
   // valid as specified
   // ... timing of LVDS 1 and LVDS 2 should not empty
   if (((FALSE == bIsEarlyConfig_TimingLvds1ManuallySet()) && (OSAL_NULL == OSAL_u32StringLength(m_strTimingLvds1)))
	   && ((FALSE == bIsEarlyConfig_TimingLvds2ManuallySet()) && (OSAL_NULL == OSAL_u32StringLength(m_strTimingLvds2)))
	   && (FALSE == bIsValidKDSCustom_TouchScreenConfig()))
   {
	   _bValid = FALSE;
	   ETG_TRACE_USR3(("bIsValidKDSEarlyConfig_TwoDisplays: invalid timing for LVDS1 and LVDS2 (both empty)!" ));
   }
   // ... clock edge should be in range from -1 and 1
   if ((FALSE == bIsEarlyConfig_ClockEdgeLvds1ManuallySet()) && ((-1 > m_s8ClockEdgeSelectLVDS1) || (1 < m_s8ClockEdgeSelectLVDS1)))
   {
	   _bValid = FALSE;
	   ETG_TRACE_USR3(("bIsValidKDSEarlyConfig_TwoDisplays: invalid clock edge %d for LVDS1!",
			   m_s8ClockEdgeSelectLVDS1 ));
   }
   if ((FALSE == bIsEarlyConfig_ClockEdgeLvds2ManuallySet()) && ((-1 > m_s8ClockEdgeSelectLVDS2) || (1 < m_s8ClockEdgeSelectLVDS2)))
   {
	   _bValid = FALSE;
	   ETG_TRACE_USR3(("bIsValidKDSEarlyConfig_TwoDisplays: invalid clock edge %d for LVDS2!",
			   m_s8ClockEdgeSelectLVDS2 ));
   }
   // ... clock generation should be in range from -1, 0, 8 ... 15
   if (FALSE == bIsEarlyConfig_ClockGenerationLvds1ManuallySet())
   {
	   if ((-1 > m_s32ClockGenerationLVDS1) || (15 < m_s32ClockGenerationLVDS1))
	   {
		   _bValid = FALSE;
		   ETG_TRACE_USR3(("bIsValidKDSEarlyConfig_TwoDisplays: invalid spread spectrum clock generation %d for LVDS1!",
				   m_s8ClockEdgeSelectLVDS1 ));
	   }
	   else if ((0 < m_s32ClockGenerationLVDS1) && (8 > m_s32ClockGenerationLVDS1))
	   {
		   _bValid = FALSE;
		   ETG_TRACE_USR3(("bIsValidKDSEarlyConfig_TwoDisplays: invalid spread spectrum clock generation %d for LVDS1!",
				   m_s8ClockEdgeSelectLVDS1 ));
	   }
   }
   if (FALSE == bIsEarlyConfig_ClockGenerationLvds2ManuallySet())
   {
	   if ((-1 > m_s32ClockGenerationLVDS2) || (15 < m_s32ClockGenerationLVDS2))
	   {
		   _bValid = FALSE;
		   ETG_TRACE_USR3(("bIsValidKDSEarlyConfig_TwoDisplays: invalid spread spectrum clock generation %d for LVDS2!",
				   m_s32ClockGenerationLVDS2 ));
	   }
	   else if ((0 < m_s32ClockGenerationLVDS2) && (8 > m_s32ClockGenerationLVDS2))
	   {
		   _bValid = FALSE;
		   ETG_TRACE_USR3(("bIsValidKDSEarlyConfig_TwoDisplays: invalid spread spectrum clock generation %d for LVDS2!",
				   m_s32ClockGenerationLVDS2 ));
	   }
   }

   return _bValid;
}

/*******************************************************************************
*
* FUNCTION:    bIsChangedEarlyConfig_TwoDisplays()
*
* DESCRIPTION: returns TRUE if KDS data not the same like early config data
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - new | else - not new}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsChangedEarlyConfig_TwoDisplays() const
{
   ETG_TRACE_USR4(("bIsChangedEarlyConfig_TwoDisplays() entered."));

   tBool _bChanged = FALSE;
   // timing LVDS 1
   dp_tclEarlyConfigTwoDisplaysTrTimingLVDS1 _oTimingLvds1;
   tChar _strTimingLvds1[sizeof(m_strTimingLvds1)];
   (tVoid)OSAL_pvMemorySet((tVoid*)_strTimingLvds1, OSAL_NULL, sizeof(_strTimingLvds1));
   tS32 _s32ReadStatus = _oTimingLvds1.s32GetData(_strTimingLvds1, sizeof(_strTimingLvds1)-1);
   if (DP_S32_NO_ERR > _oTimingLvds1.s32GetStatus())
   {
	   // not initialized
	   _bChanged = bIsLoadedKDSEarlyConfig_TwoDisplays();
	   ETG_TRACE_USR2(("bIsChangedEarlyConfig_TwoDisplays: Status Timing LVDS 1: %d.", _oTimingLvds1.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   // could be loaded
	   if (OSAL_NULL != OSAL_s32StringCompare(m_strTimingLvds1,_strTimingLvds1))
	   {
		   _bChanged = bIsLoadedKDSEarlyConfig_TwoDisplays();
		   ETG_TRACE_USR3(("bIsChangedEarlyConfig_TwoDisplays: Timing LVDS 1 Compare: %d.", OSAL_s32StringCompare(m_strTimingLvds1,_strTimingLvds1) ));
	   }
	   else
	   {
		   ETG_TRACE_USR3(("bIsChangedEarlyConfig_TwoDisplays: Timing LVDS 1 Compare: %d.", OSAL_s32StringCompare(m_strTimingLvds1,_strTimingLvds1) ));
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("bIsChangedEarlyConfig_TwoDisplays: Read Error Timing LVDS 1 detected %d!", _s32ReadStatus ));
   }
   // timing LVDS 2
   dp_tclEarlyConfigTwoDisplaysTrTimingLVDS1 _oTimingLvds2;
   tChar _strTimingLvds2[sizeof(m_strTimingLvds2)];
   (tVoid)OSAL_pvMemorySet((tVoid*)_strTimingLvds2, OSAL_NULL, sizeof(_strTimingLvds2));
   _s32ReadStatus = _oTimingLvds2.s32GetData(_strTimingLvds2, sizeof(_strTimingLvds2)-1);
   if (DP_S32_NO_ERR > _oTimingLvds2.s32GetStatus())
   {
	   // not initialized
	   _bChanged = bIsLoadedKDSEarlyConfig_TwoDisplays();
	   ETG_TRACE_USR2(("bIsChangedEarlyConfig_TwoDisplays: Status Timing LVDS 2: %d.", _oTimingLvds2.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   // could be loaded
	   if (OSAL_NULL != OSAL_s32StringCompare(m_strTimingLvds2,_strTimingLvds2))
	   {
		   _bChanged = bIsLoadedKDSEarlyConfig_TwoDisplays();
		   ETG_TRACE_USR3(("bIsChangedEarlyConfig_TwoDisplays: Timing LVDS 2 Compare: %d.", OSAL_s32StringCompare(m_strTimingLvds2,_strTimingLvds2) ));
	   }
	   else
	   {
		   ETG_TRACE_USR3(("bIsChangedEarlyConfig_TwoDisplays: Timing LVDS 2 Compare: %d.", OSAL_s32StringCompare(m_strTimingLvds2,_strTimingLvds2) ));
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("bIsChangedEarlyConfig_TwoDisplays: Read Error Timing LVDS 2 detected %d!", _s32ReadStatus ));
   }
   // clock edge LVDS 1
#ifndef VARIANT_S_FTR_ENABLE_AIVI_SCOPE1
   dp_tclEarlyConfigTwoDisplaysTrClockEdgeSelectLVDS1 _oClockEdgeLVDS1;
   if ((tS32)m_s8ClockEdgeSelectLVDS1 != _oClockEdgeLVDS1.tGetData())
   {
	   _bChanged = TRUE;
   }
#endif //#ifndef VARIANT_S_FTR_ENABLE_AIVI_SCOPE1
   // clock edge LVDS 2
#ifndef VARIANT_S_FTR_ENABLE_AIVI_SCOPE1
   dp_tclEarlyConfigTwoDisplaysTrClockEdgeSelectLVDS2 _oClockEdgeLVDS2;
   if ((tS32)m_s8ClockEdgeSelectLVDS2 != _oClockEdgeLVDS2.tGetData())
   {
	   _bChanged = TRUE;
   }
#endif //#ifndef VARIANT_S_FTR_ENABLE_AIVI_SCOPE1
   // clock generation LVDS 1
#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
   dp_tclEarlyConfigTwoDisplaysTrSpreadSpectrumClockGenerationLVDS1 _oClockGenerationLVDS1;
   if ((tS32)m_s32ClockGenerationLVDS1 != _oClockGenerationLVDS1.tGetData())
   {
	   _bChanged = TRUE;
   }
#else
   ETG_TRACE_FATAL(("bIsChangedEarlyConfig_TwoDisplays: Spread Spectrum Clock Generation LVDS 1 unsupported!" ));
#endif //#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
   // clock generation LVDS 2
#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
   dp_tclEarlyConfigTwoDisplaysTrSpreadSpectrumClockGenerationLVDS2 _oClockGenerationLVDS2;
   if ((tS32)m_s32ClockGenerationLVDS2 != _oClockGenerationLVDS2.tGetData())
   {
	   _bChanged = TRUE;
   }
#else
   ETG_TRACE_FATAL(("bIsChangedEarlyConfig_TwoDisplays: Spread Spectrum Clock Generation LVDS 2 unsupported!" ));
#endif //#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER

   return _bChanged;
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig_TwoDisplays()
*
* DESCRIPTION: write the EarlyConfig_TwoDisplays
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig_TwoDisplays()
{
   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays() entered."));

   tBool _bReturn = TRUE;
   // LVDS1
   char _strLVDS1_Timing[sizeof(m_strTimingLvds1)];
   (tVoid)memset((tVoid*)_strLVDS1_Timing, OSAL_NULL, sizeof(_strLVDS1_Timing));
   tS32 _s32LVDS1_ClockEdgeSelect = -1;
   tS32 _s32LVDS1_ClockGeneration = -1;
   tBool _bLVDS1_Customized = FALSE;
   // LVDS2
   char _strLVDS2_Timing[sizeof(m_strTimingLvds2)];
   (tVoid)memset((tVoid*)_strLVDS2_Timing, OSAL_NULL, sizeof(_strLVDS2_Timing));
   tS32 _s32LVDS2_ClockEdgeSelect = -1;
   tS32 _s32LVDS2_ClockGeneration = -1;
   tBool _bLVDS2_Customized = FALSE;
   // Dependency: EarlyConfig_LayerManager.DisplayOrder and LVDS device
   dp_tclEarlyConfigLayerManagerTrDisplayOrder _oDpDisplayOrder;
   tU8 _au8DisplayOrder[sizeof(m_au8DisplayOrder)];
   (tVoid)OSAL_pvMemorySet((tVoid*)_au8DisplayOrder, OSAL_NULL, sizeof(_au8DisplayOrder));
   (tVoid)_oDpDisplayOrder.s32GetData(_au8DisplayOrder, EARLYCONFIG_ARRAY_ELEMENTS(_au8DisplayOrder));
   // load current EarlyConfig values
   dp_tclEarlyConfigTwoDisplaysTrTimingLVDS1 _oTimingLvds1;
   tS32 _s32ReadStatus = _oTimingLvds1.s32GetData(_strLVDS1_Timing, sizeof(_strLVDS1_Timing)-1);
   if (DP_S32_NO_ERR > _oTimingLvds1.s32GetStatus())
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Could not load Timing LVDS 1 (status: %d)!", _oTimingLvds1.s32GetStatus() ));
   }
   dp_tclEarlyConfigTwoDisplaysTrClockEdgeSelectLVDS1 _oClockEdgeLvds1;
   _s32ReadStatus = _oClockEdgeLvds1.s32GetData(_s32LVDS1_ClockEdgeSelect);
   if (DP_S32_NO_ERR > _oClockEdgeLvds1.s32GetStatus())
   {
   	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Could not load Clock Edge Selected LVDS 1 (status: %d)!", _oClockEdgeLvds1.s32GetStatus() ));
   }
#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
   dp_tclEarlyConfigTwoDisplaysTrSpreadSpectrumClockGenerationLVDS1 _oClockGenerationLvds1;
   _s32ReadStatus = _oClockGenerationLvds1.s32GetData(_s32LVDS1_ClockGeneration);
   if (DP_S32_NO_ERR > _oClockGenerationLvds1.s32GetStatus())
   {
   	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Could not load Clock Generation LVDS 1 (status: %d)!", _oClockGenerationLvds1.s32GetStatus() ));
   }
#else
   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Spread Spectrum Clock Generation LVDS 1 unsupported!" ));
#endif //#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
   dp_tclEarlyConfigTwoDisplaysTrTimingLVDS2 _oTimingLvds2;
   _s32ReadStatus = _oTimingLvds2.s32GetData(_strLVDS2_Timing, sizeof(_strLVDS2_Timing)-1);
   if (DP_S32_NO_ERR > _oTimingLvds2.s32GetStatus())
   {
   	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Could not load Timing LVDS 2 (status: %d)!", _oTimingLvds2.s32GetStatus() ));
   }
   dp_tclEarlyConfigTwoDisplaysTrClockEdgeSelectLVDS2 _oClockEdgeLvds2;
   _s32ReadStatus = _oClockEdgeLvds2.s32GetData(_s32LVDS2_ClockEdgeSelect);
   if (DP_S32_NO_ERR > _oClockEdgeLvds2.s32GetStatus())
   {
   	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Could not load Clock Edge Selected LVDS 2 (status: %d)!", _oClockEdgeLvds2.s32GetStatus() ));
   }
#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
   dp_tclEarlyConfigTwoDisplaysTrSpreadSpectrumClockGenerationLVDS2 _oClockGenerationLvds2;
   _s32ReadStatus = _oClockGenerationLvds2.s32GetData(_s32LVDS2_ClockGeneration);
   if (DP_S32_NO_ERR > _oClockGenerationLvds2.s32GetStatus())
   {
   	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Could not load Clock Generation LVDS 2 (status: %d)!", _oClockGenerationLvds2.s32GetStatus() ));
   }
#else
   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Spread Spectrum Clock Generation LVDS 2 unsupported!" ));
#endif //#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER

   // loading customer touch screen configuration depending of EarlyConfig_LayerManager.DisplayOrder
   if (u8GetTouchScreenConfig() != 0 && TRUE == bIsValidKDSCustom_TouchScreenConfig())
   {
	   // for Main HMI screen LVDS1
	   if (EARLYCONFIG_C_U8_SCREENID_LVDS1 == _au8DisplayOrder[EARLYCONFIG_C_U8_SCREENTYPE_MAINHMI])
	   {
		   _bLVDS1_Customized = TRUE;
		   // Main HMI screen is mapped to LVDS1
		   (tVoid)OSAL_szStringNCopy(_strLVDS1_Timing, cstrGetCustomDisplayDriver(), sizeof(_strLVDS1_Timing)-1);
		   _s32LVDS1_ClockEdgeSelect = s32GetCustomDisplayDriverClockEdgeSelect();
		   _s32LVDS1_ClockGeneration = s32GetCustomDisplayDriverClockGeneration();
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays(): customized LVDS1 configuration (timing: %63s, clock_edge: %d, clock_generation: %d).",
				   _strLVDS1_Timing, _s32LVDS1_ClockEdgeSelect, _s32LVDS1_ClockGeneration ));
	   }
	   // for Main HMI screen LVDS2
	   if (EARLYCONFIG_C_U8_SCREENID_LVDS2 == _au8DisplayOrder[EARLYCONFIG_C_U8_SCREENTYPE_MAINHMI])
	   {
		   _bLVDS2_Customized = TRUE;
		   // Main HMI screen is mapped to LVDS2
		   (tVoid)OSAL_szStringNCopy(_strLVDS2_Timing, cstrGetCustomDisplayDriver(), sizeof(_strLVDS2_Timing)-1);
		   _s32LVDS2_ClockEdgeSelect = s32GetCustomDisplayDriverClockEdgeSelect();
		   _s32LVDS2_ClockGeneration = s32GetCustomDisplayDriverClockGeneration();
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays(): customized LVDS2 configuration (timing: %63s, clock_edge: %d, clock_generation: %d).",
				   _strLVDS2_Timing, _s32LVDS2_ClockEdgeSelect, _s32LVDS2_ClockGeneration ));
	   }
   }
   // loading PD configuration of no customized configuration
   if (TRUE == bIsValidKDSEarlyConfig_TwoDisplays())
   {
	   // LVDS1
	   if (FALSE == _bLVDS1_Customized)
	   {
		   (tVoid)OSAL_szStringNCopy(_strLVDS1_Timing, m_strTimingLvds1, sizeof(_strLVDS1_Timing)-1);
		   _s32LVDS1_ClockEdgeSelect = m_s8ClockEdgeSelectLVDS1;
		   _s32LVDS1_ClockGeneration = m_s32ClockGenerationLVDS1;
	   }
	   // LVDS2
	   if (FALSE == _bLVDS2_Customized)
	   {
		   (tVoid)OSAL_szStringNCopy(_strLVDS2_Timing, m_strTimingLvds2, sizeof(_strLVDS2_Timing)-1);
		   _s32LVDS2_ClockEdgeSelect = m_s8ClockEdgeSelectLVDS2;
		   _s32LVDS2_ClockGeneration = m_s32ClockGenerationLVDS2;
	   }
   }
   // loading all manual settings
   // ... for LVDS1
   if (TRUE == bIsEarlyConfig_TimingLvds1ManuallySet())
   {
	   dp_tclDimmingDPEarlyConfig_TimingLvds1_Manually _oManualSet;
	   _s32ReadStatus = _oManualSet.s32GetData(_strLVDS1_Timing, sizeof(_strLVDS1_Timing)-1);
	   if (DP_S32_NO_ERR > _oManualSet.s32GetStatus())
	   {
	   	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Could not load manual set Timing LVDS1 (status: %d)!", _oManualSet.s32GetStatus() ));
	   }
   }
   if (TRUE == bIsEarlyConfig_ClockEdgeLvds1ManuallySet())
   {
	   dp_tclDimmingDPEarlyConfig_ClockEdgeLvds1_Manually _oManualSet;
	   _s32ReadStatus = _oManualSet.s32GetData(_s32LVDS1_ClockEdgeSelect);
	   if (DP_S32_NO_ERR > _oManualSet.s32GetStatus())
	   {
	   	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Could not load manual set Clock Edge Selected LVDS1 (status: %d)!", _oManualSet.s32GetStatus() ));
	   }
   }
   if (TRUE == bIsEarlyConfig_ClockGenerationLvds1ManuallySet())
   {
	   dp_tclDimmingDPEarlyConfig_ClockGenerationLvds1_Manually _oManualSet;
	   _s32ReadStatus = _oManualSet.s32GetData(_s32LVDS1_ClockGeneration);
	   if (DP_S32_NO_ERR > _oManualSet.s32GetStatus())
	   {
	   	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Could not load manual set Spread Spectrum Clock Generation LVDS1 (status: %d)!", _oManualSet.s32GetStatus() ));
	   }
   }
   // ... for LVDS2
   if (TRUE == bIsEarlyConfig_TimingLvds2ManuallySet())
   {
	   dp_tclDimmingDPEarlyConfig_TimingLvds2_Manually _oManualSet;
	   _s32ReadStatus = _oManualSet.s32GetData(_strLVDS2_Timing, sizeof(_strLVDS2_Timing)-1);
	   if (DP_S32_NO_ERR > _oManualSet.s32GetStatus())
	   {
	   	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Could not load manual set Timing LVDS2 (status: %d)!", _oManualSet.s32GetStatus() ));
	   }
   }
   if (TRUE == bIsEarlyConfig_ClockEdgeLvds2ManuallySet())
   {
	   dp_tclDimmingDPEarlyConfig_ClockEdgeLvds2_Manually _oManualSet;
	   _s32ReadStatus = _oManualSet.s32GetData(_s32LVDS2_ClockEdgeSelect);
	   if (DP_S32_NO_ERR > _oManualSet.s32GetStatus())
	   {
	   	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Could not load manual set Clock Edge Selected LVDS2 (status: %d)!", _oManualSet.s32GetStatus() ));
	   }
   }
   if (TRUE == bIsEarlyConfig_ClockGenerationLvds2ManuallySet())
   {
	   dp_tclDimmingDPEarlyConfig_ClockGenerationLvds2_Manually _oManualSet;
	   _s32ReadStatus = _oManualSet.s32GetData(_s32LVDS2_ClockGeneration);
	   if (DP_S32_NO_ERR > _oManualSet.s32GetStatus())
	   {
	   	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Could not load manual set Spread Spectrum Clock Generation LVDS2 (status: %d)!", _oManualSet.s32GetStatus() ));
	   }
   }

   // write all data to EarlyConfig_TwoDisplays
   // ... for LVDS1
   if (DP_S32_NO_ERR > _oTimingLvds1.s32SetData(_strLVDS1_Timing))
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Failed to set timing LVDS 1 %s to early config!", _strLVDS1_Timing ));
	   _bReturn = FALSE;
   }
   else
   {
	   ETG_TRACE_USR2(("bWriteEarlyConfig_TwoDisplays: Successful set timing LVDS 1 %s to early config.", _strLVDS1_Timing ));
   }
#ifndef VARIANT_S_FTR_ENABLE_AIVI_SCOPE1
   if (DP_S32_NO_ERR > _oClockEdgeLvds1.s32SetData((tS32)_s32LVDS1_ClockEdgeSelect))
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Failed to set clock edge LVDS 1 %d to early config!", _s32LVDS1_ClockEdgeSelect ));
	   _bReturn = FALSE;
   }
   else
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Clock edge LVDS 1 set with %d to early config.", _s32LVDS1_ClockEdgeSelect ));
   }
#endif //#ifndef VARIANT_S_FTR_ENABLE_AIVI_SCOPE1
#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
   if (DP_S32_NO_ERR > _oClockGenerationLvds1.s32SetData((tS32)_s32LVDS1_ClockGeneration))
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Failed to set spread spectrum clock generation LVDS 1 %d to early config!", _s32LVDS1_ClockGeneration ));
	   _bReturn = FALSE;
   }
   else
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Spread spectrum clock generation LVDS 1 set with %d to early config.", _s32LVDS1_ClockGeneration ));
   }
#else
   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Spread Spectrum Clock Generation LVDS 1 (kds value: %d) unsupported!", m_s32ClockGenerationLVDS1 ));
#endif //#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
   // ... for LVDS2
   if (DP_S32_NO_ERR > _oTimingLvds2.s32SetData(_strLVDS2_Timing))
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Failed to set timing LVDS 2 %s to early config!", _strLVDS2_Timing ));
	   _bReturn = FALSE;
   }
   else
   {
	   ETG_TRACE_USR2(("bWriteEarlyConfig_TwoDisplays: Successful set timing LVDS 2 %s to early config.", _strLVDS2_Timing ));
   }
#ifndef VARIANT_S_FTR_ENABLE_AIVI_SCOPE1
   if (DP_S32_NO_ERR > _oClockEdgeLvds2.s32SetData((tS32)_s32LVDS2_ClockEdgeSelect))
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Failed to set clock edge LVDS 2 %d to early config!", _s32LVDS2_ClockEdgeSelect ));
	   _bReturn = FALSE;
   }
   else
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Clock edge LVDS 2 set with %d to early config.", _s32LVDS2_ClockEdgeSelect ));
   }
#endif //#ifndef VARIANT_S_FTR_ENABLE_AIVI_SCOPE1
#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
   if (DP_S32_NO_ERR > _oClockGenerationLvds2.s32SetData((tS32)_s32LVDS2_ClockGeneration))
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Failed to set spread spectrum clock generation LVDS 2 %d to early config!", _s32LVDS2_ClockGeneration ));
	   _bReturn = FALSE;
   }
   else
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Spread spectrum clock generation LVDS 2 set with %d to early config.", _s32LVDS2_ClockGeneration ));
   }
#else
   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoDisplays: Spread Spectrum Clock Generation LVDS 2 (kds value: %d) unsupported!", m_s32ClockGenerationLVDS2 ));
#endif //#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
	// trigger storing of datapool
	dp_tclSrvIf _oDpSrvIf;
	tU32 _u32DpID = DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER;
	tS32 _s32ErrorCode = _oDpSrvIf.s32StorePoolNow(_u32DpID);
	if (DP_S32_NO_ERR != _s32ErrorCode)
	{
	  ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoTouchDriver: storing of PDD 0x%08x failed with error code 0x%08x!", _u32DpID, _s32ErrorCode ));
	}
#endif // #ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
   return _bReturn;
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfoEarlyConfig_TwoDisplays() const
*
* DESCRIPTION: Trace information EarlyConfig_TwoDisplays
*
* PARAMETER:   const char* strIdentifier
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid dispvidctrl_tclEarlyConfig::vTraceInfoEarlyConfig_TwoDisplays(const char* strIdentifier) const
{
	ETG_TRACE_USR4(("vTraceInfoEarlyConfig_TwoDisplays() entered."));
	ETG_TRACE_FATAL(("%26s: <<< EarlyConfig_TwoDisplays >>> ", strIdentifier ));

	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, " 2DISP_TLVDS1_MANUAL",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsEarlyConfig_TimingLvds1ManuallySet()) ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, " 2DISP_CLVDS1_MANUAL",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsEarlyConfig_ClockEdgeLvds1ManuallySet()) ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "2DISP_CGLVDS1_MANUAL",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsEarlyConfig_ClockGenerationLvds1ManuallySet()) ));


	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, " 2DISP_TLVDS2_MANUAL",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsEarlyConfig_TimingLvds2ManuallySet()) ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, " 2DISP_CLVDS2_MANUAL",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsEarlyConfig_ClockEdgeLvds2ManuallySet()) ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "2DISP_CGLVDS2_MANUAL",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsEarlyConfig_ClockGenerationLvds2ManuallySet()) ));

	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "    KDS_2DISP_LOADED",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsLoadedKDSEarlyConfig_TwoDisplays()) ));

	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "     KDS_2DISP_VALID",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsValidKDSEarlyConfig_TwoDisplays()) ));

	ETG_TRACE_FATAL(("%26s: %30s=%32s ", strIdentifier, "KDS_2DISP_TIME_LVDS1",
			m_strTimingLvds1 ));
	ETG_TRACE_FATAL(("%26s: %30s=%d ",   strIdentifier, " KDS_2DISP_CES_LVDS1",
			m_s8ClockEdgeSelectLVDS1 ));
#ifdef DP_U8_KDSLEN_AIVIVARIANTCODING_SPREADSPECTRUMCLOCKGENERATIONLVDS
	ETG_TRACE_FATAL(("%26s: %30s=%d ",   strIdentifier, "KDS_2DISP_SSCG_LVDS1",
			m_s32ClockGenerationLVDS1 ));
#else
	ETG_TRACE_FATAL(("%26s: %30s=%d (unsupported) ", strIdentifier, "KDS_2DISP_SSCG_LVDS1",
			m_s32ClockGenerationLVDS1 ));
#endif //#ifdef DP_U8_KDSLEN_AIVIVARIANTCODING_SPREADSPECTRUMCLOCKGENERATIONLVDS
	ETG_TRACE_FATAL(("%26s: %30s=%32s ", strIdentifier, "KDS_2DISP_TIME_LVDS2",
			m_strTimingLvds2 ));
	ETG_TRACE_FATAL(("%26s: %30s=%d ",   strIdentifier, " KDS_2DISP_CES_LVDS2",
			m_s8ClockEdgeSelectLVDS2 ));
#ifdef DP_U8_KDSLEN_AIVIVARIANTCODING_SPREADSPECTRUMCLOCKGENERATIONLVDS
	ETG_TRACE_FATAL(("%26s: %30s=%d ",   strIdentifier, "KDS_2DISP_SSCG_LVDS2",
			m_s32ClockGenerationLVDS2 ));
#else
	ETG_TRACE_FATAL(("%26s: %30s=%d (unsupported) ", strIdentifier, "KDS_2DISP_SSCG_LVDS2",
			m_s32ClockGenerationLVDS2 ));
#endif //#ifdef DP_U8_KDSLEN_AIVIVARIANTCODING_SPREADSPECTRUMCLOCKGENERATIONLVDS

	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "    EC_2DISP_CHANGED",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsChangedEarlyConfig_TwoDisplays()) ));


	dp_tclEarlyConfigTwoDisplaysTrTimingLVDS1 _oTimingLvds1;
	tChar _strTimingLvds1[sizeof(m_strTimingLvds1)];
	(tVoid)OSAL_pvMemorySet((tVoid*)_strTimingLvds1, OSAL_NULL, sizeof(_strTimingLvds1));
	(tVoid)_oTimingLvds1.s32GetData(_strTimingLvds1, sizeof(_strTimingLvds1)-1);
	ETG_TRACE_FATAL(("%26s: %30s=%32s ", strIdentifier, " EC_2DISP_TIME_LVDS1",
			_strTimingLvds1 ));

#ifndef VARIANT_S_FTR_ENABLE_AIVI_SCOPE1
	dp_tclEarlyConfigTwoDisplaysTrClockEdgeSelectLVDS1 _oClockEdgeSelectLVDS1;
	ETG_TRACE_FATAL(("%26s: %30s=%d ",   strIdentifier, "  EC_2DISP_CES_LVDS1",
			_oClockEdgeSelectLVDS1.tGetData() ));
#else
	ETG_TRACE_FATAL(("%26s: %30s=%20s ",   strIdentifier, "  EC_2DISP_CES_LVDS1",
			"unsupported" ));
#endif

#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
	dp_tclEarlyConfigTwoDisplaysTrSpreadSpectrumClockGenerationLVDS1 _oClockGenerationLVDS1;
	ETG_TRACE_FATAL(("%26s: %30s=%d ",   strIdentifier, " EC_2DISP_SSCG_LVDS1",
			_oClockGenerationLVDS1.tGetData() ));
#else
	ETG_TRACE_FATAL(("%26s: %30s=%20s ", strIdentifier, " EC_2DISP_SSCG_LVDS1",
			"unsupported" ));
#endif //#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER

	dp_tclEarlyConfigTwoDisplaysTrTimingLVDS2 _oTimingLvds2;
	tChar _strTimingLvds2[sizeof(m_strTimingLvds2)];
	(tVoid)OSAL_pvMemorySet((tVoid*)_strTimingLvds2, OSAL_NULL, sizeof(_strTimingLvds2));
	(tVoid)_oTimingLvds2.s32GetData(_strTimingLvds2, sizeof(_strTimingLvds2)-1);
	ETG_TRACE_FATAL(("%26s: %30s=%32s ", strIdentifier, " EC_2DISP_TIME_LVDS2",
			_strTimingLvds2 ));

#ifndef VARIANT_S_FTR_ENABLE_AIVI_SCOPE1
	dp_tclEarlyConfigTwoDisplaysTrClockEdgeSelectLVDS2 _oClockEdgeSelectLVDS2;
	ETG_TRACE_FATAL(("%26s: %30s=%d ",   strIdentifier, "  EC_2DISP_CES_LVDS2",
			_oClockEdgeSelectLVDS2.tGetData() ));
#else
	ETG_TRACE_FATAL(("%26s: %30s=%20s ",   strIdentifier, "  EC_2DISP_CES_LVDS2",
			"unsupported" ));
#endif

#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
	dp_tclEarlyConfigTwoDisplaysTrSpreadSpectrumClockGenerationLVDS2 _oClockGenerationLVDS2;
	ETG_TRACE_FATAL(("%26s: %30s=%d ",   strIdentifier, " EC_2DISP_SSCG_LVDS2",
			_oClockGenerationLVDS2.tGetData() ));
#else
	ETG_TRACE_FATAL(("%26s: %30s=%20s ", strIdentifier, " EC_2DISP_SSCG_LVDS2",
			"unsupported" ));
#endif //#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig_TimingLvds1()
*
* DESCRIPTION: write the EarlyConfig_TwoDisplays timing LVDS 1 directly without validation
*
* PARAMETER:   const tChar* pStrTimingLvds (max. 31 char)
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig_TimingLvds1(const tChar* pStrTimingLvds) const
{
   ETG_TRACE_USR4(("bWriteEarlyConfig_TimingLvds1(%32s) entered.", pStrTimingLvds));

   tBool _bChanged = FALSE;
   tChar _strTimingLvds_Manual[sizeof(m_strTimingLvds1)] = {0};
   dp_tclDimmingDPEarlyConfig_TimingLvds1_Manually _oTimingLvdsManuell;
   tS32 _s32ReadStatus = _oTimingLvdsManuell.s32GetData(_strTimingLvds_Manual, sizeof(_strTimingLvds_Manual)-1);

   // check for update of manual EarlyConfig timing LVDS 1
   if (DP_S32_NO_ERR > _oTimingLvdsManuell.s32GetStatus())
   {
	   // not initialized
	   _bChanged = TRUE;
	   ETG_TRACE_USR1(("bWriteEarlyConfig_TimingLvds1: Status Timing LVDS 1 manual: %d.", _oTimingLvdsManuell.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   // could be loaded
	   tS32 _s32Compare = OSAL_s32StringCompare(pStrTimingLvds, _strTimingLvds_Manual);
	   if (OSAL_NULL != _s32Compare)
	   {
		   _bChanged = TRUE;
		   ETG_TRACE_USR1(("bWriteEarlyConfig_TimingLvds1: Timing LVDS 1 manual changed (Compare: %d).", _s32Compare ));
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TimingLvds1: Timing LVDS 1 manual not changed (Compare: %d).", _s32Compare ));
	   }
   }
   // set EarlyConfig timing LVDS 1 manual
   if (TRUE == _bChanged)
   {
	   (tVoid)OSAL_pvMemorySet((tVoid*)_strTimingLvds_Manual, OSAL_NULL, sizeof(_strTimingLvds_Manual));
	   (tVoid)strncpy(_strTimingLvds_Manual, pStrTimingLvds, sizeof(_strTimingLvds_Manual)-1);

	   // write EarlyConfig data pool
	   dp_tclEarlyConfigTwoDisplaysTrTimingLVDS1 _oTimingLvds;
	   if (DP_S32_NO_ERR != _oTimingLvds.s32SetData(_strTimingLvds_Manual))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TimingLvds1: Failed to manually set EarlyConfig timing LVDS 1 %s!",
				   _strTimingLvds_Manual ));
		   return FALSE;
	   }
	   else
	   {
		   ETG_TRACE_USR1(("bWriteEarlyConfig_TimingLvds1: Successful manually set EarlyConfig timing LVDS 1 %s.",
				   _strTimingLvds_Manual ));
	   }
	   // write own touch driver name data pool
	   if (DP_S32_NO_ERR != _oTimingLvdsManuell.s32SetData(_strTimingLvds_Manual))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TimingLvds1: Failed to manually set own timing LVDS 1 %s!",
				   _strTimingLvds_Manual ));
	   }
	   else
	   {
		   ETG_TRACE_USR1(("bWriteEarlyConfig_TimingLvds1: Successful manually set own timing LVDS 1 %s.",
				   _strTimingLvds_Manual ));
		   return TRUE;
	   }
   }

   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig_ClockEdgeLvds1()
*
* DESCRIPTION: Writes the early config LVDS1 clock edge configuration
*              (possible values: -1, 0, 1). Please use an undefined value to
*              reset manually settings.
*
* PARAMETER:   const tS32 s32ClockEdge
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig_ClockEdgeLvds1(const tS32 s32ClockEdge) const
{
   ETG_TRACE_USR4(("bWriteEarlyConfig_ClockEdgeLvds1(%d) entered.", s32ClockEdge));

   tBool _bReturn = TRUE;
   tS32 _s32Return = DP_S32_NO_ERR;

   dp_tclDimmingDPEarlyConfig_ClockEdgeLvds1_Manually _oClockEdge_Manually;
   _s32Return = _oClockEdge_Manually.s32SetData(s32ClockEdge);
   if (DP_S32_NO_ERR != _s32Return)
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_ClockEdgeLvds1: Failed to set manual two display clock edge LVDS1 %d with return %d!",
			   s32ClockEdge, _s32Return ));
	   _bReturn = FALSE;
   }
   else
   {
	   ETG_TRACE_USR2(("bWriteEarlyConfig_ClockEdgeLvds1: Successful set manual two display clock edge LVDS1 %d with return %d.",
			   s32ClockEdge, _s32Return ));
   }

   dp_tclEarlyConfigTwoDisplaysTrClockEdgeSelectLVDS1 _oClockEdge;
   _s32Return = _oClockEdge.s32SetData(s32ClockEdge);
   if (DP_S32_NO_ERR != _s32Return)
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_ClockEdgeLvds1: Failed to set two display clock edge LVDS1 %d to early config with return %d!",
			   s32ClockEdge, _s32Return ));
	   _bReturn = FALSE;
   }
   else
   {
	   ETG_TRACE_USR2(("bWriteEarlyConfig_ClockEdgeLvds1: Successful set two display clock edge LVDS1 %d to early config with return %d.",
			   s32ClockEdge, _s32Return ));
   }

   return _bReturn;
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig_ClockGenerationLvds1()
*
* DESCRIPTION: Writes the early config LVDS1 spread spectrum clock generation
*              configuration
*              (possible values: -1, 0, 8 ... 15).
*              Please use an undefined value to reset manually settings.
*
* PARAMETER:   const tS32 s32ClockEdge
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig_ClockGenerationLvds1(const tS32 s32ClockGeneration) const
{
   ETG_TRACE_USR4(("bWriteEarlyConfig_ClockGenerationLvds1(%d) entered.", s32ClockGeneration));

   tBool _bReturn = TRUE;
   tS32 _s32Return = DP_S32_NO_ERR;

   dp_tclDimmingDPEarlyConfig_ClockGenerationLvds1_Manually _oClockGeneration_Manually;
   _s32Return = _oClockGeneration_Manually.s32SetData(s32ClockGeneration);
   if (DP_S32_NO_ERR != _s32Return)
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_ClockGenerationLvds1: Failed to set manual two display spread spectrum clock generation LVDS1 %d with return %d!",
			   s32ClockGeneration, _s32Return ));
	   _bReturn = FALSE;
   }
   else
   {
	   ETG_TRACE_USR2(("bWriteEarlyConfig_ClockGenerationLvds1: Successful set manual two display spread spectrum clock generation LVDS1 %d with return %d.",
			   s32ClockGeneration, _s32Return ));
   }
#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
   dp_tclEarlyConfigTwoDisplaysTrSpreadSpectrumClockGenerationLVDS1 _oClockGeneration;
   _s32Return = _oClockGeneration.s32SetData(s32ClockGeneration);
   if (DP_S32_NO_ERR != _s32Return)
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_ClockGenerationLvds1: Failed to set two display spread spectrum clock generation LVDS1 %d to early config with return %d!",
			   s32ClockGeneration, _s32Return ));
	   _bReturn = FALSE;
   }
   else
   {
	   ETG_TRACE_USR2(("bWriteEarlyConfig_ClockGenerationLvds1: Successful set two display spread spectrum clock generation LVDS1 %d to early config with return %d.",
			   s32ClockGeneration, _s32Return ));
   }
#else
   ETG_TRACE_FATAL(("bWriteEarlyConfig_ClockGenerationLvds1: Failed to set EarlyConfig two display spread spectrum clock generation LVDS1 %d! UNSUPPORTED DATAPOOL!",
		   s32ClockGeneration ));
   _bReturn = FALSE;
#endif //#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
   return _bReturn;
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig_TimingLvds2()
*
* DESCRIPTION: write the EarlyConfig_TwoDisplays timing LVDS 2 directly without validation
*
* PARAMETER:   const tChar* pStrTimingLvds (max. 31 char)
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig_TimingLvds2(const tChar* pStrTimingLvds) const
{
   ETG_TRACE_USR4(("bWriteEarlyConfig_TimingLvds2(%32s) entered.", pStrTimingLvds));

   tBool _bChanged = FALSE;
   tChar _strTimingLvds_Manual[sizeof(m_strTimingLvds1)] = {0};
   dp_tclDimmingDPEarlyConfig_TimingLvds2_Manually _oTimingLvdsManuell;
   tS32 _s32ReadStatus = _oTimingLvdsManuell.s32GetData(_strTimingLvds_Manual, sizeof(_strTimingLvds_Manual)-1);

   // check for update of manual EarlyConfig timing LVDS 2
   if (DP_S32_NO_ERR > _oTimingLvdsManuell.s32GetStatus())
   {
	   // not initialized
	   _bChanged = TRUE;
	   ETG_TRACE_USR1(("bWriteEarlyConfig_TimingLvds2: Status Timing LVDS 2 manual: %d.", _oTimingLvdsManuell.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   // could be loaded
	   tS32 _s32Compare = OSAL_s32StringCompare(pStrTimingLvds, _strTimingLvds_Manual);
	   if (OSAL_NULL != _s32Compare)
	   {
		   _bChanged = TRUE;
		   ETG_TRACE_USR1(("bWriteEarlyConfig_TimingLvds2: Timing LVDS 1 manual changed (Compare: %d).", _s32Compare ));
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TimingLvds2: Timing LVDS 1 manual not changed (Compare: %d).", _s32Compare ));
	   }
   }
   // set EarlyConfig timing LVDS 2 manual
   if (TRUE == _bChanged)
   {
	   (tVoid)OSAL_pvMemorySet((tVoid*)_strTimingLvds_Manual, OSAL_NULL, sizeof(_strTimingLvds_Manual));
	   (tVoid)strncpy(_strTimingLvds_Manual, pStrTimingLvds, sizeof(_strTimingLvds_Manual)-1);

	   // write EarlyConfig data pool
	   dp_tclEarlyConfigTwoDisplaysTrTimingLVDS2 _oTimingLvds;
	   if (DP_S32_NO_ERR != _oTimingLvds.s32SetData(_strTimingLvds_Manual))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TimingLvds2: Failed to manually set EarlyConfig timing LVDS 2 %s!",
				   _strTimingLvds_Manual ));
		   return FALSE;
	   }
	   else
	   {
		   ETG_TRACE_USR1(("bWriteEarlyConfig_TimingLvds2: Successful manually set EarlyConfig timing LVDS 2 %s.",
				   _strTimingLvds_Manual ));
	   }
	   // write own touch driver name data pool
	   if (DP_S32_NO_ERR != _oTimingLvdsManuell.s32SetData(_strTimingLvds_Manual))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_TimingLvds2: Failed to manually set own timing LVDS 2 %s!",
				   _strTimingLvds_Manual ));
	   }
	   else
	   {
		   ETG_TRACE_USR1(("bWriteEarlyConfig_TimingLvds2: Successful manually set own timing LVDS 2 %s.",
				   _strTimingLvds_Manual ));
		   return TRUE;
	   }
   }

   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig_ClockEdgeLvds2()
*
* DESCRIPTION: Writes the early config LVDS2 clock edge configuration
*              (possible values: -1, 0, 1). Please use an undefined value to
*              reset manually settings.
*
* PARAMETER:   const tS32 s32ClockEdge
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig_ClockEdgeLvds2(const tS32 s32ClockEdge) const
{
   ETG_TRACE_USR4(("bWriteEarlyConfig_ClockEdgeLvds2(%d) entered.", s32ClockEdge));

   tBool _bReturn = TRUE;
   tS32 _s32Return = DP_S32_NO_ERR;

   dp_tclDimmingDPEarlyConfig_ClockEdgeLvds2_Manually _oClockEdge_Manually;
   _s32Return = _oClockEdge_Manually.s32SetData(s32ClockEdge);
   if (DP_S32_NO_ERR != _s32Return)
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_ClockEdgeLvds2: Failed to set manual two display clock edge LVDS2 %d with return %d!",
			   s32ClockEdge, _s32Return ));
	   _bReturn = FALSE;
   }
   else
   {
	   ETG_TRACE_USR2(("bWriteEarlyConfig_ClockEdgeLvds2: Successful set manual two display clock edge LVDS2 %d with return %d.",
			   s32ClockEdge, _s32Return ));
   }

   dp_tclEarlyConfigTwoDisplaysTrClockEdgeSelectLVDS2 _oClockEdge;
   _s32Return = _oClockEdge.s32SetData(s32ClockEdge);
   if (DP_S32_NO_ERR != _s32Return)
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_ClockEdgeLvds2: Failed to set two display clock edge LVDS2 %d to early config with return %d!",
			   s32ClockEdge, _s32Return ));
	   _bReturn = FALSE;
   }
   else
   {
	   ETG_TRACE_USR2(("bWriteEarlyConfig_ClockEdgeLvds2: Successful set two display clock edge LVDS2 %d to early config with return %d.",
			   s32ClockEdge, _s32Return ));
   }

   return _bReturn;
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig_ClockGenerationLvds2()
*
* DESCRIPTION: Writes the early config LVDS2 spread spectrum clock generation
*              configuration
*              (possible values: -1, 0, 8 ... 15).
*              Please use an undefined value to reset manually settings.
*
* PARAMETER:   const tS32 s32ClockEdge
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig_ClockGenerationLvds2(const tS32 s32ClockGeneration) const
{
   ETG_TRACE_USR4(("bWriteEarlyConfig_ClockGenerationLvds2(%d) entered.", s32ClockGeneration));

   tBool _bReturn = TRUE;
   tS32 _s32Return = DP_S32_NO_ERR;

   dp_tclDimmingDPEarlyConfig_ClockGenerationLvds2_Manually _oClockGeneration_Manually;
   _s32Return = _oClockGeneration_Manually.s32SetData(s32ClockGeneration);
   if (DP_S32_NO_ERR != _s32Return)
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_ClockGenerationLvds2: Failed to set manual two display spread spectrum clock generation LVDS2 %d with return %d!",
			   s32ClockGeneration, _s32Return ));
	   _bReturn = FALSE;
   }
   else
   {
	   ETG_TRACE_USR2(("bWriteEarlyConfig_ClockGenerationLvds2: Successful set manual two display spread spectrum clock generation LVDS2 %d with return %d.",
			   s32ClockGeneration, _s32Return ));
   }

#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
   dp_tclEarlyConfigTwoDisplaysTrSpreadSpectrumClockGenerationLVDS2 _oClockGeneration;
   _s32Return = _oClockGeneration.s32SetData(s32ClockGeneration);
   if (DP_S32_NO_ERR != _s32Return)
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_ClockGenerationLvds2: Failed to set two display pread spectrum clock generation LVDS2 %d to early config with return %d!",
			   s32ClockGeneration, _s32Return ));
	   _bReturn = FALSE;
   }
   else
   {
	   ETG_TRACE_USR2(("bWriteEarlyConfig_ClockGenerationLvds2: Successful set two display pread spectrum clock generation LVDS2 %d to early config with return %d.",
			   s32ClockGeneration, _s32Return ));
   }
#else
   ETG_TRACE_FATAL(("bWriteEarlyConfig_ClockGenerationLvds2: Failed to set EarlyConfig two display spread spectrum clock generation LVDS2 %d! UNSUPPORTED DATAPOOL!",
		   s32ClockGeneration ));
   _bReturn = FALSE;
#endif //#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER

   return _bReturn;
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig_DisplayOrder()
*
* DESCRIPTION: write the EarlyConfig_LayerManager display order
*              directly without validation
*
* PARAMETER:   const tU8* pu8DisplayOrder
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig_DisplayOrder(const tU8* pu8DisplayOrder) const
{
   ETG_TRACE_USR4(("bWriteEarlyConfig_DisplayOrder(0x%08x) entered.", pu8DisplayOrder));
   tBool _bChanged = FALSE;
   if (OSAL_NULL == pu8DisplayOrder)
	   return _bChanged;
   tU8 _au8DisplayOrderManual[EARLYCONFIG_ARRAY_ELEMENTS(m_au8DisplayOrder)] = {0};
   dp_tclDimmingDPEarlyConfig_DisplayOrder_Manually _oDisplayOrderManual;
   tS32 _s32ReadStatus = _oDisplayOrderManual.s32GetData(_au8DisplayOrderManual, EARLYCONFIG_ARRAY_ELEMENTS(_au8DisplayOrderManual));

   // check for update of manual EarlyConfig display
   if (DP_S32_NO_ERR > _oDisplayOrderManual.s32GetStatus())
   {
	   // not initialized
	   _bChanged = TRUE;
	   ETG_TRACE_USR1(("bWriteEarlyConfig_DisplayOrder: Status Display Order manual: %d.", _oDisplayOrderManual.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   // could be loaded
	   tS32 _s32Compare = OSAL_s32MemoryCompare(pu8DisplayOrder, _au8DisplayOrderManual, sizeof(_au8DisplayOrderManual));
	   if (OSAL_NULL != _s32Compare)
	   {
		   _bChanged = TRUE;
		   ETG_TRACE_USR1(("bWriteEarlyConfig_DisplayOrder: Display Order manual changed (Compare: %d).", _s32Compare ));
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_DisplayOrder: Display Order manual not changed (Compare: %d).", _s32Compare ));
	   }
   }
   // set EarlyConfig display order manual
   if (TRUE == _bChanged)
   {
	   (tVoid)OSAL_pvMemorySet((tVoid*)_au8DisplayOrderManual, OSAL_NULL, sizeof(_au8DisplayOrderManual));
	   (tVoid)OSAL_pvMemoryCopy(_au8DisplayOrderManual, pu8DisplayOrder, sizeof(_au8DisplayOrderManual));

	   // write EarlyConfig data pool
	   dp_tclEarlyConfigLayerManagerTrDisplayOrder _oDisplayOrder;
	   if (DP_S32_NO_ERR != _oDisplayOrder.s32SetData(_au8DisplayOrderManual, EARLYCONFIG_ARRAY_ELEMENTS(_au8DisplayOrderManual)))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_DisplayOrder: Failed to manually set EarlyConfig Display Order %u %u %u %u!",
				   _au8DisplayOrderManual[0], _au8DisplayOrderManual[1], _au8DisplayOrderManual[2], _au8DisplayOrderManual[3] ));
		   return FALSE;
	   }
	   else
	   {
		   ETG_TRACE_USR1(("bWriteEarlyConfig_DisplayOrder: Successful manually set EarlyConfig Display Order %u %u %u %u.",
				   _au8DisplayOrderManual[0], _au8DisplayOrderManual[1], _au8DisplayOrderManual[2], _au8DisplayOrderManual[3] ));
	   }
	   // write own display order data pool
	   if (DP_S32_NO_ERR != _oDisplayOrderManual.s32SetData(_au8DisplayOrderManual, EARLYCONFIG_ARRAY_ELEMENTS(_au8DisplayOrderManual)))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_DisplayOrder: Failed to manually set own Display Order %u %u %u %u!",
				   _au8DisplayOrderManual[0], _au8DisplayOrderManual[1], _au8DisplayOrderManual[2], _au8DisplayOrderManual[3] ));
	   }
	   else
	   {
		   ETG_TRACE_USR1(("bWriteEarlyConfig_DisplayOrder: Successful manually set own Display Order %u %u %u %u.",
				   _au8DisplayOrderManual[0], _au8DisplayOrderManual[1], _au8DisplayOrderManual[2], _au8DisplayOrderManual[3] ));
		   return TRUE;
	   }
   }

   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig_FlushdesignConfigFile()
*
* DESCRIPTION: write the EarlyConfig_LayerManager flush design config file
*              directly without validation
*
* PARAMETER:   const tChar* pStrFlushDesignConfigFile (max. 63 char)
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig_FlushDesignConfigFile(const tChar* pStrFlushDesignConfigFile) const
{
   ETG_TRACE_USR4(("bWriteEarlyConfig_FlushDesignConfigFile(%64s) entered.", pStrFlushDesignConfigFile));

   tBool _bChanged = FALSE;
   if (OSAL_NULL == pStrFlushDesignConfigFile)
	   return _bChanged;
   tChar _strFlushDesignConfigFile_Manual[sizeof(m_strFlushDesignConfigFile)] = {0};
   dp_tclDimmingDPEarlyConfig_FlushDesignConfigFile_Manually _oFlushDesignConfigFileManual;
   tS32 _s32ReadStatus = _oFlushDesignConfigFileManual.s32GetData(_strFlushDesignConfigFile_Manual, sizeof(_strFlushDesignConfigFile_Manual)-1);

   // check for update of manual EarlyConfig flush design config file
   if (DP_S32_NO_ERR > _oFlushDesignConfigFileManual.s32GetStatus())
   {
	   // not initialized
	   _bChanged = TRUE;
	   ETG_TRACE_USR1(("bWriteEarlyConfig_FlushDesignConfigFile: Status Flush Design Config File manual: %d.", _oFlushDesignConfigFileManual.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   // could be loaded
	   tS32 _s32Compare = OSAL_s32StringCompare(pStrFlushDesignConfigFile, _strFlushDesignConfigFile_Manual);
	   if (OSAL_NULL != _s32Compare)
	   {
		   _bChanged = TRUE;
		   ETG_TRACE_USR1(("bWriteEarlyConfig_FlushDesignConfigFile: Flush Design Config File manual changed (Compare: %d).", _s32Compare ));
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_FlushDesignConfigFile: Flush Design Config File manual not changed (Compare: %d).", _s32Compare ));
	   }
   }
   // set EarlyConfig flush design config file manual
   if (TRUE == _bChanged)
   {
	   (tVoid)OSAL_pvMemorySet((tVoid*)_strFlushDesignConfigFile_Manual, OSAL_NULL, sizeof(_strFlushDesignConfigFile_Manual));
	   (tVoid)strncpy(_strFlushDesignConfigFile_Manual, pStrFlushDesignConfigFile, sizeof(_strFlushDesignConfigFile_Manual)-1);

	   // write EarlyConfig data pool
	   dp_tclEarlyConfigLayerManagerTrFlushDesignConfigFile _oFlushDesignConfigFile;
	   if (DP_S32_NO_ERR != _oFlushDesignConfigFile.s32SetData(_strFlushDesignConfigFile_Manual))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_FlushDesignConfigFile: Failed to manually set EarlyConfig Flush Design Config File %s!",
				   _strFlushDesignConfigFile_Manual ));
		   return FALSE;
	   }
	   else
	   {
		   ETG_TRACE_USR1(("bWriteEarlyConfig_FlushDesignConfigFile: Successful manually set EarlyConfig Flush Design Config File %s.",
				   _strFlushDesignConfigFile_Manual ));
	   }
	   // write own flush design config file data pool
	   if (DP_S32_NO_ERR != _oFlushDesignConfigFileManual.s32SetData(_strFlushDesignConfigFile_Manual))
	   {
		   ETG_TRACE_FATAL(("bWriteEarlyConfig_FlushDesignConfigFile: Failed to manually set own Flush Design Config File %s!",
				   _strFlushDesignConfigFile_Manual ));
	   }
	   else
	   {
		   ETG_TRACE_USR1(("bWriteEarlyConfig_FlushDesignConfigFile: Successful manually set own Flush Design Config File %s.",
				   _strFlushDesignConfigFile_Manual ));
		   return TRUE;
	   }
   }

   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bLoadKDSearlyConfig_LayerManager()
*
* DESCRIPTION: loading KDS EarlyConfig_LayerManager data
*              see AIVI-21754: Multiple/Virtual screen support
*
* PARAMETER:   none
*
* RETURNVALUE: tBool
*
*******************************************************************************/
// possible flush design config file names
const char* m_astrEarlyConfigFlusDesignConfigFileNames[] =
{
		// [0x00] Standard/Default
		"/etc/layermanager/device-configuration.conf",
		// [0x01] V2 Flush
		"/etc/layermanager/aivi-cmfb-v2.conf"
};
tBool dispvidctrl_tclEarlyConfig::bLoadKDSearlyConfig_LayerManager(tVoid)
{
	ETG_TRACE_USR4(("bLoadKDSearlyConfig_LayerManager() entered."));
	tBool _bReturn = TRUE;
	m_bKDSLayerManagerLoaded = FALSE;
	// display order (see AIVI-21754)
	tU8 _u8ScreenID = EARLYCONFIG_C_U8_SCREENID_HDMI;
	tU8 _u8VirtualScreenType = OSAL_C_U8_MAX;
	tU8 _u8LastDisplayOrder = EARLYCONFIG_ARRAY_ELEMENTS(m_au8DisplayOrder) - 1; // for unused Screen ID
	tU8 _u8CountOfUnused = 0;
	// init display orders
	(tVoid)OSAL_pvMemorySet((tVoid*)m_au8DisplayOrder, -1, sizeof(m_au8DisplayOrder));
	// load and set display order for HDMI
	tS32 _s32Status = DP_s32GetConfigItem("AIVIVariantCoding", "HDMI", &_u8VirtualScreenType, sizeof(_u8VirtualScreenType), FALSE);
	if (DP_S32_NO_ERR > _s32Status)
	{
		ETG_TRACE_FATAL(("bLoadKDSearlyConfig_LayerManager: virtual screen type for ScreenID %u not configured!",
				ETG_ENUM(EC_SCREENID, _u8ScreenID) ));
		_bReturn = FALSE;
	}
	else
	{
		if (EARLYCONFIG_C_U8_SCREENTYPE_UNDEF == _u8VirtualScreenType)
		{
			++_u8CountOfUnused;
		}
		if (EARLYCONFIG_ARRAY_ELEMENTS(m_au8DisplayOrder) <= u8KDSearlyConfig_LayerManager_SetDisplayOrder(_u8ScreenID, _u8VirtualScreenType, _u8LastDisplayOrder))
		{
			ETG_TRACE_FATAL(("bLoadKDSearlyConfig_LayerManager: display order for ScreenID %u could not set!",
					ETG_ENUM(EC_SCREENID, _u8ScreenID) ));
		}
	}
	// load and set display order for LVDS1
	_u8ScreenID = EARLYCONFIG_C_U8_SCREENID_LVDS1;
	_s32Status = DP_s32GetConfigItem("AIVIVariantCoding", "LVDS1", &_u8VirtualScreenType, sizeof(_u8VirtualScreenType), FALSE);
	if (DP_S32_NO_ERR > _s32Status)
	{
		ETG_TRACE_FATAL(("bLoadKDSearlyConfig_LayerManager: virtual screen type for ScreenID %u not configured!",
				ETG_ENUM(EC_SCREENID, _u8ScreenID) ));
		_bReturn = FALSE;
	}
	else
	{
		if (EARLYCONFIG_C_U8_SCREENTYPE_UNDEF == _u8VirtualScreenType)
		{
			++_u8CountOfUnused;
		}
		if (EARLYCONFIG_ARRAY_ELEMENTS(m_au8DisplayOrder) <= u8KDSearlyConfig_LayerManager_SetDisplayOrder(_u8ScreenID, _u8VirtualScreenType, _u8LastDisplayOrder))
		{
			ETG_TRACE_FATAL(("bLoadKDSearlyConfig_LayerManager: display order for ScreenID %u could not set!",
					ETG_ENUM(EC_SCREENID, _u8ScreenID) ));
		}
	}
	// load and set display order for LVDS2
	_u8ScreenID = EARLYCONFIG_C_U8_SCREENID_LVDS2;
	_s32Status = DP_s32GetConfigItem("AIVIVariantCoding", "LVDS2", &_u8VirtualScreenType, sizeof(_u8VirtualScreenType), FALSE);
	if (DP_S32_NO_ERR > _s32Status)
	{
		ETG_TRACE_FATAL(("bLoadKDSearlyConfig_LayerManager: virtual screen type for ScreenID %u not configured!",
				ETG_ENUM(EC_SCREENID, _u8ScreenID) ));
		_bReturn = FALSE;
	}
	else
	{
		if (EARLYCONFIG_C_U8_SCREENTYPE_UNDEF == _u8VirtualScreenType)
		{
			++_u8CountOfUnused;
		}
		if (EARLYCONFIG_ARRAY_ELEMENTS(m_au8DisplayOrder) <= u8KDSearlyConfig_LayerManager_SetDisplayOrder(_u8ScreenID, _u8VirtualScreenType, _u8LastDisplayOrder))
		{
			ETG_TRACE_FATAL(("bLoadKDSearlyConfig_LayerManager: display order for ScreenID %u could not set!",
					ETG_ENUM(EC_SCREENID, _u8ScreenID) ));
		}
	}
	// load and set display order for VGA
	_u8ScreenID = EARLYCONFIG_C_U8_SCREENID_VGA;
	_s32Status = DP_s32GetConfigItem("AIVIVariantCoding", "VGA", &_u8VirtualScreenType, sizeof(_u8VirtualScreenType), FALSE);
	if (DP_S32_NO_ERR > _s32Status)
	{
		ETG_TRACE_FATAL(("bLoadKDSearlyConfig_LayerManager: virtual screen type for ScreenID %u not configured!",
				ETG_ENUM(EC_SCREENID, _u8ScreenID) ));
		_bReturn = FALSE;
	}
	else
	{
		if (EARLYCONFIG_C_U8_SCREENTYPE_UNDEF == _u8VirtualScreenType)
		{
			++_u8CountOfUnused;
		}
		if ((EARLYCONFIG_ARRAY_ELEMENTS(m_au8DisplayOrder) > _u8CountOfUnused)
			&& (EARLYCONFIG_ARRAY_ELEMENTS(m_au8DisplayOrder) <= u8KDSearlyConfig_LayerManager_SetDisplayOrder(_u8ScreenID, _u8VirtualScreenType, _u8LastDisplayOrder)))
		{
			ETG_TRACE_FATAL(("bLoadKDSearlyConfig_LayerManager: display order for ScreenID %u could not set!",
					ETG_ENUM(EC_SCREENID, _u8ScreenID) ));
		}
	}
	// virtual keys / flush design (see AIVI-21754)
	tU8 _u8FlushDesign = 0;
	_s32Status = DP_s32GetConfigItem("AIVIVariantCoding", "SoftkeysConfiguration", &_u8FlushDesign, sizeof(_u8FlushDesign), FALSE);
	if (DP_S32_NO_ERR > _s32Status)
	{
		m_u8FlushDesignConfigFile = OSAL_C_U8_MAX;
		ETG_TRACE_FATAL(("bLoadKDSearlyConfig_LayerManager: FlushDesign not configured!"));
		_bReturn = FALSE;
	}
	else
	{
		m_u8FlushDesignConfigFile = _u8FlushDesign;
		(tVoid)OSAL_pvMemorySet((tVoid*)m_strFlushDesignConfigFile, OSAL_NULL, sizeof(m_strFlushDesignConfigFile));
		if (EARLYCONFIG_ARRAY_ELEMENTS(m_astrEarlyConfigFlusDesignConfigFileNames) <= _u8FlushDesign)
		{
			ETG_TRACE_FATAL(("bLoadKDSearlyConfig_LayerManager: invalid FlushDesign %u configured!",
					_u8FlushDesign ));
			_bReturn = FALSE;
		}
		else
		{
			(tVoid)OSAL_szStringNCopy(m_strFlushDesignConfigFile, m_astrEarlyConfigFlusDesignConfigFileNames[_u8FlushDesign], sizeof(m_strFlushDesignConfigFile)-1);
		}
	}

	if (TRUE == _bReturn)
	{
        ETG_TRACE_USR2(("bLoadKDSearlyConfig_LayerManager: KDS early config layer manager loaded with display order=%d %d %d %d and flush design config file=%d (%64s)",
        		m_au8DisplayOrder[0], m_au8DisplayOrder[1], m_au8DisplayOrder[2], m_au8DisplayOrder[3], m_u8FlushDesignConfigFile, m_strFlushDesignConfigFile));
		m_bKDSLayerManagerLoaded = TRUE;
	}
	else
	{
        ETG_TRACE_ERR(("bLoadKDSearlyConfig_LayerManager: KDS early config layer manager could not loaded!"));
	}

	return bIsLoadedKDSEarlyConfig_LayerManager();
}

/*******************************************************************************
*
* FUNCTION:    u8KDSearlyConfig_LayerManager_SetDisplayOrder()
*
* DESCRIPTION: set the display order of specified screen depending of
*              virtual screen type
*
*              The attribute m_au8DisplayOrder should be initialized with 0xFF
*              before first call!
*
* PARAMETER:   tU8 u8ScreenID
*              tU8 u8VirtualScreenType
*              tU8 &u8LastDisplayOrderIndex
*
* RETURNVALUE: tU8		display order index
*
*******************************************************************************/
// assigned display order index depending of virtual screen types
// see EARLYCONFIG_C_U8_SCREENTYPE_*
const tU8 m_au8EarlyConfigDisplayOrderIndex[] =
{
		// [0x00] Not used (not configured)
		OSAL_C_U8_MAX,
		// [0x01] Screen_0: Main HMI
		0,
		// [0x02] Screen_1: RSE
		1,
		// [0x03] Screen_2: Second screen
		2,
		// [0x04] Screen_3: last screen
		3
};
tU8 dispvidctrl_tclEarlyConfig::u8KDSearlyConfig_LayerManager_SetDisplayOrder(
		tU8 u8ScreenID, tU8 u8VirtualScreenType, tU8 &u8LastDisplayOrderIndex)
{
	ETG_TRACE_USR3(("u8KDSearlyConfig_LayerManager_SetDisplayOrder(ScreenID: %d, VirtualScreenType: %02x, u8LastDisplayOrderIndex: %d) entered.",
			u8ScreenID, u8VirtualScreenType, u8LastDisplayOrderIndex));
	tU8 _u8DisplayOrderIdx = OSAL_C_U8_MAX;

	if (EARLYCONFIG_ARRAY_ELEMENTS(m_au8EarlyConfigDisplayOrderIndex) <= u8VirtualScreenType)
	{
		// Unknown virtual screen type
		ETG_TRACE_FATAL(("bLoadKDSearlyConfig_LayerManager: invalid virtual screen type %u for ScreenID %d configured!",
				ETG_ENUM(EC_SCREENTYPE, u8VirtualScreenType), u8ScreenID ));
	}
	else if (EARLYCONFIG_C_U8_SCREENTYPE_UNDEF == u8VirtualScreenType)
	{
		// unused virtual screen type
		if (EARLYCONFIG_ARRAY_ELEMENTS(m_au8DisplayOrder) > u8LastDisplayOrderIndex)
		{
			// set as last
			m_au8DisplayOrder[u8LastDisplayOrderIndex] = u8ScreenID;
			_u8DisplayOrderIdx = u8LastDisplayOrderIndex;
			--u8LastDisplayOrderIndex;
		}
	}
	else if (EARLYCONFIG_ARRAY_ELEMENTS(m_au8DisplayOrder) > m_au8EarlyConfigDisplayOrderIndex[u8VirtualScreenType])
	{
		_u8DisplayOrderIdx = m_au8EarlyConfigDisplayOrderIndex[u8VirtualScreenType];
		if (EARLYCONFIG_ARRAY_ELEMENTS(m_au8DisplayOrder) <= m_au8DisplayOrder[_u8DisplayOrderIdx])
		{
			// display order is not set before
			m_au8DisplayOrder[_u8DisplayOrderIdx] = u8ScreenID;
		}
		else
		{
			// do not set display order more than one time
			ETG_TRACE_FATAL(("bLoadKDSearlyConfig_LayerManager: display order for index %u of ScreenID %u was set before with ScreenID %u! New display order index %u",
					ETG_ENUM(EC_SCREENTYPE, _u8DisplayOrderIdx), ETG_ENUM(EC_SCREENID, u8ScreenID), ETG_ENUM(EC_SCREENID, m_au8DisplayOrder[_u8DisplayOrderIdx]), u8LastDisplayOrderIndex));
			_u8DisplayOrderIdx = OSAL_C_U8_MAX;
			// NOTE: automatically assignment after loading
		}
	}
	else
	{
		// coding error in m_au8EarlyConfigDisplayOrderIDs
		ETG_TRACE_FATAL(("bLoadKDSearlyConfig_LayerManager: invalid display order index %u for virtual screen type %u defined!",
				m_au8EarlyConfigDisplayOrderIndex[u8VirtualScreenType], ETG_ENUM(EC_SCREENTYPE, u8VirtualScreenType) ));
	}

	return _u8DisplayOrderIdx;
}

/*******************************************************************************
*
* FUNCTION:    bIsValidKDSEarlyConfig_LayerManager()
*
* DESCRIPTION: returns TRUE if loaded KDS data are valid
*              see AIVI-21754: Multiple/Virtual screen support
*
* PARAMETER:   none
*
* RETURNVALUE: tBool
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsValidKDSEarlyConfig_LayerManager(tVoid) const
{
	ETG_TRACE_USR4(("bIsValidKDSEarlyConfig_LayerManager() entered."));
	tBool _bReturn = TRUE;
	if (TRUE == bIsLoadedKDSEarlyConfig_LayerManager())
	{
		// DisplayOrder ... display order for HMI Main screen should be set, all display orders should be set
		tU8 _au8ScreenID[EARLYCONFIG_ARRAY_ELEMENTS(m_au8DisplayOrder)] = {0};
		(tVoid)OSAL_pvMemorySet((tVoid*)_au8ScreenID, -1, sizeof(_au8ScreenID));
		for (tU32 _u32Idx = 0; EARLYCONFIG_ARRAY_ELEMENTS(m_au8DisplayOrder) > _u32Idx; ++_u32Idx)
		{
			if (EARLYCONFIG_ARRAY_ELEMENTS(m_au8DisplayOrder) <= m_au8DisplayOrder[_u32Idx])
			{
				ETG_TRACE_FATAL(("bIsValidKDSEarlyConfig_LayerManager: Screen ID %u are invalid for Display Order Index %u!",
						ETG_ENUM(EC_SCREENID, m_au8DisplayOrder[_u32Idx]), _u32Idx ));
				_bReturn = FALSE;
			}
			else if (EARLYCONFIG_ARRAY_ELEMENTS(m_au8DisplayOrder) > _au8ScreenID[_u32Idx])
			{
				ETG_TRACE_FATAL(("bIsValidKDSEarlyConfig_LayerManager: Screen ID %u are used for Display Order Index %u from Screen ID %u!",
						ETG_ENUM(EC_SCREENID, m_au8DisplayOrder[_u32Idx]), _u32Idx, ETG_ENUM(EC_SCREENID, _au8ScreenID[_u32Idx]) ));
				_bReturn = FALSE;
			}
			else
			{
				ETG_TRACE_USR3(("bIsValidKDSEarlyConfig_LayerManager: Screen ID %u are valid for Display Order Index %u.",
						ETG_ENUM(EC_SCREENID, m_au8DisplayOrder[_u32Idx]), _u32Idx ));
				_au8ScreenID[_u32Idx] = m_au8DisplayOrder[_u32Idx];
			}
		}
		// FlushDesignConfigFile
		if (EARLYCONFIG_ARRAY_ELEMENTS(m_astrEarlyConfigFlusDesignConfigFileNames) <= m_u8FlushDesignConfigFile)
		{
			ETG_TRACE_FATAL(("bIsValidKDSEarlyConfig_LayerManager: Unknown flush design config %u!",
					m_u8FlushDesignConfigFile ));
			_bReturn = FALSE;
		}
		else if (0 == m_u8FlushDesignConfigFile)
		{
			ETG_TRACE_USR3(("bIsValidKDSEarlyConfig_LayerManager: use default flush design config file %s.",
					m_strFlushDesignConfigFile ));
		}
		else if (OSAL_NULL < OSAL_u32StringLength(m_strFlushDesignConfigFile))
		{
			if (FALSE == bIsExistKDSEarlyConfig_FlushDesignConfigFile())
			{
				ETG_TRACE_FATAL(("bIsValidKDSEarlyConfig_LayerManager: flush design config file %s doesn't exist!",
						m_strFlushDesignConfigFile ));
				_bReturn = FALSE;
			}
			else
			{
				ETG_TRACE_USR1(("bIsValidKDSEarlyConfig_LayerManager: flush design config file %s exist.",
						m_strFlushDesignConfigFile ));
			}
		}
		else
		{
			ETG_TRACE_FATAL(("bIsValidKDSEarlyConfig_LayerManager: no flush design config file defined!" ));
			_bReturn = FALSE;
		}
		return _bReturn;
	}
	return bIsLoadedKDSEarlyConfig_LayerManager();
}

/*******************************************************************************
*
* FUNCTION:    bIsExistKDSEarlyConfig_FlushDesignConfigFile()
*
* DESCRIPTION: returns TRUE if FlushDesignConfigFile of EarlyConfig_LayerManager
*              exist in folder /etc/layermanager
*              see AIVI-21754: Multiple/Virtual screen support
*
* PARAMETER:   none
*
* RETURNVALUE: tBool
*
*******************************************************************************/
const char m_strFlushDesignFolder[] = "/etc/layermanager";
tBool dispvidctrl_tclEarlyConfig::bIsExistKDSEarlyConfig_FlushDesignConfigFile(tVoid) const
{
	ETG_TRACE_USR4(("bIsExistKDSEarlyConfig_FlushDesignConfigFile() entered."));

	tBool _bExist = FALSE;
	char _strFilePath[sizeof(m_strFlushDesignFolder) + sizeof(m_strFlushDesignConfigFile) + 2];

	// prepare touch config file name
	(tVoid)memset((tVoid*)_strFilePath, OSAL_NULL, sizeof(_strFilePath));
	// ... set layermanager config folder
	//strcat(_strFilePath, m_strFlushDesignFolder);
	//strcat(_strFilePath, "/");
	// ... add file name
	strcat(_strFilePath, m_strFlushDesignConfigFile);
	// open the file
#ifdef DISPVIDCTRL_USE_OSAL
	OSAL_tIODescriptor _hFile = OSAL_IOOpen(_strFilePath,OSAL_EN_READONLY);
	if (OSAL_C_INVALID_HANDLE != _hFile)
#else
	int _hFile = open(_strFilePath, O_RDONLY);
	if (-1 != _hFile)
#endif // #ifdef DISPVIDCTRL_USE_OSAL
	{
	   _bExist = TRUE;
#ifdef DISPVIDCTRL_USE_OSAL
	   (tVoid)OSAL_s32IOClose(_hFile);
#else
	   close(_hFile);
#endif // #ifdef DISPVIDCTRL_USE_OSAL
	   ETG_TRACE_USR3(("bIsExistKDSEarlyConfig_FlushDesignConfigFile: existing flush design config file %s ",
			   _strFilePath ));
	}
	else
	{
	   ETG_TRACE_FATAL(("bIsExistKDSEarlyConfig_FlushDesignConfigFile: invalid flush design config file %s ",
			   _strFilePath ));
	}

	return _bExist;
}

/*******************************************************************************
*
* FUNCTION:    bIsEarlyConfig_DisplayOrder_ManuallySet()
*
* DESCRIPTION: returns TRUE if DisplayOrder of EarlyConfig_LayerManager set manually
*
* PARAMETER:   none
*
* RETURNVALUE: tBool
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsEarlyConfig_DisplayOrder_ManuallySet(tVoid) const
{
	ETG_TRACE_USR4(("bIsEarlyConfig_DisplayOrder_ManuallySet() entered."));
	dp_tclDimmingDPEarlyConfig_DisplayOrder_Manually _oDpElement;
	tU8 _au8DisplayOrder[DISPVIDCTRL_C_U16_ARRAYELEMENTS(m_au8DisplayOrder)];
	(tVoid)OSAL_pvMemorySet((tVoid*)_au8DisplayOrder, -1, sizeof(_au8DisplayOrder));
	tS32 _s32ReadStatus = _oDpElement.s32GetData(_au8DisplayOrder, DISPVIDCTRL_C_U16_ARRAYELEMENTS(_au8DisplayOrder));
	if (DP_S32_NO_ERR > _oDpElement.s32GetStatus())
	{
		ETG_TRACE_USR3(("bIsEarlyConfig_DisplayOrder_ManuallySet: DisplayOrder_Manually: %d.", _oDpElement.s32GetStatus() ));
	}
	else if (DP_S32_NO_ERR <= _s32ReadStatus)
	{
		for (tU16 _u16Idx = 0; DISPVIDCTRL_C_U16_ARRAYELEMENTS(_au8DisplayOrder) > _u16Idx; ++_u16Idx)
		{
			if (EARLYCONFIG_ARRAY_ELEMENTS(m_au8DisplayOrder) <= _au8DisplayOrder[_u16Idx])
				return FALSE;
		}
		return TRUE;
	}
	return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bIsEarlyConfig_FlushDesignConfigFile_ManuallySet()
*
* DESCRIPTION: returns TRUE if Flush Design Config File of EarlyConfig_LayerManager set manually
*
* PARAMETER:   none
*
* RETURNVALUE: tBool
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsEarlyConfig_FlushDesignConfigFile_ManuallySet(tVoid) const
{
	ETG_TRACE_USR4(("bIsEarlyConfig_FlushDesignConfigFile_ManuallySet() entered."));
	dp_tclDimmingDPEarlyConfig_FlushDesignConfigFile_Manually _oDpElement;
	tChar _strFlushDesignConfigFile[DISPVIDCTRL_C_U16_ARRAYELEMENTS(m_strFlushDesignConfigFile)];
	(tVoid)OSAL_pvMemorySet((tVoid*)_strFlushDesignConfigFile, -1, sizeof(_strFlushDesignConfigFile));
	tS32 _s32ReadStatus = _oDpElement.s32GetData(_strFlushDesignConfigFile, DISPVIDCTRL_C_U16_ARRAYELEMENTS(_strFlushDesignConfigFile));
	if (DP_S32_NO_ERR > _oDpElement.s32GetStatus())
	{
		ETG_TRACE_USR3(("bIsEarlyConfig_DisplayOrder_ManuallySet: FlushDesignConfigFile_Manually: %d.", _oDpElement.s32GetStatus() ));
	}
	else if (DP_S32_NO_ERR <= _s32ReadStatus)
	{
		if (1 < OSAL_u32StringLength(_strFlushDesignConfigFile))
			return TRUE;
	}
	return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bIsChangedEarlyConfig_LayerManager()
*
* DESCRIPTION: returns TRUE if KDS configuration data not equal than EarlyConfig
*              LayerManager data
*
* PARAMETER:   none
*
* RETURNVALUE: tBool
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsChangedEarlyConfig_LayerManager(tVoid) const
{
	ETG_TRACE_USR4(("bIsChangedEarlyConfig_LayerManager() entered."));
	tBool _bReturn = FALSE;
	if (TRUE == bIsLoadedKDSEarlyConfig_LayerManager())
	{
		// DisplayOrder
		dp_tclEarlyConfigLayerManagerTrDisplayOrder _oDpDisplayOrder;
		tU8 _au8DisplayOrder[DISPVIDCTRL_C_U16_ARRAYELEMENTS(m_au8DisplayOrder)];
		(tVoid)OSAL_pvMemorySet((tVoid*)_au8DisplayOrder, -1, sizeof(_au8DisplayOrder));
		tS32 _s32ReadStatus = _oDpDisplayOrder.s32GetData(_au8DisplayOrder, DISPVIDCTRL_C_U16_ARRAYELEMENTS(_au8DisplayOrder));
		if (DP_S32_NO_ERR > _oDpDisplayOrder.s32GetStatus())
		{
			ETG_TRACE_USR3(("bIsChangedEarlyConfig_LayerManager: status of DisplayOrder is %d.", _oDpDisplayOrder.s32GetStatus() ));
		}
		else if (DP_S32_NO_ERR <= _s32ReadStatus && OSAL_NULL != OSAL_s32MemoryCompare(_au8DisplayOrder, m_au8DisplayOrder, sizeof(_au8DisplayOrder)))
		{
			_bReturn = TRUE;
		}
		// FlushDesignConfigFile
		dp_tclEarlyConfigLayerManagerTrFlushDesignConfigFile _oDpFlushDesignConfigFile;
		tChar _strFlushDesignConfigFile[DISPVIDCTRL_C_U16_ARRAYELEMENTS(m_strFlushDesignConfigFile)];
		(tVoid)OSAL_pvMemorySet((tVoid*)_strFlushDesignConfigFile, -1, sizeof(_strFlushDesignConfigFile));
		_s32ReadStatus = _oDpFlushDesignConfigFile.s32GetData(_strFlushDesignConfigFile, DISPVIDCTRL_C_U16_ARRAYELEMENTS(_strFlushDesignConfigFile));
		if (DP_S32_NO_ERR > _oDpFlushDesignConfigFile.s32GetStatus())
		{
			ETG_TRACE_USR3(("bIsChangedEarlyConfig_LayerManager: FlushDesignConfigFile: %d.", _oDpFlushDesignConfigFile.s32GetStatus() ));
		}
		else if (DP_S32_NO_ERR <= _s32ReadStatus && OSAL_NULL != OSAL_s32StringCompare(_strFlushDesignConfigFile, m_strFlushDesignConfigFile))
		{
			_bReturn = TRUE;
		}
	}
	return _bReturn;
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig_LayerManager()
*
* DESCRIPTION: returns TRUE if EarlxConfig LayerManager configuration information
*              written successful
*
* PARAMETER:   none
*
* RETURNVALUE: tBool
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig_LayerManager(tVoid)
{
	ETG_TRACE_USR4(("bWriteEarlyConfig_LayerManager() entered."));
	if ((TRUE == bIsValidKDSEarlyConfig_LayerManager())
		&& (TRUE == bIsChangedEarlyConfig_LayerManager()))
	{
		if (TRUE != bIsEarlyConfig_DisplayOrder_ManuallySet())
		{
			dp_tclEarlyConfigLayerManagerTrDisplayOrder _oDpDisplayOrder;
			if (DP_S32_NO_ERR != _oDpDisplayOrder.s32SetData(m_au8DisplayOrder, EARLYCONFIG_ARRAY_ELEMENTS(m_au8DisplayOrder)))
			{
			   ETG_TRACE_FATAL(("bWriteEarlyConfig_LayerManager: Failed to set display order to early config!" ));
			   return FALSE;
			}
			else
			{
			   ETG_TRACE_USR2(("bWriteEarlyConfig_LayerManager: Successful set display order to early config!" ));
			}
		}
		if (TRUE != bIsEarlyConfig_FlushDesignConfigFile_ManuallySet())
		{
			dp_tclEarlyConfigLayerManagerTrFlushDesignConfigFile _oDpFlushDesignConfigFile;
			if (DP_S32_NO_ERR != _oDpFlushDesignConfigFile.s32SetData(m_strFlushDesignConfigFile))
			{
			   ETG_TRACE_FATAL(("bWriteEarlyConfig_LayerManager: Failed to set flush design config file %s to early config!",
					   m_strFlushDesignConfigFile ));
			   return FALSE;
			}
			else
			{
			   ETG_TRACE_USR2(("bWriteEarlyConfig_LayerManager: Successful set flush design config file %s to early config!",
					   m_strFlushDesignConfigFile ));
			}
		}
		// trigger storing of datapool
		dp_tclSrvIf _oDpSrvIf;
		tU32 _u32DpID = DP_U32_POOL_ID_EARLYCONFIGLAYERMANAGER;
		tS32 _s32ErrorCode = _oDpSrvIf.s32StorePoolNow(_u32DpID);
		if (DP_S32_NO_ERR != _s32ErrorCode)
		{
		  ETG_TRACE_FATAL(("bWriteEarlyConfig_LayerManager: storing of PDD 0x%08x failed with error code 0x%08x!", _u32DpID, _s32ErrorCode ));
		}
		return TRUE;
	}
	else if (TRUE == bIsValidKDSEarlyConfig_LayerManager())
	{
		return TRUE;
	}
	return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfoEarlyConfig_LayerManager() const
*
* DESCRIPTION: Trace information EarlyConfig_LayerManager
*
* PARAMETER:   const char* strIdentifier
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid dispvidctrl_tclEarlyConfig::vTraceInfoEarlyConfig_LayerManager(const char* strIdentifier) const
{
	ETG_TRACE_USR4(("vTraceInfoEarlyConfig_LayerManager() entered."));
	ETG_TRACE_FATAL(("%26s: <<< EarlyConfig_LayerManager >>> ", strIdentifier ));

	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, " LM_DISPORDER_MANUAL",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsEarlyConfig_DisplayOrder_ManuallySet()) ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, " LM_FLUSHDESN_MANUAL",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsEarlyConfig_FlushDesignConfigFile_ManuallySet()) ));

	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "       KDS_LM_LOADED",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsLoadedKDSEarlyConfig_LayerManager()) ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "        KDS_LM_VALID",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsValidKDSEarlyConfig_LayerManager()) ));

	ETG_TRACE_FATAL(("%26s: %30s=%u %u %u %u ", strIdentifier, "    KDS_LM_DISPORDER",
			ETG_ENUM(EC_SCREENID, m_au8DisplayOrder[0]),
			ETG_ENUM(EC_SCREENID, m_au8DisplayOrder[1]),
			ETG_ENUM(EC_SCREENID, m_au8DisplayOrder[2]),
			ETG_ENUM(EC_SCREENID, m_au8DisplayOrder[3]) ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier,   "KDS_LM_FLUSHDESN_ENU",
			m_u8FlushDesignConfigFile ));
	ETG_TRACE_FATAL(("%26s: %30s=%64s ", strIdentifier, "    KDS_LM_FLUSHDESN",
			m_strFlushDesignConfigFile ));

	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "       EC_LM_CHANGED",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsChangedEarlyConfig_LayerManager()) ));

	dp_tclEarlyConfigLayerManagerTrDisplayOrder _oDpDisplayOrder;
	tU8 _au8DisplayOrder[EARLYCONFIG_ARRAY_ELEMENTS(m_au8DisplayOrder)];
	(tVoid)OSAL_pvMemorySet((tVoid*)_au8DisplayOrder, OSAL_NULL, sizeof(_au8DisplayOrder));
	(tVoid)_oDpDisplayOrder.s32GetData(_au8DisplayOrder, EARLYCONFIG_ARRAY_ELEMENTS(_au8DisplayOrder));
	ETG_TRACE_FATAL(("%26s: %30s=%u %u %u %u ", strIdentifier, "     EC_LM_DISPORDER",
			ETG_ENUM(EC_SCREENID, _au8DisplayOrder[0]),
			ETG_ENUM(EC_SCREENID, _au8DisplayOrder[1]),
			ETG_ENUM(EC_SCREENID, _au8DisplayOrder[2]),
			ETG_ENUM(EC_SCREENID, _au8DisplayOrder[3]) ));

	dp_tclEarlyConfigLayerManagerTrFlushDesignConfigFile _oDpFlushDesignConfigFile;
	tChar _strFlushDesignConfigFile[EARLYCONFIG_ARRAY_ELEMENTS(m_strFlushDesignConfigFile)];
	(tVoid)OSAL_pvMemorySet((tVoid*)_strFlushDesignConfigFile, OSAL_NULL, sizeof(_strFlushDesignConfigFile));
	(tVoid)_oDpFlushDesignConfigFile.s32GetData(_strFlushDesignConfigFile, EARLYCONFIG_ARRAY_ELEMENTS(_strFlushDesignConfigFile)-1);
	ETG_TRACE_FATAL(("%26s: %30s=%64s ", strIdentifier, "     EC_LM_FLUSHDESN",
			_strFlushDesignConfigFile ));
}



/*******************************************************************************
*
* FUNCTION:    bIsEarlyConfig_TouchDriver2ManuallySet()
*
* DESCRIPTION: returns TRUE if EarlyConfig_Touch driver name 2 manually set
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsEarlyConfig_TouchDriver2ManuallySet(tVoid) const
{
   ETG_TRACE_USR4(("bIsEarlyConfig_TouchDriver2ManuallySet() entered."));

   tChar _strTouchDriverName_Manual[sizeof(m_strTouchDriverName2)] = {0};
   dp_tclDimmingDPEarlyConfig_TouchDriverName2_Manually _oTouchDriverNameManual;
   tS32 _s32ReadStatus = _oTouchDriverNameManual.s32GetData(_strTouchDriverName_Manual, sizeof(_strTouchDriverName_Manual)-1);

   // check for update of manual EarlyConfig display resolution
   if (DP_S32_NO_ERR > _oTouchDriverNameManual.s32GetStatus())
   {
	   // not initialized
	   ETG_TRACE_FATAL(("bIsEarlyConfig_TouchDriver2ManuallySet: Status Touch Driver Name 2 manual: %d.", _oTouchDriverNameManual.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   if (0 < strlen(_strTouchDriverName_Manual))
	   {
		   return TRUE;
	   }

   }
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bIsEarlyConfig_TouchDriverConfigFile2ManuallySet()
*
* DESCRIPTION: returns TRUE if EarlyConfig_Touch driver name 2 manually set
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsEarlyConfig_TouchDriverConfigFile2ManuallySet(tVoid) const
{
   ETG_TRACE_USR4(("bIsEarlyConfig_TouchDriverConfigFile2ManuallySet() entered."));

   tChar _strTouchConfigFileName_Manual[sizeof(m_strTouchConfigFileName2)] = {0};
   dp_tclDimmingDPEarlyConfig_TouchConfigFile2_Manually _oTouchConfigFileNameManual;
   tS32 _s32ReadStatus = _oTouchConfigFileNameManual.s32GetData(_strTouchConfigFileName_Manual, sizeof(_strTouchConfigFileName_Manual)-1);

   // check for update of manual EarlyConfig display resolution
   if (DP_S32_NO_ERR > _oTouchConfigFileNameManual.s32GetStatus())
   {
	   // not initialized
	   ETG_TRACE_FATAL(("bIsEarlyConfig_TouchDriverConfigFile2ManuallySet: Status Touch Driver Config File 2 manual: %d.", _oTouchConfigFileNameManual.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   if (0 < strlen(_strTouchConfigFileName_Manual))
	   {
		   return TRUE;
	   }

   }
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bIsExistKDSEarlyConfig_TouchConfigFile()
*
* DESCRIPTION: returns TRUE if loaded KDS touch config file exist
*              - file name like /lib/firmware/<driver name>/<config file name>
*
* PARAMETER:   const char* strTouchDriverName
*              const char* strTouchConfigFileName
*
* RETURNVALUE: tBool     {TRUE - exist | else - not exist}
*
*******************************************************************************/
const char m_strFirmwareFolder[] = "/lib/firmware";
tBool dispvidctrl_tclEarlyConfig::bIsExistKDSEarlyConfig_TouchConfigFile(const char* strTouchDriverName, const char* strTouchConfigFileName) const
{
   ETG_TRACE_USR4(("bIsExistKDSEarlyConfig_TouchConfigFile() entered."));

   if ((OSAL_NULL == strTouchDriverName) || (OSAL_NULL == strTouchConfigFileName))
	   return FALSE;
   if ((0 == strlen(strTouchDriverName)) || (DP_U8_KDSLEN_EARLYCONFIG_TOUCH2_TOUCHDRIVERNAME <= strlen(strTouchDriverName)))
	   return FALSE;
   if ((DP_U8_KDSLEN_EARLYCONFIG_TOUCH2_TOUCHCONFIGFILENAME <= strlen(strTouchConfigFileName)))
	   return FALSE;

   tBool _bExist = FALSE;
   char _strEarlyConfigTouchFile[sizeof(m_strFirmwareFolder) + DP_U8_KDSLEN_EARLYCONFIG_TOUCH2_TOUCHDRIVERNAME + DP_U8_KDSLEN_EARLYCONFIG_TOUCH2_TOUCHCONFIGFILENAME + 3];

   // prepare touch config file name
   (tVoid)memset((tVoid*)_strEarlyConfigTouchFile, OSAL_NULL, sizeof(_strEarlyConfigTouchFile));
   // ... set firmware folder
   strcat(_strEarlyConfigTouchFile, m_strFirmwareFolder);
   strcat(_strEarlyConfigTouchFile, "/");
   // ... add driver name as folder
   strcat(_strEarlyConfigTouchFile, strTouchDriverName);
   strcat(_strEarlyConfigTouchFile, "/");
   strcat(_strEarlyConfigTouchFile, strTouchConfigFileName);
   // open the file
#ifdef DISPVIDCTRL_USE_OSAL
   OSAL_tIODescriptor _hEarlyConfigTouchFile = OSAL_IOOpen(_strEarlyConfigTouchFile,OSAL_EN_READONLY);
   if (OSAL_C_INVALID_HANDLE != _hEarlyConfigTouchFile)
#else
   int _hEarlyConfigTouchFile = open(_strEarlyConfigTouchFile, O_RDONLY);
   if (-1 != _hEarlyConfigTouchFile)
#endif // #ifdef DISPVIDCTRL_USE_OSAL
   {
	   _bExist = TRUE;
#ifdef DISPVIDCTRL_USE_OSAL
	   (tVoid)OSAL_s32IOClose(_hEarlyConfigTouchFile);
#else
	   close(_hEarlyConfigTouchFile);
#endif // #ifdef DISPVIDCTRL_USE_OSAL
	   ETG_TRACE_USR3(("bIsExistKDSEarlyConfig_TouchConfigFile: existing touch config file %s ",
			   _strEarlyConfigTouchFile ));
   }
   else
   {
	   ETG_TRACE_FATAL(("bIsExistKDSEarlyConfig_TouchConfigFile: invalid touch config file %s ",
			   _strEarlyConfigTouchFile ));
   }

   return _bExist;
}

/*******************************************************************************
*
* FUNCTION:    bIsEarlyConfig_TouchI2CAddress2ManuallySet()
*
* DESCRIPTION: returns TRUE if EarlyConfig_Touch driver I2C address 2 manually set
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsEarlyConfig_TouchI2CAddress2ManuallySet(tVoid) const
{
   ETG_TRACE_USR4(("bIsEarlyConfig_TouchI2CAddress2ManuallySet() entered."));

   tChar _strTouchDriverI2CAddress_Manual[DP_U8_NORLEN_EARLYCONFIG_TOUCHI2CADDRESS] = {0};
   dp_tclDimmingDPEarlyConfig_TouchI2CAddress2_Manually _oTouchDriverI2CAddressManual;
   tS32 _s32ReadStatus = _oTouchDriverI2CAddressManual.s32GetData(_strTouchDriverI2CAddress_Manual, sizeof(_strTouchDriverI2CAddress_Manual)-1);

   // check for update of manual EarlyConfig display resolution
   if (DP_S32_NO_ERR > _oTouchDriverI2CAddressManual.s32GetStatus())
   {
	   // not initialized
	   ETG_TRACE_FATAL(("bIsEarlyConfig_TouchI2CAddress2ManuallySet: Status TouchDriverI2CAddress 2 manual: %d.", _oTouchDriverI2CAddressManual.s32GetStatus() ));
   }
   else if (DP_S32_NO_ERR <= _s32ReadStatus)
   {
	   if (0 < strlen(_strTouchDriverI2CAddress_Manual))
	   {
		   return TRUE;
	   }

   }
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    bIsValidKDSEarlyConfig_TwoTouchDriver()
*
* DESCRIPTION: returns TRUE if loaded KDS data of TwoTouchDriver are valid
*              - configured driver name must be known
*              - configured config file name must be known
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - valid | else - not valid}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsValidKDSEarlyConfig_TwoTouchDriver() const
{
   ETG_TRACE_USR4(("bIsValidKDSEarlyConfig_TwoTouchDriver() entered."));
   return bIsLoadedKDSEarlyConfig_TwoTouchDriver();
}

/*******************************************************************************
*
* FUNCTION:    bIsChangedEarlyConfig_TwoTouchDriver()
*
* DESCRIPTION: returns TRUE if KDS data of TwoTouchDriver not the same like
*              early config data
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - new | else - not new}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bIsChangedEarlyConfig_TwoTouchDriver() const
{
   ETG_TRACE_USR4(("bIsChangedEarlyConfig_TwoTouchDriver() entered."));
   return m_bTwoTouchChanged;
}

/*******************************************************************************
*
* FUNCTION:    bWriteEarlyConfig_TwoTouchDriver()
*
* DESCRIPTION: write the bWriteEarlyConfig_TwoTouchDriver
*
* PARAMETER:   None
*
* RETURNVALUE: tBool     {TRUE - successful | else - failed}
*
*******************************************************************************/
tBool dispvidctrl_tclEarlyConfig::bWriteEarlyConfig_TwoTouchDriver()
{
   ETG_TRACE_USR4(("bWriteEarlyConfig_TwoTouchDriver() entered."));

   tBool _bReturn = TRUE;
#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
   tChar _strTouchDriverName[sizeof(m_strTouchDriverName)] = {0};
   (tVoid)OSAL_pvMemorySet((tVoid*)_strTouchDriverName, OSAL_NULL, sizeof(_strTouchDriverName));
   tChar _strTouchConfigFileName[sizeof(m_strTouchConfigFileName)] = {0};
   (tVoid)OSAL_pvMemorySet((tVoid*)_strTouchConfigFileName, OSAL_NULL, sizeof(_strTouchConfigFileName));
   tChar _strTouchI2CAddress[DP_U8_NORLEN_EARLYCONFIG_TOUCHI2CADDRESS] = {0};
   (tVoid)OSAL_pvMemorySet((tVoid*)_strTouchI2CAddress, OSAL_NULL, sizeof(_strTouchI2CAddress));
   tChar _strTouchDriverName2[sizeof(m_strTouchDriverName)] = {0};
   (tVoid)OSAL_pvMemorySet((tVoid*)_strTouchDriverName2, OSAL_NULL, sizeof(_strTouchDriverName2));
   tChar _strTouchConfigFileName2[sizeof(m_strTouchConfigFileName)] = {0};
   (tVoid)OSAL_pvMemorySet((tVoid*)_strTouchConfigFileName2, OSAL_NULL, sizeof(_strTouchConfigFileName2));
   tChar _strTouchI2CAddress2[DP_U8_NORLEN_EARLYCONFIG_TOUCHI2CADDRESS] = {0};
   (tVoid)OSAL_pvMemorySet((tVoid*)_strTouchI2CAddress2, OSAL_NULL, sizeof(_strTouchI2CAddress2));
   // init values - loading EarlyConfig_TwoTouchDriver values
   dp_tclEarlyConfigTwoTouchDriverTrTouchDriverName _oTouchDriverName;
   (tVoid)_oTouchDriverName.s32GetData(_strTouchDriverName,sizeof(_strTouchDriverName)-1);
   dp_tclEarlyConfigTwoTouchDriverTrTouchConfigFileName _oTouchConfigFileName;
   (tVoid)_oTouchConfigFileName.s32GetData(_strTouchConfigFileName,sizeof(_strTouchConfigFileName)-1);
   dp_tclEarlyConfigTwoTouchDriverTrTouchI2CAddr _oTouchI2CAddress;
   (tVoid)_oTouchI2CAddress.s32GetData(_strTouchI2CAddress,sizeof(_strTouchI2CAddress)-1);
   dp_tclEarlyConfigTwoTouchDriverTrTouchDriverName2 _oTouchDriverName2;
   (tVoid)_oTouchDriverName2.s32GetData(_strTouchDriverName2,sizeof(_strTouchDriverName2)-1);
   dp_tclEarlyConfigTwoTouchDriverTrTouchConfigFileName2 _oTouchConfigFileName2;
   (tVoid)_oTouchConfigFileName2.s32GetData(_strTouchConfigFileName2,sizeof(_strTouchConfigFileName2)-1);
   dp_tclEarlyConfigTwoTouchDriverTrTouchI2CAddr2 _oTouchI2CAddress2;
   (tVoid)_oTouchI2CAddress2.s32GetData(_strTouchI2CAddress2,sizeof(_strTouchI2CAddress2)-1);
   // load KDS values
   if ((TRUE == bIsValidKDSEarlyConfig_TwoTouchDriver()) && (TRUE == bIsChangedEarlyConfig_TwoTouchDriver()))
   {
	   (tVoid)memset((tVoid*)_strTouchDriverName, OSAL_NULL, sizeof(_strTouchDriverName));
	   (tVoid)OSAL_szStringNCopy(_strTouchDriverName, m_strTouchDriverName, sizeof(_strTouchDriverName)-1);
	   (tVoid)memset((tVoid*)_strTouchConfigFileName, OSAL_NULL, sizeof(_strTouchConfigFileName));
	   (tVoid)OSAL_szStringNCopy(_strTouchConfigFileName, m_strTouchConfigFileName, sizeof(_strTouchConfigFileName)-1);
	   (tVoid)memset((tVoid*)_strTouchI2CAddress, OSAL_NULL, sizeof(_strTouchI2CAddress));
	   (tVoid)OSAL_szStringNCopy(_strTouchI2CAddress, m_strTouchDriverI2CAddress, sizeof(_strTouchI2CAddress)-1);
	   (tVoid)memset((tVoid*)_strTouchDriverName2, OSAL_NULL, sizeof(_strTouchDriverName2));
	   (tVoid)OSAL_szStringNCopy(_strTouchDriverName2, m_strTouchDriverName2, sizeof(_strTouchDriverName2)-1);
	   (tVoid)memset((tVoid*)_strTouchConfigFileName2, OSAL_NULL, sizeof(_strTouchConfigFileName2));
	   (tVoid)OSAL_szStringNCopy(_strTouchConfigFileName2, m_strTouchConfigFileName2, sizeof(_strTouchConfigFileName2)-1);
	   (tVoid)memset((tVoid*)_strTouchI2CAddress2, OSAL_NULL, sizeof(_strTouchI2CAddress2));
	   (tVoid)OSAL_szStringNCopy(_strTouchI2CAddress2, m_strTouchDriverI2CAddress2, sizeof(_strTouchI2CAddress2)-1);
   }
   // load manual values
   if (TRUE == bIsEarlyConfig_TouchDriverManuallySet())
   {
	   dp_tclDimmingDPEarlyConfig_TouchDriverName_Manually _oManual;
	   (tVoid)memset((tVoid*)_strTouchDriverName, OSAL_NULL, sizeof(_strTouchDriverName));
	   (tVoid)_oManual.s32GetData(_strTouchDriverName, sizeof(_strTouchDriverName)-1);
   }
   if (TRUE == bIsEarlyConfig_TouchDriverConfigFileManuallySet())
   {
	   dp_tclDimmingDPEarlyConfig_TouchConfigFile_Manually _oManual;
	   (tVoid)memset((tVoid*)_strTouchConfigFileName, OSAL_NULL, sizeof(_strTouchConfigFileName));
	   (tVoid)_oManual.s32GetData(_strTouchConfigFileName, sizeof(_strTouchConfigFileName)-1);
   }
   if (TRUE == bIsEarlyConfig_TouchI2CAddressManuallySet())
   {
	   dp_tclDimmingDPEarlyConfig_TouchConfigFile_Manually _oManual;
	   (tVoid)memset((tVoid*)_strTouchI2CAddress, OSAL_NULL, sizeof(_strTouchI2CAddress));
	   (tVoid)_oManual.s32GetData(_strTouchI2CAddress, sizeof(_strTouchI2CAddress)-1);
   }

   if (TRUE == bIsEarlyConfig_TouchDriver2ManuallySet())
   {
	   dp_tclDimmingDPEarlyConfig_TouchDriverName2_Manually _oManual;
	   (tVoid)memset((tVoid*)_strTouchDriverName2, OSAL_NULL, sizeof(_strTouchDriverName2));
	   (tVoid)_oManual.s32GetData(_strTouchDriverName2, sizeof(_strTouchDriverName2)-1);
   }
   if (TRUE == bIsEarlyConfig_TouchDriverConfigFile2ManuallySet())
   {
	   dp_tclDimmingDPEarlyConfig_TouchConfigFile2_Manually _oManual;
	   (tVoid)memset((tVoid*)_strTouchConfigFileName2, OSAL_NULL, sizeof(_strTouchConfigFileName2));
	   (tVoid)_oManual.s32GetData(_strTouchConfigFileName2, sizeof(_strTouchConfigFileName2)-1);
   }
   if (TRUE == bIsEarlyConfig_TouchI2CAddress2ManuallySet())
   {
	   dp_tclDimmingDPEarlyConfig_TouchConfigFile2_Manually _oManual;
	   (tVoid)memset((tVoid*)_strTouchI2CAddress2, OSAL_NULL, sizeof(_strTouchI2CAddress2));
	   (tVoid)_oManual.s32GetData(_strTouchI2CAddress2, sizeof(_strTouchI2CAddress2)-1);
   }
   // write values
   if (DP_S32_NO_ERR != _oTouchDriverName.s32SetData(_strTouchDriverName))
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoTouchDriver: Failed to set driver name %s to early config!", _strTouchDriverName ));
	   _bReturn = FALSE;
   }
   if (DP_S32_NO_ERR != _oTouchConfigFileName.s32SetData(_strTouchConfigFileName))
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoTouchDriver: Failed to set config file name %s to early config!", _strTouchConfigFileName ));
	   _bReturn = FALSE;
   }
   if (DP_S32_NO_ERR != _oTouchI2CAddress.s32SetData(_strTouchI2CAddress))
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoTouchDriver: Failed to set I2C address %s to early config!", _strTouchI2CAddress ));
	   _bReturn = FALSE;
   }
   if (DP_S32_NO_ERR != _oTouchDriverName2.s32SetData(_strTouchDriverName2))
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoTouchDriver: Failed to set driver name 2 %s to early config!", _strTouchDriverName2 ));
	   _bReturn = FALSE;
   }
   if (DP_S32_NO_ERR != _oTouchConfigFileName2.s32SetData(_strTouchConfigFileName2))
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoTouchDriver: Failed to set config file name 2 %s to early config!", _strTouchConfigFileName2 ));
	   _bReturn = FALSE;
   }
   if (DP_S32_NO_ERR != _oTouchI2CAddress2.s32SetData(_strTouchI2CAddress2))
   {
	   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoTouchDriver: Failed to set I2C address 2 %s to early config!", _strTouchI2CAddress2 ));
	   _bReturn = FALSE;
   }
   // trigger storing of datapool
   dp_tclSrvIf _oDpSrvIf;
   tU32 _u32DpID = DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER;
   tS32 _s32ErrorCode = _oDpSrvIf.s32StorePoolNow(_u32DpID);
   if (DP_S32_NO_ERR != _s32ErrorCode)
   {
      ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoTouchDriver: storing of PDD 0x%08x failed with error code 0x%08x!", _u32DpID, _s32ErrorCode ));
   }
#else
   ETG_TRACE_FATAL(("bWriteEarlyConfig_TwoTouchDriver: UNSUPPORTED DATAPOOL EarlyConfig_TwoTouchDriver!" ));
#endif //#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
   if (TRUE == _bReturn)
   {
	   m_bTwoTouchChanged = FALSE;
   }
   return _bReturn;
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfoEarlyConfig_TwoTouchDriver() const
*
* DESCRIPTION: Trace information EarlyConfig_TwoTouchDriver
*
* PARAMETER:   const char* strIdentifier
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid dispvidctrl_tclEarlyConfig::vTraceInfoEarlyConfig_TwoTouchDriver(const char* strIdentifier) const
{
	ETG_TRACE_USR4(("vTraceInfoEarlyConfig_TwoTouchDriver() entered."));
	ETG_TRACE_FATAL(("%26s: <<< EarlyConfig_TwoTouchDriver >>> ", strIdentifier ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "   KDS_2TOUCH_LOADED",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsLoadedKDSEarlyConfig_TouchDriver()) ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "    KDS_2TOUCH_VALID",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsValidKDSEarlyConfig_TouchDriver()) ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "   EC_2TOUCH_CHANGED",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsChangedEarlyConfig_TouchDriver()) ));

	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "   2TOUCH_DRV_MANUAL",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsEarlyConfig_TouchDriverManuallySet()) ));
	ETG_TRACE_FATAL(("%26s: %30s='%64s' ", strIdentifier, "  KDS_2TOUCH_DRVNAME",
			m_strTouchDriverName ));
#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
	dp_tclEarlyConfigTwoTouchDriverTrTouchDriverName _oTouchDriverName;
	tChar _strTouchDriverName[sizeof(m_strTouchDriverName)];
	(tVoid)OSAL_pvMemorySet((tVoid*)_strTouchDriverName, OSAL_NULL, sizeof(_strTouchDriverName));
	(tVoid)_oTouchDriverName.s32GetData(_strTouchDriverName, sizeof(_strTouchDriverName)-1);
	ETG_TRACE_FATAL(("%26s: %30s='%64s' ", strIdentifier, "   EC_2TOUCH_DRVNAME",
			_strTouchDriverName ));
#else
	ETG_TRACE_FATAL(("%26s: %30s='%64s' ", strIdentifier, "   EC_2TOUCH_DRVNAME",
			"---" ));
#endif //#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER

	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "2TOUCH_DRVCFG_MANUAL",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsEarlyConfig_TouchDriverConfigFileManuallySet()) ));
	ETG_TRACE_FATAL(("%26s: %30s='%64s' ", strIdentifier, " KDS_2TOUCH_FILENAME",
			m_strTouchConfigFileName ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "KDS_2TOUCH_FILEEXIST",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsExistKDSEarlyConfig_TouchConfigFile(m_strTouchDriverName, m_strTouchConfigFileName)) ));
#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
	dp_tclEarlyConfigTwoTouchDriverTrTouchConfigFileName _oTouchConfigFileName;
	tChar _strTouchConfigFileName[sizeof(m_strTouchConfigFileName)];
	(tVoid)OSAL_pvMemorySet((tVoid*)_strTouchConfigFileName, OSAL_NULL, sizeof(_strTouchConfigFileName));
	(tVoid)_oTouchConfigFileName.s32GetData(_strTouchConfigFileName, sizeof(_strTouchConfigFileName)-1);
	ETG_TRACE_FATAL(("%26s: %30s='%64s' ", strIdentifier, "  EC_2TOUCH_FILENAME",
			_strTouchConfigFileName ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, " EC_2TOUCH_FILEEXIST",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsExistKDSEarlyConfig_TouchConfigFile(_strTouchDriverName, _strTouchConfigFileName)) ));
#else
	ETG_TRACE_FATAL(("%26s: %30s='%64s' ", strIdentifier, "  EC_2TOUCH_FILENAME",
			"---" ));
#endif //#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER

	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "2TOUCH_DRVI2C_MANUAL",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsEarlyConfig_TouchI2CAddressManuallySet()) ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier,   " KDS_2TOUCH_I2CBUSNR",
			m_u8TouchDriverI2CAddress ));
	ETG_TRACE_FATAL(("%26s: %30s='%12s' ", strIdentifier, "   KDS_2TOUCH_I2CADR",
			m_strTouchDriverI2CAddress ));
#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
	dp_tclEarlyConfigTwoTouchDriverTrTouchI2CAddr _oTouchI2CAddress;
	tChar _strTouchI2CAddress[DP_U8_NORLEN_EARLYCONFIG_TOUCHI2CADDRESS];
	(tVoid)OSAL_pvMemorySet((tVoid*)_strTouchI2CAddress, OSAL_NULL, sizeof(_strTouchI2CAddress));
	(tVoid)_oTouchI2CAddress.s32GetData(_strTouchI2CAddress, sizeof(_strTouchI2CAddress)-1);
	ETG_TRACE_FATAL(("%26s: %30s='%64s' ", strIdentifier, "    EC_2TOUCH_I2CADR",
			_strTouchI2CAddress ));
#else
	ETG_TRACE_FATAL(("%26s: %30s='%64s' ", strIdentifier, "    EC_2TOUCH_I2CADR",
			"---" ));
#endif //#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER

	// second touch driver
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "  2TOUCH_DRV2_MANUAL",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsEarlyConfig_TouchDriver2ManuallySet()) ));
	ETG_TRACE_FATAL(("%26s: %30s='%64s' ", strIdentifier, " KDS_2TOUCH_DRVNAME2",
			m_strTouchDriverName2 ));
#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
	dp_tclEarlyConfigTwoTouchDriverTrTouchDriverName2 _oTouchDriverName2;
	tChar _strTouchDriverName2[sizeof(m_strTouchDriverName2)];
	(tVoid)OSAL_pvMemorySet((tVoid*)_strTouchDriverName2, OSAL_NULL, sizeof(_strTouchDriverName2));
	(tVoid)_oTouchDriverName2.s32GetData(_strTouchDriverName2, sizeof(_strTouchDriverName2)-1);
	ETG_TRACE_FATAL(("%26s: %30s='%64s' ", strIdentifier, "  EC_2TOUCH_DRVNAME2",
			_strTouchDriverName2 ));
#else
	ETG_TRACE_FATAL(("%26s: %30s='%64s' ", strIdentifier, "  EC_2TOUCH_DRVNAME2",
			"---" ));
#endif //#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER

	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "2TOUCH_DRVCFG_MANUA2",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsEarlyConfig_TouchDriverConfigFile2ManuallySet()) ));
	ETG_TRACE_FATAL(("%26s: %30s='%64s' ", strIdentifier, "KDS_2TOUCH_FILENAME2",
			m_strTouchConfigFileName2 ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "KDS_2TOUCH_FILEEXIS2",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsExistKDSEarlyConfig_TouchConfigFile(m_strTouchDriverName2, m_strTouchConfigFileName2)) ));
#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
	dp_tclEarlyConfigTwoTouchDriverTrTouchConfigFileName2 _oTouchConfigFileName2;
	tChar _strTouchConfigFileName2[sizeof(m_strTouchConfigFileName2)];
	(tVoid)OSAL_pvMemorySet((tVoid*)_strTouchConfigFileName2, OSAL_NULL, sizeof(_strTouchConfigFileName2));
	(tVoid)_oTouchConfigFileName2.s32GetData(_strTouchConfigFileName2, sizeof(_strTouchConfigFileName2)-1);
	ETG_TRACE_FATAL(("%26s: %30s='%64s' ", strIdentifier, " EC_2TOUCH_FILENAME2",
			_strTouchConfigFileName2 ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, " EC_2TOUCH_FILEEXIS2",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsExistKDSEarlyConfig_TouchConfigFile(_strTouchDriverName2, _strTouchConfigFileName2)) ));
#else
	ETG_TRACE_FATAL(("%26s: %30s='%64s' ", strIdentifier, " EC_2TOUCH_FILENAME2",
			"---" ));
#endif //#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER

	ETG_TRACE_FATAL(("%26s: %30s=%u ",   strIdentifier, "2TOUCH_DRVI2C_MANUA2",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsEarlyConfig_TouchI2CAddress2ManuallySet()) ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier,   "KDS_2TOUCH_I2CBUSNR2",
			m_u8TouchDriverI2CAddress2 ));
	ETG_TRACE_FATAL(("%26s: %30s='%12s' ", strIdentifier, "  KDS_2TOUCH_I2CADR2",
			m_strTouchDriverI2CAddress2 ));
#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
	dp_tclEarlyConfigTwoTouchDriverTrTouchI2CAddr2 _oTouchI2CAddress2;
	tChar _strTouchI2CAddress2[DP_U8_NORLEN_EARLYCONFIG_TOUCHI2CADDRESS];
	(tVoid)OSAL_pvMemorySet((tVoid*)_strTouchI2CAddress2, OSAL_NULL, sizeof(_strTouchI2CAddress2));
	(tVoid)_oTouchI2CAddress2.s32GetData(_strTouchI2CAddress2, sizeof(_strTouchI2CAddress2)-1);
	ETG_TRACE_FATAL(("%26s: %30s='%64s' ", strIdentifier, "   EC_2TOUCH_I2CADR2",
			_strTouchI2CAddress2 ));
#else
	ETG_TRACE_FATAL(("%26s: %30s='%64s' ", strIdentifier, "   EC_2TOUCH_I2CADR2",
			"---" ));
#endif //#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
}
