/*
 * dispvidctrl_tclTouchDriverControl.cpp
 *
 *  Created on: Jul 4, 2017
 *      Author: bne2hi
 */
/************************************************************************
* FILE:        dispvidctrl_tclTouchDriverControl.cpp
* PROJECT:
* SW-COMPONENT:vd_dimming
*----------------------------------------------------------------------
*
* DESCRIPTION: implementation of touch driver control class
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 04.07.2017| CM-AI/EPB2 Bernard | init
*
*************************************************************************/

/*******************************************************************************
| includes: system- and project- includes
|-----------------------------------------------------------------------------*/
// Basic OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

/*******************************************************************************
| set own module id
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| includes: needed interfaces from external components
|-----------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <libkmod.h>

/*******************************************************************************
| includes: internal and external interfaces from this component
|-----------------------------------------------------------------------------*/

#include "dispvidctrl_tclTouchDriverControl.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_APPLICATION
#include "trcGenProj/Header/dispvidctrl_tclTouchDriverControl.cpp.trc.h"
#endif

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#include "dp_drvass_video_if.h"

/*******************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| typedefs (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function prototype (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementation (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
*
* FUNCTION:    dispvidctrl_tclTouchDriverControl()
*
* DESCRIPTION: constructor
*
* PARAMETER:   const dispvidctrl_tclAppMain*
*
* RETURNVALUE: none
*
*******************************************************************************/
dispvidctrl_tclTouchDriverControl::dispvidctrl_tclTouchDriverControl(const dispvidctrl_tclAppMain* poMainAppl)
: I_dispvidctrl_tclTouchDriverControl(poMainAppl)
{
   ETG_TRACE_USR4(("dispvidctrl_tclTouchDriverControl() entered."));
   memset((tVoid*)m_aenState, OSAL_NULL, sizeof(m_aenState));
}

/*******************************************************************************
*
* FUNCTION:    ~dispvidctrl_tclTouchDriverControl()
*
* DESCRIPTION: destructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
dispvidctrl_tclTouchDriverControl::~dispvidctrl_tclTouchDriverControl()
{
   ETG_TRACE_USR4(("~dispvidctrl_tclTouchDriverControl() entered."));
}

/*******************************************************************************
*
* FUNCTION:    vHandleMessage()
*
* DESCRIPTION: handle internal message
*
* PARAMETER:   dispvidctrl_tclBaseIf::TMsg* pMsg
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid dispvidctrl_tclTouchDriverControl::vHandleMessage(dispvidctrl_tclBaseIf::TMsg* pMsg)
{
      DISPVIDCTRL_NULL_POINTER_CHECK(pMsg);
      ETG_TRACE_USR4(("vHandleMessage() entered %u -> data: %d.", ETG_ENUM(HC_INT_MSG , (tU32)pMsg->eCmd), pMsg->u.u32Data));
}

/*******************************************************************************
*
* FUNCTION:    vHandleTraceMessage()
*
* DESCRIPTION:    handler for trace command
*
* PARAMETER:      const tUChar* puchData: reference of received message
*
* RETURNVALUE:    None.
*
*******************************************************************************/
tVoid dispvidctrl_tclTouchDriverControl::vHandleTraceMessage(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceMessage() entered (data: 0x%08x).", puchData ));
   DISPVIDCTRL_NULL_POINTER_CHECK(puchData);

   tU32 u32MsgCode = (puchData[1]<<8) | puchData[2];
   switch (u32MsgCode)
   {
   case DISPVIDCTRL_DIMMING_TOUCHDRIVERCONTROL:
	   {
		   // Trouch driver control of specified touch driver.
		   tU8 param1 = puchData[3];
		   tU8 param2 = puchData[4];

		   // DISPVIDCTRL_DIMMING_TOUCHDRIVERCONTROL: user specific code start
		   tS32 _s32Result = OSAL_ERROR;
		   if ((EARLY_CONFIG_TOUCHDRIVER_FIRST == param1) || (EARLY_CONFIG_TOUCHDRIVER_SECOND == param1))
		   {
			   if (EARLY_CONFIG_TOUCHDRIVER_ACTION_STOP == param2)
			   {
				   _s32Result = s32StopDriver(param1);
			   }
			   else if (EARLY_CONFIG_TOUCHDRIVER_ACTION_START == param2)
			   {
				   _s32Result = s32StartDriver(param1);
			   }
			   else if (EARLY_CONFIG_TOUCHDRIVER_ACTION_RESTART == param2)
			   {
				   _s32Result = s32RestartDriver(param1);
			   }
			   else
			   {
				   _s32Result = s32TraceStatusDriver(param1);
			   }
		   }
		   else
		   {
			   if (EARLY_CONFIG_TOUCHDRIVER_ACTION_STOP == param2)
			   {
				   _s32Result = s32StopDriver(EARLY_CONFIG_TOUCHDRIVER_FIRST);
				   _s32Result = s32StopDriver(EARLY_CONFIG_TOUCHDRIVER_SECOND);
			   }
			   else if (EARLY_CONFIG_TOUCHDRIVER_ACTION_START == param2)
			   {
				   _s32Result = s32StartDriver(EARLY_CONFIG_TOUCHDRIVER_FIRST);
				   _s32Result = s32StartDriver(param1);
			   }
			   else if (EARLY_CONFIG_TOUCHDRIVER_ACTION_RESTART == param2)
			   {
				   _s32Result = s32RestartDriver(EARLY_CONFIG_TOUCHDRIVER_FIRST);
				   _s32Result = s32RestartDriver(EARLY_CONFIG_TOUCHDRIVER_SECOND);
			   }
			   else
			   {
				   _s32Result = s32TraceStatusDriver(EARLY_CONFIG_TOUCHDRIVER_FIRST);
				   _s32Result = s32TraceStatusDriver(EARLY_CONFIG_TOUCHDRIVER_SECOND);
			   }
		   }
		   // DISPVIDCTRL_DIMMING_TOUCHDRIVERCONTROL: user specific code end
	   }
	   break;

   default:
	   break;
   }
}

/*******************************************************************************
*
* FUNCTION:    vGetReferences(tVoid)
*
* DESCRIPTION:    Function to get all reference needed by this class.
*             A reference should always be the Interface class of the object
*
* PARAMETER:    None.
*
* RETURNVALUE:    None.
*
*******************************************************************************/
tVoid dispvidctrl_tclTouchDriverControl::vGetReferences(tVoid)
{
   ETG_TRACE_USR4(("vGetReferences() entered."));
}

/*******************************************************************************
*
* FUNCTION:    tVoid vStartCommunication()
*
* DESCRIPTION:    Function to start all dynamic objects e.g. threads, ...
*
* PARAMETER:    None.
*
* RETURNVALUE:    None.
*
*******************************************************************************/
tVoid dispvidctrl_tclTouchDriverControl::vStartCommunication(tVoid)
{
   ETG_TRACE_USR4(("vStartCommunication() entered."));
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfo()
*
* DESCRIPTION:    Trace information
*
* PARAMETER:    None.
*
* RETURNVALUE:    None.
*
*******************************************************************************/
tVoid dispvidctrl_tclTouchDriverControl::vTraceInfo()
{
   ETG_TRACE_USR4(("vTraceInfo() entered."));
}

/*******************************************************************************
*
* FUNCTION:     s32StartDriver()
*
* DESCRIPTION:  start touch driver of specified index
*
* PARAMETER:    None
*
* RETURNVALUE:  tS32
*
*******************************************************************************/
tS32 dispvidctrl_tclTouchDriverControl::s32StartDriver(const tU8 u8Index)
{
   ETG_TRACE_USR3(("s32StartDriver(index: %u) entered.", u8Index ));
   const char* _cstrDriverName = cstrGetDriverName(u8Index);
   if ((OSAL_NULL != _cstrDriverName) && (0 < strlen(_cstrDriverName)))
   {
	   if (enTOUCH_DRV_STARTED != m_aenState[u8Index])
	   {
		   if (OSAL_OK == s32StartDriver(_cstrDriverName))
		   {
			   m_aenState[u8Index] = enTOUCH_DRV_STARTED;
			   return OSAL_OK;
		   }
		   else
		   {
			   ETG_TRACE_FATAL(("s32StartDriver(index: %u) starting of touch driver failed!.", u8Index ));
		   }
	   }
	   else
	   {
		   ETG_TRACE_USR2(("s32StartDriver(index: %u): touch driver is started.", u8Index ));
		   return OSAL_OK;
	   }
   }
   return OSAL_ERROR;
}

/*******************************************************************************
*
* FUNCTION:     s32StopDriver()
*
* DESCRIPTION:  stop touch driver of specified index
*
* PARAMETER:    const tU8 u8Index
*
* RETURNVALUE:  tS32
*
*******************************************************************************/
tS32 dispvidctrl_tclTouchDriverControl::s32StopDriver(const tU8 u8Index)
{
   ETG_TRACE_USR3(("s32StopDriver(index: %u) entered.", u8Index ));
   const char* _cstrDriverName = cstrGetDriverName(u8Index);
   if ((OSAL_NULL != _cstrDriverName) && (0 < strlen(_cstrDriverName)))
   {
	   if (enTOUCH_DRV_STOPPED != m_aenState[u8Index])
	   {
		   if (OSAL_OK == s32StopDriver(_cstrDriverName))
		   {
			   m_aenState[u8Index] = enTOUCH_DRV_STOPPED;
			   return OSAL_OK;
		   }
		   else
		   {
			   ETG_TRACE_FATAL(("s32StopDriver(index: %u) stopping of touch driver failed!.", u8Index ));
		   }
	   }
	   else
	   {
		   ETG_TRACE_USR2(("s32StopDriver(index: %u) touch driver is stopped.", u8Index ));
		   return OSAL_OK;
	   }
   }
   return OSAL_ERROR;
}

/*******************************************************************************
*
* FUNCTION:     s32RestartDriver()
*
* DESCRIPTION:  stop and start touch driver of specified index
*
* PARAMETER:    const tU8 u8Index
*
* RETURNVALUE:  tS32
*
*******************************************************************************/
tS32 dispvidctrl_tclTouchDriverControl::s32RestartDriver(const tU8 u8Index)
{
   ETG_TRACE_USR4(("s32RestartDriver(index: %u) entered.", u8Index ));
   tS32 _s32Return = OSAL_ERROR;
   tU8 _u8Idx = (EARLY_CONFIG_TOUCHDRIVER_MAX > u8Index) ? u8Index : EARLY_CONFIG_TOUCHDRIVER_FIRST;
   char _strDriverNameCur[DP_U8_KDSLEN_EARLYCONFIG_TOUCH_TOUCHDRIVERNAME];
   (tVoid)OSAL_pvMemorySet((tVoid*)_strDriverNameCur, OSAL_NULL, sizeof(_strDriverNameCur));
   if ((OSAL_NULL != cstrGetDriverName(_u8Idx)) && (0 < OSAL_u32StringLength(cstrGetDriverName(_u8Idx))))
   { // handle first driver or single request
	   OSAL_szStringNCopy(_strDriverNameCur, cstrGetDriverName(_u8Idx), sizeof(_strDriverNameCur)-1);
	   ETG_TRACE_USR2(("s32RestartDriver: restart touch driver %u (%64s) now.", _u8Idx, _strDriverNameCur ));
	   if (OSAL_OK == s32StopDriver(_strDriverNameCur))
	   {
		   _s32Return = s32StartDriver(_strDriverNameCur);
	   }
   }
   if (EARLY_CONFIG_TOUCHDRIVER_MAX <= u8Index)
   { // handle for all drivers
	   char _strDriverNameNext[sizeof(_strDriverNameCur)];
	   (tVoid)OSAL_pvMemorySet((tVoid*)_strDriverNameNext, OSAL_NULL, sizeof(_strDriverNameNext));
	   if (OSAL_NULL != cstrGetDriverName(EARLY_CONFIG_TOUCHDRIVER_SECOND))
		   OSAL_szStringNCopy(_strDriverNameNext, cstrGetDriverName(EARLY_CONFIG_TOUCHDRIVER_SECOND), sizeof(_strDriverNameNext)-1);
	   if ((0 < OSAL_u32StringLength(_strDriverNameNext)) && (0 != OSAL_s32StringCompare(_strDriverNameCur, _strDriverNameNext)))
	   {
		   ETG_TRACE_USR2(("s32RestartDriver: restart touch driver %u (%64s) now.", EARLY_CONFIG_TOUCHDRIVER_SECOND, _strDriverNameNext ));
		   if (OSAL_OK == s32StopDriver(_strDriverNameNext))
		   {
			   _s32Return = s32StartDriver(_strDriverNameNext);
		   }
	   }
   }
   return _s32Return;
}

/*******************************************************************************
*
* FUNCTION:     cstrGetDriverName()
*
* DESCRIPTION:  returns name of specified driver by index
*
* PARAMETER:    const tU8 u8Index
*
* RETURNVALUE:  const char* cstrDriverName
*
*******************************************************************************/
char _strDriverName[DP_U8_KDSLEN_EARLYCONFIG_TOUCH_TOUCHDRIVERNAME];
const char* dispvidctrl_tclTouchDriverControl::cstrGetDriverName(const tU8 u8Index) const
{
	ETG_TRACE_USR3(("cstrGetDriverName(index: %u) entered.", u8Index ));
	memset(_strDriverName, OSAL_NULL, sizeof(_strDriverName));
	switch (u8Index)
	{
	case EARLY_CONFIG_TOUCHDRIVER_FIRST:
		{
			dp_tclDimmingDPEarlyConfig_TouchDriverName_Manually _oDriverNameManual;
#ifdef VARIANT_S_FTR_ENABLE_AIVI_NPIVI
#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
			dp_tclEarlyConfigTwoTouchDriverTrTouchDriverName _oTouchDriverName;
#else
			dp_tclEarlyConfigTouchDriverTrTouchDriverName _oTouchDriverName;
#endif //#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
#else
			dp_tclEarlyConfigTouchDriverTrTouchDriverName _oTouchDriverName;
#endif //#ifdef VARIANT_S_FTR_ENABLE_AIVI_NPIVI
			tS32 _s32ReadStatus = _oDriverNameManual.s32GetData(_strDriverName, sizeof(_strDriverName)-1);
			if(0 == strlen(_strDriverName))
			{
				_s32ReadStatus = _oTouchDriverName.s32GetData(_strDriverName, sizeof(_strDriverName)-1);
			}
		}
		break;
	case EARLY_CONFIG_TOUCHDRIVER_SECOND:
		{
			dp_tclDimmingDPEarlyConfig_TouchDriverName2_Manually _oDriverNameManual;
			tS32 _s32ReadStatus = _oDriverNameManual.s32GetData(_strDriverName, sizeof(_strDriverName)-1);
#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
			dp_tclEarlyConfigTwoTouchDriverTrTouchDriverName2 _oTouchDriverName;
			if(0 == strlen(_strDriverName))
			{
				_s32ReadStatus = _oTouchDriverName.s32GetData(_strDriverName, sizeof(_strDriverName)-1);
			}
#else
#endif //#ifdef DP_U32_POOL_ID_EARLYCONFIGTWOTOUCHDRIVER
		}
		break;
	default:
		{
		}
		break;
	}
	ETG_TRACE_USR3(("cstrGetDriverName(index: %u): driver name = '%63s' ", u8Index, _strDriverName ));
	return _strDriverName;
}

/*******************************************************************************
*
* FUNCTION:     s32TraceStatusDriver()
*
* DESCRIPTION:  trace touch driver control status of specified index
*
* PARAMETER:    const tU8 u8Index
*
* RETURNVALUE:  tS32
*
*******************************************************************************/
tS32 dispvidctrl_tclTouchDriverControl::s32TraceStatusDriver(const tU8 u8Index) const
{
	ETG_TRACE_USR3(("s32TraceStatusDriver(index: %u) entered.", u8Index ));
	tChar _strIdentStart[30] = "DIM.APPL.DISP.TCTRL.START";
	tChar _strIdentAttr[30]  = "DIM.APPL.DISP.TCTRL.ATTR_";
	tChar _strIdentEnd[30]   = "DIM.APPL.DISP.TCTRL.END__";
	ETG_TRACE_USR2(("%026s: ", _strIdentStart ));
	ETG_TRACE_USR2(("%26s: %30s=%u ", _strIdentAttr,
		   "           DRV_INDEX", u8Index ));
	if ((tU8)sizeof(m_aenState) > u8Index)
	{
		ETG_TRACE_USR2(("%26s: %30s='%64s' (len: %d) ", _strIdentAttr,
			   "            DRV_NAME", cstrGetDriverName(u8Index), strlen(cstrGetDriverName(u8Index)) ));
		ETG_TRACE_USR2(("%26s: %30s=%u ", _strIdentAttr,
			   "           DRV_STATE", ETG_CENUM(tenTOUCH_DRV_STATE, m_aenState[u8Index]) ));
	}
	ETG_TRACE_USR2(("%026s: ", _strIdentEnd ));
	return OSAL_OK;
}

/*******************************************************************************
*
* FUNCTION:     s32StartDriver()
*
* DESCRIPTION:  start specified touch driver
*
* PARAMETER:    const char* cstrDriverName
*
* RETURNVALUE:  tS32
*
*******************************************************************************/
tS32 dispvidctrl_tclTouchDriverControl::s32StartDriver(const char* cstrDriverName)
{
   ETG_TRACE_USR3(("s32StartDriver(cstrDriverName: %64s) entered.", cstrDriverName ));
   tS32 _s32Return = OSAL_OK;

   char _strCmd[EARLY_CONFIG_TOUCHDRIVER_MAX_CMD_LENGHT];
   memset(_strCmd, OSAL_NULL, sizeof(_strCmd));
   snprintf(_strCmd, sizeof(_strCmd)-1,"%s %s", EARLY_CONFIG_TOUCHDRIVER_CMD_SYSTEMMODPROB, cstrDriverName);
   int _iSystemReturn = system(_strCmd);
   ETG_TRACE_USR1(("s32StartDriver(): system(%150s) returns %d", _strCmd, _iSystemReturn ));
   return _s32Return;
}

/*******************************************************************************
*
* FUNCTION:     s32StopDriver()
*
* DESCRIPTION:  stop specified touch driver
*
* PARAMETER:    const char* cstrDriverName
*
* RETURNVALUE:  tS32
*
*******************************************************************************/
tS32 dispvidctrl_tclTouchDriverControl::s32StopDriver(const char* cstrDriverName)
{
   ETG_TRACE_USR3(("s32StopDriver(cstrDriverName: %64s) entered.", cstrDriverName ));
   tS32 _s32Return = OSAL_OK;

   char _strCmd[EARLY_CONFIG_TOUCHDRIVER_MAX_CMD_LENGHT];
   memset(_strCmd, OSAL_NULL, sizeof(_strCmd));
   snprintf(_strCmd, sizeof(_strCmd)-1,"%s -r -q %s", EARLY_CONFIG_TOUCHDRIVER_CMD_SYSTEMMODPROB, cstrDriverName);
   int _iSystemReturn = system(_strCmd);
   ETG_TRACE_USR1(("s32StopDriver(): system(%150s) returns %d", _strCmd, _iSystemReturn ));
   return _s32Return;
}
