/*
 * dispvidctrl_tclLvdsMonitor.cpp
 *
 *  Created on: Sep 28, 2017
 *      Author: huf4abt, mav4abt
 */
/************************************************************************
* FILE:        dispvidctrl_tclLvdsMonitor.cpp
* PROJECT:
* SW-COMPONENT:vd_dimming
*----------------------------------------------------------------------
*
* DESCRIPTION: implementation of lvds monitor class
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date       | Author                  | Modification
* 28.09.2017 | BEG-VS/EEC3 Mayer       | init
*            | BEG-VS/EEC3 Hueckstaedt |
*
*************************************************************************/

/*******************************************************************************
| includes: system- and project- includes
|-----------------------------------------------------------------------------*/
// Basic OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

/*******************************************************************************
| set own module id
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| includes: needed interfaces from external components
|-----------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <libkmod.h>
#include <fstream>
#include "dispvidctrl_LvdsObserver.h"
#define VD_DIAGLOG_S_IMPORT_INTERFACE_MSG
#include "vd_diaglog_if.h"

/*******************************************************************************
| includes: internal and external interfaces from this component
|-----------------------------------------------------------------------------*/
#include "dispvidctrl_tclLvdsMonitor.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_APPLICATION
#include "trcGenProj/Header/dispvidctrl_tclLvdsMonitor.cpp.trc.h"
#endif

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#include "dp_drvass_video_if.h"


//                                    second display
const Variant Variant::AIVI = Variant(false);
const Variant Variant::PIVI = Variant(true);


/*******************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| typedefs (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function prototype (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementation (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
*
* FUNCTION:    dispvidctrl_tclLvdsMonitor()
*
* DESCRIPTION: constructor
*
* PARAMETER:   const dispvidctrl_tclAppMain*
*
* RETURNVALUE: none
*
*******************************************************************************/
dispvidctrl_tclLvdsMonitor::dispvidctrl_tclLvdsMonitor(const dispvidctrl_tclAppMain* poMainAppl)
: I_dispvidctrl_tclLvdsMonitor(poMainAppl)
, m_ObserverReady(FALSE)
, m_bObserverRunning(FALSE)
, bPrimaryScreen(FALSE)
, bSecondaryScreen(FALSE)
, bPrim_status(FALSE)
, bSec_status(FALSE)
, m_poEventObserver(OSAL_NULL)
, m_deviceId(eDEVICE_UNKNOWN)
, m_variant(OSAL_NULL)
, OrientationType(0xFF)
, DisplayOutputType(0x00)
, m_poClientDiaglogIf(OSAL_NULL)
{
   ETG_TRACE_USR1(("dispvidctrl_tclLvdsMonitor() entered."));
   memset((tVoid*)m_dispState, (tU8) midw_fi_tcl_e8_TestResult::FI_EN_NORESULT, sizeof(m_dispState));
   memset((tVoid*)m_TouchControllerState, (tU8) midw_fi_tcl_e8_TestResult::FI_EN_NORESULT, sizeof(m_TouchControllerState));

   if ( OSAL_s32TimerCreate( (OSAL_tpfCallback)_pfCallbackTimer_EventObserverInit, (tPVoid)this, &m_hTimerHandle_EventObserverInit) != OSAL_OK)
   {
      ETG_TRACE_FATAL(("ERROR - Could not create EventObserverInit timer!"));
      m_hTimerHandle_EventObserverInit = OSAL_C_INVALID_HANDLE;
   }
}

/*******************************************************************************
*
* FUNCTION:    ~dispvidctrl_tclLvdsMonitor()
*
* DESCRIPTION: destructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
dispvidctrl_tclLvdsMonitor::~dispvidctrl_tclLvdsMonitor()
{
   ETG_TRACE_USR1(("~dispvidctrl_tclLvdsMonitor() entered."));

   m_poClientDiaglogIf = OSAL_NULL;
}

/*******************************************************************************
*
* FUNCTION:    vHandleMessage()
*
* DESCRIPTION: handle internal message
*
* PARAMETER:   dispvidctrl_tclBaseIf::TMsg* pMsg
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid dispvidctrl_tclLvdsMonitor::vHandleMessage(dispvidctrl_tclBaseIf::TMsg* pMsg)
{
      ETG_TRACE_USR1(("[%d ms] dispvidctrl_tclLvdsMonitor::vHandleMessage() entered %u -> data: %d.", \
                       OSAL_ClockGetElapsedTime(), ETG_CENUM(dispvidctrl_tclBaseIf::ECmdTypes , (tU32)pMsg->eCmd), pMsg->u.u32Data));

      switch (pMsg->eCmd)
       {

          case dispvidctrl_tclBaseIf::eTriggerRecheckITC:
             ETG_TRACE_USR4(("resend ITC!"));

             // reset ITC states to force resend
             memset((tVoid*)m_dispState, (tU8) midw_fi_tcl_e8_TestResult::FI_EN_NORESULT, sizeof(m_dispState));
             memset((tVoid*)m_TouchControllerState, (tU8) midw_fi_tcl_e8_TestResult::FI_EN_NORESULT, sizeof(m_TouchControllerState));
             _vEvaluateAndUpdateITCStatus();
             _vEvaluateAndUpdateControllerITCStatus();
             break;
       
          case dispvidctrl_tclBaseIf::eControlFap:
             if (pMsg->u.u32Data == 1)
             {
                if (m_deviceId != eDEVICE_UNKNOWN) {
                    m_ObserverReady = TRUE;
                    vStartStopObserver();
                    if(m_poEventObserver) {
                        m_poEventObserver->startMonitorThread();
                    }
                }
             }
             break;

          default:
             break;
       }
}

/*******************************************************************************
*
* FUNCTION:       vHandleTraceMessage()
*
* DESCRIPTION:    handler for trace command
*
* PARAMETER:      const tUChar* puchData: reference of received message
*
* RETURNVALUE:    None.
*
*******************************************************************************/
tVoid dispvidctrl_tclLvdsMonitor::vHandleTraceMessage(const tUChar* puchData)
{
   ETG_TRACE_USR1(("vHandleTraceMessage() entered."));
}

/*******************************************************************************
*
* FUNCTION:    vGetReferences(tVoid)
*
* DESCRIPTION:    Function to get all reference needed by this class.
*             A reference should always be the Interface class of the object
*
* PARAMETER:    None.
*
* RETURNVALUE:    None.
*
*******************************************************************************/
tVoid dispvidctrl_tclLvdsMonitor::vGetReferences(tVoid)
{
   ETG_TRACE_USR4(("dispvidctrl_tclLvdsMonitor::vGetReferences() entered."));

   m_poEventObserver = dynamic_cast<dispvidctrl_tclLvdsObserver*>(_cpoMain->getHandler("dispvidctrl_tclLvdsObserver"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poEventObserver);

   m_poClientDiaglogIf = dynamic_cast<Idispvidctrl_tclClientDiaglogIf*>(_cpoMain->getHandler("Idispvidctrl_tclClientDiaglogIf"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poClientDiaglogIf)


   //DEBUG CALL
   //_vRegisterFiles(_setDeviceId());


}

/*******************************************************************************
*
* FUNCTION:    tVoid vStartCommunication()
*
* DESCRIPTION:    Function to start all dynamic objects e.g. threads, ...
*
* PARAMETER:    None.
*
* RETURNVALUE:    None.
*
*******************************************************************************/
tVoid dispvidctrl_tclLvdsMonitor::vStartCommunication(tVoid)
{
   ETG_TRACE_USR4(("vStartCommunication() entered."));
   if ( m_hTimerHandle_EventObserverInit != OSAL_C_INVALID_HANDLE )
   {
      OSAL_s32TimerSetTime(m_hTimerHandle_EventObserverInit, 10000, 0);
   }
   _vRegisterFiles(_setDeviceId());
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfo()
*
* DESCRIPTION:    Trace information
*
* PARAMETER:    None.
*
* RETURNVALUE:    None.
*
*******************************************************************************/
tVoid dispvidctrl_tclLvdsMonitor::vTraceInfo()
{
   ETG_TRACE_USR4(("vTraceInfo() entered."));
}



/******************************************************************************/
/* FUNCTION     _vUpdateITCStatus()                                           */
/******************************************************************************/
/**
*  \brief       send itc status for lvds connectivity
*
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclLvdsMonitor::_vUpdateITCStatus(eDISPLAY_STATE displayType, tenITC_ID enITC_ID, tU8 itcResult)
{
    if (m_poClientDiaglogIf) {
        ETG_TRACE_USR4(("_vUpdateITCStatus entered." ));
    
        // only send if test result has changed
        if(m_dispState[displayType] != itcResult){
           ETG_TRACE_USR4(("sending ITC" ));
           m_poClientDiaglogIf->vSetITC(enITC_ID , static_cast<midw_fi_tcl_e8_TestResult::tenType>(itcResult));
           m_dispState[displayType] = itcResult;
        }else{
           ETG_TRACE_USR4(("not sending ITC! no change" ));
        }
    }
}

/******************************************************************************/
/* FUNCTION     _vUpdateControllerITCStatus()                                           */
/******************************************************************************/
/**
*  \brief       send itc status for touch controller
*
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclLvdsMonitor::_vUpdateControllerITCStatus(eDISPLAY_STATE displayType, tenITC_ID enITC_ID, tU8 itcResult)
{
    if (m_poClientDiaglogIf) {
        ETG_TRACE_USR4(("_vUpdateControllerITCStatus entered." ));

        // only send if test result has changed
        if(m_TouchControllerState[displayType] != itcResult){
           ETG_TRACE_USR4(("sending ITC for touch controller" ));
           m_poClientDiaglogIf->vSetITC(enITC_ID , static_cast<midw_fi_tcl_e8_TestResult::tenType>(itcResult));
           m_TouchControllerState[displayType] = itcResult;
        }else{
           ETG_TRACE_USR4(("not sending ITC! no change" ));
        }
    }
}

/******************************************************************************/
/* FUNCTION     _vEvaluateAndUpdateITCStatus()                                         */
/******************************************************************************/
/**
*  \brief       evaluate ITC status for primary and secondary screen
*
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclLvdsMonitor::_vEvaluateAndUpdateITCStatus()
{


   ETG_TRACE_USR4(("_vEvaluateAndUpdateITCStatus entered." ));


   if (m_deviceId != eDEVICE_UNKNOWN) {

      // evaluate primary screen
      bool bPrimaryDisplayFailure = !bPrimaryScreen;

      tU8 itcResult = bPrimaryDisplayFailure ? (tU8) midw_fi_tcl_e8_TestResult::FI_EN_FAILED : (tU8) midw_fi_tcl_e8_TestResult::FI_EN_PASSED;
      ETG_TRACE_USR4(("sending ITC %u for primary display", ETG_CENUM(midw_fi_tcl_e8_TestResult::enType , itcResult)));
      _vUpdateITCStatus(eDISPLAY_PRIMARY, enITC_ID_LVDS_OUT_CP_LINE_COMM_FAILURE, itcResult);


      // evaluate secondary screen
      if (m_variant->secDisp) {
         bool bSecondaryDisplayFailure = !bSecondaryScreen;

         tU8 itcResult = bSecondaryDisplayFailure ? (tU8) midw_fi_tcl_e8_TestResult::FI_EN_FAILED : (tU8) midw_fi_tcl_e8_TestResult::FI_EN_PASSED;
         ETG_TRACE_USR4(("sending ITC %u for secondary display", ETG_CENUM(midw_fi_tcl_e8_TestResult::enType , itcResult)));
         _vUpdateITCStatus(eDISPLAY_SECONDARY, enITC_ID_LVDS_OUT_SECOND_DISPLAY_COMM_FAILURE, itcResult);

      }
   }
}

/******************************************************************************/
/* FUNCTION     _vEvaluateAndUpdateControllerITCStatus()                                         */
/******************************************************************************/
/**
*  \brief       evaluate ITC status for Touch Controller primary and secondary screen(PIVI and Renault)
*
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclLvdsMonitor::_vEvaluateAndUpdateControllerITCStatus()
{
    ETG_TRACE_USR4(("_vEvaluateAndUpdateControllerITCStatus entered." ));

    switch(m_deviceId)
    {
        case eDEVICE_AIVI:
        {
            // evaluate primary screen
            bool bPrimaryDisplayFailure = !bPrim_status;

            tU8 itcResult = bPrimaryDisplayFailure ? (tU8) midw_fi_tcl_e8_TestResult::FI_EN_FAILED : (tU8) midw_fi_tcl_e8_TestResult::FI_EN_PASSED;
            ETG_TRACE_USR4(("sending ITC %u for primary display", ETG_CENUM(midw_fi_tcl_e8_TestResult::enType , itcResult)));
            _vUpdateControllerITCStatus(eDISPLAY_PRIMARY, enITC_ID_LVDS_OUT_CP_LINE_I2C_TIMEOUT, itcResult);
        }
        break;
        case eDEVICE_PIVI:
        {
            // evaluate primary screen
            bool bPrimaryDisplayFailure = !bPrim_status;

            tU8 itcResult = bPrimaryDisplayFailure ? (tU8) midw_fi_tcl_e8_TestResult::FI_EN_FAILED : (tU8) midw_fi_tcl_e8_TestResult::FI_EN_PASSED;
            ETG_TRACE_USR4(("sending ITC %u for primary display", ETG_CENUM(midw_fi_tcl_e8_TestResult::enType , itcResult)));
            _vUpdateControllerITCStatus(eDISPLAY_PRIMARY, enITC_ID_LVDS_OUT_SECOND_DISPLAY_I2C_TIMEOUT, itcResult);
            // evaluate secondary screen
            {
            bool bSecondaryDisplayFailure = !bSec_status;

            tU8 itcResult = bSecondaryDisplayFailure ? (tU8) midw_fi_tcl_e8_TestResult::FI_EN_FAILED : (tU8) midw_fi_tcl_e8_TestResult::FI_EN_PASSED;
            ETG_TRACE_USR4(("sending ITC %u for secondary display", ETG_CENUM(midw_fi_tcl_e8_TestResult::enType , itcResult)));
            _vUpdateControllerITCStatus(eDISPLAY_SECONDARY, enITC_ID_LVDS_OUT_CP_LINE_I2C_TIMEOUT, itcResult);
            }
        }
        break;
    }
}

/******************************************************************************/
/* FUNCTION     _setDeviceId()                                           */
/******************************************************************************/
/**
*  \brief       Check which device is in use to determine which nodes to use
*               for the observer.
*
*  \param
*  \return      none
*/
/******************************************************************************/
eDEVICE_ID dispvidctrl_tclLvdsMonitor::_setDeviceId()
{
   //Check which file(s) is/are available
   ETG_TRACE_FATAL(("dispvidctrl_tclLvdsMonitor::_setDeviceId() entered."));
   std::ifstream fileNode;

#ifdef VARIANT_S_FTR_ENABLE_AIVI_NPIVI
   m_deviceId = eDEVICE_PIVI;
   m_variant = &Variant::PIVI;
#else
   m_deviceId = eDEVICE_AIVI;
   m_variant = &Variant::AIVI;
#endif

   if (m_deviceId == eDEVICE_AIVI)
   {

      /* read configurations from System Information Common KDS Entry  via Data-pool from KDS
       *  - DisplayOutput
       *
       * 0 = Internal Display
       * 1 = External Display
       */
      tS32  s32ReturnValue;
      tU8   u8BufferGet[1] = {0};

      // DisplayOutput
      s32ReturnValue = DP_s32GetConfigItem("AIVIVariantCoding","DisplayOutput",u8BufferGet, 1, FALSE);
      if (s32ReturnValue == DP_S32_NO_ERR) {
         tU8 u8DisplayOutput = u8BufferGet[0];
         ETG_TRACE_FATAL(("dispvidctrl_tclLvdsMonitor::_setDeviceId: got DisplayOutput %i from configuration", u8DisplayOutput));
         if (u8DisplayOutput == 0) {
            m_deviceId = eDEVICE_UNKNOWN;
         }
         DisplayOutputType = u8DisplayOutput;    //FOr renault landscape detached screen identification
      }
      else {
         ETG_TRACE_FATAL((" dispvidctrl_tclLvdsMonitor :: E R R O R in _setDeviceId(): while requesting AIVIVariantCoding -> DisplayOutput: %-5d", s32ReturnValue));
         // use default values
      }

      fileNode.open(LVDS_MONITOR_AIVI_DISPLAY_SERIALIZER_NODE);
      if (fileNode.fail()) {
         m_deviceId = eDEVICE_UNKNOWN;
         ETG_TRACE_FATAL(("dispvidctrl_tclLvdsMonitor::_setDeviceId(): serializer node not present. setting device to UNKNOWN"));
      }
   }

   fileNode.close();
   ETG_TRACE_USR3(("Device set to: %i", m_deviceId ));
   return m_deviceId;
}

tBool dispvidctrl_tclLvdsMonitor::isRenaultLandscapeDetachedType()
{
    tBool bStatus =FALSE;
    if(DisplayOutputType == 1 && OrientationType == 0)	// AIVI+External Display and Landscape -> For Renault Landscape detached
    {
        bStatus = TRUE;
    }
    return bStatus;
}
tU8 dispvidctrl_tclLvdsMonitor::getDisplayOutputType()
{
     /* read configurations from System Information Common KDS Entry  via Data-pool from KDS
           *  - DisplayOutput
           *
           * 0 = Internal Display
           * 1 = External Display
           */
    tU8 u8DisplayOutput = 0xFF;
    if (DP_S32_NO_ERR == DP_s32GetConfigItem("AIVIVariantCoding","DisplayOutput",&u8DisplayOutput, 1)) {

       ETG_TRACE_FATAL(("dispvidctrl_tclLvdsMonitor::_setDeviceId: got DisplayOutput %i from configuration", u8DisplayOutput));
    }
    //DisplayOutputType = u8DisplayOutput;
    return u8DisplayOutput;
}
tU8 dispvidctrl_tclLvdsMonitor::getDisplayOrientationType()
{
/* Display Orientation/Format :  Portrait -> 0x01
                                 Landscape-> 0x00
*/
    tU8 u8Orientation=0xFF;
    //   DP_S32_NO_ERR == DP_s32GetConfigItem("AIVIVariantCoding", "DisplayOrientation", &u8Orientation, 1) -> CD config
    if ( DP_S32_NO_ERR == DP_s32GetConfigItem("HMIConfig", "DisplayFormat", &u8Orientation, 1) )
    {
       ETG_TRACE_USR4(("getDisplayOrientationType(): KDS Read:Display Orienation is 0x%02x.", u8Orientation));
    }
    else
    {
       ETG_TRACE_USR4(("getDisplayOrientationType(): KDS Read:Display Orienation ERROR."));
    }
    OrientationType = u8Orientation;
    return u8Orientation;
}

/******************************************************************************/
/* FUNCTION     _vRegisterFiles()                                           */
/******************************************************************************/
/**
*  \brief       Register files to the observer.
*
*  \param
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclLvdsMonitor::_vRegisterFiles(eDEVICE_ID pDisplayId)
{
    ETG_TRACE_USR3(("Registering file for: %i", pDisplayId));
    switch(pDisplayId)
    {
        case eDEVICE_AIVI:
        {
            m_poEventObserver->addFileToMonitor(const_cast<char*>(LVDS_MONITOR_AIVI_DISPLAY_SERIALIZER_NODE));
            if(getDisplayOrientationType() == 0x01) {
                m_poEventObserver->addFileToMonitor(const_cast<char*>(TOUCH_CONTROLLER_STATUS_PIVI_PRIMARY_PATH), 1);    //For Portrait
            } else {
                if(getDisplayOutputType() == 0x01)
                {
                    m_poEventObserver->addFileToMonitor(const_cast<char*>(TOUCH_CONTROLLER_STATUS_RENAULT_LANDSCAPE_DETACHED_PATH), 1);
                }
                else
                {
                    m_poEventObserver->addFileToMonitor(const_cast<char*>(TOUCH_CONTROLLER_STATUS_AIVI_PATH), 1);
                }
            }
            ETG_TRACE_USR3(("dispvidctrl_tclLvdsMonitor::_vRegisterFiles: Registered AIVI device."));
            break;
        }
        case eDEVICE_PIVI:
        {
            m_poEventObserver->addFileToMonitor(const_cast<char*>(LVDS_MONITOR_PIVI_PRIMARY_DISPLAY_SERIALIZER_NODE));
            m_poEventObserver->addFileToMonitor(const_cast<char*>(LVDS_MONITOR_PIVI_SECONDARY_DISPLAY_SERIALIZER_NODE));
            m_poEventObserver->addFileToMonitor(const_cast<char*>(TOUCH_CONTROLLER_STATUS_PIVI_PRIMARY_PATH), 1);
            m_poEventObserver->addFileToMonitor(const_cast<char*>(TOUCH_CONTROLLER_STATUS_PIVI_SECONDARY_PATH), 1);

            ETG_TRACE_USR3(("dispvidctrl_tclLvdsMonitor::_vRegisterFiles: Registered PIVI device."));
            break;
        }
        default:
            break;
    }
}

OSAL_tpfCallback dispvidctrl_tclLvdsMonitor::_pfCallbackTimer_EventObserverInit(tVoid* pArg)
{
   ETG_TRACE_USR4(("dispvidctrl_tclLvdsMonitor::_pfCallbackTimer_EventObserverInit(): ... ====>>    "));

   dispvidctrl_tclLvdsMonitor* po_tclControl = (dispvidctrl_tclLvdsMonitor*)pArg;
   if (po_tclControl)
   {
      if ((po_tclControl->_cpoMain) && (po_tclControl->_cpoMain->theServer()))
      {
         // post loop-back message to control _Hmi
         tChar strIfName[] = "dispvidctrl_tclLvdsMonitor";
         po_tclControl->_cpoMain->theServer()->vPostMsgControlFap(strIfName, 1);
      }
   }
   return 0;
}

tVoid dispvidctrl_tclLvdsMonitor::_vUpdateValues(tBool bResult[]) {
    switch(m_deviceId)
    {
    case eDEVICE_AIVI:
        if (bResult[eDISPLAY_PRIMARY] != bPrimaryScreen) {
            bPrimaryScreen = bResult[eDISPLAY_PRIMARY];
            ETG_TRACE_USR3(("dispvidctrl_tclLvdsMonitor::vUpdateValues: Value of primary screen was changed to: %s", bPrimaryScreen ? "true" : "false"));
            _vEvaluateAndUpdateITCStatus();
        }
        break;
    case eDEVICE_PIVI:
        if (bResult[eDISPLAY_PRIMARY] != bPrimaryScreen) {
            bPrimaryScreen = bResult[eDISPLAY_PRIMARY];
            ETG_TRACE_USR3(("dispvidctrl_tclLvdsMonitor::vUpdateValues: Value of primary screen was changed %s.", bPrimaryScreen ? "true" : "false"));
            _vEvaluateAndUpdateITCStatus();
        }
        if (bResult[eDISPLAY_SECONDARY] != bSecondaryScreen) {
            bSecondaryScreen = bResult[eDISPLAY_SECONDARY];
            ETG_TRACE_USR3(("dispvidctrl_tclLvdsMonitor::vUpdateValues: Value of secondary screen was changed %s.",  bSecondaryScreen ? "true" : "false"));
            _vEvaluateAndUpdateITCStatus();
        }
        break;
    default:
        break;
    }
}

tVoid dispvidctrl_tclLvdsMonitor::_vUpdateTouchValues(tBool bResult[]) {
    ETG_TRACE_USR4(("dispvidctrl_tclLvdsMonitor::_vUpdateTouchValues->Entered Variant =(%d)",m_deviceId));
    switch(m_deviceId)
    {
    case eDEVICE_AIVI:
        if (bResult[eDISPLAY_PRIMARY] != bPrim_status) {
            bPrim_status = bResult[eDISPLAY_PRIMARY];
            ETG_TRACE_USR4(("dispvidctrl_tclLvdsMonitor::_vUpdateTouchValues: Value of primary screen touch controller was changed to: %s", bPrim_status ? "true" : "false"));
            _vEvaluateAndUpdateControllerITCStatus();
        }
        break;
    case eDEVICE_PIVI:
        if (bResult[eDISPLAY_PRIMARY] != bPrim_status) {
            bPrim_status = bResult[eDISPLAY_PRIMARY];
            ETG_TRACE_USR3(("dispvidctrl_tclLvdsMonitor::_vUpdateTouchValues: Value of primary screen touch controller was changed to %s.", bPrim_status ? "true" : "false"));
            _vEvaluateAndUpdateControllerITCStatus();
        }
        if (bResult[eDISPLAY_SECONDARY] != bSec_status) {
            bSec_status = bResult[eDISPLAY_SECONDARY];
            ETG_TRACE_USR3(("dispvidctrl_tclLvdsMonitor::_vUpdateTouchValues: Value of secondary screen touch controller was changed to %s.",  bSec_status ? "true" : "false"));
            _vEvaluateAndUpdateControllerITCStatus();
        }
        break;
    default:
        break;
    }
}

tVoid dispvidctrl_tclLvdsMonitor::vStartStopObserver(tVoid)
{
   ETG_TRACE_USR4(("dispvidctrl_tclLvdsMonitor::vStartStopObserver() entered"));

   if (m_poEventObserver != NULL)
   {
      if (m_ObserverReady)
      {
         if (!m_bObserverRunning)
         {
            //m_poEventObserver->startMonitorThread();
            tS32 s32StartObserver = m_poEventObserver->startObserver();
            ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclLvdsMonitor::vStartStopObserver() startObserver %d", OSAL_ClockGetElapsedTime(), s32StartObserver));
            if (0 == s32StartObserver) {
                m_bObserverRunning = TRUE;
            }
         }
      }
      else // if no condition to keep the observer is active, then Stop it.
      {
         ETG_TRACE_FATAL(("dispvidctrl_tclLvdsMonitor::vStartStopObserver: Observer not Ready!"));
         if ( m_bObserverRunning )
         {
            m_poEventObserver->stopObserver();
            ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclLvdsMonitor::vStartStopObserver() stopObserver", OSAL_ClockGetElapsedTime()));
            m_bObserverRunning = FALSE;
         }
      }
   } else {
      ETG_TRACE_FATAL(("dispvidctrl_tclLvdsMonitor::vStartStopObserver: EventObserver is Null!"));
   }
   OSAL_s32TimerSetTime(m_hTimerHandle_EventObserverInit, 10000, 0);
}
