/************************************************************************
* FILE:        dispvidctrl_tclDisplaySettings.cpp
* PROJECT:
* SW-COMPONENT:vd_dimming
*----------------------------------------------------------------------
*
* DESCRIPTION: implementation of display settings class
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 17.11.2015| CM-AI/EPB2 Bernard | init
*
*************************************************************************/

/*******************************************************************************
| includes: system- and project- includes
|-----------------------------------------------------------------------------*/
// Basic OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#include "I_dispvidctrl_tclSyncHandler.h"
#include "dispvidctrl_ClientDimming.h"

/*******************************************************************************
| set own module id
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| includes: needed interfaces from external components
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| includes: internal and external interfaces from this component
|-----------------------------------------------------------------------------*/

#include "dispvidctrl_tclDisplaySettings.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_APPLICATION
#include "trcGenProj/Header/dispvidctrl_tclDisplaySettings.cpp.trc.h"
#endif

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#include "dp_drvass_video_if.h"

/*******************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------------*/
// device name for color space conversion and gamma correction as part of DRM
#define DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL                       (DISPVIDCTRL_C_U8_LEVEL_MAX)
#define DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_DEGREE_MIN                  (-180)
#define DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_DEGREE_MAX                   (180)
#define DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MIN                    (0)
#define DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MAX                  (100)
// Hue
#define DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_HUE_LEVEL_DEFAULT            (DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL/2)
#define DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_HUE_MIN                      (-45)
#define DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_HUE_MAX                       (45)
//#define DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_HUE_OFFSET_MIN               (-45)
//#define DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_HUE_OFFSET_MAX                (45)
// Saturation
#define DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_SATURATION_LEVEL_DEFAULT     (DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL)
#define DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_SATURATION_MIN                 (0)
#define DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_SATURATION_MAX               (100)
//#define DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_SATURATION_OFFSET_MIN          (0)
//#define DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_SATURATION_OFFSET_MAX        (100)
// Brightness
#define DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_BRIGHTNESS_MAX               (100)
#define DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_BRIGHTNESS_MIN               (DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_BRIGHTNESS_MAX - 40)
#define DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_BRIGHTNESS_OFFSET_MIN          (0)
#define DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_BRIGHTNESS_OFFSET_MAX         (15)
// Contrast
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
#define DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_CONTRAST_LEVEL_DEFAULT       (DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL)
#else
#define DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_CONTRAST_LEVEL_DEFAULT       (DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL/2)
#endif //#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
#define DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_CONTRAST_MIN                  (50)
#define DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_CONTRAST_MAX                 (100)
// Gamma exponent ... 1000 = 1.000
#define DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_GAMMA_LEVEL_DEFAULT          (DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL/2)
#define DISPVIDCTRL_DIM_C_U16_DISPLAYSETTINGS_GAMMA_EXPONENT_DEFAULT      (1300)
#define DISPVIDCTRL_DIM_C_U16_DISPLAYSETTINGS_GAMMA_EXPONENT_RANGE_MIN     (500)
#define DISPVIDCTRL_DIM_C_U16_DISPLAYSETTINGS_GAMMA_EXPONENT_RANGE_MAX    (2500)
#define DISPVIDCTRL_DIM_C_U16_DISPLAYSETTINGS_GAMMA_EXPONENT_MIN           (800)
#define DISPVIDCTRL_DIM_C_U16_DISPLAYSETTINGS_GAMMA_EXPONENT_MAX          (1800)
// Direct rendering manager display type
#define DISPVIDCTRL_DIM_C_S32_DISPLAYSETTINGS_DRM_DISPLAYTYPE_DEFAULT     (DRM_MODE_CONNECTOR_Unknown)
// Blacklevel
#define DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_BLACKLEVEL_DEFAULT             (DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL/2)

//FMVSS changes for brightness default = 11 for camera screen

#define FMVSS_BRIGHTNESS_DEFAULT 	11
/*******************************************************************************
| typedefs (scope: module-local)
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------------*/
#ifdef VARIANT_S_FTR_ENABLE_BRIGHTNESS_ENHANCEMENT
typedef struct
{
	tS16 s16DayDeciPercent;
	tS16 s16NightDeciPercent;
}tEnhanceBrightnesOffset;
const tEnhanceBrightnesOffset m_atCSCEnhanceBrigtnessOffset[]= {
		// DAY      NIGHT     Extended Brightness Level
		{    0,     0 },      //  0
		{    0,     0 },      //  1
		{    0,     0 },      //  2
		{    0,     0 },      //  3
		{    0,     0 },      //  4
		{    0,     0 },      //  5
		{    0,     0 },      //  6
		{    0,     0 },      //  7
		{    0,     0 },      //  8
		{    0,     0 },      //  9
		{    0,     0 },      // 10
		{    0,     0 },      // 11
		{    0,     0 },      // 12
		{    0,     0 },      // 13
		{   10,     0 },      // 14
		{   20,     0 },      // 15
		{   30,     0 },      // 16
		{   40,     0 },      // 17
		{   50,     0 },      // 18
		{   60,     0 },      // 19
		{   70,     0 },      // 20
		{  150,     0 },      // 21
		{  150,     0 },      // 22
		{  150,     0 },      // 23
		{  150,     0 },      // 24
		{  150,     0 },      // 25
		{  150,     0 },      // 26
		{  150,     0 },      // 27
		{  150,     0 },      // 28
		{  150,     0 },      // 29
		{  150,     0 },      // 30
		{  150,     0 },      // 31
		{  150,     0 }       // 32
};
#endif //#ifdef VARIANT_S_FTR_ENABLE_BRIGHTNESS_ENHANCEMENT

/*******************************************************************************
| function prototype (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementation (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
*
* FUNCTION:    dispvidctrl_tclDisplaySettings()
*
* DESCRIPTION: constructor
*
* PARAMETER:   const dispvidctrl_tclAppMain*
*
* RETURNVALUE: none
*
*******************************************************************************/
dispvidctrl_tclDisplaySettings::dispvidctrl_tclDisplaySettings(const dispvidctrl_tclAppMain* poMainAppl)
: I_dispvidctrl_tclDisplaySettings(poMainAppl)
{
   ETG_TRACE_USR4(("dispvidctrl_tclDisplaySettings() entered."));

#ifdef VARIANT_S_FTR_ENABLE_BRIGHTNESS_ENHANCEMENT
   m_u8DimMode = 0;
   (tVoid)memset((tVoid*)m_au8DimBrightnessLevel, OSAL_NULL, sizeof(m_au8DimBrightnessLevel));
#endif //#ifdef VARIANT_S_FTR_ENABLE_BRIGHTNESS_ENHANCEMENT
   m_DayBrightness = 10;
   m_NightBrightness = 10;
   m_u8ContrastLevel = DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_CONTRAST_LEVEL_DEFAULT;
   m_u8SaturationLevel = DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_SATURATION_LEVEL_DEFAULT;
   m_u8ColorLevel = DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_HUE_LEVEL_DEFAULT;
   m_u8GammaLevel = DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_GAMMA_LEVEL_DEFAULT;
   m_u8BlackLevel = DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_BLACKLEVEL_DEFAULT;

   m_poSrvDim = OSAL_NULL;
   m_poSyncHandler = OSAL_NULL;
   m_poClientDim= OSAL_NULL; 
   m_bCscInitDone = FALSE;
   
   m_bILMInit = FALSE;
#ifdef VARIANT_S_FTR_ENABLE_AIVI_NPIVI
   m_u8DRM_MainHmi_ScreenID = 2;
#else
   m_u8DRM_MainHmi_ScreenID = 1;
#endif // #ifdef VARIANT_S_FTR_ENABLE_AIVI_NPIVI
  

#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   (tVoid)memset((tVoid*)&m_tCSCProp, OSAL_NULL, sizeof(m_tCSCProp));
   m_s16BrightnessOffset = DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_BRIGHTNESS_OFFSET_MIN;
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC

#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_GAMMA
   m_u16DRM_GammaExponent = DISPVIDCTRL_DIM_C_U16_DISPLAYSETTINGS_GAMMA_EXPONENT_DEFAULT;
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_GAMMA

   m_bDisplaySettingsInitDone = FALSE;
}

/*******************************************************************************
*
* FUNCTION:    ~dispvidctrl_tclDisplaySettings()
*
* DESCRIPTION: destructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
dispvidctrl_tclDisplaySettings::~dispvidctrl_tclDisplaySettings()
{
   ETG_TRACE_USR4(("~dispvidctrl_tclDisplaySettings() entered."));

#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC

#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_GAMMA
#endif //#ifndef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_GAMMA_USE_ILM

   m_poSrvDim = OSAL_NULL;
   m_poSyncHandler = OSAL_NULL;
}

/*******************************************************************************
*
* FUNCTION:    vHandleMessage()
*
* DESCRIPTION: handle internal message
*
* PARAMETER:   dispvidctrl_tclBaseIf::TMsg* pMsg
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid dispvidctrl_tclDisplaySettings::vHandleMessage(dispvidctrl_tclBaseIf::TMsg* pMsg)
{
      DISPVIDCTRL_NULL_POINTER_CHECK(pMsg);
      ETG_TRACE_USR4(("vHandleMessage() entered %u -> data: %d.", ETG_ENUM(HC_INT_MSG , (tU32)pMsg->eCmd), pMsg->u.u32Data));
}

/*******************************************************************************
*
* FUNCTION:    vHandleTraceMessage()
*
* DESCRIPTION:    handler for trace command
*
* PARAMETER:      const tUChar* puchData: reference of received message
*
* RETURNVALUE:    None.
*
*******************************************************************************/
tVoid dispvidctrl_tclDisplaySettings::vHandleTraceMessage(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceMessage() entered (data: 0x%08x).", puchData ));
   DISPVIDCTRL_NULL_POINTER_CHECK(puchData);

    tU32 u32MsgCode = (puchData[1]<<8) | puchData[2];
   switch (u32MsgCode)
   {
   case DISPVIDCTRL_DIMMING_GETDISPLAYSETTINGS:
      {
         // Prints the display setting information.
         ETG_TRACE_FATAL(("%26s: DISPVIDCTRL_CmdGetDisplaySettings ",
               "DIM.APPL.DISP.CMD__.START" ));

         vTraceInfo();

         ETG_TRACE_FATAL(("%26s: finished without errors ",
               "DIM.APPL.DISP.CMD__.RES__" ));
      }
      break;
   case DISPVIDCTRL_DIMMING_RESETDISPLAYSETTINGS:
      {
            // Resets display settings properties and states to default values.
         ETG_TRACE_FATAL(("%26s: DISPVIDCTRL_DIMMING_ResetDisplaySettings ",
               "DIM.APPL.DISP.CMD__.START" ));

         if (0 <= s32ResetDisplaySettings())
         {
            ETG_TRACE_FATAL(("%26s: finished without errors ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
         else
         {
            ETG_TRACE_FATAL(("%26s: finished with errors! ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
      }
      break;
   case DISPVIDCTRL_DIMMING_SETCSCCONTRASTLEVEL:
      {
            tU8 param1 = puchData[3];

            // Set the contrast level (0 ... 32) of color space conversion.
         ETG_TRACE_FATAL(("%26s: DISPVIDCTRL_DIMMING_SetCSCContrastLevel %d ",
               "DIM.APPL.DISP.CMD__.START", param1 ));
         if (0 <= s32SetContrastLevel(param1))
         {
            ETG_TRACE_FATAL(("%26s: finished without errors ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
         else
         {
            ETG_TRACE_FATAL(("%26s: finished with errors! ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
      }
      break;
   case DISPVIDCTRL_DIMMING_SETCSCCONTRAST:
      {
            tU8 param1 = puchData[3];

            // Set the contrast in percent (0 ... 100) of color space conversion.
         ETG_TRACE_FATAL(("%26s: DISPVIDCTRL_DIMMING_SetCSCContrast %d ",
               "DIM.APPL.DISP.CMD__.START", param1 ));
         if (0 <= s32SetContrastCSC((tS16)param1))
         {
            ETG_TRACE_FATAL(("%26s: finished without errors ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
         else
         {
            ETG_TRACE_FATAL(("%26s: finished with errors! ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
      }
      break;
   case DISPVIDCTRL_DIMMING_SETCSCSATURATIONLEVEL:
      {
            tU8 param1 = puchData[3];

            // Set the saturation level (0 ... 32) of color space conversion.
         ETG_TRACE_FATAL(("%26s: DISPVIDCTRL_DIMMING_SetCSCSaturationLevel %d ",
               "DIM.APPL.DISP.CMD__.START", param1 ));
         if (0 <= s32SetSaturationLevel(param1))
         {
            ETG_TRACE_FATAL(("%26s: finished without errors ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
         else
         {
            ETG_TRACE_FATAL(("%26s: finished with errors! ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
      }
      break;
   case DISPVIDCTRL_DIMMING_SETCSCSATURATION:
      {
            tU8 param1 = puchData[3];

            // Set the saturation in percent (0 ... 100) of color space conversion.
         ETG_TRACE_FATAL(("%26s: DISPVIDCTRL_DIMMING_SetCSCSaturation %d ",
               "DIM.APPL.DISP.CMD__.START", param1 ));
         if (0 <= s32SetSaturationCSC((tS16)param1))
         {
            ETG_TRACE_FATAL(("%26s: finished without errors ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
         else
         {
            ETG_TRACE_FATAL(("%26s: finished with errors! ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
      }
      break;
   case DISPVIDCTRL_DIMMING_SETCSCSATURATIONOFFSET:
      {
            tU8 param1 = puchData[3];

            // Set the saturation offset in percent (0 ... 100) of color space conversion.
         ETG_TRACE_FATAL(("%26s: DISPVIDCTRL_DIMMING_SetCSCSaturationOffset %d ",
               "DIM.APPL.DISP.CMD__.START", param1 ));
         if (0 <= s32SetSaturationOffsetCSC((tS16)param1))
         {
            ETG_TRACE_FATAL(("%26s: finished without errors ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
         else
         {
            ETG_TRACE_FATAL(("%26s: finished with errors! ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
      }
      break;
   case DISPVIDCTRL_DIMMING_SETCSCCOLORLEVEL:
      {
            tU8 param1 = puchData[3];

            // Set the color level (0 ... 32) of color space conversion.
         ETG_TRACE_FATAL(("%26s: DISPVIDCTRL_DIMMING_SetCSCColorLevel %d ",
               "DIM.APPL.DISP.CMD__.START", param1 ));
         if (0 <= s32SetColorLevel(param1))
         {
            ETG_TRACE_FATAL(("%26s: finished without errors ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
         else
         {
            ETG_TRACE_FATAL(("%26s: finished with errors! ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
      }
      break;
   case DISPVIDCTRL_DIMMING_SETCSCCOLOR:
      {
         tS16 param1 = (tS16)((((tS16)puchData[3]) <<  8) | (tS16)puchData[4]);

            // Set the color / hue in degree (-180 ... 180) of color space conversion.
         ETG_TRACE_FATAL(("%26s: DISPVIDCTRL_DIMMING_SetCSCColor %d ",
               "DIM.APPL.DISP.CMD__.START", param1 ));
         if (0 <= s32SetHueCSC(param1))
         {
            ETG_TRACE_FATAL(("%26s: finished without errors ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
         else
         {
            ETG_TRACE_FATAL(("%26s: finished with errors! ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
      }
      break;
   case DISPVIDCTRL_DIMMING_SETCSCCOLOROFFSET:
      {
         tS16 param1 = (tS16)((((tS16)puchData[3]) <<  8) | (tS16)puchData[4]);

            // Set the color offset / hue offset in degree (-180 ... 180) of color space conversion.
         ETG_TRACE_FATAL(("%26s: DISPVIDCTRL_DIMMING_SetCSCColorOffset %d ",
               "DIM.APPL.DISP.CMD__.START", param1 ));
         if (0 <= s32SetHueOffsetCSC(param1))
         {
            ETG_TRACE_FATAL(("%26s: finished without errors ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
         else
         {
            ETG_TRACE_FATAL(("%26s: finished with errors! ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
      }
      break;
   case DISPVIDCTRL_DIMMING_SETCSCBRIGHTNESS:
      {
            tU8 param1 = puchData[3];

            // Set the brightness in percent (0 ... 100) of color space conversion.
         ETG_TRACE_FATAL(("%26s: DISPVIDCTRL_DIMMING_SetCSCBrightness %d ",
               "DIM.APPL.DISP.CMD__.START", param1 ));
         if (0 <= s32SetBrightnessCSC((tS16)param1))
         {
            ETG_TRACE_FATAL(("%26s: finished without errors ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
         else
         {
            ETG_TRACE_FATAL(("%26s: finished with errors! ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
      }
      break;
   case DISPVIDCTRL_DIMMING_SETCSCBRIGHTNESSOFFSET:
      {
            tU8 param1 = puchData[3];

            // Set the brightness offset in percent (0 ... 100) of color space conversion.
         ETG_TRACE_FATAL(("%26s: DISPVIDCTRL_DIMMING_SetCSCBrightnessOffset %d ",
               "DIM.APPL.DISP.CMD__.START", param1 ));
         if (0 <= s32SetBrightnessOffsetCSC((tS16)param1))
         {
            ETG_TRACE_FATAL(("%26s: finished without errors ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
         else
         {
            ETG_TRACE_FATAL(("%26s: finished with errors! ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
      }
      break;
   case DISPVIDCTRL_DIMMING_SETCSCBLACKLEVEL:
      {
            tU8 param1 = puchData[3];

            // Set the black level (0 ... 32) of color space conversion.
         ETG_TRACE_FATAL(("%26s: DISPVIDCTRL_DIMMING_SetCSCBlackLevel %d ",
               "DIM.APPL.DISP.CMD__.START", param1 ));
         if (0 <= s32SetBlackLevel(param1))
         {
            ETG_TRACE_FATAL(("%26s: finished without errors ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
         else
         {
            ETG_TRACE_FATAL(("%26s: finished with errors! ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
      }
      break;
   case DISPVIDCTRL_DIMMING_SETCSCGAMMALEVEL:
      {
            tU8 param1 = puchData[3];

            // Set the gamma adjustment level (0 ... 32) of color space conversion.
         ETG_TRACE_FATAL(("%26s: DISPVIDCTRL_DIMMING_SetCSCGammaLevel %d ",
               "DIM.APPL.DISP.CMD__.START", param1 ));
         if (0 <= s32SetGammaLevel(param1))
         {
            ETG_TRACE_FATAL(("%26s: finished without errors ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
         else
         {
            ETG_TRACE_FATAL(("%26s: finished with errors! ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
      }
      break;
   case DISPVIDCTRL_DIMMING_SETCSCGAMMA:
      {
            tU16 param1 = ((tU16)puchData[3] <<  8) | (tU16)puchData[4];

            // Set the gamma adjustment (500 = 0.500 ... 2500 = 2.500) of color space conversion.
         ETG_TRACE_FATAL(("%26s: DISPVIDCTRL_DIMMING_SetCSCGamma %d ",
               "DIM.APPL.DISP.CMD__.START", param1 ));
         if (0 <= s32SetGammaExponent(param1))
         {
            ETG_TRACE_FATAL(("%26s: finished without errors ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
         else
         {
            ETG_TRACE_FATAL(("%26s: finished with errors! ",
                  "DIM.APPL.DISP.CMD__.RES__" ));
         }
      }
      break;

   default:
      break;
   }
}

/*******************************************************************************
*
* FUNCTION:    vGetReferences(tVoid)
*
* DESCRIPTION:    Function to get all reference needed by this class.
*             A reference should always be the Interface class of the object
*
* PARAMETER:    None.
*
* RETURNVALUE:    None.
*
*******************************************************************************/
tVoid dispvidctrl_tclDisplaySettings::vGetReferences(tVoid)
{
   ETG_TRACE_USR4(("vGetReferences() entered."));
   m_poSrvDim = dynamic_cast<Idispvidctrl_tclServiceDimmingIf*>(_cpoMain->getHandler("Idispvidctrl_tclServiceDimmingIf"));
   m_poSyncHandler = dynamic_cast<I_dispvidctrl_tclSyncHandler*>(_cpoMain->getHandler("I_dispvidctrl_tclSyncHandler"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poSyncHandler);
   m_poClientDim = dynamic_cast<dispvidctrl_tclClientDimming*>(_cpoMain->getHandler("dispvidctrl_tclClientDimming"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poClientDim);
}

/*******************************************************************************
*
* FUNCTION:    tVoid vStartCommunication()
*
* DESCRIPTION:    Function to start all dynamic objects e.g. threads, ...
*
* PARAMETER:    None.
*
* RETURNVALUE:    None.
*
*******************************************************************************/
tVoid dispvidctrl_tclDisplaySettings::vStartCommunication(tVoid)
{
   ETG_TRACE_USR4(("vStartCommunication() entered."));

   tU32 u32StartTime = OSAL_ClockGetElapsedTime();
// #ifdef VARIANT_S_FTR_ENABLE_SMART
   s32SetContrastLevel(20);
// #endif
   vInitCSC();
   vInitDRM();
   m_bDisplaySettingsInitDone = TRUE;
   dispvidctrl_tclAppMain::vWriteStartupLog("[RVC-EARLY]: Display Settings initialized", OSAL_ClockGetElapsedTime() - u32StartTime);


   // init CCA properties
   if (OSAL_NULL != m_poSrvDim)
   {
      m_poSrvDim->sendDIM_DISP_CSCContrastLevelStatus(u8GetContrastLevel());
      m_poSrvDim->sendDIM_DISP_CSCSaturationLevelStatus(u8GetSaturationLevel());
      m_poSrvDim->sendDIM_DISP_CSCColorLevelStatus(u8GetColorLevel());
      m_poSrvDim->sendDIM_DISP_CSCBlackLevelStatus(u8GetBlackLevel());
      m_poSrvDim->sendDIM_DISP_CSCGammaLevelStatus(u8GetGammaLevel());
      // old CCA properties ... will be delete
      m_poSrvDim->sendDIM_DISP_ContrastLevelStatus(u8GetContrastLevel());
      m_poSrvDim->sendDIM_DISP_TintLevelStatus(u8GetSaturationLevel());
      m_poSrvDim->sendDIM_DISP_ColorLevelStatus(u8GetColorLevel());
      m_poSrvDim->sendDIM_DISP_BlackLevelStatus(u8GetBlackLevel());
      m_poSrvDim->sendDIM_DISP_YCurveLevelStatus(u8GetGammaLevel());
      vUpdateDisplaySettings(TRUE);		// This is to handle early camera start case
   }
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfo()
*
* DESCRIPTION:    Trace information
*
* PARAMETER:    None.
*
* RETURNVALUE:    None.
*
*******************************************************************************/
tVoid dispvidctrl_tclDisplaySettings::vTraceInfo()
{
   ETG_TRACE_USR4(("vTraceInfo() entered."));
   tChar _strIdentStart[30] = "DIM.APPL.DISP.SETT_.START";
   tChar _strIdentAttr[30]  = "DIM.APPL.DISP.SETT_.ATTR_";
   tChar _strIdentEnd[30]   = "DIM.APPL.DISP.SETT_.END__";
   ETG_TRACE_USR2(("%026s: ", _strIdentStart ));
   ETG_TRACE_USR2(("%26s: %30s=%u ", _strIdentAttr,
		   "            ILM_INIT", ETG_ENUM(DISPVIDCTRL_Bool, m_bILMInit) ));
   ETG_TRACE_USR2(("%26s: %30s=%d ", _strIdentAttr,
		   "        ILM_SrceenID", m_u8DRM_MainHmi_ScreenID ));
   vTraceCSC(_strIdentAttr);
   vTraceDRM(_strIdentAttr);
   ETG_TRACE_USR2(("%026s: ", _strIdentEnd ));
}

/*******************************************************************************
*
* FUNCTION:    s32SetContrastLevel()
*
* DESCRIPTION:    set contrast level
*
* PARAMETER:    tU8         {0 ... 32}
*
* RETURNVALUE:    tS32      {less 0 - failed | else - success}
*
*******************************************************************************/
tS32 dispvidctrl_tclDisplaySettings::s32SetContrastLevel(tU8 u8Level)
{
   ETG_TRACE_USR3(("s32SetContrastLevel(level: %d) entered.", u8Level ));
   tS32 _s32Return = -1;
   if (DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL < u8Level)
	   u8Level = DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL;
   if (DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL >= u8Level)
   {
	  tU8 _u8ContrastLevel = m_u8ContrastLevel;
      m_u8ContrastLevel = u8Level;
      dp_tclDimmingDPDISP_CSCContrastLevel _oDpCSCContrastLevel;
      _oDpCSCContrastLevel.vSetData(m_u8ContrastLevel);
      tS16 _s16Percent = DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_CONTRAST_MAX;
      _s16Percent -= DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_CONTRAST_MIN;
      _s16Percent *= u8Level;
      _s16Percent /= DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL;
      _s16Percent += DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_CONTRAST_MIN;
      _s32Return = s32SetContrastCSC(_s16Percent);
      // update CCA property
      if (OSAL_NULL != m_poSrvDim && m_u8ContrastLevel != _u8ContrastLevel)
      {
          m_poSrvDim->sendDIM_DISP_CSCContrastLevelStatus(m_u8ContrastLevel);
          // old CCA properties ... will be delete
          m_poSrvDim->sendDIM_DISP_ContrastLevelStatus(m_u8ContrastLevel);
      }
   }
   return _s32Return;
}

/*******************************************************************************
*
* FUNCTION:    s32SetTintLevel()
*
* DESCRIPTION:    set tint level
*
* PARAMETER:    tU8         {0 ... 32}
*
* RETURNVALUE:    tS32      {less 0 - failed | else - success}
*
*******************************************************************************/
tS32 dispvidctrl_tclDisplaySettings::s32SetSaturationLevel(tU8 u8Level)
{
   ETG_TRACE_USR3(("s32SetTintLevel(level: %d) entered.", u8Level ));
   tS32 _s32Return = -1;
   if (DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL < u8Level)
	   u8Level = DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL;
   if (DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL >= u8Level)
   {
	  tU8 _u8SaturationLevel = m_u8SaturationLevel;
      m_u8SaturationLevel = u8Level;
      dp_tclDimmingDPDISP_CSCSaturationLevel _oDpCSCSaturationLevel;
      _oDpCSCSaturationLevel.vSetData(m_u8SaturationLevel);
      tS16 _s16Percent = DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_SATURATION_MAX;
      _s16Percent -= DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_SATURATION_MIN;
      _s16Percent *= u8Level;
      _s16Percent /= DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL;
      _s16Percent += DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_SATURATION_MIN;
      _s32Return = s32SetSaturationCSC(_s16Percent);
      // update CCA property
      if (OSAL_NULL != m_poSrvDim && m_u8SaturationLevel != _u8SaturationLevel)
      {
		  m_poSrvDim->sendDIM_DISP_CSCSaturationLevelStatus(m_u8SaturationLevel);
		  // old CCA properties ... will be delete
		  m_poSrvDim->sendDIM_DISP_TintLevelStatus(m_u8SaturationLevel);
      }
   }
   return _s32Return;
}

/*******************************************************************************
*
* FUNCTION:    s32SetColorLevel()
*
* DESCRIPTION:    set color level
*
* PARAMETER:    tU8         {0 ... 32}
*
* RETURNVALUE:    tS32      {less 0 - failed | else - success}
*
*******************************************************************************/
tS32 dispvidctrl_tclDisplaySettings::s32SetColorLevel(tU8 u8Level)
{
   ETG_TRACE_USR3(("s32SetColorLevel(level: %d) entered.", u8Level ));
   tS32 _s32Return = -1;
   if (DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL < u8Level)
	   u8Level = DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL;
   if (DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL >= u8Level)
   {
	  tU8 _u8ColorLevel = m_u8ColorLevel;
      m_u8ColorLevel = u8Level;
      dp_tclDimmingDPDISP_CSCColorLevel _oDpCSCColorLevel;
      _oDpCSCColorLevel.vSetData(m_u8ColorLevel);
      tS16 _s16Degree = DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_HUE_MAX;
      _s16Degree -= DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_HUE_MIN;
      _s16Degree *= u8Level;
      _s16Degree /= DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL;
      _s16Degree += DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_HUE_MIN;
      _s32Return = s32SetHueCSC(_s16Degree);
      if (OSAL_NULL != m_poSrvDim && m_u8ColorLevel != _u8ColorLevel)
      {
          m_poSrvDim->sendDIM_DISP_CSCColorLevelStatus(m_u8ColorLevel);
          // old CCA properties ... will be delete
          m_poSrvDim->sendDIM_DISP_ColorLevelStatus(m_u8ColorLevel);
      }
   }
   return _s32Return;
}

/*******************************************************************************
*
* FUNCTION:    s32SetBlackLevel()
*
* DESCRIPTION:    set black level
*
* PARAMETER:    tU8         {0 ... 32}
*
* RETURNVALUE:    tS32      {less 0 - failed | else - success}
*
*******************************************************************************/
tS32 dispvidctrl_tclDisplaySettings::s32SetBlackLevel(tU8 u8Level)
{
   ETG_TRACE_USR3(("s32SetBlackLevel(level: %d) entered.", u8Level ));
   tS32 _s32Return = -1;
   if (DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL < u8Level)
	   u8Level = DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL;
   if (DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL >= u8Level)
   {
	  tU8 _u8BlackLevel = m_u8BlackLevel;
      m_u8BlackLevel = u8Level;
      dp_tclDimmingDPDISP_CSCBlackLevel _oDpCSCBlackLevel;
      _oDpCSCBlackLevel.vSetData(m_u8BlackLevel);
      // set csc brightness and brightness offset
	  tS16 _s16Brightness = s16GetBrightnessCSC();
	  tS16 _s16BrightnessOffset = s16GetBrightnessOffsetCSC();
      if ((DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL/2) >= u8Level)
      {
    	  // calculate csc brightness
    	  _s16Brightness = DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_BRIGHTNESS_MAX;
    	  _s16Brightness -= DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_BRIGHTNESS_MIN;
    	  _s16Brightness *= (tS16)u8Level;
    	  _s16Brightness /= DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL/2;
    	  _s16Brightness += DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_BRIGHTNESS_MIN;
    	  // calculate csc brightness offset
    	  _s16BrightnessOffset = 0;
      }
      else
      {
    	  // calculate csc brightness
    	  _s16Brightness = 100;
    	  // calculate csc brightness offset
    	  _s16BrightnessOffset = DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_BRIGHTNESS_OFFSET_MAX;
    	  _s16BrightnessOffset -= DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_BRIGHTNESS_OFFSET_MIN;
    	  _s16BrightnessOffset *= ((tS16)u8Level - (DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL/2));
    	  _s16BrightnessOffset /= DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL/2;
    	  _s16BrightnessOffset += DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_BRIGHTNESS_OFFSET_MIN;
      }
      _s32Return = s32SetBrightnessCSC(_s16Brightness);
      ETG_TRACE_USR3(("s32SetBlackLevel(level: %d): set CSC brightness to %d percent returns %d",
    		  u8Level, _s16Brightness, _s32Return ));
      _s32Return = s32SetBrightnessOffsetCSC(_s16BrightnessOffset);
      ETG_TRACE_USR3(("s32SetBlackLevel(level: %d): set CSC brightness offset to %d percent returns %d",
    		  u8Level, _s16BrightnessOffset, _s32Return ));
      _s32Return = s32UpdateCSC();
      ETG_TRACE_USR3(("s32SetBlackLevel(level: %d): update CSC returns %d",
    		  u8Level, _s32Return ));
      if (OSAL_NULL != m_poSrvDim && m_u8BlackLevel != _u8BlackLevel)
      {
          m_poSrvDim->sendDIM_DISP_CSCBlackLevelStatus(m_u8BlackLevel);
          // old CCA properties ... will be delete
          m_poSrvDim->sendDIM_DISP_BlackLevelStatus(m_u8BlackLevel);
      }
   }
   return _s32Return;
}

/*******************************************************************************
*
* FUNCTION:    s32SetGammaLevel()
*
* DESCRIPTION:    set gamma correction level
*
* PARAMETER:    tU8         {0 ... 32}
*
* RETURNVALUE:    tS32      {less 0 - failed | else - success}
*
*******************************************************************************/
tS32 dispvidctrl_tclDisplaySettings::s32SetGammaLevel(tU8 u8Level)
{
   ETG_TRACE_USR3(("s32SetGammaLevel(level: %d) entered.", u8Level ));
   tS32 _s32Return = -1;
   if (DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL < u8Level)
	   u8Level = DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL;
   if (DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL >= u8Level)
   {
	  tU8 _u8GammaLevel = m_u8GammaLevel;
	  m_u8GammaLevel = DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL - u8Level;
      dp_tclDimmingDPDISP_CSCGammaLevel _oDpCSCGammeLevel;
      _oDpCSCGammeLevel.vSetData(m_u8GammaLevel);
      tU16 _u16Exponent = DISPVIDCTRL_DIM_C_U16_DISPLAYSETTINGS_GAMMA_EXPONENT_MAX;
      _u16Exponent -= DISPVIDCTRL_DIM_C_U16_DISPLAYSETTINGS_GAMMA_EXPONENT_MIN;
      _u16Exponent *= m_u8GammaLevel;
      _u16Exponent /= DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_MAXLEVEL;
      _u16Exponent += DISPVIDCTRL_DIM_C_U16_DISPLAYSETTINGS_GAMMA_EXPONENT_MIN;
      _s32Return = s32SetGammaExponent(_u16Exponent);
      if (OSAL_NULL != m_poSrvDim && m_u8GammaLevel != _u8GammaLevel)
      {
      	 m_poSrvDim->sendDIM_DISP_CSCGammaLevelStatus(m_u8GammaLevel);
      	 // old CCA properties ... will be delete
      	 m_poSrvDim->sendDIM_DISP_YCurveLevelStatus(m_u8GammaLevel);
      }
   }
   return _s32Return;
}

/*******************************************************************************
*
* FUNCTION:    s32RestDisplaySettings()
*
* DESCRIPTION:    reset all display settings
*
* PARAMETER:    none
*
* RETURNVALUE:    tS32   { less 0 - failed | else success }
*
*******************************************************************************/
tS32 dispvidctrl_tclDisplaySettings::s32ResetDisplaySettings(tVoid)
{
   ETG_TRACE_USR3(("s32RestDisplaySettings() entered."));

   tS32 _s32Return = -1;
   tU8 _u8ContrastLevel = u8GetContrastLevel();
   tU8 _u8SaturationLevel = u8GetSaturationLevel();
   tU8 _u8ColorLevel = u8GetColorLevel();
   tU8 _u8BlackLevel = u8GetBlackLevel();
   tU8 _u8GammaLevel = u8GetGammaLevel();
   if (0 <= s32ResetCSC() && 0 <= s32ResetDRM())
   {
      // trace display settings
      vTraceInfo();
      _s32Return = 0;
   }
   // update CCA properties
   if (OSAL_NULL != m_poSrvDim)
   {
      if (u8GetContrastLevel() != _u8ContrastLevel)
      {
          m_poSrvDim->sendDIM_DISP_CSCContrastLevelStatus(u8GetContrastLevel());
          // old CCA properties ... will be delete
          m_poSrvDim->sendDIM_DISP_ContrastLevelStatus(u8GetContrastLevel());
      }
      if (u8GetSaturationLevel() != _u8SaturationLevel)
      {
          m_poSrvDim->sendDIM_DISP_CSCSaturationLevelStatus(u8GetSaturationLevel());
          // old CCA properties ... will be delete
          m_poSrvDim->sendDIM_DISP_TintLevelStatus(u8GetSaturationLevel());
      }
      if (u8GetColorLevel() != _u8ColorLevel)
      {
          m_poSrvDim->sendDIM_DISP_CSCColorLevelStatus(u8GetColorLevel());
          // old CCA properties ... will be delete
          m_poSrvDim->sendDIM_DISP_ColorLevelStatus(u8GetColorLevel());
      }
      if (u8GetBlackLevel() != _u8BlackLevel)
      {
          m_poSrvDim->sendDIM_DISP_CSCBlackLevelStatus(u8GetBlackLevel());
          // old CCA properties ... will be delete
          m_poSrvDim->sendDIM_DISP_BlackLevelStatus(u8GetBlackLevel());
      }
      if (u8GetGammaLevel() != _u8GammaLevel)
      {
     	 m_poSrvDim->sendDIM_DISP_CSCGammaLevelStatus(u8GetGammaLevel());
     	 // old CCA properties ... will be delete
     	 m_poSrvDim->sendDIM_DISP_YCurveLevelStatus(u8GetGammaLevel());
      }
   }
   return _s32Return;
}

/*******************************************************************************
*
* FUNCTION:    vInitCSC()
*
* DESCRIPTION:    initiate color space conversion
*               - open CSC device
*               - load default values
*
* PARAMETER:    none
*
* RETURNVALUE:    none
*
*******************************************************************************/
tVoid dispvidctrl_tclDisplaySettings::vInitCSC(tVoid)
{
   ETG_TRACE_USR3(("vInitCSC() entered."));

#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   // reset display settings
   tBool _bDoResetDisplaySettings = FALSE;
   m_tCSCProp.set_to_default = FALSE;
   ilmErrorTypes _tResult = enInitILM();
   if (ILM_SUCCESS == _tResult)
   {
	   // load current last mode CSC data
	   // ... contrast
	   dp_tclDimmingDPDISP_CSCContrastLevel _oDpCSCContrastLevel;
	   m_u8ContrastLevel = _oDpCSCContrastLevel.tGetData();
	   if (DISPVIDCTRL_C_U8_LEVEL_MAX < m_u8ContrastLevel)
		   _bDoResetDisplaySettings = TRUE;
	   dp_tclDimmingDPDISP_CSCContrast _oDpCSCContrast;
	   m_tCSCProp.contrast = _oDpCSCContrast.tGetData();
	   {
		   dp_tclEarlyConfigCSCTrContrastValue _oDpEC_CSC;
		   m_tCSCProp.contrast = _oDpEC_CSC.tGetData();
	   }
	   // ... saturation / tine
	   dp_tclDimmingDPDISP_CSCSaturationLevel _oDpCSCSaturationLevel;
	   m_u8SaturationLevel = _oDpCSCSaturationLevel.tGetData();
	   if (DISPVIDCTRL_C_U8_LEVEL_MAX < m_u8SaturationLevel)
		   _bDoResetDisplaySettings = TRUE;
	   dp_tclDimmingDPDISP_CSCSaturation _oDpCSCSaturation;
	   m_tCSCProp.saturation = _oDpCSCSaturation.tGetData();
	   {
		   dp_tclEarlyConfigCSCTrSaturationValue _oDpEC_CSC;
		   m_tCSCProp.saturation = _oDpEC_CSC.tGetData();
	   }
	   dp_tclDimmingDPDISP_CSCSaturationOffset _oDpCSCSaturationOffset;
	   m_tCSCProp.saturation_off = _oDpCSCSaturationOffset.tGetData();
	   {
		   dp_tclEarlyConfigCSCTrSaturationOffsetValue _oDpEC_CSC;
		   m_tCSCProp.saturation_off = _oDpEC_CSC.tGetData();
	   }
	   // ... color / hue
	   dp_tclDimmingDPDISP_CSCColorLevel _oDpCSCColorLevel;
	   m_u8ColorLevel = _oDpCSCColorLevel.tGetData();
	   if (DISPVIDCTRL_C_U8_LEVEL_MAX < m_u8ColorLevel)
		   _bDoResetDisplaySettings = TRUE;
	   dp_tclDimmingDPDISP_CSCColor _oDpCSCColor;
	   m_tCSCProp.hue = _oDpCSCColor.tGetData();
	   {
		   dp_tclEarlyConfigCSCTrHueValue _oDpEC_CSC;
		   m_tCSCProp.hue = _oDpEC_CSC.tGetData();
	   }
	   dp_tclDimmingDPDISP_CSCColorOffset _oDpCSCColorOffset;
	   m_tCSCProp.hue_off = _oDpCSCColorOffset.tGetData();
	   {
		   dp_tclEarlyConfigCSCTrHueOffsetValue _oDpEC_CSC;
		   m_tCSCProp.hue_off = _oDpEC_CSC.tGetData();
	   }
	   // ... brightness
	   m_u8BlackLevel = DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_BLACKLEVEL_DEFAULT;
	   dp_tclDimmingDPDISP_CSCBlackLevel _oDpCSCBlackLevel;
	   m_u8BlackLevel = _oDpCSCBlackLevel.tGetData();
	   if (DISPVIDCTRL_C_U8_LEVEL_MAX < m_u8BlackLevel)
		   _bDoResetDisplaySettings = TRUE;
	   dp_tclDimmingDPDISP_CSCBrigthness _oDpCSCBrigthness;
	   m_tCSCProp.brightness = _oDpCSCBrigthness.tGetData();
	   {
		   dp_tclEarlyConfigCSCTrBrightnessValue _oDpEC_CSC;
		   m_tCSCProp.brightness = _oDpEC_CSC.tGetData();
	   }
	   dp_tclDimmingDPDISP_CSCBrigthnessOffset _oDpCSCBrightnessOffset;
	   m_s16BrightnessOffset = _oDpCSCBrightnessOffset.tGetData();
	   m_tCSCProp.brightness_off = m_s16BrightnessOffset;
	   {
		   dp_tclEarlyConfigCSCTrBrightnessOffsetValue _oDpEC_CSC;
		   m_tCSCProp.brightness_off = _oDpEC_CSC.tGetData();
	   }
	   if (TRUE == _bDoResetDisplaySettings)
	   {
		   (tVoid)s32ResetCSC();
	   }
	   else
	   {
		   (tVoid)s32UpdateCSC();
	   }
	   // init done state needs to be known for switching to RVC mode
       m_bCscInitDone = TRUE;
   }
   if (ILM_SUCCESS != _tResult)
   {
	   ETG_TRACE_FATAL(("vInitCSC: update csc failed for screen %d failed!", m_u8DRM_MainHmi_ScreenID ));
   }
#else
   ETG_TRACE_USR1(("vInitCSC: CSC unavailable."));
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
}

/*******************************************************************************
*
* FUNCTION:    s32UpdateCSC()
*
* DESCRIPTION:    update color space conversion properties and identity matrix
*
* PARAMETER:    none
*
* RETURNVALUE:    tS32   { less 0 - failed | else success }
*
*******************************************************************************/
tS32 dispvidctrl_tclDisplaySettings::s32UpdateCSC(tVoid)
{
   ETG_TRACE_USR3(("s32UpdateCSC() entered."));

#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   tBool bDimmingModeRvc = FALSE;
   ilmErrorTypes _tResult = enInitILM();
   m_tCSCProp.set_to_default = FALSE;
   if (ILM_SUCCESS == _tResult)
   {
      //ETG_TRACE_FATAL(("s32UpdateCSC: Brightness       %d", m_tCSCProp.brightness));
      //ETG_TRACE_FATAL(("s32UpdateCSC: BrightnessOffset %d", m_tCSCProp.brightness_off));
      //ETG_TRACE_FATAL(("s32UpdateCSC: Contrast         %d", m_tCSCProp.contrast));
      //ETG_TRACE_FATAL(("s32UpdateCSC: Saturation       %d", m_tCSCProp.saturation));
      //ETG_TRACE_FATAL(("s32UpdateCSC: SaturationOffset %d", m_tCSCProp.saturation_off));
      //ETG_TRACE_FATAL(("s32UpdateCSC: Hue              %d", m_tCSCProp.hue));
      //ETG_TRACE_FATAL(("s32UpdateCSC: HueOffset        %d", m_tCSCProp.hue_off));

      if (m_poSyncHandler)
      {
         bDimmingModeRvc = m_poSyncHandler->bGetDimmingModeRvc();
      }
      if (bDimmingModeRvc == FALSE)
      {
         //ETG_TRACE_FATAL(("s32UpdateCSC: RVC Mode == FALSE"));
         _tResult = ilm_displaySetCSC(m_u8DRM_MainHmi_ScreenID, &m_tCSCProp);
      }
      else
      {
         ilmCSCProperties tCSCProp_RVC;
         tCSCProp_RVC = m_tCSCProp;

         // set RVC Mode defaults
         tCSCProp_RVC.brightness     = 100;
         tCSCProp_RVC.brightness_off = 1;
         tCSCProp_RVC.contrast       = 75;
         tCSCProp_RVC.saturation     = 100;
         tCSCProp_RVC.saturation_off = 0;
         tCSCProp_RVC.hue            = 0;
         tCSCProp_RVC.hue_off        = 0;

         ETG_TRACE_FATAL(("s32UpdateCSC: RVC Mode == TRUE"));
         _tResult = ilm_displaySetCSC(m_u8DRM_MainHmi_ScreenID, &tCSCProp_RVC);
      }

      ETG_TRACE_USR2(("s32UpdateCSC: ilm_displaySetCSC(screen:%d) returns %d", m_u8DRM_MainHmi_ScreenID, _tResult ));
   }
   else
   {
	   ETG_TRACE_FATAL(("s32UpdateCSC: update csc failed for screen %d failed!", m_u8DRM_MainHmi_ScreenID ));
   }
   if (ILM_SUCCESS == _tResult)
   {
	   return 0;
   }
   return -1;
#else
   ETG_TRACE_USR1(("s32UpdateCSC: CSC unavailable."));
   return 1;
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
}

/*******************************************************************************
*
* FUNCTION:    s32ResetCSC()
*
* DESCRIPTION:    reset color space conversion properties and identity matrix
*
* PARAMETER:    none
*
* RETURNVALUE:    tS32   { less 0 - failed | else success }
*
*******************************************************************************/
tS32 dispvidctrl_tclDisplaySettings::s32ResetCSC(tVoid)
{
   ETG_TRACE_USR3(("s32ResetCSC() entered."));
#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   ilmErrorTypes _tResult = enInitILM();
   (tVoid)memset((tVoid*)&m_tCSCProp, OSAL_NULL, sizeof(m_tCSCProp));
   m_tCSCProp.set_to_default = TRUE;
   if (ILM_SUCCESS == _tResult)
   {
	   _tResult = ilm_displaySetCSC(m_u8DRM_MainHmi_ScreenID, &m_tCSCProp);
	   ETG_TRACE_USR2(("s32ResetCSC: ilm_displaySetCSC(screen: %d) returns %d", m_u8DRM_MainHmi_ScreenID, _tResult ));
   }
   if (ILM_SUCCESS == _tResult)
   {
	   // default levels
	   (tVoid)s32SetContrastLevel(DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_CONTRAST_LEVEL_DEFAULT);
	   (tVoid)s32SetSaturationLevel(DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_SATURATION_LEVEL_DEFAULT);
	   (tVoid)s32SetColorLevel(DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_HUE_LEVEL_DEFAULT);
	   (tVoid)s32SetBlackLevel(DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_BLACKLEVEL_DEFAULT);
	   // update color space conversion properties and identity matrix
	   return s32UpdateCSC();
   }
   else
   {
	   ETG_TRACE_FATAL(("s32ResetCSC: reset csc failed for screen %d failed!", m_u8DRM_MainHmi_ScreenID ));
   }
   return -1;
#else
   ETG_TRACE_USR1(("s32ResetCSC: CSC unavailable."));
   return 1;
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
}

/*******************************************************************************
*
* FUNCTION:    vTraceCSC()
*
* DESCRIPTION:    trace color space conversion properties and identity matrix
*
* PARAMETER:
*
* RETURNVALUE:    none
*
*******************************************************************************/
tVoid dispvidctrl_tclDisplaySettings::vTraceCSC(const char* strIdentifier) const
{
   ETG_TRACE_USR4(("vTraceCSC() entered."));

   ETG_TRACE_USR2(("%026s: <<< color space conversion >>> ", strIdentifier ));
#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   char _strAttrName[60] = {0};

   ETG_TRACE_USR2(("%26s: %30s=%u ",
         strIdentifier, "     CSC_SET2DEFAULT", m_tCSCProp.set_to_default ));
   ETG_TRACE_USR2(("%26s: %30s=%u ",
         strIdentifier, "            CSC_TYPE", m_tCSCProp.type ));

   ETG_TRACE_USR2(("%26s: %30s=%d (min: %d percent, max: %d percent) ",
         strIdentifier, "       ContrastLevel", u8GetContrastLevel(),
         DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_CONTRAST_MIN, DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_CONTRAST_MAX ));
   ETG_TRACE_USR2(("%26s: %30s=%d (min: %d percent, max: %d percent) ",
         strIdentifier, "     SaturationLevel", u8GetSaturationLevel(),
         DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_SATURATION_MIN, DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_SATURATION_MAX ));
   ETG_TRACE_USR2(("%26s: %30s=%d (min: %d degree, max: %d degree) ",
         strIdentifier, "          ColorLevel", u8GetColorLevel(),
         DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_HUE_MIN, DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_HUE_MAX ));
   ETG_TRACE_USR2(("%26s: %30s=%d (min[0-16]: %d percent, max[0-16]: %d percent, min[17-32]: max[16] + %d percent, max[17-32]: max[16] + %d percent) ",
         strIdentifier, "          BlackLevel", u8GetBlackLevel(),
         DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_BRIGHTNESS_MIN, DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_BRIGHTNESS_MAX,
         DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_BRIGHTNESS_OFFSET_MIN, DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_BRIGHTNESS_OFFSET_MAX ));
   ETG_TRACE_USR2(("%26s: %30s=%d (min: %f, max: %f) ",
         strIdentifier, "          GammaLevel", u8GetGammaLevel(),
         ((tF32)DISPVIDCTRL_DIM_C_U16_DISPLAYSETTINGS_GAMMA_EXPONENT_MIN / 1000.0), ((tF32)DISPVIDCTRL_DIM_C_U16_DISPLAYSETTINGS_GAMMA_EXPONENT_MAX / 1000.0) ));

   ETG_TRACE_USR2(("%26s: %30s=%d [degree] (min: %d degree, max: %d degree) ",
         strIdentifier, "                 HUE", s16GetHueCSC(),
         DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_DEGREE_MIN, DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_DEGREE_MAX ));
   ETG_TRACE_USR2(("%26s: %30s=%d [percent] (min: %d percent, max: %d percent) ",
         strIdentifier, "          SATURATION", s16GetSaturationCSC(),
         DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MIN, DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MAX ));
   ETG_TRACE_USR2(("%26s: %30s=%d [percent] (min: %d percent, max: %d percent) ",
         strIdentifier, "          BRIGHTNESS", s16GetBrightnessCSC(),
         DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MIN, DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MAX ));
   ETG_TRACE_USR2(("%26s: %30s=%d [percent] (min: %d percent, max: %d percent) ",
         strIdentifier, "            CONTRAST", s16GetContrastCSC(),
         DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MIN, DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MAX ));
   ETG_TRACE_USR2(("%26s: %30s=%d [degree] (min: %d degree, max: %d degree) ",
         strIdentifier, "          HUE_OFFSET", s16GetHueOffsetCSC(),
         DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_DEGREE_MIN, DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_DEGREE_MAX ));
   ETG_TRACE_USR2(("%26s: %30s=%d [percent] (min: %d percent, max: %d percent) ",
         strIdentifier, "   SATURATION_OFFSET", s16GetSaturationOffsetCSC(),
         DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MIN, DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MAX ));
   ETG_TRACE_USR2(("%26s: %30s=%d [percent] (min: %d percent, max: %d percent) ",
         strIdentifier, "   BRIGHTNESS_OFFSET", s16GetBrightnessOffsetCSC(),
         DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MIN, DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MAX ));
   ETG_TRACE_USR2(("%26s: %30s=%d [percent] (min: %d percent, max: %d percent) ",
         strIdentifier, " O_BRIGHTNESS_OFFSET", m_tCSCProp.brightness_off,
         DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MIN, DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MAX ));
   ETG_TRACE_USR2(("%26s: %30s=%d [percent] (min: %d percent, max: %d percent) ",
         strIdentifier, "EH_BRIGHTNESS_OFFSET", s16GetEnhanceBrightnessOffset(),
         DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MIN, DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MAX ));
#ifdef VARIANT_S_FTR_ENABLE_BRIGHTNESS_ENHANCEMENT
   ETG_TRACE_USR2(("%26s: %30s=%u ",
         strIdentifier, "            DIM_MODE", ETG_ENUM(DIM_DIMMODE, m_u8DimMode) ));
   ETG_TRACE_USR2(("%26s: %30s=%u [level] ",
         strIdentifier, "  DIM_BRIGHTNESS_DAY", m_au8DimBrightnessLevel[0] ));
   ETG_TRACE_USR2(("%26s: %30s=%u [level] ",
         strIdentifier, "DIM_BRIGHTNESS_NIGHT", m_au8DimBrightnessLevel[1] ));
#endif //#ifdef VARIANT_S_FTR_ENABLE_BRIGHTNESS_ENHANCEMENT
#else
   ETG_TRACE_USR1(("vTraceCSC: CSC unavailable."));
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
}

/*******************************************************************************
*
* FUNCTION:    s16GetHueCSC()
*
* DESCRIPTION:    get hue in degree
*
* PARAMETER:    none
*
* RETURNVALUE:    tS16   { -180 ... 180 }
*
*******************************************************************************/
tS16 dispvidctrl_tclDisplaySettings::s16GetHueCSC(tVoid) const
{
   ETG_TRACE_USR4(("s16GetHueCSC() entered."));

#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   return m_tCSCProp.hue;
#else
   ETG_TRACE_USR1(("s16GetHueCSC: CSC unavailable."));
   return 0;
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
}

/*******************************************************************************
*
* FUNCTION:    s32SetHueCSC()
*
* DESCRIPTION:    set hue in degree and update CSC
*
* PARAMETER:    tS16   { -180 ... 180 }
*
* RETURNVALUE:    tS32   { less 0 - failed | else success }
*
*******************************************************************************/
tS32 dispvidctrl_tclDisplaySettings::s32SetHueCSC(tS16 s16Degree)
{
   ETG_TRACE_USR3(("s32SetHueCSC(degree: %d) entered.", s16Degree));

   tS32 _s32Return = -1;
#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   if (DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_DEGREE_MIN <= s16Degree && DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_DEGREE_MAX >= s16Degree)
   {
	   m_tCSCProp.hue = s16Degree;
      dp_tclDimmingDPDISP_CSCColor _oDpCSCColor;
      _oDpCSCColor.vSetData(s16GetHueCSC());
      dp_tclEarlyConfigCSCTrHueValue _oDpEC_CSC;
      _oDpEC_CSC.vSetData(s16GetHueCSC());
      _s32Return = s32UpdateCSC();
      if (0 > _s32Return)
      {
         ETG_TRACE_FATAL(("s32SetHueCSC: hue %d could not set! (error code: %d)", s16Degree, _s32Return));
      }
   }
   else
   {
      ETG_TRACE_FATAL(("s32SetHueCSC: hue %d out of range!", s16Degree));
   }
#else
   ETG_TRACE_USR1(("s32SetHueCSC: CSC unavailable."));
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   return _s32Return;
}

/*******************************************************************************
*
* FUNCTION:    s16GetHueOffsetCSC()
*
* DESCRIPTION:    get hue offset in degree
*
* PARAMETER:    none
*
* RETURNVALUE:    tS16   { -180 ... 180 }
*
*******************************************************************************/
tS16 dispvidctrl_tclDisplaySettings::s16GetHueOffsetCSC(tVoid) const
{
   ETG_TRACE_USR4(("s16GetHueOffsetCSC() entered."));

#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   return m_tCSCProp.hue_off;
#else
   ETG_TRACE_USR1(("s16GetHueOffsetCSC: CSC unavailable."));
   return 0;
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
}

/*******************************************************************************
*
* FUNCTION:    s32SetHueOffsetCSC()
*
* DESCRIPTION:    set hue offset in degree and update CSC
*
* PARAMETER:    tS16   { -180 ... 180 }
*
* RETURNVALUE:    tS32   { less 0 - failed | else success }
*
*******************************************************************************/
tS32 dispvidctrl_tclDisplaySettings::s32SetHueOffsetCSC(tS16 s16Degree)
{
   ETG_TRACE_USR3(("s32SetHueOffsetCSC(degree: %d) entered.", s16Degree));

   tS32 _s32Return = -1;
#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   if (DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_DEGREE_MIN <= s16Degree && DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_DEGREE_MAX >= s16Degree)
   {
	   m_tCSCProp.hue_off = s16Degree;
      dp_tclDimmingDPDISP_CSCColorOffset _oDpCSCColorOffset;
      _oDpCSCColorOffset.vSetData(s16GetHueOffsetCSC());
      dp_tclEarlyConfigCSCTrHueOffsetValue _oDpEC_CSC;
      _oDpEC_CSC.vSetData(s16GetHueOffsetCSC());
      _s32Return = s32UpdateCSC();
      if (0 > _s32Return)
      {
         ETG_TRACE_FATAL(("s32SetHueOffsetCSC: hue offset %d could not set! (error code: %d)", s16Degree, _s32Return));
      }
   }
   else
   {
      ETG_TRACE_FATAL(("s32SetHueOffsetCSC: hue offset %d out of range!", s16Degree));
   }
#else
   ETG_TRACE_USR1(("s32SetHueOffsetCSC: CSC unavailable."));
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   return _s32Return;
}

/*******************************************************************************
*
* FUNCTION:    s16GetSaturationCSC()
*
* DESCRIPTION:    get saturation in percent
*
* PARAMETER:    none
*
* RETURNVALUE:    tS16   { 0 ... 100 }
*
*******************************************************************************/
tS16 dispvidctrl_tclDisplaySettings::s16GetSaturationCSC(tVoid) const
{
   ETG_TRACE_USR4(("s16GetSaturationCSC() entered."));

#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   return m_tCSCProp.saturation;
#else
   ETG_TRACE_USR1(("s16GetSaturationCSC: CSC unavailable."));
   return 100;
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
}

/*******************************************************************************
*
* FUNCTION:    s32SetSaturationCSC()
*
* DESCRIPTION:    set saturation in percent and update CSC
*
* PARAMETER:    tS16   { 0 ... 100 }
*
* RETURNVALUE:    tS32   { less 0 - failed | else success }
*
*******************************************************************************/
tS32 dispvidctrl_tclDisplaySettings::s32SetSaturationCSC(tS16 s16Percent)
{
   ETG_TRACE_USR3(("s32SetSaturationCSC(percent: %d) entered.", s16Percent));

   tS32 _s32Return = -1;
#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   if (DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MIN <= s16Percent && DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MAX >= s16Percent)
   {
	   m_tCSCProp.saturation = s16Percent;
      dp_tclDimmingDPDISP_CSCSaturation _oDpCSCSaturation;
      _oDpCSCSaturation.vSetData(s16GetSaturationCSC());
      dp_tclEarlyConfigCSCTrSaturationValue _oDpEC_CSC;
      _oDpEC_CSC.vSetData(s16GetSaturationCSC());
      _s32Return = s32UpdateCSC();
      if (0 > _s32Return)
      {
         ETG_TRACE_FATAL(("s32SetSaturationCSC: saturation %d could not set! (error code: %d)", s16Percent, _s32Return));
      }
   }
   else
   {
      ETG_TRACE_FATAL(("s32SetSaturationCSC: saturation %d out of range", s16Percent));
   }
#else
   ETG_TRACE_USR1(("s32SetSaturationCSC: CSC unavailable."));
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   return _s32Return;
}

/*******************************************************************************
*
* FUNCTION:    s16GetSaturationOffsetCSC()
*
* DESCRIPTION:    get saturation offset in percent
*
* PARAMETER:    none
*
* RETURNVALUE:    tS16   { 0 ... 100 }
*
*******************************************************************************/
tS16 dispvidctrl_tclDisplaySettings::s16GetSaturationOffsetCSC(tVoid) const
{
   ETG_TRACE_USR4(("s16GetSaturationOffsetCSC() entered."));

#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   return m_tCSCProp.saturation_off;
#else
   ETG_TRACE_USR1(("s16GetSaturationOffsetCSC: CSC unavailable."));
   return 100;
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
}

/*******************************************************************************
*
* FUNCTION:    s32SetSaturationOffsetCSC()
*
* DESCRIPTION:    set saturation offset in percent and update CSC
*
* PARAMETER:    tS16   { 0 ... 100 }
*
* RETURNVALUE:    tS32   { less 0 - failed | else success }
*
*******************************************************************************/
tS32 dispvidctrl_tclDisplaySettings::s32SetSaturationOffsetCSC(tS16 s16Percent)
{
   ETG_TRACE_USR3(("s32SetSaturationOffsetCSC(percent: %d) entered.", s16Percent));

   tS32 _s32Return = -1;
#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   if (DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MIN <= s16Percent && DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MAX >= s16Percent)
   {
	   m_tCSCProp.saturation_off = s16Percent;
      dp_tclDimmingDPDISP_CSCSaturationOffset _oDpCSCSaturationOffset;
      _oDpCSCSaturationOffset.vSetData(s16GetSaturationOffsetCSC());
      dp_tclEarlyConfigCSCTrSaturationOffsetValue _oDpEC_CSC;
      _oDpEC_CSC.vSetData(s16GetSaturationOffsetCSC());
      _s32Return = s32UpdateCSC();
      if (0 > _s32Return)
      {
         ETG_TRACE_FATAL(("s32SetSaturationOffsetCSC: saturation offset %d could not set! (error code: %d)", s16Percent, _s32Return));
      }
   }
   else
   {
      ETG_TRACE_FATAL(("s32SetSaturationOffsetCSC: saturation offset %d out of range", s16Percent));
   }
#else
   ETG_TRACE_USR1(("s32SetSaturationOffsetCSC: CSC unavailable."));
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   return _s32Return;
}

/*******************************************************************************
*
* FUNCTION:    s16GetBrightnessCSC()
*
* DESCRIPTION:    get brightness in percent
*
* PARAMETER:    none
*
* RETURNVALUE:    tS16   { 0 ... 100 }
*
*******************************************************************************/
tS16 dispvidctrl_tclDisplaySettings::s16GetBrightnessCSC(tVoid) const
{
   ETG_TRACE_USR4(("s16GetBrightnessCSC() entered."));

#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   return m_tCSCProp.brightness;
#else
   ETG_TRACE_USR1(("s16GetBrightnessCSC: CSC unavailable."));
   return 100;
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
}

/*******************************************************************************
*
* FUNCTION:    s32SetBrightnessCSC()
*
* DESCRIPTION:    set brightness in percent and update CSC
*
* PARAMETER:    tS16   { 0 ... 100 }
*
* RETURNVALUE:    tS32   { less 0 - failed | else success }
*
*******************************************************************************/
tS32 dispvidctrl_tclDisplaySettings::s32SetBrightnessCSC(tS16 s16Percent)
{
   ETG_TRACE_USR3(("s32SetBrightnessCSC(percent: %d) entered.", s16Percent));

   tS32 _s32Return = -1;
#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   if (DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MIN <= s16Percent && DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MAX >= s16Percent)
   {
	   m_tCSCProp.brightness = s16Percent;
      dp_tclDimmingDPDISP_CSCBrigthness _oDpCSCBrightness;
      _oDpCSCBrightness.vSetData(s16GetBrightnessCSC());
      dp_tclEarlyConfigCSCTrBrightnessValue _oDpEC_CSC;
      _oDpEC_CSC.vSetData(s16GetBrightnessCSC());
      _s32Return = s32UpdateCSC();
      if (0 > _s32Return)
      {
         ETG_TRACE_FATAL(("s32SetBrightnessCSC: brightness %d could not set! (error code: %d)", s16Percent, _s32Return));
      }
   }
   else
   {
      ETG_TRACE_FATAL(("s32SetBrightnessCSC: brightness %d out of range", s16Percent));
   }
#else
   ETG_TRACE_USR1(("s32SetBrightnessCSC: CSC unavailable."));
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   return _s32Return;
}

/*******************************************************************************
*
* FUNCTION:    s16GetBrightnessOffsetCSC()
*
* DESCRIPTION:  get brightness offset in percent
*
* PARAMETER:    none
*
* RETURNVALUE:    tS16   { 0 ... 100 }
*
*******************************************************************************/
tS16 dispvidctrl_tclDisplaySettings::s16GetBrightnessOffsetCSC(tVoid) const
{
   ETG_TRACE_USR4(("s16GetBrightnessOffsetCSC() entered."));

#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   //return m_tCSC_CPROP.brightness_off;
   return m_s16BrightnessOffset;
#else
   ETG_TRACE_USR1(("s16GetBrightnessOffsetCSC: CSC unavailable."));
   return 100;
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
}

/*******************************************************************************
*
* FUNCTION:    s32SetBrightnessOffsetCSC()
*
* DESCRIPTION:    set brightness offset in percent and update CSC
*
* PARAMETER:    tS16   { 0 ... 100 }
*
* RETURNVALUE:    tS32   { less 0 - failed | else success }
*
*******************************************************************************/
tS32 dispvidctrl_tclDisplaySettings::s32SetBrightnessOffsetCSC(tS16 s16Percent)
{
   ETG_TRACE_USR3(("s32SetBrightnessOffsetCSC(percent: %d) entered.", s16Percent));

   tS32 _s32Return = -1;
#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   if (DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MIN <= s16Percent && DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MAX >= s16Percent)
   {
	   tS16 _s16EnhancePercent = s16GetEnhanceBrightnessOffset();
	   tS16 _s16Percent = s16Percent + _s16EnhancePercent + 1;
	   if (DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MAX < _s16Percent)
		   _s16Percent = DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MAX;
	   else if (DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MIN > _s16Percent)
		   _s16Percent = DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MIN;
	   m_s16BrightnessOffset = s16Percent;
	   m_tCSCProp.brightness_off = _s16Percent;
      dp_tclDimmingDPDISP_CSCBrigthnessOffset _oDpCSCBrightnessOffset;
      _oDpCSCBrightnessOffset.vSetData(s16GetBrightnessOffsetCSC());
      dp_tclEarlyConfigCSCTrBrightnessOffsetValue _oDpEC_CSC;
      _oDpEC_CSC.vSetData(m_tCSCProp.brightness_off);

      if(m_bDisplaySettingsInitDone)
      {
         _s32Return = s32UpdateCSC();
         if (0 > _s32Return)
         {
            ETG_TRACE_FATAL(("s32SetBrightnessOffsetCSC: brightness offset %d percent and enhance brightness offset %d percent could not set with %d percent! (error code: %d)",
       		     s16Percent, _s16EnhancePercent, _s16Percent, _s32Return));
         }
         else
         {
             ETG_TRACE_FATAL(("s32SetBrightnessOffsetCSC: brightness offset %d percent and enhance brightness offset %d percent set successful with %d percent. (error code: %d)",
        		     s16Percent, _s16EnhancePercent, _s16Percent, _s32Return));
         }
      }
      else
      {
         // Init not yet done, don't sent CSC value
         ETG_TRACE_FATAL(("s32SetBrightnessOffsetCSC: Init not yet done, don't sent CSC value"));
         _s32Return = 0;
      }
   }
   else
   {
      ETG_TRACE_FATAL(("s32SetBrightnessOffsetCSC: brightness offset %d out of range", s16Percent));
   }
#else
   ETG_TRACE_USR1(("s32SetBrightnessOffsetCSC: CSC unavailable."));
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   return _s32Return;
}

/*******************************************************************************
*
* FUNCTION:    s16GetContrastCSC()
*
* DESCRIPTION:    get contrast in percent
*
* PARAMETER:    none
*
* RETURNVALUE:    tS16   { 0 ... 100 }
*
*******************************************************************************/
tS16 dispvidctrl_tclDisplaySettings::s16GetContrastCSC(tVoid) const
{
   ETG_TRACE_USR4(("s16GetContrastCSC() entered."));

#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   return m_tCSCProp.contrast;
#else
   ETG_TRACE_USR1(("s16GetContrastCSC: CSC unavailable."));
   return 100;
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
}

/*******************************************************************************
*
* FUNCTION:    s32SetContrastCSC()
*
* DESCRIPTION:    set contrast in percent and update CSC
*
* PARAMETER:    tS16   { 0 ... 100 }
*
* RETURNVALUE:    tS32   { less 0 - failed | else success }
*
*******************************************************************************/
tS32 dispvidctrl_tclDisplaySettings::s32SetContrastCSC(tS16 s16Percent)
{
   ETG_TRACE_USR3(("s32SetContrastCSC(percent: %d) entered.", s16Percent));

   tS32 _s32Return = -1;
#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   if (DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MIN <= s16Percent && DISPVIDCTRL_DIM_C_S16_DISPLAYSETTINGS_PERCENT_MAX >= s16Percent)
   {
	   m_tCSCProp.contrast = s16Percent;
      dp_tclDimmingDPDISP_CSCContrast _oDpCSCContrast;
      _oDpCSCContrast.vSetData(s16GetContrastCSC());
      dp_tclEarlyConfigCSCTrContrastValue _oDpEC_CSC;
      _oDpEC_CSC.vSetData(s16GetContrastCSC());
      _s32Return = s32UpdateCSC();
      if (0 > _s32Return)
      {
         ETG_TRACE_FATAL(("s32SetContrastCSC: contrast %d could not set! (error code: %d)", s16Percent, _s32Return));
      }
   }
   else
   {
      ETG_TRACE_FATAL(("s32SetContrastCSC: contrast %d out of range", s16Percent));
   }
#else
   ETG_TRACE_USR1(("s32SetContrastCSC: CSC unavailable."));
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   return _s32Return;
}

/*******************************************************************************
*
* FUNCTION:    vInitDRM()
*
* DESCRIPTION:    initiate access to direct rendering manager
*               - create DRM helper object and set display type before
*               - sets gamma exponent and update gamma adjustment
*
* PARAMETER:    none
*
* RETURNVALUE:    none
*
*******************************************************************************/
tVoid dispvidctrl_tclDisplaySettings::vInitDRM(tVoid)
{
   ETG_TRACE_USR3(("vInitDRM() entered."));

#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_GAMMA
   ilmErrorTypes _tResult = enInitILM();
   if (ILM_SUCCESS == _tResult)
   {
	   tBool _bDoResetDisplaySettings = FALSE;
	   // loading last mode data
	   // gamma adjustment
	   dp_tclDimmingDPDISP_CSCGammaLevel _oDpCSCGammaLevel;
	   m_u8GammaLevel = _oDpCSCGammaLevel.tGetData();
	   if (DISPVIDCTRL_C_U8_LEVEL_MAX < m_u8GammaLevel)
	 	  _bDoResetDisplaySettings = TRUE;
	   dp_tclDimmingDPDISP_CSCGamma _oDpCSCGamma;
	   m_u16DRM_GammaExponent = _oDpCSCGamma.tGetData();
	   if (TRUE == _bDoResetDisplaySettings)
	   {
		   (tVoid)s32ResetDRM();
	   }
	   else
	   {
		   (tVoid)s32UpdateDRM();
	   }
   }
#else
   ETG_TRACE_USR1(("vInitDRM: gamma adjustment unavailable."));
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_GAMMA
}

/*******************************************************************************
*
* FUNCTION:    s32UpdateDRM()
*
* DESCRIPTION:    update direct rendering manager
*               - set gamma exponent
*
* PARAMETER:    none
*
* RETURNVALUE:    tS32   { less 0 - failed | else success }
*
*******************************************************************************/
tS32 dispvidctrl_tclDisplaySettings::s32UpdateDRM(tVoid)
{
   ETG_TRACE_USR3(("s32UpdateDRM() entered."));

#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_GAMMA
   ilmErrorTypes _tResult = enInitILM();
   if (ILM_SUCCESS == _tResult)
   {
	   t_ilm_display _tDisplay = (t_ilm_display)m_u8DRM_MainHmi_ScreenID;
	   t_ilm_float _tGamma = (((t_ilm_float)u16GetGammaExponent())/1000.0);
	   ilmErrorTypes _tReturn = ilm_displaySetGamma(_tDisplay, _tGamma);
	   ETG_TRACE_FATAL(("s32UpdateDRM: ilm_displaySetGamma(display: %d, gamma: %f) returns %d",
			   _tDisplay, _tGamma, _tReturn ));
   }
   if (ILM_SUCCESS == _tResult)
   {
	   return 1;
   }
#else
   ETG_TRACE_USR1(("s32UpdateDRM: gamma adjustment unavailable."));
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_GAMMA
   return -1;
}

/*******************************************************************************
*
* FUNCTION:    s32ResetDRM()
*
* DESCRIPTION:    reset direct rendering manager
*               - reset gamma exponent
*               - reset black level
*
* PARAMETER:    none
*
* RETURNVALUE:    tS32   { less 0 - failed | else success }
*
*******************************************************************************/
tS32 dispvidctrl_tclDisplaySettings::s32ResetDRM(tVoid)
{
   ETG_TRACE_USR3(("s32ResetDRM() entered."));
#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_GAMMA
   ilmErrorTypes _tResult = enInitILM();
   if (ILM_SUCCESS == _tResult)
   {
	   s32SetGammaLevel(DISPVIDCTRL_DIM_C_U8_DISPLAYSETTINGS_GAMMA_LEVEL_DEFAULT);
	   return s32UpdateDRM();
   }
   return -1;
#else
   ETG_TRACE_USR1(("s32ResetDRM: gamma adjustment unavailable."));
   return 1;
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_GAMMA
}

/*******************************************************************************
*
* FUNCTION:    vTraceDRM()
*
* DESCRIPTION:    trace direct rendering manager properties
*
* PARAMETER:
*
* RETURNVALUE:    none
*
*******************************************************************************/
tVoid dispvidctrl_tclDisplaySettings::vTraceDRM(const char* strIdentifier) const
{
   ETG_TRACE_USR4(("vTraceCSC() entered."));

   ETG_TRACE_USR2(("%026s: <<< direct rendering manager - gamma adjustment >>> ", strIdentifier ));
#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_GAMMA
   ETG_TRACE_USR2(("%26s: %30s=%u / %f ",
         strIdentifier, "        DRM_GAMMAEXP", m_u16DRM_GammaExponent, (((tF32)m_u16DRM_GammaExponent) / 1000.0) ));
#else
   ETG_TRACE_USR1(("vTraceDRM: gamma adjustment unavailable."));
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_GAMMA
}

/*******************************************************************************
*
* FUNCTION:    u16GetGammaExponent()
*
* DESCRIPTION:    get gamma exponent
*
* PARAMETER:    none
*
* RETURNVALUE:    tU16   {like 1000 for 1.000}
*
*******************************************************************************/
tU16 dispvidctrl_tclDisplaySettings::u16GetGammaExponent(tVoid) const
{
   ETG_TRACE_USR4(("u16GetGammaExponent() entered." ));

#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_GAMMA
   return m_u16DRM_GammaExponent;
#else
   ETG_TRACE_USR1(("u16GetGammaExponent: gamma adjustment unavailable."));
   return DISPVIDCTRL_DIM_C_U16_DISPLAYSETTINGS_GAMMA_EXPONENT_DEFAULT;
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_GAMMA
}

/*******************************************************************************
*
* FUNCTION:    s32SetGammaExponent()
*
* DESCRIPTION:    set gamma exponent and update direct rendering manager
*
* PARAMETER:    tU16   {like 1000 for 1.000}
*
* RETURNVALUE:    tS32   { less 0 - failed | else success }
*
*******************************************************************************/
tS32 dispvidctrl_tclDisplaySettings::s32SetGammaExponent(tU16 u16Exponent)
{
   ETG_TRACE_USR3(("s32SetGammaExponent(exponent: %d) entered.", u16Exponent ));

   tS32 _s32Return = -1;
#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_GAMMA
   if (DISPVIDCTRL_DIM_C_U16_DISPLAYSETTINGS_GAMMA_EXPONENT_RANGE_MIN <= u16Exponent && DISPVIDCTRL_DIM_C_U16_DISPLAYSETTINGS_GAMMA_EXPONENT_RANGE_MAX >= u16Exponent)
   {
      m_u16DRM_GammaExponent = u16Exponent;
      dp_tclDimmingDPDISP_CSCGamma _oDpCSCGamma;
      _oDpCSCGamma.vSetData(u16GetGammaExponent());
      dp_tclEarlyConfigCSCTrGammaFactor _oDpEC_CSC;
      _oDpEC_CSC.vSetData(((tF64)u16GetGammaExponent())/1000.0);
      _s32Return = s32UpdateDRM();
   }
   else
   {
      ETG_TRACE_USR4(("s32SetGammaExponent: gamma exponent %d out of range!", u16Exponent ));
   }
#else
   ETG_TRACE_USR1(("s32SetGammaExponent: gamma adjustment unavailable."));
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_GAMMA
   return _s32Return;
}

/*******************************************************************************
*
* FUNCTION:    s32EnhanceBrightness()
*
* DESCRIPTION: set enhance brightness and triggers a CSC update
*
* PARAMETER:   tU8 u8Level			{0 ... 32}
*              tU8 u8Mode			{DISPVIDCTRL_U8_DIMMODECAR_DAY | DISPVIDCTRL_U8_DIMMODECAR_NIGHT}
*
* RETURNVALUE: tS32   { less 0 - failed | else success }
*
*******************************************************************************/
tS32 dispvidctrl_tclDisplaySettings::s32EnhanceBrightness(tU8 u8Level, tU8 u8Mode)
{
   ETG_TRACE_USR3(("s32EnhanceBrightness(level: %d, mode: %d) entered.", u8Level, u8Mode ));

   tS32 _s32Return = -1;
#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
#ifdef VARIANT_S_FTR_ENABLE_BRIGHTNESS_ENHANCEMENT
   switch (u8Mode)
   {
   case DISPVIDCTRL_U8_DIMMODECAR_DAY:
   case DISPVIDCTRL_U8_DIMMODECAR_NIGHT:
	   {
		   if (DISPVIDCTRL_C_U16_ARRAYELEMENTS(m_atCSCEnhanceBrigtnessOffset) > (tU16)u8Level)
		   {
			   tU8 _u8Idx = (DISPVIDCTRL_U8_DIMMODECAR_NIGHT == u8Mode) ? 1 : 0;
			   _s32Return = 0;
			   m_u8DimMode = u8Mode;
			   m_au8DimBrightnessLevel[_u8Idx] = u8Level;
			   _s32Return = s32SetBrightnessOffsetCSC(s16GetBrightnessOffsetCSC());
		   }
		   else
		   {
			   ETG_TRACE_FATAL(("s32EnhanceBrightness: level %d out of range!", u8Level ));
		   }
	   }
	   break;
   default:
	   {
		   ETG_TRACE_FATAL(("s32EnhanceBrightness: mode %d out of range!", u8Mode ));
	   }
	   break;
   }
#endif //#ifdef VARIANT_S_FTR_ENABLE_BRIGHTNESS_ENHANCEMENT
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   return _s32Return;
}

/*******************************************************************************
*
* FUNCTION:    s16GetEnhanceBrightnessOffset()
*
* DESCRIPTION: get the enhance brightness offset depending of curremt
*
* PARAMETER:   tU8 u8Level			{0 ... 32}
*              tU8 u8Mode			{DISPVIDCTRL_U8_DIMMODECAR_DAY | DISPVIDCTRL_U8_DIMMODECAR_NIGHT}
*
* RETURNVALUE: tS32   { less 0 - failed | else success }
*
*******************************************************************************/
tS16 dispvidctrl_tclDisplaySettings::s16GetEnhanceBrightnessOffset(tVoid) const
{
   ETG_TRACE_USR3(("s16GetEnhanceBrightnessOffset() entered." ));

#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
#ifdef VARIANT_S_FTR_ENABLE_BRIGHTNESS_ENHANCEMENT
   switch (m_u8DimMode)
   {
   case DISPVIDCTRL_U8_DIMMODECAR_DAY:
   case DISPVIDCTRL_U8_DIMMODECAR_NIGHT:
	   {
		   tU8 _u8Idx = (DISPVIDCTRL_U8_DIMMODECAR_NIGHT == m_u8DimMode)?1:0;
		   tU8 _u8Level = m_au8DimBrightnessLevel[_u8Idx];
		   if ((tU8)DISPVIDCTRL_C_U16_ARRAYELEMENTS(m_atCSCEnhanceBrigtnessOffset) > _u8Level)
		   {
			   tS16 _s16Percent = (m_atCSCEnhanceBrigtnessOffset[_u8Level].s16DayDeciPercent / 10);
			   if(DISPVIDCTRL_U8_DIMMODECAR_NIGHT == m_u8DimMode)
				   _s16Percent = (m_atCSCEnhanceBrigtnessOffset[_u8Level].s16NightDeciPercent / 10);
			   ETG_TRACE_USR1(("s16GetEnhanceBrightnessOffset: enhance brightness offset %d [percent] ", _s16Percent ));
			   return _s16Percent;
		   }
		   else
		   {
			   ETG_TRACE_USR1(("s16GetEnhanceBrightnessOffset: level %d out of range!", _u8Level ));
		   }
	   }
	   break;
   default:
	   {
		   ETG_TRACE_USR1(("s16GetEnhanceBrightnessOffset: mode %d out of range!", m_u8DimMode ));
	   }
	   break;
   }
#endif //#ifdef VARIANT_S_FTR_ENABLE_BRIGHTNESS_ENHANCEMENT
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   return 0;
}

/*******************************************************************************
*
* FUNCTION:    enInitILM()
*
* DESCRIPTION: initialize layer manager interface
*
* PARAMETER:   none
*
* RETURNVALUE: ilmErrorTypes   { ILM_SUCCESS - success | else failed }
*
*******************************************************************************/
#if (defined VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_GAMMA) || (defined VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC)
ilmErrorTypes dispvidctrl_tclDisplaySettings::enInitILM(tVoid)
{
   ETG_TRACE_USR4(("enInitILM() entered." ));
   ilmErrorTypes _enReturn = ILM_SUCCESS;
   if (TRUE != m_bILMInit)
   {
	   _enReturn = ilm_init();
	   ETG_TRACE_USR2(("enInitILM: ilm_init returns %d", _enReturn ));
	   m_bILMInit = (ILM_SUCCESS == _enReturn);
   }
   else
   {
	   ETG_TRACE_USR4(("enInitILM: ilm_init is initialized " ));
   }
   return _enReturn;
}
#endif //#if (defined VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_GAMMA) || (defined VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC)
/******************************************************************************/
/* FUNCTION     vSetDimmingModeRvc                                            */
/******************************************************************************/
/**
*  \brief       When Dimming Mode RVC will be active, we change Dimming
*               setting to default values temporarely.
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclDisplaySettings::vSetDimmingModeRvc(tVoid)
{
	ETG_TRACE_FATAL(("vSetDimmingModeRvc entered, value of m_bCscInitDone is %d",m_bCscInitDone));
#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
   ETG_TRACE_FATAL(("vSetDimmingModeRvc"));
   if (m_bCscInitDone == TRUE)
   {
      (tVoid) s32UpdateCSC();
      vUpdateDisplaySettings(FALSE);
   }
#else
   ETG_TRACE_USR1(("vInitCSC: CSC unavailable."));
#endif //#ifdef VARIANT_S_FTR_ENABLE_DISPLAYSETTINGS_CSC
}
/******************************************************************************/
/* FUNCTION     vUpdateDisplaySettings                                            */
/******************************************************************************/
/**
*  \brief       This function is used to update the Minimum Brightness when RVC
                is activated and also to bring back the Previous Brightness
				When RVC is deactivated
*               
*  \param       none
*  \return      none
*/
/******************************************************************************/

tVoid dispvidctrl_tclDisplaySettings::vUpdateDisplaySettings(tBool bIsInit)
{
	ETG_TRACE_USR1(("Entered vUpdateDisplaySettings, value of bIsInit is %d",bIsInit));
	tBool bDimmingModeRvc = FALSE;

	tBool _bIsNight = FALSE;
	if(m_poClientDim != NULL)
	{
		switch(m_poClientDim->u8ReturnDimmingStatus())
		{
		case ::dimming_inc_fi::eDimModeStatus_DAY:
			{
				_bIsNight = FALSE;
			}
			break;
		case ::dimming_inc_fi::eDimModeStatus_NIGHT:
			{
				_bIsNight = TRUE;
			}
			break;
		case ::dimming_inc_fi::eDimModeStatus_AUTO_DAY:
			{
				_bIsNight = FALSE;
			}
			break;
		case ::dimming_inc_fi::eDimModeStatus_AUTO_NIGHT:
			{
				_bIsNight = TRUE;
			}
			break;
		default:
			{
				// Do nothing
			}
			break;
		}
	}
	ETG_TRACE_USR1(("vUpdateDisplaySettings: bIsNight=%s",_bIsNight==TRUE?"TRUE":"FALSE"));
	//Display Settings changes
	if (m_poSyncHandler)
	{
		bDimmingModeRvc = m_poSyncHandler->bGetDimmingModeRvc();
	}
	
	if (FALSE == bDimmingModeRvc)
	{
		ETG_TRACE_USR1(("vUpdateDisplaySettings: RVC Mode == FALSE"));
		if(FALSE == bIsInit)
		{
			//After coming out of camera screen, reset the brightness value to original one
			dp_tclDimmingDPDISP_BrightnessLevelExtended_Night _oBrightnessLevelNight;
			_oBrightnessLevelNight.vSetData(m_NightBrightness);

			dp_tclDimmingDPDISP_BrightnessLevelExtended_Day _oBrightnessLevelDay;
			_oBrightnessLevelDay.vSetData(m_DayBrightness);
			if(_bIsNight == TRUE)
			{
				tChar _strIfName[] = "I_dispvidctrl_tclClientDimming";
				_cpoMain->theServer()->vPostMsgDimBrightnessChanged(_strIfName, m_NightBrightness);
			}
			else
			{
				tChar _strIfName[] = "I_dispvidctrl_tclClientDimming";
				_cpoMain->theServer()->vPostMsgDimBrightnessChanged(_strIfName, m_DayBrightness);
			}
		
		}
	}
	else
	{
		//Display Settings changes for Brightness 
		ETG_TRACE_USR1(("vUpdateDisplaySettings: RVC Mode == TRUE"));
		dp_tclDimmingDPDISP_BrightnessLevelExtended_Night _oBrightnessLevelNight;
		m_NightBrightness = _oBrightnessLevelNight.tGetData();

		dp_tclDimmingDPDISP_BrightnessLevelExtended_Day _oBrightnessLevelDay;
		m_DayBrightness = _oBrightnessLevelDay.tGetData();

		tChar _strIfName[] = "I_dispvidctrl_tclClientDimming";
		_cpoMain->theServer()->vPostMsgDimBrightnessChanged(_strIfName, FMVSS_BRIGHTNESS_DEFAULT);
	
	}
}




