// Description and history of this file at the end.
// File name for preprozessor commands
#define CSM_STACK_OSAL_SELF

// csm_stack_osal.c is a file with adaptations to the environment/operating system
// according V-team CAN-SW architecture implementation rules
#define CSM_STACK_ENV_SELF

//******************************************************************************
//*  1)system- and project- includes
//*  2)needed interfaces from external components
//*  3)internal and external interfaces from this component
//******************************************************************************
// 1) 2)

#include "trace_interface.h"
#include <pthread.h>

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define DEB_PRINTF    printf                   // currently map debug print outs to printf

// 3)
#include "csm_stack_i.h"

// CBR
#if (CSM_S_CBR_AVAILABLE == CSM_C_F_ON)
  #include "cbr_i.h"                           // for Timer callbacks
#endif // CSM_S_CBR_AVAILABLE

// CNM
#if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
  #if (CSM_S_OSEK_NM == CSM_C_F_ON)
    #include "csm_c.h"                         // for Timer callbacks
  #endif // CSM_S_OSEK_NM

  #if (CSM_S_OSEKI_NM == CSM_C_F_ON)
    #include "csm_c.h"                         // for Timer callbacks
  #endif // CSM_S_OSEKI_NM

  #if (CSM_S_VWHIGH_NM == CSM_C_F_ON)
    #include "csm_c.h"                         // for Timer callbacks
  #endif // CSM_S_VWHIGH_NM
#endif // CSM_S_CNM_AVAILABLE

#include "csm_stack_m.h"
#include "csm_i.h"
#include "csm_m.h"
#include "csm_apl.h"                           // for SLOT configuration

#include "csm_proxy_i.h"                       // for prototypes

/*************** ETG Tracing ******/
#define TR_CLASS_CSMOSAL (TR_COMP_CSM + 0x13)

#if (CSM_S_TRACE == CSM_C_F_ON)                              // wir tracen generell ?
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)                        // wir wollen ETG nutzen ?
    #define ETRACE_S_IMPORT_INTERFACE_GENERIC
    //#define ETRACE_S_IMPORT_INTERFACE_ERRMEM               // was nicht gebraucht wird, darf auskommentiert werden
    #include "etrace_if.h"

    #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN                      // RN AIVI: wird in ai_projects\config\xml\ai_projects_prod.xml gesetzt
      #include "csm_etg_trace.h"                             // somit machen wir ETG tracing
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CSMOSAL
      //#define ETG_INLINE                                   // brauchen wir scheinbar nicht
      #include "trcGenProj/Header/csm_stack_osal.c.trc.h"    // so heisst unser generiertes trc file
    #endif // VARIANT_S_FTR_ENABLE_TRC_GEN
  #endif // ETG Trace
#endif // CSM_S_TRACE

/*******************************************************************************
 * defines and macros (scope: file local)
 ******************************************************************************/
#define PVOID  void*

//******************************************************************************
//* typedefs (scope: file local)
//******************************************************************************
// Typ: pointer to callback function (has argument WORD and returns VOID)
// typedef void (* can_tpfTimerCbftk) (DWORD);

/*******************************************************************************
 * variable definition (scope: file local)
 ******************************************************************************/
static LONG lUpStreamThreadId;                             // Thread ID

static OSAL_tThreadID csm_hDownStreamThread = 0;           // Thread Handle
static OSAL_tEventHandle csm_hDownStreamEvent = 0;         // Event Handle

static OSAL_tEventHandle csm_hUpStreamEvent = 0;           // Event Handle
static OSAL_tTimerHandle csm_arTimerHandle[CSM_MAX_TIMER]; // Timer Handles

static pthread_mutex_t      Mutex;                         // for ?
static pthread_mutexattr_t  Attr;                          // for ?

// global vars helping us with trace registration
static BOOLEAN hCntxtTrace        = FALSE;  // no trace registered
static WORD uwTraceStartDelayCnt  = 0;      // start value for registration delay

//*******************************************************************************
//* variable definition (scope: CAN stack)
//*******************************************************************************
// none

//*******************************************************************************
//* function prototypes (scope: file local)
//*******************************************************************************
static void CSM_vTraceOutTTFis(const WORD, const WORD, const WORD, const BYTE *, const WORD);
CSM_STATIC void csm_vCsmTraceCallback(BYTE * pu8Data);

//*******************************************************************************
//* function prototypes (scope: file global)
//*******************************************************************************
extern tVoid CSM_Proc_vSetDatapoolPDDItem(BYTE, BYTE);
extern tVoid CSM_Proc_vInitDatapoolPDD(tVoid);

//*******************************************************************************
//* function implementation (scope: file local = static)
//*******************************************************************************

#ifdef CSM_STACK_OSAL_SELF
// make LINT happy
#endif

/*******************************************************************************
 * function    csm_stack_vStartTraceCallback
 * \doxydocu
 * \brief      This function initialzes the trace callback function
 *
 * \param[in]  VOID
 *
 * \return     VOID
 *
 * \access     CAN-Downstreamtask
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 * 02.06.08 CM-DI/VTeam Brunotte
 * With the use of the macro CSM_S_USE_TRACE_CORE the trace TKernel driver is obsolete. CSM uses the
 * trace core API.
 * 11.11.07 CM-DI/VTeam Brunotte
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
#if (CSM_S_TRACE == CSM_C_F_ON)
void csm_stack_vStartTraceCallback(void)
{
  {
    OSAL_trIOCtrlLaunchChannel  MyTraceChannel;
    tS32 s32Err;

    OSAL_tIODescriptor desc = OSAL_IOOpen(OSAL_C_STRING_DEVICE_TRACE, OSAL_EN_READWRITE);
    if (desc != OSAL_ERROR)
    {
      MyTraceChannel.enTraceChannel = TR_TTFIS_FD_CSM;
      MyTraceChannel.pCallback      = (OSAL_tpfCallback) csm_vCsmTraceCallback;

      s32Err = OSAL_s32IOControl(desc, OSAL_C_S32_IOCTRL_CALLBACK_REG, (tS32) &MyTraceChannel);
      if (s32Err == OSAL_OK)
      {
        // good -> mark trace function as registered
        hCntxtTrace = TRUE;
      }
      else
      {
        // bad
        ETG_TRACE_USR4(("csm_stack_vStartTraceCallback: Error TracePlugin"));
      }
      // close trace device and clear channel
      OSAL_s32IOClose(desc);
    }
  }

  #ifdef QWERT_MP
  if (TR_chan_acess_bRegChan(TR_TTFIS_FD_CSM,(TRACE_CALLBACK) csm_vCsmTraceCallback) == TRUE)
  {
    hCntxtTrace = TRUE;
  }
  else
  {
    ETG_TRACE_FATAL(("TR_chan_acess_bRegChan(TR_TTFIS_FD_CSM) failed"));

    hCntxtTrace = FALSE;                                  // Mark trace function as un-registered
    #if (CSM_S_CYCLIC_TASK == CSM_C_F_ON)
    uwTraceStartDelayCnt = CSM_C_DELAYED_TRACE_STARTTIME; // Try it again after delayed time
    #endif
  }
  #endif
}

/*******************************************************************************
 * function    csm_stack_vStopTraceCallback
 * \doxydocu
 * \brief      This function initialzes the trace callback function
 *
 * \param[in]  VOID
 *
 * \return     VOID
 *
 * \access     CAN-Downstreamtask
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 * 02.06.08 CM-DI/VTeam Brunotte
 * With the use of the macro CSM_S_USE_TRACE_CORE the trace TKernel driver is obsolete. CSM uses the
 * trace core API.
 * 11.11.07 CM-DI/VTeam Brunotte
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void csm_stack_vStopTraceCallback (void)
{
  #ifdef QWERT
  if (hCntxtTrace > TRUE)
  {
    if (TR_chan_acess_bUnRegChan(TR_TTFIS_FD_CSM, (TRACE_CALLBACK) csm_vCsmTraceCallback)== TRUE)
    {
      hCntxtTrace = FALSE;      // Mark trace function as un-registered
      uwTraceStartDelayCnt = 0; // Prepare delay counter for next start
    }
    else
    {
      CSM_vAssert(FALSE); /*lint !e506*/ // Faulty channel unregistration shall cause an assert - 27.09.2010
    }
  }
  #endif
}
#endif // CSM_S_TRACE

//------------------------------------------------------------------------------------------------------------------------------------------------
// thread functions
//------------------------------------------------------------------------------------------------------------------------------------------------

//************************************************************************************************************************************************
static tVoid csm_vDownStreamThread(tPVoid pvData)
{
  // this thread simulates the "downstream task" of the system environment
  // after start it is running endlessly, mainly waiting for events
  DWORD hEvRequest;
  DWORD dwEventMask = 0x01;

  do
  {
    // wait for dedicated event, endlessly
    // the thread is suspended at this point !!!
    if (OSAL_OK == OSAL_s32EventWait(csm_hDownStreamEvent, dwEventMask, OSAL_EN_EVENTMASK_OR, OSAL_C_TIMEOUT_FOREVER, &hEvRequest))
    {
      if (OSAL_OK != OSAL_s32EventPost(csm_hDownStreamEvent, ~hEvRequest, OSAL_EN_EVENTMASK_AND))
      {
      }

      // the thread is re-actived by the OS if anything has happened
      // if an object has received, check for its type
      switch(hEvRequest)
      {
        // the event has been received
        case 0x01:
        {
          // on event pass execution time to CSM
          CSM_vDownstreamTask();
        }
        break;

        default:
        {
          // do nothing
        }
        break;
      }

      #if (CSM_S_TRACE == CSM_C_F_ON)
      // see above CSM_C_DELAYED_TRACE_STARTTIME
      if (uwTraceStartDelayCnt > 0)
      {
        // delay not elapsed, so go on delaying
        uwTraceStartDelayCnt--;
      }
      else if ((uwTraceStartDelayCnt == 0) && (hCntxtTrace == FALSE))
      {
        // delay elapsed and no trace registered yet
        csm_stack_vStartTraceCallback();
      }
      else
      {
        // nothing to do
      }
      #endif // CSM_S_TRACE
    } // event received
  }
  while(TRUE);/*lint !e506 yes, this is an endless loop */
}/*lint !e715 !e818 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally */

//************************************************************************************************************************************************
static tVoid csm_vUpStreamThread(tPVoid pvData)
{
  // this thread simulates the "upstream task" of the target environment
  DWORD hEvRequest;
  DWORD dwEventMask = 0x01;

  do
  {
    // wait for dedicated event, endlessly
    // the thread is suspended at this point !!!
    if (OSAL_OK == OSAL_s32EventWait(csm_hUpStreamEvent, dwEventMask, OSAL_EN_EVENTMASK_OR, OSAL_C_TIMEOUT_FOREVER, &hEvRequest))
    {
      // clear event
      if (OSAL_OK != OSAL_s32EventPost(csm_hUpStreamEvent, ~hEvRequest, OSAL_EN_EVENTMASK_AND))
      {
      }

      // the thread is re-actived by the OS if anything has happened
      // if an object has received, check for its type
      switch(hEvRequest)
      {
        case 0x01:
        {
          // on event pass execution time to CSM
          CSM_vUpstreamTask();
        }
        break;

        default:
        {
          //do nothing
        }
        break;
      }
    }
  }
  while(TRUE);/*lint !e506 yes, this is an endless loop */
}/*lint !e715 !e818 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally*/

//************************************************************************************************************************************************
static void csm_vInitTimerSystem(void)
{
  DWORD dwCount;

  // loop over all configured timers, timer 0 is a dummy, so we start with 1
  for(dwCount=1; dwCount < CSM_C_MAX_TIMER; dwCount++) //lint !e681 Reason: loop is not entered if no timer is defined
  {
    // timers are configured in csm..._mp.h
    if (OSAL_OK != OSAL_s32TimerCreate((OSAL_tpfCallback)csm_arTimer[dwCount].vpfnCallback, (tPVoid)csm_arTimer[dwCount].dwValue, &csm_arTimerHandle[dwCount]))
    {
      CSM_vAssert(FALSE); //lint !e506 self_authorized, since I want this assertion if timer creation failed
    }
  }
}

/*******************************************************************************
 * function implementation (scope: CAN stack and operating system)
 *******************************************************************************/
/*******************************************************************************
 * function    CSM_vTraceOut
 * \doxydocu
 * \brief
 *
 * \param[in ] wTraceOutLevel
 *              Trace Level.
 *
 * \param[in ] wTraceOutInterface
 *              Interface number.
 *
 * \param[in]  wTraceOutFunktion
 *              Funktion number
 *
 * \param[in]  *pbTraceOutData
 *              Pointer to trace data buffer.
 *
 * \param[in]  wTraceOutDataLength
 *              Number of bytes in trace data buffer.
 *
 * \return     VOID
 *
 * \access     all
 * \reentrant  yes
 *
 * \lastreview
 *
 * \history_begin
 * 12.11.04 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CSM_vTraceOut(const WORD wTraceOutLevel, const WORD wTraceOutInterface, const WORD wTraceOutFunktion,
                   const BYTE * pbTraceOutData, const WORD wTraceOutDataLength)
{
  // This function is implementing the old standard trace function, used in former times
  // before ETG trace was established.


  // NO trace enabled -> console output via printf
  #if (CSM_S_TRACE != CSM_C_F_ON)
  DWORD i;

  DEB_PRINTF("Trace: #%02x Funktion: %02x - ", wTraceOutInterface, wTraceOutFunktion);
  for (i=0; i<wTraceOutDataLength; i++)
  {
    DEB_PRINTF(" %02x", *(pbTraceOutData+i));
  }
  DEB_PRINTF("\n");
  #endif

  // ETG trace enabled -> convert old trace output to ETG output
  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR1(("CSM_vTraceOut: wTraceOutLevel:%d  wTraceOutInterface:%d  wTraceOutFunktion:%d  pbTraceOutData:%d  wTraceOutDataLength:%d)", wTraceOutLevel, wTraceOutInterface, wTraceOutFunktion, pbTraceOutData, wTraceOutDataLength));
  #endif
  #endif
  #endif

  // windows
  #ifdef _WIN_ONLY
  // protect trace output against manipulation from elsewhere
  EnterCriticalSection(&(CSM_TRACE_hCriticalSection));
  #endif

  // TTFis output (whatever this means ????)
  CSM_vTraceOutTTFis(wTraceOutLevel, wTraceOutInterface, wTraceOutFunktion, pbTraceOutData, wTraceOutDataLength);

  #ifdef _WIN_ONLY
  LeaveCriticalSection(&(CSM_TRACE_hCriticalSection));
  #endif
}

//*******************************************************************************
// Trace function for TTFis output
//*******************************************************************************
static void CSM_vTraceOutTTFis(const WORD wTraceOutLevel, const WORD wTraceOutInterface, const WORD wTraceOutFunktion,
                               const BYTE * pbTraceOutData, const WORD wTraceOutDataLength)
{
  // handle LINT warnings
  CSM_M_PARAMETER_INTENTIONALLY_UNUSED(wTraceOutLevel);
  CSM_M_PARAMETER_INTENTIONALLY_UNUSED(wTraceOutInterface);
  CSM_M_PARAMETER_INTENTIONALLY_UNUSED(wTraceOutFunktion);
  CSM_M_PARAMETER_INTENTIONALLY_UNUSED(pbTraceOutData);
  CSM_M_PARAMETER_INTENTIONALLY_UNUSED(wTraceOutDataLength);

  // #ToDo
  #if 0
  DEB_Trace_enTraceLevel dwLldTraceLevel;

  switch(wTraceOutLevel)
  {
    /* handle Trace Level INTERFACE and remap the Level in dependency of the Interface */
    case CSM_C_TRACEOUT_LEVEL_INTERFACE:
    {
      switch(wTraceOutInterface)
      {
        case CSM_C_TRACEOUT_INTERFACE_CSM:
        {
          if( (wTraceOutFunktion & 0xf0) == 0x30)
          {
            dwLldTraceLevel = DEB_TR_LEVEL_USER_2;
          }
          else
          {
            dwLldTraceLevel = DEB_TR_LEVEL_USER_1;
          }
          break;
        }
        case CSM_C_TRACEOUT_CSM_INTERNAL:
        {
          dwLldTraceLevel = DEB_TR_LEVEL_USER_1;
          break;
        }
        case CSM_C_TRACEOUT_INTERFACE_CNP:
        case CSM_C_TRACEOUT_INTERFACE_CNM:
        case CSM_C_TRACEOUT_INTERFACE_CBR:
        {
          dwLldTraceLevel = DEB_TR_LEVEL_USER_3;
          break;
        }
        case CSM_C_TRACEOUT_INTERFACE_CDA:
        case CSM_C_TRACEOUT_INTERFACE_CDA_SYSTEM:
        case CSM_C_TRACEOUT_INTERFACE_CSM_SYSTEM:
        case CSM_C_TRACEOUT_INTERFACE_CDA_CONTROLLER:
        case CSM_C_TRACEOUT_INTERFACE_CDA_TRANSCEIVER:
        case CSM_C_TRACEOUT_INTERFACE_OS:
        {
          dwLldTraceLevel = DEB_TR_LEVEL_USER_4;
          break;
        }
        default:
        {
          dwLldTraceLevel = DEB_TR_LEVEL_USER_4;
          break;
        }
      }
      break;
    }
    /* handle Trace Level INFO and remap the Level */
    case CSM_C_TRACEOUT_LEVEL_INFO:
    {
      dwLldTraceLevel = DEB_TR_LEVEL_SYSTEM;
      break;
    }
    /* handle Trace Level WARNING and remap the Level */
    case CSM_C_TRACEOUT_LEVEL_WARNING:
    {
      dwLldTraceLevel = DEB_TR_LEVEL_SYSTEM_MIN;
      break;
    }
    /* handle Trace Level ERROR and remap the Level */
    case CSM_C_TRACEOUT_LEVEL_ERROR:
    {
      dwLldTraceLevel = DEB_TR_LEVEL_ERRORS;
      break;
    }
    /* handle Trace Level SEVERE ERROR and remap the Level */
    case CSM_C_TRACEOUT_LEVEL_SEVERE_ERROR:
    {
      dwLldTraceLevel = DEB_TR_LEVEL_FATAL;
      break;
    }
    /* all other Levels are remapped to USER_4 */
    default:
    {
      dwLldTraceLevel = DEB_TR_LEVEL_USER_4;
      break;
    }
  }

  /* now prepare the final trace buffer with the data */
  {
    BYTE abBuffer[CSM_MAX_TRACE_LENGTH + CSM_TRACE_HEADER] ={0};
    WORD wLldTraceOutDataLength;

    abBuffer[ 0] = CSM_M_GET_LOBYTE( wTraceOutLevel);
    abBuffer[ 1] = CSM_M_GET_HIBYTE( wTraceOutLevel);
    abBuffer[ 2] = CSM_M_GET_LOBYTE( wTraceOutInterface);
    abBuffer[ 3] = CSM_M_GET_HIBYTE( wTraceOutInterface);
    abBuffer[ 4] = CSM_M_GET_LOBYTE( wTraceOutFunktion);
    abBuffer[ 5] = CSM_M_GET_HIBYTE( wTraceOutFunktion);

    if (wTraceOutDataLength < CSM_MAX_TRACE_LENGTH)
    {
      /* copy all data */
      wLldTraceOutDataLength = wTraceOutDataLength;
    }
    else
    {
      /* copy only max length data */
      wLldTraceOutDataLength = CSM_MAX_TRACE_LENGTH;
    }

    CSM_vMemCopy(&(abBuffer[CSM_TRACE_HEADER]), (void *) pbTraceOutData, wLldTraceOutDataLength);
    DEB_Trace_vTraceBuffer((int)DEB_TR_CLASS_CAN_CSM_S_TRACE, (int)dwLldTraceLevel, wLldTraceOutDataLength + CSM_TRACE_HEADER, abBuffer);
  }
  #endif
}

/*******************************************************************************
 * function    CSM_vMemCopy
 * \doxydocu
 * \brief
 *
 * \param[out] pvDestination
 *              Pointer to destination area where wLength bytes will be copied
 *              (need to be large enough).
 *
 * \param[in]  pvSource
 *              Pointer to source area where wLength bytes will be copied from.
 *
 * \param[in]  wLength
 *              Number of bytes which will be copied from source to destination
 *              area.
 *
 * \return     VOID
 *
 * \access     all
 * \reentrant  yes
 *
 * \lastreview
 *
 * \history_begin
 * 12.11.04 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
VOID CSM_vMemCopy(VOID * pDestination, const VOID * pSource, WORD wLength)
{
  memcpy( pDestination, pSource, wLength);
}

/*******************************************************************************
 * function    CSM_bMemCompare
 * \doxydocu
 * \brief
 *
 * \param[in ] pvSource1
 *              Pointer to first source area where wLength bytes will be
 *              compared.
 *
 * \param[in]  pvSource2
 *              Pointer to second source area where wLength bytes will be
 *              compared.
 *
 * \param[in]  wLength
 *              Number of bytes in source one area which will be compared with
 *              the bytes in source two area.
 *
 * \return     FALSE (==0) if pvSource1 equals to pvSource2
 *             TRUE  (!=0) if pvSource1 does not equal to pvSource2
 *
 * \access     all
 * \reentrant  yes
 *
 * \lastreview
 *
 * \history_begin
 * 12.11.04 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
BYTE CSM_bMemCompare(const VOID* pDestination, const VOID* pSource, WORD wLength)
{
  WORD wResult = memcmp(pDestination, pSource, wLength);

  if (wResult != 0)
  {
    // data are NOT equal
    return 1;
  }
  else
  {
    // data are equal
    return 0;
  }
}

/*******************************************************************************
 * function    CSM_vMemSet
 * \doxydocu
 * \brief
 *
 * \param[in ] pvDestination
 *              Pointer to first destination area where wLength bytes will be set.
 *
 * \param[in]  cubValue
 *              Value to set
 *
 *
 * \param[in]  NumObj
 *              Number of bytes to set.
 *
 * \return     VOID
 *
 * \access     all
 * \reentrant  yes
 *
 * \lastreview
 *
 * \history_begin
 * 19.06.06 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
VOID CSM_vMemSet(PVOID pvDestination, const BYTE cubValue, WORD NumObj)
{
  memset( pvDestination, cubValue, NumObj);
}

/*******************************************************************************
 * function    CSM_vMicroSecDelay
 * \doxydocu
 * \brief
 *
 * \param[in]  wDelay
 *              Delay time in microsecond, look in coded function, which maximum
 *              value is allowed
 *
 * \return     VOID
 *
 * \access     interrupt|CAN-task|application ??
 * \reentrant  yes|no ??
 *
 * \lastreview
 *
 * \history_begin
 * 12.11.04 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
#if 0
// BKA2HI: disabled since not used
#define CSM_C_DELAY_US_MSETUP ((DWORD)100)
#define CSM_C_DELAY_US_DSETUP ((DWORD)100)
#define CSM_C_DELAY_US_OSETUP ((DWORD)0)
static volatile DWORD CSM_dwDelayByte = 0;

void CSM_vMicroSecDelay(WORD wTime)
{
  DWORD dwCount;
  // loop until time is reached
  for (dwCount = 0; dwCount < ((wTime*CSM_C_DELAY_US_MSETUP)/CSM_C_DELAY_US_DSETUP); dwCount++)
  {
    CSM_dwDelayByte++;
  }
}
#endif

// BKA2HI: Trace function, currently not referenced in CSM, so disabled
#if 0
void CSM_vTraceRunningTimer()
{
   DWORD dwCount;
   DWORD u32RemainingTime, u32IntervalTime;

   printf("#################################################################################\n");
   printf("Active timer:\n");
   for (dwCount = 1; dwCount < CSM_C_MAX_TIMER; dwCount++) /*lint !e681 Reason: loop is not entered if no timer is defined*/
   {
     if (OSAL_s32TimerGetTime(csm_arTimerHandle[dwCount], &u32RemainingTime, &u32IntervalTime) != OSAL_OK)
     {
       // empty
     }
     if (u32RemainingTime > 0)
     {
       printf("Timer - ID: %d, Index: %08x, remaining: %d ms\n", dwCount, csm_arTimer[dwCount].dwValue, u32RemainingTime);
     }
   }
   printf("#################################################################################\n");
}
#endif

/*******************************************************************************
 * function    csm_vCsmTraceCallback
 * \doxydocu
 * \brief      This function is the callback from trace.
 *
 * \param[in]  pu8Data
 *                 Trace input data
 *
 * \return     VOID
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 * 06.07.12  VTeam-Khler
 * - Trace level for CSM_C_TRACE_FKT_CSM_SET_TEST_NUMBER changed to error.
 * 09.03.09  VTeam-Khler
 * - CSM_C_TRACE_FKT_CSM_SET_TEST_NUMBER added.
 * 28.02.08  VTeam-Brunotte
 * - If CSM_S_GMLAN_NM is on, two new commands with function number CSM_C_TRACE_FKT_CNM_GMLAN_GET_ACTIVE_VN
 *   and CSM_C_TRACE_FKT_CNM_GMLAN_GET_API_VN_REQ are added.
 * 29.11.07  VTeam-Khler
 * - Trace suppression for Tx signals added.
 * - lRetCode commented out (Lint warning).
 * - Indentation changed.
 * 31.01.01 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
#if (CSM_S_TRACE == CSM_C_F_ON)
CSM_STATIC void csm_vCsmTraceCallback(BYTE * pu8Data)
{
  // How does it work:
  // This function is hooked to the trace system by the StartTraceCallback function on top of this module.
  // Whenever a trace input has been manually done one the TTFis console, the data will be passed to this function.
  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_FATAL(("##### csm_vCsmTraceCallback - TTFis user input received ######"));
  #endif
  #endif
  #endif

  // evaluate Byte 1 (Trace Function ID) (Byte 0 is: 38)
  switch(*(pu8Data+1))
  {
    // Fkt ID 0x22
    case CSM_C_TRACE_FKT_CSM_COMMUNICATION_REQ:
    {
      BYTE  bBus;
      DWORD dwProtocolType;
      BYTE  abAddressField[2];
      BYTE  bAction;
      WORD  wApplID;

      bBus = *(pu8Data+2);
      dwProtocolType = ((DWORD) (*(pu8Data+3)))<<24 | ((DWORD) (*(pu8Data+4)))<<16 | ((DWORD) (*(pu8Data+5)))<<8 | (*(pu8Data+6))<<0;
      abAddressField[ 0] = *(pu8Data+7);
      abAddressField[ 1] = *(pu8Data+8);
      bAction = *(pu8Data+9);
      wApplID = (*(pu8Data+10))<<8 | (*(pu8Data+11))<<0;

      CSM_lCommunicationReq(bBus, dwProtocolType, abAddressField, bAction, wApplID);
      break;
    }

    // Fkt ID 0x21
    case CSM_C_TRACE_FKT_CSM_COMMUNICATION_IND:
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_FATAL(("csm_vCsmTraceCallback - Function ID ComInd 0x%X not supported", CSM_C_TRACE_FKT_CSM_COMMUNICATION_IND));
      #endif
      #endif
      #endif
      break;
    }

    // Fkt ID 0x20
    case CSM_C_TRACE_FKT_CSM_COMMUNICATION_CON:
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_FATAL(("csm_vCsmTraceCallback - Function ID ComCon 0x%X not supported", CSM_C_TRACE_FKT_CSM_COMMUNICATION_CON));
      #endif
      #endif
      #endif
      break;
    }

    // Fkt ID 0x25
    case CSM_C_TRACE_FKT_CSM_GET_COMMUNICATION_STATE:
    {
      BYTE  bBus;
      DWORD dwProtocolType;
      BYTE  abAddressField[2];
      BYTE  bAction=0;
      WORD  wApplID;

      bBus = *(pu8Data+2);
      dwProtocolType = ((DWORD) (*(pu8Data+3)))<<24 | ((DWORD) (*(pu8Data+4)))<<16 | ((DWORD) (*(pu8Data+5)))<<8 | (*(pu8Data+6))<<0;
      abAddressField[ 0] = *(pu8Data+7);
      abAddressField[ 1] = *(pu8Data+8);
      wApplID = (*(pu8Data+9))<<8 | (*(pu8Data+10))<<0;

      CSM_lGetCommunicationState(bBus, dwProtocolType, abAddressField, &bAction, wApplID);
      break;
    }

    // Fkt ID 0x30
    case CSM_C_TRACE_FKT_CSM_SIGNAL_READ:
    {
      DWORD dwSignalReadStatus = 0;
      BYTE  abData[8];
      DWORD dwSignalId = ((DWORD) (*(pu8Data+2)))<<24 | ((DWORD) (*(pu8Data+3)))<<16 | ((DWORD) (*(pu8Data+4)))<<8 | (*(pu8Data+5))<<0;
      BYTE  bLength = (*(pu8Data+6))<<0;

      CSM_lSignalRead(dwSignalId, abData, bLength, &dwSignalReadStatus);
      break;
    }

    // Fkt ID 0x31
    case CSM_C_TRACE_FKT_CSM_SIGNAL_WRITE:
    {
      BYTE  *pbData = pu8Data+7;
      DWORD dwSignalId = ((DWORD) (*(pu8Data+2)))<<24 | ((DWORD) (*(pu8Data+3)))<<16 | ((DWORD) (*(pu8Data+4)))<<8 | (*(pu8Data+5))<<0;
      BYTE  bTxType = (*(pu8Data+6))<<0;
      BYTE  bDataBufferLength = (*(pu8Data+0)) - 6;

      CSM_lSignalWrite(dwSignalId, pbData, bDataBufferLength, bTxType);
      break;
    }

    // Fkt ID 0x32
    case CSM_C_TRACE_FKT_CSM_SIGNAL_IND:
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_FATAL(("csm_vCsmTraceCallback - Function ID SigInd 0x%X not supported", CSM_C_TRACE_FKT_CSM_SIGNAL_IND));
      #endif
      #endif
      #endif
      break;
    }

    // Fkt ID 0x40
    case CSM_C_TRACE_FKT_CSM_DATA_REQ:
    {
      DWORD dwProtocolType;
      BYTE abAddressField[2];
      BYTE* pbData;
      WORD  wDataLength;

      dwProtocolType = ((DWORD) (*(pu8Data+2)))<<24 | ((DWORD) (*(pu8Data+3)))<<16 | ((DWORD) (*(pu8Data+4)))<<8 | (*(pu8Data+5))<<0;
      abAddressField[ 0] = *(pu8Data+6);
      abAddressField[ 1] = *(pu8Data+7);
      pbData = pu8Data+8;
      wDataLength =  (*(pu8Data+0)) - 7;

      CSM_lDataReq(dwProtocolType, abAddressField, pbData, wDataLength);
      break;
    }

    // Fkt ID 0x44
    case CSM_C_TRACE_FKT_CSM_DATA_READ:
    {
      DWORD dwProtocolType;
      BYTE abAddressField[2];
      BYTE abData[256];
      WORD  wDataLength;

      dwProtocolType = ((DWORD) (*(pu8Data+2)))<<24 | ((DWORD) (*(pu8Data+3)))<<16 | ((DWORD) (*(pu8Data+4)))<<8 | (*(pu8Data+5))<<0;
      abAddressField[ 0] = *(pu8Data+6);
      abAddressField[ 1] = *(pu8Data+7);
      wDataLength =  *(pu8Data+8);

      CSM_lDataRead(dwProtocolType, abAddressField, abData, wDataLength);
      break;
    }

    // Fkt ID 0x41
    case CSM_C_TRACE_FKT_CSM_DATA_CON:
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_FATAL(("csm_vCsmTraceCallback - Function ID DataCon 0x%X not supported", CSM_C_TRACE_FKT_CSM_DATA_CON));
      #endif
      #endif
      #endif
      break;
    }

    // Fkt ID 0x42
    case CSM_C_TRACE_FKT_CSM_DATA_IND:
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_FATAL(("csm_vCsmTraceCallback - Function ID DataInd 0x%X not supported", CSM_C_TRACE_FKT_CSM_DATA_IND));
      #endif
      #endif
      #endif
      break;
    }

    // Fkt ID 0xF2
    case CSM_C_TRACE_FKT_CSM_UPPERTST:
    {
      // this case starts/stops the upper tester in kernel or user mode
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_FATAL(("csm_vCsmTraceCallback - Function ID UpperTst 0x%X not supported", CSM_C_TRACE_FKT_CSM_UPPERTST));
      #endif
      #endif
      #endif
      break;
    }

    // Fkt ID 0x43
    case CSM_C_TRACE_FKT_CSM_DATA_IND_PROCESSED:
    {
      DWORD dwProtocolType;
      BYTE abAddressField[2];

      dwProtocolType = ((DWORD) (*(pu8Data+2)))<<24 | ((DWORD) (*(pu8Data+3)))<<16 | ((DWORD) (*(pu8Data+4)))<<8 | (*(pu8Data+5))<<0;
      abAddressField[ 0] = *(pu8Data+6);
      abAddressField[ 1] = *(pu8Data+7);

      CSM_lDataIndProcessed(dwProtocolType, abAddressField);
      break;
    }

    #if (CSM_S_TRACE_LIMITED_SIGNAL == CSM_C_F_ON)
    // Fkt ID 0xB0
    case CSM_C_TRACE_FKT_CSM_SIGNAL_SUPPRESS:
    {
      BYTE  bAction;
      BYTE* pbData;
      DWORD dwCount;
      BYTE  bTelegramIndex;
      WORD  wDataLength;

      bAction = *(pu8Data+2);
      pbData = pu8Data+3;

      if( (*(pu8Data+0)) >= 2)
      {
        // more than min lenght
        wDataLength =  (*(pu8Data+0)) - 2;

        // evaluate the action
        switch(bAction)
        {
          case 0:
          {
            // disable all RX telegrams
            for (dwCount=0; dwCount<((CBR_C_MAX_RX_TEL_NUMBER/8)+1); dwCount++)
            {
              csm_abTraceRxSignalAllowed[dwCount] = 0;
            }
            break;
          }
          case 1:
          {
            // enable all RX telegrams
            for (dwCount=0; dwCount<((CBR_C_MAX_RX_TEL_NUMBER/8)+1); dwCount++)
            {
              csm_abTraceRxSignalAllowed[dwCount] = 0xff;
            }
            break;
          }
          case 2:
          {
            // disable a single RX telegram
            for (dwCount=0; dwCount<wDataLength; dwCount++)
            {
              bTelegramIndex = *(pbData+dwCount);
              if (bTelegramIndex < CBR_C_MAX_RX_TEL_NUMBER)
              {
                csm_abTraceRxSignalAllowed[(bTelegramIndex/8)] &= ~(1U <<(bTelegramIndex%8));
              }
            }
            break;
          }
          case 3:
          {
            // enable telegram a single RX telegram
            for (dwCount=0; dwCount<wDataLength; dwCount++)
            {
              bTelegramIndex = *(pbData+dwCount);
              if( bTelegramIndex < CBR_C_MAX_RX_TEL_NUMBER)
              {
                csm_abTraceRxSignalAllowed[(bTelegramIndex/8)] |= (1U <<(bTelegramIndex%8));
              }
            }
            break;
          }
          case 4:
          {
            // disable all RX telegrams and set some telegrams
            for (dwCount=0; dwCount<((CBR_C_MAX_RX_TEL_NUMBER/8)+1); dwCount++)
            {
              csm_abTraceRxSignalAllowed[dwCount] = 0;
            }
            for (dwCount=0; dwCount<wDataLength; dwCount++)
            {
              bTelegramIndex = *(pbData+dwCount);
              if (bTelegramIndex < CBR_C_MAX_RX_TEL_NUMBER)
              {
                csm_abTraceRxSignalAllowed[(bTelegramIndex/8)] |= (1U <<(bTelegramIndex%8));
              }
            }
            break;
          }
          case 10:
          {
            // disable all TX telegrams
            for (dwCount=0; dwCount<((CBR_C_MAX_TX_TEL_NUMBER/8)+1); dwCount++)
            {
              csm_abTraceTxSignalAllowed[dwCount] = 0;
            }
            break;
          }
          case 11:
          {
            // enable all TX telegrams
            for (dwCount=0; dwCount<((CBR_C_MAX_TX_TEL_NUMBER/8)+1); dwCount++)
            {
              csm_abTraceTxSignalAllowed[dwCount] = 0xff;
            }
            break;
          }
          case 12:
          {
            // disable a single TX telegram
            for (dwCount=0; dwCount<wDataLength; dwCount++)
            {
              bTelegramIndex = *(pbData+dwCount);
              if (bTelegramIndex < CBR_C_MAX_TX_TEL_NUMBER)
              {
                csm_abTraceTxSignalAllowed[(bTelegramIndex/8)] &= ~(1U <<(bTelegramIndex%8));
              }
            }
            break;
          }
          case 13:
          {
            // enable a single TX telegram
            for (dwCount=0; dwCount<wDataLength; dwCount++)
            {
              bTelegramIndex = *(pbData+dwCount);
              if (bTelegramIndex < CBR_C_MAX_TX_TEL_NUMBER)
              {
                csm_abTraceTxSignalAllowed[(bTelegramIndex/8)] |= (1U <<(bTelegramIndex%8));
              }
            }
            break;
          }
          case 14:
          {
            // disable all TX telegrams and set some telegrams
            for (dwCount=0; dwCount<((CBR_C_MAX_TX_TEL_NUMBER/8)+1); dwCount++)
            {
              csm_abTraceTxSignalAllowed[dwCount] = 0;
            }
            for (dwCount=0; dwCount<wDataLength; dwCount++)
            {
              bTelegramIndex = *(pbData+dwCount);
              if (bTelegramIndex < CBR_C_MAX_TX_TEL_NUMBER)
              {
                csm_abTraceTxSignalAllowed[ (bTelegramIndex/8)] |= (1U <<(bTelegramIndex%8));
              }
            }
            break;
          }
          default:
          {
            break;
          }
        }
      } // len > 2
      break;

    } // CSM_C_TRACE_FKT_CSM_SIGNAL_SUPPRESS
    #endif // CSM_S_TRACE_LIMITED_SIGNAL


    // Fkt ID 0x68
    case CSM_C_TRACE_FKT_CSM_SET_TEST_NUMBER:
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      {
        #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
        ETG_TRACE_USR2(("csm_vCsmTraceCallback - Test number set: Type:0x%X  Number: 0x%X 0x%X (Hi/Lo)",
                       (tU8)  *(pu8Data+2),
                       (tU8)  *(pu8Data+3),
                       (tU8)  *(pu8Data+4)
                      ));
        #endif
      }
      #else // CSM_S_TRACE_ETG
      {
        BYTE abData[3];
        abData[0] = *(pu8Data+2);  // Test Type
        abData[1] = *(pu8Data+3);  // High byte test number
        abData[2] = *(pu8Data+4);  // Low byte test number

        CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_ERROR, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_TRACE_TEST_NUMBER, abData, 3);
      }
      #endif // CSM_S_TRACE_ETG
      #endif // CSM_S_TRACE
      break;
    }

    //*********************************************************************************************
    #if (CSM_S_MPDT == CSM_C_F_ON)
    // Special Commands for MPDT Tests, TTFis Input is: 0F FF 02 38 F5 WW XX YY ZZ
    // WW is the sub command
    // XX is delay between messages
    // YY is amount of bytes to be sent from pre-filled, very long array here
    // ZZ is amount of bytes for a second message, in case of different len must be sent for both messages

    // Fkt ID 0xF5
    case CSM_C_TRACE_FKT_MPDT_JOIN_CHECK_HELPER:
    {
      // evaluate the WW
      switch(*(pu8Data+2))
      {
        //************************
        // for T-09 (erster Fall)
        case 0x00:
        {
          #if (CSM_S_TRACE == CSM_C_F_ON)
          #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
          #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
          ETG_TRACE_USR4(("csm_vCsmTraceCallback: T-09 test - 1 SF interrupt"));
          #endif
          #endif
          #endif

          LONG  lRetCode = CSM_C_NO_ERROR;
          DWORD dwProtocolType;
          BYTE abAddressField[2];
          // we need a couple of Bytes >7 to have a multi-package message transmitted
          BYTE abData[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,};
          // the test case starts with a D-type message
          dwProtocolType = CSM_C_PTYPE_RN_MPDT_D;
          abAddressField[ 0] = 0x01; // master
          abAddressField[ 1] = 0x05; // TCU
          // start transmission with len of YY bytes
          lRetCode = CSM_lDataReq(dwProtocolType, abAddressField, abData, *(pu8Data+4));
          // BKA2HI: short break (with TTFis parameter from byte XX), otherwise we will overwrite our first D-type message from above
          // because DataReq implementation in Proxy does not consider the IPC WaitForConf flag !!!
          OSAL_s32ThreadWait(*(pu8Data+3));
          // the first message will be interrupted by a C-Type message (on v850 by transmisson control)
          dwProtocolType = CSM_C_PTYPE_RN_MPDT_C;
          // the 2. message is a single frame (6 Bytes)
          lRetCode = CSM_lDataReq(dwProtocolType, abAddressField, abData, 6);
          CSM_M_PARAMETER_INTENTIONALLY_UNUSED(lRetCode);
        }
        break;

        //*********************************
        // for T-09 (want Fall: mit 2 SF C-Type Unterbrechungen der D-Type message)
        case 0x01:
        {
          #if (CSM_S_TRACE == CSM_C_F_ON)
          #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
          #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
          ETG_TRACE_FATAL(("csm_vCsmTraceCallback: T-09 test - 2 SF interrupt"));
          #endif
          #endif
          #endif

          LONG  lRetCode = CSM_C_NO_ERROR;
          DWORD dwProtocolType;
          BYTE abAddressField[2];
          BYTE abData[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,};
          // first message (D-Type)
          dwProtocolType = CSM_C_PTYPE_RN_MPDT_D;
          abAddressField[ 0] = 0x01;  // master
          abAddressField[ 1] = 0x05;  // TCU
          // start transmission with len of YY bytes
          lRetCode = CSM_lDataReq(dwProtocolType, abAddressField, abData, *(pu8Data+4));
          // BKA2HI: short break (with TTFis parameter from byte XX), otherwise we will overwrite our first D-type message from above
          // because DataReq implementation in Proxy does not consider the IPC WaitForConf flag !!!
          OSAL_s32ThreadWait(*(pu8Data+3));
          // interrupt it with a C-Type message (SF)
          dwProtocolType = CSM_C_PTYPE_RN_MPDT_C;
          lRetCode = CSM_lDataReq(dwProtocolType, abAddressField, abData, 6);
          // BKA2HI: short break (with TTFis parameter from byte XX), otherwise we will overwrite our first D-type message from above
          // because DataReq implementation in Proxy does not consider the IPC WaitForConf flag !!!
          OSAL_s32ThreadWait(*(pu8Data+3));
          // interrupt it with a 2. C-Type message (SF)
          lRetCode = CSM_lDataReq(dwProtocolType, abAddressField, abData, 6);
          CSM_M_PARAMETER_INTENTIONALLY_UNUSED(lRetCode);
        }
        break;

        //***************************
        // for T-10
        case 0x02:
        {
          #if (CSM_S_TRACE == CSM_C_F_ON)
          #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
          #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
          ETG_TRACE_FATAL(("csm_vCsmTraceCallback: T-10 test"));
          #endif
          #endif
          #endif

          LONG  lRetCode = CSM_C_NO_ERROR;
          DWORD dwProtocolType;
          BYTE abAddressField[2];
          BYTE abData[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,};
          BYTE abData2[] = {0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99};

          // first message (D-Type)
          dwProtocolType = CSM_C_PTYPE_RN_MPDT_D;
          abAddressField[ 0] = 0x01;  // master
          abAddressField[ 1] = 0x05;  // TCU
          lRetCode = CSM_lDataReq(dwProtocolType, abAddressField, abData, *(pu8Data+4));
          // BKA2HI: short break (with TTFis parameter from byte XX), otherwise we will overwrite our first D-type message from above
          // because DataReq implementation in Proxy does not consider the IPC WaitForConf flag !!!
          OSAL_s32ThreadWait(*(pu8Data+3));
          // second message (D-Type)
          dwProtocolType = CSM_C_PTYPE_RN_MPDT_D;
          lRetCode = CSM_lDataReq(dwProtocolType, abAddressField, abData2, sizeof(abData2));
          // BKA2HI: short break (with TTFis parameter from byte XX), otherwise we will overwrite our first D-type message from above
          // because DataReq implementation in Proxy does not consider the IPC WaitForConf flag !!!
          OSAL_s32ThreadWait(*(pu8Data+3));
          // third message (C-Type)
          dwProtocolType = CSM_C_PTYPE_RN_MPDT_C;
          lRetCode = CSM_lDataReq(dwProtocolType, abAddressField, abData, 6);
          CSM_M_PARAMETER_INTENTIONALLY_UNUSED(lRetCode);
        }
        break;

        //************************
        // for T-42,
        case 42:
        {
          LONG  lRetCode = CSM_C_NO_ERROR;
          DWORD dwProtocolType;
          BYTE abAddressField[2];
          BYTE abData[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40};

          // first message
          dwProtocolType = CSM_C_PTYPE_RN_MPDT_C;
          abAddressField[ 0] = 0x01;  // master
          abAddressField[ 1] = 0x05;  // TCU
          lRetCode = CSM_lDataReq(dwProtocolType, abAddressField, abData, *(pu8Data+4));
          // second message on a different LUN
          abAddressField[ 1] = 0x07;  // RSE
          lRetCode = CSM_lDataReq(dwProtocolType, abAddressField, abData, *(pu8Data+5));
          CSM_M_PARAMETER_INTENTIONALLY_UNUSED(lRetCode);
        }
        break;

        //************************
        // for T-43 ??
        case 43:
        {
          LONG  lRetCode = CSM_C_NO_ERROR;
          DWORD dwProtocolType;
          BYTE abAddressField[2];
          BYTE abData[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31};

          dwProtocolType = CSM_C_PTYPE_RN_MPDT_C;
          abAddressField[ 0] = 0x01; // master
          abAddressField[ 1] = 0x05; // TCU
          lRetCode = CSM_lDataReq(dwProtocolType, abAddressField, abData, *(pu8Data+4));

          abAddressField[ 1] = 0x07; // RSE
          lRetCode = CSM_lDataReq(dwProtocolType, abAddressField, abData, *(pu8Data+5));
          CSM_M_PARAMETER_INTENTIONALLY_UNUSED(lRetCode);
        }
        break;

        //************************
        // for T-44 ??
        case 44:
        {
          LONG  lRetCode = CSM_C_NO_ERROR;
          DWORD dwProtocolType;
          BYTE abAddressField[2];
          BYTE abData[] = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31};

          dwProtocolType = CSM_C_PTYPE_RN_MPDT_C;
          abAddressField[ 0] = 0x01;  // master
          abAddressField[ 1] = 0x05;  // TCU
          lRetCode = CSM_lDataReq(dwProtocolType, abAddressField, abData, *(pu8Data+4));

          abAddressField[ 1] = 0x07;  // RSE
          lRetCode = CSM_lDataReq(dwProtocolType, abAddressField, abData, *(pu8Data+5));
          CSM_M_PARAMETER_INTENTIONALLY_UNUSED(lRetCode);
        }
        break;

        //**************************
        // open/close all configured MPDT channels
        case 0xff:
        {
       /*   tU8 au8AddressFieldMpdt1[2] = { CSM_C_ADDRESS_FIELD_RN_MPDT_MAS, CSM_C_ADDRESS_FIELD_RN_MPDT_NAV };
          tU8 au8AddressFieldMpdt2[2] = { CSM_C_ADDRESS_FIELD_RN_MPDT_MAS, CSM_C_ADDRESS_FIELD_RN_MPDT_AUD };
          tU8 au8AddressFieldMpdt3[2] = { CSM_C_ADDRESS_FIELD_RN_MPDT_MAS, CSM_C_ADDRESS_FIELD_RN_MPDT_SWC };
          tU8 au8AddressFieldMpdt4[2] = { CSM_C_ADDRESS_FIELD_RN_MPDT_MAS, CSM_C_ADDRESS_FIELD_RN_MPDT_TCU };
          tU8 au8AddressFieldMpdt5[2] = { CSM_C_ADDRESS_FIELD_RN_MPDT_MAS, CSM_C_ADDRESS_FIELD_RN_MPDT_BOSE };
          tU8 au8AddressFieldMpdt6[2] = { CSM_C_ADDRESS_FIELD_RN_MPDT_MAS, CSM_C_ADDRESS_FIELD_RN_MPDT_RSE };
          tU8 au8AddressFieldMpdt7[2] = { CSM_C_ADDRESS_FIELD_RN_MPDT_MAS, CSM_C_ADDRESS_FIELD_RN_MPDT_NAV2 };
          tU8 au8AddressFieldMpdt8[2] = { CSM_C_ADDRESS_FIELD_RN_MPDT_MAS, CSM_C_ADDRESS_FIELD_RN_MPDT_CD };
          tU8 au8AddressFieldMpdt9[2] = { CSM_C_ADDRESS_FIELD_RN_MPDT_MAS, CSM_C_ADDRESS_FIELD_RN_MPDT_BOSEA };

          CSM_lCommunicationReq(CSM_C_CAN_BUS_MCAN, CSM_C_PTYPE_RN_MPDT_C, au8AddressFieldMpdt1, *(pu8Data+3), CSM_C_APPL_ID_DEFAULT);
          OSAL_s32ThreadWait(100);
          CSM_lCommunicationReq(CSM_C_CAN_BUS_MCAN, CSM_C_PTYPE_RN_MPDT_C, au8AddressFieldMpdt2, *(pu8Data+3), CSM_C_APPL_ID_DEFAULT);
          OSAL_s32ThreadWait(100);
          CSM_lCommunicationReq(CSM_C_CAN_BUS_MCAN, CSM_C_PTYPE_RN_MPDT_C, au8AddressFieldMpdt3, *(pu8Data+3), CSM_C_APPL_ID_DEFAULT);
          OSAL_s32ThreadWait(100);
          CSM_lCommunicationReq(CSM_C_CAN_BUS_MCAN, CSM_C_PTYPE_RN_MPDT_C, au8AddressFieldMpdt4, *(pu8Data+3), CSM_C_APPL_ID_DEFAULT);
          OSAL_s32ThreadWait(100);
          CSM_lCommunicationReq(CSM_C_CAN_BUS_MCAN, CSM_C_PTYPE_RN_MPDT_C, au8AddressFieldMpdt5, *(pu8Data+3), CSM_C_APPL_ID_DEFAULT);
          OSAL_s32ThreadWait(100);
          CSM_lCommunicationReq(CSM_C_CAN_BUS_MCAN, CSM_C_PTYPE_RN_MPDT_C, au8AddressFieldMpdt6, *(pu8Data+3), CSM_C_APPL_ID_DEFAULT);
          OSAL_s32ThreadWait(100);
          CSM_lCommunicationReq(CSM_C_CAN_BUS_MCAN, CSM_C_PTYPE_RN_MPDT_C, au8AddressFieldMpdt7, *(pu8Data+3), CSM_C_APPL_ID_DEFAULT);
          OSAL_s32ThreadWait(100);
          CSM_lCommunicationReq(CSM_C_CAN_BUS_MCAN, CSM_C_PTYPE_RN_MPDT_C, au8AddressFieldMpdt8, *(pu8Data+3), CSM_C_APPL_ID_DEFAULT);
          OSAL_s32ThreadWait(100);
          CSM_lCommunicationReq(CSM_C_CAN_BUS_MCAN, CSM_C_PTYPE_RN_MPDT_C, au8AddressFieldMpdt9, *(pu8Data+3), CSM_C_APPL_ID_DEFAULT);*/
        }
        break;

        // LINT happy
        default:
          break;
      }
    }
    break;
    #endif // CSM_S_MPDT

    case CSM_C_TRACE_FKT_IMX_CONFIG_CHECK_HELPER:
      // evaluate sub parameter
      switch( *(pu8Data+2))
      {
        #if (CSM_S_VARIANTS_MANAGEMENT == CSM_C_F_ON)
        #if (CSM_S_SLOT_BASED_CONFIG == CSM_C_F_ON)
        // Trigger Slot Reading (just to be able to verify iMX slot config during runtime)
        case CSM_APL_C_READ_SLOT:
          #if (CSM_S_TRACE == CSM_C_F_ON)
          #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
          #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
          ETG_TRACE_FATAL(("csm_vCsmTraceCallback - Read Slot Nr"));
          #endif
          #endif
          #endif
          CSM_APL_bGetConfigurationSlotNr();
          break;
        #endif
        #endif

        case CSM_APL_C_INIT_DATAPOOL:
          #if (CSM_S_TRACE == CSM_C_F_ON)
          #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
          #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
          ETG_TRACE_FATAL(("csm_vCsmTraceCallback - ReInit Datapool"));
          #endif
          #endif
          #endif
          CSM_Proc_vInitDatapoolPDD();
          break;

        case CSM_APL_C_SET_DP_ITEM:
          #if (CSM_S_TRACE == CSM_C_F_ON)
          #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
          #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
          ETG_TRACE_FATAL(("csm_vCsmTraceCallback - Set DP Item (will sync to v850 immediately) until next reboot (will then be overwritten from KDS)"));
          #endif
          #endif
          #endif
          CSM_Proc_vSetDatapoolPDDItem((BYTE)*(pu8Data+3),(BYTE)*(pu8Data+4));
          break;

        case CSM_APL_C_CHECK_INC_COM:
        {
          BOOL state;
          state = CSM_PROXY_bIsActive();
          #if (CSM_S_TRACE == CSM_C_F_ON)
          #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
          #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
          ETG_TRACE_FATAL(("csm_vCsmTraceCallback - CSM Proxy State is:%x ", state));
          #endif
          #endif
          #endif
        }
        break;

        case CSM_APL_C_SET_DLT_LOGLEVEL:
        {
          LONG  lRetCode;
          DWORD dwProtocolType;
          BYTE abAddressField[2];
          BYTE* pbData;
          WORD  wDataLength;
          BYTE ModuleIndex = *(pu8Data+3);
          BYTE LogLevel = *(pu8Data+4);
          #if (CSM_S_TRACE == CSM_C_F_ON)
          #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
          #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
          ETG_TRACE_FATAL(("csm_vCsmTraceCallback - CSM set V850 DLT Loglevel:0x%X for a CSM Module:0x%X", LogLevel, ModuleIndex));
          #endif
          #endif
          #endif

          // AddrField will contain a command that is evaluated by DataReq for LCTRL
          dwProtocolType = CSM_C_PTYPE_PSGEN_CAN_LOCAL_CTRL;
          abAddressField[0] = CSM_C_ADDRESS_FIELD_DC_LOCAL_CTRL;
          abAddressField[1] = CSM_C_ADDRESS_FIELD_DC_LOCAL_CTRL_UNDEF;
          pbData = pu8Data+3; // Data will point to the command params
          wDataLength = 2;
          // trigger a local data request for LCTRL
          lRetCode = CSM_lDataReq(dwProtocolType, abAddressField, pbData, wDataLength);
          if (lRetCode != CSM_C_NO_ERROR)
          {
            ETG_TRACE_FATAL(("csm_vCsmTraceCallback - DataReq for AddrField 0x%X%X returned Error 0x%X", abAddressField[0], abAddressField[1], lRetCode));
          }
          else
          {
            ETG_TRACE_FATAL(("csm_vCsmTraceCallback - DataReq for AddrField 0x%X%X successfully sent", abAddressField[0], abAddressField[1]));
          }
        }
        break;

        case CSM_APL_C_SET_ISOTP_BS:
        {
          LONG  lRetCode;
          DWORD dwProtocolType;
          BYTE abAddressField[2];
          BYTE* pbData;
          WORD  wDataLength;
          BYTE Channel   = *(pu8Data+3);     // 0 or 1
          BYTE BlockSize = *(pu8Data+4);     // BS value

          #if (CSM_S_TRACE == CSM_C_F_ON)
          #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
          #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
          ETG_TRACE_FATAL(("csm_vCsmTraceCallback - CSM set ISO TP Block Size:0x%X at OSOTP channel:0x%X", BlockSize, Channel));
          #endif
          #endif
          #endif

          // AddrField will contain a command that is evaluated by DataReq for LCTRL
          dwProtocolType = CSM_C_PTYPE_PSGEN_CAN_LOCAL_CTRL;
          abAddressField[0] = CSM_C_ADDRESS_FIELD_DC_LOCAL_CTRL;
          abAddressField[1] = CSM_C_ADDRESS_FIELD_DC_LOCAL_CTRL_UNDEF;
          pbData = pu8Data+3; // Data will point to the command params
          wDataLength = 2;
          // trigger a local data request for LCTRL
          lRetCode = CSM_lDataReq(dwProtocolType, abAddressField, pbData, wDataLength);
          if (lRetCode != CSM_C_NO_ERROR)
          {
            ETG_TRACE_FATAL(("csm_vCsmTraceCallback - DataReq for AddrField 0x%X%X returned Error 0x%X", abAddressField[0], abAddressField[1], lRetCode));
          }
          else
          {
            ETG_TRACE_FATAL(("csm_vCsmTraceCallback - DataReq for AddrField 0x%X%X successfully sent", abAddressField[0], abAddressField[1]));
          }
        }
        break;

        // add functionality here if needed
        default:
          #if (CSM_S_TRACE == CSM_C_F_ON)
          #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
          #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
          ETG_TRACE_FATAL(("csm_vCsmTraceCallback - Invalid SubCommand"));
          #endif
          #endif
          #endif
          break;
      }
      break;

    default:
      #if (CSM_S_PROJ_SPECIFIC_TRACECALLBACK == CSM_C_F_ON)
        #if (CSM_S_TRACE == CSM_C_F_ON)
        #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
        #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
        ETG_TRACE_FATAL(("csm_vCsmTraceCallback - Try to execute project specific trace command"));
        #endif
        #endif
        #endif
        csm_vCsmTraceCallback_ProjectSpecific(pu8Data);
      #else
        #if (CSM_S_TRACE == CSM_C_F_ON)
        #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
        #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
        ETG_TRACE_FATAL(("csm_vCsmTraceCallback - Invalid Trace Function ID"));
        #endif
        #endif
        #endif
      #endif
      break;
  } // end of switch(*(pu8Data+1))
}
#endif // CSM_S_TRACE

/*******************************************************************************
 * function    CSM_lTimerStart
 * \doxydocu
 * \brief      CAN stack function to start a timer.
 *
 * \param[in]  bTimerId
 *              Internal (CSM) id of the timer (defined in csm_stack_mp.h)
 *
 *             WORD uwTime
 *              Time value in ms
 *
 * \return     Error code
 *
 * \access     all (except non operatins system interrupts)
 * \reentrant  only with different timer ids
 *
 * \lastreview
 *
 * \history_begin
 * 19.05.05 CM-DI/ESA3-Prhl
 * DWORD instead of BYTE is used for timer ID
 * 12.11.04 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
LONG CSM_lTimerStart(WORD wTimerId, DWORD dwTime)
{
  LONG lReturn = CSM_C_NO_ERROR;

  // determine the corresponding timer-thread handle and set intervall
  // an event is created for the thread hopefully                             */
  if (OSAL_s32TimerSetTime(csm_arTimerHandle[wTimerId], dwTime, 0) != OSAL_OK)
  {
    // failed to start timer
  }

  #if (CSM_S_TRACE == CSM_C_F_ON)
    #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR4(("CSM_lTimerStart: Id:%x",
                      ETG_CENUM(tCSM_TIMER_ID, (tU8)wTimerId)
                    ));
      #endif
    #endif
  #endif

  return (lReturn);
}

/*******************************************************************************
 * function    CSM_lTimerStop
 * \doxydocu
 * \brief      CAN stack function to stop a timer.
 *
 * \param
 * \param[in]  wTimerId
 *              Internal (CSM) id of the timer (defined in csm_stack_mp.h)
 *
 * \return     VOID
 * \return     Error code
 *
 * \access     all (except non operatins system interrupts)
 * \reentrant  only with different timer ids
 *
 * \lastreview
 *
 * \history_begin
 * 19.05.05 CM-DI/ESA3-Prhl
 * DWORD instead of BYTE is used for timer ID
 * 12.11.04 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
LONG CSM_lTimerStop(WORD wTimerId)
{
  LONG lReturn = CSM_C_NO_ERROR;

  // determine corresponding timer-thread handle and set intervall to 0
  // an event is created for the thread hopefully
  if (OSAL_s32TimerSetTime(csm_arTimerHandle[wTimerId], 0, 0) != OSAL_OK)
  {
    // failed to stop timer
  }

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR4(("CSM_lTimerStop: Id:%x",
                  ETG_CENUM(tCSM_TIMER_ID, (tU8)wTimerId)
                ));
  #endif
  #endif
  #endif

  return lReturn;
}

/*******************************************************************************
 * function    CSM_lCyclicTimerStart
 * \doxydocu
 * \brief      CAN stack function to start a recurring cyclic timer.
 *
 * \param[in]  wTimerId
 *              Internal (CSM) id of the timer (defined in csm_stack_mp.h)
 *
 * \param[in]  dwStartDelay
 *              Start delay in ms.
 *
 * \param[in]  dwCycleTime
 *              Cycle time in ms.
 *
 * \return     Error code
 *
 * \access     all (except non operatins system interrupts)
 * \reentrant  only with different timer ids
 *
 * \lastreview 16.04.08 VTeam-Prhl
 *
 * \history_begin
 * 27.03.2007 VTeam-Prhl
 * Initial version.
 * \history_end
 ******************************************************************************/
LONG CSM_lCyclicTimerStart(WORD wTimerId, DWORD dwStartDelay, DWORD dwCycleTime)
{
  LONG lReturn = CSM_C_NO_ERROR;

  if (OSAL_s32TimerSetTime( csm_arTimerHandle[wTimerId], dwStartDelay, dwCycleTime) != OSAL_OK)
  {
    // failed to start timer
  }

  return (lReturn);
}

/*******************************************************************************
 * function    CSM_lCyclicTimerStop
 * \doxydocu
 * \brief      CAN stack function to stop a recurring timer.
 *
 * \param[in]  wTimerId
 *              Internal (CSM) id of the timer (defined in csm_stack_mp.h)
 *
 * \return     VOID
 * \return     Error code
 *
 * \access     all (except non operatins system interrupts)
 * \reentrant  only with different timer ids
 *
 * \lastreview 16.04.08 VTeam-Prhl
 *
 * \history_begin
 * 27.03.2007 VTeam-Prhl
 * Initial version.
 * \history_end
 ******************************************************************************/
LONG CSM_lCyclicTimerStop(WORD wTimerId)
{
  LONG lReturn = CSM_C_NO_ERROR;

  if (OSAL_s32TimerSetTime( csm_arTimerHandle[wTimerId], 0, 0) != OSAL_OK)
  {
    // failed to stop timer
  }

  return (lReturn);
}

/*******************************************************************************
 * function    csm_vActivateUpstreamTask
 * \doxydocu
 * \brief      Activates the CSM upstream task
 *
 * \param      VOID
 *
 * \return     VOID
 *
 * \access     all (except non operatins system interrupts)
 * \reentrant  yes
 *
 * \lastreview
 *
 * \history_begin
 * 12.11.04 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
VOID csm_vActivateUpstreamTask(VOID)
{
  /* use configured event to re-activate the corresponding Thread             */
  OSAL_s32EventPost ( csm_hUpStreamEvent, 0x01, OSAL_EN_EVENTMASK_OR);
}

VOID csm_vActivateUpstreamTaskFromIrq(VOID)
{
  /* use configured event to re-activate the corresponding Thread             */
  OSAL_s32EventPost ( csm_hUpStreamEvent, 0x01, OSAL_EN_EVENTMASK_OR);
}

/*******************************************************************************
 * function    csm_vActivateDownstreamTask
 * \doxydocu
 * \brief      Activates the CSM downstream task
 *
 * \param      VOID
 *
 * \return     VOID
 *
 * \access     all (except non operatins system interrupts)
 * \reentrant  yes
 *
 * \lastreview
 *
 * \history_begin
 * 12.11.04 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
VOID csm_vActivateDownstreamTask(VOID)
{
  /* use created event to re-activate the corresponding Thread                */
  OSAL_s32EventPost ( csm_hDownStreamEvent, 0x01, OSAL_EN_EVENTMASK_OR);

}

VOID csm_vActivateDownstreamTaskFromIrq(VOID)
{
  /* use created event to re-activate the corresponding Thread                */
  OSAL_s32EventPost ( csm_hDownStreamEvent, 0x01, OSAL_EN_EVENTMASK_OR);
}

/*******************************************************************************
 * function    csm_vOSspecificUpstream
 * \doxydocu
 * \brief      Operating system specific part of the CSM upstream task
 *
 * \param      VOID
 *
 * \return     VOID
 *
 * \access     application
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 * 12.11.04 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
VOID csm_vOSspecificUpstream(VOID)
{
}

/*******************************************************************************
 * function    csm_vOSspecificDownstream
 * \doxydocu
 * \brief      Operating system specific part of the CSM downstream task
 *
 * \param      VOID
 *
 * \return     VOID
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 * 12.11.04 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
VOID csm_vOSspecificDownstream(VOID)
{
}

/*******************************************************************************
 * function    CSM_lInitEnv
 * \doxydocu
 * \brief      Initialize the environment (operating system) specific CAN stack
 *             module.
 *
 * \param[in]  bInitMode
 *               CSM_C_INIT_COLD, CSM_C_INIT_WARM, CSM_C_INIT_START
 *
 * \return     Error value
 *
 * \access     application
 * \reentrant  yes
 *
 * \lastreview
 *
 * \history_begin
 * 11.11.04 CM-DI/ESA3-Prhl
 * Initial Revision.
 * \history_end
 ******************************************************************************/
LONG CSM_lInitEnv(BYTE bInitMode)
{
  DEB_PRINTF("CSM__lInitEnv(%02X)\n", bInitMode);

  switch (bInitMode)
  {
    case CSM_C_INIT_COLD:
    {
    }
    break;

    case CSM_C_INIT_WARM:
    {
      pthread_mutexattr_init(&Attr);
      pthread_mutexattr_settype(&Attr, PTHREAD_MUTEX_RECURSIVE);
      pthread_mutex_init(&Mutex, &Attr);
    }
    break;

    case CSM_C_INIT_START:
    {
      /* register our Message Handler at DEB module                           */
      //DEB_Trace_ubRegisterHandler(tyMsgReceiver);

      /* create all required timers and their threads in a loop               */
      csm_vInitTimerSystem();

      /* create event instances for triggering the CSM tasks later            */
      if ( OSAL_OK == OSAL_s32EventCreate("EvCsmDw", &csm_hDownStreamEvent) ) {
      }
      if ( OSAL_OK == OSAL_s32EventCreate("EvCsmUp", &csm_hUpStreamEvent) ) {
      }

      {
         OSAL_trThreadAttribute  rThAttr;
         rThAttr.szName       = "CsmDown";
         rThAttr.u32Priority  = 30;
         rThAttr.s32StackSize = 10000;
         rThAttr.pfEntry      = (OSAL_tpfThreadEntry)csm_vDownStreamThread;
         rThAttr.pvArg        = (tPVoid)NULL;

         // create thread suspended
         csm_hDownStreamThread = OSAL_ThreadSpawn( &rThAttr );
         if(csm_hDownStreamThread == OSAL_ERROR) {
         }
      }

      {
         /* create "upstream task" simulation (as thread) */
         OSAL_trThreadAttribute  rThAttr;
         rThAttr.szName       = "CsmUp";
         rThAttr.u32Priority  = 30;
         rThAttr.s32StackSize = 10000;
         rThAttr.pfEntry      = (OSAL_tpfThreadEntry)csm_vUpStreamThread;
         rThAttr.pvArg        = (tPVoid)NULL;

         // create thread suspended
         lUpStreamThreadId = OSAL_ThreadSpawn( &rThAttr );
         if(lUpStreamThreadId == OSAL_ERROR) {
         }
      }

    }
    break;

    default:
    {
    }
    break;
  }
  return ((LONG) CSM_C_NO_ERROR);
}

/*******************************************************************************
 * function    CSM_lExitEnv
 * \doxydocu
 * \brief      Deactivation of the (operating system) specific CAN stack
 *             module.
 *
 * \param[in]  bExitMode
 *             Allowed values (see csm_stack_i.h):
 *             - CSM_C_EXIT_STOP: Opposite of CSM_C_INIT_START. E.g. stop timers.
 *             - CSM_C_EXIT_OFF: Opposite of CSM_C_INIT_WARM. Set ports for
 *               minimal power consumption.
 *
 * \return     Error value
 *
 * \access     application
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 * 11.11.04 CM-DI/ESA3-Prhl
 * Initial Revision.
 * \history_end
 ******************************************************************************/
LONG CSM_lExitEnv(BYTE bExitMode)
{
  LONG lReturnValue = CSM_C_NO_ERROR;

  DEB_PRINTF("CSM__lExitEnv(%02X)\n", bExitMode);

  switch(bExitMode)
  {
    case CSM_C_EXIT_STOP:
    {
    }
    break;

    case CSM_C_EXIT_OFF:
    {
    }
    break;

    default:
      break;
  }
  return lReturnValue;
}


/*******************************************************************************
 * function    CSM_iInterruptsDisable
 * \doxydocu
 * \brief      Disables all interrupts and saves program status word
 *
 * \param      VOID
 *
 * \return     Disable interrupt state or level before function call.
 *
 * \access     all
 * \reentrant  yes
 *
 * \lastreview
 *
 * \history_begin
 * 12.11.04 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
INT CSM_iInterruptsDisable(VOID)
{
  pthread_mutex_lock(&Mutex); //lint !e454 since unlock happens in enabler function !!
  return 0;
}

/*******************************************************************************
 * function    CSM_vInterruptsRestore
 * \doxydocu
 * \brief      Restores interrupts (program status word)
 *
 * \param[in]  iIsrStatus
 *              (return value of corresponding disable interrupt call)
 *
 * \return     VOID
 *
 * \access     all
 * \reentrant  yes
 *
 * \lastreview
 *
 * \history_begin
 * 12.11.04 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
VOID CSM_vInterruptsRestore(INT iIsrStatus)
{
  (void)iIsrStatus;             //lint happy
  pthread_mutex_unlock(&Mutex); //lint !e455 since lock happens in disabler function !!
}

/*******************************************************************************
 * function    CSM_vAssertFunction
 * \doxydocu
 * \brief      This Function is called after an interrupt
 *
 * \param[in]  coszExp
 *              pointer to assert expression
 *
 * \param[in]  coszFile
 *              pointer to sourcefile name
 *
 * \param[in]  u32Line
 *              Line number in source file
 *
 * \return     VOID
 *
 * \access     all
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 * 10.10.06 CM-DI/ESA3-Khler
 * - ELL-Types replaced.
 *
 * 12.11.04 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
#ifdef NOT_USED
void CSM_vAssertFunction(const CHAR * coszExp, const CHAR coszFile, DWORD dwLine)
{
  DEB_PRINTF( "Assert %s %s %d", coszExp, coszFile, dwLine);
  //TODO
}
#endif
/*****************************************************************************/

#undef CSM_STACK_OSAL_SELF

/*******************************************************************************
 * \doxydocu
 * \file          csm_stack_osal.c
 * \brief         osal environment specific adaptions for the CAN stack
 *                (timer, debug, task activation ...)
 *
 * \see           CAN-SW Architektur, written by CM-DI/ESU3
 *
 * \sw_component  CAN stack
 * \project       Pool module
 * \path          /di_can/modules_swa/csm_stack/src/
 *
 * \ingroup       CSM
 * \ingroup       CSM_STACK
 *
 * \authors       V-Team
 *
 * COPYRIGHT      (c) 2013 Bosch Car Multimedia GmbH
 *
 * \history_begin
 * 23.04.12  \main\2 Borck
 * - printf's commented out to avoid annoying console output on Triton
 *
 * 10.05.12  \main\3 Borck
 * - csm_c.h also has to be included if CSM_S_OSEKI_NM is used, otherwise the timercallback for "WakeupTemp" will not be declared as external
 *
 * 04.01.13  \main\4 Borck
 * - warnings fixed (merged from Dirk Janssens branch)
 *
 * 06.06.13  \main\5 Prhl
 * - CSM_lCyclicTimerStart() and CSM_lCyclicTimerStop() added
 * - includes added.
 *
 * 15.07.13  \main\6 Prhl
 * - Trace added.
 *
 * 07.11.13  \main\7 Prhl
 * - lint warning removed.
 *
 * 14.11.13  \main\8 Prhl
 * - TTFis input function added.
 *
 * 06.03.14  \main\9 Prhl
 * - CSM_iInterruptsDisable()/CSM_vInterruptsRestore() implemented with mutex from pthread
 *
 * 30.04.14  \main\10 Prhl
 * - tk_slp_tsk() replaced by OSAL_s32ThreadWait()
 *
 * 29.05.15  \main\11 Borck
 * - module made compileable for the case that tracing is set to OFF !!!
 *
 * 10.06.15  \main\12 Borck
 * - some ETG traces added
 * - coding style
 * - unused, old printfs removed
 *
 * 01.07.15  \main\13 Borck
 * - Special TTFis commands adapted for MPDT tests
 * - coding style at some lines
 *
 * 16.07.15  \main\14 Borck
 * - TTFis command F1 extended by the possibility to check (read) the configuration SLOT.
 *   cmd: 0F FF 02 38 F1 00
 *
 * 22.07.15  \main\15 Borck
 * - coding style
 * - header file for apl added
 * - some LINT warnings fixed
 *
 * 04.11.15  \main\16 Borck
 * - TTFis handling of ALL MPDT (open/close) channels extended for new channels
 *
 * 06.09.15  \main\17 Borck
 * - dedicated trace class for this module introduced
 * - 2 new trace commands handled to check INC state and to init the data pool after KDS changes.
 * - some unneeded stuff removed
 *
 * 26.09.15  \main\18 Borck
 * - TTFis method call added to set a datapool PDD item
 *
 * 07.11.15  \main\19 Borck
 * - unused stuff disabled or removed
 * - TTFis trace callback extended by ETG trace output
 * - TTFis trace callback extended by new commands
 * - some LINT issues fixed
 *
 * 27.01.17  \main\20 Borck
 * - TTFis interface for setting ISOTP BS added
 *
 * 24.11.17  main\g3inf4cv_csm_proxy_in\1  Jayashree
 * - Initial version for INF4CV
 * \history_end
 *//**** END OF FILE **********************************************************/
