/** File name for preprocessor commands */
#define CSM_PROXY_SELF
/** csm_proxy.c is a file according V-team CAN-SW architecture implementation rules */
#define CSM_CAN_STACK
/** Doxygen instruction:
 * \defgroup      CSM_PROXY   CSM_PROXY
 *                This group contains all files belonging to proxy server
 *                functionality of the CAN stack manager.
 * \ingroup       CSM
*/

/*******************************************************************************
 * includes:
 *  1)system- and project- includes
 *  2)needed interfaces from external components
 *...3)internal and external interfaces from this component
 ******************************************************************************/

#define USE_DGRAM_SERVICE
#include "dgram_service.h"

#ifdef USE_DGRAM_SERVICE
// LINT happy
#endif

//#include <stdio.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <pthread.h>
#include <errno.h>
#include "inc.h"
#include "inc_ports.h"
#include "inc_scc_port_extender_gpio.h"
#include <netdb.h>

#include "csm_stack_i.h"
#include "csm_stack_m.h"
#include "csm_i.h"
#include "csm_m.h"
#include "cbr_c.h"
#include "csm_cbr_c.h"
#include "cnm_oseki_i.h"                                     // for ReInitMonitors
#include "csm_lctr_i.h"                                      // for GetUserAssignedBus

#if ( CSM_S_J1939 == CSM_C_F_ON )
#include "csm_j1939_m.h"
#endif

#include "csm_c.h"                                           // for ISO-TP prototypes

// BKA2HI workaround, since platform refuses to define sensible, generic names
#ifndef NET_TP10_PORT
//#define NET_TP10_PORT (0xC700 | 0x0020)
#define NET_TP10_PORT  BAP_00_PORT
#endif

#ifndef NET_TP11_PORT
//#define NET_TP11_PORT (0xC700 | 0x0021)
#define NET_TP11_PORT  BAP_01_PORT
#endif

#ifndef CSM_C_PTYPE_RN_USDT_ISO
#define CSM_C_PTYPE_RN_USDT_ISO  ((DWORD) _USDT_ISO__)
#endif
// end of workaround

#include "csm_proxy_i.h"
#include "csm_proxy_m.h"
#include "csm_apl.h"
#include "csm_itp_m.h"     /* ISO-TP specific functions/defines of the can stack
                            * manager */
#include "cnp_itp_i.h"

/*************** ETG Tracing ******/
#define TR_CLASS_CSMPROXY (TR_COMP_CSM + 0x16)

#if (CSM_S_TRACE == CSM_C_F_ON)                              // wir tracen generell ?
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)                        // wir wollen ETG nutzen ?
    #define ETRACE_S_IMPORT_INTERFACE_GENERIC
    //#define ETRACE_S_IMPORT_INTERFACE_ERRMEM               // was nicht gebraucht wird, darf auskommentiert werden
    #include "etrace_if.h"

    #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN                      // RN AIVI: wird in ai_projects\config\xml\ai_projects_prod.xml gesetzt
      #include "csm_etg_trace.h"                             // somit machen wir ETG tracing
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CSMPROXY
      //#define ETG_INLINE                                   // brauchen wir scheinbar nicht
      #include "trcGenProj/Header/csm_proxy.c.trc.h"         // so heisst unser generiertes trc file
    #endif // VARIANT_S_FTR_ENABLE_TRC_GEN
  #endif // ETG Trace
#endif // CSM_S_TRACE

/*****************************************************************
| check of all feature switches which affect the CBR :
| The following #defines should be declared in csm_stack_p.h:
|----------------------------------------------------------------*/
#ifndef CSM_S_DOXYGEN_HIDE_NON_API_FCT
  #error "CSM_S_DOXYGEN_HIDE_NON_API_FCT must be set to CSM_C_F_ON or -OFF !"
#endif

/*******************************************************************************
 * defines and macros (scope: file local)
 ******************************************************************************/
enum
{
   // supported events for state machines
   CSM_C_EV_NONE = 0,
   CSM_C_EV_ENTRY,
   CSM_C_EV_EXIT,
   CSM_C_EV_QUIT,
   CSM_C_EV_TIMEOUT,
   CSM_C_EV_POW_ON,
   CSM_C_EV_POW_OFF,
   CSM_C_EV_COMM_ALLOFF,
   CSM_C_EV_CONF_ACTIVE,
   CSM_C_EV_CONF_INACTIVE,
   CSM_C_EV_RESTART
};

/*******************************************************************************
 * typedefs (scope: file local)
 ******************************************************************************/
typedef int tCsmProxyEv;                             // SM event
typedef tCsmProxyEv (*tpfCsmSmState) (tCsmProxyEv);  // Ptr to a state function(param:event) returning an event

typedef struct                                       // struct that implements a SM
{
   tpfCsmSmState current;                            // state function that is implementing the current state
   tpfCsmSmState next;                               // state function that is setting the next state
   tpfCsmSmState super;                              // state function implementing the parent (super) state
   int           working;                            // state transition in progress
} tCsmSm;

/*******************************************************************************
 * variable definition (scope: file local)
 ******************************************************************************/
tCsmSm rCsmSmRemoteNet;   // the SM handling the remote-net states
BOOL csm_fPowerCtrlOn;    // indicates if CSM wants the Proxy to be ON (true) or OFF (False)
                          // this initially set from external in csm_p.c if the stack is activated

/*******************************************************************************
 * variable definition (scope: CAN stack)
 ******************************************************************************/
#if (CSM_S_J1939 == CSM_C_F_ON)
extern tCSM_J1939_PGN_CALLBACK csm_j1939_arPGNCallbackFkt[CSM_C_J1939_CFG_MAXPGN][CSM_C_J1939_MAX_PGN_CALLBACKS];
extern const tCSM_J1939_PGN_CFG cnp_j1939_arCfgPGNTable[];
#endif

/*******************************************************************************
 * function prototypes (scope: file local)
 ******************************************************************************/
// State machine for component
static void csm_vSsmInit(tCsmSm*, tpfCsmSmState, tpfCsmSmState);  // SM Init, called from proxy init, setting super and initial state
static void csm_vSsmGoto(tCsmSm*, tpfCsmSmState);                 // check SM and sets the 'next' function pointer
static void csm_vSsmPutEv(tCsmSm*, tCsmProxyEv);                  // trigger SM with event, perform exit and entry functions

static tpfCsmSmState csm_vSsmGetState(const tCsmSm*);             // get the current state function pointer

static tCsmProxyEv csm_tSsmRemoteNetInactive(tCsmProxyEv);        // state function for INACTIVE
static tCsmProxyEv csm_tSsmRemoteNetStartup(tCsmProxyEv);         // state function for STARTUP
static tCsmProxyEv csm_tSsmRemoteNetActive(tCsmProxyEv);          // state function for ACTIVE
static tCsmProxyEv csm_tSsmRemoteNetWaitAllCommDisc(tCsmProxyEv); // state function for WAIT for disconnects
static tCsmProxyEv csm_tSsmRemoteNetShutdown(tCsmProxyEv);        // state function for SHUTDOWN

// NET INC channel
// RX path
void csm_vNetProcess_IPC_RxData(BYTE* pu8Data, DWORD u32Length);
static void csm_vNetProcessReject(BYTE* pu8Data, DWORD u32Length);
static void csm_vNetProcessCompStatus(BYTE* pu8Data, DWORD u32Length);
static void csm_vNetProcessBusStateInd(BYTE* pu8Data, DWORD u32Length);
static void csm_vNetProcessBusErrorInd(BYTE* pu8Data, DWORD u32Length);
static void csm_vNetProcessCommunicationCon(BYTE* pu8Data, DWORD u32Length);
static void csm_vNetProcessCommunicationInd(BYTE* pu8Data, DWORD u32Length);
static void csm_vNetProcessDataCon(BYTE* pu8Data, DWORD u32Length);

static void csm_vNetTxCompStatusReq(BYTE u8CompStatus);

// BR INC channel
// RX path
void csm_vBrProcess_IPC_RxData(BYTE* pu8Data, DWORD u32Length);
static void csm_vBrProcessReject(BYTE* pu8Data, DWORD u32Length);
static void csm_vBrProcessSignalInd(BYTE* pu8Data, DWORD u32Length);
static void csm_vBrProcessMultiSignalInd(BYTE* pu8Data, DWORD u32Length);
static void csm_vBrProcessCommunicationCon(BYTE* pu8Data, DWORD u32Length);
static void csm_vBrProcessCommunicationInd(BYTE* pu8Data, DWORD u32Length);
// TX path
static void csm_vBrTxSignalWriteReq(DWORD u32CbrTxNumber, BYTE u8TelegramTxType, const BYTE* pu8TelDataBuffer, BYTE bDLC);
static void csm_vBrTxCommReq(BYTE bUserNumber, WORD u16ApplId, BYTE u8CommState);

// TP (NP) INC channel
// RX path
void csm_vTpProcess_IPC_RxData(WORD wTpChanNum, BYTE* pu8Data, DWORD u32Length);
static void csm_vTpProcessReject(WORD wTpChanNum, BYTE* pu8Data, DWORD u32Length);
static void csm_vTpProcessDataInd(WORD wTpChanNum, BYTE* pu8Data, DWORD u32Length);
static void csm_vTpProcessDataCon(WORD wTpChanNum, BYTE* pu8Data, DWORD u32Length);
static void csm_vTpProcessCommCon(WORD wTpChanNum, BYTE* pu8Data, DWORD u32Length);
static void csm_vTpProcessCommInd(WORD wTpChanNum, BYTE* pu8Data, DWORD u32Length);

// BR and TP related
static BOOL csm_fCommReqIsPending(void);

// this pointer declaration is needed for SLOT based configuration
extern const CSM_CBR_TX * csm_arCbrTxTelegram;
// new variables to be able to handle a variable amount of RX/TX message within the configuration
// they will be initialized with real values in csm_apl.c at runtime
//extern BYTE  bCbrRxMaxNumber;
extern BYTE  bCbrTxMaxNumber;

// prototypes from csm_stack_linux_p.c
extern BYTE * CSM_PROXY_pbGetTimeoutFlags(void);
extern void CSM_CBR_vPROXYCommunicationCon(BYTE, BYTE);
extern void CSM_CBR_vPROXYCommunicationInd(BYTE, BYTE);
extern void CSM_ITP_vPROXYTriggerCommunicationCon(BYTE , BYTE );
extern void CSM_ITP_vPROXYTriggerCommunicationInd(BYTE, BYTE);
extern void CSM_ITP_vCNPDataCon(BYTE , BYTE);

// Feature switch is used only to control doxygen documentation
#if (CSM_S_DOXYGEN_HIDE_NON_API_FCT == CSM_C_F_OFF)
/*******************************************************************************
 * function implementation (scope: file local = static)
 ******************************************************************************/

/*******************************************************************************
 * function    csm_vSsmInit
 * \doxydocu
 * \brief      Initializes the state machine.
 *
 * \param      me
 *              Pointer to the state machine structure
 * \param[in]  super
 *              Super state function, which is called, if the current state
 *              ignores event
 * \param[in]  start
 *              Starting state (is immediately called with event ENTRY)
 *
 * \return     VOID
 *
 * \access     CAN task
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void csm_vSsmInit(tCsmSm* me, tpfCsmSmState super, tpfCsmSmState start)
{
  if (start == NULL)
  {
    me->current = super;
  }
  else
  {
    me->current = start;
  }
  me->next    = NULL;
  me->super   = super;
  me->working = 0;

  // trigger the SM with ENTRY event
  csm_vSsmPutEv(me, CSM_C_EV_ENTRY);

  return;
}

/*******************************************************************************
 * function    csm_vSsmGoto
 * \doxydocu
 * \brief      Triggers a transition to the next state.
 *
 * \param      me
 *              Pointer to the state machine structure
 * \param[in]  next
 *              State function, to which the next transition should go
 *
 * \return     VOID
 *
 * \access     CAN task
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static void csm_vSsmGoto(tCsmSm* me, tpfCsmSmState next)
{
  if (me->current != NULL)
  {
    // only do sth. if the current state is valid (initialized)
    if (me->current != next)
    {
      // only do sth. if the requested (next) state is not the current one
      // store the new state function in SM
      me->next = next;

      // check if SM is busy
      if (! me->working)
      {
        // if NOT busy -> trigger sm with event just to force the transition
        csm_vSsmPutEv(me, CSM_C_EV_NONE);
      }
    }
  }
}

/*******************************************************************************
 * function    csm_vSsmPutEv
 * \doxydocu
 * \brief      Puts an event into the current state. If a transition to the
 *             next state was requested, it calls the EXIT and ENTRY events.
 *
 * \param      me
 *              Pointer to the state machine structure
 * \param[in]  event
 *              Event for this state
 *
 * \return     VOID
 *
 * \access     CAN task
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static void csm_vSsmPutEv(tCsmSm* me, tCsmProxyEv event)
{
  if (me->current != NULL)
  {
    // only do sth. if the current state is valid (initialized)
    // create local event
    tCsmProxyEv processed;

    // we are now busy
    me->working = 1;

    // call the state function of the current state
    // if the next state should be entered, the NONE event is given as param
    processed = (me->current)(event);

    // check if current state has processed the event
    if (processed != CSM_C_EV_NONE && me->super != NULL)
    {
      // unhandled event found and 'super' is NOT NULL
      // then let the super state handle the event
      (me->super)(processed);
    }

    // as long as the 'next' state has not been reached
    while (me->next != NULL)
    {
      // 'next' pointer will become NULL if new state is reached
      // force the exit functionality of the still current state
      (me->current)(CSM_C_EV_EXIT);
      // make the current state the requested one and clear the 'next' pointer
      me->current = me->next;
      me->next = NULL;
      // force the entry functionality of the new current state
      // it can happen here, that the entry sets a new 'next' so that the loop continues to run
      (me->current)(CSM_C_EV_ENTRY);
    }
    // we are not busy anymore
    me->working = 0;
  }
}

/*******************************************************************************
 * function    csm_vSsmGetState
 * \doxydocu
 * \brief      Returns the current state (function pointer).
 *
 * \param      me
 *              Pointer to the state machine structure
 *
 * \return     VOID
 *
 * \access     CAN task
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static tpfCsmSmState csm_vSsmGetState(const tCsmSm* me)
{
  return (me->current);
}

/*******************************************************************************
 * function    CSM_PROXY_bIsActive
 * \doxydocu
 * \brief      Returns if the Proxy is active
 *
 * \param      void
 *
 * \return     BOOL 1 if proxy is ON, 0 if proxy is OFF
 *
 * \access     CAN task
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 * 07.11.16  A.Borck
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
BOOL CSM_PROXY_bIsActive(void)
{
  if (csm_vSsmGetState(&rCsmSmRemoteNet) == csm_tSsmRemoteNetActive)
  {
    return TRUE;
  }

  return FALSE;
}

/******************************************************************************/
/* state functions for the remote net actions */
/******************************************************************************/

/*******************************************************************************
 * function    csm_tSsmRemoteNetInactive
 * \doxydocu
 * \brief      This state represents the idle mode, waiting only for POW_ON.
 *
 * \param[in]  ev
 *              Event to feed
 *
 * \return     ev
 *              Depending on the usage of the put in event, it is given back or
 *              CSM_C_EV_NONE
 *
 * \access     CAN task
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static tCsmProxyEv csm_tSsmRemoteNetInactive(tCsmProxyEv ev)
{
  // we are in INACTIVE
  // evaluate the trigger event
  switch(ev)
  {
    // this event has currently been fired directly after the initialization
    case CSM_C_EV_ENTRY:
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR1(("csm_tSsmRemoteNetInactive - called with CSM_C_EV_ENTRY"));
      #endif
      #endif
      #endif

      // no action
      break;

    // this event has been fired after the stack has been reached ACTIVE
    case CSM_C_EV_POW_ON:
      // initialize the INC channels (opening sockets, creating threads)
      CSM_lIpcInitAll();
      // set the next state STARTUP
      csm_vSsmGoto(&rCsmSmRemoteNet, csm_tSsmRemoteNetStartup);
      break;

    // unhandled states simply return the given event
    default:
      return ev;
  }

  // signalize that we have handled the event
  return CSM_C_EV_NONE;
}

/*******************************************************************************
 * function    csm_tSsmRemoteNetStartup
 * \doxydocu
 * \brief      Waiting for the host to give us feedback to component status
 *             request "active".
 *
 * \param[in]  ev
 *              Event to feed
 *
 * \return     ev
 *              Depending on the usage of the put in event, it is given back or
 *              CSM_C_EV_NONE
 *
 * \access     CAN task
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static tCsmProxyEv csm_tSsmRemoteNetStartup(tCsmProxyEv ev)
{
  // we are in STARTUP
  // evaluate the trigger event
  switch (ev)
  {
    // entry functionality
    case CSM_C_EV_ENTRY:
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR1(("csm_tSsmRemoteNetStartup - called with CSM_C_EV_ENTRY"));
      #endif
      #endif
      #endif

      // trigger a component status request via INC (to our CCS on remote host)
      rIpcDataRemoteNet.u8CompStatusReq = CSM_C_REMOTE_NET_CTRL_COMP_STATUS_ACTIVE;
      csm_vNetTxCompStatusReq(rIpcDataRemoteNet.u8CompStatusReq);
      // start timer for the LCTRL path
      CSM_lTimerStart(CSM_TIMER_NET_LCTRL, CSM_C_TIMEOUT_NET_LCTRL);
      break;

    // exit functionality
    case CSM_C_EV_EXIT:
      // stop timer for the LCTRL path
      CSM_lTimerStop(CSM_TIMER_NET_LCTRL);
      break;

    // handle timeout for this state (INC component status request)
    case CSM_C_EV_TIMEOUT:
      // transmit request again and again
      csm_vNetTxCompStatusReq(rIpcDataRemoteNet.u8CompStatusReq);
      CSM_lTimerStart(CSM_TIMER_NET_LCTRL, CSM_C_TIMEOUT_NET_LCTRL);
      break;

    // proxy exit function has forced this event
    case CSM_C_EV_POW_OFF:
      // set the next state WAIT_ALL_COMM_DISCONNECT
      csm_vSsmGoto(&rCsmSmRemoteNet, csm_tSsmRemoteNetWaitAllCommDisc);
      break;

    // the remote component has confirmed the 'active' state so that we can proceed with next state
    case CSM_C_EV_CONF_ACTIVE:
      // set the next state ACTIVE
      csm_vSsmGoto(&rCsmSmRemoteNet, csm_tSsmRemoteNetActive);
      break;

    // unhandled events
    default:
      return ev;
  }

  return CSM_C_EV_NONE;
}

/*******************************************************************************
 * function    csm_tSsmRemoteNetActive
 * \doxydocu
 * \brief      Communication between host and proxy established/active.
 *
 * \param[in]  ev
 *              Event to feed
 *
 * \return     ev
 *              Depending on the usage of the put in event, it is given back or
 *              CSM_C_EV_NONE
 *
 * \access     CAN task
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static tCsmProxyEv csm_tSsmRemoteNetActive(tCsmProxyEv ev)
{
  // we are in ACTIVE
  // evaluate the trigger event
  switch(ev)
  {
    // entry functionality
    case CSM_C_EV_ENTRY:
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR1(("csm_tSsmRemoteNetActive - called with CSM_C_EV_ENTRY"));
      #endif
      #endif
      #endif
      // retrigger broadcast manager (to TX comm requests)
      csm_vTriggerCsmDownstreamTask(CSM_C_DOWN_REASON_IPC_LOCAL_BR_REQ_TX);
      break;

    // exit functionality
    case CSM_C_EV_EXIT:
      // none
      break;

    // proxy exit function has forced this event
    case CSM_C_EV_POW_OFF:
      // set the next state WAIT_ALL_COMM_DISCONNECT
      csm_vSsmGoto(&rCsmSmRemoteNet, csm_tSsmRemoteNetWaitAllCommDisc);
      break;

    // the remote component has confirmed/indicated the 'inactive' state
    // due to REJECT with WRONG VERSION or due to component status request
    // so that we can proceed with next state
    case CSM_C_EV_CONF_INACTIVE:
    // this restart event is currently not actively triggered from somewhere
    case CSM_C_EV_RESTART:
      // set the next state STARTUP (continue with component status request)
      csm_vSsmGoto(&rCsmSmRemoteNet, csm_tSsmRemoteNetStartup);
      break;

    // unhandled events
    case CSM_C_EV_QUIT:
    default:
      return ev;
  }

  return CSM_C_EV_NONE;
}

/*******************************************************************************
 * function    csm_tSsmRemoteNetWaitAllCommDisc
 * \doxydocu
 * \brief      Before switching off, we have to wait for all channels to be
 *             closed.
 *
 * \param[in]  ev
 *              Event to feed
 *
 * \return     ev
 *              Depending on the usage of the put in event, it is given back or
 *              CSM_C_EV_NONE
 *
 * \access     CAN task
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static tCsmProxyEv csm_tSsmRemoteNetWaitAllCommDisc(tCsmProxyEv ev)
{
  // we are in WAIT_ALL_COMM_DISCONNECT
  // evaluate the trigger event
  switch (ev)
  {
    // entry functionality
    case CSM_C_EV_ENTRY:
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR1(("csm_tSsmRemoteNetWaitAllCommDisc - called with CSM_C_EV_ENTRY"));
      #endif
      #endif
      #endif

      // check if one of the CBR users (here VCAN) or one of the TP channels need the bus
      if (csm_fCommReqIsPending() == FALSE)
      {
        // we can change directly to SHUTDOWN after this state has been reached completely
        // a Goto() within a running transition (Put) will not be executed directly.
        // Instead of, the Goto() only will set the 'next' pointer and after returning from this
        // entry action, the while loop will continue processing the next state
        csm_vSsmGoto(&rCsmSmRemoteNet, csm_tSsmRemoteNetShutdown);
      }
      break;

    // this com all off event is currently not actively triggered from somewhere
    case CSM_C_EV_COMM_ALLOFF:
      csm_vSsmGoto(&rCsmSmRemoteNet, csm_tSsmRemoteNetShutdown);
      break;

    // 'stack activated' has been called -> POWER ON event
    case CSM_C_EV_POW_ON:
      if (rIpcDataRemoteNet.u8CompStatusAct == CSM_C_REMOTE_NET_CTRL_COMP_STATUS_ACTIVE)
      {
        // we are already (or still active), go directly to state ACTIVE again
        csm_vSsmGoto(&rCsmSmRemoteNet, csm_tSsmRemoteNetActive);
      }
      else
      {
        // if not active we go to STARTUP
        csm_vSsmGoto(&rCsmSmRemoteNet, csm_tSsmRemoteNetStartup);
      }
      break;

    // unhandled events
    default:
      return ev;
  }

  return CSM_C_EV_NONE;
}

/*******************************************************************************
 * function    csm_tSsmRemoteNetShutdown
 * \doxydocu
 * \brief      Waiting for the host to give us feedback to component status
 *             request "inactive".
 *
 * \param[in]  ev
 *              Event to feed
 *
 * \return     ev
 *              Depending on the usage of the put in event, it is given back or
 *              CSM_C_EV_NONE
 *
 * \access     CAN task
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static tCsmProxyEv csm_tSsmRemoteNetShutdown(tCsmProxyEv ev)
{
  // we are in SHUTDOWN
  // evaluate the trigger event
  switch (ev)
  {
    // entry functionality
    case CSM_C_EV_ENTRY:
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR1(("csm_tSsmRemoteNetShutdown - called with CSM_C_EV_ENTRY"));
      #endif
      #endif
      #endif
      // trigger a component status request via INC (to our CCS on remote host)
      rIpcDataRemoteNet.u8CompStatusReq = CSM_C_REMOTE_NET_CTRL_COMP_STATUS_INACTIVE;
      csm_vNetTxCompStatusReq(rIpcDataRemoteNet.u8CompStatusReq);
      // start timer for the LCTRL path
      CSM_lTimerStart(CSM_TIMER_NET_LCTRL, CSM_C_TIMEOUT_NET_LCTRL);
      break;

    // exit functionality
    case CSM_C_EV_EXIT:
      // stop timer for the LCTRL path
      CSM_lTimerStop(CSM_TIMER_NET_LCTRL);
      break;

    // handle timeout for this state (INC component status request)
    case CSM_C_EV_TIMEOUT:
      // transmit request again and again
      csm_vNetTxCompStatusReq(rIpcDataRemoteNet.u8CompStatusReq);
      CSM_lTimerStart(CSM_TIMER_NET_LCTRL, CSM_C_TIMEOUT_NET_LCTRL);
      break;

    // 'stack activated' has been called -> POWER ON event
    case CSM_C_EV_POW_ON:
      // we can go directly to state STARTTUP again
      csm_vSsmGoto(&rCsmSmRemoteNet, csm_tSsmRemoteNetStartup);
      break;

    // the remote component has confirmed the 'inactive' state so that we can proceed with next state
    case CSM_C_EV_CONF_INACTIVE:
      // we can close all our IPC channels
      CSM_lIpcExitAll();
      // set next state
      csm_vSsmGoto(&rCsmSmRemoteNet, csm_tSsmRemoteNetInactive);
      break;

    // unhandled events
    default:
      return ev;
  }

  return CSM_C_EV_NONE;
}

/*******************************************************************************
 * function    csm_vNetProcess_IPC_TxResponseTimeout_LCTRL
 * \doxydocu
 * \brief      This function processes an IPC timeout on the NET channel for LCTRL protocol
 *
 * \param[in]  bTimerId Timer No. in config array
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 07.07.15  Borck
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void csm_vNetProcess_IPC_TxResponseTimeout_LCTRL(BYTE bTimerId)
{
  // the user timer nr must always be smaller than the oseki timer nr !!
  CSM_vAssert(bTimerId < CSM_TIMER_NET_OSEKI);

  // determine if user or state machine timeout -> timer IDs start with CSM_TIMER_NET_LCTRL (state machine)
  if ((bTimerId >= CSM_TIMER_NET_LCTRL_USER_0) &&
      (bTimerId < (CSM_TIMER_NET_LCTRL_USER_0 + CSM_LCTR_C_MAX_USER))
     )
  {
    // user timeout (0 or 1)
    rIpcDataRemoteNet.bCurrentProtocolClient = CSM_NET_PROTOCOL_CLIENT_NONE;              // free the protocol
    rIpcDataRemoteNet.afWaitForCommConf[bTimerId - CSM_TIMER_NET_LCTRL_USER_0] = FALSE;   // clear user related wait for conf flag
    rIpcDataRemoteNet.afWaitForDataConf[bTimerId - CSM_TIMER_NET_LCTRL_USER_0] = FALSE;   // clear user related wait for conf flag
  }
  else
  {
    // state machine timeout, so trigger an event
    csm_vSsmPutEv(&rCsmSmRemoteNet, CSM_C_EV_TIMEOUT);
  }

  // give trace that the OSEKI lock has been cleared
  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR3(("csm_vNetProcess_IPC_TxResponseTimeout_LCTRL - Protocol-Lock:%d", rIpcDataRemoteNet.bCurrentProtocolClient));
  #endif
  #endif
  #endif
}

/*******************************************************************************
 * function    csm_vNetProcess_IPC_TxResponseTimeout_OSEKI
 * \doxydocu
 * \brief      This function processes an IPC timeout on the NET channel for OSEKI protocol
 *
 * \param[in]  bTimerId Timer No. in config array
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 07.07.15  Borck
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void csm_vNetProcess_IPC_TxResponseTimeout_OSEKI(BYTE bTimerId)
{
  BYTE bLintHappy = FALSE;

  // the NET channel timer for the OSEKI path has expired
  // we must release the lock for this protocol type
  // we must clear all related flags
  // we could give a negative confirmation

  CSM_vAssert(bTimerId < CSM_MAX_TIMER);                                    // LINT happy

  rIpcDataRemoteNet.bCurrentProtocolClient = CSM_NET_PROTOCOL_CLIENT_NONE;  // free the protocol
  rIpcDataRemoteNet.afWaitForCommConf[0] = FALSE;                           // user number is always 0 for OSEKI

  // OSEKI CommRequest (ReInitMonitoring) is now working with a Confirmation on CCS/CCC Level
  // also the REJECT path is working by stopping the timer
  // so this callback should never come !!!
  CSM_vAssert(bLintHappy == TRUE);

  // give trace that the OSEKI lock has been cleared
  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR3(("csm_vNetProcess_IPC_TxResponseTimeout_OSEKI - Protocol-Lock:%d", rIpcDataRemoteNet.bCurrentProtocolClient));
  #endif
  #endif
  #endif
}

/*******************************************************************************
 * function    csm_vNetProcess_IPC_RxData
 * \doxydocu
 * \brief      This function processes the incomming data from remote net control
 *
 * \param[in]  pu8Data
 *              pointer to data
 * \param[in]  u32Length
 *              length of data
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void csm_vNetProcess_IPC_RxData(BYTE* pu8Data, DWORD u32Length)
{
  BYTE bValid = FALSE;

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR3(("csm_vNetProcess_IPC_RxData - Len:0x%04X  Data:0x%02X", u32Length, ETG_LIST_LEN(u32Length), ETG_LIST_PTR_T8(pu8Data) ));
  #endif
  #endif
  #endif

  // evaluate Service Code
  switch(*(pu8Data))
  {
    //******************************************************************
    // a NET related Request to V850 has been rejected by V850
    //******************************************************************
    case IPC_SRVC_CODE_SERVICE_REJECT:
    {
      // strip INC header and forward to strip CSM header
      csm_vNetProcessReject(pu8Data + CSM_C_REMOTE_NET_CTRL_IPCA_HEADER_LENGTH, (u32Length - CSM_C_REMOTE_NET_CTRL_IPCA_HEADER_LENGTH));
    }
    break;

    //******************************************************************
    // a NET related component status confirmation has arrived
    //******************************************************************
    case IPC_SRVC_CODE_COMP_STATUS_CON:
    {
      // strip INC header and forward to strip CSM header
      csm_vNetProcessCompStatus(pu8Data + CSM_C_REMOTE_NET_CTRL_IPCA_HEADER_LENGTH, (u32Length - CSM_C_REMOTE_NET_CTRL_IPCA_HEADER_LENGTH));
    }
    break;

    //******************************************************************
    // a NET related component status indication has arrived
    //******************************************************************
    case IPC_SRVC_CODE_COMP_STATUS_IND:
    {
      CSM_vAssert(bValid == TRUE);   // this should never happen, since we once have sent a comp state, only a confirmation comes once
      // strip INC header and forward to strip CSM header
      csm_vNetProcessCompStatus(pu8Data + CSM_C_REMOTE_NET_CTRL_IPCA_HEADER_LENGTH, (u32Length - CSM_C_REMOTE_NET_CTRL_IPCA_HEADER_LENGTH));
    }
    break;

    //******************************************************************
    // a NET related bus status indication has arrived
    //******************************************************************
    case IPC_SRVC_CODE_BUS_STATE_IND:
    {
      // strip INC header and forward to strip CSM header
      csm_vNetProcessBusStateInd(pu8Data + CSM_C_REMOTE_NET_CTRL_IPCA_HEADER_LENGTH, (u32Length - CSM_C_REMOTE_NET_CTRL_IPCA_HEADER_LENGTH));
    }
    break;

    //******************************************************************
    // a NET related bus error indication has arrived
    //******************************************************************
    case IPC_SRVC_CODE_BUS_ERROR_IND:
    {
      // strip INC header and forward to strip CSM header
      csm_vNetProcessBusErrorInd(pu8Data + CSM_C_REMOTE_NET_CTRL_IPCA_HEADER_LENGTH, (u32Length - CSM_C_REMOTE_NET_CTRL_IPCA_HEADER_LENGTH));
    }
    break;

    //******************************************************************
    // a NET related communication confirmation has arrived
    //******************************************************************
    case IPC_SRVC_CODE_COMMUNICATION_CON:
    {
      // strip INC header and forward to strip CSM header
      csm_vNetProcessCommunicationCon(pu8Data + CSM_C_REMOTE_NET_CTRL_IPCA_HEADER_LENGTH, (u32Length - CSM_C_REMOTE_NET_CTRL_IPCA_HEADER_LENGTH));
    }
    break;

    //******************************************************************
    // a NET related communication indication has arrived
    //******************************************************************
    case IPC_SRVC_CODE_COMMUNICATION_IND:
    {
      // strip INC header and forward to strip CSM header
      csm_vNetProcessCommunicationInd(pu8Data + CSM_C_REMOTE_NET_CTRL_IPCA_HEADER_LENGTH, (u32Length - CSM_C_REMOTE_NET_CTRL_IPCA_HEADER_LENGTH));
    }
    break;

    //******************************************************************
    // a NET related data confirmation has arrived
    //******************************************************************
    case IPC_SRVC_CODE_DATA_CON:
    {
      // strip INC header and forward to strip CSM header
      csm_vNetProcessDataCon(pu8Data + CSM_C_REMOTE_NET_CTRL_IPCA_HEADER_LENGTH, (u32Length - CSM_C_REMOTE_NET_CTRL_IPCA_HEADER_LENGTH));
    }
    break;

    // unknown message, report garbage
    default:
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR3(("csm_vNetProcess_IPC_RxData - Invalid service code:0x%02X", pu8Data[0]));
      #endif
      #endif
      #endif
      //CSM_vAssert(FALSE);
    }
    break;
  }
}

/*******************************************************************************
 * function    csm_vTpProcessReject
 * \doxydocu
 * \brief      This function processes a Reject message from TP channel
 *
 * \param[in]  wTpChanNum   INC TP channel number
 * \param[in]  pu8Data      pointer to data
 * \param[in]  u32Length    length of data
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 19.06.15 Borck
 *
 * \history_end
 ******************************************************************************/
static void csm_vNetProcessReject(BYTE* pu8Data, DWORD u32Length)
{
  tCsmProxyEv  tyStatusEv;
  BYTE  bUserNumber;
  DWORD dwProtocolType;
  BYTE  bBus;
  WORD  wApplId;
  BYTE  bRejMsgCode;
  BYTE  bRejReason;
  BYTE  bValidProtocol = FALSE;

  CSM_vAssert(u32Length < CSM_C_BUFFER_SIZE_REMOTE_RX);

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR3(("csm_vNetProcessReject - for SrvCode:%x  Reason:%x ",
                   ETG_CENUM(tCSM_IPC_SERVICE_CODE,  (tU8) pu8Data[0]),
                   ETG_CENUM(tCSM_IPC_REJECT_REASON, (tU8) pu8Data[1])
                ));
  #endif
  #endif
  #endif

  bRejMsgCode = *(pu8Data+0);
  bRejReason  = *(pu8Data+1);
  CSM_vAssert(bRejMsgCode < 0xFF);  // LINT happy (0xFF is hopefully invalid !!)

  dwProtocolType = ((*(pu8Data+3))<<0)| ((*(pu8Data+4))<<8);
  bBus = *(pu8Data+2);
  wApplId = ((*(pu8Data+9))<<0)| ((*(pu8Data+10))<<8);

  // differentiate between COMM_REQ and DATA_REQ
  if (bRejMsgCode == IPC_SRVC_CODE_DATA_REQ)
  {
    // this is the new case (DataReq) to be handled for LCTRL
    switch(dwProtocolType)
    {
      case CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL:
        bUserNumber = 0;                                                         // explicit always 0
        CSM_lTimerStop(CSM_TIMER_NET_LCTRL_USER_0 + bUserNumber);                // stop corresponding timer
        rIpcDataRemoteNet.bCurrentProtocolClient = CSM_NET_PROTOCOL_CLIENT_NONE; // free the protocol
        rIpcDataRemoteNet.afWaitForDataConf[bUserNumber] = FALSE;                // clear marker
        break;

      default:
        // other protocols on NET_CTRL do not support DATA_REQ
        CSM_vAssert(bValidProtocol);
        break;
    }
  }
  else if (bRejMsgCode == IPC_SRVC_CODE_COMMUNICATION_REQ)
  {
    // this was the primary standard case (CommReq) to be handled for LCTRL/OSEKI
    // evaluate protocol type
    switch(dwProtocolType)
    {
      #ifdef CSM_C_PTYPE_DC_OSEKI_NWM
      case CSM_C_PTYPE_DC_OSEKI_NWM:
        CSM_lTimerStop(CSM_TIMER_NET_OSEKI);                                      // stop running timer
        rIpcDataRemoteNet.bCurrentProtocolClient = CSM_NET_PROTOCOL_CLIENT_NONE;  // free the protocol
        rIpcDataRemoteNet.afWaitForCommConf[0] = FALSE;                           // user number is always 0 for OSEKI
        break;
      #endif

      case CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL:
        // it is important to determine the user number here
        bUserNumber = CSM_LCTR_bGetUserNumber(bBus, dwProtocolType, NULL, wApplId);
        if (bUserNumber < CSM_LCTR_C_MAX_USER)
        {
          // ein User Applikation Kommando wurde rejected
          CSM_lTimerStop(CSM_TIMER_NET_LCTRL_USER_0 + bUserNumber);                // stop corresponding timer
          rIpcDataRemoteNet.bCurrentProtocolClient = CSM_NET_PROTOCOL_CLIENT_NONE; // free the protocol
          rIpcDataRemoteNet.afWaitForCommConf[bUserNumber] = FALSE;                // clear marker
        }
        else
        {
          // invalid user number should not occur
          CSM_vAssert(bValidProtocol);
        }
        break;

      default:
        // other protocols on NET_CTRL do not support COM_REQ
        CSM_vAssert(bValidProtocol);
        break;
    }
  }
  else if (bRejMsgCode == IPC_SRVC_CODE_COMP_STATUS_REQ)
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
    #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
    ETG_TRACE_USR3(("csm_vNetProcessReject - Component Status Request REJECTED"));
    #endif
    #endif
    #endif
    if (bRejReason == (BYTE)_WRONG_CSM_VERSION) // enum is type int, although only 1 byte !!
    {
      // stoppe den SM timer damit sonst durch Timeout nicht ein neuer CompStatusReq abgesetzt wird
      CSM_lTimerStop(CSM_TIMER_NET_LCTRL);
      tyStatusEv = CSM_C_EV_CONF_INACTIVE;
      // trigger state machine with event for component status
      csm_vSsmPutEv(&rCsmSmRemoteNet, tyStatusEv);
    }
  }

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR3(("csm_vNetProcessReject - Protocol-Lock:%d", rIpcDataRemoteNet.bCurrentProtocolClient));
  #endif
  #endif
  #endif
}

/*******************************************************************************
 * function    csm_vNetProcessCompStatus
 * \doxydocu
 * \brief      This function processes the component status confirmation message
 *             from remote net control
 *
 * \param[in]  pu8Data
 *              pointer to data
 * \param[in]  u32Length
 *              length of data
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static void csm_vNetProcessCompStatus(BYTE* pu8Data, DWORD u32Length)
{
  DWORD dwCsmVersion;
  tCsmProxyEv tyStatusEv;

  CSM_vAssert(u32Length < CSM_C_BUFFER_SIZE_REMOTE_RX);

  // get current remote status from data
  rIpcDataRemoteNet.u8CompStatusAct = *pu8Data;
  // get remote version info from data
  dwCsmVersion = (((DWORD)*(pu8Data+1)) << 24) | (((DWORD)*(pu8Data+2)) << 16) | (((DWORD)*(pu8Data+3)) << 8) | (((DWORD)*(pu8Data+4)) << 0);

  // compare loacal with remote version info
  #if (CSM_S_VARIANTS_MANAGEMENT == CSM_C_F_ON)
    #if (CSM_S_SLOT_BASED_CONFIG == CSM_C_F_ON)
    BYTE bSLOT = *(pu8Data+5);
    // ensure that both: VERSION and SLOT are matching !!!
    if( (dwCsmVersion == CSM_C_VERSION) && (bSLOT == *(pu8Data+5)) )
    #endif // CSM_S_SLOT_BASED_CONFIG
  #else
  // ensure that VERSION is matching
  if( dwCsmVersion == CSM_C_VERSION)
  #endif
  {
    if (rIpcDataRemoteNet.u8CompStatusAct == CSM_C_REMOTE_NET_CTRL_COMP_STATUS_ACTIVE)
    {
      // set trigger event accordingly
      tyStatusEv = CSM_C_EV_CONF_ACTIVE;
    }
    else
    {
      tyStatusEv = CSM_C_EV_CONF_INACTIVE;
    }
    // trigger state machine with event for component status
    csm_vSsmPutEv(&rCsmSmRemoteNet, tyStatusEv);
  }
  else
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
    #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
    ETG_TRACE_FATAL(("csm_vNetProcessCompStatus - CSM Proxy VERSION CHECK 0x%08X != 0x%08X ", dwCsmVersion, CSM_C_VERSION));
    #endif
    #endif
    #endif
  }
}

/*******************************************************************************
 * function    csm_vNetProcessBusStateInd
 * \doxydocu
 * \brief      This function processes the bus state indication message
 *             from remote net control
 *
 * \param[in]  pu8Data
 *              pointer to data
 * \param[in]  u32Length
 *              length of data
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static void csm_vNetProcessBusStateInd(BYTE* pu8Data, DWORD u32Length)
{
  CSM_vAssert(u32Length < CSM_C_BUFFER_SIZE_REMOTE_RX);

  if (*pu8Data == CSM_C_STATE_BUS_UP)
  {
    // here we can place additional actions if the bus has gone UP
  }

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_FATAL(("csm_vNetProcessBusStateInd - Bus:%x  State:%x",
                    ETG_CENUM(tCSM_BUS_NUMBER,       (tU8) pu8Data[1]),
                    ETG_CENUM(tCSM_BUS_STATE,        (tU8) pu8Data[0])
                 ));
  #endif
  #endif
  #endif

  // report new state to local CSM (Bus Nr, Bus State)
  CSM_lBusStateChange(pu8Data[1], pu8Data[0]);
}

/*******************************************************************************
 * function    csm_vNetProcessBusErrorInd
 * \doxydocu
 * \brief      This function processes the bus error indication message
 *             from remote net control
 *
 * \param[in]  pu8Data
 *              pointer to data
 * \param[in]  u32Length
 *              length of data
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static void csm_vNetProcessBusErrorInd(BYTE* pu8Data, DWORD u32Length)
{
  CSM_vAssert(u32Length < CSM_C_BUFFER_SIZE_REMOTE_RX);

  #if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
  BYTE i;
  WORD wBusError, wIndic;

  BYTE bBus = pu8Data[0];
  wBusError = CSM_M_MAKE_WORD(pu8Data[2], pu8Data[1]);

  wIndic = (rIpcDataRemoteNet.au16BusError[bBus] ^ wBusError);
  rIpcDataRemoteNet.au16BusError[bBus] = wBusError;

  for (i=0; i<16; i++)
  {
    // Indicate changes
    if (((1 << i) & wIndic) != 0)
    {
      CSM_vIndicateBusError(bBus, (1 << i), ((wBusError & (1 << i)) != 0));
    }
  }
  #endif
}

/*******************************************************************************
 * function    csm_vNetProcessCommunicationCon
 * \doxydocu
 * \brief      This function processes communication confirmation msg
 *             from local control channel
 *
 * \param[in]  pu8Data
 *              pointer to data
 * \param[in]  u32Length
 *              length of data
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static void csm_vNetProcessCommunicationCon(BYTE* pu8Data, DWORD u32Length)
{
  CSM_vAssert(u32Length < CSM_C_BUFFER_SIZE_REMOTE_RX);

  BYTE bValidProtocol = FALSE;
  BYTE  bBus = (*(pu8Data + 0));
  DWORD dwProtocolType = (*(pu8Data + 1)) | ((*(pu8Data + 2)) << 8);
  BYTE  abAddr[4] = { (*(pu8Data + 3)), (*(pu8Data + 4)), (*(pu8Data + 5)), (*(pu8Data + 6)) };
  BYTE  bState = (*(pu8Data + 9));
  WORD  wApplId = (*(pu8Data + 7)) | ((*(pu8Data + 8)) << 8);

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR3(("csm_vNetProcessCommunicationCon - Bus:%x  PType:%x  State:%x  ApplId:%x",
                   ETG_CENUM(tCSM_BUS_NUMBER,        (tU8)  bBus),
                   ETG_CENUM(tCSM_PROTOCOL_TYPE,     (tU32) dwProtocolType),
                   ETG_CENUM(tCSM_CONNECTION_STATE,  (tU8)  bState),
                   ETG_CENUM(tCSM_APPL_ID,           (tU16) wApplId)
                ));
  #endif
  #endif
  #endif

  // evaluate the protocol (special handling for OSEKI !!)
  switch(dwProtocolType)
  {
    #ifdef CSM_C_PTYPE_DC_OSEKI_NWM
    case CSM_C_PTYPE_DC_OSEKI_NWM:
      if (CNM_OSEKI_CMD_REINIT_MONITOR == bState)
      {
        // it seems, that the REINIT has been successfully reached the OSEKI component in CSM
        // an execution confirmation is NOT given
        CSM_lTimerStop(CSM_TIMER_NET_OSEKI);
        rIpcDataRemoteNet.bCurrentProtocolClient = CSM_NET_PROTOCOL_CLIENT_NONE;  // free the protocol
        rIpcDataRemoteNet.afWaitForCommConf[0] = FALSE;                           // user number is always 0 for OSEKI

        #if (CSM_S_TRACE == CSM_C_F_ON)
        #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
        #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
        ETG_TRACE_USR3(("csm_vNetProcessCommunicationCon - Protocol-Lock:%d", rIpcDataRemoteNet.bCurrentProtocolClient));
        #endif
        #endif
        #endif
      }
    break;
    #endif // CSM_C_PTYPE_DC_OSEKI_NWM

    #ifdef CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL
    case CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL:
    {
      // this is the other possibility if not using OSEKI for NET channel
      BYTE bUserNumber = CSM_LCTR_bGetUserNumber(bBus, dwProtocolType, &abAddr[0], wApplId);
      if( bUserNumber < CSM_LCTR_C_MAX_USER)
      {
        CSM_lTimerStop(CSM_TIMER_NET_LCTRL_USER_0 + bUserNumber);                // stop corresponding timer
        rIpcDataRemoteNet.afWaitForCommConf[bUserNumber] = FALSE;
        rIpcDataRemoteNet.bCurrentProtocolClient = CSM_NET_PROTOCOL_CLIENT_NONE; // free the protocol
      }
      CSM_LCTR_vPROXYCommunicationCon(bBus, dwProtocolType, bState, wApplId);
    }
    break;
    #endif // CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL

    default:
      CSM_vAssert(bValidProtocol == TRUE);
    break;
  }
}

/*******************************************************************************
 * function    csm_vNetProcessCommunicationInd
 * \doxydocu
 * \brief      This function processes communication indication msg
 *             from local control channel
 *
 * \param[in]  pu8Data
 *              pointer to data
 * \param[in]  u32Length
 *              length of data
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static void csm_vNetProcessCommunicationInd(BYTE* pu8Data, DWORD u32Length)
{
  CSM_vAssert(u32Length < CSM_C_BUFFER_SIZE_REMOTE_RX);

  BYTE bBus = pu8Data[0];
  DWORD dwProtocolType = (*(pu8Data + 1)) | ((*(pu8Data + 2)) << 8);
  BYTE  abAddr[4] = { (*(pu8Data + 3)), (*(pu8Data + 4)), (*(pu8Data + 5)), (*(pu8Data + 6)) };
  WORD u16ApplId = (*(pu8Data + 7)) | ((*(pu8Data + 8)) << 8);
  BYTE bConnectState = pu8Data[9];

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR3(("csm_vNetProcessCommunicationInd - Bus:%x  PType:%x  Addr:0x%02x  State:0x%x  ApplId:%x",
                  ETG_CENUM(tCSM_BUS_NUMBER,        (tU8)  bBus),
                  ETG_CENUM(tCSM_PROTOCOL_TYPE,     (tU32) dwProtocolType),
                  ETG_LIST_LEN(4), ETG_LIST_PTR_T8((pu8Data+3)),             // Addr Field as Bytes
                  (tU8) bConnectState,                                       // Con State as Byte (different interpretations possible !!)
                  ETG_CENUM(tCSM_APPL_ID,           (tU16) u16ApplId)
                ));
  #endif
  #endif
  #endif

  switch(dwProtocolType)
  {
    #ifdef CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL
    case CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL:
    {
      CSM_LCTR_vPROXYCommunicationInd(bBus, dwProtocolType, bConnectState, u16ApplId);
    }
    break;
    #endif // CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL

    #ifdef CSM_C_PTYPE_DC_OSEKI_NWM
    case CSM_C_PTYPE_DC_OSEKI_NWM:
    {
     // CSM_OSEKI_vPROXYCommunicationInd(bBus, dwProtocolType, abAddr, bConnectState, u16ApplId);
    }
    break;
    #endif // CSM_C_PTYPE_DC_OSEKI_NWM

    default:
    {
      // LINT happy
    }
    break;
  }
}

/*******************************************************************************
 * function    csm_vNetProcessDataCon
 * \doxydocu
 * \brief      This function processes data conf msg
 *             from local control channel
 *
 * \param[in]  pu8Data
 *              pointer to data
 * \param[in]  u32Length
 *              length of data
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 07.11.16  A.Borck
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static void csm_vNetProcessDataCon(BYTE* pu8Data, DWORD u32Length)
{
  CSM_vAssert(u32Length < CSM_C_BUFFER_SIZE_REMOTE_RX);

  BYTE bBus = pu8Data[0];
  DWORD dwProtocolType = (*(pu8Data + 1)) | ((*(pu8Data + 2)) << 8);
  //BYTE  abAddr[4] = { (*(pu8Data + 3)), (*(pu8Data + 4)), (*(pu8Data + 5)), (*(pu8Data + 6)) };
  //WORD u16ApplId = (*(pu8Data + 7)) | ((*(pu8Data + 8)) << 8);
  BYTE bResult = pu8Data[9];

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR3(("csm_vNetProcessDataCon - Bus:%x  PType:%x  Addr:0x%02x  Result:0x%x",
                  ETG_CENUM(tCSM_BUS_NUMBER,        (tU8)  bBus),
                  ETG_CENUM(tCSM_PROTOCOL_TYPE,     (tU32) dwProtocolType),
                  ETG_LIST_LEN(4), ETG_LIST_PTR_T8((pu8Data+3)),             // Addr Field as Bytes
                  (tU8) bResult                                              // Con State as Byte
                ));
  #endif
  #endif
  #endif

  switch(dwProtocolType)
  {
    #ifdef CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL
    case CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL:
    {
      BYTE bUserNumber = 0;                                                    // explicit always 0
      CSM_lTimerStop(CSM_TIMER_NET_LCTRL_USER_0 + bUserNumber);                // stop corresponding timer
      rIpcDataRemoteNet.bCurrentProtocolClient = CSM_NET_PROTOCOL_CLIENT_NONE; // free the protocol
      rIpcDataRemoteNet.afWaitForDataConf[bUserNumber] = FALSE;                // clear marker

      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR3(("csm_vNetProcessDataCon - Protocol-Lock:%d", rIpcDataRemoteNet.bCurrentProtocolClient));
      #endif
      #endif
      #endif

      // Streng genommen muss jetzt LCTRL aufgerufen werden, um das Ergebnis dort abzuliefern.
      // Da das Interface aber nicht da ist, verzichten wir mal drauf, letztendlich knnte der DataReq
      // bei neg. Resoponse nur automatisch wiederholt werden. Da es aber eh nur ein TTFis Kommando war,
      // kann dies auch manuell erfolgen.
      // CSM_LCTR_vPROXYDataConf(bBus, dwProtocolType, bResult, u16ApplId);
    }
    break;
    #endif // CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL

    // unknown protocol
    default:
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR3(("csm_vNetProcessDataCon - Unknown protocol type - no action"));
      #endif
      #endif
      #endif
    }
    break;
  }
}

/*******************************************************************************
 * function    csm_vNetTxCompStatusReq
 * \doxydocu
 * \brief      This function prepares the IPC buffer for sending component
 *             status request and calls the IPC tx function
 *
 * \param[in]  u8CompStatus
 *              requested component status
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static void csm_vNetTxCompStatusReq(BYTE u8CompStatus)
{
  CSM_vAssert(rIpcDataRemoteNet.u8CompStatusReq == u8CompStatus);

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR3(("csm_vNetTxCompStatusReq - called"));
  #endif
  #endif
  #endif

  rIpcDataRemoteNet.au8BufferTx[0] = IPC_SRVC_CODE_COMP_STATUS_REQ;
  rIpcDataRemoteNet.au8BufferTx[1] = corIpcCfgRemoteNet.au8NetCompAddr[0];
  rIpcDataRemoteNet.au8BufferTx[2] = rIpcDataRemoteNet.u8CompStatusReq;
  rIpcDataRemoteNet.au8BufferTx[3] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD( CSM_C_VERSION));
  rIpcDataRemoteNet.au8BufferTx[4] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD( CSM_C_VERSION));
  rIpcDataRemoteNet.au8BufferTx[5] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD( CSM_C_VERSION));
  rIpcDataRemoteNet.au8BufferTx[6] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD( CSM_C_VERSION));

  #if (CSM_S_VARIANTS_MANAGEMENT == CSM_C_F_ON)
    #if (CSM_S_SLOT_BASED_CONFIG == CSM_C_F_ON)
    rIpcDataRemoteNet.au8BufferTx[7] = CSM_APL_bGetConfigurationSlotNr();
    CSM_lIpcDataSend(rIpcDataRemoteNet.tyIpcHandle, rIpcDataRemoteNet.au8BufferTx, 8);
    #endif // CSM_S_SLOT_BASED_CONFIG
  #else
    CSM_lIpcDataSend(rIpcDataRemoteNet.tyIpcHandle, rIpcDataRemoteNet.au8BufferTx, 7);
  #endif
}

/*******************************************************************************
 * function    csm_vBrProcess_IPC_TxResponseTimeout
 * \doxydocu
 * \brief      This function processes an IPC timeout on the BR channel
 *
 * \param[in]  bTimerId Timer No. in config array
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 07.07.15  Borck
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void csm_vBrProcess_IPC_TxResponseTimeout(BYTE bTimerId)
{
  //##ToDo: if broadcast ComReq should be handled by requirements, a proper negative confirmation must be generated upon timeout

  CSM_vAssert(bTimerId == CSM_TIMER_BRSIGNAL);

  BYTE bUserNumber;

  if (CSM_CBR_MAX_USER == 1) /*lint !e506 yes currently const value boolean, but if define should be != 1 then correct */
  {
    bUserNumber = 0;
    // clear Wait Flag to allow new requests
    if (rIpcDataBrSignalCom.afWaitForCommConf[bUserNumber])
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR3(("csm_vBrProcess_IPC_TxResponseTimeout - clear 'WaitForCommConf' Flag"));
      #endif
      #endif
      #endif
      rIpcDataBrSignalCom.afWaitForCommConf[bUserNumber] = FALSE;
    }
  }
  else
  {
    CSM_vAssert(CSM_CBR_MAX_USER == 1); /*lint !e506 yes currently const value boolean, but if define should be != 1 then correct */
  }

  //##ToDo: hier msste dann noch eine neg. confirmation getriggert werden

  // retrigger pending TX requests
  csm_vTriggerCsmDownstreamTask(CSM_C_DOWN_REASON_IPC_LOCAL_BR_REQ_TX);
}

/*******************************************************************************
 * function    csm_vBrProcess_IPC_RxData
 * \doxydocu
 * \brief      This function processes the incoming data from BR signal channel
 *
 * \param[in]  pu8Data
 *              pointer to data
 * \param[in]  u32Length
 *              length of data
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void csm_vBrProcess_IPC_RxData(BYTE* pu8Data, DWORD u32Length)
{
  BYTE bValidService = FALSE;

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR4(("csm_vBrProcess_IPC_RxData - Len:0x%04x  Data:0x%02x",
                   u32Length,
                   ETG_LIST_LEN(u32Length), ETG_LIST_PTR_T8(pu8Data)
                ));
  #endif
  #endif
  #endif

  switch(pu8Data[0])
  {
    //******************************************************************
    // a BR related Request to V850 has been rejected by V850
    //******************************************************************
    case IPC_SRVC_CODE_SERVICE_REJECT:
    {
      // strip INC header and forward to strip CSM header
      csm_vBrProcessReject(pu8Data + CSM_C_BR_SIGNAL_CTRL_IPCA_HEADER_LENGTH, (u32Length - CSM_C_BR_SIGNAL_CTRL_IPCA_HEADER_LENGTH));
    }
    break;

    //******************************************************************
    // a BR related Signal Indication has been received
    //******************************************************************
    case IPC_SRVC_CODE_SIGNAL_IND:
    {
      // strip INC header and forward to strip CSM header
      csm_vBrProcessSignalInd(pu8Data + CSM_C_BR_SIGNAL_CTRL_IPCA_HEADER_LENGTH, (u32Length - CSM_C_BR_SIGNAL_CTRL_IPCA_HEADER_LENGTH));
    }
    break;
    //******************************************************************
    // a BR related Signal Indication has been received
    //******************************************************************
    case IPC_SRVC_CODE_MULTI_SIGNAL_IND:
    {
      // strip INC header and forward to strip CSM header
      csm_vBrProcessMultiSignalInd(pu8Data + CSM_C_BR_SIGNAL_CTRL_IPCA_HEADER_LENGTH, (u32Length - CSM_C_BR_SIGNAL_CTRL_IPCA_HEADER_LENGTH));
    }
    break;

    //******************************************************************
    // a BR related Communication Confirmation has been received
    //******************************************************************
    case IPC_SRVC_CODE_COMMUNICATION_CON:
    {
      // strip INC header and forward to strip CSM header
      csm_vBrProcessCommunicationCon(pu8Data + CSM_C_BR_SIGNAL_CTRL_IPCA_HEADER_LENGTH, (u32Length - CSM_C_BR_SIGNAL_CTRL_IPCA_HEADER_LENGTH));
    }
    break;

    //******************************************************************
    // a BR related Communication indication has been received
    //******************************************************************
    case IPC_SRVC_CODE_COMMUNICATION_IND:
    {
      // strip INC header and forward to strip CSM header
      csm_vBrProcessCommunicationInd(pu8Data + CSM_C_BR_SIGNAL_CTRL_IPCA_HEADER_LENGTH, (u32Length - CSM_C_BR_SIGNAL_CTRL_IPCA_HEADER_LENGTH));
    }
    break;

    default:
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR3(("csm_vBrProcess_IPC_RxData - Invalid service code: 0x%02X", pu8Data[0]));
      #endif
      #endif
      #endif
      CSM_vAssert(bValidService == TRUE);
    }
    break;
  }
}

/*******************************************************************************
 * function    csm_vBrProcessReject
 * \doxydocu
 * \brief      This function processes data reject msg from BR signal channel
 *
 * \param[in]  pu8Data
 *              pointer to data
 * \param[in]  u32Length
 *              length of data
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static void csm_vBrProcessReject(BYTE* pu8Data, DWORD u32Length)
{
  DWORD dwProtocolType;
  //BYTE  bBus;
  BYTE  bRejMsgCode;
  BYTE  bRejReason;
  BYTE  bLocalComp;
  BYTE  bRemoteComp;
  BYTE  bValidMsgCode = FALSE;

  CSM_vAssert(u32Length < CSM_C_BUFFER_SIZE_BR_RX);

  bRejMsgCode = *(pu8Data+0);
  bRejReason  = *(pu8Data+1);
  //bBus        = *(pu8Data+2);
  dwProtocolType = ((*(pu8Data+3))<<0)| ((*(pu8Data+4))<<8);
  bLocalComp  = *(pu8Data+5);
  bRemoteComp = *(pu8Data+6);

  CSM_vAssert(bLocalComp == CSM_C_ADDRESS_FIELD_INF4CV_BR_VCAN);
  CSM_vAssert(bRemoteComp == 0xFF);
  CSM_vAssert(dwProtocolType == CSM_C_PTYPE_DC_BR_SIGNAL);

  // evaluate the rejected msg code
  switch(bRejMsgCode)
  {
    case IPC_SRVC_CODE_SIGNAL_WRITE_REQ:
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_FATAL(("csm_vBrProcessReject - Signal Write rejected - Reason:0x%02X", bRejReason));
      #endif
      #endif
      #endif
      // we can stop the timer and reset marker
      CSM_lTimerStop(CSM_TIMER_BRSIGNAL);
      rIpcDataBrSignalCom.afWaitForCommConf[0] = FALSE;
      CSM_vAssert(CSM_CBR_MAX_USER == 1);   /*lint !e506 yes currently const value boolean, but if define should be != 1 then correct */
    }
    break;

    case IPC_SRVC_CODE_COMMUNICATION_REQ:
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_FATAL(("csm_vBrProcessReject - Communication Request rejected - Reason:0x%02X", bRejReason));
      #endif
      #endif
      #endif
      // we can stop the timer and reset marker
      CSM_lTimerStop(CSM_TIMER_BRSIGNAL);
      rIpcDataBrSignalCom.afWaitForCommConf[0] = FALSE;
      CSM_vAssert(CSM_CBR_MAX_USER == 1);  /*lint !e506 yes currently const value boolean, but if define should be != 1 then correct */
    }
    break;

    // unknown rejected msg code
    default:
    {
      CSM_vAssert(bValidMsgCode == TRUE);
    }
    break;
  }
}

/*******************************************************************************
 * function    csm_vBrProcessSignalInd
 * \doxydocu
 * \brief      This function processes signal indication msg from BR signal channel
 *
 * \param[in]  pu8Data
 *              pointer to data
 * \param[in]  u32Length
 *              length of data
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static void csm_vBrProcessSignalInd(BYTE* pu8Data, DWORD u32Length)
{
  DWORD u32SignalID;
  DWORD u32SignalStatus;
  CSM_CAN_RX_DATA rData;

  BYTE bBus = (*(pu8Data + 0));
  CSM_vAssert(bBus == CSM_C_CAN_BUS_VCAN);            // only VCAN has BR

  // DWORD dwProtocolType = (*(pu8Data + 1)) | ((*(pu8Data + 2)) << 8);
  // BYTE  abAddr[4] = { (*(pu8Data + 3)), (*(pu8Data + 4)), (*(pu8Data + 5)), (*(pu8Data + 6)) };

  u32SignalID     = CSM_M_MAKE_DWORD( CSM_M_MAKE_WORD( *(pu8Data+10), *(pu8Data+9)), CSM_M_MAKE_WORD( *(pu8Data+8), *(pu8Data+7)));
  u32SignalStatus = CSM_M_MAKE_DWORD( CSM_M_MAKE_WORD( *(pu8Data+15), *(pu8Data+14)), CSM_M_MAKE_WORD( *(pu8Data+13), *(pu8Data+12)));

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR4(("csm_vBrProcessSignalInd - ID:0x%08x  Status:%x  Len:0x%02x  Data:%02x",
                   u32SignalID,
                   ETG_CENUM(tCSM_SIGNAL_STATUS_INFO, (tU32) u32SignalStatus),
                   u32Length - 16,
                   ETG_LIST_LEN(u32Length - 16),
                   ETG_LIST_PTR_T8((pu8Data + 16))
                ));
  #endif
  #endif
  #endif

  rData.bBus = CSM_M_GET_BUSNUMBER_FROM_SIGNAL_ID(u32SignalID);
  rData.dwIdentifier = 0;
  CSM_vMemCopy(rData.abData, pu8Data + 16, u32Length - 16);
  rData.bTelegramType  = CSM_C_TEL_TYPE_CBR_RX;
  rData.bTelegramIndex = CSM_M_GET_TELINDEX_FROM_SIGNAL_ID(u32SignalID);
  rData.bLengthCode    = u32Length - 16;

  // pass data to CBR
  CBR_vPROXYReceiveInd(&rData, u32SignalStatus);
}

/*******************************************************************************
 * function    csm_vBrProcessMultiSignalInd
 * \doxydocu
 * \brief      This function processes signal indication msg from BR signal channel
 *
 * \param[in]  pu8Data
 *              pointer to data
 * \param[in]  u32Length
 *              length of data
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static void csm_vBrProcessMultiSignalInd(BYTE* pu8Data, DWORD u32Length)
{
  DWORD u32SignalID;
  DWORD u32SignalStatus;
  CSM_CAN_RX_DATA rData;
  BYTE bNumberOfTelegrams;
  BYTE bCount;

  BYTE bBus = (*(pu8Data + 0));
  CSM_vAssert(bBus == CSM_C_CAN_BUS_VCAN);            // only VCAN has BR

  // DWORD dwProtocolType = (*(pu8Data + 1)) | ((*(pu8Data + 2)) << 8);
  // BYTE  abAddr[4] = { (*(pu8Data + 3)), (*(pu8Data + 4)), (*(pu8Data + 5)), (*(pu8Data + 6)) };

  bNumberOfTelegrams = (*(pu8Data + 7));

  for(bCount = 0; bCount<bNumberOfTelegrams; bCount++)
  {
    BYTE bSignalLength;
    u32SignalID     = CSM_M_MAKE_DWORD( CSM_M_MAKE_WORD( *(pu8Data+11+(bCount*17)), *(pu8Data+10+(bCount*17))), CSM_M_MAKE_WORD( *(pu8Data+9+(bCount*17)), *(pu8Data+8+(bCount*17))));
    bSignalLength   = (*(pu8Data + 12+(bCount*17)));
    u32SignalStatus = CSM_M_MAKE_DWORD( CSM_M_MAKE_WORD( *(pu8Data+16+(bCount*17)), *(pu8Data+15+(bCount*17))), CSM_M_MAKE_WORD( *(pu8Data+14+(bCount*17)), *(pu8Data+13+(bCount*17))));

    #if (CSM_S_TRACE == CSM_C_F_ON)
    #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
    #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
    ETG_TRACE_USR4(("csm_vBrProcessMultiSignalInd - Count:0x%02x  ID:0x%08x  Status:%x  Len:0x%02x  Data:%02x",
                     bCount,
                     u32SignalID,
                     ETG_CENUM(tCSM_SIGNAL_STATUS_INFO, (tU32) u32SignalStatus),
                     bSignalLength,
                     ETG_LIST_LEN( bSignalLength),
                     ETG_LIST_PTR_T8((pu8Data + 17+(bCount*17)))
                  ));
    #endif
    #endif
    #endif

    rData.bBus = CSM_M_GET_BUSNUMBER_FROM_SIGNAL_ID(u32SignalID);
    rData.dwIdentifier = 0;
    CSM_vMemCopy(rData.abData, pu8Data + 17 + (bCount*17), bSignalLength);
    rData.bTelegramType  = CSM_C_TEL_TYPE_CBR_RX;
    rData.bTelegramIndex = CSM_M_GET_TELINDEX_FROM_SIGNAL_ID(u32SignalID);
    rData.bLengthCode    = bSignalLength;

    // pass data to CBR
    CBR_vPROXYReceiveInd(&rData, u32SignalStatus);
  }
} /*lint !e715 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally */

/*******************************************************************************
 * function    csm_vBrProcessCommunicationCon
 * \doxydocu
 * \brief      This function processes communication confirmation msg
 *             from BR signal channel
 *
 * \param[in]  pu8Data
 *              pointer to data
 * \param[in]  u32Length
 *              length of data
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static void csm_vBrProcessCommunicationCon(BYTE* pu8Data, DWORD u32Length)
{
  CSM_vAssert(u32Length < CSM_C_BUFFER_SIZE_BR_RX);

  BYTE  bValidState = FALSE;
  BYTE  bBus = (*(pu8Data + 0));
  DWORD dwProtocolType = (*(pu8Data + 1)) | ((*(pu8Data + 2)) << 8);
  BYTE  abAddr[2] = { (*(pu8Data + 3)), (*(pu8Data + 4)) };
  BYTE  bState = (*(pu8Data + 5));
  WORD  wApplId = (*(pu8Data + 6)) | ((*(pu8Data + 7)) << 8);

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR3(("csm_vBrProcessCommunicationCon - Bus:%x  PType:%x  State:%x  ApplId:%x",
                  ETG_CENUM(tCSM_BUS_NUMBER,      (tU8)  bBus),
                  ETG_CENUM(tCSM_PROTOCOL_TYPE,   (tU32) dwProtocolType),
                  ETG_CENUM(tCSM_CONNECTION_STATE,(tU8)  bState),
                  ETG_CENUM(tCSM_APPL_ID,         (tU16) wApplId)
                ));
  #endif
  #endif
  #endif

  BYTE bUserNumber = CSM_CBR_bGetUserNumber(bBus, dwProtocolType, &abAddr[0], wApplId);
  if (bUserNumber < CSM_CBR_MAX_USER)
  {
    switch(bState)
    {
      case CSM_C_CONNECTED:
      case CSM_C_DISCONNECTED:
      case CSM_C_FNL_ON:
      {
        rIpcDataBrSignalCom.au8CommStatusAct[bUserNumber] = bState;
      }
      break;
      default:
      {
        rIpcDataBrSignalCom.au8CommStatusAct[bUserNumber] = CSM_C_DISCONNECTED;
        CSM_vAssert(bValidState == TRUE);
      }
      break;
    }

    if (rIpcDataBrSignalCom.afWaitForCommConf[bUserNumber] == TRUE)
    {
      // yes, we are waiting for that confirmation
      CSM_lTimerStop(CSM_TIMER_BRSIGNAL);
      rIpcDataBrSignalCom.afWaitForCommConf[bUserNumber] = FALSE;
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR3(("csm_vBrProcessCommunicationCon - got awaited confirmation"));
      #endif
      #endif
      #endif
      CSM_CBR_vPROXYCommunicationCon(bUserNumber, bState);
    }
    else
    {
      // maybe too late, timeout already occured, so convert to an indication
      CSM_CBR_vPROXYCommunicationInd(bUserNumber, bState);
    }
  }
  else
  {
    // bUserNumber >= CSM_CBR_MAX_USER: this should never happen!
    CSM_vAssert(bUserNumber < CSM_CBR_MAX_USER);
  }
}

/*******************************************************************************
 * function    csm_vBrProcessCommunicationInd
 * \doxydocu
 * \brief      This function processes communication Indication msg
 *             from BR signal channel
 *
 * \param[in]  pu8Data
 *              pointer to data
 * \param[in]  u32Length
 *              length of data
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static void csm_vBrProcessCommunicationInd(BYTE* pu8Data, DWORD u32Length)
{
  CSM_vAssert(u32Length < CSM_C_BUFFER_SIZE_BR_RX);

  BYTE  bValidState = FALSE;
  BYTE  bBus = (*(pu8Data + 0));
  DWORD dwProtocolType = (*(pu8Data + 1)) | ((*(pu8Data + 2)) << 8);
  BYTE  abAddr[2] = { (*(pu8Data + 3)), (*(pu8Data + 4)) };
  BYTE  bState = (*(pu8Data + 5));
  WORD  wApplId = (*(pu8Data + 6)) | ((*(pu8Data + 7)) << 8);

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR3(("csm_vBrProcessCommunicationInd - Bus:%x  PType:%x  State:%x  ApplId:%x",
                  ETG_CENUM(tCSM_BUS_NUMBER,      (tU8)  bBus),
                  ETG_CENUM(tCSM_PROTOCOL_TYPE,   (tU32) dwProtocolType),
                  ETG_CENUM(tCSM_CONNECTION_STATE,(tU8)  bState),
                  ETG_CENUM(tCSM_APPL_ID,         (tU16) wApplId)
                ));
  #endif
  #endif
  #endif

  BYTE bUserNumber = CSM_CBR_bGetUserNumber(bBus, dwProtocolType, &abAddr[0], wApplId);
  if (bUserNumber < CSM_CBR_MAX_USER)
  {
    switch(bState)
    {
      case CSM_C_CONNECTED:
      case CSM_C_DISCONNECTED:
      case CSM_C_FNL_ON:
      {
        rIpcDataBrSignalCom.au8CommStatusAct[bUserNumber] = bState;
      }
      break;
      default:
      {
        rIpcDataBrSignalCom.au8CommStatusAct[bUserNumber] = CSM_C_DISCONNECTED;
        CSM_vAssert(bValidState == TRUE);
      }
      break;
    }

    CSM_CBR_vPROXYCommunicationInd(bUserNumber, bState);
  }
  else
  {
    // bUserNumber >= CSM_CBR_MAX_USER: this should never happen!
    CSM_vAssert(bUserNumber < CSM_CBR_MAX_USER);
  }
}

/*******************************************************************************
 * function    csm_vTpProcess_IPC_TxResponseTimeout
 * \doxydocu
 * \brief      This function processes an IPC timeout on the NP channel for TP protocols
 *
 * \param[in]  bTimerId Timer No. in config array
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 07.07.15  Borck
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void csm_vTpProcess_IPC_TxResponseTimeout(BYTE bTimerId)
{
  BYTE bChannel = (bTimerId - CSM_TIMER_NP1_COM)/3; // determine the CAN NP channel (index) belonging to a timer
  BYTE bTimerOffset = bTimerId % 3;                 // determine the TimerOffset within the set

  if (bChannel < CSM_C_PROXY_TP_CHANNELS)           // ensure that channel is in range
  {
    // the v850 has not responded on IPC to a TP related request
    // so we now should generate a negative confirmation to the application
    // timer has been started either upon a CommunicationReq or a DataReq

    switch(bTimerOffset)
    {
      // ComCon timeout
      case 0:
        if (arIpcDataTp[bChannel].fWaitForCommConf == TRUE)
        {
            arIpcDataTp[bChannel].fWaitForCommConf = FALSE;
			if (bChannel < CSM_C_ISO_TP_MAX_CONNECTIONS)
			{
               CSM_ITP_vPROXYTriggerCommunicationCon(bChannel, CSM_C_INACTIVE); //state is unknown
			}
        }
        break;

      // DataCon C missing
      case 1:
        if (arIpcDataTp[bChannel].fWaitForDataConf_C == TRUE)
        {
            // IL is 0 for C-Type
            BYTE bInterruptLevel = 0;
            arIpcDataTp[bChannel].fWaitForDataConf_C = FALSE;
            // create a negative confirmation
            DWORD dwProtocolType = coarIpcCfgTp[ bChannel].dwProtocolType;

      #if (CSM_S_J1939 == CSM_C_F_ON)
      if (dwProtocolType == CSM_C_PTYPE_J1939 ) 
      {
          CSM_ADDR_J1939 rAddrField;

                rAddrField.bSrcNodeID  = arIpcDataTp[bChannel].au8BufferTx_C[4];
                rAddrField.bTgtNodeID  = arIpcDataTp[bChannel].au8BufferTx_C[5];
                rAddrField.wPGN        = (arIpcDataTp[bChannel].au8BufferTx_C[7] << 8) | (arIpcDataTp[bChannel].au8BufferTx_C[6]);

                if (rAddrField.wPGN < CSM_C_J1939_CFG_MAXPGN)
				{
                for(WORD wIdx = 0; wIdx < CSM_C_J1939_MAX_PGN_CALLBACKS; wIdx++)
                {
                  if(csm_j1939_arPGNCallbackFkt[rAddrField.wPGN][wIdx].pfvDataCon != NULL)
                  {
                     csm_j1939_arPGNCallbackFkt[rAddrField.wPGN][wIdx].pfvDataCon(
                     #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                                                  csm_j1939_arPGNCallbackFkt[rAddrField.wPGN][wIdx].pvHandle,
                     #endif
                                        dwProtocolType,
                                                  (const VOID*) &rAddrField,
                                                  CSM_C_CONF_FAULT);
					  }
                  }
                }
      }
      else if ((dwProtocolType == CSM_C_PTYPE_DC_USDT_ISO ) && (bChannel < CSM_C_ISO_TP_MAX_CONNECTIONS))
	  #else
	  if ((dwProtocolType == CSM_C_PTYPE_DC_USDT_ISO ) && (bChannel < CSM_C_ISO_TP_MAX_CONNECTIONS))		  
      #endif
      {
        tCSM_ADDR_ISO_TP rAddrField;

                rAddrField.bLocalComp  = coarIpcCfgTp[ bChannel].au8AddressField[0];
                rAddrField.bRemoteComp = coarIpcCfgTp[ bChannel].au8AddressField[1];

        csm_rIsoTpConnectionAct[bChannel].fTxChannelFree = TRUE;

        if ( (csm_itp_arCallbackFkt[bChannel].rCallback.pfvDataCon != NULL) )
                {
                       csm_itp_arCallbackFkt[bChannel].rCallback.pfvDataCon(
                                                          #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                                                          csm_itp_arCallbackFkt[bChannel].pvHandle,
                                                          #endif
                                                          dwProtocolType,
                                                          (const VOID*) &rAddrField,
                                                          CSM_C_CONF_FAULT); /* ISO TP uses CSM_C_CONF_OK/FAULT */


                }
      }
        }
        break;

      default:
        break;
    }
  }
}

/*******************************************************************************
 * function    csm_vTpProcess_IPC_RxData
 * \doxydocu
 * \brief      This function processes the incoming data from TP channel
 *
 * \param[in]  pu8Data
 *              pointer to data
 * \param[in]  u32Length
 *              length of data
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 13.10.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void csm_vTpProcess_IPC_RxData(WORD wTpChanNum, BYTE* pu8Data, DWORD u32Length)
{
  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR3(("csm_vTpProcess_IPC_RxData - TpChnNum:0x%02X  Len:0x%04X  Data:0x%02x", wTpChanNum, u32Length, ETG_LIST_LEN(u32Length), ETG_LIST_PTR_T8(pu8Data) ));
  #endif
  #endif
  #endif

  // evaluate Service Code
  switch(pu8Data[0])
  {
    //******************************************************************
    // a TP related Request to V850 has been rejected by V850
    //******************************************************************
    case IPC_SRVC_CODE_SERVICE_REJECT:
    {
      // strip INC header and forward to strip CSM header
      csm_vTpProcessReject(wTpChanNum, (pu8Data + CSM_C_TP_CTRL_IPCA_HEADER_LENGTH), (u32Length - CSM_C_TP_CTRL_IPCA_HEADER_LENGTH));
    }
    break;

    //******************************************************************
    // a TP related Data confirmation has arrived
    //******************************************************************
    case IPC_SRVC_CODE_DATA_CON:
    {
      // strip INC header and forward to strip CSM header
      csm_vTpProcessDataCon(wTpChanNum, (pu8Data + CSM_C_TP_CTRL_IPCA_HEADER_LENGTH), (u32Length - CSM_C_TP_CTRL_IPCA_HEADER_LENGTH));
    }
    break;

    case IPC_SRVC_CODE_DATA_IND:
    {
      // strip INC header and forward to strip CSM header
      csm_vTpProcessDataInd(wTpChanNum, (pu8Data + CSM_C_TP_CTRL_IPCA_HEADER_LENGTH), (u32Length - CSM_C_TP_CTRL_IPCA_HEADER_LENGTH));
    }
    break;

    case IPC_SRVC_CODE_COMMUNICATION_CON:
    {
      // strip INC header and forward to strip CSM header
      csm_vTpProcessCommCon(wTpChanNum, (pu8Data + CSM_C_TP_CTRL_IPCA_HEADER_LENGTH), (u32Length - CSM_C_TP_CTRL_IPCA_HEADER_LENGTH));
    }
    break;

    case IPC_SRVC_CODE_COMMUNICATION_IND:
    {
      // strip INC header and forward to strip CSM header
      csm_vTpProcessCommInd(wTpChanNum, (pu8Data + CSM_C_TP_CTRL_IPCA_HEADER_LENGTH), (u32Length - CSM_C_TP_CTRL_IPCA_HEADER_LENGTH));
    }
    break;

    default:
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR3(("csm_vTpProcess_IPC_RxData - UNHANDLED ServiceCode for ChanNum:0x%02x  Len:0x%04x", wTpChanNum, u32Length));
      #endif
      #endif
      #endif
      //CSM_vAssert(FALSE);
    }
    break;
  }
}

/*******************************************************************************
 * function    csm_vTpProcessReject
 * \doxydocu
 * \brief      This function processes a Reject message from TP channel
 *
 * \param[in]  wTpChanNum   INC TP channel number
 * \param[in]  pu8Data      pointer to data
 * \param[in]  u32Length    length of data
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 19.06.15 Borck
 *
 * \history_end
 ******************************************************************************/
static void csm_vTpProcessReject(WORD wTpChanNum, BYTE* pu8Data, DWORD u32Length)
{
  BYTE  bNoError = FALSE;
  WORD  wCount;
  BYTE  bInterruptLevel = 0;
  DWORD dwProtocolType;

  CSM_vAssert(u32Length < CSM_C_BUFFER_SIZE_TP_RX);

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR3(("csm_vTpProcessReject - for SrvCode:%x  Reason:%x ",
                   ETG_CENUM(tCSM_IPC_SERVICE_CODE,  (tU8) pu8Data[0]),
                   ETG_CENUM(tCSM_IPC_REJECT_REASON, (tU8) pu8Data[1])
                ));
  #endif
  #endif
  #endif

  // order of data after reject reason with len: Bus 1, ProtocolType 2, Addr Field 4,

  // loop over all configured channels
  for (wCount = 0; wCount < CSM_C_PROXY_TP_CHANNELS; wCount++)
  {
    // the REJECT was excactly for this channel
    if (wCount == wTpChanNum)
    {
      // check if we are waiting for a Data Confirmation if the rejected service code is DataReq
      if (pu8Data[0] == IPC_SRVC_CODE_DATA_REQ)
      {
        // Protocol Type ist wichtig hier
        dwProtocolType = ((*(pu8Data+2))<<0)| ((*(pu8Data+3))<<8);

    #if (CSM_S_J1939 == CSM_C_F_ON)
      if ((arIpcDataTp[wCount].fWaitForDataConf_C == TRUE) && (dwProtocolType == CSM_C_PTYPE_J1939))
    {
      CSM_ADDR_J1939 rAddrField;

            rAddrField.bSrcNodeID  = pu8Data[4];
            rAddrField.bTgtNodeID  = pu8Data[5];
            rAddrField.wPGN        = (WORD)((pu8Data[7] << 8) | ( pu8Data[6] ));

      CSM_lTimerStop(CSM_TIMER_NP1_TX + (((BYTE)wCount)*3));
            arIpcDataTp[wCount].fWaitForDataConf_C = FALSE;
			if (rAddrField.wPGN < CSM_C_J1939_CFG_MAXPGN)
			{
            for(WORD wIdx = 0; wIdx < CSM_C_J1939_MAX_PGN_CALLBACKS; wIdx++)
            {
              if(csm_j1939_arPGNCallbackFkt[rAddrField.wPGN][wIdx].pfvDataCon != NULL)
              {
                 csm_j1939_arPGNCallbackFkt[rAddrField.wPGN][wIdx].pfvDataCon(
                 #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                                              csm_j1939_arPGNCallbackFkt[rAddrField.wPGN][wIdx].pvHandle,
                 #endif
                                  dwProtocolType,
                                              (const VOID*) &rAddrField,
                                              CSM_C_CONF_FAULT);
				  }
              }
            }
    }
        else if ((arIpcDataTp[wCount].fWaitForDataConf_C == TRUE) && (dwProtocolType == CSM_C_PTYPE_DC_USDT_ISO) && (wCount < CSM_C_ISO_TP_MAX_CONNECTIONS))
    #else
        if ((arIpcDataTp[wCount].fWaitForDataConf_C == TRUE) && (dwProtocolType == CSM_C_PTYPE_DC_USDT_ISO) && (wCount < CSM_C_ISO_TP_MAX_CONNECTIONS))
    #endif
        {
            #if (CSM_S_TRACE == CSM_C_F_ON)
            #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
            #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
            ETG_TRACE_USR3(("csm_vTpProcessReject - Reset Wait Flag for ISO-TP Data Conf"));
            #endif
            #endif
            #endif

       // we can stop the timer and reset marker
            CSM_lTimerStop(CSM_TIMER_NP1_TX + (((BYTE)wCount)*3));
            arIpcDataTp[wCount].fWaitForDataConf_C = FALSE;

      if ( csm_itp_arCallbackFkt[wCount].rCallback.pfvDataCon != NULL )
      {
        tCSM_ADDR_ISO_TP rAddrField;

        rAddrField.bLocalComp  = coarIpcCfgTp[ wCount].au8AddressField[0];
        rAddrField.bRemoteComp = coarIpcCfgTp[ wCount].au8AddressField[1];

        csm_rIsoTpConnectionAct[wCount].fTxChannelFree = TRUE;

        csm_itp_arCallbackFkt[wCount].rCallback.pfvDataCon(
                              #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                              csm_itp_arCallbackFkt[wCount].pvHandle,
                              #endif
                              dwProtocolType,
                              (const VOID*) &rAddrField,
                              CSM_C_CONF_FAULT); /* ISO TP uses CSM_C_CONF_OK/FAULT */


      }
        }
        else
        {
          // wrong protocol type or not waiting for a data conf
          // this can for example happen if 2 CType Requests are sent without sufficiant delay (not waiting for confirmation)
          // The first confirmation has cleared the waiting flag and the second provokes this case !!
          CSM_vAssert(bNoError == TRUE);
        }
      }
      else if (pu8Data[0] == IPC_SRVC_CODE_COMMUNICATION_REQ)
      {
        if (arIpcDataTp[wCount].fWaitForCommConf == TRUE)
        {
          CSM_vAssert(bNoError == TRUE);                                     // this should not occur
          // stop the timer and reset marker
          CSM_lTimerStop(CSM_TIMER_NP1_COM + (((BYTE)wCount)*3));
          arIpcDataTp[wCount].fWaitForCommConf = FALSE;
          // create negative confirmation
		  if (wCount < CSM_C_ISO_TP_MAX_CONNECTIONS)
		  {
			CSM_ITP_vPROXYTriggerCommunicationCon((BYTE)wCount, CSM_C_INACTIVE); // state is unknown todo:check addition
		  }
        }
        else
        {
          // we are NOT waiting for a ComReq confirmation
          CSM_vAssert(bNoError == TRUE);
        }
      } // ComReq rejected
    } // channel does not match looped channel
  } // loop over channels
}

/*******************************************************************************
 * function    csm_vTpProcessDataInd
 * \doxydocu
 * \brief      This function processes data indication msg from TP channel
 *
 * \param[in]  wTpChanNum   INC TP channel number
 * \param[in]  pu8Data      pointer to data
 * \param[in]  u32Length    length of data
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * 04.06.15 Borck
 * 09.10.17 Jayashree
 * - modified from MPDT to ISOTP callbacks
 * revised version
 *
 * \history_end
 ******************************************************************************/
static void csm_vTpProcessDataInd(WORD wTpChanNum, BYTE* pu8Data, DWORD u32Length)
{
  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR3(("csm_vTpProcessDataInd - called"));
  #endif
  #endif
  #endif

  BYTE bBus;
  DWORD dwProtocolType;
  WORD wApplID;

  BYTE* pbData = pu8Data+9;                  // strip CSM header from data (bus, protocoltype, addrfield, applId)
  WORD  wLength = (WORD)(u32Length-9);       // update len, Service Code and other INC data has already been stripped

  bBus = *(pu8Data+0);
  dwProtocolType = ((*(pu8Data+1))<<0)| ((*(pu8Data+2))<<8);  // bus is the first byte in buffer
  wApplID = ((*(pu8Data+7))<<0)| ((*(pu8Data+8))<<8);


  #if (CSM_S_J1939 == CSM_C_F_ON)
  if(dwProtocolType == CSM_C_PTYPE_J1939)
  {
    CSM_ADDR_J1939 rAddrField;

  // To maintain two structs here Node ID is considered as Node Address

    rAddrField.bSrcNodeID  = pu8Data[3];
    rAddrField.bTgtNodeID  = pu8Data[4];
    rAddrField.wPGN        = (WORD)((pu8Data[6] << 8) | ( pu8Data[5] ));

    ETG_TRACE_USR1(("csm_vTpProcessDataInd: rAddrField.wPGN = %d ", rAddrField.wPGN));

  if (rAddrField.wPGN < CSM_C_J1939_CFG_MAXPGN )
    {

                for(WORD wIdx = 0; wIdx < CSM_C_J1939_MAX_PGN_CALLBACKS; wIdx++)
                {
            ETG_TRACE_USR1(("csm_vJ1939ProcessDataInd -> idx = %d", wIdx));
                  if(csm_j1939_arPGNCallbackFkt[rAddrField.wPGN][wIdx].pfvDataInd != NULL)
                  {
           ETG_TRACE_USR1(("csm_vJ1939ProcessDataInd ->PGN = %d  idx = %d", rAddrField.wPGN, wIdx));
                     csm_j1939_arPGNCallbackFkt[rAddrField.wPGN][wIdx].pfvDataInd(
                     #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                                                  csm_j1939_arPGNCallbackFkt[rAddrField.wPGN][wIdx].pvHandle,
                     #endif
                                        dwProtocolType,
                                                  (const VOID*) &rAddrField,
                                                  pu8Data+9, u32Length-9);
                  }
                }

  }
  }
  // pass data to appropriate protocol handler
  else if ((dwProtocolType == CSM_C_PTYPE_DC_USDT_ISO) && (wTpChanNum < CSM_C_ISO_TP_MAX_CONNECTIONS) )
  #else
  if ((dwProtocolType == CSM_C_PTYPE_DC_USDT_ISO) && (wTpChanNum < CSM_C_ISO_TP_MAX_CONNECTIONS) )
  #endif
  {
  tCSM_ADDR_ISO_TP rAddrField;
  BYTE bLocalComp;
  BYTE bRemoteComp;

  bLocalComp = *(pu8Data+3);
  bRemoteComp = *(pu8Data+4);

  rAddrField.bLocalComp  = bLocalComp;
  rAddrField.bRemoteComp = bRemoteComp;

  #if (CSM_S_TRACE == CSM_C_F_ON)
    #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
    #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
    ETG_TRACE_USR1(("csm_vTpProcessDataInd: TpChnNum:0x%02x  Bus:%x  Protocol:%x  Local:%x  Remote:%x  AppId:0x%x  Data:%02x",
                    (tU8) wTpChanNum,
                    ETG_CENUM(tCSM_BUS_NUMBER,        (tU8)  bBus),
                    ETG_CENUM(tCSM_PROTOCOL_TYPE,     (tU32) dwProtocolType),
                    ETG_CENUM(tCSM_ADDRESS_FIELD_USDT_ISO,(tU8)  bLocalComp),
                    ETG_CENUM(tCSM_ADDRESS_FIELD_USDT_ISO,(tU8)  bRemoteComp),
                    ETG_CENUM(tCSM_APPL_ID,           (tU16) wApplID),
                    ETG_LIST_LEN(wLength),
                    ETG_LIST_PTR_T8((pu8Data+9))         // strip CSM header for DATA
                  ));
    #endif
    #endif
    #endif

    // forwared stipped data (removed INC header)
    if (csm_itp_arCallbackFkt[wTpChanNum].rCallback.pfvDataInd != NULL)
    {
       csm_itp_arCallbackFkt[wTpChanNum].rCallback.pfvDataInd(
                                                  #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                                                    csm_itp_arCallbackFkt[wTpChanNum].pvHandle,
                                                  #endif
                                                  dwProtocolType,
                                                  (const VOID*) &rAddrField,
                                                  pu8Data+9, u32Length-9);
    }
  }
  else
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
    #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
    ETG_TRACE_FATAL(("csm_vTpProcessDataInd - Protocol Type of INC NP message is INVALID"));
    #endif
    #endif
    #endif
  }
}

/*******************************************************************************
 * function    csm_vTpProcessDataCon
 * \doxydocu
 * \brief      This function processes data confirmation msg from TP channel
 *
 * \param[in]  wTpChanNum  INC TP channel number
 * \param[in]  pu8Data     pointer to data
 * \param[in]  u32Length   length of data
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * 04.06.15 Borck
 * revised version
 *
 * \history_end
 ******************************************************************************/
static void csm_vTpProcessDataCon(WORD wTpChanNum, BYTE* pu8Data, DWORD u32Length)
{
  CSM_vAssert(u32Length < CSM_C_BUFFER_SIZE_TP_RX);

  BYTE bBus;
  DWORD dwProtocolType;
  WORD wApplID;

  BYTE bTransferResult = pu8Data[9];                    // 00 is OK, 01 is NOK
  dwProtocolType = ((*(pu8Data+1))<<0)| ((*(pu8Data+2))<<8);

  // get more details
  bBus = *(pu8Data+0);
  wApplID = ((*(pu8Data+7))<<0)| ((*(pu8Data+8))<<8);

  // pass data to appropriate protocol handler
    #if (CSM_S_J1939 == CSM_C_F_ON)
  if(dwProtocolType == CSM_C_PTYPE_J1939)
    {
     CSM_ADDR_J1939 rAddrField;

       rAddrField.bSrcNodeID  = pu8Data[3];
       rAddrField.bTgtNodeID = pu8Data[4];
       rAddrField.wPGN =  ( pu8Data[6] << 8) | ( pu8Data[5] );

     arIpcDataTp[wTpChanNum].fWaitForDataConf_C = FALSE;

     #if (CSM_S_TRACE == CSM_C_F_ON)
       #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
       #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
     ETG_TRACE_USR1(("csm_vTpProcessDataCon(WORD wTpChanNum, BYTE* pu8Data, DWORD u32Length)rAddrField.wPGN = %d",rAddrField.wPGN));
       #endif
       #else
       {
          BYTE abData[256];
          BYTE bCopyData = u32Length > (256-12) ? (256-12) : (BYTE) u32Length;
          #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
          abData[ 0] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD( csm_j1939_arPGNCallbackFkt[rAddrField.wPGN][wTpChanNum].pvHandle));
          abData[ 1] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD( csm_j1939_arPGNCallbackFkt[rAddrField.wPGN][wTpChanNum].pvHandle));
          abData[ 2] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD( csm_j1939_arPGNCallbackFkt[rAddrField.wPGN][wTpChanNum].pvHandle));
          abData[ 3] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD( csm_j1939_arPGNCallbackFkt[rAddrField.wPGN][wTpChanNum].pvHandle));
          #endif /* CSM_S_USE_CALLBACK_WITH_HANDLE */
          abData[ 4] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD( dwProtocolType));
          abData[ 5] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD( dwProtocolType));
          abData[ 6] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD( dwProtocolType));
          abData[ 7] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD( dwProtocolType));
          abData[ 8] = rAddrField.bLocalComp;
          abData[ 9] = rAddrField.bRemoteComp;
          abData[10] = CSM_M_GET_HIBYTE( rAddrField.wPGN );
          abData[11] = CSM_M_GET_LOBYTE( rAddrField.wPGN );
          CSM_vMemCopy( &(abData[12]), pu8Data, bCopyData);
          CSM_M_TRACEOUT( CSM_C_TRACEOUT_LEVEL_INTERFACE,  CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_DATA_CON, abData , (WORD) (bCopyData+12));
        }
        #endif
        #endif /* CSM_S_TRACE */

    CSM_lTimerStop(CSM_TIMER_NP1_TX + (((BYTE)wTpChanNum)*3));

        if (rAddrField.wPGN < CSM_C_J1939_CFG_MAXPGN )
    {

                for(WORD wIdx = 0; wIdx < CSM_C_J1939_MAX_PGN_CALLBACKS; wIdx++)
                {
                  if(csm_j1939_arPGNCallbackFkt[rAddrField.wPGN][wIdx].pfvDataCon != NULL)
                  {
                     csm_j1939_arPGNCallbackFkt[rAddrField.wPGN][wIdx].pfvDataCon(
                     #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                                                  csm_j1939_arPGNCallbackFkt[rAddrField.wPGN][wIdx].pvHandle,
                     #endif
                                        dwProtocolType,
                                                  (const VOID*) &rAddrField,
                                                  pu8Data[10]);
                  }
                }

    }
    }
  else if ( (dwProtocolType == CSM_C_PTYPE_DC_USDT_ISO) && (wTpChanNum < CSM_C_ISO_TP_MAX_CONNECTIONS) )
    #else
  if ((dwProtocolType == CSM_C_PTYPE_DC_USDT_ISO) && (wTpChanNum < CSM_C_ISO_TP_MAX_CONNECTIONS) )
    #endif
  {
    BYTE bLocalComp;
        BYTE bRemoteComp;
    tCSM_ADDR_ISO_TP rAddrField;

    bLocalComp = *(pu8Data+3);
        bRemoteComp = *(pu8Data+4);

    rAddrField.bLocalComp  = bLocalComp;
        rAddrField.bRemoteComp = bRemoteComp;

    #if (CSM_S_TRACE == CSM_C_F_ON)
        #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
        #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
        ETG_TRACE_USR1(("csm_vTpProcessDataCon: TpChnNum:0x%02x  Bus:%x  Protocol:%x  Local:%x  Remote:%x  Result:%x  AppId:%x",
                        (tU8) wTpChanNum,
                        ETG_CENUM(tCSM_BUS_NUMBER,        (tU8)  bBus),
                        ETG_CENUM(tCSM_PROTOCOL_TYPE,     (tU32) dwProtocolType),
                        ETG_CENUM(tCSM_ADDRESS_FIELD_USDT_ISO,(tU8)  bLocalComp),
                        ETG_CENUM(tCSM_ADDRESS_FIELD_USDT_ISO,(tU8)  bRemoteComp),
                        ETG_CENUM(tCSM_INC_TX_RESULT,     (tU8)  bTransferResult),
                        ETG_CENUM(tCSM_APPL_ID,           (tU16) wApplID)
                      ));
        #endif
        #endif
        #endif

    CSM_lTimerStop(CSM_TIMER_NP1_TX + (((BYTE)wTpChanNum)*3));          // valid channels are counting from 0 to ...

    if ( csm_itp_arCallbackFkt[wTpChanNum].rCallback.pfvDataCon != NULL )
        {
        csm_rIsoTpConnectionAct[wTpChanNum].fTxChannelFree = TRUE;
        csm_itp_arCallbackFkt[wTpChanNum].rCallback.pfvDataCon(
                            #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                            csm_itp_arCallbackFkt[wTpChanNum].pvHandle,
                            #endif
                            dwProtocolType,
                            (const VOID*) &rAddrField,
                            bTransferResult); /* ISO TP uses CSM_C_CONF_OK/FAULT */

    }
  }
  else
  {
    CSM_vAssert(FALSE);
  }
}

/*******************************************************************************
 * function    csm_vTpProcessCommCon
 * \doxydocu
 * \brief      This function processes data confirmation msg from TP channel
 *
 * \param[in]  wTpChanNum
 *              internal TP channel number
 * \param[in]  pu8Data
 *              pointer to data
 * \param[in]  u32Length
 *              length of data
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 13.10.06 CM-DI/ESA3-Haffner
 * 04.06.15 Borck
 * 09.10.17 Jayashree
 * - modified from MPDT to ISOTP callbacks
 * revised version
 *
 * \history_end
 ******************************************************************************/
static void csm_vTpProcessCommCon(WORD wTpChanNum, BYTE* pu8Data, DWORD u32Length)
{
  CSM_vAssert(u32Length < CSM_C_BUFFER_SIZE_TP_RX);

  BYTE bBus;
  DWORD dwProtocolType;
  BYTE bLocalComp;
  BYTE bRemoteComp;
  WORD wApplID;
  BYTE bConnectState;

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR3(("csm_vTpProcessCommCon - called"));
  #endif
  #endif
  #endif

  arIpcDataTp[wTpChanNum].fWaitForCommConf = FALSE;   // reset wait marker
  CSM_lTimerStop(CSM_TIMER_NP1_COM + (((BYTE)wTpChanNum)*3)); // valid channels are counting from 0 to ...

  bBus = *(pu8Data+0);
  dwProtocolType = ((*(pu8Data+1))<<0)| ((*(pu8Data+2))<<8);
  bLocalComp = *(pu8Data+3);
  bRemoteComp = *(pu8Data+4);
  wApplID = ((*(pu8Data+7))<<0)| ((*(pu8Data+8))<<8);
  bConnectState = *(pu8Data+9);

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR1(("csm_vTpProcessCommCon: TpChnNum:0x%02x  Bus:%x  Protocol:%x  Local:%x  Remote:%x  State:%x  AppId:%x",
                  (tU8) wTpChanNum,
                  ETG_CENUM(tCSM_BUS_NUMBER,        (tU8)  bBus),
                  ETG_CENUM(tCSM_PROTOCOL_TYPE,     (tU32) dwProtocolType),
                  ETG_CENUM(tCSM_ADDRESS_FIELD_USDT_ISO,(tU8)  bLocalComp),
                  ETG_CENUM(tCSM_ADDRESS_FIELD_USDT_ISO,(tU8)  bRemoteComp),
                  ETG_CENUM(tCSM_CONNECTION_STATE,  (tU8)  bConnectState),
                  ETG_CENUM(tCSM_APPL_ID,           (tU16) wApplID)
                ));
  #endif
  #endif
  #endif

  // pass data to appropriate protocol handler
  if ((dwProtocolType == CSM_C_PTYPE_DC_USDT_ISO) && (wTpChanNum < CSM_C_ISO_TP_MAX_CONNECTIONS))
  {
    //TODO: os_SetEvent( TASK_CSM, CSM_EV_CHANGED); // Trigger CSM task. Es knnte noch ein Auftrag wartem
    CSM_ITP_vPROXYTriggerCommunicationCon( wTpChanNum, bConnectState);

    /*ADDED:ama8kor*/
    /*for lint: Info 715: prio2: Symbol 'u32Length' not referenced */
    CSM_M_PARAMETER_INTENTIONALLY_UNUSED(u32Length);
  }
  #if (CSM_S_J1939 == CSM_C_F_ON)
  else if (dwProtocolType == CSM_C_PTYPE_J1939)
  {
      CSM_J1939_vPROXYTriggerCommunicationCon( wTpChanNum, bConnectState);
  }
  #endif
}

/*******************************************************************************
 * function    csm_vTpProcessCommInd
 * \doxydocu
 * \brief      This function processes data confirmation msg from TP channel
 *
 * \param[in]  wTpChanNum
 *              internal TP channel number
 * \param[in]  pu8Data
 *              pointer to data
 * \param[in]  u32Length
 *              length of data
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 13.10.06 CM-DI/ESA3-Haffner
 * 04.06.15 Borck
 * 09.10.17 Jayashree
 * - modified from MPDT to ISOTP callbacks
 * revised version
 *
 * \history_end
 ******************************************************************************/
static void csm_vTpProcessCommInd(WORD wTpChanNum, BYTE* pu8Data, DWORD u32Length)
{
  CSM_vAssert(u32Length < CSM_C_BUFFER_SIZE_TP_RX);

  BYTE bBus;
  DWORD dwProtocolType;
  BYTE bLocalComp;
  BYTE bRemoteComp;
  WORD wApplID;
  BYTE bConnectState;

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR3(("csm_vTpProcessCommInd - called"));
  #endif
  #endif
  #endif

  bBus = *(pu8Data+0);
  dwProtocolType = ((*(pu8Data+1))<<0)| ((*(pu8Data+2))<<8);
  bLocalComp = *(pu8Data+3);
  bRemoteComp = *(pu8Data+4);
  wApplID = ((*(pu8Data+7))<<0)| ((*(pu8Data+8))<<8);
  bConnectState = *(pu8Data+9);

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR1(("csm_vTpProcessCommInd: TpChnNum:0x%02x  Bus:%x  Protocol:%x  Local:%x  Remote:%x  State:%x  AppId:%x",
                  (tU8) wTpChanNum,
                  ETG_CENUM(tCSM_BUS_NUMBER,        (tU8)  bBus),
                  ETG_CENUM(tCSM_PROTOCOL_TYPE,     (tU32) dwProtocolType),
                  ETG_CENUM(tCSM_ADDRESS_FIELD_USDT_ISO,(tU8)  bLocalComp),
                  ETG_CENUM(tCSM_ADDRESS_FIELD_USDT_ISO,(tU8)  bRemoteComp),
                  ETG_CENUM(tCSM_CONNECTION_STATE,  (tU8)  bConnectState),
                  ETG_CENUM(tCSM_APPL_ID,           (tU16) wApplID)
                ));
      #endif
    #endif
  #endif

  // pass data to appropriate protocol handler
  if ((dwProtocolType == CSM_C_PTYPE_DC_USDT_ISO) && (wTpChanNum < CSM_C_ISO_TP_MAX_CONNECTIONS))
  {

    //TODO: os_SetEvent( TASK_CSM, CSM_EV_CHANGED); // Trigger CSM task. Es knnte noch ein Auftrag wartem
    CSM_ITP_vPROXYTriggerCommunicationInd( wTpChanNum, bConnectState);

    /*ADDED:ama8kor*/
    /*for lint: Info 715: prio2: Symbol 'u32Length' not referenced */
    CSM_M_PARAMETER_INTENTIONALLY_UNUSED(u32Length);
  }
  #if (CSM_S_J1939 == CSM_C_F_ON)
  else if (dwProtocolType == CSM_C_PTYPE_J1939)
  {
      CSM_J1939_vPROXYTriggerCommunicationInd( wTpChanNum, bConnectState);
  }
  #endif
}

/*******************************************************************************
 * function    CSM_PROXY_vProcess_IPC_LOCAL_NetRequests
 * \doxydocu
 * \brief      Downstream callback: triggers/serializes sending of communication/data requests for the NET channel
 *
 * \param      void
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CSM_PROXY_vProcess_IPC_LOCAL_NetRequests(void)
{
  BYTE bLoop;
  BYTE i;
  WORD wApplId;
  WORD wAddressField;
  BYTE bAction;
  LONG lRetval;
  BYTE bBus;
  BYTE bNoError = FALSE;

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR3(("CSM_PROXY_vProcess_IPC_LOCAL_NetRequests - SsmGetState:%d, SsmRemoteNetActive:%d", csm_vSsmGetState(&rCsmSmRemoteNet), csm_tSsmRemoteNetActive));
  ETG_TRACE_USR3(("CSM_PROXY_vProcess_IPC_LOCAL_NetRequests - Protocol-Lock:%d", rIpcDataRemoteNet.bCurrentProtocolClient));
  #endif
  #endif
  #endif

  // loop over LCTRL configuration (note: also the OSEKI protocol type is handled on the NET channel too (with index 0) !!!)
  if (rIpcDataRemoteNet.bCurrentProtocolClient == CSM_NET_PROTOCOL_CLIENT_LCTRL)
  {
    // LCTRL handling
    for(bLoop = 0; bLoop < CSM_LCTR_C_MAX_USER; bLoop++)
    {
      if (csm_vSsmGetState(&rCsmSmRemoteNet) == csm_tSsmRemoteNetActive)
      {
        //************************************************
        // send pending communication requests
        if (rIpcDataRemoteNet.afCommTxReq[bLoop] == TRUE)        //#error: braucht man fr jede applid
        {
          // clear the request
          rIpcDataRemoteNet.afCommTxReq[bLoop] = FALSE;          //#error: braucht man fr jede applid

          // prepare the IPC buffer
          wApplId = rIpcDataRemoteNet.awApplId[bLoop];
          bAction = rIpcDataRemoteNet.au8CommStatusReq[bLoop];

          rIpcDataRemoteNet.au8BufferTx[0]  = IPC_SRVC_CODE_COMMUNICATION_REQ;                       // service code
          rIpcDataRemoteNet.au8BufferTx[1]  = CSM_LCTR_bGetUserAssignedBus(bLoop);                   // bus
          rIpcDataRemoteNet.au8BufferTx[2]  = CSM_M_GET_LOBYTE(corIpcCfgRemoteNet.dwProtocolType);   // protocol type
          rIpcDataRemoteNet.au8BufferTx[3]  = CSM_M_GET_HIBYTE(corIpcCfgRemoteNet.dwProtocolType);
          rIpcDataRemoteNet.au8BufferTx[4]  = corIpcCfgRemoteNet.au8AddressField[0];                 // address field
          rIpcDataRemoteNet.au8BufferTx[5]  = corIpcCfgRemoteNet.au8AddressField[1];
          rIpcDataRemoteNet.au8BufferTx[6]  = 0;
          rIpcDataRemoteNet.au8BufferTx[7]  = 0;
          rIpcDataRemoteNet.au8BufferTx[8]  = CSM_M_GET_LOBYTE(wApplId);                             // Appl ID
          rIpcDataRemoteNet.au8BufferTx[9]  = CSM_M_GET_HIBYTE(wApplId);
          rIpcDataRemoteNet.au8BufferTx[10] = bAction;                                               // Action

          // trigger TX
          lRetval = CSM_lIpcDataSend(rIpcDataRemoteNet.tyIpcHandle, rIpcDataRemoteNet.au8BufferTx, 11);

          if (lRetval < 0)
          {
            // currently (09.06.2015) this will not happen, since the previously called INC DataSend function always will return OK
            CSM_vAssert(bNoError == TRUE);
          }
        } // CommTXReq was pending
        //************************************************
        // send pending data requests
        else if (rIpcDataRemoteNet.afDataTxReq[bLoop] == TRUE)
        {
          // clear the request
          rIpcDataRemoteNet.afDataTxReq[bLoop] = FALSE;

          // everything should have been prepared in the DataReq Function before
          // so trigger TX directly
          lRetval = CSM_lIpcDataSend(rIpcDataRemoteNet.tyIpcHandle, rIpcDataRemoteNet.au8BufferTx, rIpcDataRemoteNet.wDataLength);

          if (lRetval < 0)
          {
            // currently (09.06.2015) this will not happen, since the previously called INC DataSend function always will return OK
            CSM_vAssert(bNoError == TRUE);
          }
        } // DataTXReq was pending
      }  // state is ACTIVE
    }  // LCTRL loop
  }  // LCTRL

  if (rIpcDataRemoteNet.bCurrentProtocolClient == CSM_NET_PROTOCOL_CLIENT_OSEKI)
  {
    // OSEKI handling
    if (csm_vSsmGetState(&rCsmSmRemoteNet) == csm_tSsmRemoteNetActive)
    {
      // send pending communication request (for OSEKI always done with UserNumber = 0)
      if (rIpcDataRemoteNet.afCommTxReq[0] == TRUE)
      {
        // clear the request
        rIpcDataRemoteNet.afCommTxReq[0] = FALSE;

        // prepare the IPC buffer
        wApplId = rIpcDataRemoteNet.awApplId[0];
        bAction = rIpcDataRemoteNet.au8CommStatusReq[0];
        bBus    = rIpcDataRemoteNet.bBus[0];

        rIpcDataRemoteNet.au8BufferTx[0]  = IPC_SRVC_CODE_COMMUNICATION_REQ;                       // service code
        rIpcDataRemoteNet.au8BufferTx[1]  = bBus;                                                  // bus
        rIpcDataRemoteNet.au8BufferTx[2]  = CSM_M_GET_LOBYTE(CSM_C_PTYPE_DC_OSEKI_NWM);            // protocol type
        rIpcDataRemoteNet.au8BufferTx[3]  = CSM_M_GET_HIBYTE(CSM_C_PTYPE_DC_OSEKI_NWM);
        rIpcDataRemoteNet.au8BufferTx[4]  = 0x01; //replaced CSM_C_ADR_FLD_RN_NM_CMD_INDICATOR with 0x01 as its not applicable for inf4cv;                     // address field
        rIpcDataRemoteNet.au8BufferTx[5]  = 0xFF;
        rIpcDataRemoteNet.au8BufferTx[6]  = 0;
        rIpcDataRemoteNet.au8BufferTx[7]  = 0;
        rIpcDataRemoteNet.au8BufferTx[8]  = CSM_M_GET_LOBYTE(wApplId);                             // Appl ID
        rIpcDataRemoteNet.au8BufferTx[9]  = CSM_M_GET_HIBYTE(wApplId);
        rIpcDataRemoteNet.au8BufferTx[10] = bAction;                                               // Action

        // trigger TX
        lRetval = CSM_lIpcDataSend(rIpcDataRemoteNet.tyIpcHandle, rIpcDataRemoteNet.au8BufferTx, 11);

        if (lRetval < 0)
        {
          // currently (09.06.2015) this will not happen, since the previously called INC DataSend function always will return OK
          CSM_vAssert(bNoError == TRUE);
        }
      } // TX was pending
    }  // state is ACTIVE
  } // OSEKI

  // is retriggered by ENTRY of csm_tSsmRemoteNetActive()
}

/*******************************************************************************
 * function    CSM_PROXY_vProcess_IPC_LOCAL_BrRequests
 * \doxydocu
 * \brief      This function triggers/serializes sending of communication requests
 *             and signal writes to BR signal channel
 *
 * \param      void
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CSM_PROXY_vProcess_IPC_LOCAL_BrRequests(void)
{
  BYTE bLoop;
  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR4(("CSM_PROXY_vProcess_IPC_LOCAL_BrRequests - csm_vSsmGetState:%d, csm_SsmRemoteNetActive:%d", csm_vSsmGetState(&rCsmSmRemoteNet), csm_tSsmRemoteNetActive));
  #endif
  #endif
  #endif

  if (csm_vSsmGetState(&rCsmSmRemoteNet) == csm_tSsmRemoteNetActive)
  {
    // process COMM Requests
    for(bLoop = 0; bLoop < CSM_CBR_MAX_USER; bLoop++)
    {
      // send communication requests
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR4(("CSM_PROXY_vProcess_IPC_LOCAL_BrRequests - Care for pending local COM_REQs"));
      #endif
      #endif
      #endif

      // if (rIpcDataBrSignalCom.afWaitForCommConf[bLoop] == FALSE)
      {
        if (rIpcDataBrSignalCom.afCommTxReq[bLoop] == TRUE)
        {
          // clear request marker
          rIpcDataBrSignalCom.afCommTxReq[bLoop] = FALSE;
          // delegate to local function
          csm_vBrTxCommReq(bLoop, rIpcDataBrSignalCom.awApplId[bLoop], rIpcDataBrSignalCom.au8CommStatusReq[bLoop]);
        }
      }
    }

    // process signal writes
    // if (rIpcDataBrSignalCom.afWaitForCommConf[bLoop] == FALSE)
    {
      WORD  u16Count;
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR4(("CSM_PROXY_vProcess_IPC_LOCAL_BrRequests - Care for pending local SIGNAL WRITES"));
      #endif
      #endif
      #endif

      // loop over all TX telegrams
      for (u16Count = 0; u16Count < CBR_TX_MAX_NUMBER; u16Count++)
      {
        if (rIpcDataBrSignalCom.afTelTxReq[u16Count] == TRUE)
        {
          #if (CSM_S_TRACE == CSM_C_F_ON)
          #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
          #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
          ETG_TRACE_USR4(("CSM_PROXY_vProcess_IPC_LOCAL_BrRequests - TelIndex:%d  TelTxReq:%d", u16Count, rIpcDataBrSignalCom.afTelTxReq[u16Count]));
          #endif
          #endif
          #endif

          // clear request marker
          rIpcDataBrSignalCom.afTelTxReq[u16Count] = FALSE;

          // delegate to local function
          // be careful here, since the cbr names originally were and still are implying arrays
          // but for a SLOT based config they are now pointers !!!
          csm_vBrTxSignalWriteReq(u16Count, rIpcDataBrSignalCom.au8TelTxType[u16Count], (csm_arCbrTxTelegram + u16Count)->pvBuffer, (csm_arCbrTxTelegram + u16Count)->bDlc);
          break;  // quit the loop
        }
      }
    }
  }
  // retriggered by ENTRY of csm_tSsmRemoteNetActive()
}

/*******************************************************************************
 * function    csm_vBrTxSignalWriteReq
 * \doxydocu
 * \brief      This function prepares IPC buffer and calls transmit function for
 *             signal writes to BR signal channel
 *
 * \param[in]  u32CbrTxNumber
 *              signal index (telegram and signal number)
 * \param[in]  u8TelegramTxType
 *              should be CSM_C_TX_AS_CONFIGURED for proxy actions
 * \param[in]  pu8TelDataBuffer
 *              pointer to raw signal data
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 18.12.13 Prhl
 * message format changed
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static void csm_vBrTxSignalWriteReq(DWORD u32CbrTxNumber, BYTE u8TelegramTxType, const BYTE* pu8TelDataBuffer, BYTE bDlc)
{
  BYTE bBus = CSM_M_GET_BUS_NO_FROM_SIGNAL_ID(u32CbrTxNumber);
  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR4(("csm_vBrTxSignalWriteReq - called"));         // Write tracing always user 4 to prevent from flooding
  #endif
  #endif
  #endif

  rIpcDataBrSignalCom.au8BufferTx[0] = IPC_SRVC_CODE_SIGNAL_WRITE_REQ;
  rIpcDataBrSignalCom.au8BufferTx[1] = bBus;
  rIpcDataBrSignalCom.au8BufferTx[2] = CSM_M_GET_LOBYTE( corIpcCfgBrSignal.dwProtocolType);
  rIpcDataBrSignalCom.au8BufferTx[3] = CSM_M_GET_HIBYTE( corIpcCfgBrSignal.dwProtocolType);
  rIpcDataBrSignalCom.au8BufferTx[4] = corIpcCfgBrSignal.au8AddressField[0];
  rIpcDataBrSignalCom.au8BufferTx[5] = corIpcCfgBrSignal.au8AddressField[1];
  rIpcDataBrSignalCom.au8BufferTx[6] = 0;
  rIpcDataBrSignalCom.au8BufferTx[7] = 0;

  rIpcDataBrSignalCom.au8BufferTx[8] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(u32CbrTxNumber));
  rIpcDataBrSignalCom.au8BufferTx[9] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(u32CbrTxNumber));
  rIpcDataBrSignalCom.au8BufferTx[10] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(u32CbrTxNumber));
  rIpcDataBrSignalCom.au8BufferTx[11] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(u32CbrTxNumber));

  rIpcDataBrSignalCom.au8BufferTx[12] = u8TelegramTxType;

  CSM_vMemCopy(&rIpcDataBrSignalCom.au8BufferTx[13], pu8TelDataBuffer, bDlc);

  CSM_lIpcDataSend(rIpcDataBrSignalCom.tyIpcHandle, rIpcDataBrSignalCom.au8BufferTx, 13 + bDlc);
}

/*******************************************************************************
 * function    csm_vBrTxCommReq
 * \doxydocu
 * \brief      This function prepares IPC buffer and calls transmit function for
 *             communication request to BR signal channel
 *
 * \param[in]  u16ApplId
 *              application id (see csm_stack_mp.h)
 * \param[in]  u8CommState
 *              requested component status
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static void csm_vBrTxCommReq(BYTE bUserNumber, WORD wApplId, BYTE bAction)
{
  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR3(("csm_vBrTxCommReq - called"));
  #endif
  #endif
  #endif

  rIpcDataBrSignalCom.au8BufferTx[0] = IPC_SRVC_CODE_COMMUNICATION_REQ;
  rIpcDataBrSignalCom.au8BufferTx[1] = CSM_CBR_bGetUserAssignedBus(bUserNumber);
  rIpcDataBrSignalCom.au8BufferTx[2] = CSM_M_GET_LOBYTE( corIpcCfgBrSignal.dwProtocolType);
  rIpcDataBrSignalCom.au8BufferTx[3] = CSM_M_GET_HIBYTE( corIpcCfgBrSignal.dwProtocolType);
  rIpcDataBrSignalCom.au8BufferTx[4] = corIpcCfgBrSignal.au8AddressField[0];
  rIpcDataBrSignalCom.au8BufferTx[5] = corIpcCfgBrSignal.au8AddressField[1];
  rIpcDataBrSignalCom.au8BufferTx[6] = bAction;
  rIpcDataBrSignalCom.au8BufferTx[7] = CSM_M_GET_LOBYTE(wApplId);
  rIpcDataBrSignalCom.au8BufferTx[8] = CSM_M_GET_HIBYTE(wApplId);

  CSM_lIpcDataSend(rIpcDataBrSignalCom.tyIpcHandle, rIpcDataBrSignalCom.au8BufferTx, 9);
}

/*******************************************************************************
 * function    csm_fCommReqIsPending
 * \doxydocu
 * \brief      Returns, if there is still an open communication request
 *
 * \param      void
 *
 * \return     tBool
 *              TRUE|FALSE
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static BOOL csm_fCommReqIsPending(void)
{
  WORD wCount;

  for (wCount = 0; wCount < CSM_CBR_MAX_USER; wCount++)
  {
    if ((rIpcDataBrSignalCom.au8CommStatusReq[ wCount] != CSM_C_DISCONNECT)  ||
        (rIpcDataBrSignalCom.au8CommStatusAct[ wCount] != CSM_C_DISCONNECT)
       )
    {
      return TRUE;
    }
  }

  for (wCount = 0; wCount < CSM_C_PROXY_TP_CHANNELS; wCount++)
  {
    if ((arIpcDataTp[wCount].u8CommStatusReq != CSM_C_DISCONNECT)  ||
        (arIpcDataTp[wCount].u8CommStatusAct != CSM_C_DISCONNECT)
       )
    {
      return TRUE;
    }
  }

  return FALSE;
}

/*******************************************************************************
 * function implementation (scope: CAN stack and operating system)
 *******************************************************************************/

/*******************************************************************************
 * function    CSM_PROXY_lInit
 * \doxydocu
 * \brief      Initializes CAN Proxy Server module.
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *
 * \param      BYTE bInitMode
 *             Allowed values (see csm_stack_i.h):
 *                - CSM_C_INIT_COLD
 *                - CSM_C_INIT_WARM
 *                - CSM_C_INIT_START
 *
 * \return     LONG
 *              Error value:
 *              - CSM_C_NO_ERROR
 *              - CBR_E_ERROR_STATE_WHILE_INIT
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \history_begin
 *
 * 06.04.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
LONG CSM_PROXY_lInit(const BYTE bInitMode)
{
  LONG lReturn = CSM_C_NO_ERROR;

  if (bInitMode == CSM_C_INIT_COLD)
  {
    // no action yet
  }
  else if (bInitMode == CSM_C_INIT_WARM)
  {
    // no action yet
  }
  else if (bInitMode == CSM_C_INIT_START)
  {
    // finally we start now by setting:
    // the referenced state machine (RemoteNet)
    // no super state
    // the initial state (INACTIVE)
    csm_vSsmInit(&rCsmSmRemoteNet, NULL, csm_tSsmRemoteNetInactive);
  }
  else
  {
    lReturn = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_UNKNOWN_STATE);
  }

  return lReturn;
}

/*******************************************************************************
 * function    CSM_PROXY_lExit
 * \doxydocu
 * \brief      Stops CAN Proxy server.
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *
 * \param[in]  BYTE bExitMode
 *             Allowed values (see csm_stack_i.h):
 *                - CSM_C_EXIT_STOP
 *                - CSM_C_EXIT_OFF
 *
 * \return     LONG
 *              Error value: CSM_C_NO_ERROR
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \history_begin
 *
 * 06.04.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
LONG CSM_PROXY_lExit(const BYTE bExitMode )
{
  LONG lReturn = CSM_C_NO_ERROR;

  // we currently support only the STOP mode
  if (bExitMode == CSM_C_EXIT_STOP)
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
    #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
    ETG_TRACE_USR1(("CSM_PROXY_lExit - STOP"));
    #endif
    #endif
    #endif

    // switch OFF the proxy indirectly -> force a processing of the new power state
    // SM will be triggered later on with POW_OFF event
    csm_fPowerCtrlOn = FALSE;
    csm_vTriggerCsmDownstreamTask(CSM_C_DOWN_REASON_CSM_IND_RADIOSTATUSCHANGED);
    //csm_vSsmPutEv(&rCsmSmRemoteNet, CSM_C_EV_POW_OFF);
  }

  return lReturn;
}

/*******************************************************************************
 * function    CSM_PROXY_vProcess_IPC_REMOTE_NetData
 * \doxydocu
 * \brief      This function reads incoming IPC data and calls process function
 *
 * \param      void
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CSM_PROXY_vProcess_IPC_REMOTE_NetData(void)
{
  LONG lRetval;
  DWORD u32RxLength;
  BYTE bNoError = FALSE;

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR3(("CSM_PROXY_vProcess_IPC_REMOTE_NetData - called"));
  #endif
  #endif
  #endif

  // get data from IPC buffer containg new data
  lRetval = CSM_lIpcDataGet(rIpcDataRemoteNet.tyIpcHandle, rIpcDataRemoteNet.au8BufferRx, CSM_C_BUFFER_SIZE_REMOTE_RX, &u32RxLength);

  if (lRetval == CSM_C_NO_ERROR)
  {
    // process data in local function
    csm_vNetProcess_IPC_RxData(rIpcDataRemoteNet.au8BufferRx, u32RxLength);
    // free INC buffer
    rIpcDataRemoteNet.fRxBufferUsed = FALSE;
  }
  else
  {
    CSM_vAssert(bNoError == TRUE);
  }
}

/*******************************************************************************
 * function    CSM_PROXY_vProcess_IPC_REMOTE_BrData
 * \doxydocu
 * \brief      This function reads incoming IPC data and calls process function
 *
 * \param      void
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CSM_PROXY_vProcess_IPC_REMOTE_BrData(void)
{
  LONG lRetval;
  DWORD u32RxLength;
  BYTE bNoError = FALSE;

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR4(("CSM_PROXY_vProcess_IPC_REMOTE_BrData - called"));
  #endif
  #endif
  #endif

  // get data from IPC buffer containg new data
  lRetval = CSM_lIpcDataGet(rIpcDataBrSignalCom.tyIpcHandle, rIpcDataBrSignalCom.au8BufferRx, CSM_C_BUFFER_SIZE_BR_RX, &u32RxLength);

  if (lRetval == CSM_C_NO_ERROR)
  {
    // process data in local function
    csm_vBrProcess_IPC_RxData(rIpcDataBrSignalCom.au8BufferRx, u32RxLength);
    // free INC buffer
    rIpcDataBrSignalCom.fRxBufferUsed = FALSE;
  }
  else
  {
    CSM_vAssert(bNoError == TRUE);
  }
}

/*******************************************************************************
 * function    CSM_PROXY_vProcess_IPC_REMOTE_TpData
 * \doxydocu
 * \brief      This function reads incoming IPC data and calls process function
 *
 * \param      void
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CSM_PROXY_vProcess_IPC_REMOTE_TpData(void)
{
  WORD wCount;
  DWORD wProtocollType;
  BYTE bNoError = FALSE;

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR3(("CSM_PROXY_vProcess_IPC_REMOTE_TpData - called"));
  #endif
  #endif
  #endif

  // IPC data received and dispatched by Upstream task, so handle now
  for (wCount = 0; wCount < CSM_C_PROXY_TP_CHANNELS; wCount++)
  {
    // loop over all NP (internal) channels with an access index
    if (arIpcDataTp[wCount].fDataAvailable)
    {
      LONG lRetval;
      DWORD u32RxLength;
    WORD wPGN;
      // get data from IPC buffer containing new data by using the INC LUN handle
      lRetval = CSM_lIpcDataGet(arIpcDataTp[wCount].tyIpcHandle, arIpcDataTp[wCount].au8BufferRx, CSM_C_BUFFER_SIZE_TP_RX, &u32RxLength);

      if (lRetval == CSM_C_NO_ERROR)
      {
        wProtocollType = (arIpcDataTp[wCount].au8BufferRx[3] << 8) | (arIpcDataTp[wCount].au8BufferRx[2]);

        #if (CSM_S_J1939 == CSM_C_F_ON)
        if(wProtocollType == CSM_C_PTYPE_J1939)
        {
          wPGN = (arIpcDataTp[wCount].au8BufferRx[7] << 8) | (arIpcDataTp[wCount].au8BufferRx[6]);

      ETG_TRACE_USR3(("CSM_PROXY_vProcess_IPC_REMOTE if (lRetval == CSM_C_NO_ERROR) wPGN = %d, count = %d", wPGN, wCount));

      if((wPGN < CSM_C_J1939_CFG_MAXPGN ) &&
		  (wCount == cnp_j1939_arCfgPGNTable[wPGN].bCCSNPChannel) /*&& (wCount >= CSM_C_ISO_TP_MAX_CONNECTIONS)*/)
      {
        csm_vTpProcess_IPC_RxData(wCount, arIpcDataTp[wCount].au8BufferRx, u32RxLength);
        arIpcDataTp[wCount].fRxBufferUsed = FALSE;
      }
        }
    else
      #endif
    {
		if (wCount < CSM_C_ISO_TP_MAX_CONNECTIONS)
		{
			// process data in local function
			csm_vTpProcess_IPC_RxData(wCount, arIpcDataTp[wCount].au8BufferRx, u32RxLength);
            // free INC buffer
            arIpcDataTp[wCount].fRxBufferUsed = FALSE;
		}
    }
      }
      else
      {
        CSM_vAssert(bNoError == TRUE);
      }
    }
  }
}

/*******************************************************************************
 * function    CSM_PROXY_vProcess_IPC_LOCAL_TpDataReq
 * \doxydocu
 * \brief      This function sends IPC data for TP
 *
 * \param      void
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 13.10.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CSM_PROXY_vProcess_IPC_LOCAL_TpDataReq(void)
{
  WORD wCount;

  // we come here if the downstream task is processing a TP data request
  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR3(("CSM_PROXY_vProcess_IPC_LOCAL_TpDataReq - called"));
  #endif
  #endif
  #endif

  // loop over all channels
  for (wCount = 0; wCount < CSM_C_PROXY_TP_CHANNELS; wCount++)
  {
    // find out for which channel a request is pending (consider 2 types of data requests)
    if (arIpcDataTp[wCount].fDataTxReq_C == TRUE)
    {

        // ensure that no waiting for conf is active
        if (arIpcDataTp[wCount].fWaitForIncConf == FALSE)
        {
            LONG lRetval;

            arIpcDataTp[wCount].fWaitForIncConf = TRUE;                   // now actually indicate that we have send a service request over INC
                                                                          // the channel related timer has been started already in #IF1 function
            arIpcDataTp[wCount].fDataTxReq_C = FALSE;                     // clear the request

            // send request to INC, remark: the data buffer must be used for data transfer only (not for com requests)
            lRetval = CSM_lIpcDataSend(arIpcDataTp[wCount].tyIpcHandle, arIpcDataTp[wCount].au8BufferTx_C, arIpcDataTp[wCount].u32TxLength_C);

            if (lRetval < 0)
            {
                // currently (09.06.2015) this will not happen, since the previously called INC DataSend function always will return OK
                // send a negative confirmation

        DWORD dwProtocolType = coarIpcCfgTp[ wCount].dwProtocolType;
        #if (CSM_S_J1939 == CSM_C_F_ON)
        if(dwProtocolType == CSM_C_PTYPE_J1939)
                {
          CSM_ADDR_J1939 rAddrField;

          rAddrField.bSrcNodeID  = arIpcDataTp[wCount].au8BufferTx_C[4];
          rAddrField.bTgtNodeID  = arIpcDataTp[wCount].au8BufferTx_C[5];
          rAddrField.wPGN = (arIpcDataTp[wCount].au8BufferTx_C[7] << 8 ) | ( arIpcDataTp[wCount].au8BufferTx_C[6] );//  Raghu : Need to figure out which byte of address field will be  ;
                    if (rAddrField.wPGN < CSM_C_J1939_CFG_MAXPGN)
					{
					for(WORD wIdx = 0; wIdx < CSM_C_J1939_MAX_PGN_CALLBACKS; wIdx++)
                    {
                      if(csm_j1939_arPGNCallbackFkt[rAddrField.wPGN][wIdx].pfvDataCon != NULL)
                      {
                         csm_j1939_arPGNCallbackFkt[rAddrField.wPGN][wIdx].pfvDataCon(
                         #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                                                      csm_j1939_arPGNCallbackFkt[rAddrField.wPGN][wIdx].pvHandle,
                         #endif
                                             dwProtocolType,
                                                      (const VOID*) &rAddrField,
                                                      CSM_C_CONF_FAULT);
                      }
                    }
					}
                }
        else if ((dwProtocolType == CSM_C_PTYPE_DC_USDT_ISO) && (wCount < CSM_C_ISO_TP_MAX_CONNECTIONS))
        #endif
        {
                    if ( (wCount < CSM_C_ISO_TP_MAX_CONNECTIONS) && (csm_itp_arCallbackFkt[wCount].rCallback.pfvDataCon != NULL) )
                    {
            tCSM_ADDR_ISO_TP rAddrField;

                      rAddrField.bLocalComp  = coarIpcCfgTp[ wCount].au8AddressField[0];
                      rAddrField.bRemoteComp = coarIpcCfgTp[ wCount].au8AddressField[1];

            csm_rIsoTpConnectionAct[wCount].fTxChannelFree = TRUE;

            csm_itp_arCallbackFkt[wCount].rCallback.pfvDataCon(
                                                                   #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                                                                   csm_itp_arCallbackFkt[wCount].pvHandle,
                                                                   #endif
                                                                   dwProtocolType,
                                                                   (const VOID*) &rAddrField,
                                                                   CSM_C_CONF_FAULT); /* ISO TP uses CSM_C_CONF_OK/FAULT */


                    }
        }
            }
        }
    else
    {
      // still wait for INC/IPC confirmation
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_FATAL(("CSM_PROXY_vProcess_IPC_LOCAL_TpDataReq: New INC service request pending for channel: 0x%x, but previous not yet confirmed", wCount));
      #endif
      #endif
      #endif
    }
  }
    else
    {
      // no TX data request pending for looped channel
    }
  } // loop over channels
}

/*******************************************************************************
 * function    CSM_PROXY_vProcess_IPC_LOCAL_TpCommReq
 * \doxydocu
 * \brief      This function sends IPC communication requests for TP
 *
 * \param      void
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 24.10.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CSM_PROXY_vProcess_IPC_LOCAL_TpCommReq(void)
{
  WORD wCount;

  // we come here if the downstream task is processing a TP communication request
  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR3(("CSM_PROXY_vProcess_IPC_LOCAL_TpCommReq - called"));
  #endif
  #endif
  #endif

  // loop over all channels
  for (wCount = 0; wCount < CSM_C_PROXY_TP_CHANNELS; wCount++)
  {
    // find out for which channel a request is pending and ensure that no other service request is in progress
    if (arIpcDataTp[wCount].fCommTxReq == TRUE)
    {
      if (arIpcDataTp[wCount].fWaitForIncConf == FALSE)    // as long as this is active, the timer is running as well !!
      {
        // pending found and not waiting for a confirmation
        LONG lRetval;
        BYTE au8BufferTx[11];

        arIpcDataTp[wCount].fWaitForIncConf = TRUE;          // now actually indicate that we have send a service request over INC
                                                             // the channel related timer has been started already in #IF1 function
        arIpcDataTp[wCount].fCommTxReq = FALSE;              // clear the request

        // use local buffer to not destroy the DataReq buffer with CommReq stuff
        au8BufferTx[0] = IPC_SRVC_CODE_COMMUNICATION_REQ;
        au8BufferTx[1] = coarIpcCfgTp[wCount].u8Bus;
        au8BufferTx[2] = coarIpcCfgTp[wCount].u8Subcontext;
        au8BufferTx[3] = coarIpcCfgTp[wCount].u8Context;
        au8BufferTx[4] = coarIpcCfgTp[wCount].au8NetCompAddr[0];
        au8BufferTx[5] = coarIpcCfgTp[wCount].au8NetCompAddr[1];
        au8BufferTx[6] = 0;
        au8BufferTx[7] = 0;
        au8BufferTx[8] = 0xFF;   // no Appl-ID required for RN AIVI
        au8BufferTx[9] = 0xFF;   // no Appl-ID required for NR AIVI
        au8BufferTx[10] = arIpcDataTp[wCount].u8CommStatusReq;

        // send request to INC
        lRetval = CSM_lIpcDataSend(arIpcDataTp[wCount].tyIpcHandle, au8BufferTx, 11);

        // evaluate INC result
        if (lRetval < 0)
        {
          // in case of problems with INC TX:
          // BKA2HI: m.A. sollte hier von einem INACTIVE ausgegangen werden und der folgende Aufruf erfolgen.
          // So knnte die Applikation erkennen, dass es ein Problem schon bei der INC gab.
          // es gibt also weder einen Connected noch Disconnected status
		  if (wCount < CSM_C_ISO_TP_MAX_CONNECTIONS)
		  {
			CSM_ITP_vPROXYTriggerCommunicationCon((BYTE) wCount, CSM_C_INACTIVE);
		  }
          // da der Proxy keine Antwort erhalten wird, ist es hier jetzt vielleicht schlau, den laufenden Timer anzuhalten und den
          // Status zurckzusetzen
          arIpcDataTp[wCount].fWaitForIncConf = FALSE;    // INC will never deliver anything
          arIpcDataTp[wCount].fWaitForCommConf = FALSE;   // CSM will also never deliver anything
          CSM_lTimerStop((CSM_TIMER_NP1_COM + (((BYTE)wCount)*3)));
        }
      }
      else
      {
        // still wait for INC/IPC confirmation
        #if (CSM_S_TRACE == CSM_C_F_ON)
        #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
        #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
        ETG_TRACE_FATAL(("CSM_PROXY_vProcess_IPC_LOCAL_TpCommReq - New INC service request pending for channel: 0x%x, but previous not yet confirmed", wCount));
        #endif
        #endif
        #endif
      }
    } // TX request pending
  } // loop over channels
}

/*******************************************************************************
 * function    CSM_PROXY_vProcessPowerState
 * \doxydocu
 * \brief      This function processes incoming data from power module
 *
 * \param      void
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CSM_PROXY_vProcessPowerState(void)
{
  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR3(("CSM_PROXY_vProcessPowerState - called"));
  #endif
  #endif
  #endif

  if (csm_fPowerCtrlOn == TRUE)
  {
    // Proxy is currently switched ON -> forward this to SM
    csm_vSsmPutEv(&rCsmSmRemoteNet, CSM_C_EV_POW_ON);
  }
  else
  {
    // Proxy has been switched OFF (probably by Exit function) -> forward this to state machine
    csm_vSsmPutEv(&rCsmSmRemoteNet, CSM_C_EV_POW_OFF);
  }
}

/*******************************************************************************
 * function    CSM_PROXY_lCommunicationReq
 * \doxydocu
 * \brief      Requests a transport protocol connection or CAN functionality
 *
 * \param[in]  bBus
 *              Bus number
 *
 * \param[in]  dwProtocolType
 *              Protocol type, see table 2 of CAn SW architecture.
 *
 * \param[in]  pvAddressField
 *              Protocol dependend address field.
 *
 * \param[in]  bAction
 *              CSM_C_CONNECTED
 *              CSM_C_DISCONNECTED
 *
 * \param[in]  wApplID
 *              Application id
 *
 * \return     LONG
 *              Error value: CSM_C_NO_ERROR
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
LONG CSM_PROXY_lCommunicationReq(BYTE bBus, DWORD dwProtocolType, const void* pvAddressField, BYTE bAction, WORD wApplID)
{
  LONG  lReturnValue = CSM_C_NO_ERROR;
  BYTE  bNoError = FALSE;

  // in case of a remote stack, we are directly called from the CSM IF1 (CSM_lCommunicationReq)
  // do a selective trace due to the available ETG enums
  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  if ((dwProtocolType == CSM_C_PTYPE_DC_USDT_ISO))
  {
    // MPDT related trace output
    ETG_TRACE_USR1(("CSM_PROXY_lCommunicationReq - Bus:%x  Protocol:%x  Local:%x  Remote:%x  Action:%x",
                    ETG_CENUM(tCSM_BUS_NUMBER,        (tU8)  bBus),
                    ETG_CENUM(tCSM_PROTOCOL_TYPE,     (tU32) dwProtocolType),
                    ETG_CENUM(tCSM_ADDRESS_FIELD_USDT_ISO,(tU8)  *((const BYTE*)pvAddressField + 0)),
                    ETG_CENUM(tCSM_ADDRESS_FIELD_USDT_ISO,(tU8)  *((const BYTE*)pvAddressField + 1)),
                    ETG_CENUM(tCSM_ACTION_REQ,        (tU8)  bAction)
                  ));
  }
  else
  {
    // other protocols
    ETG_TRACE_USR1(("CSM_PROXY_lCommunicationReq - Bus:%x  Protocol:%x  Local:0x%x  Remote:0x%x  Action:%x",
                    ETG_CENUM(tCSM_BUS_NUMBER,        (tU8)  bBus),
                    ETG_CENUM(tCSM_PROTOCOL_TYPE,     (tU32) dwProtocolType),
                    (tU8) *((const BYTE*)pvAddressField + 0),
                    (tU8) *((const BYTE*)pvAddressField + 1),
                    ETG_CENUM(tCSM_ACTION_REQ,        (tU8)  bAction)
                  ));
  }
  #endif
  #endif
  #endif

  // evaluate protocol
  switch(dwProtocolType)
  {
    //***************************************
    // LCTRL
    case CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL:
    {
      // LCTRL can be used by configured users, to indicate, that the bus is required for something
      // e.g. the SPM to keep the bus alive
      BYTE bUserNumber;
      // check if the INC channel is just in use by LCTRL or free
      if ((rIpcDataRemoteNet.bCurrentProtocolClient == CSM_NET_PROTOCOL_CLIENT_NONE) ||
          (rIpcDataRemoteNet.bCurrentProtocolClient == CSM_NET_PROTOCOL_CLIENT_LCTRL))
      {
        // if it was free before, now take it
        rIpcDataRemoteNet.bCurrentProtocolClient = CSM_NET_PROTOCOL_CLIENT_LCTRL;
        // LCTRL is now allowed to use the INC NET channel
        bUserNumber = CSM_LCTR_bGetUserNumber(bBus, dwProtocolType, pvAddressField, wApplID);
        if (bUserNumber < CSM_LCTR_C_MAX_USER)
        {
          if (rIpcDataRemoteNet.afWaitForCommConf[bUserNumber])
          {
            // hat da einer die Confirmation nicht abgewartet?
            lReturnValue = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_STACK_BUSY);
          }
          else
          {
            // ok, let's go
            rIpcDataRemoteNet.afWaitForCommConf[bUserNumber] = TRUE;
            rIpcDataRemoteNet.awApplId[bUserNumber] = wApplID;
            rIpcDataRemoteNet.au8CommStatusReq[bUserNumber] = bAction;
            rIpcDataRemoteNet.afCommTxReq[bUserNumber] = TRUE;
            // start the LCTRL user related timer (wait for confirmation) for this channel
            CSM_lTimerStart(CSM_TIMER_NET_LCTRL_USER_0 + bUserNumber, CSM_C_TIMEOUT_NET_LCTRL);
            // trigger INC TX
            csm_vTriggerCsmDownstreamTask(CSM_C_DOWN_REASON_IPC_LOCAL_NET_REQ_TX);
          }
        }
        else
        {
          CSM_vAssert(bNoError == TRUE);
          lReturnValue = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_INVALID_PARA);
        }
      } // NET channel is currently not used
      else
      {
        // NET channel is in use (by OSEKI)
        CSM_vAssert(bNoError == TRUE);
        lReturnValue = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_STACK_BUSY);
      }
    }
    break;

    //*************************************
    case CSM_C_PTYPE_DC_OSEKI_NWM:
    {
      // this interface is only to used by APPL XXX in order to send commands to the NWM (monitoring system)
      // we share the NET channel with LCTRL, and if we use it, we will currently act as user 0
      // die UserNr spielt fr OSEKI zwar keine Rolle, wir mssen aber einen index whlen, weil fr LCTRL festgelegt werden muss,
      // welche Array-Instanz die Verwaltung der INC bertragung bernimmt. OSEKI wird explizit immer den User 0 nehmen.
      BYTE bUserNumber = 0;

      // check if the INC channel is just in use by OSEKI or free
      if ((rIpcDataRemoteNet.bCurrentProtocolClient == CSM_NET_PROTOCOL_CLIENT_NONE) ||
          (rIpcDataRemoteNet.bCurrentProtocolClient == CSM_NET_PROTOCOL_CLIENT_OSEKI))
      {
        // if it was free before, now take it
        rIpcDataRemoteNet.bCurrentProtocolClient = CSM_NET_PROTOCOL_CLIENT_OSEKI;

        // so check if some parameters are valid
        // currently the only task for CommReq is to ReInit the monitoring on V850 site
        if ((0x01 == (tU8) *((const BYTE*)pvAddressField + 0)) && /* replaced CSM_C_ADR_FLD_RN_NM_CMD_INDICATOR with 0x01 as its not applicable for inf4cv*/
            (0xFF                              == (tU8) *((const BYTE*)pvAddressField + 1)) &&
            (bAction                           == CNM_OSEKI_CMD_REINIT_MONITOR) &&
            ((tU16) CSM_C_APPL_ID_MCAN_LOCAL_CTRL_SPM == wApplID)
           )
        {
          if (rIpcDataRemoteNet.afWaitForCommConf[bUserNumber])
          {
            // hat da einer die Confirmation nicht abgewartet?
            lReturnValue = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_STACK_BUSY);
          }
          else
          {
            // ok, let's go

            // do a local ReInitMonitors, to be consistent with the afterwards triggered remote ReInit !!
            // keep in mind, that applying for VCAN is sufficiant to reset all on V850 site (MCAN will be rejected by V850)
            // for iMX site we need to apply for 2 busses
            //CNM_OSEKI_vReInitMonitoringStates(CSM_C_CAN_BUS_VCAN);
            //CNM_OSEKI_vReInitMonitoringStates(CSM_C_CAN_BUS_MCAN);

            // trigger a remote ReInit
            rIpcDataRemoteNet.bBus[bUserNumber] = bBus;
            rIpcDataRemoteNet.afWaitForCommConf[bUserNumber] = TRUE;
            rIpcDataRemoteNet.awApplId[bUserNumber] = wApplID;
            rIpcDataRemoteNet.au8CommStatusReq[bUserNumber] = bAction;
            rIpcDataRemoteNet.afCommTxReq[bUserNumber] = TRUE;
            // start the timer (wait for confirmation) for this channel
            CSM_lTimerStart(CSM_TIMER_NET_OSEKI, CSM_C_TIMEOUT_NET_OSEKI);
            // trigger INC TX
            csm_vTriggerCsmDownstreamTask(CSM_C_DOWN_REASON_IPC_LOCAL_NET_REQ_TX);
          }
        }
        else
        {
          // any other intention for CommReq is INVALID
          CSM_vAssert(bNoError == TRUE);
          lReturnValue = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_INVALID_PARA);
        }
      } // NET channel is currently not used
      else
      {
        // NET channel is in use (by LCTRL)
        CSM_vAssert(bNoError == TRUE);
        lReturnValue = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_STACK_BUSY);
      }
    }
    break; // INC Net ctrl

    //*************************************
    // Broadcasts
    case CSM_C_PTYPE_USED_STD_BR_SIGNAL:
    {
      BYTE bUserNumber = CSM_CBR_bGetUserNumber(bBus, dwProtocolType, pvAddressField, wApplID);
      // check if in range
      if (bUserNumber < CSM_CBR_MAX_USER)
      {
        if (rIpcDataBrSignalCom.afWaitForCommConf[bUserNumber])
        {
          // hat da einer die Confirmation nicht abgewartet ?
          lReturnValue = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_STACK_BUSY);
        }
        else
        {
          // ok, let's go
          rIpcDataBrSignalCom.afWaitForCommConf[bUserNumber] = TRUE;
          rIpcDataBrSignalCom.awApplId[bUserNumber] = wApplID;
          rIpcDataBrSignalCom.au8CommStatusReq[bUserNumber] = bAction;
          rIpcDataBrSignalCom.afCommTxReq[bUserNumber] = TRUE;
          // start the timer (wait for confirmation)
          CSM_lTimerStart(CSM_TIMER_BRSIGNAL, CSM_C_TIMEOUT_BRSIGNAL);       // #error braucht man fr jede applid
          // trigger INC TX
          csm_vTriggerCsmDownstreamTask(CSM_C_DOWN_REASON_IPC_LOCAL_BR_REQ_TX);
        }
      }
      else
      {
        CSM_vAssert(bNoError == TRUE);
        lReturnValue = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_INVALID_PARA);
      }
    }
    break; // INC BR channel

    //*************************************
    // Transport Protocols

    default:
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR1(("CSM_PROXY_lCommunicationReq - CSM_E_INVALID_PROTOCOL_TYPE for INC channels"));
      #endif
      #endif
      #endif

      // assertion to inform users
      CSM_vAssert(bNoError == TRUE);
      lReturnValue = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_INVALID_PROTOCOL_TYPE);
    break;
  }

  return lReturnValue;
}

/*******************************************************************************
 * function    CSM_PROXY_lDataReq
 * \doxydocu
 * \brief      Requests a data transmisson
 *
 * \param[in]  dwProtocolType
 *              Protocol type, see table 2 of CAn SW architecture.
 *
 * \param[in]  pvAddressField
 *              Protocol dependend address field.
 *
 * \param[in]  pbData
 *              pointer to tx data
 *
 * \param[in]  wDataLength
 *              length of tx data
 *
 * \return     LONG
 *              Error value: CSM_C_NO_ERROR
 *
 * \lastreview
 *
 * \history_begin
 * 11.10.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
LONG CSM_PROXY_lDataReq(DWORD dwProtocolType, const void* pvAddressField, const BYTE* pbData, WORD wDataLength)
{
  WORD wCount;
  LONG  lReturnValue = CSM_C_NO_ERROR;
  BYTE  bNoError = FALSE;

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  // do a selective trace due to the available ETG enums
  if ((dwProtocolType == CSM_C_PTYPE_DC_USDT_ISO))
  {
    // MPDT related trace output
    ETG_TRACE_USR1(("CSM_PROXY_lDataReq - Protocol:%x  Local:%x  Remote:%x  Data:%02x",
                     ETG_CENUM(tCSM_PROTOCOL_TYPE,     (tU32) dwProtocolType),
                     ETG_CENUM(tCSM_ADDRESS_FIELD_USDT_ISO,(tU8)  *((const BYTE*)pvAddressField + 0)),
                     ETG_CENUM(tCSM_ADDRESS_FIELD_USDT_ISO,(tU8)  *((const BYTE*)pvAddressField + 1)),
                     ETG_LIST_LEN(wDataLength),
                     ETG_LIST_PTR_T8(pbData)
                  ));
  }
  else
  {
    // other protocols
    ETG_TRACE_USR1(("CSM_PROXY_lDataReq - Protocol:%x  Local:%x  Remote:%x  wPgn:%x  Data:%02x",
                    ETG_CENUM(tCSM_PROTOCOL_TYPE,     (tU32) dwProtocolType),
                    (tU8) *((const BYTE*)pvAddressField + 0),
                    (tU8) *((const BYTE*)pvAddressField + 1),
                    (tU8) *((const BYTE*)pvAddressField + 2),
                    ETG_LIST_LEN(wDataLength),
                    ETG_LIST_PTR_T8(pbData)
                  ));
  }
  #endif
  #endif
  #endif

  // evaluate the protocol
  switch(dwProtocolType)
  {
  #if (CSM_S_J1939 == CSM_C_F_ON)
    case CSM_C_PTYPE_J1939:
    {
        DWORD wPGN;
    BYTE bSrcNodeID;
    BYTE bTgtNodeID;


    bSrcNodeID = *((const BYTE*)pvAddressField);
    bTgtNodeID = *((const BYTE*)pvAddressField + 1);
    wPGN   = *((const BYTE*)pvAddressField + 2);


    if (wPGN < CSM_C_J1939_CFG_MAXPGN)
        {
      wCount = cnp_j1939_arCfgPGNTable[wPGN].bCCSNPChannel ;


            if (arIpcDataTp[wCount].fWaitForDataConf_C)
            {
                ETG_TRACE_FATAL(("CSM:TP DataReq, but still waiting for CONF!"));
        lReturnValue = CSM_M_MAKE_ERROR (CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_SRVC_UNAVAILABLE);
            }
      else
      {
            if (wDataLength <= (CSM_C_BUFFER_SIZE_TP_TX_C - 10))
            {
                arIpcDataTp[wCount].fWaitForDataConf_C = TRUE;
                arIpcDataTp[wCount].au8BufferTx_C[0] = IPC_SRVC_CODE_DATA_REQ;
                arIpcDataTp[wCount].au8BufferTx_C[1] = cnp_j1939_arCfgPGNTable[wPGN].bBus;
                arIpcDataTp[wCount].au8BufferTx_C[2] = CSM_M_GET_LOBYTE( dwProtocolType);
                arIpcDataTp[wCount].au8BufferTx_C[3] = CSM_M_GET_HIBYTE( dwProtocolType);
                arIpcDataTp[wCount].au8BufferTx_C[4] = bSrcNodeID;
                arIpcDataTp[wCount].au8BufferTx_C[5] = bTgtNodeID;  /* res */
                arIpcDataTp[wCount].au8BufferTx_C[6] = CSM_M_GET_LOBYTE( wPGN);
                arIpcDataTp[wCount].au8BufferTx_C[7] = CSM_M_GET_HIBYTE( wPGN);
                arIpcDataTp[wCount].au8BufferTx_C[8] = 0xff;//no ApplId
                arIpcDataTp[wCount].au8BufferTx_C[9] = 0xff;//no ApplId
                arIpcDataTp[wCount].u32TxLength_C = wDataLength + 10;

        CSM_vMemCopy (&arIpcDataTp[wCount].au8BufferTx_C[10], (const void*)pbData, wDataLength);

        arIpcDataTp[wCount].fDataTxReq_C = TRUE;

        CSM_lTimerStart((CSM_TIMER_NP1_TX + (((BYTE)wCount)*3)), CSM_C_TIMEOUT_TP);

        csm_vTriggerCsmDownstreamTask (CSM_C_DOWN_REASON_IPC_LOCAL_TP_DATA_REQ);


            }
            else
            {

                lReturnValue = CSM_M_MAKE_ERROR (CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_DATA_LENGTH_EXCEEDED);
        }
            }
        }
        else
        {
          CSM_vAssert(FALSE);
          lReturnValue = CSM_M_MAKE_ERROR (CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_INVALID_CHANNEL);
        }
    }
    break;
    #endif

    case CSM_C_PTYPE_DC_USDT_ISO:
    {
      BOOLEAN fChanFound = FALSE;
      WORD wCount;
      WORD wMatch = 0;

      // get internal proxy channel number by comparing address field against our configured channels in configuration struct
      // TYPE_C and TYPE_D protocols are both transmitted over the same INC channel !!!
      for (wCount = 0; wCount < CSM_C_PROXY_TP_CHANNELS; wCount++)
      {
        if ((*((const BYTE*)pvAddressField + 0) == coarIpcCfgTp[wCount].au8AddressField[0])  &&
            (*((const BYTE*)pvAddressField + 1) == coarIpcCfgTp[wCount].au8AddressField[1])
           )
        {
          fChanFound = TRUE;             // indicate match
          wMatch = wCount;               // to make LINT happy by not using the loop counter for later access
          break;                         // quit loop
        }
      }

      if (fChanFound)
      {
        // wenn wir fr einen Channel noch auf die COMMReq Conf warten, macht ein DATAReq "eigentlich" noch keinen Sinn.
        if (arIpcDataTp[wMatch].fWaitForCommConf == TRUE)
        {
          // hat da einer die COMMReq Confirmation nicht abgewartet?
          #if (CSM_S_TRACE == CSM_C_F_ON)
          #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
          #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
          ETG_TRACE_FATAL(("CSM_PROXY_lDataReq - Still waiting for COMMReq Confirmation - Inform Application Developer how to correctly use the Interface !!!"));
          #endif
          #endif
          #endif
        }

        // the new implementation requires seperated handling for both MPDT protocol types
        switch(dwProtocolType)
        {
          case CSM_C_PTYPE_DC_USDT_ISO:
            // channel is valid, check if previous TX is pending
            if (arIpcDataTp[wMatch].fWaitForDataConf_C)
            {
              // ein bisschen zu frh, wir warten "eigentlich" noch auf eine C-Type confirmation
              // lReturnValue = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_STACK_BUSY);
              #if (CSM_S_TRACE == CSM_C_F_ON)
              #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
              #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
              ETG_TRACE_FATAL(("CSM_PROXY_lDataReq - new C_TYPE request, but still waiting for CONF!"));
              #endif
              #endif
              #endif
            }
            //else   //das hier ist eine abgeschaltete Verriegelung -> wir senden auch neu, wenn die conf fr alt fehlt
            {
              if (wDataLength <= (CSM_C_BUFFER_SIZE_TP_TX_C - 10))      // we need some bytes for header data
              {
                arIpcDataTp[wMatch].fWaitForDataConf_C = TRUE;          // set wait-confirmation marker

                arIpcDataTp[wMatch].au8BufferTx_C[0] = IPC_SRVC_CODE_DATA_REQ;
                arIpcDataTp[wMatch].au8BufferTx_C[1] = coarIpcCfgTp[wMatch].u8Bus;
                arIpcDataTp[wMatch].au8BufferTx_C[2] = CSM_M_GET_LOBYTE(dwProtocolType);
                arIpcDataTp[wMatch].au8BufferTx_C[3] = CSM_M_GET_HIBYTE(dwProtocolType);
                arIpcDataTp[wMatch].au8BufferTx_C[4] = *((const BYTE*)pvAddressField + 0);  // local
                arIpcDataTp[wMatch].au8BufferTx_C[5] = *((const BYTE*)pvAddressField + 1);  // remote
                arIpcDataTp[wMatch].au8BufferTx_C[6] = 0;
                arIpcDataTp[wMatch].au8BufferTx_C[7] = 0;
                arIpcDataTp[wMatch].au8BufferTx_C[8] = 0xff;//no ApplId
                arIpcDataTp[wMatch].au8BufferTx_C[9] = 0xff;//no ApplId

                // add header bytes to len
                arIpcDataTp[wMatch].u32TxLength_C = wDataLength + 10;

                CSM_vMemCopy(&arIpcDataTp[wMatch].au8BufferTx_C[10], (const void*)pbData, wDataLength);

                // trigger TX via downstream task
                arIpcDataTp[wMatch].fDataTxReq_C = TRUE;

                // start channel related timer (wait for confirmation)
                if (arIpcDataTp[wMatch].fWaitForIncConf == TRUE)
                {
                  #if (CSM_S_TRACE == CSM_C_F_ON)
                  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
                  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
                  ETG_TRACE_FATAL(("CSM_PROXY_lDataReq - Potential ERROR detected !!! Still waiting for INC confirmation, but new timer started here now"));
                  #endif
                  #endif
                  #endif
                }
                CSM_lTimerStart((CSM_TIMER_NP1_TX + (((BYTE)wMatch)*3)), CSM_C_TIMEOUT_TP);
                // trigger INC TX
                csm_vTriggerCsmDownstreamTask(CSM_C_DOWN_REASON_IPC_LOCAL_TP_DATA_REQ);
              }
              else
              {
                lReturnValue = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_DATA_LENGTH_EXCEEDED);
              }
            }
            break;


        } // seperatly handled MPDT protocol types
      } // valid channel found in config
      else
      {
        // no valid TP channel found,  no assertion required
        #if (CSM_S_TRACE == CSM_C_F_ON)
        #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
        #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
        ETG_TRACE_USR3(("CSM_PROXY_lDataReq - CSM_E_INVALID_CHANNEL for INC TP channel"));
        #endif
        #endif
        #endif

        lReturnValue = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_INVALID_CHANNEL);
      }
    }
    break;
    case CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL:
    {
      BYTE bUserNumber;

      // an upper instance (cnp_LCTR) has already checked the address field and came to the conclusion,
      // that this DataReq has to be forwarded to the remote stack

      // check if the INC channel is just in use by LCTRL or free
      if ((rIpcDataRemoteNet.bCurrentProtocolClient == CSM_NET_PROTOCOL_CLIENT_NONE) ||
          (rIpcDataRemoteNet.bCurrentProtocolClient == CSM_NET_PROTOCOL_CLIENT_LCTRL))
      {
        // if it was free before, now take it
        rIpcDataRemoteNet.bCurrentProtocolClient = CSM_NET_PROTOCOL_CLIENT_LCTRL;
        // LCTRL is now allowed to use the INC NET channel
        // die UserNr spielt fr LCTRL zwar eine Rolle, wird i.d.R. aber durch die bergebene ApplId bestimmt.
        // wir haben hier im DataReq aber nur das AddrField als Parameter und nehmen deshalb explizit die 0.
        // Die Gegenstelle muss nun natrlich um diese Systematik wissen !!

        bUserNumber = 0;
        if (bUserNumber < CSM_LCTR_C_MAX_USER)
        {
          // wenn wir hier angekommen sind, kann es sein, das noch ein CommReq zB. vom SPM auf LCTRL luft.
          if (rIpcDataRemoteNet.afWaitForCommConf[bUserNumber])
          {
            // hat da einer die Confirmation nicht abgewartet? wir geben einen Fehler zurck.
            lReturnValue = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_STACK_BUSY);
          }
          // oder ein DataReq auf LCTRL luft noch ?
          else if (rIpcDataRemoteNet.afWaitForDataConf[bUserNumber])
          {
            //  wir geben einen Fehler zurck.
            lReturnValue = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_STACK_BUSY);
          }
          else
          {
            // ok, befllen wir nun die Verwaltungsstruktur und den Buffer
            // zuerst die bgebenen Parameter
            BYTE Hibyte = *((BYTE*)pvAddressField + 0);
            BYTE Lobyte = *((BYTE*)pvAddressField + 1);
            rIpcDataRemoteNet.awAddressField[bUserNumber] = CSM_M_MAKE_WORD(Hibyte, Lobyte);
            // dann die unbenutzten
            rIpcDataRemoteNet.awApplId[bUserNumber] = 0xFFFF;                  // not used for DataReq
            rIpcDataRemoteNet.bBus[bUserNumber] = 0x00;                        // not used for DataReq
            rIpcDataRemoteNet.au8CommStatusReq[bUserNumber] = 0xFF;            // not used for DataReq
            // prepare the IPC buffer
            if (CSM_C_BUFFER_SIZE_REMOTE_TX >= (12 + wDataLength))
            {
              rIpcDataRemoteNet.au8BufferTx[0]  = IPC_SRVC_CODE_DATA_REQ;                                // service code
              rIpcDataRemoteNet.au8BufferTx[1]  = rIpcDataRemoteNet.bBus[bUserNumber];                   // Bus                                                    // bus
              rIpcDataRemoteNet.au8BufferTx[2]  = CSM_M_GET_LOBYTE(CSM_C_PTYPE_PSGEN_CAN_LOCAL_CTRL);       // protocol type
              rIpcDataRemoteNet.au8BufferTx[3]  = CSM_M_GET_HIBYTE(CSM_C_PTYPE_PSGEN_CAN_LOCAL_CTRL);
              rIpcDataRemoteNet.au8BufferTx[4]  = Hibyte;                                                // address field
              rIpcDataRemoteNet.au8BufferTx[5]  = Lobyte;
              rIpcDataRemoteNet.au8BufferTx[6]  = 0;
              rIpcDataRemoteNet.au8BufferTx[7]  = 0;
              rIpcDataRemoteNet.au8BufferTx[8]  = 0xFF;                                                  // Appl ID
              rIpcDataRemoteNet.au8BufferTx[9]  = 0xFF;
              rIpcDataRemoteNet.au8BufferTx[10] = 0xFF;                                                  // Action
              rIpcDataRemoteNet.au8BufferTx[11] = (BYTE) (wDataLength);                                  // Data len
              // copy the payload data as well
              CSM_vMemCopy(&(rIpcDataRemoteNet.au8BufferTx[12]), (const void*)pbData, wDataLength);
              rIpcDataRemoteNet.wDataLength = 12 + wDataLength;                                          // nr of bytes to TX entirely
              rIpcDataRemoteNet.afWaitForDataConf[bUserNumber] = TRUE;           // set wait marker
              rIpcDataRemoteNet.afDataTxReq[bUserNumber] = TRUE;                 // set trigger
              // start the LCTRL user related timer (wait for confirmation) for this channel
              CSM_lTimerStart(CSM_TIMER_NET_LCTRL_USER_0 + bUserNumber, CSM_C_TIMEOUT_NET_LCTRL);
              // trigger INC TX
              csm_vTriggerCsmDownstreamTask(CSM_C_DOWN_REASON_IPC_LOCAL_NET_REQ_TX);
            }
            else
            {
              CSM_vAssert(bNoError == TRUE);
              lReturnValue = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_DATA_LENGTH_EXCEEDED);
            }
          }
        }
        else
        {
          CSM_vAssert(bNoError == TRUE);
          lReturnValue = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_INVALID_PARA);
        }
      } // NET channel is currently not used
      else
      {
        // NET channel is in use (by OSEKI)
        CSM_vAssert(bNoError == TRUE);
        lReturnValue = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_STACK_BUSY);
      }
    }
    break;

    // if protocol type is NOT supported, we will come here
    default:
      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR1(("CSM_PROXY_lDataReq - CSM_E_INVALID_PROTOCOL_TYPE for INC channels"));
      #endif
      #endif
      #endif

      CSM_vAssert(bNoError == TRUE);
      lReturnValue = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_INVALID_PROTOCOL_TYPE);
    break;
  }

  return lReturnValue;
}

/*******************************************************************************
 * function    CSM_PROXY_lCbrSendTelegram
 * \doxydocu
 * \brief      This function requests/triggers the sending of a broadcast signal
 *
 * \param[in]  bBus
 *              Bus number
 * \param[in]  bTelegramType
 *              should be CSM_C_TEL_TYPE_CBR_TX
 * \param[in]  bTelegramIndex
 *              see csm_stack_mp.h
 * \param[in]  u8TelegramTxType
 *              should be CSM_C_TX_AS_CONFIGURED for proxy actions
 *
 * \return     LONG
 *              Error value: CSM_C_NO_ERROR
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
LONG CSM_PROXY_lCbrSendTelegram(BYTE u8Bus, BYTE bTelegramType, BYTE bTelegramIndex, BYTE bTelegramTxType)
{
  // check if type is really BR TX and index is valid
  CSM_vAssert(bTelegramType == CSM_C_TEL_TYPE_CBR_TX);
  CSM_vAssert(bTelegramIndex < CBR_TX_MAX_NUMBER);

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  ETG_TRACE_USR4(("CSM_PROXY_lCbrSendTelegram - called"));
  ETG_TRACE_USR4(("CSM_PROXY_lCbrSendTelegram - Bus:0x%x  TelType:0x%x  Index:0x%x  TxType:0x%x",
                  (tU8)u8Bus,
                  (tU8)bTelegramType,
                  (tU8)bTelegramIndex,
                  (tU8)bTelegramTxType
                ));
  #endif
  #endif
  #endif

  if (bTelegramIndex < CBR_TX_MAX_NUMBER)
  {
    rIpcDataBrSignalCom.afTelTxReq[ bTelegramIndex] = TRUE;
    rIpcDataBrSignalCom.au8TelTxType[ bTelegramIndex] = bTelegramTxType;
    #if (CSM_S_TRACE == CSM_C_F_ON)
    #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
    #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
    ETG_TRACE_USR4(("CSM_PROXY_lCbrSendTelegram - Trigger DownstreamTask with CSM_C_DOWN_REASON_IPC_LOCAL_BR_REQ_TX"));
    #endif
    #endif
    #endif
    csm_vTriggerCsmDownstreamTask(CSM_C_DOWN_REASON_IPC_LOCAL_BR_REQ_TX);
    return CSM_C_NO_ERROR;
  }
  else
  {
    return CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_INVALID_PARA);
  }
}

/*******************************************************************************
 * function    CSM_PROXY_vProcess_IPC_TX_ResponseTimeout
 * \doxydocu
 * \brief      This function processes timeouts for transmitted orders to INC (waiting for a response from V850)
 *
 * \param      void
 *
 * \return     void
 *
 * \lastreview
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Haffner
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CSM_PROXY_vProcess_IPC_TX_ResponseTimeout(void)
{
  BYTE i;
  BYTE * pabTimeouts;
  BYTE bNoError = FALSE;

  // get pointer to array with timeouts
  pabTimeouts = (BYTE *) CSM_PROXY_pbGetTimeoutFlags();

  // loop over timers
  for (i=0; i<CSM_MAX_TIMER; i++)
  {
    // identify which timeout
    if (*(pabTimeouts+i) == 0x01)
    {
      // current timeout flag is set !
      // clear the current flag here now !!!
      *(pabTimeouts+i) = 0x00;

      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR3(("CSM_PROXY_vProcess_IPC_TX_ResponseTimeout - TimerID:%d", i));
      #endif
      #endif
      #endif

      switch(i)
      {
        // LCTRL timeout on NET channel !!!
        case CSM_TIMER_NET_LCTRL:
        case CSM_TIMER_NET_LCTRL_USER_0:
        case CSM_TIMER_NET_LCTRL_USER_1:
          csm_vNetProcess_IPC_TxResponseTimeout_LCTRL(i);
          break;

        // OSEKI timeout on  NET channel !!!
        case CSM_TIMER_NET_OSEKI:
          csm_vNetProcess_IPC_TxResponseTimeout_OSEKI(i);
          break;

        // BR timeout
        case CSM_TIMER_BRSIGNAL:
          csm_vBrProcess_IPC_TxResponseTimeout(i);
          break;

        // TP timeout
        case CSM_TIMER_NP1_COM:
        case CSM_TIMER_NP1_TX:
        case CSM_TIMER_NP1_RX:
        case CSM_TIMER_NP2_COM:
        case CSM_TIMER_NP2_TX:
        case CSM_TIMER_NP2_RX:
        case CSM_TIMER_NP3_COM:
        case CSM_TIMER_NP3_TX:
        case CSM_TIMER_NP3_RX:
        case CSM_TIMER_NP4_COM:
        case CSM_TIMER_NP4_TX:
        case CSM_TIMER_NP4_RX:
        case CSM_TIMER_NP5_COM:
        case CSM_TIMER_NP5_TX:
        case CSM_TIMER_NP5_RX:
        case CSM_TIMER_NP6_COM:
        case CSM_TIMER_NP6_TX:
        case CSM_TIMER_NP6_RX:
        case CSM_TIMER_NP7_COM:
        case CSM_TIMER_NP7_TX:
        case CSM_TIMER_NP7_RX:
        case CSM_TIMER_NP8_COM:
        case CSM_TIMER_NP8_TX:
        case CSM_TIMER_NP8_RX:
        #if (CSM_C_PROXY_TP_CHANNELS > 8)
        case CSM_TIMER_NP9_COM:
        case CSM_TIMER_NP9_TX:
        case CSM_TIMER_NP9_RX:
        #endif
        #if (CSM_C_PROXY_TP_CHANNELS > 9)
        case CSM_TIMER_NP10_COM:
        case CSM_TIMER_NP10_TX:
        case CSM_TIMER_NP10_RX:
        #endif
        #if (CSM_C_PROXY_TP_CHANNELS > 10)
        case CSM_TIMER_NP11_COM:
        case CSM_TIMER_NP11_TX:
        case CSM_TIMER_NP11_RX:
        #endif
          csm_vTpProcess_IPC_TxResponseTimeout(i);
          break;

        default:
          CSM_vAssert(bNoError == TRUE);
          #if (CSM_S_TRACE == CSM_C_F_ON)
          #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
          #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
          ETG_TRACE_USR3(("CSM_PROXY_vProcess_IPC_TX_ResponseTimeout - Unhandled TimerID:%d", i));
          #endif
          #endif
          #endif
          break;
      } // switch
    } // timeout flag set
  } // loop
}

#endif // CSM_S_DOXYGEN_HIDE_NON_API_FCT

/*******************************************************************************
 * function implementation (scope: CAN stack #1 API)
 ******************************************************************************/
// none

#undef CSM_PROXY_SELF
#undef CSM_CAN_STACK

/*******************************************************************************
 * \doxydocu
 * \file          csm_proxy.c
 * \brief         Proxy server functionality of the can stack manager.
 *
 *                Detailed description\n
 *                (if available).
 *
 * \see           CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \sw_component  CAN stack
 * \project       Pool module
 * \path          /di_can/modules_swa/csm/src/
 *
 * \ingroup       CSM
 * \ingroup       CSM_PROXY
 *
 * \authors       CM-DI/ESA3-Haffner
 *
 * COPYRIGHT      (c) 2013 Bosch Car Multimedia GmbH
 *
 * \history_begin
 *
 * 03.05.13  \main\g32\gm_gen3_proxy\6  Pistoor
 * - Made CAN USDT running
 *
 * 07.06.13  \main\vw_mib2_mqb_proxy\1 Prhl
 * - merge from \main\gm_gen3_proxy\6
 *
 * 07.06.13  \main\vw_mib2_mqb_proxy\2 Prhl
 * - quick hack for VW MIB
 *
 * 17.07.13  \main\vw_mib2_mqb_proxy\3 Prhl
 * - bugfix for second channel with CSM proxy.
 *
 * 20.07.13  \main\vw_mib2_mqb_proxy\3 Prhl
 * - use CSM_S_VW_BAP_TP instead of CSM_S_BAP_SUPPORT.
 *
 * 25.07.13  \main\vw_mib2_mqb_proxy\4 Prhl
 * - Trace added.
 *
 * 05.08.13  \main\vw_mib2_mqb_proxy\6  Prhl
 * - protection for IPC buffer added.
 *
 * 23.08.13  \main\vw_mib2_mqb_proxy\7  Prhl
 * - adaption for DataCon and CommunicationCon added.
 *
 * 12.09.13  \main\vw_mib2_mqb_proxy\8  Feldhaus
 * - implemented communication request and confirmation for Local Control protocol
 * - Communication Indication for CBR and ITP is handled by common CSM_lBusStateChange sequence now
 * - TP channel data indication/confirmation is indicated by CSM callback array now instead of direct function calls
 *
 * 24.09.13  \main\vw_mib2_mqb_proxy\9  Prhl
 * - CSM version check added to CCS_COMP_STATUS_CON/_REQ
 * - ETG trace cleanup
 *
 * 26.09.13  \main\vw_mib2_mqb_proxy\10 Prhl
 * - Function csm_vRemoteNetCtrlProcessCommunicationInd() added.
 *
 * 15.10.13  \main\vw_mib2_mqb_proxy\11 Prhl
 * - cleanup of ComminicationReq/Con/Ind handling.
 *
 * 18.10.13  \main\vw_mib2_mqb_proxy\12 Prhl
 * - cleanup of ComminicationReq/Con/Ind handling.
 * - traces modified.
 *
 * 18.10.13  \main\vw_mib2_mqb_proxy\13 Prhl
 * - cleanup of ComminicationReq/Con/Ind (CBR) handling.
 *
 * 21.10.13  \main\vw_mib2_mqb_proxy\14 Prhl
 * - csm_vTpProcessCommInd() added.
 *
 * 23.10.13  \main\vw_mib2_mqb_proxy\15 Prhl
 * - csm_vBrSignalProcessSignalInd() update to new message format.
 *
 * 23.10.13  \main\vw_mib2_mqb_proxy\16 Prhl
 * - protection against wrong parameter added to CSM_PROXY_lBapRawSendTelegram()
 *
 * 05.11.13  \main\vw_mib2_mqb_proxy\17 Prhl
 * - Bus error indication adapted.
 *
 * 07.11.13  \main\vw_mib2_mqb_proxy\18 Prhl
 * - lint warning removed.
 *
 * 18.12.13  \main\vw_mib2_mqb_proxy\19 Prhl
 * - message format for signal send modified.
 *
 * 13.02.14  \main\vw_mib2_mqb_proxy\20 Prhl
 * - message format for COMP_STATUS_REQ modified.
 *
 * 21.04.15  \main\rn_aivi_proxy\1 Borck
 * - new include file to have a prototype from csm_cbr.c
 * - external declarations for some configurations removed
 * - access to LCTRL and CSM_CBR config now via Get-Function
 * - a few cosmetics in function signatures for better readability
 *
 * 21.04.15  \main\rn_aivi_proxy\2 Borck
 * - ISO-TP stuff disabled -> to be removed later on
 *
 * 30.04.15  \main\rn_aivi_proxy\3 Prhl
 * - MPDT stuff added -> to be check later.
 *
 * 30.04.15  \main\rn_aivi_proxy\4 Prhl
 * - only remarks added.
 *
 * 15.05.15  \main\rn_aivi_proxy\5 Prhl
 * - IPC message style for TP changed. This change require a new CSM_PROXY. (CSM_C_VERSION++)
 *
 * 15.05.15  \main\rn_aivi_proxy\6 Prhl
 * - missing feature switch added.
 *
 * 18.05.15  \main\rn_aivi_proxy\7 Prhl
 * - IPC message style for NET changed.
 * - IPC message style for TP changed (DATA_REQ).
 *
 * 19.05.15  \main\rn_aivi_proxy\8 Prhl
 * - IPC message style for BR changed.
 * - DLC added to csm_vBrSignalTxSignalWrite()
 *
 * 29.05.15  \main\rn_aivi_proxy\9 Borck
 * - made this module compileable for the case that tracing is set to OFF !!!
 *
 * 01.06.15  \main\rn_aivi_proxy\10 Borck
 * - BAP related function calls featured with BAP feature switch
 *
 * 03.06.15  \main\rn_aivi_proxy\11 Borck
 * - ETG trace strings adapted
 *
 * 03.06.15  \main\rn_aivi_proxy\12 M.Prhl
 * - support for OSEKI added to csm_vRemoteNetCtrlProcessCommunicationInd().
 *
 * 05.06.15  \main\rn_aivi_proxy\13 Borck
 * - ETG trace output changes
 * - some corrections regarding the TP Data processing
 * - last BAP stuff removed
 * - assertions removed for invalid addressfield on TP channels (DataReq and CommReq)
 * - coding style
 *
 * 10.06.15  \main\rn_aivi_proxy\14 Borck
 * - some ETG traces extended by enum fields
 * - coding style
 * - INC REJECT code will stop a running timer for the TP channels
 * - Data Confirmation will also stop a running timer for the TP channel
 * - using the correct timer ID in case of comm request for a TP channel
 *
 * 19.06.15  \main\rn_aivi_proxy\15 Borck
 * - Timeout and Reject Handling for NP (TP) path reworked (requires a new V850 version)
 * - some interface functions renamed for better understanding of the data path.
 *
 * 09.07.15  \main\rn_aivi_proxy\16 Borck
 * - some functions renamed
 * - amount of timers extended, timeout handlers added
 * - REJECT handling for NET path reworked
 * - CommCon handling modified in dependence of the protocol type
 * - Timeout handling adapted
 * - MPDT IL handling adapted (Confirmation, DatReq path)
 * - CommReq handling modified
 *
 * 16.07.15  \main\rn_aivi_proxy\17 Borck
 * - Local ReInitMonitoringStates called additionally, before the same cmd is send to v850
 *
 * 23.07.15  \main\rn_aivi_proxy\18 Borck
 * - version history NR was wrong for a LONG time !!
 * - additional header files included for prototypes !!
 * - BR function names changed
 * - some extern declarations added
 * - lot of LINT issues fixed
 * - some coding style issues
 * - buffer overflow fixed for TPCommReq
 * - local OSEKI states ReInit now done for both busses
 *
 * 24.07.15  \main\rn_aivi_proxy\19 Borck
 * - Trace output for BusStateInd adapted so that Bus State is shown correctly
 *
 * 21.09.15  \main\rn_aivi_proxy\rn_aivi_proxy_generated\1 Borck
 * - external declaration added
 *
 * 30.09.15  \main\rn_aivi_proxy\rn_aivi_proxy_generated\2 Borck
 * - LINT issue fixed
 *
 * 12.10.15  \main\rn_aivi_proxy\rn_aivi_proxy_generated\3 Borck
 * - NET channel REJECT handling modified:
 *   - suppress cyclic COMP_STATUS_REQ if REJECT reason was WRONG_VERSION
 *
 * 04.11.15  \main\rn_aivi_proxy\rn_aivi_proxy_generated\4 Borck
 * - INC TX error handling for CommReq adapted
 * - timeout handling extended for new NP channels
 *
 * 08.01.16  \main\rn_aivi_proxy\rn_aivi_proxy_generated\5 Borck
 * - trace added for ticket resolve help
 *
 * 28.01.16  \main\rn_aivi_proxy\rn_aivi_proxy_generated\6 M.Prhl
 * - handling for IPC_SRVC_CODE_MULTI_SIGNAL_IND added.
 *
 * 01.02.16  \main\rn_aivi_proxy\rn_aivi_proxy_generated\7 A.Borck
 * - BR Reject Handler modified according to changed message structure.
 *
 * 16.02.16  \main\rn_aivi_proxy\rn_aivi_proxy_generated\8 A.Borck
 * - reviewed, lots of comments added, no functional changes
 * - obsolete #defined (service codes) replaced by existing ones
 *
 * 21.06.16  \main\rn_aivi_proxy\rn_aivi_proxy_generated\9 A.Borck
 * - trace output changed slightly
 *
 * 06.09.16  \main\rn_aivi_proxy\rn_aivi_proxy_generated\10 A.Borck
 * - function added to check the INC comm state 'ACTIVE'
 * - dedicated trace class introduced for this module
 *
 * 07.09.16  \main\rn_aivi_proxy\rn_aivi_proxy_generated\11 A.Borck
 * - Trace output slightly changed
 *
 * 19.09.16  \main\rn_aivi_proxy\rn_aivi_proxy_generated\12 A.Borck
 * - MPDT channel now supporting C-Type and D-Type data
 *   - 3 Timers per channel
 *   - Reject handling modified
 *
 * 07.11.16  \main\rn_aivi_proxy\rn_aivi_proxy_generated\13 A.Borck
 * - NET path extended by DataReq and DataConf for TTFis command routing to v850 via LCTRL
 * - some cleanup
 *
 * 05.05.17  \main\rn_aivi_proxy\rn_aivi_proxy_generated\14 A.Borck
 * - prepared for MPDT/ISO-TP support, but ISO-TP support not coded yet
 *
 * 15.05.17  \main\rn_aivi_proxy\rn_aivi_proxy_generated\15 A.Borck
 * - added include file for ISO-TP interface prototypes
 * - declared new interface function for DataInd from here (Proxy), since CNP_ITP is not needed !!
 * - introduced a limit value of NP channels to be able to separate between MPDT and ISO-TP channels on INC (for AIVI this is NET_NP_8_INDEX)
 * - implemented timeout handling for ISO-TP: DataReq
 * - implemented REJECT handling for ISO-TP: DataReq
 * - implemented DataInd processing for ISO-TP protocol type
 * - implemented DataCon processing for ISO-TP protocol type
 * - implemented DataReq processing
 *
 * 01.06.17  \main\rn_aivi_proxy\rn_aivi_proxy_generated\16 A.Borck
 * - removed prototypes of CSM_ITP module (they have been added to csm_c.h)
 * - DataCon and DataInd callbacks of CSM_ITP containing CNP replaced by those containing PROXY to avoid confusion
 * - workaround for INC ports adapted
 * - LINT issues fixed
 * - fix in: Net ProcessReject for component status request (unreachable code fixed)
 *
 * 14.06.17  \main\rn_aivi_proxy\rn_aivi_proxy_generated\17 A.Borck
 * - TX timeout handling for timer x now depending on the number of configured NetTp channels (for x > 8)
 *
 * 23.06.17  \main\rn_aivi_proxy\rn_aivi_proxy_generated\18 A.Borck
 * - changed several trace output levels from USER4 to USER3 to be able to supress annoying Signal Reads/Writes by using USER3 as standard trace class
 *
 * 10.08.17  \main\rn_aivi_proxy\rn_aivi_proxy_generated\19 A.Borck
 * - final optimisations regarding TRACE output -> USER3 should be used as standard, USER4 is with CBR output
 *
 * 24.11.17  main\g3inf4cv_csm_proxy_in\1  Jayashree
 * - Modified as per INF4CV, initial version
 *
 * \history_end
 ****** END OF FILE **********************************************************/
