/* Description and history of csm_itp.c at the end of the file.               */
/** File name for preprozessor commands */
#define CSM_J1939_SELF
/** csm_itp.c is a file according V-team CAN-SW architecture implementation rules */
#define CSM_CAN_STACK


/*******************************************************************************
 * includes
 ******************************************************************************/
#include "csm_stack_i.h"
#include "csm_i.h"
#include "csm_stack_m.h"
#include "csm_m.h"
#include "cnp_i.h"

#include "csm_j1939_m.h"     /* ISO-TP specific functions/defines of the can stack
                              * manager */
#if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
#include "cnp_j1939_i.h"
#endif

#include "csm_c.h"

#if (CSM_S_J1939 == CSM_C_F_ON)
/*************** ETG Tracing ******/
#if (CSM_S_TRACE == CSM_C_F_ON)

 #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #define ETRACE_S_IMPORT_INTERFACE_GENERIC
  #include "etrace_if.h"

  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
    #include "csm_etg_trace.h"
    #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FD_CSM
    /* #define ETG_INLINE */
    #include "trcGenProj/Header/csm_j1939.c.trc.h"
  #endif
  #endif
 #endif
#endif /* (CSM_S_TRACE == CSM_C_F_ON) */

/* Feature switch is used only to control doxygen documentation */
#if (CSM_S_DOXYGEN_HIDE_NON_API_FCT == CSM_C_F_OFF)

/*******************************************************************************
 * defines and macros (scope: csm module local)
 ******************************************************************************/
#if (CSM_S_REMOTE_STACK == CSM_C_F_ON)
extern tCSM_J1939_ADDR cnp_j1939_arActAddrTable[CSM_C_MAX_CAN_BUS][CSM_C_J1939_CFG_MAXNODES];
#endif
/*******************************************************************************
 * typedefs (scope: csm module local)
 ******************************************************************************/
typedef struct
{
  BYTE bReqAction;            /**< Array to store the bAction parameter
                               *   from CSM_ITP_lCommunicationReq() from every user.
                               *   Possible values: CSM_C_STACK_DOWN, CSM_C_STACK_UP */
  BYTE bActAction;            /**< Array to store the bAction parameter
                               *   from CSM_ITP_lCommunicationReq() from every user.
                               *   Possible values: CSM_C_STACK_DOWN, CSM_C_STACK_UP */
  /*  BYTE  bBusState;         *< State of the bus. Possible values: CSM_C_STATE_BUS_DOWN, CSM_C_STATE_BUS_LISTEN, CSM_C_STATE_BUS_STARTUP, CSM_C_STATE_BUS_SHUTDOWN, CSM_C_STATE_BUS_UP */
  BOOLEAN bCommunicationReq;  /**< Trigger array Flag for Communication request,
                               *    if TRUE communication request is pending.
                               *    Possible value TRUE/FALSE */
  BOOLEAN bCommunicationCon;  /**< Trigger array Flag for Communication confirmation,
                               *   if TRUE communication confirmation is pending.
                               *   Possible value TRUE/FALSE */
  BOOLEAN bCommunicationInd;  /**< Trigger array Flag for Communication indication,
                               *   if TRUE communication indication is penning.
                               *   Possible value TRUE/FALSE */
} tCSM_J1939_COMMUNICATION_STATE;

/*******************************************************************************
 * variable definition (scope: csm module local)
 ******************************************************************************/
static tCSM_J1939_COMMUNICATION_STATE csm_j1939_arCommunicationState[CSM_C_J1939_MAX_USER];

/*******************************************************************************
 * variable definition (scope: CAN stack)
 ******************************************************************************/
/* --none-- or in csm_itp_i(p).h */

/*******************************************************************************
 * Extern
 ******************************************************************************/
#if (CSM_S_REMOTE_STACK == CSM_C_F_ON)
extern LONG CSM_PROXY_lDataReq(DWORD, const void*, const BYTE*, WORD);
#endif

/*******************************************************************************
 * function prototypes (scope: csm module local)
 ******************************************************************************/
/* --none-- or in csm_itp_m(p).h */

/*******************************************************************************
 * function implementation (scope: file local = static)
 ******************************************************************************/
static BYTE csm_j1939_bGetApplNumber(BYTE bBus, DWORD dwProtocolType,
                                     const void * pvAddressField, WORD wApplId)
{
  BYTE bApplNumber;

  for( bApplNumber = 0; bApplNumber < CSM_C_J1939_MAX_USER; bApplNumber++)
  {
    if( ( bBus == csm_j1939_arApplCfg[bApplNumber].bBus) &&
      ( dwProtocolType == csm_j1939_arApplCfg[ bApplNumber].dwProtocolType) &&
      ( wApplId == csm_j1939_arApplCfg[bApplNumber].wApplId) )
    {
      return( bApplNumber);
    }
  }
  return( bApplNumber);
}

/*******************************************************************************
 * function implementation (scope: CSM module local)
 ******************************************************************************/

/*******************************************************************************
 * function    CSM_J1939_lInit
 * \doxydocu
 * \brief      Initializes CSM J1939 Server module.
 *
 *             Initializes CSM J1939 Server module.
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *
 * \param      bInitMode
 *             Allowed values:
 *                CSM_C_INIT_COLD, CSM_C_INIT_WARM, CSM_C_INIT_START (see csm_stack_i.h)
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR, CBR_E_ERROR_STATE_WHILE_INIT
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 *
 * \history_end
 ******************************************************************************/
LONG CSM_J1939_lInit(BYTE bInitMode)
{
  LONG return_value = CSM_C_NO_ERROR;
  WORD wIdx;
  #if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
  WORD wIdx2;
  #endif

  if(bInitMode == CSM_C_INIT_COLD)
  {
    for( wIdx = 0; wIdx < CSM_C_J1939_MAX_USER; wIdx++)
    {
      csm_j1939_arCommunicationState[wIdx].bReqAction = CSM_C_STACK_DOWN;
      csm_j1939_arCommunicationState[wIdx].bActAction = CSM_C_STACK_DOWN;
      csm_j1939_arCommunicationState[wIdx].bCommunicationReq = FALSE;
      csm_j1939_arCommunicationState[wIdx].bCommunicationCon = FALSE;
      csm_j1939_arCommunicationState[wIdx].bCommunicationInd = FALSE;

      #if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
      csm_j1939_arApplCallbackFkt[wIdx].pfvCommunicationCon = NULL;
      csm_j1939_arApplCallbackFkt[wIdx].pfvCommunicationInd = NULL;
      #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
      csm_j1939_arApplCallbackFkt[wIdx].pvHandle = NULL;
      #endif /* CSM_S_USE_CALLBACK_WITH_HANDLE */
      #if (CSM_S_USE_TKSE == CSM_C_F_ON)
      csm_j1939_arApplCallbackFkt[wIdx].rTaskInfo = NULL;
      #endif /*CSM_S_USE_TKSE == CSM_C_F_ON */
      #endif
    }

    #if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
    for( wIdx = 0; wIdx < CSM_C_J1939_CFG_MAXPGN; wIdx++)
    {
      for( wIdx2 = 0; wIdx2 < CSM_C_J1939_MAX_PGN_CALLBACKS; wIdx2++)
      {
         csm_j1939_arPGNCallbackFkt[wIdx][wIdx2].pfvDataCon = NULL;
         csm_j1939_arPGNCallbackFkt[wIdx][wIdx2].pfvDataInd = NULL;
         csm_j1939_arPGNCallbackFkt[wIdx][wIdx2].pfvDataIndFF = NULL;
     #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
         csm_j1939_arPGNCallbackFkt[wIdx][wIdx2].pfvDataErrorInd = NULL;
     #endif
     #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
         csm_j1939_arPGNCallbackFkt[wIdx][wIdx2].pvHandle = NULL;
     #endif /* CSM_S_USE_CALLBACK_WITH_HANDLE */
     #if (CSM_S_USE_TKSE == CSM_C_F_ON)
         csm_j1939_arPGNCallbackFkt[wIdx][wIdx2].rTaskInfo = 0;
     #endif /*CSM_S_USE_TKSE == CSM_C_F_ON */
      }
    }
    #endif
  }
  else if (bInitMode == CSM_C_INIT_WARM)
  {
  }
  else if (bInitMode == CSM_C_INIT_START)
  {
    for(wIdx = 0; wIdx < CSM_C_J1939_CFG_MAXPGN; wIdx++)
    {
      csm_j1939_abPGNIndFlags[wIdx].fDataCon = CSM_C_CONF_NONE;
      csm_j1939_abPGNIndFlags[wIdx].fRecvInd = FALSE;
      csm_j1939_abPGNIndFlags[wIdx].fRecvIndFF = FALSE;
      csm_j1939_abPGNIndFlags[wIdx].fSendReq = FALSE;
      csm_j1939_abPGNIndFlags[wIdx].fTxTimeoutInd = FALSE;
      csm_j1939_abPGNIndFlags[wIdx].fRxTimeoutInd = FALSE;
      csm_j1939_abPGNIndFlags[wIdx].fErrorInd = FALSE;
    }
  }
  else
  {
    return CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_UNKNOWN_STATE);
  }

  return( return_value);
}

/*******************************************************************************
 * function    CSM_J1939_lExit
 * \doxydocu
 * \brief      Stops CSM CAN J1939 server.
 *
 *             Stops CAN J1939 server.
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3

 * \param[in]  bExitMode
 *             Allowed values (see csm_stack_i.h):
 *                - CSM_C_EXIT_STOP
 *                - CSM_C_EXIT_OFF
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 *
 * \history_end
 ******************************************************************************/
LONG CSM_J1939_lExit(BYTE bExitMode)
{
  LONG return_value = CSM_C_NO_ERROR;

  switch (bExitMode)
  {
    case CSM_C_EXIT_STOP:
    {
    }
    break;

    case CSM_C_EXIT_OFF:
    {
    }
    break;

    default:
    {
      return_value = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_UNKNOWN_STATE);
    }
    break;
  }

  return( return_value);
}

#if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
/*******************************************************************************
 * function    CSM_J1939_vCNPDataInd()
 * \doxydocu
 * \brief      J1939 function to indicate the application that a
 *             message was received.
 *
 * \param[in] bIsoTpChannel
 *              CAN stack internal channel Index.
 *
 * \param[in]  pbData
 *              Pointer to data which have been received
 *
 * \param[in]  wDataLength
 *              Number of data bytes
 *
 * \return     void
 *
 * \access     CAN task
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 *
 * \history_end
 ******************************************************************************/
VOID CSM_J1939_vCNPDataInd(WORD wPGN)
{
  csm_j1939_abPGNIndFlags[wPGN].fRecvInd = TRUE;
  csm_vTriggerCsmUpstreamTask(CSM_C_IND_REASON_J1939_DATA_IND);
}

/*******************************************************************************
 * function    CSM_J1939_vCNPDataCon
 * \doxydocu
 * \brief      J1939 confirmation function to confirm the data request
 *             of the application.
 *
 * \param[in] bIsoTpChannel
 *              CAN stack internal channel Index.
 *
 * \param[in]  bTransferResult
 *              Result of requested data transfer.
 *
 * \return     void
 *
 * \access     CAN task
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 *
 * \history_end
 ******************************************************************************/
VOID CSM_J1939_vCNPDataCon(WORD wPGN, BYTE bTransferResult)
{
  csm_j1939_abPGNIndFlags[wPGN].fDataCon = bTransferResult;
  csm_vTriggerCsmUpstreamTask(CSM_C_IND_REASON_J1939_DATA_CON);
}

/*******************************************************************************
 * function    CSM_J1939_vCNPDataErrorInd
 * \doxydocu
 * \brief      J1939 indication function to indicate the reception of a First Frame
 *             to the application.
 *
 * \param[in] bIsoTpChannel
 *              CAN stack internal channel Index.
 *
 * \return     void
 *
 * \access     CAN task
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 *
 * \history_end
 ******************************************************************************/
VOID CSM_J1939_vCNPDataErrorInd(WORD wPGN)
{
  csm_j1939_abPGNIndFlags[wPGN].fErrorInd = TRUE;
  csm_vTriggerCsmUpstreamTask(CSM_C_IND_REASON_J1939_ERROR_IND);
}

 /*******************************************************************************
 * function    CSM_J1939_vCNPDataIndFF
 * \doxydocu
 * \brief      J1939 indication function to indicate the reception of a First Frame
 *             to the application.
 *
 * \param[in] bIsoTpChannel
 *              CAN stack internal channel Index.
 *
 * \return     void
 *
 * \access     CAN task
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 *
 * \history_end
 ******************************************************************************/
VOID CSM_J1939_vCNPDataIndFF(WORD wPGN)
{
  csm_j1939_abPGNIndFlags[wPGN].fRecvIndFF = TRUE;
  csm_vTriggerCsmUpstreamTask(CSM_C_IND_REASON_J1939_DATA_IND_FF);
}
#endif //(CSM_S_REMOTE_STACK == CSM_C_F_OFF)

/*******************************************************************************
 * function    CSM_J1939_lDataReq
 * \doxydocu
 * \brief      Request to send a transport protocol message over J1939 to CAN
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \param[in]  dwProtocolType
 *              Protocol type, see table 2 of CAn SW architecture.
 *
 * \param[in]  pvAddressField
 *              Protocol dependend address field.
 *
 * \param[in]  pbData
 *              Pointer to data which shall be send.
 *
 * \param[in]  wDataLength
 *              Number of data bytes which shall be send.
 *
 * \return     Error/no error value
 *
 * \access     application
 * \reentrant  only for different channels
 *
 * \lastreview
 *
 * \history_begin
 *
 * \history_end
 ******************************************************************************/
LONG CSM_J1939_lDataReq( DWORD dwProtocolType, const void* pvAddressField,
                         const BYTE* pbData, WORD wDataLength)
{
  LONG lReturnValue = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_WARNING, CSM_E_INIT_WARNING);
  CSM_ADDR_J1939* prAddr = (CSM_ADDR_J1939*) pvAddressField;


  if(dwProtocolType == CSM_C_PTYPE_J1939)
  {
    if((prAddr->wPGN       <   CSM_C_J1939_CFG_MAXPGN)
         &&
       (prAddr->bTgtNodeID <   CSM_C_J1939_CFG_MAXNODES)
         &&
       (prAddr->bSrcNodeID >=  CSM_C_J1939_NODE_OWN &&
        prAddr->bSrcNodeID <  (CSM_C_J1939_NODE_OWN+CSM_C_J1939_LOCAL_NODES)))
    {
      #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
      if(csm_abCsmSubState[cnp_j1939_arCfgPGNTable[prAddr->wPGN].bBus] == CSM_C_STATE_BUS_UP)
      {
        lReturnValue = CNP_J1939_lDataReq(prAddr->bTgtNodeID, prAddr->bSrcNodeID, prAddr->wPGN, pbData, wDataLength);
        if(lReturnValue == CSM_C_NO_ERROR)
        {
          csm_j1939_abPGNIndFlags[prAddr->wPGN].fSendReq = TRUE;
          csm_vTriggerCsmDownstreamTask(CSM_C_DOWN_REASON_J1939_DATA_REQ);
        }
      }
      else
      {
        lReturnValue = CNP_M_ITP_MAKE_ERROR (CSM_C_ERR_ERROR, CSM_E_WRONG_BUS_STATE);
      }
    #else
      lReturnValue = CSM_PROXY_lDataReq(dwProtocolType, pvAddressField, pbData, wDataLength);
        #endif
    }
    else
    {
      lReturnValue = CNP_M_ITP_MAKE_ERROR (CSM_C_ERR_ERROR, CSM_E_INVALID_PARA);
    }
  }
  else
  {
    lReturnValue = CNP_M_ITP_MAKE_ERROR (CSM_C_ERR_ERROR, CSM_E_INVALID_PROTOCOL_TYPE);
  }

  return (lReturnValue);
}

/*******************************************************************************
 * function    CSM_J1939_lCommunicationReq
 * \doxydocu
 * \brief      Requests the CAN bus for J1939 communication.
 *
 * \see        CAN-SW Architecture, written by V-Team
 *
 * \param[in]  bBus
 *             bus number
 *
 * \param[in]  dwProtocolType
 *             protocol type, see table 2 of CAn SW architecture
 *
 * \param[in]  *pvAddressField
 *             addressing, dependent on protocol
 *
 * \param[in]  bAction
 *             Allowed values: CSM_C_STACK_UP, CSM_C_STACK_DOWN
 *
 * \param[in]  wApplID (I)
 *               Appl Id
 *
 * \return     Error value
 *
 * \access     application
 * \reentrant  no, but the you can call it with diffrend wApplID without a problem
 *
 * \lastreview
 *
 * \history_begin
 *
 * \history_end
 ******************************************************************************/
LONG CSM_J1939_lCommunicationReq(BYTE bBus, DWORD dwProtocolType,
                                 const void* pvAddressField, BYTE bAction, WORD wApplID)
{
  LONG lReturnValue = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_WARNING, CSM_E_INIT_WARNING);
  BYTE bUser = csm_j1939_bGetApplNumber(bBus, dwProtocolType, pvAddressField, wApplID);


  if(  bUser < CSM_C_J1939_MAX_USER)
  {
  

#if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
  #if (CSM_S_J1939_STARTSTOP_BROADCAST == CSM_C_F_ON)
  if(bAction == CSM_C_NORMAL_COMMUNICATION_ON)
  {
    CNP_J1939_vSetBroadcastMode(bBus, FALSE);
    lReturnValue = CSM_C_NO_ERROR;
  }
  else if(bAction == CSM_C_NORMAL_COMMUNICATION_OFF)
  {
    CNP_J1939_vSetBroadcastMode(bBus, TRUE);
    lReturnValue = CSM_C_NO_ERROR;
  }
  else
#endif
#endif
  {
     /* Check for correct value of bAction: all but up is down  */
     bAction = CSM_C_STACK_UP == bAction ? CSM_C_STACK_UP : CSM_C_STACK_DOWN;

     if( csm_j1939_arCommunicationState[bUser].bReqAction != bAction)
     {/* Change of ReqAction */
       csm_j1939_arCommunicationState[bUser].bReqAction = bAction;
       csm_j1939_arCommunicationState[bUser].bCommunicationReq = TRUE;
       csm_vTriggerCsmDownstreamTask( CSM_C_DOWN_REASON_J1939_COMMUNICATION_REQ);

       lReturnValue = CSM_C_NO_ERROR;
     }
     else
     {/* Trigger Communication Confirmation directly */
       csm_j1939_arCommunicationState[bUser].bCommunicationCon = TRUE;
       csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_J1939_COMMUNICATION_CON);

       lReturnValue = CSM_C_NO_ERROR;
     }
  }
  } // bUSER < CSM_C_J1939_MAX_USER
  else
  {
    lReturnValue = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_C_INVALID_PARA);
  }

  return lReturnValue;
}

/*******************************************************************************
 * function    CSM_J1939_fCheckCommunicationReq
 * \doxydocu
 * \brief      Calculate Communication request.
 *
 *             Calculate Communication request.
 *             This function is called to calculate the CommunicationReq state.
 *
 * \param[in]  bBus
 *
 * \return     BOOL
 *               FALSE: the bus is not needed by CSM_J1939
 *               TRUE: the bus is neened by CSM_J1939
 *
 * \access     CAN-Task
 * \reentrant  yes, but only call by CAN-Task
 *
 * \lastreview
 *
 * \history_begin
 *
 * \history_end
 ******************************************************************************/
BOOL CSM_J1939_fCheckCommunicationReq(BYTE bBus)
{
  WORD wCount;

  for( wCount = 0; wCount < CSM_C_J1939_MAX_USER; wCount++)
  {
    if( csm_j1939_arApplCfg[wCount].bBus == bBus)
    {
      if( csm_j1939_arCommunicationState[wCount].bReqAction != CSM_C_STACK_DOWN)
      {
        return( TRUE);
      }
    }
  }
  return( FALSE);
}

/*******************************************************************************
 * function    CSM_J1939_lGetCommunicationState
 * \doxydocu
 * \brief      The actual state of the "connection" can be
 *             checked with this function.
 *
 * \param[in]  bBus
 *              Bus number, only needed for "broadcast" protocol types
 *
 * \param[in]  dwProtocolType
 *              Protocol type, see corresponding table of CAN SW architecture
 *
 * \param[in]  pvAddressField
 *              Protocol dependend address field
 *
 * \param[out] pbConnectState
 *              CSM_C_CONNECTED     /  CSM_C_STACK_UP
 *              CSM_C_DISCONNECTED  /  CSM_C_STACK_DOWN
 *
 * \param[in]  wApplId
 *              Application Id, only needed for "broadcast" protocol types
 *
 * \return     Error value (wrong Parameter) if < CSM_C_NO_ERROR
 *             else no error
 *
 * \access     CAN-Task
 * \reentrant  yes, but only call by CAN-Task
 *
 * \lastreview
 *
 * \history_begin
 *
 * \history_end
 ******************************************************************************/
LONG CSM_J1939_lGetCommunicationState(BYTE bBus, DWORD dwProtocolType,
                                      const void* pvAddressField,
                                      BYTE* pbConnectState, WORD wApplId)
{
  LONG lRet = CSM_C_NO_ERROR;
  BYTE bUser = csm_j1939_bGetApplNumber(bBus, dwProtocolType, pvAddressField, wApplId);

  if( bBus < CSM_C_MAX_CAN_BUS && bUser < CSM_C_J1939_MAX_USER)
  {
    *pbConnectState = csm_j1939_arCommunicationState[bUser].bActAction;
  }
  else
  {
    *pbConnectState = CSM_C_STACK_DOWN;
    lRet = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_C_INVALID_PARA);
  }

  return (lRet);
}

LONG CSM_J1939_lDataIndProcessed(DWORD dwProtocolType, const void* pvAddressField)
{
  LONG lReturnValue = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_WARNING, CSM_E_INIT_WARNING);

  if(dwProtocolType == CSM_C_PTYPE_J1939)
  {
    CSM_ADDR_J1939* prAddr = (CSM_ADDR_J1939*) pvAddressField;
    if(prAddr->wPGN < CSM_C_J1939_CFG_MAXPGN)
    {
    #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
      CNP_J1939_vClearPGNRxBuffer(prAddr->wPGN);
    #endif // (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
      lReturnValue = CSM_C_NO_ERROR;
    }
    else
    {
      lReturnValue = CNP_M_ITP_MAKE_ERROR (CSM_C_ERR_ERROR, CSM_E_INVALID_PARA);
    }
  }
  else
  {
    lReturnValue = CNP_M_ITP_MAKE_ERROR (CSM_C_ERR_ERROR, CSM_E_INVALID_PROTOCOL_TYPE);
  }

  return lReturnValue;
}

#if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
void CSM_J1939_vPGNTxTimeout(DWORD dwIndex)
{
  if(dwIndex < CSM_C_J1939_CFG_MAXPGN)
  {
    csm_j1939_abPGNIndFlags[dwIndex].fTxTimeoutInd = TRUE;
    csm_vTriggerCsmDownstreamTask(CSM_C_DOWN_REASON_J1939_PGN_TIMEOUT);
  }
}

void CSM_J1939_vPGNRxTimeout(DWORD dwIndex)
{
  if(dwIndex < CSM_C_J1939_CFG_MAXPGN)
  {
    csm_j1939_abPGNIndFlags[dwIndex].fRxTimeoutInd = TRUE;
    csm_vTriggerCsmDownstreamTask(CSM_C_DOWN_REASON_J1939_PGN_TIMEOUT);
  }
}


#if (CSM_S_J1939_STARTSTOP_BROADCAST == CSM_C_F_ON)
void CSM_J1939_vStopBroadcastTimeout(DWORD dwParam)
{
  if(dwParam < CSM_C_MAX_CAN_BUS)
  {
    CNP_J1939_vStopBroadcastTimeout(dwParam);
  }
}
#endif
#endif // (CSM_S_REMOTE_STACK == CSM_C_F_OFF)

/*******************************************************************************
 * function    CSM_J1939_vProcessCommunicationReq
 * \doxydocu
 * \brief      Handle communication request.
 *
 *             Handle communication request.
 *             This function is call in downstream-task after trigger with
 *             CSM_C_DOWN_REASON_CSM_ITP_COMMUNICATION_REQ.
 *
 * \return     void
 *
 * \access     CAN-Task
 * \reentrant  no, but only call by CAN-Task
 *
 * \lastreview
 *
 * \history_begin
 *
 * \history_end
 ******************************************************************************/
VOID CSM_J1939_vProcessCommunicationReq(VOID)
{
  BOOL fStateChanged = FALSE;
  BYTE bUserNumber;

  for( bUserNumber = 0; bUserNumber < CSM_C_J1939_MAX_USER; bUserNumber++)
  {
    if( csm_j1939_arCommunicationState[bUserNumber].bCommunicationReq == TRUE)
    {
      csm_j1939_arCommunicationState[bUserNumber].bCommunicationReq = FALSE;
      csm_j1939_arCommunicationState[bUserNumber].bCommunicationCon = TRUE;
      fStateChanged = TRUE;
    }
  }

  if(fStateChanged == TRUE)
  {
    csm_vTriggerCsmDownstreamTask( CSM_C_DOWN_REASON_CSM_BUS_STATE_CHANGED);
    csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_J1939_COMMUNICATION_CON);
  }
}

/*******************************************************************************
 * function    CSM_J1939_vProcessCommunicationCon
 * \doxydocu
 * \brief      Handle communication confimation.
 *
 *             Handle communication confimation.
 *             This function is called in upstream-task after trigger
 *             with CSM_C_IND_REASON_CSM_ITP_COMMUNICATION_CON.
 *
 * \return     void
 *
 * \access     Upstream-Task
 * \reentrant  no, but only call by Upstream-Task
 *
 * \lastreview
 *
 * \history_begin
 *
 * \history_end
 ******************************************************************************/
VOID CSM_J1939_vProcessCommunicationCon(VOID)
{
  BYTE bUserNumber;

  for( bUserNumber = 0; bUserNumber < CSM_C_J1939_MAX_USER; bUserNumber++)
  {
    if( csm_j1939_arCommunicationState[bUserNumber].bCommunicationCon == TRUE)
    {
      csm_j1939_arCommunicationState[bUserNumber].bCommunicationCon = FALSE;

      if( csm_j1939_arApplCallbackFkt[bUserNumber].pfvCommunicationCon != NULL)
      {
        #if (CSM_S_TRACE == CSM_C_F_ON)
       #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
            #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      //TODO
            #endif
       #else
            {
              BYTE abData[ 10];
              abData[ 0] = csm_j1939_arApplCfg[ bUserNumber].bBus;
              abData[ 1] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD( csm_j1939_arApplCfg[ bUserNumber].dwProtocolType));
              abData[ 2] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD( csm_j1939_arApplCfg[ bUserNumber].dwProtocolType));
              abData[ 3] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD( csm_j1939_arApplCfg[ bUserNumber].dwProtocolType));
              abData[ 4] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD( csm_j1939_arApplCfg[ bUserNumber].dwProtocolType));
              abData[ 5] = 0x00;
              abData[ 6] = 0x00;
              abData[ 7] = csm_j1939_arCommunicationState[bUserNumber].bActAction;
              abData[ 8] = CSM_M_GET_HIBYTE( csm_j1939_arApplCfg[ bUserNumber].wApplId);
              abData[ 9] = CSM_M_GET_LOBYTE( csm_j1939_arApplCfg[ bUserNumber].wApplId);
              CSM_M_TRACEOUT( CSM_C_TRACEOUT_LEVEL_INTERFACE,  CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_COMMUNICATION_CON, abData ,  10);
            }
       #endif
        #endif /* CSM_S_TRACE */
        CSM_M_SET_TASK_CONTEXT( &(csm_j1939_arApplCallbackFkt[bUserNumber].rTaskInfo),
        csm_j1939_arApplCallbackFkt[bUserNumber].pfvCommunicationCon(
        #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON )
                                           csm_j1939_arApplCallbackFkt[bUserNumber].pvHandle,
        #endif
                                           csm_j1939_arApplCfg[bUserNumber].bBus,
                                           csm_j1939_arApplCfg[bUserNumber].dwProtocolType,
                                           NULL,
                                           csm_j1939_arCommunicationState[bUserNumber].bActAction,
                                           csm_j1939_arApplCfg[bUserNumber].wApplId) );
      }
    }
  }
}

/*******************************************************************************
 * function    CSM_J1939_vProcessCommunicationInd
 * \doxydocu
 * \brief      Handle communication indication.
 *
 *             Handle communication indication.
 *             This function is called in upstream-task after trigger
 *             with CSM_C_IND_REASON_CSM_ITP_COMMUNICATION_IND.
 *
 * \return     void
 *
 * \access     Upstream-Task
 * \reentrant  no, but only call by Upstream-Task
 *
 * \lastreview
 *
 * \history_begin
 *
 * \history_end
 ******************************************************************************/
VOID CSM_J1939_vProcessCommunicationInd(VOID)
{
  BYTE bUserNumber;

  for( bUserNumber = 0; bUserNumber < CSM_C_J1939_MAX_USER; bUserNumber++)
  {
    if( csm_j1939_arCommunicationState[bUserNumber].bCommunicationInd == TRUE)
    {
      csm_j1939_arCommunicationState[bUserNumber].bCommunicationInd = FALSE;

      if( csm_j1939_arApplCallbackFkt[bUserNumber].pfvCommunicationInd != NULL)
      {
        #if (CSM_S_TRACE == CSM_C_F_ON)
       #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
            #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      //TODO
            #endif
       #else
            {
              BYTE abData[ 10];
              abData[ 0] = csm_j1939_arApplCfg[ bUserNumber].bBus;
              abData[ 1] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD( csm_j1939_arApplCfg[ bUserNumber].dwProtocolType));
              abData[ 2] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD( csm_j1939_arApplCfg[ bUserNumber].dwProtocolType));
              abData[ 3] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD( csm_j1939_arApplCfg[ bUserNumber].dwProtocolType));
              abData[ 4] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD( csm_j1939_arApplCfg[ bUserNumber].dwProtocolType));
              abData[ 5] = 0x00;
              abData[ 6] = 0x00;
              abData[ 7] = csm_j1939_arCommunicationState[bUserNumber].bActAction;
              abData[ 8] = CSM_M_GET_HIBYTE( csm_j1939_arApplCfg[ bUserNumber].wApplId);
              abData[ 9] = CSM_M_GET_LOBYTE( csm_j1939_arApplCfg[ bUserNumber].wApplId);
              CSM_M_TRACEOUT( CSM_C_TRACEOUT_LEVEL_INTERFACE,  CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_COMMUNICATION_IND, abData ,  10);
            }
       #endif
        #endif /* CSM_S_TRACE */
        CSM_M_SET_TASK_CONTEXT( &(csm_j1939_arApplCallbackFkt[bUserNumber].rTaskInfo),
        csm_j1939_arApplCallbackFkt[bUserNumber].pfvCommunicationInd(
        #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                                     csm_j1939_arApplCallbackFkt[bUserNumber].pvHandle,
        #endif
                                     csm_j1939_arApplCfg[bUserNumber].bBus,
                                     csm_j1939_arApplCfg[bUserNumber].dwProtocolType,
                                     NULL,
                                     csm_j1939_arCommunicationState[bUserNumber].bActAction,
                                     csm_j1939_arApplCfg[bUserNumber].wApplId) );
      }
    }
  }
}


VOID CSM_J1939_vProcessDataReq(VOID)
{
  WORD wIdx;
  for(wIdx = 0; wIdx < CSM_C_J1939_CFG_MAXPGN; wIdx++)
  {
    if(csm_j1939_abPGNIndFlags[wIdx].fSendReq)
    {
      csm_j1939_abPGNIndFlags[wIdx].fSendReq = FALSE;
    #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
      CNP_J1939_vProcessDataReq(wIdx);
    #endif
    }
  }
}

#if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
VOID CSM_J1939_vProcessDataInd(VOID)
{
  WORD wPGN;
  CSM_ADDR_J1939 rAddr;

  for( wPGN = 0; wPGN < CSM_C_J1939_CFG_MAXPGN; wPGN++)
  {
    if( csm_j1939_abPGNIndFlags[wPGN].fRecvInd == TRUE )
    {
      csm_j1939_abPGNIndFlags[wPGN].fRecvInd = FALSE;

      rAddr.bSrcNodeID = CNP_J1939_bGetNodeID(cnp_j1939_arCfgPGNTable[wPGN].bBus, cnp_j1939_arActPGNTable[wPGN].rRxData.bSrcAddr);
      rAddr.bTgtNodeID = CNP_J1939_bGetNodeID(cnp_j1939_arCfgPGNTable[wPGN].bBus, cnp_j1939_arActPGNTable[wPGN].rRxData.bTgtAddr);
      rAddr.wPGN = wPGN;

      /* only indicate to application, if bus is up */
      if(csm_abCsmSubState[cnp_j1939_arCfgPGNTable[wPGN].bBus] == CSM_C_STATE_BUS_UP)
      {
         #if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
         WORD wIdx;
         for( wIdx = 0; wIdx < CSM_C_J1939_MAX_PGN_CALLBACKS; wIdx++)
         {
           if(csm_j1939_arPGNCallbackFkt[wPGN][wIdx].pfvDataInd != NULL)
           {
              CSM_M_SET_TASK_CONTEXT( &(csm_j1939_arPGNCallbackFkt[wPGN][wIdx].rTaskInfo),
              csm_j1939_arPGNCallbackFkt[wPGN][wIdx].pfvDataInd(
              #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                                           csm_j1939_arPGNCallbackFkt[wPGN][wIdx].pvHandle,
              #endif
                                           CSM_C_PTYPE_J1939, &rAddr,
                                           cnp_j1939_arActPGNTable[wPGN].rRxData.pbBuffer,
                                           cnp_j1939_arActPGNTable[wPGN].rRxData.wBufferLen) );
           }
         }
         #else
         if(csm_j1939_arPGNCallbackFkt[wPGN].pfvDataInd != NULL)
         {
           CSM_M_SET_TASK_CONTEXT( &(csm_j1939_arPGNCallbackFkt[wPGN].rTaskInfo),
           csm_j1939_arPGNCallbackFkt[wPGN].pfvDataInd(
           #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                                        csm_j1939_arPGNCallbackFkt[wPGN].pvHandle,
           #endif
                                        CSM_C_PTYPE_J1939, &rAddr,
                                        cnp_j1939_arActPGNTable[wPGN].rRxData.pbBuffer,
                                        cnp_j1939_arActPGNTable[wPGN].rRxData.wBufferLen) );
         }
         #endif
      }
      CNP_J1939_vClearPGNRxBuffer(wPGN);
    }
  }
}

VOID CSM_J1939_vProcessDataCon(VOID)
{
  WORD wPGN;
  #if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
  WORD wIdx;
  #endif
  CSM_ADDR_J1939 rAddr;
  BYTE bConf;

  for( wPGN = 0; wPGN < CSM_C_J1939_CFG_MAXPGN; wPGN++)
  {
    if( csm_j1939_abPGNIndFlags[wPGN].fDataCon != CSM_C_CONF_NONE)
    {
      bConf = csm_j1939_abPGNIndFlags[wPGN].fDataCon;
      rAddr.bSrcNodeID = CNP_J1939_bGetNodeID(cnp_j1939_arCfgPGNTable[wPGN].bBus,
                                              cnp_j1939_arActPGNTable[wPGN].rTxData.bSrcAddr);
      rAddr.bTgtNodeID = CNP_J1939_bGetNodeID(cnp_j1939_arCfgPGNTable[wPGN].bBus,
                                              cnp_j1939_arActPGNTable[wPGN].rTxData.bTgtAddr);
      rAddr.wPGN = wPGN;

      csm_j1939_abPGNIndFlags[wPGN].fDataCon = CSM_C_CONF_NONE;

      /* Clear the PGN Tx structure before the indication function is called
       * so that the callback can use this PGN directly. */
      CNP_J1939_vClearPGNTxBuffer(wPGN);

      #if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
      for( wIdx = 0; wIdx < CSM_C_J1939_MAX_PGN_CALLBACKS; wIdx++)
      {
        if(csm_j1939_arPGNCallbackFkt[wPGN][wIdx].pfvDataCon != NULL)
        {
           CSM_M_SET_TASK_CONTEXT( &(csm_j1939_arPGNCallbackFkt[wPGN][wIdx].rTaskInfo),
           csm_j1939_arPGNCallbackFkt[wPGN][wIdx].pfvDataCon(
           #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                                                       csm_j1939_arPGNCallbackFkt[wPGN][wIdx].pvHandle,
           #endif
                                                       CSM_C_PTYPE_J1939, &rAddr, bConf));
        }
      }
      #else
      if( csm_j1939_arPGNCallbackFkt[wPGN].pfvDataCon != NULL)
      {
        CSM_M_SET_TASK_CONTEXT( &(csm_j1939_arPGNCallbackFkt[wPGN].rTaskInfo),
        csm_j1939_arPGNCallbackFkt[wPGN].pfvDataCon(
        #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                                                    csm_j1939_arPGNCallbackFkt[wPGN].pvHandle,
        #endif
                                                    CSM_C_PTYPE_J1939, &rAddr, bConf));
      }
      #endif
    }
  }
}

VOID CSM_J1939_vProcessErrorInd(VOID)
{
}


VOID CSM_J1939_vProcessDataIndFF(VOID)
{
  WORD wPGN;
  #if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
  WORD wIdx;
  #endif
  CSM_ADDR_J1939 rAddr;

  for( wPGN = 0; wPGN < CSM_C_J1939_CFG_MAXPGN; wPGN++)
  {
    if( csm_j1939_abPGNIndFlags[wPGN].fRecvIndFF == TRUE )
    {
      csm_j1939_abPGNIndFlags[wPGN].fRecvIndFF = FALSE;

      rAddr.bSrcNodeID = CNP_J1939_bGetNodeID(cnp_j1939_arCfgPGNTable[wPGN].bBus,
                                              cnp_j1939_arActPGNTable[wPGN].rTxData.bSrcAddr);
      rAddr.bTgtNodeID = CNP_J1939_bGetNodeID(cnp_j1939_arCfgPGNTable[wPGN].bBus,
                                              cnp_j1939_arActPGNTable[wPGN].rTxData.bTgtAddr);
      rAddr.wPGN = wPGN;

      #if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
      for( wIdx = 0; wIdx < CSM_C_J1939_MAX_PGN_CALLBACKS; wIdx++)
      {
        if(csm_j1939_arPGNCallbackFkt[wPGN][wIdx].pfvDataIndFF != NULL)
        {
           CSM_M_SET_TASK_CONTEXT( &(csm_j1939_arPGNCallbackFkt[wPGN][wIdx].rTaskInfo),
           csm_j1939_arPGNCallbackFkt[wPGN][wIdx].pfvDataIndFF(
           #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                                        csm_j1939_arPGNCallbackFkt[wPGN][wIdx].pvHandle,
           #endif
                                        CSM_C_PTYPE_J1939, &rAddr,
                                        cnp_j1939_arActPGNTable[wPGN].rRxData.wBufferLen) );
        }
      }
      #else
      if( (csm_j1939_arPGNCallbackFkt[wPGN].pfvDataIndFF != NULL) &&
          (csm_abCsmSubState[cnp_j1939_arCfgPGNTable[wPGN].bBus] == CSM_C_STATE_BUS_UP) )
          /* only indicate to application, if bus is up */
      {
        CSM_M_SET_TASK_CONTEXT( &(csm_j1939_arPGNCallbackFkt[wPGN].rTaskInfo),
        csm_j1939_arPGNCallbackFkt[wPGN].pfvDataIndFF(
        #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                                     csm_j1939_arPGNCallbackFkt[wPGN].pvHandle,
        #endif
                                     CSM_C_PTYPE_J1939, &rAddr,
                                     cnp_j1939_arActPGNTable[wPGN].rRxData.wBufferLen) );
      }
      #endif
    }
  }
}

VOID CSM_J1939_vProcessDataIndProcessed(VOID)
{
}

VOID CSM_J1939_vProcessPGNTimeout(VOID)
{
  WORD wIdx;
  for(wIdx = 0; wIdx < CSM_C_J1939_CFG_MAXPGN; wIdx++)
  {
    if(csm_j1939_abPGNIndFlags[wIdx].fTxTimeoutInd == TRUE)
    {
      csm_j1939_abPGNIndFlags[wIdx].fTxTimeoutInd = FALSE;
      CNP_J1939_vPGNTxTimeout(wIdx);
    }
    if(csm_j1939_abPGNIndFlags[wIdx].fRxTimeoutInd == TRUE)
    {
      csm_j1939_abPGNIndFlags[wIdx].fRxTimeoutInd = FALSE;
      CNP_J1939_vPGNRxTimeout(wIdx);
    }
  }
}

BYTE CSM_J1939_bGetCCSChannelId(WORD wPgn)
{
  if(wPgn < CSM_C_J1939_CFG_MAXPGN)
  return cnp_j1939_arCfgPGNTable[wPgn].bCCSNPChannel;
  else
    return CSM_C_J1939_CFG_MAXPGN+1;
}
#endif /* CSM_S_REMOTE_STACK */
/*******************************************************************************
 * function    CSM_J1939_vCSMBusStateInd
 * \doxydocu
 * \brief      Bus State indication callback.
 *
 *             This funktion is called after the bus state is changed.
 *
 * \param[in]  bBus
 *             Bus
 *
 * \param[in]  bCsmActState
 *             New bus state.
 *             possible values:
 *                CSM_C_STATE_BUS_DOWN, CSM_C_STATE_BUS_LISTEN, CSM_C_STATE_BUS_STARTUP,
 *                CSM_C_STATE_BUS_UP, CSM_C_STATE_BUS_SHUTDOWN
 *
 * \param[in]  bVN
 *             Bit field for virtual networks
 *
 * \return     void
 *
 * \access     interrupt
 * \reentrant  yes
 *
 * \lastreview
 *
 * \history_begin
 *
 * \history_end
 ******************************************************************************/
VOID CSM_J1939_vCSMBusStateInd(BYTE bBus, BYTE bCsmActState)
{
  WORD wCount;

  switch( bCsmActState)
  {
  case CSM_C_STATE_BUS_DOWN:
  case CSM_C_STATE_BUS_LISTEN:
  case CSM_C_STATE_BUS_STARTUP:
  case CSM_C_STATE_BUS_SHUTDOWN:
    {
      for( wCount = 0; wCount < CSM_C_J1939_MAX_USER; wCount++)
      {
        if( csm_j1939_arApplCfg[ wCount].bBus == bBus)
        {
          if( csm_j1939_arCommunicationState[wCount].bActAction != CSM_C_STACK_DOWN)
          {
            csm_j1939_arCommunicationState[wCount].bActAction = CSM_C_STACK_DOWN;
            csm_j1939_arCommunicationState[wCount].bCommunicationInd = TRUE;
            csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_J1939_COMMUNICATION_IND);
          }
        }
      }
    }
    break;
  case CSM_C_STATE_BUS_UP:
    {
      for( wCount = 0; wCount < CSM_C_J1939_MAX_USER; wCount++)
      {
        if( csm_j1939_arApplCfg[ wCount].bBus == bBus)
        {
          if( csm_j1939_arCommunicationState[wCount].bActAction != CSM_C_STACK_UP)
          {
            csm_j1939_arCommunicationState[wCount].bActAction = CSM_C_STACK_UP;
            csm_j1939_arCommunicationState[wCount].bCommunicationInd = TRUE;
            csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_J1939_COMMUNICATION_IND);
          }
        }
      }
    }
    break;
  default:
    {
      CSM_vAssert( FALSE);
    }
    break;
  }
}

#if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
LONG CSM_J1939_lApplCallbackInit(
                #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                          VOID*       pvHandle,
                #endif
                 BYTE bBus, DWORD dwProtocolType, const void* pvAddressField, tCSM_J1939_APPL_CALLBACK* prCallBackFkt)
{
  WORD wIdx;
  LONG lRet = CSM_C_NO_ERROR;
  CSM_ADDR_J1939 *pvAddr = (CSM_ADDR_J1939*) pvAddressField;

  CSM_vAssert(pvAddr != NULL);
  /* look if a communication function shall be registered */
  if(prCallBackFkt->pfvCommunicationCon != NULL || prCallBackFkt->pfvCommunicationInd != NULL)
  {
     BYTE bUser = csm_j1939_bGetApplNumber(bBus, dwProtocolType, NULL, pvAddr->bSrcNodeID);
     if(bUser < CSM_C_J1939_MAX_USER)
     {
       csm_j1939_arApplCallbackFkt[bUser].pfvCommunicationCon = prCallBackFkt->pfvCommunicationCon;
       csm_j1939_arApplCallbackFkt[bUser].pfvCommunicationInd = prCallBackFkt->pfvCommunicationInd;
       #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
        csm_j1939_arApplCallbackFkt[bUser].pvHandle = pvHandle;
       #endif
     }
     else
     {
       lRet = CNP_M_ITP_MAKE_ERROR (CSM_C_ERR_ERROR, CSM_E_NO_FREE_CALLBACK);
     }
  }

  /* and look if a data function shall be registered */
  if(prCallBackFkt->pfvDataCon != NULL || prCallBackFkt->pfvDataInd != NULL)
  {
     if (pvAddr->wPGN < CSM_C_J1939_CFG_MAXPGN)
     {
       /* PGN CallBack Function Registration */
       /* registration of first-frame and error callbacks is not (yet) supported */
       for( wIdx = 0; wIdx < CSM_C_J1939_MAX_PGN_CALLBACKS; wIdx++)
       {
         if(csm_j1939_arPGNCallbackFkt[pvAddr->wPGN][wIdx].pfvDataCon == NULL &&
            csm_j1939_arPGNCallbackFkt[pvAddr->wPGN][wIdx].pfvDataInd == NULL)
         {
           /* free location found, use for this PGN */
           csm_j1939_arPGNCallbackFkt[pvAddr->wPGN][wIdx].pfvDataCon = prCallBackFkt->pfvDataCon;
           csm_j1939_arPGNCallbackFkt[pvAddr->wPGN][wIdx].pfvDataInd = prCallBackFkt->pfvDataInd;
           #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
            csm_j1939_arPGNCallbackFkt[pvAddr->wPGN][wIdx].pvHandle = pvHandle;
           #endif
           break;
         }
       }

       if( wIdx == CSM_C_J1939_MAX_PGN_CALLBACKS )
       {
         lRet = CNP_M_ITP_MAKE_ERROR (CSM_C_ERR_ERROR, CSM_E_NO_FREE_CALLBACK);
       }
     }
     else
     {
        lRet = CNP_M_ITP_MAKE_ERROR (CSM_C_ERR_ERROR, CSM_E_NO_FREE_CALLBACK);
     }
  }

  return(lRet);
}
#endif /* (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON) */


#if (CSM_S_REMOTE_STACK == CSM_C_F_ON)
/*******************************************************************************
 * function    CSM_J1939_vPROXYTriggerCommunicationCon
 * \doxydocu
 * \brief      Callback function from PROXY to trigger a communication confirmation
 *  for J1939 protocol
 * \param[in]  bUser
 *                 Data
 *
 * \return     void
 *
 * \access     CSM task level
 * \reentrant  no, but only call from interrupt
 *
 * \lastreview
 *
 * \history_begin
 *
 * 20.08.14 Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CSM_J1939_vPROXYTriggerCommunicationCon(BYTE bUser, BYTE bAction)
{
  csm_j1939_arCommunicationState[bUser].bActAction = bAction;
  csm_j1939_arCommunicationState[bUser].bCommunicationCon = TRUE;
  csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_J1939_COMMUNICATION_CON);
}

/*******************************************************************************
 * function    CSM_J1939_vPROXYTriggerCommunicationInd
 * \doxydocu
 * \brief      Callback function from PROXY to trigger a communication indication
 * for J1939 protocol
 * \param[in]  bUser
 *                 Data
 *
 * \return     void
 *
 * \access     CSM task level
 * \reentrant  no, but only call from interrupt
 *
 * \lastreview
 *
 * \history_begin
 *
 * 20.08.14 Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CSM_J1939_vPROXYTriggerCommunicationInd(BYTE bUser, BYTE bAction)
{
  csm_j1939_arCommunicationState[bUser].bActAction = bAction;
  csm_j1939_arCommunicationState[bUser].bCommunicationInd = TRUE;
  csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_J1939_COMMUNICATION_IND);
}

#endif /* CSM_S_REMOTE_STACK == CSM_C_F_ON */
#endif /* CSM_S_DOXYGEN_HIDE_NON_API_FCT */

#endif //(CSM_S_J1939 == CSM_C_F_ON)

#undef CSM_J1939_SELF
#undef CSM_CAN_STACK

