/* Description and history of csm_itp.c at the end of the file.               */
/** File name for preprozessor commands */
#define CSM_ITP_SELF
/** csm_itp.c is a file according V-team CAN-SW architecture implementation rules */
#define CSM_CAN_STACK

/*******************************************************************************
 * includes
 ******************************************************************************/
#include "csm_stack_i.h"
#include "csm_i.h"
#include "csm_stack_m.h"
#include "csm_m.h"

/*ADDED: ama8kor*/
/*for compiler warning removal - function call without a prototype*/
extern LONG CSM_PROXY_lDataReq(DWORD, const void*, const BYTE*, WORD);
/*till here*/


/*#include "csm_itp_i.h" no interface to the application */
#include "csm_itp_m.h"     /* ISO-TP specific functions/defines of the can stack manager */
#include "cnp_itp_i.h"

#include "csm_c.h"

#if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
  #if (CSM_S_OSEK_NM == CSM_C_F_ON)
    #if ( CNM_OSEK_S_PAYLOAD == CSM_C_F_ON )
    #include "cnm_osek_i.h"
    #endif
  #endif /* CSM_S_OSEK_NM */
#endif /* CSM_S_CNM_AVAILABLE */

/*************** ETG Tracing ******/
#if (CSM_S_TRACE == CSM_C_F_ON)
  #define ETRACE_S_IMPORT_INTERFACE_GENERIC
  #include "etrace_if.h"

  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
    #include "csm_etg_trace.h"
    #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FD_CSM
    /* #define ETG_INLINE */
    #include "trcGenProj/Header/csm_itp.c.trc.h"
  #endif
  #endif
#endif /* (CSM_S_TRACE == CSM_C_F_ON) */

#ifndef CSM_S_CNP_AVAILABLE
  #error "CSM_S_CNP_AVAILABLE must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_ISO_TP
  #error "CSM_S_ISO_TP must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_ITP_FF_INDICATION_AVAILABLE
  #error "CSM_S_ITP_FF_INDICATION_AVAILABLE must be set to CSM_C_F_ON or -OFF !"
#endif
#ifndef CSM_S_ITP_ERROR_INDICATION_AVAILABLE
  #error "CSM_S_ITP_ERROR_INDICATION_AVAILABLE must be set to CSM_C_F_ON or -OFF !"
#endif

/*******************************************************************************
 * defines and macros (scope: csm module local)
 ******************************************************************************/
/* --none-- or in csm_itp_m(p).h */

/*******************************************************************************
 * typedefs (scope: csm module local)
 ******************************************************************************/
#if (CSM_S_ITP_COMMUNICATION_FCT == CSM_C_F_ON)
/** data structure for CSM ISO TP communication state */
typedef struct
{
  BYTE abReqAction[ CSM_C_ISO_TP_MAX_CONNECTIONS];            /**< Array to store the bAction parameter
                                                               *   from CSM_ITP_lCommunicationReq() from every user.
                                                               *   Possible values: CSM_C_STACK_DOWN, CSM_C_STACK_UP */
  BYTE abActAction[ CSM_C_ISO_TP_MAX_CONNECTIONS];            /**< Array to store the bAction parameter
                                                               *   from CSM_ITP_lCommunicationReq() from every user.
                                                               *   Possible values: CSM_C_STACK_DOWN, CSM_C_STACK_UP */
  /*  BYTE  bBusState;                                *< State of the bus. Possible values: CSM_C_STATE_BUS_DOWN, CSM_C_STATE_BUS_LISTEN, CSM_C_STATE_BUS_STARTUP, CSM_C_STATE_BUS_SHUTDOWN, CSM_C_STATE_BUS_UP */
  BOOLEAN afCommunicationReq[ CSM_C_ISO_TP_MAX_CONNECTIONS];  /**< Trigger array Flag for Communication request,
                                                               *    if TRUE communication request is pending.
                                                               *    Possible value TRUE/FALSE */
  BOOLEAN afCommunicationCon[ CSM_C_ISO_TP_MAX_CONNECTIONS];  /**< Trigger array Flag for Communication confirmation,
                                                               *   if TRUE communication confirmation is pending.
                                                               *   Possible value TRUE/FALSE */
  BOOLEAN afCommunicationInd[ CSM_C_ISO_TP_MAX_CONNECTIONS];  /**< Trigger array Flag for Communication indication,
                                                               *   if TRUE communication indication is penning.
                                                               *   Possible value TRUE/FALSE */
}tCSM_ITP_COMMUNICATION_STATE ;
#endif

/*******************************************************************************
 * variable definition (scope: csm module local)
 ******************************************************************************/
#if (CSM_S_ITP_COMMUNICATION_FCT == CSM_C_F_ON)
  static tCSM_ITP_COMMUNICATION_STATE CSM_ITP_rCommunicationState; /**< This variable stores the CSM ISO TP state */
#endif

/*******************************************************************************
 * variable definition (scope: CAN stack)
 ******************************************************************************/
/* --none-- or in csm_itp_i(p).h */

/*******************************************************************************
 * function prototypes (scope: csm module local)
 ******************************************************************************/
/* --none-- or in csm_itp_i(p).h */

/* Feature switch is used only to control doxygen documentation */
#if (CSM_S_DOXYGEN_HIDE_NON_API_FCT == CSM_C_F_OFF)

/*******************************************************************************
 * function implementation (scope: file local = static)
 ******************************************************************************/
/* --none-- */

/*******************************************************************************
 * function implementation (scope: CSM module local)
 ******************************************************************************/
/*******************************************************************************
 * function    CSM_ITP_lInit
 * \doxydocu
 * \brief      Initializes CSM ISO-TP Server module.
 *
 *             Initializes CSM ISO-TP Server module.
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *
 * \param      bInitMode
 *             Allowed values:
 *                CSM_C_INIT_COLD, CSM_C_INIT_WARM, CSM_C_INIT_START (see csm_stack_i.h)
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR, CBR_E_ERROR_STATE_WHILE_INIT
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview 02.08.06 CM-DI/ESA3-Zurmhl
 *
 * \history_begin
 * 02.08.06 CM-DI/ESA3-Zurmhl
 * csm_itp_arCallbackFkt[wIsoTpChannel].pvHandle initialized to NULL
 *
 * 28.07.06 CM-DI/ESA3-K"ohler
 * CSM_ITP_rCommunicationState encapsulated with feature switch CSM_S_ITP_COMMUNICATION_FCT.
 *
 * 21.06.05 CM-DI/ESA3-K"ohler
 * Initialization of ITP variables for upstream task indication moved from
 * CSM_lInit() to CSM_ITP_lInit().
 *
 * 27.05.05 CM-DI/ESU3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
LONG CSM_ITP_lInit(
  BYTE bInitMode)
{
  LONG return_value = CSM_C_NO_ERROR;
  WORD wIsoTpChannel;

  if(bInitMode == CSM_C_INIT_COLD)
  {
  }
  else if (bInitMode == CSM_C_INIT_WARM)
  {
    /* Init variables for Upstream Task */
    CSM_ITP_vResetIndicationFlags();

    for( wIsoTpChannel = 0; wIsoTpChannel < CSM_C_ISO_TP_MAX_CONNECTIONS; wIsoTpChannel++)
    {
//TODO: MP cleanup for Proxy needed
      csm_rIsoTpConnectionAct[wIsoTpChannel].fTxChannelFree    = TRUE;
// Vielleicht sollte man die nderung in csm_lCheckItpDataReqErrors() machen
//TODO: MP cleanup for Proxy needed

      #if (CSM_S_ITP_COMMUNICATION_FCT == CSM_C_F_ON)
        CSM_ITP_rCommunicationState.abReqAction[ wIsoTpChannel] = CSM_C_STACK_DOWN;
        CSM_ITP_rCommunicationState.abActAction[ wIsoTpChannel] = CSM_C_STACK_DOWN;
        CSM_ITP_rCommunicationState.afCommunicationCon[ wIsoTpChannel] = FALSE;
        CSM_ITP_rCommunicationState.afCommunicationInd[ wIsoTpChannel] = FALSE;
        CSM_ITP_rCommunicationState.afCommunicationReq[ wIsoTpChannel] = FALSE;
      #endif
      #if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
        csm_itp_arCallbackFkt[wIsoTpChannel].rCallback.pfvCommunicationCon = NULL;
        csm_itp_arCallbackFkt[wIsoTpChannel].rCallback.pfvCommunicationInd = NULL;
        csm_itp_arCallbackFkt[wIsoTpChannel].rCallback.pfvDataCon = NULL;
        csm_itp_arCallbackFkt[wIsoTpChannel].rCallback.pfvDataInd = NULL;
        #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
          csm_itp_arCallbackFkt[wIsoTpChannel].rCallback.pfvDataIndFF = NULL;
        #endif /*CSM_S_ITP_FF_INDICATION_AVAILABLE*/
        #if (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
          csm_itp_arCallbackFkt[wIsoTpChannel].rCallback.pfvDataErrorInd = NULL;
        #endif /*CSM_S_ITP_ERROR_INDICATION_AVAILABLE*/

        #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
          csm_itp_arCallbackFkt[wIsoTpChannel].pvHandle = NULL;
        #endif /* CSM_S_USE_CALLBACK_WITH_HANDLE */
      #endif
    }
  }
  else if (bInitMode == CSM_C_INIT_START)
  {
  }
  else
  {
    return CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_UNKNOWN_STATE);
  }

  return( return_value);
}
/*******************************************************************************
 * function    CSM_ITP_lExit
 * \doxydocu
 * \brief      Stops CSM CAN ITP server.
 *
 *             Stops CAN ITP server.
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *             When the stop state is called, the CBR state is set to DOWN. This state
 *             switch is not indicated to registered applications.
 * \param[in]  bExitMode
 *             Allowed values (see csm_stack_i.h):
 *                - CSM_C_EXIT_STOP
 *                - CSM_C_EXIT_OFF
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview 25.07.08 VTeam-Zurmhl
 *
 * \history_begin
 * 12.06.08 Brunotte
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
LONG CSM_ITP_lExit(
   BYTE bExitMode )
{
  LONG return_value = CSM_C_NO_ERROR;
  WORD wIsoTpChannel;

  switch (bExitMode)
  {
    case CSM_C_EXIT_STOP:
    {
      /* Init variables for Upstream Task */
      CSM_ITP_vResetIndicationFlags();

      for( wIsoTpChannel = 0; wIsoTpChannel < CSM_C_ISO_TP_MAX_CONNECTIONS; wIsoTpChannel++)
      {
        #if (CSM_S_ITP_COMMUNICATION_FCT == CSM_C_F_ON)
          CSM_ITP_rCommunicationState.abReqAction[ wIsoTpChannel] = CSM_C_STACK_DOWN;
          CSM_ITP_rCommunicationState.abActAction[ wIsoTpChannel] = CSM_C_STACK_DOWN;
          CSM_ITP_rCommunicationState.afCommunicationCon[ wIsoTpChannel] = FALSE;
          CSM_ITP_rCommunicationState.afCommunicationInd[ wIsoTpChannel] = FALSE;
          CSM_ITP_rCommunicationState.afCommunicationReq[ wIsoTpChannel] = FALSE;
        #endif
      }
    }
    break;

    case CSM_C_EXIT_OFF:
    {
      for( wIsoTpChannel = 0; wIsoTpChannel < CSM_C_ISO_TP_MAX_CONNECTIONS; wIsoTpChannel++)
      {
        #if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
          csm_itp_arCallbackFkt[wIsoTpChannel].rCallback.pfvCommunicationCon = NULL;
          csm_itp_arCallbackFkt[wIsoTpChannel].rCallback.pfvCommunicationInd = NULL;
          csm_itp_arCallbackFkt[wIsoTpChannel].rCallback.pfvDataCon = NULL;
          csm_itp_arCallbackFkt[wIsoTpChannel].rCallback.pfvDataInd = NULL;
          #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
            csm_itp_arCallbackFkt[wIsoTpChannel].rCallback.pfvDataIndFF = NULL;
          #endif /*CSM_S_ITP_FF_INDICATION_AVAILABLE*/
          #if (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
            csm_itp_arCallbackFkt[wIsoTpChannel].rCallback.pfvDataErrorInd = NULL;
          #endif /*CSM_S_ITP_ERROR_INDICATION_AVAILABLE*/

          #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
            csm_itp_arCallbackFkt[wIsoTpChannel].pvHandle = NULL;
          #endif /* CSM_S_USE_CALLBACK_WITH_HANDLE */
        #endif
      }
    }
    break;

    default:
    {
      return_value = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_UNKNOWN_STATE);
    }
    break;
  }

  return( return_value);
}


/*******************************************************************************
 * function    csm_vGetItpAdrParameter
 * \doxydocu
 * \brief      This function copies the ISO-TP adressing parameter protocol
 *             type and address field to the first and second function parameter.
 *             This function is used in the upstream task.
 *
 * \param[out] pdwProtocollType
 *              see IPCA-Context
 *
 * \param[out] prAddrField
 *              ITP address field
 *
 * \param[in]  bIsoTpChannel
 *              ISO-TP channel index
 *
 * \return     VOID
 *
 * \access     all
 * \reentrant  yes
 *
 * \lastreview 11.05.17 Borck
 *
 * \history_begin
 * 25.07.08 VTeam-Zurmhl
 * - Parameter checking added
 *
 * 25.07.02 CM-DI/ESA3-K"ohler
 * Protocol type is now in csm_rIsoTpConnection table.
 *
 * 16.09.02 CM-DI/EHS3-K"ohler
 * Initial Revision.
 * \history_end
 ******************************************************************************/
VOID csm_vGetItpAdrParameter(DWORD * pdwProtocollType, tCSM_ADDR_ISO_TP * prAddrField, BYTE bIsoTpChannel)
{
  if (bIsoTpChannel < CSM_C_ISO_TP_MAX_CONNECTIONS)
  {
    if ((CSM_C_ITP_AMODE_NORMAL       == csm_rIsoTpConnection[bIsoTpChannel].bAddrMode) ||
        (CSM_C_ITP_AMODE_NORMAL_FIXED == csm_rIsoTpConnection[bIsoTpChannel].bAddrMode)
        #if (CSM_S_ITP_EXTENDED_ADDRESSING == CSM_C_F_ON)
        || (CSM_C_ITP_AMODE_EXTENDED == csm_rIsoTpConnection[bIsoTpChannel].bAddrMode)
        #endif
       )
    {
      *pdwProtocollType         = csm_rIsoTpConnection[bIsoTpChannel].dwProtocolType;
       prAddrField->bLocalComp  = csm_rIsoTpConnection[bIsoTpChannel].bLocalComp;
       prAddrField->bRemoteComp = csm_rIsoTpConnection[bIsoTpChannel].bRemoteComp;
    }
    #ifdef CSM_C_PTYPE_OPEL_USDT_ISO_EXT_MUX
    else if (CSM_C_ITP_AMODE_EXTENDED_MUX == csm_rIsoTpConnection[bIsoTpChannel].bAddrMode)
    {
      *pdwProtocollType         = CSM_C_PTYPE_OPEL_USDT_ISO_EXT_MUX;
       prAddrField->bLocalComp  = CSM_C_ISO_TP_LC_EHU_FUDE;                      /* Opel special, two channels between EHU and DIS */
       prAddrField->bRemoteComp = CNP_rItpAssRxBuf[bIsoTpChannel].rMsg.bSender;
    }
    #endif // CSM_C_PTYPE_OPEL_USDT_ISO_EXT_MUX
    else
    {
      CSM_vAssert(FALSE); /*lint !e506 PQM_authorized_multi_226 */ /* Invalid addressing mode shall cause an assert - 21.01.2011 krv2hi */
    }
  }
  else
  {
    CSM_vAssert(FALSE); /*lint !e506 PQM_authorized_multi_226 */ /* Invalid channel shall cause an assert - 21.01.2011 krv2hi */
  }
}

/*******************************************************************************
 * function    csm_bGetItpNATelIndex
 * \doxydocu
 * \brief      This function checks if the passed address field fits to a
 *             configurated ISO-TP normal addressing channel and returns the
 *             telegram/channel index.
 *
 * \param[in]  pvAddressField
 *             pointer to tCSM_ADDR_ISO_TP address field
 *
 * \return     bIsoTpChannel    channel index
 *              0 <= bIsoTpChannel < CSM_C_ISO_TP_MAX_CONNECTIONS : valid index
 *              else channel not found
 *
 * \access     all
 * \reentrant  yes
 *
 * \lastreview 11.05.17 Borck
 *
 * \history_begin
 * 10.10.06 CM-DI/ESA3-Zurmhl
 * pvAddressField changed: PVOID-> void*.
 *
 * 12.02.03 CM-DI/ESU3-K"ohler
 * pvAddressField changed to pointer to const.
 *
 * 02.12.02 CM-DI/EHS3-K"ohler
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#ifdef CSM_C_PTYPE_USED_STD_USDT_ISO
// should work with other ISO-TP (normal addressing) protocol types if they use the same address field
BYTE csm_bGetItpNATelIndex(const void* pvAddressField)
{
  tCSM_ADDR_ISO_TP * pUsdtIsoAddrField = (tCSM_ADDR_ISO_TP *) pvAddressField;
  BYTE bIsoTpChannel = 0;

  while (bIsoTpChannel < CSM_C_ISO_TP_MAX_CONNECTIONS)
  {
    if ((pUsdtIsoAddrField->bLocalComp  == csm_rIsoTpConnection[bIsoTpChannel].bLocalComp) &&
        (pUsdtIsoAddrField->bRemoteComp == csm_rIsoTpConnection[bIsoTpChannel].bRemoteComp))
    {
      break;  // channel found, leave loop
    }
    else
    {
      bIsoTpChannel++;
    }
  }

  return bIsoTpChannel;  // CSM_C_ISO_TP_MAX_CONNECTIONS -> no channel found
}
#endif // CSM_C_PTYPE_USED_STD_USDT_ISO

/*******************************************************************************
 * function    csm_bGetItpExtTelIndex
 * \doxydocu
 * \brief      This function checks if the passed address field fits to a
 *             configurated ISO-TP extended addressing channel and returns the
 *             telegram/channel index.
 *
 * \param[in]  pvAddressField
 *             pointer to tCSM_ADDR_ISO_TP address field
 *
 * \return     bIsoTpChannel    channel index
 *              0 <= bIsoTpChannel < CSM_C_ISO_TP_MAX_CONNECTIONS : valid index
 *              else channel not found
 *
 *
 * \access     all
 * \reentrant  yes
 *
 * \lastreview  10.10.06 CM-DI/ESA3-Zurmhl
 *
 * \history_begin
 * 10.10.06 CM-DI/ESA3-Zurmhl
 * pvAddressField changed: PVOID-> void*.
 *
 * 14.02.03 CM-DI/ESU3-Pistoor
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#ifdef CSM_C_PTYPE_FIAT_USDT_ISO_EXT
// should work with other ISO-TP extended multiplexed addressing protocol types if they use the same address field
BYTE csm_bGetItpExtTelIndex(const void* pvAddressField)
{
  tCSM_ADDR_ISO_TP * pUsdtIsoAddrField = (tCSM_ADDR_ISO_TP *) pvAddressField;
  BYTE bIsoTpChannel = 0;

  while (bIsoTpChannel < CSM_C_ISO_TP_MAX_CONNECTIONS)
  {
    if (pUsdtIsoAddrField->bLocalComp ==
        csm_rIsoTpConnection[bIsoTpChannel].bLocalComp)
      /* bLocalComp has to define clearly the channel, bRemoteComp defines
       * a variable receiver. */
    {
      break;  /* channel found, leave loop */
    }
    else
    {
      bIsoTpChannel++;
    }
  } /* end while bIsoTpChannel */

  return (bIsoTpChannel);  /* CSM_C_ISO_TP_MAX_CONNECTIONS -> no channel found */
} /* csm_bGetItpExtTelIndex() */
#endif /* CSM_C_PTYPE_OPEL_USDT_ISO_EXT_MUX */


#ifdef CSM_C_PTYPE_OPEL_USDT_ISO_EXT_MUX /* should work with other ISO-TP
                                  * extended multiplexed addressing) protocol
                                  * types when they use the same address field */
/*******************************************************************************
 * function    csm_bGetItpExtMuxTelIndex
 * \doxydocu
 * \brief      This function checks if the passed address field fits to a
 *             configurated ISO-TP extended multiplex addressing channel and
 *             returns the telegram/channel index.
 *
 * \param[in]  pvAddressField
 *             pointer to tCSM_ADDR_ISO_TP address field
 *
 * \return     telegram/channel index
 *              0 < bTelIdx < CSM_C_ISO_TP_MAX_CONNECTIONS : valid index
 *              else channel not found
 *
 * \access     all
 * \reentrant  yes
 *
 * \lastreview 07.11.06 CM-DI/ESA3-Khler
 *
 * \history_begin
 * 10.10.06 CM-DI/ESA3-Zurmhl
 * pvAddressField changed: PVOID-> void*.
 *
 * 02.12.02 CM-DI/EHS3-K"ohler
 * Initial Revision.
 * \history_end
 ******************************************************************************/
BYTE csm_bGetItpExtMuxTelIndex(const void* pvAddressField)
{
tCSM_ADDR_ISO_TP * pUsdtIsoAddrField = (tCSM_ADDR_ISO_TP *) pvAddressField;
BYTE bIsoTpChannel = 0;

  while (bIsoTpChannel < CSM_C_ISO_TP_MAX_CONNECTIONS)
  {
    if (pUsdtIsoAddrField->bLocalComp ==
        csm_rIsoTpConnection[bIsoTpChannel].bLocalComp)
      /* bLocalComp has to define clearly the channel, bRemoteComp defines
       * a variable receiver. */
    {
      break;  /* channel found, leave loop */
    }
    else
    {
      bIsoTpChannel++;
    }
  } /* end while bIsoTpChannel */
  return (bIsoTpChannel);  /* CSM_C_ISO_TP_MAX_CONNECTIONS -> no channel found */
} /* csm_bGetItpExtMuxTelIndex() */
#endif /* CSM_C_PTYPE_OPEL_USDT_ISO_EXT_MUX */

/*******************************************************************************
 * function    csm_lCheckItpDataReqErrors
 * \doxydocu
 * \brief      This function checks if there are conditions which prevent that
 *             a data request for an ISO-TP channel can be accepted.
 *
 * \param[in]  bIsoTpChannel
 *              ISO-TP channel index
 *
 * \param[in]  pbData
 *              pointer to data which shall be send
 *
 * \param[in]  wDataLength
 *              number of data bytes which shall be send
 *
 * \return     Error value
 *
 * \access     all
 * \reentrant  yes
 *
 * \lastreview 11.05.17 Borck
 *
 * \history_begin
 * 03.04.08 VTeam-Khler
 * - DataCon callback pointer checked, too.
 *
 * 28.09.06 CM-DI/ESA3-Zurmhl
 * - lReturnValue initialized with new default warning
 *
 * 06.12.02 CM-DI/EHS3-Koehler
 * Initial Revision.
 * \history_end
 ******************************************************************************/
LONG csm_lCheckItpDataReqErrors(BYTE bIsoTpChannel, const BYTE * pbData, WORD wDataLength)
{
  // preset error
  LONG  lReturnValue = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_WARNING, CSM_E_INIT_WARNING);

  // check ISO-TP data request error cases
  if (bIsoTpChannel >= CSM_C_ISO_TP_MAX_CONNECTIONS)
  {
    // channel number invalid
    lReturnValue = CSM_M_MAKE_ERROR(CSM_C_ERR_WARNING, CSM_C_ERR_LOC_CSM, CSM_E_INVALID_CHANNEL);
  }

  #if (CSM_S_ICCC == CSM_C_F_ON)
  else if ((CSM_C_CONNECTED != csm_rIsoTpConnectionAct[bIsoTpChannel].bfConnectionState))
  {
    // not connected
    lReturnValue = CSM_M_MAKE_ERROR(CSM_C_ERR_WARNING, CSM_C_ERR_LOC_CSM, CSM_E_NO_CONNECTION);
  }
  #endif // CSM_S_ICCC

  else if (!csm_rIsoTpConnectionAct[bIsoTpChannel].fTxChannelFree)
  {
    // already in TX state
    lReturnValue = CSM_M_MAKE_ERROR(CSM_C_ERR_WARNING, CSM_C_ERR_LOC_CSM, CSM_E_STACK_BUSY);
  }
  else if (wDataLength > csm_rIsoTpConnection[bIsoTpChannel].wMaxTxLen)
  {
    // too many data
    lReturnValue = CSM_M_MAKE_ERROR(CSM_C_ERR_WARNING, CSM_C_ERR_LOC_CSM, CSM_E_DATA_LENGTH_EXCEEDED);
  }
  else if (pbData == NULL)
  {
    // invalid data pointer
    lReturnValue = CSM_M_MAKE_ERROR(CSM_C_ERR_WARNING, CSM_C_ERR_LOC_CSM, CSM_E_NULL_POINTER);
  }
  else if (NULL == csm_itp_arCallbackFkt[bIsoTpChannel].rCallback.pfvDataCon)
  {
    // callback for DataCon missing
    lReturnValue = CSM_M_MAKE_ERROR(CSM_C_ERR_WARNING, CSM_C_ERR_LOC_CSM, CSM_E_CALLBACK_MISSING);
  }
  else
  {
    lReturnValue = CSM_C_NO_ERROR;
  }

  return lReturnValue;
}

//*******************************************************************************
//* function implementation (scope: CAN stack and operating system)
//*******************************************************************************

/*******************************************************************************
 * function    CSM_ITP_vCNPDataInd()
 * \doxydocu
 * \brief      ISO TP function to indicate the application that a
 *             message was received.
 *
 * \param[in] bIsoTpChannel
 *              CAN stack internal channel Index.
 *
 * \param[in]  pbData
 *              Pointer to data which have been received
 *
 * \param[in]  wDataLength
 *              Number of data bytes
 *
 * \return     VOID
 *
 * \access     CAN task
 * \reentrant  no
 *
 * \lastreview 25.07.08 VTeam-Zurmhl
 *
 * \history_begin
 *
 * 10.05.17 Borck
 * - this function is not required for a REMOTE stack
 *
 * 08.04.05 CM-DI/ESA3-Koehler
 * - Callback function array used instead of ITP ROM data struct.
 *
 * 26.01.05 CM-DI/ESA3-Koehler
 * - New naming conventions.
 * - Parameter bBus removed.
 *
 * 07.01.05 CM-DI/ESA3-Koehler
 * CSM_CNP_vIsoTp_DataInd() renamed CSM_ITP_vCNP_DataInd().
 *
 * 14.03.03 CM-DI/ESU3-Koehler
 * Workaround CSM_S_DIAGNOSIS_DATA_IND_WITHOUT_UPSTREAM_TASK added.
 *
 * CM-DI/EHS3-Koehler
 * Call of UpstreamTask.
 *
 * 06.09.02 CM-DI/EHS3-Koehler
 * Extended Addressing added
 *
 * ??.??.02 CM-DI/EHS3-Koehler
 * Initial Revision. Only dummy implementation for Opel A-Muster.
 * \history_end
 ******************************************************************************/
#if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
VOID CSM_ITP_vCNPDataInd(BYTE bIsoTpChannel, BYTE * pbData, WORD wDataLength)
{
  // check for valid access index
  if (bIsoTpChannel < CSM_C_ISO_TP_MAX_CONNECTIONS)
  {
    CSM_M_SET_ITP_INDICATION_FLAG(bIsoTpChannel);
    csm_vTriggerCsmUpstreamTask(CSM_C_IND_REASON_ITP_DATA_IND);
  }
}/*lint !e715 !e818 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally , PQM_authorized_multi_123. Reason: Interface not defined as const*/ /* end CSM_ITP_vCNPDataInd() */
#endif


/*******************************************************************************
 * function    CSM_ITP_vPROXYTriggerCommunicationCon
 * \doxydocu
 * \brief      Callback function from PROXY to trigger a communication confirmation
 *
 * \param[in]  bIsoTpChannel
 *                 Data
 *
 * \return     void
 *
 * \access     CSM task level
 * \reentrant  no, but only call from interrupt
 *
 * \lastreview
 *
 * \history_begin
 *
 * 23.08.13 Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
#if (CSM_S_REMOTE_STACK == CSM_C_F_ON)
void CSM_ITP_vPROXYTriggerCommunicationCon(BYTE bIsoTpChannel, BYTE bAction)
{
  #if (CSM_S_ITP_COMMUNICATION_FCT == CSM_C_F_ON)
  CSM_ITP_rCommunicationState.abActAction[ bIsoTpChannel] = bAction;
  CSM_ITP_rCommunicationState.afCommunicationCon[ bIsoTpChannel] = TRUE;
  csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_CNP_ITP_COMMUNICATION_CON);
  #endif
}

/*******************************************************************************
 * function    CSM_ITP_vPROXYTriggerCommunicationInd
 * \doxydocu
 * \brief      Callback function from PROXY to trigger a communication indication
 *
 * \param[in]  bIsoTpChannel
 *                 Data
 *
 * \return     void
 *
 * \access     CSM task level
 * \reentrant  no, but only call from interrupt
 *
 * \lastreview
 *
 * \history_begin
 *
 * 21.10.13 Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CSM_ITP_vPROXYTriggerCommunicationInd(BYTE bIsoTpChannel, BYTE bAction)
{
  #if (CSM_S_ITP_COMMUNICATION_FCT == CSM_C_F_ON)
  CSM_ITP_rCommunicationState.abActAction[ bIsoTpChannel] = bAction;
  CSM_ITP_rCommunicationState.afCommunicationInd[ bIsoTpChannel] = TRUE;
  csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_CNP_ITP_COMMUNICATION_IND);
  #endif
}
#endif // CSM_S_REMOTE_STACK is ON

/*******************************************************************************
 * function    CSM_ITP_vCNPDataCon
 * \doxydocu
 * \brief      ISO TP confirmation function to confirm the data request
 *             of the application.
 *
 * \param[in]  bIsoTpChannel
 *              CAN stack internal channel Index.
 *
 * \param[in]  bTransferResult
 *              Result of requested data transfer.
 *
 * \return     VOID
 *
 * \access     CAN task
 * \reentrant  no
 *
 * \lastreview 10.05.17 Borck
 *
 * \history_begin
 *
 * 10.05.17 Borck
 * - this function is required for Remote and Local Stack to compile !!!
 *
 * 08.04.05 CM-DI/ESA3-Looby
 * - Altered for Renault ASDT high level retry.
 *
 * 08.04.05 CM-DI/ESA3-Koehler
 * - Callback function array used instead of ITP ROM data struct.
 *
 * 26.01.05 CM-DI/ESA3-Koehler
 * - New naming conventions.
 * - Parameter bBus removed.
 *
 * 15.07.02 CM-DI/ESU3-Koehler
 * CSM_S_DIAGNOSIS_DATA_IND_WITHOUT_UPSTREAM_TASK added.
 *
 * 06.09.02 CM-DI/EHS3-Koehler
 * Extended Addressing added
 *
 * 29.05.02 CM-DI/EHS3-Koehler
 * Continuously usage of struct csm_rIsoTpConnection1[].
 *
 * ??.??.?? CM-DI/EHS3-Koehler
 * Initial Revision. Only dummy implementation for Opel A-Muster.
 * \history_end
 ******************************************************************************/
#if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
VOID CSM_ITP_vCNPDataCon(BYTE bIsoTpChannel, BYTE bTransferResult)
{
  // check for valid access index
  if (bIsoTpChannel < CSM_C_ISO_TP_MAX_CONNECTIONS)
  {
    CSM_M_SET_ITP_CONFIRMATION_FLAG(bIsoTpChannel);
    CSM_abItpConfirmationResults[bIsoTpChannel] = bTransferResult;
    csm_vTriggerCsmUpstreamTask(CSM_C_IND_REASON_ITP_DATA_CON);
  }
}
#endif



/*******************************************************************************
 * function    CSM_ITP_vCNPDataErrorInd
 * \doxydocu
 * \brief      ISO TP indication function to indicate the reception of a First Frame
 *             to the application.
 *
 * \param[in] bIsoTpChannel
 *              CAN stack internal channel Index.
 *
 * \return     VOID
 *
 * \access     CAN task
 * \reentrant  no
 *
 * \lastreview 09.05.11 Prhl
 *
 * \history_begin
 *
 * 23.11.08 CM-PJ/FO45-Pistoor
 * - Initial version
 *
 * \history_end
 ******************************************************************************/
#if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
#if (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
VOID CSM_ITP_vCNPDataErrorInd(BYTE bIsoTpChannel)
{
  // check for valid access index
  if (bIsoTpChannel < CSM_C_ISO_TP_MAX_CONNECTIONS)
  {
    CSM_M_SET_ITP_ERROR_INDICATION_FLAG(bIsoTpChannel);
    csm_vTriggerCsmUpstreamTask(CSM_C_IND_REASON_ITP_ERROR_IND);
  }
}
#endif // CSM_S_ITP_ERROR_INDICATION_AVAILABLE

 /*******************************************************************************
 * function    CSM_ITP_vCNPDataIndFF
 * \doxydocu
 * \brief      ISO TP indication function to indicate the reception of a First Frame
 *             to the application.
 *
 * \param[in] bIsoTpChannel
 *              CAN stack internal channel Index.
 *
 * \return     VOID
 *
 * \access     CAN task
 * \reentrant  no
 *
 * \lastreview 09.05.11 Prhl
 *
 * \history_begin
 *
 * 23.11.08 CM-PJ/FO45-Pistoor
 * - Initial version
 *
 * \history_end
 ******************************************************************************/
#if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
VOID CSM_ITP_vCNPDataIndFF(BYTE bIsoTpChannel)
{
  // check for valid access index
  if (bIsoTpChannel < CSM_C_ISO_TP_MAX_CONNECTIONS)
  {
    CSM_M_SET_ITP_INDICATION_FF_FLAG(bIsoTpChannel);
    csm_vTriggerCsmUpstreamTask(CSM_C_IND_REASON_ITP_DATA_IND_FF);
  }
}
#endif // CSM_S_ITP_FF_INDICATION_AVAILABLE

#endif // CSM_S_REMOTE_STACK is OFF

/*******************************************************************************
 * function    CSM_ITP_lApplCallbackInit
 * \doxydocu
 * \brief      Initialize application callback function
 *
 *             Initialize application  callback function.
 *             Please call this function only once for each channel

 * \see        CAN-SW Architecture, written by CM-DI/ESA3
 *
 * \param[in]  pvHandle
 *             pointer to handle. For future use! May be a deinit() function
 *
 * \param[in]  bBus
 *             CAN bus number
 *
 * \param[in]  dwProtocolType
 *             protocol type, see table 2 of CAN SW architecture
 *
 * \param[in]  pvAddressField
 *             addressing, dependent on protocol
 *
 * \param[in]  prCallBackFkt
 *             pointer to callback function
 *
 * \return     Error value
 *
 * \access     application
 * \reentrant  yes
 *
 *
 * \lastreview 13.02.09 VTeam-Zurmhl
 *                     18.04.08 VTeam-Prhl
 *
 * \history_begin
 * 29.08.08 VTeam-Brunotte
 * - macro CSM_S_USE_TKSE check added.
 * 20.07.07 VTeam-Khler
 * - Pending data indication is checked and retriggered after callback
 *   registration.
 *
 * 10.07.07 VTeam-Khler
 * - handle stored, too.
 *
 * 10.10.06 CM-DI/ESA3-Zurmhl
 * pvAddressField changed: PVOID-> void*.
 *
 * 28.09.06 CM-DI/ESA3-Zurmhl
 * - lReturnValue initialized with new default warning
 *
 * 31.08.06 CM-DI/ESA3-Khler
 * - Return value used, #ifdef CSM_C_PTYPE_USED_STD_USDT_ISO added.
 *
 * 14.07.06 CM-DI/ESA3-Prhl
 * add bBus
 *
 * 18.02.05 CM-DI/ESU3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
#if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
LONG CSM_ITP_lApplCallbackInit(void * pvHandle, BYTE bBus, DWORD dwProtocolType, const void * pvAddressField, tCSM_ISO_TP_USDT_APPL_CALLBACK * prCallBackFkt)
{
  LONG lRetVal = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_WARNING, CSM_E_INIT_WARNING);
  BYTE bIsoTpChannel;

  #ifdef CSM_C_PTYPE_USED_STD_USDT_ISO
    bIsoTpChannel = csm_bGetItpNATelIndex(pvAddressField);
  #else
    // ##ToDo: general solution for normal addressing and other connection protocol type to be checked
    bIsoTpChannel = CSM_C_ISO_TP_MAX_CONNECTIONS;
  #endif

  if( bIsoTpChannel < CSM_C_ISO_TP_MAX_CONNECTIONS)
  {
    #if (CSM_S_USE_TKSE == CSM_C_F_ON)
    CSM_M_GET_TASK_CONTEXT( &(csm_itp_arCallbackFkt[ bIsoTpChannel].rTaskInfo));
    #endif

    // Assert if any callback is already stored for this channel.
    // It is forbidden to overwrite an initialisation and we assume that
    // this is an error (an application initialises a wrong channel).
    if (prCallBackFkt->pfvCommunicationCon != NULL)
    {
      // only for funny diagnose software
      CSM_vAssert(csm_itp_arCallbackFkt[bIsoTpChannel].rCallback.pfvCommunicationCon == NULL);
    }

    if (prCallBackFkt->pfvCommunicationInd != NULL)
    {
      // only for funny diagnose software
      CSM_vAssert(csm_itp_arCallbackFkt[bIsoTpChannel].rCallback.pfvCommunicationInd == NULL);
    }

    if (prCallBackFkt->pfvDataCon != NULL)
    {
      // only for funny diagnose software
      CSM_vAssert(csm_itp_arCallbackFkt[bIsoTpChannel].rCallback.pfvDataCon == NULL);
    }

    if (prCallBackFkt->pfvDataInd != NULL)
    {
      // only for funny diagnose software
      CSM_vAssert(csm_itp_arCallbackFkt[bIsoTpChannel].rCallback.pfvDataInd == NULL);
    }

    #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
    {
      if (prCallBackFkt->pfvDataIndFF != NULL)
      {
        // only for funny diagnose software
        CSM_vAssert(csm_itp_arCallbackFkt[bIsoTpChannel].rCallback.pfvDataIndFF == NULL);
      }
    }
    #endif

    #if (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
    {
      if (prCallBackFkt->pfvDataErrorInd != NULL)
      {
        // only for funny diagnose software
        CSM_vAssert(csm_itp_arCallbackFkt[ bIsoTpChannel].rCallback.pfvDataErrorInd == NULL);
      }
    }
    #endif

    #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
    csm_itp_arCallbackFkt[bIsoTpChannel].pvHandle = pvHandle; /*lint !e539 */ /*  preprocessor indentation - 21.01.2011 krv2hi */
    #endif

    csm_itp_arCallbackFkt[bIsoTpChannel].rCallback.pfvCommunicationCon = prCallBackFkt->pfvCommunicationCon;
    csm_itp_arCallbackFkt[bIsoTpChannel].rCallback.pfvCommunicationInd = prCallBackFkt->pfvCommunicationInd;
    csm_itp_arCallbackFkt[bIsoTpChannel].rCallback.pfvDataCon = prCallBackFkt->pfvDataCon;
    csm_itp_arCallbackFkt[bIsoTpChannel].rCallback.pfvDataInd = prCallBackFkt->pfvDataInd;

    #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
    csm_itp_arCallbackFkt[bIsoTpChannel].rCallback.pfvDataIndFF = prCallBackFkt->pfvDataIndFF; /*lint !e539 */ /*  preprocessor indentation - 21.01.2011 krv2hi */
    #endif

    #if (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
    csm_itp_arCallbackFkt[bIsoTpChannel].rCallback.pfvDataErrorInd = prCallBackFkt->pfvDataErrorInd; /*lint !e539 */ /*  preprocessor indentation - 21.01.2011 krv2hi */
    #endif

    #ifdef CSM_C_IND_REASON_ITP_DATA_IND
    if ((CSM_M_IS_ITP_INDICATION_FLAG_SET(bIsoTpChannel)) &&
        (csm_itp_arCallbackFkt[bIsoTpChannel].rCallback.pfvDataInd != NULL))
    {
      // If indication flag is already stored (message already received)
      // and application has just registered a data indication function
      // then trigger data indication call via upstream task.
      CSM_M_SET_ITP_INDICATION_FLAG(bIsoTpChannel);
      csm_vTriggerCsmUpstreamTask(CSM_C_IND_REASON_ITP_DATA_IND);
    }
    #endif

    lRetVal = CSM_C_NO_ERROR;
  }
  else
  {
    lRetVal = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_INVALID_PARA);
    CSM_vAssert(FALSE); /*lint !e506 PQM_authorized_multi_226 */ /* Invalid channel shall cause an assert - 21.01.2011 krv2hi */
  }
  return(lRetVal);
}/*lint !e715 !e818 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally , PQM_authorized_multi_123. Reason: Interface not defined as const*/
#endif // CSM_S_CALLBACK_INIT_BY_APPLICATION

/*******************************************************************************
 * function    CSM_ITP_lGetCommunicationState
 * \doxydocu
 * \brief      The actual state of the "connection" can be
 *             checked with this function.
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \param[in]  bBus
 *              Bus number, only needed for "broadcast" protocol types
 *
 * \param[in]  dwProtocolType
 *              Protocol type, see corresponding table of CAN SW architecture
 *
 * \param[in]  pvAddressField
 *              Protocol dependend address field
 *
 * \param[out] pbConnectState
 *              CSM_C_CONNECTED     /  CSM_C_STACK_UP
 *              CSM_C_DISCONNECTED  /  CSM_C_STACK_DOWN
 *
 * \param[in]  wApplId
 *              Application Id, only needed for "broadcast" protocol types
 *
 * \return     Error value (wrong Parameter) if < CSM_C_NO_ERROR
 *             else no error
 *
 * \access     all
 * \reentrant  yes
 *
 * \lastreview 18.04.08 VTeam-Prhl
 *
 * \history_begin
 * 22.08.06 CM-DI/ESA3-Prhl
 * First version
 *
 * \history_end
 ******************************************************************************/
LONG CSM_ITP_lGetCommunicationState(
  BYTE  bBus,
  DWORD dwProtocolType,
  const void* pvAddressField,
  BYTE* pbConnectState,
  WORD  wApplId)
{
  LONG  lRet = CSM_C_NO_ERROR;

  BYTE bIsoTpChannel;

  CSM_M_PARAMETER_INTENTIONALLY_UNUSED(bBus);
  CSM_M_PARAMETER_INTENTIONALLY_UNUSED(wApplId);
  CSM_M_PARAMETER_INTENTIONALLY_UNUSED(dwProtocolType);

  #ifdef CSM_C_PTYPE_USED_STD_USDT_ISO

  bIsoTpChannel = csm_bGetItpNATelIndex(pvAddressField);

  #endif
  if( bIsoTpChannel < CSM_C_ISO_TP_MAX_CONNECTIONS)
  {
  #if (CSM_S_ITP_COMMUNICATION_FCT == CSM_C_F_ON)
    *pbConnectState = CSM_ITP_rCommunicationState.abActAction[ bIsoTpChannel];
  #endif
  }
  else
  {
    lRet = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_C_INVALID_PARA);
  }

  return (lRet);
}/*lint !e715 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally*//* CSM_ITP_lGetCommunicationState() */

/*******************************************************************************
 * function    CSM_ITP_lDataReq
 * \doxydocu
 * \brief      Request to send a transport protocol message over ISO-TP to CAN
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \param[in]  dwProtocolType
 *              Protocol type, see table 2 of CAn SW architecture.
 *
 * \param[in]  pvAddressField
 *              Protocol dependend address field.
 *
 * \param[in]  pbData
 *              Pointer to data which shall be send.
 *
 * \param[in]  wDataLength
 *              Number of data bytes which shall be send.
 *
 * \return     Error/no error value
 *
 * \access     application
 * \reentrant  only for different channels
 *
 * \lastreview 18.04.08 VTeam-Prhl
 *
 * \history_begin
 * 25.01.08 VTeam-Prhl
 * some code moved to cnp_itp.c
 *
 * 10.10.06 CM-DI/ESA3-Zurmhl
 * pvAddressField changed: PVOID-> void*.
 *
 * 28.09.06 CM-DI/ESA3-Zurmhl
 * - lReturnValue initialized with new default warning
 *
 * 18.02.05 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
LONG CSM_ITP_lDataReq(DWORD dwProtocolType, const void * pvAddressField, const BYTE * pbData, WORD wDataLength)
{
  // Default return value Error
  LONG lReturnValue;
  BYTE bIsoTpChannel;
 #ifdef CSM_C_PTYPE_USED_STD_USDT_ISO
  // get corresponding channel
  bIsoTpChannel = csm_bGetItpNATelIndex(pvAddressField);
 #endif
  // check some parameters
  lReturnValue = csm_lCheckItpDataReqErrors(bIsoTpChannel, pbData, wDataLength);

  /* no error found, call ITP module, it stores the data and continues via downstream task */
  if (CSM_C_NO_ERROR == lReturnValue)
  {
    #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
    lReturnValue = CNP_ITP_lDataReq(bIsoTpChannel, pbData, wDataLength);
    #else
    lReturnValue = CSM_PROXY_lDataReq(dwProtocolType, pvAddressField, pbData, wDataLength);
    #endif
  }
  return lReturnValue;
}/*lint !e715 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally*/ /* end CSM_ITP_lDataReq() */


#if (CSM_S_ITP_COMMUNICATION_FCT == CSM_C_F_ON)
/*******************************************************************************
 * function    CSM_ITP_lCommunicationReq
 * \doxydocu
 * \brief      Requests the CAN bus for ISO-TP communication.
 *
 * \see        CAN-SW Architecture, written by V-Team
 *
 * \param[in]  bBus
 *             bus number
 *
 * \param[in]  dwProtocolType
 *             protocol type, see table 2 of CAn SW architecture
 *
 * \param[in]  *pvAddressField
 *             addressing, dependent on protocol
 *
 * \param[in]  bAction
 *             Allowed values: CSM_C_STACK_UP, CSM_C_STACK_DOWN
 *
 * \param[in]  wApplID (I)
 *               Appl Id
 *
 * \return     Error value
 *
 * \access     application
 * \reentrant  no, but the you can call it with diffrend wApplID without a problem
 *
 * \lastreview 24.04.07 VTeam-Prhl
 *
 * \history_begin
 *
 * 10.10.06 CM-DI/ESA3-Zurmhl
 * pvAddressField changed: PVOID-> void*.
 *
 * 28.09.06 CM-DI/ESA3-Zurmhl
 * - lReturnValue initialized with new default warning
 *
 * 31.08.06 CM-DI/ESA3-Khler
 * - Return value used, #ifdef CSM_C_PTYPE_USED_STD_USDT_ISO added.
 *
 * 18.02.05 CM-DI/ESA3-Prhl
 * - Initial Revision.
 *
 * \history_end
 ******************************************************************************/
LONG CSM_ITP_lCommunicationReq(
  BYTE  bBus,
  DWORD dwProtocolType,
  const void* pvAddressField,
  BYTE  bAction,
  WORD wApplID)
{
  LONG  lReturnValue = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_WARNING, CSM_E_INIT_WARNING);
  BYTE bIsoTpChannel;

  /* Check for correct value of bAction: all but up is down  */
  bAction= CSM_C_STACK_UP == bAction ? CSM_C_STACK_UP : CSM_C_STACK_DOWN;

  bIsoTpChannel =
    #ifdef CSM_C_PTYPE_USED_STD_USDT_ISO
      csm_bGetItpNATelIndex(pvAddressField);
    #else
      /* \todo general solution for normal addressing and other connection */
      CSM_C_ISO_TP_MAX_CONNECTIONS;
    #endif

  if( bIsoTpChannel < CSM_C_ISO_TP_MAX_CONNECTIONS)
  {
    if( CSM_ITP_rCommunicationState.abReqAction[ bIsoTpChannel] != bAction)
    {/* Change of ReqAction */
      CSM_ITP_rCommunicationState.abReqAction[ bIsoTpChannel] = bAction;
      CSM_ITP_rCommunicationState.afCommunicationReq[ bIsoTpChannel] = TRUE;
      csm_vTriggerCsmDownstreamTask( CSM_C_DOWN_REASON_CNP_ITP_COMMUNICATION_REQ);

      lReturnValue = CSM_C_NO_ERROR;
    }
    else
    {/* Trigger Communication Confirmation directly */
      CSM_ITP_rCommunicationState.afCommunicationCon[ bIsoTpChannel] = TRUE;
      csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_CNP_ITP_COMMUNICATION_CON);

      lReturnValue = CSM_C_NO_ERROR;
    }
  }
  else
  {
    lReturnValue = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_INVALID_PARA);
  }

  return lReturnValue;
}/*lint !e715 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally */ /* end CSM_ITP_lCommunicationReq() */


/*******************************************************************************
 * function    CSM_ITP_fCheckCommunicationReq
 * \doxydocu
 * \brief      Calculate Communication request.
 *
 *             Calculate Communication request.
 *             This function is called to calculate the CommunicationReq state.
 *
 * \param[in]  bBus
 *
 * \return     BOOL
 *               FALSE: the bus is not needed by CSM_ITP
 *               TRUE: the bus is neened by CSM_ITP
 *
 * \access     CAN-Task
 * \reentrant  yes, but only call by CAN-Task
 *
 * \lastreview  06.11.06 CM-DI/ESA3-Khler
 *
 * \history_begin
 * 06.11.06 CM-DI/ESA3-Khler
 * Bus check added.
 * 18.02.05 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
BOOL
CSM_ITP_fCheckCommunicationReq(
  BYTE bBus)
{
  WORD wConnection;

  for( wConnection = 0; wConnection < CSM_C_ISO_TP_MAX_CONNECTIONS; wConnection++)
  {
    if((bBus == csm_rIsoTpConnection[wConnection].bBus) &&   /* connection belongs to the requested bus */
       (CSM_ITP_rCommunicationState.abReqAction[wConnection] != CSM_C_STACK_DOWN))
    {
      return( TRUE); /* CSM_ITP needs bus access */
    }
  }
  return( FALSE); /* CSM_ITP does'nt need bus access (any longer) */
}


/*******************************************************************************
 * function    CSM_ITP_vEndCommunicationRequests
 * \doxydocu
 * \brief      End all Communication requests.
 *
 *             End all Communication requests.
 *             This function is called to end all active requests.
 *             Requests are not disabled afterwards
 *             (and can successfully be requested again).
 *
 * \param[in]  none
 *
 * \return     none
 *
 * \access     CAN-Task
 * \reentrant  yes, but only call by CAN-Task
 *
 * \lastreview 09.05.11 Prhl
 *
 * \history_begin
 * 12.08.09 CM-DI/VTeam-Franke
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void
CSM_ITP_vEndCommunicationRequests(void)
{
  WORD bIsoTpChannel;
  BOOL fDisconnectNecessary = FALSE;

  /* for all channels: */
  for( bIsoTpChannel = 0; bIsoTpChannel < CSM_C_ISO_TP_MAX_CONNECTIONS; bIsoTpChannel++)
  {
    /* if connection has active request: */
    if (CSM_ITP_rCommunicationState.abReqAction[bIsoTpChannel] != CSM_C_STACK_DOWN)
    {
      /* disconnect channel: */
      CSM_ITP_rCommunicationState.afCommunicationReq[bIsoTpChannel] = TRUE;
      CSM_ITP_rCommunicationState.abReqAction[bIsoTpChannel] = CSM_C_STACK_DOWN;

      fDisconnectNecessary = TRUE;
    } /* if user has active request */
  } /* for all connections */

  if (fDisconnectNecessary)
  {
    csm_vTriggerCsmDownstreamTask(CSM_C_DOWN_REASON_CNP_ITP_COMMUNICATION_REQ);
  }
} /* CSM_ITP_vEndCommunicationRequests */

/*******************************************************************************
 * function    CSM_ITP_vProcessCommunicationReq
 * \doxydocu
 * \brief      Handle communication request.
 *
 *             Handle communication request.
 *             This function is call in downstream-task after trigger with CSM_C_DOWN_REASON_CSM_ITP_COMMUNICATION_REQ.
 *
 * \return     VOID
 *
 * \access     CAN-Task
 * \reentrant  no, but only call by CAN-Task
 *
 * \lastreview 18.04.08 VTeam-Prhl
 *
 * \history_begin
 * 27.10.06 CM-DI/ESA3-Looby
 * Enhanced For Renault ASDT.
 *
 * 18.02.05 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
VOID
CSM_ITP_vProcessCommunicationReq(
  VOID)
{
  WORD wCount;
  BOOL fStateChanged = FALSE;

  for( wCount = 0; wCount < CSM_C_ISO_TP_MAX_CONNECTIONS; wCount++)
  {
    if( CSM_ITP_rCommunicationState.afCommunicationReq[ wCount] == TRUE)
    {
      CSM_ITP_rCommunicationState.afCommunicationReq[ wCount] = FALSE;

      #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
      #if ( CSM_S_OSEK_NM == CSM_C_F_ON )
        #if ( CNM_OSEK_S_PAYLOAD == CSM_C_F_ON )
          if( CSM_ITP_rCommunicationState.abReqAction[ wCount] == CSM_C_DISCONNECTED)
          {
            CNM_OSEK_lClrPayload( csm_rIsoTpConnection[ wCount].bBus, csm_rIsoTpConnection[ wCount].dwNmPayload);
          }
          else
          {
            CNM_OSEK_lSetPayload( csm_rIsoTpConnection[ wCount].bBus, csm_rIsoTpConnection[ wCount].dwNmPayload);
          }
        #endif /* CNM_OSEK_S_PAYLOAD == CSM_C_F_ON */
      #endif /* CSM_S_OSEK_NM == CSM_C_F_ON */
      #endif /* CSM_S_CNM_AVAILABLE */
      {
        /* hier knnten Funktionen stehen die auf CAN-Task level nach einem CommunicationReq gemacht werden sollen */

        /* Set Communication Confirmation */
        CSM_ITP_rCommunicationState.afCommunicationCon[ wCount] = TRUE;

        fStateChanged = TRUE;
      }
    }
  }

  if (fStateChanged == TRUE)
  {
    csm_vTriggerCsmDownstreamTask( CSM_C_DOWN_REASON_CSM_BUS_STATE_CHANGED);
    csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_CNP_ITP_COMMUNICATION_CON);
  }
}

/*******************************************************************************
 * function    CSM_ITP_vProcessCommunicationCon
 * \doxydocu
 * \brief      Handle communication confimation.
 *
 *             Handle communication confimation.
 *             This function is called in upstream-task after trigger
 *             with CSM_C_IND_REASON_CSM_ITP_COMMUNICATION_CON.
 *
 * \return     VOID
 *
 * \access     Upstream-Task
 * \reentrant  no, but only call by Upstream-Task
 *
 * \lastreview 01.06.05 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 01.06.05 CM-DI/ESA3-Prhl
 * all call back parameter now used from configuration struct
 *
 * 18.02.05 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
VOID CSM_ITP_vProcessCommunicationCon(VOID)
{
  WORD wCount;

  for( wCount = 0; wCount < CSM_C_ISO_TP_MAX_CONNECTIONS; wCount++)
  {
    if( CSM_ITP_rCommunicationState.afCommunicationCon[ wCount] == TRUE)
    {
      CSM_ITP_rCommunicationState.afCommunicationCon[ wCount] = FALSE;

      if( csm_itp_arCallbackFkt[ wCount].rCallback.pfvCommunicationCon != NULL)
      {
        tCSM_ADDR_ISO_TP rAddressField;

        rAddressField.bLocalComp = csm_rIsoTpConnection[ wCount].bLocalComp;
        rAddressField.bRemoteComp = csm_rIsoTpConnection[ wCount].bRemoteComp;

        #if (CSM_S_TRACE == CSM_C_F_ON)
          #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
            #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
            ETG_TRACE_USR1(("CommunicationCon       --> Bus: %x Protocol: %x Addr: %02x%02x Action: %02x ApplID: %04x",
            ETG_CENUM(tCSM_BUS_NUMBER,      (tU8)  csm_rIsoTpConnection[ wCount].bBus),
            ETG_CENUM(tCSM_PROTOCOL_TYPE,   (tU32) csm_rIsoTpConnection[ wCount].dwProtocolType),
            rAddressField.bLocalComp,
            rAddressField.bRemoteComp,
            CSM_ITP_rCommunicationState.abActAction[ wCount],
            CSM_C_APPL_ID_DEFAULT ));
            #endif
          #else
        {
          BYTE abData[ 10];
          abData[ 0] = csm_rIsoTpConnection[ wCount].bBus;
          abData[ 1] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD( csm_rIsoTpConnection[ wCount].dwProtocolType));
          abData[ 2] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD( csm_rIsoTpConnection[ wCount].dwProtocolType));
          abData[ 3] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD( csm_rIsoTpConnection[ wCount].dwProtocolType));
          abData[ 4] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD( csm_rIsoTpConnection[ wCount].dwProtocolType));

          abData[ 5] = rAddressField.bLocalComp;
          abData[ 6] = rAddressField.bRemoteComp;

          abData[ 7] = CSM_ITP_rCommunicationState.abActAction[ wCount];
          abData[ 8] = CSM_M_GET_HIBYTE( CSM_C_APPL_ID_DEFAULT);
          abData[ 9] = CSM_M_GET_LOBYTE( CSM_C_APPL_ID_DEFAULT);

          CSM_M_TRACEOUT( CSM_C_TRACEOUT_LEVEL_INTERFACE,  CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_COMMUNICATION_CON, abData ,  10);
        }
          #endif
        #endif /* CSM_S_TRACE */

        #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
          CSM_M_SET_TASK_CONTEXT( &(csm_itp_arCallbackFkt[ wCount].rTaskInfo),
                                  csm_itp_arCallbackFkt[ wCount].rCallback.pfvCommunicationCon(
                                                           csm_itp_arCallbackFkt[ wCount].pvHandle,
                                                           csm_rIsoTpConnection[ wCount].bBus,
                                                           csm_rIsoTpConnection[ wCount].dwProtocolType,
                                                           (const VOID*) &rAddressField,
                                                           CSM_ITP_rCommunicationState.abActAction[ wCount],
                                                           CSM_C_APPL_ID_DEFAULT) );
        #else /* CSM_S_USE_CALLBACK_WITH_HANDLE */
          CSM_M_SET_TASK_CONTEXT( &(csm_itp_arCallbackFkt[ wCount].rTaskInfo),
                                  csm_itp_arCallbackFkt[ wCount].rCallback.pfvCommunicationCon(
                                                           csm_rIsoTpConnection[ wCount].bBus,
                                                           csm_rIsoTpConnection[ wCount].dwProtocolType,
                                                           (const VOID*) &rAddressField,
                                                           CSM_ITP_rCommunicationState.abActAction[ wCount],
                                                           CSM_C_APPL_ID_DEFAULT) );
        #endif /* CSM_S_USE_CALLBACK_WITH_HANDLE */
      }
    }
  }
}  /* CSM_ITP_vProcessCommunicationCon() */

/*******************************************************************************
 * function    CSM_ITP_vProcessCommunicationInd
 * \doxydocu
 * \brief      Handle communication indication.
 *
 *             Handle communication indication.
 *             This function is called in upstream-task after trigger
 *             with CSM_C_IND_REASON_CSM_ITP_COMMUNICATION_IND.
 *
 * \return     VOID
 *
 * \access     Upstream-Task
 * \reentrant  no, but only call by Upstream-Task
 *
 * \lastreview 01.06.05 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 01.06.05 CM-DI/ESA3-Prhl
 * all call back parameter now used from configuration struct
 *
 * 18.02.05 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
VOID CSM_ITP_vProcessCommunicationInd(VOID)
{
  WORD wCount;

  for( wCount = 0; wCount < CSM_C_ISO_TP_MAX_CONNECTIONS; wCount++)
  {
    if( CSM_ITP_rCommunicationState.afCommunicationInd[ wCount] == TRUE)
    {
      CSM_ITP_rCommunicationState.afCommunicationInd[ wCount] = FALSE;

      if( csm_itp_arCallbackFkt[ wCount].rCallback.pfvCommunicationInd != NULL)
      {
        tCSM_ADDR_ISO_TP rAddressField;

        rAddressField.bLocalComp = csm_rIsoTpConnection[ wCount].bLocalComp;
        rAddressField.bRemoteComp = csm_rIsoTpConnection[ wCount].bRemoteComp;

        #if (CSM_S_TRACE == CSM_C_F_ON)
          #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
            #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
            ETG_TRACE_USR1(("CommunicationInd       --> Bus: %x Protocol: %x Addr: %02x%02x Action: %02x ApplID: %04x",
            ETG_CENUM(tCSM_BUS_NUMBER,      (tU8)  csm_rIsoTpConnection[ wCount].bBus),
            ETG_CENUM(tCSM_PROTOCOL_TYPE,   (tU32) csm_rIsoTpConnection[ wCount].dwProtocolType),
            rAddressField.bLocalComp,
            rAddressField.bRemoteComp,
            CSM_ITP_rCommunicationState.abActAction[ wCount],
            CSM_C_APPL_ID_DEFAULT ));
            #endif
          #else
        {
          BYTE abData[ 10];
          abData[ 0] = csm_rIsoTpConnection[ wCount].bBus;
          abData[ 1] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD( csm_rIsoTpConnection[ wCount].dwProtocolType));
          abData[ 2] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD( csm_rIsoTpConnection[ wCount].dwProtocolType));
          abData[ 3] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD( csm_rIsoTpConnection[ wCount].dwProtocolType));
          abData[ 4] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD( csm_rIsoTpConnection[ wCount].dwProtocolType));

          abData[ 5] = rAddressField.bLocalComp;
          abData[ 6] = rAddressField.bRemoteComp;

          abData[ 7] = CSM_ITP_rCommunicationState.abActAction[ wCount];
          abData[ 8] = CSM_M_GET_HIBYTE( CSM_C_APPL_ID_DEFAULT);
          abData[ 9] = CSM_M_GET_LOBYTE( CSM_C_APPL_ID_DEFAULT);

          CSM_M_TRACEOUT( CSM_C_TRACEOUT_LEVEL_INTERFACE,  CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_COMMUNICATION_IND, abData ,  10);
        }
          #endif
        #endif /* CSM_S_TRACE */

        #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
          CSM_M_SET_TASK_CONTEXT( &(csm_itp_arCallbackFkt[ wCount].rTaskInfo),
                                  csm_itp_arCallbackFkt[ wCount].rCallback.pfvCommunicationInd(
                                        csm_itp_arCallbackFkt[ wCount].pvHandle,
                                        csm_rIsoTpConnection[ wCount].bBus,
                                        csm_rIsoTpConnection[ wCount].dwProtocolType,
                                        (const VOID*) &rAddressField,
                                        CSM_ITP_rCommunicationState.abActAction[ wCount],
                                        CSM_C_APPL_ID_DEFAULT) );
        #else /* CSM_S_USE_CALLBACK_WITH_HANDLE */
          CSM_M_SET_TASK_CONTEXT( &(csm_itp_arCallbackFkt[ wCount].rTaskInfo),
                                  csm_itp_arCallbackFkt[ wCount].rCallback.pfvCommunicationInd(
                                        csm_rIsoTpConnection[ wCount].bBus,
                                        csm_rIsoTpConnection[ wCount].dwProtocolType,
                                        (const VOID*) &rAddressField,
                                        CSM_ITP_rCommunicationState.abActAction[ wCount],
                                        CSM_C_APPL_ID_DEFAULT) );
        #endif /* CSM_S_USE_CALLBACK_WITH_HANDLE */
      }
    }
  }
} /* end CSM_ITP_vProcessCommunicationInd() */
#endif /* CSM_S_ITP_COMMUNICATION_FCT */



/*******************************************************************************
 * function    CSM_ITP_vCSMBusStateInd
 * \doxydocu
 * \brief      Bus State indication callback.
 *
 *             This funktion is called after the bus state is changed.
 *
 * \param[in]  bBus
 *             Bus
 *
 * \param[in]  bCsmActState
 *             New bus state.
 *             possible values:
 *                CSM_C_STATE_BUS_DOWN, CSM_C_STATE_BUS_LISTEN, CSM_C_STATE_BUS_STARTUP,
 *                CSM_C_STATE_BUS_UP, CSM_C_STATE_BUS_SHUTDOWN
 *
 * \param[in]  bVN
 *             Bit field for virtual networks
 *
 * \return     VOID
 *
 * \access     interrupt
 * \reentrant  yes
 *
 * \lastreview  30.07.07 VTeam-Prhl
 *
 * \history_begin
 * 18.02.05 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
VOID CSM_ITP_vCSMBusStateInd(
  BYTE bBus,
  BYTE bCsmActState
  #if (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
    , BYTE bVN         /* Up to now not used. */
  #endif /* CSM_S_VIRTUAL_NETWORK */
  )
{
    WORD wCount;
    switch( bCsmActState)
    {
      case CSM_C_STATE_BUS_DOWN:
      {
        for( wCount = 0; wCount < CSM_C_ISO_TP_MAX_CONNECTIONS; wCount++)
        {
          if( bBus == csm_rIsoTpConnection[wCount].bBus)
          {
      /* connection belongs to the requested bus */
      #if (CSM_S_ITP_COMMUNICATION_FCT == CSM_C_F_ON)
            if( CSM_ITP_rCommunicationState.abActAction[ wCount] != CSM_C_STACK_DOWN)
            {
              CSM_ITP_rCommunicationState.abActAction[ wCount] = CSM_C_STACK_DOWN;
              CSM_ITP_rCommunicationState.afCommunicationInd[ wCount] = TRUE;
              csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_CNP_ITP_COMMUNICATION_IND);
            }
      #endif
          }
        }
      }
      break;
      case CSM_C_STATE_BUS_LISTEN:
      case CSM_C_STATE_BUS_STARTUP:
      case CSM_C_STATE_BUS_UP:
      case CSM_C_STATE_BUS_SHUTDOWN:
      {
        for( wCount = 0; wCount < CSM_C_ISO_TP_MAX_CONNECTIONS; wCount++)
        {
          if( bBus == csm_rIsoTpConnection[wCount].bBus)
          {/* connection belongs to the requested bus */
      #if (CSM_S_ITP_COMMUNICATION_FCT == CSM_C_F_ON)
            if( CSM_ITP_rCommunicationState.abActAction[ wCount] != CSM_C_STACK_UP)
            {
              CSM_ITP_rCommunicationState.abActAction[ wCount] = CSM_C_STACK_UP;
              CSM_ITP_rCommunicationState.afCommunicationInd[ wCount] = TRUE;
              csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_CNP_ITP_COMMUNICATION_IND);
            }
      #endif
          }
        }
      }
      break;
      default:
      {
        CSM_vAssert( FALSE); /*lint !e506 PQM_authorized_multi_226 */ /* Invalid bus state shall cause an assert - 21.01.2011 krv2hi */
      }
      break;
    }
  #if (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
    CSM_M_PARAMETER_INTENTIONALLY_UNUSED(bVN);
  #endif
}

/*******************************************************************************
 * function    CSM_ITP_vResetIndicationFlags
 * \doxydocu
 * \brief      reset all indication flags
 *
 * \param[in]  void
 *
 * \return     void
 *
 * \access     CAN task
 * \reentrant  no
 *
 * \lastreview 09.05.11 Prhl
 *
 * \history_begin
 *
 * 07.10.10 V-Team Franke
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CSM_ITP_vResetIndicationFlags(void)
{
  #ifdef CSM_C_IND_REASON_ITP_DATA_IND
    CSM_dwItpIndicationFlags    = 0x00000000;
  #endif
  #ifdef CSM_C_IND_REASON_ITP_DATA_CON
    CSM_dwItpConfirmationFlags  = 0x00000000;
    /*CSM_abItpConfirmationResults[CSM_C_ISO_TP_MAX_CONNECTIONS]; initialisation not necessary
     * value is added before the flag is set */
  #endif /* CSM_C_IND_REASON_ITP_DATA_CON */
  #ifdef CSM_C_IND_REASON_ITP_COMM_IND
    CSM_dwItpCommIndFlags       = 0x00000000;
  #endif
} /* CSM_ITP_vResetIndicationFlags */


#if (CSM_S_DEBUG == CSM_C_F_ON)
#define TP_DEBUG_LEVEL_X  1  /* dummy: Debug levels have to be defined later */
/*******************************************************************************
 * function    CSM_ITP_vCNPErrorInd
 * \doxydocu
 * \brief      !!! Presently only dummy implementation for testing purposes
 *             !!! to be adapted if used for specific project
 *
 * \param[in] bISOTpChannel
 *              CAN stack internal channel Index.
 *
 * \param[in]  bError
 *              CNP_ITP_ERR...
 *
 * \return     VOID
 *
 * \access     CAN-Task
 * \reentrant  No
 *
 * \lastreview  04.07.08 VTeam-Zurmhl
 *
 * \history_begin
 *
 * 31.08.04 VTeam
 *  - initial version
 * \history_end
 ******************************************************************************/
VOID  CSM_ITP_vCNPErrorInd (BYTE bIsoTpChannel, BYTE bError)
{   /* ESD debug output */
  char bDebString[] = {'C','N','P',' ','C','H','A','N','N','E','L',' ',' ',':',' '};

  bDebString[12] = '0' + bIsoTpChannel;


  CSM_vDebPrintf(TP_DEBUG_LEVEL_X, bDebString);
  switch (bError)
  {
    case CNP_ITP_C_ERR_WRONG_SN:
    {
      CSM_vDebPrintf(TP_DEBUG_LEVEL_X, "Wrong SN (Rx)\n");
      break;
    }
    case CNP_ITP_C_ERR_TIMEOUT_CR:
    {
      CSM_vDebPrintf(TP_DEBUG_LEVEL_X, "CF timeout (Rx)\n");
      break;
    }
    case CNP_ITP_C_ERR_APPL_BUF:
    {
      CSM_vDebPrintf(TP_DEBUG_LEVEL_X, "Appl. buffer full (RX)\n");
      break;
    }
    case CNP_ITP_C_ERR_UNEXP_PDU:
    {
      CSM_vDebPrintf(TP_DEBUG_LEVEL_X, "Unexpected PDU (RX/TX)\n");
      break;
    }
    case CNP_ITP_C_ERR_TIMEOUT_BS:
    {
      CSM_vDebPrintf(TP_DEBUG_LEVEL_X, "FC timeout (TX)\n");
      break;
    }
    case CNP_ITP_C_ERR_WFT_OVRN:
    {
      CSM_vDebPrintf(TP_DEBUG_LEVEL_X, "Too many WAIT frames (TX)\n");
      break;
    }
    case CNP_ITP_C_ERR_RX_MSG_TOO_LONG:
    {
      CSM_vDebPrintf(TP_DEBUG_LEVEL_X, "RX message to long for assembly buffer (RX)\n");
      break;
    }
    case CNP_ITP_C_ERR_TX_MSG_TOO_LONG:
    {
      CSM_vDebPrintf(TP_DEBUG_LEVEL_X, "TX message to long for Tx-buffer (TX)\n");
      break;
    }

    default:
    {
      break;
    }
  }

} /* end CSM_CNP_vIsoTp_ErrorInd() */
#endif /* CSM_S_DEBUG */



#endif /* CSM_S_DOXYGEN_HIDE_NON_API_FCT */
/*******************************************************************************
 * function implementation (scope: CAN stack #1 API)
 ******************************************************************************/


#undef CSM_ITP_SELF
#undef CSM_CAN_STACK
/*******************************************************************************
 * \doxydocu
 * \file          csm_itp.c
 * \brief         ISO-TP part of the CAN Stack Manager
 *
 * \see           CAN-SW Architektur, written by CM-DI/ESU3
 *
 * \sw_component  CAN stack
 * \project       Pool module
 * \path          /di_can/modules_swa/csm/src/
 *
 * \ingroup       CSM
 * \ingroup       CSM_ITP
 *
 * \authors       CM-DI/ESU3-K"ohler, -Zurm"uhl
 *
 * COPYRIGHT      (c) 2004 Blaupunkt GmbH
 *
 * \history_begin
 *
 * 31.08.04  main\1 CM-DI/ESA3-K"ohler
 * Initial revision.
 * Code moved from csm.c, doxygen comments added.
 *
 * 28.01.05  main\2 CM-DI/ESA3-K"ohler
 * - Include cnp_itp_i.h added.
 * - csm_vGetItpAdrParameter() not any longer static.
 * - New naming convention.
 * - Parameter bBus for CSM_ITP_vCNPDataInd(), CSM_ITP_vCNPDataCon()
 *   and CSM_ITP_vCNPErrorInd() removed.
 *
 * 24.02.05  main\3 CM-DI/ESA3-Prhl
 * - Communication requenst handling added
 *
 * 08.04.05  main\4  CM-DI/ESA3-K"ohler
 * - Function CSM_ITP_lApplCallbackInit() encapsulated with feature switch
 *   CSM_S_CALLBACK_INIT_BY_APPLICATION.
 * - Callback function array used instead of ITP ROM data struct.
 *
 * 08.04.05  main\5 CM-DI/ESA3-Zurmhl
 * - Callback function types renamed from vPFNTp to pfv
 *
 * 27.05.05  main\6 CM-DI/ESA3-Prhl
 * - Function CSM_ITP_lInit() added
 *
 * 20.06.05  main\7  CM-DI/ESA3-K"ohler
 * Initialization of ITP variables for upstream task indication moved from
 * CSM_lInit() to CSM_ITP_lInit().
 *
 * 07.07.05  main\8 CM-DI/ESA3-Zurmhl
 * - Revision of error codes (adaption to new conventions etc.)
 *
 * 19.09.05  \main\9 CM-DI/ESA3-Prhl
 * - add handle to all ...CallbackInit() funktions
 *
 * 04.04.06  \main\10 CM-DI/ESA3-Prhl
 * - add const
 *
 * 01.06.06  \main\11 CM-DI/ESA3-Prhl
 * - function CSM_ITP_vProcessCommunicationInd(), CSM_ITP_vProcessCommunicationCon():
 *       all call back parameter now used from configuration struct
 *
 * 02.06.06  \main\12 CM-DI/ESA3-Prhl
 * - comments corrected
 *
 * 14.06.06  \main\13 CM-DI/ESA3-Prhl
 * - virtual network support added
 *
 * 14.07.06  \main\14 CM-DI/ESA3-Prhl
 * - modify CSM_ITP_lApplCallbackInit()
 *   add bBus
 *
 * 19.07.06 \main\15 CM-DI/ESA3-Prhl
 * - modify feature CSM_S_USE_CALLBACK_WITH_HANDLE.
 *   add PVOID pvHandle to tCSM_ISO_TP_USDT_APPL_CALLBACK if feature CSM_S_USE_CALLBACK_WITH_HANDLE is active
 *
 * 25.07.06 \main\16 CM-DI/ESA3-K"ohler
 * - csm_bGetItpNATelIndex() available for CSM_C_PTYPE_RN_USDT_ISO, too.
 * - Usage of protocol types csm_vGetItpAdrParameter() changed.
 *
 * 02.08.06 \main\17 CM-DI/ESA3-K"ohler
 * - tCSM_ITP_COMMUNICATION_STATE CSM_ITP_rCommunicationState encapsulated with feature switch
 *   CSM_S_ITP_COMMUNICATION_FCT.
 *
 * 02.08.06  \main\18  CM-DI/ESA3-Zurmhl
 * - Some constant names changed: CNP_ITP_XYZ -> CNP_IPT_C_XYZ
 * - Some comments changed, identation of "#" changed
 *
 * 21.08.06  \main\19  CM-DI/ESA3-Khler
 * - ISO-TP normal addressing protocol type placeholder
 *   CSM_C_PTYPE_USED_STD_USDT_ISO added.
 * - tCSM_ITP_STATE and CSM_ITP_rState renamed tCSM_ITP_COMMUNICATION_STATE
 *   and CSM_ITP_rCommunicationState.
 *
 * 31.08.06  \main\20  CM-DI/ESA3-Khler
 * - Return value used and #ifdef CSM_C_PTYPE_USED_STD_USDT_ISO added in
 *   CSM_ITP_lApplCallbackInit() and CSM_ITP_lCommunicationReq().
 * - Include csm_m.h added.
 *
 * 28.09.06  \main\21  CM-DI/ESA3-Zurmhl
 * - some lReturnValue initialized with new default warning (CSM_E_INIT_WARNING)
 *
 * 10.10.06  \main\22  CM-DI/ESA3-Zurmhl
 * - PVOID replaced by void* (because of const PVOID confusion!)
 *
 * 30.10.06  \main\23  CM-DI/ESA3-Looby
 * - Added functions CSM_ITP_vCNPRnASDTConnectCon,CSM_ITP_vCNPRnASDTConnectInd
 * - and CSM_ITP_vRnASDTForceDisconnect(),CSM_ITP_lNMRnASDTConnected().
 * - ALtered CSM_ITP_vProcessCommunicationReq()
 *
 * 31.10.06  \main\24  CM-DI/ESA3-Looby
 * - Added test for ASDT ( to exclude USDT ) in csm_lCheckItpDataReqErrors.
 * - Removed ASDT switch in CSM_ITP_lDataReq, as it shouldn't have been there.
 *
 * 07.11.06  \main\25  CM-DI/ESA3-Khler
 * - CSM_ADDR_OPEL_USDT_ISO replaced by tCSM_ADDR_ISO_TP.
 * - Code reading.
 *
 * 01.12.06  \main\26  CM-DI/ESA3-Prhl
 * - add macros CSM_M_GET_TASK_CONTEXT and CSM_M_SET_TASK_CONTEXT for all callback functions
 *
 * 12.01.07  \main\27  CM-DI/ESA3-Prhl
 * - make ready to compile for WIN32. some #if moved

 * 18.10.07  \main\28  CM-DI/ESA3-Looby
 * - Altered CSM_ITP_vCNPDataCon(),CSM_ITP_vRnASDTForceDisconnect().
 * - Added csm_itp_vHLRetryTLBTimer() and csm_itp_vRN_ASDT_HighLevelRetry().
 *
 * 10.02.07  \main\29  CM-DI/ESA3-Looby
 * - Altered Renault CSM_ITP_vRnASDTForceDisconnect().
 *
 * 15.02.07  \main\30  CM-DI/ESA3-Looby
 * - Renault NM calles ASDT TP connection confirmation.
 *
 * 03.03.07  \main\31  CM-DI/ESA3-Looby
 * - Added functions CSM_ITP_vCNPNMLimpHome .
 *
 * 23.03.07  \main\32  CM-DI/ESA3-Looby
 * - Renault specific changes for C-box,initial version.
 *
 * 24.04.07  \main\33  VTeam-Prhl
 * - fix CSM_ITP_lCommunicationReq(). lReturnValue was not set in one case.
 *
 * 22.05.07  \main\34  VTeam-Looby
 * - Altered Renault specific functions regarding for C-box/Amp.
 *
 * 22.05.07  \main\35  VTeam-Looby
 * - Altered Renault specific function CSM_ITP_vRnASDTConnectionCon.
 *
 * 25.05.07  \main\36  VTeam-Prhl
 * - some traces added
 *
 * 15.06.07  \main\37  VTeam-Prhl
 * - this makes some tools happy.
 *
 * 19.06.07  \main\38  VTeam-Prhl
 * - some asserts aded to CSM_ITP_lApplCallbackInit()
 * - handling for network management payload added
 *
 * 29.06.07  \main\39  VTeam-Prhl
 * - workaround for funny VW diagnose
 *
 * 05.07.07  \main\40  VTeam-Looby
 * - Altered Renault specific ASDT High-level retry.
 *
 * 13.07.07  \main\41  VTeam-Khler
 * - Handle stored in CSM_ITP_lApplCallbackInit().
 *
 * 20.07.07  \main\42  VTeam-Khler
 * - Pending data indication is checked and retriggered after callback
 *   registration.
 *
 * 31.07.07  \main\43  VTeam-Prhl
 * - lint warnings removed or commented out.
 *
 * 08.10.07  \main\44  VTeam-Zurmhl
 * - ubyte replaced by BYTE in csm_itp_vRN_ASDT_HighLevelRetry()
 *
 * 15.11.07  \main\45  VTeam-Prhl
 * - CNM_OSEK_S_VW_PAYLOAD replaced by CNM_OSEK_S_PAYLOAD
 *
 * 25.01.08  \main\46  VTeam-Prhl
 * - CSM_ITP_lDataReq() modified
 * - RN_ASDT code removed
 * - old feature switchs removed
 *
 * 05.02.08  \main\47  VTeam-Prhl
 * - feature CSM_S_ITP_FF_INDICATION_AVAILABLE and CSM_S_ITP_ERROR_INDICATION_AVAILABLE added
 *   to CSM_ITP_lApplCallbackInit() and CSM_ITP_lInit().
 *
 * 06.02.08  \main\48  VTeam-Khler
 * - Lint warning removed.
 *
 * 11.02.08  \main\49  VTeam-Prhl
 * - lint authorisation added.
 *
 * 21.02.08  \main\50  VTeam-Prhl
 * - lint authorisation added.
 *
 * 22.02.08  \main\51  VTeam-Prhl
 * - make doxygen happy.
 *
 * 10.04.08  \main\52  VTeam-Khler
 * - DataCon callback pointer checked in csm_lCheckItpDataReqErrors(), too.
 *
 * 18.04.08  \main\53  VTeam-Prhl
 * - some reviews added.
 *
 * 18.06.08 \main\54 VTeam-Brunotte
 * - CSM_ITP_lExit added. In stop state communication state is switched to down.
 *   In off state all callback functions are removed.
 *
 * 04.07.08  \main\55  VTeam-Zurmhl
 * - CSM_ITP_vCNPErrorInd() added again (active only for feature switch CSM_S_DEBUG)
 *
 * 25.07.08  \main\56 VTeam-Zurmhl
 * - Some reviewing done, some comments updated
 * - csm_vGetItpAdrParameter(): parameter check added
 *
 * 29.08.08 VTeam-Brunotte
 * - Macro CSM_S_USE_TKSE check in CSM_ITP_lApplCallbackInit added.
 *
 * 13.02.09  \main\58 VTeam-Zurmhl
 * - Review done (no change in code)
 *
 * 25.08.09  \main\59 VTeam-Franke
 * - new: CSM_ITP_vEndCommunicationRequests, CSM_ITP_vCNPDataErrorInd, CSM_ITP_vCNPDataIndFF
 *
 * 31.08.10  \main\60  Prhl
 * - CSM_ITP_lGetCommunicationState() added
 *
 * 08.10.10  \main\61  VTeam-Franke
 * - ISO TP: enhance maximum channels to 32
 *   use along cnp_itp.c     \main\70
 *             cnp_itp_i.h   \main\26
 *             csm.c         \main\270
 *             csm_stack_m.h \main\207
 *
 * 10.11.10  \main\62  Prhl
 * - CSM_ITP_vResetIndicationFlags() added
 *   use along csm.c        \main\273
 *             csm_itp.c    \main\62
 *             csm_itp_m.h  \main\22
 *             cnp_itp.c    \main\72
 *             cnp_itp_i.h  \main\27
 *
 * 03.01.11  \main\63  Pistoor
 * - Removed LINT warnings
 *
 * 25.01.11  \main\64  VTeam-Khler
 * - Struct tCSM_LOCAL_CTRL_APPL_CALLBACK_ADMIN used instead of
 *   tCSM_ISO_TP_USDT_APPL_CALLBACK for array csm_itp_arCallbackFkt[] to
 *   separate callback pointer from application and internal administration data.
 *   csm_stack_i.h main\134
 *   csm_stack_m.h main\211, csm_stack_mp.h and cnp_itp.c update necessary.
 * - Lint warnings removed.
 *
 * 18.02.11  \main\65  VTeam-Khler
 * - Lint deactivation authorization strings added.
 *
 * 09.05.11  \main\66  Prhl
 * - only code review
 *
 * 13.07.11  \main\67  Borck
 * - ETG tracing introduced.
 *
 * 10.01.13  \main\68  Feldhaus
 * - added check for new Normal Fixed address mode
 *
 * 08.02.13  \main\69  Feldhaus
 * - removed feature switch CSM_S_CALLBACK_INIT_BY_APPLICATION in CSM_ITP_vCSMBusStateInd
 *   because this code is also required without this switch.
 *
 * 03.05.13  \main\70  Pistoor
 * - added CSM_ITP_vPROXYReceiveInd() in case of remote enabled
 *
 * 03.06.13  \main\71  Pistoor
 * - aremoved second initalisation of fvDataInd in ApplcationCallbackInit
 *
 * 17.07.13  \main\72  Prhl
 * - bugfix for second channel with CSM proxy.
 *
 * 23.08.13  \main\73  Prhl
 * - CSM_ITP_vPROXYTriggerCommunicationCon() added.
 *
 * 12.09.13  \main\74  Feldhaus
 * - further modifications for CSM Proxy support
 *
 * 15.10.13  \main\gen3_linux\2  Prhl
 * - Bugfix for removed CNP_ITP part.
 *
 * 18.10.13  \main\gen3_linux\3  Prhl
 * - trace fixed.
 *
 * 21.10.13  \main\gen3_linux\4  Prhl
 * - CSM_ITP_vPROXYTriggerCommunicationInd() added.
 *
 * 06.12.13  \main\gen3_linux\5  Prhl
 * - lint warning removed. missing #ifdef around feature switches.
 *
 * 12.05.17  \main\gen3_linux\rn_aivi_proxy\1  Borck
 * - include csm proxy interface, if using a remote stack
 * - mapping DataCon processing function from a CNP-located one to local one here (to be able to compile)
 * - some code cleanup, -> no functional changes
 * - added callback for DataInd from Proxy -> callbacks for AccessLayer are called in downstream task directly
 * - callback for DataCon from Proxy (name still containing the confusing CNP) is checking channel index to be valid
 * - DataCon processing function added, to get rid of the CNP module which is obsolete for the proxy variant of CSM
 *
 * 30.05.17  \main\gen3_linux\rn_aivi_proxy\2  Borck
 * - proper DataInd Callback handling via Upstream task now (previous solution used downstream task)
 *
 * 31.05.17  \main\gen3_linux\rn_aivi_proxy\3  Borck
 * - dedicated DataCon handler for proxy (to get rid of confusing names that contain CNP when having a proxy system without CNP module)
 * - LINT issues fixed
 *
 * \history_end p
 *//**** END OF FILE **********************************************************/
