/* Description and history of csm.c at the end of the file.                   */
/** File name for preprozessor commands */
#define CSM_SELF
/** csm.c is a file according V-team CAN-SW architecture implementation rules */
#define CSM_CAN_STACK

/*******************************************************************************
 * includes:
 ******************************************************************************/
/*************** CSM and stack **************/
#include "csm_stack_i.h"
#include "csm_i.h"
#include "csm_stack_m.h"


#if ( CSM_S_CSM_VW == CSM_C_F_ON )
  #include "csm_vw_i.h"  /* Include VW adoptions and if extensions. */
#endif

#if ( CSM_S_VW_BAP_TP == CSM_C_F_ON )
  #include "csm_bap_i.h"
#endif

#include "csm_m.h"
#include "csm_c.h"
#include "csm_apl.h"

#if  (CSM_S_CSM_LOCAL_CTRL == CSM_C_F_ON)
#include "csm_lctr_i.h"
#endif /*  (CSM_S_CSM_LOCAL_CTRL == CSM_C_F_ON)  */

#if (CSM_S_REMOTE_STACK == CSM_C_F_ON)
 /************ CSM PROXY ***********/
 #include "csm_proxy_i.h"
#else
  /*************** CDA **************/
  #if (CSM_S_CDA_AVAILABLE == CSM_C_F_ON)
    #include "cda_i.h"
  #endif /*CSM_S_CDA_AVAILABLE == CSM_C_F_ON*/
#endif

/*************** CNM **************/
#if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
  #include "cnm_i.h"

  #if (CSM_S_OSEK_NM == CSM_C_F_ON)
    #include "cnm_osek_i.h"
  #endif /* CSM_S_OSEK_NM */
  #if (CSM_S_OSEKI_NM == CSM_C_F_ON)
    #include "cnm_oseki_i.h"
  #endif /* CSM_S_OSEKI_NM */
  #if (CSM_S_VWHIGH_NM == CSM_C_F_ON)
    #include "cnm_vwhigh_i.h"
  #endif /* CSM_S_VWHIGH_NM */
  #if (CSM_S_GMLAN_NM == CSM_C_F_ON)
    #include "cnm_gmlan_i.h"
  #endif /* CSM_S_GMLAN_NM */
  #if (CSM_S_FIATSLAVE_NM == CSM_C_F_ON)
    #include "cnm_FiatSlave_i.h"
  #endif
  #if (CSM_S_IVLAN_NM == CSM_C_F_ON)
    #include "cnm_ivlan_i.h"
  #endif
  #if (CSM_S_J1939_NM == CSM_C_F_ON)
    #include "cnm_J1939_i.h"
  #endif
#endif /* CSM_S_CNM_AVAILABLE */

/*************** CNP **************/
#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
  #include "cnp_i.h"

  #if (CSM_S_ISO_TP == CSM_C_F_ON)
    #include "csm_itp_m.h"
    #include "cnp_itp_i.h"
  #endif

  #if (CSM_S_LIN_DIAG == CSM_C_F_ON)
    #include "csm_lin_diag_m.h"
  #endif /* CSM_S_LIN_DIAG == CSM_C_F_ON */

  #if (CSM_S_UUDT_TX == CSM_C_F_ON) || (CSM_S_UUDT_RX == CSM_C_F_ON)
    #include "cnp_uudt_i.h"
  #endif

  #if (CSM_S_MCNET == CSM_C_F_ON)
    #include "csm_mcnet_i.h"  /* U.Z.(6) */
    #include "cnp_mcnet_i.h"
    #if ((CSM_S_MCNET_MASTER_NWM == CSM_C_F_ON) || (CSM_S_MCNET_SLAVE_NWM == CSM_C_F_ON))
      #include "cnm_mcnet_i.h"
    #endif
  #endif

  #if (CSM_S_TP20 == CSM_C_F_ON)
    #include "csm_tp20_i.h"
    //  #include "cnp_tp20_i.h"
  #endif

  #if (CSM_S_MPDT == CSM_C_F_ON)
    #include "csm_mpdt_m.h"
    #include "csm_mpdt_i.h"
    #include "cnp_mpdt_i.h"
  #endif

  #if (CSM_S_J1939 == CSM_C_F_ON)
    #include "csm_j1939_m.h"
  #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
    #include "cnp_J1939_i.h"
  #endif
  #endif
#endif /* CSM_S_CNP_AVAILABLE */

/*************** CBR **************/
#if (CSM_S_CBR_AVAILABLE == CSM_C_F_ON)
  #if (CSM_S_CBR_USES_COMMUNICATION_REQ == CSM_C_F_ON)
    #include "csm_cbr_m.h"
  #endif
  #include "cbr_i.h"
#endif /* CSM_S_CBR_AVAILABLE */

#if (CSM_S_CBR_AVAILABLE == CSM_C_F_ON)
  #if (CSM_S_CBR_QUEUED_SIGNALS_USED == CSM_C_F_ON)
  extern VOID CSM_vCCS_ClearQSigInd(VOID);
  #endif
#endif /* CSM_S_CBR_AVAILABLE */

#if (!defined CSM_S_v850_SIGNAL_ABSTRACTION)
#error "symbol CSM_S_v850_SIGNAL_ABSTRACTION must be defined as either CSM_C_F_ON or CSM_C_F_OFF"
#endif

#if (!defined CSM_S_iMX_SIGNAL_ABSTRACTION)
#error "symbol CSM_S_iMX_SIGNAL_ABSTRACTION must be defined as either CSM_C_F_ON or CSM_C_F_OFF"
#endif

#if ((CSM_S_v850_SIGNAL_ABSTRACTION == CSM_C_F_ON) || (CSM_S_iMX_SIGNAL_ABSTRACTION == CSM_C_F_ON))
#include "cbr_al.h"
#endif

#if (CSM_S_CBR_AVAILABLE == CSM_C_F_ON)
#if (CSM_S_CCS_USED == CSM_C_F_ON)
  #include "ccs_prot.h"
#endif /* CSM_S_CCS_USED */
#endif /* CSM_S_CBR_AVAILABLE */

/*************** ETG Tracing ******/
#if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
    #define ETRACE_S_IMPORT_INTERFACE_GENERIC
    #include "etrace_if.h"

  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
    #include "csm_etg_trace.h"
    #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FD_CSM
    /* #define ETG_INLINE */
    #include "trcGenProj/Header/csm.c.trc.h"
  #endif
  #endif
#endif /* (CSM_S_TRACE == CSM_C_F_ON) */


/*****************************************************************
| check of all feature switches which affects this file :
| The following #defines should be declared in csm_stack_p.h:
|----------------------------------------------------------------*/
#ifndef CSM_S_VIRTUAL_NETWORK
  #error "CSM_S_VIRTUAL_NETWORK must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_CDA_AVAILABLE
  #error "CSM_S_CDA_AVAILABLE must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_STATE_MASCHINE
  #error "CSM_S_STATE_MASCHINE must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_BLOCKABLE_UPSTREAM_TASK
  #error "CSM_S_BLOCKABLE_UPSTREAM_TASK must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_BUS_WAKE_MIN
  #error "CSM_S_BUS_WAKE_MIN must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_BUS_WAKE_TEMP
  #error "CSM_S_BUS_WAKE_TEMP must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_CALLBACK_INIT_BY_APPLICATION
  #error "CSM_S_CALLBACK_INIT_BY_APPLICATION must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_CAN_WAKEUP_FUNCTIONS_VIA_CSM
  #error "CSM_S_CAN_WAKEUP_FUNCTIONS_VIA_CSM must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_CBR_AVAILABLE
  #error "CSM_S_CBR_AVAILABLE must be set to CSM_C_F_ON or -OFF !"
#else
  #if (CSM_S_CBR_AVAILABLE == CSM_C_F_ON)
    #ifndef CSM_S_CBR_USES_COMMUNICATION_REQ
      #error "CSM_S_CBR_USES_COMMUNICATION_REQ must be set to CSM_C_F_ON or -OFF !"
    #endif

    #ifndef CSM_S_CBR_QUEUED_SIGNALS_USED
      #error "CSM_S_CBR_QUEUED_SIGNALS_USED must be set to CSM_C_F_ON or -OFF !"
    #endif
    #ifndef CSM_S_CCS_USED
      #error "CSM_S_CCS_USED must be set to CSM_C_F_ON or -OFF !"
    #endif
  #endif
#endif

#ifndef CSM_S_CDA_AVAILABLE
  #error "CSM_S_CDA_AVAILABLE must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_CNM_AVAILABLE
  #error "CSM_S_CNM_AVAILABLE must be set to CSM_C_F_ON or -OFF !"
#else
  #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
    #ifndef CSM_S_FIATSLAVE_NM
      #error "CSM_S_FIATSLAVE_NM must be set to CSM_C_F_ON or -OFF !"
    #endif

    #ifndef CSM_S_OSEK_NM
      #error "CSM_S_OSEK_NM must be set to CSM_C_F_ON or -OFF !"
      #ifndef CNM_OSEK_S_PAYLOAD
        #error "CNM_OSEK_S_PAYLOAD must be set to CSM_C_F_ON or -OFF !"
      #endif
    #endif

    #ifndef CSM_S_GMLAN_NM
      #error "CSM_S_GMLAN_NM must be set to CSM_C_F_ON or -OFF !"
    #endif

    #ifndef CSM_S_IVLAN_NM
      #error "CSM_S_IVLAN_NM must be set to CSM_C_F_ON or -OFF !"
    #endif

    #ifndef CSM_S_LAST_NM_MSG
      #error "CSM_S_LAST_NM_MSG must be set to CSM_C_F_ON or -OFF !"
    #endif
  #endif /* CSM_S_CNM_AVAILABLE */
#endif

#ifndef CSM_S_CNP_AVAILABLE
  #error "CSM_S_CNP_AVAILABLE must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_CSM_LOCAL_CTRL
  #error "CSM_S_CSM_LOCAL_CTRL must be set to CSM_C_F_ON or -OFF !"
#else
  #if (CSM_S_CSM_LOCAL_CTRL == CSM_C_F_ON)
    #ifndef CSM_S_CSM_LOCAL_CTRL_USER_FUNCTION
      #error "CSM_S_CSM_LOCAL_CTRL_USER_FUNCTION must be set to CSM_C_F_ON or -OFF !"
    #endif
  #endif /* CSM_S_CSM_LOCAL_CTRL == CSM_C_F_ON */
#endif

#ifndef CSM_S_CSM_VW
  #error "CSM_S_CSM_VW must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_DEBUG
  #error "CSM_S_DEBUG must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_DOXYGEN_HIDE_NON_API_FCT
  #error "CSM_S_DOXYGEN_HIDE_NON_API_FCT must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_ENABLE_ALWAYS_INDICATE_STATE_CHANGE
  #error "CSM_S_ENABLE_ALWAYS_INDICATE_STATE_CHANGE must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_ENABLE_INDICATE_BUSERROR
  #error "CSM_S_ENABLE_INDICATE_BUSERROR must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_ENABLE_INDICATE_STATIC_ERROR
  #error "CSM_S_ENABLE_INDICATE_STATIC_ERROR must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_FNL
  #error "CSM_S_FNL must be set to CSM_C_F_ON or -OFF !"
#endif

#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
  #ifndef CSM_S_UUDT_RX
    #error "CSM_S_UUDT_RX must be set to CSM_C_F_ON or -OFF !"
  #endif

  #ifndef CSM_S_UUDT_TX
    #error "CSM_S_UUDT_TX must be set to CSM_C_F_ON or -OFF !"
  #endif

  #ifndef CSM_S_ISO_TP
    #error "CSM_S_ISO_TP must be set to CSM_C_F_ON or -OFF !"
  #else
    #if (CSM_S_ISO_TP == CSM_C_F_ON)
      #ifndef CSM_S_ITP_APPL_SETS_RX_BUFFER_FREE
        #error "CSM_S_ITP_APPL_SETS_RX_BUFFER_FREE must be set to CSM_C_F_ON or -OFF !"
      #endif
      #ifndef CSM_S_ITP_COMMUNICATION_FCT
        #error "CSM_S_ITP_COMMUNICATION_FCT must be set to CSM_C_F_ON or -OFF !"
      #endif
      #ifndef CSM_S_ITP_APPL_TX_BUFFER_USED
        #error "CSM_S_ITP_APPL_TX_BUFFER_USED must be set to CSM_C_F_ON or -OFF !"
      #endif
      #if (CSM_S_ITP_ASDT_EXTENSIONS == CSM_C_F_ON)
        #ifndef CSM_S_ITP_ASDT_EXTENSION_RENAULT
          #error "CSM_S_ITP_ASDT_EXTENSION_RENAULT must be set to CSM_C_F_ON or -OFF !"
        #endif
      #endif
    #endif
  #endif

  #ifndef CSM_S_MCNET
    #error "CSM_S_MCNET must be set to CSM_C_F_ON or -OFF !"
  #endif
  #if (CSM_S_MCNET == CSM_C_F_ON)
    #ifndef CSM_S_MCNET_ASDT
      #error "CSM_S_MCNET_ASDT must be set to CSM_C_F_ON or -OFF !"
    #endif

    #ifndef CSM_S_MCNET_MASTER_NWM
      #error "CSM_S_MCNET_MASTER_NWM must be set to CSM_C_F_ON or -OFF !"
    #endif

    #ifndef CSM_S_MCNET_SLAVE_NWM
      #error "CSM_S_MCNET_SLAVE_NWM must be set to CSM_C_F_ON or -OFF !"
    #endif

    #ifndef CSM_S_MCNET_USDT_TX
      #error "CSM_S_MCNET_USDT_TX must be set to CSM_C_F_ON or -OFF !"
    #endif
  #endif /* (CSM_S_MCNET == CSM_C_F_ON) */

  #ifndef CSM_S_VW_TP_1_6
    #error "CSM_S_VW_TP_1_6 must be set to CSM_C_F_ON or -OFF !"
  #endif

  #ifndef CSM_S_VW_TP_2_0
    #error "CSM_S_VW_TP_2_0 must be set to CSM_C_F_ON or -OFF !"
  #endif

  #if (CSM_S_LIN_DIAG == CSM_C_F_ON)
    #ifndef CSM_C_PTYPE_USED_STD_LIN_DIAG
      #error CSM_C_PTYPE_USED_STD_LIN_DIAG must be defined!
    #endif /* CSM_C_PTYPE_USED_STD_LIN_DIAG */
  #endif /* CSM_S_LIN_DIAG == CSM_C_F_ON */

#endif /* CSM_S_CNP_AVAILABLE */

#ifndef CSM_S_VW_BAP_TP  /* according to csm_stack_p.h do not remove this to cnp section */
  #error "CSM_S_VW_BAP_TP must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_REMOTE_STACK
  #error "CSM_S_REMOTE_STACK must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_RX_OVERRUN_INDICATION
  #error "CSM_S_RX_OVERRUN_INDICATION must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_STATE_MASCHINE
  #error "CSM_S_STATE_MASCHINE must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_TEST
  #error "CSM_S_TEST must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_TRACE
  #error "CSM_S_TRACE must be set to CSM_C_F_ON or -OFF !"
#endif
#if (CSM_S_TRACE == CSM_C_F_ON)
  #ifndef CSM_S_TRACE_LIMITED_SIGNAL
    #error "CSM_S_TRACE_LIMITED_SIGNAL must be set to CSM_C_F_ON or -OFF !"
  #endif
#endif /* (CSM_S_TRACE == CSM_C_F_ON) */

#ifndef CSM_S_USE_CALLBACK_WITH_HANDLE
  #error "CSM_S_USE_CALLBACK_WITH_HANDLE must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_VARIANTS_MANAGEMENT
  #error "CSM_S_VARIANTS_MANAGEMENT must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_VIRTUAL_NETWORK
  #error "CSM_S_VIRTUAL_NETWORK must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_USE_TKSE
  #error "CSM_S_USE_TKSE must be set to CSM_C_F_ON or -OFF !"
#endif

/*******************************************************************************
 * defines and macros (scope: file local)
 ******************************************************************************/
/* -- none --*/

/*******************************************************************************
 * typedefs (scope: file local)
 ******************************************************************************/
typedef LONG (*tCSM_PFN_INIT) ( BYTE );   /**< function pointer for ..._lInit()*/
typedef LONG (*tCSM_PFN_EXIT) ( BYTE );   /**< function pointer for ..._lExit()*/

/*******************************************************************************
 * variable definition (scope: file local)
 ******************************************************************************/
CSM_STATIC const tCSM_PFN_INIT csm_alpfInit[] = /**< list of all ..._lInit() functions */
{
  #if  (CSM_S_CSM_LOCAL_CTRL == CSM_C_F_ON)
  CSM_LCTR_lInit,
  #endif /*  (CSM_S_CSM_LOCAL_CTRL == CSM_C_F_ON)  */
  #if (CSM_S_CSM_VW == CSM_C_F_ON)
  CSM_VW_lInit,
  #endif
  CSM_lInitEnv,
  #if (CSM_S_VARIANTS_MANAGEMENT == CSM_C_F_ON)
    CSM_lInitVariants,
  #endif
  #if (CSM_S_REMOTE_STACK == CSM_C_F_ON)
    CSM_PROXY_lInit,
  #else
  #if (CSM_S_CDA_AVAILABLE == CSM_C_F_ON)
    CDA_lInit,
  #endif /*CSM_S_CDA_AVAILABLE == CSM_C_F_ON*/
  #endif
  #if (CSM_S_CBR_AVAILABLE == CSM_C_F_ON)
    #if (CSM_S_CBR_USES_COMMUNICATION_REQ == CSM_C_F_ON)
      CSM_CBR_lInit,
    #endif
    CBR_lInit,
  #endif /* CSM_S_CBR_AVAILABLE */
  #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
    CNM_lInit,
  #endif /* CSM_S_NM_AVAILABLE */
  #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
    #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
    CNP_lInit,
    #endif
    #if (CSM_S_MCNET == CSM_C_F_ON)
      CSM_MCNET_lInit,
    #endif /* CSM_S_MCNET */
    #if (CSM_S_ISO_TP == CSM_C_F_ON)
      #if (CSM_S_ITP_COMMUNICATION_FCT == CSM_C_F_ON)
        CSM_ITP_lInit,
      #endif
    #endif /* CSM_S_ISO_TP */
    #if (CSM_S_MPDT == CSM_C_F_ON)
      CSM_MPDT_lInit,
    #endif /* CSM_S_MPDT */
    #if (CSM_S_J1939 == CSM_C_F_ON)
      CSM_J1939_lInit,
    #endif
  #endif /* CSM_S_CNP_AVAILABLE */
  /* BAP outside of CNP! */
  #if ( CSM_S_VW_BAP_TP == CSM_C_F_ON )
    CSM_BAP_lInit,
  #endif
  #ifndef CSM_lInitApl /* if CSM_lInitApl is not an (empty) define then it is a
                        * function and has to be called */
  CSM_lInitApl
  #endif
};

/* ************************************************************************* */
CSM_STATIC const tCSM_PFN_EXIT csm_alpfExit[] = /**< list of all ..._lExit() functions */
{
  #ifndef CSM_lExitApl /* if CSM_lExitApl is not an (empty) define then it is a
                        * function and has to be called */
    CSM_lExitApl,
  #endif
  #if ( CSM_S_VW_BAP_TP == CSM_C_F_ON )
    CSM_BAP_lExit,
  #endif
  #if (CSM_S_REMOTE_STACK == CSM_C_F_ON)
    CSM_PROXY_lExit,
  #else
  #if (CSM_S_CDA_AVAILABLE == CSM_C_F_ON)
    CDA_lExit,
  #endif /*CSM_S_CDA_AVAILABLE == CSM_C_F_ON*/
  #endif
  #if (CSM_S_CBR_AVAILABLE == CSM_C_F_ON)
    #if (CSM_S_CBR_USES_COMMUNICATION_REQ == CSM_C_F_ON)
      CSM_CBR_lExit,
    #endif
    CBR_lExit,
  #endif /* CSM_S_CBR_AVAILABLE */
  #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
    CNM_lExit,
  #endif /* CSM_S_NM_AVAILABLE */
  #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
    #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
    CNP_lExit ,
    #endif
  #if (CSM_S_MCNET == CSM_C_F_ON)
    CSM_MCNET_lExit,
  #endif /* CSM_S_MCNET */
  #if (CSM_S_CSM_VW == CSM_C_F_ON)
    CSM_VW_lExit,
  #endif
  #if (CSM_S_ISO_TP == CSM_C_F_ON)
    #if (CSM_S_ITP_COMMUNICATION_FCT == CSM_C_F_ON)
      CSM_ITP_lExit,
    #endif /* (CSM_S_ITP_COMMUNICATION_FCT == CSM_C_F_ON) */
  #endif /* CSM_S_ISO_TP */
  #if (CSM_S_MPDT == CSM_C_F_ON)
    CSM_MPDT_lExit,
  #endif /* CSM_S_MPDT */
  #if (CSM_S_J1939 == CSM_C_F_ON)
    CSM_J1939_lExit,
  #endif

  #endif /* CSM_S_CNP_AVAILABLE */
  #if  (CSM_S_CSM_LOCAL_CTRL == CSM_C_F_ON)
  CSM_LCTR_lExit,
  #endif /*  (CSM_S_CSM_LOCAL_CTRL == CSM_C_F_ON)  */

  CSM_lExitEnv
};

#if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
#if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
  tCSM_BUS_ERROR_IND_CALLBACK csm_arBusErrorIndCallbackFkt[ CSM_C_MAX_ERROR_IND_USER];
#else
extern tCSM_BUS_ERROR_IND_CALLBACK csm_arBusErrorIndCallbackFkt[ CSM_C_MAX_ERROR_IND_USER];
#endif
#endif /* CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON */

/* EM trace error is used to store an error code that leads to em_trace       */
/* outputs in dedicated situations                                            */
static DWORD dwCSM_EmTraceError;

/*******************************************************************************
 * function prototypes (scope: file local = static)
 ******************************************************************************/
//??void CSM_vCheckandWriteErrorToEM(void);

/*******************************************************************************
 * variable definition (scope: CAN stack)
 ******************************************************************************/
CSM_STATIC const tCSM_PFN_BUS_STATE_IND csm_avpfBusStateInd[] =
{
#if (CSM_S_CSM_LOCAL_CTRL == CSM_C_F_ON)
  CSM_LCTR_vCSMBusStateInd,
#endif /* CSM_S_CSM_LOCAL_CTRL == CSM_C_F_ON */
#if (CSM_S_CBR_AVAILABLE == CSM_C_F_ON)
#if (CSM_S_CBR_USES_COMMUNICATION_REQ == CSM_C_F_ON)
  CSM_CBR_vCSMBusStateInd,
#endif /* CSM_S_CBR_USES_COMMUNICATION_REQ */
#endif /* CSM_S_CBR_AVAILABLE */
#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
#if (CSM_S_ISO_TP == CSM_C_F_ON)
  CSM_ITP_vCSMBusStateInd,
#endif /* CSM_S_ISO_TP */
#if (CSM_S_MCNET == CSM_C_F_ON)
  CNP_MCNET_vCSMBusStateInd,
#endif /* CSM_S_MCNET */
#if (CSM_S_J1939 == CSM_C_F_ON)
  CSM_J1939_vCSMBusStateInd,
#endif /* CSM_S_J1939 */
#if (CSM_S_CCS_USED == CSM_C_F_ON)
  CCS_NET_vCSM_BusStateInd,
#endif /* CSM_S_CCS_USED */
#endif /* CSM_S_CNP_AVAILABLE */
NULL,
};

#if (CSM_S_BUS_WAKE_TEMP == CSM_C_F_ON)
CSM_STATIC BOOL csm_fWakeTemp = FALSE;
#endif
#if (CSM_S_BUS_WAKE_MIN == CSM_C_F_ON)
CSM_STATIC BOOL csm_afWakeMin[ CSM_C_MAX_CAN_BUS] = {FALSE};
#endif

#if (CSM_S_DEBUG == CSM_C_F_ON)
/*****************************************************************
| variable csm_bNumberOfBusoff
|
| comment: number of busoff retries for each bus
|
| range: 0..(2^8 - 1)
|
| access level (loop / interrupt): interrupt level
|----------------------------------------------------------------*/
BYTE csm_bNumberOfBusoff[CSM_C_MAX_CAN_BUS];
#endif

/*****************************************************************
| variable csm_fStateMachineRuns
|
| comment: flag, if state machine runs actually
|
| range: TRUE, FALSE
|
| access level (loop / interrupt): main loop level
|----------------------------------------------------------------*/
BOOLEAN csm_fStateMachineRuns;

/*****************************************************************
| variable csm_lCommunicationRequests
|
| comment: bitfield for requested channels for each protocol on
|          every channel (used by CSM_lSetBusHold, CSM_lClearBusHold)
|
| range: 0..(2^32 - 1)
|
| access level (loop / interrupt): main loop level
|----------------------------------------------------------------*/
#ifdef CSM_C_MAX_PTYPE_1
#define CSM_C_MAX_POSSIBLE_APPL_ID  32 /* because a DWORD is used as storage location */
  DWORD csm_adwCommunicationRequests1[CSM_C_MAX_PTYPE_1];
  #if CSM_C_MAX_CAN_BUS > 1
  DWORD csm_adwCommunicationRequests2[CSM_C_MAX_PTYPE_2];
  #if CSM_C_MAX_CAN_BUS > 2
  DWORD csm_adwCommunicationRequests3[CSM_C_MAX_PTYPE_3];
  #if CSM_C_MAX_CAN_BUS > 3
     #error Insert here functionality for next CAN bus
  #endif /* #if (CSM_C_MAX_CAN_BUS > 3) */
  #endif /* #if (CSM_C_MAX_CAN_BUS > 2) */
  #endif /* #if (CSM_C_MAX_CAN_BUS > 1) */
#endif /* CSM_C_MAX_PTYPE_1 */

/*****************************************************************
| variable csm_bCsmState
|
| comment: Main states of the stack manager
|          (undefined, inactive, active, ...)
| range:   0..(2^8 - 1)
|
| access level (loop / interrupt): main loop level
|----------------------------------------------------------------*/
#ifdef GLB_CAR_RADIO
  BYTE csm_bCsmState; /**< Main states of the stack manager possible values: CSM_C_STATE_UNDEFINED,
                       *   CSM_C_STATE_OFF, CSM_C_STATE_INITIALIZED, CSM_C_STATE_ACTIVE
                       *   Initialized variables not allowed in GLB_CAR_RADIO environment.
                       *   Global variables are set to 0 (= undefined) after reset. */
#else
  BYTE csm_bCsmState = CSM_C_STATE_UNDEFINED; /**< Main states of the stack manager possible values:
                                               *   CSM_C_STATE_UNDEFINED, CSM_C_STATE_OFF,
                                               *   CSM_C_STATE_INITIALIZED, CSM_C_STATE_ACTIVE. */
#endif
/* initialised via CSM_lInit(CSM_C_INIT_COLD) */

/*****************************************************************
| variable csm_abCsmSubState[]
|
| comment: sub states of the stack manager for each CAN bus
| range:   0..(2^8 - 1)
|
| access level (loop / interrupt): main loop level
|----------------------------------------------------------------*/
BYTE csm_abCsmSubState[CSM_C_MAX_CAN_BUS];

/*****************************************************************
| variable csm_bBusOffState[]
|
| comment: busoff state of the CAN stack for each CAN bus
|
| range:   TRUE, FALSE
|
| access level (loop / interrupt): interrupt / main loop level
|----------------------------------------------------------------*/
tBusOffState csm_bBusOffState[CSM_C_MAX_CAN_BUS];

#if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
/*******************************************************************************
| comment: buffer for indication of the bus error and bus nr.
|
| range: 0..255
|
| access level (loop / interrupt): main loop level
|-----------------------------------------------------------------------------*/
static BYTE csm_abActBusError[ CSM_C_MAX_CAN_BUS]; /* present error state of CAN-bus */
static BOOL csm_afTriggerErrorInd[ CSM_C_MAX_CAN_BUS]; /* trigger flag for each bus */
#endif

#if (CSM_S_ENABLE_INDICATE_STATIC_ERROR == CSM_C_F_ON)
/*******************************************************************************
| comment: buffer for indication of the static initialisation error .
|
| range: 0..255
|
| access level (loop / interrupt): main loop level
|-----------------------------------------------------------------------------*/
static LONG csm_lActStaticError[ CSM_C_MAX_CAN_BUS]; /* present error state of CAN-bus */
#endif



/*******************************************************************************
 * constants (scope: file local)
 ******************************************************************************/
/*-- none --*/


/* Feature switch is used only to control doxygen documentation */
#if (CSM_S_DOXYGEN_HIDE_NON_API_FCT == CSM_C_F_OFF)
/*******************************************************************************
 * function implementation (scope: file local = static)
 ******************************************************************************/

/*******************************************************************************
 * function    csm_lExecStateMachine
 * \doxydocu
 * \brief      State machine transition execution
 *
 * \param[in]  bBus
 *              Bus number

 * \param[in]  bState
 *              Old/current state

 * \param[in]  bEvent
 *              Event
 *
 * \return     positive or "0": New state
 *             negative       : Error value
 *
 * \access     CAN-task or application (should be configured)
 * \reentrant  no
 *
 * \lastreview 21.03.11 Borck
 *
 * \history_begin
 * 15.05.03  CM-DI/ESU3-Pistoor
 * - Added security check for NULL pointer before call of
 *   state indication function
 *
 * 25.11.02  CM-DI/EHS3-Pistoor
 * - Added bus number to state callback function
 *
 * 30.10.02  CM-DI/EHS3-Pistoor
 * - When no entry was found, return immediately.
 *   Otherwise the loop would be endless.
 *
 * 25.02.02  CM-DI/EHS3-Pistoor
 * - Initial version
 * \history_end
 ******************************************************************************/
#if (CSM_S_STATE_MASCHINE == CSM_C_F_ON)
CSM_STATIC LONG csm_lExecStateMachine (BYTE bBus, BYTE bState, BYTE bEvent)
{
   DWORD   dwLine;
   DWORD   dwReturn = bState;
   BOOLEAN fResult = FALSE;
   BOOLEAN fEntryFound = FALSE;

   /* Here you can change (your code instead of the while loop),
    * if you have to wait for an external acknowledge */
   do
   {
      /* loop over all configured state transitions                           */
      for (dwLine=0; dwLine<CSM_M_ELEMENTS(csm_arStateMachineTable); dwLine++)
      {
         /* if a matching line in the state table is found                    */
         if ((csm_arStateMachineTable[dwLine].dwEntryState == dwReturn) &&
             (csm_arStateMachineTable[dwLine].bEvent == bEvent) )
         {
            fEntryFound = TRUE;
            /* call event processing function                                 */
            fResult = csm_arStateMachineTable[dwLine].pvFunctionCall(bBus);

            /* get new state depending on function result                     */
            if (fResult)
            {
               dwReturn = csm_arStateMachineTable[dwLine].dwExitStateIfFuncTrue;
            }
            else
            {
               dwReturn = csm_arStateMachineTable[dwLine].dwExitStateIfFuncFalse;
            }
            if((fResult == TRUE)&&(csm_abCsmSubState[bBus] != dwReturn))
            {
               csm_abCsmSubState[bBus] = (BYTE ) (dwReturn & 0xff);
               #if (CSM_S_ENABLE_ALWAYS_INDICATE_STATE_CHANGE == CSM_C_F_ON)
               /* State indications to the appications are called from
                * csm_vOSspecificUpstream()
                */
               CSM_vSYS_BusStateInd(bBus, dwReturn);
               #endif
            }
            break; /* process 1 table line only */
         } /* if a matching line in the state table is found */
      }

      /* check if at least 1 transition has been found in table               */
      if ( (fEntryFound == FALSE) || ( dwLine == CSM_M_ELEMENTS(csm_arStateMachineTable)))
      {
        /* no entry was found -> error                                        */
        CSM_vAssert( FALSE);
        return CSM_M_MAKE_CSM_ERROR( CSM_C_ERR_WARNING, CSM_E_STATE_NO_VALID_ENTRY_IN_STATEMACHINE);
      }
   }while((csm_arStateMachineTable[dwLine].fReachedState == FALSE) &&
          (fResult == TRUE));

   return (LONG)dwReturn;
} /* end csm_lExecStateMachine() */
#endif /* CSM_S_STATE_MASCHINE */

/*******************************************************************************
 * function    csm_lSwitchStateMachine
 * \doxydocu
 * \brief      State machine transition
 *
 * \param[in]  bBus
 *              Bus number
 *
 * \param[in]  bState
 *              Old/current state
 *
 * \param[in]  bEvent
 *              Event
 *
 * \return     positive or "0": New state
 *             negative       : Error value
 *
 * \access     CAN-task or application (should be configured)
 * \reentrant  yes ??
 *
 * \lastreview 21.03.11 Borck
 *             13.10.08 VTeam-Funke
 *             29.05.06 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 13.10.08 VTeam-Funke
 * - align return value to same name
 * - reduce return points of function
 *
 * 10.05.06 CM-DI/ESA3-Battis
 * behaviour of state machine changed: transition during busoff is
 * permitted now.
 *
 * 21.08.02 CM-DI/EHS3-Pistoor
 * Execution only, if state not busoff
 *
 * 24.06.02 CM-DI/EHS3-Pistoor
 * Added busoff hook functionality
 *
 * 25.02.02 CM-DI/EHS3-Pistoor
 * Initial version
 * \history_end
 ******************************************************************************/
#if (CSM_S_STATE_MASCHINE == CSM_C_F_ON)
CSM_STATIC LONG csm_lSwitchStateMachine(BYTE bBus, BYTE bState, BYTE bEvent)
{
  LONG lRet = CSM_C_NO_ERROR;

  /* check if CSM is active                                                   */
  if (csm_bCsmState == CSM_C_STATE_ACTIVE)
  {
    /* check if statemachine is already running                               */
    if (csm_fStateMachineRuns == TRUE)
    {
      lRet = CSM_M_MAKE_CSM_ERROR( CSM_C_ERR_WARNING, CSM_E_REENTRANT_CALL_OF_STATEMACHINE);
    }
    else
    {
      /* mark SM as running, execute transition, unmark as running            */
      csm_fStateMachineRuns = TRUE;
      lRet = csm_lExecStateMachine(bBus, bState, bEvent);
      csm_fStateMachineRuns = FALSE;
    }
  }
  else
  {
    /* if CSM is not active, a state transition cannot be done -> error       */
    lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_WARNING, CSM_E_STATE_IS_NOT_ACTIVE);
  }

  return lRet;
} /* csm_lSwitchStateMachine() */
#endif /* CSM_S_STATE_MASCHINE */

/*******************************************************************************
 * function    csm_fChangeDummy
 * \doxydocu
 * \brief      Dummy function of the CAN stack manager state machine
 *
 *             This function is called, if nothing is to do but a function
 *             pointer is needed,
 *
 * \param[in]  bBus
 *             bus number
 *
 * \return     allways TRUE (= function successful)
 *
 * \access     all
 * \reentrant  yes|no
 *
 * \lastreview  29.05.06 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 18.01.04 CM-DI/ESU3-Name
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_STATE_MASCHINE == CSM_C_F_ON)
CSM_STATIC BOOLEAN csm_fChangeDummy(BYTE bBus)
{

  #if (CSM_S_TEST == CSM_C_F_ON)
    CSM_vTestPrintf("->Dummy\n");
  #endif

  return TRUE;
}/*lint !e715 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally*/
#endif /* CSM_S_STATE_MASCHINE */

/*******************************************************************************
 * function    CSM_fCheckCommunicationReq
 * \doxydocu
 * \brief      Calculate communication request
 *
 *             This function is called from CSM_bCalcBusState() to determine
 *             pending communication requests from application or elsewhere
 *
 * \param[in]  bBus
 *               Bus number
 *
 * \return     BOOL
 *               TRUE:  bus is in use
 *               False: bus is not in use
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview  29.05.06 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 19.10.04  CM-DI/ESA3-Koehler
 * - CSM_ITP_fCheckCommunicationReq() call featured with switch
 *   CSM_S_ITP_COMMUNICATION_FCT.
 *
 * 19.10.04  CM-DI/ESA3-Prhl
 * - Initial version
 * \history_end
 ******************************************************************************/
#ifdef CSM_C_DOWN_REASON_CSM_BUS_STATE_CHANGED
CSM_STATIC BOOL CSM_fCheckCommunicationReq(BYTE bBus)
{
  BOOL fBusUp = FALSE;
  BOOL fRetVal;

  // Funktionsnachlauf Behandlung
  #if (CSM_S_FNL == CSM_C_F_ON)
  if (CSM_CBR_afFnlWake[bBus] == TRUE)
  {
    // der CSM_CBR braucht den BUS
    return TRUE;
  }
  #endif // CSM_S_FNL

  #if (CSM_S_BUS_WAKE_MIN == CSM_C_F_ON)
  if (csm_afWakeMin[bBus] == TRUE)
  {
    // der CSM braucht den BUS
    return TRUE;
  }
  #endif // CSM_S_BUS_WAKE_MIN

  #if (CSM_S_BUS_WAKE_TEMP == CSM_C_F_ON)
  if( csm_fWakeTemp == TRUE)
  {
    // der CSM braucht den BUS
    return TRUE;
  }
  #endif // CSM_S_BUS_WAKE_TEMP

  #if (CSM_S_CSM_LOCAL_CTRL == CSM_C_F_ON)
  fRetVal = CSM_LCTR_fCheckCommunicationReq(bBus);
  if (fRetVal != FALSE)
  {
    #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
      #if (CSM_S_OSEK_NM == CSM_C_F_ON )
        #if ( CNM_OSEK_S_VW_PAYLOAD == CSM_C_F_ON )
        // wecken ber CSM_lCommunicationReq()
        CNM_OSEK_lSetPayload( bBus, CNM_OSEK_C_NMINFO_WAKEUP_COM_REQ);
        #endif // CNM_OSEK_S_VW_PAYLOAD
      #endif // CSM_S_OSEK_NM
    #endif // CSM_S_CNM_AVAILABLE
    fBusUp = TRUE;
  }
  #endif // CSM_S_CSM_LOCAL_CTRL

  #if (CSM_S_CBR_AVAILABLE == CSM_C_F_ON)
    #if (CSM_S_CBR_USES_COMMUNICATION_REQ == CSM_C_F_ON)
    fRetVal = CSM_CBR_fCheckCommunicationReq( bBus);
    if (fRetVal != FALSE)
    {
      #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
        #if ( CSM_S_OSEK_NM == CSM_C_F_ON )
          #if ( CNM_OSEK_S_VW_PAYLOAD == CSM_C_F_ON )
          // wecken ber CSM_lCommunicationReq()
          CNM_OSEK_lSetPayload(bBus, CNM_OSEK_C_NMINFO_WAKEUP_COM_REQ);
          #endif // CNM_OSEK_S_VW_PAYLOAD
        #endif // CSM_S_OSEK_NM
      #endif // CSM_S_CNM_AVAILABLE
      fBusUp = TRUE;
    }
    #endif // CSM_S_CBR_USES_COMMUNICATION_REQ
  #endif // CSM_S_CBR_AVAILABLE

  // #ToDo:  This shall also cover Local Ctrl. later !
  fRetVal = CSM_bGetBusHold(bBus); //lint !e838 yes, sometime a variable get a new value
  if (fRetVal != FALSE)
  {
    fBusUp = TRUE;
  }

  #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
  {
    #if (CSM_S_MPDT == CSM_C_F_ON)
    fRetVal = CSM_MPDT_fCheckCommunicationReq(bBus);
    if (fRetVal != FALSE)
    {
      fBusUp = TRUE;
    }
    #endif

    #if (CSM_S_MCNET == CSM_C_F_ON)
    fRetVal = CSM_MCNET_fCheckCommunicationReq(bBus);
    if (fRetVal != FALSE)
    {
      #if (CSM_S_OSEK_NM == CSM_C_F_ON )
        #if ( CNM_OSEK_S_VW_PAYLOAD == CSM_C_F_ON )
        // wecken ber CSM_lCommunicationReq()
        CNM_OSEK_lSetPayload( bBus, CNM_OSEK_C_NMINFO_WAKEUP_COM_REQ);
        #endif // CNM_OSEK_S_VW_PAYLOAD
      #endif // CSM_S_OSEK_NM
      fBusUp = TRUE;
    }
    #endif

    #if (CSM_S_ISO_TP == CSM_C_F_ON)
      #if (CSM_S_ITP_COMMUNICATION_FCT == CSM_C_F_ON)
      fRetVal = CSM_ITP_fCheckCommunicationReq( bBus);
      if (fRetVal != FALSE)
      {
        #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
          #if ( CSM_S_OSEK_NM == CSM_C_F_ON )
            #if ( CNM_OSEK_S_VW_PAYLOAD == CSM_C_F_ON )
            // wecken ber CSM_lCommunicationReq()
            CNM_OSEK_lSetPayload( bBus, CNM_OSEK_C_NMINFO_WAKEUP_COM_REQ);
            #endif // CNM_OSEK_S_VW_PAYLOAD
          #endif // CSM_S_OSEK_NM
        #endif // CSM_S_CNM_AVAILABLE
        fBusUp = TRUE;
      }
      #endif // CSM_S_ITP_COMMUNICATION_FCT
    #endif // CSM_S_ISO_TP

    #if (CSM_S_J1939 == CSM_C_F_ON)
    fRetVal = CSM_J1939_fCheckCommunicationReq(bBus);
    if (fRetVal != FALSE)
    {
      fBusUp = TRUE;
    }
    #endif

    #if (CSM_S_VW_TP_2_0 == CSM_C_F_ON)
    // #ToDo:
    // fRetVal = CSM_VWTP20_fCheckCommunicationReq(bBus); + OSEK-Payload
    if (fRetVal != FALSE)
    {
      fBusUp = TRUE;
    }
    #endif
  }
  #endif // CSM_S_CNP_AVAILABLE

  return fBusUp;
}

/*******************************************************************************
 * function    CSM_bCalcBusState
 * \doxydocu
 * \brief      Calculate bus state
 *
 *             This function is call from CSM_vProcessBusStateChange() to calculate the bus state.
 *
 * \param[in]  bBus
 *              Bus number
 *
 * \return     void
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview 24.07.08  VTeam-Zurmhl
 *
 * \history_begin
 * 19.10.04  CM-DI/ESA3-Prhl
 * - Initial version
 * \history_end
 ******************************************************************************/
CSM_STATIC BYTE CSM_bCalcBusState(BYTE bBus)
{
  BYTE bNewBusState = CSM_C_STATE_BUS_DOWN;
  BYTE bCsmBusState;

  #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
  BYTE bCnmBusState;
  #endif

  // 1. Wurde von einer Applikation der Bus angefordert?
  //    Wenn ja dann ist CSM_C_STATE_BUS_UP die Antwort.
  // 2. In welchen Zustand mchte das Netzwerkmanagement wechseln?
  //    Wenn das Netzwerkmanagement nach CSM_C_STATE_BUS_DOWN will dann geht es bei 3. weiter. Andernfalls
  //    wird in den Busstate gewechselt in den das Netzwerkmanagement will.
  // 3. Mchte der CSM selbst den Busstate vorgeben, z.B. weil ein Weckgrund vorlag und der Bus N Sekunden
  //    in CSM_C_STATE_BUS_LISTEN gehalten werden soll oder weil einmalig nach CSM_C_STATE_BUS_SHUTDOWN/
  //    CSM_C_EVENT_BUS_STARTUP gewechselt werden soll.

  #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
  // determine the NWM bus state
  bCnmBusState = CNM_bCalculateBusState(bBus);
  #endif

  // determine the project specific bus state of the CSM
  bCsmBusState = csm_bCalculateBusState(bBus);

  // bus may not change to BUS_UP immediately with J1939_NM (only after AddrClaim)
  #if (CSM_S_J1939_NM == CSM_C_F_OFF)
  // if communication requests activated and any application needs bus:
  if( (!(csm_fIgnoreCommReq)) && CSM_fCheckCommunicationReq( bBus)) /*lint !e774 !e506 PQM_authorized_127 csm_fIgnoreCommReq is FALSE if CSM_S_DEACTIVATE_COMMUNICATION_REQUESTS if CSM_C_F_OFF */
  {
    // eine Applikation braucht den Bus
    bNewBusState = CSM_C_STATE_BUS_UP;
    #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
      #if (CSM_S_VWHIGH_NM == CSM_C_F_ON)
      CNM_VWHIGH_vSetPayload(bBus, CNM_VWHIGH_C_NMINFO_ACTIVE_WAKEUP_BIT );
      #endif // CSM_S_VWHIGH_NM
    #endif // CSM_S_CNM_AVAILABLE
  }
  else
  #endif // CSM_S_J1939_NM == CSM_C_F_OFF !!!
  {
    #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
      #if (CSM_S_VWHIGH_NM == CSM_C_F_ON)
      CNM_VWHIGH_vClrPayload(bBus, CNM_VWHIGH_C_NMINFO_ACTIVE_WAKEUP_BIT);
      #endif // CSM_S_VWHIGH_NM
    #endif // CSM_S_CNM_AVAILABLE

    #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
    bNewBusState = bCnmBusState;

    if( bNewBusState == CSM_C_STATE_BUS_DOWN)
    {
      bNewBusState = bCsmBusState; // from csm_bCalculateBusState
    }
    #else  // CSM_S_CNM_AVAILABLE
    bNewBusState = bCsmBusState;   //lint !e838 yes, sometime a variable get a new value
    #endif // CSM_S_CNM_AVAILABLE

    if( bCsmBusState == CSM_C_STATE_BUS_UP)
    {
      // UP gewinnt immer
      bNewBusState = bCsmBusState;
    }
  }

  #if (CSM_S_TRACE == CSM_C_F_ON)
    #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
    {
      #if (CSM_S_CNM_AVAILABLE == CSM_C_F_OFF)
      BYTE bCnmBusState = 0;
      #endif

      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR2(("CSM_bCalcBusState: Bus:%x  CSMState:%x  NMState:%x  ResultState:%x  IgnoreCommReq:%x",
      ETG_CENUM(tCSM_BUS_NUMBER, (tU8)  bBus),
      ETG_CENUM(tCSM_BUS_STATE,  (tU8)  bCsmBusState),
      ETG_CENUM(tCSM_BUS_STATE,  (tU8)  bCnmBusState),
      ETG_CENUM(tCSM_BUS_STATE,  (tU8)  bNewBusState),
      csm_fIgnoreCommReq ));
      #endif
    }
    #else
    {
      // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
      BYTE abData[5];

      abData[0] = bBus;
      abData[1] = bCsmBusState;
      #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
        abData[2] = bCnmBusState;
      #else
        abData[2] = 0;
      #endif
      abData[3] = bNewBusState;
      abData[4] = (BYTE)csm_fIgnoreCommReq;

      CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_CSM_INTERNAL, CSM_C_TRACE_FKT_CSM_CALC_BUSSTATE, abData, 5);
    }
    #endif
  #endif

  return bNewBusState;
}
#endif // CSM_C_DOWN_REASON_CSM_BUS_STATE_CHANGED

/*******************************************************************************
 * function implementation (scope: CAN stack and operating system)
 *******************************************************************************/
#ifdef CSM_C_DOWN_REASON_CSM_BUS_STATE_CHANGED
/*******************************************************************************
 * function    CSM_vProcessBusStateChange
 * \doxydocu
 * \brief      Change Busstate if necessary
 *
 *             This function is call from downstream-task to change the bus state.
 *             The trigger is CSM_C_DOWN_REASON_CSM_BUS_STATE_CHANGED
 *
 * \param[in]  bBus
 *              Bus number
 *
 * \return     void
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview  11.12.09 Prhl
 *
 * \history_begin
 * 26.02.09  VTeam-Khler
 * - lReachedState is set to bNewState if VNs have changed but not the bus state.
 *
 * 15.07.08  VTeam-Khler
 * - lReachedState is set to bNewState if CSM_S_STATE_MASCHINE != CSM_C_F_ON
 *
 * 19.06.08  VTeam-Khler
 * - CSM_lBusStateChange() calls encapsulated with feature switch
 *   CSM_S_STATE_MASCHINE
 *
 * 05.03.08  VTeam Brunotte
 * - workflow changed : 1 calculate new states
 *                      2 on state change set Indication flags (seperatly for VN and bus state)
 *                      3 trace out bus state changed message
 *                      4 store new states (VN and bus state )
 *                      5 Call all callbacks with new states, if flags are set.
 * 13.02.08  VTeam Brunotte
 * - Static variable bVN changed to array abVN with dependece to CSM_C_MAX_CAN_BUS
 * 26.11.07  VTeam-Khler
 * - Indentation according VTeam rules.
 * - Trace databytes for virtual networks corrected.
 *
 * 14.06.06  CM-DI/ESA3-Prhl
 * virtual network support added
 *
 * 16.06.05  CM-DI/ESA3-K"ohler
 * if (csm_avpfBusStateInd[ dwCount] != NULL) added.
 *
 * 19.10.04  CM-DI/ESA3-Prhl
 * - Initial version
 * \history_end
 ******************************************************************************/
void CSM_vProcessBusStateChange(BYTE bBus)
{
  #if (CSM_S_REMOTE_STACK == CSM_C_F_ON)
  //TODO MP: Wenn CSM_C_DOWN_REASON_CSM_BUS_STATE_CHANGED entfernt wurde, dann das #ifdef CSM_S_REMOTE_STACK wieder entfernen.
  return;
  #else // CSM_S_REMOTE_STACK
  BYTE bNewState;
  LONG lReachedState = 0;
  DWORD dwCount;
  BOOL fStateIndication = FALSE;

  #if (CSM_S_BUS_WAKE_MIN == CSM_C_F_ON)
  BYTE bOldState = csm_abCsmSubState[ bBus];
  #endif /*CSM_S_BUS_WAKE_MIN == CSM_C_F_ON */

  #if (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
  static BYTE abVN [CSM_C_MAX_CAN_BUS]= {0, 0};
         BYTE bNewVN = CNM_GMLAN_bGetVN( bBus);
  BOOL fVNIndication = FALSE;
  #endif /* CSM_S_VIRTUAL_NETWORK */

  /* determine the new bus state                                              */
  bNewState = CSM_bCalcBusState( bBus);

  /* handle some NM things -------------------------------------------------- */
  #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
  #if ( CSM_S_OSEK_NM == CSM_C_F_ON )
    #if ( CNM_OSEK_S_VW_PAYLOAD == CSM_C_F_ON )
      #if (CSM_S_FNL == CSM_C_F_ON)
        if( CSM_CBR_afFnlWake[ bBus] == TRUE)
        {
          CNM_OSEK_lSetPayload( bBus, CNM_OSEK_C_NMINFO_NL_FNL);
        }
        else
        {
          CNM_OSEK_lClrPayload( bBus, CNM_OSEK_C_NMINFO_NL_FNL);
        }
      #endif /* (CSM_S_FNL == CSM_C_F_ON) */

      #if (CSM_S_BUS_WAKE_TEMP == CSM_C_F_ON)
        if( csm_fWakeTemp == TRUE)
        {
          CNM_OSEK_lSetPayload( bBus, CNM_OSEK_C_NMINFO_WAKE_TEMP);
        }
        else
        {
          CNM_OSEK_lClrPayload( bBus, CNM_OSEK_C_NMINFO_WAKE_TEMP);
        }
      #endif

      #if (CSM_S_BUS_WAKE_MIN == CSM_C_F_ON)
        if( csm_afWakeMin[ bBus] == TRUE)
        {
          CNM_OSEK_lSetPayload( bBus, CNM_OSEK_C_NMINFO_MIN_ACTIVE_TIME);
        }
        else
        {
          CNM_OSEK_lClrPayload( bBus, CNM_OSEK_C_NMINFO_MIN_ACTIVE_TIME);
        }
      #endif /* (CSM_S_BUS_WAKE_MIN == CSM_C_F_ON) */
    #endif /* CNM_OSEK_S_VW_PAYLOAD == CSM_C_F_ON */
  #endif /* CSM_S_OSEK_NM == CSM_C_F_ON */
  #endif /* CSM_S_CNM_AVAILABLE */

  /* handle some virtual network things ------------------------------------- */
  #if (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
    if (bNewVN != abVN[bBus])                           /* VNs changed */
    {
      fVNIndication = TRUE;
    }
  #endif /* CSM_S_VIRTUAL_NETWORK */

  /* handle a bus state change ---------------------------------------------- */
  if( csm_abCsmSubState[ bBus] != bNewState)
  {
    /* the bus state has to be changed                                        */
    #if (CSM_S_STATE_MASCHINE == CSM_C_F_ON)
    /* trigger our statemachine with state change event                       */
    lReachedState = CSM_lBusStateChange( bBus, bNewState);
    #else
    lReachedState = bNewState;
    #endif /* CSM_S_STATE_MASCHINE */

    #if (CSM_S_BUS_WAKE_MIN == CSM_C_F_ON)
    /* always extend up time if bus is leaving the DOWN state                 */
    if( ((bOldState == CSM_C_STATE_BUS_UNDEFINED) ||
         (bOldState == CSM_C_STATE_BUS_DOWN) ||
         (bOldState == CSM_C_STATE_BUS_LISTEN)) &&
        ((lReachedState == CSM_C_STATE_BUS_STARTUP) ||
         (lReachedState == CSM_C_STATE_BUS_UP) ||
         (lReachedState == CSM_C_STATE_BUS_SHUTDOWN) ) )
    {
      /* old state was more or less DOWN, the new state is more or less UP    */
      /* so turn on the bus for its minimal ON time                           */
      csm_afWakeMin[ bBus] = TRUE;
      csm_vTriggerCsmDownstreamTask( CSM_C_DOWN_REASON_CSM_BUS_STATE_CHANGED);
      CSM_lTimerStart( CSM_TIMER_WAKE_MIN_0 + bBus, CSM_C_WAKE_MIN_TIME);
    }
    #endif /* (CSM_S_BUS_WAKE_MIN == CSM_C_F_ON) */

    /* evaluate the current state                                             */
    if ((lReachedState > 0) && (lReachedState <= 5))
    {
      fStateIndication = TRUE;
    }
    else
    {
      /* something's going wrong with that state transition request */
      if(lReachedState < 0) /* negative value, means error */
      {
        if((CSM_M_GET_EG(lReachedState) == CSM_C_ERR_ERROR) || (CSM_M_GET_EG(lReachedState) == CSM_C_ERR_SEVERE_ERROR))
        {
          CSM_vAssert(FALSE);
        }
      }
    }
  }
  #if (CSM_S_TRACE == CSM_C_F_ON) && (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
  else if (fVNIndication == TRUE)
  {
    /* Up to now lReachedState is 0 (undefined) and this will be traced out.  */
    /* This is no functional problem but it can lead to confusion.            */
    lReachedState = bNewState;
  }
  #endif /* (CSM_S_TRACE == CSM_C_F_ON) && (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON) */

  /* trace the state change ------------------------------------------------- */
  #if (CSM_S_TRACE == CSM_C_F_ON)
    #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      if ((fStateIndication == TRUE)
        #if (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
         || (fVNIndication == TRUE)
        #endif
         )
      {
        #if (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
        ETG_TRACE_USR2(("Bus state changed      --> Bus: %x State: %x NewState: %x Reached: %x VN: %x NewVN: %x",
        ETG_CENUM(tCSM_BUS_NUMBER,    (tU8)  bBus),
        ETG_CENUM(tCSM_BUS_STATE,     (tU8)  csm_abCsmSubState[ bBus]),
        ETG_CENUM(tCSM_BUS_STATE,     (tU8)  bNewState),
        ETG_CENUM(tCSM_BUS_STATE,     (tU8)  lReachedState),
        (tU8) abVN[bBus],
        (tU8) bNewVN ));
        #else
        ETG_TRACE_USR2(("Bus state changed      --> Bus: %x State: %x NewState: %x Reached: %x",
        ETG_CENUM(tCSM_BUS_NUMBER,    (tU8)  bBus),
        ETG_CENUM(tCSM_BUS_STATE,     (tU8)  csm_abCsmSubState[ bBus]),
        ETG_CENUM(tCSM_BUS_STATE,     (tU8)  bNewState),
        ETG_CENUM(tCSM_BUS_STATE,     (tU8)  lReachedState) ));
        #endif /* CSM_S_VIRTUAL_NETWORK */
      }
      #endif
    #else
    if ((fStateIndication == TRUE)
      #if (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
        || (fVNIndication == TRUE)
      #endif
       )
    {
      #if (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
        #define CSM_C_LOCAL_NUMBER_OF_TRACE_BYTES 6
      #else
        #define CSM_C_LOCAL_NUMBER_OF_TRACE_BYTES 4
      #endif
      // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
      BYTE abData[CSM_C_LOCAL_NUMBER_OF_TRACE_BYTES]={0};

      abData[0] = bBus;
      abData[1] = csm_abCsmSubState[bBus];
      abData[2] = bNewState;
      abData[3] = (BYTE)lReachedState;
      #if (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
        abData[4] = abVN[bBus];
        abData[5] = bNewVN;
      #endif // CSM_S_VIRTUAL_NETWORK
      CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_CSM_INTERNAL, CSM_C_TRACE_FKT_CSM_BUSSTATE_CHANGED, abData, CSM_C_LOCAL_NUMBER_OF_TRACE_BYTES);
    }
    #endif
  #endif // CSM_S_TRACE

  // Store new bus state
  if (fStateIndication == TRUE)
  {
    csm_abCsmSubState[bBus] = (BYTE)lReachedState; // Der erreichte Busstate wird gemerkt
  }

  #if (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
  // Store new VN
  if (fVNIndication == TRUE)
  {
    abVN[bBus] = bNewVN;
  }
  #endif

  if ((fStateIndication == TRUE)
      #if (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
      || (fVNIndication == TRUE)
      #endif
     )
  {
    // call BusStateInd-Callbackfkt
    for (dwCount = 0; dwCount < CSM_M_ELEMENTS( csm_avpfBusStateInd); dwCount++)
    {
      if (csm_avpfBusStateInd[dwCount] != NULL)
      {
        #if (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
        csm_avpfBusStateInd[dwCount](bBus, csm_abCsmSubState[bBus], abVN[bBus]);
        #else
        csm_avpfBusStateInd[dwCount](bBus, csm_abCsmSubState[bBus]);
        #endif // CSM_S_VIRTUAL_NETWORK
      }
    }
  }
  #endif // CSM_S_REMOTE_STACK
}/*lint !e715 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally*/
#endif /* CSM_C_DOWN_REASON_CSM_BUS_STATE_CHANGED */

/*******************************************************************************
 * function    CSM_lBusStateChange
 * \doxydocu
 * \brief      Request to change the bus state (CSM substate)
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \param[in]  bBus
 *              Bus number
 *
 * \param[in]  bCsmTargetState
 *              Requested CSM substate (= bus state):
 *              CSM_C_STATE_BUS_DOWN, CSM_C_STATE_BUS_LISTEN, CSM_C_STATE_BUS_UP
 *
 * \return     Error value or act state.
 *
 * \access     interrupt
 * \reentrant  yes|no
 *
 * \lastreview 21.03.11 Borck
 *             13.10.08 VTeam-Funke
 *             29.05.06 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 13.10.08 VTeam-Funke
 * - align return value to same name
 *
 * 03.05.05 CM-DI/ESU3-Zurmuehl
 * CSM_API modifier removed (no API function any longer)
 *
 * 27.02.04 CM-DI/ESU3-Koehler
 * Target states startup and shutdown added.
 *
 * 16.10.03 CM-DI/ESU3-Zurmuehl
 * CSM_API modifier added to function definition
 *
 * 25.02.02 CM-DI/ECHS3-Pistoor
 * Added new state machine and adapted this function
 *
 * 10.01.02 CM-DI/ECC13-Koehler
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_STATE_MASCHINE == CSM_C_F_ON)
LONG CSM_lBusStateChange(BYTE bBus, BYTE bCsmTargetState)
{
  LONG lRet = CSM_C_NO_ERROR;
  BYTE bEvent;

  switch(bCsmTargetState)
  {
    case CSM_C_STATE_BUS_DOWN:
      bEvent = CSM_C_EVENT_BUS_DOWN;
      break;

    case CSM_C_STATE_BUS_LISTEN:
      bEvent = CSM_C_EVENT_BUS_LISTEN;
      break;

    case CSM_C_STATE_BUS_STARTUP:
      bEvent = CSM_C_EVENT_BUS_STARTUP;
      break;

    case CSM_C_STATE_BUS_UP:
      bEvent = CSM_C_EVENT_BUS_UP;
      break;

    case CSM_C_STATE_BUS_SHUTDOWN:
      bEvent = CSM_C_EVENT_BUS_SHUTDOWN;
      break;

    default:
      /* exit function with error code */
      return CSM_M_MAKE_CSM_ERROR( CSM_C_ERR_WARNING, CSM_E_UNKNOWN_STATE_CHANGE_COMMAND);
  };

  lRet = csm_lSwitchStateMachine(bBus, csm_abCsmSubState[bBus], bEvent); /*lint !e838 yes, sometime a variable get a new value*/

  return lRet;
} /* end CSM_lBusStateChange() */
#endif /* CSM_S_STATE_MASCHINE */

/*******************************************************************************
 * function    CSM_vCDAErrorInd
 * \doxydocu
 * \brief      Bus error indication function
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \param[in]  bBus
 *              Bus number
 *
 * \param[in]  bErrorType
 *                CSM_C_BUS_OFF
 *                CSM_C_TRANSC_ERR_STABLE
 *                CSM_C_NO_NETWORK
 *
 * \param[in]  dwNumberOfRepeatedBusoff
 *                ???
 *
 * \return     VOID
 *
 * \access     interrupt
 * \reentrant  yes|no
 *
 * \lastreview 24.07.08  VTeam-Zurmhl ..Finding: Fct. still a little bit incomplete
 *
 * \history_begin
 * 30.08.04 CM-DI/ESA3-Battis
 * retval removed because function is void
 *
 * 14.05.03 CM-DI/ESU3-Pistoor
 * - Added functionality for busoff indication
 * - Added parameter dwNumberOfRepeatedBusoff
 *
 * 04.04.01 K7/EFG13-Pistoor
 * Initial Revision.
 * \history_end
 ******************************************************************************/
/*-
 * The VW instance of the CAN-Stack doesn't use the current bus-off
 * treatment of the CSM. Because state changes of the CSM state maschine
 * are not allowed from interrupt level!
 * The design [CAN-SWA] of the bus-off treatment is still a work in progress
 * as well.
 * I'll implement the specified treatment, if it is fixed in the CAN-SWA
 * documentation.
 *
 * 28.07.2004 CM-CR/ESD3 Thomas Mehring
 */
void CSM_vCDAErrorInd(BYTE bBus, BYTE bErrorType, DWORD dwNumberOfRepeatedBusoff)
{
  /* TODO: was machen wir hier mit dwNumberOfRepeatedBusoff ?? */
  #ifdef CSM_C_DOWN_REASON_CDA_BUSOFF_ENTRY
  if(bErrorType & CSM_C_BUS_OFF)
  {
    csm_bBusOffState[ bBus].fEntryIndication = TRUE;
    csm_bBusOffState[ bBus].fState = TRUE;
    csm_vTriggerCsmDownstreamTaskFromIrq( CSM_C_DOWN_REASON_CDA_BUSOFF_ENTRY);
  }
  else
  #endif
  {
    /* nothing except busoff handled here up to now !*/
    CSM_vAssert(FALSE);
  }

} /* CSM_vCDAErrorInd() *//*lint !e715 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally*/

#endif /* CSM_S_DOXYGEN_HIDE_NON_API_FCT */

/*******************************************************************************
 * function implementation (scope: CAN stack #1 API)
 ******************************************************************************/

/*******************************************************************************
 * function    CSM_lInit
 * \doxydocu
 * \brief      Init the CAN stack.
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \param[in]  bInitMode
 *              CSM_C_INIT_COLD, CSM_C_INIT_WARM, CSM_C_INIT_START
 *
 * \return     Error value.
 *
 * \access     application
 * \reentrant  no
 *
 * \lastreview 11.12.09 Prhl
 *             13.10.08 VTeam-Funke ..finding problems in error handling of case CSM_C_INIT_START
 *             16.04.08 VTeam-Prhl
 *
 * \history_begin
 *
 * 23.07.09 VTeam-Jablonski
 * - Init of csm_arBusErrorIndCallbackFkt moved to CSM_lnit
 *
 * 13.10.08 VTeam-Funke
 * - overworked error handling in case of CSM_C_INIT_START
 *
 * 22.02.08 VTeam-Prhl
 * csm_bCsmState is not set if one of the ...lInit() calls has failed.
 * 21.06.05 CM-DI/ESA3-K"ohler
 * Initialization of ITP variables for upstream task indication moved from
 * CSM_lInit() to CSM_ITP_lInit().
 * csm_vInitApplTelRxQueue added for warm init.
 *
 * 16.06.05 CM-DI/ESA3-K"ohler
 * csm_rLocalControlState initialized in cold init.
 *
 * 11.10.04 CM-DI/ESA3-K"ohler
 * csm_vTriggerCsmDownstreamTask( CSM_C_DOWN_REASON_CSM_POST_INIT_START ) call
 * encapsulated with feature switch.
 *
 * 02.03.04 CM-DI/ESU3-K"ohler
 * - csm_vFordInitStart() moved from csm.c to Ford CSM_lInitApl().
 *
 * 27.02.04 CM-DI/ESU3-K"ohler
 * CSM_lInitEnv(CSM_C_INIT_START) moved up to second position after CSM init.
 * The CAN busses will be activated after initialization only if activity was
 * detected before.
 *
 * 15.12.03 CM-DI/ESU3-K"ohler
 * Initialization of MCNet variables moved to CSM_MCNET_lInit.
 *
 * 16.10.03 CM-DI/ESU3-Zurmuehl
 * CSM_API modifier added to function definition
 *
 * 06.10.03 CM-DI/ESU3-K"ohler
 * MCNet added.
 *
 * 15.07.03 CM-DI/ESU3-K"ohler
 * Init order standardized.
 * CSM_lInitApl(CSM_C_INIT_COLD) added.
 *
 * 14.05.03 CM-DI/ESU3-K"ohler
 * CSM stack manager states are checked.

 * 18.09.02 CM-DI/EHS3-K"ohler
 * Bus state change to listen mode moved to CSM_vStart().
 *
 * 12.09.02 CM-DI/EHS3-K"ohler
 * Two variables for application indication added.
 *
 * 10.04.02 CM-DI/EHS3-K"ohler
 * Init() for different CNMs and CNPs added.
 *
 * 20.02.02 CM-DI/EHS3-Pistoor
 * Added functionality for Broadcast Server and Fiat network management
 *
 * 10.01.02 CM-DI/EHS3-K"ohler
 * Initial Revision.
 * \history_end
 ******************************************************************************/
CSM_API LONG CSM_lInit(BYTE bInitMode)
{
  LONG lRet = CSM_C_NO_ERROR;
  BYTE bBusIndex;
  DWORD dwCount;
  #if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
  BYTE bUserNumber;
  #endif /*  CSM_S_ENABLE_INDICATE_BUSERROR*/

  /***************  CSM_C_INIT_COLD *******************************************/
  if (bInitMode == CSM_C_INIT_COLD)
  {
    csm_bCsmState = CSM_C_STATE_UNDEFINED;
    csm_fStateMachineRuns = FALSE;
    dwCSM_EmTraceError = 0;

    #if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
    for (bBusIndex = 0; bBusIndex < CSM_C_MAX_CAN_BUS; bBusIndex++)
    {
      csm_abActBusError[ bBusIndex] = 0;
      csm_afTriggerErrorInd[ bBusIndex] = FALSE;
    }
    #endif

    #if (CSM_S_ENABLE_INDICATE_STATIC_ERROR == CSM_C_F_ON)
    for (bBusIndex = 0; bBusIndex < CSM_C_MAX_CAN_BUS; bBusIndex++)
    {
      csm_lActStaticError[ bBusIndex] = CSM_C_NO_ERROR;
    }
    #endif

    for( dwCount = 0; dwCount < CSM_M_ELEMENTS( csm_alpfInit); dwCount++)
    {
      CSM_M_SET_HIGHEST_ERROR_GRADE(lRet, csm_alpfInit[ dwCount]( CSM_C_INIT_COLD));
    }

    /* Set states */
    for (bBusIndex=0; bBusIndex < CSM_C_MAX_CAN_BUS; bBusIndex++) /* VK: Loop here or in warm init ??? */
    {
      csm_abCsmSubState[bBusIndex] = CSM_C_STATE_BUS_UNDEFINED;
      #if (CSM_S_DEBUG == CSM_C_F_ON)
      csm_bNumberOfBusoff[bBusIndex] = 0;
      #endif
    }

    #if (CSM_S_CAN_WAKEUP_FUNCTIONS_VIA_CSM == CSM_C_F_ON)
    CSM_bCanWasWakeupReason = 0;        /* clear all wakeup reasons */
    #endif /* CSM_S_CAN_WAKEUP_FUNCTIONS_VIA_CSM */

    if(lRet >= CSM_C_NO_ERROR)
    {
      csm_bCsmState = CSM_C_STATE_OFF;
    }
    else
    {
      CSM_vAssert(FALSE);
    }
  }
  /***************  CSM_C_INIT_WARM *******************************************/
  else if (bInitMode == CSM_C_INIT_WARM)
  {
    if (csm_bCsmState == CSM_C_STATE_OFF)
    {
      #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
      csm_fActivationFlag = FALSE;
      #endif
      csm_fStateMachineRuns = FALSE;
      #if (CSM_S_BLOCKABLE_UPSTREAM_TASK == CSM_C_F_ON)
      csm_fBlockUpstreamTask = FALSE;
      #endif

      for (bBusIndex=0; bBusIndex < CSM_C_MAX_CAN_BUS; bBusIndex++)
      {
        csm_bBusOffState[bBusIndex].fEntryIndication = FALSE;
        csm_bBusOffState[bBusIndex].fRecoveryIndication = FALSE;
        csm_bBusOffState[bBusIndex].fState = FALSE;
      }

      /* Init variables for Downstream Task                                   */
      csm_dwDownReasonBitfield = 0;
      /* Init variables for Upstream Task                                     */
      csm_dwIndicationReasonBitfield = 0;

      #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
      #if (CSM_S_ISO_TP == CSM_C_F_ON)
      CSM_ITP_vResetIndicationFlags();
      #endif
      #endif

      #if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
      #if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
      for(bUserNumber=0; bUserNumber < CSM_C_MAX_ERROR_IND_USER; bUserNumber++)
      {
        csm_arBusErrorIndCallbackFkt[ bUserNumber].pfvBusErrorInd = NULL;
        #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
        csm_arBusErrorIndCallbackFkt[ bUserNumber].pvHandle = NULL;
        #endif
      }
      #endif /*  CSM_S_CALLBACK_INIT_BY_APPLICATION*/
      #endif /*  CSM_S_ENABLE_INDICATE_BUSERROR*/

      for(dwCount = 0; dwCount < CSM_M_ELEMENTS( csm_alpfInit); dwCount++)
      {
        CSM_M_SET_HIGHEST_ERROR_GRADE(lRet, csm_alpfInit[ dwCount]( CSM_C_INIT_WARM));
      }

      if (lRet >= CSM_C_NO_ERROR)
      {
        csm_bCsmState = CSM_C_STATE_INITIALIZED;
      }
      else
      {
        CSM_vAssert(FALSE);
      }
    }
    else  /* invalid CSM state for warm init */
    {
      CSM_vAssert(FALSE); /* warm init without cold init or off */
      lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_ERROR_STATE_WHILE_WARM_INIT);
    }
  }
  /***************  CSM_C_INIT_START *******************************************/
  else if (bInitMode == CSM_C_INIT_START)
  {
    if (csm_bCsmState == CSM_C_STATE_INITIALIZED)
    {
      for( dwCount = 0; dwCount < CSM_M_ELEMENTS( csm_alpfInit); dwCount++)
      {
        CSM_M_SET_HIGHEST_ERROR_GRADE(lRet, csm_alpfInit[ dwCount]( CSM_C_INIT_START));
      }

      for (bBusIndex=0; bBusIndex < CSM_C_MAX_CAN_BUS; bBusIndex++)
      {
        csm_abCsmSubState[bBusIndex] = CSM_C_STATE_BUS_DOWN;
      }

      if (lRet >= CSM_C_NO_ERROR)
      {
        csm_bCsmState = CSM_C_STATE_ACTIVE;    /* CSM stack initialization finished */

        #ifdef CSM_C_DOWN_REASON_CSM_STACK_ACTIVATED
        /* Call CSM_vStackActivated() on downstreamtask level. */
        csm_vTriggerCsmDownstreamTask( CSM_C_DOWN_REASON_CSM_STACK_ACTIVATED );
        #else
        /* Call CSM_vStackActivated() right here. */
        /* This function must be implemented in csm_p.c */
        CSM_vStackActivated(); /* Inform sub components about stack activation. */
        #endif

        #ifdef CSM_C_DOWN_REASON_CSM_BUS_STATE_CHANGED
        #if ( CSM_S_CSM_VW == CSM_C_F_OFF )
        csm_vTriggerCsmDownstreamTask( CSM_C_DOWN_REASON_CSM_BUS_STATE_CHANGED); /* Trigger bus state handler */
        #endif
        #endif

        #if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
        /*-
         * REMARK:
         * After restart inform other components (especially HMI processor),
         * about current (last known) bus error state.
         * Because it may be possible, that a bus error indication has
         * gone astray, if it was generated in the shutdown phase (HMI already
         * off). (Remark: CCS queues are cleared with ...start())
         *
         * 30.06.2005 (CM-CR/ESD3) Thomas Mehring
         */

        csm_vTriggerCsmUpstreamTask(CSM_C_IND_REASON_BUS_ERROR);
        #endif
      }
      else
      {
        CSM_vAssert(FALSE);
      }
    }
    else /* CAN stack != initialized */
    {
      CSM_vAssert(FALSE); /* no init or stop before start */
      lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_ERROR_STATE_WHILE_START);
    }
  } /* CSM_C_INIT_START */
  /****************************************************************************/
  else
  {
    /* undefined init mode, so return error                                   */
    lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_ERROR_STATE_WHILE_INIT);
  }

  return lRet;
} /* end CSM_lInit() */

/*******************************************************************************
 * function    CSM_lExit
 * \doxydocu
 * \brief      Switch CAN stack to inactive.
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \param[in]  bExitMode
 *               CSM_C_EXIT_STOP, CSM_C_EXIT_OFF
 *
 * \return     Error value.
 *
 * \access     application
 * \reentrant  no
 *
 * \lastreview 21.03.11 Borck
 *             13.10.08 VTeam-Funke
 *             29.05.06 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 13.10.08 VTeam-Funke
 * - align return value to same name
 *
 * 17.11.03 CM-DI/ESU3-Zurmuehl
 * CSM_lMCNet_Exit() added
 * CSM_API modifier added to function definition
 *
 * 11.08.03 CM-DI/ESU3-Pistoor
 *  Moved function calls of CSM_vStopRxCBRManager() and CSM_vStopTxCBRManager()
 * to CBR_lTxStop() and CBR_lRxStop()
 *
 * 14.05.03 CM-DI/ESU3-Koehler
 * CSM stack manager states are checked.
 *
 * 22.01.03 CM-DI/EHS3-Koehler
 * csm_abCsmSubStates are set to CSM_C_STATE_BUS_DOWN for CSM_C_EXIT_STOP.
 *
 * 10.04.02 CM-DI/ECC13-Koehler
 * Exit() for different CNMs and CNPs added.
 *
 * 20.02.02 CM-DI/EHS3-Pistoor
 * Added functionality for Broadcast Server and Fiat network management
 *
 * 10.01.02 CM-DI/ECC13-Koehler
 * Initial Revision.
 * \history_end
 ******************************************************************************/
CSM_API LONG CSM_lExit(BYTE bExitMode)
{
  LONG lRet = CSM_C_NO_ERROR;
  DWORD dwCount;

  /* ToDo: wenn noch ein Bus an ist, dann fahr ihn zwangsweise herunter       */

  /***********************  CSM_C_EXIT_STOP ***********************************/
  if (CSM_C_EXIT_STOP == bExitMode)
  {
    if (CSM_C_STATE_ACTIVE == csm_bCsmState)
    {
      BYTE bBusIndex;

      /* loop over all configured exit functions                              */
      for (dwCount = 0; dwCount < CSM_M_ELEMENTS( csm_alpfExit); dwCount++)
      {
        CSM_M_SET_HIGHEST_ERROR_GRADE(lRet, csm_alpfExit[ dwCount]( CSM_C_EXIT_STOP));
      }

      /* set "bus down" states for all busses                                 */
      for (bBusIndex=0; bBusIndex < CSM_C_MAX_CAN_BUS; bBusIndex++)
      {
        csm_abCsmSubState[bBusIndex] = CSM_C_STATE_BUS_DOWN;
        #if (CSM_S_DEBUG == CSM_C_F_ON)
        csm_bNumberOfBusoff[bBusIndex] = 0;
        #endif
      }
      csm_bCsmState = CSM_C_STATE_INITIALIZED;
    }
    else
    {
      /* error: CSM was not started correctly before                          */
      CSM_vAssert(FALSE);
      lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_ERROR_STATE_WHILE_STOP);
    }
  }
  /***********************  CSM_C_EXIT_OFF  ***********************************/
  else if (CSM_C_EXIT_OFF == bExitMode)
  {
    if (CSM_C_STATE_ACTIVE == csm_bCsmState)
    {
      /* error: off without stop                                              */
      CSM_vAssert(FALSE);
      /* CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_ERROR_STATE_WHILE_OFF);  */
      /* correct the missing transition by doing it now                       */
      lRet = CSM_lExit(CSM_C_EXIT_STOP);
      /* now in state initialized                                             */
    }

    if (CSM_C_STATE_INITIALIZED == csm_bCsmState)
    {
      /* loop over all configured exit functions                              */
      for (dwCount = 0; dwCount < CSM_M_ELEMENTS(csm_alpfExit); dwCount++)
      {
        CSM_M_SET_HIGHEST_ERROR_GRADE(lRet, csm_alpfExit[ dwCount]( CSM_C_EXIT_OFF));
      }

      csm_bCsmState = CSM_C_STATE_OFF;
      /* lRet stays at it is: default value no error or return value of CSM_lExit */
    }
  }
  /****************************************************************************/
  else
  {
    /* invalid exit mode, so return an error                                  */
    lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_ERROR_STATE_WHILE_EXIT);
  }

  return lRet;
} /* end CSM_lExit() */

#if ((CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON) || (CSM_S_VW_BAP_TP == CSM_C_F_ON))
/*******************************************************************************
 * function    CSM_lApplCallbackInit
 * \doxydocu
 * \brief
 *
 * \param[in]  pvHandle
 *              pointer to handle. For future use! May be a deinit() function
 *
 * \param[in]  bBus (I)
 *             CAN bus number
 *
 * \param[in]  dwProtocolType
 *              protocol type, see table 2 of CAN SW architecture
 *
 * \param[in]  pvAddressField
 *              addressing, depends on protocol
 *
 * \param[in]  pvCallBackFkt
 *
 * \return     Error/no error value
 *
 * \access     all
 * \reentrant  only for different protocol type/addreess fields
 *
 * \lastreview  21.03.11 Borck
 *              27.10.08 VTeam-Zurmhl
 *              13.10.08 VTeam-Funke
 *
 * \history_begin
 * 27.10.08  VTeam-Zurmhl
 * - erroneous code fragments for CSM_C_PTYPE_MCNET_USDT removed, #error added
 *
 * 13.10.08  VTeam-Funke
 * - added static error handling
 * - error case is printed out, now
 * - align return value to same name
 *
 * 13.03.08  VTeam-Khler
 * - CSM_C_PTYPE_USED_STD_BR_SIGNAL added.
 *
 * - CSM_C_PTYPE_USED_STD_BR_SIGNAL added.
 * 17.08.07 RBIN/EDI2-Basavaraj
 * - Added switch case for handling CSM_C_PTYPE_VCC_BR_SIGNAL.
 *
 * 17.10.06 CM-DI/ESA3-Khler
 * - PVOID replaced by void *.
 *
 * 21.08.06 CM-DI/ESA3-Khler
 * CSM_C_PTYPE_USED_STD_USDT_ISO instead of _FORD_, _GM_ ...
 *
 * 14.07.06 CM-DI/ESA3-Prhl
 * add bBus
 *
 * 11.05.06 CM-DI/ESA3-Battis
 * pvCallBackFkt made const PVOID
 *
 * 27.02.06 CM-DI/ESA3-K"ohler
 * Parameter pvAddressField is now a pointer to const.
 *
 * 01.10.03 CM-DI/ESU3-Prhl
 * Initial Revision.
 * \history_end
 ******************************************************************************/
CSM_API LONG CSM_lApplCallbackInit(void * pvHandle, BYTE bBus, DWORD dwProtocolType,
                                   const void * pvAddressField, const void * pvCallBackFkt)
{
  /* default return is "no error"                                             */
  LONG lRet = CSM_C_NO_ERROR;

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  /* ensure proper ETG tracing of invalid adressfield                         */
  BYTE Adr0 = 0x00;
  BYTE Adr1 = 0x00;

  if (pvAddressField != NULL)
  {
    Adr0 = *(((BYTE *)pvAddressField)+0);
    Adr1 = *(((BYTE *)pvAddressField)+1);
  }
  #endif
  #endif
  #endif

  /* check dedicated EM trace var and write error code to EM in case of       */
//??  CSM_vCheckandWriteErrorToEM();

  /* Check if CSM is in correct state for processing this API function call   */
  if ( (csm_bCsmState != CSM_C_STATE_ACTIVE) && (csm_bCsmState != CSM_C_STATE_INITIALIZED) )
  {
    #if (CSM_S_ENABLE_INDICATE_STATIC_ERROR == CSM_C_F_ON)
    /* Check if CSM has static errors -> static errors win over anything      */
    if ( (bBus < CSM_C_MAX_CAN_BUS) && (csm_lActStaticError[bBus] != CSM_C_NO_ERROR))
    {
      /* CSM has serious initialisation error */
      lRet = csm_lActStaticError[bBus];
    }
    else
    #endif
    {
      /* CSM not in state ACTIVE or INITIALIZED */
      lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_CSM_STATE_FORBIDS_CALL);
    }
  }
  else
  {
    /* before we continue we check for static errors                          */
    #if (CSM_S_ENABLE_INDICATE_STATIC_ERROR == CSM_C_F_ON)
    /* Check if CSM has static errors -> static errors win over anything      */
    if ( (bBus < CSM_C_MAX_CAN_BUS) && (csm_lActStaticError[bBus] != CSM_C_NO_ERROR))
    {
      /* CSM has serious initialisation error */
      lRet = csm_lActStaticError[bBus];
    }
    else
    #endif
    {
      /* the call of this function is valid in the current state, so process it */
      switch (dwProtocolType)
      {
        /**************************/
        /* Local CTRL             */
        /**************************/
        #ifdef CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL
        case CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL:
        {
          lRet = CSM_LCTR_lApplCallbackInit( pvHandle, bBus, dwProtocolType, pvAddressField, (tCSM_LOCAL_CTRL_APPL_CALLBACK*) pvCallBackFkt);
        }
        break;
        #endif /* CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL */

        /**************************/
        /* GM                     */
        /**************************/
        #ifdef CSM_C_PTYPE_GM_NWM
        case CSM_C_PTYPE_GM_NWM:
        {
          lRet = CSM_GMLAN_lApplCallbackInit( pvHandle, bBus, dwProtocolType, pvAddressField, (tCSM_GMLAN_NWM_APPL_CALLBACK*) pvCallBackFkt);
        }
        break;
        #endif /* CSM_C_PTYPE_GM_NWM */

        /**************************/
        /* PAG OSEK               */
        /**************************/
        #ifdef CSM_C_PTYPE_PAG_OSEK_NWM
        case CSM_C_PTYPE_PAG_OSEK_NWM:
        {
          lRet = CSM_OSEK_lApplCallbackInit( pvHandle, bBus, dwProtocolType, pvAddressField, (tCSM_OSEK_NWM_APPL_CALLBACK*) pvCallBackFkt);
        }
        break;
        #endif /* CSM_C_PTYPE_PAG_OSEK_NWM */

        /**************************/
        /* USDT ISO               */
        /**************************/
        #ifdef CSM_C_PTYPE_USED_STD_USDT_ISO
        case CSM_C_PTYPE_USED_STD_USDT_ISO:
        {
          lRet = CSM_ITP_lApplCallbackInit( pvHandle, bBus, dwProtocolType, pvAddressField, (tCSM_ISO_TP_USDT_APPL_CALLBACK*) pvCallBackFkt);
        }
        break;
        #endif /* CSM_C_PTYPE_USED_STD_USDT_ISO */

        /**************************/
        /* UUDT                   */
        /**************************/
        #ifdef CSM_C_PTYPE_USED_STD_UUDT_TX
        case CSM_C_PTYPE_USED_STD_UUDT_TX:
        {
          lRet = CNP_UUDT_lTxApplCallbackInit( pvHandle, bBus, dwProtocolType, pvAddressField, (tCSM_UUDT_TX_APPL_CALLBACK*) pvCallBackFkt);
        }
        break;
        #endif /* CSM_C_PTYPE_USED_STD_UUDT_TX */

        #ifdef CSM_C_PTYPE_USED_STD_UUDT_RX
        case CSM_C_PTYPE_USED_STD_UUDT_RX:
        {
          lRet = CNP_UUDT_lRxApplCallbackInit( pvHandle, bBus, dwProtocolType, pvAddressField, (tCSM_UUDT_RX_APPL_CALLBACK*) pvCallBackFkt);
        }
        break;
        #endif /* CSM_C_PTYPE_USED_STD_UUDT_RX */

        /**************************/
        /* Broadcasts             */
        /**************************/
        #ifdef CSM_C_PTYPE_USED_STD_BR_SIGNAL
        case CSM_C_PTYPE_USED_STD_BR_SIGNAL:
        {
          lRet = CSM_CBR_lApplCallbackInit( pvHandle, bBus, dwProtocolType, pvAddressField, (tCSM_BR_SIGNAL_APPL_CALLBACK*) pvCallBackFkt);
        }
        break;
        #endif /* CSM_C_PTYPE_USED_STD_BR_SIGNAL */

        /******************************/
        /* Diverse Network protocols  */
        /******************************/
        #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
        /*--------------------------*/
        /*  VW TP2.0                */
        /*--------------------------*/
        #if (CSM_S_TP20 == CSM_C_F_ON)
        #ifdef CSM_C_PTYPE_VW_ASDT_TP_20
        case CSM_C_PTYPE_VW_ASDT_TP_20:
        {
          lRet = CSM_TP20_lApplCallbackInit( pvHandle, bBus, dwProtocolType, pvAddressField, (tCSM_TP20_APPL_CALLBACK*) pvCallBackFkt);
        }
        break;
        #endif /* CSM_C_PTYPE_VW_ASDT_TP_20 */
        #endif /* (CSM_S_TP20 == CSM_C_F_ON) */

        /*--------------------------*/
        /*  Renault Nissan MPDT     */
        /*--------------------------*/
        #if (CSM_S_MPDT == CSM_C_F_ON)
        #ifdef CSM_C_PTYPE_RN_MPDT_C
        case CSM_C_PTYPE_RN_MPDT_C:
        {
          lRet = CSM_MPDT_lApplCallbackInit( pvHandle, bBus, dwProtocolType, pvAddressField, (tCSM_MPDT_APPL_CALLBACK*) pvCallBackFkt);
        }
        break;
        #endif /* CSM_C_PTYPE_RN_MPDT_C */

        #ifdef CSM_C_PTYPE_RN_MPDT_D
        case CSM_C_PTYPE_RN_MPDT_D:
        {
          lRet = CSM_MPDT_lApplCallbackInit( pvHandle, bBus, dwProtocolType, pvAddressField, (tCSM_MPDT_APPL_CALLBACK*) pvCallBackFkt);
        }
        break;
        #endif /* CSM_C_PTYPE_RN_MPDT_D */
        #endif /* (CSM_S_MPDT == CSM_C_F_ON) */

        /*--------------------------*/
        /*  MCNet                   */
        /*--------------------------*/
        #if (CSM_S_MCNET == CSM_C_F_ON)
        #ifdef CSM_C_PTYPE_MCNET_ASDT_STD
        case CSM_C_PTYPE_MCNET_ASDT_STD:
        {
          lRet = CSM_MCNET_lApplCallbackInit( pvHandle, bBus, dwProtocolType, pvAddressField, (tCSM_MCNET_ASDT_APPL_CALLBACK*) pvCallBackFkt);
        }
        break;
        #endif /* CSM_C_PTYPE_MCNET_ASDT_STD */

        #ifdef CSM_C_PTYPE_MCNET_USDT
        case CSM_C_PTYPE_MCNET_USDT:
        {
          lRet = CSM_MCNET_lApplCallbackInit( pvHandle, bBus, dwProtocolType, pvAddressField, pvCallBackFkt);
        }
        break;
        #endif /* CSM_C_PTYPE_MCNET_USDT */
        #endif /* (CSM_S_MCNET == CSM_C_F_ON) */

        /*--------------------------*/
        /*  LIN                     */
        /*--------------------------*/
        #if (CSM_S_LIN_DIAG == CSM_C_F_ON)
        case CSM_C_PTYPE_USED_STD_LIN_DIAG:
        {
          lRet = CSM_LIN_DIAG_lApplCallbackInit( pvHandle, bBus, dwProtocolType, pvAddressField, (tCSM_ISO_TP_USDT_APPL_CALLBACK*) pvCallBackFkt);
        }
        break;
        #endif /* (CSM_S_LIN_DIAG == CSM_C_F_ON) */
        #endif /* (CSM_S_CNP_AVAILABLE == CSM_C_F_ON) */

        /******************************/
        /* VW BAP                     */
        /******************************/
        #if ( CSM_S_VW_BAP_TP == CSM_C_F_ON )
        #ifdef CSM_C_PTYPE_VW_BAP
        case CSM_C_PTYPE_VW_BAP:
        {
          lRet = CSM_BAP_lApplCallbackInit( pvHandle, bBus, dwProtocolType, pvAddressField, (tCSM_VW_BAP_APPL_CALLBACK*)pvCallBackFkt);
        }
        break;
        #endif
        #endif /* CSM_S_VW_BAP_TP */

        /******************************/
        /* Diamler OSEKI NWM           */
        /******************************/
        #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
        #if (CSM_S_OSEKI_NM == CSM_C_F_ON)
        #ifdef CSM_C_PTYPE_DC_OSEKI_NWM
        case CSM_C_PTYPE_DC_OSEKI_NWM:
        {
          lRet = CSM_OSEKI_lApplCallbackInit(pvHandle, bBus, dwProtocolType, pvAddressField, (tCSM_OSEKI_NWM_APPL_CALLBACK*) pvCallBackFkt);
        }
        break;
        #endif /* CSM_C_PTYPE_DC_OSEKI_NWM */
        #endif /* CSM_S_OSEKI_NM */
        #endif /* CSM_S_CNM_AVAILABLE == CSM_C_F_ON */

        /******************************/
        /* J1939 Protocoll            */
        /******************************/
        #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
        #if (CSM_S_J1939 == CSM_C_F_ON)
        case CSM_C_PTYPE_J1939:
        {
          lRet = CSM_J1939_lApplCallbackInit(pvHandle, bBus, dwProtocolType, pvAddressField, (tCSM_J1939_APPL_CALLBACK*) pvCallBackFkt);
        }
        break;
        #endif
        #endif /* CSM_S_CNP_AVAILABLE */

        /******************************/
        /* unknown protocol           */
        /******************************/
        default:
        {
          lRet = CSM_M_MAKE_CSM_ERROR (CSM_C_ERR_ERROR, CSM_E_INVALID_PROTOCOL_TYPE);
          CSM_vAssert(FALSE);
        }
        break;
      } /* end switch(dwProtocolType) */ /*lint !e764 */ /* Default tree is used for all protocol types except CSM_C_PTYPE_VW_BAP - 22.08.2008 krv2hi */
    } /* no static error present */
  } /* end of valid state, protocol dependent processing of the call          */

  /* now do some trace output                                                 */
  #if (CSM_S_TRACE == CSM_C_F_ON)
    #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR1(("####1 CSM_lApplCallbackInit: Bus:%x  Protocol:%x  Addr:%02x%02x  Result:%x  Handle:%x",
                      ETG_CENUM(tCSM_BUS_NUMBER,      (tU8)  bBus),
                      ETG_CENUM(tCSM_PROTOCOL_TYPE,   (tU32) dwProtocolType),
                      Adr0,
                      Adr1,
                      lRet,
                      pvHandle
                    ));
      #endif
    #else
    {
      // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
      BYTE abData[15]={0};

      abData[0] = bBus;
      abData[1] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(dwProtocolType));
      abData[2] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(dwProtocolType));
      abData[3] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(dwProtocolType));
      abData[4] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(dwProtocolType));
      if (pvAddressField != NULL)
      {
        abData[5] = *(((BYTE *)pvAddressField)+0);
        abData[6] = *(((BYTE *)pvAddressField)+1);
      }
      abData[7] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(lRet));
      abData[8] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(lRet));
      abData[9] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(lRet));
      abData[10] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(lRet));

      abData[11] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(pvHandle));
      abData[12] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(pvHandle));
      abData[13] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(pvHandle));
      abData[14] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(pvHandle));

      CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_APPL_CALLBACK_INIT, abData, 15);
    }
    #endif
  #endif

  return lRet;
} //lint !e715 !e818 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally , PQM_authorized_multi_123. Reason: Interface not defined as const
#endif // CSM_S_CALLBACK_INIT_BY_APPLICATION

/*******************************************************************************
 * function    CSM_lBusErrorIndCallbackInit
 * \doxydocu
 * \brief      Stores the callback function
 *
 * \param[in]  pvHandle
 *              pointer to handle. For future use! May be a deinit() function
 *
 * \param[in]  vPFNBusErrorInd
 *              pointer to callback function
 *
 * \return     Error value
 *
 * \access     interrupt|CAN-task|application ??
 * \reentrant  yes|no ??
 *
 * \lastreview 02.02.09 Prhl
 *             13.10.08 VTeam-Funke
 *             09.11.06 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 13.10.08  VTeam-Funke
 * - only one return point of function
 * - align return value to same name
 *
 * 17.10.06 CM-DI/ESA3-Khler
 * - PVOID replaced by void *.
 *
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
CSM_API LONG CSM_lBusErrorIndCallbackInit(void * pvHandle, tCSM_PFN_BUS_ERROR_IND vPFNBusErrorInd)
{
  LONG lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_WARNING, CSM_E_RESOURCE);

  #if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
  DWORD dwCount;

  // ToDo: This function has to be implemented if it is needed by new projects.

  // Check if CSM is in correct state for processing this API function call
  if ((csm_bCsmState != CSM_C_STATE_ACTIVE) && (csm_bCsmState != CSM_C_STATE_INITIALIZED))
  {
    // CSM not in state ACTIVE or INITIALIZED
    lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_CSM_STATE_FORBIDS_CALL);
  }
  else
  {
    for (dwCount = 0; dwCount < CSM_M_ELEMENTS(csm_arBusErrorIndCallbackFkt); dwCount++)
    {
      if (csm_arBusErrorIndCallbackFkt[dwCount].pfvBusErrorInd == NULL)
      {
        // store callback on the first free place                           */
        csm_arBusErrorIndCallbackFkt[dwCount].pfvBusErrorInd = vPFNBusErrorInd;
        csm_arBusErrorIndCallbackFkt[dwCount].pvHandle = pvHandle;
        lRet = CSM_C_NO_ERROR;
        break;
      }
    }
  }
  #endif // CSM_S_ENABLE_INDICATE_BUSERROR

  return lRet;
}//lint !e715 !e818 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally , PQM_authorized_multi_123. Reason: Interface not defined as const*/

/*******************************************************************************
 * function    CSM_lSignalCallbackInit
 * \doxydocu
 * \brief      Stores the callback function
 *
 * \param[in]  pvHandle
 *              pointer to handle. For future use! May be a deinit() function
 *
 * \param[in]  dwSignalId
 *              signal identifier (signal name)
 *
 * \param[in]  vPFNCBRSignalInd
 *              pointer to callback function
 *
 * \return     Error value
 *
 * \access     interrupt|CAN-task|application ??
 * \reentrant  yes|no ??
 *
 * \lastreview 21.03.11 Borck
 *             13.10.08 VTeam-Funke
 *             09.11.06 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 13.10.08 VTeam-Funke
 * - align return value to same name
 *
 * 17.10.06 CM-DI/ESA3-Khler
 * - PVOID replaced by void *.
 *
 * Initial Revision.
 * \history_end
 ******************************************************************************/
CSM_API LONG CSM_lSignalCallbackInit(void * pvHandle, DWORD dwSignalId, tCSM_PFN_CBR_SIGNAL_IND vPFNCBRSignalInd)
{
  LONG lRet;

  #if (CSM_S_CBR_AVAILABLE == CSM_C_F_ON)
  /* Check if CSM is in correct state for processing this API function call   */
  if ( (csm_bCsmState == CSM_C_STATE_ACTIVE) || (csm_bCsmState == CSM_C_STATE_INITIALIZED) )
  {
    /* signals will be handled by broadcast server, so forward this service   */
    #if ((CSM_S_iMX_SIGNAL_ABSTRACTION == CSM_C_F_ON) || (CSM_S_v850_SIGNAL_ABSTRACTION == CSM_C_F_ON))
    lRet = CBR_AL_lSignalCallBackInit(pvHandle, dwSignalId, vPFNCBRSignalInd);
    #else
    lRet = CBR_lSignalCallBackInit(pvHandle, dwSignalId, vPFNCBRSignalInd);
    #endif
  }
  else
  {
    /* CSM not in state ACTIVE or INITIALIZED                                 */
    lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_CSM_STATE_FORBIDS_CALL);
  }

  #if (CSM_S_TRACE == CSM_C_F_ON)
    #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR1(("####1 CSM_lSignalCallbackInit: Signal:%08x  Handle:%08x  Callback:%08x  Return:%08x",
                      (tU32)  dwSignalId,
                      (tU32)  pvHandle,
                      (tU32)  vPFNCBRSignalInd,
                      (tU32)  lRet
                    ));
      #endif
    #else
    {
      // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
      BYTE abData[16]={0};

      abData[0] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(pvHandle));
      abData[1] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(pvHandle));
      abData[2] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(pvHandle));
      abData[3] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(pvHandle));

      abData[4] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(dwSignalId));
      abData[5] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(dwSignalId));
      abData[6] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(dwSignalId));
      abData[7] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(dwSignalId));

      abData[8] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(vPFNCBRSignalInd));
      abData[9] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(vPFNCBRSignalInd));
      abData[10] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(vPFNCBRSignalInd));
      abData[11] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(vPFNCBRSignalInd));

      abData[12] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(lRet));
      abData[13] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(lRet));
      abData[14] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(lRet));
      abData[15] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(lRet));

      CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_SIGNAL_CALLBACK_INIT, abData, 16);
    }
    #endif
  #endif // CSM_S_TRACE
  #else // CSM_S_CBR_AVAILABLE
  lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_GENERAL_ERROR);
  #endif // CSM_S_CBR_AVAILABLE

  return lRet;
} //*lint !e715 !e818 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally , PQM_authorized_multi_123. Reason: Interface not defined as const
#endif // CSM_S_CALLBACK_INIT_BY_APPLICATION

/*******************************************************************************
 * function    CSM_lSetBusHold
 * \doxydocu
 * \brief      Sets an request to hold the selected bus (bBus) active. The
 *             bus should stay active, as any bit is still set.
 *
 * \param[in]  bBus
 *              bus no.
 *
 * \param[in]  dwProtocolType
 *              protocol tpye (e.g. TP, broadcast)
 *
 * \param[in]  wApplId
 *              application ID to handle several application sharing a protocol
 *
 * \return     Error value
 *
 * \access     interrupt|CAN-task|application ??
 * \reentrant  yes|no ??
 *
 * \lastreview 21.03.11 Borck
 *             13.10.08 VTeam-Funke
 *             28.06.2006 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 13.10.08 VTeam-Funke
 * - align return value to same name
 *
 * 27.06.2006 CM-DI/ESA3-Zurmhl
 * adaptation for third CAN bus
 *
 * 25.11.2005 CM-DI/ESA3-Franke
 * bug fixes, simplifications
 *
 * 18.02.2004 CM-CR/ESD-Peter Tuschik
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#ifdef CSM_C_MAX_PTYPE_1
LONG CSM_lSetBusHold(BYTE  bBus, DWORD dwProtocolType, WORD  wApplID)
{
  LONG lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_WARNING, CSM_E_INVALID_PARA);

  /* disable INTs                                                             */
  INT iIsrStat = CSM_iInterruptsDisable();

  /* check parameters                                                         */
  if ( (bBus < CSM_C_MAX_CAN_BUS) &&
       (wApplID < CSM_C_MAX_APPL_ID) &&
       (wApplID < CSM_C_MAX_POSSIBLE_APPL_ID))
  {
    LONG lMaxPTypeIdx = 0;                  /* hfm2hi: just to be initialized */
    LONG lPTypeIdx = 0;
    DWORD *pdwRequest = NULL;
    const DWORD *pdwPType = NULL;

    /* select the bus according to the given parameter                        */
    if (bBus == CSM_C_CAN_1)
    {
       pdwRequest = &csm_adwCommunicationRequests1[0];
       pdwPType = &CSM_caPType1[0];
       lMaxPTypeIdx = CSM_C_MAX_PTYPE_1;
    }

    #if CSM_C_MAX_CAN_BUS > 1
    else if (bBus == CSM_C_CAN_2)
      {
         pdwRequest = &csm_adwCommunicationRequests2[0];
         pdwPType = &CSM_caPType2[0];
         lMaxPTypeIdx = CSM_C_MAX_PTYPE_2;
      }

      #if (CSM_C_MAX_CAN_BUS > 2)
      else if (bBus == CSM_C_CAN_3)
        {
          pdwRequest = &csm_adwCommunicationRequests3[0];
          pdwPType = &CSM_caPType3[0];
          lMaxPTypeIdx = CSM_C_MAX_PTYPE_3;
        }

        #if (CSM_C_MAX_CAN_BUS > 3)
          #error Insert here functionality for next CAN bus
        #endif /* #if (CSM_C_MAX_CAN_BUS > 3) */
      #endif /* #if (CSM_C_MAX_CAN_BUS > 2) */
    #endif /* #if (CSM_C_MAX_CAN_BUS > 1) */

    /* loop over all available protocol types:                                */
    for (lPTypeIdx = 0; lPTypeIdx < lMaxPTypeIdx; lPTypeIdx++)
    {
      if (*(pdwPType+lPTypeIdx) == dwProtocolType)  /*lint !e613*/ /* Possible use of null pointer 'pdwPType': */
      {                                             /* bBus is checked and for each id an array address will be assigned. */
        DWORD dwMask;

        dwMask = 1u << wApplID;             /* calc bit # from ID */
        pdwRequest[lPTypeIdx] |= dwMask;    /* and set request */ /*lint !e613*/ /* Possible use of null pointer 'pdwRequest': */
                                                                  /* bBus is checked and for each id an array address will be assigned. */
        lRet = CSM_C_NO_ERROR;
        break;
      }
    } /* for all protocol types */

    if (lRet != CSM_C_NO_ERROR)
    {
      lRet = CSM_M_MAKE_CSM_ERROR( CSM_C_ERR_WARNING, CSM_E_INVALID_PROTOCOL_TYPE);
    } /* (lRet != CSM_C_NO_ERROR) */
  }

  /* On this Assert first check the function parameters !                     */
  CSM_vAssert(lRet == CSM_C_NO_ERROR);

  /* enable INTs                                                              */
  CSM_vInterruptsRestore(iIsrStat);

  return lRet;
} /* end of CSM_lSetBusHold () */

/*******************************************************************************
 * function    CSM_lClearBusHold
 * \doxydocu
 * \brief      Clear an request to hold the selected bus (bBus) active. The
 *             bus should stay active, as any bit is still set.
 *
 * \param[in]  bBus
 *              bus no.
 *
 * \param[in]  dwProtocolType
 *              protocol tpye (e.g. TP, broadcast)
 *
 * \param[in]  wApplId
 *              application ID to handle several application sharing a protocol
 *
 * \return     Error value
 *
 * \access     interrupt|CAN-task|application ??
 * \reentrant  yes|no ??
 *
 * \lastreview 21.03.11 Borck
 *             13.10.08 VTeam-Funke
 *             28.06.2006 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 13.10.08 VTeam-Funke
 * - align return value to same name
 *
 * 27.06.2006 CM-DI/ESA3-Zurmhl
 * adaptation for third CAN bus
 *
 * 25.11.2005 CM-DI/ESA3 Franke
 * bug fixes, simplifications
 *
 * 18.02.2004 CM-CR/ESD-Peter Tuschik
 * Initial Revision.
 * \history_end
 ******************************************************************************/
LONG CSM_lClearBusHold(BYTE  bBus, DWORD dwProtocolType, WORD  wApplID)
{
  LONG lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_WARNING, CSM_E_INVALID_PARA);

  /* disable INTs                                                             */
  INT iIsrStat = CSM_iInterruptsDisable();

  /* check parameters                                                         */
  if ((bBus < CSM_C_MAX_CAN_BUS) &&
      (wApplID < CSM_C_MAX_APPL_ID) &&
      (wApplID < CSM_C_MAX_POSSIBLE_APPL_ID))
  {
    LONG lMaxPTypeIdx = 0;                  /* hfm2hi: just to be initialized */
    LONG lPTypeIdx = 0;
    DWORD *pdwRequest = NULL;
    const DWORD *pdwPType = NULL;

    /* select the bus according to the given parameter                        */
    if (bBus == CSM_C_CAN_1)
    {
       pdwRequest = &csm_adwCommunicationRequests1[0];
       pdwPType = &CSM_caPType1[0];
       lMaxPTypeIdx = CSM_C_MAX_PTYPE_1;
    }

    #if CSM_C_MAX_CAN_BUS > 1
    else if (bBus == CSM_C_CAN_2)
      {
         pdwRequest = &csm_adwCommunicationRequests2[0];
         pdwPType = &CSM_caPType2[0];
         lMaxPTypeIdx = CSM_C_MAX_PTYPE_2;
      }

      #if (CSM_C_MAX_CAN_BUS > 2)
      else if (bBus == CSM_C_CAN_3)
        {
          pdwRequest = &csm_adwCommunicationRequests3[0];
          pdwPType = &CSM_caPType3[0];
          lMaxPTypeIdx = CSM_C_MAX_PTYPE_3;
        }

        #if (CSM_C_MAX_CAN_BUS > 3)
          #error Insert here functionality for next CAN bus
        #endif /* #if (CSM_C_MAX_CAN_BUS > 3) */
      #endif /* #if (CSM_C_MAX_CAN_BUS > 2) */
    #endif /* #if (CSM_C_MAX_CAN_BUS > 1) */

    /* loop over all available protocol types:                                */
    for (lPTypeIdx = 0;lPTypeIdx < lMaxPTypeIdx;lPTypeIdx++)
    {
      if (*(pdwPType+lPTypeIdx) == dwProtocolType)            /*lint !e613*/ /* Possible use of null pointer 'pdwPType': */
      {                                                       /* bBus is checked and for each id an array address will be assigned. */
        DWORD dwMask;

        dwMask = 1u << wApplID;            /* calc bit # from ID */
        pdwRequest[lPTypeIdx] &= ~dwMask; /* and clear request */ /*lint !e613*/ /* Possible use of null pointer 'pdwRequest': */
                                                                  /* bBus is checked and for each id an array address will be assigned. */
        lRet = CSM_C_NO_ERROR;
        break;
      }
    } /* for all protocol types */

    if (lRet != CSM_C_NO_ERROR)
    {
      lRet = CSM_M_MAKE_CSM_ERROR( CSM_C_ERR_ERROR, CSM_E_INVALID_PROTOCOL_TYPE);
    } /* (lRet != CSM_C_NO_ERROR) */
  }

  /* On this Assert first check the function parameters !                     */
  CSM_vAssert(lRet == CSM_C_NO_ERROR);

  /* enable INTs                                                              */
  CSM_vInterruptsRestore(iIsrStat);

  return lRet;
} /* end of CSM_lClearBusHold () */

#endif /* CSM_C_MAX_PTYPE_1 */

/*******************************************************************************
 * function    CSM_bGetBusHold
 * \doxydocu
 * \brief      return TRUE, if Bus should stay active
 *
 * \param[in]  bBus
 *              bus no.
 *
 * \return     Error value
 *
 * \access     interrupt|CAN-task|application ??
 * \reentrant  yes|no ??
 *
 * \lastreview 25.04.2008 VTeam-Prhl
 *
 * \history_begin
 * 27.11.2007 VTeam-Khler
 * Error handling if bBus >= CSM_C_MAX_CAN_BUS.
 *
 * 25.10.2006 CM-DI/ESA3-Haffner
 * initialize variables
 *
 * 27.06.2006 CM-DI/ESA3-Zurmhl
 * adaptation for third CAN bus
 *
 * 25.11.2005 CM-DI/ESA3 Franke
 * bug fixes, simplifications
 *
 * 18.02.2004 CM-CR/ESD-Peter Tuschik
 * Initial Revision.
 * \history_end
 ******************************************************************************/
BOOL CSM_bGetBusHold(BYTE bBus)
{
#ifdef CSM_C_MAX_PTYPE_1
  DWORD dwState = 0;

  if (bBus < CSM_C_MAX_CAN_BUS)
  {
  LONG lPTypeIdx;
  LONG lMaxPTypeIdx = 0;
  DWORD *pdwRequest = NULL;

    if (bBus == CSM_C_CAN_1)
    {
      pdwRequest = &csm_adwCommunicationRequests1[0];
      lMaxPTypeIdx = CSM_C_MAX_PTYPE_1;
    }

    #if CSM_C_MAX_CAN_BUS > 1
      else if (bBus == CSM_C_CAN_2)
      {
        pdwRequest = &csm_adwCommunicationRequests2[0];
        lMaxPTypeIdx = CSM_C_MAX_PTYPE_2;
      }
      #if (CSM_C_MAX_CAN_BUS > 2)
        else if (bBus == CSM_C_CAN_3)
        {
          pdwRequest = &csm_adwCommunicationRequests3[0];
          lMaxPTypeIdx = CSM_C_MAX_PTYPE_3;
        }
        #if (CSM_C_MAX_CAN_BUS > 3)
          #error Insert here functionality for next CAN bus
        #endif /* #if (CSM_C_MAX_CAN_BUS > 3) */
      #endif  /* #if (CSM_C_MAX_CAN_BUS > 2) */
    #endif   /* #if (CSM_C_MAX_CAN_BUS > 1) */

    /* check all requests for channel: */
    /* for all protocol types: */
    for (lPTypeIdx = 0; lPTypeIdx < lMaxPTypeIdx; lPTypeIdx++)
    {
      dwState |= *(pdwRequest+lPTypeIdx);  /*lint !e613*/ /* Possible use of null pointer 'pdwRequest': */
    }                                      /* bBus is checked and for each id an array address will be assigned. */
  }
  else
  {
    CSM_vAssert(FALSE); /*lint !e506 */ /* Wrong bus index shall cause an assert - krv2hi */
    /* dwState = 0; is initialization */
  }

  return (dwState != 0);                    /* return any request pending ? */

#else
  return FALSE;
#endif /* CSM_C_MAX_PTYPE_1 */

} /* end of CSM_bGetBusHold () *//*lint !e715 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally*/

/*******************************************************************************
 * function    CSM_lGetCommunicationState
 * \doxydocu
 * \brief      The actual state of a transport protocol connection can be
 *             checked with this function. In addition, state information
 *             of broadcast, NM or stack "connections" can be retrieved, too
 *
 *             This function shall replace the - obsolete - function
 *             CSM_lGetConnectState().
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \param[in]  bBus
 *              Bus number, only needed for "broadcast" protocol types
 *
 * \param[in]  dwProtocolType
 *              Protocol type, see corresponding table of CAN SW architecture
 *
 * \param[in]  pvAddressField
 *              Protocol dependend address field
 *
 * \param[out] pbConnectState
 *              CSM_C_CONNECTED     /  CSM_C_STACK_UP
 *              CSM_C_DISCONNECTED  /  CSM_C_STACK_DOWN
 *
 * \param[in]  wApplId
 *              Application Id, only needed for "broadcast" protocol types
 *
 * \return     Error value (wrong Parameter) if < CSM_C_NO_ERROR
 *             else no error
 *
 * \access     all
 * \reentrant  yes
 *
 * \lastreview 02.02.09  Franke
 *             13.10.08  VTeam-Funke
 *             12.09.07  VTeam-Zurmhl
 *
 * \history_begin
 * 21.06.12  Khler
 * - CSM_SELF_TEST_lGetCommunicationState() added.
 * 02.02.09  Prhl
 * - CSM_C_PTYPE_???_BR_SIGNAL replaced by CSM_C_PTYPE_USED_STD_BR_SIGNAL.
 *
 * 13.10.08  VTeam-Funke
 * - added static error handling
 *
 * 25.09.07 RBIN/EDI2-Basavaraj
 * - Added switch case for handling CSM_C_PTYPE_VCC_BR_SIGNAL.
 *
 * 11.09.07  VTeam-Khler
 * Connect state is set per default to disconnect/stack down if error occured.
 *
 * ......
 *
 * 21.03.05 CM-DI/ESA3-Zurmhl
 * First version (only function body)
 *
 * 11.08.05 CM-DI/ESA3-Zurmhl
 * Last change
 *
 * 02.12.05 CM-DI/ESA3-Khler
 * Type of parameter pvAddressField changed to const void*.
 *
 * \history_end
 ******************************************************************************/
CSM_API LONG CSM_lGetCommunicationState(BYTE  bBus, DWORD dwProtocolType, const void * pvAddressField, BYTE * pbConnectState, WORD wApplId)
{
  LONG  lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_WARNING, CSM_E_INIT_WARNING);

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  /* ensure proper ETG tracing of invalid adressfield                         */
  BYTE Adr0 = 0x00;
  BYTE Adr1 = 0x00;

  if (pvAddressField != NULL)
  {
    Adr0 = *(((BYTE *)pvAddressField)+0);
    Adr1 = *(((BYTE *)pvAddressField)+1);
  }
  #endif
  #endif
  #endif

  /* Check if CSM is in correct state for processing this API function call   */
  if ( (csm_bCsmState == CSM_C_STATE_ACTIVE) || (csm_bCsmState == CSM_C_STATE_INITIALIZED) )
  {
    /* before we continue we check for static errors                          */
    #if (CSM_S_ENABLE_INDICATE_STATIC_ERROR == CSM_C_F_ON)
    /* Check if CSM has static errors -> static errors win over anything      */
    if ( (bBus < CSM_C_MAX_CAN_BUS) && (csm_lActStaticError[bBus] != CSM_C_NO_ERROR))
    {
      /* CSM has serious initialisation error */
      lRet = csm_lActStaticError[bBus];
    }
    else
    #endif
    {
      /* calling this function is valid in the current state, so process it   */
      switch (dwProtocolType)
      {
        /**************************/
        /* Local CTRL             */
        /**************************/
        #ifdef CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL
        case CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL:
        {
          lRet = CSM_LCTR_lGetCommunicationState(bBus, dwProtocolType, pvAddressField, pbConnectState, wApplId);
        }
        break;
        #endif /* CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL */

        /**************************/
        /* GM NWM                 */
        /**************************/
        #ifdef CSM_C_PTYPE_GM_NWM
        case CSM_C_PTYPE_GM_NWM:
        {
          lRet = CSM_GMLAN_lGetCommunicationState(bBus, dwProtocolType, pvAddressField, pbConnectState, wApplId);
        }
        break;
        #endif /* CSM_C_PTYPE_GM_NWM */

        /**************************/
        /* Broadcasts             */
        /**************************/
        #ifdef CSM_C_PTYPE_USED_STD_BR_SIGNAL
        case CSM_C_PTYPE_USED_STD_BR_SIGNAL:
        {
          #if (CSM_S_CSM_VW == CSM_C_F_OFF) && (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
          /*Battis: use this only in non radio projects */
          lRet = CSM_CBR_lGetCommunicationState(bBus, dwProtocolType, pvAddressField, pbConnectState, wApplId);
          #endif
        }
        break;
        #endif /* CSM_C_PTYPE_USED_STD_BR_SIGNAL */

        /**************************/
        /* USDT ISO               */
        /**************************/
        #ifdef CSM_C_PTYPE_USED_STD_USDT_ISO
        case CSM_C_PTYPE_USED_STD_USDT_ISO:
        {
          lRet = CSM_ITP_lGetCommunicationState(bBus, dwProtocolType, pvAddressField, pbConnectState, wApplId);
        }
        break;
        #endif /* CSM_C_PTYPE_USED_STD_USDT_ISO */

        /**************************/
        /* MCNET ASDT             */
        /**************************/
        #if defined CSM_C_PTYPE_MCNET_ASDT_STD    /*--- Protocol Type: MCNet ASDT --> -*/
        case CSM_C_PTYPE_MCNET_ASDT_STD:
        {
          lRet = CSM_MCNET_lGetCommunicationState(bBus, dwProtocolType, pvAddressField, pbConnectState, wApplId);
        }
        break;
        #endif  /* CSM_C_PTYPE_MCNET_ASDT_STD */

        /*****************************/
        /* Diverse Network Protocols */
        /*****************************/
        #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)

        /*---------------------------*/
        /* VW TP2.0                  */
        /*---------------------------*/
        #if (CSM_S_TP20 == CSM_C_F_ON)
        #ifdef CSM_C_PTYPE_VW_ASDT_TP_20
        case CSM_C_PTYPE_VW_ASDT_TP_20:
        {
          lRet = CSM_TP20_lGetCommunicationState(bBus, dwProtocolType, pvAddressField, pbConnectState, wApplId);
        }
        break;
        #endif /* CSM_C_PTYPE_VW_ASDT_TP_20 */
        #endif /* (CSM_S_TP20 == CSM_C_F_ON) */

        /*---------------------------*/
        /* Renault Nissan MPDT       */
        /*---------------------------*/
        #if (CSM_S_MPDT == CSM_C_F_ON)
        #ifdef CSM_C_PTYPE_RN_MPDT_C
        case CSM_C_PTYPE_RN_MPDT_C:
        {
          lRet = CSM_MPDT_lGetCommunicationState(bBus, dwProtocolType, pvAddressField, pbConnectState, wApplId);
        }
        break;
        #endif /* CSM_C_PTYPE_RN_MPDT_C */

        #ifdef CSM_C_PTYPE_RN_MPDT_D
        case CSM_C_PTYPE_RN_MPDT_D:
        {
          lRet = CSM_MPDT_lGetCommunicationState(bBus, dwProtocolType, pvAddressField, pbConnectState, wApplId);
        }
        break;
        #endif /* CSM_C_PTYPE_RN_MPDT_D */
        #endif /* (CSM_S_MPDT == CSM_C_F_ON) */

        /*---------------------------*/
        /* J1939                     */
        /*---------------------------*/
        #if (CSM_S_J1939 == CSM_C_F_ON)
        #ifdef CSM_C_PTYPE_J1939
        case CSM_C_PTYPE_J1939:
        {
          lRet = CSM_J1939_lGetCommunicationState(bBus, dwProtocolType, pvAddressField, pbConnectState, wApplId);
        }
        break;
        #endif /* CSM_C_PTYPE_J1939 */
        #endif /* (CSM_S_J1939 == CSM_C_F_ON) */
        #endif /* (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)*/

        /*****************************/
        /* PAG OSEK NWM              */
        /*****************************/
        #ifdef CSM_C_PTYPE_PAG_OSEK_NWM
        case CSM_C_PTYPE_PAG_OSEK_NWM:
        {
          lRet = CSM_OSEK_lGetCommunicationState(bBus, dwProtocolType, pvAddressField, pbConnectState, wApplId);
        }
        break;
        #endif /* CSM_C_PTYPE_PAG_OSEK_NWM */

        /*****************************/
        /* Nissan OSEKI NWM          */
        /*****************************/
        #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
        #if (CSM_S_OSEKI_NM == CSM_C_F_ON)
        #ifdef CSM_C_PTYPE_DC_OSEKI_NWM
        case CSM_C_PTYPE_DC_OSEKI_NWM:
        {
          /* this call will retrieve the monitor state for a given connection   */
          /* that is specified by the adressfield. The Appl ID is a value that  */
          /* identifies the calling component.                                  */
          lRet = CSM_OSEKI_lGetCommunicationState(bBus, dwProtocolType, pvAddressField, pbConnectState, wApplId);
        }
        break;
        #endif /* CSM_C_PTYPE_DC_OSEKI_NWM */
        #endif /* CSM_S_OSEKI_NM */
        #endif /* CSM_S_CNM_AVAILABLE == CSM_C_F_ON */

        /*******************************/
        /* CSM self test called by SPM */
        /*******************************/
        #ifdef  CSM_C_PTYPE_GM_CSM_SELF_TEST
          case CSM_C_PTYPE_GM_CSM_SELF_TEST:
          {
            lRet = CSM_SELF_TEST_lGetCommunicationState(bBus, dwProtocolType, pvAddressField, pbConnectState, wApplId);
          }
          break;
        #endif /* CSM_C_PTYPE_GM_CSM_SELF_TEST */

        /*****************************/
        /* Unknown Protocol          */
        /*****************************/
        default:
        {
          lRet = CSM_M_MAKE_CSM_ERROR( CSM_C_ERR_WARNING, CSM_E_INVALID_PROTOCOL_TYPE);
          CSM_vAssert(FALSE);
        }
        break;
      } /* end switch(dwProtocolType) */ /*lint !e764 */ /* Default tree is used for all protocol types except CSM_C_PTYPE_VW_BAP - 22.08.2008 krv2hi */
    }
  } /* end of valid state, protocol dependent processing of the call          */
  else
  {
    /* Check if CSM has static errors -> static errors win over anything      */
    #if (CSM_S_ENABLE_INDICATE_STATIC_ERROR == CSM_C_F_ON)
    /* Check if CSM has not static errors */
    if ( (bBus < CSM_C_MAX_CAN_BUS) && (csm_lActStaticError[bBus] != CSM_C_NO_ERROR))
    {
      /* CSM has serious initialisation error */
      lRet = csm_lActStaticError[bBus];
    }
    else
    #endif
    {
      /* CSM not in state ACTIVE or INITIALIZED                               */
      lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_CSM_STATE_FORBIDS_CALL);
    }
  }

  /* Set connect state per default to disconnect if error occured             */
  if (lRet < CSM_C_NO_ERROR)
  {
    * pbConnectState = CSM_C_DISCONNECTED; /* same as CSM_C_STACK_DOWN */
  }

  #if (CSM_S_TRACE == CSM_C_F_ON)
    #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      {
        // ETG trace
        BYTE ubErrorGrade = CSM_M_GET_EG(lRet);
        BYTE ubErrorLoc   = CSM_M_GET_EL(lRet);
        WORD ubErrorCode  = CSM_M_GET_EC(lRet);

        // new ETG enums have been defined in interface header (CSM wide) and are used here now
        // only project specific enums (like address field or appl id are not used here)
        ETG_TRACE_USR1(("####1 CSM_lGetCommunicationState: Bus:%x  Protocol:%x  Addr:%02x%02x  State:%x  ApplID:%04x  Error:%x  ErrorLoc:%x  ErrorGrade:%x",
                        ETG_CENUM(tCSM_BUS_NUMBER,       (tU8)  bBus),
                        ETG_CENUM(tCSM_PROTOCOL_TYPE,    (tU32) dwProtocolType),
                        Adr0,                            // generic without enum here
                        Adr1,                            // generic without enum here
                        ETG_CENUM(tCSM_CONNECTION_STATE, (tU8)  (* pbConnectState)),
                        wApplId,                         // generic without enmum here
                        ETG_CENUM(tCSM_ERROR_CODE,       (tU16) ubErrorCode),
                        ETG_CENUM(tCSM_ERROR_LOCATION,   (tU8)  ubErrorLoc),
                        ETG_CENUM(tCSM_ERROR_GRADE,      (tU8)  ubErrorGrade)
                      ));
      }
      #endif
    #else
    {
      // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
      BYTE abData[11] = {0};

      abData[0] = bBus;
      abData[1] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(dwProtocolType));
      abData[2] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(dwProtocolType));
      abData[3] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(dwProtocolType));
      abData[4] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(dwProtocolType));
      if( pvAddressField != NULL)
      {
        abData[5] = *(((BYTE *)pvAddressField)+0);
        abData[6] = *(((BYTE *)pvAddressField)+1);
      }
      abData[7] = *pbConnectState;
      abData[8] = CSM_M_GET_HIBYTE(wApplId);
      abData[9] = CSM_M_GET_LOBYTE(wApplId);

      abData[10] = (lRet >= CSM_C_NO_ERROR) ? 0x00 : 0xFF;

      CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_GET_COMMUNICATION_STATE, abData, 11);
    }
    #endif
  #endif // CSM_S_TRACE

  return lRet;
} //lint !e715 !e818 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally , PQM_authorized_multi_123. Reason: Interface not defined as const

/*******************************************************************************
 * function    CSM_lConTest
 * \doxydocu
 * \brief      Requests a connection test for a transport protocol connection.
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \param[in]  dwProtocolType
 *              Protocol type, see table 2 of CAn SW architecture
 *
 * \param[in]  pvAdressField
 *              Protocol dependend address field
 *
 * \param[in]  bAction
 *              CSM_C_ENABLE
 *              CSM_C_DISABLE
 *
 * \return     Error/no error value
 *
 * \access     application
 * \reentrant  only for different channels
 *
 * \lastreview 25.01.08 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 27.11.06  VTeam-Khler
 *  - Return value initialized with CSM_E_INIT_WARNING instead of no error.
 *
 * 17.10.06 CM-DI/ESA3-Khler
 * - PVOID replaced by void *.
 *
 * 13.01.04 CM-DI/ESU3-Zurmuehl
 * Parameter bBus deleted
 *
 * 17.11.03 CM-DI/ESU3-Zurmuehl
 * Initial Revision.
 * \history_end
 ******************************************************************************/
/* U.Z.(6): Noch vollstaendig zu implementieren!                    */
/* Fuer MCNet z.Zt. nur automatisch gemacht, nicht von Applikation! */

CSM_API LONG CSM_lConTest (DWORD dwProtocolType, void* pvAdressField, BYTE  bAction)
{
  LONG lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_WARNING, CSM_E_INIT_WARNING);
#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
  #if (CSM_S_MCNET == CSM_C_F_OFF)
    /* (CSM_S_MCNET == CSM_C_F_ON) and (CSM_S_MCNET_ASDT == CSM_C_F_OFF) could be checked, too */
    CSM_M_PARAMETER_INTENTIONALLY_UNUSED(dwProtocolType);
    CSM_M_PARAMETER_INTENTIONALLY_UNUSED(pvAdressField);
    CSM_M_PARAMETER_INTENTIONALLY_UNUSED(bAction);
  #endif

  /* Check if CSM is in correct state for processing this API function call */
  if (csm_bCsmState == CSM_C_STATE_ACTIVE)
  {
    #if (CSM_S_MCNET == CSM_C_F_ON)  /* Might also be useful for VW TP (?) */
      #if (CSM_S_MCNET_ASDT == CSM_C_F_ON)
        if (CSM_C_PTYPE_MCNET_ASDT_STD == dwProtocolType)   /* Might also be useful for VW TP (?) */
        {
          BYTE bBus;
          BYTE bConNumber;

          if ( (lRet = CSM_MCNET_lGetConNumber (dwProtocolType, pvAdressField, &bBus, &bConNumber) ) == CSM_C_NO_ERROR)
          {
            lRet = CSM_MCNET_lConTest (bConNumber, bAction);
          }
        }
      #endif /* CSM_S_MCNET_ASDT  ON */
    #endif /* CSM_S_MCNET */
  }
  else
  {
    /* CSM not in state ACTIVE */
    lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_CSM_STATE_FORBIDS_CALL);
  }
#endif /* CSM_S_CNP_AVAILABLE == CSM_C_F_ON */
  return (lRet);
} /* end CSM_lConTest() *//*lint !e715 !e818 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally , PQM_authorized_multi_123. Reason: Interface not defined as const*/


/*******************************************************************************
 * function    CSM_lDataReq
 * \doxydocu
 * \brief      Request to send a transport protocol message to CAN
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \param[in]  dwProtocolType
 *              Protocol type, see table 2 of CAn SW architecture.
 *
 * \param[in]  pvAddressField
 *              Protocol dependend address field.
 *
 * \param[in]  pbData
 *              Pointer to data which shall be send.
 *
 * \param[in]  wDataLength
 *              Number of data bytes which shall be send.
 *
 * \return     Error/no error value
 *
 * \access     application
 * \reentrant  only for different channels
 *
 * \lastreview 21.03.11 Borck
 *             13.10.08 VTeam-Funke
 *             25.01.2008 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 13.10.08 VTeam-Funke
 * - align return value to same name
 *
 * 27.11.07 VTeam-Khler
 * - Indentation changed.
 * - CSM_S_MCNET_ASDT and CSM_S_MCNET_USDT_TX encapsulated with CSM_S_MCNET
 *
 * 25.10.06 CM-DI/ESA3-Haffner
 * - Changed proxy handling, all PTYPEs (except BAP) handled in csm_proxy.
 *
 * 16.08.06 CM-DI/ESA3-Khler
 * - Protocol type SED_STD_USDT_ISO added instead of FORD/GM/RN/...
 * - Trace handling changed.
 *
 * 27.06.06 CM-DI/ESA3-Zurmhl
 * Explicit type conversion (WORD) in CSM_M_TRACEOUT added
 *
 * 15.08.05 CM-DI/ESA3-Zurmhl
 * CSM_lDataReq now calls CSM_MCNET_lDataReq for MCNet ASDT & USDT
 *
 * 14.10.04 CM-DI/ESA3-Koehler
 * Statement without effect for ITP extended addressing removed.
 * CSM_C_PTYPE_OPEL_BR_SIGNAL removed.
 *
 * 25.02.04 CM-DI/ESU3-Koehler
 * Function call CSM_vTraceOut encapsulated with feature switch CSM_S_TRACE.
 *
 * 16.10.03 CM-DI/ESU3-Zurmuehl
 * CSM_API modifier added to function definition
 *
 * 15.07.03 CM-DI/ESU3-Koehler
 *(BYTE *) cast added to avoid warnings.
 *
 * 12.02.03 CM-DI/ESU3-Koehler
 * Addressfield- and data-pointer for message interface changed to pointer to
 * const values.
 *
 * 11.12.02 CM-DI/EHS3-Koehler
 * Downstream task is called.
 *
 * 25.10.02 CM-DI/EHS3-Koehler
 * Usage of fTxChannelFree for ITP changed: channel stays free, if data request
 * fails
 *
 * 13.09.02 CM-DI/EHS3-Koehler
 * CSM_C_PTYPE_OPEL_USDT_ISO: parameter test changed.
 * CSM_C_PTYPE_OPEL_USDT_ISO_EXT_MUX added.
 * NMD_fNmMsgSending added to accept/deny DataReq from Application.
 *
 * 02.08.02 CM-DI/EHS3-Koehler
 * CSM_C_PTYPE_OPEL_BR_TEL frames will be send without checking for the CAN-ID.
 *
 * 04.06.02 CM-DI/EHS3-Koehler
 * CSM_C_PTYPE_OPEL_USDT_ISO: parameter check expanded, ISO-TP struct used
 * for RemoteComp.
 *
 * 03.04.02 CM-DI/EHS3-Koehler
 * ProtocolType  * CSM_C_PTYPE_OPEL_USDT_ISO
 *
 * 27.03.02 CM-DI/EHS3-Pistoor
 * CSM_M_MAKE_CSM_ERROR added.
 *
 * 25.02.02 CM-DI/EHS3-Koehler
 * ProtocolType CSM_C_PTYPE_OPEL_UUDT_ISO added.
 *
 * 15.02.02 CM-DI/EHS3-Koehler
 * Initial Revision.
 * \history_end
 ******************************************************************************/
CSM_API LONG CSM_lDataReq(DWORD dwProtocolType, const void * pvAddressField, const BYTE * pbData, WORD wDataLength)
{
  /* Default return value Error                                               */
  LONG lRet = CSM_C_NO_ERROR;

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  /* ensure proper ETG tracing of invalid adressfield                         */
  BYTE Adr0 = 0x00;
  BYTE Adr1 = 0x00;

  if (pvAddressField != NULL)
  {
    Adr0 = *(((BYTE *)pvAddressField)+0);
    Adr1 = *(((BYTE *)pvAddressField)+1);
  }
  #endif
  #endif
  #endif

  #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
  /* we have at least one network protocol configured                         */

  /******************************************/
  /* general TRACE output                   */
  /******************************************/
  #if (CSM_S_TRACE == CSM_C_F_ON)
    #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR1(("####1 CSM_lDataReq: Protocol:%x  Addr:%02x%02x  Len:%x  Data:%02x",
                      ETG_CENUM(tCSM_PROTOCOL_TYPE,   (tU32) dwProtocolType),
                      Adr0,
                      Adr1,
                      (tU16) wDataLength,
                      ETG_LIST_LEN(wDataLength),
                      ETG_LIST_PTR_T8(pbData)
                    ));
      #endif
    #else
    {
      // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
      BYTE abData[CSM_MAX_TRACE_LENGTH];

      BYTE bCopyData = wDataLength > (CSM_MAX_TRACE_LENGTH-8) ? (CSM_MAX_TRACE_LENGTH-8) : (BYTE)wDataLength; // Do not copy and trace out to much bytes.
      abData[0] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(dwProtocolType));
      abData[1] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(dwProtocolType));
      abData[2] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(dwProtocolType));
      abData[3] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(dwProtocolType));
      abData[4] = *((((BYTE *)pvAddressField)+0));
      abData[5] = *((((BYTE *)pvAddressField)+1));
      abData[6] = CSM_M_GET_HIBYTE(wDataLength);
      abData[7] = CSM_M_GET_LOBYTE(wDataLength);

      CSM_vMemCopy(&(abData[8]), pbData, bCopyData);
      CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_DATA_REQ, abData, (WORD) (bCopyData+8));
    }
    #endif
  #endif // CSM_S_TRACE

  // only process data request if CSM is active
  if (csm_bCsmState == CSM_C_STATE_ACTIVE)
  {
    /******************************************/
    /* LOCAL Stack                            */
    /******************************************/
    /* select source protocol type for this data request                      */
    switch (dwProtocolType)
    {
      /****************/
      /* VW TP        */
      /****************/
      #if (   ( CSM_S_VW_TP_1_6 == CSM_C_F_ON ) \
           || ( CSM_S_VW_TP_2_0 == CSM_C_F_ON ))
      case CSM_C_PTYPE_VW_ASDT_TP_20:
      case CSM_C_PTYPE_VW_ASDT_TP_16:
           lRet = CSM_VW_lDataReq(dwProtocolType,
                                          pvAddressField,
                                          pbData,
                                          wDataLength);
      break;
      #endif

      /****************/
      /* USDT ISO     */
      /****************/
      #ifdef CSM_C_PTYPE_USED_STD_USDT_ISO
        case CSM_C_PTYPE_USED_STD_USDT_ISO:
        {
          lRet = CSM_ITP_lDataReq( dwProtocolType, pvAddressField, pbData, wDataLength);
          break;
        }
      #endif // CSM_C_PTYPE_USED_STD_USDT_ISO

      /****************/
      /* UUDT         */
      /****************/
      #ifdef CSM_C_PTYPE_USED_STD_UUDT_TX
        case CSM_C_PTYPE_USED_STD_UUDT_TX:
        {
          lRet = CNP_UUDT_lDataReq( dwProtocolType, pvAddressField, pbData, wDataLength);
        }
        break;
      #endif // CSM_C_PTYPE_USED_STD_UUDT_TX

      /****************/
      /* ISO TP       */
      /****************/
      #if (CSM_S_ISO_TP == CSM_C_F_ON)
      #if (CSM_S_ITP_ASDT_EXTENSIONS == CSM_C_F_ON)
        #if (   ( CSM_S_ITP_ASDT_EXTENSION_RENAULT == CSM_C_F_ON ) )
          case CSM_C_PTYPE_RN_ASDT:
            lRet = CSM_ITP_lDataReq(dwProtocolType,
                                            pvAddressField,
                                            pbData,
                                            wDataLength);
            break;
        #endif // CSM_C_PTYPE_RN_ASDT
      #endif // CSM_S_ITP_ASDT_EXTENSIONS
      #endif // CSM_S_ISO_TP

      /****************/
      /* OPEL USDT    */
      /****************/
      #if(defined CSM_C_PTYPE_OPEL_USDT_ISO_EXT || defined CSM_C_PTYPE_FIAT_USDT_ISO_EXT)
        #ifdef CSM_C_PTYPE_OPEL_USDT_ISO_EXT
          case CSM_C_PTYPE_OPEL_USDT_ISO_EXT:
        #endif
        #ifdef CSM_C_PTYPE_FIAT_USDT_ISO_EXT
          case CSM_C_PTYPE_FIAT_USDT_ISO_EXT:
        #endif
        {
          BYTE bIsoTpChannel;
          CSM_ADDR_OPEL_USDT_ISO *pUsdtIsoAddrField;

          pUsdtIsoAddrField = (CSM_ADDR_OPEL_USDT_ISO *) pvAddressField;

          bIsoTpChannel = 0; /* csm_bGetItpExtTelIndex(pvAddressField); Das scheint hier ein Spezialfall fr ein Fiat
                              * Projekt mit nur einem Kanal zu sein */     /* first difference to normal addressing */
          lRet = csm_lCheckItpDataReqErrors(bIsoTpChannel, pbData, wDataLength);

          /* no error found, call ITP module via downstream task */
          if (CSM_C_NO_ERROR == lRet)
          {
            #if (CSM_S_ITP_APPL_TX_BUFFER_USED == CSM_C_F_ON)
              if (csm_rIsoTpConnection[bIsoTpChannel].fApplTxBufferUsed)
              {
                /* no CSM buffer for this channel available: use application buffer */
                CNP_rItpTxBuf[bIsoTpChannel].rMsg.pbData = (BYTE*)pbData; /* ITP will not change the data but
                                                                           * with CSM buffer the same struct is used
                                                                           * so it is no const BYTE* */
              }
              else
            #endif
              {
                /*CSM_ASSERT(NULL != CNP_rItpTxBuf[bIsoTpChannel].rMsg.pbData) configuration error */
                CSM_vMemCopy(CNP_rItpTxBuf[bIsoTpChannel].rMsg.pbData, pbData, wDataLength);
              }

              CNP_rItpTxBuf[bIsoTpChannel].rMsg.wDataLength = wDataLength;
              CNP_rItpTxBuf[bIsoTpChannel].rMsg.bSender = csm_rIsoTpConnection[bIsoTpChannel].bLocalComp;
              CNP_rItpTxBuf[bIsoTpChannel].rMsg.bReceiver = csm_rIsoTpConnection[bIsoTpChannel].bRemoteComp;/* second difference to normal addressing */
              csm_rIsoTpConnectionAct[bIsoTpChannel].fTxChannelFree = FALSE; /* ITP DataReq error =>  Call DataConf and set channel free */

              CSM_M_SET_ITP_DATAREQUEST_FLAG(bIsoTpChannel);
              csm_vTriggerCsmDownstreamTask(CSM_C_DOWN_REASON_ITP_DATA_REQ);
          }
          break;
        }
      #endif // CSM_C_PTYPE_OPEL_USDT_ISO_EXT || defined CSM_C_PTYPE_FIAT_USDT_ISO_EXT)

      /*******************/
      /* more Opel USDT  */
      /*******************/
      #ifdef CSM_C_PTYPE_OPEL_USDT_ISO_EXT_MUX
        case CSM_C_PTYPE_OPEL_USDT_ISO_EXT_MUX:
        {
          BYTE bIsoTpChannel;
          CSM_ADDR_OPEL_USDT_ISO *pUsdtIsoAddrField;

          pUsdtIsoAddrField = (CSM_ADDR_OPEL_USDT_ISO *) pvAddressField;

          bIsoTpChannel = csm_bGetItpExtMuxTelIndex(pvAddressField);     /* first difference to normal addressing */
          lRet  = csm_lCheckItpDataReqErrors(bIsoTpChannel, pbData, wDataLength);

          /* no error found, call ITP module via downstream task */
          if (CSM_C_NO_ERROR == lRet)
          {
            #if (CSM_S_ITP_APPL_TX_BUFFER_USED == CSM_C_F_ON)
              if (csm_rIsoTpConnection[bIsoTpChannel].fApplTxBufferUsed)
              {
                /* no CSM buffer for this channel available: use application buffer */
                CNP_rItpTxBuf[bIsoTpChannel].rMsg.pbData = (BYTE*) pbData;/* ITP will not change the data but
                                                                           * with CSM buffer the same struct is used
                                                                           * so it is no const BYTE* */
              }
              else
            #endif
            {
              /*CSM_ASSERT(NULL != CNP_rItpTxBuf[bIsoTpChannel].rMsg.pbData) configuration error */
              CSM_vMemCopy(CNP_rItpTxBuf[bIsoTpChannel].rMsg.pbData, pbData, wDataLength);
            }

              CNP_rItpTxBuf[bIsoTpChannel].rMsg.wDataLength = wDataLength;
              CNP_rItpTxBuf[bIsoTpChannel].rMsg.bSender =
                csm_rIsoTpConnection[bIsoTpChannel].bLocalComp; /* V.K. not necessary/used ? */
              CNP_rItpTxBuf[bIsoTpChannel].rMsg.bReceiver = /* second difference to normal addressing */
                pUsdtIsoAddrField->bRemoteComp; /* extended multiplexed addressing, receiver determined by application */
              csm_rIsoTpConnectionAct[bIsoTpChannel].fTxChannelFree = FALSE; /* ITP DataReq error =>  Call DataConf and set channel free */

              CSM_M_SET_ITP_DATAREQUEST_FLAG(bIsoTpChannel);
              csm_vTriggerCsmDownstreamTask(CSM_C_DOWN_REASON_ITP_DATA_REQ);
          }
          break;
        }
      #endif // CSM_C_PTYPE_OPEL_USDT_ISO_EXT_MUX

      /*******************/
      /* Opel Broadcasts */
      /*******************/
      #ifdef CSM_C_PTYPE_OPEL_BR_SIGNAL
        case CSM_C_PTYPE_OPEL_BR_SIGNAL:
        {
          lRet = CSM_M_MAKE_CSM_ERROR( CSM_C_ERR_WARNING,
                                               CSM_E_ERROR_STATE_WHILE_DATA_REQ);
          break;
        }
      #endif // CSM_C_PTYPE_OPEL_BR_SIGNAL

      /****************/
      /* MCNet        */
      /****************/
      #if (CSM_S_MCNET == CSM_C_F_ON)
        #if (CSM_S_MCNET_ASDT == CSM_C_F_ON)
          case CSM_C_PTYPE_MCNET_ASDT_STD:
          case CSM_C_PTYPE_MCNET_ASDT_EXP:
          {
            lRet = CSM_MCNET_lDataReq (dwProtocolType, pvAddressField, pbData, wDataLength);
          }
          break;
        #endif // CSM_S_MCNET_ASDT

        #if (CSM_S_MCNET_USDT_TX == CSM_C_F_ON)
          case CSM_C_PTYPE_MCNET_USDT:
          {
            lRet = CSM_MCNET_lDataReq (dwProtocolType, pvAddressField, pbData, wDataLength);
          }
          break;
        #endif // CSM_S_MCNET_USDT_TX
      #endif // CSM_S_MCNET

      /****************/
      /* VW BAP       */
      /****************/
      #if (CSM_S_VW_BAP_TP == CSM_C_F_ON)
        case CSM_C_PTYPE_VW_BAP:
        {
          lRet = CSM_BAP_lDataReq ( *((BYTE*)pvAddressField), pbData, wDataLength);
        }
        break;
      #endif // CSM_S_VW_BAP_TP

      /****************/
      /* VW TP 2.0    */
      /****************/
      #if (CSM_S_TP20 == CSM_C_F_ON)
        #ifdef CSM_C_PTYPE_VW_ASDT_TP_20
          case CSM_C_PTYPE_VW_ASDT_TP_20:
          {
            lRet = CSM_TP20_lDataReq( dwProtocolType,
                                              pvAddressField,
                                              pbData,
                                              wDataLength);
          }
          break;
        #endif // CSM_C_PTYPE_VW_ASDT_TP_20
      #endif // CSM_S_TP20

      /************************/
      /* Renault Nissan MPDT  */
      /************************/
      #if (CSM_S_MPDT == CSM_C_F_ON)
        #ifdef CSM_C_PTYPE_RN_MPDT_C
          case CSM_C_PTYPE_RN_MPDT_C:
          {
            lRet = CSM_MPDT_lDataReq( dwProtocolType,
                                              pvAddressField,
                                              pbData,
                                              wDataLength);
          }
          break;
        #endif /* CSM_C_PTYPE_RN_MPDT_C */
        #ifdef CSM_C_PTYPE_RN_MPDT_D
          case CSM_C_PTYPE_RN_MPDT_D:
          {
            lRet = CSM_MPDT_lDataReq( dwProtocolType,
                                              pvAddressField,
                                              pbData,
                                              wDataLength);
          }
          break;
        #endif // CSM_C_PTYPE_RN_MPDT_D
      #endif // CSM_S_MPDT

      /****************/
      /* LIN Diag     */
      /****************/
      #ifdef CSM_C_PTYPE_USED_STD_LIN_DIAG
        case CSM_C_PTYPE_USED_STD_LIN_DIAG:
        {
          lRet = CSM_LIN_DIAG_lDataReq( dwProtocolType,
                                         pvAddressField,
                                         pbData,
                                         wDataLength);
        }
        break;
      #endif // CSM_C_PTYPE_USED_STD_LIN_DIAG

      /****************/
      /* LCTRL        */
      /****************/
      #ifdef CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL
              case CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL:
              {
                lRet = CSM_LCTR_lDataReq( dwProtocolType,
                                               pvAddressField,
                                               pbData,
                                               wDataLength);
              }
              break;
      #endif // CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL

      /****************/
      /* J1939        */
      /****************/
    #if (CSM_S_J1939 == CSM_C_F_ON)
      #ifdef CSM_C_PTYPE_J1939
        case CSM_C_PTYPE_J1939:
        {
          lRet = CSM_J1939_lDataReq(dwProtocolType,
                                     pvAddressField,
                                     pbData,
                                     wDataLength);
        }
        break;
      #endif // CSM_C_PTYPE_J1939
      #endif // (CSM_S_J1939 == CSM_C_F_ON)

      /****************/
      /* UNKNOWN      */
      /****************/
      default:
      {
        lRet = CSM_M_MAKE_CSM_ERROR( CSM_C_ERR_WARNING, CSM_E_INVALID_PROTOCOL_TYPE);
        break;
      }
    }
  }
  else
  {
    /* error: CSM stack is NOT active                                         */
    lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_ERROR_STATE_WHILE_DATA_REQ);
  }

  /********************************/
  /* error TRACE output           */
  /********************************/
  #if (CSM_S_TRACE == CSM_C_F_ON)
    #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      if (lRet < CSM_C_NO_ERROR)
      {
        // ETG trace
        BYTE ubErrorGrade = CSM_M_GET_EG(lRet);
        BYTE ubErrorLoc   = CSM_M_GET_EL(lRet);
        WORD ubErrorCode  = CSM_M_GET_EC(lRet);

        ETG_TRACE_USR1(("CSM_lDataReq: ERROR:%x  LOCATION:%x  GRADE:%x",
                        ETG_CENUM(tCSM_ERROR_CODE,    (tU16) ubErrorCode),
                        ETG_CENUM(tCSM_ERROR_LOCATION, (tU8) ubErrorLoc),
                        ETG_CENUM(tCSM_ERROR_GRADE,    (tU8) ubErrorGrade)
                      ));
      }
      #endif
    #else
    if (lRet < CSM_C_NO_ERROR)
    {
      // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
      BYTE abData[10];

      abData[0] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(dwProtocolType));
      abData[1] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(dwProtocolType));
      abData[2] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(dwProtocolType));
      abData[3] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(dwProtocolType));
      if (pvAddressField != NULL)
      {
        abData[4] = *(((BYTE *)pvAddressField)+0);
        abData[5] = *(((BYTE *)pvAddressField)+1);
      }

      abData[6] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(lRet));
      abData[7] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(lRet));
      abData[8] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(lRet));
      abData[9] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(lRet));

      CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_ERROR, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_DATA_REQ_RETVAL, abData, 10);
    }
    #endif
  #endif // CSM_S_TRACE

  #else // CSM_S_CNP_AVAILABLE != CSM_C_F_ON
    // error: no network protocol configured                                    */
    lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_GENERAL_ERROR); /*lint !e838 yes, sometime a variable get a new value*/
  #endif // CSM_S_CNP_AVAILABLE

  return lRet;

} //lint !e715 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally

/*******************************************************************************
 * function    CSM_lDataRead
 * \doxydocu
 * \brief      Synchronous read request on the CSM
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \param[in]  dwProtocolType
 *              Protocol type, see table 2 of CAN SW architecture.
 *
 * \param[in]  pvAddressField
 *              Protocol dependend address field.
 *
 * \param[in]  pbData
 *              Pointer to buffer in which data shall be read.
 *
 * \param[in]  wDataLength
 *              Number of data bytes which shall be send.
 *
 * \return     Error/no error value
 *
 * \access     application
 * \reentrant  only for different channels
 *
 * \lastreview 21.03.11 Borck
 *             13.10.08 VTeam-Funke
 *             09.11.2006 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 13.10.08 VTeam-Funke
 * - align return value to same name
 *
 * 21.08.06 CM-DI/ESA3-Khler
 * Trace handling changed.
 *
 * 27.06.06 CM-DI/ESA3-Zurmhl
 * Explicit type conversion (WORD) in CSM_M_TRACEOUT added
 *
 * 31.03.06 CM-DI/ESA3-Battis
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
CSM_API LONG CSM_lDataRead(DWORD dwProtocolType, const VOID * pvAddressField, BYTE * pbData, WORD wDataLength)
{
  LONG lRet = CSM_C_NO_ERROR;

  #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
  /* we have at least one network protocol configured                         */

  if (csm_bCsmState == CSM_C_STATE_ACTIVE)
  {
    /* CSM stack is active                                                    */
    switch (dwProtocolType)
    {
      #if (CSM_S_VW_BAP_TP == CSM_C_F_ON)
        case CSM_C_PTYPE_VW_BAP:
        {
          lRet = CSM_BAP_lDataRead (*(BYTE*)pvAddressField, (tCSM_BAPInOut*)((void *)pbData), wDataLength);
          break;
        }
      #endif

      default:
      {
        lRet = CSM_M_MAKE_CSM_ERROR( CSM_C_ERR_WARNING, CSM_E_INVALID_PROTOCOL_TYPE);
        break;
      }
    } /*lint !e764 */ /* Default tree is used for all protocol types except CSM_C_PTYPE_VW_BAP - 22.08.2008 krv2hi */
  }
  else
  {
    /* error: CSM stack is NOT active                                         */
    lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_ERROR_STATE_WHILE_DATA_READ);
  }

  #if (CSM_S_TRACE == CSM_C_F_ON)
    #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
    {
      BYTE Adr0 = *((((BYTE *)pvAddressField)+0));
      BYTE Adr1 = *((((BYTE *)pvAddressField)+1));

      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR2(("CSM_lDataRead - Prot:%x  AddrField:0x%X 0x%X  Data:no output here",
                      ETG_CENUM(tCSM_PROTOCOL_TYPE,   (tU32) dwProtocolType),
                      Adr0,
                      Adr1
                    ));
      #endif
    }
    #else
    {
      // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
      BYTE abData[CSM_MAX_TRACE_LENGTH];

      BYTE bCopyData = wDataLength > (CSM_MAX_TRACE_LENGTH-10) ? (CSM_MAX_TRACE_LENGTH-10) : (BYTE)wDataLength; // Do not copy and trace out to much bytes.
      abData[0] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(dwProtocolType));
      abData[1] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(dwProtocolType));
      abData[2] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(dwProtocolType));
      abData[3] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(dwProtocolType));
      abData[4] = *((((BYTE *)pvAddressField)+0));
      abData[5] = *((((BYTE *)pvAddressField)+1));
      abData[6] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(lRet));
      abData[7] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(lRet));
      abData[8] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(lRet));
      abData[9] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(lRet));

      CSM_vMemCopy(&(abData[10]), pbData, bCopyData);
      CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_DATA_READ, abData, (WORD) (bCopyData+10) );
    }
    #endif
  #endif // CSM_S_TRACE

  #else // CSM_S_CNP_AVAILABLE
    // error: no network protocol configured
    lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_GENERAL_ERROR); /*lint !e838 yes, sometime a variable get a new value*/
  #endif // CSM_S_CNP_AVAILABLE

  return lRet;
} //lint !e715 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally  //lint !e818 (pbData could be pointer to const) specified interface - 21.08.2008 krv2hi */

/*******************************************************************************
 * function    CSM_lDataIndProcessed
 * \doxydocu
 * \brief      The received and indicated data are processed.
 *
 *             This function is called from application after YYY_vCSMDataInd()
 *             to indicate that the data is processed (only for some configured)
 *             channels). The CSM frees the Rx buffer.
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \param[in]  dwProtocolType
 *              Protocol type, see table 2 of CAn SW architecture.
 *
 * \param[in]  pvAddressField
 *              Protocol dependend address field.
 *
 * \return     Error/no error value
 *
 * \access     application
 * \reentrant  only for different channels
 *
 * \lastreview 21.03.11 Borck
 *             13.10.08 VTeam-Funke
 *             16.09.08 Prhl
 *
 * \history_begin
 * 13.10.08 VTeam-Funke
 * - align return value to same name
 * - reduce return points of function
 *
 * 01.10.08  Prhl
 * - support for CSM_C_PTYPE_RN_MPDT_C and CSM_C_PTYPE_RN_MPDT_D added.
 *
 * 22.08.08  VTeam-Khler
 * - Return value initialized with CSM_E_INIT_WARNING instead of CSM_C_NO_ERROR.
 * - Switch replaced by if for standard rx UUDT protocol type to avoid lint
 *   warning.
 * - Indentation changed.
 * - CSM_C_PTYPE_OPEL_USDT_ISO replaced by CSM_C_PTYPE_USED_STD_USDT_ISO in #if
 *   statement
 *
 * 17.10.06 CM-DI/ESA3-Khler
 * - PVOID replaced by void *.
 *
 * 21.08.06 CM-DI/ESA3-Khler
 * - Trace handling changed.
 *
 * 16.10.03 CM-DI/ESU3-Zurmuehl
 * CSM_API modifier added to function definition
 *
 * 20.08.03 CM-DI/ESU3-Franke
 * added call of cnp_lDataIndProcessed for CSM_C_PTYPE_FORD_USDT_ISO
 *
 * 16.06.03 CM-DI/ESU3-Koehler
 * CNP_lIsoTp_DataIndProcessed used insead of CNP_vItpFreeAssBuffer.
 *
 * 17.03.03 CM-DI/ESU3-Koehler
 * csm_rIsoTpConnection[bIsoTpChannel].fApplSetsRxBufferFree will be checked.
 * #ifdefs changed
 *
 * 12.12.02 CM-DI/EHS3-Koehler
 * CSM_C_PTYPE_OPEL_USDT_ISO and CSM_C_PTYPE_OPEL_USDT_ISO_EXT_MUX added.
 *
 * 15.11.02 CM-DI/EHS3-Pistoor
 * Initial Revision.
 * \history_end
 ******************************************************************************/
CSM_API LONG CSM_lDataIndProcessed(DWORD dwProtocolType, const void * pvAddressField)
{
  LONG lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_WARNING, CSM_E_INIT_WARNING);

  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  /* ensure proper ETG tracing of invalid adressfield                         */
  BYTE Adr0 = 0x00;
  BYTE Adr1 = 0x00;

  if (pvAddressField != NULL)
  {
    Adr0 = *(((BYTE *)pvAddressField)+0);
    Adr1 = *(((BYTE *)pvAddressField)+1);
  }
  #endif
  #endif
  #endif

  #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
    #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
        #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
        ETG_TRACE_USR1(("DataIndProcessed       --> Protocol: %x Addr: %02x%02x",
        ETG_CENUM(tCSM_PROTOCOL_TYPE,   (tU32) dwProtocolType),
        Adr0,
        Adr1 ));
        #endif
      #else
      {
        // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
        BYTE abData[6];

        abData[0] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(dwProtocolType));
        abData[1] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(dwProtocolType));
        abData[2] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(dwProtocolType));
        abData[3] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(dwProtocolType));
        abData[4] = *((((BYTE *)pvAddressField)+0));
        abData[5] = *((((BYTE *)pvAddressField)+1));
        CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_DATA_IND_PROCESSED, abData, 6);
      }
      #endif
    #endif // CSM_S_TRACE

    // Check if CSM is in correct state for processing this API function call
    if ( (csm_bCsmState != CSM_C_STATE_ACTIVE) && (csm_bCsmState != CSM_C_STATE_INITIALIZED) )
    {
      // CSM not in state ACTIVE or INITIALIZED
      lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_CSM_STATE_FORBIDS_CALL);
    }
    else
    {
      /****************/
      /* ISO TP       */
      /****************/
      #if ((CSM_S_ISO_TP == CSM_C_F_ON) && (CSM_S_ITP_APPL_SETS_RX_BUFFER_FREE  == CSM_C_F_ON))
      /* no switch used to save ROM in V850 Projects */
      #if (defined CSM_C_PTYPE_USED_STD_USDT_ISO) || (defined CSM_C_PTYPE_OPEL_USDT_ISO_EXT_MUX) || (defined CSM_C_PTYPE_FIAT_USDT_ISO_EXT)
      {
        #if (defined CSM_C_PTYPE_USED_STD_USDT_ISO)
          if (CSM_C_PTYPE_USED_STD_USDT_ISO_USDT_ISO == dwProtocolType)
          {
            bIsoTpChannel = csm_bGetItpNATelIndex(pvAddressField);
          }
          #if (defined CSM_C_PTYPE_OPEL_USDT_ISO_EXT_MUX) /* extended mux and normal must be defined */
            else if (CSM_C_PTYPE_OPEL_USDT_ISO_EXT_MUX == dwProtocolType)
            {
              bIsoTpChannel = csm_bGetItpExtMuxTelIndex(pvAddressField);
            }
          #endif /* CSM_C_PTYPE_OPEL_USDT_ISO_EXT_MUX */
        #elif (defined CSM_C_PTYPE_FIAT_USDT_ISO_EXT)
            if (CSM_C_PTYPE_FIAT_USDT_ISO_EXT == dwProtocolType)
            {
              bIsoTpChannel = csm_bGetItpExtTelIndex(pvAddressField);
            }
        #endif
        else
        {
          /* exit because of wrong protocol type */
          return( CSM_M_MAKE_CSM_ERROR( CSM_C_ERR_WARNING, CSM_E_INVALID_PROTOCOL_TYPE));
        }

        /* Valid protocol type => check channel/address and configuration */
        if ((bIsoTpChannel < CSM_C_ISO_TP_MAX_CONNECTIONS) &&
            (csm_rIsoTpConnection[bIsoTpChannel].fApplSetsRxBufferFree))
        {
          lRet = CNP_ITP_lDataIndProcessed(bIsoTpChannel);
        }
        else
        {
          lRet = CSM_M_MAKE_CSM_ERROR( CSM_C_ERR_WARNING, CSM_E_INVALID_ADDRESS);
        }
      }
      #endif /* CSM_C_PTYPE_OPEL_USDT_ISO || ... */
      #endif /* CSM_S_ISO_TP && CSM_S_ITP_APPL_SETS_RX_BUFFER_FREE */

      /****************/
      /* Ford USDT    */
      /****************/
      #if (defined CSM_C_PTYPE_FORD_USDT_ISO)
      if (dwProtocolType == CSM_C_PTYPE_FORD_USDT_ISO)
      {
        /*TODO:*/
        /*      lRet = cnp_lDataIndProcessed(pvAddressField);*/
      } /* if dwProtocolType == CSM_C_PTYPE_FORD_USDT_ISO */
      #endif /* CSM_C_PTYPE_FORD_USDT_ISO */

      /***********************/
      /* Renault Nissan MPDT */
      /***********************/
      #ifdef CSM_C_PTYPE_RN_MPDT_C
      if (dwProtocolType == CSM_C_PTYPE_RN_MPDT_C)
      {
        lRet =  CSM_MPDT_lDataIndProcessed( dwProtocolType,
                                                    pvAddressField);
      }
      #endif /* CSM_C_PTYPE_RN_MPDT_C */

      #ifdef CSM_C_PTYPE_RN_MPDT_D
      if (dwProtocolType == CSM_C_PTYPE_RN_MPDT_D)
      {
        lRet = CSM_MPDT_lDataIndProcessed(dwProtocolType, pvAddressField);
      }
      #endif /* CSM_C_PTYPE_RN_MPDT_D */

      /****************/
      /* UUDT       */
      /****************/
      #ifdef CSM_C_PTYPE_USED_STD_UUDT_RX
      if (dwProtocolType == CSM_C_PTYPE_USED_STD_UUDT_RX)
      {
        lRet = CNP_UUDT_lDataIndProcessed( dwProtocolType, pvAddressField);
      }
      #endif /* CSM_C_PTYPE_USED_STD_UUDT_RX */

      /****************/
      /* MCNET        */
      /****************/
      #if (CSM_S_MCNET == CSM_C_F_ON)
      switch (dwProtocolType)
      {
        #if (defined CSM_C_PTYPE_MCNET_ASDT_STD)
          case CSM_C_PTYPE_MCNET_ASDT_STD:
          {
            BYTE bTpChannel =CSM_MCNET_bGetAsdtConnection(pvAddressField);

            if( bTpChannel < CSM_C_MCNET_MAX_CONNECTIONS)
            {
              CNP_MCNET_lAsdtDataIndProcessed( CNP_MCNET_C_LDATA_SERV-1, bTpChannel);/*lint !e778*/          /* => CSM sets it free */
            }
            else
            {
              /*CSM_ASSERT( FALSE);*/
              lRet = CSM_M_MAKE_CSM_ERROR (CSM_C_ERR_ERROR, CSM_E_INVALID_ADDRESS);  /* Error: State unknown */
            }
          }
          break;
        #endif /* CSM_C_PTYPE_MCNET_ASDT_STD */
        #if (defined CSM_C_PTYPE_MCNET_ASDT_EXP)
          case CSM_C_PTYPE_MCNET_ASDT_EXP:
          {
            BYTE bTpChannel =CSM_MCNET_bGetAsdtConnection(pvAddressField);

            if( bTpChannel < CSM_C_MCNET_MAX_CONNECTIONS)
            {
              CNP_MCNET_lAsdtDataIndProcessed( CNP_MCNET_C_EDATA_SERV-1, bTpChannel);          /* => CSM sets it free */
            }
            else
            {
              /*CSM_ASSERT( FALSE);*/
              lRet = CSM_M_MAKE_CSM_ERROR (CSM_C_ERR_ERROR, CSM_E_INVALID_ADDRESS);  /* Error: State unknown */
            }
          }
          break;
        #endif /* CSM_C_PTYPE_MCNET_ASDT_EXP */
        default:
        {
        }
        break;
      }  /* end switch (dwProtocolType) */
      #endif /*(CSM_S_MCNET == CSM_C_F_ON)*/

      /****************/
      /* VW TP2.0     */
      /****************/
      #if (CSM_S_TP20 == CSM_C_F_ON)
      #ifdef CSM_C_PTYPE_VW_ASDT_TP_20
        switch (dwProtocolType)
        {
          case CSM_C_PTYPE_VW_ASDT_TP_20:
          {
            lRet =  CSM_TP20_lDataIndProcessed( dwProtocolType, pvAddressField);
          }
          break;
        default:
        {
        }
        break;
        }
      #endif /* CSM_C_PTYPE_VW_ASDT_TP_20 */
      #endif /* (CSM_S_TP20 == CSM_C_F_ON) */

      /****************/
      /* VW TP        */
      /****************/
      #if (( CSM_S_VW_TP_1_6 == CSM_C_F_ON ) || ( CSM_S_VW_TP_2_0 == CSM_C_F_ON ))
      if ( (dwProtocolType == CSM_C_PTYPE_VW_ASDT_TP_20)
           || (dwProtocolType == CSM_C_PTYPE_VW_ASDT_TP_16))
      {
        lRet = CSM_VW_lDataIndProcessed(dwProtocolType, pvAddressField);
      }
      else
      {
        lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_INVALID_PROTOCOL_TYPE);
      }
      #endif /* CSM_VW_TP */

      /****************/
      /* J1939        */
      /****************/
    #if (CSM_S_J1939 == CSM_C_F_ON)
      #ifdef CSM_C_PTYPE_J1939
      if(dwProtocolType == CSM_C_PTYPE_J1939)
      {
        lRet = CSM_J1939_lDataIndProcessed(dwProtocolType, pvAddressField);
      }
      #endif
      #endif
    }
  #else /* CSM_S_CNP_AVAILABLE == CSM_C_F_ON */
    lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_GENERAL_ERROR); /*lint !e838 yes, sometime a variable get a new value*/
  #endif /* CSM_S_CNP_AVAILABLE == CSM_C_F_ON */

  return (lRet);
} //lint !e715 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally

#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
#if ( ( CSM_S_VW_TP_2_0 == CSM_C_F_ON ) || ( CSM_S_VW_TP_1_6 == CSM_C_F_ON ) )
/*******************************************************************************
 * function    CSM_lConfigChannel
 * \doxydocu
 * \brief      Implements channel configuration.
 *
 * \param[in]  dwProtocolType
 *              Protocol type
 *
 * \param[in]  pvAddressField
 *              Protocol dependend address field.
 *
 * \param[in]  lConfData
 *              Configuration data for TP
 *
 * \return     Error/no error value
 *
 * \access     main loop
 * \reentrant  no
 *
 * \lastreview  24.07.08 VTeam-Zurmhl .. Finding: Maybe an obsolete fct. (not used anymore?)
 *
 * \history_begin
 *
 * 10.05.2004 CM-CR/ESD4 P.Tuschik, CM-CR/ESD3 T.Mehring
 * Initial Revision.
 * \history_end
 ******************************************************************************/
LONG CSM_lConfigChannel(DWORD dwProtocolType, const void* pvAddressField, LONG lConfData)
{
  return CSM_VW_lConfigChannel(dwProtocolType, pvAddressField, lConfData);
}
#endif  //CSM_S_VW_TP_2_0
#endif // CSM_S_CNP_AVAILABLE

/*******************************************************************************
 * function    CSM_lCommunicationReq
 * \doxydocu
 * \brief      Requests a transport protocol connection or CAN functionality.
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \param[in]  bBus
 *              Bus number
 *
 * \param[in]  dwProtocolType
 *              Protocol type, see table 2 of CAn SW architecture.
 *
 * \param[in]  pvAddressField
 *              Protocol dependend address field.
 *
 * \param[in]  bAction
 *              CSM_C_CONNECTED
 *              CSM_C_DISCONNECTED
 *
 * \param[in]  wApplID
 *              Application id
 *
 * \return     Error/no error value
 *
 * \access     application
 * \reentrant  no
 *
 * \lastreview  21.03.11 Borck
                29.04.10 VTeam-Franke
 *              13.10.08 VTeam-Funke
 *              16.04.08 VTeam-Prhl
 *
 * \history_begin
 *
 * 29.04.10  VTeam-Franke
 * -  bBus validity check bug fix
 *
 * 13.10.08  VTeam-Funke
 * - added static error handling
 * - reduce return points of function to get trace of returnvalue!
 *
 * 13.03.08  VTeam-Khler
 * - CSM_C_PTYPE_USED_STD_BR_SIGNAL added.
 * - indentation changed
 *
 * 25.09.07 RBIN/EDI2-Basavaraj
 * - Added switch case for handling CSM_C_PTYPE_VCC_BR_SIGNAL.
 *
 * 25.10.06 CM-DI/ESA3-Haffner
 * - Changed proxy handling, all PTYPEs (except BAP) handled in csm_proxy.
 *
 * 17.10.06 CM-DI/ESA3-Khler
 * - PVOID replaced by void *.
 *
 * 26.09.06 CM-DI/ESA3-Khler
 * - Trace output changed.
 *
 * 21.08.06 CM-DI/ESA3-Khler
 * - Protocol type USED_STD_USDT_ISO added instead of FORD/GM/...
 * - Cases for BR_SIGNAL protocol types rearanged.
 * - Trace handling changed.
 *
 * 28.06.06 CM-DI/ESA3-Haffner
 * - routing to new CSM proxy implemented
 *
 * 16.10.03 CM-DI/ESU3-Zurmuehl
 * CSM_API modifier added to function definition
 *
 * 20.08.03 CM-DI/ESU3-Franke
 * - extensions for Ford TP_EX
 * - answer, but ignore communication requests from CCS
 * - treat action != connect as disconnect
 *
 * 19.05.03 CM-DI/ESU3-Franke
 * CCS communication confirmation for Ford TP_EX added
 *
 * 20.02.03 CM-DI/ESU3-Pistoor
 * Made parameter pvAddressField to const
 *
 * 27.02.02 CM-DI/EHS3-Pistoor
 * Initial Revision.
 * \history_end
 ******************************************************************************/
CSM_API LONG CSM_lCommunicationReq(BYTE bBus, DWORD dwProtocolType, const void * pvAddressField, BYTE bAction, WORD wApplID)
{
  LONG  lRet = CSM_C_NO_ERROR;

  /* ensure proper ETG tracing of invalid adressfield                         */
  #if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  BYTE Adr0 = 0x00;
  BYTE Adr1 = 0x00;

  if (pvAddressField != NULL)
  {
    Adr0 = *(((BYTE *)pvAddressField)+0);
    Adr1 = *(((BYTE *)pvAddressField)+1);
  }
  #endif
  #endif
  #endif

  /*************************/
  /* general TRACE output  */
  /*************************/
  #if (CSM_S_TRACE == CSM_C_F_ON)
    #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR1(("####1 CSM_lCommunicationReq: Bus:%x  Protocol:%x  Addr:%02x%02x  Action:%x  ApplID:%x",
                      ETG_CENUM(tCSM_BUS_NUMBER,      (tU8)  bBus),
                      ETG_CENUM(tCSM_PROTOCOL_TYPE,   (tU32) dwProtocolType),
                      Adr0,
                      Adr1,
                      bAction,
                      ETG_CENUM(tCSM_APPL_ID,         (tU16) wApplID)
                    ));
      #endif
    #else
    {
      // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
      BYTE abData[10];

      abData[0] = bBus;
      abData[1] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(dwProtocolType));
      abData[2] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(dwProtocolType));
      abData[3] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(dwProtocolType));
      abData[4] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(dwProtocolType));
      if (pvAddressField != NULL)
      {
        abData[5] = *(((BYTE *)pvAddressField)+0);
        abData[6] = *(((BYTE *)pvAddressField)+1);
      }
      abData[7] = bAction;
      abData[8] = CSM_M_GET_HIBYTE(wApplID);
      abData[9] = CSM_M_GET_LOBYTE(wApplID);

      CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_COMMUNICATION_REQ, abData, 10);
    }
    #endif
  #endif

  // Check if CSM is in correct state for processing this API function call
  if (csm_bCsmState != CSM_C_STATE_ACTIVE)
  {
    // CSM not in state ACTIVE
    #if (CSM_S_ENABLE_INDICATE_STATIC_ERROR == CSM_C_F_ON)
    // Check if CSM has not static errors
    if ((bBus < CSM_C_MAX_CAN_BUS) && (csm_lActStaticError[bBus] != CSM_C_NO_ERROR))
    {
      // CSM has serious initialisation error
      lRet = csm_lActStaticError[bBus];
    }
    else
    #endif
    {
      lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_CSM_STATE_FORBIDS_CALL);
    }
  }
  else
  {
    if (bBus < CSM_C_MAX_CAN_BUS)
    {
      #if (CSM_S_REMOTE_STACK == CSM_C_F_ON)
      /****************/
      /* REMOTE Stack */
      /****************/
      /* all protocol types handled in csm_proxy, except BAP                  */
      switch(dwProtocolType)
      {
        #if (CSM_S_VW_BAP_TP == CSM_C_F_ON)
        case CSM_C_PTYPE_VW_BAP:
        {
          lRet = CSM_BAP_lCommunicationReq(bBus, *((BYTE*)pvAddressField), bAction);
        }
        break;
        #endif /* CSM_S_VW_BAP_TP */

        // MPDT and other TPs are forwarded to remote processor (Proxy)
        #if (CSM_S_MPDT == CSM_C_F_ON)
        case CSM_C_PTYPE_RN_MPDT_C:
        case CSM_C_PTYPE_RN_MPDT_D:
        #endif
        default:
        {
          lRet = CSM_PROXY_lCommunicationReq( bBus, dwProtocolType, pvAddressField, bAction, wApplID);
        }
        break;
      }
      #else  /* CSM_S_REMOTE_STACK */
      /****************/
      /* LOCAL Stack  */
      /****************/
      switch(dwProtocolType)
      {
        #if ( CSM_S_CSM_VW == CSM_C_F_ON )
        case CSM_C_PTYPE_VW_ASDT_TP_20:
        case CSM_C_PTYPE_VW_ASDT_TP_16:
        {
          lRet = CSM_VW_lCommunicationReq(bBus, dwProtocolType, pvAddressField,
                                          bAction, wApplID);
        }
        break;
        #endif /* CSM_S_CSM_VW */

        #ifdef CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL
        case CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL:
        {
          lRet = CSM_LCTR_lCommunicationReq( bBus, dwProtocolType, pvAddressField, bAction, wApplID);
        }
        break;
        #endif /* CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL */

        #ifdef CSM_C_PTYPE_USED_STD_USDT_ISO
        case CSM_C_PTYPE_USED_STD_USDT_ISO:
        {
          #if (CSM_S_ITP_COMMUNICATION_FCT == CSM_C_F_ON)
          lRet = CSM_ITP_lCommunicationReq( bBus, dwProtocolType, pvAddressField, bAction, wApplID);
          #else
          lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_CONFIGURATION);
          #endif
        }
        break;
        #endif /* CSM_C_PTYPE_USED_STD_USDT_ISO */

        #ifdef CSM_C_PTYPE_USED_STD_BR_SIGNAL
        case CSM_C_PTYPE_USED_STD_BR_SIGNAL:
        {
          lRet = CSM_CBR_lCommunicationReq( bBus, dwProtocolType, pvAddressField, bAction, wApplID);
        }
        break;
        #endif /* CSM_C_PTYPE_USED_STD_BR_SIGNAL */

        #ifdef CSM_C_PTYPE_FIAT_USDT_ISO_EXT
        case CSM_C_PTYPE_FIAT_USDT_ISO_EXT:
          /*TODO : if CSM_lCommunicationReq is used by the application, add flags here ! */
        break;
        #endif /* CSM_C_PTYPE_FIAT_USDT_ISO_EXT */

        #ifdef CSM_C_PTYPE_MCNET_ASDT_STD
        case CSM_C_PTYPE_MCNET_ASDT_STD:
        {
          lRet = CSM_MCNET_lCommunicationReq( bBus, dwProtocolType, pvAddressField, bAction, wApplID);
        }
        break;
        #endif /* CSM_C_PTYPE_MCNET_ASDT_STD */

        #ifdef CSM_C_PTYPE_VW_BAP
        case CSM_C_PTYPE_VW_BAP:
        {
          if (pvAddressField != NULL)
          {
            lRet = CSM_BAP_lCommunicationReq(bBus, *((BYTE*)pvAddressField), bAction);
          }
          else
          {
            lRet = CSM_M_MAKE_CSM_ERROR( CSM_C_ERR_WARNING, CSM_E_NULL_POINTER);
          }
        }
        break;
        #endif /* CSM_C_PTYPE_VW_BAP */

        #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
        #if (CSM_S_TP20 == CSM_C_F_ON)
        #ifdef CSM_C_PTYPE_VW_ASDT_TP_20
        case CSM_C_PTYPE_VW_ASDT_TP_20:
        {
          lRet = CSM_TP20_lCommunicationReq( bBus, dwProtocolType, pvAddressField, bAction, wApplID);
        }
        break;
        #endif /* CSM_C_PTYPE_VW_ASDT_TP_20 */
        #endif /* (CSM_S_TP20 == CSM_C_F_ON) */

        #if (CSM_S_MPDT == CSM_C_F_ON)
        #ifdef CSM_C_PTYPE_RN_MPDT_C
        case CSM_C_PTYPE_RN_MPDT_C:
        {
          lRet = CSM_MPDT_lCommunicationReq( bBus, dwProtocolType, pvAddressField, bAction, wApplID);
        }
        break;
        #endif /* CSM_C_PTYPE_RN_MPDT_C */
        #endif /* (CSM_S_MPDT == CSM_C_F_ON) */
        #endif /* (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)*/

        #ifdef CSM_C_PTYPE_MCNET_LOCAL_CTRL
          #error so geht es nicht CSM_lBusStateChange() darf nicht auf Aplikationslevel aufgerufen werden!!
        case CSM_C_PTYPE_MCNET_LOCAL_CTRL:
        {
          #if (CSM_S_STATE_MASCHINE == CSM_C_F_ON)
          if ( (bAction == CSM_C_CONNECTED))
          {
            /* TODO ReqCommunicationState = CSM_C_CONNECTED */
            if (csm_abCsmSubState[ bBus] != CSM_C_STATE_BUS_UP)
            {
              BYTE bNewState;

              bNewState = CSM_lBusStateChange(bBus, CSM_C_STATE_BUS_UP);
              if (bNewState == CSM_C_STATE_BUS_UP)
              {
                /* TODO CSM_lCommunicationCon( CSM_C_CONNECTED); */
              }
              else
              {
                /* TODO fWaitForUp = TRUE */
              }
            }
            else
            {
              /* TODO CSM_lCommunicationCon( CSM_C_CONNECTED); */
            }
          }
          else
          {
            /* TODO ReqCommunicationState = CSM_C_DISCONNECTED */
            /* TODO */
            CSM_lBusStateChange(bBus, CSM_C_STATE_BUS_DOWN);
          }
          #endif /* CSM_S_STATE_MASCHINE */
        }
        break;
        #endif /* CSM_C_PTYPE_MCNET_LOCAL_CTRL */

        #ifdef CSM_C_PTYPE_J1939
        case CSM_C_PTYPE_J1939:
        {
          lRet = CSM_J1939_lCommunicationReq( bBus, dwProtocolType, pvAddressField, bAction, wApplID);
        }
        break;
        #endif /*CSM_C_PTYPE_J1939*/

        /******************************/
        /* Nissan OSEKI NWM           */
        /******************************/
        #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
        #if (CSM_S_OSEKI_NM == CSM_C_F_ON)
        #ifdef CSM_C_PTYPE_RN_OSEKI_NWM
        /* The communication request for NM is used to pass commands          */
        /* directly from middleware components to the CSM(CNM) bypassing the  */
        /* CSM_DRV layer. This is a very restricted and simple solution.      */
        case CSM_C_PTYPE_RN_OSEKI_NWM:
        {
          lRet = CNM_OSEKI_lCommunicationReq(bBus, dwProtocolType, pvAddressField, bAction, wApplID);
        }
        break;
        #endif
        #endif
        #endif

        default:
          lRet = CSM_M_MAKE_CSM_ERROR( CSM_C_ERR_WARNING, CSM_E_INVALID_PROTOCOL_TYPE);
          break;

          #if 0
            /* Das war mal unser Lsungsvorschlag zu diesem Thema fr alle anderen Proto-
               kolltypen.
               Es ist ja eigentlich ein ziemlich einfaches Thema, bei einer Anforderung
               ein Bit zu setzen und bei Abfrage von CSM_bGetBusHold entsprechend den
               Busstatus anzupassen. Die uerst umfangreiche Lsung mit dem Compiler-
               schalter CSM_C_PTYPE_LOCAL_CTRL zeigt, wie kompliziert man diesen billigen
               Mechanismus machen kann.
               Es ist allerdings eine echte Herausforderung sich den Code zu Gemte zu
               fhren, aber nach 1 Tag intensivem Studium schafft man es dann doch zu
               verstehen, wie man die Sache endlich bewltigt. Bis jetzt hat sich aber
               immer wieder besttigt: Die einfachsten Lsungen sind meist die Besten !! */
            /* Aber leider war die einfach Lsung zu einfach. Bei der einfachen Lsung
               kann es vorkommen, dass der Bus angeht obwohl er ausgehen soll oder umge-
               kehrt. Zum Beispiel wenn Task A die letzte Task ist, die den Bus braucht und nun
               CSM_lBusStateChange( bBus, CSM_C_STATE_BUS_DOWN) ausgefhrt werden soll. Aber
               Task B jetzt Task A unterbricht und den Bus anfordert. Task A luft weiter nach
               dem Task B CSM_lBusStateChange(bBus, CSM_C_STATE_BUS_UP) beendet hat und fhrt
               nun CSM_lBusStateChange( bBus, CSM_C_STATE_BUS_DOWN) aus. Jetzt ist es passiert
               das Task B den Bus braucht er aber aus ist. */
            default:
            {
              BOOL bOld = CSM_bGetBusHold(bBus);         /* get old Bus Requests */
              if (bAction == CSM_C_CONNECTED)
                lRet = CSM_lSetBusHold(bBus, dwProtocolType, wApplID);
              else lRet = CSM_lClearBusHold(bBus, dwProtocolType, wApplID);
              if (CSM_bGetBusHold(bBus))
              {  if (!bOld)                         /* if state changes from OFF */
                 {                                     /* to ON, then set bus up */
                   #if (CSM_S_STATE_MASCHINE == CSM_C_F_ON)
                     CSM_lBusStateChange(bBus, CSM_C_STATE_BUS_UP);
                   #endif /* CSM_S_STATE_MASCHINE */
                 }
              }
              else                                 /* if no more request pending */
              {                                             /* then set bus down */
                #if (CSM_S_STATE_MASCHINE == CSM_C_F_ON)
                  CSM_M_SET_HIGHEST_ERROR_GRADE(lRet, CSM_lBusStateChange(bBus, CSM_C_STATE_BUS_DOWN));//lint !e774 PQM_authorized_127
                #endif /* CSM_S_STATE_MASCHINE */
              }
            }
            break;
          #endif  /* 0 */

      }; /* switch(dwProtocolType) *//*lint !e764 Reason: if no protocoll type is defined in a project no case is available*/
      #endif /* CSM_S_REMOTE_STACK */
    }
    else
    {
      lRet = CSM_M_MAKE_CSM_ERROR( CSM_C_ERR_WARNING, CSM_E_INVALID_BUS_NUMBER);
    }
  }

  // trace ERRORS if there are some
  if (lRet < CSM_C_NO_ERROR)
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
        #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
        // ETG trace
        BYTE ubErrorGrade = CSM_M_GET_EG(lRet);
        BYTE ubErrorLoc   = CSM_M_GET_EL(lRet);
        WORD ubErrorCode  = CSM_M_GET_EC(lRet);

        ETG_TRACE_USR1(("CommunicationReq: ERROR:%x  LOCATION:%x  GRADE:%x",
                        ETG_CENUM(tCSM_ERROR_CODE,    (tU16) ubErrorCode),
                        ETG_CENUM(tCSM_ERROR_LOCATION, (tU8) ubErrorLoc),
                        ETG_CENUM(tCSM_ERROR_GRADE,    (tU8) ubErrorGrade)
                      ));
        #endif
      #else
        // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
        BYTE abData[14];

        abData[0] = bBus;
        abData[1] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(dwProtocolType));
        abData[2] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(dwProtocolType));
        abData[3] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(dwProtocolType));
        abData[4] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(dwProtocolType));
        if (pvAddressField != NULL)
        {
          abData[5] = *(((BYTE *)pvAddressField)+0);
          abData[6] = *(((BYTE *)pvAddressField)+1);
        }
        abData[7] = bAction;
        abData[8] = CSM_M_GET_HIBYTE(wApplID);
        abData[9] = CSM_M_GET_LOBYTE(wApplID);

        abData[10] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(lRet));
        abData[11] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(lRet));
        abData[12] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(lRet));
        abData[13] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(lRet));

        CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_ERROR, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_COMMUNICATION_REQ_RETVAL, abData, 14);
      #endif // standard trace
    #endif // CSM_S_TRACE
  }

  return lRet;
} //lint !e715 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally

/*******************************************************************************
 * function    CSM_lSignalWrite
 * \doxydocu
 * \brief      Stores signal data and triggers transmission.
 *
 *             Stores signal data in telegram buffer.\n
 *             If transmission type is CSM_C_TX_IMMEDIATELY, telegram has to be
 *             transmitted immediately, otherwise (CSM_C_TX_AS_CONFIGURED) as
 *             configured (for cyclic/event mixed modes cyclic).
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \param[in]  dwSignalId
 *              Signal identifier (signal name)
 *
 * \param[in]  pvNewSignalData
 *              Pointer to data buffer
 *
 * \param[in]  bDataBufferLength
 *              Length of data buffer (security element)
 *
 * \param[in]  bTxType
 *              transmission mode:
 *                CSM_C_TX_AS_CONFIGURED -   like in config.-Tabelle
 *                CSM_C_TX_IMMEDIATELY -     immediately transmission
 *
 * \return     Error/no error value
 *
 * \access     application
 * \reentrant  no
 *
 * \lastreview 21.03.11 Borck
 *             13.10.08 VTeam-Funke
 *             25.01.08 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 13.10.08  VTeam-Funke
 * - added static error handling
 * - reduce return points of function
 *
 * 17.10.06 CM-DI/ESA3-Khler
 * - PVOID replaced by void *.
 *
 * 21.08.06 CM-DI/ESA3-Khler
 * * - Trace handling changed.
 *
 * 16.10.03 CM-DI/ESU3-Zurmuehl
 * CSM_API modifier added to function definition
 *
 * 27.02.02 CM-DI/EHS3-Pistoor
 * Initial Revision.
 * \history_end
 ******************************************************************************/
CSM_API LONG CSM_lSignalWrite(DWORD dwSignalId, const void * pvNewSignalData, BYTE bDataBufferLength, BYTE bTxType)
{
  LONG lRet = CSM_C_NO_ERROR;

  #if (CSM_S_CBR_AVAILABLE == CSM_C_F_ON)
    if (( csm_bCsmState != CSM_C_STATE_INITIALIZED) && ( csm_bCsmState != CSM_C_STATE_ACTIVE))
    {
      // error: CSM has not been initialized yet
      CSM_vAssert(FALSE);
      return (CSM_M_MAKE_ERROR( CSM_C_ERR_SEVERE_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_CSM_STATE_FORBIDS_CALL));
    }

    // CSM is initialized so check project dependency
    #if ( CSM_S_CSM_VW == CSM_C_F_ON )
      // special VW handling
      lRet = CSM_VW_lSignalWrite(dwSignalId, pvNewSignalData, bDataBufferLength, bTxType);
    #else
      // all other projects except VW
      #if (CSM_S_BUS_WAKE_TEMP == CSM_C_F_ON)
      if( (bTxType & CSM_C_TX_BUS_WAKE_TEMP) == CSM_C_TX_BUS_WAKE_TEMP)
      {
        csm_fWakeTemp = TRUE;
        csm_vTriggerCsmDownstreamTask( CSM_C_DOWN_REASON_CSM_BUS_STATE_CHANGED);
        CSM_lTimerStart( CSM_TIMER_WAKE_TEMP, CSM_C_WAKE_TEMP_TIME);
      }
      #endif // CSM_S_BUS_WAKE_TEMP == CSM_C_F_ON

      #if ((CSM_S_iMX_SIGNAL_ABSTRACTION == CSM_C_F_ON) || (CSM_S_v850_SIGNAL_ABSTRACTION == CSM_C_F_ON))
        // generic handling via broadcast abstraction layer
        #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
        CSM_M_SET_HIGHEST_ERROR_GRADE(lRet, CBR_AL_lSignalWrite(dwSignalId, pvNewSignalData, bDataBufferLength, (bTxType & ~CSM_C_TX_BUS_WAKE_TEMP))); //lint !e774 PQM_authorized_127
        #else
        CSM_M_SET_HIGHEST_ERROR_GRADE(lRet, CBR_AL_lSignalWrite(dwSignalId, pvNewSignalData, bDataBufferLength, (bTxType))); //lint !e774 PQM_authorized_127
        #endif // REMOTE_STACK
      #else
        // generic handling via broadcast server
        #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
        CSM_M_SET_HIGHEST_ERROR_GRADE(lRet, CBR_lSignalWrite(dwSignalId, pvNewSignalData, bDataBufferLength, (bTxType & ~CSM_C_TX_BUS_WAKE_TEMP))); //lint !e774 PQM_authorized_127
        #else
        CSM_M_SET_HIGHEST_ERROR_GRADE(lRet, CBR_lSignalWrite(dwSignalId, pvNewSignalData, bDataBufferLength, (bTxType))); //lint !e774 PQM_authorized_127
        #endif  // REMOTE_STACK
      #endif // broadcast abstraction layer

      #if (CSM_S_TRACE == CSM_C_F_ON)
        #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
          #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
          {
            BYTE ubErrorGrade = CSM_M_GET_EG(lRet);
            BYTE ubErrorLoc   = CSM_M_GET_EL(lRet);
            WORD ubErrorCode  = CSM_M_GET_EC(lRet);

            ETG_TRACE_USR4(("####1 CSM_lSignalWrite: Signal:%08x  Typ:%x  Error:%x  ErrLoc:%x  ErrGrade:%x  Len:%x  Data:%02x",
                            (tU32)  dwSignalId,
                            (tU8)   bTxType,
                            ETG_CENUM(tCSM_ERROR_CODE,     (tU16) ubErrorCode),
                            ETG_CENUM(tCSM_ERROR_LOCATION, (tU8)  ubErrorLoc),
                            ETG_CENUM(tCSM_ERROR_GRADE,    (tU8)  ubErrorGrade),
                            (tU8)   bDataBufferLength,
                            ETG_LIST_LEN(bDataBufferLength),
                            ETG_LIST_PTR_T8(pvNewSignalData)
                          ));
          }
          #endif
        #else
        #if (CSM_S_TRACE_LIMITED_SIGNAL == CSM_C_F_ON)
        if( CSM_fTraceCheckTxSignal( dwSignalId) )
        #endif // CSM_S_TRACE_LIMITED_SIGNAL
        {
          // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
          BYTE abData[18]={0};

          abData[0] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(dwSignalId));
          abData[1] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(dwSignalId));
          abData[2] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(dwSignalId));
          abData[3] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(dwSignalId));

          abData[4] = bTxType;
          abData[5] = bDataBufferLength;

          abData[6] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(lRet));
          abData[7] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(lRet));
          abData[8] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(lRet));
          abData[9] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(lRet));

          abData[10] = *(((BYTE *)pvNewSignalData)+0);
          abData[11] = *(((BYTE *)pvNewSignalData)+1);
          abData[12] = *(((BYTE *)pvNewSignalData)+2);
          abData[13] = *(((BYTE *)pvNewSignalData)+3);
          abData[14] = *(((BYTE *)pvNewSignalData)+4);
          abData[15] = *(((BYTE *)pvNewSignalData)+5);
          abData[16] = *(((BYTE *)pvNewSignalData)+6);
          abData[17] = *(((BYTE *)pvNewSignalData)+7);

          CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_SIGNAL_WRITE, abData, 10+bDataBufferLength);
        }
        #endif
      #endif // CSM_S_TRACE
    #endif // CSM_S_CSM_VW
  #else // CSM_S_CBR_AVAILABLE
    // error: without broadcast server no signal processing
    lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_GENERAL_ERROR); //lint !e838 yes, sometime a variable gets a new value
  #endif // CSM_S_CBR_AVAILABLE
  return lRet;
}//lint !e715 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally

/*******************************************************************************
 * function    CSM_lFullFrameSignalWrite
 * \doxydocu
 * \brief      Proxy (CCC) has requested the storage of a full frame due to changed signal data.
 *             Since this can collide with locally written gateway signals for the same frame,
 *             a special handling is required. This function should only be called (directly or indirectly) from the CCS.
 *             Pass all parameters directly to the CBR which knows what to do
 *
 * \history_begin
 * 22.06.16  Andreas Borck
 * - initial version
 *
 * \history_end
 ******************************************************************************/
#if (CSM_S_CBR_GATEWAY == CSM_C_F_ON)
CSM_API LONG CSM_lFullFrameSignalWrite(DWORD dwSignalId, const void * pvNewSignalData, BYTE bDataBufferLength, BYTE bTxType)
{
  LONG lRet = CSM_C_NO_ERROR;

  #if (CSM_S_CBR_AVAILABLE == CSM_C_F_ON)
    // only process request if a CBR is available
    if (( csm_bCsmState != CSM_C_STATE_INITIALIZED) && ( csm_bCsmState != CSM_C_STATE_ACTIVE))
    {
      // error: CSM has not been initialized yet
      CSM_vAssert(FALSE);
      return (CSM_M_MAKE_ERROR( CSM_C_ERR_SEVERE_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_CSM_STATE_FORBIDS_CALL));
    }

    // Full frame access is only intended to be called from CCS (IPC) and will feed us with native signal IDs,
    // so remote stack MUST NOT be active (otherwise we are compiling on the iMX proxy system)
    #if (CSM_S_REMOTE_STACK == CSM_C_F_ON)
      #error "Check your configuration - This function must not be active for this feature switch combination"
    #else
      // simply pass request to dedicated function in CBR
      CSM_M_SET_HIGHEST_ERROR_GRADE(lRet, CBR_lFullFrameSignalWrite(dwSignalId, pvNewSignalData, bDataBufferLength, (bTxType & ~CSM_C_TX_BUS_WAKE_TEMP))); //lint !e774 PQM_authorized_127
    #endif
      #if (CSM_S_TRACE == CSM_C_F_ON)
        #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
          #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
          {
            BYTE ubErrorGrade = CSM_M_GET_EG(lRet);
            BYTE ubErrorLoc   = CSM_M_GET_EL(lRet);
            WORD ubErrorCode  = CSM_M_GET_EC(lRet);

            ETG_TRACE_USR1(("####1 CSM_lFullFrameSignalWrite: Signal:%08x  Typ:%x  Error:%x  ErrLoc:%x  ErrGrade:%x  Len:%x  Data:%02x",
                            (tU32)  dwSignalId,
                            (tU8)   bTxType,
                            ETG_CENUM(tCSM_ERROR_CODE,     (tU16) ubErrorCode),
                            ETG_CENUM(tCSM_ERROR_LOCATION, (tU8)  ubErrorLoc),
                            ETG_CENUM(tCSM_ERROR_GRADE,    (tU8)  ubErrorGrade),
                            (tU8)   bDataBufferLength,
                            ETG_LIST_LEN(bDataBufferLength),
                            ETG_LIST_PTR_T8(pvNewSignalData)
                          ));
          }
          #endif
        #else
        #if (CSM_S_TRACE_LIMITED_SIGNAL == CSM_C_F_ON)
        if( CSM_fTraceCheckTxSignal( dwSignalId) )
        #endif /* CSM_S_TRACE_LIMITED_SIGNAL */
        {
          // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
          BYTE abData[18]={0};

          abData[0] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(dwSignalId));
          abData[1] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(dwSignalId));
          abData[2] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(dwSignalId));
          abData[3] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(dwSignalId));

          abData[4] = bTxType;
          abData[5] = bDataBufferLength;

          abData[6] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(lRet));
          abData[7] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(lRet));
          abData[8] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(lRet));
          abData[9] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(lRet));

          abData[10] = *(((BYTE *)pvNewSignalData)+0);
          abData[11] = *(((BYTE *)pvNewSignalData)+1);
          abData[12] = *(((BYTE *)pvNewSignalData)+2);
          abData[13] = *(((BYTE *)pvNewSignalData)+3);
          abData[14] = *(((BYTE *)pvNewSignalData)+4);
          abData[15] = *(((BYTE *)pvNewSignalData)+5);
          abData[16] = *(((BYTE *)pvNewSignalData)+6);
          abData[17] = *(((BYTE *)pvNewSignalData)+7);

          CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_SIGNAL_WRITE, abData, 10+bDataBufferLength);
        }
        #endif
      #endif // CSM_S_TRACE

  #else // CSM_S_CBR_AVAILABLE
    // error: without broadcast server no signal processing
    lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_GENERAL_ERROR); //lint !e838 yes, sometime a variable gets a new value
  #endif // CSM_S_CBR_AVAILABLE

  return lRet;
}
#endif

/*******************************************************************************
 * function    CSM_lMultipleSignalWrite
 * \doxydocu
 * \brief      Stores multiple signal data and triggers transmission.
 *
 *             Stores multiple signal data in telegram buffer.\n
 *             If transmission type is CSM_C_TX_IMMEDIATELY, telegram has to be
 *             transmitted immediately, otherwise (CSM_C_TX_AS_CONFIGURED) as
 *             configured (for cyclic/event mixed modes cyclic).
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \param[in]  *prSignals (I)
 *             pointer to signal data structure
 *
 * \param[in]  bNumberOfSignals (I)
 *             number of signals
 *
 * \return     Error/no error value
 *
 * \access     application
 * \reentrant  no
 *
 * \lastreview 21.03.11 Borck
 *             13.10.08 VTeam-Funke
 *             24.07.08 VTeam-Zurmhl
 *
 * \history_begin
 * 13.10.08 VTeam-Funke
 * - align return value to same name
 * - reduce return points of function
 *
 * 25.04.08 VTeam-Prhl
 * Initial Revision.
 * \history_end
 ******************************************************************************/
CSM_API LONG CSM_lMultipleSignalWrite(const tCSM_MultipleSignalWrite * prSignalList, const BYTE bNumberOfSignals)
{
  LONG lRet = CSM_C_NO_ERROR;
  #if (CSM_S_CBR_AVAILABLE == CSM_C_F_ON)

    #if (CSM_S_TRACE == CSM_C_F_ON)||(CSM_S_BUS_WAKE_TEMP == CSM_C_F_ON)
    BYTE bCount;
    #endif
    if (( csm_bCsmState != CSM_C_STATE_INITIALIZED) && ( csm_bCsmState != CSM_C_STATE_ACTIVE))
    {
      // error: CSM has not been initialized yet
      CSM_vAssert( FALSE);
      lRet = (CSM_M_MAKE_ERROR( CSM_C_ERR_SEVERE_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_CSM_STATE_FORBIDS_CALL));
    }
    else
    {
      #if (CSM_S_BUS_WAKE_TEMP == CSM_C_F_ON)
      for (bCount = 0; bCount < bNumberOfSignals; bCount++)
      {
        if( (prSignalList->bTxType & CSM_C_TX_BUS_WAKE_TEMP) == CSM_C_TX_BUS_WAKE_TEMP)
        {
          csm_fWakeTemp = TRUE;
          csm_vTriggerCsmDownstreamTask( CSM_C_DOWN_REASON_CSM_BUS_STATE_CHANGED);
          CSM_lTimerStart(CSM_TIMER_WAKE_TEMP, CSM_C_WAKE_TEMP_TIME);
        }
      }
      #endif // CSM_S_BUS_WAKE_TEMP

      #if ((CSM_S_iMX_SIGNAL_ABSTRACTION == CSM_C_F_ON) || (CSM_S_v850_SIGNAL_ABSTRACTION == CSM_C_F_ON))
      CSM_M_SET_HIGHEST_ERROR_GRADE(lRet, CBR_AL_lMultipleSignalWrite(prSignalList, bNumberOfSignals));//lint !e774 PQM_authorized_127
      #else
      // generic handling via broadcast server
      CSM_M_SET_HIGHEST_ERROR_GRADE(lRet, CBR_lMultipleSignalWrite(prSignalList, bNumberOfSignals));//lint !e774 PQM_authorized_127
      #endif

      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      {
        for( bCount = 0; bCount < bNumberOfSignals; bCount++)
        {
          #if (CSM_S_TRACE_LIMITED_SIGNAL == CSM_C_F_ON)
          if (CSM_fTraceCheckTxSignal(prSignalList[bCount].dwSignalId))
          #endif // CSM_S_TRACE_LIMITED_SIGNAL
          {
            #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
            ETG_TRACE_USR4(("CSM_lMultipleSignalWrite - SigId:0x%X  TxType:0x%X  Len:0x%X  Result:0x%X  Data:0x%X 0x%X 0x%X 0x%X 0x%X 0x%X 0x%X 0x%X",
                            (tU32) prSignalList[bCount].dwSignalId,
                            (tU8)  prSignalList[bCount].bTxType,
                            (tU8)  prSignalList[bCount].bDataBufferLength,
                            (tU32) lRet,
                            (tU8) *(((BYTE *)prSignalList[bCount].pvNewSignalData)+0),
                            (tU8) *(((BYTE *)prSignalList[bCount].pvNewSignalData)+1),
                            (tU8) *(((BYTE *)prSignalList[bCount].pvNewSignalData)+2),
                            (tU8) *(((BYTE *)prSignalList[bCount].pvNewSignalData)+3),
                            (tU8) *(((BYTE *)prSignalList[bCount].pvNewSignalData)+4),
                            (tU8) *(((BYTE *)prSignalList[bCount].pvNewSignalData)+5),
                            (tU8) *(((BYTE *)prSignalList[bCount].pvNewSignalData)+6),
                            (tU8) *(((BYTE *)prSignalList[bCount].pvNewSignalData)+7)
                          ));
            #endif
          }
        }
      }
      #else // CSM_S_TRACE_ETG
      for( bCount = 0; bCount < bNumberOfSignals; bCount++)
      {
        #if (CSM_S_TRACE_LIMITED_SIGNAL == CSM_C_F_ON)
        if( CSM_fTraceCheckTxSignal( prSignalList[bCount].dwSignalId) )
        #endif // CSM_S_TRACE_LIMITED_SIGNAL
        {
          // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
          BYTE abData[18]={0};

          abData[0] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD( prSignalList[bCount].dwSignalId));
          abData[1] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD( prSignalList[bCount].dwSignalId));
          abData[2] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD( prSignalList[bCount].dwSignalId));
          abData[3] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD( prSignalList[bCount].dwSignalId));

          abData[4] = prSignalList[bCount].bTxType;
          abData[5] = prSignalList[bCount].bDataBufferLength;

          abData[6] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(lRet));
          abData[7] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(lRet));
          abData[8] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(lRet));
          abData[9] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(lRet));

          abData[10] = *(((BYTE *)prSignalList[bCount].pvNewSignalData)+0);
          abData[11] = *(((BYTE *)prSignalList[bCount].pvNewSignalData)+1);
          abData[12] = *(((BYTE *)prSignalList[bCount].pvNewSignalData)+2);
          abData[13] = *(((BYTE *)prSignalList[bCount].pvNewSignalData)+3);
          abData[14] = *(((BYTE *)prSignalList[bCount].pvNewSignalData)+4);
          abData[15] = *(((BYTE *)prSignalList[bCount].pvNewSignalData)+5);
          abData[16] = *(((BYTE *)prSignalList[bCount].pvNewSignalData)+6);
          abData[17] = *(((BYTE *)prSignalList[bCount].pvNewSignalData)+7);

          CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_MULTIPLE_SIGNAL_WRITE, abData,
                         10+prSignalList[bCount].bDataBufferLength);
        }
      }
      #endif // CSM_S_TRACE_ETG
      #endif // CSM_S_TRACE
    } // CSM is initialized

  #else // CSM_S_CBR_AVAILABLE
    // error: no broadcast server configured
    lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_GENERAL_ERROR); /*lint !e838 yes, sometime a variable get a new value*/
  #endif // CSM_S_CBR_AVAILABLE
  return lRet;
} //lint !e715 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally

/*******************************************************************************
 * function    CSM_lSignalRead
 * \doxydocu
 * \brief      Reads signal data from telegram buffer.
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \param[in]  dwSignalId
 *              Signal identifier (signal name)
 *
 * \param[out] pvActSignalData
 *              Pointer to data of signal data
 *
 * \param[in]  bDataBufferLength
 *              Length of output data buffer in bytes (security element).\n
 *              Best way: sizeof(databuffer).
 *
 * \param[in]  pdwSignalStatus (->O)
 *              Pointer to global signal state (bitfield)
 *                SIGNAL_NOT_RECEIVED  no signal received
 *                SIGNAL_DATA_INVALID  data invalid (for example range error)
 *                SIGNAL_TIMEOUT       timeout (receive)
 *
 * \return     Error/no error value
 *
 * \access     application
 * \reentrant  no
 *
 * \lastreview 21.03.11 Borck
 *             13.10.08 VTeam-Funke
 *             09.11.06 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 13.10.08 VTeam-Funke
 * - align return value to same name
 * - reduce return points of function
 *
 * 17.10.06 CM-DI/ESA3-Khler
 * - PVOID replaced by void *.
 *
 *
 * 21.08.06 CM-DI/ESA3-Khler
 * - Trace handling changed.
 *
 * 16.10.03 CM-DI/ESU3-Zurmuehl
 * CSM_API modifier added to function definition
 *
 * 27.02.02 CM-DI/EHS3-Pistoor
 * Initial Revision.
 * \history_end
 ******************************************************************************/
CSM_API LONG CSM_lSignalRead(DWORD  dwSignalId, void * pvActSignalData, BYTE bDataBufferLength, DWORD * pdwSignalStatus)
{
  LONG lRet = CSM_C_NO_ERROR;
  #if (CSM_S_CBR_AVAILABLE == CSM_C_F_ON)
  if (( csm_bCsmState != CSM_C_STATE_INITIALIZED) && (csm_bCsmState != CSM_C_STATE_ACTIVE))
  {
    // error: CSM has not been initialized yet
    CSM_vAssert(FALSE);
    lRet = (CSM_M_MAKE_ERROR( CSM_C_ERR_SEVERE_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_CSM_STATE_FORBIDS_CALL));
  }
  else
  {
    #if ((CSM_S_iMX_SIGNAL_ABSTRACTION == CSM_C_F_ON) || (CSM_S_v850_SIGNAL_ABSTRACTION == CSM_C_F_ON))
    // generic handling via broadcast abstraction layer
    CSM_M_SET_HIGHEST_ERROR_GRADE(lRet, CBR_AL_lSignalRead(dwSignalId, pvActSignalData, bDataBufferLength, pdwSignalStatus)); //lint !e774 PQM_authorized_127
    #else
    // generic handling via broadcast server
    CSM_M_SET_HIGHEST_ERROR_GRADE(lRet, CBR_lSignalRead(dwSignalId, pvActSignalData, bDataBufferLength, pdwSignalStatus)); //lint !e774 PQM_authorized_127
    #endif

    #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
        #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
          #if (CSM_S_LITTLE_ENDIAN_CPU == CSM_C_F_ON)
          ETG_TRACE_USR4(("####1 CSM_lSignalRead: Signal:%08x  Status:%08x  Return:%08x  Len:%x  Data(LittleEndian[Lo/Hi]):%02x",
                         (tU32)  dwSignalId,
                         (tU32)  *pdwSignalStatus,
                         (tU32)  lRet,
                         (tU8)   bDataBufferLength,
                         ETG_LIST_LEN(bDataBufferLength),
                         ETG_LIST_PTR_T8(pvActSignalData)
                        ));
          #else
          ETG_TRACE_USR1(("####1 CSM_lSignalRead: Signal:%08x  Status:%08x  Return:%08x  Len:%x  Data:%02x",
                         (tU32)  dwSignalId,
                         (tU32)  *pdwSignalStatus,
                         (tU32)  lRet,
                         (tU8)   bDataBufferLength,
                         ETG_LIST_LEN(bDataBufferLength),
                         ETG_LIST_PTR_T8(pvActSignalData)
                        ));
          #endif
        #endif
      #else
      #if (CSM_S_TRACE_LIMITED_SIGNAL == CSM_C_F_ON)
      if( CSM_fTraceCheckSignal( dwSignalId) )
      #endif // CSM_S_TRACE_LIMITED_SIGNAL
      {
        // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
        BYTE abData[21]={0};

        abData[0] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(dwSignalId));
        abData[1] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(dwSignalId));
        abData[2] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(dwSignalId));
        abData[3] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(dwSignalId));

        abData[4] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(*pdwSignalStatus));
        abData[5] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(*pdwSignalStatus));
        abData[6] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(*pdwSignalStatus));
        abData[7] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(*pdwSignalStatus));

        abData[8] = bDataBufferLength;

        abData[9] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(lRet));
        abData[10] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(lRet));
        abData[11] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(lRet));
        abData[12] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(lRet));

        abData[13] = *(((BYTE *)pvActSignalData)+0);
        abData[14] = *(((BYTE *)pvActSignalData)+1);
        abData[15] = *(((BYTE *)pvActSignalData)+2);
        abData[16] = *(((BYTE *)pvActSignalData)+3);
        abData[17] = *(((BYTE *)pvActSignalData)+4);
        abData[18] = *(((BYTE *)pvActSignalData)+5);
        abData[19] = *(((BYTE *)pvActSignalData)+6);
        abData[20] = *(((BYTE *)pvActSignalData)+7);

        CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_SIGNAL_READ, abData, 13+bDataBufferLength);
      }
      #endif
    #endif // CSM_S_TRACE
  }
  #else // CSM_S_CBR_AVAILABLE
  // error: no broadcast server configured                                    */
  lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_GENERAL_ERROR); /*lint !e838 yes, sometime a variable get a new value*/
  #endif // CSM_S_CBR_AVAILABLE
  return lRet;
}/*lint !e715 !e818 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally , PQM_authorized_multi_123. Reason: Interface not defined as const*/

/*******************************************************************************
 * function    CSM_lMultipleSignalRead
 * \doxydocu
 * \brief      Reads multiple signal data from telegram buffer.
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \param[in]  *prSignals (I)
 *             pointer to signal data structure
 *
 * \param[in]  bNumberOfSignals (I)
 *             number of signals
 *
 * \return     Error/no error value
 *
 * \access     application
 * \reentrant  no
 *
 * \lastreview 21.03.11 Borck
 *             13.10.08 VTeam-Funke
 *             24.07.08 VTeam-Zurmhl
 *
 * \history_begin
 * 13.10.08 VTeam-Funke
 * - align return value to same name
 * - reduce return points of function
 *
 * 29.04.08 VTeam-Prhl
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if ((CSM_S_iMX_SIGNAL_ABSTRACTION == CSM_C_F_ON) || (CSM_S_v850_SIGNAL_ABSTRACTION == CSM_C_F_ON))
#else
#endif
CSM_API LONG CSM_lMultipleSignalRead(tCSM_MultipleSignalRead * prSignalList, const BYTE  bNumberOfSignals)
{
  LONG lRet = CSM_C_NO_ERROR;

  #if (CSM_S_CBR_AVAILABLE == CSM_C_F_ON)
  #if (CSM_S_TRACE == CSM_C_F_ON)
  BYTE bCount;
  #endif /* CSM_S_TRACE */

  if ((csm_bCsmState != CSM_C_STATE_INITIALIZED) && (csm_bCsmState != CSM_C_STATE_ACTIVE))
  {
    /* error: CSM has not been initialized yet                                */
    CSM_vAssert(FALSE);
    lRet = (CSM_M_MAKE_ERROR( CSM_C_ERR_SEVERE_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_CSM_STATE_FORBIDS_CALL));
  }
  else
  {
    /* generic handling via broadcast server                                  */
#if ((CSM_S_iMX_SIGNAL_ABSTRACTION == CSM_C_F_ON) || (CSM_S_v850_SIGNAL_ABSTRACTION == CSM_C_F_ON))
    CSM_M_SET_HIGHEST_ERROR_GRADE(lRet, CBR_AL_lMultipleSignalRead( prSignalList, bNumberOfSignals));//lint !e774 PQM_authorized_127
#else
    CSM_M_SET_HIGHEST_ERROR_GRADE(lRet, CBR_lMultipleSignalRead( prSignalList, bNumberOfSignals));//lint !e774 PQM_authorized_127
#endif

    #if (CSM_S_TRACE == CSM_C_F_ON)
    for( bCount = 0; bCount < bNumberOfSignals; bCount++)
    {
      #if (CSM_S_TRACE_LIMITED_SIGNAL == CSM_C_F_ON)
      if( CSM_fTraceCheckSignal( prSignalList[bCount].dwSignalId) )
      #endif // CSM_S_TRACE_LIMITED_SIGNAL
      {
        #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
          #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
          ETG_TRACE_USR4(("####1 CSM_lMultipleSignalRead: Signal:%08x  Status:%08x  Return:%08x  Len:%x  Data:%02x",
                          (tU32)  prSignalList[bCount].dwSignalId,
                          (tU32)  *prSignalList[bCount].pdwSignalStatus,
                          (tU32)  lRet,
                          (tU8)   prSignalList[bCount].bDataBufferLength,
                          ETG_LIST_LEN(prSignalList[bCount].bDataBufferLength),
                          ETG_LIST_PTR_T8(prSignalList[bCount].pvActSignalData)
                        ));
          #endif
        #else
        // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
        BYTE abData[21]={0};

        abData[0] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(prSignalList[bCount].dwSignalId));
        abData[1] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(prSignalList[bCount].dwSignalId));
        abData[2] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(prSignalList[bCount].dwSignalId));
        abData[3] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(prSignalList[bCount].dwSignalId));

        abData[4] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(*prSignalList[bCount].pdwSignalStatus));
        abData[5] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(*prSignalList[bCount].pdwSignalStatus));
        abData[6] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(*prSignalList[bCount].pdwSignalStatus));
        abData[7] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(*prSignalList[bCount].pdwSignalStatus));

        abData[8] = prSignalList[bCount].bDataBufferLength;

        abData[9] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(lRet));
        abData[10] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(lRet));
        abData[11] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(lRet));
        abData[12] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(lRet));

        abData[13] = *(((BYTE *)prSignalList[bCount].pvActSignalData)+0);
        abData[14] = *(((BYTE *)prSignalList[bCount].pvActSignalData)+1);
        abData[15] = *(((BYTE *)prSignalList[bCount].pvActSignalData)+2);
        abData[16] = *(((BYTE *)prSignalList[bCount].pvActSignalData)+3);
        abData[17] = *(((BYTE *)prSignalList[bCount].pvActSignalData)+4);
        abData[18] = *(((BYTE *)prSignalList[bCount].pvActSignalData)+5);
        abData[19] = *(((BYTE *)prSignalList[bCount].pvActSignalData)+6);
        abData[20] = *(((BYTE *)prSignalList[bCount].pvActSignalData)+7);

        CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_MULTIPLE_SIGNAL_READ, abData, 13+prSignalList[bCount].bDataBufferLength);
        #endif // CSM_S_TRACE_ETG
      }
    }
    #endif // CSM_S_TRACE
  }
  #else // CSM_S_CBR_AVAILABLE
    lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_GENERAL_ERROR); /*lint !e838 yes, sometime a variable get a new value*/
  #endif // CSM_S_CBR_AVAILABLE
  return lRet;
}/*lint !e715 !e818 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally , PQM_authorized_multi_123. Reason: Interface not defined as const*/


/*******************************************************************************
 * function    CSM_vWakeupIrqEnable
 * \doxydocu
 * \brief      Enables wakeup interrupt.
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \return     void
 *
 * \access     application
 * \reentrant  no
 *
 * \lastreview 29.06.06 CM-DI/ESA3-Battis
 *
 * \history_begin
 * 28.06.06 CM-DI/ESA3-Haffner
 * changes for new proxy
 *
 * 27.06.2006 CM-DI/ESA3-Zurmhl
 * adaptation for third CAN bus
 *
 * 15.01.03 CM-DI/EHS3-K"ohler
 * Clear wakeup reason added.
 *
 * 22.03.01 K7/EFG13-Pistoor
 * Initial Revision.
 * \history_end
 ******************************************************************************/
void CSM_vWakeupIrqEnable(void)
{
#if (CSM_S_CDA_AVAILABLE == CSM_C_F_ON)
  #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
    #if (CSM_S_CAN_WAKEUP_FUNCTIONS_VIA_CSM == CSM_C_F_ON)
      CSM_M_CLEAR_CAN_WAKEUP_REASON(CSM_C_CAN_1);
    #endif
    CDA_vWakeupIrqEnable(CSM_C_CAN_1);

    #if CSM_C_MAX_CAN_BUS > 1
      #if (CSM_S_CAN_WAKEUP_FUNCTIONS_VIA_CSM == CSM_C_F_ON)
        CSM_M_CLEAR_CAN_WAKEUP_REASON(CSM_C_CAN_2);
      #endif
      CDA_vWakeupIrqEnable(CSM_C_CAN_2);

      #if CSM_C_MAX_CAN_BUS > 2
        #if (CSM_S_CAN_WAKEUP_FUNCTIONS_VIA_CSM == CSM_C_F_ON)
          CSM_M_CLEAR_CAN_WAKEUP_REASON(CSM_C_CAN_3);
        #endif
        CDA_vWakeupIrqEnable(CSM_C_CAN_3);

        #if (CSM_C_MAX_CAN_BUS > 3)
          #error Insert here functionality for next CAN bus
        #endif /* #if (CSM_C_MAX_CAN_BUS > 3) */
      #endif /* #if (CSM_C_MAX_CAN_BUS > 2) */
    #endif /* #if (CSM_C_MAX_CAN_BUS > 1) */
  #endif /* CSM_S_REMOTE_STACK */
#endif /*CSM_S_CDA_AVAILABLE == CSM_C_F_ON*/

  return;
}


/*******************************************************************************
 * function    CSM_vWakeupIrqDisable
 * \doxydocu
 * \brief      Disables wakeup interrupt.
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \return     void
 *
 * \access     application
 * \reentrant  no
 *
 * \lastreview 29.06.06 CM-DI/ESA3-Battis
 *
 * \history_begin
 * 28.06.06 CM-DI/ESA3-Haffner
 * changes for new proxy
 *
 * 27.06.2006 CM-DI/ESA3-Zurmhl
 * adaptation for third CAN bus
 *
 * 22.03.01 K7/EFG13-Pistoor
 * Initial Revision.
 * \history_end
 ******************************************************************************/
void CSM_vWakeupIrqDisable(void)
{
#if (CSM_S_CDA_AVAILABLE == CSM_C_F_ON)
  #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
    CDA_vWakeupIrqDisable(CSM_C_CAN_1);
    #if CSM_C_MAX_CAN_BUS > 1
      CDA_vWakeupIrqDisable(CSM_C_CAN_2);
      #if CSM_C_MAX_CAN_BUS > 2
        CDA_vWakeupIrqDisable(CSM_C_CAN_3);
        #if (CSM_C_MAX_CAN_BUS > 3)
          #error Insert here functionality for next CAN bus
        #endif /* #if (CSM_C_MAX_CAN_BUS > 3) */
      #endif /* #if (CSM_C_MAX_CAN_BUS > 2) */
    #endif /* #if (CSM_C_MAX_CAN_BUS > 1) */
  #endif /* CSM_S_REMOTE_STACK */
#endif /*CSM_S_CDA_AVAILABLE == CSM_C_F_ON*/

  return;
}


#if (CSM_S_CBR_AVAILABLE == CSM_C_F_ON)
#if (CSM_S_CCS_USED == CSM_C_F_ON)

/*******************************************************************************
 * function    CSM_lGetAllStateIndicationsAgain
 * \doxydocu
 * \brief      Requests a signalisation of all states
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \return     Error value
 *
 * \access     application
 * \reentrant  no
 *
 * \lastreview 28.06.2006 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 27.06.2006 CM-DI/ESA3-Zurmhl
 * adaptation for third CAN bus
 *
 * 04.11.04 CM-DI/ESA3-Battis
 * Cleaned for new CBR.
 * \history_end
 ******************************************************************************/
CSM_API LONG CSM_lGetAllStateIndicationsAgain(VOID)
{
#if (CSM_S_CCS_USED == CSM_C_F_ON)
  BYTE bBus;
#endif /* CSM_S_CCS_USED */

#if (CSM_S_CBR_AVAILABLE == CSM_C_F_ON)
#if (CSM_S_CBR_QUEUED_SIGNALS_USED == CSM_C_F_ON)
  CSM_vCCS_ClearQSigInd();         /* clear pending indications from queue. */
#endif
#endif /* CSM_S_CBR_AVAILABLE == CSM_C_F_ON */

#if (CSM_S_CCS_USED == CSM_C_F_ON)
  for( bBus=0; bBus < CSM_C_MAX_CAN_BUS; bBus++)
  {
    #if (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
    CCS_NET_vCSM_BusStateInd( bBus, csm_abCsmSubState[ bBus], abVN[bBus]);
    #else
    CCS_NET_vCSM_BusStateInd( bBus, csm_abCsmSubState[ bBus]);
    #endif /* CSM_S_VIRTUAL_NETWORK */
  }
#endif /* CSM_S_CCS_USED */

//  CNM_vSendAllIncicationsAgain();     /* force send all communication indications immediately */
#if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
#if  (CSM_S_OSEKI_NM == CSM_C_F_ON)
  CNM_OSEKI_vSendAllIndicationsAgain();/* force send all communication indications immediately */
#endif /*  (CSM_S_OSEKI_NM == CSM_C_F_ON)  */
#endif /*  (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)  */
#if  (CSM_S_CSM_LOCAL_CTRL == CSM_C_F_ON)
  CSM_LCTR_vSendAllIndicationsAgain();/* force send all communication indications immediately */
#endif /*  (CSM_S_CSM_LOCAL_CTRL == CSM_C_F_ON)  */
#if (CSM_S_ISO_TP == CSM_C_F_ON)
#if (CSM_S_ITP_COMMUNICATION_FCT == CSM_C_F_ON)
  CSM_ITP_vSendAllIndicationsAgain(); /* force send all communication indications immediately */
#endif
#endif /* CSM_S_ISO_TP */
#if (CSM_S_CBR_AVAILABLE == CSM_C_F_ON)
  CBR_vSendAllSignalsAgain();         /* force send all signals immediately */
#if (CSM_S_CBR_USES_COMMUNICATION_REQ == CSM_C_F_ON)
  CSM_CBR_vSendAllIndicationsAgain(); /* force send all communication indications immediately */
#endif
#endif /* CSM_S_CBR_AVAILABLE */

  #if (CSM_S_ENABLE_ALWAYS_INDICATE_STATE_CHANGE == CSM_C_F_ON)
     CSM_vSYS_BusStateInd(CSM_C_CAN_1,csm_abCsmSubState[CSM_C_CAN_1]);
     #if (CSM_C_MAX_CAN_BUS > 1)
       CSM_vSYS_BusStateInd(CSM_C_CAN_2,csm_abCsmSubState[CSM_C_CAN_2]);
     #endif /* (CSM_C_MAX_CAN_BUS > 1) */
     #if (CSM_C_MAX_CAN_BUS > 2)
       CSM_vSYS_BusStateInd(CSM_C_CAN_3,csm_abCsmSubState[CSM_C_CAN_3]);
     #endif /* (CSM_C_MAX_CAN_BUS > 2) */
     #if (CSM_C_MAX_CAN_BUS > 3)
       #error add state indications for next CAN bus
     #endif /* (CSM_C_MAX_CAN_BUS > 3) */

  #endif /* if (CSM_S_ENABLE_ALWAYS_INDICATE_STATE_CHANGE == CSM_C_F_ON) */
  return CSM_C_NO_ERROR;
} /* end CSM_lGetAllStateIndicationsAgain() */
#endif /* CSM_S_CCS_USED == CSM_C_F_ON */
#endif /* CSM_S_CBR_AVAILABLE */

#if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
#if (CSM_S_LAST_NM_MSG == CSM_C_F_ON)
/*******************************************************************************
 * function    CSM_vCNMLastNmMsg
 * \doxydocu
 * \brief      NM indicates that the last message was send. Bus state
 *             changes from shutdown to listen.
 *             Function can be used for OSEK or IVLAN NM.
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \param[in]  bBus
 *              Bus number
 *
 * \return     void
 *
 * \access     CAN task
 * \reentrant  no
 *
 * \lastreview 24.07.08 VTeam-Zurmhl .. Finding: Incomplete and probably obsolete
 *                      function. OSEK NM currently uses different strategy.
 *
 * \history_begin
 * 28.09.05  CM-DI/ESA3-K"ohler
 * - Warning commented out
 *
 * 14.01.05  CM-DI/ESA3-K"ohler
 * CDA_vClearAllTxObjectsIfTxQueueIsFull(bBus) call removed and
 * CDA_lCanRxTxErrInterruptDisableAndSave/Restore replaced by
 * CDA_iCanInterruptsDisable/Restore.
 *
 * 15.07.03 CM-DI/ESU3-K"ohler
 * CSM_vCNPLastNmMsg renamed CSM_vCNMLastNmMsg.
 * CDA_vClearAllTxObjectsIfTxQueueIsFull() added.
 *
 * 12.03.03 CM-DI/ESU3-K"ohler
 * Initial Revision.
 * \history_end
 ******************************************************************************/
VOID CSM_vCNMLastNmMsg(BYTE bBus)
{
INT iInterruptState;

  iInterruptState = CDA_iCanInterruptsDisable(bBus);
  {
    /* Was hier eigentlich gemacht werden soll:
     * pruefen, ob einige Nachrichten nicht verschickt werden konnten
     * (z.B. Bus abgeklemmt) und in diesem Fall Sendeauftraege loeschen, damit
     * sie nicht noch spaeter vor dem Abschalten des CAN automatisch vom
     * Tx-Interrupt rausgeschickt werden. Indikator hier: queue ist voll. */
    /*CDA_vClearAllTxObjectsIfTxQueueIsFull(bBus);*/
    /* #warning "Transmit queue loechen" */
  }
  CDA_vCanInterruptsRestore(bBus, iInterruptState);

  if (CSM_C_STATE_BUS_SHUTDOWN == csm_abCsmSubState[bBus])
  {
    #if (CSM_S_STATE_MASCHINE == CSM_C_F_ON)
      CSM_lBusStateChange(bBus, CSM_C_STATE_BUS_LISTEN);
    #endif /* CSM_S_STATE_MASCHINE */
  }
}
#endif /* CSM_S_LAST_NM_MSG */
#endif /* CSM_S_CNM_AVAILABLE */


#if (CSM_S_RX_OVERRUN_INDICATION == CSM_C_F_ON)
/*******************************************************************************
 * function    CSM_vRxOverrunOccured
 * \doxydocu
 * \brief      Any CSM module indicates that an Rx overrun has occured
 *             (CDA: Rx register or Rx queue, CSM mainloop queue, ...)
 *             This function can be used for example to indicate NM,
 *             diagnosis or for debug purposes.
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \param[in]  bBus
 *              Bus number
 *
 * \return     void
 *
 * \access     interrupt
 * \reentrant  no
 *
 * \lastreview 24.07.08 VTeam-Zurmhl .. Finding: Probably obsolete fct.
 *
 * \history_begin
 * 17.06.03 CM-DI/ESU3-Pistoor
 * Added functionality for global usage
 *
 * 19.03.03 CM-DI/ESU3-K"ohler
 * Initial Revision.
 * \history_end
 ******************************************************************************/
VOID CSM_vRxOverrunOccured(BYTE bBus)
{
    DWORD dwLoop;
    for(dwLoop = 0; dwLoop < CSM_M_ELEMENTS(csm_pvOverrunCallback); dwLoop++)
    {
      if(csm_pvOverrunCallback[dwLoop].pvFunctionCall != NULL)
      {
        csm_pvOverrunCallback[dwLoop].pvFunctionCall(bBus);
      }
    }
}
#endif /* CSM_S_RX_OVERRUN_INDICATION */


#ifdef CSM_C_DOWN_REASON_CDA_BUSOFF_RECOVERY
/*******************************************************************************
 * function    CSM_vCNMBusoffReinit
 * \doxydocu
 * \brief      reinit for any network management
 *             called after busoff
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \param[in]  bBus
 *              Bus number
 *
 * \return     void
 *
 * \access     interrupt
 * \reentrant  no
 *
 * \lastreview 24.07.08 VTeam-Zurmhl
 *
 * \history_begin
 * 30.08.04 CM-DI/ESA3-Battis
 * retval removed because function is void
 *
 * 03.04.03 CM-DI/EHS3-Pistoor
 * Initial Revision.
 * \history_end
 ******************************************************************************/
void CSM_vCNMBusoffReinit(BYTE bBus)
{
  csm_bBusOffState[ bBus].fRecoveryIndication = TRUE;
  csm_bBusOffState[ bBus].fState = FALSE;

  csm_vTriggerCsmDownstreamTask( CSM_C_DOWN_REASON_CDA_BUSOFF_RECOVERY);

  return;
}


/*******************************************************************************
 * function    CSM_vCNMBusoffReinitTimerExpired
 * \doxydocu
 * \brief      reinit timer timeout for any network management called after
 *             busoff on CAN
 *
 * \param[in]  bBus
 *              Bus number
 *
 * \return     void
 *
 * \access     interrupt
 * \reentrant  no
 *
 * \lastreview 14.06.06 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 24.09.04 CM-DI/ESA3-Battis
 * new implementation
 *
 * 30.08.04 CM-DI/ESA3-Battis
 * retval removed because function is void
 *
 * 03.04.03 CM-DI/EHS3-Pistoor
 * Initial Revision.
 * \history_end
 ******************************************************************************/
VOID CSM_vCNMBusoffReinitTimerExpired( DWORD dwTimerValue)
{
#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON) || (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
  BYTE bBus = (BYTE) (dwTimerValue & 0xff);
#endif
  #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
    #if (CSM_S_MPDT == CSM_C_F_ON)
    CNP_MPDT_vReinitTimerExpired( bBus);
    #endif /* CSM_S_MPDT */
    #if (CSM_S_J1939 == CSM_C_F_ON)
    CNP_J1939_vReinitTimerExpired(bBus);
    #endif
  #endif /* CSM_S_CNP_AVAILABLE */

  #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
    #if (CSM_S_OSEK_NM == CSM_C_F_ON)
    CNM_OSEK_vReinitTimerExpired( bBus);
    #endif /* CSM_S_OSEK_NM */

    #if (CSM_S_OSEKI_NM == CSM_C_F_ON)
    CNM_OSEKI_vReinitTimerExpired( bBus);
    #endif /* CSM_S_OSEKI_NM */

    #if (CSM_S_VWHIGH_NM == CSM_C_F_ON)
    CNM_VWHIGH_vReinitTimerExpired( bBus);
    #endif /* CSM_S_VWHIGH_NM */

    #if (CSM_S_GMLAN_NM == CSM_C_F_ON)
    CNM_GMLAN_vReinitTimerExpired( bBus);
    #endif /* CSM_S_GMLAN_NM */

    #if (CSM_S_FIATSLAVE_NM == CSM_C_F_ON)
    CNM_vFiatSlave_ReinitTimerExpired( bBus);
    #endif

    #if (CSM_S_IVLAN_NM == CSM_C_F_ON)
    CNM_IVLAN_vReinitTimerExpired( bBus);
    #endif

    #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
      #if (CSM_S_MCNET == CSM_C_F_ON)
        #if (CSM_S_MCNET_MASTER_NWM == CSM_C_F_ON) || (CSM_S_MCNET_SLAVE_NWM == CSM_C_F_ON)
        CNM_MCNET_vReinitTimerExpired( bBus);
        #endif
      #endif
    #endif
  #endif /* CSM_S_CNM_AVAILABLE */
}/*lint !e715 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally*/

/*******************************************************************************
 * function    CSM_vCNMBusoffReinitTimer1Expired
 * \doxydocu
 * \brief      reinit timer timeout for any network management called after
 *             busoff on CAN1
 *
 * \return     void
 *
 * \access     interrupt
 * \reentrant  no
 *
 * \lastreview 28.06.2006 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 03.04.03 CM-DI/EHS3-Pistoor
 * Initial Revision.
 * \history_end
 ******************************************************************************/
VOID CSM_vCNMBusoffReinitTimer1Expired(VOID)
{
   CSM_vCNMBusoffReinitTimerExpired(CSM_C_CAN_1);
   return;
}


#if (CSM_C_MAX_CAN_BUS > 1)
/*******************************************************************************
 * function    CSM_vCNMBusoffReinitTimer2Expired
 * \doxydocu
 * \brief      reinit timer timeout for any network management called after
 *             busoff on CAN2
 *
 * \return     void
 *
 * \access     interrupt
 * \reentrant  no
 *
 * \lastreview 28.06.2006 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 03.04.03 CM-DI/EHS3-Pistoor
 * Initial Revision.
 * \history_end
 ******************************************************************************/
VOID CSM_vCNMBusoffReinitTimer2Expired(VOID)
{
  CSM_vCNMBusoffReinitTimerExpired(CSM_C_CAN_2);
  return;
}
#endif /* #if (CSM_C_MAX_CAN_BUS > 1) */


#if (CSM_C_MAX_CAN_BUS > 2)
/*******************************************************************************
 * function    CSM_vCNMBusoffReinitTimer3Expired
 * \doxydocu
 * \brief      reinit timer timeout for any network management called after
 *             busoff on CAN3
 *
 * \return     void
 *
 * \access     interrupt
 * \reentrant  no
 *
 * \lastreview 28.06.2006 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 27.06.06 CM-DI/ESA3-Zurmhl
 * Initial Revision.(added for support of third CAN)
 * \history_end
 ******************************************************************************/
VOID CSM_vCNMBusoffReinitTimer3Expired(VOID)
{
  CSM_vCNMBusoffReinitTimerExpired(CSM_C_CAN_3);
  return;
}
#endif /* #if (CSM_C_MAX_CAN_BUS > 2) */

#if (CSM_C_MAX_CAN_BUS > 3)
  #error define function CSM_vCNMBusoffReinitTimer4Expired()
#endif
#endif /* #ifdef CSM_C_DOWN_REASON_CDA_BUSOFF_RECOVERY */

#if (defined CSM_C_IND_REASON_BUS_STATE_CHANGE) && (defined CSM_C_DOWN_REASON_BUS_STATE_CHANGE)
  #error "Only one way is allowed: BusStateChange either via downstream or via upstream/indication task "
#endif

/*******************************************************************************
 * function    csm_vWakeTempTimeout
 * \doxydocu
 * \brief      Wake temp timeout callback.
 *
 *             This funktion is call after the wake temp timeout is expired.
 *             Only the downstream-task is triggered is this function.
 *
 * \param[in]  dwDummy
 *             not used
 *
 * \return     VOID
 *
 * \access     interrupt
 * \reentrant  yes
 *
 * \lastreview 24.07.08  VTeam-Zurmhl
 *
 * \history_begin
 * 12.02.05 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
#if (CSM_S_BUS_WAKE_TEMP == CSM_C_F_ON)
VOID csm_vWakeTempTimeout(DWORD dwDummy)
{
  csm_fWakeTemp = FALSE; /* clear wake temp flag */
  csm_vTriggerCsmDownstreamTaskFromIrq( CSM_C_DOWN_REASON_CSM_BUS_STATE_CHANGED);
}/*lint !e715 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally*/
#endif /* (CSM_S_BUS_WAKE_TEMP == CSM_C_F_ON) */


/*******************************************************************************
 * function    csm_vWakeMinTimeout
 * \doxydocu
 * \brief      Wake min timeout callback.
 *
 *             This funktion is call after the wake min timeout is expired.
 *             Only the downstream-task is triggered is this function.
 *
 * \param[in]  dwDummy
 *             not used
 *
 * \return     VOID
 *
 * \access     interrupt
 * \reentrant  yes
 *
 * \lastreview 21.03.11 Borck
 *
 * \history_begin
 * 23.02.07 VTeam-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
#if (CSM_S_BUS_WAKE_MIN == CSM_C_F_ON)
VOID csm_vWakeMinTimeout(DWORD dwDummy)
{
  BYTE bBus = (BYTE)dwDummy;

  csm_afWakeMin[ bBus] = FALSE;                        /* clear wake min flag */
  csm_vTriggerCsmDownstreamTaskFromIrq(CSM_C_DOWN_REASON_CSM_BUS_STATE_CHANGED);
}
#endif /* (CSM_S_BUS_WAKE_MIN == CSM_C_F_ON) */


/*******************************************************************************
 * function    CSM_vIndicateStaticError(BYTE bBus, LONG lError)
 * \doxydocu
 * \brief      Set static error state.
 *
 * \param      BYTE BusNumber, LONG lError
 *
 * \return     void
 *
 * \access     *
 * \reentrant  yes
 *
 * \lastreview  16.04.08 VTeam-Prhl
 *
 * \history_begin
 * 21.02.08 RBIN/EDI  Basavaraj Patil
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_ENABLE_INDICATE_STATIC_ERROR == CSM_C_F_ON)
void CSM_vIndicateStaticError(BYTE bBus, LONG lError)
{
  INT iIsrStat = CSM_iInterruptsDisable();

  if (bBus < CSM_C_MAX_CAN_BUS)
  {
    csm_lActStaticError[ bBus] = lError;
  }
  else
  {
    /* Bus no. does not fit to CSM_C_MAX_CAN_BUS !! */
    CSM_vAssert(0);
  }

  CSM_vInterruptsRestore(iIsrStat);

}
#endif // CSM_S_ENABLE_INDICATE_STATIC_ERROR

/*******************************************************************************
 * function    CSM_vSetEmTraceError(WORD wErrorId)
 * \doxydocu
 * \brief      Set error for error memory (em trace).
 *
 * \param      WORD error
 *
 * \return     void
 *
 * \access     CSM
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 * 07.06.11 Borck
 * Initial Revision.
 * \history_end
 ******************************************************************************/
void CSM_vSetEmTraceError(DWORD dwErrorId)
{
  dwCSM_EmTraceError |= dwErrorId;
}

/*******************************************************************************
 * function    CSM_dwGetEmTraceError(void)
 * \doxydocu
 * \brief      Retrieve error entered for error memory (em trace).
 *
 * \param      void
 *
 * \return     DWORD error
 *
 * \access     CSM
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 * 07.06.11 Borck
 * Initial Revision.
 * \history_end
 ******************************************************************************/
DWORD CSM_dwGetEmTraceError(void)
{
  return dwCSM_EmTraceError;
}

/*******************************************************************************
 * function    CSM_vIndicateBusError(BYTE bBus, BYTE bError, BYTE bState)
 * \doxydocu
 * \brief      Set bus off state.
 *
 * \param      BYTE BusNumber, BYTE bError, BYTE BusState
 *
 * \return     void
 *
 * \access     *
 * \reentrant  yes
 *
 * \lastreview  17.04.08  VTeam-Prhl
 *
 * \history_begin
 * 16.04.08 RBIN/EDI  Basavaraj Patil
 * - Moved the call of CSM_iInterruptsDisable after the if statement.
 *
 * 29.06.05 CM-CR/ESD  Thomas Mehring / Peter Tuschik
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
void CSM_vIndicateBusError(BYTE bBus, BYTE bError, BYTE bState)
{
  if (bBus < CSM_C_MAX_CAN_BUS)
  {
    INT iIsrStat = CSM_iInterruptsDisable();

    if (bState == TRUE)      /* and set or clear error. */
    {
       csm_abActBusError[ bBus] |= bError;
    }
    else
    {
       csm_abActBusError[ bBus] &= ~bError;
    }

    CSM_vInterruptsRestore(iIsrStat);

    /* Activate upstream task. */
    csm_afTriggerErrorInd[ bBus] = TRUE;// set trigger flag for this bus
    csm_vTriggerCsmUpstreamTask(CSM_C_IND_REASON_BUS_ERROR);
  }
  else
  {
    /* Bus no. does not fit to CSM_C_MAX_CAN_BUS !! */
    CSM_vAssert(0);
  }
}
#endif /* (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON) */

/*******************************************************************************
 * function    CSM_bGetBusError(BYTE bBus)
 * \doxydocu
 * \brief      Get current bus state.
 *
 * \param      BYTE BusNumber
 *
 * \return     Current bus state
 *
 * \access     *
 * \reentrant  yes
 *
 * \lastreview 21.03.11  Borck
 *
 * \history_begin
 * 29.04.08 VTeam-Funke
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
BYTE CSM_bGetBusError(BYTE bBus)
{
  BYTE bReturn = 0;

  if (bBus < CSM_C_MAX_CAN_BUS)
  {
    INT iIsrStat = CSM_iInterruptsDisable();

    bReturn = csm_abActBusError[ bBus];

    CSM_vInterruptsRestore(iIsrStat);
  }
  else
  {
    /* error: invalid BusNr.                                                  */
    CSM_vAssert(0);
  }
  return bReturn;
}
#endif // CSM_S_ENABLE_INDICATE_BUSERROR

/*******************************************************************************
 * function    CSM_vProcessErrorInd( void)
 * \doxydocu
 * \brief      Process error indication
 *
 * \return     void
 *
 * \access     *
 * \reentrant  no, but only called from upstream task
 *
 * \lastreview  05.07.07 CM-DI/ERM13-Battis
 *
 * \history_begin
 * 26.06.07 VTeam-Prhl
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
void CSM_vProcessErrorInd(void)
{
  BYTE  bBus;
  DWORD dwCount;

  for (bBus=0; bBus < CSM_C_MAX_CAN_BUS; bBus++)
  {
    if (csm_afTriggerErrorInd[bBus] == TRUE) // check trigger flag for this bus
    {
      csm_afTriggerErrorInd[bBus] = FALSE; // clear trigger flag for this bus

      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      {
        #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
        ETG_TRACE_USR2(("CSM_vProcessErrorInd - Bus:0x%X  BusError:0x%X",
                        ETG_CENUM(tCSM_BUS_NUMBER,               (tU8)  bBus),
                        (tU8) csm_abActBusError[bBus]
                      ));
        #endif
      }
      #else // CSM_S_TRACE_ETG
      {
        // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
        BYTE abData[2]={0};

        abData[0] = bBus;
        abData[1] = csm_abActBusError[bBus];
        CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_ERROR_IND, abData, 2);
      }
      #endif // CSM_S_TRACE_ETG
      #endif // CSM_S_TRACE

      for (dwCount = 0; dwCount < CSM_M_ELEMENTS(csm_arBusErrorIndCallbackFkt); dwCount++)
      {
        if (csm_arBusErrorIndCallbackFkt[dwCount].pfvBusErrorInd != NULL)
        {
          #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
          CSM_M_SET_TASK_CONTEXT( &(csm_arBusErrorIndCallbackFkt[dwCount].rTaskInfo),
                                  csm_arBusErrorIndCallbackFkt[dwCount].pfvBusErrorInd
                                  (
                                    csm_arBusErrorIndCallbackFkt[dwCount].pvHandle,
                                    bBus,
                                    csm_abActBusError[bBus]
                                  )
                                );
          #else // CSM_S_USE_CALLBACK_WITH_HANDLE
            csm_arBusErrorIndCallbackFkt[dwCount].pfvBusErrorInd(bBus, csm_abActBusError[bBus]);
          #endif // CSM_S_USE_CALLBACK_WITH_HANDLE
        }
      }
    }
  }
}
#endif // CSM_S_ENABLE_INDICATE_BUSERROR

#undef CSM_SELF
#undef CSM_CAN_STACK
/*******************************************************************************
 * \doxydocu
 * \file          csm.c
 * \brief         CAN stack manager module
 *
 * \see           CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \sw_component  CAN stack
 * \project       Pool module
 * \path          /di_can/modules_swa/csm/src/
 *
 * \ingroup       CSM
 * \ingroup       CSM_IF
 *
 * \authors       V-Team
 *
 * COPYRIGHT      (c) 2004 Blaupunkt GmbH
 *
 * \history_begin
 *
 * 27.02.04  Rev. main\3  CM-DI/ESU3-K"ohler
 * - CAN activation after initialization only if wakeup was detected.
 * - Function CSM_vTriggerBusStateChange() added.
 * - CSM_vCDAWakeup completely new coded.
 *
 * 02.03.04  Rev. main\4  CM-DI/ESU3-K"ohler
 * - All csm_fChangeXY functions except csm_fChangeDummy and csm_fChangeBusOff
 *   moved to project files csm_p.c.
 * - CSM_lMessageWrite() moved from csm.c to Ford csm_p.c.
 * - csm_vFordInitStart() moved from csm.c to Ford CSM_lInitApl().
 * - $Header removed (is not available for ClearCase).
 *
 * 10.03.04  Rev. main\5  CM-DI/ESU3-K"ohler
 * - CSM_vCDAWakeup() renamed CSM_vCDAWakeupInd().
 *
 * 02.04.04  Rev. main\6  CM-DI/ESU3-Franke
 * - CSM_vCBRSignalInd/CSM_vCBRMultipleSignalInd: Added check for NULL pointer in
 *   TelegramIndex
 *
 * 02.04.04  Rev. main\7  CM-DI/ESU3-Franke
 * - CSM_vCBRSignalInd: bracket corrected
 *
 * 25.05.04  Rev. main\8  CM-DI/ESU3-Prhl
 * - integration from mcnet from SW_NAVI PVCS
 *
 * 25.05.04  Rev. main\9  CM-DI/ESU3-Prhl
 * - CSM_lBusStateIndCallBackInit() added
 * - CSM_lBusErrorIndCallBackInit() added
 * - CSM_lSignalCallBackInit() added
 * - CSM_lCommunicationReq() wApplID added
 *
 * 18.06.04  Rev. main\10 CM-DI/ESU3-Prhl
 * - #ifdef ... added to make it compilable
 *
 * 18.06.04  Rev. main\11 CM-DI/ESU3-Prhl
 * - modify CSM_lSignalWrite()
 *
 * 23.06.04  Rev. main\12 CM-DI/ESU3-Prhl
 * - remove CSM_S_FORD_TP_EX switch
 * - remove CSM_S_MCNET_MP switch
 * - tU8,tU16.. removed
 *
 * 25.06.04  Rev. main\13 CM-DI/ESU3-Prhl
 * - CSM_lCommunicationReq() modified
 *
 * 28.06.04  Rev. main\14 CM-DI/ESU3-Prhl
 * - CSM_lSignalCallBackInit() modified
 * - warnings removed
 *
 * 23.07.04  main\15 CM-DI/ESU3-Battis
 * - several CSM_S_CNP_AVAILABLE inserted
 * - Bugfix for main\14
 *
 * 23.07.04  main\16 CM-DI/ESU3-Prhl
 * - #if (CSM_S_CBR_AVAILABLE == CSM_C_F_ON) is used now
 *
 * 25.08.04  main\17 CM-DI/ESA3-Battis
 * - call of CDA_lExit( CSM_C_EXIT_OFF) in  CSM_lExit() inserted
 *
 * 27.08.04  main\18 CM-DI/ESA3-K"ohler
 * - Bus state change in CSM_vCDAWakeupInd() is triggered in state listen too
 *   because the interrupt may occur after switching of the HW during the
 *   transition from listen to down.
 * - First doxygen comment added.
 *
 * 30.08.04 main\19 DM-DI/ESA3-Battis
 * - warnings removed
 *
 * 02.09.04  main\20 CM-DI/ESA3-K"ohler
 * - ISO-TP functions moved to csm_itp.c.
 * - Additional doxygen comments added.
 *
 * 03.09.04  main\21 CM-DI/ESA3-Prhl
 * - new bus state handling added
 *
 * 07.09.04  main\22 CM-DI/ESA3-Prhl
 * - call CBR_lExit() changed
 *
 * 10.09.04  main\23 CM-DI/ESA3-Battis
 * - CSM_bCalculateBusState() moved to csm_p.c
 *
 * 13.09.04  main\24 CM-DI/ESA3-Battis
 * -  call of csm_vPostInitStart() inserted in CSM_lInit()
 *
 * 22.09.04  main\25 CM-DI/ESA3-Battis
 * - initialization of CSM_bCanWasWakeupReason added
 *
 * 24.09.04  main\26 CM-DI/ESA3-Battis
 * - csm_fChangeBusOff() is deprecated
 * - Busoff in CSM_vCDAErrorInd() is reported to the DownsteamTask
 * - call of csm_vPostInitStart() from CSM_lInit() is moved to DownsteamTask
 * - function CSM_vCNMBusoffReinit() added
 * - new implementation of CSM_vCNMBusoffReinitTimerTimeout()
 *
 * 30.09.04  main\27 CM-DI/ESA3-Battis
 * - feature switch used in CSM_vCNMBusoffReinitTimerTimeout()
 *
 * 11.10.04  main\28 CM-DI/ESA3-K"ohler
 * - Some changes in CSM_fCheckCommunicationReq().
 * - Some feature switches added.
 *
 * 18.10.04  main\29 CM-DI/ESA3-K"ohler
 * - Statement without effect for ITP extended addressing data request removed.
 * - Comments changed, Doxygen comments added.
 *
 * 15.10.04  main\30 CM-DI/ESA3-Prhl
 * - csm_bCalculateBusState changed to CSM_bCalculateBusState
 * - Trigger the bus state handler at the end of CSM_lInit( CSM_C_INIT_START)
 *
 * 22.10.04  main\31 CM-DI/ESA3-Prhl
 * - busoff handling for OSEK nm added
 *
 * 22.10.04  main\32 CM-DI/ESA3-Prhl
 * - old code removed
 *
 * 29.10.04  main\33 CM-DI/ESA3-Battis
 * - variable definition moved
 * - variable csm_bCalculateBusState renamed
 *
 * 22.11.04  main\34 CM-DI/ESA3-Battis
 * - new CBR integration
 *
 * 26.11.04  main\35 CM-DI/ESA3-Battis
 * - special implementation of CSM_vCDAWakeupInd for Fiat SB05
 *   needs still be checked later on
 *
 * 02.12.04  main\36 CM-DI/ESA3-Battis
 * - call of CSM_lExitApl() inserted in CSM_lExit()
 *
 * 22.12.04  main\37 CM-DI/ESA3-Battis
 * - warnings removed
 *
 * 18.01.05  main\38 CM-DI/ESA3-K"ohler
 * - CDA_vClearAllTxObjectsIfTxQueueIsFull(bBus) call in CSM_vCNMLastNmMsg()
 *   removed and CDA_lCanRxTxErrInterruptDisableAndSave/Restore() replaced by
 *   CDA_iCanInterruptsDisable/Restore().
 *
 * 18.01.05  main\39 CM-DI/ESA3-K"ohler
 * - C++ comment removed.
 *
 * 03.02.05  main\40 CM-DI/ESA3-Battis
 * -  timer functions renamed
 *
 * 07.02.05  main\41 CM-DI/ESA3-Prhl
 * -  rename tCSM_MCNET_USDT_TX_APPL_CALLBACK to tCSM_MCNET_USDT_APPL_CALLBACK
 *
 * 09.02.05  main\42 CM-DI/ESA3-Prhl
 * -  add some missing #if ... #endif
 * -  #warning moved
 *
 * 17.02.05  main\43 CM-DI/ESA3-Battis
 * - warning in CSM_vCDAWakeupInd() suppressed for FIAT NM
 *
 * 17.02.05  main\44 CM-DI/ESA3-Prhl
 * -  CSM_C_PTYPE_LOCAL_CTRL added for CSM_lCommunicationReq()
 * -  CSM_lInit() and CSM_lExit() modified
 * -  interface #1 function: traces added
 *
 * 18.02.05  main\45 CM-DI/ESA3-Prhl
 * - some changed for CSM_C_PTYPE_FORD_USDT_ISO
 *
 * 18.02.05  main\46 CM-DI/ESA3-Battis
 * - Asserts inserted in CSM_lSignalRead() and CSM_lSignalWrite()
 *
 * 22.02.05  main\47 CM-DI/ESA3-Battis
 * application callback deleted in CSM_vCDAErrorInd()
 *
 * 22.02.05  main\48 CM-DI/ESA3-Prhl
 * ISO TP integration
 *
 * 24.02.05  main\49 CM-DI/ESA3-Prhl
 * #ifdef #endif for local control modified
 *
 * 08.03.05  main\50 CM-DI/ESA3-Zurmhl
 * - CSM_API modifier added to CSM_lBusStateChange() declaration
 *   (even for "non intelligent power mgt." this may be needed for testing under Windows)
 *
 * 08.03.05  main\51 CM-DI/ESA3-Battis
 * - call of CSM_lInitVariants added to csm_alpfInit[]
 *
 * 09.03.05  main\52 CM-DI/ESA3-Prhl
 * - bus wake temp added to CSM_lSignalWrite()
 *
 * 10.03.05  main\53 CM-DI/ESA3-Battis
 * - bugfix
 *
 * 21.03.05  main\54 CM-DI/ESA3-Zurmhl
 * - function body for CSM_lGetCommunicationState() added
 * - CSM_API modifier added to several callback init functions
 *
 * 01.04.05  main\55 CM-DI/ESA3-Zurmhl
 * - keyword "static" changed to "CSM_STATIC" for global variables and functions
 *
 * 05.04.05  main\56 CM-DI/ESA3-Battis
 * - some code featured with  CSM_S_CBR_USES_COMMUNICATION_REQ
 *
 * 05.04.05  main\57 CM-DI/ESA3-Battis
 * - include bugfix
 *
 * 07.04.05  main\58 CM-DI/ESA3-K"ohler
 * - Macro CSM_M_ELEMENTS removed.
 *
 * 08.04.05  main\59 CM-DI/ESA3-Zurmhl
 * - Callback function types renamed from vPFNTp to pfv
 *
 * 27.04.05  main\60 CM-DI/ESA3-Prhl
 * - CSM_lGetAllStateIndicationsAgain() added and old function removed
 *
 * 28.04.05  main\61 CM-DI/ESA3-Zurmhl
 * - CSM_API added to function CSM_lGetAllStateIndicationsAgain
 *
 * 28.04.05  main\62 CM-DI/ESA3-Battis
 * - bugfix
 *
 * 03.05.05  main\63 CM-DI/ESA3-Zurmhl
 * - CSM_S_INTELLIGENT_POWER_MANAGEMENT deleted
 * - CSM_API modifier removed for CSM_lBusStateChange()
 *
 * 27.05.05  main\64 CM-DI/ESA3-Zurmhl
 * - header file "csm_apl.h" included
 *
 * 27.05.05  main\65 CM-DI/ESA3-Prhl
 * - add CSM_ITP_lInit() to csm_alpfInit[]
 *
 * 03.06.05  main\66  CM-DI/ESA3-K"ohler
 * - Indentation in csm_alpfInit[] changed and ITP moved into CNP feature switch.
 *
 * 09.06.05  main\67  CM-DI/ESA3-K"ohler
 * - CSM_lInit/ExitApl featured in csm_alpfInit/Exit function table.
 *
 * 09.06.05  main\68  CM-DI/ESA3-Battis
 * - MISRA happyness
 *
 * 16.06.05  main\69  CM-DI/ESA3-Khler
 * - csm_rLocalControlState initialized now in cold init.
 * - if != NULL added in CSM_vProcessBusStateChange().
 *
 * 17.06.05  main\70  CM-DI/ESA3-K"ohler
 * - CSM_ITP_fCheckCommunicationReq() call featured with switch
 *   CSM_S_ITP_COMMUNICATION_FCT in CSM_fCheckCommunicationReq().
 *
 * 20.06.05  main\71  CM-DI/ESA3-K"ohler
 * - Initialization of ITP variables for upstream task indication moved from
 *   CSM_lInit() to CSM_ITP_lInit().
 * - csm_vInitApplTelRxQueue added for warm init.
 *
 * 30.06.05  main\72  CM-DI/ESA3-K"ohler
 * - Wakeup reason is stored even if CAN is initialized in CSM_vCDAWakeupInd().
 *
 * 05.07.05  main\73  CM-DI/ESA3-Zurmhl
 * - Some error codes revised
 *
 * 07.07.05  main\74  CM-DI/ESA3-Zurmhl
 * - Additional error codes revised
 *
 * 14.07.05  main\75  CM-DI/ESA3-Zurmhl
 * - "CSM_lSignalCallBackInit" changed to "CSM_lSignalCallbackInit"
 * - "CSM_lBusErrorIndCallBackInit" changed to "CSM_lBusErrorIndCallbackInit"
 * - "CSM_lBusStateIndCallBackInit" changed to "CSM_lBusStateIndCallbackInit"
 *
 * 04.08.05  main\76  CM-DI/ESA3-Zurmhl
 * - feature switch corrected: #if (CSM_S_RX_OVERRUN_INDICATION == CSM_C_F_ON)
 *
 * 08.08.05  main\77  CM-DI/ESA3-Battis
 * - bugfix in CSM_vProcessBusStateChange().
 *
 * 09.08.05  main\78  CM-DI/ESA3-Zurmhl
 * - Structure CNP_MCNET_MSG: rDataLength -> wDataLength
 *
 * 11.08.05  main\79  CM-DI/ESA3-Zurmhl
 * - CSM_lGetCommunicationState() enhanced for MCNet ASDT
 *
 * 15.08.05  main\80  CM-DI/ESA3-Zurmhl
 * - CSM_lDataReq now calls CSM_MCNET_lDataReq for MCNet ASDT & USDT
 * - For several API functions, now state csm_bCsmState is checked at entry
 *
 * 30.08.05  main\81  CM-DI/ESA3-Franke
 * - corrected CSM_S_CBR_AVAILABLE encapsulation of SignalWrite and -Read
 *
 * 15.09.05  main\82  CM-DI/ESA3-Zurmhl
 * - For additional API functions, now state csm_bCsmState is checked at entry,
 *   common Error Grade for all such cases now: CSM_C_ERR_ERROR
 *
 * 19.09.05  \main\83 CM-DI/ESA3-Prhl
 * - CSM_vTraceOut() replaced by CSM_M_TRACEOUT()
 *
 * 19.09.05  \main\84 CM-DI/ESA3-Prhl
 * - add handle to all ...CallbackInit() funktions
 *
 * 19.09.05  \main\85 CM-DI/ESA3-Prhl
 * - add handle to all ...CallbackInit() funktions
 *
 * 19.09.05  \main\86 CM-DI/ESA3-Prhl
 * - change defines for trace
 *
 * 28.09.05  \main\87 CM-DI/ESA3-Khler
 * - Two warnings in Opel projects removed.
 *
 * 28.09.05  \main\88 CM-DI/ESA3-Battis
 * - First version after VW merge
 *   USE WITH CAUTION !
 *
 * 07.10.05  \main\89 CM-DI/ESA3-Battis
 * - Busoff handling (substate transition) removed from CSM state machine
 *
 * 20.10.05  \main\90 CM-DI/ESA3-Battis
 * - MCNet communication request integrated.
 *
 * 04.11.05  \main\91 CM-DI/ESA3-Prhl
 * - make ready to compile
 *
 * 04.11.05  \main\92 CM-DI/ESA3-Battis
 * - Set/Get BusHold handling encapsulated with CSM_C_MAX_PTYPE_1
 *
 * 14.11.05  \main\93 CM-DI/ESA3-Prhl
 * - move CSM_vCDAWakeupInd() to CSM_P.C
 *
 * 15.11.05  \main\94 CM-DI/ESA3-Prhl
 * - remove some compiler warnings
 *
 * 16.11.05  \main\95 CM-DI/ESA3-Prhl
 * - tCSM_LOCAL_CTRL_SIGNAL_APPL_CALLBACK renamed to tCSM_LOCAL_CTRL_APPL_CALLBACK
 *
 * 17.11.05  \main\96 CM-DI/ESA3-Battis
 * - usage of several feature switches corrected
 * - PostInitStart concept merged with StackActived Concept:
 *   if CSM_C_DOWN_REASON_CSM_STACK_ACTIVATED is available, then CSM_vStackActivated()
 *   is called from downstream task.
 *
 * 22.11.05  \main\97 CM-DI/ESA3-Battis
 * - multi channel busoff handling integrated.
 *   CSM_C_DOWN_REASON_CDA_BUSOFF_ENTRY_CAN0 renamed to
 *      CSM_C_DOWN_REASON_CDA_BUSOFF_ENTRY
 *   CSM_C_DOWN_REASON_CDA_BUSOFF_RECOVERY_CAN0 renamed to
 *      CSM_C_DOWN_REASON_CDA_BUSOFF_RECOVERY
 *
 * 25.11.2005 \main\98 CM-DI/ESA3 Franke
 * - CSM_lSet/Clear/GetBusHold:
 *   bug fixes, simplifications
 *   project specific define CSM_C_MAX_APPL_ID needed for correct check on
 *   valid ApplID
 *
 * 29.11.05  \main\99 CM-DI/ESA3-Battis
 * - Down reason CSM_C_DOWN_REASON_CDA_BUSOFF_ENTRY considered in CSM_vCDAErrorInd()
 *
 * 02.12.05  \main\100  CM-DI/ESA3-Khler
 * - Type of parameter pvAddressField in CSM_lGetCommunicationState() changed to
 *   const void*.
 *
 * 25.01.06  \main\101  CM-DI/ESA3-Battis
 * - precompiler switched on csm_avpfBusStateInd[] for CBR_vCSMBusStateInd() changed
 * - define CSM_C_MAX_POSSIBLE_APPL_ID added for CSM_lSetBusHold() and
 *   CSM_lClearBusHold()
 *
 * 23.02.06  \main\102  CM-DI/ESA3-Khler
 * - OLD_CODE_MP removed.
 *
 * 27.02.06  \main\103  CM-DI/ESA3-Khler
 * - Parameter pvAddressField in CSM_lApplCallbackInit() and
 *   csm_lLocalCtrlApplCallbackInit() is now a pointer to const.
 *
 * 03.04.06  \main\104  CM-DI/ESA3-Battis
 * - BAP integration on interface #1
 * - new interface #1 function (for BAP) CSM_lDataRead()
 *
 * 04.04.06  \main\105  CM-DI/ESA3-Battis
 * - csm_bap_i.h include removed because it's included from csm_ip.h now
 *
 * 04.04.06  \main\106  CM-DI/ESA3-Battis
 * - CSM_BAP_lExit inserted into csm_alpfExit[]
 *
 * 21.04.06  \main\107  CM-DI/ESA3-Battis
 * - BAP include changed to csm_bap_i.h
 *
 * 10.05.06  \main\108  CM-DI/ESA3-Battis
 * - behaviour of state machine changed: transition during busoff is
 *   permitted now.
 *
 * 10.05.06  \main\109  CM-DI/ESA3-Battis
 * - csm_bap_i.h include moved to csm_i.h
 *
 * 11.05.06  \main\110  CM-DI/ESA3-Battis
 * - pvCallBackFkt made const PVOID in CSM_lApplCallbackInit
 *
 * 29.05.06  \main\111  CM-DI/ESA3-Prhl
 * - switch CSM_S_CSM_LOCAL_CTRL added
 * - some reviews
 * - CSM_C_PTYPE_GM_CAN_LOCAL_CTRL, CSM_C_PTYPE_GM_BR_SIGNAL and CSM_C_PTYPE_GM_USDT_ISO integrated
 *
 * 01.06.06  \main\112  CM-DI/ESA3-Prhl
 * - tCSM_CSM_CFG moved to csm_stack_m.h
 * - CSM_MAX_LOCAL_CTRL_USER added defined in csm_stack_mp.h
 * - CSM_vLocalCtrlProcessCommunicationCon() and CSM_vLocalCtrlProcessCommunicationInd() modified
 *
 * 02.06.06  \main\113 CM-DI/ESA3-Prhl
 * - comments corrected
 *
 * 06.06.06  \main\114 CM-DI/ESA3-Prhl
 * - CSM_C_PTYPE_GM_BR_SIGNAL and CSM_C_PTYPE_GM_USDT_ISO support added to CSM_lCommunicationReq()
 *
 * 14.06.06  \main\115 CM-DI/ESA3-Prhl
 * - virtual network support added
 * - GMLAN support added
 *
 * 14.06.06  \main\116 CM-DI/ESA3-Prhl
 * - CBR_vCSMBusStateInd() renamed to CSM_CBR_vCSMBusStateInd()
 *
 * 26.06.06  \main\117 CM-DI/ESA3-Prhl
 * - third CAN bus is now possible
 *
 * 26.06.06  \main\118  CM-DI/ESA3-Zurmhl
 * - adaptions for third CAN bus
 *
 * 27.06.06  \main\119  CM-DI/ESA3-Zurmhl
 * - further adaptions for third CAN bus
 *
 * 28.06.06  \main\120  CM-DI/ESA3-Haffner
 * - changes for new proxy
 *
 * 07.07.06  \main\121  CM-DI/ESA3-Prhl
 * - change tU16 to WORD
 *
 * 14.07.06  \main\122  CM-DI/ESA3-Prhl
 * - modify csm_lLocalCtrlApplCallbackInit(). add bBus.
 * - modify CSM_lApplCallbackInit(). add bBus.
 * - add feature CSM_S_USE_CALLBACK_WITH_HANDLE.
 *   Add the pvHandle from CSM_lApplCallbackInit() to callback routines.
 *   This is used to handle the process border jump (e.g. TENGINE subsystem)
 *
 * 19.07.06  \main\123  CM-DI/ESA3-Prhl
 * - modify feature CSM_S_USE_CALLBACK_WITH_HANDLE.
 *   add PVOID pvHandle to tCSM_ISO_TP_USDT_APPL_CALLBACK if feature CSM_S_USE_CALLBACK_WITH_HANDLE is active
 *
 * 14.08.06  \main\124  CM-DI/ESA3-Prhl
 * - move #ifdef ... . Now a empty CSM_lConTest() is always avalible
 *
 * 21.08.06  \main\125  CM-DI/ESA3-Khler
 * - CSM_TRACE_C_FKT renamed CSM_C_TRACE_FKT.
 * - ISO-TP normal addressing protocol type placeholder
 *   CSM_C_PTYPE_USED_STD_USDT_ISO added.
 * - Trace handling changed.
 *
 * 21.08.06  \main\126  CM-DI/ESA3-Prhl
 * - add CSM_C_PTYPE_GM_CAN_LOCAL_CTRL and CSM_C_PTYPE_GM_BR_SIGNAL to CSM_lGetCommunicationState()
 * - add csm_lLocalCtrlGetCommunicationState()
 *
 * 01.09.06  \main\127  CM-DI/ESA3-Khler
 * - Trace for CSM_lSignalRead corrected.
 *
 * 01.09.06  \main\128  CM-DI/ESA3-Khler
 * - csm_lLocalCtrlGetCommunicationState() featured with CSM_S_CSM_LOCAL_CTRL, too.
 *
 * 11.09.06  \main\129
 * -  Add protocoll type CSM_C_PTYPE_VW_DDP_TP_20.
 *
 * 26.09.06  \main\130  CM-DI/ESA3-Khler
 * - Trace output in CSM_lCommunicationReq() changed.
 *
 * 11.10.06  \main\130  CM-DI/ESA3-Looby
 * - Added tp type CSM_C_PTYPE_RN_ASDT in DataReq.
 *
 * 13.10.06  \main\133  CM-CR/ESD4-Tuschik
 * - Remove Protocoll type CSM_C_PTYPE_VW_DDP_TP_20.
 *
 * 17.10.06  \main\134  CM-DI/ESA3-Khler
 * - PVOID replaced by void *.
 *
 * 26.10.06  \main\135  CM-DI/ESA3-Haffner
 * - Moved CSM_S_VW_BAP_TP out CNP of dependency.
 * - Initialize variables to avoid warnings.
 * - In CSM_lCommunicationReq and CSM_lDataReq all protocol types except BAP are
 *   handled in csm_proxy (remote stack only).
 *
 * 07.11.06  \main\136  CM-DI/ESA3-Battis
 * - const removed from pvActSignalData of CSM_lSignalRead()
 *
 * 08.11.06  \main\137  CM-DI/ESA3-Prhl
 * - merge from \main\gm_ge09_vr55\7
 * - add feature CSM_S_STATE_MASCHINE.
 * - add feature CSM_S_CDA_AVAILABLE.
 *
 * 08.11.06  \main\138  CM-DI/ESA3-Prhl
 * - multi bus support added for CSM_C_PTYPE_???_LOCAL_CTRL
 *
 * 14.11.06  \main\139  CM-DI/ESA3-Prhl
 * - bugfix for CSM_C_PTYPE_???_LOCAL_CTRL. wrong size for csm_arLocalCtrlCallbackFkt
 *
 * 28.11.06  \main\140  CM-DI/ESA3-Zurmhl
 * - bugfix in CSM_vLocalCtrlProcessCommunicationInd()
 *
 * 01.12.06  \main\141  CM-DI/ESA3-Prhl
 * - add macros CSM_M_GET_TASK_CONTEXT and CSM_M_SET_TASK_CONTEXT for all callback functions
 *
 * 05.12.06  \main\142  CM-DI/ESA3-Prhl
 * - remove bug from csm_lLocalCtrlApplCallbackInit()
 *
 * 12.01.07  \main\143  CM-DI/ESA3-Battis
 * - bus error indications to application not supporting all buses in parallel
 *
 * 12.01.07  \main\144  CM-DI/ESA3-Prhl
 * - make ready to compile for WIN32. some #if moved
 *
 * 12.01.07  \main\145  CM-DI/ESA3-Prhl
 * - add UUDT
 *
 * 29.01.07  \main\146  CM-DI/ESA3-Prhl
 * - add #if/#endif round cnm_mcnet_i.h
 *
 * 09.02.07  \main\147  VTeam-Prhl
 * - add some cases with CSM_C_PTYPE_VW_BR_SIGNAL and CSM_C_PTYPE_VW_BR_SIGNAL in
 *   functions CSM_lApplCallbackInit(), CSM_lGetCommunicationState() and CSM_lCommunicationReq().
 *
 * 14.02.07  \main\148  VTeam-Prhl
 * - add some CSM_M_TRACEOUT()
 *
 * 15.02.07  \main\149  VTeam-Prhl
 * - some CSM_M_TRACEOUT() modified
 *
 * 16.02.07  \main\150  VTeam-Prhl
 * - add some CSM_M_TRACEOUT()
 *
 * 22.02.07  \main\151  VTeam-Prhl
 * - temporary checkin for FNL
 *
 * 23.02.07  \main\151  VTeam-Prhl
 * - Procokoll type GM_NWM integrated
 *
 * 15.03.07  \main\153  VTeam-Prhl
 * - BUS_WAKE_MIN feature added
 *
 * 21.03.07  \main\154  VTeam-Battis
 * - feature switchs checks added
 * - some cpp comments removed.
 * - handling for CSM_C_PTYPE_VW_BR_SIGNAL adjusted.
 *
 * 21.03.07  \main\155  VTeam-Battis
 * - feature switchs check added
 * - adjustments for CSM_C_PTYPE_VW_BR_SIGNAL handling in radio and navi projects
 *
 * 22.03.07  \main\156  VTeam-Prhl
 * - bugfix for BUS_WAKE_MIN feature
 * - old OPEL feature switch removed
 *
 * 22.03.07  \main\157  VTeam-Prhl
 * - adjustments for CSM_C_PTYPE_VW_BR_SIGNAL handling in radio and navi projects
 *
 * 22.03.07  \main\158  VTeam-Prhl
 * - OSEK payload handling added
 *
 * 28.03.07  \main\159  VTeam-Prhl
 * - missing csm_bap_i.h include added
 *
 * 10.04.07  \main\160  VTeam-Prhl
 * - only Trace output changed.
 *
 * 24.04.07  \main\161  VTeam-Prhl
 * - Trace output added to CSM_lApplCallbackInit().
 * - in Function csm_bLocalCtrlGetUserNumber() pvAddressField is not checked now.
 *
 * 15.05.07  \main\162  VTeam-Prhl
 * - Trace output changed in CSM_lApplCallbackInit(), CSM_lDataReq() and CSM_lDataRead().
 * - TP20 function added to interface 1# function.
 *
 * 15.05.07  \main\163  VTeam-Prhl
 * - Trace output added in CSM_lSignalCallbackInit()
 *
 * 15.05.07  \main\164  VTeam-Prhl
 * - spaces after # deleted
 *
 * 04.06.07  \main\165  VTeam-Prhl
 * - csm_vTriggerCsmDownstreamTask() changed to csm_vTriggerCsmDownstreamTaskFromIrq() in CSM_vCDAErrorInd(),
 *   because CSM_vCDAErrorInd() is call from LISR.
 *
 * 14.06.07  \main\166  VTeam-Battis
 * - BAP include removed
 * - feature switch checks adjusted for ISO TP
 * - CSM_vProcessBusStateChange() removed from compilation if
 *   CSM_C_DOWN_REASON_CSM_BUS_STATE_CHANGED is not defined
 * - feature switch usage corrected.
 *
 * 15.06.07  \main\167  VTeam-Prhl
 * - stupid asserts removed
 *
 * 15.06.07  \main\168  VTeam-Battis
 * - BAP include added.
 *
 * 19.06.07  \main\169  VTeam-Prhl
 * - some asserts aded to csm_lLocalCtrlApplCallbackInit()
 *
 * 26.06.07  \main\170  VTeam-Prhl
 * - error indication handling added
 *
 * 28.06.07  \main\171  VTeam-Prhl
 * - trace output moved in CSM_vProcessErrorInd().
 *
 * 02.07.07  \main\172  VTeam-Prhl
 * - feature CSM_S_CSM_LOCAL_CTRL_USER_FUNCTION added.
 *
 * 05.07.07  \main\173  VTeam-Battis
 * - code cleaned up / review done.
 *
 * 06.07.07  \main\174 VTeam-Prhl
 * - some lint warnings removed
 *
 * 09.07.07  \main\175 VTeam-Battis
 * - adjustment for VW remote stack.
 *
 * 11.07.07  \main\176 VTeam-Prhl
 * - trace changed for CSM_lSignalWrite() and CSM_lSignalRead()
 * - csm_dwDownReasonBitfield and csm_dwIndicationReasonBitfield set to 0 during warm init.
 * - make lint happy
 *
 * 11.07.07  \main\177 VTeam-Prhl
 * - trace changed for CSM_lSignalWrite()
 *
 * 11.07.07  \main\178 VTeam-Battis
 * - small adjustment for VW remote stack
 *
 * 11.07.07  \main\179 VTeam-Prhl
 * - trace changed for CSM_lSignalRead()
 *
 * 12.07.07  \main\180 VTeam-Brunotte
 * - small adjustment for error free compiling in GMGE09
 *
 * 13.07.07  \main\181 VTeam-Prhl
 * - CNP_MCNET_vCSMBusStateInd added to csm_avpfBusStateInd[].
 *
 * 19.07.07  \main\182 VTeam-Brunotte
 * - CSM_CBR_lExit added to csm_alpfExit
 * - Warning in CSM_vProcessBusStateChange eliminated
 *
 * 24.07.07  \main\183 VTeam-Prhl
 * - CSM_MCNET_fCheckCommunicationReq() added to CSM_fCheckCommunicationReq()
 *
 * 26.07.07  \main\184 VTeam-Prhl
 * - bugfix for CSM_lBusErrorIndCallbackInit(). Callback is only added onces now.
 *
 * 31.07.07  \main\185 VTeam-Prhl
 * - lint warnings removed or commented out.
 *
 * 01.07.07  \main\186 VTeam-Prhl
 * - deleted "return(TRUE)" added again csm_fLocalCtrlCheckCommunicationReq().
 *
 * 10.08.07  \main\187 VTeam-Prhl
 * - CSM_lSignalWrite() bugfix for monitoring bTxType.
 * - OSEK payload handling changed.
 * - dis/enable signal monitoring added.
 *
 * 13.08.07  \main\188 VTeam-Prhl
 * - check for CSM_S_TRACE_LIMITED_SIGNAL added.
 *
 * 13.09.07  \main\189  VTeam-Khler
 * - Connect state is set per default to disconnect/stack down in
 *   CSM_lGetCommunicationState() if error occured.
 * - Some CSM_C error codes replaced by CSM_E.
 *
 * 25.09.07  \main\190  VTeam-Basavaraj, -Khler
 * - Adaptations for the PAG_ICM project.
 * - Re-ordered the #ifndef checks as the CNP and CDA is not available.
 * - added relevant CSM_CBR processing code with the "CSM_C_PTYPE_VCC_BR_SIGNAL" protocol switch.
 *
 * 26.09.07  \main\191  VTeam-Funke
 * - move CSM_S_UUDT_RX and CSM_S_UUDT_TX request inside CSM_S_CNP_AVAILABLE request
 *
 * 26.09.07  \main\192  VTeam-Funke
 * - add a return value for CSM_CBR_lGetCommunicationState
 *
 * 11.10.07  \main\193  VTeam-Battis
 * - compiler warning removed.
 *
 * 13.11.07  \main\194  VTeam-Prhl
 * - CSM_S_TRACE_LIMITED_SIGNAL adde to CSM_lSignalWrite()
 *
 * 14.11.07  \main\195  VTeam-Funke
 * - replaced CSM_C_PTYPE_VCC_BR_SIGNAL by CSM_C_PTYPE_PAG_BR_SIGNAL
 *
 * 15.11.07  \main\196  VTeam-Prhl
 * - some old code removed.
 *
 * 15.11.07  \main\197  VTeam-Prhl
 * - CNM_OSEK_S_VW_PAYLOAD replaced by CNM_OSEK_S_PAYLOAD
 *
 * 27.11.07  \main\198  VTeam-Khler
 * - Trace databytes for virtual networks in CSM_vProcessBusStateChange()
 *   corrected.
 * - Lint warnings removed.
 * - csm_dwCommunicationRequests replaced by csm_adwCommunicationRequests.
 * - CSM_S_MCNET_ASDT encapsulated with CSM_S_MCNET.
 * - Some Indentations changed.
 *
 * 27.11.07  \main\199  VTeam-Funke
 * - ???
 *
 * 25.01.08  \main\200  VTeam-Prhl
 * - only Asserts in csm_lLocalCtrlApplCallbackInit() removed. Now we are able to deregister callbacks.
 *
 * 30.01.08  \main\201  VTeam-Khler
 * - Some CSM_M_PARAMETER_INTENTIONALLY_UNUSED() added to remove lint warnings.
 *
 * 11.02.08  \main\202  VTeam-Prhl
 * - CSM_lBusStateIndCallbackInit() removed
 * - now it is compilable without CSM_S_VIRTUAL_NETWORK = CSM_C_F_ON
 *
 * 11.02.08  \main\203  VTeam-Prhl
 * - lint authorisation added.
 *
 * 13.02.08  \main\204  VTeam-Brunotte
 * - In CSM_vProcessBusStateChange: Static variable bVN changed to array abVN with
 *   dependence to CSM_C_MAX_CAN_BUS.
 *
 * 21.02.08  \main\205  VTeam-Prhl
 * - lint authorisation added.
 *
 * 22.02.08  \main\206  VTeam-Prhl
 * - make doxygen happy.
 *
 * 22.02.08  \main\207  VTeam-Prhl
 * - CSM_lInit() modified. csm_bCsmState is not set if one of the ...lInit() calls has failed.
 *
 * 25.02.08  \main\208  VTeam-Funke
 * - Use of CSM_S_ENABLE_INDICATE_STATIC_ERROR added
 *   This responses an additional error code in case of CommunicationRequest fails.
 *
 * 27.02.08  \main\209  VTeam-Lansley / Funke
 * - added protocoltype CSM_C_PTYPE_PAG_OSEK_NWM
 *
 * 27.02.08  \main\210  VTeam-Brunotte
 * - CSM_vProcessBusStateChange work flow changed
 *
 * 14.03.08  \main\211  VTeam-Khler
 * - CSM_C_PTYPE_USED_STD_BR_SIGNAL added in CSM_lCommunicationReq() and
 *   CSM_lApplCallbackInit().
 *
 * 10.04.08  \main\212  VTeam-Prhl
 * - CSM_lMultipleSignalRead() added.
 *
 * 16.04.08  \main\213  VTeam-Basavaraj, Funke
 * - Moved the call of CSM_iInterruptsDisable after the if statement in the function CSM_vIndicateBusError
 *
 * 16.04.08  \main\214  VTeam-Prhl
 * - comment changed for CSM_lMultipleSignalRead() .
 *
 * 21.04.08  \main\215  VTeam-Prhl
 * - some feature switches moved. Now we can compile without CNM.
 *
 * 29.04.08  \main\216  VTeam-Prhl
 * - CSM_lMultipleSignalWrite() added.
 * - CSM_lMultipleSignalRead() modified.
 *
 * 29.04.08  \main\217  VTeam-Funke
 * - CSM_bGetBusError() added.
 *
 * 20.05.08  \main\218  VTeam-Funke
 * - trace out pvHandle in CSM_lApplCallbackInit().
 *
 * 30.05.08  \main\219  VTeam-Prhl
 * - some feature switches moved to be ready to link without CBR or CNP.
 *
 * 12.06.08  \main\220  VTeam-Brunotte
 * - CSM_ITP_lExit call inserted in csm_alpfExit.
 * - MultipleSignalWrite/Read is commented out.
 *
 * 12.06.08  \main\221  VTeam-Brunotte
 * - MultipleSignalWrite/Read is available again.
 *
 * 18.06.08  \main\222  VTeam-Brunotte
 * - MultipleSignalWrite/Read is commented out.
 * - CSM_LOCAL_lInit and CSM_LOCAL_lExit added and placed into the csm_alpfInit
 *   and csm_alpfExit arrays. These functions initialize the callback function arrays.
 *
 * 18.06.08  \main\223  VTeam-Brunotte
 * - MultipleSignalWrite/Read is available again.
 *
 * 19.06.08  \main\224  VTeam-Khler
 * - CSM_lBusStateChange() calls encapsulated with feature switch
 *   CSM_S_STATE_MASCHINE
 *
 * 24.06.08  \main\225  VTeam-Funke
 * - replaced project specific LOCAL_CTRL types by
 *   CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL
 *
 * 15.07.08  \main\226  VTeam-Khler
 * - lReachedState is set to bNewState if CSM_S_STATE_MASCHINE != CSM_C_F_ON in
 *   CSM_vProcessBusStateChange()
 *
 * 15.07.08  \main\227 VTeam-Funke / Basvaraj
 * - changes for reducing Lint and Compiler warnings
 *
 * 24.07.08  \main\228 VTeam-Zurmhl
 * - Some reviewing done, some comments updated
 * - FS CSM_S_USE_TKSE added at some places
 *
 * 01.09.08  \main\229  VTeam-Khler
 * - Lint warnings removed.
 * - Some modifications in CSM_lDataIndProcessed().
 *
 * 15.09.08  \main\230  Prhl
 * - Lint warnings removed.
 *
 * 15.09.08  \main\231  Prhl
 * - Lint warnings removed.
 * - MPDT added.
 *
 * 15.09.08  \main\232  Prhl
 * - code review done.
 *
 * 22.09.08  \main\233  Prhl
 * - some MPDT code added.
 *
 * 22.09.08  \main\234  VTeam-Pistoor
 * - Lint warnings removed.
 *
 * 29.09.08  \main\235  Prhl
 * - bus off handling for MPDT integrated.
 *
 * 30.09.08  \main\236  VTeam-Franke
 * - added callbacks for CSM_C_PTYPE_FORD_USDT_LIN_DIAG
 *
 * 01.10.08  \main\236  Prhl
 * - function CSM_lDataIndProcessed() modified. Support for CSM_C_PTYPE_RN_MPDT_C and CSM_C_PTYPE_RN_MPDT_D added.
 *
 * 13.10.08  \main\239  VTeam-Funke
 * - align return value to same name
 * - reduce return points of function
 * - added static error handling
 * - overworked error handling in case of CSM_C_INIT_START
 *
 * 27.10.08  \main\240  VTeam-Zurmhl
 * - erroneous code fragments for MCNET_USDT removed in CSM_lApplCallbackInit()
 *
 * 24.11.08  \main\241  VTeam-Khler
 * - Lint warnings removed.
 *
 * 29.11.08  \main\242  VTeam-Pistoor
 * - added bus error indication functionality
 *
 * 28.01.09  \main\243  VTeam-Pistoor
 * - renamed CSM_USDT_LIN_DIAG to CSM_LIN_DIAG
 *
 * 02.02.09  \main\244  Prhl
 * - feature switch CSM_S_ENABLE_INDICATE_BUSERROR moved at CSM_lBusErrorIndCallbackInit()
 * - CSM_C_PTYPE_USED_STD_BR_SIGNAL used at CSM_lGetCommunicationState()
 *
 * 03.02.09  \main\245  VTeam-Franke
 * - CSM_S_LIN_DIAG encapsulation
 *
 * 26.02.09  \main\246  VTeam-Khler
 * - lReachedState handling changed in CSM_vProcessBusStateChange()
 * - Indentation for includes changed.
 *
 * 05.06.09  \main\247  VTeam-Pistoor
 * - Changed error grade in case of initial default from CSM_C_ERR_ERROR to CSM_C_ERR_WARNING
 *
 * 17.07.09  \main\248  Prhl
 * - local control functions are moved to modul csm_lctr.c
 *   You have to use some more updated files!
 *     csm.c        \main\248 or newer
 *     csm_m.h      \main\41 or newer
 *     csm_stack.c  \main\114  or newer
 *   You have to update the content of your project file csm_stack_mp.h
 *     CSM_MAX_LOCAL_CTRL_USER renamed to CSM_LCTR_C_MAX_USER
 *     csm_arCsmCfg[] is now part of "#ifdef CSM_LCTR_SELF" and not part of "#ifdef CSM_SELF"
 *   Do not forget to add csm_lctr.c and csm_lctr_i.h to your project src.sml files.
 *
 * 23.07.09  \main\249 VTeam-Jablonski
 * - Init of csm_arBusErrorIndCallbackFkt moved to CSM_lnit
 *
 * 23.07.09  \main\250 VTeam-Jablonski
 * - bUserNumber defined
 *
 * 17.08.09  \main\251 VTeam-Franke
 * - new: CSM_lCommunicationReq(CSM_C_DISCONNECT_ALL)
 *
 * 25.08.09  \main\252  VTeam-Khler
 * - Variable csm_bCsmState is initialized CSM_C_STATE_UNDEFINED again if
 *   GLB_CAR_RADIO environment is not defined.
 *
 * 09.09.09  \main\253 Prhl
 * - main\251 change removed.
 *
 * 17.08.09  \main\254 VTeam-Franke
 * - new flag for CommunicationRequest
 *   (CSM_C_ACTIVATE/DEACTIVATE_COMMUNICATION_REQUESTS)
 *   use along with csm_m.h \main\42 and csm_stack_p.h \main\76
 *
 * 23.09.09  \main\255 VTeam-Franke
 * - lint warning removed
 *
 * 29.09.09  \main\256 VTeam-Jablonski
 * - CSM_LCTR_lDataReq added
 *
 * 13.10.09  \main\257 Prhl
 * - missing feature switch added.
 *
 * 14.10.09  \main\258 Prhl
 * - missing feature switch added.
 *
 * 11.12.09  \main\259 Prhl
 * - only review done
 *
 * 22.01.10  \main\260 Prhl
 * - old IVLAN stuff removed
 *
 * 25.01.10  \main\261 Prhl
 * - Busoff handling for IVLAN added in CSM_vCNMBusoffReinitTimerExpired()
 *
 * 22.02.10  \main\262 Riecker
 * - #define CSM_C_IND_REASON_ITP_COMM_IND          0x0010   added
 *
 * 29.04.10  \main\263 Franke
 * - CSM_lCommunicationReq: bBus validity check bug fix
 *
 * 20.05.10  \main\263 Franke
 * - OSEKI header file included, if feature is ON
 * - CNM_OSEKI_vReinitTimerExpired() called if feature is ON
 *
 * 26.05.10  \main\264 Riecker
 * - move #define CSM_C_IND_REASON_ITP_COMM_IND  to header file
 *
 * 15.06.10  \main\265 Borck
 * - CSM_fCheckCommunicationReq extendet by MPDT support, additionally beautified !!!
 *
 * 31.08.10  \main\267 Prhl
 * - CSM_ITP_lGetCommunicationState() call added to CSM_lGetCommunicationState()
 *
 * 07.09.10  \main\268 Feldhaus
 * - CSM extension for J1939 protocol
 *
 * 07.10.10  \main\269 Franke
 * - ISO TP: enhance maximum channels to 32
 *   use along csm_stack_m.h \main\207
 *             csm_itp.c  \main\61
 *                        \main\ford_mfd\10
 *             cnp_itp.c  \main\70
 *                        \main\vector_adaptation\29
 *             cnp_itp_i.h \main\26
 *
 * 25.10.10  \main\270 Feldhaus
 * - bugfix for compilation error of J1939_NM feature switch being off
 *
 * 26.10.10  \main\271 Borck
 * - CSM_lApplCallbackInit() extended by CSM_C_PTYPE_RN_OSEKI_NWM, source code re-aligned within function
 * - CSM_lGetCommunicationState() extended by CSM_C_PTYPE_RN_OSEKI_NWM, source code re-aligned within function
 *
 * 10.11.10  \main\273 Prhl
 * - CNP_ITP_vResetIndicationFlags() renamed to CSM_ITP_vResetIndicationFlags()
 *   use along csm.c        \main\273
 *             csm_itp.c    \main\62
 *             csm_itp_m.h  \main\22
 *             cnp_itp.c    \main\72
 *             cnp_itp_i.h  \main\27
 *
 * 01.12.10  \main\274 Borck
 * - CSM_lApplCallbackInit() extended by returning a static error, if it exists, otherwise behave as normal
 * - CSM_lGetCommunicationState() extended by returning a static error, if it exists, otherwise behave as normal
 * - at some places code readability has been improved a lttle bit.
 *
 * 14.01.11  \main\275 Borck
 * - LINT info 715 fixed.
 *
 * 21.01.11  \main\276 Funke
 * - add missing condition forCSM_S_ENABLE_INDICATE_STATIC_ERROR
 *
 * 01.02.11  \main\277 Borck
 * - compiler warnings fixed
 *
 * 09.03.11  \main\279 Feldhaus
 * - busoff handling for J1939 protocol is handled only in CNP and removed from CNM
 *
 * 21.03.11  \main\280 Borck
 * - reviewed
 *
 * 06.06.11  \main\282 Borck
 * - CSM_lCommunicationReq() extended for OSEKI NM support.
 *
 * 08.06.11  \main\283 Borck
 * - static var wCSM_EmTraceError introduced along with get and set function.
 *   this allows storing some simple trace information, e.g. while controller init,
 *   which is then later been written to em.
 * - this version requires at least \csm_stack\src\csm_stack_m.h@@\main\221
 * - on ApplCallbackInit, the new var is checked for error and em is written in case of.
 *
 * 21.06.11  \main\284 Borck
 * - ETG tracing introduced for CalcBusState(), ApplCallbackInit() and CommunicationReq()
 *   as examples. Additional changes in csm_stack_i.h and csm_stack_ip.h required.
 * - usage of at least: \csm_stack_i.h\main\136 required if ETG tracing is enabled.
 * - for reference:     \csm_stack_ip.h\nissan_lcn2\26
 *
 * 11.07.11  \main\285 Borck
 * - ETG tracing extended for some more functions now.
 * - EM Trace variable initialized at cold init.
 *
 * 12.07.11  \main\286 Borck
 * - ETG tracing extended again.
 *
 * 02.08.11  \main\287 Prhl
 * - CSM_vCheckandWriteErrorToEM() removed. CSM_vCheckandWriteErrorToEM() has to be moved to a plattfrom or project depended file!
 *
 * 10.08.11  \main\288 Prhl
 * - lint fix moved to the end of the function CSM_lBusErrorIndCallbackInit().
 *   Now I can compile it with CSM_S_ENABLE_INDICATE_BUSERROR on.
 *
 * 20.09.11  \main\289 Feldhaus
 * - added CSM_MCNET_lApplCallbackInit for protocoltype MCNET_USDT.
 *
 * 21.06.12  \main\290  Khler
 * - CSM_SELF_TEST_lGetCommunicationState() for protocol type
 *   CSM_C_PTYPE_GM_CSM_SELF_TEST added.
 *
 * 11.04.13  \main\291  Feldhaus
 * - embraced all declarations of callback array csm_arBusErrorIndCallbackFkt
 *   with feature switches CSM_S_CALLBACK_INIT_BY_APPLICATION and
 *   CSM_S_USE_CALLBACK_WITH_HANDLE.
 *
 * 10.05.13  \main\292  Prhl
 * - VW NM HIGH integrated.
 *
 * 26.07.13  \main\293  Prhl
 * - CCS_NET_vCSM_BusStateInd() adde to csm_avpfBusStateInd[]
 *
 * 01.08.13  \main\294  Feldhaus
 * - removed static definition for CSM_lCheckCommunicationReq which is needed by VW HIGH NM stat management
 * - added special feature switch CSM_S_VWHIGH_NM == CSM_C_F_OFF for CAN state calculation
 *
 * 13.08.13  \main\295  Prhl
 * - compiler warnings removed.
 *
 * 13.08.13  \main\296  Prhl
 * - CSM_BAP_lApplCallbackInit() prototype changed. Now CSM_BAP_lApplCallbackInit() similar to all other ..._lApplCallbackInit()
 *
 * 26.08.13  \main\297  Prhl
 * - first step of LOCAL CTRL adaption for CSM Proxy.
 *
 * 26.08.13  \main\298  Prhl
 * - TODOs added.
 *
 * 12.09.13  \main\299  Feldhaus
 * - final step of local control adaptation for CSM Proxy
 *
 * 26.09.13  \main\300  Prhl
 * - modification for local control adaptation for CSM Proxy
 *
 * 10.10.13  \main\301  Feldhaus
 * - CSM_ITP_lCommunicationReq is only called when CSM_S_ITP_COMMUNICATION_FCT is CSM_C_F_ON
 *   (returns an error otherwise)
 *
 * 22.10.13  \main\302  Prhl
 * - call of CSM_LCTR_vSendAllIncicationsAgain(), CSM_ITP_vSendAllIncicationsAgain() and CSM_CBR_vSendAllIncicationsAgain() added to CSM_lGetAllStateIndicationsAgain().
 *
 * 26.10.13  \main\303  Prhl
 * - renamed to CSM_LCTR_vSendAllIndicationsAgain(), CSM_ITP_vSendAllIndicationsAgain() and CSM_CBR_vSendAllIndicationsAgain() in CSM_lGetAllStateIndicationsAgain().
 *
 * 26.10.13  \main\304  Prhl
 * - ETG traces for CSM_lMultipleSignalRead() added.
 *
 * 01.11.13  \main\305  Prhl
 * - interface for  CSM_BAP_lDataReq() changed.
 *
 * 07.11.13  \main\306 Prhl
 * - lint warning removed.
 *
 * 04.12.13  \main\307 Prhl
 * - lint warning removed.
 *
 * 06.12.13  \main\308 Prhl
 * - lint warning removed. missing #ifdef around feature switches.
 *
 * 18.12.13  \main\309 Prhl
 * - old feature switches removed around CSM_CBR_lApplCallbackInit()
 *
 * 31.01.14  \main\310 Prhl
 * - call CCS_NET_vCSM_BusStateInd() in CSM_lGetAllStateIndicationsAgain() to send the bus state to the main processor.
 *
 * 21.02.14  \main\311 Prhl
 * - withdraw change from \main\294
 *
 * 26.02.14  \main\312 Prhl
 * - payload handling for VWHIGH NM of CNM_VWHIGH_C_NMINFO_ACTIVE_WAKEUP_BIT added
 *
 * 07.08.14  \main\313 Feldhaus
 * - added CSM_J1939_lGetCommunicationState
 *
 * 09.12.14  \main\314 Prhl
 * - LINT warning removed.
 *
 * 25.02.15  \main\315 Borck
 * - J1939 CNM include file featured with the correct feature switch for NM
 *
 * 06.03.15  \main\316 Prhl
 * - trace without ETG for Autosar enviroment.
 *
 * 08.06.15  \main\317 Borck
 * - ETG trace for errors upon Communication Request
 * - ETG trace for errors upon Data Request
 * - ETG trace for errors upon Signal Write
 * - ETG trace for errors upon Get Communication State
 * - all extensions require new csm_stack_i.h (at least version 152)
 * - ETG traces of CSM IF1 extended by tag ####1 (better be identified in output)
 * - some coding style cleanup
 *
 * 21.07.15  \main\318 Borck
 * - Signal read trace gives "Little Endian Hint" for data bytes
 *
 * 23.07.15  \main\319 Borck
 * - some LINT issues and coding style fixed
 *
 * 10.09.15  \main\320 Prhl
 * - CSM_lSignalWrite() changed. bTxType will not be filtered if it is a proxy software.
 *
 * 24.09.15  \main\321 Borck
 * - first attempt to integrate the SLOT based CBR Signal abstraction.
 *   -- only SignalRead, Write and SignalCallbackInit are handled.
 *   -- MultipleRead and Write are still unhandled !!
 *
 * 05.10.15  \main\322 Borck
 * - Multi-Signal Support disabled if Signal Abstraction is enabled (Implicit by renamed Read-Write functions)
 *
 * 06.10.15  \main\323 Borck
 * - optimized integration of the CBR Signal abstraction:
 *   -- include header file of cbr_al
 *   -- checking of new feature switches
 *   -- Signal functions for Read/Write/CallbackInit are again located always here !!
 *   -- decision to call CBR or CBR_AL is made within the 3 functions
 *   -- MultiSignal Read/Write is still hidden (disabled) if CBR Signal Abstraction is active
 *
 * 06.10.15  \main\324 Borck
 * - Typo fixed
 *
 * 04.11.15  \main\325 M.Prhl
 * - CNM_OSEKI_vSendAllIndicationsAgain() call added to CSM_lGetAllStateIndicationsAgain()
 *
 * 03.06.16  \main\326 M.Prhl
 * - CSM_lMultipleSignalWrite() is now available again if CBR Signal Abstraction is active
 *
 * 03.06.16  \main\327 M.Prhl
 * - CSM_lMultipleSignalRead() is now available again if CBR Signal Abstraction is active
 *
 * 23.06.16  \main\328 A.Borck
 * - API extended to provide proxy full frame signal write (required for CBR gateway feature)
 *
 * 14.07.16  \main\329 M.Prhl
 * - traces added to CSM_lFullFrameSignalWrite.
 *
 * 22.09.16  \main\330 A.Borck
 * - suppress TempWakeUp-Flag when passing TX-Type parameter to CBR_lFullFrameSignalWrite (same handling as for call of CBR_lSignalWrite)
 *
 * 01.11.16  \main\331 A.Borck
 * - added ETG tracing for some sections that are used by the project (so that old trace output is disabled on iMX site)
 *
 * 11.11.16  \main\332 A.Borck
 * - trace buffer sizes have been checked against CSM_MAX_TRACE_LENGTH due to reduction of the same in csm_stack_m.h
 *
 * 19.05.17  \main\333 A.Borck
 * - Trace for SignalRead set under USER4 Level to avoid annoying, cyclic trace output. USER3 will be the preferred trace class to be set.
 *
 * 21.06.17  \main\334 A.Borck
 * - Trace for SignalWrite set under USER4 Level to avoid annoying, cyclic trace output. USER3 will be the preferred trace class to be set.
 *
 * 24.11.17  main\g3inf4cv_csm_proxy_in\1  Jayashree
 * - Modified protocoltypes as per INF4CV
 * \history_end
 *//**** END OF FILE **********************************************************/
