/*****************************************************************************
| FILE:         ProcCsm.cpp
| PROJECT:      NISSAN LCN2kai
| SW-COMPONENT: Process
|-----------------------------------------------------------------------------
| DESCRIPTION:  Process Base Entry
|
|-----------------------------------------------------------------------------
| COPYRIGHT:    (c) 2013 Bosch Car Multimegia GmbH
| HISTORY:
| Date      | Modification                                    | Author
| 18.08.11  |                                                 | Kollai
| 09.03.12  |                                                 | Kempen
| 08.07.13  |                                                 | Prhl
| 15.07.13  | BAP support                                     | Prhl
| 26.08.13  | local control support                           | Prhl
| 07.11.13  | lint fixes                                      | Prhl
| 20.11.13  | MsgQueueAdmin + ETG traces                      | Kempen
| 02.06.15  | small fix for ETG trace                         | Borck
| 02.06.15  | OSEKI support                                   | Prhl
| 05.10.15  | No Multi-Signal Support for Abstract Signals    | Borck
| 26.10.16  | DataPool support for KDS items introduced       | Borck
| 29.01.16  | DataPool support for new KDS items introduced   | Prhl
| 23.05.16  | DataPool support for TrafficSignRec modified    | Borck
| 16.10.16  | Reviewed, analysed and extended                 | Borck
| 22.11.16  | HandleInit is now storing the queue item
|             element size -> CSMAL_lStoreHandleData extended | Borck
| 30.11.16  | Trace output reworked                           | Borck
| 22.02.17  | DataPool support for more KDS items introduced  | Borck
| 28.02.17  | ETG output fix                                  | Borck
| --.--.--  | ----------------                                | -----
|
|*****************************************************************************/
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define SYSTEM_S_IMPORT_INTERFACE_EXH
#define SYSTEM_S_IMPORT_INTERFACE_VECTOR
#include "stl_pif.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#include "dp_effc_if.h"

#define CSM_S_IMPORT_INTERFACE_GENERIC_USER
#include "csm_if.h"
#include "csm_stack_m.h"
#include "csm_access_krnl.h"
#include "ProcCsm.h"

// Trace class definition for ETG (extented trace generator).
#define TR_CLASS_CSMPROC (TR_COMP_CSM + 0x10)

#if (CSM_S_TRACE == CSM_C_F_ON)
  #define ETRACE_S_IMPORT_INTERFACE_GENERIC
  #define ET_TRACE_INFO_ON
  #include "etrace_if.h"

  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
    #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CSMPROC
    #include "trcGenProj/Header/ProcCsm.cpp.trc.h"
  #endif
#endif

//
#define CSM_EV_SHUTDOWN_NAME  "CSM_SDOWN"
#define CSM_EV_SHUTDOWN       0x00000001

#define CSM_PROC_WORKER_THREAD_NAME       "PROC_CSM_WORK"
#define CSM_PROC_WORKER_THREAD_STACKSIZE  4096
#define CSM_PROC_WORKER_THREAD_PRIO       100


#define CSM_C_PROC_MESSAGE_QUEUE_MAX_ENTRIES  20

typedef struct
{
   OSAL_tProcessID     hProcessID;
   tU32              u32MsgQueueCnt;
   OSAL_tMQueueHandle  hMessageQueueID;
} trCSMParentMessageQueueAdmin;

bpstl::vector<trCSMParentMessageQueueAdmin, bpstl::allocator<trCSMParentMessageQueueAdmin> > vecParentMessageQueueAdmin;

OSAL_tMQueueHandle hMessageQueueProc = OSAL_C_INVALID_HANDLE;

// used for variant "late CSM startup"
DWORD CSM_Proc_dwGetKDSValue(BYTE* pTarget, BYTE bItem, DWORD* dwError);
// used for new variant "early CSM startup"
BYTE _CSM_Proc_vGetDatapoolEarlyNorItem(BYTE* pTarget, BYTE bItem);

tVoid _CSM_Proc_vInitDatapoolPDD(tVoid);
tVoid _CSM_Proc_vSetDatapoolPDDItem(BYTE, BYTE);

tVoid vStartWorkerThread( tVoid );
tVoid vWorkerThreadFunction( tPVoid pvArg );

tVoid vHandleInit( trCSMProcMessageQueueItemInit rInit );
tVoid vHandleRegisterAppl( trCSMProcMessageQueueItemRegisterAppl rRegisterAppl );
tVoid vHandleGetCommunicationState( trCSMProcMessageQueueItemGetCommunicationState rGetCommunicationState );
tVoid vHandleSignalRead( trCSMProcMessageQueueItemSignalRead rSignalRead );
tVoid vHandleMultipleSignalRead( trCSMProcMessageQueueItemMultipleSignalRead rMultipleSignalRead );
tVoid vHandleDataRead( trCSMProcMessageQueueItemDataRead rDataRead );

tVoid vSendGeneralResult( tCString coszDebugString, tenCSMParentMessageQueueItemID enItemID, OSAL_tProcessID hProcessID, tU32 u32MsgQueueCnt, tS32 s32Result );

OSAL_tMQueueHandle hGetParentMessageQueueID( OSAL_tProcessID hProcessID, tU32 u32MsgQueueCnt );
tVoid vCloseAllParentMessageQueues( tVoid );

//**************************************************************************************************
// function implementation
//**************************************************************************************************
tS32 s32RunEnv( tVoid )
{
   tS32 s32RetVal = OSAL_OK;

   // open Trace ------------------------------
   #if (CSM_S_TRACE == CSM_C_F_ON)
   ET_TRACE_OPEN;
   ETG_TRACE_USR1(("ProcCsm - Started"));
   #endif

   // initialize exception handling
   if ( TRUE == exh_bInitExceptionHandling() )
   {
      tBool bExcHandling = exh_bCreateExceptionContext( 0, 0 );
      if ( TRUE == bExcHandling )
      {
         OSAL_tEventHandle  hEvShutdown = OSAL_C_INVALID_HANDLE;
         OSAL_tEventMask    hEvMask     = CSM_EV_SHUTDOWN;
         OSAL_tEventMask    hEvRequest  = 0;

         // first create event
         if ( OSAL_s32EventCreate( CSM_EV_SHUTDOWN_NAME, &hEvShutdown ) != OSAL_ERROR )
         {
            // from now on (09.06.2017) Data pool (v850 mirror) is initialized/updated only on trigger from VehicleData (KDS is changed via Diagnosis)
            // _CSM_Proc_vInitDatapoolPDD();

            CSM_lInit(CSM_C_INIT_COLD);
            CSM_lInit(CSM_C_INIT_WARM);
            CSM_lInit(CSM_C_INIT_START);

            CSMAL_lInit();

            // now setup and start the worker thread
            vStartWorkerThread();

            #if (CSM_S_TRACE == CSM_C_F_ON)
               ETG_TRACE_USR1(("ProcCsm - Wait for Shutdown-event %08x = %s!", hEvShutdown, CSM_EV_SHUTDOWN_NAME));
            #endif
            tS32 sRet = OSAL_s32EventWait( hEvShutdown, hEvMask, OSAL_EN_EVENTMASK_OR, OSAL_C_TIMEOUT_FOREVER, &hEvRequest );
            #if (CSM_S_TRACE == CSM_C_F_ON)
               ETG_TRACE_USR1(("ProcCsm - Shutdown-event received -> stop CSM (%d(%08x)) -> %d!", sRet, sRet, hEvRequest));
            #endif

            vCloseAllParentMessageQueues();

            CSMAL_lExit();

            CSM_lExit(CSM_C_EXIT_STOP);
            CSM_lExit(CSM_C_EXIT_OFF);

            OSAL_s32EventClose( hEvShutdown );
            OSAL_s32EventDelete( CSM_EV_SHUTDOWN_NAME );
         }
         else
         {
            #if (CSM_S_TRACE == CSM_C_F_ON)
               ETG_TRACE_ERR(("ProcCsm - Failed to create Shutdown event --> Delete proc."));
            #endif
         }

         // free Exceptionhandling
         exh_vFreeExceptionContext();
      } // bExcHandling
      exh_vFreeExceptionHandling();
   } // exh_bInitExceptionHandling

   return s32RetVal;
}
//**************************************************************************************************
// At startup , the slot number information is now read from EarlyNOR in csm_apl.c
// thats why we need a wrapper function here.
//**************************************************************************************************
extern "C"
{
  BYTE CSM_Proc_vGetDatapoolEarlyNorItem(BYTE* pTarget, BYTE bItem)
  {
    tS32 lRet = 0;
    // call our internal CPP function
    lRet = _CSM_Proc_vGetDatapoolEarlyNorItem(pTarget, bItem);
    return lRet;
  }
}

BYTE _CSM_Proc_vGetDatapoolEarlyNorItem(BYTE* pTarget, BYTE bItem)
{
  BYTE bReturn = 0;
  // DataPool Access Variables
  tS32 s32DPReturnValue=DP_S32_NO_ERR;
  TCSMEarlyNorConfiguration rCSMEarlyNorConfiguration;
  dp_tclCsmEarlyNorConfigCsmEarlyNorConfig rCsmEarlyNorConfig;

  switch(bItem)
  {
    // read SLOT
    case CSM_APL_C_READ_EARLYNOR_SLOT:
      s32DPReturnValue=rCsmEarlyNorConfig.s32GetData(rCSMEarlyNorConfiguration);
      if(s32DPReturnValue>=DP_S32_NO_ERR)
      {
        *pTarget=rCSMEarlyNorConfiguration.bEarlyNorSlot;
        bReturn = 1;
      }
      else
      {
        #if (CSM_S_TRACE == CSM_C_F_ON)
        #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
          #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
          ETG_TRACE_FATAL(("_CSM_Proc_vGetDatapoolEarlyNorItem - WARNING - Early Nor value retrieval failed , ErrorCode:0x%x ", s32DPReturnValue));
          #endif
        #endif
        #endif
      }
      break;

    default:
      //do nothing
      break;
  }
  return bReturn;
}

//**************************************************************************************************
// The DataPool Initialisation is now being done at runtime, if KDS is changed via Diagnosis
// The usage call is being done from within a static C-library:
// thats why we need a wrapper function here to be able to distinguish between calls from CPP and C code
//**************************************************************************************************

// define a wrapper C-function for DP-Initialisation from static c-library
extern "C"
{
  tVoid CSM_Proc_vInitDatapoolPDD(tVoid)
  {
    // call our internal CPP function
    _CSM_Proc_vInitDatapoolPDD();
  }
}

//**************************************************************************************************
// Introduced a possibility to modify DataPool items at runtime via TTFis
// 1) Intention is ONLY to quickly modifiy a persistent v850 datapool item for test purposes !!!
// 2) Keep in mind: to see changes and impact, a restart (v850 init) is required for changed v850 behaviour !!!
// 3) Keep in mind that after the 2. restart the changes are gone again, since the iMX items have been retrieved from KDS
//    and have been mirrored to v850 as well !!! so this is a sort of test feature with temporary impact only.
// 4) SLOT item is not modifyable !!
//**************************************************************************************************
extern "C"
{
  tVoid CSM_Proc_vSetDatapoolPDDItem(BYTE bItem, BYTE bValue)
  {
    // call our internal CPP function
    _CSM_Proc_vSetDatapoolPDDItem(bItem, bValue);
  }
}

tVoid _CSM_Proc_vSetDatapoolPDDItem(BYTE bItem, BYTE bValue)
{
  // create instances of our DP classes
  dp_tclCsmDpPDDSteeringWheelType    bSteeringWheelType;
  dp_tclCsmDpPDDVoiceRec             bVoiceRec;
  dp_tclCsmDpPDDFuelType             bFuelType;
  dp_tclCsmDpPDDCameraSystem         bCameraSystem;
  dp_tclCsmDpPDDIpaHfp               bIpaHfp;
  dp_tclCsmDpPDDSonar                bSonar;
  dp_tclCsmDpPDDAntiTheft            bAntiTheft;
  dp_tclCsmDpPDDAnticipatoryLine     bAnticipatoryLine;
  dp_tclCsmDpPDDSteeringMsg          bSteeringMsg;
  dp_tclCsmDpPDDTrafficSignRec       bTrafficSignRec;
  dp_tclCsmDpPDDNaviFunc             bNaviFunc;
  dp_tclCsmDpPDDRCSType              bRCSType;
  dp_tclCsmDpPDDITCommanderType      bITCommanderType;
  dp_tclCsmDpPDDVCANMCANGateway      bVCANMCANGateway;
  dp_tclCsmDpPDDFuelConsumptionInfo  bFuelConsumptionInfo;
  dp_tclCsmDpPDDTPMS                 bTPMS;
  dp_tclCsmDpPDDVDC                  bVDC;
  dp_tclCsmDpPDDAdFunction           bAdFunction;
  dp_tclCsmDpPDDVehicleSetting       bVehicleSetting;
  dp_tclCsmDpPDDHudSetting           bHudSetting;
  dp_tclCsmDpPDDHvacType             bHvacType;
  dp_tclCsmDpPDDDigitalAssi          bDigitalAssi;
  dp_tclCsmDpPDDSendCountryInfo      bSendCountryInfo;

  // evaluate params
  if (bItem < CSM_C_ADDRESS_FIELD_DP_MAX_ITEMS)
  {
    switch(bItem)
    {
      default:
      case CSM_C_ADDRESS_FIELD_DP_SLOT:
        #if (CSM_S_TRACE == CSM_C_F_ON)
          #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
            #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
            ETG_TRACE_FATAL(("_CSM_Proc_vSetDatapoolPDDItem - Item NOT modifyable !!!"));
            #endif
          #endif
        #endif
        break;
      case CSM_C_ADDRESS_FIELD_DP_STEERING_WHEEL_TYPE:
        bSteeringWheelType.vSetData(bValue);
        break;
      case CSM_C_ADDRESS_FIELD_DP_VOICE_REC:
        bVoiceRec.vSetData(bValue);
        break;
      case CSM_C_ADDRESS_FIELD_DP_FUEL_TYPE:
        bFuelType.vSetData(bValue);
        break;
      case CSM_C_ADDRESS_FIELD_DP_CAMERA_SYSTEM:
        bCameraSystem.vSetData(bValue);
        break;
      case CSM_C_ADDRESS_FIELD_DP_IPA_HFP:
        bIpaHfp.vSetData(bValue);
        break;
      case CSM_C_ADDRESS_FIELD_DP_SONAR:
        bSonar.vSetData(bValue);
        break;
      case CSM_C_ADDRESS_FIELD_DP_ANTI_THEFT:
        bAntiTheft.vSetData(bValue);
        break;
      case CSM_C_ADDRESS_FIELD_DP_ANTICIPATORY_LINE:
        bAnticipatoryLine.vSetData(bValue);
        break;
      case CSM_C_ADDRESS_FIELD_DP_STEERING_MSG:
        bSteeringMsg.vSetData(bValue);
        break;
      case CSM_C_ADDRESS_FIELD_DP_TRAFFIC_SIGN_REC:
        bTrafficSignRec.vSetData(bValue);
        break;
      case CSM_C_ADDRESS_FIELD_DP_NAVI_FUNCTION:
        bNaviFunc.vSetData(bValue);
        break;
      case CSM_C_ADDRESS_FIELD_DP_RCSTYPE:
        bRCSType.vSetData(bValue);
        break;
      case CSM_C_ADDRESS_FIELD_DP_ITCOMMANDERTYPE:
        bITCommanderType.vSetData(bValue);
        break;
      case CSM_C_ADDRESS_FIELD_DP_VCANMCAN_GATEWAY:
        bVCANMCANGateway.vSetData(bValue);
        break;
      case CSM_C_ADDRESS_FIELD_DP_FUEL_CONSUMPTION_INFO:
        bFuelConsumptionInfo.vSetData(bValue);
        break;
      case   CSM_C_ADDRESS_FIELD_DP_TPMS:
        bTPMS.vSetData(bValue);
        break;
      case   CSM_C_ADDRESS_FIELD_DP_VDC:
        bVDC.vSetData(bValue);
        break;
      case   CSM_C_ADDRESS_FIELD_DP_AD_FUNCTION:
        bAdFunction.vSetData(bValue);
        break;
      case   CSM_C_ADDRESS_FIELD_DP_VEHICLE_SETTING:
        bVehicleSetting.vSetData(bValue);
        break;
      case   CSM_C_ADDRESS_FIELD_DP_HUD_SETTING:
        bHudSetting.vSetData(bValue);
        break;
      case   CSM_C_ADDRESS_FIELD_DP_HVAC_TYPE:
        bHvacType.vSetData(bValue);
        break;
      case   CSM_C_ADDRESS_FIELD_DP_DIGITAL_ASSI:
        bDigitalAssi.vSetData(bValue);
        break;
      case   CSM_C_ADDRESS_FIELD_DP_SEND_COUNTRY_INFO:
        bSendCountryInfo.vSetData(bValue);
        break;
    }
    #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
        #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
        ETG_TRACE_USR3(("_CSM_Proc_vSetDatapoolPDDItem - Temporary set for 1 single v850 restart: Item:%x  Value:%x", bItem, bValue));
        #endif
      #endif
    #endif
  }
  else
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
        #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
        ETG_TRACE_FATAL(("_CSM_Proc_vSetDatapoolPDDItem - Item %x out of range !!!" , bItem));
        #endif
      #endif
    #endif
  }
}

// define the CPP function for internal DP-Initialisation from within this CPP module
// init the PDD datapool part (v850 mirror) from KDS (only upon trigger from DIAG !!!)
tVoid _CSM_Proc_vInitDatapoolPDD(tVoid)
{
  BYTE i;
  typedef struct CSM_DP_Item
  {
    BYTE bItem;
    BYTE bValue;
  } CSM_DP_Item;
  TCSMEarlyNorConfiguration rCSMEarlyNorConfiguration;
  // DataPool Access Variables
  tS32 s32DPReturnValue=DP_S32_NO_ERR;
  CSM_DP_Item aDPItems[CSM_C_ADDRESS_FIELD_DP_MAX_ITEMS] =
  {
    // the order and position of these elements should match the defined CSM data pool structure
    {CSM_C_ADDRESS_FIELD_DP_SLOT                 ,0x00},                // 0 Slot
    {CSM_C_ADDRESS_FIELD_DP_STEERING_WHEEL_TYPE  ,0x00},                // 1 Steering Wheel Type
    {CSM_C_ADDRESS_FIELD_DP_VOICE_REC            ,0x00},                // 2 Voice recognition
    {CSM_C_ADDRESS_FIELD_DP_FUEL_TYPE            ,0x00},                // 3 Fuel type
    {CSM_C_ADDRESS_FIELD_DP_CAMERA_SYSTEM        ,0x00},                // 4 Camera system Type
    {CSM_C_ADDRESS_FIELD_DP_IPA_HFP              ,0x00},                // 5 IPA HFP
    {CSM_C_ADDRESS_FIELD_DP_SONAR                ,0x00},                // 6 Sonar
    {CSM_C_ADDRESS_FIELD_DP_ANTI_THEFT           ,0x00},                // 7 Anti Theft
    {CSM_C_ADDRESS_FIELD_DP_ANTICIPATORY_LINE    ,0x00},                // 8 Anticipatory Line
    {CSM_C_ADDRESS_FIELD_DP_STEERING_MSG         ,0x00},                // 9 Steering Msg ID
    {CSM_C_ADDRESS_FIELD_DP_TRAFFIC_SIGN_REC     ,0x00},                //10 Traffic Sign Recognition
    {CSM_C_ADDRESS_FIELD_DP_NAVI_FUNCTION        ,0x00},                //11 Navi Function
    {CSM_C_ADDRESS_FIELD_DP_RCSTYPE              ,0x00},                //12 RSC Type
    {CSM_C_ADDRESS_FIELD_DP_ITCOMMANDERTYPE      ,0x00},                //13 IT Commander Type
    {CSM_C_ADDRESS_FIELD_DP_VCANMCAN_GATEWAY     ,0x00},                //14 Bridge On Off
    {CSM_C_ADDRESS_FIELD_DP_FUEL_CONSUMPTION_INFO,0x00},                //15
    {CSM_C_ADDRESS_FIELD_DP_TPMS                 ,0x00},                //16
    {CSM_C_ADDRESS_FIELD_DP_VDC                  ,0x00},                //17
    {CSM_C_ADDRESS_FIELD_DP_AD_FUNCTION          ,0x00},                //18
    {CSM_C_ADDRESS_FIELD_DP_VEHICLE_SETTING      ,0x00},                //19
    {CSM_C_ADDRESS_FIELD_DP_HUD_SETTING          ,0x00},                //20
    {CSM_C_ADDRESS_FIELD_DP_HVAC_TYPE            ,0x00},                //21
    {CSM_C_ADDRESS_FIELD_DP_DIGITAL_ASSI         ,0x00},                //22
    {CSM_C_ADDRESS_FIELD_DP_SEND_COUNTRY_INFO    ,0x00},                //23
  };

  DWORD dwOsalReturn;
  DWORD dwOsalError;

  // load KDS values (loop over all configured DP items)
  for (i=0; i < CSM_C_ADDRESS_FIELD_DP_MAX_ITEMS; i++)
  {
    if (i != (aDPItems[i].bItem))
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
        #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
          #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
          ETG_TRACE_FATAL(("_CSM_Proc_vInitDatapoolPDD - ERROR - Configuration Problem for PDD DataPool (Index: %d)", i));
          #endif
        #endif
      #endif
    }

    // get KDS value into our configuration array
    dwOsalReturn = CSM_Proc_dwGetKDSValue(&(aDPItems[i].bValue), (aDPItems[i].bItem), &dwOsalError);

    if (1 != dwOsalReturn)
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
        #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
          #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
          ETG_TRACE_FATAL(("_CSM_Proc_vInitDatapoolPDD - WARNING - KDS value retrieval failed (0x270565 -> unsupported), ErrorCode:0x%x Item: %d", dwOsalError, i));
          #endif
        #endif
      #endif
    }
  }

  // create instances of our DP classes
  dp_tclCsmDpPDDSlotNr               bSlotNr;
  dp_tclCsmDpPDDSteeringWheelType    bSteeringWheelType;
  dp_tclCsmDpPDDVoiceRec             bVoiceRec;
  dp_tclCsmDpPDDFuelType             bFuelType;
  dp_tclCsmDpPDDCameraSystem         bCameraSystem;
  dp_tclCsmDpPDDIpaHfp               bIpaHfp;
  dp_tclCsmDpPDDSonar                bSonar;
  dp_tclCsmDpPDDAntiTheft            bAntiTheft;
  dp_tclCsmDpPDDAnticipatoryLine     bAnticipatoryLine;
  dp_tclCsmDpPDDSteeringMsg          bSteeringMsg;
  dp_tclCsmDpPDDTrafficSignRec       bTrafficSignRec;
  dp_tclCsmDpPDDNaviFunc             bNaviFunc;
  dp_tclCsmDpPDDRCSType              bRCSType;
  dp_tclCsmDpPDDITCommanderType      bITCommanderType;
  dp_tclCsmDpPDDVCANMCANGateway      bVCANMCANGateway;
  dp_tclCsmDpPDDFuelConsumptionInfo  bFuelConsumptionInfo;
  dp_tclCsmDpPDDTPMS                 bTPMS;
  dp_tclCsmDpPDDVDC                  bVDC;
  dp_tclCsmDpPDDAdFunction           bAdFunction;
  dp_tclCsmDpPDDVehicleSetting       bVehicleSetting;
  dp_tclCsmDpPDDHudSetting           bHudSetting;
  dp_tclCsmDpPDDHvacType             bHvacType;
  dp_tclCsmDpPDDDigitalAssi          bDigitalAssi;
  dp_tclCsmDpPDDSendCountryInfo      bSendCountryInfo;

  // create instances of our EarlyNor DP classes
  dp_tclCsmEarlyNorConfigCsmEarlyNorConfig  rCsmEarlyNorConfig;

  #if (CSM_S_TRACE == CSM_C_F_ON)
    #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR3(("CSM_Proc_vInitDatapoolPDD - SlotNr:           %x", aDPItems[CSM_C_ADDRESS_FIELD_DP_SLOT].bValue));
      ETG_TRACE_USR3(("CSM_Proc_vInitDatapoolPDD - SteeringWheelType:%x", aDPItems[CSM_C_ADDRESS_FIELD_DP_STEERING_WHEEL_TYPE].bValue));
      ETG_TRACE_USR3(("CSM_Proc_vInitDatapoolPDD - VoiceRec:         %x", aDPItems[CSM_C_ADDRESS_FIELD_DP_VOICE_REC].bValue));
      ETG_TRACE_USR3(("CSM_Proc_vInitDatapoolPDD - Fueltype:         %x", aDPItems[CSM_C_ADDRESS_FIELD_DP_FUEL_TYPE].bValue));
      ETG_TRACE_USR3(("CSM_Proc_vInitDatapoolPDD - CamerasystemType: %x", aDPItems[CSM_C_ADDRESS_FIELD_DP_CAMERA_SYSTEM].bValue));
      ETG_TRACE_USR3(("CSM_Proc_vInitDatapoolPDD - IPA HFP:          %x", aDPItems[CSM_C_ADDRESS_FIELD_DP_IPA_HFP].bValue));
      ETG_TRACE_USR3(("CSM_Proc_vInitDatapoolPDD - Sonar:            %x", aDPItems[CSM_C_ADDRESS_FIELD_DP_SONAR].bValue));
      ETG_TRACE_USR3(("CSM_Proc_vInitDatapoolPDD - AntiTheft:        %x", aDPItems[CSM_C_ADDRESS_FIELD_DP_ANTI_THEFT].bValue));
      ETG_TRACE_USR3(("CSM_Proc_vInitDatapoolPDD - AnticipatoryLine: %x", aDPItems[CSM_C_ADDRESS_FIELD_DP_ANTICIPATORY_LINE].bValue));
      ETG_TRACE_USR3(("CSM_Proc_vInitDatapoolPDD - SteeringMsg:      %x", aDPItems[CSM_C_ADDRESS_FIELD_DP_STEERING_MSG].bValue));
      ETG_TRACE_USR3(("CSM_Proc_vInitDatapoolPDD - TrafficSignRec:   %x", aDPItems[CSM_C_ADDRESS_FIELD_DP_TRAFFIC_SIGN_REC].bValue));
      ETG_TRACE_USR3(("CSM_Proc_vInitDatapoolPDD - NaviFunc:         %x", aDPItems[CSM_C_ADDRESS_FIELD_DP_NAVI_FUNCTION].bValue));
      ETG_TRACE_USR3(("CSM_Proc_vInitDatapoolPDD - RCSType:          %x", aDPItems[CSM_C_ADDRESS_FIELD_DP_RCSTYPE].bValue));
      ETG_TRACE_USR3(("CSM_Proc_vInitDatapoolPDD - ITCommanderType:  %x", aDPItems[CSM_C_ADDRESS_FIELD_DP_ITCOMMANDERTYPE].bValue));
      ETG_TRACE_USR3(("CSM_Proc_vInitDatapoolPDD - VCANMCANGateway:  %x", aDPItems[CSM_C_ADDRESS_FIELD_DP_VCANMCAN_GATEWAY].bValue));
      ETG_TRACE_USR3(("CSM_Proc_vInitDatapoolPDD - FuelConsumptInf:  %x", aDPItems[CSM_C_ADDRESS_FIELD_DP_FUEL_CONSUMPTION_INFO].bValue));
      ETG_TRACE_USR3(("CSM_Proc_vInitDatapoolPDD - TPMS:             %x", aDPItems[CSM_C_ADDRESS_FIELD_DP_TPMS].bValue));
      ETG_TRACE_USR3(("CSM_Proc_vInitDatapoolPDD - VDC:              %x", aDPItems[CSM_C_ADDRESS_FIELD_DP_VDC].bValue));
      ETG_TRACE_USR3(("CSM_Proc_vInitDatapoolPDD - AdFunction:       %x", aDPItems[CSM_C_ADDRESS_FIELD_DP_AD_FUNCTION].bValue));
      ETG_TRACE_USR3(("CSM_Proc_vInitDatapoolPDD - VehicleSetting:   %x", aDPItems[CSM_C_ADDRESS_FIELD_DP_VEHICLE_SETTING].bValue));
      ETG_TRACE_USR3(("CSM_Proc_vInitDatapoolPDD - HudSetting:       %x", aDPItems[CSM_C_ADDRESS_FIELD_DP_HUD_SETTING].bValue));
      ETG_TRACE_USR3(("CSM_Proc_vInitDatapoolPDD - HvacType:         %x", aDPItems[CSM_C_ADDRESS_FIELD_DP_HVAC_TYPE].bValue));
      ETG_TRACE_USR3(("CSM_Proc_vInitDatapoolPDD - DigitalAssi:      %x", aDPItems[CSM_C_ADDRESS_FIELD_DP_DIGITAL_ASSI].bValue));
      ETG_TRACE_USR3(("CSM_Proc_vInitDatapoolPDD - SendCountryInfo   %x", aDPItems[CSM_C_ADDRESS_FIELD_DP_SEND_COUNTRY_INFO].bValue));
      #endif
    #endif
  #endif

  // write items to PDD datapool section (item by item)
  bSlotNr.vSetData(aDPItems[CSM_C_ADDRESS_FIELD_DP_SLOT].bValue);
  bSteeringWheelType.vSetData(aDPItems[CSM_C_ADDRESS_FIELD_DP_STEERING_WHEEL_TYPE].bValue);
  bVoiceRec.vSetData(aDPItems[CSM_C_ADDRESS_FIELD_DP_VOICE_REC].bValue);
  bFuelType.vSetData(aDPItems[CSM_C_ADDRESS_FIELD_DP_FUEL_TYPE].bValue);
  bCameraSystem.vSetData(aDPItems[CSM_C_ADDRESS_FIELD_DP_CAMERA_SYSTEM].bValue);
  bIpaHfp.vSetData(aDPItems[CSM_C_ADDRESS_FIELD_DP_IPA_HFP].bValue);
  bSonar.vSetData(aDPItems[CSM_C_ADDRESS_FIELD_DP_SONAR].bValue);
  bAntiTheft.vSetData(aDPItems[CSM_C_ADDRESS_FIELD_DP_ANTI_THEFT].bValue);
  bAnticipatoryLine.vSetData(aDPItems[CSM_C_ADDRESS_FIELD_DP_ANTICIPATORY_LINE].bValue);
  bSteeringMsg.vSetData(aDPItems[CSM_C_ADDRESS_FIELD_DP_STEERING_MSG].bValue);
  bTrafficSignRec.vSetData(aDPItems[CSM_C_ADDRESS_FIELD_DP_TRAFFIC_SIGN_REC].bValue);
  bNaviFunc.vSetData(aDPItems[CSM_C_ADDRESS_FIELD_DP_NAVI_FUNCTION].bValue);
  bRCSType.vSetData(aDPItems[CSM_C_ADDRESS_FIELD_DP_RCSTYPE].bValue);
  bITCommanderType.vSetData(aDPItems[CSM_C_ADDRESS_FIELD_DP_ITCOMMANDERTYPE].bValue);
  bVCANMCANGateway.vSetData(aDPItems[CSM_C_ADDRESS_FIELD_DP_VCANMCAN_GATEWAY].bValue);
  bFuelConsumptionInfo.vSetData(aDPItems[CSM_C_ADDRESS_FIELD_DP_FUEL_CONSUMPTION_INFO].bValue);
  bTPMS.vSetData(aDPItems[CSM_C_ADDRESS_FIELD_DP_TPMS].bValue);
  bVDC.vSetData(aDPItems[CSM_C_ADDRESS_FIELD_DP_VDC].bValue);
  bAdFunction.vSetData(aDPItems[CSM_C_ADDRESS_FIELD_DP_AD_FUNCTION].bValue);
  bVehicleSetting.vSetData(aDPItems[CSM_C_ADDRESS_FIELD_DP_VEHICLE_SETTING].bValue);
  bHudSetting.vSetData(aDPItems[CSM_C_ADDRESS_FIELD_DP_HUD_SETTING].bValue);
  bHvacType.vSetData(aDPItems[CSM_C_ADDRESS_FIELD_DP_HVAC_TYPE].bValue);
  bDigitalAssi.vSetData(aDPItems[CSM_C_ADDRESS_FIELD_DP_DIGITAL_ASSI].bValue);
  bSendCountryInfo.vSetData(aDPItems[CSM_C_ADDRESS_FIELD_DP_SEND_COUNTRY_INFO].bValue);

  // write items to Early Nor datapool section
  s32DPReturnValue= rCsmEarlyNorConfig.s32GetData(rCSMEarlyNorConfiguration);
  if (s32DPReturnValue>=DP_S32_NO_ERR)
  {
    rCSMEarlyNorConfiguration.bEarlyNorSlot=aDPItems[CSM_C_ADDRESS_FIELD_DP_SLOT].bValue;
    rCsmEarlyNorConfig.vSetData(rCSMEarlyNorConfiguration);
  }
  else
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
        #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
        ETG_TRACE_FATAL(("_CSM_Proc_vInitDatapoolPDD - WARNING - EarlyNor value retrieval failed , ErrorCode:0x%x ", s32DPReturnValue));
        #endif
      #endif
    #endif
  }
}

//**************************************************************************************************
// get an entry from KDS
//**************************************************************************************************
DWORD CSM_Proc_dwGetKDSValue(BYTE* pTarget, BYTE bItem, DWORD* dwError)
{
  // DataPool Access Variables
  tS32 s32DPReturnValue;
  tU8  u8DPBufferGet[255] = {0};

  DWORD dwReturn    = 1; // presume NO error

  // items defined as indicees (addressfields)
  switch(bItem)
  {
    //--------------------------------------------------------------------------------------------
    // read data from Vehicle Information (64 Bytes max.)
    case CSM_C_ADDRESS_FIELD_DP_SLOT:
      s32DPReturnValue = DP_s32GetConfigItem("VehicleInformation", "VCANDiversity", u8DPBufferGet, 1, FALSE);
      if (s32DPReturnValue == DP_S32_NO_ERR)
      {
        *pTarget = u8DPBufferGet[0];                       // direct value, full byte
        ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - SLOTNr - Index:%x Value:%x", bItem, *pTarget));
      }
      else
      {
        ETG_TRACE_FATAL(("CSM_Proc_dwGetKDSValue - ERROR - SLOTNr - Index:%x", bItem));
        dwReturn = 0;
        *dwError = s32DPReturnValue;
      }
      break;
    //----------------------------------------------
    case CSM_C_ADDRESS_FIELD_DP_STEERING_WHEEL_TYPE:
      s32DPReturnValue = DP_s32GetConfigItem("VehicleInformation", "SteeringSW", u8DPBufferGet, 1, FALSE);
      if (s32DPReturnValue == DP_S32_NO_ERR)
      {
        *pTarget = u8DPBufferGet[0];                       // masked value, right aligned
        ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - SteeringWheelType - Index:%x Value:%x", bItem, *pTarget));
      }
      else
      {
        ETG_TRACE_FATAL(("CSM_Proc_dwGetKDSValue - ERROR - SteeringWheelType - Index:%x", bItem));
        dwReturn = 0;
        *dwError = s32DPReturnValue;
      }
      break;
    // -----------------------------------
    case CSM_C_ADDRESS_FIELD_DP_FUEL_TYPE:
      s32DPReturnValue = DP_s32GetConfigItem("VehicleInformation", "FuelType", u8DPBufferGet, 1, FALSE);
      if (s32DPReturnValue == DP_S32_NO_ERR)
      {
        *pTarget = u8DPBufferGet[0];                       // masked value, right aligned, directly as described in PDF interface doc
        ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - FuelType - Index:%x Value:%x", bItem, *pTarget));
      }
      else
      {
        ETG_TRACE_FATAL(("CSM_Proc_dwGetKDSValue - ERROR - FuelType - Index:%x", bItem));
        dwReturn = 0;
        *dwError = s32DPReturnValue;
      }
      break;
    //----------------------------------
    case CSM_C_ADDRESS_FIELD_DP_IPA_HFP:
      s32DPReturnValue = DP_s32GetConfigItem("VehicleInformation", "IPAHFP", u8DPBufferGet, 1, FALSE);
      if (s32DPReturnValue == DP_S32_NO_ERR)
      {
        *pTarget = u8DPBufferGet[0];                       // masked value, right aligned, directly as described in PDF interface doc
        ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - IpaHfpType - Index:%x Value:%x", bItem, *pTarget));
      }
      else
      {
        ETG_TRACE_FATAL(("CSM_Proc_dwGetKDSValue - ERROR - IpaHfpType - Index:%x", bItem));
        dwReturn = 0;
        *dwError = s32DPReturnValue;
      }
      break;
    //----------------------------------------------
    case CSM_C_ADDRESS_FIELD_DP_TRAFFIC_SIGN_REC:
      s32DPReturnValue = DP_s32GetConfigItem("VehicleInformation", "TrafficSignRecognition", u8DPBufferGet, 1, FALSE);
      if (s32DPReturnValue == DP_S32_NO_ERR)
      {
        *pTarget = u8DPBufferGet[0];                       // masked value, right aligned
        ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - Traffic Sign Recognition - Index:%x Value:%x", bItem, *pTarget));
      }
      else
      {
        ETG_TRACE_FATAL(("CSM_Proc_dwGetKDSValue - ERROR - Traffic Sign Recognition - Index:%x", bItem));
        dwReturn = 0;
        *dwError = s32DPReturnValue;
      }
      break;

    //--------------------------------------------------------------------------------------------
    // read from Camera System (88 Bytes max.)
    case CSM_C_ADDRESS_FIELD_DP_CAMERA_SYSTEM:
      s32DPReturnValue = DP_s32GetConfigItem("CameraSystem", "CameraSystem", u8DPBufferGet, 1, FALSE);
      if (s32DPReturnValue == DP_S32_NO_ERR)
      {
        *pTarget = u8DPBufferGet[0];                       // masked value, right aligned, directly as described in PDF interface doc
        ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - CameraSystem - Index:%x Value:%x", bItem, *pTarget));
      }
      else
      {
        ETG_TRACE_FATAL(("CSM_Proc_dwGetKDSValue - ERROR - CameraSystem - Index:%x", bItem));
        dwReturn = 0;
        *dwError = s32DPReturnValue;
      }
      break;
    //--------------------------------------------
    case CSM_C_ADDRESS_FIELD_DP_ANTICIPATORY_LINE:
      s32DPReturnValue = DP_s32GetConfigItem("CameraSystem", "AnticipatoryLine", u8DPBufferGet, 1, FALSE);
      if (s32DPReturnValue == DP_S32_NO_ERR)
      {
        *pTarget = u8DPBufferGet[0];                       // masked value, right aligned, directly as described in PDF interface doc
        ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - AnticipatoryLine - Index:%x Value:%x", bItem, *pTarget));
      }
      else
      {
        ETG_TRACE_FATAL(("CSM_Proc_dwGetKDSValue - ERROR - AnticipatoryLine - Index:%x", bItem));
        dwReturn = 0;
        *dwError = s32DPReturnValue;
      }
      break;
    //--------------------------------
    case CSM_C_ADDRESS_FIELD_DP_SONAR:
      s32DPReturnValue = DP_s32GetConfigItem("CameraSystem","SonarSystemAvailability",u8DPBufferGet, 1, FALSE);
      if (s32DPReturnValue == DP_S32_NO_ERR)
      {
        *pTarget = u8DPBufferGet[0];                       // masked value, right aligned
        ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - SonarSystemAvailability - Index:%x Value:%x", bItem, *pTarget));
      }
      else
      {
        ETG_TRACE_FATAL(("CSM_Proc_dwGetKDSValue - ERROR - SonarSystemAvailability - Index:%x", bItem));
        dwReturn = 0;
        *dwError = s32DPReturnValue;
      }
      break;
    //---------------------------------------
    case CSM_C_ADDRESS_FIELD_DP_STEERING_MSG:
      s32DPReturnValue = DP_s32GetConfigItem("CameraSystem", "CameraSteeringMessage", u8DPBufferGet, 1, FALSE);
      if (s32DPReturnValue == DP_S32_NO_ERR)
      {
        *pTarget = u8DPBufferGet[0];                       // direct value, full byte
        ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - CameraSteeringMessage - Index:%x Value:%x", bItem, *pTarget));
      }
      else
      {
        ETG_TRACE_FATAL(("CSM_Proc_dwGetKDSValue - ERROR - CameraSteeringMessage - Index:%x", bItem));
        dwReturn = 0;
        *dwError = s32DPReturnValue;
      }
      break;

    //--------------------------------------------------------------------------------------------
    // read data from Anti Theft Map (3 bytes)
    case CSM_C_ADDRESS_FIELD_DP_ANTI_THEFT:
      s32DPReturnValue = DP_s32GetConfigItem("AntitheftMapSelection", "AntitheftFunction", u8DPBufferGet, 1, FALSE);
      if (s32DPReturnValue == DP_S32_NO_ERR)
      {
        *pTarget = u8DPBufferGet[0];                       // direct value, full byte
        ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - AntitheftFunction - Index:%x Value:%x", bItem, *pTarget));
      }
      else
      {
        ETG_TRACE_FATAL(("CSM_Proc_dwGetKDSValue - ERROR - AntitheftFunction - Index:%x", bItem));
        dwReturn = 0;
        *dwError = s32DPReturnValue;
      }
      break;
    //--------------------------------------------------------------------------------------------
    // read data from System Information Common (64 bytes max.)
    case CSM_C_ADDRESS_FIELD_DP_VOICE_REC:
      s32DPReturnValue = DP_s32GetConfigItem("VehicleInformation", "BlowerReductionRequest", u8DPBufferGet, 1, FALSE);
      if (s32DPReturnValue == DP_S32_NO_ERR)
      {
        *pTarget = u8DPBufferGet[0];                       // masked value, right aligned
        ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - BlowerReductionRequest - Index:%x Value:%x", bItem, *pTarget));
      }
      else
      {
        ETG_TRACE_FATAL(("CSM_Proc_dwGetKDSValue - ERROR - BlowerReductionRequest - Index:%x", bItem));
        dwReturn = 0;
        *dwError = s32DPReturnValue;
      }
      break;
    //----------------------------------------------
    case CSM_C_ADDRESS_FIELD_DP_RCSTYPE:
      s32DPReturnValue = DP_s32GetConfigItem("VehicleInformation", "RCSRearArmrestSwitch", u8DPBufferGet, 1, FALSE);
      if (s32DPReturnValue == DP_S32_NO_ERR)
      {
        *pTarget = u8DPBufferGet[0];                       // masked value, right aligned
        ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - RCSRearArmrestSwitch - Index:%x Value:%x", bItem, *pTarget));
      }
      else
      {
        ETG_TRACE_FATAL(("CSM_Proc_dwGetKDSValue - ERROR - RCSRearArmrestSwitch - Index:%x", bItem));
        dwReturn = 0;
        *dwError = s32DPReturnValue;
      }
      break;
    //----------------------------------------------
    case CSM_C_ADDRESS_FIELD_DP_ITCOMMANDERTYPE:
      s32DPReturnValue = DP_s32GetConfigItem("VehicleInformation", "ITCommanderCentralSwitch", u8DPBufferGet, 1, FALSE);
      if (s32DPReturnValue == DP_S32_NO_ERR)
      {
        *pTarget = u8DPBufferGet[0];                       // masked value, right aligned
        ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - ITCommanderCentralSwitch - Index:%x Value:%x", bItem, *pTarget));
      }
      else
      {
        ETG_TRACE_FATAL(("CSM_Proc_dwGetKDSValue - ERROR - ITCommanderCentralSwitch - Index:%x", bItem));
        dwReturn = 0;
        *dwError = s32DPReturnValue;
      }
      break;

    case CSM_C_ADDRESS_FIELD_DP_NAVI_FUNCTION:
      s32DPReturnValue = DP_s32GetConfigItem("SystemConfiguration1", "NaviFunction", u8DPBufferGet, 1, FALSE);
      if (s32DPReturnValue == DP_S32_NO_ERR)
      {
        *pTarget = u8DPBufferGet[0];                       // masked value, right aligned
        ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - NaviFunction - Index:%x Value:%x", bItem, *pTarget));
      }
      else
      {
        ETG_TRACE_FATAL(("CSM_Proc_dwGetKDSValue - ERROR - NaviFunction - Index:%x", bItem));
        dwReturn = 0;
        *dwError = s32DPReturnValue;
      }
      break;

    case CSM_C_ADDRESS_FIELD_DP_VCANMCAN_GATEWAY:
      s32DPReturnValue = DP_s32GetConfigItem("VehicleInformation", "VCANMCANGatewayForANCASC", u8DPBufferGet, 1, FALSE);
      if (s32DPReturnValue == DP_S32_NO_ERR)
      {
        *pTarget = u8DPBufferGet[0];                       // masked value, right aligned
        ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - VCANMCANGatewayForANCASC - Index:%x Value:%x", bItem, *pTarget));
      }
      else
      {
        ETG_TRACE_FATAL(("CSM_Proc_dwGetKDSValue - ERROR - VCANMCANGatewayForANCASC - Index:%x", bItem));
        dwReturn = 0;
        *dwError = s32DPReturnValue;
      }
      break;

    case CSM_C_ADDRESS_FIELD_DP_FUEL_CONSUMPTION_INFO:
      s32DPReturnValue = DP_s32GetConfigItem("VehicleInformation", "FuelConsumptionInfo", u8DPBufferGet, 1, FALSE);
      if (s32DPReturnValue == DP_S32_NO_ERR)
      {
        *pTarget = u8DPBufferGet[0];                       // masked value, right aligned
        ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - FuelConsumptionInfo - Index:%x Value:%x", bItem, *pTarget));
      }
      else
      {
        ETG_TRACE_FATAL(("CSM_Proc_dwGetKDSValue - ERROR - FuelConsumptionInfo - Index:%x", bItem));
        dwReturn = 0;
        *dwError = s32DPReturnValue;
      }
      break;

    case CSM_C_ADDRESS_FIELD_DP_TPMS:
      s32DPReturnValue = DP_s32GetConfigItem("VehicleInformation", "TPMS", u8DPBufferGet, 1, FALSE);
      if (s32DPReturnValue == DP_S32_NO_ERR)
      {
        *pTarget = u8DPBufferGet[0];                       // masked value, right aligned
        ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - TPMS - Index:%x Value:%x", bItem, *pTarget));
      }
      else
      {
        ETG_TRACE_FATAL(("CSM_Proc_dwGetKDSValue - ERROR - TPMS - Index:%x", bItem));
        dwReturn = 0;
        *dwError = s32DPReturnValue;
      }
      break;

    case CSM_C_ADDRESS_FIELD_DP_VDC:
      s32DPReturnValue = DP_s32GetConfigItem("SystemConfiguration1", "VDCSetting", u8DPBufferGet, 1, FALSE);
      if (s32DPReturnValue == DP_S32_NO_ERR)
      {
        *pTarget = u8DPBufferGet[0];                       // masked value, right aligned
        ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - VDC - Index:%x Value:%x", bItem, *pTarget));
      }
      else
      {
        ETG_TRACE_FATAL(("CSM_Proc_dwGetKDSValue - ERROR - VDC - Index:%x", bItem));
        dwReturn = 0;
        *dwError = s32DPReturnValue;
      }
      break;

    case CSM_C_ADDRESS_FIELD_DP_AD_FUNCTION:
      s32DPReturnValue = DP_s32GetConfigItem("VehicleInformation", "ADFunction", u8DPBufferGet, 1, FALSE);
      if (s32DPReturnValue == DP_S32_NO_ERR)
      {
        *pTarget = u8DPBufferGet[0];                       // masked value, right aligned
        ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - ADFunction - Index:%x Value:%x", bItem, *pTarget));
      }
      else
      {
        ETG_TRACE_FATAL(("CSM_Proc_dwGetKDSValue - ERROR - ADFunction - Index:%x", bItem));
        dwReturn = 0;
        *dwError = s32DPReturnValue;
      }
      break;

    case CSM_C_ADDRESS_FIELD_DP_VEHICLE_SETTING:
      s32DPReturnValue = DP_s32GetConfigItem("SystemConfiguration1", "VehicleSetting", u8DPBufferGet, 1, FALSE);
      if (s32DPReturnValue == DP_S32_NO_ERR)
      {
        *pTarget = u8DPBufferGet[0];                       // masked value, right aligned
        ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - VehicleSetting - Index:%x Value:%x", bItem, *pTarget));
      }
      else
      {
        ETG_TRACE_FATAL(("CSM_Proc_dwGetKDSValue - ERROR - VehicleSetting - Index:%x", bItem));
        dwReturn = 0;
        *dwError = s32DPReturnValue;
      }
      break;

    case CSM_C_ADDRESS_FIELD_DP_HUD_SETTING:
      s32DPReturnValue = DP_s32GetConfigItem("VehicleInformation", "HUDSetting", u8DPBufferGet, 1, FALSE);
      if (s32DPReturnValue == DP_S32_NO_ERR)
      {
        *pTarget = u8DPBufferGet[0];                       // masked value, right aligned
        ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - HUDSetting - Index:%x Value:%x", bItem, *pTarget));
      }
      else
      {
        ETG_TRACE_FATAL(("CSM_Proc_dwGetKDSValue - ERROR - HUDSetting - Index:%x", bItem));
        dwReturn = 0;
        *dwError = s32DPReturnValue;
      }
      break;

    case CSM_C_ADDRESS_FIELD_DP_HVAC_TYPE:
      s32DPReturnValue = DP_s32GetConfigItem("SystemConfiguration1", "HVACType", u8DPBufferGet, 1, FALSE);
      if (s32DPReturnValue == DP_S32_NO_ERR)
      {
        *pTarget = u8DPBufferGet[0];                       // masked value, right aligned
        ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - HVACType - Index:%x Value:%x", bItem, *pTarget));
      }
      else
      {
        ETG_TRACE_FATAL(("CSM_Proc_dwGetKDSValue - ERROR - HVACType - Index:%x", bItem));
        dwReturn = 0;
        *dwError = s32DPReturnValue;
      }
      break;

    case CSM_C_ADDRESS_FIELD_DP_DIGITAL_ASSI:
      s32DPReturnValue = DP_s32GetConfigItem("SystemConfiguration1", "DigitalAssistant", u8DPBufferGet, 1, FALSE);
      if (s32DPReturnValue == DP_S32_NO_ERR)
      {
        *pTarget = u8DPBufferGet[0];                       // masked value, right aligned
        ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - DigitalAssistant - Index:%x Value:%x", bItem, *pTarget));
      }
      else
      {
        ETG_TRACE_FATAL(("CSM_Proc_dwGetKDSValue - ERROR - DigitalAssistant - Index:%x", bItem));
        dwReturn = 0;
        *dwError = s32DPReturnValue;
      }
      break;

    case CSM_C_ADDRESS_FIELD_DP_SEND_COUNTRY_INFO:
      s32DPReturnValue = DP_s32GetConfigItem("VehicleInformation", "SendingCountryInformation", u8DPBufferGet, 1, FALSE);
      if (s32DPReturnValue == DP_S32_NO_ERR)
      {
        *pTarget = u8DPBufferGet[0];                       // masked value, right aligned
        ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - SendingCountryInfo - Index:%x Value:%x", bItem, *pTarget));
      }
      else
      {
        ETG_TRACE_FATAL(("CSM_Proc_dwGetKDSValue - ERROR - SendingCountryInfo - Index:%x", bItem));
        dwReturn = 0;
        *dwError = s32DPReturnValue;
      }
      break;

    default:
      // announce an error
      dwReturn = 0;
      *dwError = 0x270565; // means that the item is not yet supported
      break;
  }

  #if (CSM_S_TRACE == CSM_C_F_ON)
    #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR3(("CSM_Proc_dwGetKDSValue - Index:%x Value:%x ", bItem, *pTarget));
      #endif
    #endif
  #endif

  // return 0 if error, else 1
  return dwReturn;
}

//**************************************************************************************************
// start our worker thread
//**************************************************************************************************
tVoid vStartWorkerThread( tVoid )
{
   OSAL_trThreadAttribute  rAttr;
   OSAL_tThreadID          hThreadId = 0;
   tC8                     szThreadName[OSAL_C_U32_MAX_NAMELENGTH] = CSM_PROC_WORKER_THREAD_NAME;

   rAttr.szName       = szThreadName;
   rAttr.s32StackSize = CSM_PROC_WORKER_THREAD_STACKSIZE;
   rAttr.u32Priority  = CSM_PROC_WORKER_THREAD_PRIO;
   rAttr.pfEntry      = (OSAL_tpfThreadEntry)vWorkerThreadFunction;
   rAttr.pvArg        = NULL;

   hThreadId = OSAL_ThreadSpawn(&rAttr);
}

//**************************************************************************************************
// our worker thread function
//**************************************************************************************************
tVoid vWorkerThreadFunction( tPVoid pvArg )
{
  (tVoid)pvArg;
  tS32 s32Qcreated;

  #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_USR1(("ProcCsm - start worker thread of CSM process"));
  #endif

  // first, create required input message queue
  s32Qcreated = OSAL_s32MessageQueueCreate( CSM_C_PROC_MESSAGE_QUEUE_NAME,
                                             CSM_C_PROC_MESSAGE_QUEUE_MAX_ENTRIES,
                                             sizeof(trCSMProcMessageQueueItem),
                                             OSAL_EN_READWRITE,
                                             &hMessageQueueProc
                                           );

  // check if queue had been created
  if (s32Qcreated == OSAL_OK)
  {
    tS32 s32ReadBytes;
    trCSMProcMessageQueueItem rItem;

    // forever wait for queue events
    while (TRUE)
    {
      // this call returns only if a message is available
      s32ReadBytes = OSAL_s32MessageQueueWait( hMessageQueueProc,
                                               (tU8*)&rItem,
                                               sizeof(rItem),
                                               OSAL_NULL,  // equal priorities
                                               OSAL_C_U32_INFINITE
                                             );

      // after message has been arrived, process it if valid
      if ( s32ReadBytes > 0 )
      {
        // identify the message type
        switch(rItem.enID )
        {
          case CSM_PROC_MESSAGE_QUEUE_ID_INIT:
          {
            // the INIT message is sent by the AccessUser instance from within the Pre-Init function
            // call the handler function and pass the data element with the appropriate UNION type
            vHandleInit(rItem.uData.rInit);
          }
          break;

          case CSM_PROC_MESSAGE_QUEUE_ID_REGISTER_APPL:
          {
            // a User Application wants to register a bunch of callback functions
            // call the handler function and pass the data element with the appropriate UNION type
            vHandleRegisterAppl(rItem.uData.rRegisterAppl);
          }
          break;

          case CSM_PROC_MESSAGE_QUEUE_ID_UNREGISTER_APPL:
          {
            tS32 s32RetVal;
            #if (CSM_S_TRACE == CSM_C_F_ON)
            ETG_TRACE_USR1(("ProcCsm - message UNREGISTER APPL received - Handle:0x%08X",
                            rItem.uData.rUnregisterAppl.pvHandle));
            #endif
            s32RetVal = CSMAL_lUnregisterAppl( rItem.uData.rUnregisterAppl.pvHandle );

            vSendGeneralResult( (tCString)"UNREGISTER APPL",
                                CSM_PARENT_MESSAGE_QUEUE_ID_UNREGISTER_APPL,
                                rItem.uData.rUnregisterAppl.hProcessID,
                                rItem.uData.rUnregisterAppl.u32MsgQueueCnt,
                                s32RetVal
                              );
          }
          break;

          case CSM_PROC_MESSAGE_QUEUE_ID_REGISTER_SIGNAL:
          {
            tS32 s32RetVal;
            #if (CSM_S_TRACE == CSM_C_F_ON)
            ETG_TRACE_USR1(("ProcCsm - message REGISTER SIGNAL received - Handle:0x%08X  Signal:0x%08X  pCallback:0x%08X",
                            rItem.uData.rRegisterSignal.pvHandle,
                            rItem.uData.rRegisterSignal.u32SignalId,
                            rItem.uData.rRegisterSignal.vPFNCBRSignalInd));
            #endif

            s32RetVal = CSMAL_lRegisterSignal( rItem.uData.rRegisterSignal.pvHandle,
                                               rItem.uData.rRegisterSignal.u32SignalId,
                                               rItem.uData.rRegisterSignal.vPFNCBRSignalInd
                                             );

            vSendGeneralResult( (tCString)"REGISTER SIGNAL",
                                 CSM_PARENT_MESSAGE_QUEUE_ID_REGISTER_SIGNAL,
                                 rItem.uData.rRegisterSignal.hProcessID,
                                 rItem.uData.rRegisterSignal.u32MsgQueueCnt,
                                 s32RetVal
                              );
          }

          break;

          case CSM_PROC_MESSAGE_QUEUE_ID_UNREGISTER_SIGNAL:
          {
            tS32 s32RetVal;
            #if (CSM_S_TRACE == CSM_C_F_ON)
            ETG_TRACE_USR1(("ProcCsm - message UNREGISTER SIGNAL received - Handle:0x%08X  Signal:0x%08X",
                            rItem.uData.rUnregisterSignal.pvHandle,
                            rItem.uData.rUnregisterSignal.u32SignalId));
            #endif

            s32RetVal = CSMAL_lUnregisterSignal( rItem.uData.rUnregisterSignal.pvHandle,
                                                 rItem.uData.rUnregisterSignal.u32SignalId
                                               );

            vSendGeneralResult( (tCString)"UNREGISTER SIGNAL",
                                CSM_PARENT_MESSAGE_QUEUE_ID_UNREGISTER_SIGNAL,
                                rItem.uData.rUnregisterSignal.hProcessID,
                                rItem.uData.rUnregisterSignal.u32MsgQueueCnt,
                                s32RetVal
                              );
          }
          break;

          case CSM_PROC_MESSAGE_QUEUE_ID_UNREGISTER_ALL_SIGNALS:
          {
            tS32 s32RetVal;
            #if (CSM_S_TRACE == CSM_C_F_ON)
            ETG_TRACE_USR1(("ProcCsm - message UNREGISTER ALL SIGNALS received - Handle:0x%08X",
                            rItem.uData.rUnregisterAllSignals.pvHandle));
            #endif

            s32RetVal = CSMAL_lUnregisterAllSignals(rItem.uData.rUnregisterAllSignals.pvHandle);

            vSendGeneralResult( (tCString)"UNREGISTER ALL SIGNALS",
                                 CSM_PARENT_MESSAGE_QUEUE_ID_UNREGISTER_ALL_SIGNALS,
                                 rItem.uData.rUnregisterAllSignals.hProcessID,
                                 rItem.uData.rUnregisterAllSignals.u32MsgQueueCnt,
                                 s32RetVal
                              );
          }
          break;

          case CSM_PROC_MESSAGE_QUEUE_ID_REGISTER_BUS_ERROR_IND:
          {
            tS32 s32RetVal;
            #if (CSM_S_TRACE == CSM_C_F_ON)
            ETG_TRACE_USR1(("ProcCsm - message REGISTER BUS ERROR IND received - Handle:0x%08X  pCallback:0x%08X",
                            rItem.uData.rRegisterBusErrorInd.pvHandle,
                            rItem.uData.rRegisterBusErrorInd.vPFNBusErrorInd));
            #endif

            #if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
            s32RetVal = CSMAL_lRegisterBusErrorInd( rItem.uData.rRegisterBusErrorInd.pvHandle,
                                                    rItem.uData.rRegisterBusErrorInd.vPFNBusErrorInd
                                                  );
            #else
            s32RetVal = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_SRVC_UNAVAILABLE);
            #endif
            vSendGeneralResult( (tCString)"REGISTER BUS ERROR IND",
                                CSM_PARENT_MESSAGE_QUEUE_ID_REGISTER_BUS_ERROR_IND,
                                rItem.uData.rRegisterBusErrorInd.hProcessID,
                                rItem.uData.rRegisterBusErrorInd.u32MsgQueueCnt,
                                s32RetVal
                              );
          }
          break;

          case CSM_PROC_MESSAGE_QUEUE_ID_COMMUNICATION_REQ:
          {
            tS32 s32RetVal;
            #if (CSM_S_TRACE == CSM_C_F_ON)
             ETG_TRACE_USR1(("ProcCsm - message COMMUNICATION REQ received - Bus:%X  Prot:0x%08X  AddrField:0x%02X%02X  Action:%X  AppID:0x%04X",
                             rItem.uData.rCommunicationReq.u8Bus,
                             rItem.uData.rCommunicationReq.u32ProtocolType,
                             rItem.uData.rCommunicationReq.au8AddressField[0],
                             rItem.uData.rCommunicationReq.au8AddressField[1],
                             rItem.uData.rCommunicationReq.u8Action,
                             rItem.uData.rCommunicationReq.u16ApplID));
            #endif

            s32RetVal = CSM_lCommunicationReq( rItem.uData.rCommunicationReq.u8Bus,
                                               rItem.uData.rCommunicationReq.u32ProtocolType,
                                               (tPCVoid)&(rItem.uData.rCommunicationReq.au8AddressField[0]),
                                               rItem.uData.rCommunicationReq.u8Action,
                                               rItem.uData.rCommunicationReq.u16ApplID
                                             );

            vSendGeneralResult( (tCString)"COMMUNICATION REQ",
                                CSM_PARENT_MESSAGE_QUEUE_ID_COMMUNICATION_REQ,
                                rItem.uData.rCommunicationReq.hProcessID,
                                rItem.uData.rCommunicationReq.u32MsgQueueCnt,
                                s32RetVal
                              );
          }
          break;

          case CSM_PROC_MESSAGE_QUEUE_ID_DATA_REQUEST:
          {
            tS32 s32RetVal;
            #if (CSM_S_TRACE == CSM_C_F_ON)
            ETG_TRACE_USR1(("ProcCsm - message DATA REQ received - Prot:0x%08X  AddrField:0x%02X%02X  Len:%d  Data:0x%02X 0x%02X ....",
                           rItem.uData.rDataRequest.u32ProtocolType,
                           rItem.uData.rDataRequest.au8AddressField[0],
                           rItem.uData.rDataRequest.au8AddressField[1],
                           rItem.uData.rDataRequest.u16DataLength,
                           rItem.uData.rDataRequest.au8Data[0],
                           rItem.uData.rDataRequest.au8Data[1]));
            #endif

            s32RetVal = CSM_lDataReq( rItem.uData.rDataRequest.u32ProtocolType,
                                      (tPCVoid)&(rItem.uData.rDataRequest.au8AddressField[0]),
                                      rItem.uData.rDataRequest.au8Data,
                                      rItem.uData.rDataRequest.u16DataLength
                                    );

            vSendGeneralResult( (tCString)"DATA REQ",
                                CSM_PARENT_MESSAGE_QUEUE_ID_DATA_REQUEST,
                                rItem.uData.rDataRequest.hProcessID,
                                rItem.uData.rDataRequest.u32MsgQueueCnt,
                                s32RetVal
                              );
          }
          break;

          case CSM_PROC_MESSAGE_QUEUE_ID_DATA_IND_PROCESSED:
          {
            tS32 s32RetVal;
            #if (CSM_S_TRACE == CSM_C_F_ON)
            ETG_TRACE_USR1(("ProcCsm - message DATA IND PROCESSED received - Prot:0x%08X  AddrField:0x%02X%02X",
                           rItem.uData.rDataIndProcessed.u32ProtocolType,
                           rItem.uData.rDataIndProcessed.au8AddressField[0],
                           rItem.uData.rDataIndProcessed.au8AddressField[1]));
            #endif

            s32RetVal = CSM_lDataIndProcessed( rItem.uData.rDataIndProcessed.u32ProtocolType,
                                               (tPCVoid)&(rItem.uData.rDataIndProcessed.au8AddressField[0])
                                             );

            vSendGeneralResult( (tCString)"IND PROCESSED",
                                CSM_PARENT_MESSAGE_QUEUE_ID_DATA_IND_PROCESSED,
                                rItem.uData.rDataIndProcessed.hProcessID,
                                rItem.uData.rDataIndProcessed.u32MsgQueueCnt,
                                s32RetVal
                              );
          }
          break;

          case CSM_PROC_MESSAGE_QUEUE_ID_GET_COMMUNICATION_STATE:
          {
            vHandleGetCommunicationState( rItem.uData.rGetCommunicationState );
          }
          break;

          case CSM_PROC_MESSAGE_QUEUE_ID_SIGNAL_WRITE:
          {
            tS32 s32RetVal;
            #if (CSM_S_TRACE == CSM_C_F_ON)
            ETG_TRACE_USR4(("ProcCsm - message SIGNAL WRITE received - Signal:0x%08X  Len:%d  Type:0x%02X",
                             rItem.uData.rSignalWrite.u32SignalId,
                             rItem.uData.rSignalWrite.u8Length,
                             rItem.uData.rSignalWrite.u8TxType));
            #endif

            s32RetVal = CSM_lSignalWrite( rItem.uData.rSignalWrite.u32SignalId,
                                          (tPCVoid)&(rItem.uData.rSignalWrite.au8Data[0]),
                                          rItem.uData.rSignalWrite.u8Length,
                                          rItem.uData.rSignalWrite.u8TxType
                                        );

            vSendGeneralResult( (tCString)"SIGNAL WRITE",
                                CSM_PARENT_MESSAGE_QUEUE_ID_SIGNAL_WRITE,
                                rItem.uData.rSignalWrite.hProcessID,
                                rItem.uData.rSignalWrite.u32MsgQueueCnt,
                                s32RetVal
                              );
          }
          break;

          case CSM_PROC_MESSAGE_QUEUE_ID_MULTIPLE_SIGNAL_WRITE:
          {
            tS32  s32RetVal;
            tU8   u8NumberOfSignals;
            tCSM_MultipleSignalWrite *prSignalList;

            u8NumberOfSignals = rItem.uData.rMultipleSignalWrite.u8NumberOfSignals;

            #if (CSM_S_TRACE == CSM_C_F_ON)
            ETG_TRACE_USR4(("ProcCsm - message MULTIPLE SIGNAL WRITE received - Number of signals:%d", u8NumberOfSignals));
            #endif

            prSignalList = OSAL_NEW tCSM_MultipleSignalWrite[u8NumberOfSignals];

            if ( NULL != prSignalList )
            {
              for ( tU8 u8SignalCnt = 0; u8SignalCnt < u8NumberOfSignals; u8SignalCnt++ )
              {
                #if (CSM_S_TRACE == CSM_C_F_ON)
                ETG_TRACE_USR4(("ProcCsm - Signal:0x%08X  Len:%d  Type:0x%02X",
                                rItem.uData.rMultipleSignalWrite.arSignalData[u8SignalCnt].u32SignalId,
                                rItem.uData.rMultipleSignalWrite.arSignalData[u8SignalCnt].u8Length,
                                rItem.uData.rMultipleSignalWrite.arSignalData[u8SignalCnt].u8TxType));
                #endif

                (prSignalList + u8SignalCnt)->dwSignalId        = rItem.uData.rMultipleSignalWrite.arSignalData[u8SignalCnt].u32SignalId;
                (prSignalList + u8SignalCnt)->pvNewSignalData   = (tPVoid)&(rItem.uData.rMultipleSignalWrite.arSignalData[u8SignalCnt].au8Data[0]);
                (prSignalList + u8SignalCnt)->bDataBufferLength = rItem.uData.rMultipleSignalWrite.arSignalData[u8SignalCnt].u8Length;
                (prSignalList + u8SignalCnt)->bTxType           = rItem.uData.rMultipleSignalWrite.arSignalData[u8SignalCnt].u8TxType;
              }

              s32RetVal = CSM_lMultipleSignalWrite( prSignalList, u8NumberOfSignals );
              OSAL_DELETE[] prSignalList;
            }
            else
            {
              s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_RESOURCE);
            }

            vSendGeneralResult( (tCString)"MULTIPLE SIGNAL WRITE",
                                CSM_PARENT_MESSAGE_QUEUE_ID_MULTIPLE_SIGNAL_WRITE,
                                rItem.uData.rMultipleSignalWrite.hProcessID,
                                rItem.uData.rMultipleSignalWrite.u32MsgQueueCnt,
                                s32RetVal
                              );
          }
          break;

          case CSM_PROC_MESSAGE_QUEUE_ID_SIGNAL_READ:
          {
            vHandleSignalRead( rItem.uData.rSignalRead );
          }
          break;

          case CSM_PROC_MESSAGE_QUEUE_ID_MULTIPLE_SIGNAL_READ:
          {
            vHandleMultipleSignalRead( rItem.uData.rMultipleSignalRead );
          }
          break;

          case CSM_PROC_MESSAGE_QUEUE_ID_GET_ALL_STATE_IND_AGAIN:
          {
            tS32 s32RetVal;
            #if (CSM_S_TRACE == CSM_C_F_ON)
            ETG_TRACE_USR1(("ProcCsm - message GET ALL STATE IND AGAIN received"));
            #endif

            #if ( (CSM_S_CBR_AVAILABLE == CSM_C_F_ON) && (CSM_S_CCS_USED == CSM_C_F_ON) )
            s32RetVal = CSM_lGetAllStateIndicationsAgain();
            #else
            s32RetVal = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_SRVC_UNAVAILABLE );
            #endif

            vSendGeneralResult( (tCString)"GET ALL STATE IND AGAIN",
                                CSM_PARENT_MESSAGE_QUEUE_ID_GET_ALL_STATE_IND_AGAIN,
                                rItem.uData.rGetAllStateIndAgain.hProcessID,
                                rItem.uData.rGetAllStateIndAgain.u32MsgQueueCnt,
                                s32RetVal
                              );
          }
          break;

          case CSM_PROC_MESSAGE_QUEUE_ID_CON_TEST:
          {
            tS32 s32RetVal;
            #if (CSM_S_TRACE == CSM_C_F_ON)
            ETG_TRACE_USR1(("ProcCsm - message CON TEST received - Prot:0x%08X  AddrField:0x%02X%02X  Action:%X",
                            rItem.uData.rConTest.u32ProtocolType,
                            rItem.uData.rConTest.au8AddressField[0],
                            rItem.uData.rConTest.au8AddressField[1],
                            rItem.uData.rConTest.u8Action));
            #endif

            s32RetVal = CSM_lConTest( rItem.uData.rConTest.u32ProtocolType,
                                      (tPVoid)&(rItem.uData.rConTest.au8AddressField[0]),
                                      rItem.uData.rConTest.u8Action );

            vSendGeneralResult( (tCString)"COMMUNICATION REQ",
                                CSM_PARENT_MESSAGE_QUEUE_ID_CON_TEST,
                                rItem.uData.rConTest.hProcessID,
                                rItem.uData.rConTest.u32MsgQueueCnt,
                                s32RetVal
                              );
          }
          break;

          case CSM_PROC_MESSAGE_QUEUE_ID_DATA_READ:
          {
            vHandleDataRead( rItem.uData.rDataRead );
          }
          break;

          case CSM_PROC_MESSAGE_QUEUE_ID_MAX:
          default:
          {
            // do nothing
          }
          break;
        } // switch message type
      } // message > 0 bytes received
    } // endless while loop
  } // message queue has been created

  // if a message queue creation failed somehow, it could be required to properly remove some remaining artefacts
  if (hMessageQueueProc != OSAL_C_INVALID_HANDLE)
  {
    if (OSAL_OK == OSAL_s32MessageQueueClose(hMessageQueueProc))
    {
      OSAL_s32MessageQueueDelete(CSM_C_PROC_MESSAGE_QUEUE_NAME);
    }
  }
}

//**************************************************************************************************
// message handler function: INIT
//**************************************************************************************************
tVoid vHandleInit(trCSMProcMessageQueueItemInit rInit)
{
  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR1(("ProcCsm->vHandleInit - ProcessID:0x%08X  MsgQueueCnt:%04d  QueueElemSize:%04d",
                 rInit.hProcessID,
                 rInit.u32MsgQueueCnt,
                 rInit.u32MsgQueueElementSize));
  #endif

  // Process ID and MessageQueue Nr -> get the correct handle for the queue (and open it if not yet open)
  OSAL_tMQueueHandle hMessageQueueID = hGetParentMessageQueueID(rInit.hProcessID, rInit.u32MsgQueueCnt);

  if (OSAL_C_INVALID_HANDLE != hMessageQueueID)
  {
    pCSMHandle  pAccessUsrHandle;
    trCSMParentMessageQueueItem  rMessage;
    tS32  s32Res;

    // get the next free CSM access handle
    if (CSMAL_lGetAccessHandle(&pAccessUsrHandle) >= CSM_C_NO_ERROR)
    {
      // pass the ProcessId, MessageQueueNr, Handle and ServiceQueueElementSize to the CSMAL
      // all data will be stored within the handle table
      CSMAL_lStoreHandleData((void *)pAccessUsrHandle, (LONG)rInit.u32MsgQueueCnt, (LONG)rInit.hProcessID, (LONG)rInit.u32MsgQueueElementSize);
    }

    // create the response message for the Application
    rMessage.enID = CSM_PARENT_MESSAGE_QUEUE_ID_INIT;
    rMessage.uData.rInit.pvHandle = pAccessUsrHandle;

    // send the CSMAL handle back to the Application (AccessUser instance)
    s32Res = OSAL_s32MessageQueuePost(hMessageQueueID, (tPCU8)&rMessage, sizeof(rMessage), OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST);
  }
  else
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_ERR(("ProcCsm->vHandleInit - could not open message queue"));
    #endif
  }
}

//**************************************************************************************************
// message handler function: REGISTER APPL (ApplCallbackInit)
//**************************************************************************************************
tVoid vHandleRegisterAppl(trCSMProcMessageQueueItemRegisterAppl rRegisterAppl)
{
  tS32  s32RetVal;
  tBool bRegistered = FALSE;

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR1(("ProcCsm - message REGISTER APPL received - Handle:0x%08X  Bus:%X  Prot:0x%08X  AddrField:0x%02X%02X  ApplId:0x%04X",
                  rRegisterAppl.pvHandle,
                  rRegisterAppl.u8Bus,
                  rRegisterAppl.u32ProtocolType,
                  rRegisterAppl.au8AddressField[0],
                  rRegisterAppl.au8AddressField[1],
                  rRegisterAppl.u16ApplId));

  #endif

  // evaluate protocol type to use the required data for this
  switch ( rRegisterAppl.u32ProtocolType )
  {
    #ifdef CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL
    case CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL:
    {
      tCSM_LOCAL_CTRL_APPL_CALLBACK rCallBackFkt;
      rCallBackFkt.pfvCommunicationCon = (tCSM_PFN_TP_COMMUNICATION_CON)rRegisterAppl.pcvCallBackCommunicationCon;
      rCallBackFkt.pfvCommunicationInd = (tCSM_PFN_TP_COMMUNICATION_IND)rRegisterAppl.pcvCallBackCommunicationInd;
      rCallBackFkt.u16ApplID =           rRegisterAppl.u16ApplId;

      s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                       rRegisterAppl.u8Bus,
                                       rRegisterAppl.u32ProtocolType,
                                       (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                       (tPCVoid)&(rCallBackFkt) );
      bRegistered = TRUE;
    }
    break;
    #endif // CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL

    #ifdef CSM_C_PTYPE_GM_NWM
    case CSM_C_PTYPE_GM_NWM:
    {
      tCSM_GMLAN_NWM_APPL_CALLBACK rCallBackFkt;
      rCallBackFkt.pfvCommunicationInd = (tCSM_PFN_TP_COMMUNICATION_IND)rRegisterAppl.pcvCallBackCommunicationInd;

      s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                       rRegisterAppl.u8Bus,
                                       rRegisterAppl.u32ProtocolType,
                                       (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                       (tPCVoid)&(rCallBackFkt) );
      bRegistered = TRUE;
    }
    break;
    #endif // CSM_C_PTYPE_GM_NWM

    #ifdef CSM_C_PTYPE_USED_STD_USDT_ISO
    case CSM_C_PTYPE_USED_STD_USDT_ISO:
    {
      #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
        #if (CSM_S_ISO_TP == CSM_C_F_ON)
        tCSM_ISO_TP_USDT_APPL_CALLBACK rCallBackFkt;
        rCallBackFkt.pfvCommunicationCon = (tCSM_PFN_TP_COMMUNICATION_CON)rRegisterAppl.pcvCallBackCommunicationCon;
        rCallBackFkt.pfvCommunicationInd = (tCSM_PFN_TP_COMMUNICATION_IND)rRegisterAppl.pcvCallBackCommunicationInd;
        rCallBackFkt.pfvDataCon          = (tCSM_PFN_TP_DATA_CON)rRegisterAppl.pcvCallBackDataCon;
        rCallBackFkt.pfvDataInd          = (tCSM_PFN_TP_DATA_IND)rRegisterAppl.pcvCallBackDataInd;

        #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
        rCallBackFkt.pfvDataIndFF        = (tCSM_PFN_TP_DATA_IND_FF)rRegisterAppl.pcvCallBackDataIndFF;
        #endif

        #if (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
        rCallBackFkt.pfvDataErrorInd     = (tCSM_PFN_TP_DATA_ERROR_IND)rRegisterAppl.pcvCallBackDataErrorInd;
        #endif

        s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                         rRegisterAppl.u8Bus,
                                         rRegisterAppl.u32ProtocolType,
                                         (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                         (tPCVoid)&(rCallBackFkt) );
        bRegistered = TRUE;
        #endif
      #endif
    }
    break;
    #endif // CSM_C_PTYPE_USED_STD_USDT_ISO

    #ifdef CSM_C_PTYPE_USED_STD_LIN_DIAG
    case CSM_C_PTYPE_USED_STD_LIN_DIAG:
    {
      #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
        #if (CSM_S_ISO_TP == CSM_C_F_ON)
        tCSM_ISO_TP_USDT_APPL_CALLBACK rCallBackFkt;
        rCallBackFkt.pfvCommunicationCon = (tCSM_PFN_TP_COMMUNICATION_CON)rRegisterAppl.pcvCallBackCommunicationCon;
        rCallBackFkt.pfvCommunicationInd = (tCSM_PFN_TP_COMMUNICATION_IND)rRegisterAppl.pcvCallBackCommunicationInd;
        rCallBackFkt.pfvDataCon          = (tCSM_PFN_TP_DATA_CON)rRegisterAppl.pcvCallBackDataCon;
        rCallBackFkt.pfvDataInd          = (tCSM_PFN_TP_DATA_IND)rRegisterAppl.pcvCallBackDataInd;

        #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
        rCallBackFkt.pfvDataIndFF        = (tCSM_PFN_TP_DATA_IND_FF)rRegisterAppl.pcvCallBackDataIndFF;
        #endif

        #if (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
        rCallBackFkt.pfvDataErrorInd     = (tCSM_PFN_TP_DATA_ERROR_IND)rRegisterAppl.pcvCallBackDataErrorInd;
        #endif

        s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                         rRegisterAppl.u8Bus,
                                         rRegisterAppl.u32ProtocolType,
                                         (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                         (tPCVoid)&(rCallBackFkt) );
        bRegistered = TRUE;
        #endif
      #endif
    }
    break;
    #endif // CSM_C_PTYPE_USED_STD_LIN_DIAG

    #ifdef CSM_C_PTYPE_USED_STD_LIN_USDT
    case CSM_C_PTYPE_USED_STD_LIN_USDT:
    {
      #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
        #if (CSM_S_ISO_TP == CSM_C_F_ON)
        tCSM_ISO_TP_USDT_APPL_CALLBACK rCallBackFkt;
        rCallBackFkt.pfvCommunicationCon = (tCSM_PFN_TP_COMMUNICATION_CON)rRegisterAppl.pcvCallBackCommunicationCon;
        rCallBackFkt.pfvCommunicationInd = (tCSM_PFN_TP_COMMUNICATION_IND)rRegisterAppl.pcvCallBackCommunicationInd;
        rCallBackFkt.pfvDataCon          = (tCSM_PFN_TP_DATA_CON)rRegisterAppl.pcvCallBackDataCon;
        rCallBackFkt.pfvDataInd          = (tCSM_PFN_TP_DATA_IND)rRegisterAppl.pcvCallBackDataInd;

        #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
        rCallBackFkt.pfvDataIndFF        = (tCSM_PFN_TP_DATA_IND_FF)rRegisterAppl.pcvCallBackDataIndFF;
        #endif

        #if (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
        rCallBackFkt.pfvDataErrorInd     = (tCSM_PFN_TP_DATA_ERROR_IND)rRegisterAppl.pcvCallBackDataErrorInd;
        #endif

        s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                         rRegisterAppl.u8Bus,
                                         rRegisterAppl.u32ProtocolType,
                                         (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                         (tPCVoid)&(rCallBackFkt) );
        bRegistered = TRUE;
        #endif
      #endif
    }
    break;
    #endif // CSM_C_PTYPE_USED_STD_LIN_USDT

    #ifdef CSM_C_PTYPE_USED_STD_UUDT_TX
    case CSM_C_PTYPE_USED_STD_UUDT_TX:
    {
      #if (CSM_S_UUDT_TX == CSM_C_F_ON)
      tCSM_UUDT_TX_APPL_CALLBACK rCallBackFkt;
      rCallBackFkt.pfvDataCon = (tCSM_PFN_TP_DATA_CON)rRegisterAppl.pcvCallBackDataCon;

      s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                       rRegisterAppl.u8Bus,
                                       rRegisterAppl.u32ProtocolType,
                                       (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                       (tPCVoid)&(rCallBackFkt) );
      bRegistered = TRUE;
      #endif
    }
    break;
    #endif // CSM_C_PTYPE_USED_STD_UUDT_TX

    #ifdef CSM_C_PTYPE_USED_STD_UUDT_RX
    case CSM_C_PTYPE_USED_STD_UUDT_RX:
    {
      #if (CSM_S_UUDT_RX == CSM_C_F_ON)
      tCSM_UUDT_RX_APPL_CALLBACK rCallBackFkt;
      rCallBackFkt.pfvDataInd = (tCSM_PFN_TP_DATA_IND)rRegisterAppl.pcvCallBackDataInd;

      #if (CSM_S_UUDT_RX_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
      rCallBackFkt.pfvDataErrorInd = (tCSM_PFN_TP_DATA_ERROR_IND)rRegisterAppl.pcvCallBackDataErrorInd;
      #endif

      s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                       rRegisterAppl.u8Bus,
                                       rRegisterAppl.u32ProtocolType,
                                       (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                       (tPCVoid)&(rCallBackFkt) );
      bRegistered = TRUE;
      #endif
    }
    break;
    #endif // CSM_C_PTYPE_USED_STD_UUDT_RX

    #ifdef CSM_C_PTYPE_USED_STD_BR_SIGNAL
    case CSM_C_PTYPE_USED_STD_BR_SIGNAL:
    {
      tCSM_BR_SIGNAL_APPL_CALLBACK rCallBackFkt;
      rCallBackFkt.pfvCommunicationCon = (tCSM_PFN_TP_COMMUNICATION_CON)rRegisterAppl.pcvCallBackCommunicationCon;
      rCallBackFkt.pfvCommunicationInd = (tCSM_PFN_TP_COMMUNICATION_IND)rRegisterAppl.pcvCallBackCommunicationInd;
      rCallBackFkt.u16ApplID =           rRegisterAppl.u16ApplId;

      s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                       rRegisterAppl.u8Bus,
                                       rRegisterAppl.u32ProtocolType,
                                       (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                       (tPCVoid)&(rCallBackFkt) );
      bRegistered = TRUE;
    }
    break;
    #endif // CSM_C_PTYPE_USED_STD_BR_SIGNAL

    #ifdef CSM_C_PTYPE_MCNET_ASDT_STD
    case CSM_C_PTYPE_MCNET_ASDT_STD:
    {
      // not supported yet
    }
    #endif

    #ifdef CSM_C_PTYPE_MCNET_USDT
    case CSM_C_PTYPE_MCNET_USDT:
    {
      // not supported yet
    }
    #endif

    #ifdef CSM_C_PTYPE_VW_BAP
    case CSM_C_PTYPE_VW_BAP:
    {
      #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
      tCSM_VW_BAP_APPL_CALLBACK rCallBackFkt;
      rCallBackFkt.pfvCommunicationCon = (tCSM_PFN_TP_COMMUNICATION_CON)rRegisterAppl.pcvCallBackCommunicationCon;
      rCallBackFkt.pfvDataCon          = (tCSM_PFN_TP_DATA_CON)rRegisterAppl.pcvCallBackDataCon;
      rCallBackFkt.pfvDataInd          = (tCSM_PFN_TP_DATA_IND)rRegisterAppl.pcvCallBackDataInd;

      s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                       rRegisterAppl.u8Bus,
                                       rRegisterAppl.u32ProtocolType,
                                       (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                       (tPCVoid)&(rCallBackFkt) );
      bRegistered = TRUE;
      #endif
    }
    break;
    #endif // CSM_C_PTYPE_VW_BAP

    #ifdef CSM_C_PTYPE_PAG_OSEK_NWM
    case CSM_C_PTYPE_PAG_OSEK_NWM:
    {
      tCSM_OSEK_NWM_APPL_CALLBACK rCallBackFkt;
      rCallBackFkt.pfvCommunicationInd = (tCSM_PFN_TP_COMMUNICATION_IND)rRegisterAppl.pcvCallBackCommunicationInd;

      s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                       rRegisterAppl.u8Bus,
                                       rRegisterAppl.u32ProtocolType,
                                       (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                       (tPCVoid)&(rCallBackFkt) );
      bRegistered = TRUE;
    }
    break;
    #endif // CSM_C_PTYPE_PAG_OSEK_NWM

    #ifdef CSM_C_PTYPE_RN_MPDT_C
    case CSM_C_PTYPE_RN_MPDT_C:
    case CSM_C_PTYPE_RN_MPDT_D:
    {
      #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
      tCSM_MPDT_APPL_CALLBACK rCallBackFkt;
      rCallBackFkt.pfvCommunicationCon = (tCSM_PFN_TP_COMMUNICATION_CON)rRegisterAppl.pcvCallBackCommunicationCon;
      rCallBackFkt.pfvCommunicationInd = (tCSM_PFN_TP_COMMUNICATION_IND)rRegisterAppl.pcvCallBackCommunicationInd;
      rCallBackFkt.pfvCDataCon         = (tCSM_PFN_TP_DATA_CON)rRegisterAppl.pcvCallBackDataCon;
      rCallBackFkt.pfvCDataInd         = (tCSM_PFN_TP_DATA_IND)rRegisterAppl.pcvCallBackDataInd;
      rCallBackFkt.pfvDDataCon         = (tCSM_PFN_TP_DATA_CON)rRegisterAppl.pcvCallBackDDataCon;
      rCallBackFkt.pfvDDataInd         = (tCSM_PFN_TP_DATA_IND)rRegisterAppl.pcvCallBackDDataInd;

      s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                       rRegisterAppl.u8Bus,
                                       rRegisterAppl.u32ProtocolType,
                                       (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                       (tPCVoid)&(rCallBackFkt) );
      bRegistered = TRUE;
      #endif
    }
    break;
    #endif // CSM_C_PTYPE_RN_MPDT_C

  #ifdef CSM_C_PTYPE_J1939
    case CSM_C_PTYPE_J1939:
    {
  #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
    #if (CSM_S_J1939 == CSM_C_F_ON)
      tCSM_J1939_APPL_CALLBACK rCallBackFkt;

      rCallBackFkt.pfvCommunicationCon = (tCSM_PFN_TP_COMMUNICATION_CON)rRegisterAppl.pcvCallBackCommunicationCon;
      rCallBackFkt.pfvCommunicationInd = (tCSM_PFN_TP_COMMUNICATION_IND)rRegisterAppl.pcvCallBackCommunicationInd;
      rCallBackFkt.pfvDataCon          = (tCSM_PFN_TP_DATA_CON)rRegisterAppl.pcvCallBackDataCon;
      rCallBackFkt.pfvDataInd          = (tCSM_PFN_TP_DATA_IND)rRegisterAppl.pcvCallBackDataInd;
    #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
      rCallBackFkt.pfvDataIndFF        = (tCSM_PFN_TP_DATA_IND_FF)rRegisterAppl.pcvCallBackDataIndFF;
    #endif
    #if (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
      rCallBackFkt.pfvDataErrorInd     = (tCSM_PFN_TP_DATA_ERROR_IND)rRegisterAppl.pcvCallBackDataErrorInd;
    #endif

      s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                      rRegisterAppl.u8Bus,
                      rRegisterAppl.u32ProtocolType,
                      (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                      (tPCVoid)&(rCallBackFkt) );
      bRegistered = TRUE;
    #endif
  #endif
    }
    break;
  #endif /* CSM_C_PTYPE_J1939 */

    #ifdef CSM_C_PTYPE_RN_OSEKI_NWM
    case CSM_C_PTYPE_RN_OSEKI_NWM:
    {
      tCSM_OSEKI_NWM_APPL_CALLBACK rCallBackFkt;
      rCallBackFkt.u16ApplID =           rRegisterAppl.u16ApplId;
      rCallBackFkt.pfvCommunicationInd = (tCSM_PFN_TP_COMMUNICATION_IND)rRegisterAppl.pcvCallBackCommunicationInd;

      s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                       rRegisterAppl.u8Bus,
                                       rRegisterAppl.u32ProtocolType,
                                       (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                       (tPCVoid)&(rCallBackFkt) );
      bRegistered = TRUE;
    }
    break;
    #endif // CSM_C_PTYPE_RN_OSEKI_NWM

    default:
    {
      // no valid protocol type
    }
    break;
  }

  // check if any registration happened
  if (FALSE == bRegistered)
  {
    // no -> error
    s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_INVALID_PROTOCOL_TYPE);
  }

  // create response message with either error or registration result
  vSendGeneralResult( (tCString)"REGISTER APPL",
                      CSM_PARENT_MESSAGE_QUEUE_ID_REGISTER_APPL,
                      rRegisterAppl.hProcessID,
                      rRegisterAppl.u32MsgQueueCnt,
                      s32RetVal);
}


tVoid vHandleGetCommunicationState( trCSMProcMessageQueueItemGetCommunicationState rGetCommunicationState )
{
   #if (CSM_S_TRACE == CSM_C_F_ON)
   ETG_TRACE_USR1(("ProcCsm - message GET COMMUNICATION STATE received - ProcessID:%08X  MsgQueueCnt:%04d  Bus:0x%X  Prot:0x%08X  AddrField:0x%02X%02X  AppID:0x%04X",
                   rGetCommunicationState.hProcessID,
                   rGetCommunicationState.u32MsgQueueCnt,
                   rGetCommunicationState.u8Bus,
                   rGetCommunicationState.u32ProtocolType,
                   rGetCommunicationState.au8AddressField[0],
                   rGetCommunicationState.au8AddressField[1],
                   rGetCommunicationState.u16ApplID));
   #endif

   OSAL_tMQueueHandle hMessageQueueID = hGetParentMessageQueueID( rGetCommunicationState.hProcessID, rGetCommunicationState.u32MsgQueueCnt );

   if ( OSAL_C_INVALID_HANDLE != hMessageQueueID )
   {
      tS32  s32Result;
      tU8    u8State = CSM_C_INACTIVE;
      trCSMParentMessageQueueItem  rMessage;

      s32Result = CSM_lGetCommunicationState( rGetCommunicationState.u8Bus,
                                              rGetCommunicationState.u32ProtocolType,
                                              (tPCVoid)&(rGetCommunicationState.au8AddressField[0]),
                                              &u8State,
                                              rGetCommunicationState.u16ApplID );

      rMessage.enID = CSM_PARENT_MESSAGE_QUEUE_ID_GET_COMMUNICATION_STATE;
      rMessage.uData.rGetCommunicationState.s32Result = s32Result;
      rMessage.uData.rGetCommunicationState.u8State = u8State;

      s32Result = OSAL_s32MessageQueuePost( hMessageQueueID, (tPCU8)&rMessage, sizeof(rMessage), OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST );
   }
   else
   {
      #if (CSM_S_TRACE == CSM_C_F_ON)
         ETG_TRACE_ERR(("ProcCsm->vHandleGetCommunicationState - could not open message queue"));
      #endif
   }
}


tVoid vHandleSignalRead( trCSMProcMessageQueueItemSignalRead rSignalRead )
{
   #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR4(("ProcCsm - message SIGNAL READ received - ProcessID:0x%08X  MsgQueueCnt:%04d  Signal:0x%08X  Len:%d",
                      rSignalRead.hProcessID,
                      rSignalRead.u32MsgQueueCnt,
                      rSignalRead.u32SignalId,
                      rSignalRead.u8Length));
   #endif

   OSAL_tMQueueHandle hMessageQueueID = hGetParentMessageQueueID( rSignalRead.hProcessID, rSignalRead.u32MsgQueueCnt );

   if ( OSAL_C_INVALID_HANDLE != hMessageQueueID )
   {
      tS32  s32Result;
      trCSMParentMessageQueueItem  rMessage;

      if ( rSignalRead.u8Length <= CSM_C_ACCESS_LAYER_SIGNAL_DATA_MAXLENGTH )
      {
         s32Result = CSM_lSignalRead( rSignalRead.u32SignalId, (tPVoid)&(rMessage.uData.rSignalRead.au8Data[0]), rSignalRead.u8Length, &(rMessage.uData.rSignalRead.u32SignalStatus) );
      }
      else
      {
         s32Result = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_DATA_LENGTH_EXCEEDED);
      }

      rMessage.enID = CSM_PARENT_MESSAGE_QUEUE_ID_SIGNAL_READ;
      rMessage.uData.rSignalRead.s32Result = s32Result;

      s32Result = OSAL_s32MessageQueuePost( hMessageQueueID, (tPCU8)&rMessage, sizeof(rMessage), OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST );
   }
   else
   {
      #if (CSM_S_TRACE == CSM_C_F_ON)
         ETG_TRACE_ERR(("ProcCsm->vHandleSignalRead - could not open message queue"));
      #endif
   }
}


tVoid vHandleMultipleSignalRead( trCSMProcMessageQueueItemMultipleSignalRead rMultipleSignalRead )
{
   tU8  u8NumberOfSignals = rMultipleSignalRead.u8NumberOfSignals;

   #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR4(("ProcCsm - message MULTIPLE SIGNAL READ received - ProcessID:0x%08X  MsgQueueCnt:%04d  NumberOfSignals:%d",
                      rMultipleSignalRead.hProcessID,
                      rMultipleSignalRead.u32MsgQueueCnt,
                      u8NumberOfSignals));
   #endif

   OSAL_tMQueueHandle hMessageQueueID = hGetParentMessageQueueID( rMultipleSignalRead.hProcessID, rMultipleSignalRead.u32MsgQueueCnt );

   if ( OSAL_C_INVALID_HANDLE != hMessageQueueID )
   {
      tS32  s32Result = CSM_C_NO_ERROR;
      trCSMParentMessageQueueItem  rMessage;

      if ( u8NumberOfSignals <= CSM_C_ACCESS_LAYER_MULTI_SIGNAL_MAXNUMBER )
      {
         tCSM_MultipleSignalRead *prSignalList = OSAL_NEW tCSM_MultipleSignalRead[u8NumberOfSignals];

         if ( NULL != prSignalList )
         {
            for ( tU8 u8SignalCnt = 0; u8SignalCnt < u8NumberOfSignals; u8SignalCnt++ )
            {
               #if (CSM_S_TRACE == CSM_C_F_ON)
               ETG_TRACE_USR4(("ProcCsm - Multiple Signal Read - Signal:0x%08X  Len:%d",
                               rMultipleSignalRead.arSignalData[u8SignalCnt].u32SignalId,
                               rMultipleSignalRead.arSignalData[u8SignalCnt].u8Length));
               #endif

               if ( rMultipleSignalRead.arSignalData[u8SignalCnt].u8Length <= CSM_C_ACCESS_LAYER_SIGNAL_DATA_MAXLENGTH )
               {
                  // fill the requested values to the output message
                  rMessage.uData.rMultipleSignalRead.arSignalData[u8SignalCnt].u32SignalId =
                     rMultipleSignalRead.arSignalData[u8SignalCnt].u32SignalId;
                  rMessage.uData.rMultipleSignalRead.arSignalData[u8SignalCnt].u8Length =
                     rMultipleSignalRead.arSignalData[u8SignalCnt].u8Length;

                  // handle over the requested values and the addresses of the output message parameters to the signal list
                  (prSignalList + u8SignalCnt)->dwSignalId        = rMultipleSignalRead.arSignalData[u8SignalCnt].u32SignalId;
                  (prSignalList + u8SignalCnt)->bDataBufferLength = rMultipleSignalRead.arSignalData[u8SignalCnt].u8Length;
                  (prSignalList + u8SignalCnt)->pvActSignalData   = (tPVoid)&(rMessage.uData.rMultipleSignalRead.arSignalData[u8SignalCnt].au8Data[0]);
                  (prSignalList + u8SignalCnt)->pdwSignalStatus   = (DWORD*)&(rMessage.uData.rMultipleSignalRead.arSignalData[u8SignalCnt].u32SignalStatus);
               }
               else
               {
                  s32Result = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_DATA_LENGTH_EXCEEDED);
                  break;  // leave for-loop
               }
            }

            if ( CSM_C_NO_ERROR == s32Result )
            {
               // now read the data to the list, i.e. to the addresses of the output message parameters
               #if ((CSM_S_iMX_SIGNAL_ABSTRACTION == CSM_C_F_ON) || (CSM_S_v850_SIGNAL_ABSTRACTION == CSM_C_F_ON))
               s32Result = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_SRVC_UNAVAILABLE );
               #else
               s32Result = CSM_lMultipleSignalRead( prSignalList, u8NumberOfSignals );
               #endif
            }

            OSAL_DELETE[] prSignalList;
         }
         else
         {
            s32Result = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_RESOURCE);
         }
      }
      else
      {
         s32Result = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_INVALID_PARA);
      }

      rMessage.enID = CSM_PARENT_MESSAGE_QUEUE_ID_MULTIPLE_SIGNAL_READ;
      rMessage.uData.rMultipleSignalRead.s32Result = s32Result;
      if ( CSM_C_NO_ERROR == s32Result )
      {
         rMessage.uData.rMultipleSignalRead.u8NumberOfSignals = u8NumberOfSignals;
      }
      else
      {
         rMessage.uData.rMultipleSignalRead.u8NumberOfSignals = 0;
      }

      s32Result = OSAL_s32MessageQueuePost( hMessageQueueID, (tPCU8)&rMessage, sizeof(rMessage), OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST );
   }
   else
   {
      #if (CSM_S_TRACE == CSM_C_F_ON)
         ETG_TRACE_ERR(("ProcCsm->vHandleMultipleSignalRead - could not open message queue"));
      #endif
   }
}


tVoid vHandleDataRead( trCSMProcMessageQueueItemDataRead rDataRead )
{
   #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR1(("ProcCsm - message DATA READ received - ProcessID:0x%08X  MsgQueueCnt:%04d  Prot:0x%08X  AddrField:0x%02X%02X  Len:%d",
                       rDataRead.hProcessID,
                       rDataRead.u32MsgQueueCnt,
                       rDataRead.u32ProtocolType,
                       rDataRead.au8AddressField[0],
                       rDataRead.au8AddressField[1],
                       rDataRead.u16DataLength));
   #endif

   OSAL_tMQueueHandle hMessageQueueID = hGetParentMessageQueueID( rDataRead.hProcessID, rDataRead.u32MsgQueueCnt );

   if ( OSAL_C_INVALID_HANDLE != hMessageQueueID )
   {
      tS32  s32Result;
      trCSMParentMessageQueueItem  rMessage;

      if ( rDataRead.u16DataLength <= CSM_C_ACCESS_LAYER_DATA_READ_MAXLENGTH )
      {
         s32Result = CSM_lDataRead( rDataRead.u32ProtocolType, (tPCVoid)&(rDataRead.au8AddressField[0]), &(rMessage.uData.rDataRead.au8Data[0]), rDataRead.u16DataLength );
      }
      else
      {
         s32Result = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_DATA_LENGTH_EXCEEDED);
      }

      rMessage.enID = CSM_PARENT_MESSAGE_QUEUE_ID_DATA_READ;
      rMessage.uData.rDataRead.s32Result = s32Result;

      s32Result = OSAL_s32MessageQueuePost( hMessageQueueID, (tPCU8)&rMessage, sizeof(rMessage), OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST );
   }
   else
   {
      #if (CSM_S_TRACE == CSM_C_F_ON)
         ETG_TRACE_ERR(("ProcCsm->vHandleDataRead - could not open message queue"));
      #endif
   }
}


tVoid vSendGeneralResult( tCString coszDebugString, tenCSMParentMessageQueueItemID enItemID, OSAL_tProcessID hProcessID, tU32 u32MsgQueueCnt, tS32 s32Result )
{
   #if (CSM_S_TRACE == CSM_C_F_ON)
   ETG_TRACE_USR4(("vSendGeneralResult for %s", coszDebugString));
   ETG_TRACE_USR4(("ProcessID:0x%08X  MsgQueueCnt:%04d  Result:0x%08X",
                   hProcessID,
                   u32MsgQueueCnt,
                   s32Result));
   #endif

   OSAL_tMQueueHandle hMessageQueueID = hGetParentMessageQueueID( hProcessID, u32MsgQueueCnt );

   if ( OSAL_C_INVALID_HANDLE != hMessageQueueID )
   {
      trCSMParentMessageQueueItem  rMessage;

      rMessage.enID = enItemID;
      rMessage.uData.rGeneralResult.s32Result = s32Result;

      s32Result = OSAL_s32MessageQueuePost( hMessageQueueID, (tPCU8)&rMessage, sizeof(rMessage), OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST );
   }
   else
   {
      #if (CSM_S_TRACE == CSM_C_F_ON)
         ETG_TRACE_ERR(("vSendGeneralResult - could not open message queue"));
      #endif
   }
}

//**************************************************************************************************
// Helper function: get the correct handle for a clients message queue
//**************************************************************************************************
OSAL_tMQueueHandle hGetParentMessageQueueID(OSAL_tProcessID hProcessID, tU32 u32MsgQueueCnt)
{
  OSAL_tMQueueHandle hResult = OSAL_C_INVALID_HANDLE;
  tU32 u32Idx = 0;

  // search whether queue is already opened
  // loop over all known message queues
  while ((u32Idx < vecParentMessageQueueAdmin.size() ) && (OSAL_C_INVALID_HANDLE == hResult))
  {
    // search for ProcessId and MsgQnr in list
    if ( (vecParentMessageQueueAdmin[u32Idx].hProcessID == hProcessID)
         &&
         (vecParentMessageQueueAdmin[u32Idx].u32MsgQueueCnt == u32MsgQueueCnt)
       )
    {
      hResult = vecParentMessageQueueAdmin[u32Idx].hMessageQueueID;
    }
    u32Idx++;
  }

  // no matching open message queue found -> open it
  if (OSAL_C_INVALID_HANDLE == hResult)
  {
    tChar acMessageQueueName[OSAL_C_U32_MAX_NAMELENGTH];
    tChar acNamePrefix[] = CSMAL_C_PARENT_MESSAGE_QUEUE_NAME_PREFIX;

    sprintf(acMessageQueueName, "%s%08x_%04d", acNamePrefix, hProcessID, u32MsgQueueCnt);

    if (OSAL_OK == OSAL_s32MessageQueueOpen(acMessageQueueName, OSAL_EN_READWRITE, &hResult))
    {
      trCSMParentMessageQueueAdmin rNewMessageQueueAdmin;

      rNewMessageQueueAdmin.hProcessID      = hProcessID;
      rNewMessageQueueAdmin.u32MsgQueueCnt  = u32MsgQueueCnt;
      rNewMessageQueueAdmin.hMessageQueueID = hResult;

      // append the parent applications message queue to list
      vecParentMessageQueueAdmin.push_back(rNewMessageQueueAdmin);

      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR1(("ProcCsm - new parent message queue opened - ID:0x%08X  Name:%s",
                      hResult, acMessageQueueName));
      #endif
    }
  }
  return hResult;
}


tVoid vCloseAllParentMessageQueues( tVoid )
{
   for ( tU32 u32Idx = 0; u32Idx < vecParentMessageQueueAdmin.size(); ++u32Idx )
   {
      if ( OSAL_C_INVALID_HANDLE != vecParentMessageQueueAdmin[u32Idx].hMessageQueueID )
      {
         OSAL_s32MessageQueueClose( vecParentMessageQueueAdmin[u32Idx].hMessageQueueID );
      }
   }

   vecParentMessageQueueAdmin.clear();
}
